import type { ChatGPTAPIBrowser } from 'chatgpt3';
import PQueue from 'p-queue';
import { ConsoleLogger } from '@nestjs/common';
import pTimeout from 'p-timeout';
import { OpenAIAccount } from '../utility/config';

export class AccountState {
  private lock = new PQueue({ concurrency: 1 });
  occupyTimestamp = 0;
  private logger = new ConsoleLogger(`OpenAI Account ${this.loginInfo.email}`);
  constructor(
    public loginInfo: OpenAIAccount,
    private instance: ChatGPTAPIBrowser,
  ) {}

  async init() {
    this.logger.log('Initializing ChatGPT API');
    try {
      await pTimeout(this.instance.initSession(), 120000);
    } catch (e) {
      this.logger.error('Failed to initialize ChatGPT API');
      this.logger.error(e);
      return false;
    }
    this.logger.log('Initialized ChatGPT API');
    return true;
  }

  async close() {
    return this.instance.closeSession();
  }

  isFree() {
    return this.occupyTimestamp === 0;
  }

  queueSize() {
    return this.lock.size + this.lock.pending;
  }

  getEmail() {
    return this.loginInfo.email;
  }

  async sendMessage(...args: Parameters<ChatGPTAPIBrowser['sendMessage']>) {
    return this.lock.add(async () => {
      this.occupyTimestamp = Date.now();
      try {
        this.logger.log(`Sending message`);
        const result = await this.instance.sendMessage(...args);
        this.logger.log(`Sent message`);
        return result;
      } catch (e) {
        this.logger.error(`Failed to send message`);
        this.logger.error(e);
        return;
      } finally {
        this.occupyTimestamp = 0;
      }
    });
  }
}
