import {
  ConsoleLogger,
  Injectable,
  OnModuleDestroy,
  OnModuleInit,
} from '@nestjs/common';
import { ProxyPoolService } from '../proxy-pool/proxy-pool.service';
import { ChatGPTAPIBrowser } from 'chatgpt3';
import { OpenAIAccount } from '../utility/config';
import { AccountState } from './account-state';
import { AccountProviderService } from '../account-provider/account-provider.service';
import { AccountPoolStatusDto } from './account-pool-status.dto';
import { Interval } from '@nestjs/schedule';
import BetterLock from 'better-lock';
import { resolve } from 'path';

interface AddAccountPayload extends OpenAIAccount {
  resolve: (success: boolean) => void;
  reject: (error: any) => void;
  retry: boolean;
}

@Injectable()
export class AccountPoolService
  extends ConsoleLogger
  implements OnModuleInit, OnModuleDestroy
{
  constructor(
    private proxyPoolService: ProxyPoolService,
    private accountProvider: AccountProviderService,
  ) {
    super('AccountPoolService');
  }

  private ChatGPTAPIBrowserConstructor: typeof ChatGPTAPIBrowser;
  private accounts = new Map<string, AccountState>();

  private async initAccount(
    account: OpenAIAccount,
    retry = true,
    proxy?: string,
  ): Promise<boolean> {
    if (this.accounts.has(account.email)) {
      return false;
    }
    if (!proxy) {
      proxy = await this.proxyPoolService.getProxy();
    }
    const state = new AccountState(
      account,
      new this.ChatGPTAPIBrowserConstructor({
        email: account.email,
        password: account.password,
        isProAccount: !!account.pro,
        isGoogleLogin: account.loginType === 'google',
        isMicrosoftLogin: account.loginType === 'microsoft',
        proxyServer: proxy || undefined,
      }),
    );
    this.log(`Adding account ${account.email}`);
    const success = await state.init();
    if (success) {
      this.accounts.set(account.email, state);
      this.log(`Added account ${account.email}`);
      return true;
    } else if (retry) {
      await state.close();
      return this.initAccountQueued(account, retry);
    } else {
      await state.close();
      return false;
    }
  }

  private initAccountQueue = new Map<string, AddAccountPayload[]>();
  private async initAccountQueued(account: OpenAIAccount, retry = true) {
    if (this.accounts.has(account.email)) {
      return false;
    }
    const proxy = (await this.proxyPoolService.getProxy()) || '';
    if (!this.initAccountQueue.has(proxy)) {
      this.initAccountQueue.set(proxy, []);
    }
    const queue = this.initAccountQueue.get(proxy);
    return new Promise<boolean>((resolve, reject) =>
      queue.push({
        ...account,
        retry,
        resolve,
        reject,
      }),
    );
  }

  @Interval(1000 * 10)
  private async initAccountQueueResolve() {
    const entries = [...this.initAccountQueue.entries()];
    await Promise.all(
      entries.map(async ([proxy, queue]) => {
        const item = queue.shift();
        if (!item) {
          return;
        }
        try {
          item.resolve(await this.initAccount(item, item.retry, proxy));
        } catch (e) {
          item.reject(e);
        }
      }),
    );
  }

  private accountInfos: OpenAIAccount[];

  @Interval(1000 * 60 * 5)
  async syncAccounts() {
    this.accountInfos = await this.accountProvider.get();
    const accountEmails = new Set(this.accountInfos.map((a) => a.email));
    const addedAccounts = this.accountInfos.filter(
      (a) => !this.accounts.has(a.email),
    );
    const removedAccounts = Array.from(this.accounts.keys()).filter(
      (a) => !accountEmails.has(a),
    );
    this.log(
      `Sync accounts: ${addedAccounts.length} added, ${removedAccounts.length} removed`,
    );
    await Promise.all([
      ...addedAccounts.map((a) => this.addAccount(a)),
      ...removedAccounts.map((a) => this.removeAccount(a)),
    ]);
  }

  async onModuleInit() {
    this.ChatGPTAPIBrowserConstructor = (
      await eval("import('chatgpt3')")
    ).ChatGPTAPIBrowser;
    this.accountInfos = await this.accountProvider.get();
    this.accountInfos.forEach((a, i) => this.addAccount(a, true, i === 0));
  }

  private addAccountLock = new BetterLock();

  async addAccount(account: OpenAIAccount, retry = false, noQueue = false) {
    return this.addAccountLock.acquire(account.email, () =>
      noQueue
        ? this.initAccount(account, retry)
        : this.initAccountQueued(account, retry),
    );
  }

  async onModuleDestroy() {
    await Promise.all(Array.from(this.accounts.values()).map((a) => a.close()));
  }

  randomAccount(exclude: string[] = []) {
    const accounts = Array.from(this.accounts.values()).filter(
      (a) => !exclude.includes(a.loginInfo.email),
    );
    const freeAccounts = accounts.filter((a) => a.isFree());
    if (freeAccounts.length) {
      return freeAccounts[Math.floor(Math.random() * freeAccounts.length)];
    }
    if (!accounts.length) {
      return;
    }
    let useAccount: AccountState;
    for (const account of accounts) {
      if (
        !useAccount ||
        account.queueSize() < useAccount.queueSize() ||
        (account.queueSize() === useAccount.queueSize() &&
          account.occupyTimestamp < useAccount.occupyTimestamp)
      ) {
        useAccount = account;
      }
    }
    return useAccount;
  }

  hasAccount(email: string) {
    return this.accounts.has(email);
  }

  getAccount(email: string, exclude: string[] = []) {
    if (!email || exclude.includes(email)) {
      return this.randomAccount(exclude);
    }
    return this.accounts.get(email) || this.randomAccount(exclude);
  }

  async removeAccount(email: string) {
    const state = this.accounts.get(email);
    if (!state) {
      return false;
    }
    this.log(`Removing account ${email}`);
    this.accounts.delete(email);
    await state.close();
    this.log(`Removed account ${email}`);
    return true;
  }

  getAccountStatus() {
    const accounts = Array.from(this.accounts.values());
    const plain = {
      free: accounts.filter((a) => a.isFree()).length,
      online: accounts.length,
      total: this.accountInfos.length,
      queuing: accounts.reduce((p, c) => p + c.queueSize(), 0),
    };
    const dto = new AccountPoolStatusDto();
    Object.assign(dto, plain);
    return dto;
  }
}
