import {
  ConsoleLogger,
  Injectable,
  OnModuleDestroy,
  OnModuleInit,
} from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import type { ChatGPTAPIBrowser } from 'chatgpt3';
import { TalkDto } from './talk.dto';
import { ConversationService } from '../conversation/conversation.service';
import { v4 as uuid } from 'uuid';
import { ConversationDto } from '../conversation/conversation.dto';

@Injectable()
export class ChatgptService
  extends ConsoleLogger
  implements OnModuleInit, OnModuleDestroy
{
  constructor(
    private config: ConfigService,
    private conversationService: ConversationService,
  ) {
    super('ChatGPT');
  }

  private ChatGPTAPIBrowserConstructor: typeof ChatGPTAPIBrowser;
  private api: ChatGPTAPIBrowser;

  async onModuleInit() {
    this.ChatGPTAPIBrowserConstructor = (
      await eval("import('chatgpt3')")
    ).ChatGPTAPIBrowser;
    this.api = new this.ChatGPTAPIBrowserConstructor({
      email: this.config.get('OPENAI_EMAIL'),
      password: this.config.get('OPENAI_PASSWORD'),
      isProAccount: !!this.config.get('OPENAI_PRO'),
      isGoogleLogin: this.config.get('OPENAI_LOGIN_TYPE') === 'google',
      isMicrosoftLogin: this.config.get('OPENAI_LOGIN_TYPE') === 'microsoft',
    });
    this.log('Initializing ChatGPT API');
    await this.api.initSession();
  }

  async onModuleDestroy() {
    await this.api.closeSession();
  }

  getAPI() {
    return this.api;
  }

  async chat(question: TalkDto) {
    const session = question.session || uuid();
    const previousConversation = await this.conversationService.getConversation(
      session,
    );
    const result = await this.api.sendMessage(question.text, {
      ...(previousConversation
        ? {
            conversationId: previousConversation.conversationId,
            parentMessageId: previousConversation.messageId,
          }
        : {}),
      timeoutMs: 300000,
    });
    await this.conversationService.saveConversation(session, result);
    const dto = new ConversationDto();
    dto.session = session;
    dto.conversationId = result.conversationId;
    dto.messageId = result.messageId;
    dto.text = result.response.replace(/^<!--(.*)-->$/gm, '');
    return dto;
  }
}
