import Aliyun from '@alicloud/pop-core';
import axios from 'axios';
import _ from 'lodash';
import YAML from 'yaml';
import fs from 'fs';
import net from 'net';
import { assert } from 'console';
import { ping } from 'icmp';
import delay from 'delay';
import PQueue from 'p-queue';
import https from 'https';
import pTimeout from 'p-timeout';

interface Source {
  protocol: string;
  port: number;
  testDomains?: string[];
  source?: string;
}

interface RecordRule {
  match?: string;
  sources: Source[];
}

interface Config {
  aliyun: Aliyun.Config;
  domain: string;
  cdnRecords: RecordRule[];
  timeout: number;
  retryCount: number;
  interval: number;
  cocurrent: number;
  maintainance?: string[];
}

interface DomainRecordObject {
  Record: DomainRecord[];
}

interface DomainRecordReturnResult {
  RequestId: string;
  TotalCount: number;
  PageNumber: number;
  PageSize: number;
  DomainRecords: DomainRecordObject;
}

interface DomainRecord {
  DomainName: string;
  RecordId: string;
  RR: string;
  Type: string;
  Value: string;
  TTL: number;
  Priority: number;
  Line: string;
  Status: string;
  Locked: boolean;
  Weight: number;
  Remark: string;
}

let config: Config;

interface DomainRecordInfo {
  recordRule: RecordRule;
  record: DomainRecord;
  sources: Source[];
  isCDN: boolean;
  good: ConnectResult;
}

const requestOption = {
  method: 'POST',
};

enum ConnectResult {
  SourceBad,
  CDNBad,
  Good,
}

class Checker {
  private http = axios.create({
    httpsAgent: new https.Agent({
      rejectUnauthorized: false,
    }),
  });
  private client: Aliyun;
  private cdnRecordsRegex: RegExp[];
  static order = 0;
  private id: number;
  private records: DomainRecordInfo[];
  private nonCDNRecords: DomainRecordInfo[];
  private CDNRecords: DomainRecordInfo[];
  private checkMethods = new Map<
    string,
    (record: Source, address: string) => Promise<ConnectResult>
  >();
  private availableRecordRules: RecordRule[];

  private async filterRecordRules() {
    this.availableRecordRules = [];
    await Promise.all(
      this.config.cdnRecords.map(async (rule) => {
        const { sources } = rule;
        const goodSources: Source[] = [];
        await Promise.all(
          sources.map(async (source) => {
            if (source.protocol === 'tcp' && source.source) {
              this.message(
                `Checking source ${this.getSourcePattern(
                  source,
                  source.source,
                )}.`,
              );
              try {
                const ms = await this.checkTcpProcess(
                  source.source,
                  source.port,
                );
                goodSources.push(source);
                this.message(
                  `Source ${this.getSourcePattern(
                    source,
                    source.source,
                  )} is good: ${ms} ms.`,
                );
              } catch (e) {
                this.message(
                  `Skipping rule ${
                    rule.match
                  } for source ${this.getSourcePattern(
                    source,
                    source.source,
                  )} unhealthy: ${e.toString()}`,
                );
              }
            } else if (
              source.protocol === 'http' ||
              source.protocol === 'https'
            ) {
              const availableTestDomains: string[] = [];
              await Promise.all(
                source.testDomains.map(async (domain) => {
                  this.message(
                    `Checking source domain ${this.getSourcePattern(
                      source,
                      domain,
                    )}.`,
                  );
                  const errMessage = await this.tryConnectHttp(
                    source.protocol,
                    domain,
                    source.port,
                    domain,
                  );
                  if (errMessage) {
                    this.message(
                      `Skipping domain ${this.getSourcePattern(
                        source,
                        domain,
                      )} of rule ${rule.match} for bad source: ${errMessage}`,
                    );
                  } else {
                    this.message(
                      `Source domain ${this.getSourcePattern(
                        source,
                        domain,
                      )} is good.`,
                    );
                    availableTestDomains.push(domain);
                  }
                }),
              );
              if (availableTestDomains.length) {
                source.testDomains = availableTestDomains;
                goodSources.push(source);
              } else {
                this.message(
                  `Skipping source ${rule.match} ${source.testDomains.join(
                    ',',
                  )} for no available source domains.`,
                );
              }
            } else {
              goodSources.push(source);
            }
          }),
        );
        if (goodSources.length) {
          rule.sources = goodSources;
          this.availableRecordRules.push(rule);
        } else {
          this.message(`Skipping rule ${rule.match} for no available sources.`);
        }
      }),
    );
  }

  private async connectHttpProcess(url: string, hostHeader: string) {
    try {
      await this.http.get(url, {
        headers: { Host: hostHeader },
        timeout: this.config.timeout,
        validateStatus: (status) => status < 500,
      });
      return null;
    } catch (e) {
      return `${url}: ${e.toString()}`;
    }
  }

  private queue: PQueue;

  private async tryConnectHttp(
    protocol: string,
    address: string,
    port: number,
    hostHeader: string,
  ) {
    const url = `${protocol}://${address}:${port}`;
    // const urlKey = `${url}-${hostHeader}`;
    return this.queue.add(() => this.connectHttpProcess(url, hostHeader));
  }
  async checkHttpOrHttps(record: Source, address: string) {
    let good = false;
    if (!record.testDomains.length) {
      return ConnectResult.SourceBad;
    }
    for (const testDomain of record.testDomains) {
      /*const sourceErrorMessage = await this.tryConnectHttp(
        record.protocol,
        testDomain,
        record.port,
        testDomain
      );
      if (sourceErrorMessage != null) {
        this.message(
          `Source ${testDomain}:${record.port} is broken: ${sourceErrorMessage}`,
        );
        // return ConnectResult.SourceBad;
        continue;
      }*/
      const nodeErrorMessage = await this.tryConnectHttp(
        record.protocol,
        address,
        record.port,
        testDomain,
      );
      if (nodeErrorMessage != null) {
        this.message(
          `Connection ${this.getSourcePattern(
            record,
            address,
          )} is broken: ${nodeErrorMessage}`,
        );
        return ConnectResult.CDNBad;
      } else {
        good = true;
      }
    }
    if (!good) {
      this.message(
        `Connection ${this.getSourcePattern(
          record,
          address,
        )} skipped for no sources.`,
      );
      return ConnectResult.SourceBad;
    }
    this.message(
      `Connection ${this.getSourcePattern(record, address)} is good.`,
    );
    return ConnectResult.Good;
  }
  async checkTcpProcess(address: string, port: number) {
    return new Promise<number>((resolve, reject) => {
      const hrstart = process.hrtime();
      const socket = new net.Socket();

      socket.connect(port, address);
      socket.setTimeout(this.config.timeout);

      socket.on('connect', () => {
        socket.destroy();
        resolve(milliseconds());
      });

      socket.on('error', (error) => {
        socket.destroy();
        reject(error);
      });

      socket.on('timeout', (error) => {
        socket.destroy();
        reject(error || 'socket TIMEOUT');
      });

      function milliseconds() {
        const hrend = process.hrtime(hrstart);
        const ms = hrend[0] * 1e3 + hrend[1] / 1e6;
        return Math.floor(ms);
      }
    });
  }
  async checkTcp(record: Source, address: string) {
    /*if (record.source) {
      try {
        await this.checkTcpProcess(record.source, record.port);
      } catch (e) {
        this.message(
          `Source ${address}:${record.port} failed: ${e.toString()}`,
        );
        return ConnectResult.SourceBad;
      }
    }*/
    try {
      const ms = await this.checkTcpProcess(address, record.port);
      this.message(
        `Connection ${this.getSourcePattern(
          record,
          address,
        )} is good: ${ms} ms.`,
      );
      return ConnectResult.Good;
    } catch (e) {
      this.message(
        `Connection ${this.getSourcePattern(
          record,
          address,
        )} failed: ${e.toString()}`,
      );
      return ConnectResult.CDNBad;
    }
  }
  async checkIcmp(record: Source, address: string) {
    try {
      await ping(address, this.config.timeout);
      this.message(
        `Connection ${this.getSourcePattern(record, address)} is good.`,
      );
      return ConnectResult.Good;
    } catch (e) {
      this.message(
        `Connection ${this.getSourcePattern(
          record,
          address,
        )} failed: ${e.toString()}`,
      );
      return ConnectResult.CDNBad;
    }
  }

  maintainance: Set<string>;

  constructor(private config: Config) {
    this.queue = new PQueue({ concurrency: this.config.cocurrent || 10 });
    this.client = new Aliyun(config.aliyun);
    this.cdnRecordsRegex = config.cdnRecords.map((m) => new RegExp(m.match));
    this.id = ++Checker.order;
    this.checkMethods.set('http', (record: Source, address: string) => {
      return this.checkHttpOrHttps(record, address);
    });
    this.checkMethods.set('https', (record: Source, address: string) => {
      return this.checkHttpOrHttps(record, address);
    });
    this.checkMethods.set('tcp', (record: Source, address: string) => {
      return this.checkTcp(record, address);
    });
    this.checkMethods.set('icmp', (record: Source, address: string) => {
      return this.checkIcmp(record, address);
    });
    this.maintainance = new Set(config.maintainance || []);
  }
  private message(msg: string) {
    console.log(`${this.id} => ${msg}`);
  }
  getRecordPrefix(m: DomainRecord) {
    if (!m.Value.endsWith(this.config.domain)) {
      return null;
    }
    return m.Value.slice(0, m.Value.length - 1 - this.config.domain.length);
  }
  isCDNRecord(m: DomainRecord) {
    const valuePrefix = this.getRecordPrefix(m);
    return (
      valuePrefix && this.cdnRecordsRegex.some((r) => !!valuePrefix.match(r))
    );
  }
  getSourcePattern(source: Source, address: string) {
    return `${source.protocol}://${address}:${source.port}`;
  }
  getRecordPattern(recordInfo: DomainRecordInfo) {
    const record = recordInfo.record;
    return `${record.RR}.${this.config.domain} => ${recordInfo.sources
      .map((s) => this.getSourcePattern(s, record.Value))
      .join(', ')}`;
  }
  async getRecords(): Promise<DomainRecordInfo[]> {
    this.message(`Fetching domain records of ${this.config.domain}.`);
    const res: DomainRecordInfo[] = [];
    for (let i = 1; ; ++i) {
      const ret: DomainRecordReturnResult = await this.client.request(
        'DescribeDomainRecords',
        {
          DomainName: this.config.domain,
          PageNumber: i,
          PageSize: 500,
        },
        requestOption,
      );
      //this.message(`${ret.TotalCount} records found.`);
      if (!ret.DomainRecords.Record.length) {
        break;
      }
      for (const record of ret.DomainRecords.Record.filter((m) => {
        return (
          m.RR &&
          m.Type === 'CNAME' &&
          this.cdnRecordsRegex.some((r) => !!m.RR.match(r))
        );
      })) {
        const matchCDNRecord = this.availableRecordRules.find((r) =>
          record.RR.match(r.match),
        );
        if (!matchCDNRecord) {
          // source down
          this.message(
            `Will skip record ${record.RR}.${this.config.domain} => ${record.Value} because source is down.`,
          );
          continue;
        }
        const isCDN = this.isCDNRecord(record);
        const recordInfo: DomainRecordInfo = {
          recordRule: matchCDNRecord,
          record,
          sources: matchCDNRecord.sources,
          isCDN,
          good: null,
        };
        this.message(`Found record ${this.getRecordPattern(recordInfo)}`);
        res.push(recordInfo);
      }
    }
    return res;
  }

  async checkSource(source: Source, address: string): Promise<ConnectResult> {
    if (this.maintainance.has(address)) {
      this.message(
        `Skip ${this.getSourcePattern(
          source,
          address,
        )} because it is in maintainance.`,
      );
      return ConnectResult.CDNBad;
    }
    const checkMethodFunction = this.checkMethods.get(source.protocol);
    assert(
      checkMethodFunction,
      `Check method ${source.protocol} not supported.`,
    );
    let lastResult: ConnectResult;
    for (let i = 1; i <= this.config.retryCount; ++i) {
      const prom = checkMethodFunction(source, address);
      try {
        lastResult = await pTimeout(prom, this.config.timeout * 2 + 1000);
      } catch (e) {
        this.message(
          `Connection ${this.getSourcePattern(source, address)} timeout.`,
        );
        lastResult = ConnectResult.CDNBad;
      }
      if (lastResult == ConnectResult.Good) {
        return lastResult;
      }
    }
    if (lastResult === ConnectResult.CDNBad) {
      this.message(
        `Connection ${this.getSourcePattern(source, address)} is bad.`,
      );
    } else {
      this.message(
        `Connection ${this.getSourcePattern(
          source,
          address,
        )} skipped for source broken.`,
      );
    }
    return lastResult;
  }

  async checkNode(recordInfo: DomainRecordInfo): Promise<ConnectResult> {
    const record = recordInfo.recordRule;
    const address = recordInfo.record.Value;
    const checks = await Promise.all(
      record.sources.map(async (source) => this.checkSource(source, address)),
    );
    if (checks.some((r) => r === ConnectResult.CDNBad)) {
      this.message(`Node ${this.getRecordPattern(recordInfo)} is bad.`);
      return ConnectResult.CDNBad;
    }
    if (checks.every((r) => r === ConnectResult.SourceBad)) {
      this.message(
        `Node ${this.getRecordPattern(
          recordInfo,
        )} skipped for no good sources.`,
      );
      return ConnectResult.SourceBad;
    }
    this.message(`Node ${this.getRecordPattern(recordInfo)} is good.`);
    return ConnectResult.Good;
  }

  async checkRecord(recordInfo: DomainRecordInfo) {
    const record = recordInfo.record;
    this.message(
      `Checking record ${this.getRecordPattern(
        recordInfo,
      )} with old status of ${record.Status}.`,
    );
    const good = await this.checkNode(recordInfo);
    await this.handleRecordResult(recordInfo, good);
  }
  isRecordGood(recordInfo: DomainRecordInfo): boolean {
    if (recordInfo.isCDN) {
      const valuePrefix = this.getRecordPrefix(recordInfo.record);
      return this.records.some((r) => {
        return r.record.RR === valuePrefix && this.isRecordGood(r);
      });
    } else {
      return recordInfo.good !== ConnectResult.CDNBad;
    }
  }
  async checkCDNRecord(recordInfo: DomainRecordInfo) {
    this.message(
      `Checking CDN record ${this.getRecordPattern(
        recordInfo,
      )} with old status of ${recordInfo.record.Status}.`,
    );
    const good = this.isRecordGood(recordInfo);
    this.message(
      `CDN Record ${this.getRecordPattern(recordInfo)} is ${
        good ? 'good' : 'bad'
      }.`,
    );
    await this.handleRecordResult(
      recordInfo,
      good ? ConnectResult.Good : ConnectResult.CDNBad,
    );
  }

  private requestQueue = new PQueue({ concurrency: 1 });
  async handleRecordResult(recordInfo: DomainRecordInfo, good: ConnectResult) {
    const record = recordInfo.record;
    const status = record.Status;
    recordInfo.good = good;
    if (good === ConnectResult.SourceBad) {
      return;
    }
    const targetStatus = good === ConnectResult.Good ? 'ENABLE' : 'DISABLE';
    if (status != targetStatus) {
      this.message(
        `Changing record status of ${this.getRecordPattern(
          recordInfo,
        )} from ${status} to ${targetStatus}.`,
      );
      await this.requestQueue.add(() =>
        this.client.request(
          'SetDomainRecordStatus',
          {
            RecordId: record.RecordId,
            Status: targetStatus,
          },
          requestOption,
        ),
      );
    }
  }
  async start() {
    this.message(`Started.`);
    await this.filterRecordRules();
    this.records = await this.getRecords();
    this.nonCDNRecords = this.records.filter((m) => !m.isCDN);
    this.CDNRecords = this.records.filter((m) => !!m.isCDN);
    await Promise.all(
      this.nonCDNRecords.map((r) => {
        return this.checkRecord(r);
      }),
    );
    await Promise.all(
      this.CDNRecords.map((r) => {
        return this.checkCDNRecord(r);
      }),
    );
    this.message(`Finished.`);
  }
}

async function run() {
  const checker = new Checker(_.cloneDeep(config));
  await checker.start();
}

async function main() {
  config = YAML.parse(await fs.promises.readFile('./config.yaml', 'utf8'));
  //await run();
  while (true) {
    await run();
    await delay(config.interval);
  }
}

main();
