import Aliyun from "@alicloud/pop-core";
import axios from "axios";
import _ from "underscore";
import YAML from "yaml";
import fs from "fs";
import {CronJob} from "cron";

interface CDNRecord {
	match: string;
	port: number;
}

interface Config {
	aliyun: Aliyun.Config;
	domain: string;
	cdnRecords: CDNRecord[];
	testDomains: string[];
	timeout: number;
	retryCount: number;
	cronString: string;
}

interface DomainRecordObject {
	Record: DomainRecord[];
}

interface DomainRecordReturnResult {
	RequestId: string;
	TotalCount: number;
	PageNumber: number;
	PageSize: number;
	DomainRecords: DomainRecordObject;
}

interface DomainRecord {
	DomainName: string;
	RecordId: string;
	RR: string;
	Type: string;
	Value: string;
	TTL: number;
	Priority: number;
	Line: string;
	Status: string;
	Locked: boolean;
	Weight: number;
	Remark: string;
}

let config: Config;

interface DomainRecordInfo {
	record: DomainRecord;
	port: number;
	isCDN: boolean;
	good: boolean;
}

const requestOption = {
	method: "POST"
}

class Checker {
	config: Config;
	client: Aliyun;
	cdnRecordsRegex: RegExp[];
	static order: number = 0;
	id: number;
	constructor(config: Config) {
		this.config = config;
		this.client = new Aliyun(config.aliyun);
		this.cdnRecordsRegex = config.cdnRecords.map(m => new RegExp(m.match));
		this.id = ++Checker.order;
	}
	private message(msg: string) {
		console.log(`${this.id} => ${msg}`);
	}
	getRecordPrefix(m: DomainRecord) {
		if (!m.Value.endsWith(this.config.domain)) {
			return null;
		}
		return m.Value.slice(0, m.Value.length - 1 - this.config.domain.length);
	}
	isCDNRecord(m: DomainRecord) {
		const valuePrefix = this.getRecordPrefix(m);
		return valuePrefix && _.any(this.cdnRecordsRegex, r => !!valuePrefix.match(r));
	}
	getRecordPattern(recordInfo: DomainRecordInfo) {
		const record = recordInfo.record;
		return `${record.RR}.${this.config.domain} => ${record.Value}:${recordInfo.port}`;
	}
	async getRecords(): Promise<DomainRecordInfo[]> {
		this.message(`Fetching domain records of ${this.config.domain}.`)
		const res: DomainRecordInfo[] = [];
		for (let i = 1; ; ++i) {
			const ret: DomainRecordReturnResult = await this.client.request("DescribeDomainRecords", {
				DomainName: this.config.domain,
				PageNumber: i,
				PageSize: 500,
			}, requestOption);
			//this.message(`${ret.TotalCount} records found.`);
			if (!ret.DomainRecords.Record.length) {
				break;
			}
			for (let record of ret.DomainRecords.Record.filter(m => {
				return m.RR && m.Type === "CNAME" && _.any(this.cdnRecordsRegex, r => !!m.RR.match(r))
			})) {
				const port = _.find(this.config.cdnRecords, r => record.RR.match(r.match)).port;
				const isCDN = this.isCDNRecord(record);
				const recordInfo: DomainRecordInfo = { record, port, isCDN, good: false };
				this.message(`Found record ${this.getRecordPattern(recordInfo)}`);
				res.push(recordInfo);
			}
		}
		return res;
	}
	async checkNode(address: string, port: number): Promise<boolean> {
		let currentTestDomain: string;
		for (let i = 1; i <= this.config.retryCount; ++i) {
			try {
				for (let testDomain of this.config.testDomains) {
					currentTestDomain = testDomain;
					await axios.get(`https://${address}:${port}`, {
						headers: {
							Host: testDomain
						},
						timeout: this.config.timeout,
						validateStatus: status => status < 500
					});
				}
				this.message(`Node ${address}:${port} is good.`);
				return true;
			} catch (e) {
				this.message(`Node ${address}:${port} Failed in checking ${currentTestDomain} ${i}: ${e.toString()}`);
			}
		}
		this.message(`Node ${address}:${port} is bad.`);
		return false;
	}
	async checkRecord(recordInfo: DomainRecordInfo) {
		const record = recordInfo.record;
		this.message(`Checking record ${this.getRecordPattern(recordInfo)} with old status of ${record.Status}.`)
		const good = await this.checkNode(record.Value, recordInfo.port);
		await this.handleRecordResult(recordInfo, good);
	}
	async checkCDNRecord(recordInfo: DomainRecordInfo, nonCDNRecords: DomainRecordInfo[]) {
		const record = recordInfo.record;
		this.message(`Checking CDN record ${this.getRecordPattern(recordInfo)} with old status of ${record.Status}.`)
		const valuePrefix = this.getRecordPrefix(record);
		const good = _.any(nonCDNRecords, r => {
			return r.record.RR === valuePrefix && r.good;
		});
		this.message(`CDN Record ${this.getRecordPattern(recordInfo)} is ${good ? "good" : "bad"}.`);
		await this.handleRecordResult(recordInfo, good);
	}
	async handleRecordResult(recordInfo: DomainRecordInfo, good: boolean) {
		const record = recordInfo.record;
		const status = record.Status;
		recordInfo.good = good;
		const targetStatus = good ? "ENABLE" : "DISABLE";
		if (status != targetStatus) {
			this.message(`Changing record status of ${this.getRecordPattern(recordInfo)} from ${status} to ${targetStatus}.`);
			await this.client.request("SetDomainRecordStatus", {
				RecordId: record.RecordId,
				Status: targetStatus
			}, requestOption);
		}
	}
	async start() {
		this.message(`Started.`);
		const records = await this.getRecords();
		const nonCDNRecords = records.filter(m => !m.isCDN);
		const CDNRecords = records.filter(m => !!m.isCDN);
		await Promise.all(nonCDNRecords.map(r => {
			return this.checkRecord(r);
		}));
		//await Promise.all(CDNRecords.map(r => {
		//	return this.checkCDNRecord(r, nonCDNRecords);
		//}));
		this.message(`Finished.`);
	}
}

async function run() {
	const checker = new Checker(config);
	await checker.start();
}

async function main() {
	config = YAML.parse(await fs.promises.readFile("./config.yaml", "utf8"));
	//await run();
	(new CronJob(config.cronString, run, null, true, "Asia/Shanghai", null, true)).start();
}

main();
