import Aliyun from '@alicloud/pop-core';
import axios from 'axios';
import _ from 'lodash';
import YAML from 'yaml';
import fs from 'fs';
import { CronJob } from 'cron';
import net from 'net';
import { assert } from 'console';
import { ping } from 'icmp';
import delay from 'delay';

interface RecordRule {
  match?: string;
  protocol: string;
  port: number;
  testDomains?: string[];
  source?: string;
}

interface Config {
  aliyun: Aliyun.Config;
  domain: string;
  cdnRecords: RecordRule[];
  timeout: number;
  retryCount: number;
  interval: number;
}

interface DomainRecordObject {
  Record: DomainRecord[];
}

interface DomainRecordReturnResult {
  RequestId: string;
  TotalCount: number;
  PageNumber: number;
  PageSize: number;
  DomainRecords: DomainRecordObject;
}

interface DomainRecord {
  DomainName: string;
  RecordId: string;
  RR: string;
  Type: string;
  Value: string;
  TTL: number;
  Priority: number;
  Line: string;
  Status: string;
  Locked: boolean;
  Weight: number;
  Remark: string;
}

let config: Config;

interface DomainRecordInfo {
  recordRule: RecordRule;
  record: DomainRecord;
  protocol: string;
  port: number;
  isCDN: boolean;
  good: ConnectResult;
}

const requestOption = {
  method: 'POST',
};

enum ConnectResult {
  SourceBad,
  CDNBad,
  Good,
}

class Checker {
  private client: Aliyun;
  private cdnRecordsRegex: RegExp[];
  static order = 0;
  private id: number;
  private records: DomainRecordInfo[];
  private nonCDNRecords: DomainRecordInfo[];
  private CDNRecords: DomainRecordInfo[];
  private checkMethods = new Map<
    string,
    (record: RecordRule, address: string) => Promise<ConnectResult>
  >();
  private availableRecordRules: RecordRule[];

  private async filterRecordRules() {
    this.availableRecordRules = [];
    await Promise.all(
      this.config.cdnRecords.map(async (rule) => {
        if (rule.protocol === 'tcp' && rule.source) {
          this.message(`Checking source ${rule.source}:${rule.port}.`);
          try {
            const ms = await this.checkTcpProcess(rule.source, rule.port);
            this.availableRecordRules.push(rule);
            this.message(
              `Source ${rule.source}:${rule.port} is good: ${ms} ms.`,
            );
          } catch (e) {
            this.message(
              `Skipping rule ${rule.match} fo source ${rule.source}:${
                rule.port
              } unhealthy: ${e.toString()}`,
            );
          }
        } else if (
          rule.source &&
          (rule.protocol === 'http' || rule.protocol === 'https')
        ) {
          const availableTestDomains: string[] = [];
          for (const domain of rule.testDomains) {
            this.message(`Checking source domain ${domain}:${rule.port}.`);
            const errMessage = await this.tryConnectHttp(
              rule.protocol,
              domain,
              rule.port,
              domain,
            );
            if (errMessage) {
              this.message(
                `Skipping domain ${domain} of rule ${rule.match} for bad source: ${errMessage}`,
              );
            } else {
              this.message(`Source domain ${domain} is good.`);
              availableTestDomains.push(domain);
            }
          }
          if (availableTestDomains.length) {
            rule.testDomains = availableTestDomains;
            this.availableRecordRules.push(rule);
          } else {
            this.message(
              `Skipping rule ${rule.match} for no available sources.`,
            );
          }
        } else {
          this.availableRecordRules.push(rule);
        }
      }),
    );
  }

  private async connectHttpProcess(url: string, hostHeader: string) {
    try {
      await axios.get(url, {
        headers: { Host: hostHeader },
        timeout: this.config.timeout,
        validateStatus: (status) => status < 500,
      });
      return null;
    } catch (e) {
      return `${url}: ${e.toString()}`;
    }
  }

  private async tryConnectHttp(
    protocol: string,
    address: string,
    port: number,
    hostHeader: string,
  ) {
    const url = `${protocol}://${address}:${port}`;
    // const urlKey = `${url}-${hostHeader}`;
    return this.connectHttpProcess(url, hostHeader);
  }
  async checkHttpOrHttps(record: RecordRule, address: string) {
    let good = false;
    if (!record.testDomains.length) {
      return ConnectResult.SourceBad;
    }
    for (const testDomain of record.testDomains) {
      /*const sourceErrorMessage = await this.tryConnectHttp(
        record.protocol,
        testDomain,
        record.port,
        testDomain
      );
      if (sourceErrorMessage != null) {
        this.message(
          `Source ${testDomain}:${record.port} is broken: ${sourceErrorMessage}`,
        );
        // return ConnectResult.SourceBad;
        continue;
      }*/
      const nodeErrorMessage = await this.tryConnectHttp(
        record.protocol,
        address,
        record.port,
        testDomain,
      );
      if (nodeErrorMessage != null) {
        this.message(
          `Node ${address}:${record.port} is broken: ${nodeErrorMessage}`,
        );
        return ConnectResult.CDNBad;
      } else {
        good = true;
      }
    }
    if (!good) {
      this.message(`Node ${address}:${record.port} skipped for no sources.`);
      return ConnectResult.SourceBad;
    }
    this.message(`Node ${address}:${record.port} is good.`);
    return ConnectResult.Good;
  }
  async checkTcpProcess(address: string, port: number) {
    return new Promise<number>((resolve, reject) => {
      const hrstart = process.hrtime();
      const socket = new net.Socket();

      socket.connect(port, address);
      socket.setTimeout(this.config.timeout || 1000);

      socket.on('connect', () => {
        socket.destroy();
        resolve(milliseconds());
      });

      socket.on('error', (error) => {
        socket.destroy();
        reject(error);
      });

      socket.on('timeout', (error) => {
        socket.destroy();
        reject(error || 'socket TIMEOUT');
      });

      function milliseconds() {
        const hrend = process.hrtime(hrstart);
        const ms = hrend[0] * 1e3 + hrend[1] / 1e6;
        return Math.floor(ms);
      }
    });
  }
  async checkTcp(record: RecordRule, address: string) {
    /*if (record.source) {
      try {
        await this.checkTcpProcess(record.source, record.port);
      } catch (e) {
        this.message(
          `Source ${address}:${record.port} failed: ${e.toString()}`,
        );
        return ConnectResult.SourceBad;
      }
    }*/
    try {
      const ms = await this.checkTcpProcess(address, record.port);
      this.message(`Node ${address}:${record.port} is good: ${ms} ms.`);
      return ConnectResult.Good;
    } catch (e) {
      this.message(`Node ${address}:${record.port} failed: ${e.toString()}`);
      return ConnectResult.CDNBad;
    }
  }
  async checkIcmp(address: string) {
    try {
      await ping(address, this.config.timeout);
      this.message(`Node ${address}:ICMP is good.`);
      return ConnectResult.Good;
    } catch (e) {
      this.message(`Node ${address}:ICMP failed: ${e.toString()}`);
      return ConnectResult.CDNBad;
    }
  }
  constructor(private config: Config) {
    this.client = new Aliyun(config.aliyun);
    this.cdnRecordsRegex = config.cdnRecords.map((m) => new RegExp(m.match));
    this.id = ++Checker.order;
    this.checkMethods.set('http', (record: RecordRule, address: string) => {
      return this.checkHttpOrHttps(record, address);
    });
    this.checkMethods.set('https', (record: RecordRule, address: string) => {
      return this.checkHttpOrHttps(record, address);
    });
    this.checkMethods.set('tcp', (record: RecordRule, address: string) => {
      return this.checkTcp(record, address);
    });
    this.checkMethods.set('icmp', (record: RecordRule, address: string) => {
      return this.checkIcmp(address);
    });
  }
  private message(msg: string) {
    console.log(`${this.id} => ${msg}`);
  }
  getRecordPrefix(m: DomainRecord) {
    if (!m.Value.endsWith(this.config.domain)) {
      return null;
    }
    return m.Value.slice(0, m.Value.length - 1 - this.config.domain.length);
  }
  isCDNRecord(m: DomainRecord) {
    const valuePrefix = this.getRecordPrefix(m);
    return (
      valuePrefix && this.cdnRecordsRegex.some((r) => !!valuePrefix.match(r))
    );
  }
  getRecordPattern(recordInfo: DomainRecordInfo) {
    const record = recordInfo.record;
    return `${record.RR}.${this.config.domain} => ${record.Value}:${recordInfo.port}`;
  }
  async getRecords(): Promise<DomainRecordInfo[]> {
    this.message(`Fetching domain records of ${this.config.domain}.`);
    const res: DomainRecordInfo[] = [];
    for (let i = 1; ; ++i) {
      const ret: DomainRecordReturnResult = await this.client.request(
        'DescribeDomainRecords',
        {
          DomainName: this.config.domain,
          PageNumber: i,
          PageSize: 500,
        },
        requestOption,
      );
      //this.message(`${ret.TotalCount} records found.`);
      if (!ret.DomainRecords.Record.length) {
        break;
      }
      for (const record of ret.DomainRecords.Record.filter((m) => {
        return (
          m.RR &&
          m.Type === 'CNAME' &&
          this.cdnRecordsRegex.some((r) => !!m.RR.match(r))
        );
      })) {
        const matchCDNRecord = this.availableRecordRules.find((r) =>
          record.RR.match(r.match),
        );
        if (!matchCDNRecord) {
          // source down
          this.message(
            `Will skip record ${record.RR}.${this.config.domain} => ${record.Value} because source is down.`,
          );
          continue;
        }
        const { port, protocol } = matchCDNRecord;
        const isCDN = this.isCDNRecord(record);
        const recordInfo: DomainRecordInfo = {
          recordRule: matchCDNRecord,
          record,
          protocol,
          port,
          isCDN,
          good: null,
        };
        this.message(`Found record ${this.getRecordPattern(recordInfo)}`);
        res.push(recordInfo);
      }
    }
    return res;
  }
  async checkNode(record: RecordRule, address: string): Promise<ConnectResult> {
    const checkMethodFunction = this.checkMethods.get(record.protocol);
    assert(
      checkMethodFunction,
      `Check method ${record.protocol} not supported.`,
    );
    let lastResult: ConnectResult;
    for (let i = 1; i <= this.config.retryCount; ++i) {
      const result = await checkMethodFunction(record, address);
      if (result == ConnectResult.Good) {
        return result;
      }
      lastResult = result;
    }
    if (lastResult === ConnectResult.CDNBad) {
      this.message(`Node ${address}:${record.port} is bad.`);
    } else {
      this.message(`Node ${address}:${record.port} skipped for source broken.`);
    }
    return lastResult;
  }
  async checkRecord(recordInfo: DomainRecordInfo) {
    const record = recordInfo.record;
    this.message(
      `Checking record ${this.getRecordPattern(
        recordInfo,
      )} with old status of ${record.Status}.`,
    );
    const good = await this.checkNode(recordInfo.recordRule, record.Value);
    await this.handleRecordResult(recordInfo, good);
  }
  isRecordGood(recordInfo: DomainRecordInfo): boolean {
    if (recordInfo.isCDN) {
      const valuePrefix = this.getRecordPrefix(recordInfo.record);
      return this.records.some((r) => {
        return r.record.RR === valuePrefix && this.isRecordGood(r);
      });
    } else {
      return recordInfo.good !== ConnectResult.CDNBad;
    }
  }
  async checkCDNRecord(recordInfo: DomainRecordInfo) {
    this.message(
      `Checking CDN record ${this.getRecordPattern(
        recordInfo,
      )} with old status of ${recordInfo.record.Status}.`,
    );
    const good = this.isRecordGood(recordInfo);
    this.message(
      `CDN Record ${this.getRecordPattern(recordInfo)} is ${
        good ? 'good' : 'bad'
      }.`,
    );
    await this.handleRecordResult(
      recordInfo,
      good ? ConnectResult.Good : ConnectResult.CDNBad,
    );
  }
  async handleRecordResult(recordInfo: DomainRecordInfo, good: ConnectResult) {
    const record = recordInfo.record;
    const status = record.Status;
    recordInfo.good = good;
    if (good === ConnectResult.SourceBad) {
      return;
    }
    const targetStatus = good === ConnectResult.Good ? 'ENABLE' : 'DISABLE';
    if (status != targetStatus) {
      this.message(
        `Changing record status of ${this.getRecordPattern(
          recordInfo,
        )} from ${status} to ${targetStatus}.`,
      );
      await this.client.request(
        'SetDomainRecordStatus',
        {
          RecordId: record.RecordId,
          Status: targetStatus,
        },
        requestOption,
      );
    }
  }
  async start() {
    this.message(`Started.`);
    await this.filterRecordRules();
    this.records = await this.getRecords();
    this.nonCDNRecords = this.records.filter((m) => !m.isCDN);
    this.CDNRecords = this.records.filter((m) => !!m.isCDN);
    await Promise.all(
      this.nonCDNRecords.map((r) => {
        return this.checkRecord(r);
      }),
    );
    await Promise.all(
      this.CDNRecords.map((r) => {
        return this.checkCDNRecord(r);
      }),
    );
    this.message(`Finished.`);
  }
}

async function run() {
  const checker = new Checker(_.cloneDeep(config));
  await checker.start();
}

async function main() {
  config = YAML.parse(await fs.promises.readFile('./config.yaml', 'utf8'));
  //await run();
  while (true) {
    await run();
    await delay(config.interval);
  }
}

main();
