import cluster from "cluster";

export interface SendData {
	id: number,
	proto: string,
	param: any
}

class Task {
	id: number;
	solved: boolean;
	callback: (ret: any) => void;
	constructor(id: number, proto: string, param: any, callback: (ret: any) => void, worker?: cluster.Worker) {
		this.id = id;
		this.callback = callback;
		this.solved = false;
		const sendData: SendData = {
			id,
			proto,
			param
		}
		if (cluster.isMaster && worker) {
			worker.send(sendData);
		} else {
			process.send(sendData);
		}
	}
	solve(ret: any) {
		if (this.solved) {
			return;
		}
		this.solved = true;
		this.callback(ret);
	}
}

async function sleep(ms: number) {
	return new Promise(callback => {
		setTimeout(callback, ms);
	})
}

export class Processor {
	curID: number = 0;
	queue: Map<number, Task>;
	handlers: Map<string, (param: any, id: number, workerID?: number) => Promise<any>>;
	nproc: number;
	workerReadyCallbacks: Map<number, () => void>
	constructor(nproc?: number) {
		if (cluster.isMaster && nproc === undefined) {
			throw "A value of nproc is needed.";
		}
		this.queue = new Map();
		this.handlers = new Map();
		this.workerReadyCallbacks = new Map();
		this.nproc = nproc || 0;
		if (cluster.isMaster) {
			this.addHandler("ready", async (param: any, dataID: number, workerID: number) => {
				const callback = this.workerReadyCallbacks.get(workerID);
				if (callback) {
					callback();
				}
			});
			cluster.on("message", Processor.masterHandler(this));
		} else {
			process.on("message", Processor.workerHandler(this));
		}
	}

	async startWorkers(env?: any) {
		if (cluster.isMaster) {
			let readyPromises = [];
			for (let i = 0; i < this.nproc; ++i) {
				readyPromises.push(new Promise(callback => {
					const worker = cluster.fork(env);
					const ID = worker.id;
					this.workerReadyCallbacks.set(ID, callback);
				}));
			}
			await Promise.all(readyPromises);
		}
	}

	async ready() {
		if (cluster.isWorker) {
			await this.addTask("ready", null);
		}
	}

	addHandler(proto: string, handler: (param: any, id: number, workerID?: number) => Promise<any>) {
		this.handlers.set(proto, handler);
	}

	solveTask(data: SendData) {
		const task = this.queue.get(data.id);
		this.queue.delete(data.id);
		if (task && !task.solved) {
			task.solve(data.param);
		}
	}

	static masterHandler(_this: Processor) {
		return async (worker: cluster.Worker, data: SendData) => {
			if (data.proto === "solve") {
				_this.solveTask(data);
			} else if (_this.handlers.has(data.proto)) {
				const handler = _this.handlers.get(data.proto);
				const ret = await handler(data.param, data.id, worker.id);
				const sendData: SendData = {
					id: data.id,
					proto: "solve",
					param: ret
				};
				worker.send(sendData);
			} else {
				console.error(`Unknown task: ${data.proto}`);
				const sendData: SendData = {
					id: data.id,
					proto: "solve",
					param: null
				};
				worker.send(sendData);
			}
		};
	}

	static workerHandler(_this: Processor) {
		return async (data: SendData) => {
			if (data.proto === "solve") {
				_this.solveTask(data);
			} else if (_this.handlers.has(data.proto)) {
				const handler = _this.handlers.get(data.proto);
				const ret = await handler(data.param, data.id);
				const sendData: SendData = {
					id: data.id,
					proto: "solve",
					param: ret
				};
				process.send(sendData);
			} else {
				console.error(`Unknown task: ${data.proto}`);
				const sendData: SendData = {
					id: data.id,
					proto: "solve",
					param: null
				};
				process.send(sendData);
			}
		};
	}

	addTask(proto: string, param: any, targetWorker?: string): any {
		return new Promise((callback: (ret: any) => void) => {
			let worker: cluster.Worker;
			if (cluster.isMaster) {
				if (!targetWorker) {
					targetWorker = Object.keys(cluster.workers)[this.curID % this.nproc];
				}
				worker = cluster.workers[targetWorker]
			}
			const task = new Task(++this.curID, proto, param, callback, worker);
			this.queue.set(task.id, task);
		});
	}
}
