import config from './config';
import mysql, { ConnectionConfig } from "promise-mysql";
import axios from "axios";
import { Processor } from "./processor";
import { CutData } from "./utility";
import _ from 'underscore';

async function getRealUsername(originalUsername: string) {
	try {
		const { data: {user: { name, username }} } = (await axios.get(`https://ygobbs.com/users/${encodeURIComponent(originalUsername)}.json`, {
			responseType: "json"
		}));
		return _.uniq([originalUsername, username, name]);
	} catch (e) {
		console.error(`User ${originalUsername} not found: ${e.toString()}`);
		return [originalUsername];
	}
}

async function getAvatarURL(usernames: string[]): Promise<string> {
	for (const username of usernames) {
		try {
			const { data: { user: { avatar: _avatar } } } = (await axios.get(`https://api.moecube.com/accounts/users/${encodeURIComponent(username)}.json`, {
				responseType: "json"
			}));
			const avatar = _avatar as string;
			if (avatar?.match(/^http(s)?:\/\//) && !avatar.endsWith('default_avatar.jpg')) {
				return avatar;
			}
		} catch (e) { 
			console.error(`Avatar of ${username} not found: ${e.toString()}`);
		}
	}
	return "https://cdn02.moecube.com:444/accounts/default_avatar.jpg";
}

async function getURLFromUsername(username: string) {
	const possibleNames = await getRealUsername(username);
	return getAvatarURL(possibleNames);
}

interface QueryResult{
	content: string;
}

export class Avatar {
	db: mysql.Pool;
	inited: boolean;
	processor: Processor;
	async init() {
		console.error(`Initing...`);
		console.error(`Connectiing to database...`);
		this.db = await mysql.createPool({
			host: config.dbHost,
			port: config.dbPort as number,
			user: config.dbUser,
			password: config.dbPassword,
			database: config.dbName
		});
		console.error(`Creating table...`);
		await this.db.query("CREATE TABLE IF NOT EXISTS `avatar_cache` (\n" +
				"  `id` bigint(20) NOT NULL AUTO_INCREMENT,\n" +
				"  `url` text COLLATE utf8_unicode_ci NOT NULL,\n" +
				"  `size` smallint UNSIGNED NOT NULL,\n" +
				"  `content` MEDIUMTEXT COLLATE utf8_unicode_ci NOT NULL,\n" + 
				"  PRIMARY KEY (`id`),\n" +
				"  INDEX(`url`(42)),\n" +
				"  INDEX(`url`(42), `size`)\n" +
			") ENGINE=InnoDB AUTO_INCREMENT=2 DEFAULT CHARSET=utf8 COLLATE=utf8_unicode_ci;");
		console.error(`Staring workers...`);
		this.processor = new Processor(config.workers);
		await this.processor.startWorkers();
		console.error(`Ready.`);
		this.inited = true;
	}
	async getImage(username: string, size: number, filename: string) {
		if (!this.inited) {
			return null;
		}
		if (size < 8 || size > 1000) {
			return null;
		}
		const url = await getURLFromUsername(username);
		let buffer: Buffer;
		let content: string;
		let queryResult: QueryResult[]
		try {
			queryResult = await this.db.query("select content from `avatar_cache` where url = ? and size = ? limit 1", [url, size]);
		} catch (e) {
			console.error(`Failed to fetch avatar cache ${url} ${size}: ${e.toString()}`);
		}
		if (queryResult && queryResult.length) {
			content = queryResult[0].content;
			buffer = Buffer.from(content, "base64");
			return buffer;
		}
		console.error(`Resizing image of ${username} ${url} with size ${size}.`);
		try {
			queryResult = await this.db.query("select content from `avatar_cache` where url = ? and size = 0 limit 1", [url, size]);
		} catch (e) {
			console.error(`Failed to fetch original avatar cache ${url} ${size}: ${e.toString()}`);
		}
		if (queryResult && queryResult.length) {
			content = queryResult[0].content;
			buffer = Buffer.from(content, "base64");
		} else {
			try {
				buffer = (await axios.get(url, {
					responseType: "arraybuffer"
				})).data as Buffer;
			} catch (e) {
				console.error(`Error fetching original image of ${username} ${url} : ${e.toString()}`);
				return null;
			}
			content = buffer.toString("base64");
			try {
				await this.db.query("insert into `avatar_cache` set ?", {
					url,
					size: 0,
					content
				});
			} catch (e) {
				console.error(`Failed to save original avatar cache ${url} ${size}: ${e.toString()}`);
			}
		}
		const resizedContentArray: number[] = await this.processor.addTask("cut", {
				image: buffer.toJSON().data,
				filename,
				size
			} as CutData);
		if(!resizedContentArray) {
			console.error(`Error resizing image of ${username} ${url} with size ${size}.`);
			return null;
		}
		const resizedBuffer = Buffer.from(resizedContentArray);
		try {
			await this.db.query("insert into `avatar_cache` set ?", {
				url,
				size,
				content: resizedBuffer.toString("base64")
			});
		} catch (e) {
			console.error(`Failed to save avatar cache ${url} ${size}: ${e.toString()}`);
		}
		return resizedBuffer;
	}
}
