import YGOProDeck from '..';
import { BufferReader } from '../src/utils';

const deck = new YGOProDeck();
deck.main.push(123);
deck.main.push(123);
deck.main.push(123);
deck.main.push(456);
deck.main.push(456);
deck.main.push(789);
deck.extra.push(1234);
deck.extra.push(5678);
deck.side.push(12345);

describe('Sample test.', () => {
  it('should encode and decode deck string', () => {
    const encoded = deck.toEncodedString();
    const buf = Buffer.from(
      encoded.replace(/[-_]/g, (m0) => (m0 == '-' ? '+' : '/')),
      'base64',
    );
    expect(buf.readUInt32LE(0) & 0xfffffff).toBe(123);
    expect((buf.readUInt32LE(0) >>> 28) & 0x3).toBe(0);
    expect(((buf.readUInt32LE(0) >>> 30) & 0x3) + 1).toBe(3);
    expect(buf.readUInt32LE(4) & 0xfffffff).toBe(456);
    expect((buf.readUInt32LE(4) >>> 28) & 0x3).toBe(0);
    expect(((buf.readUInt32LE(4) >>> 30) & 0x3) + 1).toBe(2);
    expect(buf.readUInt32LE(8) & 0xfffffff).toBe(789);
    expect((buf.readUInt32LE(8) >>> 28) & 0x3).toBe(0);
    expect(((buf.readUInt32LE(8) >>> 30) & 0x3) + 1).toBe(1);
    expect(buf.readUInt32LE(12) & 0xfffffff).toBe(1234);
    expect((buf.readUInt32LE(12) >>> 28) & 0x3).toBe(1);
    expect(((buf.readUInt32LE(12) >>> 30) & 0x3) + 1).toBe(1);
    expect(buf.readUInt32LE(16) & 0xfffffff).toBe(5678);
    expect((buf.readUInt32LE(16) >>> 28) & 0x3).toBe(1);
    expect(((buf.readUInt32LE(16) >>> 30) & 0x3) + 1).toBe(1);
    expect(buf.readUInt32LE(20) & 0xfffffff).toBe(12345);
    expect((buf.readUInt32LE(20) >>> 28) & 0x3).toBe(2);
    expect(((buf.readUInt32LE(20) >>> 30) & 0x3) + 1).toBe(1);
    expect(buf.length).toBe(24);
    const decoded = new YGOProDeck().fromEncodedString(encoded);
    expect(decoded.main).toStrictEqual(deck.main);
    expect(decoded.extra).toStrictEqual(deck.extra);
    expect(decoded.side).toStrictEqual(deck.side);
  });
  it('should encode and decode ydk text', () => {
    const ydk = deck.toYdkString();
    console.log(ydk);
    expect(ydk.split('\n')).toHaveLength(13);
    const decoded = new YGOProDeck().fromYdkString(ydk);
    expect(decoded.main).toStrictEqual(deck.main);
    expect(decoded.extra).toStrictEqual(deck.extra);
    expect(decoded.side).toStrictEqual(deck.side);
  });

  it('should encode and decode ygomobile', () => {
    const uri =
      'http://deck.ourygo.top?name=%E7%99%BD%E9%93%B6%E5%9F%8E%E7%A0%81&ygotype=deck&v=1&d=J-xK4Mka02AuEAMf2dV6mj7aRemuJNQJK8BwrcEYh0MqOJqBJSgYqUwxPEVhOG8R18WWVKmzkT-xEdwxbmGVkkOdrVIpufaYI3Hs8oOrcya8Bi40h9G79iFW80rq-o6P-AHsusPY5nmvHLol0DIqEykESVlf6VSbxVJp-j7XZtTE0XvmJW80rqH28R4rgyRovOusVJzbutenYFBA_cyK6d3UWcQkJQlLjaroWavH-INFA56k5DQNWOQ1gpvxrKVBLgEk1olpolKmSgriramLlgtBK1EQ6C6oi94ZyHe7N6T7mqE6peds7mahrORP6A';
    const correctYdk = `#created by OURYGO
#main
22812963
55410872
102380
89631139
89631139
89631139
95718355
95718355
95718355
59323650
22938501
96540807
6637331
33854624
72656408
63198739
63198739
14558127
14558127
23434538
23434538
94145021
17947697
17947697
97268402
17725109
18144507
25311006
31786838
31786838
31786838
80326401
80326401
80326401
24224830
24224830
29095457
29095457
48130397
65681983
67171933
67171933
67171933
71143015
56506740
56506740
24382602
10045474
10045474
22634473
43219114
27781371
62089826
85442146
48130397
8240199
22812963
22812963
#extra
12381100
43228023
56532353
2129638
11443677
11765832
21123811
89604813
63436931
10515412
59822133
59822133
74586817
29301450
42097666
!side
20292186
85103922
34267821
87126721
87126721
84192580
7608148
12444060
15693423
15693423
20899496
30748475
82732705
94145021
94145021`;
    const deck = new YGOProDeck().fromYGOMobileDeckURL(uri);
    expect(deck.main).toHaveLength(58);
    expect(deck.extra).toHaveLength(15);
    expect(deck.side).toHaveLength(15);
    expect(deck.main[0]).toBe(22812963);
    expect(deck.extra[0]).toBe(12381100);
    expect(deck.side[0]).toBe(20292186);

    const correctDeck = new YGOProDeck().fromYdkString(correctYdk);
    expect(deck.main).toStrictEqual(correctDeck.main);
    expect(deck.extra).toStrictEqual(correctDeck.extra);
    expect(deck.side).toStrictEqual(correctDeck.side);

    const uri2 = deck.toYGOMobileDeckURL();

    expect(uri2).toBe(uri);
  });

  it('should encode and decode ydke URL', () => {
    const deck = new YGOProDeck();
    deck.main.push(111111);
    deck.main.push(111111);
    deck.main.push(111111);
    deck.main.push(222222);
    deck.main.push(333333);
    deck.extra.push(444444);
    deck.extra.push(555555);
    deck.side.push(666666);

    const uri = deck.toYdkeURL();
    expect(uri.startsWith('ydke://')).toBe(true);

    const decoded = new YGOProDeck().fromYdkeURL(uri);

    expect(decoded.main).toStrictEqual(deck.main);
    expect(decoded.extra).toStrictEqual(deck.extra);
    expect(decoded.side).toStrictEqual(deck.side);
  });

  it('should encode and decode update deck payload', () => {
    const uri =
      'http://deck.ourygo.top?name=%E7%99%BD%E9%93%B6%E5%9F%8E%E7%A0%81&ygotype=deck&v=1&d=J-xK4Mka02AuEAMf2dV6mj7aRemuJNQJK8BwrcEYh0MqOJqBJSgYqUwxPEVhOG8R18WWVKmzkT-xEdwxbmGVkkOdrVIpufaYI3Hs8oOrcya8Bi40h9G79iFW80rq-o6P-AHsusPY5nmvHLol0DIqEykESVlf6VSbxVJp-j7XZtTE0XvmJW80rqH28R4rgyRovOusVJzbutenYFBA_cyK6d3UWcQkJQlLjaroWavH-INFA56k5DQNWOQ1gpvxrKVBLgEk1olpolKmSgriramLlgtBK1EQ6C6oi94ZyHe7N6T7mqE6peds7mahrORP6A';
    const deck = new YGOProDeck().fromYGOMobileDeckURL(uri);
    const updateDeckPayload = deck.toUpdateDeckPayload();
    const bufferReader = new BufferReader(updateDeckPayload);
    const mainc = bufferReader.readUint32LE();
    const sidec = bufferReader.readUint32LE();
    expect(mainc).toBe(58 + 15);
    expect(sidec).toBe(15);
    const deckAgain = new YGOProDeck().fromUpdateDeckPayload(
      updateDeckPayload,
      (id, i, mainc) => i >= 58,
    );
    expect(deckAgain.main).toHaveLength(58);
    expect(deckAgain.extra).toHaveLength(15);
    expect(deckAgain.side).toHaveLength(15);
    expect(deckAgain.main).toStrictEqual(deck.main);
    expect(deckAgain.extra).toStrictEqual(deck.extra);
    expect(deckAgain.side).toStrictEqual(deck.side);
  });

  it('should compare decks correctly', () => {
    const a = new YGOProDeck({
      main: [1, 2, 3, 4, 5],
      extra: [11, 12, 13],
      side: [21, 22],
    });
    const b = new YGOProDeck({
      // different order
      main: [5, 4, 3, 2, 1],
      extra: [13, 12, 11],
      side: [22, 21],
    });
    const c = new YGOProDeck({
      // different content
      main: [1, 2, 3, 4, 6],
      extra: [11, 12, 13],
      side: [21, 22],
    });
    const aa = new YGOProDeck(a);
    expect(a.isEqual(aa)).toBe(true);
    expect(a.isEqual(b)).toBe(false);
    expect(a.isEqual(b, { ignoreOrder: true })).toBe(true);
    expect(a.isEqual(c)).toBe(false);
  });
});
