// ydke.ts

export interface TypedDeck {
  main: number[];
  extra: number[];
  side: number[];
}

function base64ToUint32Array(base64: string): number[] {
  const binary = atob(base64);
  const bytes = new Uint8Array(binary.length);
  for (let i = 0; i < binary.length; i++) {
    bytes[i] = binary.charCodeAt(i);
  }
  const view = new DataView(bytes.buffer);
  const result: number[] = [];
  for (let i = 0; i < bytes.length; i += 4) {
    result.push(view.getUint32(i, true)); // little-endian
  }
  return result;
}

function uint32ArrayToBase64(data: number[]): string {
  const buffer = new ArrayBuffer(data.length * 4);
  const view = new DataView(buffer);
  data.forEach((val, i) => view.setUint32(i * 4, val, true));
  const bytes = new Uint8Array(buffer);
  let binary = '';
  for (let i = 0; i < bytes.length; i++) {
    binary += String.fromCharCode(bytes[i]);
  }
  return btoa(binary);
}

export function toYdkeURL(deck: TypedDeck): string {
  return (
    'ydke://' +
    uint32ArrayToBase64(deck.main) +
    '!' +
    uint32ArrayToBase64(deck.extra) +
    '!' +
    uint32ArrayToBase64(deck.side) +
    '!'
  );
}

export function fromYdkeURL(ydke: string): TypedDeck {
  if (!ydke.startsWith('ydke://')) {
    throw new Error('Invalid ydke:// URI');
  }
  const [mainStr, extraStr, sideStr] = ydke.slice(7).split('!');
  if (mainStr === undefined || extraStr === undefined || sideStr === undefined) {
    throw new Error('Incomplete ydke:// URI');
  }
  return {
    main: base64ToUint32Array(mainStr),
    extra: base64ToUint32Array(extraStr),
    side: base64ToUint32Array(sideStr),
  };
}
