import { fromBase64Url, toBase64Url } from "./base64";

// === 常量 ===
const QUERY_YGO_TYPE = 'ygotype';
const QUERY_VERSION = 'v';
const ARG_DECK = 'deck';
const QUERY_DECK = 'd';
const QUERY_NAME = 'name';
const URL_SCHEME_HTTP = 'http';
const URL_HOST_DECK = 'deck.ourygo.top';

// === BitWriter 类 ===
class BitWriter {
  private buffer: number[] = [];
  private current = 0;
  private bitPos = 0;

  writeBits(value: number, length: number) {
    while (length > 0) {
      const remain = 8 - this.bitPos;
      const take = Math.min(remain, length);
      const shift = length - take;
      this.current |= ((value >>> shift) & ((1 << take) - 1)) << (remain - take);
      this.bitPos += take;
      length -= take;

      if (this.bitPos === 8) {
        this.buffer.push(this.current);
        this.current = 0;
        this.bitPos = 0;
      }
    }
  }

  finish(): Uint8Array {
    if (this.bitPos > 0) this.buffer.push(this.current);
    return new Uint8Array(this.buffer);
  }
}

// === BitReader 类 ===
class BitReader {
  private index = 0;
  private bitPos = 0;

  constructor(private bytes: Uint8Array) {}

  readBits(length: number): number {
    let result = 0;
    while (length > 0) {
      const remain = 8 - this.bitPos;
      const take = Math.min(remain, length);
      const bits = (this.bytes[this.index] >>> (remain - take)) & ((1 << take) - 1);
      result = (result << take) | bits;
      this.bitPos += take;
      length -= take;

      if (this.bitPos === 8) {
        this.bitPos = 0;
        this.index++;
      }
    }
    return result;
  }
}

// === 工具函数 ===
function countUnique(cards: number[]): number {
  let count = 0;
  for (let i = 0; i < cards.length; ) {
    const id = cards[i];
    count++;
    while (i < cards.length && cards[i] === id) i++;
  }
  return count;
}

// === 主函数：编码 ===
export function toYGOMobileDeckURL(
  main: number[],
  extra: number[],
  side: number[],
  customParams: Record<string, string> = {},
): string {
  const mNum = countUnique(main);
  const eNum = countUnique(extra);
  const sNum = countUnique(side);

  const writer = new BitWriter();
  writer.writeBits(mNum, 8);
  writer.writeBits(eNum, 4);
  writer.writeBits(sNum, 4);

  const encodeSection = (cards: number[]) => {
    for (let i = 0; i < cards.length; ) {
      const id = cards[i];
      let count = 1;
      while (i + count < cards.length && cards[i + count] === id && count < 3) count++;
      const prefix = count === 2 ? 0b10 : count === 3 ? 0b11 : 0b01;
      writer.writeBits(prefix, 2);
      writer.writeBits(id, 27);
      i += count;
    }
  };

  encodeSection(main);
  encodeSection(extra);
  encodeSection(side);

  const encoded = toBase64Url(writer.finish());

  const searchParams = new URLSearchParams();
  for (const [k, v] of Object.entries(customParams)) {
    searchParams.set(k, v);
  }
  searchParams.set(QUERY_YGO_TYPE, ARG_DECK);
  searchParams.set(QUERY_VERSION, '1');
  searchParams.set(QUERY_DECK, encoded);

  return `${URL_SCHEME_HTTP}://${URL_HOST_DECK}?${searchParams.toString()}`;
}

// === 主函数：解码 ===
export function fromYGOMobileDeckURL(uri: string): {
  main: number[];
  extra: number[];
  side: number[];
  name?: string;
} {
  const url = new URL(uri);
  if (url.searchParams.get(QUERY_YGO_TYPE) !== ARG_DECK) {
    throw new Error('Invalid deck URL');
  }

  const encoded = url.searchParams.get(QUERY_DECK);
  if (!encoded) throw new Error('Missing deck data');

  const bytes = fromBase64Url(encoded);
  const reader = new BitReader(bytes);

  const mNum = reader.readBits(8);
  const eNum = reader.readBits(4);
  const sNum = reader.readBits(4);
  const total = mNum + eNum + sNum;

  const result = {
    main: [] as number[],
    extra: [] as number[],
    side: [] as number[],
  };

  for (let i = 0; i < total; i++) {
    const prefix = reader.readBits(2);
    const count = prefix === 0b10 ? 2 : prefix === 0b11 ? 3 : 1;
    const id = reader.readBits(27);
    const target =
      i < mNum ? result.main : i < mNum + eNum ? result.extra : result.side;
    for (let j = 0; j < count; j++) {
      target.push(id);
    }
  }

  return {
    ...result,
    name: url.searchParams.get(QUERY_NAME) ?? undefined,
  };
}
