export function countItems<T>(arr: T[]) {
  const map = new Map<T, number>();
  for (const item of arr) {
    map.set(item, (map.get(item) || 0) + 1);
  }
  const blocks: { item: T, count: number }[] = [];
  for (const [item, count] of map.entries()) {
    blocks.push({ item, count });
  }
  return blocks;
}

export function countConsecutiveItems<T>(arr: T[]) {
  const result: { item: T; count: number }[] = [];

  if (arr.length === 0) return result;

  let currentItem = arr[0];
  let count = 1;

  for (let i = 1; i < arr.length; i++) {
    if (arr[i] === currentItem) {
      count++;
    } else {
      result.push({ item: currentItem, count });
      currentItem = arr[i];
      count = 1;
    }
  }

  // 推入最后一组
  result.push({ item: currentItem, count });

  return result;
}

abstract class BufferCursor {
  buffer: Uint8Array;
  pointer = 0;

  constructor(bufOrLength: Uint8Array | number) {
    if (typeof bufOrLength === 'number') {
      this.buffer = new Uint8Array(bufOrLength);
    } else {
      this.buffer = bufOrLength;
    }
  }

  /**
   * 安全地将 pointer 向前移动指定字节数。
   * @param bytes 要移动的字节数
   * @returns 原来 pointer 的值
   * @throws RangeError 如果越界
   */
  protected increasePointer(bytes: number): number {
    const old = this.pointer;
    const next = old + bytes;
    if (next > this.buffer.length) {
      throw new RangeError(
        `Pointer overflow: tried to move to ${next}, but buffer length is ${this.buffer.length}`
      );
    }
    this.pointer = next;
    return old;
  }
}

export class BufferWriter extends BufferCursor {
  constructor(length: number) {
    super(length);
  }

  writeUint32LE(value: number) {
    const idx = this.increasePointer(4);
    this.buffer[idx    ] =  value         & 0xff;
    this.buffer[idx + 1] = (value >>> 8)  & 0xff;
    this.buffer[idx + 2] = (value >>> 16) & 0xff;
    this.buffer[idx + 3] = (value >>> 24) & 0xff;
    return this;
  }

  writeUint32BE(value: number) {
    const idx = this.increasePointer(4);
    this.buffer[idx    ] = (value >>> 24) & 0xff;
    this.buffer[idx + 1] = (value >>> 16) & 0xff;
    this.buffer[idx + 2] = (value >>> 8)  & 0xff;
    this.buffer[idx + 3] =  value         & 0xff;
    return this;
  }

  writeUint16LE(value: number) {
    const idx = this.increasePointer(2);
    this.buffer[idx    ] =  value        & 0xff;
    this.buffer[idx + 1] = (value >>> 8) & 0xff;
    return this;
  }

  writeUint16BE(value: number) {
    const idx = this.increasePointer(2);
    this.buffer[idx    ] = (value >>> 8) & 0xff;
    this.buffer[idx + 1] =  value        & 0xff;
    return this;
  }

  writeUint8(value: number) {
    const idx = this.increasePointer(1);
    this.buffer[idx] = value & 0xff;
    return this;
  }

  writeString(str: string) {
    for (let i = 0; i < str.length; i++) {
      const idx = this.increasePointer(1);
      this.buffer[idx] = str.charCodeAt(i);
    }
    return this;
  }
}

export class BufferReader extends BufferCursor {
  constructor(buf: Uint8Array) {
    super(buf);
  }

  readUint32LE(): number {
    const idx = this.increasePointer(4);
    const b = this.buffer;
    // >>>0 保证无符号输出
    return (
      (b[idx    ]      ) |
      (b[idx + 1] <<  8) |
      (b[idx + 2] << 16) |
      (b[idx + 3] << 24)
    ) >>> 0;
  }

  readUint32BE(): number {
    const idx = this.increasePointer(4);
    const b = this.buffer;
    return (
      (b[idx    ] << 24) |
      (b[idx + 1] << 16) |
      (b[idx + 2] <<  8) |
      (b[idx + 3]      )
    ) >>> 0;
  }

  readUint16LE(): number {
    const idx = this.increasePointer(2);
    const b = this.buffer;
    return (b[idx] | (b[idx + 1] << 8));
  }

  readUint16BE(): number {
    const idx = this.increasePointer(2);
    const b = this.buffer;
    return ((b[idx] << 8) | b[idx + 1]);
  }

  readUint8(): number {
    const idx = this.increasePointer(1);
    return this.buffer[idx];
  }

  readString(length: number): string {
    let s = '';
    for (let i = 0; i < length; i++) {
      const idx = this.increasePointer(1);
      s += String.fromCharCode(this.buffer[idx]);
    }
    return s;
  }

  readRemaining(): Uint8Array {
    const remaining = this.buffer.length - this.pointer;
    const idx = this.increasePointer(remaining);
    return this.buffer.subarray(idx, this.buffer.length);
  }
}
