import { CommonModule } from '@angular/common';
import {
  ChangeDetectionStrategy,
  Component,
  computed,
  signal,
} from '@angular/core';
import JSZip from 'jszip';
import initSqlJs, { type Database, type SqlJsStatic } from 'sql.js';
import {
  OcgcoreMessageType,
  ZipScriptReader,
  createOcgcoreWrapper,
  SqljsCardReader,
  testCard,
  type ScriptReader,
  type TestCardMessage,
} from 'koishipro-core.js';
import { YGOPRO_CDB_URL, YGOPRO_SCRIPT_ZIP_URL } from './resource-urls';
import { OcgcoreCommonConstants } from 'ygopro-msg-encode';
import { YGOProCdb } from 'ygopro-cdb-encode';

type PackageStatus = 'pending' | 'running' | 'done' | 'error';
type CdbStatus = 'pending' | 'running' | 'done' | 'error';
type CardStatus = 'pending' | 'running' | 'passed' | 'failed' | 'error';
type FilterOption = 'all' | 'passed' | 'failed' | 'skip' | 'empty';
type DisplayStatus =
  | 'pending'
  | 'running'
  | 'normal'
  | 'abnormal'
  | 'error'
  | 'empty';

type CardTestResult = {
  id: number;
  name: string;
  status: CardStatus;
  scriptErrors: string[];
  logs: TestCardMessage[];
  skipTest: boolean;
  detail?: string;
};

type CdbTestResult = {
  fileName: string;
  status: CdbStatus;
  cardFilter: FilterOption;
  cards: CardTestResult[];
  totalTestableCards: number;
  totalCards: number;
  testedCards: number;
  passedCards: number;
  failedCards: number;
  error?: string;
};

type PackageTestResult = {
  fileName: string;
  status: PackageStatus;
  cdbFilter: FilterOption;
  cdbs: CdbTestResult[];
  error?: string;
};

type ProgressSnapshot = {
  total: number;
  done: number;
  failed: number;
  percent: number;
};

type ZipCdbEntry = {
  zipName: string;
  displayName: string;
};

@Component({
  selector: 'app-root',
  imports: [CommonModule],
  templateUrl: './app.html',
  styleUrl: './app.css',
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class App {
  protected readonly title = 'YGOPro 脚本测试器';
  protected readonly messageType = OcgcoreMessageType;
  protected readonly selectedFiles = signal<File[]>([]);
  protected readonly reports = signal<PackageTestResult[]>([]);
  protected readonly running = signal(false);
  protected readonly statusMessage = signal('等待上传扩展包。');
  protected readonly globalError = signal<string | null>(null);

  protected readonly canStart = computed(
    () => this.selectedFiles().length > 0 && !this.running(),
  );

  protected readonly progress = computed<ProgressSnapshot>(() => {
    let total = 0;
    let done = 0;
    let failed = 0;
    for (const report of this.reports()) {
      for (const cdb of report.cdbs) {
        total += cdb.totalTestableCards;
        done += cdb.testedCards;
        failed += cdb.failedCards;
      }
    }
    const percent = total === 0 ? 0 : Math.round((done / total) * 100);
    return { total, done, failed, percent };
  });

  protected readonly progressValueText = computed(() => {
    const snapshot = this.progress();
    if (snapshot.total === 0) {
      return '尚未开始';
    }
    return `已完成 ${snapshot.done} / ${snapshot.total}`;
  });

  protected readonly hasResults = computed(() => this.reports().length > 0);
  protected readonly zipFilter = signal<FilterOption>('all');
  protected readonly visibleReports = computed(() =>
    this.reports().filter((report) =>
      this.matchesFilter(
        this.packageDisplayStatus(report),
        this.resolveFilter(this.zipFilter(), this.zipFilterOptions()),
      ),
    ),
  );

  protected readonly overallStats = computed(() => {
    const reports = this.reports();
    const zipTotal = reports.length;
    let zipPassed = 0;
    let zipFailed = 0;
    let cdbTotal = 0;
    let cdbPassed = 0;
    let cdbFailed = 0;
    let cardTotal = 0;
    let cardPassed = 0;
    let cardFailed = 0;

    for (const report of reports) {
      const zipStatus = this.packageDisplayStatus(report);
      if (zipStatus === 'normal') {
        zipPassed += 1;
      } else if (zipStatus === 'abnormal' || zipStatus === 'error') {
        zipFailed += 1;
      }
      for (const cdb of report.cdbs) {
        cdbTotal += 1;
        const cdbStatus = this.cdbDisplayStatus(cdb);
        if (cdbStatus === 'normal') {
          cdbPassed += 1;
        } else if (cdbStatus === 'abnormal' || cdbStatus === 'error') {
          cdbFailed += 1;
        }
        cardTotal += cdb.totalTestableCards;
        cardPassed += cdb.passedCards;
        cardFailed += cdb.failedCards;
      }
    }

    return {
      zipTotal,
      zipPassed,
      zipFailed,
      cdbTotal,
      cdbPassed,
      cdbFailed,
      cardTotal,
      cardPassed,
      cardFailed,
    };
  });

  private sqlModule: SqlJsStatic | null = null;
  private baseDb: Database | null = null;
  private baseScriptReader: ScriptReader | null = null;
  private ocgcoreWasmBinary: Uint8Array | null = null;

  protected onFileInputClick(event: Event): void {
    const target = event.target as HTMLInputElement | null;
    if (target) {
      target.value = '';
    }
  }

  protected onFileSelection(event: Event): void {
    const target = event.target as HTMLInputElement | null;
    const files = target?.files ? Array.from(target.files) : [];
    this.selectedFiles.set(files);
    this.reports.set([]);
    this.globalError.set(null);
    this.zipFilter.set('all');
    if (files.length === 0) {
      this.statusMessage.set('等待上传扩展包。');
    } else {
      this.statusMessage.set(`已选择 ${files.length} 个扩展包，开始测试。`);
      void this.startTests();
    }
  }

  protected clearSelection(): void {
    this.selectedFiles.set([]);
    this.reports.set([]);
    this.globalError.set(null);
    this.zipFilter.set('all');
    this.statusMessage.set('已清空选择。');
  }

  protected async startTests(): Promise<void> {
    if (!this.canStart()) {
      return;
    }

    const files = this.selectedFiles();
    this.running.set(true);
    this.globalError.set(null);
    this.reports.set(
      files.map((file) => ({
        fileName: file.name,
        status: 'pending',
        cdbFilter: 'all',
        cdbs: [],
      })),
    );

    try {
      this.statusMessage.set('正在准备基础资源...');
      await this.ensureBaseResources();

      for (let index = 0; index < files.length; index += 1) {
        await this.testPackage(files[index], index);
      }

      this.statusMessage.set('全部测试完成。');
    } catch (error) {
      this.globalError.set(this.formatError(error));
      this.statusMessage.set('基础资源加载失败，测试已中止。');
      this.reports.update((reports) =>
        reports.map((report) => ({
          ...report,
          status: 'error',
          error: '基础资源加载失败',
        })),
      );
    } finally {
      this.running.set(false);
    }
  }

  protected packageBadgeClass(report: PackageTestResult): string {
    switch (this.packageDisplayStatus(report)) {
      case 'running':
        return 'badge text-bg-primary';
      case 'normal':
        return 'badge text-bg-success';
      case 'abnormal':
        return 'badge text-bg-warning';
      case 'empty':
        return 'badge text-bg-secondary';
      case 'error':
        return 'badge text-bg-danger';
      default:
        return 'badge text-bg-secondary';
    }
  }

  protected packageStatusLabel(report: PackageTestResult): string {
    switch (this.packageDisplayStatus(report)) {
      case 'running':
        return '测试中';
      case 'normal':
        return '正常';
      case 'abnormal':
        return '异常';
      case 'empty':
        return '无内容';
      case 'error':
        return '出错';
      default:
        return '等待';
    }
  }

  protected cdbBadgeClass(cdb: CdbTestResult): string {
    switch (this.cdbDisplayStatus(cdb)) {
      case 'running':
        return 'badge text-bg-primary';
      case 'normal':
        return 'badge text-bg-success';
      case 'abnormal':
        return 'badge text-bg-warning';
      case 'empty':
        return 'badge text-bg-secondary';
      case 'error':
        return 'badge text-bg-danger';
      default:
        return 'badge text-bg-secondary';
    }
  }

  protected cdbStatusLabel(cdb: CdbTestResult): string {
    switch (this.cdbDisplayStatus(cdb)) {
      case 'running':
        return '测试中';
      case 'normal':
        return '正常';
      case 'abnormal':
        return '异常';
      case 'empty':
        return '无内容';
      case 'error':
        return '出错';
      default:
        return '等待';
    }
  }

  protected cardBadgeClass(status: CardStatus): string {
    switch (status) {
      case 'running':
        return 'badge text-bg-primary';
      case 'passed':
        return 'badge text-bg-success';
      case 'failed':
      case 'error':
        return 'badge text-bg-danger';
      default:
        return 'badge text-bg-secondary';
    }
  }

  protected cardStatusLabel(status: CardStatus): string {
    switch (status) {
      case 'running':
        return '测试中';
      case 'passed':
        return '通过';
      case 'failed':
        return '失败';
      case 'error':
        return '出错';
      default:
        return '等待';
    }
  }

  protected cardBadgeClassForCard(card: CardTestResult): string {
    if (card.skipTest) {
      return 'badge text-bg-secondary';
    }
    return this.cardBadgeClass(card.status);
  }

  protected cardStatusLabelForCard(card: CardTestResult): string {
    if (card.skipTest) {
      return '无需测试';
    }
    return this.cardStatusLabel(card.status);
  }

  protected visibleCdbs(report: PackageTestResult): CdbTestResult[] {
    const filter = this.resolveFilter(report.cdbFilter, this.cdbFilterOptions(report));
    return report.cdbs.filter((cdb) =>
      this.matchesFilter(this.cdbDisplayStatus(cdb), filter),
    );
  }

  protected visibleCards(cdb: CdbTestResult): CardTestResult[] {
    const filter = this.resolveFilter(cdb.cardFilter, this.cardFilterOptions(cdb));
    return cdb.cards.filter((card) =>
      this.matchesFilter(this.cardDisplayStatus(card), filter),
    );
  }

  protected zipFilterOptions(): FilterOption[] {
    return this.filterOptionsForStatuses(
      this.reports().map((report) => this.packageDisplayStatus(report)),
    );
  }

  protected cdbFilterOptions(report: PackageTestResult): FilterOption[] {
    return this.filterOptionsForStatuses(
      report.cdbs.map((cdb) => this.cdbDisplayStatus(cdb)),
    );
  }

  protected cardFilterOptions(cdb: CdbTestResult): FilterOption[] {
    return this.filterOptionsForStatuses(
      cdb.cards.map((card) => this.cardDisplayStatus(card)),
    );
  }

  protected onZipFilterChange(event: Event): void {
    const target = event.target as HTMLSelectElement | null;
    const value = target?.value ?? 'all';
    if (!this.isFilterOption(value)) {
      return;
    }
    this.zipFilter.set(value);
  }

  protected onPackageFilterChange(
    event: Event,
    report: PackageTestResult,
  ): void {
    const target = event.target as HTMLSelectElement | null;
    const value = target?.value ?? 'all';
    if (!this.isFilterOption(value)) {
      return;
    }
    const packageIndex = this.packageIndexOf(report);
    if (packageIndex < 0) {
      return;
    }
    this.updatePackage(packageIndex, (current) => ({
      ...current,
      cdbFilter: value,
    }));
  }

  protected onCdbFilterChange(
    event: Event,
    report: PackageTestResult,
    cdb: CdbTestResult,
  ): void {
    const target = event.target as HTMLSelectElement | null;
    const value = target?.value ?? 'all';
    if (!this.isFilterOption(value)) {
      return;
    }
    const packageIndex = this.packageIndexOf(report);
    if (packageIndex < 0) {
      return;
    }
    const cdbIndex = this.cdbIndexOf(report, cdb);
    if (cdbIndex < 0) {
      return;
    }
    this.updateCdb(packageIndex, cdbIndex, (current) => ({
      ...current,
      cardFilter: value,
    }));
  }

  protected packageIndexOf(report: PackageTestResult): number {
    return this.reports().indexOf(report);
  }

  protected cdbIndexOf(report: PackageTestResult, cdb: CdbTestResult): number {
    return report.cdbs.indexOf(cdb);
  }

  protected formatFileSize(bytes: number): string {
    if (bytes >= 1024 * 1024) {
      return `${(bytes / (1024 * 1024)).toFixed(1)} MB`;
    }
    if (bytes >= 1024) {
      return `${Math.round(bytes / 1024)} KB`;
    }
    return `${bytes} bytes`;
  }

  protected exportPdf(): void {
    if (typeof window !== 'undefined') {
      window.print();
    }
  }

  protected packageCardSummary(report: PackageTestResult): string {
    let total = 0;
    let passed = 0;
    let failed = 0;
    for (const cdb of report.cdbs) {
      total += cdb.totalTestableCards;
      passed += cdb.passedCards;
      failed += cdb.failedCards;
    }
    if (total === 0) {
      return '没有需要测试的卡片';
    }
    return `卡片 ${total}，通过 ${passed}，失败 ${failed}`;
  }

  protected packageCdbSummary(report: PackageTestResult): string {
    if (report.cdbs.length === 0) {
      return 'CDB 0';
    }
    let passed = 0;
    let failed = 0;
    for (const cdb of report.cdbs) {
      const status = this.cdbDisplayStatus(cdb);
      if (status === 'normal') {
        passed += 1;
      } else if (status === 'abnormal' || status === 'error') {
        failed += 1;
      }
    }
    return `CDB ${report.cdbs.length}，正常 ${passed}，异常 ${failed}`;
  }

  private async testPackage(file: File, packageIndex: number): Promise<void> {
    this.updatePackage(packageIndex, (report) => ({
      ...report,
      status: 'running',
      error: undefined,
    }));

    try {
      this.statusMessage.set(`正在读取扩展包 ${file.name}...`);
      const buffer = await file.arrayBuffer();
      const zip = await JSZip.loadAsync(buffer);
      const cdbEntries = this.extractRootCdbEntries(zip);
      const cdbReports = cdbEntries.map((entry) =>
        this.createCdbReport(entry.displayName),
      );

      this.updatePackage(packageIndex, (report) => ({
        ...report,
        cdbs: cdbReports,
      }));

      if (cdbEntries.length === 0) {
        this.updatePackage(packageIndex, (report) => ({
          ...report,
          status: 'done',
        }));
        return;
      }

      const baseDb = await this.getBaseDatabase();
      const baseScriptReader = await this.getBaseScriptReader();
      const wasmBinary = await this.getOcgcoreWasmBinary();
      const packageScriptReader = await ZipScriptReader(buffer);

      for (let cdbIndex = 0; cdbIndex < cdbEntries.length; cdbIndex += 1) {
        await this.testCdb({
          packageIndex,
          cdbIndex,
          entry: cdbEntries[cdbIndex],
          zip,
          baseDb,
          baseScriptReader,
          packageScriptReader,
          wasmBinary,
        });
      }

      const snapshot = this.reports()[packageIndex];
      const hasError =
        snapshot?.cdbs.some((cdb) => cdb.status === 'error') ?? false;
      this.updatePackage(packageIndex, (report) => ({
        ...report,
        status: hasError ? 'error' : 'done',
      }));
    } catch (error) {
      this.updatePackage(packageIndex, (report) => ({
        ...report,
        status: 'error',
        error: this.formatError(error),
      }));
    }
  }

  private async testCdb(args: {
    packageIndex: number;
    cdbIndex: number;
    entry: ZipCdbEntry;
    zip: JSZip;
    baseDb: Database;
    baseScriptReader: ScriptReader;
    packageScriptReader: ScriptReader;
    wasmBinary: Uint8Array;
  }): Promise<void> {
    const {
      packageIndex,
      cdbIndex,
      entry,
      zip,
      baseDb,
      baseScriptReader,
      packageScriptReader,
      wasmBinary,
    } = args;

    this.updateCdb(packageIndex, cdbIndex, (cdb) => ({
      ...cdb,
      status: 'running',
      error: undefined,
    }));

    let db: Database | null = null;

    try {
      this.statusMessage.set(`正在解析 ${entry.displayName}...`);
      const zipEntry = zip.file(entry.zipName);
      if (!zipEntry) {
        throw new Error(`未找到 ${entry.displayName}`);
      }
      const sql = await this.getSqlModule();
      const cdbBytes = await zipEntry.async('uint8array');
      db = new sql.Database(cdbBytes);
      const cdb = new YGOProCdb(db);
      const cards = this.queryTestCards(cdb);

      if (cards.length === 0) {
        this.updateCdb(packageIndex, cdbIndex, (cdb) => ({
          ...cdb,
          status: 'done',
          totalTestableCards: 0,
          totalCards: 0,
          testedCards: 0,
          passedCards: 0,
          failedCards: 0,
          cards: [],
        }));
        return;
      }

      const cardResults: CardTestResult[] = cards.map((card) => ({
        id: card.id,
        name: card.name,
        status: 'pending',
        scriptErrors: [],
        logs: [],
        skipTest: card.skipTest,
      }));
      const testableCards = cardResults.filter((card) => !card.skipTest);

      this.updateCdb(packageIndex, cdbIndex, (cdb) => ({
        ...cdb,
        cards: cardResults,
        totalCards: cardResults.length,
        totalTestableCards: testableCards.length,
        testedCards: 0,
        passedCards: 0,
        failedCards: 0,
      }));

      for (let cardIndex = 0; cardIndex < cardResults.length; cardIndex += 1) {
        const card = cardResults[cardIndex];
        if (card.skipTest) {
          continue;
        }
        this.statusMessage.set(`正在测试 ${entry.displayName} - ${card.name}`);
        this.updateCard(packageIndex, cdbIndex, cardIndex, (item) => ({
          ...item,
          status: 'running',
        }));

        const result = await this.runCardTest({
          cardId: card.id,
          cardName: card.name,
          cdbDb: db,
          baseDb,
          packageScriptReader,
          baseScriptReader,
          wasmBinary,
        });

        this.updateCard(packageIndex, cdbIndex, cardIndex, () => result);
        await this.yieldToUi();
      }

      this.updateCdb(packageIndex, cdbIndex, (cdb) => ({
        ...cdb,
        status: 'done',
      }));
    } catch (error) {
      this.updateCdb(packageIndex, cdbIndex, (cdb) => ({
        ...cdb,
        status: 'error',
        error: this.formatError(error),
      }));
    } finally {
      if (db) {
        db.close();
      }
    }
  }

  private extractRootCdbEntries(zip: JSZip): ZipCdbEntry[] {
    const entries: ZipCdbEntry[] = [];
    for (const entry of Object.values(zip.files)) {
      if (entry.dir) {
        continue;
      }
      const normalized = this.normalizeZipPath(entry.name);
      if (!normalized.toLowerCase().endsWith('.cdb')) {
        continue;
      }
      if (normalized.includes('/')) {
        continue;
      }
      entries.push({ zipName: entry.name, displayName: normalized });
    }
    return entries;
  }

  private normalizeZipPath(name: string): string {
    return name.replace(/\\/g, '/').replace(/^\.\//, '').replace(/^\//, '');
  }

  private createCdbReport(fileName: string): CdbTestResult {
    return {
      fileName,
      status: 'pending',
      cardFilter: 'all',
      cards: [],
      totalTestableCards: 0,
      totalCards: 0,
      testedCards: 0,
      passedCards: 0,
      failedCards: 0,
    };
  }

  private async ensureBaseResources(): Promise<void> {
    await this.getSqlModule();
    await this.getBaseDatabase();
    await this.getBaseScriptReader();
    await this.getOcgcoreWasmBinary();
  }

  private async getSqlModule(): Promise<SqlJsStatic> {
    if (this.sqlModule) {
      return this.sqlModule;
    }
    this.sqlModule = await initSqlJs({
      locateFile: (file) => `assets/sql.js/${file}`,
    });
    return this.sqlModule;
  }

  private async getBaseDatabase(): Promise<Database> {
    if (this.baseDb) {
      return this.baseDb;
    }
    const sql = await this.getSqlModule();
    const response = await fetch(YGOPRO_CDB_URL);
    if (!response.ok) {
      throw new Error(`无法加载基础卡片数据库：${response.status}`);
    }
    const buffer = await response.arrayBuffer();
    this.baseDb = new sql.Database(new Uint8Array(buffer));
    return this.baseDb;
  }

  private async getBaseScriptReader(): Promise<ScriptReader> {
    if (this.baseScriptReader) {
      return this.baseScriptReader;
    }
    const response = await fetch(YGOPRO_SCRIPT_ZIP_URL);
    if (!response.ok) {
      throw new Error(`无法加载基础脚本包：${response.status}`);
    }
    const buffer = await response.arrayBuffer();
    this.baseScriptReader = await ZipScriptReader(buffer);
    return this.baseScriptReader;
  }

  private async getOcgcoreWasmBinary(): Promise<Uint8Array> {
    if (this.ocgcoreWasmBinary) {
      return this.ocgcoreWasmBinary;
    }
    const response = await fetch('assets/ocgcore/libocgcore.wasm');
    if (!response.ok) {
      throw new Error(`无法加载 ocgcore wasm：${response.status}`);
    }
    const buffer = await response.arrayBuffer();
    this.ocgcoreWasmBinary = new Uint8Array(buffer);
    return this.ocgcoreWasmBinary;
  }

  private async runCardTest(args: {
    cardId: number;
    cardName: string;
    cdbDb: Database;
    baseDb: Database;
    packageScriptReader: ScriptReader;
    baseScriptReader: ScriptReader;
    wasmBinary: Uint8Array;
  }): Promise<CardTestResult> {
    const {
      cardId,
      cardName,
      cdbDb,
      baseDb,
      packageScriptReader,
      baseScriptReader,
      wasmBinary,
    } = args;
    const wrapper = await createOcgcoreWrapper({ wasmBinary });
    try {
      const cardReader = SqljsCardReader(cdbDb, baseDb);
      wrapper.setCardReader(cardReader, true);
      wrapper.setScriptReader(packageScriptReader, true);
      wrapper.setScriptReader(baseScriptReader);

      const logs = testCard(wrapper, cardId);
      const scriptErrors = logs
        .filter((entry) => entry.type === OcgcoreMessageType.ScriptError)
        .map((entry) => entry.message);

      return {
        id: cardId,
        name: cardName,
        status: scriptErrors.length === 0 ? 'passed' : 'failed',
        scriptErrors,
        logs,
        skipTest: false,
      };
    } catch (error) {
      return {
        id: cardId,
        name: cardName,
        status: 'error',
        scriptErrors: [],
        logs: [],
        skipTest: false,
        detail: this.formatErrorStack(error),
      };
    } finally {
      wrapper.finalize();
    }
  }

  private queryTestCards(cdb: YGOProCdb): Array<{ id: number; name: string; skipTest: boolean }> {
    const normalMonsterType =
      (OcgcoreCommonConstants.TYPE_NORMAL | OcgcoreCommonConstants.TYPE_MONSTER) >>> 0;
    const tokenType = OcgcoreCommonConstants.TYPE_TOKEN >>> 0;

    const entries = cdb.find();
    const result: Array<{ id: number; name: string; skipTest: boolean }> = [];
    for (const entry of entries) {
      const id = entry.code;
      if (!Number.isFinite(id)) {
        continue;
      }
      const typeValue = (entry.type ?? 0) >>> 0;
      const skipTest =
        typeValue === normalMonsterType || (typeValue & tokenType) !== 0;
      result.push({
        id,
        name: entry.name || '未命名卡片',
        skipTest,
      });
    }
    return result;
  }

  private updatePackage(
    packageIndex: number,
    updater: (report: PackageTestResult) => PackageTestResult,
  ): void {
    this.reports.update((reports) =>
      reports.map((report, index) =>
        index === packageIndex ? updater(report) : report,
      ),
    );
  }

  private updateCdb(
    packageIndex: number,
    cdbIndex: number,
    updater: (cdb: CdbTestResult) => CdbTestResult,
  ): void {
    this.reports.update((reports) =>
      reports.map((report, reportIndex) => {
        if (reportIndex !== packageIndex) {
          return report;
        }
        const cdbs = report.cdbs.map((cdb, index) =>
          index === cdbIndex ? updater(cdb) : cdb,
        );
        return { ...report, cdbs };
      }),
    );
  }

  private updateCard(
    packageIndex: number,
    cdbIndex: number,
    cardIndex: number,
    updater: (card: CardTestResult) => CardTestResult,
  ): void {
    this.reports.update((reports) =>
      reports.map((report, reportIndex) => {
        if (reportIndex !== packageIndex) {
          return report;
        }
        const cdbs = report.cdbs.map((cdb, currentCdbIndex) => {
          if (currentCdbIndex !== cdbIndex) {
            return cdb;
          }
          const oldCard = cdb.cards[cardIndex];
          const newCard = updater(oldCard);
          const cards = cdb.cards.map((card, index) =>
            index === cardIndex ? newCard : card,
          );
          const counters = this.updateCardCounters(cdb, oldCard.status, newCard.status);
          return { ...cdb, cards, ...counters };
        });
        return { ...report, cdbs };
      }),
    );
  }

  private updateCardCounters(
    cdb: CdbTestResult,
    previous: CardStatus,
    next: CardStatus,
  ): Pick<CdbTestResult, 'testedCards' | 'passedCards' | 'failedCards'> {
    let testedCards = cdb.testedCards;
    let passedCards = cdb.passedCards;
    let failedCards = cdb.failedCards;

    if (this.isTested(previous)) {
      testedCards -= 1;
    }
    if (previous === 'passed') {
      passedCards -= 1;
    }
    if (this.isFailed(previous)) {
      failedCards -= 1;
    }

    if (this.isTested(next)) {
      testedCards += 1;
    }
    if (next === 'passed') {
      passedCards += 1;
    }
    if (this.isFailed(next)) {
      failedCards += 1;
    }

    return { testedCards, passedCards, failedCards };
  }

  private isTested(status: CardStatus): boolean {
    return status === 'passed' || status === 'failed' || status === 'error';
  }

  private isFailed(status: CardStatus): boolean {
    return status === 'failed' || status === 'error';
  }

  private cardDisplayStatus(
    card: CardTestResult,
  ): 'passed' | 'failed' | 'pending' | 'skip' {
    if (card.skipTest) {
      return 'skip';
    }
    if (card.status === 'passed') {
      return 'passed';
    }
    if (card.status === 'failed' || card.status === 'error') {
      return 'failed';
    }
    return 'pending';
  }

  private cdbDisplayStatus(cdb: CdbTestResult): DisplayStatus {
    if (cdb.status === 'error') {
      return 'error';
    }
    if (cdb.status === 'running') {
      return 'running';
    }
    if (cdb.totalTestableCards === 0 && cdb.status === 'done') {
      return 'empty';
    }
    if (cdb.failedCards > 0) {
      return 'abnormal';
    }
    if (cdb.totalTestableCards > 0 && cdb.testedCards === cdb.totalTestableCards) {
      return 'normal';
    }
    return 'pending';
  }

  private packageDisplayStatus(report: PackageTestResult): DisplayStatus {
    if (report.status === 'error') {
      return 'error';
    }
    if (report.status === 'running') {
      return 'running';
    }
    if (report.status === 'done' && report.cdbs.length === 0) {
      return 'empty';
    }
    if (
      report.status === 'done' &&
      report.cdbs.length > 0 &&
      report.cdbs.every((cdb) => this.cdbDisplayStatus(cdb) === 'empty')
    ) {
      return 'empty';
    }
    const hasFailed =
      report.cdbs.some((cdb) => this.cdbDisplayStatus(cdb) === 'abnormal') ||
      report.cdbs.some((cdb) => this.cdbDisplayStatus(cdb) === 'error');
    if (hasFailed) {
      return 'abnormal';
    }
    const allDone = report.cdbs.every((cdb) => {
      const status = this.cdbDisplayStatus(cdb);
      return status === 'normal' || status === 'empty' || status === 'error';
    });
    if (allDone && report.cdbs.length > 0) {
      return 'normal';
    }
    return 'pending';
  }

  private matchesFilter(
    status: 'passed' | 'failed' | DisplayStatus | 'skip',
    filter: FilterOption,
  ): boolean {
    if (filter === 'all') {
      return true;
    }
    if (filter === 'passed') {
      return status === 'passed' || status === 'normal';
    }
    if (filter === 'skip') {
      return status === 'skip';
    }
    if (filter === 'empty') {
      return status === 'empty';
    }
    return status === 'failed' || status === 'abnormal' || status === 'error';
  }

  private filterOptionsForStatuses(
    statuses: Array<'passed' | 'failed' | DisplayStatus | 'pending' | 'skip'>,
  ): FilterOption[] {
    let hasPassed = false;
    let hasFailed = false;
    let hasSkip = false;
    let hasEmpty = false;
    for (const status of statuses) {
      if (status === 'passed' || status === 'normal') {
        hasPassed = true;
      } else if (status === 'failed' || status === 'abnormal' || status === 'error') {
        hasFailed = true;
      } else if (status === 'skip') {
        hasSkip = true;
      } else if (status === 'empty') {
        hasEmpty = true;
      }
    }
    const options: FilterOption[] = [];
    if (hasPassed) {
      options.push('passed');
    }
    if (hasFailed) {
      options.push('failed');
    }
    if (hasSkip) {
      options.push('skip');
    }
    if (hasEmpty) {
      options.push('empty');
    }
    return options;
  }

  private resolveFilter(
    current: FilterOption,
    options: FilterOption[],
  ): FilterOption {
    if (options.includes(current)) {
      return current;
    }
    return 'all';
  }

  private isFilterOption(value: string): value is FilterOption {
    return (
      value === 'all' ||
      value === 'passed' ||
      value === 'failed' ||
      value === 'skip' ||
      value === 'empty'
    );
  }

  private formatError(error: unknown): string {
    if (error instanceof Error) {
      return error.message;
    }
    if (typeof error === 'string') {
      return error;
    }
    return '未知错误';
  }

  private formatErrorStack(error: unknown): string {
    if (error instanceof Error) {
      return error.stack ?? error.message;
    }
    if (typeof error === 'string') {
      return error;
    }
    return '未知错误';
  }

  private async yieldToUi(): Promise<void> {
    await new Promise<void>((resolve) => {
      setTimeout(() => resolve(), 0);
    });
  }
}
