package moe.mycard.tabulator.model.service.ds;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import lombok.val;
import moe.mycard.tabulator.exception.CavCore;
import moe.mycard.tabulator.model.dto.GlobalAskBody;
import moe.mycard.tabulator.model.dto.tournament.RTree;
import moe.mycard.tabulator.model.dto.tournament.SeatData;
import moe.mycard.tabulator.model.dto.tournament.SeatNode;
import moe.mycard.tabulator.model.dto.tournament.TTree;
import moe.mycard.tabulator.model.po.ParticipantPO;
import moe.mycard.tabulator.model.po.TournamentPO;
import moe.mycard.tabulator.model.service.IParticipantService;
import moe.mycard.tabulator.model.service.ITournamentService;
import moe.mycard.tabulator.model.vo.req.ImportParticipantPO;
import moe.mycard.tabulator.model.vo.req.TidReq;
import moe.mycard.tabulator.model.vo.req.UpdateSeatNode;
import moe.mycard.tabulator.tool.UtilTime;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
public class TournamentDoService {

  private static final Map<Integer, TTree> tidMapTree = new HashMap<>();

  private static final Map<Integer, Map<Integer, SeatNode>> tidMapSeatIdMapNode = new HashMap<>();

  private final CavCore c = CavCore.getC();
  @Resource private IParticipantService participantService;

  @Resource private ITournamentService tournamentService;

  public static void main(String[] args) {
    RTree rTree = new RTree();
    rTree.setSeatNode(
        new SeatNode() {
          {
            setDepthDesc(1);
          }
        });
    JSONObject jsonObject = (JSONObject) JSONObject.toJSON(rTree);
    System.out.println(jsonObject);
  }

  /**
   * 导入参赛者名录
   *
   * @param body 比赛id and 参赛名录
   * @return 参赛者数量
   */
  @Transactional(rollbackFor = Exception.class)
  public int importParticipant(GlobalAskBody<ImportParticipantPO> body) {
    c.cn(body);

    val directory = body.getParams().getDirectory();
    val tid = body.getParams().getTournamentId();

    c.c(StringUtils.isBlank(directory), "空名录");

    char[] str = directory.toCharArray();

    List<String> participantList = new ArrayList<>();
    List<ParticipantPO> participantPOS = new ArrayList<>();
    val date = UtilTime.getDateL();

    StringBuilder sb = new StringBuilder();
    for (var c : str) {
      if (c == 10 || c == 13) {
        participantList.add(sb.toString());
        sb = new StringBuilder();
      } else {
        sb.append(c);
      }
    }

    participantList.forEach(
        deck -> {
          val e = deck.split("\\+");
          val savePO = new ParticipantPO();
          savePO.setParticipantName(e[0]);
          savePO.setParticipantQQ(e[1]);
          savePO.setParticipantDeck(deck);
          savePO.setTournamentId(tid);
          savePO.setTimeCreate(date);
          savePO.setIsValid(true);
          participantPOS.add(savePO);
        });

    participantService.saveBatch(participantPOS);

    return participantPOS.size();
  }

  /**
   * 制表
   *
   * @param body 比赛id
   * @return 表树
   */
  public TTree tabulation(GlobalAskBody<TidReq> body) {
    c.cn(body);

    val tid = body.getParams().getTournamentId();
    val participantPOS =
        participantService.list(
            new LambdaQueryWrapper<ParticipantPO>()
                .select(ParticipantPO::getParticipantDeck)
                .eq(ParticipantPO::getTournamentId, tid)
                .eq(ParticipantPO::getIsValid, true));
    Collections.shuffle(participantPOS);
    c.cnList(participantPOS, "参赛名录为空");

    val tournamentPO = tournamentService.getById(tid);

    var id = 1;
    var size = participantPOS.size();
    val date = UtilTime.getDateL();
    Map<Integer, List<Integer>> roundMapSeatArray = new HashMap<>();

    if (participantPOS.size() % 2 != 0) {
      size = size + 1;
    }

    val depth = getDepth(size, 0);

    SeatNode[] nodes = new SeatNode[size];
    for (int i = 0; i < size - 1 - 1; i++) {
      val participantPO = participantPOS.get(i);

      SeatData nodeData = new SeatData();
      roundMapSeatArray.computeIfAbsent(tid, k -> new ArrayList<>());
      roundMapSeatArray.get(tid).add(id);
      nodeData.setSeatId(id);
      nodeData.setParticipantId(participantPO.getParticipantId());
      nodeData.setDeck(participantPO.getParticipantDeck());
      nodeData.setLock(false);
      nodeData.setMatchType(0);
      nodeData.setDataType(1);
      nodeData.setTimeCreate(date);
      nodeData.setTimeMatch(null);

      SeatNode node = new SeatNode();
      node.setSeatId(nodeData.getSeatId());
      node.setData(nodeData);
      node.setNext(null);
      node.setPrevious(null);
      node.setDepth(depth);
      node.setDepthDesc(1);

      nodes[i] = node;
      tidMapSeatIdMapNode.computeIfAbsent(tid, k -> new HashMap<>());
      tidMapSeatIdMapNode.get(tid).put(id, node);
      id++;
    }
    nodes[size - 1] = SeatNode.vacancy(id, 2, depth, 1, date);

    val seatTree = leafToTree(id, depth, 2, date, nodes, roundMapSeatArray);

    List<RTree> rTreeList = new ArrayList<>();
    roundMapSeatArray.forEach(
        (kev, value) -> {
          RTree rTree = new RTree();
          rTree.setRound(kev);
          rTree.setSeatNode(seatTree);
          rTree.setSeatIdList(value);
          rTreeList.add(rTree);
        });

    TTree tTree = new TTree();
    tTree.setTournamentSystem(tournamentPO.getTournamentSystem());
    tTree.setTournamentId(tid);
    tTree.setTournamentName(tournamentPO.getTournamentName());
    tTree.setSign("sign");
    tTree.setRoundTreeList(rTreeList);

    val updatePO = new TournamentPO();
    updatePO.setTournamentId(tid);
    updatePO.setTimeUpdate(date);
    updatePO.setTournamentSeat(((JSONObject) JSONObject.toJSON(tTree)).toString());
    updatePO.setTournamentRound(1);
    tournamentService.updateById(updatePO);

    tidMapTree.put(tid, tTree);
    return tTree;
  }

  /**
   * 获取比赛详情
   *
   * @param body 比赛id
   * @return 比赛详情
   */
  public TournamentPO tournamentInfo(GlobalAskBody<TidReq> body) {
    c.cn(body);

    return tournamentService.getById(body.getParams().getTournamentId());
  }

  /*
  ======================================================================================================================
   */

  /**
   * 编辑座位信息
   *
   * @param body 座位信息
   */
  public void editSeat(GlobalAskBody<UpdateSeatNode> body) {
    c.cn(body);

    val seatNode = body.getParams();
    c.cn(seatNode.getTournamentId());
    c.cn(seatNode.getSeatId());

    val seat = tidMapSeatIdMapNode.get(seatNode.getTournamentId()).get(seatNode.getSeatId());

    c.cn(seat);

    if (StringUtils.isNotBlank(seatNode.getDeck())) {
      seat.getData().setDeck(seatNode.getDeck());
    }
    if (seatNode.getMatchFraction() != null) {
      seat.getData().setMatchFraction(seatNode.getMatchFraction());
    }
    if (StringUtils.isNotBlank(seatNode.getMatchRemark())) {
      seat.getData().setMatchRemark(seatNode.getMatchRemark());
    }
  }

  /**
   * 获取树深 64/2 32/2 16/2 8/2 4/2 2/2 1
   *
   * @param size 参赛总数
   */
  private int getDepth(int size, int depth) {
    if (size < 2) {
      return depth + 1;
    } else {
      if (size % 2 != 0) {
        size = size + 1;
      }
      return getDepth(size / 2, depth + 1);
    }
  }

  /**
   * 叶子推算树
   *
   * @param id 座位id
   * @param depth 树深
   * @param depthDesc 树深倒叙
   * @param date 创建时间
   * @param seatNodes 子叶子
   * @param roundMapSeatArray 轮数映射位置集合
   * @return 对局树集合
   */
  private SeatNode leafToTree(
      int id,
      int depth,
      int depthDesc,
      LocalDateTime date,
      SeatNode[] seatNodes,
      Map<Integer, List<Integer>> roundMapSeatArray) {
    boolean combination = true;
    List<SeatNode> curList = new ArrayList<>();

    // ===== 顶层节点 =====
    if (seatNodes.length < 2) {
      val curNode = seatNodes[0];
      val next = SeatNode.emptySeat(id, depth, depthDesc, date, new SeatNode[] {curNode});
      curNode.setNext(new SeatNode[] {next});
      return next;
    }
    // ===== 顶层节点 =====

    // ===== 当前节点集合 =====
    for (int i = 0; i < seatNodes.length - 1; i++) {
      val curNode = seatNodes[i];
      SeatNode next;
      if (combination) {
        next = SeatNode.emptySeat(id++, depth, depthDesc, date, new SeatNode[] {curNode});
        curNode.setNext(new SeatNode[] {next});
        next.setPrevious(new SeatNode[] {curNode});
      } else {
        next = seatNodes[i - 1].getNext()[0];
        curNode.setNext(new SeatNode[] {next});
        next.setPrevious(new SeatNode[] {curNode, next.getPrevious()[0]});
      }
      curList.add(curNode);
      combination = !combination;
    }
    // ===== 当前节点集合 =====

    // ===== 轮空位 =====
    if (!combination) {
      val next = seatNodes[seatNodes.length - 1].getNext()[0];
      val cur = SeatNode.emptySeat(id++, depth, depthDesc, date, null);
      cur.setNext(new SeatNode[] {next});
      next.setPrevious(new SeatNode[] {cur, next.getPrevious()[0]});
      curList.add(cur);
    }
    // ===== 轮空位 =====

    roundMapSeatArray.put(
        depthDesc, curList.stream().map(SeatNode::getSeatId).collect(Collectors.toList()));

    return leafToTree(
        id, depth - 1, depthDesc + 1, date, curList.toArray(new SeatNode[0]), roundMapSeatArray);
  }
}
