package moe.mycard.tabulator.exception;

import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import moe.mycard.tabulator.model.dto.ReturnMessage;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import javax.servlet.http.HttpServletRequest;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;

/**
 * 全局异常处理器
 *
 * @author SPiCa
 */
@RestControllerAdvice
public class GlobalExceptionProcessor extends Method {

  /** 全局未知异常捕获 */
  @ExceptionHandler(Exception.class)
  public ReturnMessage<Void> exception(Exception ex, HttpServletRequest request) {
    log(true, ex, request);
    return ReturnMessage.error();
  }

  /** 请求类型错误 */
  @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
  public ReturnMessage<Void> exception(
      HttpRequestMethodNotSupportedException ex, HttpServletRequest request) {
    log(false, ex, request);
    return ReturnMessage.error("请求类型错误,该资源不支持此请求类型");
  }

  /** 内置异常捕获 */
  @ExceptionHandler(CavException.class)
  public ReturnMessage<Void> exception(CavException ex, HttpServletRequest request) {
    log(true, ex, request);
    return ex.result();
  }
}

@Slf4j
class Method {

  private static final ObjectMapper om = new ObjectMapper();

  private static String key2Val(String json, String key) {
    try {
      JsonNode node = om.readTree(json);
      if (node != null) {
        return node.findValue(key).asText();
      }
    } catch (JsonProcessingException e) {
      log.error("【log.23101】获取json节点的值错误", e);
    }
    return null;
  }

  private static String substringBetween(String str, String open, String close) {
    if (str != null && open != null && close != null) {
      int start = str.indexOf(open);
      if (start != -1) {
        int end = str.indexOf(close, start + open.length());
        if (end != -1) {
          return str.substring(start + open.length(), end);
        }
      }
    }
    return null;
  }

  /**
   * 打印错误消息
   *
   * @param ex 异常信息
   * @param request 请求对象
   */
  protected void log(boolean all, Exception ex, HttpServletRequest request) {
    log.error(
        "{}：************************ STR *******************************", all ? "未知异常" : "预知异常");
    log.error("异常名称：【{}】", getMsg(ex));
    log.error("请求类型：【{}】", request.getMethod());
    log.error("请求地址：【{}】", request.getRequestURL());
    Enumeration<String> es = request.getParameterNames();
    String key;
    if (es.hasMoreElements()) {
      while (es.hasMoreElements()) {
        key = es.nextElement();
        String value = request.getParameter(key);
        if ("pass".equals(key) || "adminPass".equals(key)) value = "******";
        log.error("请求参数：【{}:{}】", key, value);
      }
    } else {
      log.error("请求参数：【】");
    }
    if (all) {
      Arrays.stream(ex.getStackTrace()).forEach(s -> log.error(s.toString()));
      // 递归打印Cause
      printCause(ex.getCause());
    }
    log.error(
        "{}：************************ END *******************************", all ? "未知异常" : "预知异常");
  }

  /** 获取feign 异常时候被调用接口抛出的异常消息 */
  private String getMsg(Exception e) {
    Throwable t = e.getCause();
    StringBuilder msg = new StringBuilder();
    String exceptionName = "feign.FeignException";
    if (t != null && exceptionName.equals(t.getClass().getName())) {
      String open = "{";
      String close = "}";
      String json = substringBetween(t.getMessage(), open, close);
      String m = key2Val(open + json + close, "message");
      if (StringUtils.isNotBlank(m)) {
        msg.append(m);
      }
    } else if (e instanceof MethodArgumentNotValidException) {
      MethodArgumentNotValidException me = (MethodArgumentNotValidException) e;
      List<ObjectError> errors = me.getBindingResult().getAllErrors();
      if (CollectionUtils.isNotEmpty(errors)) {
        for (ObjectError oe : errors) {
          if (StringUtils.isNotBlank(msg)) {
            msg.append(',');
          }
          msg.append(oe.getDefaultMessage());
        }
      }
    } else {
      msg.append(e.getMessage());
    }

    return msg.toString();
  }

  /** 递归CauseBy */
  private void printCause(Throwable cause) {
    if (cause != null) {
      log.error("Cause by:");
      StackTraceElement[] error = cause.getStackTrace();
      if (error.length > 0) {
        log.error(error[0].toString());
      }
      printCause(cause.getCause());
    }
  }
}
