import { Injectable } from '@nestjs/common';
import { BlankReturnMessageDto, CrudService } from 'nicot';
import { Tournament, TournamentStatus } from './entities/Tournament.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { MycardUser } from 'nestjs-mycard';

@Injectable()
export class TournamentService extends CrudService(Tournament, {
  relations: ['participants'],
}) {
  constructor(@InjectRepository(Tournament) repo) {
    super(repo);
  }

  getTournament(id: number, user: MycardUser) {
    return this.findOne(id, (qb) =>
      Tournament.extraQueryForUser(user, qb, this.entityAliasName),
    );
  }

  getTournaments(dto: Partial<Tournament>, user: MycardUser) {
    return this.findAll(dto, (qb) =>
      Tournament.extraQueryForUser(user, qb, this.entityAliasName),
    );
  }

  createTournament(tournament: Tournament, user: MycardUser) {
    tournament.creator = user.id;
    return this.create(tournament);
  }

  async updateTournament(
    id: number,
    dto: Partial<Tournament>,
    user: MycardUser,
  ) {
    await this.checkPermissionOfTournament(id, user);
    return this.update(id, dto);
  }

  async deleteTournament(id: number, user: MycardUser) {
    await this.checkPermissionOfTournament(id, user);
    return this.delete(id);
  }

  async checkPermissionOfTournament(id: number, user: MycardUser) {
    const tournament = await this.repo.findOne({
      where: { id },
      select: ['id', 'creator', 'collaborators'],
    });
    if (!tournament) {
      throw new BlankReturnMessageDto(404, '未找到该比赛。').toException();
    }
    return tournament.checkPermission(user);
  }

  async canModifyParticipants(id: number, user: MycardUser) {
    const tournamnt = await this.checkPermissionOfTournament(id, user);
    if (tournamnt.status !== TournamentStatus.Ready) {
      throw new BlankReturnMessageDto(
        403,
        '比赛已经开始，不能修改参赛者。',
      ).toException();
    }
    return tournamnt;
  }
}
