import { Column, Entity, Index, OneToMany, SelectQueryBuilder } from 'typeorm';
import {
  applyQueryProperty,
  BlankReturnMessageDto,
  DateColumn,
  EnumColumn,
  IntColumn,
  NotChangeable,
  NotColumn,
  NotWritable,
} from 'nicot';
import { MycardUser } from 'nestjs-mycard';
import { DescBase } from '../../utility/NamedBase.entity';
import { Participant } from '../../participant/entities/participant.entity';
import { IsArray, IsInt, IsPositive } from 'class-validator';
import { ApiProperty } from '@nestjs/swagger';

export enum TournamentRule {
  SingleElimination = 'SingleElimination',
  Swiss = 'Swiss',
}

export enum TournamentVisibility {
  Public = 'Public',
  Internal = 'Internal',
  Private = 'Private',
}

export enum TournamentStatus {
  Ready = 'Ready',
  Running = 'Running',
  Finished = 'Finished',
}

@Entity()
export class Tournament extends DescBase {
  @NotChangeable()
  @EnumColumn(TournamentRule, {
    default: TournamentRule.SingleElimination,
    description: '规则',
  })
  rule: TournamentRule;

  @EnumColumn(TournamentVisibility, {
    default: TournamentVisibility.Public,
    description: '可见性',
  })
  visibility: TournamentVisibility;

  @NotWritable()
  @EnumColumn(TournamentStatus, {
    default: TournamentStatus.Ready,
    description: '状态',
  })
  status: TournamentStatus;

  @NotWritable()
  @Index()
  @IntColumn('int', {
    description: '创建者 MC ID',
    unsigned: true,
    columnExtras: { nullable: false },
  })
  creator: number;

  @Index()
  @IsArray()
  @IsPositive({ each: true })
  @IsInt({ each: true })
  @ApiProperty({ type: [Number], description: '协作者 MC ID' })
  @Column('int', { array: true, default: [], comment: '协作者 MC ID' })
  collaborators: number[];

  @NotWritable()
  @Index()
  @DateColumn({ description: '创建时间', columnExtras: { nullable: false } })
  createdAt: Date;

  @NotColumn()
  @OneToMany(() => Participant, (participant) => participant.tournament)
  participants: Participant[];

  async beforeCreate() {
    this.createdAt = new Date();
  }

  applyQuery(qb: SelectQueryBuilder<Tournament>, entityName: string) {
    super.applyQuery(qb, entityName);
    applyQueryProperty(
      this,
      qb,
      entityName,
      'rule',
      'visibility',
      'creator',
      'createdAt',
    );
  }

  static extraQueryForUser(
    user: MycardUser,
    qb: SelectQueryBuilder<any>,
    entityName: string,
  ) {
    if (!user) {
      qb.andWhere(`${entityName}.visibility = :public`, {
        public: TournamentVisibility.Public,
      });
    } else {
      qb.andWhere(
        `(${entityName}.visibility != :private OR ${entityName}.creator = :self OR :self = ANY(${entityName}.collaborators))`,
        {
          private: TournamentVisibility.Private,
          self: user.id,
        },
      );
    }
  }

  checkPermission(user: MycardUser) {
    if (
      this.creator !== user.id &&
      !user.admin &&
      !this.collaborators.includes(user.id)
    ) {
      throw new BlankReturnMessageDto(403, '您无权操作该比赛。').toException();
    }
    return this;
  }
}
