import { Injectable } from '@nestjs/common';
import { BlankReturnMessageDto, CrudService, Inner } from 'nicot';
import { Participant } from './entities/participant.entity';
import { InjectRepository } from '@nestjs/typeorm';
import {
  Tournament,
  TournamentStatus,
} from '../tournament/entities/Tournament.entity';
import { MycardUser } from 'nestjs-mycard';
import { TournamentService } from '../tournament/tournament.service';

@Injectable()
export class ParticipantService extends CrudService(Participant, {
  relations: [Inner('tournament')],
}) {
  constructor(
    @InjectRepository(Participant) repo,
    private readonly tournamentService: TournamentService,
  ) {
    super(repo);
  }

  getParticipant(id: number, user: MycardUser) {
    return this.findOne(id, (qb) =>
      Tournament.extraQueryForUser(user, qb, 'tournament'),
    );
  }

  getParticipants(dto: Partial<Participant>, user: MycardUser) {
    return this.findAll(dto, (qb) =>
      Tournament.extraQueryForUser(user, qb, 'tournament'),
    );
  }

  async createParticipant(dto: Participant, user: MycardUser) {
    await this.tournamentService.canModifyParticipants(dto.tournamentId, user);
    return this.create(dto);
  }

  async updateParticipant(
    id: number,
    dto: Partial<Participant>,
    user: MycardUser,
  ) {
    await this.checkPermissionOfParticipant(id, user);
    return this.update(id, dto);
  }

  async deleteParticipant(id: number, user: MycardUser) {
    await this.checkPermissionOfParticipant(id, user);
    return this.delete(id);
  }

  async checkPermissionOfParticipant(id: number, user: MycardUser) {
    const participant = await this.repo.findOne({
      where: { id },
      select: {
        id: true,
        tournament: {
          id: true,
          creator: true,
          status: true,
          collaborators: true,
        },
      },
      relations: ['tournament'],
    });
    if (!participant?.tournament) {
      throw new BlankReturnMessageDto(404, '未找到该参赛者。').toException();
    }
    if (participant.tournament.status !== TournamentStatus.Ready) {
      throw new BlankReturnMessageDto(
        400,
        '比赛已开始，无法修改参赛者。',
      ).toException();
    }
    return participant.tournament.checkPermission(user);
  }

  async importParticipants(participants: Participant[], user: MycardUser) {
    return this.importEntities(participants, async (p) => {
      try {
        await this.tournamentService.checkPermissionOfTournament(
          p.tournamentId,
          user,
        );
      } catch (e) {
        return `玩家 ${p.name} 对应的比赛 ${p.tournamentId} 不存在或您没有权限。`;
      }
    });
  }
}
