import { Injectable } from '@nestjs/common';
import { BlankReturnMessageDto, CrudService } from 'nicot';
import { Tournament, TournamentStatus } from './entities/Tournament.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { MycardUser } from 'nestjs-mycard';
import { Match } from '../match/entities/match.entity';
import { Repository } from 'typeorm';

@Injectable()
export class TournamentService extends CrudService(Tournament, {
  relations: [
    'participants',
    'matches',
    'matches.player1',
    'matches.player2',
    'matches.winner',
  ],
}) {
  constructor(
    @InjectRepository(Tournament) repo,
    @InjectRepository(Match) private readonly matchRepo: Repository<Match>,
  ) {
    super(repo);
  }

  getTournament(id: number, user: MycardUser) {
    return this.findOne(id, (qb) =>
      Tournament.extraQueryForUser(user, qb, this.entityAliasName),
    );
  }

  getTournaments(dto: Partial<Tournament>, user: MycardUser) {
    return this.findAll(dto, (qb) =>
      Tournament.extraQueryForUser(user, qb, this.entityAliasName),
    );
  }

  createTournament(tournament: Tournament, user: MycardUser) {
    tournament.creator = user.id;
    return this.create(tournament);
  }

  async updateTournament(
    id: number,
    dto: Partial<Tournament>,
    user: MycardUser,
  ) {
    await this.checkPermissionOfTournament(id, user);
    return this.update(id, dto);
  }

  async deleteTournament(id: number, user: MycardUser) {
    await this.checkPermissionOfTournament(id, user);
    return this.delete(id);
  }

  async checkPermissionOfTournament(
    id: number,
    user: MycardUser,
    extraGets: (keyof Tournament)[] = [],
  ) {
    const tournament = await this.repo.findOne({
      where: { id },
      select: ['id', 'creator', 'collaborators', ...extraGets],
    });
    if (!tournament) {
      throw new BlankReturnMessageDto(404, '未找到该比赛。').toException();
    }
    return tournament.checkPermission(user);
  }

  async canModifyParticipants(id: number, user: MycardUser) {
    const tournamnt = await this.checkPermissionOfTournament(id, user);
    if (tournamnt.status !== TournamentStatus.Ready) {
      throw new BlankReturnMessageDto(
        403,
        '比赛已经开始，不能修改参赛者。',
      ).toException();
    }
    return tournamnt;
  }

  async resetTournament(id: number, user: MycardUser) {
    const tournament = await this.checkPermissionOfTournament(id, user, [
      'status',
    ]);
    if (tournament.status === TournamentStatus.Ready) {
      throw new BlankReturnMessageDto(
        403,
        '比赛还未开始，不能重置。',
      ).toException();
    }
    await this.repo.update(id, { status: TournamentStatus.Ready });
    await this.matchRepo.softDelete({ tournamentId: id });
    return new BlankReturnMessageDto(200, 'success');
  }

  async startTournament(id: number, user: MycardUser) {
    const tournament = await this.checkPermissionOfTournament(id, user, [
      'status',
    ]);
    if (tournament.status !== TournamentStatus.Ready) {
      throw new BlankReturnMessageDto(
        403,
        '比赛已经开始，不能重复开始。',
      ).toException();
    }
    await this.repo.update(id, { status: TournamentStatus.Running });
    // TODO: generate matches
    return new BlankReturnMessageDto(200, 'success');
  }

  async endTournament(id: number, user: MycardUser) {
    const tournament = await this.checkPermissionOfTournament(id, user, [
      'status',
    ]);
    if (tournament.status !== TournamentStatus.Running) {
      throw new BlankReturnMessageDto(
        403,
        '比赛还未开始，不能结束。',
      ).toException();
    }
    await this.repo.update(id, { status: TournamentStatus.Finished });
    // TODO: calculate results
    return new BlankReturnMessageDto(200, 'success');
  }
}
