import { Entity, ManyToOne, OneToMany, SelectQueryBuilder } from 'typeorm';
import { NamedBase } from '../../utility/NamedBase.entity';
import {
  applyQueryProperty,
  BoolColumn,
  NotChangeable,
  NotColumn,
} from 'nicot';
import { TournamentIdColumn } from '../../utility/decorators/tournament-id-column';
import {
  Tournament,
  TournamentVisibility,
} from '../../tournament/entities/Tournament.entity';
import { MycardUser } from 'nestjs-mycard';
import { Match } from '../../match/entities/match.entity';
import { ApiProperty } from '@nestjs/swagger';

@Entity()
export class Participant extends NamedBase {
  @BoolColumn({ default: false, description: '是否已经退赛' })
  quit: boolean;

  @NotChangeable()
  @TournamentIdColumn(true)
  tournamentId: number;

  @NotColumn()
  @ManyToOne(() => Tournament, (tournament) => tournament.participants)
  tournament: Tournament;

  @NotColumn()
  @OneToMany(() => Match, (match) => match.player1)
  matches1: Match[];

  @NotColumn()
  @OneToMany(() => Match, (match) => match.player2)
  matches2: Match[];

  @NotColumn()
  @OneToMany(() => Match, (match) => match.winner)
  wonMatches: Match[];

  @NotColumn()
  @ApiProperty({ type: [Match], description: '参与的比赛。' })
  matches: Match[];

  getMatches() {
    return this.matches1.concat(this.matches2);
  }

  async afterGet() {
    await super.afterGet();
    this.matches = this.getMatches();
  }

  checkPermission(user: MycardUser) {
    return this.tournament?.checkPermission(user);
  }

  applyQuery(qb: SelectQueryBuilder<Participant>, entityName: string) {
    super.applyQuery(qb, entityName);
    applyQueryProperty(this, qb, entityName, 'quit', 'tournamentId');
  }
}
