import { Entity, ManyToOne } from 'typeorm';
import { EnumColumn, IdBase, IntColumn, NotChangeable, NotColumn } from 'nicot';
import { TournamentIdColumn } from '../../utility/decorators/tournament-id-column';
import { Tournament } from '../../tournament/entities/Tournament.entity';
import { Participant } from '../../participant/entities/participant.entity';

enum MatchStatus {
  Pending = 'Pending',
  Running = 'Running',
  Finished = 'Finished',
}

@Entity()
export class Match extends IdBase() {
  @NotChangeable()
  @TournamentIdColumn(true)
  tournamentId: number;

  @NotColumn()
  @ManyToOne(() => Tournament, (tournament) => tournament.matches)
  tournament: Tournament;

  @NotChangeable()
  @IntColumn('smallint', {
    unsigned: true,
    required: true,
    description: '比赛轮次。',
  })
  round: number;

  @EnumColumn(MatchStatus, {
    default: MatchStatus.Pending,
    description: '比赛状态',
  })
  @NotChangeable()
  status: MatchStatus;

  @NotChangeable()
  @IntColumn('bigint', {
    unsigned: true,
    description: '玩家 1 ID',
  })
  player1Id: number;
  @IntColumn('mediumint', { description: '玩家 1 分数', required: false })
  player1Score: number;

  @NotColumn()
  @ManyToOne(() => Participant, (participant) => participant.matches1)
  player1: Participant;

  @NotChangeable()
  @IntColumn('bigint', {
    unsigned: true,
    description: '玩家 2 ID',
  })
  player2Id: number;
  @IntColumn('mediumint', { description: '玩家 2 分数', required: false })
  player2Score: number;

  @NotColumn()
  @ManyToOne(() => Participant, (participant) => participant.matches2)
  player2: Participant;

  @IntColumn('bigint', {
    unsigned: true,
    description: '胜者 ID',
  })
  winnerId: number;

  @NotColumn()
  @ManyToOne(() => Participant, (participant) => participant.wonMatches)
  winner: Participant;

  @NotChangeable()
  @IntColumn('bigint', {
    unsigned: true,
    description: '晋级通往的比赛 ID',
    required: false,
  })
  childMatchId: number;

  @NotColumn()
  @ManyToOne(() => Match, (match) => match.parentMatches)
  childMatch: Match;
  @NotColumn()
  @ManyToOne(() => Match, (match) => match.childMatch)
  parentMatches: Match[];
}
