import { Injectable } from '@nestjs/common';
import { BlankReturnMessageDto, CrudService } from 'nicot';
import { Tournament, TournamentStatus } from './entities/Tournament.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { MycardUser } from 'nestjs-mycard';
import { Match, MatchStatus } from '../match/entities/match.entity';
import { In, Repository } from 'typeorm';
import { Participant } from '../participant/entities/participant.entity';

@Injectable()
export class TournamentService extends CrudService(Tournament, {
  relations: [
    'participants',
    'matches',
    'matches.player1',
    'matches.player2',
    'matches.winner',
  ],
}) {
  constructor(
    @InjectRepository(Tournament) repo,
    @InjectRepository(Match) private readonly matchRepo: Repository<Match>,
  ) {
    super(repo);
  }

  async getTournament(id: number, user: MycardUser | number) {
    const result = await this.findOne(id, (qb) =>
      Tournament.extraQueryForUser(user, qb, this.entityAliasName),
    );
    result.data?.analytics();
    return result;
  }

  getTournaments(dto: Partial<Tournament>, user: MycardUser | number) {
    return this.findAll(dto, (qb) =>
      Tournament.extraQueryForUser(user, qb, this.entityAliasName),
    );
  }

  createTournament(tournament: Tournament, user: MycardUser | number) {
    tournament.creator = typeof user === 'number' ? user : user.id;
    return this.create(tournament);
  }

  async updateTournament(
    id: number,
    dto: Partial<Tournament>,
    user: MycardUser,
  ) {
    await this.checkPermissionOfTournament(id, user);
    return this.update(id, dto);
  }

  async deleteTournament(id: number, user: MycardUser | number) {
    await this.checkPermissionOfTournament(id, user);
    return this.delete(id);
  }

  async checkPermissionOfTournament(
    id: number,
    user: MycardUser | number,
    extraGets: (keyof Tournament)[] = [],
    relations: string[] = [],
  ) {
    const tournament = await this.repo.findOne({
      where: { id },
      select: ['id', 'creator', 'collaborators', ...extraGets],
      relations,
    });
    if (!tournament) {
      throw new BlankReturnMessageDto(404, '未找到该比赛。').toException();
    }
    return tournament.checkPermission(user);
  }

  async canModifyParticipants(id: number, user: MycardUser | number) {
    const tournamnt = await this.checkPermissionOfTournament(id, user, [
      'status',
    ]);
    if (tournamnt.status !== TournamentStatus.Ready) {
      throw new BlankReturnMessageDto(
        403,
        '比赛已经开始，不能修改参赛者。',
      ).toException();
    }
    return tournamnt;
  }

  async resetTournament(id: number, user: MycardUser | number) {
    const tournament = await this.checkPermissionOfTournament(id, user, [
      'status',
    ]);
    if (tournament.status === TournamentStatus.Ready) {
      throw new BlankReturnMessageDto(
        403,
        '比赛还未开始，不能重置。',
      ).toException();
    }
    await this.repo.manager.transaction(async (tdb) => {
      await tdb.update(Tournament, id, { status: TournamentStatus.Ready });
      await tdb.softDelete(Match, { tournamentId: id });
    });
    return new BlankReturnMessageDto(200, 'success');
  }

  async startTournament(id: number, user: MycardUser | number) {
    const tournament = await this.checkPermissionOfTournament(
      id,
      user,
      ['status', 'rule', 'ruleSettings'],
      ['participants'],
    );
    if (tournament.status !== TournamentStatus.Ready) {
      throw new BlankReturnMessageDto(
        403,
        '比赛已经开始，不能重复开始。',
      ).toException();
    }
    if (tournament.participants.length < 2) {
      throw new BlankReturnMessageDto(
        403,
        '参赛者数量不足，不能开始。',
      ).toException();
    }
    await this.repo.manager.transaction(async (edb) => {
      await tournament.getRuleHandler(edb.getRepository(Match)).initialize();
      await edb.update(Tournament, id, { status: TournamentStatus.Running });
    });
    return new BlankReturnMessageDto(200, 'success');
  }

  async endTournament(id: number, user: MycardUser | number) {
    const tournament = await this.checkPermissionOfTournament(id, user, [
      'status',
    ]);
    if (tournament.status !== TournamentStatus.Running) {
      throw new BlankReturnMessageDto(
        403,
        '比赛还未开始，不能结束。',
      ).toException();
    }
    if (
      await this.matchRepo.exists({
        where: {
          tournamentId: id,
          status: In([MatchStatus.Running, MatchStatus.Pending]),
        },
        take: 1,
      })
    ) {
      throw new BlankReturnMessageDto(
        403,
        '比赛还有未完成的对局，不能结束。',
      ).toException();
    }
    await this.repo.update(id, { status: TournamentStatus.Finished });
    return new BlankReturnMessageDto(200, 'success');
  }

  async clearTournamentParticipants(id: number, user: MycardUser | number) {
    const tournament = await this.checkPermissionOfTournament(id, user, [
      'status',
    ]);
    if (tournament.status !== TournamentStatus.Ready) {
      throw new BlankReturnMessageDto(
        403,
        '比赛已经开始，不能清空参赛者。',
      ).toException();
    }
    await this.repo.manager.softDelete(Participant, {
      tournamentId: id,
    });
    return new BlankReturnMessageDto(200, 'success');
  }

  async afterMatchUpdate(id: number) {
    if (
      !(await this.matchRepo.exists({
        where: { tournamentId: id, status: MatchStatus.Running },
      })) &&
      (await this.matchRepo.exists({
        where: { tournamentId: id, status: MatchStatus.Pending },
      }))
    ) {
      const tournament = await this.repo.findOne({
        where: { id },
        select: ['id', 'status', 'rule', 'ruleSettings'],
        relations: ['participants', 'matches'],
      });
      await this.repo.manager.transaction(async (edb) => {
        const repo = edb.getRepository(Match);
        const updates = tournament.getRuleHandler().nextRound();
        await Promise.all(
          updates.map((update) =>
            repo.update(update.id, {
              player1Id: update.player1Id,
              player2Id: update.player2Id,
              status: update.status,
            }),
          ),
        );
      });
    }
  }
}
