import { Controller, HttpCode, Post } from '@nestjs/common';
import { TournamentService } from './tournament.service';
import { ApiError, BlankReturnMessageDto, RestfulFactory } from 'nicot';
import { Tournament } from './entities/Tournament.entity';
import { ApiMycardUser, MycardUser, PutMycardUser } from 'nestjs-mycard';
import {
  ApiOkResponse,
  ApiOperation,
  ApiParam,
  ApiTags,
} from '@nestjs/swagger';

const factory = new RestfulFactory(Tournament, {
  relations: [
    'participants',
    'matches',
    'matches.player1',
    'matches.player2',
    'matches.winner',
    'matchTree',
    'matchTree.player1',
    'matchTree.player2',
    'matchTree.winner',
    'matchTree.parentMatches',
    'matchTree.parentMatches.player1',
    'matchTree.parentMatches.player2',
    'matchTree.parentMatches.winner',
    'matchTree.parentMatches.parentMatches',
    'matchTree.parentMatches.parentMatches.player1',
    'matchTree.parentMatches.parentMatches.player2',
    'matchTree.parentMatches.parentMatches.winner',
  ],
});
class CreateTournamentDto extends factory.createDto {}
class FindTournamentDto extends factory.findAllDto {}
class UpdateTournamentDto extends factory.updateDto {}

@ApiTags('tournament')
@Controller('tournament')
@ApiMycardUser({ optional: true })
@ApiError(404, '相关资源未找到')
@ApiError(403, '无法操作，弹 Toast 即可')
export class TournamentController {
  constructor(private readonly tournamentService: TournamentService) {}

  @factory.create()
  async create(
    @factory.createParam() tournament: CreateTournamentDto,
    @PutMycardUser() user: MycardUser,
  ) {
    return this.tournamentService.createTournament(tournament, user);
  }

  @factory.findOne()
  async findOne(
    @factory.idParam() id: number,
    @PutMycardUser(false) user: MycardUser,
  ) {
    return this.tournamentService.getTournament(id, user);
  }

  @factory.findAll()
  async findAll(
    @factory.findAllParam() dto: FindTournamentDto,
    @PutMycardUser(false) user: MycardUser,
  ) {
    return this.tournamentService.getTournaments(dto, user);
  }

  @factory.update()
  async update(
    @factory.idParam() id: number,
    @factory.updateParam() dto: UpdateTournamentDto,
    @PutMycardUser() user: MycardUser,
  ) {
    return this.tournamentService.updateTournament(id, dto, user);
  }

  @factory.delete()
  async delete(
    @factory.idParam() id: number,
    @PutMycardUser() user: MycardUser,
  ) {
    return this.tournamentService.deleteTournament(id, user);
  }

  @Post(':id/start')
  @HttpCode(200)
  @ApiOperation({ summary: '开始比赛' })
  @ApiParam({ name: 'id', description: 'Tournament ID' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async start(
    @factory.idParam() id: number,
    @PutMycardUser() user: MycardUser,
  ) {
    return this.tournamentService.startTournament(id, user);
  }

  @Post(':id/reset')
  @HttpCode(200)
  @ApiOperation({ summary: '重置比赛' })
  @ApiParam({ name: 'id', description: 'Tournament ID' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async reset(
    @factory.idParam() id: number,
    @PutMycardUser() user: MycardUser,
  ) {
    return this.tournamentService.resetTournament(id, user);
  }

  @Post(':id/finish')
  @HttpCode(200)
  @ApiOperation({ summary: '结束比赛' })
  @ApiParam({ name: 'id', description: 'Tournament ID' })
  @ApiOkResponse({ type: BlankReturnMessageDto })
  async finish(
    @factory.idParam() id: number,
    @PutMycardUser() user: MycardUser,
  ) {
    return this.tournamentService.endTournament(id, user);
  }
}
