import {
  Column,
  Entity,
  Index,
  OneToMany,
  Repository,
  SelectQueryBuilder,
} from 'typeorm';
import {
  BlankReturnMessageDto,
  DateColumn,
  EnumColumn,
  IntColumn,
  JsonColumn,
  NotChangeable,
  NotColumn,
  NotInResult,
  NotQueryable,
  NotWritable,
  QueryEqual,
  RelationComputed,
} from 'nicot';
import { MycardUser } from 'nestjs-mycard';
import { DescBase } from '../../utility/NamedBase.entity';
import {
  Participant,
  ParticipantScore,
} from '../../participant/entities/participant.entity';
import { IsArray, IsInt, IsOptional, IsPositive } from 'class-validator';
import { ApiProperty, PartialType } from '@nestjs/swagger';
import { Match } from '../../match/entities/match.entity';
import { TournamentRules } from '../../tournament-rules/rule-map';
import _ from 'lodash';
import { RenameClass } from 'nicot/dist/src/utility/rename-class';
import { QuerySinceDaysAgo } from '../../utility/query-since-days-ago';

export enum TournamentRule {
  SingleElimination = 'SingleElimination',
  Swiss = 'Swiss',
}

export enum TournamentVisibility {
  Public = 'Public',
  Internal = 'Internal',
  Private = 'Private',
}

export enum TournamentStatus {
  Ready = 'Ready',
  Running = 'Running',
  Finished = 'Finished',
}

export class RuleSettings {
  @ApiProperty({ description: '瑞士轮局数。' })
  rounds: number;

  @ApiProperty({ description: '瑞士轮胜利分。' })
  winScore: number;

  @ApiProperty({ description: '瑞士轮平局分。' })
  drawScore: number;

  @ApiProperty({ description: '瑞士轮轮空分。' })
  byeScore: number;

  @ApiProperty({ description: '淘汰赛中是否是季军塞' })
  hasThirdPlaceMatch: boolean;
}

@Entity()
export class Tournament extends DescBase {
  @QueryEqual()
  @NotChangeable()
  @EnumColumn(TournamentRule, {
    default: TournamentRule.SingleElimination,
    description: '规则',
  })
  rule: TournamentRule;

  getRuleHandler(repo?: Repository<Match>) {
    const ruleClass = TournamentRules.get(this.rule);
    return new ruleClass(this, repo);
  }

  @NotChangeable()
  @JsonColumn(RenameClass(PartialType(RuleSettings), 'RuleSettingsPartial'), {
    description: '比赛规则参数',
  })
  ruleSettings: RuleSettings;

  @QueryEqual()
  @EnumColumn(TournamentVisibility, {
    default: TournamentVisibility.Public,
    description: '可见性',
  })
  visibility: TournamentVisibility;

  @NotWritable()
  @EnumColumn(TournamentStatus, {
    default: TournamentStatus.Ready,
    description: '状态',
  })
  status: TournamentStatus;

  @NotWritable()
  @Index()
  @IntColumn('int', {
    description: '创建者 MC ID',
    unsigned: true,
    columnExtras: { nullable: false },
  })
  creator: number;

  @NotQueryable()
  @Index()
  @IsArray()
  @IsOptional()
  @IsPositive({ each: true })
  @IsInt({ each: true })
  @ApiProperty({ type: [Number], description: '协作者 MC ID', required: false })
  @Column('int', { array: true, default: [], comment: '协作者 MC ID' })
  collaborators: number[];

  @NotQueryable()
  @NotWritable()
  @Index()
  @DateColumn({ description: '创建时间', columnExtras: { nullable: false } })
  createdAt: Date;

  @NotWritable()
  @NotInResult()
  @IsInt()
  @IsPositive()
  @QuerySinceDaysAgo('createdAt')
  @ApiProperty({
    description: '查询最多创建时间在多少天前的比赛',
    required: false,
  })
  createdAtSinceDaysAgo: number;

  @NotColumn()
  @OneToMany(() => Participant, (participant) => participant.tournament)
  participants: Participant[];

  @NotColumn()
  @OneToMany(() => Match, (match) => match.tournament)
  matches: Match[];

  @NotColumn()
  @ApiProperty({
    description: '对阵图树',
  })
  @RelationComputed()
  matchTree: Match;

  async beforeCreate() {
    this.createdAt = new Date();
  }

  override isValidInCreate() {
    if (!this.getRuleHandler()) {
      return '该规则目前不受支持。';
    }
    return;
  }

  static extraQueryForUser(
    user: MycardUser | number,
    qb: SelectQueryBuilder<any>,
    entityName: string,
  ) {
    if (!user) {
      qb.andWhere(`${entityName}.visibility = :public`, {
        public: TournamentVisibility.Public,
      });
    } else {
      qb.andWhere(
        `(${entityName}.visibility != :private OR ${entityName}.creator = :self OR :self = ANY(${entityName}.collaborators))`,
        {
          private: TournamentVisibility.Private,
          self: typeof user === 'number' ? user : user.id,
        },
      );
    }
  }

  checkPermissionWithUserId(userId: number) {
    if (
      !userId ||
      (this.creator !== userId && !this.collaborators.includes(userId))
    ) {
      throw new BlankReturnMessageDto(403, '您无权操作该比赛。').toException();
    }
    return this;
  }
  checkPermission(user: MycardUser | number) {
    if (typeof user === 'number') {
      return this.checkPermissionWithUserId(user);
    }
    if (user.admin) return this;
    return this.checkPermissionWithUserId(user.id);
  }

  calculateScore() {
    const rule = this.getRuleHandler();
    this.participants.forEach((p) => {
      p.score = new ParticipantScore();
      Object.assign(p.score, rule.participantScore(p));
    });
    this.participants.forEach((p) => {
      const editScore = rule.participantScoreAfter(p);
      Object.assign(p.score, editScore);
    });
    this.participants = _.sortBy(
      this.participants,
      (p) => -p.score.score,
      (p) => -p.score.tieBreaker,
      (p) => p.id,
    );
    this.participants.forEach((p, i) => {
      p.score.rank = i + 1;
    });
  }

  calculateTree() {
    const finalMatch = _.maxBy(
      this.matches.filter((m) => !m.isThirdPlaceMatch),
      (m) => m.round,
    );
    this.matchTree = finalMatch.clone().buildTree(this.matches);
  }

  analytics() {
    if (!this.participants || this.status === TournamentStatus.Ready) {
      return;
    }
    this.calculateScore();
    if (this.rule === TournamentRule.SingleElimination) {
      this.calculateTree();
    }
  }
}
