import { TournamentRuleBase } from '../base';
import { RuleSettings } from '../../tournament/entities/Tournament.entity';
import { Match, MatchStatus } from '../../match/entities/match.entity';
import _ from 'lodash';
import {
  Participant,
  ParticipantScore,
} from '../../participant/entities/participant.entity';

export class Swiss extends TournamentRuleBase {
  private settings: Partial<RuleSettings> = {
    rounds:
      this.tournament.ruleSettings?.rounds ??
      Math.ceil(Math.log2(this.tournament.participants?.length || 2)),
    winScore: this.tournament.ruleSettings?.winScore ?? 3,
    drawScore: this.tournament.ruleSettings?.drawScore ?? 1,
    byeScore: this.tournament.ruleSettings?.byeScore ?? 3,
  };

  async initialize() {
    const matchCountPerRound = Math.floor(
      this.tournament.participants.length / 2,
    );

    const allMatches: Match[] = [];
    for (let r = 1; r <= this.settings.rounds; ++r) {
      const matches: Match[] = [];
      for (let i = 0; i < matchCountPerRound; ++i) {
        const match = new Match();
        match.round = r;
        match.status = MatchStatus.Pending;
        matches.push(match);
      }
      if (r === 1) {
        const participants = _.sortBy(
          this.tournament.participants,
          (p) => -p.id,
        );
        for (const match of matches) {
          match.status = MatchStatus.Running;
          match.player1Id = participants.pop().id;
          match.player2Id = participants.pop().id;
        }
      }
      allMatches.push(...matches);
    }
    await this.saveMatch(allMatches);
  }

  nextRound(): Partial<Match[]> {
    this.tournament.calculateScore();
    const participants = this.tournament.participants
      .filter((p) => !p.quit)
      .reverse();
    const nextRoundCount = this.nextRoundCount();
    const matches = this.tournament.matches.filter(
      (m) => m.round === nextRoundCount,
    );
    for (const match of matches) {
      match.status = MatchStatus.Running;
      match.player1Id = participants.pop()?.id;
      match.player2Id = participants.pop()?.id;
      if (!match.player1Id || !match.player2Id) {
        match.status = MatchStatus.Abandoned;
        match.player1Id = null;
        match.player2Id = null;
      }
    }
    return matches;
  }

  participantScore(participant: Participant): Partial<ParticipantScore> {
    const data = super.participantScore(participant);
    let bye = 0;
    for (let i = 1; i <= this.currentRoundCount(); ++i) {
      if (
        !this.tournament.matches.some(
          (m) => m.round === i && m.participated(participant.id),
        )
      ) {
        ++bye;
      }
    }
    return {
      ...data,
      bye,
      score:
        data.win * this.settings.winScore +
        data.draw * this.settings.drawScore +
        bye * this.settings.byeScore,
    };
  }

  participantScoreAfter(participant: Participant): Partial<ParticipantScore> {
    const opponentIds = this.specificMatches(MatchStatus.Finished)
      .filter((m) => m.participated(participant.id))
      .map((m) => m.opponentId(participant.id));

    const opponents = opponentIds.map((id) => this.participantMap.get(id));
    return {
      tieBreaker: _.sumBy(opponents, (p) => p.score.score),
    };
  }
}
