import { TournamentRuleBase } from '../base';
import { Match, MatchStatus } from '../../match/entities/match.entity';
import {
  Participant,
  ParticipantScore,
} from '../../participant/entities/participant.entity';
import _ from 'lodash';

export class SingleElimination extends TournamentRuleBase {
  totalRoundCount() {
    return Math.ceil(Math.log2(this.tournament.participants.length));
  }
  async initialize() {
    const roundCount = this.totalRoundCount();
    const matchStack: Record<number, Match[]> = {};
    for (let i = roundCount; i > 0; i--) {
      let matches: Match[];
      if (i === roundCount) {
        matches = [new Match()];
      } else {
        const nextRoundMatches = matchStack[i + 1];
        matches = nextRoundMatches.flatMap((m) => [
          new Match().setChildMatch(m),
          new Match().setChildMatch(m),
        ]);
        if (i === 1) {
          // add participants to first round
          const participants = _.sortBy(
            this.tournament.participants,
            (p) => -p.id,
          );
          const neededMatchesCount =
            participants.length - nextRoundMatches.length * 2;
          matches = matches.slice(0, neededMatchesCount);
          for (const match of matches) {
            match.player1Id = participants.pop().id;
            match.player2Id = participants.pop().id;
            match.status = MatchStatus.Running;
          }
        }
      }
      matches.forEach((m) => (m.round = i));

      matches = await this.saveMatch(matches);
      matchStack[i] = matches;
    }
    if (
      this.tournament.participants.length >= 4 &&
      this.tournament.ruleSettings.hasThirdPlaceMatch
    ) {
      // add third place match
      const thirdPlaceMatch = new Match();
      thirdPlaceMatch.isThirdPlaceMatch = true;
      thirdPlaceMatch.round = roundCount;
      await this.saveMatch([thirdPlaceMatch]);
    }
  }

  nextRound() {
    const finishedMatches = this.specificMatches(MatchStatus.Finished);
    const survivedParticipants = this.tournament.participants
      .filter(
        (p) =>
          !finishedMatches.some(
            (m) => m.participated(p.id) && m.winnerId !== p.id,
          ),
      )
      .reverse();
    const nextRoundCount = this.nextRoundCount();
    const matches = this.specificMatches(MatchStatus.Pending).filter(
      (m) => m.round === nextRoundCount && !m.isThirdPlaceMatch,
    );
    for (const match of matches) {
      match.player1Id = survivedParticipants.pop().id;
      match.player2Id = survivedParticipants.pop().id;
      match.status = MatchStatus.Running;
    }
    if (
      nextRoundCount === this.totalRoundCount() &&
      this.tournament.ruleSettings.hasThirdPlaceMatch
    ) {
      const thirdPlaceMatch = this.tournament.matches.find(
        (m) => m.isThirdPlaceMatch && m.status === MatchStatus.Pending,
      );
      const losers = finishedMatches
        .filter((m) => m.round === nextRoundCount - 1)
        .map((m) => m.loserId());
      if (thirdPlaceMatch && losers.length >= 2) {
        thirdPlaceMatch.player1Id = losers[0];
        thirdPlaceMatch.player2Id = losers[1];
        thirdPlaceMatch.status = MatchStatus.Running;
        matches.push(thirdPlaceMatch);
      }
    }
    return matches;
  }
  participantScore(participant: Participant): Partial<ParticipantScore> {
    const matches = this.specificMatches(MatchStatus.Finished);
    const scores = matches
      .filter((m) => m.participated(participant.id))
      .map((m) => {
        if (m.winnerId !== participant.id) {
          return 0;
        }
        let point = Math.pow(2, m.round);
        if (m.isThirdPlaceMatch) {
          point *= 0.25;
        }
        return point;
      });
    const standardRoundCount = Math.log2(this.tournament.participants.length);
    return {
      ...super.participantScore(participant),
      bye:
        standardRoundCount - Math.ceil(standardRoundCount) &&
        !matches.some((m) => m.participated(participant.id) && m.round === 1)
          ? 1
          : 0,
      score: scores.reduce((a, b) => a + b, 0),
      tieBreaker: 0,
    };
  }
}
