import {
  IsArray,
  IsNotEmpty,
  IsOptional,
  IsString,
  Matches,
  MaxLength,
  MinLength,
  Validate,
  ValidateNested,
  ValidationArguments,
  ValidatorConstraint,
  ValidatorConstraintInterface,
} from 'class-validator';
import { ApiProperty, OmitType } from '@nestjs/swagger';
import _ from 'lodash';
import { Participant } from '../participant/entities/participant.entity';
import { Type } from 'class-transformer';
import { RestfulFactory } from 'nicot';
import { Match, MatchStatus } from '../match/entities/match.entity';
import { Tournament } from '../tournament/entities/Tournament.entity';

export const participantRestfulFactory = new RestfulFactory(Participant, {
  entityClassName: 'SRVProParticipant',
  relations: [],
});

export class WithApiKeyBody {
  @IsString()
  @IsNotEmpty()
  @MaxLength(64)
  @MinLength(64)
  @ApiProperty({
    description: 'API Key',
    type: String,
    example: _.range(64)
      .map(() => 'x')
      .join(''),
  })
  api_key: string;
}

@ValidatorConstraint({ name: 'IsWinnerId', async: false })
export class IsWinnerId implements ValidatorConstraintInterface {
  validate(value: any, args: ValidationArguments) {
    if (value === 'tie') return true; // 'tie' 是合法的
    if (typeof value === 'number' && value > 0) return true; // 正数也是合法的
    return false; // 其他情况不合法
  }

  defaultMessage(args: ValidationArguments) {
    return `${args.property} must be either "tie" or a positive number`;
  }
}

class SRVProCreateParticipantDto extends OmitType(
  participantRestfulFactory.createDto,
  ['tournamentId', 'quit'],
) {}

export class SRVProUploadParticipantDto extends WithApiKeyBody {
  @IsArray()
  @ValidateNested({
    each: true,
  })
  @Type(() => SRVProCreateParticipantDto)
  @ApiProperty({
    type: () => [SRVProCreateParticipantDto],
    description: '参与者列表',
  })
  participants: Participant[];
}

const WinnderIdProperty = () =>
  ApiProperty({
    description: '比赛胜者 ID。不写代表比赛还没结束，写 tie 代表平局',
    required: false,
    oneOf: [
      { type: 'string', example: 'tie', enum: ['tie'] },
      { type: 'number', example: 1, minimum: 1 },
    ],
  });

export class SRVProUploadMatch {
  @IsString()
  @Matches(/^-?\d+--?\d+$/)
  @ApiProperty({
    description: '比赛比分',
    example: '2-1',
  })
  scores_csv: string;

  @IsOptional()
  @Validate(IsWinnerId)
  @WinnderIdProperty()
  winner_id?: 'tie' | number;
}

export const srvproUploadMatchToUpdateMatch = (
  srvproMatch: SRVProUploadMatch,
): Partial<Match> => {
  const csvMatch = srvproMatch.scores_csv.match(/^(-?\d+)-(-?\d+)$/);
  const player1Score = parseInt(csvMatch[1]);
  const player2Score = parseInt(csvMatch[2]);
  return {
    winnerId:
      srvproMatch.winner_id === 'tie'
        ? null
        : srvproMatch.winner_id
          ? srvproMatch.winner_id
          : undefined,
    player1Score,
    player2Score,
  };
};

export class SRVProUploadMatchDto extends WithApiKeyBody {
  @ValidateNested()
  @ApiProperty({
    description: '比分信息',
  })
  match: SRVProUploadMatch;
}

export class SRVProParticipantDto {
  @ApiProperty({
    type: () => participantRestfulFactory.entityResultDto,
  })
  participant: Participant;

  constructor(participant: Participant) {
    this.participant = participant;
  }
}

export class SRVProMatch {
  @ApiProperty({
    description: '比赛 ID',
  })
  id: number;
  @ApiProperty({
    description: '比赛状态',
    enum: ['pending', 'open', 'complete'],
    type: String,
  })
  state: 'pending' | 'open' | 'complete'; // pending: 还未开始，open: 进行中，complete: 已结束
  @ApiProperty({
    description: '玩家 1 ID',
  })
  player1_id: number;
  @ApiProperty({
    description: '玩家 2 ID',
  })
  player2_id: number;
  @WinnderIdProperty()
  winner_id?: number | 'tie'; // 如果存在，则代表该比赛已经结束
  @ApiProperty({
    description: '比赛的比分',
  })
  scores_csv?: string; // 比分，2-1 这样的格式，请保证和上传的情况相同

  constructor(match: Match) {
    this.id = match.id;
    this.state =
      match.status === MatchStatus.Pending
        ? 'pending'
        : match.status === MatchStatus.Running
          ? 'open'
          : 'complete';
    this.player1_id = match.player1Id;
    this.player2_id = match.player2Id;
    this.winner_id =
      match.winnerId ||
      (match.status === MatchStatus.Finished ? 'tie' : undefined);
    this.scores_csv =
      match.player1Score && match.player2Score
        ? `${match.player1Score}-${match.player2Score}`
        : undefined;
  }
}

export class SRVProMatchDto {
  @ApiProperty({
    description: '比赛信息',
  })
  match: SRVProMatch;

  constructor(match: Match) {
    this.match = new SRVProMatch(match);
  }
}

export class SRVProTournament {
  @ApiProperty({
    description: '比赛 ID',
  })
  id: number;
  @ApiProperty({ type: () => [SRVProParticipantDto] })
  participants: SRVProParticipantDto[];
  @ApiProperty({ type: () => [SRVProMatchDto] })
  matches: SRVProMatchDto[];

  constructor(tournament: Tournament) {
    this.id = tournament.id;
    this.participants =
      tournament.participants?.map((p) => new SRVProParticipantDto(p)) || [];
    this.matches = tournament.matches?.map((m) => new SRVProMatchDto(m)) || [];
  }
}

export class SRVProTournamentDto {
  @ApiProperty({
    description: '比赛信息',
  })
  tournament: SRVProTournament;

  constructor(tournament: Tournament) {
    this.tournament = new SRVProTournament(tournament);
  }
}
