import {
  Controller,
  Delete,
  Get,
  HttpCode,
  Param,
  ParseIntPipe,
  Post,
  Put,
} from '@nestjs/common';
import {
  ApiOkResponse,
  ApiOperation,
  ApiParam,
  ApiTags,
} from '@nestjs/swagger';
import { ApiKeyError, ApiKeyQuery } from '../api-key/api-require-key';
import { SrvproService } from './srvpro.service';
import {
  participantRestfulFactory,
  SRVProTournamentDto,
  SRVProUploadMatchDto,
  SRVProUploadParticipantDto,
} from './srvpro.dto';
import { ApiBlankResponse, ApiError, DataBody } from 'nicot';
import { PutApiKeyUserId } from '../api-key/api-key.pipe';

const ApiTournamentIdParam = () =>
  ApiParam({
    name: 'tournamentId',
    description: '赛事 ID',
    required: true,
    type: Number,
  });

@ApiTags('srvpro')
@Controller('srvpro/v1')
@ApiKeyError()
@ApiError(404, '信息不存在')
@ApiError(403, '无法操作或没有权限')
export class SrvproController {
  constructor(private service: SrvproService) {}

  // GET /v1/tournaments/${tournament_id}.json?api_key=xxx&include_participants=1&include_matches=1 returns { tournament: Tournament }
  @Get('tournaments/:tournamentId.json')
  @ApiOperation({
    summary: '获取赛事信息',
  })
  @ApiKeyQuery()
  @ApiTournamentIdParam()
  @ApiOkResponse({
    type: SRVProTournamentDto,
  })
  async getTournament(
    @Param('tournamentId', ParseIntPipe) tournamentId: number,
    @PutApiKeyUserId() userId: number,
  ) {
    return this.service.getTournament(tournamentId, userId);
  }

  // PUT /v1/tournaments/${tournament_id}/matches/${match_id}.json { api_key: string, match: MatchPost } returns ANY
  @Put('tournaments/:tournamentId/matches/:matchId.json')
  @HttpCode(200)
  @ApiOperation({
    summary: '上传赛事成绩',
  })
  @ApiTournamentIdParam()
  @ApiParam({
    name: 'matchId',
    description: '对局 ID',
    required: true,
    type: Number,
  })
  @ApiBlankResponse()
  async putScore(
    @Param('tournamentId', ParseIntPipe) tournamentId: number,
    @Param('matchId', ParseIntPipe) matchId: number,
    @DataBody() body: SRVProUploadMatchDto,
    @PutApiKeyUserId() userId: number,
  ) {
    return this.service.putScore(tournamentId, matchId, body, userId);
  }

  // DELETE /v1/tournaments/${tournament_id}/participants/clear.json?api_key=xxx returns ANY
  @Delete('tournaments/:tournamentId/participants/clear.json')
  @HttpCode(200)
  @ApiOperation({
    summary: '清空赛事选手',
  })
  @ApiKeyQuery()
  @ApiTournamentIdParam()
  @ApiBlankResponse()
  async clearParticipants(
    @Param('tournamentId', ParseIntPipe) tournamentId: number,
    @PutApiKeyUserId() userId: number,
  ) {
    return this.service.clearParticipants(tournamentId, userId);
  }

  // POST /v1/tournaments/${tournament_id}/participants/bulk_add.json { api_key: string, participants: { name: string }[] } returns ANY
  @Post('tournaments/:tournamentId/participants/bulk_add.json')
  @HttpCode(200)
  @ApiOperation({
    summary: '批量上传选手',
  })
  @ApiTournamentIdParam()
  @ApiOkResponse({
    type: participantRestfulFactory.importReturnMessageDto,
  })
  async uploadParticipants(
    @Param('tournamentId', ParseIntPipe) tournamentId: number,
    @DataBody() body: SRVProUploadParticipantDto,
    @PutApiKeyUserId() userId: number,
  ) {
    return this.service.uploadParticipants(tournamentId, body, userId);
  }
}
