import { Entity, ManyToOne, OneToMany } from 'typeorm';
import { NamedBase } from '../../utility/NamedBase.entity';
import {
  BoolColumn,
  NotChangeable,
  NotColumn,
  NotInResult,
  NotQueryable,
  QueryEqual,
  QueryMatchBoolean,
  StringColumn,
} from 'nicot';
import { TournamentIdColumn } from '../../utility/decorators/tournament-id-column';
import { Tournament } from '../../tournament/entities/Tournament.entity';
import { MycardUser } from 'nestjs-mycard';
import { Match } from '../../match/entities/match.entity';
import { ApiProperty } from '@nestjs/swagger';
import { IsBase64 } from 'class-validator';

export class ParticipantScore {
  @ApiProperty({ description: '排名' })
  rank: number;
  @ApiProperty({ description: '得分' })
  score: number;
  @ApiProperty({ description: '胜场' })
  win: number;
  @ApiProperty({ description: '平场' })
  draw: number;
  @ApiProperty({ description: '负场' })
  lose: number;
  @ApiProperty({ description: '轮空' })
  bye: number;
  @ApiProperty({ description: '平局分' })
  tieBreaker: number;
}

@Entity({ orderBy: { id: 'ASC' } })
export class Participant extends NamedBase {
  @QueryMatchBoolean()
  @BoolColumn({ default: false, description: '是否已经退赛' })
  quit: boolean;

  @QueryEqual()
  @NotChangeable()
  @TournamentIdColumn(true)
  tournamentId: number;

  @NotColumn()
  @ManyToOne(() => Tournament, (tournament) => tournament.participants)
  tournament: Tournament;

  @NotColumn()
  @OneToMany(() => Match, (match) => match.player1)
  matches1: Match[];

  @NotColumn()
  @OneToMany(() => Match, (match) => match.player2)
  matches2: Match[];

  @NotColumn()
  @OneToMany(() => Match, (match) => match.winner)
  wonMatches: Match[];

  @NotColumn({
    description: '该选手的成绩。',
    required: false,
  })
  score: ParticipantScore;

  getMatches() {
    return this.matches1.concat(this.matches2);
  }

  async afterGet() {
    await super.afterGet();
    //this.matches = this.getMatches();
  }

  checkPermission(user: MycardUser | number) {
    return this.tournament?.checkPermission(user);
  }

  @IsBase64()
  @NotQueryable()
  @StringColumn(1024, {
    description:
      '卡组 base64，影响 srvpro 的卡组。用库 ygopro-deck-encode 生成。',
  })
  deckbuf?: string;
}
