import { Injectable } from '@nestjs/common';
import { BlankReturnMessageDto, CrudService, Inner } from 'nicot';
import { Match, MatchStatus } from './entities/match.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { MycardUser } from 'nestjs-mycard';
import {
  Tournament,
  TournamentRule,
  TournamentStatus,
} from '../tournament/entities/Tournament.entity';
import { TournamentService } from '../tournament/tournament.service';
import { MoreThan } from 'typeorm';

@Injectable()
export class MatchService extends CrudService(Match, {
  relations: [Inner('tournament'), 'player1', 'player2', 'winner'],
}) {
  constructor(
    @InjectRepository(Match) repo,
    private tournamentService: TournamentService,
  ) {
    super(repo);
  }

  getMatch(id: number, user: MycardUser) {
    return this.findOne(id, (qb) =>
      Tournament.extraQueryForUser(user, qb, 'tournament'),
    );
  }

  getMatches(dto: Partial<Match>, user: MycardUser) {
    return this.findAll(dto, (qb) =>
      Tournament.extraQueryForUser(user, qb, 'tournament'),
    );
  }

  async updateMatch(
    id: number,
    dto: Partial<Match>,
    user: MycardUser | number,
  ) {
    const match = await this.repo.findOne({
      where: { id },
      relations: ['tournament'],
      select: {
        id: true,
        status: true,
        round: true,
        tournamentId: true,
        player1Id: true,
        player2Id: true,
        tournament: {
          id: true,
          status: true,
          creator: true,
          collaborators: true,
          rule: true,
        },
      },
    });
    if (!match) {
      throw new BlankReturnMessageDto(404, '对局不存在。').toException();
    }
    match.tournament.checkPermission(user);
    if (dto.winnerId && !match.participated(dto.winnerId)) {
      throw new BlankReturnMessageDto(
        400,
        '对局胜者不在对局中。',
      ).toException();
    }
    if (match.status === MatchStatus.Pending) {
      throw new BlankReturnMessageDto(
        400,
        '对局尚未开始，无法修改。',
      ).toException();
    }
    if (match.tournament.status === TournamentStatus.Finished) {
      throw new BlankReturnMessageDto(
        400,
        '比赛已结束，无法修改。',
      ).toException();
    }
    if (
      dto.winnerId === null &&
      match.tournament.rule !== TournamentRule.Swiss
    ) {
      throw new BlankReturnMessageDto(
        400,
        '非瑞士轮对局胜者不能为空。',
      ).toException();
    }
    if (dto.winnerId !== undefined) {
      dto.status = MatchStatus.Finished;
    }
    if (match.status === MatchStatus.Finished && dto.winnerId !== undefined) {
      // clean all other matches in greater rounds
      await this.repo.update(
        { round: MoreThan(match.round), tournamentId: match.tournamentId },
        {
          winnerId: null,
          status: MatchStatus.Pending,
          player1Id: null,
          player2Id: null,
          player1Score: null,
          player2Score: null,
        },
      );
    }
    dto.winnerId ||= null;
    const result = await this.update(id, dto);
    await this.tournamentService.afterMatchUpdate(match.tournamentId);
    return result;
  }
}
