import { Entity, Index, ManyToOne, OneToMany } from 'typeorm';
import {
  BoolColumn,
  EnumColumn,
  IdBase,
  IntColumn,
  NotChangeable,
  NotColumn,
  NotInResult,
  QueryEqual,
  QueryMatchBoolean,
} from 'nicot';
import { TournamentIdColumn } from '../../utility/decorators/tournament-id-column';
import { Tournament } from '../../tournament/entities/Tournament.entity';
import { Participant } from '../../participant/entities/participant.entity';

export enum MatchStatus {
  Pending = 'Pending',
  Running = 'Running',
  Finished = 'Finished',
  Abandoned = 'Abandoned',
}

@Entity()
export class Match extends IdBase() {
  @NotChangeable()
  @TournamentIdColumn(true)
  @QueryEqual()
  tournamentId: number;

  @NotColumn()
  @ManyToOne(() => Tournament, (tournament) => tournament.matches)
  tournament: Tournament;

  @Index()
  @NotChangeable()
  @IntColumn('smallint', {
    unsigned: true,
    required: true,
    description: '比赛轮次。',
  })
  @QueryEqual()
  round: number;

  @NotChangeable()
  @BoolColumn({ default: false, description: '是否为季军赛。' })
  @QueryMatchBoolean()
  isThirdPlaceMatch: boolean;

  @EnumColumn(MatchStatus, {
    default: MatchStatus.Pending,
    description: '比赛状态',
  })
  @NotChangeable()
  @QueryEqual()
  status: MatchStatus;

  @NotChangeable()
  @IntColumn('bigint', {
    unsigned: true,
    description: '玩家 1 ID',
  })
  @QueryEqual()
  player1Id: number;
  @IntColumn('smallint', { description: '玩家 1 分数', required: false })
  @QueryEqual()
  player1Score: number;

  @NotColumn()
  @ManyToOne(() => Participant, (participant) => participant.matches1)
  player1: Participant;

  @NotChangeable()
  @IntColumn('bigint', {
    unsigned: true,
    description: '玩家 2 ID',
  })
  @QueryEqual()
  player2Id: number;
  @IntColumn('smallint', { description: '玩家 2 分数', required: false })
  @QueryEqual()
  player2Score: number;

  @NotColumn()
  @ManyToOne(() => Participant, (participant) => participant.matches2)
  player2: Participant;

  @IntColumn('bigint', {
    unsigned: true,
    description:
      '胜者 ID。update 的时候填写 null 代表平局，undefined 代表还未结束。',
  })
  @QueryEqual()
  winnerId: number;

  @NotColumn()
  @ManyToOne(() => Participant, (participant) => participant.wonMatches)
  winner: Participant;

  loserId() {
    return this.player1Id === this.winnerId ? this.player2Id : this.player1Id;
  }

  loser() {
    return this.player1Id === this.winnerId ? this.player2 : this.player1;
  }

  @NotChangeable()
  @IntColumn('bigint', {
    unsigned: true,
    description: '晋级通往的比赛 ID',
    required: false,
  })
  @QueryEqual()
  childMatchId: number;

  setChildMatch(match: Match) {
    this.childMatchId = match.id;
    return this;
  }

  @NotColumn()
  @ManyToOne(() => Match, (match) => match.parentMatches)
  childMatch: Match;
  @NotColumn()
  @OneToMany(() => Match, (match) => match.childMatch)
  parentMatches: Match[];

  participated(id: number) {
    return this.player1Id === id || this.player2Id === id;
  }

  opponentId(id: number) {
    return this.player1Id === id ? this.player2Id : this.player1Id;
  }

  clone() {
    const match = new Match();
    Object.assign(match, this);
    return match;
  }

  buildTree(matches: Match[]) {
    this.parentMatches = matches
      .filter((match) => match.childMatchId === this.id)
      .map((match) => match.clone().buildTree(matches));
    return this;
  }
}
