import { Injectable, ConsoleLogger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { Request, Response } from 'express';
import { Aragami, CacheKey, CacheTTL } from 'aragami';
import { InjectAragami } from 'nestjs-aragami';
import { PuppeteerService } from './puppeteer/puppeteer.service';
import { BlankReturnMessageDto } from './dto/ReturnMessage.dto';
import { HttpService } from '@nestjs/axios';
import { lastValueFrom } from 'rxjs';

@CacheTTL(3600000)
class SiteCache {
  @CacheKey()
  url: string;
  html: string;
}

@Injectable()
export class AppService extends ConsoleLogger {
  constructor(
    private readonly config: ConfigService,
    @InjectAragami() private readonly aragami: Aragami,
    private readonly puppeteerService: PuppeteerService,
    private readonly httpService: HttpService,
  ) {
    super('app');
  }

  async render(req: Request, res: Response) {
    const url = req.originalUrl.startsWith('/')
      ? req.originalUrl.slice(1)
      : req.originalUrl;
    return this.aragami.lock(url, async () => {
      const siteCache = await this.aragami.get(SiteCache, url);
      if (siteCache) {
        this.log(`Cache hit for ${url}`);
        return siteCache.html;
      }
      const testHead = await lastValueFrom(
        this.httpService.head(url, {
          validateStatus: () => true,
        }),
      );
      if (testHead.status >= 400) {
        throw new BlankReturnMessageDto(
          testHead.status,
          testHead.statusText,
        ).toException();
      }
      if (testHead.status >= 300) {
        const location = testHead.headers.location;
        if (location) {
          res.header('Location', location);
          throw new BlankReturnMessageDto(
            testHead.status,
            testHead.statusText,
          ).toException();
        }
      }
      this.log(`Rendering ${url} requested by ${req.header('User-Agent')}`);
      try {
        const html = await this.puppeteerService.render(url);
        await this.aragami.set(SiteCache, { url, html });
        return html;
      } catch (e) {
        this.error(`Error rendering ${url}`, e);
        throw new BlankReturnMessageDto(
          404,
          `Page not found: ${url}`,
        ).toException();
      }
    });
  }
}
