import { createParamDecorator, ExecutionContext } from '@nestjs/common';
import { Request } from 'express';
import axios, { AxiosResponse } from 'axios';

export interface MyCardUser {
  id: number;
  username: string;
  name: string;
  email: string;
  password_hash: string;
  salt: string;
  active: boolean;
  admin: boolean;
  avatar: string;
  locale: string;
  registration_ip_address: string;
  ip_address: string;
  created_at: string;
  updated_at: string;
}

export interface FetchMyCardUserOptions {
  mycardAccountsUrl: string;
  field: string;
}

export async function fetchUserWithToken(
  mycardAccountsUrl: string,
  token: string,
) {
  let authResult: AxiosResponse<MyCardUser>;
  try {
    authResult = await axios.get<MyCardUser>(`${mycardAccountsUrl}/authUser`, {
      responseType: 'json',
      validateStatus: (s) => true,
      headers: { Authorization: `Bearer ${token}` },
    });
  } catch (e) {
    return null;
  }
  if (authResult.status >= 400) {
    return null;
  }
  return authResult.data;
}

export const FetchMyCardUser = createParamDecorator(
  async (options: FetchMyCardUserOptions, context: ExecutionContext) => {
    const _options = {
      mycardAccountsUrl:
        process.env.MYCARD_ACCOUNTS_URL ||
        `https://sapi.moecube.com:444/accounts`,
      field: 'sso',
      ...options,
    };
    const req = context.switchToHttp().getRequest<Request>();
    let token: string;
    if (!token && req.headers) {
      const authorizationHeader = req.headers['authorization'] as string;
      if (authorizationHeader) {
        token = authorizationHeader.split(' ').pop();
      }
    }
    if (!token && req.query) {
      token = req.query[_options.field] as string;
    }
    if (!token && req.body) {
      token = req.body[_options.field] as string;
    }
    if (!token) {
      return null;
    }
    return fetchUserWithToken(_options.mycardAccountsUrl, token);
  },
);
