import {
  ConsoleLogger,
  Injectable,
  OnApplicationBootstrap,
} from '@nestjs/common';
import { UseEvent, WireContextService } from 'koishi-nestjs';
import { Session } from 'koishi';
import S3Assets from '@koishijs/plugin-assets-s3';
import { InjectCenterAccount } from '../utility/constant';
import { ElasticsearchService } from '@nestjs/elasticsearch';

@Injectable()
export class MessageService
  extends ConsoleLogger
  implements OnApplicationBootstrap
{
  constructor(
    @InjectCenterAccount()
    private readonly centerAccount: string,
    private readonly elasticsearchService: ElasticsearchService,
  ) {
    super('message');
  }

  onApplicationBootstrap(): any {
    if (this.assets) {
      this.log(`Assets transformation enabled.`);
    }
  }

  @UseEvent('message')
  async onMessage(session: Session) {
    if (
      session.targetId === this.centerAccount ||
      // eslint-disable-next-line @typescript-eslint/ban-ts-comment
      // @ts-ignore
      session.subtype === 'guild'
    ) {
      return;
    }
    let content = session.content;
    if (this.assets) {
      try {
        content = await this.assets.transform(content);
      } catch (e) {
        content = session.content;
        this.error(`Failed to transform assets in message: ${e.message}`);
      }
    }
    // targetId, for private messages only
    const fromId = session.userId;
    const targetId = session.targetId || session.guildId;
    const fromName = session.username;
    this.log(
      `Received message from ${fromName}(${fromId}) to ${targetId}: ${content}`,
    );

    await this.elasticsearchService.index({
      index: `messages`,
      body: {
        selfId: session.selfId,
        type: session.subtype as string,
        fromId,
        fromName,
        targetId,
        content,
      },
    });
  }

  @WireContextService()
  private assets: S3Assets;
}
