import { ConsoleLogger, Injectable } from '@nestjs/common';
import { BotService } from '../bot/bot.service';
import { JwtService } from '@nestjs/jwt';
import { BlankReturnMessageDto } from '../dto/ReturnMessage.dto';
import { InjectCenterAccount } from '../utility/constant';
import { Bot } from 'koishi';

@Injectable()
export class AuthService extends ConsoleLogger {
  constructor(
    private readonly botService: BotService,
    private readonly jwt: JwtService,
    @InjectCenterAccount()
    private readonly centerAccount: string,
  ) {
    super('AuthService');
  }

  private async checkCenterFriend(bot: Bot) {
    let friends: Bot.User[];
    try {
      friends = await bot.getFriendList();
    } catch (e) {
      this.error(`Failed to get friend list of ${bot.selfId}: ${e.message}`);
      throw new BlankReturnMessageDto(
        500,
        'Failed fetching friend list.',
      ).toException();
    }
    const centerFriend = friends?.find(
      (friend) => friend.userId === this.centerAccount,
    );
    if (!centerFriend) {
      throw new BlankReturnMessageDto(
        404,
        'Center account is not a friend.',
      ).toException();
    }
  }

  async sendCode(selfId: string) {
    const bot = this.botService.getBot(selfId);
    if (!bot) {
      throw new BlankReturnMessageDto(404, 'Bot not found').toException();
    }
    await this.checkCenterFriend(bot);
    try {
      const code = await this.jwt.signAsync({ sub: selfId });
      await bot.sendPrivateMessage(this.centerAccount, `您的密钥是:\n${code}`);
    } catch (e) {
      this.error(`sendCode error for ${selfId}: ${e.message}`);
      throw new BlankReturnMessageDto(500, 'Send message failed').toException();
    }
  }

  async verify(token: string) {
    try {
      const { sub } = await this.jwt.verifyAsync<{ sub: string }>(token);
      return sub;
    } catch (e) {
      throw new BlankReturnMessageDto(403, 'Invalid token').toException();
    }
  }
}
