/*
 * Copyright 2020 Mamoe Technologies and contributors.
 *
 * 此源代码的使用受 GNU AFFERO GENERAL PUBLIC LICENSE version 3 许可证的约束, 可以在以下链接找到该许可证.
 * Use of this source code is governed by the GNU AGPLv3 license that can be found through the following link.
 *
 * https://github.com/mamoe/mirai/blob/master/LICENSE
 */

package net.mamoe.mirai.qqandroid.network.protocol.packet.login

import kotlinx.io.core.ByteReadPacket
import kotlinx.serialization.Serializable
import kotlinx.serialization.protobuf.ProtoId
import net.mamoe.mirai.event.AbstractEvent
import net.mamoe.mirai.event.Event
import net.mamoe.mirai.qqandroid.QQAndroidBot
import net.mamoe.mirai.qqandroid.network.Packet
import net.mamoe.mirai.qqandroid.network.protocol.data.jce.FileStoragePushFSSvcListFuckKotlin
import net.mamoe.mirai.qqandroid.network.protocol.data.jce.PushResp
import net.mamoe.mirai.qqandroid.network.protocol.data.jce.RequestPacket
import net.mamoe.mirai.qqandroid.network.protocol.packet.IncomingPacketFactory
import net.mamoe.mirai.qqandroid.network.protocol.packet.OutgoingPacket
import net.mamoe.mirai.qqandroid.network.protocol.packet.buildResponseUniPacket
import net.mamoe.mirai.qqandroid.utils.ByteArrayPool
import net.mamoe.mirai.qqandroid.utils.hexToBytes
import net.mamoe.mirai.qqandroid.utils.io.ProtoBuf
import net.mamoe.mirai.qqandroid.utils.io.serialization.*
import net.mamoe.mirai.qqandroid.utils.io.withUse
import net.mamoe.mirai.qqandroid.utils.toReadPacket
import net.mamoe.mirai.qqandroid.utils.toUHexString
import net.mamoe.mirai.utils.warning
import net.mamoe.mirai.qqandroid.network.protocol.data.jce.PushReq as PushReqJceStruct


internal class ConfigPushSvc {
    object PushReq : IncomingPacketFactory<PushReq.PushReqResponse?>(
        receivingCommandName = "ConfigPushSvc.PushReq",
        responseCommandName = "ConfigPushSvc.PushResp"
    ) {
        override val canBeCached: Boolean get() = false

        sealed class PushReqResponse : Packet, Event, AbstractEvent() {
            class Success(
                val struct: PushReqJceStruct
            ) : PushReqResponse() {
                override fun toString(): String {
                    return "ConfigPushSvc.PushReq.PushReqResponse.Success"
                }
            }

            @Serializable
            data class ChangeServer(
                @ProtoId(1) val unknown: Int, // =08
                @ProtoId(2) val serverList: List<ServerInfo>
            ) : ProtoBuf, PushReqResponse() {

                @Serializable
                data class ServerInfo(
                    @ProtoId(1) val host: String,
                    @ProtoId(2) val port: Int,
                    @ProtoId(3) val unknown: Int
                ) : ProtoBuf {
                    override fun toString(): String {
                        return "$host:$port"
                    }
                }
            }
        }


        override suspend fun ByteReadPacket.decode(bot: QQAndroidBot, sequenceId: Int): PushReqResponse? {
            ByteArrayPool.useInstance(this.remaining.toInt()) { buffer ->
                val length = this.readAvailable(buffer)

                kotlin.runCatching {
                    val configPush =
                        buffer.toReadPacket(length = length).readUniPacket(PushReqJceStruct.serializer(), "PushReq")

                    // 09 00 01 0A 16 0C 31 38 33 2E 35 37 2E 35 33 2E 31 36 21 1F 90 0B 19 00 01 0A 16 0B 31 38 30 2E 39 36 2E 31 2E 33 30 20 50 0B 29 00 02 0A 16 0D 36 31 2E 31 38 33 2E 31 36 34 2E 34 34 20 50 0B 0A 16 0D 31 31 33 2E 39 36 2E 32 33 32 2E 39 32 20 50 0B 39 00 06 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 34 31 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 34 34 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 34 35 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 32 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 33 20 50 0B 0A 16 11 73 63 61 6E 6E 6F 6E 2E 33 67 2E 71 71 2E 63 6F 6D 20 50 0B 49 00 04 0A 16 0E 31 31 33 2E 39 36 2E 32 33 32 2E 31 30 38 20 50 0B 0A 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 35 21 1F 90 0B 0A 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 21 01 BB 0B 0A 16 0D 31 38 30 2E 31 36 33 2E 32 35 2E 33 38 20 50 0B 5A 09 00 03 0A 00 01 19 00 04 0A 00 01 16 0E 31 31 33 2E 39 36 2E 32 33 32 2E 31 30 38 20 50 0B 0A 00 01 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 35 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 21 01 BB 0B 0A 00 01 16 0D 31 38 30 2E 31 36 33 2E 32 35 2E 33 38 20 50 0B 29 0C 3C 0B 0A 00 05 19 00 04 0A 00 01 16 0E 31 31 33 2E 39 36 2E 32 33 32 2E 31 30 38 20 50 0B 0A 00 01 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 35 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 21 01 BB 0B 0A 00 01 16 0D 31 38 30 2E 31 36 33 2E 32 35 2E 33 38 20 50 0B 29 0C 3C 0B 0A 00 0A 19 00 04 0A 00 01 16 0E 31 31 33 2E 39 36 2E 32 33 32 2E 31 30 38 20 50 0B 0A 00 01 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 35 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 21 01 BB 0B 0A 00 01 16 0D 31 38 30 2E 31 36 33 2E 32 35 2E 33 38 20 50 0B 29 00 05 0A 0C 11 20 00 20 10 30 01 0B 0A 00 01 11 20 00 20 08 30 02 0B 0A 00 02 11 20 00 20 08 30 01 0B 0A 00 03 12 00 01 00 00 20 08 30 02 0B 0A 00 04 11 20 00 20 08 30 02 0B 3C 0B 1D 00 00 68 CA 62 F1 01 C2 AF E6 CF 29 4B 18 71 B5 EE 6B 63 EB F0 0B AB EE A0 5C 20 B9 83 E2 52 F7 BF C7 46 80 BC C3 7F 22 6B 6E 23 42 D0 8F C8 6A C4 F4 49 AA E7 94 EF D4 80 0A E4 8B BF E2 C0 4F FC C5 3F 97 1A E8 0F 0F 7D 06 47 62 C3 C8 07 4F E6 F6 E9 DB CB 4C F5 95 6A AD EC FD D0 46 A5 16 8D 30 02 D5 8A 86 2E 5F E8 D6 8C 2D 00 00 10 33 6E 59 70 73 47 38 52 6E 48 6A 64 51 48 46 54 32 76 E4 B8 DD 40 01 5D 00 01 02 54 8A 50 D0 04 0A 68 CA 62 F1 01 C2 AF E6 CF 29 4B 18 71 B5 EE 6B 63 EB F0 0B AB EE A0 5C 20 B9 83 E2 52 F7 BF C7 46 80 BC C3 7F 22 6B 6E 23 42 D0 8F C8 6A C4 F4 49 AA E7 94 EF D4 80 0A E4 8B BF E2 C0 4F FC C5 3F 97 1A E8 0F 0F 7D 06 47 62 C3 C8 07 4F E6 F6 E9 DB CB 4C F5 95 6A AD EC FD D0 46 A5 16 8D 30 02 D5 8A 86 2E 5F E8 D6 8C 12 10 33 6E 59 70 73 47 38 52 6E 48 6A 64 51 48 46 54 1A 40 08 01 12 0D 08 01 15 71 60 E8 6C 18 50 20 01 28 01 12 0E 08 01 15 B7 03 E9 E1 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 96 4C A8 18 BB 03 20 02 28 00 12 0D 08 01 15 B4 A3 19 26 18 50 20 04 28 00 1A 40 08 05 12 0D 08 01 15 71 60 E8 6C 18 50 20 01 28 01 12 0E 08 01 15 B7 03 E9 E1 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 96 4C A8 18 BB 03 20 02 28 00 12 0D 08 01 15 B4 A3 19 26 18 50 20 04 28 00 1A 78 08 0A 12 0D 08 01 15 71 60 E8 6C 18 50 20 01 28 01 12 0E 08 01 15 B7 03 E9 E1 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 96 4C A8 18 BB 03 20 02 28 00 12 0D 08 01 15 B4 A3 19 26 18 50 20 04 28 00 22 09 08 00 10 80 40 18 10 20 01 22 09 08 01 10 80 40 18 08 20 02 22 09 08 02 10 80 40 18 08 20 01 22 0A 08 03 10 80 80 04 18 08 20 02 22 09 08 04 10 80 40 18 08 20 02 20 01 32 04 08 00 10 01 3A 2A 08 10 10 10 18 09 20 09 28 0F 30 0F 38 05 40 05 48 5A 50 01 58 5A 60 5A 68 5A 70 5A 78 0A 80 01 0A 88 01 0A 90 01 0A 98 01 0A 42 0A 08 00 10 00 18 00 20 00 28 00 4A 06 08 01 10 01 18 03 52 42 08 01 12 0A 08 00 10 80 80 04 18 10 20 02 12 0A 08 01 10 80 80 04 18 08 20 02 12 0A 08 02 10 80 80 01 18 08 20 01 12 0A 08 03 10 80 80 04 18 08 20 02 12 0A 08 04 10 80 80 04 18 08 20 02 18 00 20 00 5A 40 08 01 12 0A 08 00 10 80 80 04 18 10 20 02 12 0A 08 01 10 80 80 04 18 08 20 02 12 0A 08 02 10 80 80 01 18 08 20 01 12 0A 08 03 10 80 80 04 18 08 20 02 12 0A 08 04 10 80 80 04 18 08 20 02 18 00 70 02 78 02 80 01 FA 01 0B 69 00 01 0A 16 26 69 6D 67 63 61 63 68 65 2E 71 71 2E 63 6F 6D 2E 73 63 68 65 64 2E 70 31 76 36 2E 74 64 6E 73 76 36 2E 63 6F 6D 2E 20 50 0B 79 00 02 0A 16 0E 31 30 31 2E 32 32 37 2E 31 33 31 2E 36 37 20 50 0B 0A 16 0D 36 31 2E 31 35 31 2E 31 38 33 2E 32 31 20 50 0B 8A 06 0F 31 37 31 2E 31 31 32 2E 32 32 36 2E 32 33 37 10 03 0B 9A 09 00 0B 0A 00 0F 19 00 01 0A 12 71 19 A3 B4 20 50 0B 29 0C 0B 0A 00 04 19 00 01 0A 12 0B 27 59 65 20 50 0B 29 0C 0B 0A 00 0D 19 00 02 0A 12 55 31 BA DE 20 50 0B 0A 12 5B A0 6A 72 20 50 0B 29 0C 0B 0A 00 03 19 00 02 0A 12 C3 B9 D3 74 20 50 0B 0A 12 CC 43 E4 DD 20 50 0B 29 0C 0B 0A 00 07 19 00 01 0A 12 75 A2 E3 65 20 50 0B 29 0C 0B 0A 00 09 19 00 02 0A 12 BC 6C 24 B7 20 50 0B 0A 12 A6 6C 24 B7 20 50 0B 29 0C 0B 0A 00 0A 19 00 02 0A 12 11 B4 12 0E 20 50 0B 0A 12 15 8C D7 0E 20 50 0B 29 0C 0B 0A 00 05 19 00 01 0A 12 1D E2 03 B7 20 50 0B 29 0C 0B 0A 00 08 19 00 02 0A 12 DE 3F 5B 65 20 50 0B 0A 12 78 09 61 B4 20 50 0B 29 0C 0B 0A 00 06 19 00 02 0A 12 16 CF 97 3D 20 50 0B 0A 12 54 10 59 65 20 50 0B 29 0C 0B 0A 00 0E 19 00 02 0A 12 76 01 B1 6F 20 50 0B 0A 12 6B 89 31 3A 20 50 0B 29 0C 0B 0B AD 00 01 01 5B 08 01 10 A4 F6 AA 16 18 00 22 0A 31 39 39 34 37 30 31 30 32 31 28 AB E1 89 EF 0E 32 12 08 8E A4 D8 A5 09 10 50 18 89 D8 AC F0 08 20 50 28 64 32 12 08 8E A4 C4 DD 08 10 50 18 89 F4 DE E0 05 20 50 28 64 32 13 08 B4 C7 DA B0 02 10 50 18 8A EE D4 F2 0D 20 50 28 C8 01 32 13 08 B4 C7 DA A0 02 10 50 18 8A EC D0 86 0E 20 50 28 C8 01 32 13 08 8C 9D 9B 85 07 10 50 18 89 D6 AD 9C 09 20 50 28 AC 02 32 13 08 B7 81 97 F6 06 10 50 18 8A EC D4 96 02 20 50 28 AC 02 3A 1E 0A 10 24 0E 00 E1 A9 00 00 50 00 00 00 00 00 00 00 29 10 50 18 89 EC 8C B1 05 20 50 28 64 3A 1E 0A 10 24 0E 00 E1 A9 00 00 50 00 00 00 00 00 00 00 64 10 50 18 89 EC 8C D1 07 20 50 28 64 3A 1F 0A 10 24 0E 00 FF F1 01 00 10 00 00 00 00 00 00 01 6F 10 50 18 E4 E6 B1 F0 04 20 50 28 C8 01 3A 1F 0A 10 24 0E 00 FF F1 01 00 10 00 00 00 00 00 00 01 72 10 50 18 E4 E6 AD F0 0E 20 50 28 C8 01 3A 1F 0A 10 24 09 8C 1E 75 B0 00 13 00 00 00 00 00 00 00 36 10 50 18 89 EC 9C E8 0D 20 50 28 AC 02 3A 1F 0A 10 24 09 8C 54 10 03 00 10 00 00 00 00 00 00 00 55 10 50 18 89 CA 8C A0 01 20 50 28 AC 02

                    // ??
                    // 19 00 0A 0A 16 0C 31 31 33 2E 39 36 2E 31 32 2E 38 35 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 33 33 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 0D 31 31 33 2E 39 36 2E 31 32 2E 32 32 34 21 1F 90 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0C 34 32 2E 38 31 2E 31 37 32 2E 36 33 21 01 BB 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 37 36 20 50 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 0D 31 31 33 2E 39 36 2E 31 32 2E 32 31 37 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 37 30 2E 31 32 32 21 01 BB 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 38 2E 31 34 21 1F 90 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 11 6D 73 66 77 69 66 69 2E 33 67 2E 71 71 2E 63 6F 6D 21 1F 90 30 01 4C 5C 60 08 7C 86 06 6F 74 68 65 72 73 96 06 6F 74 68 65 72 73 0B 0A 16 0D 34 32 2E 38 31 2E 31 37 32 2E 32 30 37 20 50 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 39 00 0A 0A 16 0C 31 31 33 2E 39 36 2E 31 32 2E 38 35 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 33 33 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 0D 31 31 33 2E 39 36 2E 31 32 2E 32 32 34 21 1F 90 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0C 34 32 2E 38 31 2E 31 37 32 2E 36 33 21 01 BB 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 37 36 20 50 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 0D 31 31 33 2E 39 36 2E 31 32 2E 32 31 37 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 37 30 2E 31 32 32 21 01 BB 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 38 2E 31 34 21 1F 90 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 11 6D 73 66 77 69 66 69 2E 33 67 2E 71 71 2E 63 6F 6D 21 1F 90 30 01 4C 5C 60 08 7C 86 06 6F 74 68 65 72 73 96 06 6F 74 68 65 72 73 0B 0A 16 0D 34 32 2E 38 31 2E 31 37 32 2E 32 30 37 20 50 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 4C 5C 6C 70 01 89 00 04 0A 16 0C 31 31 33 2E 39 36 2E 31 33 2E 34 34 20 50 30 01 4C 50 03 60 08 7C 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 32 20 50 30 01 4C 50 03 60 08 7C 86 02 73 68 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 20 50 30 01 4C 50 03 60 08 7C 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 20 50 30 01 4C 50 03 60 08 7C 86 02 74 6A 96 03 74 65 6C 0B 99 00 04 0A 16 0C 31 31 33 2E 39 36 2E 31 33 2E 34 34 20 50 30 01 4C 50 03 60 08 7C 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 32 20 50 30 01 4C 50 03 60 08 7C 86 02 73 68 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 20 50 30 01 4C 50 03 60 08 7C 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 20 50 30 01 4C 50 03 60 08 7C 86 02 74 6A 96 03 74 65 6C 0B A9 0C B9 0C C9 0C D9 0C E0 01 FC 0F F6 10 28 32 30 32 30 2D 30 34 2D 31 32 20 31 39 3A 31 30 3A 30 38 20 64 65 6C 69 76 65 72 79 69 6E 67 20 61 20 70 6F 6C 69 63 79

                    kotlin.runCatching {
                        configPush.jcebuf.loadAs(FileStoragePushFSSvcListFuckKotlin.serializer()).also {
                            bot.client.fileStoragePushFSSvcList = it
                        }
                    }.getOrElse {
                        val default = """
                            09 00 01 0A 16 0C 31 38 33 2E 35 37 2E 35 33 2E 31 36 21 1F 90 0B 19 00 01 0A 16 0B 31 38 30 2E 39 36 2E 31 2E 33 30 20 50 0B 29 00 02 0A 16 0D 36 31 2E 31 38 33 2E 31 36 34 2E 34 34 20 50 0B 0A 16 0D 31 31 33 2E 39 36 2E 32 33 32 2E 39 32 20 50 0B 39 00 06 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 34 31 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 34 34 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 34 35 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 32 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 33 20 50 0B 0A 16 11 73 63 61 6E 6E 6F 6E 2E 33 67 2E 71 71 2E 63 6F 6D 20 50 0B 49 00 04 0A 16 0E 31 31 33 2E 39 36 2E 32 33 32 2E 31 30 38 20 50 0B 0A 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 35 21 1F 90 0B 0A 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 21 01 BB 0B 0A 16 0D 31 38 30 2E 31 36 33 2E 32 35 2E 33 38 20 50 0B 5A 09 00 03 0A 00 01 19 00 04 0A 00 01 16 0E 31 31 33 2E 39 36 2E 32 33 32 2E 31 30 38 20 50 0B 0A 00 01 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 35 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 21 01 BB 0B 0A 00 01 16 0D 31 38 30 2E 31 36 33 2E 32 35 2E 33 38 20 50 0B 29 0C 3C 0B 0A 00 05 19 00 04 0A 00 01 16 0E 31 31 33 2E 39 36 2E 32 33 32 2E 31 30 38 20 50 0B 0A 00 01 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 35 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 21 01 BB 0B 0A 00 01 16 0D 31 38 30 2E 31 36 33 2E 32 35 2E 33 38 20 50 0B 29 0C 3C 0B 0A 00 0A 19 00 04 0A 00 01 16 0E 31 31 33 2E 39 36 2E 32 33 32 2E 31 30 38 20 50 0B 0A 00 01 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 35 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 21 01 BB 0B 0A 00 01 16 0D 31 38 30 2E 31 36 33 2E 32 35 2E 33 38 20 50 0B 29 00 05 0A 0C 11 20 00 20 10 30 01 0B 0A 00 01 11 20 00 20 08 30 02 0B 0A 00 02 11 20 00 20 08 30 01 0B 0A 00 03 12 00 01 00 00 20 08 30 02 0B 0A 00 04 11 20 00 20 08 30 02 0B 3C 0B 1D 00 00 68 CA 62 F1 01 C2 AF E6 CF 29 4B 18 71 B5 EE 6B 63 EB F0 0B AB EE A0 5C 20 B9 83 E2 52 F7 BF C7 46 80 BC C3 7F 22 6B 6E 23 42 D0 8F C8 6A C4 F4 49 AA E7 94 EF D4 80 0A E4 8B BF E2 C0 4F FC C5 3F 97 1A E8 0F 0F 7D 06 47 62 C3 C8 07 4F E6 F6 E9 DB CB 4C F5 95 6A AD EC FD D0 46 A5 16 8D 30 02 D5 8A 86 2E 5F E8 D6 8C 2D 00 00 10 33 6E 59 70 73 47 38 52 6E 48 6A 64 51 48 46 54 32 76 E4 B8 DD 40 01 5D 00 01 02 54 8A 50 D0 04 0A 68 CA 62 F1 01 C2 AF E6 CF 29 4B 18 71 B5 EE 6B 63 EB F0 0B AB EE A0 5C 20 B9 83 E2 52 F7 BF C7 46 80 BC C3 7F 22 6B 6E 23 42 D0 8F C8 6A C4 F4 49 AA E7 94 EF D4 80 0A E4 8B BF E2 C0 4F FC C5 3F 97 1A E8 0F 0F 7D 06 47 62 C3 C8 07 4F E6 F6 E9 DB CB 4C F5 95 6A AD EC FD D0 46 A5 16 8D 30 02 D5 8A 86 2E 5F E8 D6 8C 12 10 33 6E 59 70 73 47 38 52 6E 48 6A 64 51 48 46 54 1A 40 08 01 12 0D 08 01 15 71 60 E8 6C 18 50 20 01 28 01 12 0E 08 01 15 B7 03 E9 E1 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 96 4C A8 18 BB 03 20 02 28 00 12 0D 08 01 15 B4 A3 19 26 18 50 20 04 28 00 1A 40 08 05 12 0D 08 01 15 71 60 E8 6C 18 50 20 01 28 01 12 0E 08 01 15 B7 03 E9 E1 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 96 4C A8 18 BB 03 20 02 28 00 12 0D 08 01 15 B4 A3 19 26 18 50 20 04 28 00 1A 78 08 0A 12 0D 08 01 15 71 60 E8 6C 18 50 20 01 28 01 12 0E 08 01 15 B7 03 E9 E1 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 96 4C A8 18 BB 03 20 02 28 00 12 0D 08 01 15 B4 A3 19 26 18 50 20 04 28 00 22 09 08 00 10 80 40 18 10 20 01 22 09 08 01 10 80 40 18 08 20 02 22 09 08 02 10 80 40 18 08 20 01 22 0A 08 03 10 80 80 04 18 08 20 02 22 09 08 04 10 80 40 18 08 20 02 20 01 32 04 08 00 10 01 3A 2A 08 10 10 10 18 09 20 09 28 0F 30 0F 38 05 40 05 48 5A 50 01 58 5A 60 5A 68 5A 70 5A 78 0A 80 01 0A 88 01 0A 90 01 0A 98 01 0A 42 0A 08 00 10 00 18 00 20 00 28 00 4A 06 08 01 10 01 18 03 52 42 08 01 12 0A 08 00 10 80 80 04 18 10 20 02 12 0A 08 01 10 80 80 04 18 08 20 02 12 0A 08 02 10 80 80 01 18 08 20 01 12 0A 08 03 10 80 80 04 18 08 20 02 12 0A 08 04 10 80 80 04 18 08 20 02 18 00 20 00 5A 40 08 01 12 0A 08 00 10 80 80 04 18 10 20 02 12 0A 08 01 10 80 80 04 18 08 20 02 12 0A 08 02 10 80 80 01 18 08 20 01 12 0A 08 03 10 80 80 04 18 08 20 02 12 0A 08 04 10 80 80 04 18 08 20 02 18 00 70 02 78 02 80 01 FA 01 0B 69 00 01 0A 16 26 69 6D 67 63 61 63 68 65 2E 71 71 2E 63 6F 6D 2E 73 63 68 65 64 2E 70 31 76 36 2E 74 64 6E 73 76 36 2E 63 6F 6D 2E 20 50 0B 79 00 02 0A 16 0E 31 30 31 2E 32 32 37 2E 31 33 31 2E 36 37 20 50 0B 0A 16 0D 36 31 2E 31 35 31 2E 31 38 33 2E 32 31 20 50 0B 8A 06 0F 31 37 31 2E 31 31 32 2E 32 32 36 2E 32 33 37 10 03 0B 9A 09 00 0B 0A 00 0F 19 00 01 0A 12 71 19 A3 B4 20 50 0B 29 0C 0B 0A 00 04 19 00 01 0A 12 0B 27 59 65 20 50 0B 29 0C 0B 0A 00 0D 19 00 02 0A 12 55 31 BA DE 20 50 0B 0A 12 5B A0 6A 72 20 50 0B 29 0C 0B 0A 00 03 19 00 02 0A 12 C3 B9 D3 74 20 50 0B 0A 12 CC 43 E4 DD 20 50 0B 29 0C 0B 0A 00 07 19 00 01 0A 12 75 A2 E3 65 20 50 0B 29 0C 0B 0A 00 09 19 00 02 0A 12 BC 6C 24 B7 20 50 0B 0A 12 A6 6C 24 B7 20 50 0B 29 0C 0B 0A 00 0A 19 00 02 0A 12 11 B4 12 0E 20 50 0B 0A 12 15 8C D7 0E 20 50 0B 29 0C 0B 0A 00 05 19 00 01 0A 12 1D E2 03 B7 20 50 0B 29 0C 0B 0A 00 08 19 00 02 0A 12 DE 3F 5B 65 20 50 0B 0A 12 78 09 61 B4 20 50 0B 29 0C 0B 0A 00 06 19 00 02 0A 12 16 CF 97 3D 20 50 0B 0A 12 54 10 59 65 20 50 0B 29 0C 0B 0A 00 0E 19 00 02 0A 12 76 01 B1 6F 20 50 0B 0A 12 6B 89 31 3A 20 50 0B 29 0C 0B 0B AD 00 01 01 5B 08 01 10 A4 F6 AA 16 18 00 22 0A 31 39 39 34 37 30 31 30 32 31 28 AB E1 89 EF 0E 32 12 08 8E A4 D8 A5 09 10 50 18 89 D8 AC F0 08 20 50 28 64 32 12 08 8E A4 C4 DD 08 10 50 18 89 F4 DE E0 05 20 50 28 64 32 13 08 B4 C7 DA B0 02 10 50 18 8A EE D4 F2 0D 20 50 28 C8 01 32 13 08 B4 C7 DA A0 02 10 50 18 8A EC D0 86 0E 20 50 28 C8 01 32 13 08 8C 9D 9B 85 07 10 50 18 89 D6 AD 9C 09 20 50 28 AC 02 32 13 08 B7 81 97 F6 06 10 50 18 8A EC D4 96 02 20 50 28 AC 02 3A 1E 0A 10 24 0E 00 E1 A9 00 00 50 00 00 00 00 00 00 00 29 10 50 18 89 EC 8C B1 05 20 50 28 64 3A 1E 0A 10 24 0E 00 E1 A9 00 00 50 00 00 00 00 00 00 00 64 10 50 18 89 EC 8C D1 07 20 50 28 64 3A 1F 0A 10 24 0E 00 FF F1 01 00 10 00 00 00 00 00 00 01 6F 10 50 18 E4 E6 B1 F0 04 20 50 28 C8 01 3A 1F 0A 10 24 0E 00 FF F1 01 00 10 00 00 00 00 00 00 01 72 10 50 18 E4 E6 AD F0 0E 20 50 28 C8 01 3A 1F 0A 10 24 09 8C 1E 75 B0 00 13 00 00 00 00 00 00 00 36 10 50 18 89 EC 9C E8 0D 20 50 28 AC 02 3A 1F 0A 10 24 09 8C 54 10 03 00 10 00 00 00 00 00 00 00 55 10 50 18 89 CA 8C A0 01 20 50 28 AC 02
                        """.trimIndent().hexToBytes()

                        val client = bot.client


                        client.fileStoragePushFSSvcList =
                            default.loadAs(FileStoragePushFSSvcListFuckKotlin.serializer())

                        Unit
                        // 19 00 0A 0A 16 0F 31 31 34 2E 32 32 31 2E 31 34 34 2E 31 36 30 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 0C 31 31 33 2E 39 36 2E 31 33 2E 39 35 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0D 31 31 33 2E 39 36 2E 31 32 2E 32 31 37 21 1F 90 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 37 30 2E 31 32 32 21 01 BB 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0F 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 31 32 20 50 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 0A 31 34 2E 32 32 2E 33 2E 33 35 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 37 32 2E 31 35 36 21 01 BB 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 31 39 21 1F 90 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 11 6D 73 66 77 69 66 69 2E 33 67 2E 71 71 2E 63 6F 6D 21 1F 90 30 01 4C 5C 60 08 7C 86 06 6F 74 68 65 72 73 96 06 6F 74 68 65 72 73 0B 0A 16 0D 34 32 2E 38 31 2E 31 37 30 2E 31 32 32 20 50 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 39 00 0A 0A 16 0F 31 31 34 2E 32 32 31 2E 31 34 34 2E 31 36 30 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 0C 31 31 33 2E 39 36 2E 31 33 2E 39 35 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0D 31 31 33 2E 39 36 2E 31 32 2E 32 31 37 21 1F 90 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 37 30 2E 31 32 32 21 01 BB 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0F 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 31 32 20 50 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 0A 31 34 2E 32 32 2E 33 2E 33 35 21 36 B0 30 01 4C 5C 60 08 70 01 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 37 32 2E 31 35 36 21 01 BB 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 31 39 21 1F 90 30 01 4C 5C 60 08 70 01 86 02 73 68 96 03 74 65 6C 0B 0A 16 11 6D 73 66 77 69 66 69 2E 33 67 2E 71 71 2E 63 6F 6D 21 1F 90 30 01 4C 5C 60 08 7C 86 06 6F 74 68 65 72 73 96 06 6F 74 68 65 72 73 0B 0A 16 0D 34 32 2E 38 31 2E 31 37 30 2E 31 32 32 20 50 30 01 4C 5C 60 08 70 01 86 02 74 6A 96 03 74 65 6C 0B 4C 5C 6C 70 01 89 00 04 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 32 20 50 30 01 4C 50 03 60 08 7C 86 02 73 68 96 03 74 65 6C 0B 0A 16 0C 31 31 33 2E 39 36 2E 31 33 2E 34 34 20 50 30 01 4C 50 03 60 08 7C 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 20 50 30 01 4C 50 03 60 08 7C 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 20 50 30 01 4C 50 03 60 08 7C 86 02 74 6A 96 03 74 65 6C 0B 99 00 04 0A 16 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 32 20 50 30 01 4C 50 03 60 08 7C 86 02 73 68 96 03 74 65 6C 0B 0A 16 0C 31 31 33 2E 39 36 2E 31 33 2E 34 34 20 50 30 01 4C 50 03 60 08 7C 86 02 73 7A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 20 50 30 01 4C 50 03 60 08 7C 86 02 74 6A 96 03 74 65 6C 0B 0A 16 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 20 50 30 01 4C 50 03 60 08 7C 86 02 74 6A 96 03 74 65 6C 0B A9 0C B9 0C C9 0C D9 0C E0 01 FC 0F F6 10 28 32 30 32 30 2D 30 34 2D 31 34 20 32 31 3A 33 31 3A 31 36 20 64 65 6C 69 76 65 72 79 69 6E 67 20 61 20 70 6F 6C 69 63 79
                        // failed
                    }

                    return PushReqResponse.Success(configPush)
                }.recoverCatching {
                    // 02 00 00 01 2B 00 01 00 02 CB 1E 16 27 08 08 12 13 0A 0C 34 32 2E 38 31 2E 31 36 39 2E 34 36 10 90 3F 18 00 12 12 0A 0C 34 32 2E 38 31 2E 31 37 32 2E 38 31 10 50 18 00 12 15 0A 0E 31 31 34 2E 32 32 31 2E 31 34 38 2E 35 39 10 B0 6D 18 00 12 14 0A 0D 34 32 2E 38 31 2E 31 37 32 2E 31 34 37 10 BB 03 18 00 12 13 0A 0D 31 32 35 2E 39 34 2E 36 30 2E 31 34 36 10 50 18 00 12 15 0A 0F 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 31 35 10 50 18 00 12 18 0A 11 6D 73 66 77 69 66 69 2E 33 67 2E 71 71 2E 63 6F 6D 10 90 3F 18 00 12 12 0A 0C 34 32 2E 38 31 2E 31 37 32 2E 32 32 10 50 18 00 1A 13 0A 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 10 50 18 00 1A 14 0A 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 32 10 50 18 00 1A 14 0A 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 32 10 50 18 00 1A 13 0A 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 10 50 18 00 1A 13 0A 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 10 50 18 00 03

                    // tail = 0x03
                    buffer.toReadPacket(offset = 13, length = length - 1).withUse {
                        return readProtoBuf(PushReqResponse.ChangeServer.serializer())
                    }
                }.getOrElse {
                    bot.network.logger.warning { "Cannot decode ConfigPushSvc.PushReq, data=${buffer.toUHexString()}" }
                    bot.network.logger.warning(it)
                    return null
                }
            }

            // 114.221.144.89 8080
            // 113.96.13.125 80
            // 42.81.172.63 14000

            // println(bytes.toUHexString())
            // 10 02 2C 3C 4C 56 23 51 51 53 65 72 76 69 63 65 2E 43 6F 6E 66 69 67 50 75 73 68 53 76 63 2E 4D 61 69 6E 53 65 72 76 61 6E 74 66 07 50 75 73 68 52 65 71 7D 00 01 08 7E 08 00 01 06 07 50 75 73 68 52 65 71 18 00 01 06 12 43 6F 6E 66 69 67 50 75 73 68 2E 50 75 73 68 52 65 71 1D 00 01 08 56 0A 10 02 2D 00 01 08 44 09 00 01 0A 16 0C 31 38 33 2E 35 37 2E 35 33 2E 31 36 21 1F 90 0B 19 00 01 0A 16 0B 31 38 30 2E 39 36 2E 31 2E 33 30 20 50 0B 29 00 02 0A 16 0D 36 31 2E 31 38 33 2E 31 36 34 2E 32 37 20 50 0B 0A 16 0B 31 34 2E 31 37 2E 34 33 2E 34 38 20 50 0B 39 00 06 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 33 35 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 34 31 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 34 34 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 34 35 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 32 20 50 0B 0A 16 11 73 63 61 6E 6E 6F 6E 2E 33 67 2E 71 71 2E 63 6F 6D 20 50 0B 49 00 04 0A 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 37 30 21 01 BB 0B 0A 16 0C 35 39 2E 33 36 2E 38 39 2E 32 35 32 21 1F 90 0B 0A 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 20 50 0B 0A 16 0C 31 30 31 2E 39 31 2E 35 2E 31 38 37 21 01 BB 0B 5A 09 00 03 0A 00 01 19 00 04 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 37 30 21 01 BB 0B 0A 00 01 16 0C 35 39 2E 33 36 2E 38 39 2E 32 35 32 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 20 50 0B 0A 00 01 16 0C 31 30 31 2E 39 31 2E 35 2E 31 38 37 21 01 BB 0B 29 0C 3C 0B 0A 00 05 19 00 04 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 37 30 21 01 BB 0B 0A 00 01 16 0C 35 39 2E 33 36 2E 38 39 2E 32 35 32 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 20 50 0B 0A 00 01 16 0C 31 30 31 2E 39 31 2E 35 2E 31 38 37 21 01 BB 0B 29 0C 3C 0B 0A 00 0A 19 00 04 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 37 30 21 01 BB 0B 0A 00 01 16 0C 35 39 2E 33 36 2E 38 39 2E 32 35 32 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 36 38 20 50 0B 0A 00 01 16 0C 31 30 31 2E 39 31 2E 35 2E 31 38 37 21 01 BB 0B 29 00 05 0A 0C 11 20 00 20 10 30 01 0B 0A 00 01 11 20 00 20 08 30 02 0B 0A 00 02 11 20 00 20 08 30 01 0B 0A 00 03 12 00 01 00 00 20 08 30 02 0B 0A 00 04 11 20 00 20 08 30 02 0B 3C 0B 1D 00 00 68 67 84 36 BC AB 01 05 B2 79 8B F8 C3 51 00 DA AF 1E EF D8 E6 01 AF 08 05 B5 8B A6 61 9B 1B 1C 5A 0B AC FB D8 4C FB 2D 47 3F D0 8D 56 2D 6C FF 9B 48 B0 1D BC 14 34 F8 64 36 F2 0D EA 8B 63 C5 CC 37 54 0A A0 81 27 7D B8 91 EB 88 DC 69 2B 5C 88 BD 7B D3 B6 31 33 46 E1 BA BE A3 88 52 17 8B E6 11 5F DA C0 D2 DA 31 BB 2D 00 00 10 78 56 76 4A 44 49 62 33 6B 76 53 52 61 62 74 52 32 76 E4 B8 DD 40 01 5D 00 01 02 4F 8A 50 CB 04 0A 68 67 84 36 BC AB 01 05 B2 79 8B F8 C3 51 00 DA AF 1E EF D8 E6 01 AF 08 05 B5 8B A6 61 9B 1B 1C 5A 0B AC FB D8 4C FB 2D 47 3F D0 8D 56 2D 6C FF 9B 48 B0 1D BC 14 34 F8 64 36 F2 0D EA 8B 63 C5 CC 37 54 0A A0 81 27 7D B8 91 EB 88 DC 69 2B 5C 88 BD 7B D3 B6 31 33 46 E1 BA BE A3 88 52 17 8B E6 11 5F DA C0 D2 DA 31 BB 12 10 78 56 76 4A 44 49 62 33 6B 76 53 52 61 62 74 52 1A 41 08 01 12 0E 08 01 15 7B 96 4C AA 18 BB 03 20 02 28 01 12 0E 08 01 15 3B 24 59 FC 18 90 3F 20 01 28 01 12 0D 08 01 15 7B 96 4C A8 18 50 20 02 28 00 12 0E 08 01 15 65 5B 05 BB 18 BB 03 20 04 28 00 1A 41 08 05 12 0E 08 01 15 7B 96 4C AA 18 BB 03 20 02 28 01 12 0E 08 01 15 3B 24 59 FC 18 90 3F 20 01 28 01 12 0D 08 01 15 7B 96 4C A8 18 50 20 02 28 00 12 0E 08 01 15 65 5B 05 BB 18 BB 03 20 04 28 00 1A 79 08 0A 12 0E 08 01 15 7B 96 4C AA 18 BB 03 20 02 28 01 12 0E 08 01 15 3B 24 59 FC 18 90 3F 20 01 28 01 12 0D 08 01 15 7B 96 4C A8 18 50 20 02 28 00 12 0E 08 01 15 65 5B 05 BB 18 BB 03 20 04 28 00 22 09 08 00 10 80 40 18 10 20 01 22 09 08 01 10 80 40 18 08 20 02 22 09 08 02 10 80 40 18 08 20 01 22 0A 08 03 10 80 80 04 18 08 20 02 22 09 08 04 10 80 40 18 08 20 02 20 01 32 04 08 00 10 01 3A 2A 08 10 10 10 18 09 20 09 28 0F 30 0F 38 05 40 05 48 5A 50 01 58 5A 60 5A 68 5A 70 5A 78 0A 80 01 0A 88 01 0A 90 01 0A 98 01 0A 42 0A 08 00 10 00 18 00 20 00 28 00 4A 06 08 01 10 01 18 03 52 42 08 01 12 0A 08 00 10 80 80 04 18 10 20 02 12 0A 08 01 10 80 80 04 18 08 20 02 12 0A 08 02 10 80 80 01 18 08 20 01 12 0A 08 03 10 80 80 04 18 08 20 02 12 0A 08 04 10 80 80 04 18 08 20 02 18 00 20 00 5A 40 08 01 12 0A 08 00 10 80 80 04 18 10 20 02 12 0A 08 01 10 80 80 04 18 08 20 02 12 0A 08 02 10 80 80 01 18 08 20 01 12 0A 08 03 10 80 80 04 18 08 20 02 12 0A 08 04 10 80 80 04 18 08 20 02 18 00 0B 69 00 01 0A 16 26 69 6D 67 63 61 63 68 65 2E 71 71 2E 63 6F 6D 2E 73 63 68 65 64 2E 70 31 76 36 2E 74 64 6E 73 76 36 2E 63 6F 6D 2E 20 50 0B 79 00 02 0A 16 0E 31 30 31 2E 32 32 37 2E 31 33 31 2E 36 37 20 50 0B 0A 16 0C 31 30 31 2E 38 39 2E 33 39 2E 32 31 20 50 0B 8A 06 0E 31 37 31 2E 31 31 32 2E 32 32 34 2E 31 30 10 03 0B 9A 09 00 0C 0A 00 0F 19 00 01 0A 12 71 19 A3 B4 20 50 0B 29 0C 0B 0A 00 04 19 00 01 0A 12 0B 27 59 65 20 50 0B 29 0C 0B 0A 00 0D 19 00 02 0A 12 5B A0 6A 72 20 50 0B 0A 12 71 EB 3F 3B 20 50 0B 29 0C 0B 0A 00 03 19 00 02 0A 12 5B A0 6A 72 20 50 0B 0A 12 71 EB 3F 3B 20 50 0B 29 0C 0B 0A 00 07 19 00 01 0A 12 75 A2 E3 65 20 50 0B 29 0C 0B 0A 00 09 19 00 02 0A 12 15 8C D7 0E 20 50 0B 0A 12 18 8C D7 0E 20 50 0B 29 0C 0B 0A 00 0A 19 00 02 0A 12 15 8C D7 0E 20 50 0B 0A 12 18 8C D7 0E 20 50 0B 29 0C 0B 0A 00 0B 19 00 02 0A 12 6D 01 B1 6F 20 50 0B 0A 12 4D 01 B1 6F 20 50 0B 29 0C 0B 0A 00 05 19 00 01 0A 12 1D E2 03 B7 20 50 0B 29 0C 0B 0A 00 08 19 00 02 0A 12 DF 3F 5B 65 20 50 0B 0A 12 DE 3F 5B 65 20 50 0B 29 0C 0B 0A 00 06 19 00 02 0A 12 2C B7 97 3D 20 50 0B 0A 12 1B 45 5B 65 20 50 0B 29 0C 0B 0A 00 0E 19 00 01 0A 12 76 01 B1 6F 20 50 0B 29 0C 0B 0B AD 00 01 01 5A 08 01 10 80 EE D3 1D 18 00 22 0A 31 39 39 34 37 30 31 30 32 31 28 AB E1 81 57 32 12 08 8E A4 D8 9D 0A 10 50 18 89 D8 AC A0 0E 20 50 28 64 32 12 08 8E A4 D8 B5 0A 10 50 18 89 D8 AC B0 09 20 50 28 64 32 13 08 B4 C7 DA F0 01 10 50 18 8A EE D4 92 08 20 50 28 C8 01 32 13 08 E5 B6 F9 A1 09 10 50 18 89 88 E0 B4 08 20 50 28 C8 01 32 13 08 DF CF DE A2 07 10 50 18 8A EE D4 82 0E 20 50 28 AC 02 32 13 08 F8 98 CB C0 0A 10 50 18 E4 E0 8D A5 0E 20 50 28 AC 02 3A 1E 0A 10 24 0E 00 E1 A9 00 00 10 00 00 00 00 00 00 00 1F 10 50 18 89 88 E0 CC 04 20 50 28 64 3A 1E 0A 10 24 0E 00 E1 A9 00 00 10 00 00 00 00 00 00 00 42 10 50 18 89 88 E0 F4 0C 20 50 28 64 3A 1F 0A 10 24 02 4E 00 80 20 00 02 00 00 00 00 00 00 00 A9 10 50 18 89 E6 80 B8 02 20 50 28 C8 01 3A 1F 0A 10 24 02 4E 00 80 20 00 02 00 00 00 00 00 00 00 A7 10 50 18 89 E6 80 A0 06 20 50 28 C8 01 3A 1F 0A 10 24 08 80 F1 00 31 00 10 00 00 00 00 00 00 00 40 10 50 18 89 88 8C DC 06 20 50 28 AC 02 3A 1F 0A 10 24 02 4E 00 80 10 00 00 00 00 00 00 00 00 01 58 10 50 18 89 DC C4 DC 03 20 50 28 AC 02 33 00 00 00 02 6C 97 2A BD 0B 8C 98 0C A8 0C
            // 10 02 2C 3C 4C 56 23 51 51 53 65 72 76 69 63 65 2E 43 6F 6E 66 69 67 50 75 73 68 53 76 63 2E 4D 61 69 6E 53 65 72 76 61 6E 74 66 07 50 75 73 68 52 65 71 7D 00 01 08 82 08 00 01 06 07 50 75 73 68 52 65 71 18 00 01 06 12 43 6F 6E 66 69 67 50 75 73 68 2E 50 75 73 68 52 65 71 1D 00 01 08 5A 0A 10 02 2D 00 01 08 4C 09 00 01 0A 16 0C 31 38 33 2E 35 37 2E 35 33 2E 31 36 21 1F 90 0B 19 00 01 0A 16 0B 31 38 30 2E 39 36 2E 31 2E 33 30 20 50 0B 29 00 02 0A 16 0D 36 31 2E 31 38 33 2E 31 36 34 2E 33 35 20 50 0B 0A 16 0D 31 38 30 2E 39 37 2E 31 30 2E 31 31 31 20 50 0B 39 00 06 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 32 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 33 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 34 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 33 33 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 33 34 20 50 0B 0A 16 11 73 63 61 6E 6E 6F 6E 2E 33 67 2E 71 71 2E 63 6F 6D 20 50 0B 49 00 04 0A 16 0C 31 38 33 2E 33 2E 32 32 35 2E 35 38 20 50 0B 0A 16 0C 35 39 2E 33 36 2E 38 39 2E 32 35 32 21 1F 90 0B 0A 16 0F 31 32 33 2E 31 35 31 2E 31 39 30 2E 31 36 32 21 01 BB 0B 0A 16 0C 31 30 31 2E 39 31 2E 35 2E 31 39 34 20 50 0B 5A 09 00 03 0A 00 01 19 00 04 0A 00 01 16 0C 31 38 33 2E 33 2E 32 32 35 2E 35 38 20 50 0B 0A 00 01 16 0C 35 39 2E 33 36 2E 38 39 2E 32 35 32 21 1F 90 0B 0A 00 01 16 0F 31 32 33 2E 31 35 31 2E 31 39 30 2E 31 36 32 21 01 BB 0B 0A 00 01 16 0C 31 30 31 2E 39 31 2E 35 2E 31 39 34 20 50 0B 29 0C 3C 0B 0A 00 05 19 00 04 0A 00 01 16 0C 31 38 33 2E 33 2E 32 32 35 2E 35 38 20 50 0B 0A 00 01 16 0C 35 39 2E 33 36 2E 38 39 2E 32 35 32 21 1F 90 0B 0A 00 01 16 0F 31 32 33 2E 31 35 31 2E 31 39 30 2E 31 36 32 21 01 BB 0B 0A 00 01 16 0C 31 30 31 2E 39 31 2E 35 2E 31 39 34 20 50 0B 29 0C 3C 0B 0A 00 0A 19 00 04 0A 00 01 16 0C 31 38 33 2E 33 2E 32 32 35 2E 35 38 20 50 0B 0A 00 01 16 0C 35 39 2E 33 36 2E 38 39 2E 32 35 32 21 1F 90 0B 0A 00 01 16 0F 31 32 33 2E 31 35 31 2E 31 39 30 2E 31 36 32 21 01 BB 0B 0A 00 01 16 0C 31 30 31 2E 39 31 2E 35 2E 31 39 34 20 50 0B 29 00 05 0A 0C 11 20 00 20 10 30 01 0B 0A 00 01 11 20 00 20 08 30 02 0B 0A 00 02 11 20 00 20 08 30 01 0B 0A 00 03 12 00 01 00 00 20 08 30 02 0B 0A 00 04 11 20 00 20 08 30 02 0B 3C 0B 1D 00 00 68 2E 0A 92 69 8B 45 38 A1 78 17 9F B5 DB FF 79 C1 EE A3 D7 68 B5 FC C1 3D 20 DD 4B 3B 3E D3 58 65 7E 0B D2 B8 A1 7B 0D 46 52 7E 10 53 18 0E 83 5D C3 3C 5F 1D DB 74 AE 63 3D 51 A1 2C 53 2D 2F CA DD 1C 37 A8 D6 C5 EA FE C3 2D 7B 16 CC 12 13 D3 27 19 57 31 8D 89 8C 44 71 21 BE 87 30 F2 A3 3D 2D 93 E1 0F F1 D1 22 45 2D 00 00 10 47 75 6D 44 64 44 51 6B 76 58 6E 61 66 4A 52 35 32 76 E4 B8 DD 40 01 5D 00 01 02 54 8A 50 D0 04 0A 68 2E 0A 92 69 8B 45 38 A1 78 17 9F B5 DB FF 79 C1 EE A3 D7 68 B5 FC C1 3D 20 DD 4B 3B 3E D3 58 65 7E 0B D2 B8 A1 7B 0D 46 52 7E 10 53 18 0E 83 5D C3 3C 5F 1D DB 74 AE 63 3D 51 A1 2C 53 2D 2F CA DD 1C 37 A8 D6 C5 EA FE C3 2D 7B 16 CC 12 13 D3 27 19 57 31 8D 89 8C 44 71 21 BE 87 30 F2 A3 3D 2D 93 E1 0F F1 D1 22 45 12 10 47 75 6D 44 64 44 51 6B 76 58 6E 61 66 4A 52 35 1A 40 08 01 12 0D 08 01 15 B7 03 E1 3A 18 50 20 01 28 01 12 0E 08 01 15 3B 24 59 FC 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 97 BE A2 18 BB 03 20 02 28 00 12 0D 08 01 15 65 5B 05 C2 18 50 20 04 28 00 1A 40 08 05 12 0D 08 01 15 B7 03 E1 3A 18 50 20 01 28 01 12 0E 08 01 15 3B 24 59 FC 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 97 BE A2 18 BB 03 20 02 28 00 12 0D 08 01 15 65 5B 05 C2 18 50 20 04 28 00 1A 78 08 0A 12 0D 08 01 15 B7 03 E1 3A 18 50 20 01 28 01 12 0E 08 01 15 3B 24 59 FC 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 97 BE A2 18 BB 03 20 02 28 00 12 0D 08 01 15 65 5B 05 C2 18 50 20 04 28 00 22 09 08 00 10 80 40 18 10 20 01 22 09 08 01 10 80 40 18 08 20 02 22 09 08 02 10 80 40 18 08 20 01 22 0A 08 03 10 80 80 04 18 08 20 02 22 09 08 04 10 80 40 18 08 20 02 20 01 32 04 08 00 10 01 3A 2A 08 10 10 10 18 09 20 09 28 0F 30 0F 38 05 40 05 48 5A 50 01 58 5A 60 5A 68 5A 70 5A 78 0A 80 01 0A 88 01 0A 90 01 0A 98 01 0A 42 0A 08 00 10 00 18 00 20 00 28 00 4A 06 08 01 10 01 18 03 52 42 08 01 12 0A 08 00 10 80 80 04 18 10 20 02 12 0A 08 01 10 80 80 04 18 08 20 02 12 0A 08 02 10 80 80 01 18 08 20 01 12 0A 08 03 10 80 80 04 18 08 20 02 12 0A 08 04 10 80 80 04 18 08 20 02 18 00 20 00 5A 40 08 01 12 0A 08 00 10 80 80 04 18 10 20 02 12 0A 08 01 10 80 80 04 18 08 20 02 12 0A 08 02 10 80 80 01 18 08 20 01 12 0A 08 03 10 80 80 04 18 08 20 02 12 0A 08 04 10 80 80 04 18 08 20 02 18 00 70 02 78 02 80 01 FA 01 0B 69 00 01 0A 16 26 69 6D 67 63 61 63 68 65 2E 71 71 2E 63 6F 6D 2E 73 63 68 65 64 2E 70 31 76 36 2E 74 64 6E 73 76 36 2E 63 6F 6D 2E 20 50 0B 79 00 02 0A 16 0C 31 30 31 2E 39 31 2E 35 2E 32 31 30 20 50 0B 0A 16 0C 31 30 31 2E 38 39 2E 33 39 2E 32 31 20 50 0B 8A 06 0F 31 37 31 2E 31 31 32 2E 32 32 36 2E 32 31 34 10 03 0B 9A 09 00 0C 0A 00 0F 19 00 01 0A 12 71 19 A3 B4 20 50 0B 29 0C 0B 0A 00 04 19 00 01 0A 12 0B 27 59 65 20 50 0B 29 0C 0B 0A 00 0D 19 00 02 0A 12 0E EB 3F 3B 20 50 0B 0A 12 8D 9D 31 3A 20 50 0B 29 0C 0B 0A 00 03 19 00 02 0A 12 0E 01 B1 6F 20 50 0B 0A 12 91 43 E4 DD 20 50 0B 29 0C 0B 0A 00 07 19 00 01 0A 12 75 A2 E3 65 20 50 0B 29 0C 0B 0A 00 09 19 00 02 0A 12 35 8C D7 0E 20 50 0B 0A 12 19 8C D7 0E 20 50 0B 29 0C 0B 0A 00 0A 19 00 02 0A 12 8C B4 12 0E 20 50 0B 0A 12 7B B4 12 0E 20 50 0B 29 0C 0B 0A 00 0B 19 00 02 0A 12 6D 01 B1 6F 20 50 0B 0A 12 4D 01 B1 6F 20 50 0B 29 0C 0B 0A 00 05 19 00 01 0A 12 1D E2 03 B7 20 50 0B 29 0C 0B 0A 00 08 19 00 02 0A 12 78 09 61 B4 20 50 0B 0A 12 78 08 61 B4 20 50 0B 29 0C 0B 0A 00 06 19 00 02 0A 12 64 09 61 B4 20 50 0B 0A 12 6B D4 E2 65 20 50 0B 29 0C 0B 0A 00 0E 19 00 02 0A 12 6B 89 31 3A 20 50 0B 0A 12 76 01 B1 6F 20 50 0B 29 0C 0B 0B AD 00 01 01 5B 08 01 10 96 FA C9 03 18 00 22 0A 31 39 39 34 37 30 31 30 32 31 28 AB E1 89 B7 0D 32 12 08 B7 87 F4 97 05 10 50 18 8A F0 D8 F6 08 20 50 28 64 32 12 08 B7 87 F4 DF 03 10 50 18 8A F0 D4 BA 01 20 50 28 64 32 13 08 B4 C7 DA A0 01 10 50 18 8A EC DC D6 01 20 50 28 C8 01 32 13 08 E5 B6 BD D0 03 10 50 18 89 D6 AD CC 02 20 50 28 C8 01 32 13 08 DF CF DE AA 07 10 50 18 8A EE D4 8A 0E 20 50 28 AC 02 32 13 08 B7 81 97 86 07 10 50 18 8A EC D4 86 02 20 50 28 AC 02 3A 1E 0A 10 24 0E 00 E1 A9 00 00 10 00 00 00 00 00 00 00 4A 10 50 18 89 88 E0 DC 04 20 50 28 64 3A 1E 0A 10 24 0E 00 E1 A9 00 00 10 00 00 00 00 00 00 00 1F 10 50 18 89 88 E0 CC 04 20 50 28 64 3A 1F 0A 10 24 02 4E 00 80 20 00 02 00 00 00 00 00 00 00 A9 10 50 18 89 E6 80 B8 02 20 50 28 C8 01 3A 1F 0A 10 24 02 4E 00 80 20 00 02 00 00 00 00 00 00 00 72 10 50 18 89 E6 80 D8 02 20 50 28 C8 01 3A 1F 0A 10 24 08 80 F1 00 31 00 50 00 00 00 00 00 00 00 5A 10 50 18 89 88 8C A4 06 20 50 28 AC 02 3A 1F 0A 10 24 02 4E 00 80 10 00 00 00 00 00 00 00 00 01 51 10 50 18 89 DC C0 E4 0F 20 50 28 AC 02 32 4A D6 38 1D 0B 8C 98 0C A8 0C
            // 10 02 2C 3C 4C 56 23 51 51 53 65 72 76 69 63 65 2E 43 6F 6E 66 69 67 50 75 73 68 53 76 63 2E 4D 61 69 6E 53 65 72 76 61 6E 74 66 07 50 75 73 68 52 65 71 7D 00 01 08 70 08 00 01 06 07 50 75 73 68 52 65 71 18 00 01 06 12 43 6F 6E 66 69 67 50 75 73 68 2E 50 75 73 68 52 65 71 1D 00 01 08 48 0A 10 02 2D 00 01 08 36 09 00 01 0A 16 0C 31 38 33 2E 35 37 2E 35 33 2E 31 36 21 1F 90 0B 19 00 01 0A 16 0B 31 38 30 2E 39 36 2E 31 2E 33 30 20 50 0B 29 00 02 0A 16 0D 36 31 2E 31 38 33 2E 31 36 34 2E 33 36 20 50 0B 0A 16 0C 35 39 2E 33 36 2E 31 31 33 2E 36 32 20 50 0B 39 00 06 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 32 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 33 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 35 34 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 33 33 20 50 0B 0A 16 0E 31 31 39 2E 31 34 37 2E 31 39 2E 32 33 34 20 50 0B 0A 16 11 73 63 61 6E 6E 6F 6E 2E 33 67 2E 71 71 2E 63 6F 6D 20 50 0B 49 00 04 0A 16 0D 31 34 2E 31 38 2E 31 38 30 2E 31 38 34 20 50 0B 0A 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 37 21 1F 90 0B 0A 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 37 30 21 01 BB 0B 0A 16 0C 31 30 31 2E 39 31 2E 35 2E 31 39 34 20 50 0B 5A 09 00 03 0A 00 01 19 00 04 0A 00 01 16 0D 31 34 2E 31 38 2E 31 38 30 2E 31 38 34 20 50 0B 0A 00 01 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 37 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 37 30 21 01 BB 0B 0A 00 01 16 0C 31 30 31 2E 39 31 2E 35 2E 31 39 34 20 50 0B 29 0C 3C 0B 0A 00 05 19 00 04 0A 00 01 16 0D 31 34 2E 31 38 2E 31 38 30 2E 31 38 34 20 50 0B 0A 00 01 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 37 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 37 30 21 01 BB 0B 0A 00 01 16 0C 31 30 31 2E 39 31 2E 35 2E 31 39 34 20 50 0B 29 0C 3C 0B 0A 00 0A 19 00 04 0A 00 01 16 0D 31 34 2E 31 38 2E 31 38 30 2E 31 38 34 20 50 0B 0A 00 01 16 0D 31 38 33 2E 33 2E 32 33 33 2E 32 32 37 21 1F 90 0B 0A 00 01 16 0E 31 32 33 2E 31 35 30 2E 37 36 2E 31 37 30 21 01 BB 0B 0A 00 01 16 0C 31 30 31 2E 39 31 2E 35 2E 31 39 34 20 50 0B 29 00 05 0A 0C 11 20 00 20 10 30 01 0B 0A 00 01 11 20 00 20 08 30 02 0B 0A 00 02 11 20 00 20 08 30 01 0B 0A 00 03 12 00 01 00 00 20 08 30 02 0B 0A 00 04 11 20 00 20 08 30 02 0B 3C 0B 1D 00 00 68 24 A8 EE 05 B6 9E 51 AB 2C 10 75 F5 32 F5 CB F1 35 79 02 F4 B2 B9 BB FB B9 8E 20 16 39 61 79 24 47 59 C2 82 EB 13 9B 62 D5 66 66 CB EC F3 54 4B 3D E5 14 37 F2 0B C5 01 7D 21 E7 BC EF 30 8E 5C D6 C0 36 5B E6 3F 68 A9 DE 22 09 39 F5 A8 D6 9A 9B 58 09 37 80 EF B7 D4 C5 84 97 30 F4 69 A2 31 BC 2F 76 1E 06 F0 C5 7E 2D 00 00 10 36 39 42 41 71 51 46 65 43 4E 64 6A 62 65 59 39 32 76 E4 B8 DD 40 01 5D 00 01 02 54 8A 50 D0 04 0A 68 24 A8 EE 05 B6 9E 51 AB 2C 10 75 F5 32 F5 CB F1 35 79 02 F4 B2 B9 BB FB B9 8E 20 16 39 61 79 24 47 59 C2 82 EB 13 9B 62 D5 66 66 CB EC F3 54 4B 3D E5 14 37 F2 0B C5 01 7D 21 E7 BC EF 30 8E 5C D6 C0 36 5B E6 3F 68 A9 DE 22 09 39 F5 A8 D6 9A 9B 58 09 37 80 EF B7 D4 C5 84 97 30 F4 69 A2 31 BC 2F 76 1E 06 F0 C5 7E 12 10 36 39 42 41 71 51 46 65 43 4E 64 6A 62 65 59 39 1A 40 08 01 12 0D 08 01 15 0E 12 B4 B8 18 50 20 01 28 01 12 0E 08 01 15 B7 03 E9 E3 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 96 4C AA 18 BB 03 20 02 28 00 12 0D 08 01 15 65 5B 05 C2 18 50 20 04 28 00 1A 40 08 05 12 0D 08 01 15 0E 12 B4 B8 18 50 20 01 28 01 12 0E 08 01 15 B7 03 E9 E3 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 96 4C AA 18 BB 03 20 02 28 00 12 0D 08 01 15 65 5B 05 C2 18 50 20 04 28 00 1A 78 08 0A 12 0D 08 01 15 0E 12 B4 B8 18 50 20 01 28 01 12 0E 08 01 15 B7 03 E9 E3 18 90 3F 20 01 28 01 12 0E 08 01 15 7B 96 4C AA 18 BB 03 20 02 28 00 12 0D 08 01 15 65 5B 05 C2 18 50 20 04 28 00 22 09 08 00 10 80 40 18 10 20 01 22 09 08 01 10 80 40 18 08 20 02 22 09 08 02 10 80 40 18 08 20 01 22 0A 08 03 10 80 80 04 18 08 20 02 22 09 08 04 10 80 40 18 08 20 02 20 01 32 04 08 00 10 01 3A 2A 08 10 10 10 18 09 20 09 28 0F 30 0F 38 05 40 05 48 5A 50 01 58 5A 60 5A 68 5A 70 5A 78 0A 80 01 0A 88 01 0A 90 01 0A 98 01 0A 42 0A 08 00 10 00 18 00 20 00 28 00 4A 06 08 01 10 01 18 03 52 42 08 01 12 0A 08 00 10 80 80 04 18 10 20 02 12 0A 08 01 10 80 80 04 18 08 20 02 12 0A 08 02 10 80 80 01 18 08 20 01 12 0A 08 03 10 80 80 04 18 08 20 02 12 0A 08 04 10 80 80 04 18 08 20 02 18 00 20 00 5A 40 08 01 12 0A 08 00 10 80 80 04 18 10 20 02 12 0A 08 01 10 80 80 04 18 08 20 02 12 0A 08 02 10 80 80 01 18 08 20 01 12 0A 08 03 10 80 80 04 18 08 20 02 12 0A 08 04 10 80 80 04 18 08 20 02 18 00 70 02 78 02 80 01 FA 01 0B 69 00 01 0A 16 26 69 6D 67 63 61 63 68 65 2E 71 71 2E 63 6F 6D 2E 73 63 68 65 64 2E 70 31 76 36 2E 74 64 6E 73 76 36 2E 63 6F 6D 2E 20 50 0B 79 00 02 0A 16 0C 31 30 31 2E 39 31 2E 35 2E 32 31 30 20 50 0B 0A 16 0E 31 30 31 2E 32 32 37 2E 31 33 31 2E 36 37 20 50 0B 8A 06 0F 31 37 31 2E 31 31 32 2E 32 32 36 2E 32 33 37 10 03 0B 9A 09 00 0B 0A 00 0F 19 00 01 0A 12 71 19 A3 B4 20 50 0B 29 0C 0B 0A 00 04 19 00 01 0A 12 0B 27 59 65 20 50 0B 29 0C 0B 0A 00 0D 19 00 02 0A 12 C3 B9 D3 74 20 50 0B 0A 12 CC 43 E4 DD 20 50 0B 29 0C 0B 0A 00 03 19 00 02 0A 12 0D 31 BA DE 20 50 0B 0A 12 C3 B9 D3 74 20 50 0B 29 0C 0B 0A 00 07 19 00 01 0A 12 75 A2 E3 65 20 50 0B 29 0C 0B 0A 00 09 19 00 02 0A 12 17 D0 60 71 20 50 0B 0A 12 30 E8 60 71 20 50 0B 29 0C 0B 0A 00 0A 19 00 02 0A 12 E8 E9 03 B7 20 50 0B 0A 12 2E AF 12 0E 20 50 0B 29 0C 0B 0A 00 05 19 00 01 0A 12 1D E2 03 B7 20 50 0B 29 0C 0B 0A 00 08 19 00 02 0A 12 13 75 61 B4 20 50 0B 0A 12 15 75 61 B4 20 50 0B 29 0C 0B 0A 00 06 19 00 02 0A 12 55 10 59 65 20 50 0B 0A 12 47 10 59 65 20 50 0B 29 0C 0B 0A 00 0E 19 00 02 0A 12 6B 89 31 3A 20 50 0B 0A 12 76 01 B1 6F 20 50 0B 29 0C 0B 0B AD 00 01 01 5B 08 01 10 BD 8F A6 16 18 00 22 0A 31 39 39 34 37 30 31 30 32 31 28 AB E1 89 EF 0E 32 12 08 B7 87 FC D7 05 10 50 18 8A F0 C8 98 06 20 50 28 64 32 12 08 B7 87 F4 87 02 10 50 18 8A F0 CC E8 02 20 50 28 64 32 13 08 E5 B6 BD E0 05 10 50 18 89 D6 AD BC 06 20 50 28 C8 01 32 13 08 E5 B2 F1 E8 0E 10 50 18 89 B0 C4 D1 08 20 50 28 C8 01 32 13 08 DF CD DE BC 0A 10 50 18 E4 E0 8D ED 0E 20 50 28 AC 02 32 13 08 B7 81 CF F5 0E 10 50 18 89 EC F4 F0 0C 20 50 28 AC 02 3A 1E 0A 10 24 0E 00 E1 AA 00 00 13 00 00 00 00 00 00 00 18 10 50 18 89 EC A0 88 0D 20 50 28 64 3A 1E 0A 10 24 0E 00 E1 A9 00 00 50 00 00 00 00 00 00 00 53 10 50 18 89 EC 8C B9 03 20 50 28 64 3A 1F 0A 10 24 0E 00 FF F1 01 00 10 00 00 00 00 00 00 01 72 10 50 18 E4 E6 AD F0 0E 20 50 28 C8 01 3A 1F 0A 10 24 0E 00 FF F1 00 80 19 00 00 00 00 00 00 00 11 10 50 18 8A F0 D0 A2 0C 20 50 28 C8 01 3A 1F 0A 10 24 09 8C 1E 8F D0 00 50 00 00 00 00 00 00 00 5A 10 50 18 89 EC A0 C8 0A 20 50 28 AC 02 3A 1F 0A 10 24 09 8C 54 10 03 10 19 00 00 00 00 00 00 00 13 10 50 18 E4 E6 A9 B4 07 20 50 28 AC 02 33 00 00 00 01 D3 68 60 BF 0B 8C 98 0C A8 0C

            // 02 00 00 01 2C 00 01 00 02 40 63 83 C1
            //
            // 08 08
            //
            // 12
            // 15
            // 0A 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 38 39
            // 10 90 3F
            // 18 00
            //
            // 12
            // 13
            // 0A 0D 31 31 33 2E 39 36 2E 31 33 2E 31 32 35
            // 10 50
            // 18 00
            //
            // 12
            // 13
            // 0A 0C 34 32 2E 38 31 2E 31 37 32 2E 36 33
            // 10 B0 6D
            // 18 00
            //
            // 12
            // 15
            // 0A 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 33 34
            // 10 BB 03
            // 18 00
            //
            // 12
            // 13
            // 0A 0D 31 32 35 2E 39 34 2E 36 30 2E 31 34 36
            // 10 50
            // 18 00
            //
            // 12
            // 13
            // 0A 0D 34 32 2E 38 31 2E 31 37 32 2E 31 34 37
            // 10 50
            // 18 00
            //
            // 12
            // 18
            // 0A 11 6D 73 66 77 69 66 69 2E 33 67 2E 71 71 2E 63 6F 6D
            // 10 90 3F
            // 18 00
            //
            // 12
            // 13
            // 0A 0D 34 32 2E 38 31 2E 31 37 32 2E 31 34 37
            // 10 50
            // 18 00
            //
            // 1A
            // 14
            // 0A 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 32
            // 10 50
            // 18 00
            //
            // 1A
            // 13
            // 0A 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35
            // 10 50
            // 18 00
            //
            // 1A
            // 13
            // 0A 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35
            // 10 50
            // 18 00
            //
            // 1A
            // 14
            // 0A 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 32
            // 10 50
            // 18 00
            //
            // 1A
            // 13
            // 0A 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35
            // 10 50
            // 18 00
            //
            // 03


            // 02 00 00 01 2B 00 01 00 02 CB 1E 16 27 08 08 12 13 0A 0C 34 32 2E 38 31 2E 31 36 39 2E 34 36 10 90 3F 18 00 12 12 0A 0C 34 32 2E 38 31 2E 31 37 32 2E 38 31 10 50 18 00 12 15 0A 0E 31 31 34 2E 32 32 31 2E 31 34 38 2E 35 39 10 B0 6D 18 00 12 14 0A 0D 34 32 2E 38 31 2E 31 37 32 2E 31 34 37 10 BB 03 18 00 12 13 0A 0D 31 32 35 2E 39 34 2E 36 30 2E 31 34 36 10 50 18 00 12 15 0A 0F 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 31 35 10 50 18 00 12 18 0A 11 6D 73 66 77 69 66 69 2E 33 67 2E 71 71 2E 63 6F 6D 10 90 3F 18 00 12 12 0A 0C 34 32 2E 38 31 2E 31 37 32 2E 32 32 10 50 18 00 1A 13 0A 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 10 50 18 00 1A 14 0A 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 32 10 50 18 00 1A 14 0A 0E 31 31 34 2E 32 32 31 2E 31 34 34 2E 32 32 10 50 18 00 1A 13 0A 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 10 50 18 00 1A 13 0A 0D 34 32 2E 38 31 2E 31 36 39 2E 31 30 35 10 50 18 00 03


            /*
            + '
42.81.169.46 ?
42.81.172.81P
114.221.148.59 m
42.81.172.147
125.94.60.146P
114.221.144.215P
msfwifi.3g.qq.com ?
42.81.172.22P
42.81.169.105P
114.221.144.22P
114.221.144.22P
42.81.169.105P
42.81.169.105P
             */
        }

        override suspend fun QQAndroidBot.handle(packet: PushReqResponse?, sequenceId: Int): OutgoingPacket? {
            if (packet == null) {
                return null
            }
            when (packet) {
                is PushReqResponse.Success -> {
                    return buildResponseUniPacket(
                        client,
                        sequenceId = sequenceId
                    ) {
                        writeJceStruct(
                            RequestPacket.serializer(),
                            RequestPacket(
                                iRequestId = 0,
                                iVersion = 3,
                                sServantName = "QQService.ConfigPushSvc.MainServant",
                                sFuncName = "PushResp",
                                sBuffer = jceRequestSBuffer(
                                    "PushResp",
                                    PushResp.serializer(),
                                    PushResp(
                                        type = packet.struct.type,
                                        seq = packet.struct.seq,
                                        jcebuf = if (packet.struct.type == 3) packet.struct.jcebuf else null
                                    )
                                )
                            )
                        )
                        // writePacket(this.build().debugPrintThis())
                    }
                }
                else -> {
                    // handled in QQABot
                    return null
                }
            }
        }
    }
}