// Audio playback and capture library. Public domain. See "unlicense" statement at the end of this file.
// mini_al - v0.8.10-rc - 2018-xx-xx
//
// David Reid - davidreidsoftware@gmail.com

// ABOUT
// =====
// mini_al is a small library for making it easy to connect to a playback or capture device and send
// or receive data from that device.
//
// mini_al uses an asynchronous API. Every device is created with it's own thread, with audio data
// being delivered to or from the device via a callback. Synchronous APIs are not supported in the
// interest of keeping the library as simple and light-weight as possible.
//
// Supported Backends:
//   - WASAPI
//   - DirectSound
//   - WinMM
//   - Core Audio (Apple)
//   - ALSA
//   - PulseAudio
//   - JACK
//   - sndio (OpenBSD)
//   - audio(4) (NetBSD and OpenBSD)
//   - OSS (FreeBSD)
//   - OpenSL|ES (Android only)
//   - OpenAL
//   - SDL
//   - Null (Silence)
//
// Supported Formats:
//   - Unsigned 8-bit PCM
//   - Signed 16-bit PCM
//   - Signed 24-bit PCM (tightly packed)
//   - Signed 32-bit PCM
//   - IEEE 32-bit floating point PCM
//
//
// USAGE
// =====
// mini_al is a single-file library. To use it, do something like the following in one .c file.
//   #define MINI_AL_IMPLEMENTATION
//   #include "mini_al.h"
//
// You can then #include this file in other parts of the program as you would with any other header file.
//
// If you want to disable a specific backend, #define the appropriate MAL_NO_* option before the implementation.
//
// Note that GCC and Clang requires "-msse2", "-mavx2", etc. for SIMD optimizations.
//
//
// Building for Windows
// --------------------
// The Windows build should compile clean on all popular compilers without the need to configure any include paths
// nor link to any libraries.
//
// Building for macOS and iOS
// --------------------------
// The macOS build should compile clean without the need to download any dependencies or link to any libraries or
// frameworks. The iOS build needs to be compiled as Objective-C (sorry) and will need to link the relevant frameworks
// but should Just Work with Xcode.
//
// Building for Linux
// ------------------
// The Linux build only requires linking to -ldl, -lpthread and -lm. You do not need any development packages for any
// of the supported backends.
//
// Building for BSD
// ----------------
// The BSD build only requires linking to -ldl, -lpthread and -lm. NetBSD uses audio(4), OpenBSD uses sndio and
// FreeBSD uses OSS.
//
// Building for Android
// --------------------
// The Android build uses OpenSL|ES, and will require an appropriate API level that supports OpenSL|ES. mini_al has
// been tested against API levels 16 and 21.
//
// Building for Emscripten
// -----------------------
// The Emscripten build currently uses SDL 1.2 for it's backend which means specifying "-s USE_SDL=2" is unecessary
// as of this version.
//
//
// Playback Example
// ----------------
//   mal_uint32 on_send_samples(mal_device* pDevice, mal_uint32 frameCount, void* pSamples)
//   {
//       // This callback is set at initialization time and will be called when a playback device needs more
//       // data. You need to write as many frames as you can to pSamples (but no more than frameCount) and
//       // then return the number of frames you wrote.
//       //
//       // The user data (pDevice->pUserData) is set by mal_device_init().
//       return (mal_uint32)mal_decoder_read((mal_decoder*)pDevice->pUserData, frameCount, pSamples);
//   }
//
//   ...
//
//   mal_device_config config = mal_device_config_init_playback(decoder.outputFormat, decoder.outputChannels, decoder.outputSampleRate, on_send_frames_to_device);
//
//   mal_device device;
//   mal_result result = mal_device_init(NULL, mal_device_type_playback, NULL, &config, &decoder /*pUserData*/, &device);
//   if (result != MAL_SUCCESS) {
//       return -1;
//   }
//
//   mal_device_start(&device);     // The device is sleeping by default so you'll need to start it manually.
//
//   ...
//
//   mal_device_uninit(&device);    // This will stop the device so no need to do that manually.
//
//
//
// NOTES
// =====
// - This library uses an asynchronous API for delivering and requesting audio data. Each device will have
//   it's own worker thread which is managed by the library.
// - If mal_device_init() is called with a device that's not aligned to the platform's natural alignment
//   boundary (4 bytes on 32-bit, 8 bytes on 64-bit), it will _not_ be thread-safe. The reason for this
//   is that it depends on members of mal_device being correctly aligned for atomic assignments.
// - Sample data is always native-endian and interleaved. For example, mal_format_s16 means signed 16-bit
//   integer samples, interleaved. Let me know if you need non-interleaved and I'll look into it.
// - The sndio backend is currently only enabled on OpenBSD builds.
// - The audio(4) backend is supported on OpenBSD, but you may need to disable sndiod before you can use it.
// - If you are using the platform's default device, mini_al will try automatically switching the internal
//   device when the device is unplugged. This feature is disabled when the device is opened in exclusive
//   mode.
//
//
//
// BACKEND NUANCES
// ===============
//
// Android
// -------
// - To capture audio on Android, remember to add the RECORD_AUDIO permission to your manifest:
//     <uses-permission android:name="android.permission.RECORD_AUDIO" />
// - Only a single mal_context can be active at any given time. This is due to a limitation with OpenSL|ES.
//
// UWP
// ---
// - UWP only supports default playback and capture devices.
// - UWP requires the Microphone capability to be enabled in the application's manifest (Package.appxmanifest):
//       <Package ...>
//           ...
//           <Capabilities>
//               <DeviceCapability Name="microphone" />
//           </Capabilities>
//       </Package>
//
//
// OPTIONS
// =======
// #define these options before including this file.
//
// #define MAL_NO_WASAPI
//   Disables the WASAPI backend.
//
// #define MAL_NO_DSOUND
//   Disables the DirectSound backend.
//
// #define MAL_NO_WINMM
//   Disables the WinMM backend.
//
// #define MAL_NO_ALSA
//   Disables the ALSA backend.
//
// #define MAL_NO_PULSEAUDIO
//   Disables the PulseAudio backend.
//
// #define MAL_NO_JACK
//   Disables the JACK backend.
//
// #define MAL_NO_COREAUDIO
//   Disables the Core Audio backend.
//
// #define MAL_NO_SNDIO
//   Disables the sndio backend.
//
// #define MAL_NO_AUDIO4
//   Disables the audio(4) backend.
//
// #define MAL_NO_OSS
//   Disables the OSS backend.
//
// #define MAL_NO_OPENSL
//   Disables the OpenSL|ES backend.
//
// #define MAL_NO_OPENAL
//   Disables the OpenAL backend.
//
// #define MAL_NO_SDL
//   Disables the SDL backend.
//
// #define MAL_NO_NULL
//   Disables the null backend.
//
// #define MAL_DEFAULT_PERIODS
//   When a period count of 0 is specified when a device is initialized, it will default to this.
//
// #define MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY
// #define MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE
//   When a buffer size of 0 is specified when a device is initialized it will default to a buffer of this size, depending
//   on the chosen performance profile. These can be increased or decreased depending on your specific requirements.
//
// #define MAL_NO_DECODING
//   Disables the decoding APIs.
//
// #define MAL_NO_DEVICE_IO
//   Disables playback and recording. This will disable mal_context and mal_device APIs. This is useful if you only want to
//   use mini_al's data conversion and/or decoding APIs. 
//
// #define MAL_NO_STDIO
//   Disables file IO APIs.
//
// #define MAL_NO_SSE2
//   Disables SSE2 optimizations.
//
// #define MAL_NO_AVX2
//   Disables AVX2 optimizations.
//
// #define MAL_NO_AVX512
//   Disables AVX-512 optimizations.
//
// #define MAL_NO_NEON
//   Disables NEON optimizations.
//
// #define MAL_LOG_LEVEL <Level>
//   Sets the logging level. Set level to one of the following:
//     MAL_LOG_LEVEL_VERBOSE
//     MAL_LOG_LEVEL_INFO
//     MAL_LOG_LEVEL_WARNING
//     MAL_LOG_LEVEL_ERROR
//
// #define MAL_DEBUT_OUTPUT
//   Enable printf() debug output.
//
// #ifndef MAL_COINIT_VALUE
//   Windows only. The value to pass to internal calls to CoInitializeEx(). Defaults to COINIT_MULTITHREADED.

#ifndef mini_al_h
#define mini_al_h

#ifdef __cplusplus
extern "C" {
#endif

#if defined(_MSC_VER)
    #pragma warning(push)
    #pragma warning(disable:4201)   // nonstandard extension used: nameless struct/union
    #pragma warning(disable:4324)   // structure was padded due to alignment specifier
#endif

// Platform/backend detection.
#ifdef _WIN32
    #define MAL_WIN32
    #if defined(WINAPI_FAMILY) && (WINAPI_FAMILY == WINAPI_FAMILY_PC_APP || WINAPI_FAMILY == WINAPI_FAMILY_PHONE_APP)
        #define MAL_WIN32_UWP
    #else
        #define MAL_WIN32_DESKTOP
    #endif
#else
    #define MAL_POSIX
    #include <pthread.h>    // Unfortunate #include, but needed for pthread_t, pthread_mutex_t and pthread_cond_t types.

    #ifdef __unix__
        #define MAL_UNIX
        #if defined(__DragonFly__) || defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
            #define MAL_BSD
        #endif
    #endif
    #ifdef __linux__
        #define MAL_LINUX
    #endif
    #ifdef __APPLE__
        #define MAL_APPLE
    #endif
    #ifdef __ANDROID__
        #define MAL_ANDROID
    #endif
    #ifdef __EMSCRIPTEN__
        #define MAL_EMSCRIPTEN
    #endif
#endif

#include <stddef.h> // For size_t.

#ifndef MAL_HAS_STDINT
    #if defined(_MSC_VER)
        #if _MSC_VER >= 1600
            #define MAL_HAS_STDINT
        #endif
    #else
        #if defined(__has_include)
            #if __has_include(<stdint.h>)
                #define MAL_HAS_STDINT
            #endif
        #endif
    #endif
#endif

#if !defined(MAL_HAS_STDINT) && (defined(__GNUC__) || defined(__clang__))   // Assume support for stdint.h on GCC and Clang.
    #define MAL_HAS_STDINT
#endif

#ifndef MAL_HAS_STDINT
typedef   signed char               mal_int8;
typedef unsigned char               mal_uint8;
typedef   signed short              mal_int16;
typedef unsigned short              mal_uint16;
typedef   signed int                mal_int32;
typedef unsigned int                mal_uint32;
    #if defined(_MSC_VER)
    typedef   signed __int64        mal_int64;
    typedef unsigned __int64        mal_uint64;
    #else
    typedef   signed long long int  mal_int64;
    typedef unsigned long long int  mal_uint64;
    #endif
    #if defined(_WIN32)
        #if defined(_WIN64)
        typedef mal_uint64          mal_uintptr;
        #else
        typedef mal_uint32          mal_uintptr;
        #endif
    #elif defined(__GNUC__)
        #if defined(__LP64__)
        typedef mal_uint64          mal_uintptr;
        #else
        typedef mal_uint32          mal_uintptr;
        #endif
    #else
        typedef mal_uint64          mal_uintptr;    // Fallback.
    #endif
#else
#include <stdint.h>
typedef int8_t                      mal_int8;
typedef uint8_t                     mal_uint8;
typedef int16_t                     mal_int16;
typedef uint16_t                    mal_uint16;
typedef int32_t                     mal_int32;
typedef uint32_t                    mal_uint32;
typedef int64_t                     mal_int64;
typedef uint64_t                    mal_uint64;
typedef uintptr_t                   mal_uintptr;
#endif
typedef mal_uint8                   mal_bool8;
typedef mal_uint32                  mal_bool32;
#define MAL_TRUE                    1
#define MAL_FALSE                   0

typedef void* mal_handle;
typedef void* mal_ptr;
typedef void (* mal_proc)(void);

#if defined(_MSC_VER) && !defined(_WCHAR_T_DEFINED)
typedef mal_uint16 wchar_t;
#endif

// Define NULL for some compilers.
#ifndef NULL
#define NULL 0
#endif

#if defined(SIZE_MAX)
    #define MAL_SIZE_MAX    SIZE_MAX
#else
    #define MAL_SIZE_MAX    0xFFFFFFFF  /* When SIZE_MAX is not defined by the standard library just default to the maximum 32-bit unsigned integer. */
#endif


#ifdef _MSC_VER
#define MAL_INLINE __forceinline
#else
#ifdef __GNUC__
#define MAL_INLINE inline __attribute__((always_inline))
#else
#define MAL_INLINE inline
#endif
#endif

#ifdef _MSC_VER
#define MAL_ALIGN(alignment) __declspec(align(alignment))
#elif !defined(__DMC__)
#define MAL_ALIGN(alignment) __attribute__((aligned(alignment)))
#else
#define MAL_ALIGN(alignment)
#endif

#ifdef _MSC_VER
#define MAL_ALIGNED_STRUCT(alignment) MAL_ALIGN(alignment) struct
#else
#define MAL_ALIGNED_STRUCT(alignment) struct MAL_ALIGN(alignment)
#endif

// SIMD alignment in bytes. Currently set to 64 bytes in preparation for future AVX-512 optimizations.
#define MAL_SIMD_ALIGNMENT  64


// Logging levels
#define MAL_LOG_LEVEL_VERBOSE   4
#define MAL_LOG_LEVEL_INFO      3
#define MAL_LOG_LEVEL_WARNING   2
#define MAL_LOG_LEVEL_ERROR     1

#ifndef MAL_LOG_LEVEL
#define MAL_LOG_LEVEL           MAL_LOG_LEVEL_ERROR
#endif

typedef struct mal_context mal_context;
typedef struct mal_device mal_device;

typedef mal_uint8 mal_channel;
#define MAL_CHANNEL_NONE                                0
#define MAL_CHANNEL_MONO                                1
#define MAL_CHANNEL_FRONT_LEFT                          2
#define MAL_CHANNEL_FRONT_RIGHT                         3
#define MAL_CHANNEL_FRONT_CENTER                        4
#define MAL_CHANNEL_LFE                                 5
#define MAL_CHANNEL_BACK_LEFT                           6
#define MAL_CHANNEL_BACK_RIGHT                          7
#define MAL_CHANNEL_FRONT_LEFT_CENTER                   8
#define MAL_CHANNEL_FRONT_RIGHT_CENTER                  9
#define MAL_CHANNEL_BACK_CENTER                         10
#define MAL_CHANNEL_SIDE_LEFT                           11
#define MAL_CHANNEL_SIDE_RIGHT                          12
#define MAL_CHANNEL_TOP_CENTER                          13
#define MAL_CHANNEL_TOP_FRONT_LEFT                      14
#define MAL_CHANNEL_TOP_FRONT_CENTER                    15
#define MAL_CHANNEL_TOP_FRONT_RIGHT                     16
#define MAL_CHANNEL_TOP_BACK_LEFT                       17
#define MAL_CHANNEL_TOP_BACK_CENTER                     18
#define MAL_CHANNEL_TOP_BACK_RIGHT                      19
#define MAL_CHANNEL_AUX_0                               20
#define MAL_CHANNEL_AUX_1                               21
#define MAL_CHANNEL_AUX_2                               22
#define MAL_CHANNEL_AUX_3                               23
#define MAL_CHANNEL_AUX_4                               24
#define MAL_CHANNEL_AUX_5                               25
#define MAL_CHANNEL_AUX_6                               26
#define MAL_CHANNEL_AUX_7                               27
#define MAL_CHANNEL_AUX_8                               28
#define MAL_CHANNEL_AUX_9                               29
#define MAL_CHANNEL_AUX_10                              30
#define MAL_CHANNEL_AUX_11                              31
#define MAL_CHANNEL_AUX_12                              32
#define MAL_CHANNEL_AUX_13                              33
#define MAL_CHANNEL_AUX_14                              34
#define MAL_CHANNEL_AUX_15                              35
#define MAL_CHANNEL_AUX_16                              36
#define MAL_CHANNEL_AUX_17                              37
#define MAL_CHANNEL_AUX_18                              38
#define MAL_CHANNEL_AUX_19                              39
#define MAL_CHANNEL_AUX_20                              40
#define MAL_CHANNEL_AUX_21                              41
#define MAL_CHANNEL_AUX_22                              42
#define MAL_CHANNEL_AUX_23                              43
#define MAL_CHANNEL_AUX_24                              44
#define MAL_CHANNEL_AUX_25                              45
#define MAL_CHANNEL_AUX_26                              46
#define MAL_CHANNEL_AUX_27                              47
#define MAL_CHANNEL_AUX_28                              48
#define MAL_CHANNEL_AUX_29                              49
#define MAL_CHANNEL_AUX_30                              50
#define MAL_CHANNEL_AUX_31                              51
#define MAL_CHANNEL_LEFT                                MAL_CHANNEL_FRONT_LEFT
#define MAL_CHANNEL_RIGHT                               MAL_CHANNEL_FRONT_RIGHT
#define MAL_CHANNEL_POSITION_COUNT                      MAL_CHANNEL_AUX_31 + 1

typedef int mal_result;
#define MAL_SUCCESS                                      0
#define MAL_ERROR                                       -1      // A generic error.
#define MAL_INVALID_ARGS                                -2
#define MAL_INVALID_OPERATION                           -3
#define MAL_OUT_OF_MEMORY                               -4
#define MAL_FORMAT_NOT_SUPPORTED                        -5
#define MAL_NO_BACKEND                                  -6
#define MAL_NO_DEVICE                                   -7
#define MAL_API_NOT_FOUND                               -8
#define MAL_DEVICE_BUSY                                 -9
#define MAL_DEVICE_NOT_INITIALIZED                      -10
#define MAL_DEVICE_NOT_STARTED                          -11
#define MAL_DEVICE_NOT_STOPPED                          -12
#define MAL_DEVICE_ALREADY_STARTED                      -13
#define MAL_DEVICE_ALREADY_STARTING                     -14
#define MAL_DEVICE_ALREADY_STOPPED                      -15
#define MAL_DEVICE_ALREADY_STOPPING                     -16
#define MAL_FAILED_TO_MAP_DEVICE_BUFFER                 -17
#define MAL_FAILED_TO_UNMAP_DEVICE_BUFFER               -18
#define MAL_FAILED_TO_INIT_BACKEND                      -19
#define MAL_FAILED_TO_READ_DATA_FROM_CLIENT             -20
#define MAL_FAILED_TO_READ_DATA_FROM_DEVICE             -21
#define MAL_FAILED_TO_SEND_DATA_TO_CLIENT               -22
#define MAL_FAILED_TO_SEND_DATA_TO_DEVICE               -23
#define MAL_FAILED_TO_OPEN_BACKEND_DEVICE               -24
#define MAL_FAILED_TO_START_BACKEND_DEVICE              -25
#define MAL_FAILED_TO_STOP_BACKEND_DEVICE               -26
#define MAL_FAILED_TO_CONFIGURE_BACKEND_DEVICE          -27
#define MAL_FAILED_TO_CREATE_MUTEX                      -28
#define MAL_FAILED_TO_CREATE_EVENT                      -29
#define MAL_FAILED_TO_CREATE_THREAD                     -30
#define MAL_INVALID_DEVICE_CONFIG                       -31
#define MAL_ACCESS_DENIED                               -32
#define MAL_TOO_LARGE                                   -33
#define MAL_DEVICE_UNAVAILABLE                          -34
#define MAL_TIMEOUT                                     -35

// Standard sample rates.
#define MAL_SAMPLE_RATE_8000                            8000
#define MAL_SAMPLE_RATE_11025                           11025
#define MAL_SAMPLE_RATE_16000                           16000
#define MAL_SAMPLE_RATE_22050                           22050
#define MAL_SAMPLE_RATE_24000                           24000
#define MAL_SAMPLE_RATE_32000                           32000
#define MAL_SAMPLE_RATE_44100                           44100
#define MAL_SAMPLE_RATE_48000                           48000
#define MAL_SAMPLE_RATE_88200                           88200
#define MAL_SAMPLE_RATE_96000                           96000
#define MAL_SAMPLE_RATE_176400                          176400
#define MAL_SAMPLE_RATE_192000                          192000
#define MAL_SAMPLE_RATE_352800                          352800
#define MAL_SAMPLE_RATE_384000                          384000

#define MAL_MIN_PCM_SAMPLE_SIZE_IN_BYTES                1   // For simplicity, mini_al does not support PCM samples that are not byte aligned.
#define MAL_MAX_PCM_SAMPLE_SIZE_IN_BYTES                8
#define MAL_MIN_CHANNELS                                1
#define MAL_MAX_CHANNELS                                32
#define MAL_MIN_SAMPLE_RATE                             MAL_SAMPLE_RATE_8000
#define MAL_MAX_SAMPLE_RATE                             MAL_SAMPLE_RATE_384000
#define MAL_SRC_SINC_MIN_WINDOW_WIDTH                   2
#define MAL_SRC_SINC_MAX_WINDOW_WIDTH                   32
#define MAL_SRC_SINC_DEFAULT_WINDOW_WIDTH               32
#define MAL_SRC_SINC_LOOKUP_TABLE_RESOLUTION            8
#define MAL_SRC_INPUT_BUFFER_SIZE_IN_SAMPLES            256

typedef enum
{
    mal_stream_format_pcm = 0,
} mal_stream_format;

typedef enum
{
    mal_stream_layout_interleaved = 0,
    mal_stream_layout_deinterleaved
} mal_stream_layout;

typedef enum
{
    mal_dither_mode_none = 0,
    mal_dither_mode_rectangle,
    mal_dither_mode_triangle
} mal_dither_mode;

typedef enum
{
    // I like to keep these explicitly defined because they're used as a key into a lookup table. When items are
    // added to this, make sure there are no gaps and that they're added to the lookup table in mal_get_bytes_per_sample().
    mal_format_unknown = 0,     // Mainly used for indicating an error, but also used as the default for the output format for decoders.
    mal_format_u8      = 1,
    mal_format_s16     = 2,     // Seems to be the most widely supported format.
    mal_format_s24     = 3,     // Tightly packed. 3 bytes per sample.
    mal_format_s32     = 4,
    mal_format_f32     = 5,
    mal_format_count
} mal_format;

typedef enum
{
    mal_channel_mix_mode_planar_blend = 0,  // Simple averaging based on the plane(s) the channel is sitting on.
    mal_channel_mix_mode_simple,            // Drop excess channels; zeroed out extra channels.
    mal_channel_mix_mode_default = mal_channel_mix_mode_planar_blend
} mal_channel_mix_mode;

typedef enum
{
    mal_standard_channel_map_microsoft,
    mal_standard_channel_map_alsa,
    mal_standard_channel_map_rfc3551,   // Based off AIFF.
    mal_standard_channel_map_flac,
    mal_standard_channel_map_vorbis,
    mal_standard_channel_map_sound4,    // FreeBSD's sound(4).
    mal_standard_channel_map_sndio,     // www.sndio.org/tips.html
    mal_standard_channel_map_default = mal_standard_channel_map_microsoft
} mal_standard_channel_map;

typedef enum
{
    mal_performance_profile_low_latency = 0,
    mal_performance_profile_conservative
} mal_performance_profile;


typedef struct mal_format_converter mal_format_converter;
typedef mal_uint32 (* mal_format_converter_read_proc)              (mal_format_converter* pConverter, mal_uint32 frameCount, void* pFramesOut, void* pUserData);
typedef mal_uint32 (* mal_format_converter_read_deinterleaved_proc)(mal_format_converter* pConverter, mal_uint32 frameCount, void** ppSamplesOut, void* pUserData);

typedef struct
{
    mal_format formatIn;
    mal_format formatOut;
    mal_uint32 channels;
    mal_stream_format streamFormatIn;
    mal_stream_format streamFormatOut;
    mal_dither_mode ditherMode;
    mal_bool32 noSSE2   : 1;
    mal_bool32 noAVX2   : 1;
    mal_bool32 noAVX512 : 1;
    mal_bool32 noNEON   : 1;
    mal_format_converter_read_proc onRead;
    mal_format_converter_read_deinterleaved_proc onReadDeinterleaved;
    void* pUserData;
} mal_format_converter_config;

struct mal_format_converter
{
    mal_format_converter_config config;
    mal_bool32 useSSE2   : 1;
    mal_bool32 useAVX2   : 1;
    mal_bool32 useAVX512 : 1;
    mal_bool32 useNEON   : 1;
    void (* onConvertPCM)(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode);
    void (* onInterleavePCM)(void* dst, const void** src, mal_uint64 frameCount, mal_uint32 channels);
    void (* onDeinterleavePCM)(void** dst, const void* src, mal_uint64 frameCount, mal_uint32 channels);
};



typedef struct mal_channel_router mal_channel_router;
typedef mal_uint32 (* mal_channel_router_read_deinterleaved_proc)(mal_channel_router* pRouter, mal_uint32 frameCount, void** ppSamplesOut, void* pUserData);

typedef struct
{
    mal_uint32 channelsIn;
    mal_uint32 channelsOut;
    mal_channel channelMapIn[MAL_MAX_CHANNELS];
    mal_channel channelMapOut[MAL_MAX_CHANNELS];
    mal_channel_mix_mode mixingMode;
    mal_bool32 noSSE2   : 1;
    mal_bool32 noAVX2   : 1;
    mal_bool32 noAVX512 : 1;
    mal_bool32 noNEON   : 1;
    mal_channel_router_read_deinterleaved_proc onReadDeinterleaved;
    void* pUserData;
} mal_channel_router_config;

struct mal_channel_router
{
    mal_channel_router_config config;
    mal_bool32 isPassthrough   : 1;
    mal_bool32 isSimpleShuffle : 1;
    mal_bool32 useSSE2         : 1;
    mal_bool32 useAVX2         : 1;
    mal_bool32 useAVX512       : 1;
    mal_bool32 useNEON         : 1;
    mal_uint8 shuffleTable[MAL_MAX_CHANNELS];
    float weights[MAL_MAX_CHANNELS][MAL_MAX_CHANNELS];
};



typedef struct mal_src mal_src;
//typedef mal_uint32 (* mal_src_read_proc)(mal_src* pSRC, mal_uint32 frameCount, void* pFramesOut, void* pUserData); // Returns the number of frames that were read.
typedef mal_uint32 (* mal_src_read_deinterleaved_proc)(mal_src* pSRC, mal_uint32 frameCount, void** ppSamplesOut, void* pUserData); // Returns the number of frames that were read.

typedef enum
{
    mal_src_algorithm_sinc = 0,
    mal_src_algorithm_linear,
    mal_src_algorithm_none,
    mal_src_algorithm_default = mal_src_algorithm_sinc
} mal_src_algorithm;

typedef enum
{
    mal_src_sinc_window_function_hann = 0,
    mal_src_sinc_window_function_rectangular,
    mal_src_sinc_window_function_default = mal_src_sinc_window_function_hann
} mal_src_sinc_window_function;

typedef struct
{
    mal_src_sinc_window_function windowFunction;
    mal_uint32 windowWidth;
} mal_src_config_sinc;

typedef struct
{
    mal_uint32 sampleRateIn;
    mal_uint32 sampleRateOut;
    mal_uint32 channels;
    mal_src_algorithm algorithm;
    mal_bool32 neverConsumeEndOfInput : 1;
    mal_bool32 noSSE2   : 1;
    mal_bool32 noAVX2   : 1;
    mal_bool32 noAVX512 : 1;
    mal_bool32 noNEON   : 1;
    mal_src_read_deinterleaved_proc onReadDeinterleaved;
    void* pUserData;
    union
    {
        mal_src_config_sinc sinc;
    };
} mal_src_config;

MAL_ALIGNED_STRUCT(MAL_SIMD_ALIGNMENT) mal_src
{
    union
    {
        struct
        {
            MAL_ALIGN(MAL_SIMD_ALIGNMENT) float input[MAL_MAX_CHANNELS][MAL_SRC_INPUT_BUFFER_SIZE_IN_SAMPLES];
            float timeIn;
            mal_uint32 leftoverFrames;
        } linear;

        struct
        {
            MAL_ALIGN(MAL_SIMD_ALIGNMENT) float input[MAL_MAX_CHANNELS][MAL_SRC_SINC_MAX_WINDOW_WIDTH*2 + MAL_SRC_INPUT_BUFFER_SIZE_IN_SAMPLES];
            float timeIn;
            mal_uint32 inputFrameCount;     // The number of frames sitting in the input buffer, not including the first half of the window.
            mal_uint32 windowPosInSamples;  // An offset of <input>.
            float table[MAL_SRC_SINC_MAX_WINDOW_WIDTH*1 * MAL_SRC_SINC_LOOKUP_TABLE_RESOLUTION]; // Precomputed lookup table. The +1 is used to avoid the need for an overflow check.
        } sinc;
    };

    mal_src_config config;
    mal_bool32 isEndOfInputLoaded : 1;
    mal_bool32 useSSE2   : 1;
    mal_bool32 useAVX2   : 1;
    mal_bool32 useAVX512 : 1;
    mal_bool32 useNEON   : 1;
};

typedef struct mal_dsp mal_dsp;
typedef mal_uint32 (* mal_dsp_read_proc)(mal_dsp* pDSP, mal_uint32 frameCount, void* pSamplesOut, void* pUserData);

typedef struct
{
    mal_format formatIn;
    mal_uint32 channelsIn;
    mal_uint32 sampleRateIn;
    mal_channel channelMapIn[MAL_MAX_CHANNELS];
    mal_format formatOut;
    mal_uint32 channelsOut;
    mal_uint32 sampleRateOut;
    mal_channel channelMapOut[MAL_MAX_CHANNELS];
    mal_channel_mix_mode channelMixMode;
    mal_dither_mode ditherMode;
    mal_src_algorithm srcAlgorithm;
    mal_bool32 allowDynamicSampleRate;
    mal_bool32 neverConsumeEndOfInput : 1;  // <-- For SRC.
    mal_bool32 noSSE2   : 1;
    mal_bool32 noAVX2   : 1;
    mal_bool32 noAVX512 : 1;
    mal_bool32 noNEON   : 1;
    mal_dsp_read_proc onRead;
    void* pUserData;
    union
    {
        mal_src_config_sinc sinc;
    };
} mal_dsp_config;

MAL_ALIGNED_STRUCT(MAL_SIMD_ALIGNMENT) mal_dsp
{
    mal_dsp_read_proc onRead;
    void* pUserData;
    mal_format_converter formatConverterIn;             // For converting data to f32 in preparation for further processing.
    mal_format_converter formatConverterOut;            // For converting data to the requested output format. Used as the final step in the processing pipeline.
    mal_channel_router channelRouter;                   // For channel conversion.
    mal_src src;                                        // For sample rate conversion.
    mal_bool32 isDynamicSampleRateAllowed     : 1;      // mal_dsp_set_input_sample_rate() and mal_dsp_set_output_sample_rate() will fail if this is set to false.
    mal_bool32 isPreFormatConversionRequired  : 1;
    mal_bool32 isPostFormatConversionRequired : 1;
    mal_bool32 isChannelRoutingRequired       : 1;
    mal_bool32 isSRCRequired                  : 1;
    mal_bool32 isChannelRoutingAtStart        : 1;
    mal_bool32 isPassthrough                  : 1;      // <-- Will be set to true when the DSP pipeline is an optimized passthrough.
};


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// DATA CONVERSION
// ===============
//
// This section contains the APIs for data conversion. You will find everything here for channel mapping, sample format conversion, resampling, etc.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Channel Maps
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// Helper for retrieving a standard channel map.
void mal_get_standard_channel_map(mal_standard_channel_map standardChannelMap, mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS]);

// Copies a channel map.
void mal_channel_map_copy(mal_channel* pOut, const mal_channel* pIn, mal_uint32 channels);


// Determines whether or not a channel map is valid.
//
// A blank channel map is valid (all channels set to MAL_CHANNEL_NONE). The way a blank channel map is handled is context specific, but
// is usually treated as a passthrough.
//
// Invalid channel maps:
//   - A channel map with no channels
//   - A channel map with more than one channel and a mono channel
mal_bool32 mal_channel_map_valid(mal_uint32 channels, const mal_channel channelMap[MAL_MAX_CHANNELS]);

// Helper for comparing two channel maps for equality.
//
// This assumes the channel count is the same between the two.
mal_bool32 mal_channel_map_equal(mal_uint32 channels, const mal_channel channelMapA[MAL_MAX_CHANNELS], const mal_channel channelMapB[MAL_MAX_CHANNELS]);

// Helper for determining if a channel map is blank (all channels set to MAL_CHANNEL_NONE).
mal_bool32 mal_channel_map_blank(mal_uint32 channels, const mal_channel channelMap[MAL_MAX_CHANNELS]);

// Helper for determining whether or not a channel is present in the given channel map.
mal_bool32 mal_channel_map_contains_channel_position(mal_uint32 channels, const mal_channel channelMap[MAL_MAX_CHANNELS], mal_channel channelPosition);


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Format Conversion
// =================
// The format converter serves two purposes:
//   1) Conversion between data formats (u8 to f32, etc.)
//   2) Interleaving and deinterleaving
//
// When initializing a converter, you specify the input and output formats (u8, s16, etc.) and read callbacks. There are two read callbacks - one for
// interleaved input data (onRead) and another for deinterleaved input data (onReadDeinterleaved). You implement whichever is most convenient for you. You
// can implement both, but it's not recommended as it just introduces unnecessary complexity.
//
// To read data as interleaved samples, use mal_format_converter_read(). Otherwise use mal_format_converter_read_deinterleaved().
//
// Dithering
// ---------
// The format converter also supports dithering. Dithering can be set using ditherMode variable in the config, like so.
//
//   pConfig->ditherMode = mal_dither_mode_rectangle;
//
// The different dithering modes include the following, in order of efficiency:
//   - None:      mal_dither_mode_none
//   - Rectangle: mal_dither_mode_rectangle
//   - Triangle:  mal_dither_mode_triangle
//
// Note that even if the dither mode is set to something other than mal_dither_mode_none, it will be ignored for conversions where dithering is not needed.
// Dithering is available for the following conversions:
//   - s16 -> u8
//   - s24 -> u8
//   - s32 -> u8
//   - f32 -> u8
//   - s24 -> s16
//   - s32 -> s16
//   - f32 -> s16
//
// Note that it is not an error to pass something other than mal_dither_mode_none for conversions where dither is not used. It will just be ignored.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// Initializes a format converter.
mal_result mal_format_converter_init(const mal_format_converter_config* pConfig, mal_format_converter* pConverter);

// Reads data from the format converter as interleaved channels.
mal_uint64 mal_format_converter_read(mal_format_converter* pConverter, mal_uint64 frameCount, void* pFramesOut, void* pUserData);

// Reads data from the format converter as deinterleaved channels.
mal_uint64 mal_format_converter_read_deinterleaved(mal_format_converter* pConverter, mal_uint64 frameCount, void** ppSamplesOut, void* pUserData);


// Helper for initializing a format converter config.
mal_format_converter_config mal_format_converter_config_init_new(void);
mal_format_converter_config mal_format_converter_config_init(mal_format formatIn, mal_format formatOut, mal_uint32 channels, mal_format_converter_read_proc onRead, void* pUserData);
mal_format_converter_config mal_format_converter_config_init_deinterleaved(mal_format formatIn, mal_format formatOut, mal_uint32 channels, mal_format_converter_read_deinterleaved_proc onReadDeinterleaved, void* pUserData);



//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Channel Routing
// ===============
// There are two main things you can do with the channel router:
//   1) Rearrange channels
//   2) Convert from one channel count to another
//
// Channel Rearrangement
// ---------------------
// A simple example of channel rearrangement may be swapping the left and right channels in a stereo stream. To do this you just pass in the same channel
// count for both the input and output with channel maps that contain the same channels (in a different order).
//
// Channel Conversion
// ------------------
// The channel router can also convert from one channel count to another, such as converting a 5.1 stream to stero. When changing the channel count, the
// router will first perform a 1:1 mapping of channel positions that are present in both the input and output channel maps. The second thing it will do
// is distribute the input mono channel (if any) across all output channels, excluding any None and LFE channels. If there is an output mono channel, all
// input channels will be averaged, excluding any None and LFE channels.
//
// The last case to consider is when a channel position in the input channel map is not present in the output channel map, and vice versa. In this case the
// channel router will perform a blend of other related channels to produce an audible channel. There are several blending modes.
//   1) Simple
//      Unmatched channels are silenced.
//   2) Planar Blending
//      Channels are blended based on a set of planes that each speaker emits audio from.
//
// Planar Blending
// ---------------
// In this mode, channel positions are associated with a set of planes where the channel conceptually emits audio from. An example is the front/left speaker.
// This speaker is positioned to the front of the listener, so you can think of it as emitting audio from the front plane. It is also positioned to the left
// of the listener so you can think of it as also emitting audio from the left plane. Now consider the (unrealistic) situation where the input channel map
// contains only the front/left channel position, but the output channel map contains both the front/left and front/center channel. When deciding on the audio
// data to send to the front/center speaker (which has no 1:1 mapping with an input channel) we need to use some logic based on our available input channel
// positions.
//
// As mentioned earlier, our front/left speaker is, conceptually speaking, emitting audio from the front _and_ the left planes. Similarly, the front/center
// speaker is emitting audio from _only_ the front plane. What these two channels have in common is that they are both emitting audio from the front plane.
// Thus, it makes sense that the front/center speaker should receive some contribution from the front/left channel. How much contribution depends on their
// planar relationship (thus the name of this blending technique).
//
// Because the front/left channel is emitting audio from two planes (front and left), you can think of it as though it's willing to dedicate 50% of it's total
// volume to each of it's planes (a channel position emitting from 1 plane would be willing to given 100% of it's total volume to that plane, and a channel
// position emitting from 3 planes would be willing to given 33% of it's total volume to each plane). Similarly, the front/center speaker is emitting audio
// from only one plane so you can think of it as though it's willing to _take_ 100% of it's volume from front plane emissions. Now, since the front/left
// channel is willing to _give_ 50% of it's total volume to the front plane, and the front/center speaker is willing to _take_ 100% of it's total volume
// from the front, you can imagine that 50% of the front/left speaker will be given to the front/center speaker.
//
// Usage
// -----
// To use the channel router you need to specify three things:
//   1) The input channel count and channel map
//   2) The output channel count and channel map
//   3) The mixing mode to use in the case where a 1:1 mapping is unavailable
//
// Note that input and output data is always deinterleaved 32-bit floating point.
//
// Initialize the channel router with mal_channel_router_init(). You will need to pass in a config object which specifies the input and output configuration,
// mixing mode and a callback for sending data to the router. This callback will be called when input data needs to be sent to the router for processing.
//
// Read data from the channel router with mal_channel_router_read_deinterleaved(). Output data is always 32-bit floating point.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// Initializes a channel router where it is assumed that the input data is non-interleaved.
mal_result mal_channel_router_init(const mal_channel_router_config* pConfig, mal_channel_router* pRouter);

// Reads data from the channel router as deinterleaved channels.
mal_uint64 mal_channel_router_read_deinterleaved(mal_channel_router* pRouter, mal_uint64 frameCount, void** ppSamplesOut, void* pUserData);

// Helper for initializing a channel router config.
mal_channel_router_config mal_channel_router_config_init(mal_uint32 channelsIn, const mal_channel channelMapIn[MAL_MAX_CHANNELS], mal_uint32 channelsOut, const mal_channel channelMapOut[MAL_MAX_CHANNELS], mal_channel_mix_mode mixingMode, mal_channel_router_read_deinterleaved_proc onRead, void* pUserData);


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Sample Rate Conversion
// ======================
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// Initializes a sample rate conversion object.
mal_result mal_src_init(const mal_src_config* pConfig, mal_src* pSRC);

// Dynamically adjusts the input sample rate.
//
// DEPRECATED. Use mal_src_set_sample_rate() instead.
mal_result mal_src_set_input_sample_rate(mal_src* pSRC, mal_uint32 sampleRateIn);

// Dynamically adjusts the output sample rate.
//
// This is useful for dynamically adjust pitch. Keep in mind, however, that this will speed up or slow down the sound. If this
// is not acceptable you will need to use your own algorithm.
//
// DEPRECATED. Use mal_src_set_sample_rate() instead.
mal_result mal_src_set_output_sample_rate(mal_src* pSRC, mal_uint32 sampleRateOut);

// Dynamically adjusts the sample rate.
//
// This is useful for dynamically adjust pitch. Keep in mind, however, that this will speed up or slow down the sound. If this
// is not acceptable you will need to use your own algorithm.
mal_result mal_src_set_sample_rate(mal_src* pSRC, mal_uint32 sampleRateIn, mal_uint32 sampleRateOut);

// Reads a number of frames.
//
// Returns the number of frames actually read.
mal_uint64 mal_src_read_deinterleaved(mal_src* pSRC, mal_uint64 frameCount, void** ppSamplesOut, void* pUserData);


// Helper for creating a sample rate conversion config.
mal_src_config mal_src_config_init_new(void);
mal_src_config mal_src_config_init(mal_uint32 sampleRateIn, mal_uint32 sampleRateOut, mal_uint32 channels, mal_src_read_deinterleaved_proc onReadDeinterleaved, void* pUserData);


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// DSP
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// Initializes a DSP object.
mal_result mal_dsp_init(const mal_dsp_config* pConfig, mal_dsp* pDSP);

// Dynamically adjusts the input sample rate.
//
// This will fail is the DSP was not initialized with allowDynamicSampleRate.
//
// DEPRECATED. Use mal_dsp_set_sample_rate() instead.
mal_result mal_dsp_set_input_sample_rate(mal_dsp* pDSP, mal_uint32 sampleRateOut);

// Dynamically adjusts the output sample rate.
//
// This is useful for dynamically adjust pitch. Keep in mind, however, that this will speed up or slow down the sound. If this
// is not acceptable you will need to use your own algorithm.
//
// This will fail is the DSP was not initialized with allowDynamicSampleRate.
//
// DEPRECATED. Use mal_dsp_set_sample_rate() instead.
mal_result mal_dsp_set_output_sample_rate(mal_dsp* pDSP, mal_uint32 sampleRateOut);

// Dynamically adjusts the output sample rate.
//
// This is useful for dynamically adjust pitch. Keep in mind, however, that this will speed up or slow down the sound. If this
// is not acceptable you will need to use your own algorithm.
//
// This will fail is the DSP was not initialized with allowDynamicSampleRate.
mal_result mal_dsp_set_sample_rate(mal_dsp* pDSP, mal_uint32 sampleRateIn, mal_uint32 sampleRateOut);


// Reads a number of frames and runs them through the DSP processor.
mal_uint64 mal_dsp_read(mal_dsp* pDSP, mal_uint64 frameCount, void* pFramesOut, void* pUserData);

// Helper for initializing a mal_dsp_config object.
mal_dsp_config mal_dsp_config_init_new(void);
mal_dsp_config mal_dsp_config_init(mal_format formatIn, mal_uint32 channelsIn, mal_uint32 sampleRateIn, mal_format formatOut, mal_uint32 channelsOut, mal_uint32 sampleRateOut, mal_dsp_read_proc onRead, void* pUserData);
mal_dsp_config mal_dsp_config_init_ex(mal_format formatIn, mal_uint32 channelsIn, mal_uint32 sampleRateIn, mal_channel channelMapIn[MAL_MAX_CHANNELS], mal_format formatOut, mal_uint32 channelsOut, mal_uint32 sampleRateOut,  mal_channel channelMapOut[MAL_MAX_CHANNELS], mal_dsp_read_proc onRead, void* pUserData);


// High-level helper for doing a full format conversion in one go. Returns the number of output frames. Call this with pOut set to NULL to
// determine the required size of the output buffer.
//
// A return value of 0 indicates an error.
//
// This function is useful for one-off bulk conversions, but if you're streaming data you should use the DSP APIs instead.
mal_uint64 mal_convert_frames(void* pOut, mal_format formatOut, mal_uint32 channelsOut, mal_uint32 sampleRateOut, const void* pIn, mal_format formatIn, mal_uint32 channelsIn, mal_uint32 sampleRateIn, mal_uint64 frameCountIn);
mal_uint64 mal_convert_frames_ex(void* pOut, mal_format formatOut, mal_uint32 channelsOut, mal_uint32 sampleRateOut, mal_channel channelMapOut[MAL_MAX_CHANNELS], const void* pIn, mal_format formatIn, mal_uint32 channelsIn, mal_uint32 sampleRateIn, mal_channel channelMapIn[MAL_MAX_CHANNELS], mal_uint64 frameCountIn);



//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Miscellaneous Helpers
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// malloc(). Calls MAL_MALLOC().
void* mal_malloc(size_t sz);

// realloc(). Calls MAL_REALLOC().
void* mal_realloc(void* p, size_t sz);

// free(). Calls MAL_FREE().
void mal_free(void* p);

// Performs an aligned malloc, with the assumption that the alignment is a power of 2.
void* mal_aligned_malloc(size_t sz, size_t alignment);

// Free's an aligned malloc'd buffer.
void mal_aligned_free(void* p);

// Retrieves a friendly name for a format.
const char* mal_get_format_name(mal_format format);

// Blends two frames in floating point format.
void mal_blend_f32(float* pOut, float* pInA, float* pInB, float factor, mal_uint32 channels);

// Retrieves the size of a sample in bytes for the given format.
//
// This API is efficient and is implemented using a lookup table.
//
// Thread Safety: SAFE
//   This is API is pure.
mal_uint32 mal_get_bytes_per_sample(mal_format format);
static MAL_INLINE mal_uint32 mal_get_bytes_per_frame(mal_format format, mal_uint32 channels) { return mal_get_bytes_per_sample(format) * channels; }


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Format Conversion
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void mal_pcm_u8_to_s16(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_u8_to_s24(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_u8_to_s32(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_u8_to_f32(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s16_to_u8(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s16_to_s24(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s16_to_s32(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s16_to_f32(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s24_to_u8(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s24_to_s16(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s24_to_s32(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s24_to_f32(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s32_to_u8(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s32_to_s16(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s32_to_s24(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_s32_to_f32(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_f32_to_u8(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_f32_to_s16(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_f32_to_s24(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_f32_to_s32(void* pOut, const void* pIn, mal_uint64 count, mal_dither_mode ditherMode);
void mal_pcm_convert(void* pOut, mal_format formatOut, const void* pIn, mal_format formatIn, mal_uint64 sampleCount, mal_dither_mode ditherMode);



//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// DEVICE I/O
// ==========
//
// This section contains the APIs for device playback and capture. Here is where you'll find mal_device_init(), etc.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef MAL_NO_DEVICE_IO
// Some backends are only supported on certain platforms.
#if defined(MAL_WIN32)
    #define MAL_SUPPORT_WASAPI
    #if defined(MAL_WIN32_DESKTOP)  // DirectSound and WinMM backends are only supported on desktop's.
        #define MAL_SUPPORT_DSOUND
        #define MAL_SUPPORT_WINMM
        #define MAL_SUPPORT_JACK    // JACK is technically supported on Windows, but I don't know how many people use it in practice...
    #endif
#endif
#if defined(MAL_UNIX)
    #if defined(MAL_LINUX)
        #if !defined(MAL_ANDROID)   // ALSA is not supported on Android.
            #define MAL_SUPPORT_ALSA
        #endif
    #endif
    #if !defined(MAL_BSD) && !defined(MAL_ANDROID) && !defined(MAL_EMSCRIPTEN)
        #define MAL_SUPPORT_PULSEAUDIO
        #define MAL_SUPPORT_JACK
    #endif
    #if defined(MAL_ANDROID)
        #define MAL_SUPPORT_OPENSL
    #endif
    #if defined(__OpenBSD__)    // <-- Change this to "#if defined(MAL_BSD)" to enable sndio on all BSD flavors.
        #define MAL_SUPPORT_SNDIO   // sndio is only supported on OpenBSD for now. May be expanded later if there's demand.
    #endif
    #if defined(__NetBSD__) || defined(__OpenBSD__)
        #define MAL_SUPPORT_AUDIO4  // Only support audio(4) on platforms with known support.
    #endif
    #if defined(__FreeBSD__) || defined(__DragonFly__)
        #define MAL_SUPPORT_OSS     // Only support OSS on specific platforms with known support.
    #endif
#endif
#if defined(MAL_APPLE)
    #define MAL_SUPPORT_COREAUDIO
#endif

#define MAL_SUPPORT_SDL     // All platforms support SDL.

// Explicitly disable OpenAL and Null backends for Emscripten because they both use a background thread which is not properly supported right now.
#if !defined(MAL_EMSCRIPTEN)
#define MAL_SUPPORT_OPENAL
#define MAL_SUPPORT_NULL    // All platforms support the null backend.
#endif


#if !defined(MAL_NO_WASAPI) && defined(MAL_SUPPORT_WASAPI)
    #define MAL_ENABLE_WASAPI
#endif
#if !defined(MAL_NO_DSOUND) && defined(MAL_SUPPORT_DSOUND)
    #define MAL_ENABLE_DSOUND
#endif
#if !defined(MAL_NO_WINMM) && defined(MAL_SUPPORT_WINMM)
    #define MAL_ENABLE_WINMM
#endif
#if !defined(MAL_NO_ALSA) && defined(MAL_SUPPORT_ALSA)
    #define MAL_ENABLE_ALSA
#endif
#if !defined(MAL_NO_PULSEAUDIO) && defined(MAL_SUPPORT_PULSEAUDIO)
    #define MAL_ENABLE_PULSEAUDIO
#endif
#if !defined(MAL_NO_JACK) && defined(MAL_SUPPORT_JACK)
    #define MAL_ENABLE_JACK
#endif
#if !defined(MAL_NO_COREAUDIO) && defined(MAL_SUPPORT_COREAUDIO)
    #define MAL_ENABLE_COREAUDIO
#endif
#if !defined(MAL_NO_SNDIO) && defined(MAL_SUPPORT_SNDIO)
    #define MAL_ENABLE_SNDIO
#endif
#if !defined(MAL_NO_AUDIO4) && defined(MAL_SUPPORT_AUDIO4)
    #define MAL_ENABLE_AUDIO4
#endif
#if !defined(MAL_NO_OSS) && defined(MAL_SUPPORT_OSS)
    #define MAL_ENABLE_OSS
#endif
#if !defined(MAL_NO_OPENSL) && defined(MAL_SUPPORT_OPENSL)
    #define MAL_ENABLE_OPENSL
#endif
#if !defined(MAL_NO_OPENAL) && defined(MAL_SUPPORT_OPENAL)
    #define MAL_ENABLE_OPENAL
#endif
#if !defined(MAL_NO_SDL) && defined(MAL_SUPPORT_SDL)
    #define MAL_ENABLE_SDL
#endif
#if !defined(MAL_NO_NULL) && defined(MAL_SUPPORT_NULL)
    #define MAL_ENABLE_NULL
#endif

#ifdef MAL_SUPPORT_WASAPI
// We need a IMMNotificationClient object for WASAPI. 
typedef struct
{
    void* lpVtbl;
    mal_uint32 counter;
    mal_device* pDevice;
} mal_IMMNotificationClient;
#endif


typedef enum
{
    mal_backend_null,
    mal_backend_wasapi,
    mal_backend_dsound,
    mal_backend_winmm,
    mal_backend_alsa,
    mal_backend_pulseaudio,
    mal_backend_jack,
    mal_backend_coreaudio,
    mal_backend_sndio,
    mal_backend_audio4,
    mal_backend_oss,
    mal_backend_opensl,
    mal_backend_openal,
    mal_backend_sdl
} mal_backend;

// Thread priorties should be ordered such that the default priority of the worker thread is 0.
typedef enum
{
    mal_thread_priority_idle     = -5,
    mal_thread_priority_lowest   = -4,
    mal_thread_priority_low      = -3,
    mal_thread_priority_normal   = -2,
    mal_thread_priority_high     = -1,
    mal_thread_priority_highest  =  0,
    mal_thread_priority_realtime =  1,
    mal_thread_priority_default  =  0
} mal_thread_priority;

typedef struct
{
    mal_context* pContext;

    union
    {
#ifdef MAL_WIN32
        struct
        {
            /*HANDLE*/ mal_handle hThread;
        } win32;
#endif
#ifdef MAL_POSIX
        struct
        {
            pthread_t thread;
        } posix;
#endif

        int _unused;
    };
} mal_thread;

typedef struct
{
    mal_context* pContext;

    union
    {
#ifdef MAL_WIN32
        struct
        {
            /*HANDLE*/ mal_handle hMutex;
        } win32;
#endif
#ifdef MAL_POSIX
        struct
        {
            pthread_mutex_t mutex;
        } posix;
#endif

        int _unused;
    };
} mal_mutex;

typedef struct
{
    mal_context* pContext;

    union
    {
#ifdef MAL_WIN32
        struct
        {
            /*HANDLE*/ mal_handle hEvent;
        } win32;
#endif
#ifdef MAL_POSIX
        struct
        {
            pthread_mutex_t mutex;
            pthread_cond_t condition;
            mal_uint32 value;
        } posix;
#endif

        int _unused;
    };
} mal_event;


#define MAL_MAX_PERIODS_DSOUND                          4
#define MAL_MAX_PERIODS_OPENAL                          4

typedef void       (* mal_log_proc) (mal_context* pContext, mal_device* pDevice, const char* message);
typedef void       (* mal_recv_proc)(mal_device* pDevice, mal_uint32 frameCount, const void* pSamples);
typedef mal_uint32 (* mal_send_proc)(mal_device* pDevice, mal_uint32 frameCount, void* pSamples);
typedef void       (* mal_stop_proc)(mal_device* pDevice);

typedef enum
{
    mal_device_type_playback,
    mal_device_type_capture
} mal_device_type;

typedef enum
{
    mal_share_mode_shared = 0,
    mal_share_mode_exclusive,
} mal_share_mode;

typedef union
{
#ifdef MAL_SUPPORT_WASAPI
    wchar_t wasapi[64];             // WASAPI uses a wchar_t string for identification.
#endif
#ifdef MAL_SUPPORT_DSOUND
    mal_uint8 dsound[16];           // DirectSound uses a GUID for identification.
#endif
#ifdef MAL_SUPPORT_WINMM
    /*UINT_PTR*/ mal_uint32 winmm;  // When creating a device, WinMM expects a Win32 UINT_PTR for device identification. In practice it's actually just a UINT.
#endif
#ifdef MAL_SUPPORT_ALSA
    char alsa[256];                 // ALSA uses a name string for identification.
#endif
#ifdef MAL_SUPPORT_PULSEAUDIO
    char pulse[256];                // PulseAudio uses a name string for identification.
#endif
#ifdef MAL_SUPPORT_JACK
    int jack;                       // JACK always uses default devices.
#endif
#ifdef MAL_SUPPORT_COREAUDIO
    char coreaudio[256];            // Core Audio uses a string for identification.
#endif
#ifdef MAL_SUPPORT_SNDIO
    char sndio[256];                // "snd/0", etc.
#endif
#ifdef MAL_SUPPORT_AUDIO4
    char audio4[256];               // "/dev/audio", etc.
#endif
#ifdef MAL_SUPPORT_OSS
    char oss[64];                   // "dev/dsp0", etc. "dev/dsp" for the default device.
#endif
#ifdef MAL_SUPPORT_OPENSL
    mal_uint32 opensl;              // OpenSL|ES uses a 32-bit unsigned integer for identification.
#endif
#ifdef MAL_SUPPORT_OPENAL
    char openal[256];               // OpenAL seems to use human-readable device names as the ID.
#endif
#ifdef MAL_SUPPORT_SDL
    int sdl;                        // SDL devices are identified with an index.
#endif
#ifdef MAL_SUPPORT_NULL
    int nullbackend;                // The null backend uses an integer for device IDs.
#endif
} mal_device_id;

typedef struct
{
    // Basic info. This is the only information guaranteed to be filled in during device enumeration.
    mal_device_id id;
    char name[256];

    // Detailed info. As much of this is filled as possible with mal_context_get_device_info(). Note that you are allowed to initialize
    // a device with settings outside of this range, but it just means the data will be converted using mini_al's data conversion
    // pipeline before sending the data to/from the device. Most programs will need to not worry about these values, but it's provided
    // here mainly for informational purposes or in the rare case that someone might find it useful.
    //
    // These will be set to 0 when returned by mal_context_enumerate_devices() or mal_context_get_devices().
    mal_uint32 formatCount;
    mal_format formats[mal_format_count];
    mal_uint32 minChannels;
    mal_uint32 maxChannels;
    mal_uint32 minSampleRate;
    mal_uint32 maxSampleRate;
} mal_device_info;

typedef struct
{
    mal_int64 counter;
} mal_timer;

typedef struct
{
    mal_format format;
    mal_uint32 channels;
    mal_uint32 sampleRate;
    mal_channel channelMap[MAL_MAX_CHANNELS];
    mal_uint32 bufferSizeInFrames;
    mal_uint32 bufferSizeInMilliseconds;
    mal_uint32 periods;
    mal_share_mode shareMode;
    mal_performance_profile performanceProfile;
    mal_recv_proc onRecvCallback;
    mal_send_proc onSendCallback;
    mal_stop_proc onStopCallback;

    struct
    {
        mal_bool32 noMMap;  // Disables MMap mode.
    } alsa;

    struct
    {
        const char* pStreamName;
    } pulse;
} mal_device_config;

typedef struct
{
    mal_log_proc onLog;
    mal_thread_priority threadPriority;

    struct
    {
        mal_bool32 useVerboseDeviceEnumeration;
    } alsa;

    struct
    {
        const char* pApplicationName;
        const char* pServerName;
        mal_bool32 tryAutoSpawn; // Enables autospawning of the PulseAudio daemon if necessary.
    } pulse;

    struct
    {
        const char* pClientName;
        mal_bool32 tryStartServer;
    } jack;
} mal_context_config;

typedef mal_bool32 (* mal_enum_devices_callback_proc)(mal_context* pContext, mal_device_type type, const mal_device_info* pInfo, void* pUserData);

struct mal_context
{
    mal_backend backend;                    // DirectSound, ALSA, etc.
    mal_context_config config;
    mal_mutex deviceEnumLock;               // Used to make mal_context_get_devices() thread safe.
    mal_mutex deviceInfoLock;               // Used to make mal_context_get_device_info() thread safe.
    mal_uint32 deviceInfoCapacity;          // Total capacity of pDeviceInfos.
    mal_uint32 playbackDeviceInfoCount;
    mal_uint32 captureDeviceInfoCount;
    mal_device_info* pDeviceInfos;          // Playback devices first, then capture.
    mal_bool32 isBackendAsynchronous : 1;   // Set when the context is initialized. Set to 1 for asynchronous backends such as Core Audio and JACK. Do not modify.

    mal_result (* onUninit             )(mal_context* pContext);
    mal_bool32 (* onDeviceIDEqual      )(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1);
    mal_result (* onEnumDevices        )(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData);    // Return false from the callback to stop enumeration.
    mal_result (* onGetDeviceInfo      )(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo);
    mal_result (* onDeviceInit         )(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice);
    void       (* onDeviceUninit       )(mal_device* pDevice);
    mal_result (* onDeviceReinit       )(mal_device* pDevice);
    mal_result (* onDeviceStart        )(mal_device* pDevice);
    mal_result (* onDeviceStop         )(mal_device* pDevice);
    mal_result (* onDeviceBreakMainLoop)(mal_device* pDevice);
    mal_result (* onDeviceMainLoop     )(mal_device* pDevice);

    union
    {
#ifdef MAL_SUPPORT_WASAPI
        struct
        {
            int _unused;
        } wasapi;
#endif
#ifdef MAL_SUPPORT_DSOUND
        struct
        {
            /*HMODULE*/ mal_handle hDSoundDLL;
            mal_proc DirectSoundCreate;
            mal_proc DirectSoundEnumerateA;
            mal_proc DirectSoundCaptureCreate;
            mal_proc DirectSoundCaptureEnumerateA;
        } dsound;
#endif
#ifdef MAL_SUPPORT_WINMM
        struct
        {
            /*HMODULE*/ mal_handle hWinMM;
            mal_proc waveOutGetNumDevs;
            mal_proc waveOutGetDevCapsA;
            mal_proc waveOutOpen;
            mal_proc waveOutClose;
            mal_proc waveOutPrepareHeader;
            mal_proc waveOutUnprepareHeader;
            mal_proc waveOutWrite;
            mal_proc waveOutReset;
            mal_proc waveInGetNumDevs;
            mal_proc waveInGetDevCapsA;
            mal_proc waveInOpen;
            mal_proc waveInClose;
            mal_proc waveInPrepareHeader;
            mal_proc waveInUnprepareHeader;
            mal_proc waveInAddBuffer;
            mal_proc waveInStart;
            mal_proc waveInReset;
        } winmm;
#endif
#ifdef MAL_SUPPORT_ALSA
        struct
        {
            mal_handle asoundSO;
            mal_proc snd_pcm_open;
            mal_proc snd_pcm_close;
            mal_proc snd_pcm_hw_params_sizeof;
            mal_proc snd_pcm_hw_params_any;
            mal_proc snd_pcm_hw_params_set_format;
            mal_proc snd_pcm_hw_params_set_format_first;
            mal_proc snd_pcm_hw_params_get_format_mask;
            mal_proc snd_pcm_hw_params_set_channels_near;
            mal_proc snd_pcm_hw_params_set_rate_resample;
            mal_proc snd_pcm_hw_params_set_rate_near;
            mal_proc snd_pcm_hw_params_set_buffer_size_near;
            mal_proc snd_pcm_hw_params_set_periods_near;
            mal_proc snd_pcm_hw_params_set_access;
            mal_proc snd_pcm_hw_params_get_format;
            mal_proc snd_pcm_hw_params_get_channels;
            mal_proc snd_pcm_hw_params_get_channels_min;
            mal_proc snd_pcm_hw_params_get_channels_max;
            mal_proc snd_pcm_hw_params_get_rate;
            mal_proc snd_pcm_hw_params_get_rate_min;
            mal_proc snd_pcm_hw_params_get_rate_max;
            mal_proc snd_pcm_hw_params_get_buffer_size;
            mal_proc snd_pcm_hw_params_get_periods;
            mal_proc snd_pcm_hw_params_get_access;
            mal_proc snd_pcm_hw_params;
            mal_proc snd_pcm_sw_params_sizeof;
            mal_proc snd_pcm_sw_params_current;
            mal_proc snd_pcm_sw_params_set_avail_min;
            mal_proc snd_pcm_sw_params_set_start_threshold;
            mal_proc snd_pcm_sw_params;
            mal_proc snd_pcm_format_mask_sizeof;
            mal_proc snd_pcm_format_mask_test;
            mal_proc snd_pcm_get_chmap;
            mal_proc snd_pcm_prepare;
            mal_proc snd_pcm_start;
            mal_proc snd_pcm_drop;
            mal_proc snd_device_name_hint;
            mal_proc snd_device_name_get_hint;
            mal_proc snd_card_get_index;
            mal_proc snd_device_name_free_hint;
            mal_proc snd_pcm_mmap_begin;
            mal_proc snd_pcm_mmap_commit;
            mal_proc snd_pcm_recover;
            mal_proc snd_pcm_readi;
            mal_proc snd_pcm_writei;
            mal_proc snd_pcm_avail;
            mal_proc snd_pcm_avail_update;
            mal_proc snd_pcm_wait;
            mal_proc snd_pcm_info;
            mal_proc snd_pcm_info_sizeof;
            mal_proc snd_pcm_info_get_name;
            mal_proc snd_config_update_free_global;

            mal_mutex internalDeviceEnumLock;
        } alsa;
#endif
#ifdef MAL_SUPPORT_PULSEAUDIO
        struct
        {
            mal_handle pulseSO;
            mal_proc pa_mainloop_new;
            mal_proc pa_mainloop_free;
            mal_proc pa_mainloop_get_api;
            mal_proc pa_mainloop_iterate;
            mal_proc pa_mainloop_wakeup;
            mal_proc pa_context_new;
            mal_proc pa_context_unref;
            mal_proc pa_context_connect;
            mal_proc pa_context_disconnect;
            mal_proc pa_context_set_state_callback;
            mal_proc pa_context_get_state;
            mal_proc pa_context_get_sink_info_list;
            mal_proc pa_context_get_source_info_list;
            mal_proc pa_context_get_sink_info_by_name;
            mal_proc pa_context_get_source_info_by_name;
            mal_proc pa_operation_unref;
            mal_proc pa_operation_get_state;
            mal_proc pa_channel_map_init_extend;
            mal_proc pa_channel_map_valid;
            mal_proc pa_channel_map_compatible;
            mal_proc pa_stream_new;
            mal_proc pa_stream_unref;
            mal_proc pa_stream_connect_playback;
            mal_proc pa_stream_connect_record;
            mal_proc pa_stream_disconnect;
            mal_proc pa_stream_get_state;
            mal_proc pa_stream_get_sample_spec;
            mal_proc pa_stream_get_channel_map;
            mal_proc pa_stream_get_buffer_attr;
            mal_proc pa_stream_get_device_name;
            mal_proc pa_stream_set_write_callback;
            mal_proc pa_stream_set_read_callback;
            mal_proc pa_stream_flush;
            mal_proc pa_stream_drain;
            mal_proc pa_stream_cork;
            mal_proc pa_stream_trigger;
            mal_proc pa_stream_begin_write;
            mal_proc pa_stream_write;
            mal_proc pa_stream_peek;
            mal_proc pa_stream_drop;
        } pulse;
#endif
#ifdef MAL_SUPPORT_JACK
        struct
        {
            mal_handle jackSO;
            mal_proc jack_client_open;
            mal_proc jack_client_close;
            mal_proc jack_client_name_size;
            mal_proc jack_set_process_callback;
            mal_proc jack_set_buffer_size_callback;
            mal_proc jack_on_shutdown;
            mal_proc jack_get_sample_rate;
            mal_proc jack_get_buffer_size;
            mal_proc jack_get_ports;
            mal_proc jack_activate;
            mal_proc jack_deactivate;
            mal_proc jack_connect;
            mal_proc jack_port_register;
            mal_proc jack_port_name;
            mal_proc jack_port_get_buffer;
            mal_proc jack_free;
        } jack;
#endif
#ifdef MAL_SUPPORT_COREAUDIO
        struct
        {
            mal_handle hCoreFoundation;
            mal_proc CFStringGetCString;
            
            mal_handle hCoreAudio;
            mal_proc AudioObjectGetPropertyData;
            mal_proc AudioObjectGetPropertyDataSize;
            mal_proc AudioObjectSetPropertyData;
            mal_proc AudioObjectAddPropertyListener;
            
            mal_handle hAudioUnit;  // Could possibly be set to AudioToolbox on later versions of macOS.
            mal_proc AudioComponentFindNext;
            mal_proc AudioComponentInstanceDispose;
            mal_proc AudioComponentInstanceNew;
            mal_proc AudioOutputUnitStart;
            mal_proc AudioOutputUnitStop;
            mal_proc AudioUnitAddPropertyListener;
            mal_proc AudioUnitGetProperty;
            mal_proc AudioUnitSetProperty;
            mal_proc AudioUnitInitialize;
            mal_proc AudioUnitRender;
        } coreaudio;
#endif
#ifdef MAL_SUPPORT_SNDIO
        struct
        {
            mal_handle sndioSO;
            mal_proc sio_open;
            mal_proc sio_close;
            mal_proc sio_setpar;
            mal_proc sio_getpar;
            mal_proc sio_getcap;
            mal_proc sio_start;
            mal_proc sio_stop;
            mal_proc sio_read;
            mal_proc sio_write;
            mal_proc sio_onmove;
            mal_proc sio_nfds;
            mal_proc sio_pollfd;
            mal_proc sio_revents;
            mal_proc sio_eof;
            mal_proc sio_setvol;
            mal_proc sio_onvol;
            mal_proc sio_initpar;
        } sndio;
#endif
#ifdef MAL_SUPPORT_AUDIO4
        struct
        {
            int _unused;
        } audio4;
#endif
#ifdef MAL_SUPPORT_OSS
        struct
        {
            int versionMajor;
            int versionMinor;
        } oss;
#endif
#ifdef MAL_SUPPORT_OPENSL
        struct
        {
            int _unused;
        } opensl;
#endif
#ifdef MAL_SUPPORT_OPENAL
        struct
        {
            /*HMODULE*/ mal_handle hOpenAL;     // OpenAL32.dll, etc.
            mal_proc alcCreateContext;
            mal_proc alcMakeContextCurrent;
            mal_proc alcProcessContext;
            mal_proc alcSuspendContext;
            mal_proc alcDestroyContext;
            mal_proc alcGetCurrentContext;
            mal_proc alcGetContextsDevice;
            mal_proc alcOpenDevice;
            mal_proc alcCloseDevice;
            mal_proc alcGetError;
            mal_proc alcIsExtensionPresent;
            mal_proc alcGetProcAddress;
            mal_proc alcGetEnumValue;
            mal_proc alcGetString;
            mal_proc alcGetIntegerv;
            mal_proc alcCaptureOpenDevice;
            mal_proc alcCaptureCloseDevice;
            mal_proc alcCaptureStart;
            mal_proc alcCaptureStop;
            mal_proc alcCaptureSamples;
            mal_proc alEnable;
            mal_proc alDisable;
            mal_proc alIsEnabled;
            mal_proc alGetString;
            mal_proc alGetBooleanv;
            mal_proc alGetIntegerv;
            mal_proc alGetFloatv;
            mal_proc alGetDoublev;
            mal_proc alGetBoolean;
            mal_proc alGetInteger;
            mal_proc alGetFloat;
            mal_proc alGetDouble;
            mal_proc alGetError;
            mal_proc alIsExtensionPresent;
            mal_proc alGetProcAddress;
            mal_proc alGetEnumValue;
            mal_proc alGenSources;
            mal_proc alDeleteSources;
            mal_proc alIsSource;
            mal_proc alSourcef;
            mal_proc alSource3f;
            mal_proc alSourcefv;
            mal_proc alSourcei;
            mal_proc alSource3i;
            mal_proc alSourceiv;
            mal_proc alGetSourcef;
            mal_proc alGetSource3f;
            mal_proc alGetSourcefv;
            mal_proc alGetSourcei;
            mal_proc alGetSource3i;
            mal_proc alGetSourceiv;
            mal_proc alSourcePlayv;
            mal_proc alSourceStopv;
            mal_proc alSourceRewindv;
            mal_proc alSourcePausev;
            mal_proc alSourcePlay;
            mal_proc alSourceStop;
            mal_proc alSourceRewind;
            mal_proc alSourcePause;
            mal_proc alSourceQueueBuffers;
            mal_proc alSourceUnqueueBuffers;
            mal_proc alGenBuffers;
            mal_proc alDeleteBuffers;
            mal_proc alIsBuffer;
            mal_proc alBufferData;
            mal_proc alBufferf;
            mal_proc alBuffer3f;
            mal_proc alBufferfv;
            mal_proc alBufferi;
            mal_proc alBuffer3i;
            mal_proc alBufferiv;
            mal_proc alGetBufferf;
            mal_proc alGetBuffer3f;
            mal_proc alGetBufferfv;
            mal_proc alGetBufferi;
            mal_proc alGetBuffer3i;
            mal_proc alGetBufferiv;

            mal_bool32 isEnumerationSupported : 1;
            mal_bool32 isFloat32Supported     : 1;
            mal_bool32 isMCFormatsSupported   : 1;
        } openal;
#endif
#ifdef MAL_SUPPORT_SDL
        struct
        {
            mal_handle hSDL;    // SDL
            mal_proc SDL_InitSubSystem;
            mal_proc SDL_QuitSubSystem;
            mal_proc SDL_CloseAudio;
            mal_proc SDL_OpenAudio;
            mal_proc SDL_PauseAudio;
            mal_proc SDL_GetNumAudioDevices;
            mal_proc SDL_GetAudioDeviceName;
            mal_proc SDL_CloseAudioDevice;
            mal_proc SDL_OpenAudioDevice;
            mal_proc SDL_PauseAudioDevice;

            mal_bool32 usingSDL1;
        } sdl;
#endif
#ifdef MAL_SUPPORT_NULL
        struct
        {
            int _unused;
        } null_backend;
#endif
    };

    union
    {
#ifdef MAL_WIN32
        struct
        {
            /*HMODULE*/ mal_handle hOle32DLL;
            mal_proc CoInitializeEx;
            mal_proc CoUninitialize;
            mal_proc CoCreateInstance;
            mal_proc CoTaskMemFree;
            mal_proc PropVariantClear;
            mal_proc StringFromGUID2;

            /*HMODULE*/ mal_handle hUser32DLL;
            mal_proc GetForegroundWindow;
            mal_proc GetDesktopWindow;

            /*HMODULE*/ mal_handle hAdvapi32DLL;
            mal_proc RegOpenKeyExA;
            mal_proc RegCloseKey;
            mal_proc RegQueryValueExA;
        } win32;
#endif
#ifdef MAL_POSIX
        struct
        {
            mal_handle pthreadSO;
            mal_proc pthread_create;
            mal_proc pthread_join;
            mal_proc pthread_mutex_init;
            mal_proc pthread_mutex_destroy;
            mal_proc pthread_mutex_lock;
            mal_proc pthread_mutex_unlock;
            mal_proc pthread_cond_init;
            mal_proc pthread_cond_destroy;
            mal_proc pthread_cond_wait;
            mal_proc pthread_cond_signal;
            mal_proc pthread_attr_init;
            mal_proc pthread_attr_destroy;
            mal_proc pthread_attr_setschedpolicy;
            mal_proc pthread_attr_getschedparam;
            mal_proc pthread_attr_setschedparam;
        } posix;
#endif
        int _unused;
    };
};

MAL_ALIGNED_STRUCT(MAL_SIMD_ALIGNMENT) mal_device
{
    mal_context* pContext;
    mal_device_type type;
    mal_format format;
    mal_uint32 channels;
    mal_uint32 sampleRate;
    mal_channel channelMap[MAL_MAX_CHANNELS];
    mal_uint32 bufferSizeInFrames;
    mal_uint32 bufferSizeInMilliseconds;
    mal_uint32 periods;
    mal_uint32 state;
    mal_recv_proc onRecv;
    mal_send_proc onSend;
    mal_stop_proc onStop;
    void* pUserData;                // Application defined data.
    char name[256];
    mal_device_config initConfig;   // The configuration passed in to mal_device_init(). Mainly used for reinitializing the backend device.
    mal_mutex lock;
    mal_event wakeupEvent;
    mal_event startEvent;
    mal_event stopEvent;
    mal_thread thread;
    mal_result workResult;          // This is set by the worker thread after it's finished doing a job.
    mal_bool32 usingDefaultFormat     : 1;
    mal_bool32 usingDefaultChannels   : 1;
    mal_bool32 usingDefaultSampleRate : 1;
    mal_bool32 usingDefaultChannelMap : 1;
    mal_bool32 usingDefaultBufferSize : 1;
    mal_bool32 usingDefaultPeriods    : 1;
    mal_bool32 exclusiveMode          : 1;
    mal_bool32 isOwnerOfContext       : 1;  // When set to true, uninitializing the device will also uninitialize the context. Set to true when NULL is passed into mal_device_init().
    mal_bool32 isDefaultDevice        : 1;  // Used to determine if the backend should try reinitializing if the default device is unplugged.
    mal_format internalFormat;
    mal_uint32 internalChannels;
    mal_uint32 internalSampleRate;
    mal_channel internalChannelMap[MAL_MAX_CHANNELS];
    mal_dsp dsp;                    // Samples run through this to convert samples to a format suitable for use by the backend.
    mal_uint32 _dspFrameCount;      // Internal use only. Used when running the device -> DSP -> client pipeline. See mal_device__on_read_from_device().
    const mal_uint8* _dspFrames;    // ^^^ AS ABOVE ^^^

    union
    {
#ifdef MAL_SUPPORT_WASAPI
        struct
        {
            /*IAudioClient**/ mal_ptr pAudioClient;
            /*IAudioRenderClient**/ mal_ptr pRenderClient;
            /*IAudioCaptureClient**/ mal_ptr pCaptureClient;
            /*IMMDeviceEnumerator**/ mal_ptr pDeviceEnumerator; /* <-- Used for IMMNotificationClient notifications. Required for detecting default device changes. */
            mal_IMMNotificationClient notificationClient;
            /*HANDLE*/ mal_handle hEvent;
            /*HANDLE*/ mal_handle hBreakEvent;  /* <-- Used to break from WaitForMultipleObjects() in the main loop. */
            mal_bool32 breakFromMainLoop;
            mal_bool32 hasDefaultDeviceChanged; /* <-- Make sure this is always a whole 32-bits because we use atomic assignments. */
        } wasapi;
#endif
#ifdef MAL_SUPPORT_DSOUND
        struct
        {
            /*LPDIRECTSOUND*/ mal_ptr pPlayback;
            /*LPDIRECTSOUNDBUFFER*/ mal_ptr pPlaybackPrimaryBuffer;
            /*LPDIRECTSOUNDBUFFER*/ mal_ptr pPlaybackBuffer;
            /*LPDIRECTSOUNDCAPTURE*/ mal_ptr pCapture;
            /*LPDIRECTSOUNDCAPTUREBUFFER*/ mal_ptr pCaptureBuffer;
            /*LPDIRECTSOUNDNOTIFY*/ mal_ptr pNotify;
            /*HANDLE*/ mal_handle pNotifyEvents[MAL_MAX_PERIODS_DSOUND];  // One event handle for each period.
            /*HANDLE*/ mal_handle hStopEvent;
            mal_uint32 lastProcessedFrame;      // This is circular.
            mal_bool32 breakFromMainLoop;
        } dsound;
#endif
#ifdef MAL_SUPPORT_WINMM
        struct
        {
            /*HWAVEOUT, HWAVEIN*/ mal_handle hDevice;
            /*HANDLE*/ mal_handle hEvent;
            mal_uint32 fragmentSizeInFrames;
            mal_uint32 fragmentSizeInBytes;
            mal_uint32 iNextHeader;             // [0,periods). Used as an index into pWAVEHDR.
            /*WAVEHDR**/ mal_uint8* pWAVEHDR;   // One instantiation for each period.
            mal_uint8* pIntermediaryBuffer;
            mal_uint8* _pHeapData;              // Used internally and is used for the heap allocated data for the intermediary buffer and the WAVEHDR structures.
            mal_bool32 breakFromMainLoop;
        } winmm;
#endif
#ifdef MAL_SUPPORT_ALSA
        struct
        {
            /*snd_pcm_t**/ mal_ptr pPCM;
            mal_bool32 isUsingMMap       : 1;
            mal_bool32 breakFromMainLoop : 1;
            void* pIntermediaryBuffer;
        } alsa;
#endif
#ifdef MAL_SUPPORT_PULSEAUDIO
        struct
        {
            /*pa_mainloop**/ mal_ptr pMainLoop;
            /*pa_mainloop_api**/ mal_ptr pAPI;
            /*pa_context**/ mal_ptr pPulseContext;
            /*pa_stream**/ mal_ptr pStream;
            /*pa_context_state*/ mal_uint32 pulseContextState;
            mal_uint32 fragmentSizeInBytes;
            mal_bool32 breakFromMainLoop : 1;
        } pulse;
#endif
#ifdef MAL_SUPPORT_JACK
        struct
        {
            /*jack_client_t**/ mal_ptr pClient;
            /*jack_port_t**/ mal_ptr pPorts[MAL_MAX_CHANNELS];
            float* pIntermediaryBuffer; // Typed as a float because JACK is always floating point.
        } jack;
#endif
#ifdef MAL_SUPPORT_COREAUDIO
        struct
        {
            mal_uint32 deviceObjectID;
            /*AudioComponent*/ mal_ptr component;   // <-- Can this be per-context?
            /*AudioUnit*/ mal_ptr audioUnit;
            /*AudioBufferList**/ mal_ptr pAudioBufferList;  // Only used for input devices.
            mal_bool32 isSwitchingDevice;   /* <-- Set to true when the default device has changed and mini_al is in the process of switching. */
        } coreaudio;
#endif
#ifdef MAL_SUPPORT_SNDIO
        struct
        {
            mal_ptr handle;
            mal_uint32 fragmentSizeInFrames;
            mal_bool32 breakFromMainLoop;
            void* pIntermediaryBuffer;
        } sndio;
#endif
#ifdef MAL_SUPPORT_AUDIO4
        struct
        {
            int fd;
            mal_uint32 fragmentSizeInFrames;
            mal_bool32 breakFromMainLoop;
            void* pIntermediaryBuffer;
        } audio4;
#endif
#ifdef MAL_SUPPORT_OSS
        struct
        {
            int fd;
            mal_uint32 fragmentSizeInFrames;
            mal_bool32 breakFromMainLoop;
            void* pIntermediaryBuffer;
        } oss;
#endif
#ifdef MAL_SUPPORT_OPENSL
        struct
        {
            /*SLObjectItf*/ mal_ptr pOutputMixObj;
            /*SLOutputMixItf*/ mal_ptr pOutputMix;
            /*SLObjectItf*/ mal_ptr pAudioPlayerObj;
            /*SLPlayItf*/ mal_ptr pAudioPlayer;
            /*SLObjectItf*/ mal_ptr pAudioRecorderObj;
            /*SLRecordItf*/ mal_ptr pAudioRecorder;
            /*SLAndroidSimpleBufferQueueItf*/ mal_ptr pBufferQueue;
            mal_uint32 periodSizeInFrames;
            mal_uint32 currentBufferIndex;
            mal_uint8* pBuffer;                 // This is malloc()'d and is used for storing audio data. Typed as mal_uint8 for easy offsetting.
        } opensl;
#endif
#ifdef MAL_SUPPORT_OPENAL
        struct
        {
            /*ALCcontext**/ mal_ptr pContextALC;
            /*ALCdevice**/ mal_ptr pDeviceALC;
            /*ALuint*/ mal_uint32 sourceAL;
            /*ALuint*/ mal_uint32 buffersAL[MAL_MAX_PERIODS_OPENAL];
            /*ALenum*/ mal_uint32 formatAL;
            mal_uint32 subBufferSizeInFrames;   // This is the size of each of the OpenAL buffers (buffersAL).
            mal_uint8* pIntermediaryBuffer;     // This is malloc()'d and is used as the destination for reading from the client. Typed as mal_uint8 for easy offsetting.
            mal_uint32 iNextBuffer;             // The next buffer to unenqueue and then re-enqueue as new data is read.
            mal_bool32 breakFromMainLoop;
        } openal;
#endif
#ifdef MAL_SUPPORT_SDL
        struct
        {
            mal_uint32 deviceID;
        } sdl;
#endif
#ifdef MAL_SUPPORT_NULL
        struct
        {
            mal_timer timer;
            mal_uint32 lastProcessedFrame;      // This is circular.
            mal_bool32 breakFromMainLoop;
            mal_uint8* pBuffer;                 // This is malloc()'d and is used as the destination for reading from the client. Typed as mal_uint8 for easy offsetting.
        } null_device;
#endif
    };
};
#if defined(_MSC_VER)
    #pragma warning(pop)
#endif

// Initializes a context.
//
// The context is used for selecting and initializing the relevant backends.
//
// Note that the location of the context cannot change throughout it's lifetime. Consider allocating
// the mal_context object with malloc() if this is an issue. The reason for this is that a pointer
// to the context is stored in the mal_device structure.
//
// <backends> is used to allow the application to prioritize backends depending on it's specific
// requirements. This can be null in which case it uses the default priority, which is as follows:
//   - WASAPI
//   - DirectSound
//   - WinMM
//   - Core Audio (Apple)
//   - sndio
//   - audio(4)
//   - OSS
//   - PulseAudio
//   - ALSA
//   - JACK
//   - OpenSL|ES
//   - OpenAL
//   - SDL
//   - Null
//
// <pConfig> is used to configure the context. Use the onLog config to set a callback for whenever a
// log message is posted. The priority of the worker thread can be set with the threadPriority config.
//
// It is recommended that only a single context is active at any given time because it's a bulky data
// structure which performs run-time linking for the relevant backends every time it's initialized.
//
// Return Value:
//   MAL_SUCCESS if successful; any other error code otherwise.
//
// Thread Safety: UNSAFE
mal_result mal_context_init(const mal_backend backends[], mal_uint32 backendCount, const mal_context_config* pConfig, mal_context* pContext);

// Uninitializes a context.
//
// Results are undefined if you call this while any device created by this context is still active.
//
// Return Value:
//   MAL_SUCCESS if successful; any other error code otherwise.
//
// Thread Safety: UNSAFE
mal_result mal_context_uninit(mal_context* pContext);

// Enumerates over every device (both playback and capture).
//
// This is a lower-level enumeration function to the easier to use mal_context_get_devices(). Use
// mal_context_enumerate_devices() if you would rather not incur an internal heap allocation, or
// it simply suits your code better.
//
// Do _not_ assume the first enumerated device of a given type is the default device.
//
// Some backends and platforms may only support default playback and capture devices.
//
// Note that this only retrieves the ID and name/description of the device. The reason for only
// retrieving basic information is that it would otherwise require opening the backend device in
// order to probe it for more detailed information which can be inefficient. Consider using
// mal_context_get_device_info() for this, but don't call it from within the enumeration callback.
//
// In general, you should not do anything complicated from within the callback. In particular, do
// not try initializing a device from within the callback.
//
// Consider using mal_context_get_devices() for a simpler and safer API, albeit at the expense of
// an internal heap allocation.
//
// Returning false from the callback will stop enumeration. Returning true will continue enumeration.
//
// Return Value:
//   MAL_SUCCESS if successful; any other error code otherwise.
//
// Thread Safety: SAFE
//   This is guarded using a simple mutex lock.
mal_result mal_context_enumerate_devices(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData);

// Retrieves basic information about every active playback and/or capture device.
//
// You can pass in NULL for the playback or capture lists in which case they'll be ignored.
//
// It is _not_ safe to assume the first device in the list is the default device.
//
// The returned pointers will become invalid upon the next call this this function, or when the
// context is uninitialized. Do not free the returned pointers.
//
// This function follows the same enumeration rules as mal_context_enumerate_devices(). See
// documentation for mal_context_enumerate_devices() for more information.
//
// Return Value:
//   MAL_SUCCESS if successful; any other error code otherwise.
//
// Thread Safety: SAFE
//   Since each call to this function invalidates the pointers from the previous call, you
//   should not be calling this simultaneously across multiple threads. Instead, you need to
//   make a copy of the returned data with your own higher level synchronization.
mal_result mal_context_get_devices(mal_context* pContext, mal_device_info** ppPlaybackDeviceInfos, mal_uint32* pPlaybackDeviceCount, mal_device_info** ppCaptureDeviceInfos, mal_uint32* pCaptureDeviceCount);

// Retrieves information about a device with the given ID.
//
// Do _not_ call this from within the mal_context_enumerate_devices() callback.
//
// It's possible for a device to have different information and capabilities depending on wether or
// not it's opened in shared or exclusive mode. For example, in shared mode, WASAPI always uses
// floating point samples for mixing, but in exclusive mode it can be anything. Therefore, this
// function allows you to specify which share mode you want information for. Note that not all
// backends and devices support shared or exclusive mode, in which case this function will fail
// if the requested share mode is unsupported.
//
// This leaves pDeviceInfo unmodified in the result of an error.
//
// Return Value:
//   MAL_SUCCESS if successful; any other error code otherwise.
//
// Thread Safety: SAFE
//   This is guarded using a simple mutex lock.
mal_result mal_context_get_device_info(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo);

// Initializes a device.
//
// The context can be null in which case it uses the default. This is equivalent to passing in a
// context that was initialized like so:
//
//     mal_context_init(NULL, 0, NULL, &context);
//
// Do not pass in null for the context if you are needing to open multiple devices. You can,
// however, use null when initializing the first device, and then use device.pContext for the
// initialization of other devices.
//
// The device ID (pDeviceID) can be null, in which case the default device is used. Otherwise, you
// can retrieve the ID by calling mal_context_get_devices() and using the ID from the returned data.
// Set pDeviceID to NULL to use the default device. Do _not_ rely on the first device ID returned
// by mal_context_enumerate_devices() or mal_context_get_devices() to be the default device.
//
// The device's configuration is controlled with pConfig. This allows you to configure the sample
// format, channel count, sample rate, etc. Before calling mal_device_init(), you will most likely
// want to initialize a mal_device_config object using mal_device_config_init(),
// mal_device_config_init_playback(), etc. You can also pass in NULL for the device config in
// which case it will use defaults, but will require you to call mal_device_set_recv_callback() or
// mal_device_set_send_callback() before starting the device.
//
// Passing in 0 to any property in pConfig will force the use of a default value. In the case of
// sample format, channel count, sample rate and channel map it will default to the values used by
// the backend's internal device. For the size of the buffer you can set bufferSizeInFrames or
// bufferSizeInMilliseconds (if both are set it will prioritize bufferSizeInFrames). If both are
// set to zero, it will default to MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY or
// MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE, depending on whether or not performanceProfile
// is set to mal_performance_profile_low_latency or mal_performance_profile_conservative.
//
// When sending or receiving data to/from a device, mini_al will internally perform a format
// conversion to convert between the format specified by pConfig and the format used internally by
// the backend. If you pass in NULL for pConfig or 0 for the sample format, channel count,
// sample rate _and_ channel map, data transmission will run on an optimized pass-through fast path.
//
// The <periods> property controls how frequently the background thread is woken to check for more
// data. It's tied to the buffer size, so as an example, if your buffer size is equivalent to 10
// milliseconds and you have 2 periods, the CPU will wake up approximately every 5 milliseconds.
//
// When compiling for UWP you must ensure you call this function on the main UI thread because the
// operating system may need to present the user with a message asking for permissions. Please refer
// to the official documentation for ActivateAudioInterfaceAsync() for more information.
//
// Return Value:
//   MAL_SUCCESS if successful; any other error code otherwise.
//
// Thread Safety: UNSAFE
//   It is not safe to call this function simultaneously for different devices because some backends
//   depend on and mutate global state (such as OpenSL|ES). The same applies to calling this at the
//   same time as mal_device_uninit().
mal_result mal_device_init(mal_context* pContext, mal_device_type type, mal_device_id* pDeviceID, const mal_device_config* pConfig, void* pUserData, mal_device* pDevice);

// Initializes a device without a context, with extra parameters for controlling the configuration
// of the internal self-managed context.
//
// See mal_device_init() and mal_context_init().
mal_result mal_device_init_ex(const mal_backend backends[], mal_uint32 backendCount, const mal_context_config* pContextConfig, mal_device_type type, mal_device_id* pDeviceID, const mal_device_config* pConfig, void* pUserData, mal_device* pDevice);

// Uninitializes a device.
//
// This will explicitly stop the device. You do not need to call mal_device_stop() beforehand, but it's
// harmless if you do.
//
// Return Value:
//   MAL_SUCCESS if successful; any other error code otherwise.
//
// Thread Safety: UNSAFE
//   As soon as this API is called the device should be considered undefined. All bets are off if you
//   try using the device at the same time as uninitializing it.
void mal_device_uninit(mal_device* pDevice);

// Sets the callback to use when the application has received data from the device.
//
// Thread Safety: SAFE
//   This API is implemented as a simple atomic assignment.
//
// DEPRECATED. Set this when the device is initialized with mal_device_init*().
void mal_device_set_recv_callback(mal_device* pDevice, mal_recv_proc proc);

// Sets the callback to use when the application needs to send data to the device for playback.
//
// Note that the implementation of this callback must copy over as many samples as is available. The
// return value specifies how many samples were written to the output buffer. The backend will fill
// any leftover samples with silence.
//
// Thread Safety: SAFE
//   This API is implemented as a simple atomic assignment.
//
// DEPRECATED. Set this when the device is initialized with mal_device_init*().
void mal_device_set_send_callback(mal_device* pDevice, mal_send_proc proc);

// Sets the callback to use when the device has stopped, either explicitly or as a result of an error.
//
// Thread Safety: SAFE
//   This API is implemented as a simple atomic assignment.
void mal_device_set_stop_callback(mal_device* pDevice, mal_stop_proc proc);

// Activates the device. For playback devices this begins playback. For capture devices it begins
// recording.
//
// For a playback device, this will retrieve an initial chunk of audio data from the client before
// returning. The reason for this is to ensure there is valid audio data in the buffer, which needs
// to be done _before_ the device begins playback.
//
// This API waits until the backend device has been started for real by the worker thread. It also
// waits on a mutex for thread-safety.
//
// Return Value:
//   - MAL_SUCCESS if successful; any other error code otherwise.
//   - MAL_INVALID_ARGS
//       One or more of the input arguments is invalid.
//   - MAL_DEVICE_NOT_INITIALIZED
//       The device is not currently or was never initialized.
//   - MAL_DEVICE_BUSY
//       The device is in the process of stopping. This will only happen if mal_device_start() and
//       mal_device_stop() is called simultaneous on separate threads. This will never be returned in
//       single-threaded applications.
//   - MAL_DEVICE_ALREADY_STARTING
//       The device is already in the process of starting. This will never be returned in single-threaded
//       applications.
//   - MAL_DEVICE_ALREADY_STARTED
//       The device is already started.
//   - MAL_FAILED_TO_READ_DATA_FROM_CLIENT
//       Failed to read the initial chunk of audio data from the client. This initial chunk of data is
//       required so that the device has valid audio data as soon as it starts playing. This will never
//       be returned for capture devices.
//   - MAL_FAILED_TO_START_BACKEND_DEVICE
//       There was a backend-specific error starting the device.
//
// Thread Safety: SAFE
mal_result mal_device_start(mal_device* pDevice);

// Puts the device to sleep, but does not uninitialize it. Use mal_device_start() to start it up again.
//
// This API needs to wait on the worker thread to stop the backend device properly before returning. It
// also waits on a mutex for thread-safety. In addition, some backends need to wait for the device to
// finish playback/recording of the current fragment which can take some time (usually proportionate to
// the buffer size that was specified at initialization time).
//
// Return Value:
//   - MAL_SUCCESS if successful; any other error code otherwise.
//   - MAL_INVALID_ARGS
//       One or more of the input arguments is invalid.
//   - MAL_DEVICE_NOT_INITIALIZED
//       The device is not currently or was never initialized.
//   - MAL_DEVICE_BUSY
//       The device is in the process of starting. This will only happen if mal_device_start() and
//       mal_device_stop() is called simultaneous on separate threads. This will never be returned in
//       single-threaded applications.
//   - MAL_DEVICE_ALREADY_STOPPING
//       The device is already in the process of stopping. This will never be returned in single-threaded
//       applications.
//   - MAL_DEVICE_ALREADY_STOPPED
//       The device is already stopped.
//   - MAL_FAILED_TO_STOP_BACKEND_DEVICE
//       There was a backend-specific error stopping the device.
//
// Thread Safety: SAFE
mal_result mal_device_stop(mal_device* pDevice);

// Determines whether or not the device is started.
//
// This is implemented as a simple accessor.
//
// Return Value:
//   True if the device is started, false otherwise.
//
// Thread Safety: SAFE
//   If another thread calls mal_device_start() or mal_device_stop() at this same time as this function
//   is called, there's a very small chance the return value will be out of sync.
mal_bool32 mal_device_is_started(mal_device* pDevice);

// Retrieves the size of the buffer in bytes for the given device.
//
// This API is efficient and is implemented with just a few 32-bit integer multiplications.
//
// Thread Safety: SAFE
//   This is calculated from constant values which are set at initialization time and never change.
mal_uint32 mal_device_get_buffer_size_in_bytes(mal_device* pDevice);


// Helper function for initializing a mal_context_config object.
mal_context_config mal_context_config_init(mal_log_proc onLog);

// Initializes a default device config.
//
// A default configuration will configure the device such that the format, channel count, sample rate and channel map are
// the same as the backend's internal configuration. This means the application loses explicit control of these properties,
// but in return gets an optimized fast path for data transmission since mini_al will be releived of all format conversion
// duties. You will not typically want to use default configurations unless you have some specific low-latency requirements.
//
// mal_device_config_init(), mal_device_config_init_playback(), etc. will allow you to explicitly set the sample format,
// channel count, etc.
mal_device_config mal_device_config_init_default(void);
mal_device_config mal_device_config_init_default_capture(mal_recv_proc onRecvCallback);
mal_device_config mal_device_config_init_default_playback(mal_send_proc onSendCallback);

// Helper function for initializing a mal_device_config object.
//
// This is just a helper API, and as such the returned object can be safely modified as needed.
//
// The default channel mapping is based on the channel count, as per the table below. Note that these
// can be freely changed after this function returns if you are needing something in particular.
//
// |---------------|------------------------------|
// | Channel Count | Mapping                      |
// |---------------|------------------------------|
// | 1 (Mono)      | 0: MAL_CHANNEL_MONO          |
// |---------------|------------------------------|
// | 2 (Stereo)    | 0: MAL_CHANNEL_FRONT_LEFT    |
// |               | 1: MAL_CHANNEL_FRONT_RIGHT   |
// |---------------|------------------------------|
// | 3             | 0: MAL_CHANNEL_FRONT_LEFT    |
// |               | 1: MAL_CHANNEL_FRONT_RIGHT   |
// |               | 2: MAL_CHANNEL_FRONT_CENTER  |
// |---------------|------------------------------|
// | 4 (Surround)  | 0: MAL_CHANNEL_FRONT_LEFT    |
// |               | 1: MAL_CHANNEL_FRONT_RIGHT   |
// |               | 2: MAL_CHANNEL_FRONT_CENTER  |
// |               | 3: MAL_CHANNEL_BACK_CENTER   |
// |---------------|------------------------------|
// | 5             | 0: MAL_CHANNEL_FRONT_LEFT    |
// |               | 1: MAL_CHANNEL_FRONT_RIGHT   |
// |               | 2: MAL_CHANNEL_FRONT_CENTER  |
// |               | 3: MAL_CHANNEL_BACK_LEFT     |
// |               | 4: MAL_CHANNEL_BACK_RIGHT    |
// |---------------|------------------------------|
// | 6 (5.1)       | 0: MAL_CHANNEL_FRONT_LEFT    |
// |               | 1: MAL_CHANNEL_FRONT_RIGHT   |
// |               | 2: MAL_CHANNEL_FRONT_CENTER  |
// |               | 3: MAL_CHANNEL_LFE           |
// |               | 4: MAL_CHANNEL_SIDE_LEFT     |
// |               | 5: MAL_CHANNEL_SIDE_RIGHT    |
// |---------------|------------------------------|
// | 7             | 0: MAL_CHANNEL_FRONT_LEFT    |
// |               | 1: MAL_CHANNEL_FRONT_RIGHT   |
// |               | 2: MAL_CHANNEL_FRONT_CENTER  |
// |               | 3: MAL_CHANNEL_LFE           |
// |               | 4: MAL_CHANNEL_BACK_CENTER   |
// |               | 4: MAL_CHANNEL_SIDE_LEFT     |
// |               | 5: MAL_CHANNEL_SIDE_RIGHT    |
// |---------------|------------------------------|
// | 8 (7.1)       | 0: MAL_CHANNEL_FRONT_LEFT    |
// |               | 1: MAL_CHANNEL_FRONT_RIGHT   |
// |               | 2: MAL_CHANNEL_FRONT_CENTER  |
// |               | 3: MAL_CHANNEL_LFE           |
// |               | 4: MAL_CHANNEL_BACK_LEFT     |
// |               | 5: MAL_CHANNEL_BACK_RIGHT    |
// |               | 6: MAL_CHANNEL_SIDE_LEFT     |
// |               | 7: MAL_CHANNEL_SIDE_RIGHT    |
// |---------------|------------------------------|
// | Other         | All channels set to 0. This  |
// |               | is equivalent to the same    |
// |               | mapping as the device.       |
// |---------------|------------------------------|
//
// Thread Safety: SAFE
//
// Efficiency: HIGH
//   This just returns a stack allocated object and consists of just a few assignments.
mal_device_config mal_device_config_init_ex(mal_format format, mal_uint32 channels, mal_uint32 sampleRate, mal_channel channelMap[MAL_MAX_CHANNELS], mal_recv_proc onRecvCallback, mal_send_proc onSendCallback);

// A simplified version of mal_device_config_init_ex().
static MAL_INLINE mal_device_config mal_device_config_init(mal_format format, mal_uint32 channels, mal_uint32 sampleRate, mal_recv_proc onRecvCallback, mal_send_proc onSendCallback) { return mal_device_config_init_ex(format, channels, sampleRate, NULL, onRecvCallback, onSendCallback); }

// A simplified version of mal_device_config_init() for capture devices.
static MAL_INLINE mal_device_config mal_device_config_init_capture_ex(mal_format format, mal_uint32 channels, mal_uint32 sampleRate, mal_channel channelMap[MAL_MAX_CHANNELS], mal_recv_proc onRecvCallback) { return mal_device_config_init_ex(format, channels, sampleRate, channelMap, onRecvCallback, NULL); }
static MAL_INLINE mal_device_config mal_device_config_init_capture(mal_format format, mal_uint32 channels, mal_uint32 sampleRate, mal_recv_proc onRecvCallback) { return mal_device_config_init_capture_ex(format, channels, sampleRate, NULL, onRecvCallback); }

// A simplified version of mal_device_config_init() for playback devices.
static MAL_INLINE mal_device_config mal_device_config_init_playback_ex(mal_format format, mal_uint32 channels, mal_uint32 sampleRate, mal_channel channelMap[MAL_MAX_CHANNELS], mal_send_proc onSendCallback) { return mal_device_config_init_ex(format, channels, sampleRate, channelMap, NULL, onSendCallback); }
static MAL_INLINE mal_device_config mal_device_config_init_playback(mal_format format, mal_uint32 channels, mal_uint32 sampleRate, mal_send_proc onSendCallback) { return mal_device_config_init_playback_ex(format, channels, sampleRate, NULL, onSendCallback); }



//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Utiltities
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// Creates a mutex.
//
// A mutex must be created from a valid context. A mutex is initially unlocked.
mal_result mal_mutex_init(mal_context* pContext, mal_mutex* pMutex);

// Deletes a mutex.
void mal_mutex_uninit(mal_mutex* pMutex);

// Locks a mutex with an infinite timeout.
void mal_mutex_lock(mal_mutex* pMutex);

// Unlocks a mutex.
void mal_mutex_unlock(mal_mutex* pMutex);


// Retrieves a friendly name for a backend.
const char* mal_get_backend_name(mal_backend backend);

// Adjust buffer size based on a scaling factor.
//
// This just multiplies the base size by the scaling factor, making sure it's a size of at least 1.
mal_uint32 mal_scale_buffer_size(mal_uint32 baseBufferSize, float scale);

// Calculates a buffer size in milliseconds from the specified number of frames and sample rate.
mal_uint32 mal_calculate_buffer_size_in_milliseconds_from_frames(mal_uint32 bufferSizeInFrames, mal_uint32 sampleRate);

// Calculates a buffer size in frames from the specified number of milliseconds and sample rate.
mal_uint32 mal_calculate_buffer_size_in_frames_from_milliseconds(mal_uint32 bufferSizeInMilliseconds, mal_uint32 sampleRate);

// Retrieves the default buffer size in milliseconds based on the specified performance profile.
mal_uint32 mal_get_default_buffer_size_in_milliseconds(mal_performance_profile performanceProfile);

// Calculates a buffer size in frames for the specified performance profile and scale factor.
mal_uint32 mal_get_default_buffer_size_in_frames(mal_performance_profile performanceProfile, mal_uint32 sampleRate);

#endif  // MAL_NO_DEVICE_IO




//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Decoding
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef MAL_NO_DECODING

typedef struct mal_decoder mal_decoder;

typedef enum
{
    mal_seek_origin_start,
    mal_seek_origin_current
} mal_seek_origin;

typedef size_t     (* mal_decoder_read_proc)         (mal_decoder* pDecoder, void* pBufferOut, size_t bytesToRead); // Returns the number of bytes read.
typedef mal_bool32 (* mal_decoder_seek_proc)         (mal_decoder* pDecoder, int byteOffset, mal_seek_origin origin);
typedef mal_result (* mal_decoder_seek_to_frame_proc)(mal_decoder* pDecoder, mal_uint64 frameIndex);
typedef mal_result (* mal_decoder_uninit_proc)       (mal_decoder* pDecoder);

typedef struct
{
    mal_format format;      // Set to 0 or mal_format_unknown to use the stream's internal format.
    mal_uint32 channels;    // Set to 0 to use the stream's internal channels.
    mal_uint32 sampleRate;  // Set to 0 to use the stream's internal sample rate.
    mal_channel channelMap[MAL_MAX_CHANNELS];
    mal_channel_mix_mode channelMixMode;
    mal_dither_mode ditherMode;
    mal_src_algorithm srcAlgorithm;
    union
    {
        mal_src_config_sinc sinc;
    } src;
} mal_decoder_config;

struct mal_decoder
{
    mal_decoder_read_proc onRead;
    mal_decoder_seek_proc onSeek;
    void* pUserData;
    mal_format  internalFormat;
    mal_uint32  internalChannels;
    mal_uint32  internalSampleRate;
    mal_channel internalChannelMap[MAL_MAX_CHANNELS];
    mal_format  outputFormat;
    mal_uint32  outputChannels;
    mal_uint32  outputSampleRate;
    mal_channel outputChannelMap[MAL_MAX_CHANNELS];
    mal_dsp dsp;                // <-- Format conversion is achieved by running frames through this.
    mal_decoder_seek_to_frame_proc onSeekToFrame;
    mal_decoder_uninit_proc onUninit;
    void* pInternalDecoder;     // <-- The drwav/drflac/stb_vorbis/etc. objects.
    struct
    {
        const mal_uint8* pData;
        size_t dataSize;
        size_t currentReadPos;
    } memory;   // Only used for decoders that were opened against a block of memory.
};

mal_decoder_config mal_decoder_config_init(mal_format outputFormat, mal_uint32 outputChannels, mal_uint32 outputSampleRate);

mal_result mal_decoder_init(mal_decoder_read_proc onRead, mal_decoder_seek_proc onSeek, void* pUserData, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_wav(mal_decoder_read_proc onRead, mal_decoder_seek_proc onSeek, void* pUserData, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_flac(mal_decoder_read_proc onRead, mal_decoder_seek_proc onSeek, void* pUserData, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_vorbis(mal_decoder_read_proc onRead, mal_decoder_seek_proc onSeek, void* pUserData, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_mp3(mal_decoder_read_proc onRead, mal_decoder_seek_proc onSeek, void* pUserData, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_raw(mal_decoder_read_proc onRead, mal_decoder_seek_proc onSeek, void* pUserData, const mal_decoder_config* pConfigIn, const mal_decoder_config* pConfigOut, mal_decoder* pDecoder);

mal_result mal_decoder_init_memory(const void* pData, size_t dataSize, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_memory_wav(const void* pData, size_t dataSize, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_memory_flac(const void* pData, size_t dataSize, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_memory_vorbis(const void* pData, size_t dataSize, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_memory_mp3(const void* pData, size_t dataSize, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_memory_raw(const void* pData, size_t dataSize, const mal_decoder_config* pConfigIn, const mal_decoder_config* pConfigOut, mal_decoder* pDecoder);

#ifndef MAL_NO_STDIO
mal_result mal_decoder_init_file(const char* pFilePath, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
mal_result mal_decoder_init_file_wav(const char* pFilePath, const mal_decoder_config* pConfig, mal_decoder* pDecoder);
#endif

mal_result mal_decoder_uninit(mal_decoder* pDecoder);

mal_uint64 mal_decoder_read(mal_decoder* pDecoder, mal_uint64 frameCount, void* pFramesOut);
mal_result mal_decoder_seek_to_frame(mal_decoder* pDecoder, mal_uint64 frameIndex);


// Helper for opening and decoding a file into a heap allocated block of memory. Free the returned pointer with mal_free(). On input,
// pConfig should be set to what you want. On output it will be set to what you got.
#ifndef MAL_NO_STDIO
mal_result mal_decode_file(const char* pFilePath, mal_decoder_config* pConfig, mal_uint64* pFrameCountOut, void** ppDataOut);
#endif
mal_result mal_decode_memory(const void* pData, size_t dataSize, mal_decoder_config* pConfig, mal_uint64* pFrameCountOut, void** ppDataOut);

#endif  // MAL_NO_DECODING


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Generation
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

typedef struct
{
    double amplitude;
    double periodsPerSecond;
    double delta;
    double time;
} mal_sine_wave;

mal_result mal_sine_wave_init(double amplitude, double period, mal_uint32 sampleRate, mal_sine_wave* pSineWave);
mal_uint64 mal_sine_wave_read(mal_sine_wave* pSineWave, mal_uint64 count, float* pSamples);
mal_uint64 mal_sine_wave_read_ex(mal_sine_wave* pSineWave, mal_uint64 frameCount, mal_uint32 channels, mal_stream_layout layout, float** ppFrames);


#ifdef __cplusplus
}
#endif
#endif  //mini_al_h


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// IMPLEMENTATION
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#if defined(MINI_AL_IMPLEMENTATION) || defined(MAL_IMPLEMENTATION)
#include <assert.h>
#include <limits.h> // For INT_MAX
#include <math.h>   // sin(), etc.

#if defined(MAL_DEBUG_OUTPUT)
#include <stdio.h>  // for printf() for debug output
#endif

#ifdef MAL_WIN32
#include <windows.h>
#else
#include <stdlib.h> // For malloc()/free()
#include <string.h> // For memset()
#endif

#if defined(MAL_APPLE) && (__MAC_OS_X_VERSION_MIN_REQUIRED < 101200)
#include <mach/mach_time.h> // For mach_absolute_time()
#endif

#ifdef MAL_POSIX
#include <unistd.h>
#include <dlfcn.h>
#endif

#if !defined(MAL_64BIT) && !defined(MAL_32BIT)
#ifdef _WIN32
#ifdef _WIN64
#define MAL_64BIT
#else
#define MAL_32BIT
#endif
#endif
#endif

#if !defined(MAL_64BIT) && !defined(MAL_32BIT)
#ifdef __GNUC__
#ifdef __LP64__
#define MAL_64BIT
#else
#define MAL_32BIT
#endif
#endif
#endif

#if !defined(MAL_64BIT) && !defined(MAL_32BIT)
#include <stdint.h>
#if INTPTR_MAX == INT64_MAX
#define MAL_64BIT
#else
#define MAL_32BIT
#endif
#endif

// Architecture Detection
#if defined(__x86_64__) || defined(_M_X64)
#define MAL_X64
#elif defined(__i386) || defined(_M_IX86)
#define MAL_X86
#elif defined(__arm__) || defined(_M_ARM)
#define MAL_ARM
#endif

// Cannot currently support AVX-512 if AVX is disabled.
#if !defined(MAL_NO_AVX512) && defined(MAL_NO_AVX2)
#define MAL_NO_AVX512
#endif

// Intrinsics Support
#if defined(MAL_X64) || defined(MAL_X86)
    #if defined(_MSC_VER) && !defined(__clang__)
        // MSVC.
        #if !defined(MAL_NO_SSE2)   // Assume all MSVC compilers support SSE2 intrinsics.
            #define MAL_SUPPORT_SSE2
        #endif
        //#if _MSC_VER >= 1600 && !defined(MAL_NO_AVX)    // 2010
        //    #define MAL_SUPPORT_AVX
        //#endif
        #if _MSC_VER >= 1700 && !defined(MAL_NO_AVX2)   // 2012
            #define MAL_SUPPORT_AVX2
        #endif
        #if _MSC_VER >= 1910 && !defined(MAL_NO_AVX512) // 2017
            #define MAL_SUPPORT_AVX512
        #endif
    #else
        // Assume GNUC-style.
        #if defined(__SSE2__) && !defined(MAL_NO_SSE2)
            #define MAL_SUPPORT_SSE2
        #endif
        //#if defined(__AVX__) && !defined(MAL_NO_AVX)
        //    #define MAL_SUPPORT_AVX
        //#endif
        #if defined(__AVX2__) && !defined(MAL_NO_AVX2)
            #define MAL_SUPPORT_AVX2
        #endif
        #if defined(__AVX512F__) && !defined(MAL_NO_AVX512)
            #define MAL_SUPPORT_AVX512
        #endif
    #endif

    // If at this point we still haven't determined compiler support for the intrinsics just fall back to __has_include.
    #if !defined(__GNUC__) && !defined(__clang__) && defined(__has_include)
        #if !defined(MAL_SUPPORT_SSE2)   && !defined(MAL_NO_SSE2)   && __has_include(<emmintrin.h>)
            #define MAL_SUPPORT_SSE2
        #endif
        //#if !defined(MAL_SUPPORT_AVX)    && !defined(MAL_NO_AVX)    && __has_include(<immintrin.h>)
        //    #define MAL_SUPPORT_AVX
        //#endif
        #if !defined(MAL_SUPPORT_AVX2)   && !defined(MAL_NO_AVX2)   && __has_include(<immintrin.h>)
            #define MAL_SUPPORT_AVX2
        #endif
        #if !defined(MAL_SUPPORT_AVX512) && !defined(MAL_NO_AVX512) && __has_include(<zmmintrin.h>)
            #define MAL_SUPPORT_AVX512
        #endif
    #endif

    #if defined(MAL_SUPPORT_AVX512)
        #include <immintrin.h>  // Not a mistake. Intentionally including <immintrin.h> instead of <zmmintrin.h> because otherwise the compiler will complain.
    #elif defined(MAL_SUPPORT_AVX2) || defined(MAL_SUPPORT_AVX)
        #include <immintrin.h>
    #elif defined(MAL_SUPPORT_SSE2)
        #include <emmintrin.h>
    #endif
#endif

#if defined(MAL_ARM)
    #if !defined(MAL_NO_NEON) && (defined(__ARM_NEON) || defined(__aarch64__) || defined(_M_ARM64))
        #define MAL_SUPPORT_NEON
    #endif

    // Fall back to looking for the #include file.
    #if !defined(__GNUC__) && !defined(__clang__) && defined(__has_include)
        #if !defined(MAL_SUPPORT_NEON) && !defined(MAL_NO_NEON) && __has_include(<arm_neon.h>)
            #define MAL_SUPPORT_NEON
        #endif
    #endif

    #if defined(MAL_SUPPORT_NEON)
        #include <arm_neon.h>
    #endif
#endif


#if defined(MAL_X64) || defined(MAL_X86)
    #if defined(_MSC_VER) && !defined(__clang__)
        #if _MSC_VER >= 1400
            #include <intrin.h>
            static MAL_INLINE void mal_cpuid(int info[4], int fid)
            {
                __cpuid(info, fid);
            }
        #else
            #define MAL_NO_CPUID
        #endif

        #if _MSC_VER >= 1600
            static MAL_INLINE unsigned __int64 mal_xgetbv(int reg)
            {
                return _xgetbv(reg);
            }
        #else
            #define MAL_NO_XGETBV
        #endif
    #elif (defined(__GNUC__) || defined(__clang__)) && !defined(MAL_ANDROID)
        static MAL_INLINE void mal_cpuid(int info[4], int fid)
        {
            // It looks like the -fPIC option uses the ebx register which GCC complains about. We can work around this by just using a different register, the
            // specific register of which I'm letting the compiler decide on. The "k" prefix is used to specify a 32-bit register. The {...} syntax is for
            // supporting different assembly dialects.
            //
            // What's basically happening is that we're saving and restoring the ebx register manually.
            #if defined(DRFLAC_X86) && defined(__PIC__)
                __asm__ __volatile__ (
                    "xchg{l} {%%}ebx, %k1;"
                    "cpuid;"
                    "xchg{l} {%%}ebx, %k1;"
                    : "=a"(info[0]), "=&r"(info[1]), "=c"(info[2]), "=d"(info[3]) : "a"(fid), "c"(0)
                );
            #else
                __asm__ __volatile__ (
                    "cpuid" : "=a"(info[0]), "=b"(info[1]), "=c"(info[2]), "=d"(info[3]) : "a"(fid), "c"(0)
                );
            #endif
        }

        static MAL_INLINE unsigned long long mal_xgetbv(int reg)
        {
            unsigned int hi;
            unsigned int lo;

            __asm__ __volatile__ (
                "xgetbv" : "=a"(lo), "=d"(hi) : "c"(reg)
            );

            return ((unsigned long long)hi << 32ULL) | (unsigned long long)lo;
        }
    #else
        #define MAL_NO_CPUID
        #define MAL_NO_XGETBV
    #endif
#else
    #define MAL_NO_CPUID
    #define MAL_NO_XGETBV
#endif

static MAL_INLINE mal_bool32 mal_has_sse2()
{
#if defined(MAL_SUPPORT_SSE2)
    #if (defined(MAL_X64) || defined(MAL_X86)) && !defined(MAL_NO_SSE2)
        #if defined(MAL_X64)
            return MAL_TRUE;    // 64-bit targets always support SSE2.
        #elif (defined(_M_IX86_FP) && _M_IX86_FP == 2) || defined(__SSE2__)
            return MAL_TRUE;    // If the compiler is allowed to freely generate SSE2 code we can assume support.
        #else
            #if defined(MAL_NO_CPUID)
                return MAL_FALSE;
            #else
                int info[4];
                mal_cpuid(info, 1);
                return (info[3] & (1 << 26)) != 0;
            #endif
        #endif
    #else
        return MAL_FALSE;       // SSE2 is only supported on x86 and x64 architectures.
    #endif
#else
    return MAL_FALSE;           // No compiler support.
#endif
}

#if 0
static MAL_INLINE mal_bool32 mal_has_avx()
{
#if defined(MAL_SUPPORT_AVX)
    #if (defined(MAL_X64) || defined(MAL_X86)) && !defined(MAL_NO_AVX)
        #if defined(_AVX_) || defined(__AVX__)
            return MAL_TRUE;    // If the compiler is allowed to freely generate AVX code we can assume support.
        #else
            // AVX requires both CPU and OS support.
            #if defined(MAL_NO_CPUID) || defined(MAL_NO_XGETBV)
                return MAL_FALSE;
            #else
                int info[4];
                mal_cpuid(info, 1);
                if (((info[2] & (1 << 27)) != 0) && ((info[2] & (1 << 28)) != 0)) {
                    mal_uint64 xrc = mal_xgetbv(0);
                    if ((xrc & 0x06) == 0x06) {
                        return MAL_TRUE;
                    } else {
                        return MAL_FALSE;
                    }
                } else {
                    return MAL_FALSE;
                }
            #endif
        #endif
    #else
        return MAL_FALSE;       // AVX is only supported on x86 and x64 architectures.
    #endif
#else
    return MAL_FALSE;           // No compiler support.
#endif
}
#endif

static MAL_INLINE mal_bool32 mal_has_avx2()
{
#if defined(MAL_SUPPORT_AVX2)
    #if (defined(MAL_X64) || defined(MAL_X86)) && !defined(MAL_NO_AVX2)
        #if defined(_AVX2_) || defined(__AVX2__)
            return MAL_TRUE;    // If the compiler is allowed to freely generate AVX2 code we can assume support.
        #else
            // AVX2 requires both CPU and OS support.
            #if defined(MAL_NO_CPUID) || defined(MAL_NO_XGETBV)
                return MAL_FALSE;
            #else
                int info1[4];
                int info7[4];
                mal_cpuid(info1, 1);
                mal_cpuid(info7, 7);
                if (((info1[2] & (1 << 27)) != 0) && ((info7[1] & (1 << 5)) != 0)) {
                    mal_uint64 xrc = mal_xgetbv(0);
                    if ((xrc & 0x06) == 0x06) {
                        return MAL_TRUE;
                    } else {
                        return MAL_FALSE;
                    }
                } else {
                    return MAL_FALSE;
                }
            #endif
        #endif
    #else
        return MAL_FALSE;       // AVX2 is only supported on x86 and x64 architectures.
    #endif
#else
    return MAL_FALSE;           // No compiler support.
#endif
}

static MAL_INLINE mal_bool32 mal_has_avx512f()
{
#if defined(MAL_SUPPORT_AVX512)
    #if (defined(MAL_X64) || defined(MAL_X86)) && !defined(MAL_NO_AVX512)
        #if defined(__AVX512F__)
            return MAL_TRUE;    // If the compiler is allowed to freely generate AVX-512F code we can assume support.
        #else
            // AVX-512 requires both CPU and OS support.
            #if defined(MAL_NO_CPUID) || defined(MAL_NO_XGETBV)
                return MAL_FALSE;
            #else
                int info1[4];
                int info7[4];
                mal_cpuid(info1, 1);
                mal_cpuid(info7, 7);
                if (((info1[2] & (1 << 27)) != 0) && ((info7[1] & (1 << 16)) != 0)) {
                    mal_uint64 xrc = mal_xgetbv(0);
                    if ((xrc & 0xE6) == 0xE6) {
                        return MAL_TRUE;
                    } else {
                        return MAL_FALSE;
                    }
                } else {
                    return MAL_FALSE;
                }
            #endif
        #endif
    #else
        return MAL_FALSE;       // AVX-512F is only supported on x86 and x64 architectures.
    #endif
#else
    return MAL_FALSE;           // No compiler support.
#endif
}

static MAL_INLINE mal_bool32 mal_has_neon()
{
#if defined(MAL_SUPPORT_NEON)
    #if defined(MAL_ARM) && !defined(MAL_NO_NEON)
        #if (defined(__ARM_NEON) || defined(__aarch64__) || defined(_M_ARM64))
            return MAL_TRUE;    // If the compiler is allowed to freely generate NEON code we can assume support.
        #else
            // TODO: Runtime check.
            return MAL_FALSE;
        #endif
    #else
        return MAL_FALSE;       // NEON is only supported on ARM architectures.
    #endif
#else
    return MAL_FALSE;           // No compiler support.
#endif
}


static MAL_INLINE mal_bool32 mal_is_little_endian()
{
#if defined(MAL_X86) || defined(MAL_X64)
    return MAL_TRUE;
#else
    int n = 1;
    return (*(char*)&n) == 1;
#endif
}

static MAL_INLINE mal_bool32 mal_is_big_endian()
{
    return !mal_is_little_endian();
}


#ifndef MAL_COINIT_VALUE
#define MAL_COINIT_VALUE    0   /* 0 = COINIT_MULTITHREADED*/
#endif



#ifndef MAL_PI
#define MAL_PI      3.14159265358979323846264f
#endif
#ifndef MAL_PI_D
#define MAL_PI_D    3.14159265358979323846264
#endif
#ifndef MAL_TAU
#define MAL_TAU     6.28318530717958647693f
#endif
#ifndef MAL_TAU_D
#define MAL_TAU_D   6.28318530717958647693
#endif


// The default format when mal_format_unknown (0) is requested when initializing a device.
#ifndef MAL_DEFAULT_FORMAT
#define MAL_DEFAULT_FORMAT                                  mal_format_f32
#endif

// The default channel count to use when 0 is used when initializing a device.
#ifndef MAL_DEFAULT_CHANNELS
#define MAL_DEFAULT_CHANNELS                                2
#endif

// The default sample rate to use when 0 is used when initializing a device.
#ifndef MAL_DEFAULT_SAMPLE_RATE
#define MAL_DEFAULT_SAMPLE_RATE                             48000
#endif

// Default periods when none is specified in mal_device_init(). More periods means more work on the CPU.
#ifndef MAL_DEFAULT_PERIODS
#define MAL_DEFAULT_PERIODS                                 2
#endif

// The base buffer size in milliseconds for low latency mode.
#ifndef MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY
#define MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY    25
#endif

// The base buffer size in milliseconds for conservative mode.
#ifndef MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE
#define MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE   150
#endif


// Standard sample rates, in order of priority.
mal_uint32 g_malStandardSampleRatePriorities[] = {
    MAL_SAMPLE_RATE_48000,  // Most common
    MAL_SAMPLE_RATE_44100,

    MAL_SAMPLE_RATE_32000,  // Lows
    MAL_SAMPLE_RATE_24000,
    MAL_SAMPLE_RATE_22050,

    MAL_SAMPLE_RATE_88200,  // Highs
    MAL_SAMPLE_RATE_96000,
    MAL_SAMPLE_RATE_176400,
    MAL_SAMPLE_RATE_192000,

    MAL_SAMPLE_RATE_16000,  // Extreme lows
    MAL_SAMPLE_RATE_11025,
    MAL_SAMPLE_RATE_8000,

    MAL_SAMPLE_RATE_352800, // Extreme highs
    MAL_SAMPLE_RATE_384000
};

mal_format g_malFormatPriorities[] = {
    mal_format_s16,         // Most common
    mal_format_f32,
    
    //mal_format_s24_32,    // Clean alignment
    mal_format_s32,
    
    mal_format_s24,         // Unclean alignment
    
    mal_format_u8           // Low quality
};



///////////////////////////////////////////////////////////////////////////////
//
// Standard Library Stuff
//
///////////////////////////////////////////////////////////////////////////////
#ifndef MAL_MALLOC
#ifdef MAL_WIN32
#define MAL_MALLOC(sz) HeapAlloc(GetProcessHeap(), 0, (sz))
#else
#define MAL_MALLOC(sz) malloc((sz))
#endif
#endif

#ifndef MAL_REALLOC
#ifdef MAL_WIN32
#define MAL_REALLOC(p, sz) (((sz) > 0) ? ((p) ? HeapReAlloc(GetProcessHeap(), 0, (p), (sz)) : HeapAlloc(GetProcessHeap(), 0, (sz))) : ((VOID*)(size_t)(HeapFree(GetProcessHeap(), 0, (p)) & 0)))
#else
#define MAL_REALLOC(p, sz) realloc((p), (sz))
#endif
#endif

#ifndef MAL_FREE
#ifdef MAL_WIN32
#define MAL_FREE(p) HeapFree(GetProcessHeap(), 0, (p))
#else
#define MAL_FREE(p) free((p))
#endif
#endif

#ifndef MAL_ZERO_MEMORY
#ifdef MAL_WIN32
#define MAL_ZERO_MEMORY(p, sz) ZeroMemory((p), (sz))
#else
#define MAL_ZERO_MEMORY(p, sz) memset((p), 0, (sz))
#endif
#endif

#ifndef MAL_COPY_MEMORY
#ifdef MAL_WIN32
#define MAL_COPY_MEMORY(dst, src, sz) CopyMemory((dst), (src), (sz))
#else
#define MAL_COPY_MEMORY(dst, src, sz) memcpy((dst), (src), (sz))
#endif
#endif

#ifndef MAL_ASSERT
#ifdef MAL_WIN32
#define MAL_ASSERT(condition) assert(condition)
#else
#define MAL_ASSERT(condition) assert(condition)
#endif
#endif

#define mal_zero_memory MAL_ZERO_MEMORY
#define mal_copy_memory MAL_COPY_MEMORY
#define mal_assert      MAL_ASSERT

#define mal_zero_object(p)          mal_zero_memory((p), sizeof(*(p)))
#define mal_countof(x)              (sizeof(x) / sizeof(x[0]))
#define mal_max(x, y)               (((x) > (y)) ? (x) : (y))
#define mal_min(x, y)               (((x) < (y)) ? (x) : (y))
#define mal_clamp(x, lo, hi)        (mal_max(lo, mal_min(x, hi)))
#define mal_offset_ptr(p, offset)   (((mal_uint8*)(p)) + (offset))

#define mal_buffer_frame_capacity(buffer, channels, format) (sizeof(buffer) / mal_get_bytes_per_sample(format) / (channels))


// Return Values:
//   0:  Success
//   22: EINVAL
//   34: ERANGE
//
// Not using symbolic constants for errors because I want to avoid #including errno.h
int mal_strcpy_s(char* dst, size_t dstSizeInBytes, const char* src)
{
    if (dst == 0) {
        return 22;
    }
    if (dstSizeInBytes == 0) {
        return 34;
    }
    if (src == 0) {
        dst[0] = '\0';
        return 22;
    }

    size_t i;
    for (i = 0; i < dstSizeInBytes && src[i] != '\0'; ++i) {
        dst[i] = src[i];
    }

    if (i < dstSizeInBytes) {
        dst[i] = '\0';
        return 0;
    }

    dst[0] = '\0';
    return 34;
}

int mal_strncpy_s(char* dst, size_t dstSizeInBytes, const char* src, size_t count)
{
    if (dst == 0) {
        return 22;
    }
    if (dstSizeInBytes == 0) {
        return 34;
    }
    if (src == 0) {
        dst[0] = '\0';
        return 22;
    }

    size_t maxcount = count;
    if (count == ((size_t)-1) || count >= dstSizeInBytes) {        // -1 = _TRUNCATE
        maxcount = dstSizeInBytes - 1;
    }

    size_t i;
    for (i = 0; i < maxcount && src[i] != '\0'; ++i) {
        dst[i] = src[i];
    }

    if (src[i] == '\0' || i == count || count == ((size_t)-1)) {
        dst[i] = '\0';
        return 0;
    }

    dst[0] = '\0';
    return 34;
}

int mal_strcat_s(char* dst, size_t dstSizeInBytes, const char* src)
{
    if (dst == 0) {
        return 22;
    }
    if (dstSizeInBytes == 0) {
        return 34;
    }
    if (src == 0) {
        dst[0] = '\0';
        return 22;
    }

    char* dstorig = dst;

    while (dstSizeInBytes > 0 && dst[0] != '\0') {
        dst += 1;
        dstSizeInBytes -= 1;
    }

    if (dstSizeInBytes == 0) {
        return 22;  // Unterminated.
    }


    while (dstSizeInBytes > 0 && src[0] != '\0') {
        *dst++ = *src++;
        dstSizeInBytes -= 1;
    }

    if (dstSizeInBytes > 0) {
        dst[0] = '\0';
    } else {
        dstorig[0] = '\0';
        return 34;
    }

    return 0;
}

int mal_itoa_s(int value, char* dst, size_t dstSizeInBytes, int radix)
{
    if (dst == NULL || dstSizeInBytes == 0) {
        return 22;
    }
    if (radix < 2 || radix > 36) {
        dst[0] = '\0';
        return 22;
    }

    int sign = (value < 0 && radix == 10) ? -1 : 1;     // The negative sign is only used when the base is 10.

    unsigned int valueU;
    if (value < 0) {
        valueU = -value;
    } else {
        valueU = value;
    }

    char* dstEnd = dst;
    do
    {
        int remainder = valueU % radix;
        if (remainder > 9) {
            *dstEnd = (char)((remainder - 10) + 'a');
        } else {
            *dstEnd = (char)(remainder + '0');
        }

        dstEnd += 1;
        dstSizeInBytes -= 1;
        valueU /= radix;
    } while (dstSizeInBytes > 0 && valueU > 0);

    if (dstSizeInBytes == 0) {
        dst[0] = '\0';
        return 22;  // Ran out of room in the output buffer.
    }

    if (sign < 0) {
        *dstEnd++ = '-';
        dstSizeInBytes -= 1;
    }

    if (dstSizeInBytes == 0) {
        dst[0] = '\0';
        return 22;  // Ran out of room in the output buffer.
    }

    *dstEnd = '\0';


    // At this point the string will be reversed.
    dstEnd -= 1;
    while (dst < dstEnd) {
        char temp = *dst;
        *dst = *dstEnd;
        *dstEnd = temp;

        dst += 1;
        dstEnd -= 1;
    }

    return 0;
}

int mal_strcmp(const char* str1, const char* str2)
{
    if (str1 == str2) return  0;

    // These checks differ from the standard implementation. It's not important, but I prefer
    // it just for sanity.
    if (str1 == NULL) return -1;
    if (str2 == NULL) return  1;

    for (;;) {
        if (str1[0] == '\0') {
            break;
        }
        if (str1[0] != str2[0]) {
            break;
        }

        str1 += 1;
        str2 += 1;
    }

    return ((unsigned char*)str1)[0] - ((unsigned char*)str2)[0];
}


// Thanks to good old Bit Twiddling Hacks for this one: http://graphics.stanford.edu/~seander/bithacks.html#RoundUpPowerOf2
static MAL_INLINE unsigned int mal_next_power_of_2(unsigned int x)
{
    x--;
    x |= x >> 1;
    x |= x >> 2;
    x |= x >> 4;
    x |= x >> 8;
    x |= x >> 16;
    x++;

    return x;
}

static MAL_INLINE unsigned int mal_prev_power_of_2(unsigned int x)
{
    return mal_next_power_of_2(x) >> 1;
}

static MAL_INLINE unsigned int mal_round_to_power_of_2(unsigned int x)
{
    unsigned int prev = mal_prev_power_of_2(x);
    unsigned int next = mal_next_power_of_2(x);
    if ((next - x) > (x - prev)) {
        return prev;
    } else {
        return next;
    }
}

static MAL_INLINE unsigned int mal_count_set_bits(unsigned int x)
{
    unsigned int count = 0;
    while (x != 0) {
        if (x & 1) {
            count += 1;
        }
        
        x = x >> 1;
    }
    
    return count;
}



// Clamps an f32 sample to -1..1
static MAL_INLINE float mal_clip_f32(float x)
{
    if (x < -1) return -1;
    if (x > +1) return +1;
    return x;
}

static MAL_INLINE float mal_mix_f32(float x, float y, float a)
{
    return x*(1-a) + y*a;
}
static MAL_INLINE float mal_mix_f32_fast(float x, float y, float a)
{
    float r0 = (y - x);
    float r1 = r0*a;
    return x + r1;
    //return x + (y - x)*a;
}

#if defined(MAL_SUPPORT_SSE2)
static MAL_INLINE __m128 mal_mix_f32_fast__sse2(__m128 x, __m128 y, __m128 a)
{
    return _mm_add_ps(x, _mm_mul_ps(_mm_sub_ps(y, x), a));
}
#endif
#if defined(MAL_SUPPORT_AVX2)
static MAL_INLINE __m256 mal_mix_f32_fast__avx2(__m256 x, __m256 y, __m256 a)
{
    return _mm256_add_ps(x, _mm256_mul_ps(_mm256_sub_ps(y, x), a));
}
#endif
#if defined(MAL_SUPPORT_AVX512)
static MAL_INLINE __m512 mal_mix_f32_fast__avx512(__m512 x, __m512 y, __m512 a)
{
    return _mm512_add_ps(x, _mm512_mul_ps(_mm512_sub_ps(y, x), a));
}
#endif
#if defined(MAL_SUPPORT_NEON)
static MAL_INLINE float32x4_t mal_mix_f32_fast__neon(float32x4_t x, float32x4_t y, float32x4_t a)
{
    return vaddq_f32(x, vmulq_f32(vsubq_f32(y, x), a));
}
#endif


static MAL_INLINE double mal_mix_f64(double x, double y, double a)
{
    return x*(1-a) + y*a;
}
static MAL_INLINE double mal_mix_f64_fast(double x, double y, double a)
{
    return x + (y - x)*a;
}

static MAL_INLINE float mal_scale_to_range_f32(float x, float lo, float hi)
{
    return lo + x*(hi-lo);
}



// Random Number Generation
//
// mini_al uses the LCG random number generation algorithm. This is good enough for audio.
//
// Note that mini_al's LCG implementation uses global state which is _not_ thread-local. When this is called across
// multiple threads, results will be unpredictable. However, it won't crash and results will still be random enough
// for mini_al's purposes.
#define MAL_LCG_M   4294967296
#define MAL_LCG_A   1103515245
#define MAL_LCG_C   12345
static mal_int32 g_malLCG;

void mal_seed(mal_int32 seed)
{
    g_malLCG = seed;
}

mal_int32 mal_rand_s32()
{
    mal_int32 lcg = g_malLCG;
    mal_int32 r = (MAL_LCG_A * lcg + MAL_LCG_C) % MAL_LCG_M;
    g_malLCG = r;
    return r;
}

double mal_rand_f64()
{
    return (mal_rand_s32() + 0x80000000) / (double)0x7FFFFFFF;
}

float mal_rand_f32()
{
    return (float)mal_rand_f64();
}

static MAL_INLINE float mal_rand_range_f32(float lo, float hi)
{
    return mal_scale_to_range_f32(mal_rand_f32(), lo, hi);
}

static MAL_INLINE mal_int32 mal_rand_range_s32(mal_int32 lo, mal_int32 hi)
{
    double x = mal_rand_f64();
    return lo + (mal_int32)(x*(hi-lo));
}


static MAL_INLINE float mal_dither_f32_rectangle(float ditherMin, float ditherMax)
{
    return mal_rand_range_f32(ditherMin, ditherMax);
}

static MAL_INLINE float mal_dither_f32_triangle(float ditherMin, float ditherMax)
{
    float a = mal_rand_range_f32(ditherMin, 0);
    float b = mal_rand_range_f32(0, ditherMax);
    return a + b;
}

static MAL_INLINE float mal_dither_f32(mal_dither_mode ditherMode, float ditherMin, float ditherMax)
{
    if (ditherMode == mal_dither_mode_rectangle) {
        return mal_dither_f32_rectangle(ditherMin, ditherMax);
    }
    if (ditherMode == mal_dither_mode_triangle) {
        return mal_dither_f32_triangle(ditherMin, ditherMax);
    }

    return 0;
}

static MAL_INLINE mal_int32 mal_dither_s32(mal_dither_mode ditherMode, mal_int32 ditherMin, mal_int32 ditherMax)
{
    if (ditherMode == mal_dither_mode_rectangle) {
        mal_int32 a = mal_rand_range_s32(ditherMin, ditherMax);
        return a;
    }
    if (ditherMode == mal_dither_mode_triangle) {
        mal_int32 a = mal_rand_range_s32(ditherMin, 0);
        mal_int32 b = mal_rand_range_s32(0, ditherMax);
        return a + b;
    }

    return 0;
}


// Splits a buffer into parts of equal length and of the given alignment. The returned size of the split buffers will be a
// multiple of the alignment. The alignment must be a power of 2.
void mal_split_buffer(void* pBuffer, size_t bufferSize, size_t splitCount, size_t alignment, void** ppBuffersOut, size_t* pSplitSizeOut)
{
    if (pSplitSizeOut) {
        *pSplitSizeOut = 0;
    }

    if (pBuffer == NULL || bufferSize == 0 || splitCount == 0) {
        return;
    }

    if (alignment == 0) {
        alignment = 1;
    }

    mal_uintptr pBufferUnaligned = (mal_uintptr)pBuffer;
    mal_uintptr pBufferAligned = (pBufferUnaligned + (alignment-1)) & ~(alignment-1);
    size_t unalignedBytes = (size_t)(pBufferAligned - pBufferUnaligned);

    size_t splitSize = 0;
    if (bufferSize >= unalignedBytes) {
        splitSize = (bufferSize - unalignedBytes) / splitCount;
        splitSize = splitSize & ~(alignment-1);
    }

    if (ppBuffersOut != NULL) {
        for (size_t i = 0; i < splitCount; ++i) {
            ppBuffersOut[i] = (mal_uint8*)(pBufferAligned + (splitSize*i));
        }
    }

    if (pSplitSizeOut) {
        *pSplitSizeOut = splitSize;
    }
}


///////////////////////////////////////////////////////////////////////////////
//
// Atomics
//
///////////////////////////////////////////////////////////////////////////////
#if defined(_WIN32) && !defined(__GNUC__)
#define mal_memory_barrier()            MemoryBarrier()
#define mal_atomic_exchange_32(a, b)    InterlockedExchange((LONG*)a, (LONG)b)
#define mal_atomic_exchange_64(a, b)    InterlockedExchange64((LONGLONG*)a, (LONGLONG)b)
#define mal_atomic_increment_32(a)      InterlockedIncrement((LONG*)a)
#define mal_atomic_decrement_32(a)      InterlockedDecrement((LONG*)a)
#else
#define mal_memory_barrier()            __sync_synchronize()
#define mal_atomic_exchange_32(a, b)    (void)__sync_lock_test_and_set(a, b); __sync_synchronize()
#define mal_atomic_exchange_64(a, b)    (void)__sync_lock_test_and_set(a, b); __sync_synchronize()
#define mal_atomic_increment_32(a)      __sync_add_and_fetch(a, 1)
#define mal_atomic_decrement_32(a)      __sync_sub_and_fetch(a, 1)
#endif

#ifdef MAL_64BIT
#define mal_atomic_exchange_ptr mal_atomic_exchange_64
#endif
#ifdef MAL_32BIT
#define mal_atomic_exchange_ptr mal_atomic_exchange_32
#endif


mal_uint32 mal_get_standard_sample_rate_priority_index(mal_uint32 sampleRate)   // Lower = higher priority
{
    for (mal_uint32 i = 0; i < mal_countof(g_malStandardSampleRatePriorities); ++i) {
        if (g_malStandardSampleRatePriorities[i] == sampleRate) {
            return i;
        }
    }

    return (mal_uint32)-1;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// DEVICE I/O
// ==========
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef MAL_NO_DEVICE_IO
// Unfortunately using runtime linking for pthreads causes problems. This has occurred for me when testing on FreeBSD. When
// using runtime linking, deadlocks can occur (for me it happens when loading data from fread()). It turns out that doing
// compile-time linking fixes this. I'm not sure why this happens, but the safest way I can think of to fix this is to simply
// disable runtime linking by default. To enable runtime linking, #define this before the implementation of this file. I am
// not officially supporting this, but I'm leaving it here in case it's useful for somebody, somewhere.
//#define MAL_USE_RUNTIME_LINKING_FOR_PTHREAD

// Disable run-time linking on certain backends.
#ifndef MAL_NO_RUNTIME_LINKING
    #if defined(MAL_ANDROID) || defined(MAL_EMSCRIPTEN)
        #define MAL_NO_RUNTIME_LINKING
    #endif
#endif

// Check if we have the necessary development packages for each backend at the top so we can use this to determine whether or not
// certain unused functions and variables can be excluded from the build to avoid warnings.
#ifdef MAL_ENABLE_WASAPI
    #define MAL_HAS_WASAPI      // Every compiler should support WASAPI
#endif
#ifdef MAL_ENABLE_DSOUND
    #define MAL_HAS_DSOUND      // Every compiler should support DirectSound.
#endif
#ifdef MAL_ENABLE_WINMM
    #define MAL_HAS_WINMM       // Every compiler I'm aware of supports WinMM.
#endif
#ifdef MAL_ENABLE_ALSA
    #define MAL_HAS_ALSA
    #ifdef MAL_NO_RUNTIME_LINKING
        #ifdef __has_include
            #if !__has_include(<alsa/asoundlib.h>)
                #undef MAL_HAS_ALSA
            #endif
        #endif
    #endif
#endif
#ifdef MAL_ENABLE_PULSEAUDIO
    #define MAL_HAS_PULSEAUDIO  // Development packages are unnecessary for PulseAudio.
    #ifdef MAL_NO_RUNTIME_LINKING
        #ifdef __has_include
            #if !__has_include(<pulse/pulseaudio.h>)
                #undef MAL_HAS_PULSEAUDIO
            #endif
        #endif
    #endif
#endif
#ifdef MAL_ENABLE_JACK
    #define MAL_HAS_JACK
    #ifdef MAL_NO_RUNTIME_LINKING
        #ifdef __has_include
            #if !__has_include(<jack/jack.h>)
                #undef MAL_HAS_JACK
            #endif
        #endif
    #endif
#endif
#ifdef MAL_ENABLE_COREAUDIO
    #define MAL_HAS_COREAUDIO
#endif
#ifdef MAL_ENABLE_SNDIO
    #define MAL_HAS_SNDIO
#endif
#ifdef MAL_ENABLE_AUDIO4
    #define MAL_HAS_AUDIO4      // When enabled, always assume audio(4) is available.
#endif
#ifdef MAL_ENABLE_OSS
    #define MAL_HAS_OSS         // OSS is the only supported backend for Unix and BSD, so it must be present else this library is useless.
#endif
#ifdef MAL_ENABLE_OPENSL
    #define MAL_HAS_OPENSL      // OpenSL is the only supported backend for Android. It must be present.
#endif
#ifdef MAL_ENABLE_OPENAL
    #define MAL_HAS_OPENAL
    #ifdef MAL_NO_RUNTIME_LINKING
        #ifdef __has_include
            #if !__has_include(<AL/al.h>)
                #undef MAL_HAS_OPENAL
            #endif
        #endif
    #endif
#endif
#ifdef MAL_ENABLE_SDL
    #define MAL_HAS_SDL

    // SDL headers are necessary if using compile-time linking.
    #ifdef MAL_NO_RUNTIME_LINKING
        #ifdef __has_include
            #ifdef MAL_EMSCRIPTEN
                #if !__has_include(<SDL/SDL_audio.h>)
                    #undef MAL_HAS_SDL
                #endif
            #else
                #if !__has_include(<SDL2/SDL_audio.h>)
                    #undef MAL_HAS_SDL
                #endif
            #endif
        #endif
    #endif
#endif
#ifdef MAL_ENABLE_NULL
    #define MAL_HAS_NULL    // Everything supports the null backend.
#endif

const mal_backend g_malDefaultBackends[] = {
    mal_backend_wasapi,
    mal_backend_dsound,
    mal_backend_winmm,
    mal_backend_coreaudio,
    mal_backend_sndio,
    mal_backend_audio4,
    mal_backend_oss,
    mal_backend_pulseaudio,
    mal_backend_alsa,
    mal_backend_jack,
    mal_backend_opensl,
    mal_backend_openal,
    mal_backend_sdl,
    mal_backend_null
};

const char* mal_get_backend_name(mal_backend backend)
{
    switch (backend)
    {
        case mal_backend_null:       return "Null";
        case mal_backend_wasapi:     return "WASAPI";
        case mal_backend_dsound:     return "DirectSound";
        case mal_backend_winmm:      return "WinMM";
        case mal_backend_alsa:       return "ALSA";
        case mal_backend_pulseaudio: return "PulseAudio";
        case mal_backend_jack:       return "JACK";
        case mal_backend_coreaudio:  return "Core Audio";
        case mal_backend_sndio:      return "sndio";
        case mal_backend_audio4:     return "audio(4)";
        case mal_backend_oss:        return "OSS";
        case mal_backend_opensl:     return "OpenSL|ES";
        case mal_backend_openal:     return "OpenAL";
        case mal_backend_sdl:        return "SDL";
        default:                     return "Unknown";
    }
}



#ifdef MAL_WIN32
    #define MAL_THREADCALL WINAPI
    typedef unsigned long mal_thread_result;
#else
    #define MAL_THREADCALL
    typedef void* mal_thread_result;
#endif
typedef mal_thread_result (MAL_THREADCALL * mal_thread_entry_proc)(void* pData);

#ifdef MAL_WIN32
typedef HRESULT (WINAPI * MAL_PFN_CoInitializeEx)(LPVOID pvReserved, DWORD  dwCoInit);
typedef void    (WINAPI * MAL_PFN_CoUninitialize)();
typedef HRESULT (WINAPI * MAL_PFN_CoCreateInstance)(REFCLSID rclsid, LPUNKNOWN pUnkOuter, DWORD dwClsContext, REFIID riid, LPVOID *ppv);
typedef void    (WINAPI * MAL_PFN_CoTaskMemFree)(LPVOID pv);
typedef HRESULT (WINAPI * MAL_PFN_PropVariantClear)(PROPVARIANT *pvar);
typedef int     (WINAPI * MAL_PFN_StringFromGUID2)(const GUID* const rguid, LPOLESTR lpsz, int cchMax);

typedef HWND (WINAPI * MAL_PFN_GetForegroundWindow)();
typedef HWND (WINAPI * MAL_PFN_GetDesktopWindow)();

// Microsoft documents these APIs as returning LSTATUS, but the Win32 API shipping with some compilers do not define it. It's just a LONG.
typedef LONG (WINAPI * MAL_PFN_RegOpenKeyExA)(HKEY hKey, LPCSTR lpSubKey, DWORD ulOptions, REGSAM samDesired, PHKEY phkResult);
typedef LONG (WINAPI * MAL_PFN_RegCloseKey)(HKEY hKey);
typedef LONG (WINAPI * MAL_PFN_RegQueryValueExA)(HKEY hKey, LPCSTR lpValueName, LPDWORD lpReserved, LPDWORD lpType, LPBYTE lpData, LPDWORD lpcbData);
#endif


#define MAL_STATE_UNINITIALIZED     0
#define MAL_STATE_STOPPED           1   // The device's default state after initialization.
#define MAL_STATE_STARTED           2   // The worker thread is in it's main loop waiting for the driver to request or deliver audio data.
#define MAL_STATE_STARTING          3   // Transitioning from a stopped state to started.
#define MAL_STATE_STOPPING          4   // Transitioning from a started state to stopped.

#define MAL_DEFAULT_PLAYBACK_DEVICE_NAME    "Default Playback Device"
#define MAL_DEFAULT_CAPTURE_DEVICE_NAME     "Default Capture Device"


///////////////////////////////////////////////////////////////////////////////
//
// Timing
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_WIN32
LARGE_INTEGER g_mal_TimerFrequency = {{0}};
void mal_timer_init(mal_timer* pTimer)
{
    if (g_mal_TimerFrequency.QuadPart == 0) {
        QueryPerformanceFrequency(&g_mal_TimerFrequency);
    }

    LARGE_INTEGER counter;
    QueryPerformanceCounter(&counter);
    pTimer->counter = counter.QuadPart;
}

double mal_timer_get_time_in_seconds(mal_timer* pTimer)
{
    LARGE_INTEGER counter;
    if (!QueryPerformanceCounter(&counter)) {
        return 0;
    }

    return (double)(counter.QuadPart - pTimer->counter) / g_mal_TimerFrequency.QuadPart;
}
#elif defined(MAL_APPLE) && (__MAC_OS_X_VERSION_MIN_REQUIRED < 101200)
mal_uint64 g_mal_TimerFrequency = 0;
void mal_timer_init(mal_timer* pTimer)
{
    mach_timebase_info_data_t baseTime;
    mach_timebase_info(&baseTime);
    g_mal_TimerFrequency = (baseTime.denom * 1e9) / baseTime.numer;

    pTimer->counter = mach_absolute_time();
}

double mal_timer_get_time_in_seconds(mal_timer* pTimer)
{
    mal_uint64 newTimeCounter = mach_absolute_time();
    mal_uint64 oldTimeCounter = pTimer->counter;

    return (newTimeCounter - oldTimeCounter) / g_mal_TimerFrequency;
}
#else
#if defined(CLOCK_MONOTONIC)
    #define MAL_CLOCK_ID CLOCK_MONOTONIC
#else
    #define MAL_CLOCK_ID CLOCK_REALTIME
#endif

void mal_timer_init(mal_timer* pTimer)
{
    struct timespec newTime;
    clock_gettime(MAL_CLOCK_ID, &newTime);

    pTimer->counter = (newTime.tv_sec * 1000000000) + newTime.tv_nsec;
}

double mal_timer_get_time_in_seconds(mal_timer* pTimer)
{
    struct timespec newTime;
    clock_gettime(MAL_CLOCK_ID, &newTime);

    mal_uint64 newTimeCounter = (newTime.tv_sec * 1000000000) + newTime.tv_nsec;
    mal_uint64 oldTimeCounter = pTimer->counter;

    return (newTimeCounter - oldTimeCounter) / 1000000000.0;
}
#endif


///////////////////////////////////////////////////////////////////////////////
//
// Dynamic Linking
//
///////////////////////////////////////////////////////////////////////////////
mal_handle mal_dlopen(const char* filename)
{
#ifdef _WIN32
#ifdef MAL_WIN32_DESKTOP
    return (mal_handle)LoadLibraryA(filename);
#else
    // *sigh* It appears there is no ANSI version of LoadPackagedLibrary()...
    WCHAR filenameW[4096];
    if (MultiByteToWideChar(CP_UTF8, 0, filename, -1, filenameW, sizeof(filenameW)) == 0) {
        return NULL;
    }

    return (mal_handle)LoadPackagedLibrary(filenameW, 0);
#endif
#else
    return (mal_handle)dlopen(filename, RTLD_NOW);
#endif
}

void mal_dlclose(mal_handle handle)
{
#ifdef _WIN32
    FreeLibrary((HMODULE)handle);
#else
    dlclose((void*)handle);
#endif
}

mal_proc mal_dlsym(mal_handle handle, const char* symbol)
{
#ifdef _WIN32
    return (mal_proc)GetProcAddress((HMODULE)handle, symbol);
#else
    return (mal_proc)dlsym((void*)handle, symbol);
#endif
}


///////////////////////////////////////////////////////////////////////////////
//
// Threading
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_WIN32
int mal_thread_priority_to_win32(mal_thread_priority priority)
{
    switch (priority) {
        case mal_thread_priority_idle:     return THREAD_PRIORITY_IDLE;
        case mal_thread_priority_lowest:   return THREAD_PRIORITY_LOWEST;
        case mal_thread_priority_low:      return THREAD_PRIORITY_BELOW_NORMAL;
        case mal_thread_priority_normal:   return THREAD_PRIORITY_NORMAL;
        case mal_thread_priority_high:     return THREAD_PRIORITY_ABOVE_NORMAL;
        case mal_thread_priority_highest:  return THREAD_PRIORITY_HIGHEST;
        case mal_thread_priority_realtime: return THREAD_PRIORITY_TIME_CRITICAL;
        default: return mal_thread_priority_normal;
    }
}

mal_result mal_thread_create__win32(mal_context* pContext, mal_thread* pThread, mal_thread_entry_proc entryProc, void* pData)
{
    pThread->win32.hThread = CreateThread(NULL, 0, entryProc, pData, 0, NULL);
    if (pThread->win32.hThread == NULL) {
        return MAL_FAILED_TO_CREATE_THREAD;
    }

    SetThreadPriority((HANDLE)pThread->win32.hThread, mal_thread_priority_to_win32(pContext->config.threadPriority));

    return MAL_SUCCESS;
}

void mal_thread_wait__win32(mal_thread* pThread)
{
    WaitForSingleObject(pThread->win32.hThread, INFINITE);
}

void mal_sleep__win32(mal_uint32 milliseconds)
{
    Sleep((DWORD)milliseconds);
}


mal_result mal_mutex_init__win32(mal_context* pContext, mal_mutex* pMutex)
{
    (void)pContext;

    pMutex->win32.hMutex = CreateEventA(NULL, FALSE, TRUE, NULL);
    if (pMutex->win32.hMutex == NULL) {
        return MAL_FAILED_TO_CREATE_MUTEX;
    }

    return MAL_SUCCESS;
}

void mal_mutex_uninit__win32(mal_mutex* pMutex)
{
    CloseHandle(pMutex->win32.hMutex);
}

void mal_mutex_lock__win32(mal_mutex* pMutex)
{
    WaitForSingleObject(pMutex->win32.hMutex, INFINITE);
}

void mal_mutex_unlock__win32(mal_mutex* pMutex)
{
    SetEvent(pMutex->win32.hMutex);
}


mal_result mal_event_init__win32(mal_context* pContext, mal_event* pEvent)
{
    (void)pContext;

    pEvent->win32.hEvent = CreateEventW(NULL, FALSE, FALSE, NULL);
    if (pEvent->win32.hEvent == NULL) {
        return MAL_FAILED_TO_CREATE_EVENT;
    }

    return MAL_SUCCESS;
}

void mal_event_uninit__win32(mal_event* pEvent)
{
    CloseHandle(pEvent->win32.hEvent);
}

mal_bool32 mal_event_wait__win32(mal_event* pEvent)
{
    return WaitForSingleObject(pEvent->win32.hEvent, INFINITE) == WAIT_OBJECT_0;
}

mal_bool32 mal_event_signal__win32(mal_event* pEvent)
{
    return SetEvent(pEvent->win32.hEvent);
}
#endif


#ifdef MAL_POSIX
#include <sched.h>

typedef int (* mal_pthread_create_proc)(pthread_t *thread, const pthread_attr_t *attr, void *(*start_routine) (void *), void *arg);
typedef int (* mal_pthread_join_proc)(pthread_t thread, void **retval);
typedef int (* mal_pthread_mutex_init_proc)(pthread_mutex_t *__mutex, const pthread_mutexattr_t *__mutexattr);
typedef int (* mal_pthread_mutex_destroy_proc)(pthread_mutex_t *__mutex);
typedef int (* mal_pthread_mutex_lock_proc)(pthread_mutex_t *__mutex);
typedef int (* mal_pthread_mutex_unlock_proc)(pthread_mutex_t *__mutex);
typedef int (* mal_pthread_cond_init_proc)(pthread_cond_t *__restrict __cond, const pthread_condattr_t *__restrict __cond_attr);
typedef int (* mal_pthread_cond_destroy_proc)(pthread_cond_t *__cond);
typedef int (* mal_pthread_cond_signal_proc)(pthread_cond_t *__cond);
typedef int (* mal_pthread_cond_wait_proc)(pthread_cond_t *__restrict __cond, pthread_mutex_t *__restrict __mutex);
typedef int (* mal_pthread_attr_init_proc)(pthread_attr_t *attr);
typedef int (* mal_pthread_attr_destroy_proc)(pthread_attr_t *attr);
typedef int (* mal_pthread_attr_setschedpolicy_proc)(pthread_attr_t *attr, int policy);
typedef int (* mal_pthread_attr_getschedparam_proc)(const pthread_attr_t *attr, struct sched_param *param);
typedef int (* mal_pthread_attr_setschedparam_proc)(pthread_attr_t *attr, const struct sched_param *param);

mal_bool32 mal_thread_create__posix(mal_context* pContext, mal_thread* pThread, mal_thread_entry_proc entryProc, void* pData)
{
    pthread_attr_t* pAttr = NULL;

#if !defined(__EMSCRIPTEN__)
    // Try setting the thread priority. It's not critical if anything fails here.
    pthread_attr_t attr;
    if (((mal_pthread_attr_init_proc)pContext->posix.pthread_attr_init)(&attr) == 0) {
        int scheduler = -1;
        if (pContext->config.threadPriority == mal_thread_priority_idle) {
#ifdef SCHED_IDLE
            if (((mal_pthread_attr_setschedpolicy_proc)pContext->posix.pthread_attr_setschedpolicy)(&attr, SCHED_IDLE) == 0) {
                scheduler = SCHED_IDLE;
            }
#endif
        } else if (pContext->config.threadPriority == mal_thread_priority_realtime) {
#ifdef SCHED_FIFO
            if (((mal_pthread_attr_setschedpolicy_proc)pContext->posix.pthread_attr_setschedpolicy)(&attr, SCHED_FIFO) == 0) {
                scheduler = SCHED_FIFO;
            }
#endif
#ifdef MAL_LINUX
        } else {
            scheduler = sched_getscheduler(0);
#endif
        }

        if (scheduler != -1) {
            int priorityMin = sched_get_priority_min(scheduler);
            int priorityMax = sched_get_priority_max(scheduler);
            int priorityStep = (priorityMax - priorityMin) / 7;  // 7 = number of priorities supported by mini_al.

            struct sched_param sched;
            if (((mal_pthread_attr_getschedparam_proc)pContext->posix.pthread_attr_getschedparam)(&attr, &sched) == 0) {
                if (pContext->config.threadPriority == mal_thread_priority_idle) {
                    sched.sched_priority = priorityMin;
                } else if (pContext->config.threadPriority == mal_thread_priority_realtime) {
                    sched.sched_priority = priorityMax;
                } else {
                    sched.sched_priority += ((int)pContext->config.threadPriority + 5) * priorityStep;  // +5 because the lowest priority is -5.
                    if (sched.sched_priority < priorityMin) {
                        sched.sched_priority = priorityMin;
                    }
                    if (sched.sched_priority > priorityMax) {
                        sched.sched_priority = priorityMax;
                    }
                }

                if (((mal_pthread_attr_setschedparam_proc)pContext->posix.pthread_attr_setschedparam)(&attr, &sched) == 0) {
                    pAttr = &attr;
                }
            }
        }

        ((mal_pthread_attr_destroy_proc)pContext->posix.pthread_attr_destroy)(&attr);
    }
#endif

    int result = ((mal_pthread_create_proc)pContext->posix.pthread_create)(&pThread->posix.thread, pAttr, entryProc, pData);
    if (result != 0) {
        return MAL_FAILED_TO_CREATE_THREAD;
    }

    return MAL_SUCCESS;
}

void mal_thread_wait__posix(mal_thread* pThread)
{
    ((mal_pthread_join_proc)pThread->pContext->posix.pthread_join)(pThread->posix.thread, NULL);
}

void mal_sleep__posix(mal_uint32 milliseconds)
{
    usleep(milliseconds * 1000);    // <-- usleep is in microseconds.
}


mal_result mal_mutex_init__posix(mal_context* pContext, mal_mutex* pMutex)
{
    int result = ((mal_pthread_mutex_init_proc)pContext->posix.pthread_mutex_init)(&pMutex->posix.mutex, NULL);
    if (result != 0) {
        return MAL_FAILED_TO_CREATE_MUTEX;
    }

    return MAL_SUCCESS;
}

void mal_mutex_uninit__posix(mal_mutex* pMutex)
{
    ((mal_pthread_mutex_destroy_proc)pMutex->pContext->posix.pthread_mutex_destroy)(&pMutex->posix.mutex);
}

void mal_mutex_lock__posix(mal_mutex* pMutex)
{
    ((mal_pthread_mutex_lock_proc)pMutex->pContext->posix.pthread_mutex_lock)(&pMutex->posix.mutex);
}

void mal_mutex_unlock__posix(mal_mutex* pMutex)
{
    ((mal_pthread_mutex_unlock_proc)pMutex->pContext->posix.pthread_mutex_unlock)(&pMutex->posix.mutex);
}


mal_result mal_event_init__posix(mal_context* pContext, mal_event* pEvent)
{
    if (((mal_pthread_mutex_init_proc)pContext->posix.pthread_mutex_init)(&pEvent->posix.mutex, NULL) != 0) {
        return MAL_FAILED_TO_CREATE_MUTEX;
    }

    if (((mal_pthread_cond_init_proc)pContext->posix.pthread_cond_init)(&pEvent->posix.condition, NULL) != 0) {
        return MAL_FAILED_TO_CREATE_EVENT;
    }

    pEvent->posix.value = 0;
    return MAL_SUCCESS;
}

void mal_event_uninit__posix(mal_event* pEvent)
{
    ((mal_pthread_cond_destroy_proc)pEvent->pContext->posix.pthread_cond_destroy)(&pEvent->posix.condition);
    ((mal_pthread_mutex_destroy_proc)pEvent->pContext->posix.pthread_mutex_destroy)(&pEvent->posix.mutex);
}

mal_bool32 mal_event_wait__posix(mal_event* pEvent)
{
    ((mal_pthread_mutex_lock_proc)pEvent->pContext->posix.pthread_mutex_lock)(&pEvent->posix.mutex);
    {
        while (pEvent->posix.value == 0) {
            ((mal_pthread_cond_wait_proc)pEvent->pContext->posix.pthread_cond_wait)(&pEvent->posix.condition, &pEvent->posix.mutex);
        }

        pEvent->posix.value = 0;  // Auto-reset.
    }
    ((mal_pthread_mutex_unlock_proc)pEvent->pContext->posix.pthread_mutex_unlock)(&pEvent->posix.mutex);

    return MAL_TRUE;
}

mal_bool32 mal_event_signal__posix(mal_event* pEvent)
{
    ((mal_pthread_mutex_lock_proc)pEvent->pContext->posix.pthread_mutex_lock)(&pEvent->posix.mutex);
    {
        pEvent->posix.value = 1;
        ((mal_pthread_cond_signal_proc)pEvent->pContext->posix.pthread_cond_signal)(&pEvent->posix.condition);
    }
    ((mal_pthread_mutex_unlock_proc)pEvent->pContext->posix.pthread_mutex_unlock)(&pEvent->posix.mutex);

    return MAL_TRUE;
}
#endif

mal_result mal_thread_create(mal_context* pContext, mal_thread* pThread, mal_thread_entry_proc entryProc, void* pData)
{
    if (pContext == NULL || pThread == NULL || entryProc == NULL) return MAL_FALSE;

    pThread->pContext = pContext;

#ifdef MAL_WIN32
    return mal_thread_create__win32(pContext, pThread, entryProc, pData);
#endif
#ifdef MAL_POSIX
    return mal_thread_create__posix(pContext, pThread, entryProc, pData);
#endif
}

void mal_thread_wait(mal_thread* pThread)
{
    if (pThread == NULL) return;

#ifdef MAL_WIN32
    mal_thread_wait__win32(pThread);
#endif
#ifdef MAL_POSIX
    mal_thread_wait__posix(pThread);
#endif
}

void mal_sleep(mal_uint32 milliseconds)
{
#ifdef MAL_WIN32
    mal_sleep__win32(milliseconds);
#endif
#ifdef MAL_POSIX
    mal_sleep__posix(milliseconds);
#endif
}


mal_result mal_mutex_init(mal_context* pContext, mal_mutex* pMutex)
{
    if (pContext == NULL || pMutex == NULL) {
        return MAL_INVALID_ARGS;
    }

    pMutex->pContext = pContext;

#ifdef MAL_WIN32
    return mal_mutex_init__win32(pContext, pMutex);
#endif
#ifdef MAL_POSIX
    return mal_mutex_init__posix(pContext, pMutex);
#endif
}

void mal_mutex_uninit(mal_mutex* pMutex)
{
    if (pMutex == NULL || pMutex->pContext == NULL) return;

#ifdef MAL_WIN32
    mal_mutex_uninit__win32(pMutex);
#endif
#ifdef MAL_POSIX
    mal_mutex_uninit__posix(pMutex);
#endif
}

void mal_mutex_lock(mal_mutex* pMutex)
{
    if (pMutex == NULL || pMutex->pContext == NULL) return;

#ifdef MAL_WIN32
    mal_mutex_lock__win32(pMutex);
#endif
#ifdef MAL_POSIX
    mal_mutex_lock__posix(pMutex);
#endif
}

void mal_mutex_unlock(mal_mutex* pMutex)
{
    if (pMutex == NULL || pMutex->pContext == NULL) return;

#ifdef MAL_WIN32
    mal_mutex_unlock__win32(pMutex);
#endif
#ifdef MAL_POSIX
    mal_mutex_unlock__posix(pMutex);
#endif
}


mal_result mal_event_init(mal_context* pContext, mal_event* pEvent)
{
    if (pContext == NULL || pEvent == NULL) return MAL_FALSE;

    pEvent->pContext = pContext;

#ifdef MAL_WIN32
    return mal_event_init__win32(pContext, pEvent);
#endif
#ifdef MAL_POSIX
    return mal_event_init__posix(pContext, pEvent);
#endif
}

void mal_event_uninit(mal_event* pEvent)
{
    if (pEvent == NULL || pEvent->pContext == NULL) return;

#ifdef MAL_WIN32
    mal_event_uninit__win32(pEvent);
#endif
#ifdef MAL_POSIX
    mal_event_uninit__posix(pEvent);
#endif
}

mal_bool32 mal_event_wait(mal_event* pEvent)
{
    if (pEvent == NULL || pEvent->pContext == NULL) return MAL_FALSE;

#ifdef MAL_WIN32
    return mal_event_wait__win32(pEvent);
#endif
#ifdef MAL_POSIX
    return mal_event_wait__posix(pEvent);
#endif
}

mal_bool32 mal_event_signal(mal_event* pEvent)
{
    if (pEvent == NULL || pEvent->pContext == NULL) return MAL_FALSE;

#ifdef MAL_WIN32
    return mal_event_signal__win32(pEvent);
#endif
#ifdef MAL_POSIX
    return mal_event_signal__posix(pEvent);
#endif
}


mal_uint32 mal_get_best_sample_rate_within_range(mal_uint32 sampleRateMin, mal_uint32 sampleRateMax)
{
    // Normalize the range in case we were given something stupid.
    if (sampleRateMin < MAL_MIN_SAMPLE_RATE) {
        sampleRateMin = MAL_MIN_SAMPLE_RATE;
    }
    if (sampleRateMax > MAL_MAX_SAMPLE_RATE) {
        sampleRateMax = MAL_MAX_SAMPLE_RATE;
    }
    if (sampleRateMin > sampleRateMax) {
        sampleRateMin = sampleRateMax;
    }

    if (sampleRateMin == sampleRateMax) {
        return sampleRateMax;
    } else {
        for (size_t iStandardRate = 0; iStandardRate < mal_countof(g_malStandardSampleRatePriorities); ++iStandardRate) {
            mal_uint32 standardRate = g_malStandardSampleRatePriorities[iStandardRate];
            if (standardRate >= sampleRateMin && standardRate <= sampleRateMax) {
                return standardRate;
            }
        }
    }

    // Should never get here.
    mal_assert(MAL_FALSE);
    return 0;
}

mal_uint32 mal_get_closest_standard_sample_rate(mal_uint32 sampleRateIn)
{
    mal_uint32 closestRate = 0;
    mal_uint32 closestDiff = 0xFFFFFFFF;

    for (size_t iStandardRate = 0; iStandardRate < mal_countof(g_malStandardSampleRatePriorities); ++iStandardRate) {
        mal_uint32 standardRate = g_malStandardSampleRatePriorities[iStandardRate];

        mal_uint32 diff;
        if (sampleRateIn > standardRate) {
            diff = sampleRateIn - standardRate;
        } else {
            diff = standardRate - sampleRateIn;
        }

        if (diff == 0) {
            return standardRate;    // The input sample rate is a standard rate.
        }

        if (closestDiff > diff) {
            closestDiff = diff;
            closestRate = standardRate;
        }
    }

    return closestRate;
}


mal_uint32 mal_scale_buffer_size(mal_uint32 baseBufferSize, float scale)
{
    return mal_max(1, (mal_uint32)(baseBufferSize*scale));
}

mal_uint32 mal_calculate_buffer_size_in_milliseconds_from_frames(mal_uint32 bufferSizeInFrames, mal_uint32 sampleRate)
{
    return bufferSizeInFrames / (sampleRate/1000);
}

mal_uint32 mal_calculate_buffer_size_in_frames_from_milliseconds(mal_uint32 bufferSizeInMilliseconds, mal_uint32 sampleRate)
{
    return bufferSizeInMilliseconds * (sampleRate/1000); 
}

mal_uint32 mal_get_default_buffer_size_in_milliseconds(mal_performance_profile performanceProfile)
{
    if (performanceProfile == mal_performance_profile_low_latency) {
        return MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY;
    } else {
        return MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE;
    }
}

mal_uint32 mal_get_default_buffer_size_in_frames(mal_performance_profile performanceProfile, mal_uint32 sampleRate)
{
    mal_uint32 bufferSizeInMilliseconds = mal_get_default_buffer_size_in_milliseconds(performanceProfile);
    if (bufferSizeInMilliseconds == 0) {
        bufferSizeInMilliseconds = 1;
    }

    mal_uint32 sampleRateMS = (sampleRate/1000);
    if (sampleRateMS == 0) {
        sampleRateMS = 1;
    }

    return bufferSizeInMilliseconds * sampleRateMS;
}


const char* mal_log_level_to_string(mal_uint32 logLevel)
{
    switch (logLevel)
    {
        case MAL_LOG_LEVEL_VERBOSE: return "";
        case MAL_LOG_LEVEL_INFO:    return "INFO";
        case MAL_LOG_LEVEL_WARNING: return "WARNING";
        case MAL_LOG_LEVEL_ERROR:   return "ERROR";
        default:                    return "ERROR";
    }
}

// Posts a log message.
void mal_log(mal_context* pContext, mal_device* pDevice, mal_uint32 logLevel, const char* message)
{
    if (pContext == NULL) return;
    
#if defined(MAL_LOG_LEVEL)
    if (logLevel <= MAL_LOG_LEVEL) {
    #if defined(MAL_DEBUG_OUTPUT)
        if (logLevel <= MAL_LOG_LEVEL) {
            printf("%s: %s", mal_log_level_to_string(logLevel), message);
        }
    #endif
    
        mal_log_proc onLog = pContext->config.onLog;
        if (onLog) {
            onLog(pContext, pDevice, message);
        }
    }
#endif
}

// Posts an error. Throw a breakpoint in here if you're needing to debug. The return value is always "resultCode".
mal_result mal_context_post_error(mal_context* pContext, mal_device* pDevice, mal_uint32 logLevel, const char* message, mal_result resultCode)
{
    // Derive the context from the device if necessary.
    if (pContext == NULL) {
        if (pDevice != NULL) {
            pContext = pDevice->pContext;
        }
    }

    mal_log(pContext, pDevice, logLevel, message);
    return resultCode;
}

mal_result mal_post_error(mal_device* pDevice, mal_uint32 logLevel, const char* message, mal_result resultCode)
{
    return mal_context_post_error(NULL, pDevice, logLevel, message, resultCode);
}


// The callback for reading from the client -> DSP -> device.
mal_uint32 mal_device__on_read_from_client(mal_dsp* pDSP, mal_uint32 frameCount, void* pFramesOut, void* pUserData)
{
    (void)pDSP;

    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    mal_send_proc onSend = pDevice->onSend;
    if (onSend) {
        return onSend(pDevice, frameCount, pFramesOut);
    }

    return 0;
}

// The callback for reading from the device -> DSP -> client.
mal_uint32 mal_device__on_read_from_device(mal_dsp* pDSP, mal_uint32 frameCount, void* pFramesOut, void* pUserData)
{
    (void)pDSP;

    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    if (pDevice->_dspFrameCount == 0) {
        return 0;   // Nothing left.
    }

    mal_uint32 framesToRead = frameCount;
    if (framesToRead > pDevice->_dspFrameCount) {
        framesToRead = pDevice->_dspFrameCount;
    }

    mal_uint32 bytesToRead = framesToRead * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat);
    mal_copy_memory(pFramesOut, pDevice->_dspFrames, bytesToRead);
    pDevice->_dspFrameCount -= framesToRead;
    pDevice->_dspFrames += bytesToRead;

    return framesToRead;
}

// A helper function for reading sample data from the client. Returns the number of samples read from the client. Remaining samples
// are filled with silence.
static MAL_INLINE mal_uint32 mal_device__read_frames_from_client(mal_device* pDevice, mal_uint32 frameCount, void* pSamples)
{
    mal_assert(pDevice != NULL);
    mal_assert(frameCount > 0);
    mal_assert(pSamples != NULL);

    mal_uint32 framesRead     = (mal_uint32)mal_dsp_read(&pDevice->dsp, frameCount, pSamples, pDevice->dsp.pUserData);
    mal_uint32 samplesRead    = framesRead * pDevice->internalChannels;
    mal_uint32 sampleSize     = mal_get_bytes_per_sample(pDevice->internalFormat);
    mal_uint32 consumedBytes  = samplesRead*sampleSize;
    mal_uint32 remainingBytes = ((frameCount * pDevice->internalChannels) - samplesRead)*sampleSize;
    mal_zero_memory((mal_uint8*)pSamples + consumedBytes, remainingBytes);

    return samplesRead;
}

// A helper for sending sample data to the client.
static MAL_INLINE void mal_device__send_frames_to_client(mal_device* pDevice, mal_uint32 frameCount, const void* pSamples)
{
    mal_assert(pDevice != NULL);
    mal_assert(frameCount > 0);
    mal_assert(pSamples != NULL);

    mal_recv_proc onRecv = pDevice->onRecv;
    if (onRecv) {
        pDevice->_dspFrameCount = frameCount;
        pDevice->_dspFrames = (const mal_uint8*)pSamples;

        mal_uint8 chunkBuffer[4096];
        mal_uint32 chunkFrameCount = sizeof(chunkBuffer) / mal_get_bytes_per_frame(pDevice->format, pDevice->channels);

        for (;;) {
            mal_uint32 framesJustRead = (mal_uint32)mal_dsp_read(&pDevice->dsp, chunkFrameCount, chunkBuffer, pDevice->dsp.pUserData);
            if (framesJustRead == 0) {
                break;
            }

            onRecv(pDevice, framesJustRead, chunkBuffer);

            if (framesJustRead < chunkFrameCount) {
                break;
            }
        }
    }
}

// A helper for changing the state of the device.
static MAL_INLINE void mal_device__set_state(mal_device* pDevice, mal_uint32 newState)
{
    mal_atomic_exchange_32(&pDevice->state, newState);
}

// A helper for getting the state of the device.
static MAL_INLINE mal_uint32 mal_device__get_state(mal_device* pDevice)
{
    return pDevice->state;
}


#ifdef MAL_WIN32
    GUID MAL_GUID_KSDATAFORMAT_SUBTYPE_PCM        = {0x00000001, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};
    GUID MAL_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT = {0x00000003, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};
    //GUID MAL_GUID_KSDATAFORMAT_SUBTYPE_ALAW       = {0x00000006, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};
    //GUID MAL_GUID_KSDATAFORMAT_SUBTYPE_MULAW      = {0x00000007, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};
#endif


mal_bool32 mal_context__device_id_equal(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);

    if (pID0 == pID1) return MAL_TRUE;

    if ((pID0 == NULL && pID1 != NULL) ||
        (pID0 != NULL && pID1 == NULL)) {
        return MAL_FALSE;
    }

    if (pContext->onDeviceIDEqual) {
        return pContext->onDeviceIDEqual(pContext, pID0, pID1);
    }

    return MAL_FALSE;
}


typedef struct
{
    mal_device_type deviceType;
    const mal_device_id* pDeviceID;
    char* pName;
    size_t nameBufferSize;
    mal_bool32 foundDevice;
} mal_context__try_get_device_name_by_id__enum_callback_data;

mal_bool32 mal_context__try_get_device_name_by_id__enum_callback(mal_context* pContext, mal_device_type deviceType, const mal_device_info* pDeviceInfo, void* pUserData)
{
    mal_context__try_get_device_name_by_id__enum_callback_data* pData = (mal_context__try_get_device_name_by_id__enum_callback_data*)pUserData;
    mal_assert(pData != NULL);

    if (pData->deviceType == deviceType) {
        if (pContext->onDeviceIDEqual(pContext, pData->pDeviceID, &pDeviceInfo->id)) {
            mal_strncpy_s(pData->pName, pData->nameBufferSize, pDeviceInfo->name, (size_t)-1);
            pData->foundDevice = MAL_TRUE;
        }
    }

    return !pData->foundDevice;
}

// Generic function for retrieving the name of a device by it's ID.
//
// This function simply enumerates every device and then retrieves the name of the first device that has the same ID.
mal_result mal_context__try_get_device_name_by_id(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, char* pName, size_t nameBufferSize)
{
    mal_assert(pContext != NULL);
    mal_assert(pName != NULL);

    if (pDeviceID == NULL) {
        return MAL_NO_DEVICE;
    }

    mal_context__try_get_device_name_by_id__enum_callback_data data;
    data.deviceType = type;
    data.pDeviceID = pDeviceID;
    data.pName = pName;
    data.nameBufferSize = nameBufferSize;
    data.foundDevice = MAL_FALSE;
    mal_result result = mal_context_enumerate_devices(pContext, mal_context__try_get_device_name_by_id__enum_callback, &data);
    if (result != MAL_SUCCESS) {
        return result;
    }

    if (!data.foundDevice) {
        return MAL_NO_DEVICE;
    } else {
        return MAL_SUCCESS;
    }
}


mal_uint32 mal_get_format_priority_index(mal_format format) // Lower = better.
{
    for (mal_uint32 i = 0; i < mal_countof(g_malFormatPriorities); ++i) {
        if (g_malFormatPriorities[i] == format) {
            return i;
        }
    }

    // Getting here means the format could not be found or is equal to mal_format_unknown.
    return (mal_uint32)-1;
}

void mal_device__post_init_setup(mal_device* pDevice);

///////////////////////////////////////////////////////////////////////////////
//
// Null Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_NULL

mal_bool32 mal_context_is_device_id_equal__null(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return pID0->nullbackend == pID1->nullbackend;
}

mal_result mal_context_enumerate_devices__null(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    mal_bool32 cbResult = MAL_TRUE;

    // Playback.
    if (cbResult) {
        mal_device_info deviceInfo;
        mal_zero_object(&deviceInfo);
        mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), "NULL Playback Device", (size_t)-1);
        cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
    }

    // Capture.
    if (cbResult) {
        mal_device_info deviceInfo;
        mal_zero_object(&deviceInfo);
        mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), "NULL Capture Device", (size_t)-1);
        cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
    }

    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info__null(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);

    (void)pContext;
    (void)shareMode;

    if (pDeviceID != NULL && pDeviceID->nullbackend != 0) {
        return MAL_NO_DEVICE;   // Don't know the device.
    }

    // Name / Description
    if (deviceType == mal_device_type_playback) {
        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), "NULL Playback Device", (size_t)-1);
    } else {
        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), "NULL Capture Device", (size_t)-1);
    }

    // Support everything on the null backend.
    pDeviceInfo->formatCount = mal_format_count - 1;    // Minus one because we don't want to include mal_format_unknown.
    for (mal_uint32 iFormat = 0; iFormat < pDeviceInfo->formatCount; ++iFormat) {
        pDeviceInfo->formats[iFormat] = (mal_format)(iFormat + 1);  // +1 to skip over mal_format_unknown.
    }

    pDeviceInfo->minChannels   = 1;
    pDeviceInfo->maxChannels   = MAL_MAX_CHANNELS;
    pDeviceInfo->minSampleRate = MAL_SAMPLE_RATE_8000;
    pDeviceInfo->maxSampleRate = MAL_SAMPLE_RATE_384000;

    return MAL_SUCCESS;
}


void mal_device_uninit__null(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);
    mal_free(pDevice->null_device.pBuffer);
}

mal_result mal_device_init__null(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pContext;
    (void)type;
    (void)pDeviceID;
    (void)pConfig;

    mal_assert(pDevice != NULL);
    mal_zero_object(&pDevice->null_device);

    if (type == mal_device_type_playback) {
        mal_strncpy_s(pDevice->name, sizeof(pDevice->name), "NULL Playback Device", (size_t)-1);
    } else {
        mal_strncpy_s(pDevice->name, sizeof(pDevice->name), "NULL Capture Device", (size_t)-1);
    }

    if (pDevice->bufferSizeInFrames == 0) {
        pDevice->bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, pDevice->sampleRate);
    }

    pDevice->null_device.pBuffer = (mal_uint8*)mal_malloc(pDevice->bufferSizeInFrames * pDevice->channels * mal_get_bytes_per_sample(pDevice->format));
    if (pDevice->null_device.pBuffer == NULL) {
        return MAL_OUT_OF_MEMORY;
    }

    mal_zero_memory(pDevice->null_device.pBuffer, mal_device_get_buffer_size_in_bytes(pDevice));

    return MAL_SUCCESS;
}

mal_result mal_device__start_backend__null(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_timer_init(&pDevice->null_device.timer);
    pDevice->null_device.lastProcessedFrame = 0;

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__null(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);
    (void)pDevice;

    return MAL_SUCCESS;
}

mal_result mal_device__break_main_loop__null(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->null_device.breakFromMainLoop = MAL_TRUE;
    return MAL_SUCCESS;
}

mal_bool32 mal_device__get_current_frame__null(mal_device* pDevice, mal_uint32* pCurrentPos)
{
    mal_assert(pDevice != NULL);
    mal_assert(pCurrentPos != NULL);
    *pCurrentPos = 0;

    mal_uint64 currentFrameAbs = (mal_uint64)(mal_timer_get_time_in_seconds(&pDevice->null_device.timer) * pDevice->sampleRate) / pDevice->channels;

    *pCurrentPos = (mal_uint32)(currentFrameAbs % pDevice->bufferSizeInFrames);
    return MAL_TRUE;
}

mal_uint32 mal_device__get_available_frames__null(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_uint32 currentFrame;
    if (!mal_device__get_current_frame__null(pDevice, &currentFrame)) {
        return 0;
    }

    // In a playback device the last processed frame should always be ahead of the current frame. The space between
    // the last processed and current frame (moving forward, starting from the last processed frame) is the amount
    // of space available to write.
    //
    // For a recording device it's the other way around - the last processed frame is always _behind_ the current
    // frame and the space between is the available space.
    mal_uint32 totalFrameCount = pDevice->bufferSizeInFrames;
    if (pDevice->type == mal_device_type_playback) {
        mal_uint32 committedBeg = currentFrame;
        mal_uint32 committedEnd = pDevice->null_device.lastProcessedFrame;
        if (committedEnd <= committedBeg) {
            committedEnd += totalFrameCount;    // Wrap around.
        }

        mal_uint32 committedSize = (committedEnd - committedBeg);
        mal_assert(committedSize <= totalFrameCount);

        return totalFrameCount - committedSize;
    } else {
        mal_uint32 validBeg = pDevice->null_device.lastProcessedFrame;
        mal_uint32 validEnd = currentFrame;
        if (validEnd < validBeg) {
            validEnd += totalFrameCount;        // Wrap around.
        }

        mal_uint32 validSize = (validEnd - validBeg);
        mal_assert(validSize <= totalFrameCount);

        return validSize;
    }
}

mal_uint32 mal_device__wait_for_frames__null(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    while (!pDevice->null_device.breakFromMainLoop) {
        mal_uint32 framesAvailable = mal_device__get_available_frames__null(pDevice);
        if (framesAvailable > 0) {
            return framesAvailable;
        }

        mal_sleep(16);
    }

    // We'll get here if the loop was terminated. Just return whatever's available.
    return mal_device__get_available_frames__null(pDevice);
}

mal_result mal_device__main_loop__null(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->null_device.breakFromMainLoop = MAL_FALSE;
    while (!pDevice->null_device.breakFromMainLoop) {
        mal_uint32 framesAvailable = mal_device__wait_for_frames__null(pDevice);
        if (framesAvailable == 0) {
            continue;
        }

        // If it's a playback device, don't bother grabbing more data if the device is being stopped.
        if (pDevice->null_device.breakFromMainLoop && pDevice->type == mal_device_type_playback) {
            return MAL_FALSE;
        }

        if (framesAvailable + pDevice->null_device.lastProcessedFrame > pDevice->bufferSizeInFrames) {
            framesAvailable = pDevice->bufferSizeInFrames - pDevice->null_device.lastProcessedFrame;
        }

        mal_uint32 sampleCount = framesAvailable * pDevice->channels;
        mal_uint32 lockOffset  = pDevice->null_device.lastProcessedFrame * pDevice->channels * mal_get_bytes_per_sample(pDevice->format);
        mal_uint32 lockSize    = sampleCount * mal_get_bytes_per_sample(pDevice->format);

        if (pDevice->type == mal_device_type_playback) {
            if (pDevice->null_device.breakFromMainLoop) {
                return MAL_FALSE;
            }

            mal_device__read_frames_from_client(pDevice, framesAvailable, pDevice->null_device.pBuffer + lockOffset);
        } else {
            mal_zero_memory(pDevice->null_device.pBuffer + lockOffset, lockSize);
            mal_device__send_frames_to_client(pDevice, framesAvailable, pDevice->null_device.pBuffer + lockOffset);
        }

        pDevice->null_device.lastProcessedFrame = (pDevice->null_device.lastProcessedFrame + framesAvailable) % pDevice->bufferSizeInFrames;
    }

    return MAL_SUCCESS;
}


mal_result mal_context_uninit__null(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_null);

    (void)pContext;
    return MAL_SUCCESS;
}

mal_result mal_context_init__null(mal_context* pContext)
{
    mal_assert(pContext != NULL);

    pContext->onUninit              = mal_context_uninit__null;
    pContext->onDeviceIDEqual       = mal_context_is_device_id_equal__null;
    pContext->onEnumDevices         = mal_context_enumerate_devices__null;
    pContext->onGetDeviceInfo       = mal_context_get_device_info__null;
    pContext->onDeviceInit          = mal_device_init__null;
    pContext->onDeviceUninit        = mal_device_uninit__null;
    pContext->onDeviceStart         = mal_device__start_backend__null;
    pContext->onDeviceStop          = mal_device__stop_backend__null;
    pContext->onDeviceBreakMainLoop = mal_device__break_main_loop__null;
    pContext->onDeviceMainLoop      = mal_device__main_loop__null;

    // The null backend always works.
    return MAL_SUCCESS;
}
#endif


///////////////////////////////////////////////////////////////////////////////
//
// WIN32 COMMON
//
///////////////////////////////////////////////////////////////////////////////
#if defined(MAL_WIN32)
#include "objbase.h"
#if defined(MAL_WIN32_DESKTOP)
    #define mal_CoInitializeEx(pContext, pvReserved, dwCoInit)                          ((MAL_PFN_CoInitializeEx)pContext->win32.CoInitializeEx)(pvReserved, dwCoInit)
    #define mal_CoUninitialize(pContext)                                                ((MAL_PFN_CoUninitialize)pContext->win32.CoUninitialize)()
    #define mal_CoCreateInstance(pContext, rclsid, pUnkOuter, dwClsContext, riid, ppv)  ((MAL_PFN_CoCreateInstance)pContext->win32.CoCreateInstance)(rclsid, pUnkOuter, dwClsContext, riid, ppv)
    #define mal_CoTaskMemFree(pContext, pv)                                             ((MAL_PFN_CoTaskMemFree)pContext->win32.CoTaskMemFree)(pv)
    #define mal_PropVariantClear(pContext, pvar)                                        ((MAL_PFN_PropVariantClear)pContext->win32.PropVariantClear)(pvar)
#else
    #define mal_CoInitializeEx(pContext, pvReserved, dwCoInit)                          CoInitializeEx(pvReserved, dwCoInit)
    #define mal_CoUninitialize(pContext)                                                CoUninitialize()
    #define mal_CoCreateInstance(pContext, rclsid, pUnkOuter, dwClsContext, riid, ppv)  CoCreateInstance(rclsid, pUnkOuter, dwClsContext, riid, ppv)
    #define mal_CoTaskMemFree(pContext, pv)                                             CoTaskMemFree(pv)
    #define mal_PropVariantClear(pContext, pvar)                                        PropVariantClear(pvar)
#endif

// There's a few common headers for Win32 backends which include here for simplicity. Note that we should never
// include any files that do not come standard with modern compilers, and we may need to manually define a few
// symbols.
#include <mmreg.h>
#include <mmsystem.h>

#if !defined(MAXULONG_PTR)
typedef size_t DWORD_PTR;
#endif

#if !defined(WAVE_FORMAT_44M08)
#define WAVE_FORMAT_44M08 0x00000100
#define WAVE_FORMAT_44S08 0x00000200
#define WAVE_FORMAT_44M16 0x00000400
#define WAVE_FORMAT_44S16 0x00000800
#define WAVE_FORMAT_48M08 0x00001000
#define WAVE_FORMAT_48S08 0x00002000
#define WAVE_FORMAT_48M16 0x00004000
#define WAVE_FORMAT_48S16 0x00008000
#define WAVE_FORMAT_96M08 0x00010000
#define WAVE_FORMAT_96S08 0x00020000
#define WAVE_FORMAT_96M16 0x00040000
#define WAVE_FORMAT_96S16 0x00080000
#endif

#ifndef SPEAKER_FRONT_LEFT
#define SPEAKER_FRONT_LEFT            0x1
#define SPEAKER_FRONT_RIGHT           0x2
#define SPEAKER_FRONT_CENTER          0x4
#define SPEAKER_LOW_FREQUENCY         0x8
#define SPEAKER_BACK_LEFT             0x10
#define SPEAKER_BACK_RIGHT            0x20
#define SPEAKER_FRONT_LEFT_OF_CENTER  0x40
#define SPEAKER_FRONT_RIGHT_OF_CENTER 0x80
#define SPEAKER_BACK_CENTER           0x100
#define SPEAKER_SIDE_LEFT             0x200
#define SPEAKER_SIDE_RIGHT            0x400
#define SPEAKER_TOP_CENTER            0x800
#define SPEAKER_TOP_FRONT_LEFT        0x1000
#define SPEAKER_TOP_FRONT_CENTER      0x2000
#define SPEAKER_TOP_FRONT_RIGHT       0x4000
#define SPEAKER_TOP_BACK_LEFT         0x8000
#define SPEAKER_TOP_BACK_CENTER       0x10000
#define SPEAKER_TOP_BACK_RIGHT        0x20000
#endif

// The SDK that comes with old versions of MSVC (VC6, for example) does not appear to define WAVEFORMATEXTENSIBLE. We
// define our own implementation in this case.
#if (defined(_MSC_VER) && !defined(_WAVEFORMATEXTENSIBLE_)) || defined(__DMC__)
typedef struct
{
    WAVEFORMATEX Format;
    union
    {
        WORD wValidBitsPerSample;
        WORD wSamplesPerBlock;
        WORD wReserved;
    } Samples;
    DWORD dwChannelMask;
    GUID SubFormat;
} WAVEFORMATEXTENSIBLE;
#endif

#ifndef WAVE_FORMAT_EXTENSIBLE
#define WAVE_FORMAT_EXTENSIBLE  0xFFFE
#endif

#ifndef WAVE_FORMAT_IEEE_FLOAT
#define WAVE_FORMAT_IEEE_FLOAT  0x0003
#endif

GUID MAL_GUID_NULL = {0x00000000, 0x0000, 0x0000, {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}};

// Converts an individual Win32-style channel identifier (SPEAKER_FRONT_LEFT, etc.) to mini_al.
mal_uint8 mal_channel_id_to_mal__win32(DWORD id)
{
    switch (id)
    {
        case SPEAKER_FRONT_LEFT:            return MAL_CHANNEL_FRONT_LEFT;
        case SPEAKER_FRONT_RIGHT:           return MAL_CHANNEL_FRONT_RIGHT;
        case SPEAKER_FRONT_CENTER:          return MAL_CHANNEL_FRONT_CENTER;
        case SPEAKER_LOW_FREQUENCY:         return MAL_CHANNEL_LFE;
        case SPEAKER_BACK_LEFT:             return MAL_CHANNEL_BACK_LEFT;
        case SPEAKER_BACK_RIGHT:            return MAL_CHANNEL_BACK_RIGHT;
        case SPEAKER_FRONT_LEFT_OF_CENTER:  return MAL_CHANNEL_FRONT_LEFT_CENTER;
        case SPEAKER_FRONT_RIGHT_OF_CENTER: return MAL_CHANNEL_FRONT_RIGHT_CENTER;
        case SPEAKER_BACK_CENTER:           return MAL_CHANNEL_BACK_CENTER;
        case SPEAKER_SIDE_LEFT:             return MAL_CHANNEL_SIDE_LEFT;
        case SPEAKER_SIDE_RIGHT:            return MAL_CHANNEL_SIDE_RIGHT;
        case SPEAKER_TOP_CENTER:            return MAL_CHANNEL_TOP_CENTER;
        case SPEAKER_TOP_FRONT_LEFT:        return MAL_CHANNEL_TOP_FRONT_LEFT;
        case SPEAKER_TOP_FRONT_CENTER:      return MAL_CHANNEL_TOP_FRONT_CENTER;
        case SPEAKER_TOP_FRONT_RIGHT:       return MAL_CHANNEL_TOP_FRONT_RIGHT;
        case SPEAKER_TOP_BACK_LEFT:         return MAL_CHANNEL_TOP_BACK_LEFT;
        case SPEAKER_TOP_BACK_CENTER:       return MAL_CHANNEL_TOP_BACK_CENTER;
        case SPEAKER_TOP_BACK_RIGHT:        return MAL_CHANNEL_TOP_BACK_RIGHT;
        default: return 0;
    }
}

// Converts an individual mini_al channel identifier (MAL_CHANNEL_FRONT_LEFT, etc.) to Win32-style.
DWORD mal_channel_id_to_win32(DWORD id)
{
    switch (id)
    {
        case MAL_CHANNEL_MONO:               return SPEAKER_FRONT_CENTER;
        case MAL_CHANNEL_FRONT_LEFT:         return SPEAKER_FRONT_LEFT;
        case MAL_CHANNEL_FRONT_RIGHT:        return SPEAKER_FRONT_RIGHT;
        case MAL_CHANNEL_FRONT_CENTER:       return SPEAKER_FRONT_CENTER;
        case MAL_CHANNEL_LFE:                return SPEAKER_LOW_FREQUENCY;
        case MAL_CHANNEL_BACK_LEFT:          return SPEAKER_BACK_LEFT;
        case MAL_CHANNEL_BACK_RIGHT:         return SPEAKER_BACK_RIGHT;
        case MAL_CHANNEL_FRONT_LEFT_CENTER:  return SPEAKER_FRONT_LEFT_OF_CENTER;
        case MAL_CHANNEL_FRONT_RIGHT_CENTER: return SPEAKER_FRONT_RIGHT_OF_CENTER;
        case MAL_CHANNEL_BACK_CENTER:        return SPEAKER_BACK_CENTER;
        case MAL_CHANNEL_SIDE_LEFT:          return SPEAKER_SIDE_LEFT;
        case MAL_CHANNEL_SIDE_RIGHT:         return SPEAKER_SIDE_RIGHT;
        case MAL_CHANNEL_TOP_CENTER:         return SPEAKER_TOP_CENTER;
        case MAL_CHANNEL_TOP_FRONT_LEFT:     return SPEAKER_TOP_FRONT_LEFT;
        case MAL_CHANNEL_TOP_FRONT_CENTER:   return SPEAKER_TOP_FRONT_CENTER;
        case MAL_CHANNEL_TOP_FRONT_RIGHT:    return SPEAKER_TOP_FRONT_RIGHT;
        case MAL_CHANNEL_TOP_BACK_LEFT:      return SPEAKER_TOP_BACK_LEFT;
        case MAL_CHANNEL_TOP_BACK_CENTER:    return SPEAKER_TOP_BACK_CENTER;
        case MAL_CHANNEL_TOP_BACK_RIGHT:     return SPEAKER_TOP_BACK_RIGHT;
        default: return 0;
    }
}

// Converts a channel mapping to a Win32-style channel mask.
DWORD mal_channel_map_to_channel_mask__win32(const mal_channel channelMap[MAL_MAX_CHANNELS], mal_uint32 channels)
{
    DWORD dwChannelMask = 0;
    for (mal_uint32 iChannel = 0; iChannel < channels; ++iChannel) {
        dwChannelMask |= mal_channel_id_to_win32(channelMap[iChannel]);
    }

    return dwChannelMask;
}

// Converts a Win32-style channel mask to a mini_al channel map.
void mal_channel_mask_to_channel_map__win32(DWORD dwChannelMask, mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    if (channels == 1 && dwChannelMask == 0) {
        channelMap[0] = MAL_CHANNEL_MONO;
    } else if (channels == 2 && dwChannelMask == 0) {
        channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
        channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
    } else {
        if (channels == 1 && (dwChannelMask & SPEAKER_FRONT_CENTER) != 0) {
            channelMap[0] = MAL_CHANNEL_MONO;
        } else {
            // Just iterate over each bit.
            mal_uint32 iChannel = 0;
            for (mal_uint32 iBit = 0; iBit < 32; ++iBit) {
                DWORD bitValue = (dwChannelMask & (1UL << iBit));
                if (bitValue != 0) {
                    // The bit is set.
                    channelMap[iChannel] = mal_channel_id_to_mal__win32(bitValue);
                    iChannel += 1;
                }
            }
        }
    }
}

#ifdef __cplusplus
mal_bool32 mal_is_guid_equal(const void* a, const void* b)
{
    return IsEqualGUID(*(const GUID*)a, *(const GUID*)b);
}
#else
#define mal_is_guid_equal(a, b) IsEqualGUID((const GUID*)a, (const GUID*)b)
#endif

mal_format mal_format_from_WAVEFORMATEX(const WAVEFORMATEX* pWF)
{
    mal_assert(pWF != NULL);

    if (pWF->wFormatTag == WAVE_FORMAT_EXTENSIBLE) {
        const WAVEFORMATEXTENSIBLE* pWFEX = (const WAVEFORMATEXTENSIBLE*)pWF;
        if (mal_is_guid_equal(&pWFEX->SubFormat, &MAL_GUID_KSDATAFORMAT_SUBTYPE_PCM)) {
            if (pWFEX->Samples.wValidBitsPerSample == 32) {
                return mal_format_s32;
            }
            if (pWFEX->Samples.wValidBitsPerSample == 24) {
                if (pWFEX->Format.wBitsPerSample == 32) {
                    //return mal_format_s24_32;
                }
                if (pWFEX->Format.wBitsPerSample == 24) {
                    return mal_format_s24;
                }
            }
            if (pWFEX->Samples.wValidBitsPerSample == 16) {
                return mal_format_s16;
            }
            if (pWFEX->Samples.wValidBitsPerSample == 8) {
                return mal_format_u8;
            }
        }
        if (mal_is_guid_equal(&pWFEX->SubFormat, &MAL_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT)) {
            if (pWFEX->Samples.wValidBitsPerSample == 32) {
                return mal_format_f32;
            }
            //if (pWFEX->Samples.wValidBitsPerSample == 64) {
            //    return mal_format_f64;
            //}
        }
    } else {
        if (pWF->wFormatTag == WAVE_FORMAT_PCM) {
            if (pWF->wBitsPerSample == 32) {
                return mal_format_s32;
            }
            if (pWF->wBitsPerSample == 24) {
                return mal_format_s24;
            }
            if (pWF->wBitsPerSample == 16) {
                return mal_format_s16;
            }
            if (pWF->wBitsPerSample == 8) {
                return mal_format_u8;
            }
        }
        if (pWF->wFormatTag == WAVE_FORMAT_IEEE_FLOAT) {
            if (pWF->wBitsPerSample == 32) {
                return mal_format_f32;
            }
            if (pWF->wBitsPerSample == 64) {
                //return mal_format_f64;
            }
        }
    }

    return mal_format_unknown;
}
#endif


///////////////////////////////////////////////////////////////////////////////
//
// WASAPI Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_WASAPI
//#if defined(_MSC_VER)
//    #pragma warning(push)
//    #pragma warning(disable:4091)   // 'typedef ': ignored on left of '' when no variable is declared
//#endif
//#include <audioclient.h>
//#include <mmdeviceapi.h>
//#if defined(_MSC_VER)
//    #pragma warning(pop)
//#endif


// Some compilers don't define VerifyVersionInfoW. Need to write this ourselves.
#if defined(__DMC__)
#define _WIN32_WINNT_VISTA      0x0600
#define VER_MINORVERSION        0x01
#define VER_MAJORVERSION        0x02
#define VER_SERVICEPACKMAJOR    0x20
#define VER_GREATER_EQUAL       0x03

typedef struct  {
    DWORD dwOSVersionInfoSize;
    DWORD dwMajorVersion;
    DWORD dwMinorVersion;
    DWORD dwBuildNumber;
    DWORD dwPlatformId;
    WCHAR szCSDVersion[128];
    WORD  wServicePackMajor;
    WORD  wServicePackMinor;
    WORD  wSuiteMask;
    BYTE  wProductType;
    BYTE  wReserved;
} mal_OSVERSIONINFOEXW;

BOOL WINAPI VerifyVersionInfoW(mal_OSVERSIONINFOEXW* lpVersionInfo, DWORD dwTypeMask, DWORDLONG dwlConditionMask);
ULONGLONG WINAPI VerSetConditionMask(ULONGLONG dwlConditionMask, DWORD dwTypeBitMask, BYTE dwConditionMask);
#else
typedef OSVERSIONINFOEXW mal_OSVERSIONINFOEXW;
#endif


#ifndef PROPERTYKEY_DEFINED
#define PROPERTYKEY_DEFINED
typedef struct
{
    GUID fmtid;
    DWORD pid;
} PROPERTYKEY;
#endif

// Some compilers don't define PropVariantInit(). We just do this ourselves since it's just a memset().
static MAL_INLINE void mal_PropVariantInit(PROPVARIANT* pProp)
{
    mal_zero_object(pProp);
}


const PROPERTYKEY MAL_PKEY_Device_FriendlyName             = {{0xA45C254E, 0xDF1C, 0x4EFD, {0x80, 0x20, 0x67, 0xD1, 0x46, 0xA8, 0x50, 0xE0}}, 14};
const PROPERTYKEY MAL_PKEY_AudioEngine_DeviceFormat        = {{0xF19F064D, 0x82C,  0x4E27, {0xBC, 0x73, 0x68, 0x82, 0xA1, 0xBB, 0x8E, 0x4C}},  0};

const IID MAL_IID_IUnknown                                 = {0x00000000, 0x0000, 0x0000, {0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46}}; // 00000000-0000-0000-C000-000000000046
const IID MAL_IID_IAgileObject                             = {0x94EA2B94, 0xE9CC, 0x49E0, {0xC0, 0xFF, 0xEE, 0x64, 0xCA, 0x8F, 0x5B, 0x90}}; // 94EA2B94-E9CC-49E0-C0FF-EE64CA8F5B90

const IID MAL_IID_IAudioClient                             = {0x1CB9AD4C, 0xDBFA, 0x4C32, {0xB1, 0x78, 0xC2, 0xF5, 0x68, 0xA7, 0x03, 0xB2}}; // 1CB9AD4C-DBFA-4C32-B178-C2F568A703B2 = __uuidof(IAudioClient)
const IID MAL_IID_IAudioClient2                            = {0x726778CD, 0xF60A, 0x4EDA, {0x82, 0xDE, 0xE4, 0x76, 0x10, 0xCD, 0x78, 0xAA}}; // 726778CD-F60A-4EDA-82DE-E47610CD78AA = __uuidof(IAudioClient2)
const IID MAL_IID_IAudioClient3                            = {0x7ED4EE07, 0x8E67, 0x4CD4, {0x8C, 0x1A, 0x2B, 0x7A, 0x59, 0x87, 0xAD, 0x42}}; // 7ED4EE07-8E67-4CD4-8C1A-2B7A5987AD42 = __uuidof(IAudioClient3)
const IID MAL_IID_IAudioRenderClient                       = {0xF294ACFC, 0x3146, 0x4483, {0xA7, 0xBF, 0xAD, 0xDC, 0xA7, 0xC2, 0x60, 0xE2}}; // F294ACFC-3146-4483-A7BF-ADDCA7C260E2 = __uuidof(IAudioRenderClient)
const IID MAL_IID_IAudioCaptureClient                      = {0xC8ADBD64, 0xE71E, 0x48A0, {0xA4, 0xDE, 0x18, 0x5C, 0x39, 0x5C, 0xD3, 0x17}}; // C8ADBD64-E71E-48A0-A4DE-185C395CD317 = __uuidof(IAudioCaptureClient)
const IID MAL_IID_IMMNotificationClient                    = {0x7991EEC9, 0x7E89, 0x4D85, {0x83, 0x90, 0x6C, 0x70, 0x3C, 0xEC, 0x60, 0xC0}}; // 7991EEC9-7E89-4D85-8390-6C703CEC60C0 = __uuidof(IMMNotificationClient)
#ifndef MAL_WIN32_DESKTOP
const IID MAL_IID_DEVINTERFACE_AUDIO_RENDER                = {0xE6327CAD, 0xDCEC, 0x4949, {0xAE, 0x8A, 0x99, 0x1E, 0x97, 0x6A, 0x79, 0xD2}}; // E6327CAD-DCEC-4949-AE8A-991E976A79D2
const IID MAL_IID_DEVINTERFACE_AUDIO_CAPTURE               = {0x2EEF81BE, 0x33FA, 0x4800, {0x96, 0x70, 0x1C, 0xD4, 0x74, 0x97, 0x2C, 0x3F}}; // 2EEF81BE-33FA-4800-9670-1CD474972C3F
const IID MAL_IID_IActivateAudioInterfaceCompletionHandler = {0x41D949AB, 0x9862, 0x444A, {0x80, 0xF6, 0xC2, 0x61, 0x33, 0x4D, 0xA5, 0xEB}}; // 41D949AB-9862-444A-80F6-C261334DA5EB
#endif

const IID MAL_CLSID_MMDeviceEnumerator_Instance            = {0xBCDE0395, 0xE52F, 0x467C, {0x8E, 0x3D, 0xC4, 0x57, 0x92, 0x91, 0x69, 0x2E}}; // BCDE0395-E52F-467C-8E3D-C4579291692E = __uuidof(MMDeviceEnumerator)
const IID MAL_IID_IMMDeviceEnumerator_Instance             = {0xA95664D2, 0x9614, 0x4F35, {0xA7, 0x46, 0xDE, 0x8D, 0xB6, 0x36, 0x17, 0xE6}}; // A95664D2-9614-4F35-A746-DE8DB63617E6 = __uuidof(IMMDeviceEnumerator)
#ifdef __cplusplus
#define MAL_CLSID_MMDeviceEnumerator MAL_CLSID_MMDeviceEnumerator_Instance
#define MAL_IID_IMMDeviceEnumerator  MAL_IID_IMMDeviceEnumerator_Instance
#else
#define MAL_CLSID_MMDeviceEnumerator &MAL_CLSID_MMDeviceEnumerator_Instance
#define MAL_IID_IMMDeviceEnumerator  &MAL_IID_IMMDeviceEnumerator_Instance
#endif

typedef struct mal_IUnknown                                 mal_IUnknown;
#ifdef MAL_WIN32_DESKTOP
#define MAL_MM_DEVICE_STATE_ACTIVE                          1
#define MAL_MM_DEVICE_STATE_DISABLED                        2
#define MAL_MM_DEVICE_STATE_NOTPRESENT                      4
#define MAL_MM_DEVICE_STATE_UNPLUGGED                       8

typedef struct mal_IMMDeviceEnumerator                      mal_IMMDeviceEnumerator;
typedef struct mal_IMMDeviceCollection                      mal_IMMDeviceCollection;
typedef struct mal_IMMDevice                                mal_IMMDevice;
#else
typedef struct mal_IActivateAudioInterfaceCompletionHandler mal_IActivateAudioInterfaceCompletionHandler;
typedef struct mal_IActivateAudioInterfaceAsyncOperation    mal_IActivateAudioInterfaceAsyncOperation;
#endif
typedef struct mal_IPropertyStore                           mal_IPropertyStore;
typedef struct mal_IAudioClient                             mal_IAudioClient;
typedef struct mal_IAudioClient2                            mal_IAudioClient2;
typedef struct mal_IAudioClient3                            mal_IAudioClient3;
typedef struct mal_IAudioRenderClient                       mal_IAudioRenderClient;
typedef struct mal_IAudioCaptureClient                      mal_IAudioCaptureClient;

typedef mal_int64                                           MAL_REFERENCE_TIME;

#define MAL_AUDCLNT_STREAMFLAGS_CROSSPROCESS                0x00010000
#define MAL_AUDCLNT_STREAMFLAGS_LOOPBACK                    0x00020000
#define MAL_AUDCLNT_STREAMFLAGS_EVENTCALLBACK               0x00040000
#define MAL_AUDCLNT_STREAMFLAGS_NOPERSIST                   0x00080000
#define MAL_AUDCLNT_STREAMFLAGS_RATEADJUST                  0x00100000
#define MAL_AUDCLNT_STREAMFLAGS_SRC_DEFAULT_QUALITY         0x08000000
#define MAL_AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM              0x80000000
#define MAL_AUDCLNT_SESSIONFLAGS_EXPIREWHENUNOWNED          0x10000000
#define MAL_AUDCLNT_SESSIONFLAGS_DISPLAY_HIDE               0x20000000
#define MAL_AUDCLNT_SESSIONFLAGS_DISPLAY_HIDEWHENEXPIRED    0x40000000

// We only care about a few error codes.
#define MAL_AUDCLNT_E_INVALID_DEVICE_PERIOD                 (-2004287456)
#define MAL_AUDCLNT_E_BUFFER_SIZE_NOT_ALIGNED               (-2004287463)
#define MAL_AUDCLNT_S_BUFFER_EMPTY                          (143196161)
#define MAL_AUDCLNT_E_DEVICE_IN_USE                         (-2004287478)

typedef enum
{
    mal_eRender  = 0,
    mal_eCapture = 1,
    mal_eAll     = 2
} mal_EDataFlow;

typedef enum
{
    mal_eConsole        = 0,
    mal_eMultimedia     = 1,
    mal_eCommunications = 2
} mal_ERole;

typedef enum
{
    MAL_AUDCLNT_SHAREMODE_SHARED,
    MAL_AUDCLNT_SHAREMODE_EXCLUSIVE
} MAL_AUDCLNT_SHAREMODE;

typedef enum
{
    MAL_AudioCategory_Other = 0,    // <-- mini_al is only caring about Other.
} MAL_AUDIO_STREAM_CATEGORY;

typedef struct
{
    UINT32 cbSize;
    BOOL bIsOffload;
    MAL_AUDIO_STREAM_CATEGORY eCategory;
} mal_AudioClientProperties;

// IUnknown
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IUnknown* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IUnknown* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IUnknown* pThis);
} mal_IUnknownVtbl;
struct mal_IUnknown
{
    mal_IUnknownVtbl* lpVtbl;
};
HRESULT mal_IUnknown_QueryInterface(mal_IUnknown* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IUnknown_AddRef(mal_IUnknown* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IUnknown_Release(mal_IUnknown* pThis)                                                { return pThis->lpVtbl->Release(pThis); }

#ifdef MAL_WIN32_DESKTOP
    // IMMNotificationClient
    typedef struct
    {
        // IUnknown
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IMMNotificationClient* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IMMNotificationClient* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (mal_IMMNotificationClient* pThis);

        // IMMNotificationClient
        HRESULT (STDMETHODCALLTYPE * OnDeviceStateChanged)  (mal_IMMNotificationClient* pThis, LPCWSTR pDeviceID, DWORD dwNewState);
        HRESULT (STDMETHODCALLTYPE * OnDeviceAdded)         (mal_IMMNotificationClient* pThis, LPCWSTR pDeviceID);
        HRESULT (STDMETHODCALLTYPE * OnDeviceRemoved)       (mal_IMMNotificationClient* pThis, LPCWSTR pDeviceID);
        HRESULT (STDMETHODCALLTYPE * OnDefaultDeviceChanged)(mal_IMMNotificationClient* pThis, mal_EDataFlow dataFlow, mal_ERole role, LPCWSTR pDefaultDeviceID);
        HRESULT (STDMETHODCALLTYPE * OnPropertyValueChanged)(mal_IMMNotificationClient* pThis, LPCWSTR pDeviceID, const PROPERTYKEY key);
    } mal_IMMNotificationClientVtbl;

    // IMMDeviceEnumerator
    typedef struct
    {
        // IUnknown
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IMMDeviceEnumerator* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IMMDeviceEnumerator* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (mal_IMMDeviceEnumerator* pThis);

        // IMMDeviceEnumerator
        HRESULT (STDMETHODCALLTYPE * EnumAudioEndpoints)                    (mal_IMMDeviceEnumerator* pThis, mal_EDataFlow dataFlow, DWORD dwStateMask, mal_IMMDeviceCollection** ppDevices);
        HRESULT (STDMETHODCALLTYPE * GetDefaultAudioEndpoint)               (mal_IMMDeviceEnumerator* pThis, mal_EDataFlow dataFlow, mal_ERole role, mal_IMMDevice** ppEndpoint);
        HRESULT (STDMETHODCALLTYPE * GetDevice)                             (mal_IMMDeviceEnumerator* pThis, LPCWSTR pID, mal_IMMDevice** ppDevice);
        HRESULT (STDMETHODCALLTYPE * RegisterEndpointNotificationCallback)  (mal_IMMDeviceEnumerator* pThis, mal_IMMNotificationClient* pClient);
        HRESULT (STDMETHODCALLTYPE * UnregisterEndpointNotificationCallback)(mal_IMMDeviceEnumerator* pThis, mal_IMMNotificationClient* pClient);
    } mal_IMMDeviceEnumeratorVtbl;
    struct mal_IMMDeviceEnumerator
    {
        mal_IMMDeviceEnumeratorVtbl* lpVtbl;
    };
    HRESULT mal_IMMDeviceEnumerator_QueryInterface(mal_IMMDeviceEnumerator* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    ULONG   mal_IMMDeviceEnumerator_AddRef(mal_IMMDeviceEnumerator* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    ULONG   mal_IMMDeviceEnumerator_Release(mal_IMMDeviceEnumerator* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    HRESULT mal_IMMDeviceEnumerator_EnumAudioEndpoints(mal_IMMDeviceEnumerator* pThis, mal_EDataFlow dataFlow, DWORD dwStateMask, mal_IMMDeviceCollection** ppDevices) { return pThis->lpVtbl->EnumAudioEndpoints(pThis, dataFlow, dwStateMask, ppDevices); }
    HRESULT mal_IMMDeviceEnumerator_GetDefaultAudioEndpoint(mal_IMMDeviceEnumerator* pThis, mal_EDataFlow dataFlow, mal_ERole role, mal_IMMDevice** ppEndpoint) { return pThis->lpVtbl->GetDefaultAudioEndpoint(pThis, dataFlow, role, ppEndpoint); }
    HRESULT mal_IMMDeviceEnumerator_GetDevice(mal_IMMDeviceEnumerator* pThis, LPCWSTR pID, mal_IMMDevice** ppDevice)       { return pThis->lpVtbl->GetDevice(pThis, pID, ppDevice); }
    HRESULT mal_IMMDeviceEnumerator_RegisterEndpointNotificationCallback(mal_IMMDeviceEnumerator* pThis, mal_IMMNotificationClient* pClient) { return pThis->lpVtbl->RegisterEndpointNotificationCallback(pThis, pClient); }
    HRESULT mal_IMMDeviceEnumerator_UnregisterEndpointNotificationCallback(mal_IMMDeviceEnumerator* pThis, mal_IMMNotificationClient* pClient) { return pThis->lpVtbl->UnregisterEndpointNotificationCallback(pThis, pClient); }


    // IMMDeviceCollection
    typedef struct
    {
        // IUnknown
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IMMDeviceCollection* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IMMDeviceCollection* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (mal_IMMDeviceCollection* pThis);

        // IMMDeviceCollection
        HRESULT (STDMETHODCALLTYPE * GetCount)(mal_IMMDeviceCollection* pThis, UINT* pDevices);
        HRESULT (STDMETHODCALLTYPE * Item)    (mal_IMMDeviceCollection* pThis, UINT nDevice, mal_IMMDevice** ppDevice);
    } mal_IMMDeviceCollectionVtbl;
    struct mal_IMMDeviceCollection
    {
        mal_IMMDeviceCollectionVtbl* lpVtbl;
    };
    HRESULT mal_IMMDeviceCollection_QueryInterface(mal_IMMDeviceCollection* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    ULONG   mal_IMMDeviceCollection_AddRef(mal_IMMDeviceCollection* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    ULONG   mal_IMMDeviceCollection_Release(mal_IMMDeviceCollection* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    HRESULT mal_IMMDeviceCollection_GetCount(mal_IMMDeviceCollection* pThis, UINT* pDevices)                               { return pThis->lpVtbl->GetCount(pThis, pDevices); }
    HRESULT mal_IMMDeviceCollection_Item(mal_IMMDeviceCollection* pThis, UINT nDevice, mal_IMMDevice** ppDevice)           { return pThis->lpVtbl->Item(pThis, nDevice, ppDevice); }


    // IMMDevice
    typedef struct
    {
        // IUnknown
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IMMDevice* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IMMDevice* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (mal_IMMDevice* pThis);

        // IMMDevice
        HRESULT (STDMETHODCALLTYPE * Activate)         (mal_IMMDevice* pThis, const IID* const iid, DWORD dwClsCtx, PROPVARIANT* pActivationParams, void** ppInterface);
        HRESULT (STDMETHODCALLTYPE * OpenPropertyStore)(mal_IMMDevice* pThis, DWORD stgmAccess, mal_IPropertyStore** ppProperties);
        HRESULT (STDMETHODCALLTYPE * GetId)            (mal_IMMDevice* pThis, LPWSTR *pID);
        HRESULT (STDMETHODCALLTYPE * GetState)         (mal_IMMDevice* pThis, DWORD *pState);
    } mal_IMMDeviceVtbl;
    struct mal_IMMDevice
    {
        mal_IMMDeviceVtbl* lpVtbl;
    };
    HRESULT mal_IMMDevice_QueryInterface(mal_IMMDevice* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    ULONG   mal_IMMDevice_AddRef(mal_IMMDevice* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    ULONG   mal_IMMDevice_Release(mal_IMMDevice* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    HRESULT mal_IMMDevice_Activate(mal_IMMDevice* pThis, const IID* const iid, DWORD dwClsCtx, PROPVARIANT* pActivationParams, void** ppInterface) { return pThis->lpVtbl->Activate(pThis, iid, dwClsCtx, pActivationParams, ppInterface); }
    HRESULT mal_IMMDevice_OpenPropertyStore(mal_IMMDevice* pThis, DWORD stgmAccess, mal_IPropertyStore** ppProperties) { return pThis->lpVtbl->OpenPropertyStore(pThis, stgmAccess, ppProperties); }
    HRESULT mal_IMMDevice_GetId(mal_IMMDevice* pThis, LPWSTR *pID)                                     { return pThis->lpVtbl->GetId(pThis, pID); }
    HRESULT mal_IMMDevice_GetState(mal_IMMDevice* pThis, DWORD *pState)                                { return pThis->lpVtbl->GetState(pThis, pState); }
#else
    // IActivateAudioInterfaceAsyncOperation
    typedef struct
    {
        // IUnknown
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IActivateAudioInterfaceAsyncOperation* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IActivateAudioInterfaceAsyncOperation* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (mal_IActivateAudioInterfaceAsyncOperation* pThis);

        // IActivateAudioInterfaceAsyncOperation
        HRESULT (STDMETHODCALLTYPE * GetActivateResult)(mal_IActivateAudioInterfaceAsyncOperation* pThis, HRESULT *pActivateResult, mal_IUnknown** ppActivatedInterface);
    } mal_IActivateAudioInterfaceAsyncOperationVtbl;
    struct mal_IActivateAudioInterfaceAsyncOperation
    {
        mal_IActivateAudioInterfaceAsyncOperationVtbl* lpVtbl;
    };
    HRESULT mal_IActivateAudioInterfaceAsyncOperation_QueryInterface(mal_IActivateAudioInterfaceAsyncOperation* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    ULONG   mal_IActivateAudioInterfaceAsyncOperation_AddRef(mal_IActivateAudioInterfaceAsyncOperation* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    ULONG   mal_IActivateAudioInterfaceAsyncOperation_Release(mal_IActivateAudioInterfaceAsyncOperation* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    HRESULT mal_IActivateAudioInterfaceAsyncOperation_GetActivateResult(mal_IActivateAudioInterfaceAsyncOperation* pThis, HRESULT *pActivateResult, mal_IUnknown** ppActivatedInterface) { return pThis->lpVtbl->GetActivateResult(pThis, pActivateResult, ppActivatedInterface); }
#endif

// IPropertyStore
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IPropertyStore* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IPropertyStore* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IPropertyStore* pThis);

    // IPropertyStore
    HRESULT (STDMETHODCALLTYPE * GetCount)(mal_IPropertyStore* pThis, DWORD* pPropCount);
    HRESULT (STDMETHODCALLTYPE * GetAt)   (mal_IPropertyStore* pThis, DWORD propIndex, PROPERTYKEY* pPropKey);
    HRESULT (STDMETHODCALLTYPE * GetValue)(mal_IPropertyStore* pThis, const PROPERTYKEY* const pKey, PROPVARIANT* pPropVar);
    HRESULT (STDMETHODCALLTYPE * SetValue)(mal_IPropertyStore* pThis, const PROPERTYKEY* const pKey, const PROPVARIANT* const pPropVar);
    HRESULT (STDMETHODCALLTYPE * Commit)  (mal_IPropertyStore* pThis);
} mal_IPropertyStoreVtbl;
struct mal_IPropertyStore
{
    mal_IPropertyStoreVtbl* lpVtbl;
};
HRESULT mal_IPropertyStore_QueryInterface(mal_IPropertyStore* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IPropertyStore_AddRef(mal_IPropertyStore* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IPropertyStore_Release(mal_IPropertyStore* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IPropertyStore_GetCount(mal_IPropertyStore* pThis, DWORD* pPropCount)                            { return pThis->lpVtbl->GetCount(pThis, pPropCount); }
HRESULT mal_IPropertyStore_GetAt(mal_IPropertyStore* pThis, DWORD propIndex, PROPERTYKEY* pPropKey)          { return pThis->lpVtbl->GetAt(pThis, propIndex, pPropKey); }
HRESULT mal_IPropertyStore_GetValue(mal_IPropertyStore* pThis, const PROPERTYKEY* const pKey, PROPVARIANT* pPropVar) { return pThis->lpVtbl->GetValue(pThis, pKey, pPropVar); }
HRESULT mal_IPropertyStore_SetValue(mal_IPropertyStore* pThis, const PROPERTYKEY* const pKey, const PROPVARIANT* const pPropVar) { return pThis->lpVtbl->SetValue(pThis, pKey, pPropVar); }
HRESULT mal_IPropertyStore_Commit(mal_IPropertyStore* pThis)                                                 { return pThis->lpVtbl->Commit(pThis); }


// IAudioClient
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IAudioClient* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IAudioClient* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IAudioClient* pThis);

    // IAudioClient
    HRESULT (STDMETHODCALLTYPE * Initialize)       (mal_IAudioClient* pThis, MAL_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MAL_REFERENCE_TIME bufferDuration, MAL_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
    HRESULT (STDMETHODCALLTYPE * GetBufferSize)    (mal_IAudioClient* pThis, mal_uint32* pNumBufferFrames);
    HRESULT (STDMETHODCALLTYPE * GetStreamLatency) (mal_IAudioClient* pThis, MAL_REFERENCE_TIME* pLatency);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPadding)(mal_IAudioClient* pThis, mal_uint32* pNumPaddingFrames);
    HRESULT (STDMETHODCALLTYPE * IsFormatSupported)(mal_IAudioClient* pThis, MAL_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch);
    HRESULT (STDMETHODCALLTYPE * GetMixFormat)     (mal_IAudioClient* pThis, WAVEFORMATEX** ppDeviceFormat);
    HRESULT (STDMETHODCALLTYPE * GetDevicePeriod)  (mal_IAudioClient* pThis, MAL_REFERENCE_TIME* pDefaultDevicePeriod, MAL_REFERENCE_TIME* pMinimumDevicePeriod);
    HRESULT (STDMETHODCALLTYPE * Start)            (mal_IAudioClient* pThis);
    HRESULT (STDMETHODCALLTYPE * Stop)             (mal_IAudioClient* pThis);
    HRESULT (STDMETHODCALLTYPE * Reset)            (mal_IAudioClient* pThis);
    HRESULT (STDMETHODCALLTYPE * SetEventHandle)   (mal_IAudioClient* pThis, HANDLE eventHandle);
    HRESULT (STDMETHODCALLTYPE * GetService)       (mal_IAudioClient* pThis, const IID* const riid, void** pp);
} mal_IAudioClientVtbl;
struct mal_IAudioClient
{
    mal_IAudioClientVtbl* lpVtbl;
};
HRESULT mal_IAudioClient_QueryInterface(mal_IAudioClient* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IAudioClient_AddRef(mal_IAudioClient* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IAudioClient_Release(mal_IAudioClient* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IAudioClient_Initialize(mal_IAudioClient* pThis, MAL_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MAL_REFERENCE_TIME bufferDuration, MAL_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid) { return pThis->lpVtbl->Initialize(pThis, shareMode, streamFlags, bufferDuration, periodicity, pFormat, pAudioSessionGuid); }
HRESULT mal_IAudioClient_GetBufferSize(mal_IAudioClient* pThis, mal_uint32* pNumBufferFrames)                { return pThis->lpVtbl->GetBufferSize(pThis, pNumBufferFrames); }
HRESULT mal_IAudioClient_GetStreamLatency(mal_IAudioClient* pThis, MAL_REFERENCE_TIME* pLatency)             { return pThis->lpVtbl->GetStreamLatency(pThis, pLatency); }
HRESULT mal_IAudioClient_GetCurrentPadding(mal_IAudioClient* pThis, mal_uint32* pNumPaddingFrames)           { return pThis->lpVtbl->GetCurrentPadding(pThis, pNumPaddingFrames); }
HRESULT mal_IAudioClient_IsFormatSupported(mal_IAudioClient* pThis, MAL_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch) { return pThis->lpVtbl->IsFormatSupported(pThis, shareMode, pFormat, ppClosestMatch); }
HRESULT mal_IAudioClient_GetMixFormat(mal_IAudioClient* pThis, WAVEFORMATEX** ppDeviceFormat)            { return pThis->lpVtbl->GetMixFormat(pThis, ppDeviceFormat); }
HRESULT mal_IAudioClient_GetDevicePeriod(mal_IAudioClient* pThis, MAL_REFERENCE_TIME* pDefaultDevicePeriod, MAL_REFERENCE_TIME* pMinimumDevicePeriod) { return pThis->lpVtbl->GetDevicePeriod(pThis, pDefaultDevicePeriod, pMinimumDevicePeriod); }
HRESULT mal_IAudioClient_Start(mal_IAudioClient* pThis)                                                  { return pThis->lpVtbl->Start(pThis); }
HRESULT mal_IAudioClient_Stop(mal_IAudioClient* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
HRESULT mal_IAudioClient_Reset(mal_IAudioClient* pThis)                                                  { return pThis->lpVtbl->Reset(pThis); }
HRESULT mal_IAudioClient_SetEventHandle(mal_IAudioClient* pThis, HANDLE eventHandle)                     { return pThis->lpVtbl->SetEventHandle(pThis, eventHandle); }
HRESULT mal_IAudioClient_GetService(mal_IAudioClient* pThis, const IID* const riid, void** pp)           { return pThis->lpVtbl->GetService(pThis, riid, pp); }

// IAudioClient2
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IAudioClient2* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IAudioClient2* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IAudioClient2* pThis);

    // IAudioClient
    HRESULT (STDMETHODCALLTYPE * Initialize)       (mal_IAudioClient2* pThis, MAL_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MAL_REFERENCE_TIME bufferDuration, MAL_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
    HRESULT (STDMETHODCALLTYPE * GetBufferSize)    (mal_IAudioClient2* pThis, mal_uint32* pNumBufferFrames);
    HRESULT (STDMETHODCALLTYPE * GetStreamLatency) (mal_IAudioClient2* pThis, MAL_REFERENCE_TIME* pLatency);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPadding)(mal_IAudioClient2* pThis, mal_uint32* pNumPaddingFrames);
    HRESULT (STDMETHODCALLTYPE * IsFormatSupported)(mal_IAudioClient2* pThis, MAL_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch);
    HRESULT (STDMETHODCALLTYPE * GetMixFormat)     (mal_IAudioClient2* pThis, WAVEFORMATEX** ppDeviceFormat);
    HRESULT (STDMETHODCALLTYPE * GetDevicePeriod)  (mal_IAudioClient2* pThis, MAL_REFERENCE_TIME* pDefaultDevicePeriod, MAL_REFERENCE_TIME* pMinimumDevicePeriod);
    HRESULT (STDMETHODCALLTYPE * Start)            (mal_IAudioClient2* pThis);
    HRESULT (STDMETHODCALLTYPE * Stop)             (mal_IAudioClient2* pThis);
    HRESULT (STDMETHODCALLTYPE * Reset)            (mal_IAudioClient2* pThis);
    HRESULT (STDMETHODCALLTYPE * SetEventHandle)   (mal_IAudioClient2* pThis, HANDLE eventHandle);
    HRESULT (STDMETHODCALLTYPE * GetService)       (mal_IAudioClient2* pThis, const IID* const riid, void** pp);

    // IAudioClient2
    HRESULT (STDMETHODCALLTYPE * IsOffloadCapable)   (mal_IAudioClient2* pThis, MAL_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable);
    HRESULT (STDMETHODCALLTYPE * SetClientProperties)(mal_IAudioClient2* pThis, const mal_AudioClientProperties* pProperties);
    HRESULT (STDMETHODCALLTYPE * GetBufferSizeLimits)(mal_IAudioClient2* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MAL_REFERENCE_TIME* pMinBufferDuration, MAL_REFERENCE_TIME* pMaxBufferDuration);
} mal_IAudioClient2Vtbl;
struct mal_IAudioClient2
{
    mal_IAudioClient2Vtbl* lpVtbl;
};
HRESULT mal_IAudioClient2_QueryInterface(mal_IAudioClient2* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IAudioClient2_AddRef(mal_IAudioClient2* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IAudioClient2_Release(mal_IAudioClient2* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IAudioClient2_Initialize(mal_IAudioClient2* pThis, MAL_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MAL_REFERENCE_TIME bufferDuration, MAL_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid) { return pThis->lpVtbl->Initialize(pThis, shareMode, streamFlags, bufferDuration, periodicity, pFormat, pAudioSessionGuid); }
HRESULT mal_IAudioClient2_GetBufferSize(mal_IAudioClient2* pThis, mal_uint32* pNumBufferFrames)                { return pThis->lpVtbl->GetBufferSize(pThis, pNumBufferFrames); }
HRESULT mal_IAudioClient2_GetStreamLatency(mal_IAudioClient2* pThis, MAL_REFERENCE_TIME* pLatency)             { return pThis->lpVtbl->GetStreamLatency(pThis, pLatency); }
HRESULT mal_IAudioClient2_GetCurrentPadding(mal_IAudioClient2* pThis, mal_uint32* pNumPaddingFrames)           { return pThis->lpVtbl->GetCurrentPadding(pThis, pNumPaddingFrames); }
HRESULT mal_IAudioClient2_IsFormatSupported(mal_IAudioClient2* pThis, MAL_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch) { return pThis->lpVtbl->IsFormatSupported(pThis, shareMode, pFormat, ppClosestMatch); }
HRESULT mal_IAudioClient2_GetMixFormat(mal_IAudioClient2* pThis, WAVEFORMATEX** ppDeviceFormat)            { return pThis->lpVtbl->GetMixFormat(pThis, ppDeviceFormat); }
HRESULT mal_IAudioClient2_GetDevicePeriod(mal_IAudioClient2* pThis, MAL_REFERENCE_TIME* pDefaultDevicePeriod, MAL_REFERENCE_TIME* pMinimumDevicePeriod) { return pThis->lpVtbl->GetDevicePeriod(pThis, pDefaultDevicePeriod, pMinimumDevicePeriod); }
HRESULT mal_IAudioClient2_Start(mal_IAudioClient2* pThis)                                                  { return pThis->lpVtbl->Start(pThis); }
HRESULT mal_IAudioClient2_Stop(mal_IAudioClient2* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
HRESULT mal_IAudioClient2_Reset(mal_IAudioClient2* pThis)                                                  { return pThis->lpVtbl->Reset(pThis); }
HRESULT mal_IAudioClient2_SetEventHandle(mal_IAudioClient2* pThis, HANDLE eventHandle)                     { return pThis->lpVtbl->SetEventHandle(pThis, eventHandle); }
HRESULT mal_IAudioClient2_GetService(mal_IAudioClient2* pThis, const IID* const riid, void** pp)           { return pThis->lpVtbl->GetService(pThis, riid, pp); }
HRESULT mal_IAudioClient2_IsOffloadCapable(mal_IAudioClient2* pThis, MAL_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable) { return pThis->lpVtbl->IsOffloadCapable(pThis, category, pOffloadCapable); }
HRESULT mal_IAudioClient2_SetClientProperties(mal_IAudioClient2* pThis, const mal_AudioClientProperties* pProperties)           { return pThis->lpVtbl->SetClientProperties(pThis, pProperties); }
HRESULT mal_IAudioClient2_GetBufferSizeLimits(mal_IAudioClient2* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MAL_REFERENCE_TIME* pMinBufferDuration, MAL_REFERENCE_TIME* pMaxBufferDuration) { return pThis->lpVtbl->GetBufferSizeLimits(pThis, pFormat, eventDriven, pMinBufferDuration, pMaxBufferDuration); }


// IAudioClient3
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IAudioClient3* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IAudioClient3* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IAudioClient3* pThis);

    // IAudioClient
    HRESULT (STDMETHODCALLTYPE * Initialize)       (mal_IAudioClient3* pThis, MAL_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MAL_REFERENCE_TIME bufferDuration, MAL_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
    HRESULT (STDMETHODCALLTYPE * GetBufferSize)    (mal_IAudioClient3* pThis, mal_uint32* pNumBufferFrames);
    HRESULT (STDMETHODCALLTYPE * GetStreamLatency) (mal_IAudioClient3* pThis, MAL_REFERENCE_TIME* pLatency);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPadding)(mal_IAudioClient3* pThis, mal_uint32* pNumPaddingFrames);
    HRESULT (STDMETHODCALLTYPE * IsFormatSupported)(mal_IAudioClient3* pThis, MAL_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch);
    HRESULT (STDMETHODCALLTYPE * GetMixFormat)     (mal_IAudioClient3* pThis, WAVEFORMATEX** ppDeviceFormat);
    HRESULT (STDMETHODCALLTYPE * GetDevicePeriod)  (mal_IAudioClient3* pThis, MAL_REFERENCE_TIME* pDefaultDevicePeriod, MAL_REFERENCE_TIME* pMinimumDevicePeriod);
    HRESULT (STDMETHODCALLTYPE * Start)            (mal_IAudioClient3* pThis);
    HRESULT (STDMETHODCALLTYPE * Stop)             (mal_IAudioClient3* pThis);
    HRESULT (STDMETHODCALLTYPE * Reset)            (mal_IAudioClient3* pThis);
    HRESULT (STDMETHODCALLTYPE * SetEventHandle)   (mal_IAudioClient3* pThis, HANDLE eventHandle);
    HRESULT (STDMETHODCALLTYPE * GetService)       (mal_IAudioClient3* pThis, const IID* const riid, void** pp);

    // IAudioClient2
    HRESULT (STDMETHODCALLTYPE * IsOffloadCapable)   (mal_IAudioClient3* pThis, MAL_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable);
    HRESULT (STDMETHODCALLTYPE * SetClientProperties)(mal_IAudioClient3* pThis, const mal_AudioClientProperties* pProperties);
    HRESULT (STDMETHODCALLTYPE * GetBufferSizeLimits)(mal_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MAL_REFERENCE_TIME* pMinBufferDuration, MAL_REFERENCE_TIME* pMaxBufferDuration);

    // IAudioClient3
    HRESULT (STDMETHODCALLTYPE * GetSharedModeEnginePeriod)       (mal_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, UINT32* pDefaultPeriodInFrames, UINT32* pFundamentalPeriodInFrames, UINT32* pMinPeriodInFrames, UINT32* pMaxPeriodInFrames);
    HRESULT (STDMETHODCALLTYPE * GetCurrentSharedModeEnginePeriod)(mal_IAudioClient3* pThis, WAVEFORMATEX** ppFormat, UINT32* pCurrentPeriodInFrames);
    HRESULT (STDMETHODCALLTYPE * InitializeSharedAudioStream)     (mal_IAudioClient3* pThis, DWORD streamFlags, UINT32 periodInFrames, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
} mal_IAudioClient3Vtbl;
struct mal_IAudioClient3
{
    mal_IAudioClient3Vtbl* lpVtbl;
};
HRESULT mal_IAudioClient3_QueryInterface(mal_IAudioClient3* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IAudioClient3_AddRef(mal_IAudioClient3* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IAudioClient3_Release(mal_IAudioClient3* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IAudioClient3_Initialize(mal_IAudioClient3* pThis, MAL_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MAL_REFERENCE_TIME bufferDuration, MAL_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid) { return pThis->lpVtbl->Initialize(pThis, shareMode, streamFlags, bufferDuration, periodicity, pFormat, pAudioSessionGuid); }
HRESULT mal_IAudioClient3_GetBufferSize(mal_IAudioClient3* pThis, mal_uint32* pNumBufferFrames)                { return pThis->lpVtbl->GetBufferSize(pThis, pNumBufferFrames); }
HRESULT mal_IAudioClient3_GetStreamLatency(mal_IAudioClient3* pThis, MAL_REFERENCE_TIME* pLatency)             { return pThis->lpVtbl->GetStreamLatency(pThis, pLatency); }
HRESULT mal_IAudioClient3_GetCurrentPadding(mal_IAudioClient3* pThis, mal_uint32* pNumPaddingFrames)           { return pThis->lpVtbl->GetCurrentPadding(pThis, pNumPaddingFrames); }
HRESULT mal_IAudioClient3_IsFormatSupported(mal_IAudioClient3* pThis, MAL_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch) { return pThis->lpVtbl->IsFormatSupported(pThis, shareMode, pFormat, ppClosestMatch); }
HRESULT mal_IAudioClient3_GetMixFormat(mal_IAudioClient3* pThis, WAVEFORMATEX** ppDeviceFormat)            { return pThis->lpVtbl->GetMixFormat(pThis, ppDeviceFormat); }
HRESULT mal_IAudioClient3_GetDevicePeriod(mal_IAudioClient3* pThis, MAL_REFERENCE_TIME* pDefaultDevicePeriod, MAL_REFERENCE_TIME* pMinimumDevicePeriod) { return pThis->lpVtbl->GetDevicePeriod(pThis, pDefaultDevicePeriod, pMinimumDevicePeriod); }
HRESULT mal_IAudioClient3_Start(mal_IAudioClient3* pThis)                                                  { return pThis->lpVtbl->Start(pThis); }
HRESULT mal_IAudioClient3_Stop(mal_IAudioClient3* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
HRESULT mal_IAudioClient3_Reset(mal_IAudioClient3* pThis)                                                  { return pThis->lpVtbl->Reset(pThis); }
HRESULT mal_IAudioClient3_SetEventHandle(mal_IAudioClient3* pThis, HANDLE eventHandle)                     { return pThis->lpVtbl->SetEventHandle(pThis, eventHandle); }
HRESULT mal_IAudioClient3_GetService(mal_IAudioClient3* pThis, const IID* const riid, void** pp)           { return pThis->lpVtbl->GetService(pThis, riid, pp); }
HRESULT mal_IAudioClient3_IsOffloadCapable(mal_IAudioClient3* pThis, MAL_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable) { return pThis->lpVtbl->IsOffloadCapable(pThis, category, pOffloadCapable); }
HRESULT mal_IAudioClient3_SetClientProperties(mal_IAudioClient3* pThis, const mal_AudioClientProperties* pProperties)           { return pThis->lpVtbl->SetClientProperties(pThis, pProperties); }
HRESULT mal_IAudioClient3_GetBufferSizeLimits(mal_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MAL_REFERENCE_TIME* pMinBufferDuration, MAL_REFERENCE_TIME* pMaxBufferDuration) { return pThis->lpVtbl->GetBufferSizeLimits(pThis, pFormat, eventDriven, pMinBufferDuration, pMaxBufferDuration); }
HRESULT mal_IAudioClient3_GetSharedModeEnginePeriod(mal_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, UINT32* pDefaultPeriodInFrames, UINT32* pFundamentalPeriodInFrames, UINT32* pMinPeriodInFrames, UINT32* pMaxPeriodInFrames) { return pThis->lpVtbl->GetSharedModeEnginePeriod(pThis, pFormat, pDefaultPeriodInFrames, pFundamentalPeriodInFrames, pMinPeriodInFrames, pMaxPeriodInFrames); }
HRESULT mal_IAudioClient3_GetCurrentSharedModeEnginePeriod(mal_IAudioClient3* pThis, WAVEFORMATEX** ppFormat, UINT32* pCurrentPeriodInFrames) { return pThis->lpVtbl->GetCurrentSharedModeEnginePeriod(pThis, ppFormat, pCurrentPeriodInFrames); }
HRESULT mal_IAudioClient3_InitializeSharedAudioStream(mal_IAudioClient3* pThis, DWORD streamFlags, UINT32 periodInFrames, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGUID) { return pThis->lpVtbl->InitializeSharedAudioStream(pThis, streamFlags, periodInFrames, pFormat, pAudioSessionGUID); }


// IAudioRenderClient
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IAudioRenderClient* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IAudioRenderClient* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IAudioRenderClient* pThis);

    // IAudioRenderClient
    HRESULT (STDMETHODCALLTYPE * GetBuffer)    (mal_IAudioRenderClient* pThis, mal_uint32 numFramesRequested, BYTE** ppData);
    HRESULT (STDMETHODCALLTYPE * ReleaseBuffer)(mal_IAudioRenderClient* pThis, mal_uint32 numFramesWritten, DWORD dwFlags);
} mal_IAudioRenderClientVtbl;
struct mal_IAudioRenderClient
{
    mal_IAudioRenderClientVtbl* lpVtbl;
};
HRESULT mal_IAudioRenderClient_QueryInterface(mal_IAudioRenderClient* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IAudioRenderClient_AddRef(mal_IAudioRenderClient* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IAudioRenderClient_Release(mal_IAudioRenderClient* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IAudioRenderClient_GetBuffer(mal_IAudioRenderClient* pThis, mal_uint32 numFramesRequested, BYTE** ppData)    { return pThis->lpVtbl->GetBuffer(pThis, numFramesRequested, ppData); }
HRESULT mal_IAudioRenderClient_ReleaseBuffer(mal_IAudioRenderClient* pThis, mal_uint32 numFramesWritten, DWORD dwFlags)  { return pThis->lpVtbl->ReleaseBuffer(pThis, numFramesWritten, dwFlags); }


// IAudioCaptureClient
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IAudioCaptureClient* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IAudioCaptureClient* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IAudioCaptureClient* pThis);

    // IAudioRenderClient
    HRESULT (STDMETHODCALLTYPE * GetBuffer)        (mal_IAudioCaptureClient* pThis, BYTE** ppData, mal_uint32* pNumFramesToRead, DWORD* pFlags, mal_uint64* pDevicePosition, mal_uint64* pQPCPosition);
    HRESULT (STDMETHODCALLTYPE * ReleaseBuffer)    (mal_IAudioCaptureClient* pThis, mal_uint32 numFramesRead);
    HRESULT (STDMETHODCALLTYPE * GetNextPacketSize)(mal_IAudioCaptureClient* pThis, mal_uint32* pNumFramesInNextPacket);
} mal_IAudioCaptureClientVtbl;
struct mal_IAudioCaptureClient
{
    mal_IAudioCaptureClientVtbl* lpVtbl;
};
HRESULT mal_IAudioCaptureClient_QueryInterface(mal_IAudioCaptureClient* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IAudioCaptureClient_AddRef(mal_IAudioCaptureClient* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IAudioCaptureClient_Release(mal_IAudioCaptureClient* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IAudioCaptureClient_GetBuffer(mal_IAudioCaptureClient* pThis, BYTE** ppData, mal_uint32* pNumFramesToRead, DWORD* pFlags, mal_uint64* pDevicePosition, mal_uint64* pQPCPosition) { return pThis->lpVtbl->GetBuffer(pThis, ppData, pNumFramesToRead, pFlags, pDevicePosition, pQPCPosition); }
HRESULT mal_IAudioCaptureClient_ReleaseBuffer(mal_IAudioCaptureClient* pThis, mal_uint32 numFramesRead)                    { return pThis->lpVtbl->ReleaseBuffer(pThis, numFramesRead); }
HRESULT mal_IAudioCaptureClient_GetNextPacketSize(mal_IAudioCaptureClient* pThis, mal_uint32* pNumFramesInNextPacket)      { return pThis->lpVtbl->GetNextPacketSize(pThis, pNumFramesInNextPacket); }

#ifndef MAL_WIN32_DESKTOP
#include <mmdeviceapi.h>
typedef struct mal_completion_handler_uwp mal_completion_handler_uwp;

typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_completion_handler_uwp* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_completion_handler_uwp* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_completion_handler_uwp* pThis);

    // IActivateAudioInterfaceCompletionHandler
    HRESULT (STDMETHODCALLTYPE * ActivateCompleted)(mal_completion_handler_uwp* pThis, mal_IActivateAudioInterfaceAsyncOperation* pActivateOperation);
} mal_completion_handler_uwp_vtbl;
struct mal_completion_handler_uwp
{
    mal_completion_handler_uwp_vtbl* lpVtbl;
    mal_uint32 counter;
    HANDLE hEvent;
};

HRESULT STDMETHODCALLTYPE mal_completion_handler_uwp_QueryInterface(mal_completion_handler_uwp* pThis, const IID* const riid, void** ppObject)
{
    // We need to "implement" IAgileObject which is just an indicator that's used internally by WASAPI for some multithreading management. To
    // "implement" this, we just make sure we return pThis when the IAgileObject is requested.
    if (!mal_is_guid_equal(riid, &MAL_IID_IUnknown) && !mal_is_guid_equal(riid, &MAL_IID_IActivateAudioInterfaceCompletionHandler) && !mal_is_guid_equal(riid, &MAL_IID_IAgileObject)) {
        *ppObject = NULL;
        return E_NOINTERFACE;
    }

    // Getting here means the IID is IUnknown or IMMNotificationClient.
    *ppObject = (void*)pThis;
    ((mal_completion_handler_uwp_vtbl*)pThis->lpVtbl)->AddRef(pThis);
    return S_OK;
}

ULONG STDMETHODCALLTYPE mal_completion_handler_uwp_AddRef(mal_completion_handler_uwp* pThis)
{
    return (ULONG)mal_atomic_increment_32(&pThis->counter);
}

ULONG STDMETHODCALLTYPE mal_completion_handler_uwp_Release(mal_completion_handler_uwp* pThis)
{
    mal_uint32 newRefCount = mal_atomic_decrement_32(&pThis->counter);
    if (newRefCount == 0) {
        return 0;   // We don't free anything here because we never allocate the object on the heap.
    }

    return (ULONG)newRefCount;
}

HRESULT STDMETHODCALLTYPE mal_completion_handler_uwp_ActivateCompleted(mal_completion_handler_uwp* pThis, mal_IActivateAudioInterfaceAsyncOperation* pActivateOperation)
{
    (void)pActivateOperation;
    SetEvent(pThis->hEvent);
    return S_OK;
}


static mal_completion_handler_uwp_vtbl g_malCompletionHandlerVtblInstance = {
    mal_completion_handler_uwp_QueryInterface,
    mal_completion_handler_uwp_AddRef,
    mal_completion_handler_uwp_Release,
    mal_completion_handler_uwp_ActivateCompleted
};

mal_result mal_completion_handler_uwp_init(mal_completion_handler_uwp* pHandler)
{
    mal_assert(pHandler != NULL);
    mal_zero_object(pHandler);

    pHandler->lpVtbl = &g_malCompletionHandlerVtblInstance;
    pHandler->counter = 1;
    pHandler->hEvent = CreateEventA(NULL, FALSE, FALSE, NULL);
    if (pHandler->hEvent == NULL) {
        return MAL_ERROR;
    }

    return MAL_SUCCESS;
}

void mal_completion_handler_uwp_uninit(mal_completion_handler_uwp* pHandler)
{
    if (pHandler->hEvent != NULL) {
        CloseHandle(pHandler->hEvent);
    }
}

void mal_completion_handler_uwp_wait(mal_completion_handler_uwp* pHandler)
{
    WaitForSingleObject(pHandler->hEvent, INFINITE);
}
#endif  // !MAL_WIN32_DESKTOP

// We need a virtual table for our notification client object that's used for detecting changes to the default device.
#ifdef MAL_WIN32_DESKTOP
HRESULT STDMETHODCALLTYPE mal_IMMNotificationClient_QueryInterface(mal_IMMNotificationClient* pThis, const IID* const riid, void** ppObject)
{
    // We care about two interfaces - IUnknown and IMMNotificationClient. If the requested IID is something else
    // we just return E_NOINTERFACE. Otherwise we need to increment the reference counter and return S_OK.
    if (!mal_is_guid_equal(riid, &MAL_IID_IUnknown) && !mal_is_guid_equal(riid, &MAL_IID_IMMNotificationClient)) {
        *ppObject = NULL;
        return E_NOINTERFACE;
    }

    // Getting here means the IID is IUnknown or IMMNotificationClient.
    *ppObject = (void*)pThis;
    ((mal_IMMNotificationClientVtbl*)pThis->lpVtbl)->AddRef(pThis);
    return S_OK;
}

ULONG STDMETHODCALLTYPE mal_IMMNotificationClient_AddRef(mal_IMMNotificationClient* pThis)
{
    return (ULONG)mal_atomic_increment_32(&pThis->counter);
}

ULONG STDMETHODCALLTYPE mal_IMMNotificationClient_Release(mal_IMMNotificationClient* pThis)
{
    mal_uint32 newRefCount = mal_atomic_decrement_32(&pThis->counter);
    if (newRefCount == 0) {
        return 0;   // We don't free anything here because we never allocate the object on the heap.
    }

    return (ULONG)newRefCount;
}


HRESULT STDMETHODCALLTYPE mal_IMMNotificationClient_OnDeviceStateChanged(mal_IMMNotificationClient* pThis, LPCWSTR pDeviceID, DWORD dwNewState)
{
#ifdef MAL_DEBUG_OUTPUT
    printf("IMMNotificationClient_OnDeviceStateChanged(pDeviceID=%S, dwNewState=%u)\n", pDeviceID, (unsigned int)dwNewState);
#endif

    (void)pThis;
    (void)pDeviceID;
    (void)dwNewState;
    return S_OK;
}

HRESULT STDMETHODCALLTYPE mal_IMMNotificationClient_OnDeviceAdded(mal_IMMNotificationClient* pThis, LPCWSTR pDeviceID)
{
#ifdef MAL_DEBUG_OUTPUT
    printf("IMMNotificationClient_OnDeviceAdded(pDeviceID=%S)\n", pDeviceID);
#endif

    // We don't need to worry about this event for our purposes.
    (void)pThis;
    (void)pDeviceID;
    return S_OK;
}

HRESULT STDMETHODCALLTYPE mal_IMMNotificationClient_OnDeviceRemoved(mal_IMMNotificationClient* pThis, LPCWSTR pDeviceID)
{
#ifdef MAL_DEBUG_OUTPUT
    printf("IMMNotificationClient_OnDeviceRemoved(pDeviceID=%S)\n", pDeviceID);
#endif

    // We don't need to worry about this event for our purposes.
    (void)pThis;
    (void)pDeviceID;
    return S_OK;
}

HRESULT STDMETHODCALLTYPE mal_IMMNotificationClient_OnDefaultDeviceChanged(mal_IMMNotificationClient* pThis, mal_EDataFlow dataFlow, mal_ERole role, LPCWSTR pDefaultDeviceID)
{
#ifdef MAL_DEBUG_OUTPUT
    printf("IMMNotificationClient_OnDefaultDeviceChanged(dataFlow=%d, role=%d, pDefaultDeviceID=%S)\n", dataFlow, role, pDefaultDeviceID);
#endif

    // We only ever use the eConsole role in mini_al.
    if (role != mal_eConsole) {
        return S_OK;
    }

    // We only care about devices with the same data flow and role as the current device.
    if ((pThis->pDevice->type == mal_device_type_playback && dataFlow != mal_eRender ) ||
        (pThis->pDevice->type == mal_device_type_capture  && dataFlow != mal_eCapture)) {
        return S_OK;
    }

    // Not currently supporting automatic stream routing in exclusive mode. This is not working correctly on my machine due to
    // AUDCLNT_E_DEVICE_IN_USE errors when reinitializing the device. If this is a bug in mini_al, we can try re-enabling this once
    // it's fixed.
    if (pThis->pDevice->exclusiveMode) {
        return S_OK;
    }

    // We don't change the device here - we change it in the worker thread to keep synchronization simple. To this I'm just setting a flag to
    // indicate that the default device has changed.
    mal_atomic_exchange_32(&pThis->pDevice->wasapi.hasDefaultDeviceChanged, MAL_TRUE);
    SetEvent(pThis->pDevice->wasapi.hBreakEvent);   // <-- The main loop will be waiting on some events. We want to break from this wait ASAP so we can change the device as quickly as possible.

    
    (void)pDefaultDeviceID;
    return S_OK;
}

HRESULT STDMETHODCALLTYPE mal_IMMNotificationClient_OnPropertyValueChanged(mal_IMMNotificationClient* pThis, LPCWSTR pDeviceID, const PROPERTYKEY key)
{
#ifdef MAL_DEBUG_OUTPUT
    printf("IMMNotificationClient_OnPropertyValueChanged(pDeviceID=%S)\n", pDeviceID);
#endif

    (void)pThis;
    (void)pDeviceID;
    (void)key;
    return S_OK;
}

static mal_IMMNotificationClientVtbl g_malNotificationCientVtbl = {
    mal_IMMNotificationClient_QueryInterface,
    mal_IMMNotificationClient_AddRef,
    mal_IMMNotificationClient_Release,
    mal_IMMNotificationClient_OnDeviceStateChanged,
    mal_IMMNotificationClient_OnDeviceAdded,
    mal_IMMNotificationClient_OnDeviceRemoved,
    mal_IMMNotificationClient_OnDefaultDeviceChanged,
    mal_IMMNotificationClient_OnPropertyValueChanged
};
#endif  // MAL_WIN32_DESKTOP

mal_bool32 mal_context_is_device_id_equal__wasapi(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return memcmp(pID0->wasapi, pID1->wasapi, sizeof(pID0->wasapi)) == 0;
}

void mal_set_device_info_from_WAVEFORMATEX(const WAVEFORMATEX* pWF, mal_device_info* pInfo)
{
    mal_assert(pWF != NULL);
    mal_assert(pInfo != NULL);

    pInfo->formatCount   = 1;
    pInfo->formats[0]    = mal_format_from_WAVEFORMATEX(pWF);
    pInfo->minChannels   = pWF->nChannels;
    pInfo->maxChannels   = pWF->nChannels;
    pInfo->minSampleRate = pWF->nSamplesPerSec;
    pInfo->maxSampleRate = pWF->nSamplesPerSec;
}

#ifndef MAL_WIN32_DESKTOP
mal_result mal_context_get_IAudioClient_UWP__wasapi(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_IAudioClient** ppAudioClient, mal_IUnknown** ppActivatedInterface)
{
    mal_assert(pContext != NULL);
    mal_assert(ppAudioClient != NULL);

    mal_IActivateAudioInterfaceAsyncOperation *pAsyncOp = NULL;
    mal_completion_handler_uwp completionHandler;

    IID iid;
    if (pDeviceID != NULL) {
        mal_copy_memory(&iid, pDeviceID->wasapi, sizeof(iid));
    } else {
        if (deviceType == mal_device_type_playback) {
            iid = MAL_IID_DEVINTERFACE_AUDIO_RENDER;
        } else {
            iid = MAL_IID_DEVINTERFACE_AUDIO_CAPTURE;
        }
    }

    LPOLESTR iidStr;
#if defined(__cplusplus)
    HRESULT hr = StringFromIID(iid, &iidStr);
#else
    HRESULT hr = StringFromIID(&iid, &iidStr);
#endif
    if (FAILED(hr)) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to convert device IID to string for ActivateAudioInterfaceAsync(). Out of memory.", MAL_OUT_OF_MEMORY);
    }

    mal_result result = mal_completion_handler_uwp_init(&completionHandler);
    if (result != MAL_SUCCESS) {
        mal_CoTaskMemFree(pContext, iidStr);
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to create event for waiting for ActivateAudioInterfaceAsync().", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

#if defined(__cplusplus)
    hr = ActivateAudioInterfaceAsync(iidStr, MAL_IID_IAudioClient, NULL, (IActivateAudioInterfaceCompletionHandler*)&completionHandler, (IActivateAudioInterfaceAsyncOperation**)&pAsyncOp);
#else
    hr = ActivateAudioInterfaceAsync(iidStr, &MAL_IID_IAudioClient, NULL, (IActivateAudioInterfaceCompletionHandler*)&completionHandler, (IActivateAudioInterfaceAsyncOperation**)&pAsyncOp);
#endif
    if (FAILED(hr)) {
        mal_completion_handler_uwp_uninit(&completionHandler);
        mal_CoTaskMemFree(pContext, iidStr);
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] ActivateAudioInterfaceAsync() failed.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    mal_CoTaskMemFree(pContext, iidStr);

    // Wait for the async operation for finish.
    mal_completion_handler_uwp_wait(&completionHandler);
    mal_completion_handler_uwp_uninit(&completionHandler);

    HRESULT activateResult;
    mal_IUnknown* pActivatedInterface;
    hr = mal_IActivateAudioInterfaceAsyncOperation_GetActivateResult(pAsyncOp, &activateResult, &pActivatedInterface);
    mal_IActivateAudioInterfaceAsyncOperation_Release(pAsyncOp);

    if (FAILED(hr) || FAILED(activateResult)) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to activate device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    // Here is where we grab the IAudioClient interface.
    hr = mal_IUnknown_QueryInterface(pActivatedInterface, &MAL_IID_IAudioClient, (void**)ppAudioClient);
    if (FAILED(hr)) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to query IAudioClient interface.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    if (ppActivatedInterface) {
        *ppActivatedInterface = pActivatedInterface;
    } else {
        mal_IUnknown_Release(pActivatedInterface);
    }

    return MAL_SUCCESS;
}
#endif

mal_result mal_context_get_device_info_from_IAudioClient__wasapi(mal_context* pContext, /*mal_IMMDevice**/void* pMMDevice, mal_IAudioClient* pAudioClient, mal_share_mode shareMode, mal_device_info* pInfo)
{
    mal_assert(pAudioClient != NULL);
    mal_assert(pInfo != NULL);

    // We use a different technique to retrieve the device information depending on whether or not we are using shared or exclusive mode.
    if (shareMode == mal_share_mode_shared) {
        // Shared Mode. We use GetMixFormat() here.
        WAVEFORMATEX* pWF = NULL;
        HRESULT hr = mal_IAudioClient_GetMixFormat((mal_IAudioClient*)pAudioClient, (WAVEFORMATEX**)&pWF);
        if (SUCCEEDED(hr)) {
            mal_set_device_info_from_WAVEFORMATEX(pWF, pInfo);
            return MAL_SUCCESS;
        } else {
            return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve mix format for device info retrieval.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    } else {
        // Exlcusive Mode. We repeatedly call IsFormatSupported() here. This is not currently support on UWP.
#ifdef MAL_WIN32_DESKTOP
        // The first thing to do is get the format from PKEY_AudioEngine_DeviceFormat. This should give us a channel count we assume is
        // correct which will simplify our searching.
        mal_IPropertyStore *pProperties;
        HRESULT hr = mal_IMMDevice_OpenPropertyStore((mal_IMMDevice*)pMMDevice, STGM_READ, &pProperties);
        if (SUCCEEDED(hr)) {
            PROPVARIANT var;
            mal_PropVariantInit(&var);

            hr = mal_IPropertyStore_GetValue(pProperties, &MAL_PKEY_AudioEngine_DeviceFormat, &var);
            if (SUCCEEDED(hr)) {
                WAVEFORMATEX* pWF = (WAVEFORMATEX*)var.blob.pBlobData;
                mal_set_device_info_from_WAVEFORMATEX(pWF, pInfo);

                // In my testing, the format returned by PKEY_AudioEngine_DeviceFormat is suitable for exclusive mode so we check this format
                // first. If this fails, fall back to a search.
                hr = mal_IAudioClient_IsFormatSupported((mal_IAudioClient*)pAudioClient, MAL_AUDCLNT_SHAREMODE_EXCLUSIVE, pWF, NULL);
                mal_PropVariantClear(pContext, &var);

                if (FAILED(hr)) {
                    // The format returned by PKEY_AudioEngine_DeviceFormat is not supported, so fall back to a search. We assume the channel
                    // count returned by MAL_PKEY_AudioEngine_DeviceFormat is valid and correct. For simplicity we're only returning one format.
                    mal_uint32 channels = pInfo->minChannels;

                    mal_format formatsToSearch[] = {
                        mal_format_s16,
                        mal_format_s24,
                        //mal_format_s24_32,
                        mal_format_f32,
                        mal_format_s32,
                        mal_format_u8
                    };

                    mal_channel defaultChannelMap[MAL_MAX_CHANNELS];
                    mal_get_standard_channel_map(mal_standard_channel_map_microsoft, channels, defaultChannelMap);

                    WAVEFORMATEXTENSIBLE wf;
                    mal_zero_object(&wf);
                    wf.Format.cbSize     = sizeof(wf);
                    wf.Format.wFormatTag = WAVE_FORMAT_EXTENSIBLE;
                    wf.Format.nChannels  = (WORD)channels;
                    wf.dwChannelMask     = mal_channel_map_to_channel_mask__win32(defaultChannelMap, channels);

                    mal_bool32 found = MAL_FALSE;
                    for (mal_uint32 iFormat = 0; iFormat < mal_countof(formatsToSearch); ++iFormat) {
                        mal_format format = formatsToSearch[iFormat];

                        wf.Format.wBitsPerSample       = (WORD)mal_get_bytes_per_sample(format)*8;
                        wf.Format.nBlockAlign          = (wf.Format.nChannels * wf.Format.wBitsPerSample) / 8;
                        wf.Format.nAvgBytesPerSec      = wf.Format.nBlockAlign * wf.Format.nSamplesPerSec;
                        wf.Samples.wValidBitsPerSample = /*(format == mal_format_s24_32) ? 24 :*/ wf.Format.wBitsPerSample;
                        if (format == mal_format_f32) {
                            wf.SubFormat = MAL_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT;
                        } else {
                            wf.SubFormat = MAL_GUID_KSDATAFORMAT_SUBTYPE_PCM;
                        }

                        for (mal_uint32 iSampleRate = 0; iSampleRate < mal_countof(g_malStandardSampleRatePriorities); ++iSampleRate) {
                            wf.Format.nSamplesPerSec = g_malStandardSampleRatePriorities[iSampleRate];

                            hr = mal_IAudioClient_IsFormatSupported((mal_IAudioClient*)pAudioClient, MAL_AUDCLNT_SHAREMODE_EXCLUSIVE, (WAVEFORMATEX*)&wf, NULL);
                            if (SUCCEEDED(hr)) {
                                mal_set_device_info_from_WAVEFORMATEX((WAVEFORMATEX*)&wf, pInfo);
                                found = MAL_TRUE;
                                break;
                            }
                        }

                        if (found) {
                            break;
                        }
                    }

                    if (!found) {
                        mal_IPropertyStore_Release(pProperties);
                        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to find suitable device format for device info retrieval.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
                    }
                }
            } else {
                mal_IPropertyStore_Release(pProperties);
                return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve device format for device info retrieval.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
            }
        } else {
            return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to open property store for device info retrieval.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        return MAL_SUCCESS;
#else
        // Exclusive mode not fully supported in UWP right now.
        return MAL_ERROR;
#endif
    }
}

#ifdef MAL_WIN32_DESKTOP
mal_result mal_context_get_MMDevice__wasapi(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_IMMDevice** ppMMDevice)
{
    mal_assert(pContext != NULL);
    mal_assert(ppMMDevice != NULL);

    mal_IMMDeviceEnumerator* pDeviceEnumerator;
    HRESULT hr = mal_CoCreateInstance(pContext, MAL_CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, MAL_IID_IMMDeviceEnumerator, (void**)&pDeviceEnumerator);
    if (FAILED(hr)) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to create IMMDeviceEnumerator.", MAL_FAILED_TO_INIT_BACKEND);
    }

    if (pDeviceID == NULL) {
        hr = mal_IMMDeviceEnumerator_GetDefaultAudioEndpoint(pDeviceEnumerator, (deviceType == mal_device_type_playback) ? mal_eRender : mal_eCapture, mal_eConsole, ppMMDevice);
    } else {
        hr = mal_IMMDeviceEnumerator_GetDevice(pDeviceEnumerator, pDeviceID->wasapi, ppMMDevice);
    }

    mal_IMMDeviceEnumerator_Release(pDeviceEnumerator);
    if (FAILED(hr)) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve IMMDevice.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info_from_MMDevice__wasapi(mal_context* pContext, mal_IMMDevice* pMMDevice, mal_share_mode shareMode, mal_bool32 onlySimpleInfo, mal_device_info* pInfo)
{
    mal_assert(pContext != NULL);
    mal_assert(pMMDevice != NULL);
    mal_assert(pInfo != NULL);

    // ID.
    LPWSTR id;
    HRESULT hr = mal_IMMDevice_GetId(pMMDevice, &id);
    if (SUCCEEDED(hr)) {
        size_t idlen = wcslen(id);
        if (idlen+1 > mal_countof(pInfo->id.wasapi)) {
            mal_CoTaskMemFree(pContext, id);
            mal_assert(MAL_FALSE);  // NOTE: If this is triggered, please report it. It means the format of the ID must haved change and is too long to fit in our fixed sized buffer.
            return MAL_ERROR;
        }

        mal_copy_memory(pInfo->id.wasapi, id, idlen * sizeof(wchar_t));
        pInfo->id.wasapi[idlen] = '\0';

        mal_CoTaskMemFree(pContext, id);
    }

    {
        mal_IPropertyStore *pProperties;
        hr = mal_IMMDevice_OpenPropertyStore(pMMDevice, STGM_READ, &pProperties);
        if (SUCCEEDED(hr)) {
            PROPVARIANT var;

            // Description / Friendly Name
            mal_PropVariantInit(&var);
            hr = mal_IPropertyStore_GetValue(pProperties, &MAL_PKEY_Device_FriendlyName, &var);
            if (SUCCEEDED(hr)) {
                WideCharToMultiByte(CP_UTF8, 0, var.pwszVal, -1, pInfo->name, sizeof(pInfo->name), 0, FALSE);
                mal_PropVariantClear(pContext, &var);
            }

            mal_IPropertyStore_Release(pProperties);
        }
    }

    // Format
    if (!onlySimpleInfo) {
        mal_IAudioClient* pAudioClient;
        hr = mal_IMMDevice_Activate(pMMDevice, &MAL_IID_IAudioClient, CLSCTX_ALL, NULL, (void**)&pAudioClient);
        if (SUCCEEDED(hr)) {
            mal_result result = mal_context_get_device_info_from_IAudioClient__wasapi(pContext, pMMDevice, pAudioClient, shareMode, pInfo);
            
            mal_IAudioClient_Release(pAudioClient);
            return result;
        } else {
            return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to activate audio client for device info retrieval.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_context_enumerate_device_collection__wasapi(mal_context* pContext, mal_IMMDeviceCollection* pDeviceCollection, mal_device_type deviceType, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    UINT deviceCount;
    HRESULT hr = mal_IMMDeviceCollection_GetCount(pDeviceCollection, &deviceCount);
    if (FAILED(hr)) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to get playback device count.", MAL_NO_DEVICE);
    }

    for (mal_uint32 iDevice = 0; iDevice < deviceCount; ++iDevice) {
        mal_device_info deviceInfo;
        mal_zero_object(&deviceInfo);

        mal_IMMDevice* pMMDevice;
        hr = mal_IMMDeviceCollection_Item(pDeviceCollection, iDevice, &pMMDevice);
        if (SUCCEEDED(hr)) {
            mal_result result = mal_context_get_device_info_from_MMDevice__wasapi(pContext, pMMDevice, mal_share_mode_shared, MAL_TRUE, &deviceInfo);   // MAL_TRUE = onlySimpleInfo.

            mal_IMMDevice_Release(pMMDevice);
            if (result == MAL_SUCCESS) {
                mal_bool32 cbResult = callback(pContext, deviceType, &deviceInfo, pUserData);
                if (cbResult == MAL_FALSE) {
                    break;
                }
            }
        }
    }

    return MAL_SUCCESS;
}
#endif

mal_result mal_context_enumerate_devices__wasapi(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    // Different enumeration for desktop and UWP.
#ifdef MAL_WIN32_DESKTOP
    // Desktop
    mal_IMMDeviceEnumerator* pDeviceEnumerator;
    HRESULT hr = mal_CoCreateInstance(pContext, MAL_CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, MAL_IID_IMMDeviceEnumerator, (void**)&pDeviceEnumerator);
    if (FAILED(hr)) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to create device enumerator.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    mal_IMMDeviceCollection* pDeviceCollection;

    // Playback.
    hr = mal_IMMDeviceEnumerator_EnumAudioEndpoints(pDeviceEnumerator, mal_eRender, MAL_MM_DEVICE_STATE_ACTIVE, &pDeviceCollection);
    if (SUCCEEDED(hr)) {
        mal_context_enumerate_device_collection__wasapi(pContext, pDeviceCollection, mal_device_type_playback, callback, pUserData);
        mal_IMMDeviceCollection_Release(pDeviceCollection);
    }

    // Capture.
    hr = mal_IMMDeviceEnumerator_EnumAudioEndpoints(pDeviceEnumerator, mal_eCapture, MAL_MM_DEVICE_STATE_ACTIVE, &pDeviceCollection);
    if (SUCCEEDED(hr)) {
        mal_context_enumerate_device_collection__wasapi(pContext, pDeviceCollection, mal_device_type_capture, callback, pUserData);
        mal_IMMDeviceCollection_Release(pDeviceCollection);
    }

    mal_IMMDeviceEnumerator_Release(pDeviceEnumerator);
#else
    // UWP
    //
    // The MMDevice API is only supported on desktop applications. For now, while I'm still figuring out how to properly enumerate
    // over devices without using MMDevice, I'm restricting devices to defaults.
    //
    // Hint: DeviceInformation::FindAllAsync() with DeviceClass.AudioCapture/AudioRender. https://blogs.windows.com/buildingapps/2014/05/15/real-time-audio-in-windows-store-and-windows-phone-apps/
    if (callback) {
        mal_bool32 cbResult = MAL_TRUE;

        // Playback.
        if (cbResult) {
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
            cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
        }

        // Capture.
        if (cbResult) {
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
            cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
        }
    }
#endif

    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info__wasapi(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
#ifdef MAL_WIN32_DESKTOP
    mal_IMMDevice* pMMDevice = NULL;
    mal_result result = mal_context_get_MMDevice__wasapi(pContext, deviceType, pDeviceID, &pMMDevice);
    if (result != MAL_SUCCESS) {
        return result;
    }

    result = mal_context_get_device_info_from_MMDevice__wasapi(pContext, pMMDevice, shareMode, MAL_FALSE, pDeviceInfo);   // MAL_FALSE = !onlySimpleInfo.

    mal_IMMDevice_Release(pMMDevice);
    return result;
#else
    // UWP currently only uses default devices.
    if (deviceType == mal_device_type_playback) {
        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    } else {
        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    }

    // Not currently supporting exclusive mode on UWP.
    if (shareMode == mal_share_mode_exclusive) {
        return MAL_ERROR;
    }

    mal_IAudioClient* pAudioClient;
    mal_result result = mal_context_get_IAudioClient_UWP__wasapi(pContext, deviceType, pDeviceID, &pAudioClient, NULL);
    if (result != MAL_SUCCESS) {
        return result;
    }

    result = mal_context_get_device_info_from_IAudioClient__wasapi(pContext, NULL, pAudioClient, shareMode, pDeviceInfo);

    mal_IAudioClient_Release(pAudioClient);
    return result;
#endif
}

void mal_device_uninit__wasapi(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

#ifdef MAL_WIN32_DESKTOP
    if (pDevice->wasapi.pDeviceEnumerator) {
        ((mal_IMMDeviceEnumerator*)pDevice->wasapi.pDeviceEnumerator)->lpVtbl->UnregisterEndpointNotificationCallback((mal_IMMDeviceEnumerator*)pDevice->wasapi.pDeviceEnumerator, &pDevice->wasapi.notificationClient);
        mal_IMMDeviceEnumerator_Release((mal_IMMDeviceEnumerator*)pDevice->wasapi.pDeviceEnumerator);
    }
#endif

    if (pDevice->wasapi.pRenderClient) {
        mal_IAudioRenderClient_Release((mal_IAudioRenderClient*)pDevice->wasapi.pRenderClient);
    }
    if (pDevice->wasapi.pCaptureClient) {
        mal_IAudioCaptureClient_Release((mal_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
    }
    if (pDevice->wasapi.pAudioClient) {
        mal_IAudioClient_Release((mal_IAudioClient*)pDevice->wasapi.pAudioClient);
    }

    if (pDevice->wasapi.hEvent) {
        CloseHandle(pDevice->wasapi.hEvent);
    }
    if (pDevice->wasapi.hBreakEvent) {
        CloseHandle(pDevice->wasapi.hBreakEvent);
    }
}

typedef struct
{
    // Input.
    mal_format formatIn;
    mal_uint32 channelsIn;
    mal_uint32 sampleRateIn;
    mal_channel channelMapIn[MAL_MAX_CHANNELS];
    mal_uint32 bufferSizeInFramesIn;
    mal_uint32 bufferSizeInMillisecondsIn;
    mal_uint32 periodsIn;
    mal_bool32 usingDefaultFormat;
    mal_bool32 usingDefaultChannels;
    mal_bool32 usingDefaultSampleRate;
    mal_bool32 usingDefaultChannelMap;
    mal_share_mode shareMode;

    // Output.
    mal_IAudioClient* pAudioClient;
    mal_IAudioRenderClient* pRenderClient;
    mal_IAudioCaptureClient* pCaptureClient;
    mal_format formatOut;
    mal_uint32 channelsOut;
    mal_uint32 sampleRateOut;
    mal_channel channelMapOut[MAL_MAX_CHANNELS];
    mal_uint32 bufferSizeInFramesOut;
    mal_uint32 periodsOut;
    mal_bool32 exclusiveMode;
    char deviceName[256];
} mal_device_init_internal_data__wasapi;

mal_result mal_device_init_internal__wasapi(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, mal_device_init_internal_data__wasapi* pData)
{
    (void)pContext;

    mal_assert(pContext != NULL);
    mal_assert(pData != NULL);

    pData->pAudioClient = NULL;
    pData->pRenderClient = NULL;
    pData->pCaptureClient = NULL;
    

    HRESULT hr;
    mal_result result = MAL_SUCCESS;
    const char* errorMsg = "";
    MAL_AUDCLNT_SHAREMODE shareMode = MAL_AUDCLNT_SHAREMODE_SHARED;
    MAL_REFERENCE_TIME bufferDurationInMicroseconds;
    mal_bool32 wasInitializedUsingIAudioClient3 = MAL_FALSE;
    WAVEFORMATEXTENSIBLE wf;

#ifdef MAL_WIN32_DESKTOP
    mal_IMMDevice* pMMDevice = NULL;
    result = mal_context_get_MMDevice__wasapi(pContext, type, pDeviceID, &pMMDevice);
    if (result != MAL_SUCCESS) {
        goto done;
    }

    hr = mal_IMMDevice_Activate(pMMDevice, &MAL_IID_IAudioClient, CLSCTX_ALL, NULL, (void**)&pData->pAudioClient);
    if (FAILED(hr)) {
        errorMsg = "[WASAPI] Failed to activate device.", result = MAL_FAILED_TO_OPEN_BACKEND_DEVICE;
        goto done;
    }
#else
    mal_IUnknown* pActivatedInterface = NULL;
    result = mal_context_get_IAudioClient_UWP__wasapi(pContext, type, pDeviceID, &pData->pAudioClient, &pActivatedInterface);
    if (result != MAL_SUCCESS) {
        goto done;
    }
#endif

    // Try enabling hardware offloading.
    mal_IAudioClient2* pAudioClient2;
    hr = mal_IAudioClient_QueryInterface(pData->pAudioClient, &MAL_IID_IAudioClient2, (void**)&pAudioClient2);
    if (SUCCEEDED(hr)) {
        BOOL isHardwareOffloadingSupported = 0;
        hr = mal_IAudioClient2_IsOffloadCapable(pAudioClient2, MAL_AudioCategory_Other, &isHardwareOffloadingSupported);
        if (SUCCEEDED(hr) && isHardwareOffloadingSupported) {
            mal_AudioClientProperties clientProperties;
            mal_zero_object(&clientProperties);
            clientProperties.cbSize = sizeof(clientProperties);
            clientProperties.bIsOffload = 1;
            clientProperties.eCategory = MAL_AudioCategory_Other;
            mal_IAudioClient2_SetClientProperties(pAudioClient2, &clientProperties);
        }
    }


    // Here is where we try to determine the best format to use with the device. If the client if wanting exclusive mode, first try finding the best format for that. If this fails, fall back to shared mode.
    result = MAL_FORMAT_NOT_SUPPORTED;
    if (pData->shareMode == mal_share_mode_exclusive) {
    #ifdef MAL_WIN32_DESKTOP
        // In exclusive mode on desktop we always use the backend's native format.
        mal_IPropertyStore* pStore = NULL;
        hr = mal_IMMDevice_OpenPropertyStore(pMMDevice, STGM_READ, &pStore);
        if (SUCCEEDED(hr)) {
            PROPVARIANT prop;
            mal_PropVariantInit(&prop);
            hr = mal_IPropertyStore_GetValue(pStore, &MAL_PKEY_AudioEngine_DeviceFormat, &prop);
            if (SUCCEEDED(hr)) {
                WAVEFORMATEX* pActualFormat = (WAVEFORMATEX*)prop.blob.pBlobData;
                hr = mal_IAudioClient_IsFormatSupported((mal_IAudioClient*)pData->pAudioClient, MAL_AUDCLNT_SHAREMODE_EXCLUSIVE, pActualFormat, NULL);
                if (SUCCEEDED(hr)) {
                    mal_copy_memory(&wf, pActualFormat, sizeof(WAVEFORMATEXTENSIBLE));
                }

                mal_PropVariantClear(pContext, &prop);
            }

            mal_IPropertyStore_Release(pStore);
        }
    #else
        // I do not know how to query the device's native format on UWP so for now I'm just disabling support for
        // exclusive mode. The alternative is to enumerate over different formats and check IsFormatSupported()
        // until you find one that works.
        //
        // TODO: Add support for exclusive mode to UWP.
        hr = S_FALSE;
    #endif

        if (hr == S_OK) {
            shareMode = MAL_AUDCLNT_SHAREMODE_EXCLUSIVE;
            result = MAL_SUCCESS;
        }
    }

    // Fall back to shared mode if necessary.
    if (result != MAL_SUCCESS) {
        // In shared mode we are always using the format reported by the operating system.
        WAVEFORMATEXTENSIBLE* pNativeFormat = NULL;
        hr = mal_IAudioClient_GetMixFormat((mal_IAudioClient*)pData->pAudioClient, (WAVEFORMATEX**)&pNativeFormat);
        if (hr != S_OK) {
            result = MAL_FORMAT_NOT_SUPPORTED;
        } else {
            mal_copy_memory(&wf, pNativeFormat, sizeof(wf));
            result = MAL_SUCCESS;
        }

        mal_CoTaskMemFree(pContext, pNativeFormat);

        shareMode = MAL_AUDCLNT_SHAREMODE_SHARED;
    }

    // Return an error if we still haven't found a format.
    if (result != MAL_SUCCESS) {
        errorMsg = "[WASAPI] Failed to find best device mix format.", result = MAL_FORMAT_NOT_SUPPORTED;
        goto done;
    }

    pData->formatOut = mal_format_from_WAVEFORMATEX((WAVEFORMATEX*)&wf);
    pData->channelsOut = wf.Format.nChannels;
    pData->sampleRateOut = wf.Format.nSamplesPerSec;

    // Get the internal channel map based on the channel mask.
    mal_channel_mask_to_channel_map__win32(wf.dwChannelMask, pData->channelsOut, pData->channelMapOut);

    // If we're using a default buffer size we need to calculate it based on the efficiency of the system.
    pData->periodsOut = pData->periodsIn;
    pData->bufferSizeInFramesOut = pData->bufferSizeInFramesIn;
    if (pData->bufferSizeInFramesOut == 0) {
        pData->bufferSizeInFramesOut = mal_calculate_buffer_size_in_frames_from_milliseconds(pData->bufferSizeInMillisecondsIn, pData->sampleRateOut);
    }

    bufferDurationInMicroseconds = ((mal_uint64)pData->bufferSizeInFramesOut * 1000 * 1000) / pData->sampleRateOut;


    // Slightly different initialization for shared and exclusive modes. We try exclusive mode first, and if it fails, fall back to shared mode.
    if (shareMode == MAL_AUDCLNT_SHAREMODE_EXCLUSIVE) {
        // Exclusive.
        MAL_REFERENCE_TIME bufferDuration = bufferDurationInMicroseconds*10;

        // If the periodicy is too small, Initialize() will fail with AUDCLNT_E_INVALID_DEVICE_PERIOD. In this case we should just keep increasing
        // it and trying it again.
        hr = E_FAIL;
        for (;;) {
            hr = mal_IAudioClient_Initialize((mal_IAudioClient*)pData->pAudioClient, shareMode, MAL_AUDCLNT_STREAMFLAGS_EVENTCALLBACK, bufferDuration, bufferDuration, (WAVEFORMATEX*)&wf, NULL);
            if (hr == MAL_AUDCLNT_E_INVALID_DEVICE_PERIOD) {
                if (bufferDuration > 500*10000) {
                    break;
                } else {
                    if (bufferDuration == 0) {  // <-- Just a sanity check to prevent an infinit loop. Should never happen, but it makes me feel better.
                        break;
                    }

                    bufferDuration = bufferDuration * 2;
                    continue;
                }
            } else {
                break;
            }
        }
        
        if (hr == MAL_AUDCLNT_E_BUFFER_SIZE_NOT_ALIGNED) {
            UINT bufferSizeInFrames;
            hr = mal_IAudioClient_GetBufferSize((mal_IAudioClient*)pData->pAudioClient, &bufferSizeInFrames);
            if (SUCCEEDED(hr)) {
                bufferDuration = (MAL_REFERENCE_TIME)((10000.0 * 1000 / wf.Format.nSamplesPerSec * bufferSizeInFrames) + 0.5);

                // Unfortunately we need to release and re-acquire the audio client according to MSDN. Seems silly - why not just call IAudioClient_Initialize() again?!
                mal_IAudioClient_Release((mal_IAudioClient*)pData->pAudioClient);

            #ifdef MAL_WIN32_DESKTOP
                hr = mal_IMMDevice_Activate(pMMDevice, &MAL_IID_IAudioClient, CLSCTX_ALL, NULL, (void**)&pData->pAudioClient);
            #else
                hr = mal_IUnknown_QueryInterface(pActivatedInterface, &MAL_IID_IAudioClient, (void**)&pData->pAudioClient);
            #endif

                if (SUCCEEDED(hr)) {
                    hr = mal_IAudioClient_Initialize((mal_IAudioClient*)pData->pAudioClient, shareMode, MAL_AUDCLNT_STREAMFLAGS_EVENTCALLBACK, bufferDuration, bufferDuration, (WAVEFORMATEX*)&wf, NULL);
                }
            }
        }

        if (FAILED(hr)) {
            // Failed to initialize in exclusive mode. We don't return an error here, but instead fall back to shared mode.
            shareMode = MAL_AUDCLNT_SHAREMODE_SHARED;
        }
    }

    if (shareMode == MAL_AUDCLNT_SHAREMODE_SHARED) {
        // Shared.

        // Low latency shared mode via IAudioClient3.
        mal_IAudioClient3* pAudioClient3 = NULL;
        hr = mal_IAudioClient_QueryInterface(pData->pAudioClient, &MAL_IID_IAudioClient3, (void**)&pAudioClient3);
        if (SUCCEEDED(hr)) {
            UINT32 defaultPeriodInFrames;
            UINT32 fundamentalPeriodInFrames;
            UINT32 minPeriodInFrames;
            UINT32 maxPeriodInFrames;
            hr = mal_IAudioClient3_GetSharedModeEnginePeriod(pAudioClient3, (WAVEFORMATEX*)&wf, &defaultPeriodInFrames, &fundamentalPeriodInFrames, &minPeriodInFrames, &maxPeriodInFrames);
            if (SUCCEEDED(hr)) {
                UINT32 desiredPeriodInFrames = pData->bufferSizeInFramesOut / pData->periodsOut;
                
                // Make sure the period size is a multiple of fundamentalPeriodInFrames.
                desiredPeriodInFrames = desiredPeriodInFrames / fundamentalPeriodInFrames;
                desiredPeriodInFrames = desiredPeriodInFrames * fundamentalPeriodInFrames;

                // The period needs to be clamped between minPeriodInFrames and maxPeriodInFrames.
                desiredPeriodInFrames = mal_clamp(desiredPeriodInFrames, minPeriodInFrames, maxPeriodInFrames);
                
                hr = mal_IAudioClient3_InitializeSharedAudioStream(pAudioClient3, MAL_AUDCLNT_STREAMFLAGS_EVENTCALLBACK, desiredPeriodInFrames, (WAVEFORMATEX*)&wf, NULL);
                if (SUCCEEDED(hr)) {
                    wasInitializedUsingIAudioClient3 = MAL_TRUE;
                    pData->bufferSizeInFramesOut = desiredPeriodInFrames * pData->periodsOut;
                }
            }

            mal_IAudioClient3_Release(pAudioClient3);
            pAudioClient3 = NULL;
        }

        // If we don't have an IAudioClient3 then we need to use the normal initialization routine.
        if (!wasInitializedUsingIAudioClient3) {
            MAL_REFERENCE_TIME bufferDuration = bufferDurationInMicroseconds*10;
            hr = mal_IAudioClient_Initialize((mal_IAudioClient*)pData->pAudioClient, shareMode, MAL_AUDCLNT_STREAMFLAGS_EVENTCALLBACK, bufferDuration, 0, (WAVEFORMATEX*)&wf, NULL);
            if (FAILED(hr)) {
                if (hr == E_ACCESSDENIED) {
                    errorMsg = "[WASAPI] Failed to initialize device. Access denied.", result = MAL_ACCESS_DENIED;
                } else if (hr == MAL_AUDCLNT_E_DEVICE_IN_USE) {
                    errorMsg = "[WASAPI] Failed to initialize device. Device in use.", result = MAL_DEVICE_BUSY;
                } else {
                    errorMsg = "[WASAPI] Failed to initialize device.", result = MAL_FAILED_TO_OPEN_BACKEND_DEVICE;
                }

                goto done;
            }
        }
    }

    if (!wasInitializedUsingIAudioClient3) {
        hr = mal_IAudioClient_GetBufferSize((mal_IAudioClient*)pData->pAudioClient, &pData->bufferSizeInFramesOut);
        if (FAILED(hr)) {
            errorMsg = "[WASAPI] Failed to get audio client's actual buffer size.", result = MAL_FAILED_TO_OPEN_BACKEND_DEVICE;
            goto done;
        }
    }

    if (type == mal_device_type_playback) {
        hr = mal_IAudioClient_GetService((mal_IAudioClient*)pData->pAudioClient, &MAL_IID_IAudioRenderClient, (void**)&pData->pRenderClient);
    } else {
        hr = mal_IAudioClient_GetService((mal_IAudioClient*)pData->pAudioClient, &MAL_IID_IAudioCaptureClient, (void**)&pData->pCaptureClient);
    }

    if (FAILED(hr)) {
        errorMsg = "[WASAPI] Failed to get audio client service.", result = MAL_API_NOT_FOUND;
        goto done;
    }


    if (shareMode == MAL_AUDCLNT_SHAREMODE_SHARED) {
        pData->exclusiveMode = MAL_FALSE;
    } else /*if (shareMode == MAL_AUDCLNT_SHAREMODE_EXCLUSIVE)*/ {
        pData->exclusiveMode = MAL_TRUE;
    }


    // Grab the name of the device.
#ifdef MAL_WIN32_DESKTOP
    mal_IPropertyStore *pProperties;
    hr = mal_IMMDevice_OpenPropertyStore(pMMDevice, STGM_READ, &pProperties);
    if (SUCCEEDED(hr)) {
        PROPVARIANT varName;
        mal_PropVariantInit(&varName);
        hr = mal_IPropertyStore_GetValue(pProperties, &MAL_PKEY_Device_FriendlyName, &varName);
        if (SUCCEEDED(hr)) {
            WideCharToMultiByte(CP_UTF8, 0, varName.pwszVal, -1, pData->deviceName, sizeof(pData->deviceName), 0, FALSE);
            mal_PropVariantClear(pContext, &varName);
        }

        mal_IPropertyStore_Release(pProperties);
    }
#endif

done:
    // Clean up.
#ifdef MAL_WIN32_DESKTOP
    if (pMMDevice != NULL) {
        mal_IMMDevice_Release(pMMDevice);
    }
#else
    if (pActivatedInterface != NULL) {
        mal_IUnknown_Release(pActivatedInterface);
    }
#endif

    if (result != MAL_SUCCESS) {
        if (pData->pRenderClient) {
            mal_IAudioRenderClient_Release((mal_IAudioRenderClient*)pData->pRenderClient);
            pData->pRenderClient = NULL;
        }
        if (pData->pCaptureClient) {
            mal_IAudioCaptureClient_Release((mal_IAudioCaptureClient*)pData->pCaptureClient);
            pData->pCaptureClient = NULL;
        }
        if (pData->pAudioClient) {
            mal_IAudioClient_Release((mal_IAudioClient*)pData->pAudioClient);
            pData->pAudioClient = NULL;
        }

        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, errorMsg, result);
    } else {
        return MAL_SUCCESS;
    }
}

mal_result mal_device_reinit__wasapi(mal_device* pDevice)
{
    mal_device_init_internal_data__wasapi data;
    data.formatIn = pDevice->format;
    data.channelsIn = pDevice->channels;
    data.sampleRateIn = pDevice->sampleRate;
    mal_copy_memory(data.channelMapIn, pDevice->channelMap, sizeof(pDevice->channelMap));
    data.bufferSizeInFramesIn = pDevice->bufferSizeInFrames;
    data.bufferSizeInMillisecondsIn = pDevice->bufferSizeInMilliseconds;
    data.periodsIn = pDevice->periods;
    data.usingDefaultFormat = pDevice->usingDefaultFormat;
    data.usingDefaultChannels = pDevice->usingDefaultChannels;
    data.usingDefaultSampleRate = pDevice->usingDefaultSampleRate;
    data.usingDefaultChannelMap = pDevice->usingDefaultChannelMap;
    data.shareMode = pDevice->initConfig.shareMode;
    mal_result result = mal_device_init_internal__wasapi(pDevice->pContext, pDevice->type, NULL, &data);
    if (result != MAL_SUCCESS) {
        return result;
    }

    // At this point we have some new objects ready to go. We need to uninitialize the previous ones and then set the new ones.
    if (pDevice->wasapi.pRenderClient) {
        mal_IAudioRenderClient_Release((mal_IAudioRenderClient*)pDevice->wasapi.pRenderClient);
        pDevice->wasapi.pRenderClient = NULL;
    }
    if (pDevice->wasapi.pCaptureClient) {
        mal_IAudioCaptureClient_Release((mal_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
        pDevice->wasapi.pCaptureClient = NULL;
    }
    if (pDevice->wasapi.pAudioClient) {
        mal_IAudioClient_Release((mal_IAudioClient*)pDevice->wasapi.pAudioClient);
        pDevice->wasapi.pAudioClient = NULL;
    }

    pDevice->wasapi.pAudioClient = data.pAudioClient;
    pDevice->wasapi.pRenderClient = data.pRenderClient;
    pDevice->wasapi.pCaptureClient = data.pCaptureClient;
    
    pDevice->internalFormat = data.formatOut;
    pDevice->internalChannels = data.channelsOut;
    pDevice->internalSampleRate = data.sampleRateOut;
    mal_copy_memory(pDevice->internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
    pDevice->bufferSizeInFrames = data.bufferSizeInFramesOut;
    pDevice->periods = data.periodsOut;
    pDevice->exclusiveMode = data.exclusiveMode;
    mal_strcpy_s(pDevice->name, sizeof(pDevice->name), data.deviceName);

    mal_IAudioClient_SetEventHandle((mal_IAudioClient*)pDevice->wasapi.pAudioClient, pDevice->wasapi.hEvent);

    return MAL_SUCCESS;
}

mal_result mal_device_init__wasapi(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pContext;
    (void)pConfig;

    mal_assert(pDevice != NULL);
    mal_zero_object(&pDevice->wasapi);

    mal_result result = MAL_SUCCESS;
    const char* errorMsg = "";

    mal_device_init_internal_data__wasapi data;
    data.formatIn = pDevice->format;
    data.channelsIn = pDevice->channels;
    data.sampleRateIn = pDevice->sampleRate;
    mal_copy_memory(data.channelMapIn, pDevice->channelMap, sizeof(pDevice->channelMap));
    data.bufferSizeInFramesIn = pDevice->bufferSizeInFrames;
    data.bufferSizeInMillisecondsIn = pDevice->bufferSizeInMilliseconds;
    data.periodsIn = pDevice->periods;
    data.usingDefaultFormat = pDevice->usingDefaultFormat;
    data.usingDefaultChannels = pDevice->usingDefaultChannels;
    data.usingDefaultSampleRate = pDevice->usingDefaultSampleRate;
    data.usingDefaultChannelMap = pDevice->usingDefaultChannelMap;
    data.shareMode = pDevice->initConfig.shareMode;
    result = mal_device_init_internal__wasapi(pDevice->pContext, type, pDeviceID, &data);
    if (result != MAL_SUCCESS) {
        return result;
    }

    pDevice->wasapi.pAudioClient = data.pAudioClient;
    pDevice->wasapi.pRenderClient = data.pRenderClient;
    pDevice->wasapi.pCaptureClient = data.pCaptureClient;
    
    pDevice->internalFormat = data.formatOut;
    pDevice->internalChannels = data.channelsOut;
    pDevice->internalSampleRate = data.sampleRateOut;
    mal_copy_memory(pDevice->internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
    pDevice->bufferSizeInFrames = data.bufferSizeInFramesOut;
    pDevice->periods = data.periodsOut;
    pDevice->exclusiveMode = data.exclusiveMode;
    mal_strcpy_s(pDevice->name, sizeof(pDevice->name), data.deviceName);



    // We need to get notifications of when the default device changes. We do this through a device enumerator by
    // registering a IMMNotificationClient with it. We only care about this if it's the default device.
#ifdef MAL_WIN32_DESKTOP
    mal_IMMDeviceEnumerator* pDeviceEnumerator;
    HRESULT hr = mal_CoCreateInstance(pContext, MAL_CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, MAL_IID_IMMDeviceEnumerator, (void**)&pDeviceEnumerator);
    if (FAILED(hr)) {
        errorMsg = "[WASAPI] Failed to create device enumerator.", result = MAL_FAILED_TO_OPEN_BACKEND_DEVICE;
        goto done;
    }

    pDevice->wasapi.notificationClient.lpVtbl  = (void*)&g_malNotificationCientVtbl;
    pDevice->wasapi.notificationClient.counter = 1;
    pDevice->wasapi.notificationClient.pDevice = pDevice;

    hr = pDeviceEnumerator->lpVtbl->RegisterEndpointNotificationCallback(pDeviceEnumerator, &pDevice->wasapi.notificationClient);
    if (SUCCEEDED(hr)) {
        pDevice->wasapi.pDeviceEnumerator = (mal_ptr)pDeviceEnumerator;
    } else {
        // Not the end of the world if we fail to register the notification callback. We just won't support automatic stream routing.
        mal_IMMDeviceEnumerator_Release(pDeviceEnumerator);
    }
#endif


    // We need to create and set the event for event-driven mode. This event is signalled whenever a new chunk of audio
    // data needs to be written or read from the device.
    pDevice->wasapi.hEvent = CreateEventA(NULL, FALSE, FALSE, NULL);
    if (pDevice->wasapi.hEvent == NULL) {
        errorMsg = "[WASAPI] Failed to create main event for main loop.", result = MAL_FAILED_TO_CREATE_EVENT;
        goto done;
    }

    mal_IAudioClient_SetEventHandle((mal_IAudioClient*)pDevice->wasapi.pAudioClient, pDevice->wasapi.hEvent);


    // When the device is playing the worker thread will be waiting on a bunch of notification events. To return from
    // this wait state we need to signal a special event.
    pDevice->wasapi.hBreakEvent = CreateEventA(NULL, FALSE, FALSE, NULL);
    if (pDevice->wasapi.hBreakEvent == NULL) {
        errorMsg = "[WASAPI] Failed to create break event for main loop break notification.", result = MAL_FAILED_TO_CREATE_EVENT;
        goto done;
    }

    result = MAL_SUCCESS;

done:
    // Clean up.
    if (result != MAL_SUCCESS) {
        mal_device_uninit__wasapi(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, errorMsg, result);
    } else {
        return MAL_SUCCESS;
    }
}

mal_result mal_device__start_backend__wasapi(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // Playback devices need to have an initial chunk of data loaded.
    if (pDevice->type == mal_device_type_playback) {
        BYTE* pData;
        HRESULT hr = mal_IAudioRenderClient_GetBuffer((mal_IAudioRenderClient*)pDevice->wasapi.pRenderClient, pDevice->bufferSizeInFrames, &pData);
        if (FAILED(hr)) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve buffer from internal playback device.", MAL_FAILED_TO_MAP_DEVICE_BUFFER);
        }

        mal_device__read_frames_from_client(pDevice, pDevice->bufferSizeInFrames, pData);

        hr = mal_IAudioRenderClient_ReleaseBuffer((mal_IAudioRenderClient*)pDevice->wasapi.pRenderClient, pDevice->bufferSizeInFrames, 0);
        if (FAILED(hr)) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to release internal buffer for playback device.", MAL_FAILED_TO_UNMAP_DEVICE_BUFFER);
        }
    }

    HRESULT hr = mal_IAudioClient_Start((mal_IAudioClient*)pDevice->wasapi.pAudioClient);
    if (FAILED(hr)) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
    }

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__wasapi(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->wasapi.pAudioClient == NULL) {
        return MAL_DEVICE_NOT_INITIALIZED;
    }

    HRESULT hr = mal_IAudioClient_Stop((mal_IAudioClient*)pDevice->wasapi.pAudioClient);
    if (FAILED(hr)) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to stop internal device.", MAL_FAILED_TO_STOP_BACKEND_DEVICE);
    }

    // The client needs to be reset or else we won't be able to resume it again.
    hr = mal_IAudioClient_Reset((mal_IAudioClient*)pDevice->wasapi.pAudioClient);
    if (FAILED(hr)) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to reset internal device.", MAL_FAILED_TO_STOP_BACKEND_DEVICE);
    }

    return MAL_SUCCESS;
}

mal_result mal_device__break_main_loop__wasapi(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // The main loop will be waiting on a bunch of events via the WaitForMultipleObjects() API. One of those events
    // is a special event we use for forcing that function to return.
    pDevice->wasapi.breakFromMainLoop = MAL_TRUE;
    SetEvent(pDevice->wasapi.hBreakEvent);
    return MAL_SUCCESS;
}

mal_result mal_device__get_available_frames__wasapi(mal_device* pDevice, mal_uint32* pFrameCount)
{
    mal_assert(pDevice != NULL);
    mal_assert(pFrameCount != NULL);
    
    *pFrameCount = 0;

    mal_uint32 paddingFramesCount;
    HRESULT hr = mal_IAudioClient_GetCurrentPadding((mal_IAudioClient*)pDevice->wasapi.pAudioClient, &paddingFramesCount);
    if (FAILED(hr)) {
        return MAL_DEVICE_UNAVAILABLE;
    }

    // Slightly different rules for exclusive and shared modes.
    if (pDevice->exclusiveMode) {
        *pFrameCount = paddingFramesCount;
    } else {
        if (pDevice->type == mal_device_type_playback) {
            *pFrameCount = pDevice->bufferSizeInFrames - paddingFramesCount;
        } else {
            *pFrameCount = paddingFramesCount;
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_device__wait_for_frames__wasapi(mal_device* pDevice, mal_uint32* pFrameCount)
{
    mal_assert(pDevice != NULL);

    mal_result result;

    while (!pDevice->wasapi.breakFromMainLoop) {
        // Wait for a buffer to become available or for the stop event to be signalled.
        HANDLE hEvents[2];
        hEvents[0] = (HANDLE)pDevice->wasapi.hEvent;
        hEvents[1] = (HANDLE)pDevice->wasapi.hBreakEvent;
        if (WaitForMultipleObjects(mal_countof(hEvents), hEvents, FALSE, INFINITE) == WAIT_FAILED) {
            break;
        }

        // Break from the main loop if the device isn't started anymore. Likely what's happened is the application
        // has requested that the device be stopped.
        if (!mal_device_is_started(pDevice)) {
            break;
        }

        // Make sure we break from the main loop if requested from an external factor.
        if (pDevice->wasapi.breakFromMainLoop) {
            break;
        }

        // We may want to reinitialize the device. Only do this if this device is the default.
        mal_bool32 needDeviceReinit = MAL_FALSE;

        mal_bool32 hasDefaultDeviceChanged = pDevice->wasapi.hasDefaultDeviceChanged;
        if (hasDefaultDeviceChanged && pDevice->isDefaultDevice) {
            needDeviceReinit = MAL_TRUE;
        }

        if (!needDeviceReinit) {
            result = mal_device__get_available_frames__wasapi(pDevice, pFrameCount);
            if (result != MAL_SUCCESS) {
                if (!pDevice->exclusiveMode) {
                    needDeviceReinit = MAL_TRUE;
                } else {
                    return result;
                }
            }
        }


        mal_atomic_exchange_32(&pDevice->wasapi.hasDefaultDeviceChanged, MAL_FALSE);

        // Here is where the device is re-initialized if required.
        if (needDeviceReinit) {
            #ifdef MAL_DEBUG_OUTPUT
                printf("=== CHANGING DEVICE ===\n");
            #endif

            if (pDevice->pContext->onDeviceReinit) {
                mal_result reinitResult = pDevice->pContext->onDeviceReinit(pDevice);
                if (reinitResult != MAL_SUCCESS) {
                    return reinitResult;
                }

                mal_device__post_init_setup(pDevice);

                // Start playing the device again, and then continue the loop from the top.
                if (mal_device__get_state(pDevice) == MAL_STATE_STARTED) {
                    if (pDevice->pContext->onDeviceStart) {
                        pDevice->pContext->onDeviceStart(pDevice);
                    }
                    continue;
                }
            }
        }
        

        if (*pFrameCount > 0) {
            return MAL_SUCCESS;
        }
    }

    // We'll get here if the loop was terminated. Just return whatever's available.
    return mal_device__get_available_frames__wasapi(pDevice, pFrameCount);
}

mal_result mal_device__main_loop__wasapi(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // Make sure the break event is not signaled to ensure we don't end up immediately returning from WaitForMultipleObjects().
    ResetEvent(pDevice->wasapi.hBreakEvent);

    pDevice->wasapi.breakFromMainLoop = MAL_FALSE;
    while (!pDevice->wasapi.breakFromMainLoop) {
        mal_uint32 framesAvailable;
        mal_result result = mal_device__wait_for_frames__wasapi(pDevice, &framesAvailable);
        if (result != MAL_SUCCESS) {
            return result;
        }

        if (framesAvailable == 0) {
            continue;
        }

        // If it's a playback device, don't bother grabbing more data if the device is being stopped.
        if (pDevice->wasapi.breakFromMainLoop && pDevice->type == mal_device_type_playback) {
            return MAL_SUCCESS;
        }

        if (pDevice->type == mal_device_type_playback) {
            BYTE* pData;
            HRESULT hr = mal_IAudioRenderClient_GetBuffer((mal_IAudioRenderClient*)pDevice->wasapi.pRenderClient, framesAvailable, &pData);
            if (FAILED(hr)) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from playback device in preparation for sending new data to the device.", MAL_FAILED_TO_MAP_DEVICE_BUFFER);
            }

            mal_device__read_frames_from_client(pDevice, framesAvailable, pData);

            hr = mal_IAudioRenderClient_ReleaseBuffer((mal_IAudioRenderClient*)pDevice->wasapi.pRenderClient, framesAvailable, 0);
            if (FAILED(hr)) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WASAPI] Failed to release internal buffer from playback device in preparation for sending new data to the device.", MAL_FAILED_TO_UNMAP_DEVICE_BUFFER);
            }
        } else {
            mal_uint32 framesRemaining = framesAvailable;
            while (framesRemaining > 0) {
                BYTE* pData;
                mal_uint32 framesToSend;
                DWORD flags;
                HRESULT hr = mal_IAudioCaptureClient_GetBuffer((mal_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, &pData, &framesToSend, &flags, NULL, NULL);
                if (FAILED(hr)) {
                    mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WASAPI] WARNING: Failed to retrieve internal buffer from capture device in preparation for sending new data to the client.", MAL_FAILED_TO_MAP_DEVICE_BUFFER);
                    break;
                }

                if (hr != MAL_AUDCLNT_S_BUFFER_EMPTY) {
                    mal_device__send_frames_to_client(pDevice, framesToSend, pData);

                    hr = mal_IAudioCaptureClient_ReleaseBuffer((mal_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, framesToSend);
                    if (FAILED(hr)) {
                        mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WASAPI] WARNING: Failed to release internal buffer from capture device in preparation for sending new data to the client.", MAL_FAILED_TO_UNMAP_DEVICE_BUFFER);
                        break;
                    }

                    if (framesRemaining >= framesToSend) {
                        framesRemaining -= framesToSend;
                    } else {
                        framesRemaining = 0;
                    }
                }
            }
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_context_uninit__wasapi(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_wasapi);
    (void)pContext;

    return MAL_SUCCESS;
}

mal_result mal_context_init__wasapi(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    (void)pContext;

    mal_result result = MAL_SUCCESS;

#ifdef MAL_WIN32_DESKTOP
    // WASAPI is only supported in Vista SP1 and newer. The reason for SP1 and not the base version of Vista is that event-driven
    // exclusive mode does not work until SP1.
    mal_OSVERSIONINFOEXW osvi;
    mal_zero_object(&osvi);
    osvi.dwOSVersionInfoSize = sizeof(osvi);
    osvi.dwMajorVersion = HIBYTE(_WIN32_WINNT_VISTA);
    osvi.dwMinorVersion = LOBYTE(_WIN32_WINNT_VISTA);
    osvi.wServicePackMajor = 1;
    if (VerifyVersionInfoW(&osvi, VER_MAJORVERSION | VER_MINORVERSION | VER_SERVICEPACKMAJOR, VerSetConditionMask(VerSetConditionMask(VerSetConditionMask(0, VER_MAJORVERSION, VER_GREATER_EQUAL), VER_MINORVERSION, VER_GREATER_EQUAL), VER_SERVICEPACKMAJOR, VER_GREATER_EQUAL))) {
        result = MAL_SUCCESS;
    } else {
        result = MAL_NO_BACKEND;
    }
#endif

    if (result != MAL_SUCCESS) {
        return result;
    }

    pContext->onUninit              = mal_context_uninit__wasapi;
    pContext->onDeviceIDEqual       = mal_context_is_device_id_equal__wasapi;
    pContext->onEnumDevices         = mal_context_enumerate_devices__wasapi;
    pContext->onGetDeviceInfo       = mal_context_get_device_info__wasapi;
    pContext->onDeviceInit          = mal_device_init__wasapi;
    pContext->onDeviceUninit        = mal_device_uninit__wasapi;
    pContext->onDeviceReinit        = mal_device_reinit__wasapi;
    pContext->onDeviceStart         = mal_device__start_backend__wasapi;
    pContext->onDeviceStop          = mal_device__stop_backend__wasapi;
    pContext->onDeviceBreakMainLoop = mal_device__break_main_loop__wasapi;
    pContext->onDeviceMainLoop      = mal_device__main_loop__wasapi;

    return result;
}
#endif

///////////////////////////////////////////////////////////////////////////////
//
// DirectSound Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_DSOUND
//#include <dsound.h>

GUID MAL_GUID_IID_DirectSoundNotify = {0xb0210783, 0x89cd, 0x11d0, {0xaf, 0x08, 0x00, 0xa0, 0xc9, 0x25, 0xcd, 0x16}};

// mini_al only uses priority or exclusive modes.
#define MAL_DSSCL_NORMAL                 1
#define MAL_DSSCL_PRIORITY               2
#define MAL_DSSCL_EXCLUSIVE              3
#define MAL_DSSCL_WRITEPRIMARY           4

#define MAL_DSCAPS_PRIMARYMONO           0x00000001
#define MAL_DSCAPS_PRIMARYSTEREO         0x00000002
#define MAL_DSCAPS_PRIMARY8BIT           0x00000004
#define MAL_DSCAPS_PRIMARY16BIT          0x00000008
#define MAL_DSCAPS_CONTINUOUSRATE        0x00000010
#define MAL_DSCAPS_EMULDRIVER            0x00000020
#define MAL_DSCAPS_CERTIFIED             0x00000040
#define MAL_DSCAPS_SECONDARYMONO         0x00000100
#define MAL_DSCAPS_SECONDARYSTEREO       0x00000200
#define MAL_DSCAPS_SECONDARY8BIT         0x00000400
#define MAL_DSCAPS_SECONDARY16BIT        0x00000800

#define MAL_DSBCAPS_PRIMARYBUFFER        0x00000001
#define MAL_DSBCAPS_STATIC               0x00000002
#define MAL_DSBCAPS_LOCHARDWARE          0x00000004
#define MAL_DSBCAPS_LOCSOFTWARE          0x00000008
#define MAL_DSBCAPS_CTRL3D               0x00000010
#define MAL_DSBCAPS_CTRLFREQUENCY        0x00000020
#define MAL_DSBCAPS_CTRLPAN              0x00000040
#define MAL_DSBCAPS_CTRLVOLUME           0x00000080
#define MAL_DSBCAPS_CTRLPOSITIONNOTIFY   0x00000100
#define MAL_DSBCAPS_CTRLFX               0x00000200
#define MAL_DSBCAPS_STICKYFOCUS          0x00004000
#define MAL_DSBCAPS_GLOBALFOCUS          0x00008000
#define MAL_DSBCAPS_GETCURRENTPOSITION2  0x00010000
#define MAL_DSBCAPS_MUTE3DATMAXDISTANCE  0x00020000
#define MAL_DSBCAPS_LOCDEFER             0x00040000
#define MAL_DSBCAPS_TRUEPLAYPOSITION     0x00080000

#define MAL_DSBPLAY_LOOPING              0x00000001
#define MAL_DSBPLAY_LOCHARDWARE          0x00000002
#define MAL_DSBPLAY_LOCSOFTWARE          0x00000004
#define MAL_DSBPLAY_TERMINATEBY_TIME     0x00000008
#define MAL_DSBPLAY_TERMINATEBY_DISTANCE 0x00000010
#define MAL_DSBPLAY_TERMINATEBY_PRIORITY 0x00000020

#define MAL_DSCBSTART_LOOPING            0x00000001

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwReserved;
    WAVEFORMATEX* lpwfxFormat;
    GUID guid3DAlgorithm;
} MAL_DSBUFFERDESC;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwReserved;
    WAVEFORMATEX* lpwfxFormat;
    DWORD dwFXCount;
    void* lpDSCFXDesc;  // <-- mini_al doesn't use this, so set to void*.
} MAL_DSCBUFFERDESC;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwMinSecondarySampleRate;
    DWORD dwMaxSecondarySampleRate;
    DWORD dwPrimaryBuffers;
    DWORD dwMaxHwMixingAllBuffers;
    DWORD dwMaxHwMixingStaticBuffers;
    DWORD dwMaxHwMixingStreamingBuffers;
    DWORD dwFreeHwMixingAllBuffers;
    DWORD dwFreeHwMixingStaticBuffers;
    DWORD dwFreeHwMixingStreamingBuffers;
    DWORD dwMaxHw3DAllBuffers;
    DWORD dwMaxHw3DStaticBuffers;
    DWORD dwMaxHw3DStreamingBuffers;
    DWORD dwFreeHw3DAllBuffers;
    DWORD dwFreeHw3DStaticBuffers;
    DWORD dwFreeHw3DStreamingBuffers;
    DWORD dwTotalHwMemBytes;
    DWORD dwFreeHwMemBytes;
    DWORD dwMaxContigFreeHwMemBytes;
    DWORD dwUnlockTransferRateHwBuffers;
    DWORD dwPlayCpuOverheadSwBuffers;
    DWORD dwReserved1;
    DWORD dwReserved2;
} MAL_DSCAPS;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwUnlockTransferRate;
    DWORD dwPlayCpuOverhead;
} MAL_DSBCAPS;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwFormats;
    DWORD dwChannels;
} MAL_DSCCAPS;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwReserved;
} MAL_DSCBCAPS;

typedef struct
{
    DWORD  dwOffset;
    HANDLE hEventNotify;
} MAL_DSBPOSITIONNOTIFY;

typedef struct mal_IDirectSound              mal_IDirectSound;
typedef struct mal_IDirectSoundBuffer        mal_IDirectSoundBuffer;
typedef struct mal_IDirectSoundCapture       mal_IDirectSoundCapture;
typedef struct mal_IDirectSoundCaptureBuffer mal_IDirectSoundCaptureBuffer;
typedef struct mal_IDirectSoundNotify        mal_IDirectSoundNotify;


// COM objects. The way these work is that you have a vtable (a list of function pointers, kind of
// like how C++ works internally), and then you have a structure with a single member, which is a
// pointer to the vtable. The vtable is where the methods of the object are defined. Methods need
// to be in a specific order, and parent classes need to have their methods declared first.

// IDirectSound
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IDirectSound* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IDirectSound* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IDirectSound* pThis);

    // IDirectSound
    HRESULT (STDMETHODCALLTYPE * CreateSoundBuffer)   (mal_IDirectSound* pThis, const MAL_DSBUFFERDESC* pDSBufferDesc, mal_IDirectSoundBuffer** ppDSBuffer, void* pUnkOuter);
    HRESULT (STDMETHODCALLTYPE * GetCaps)             (mal_IDirectSound* pThis, MAL_DSCAPS* pDSCaps);
    HRESULT (STDMETHODCALLTYPE * DuplicateSoundBuffer)(mal_IDirectSound* pThis, mal_IDirectSoundBuffer* pDSBufferOriginal, mal_IDirectSoundBuffer** ppDSBufferDuplicate);
    HRESULT (STDMETHODCALLTYPE * SetCooperativeLevel) (mal_IDirectSound* pThis, HWND hwnd, DWORD dwLevel);
    HRESULT (STDMETHODCALLTYPE * Compact)             (mal_IDirectSound* pThis);
    HRESULT (STDMETHODCALLTYPE * GetSpeakerConfig)    (mal_IDirectSound* pThis, DWORD* pSpeakerConfig);
    HRESULT (STDMETHODCALLTYPE * SetSpeakerConfig)    (mal_IDirectSound* pThis, DWORD dwSpeakerConfig);
    HRESULT (STDMETHODCALLTYPE * Initialize)          (mal_IDirectSound* pThis, const GUID* pGuidDevice);
} mal_IDirectSoundVtbl;
struct mal_IDirectSound
{
    mal_IDirectSoundVtbl* lpVtbl;
};
HRESULT mal_IDirectSound_QueryInterface(mal_IDirectSound* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IDirectSound_AddRef(mal_IDirectSound* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IDirectSound_Release(mal_IDirectSound* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IDirectSound_CreateSoundBuffer(mal_IDirectSound* pThis, const MAL_DSBUFFERDESC* pDSBufferDesc, mal_IDirectSoundBuffer** ppDSBuffer, void* pUnkOuter) { return pThis->lpVtbl->CreateSoundBuffer(pThis, pDSBufferDesc, ppDSBuffer, pUnkOuter); }
HRESULT mal_IDirectSound_GetCaps(mal_IDirectSound* pThis, MAL_DSCAPS* pDSCaps)                           { return pThis->lpVtbl->GetCaps(pThis, pDSCaps); }
HRESULT mal_IDirectSound_DuplicateSoundBuffer(mal_IDirectSound* pThis, mal_IDirectSoundBuffer* pDSBufferOriginal, mal_IDirectSoundBuffer** ppDSBufferDuplicate) { return pThis->lpVtbl->DuplicateSoundBuffer(pThis, pDSBufferOriginal, ppDSBufferDuplicate); }
HRESULT mal_IDirectSound_SetCooperativeLevel(mal_IDirectSound* pThis, HWND hwnd, DWORD dwLevel)          { return pThis->lpVtbl->SetCooperativeLevel(pThis, hwnd, dwLevel); }
HRESULT mal_IDirectSound_Compact(mal_IDirectSound* pThis)                                                { return pThis->lpVtbl->Compact(pThis); }
HRESULT mal_IDirectSound_GetSpeakerConfig(mal_IDirectSound* pThis, DWORD* pSpeakerConfig)                { return pThis->lpVtbl->GetSpeakerConfig(pThis, pSpeakerConfig); }
HRESULT mal_IDirectSound_SetSpeakerConfig(mal_IDirectSound* pThis, DWORD dwSpeakerConfig)                { return pThis->lpVtbl->SetSpeakerConfig(pThis, dwSpeakerConfig); }
HRESULT mal_IDirectSound_Initialize(mal_IDirectSound* pThis, const GUID* pGuidDevice)                    { return pThis->lpVtbl->Initialize(pThis, pGuidDevice); }


// IDirectSoundBuffer
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IDirectSoundBuffer* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IDirectSoundBuffer* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IDirectSoundBuffer* pThis);

    // IDirectSoundBuffer
    HRESULT (STDMETHODCALLTYPE * GetCaps)           (mal_IDirectSoundBuffer* pThis, MAL_DSBCAPS* pDSBufferCaps);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPosition)(mal_IDirectSoundBuffer* pThis, DWORD* pCurrentPlayCursor, DWORD* pCurrentWriteCursor);
    HRESULT (STDMETHODCALLTYPE * GetFormat)         (mal_IDirectSoundBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten);
    HRESULT (STDMETHODCALLTYPE * GetVolume)         (mal_IDirectSoundBuffer* pThis, LONG* pVolume);
    HRESULT (STDMETHODCALLTYPE * GetPan)            (mal_IDirectSoundBuffer* pThis, LONG* pPan);
    HRESULT (STDMETHODCALLTYPE * GetFrequency)      (mal_IDirectSoundBuffer* pThis, DWORD* pFrequency);
    HRESULT (STDMETHODCALLTYPE * GetStatus)         (mal_IDirectSoundBuffer* pThis, DWORD* pStatus);
    HRESULT (STDMETHODCALLTYPE * Initialize)        (mal_IDirectSoundBuffer* pThis, mal_IDirectSound* pDirectSound, const MAL_DSBUFFERDESC* pDSBufferDesc);
    HRESULT (STDMETHODCALLTYPE * Lock)              (mal_IDirectSoundBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * Play)              (mal_IDirectSoundBuffer* pThis, DWORD dwReserved1, DWORD dwPriority, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * SetCurrentPosition)(mal_IDirectSoundBuffer* pThis, DWORD dwNewPosition);
    HRESULT (STDMETHODCALLTYPE * SetFormat)         (mal_IDirectSoundBuffer* pThis, const WAVEFORMATEX* pFormat);
    HRESULT (STDMETHODCALLTYPE * SetVolume)         (mal_IDirectSoundBuffer* pThis, LONG volume);
    HRESULT (STDMETHODCALLTYPE * SetPan)            (mal_IDirectSoundBuffer* pThis, LONG pan);
    HRESULT (STDMETHODCALLTYPE * SetFrequency)      (mal_IDirectSoundBuffer* pThis, DWORD dwFrequency);
    HRESULT (STDMETHODCALLTYPE * Stop)              (mal_IDirectSoundBuffer* pThis);
    HRESULT (STDMETHODCALLTYPE * Unlock)            (mal_IDirectSoundBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2);
    HRESULT (STDMETHODCALLTYPE * Restore)           (mal_IDirectSoundBuffer* pThis);
} mal_IDirectSoundBufferVtbl;
struct mal_IDirectSoundBuffer
{
    mal_IDirectSoundBufferVtbl* lpVtbl;
};
HRESULT mal_IDirectSoundBuffer_QueryInterface(mal_IDirectSoundBuffer* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IDirectSoundBuffer_AddRef(mal_IDirectSoundBuffer* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IDirectSoundBuffer_Release(mal_IDirectSoundBuffer* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IDirectSoundBuffer_GetCaps(mal_IDirectSoundBuffer* pThis, MAL_DSBCAPS* pDSBufferCaps)                    { return pThis->lpVtbl->GetCaps(pThis, pDSBufferCaps); }
HRESULT mal_IDirectSoundBuffer_GetCurrentPosition(mal_IDirectSoundBuffer* pThis, DWORD* pCurrentPlayCursor, DWORD* pCurrentWriteCursor) { return pThis->lpVtbl->GetCurrentPosition(pThis, pCurrentPlayCursor, pCurrentWriteCursor); }
HRESULT mal_IDirectSoundBuffer_GetFormat(mal_IDirectSoundBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten) { return pThis->lpVtbl->GetFormat(pThis, pFormat, dwSizeAllocated, pSizeWritten); }
HRESULT mal_IDirectSoundBuffer_GetVolume(mal_IDirectSoundBuffer* pThis, LONG* pVolume)                               { return pThis->lpVtbl->GetVolume(pThis, pVolume); }
HRESULT mal_IDirectSoundBuffer_GetPan(mal_IDirectSoundBuffer* pThis, LONG* pPan)                                     { return pThis->lpVtbl->GetPan(pThis, pPan); }
HRESULT mal_IDirectSoundBuffer_GetFrequency(mal_IDirectSoundBuffer* pThis, DWORD* pFrequency)                        { return pThis->lpVtbl->GetFrequency(pThis, pFrequency); }
HRESULT mal_IDirectSoundBuffer_GetStatus(mal_IDirectSoundBuffer* pThis, DWORD* pStatus)                              { return pThis->lpVtbl->GetStatus(pThis, pStatus); }
HRESULT mal_IDirectSoundBuffer_Initialize(mal_IDirectSoundBuffer* pThis, mal_IDirectSound* pDirectSound, const MAL_DSBUFFERDESC* pDSBufferDesc) { return pThis->lpVtbl->Initialize(pThis, pDirectSound, pDSBufferDesc); }
HRESULT mal_IDirectSoundBuffer_Lock(mal_IDirectSoundBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags) { return pThis->lpVtbl->Lock(pThis, dwOffset, dwBytes, ppAudioPtr1, pAudioBytes1, ppAudioPtr2, pAudioBytes2, dwFlags); }
HRESULT mal_IDirectSoundBuffer_Play(mal_IDirectSoundBuffer* pThis, DWORD dwReserved1, DWORD dwPriority, DWORD dwFlags) { return pThis->lpVtbl->Play(pThis, dwReserved1, dwPriority, dwFlags); }
HRESULT mal_IDirectSoundBuffer_SetCurrentPosition(mal_IDirectSoundBuffer* pThis, DWORD dwNewPosition)                { return pThis->lpVtbl->SetCurrentPosition(pThis, dwNewPosition); }
HRESULT mal_IDirectSoundBuffer_SetFormat(mal_IDirectSoundBuffer* pThis, const WAVEFORMATEX* pFormat)                 { return pThis->lpVtbl->SetFormat(pThis, pFormat); }
HRESULT mal_IDirectSoundBuffer_SetVolume(mal_IDirectSoundBuffer* pThis, LONG volume)                                 { return pThis->lpVtbl->SetVolume(pThis, volume); }
HRESULT mal_IDirectSoundBuffer_SetPan(mal_IDirectSoundBuffer* pThis, LONG pan)                                       { return pThis->lpVtbl->SetPan(pThis, pan); }
HRESULT mal_IDirectSoundBuffer_SetFrequency(mal_IDirectSoundBuffer* pThis, DWORD dwFrequency)                        { return pThis->lpVtbl->SetFrequency(pThis, dwFrequency); }
HRESULT mal_IDirectSoundBuffer_Stop(mal_IDirectSoundBuffer* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
HRESULT mal_IDirectSoundBuffer_Unlock(mal_IDirectSoundBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2) { return pThis->lpVtbl->Unlock(pThis, pAudioPtr1, dwAudioBytes1, pAudioPtr2, dwAudioBytes2); }
HRESULT mal_IDirectSoundBuffer_Restore(mal_IDirectSoundBuffer* pThis)                                                { return pThis->lpVtbl->Restore(pThis); }


// IDirectSoundCapture
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IDirectSoundCapture* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IDirectSoundCapture* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IDirectSoundCapture* pThis);

    // IDirectSoundCapture
    HRESULT (STDMETHODCALLTYPE * CreateCaptureBuffer)(mal_IDirectSoundCapture* pThis, const MAL_DSCBUFFERDESC* pDSCBufferDesc, mal_IDirectSoundCaptureBuffer** ppDSCBuffer, void* pUnkOuter);
    HRESULT (STDMETHODCALLTYPE * GetCaps)            (mal_IDirectSoundCapture* pThis, MAL_DSCCAPS* pDSCCaps);
    HRESULT (STDMETHODCALLTYPE * Initialize)         (mal_IDirectSoundCapture* pThis, const GUID* pGuidDevice);
} mal_IDirectSoundCaptureVtbl;
struct mal_IDirectSoundCapture
{
    mal_IDirectSoundCaptureVtbl* lpVtbl;
};
HRESULT mal_IDirectSoundCapture_QueryInterface(mal_IDirectSoundCapture* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IDirectSoundCapture_AddRef(mal_IDirectSoundCapture* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IDirectSoundCapture_Release(mal_IDirectSoundCapture* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IDirectSoundCapture_CreateCaptureBuffer(mal_IDirectSoundCapture* pThis, const MAL_DSCBUFFERDESC* pDSCBufferDesc, mal_IDirectSoundCaptureBuffer** ppDSCBuffer, void* pUnkOuter) { return pThis->lpVtbl->CreateCaptureBuffer(pThis, pDSCBufferDesc, ppDSCBuffer, pUnkOuter); }
HRESULT mal_IDirectSoundCapture_GetCaps            (mal_IDirectSoundCapture* pThis, MAL_DSCCAPS* pDSCCaps)             { return pThis->lpVtbl->GetCaps(pThis, pDSCCaps); }
HRESULT mal_IDirectSoundCapture_Initialize         (mal_IDirectSoundCapture* pThis, const GUID* pGuidDevice)           { return pThis->lpVtbl->Initialize(pThis, pGuidDevice); }


// IDirectSoundCaptureBuffer
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IDirectSoundCaptureBuffer* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IDirectSoundCaptureBuffer* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IDirectSoundCaptureBuffer* pThis);

    // IDirectSoundCaptureBuffer
    HRESULT (STDMETHODCALLTYPE * GetCaps)           (mal_IDirectSoundCaptureBuffer* pThis, MAL_DSCBCAPS* pDSCBCaps);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPosition)(mal_IDirectSoundCaptureBuffer* pThis, DWORD* pCapturePosition, DWORD* pReadPosition);
    HRESULT (STDMETHODCALLTYPE * GetFormat)         (mal_IDirectSoundCaptureBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten);
    HRESULT (STDMETHODCALLTYPE * GetStatus)         (mal_IDirectSoundCaptureBuffer* pThis, DWORD* pStatus);
    HRESULT (STDMETHODCALLTYPE * Initialize)        (mal_IDirectSoundCaptureBuffer* pThis, mal_IDirectSoundCapture* pDirectSoundCapture, const MAL_DSCBUFFERDESC* pDSCBufferDesc);
    HRESULT (STDMETHODCALLTYPE * Lock)              (mal_IDirectSoundCaptureBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * Start)             (mal_IDirectSoundCaptureBuffer* pThis, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * Stop)              (mal_IDirectSoundCaptureBuffer* pThis);
    HRESULT (STDMETHODCALLTYPE * Unlock)            (mal_IDirectSoundCaptureBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2);
} mal_IDirectSoundCaptureBufferVtbl;
struct mal_IDirectSoundCaptureBuffer
{
    mal_IDirectSoundCaptureBufferVtbl* lpVtbl;
};
HRESULT mal_IDirectSoundCaptureBuffer_QueryInterface(mal_IDirectSoundCaptureBuffer* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IDirectSoundCaptureBuffer_AddRef(mal_IDirectSoundCaptureBuffer* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IDirectSoundCaptureBuffer_Release(mal_IDirectSoundCaptureBuffer* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IDirectSoundCaptureBuffer_GetCaps(mal_IDirectSoundCaptureBuffer* pThis, MAL_DSCBCAPS* pDSCBCaps)                       { return pThis->lpVtbl->GetCaps(pThis, pDSCBCaps); }
HRESULT mal_IDirectSoundCaptureBuffer_GetCurrentPosition(mal_IDirectSoundCaptureBuffer* pThis, DWORD* pCapturePosition, DWORD* pReadPosition) { return pThis->lpVtbl->GetCurrentPosition(pThis, pCapturePosition, pReadPosition); }
HRESULT mal_IDirectSoundCaptureBuffer_GetFormat(mal_IDirectSoundCaptureBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten) { return pThis->lpVtbl->GetFormat(pThis, pFormat, dwSizeAllocated, pSizeWritten); }
HRESULT mal_IDirectSoundCaptureBuffer_GetStatus(mal_IDirectSoundCaptureBuffer* pThis, DWORD* pStatus)                              { return pThis->lpVtbl->GetStatus(pThis, pStatus); }
HRESULT mal_IDirectSoundCaptureBuffer_Initialize(mal_IDirectSoundCaptureBuffer* pThis, mal_IDirectSoundCapture* pDirectSoundCapture, const MAL_DSCBUFFERDESC* pDSCBufferDesc) { return pThis->lpVtbl->Initialize(pThis, pDirectSoundCapture, pDSCBufferDesc); }
HRESULT mal_IDirectSoundCaptureBuffer_Lock(mal_IDirectSoundCaptureBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags) { return pThis->lpVtbl->Lock(pThis, dwOffset, dwBytes, ppAudioPtr1, pAudioBytes1, ppAudioPtr2, pAudioBytes2, dwFlags); }
HRESULT mal_IDirectSoundCaptureBuffer_Start(mal_IDirectSoundCaptureBuffer* pThis, DWORD dwFlags)                                   { return pThis->lpVtbl->Start(pThis, dwFlags); }
HRESULT mal_IDirectSoundCaptureBuffer_Stop(mal_IDirectSoundCaptureBuffer* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
HRESULT mal_IDirectSoundCaptureBuffer_Unlock(mal_IDirectSoundCaptureBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2) { return pThis->lpVtbl->Unlock(pThis, pAudioPtr1, dwAudioBytes1, pAudioPtr2, dwAudioBytes2); }


// IDirectSoundNotify
typedef struct
{
    // IUnknown
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(mal_IDirectSoundNotify* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (mal_IDirectSoundNotify* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (mal_IDirectSoundNotify* pThis);

    // IDirectSoundNotify
    HRESULT (STDMETHODCALLTYPE * SetNotificationPositions)(mal_IDirectSoundNotify* pThis, DWORD dwPositionNotifies, const MAL_DSBPOSITIONNOTIFY* pPositionNotifies);
} mal_IDirectSoundNotifyVtbl;
struct mal_IDirectSoundNotify
{
    mal_IDirectSoundNotifyVtbl* lpVtbl;
};
HRESULT mal_IDirectSoundNotify_QueryInterface(mal_IDirectSoundNotify* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   mal_IDirectSoundNotify_AddRef(mal_IDirectSoundNotify* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   mal_IDirectSoundNotify_Release(mal_IDirectSoundNotify* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT mal_IDirectSoundNotify_SetNotificationPositions(mal_IDirectSoundNotify* pThis, DWORD dwPositionNotifies, const MAL_DSBPOSITIONNOTIFY* pPositionNotifies) { return pThis->lpVtbl->SetNotificationPositions(pThis, dwPositionNotifies, pPositionNotifies); }


typedef BOOL    (CALLBACK * mal_DSEnumCallbackAProc)             (LPGUID pDeviceGUID, LPCSTR pDeviceDescription, LPCSTR pModule, LPVOID pContext);
typedef HRESULT (WINAPI   * mal_DirectSoundCreateProc)           (const GUID* pcGuidDevice, mal_IDirectSound** ppDS8, LPUNKNOWN pUnkOuter);
typedef HRESULT (WINAPI   * mal_DirectSoundEnumerateAProc)       (mal_DSEnumCallbackAProc pDSEnumCallback, LPVOID pContext);
typedef HRESULT (WINAPI   * mal_DirectSoundCaptureCreateProc)    (const GUID* pcGuidDevice, mal_IDirectSoundCapture** ppDSC8, LPUNKNOWN pUnkOuter);
typedef HRESULT (WINAPI   * mal_DirectSoundCaptureEnumerateAProc)(mal_DSEnumCallbackAProc pDSEnumCallback, LPVOID pContext);


// Retrieves the channel count and channel map for the given speaker configuration. If the speaker configuration is unknown,
// the channel count and channel map will be left unmodified.
void mal_get_channels_from_speaker_config__dsound(DWORD speakerConfig, WORD* pChannelsOut, DWORD* pChannelMapOut)
{
    WORD channels = 0;
    if (pChannelsOut != NULL) {
        channels = *pChannelsOut;
    }

    DWORD channelMap = 0;
    if (pChannelMapOut != NULL) {
        channelMap = *pChannelMapOut;
    }

    // The speaker configuration is a combination of speaker config and speaker geometry. The lower 8 bits is what we care about. The upper
    // 16 bits is for the geometry.
    switch ((BYTE)(speakerConfig)) {
        case 1 /*DSSPEAKER_HEADPHONE*/:                          channels = 2; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT; break;
        case 2 /*DSSPEAKER_MONO*/:                               channels = 1; channelMap = SPEAKER_FRONT_CENTER; break;
        case 3 /*DSSPEAKER_QUAD*/:                               channels = 4; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT; break;
        case 4 /*DSSPEAKER_STEREO*/:                             channels = 2; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT; break;
        case 5 /*DSSPEAKER_SURROUND*/:                           channels = 4; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_BACK_CENTER; break;
        case 6 /*DSSPEAKER_5POINT1_BACK*/ /*DSSPEAKER_5POINT1*/: channels = 6; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT; break;
        case 7 /*DSSPEAKER_7POINT1_WIDE*/ /*DSSPEAKER_7POINT1*/: channels = 8; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT | SPEAKER_FRONT_LEFT_OF_CENTER | SPEAKER_FRONT_RIGHT_OF_CENTER; break;
        case 8 /*DSSPEAKER_7POINT1_SURROUND*/:                   channels = 8; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT | SPEAKER_SIDE_LEFT | SPEAKER_SIDE_RIGHT; break;
        case 9 /*DSSPEAKER_5POINT1_SURROUND*/:                   channels = 6; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_SIDE_LEFT | SPEAKER_SIDE_RIGHT; break;
        default: break;
    }

    if (pChannelsOut != NULL) {
        *pChannelsOut = channels;
    }

    if (pChannelMapOut != NULL) {
        *pChannelMapOut = channelMap;
    }
}


mal_result mal_context_create_IDirectSound__dsound(mal_context* pContext, mal_share_mode shareMode, const mal_device_id* pDeviceID, mal_IDirectSound** ppDirectSound)
{
    mal_assert(pContext != NULL);
    mal_assert(ppDirectSound != NULL);

    *ppDirectSound = NULL;
    mal_IDirectSound* pDirectSound = NULL;

    if (FAILED(((mal_DirectSoundCreateProc)pContext->dsound.DirectSoundCreate)((pDeviceID == NULL) ? NULL : (const GUID*)pDeviceID->dsound, &pDirectSound, NULL))) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[DirectSound] DirectSoundCreate() failed for playback device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    // The cooperative level must be set before doing anything else.
    HWND hWnd = ((MAL_PFN_GetForegroundWindow)pContext->win32.GetForegroundWindow)();
    if (hWnd == NULL) {
        hWnd = ((MAL_PFN_GetDesktopWindow)pContext->win32.GetDesktopWindow)();
    }
    if (FAILED(mal_IDirectSound_SetCooperativeLevel(pDirectSound, hWnd, (shareMode == mal_share_mode_exclusive) ? MAL_DSSCL_EXCLUSIVE : MAL_DSSCL_PRIORITY))) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_SetCooperateiveLevel() failed for playback device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    *ppDirectSound = pDirectSound;
    return MAL_SUCCESS;
}

mal_result mal_context_create_IDirectSoundCapture__dsound(mal_context* pContext, mal_share_mode shareMode, const mal_device_id* pDeviceID, mal_IDirectSoundCapture** ppDirectSoundCapture)
{
    mal_assert(pContext != NULL);
    mal_assert(ppDirectSoundCapture != NULL);

    // Everything is shared in capture mode by the looks of it.
    (void)shareMode;

    *ppDirectSoundCapture = NULL;
    mal_IDirectSoundCapture* pDirectSoundCapture = NULL;

    if (FAILED(((mal_DirectSoundCaptureCreateProc)pContext->dsound.DirectSoundCaptureCreate)((pDeviceID == NULL) ? NULL : (const GUID*)pDeviceID->dsound, &pDirectSoundCapture, NULL))) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[DirectSound] DirectSoundCaptureCreate() failed for capture device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    *ppDirectSoundCapture = pDirectSoundCapture;
    return MAL_SUCCESS;
}

mal_result mal_context_get_format_info_for_IDirectSoundCapture__dsound(mal_context* pContext, mal_IDirectSoundCapture* pDirectSoundCapture, WORD* pChannels, WORD* pBitsPerSample, DWORD* pSampleRate)
{
    mal_assert(pContext != NULL);
    mal_assert(pDirectSoundCapture != NULL);

    if (pChannels) {
        *pChannels = 0;
    }
    if (pBitsPerSample) {
        *pBitsPerSample = 0;
    }
    if (pSampleRate) {
        *pSampleRate = 0;
    }

    MAL_DSCCAPS caps;
    mal_zero_object(&caps);
    caps.dwSize = sizeof(caps);
    if (FAILED(mal_IDirectSoundCapture_GetCaps(pDirectSoundCapture, &caps))) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCapture_GetCaps() failed for capture device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    if (pChannels) {
        *pChannels = (WORD)caps.dwChannels;
    }

    // The device can support multiple formats. We just go through the different formats in order of priority and
    // pick the first one. This the same type of system as the WinMM backend.
    WORD bitsPerSample = 16;
    DWORD sampleRate = 48000;

    if (caps.dwChannels == 1) {
        if ((caps.dwFormats & WAVE_FORMAT_48M16) != 0) {
            sampleRate = 48000;
        } else if ((caps.dwFormats & WAVE_FORMAT_44M16) != 0) {
            sampleRate = 44100;
        } else if ((caps.dwFormats & WAVE_FORMAT_2M16) != 0) {
            sampleRate = 22050;
        } else if ((caps.dwFormats & WAVE_FORMAT_1M16) != 0) {
            sampleRate = 11025;
        } else if ((caps.dwFormats & WAVE_FORMAT_96M16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((caps.dwFormats & WAVE_FORMAT_48M08) != 0) {
                sampleRate = 48000;
            } else if ((caps.dwFormats & WAVE_FORMAT_44M08) != 0) {
                sampleRate = 44100;
            } else if ((caps.dwFormats & WAVE_FORMAT_2M08) != 0) {
                sampleRate = 22050;
            } else if ((caps.dwFormats & WAVE_FORMAT_1M08) != 0) {
                sampleRate = 11025;
            } else if ((caps.dwFormats & WAVE_FORMAT_96M08) != 0) {
                sampleRate = 96000;
            } else {
                bitsPerSample = 16;  // Didn't find it. Just fall back to 16-bit.
            }
        }
    } else if (caps.dwChannels == 2) {
        if ((caps.dwFormats & WAVE_FORMAT_48S16) != 0) {
            sampleRate = 48000;
        } else if ((caps.dwFormats & WAVE_FORMAT_44S16) != 0) {
            sampleRate = 44100;
        } else if ((caps.dwFormats & WAVE_FORMAT_2S16) != 0) {
            sampleRate = 22050;
        } else if ((caps.dwFormats & WAVE_FORMAT_1S16) != 0) {
            sampleRate = 11025;
        } else if ((caps.dwFormats & WAVE_FORMAT_96S16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((caps.dwFormats & WAVE_FORMAT_48S08) != 0) {
                sampleRate = 48000;
            } else if ((caps.dwFormats & WAVE_FORMAT_44S08) != 0) {
                sampleRate = 44100;
            } else if ((caps.dwFormats & WAVE_FORMAT_2S08) != 0) {
                sampleRate = 22050;
            } else if ((caps.dwFormats & WAVE_FORMAT_1S08) != 0) {
                sampleRate = 11025;
            } else if ((caps.dwFormats & WAVE_FORMAT_96S08) != 0) {
                sampleRate = 96000;
            } else {
                bitsPerSample = 16;  // Didn't find it. Just fall back to 16-bit.
            }
        }
    }

    if (pBitsPerSample) {
        *pBitsPerSample = bitsPerSample;
    }
    if (pSampleRate) {
        *pSampleRate = sampleRate;
    }

    return MAL_SUCCESS;
}

mal_bool32 mal_context_is_device_id_equal__dsound(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return memcmp(pID0->dsound, pID1->dsound, sizeof(pID0->dsound)) == 0;
}


typedef struct
{
    mal_context* pContext;
    mal_device_type deviceType;
    mal_enum_devices_callback_proc callback;
    void* pUserData;
    mal_bool32 terminated;
} mal_context_enumerate_devices_callback_data__dsound;

BOOL CALLBACK mal_context_enumerate_devices_callback__dsound(LPGUID lpGuid, LPCSTR lpcstrDescription, LPCSTR lpcstrModule, LPVOID lpContext)
{
    (void)lpcstrModule;

    mal_context_enumerate_devices_callback_data__dsound* pData = (mal_context_enumerate_devices_callback_data__dsound*)lpContext;
    mal_assert(pData != NULL);

    mal_device_info deviceInfo;
    mal_zero_object(&deviceInfo);

    // ID.
    if (lpGuid != NULL) {
        mal_copy_memory(deviceInfo.id.dsound, lpGuid, 16);
    } else {
        mal_zero_memory(deviceInfo.id.dsound, 16);
    }

    // Name / Description
    mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), lpcstrDescription, (size_t)-1);


    // Call the callback function, but make sure we stop enumerating if the callee requested so.
    pData->terminated = !pData->callback(pData->pContext, pData->deviceType, &deviceInfo, pData->pUserData);
    if (pData->terminated) {
        return FALSE;   // Stop enumeration.
    } else {
        return TRUE;    // Continue enumeration.
    }
}

mal_result mal_context_enumerate_devices__dsound(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    mal_context_enumerate_devices_callback_data__dsound data;
    data.pContext = pContext;
    data.callback = callback;
    data.pUserData = pUserData;
    data.terminated = MAL_FALSE;

    // Playback.
    if (!data.terminated) {
        data.deviceType = mal_device_type_playback;
        ((mal_DirectSoundEnumerateAProc)pContext->dsound.DirectSoundEnumerateA)(mal_context_enumerate_devices_callback__dsound, &data);
    }

    // Capture.
    if (!data.terminated) {
        data.deviceType = mal_device_type_capture;
        ((mal_DirectSoundCaptureEnumerateAProc)pContext->dsound.DirectSoundCaptureEnumerateA)(mal_context_enumerate_devices_callback__dsound, &data);
    }

    return MAL_SUCCESS;
}


typedef struct
{
    const mal_device_id* pDeviceID;
    mal_device_info* pDeviceInfo;
    mal_bool32 found;
} mal_context_get_device_info_callback_data__dsound;

BOOL CALLBACK mal_context_get_device_info_callback__dsound(LPGUID lpGuid, LPCSTR lpcstrDescription, LPCSTR lpcstrModule, LPVOID lpContext)
{
    (void)lpcstrModule;

    mal_context_get_device_info_callback_data__dsound* pData = (mal_context_get_device_info_callback_data__dsound*)lpContext;
    mal_assert(pData != NULL);

    if ((pData->pDeviceID == NULL || mal_is_guid_equal(pData->pDeviceID->dsound, &MAL_GUID_NULL)) && (lpGuid == NULL || mal_is_guid_equal(lpGuid, &MAL_GUID_NULL))) {
        // Default device.
        mal_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), lpcstrDescription, (size_t)-1);
        pData->found = MAL_TRUE;
        return FALSE;   // Stop enumeration.
    } else {
        // Not the default device.
        if (lpGuid != NULL) {
            if (memcmp(pData->pDeviceID->dsound, lpGuid, sizeof(pData->pDeviceID->dsound)) == 0) {
                mal_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), lpcstrDescription, (size_t)-1);
                pData->found = MAL_TRUE;
                return FALSE;   // Stop enumeration.
            }
        }
    }

    return TRUE;
}

mal_result mal_context_get_device_info__dsound(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    (void)shareMode;

    if (pDeviceID != NULL) {
        // ID.
        mal_copy_memory(pDeviceInfo->id.dsound, pDeviceID->dsound, 16);

        // Name / Description. This is retrieved by enumerating over each device until we find that one that matches the input ID.
        mal_context_get_device_info_callback_data__dsound data;
        data.pDeviceID = pDeviceID;
        data.pDeviceInfo = pDeviceInfo;
        data.found = MAL_FALSE;
        if (deviceType == mal_device_type_playback) {
            ((mal_DirectSoundEnumerateAProc)pContext->dsound.DirectSoundEnumerateA)(mal_context_get_device_info_callback__dsound, &data);
        } else {
            ((mal_DirectSoundCaptureEnumerateAProc)pContext->dsound.DirectSoundCaptureEnumerateA)(mal_context_get_device_info_callback__dsound, &data);
        }

        if (!data.found) {
            return MAL_NO_DEVICE;
        }
    } else {
        // I don't think there's a way to get the name of the default device with DirectSound. In this case we just need to use defaults.

        // ID
        mal_zero_memory(pDeviceInfo->id.dsound, 16);

        // Name / Description/
        if (deviceType == mal_device_type_playback) {
            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        } else {
            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        }
    }

    // Retrieving detailed information is slightly different depending on the device type.
    if (deviceType == mal_device_type_playback) {
        // Playback.
        mal_IDirectSound* pDirectSound;
        mal_result result = mal_context_create_IDirectSound__dsound(pContext, shareMode, pDeviceID, &pDirectSound);
        if (result != MAL_SUCCESS) {
            return result;
        }

        MAL_DSCAPS caps;
        mal_zero_object(&caps);
        caps.dwSize = sizeof(caps);
        if (FAILED(mal_IDirectSound_GetCaps(pDirectSound, &caps))) {
            return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_GetCaps() failed for playback device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if ((caps.dwFlags & MAL_DSCAPS_PRIMARYSTEREO) != 0) {
            // It supports at least stereo, but could support more.
            WORD channels = 2;

            // Look at the speaker configuration to get a better idea on the channel count.
            DWORD speakerConfig;
            if (SUCCEEDED(mal_IDirectSound_GetSpeakerConfig(pDirectSound, &speakerConfig))) {
                mal_get_channels_from_speaker_config__dsound(speakerConfig, &channels, NULL);
            }

            pDeviceInfo->minChannels = channels;
            pDeviceInfo->maxChannels = channels;
        } else {
            // It does not support stereo, which means we are stuck with mono.
            pDeviceInfo->minChannels = 1;
            pDeviceInfo->maxChannels = 1;
        }

        // Sample rate.
        if ((caps.dwFlags & MAL_DSCAPS_CONTINUOUSRATE) != 0) {
            pDeviceInfo->minSampleRate = caps.dwMinSecondarySampleRate;
            pDeviceInfo->maxSampleRate = caps.dwMaxSecondarySampleRate;

            // On my machine the min and max sample rates can return 100 and 200000 respectively. I'd rather these be within
            // the range of our standard sample rates so I'm clamping.
            if (caps.dwMinSecondarySampleRate < MAL_MIN_SAMPLE_RATE && caps.dwMaxSecondarySampleRate >= MAL_MIN_SAMPLE_RATE) {
                pDeviceInfo->minSampleRate = MAL_MIN_SAMPLE_RATE;
            }
            if (caps.dwMaxSecondarySampleRate > MAL_MAX_SAMPLE_RATE && caps.dwMinSecondarySampleRate <= MAL_MAX_SAMPLE_RATE) {
                pDeviceInfo->maxSampleRate = MAL_MAX_SAMPLE_RATE;
            }
        } else {
            // Only supports a single sample rate. Set both min an max to the same thing. Do not clamp within the standard rates.
            pDeviceInfo->minSampleRate = caps.dwMaxSecondarySampleRate;
            pDeviceInfo->maxSampleRate = caps.dwMaxSecondarySampleRate;
        }

        // DirectSound can support all formats.
        pDeviceInfo->formatCount = mal_format_count - 1;    // Minus one because we don't want to include mal_format_unknown.
        for (mal_uint32 iFormat = 0; iFormat < pDeviceInfo->formatCount; ++iFormat) {
            pDeviceInfo->formats[iFormat] = (mal_format)(iFormat + 1);  // +1 to skip over mal_format_unknown.
        }

        mal_IDirectSound_Release(pDirectSound);
    } else {
        // Capture. This is a little different to playback due to the say the supported formats are reported. Technically capture
        // devices can support a number of different formats, but for simplicity and consistency with mal_device_init() I'm just
        // reporting the best format.
        mal_IDirectSoundCapture* pDirectSoundCapture;
        mal_result result = mal_context_create_IDirectSoundCapture__dsound(pContext, shareMode, pDeviceID, &pDirectSoundCapture);
        if (result != MAL_SUCCESS) {
            return result;
        }

        WORD channels;
        WORD bitsPerSample;
        DWORD sampleRate;
        result = mal_context_get_format_info_for_IDirectSoundCapture__dsound(pContext, pDirectSoundCapture, &channels, &bitsPerSample, &sampleRate);
        if (result != MAL_SUCCESS) {
            mal_IDirectSoundCapture_Release(pDirectSoundCapture);
            return result;
        }

        pDeviceInfo->minChannels = channels;
        pDeviceInfo->maxChannels = channels;
        pDeviceInfo->minSampleRate = sampleRate;
        pDeviceInfo->maxSampleRate = sampleRate;
        pDeviceInfo->formatCount = 1;
        if (bitsPerSample == 8) {
            pDeviceInfo->formats[0] = mal_format_u8;
        } else if (bitsPerSample == 16) {
            pDeviceInfo->formats[0] = mal_format_s16;
        } else if (bitsPerSample == 24) {
            pDeviceInfo->formats[0] = mal_format_s24;
        } else if (bitsPerSample == 32) {
            pDeviceInfo->formats[0] = mal_format_s32;
        } else {
            mal_IDirectSoundCapture_Release(pDirectSoundCapture);
            return MAL_FORMAT_NOT_SUPPORTED;
        }

        mal_IDirectSoundCapture_Release(pDirectSoundCapture);
    }

    return MAL_SUCCESS;
}


typedef struct
{
    mal_uint32 deviceCount;
    mal_uint32 infoCount;
    mal_device_info* pInfo;
} mal_device_enum_data__dsound;

BOOL CALLBACK mal_enum_devices_callback__dsound(LPGUID lpGuid, LPCSTR lpcstrDescription, LPCSTR lpcstrModule, LPVOID lpContext)
{
    (void)lpcstrModule;

    mal_device_enum_data__dsound* pData = (mal_device_enum_data__dsound*)lpContext;
    mal_assert(pData != NULL);

    if (pData->pInfo != NULL) {
        if (pData->infoCount > 0) {
            mal_zero_object(pData->pInfo);
            mal_strncpy_s(pData->pInfo->name, sizeof(pData->pInfo->name), lpcstrDescription, (size_t)-1);

            if (lpGuid != NULL) {
                mal_copy_memory(pData->pInfo->id.dsound, lpGuid, 16);
            } else {
                mal_zero_memory(pData->pInfo->id.dsound, 16);
            }

            pData->pInfo += 1;
            pData->infoCount -= 1;
            pData->deviceCount += 1;
        }
    } else {
        pData->deviceCount += 1;
    }

    return TRUE;
}

void mal_device_uninit__dsound(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->dsound.pNotify != NULL) {
        mal_IDirectSoundNotify_Release((mal_IDirectSoundNotify*)pDevice->dsound.pNotify);
    }

    if (pDevice->dsound.hStopEvent) {
        CloseHandle(pDevice->dsound.hStopEvent);
    }
    for (mal_uint32 i = 0; i < pDevice->periods; ++i) {
        if (pDevice->dsound.pNotifyEvents[i]) {
            CloseHandle(pDevice->dsound.pNotifyEvents[i]);
        }
    }

    if (pDevice->dsound.pCaptureBuffer != NULL) {
        mal_IDirectSoundCaptureBuffer_Release((mal_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer);
    }
    if (pDevice->dsound.pCapture != NULL) {
        mal_IDirectSoundCapture_Release((mal_IDirectSoundCapture*)pDevice->dsound.pCapture);
    }

    if (pDevice->dsound.pPlaybackBuffer != NULL) {
        mal_IDirectSoundBuffer_Release((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer);
    }
    if (pDevice->dsound.pPlaybackPrimaryBuffer != NULL) {
        mal_IDirectSoundBuffer_Release((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackPrimaryBuffer);
    }
    if (pDevice->dsound.pPlayback != NULL) {
        mal_IDirectSound_Release((mal_IDirectSound*)pDevice->dsound.pPlayback);
    }
}

mal_result mal_device_init__dsound(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pContext;

    mal_assert(pDevice != NULL);
    mal_zero_object(&pDevice->dsound);

    // Check that we have a valid format.
    GUID subformat;
    switch (pConfig->format)
    {
        case mal_format_u8:
        case mal_format_s16:
        case mal_format_s24:
        //case mal_format_s24_32:
        case mal_format_s32:
        {
            subformat = MAL_GUID_KSDATAFORMAT_SUBTYPE_PCM;
        } break;

        case mal_format_f32:
        {
            subformat = MAL_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT;
        } break;

        default:
        return MAL_FORMAT_NOT_SUPPORTED;
    }

    WAVEFORMATEXTENSIBLE wf;
    mal_zero_object(&wf);
    wf.Format.cbSize               = sizeof(wf);
    wf.Format.wFormatTag           = WAVE_FORMAT_EXTENSIBLE;
    wf.Format.nChannels            = (WORD)pConfig->channels;
    wf.Format.nSamplesPerSec       = (DWORD)pConfig->sampleRate;
    wf.Format.wBitsPerSample       = (WORD)mal_get_bytes_per_sample(pConfig->format)*8;
    wf.Format.nBlockAlign          = (wf.Format.nChannels * wf.Format.wBitsPerSample) / 8;
    wf.Format.nAvgBytesPerSec      = wf.Format.nBlockAlign * wf.Format.nSamplesPerSec;
    wf.Samples.wValidBitsPerSample = wf.Format.wBitsPerSample;
    wf.dwChannelMask               = mal_channel_map_to_channel_mask__win32(pConfig->channelMap, pConfig->channels);
    wf.SubFormat                   = subformat;

    DWORD bufferSizeInBytes = 0;

    // Unfortunately DirectSound uses different APIs and data structures for playback and catpure devices :(
    if (type == mal_device_type_playback) {
        mal_result result = mal_context_create_IDirectSound__dsound(pContext, pConfig->shareMode, pDeviceID, (mal_IDirectSound**)&pDevice->dsound.pPlayback);
        if (result != MAL_SUCCESS) {
            mal_device_uninit__dsound(pDevice);
            return result;
        }

        MAL_DSBUFFERDESC descDSPrimary;
        mal_zero_object(&descDSPrimary);
        descDSPrimary.dwSize  = sizeof(MAL_DSBUFFERDESC);
        descDSPrimary.dwFlags = MAL_DSBCAPS_PRIMARYBUFFER | MAL_DSBCAPS_CTRLVOLUME;
        if (FAILED(mal_IDirectSound_CreateSoundBuffer((mal_IDirectSound*)pDevice->dsound.pPlayback, &descDSPrimary, (mal_IDirectSoundBuffer**)&pDevice->dsound.pPlaybackPrimaryBuffer, NULL))) {
            mal_device_uninit__dsound(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_CreateSoundBuffer() failed for playback device's primary buffer.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }


        // We may want to make some adjustments to the format if we are using defaults.
        MAL_DSCAPS caps;
        mal_zero_object(&caps);
        caps.dwSize = sizeof(caps);
        if (FAILED(mal_IDirectSound_GetCaps((mal_IDirectSound*)pDevice->dsound.pPlayback, &caps))) {
            mal_device_uninit__dsound(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_GetCaps() failed for playback device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (pDevice->usingDefaultChannels) {
            if ((caps.dwFlags & MAL_DSCAPS_PRIMARYSTEREO) != 0) {
                // It supports at least stereo, but could support more.
                wf.Format.nChannels = 2;

                // Look at the speaker configuration to get a better idea on the channel count.
                DWORD speakerConfig;
                if (SUCCEEDED(mal_IDirectSound_GetSpeakerConfig((mal_IDirectSound*)pDevice->dsound.pPlayback, &speakerConfig))) {
                    mal_get_channels_from_speaker_config__dsound(speakerConfig, &wf.Format.nChannels, &wf.dwChannelMask);
                }
            } else {
                // It does not support stereo, which means we are stuck with mono.
                wf.Format.nChannels = 1;
            }
        }

        if (pDevice->usingDefaultSampleRate) {
            // We base the sample rate on the values returned by GetCaps().
            if ((caps.dwFlags & MAL_DSCAPS_CONTINUOUSRATE) != 0) {
                wf.Format.nSamplesPerSec = mal_get_best_sample_rate_within_range(caps.dwMinSecondarySampleRate, caps.dwMaxSecondarySampleRate);
            } else {
                wf.Format.nSamplesPerSec = caps.dwMaxSecondarySampleRate;
            }
        }

        wf.Format.nBlockAlign     = (wf.Format.nChannels * wf.Format.wBitsPerSample) / 8;
        wf.Format.nAvgBytesPerSec = wf.Format.nBlockAlign * wf.Format.nSamplesPerSec;

        // From MSDN:
        //
        // The method succeeds even if the hardware does not support the requested format; DirectSound sets the buffer to the closest
        // supported format. To determine whether this has happened, an application can call the GetFormat method for the primary buffer
        // and compare the result with the format that was requested with the SetFormat method.
        if (FAILED(mal_IDirectSoundBuffer_SetFormat((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackPrimaryBuffer, (WAVEFORMATEX*)&wf))) {
            mal_device_uninit__dsound(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] Failed to set format of playback device's primary buffer.", MAL_FORMAT_NOT_SUPPORTED);
        }

        // Get the _actual_ properties of the buffer.
        char rawdata[1024];
        WAVEFORMATEXTENSIBLE* pActualFormat = (WAVEFORMATEXTENSIBLE*)rawdata;
        if (FAILED(mal_IDirectSoundBuffer_GetFormat((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackPrimaryBuffer, (WAVEFORMATEX*)pActualFormat, sizeof(rawdata), NULL))) {
            mal_device_uninit__dsound(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] Failed to retrieve the actual format of the playback device's primary buffer.", MAL_FORMAT_NOT_SUPPORTED);
        }

        pDevice->internalFormat = mal_format_from_WAVEFORMATEX((WAVEFORMATEX*)pActualFormat);
        pDevice->internalChannels = pActualFormat->Format.nChannels;
        pDevice->internalSampleRate = pActualFormat->Format.nSamplesPerSec;

        // Get the internal channel map based on the channel mask.
        if (pActualFormat->Format.wFormatTag == WAVE_FORMAT_EXTENSIBLE) {
            mal_channel_mask_to_channel_map__win32(pActualFormat->dwChannelMask, pDevice->internalChannels, pDevice->internalChannelMap);
        } else {
            mal_channel_mask_to_channel_map__win32(wf.dwChannelMask, pDevice->internalChannels, pDevice->internalChannelMap);
        }

        // We need to wait until we know the sample rate before we can calculate the buffer size.
        if (pDevice->bufferSizeInFrames == 0) {
            pDevice->bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, pDevice->internalSampleRate);
        }

        bufferSizeInBytes = pDevice->bufferSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat);



        // Meaning of dwFlags (from MSDN):
        //
        // DSBCAPS_CTRLPOSITIONNOTIFY
        //   The buffer has position notification capability.
        //
        // DSBCAPS_GLOBALFOCUS
        //   With this flag set, an application using DirectSound can continue to play its buffers if the user switches focus to
        //   another application, even if the new application uses DirectSound.
        //
        // DSBCAPS_GETCURRENTPOSITION2
        //   In the first version of DirectSound, the play cursor was significantly ahead of the actual playing sound on emulated
        //   sound cards; it was directly behind the write cursor. Now, if the DSBCAPS_GETCURRENTPOSITION2 flag is specified, the
        //   application can get a more accurate play cursor.
        MAL_DSBUFFERDESC descDS;
        mal_zero_object(&descDS);
        descDS.dwSize = sizeof(descDS);
        descDS.dwFlags = MAL_DSBCAPS_CTRLPOSITIONNOTIFY | MAL_DSBCAPS_GLOBALFOCUS | MAL_DSBCAPS_GETCURRENTPOSITION2;
        descDS.dwBufferBytes = bufferSizeInBytes;
        descDS.lpwfxFormat = (WAVEFORMATEX*)&wf;
        if (FAILED(mal_IDirectSound_CreateSoundBuffer((mal_IDirectSound*)pDevice->dsound.pPlayback, &descDS, (mal_IDirectSoundBuffer**)&pDevice->dsound.pPlaybackBuffer, NULL))) {
            mal_device_uninit__dsound(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_CreateSoundBuffer() failed for playback device's secondary buffer.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        // Notifications are set up via a DIRECTSOUNDNOTIFY object which is retrieved from the buffer.
        if (FAILED(mal_IDirectSoundBuffer_QueryInterface((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, &MAL_GUID_IID_DirectSoundNotify, (void**)&pDevice->dsound.pNotify))) {
            mal_device_uninit__dsound(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_QueryInterface() failed for playback device's IDirectSoundNotify object.", MAL_API_NOT_FOUND);
        }
    } else {
        // The default buffer size is treated slightly differently for DirectSound which, for some reason, seems to
        // have worse latency with capture than playback (sometimes _much_ worse).
        if (pDevice->usingDefaultBufferSize) {
            pDevice->bufferSizeInFrames *= 2; // <-- Might need to fiddle with this to find a more ideal value. May even be able to just add a fixed amount rather than scaling.
        }

        mal_result result = mal_context_create_IDirectSoundCapture__dsound(pContext, pConfig->shareMode, pDeviceID, (mal_IDirectSoundCapture**)&pDevice->dsound.pCapture);
        if (result != MAL_SUCCESS) {
            mal_device_uninit__dsound(pDevice);
            return result;
        }

        result = mal_context_get_format_info_for_IDirectSoundCapture__dsound(pContext, (mal_IDirectSoundCapture*)pDevice->dsound.pCapture, &wf.Format.nChannels, &wf.Format.wBitsPerSample, &wf.Format.nSamplesPerSec);
        if (result != MAL_SUCCESS) {
            mal_device_uninit__dsound(pDevice);
            return result;
        }

        wf.Format.nBlockAlign          = (wf.Format.nChannels * wf.Format.wBitsPerSample) / 8;
        wf.Format.nAvgBytesPerSec      = wf.Format.nBlockAlign * wf.Format.nSamplesPerSec;
        wf.Samples.wValidBitsPerSample = wf.Format.wBitsPerSample;
        wf.SubFormat                   = MAL_GUID_KSDATAFORMAT_SUBTYPE_PCM;

        if (pDevice->bufferSizeInFrames == 0) {
            pDevice->bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, wf.Format.nSamplesPerSec);
        }

        bufferSizeInBytes = pDevice->bufferSizeInFrames * wf.Format.nChannels * mal_get_bytes_per_sample(pDevice->format);

        MAL_DSCBUFFERDESC descDS;
        mal_zero_object(&descDS);
        descDS.dwSize = sizeof(descDS);
        descDS.dwFlags = 0;
        descDS.dwBufferBytes = bufferSizeInBytes;
        descDS.lpwfxFormat = (WAVEFORMATEX*)&wf;
        if (FAILED(mal_IDirectSoundCapture_CreateCaptureBuffer((mal_IDirectSoundCapture*)pDevice->dsound.pCapture, &descDS, (mal_IDirectSoundCaptureBuffer**)&pDevice->dsound.pCaptureBuffer, NULL))) {
            mal_device_uninit__dsound(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCapture_CreateCaptureBuffer() failed for capture device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        // Get the _actual_ properties of the buffer.
        char rawdata[1024];
        WAVEFORMATEXTENSIBLE* pActualFormat = (WAVEFORMATEXTENSIBLE*)rawdata;
        if (FAILED(mal_IDirectSoundCaptureBuffer_GetFormat((mal_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, (WAVEFORMATEX*)pActualFormat, sizeof(rawdata), NULL))) {
            mal_device_uninit__dsound(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] Failed to retrieve the actual format of the capture device's buffer.", MAL_FORMAT_NOT_SUPPORTED);
        }

        pDevice->internalFormat = mal_format_from_WAVEFORMATEX((WAVEFORMATEX*)pActualFormat);
        pDevice->internalChannels = pActualFormat->Format.nChannels;
        pDevice->internalSampleRate = pActualFormat->Format.nSamplesPerSec;

        // Get the internal channel map based on the channel mask.
        if (pActualFormat->Format.wFormatTag == WAVE_FORMAT_EXTENSIBLE) {
            mal_channel_mask_to_channel_map__win32(pActualFormat->dwChannelMask, pDevice->internalChannels, pDevice->internalChannelMap);
        } else {
            mal_channel_mask_to_channel_map__win32(wf.dwChannelMask, pDevice->internalChannels, pDevice->internalChannelMap);
        }


        // Notifications are set up via a DIRECTSOUNDNOTIFY object which is retrieved from the buffer.
        if (FAILED(mal_IDirectSoundCaptureBuffer_QueryInterface((mal_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, &MAL_GUID_IID_DirectSoundNotify, (void**)&pDevice->dsound.pNotify))) {
            mal_device_uninit__dsound(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCaptureBuffer_QueryInterface() failed for capture device's IDirectSoundNotify object.", MAL_API_NOT_FOUND);
        }
    }

    // We need a notification for each period. The notification offset is slightly different depending on whether or not the
    // device is a playback or capture device. For a playback device we want to be notified when a period just starts playing,
    // whereas for a capture device we want to be notified when a period has just _finished_ capturing.
    mal_uint32 periodSizeInBytes = pDevice->bufferSizeInFrames / pDevice->periods;
    MAL_DSBPOSITIONNOTIFY notifyPoints[MAL_MAX_PERIODS_DSOUND];  // One notification event for each period.
    for (mal_uint32 i = 0; i < pDevice->periods; ++i) {
        pDevice->dsound.pNotifyEvents[i] = CreateEventA(NULL, FALSE, FALSE, NULL);
        if (pDevice->dsound.pNotifyEvents[i] == NULL) {
            mal_device_uninit__dsound(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] Failed to create event for buffer notifications.", MAL_FAILED_TO_CREATE_EVENT);
        }

        // The notification offset is in bytes.
        notifyPoints[i].dwOffset = i * periodSizeInBytes;
        notifyPoints[i].hEventNotify = pDevice->dsound.pNotifyEvents[i];
    }

    if (FAILED(mal_IDirectSoundNotify_SetNotificationPositions((mal_IDirectSoundNotify*)pDevice->dsound.pNotify, pDevice->periods, notifyPoints))) {
        mal_device_uninit__dsound(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundNotify_SetNotificationPositions() failed.", MAL_FAILED_TO_CREATE_EVENT);
    }

    // When the device is playing the worker thread will be waiting on a bunch of notification events. To return from
    // this wait state we need to signal a special event.
    pDevice->dsound.hStopEvent = CreateEventA(NULL, FALSE, FALSE, NULL);
    if (pDevice->dsound.hStopEvent == NULL) {
        mal_device_uninit__dsound(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] Failed to create event for main loop break notification.", MAL_FAILED_TO_CREATE_EVENT);
    }

    return MAL_SUCCESS;
}


mal_result mal_device__start_backend__dsound(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->type == mal_device_type_playback) {
        // Before playing anything we need to grab an initial group of samples from the client.
        mal_uint32 framesToRead = pDevice->bufferSizeInFrames / pDevice->periods;
        mal_uint32 desiredLockSize = framesToRead * pDevice->channels * mal_get_bytes_per_sample(pDevice->format);

        void* pLockPtr;
        DWORD actualLockSize;
        void* pLockPtr2;
        DWORD actualLockSize2;
        if (SUCCEEDED(mal_IDirectSoundBuffer_Lock((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0, desiredLockSize, &pLockPtr, &actualLockSize, &pLockPtr2, &actualLockSize2, 0))) {
            framesToRead = actualLockSize / mal_get_bytes_per_sample(pDevice->format) / pDevice->channels;
            mal_device__read_frames_from_client(pDevice, framesToRead, pLockPtr);
            mal_IDirectSoundBuffer_Unlock((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, pLockPtr, actualLockSize, pLockPtr2, actualLockSize2);

            pDevice->dsound.lastProcessedFrame = framesToRead;
            if (FAILED(mal_IDirectSoundBuffer_Play((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0, 0, MAL_DSBPLAY_LOOPING))) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Play() failed.", MAL_FAILED_TO_START_BACKEND_DEVICE);
            }
        } else {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Lock() failed.", MAL_FAILED_TO_MAP_DEVICE_BUFFER);
        }
    } else {
        if (FAILED(mal_IDirectSoundCaptureBuffer_Start((mal_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, MAL_DSCBSTART_LOOPING))) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCaptureBuffer_Start() failed.", MAL_FAILED_TO_START_BACKEND_DEVICE);
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__dsound(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->type == mal_device_type_playback) {
        if (FAILED(mal_IDirectSoundBuffer_Stop((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer))) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Stop() failed.", MAL_FAILED_TO_STOP_BACKEND_DEVICE);
        }

        mal_IDirectSoundBuffer_SetCurrentPosition((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0);
    } else {
        if (FAILED(mal_IDirectSoundCaptureBuffer_Stop((mal_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer))) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCaptureBuffer_Stop() failed.", MAL_FAILED_TO_STOP_BACKEND_DEVICE);
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_device__break_main_loop__dsound(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // The main loop will be waiting on a bunch of events via the WaitForMultipleObjects() API. One of those events
    // is a special event we use for forcing that function to return.
    pDevice->dsound.breakFromMainLoop = MAL_TRUE;
    SetEvent(pDevice->dsound.hStopEvent);
    return MAL_SUCCESS;
}

mal_bool32 mal_device__get_current_frame__dsound(mal_device* pDevice, mal_uint32* pCurrentPos)
{
    mal_assert(pDevice != NULL);
    mal_assert(pCurrentPos != NULL);
    *pCurrentPos = 0;

    DWORD dwCurrentPosition;
    if (pDevice->type == mal_device_type_playback) {
        if (FAILED(mal_IDirectSoundBuffer_GetCurrentPosition((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, NULL, &dwCurrentPosition))) {
            return MAL_FALSE;
        }
    } else {
        if (FAILED(mal_IDirectSoundCaptureBuffer_GetCurrentPosition((mal_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, &dwCurrentPosition, NULL))) {
            return MAL_FALSE;
        }
    }

    *pCurrentPos = (mal_uint32)dwCurrentPosition / mal_get_bytes_per_sample(pDevice->format) / pDevice->channels;
    return MAL_TRUE;
}

mal_uint32 mal_device__get_available_frames__dsound(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_uint32 currentFrame;
    if (!mal_device__get_current_frame__dsound(pDevice, &currentFrame)) {
        return 0;
    }

    // In a playback device the last processed frame should always be ahead of the current frame. The space between
    // the last processed and current frame (moving forward, starting from the last processed frame) is the amount
    // of space available to write.
    //
    // For a recording device it's the other way around - the last processed frame is always _behind_ the current
    // frame and the space between is the available space.
    mal_uint32 totalFrameCount = pDevice->bufferSizeInFrames;
    if (pDevice->type == mal_device_type_playback) {
        mal_uint32 committedBeg = currentFrame;
        mal_uint32 committedEnd;
        committedEnd = pDevice->dsound.lastProcessedFrame;
        if (committedEnd <= committedBeg) {
            committedEnd += totalFrameCount;
        }

        mal_uint32 committedSize = (committedEnd - committedBeg);
        mal_assert(committedSize <= totalFrameCount);

        return totalFrameCount - committedSize;
    } else {
        mal_uint32 validBeg = pDevice->dsound.lastProcessedFrame;
        mal_uint32 validEnd = currentFrame;
        if (validEnd < validBeg) {
            validEnd += totalFrameCount;        // Wrap around.
        }

        mal_uint32 validSize = (validEnd - validBeg);
        mal_assert(validSize <= totalFrameCount);

        return validSize;
    }
}

mal_uint32 mal_device__wait_for_frames__dsound(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // The timeout to use for putting the thread to sleep is based on the size of the buffer and the period count.
    DWORD timeoutInMilliseconds = (pDevice->bufferSizeInFrames / (pDevice->sampleRate/1000)) / pDevice->periods;
    if (timeoutInMilliseconds < 1) {
        timeoutInMilliseconds = 1;
    }

    unsigned int eventCount = pDevice->periods + 1;
    HANDLE pEvents[MAL_MAX_PERIODS_DSOUND + 1];   // +1 for the stop event.
    mal_copy_memory(pEvents, pDevice->dsound.pNotifyEvents, sizeof(HANDLE) * pDevice->periods);
    pEvents[eventCount-1] = pDevice->dsound.hStopEvent;

    while (!pDevice->dsound.breakFromMainLoop) {
        mal_uint32 framesAvailable = mal_device__get_available_frames__dsound(pDevice);
        if (framesAvailable > 0) {
            return framesAvailable;
        }

        // If we get here it means we weren't able to find any frames. We'll just wait here for a bit.
        WaitForMultipleObjects(eventCount, pEvents, FALSE, timeoutInMilliseconds);
    }

    // We'll get here if the loop was terminated. Just return whatever's available.
    return mal_device__get_available_frames__dsound(pDevice);
}

mal_result mal_device__main_loop__dsound(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // Make sure the stop event is not signaled to ensure we don't end up immediately returning from WaitForMultipleObjects().
    ResetEvent(pDevice->dsound.hStopEvent);

    pDevice->dsound.breakFromMainLoop = MAL_FALSE;
    while (!pDevice->dsound.breakFromMainLoop) {
        mal_uint32 framesAvailable = mal_device__wait_for_frames__dsound(pDevice);
        if (framesAvailable == 0) {
            continue;
        }

        // If it's a playback device, don't bother grabbing more data if the device is being stopped.
        if (pDevice->dsound.breakFromMainLoop && pDevice->type == mal_device_type_playback) {
            return MAL_FALSE;
        }

        DWORD lockOffset = pDevice->dsound.lastProcessedFrame * pDevice->channels * mal_get_bytes_per_sample(pDevice->format);
        DWORD lockSize   = framesAvailable * pDevice->channels * mal_get_bytes_per_sample(pDevice->format);

        if (pDevice->type == mal_device_type_playback) {
            void* pLockPtr;
            DWORD actualLockSize;
            void* pLockPtr2;
            DWORD actualLockSize2;
            if (FAILED(mal_IDirectSoundBuffer_Lock((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, lockOffset, lockSize, &pLockPtr, &actualLockSize, &pLockPtr2, &actualLockSize2, 0))) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Lock() failed.", MAL_FAILED_TO_MAP_DEVICE_BUFFER);
            }

            mal_uint32 frameCount = actualLockSize / mal_get_bytes_per_sample(pDevice->format) / pDevice->channels;
            mal_device__read_frames_from_client(pDevice, frameCount, pLockPtr);
            pDevice->dsound.lastProcessedFrame = (pDevice->dsound.lastProcessedFrame + frameCount) % pDevice->bufferSizeInFrames;

            mal_IDirectSoundBuffer_Unlock((mal_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, pLockPtr, actualLockSize, pLockPtr2, actualLockSize2);
        } else {
            void* pLockPtr;
            DWORD actualLockSize;
            void* pLockPtr2;
            DWORD actualLockSize2;
            if (FAILED(mal_IDirectSoundCaptureBuffer_Lock((mal_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, lockOffset, lockSize, &pLockPtr, &actualLockSize, &pLockPtr2, &actualLockSize2, 0))) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCaptureBuffer_Lock() failed.", MAL_FAILED_TO_MAP_DEVICE_BUFFER);
            }

            mal_uint32 frameCount = actualLockSize / mal_get_bytes_per_sample(pDevice->format) / pDevice->channels;
            mal_device__send_frames_to_client(pDevice, frameCount, pLockPtr);
            pDevice->dsound.lastProcessedFrame = (pDevice->dsound.lastProcessedFrame + frameCount) % pDevice->bufferSizeInFrames;

            mal_IDirectSoundCaptureBuffer_Unlock((mal_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, pLockPtr, actualLockSize, pLockPtr2, actualLockSize2);
        }
    }

    return MAL_SUCCESS;
}


mal_result mal_context_uninit__dsound(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_dsound);

    mal_dlclose(pContext->dsound.hDSoundDLL);

    return MAL_SUCCESS;
}

mal_result mal_context_init__dsound(mal_context* pContext)
{
    mal_assert(pContext != NULL);

    pContext->dsound.hDSoundDLL = mal_dlopen("dsound.dll");
    if (pContext->dsound.hDSoundDLL == NULL) {
        return MAL_API_NOT_FOUND;
    }

    pContext->dsound.DirectSoundCreate            = mal_dlsym(pContext->dsound.hDSoundDLL, "DirectSoundCreate");
    pContext->dsound.DirectSoundEnumerateA        = mal_dlsym(pContext->dsound.hDSoundDLL, "DirectSoundEnumerateA");
    pContext->dsound.DirectSoundCaptureCreate     = mal_dlsym(pContext->dsound.hDSoundDLL, "DirectSoundCaptureCreate");
    pContext->dsound.DirectSoundCaptureEnumerateA = mal_dlsym(pContext->dsound.hDSoundDLL, "DirectSoundCaptureEnumerateA");

    pContext->onUninit              = mal_context_uninit__dsound;
    pContext->onDeviceIDEqual       = mal_context_is_device_id_equal__dsound;
    pContext->onEnumDevices         = mal_context_enumerate_devices__dsound;
    pContext->onGetDeviceInfo       = mal_context_get_device_info__dsound;
    pContext->onDeviceInit          = mal_device_init__dsound;
    pContext->onDeviceUninit        = mal_device_uninit__dsound;
    pContext->onDeviceStart         = mal_device__start_backend__dsound;
    pContext->onDeviceStop          = mal_device__stop_backend__dsound;
    pContext->onDeviceBreakMainLoop = mal_device__break_main_loop__dsound;
    pContext->onDeviceMainLoop      = mal_device__main_loop__dsound;

    return MAL_SUCCESS;
}
#endif



///////////////////////////////////////////////////////////////////////////////
//
// WinMM Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_WINMM

// Some older compilers don't have WAVEOUTCAPS2A and WAVEINCAPS2A, so we'll need to write this ourselves. These structures
// are exactly the same as the older ones but they have a few GUIDs for manufacturer/product/name identification. I'm keeping
// the names the same as the Win32 library for consistency, but namespaced to avoid naming conflicts with the Win32 version.
typedef struct
{
    WORD wMid;
    WORD wPid;
    MMVERSION vDriverVersion;
    CHAR szPname[MAXPNAMELEN];
    DWORD dwFormats;
    WORD wChannels;
    WORD wReserved1;
    DWORD dwSupport;
    GUID ManufacturerGuid;
    GUID ProductGuid;
    GUID NameGuid;
} MAL_WAVEOUTCAPS2A;
typedef struct
{
    WORD wMid;
    WORD wPid;
    MMVERSION vDriverVersion;
    CHAR szPname[MAXPNAMELEN];
    DWORD dwFormats;
    WORD wChannels;
    WORD wReserved1;
    GUID ManufacturerGuid;
    GUID ProductGuid;
    GUID NameGuid;
} MAL_WAVEINCAPS2A;

typedef UINT     (WINAPI * MAL_PFN_waveOutGetNumDevs)(void);
typedef MMRESULT (WINAPI * MAL_PFN_waveOutGetDevCapsA)(mal_uintptr uDeviceID, LPWAVEOUTCAPSA pwoc, UINT cbwoc);
typedef MMRESULT (WINAPI * MAL_PFN_waveOutOpen)(LPHWAVEOUT phwo, UINT uDeviceID, LPCWAVEFORMATEX pwfx, DWORD_PTR dwCallback, DWORD_PTR dwInstance, DWORD fdwOpen);
typedef MMRESULT (WINAPI * MAL_PFN_waveOutClose)(HWAVEOUT hwo);
typedef MMRESULT (WINAPI * MAL_PFN_waveOutPrepareHeader)(HWAVEOUT hwo, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MAL_PFN_waveOutUnprepareHeader)(HWAVEOUT hwo, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MAL_PFN_waveOutWrite)(HWAVEOUT hwo, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MAL_PFN_waveOutReset)(HWAVEOUT hwo);
typedef UINT     (WINAPI * MAL_PFN_waveInGetNumDevs)(void);
typedef MMRESULT (WINAPI * MAL_PFN_waveInGetDevCapsA)(mal_uintptr uDeviceID, LPWAVEINCAPSA pwic, UINT cbwic);
typedef MMRESULT (WINAPI * MAL_PFN_waveInOpen)(LPHWAVEIN phwi, UINT uDeviceID, LPCWAVEFORMATEX pwfx, DWORD_PTR dwCallback, DWORD_PTR dwInstance, DWORD fdwOpen);
typedef MMRESULT (WINAPI * MAL_PFN_waveInClose)(HWAVEIN hwi);
typedef MMRESULT (WINAPI * MAL_PFN_waveInPrepareHeader)(HWAVEIN hwi, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MAL_PFN_waveInUnprepareHeader)(HWAVEIN hwi, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MAL_PFN_waveInAddBuffer)(HWAVEIN hwi, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MAL_PFN_waveInStart)(HWAVEIN hwi);
typedef MMRESULT (WINAPI * MAL_PFN_waveInReset)(HWAVEIN hwi);

mal_result mal_result_from_MMRESULT(MMRESULT resultMM)
{
    switch (resultMM) {
        case MMSYSERR_NOERROR:      return MAL_SUCCESS;
        case MMSYSERR_BADDEVICEID:  return MAL_INVALID_ARGS;
        case MMSYSERR_INVALHANDLE:  return MAL_INVALID_ARGS;
        case MMSYSERR_NOMEM:        return MAL_OUT_OF_MEMORY;
        case MMSYSERR_INVALFLAG:    return MAL_INVALID_ARGS;
        case MMSYSERR_INVALPARAM:   return MAL_INVALID_ARGS;
        case MMSYSERR_HANDLEBUSY:   return MAL_DEVICE_BUSY;
        case MMSYSERR_ERROR:        return MAL_ERROR;
        default:                    return MAL_ERROR;
    }
}

char* mal_find_last_character(char* str, char ch)
{
    if (str == NULL) {
        return NULL;
    }

    char* last = NULL;
    while (*str != '\0') {
        if (*str == ch) {
            last = str;
        }

        str += 1;
    }

    return last;
}


// Our own "WAVECAPS" structure that contains generic information shared between WAVEOUTCAPS2 and WAVEINCAPS2 so
// we can do things generically and typesafely. Names are being kept the same for consistency.
typedef struct
{
    CHAR szPname[MAXPNAMELEN];
    DWORD dwFormats;
    WORD wChannels;
    GUID NameGuid;
} MAL_WAVECAPSA;

mal_result mal_get_best_info_from_formats_flags__winmm(DWORD dwFormats, WORD channels, WORD* pBitsPerSample, DWORD* pSampleRate)
{
    if (pBitsPerSample) {
        *pBitsPerSample = 0;
    }
    if (pSampleRate) {
        *pSampleRate = 0;
    }

    WORD bitsPerSample = 0;
    DWORD sampleRate = 0;

    if (channels == 1) {
        bitsPerSample = 16;
        if ((dwFormats & WAVE_FORMAT_48M16) != 0) {
            sampleRate = 48000;
        } else if ((dwFormats & WAVE_FORMAT_44M16) != 0) {
            sampleRate = 44100;
        } else if ((dwFormats & WAVE_FORMAT_2M16) != 0) {
            sampleRate = 22050;
        } else if ((dwFormats & WAVE_FORMAT_1M16) != 0) {
            sampleRate = 11025;
        } else if ((dwFormats & WAVE_FORMAT_96M16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((dwFormats & WAVE_FORMAT_48M08) != 0) {
                sampleRate = 48000;
            } else if ((dwFormats & WAVE_FORMAT_44M08) != 0) {
                sampleRate = 44100;
            } else if ((dwFormats & WAVE_FORMAT_2M08) != 0) {
                sampleRate = 22050;
            } else if ((dwFormats & WAVE_FORMAT_1M08) != 0) {
                sampleRate = 11025;
            } else if ((dwFormats & WAVE_FORMAT_96M08) != 0) {
                sampleRate = 96000;
            } else {
                return MAL_FORMAT_NOT_SUPPORTED;
            }
        }
    } else {
        bitsPerSample = 16;
        if ((dwFormats & WAVE_FORMAT_48S16) != 0) {
            sampleRate = 48000;
        } else if ((dwFormats & WAVE_FORMAT_44S16) != 0) {
            sampleRate = 44100;
        } else if ((dwFormats & WAVE_FORMAT_2S16) != 0) {
            sampleRate = 22050;
        } else if ((dwFormats & WAVE_FORMAT_1S16) != 0) {
            sampleRate = 11025;
        } else if ((dwFormats & WAVE_FORMAT_96S16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((dwFormats & WAVE_FORMAT_48S08) != 0) {
                sampleRate = 48000;
            } else if ((dwFormats & WAVE_FORMAT_44S08) != 0) {
                sampleRate = 44100;
            } else if ((dwFormats & WAVE_FORMAT_2S08) != 0) {
                sampleRate = 22050;
            } else if ((dwFormats & WAVE_FORMAT_1S08) != 0) {
                sampleRate = 11025;
            } else if ((dwFormats & WAVE_FORMAT_96S08) != 0) {
                sampleRate = 96000;
            } else {
                return MAL_FORMAT_NOT_SUPPORTED;
            }
        }
    }

    if (pBitsPerSample) {
        *pBitsPerSample = bitsPerSample;
    }
    if (pSampleRate) {
        *pSampleRate = sampleRate;
    }

    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info_from_WAVECAPS(mal_context* pContext, MAL_WAVECAPSA* pCaps, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    mal_assert(pCaps != NULL);
    mal_assert(pDeviceInfo != NULL);

    // Name / Description
    //
    // Unfortunately the name specified in WAVE(OUT/IN)CAPS2 is limited to 31 characters. This results in an unprofessional looking
    // situation where the names of the devices are truncated. To help work around this, we need to look at the name GUID and try
    // looking in the registry for the full name. If we can't find it there, we need to just fall back to the default name.

    // Set the default to begin with.
    mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), pCaps->szPname, (size_t)-1);

    // Now try the registry. There's a few things to consider here:
    // - The name GUID can be null, in which we case we just need to stick to the original 31 characters.
    // - If the name GUID is not present in the registry we'll also need to stick to the original 31 characters.
    // - I like consistency, so I want the returned device names to be consistent with those returned by WASAPI and DirectSound. The
    //   problem, however is that WASAPI and DirectSound use "<component> (<name>)" format (such as "Speakers (High Definition Audio)"),
    //   but WinMM does not specificy the component name. From my admittedly limited testing, I've notice the component name seems to
    //   usually fit within the 31 characters of the fixed sized buffer, so what I'm going to do is parse that string for the component
    //   name, and then concatenate the name from the registry.
    if (!mal_is_guid_equal(&pCaps->NameGuid, &MAL_GUID_NULL)) {
        wchar_t guidStrW[256];
        if (((MAL_PFN_StringFromGUID2)pContext->win32.StringFromGUID2)(&pCaps->NameGuid, guidStrW, mal_countof(guidStrW)) > 0) {
            char guidStr[256];
            WideCharToMultiByte(CP_UTF8, 0, guidStrW, -1, guidStr, sizeof(guidStr), 0, FALSE);

            char keyStr[1024];
            mal_strcpy_s(keyStr, sizeof(keyStr), "SYSTEM\\CurrentControlSet\\Control\\MediaCategories\\");
            mal_strcat_s(keyStr, sizeof(keyStr), guidStr);

            HKEY hKey;
            LONG result = ((MAL_PFN_RegOpenKeyExA)pContext->win32.RegOpenKeyExA)(HKEY_LOCAL_MACHINE, keyStr, 0, KEY_READ, &hKey);
            if (result == ERROR_SUCCESS) {
                BYTE nameFromReg[512];
                DWORD nameFromRegSize = sizeof(nameFromReg);
                result = ((MAL_PFN_RegQueryValueExA)pContext->win32.RegQueryValueExA)(hKey, "Name", 0, NULL, (LPBYTE)nameFromReg, (LPDWORD)&nameFromRegSize);
                ((MAL_PFN_RegCloseKey)pContext->win32.RegCloseKey)(hKey);

                if (result == ERROR_SUCCESS) {
                    // We have the value from the registry, so now we need to construct the name string.
                    char name[1024];
                    if (mal_strcpy_s(name, sizeof(name), pDeviceInfo->name) == 0) {
                        char* nameBeg = mal_find_last_character(name, '(');
                        if (nameBeg != NULL) {
                            size_t leadingLen = (nameBeg - name);
                            mal_strncpy_s(nameBeg + 1, sizeof(name) - leadingLen, (const char*)nameFromReg, (size_t)-1);

                            // The closing ")", if it can fit.
                            if (leadingLen + nameFromRegSize < sizeof(name)-1) {
                                mal_strcat_s(name, sizeof(name), ")");
                            }

                            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), name, (size_t)-1);
                        }
                    }
                }
            }
        }
    }

    
    WORD bitsPerSample;
    DWORD sampleRate;
    mal_result result = mal_get_best_info_from_formats_flags__winmm(pCaps->dwFormats, pCaps->wChannels, &bitsPerSample, &sampleRate);
    if (result != MAL_SUCCESS) {
        return result;
    }

    pDeviceInfo->minChannels = pCaps->wChannels;
    pDeviceInfo->maxChannels = pCaps->wChannels;
    pDeviceInfo->minSampleRate = sampleRate;
    pDeviceInfo->maxSampleRate = sampleRate;
    pDeviceInfo->formatCount = 1;
    if (bitsPerSample == 8) {
        pDeviceInfo->formats[0] = mal_format_u8;
    } else if (bitsPerSample == 16) {
        pDeviceInfo->formats[0] = mal_format_s16;
    } else if (bitsPerSample == 24) {
        pDeviceInfo->formats[0] = mal_format_s24;
    } else if (bitsPerSample == 32) {
        pDeviceInfo->formats[0] = mal_format_s32;
    } else {
        return MAL_FORMAT_NOT_SUPPORTED;
    }

    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info_from_WAVEOUTCAPS2(mal_context* pContext, MAL_WAVEOUTCAPS2A* pCaps, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    mal_assert(pCaps != NULL);
    mal_assert(pDeviceInfo != NULL);

    MAL_WAVECAPSA caps;
    mal_copy_memory(caps.szPname, pCaps->szPname, sizeof(caps.szPname));
    caps.dwFormats = pCaps->dwFormats;
    caps.wChannels = pCaps->wChannels;
    caps.NameGuid = pCaps->NameGuid;
    return mal_context_get_device_info_from_WAVECAPS(pContext, &caps, pDeviceInfo);
}

mal_result mal_context_get_device_info_from_WAVEINCAPS2(mal_context* pContext, MAL_WAVEINCAPS2A* pCaps, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    mal_assert(pCaps != NULL);
    mal_assert(pDeviceInfo != NULL);

    MAL_WAVECAPSA caps;
    mal_copy_memory(caps.szPname, pCaps->szPname, sizeof(caps.szPname));
    caps.dwFormats = pCaps->dwFormats;
    caps.wChannels = pCaps->wChannels;
    caps.NameGuid = pCaps->NameGuid;
    return mal_context_get_device_info_from_WAVECAPS(pContext, &caps, pDeviceInfo);
}


mal_bool32 mal_context_is_device_id_equal__winmm(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return pID0->winmm == pID1->winmm;
}

mal_result mal_context_enumerate_devices__winmm(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    // Playback.
    UINT playbackDeviceCount = ((MAL_PFN_waveOutGetNumDevs)pContext->winmm.waveOutGetNumDevs)();
    for (UINT iPlaybackDevice = 0; iPlaybackDevice < playbackDeviceCount; ++iPlaybackDevice) {
        MAL_WAVEOUTCAPS2A caps;
        mal_zero_object(&caps);
        MMRESULT result = ((MAL_PFN_waveOutGetDevCapsA)pContext->winmm.waveOutGetDevCapsA)(iPlaybackDevice, (WAVEOUTCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            deviceInfo.id.winmm = iPlaybackDevice;

            if (mal_context_get_device_info_from_WAVEOUTCAPS2(pContext, &caps, &deviceInfo) == MAL_SUCCESS) {
                mal_bool32 cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
                if (cbResult == MAL_FALSE) {
                    return MAL_SUCCESS; // Enumeration was stopped.
                }
            }
        }
    }

    // Capture.
    UINT captureDeviceCount = ((MAL_PFN_waveInGetNumDevs)pContext->winmm.waveInGetNumDevs)();
    for (UINT iCaptureDevice = 0; iCaptureDevice < captureDeviceCount; ++iCaptureDevice) {
        MAL_WAVEINCAPS2A caps;
        mal_zero_object(&caps);
        MMRESULT result = ((MAL_PFN_waveInGetDevCapsA)pContext->winmm.waveInGetDevCapsA)(iCaptureDevice, (WAVEINCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            deviceInfo.id.winmm = iCaptureDevice;

            if (mal_context_get_device_info_from_WAVEINCAPS2(pContext, &caps, &deviceInfo) == MAL_SUCCESS) {
                mal_bool32 cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
                if (cbResult == MAL_FALSE) {
                    return MAL_SUCCESS; // Enumeration was stopped.
                }
            }
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info__winmm(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    (void)shareMode;

    UINT winMMDeviceID = 0;
    if (pDeviceID != NULL) {
        winMMDeviceID = (UINT)pDeviceID->winmm;
    }

    pDeviceInfo->id.winmm = winMMDeviceID;

    if (deviceType == mal_device_type_playback) {
        MAL_WAVEOUTCAPS2A caps;
        mal_zero_object(&caps);
        MMRESULT result = ((MAL_PFN_waveOutGetDevCapsA)pContext->winmm.waveOutGetDevCapsA)(winMMDeviceID, (WAVEOUTCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            return mal_context_get_device_info_from_WAVEOUTCAPS2(pContext, &caps, pDeviceInfo);
        }
    } else {
        MAL_WAVEINCAPS2A caps;
        mal_zero_object(&caps);
        MMRESULT result = ((MAL_PFN_waveInGetDevCapsA)pContext->winmm.waveInGetDevCapsA)(winMMDeviceID, (WAVEINCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            return mal_context_get_device_info_from_WAVEINCAPS2(pContext, &caps, pDeviceInfo);
        }
    }

    return MAL_NO_DEVICE;
}


void mal_device_uninit__winmm(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->type == mal_device_type_playback) {
        ((MAL_PFN_waveOutClose)pDevice->pContext->winmm.waveOutClose)((HWAVEOUT)pDevice->winmm.hDevice);
    } else {
        ((MAL_PFN_waveInClose)pDevice->pContext->winmm.waveInClose)((HWAVEIN)pDevice->winmm.hDevice);
    }

    mal_free(pDevice->winmm._pHeapData);
    CloseHandle((HANDLE)pDevice->winmm.hEvent);

    mal_zero_object(&pDevice->winmm);   // Safety.
}

mal_result mal_device_init__winmm(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pContext;

    mal_uint32 heapSize;

    mal_assert(pDevice != NULL);
    mal_zero_object(&pDevice->winmm);

    UINT winMMDeviceID = 0;
    if (pDeviceID != NULL) {
        winMMDeviceID = (UINT)pDeviceID->winmm;
    }

    const char* errorMsg = "";
    mal_result errorCode = MAL_ERROR;
    mal_result result = MAL_SUCCESS;

    // WinMM doesn't seem to have a good way to query the format of the device. Therefore, we'll restrict the formats to the
    // standard formats documented here https://msdn.microsoft.com/en-us/library/windows/desktop/dd743855(v=vs.85).aspx. If
    // that link goes stale, just look up the documentation for WAVEOUTCAPS or WAVEINCAPS.
    WAVEFORMATEX wf;
    mal_zero_object(&wf);
    wf.cbSize          = sizeof(wf);
    wf.wFormatTag      = WAVE_FORMAT_PCM;
    wf.nChannels       = (WORD)pConfig->channels;
    wf.nSamplesPerSec  = (DWORD)pConfig->sampleRate;
    wf.wBitsPerSample  = (WORD)mal_get_bytes_per_sample(pConfig->format)*8;

    if (wf.nChannels > 2) {
        wf.nChannels = 2;
    }

    if (wf.wBitsPerSample != 8 && wf.wBitsPerSample != 16) {
        if (wf.wBitsPerSample <= 8) {
            wf.wBitsPerSample = 8;
        } else {
            wf.wBitsPerSample = 16;
        }
    }

    if (wf.nSamplesPerSec <= 11025) {
        wf.nSamplesPerSec = 11025;
    } else if (wf.nSamplesPerSec <= 22050) {
        wf.nSamplesPerSec = 22050;
    } else if (wf.nSamplesPerSec <= 44100) {
        wf.nSamplesPerSec = 44100;
    } else if (wf.nSamplesPerSec <= 48000) {
        wf.nSamplesPerSec = 48000;
    } else {
        wf.nSamplesPerSec = 96000;
    }


    // Change the format based on the closest match of the supported standard formats.
    DWORD dwFormats = 0;
    WORD wChannels = 0;
    if (type == mal_device_type_playback) {
        WAVEOUTCAPSA caps;
        if (((MAL_PFN_waveOutGetDevCapsA)pContext->winmm.waveOutGetDevCapsA)(winMMDeviceID, &caps, sizeof(caps)) == MMSYSERR_NOERROR) {
            dwFormats = caps.dwFormats;
            wChannels = caps.wChannels;
        } else {
            errorMsg = "[WinMM] Failed to retrieve internal device caps.", errorCode = MAL_FORMAT_NOT_SUPPORTED;
            goto on_error;
        }
    } else {
        WAVEINCAPSA caps;
        if (((MAL_PFN_waveInGetDevCapsA)pContext->winmm.waveInGetDevCapsA)(winMMDeviceID, &caps, sizeof(caps)) == MMSYSERR_NOERROR) {
            dwFormats = caps.dwFormats;
            wChannels = caps.wChannels;
        } else {
            errorMsg = "[WinMM] Failed to retrieve internal device caps.", errorCode = MAL_FORMAT_NOT_SUPPORTED;
            goto on_error;
        }
    }

    if (dwFormats == 0) {
        errorMsg = "[WinMM] Failed to retrieve the supported formats for the internal device.", errorCode = MAL_FORMAT_NOT_SUPPORTED;
        goto on_error;
    }

    wf.nChannels = wChannels;

    result = mal_get_best_info_from_formats_flags__winmm(dwFormats, wChannels, &wf.wBitsPerSample, &wf.nSamplesPerSec);
    if (result != MAL_SUCCESS) {
        errorMsg = "[WinMM] Could not find appropriate format for internal device.", errorCode = result;
        goto on_error;
    }

    wf.nBlockAlign     = (wf.nChannels * wf.wBitsPerSample) / 8;
    wf.nAvgBytesPerSec = wf.nBlockAlign * wf.nSamplesPerSec;


    // We use an event to know when a new fragment needs to be enqueued.
    pDevice->winmm.hEvent = (mal_handle)CreateEvent(NULL, TRUE, TRUE, NULL);
    if (pDevice->winmm.hEvent == NULL) {
        errorMsg = "[WinMM] Failed to create event for fragment enqueing.", errorCode = MAL_FAILED_TO_CREATE_EVENT;
        goto on_error;
    }


    if (type == mal_device_type_playback) {
        MMRESULT resultMM = ((MAL_PFN_waveOutOpen)pContext->winmm.waveOutOpen)((LPHWAVEOUT)&pDevice->winmm.hDevice, winMMDeviceID, &wf, (DWORD_PTR)pDevice->winmm.hEvent, (DWORD_PTR)pDevice, CALLBACK_EVENT | WAVE_ALLOWSYNC);
        if (resultMM != MMSYSERR_NOERROR) {
            errorMsg = "[WinMM] Failed to open playback device.", errorCode = MAL_FAILED_TO_OPEN_BACKEND_DEVICE;
            goto on_error;
        }
    } else {
        MMRESULT resultMM = ((MAL_PFN_waveInOpen)pDevice->pContext->winmm.waveInOpen)((LPHWAVEIN)&pDevice->winmm.hDevice, winMMDeviceID, &wf, (DWORD_PTR)pDevice->winmm.hEvent, (DWORD_PTR)pDevice, CALLBACK_EVENT | WAVE_ALLOWSYNC);
        if (resultMM != MMSYSERR_NOERROR) {
            errorMsg = "[WinMM] Failed to open capture device.", errorCode = MAL_FAILED_TO_OPEN_BACKEND_DEVICE;
            goto on_error;
        }
    }


    // The internal formats need to be set based on the wf object.
    if (wf.wFormatTag == WAVE_FORMAT_PCM) {
        switch (wf.wBitsPerSample) {
            case 8:  pDevice->internalFormat = mal_format_u8;  break;
            case 16: pDevice->internalFormat = mal_format_s16; break;
            case 24: pDevice->internalFormat = mal_format_s24; break;
            case 32: pDevice->internalFormat = mal_format_s32; break;
            default: mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] The device's internal format is not supported by mini_al.", MAL_FORMAT_NOT_SUPPORTED);
        }
    } else {
        errorMsg = "[WinMM] The device's internal format is not supported by mini_al.", errorCode = MAL_FORMAT_NOT_SUPPORTED;
        goto on_error;
    }

    pDevice->internalChannels = wf.nChannels;
    pDevice->internalSampleRate = wf.nSamplesPerSec;


    // Just use the default channel mapping. WinMM only supports mono or stereo anyway so it'll reliably be left/right order for stereo.
    mal_get_standard_channel_map(mal_standard_channel_map_microsoft, pDevice->internalChannels, pDevice->internalChannelMap);


    if (pDevice->bufferSizeInFrames == 0) {
        pDevice->bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, pDevice->internalSampleRate);
        if (pDevice->usingDefaultBufferSize) {
            float bufferSizeScaleFactor = 4;    // <-- Latency with WinMM seems pretty bad from my testing...
            pDevice->bufferSizeInFrames = mal_scale_buffer_size(pDevice->bufferSizeInFrames, bufferSizeScaleFactor);
        }
    }

    // The size of the intermediary buffer needs to be able to fit every fragment.
    pDevice->winmm.fragmentSizeInFrames = pDevice->bufferSizeInFrames / pDevice->periods;
    pDevice->winmm.fragmentSizeInBytes = pDevice->winmm.fragmentSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat);

    heapSize = (sizeof(WAVEHDR) * pDevice->periods) + (pDevice->winmm.fragmentSizeInBytes * pDevice->periods);
    pDevice->winmm._pHeapData = (mal_uint8*)mal_malloc(heapSize);
    if (pDevice->winmm._pHeapData == NULL) {
        errorMsg = "[WinMM] Failed to allocate memory for the intermediary buffer.", errorCode = MAL_OUT_OF_MEMORY;
        goto on_error;
    }

    mal_zero_memory(pDevice->winmm._pHeapData, pDevice->winmm.fragmentSizeInBytes * pDevice->periods);

    pDevice->winmm.pWAVEHDR = pDevice->winmm._pHeapData;
    pDevice->winmm.pIntermediaryBuffer = pDevice->winmm._pHeapData + (sizeof(WAVEHDR) * pDevice->periods);


    return MAL_SUCCESS;

on_error:
    if (pDevice->type == mal_device_type_playback) {
        ((MAL_PFN_waveOutClose)pContext->winmm.waveOutClose)((HWAVEOUT)pDevice->winmm.hDevice);
    } else {
        ((MAL_PFN_waveInClose)pContext->winmm.waveInClose)((HWAVEIN)pDevice->winmm.hDevice);
    }

    mal_free(pDevice->winmm._pHeapData);
    return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, errorMsg, errorCode);
}


mal_result mal_device__start_backend__winmm(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->winmm.hDevice == NULL) {
        return MAL_INVALID_ARGS;
    }

    if (pDevice->type == mal_device_type_playback) {
        // Playback. The device is started when we call waveOutWrite() with a block of data. From MSDN:
        //
        //     Unless the device is paused by calling the waveOutPause function, playback begins when the first data block is sent to the device.
        //
        // When starting the device we commit every fragment. We signal the event before calling waveOutWrite().
        mal_uint32 i;
        for (i = 0; i < pDevice->periods; ++i) {
            mal_zero_object(&((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i]);
            ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].lpData = (LPSTR)(pDevice->winmm.pIntermediaryBuffer + (pDevice->winmm.fragmentSizeInBytes * i));
            ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwBufferLength = pDevice->winmm.fragmentSizeInBytes;
            ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwFlags = 0L;
            ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwLoops = 0L;
            mal_device__read_frames_from_client(pDevice, pDevice->winmm.fragmentSizeInFrames, ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].lpData);

            if (((MAL_PFN_waveOutPrepareHeader)pDevice->pContext->winmm.waveOutPrepareHeader)((HWAVEOUT)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR)) != MMSYSERR_NOERROR) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to start backend device. Failed to prepare header.", MAL_FAILED_TO_START_BACKEND_DEVICE);
            }
        }

        ResetEvent(pDevice->winmm.hEvent);

        for (i = 0; i < pDevice->periods; ++i) {
            if (((MAL_PFN_waveOutWrite)pDevice->pContext->winmm.waveOutWrite)((HWAVEOUT)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR)) != MMSYSERR_NOERROR) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to start backend device. Failed to send data to the backend device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
            }
        }
    } else {
        // Capture.
        for (mal_uint32 i = 0; i < pDevice->periods; ++i) {
            mal_zero_object(&((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i]);
            ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].lpData = (LPSTR)(pDevice->winmm.pIntermediaryBuffer + (pDevice->winmm.fragmentSizeInBytes * i));
            ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwBufferLength = pDevice->winmm.fragmentSizeInBytes;
            ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwFlags = 0L;
            ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwLoops = 0L;

            MMRESULT resultMM = ((MAL_PFN_waveInPrepareHeader)pDevice->pContext->winmm.waveInPrepareHeader)((HWAVEIN)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR));
            if (resultMM != MMSYSERR_NOERROR) {
                mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to prepare header for capture device in preparation for adding a new capture buffer for the device.", mal_result_from_MMRESULT(resultMM));
                break;
            }

            resultMM = ((MAL_PFN_waveInAddBuffer)pDevice->pContext->winmm.waveInAddBuffer)((HWAVEIN)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR));
            if (resultMM != MMSYSERR_NOERROR) {
                mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to add new capture buffer to the internal capture device.", mal_result_from_MMRESULT(resultMM));
                break;
            }
        }

        ResetEvent(pDevice->winmm.hEvent);

        if (((MAL_PFN_waveInStart)pDevice->pContext->winmm.waveInStart)((HWAVEIN)pDevice->winmm.hDevice) != MMSYSERR_NOERROR) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to start backend device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
        }
    }

    pDevice->winmm.iNextHeader = 0;
    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__winmm(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->winmm.hDevice == NULL) {
        return MAL_INVALID_ARGS;
    }

    if (pDevice->type == mal_device_type_playback) {
        MMRESULT resultMM = ((MAL_PFN_waveOutReset)pDevice->pContext->winmm.waveOutReset)((HWAVEOUT)pDevice->winmm.hDevice);
        if (resultMM != MMSYSERR_NOERROR) {
            mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] WARNING: Failed to reset playback device.", mal_result_from_MMRESULT(resultMM));
        }

        // Unprepare all WAVEHDR structures.
        for (mal_uint32 i = 0; i < pDevice->periods; ++i) {
            resultMM = ((MAL_PFN_waveOutUnprepareHeader)pDevice->pContext->winmm.waveOutUnprepareHeader)((HWAVEOUT)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR));
            if (resultMM != MMSYSERR_NOERROR) {
                mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] WARNING: Failed to unprepare header for playback device.", mal_result_from_MMRESULT(resultMM));
            }
        }
    } else {
        MMRESULT resultMM = ((MAL_PFN_waveInReset)pDevice->pContext->winmm.waveInReset)((HWAVEIN)pDevice->winmm.hDevice);
        if (resultMM != MMSYSERR_NOERROR) {
            mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] WARNING: Failed to reset capture device.", mal_result_from_MMRESULT(resultMM));
        }

        // Unprepare all WAVEHDR structures.
        for (mal_uint32 i = 0; i < pDevice->periods; ++i) {
            resultMM = ((MAL_PFN_waveInUnprepareHeader)pDevice->pContext->winmm.waveInUnprepareHeader)((HWAVEIN)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR));
            if (resultMM != MMSYSERR_NOERROR) {
                mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] WARNING: Failed to unprepare header for playback device.", mal_result_from_MMRESULT(resultMM));
            }
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_device__break_main_loop__winmm(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->winmm.breakFromMainLoop = MAL_TRUE;
    SetEvent((HANDLE)pDevice->winmm.hEvent);

    return MAL_SUCCESS;
}

mal_result mal_device__main_loop__winmm(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_uint32 counter;

    pDevice->winmm.breakFromMainLoop = MAL_FALSE;
    while (!pDevice->winmm.breakFromMainLoop) {
        // Wait for a block of data to finish processing...
        if (WaitForSingleObject((HANDLE)pDevice->winmm.hEvent, INFINITE) != WAIT_OBJECT_0) {
            break;
        }

        // Break from the main loop if the device isn't started anymore. Likely what's happened is the application
        // has requested that the device be stopped.
        if (!mal_device_is_started(pDevice)) {
            break;
        }

        // Any headers that are marked as done need to be handled. We start by processing the completed blocks. Then we reset the event
        // and then write or add replacement buffers to the device.
        mal_uint32 iFirstHeader = pDevice->winmm.iNextHeader;
        for (counter = 0; counter < pDevice->periods; ++counter) {
            mal_uint32 i = pDevice->winmm.iNextHeader;
            if ((((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwFlags & WHDR_DONE) == 0) {
                break;
            }

            if (pDevice->type == mal_device_type_playback) {
                // Playback.
                MMRESULT resultMM = ((MAL_PFN_waveOutUnprepareHeader)pDevice->pContext->winmm.waveOutUnprepareHeader)((HWAVEOUT)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR));
                if (resultMM != MMSYSERR_NOERROR) {
                    mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to unprepare header for playback device in preparation for sending a new block of data to the device for playback.", mal_result_from_MMRESULT(resultMM));
                    return MAL_DEVICE_UNAVAILABLE;
                }

                mal_zero_object(&((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i]);
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].lpData = (LPSTR)(pDevice->winmm.pIntermediaryBuffer + (pDevice->winmm.fragmentSizeInBytes * i));
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwBufferLength = pDevice->winmm.fragmentSizeInBytes;
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwFlags = 0L;
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwLoops = 0L;
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwUser = 1;     // <-- Used in the next section to identify the buffers that needs to be re-written to the device.
                mal_device__read_frames_from_client(pDevice, pDevice->winmm.fragmentSizeInFrames, ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].lpData);

                resultMM = ((MAL_PFN_waveOutPrepareHeader)pDevice->pContext->winmm.waveOutPrepareHeader)((HWAVEOUT)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR));
                if (resultMM != MMSYSERR_NOERROR) {
                    mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to prepare header for playback device in preparation for sending a new block of data to the device for playback.", mal_result_from_MMRESULT(resultMM));
                    return MAL_DEVICE_UNAVAILABLE;
                }
            } else {
                // Capture.
                mal_uint32 framesCaptured = (mal_uint32)(((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwBytesRecorded) / pDevice->internalChannels / mal_get_bytes_per_sample(pDevice->internalFormat);
                if (framesCaptured > 0) {
                    mal_device__send_frames_to_client(pDevice, framesCaptured, ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].lpData);
                }

                MMRESULT resultMM = ((MAL_PFN_waveInUnprepareHeader)pDevice->pContext->winmm.waveInUnprepareHeader)((HWAVEIN)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR));
                if (resultMM != MMSYSERR_NOERROR) {
                    mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to unprepare header for capture device in preparation for adding a new capture buffer for the device.", mal_result_from_MMRESULT(resultMM));
                    return MAL_DEVICE_UNAVAILABLE;
                }

                mal_zero_object(&((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i]);
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].lpData = (LPSTR)(pDevice->winmm.pIntermediaryBuffer + (pDevice->winmm.fragmentSizeInBytes * i));
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwBufferLength = pDevice->winmm.fragmentSizeInBytes;
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwFlags = 0L;
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwLoops = 0L;
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwUser = 1;     // <-- Used in the next section to identify the buffers that needs to be re-added to the device.

                resultMM = ((MAL_PFN_waveInPrepareHeader)pDevice->pContext->winmm.waveInPrepareHeader)((HWAVEIN)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR));
                if (resultMM != MMSYSERR_NOERROR) {
                    mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to prepare header for capture device in preparation for adding a new capture buffer for the device.", mal_result_from_MMRESULT(resultMM));
                    return MAL_DEVICE_UNAVAILABLE;
                }
            }

            pDevice->winmm.iNextHeader = (pDevice->winmm.iNextHeader + 1) % pDevice->periods;
        }

        ResetEvent((HANDLE)pDevice->winmm.hEvent);

        for (counter = 0; counter < pDevice->periods; ++counter) {
            mal_uint32 i = (iFirstHeader + counter) % pDevice->periods;

            if (((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwUser == 1) {
                ((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i].dwUser = 0;

                if (pDevice->type == mal_device_type_playback) {
                    // Playback.
                    MMRESULT resultMM = ((MAL_PFN_waveOutWrite)pDevice->pContext->winmm.waveOutWrite)((HWAVEOUT)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR));
                    if (resultMM != MMSYSERR_NOERROR) {
                        mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to write data to the internal playback device.", mal_result_from_MMRESULT(resultMM));
                        return MAL_DEVICE_UNAVAILABLE;
                    }
                } else {
                    // Capture.
                    MMRESULT resultMM = ((MAL_PFN_waveInAddBuffer)pDevice->pContext->winmm.waveInAddBuffer)((HWAVEIN)pDevice->winmm.hDevice, &((LPWAVEHDR)pDevice->winmm.pWAVEHDR)[i], sizeof(WAVEHDR));
                    if (resultMM != MMSYSERR_NOERROR) {
                        mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[WinMM] Failed to add new capture buffer to the internal capture device.", mal_result_from_MMRESULT(resultMM));
                        return MAL_DEVICE_UNAVAILABLE;
                    }
                }
            }
        }
    }

    return MAL_SUCCESS;
}


mal_result mal_context_uninit__winmm(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_winmm);

    mal_dlclose(pContext->winmm.hWinMM);
    return MAL_SUCCESS;
}

mal_result mal_context_init__winmm(mal_context* pContext)
{
    mal_assert(pContext != NULL);

    pContext->winmm.hWinMM = mal_dlopen("winmm.dll");
    if (pContext->winmm.hWinMM == NULL) {
        return MAL_NO_BACKEND;
    }

    pContext->winmm.waveOutGetNumDevs      = mal_dlsym(pContext->winmm.hWinMM, "waveOutGetNumDevs");
    pContext->winmm.waveOutGetDevCapsA     = mal_dlsym(pContext->winmm.hWinMM, "waveOutGetDevCapsA");
    pContext->winmm.waveOutOpen            = mal_dlsym(pContext->winmm.hWinMM, "waveOutOpen");
    pContext->winmm.waveOutClose           = mal_dlsym(pContext->winmm.hWinMM, "waveOutClose");
    pContext->winmm.waveOutPrepareHeader   = mal_dlsym(pContext->winmm.hWinMM, "waveOutPrepareHeader");
    pContext->winmm.waveOutUnprepareHeader = mal_dlsym(pContext->winmm.hWinMM, "waveOutUnprepareHeader");
    pContext->winmm.waveOutWrite           = mal_dlsym(pContext->winmm.hWinMM, "waveOutWrite");
    pContext->winmm.waveOutReset           = mal_dlsym(pContext->winmm.hWinMM, "waveOutReset");
    pContext->winmm.waveInGetNumDevs       = mal_dlsym(pContext->winmm.hWinMM, "waveInGetNumDevs");
    pContext->winmm.waveInGetDevCapsA      = mal_dlsym(pContext->winmm.hWinMM, "waveInGetDevCapsA");
    pContext->winmm.waveInOpen             = mal_dlsym(pContext->winmm.hWinMM, "waveInOpen");
    pContext->winmm.waveInClose            = mal_dlsym(pContext->winmm.hWinMM, "waveInClose");
    pContext->winmm.waveInPrepareHeader    = mal_dlsym(pContext->winmm.hWinMM, "waveInPrepareHeader");
    pContext->winmm.waveInUnprepareHeader  = mal_dlsym(pContext->winmm.hWinMM, "waveInUnprepareHeader");
    pContext->winmm.waveInAddBuffer        = mal_dlsym(pContext->winmm.hWinMM, "waveInAddBuffer");
    pContext->winmm.waveInStart            = mal_dlsym(pContext->winmm.hWinMM, "waveInStart");
    pContext->winmm.waveInReset            = mal_dlsym(pContext->winmm.hWinMM, "waveInReset");

    pContext->onUninit              = mal_context_uninit__winmm;
    pContext->onDeviceIDEqual       = mal_context_is_device_id_equal__winmm;
    pContext->onEnumDevices         = mal_context_enumerate_devices__winmm;
    pContext->onGetDeviceInfo       = mal_context_get_device_info__winmm;
    pContext->onDeviceInit          = mal_device_init__winmm;
    pContext->onDeviceUninit        = mal_device_uninit__winmm;
    pContext->onDeviceStart         = mal_device__start_backend__winmm;
    pContext->onDeviceStop          = mal_device__stop_backend__winmm;
    pContext->onDeviceBreakMainLoop = mal_device__break_main_loop__winmm;
    pContext->onDeviceMainLoop      = mal_device__main_loop__winmm;

    return MAL_SUCCESS;
}
#endif




///////////////////////////////////////////////////////////////////////////////
//
// ALSA Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_ALSA

#ifdef MAL_NO_RUNTIME_LINKING
#include <alsa/asoundlib.h>
typedef snd_pcm_uframes_t                       mal_snd_pcm_uframes_t;
typedef snd_pcm_sframes_t                       mal_snd_pcm_sframes_t;
typedef snd_pcm_stream_t                        mal_snd_pcm_stream_t;
typedef snd_pcm_format_t                        mal_snd_pcm_format_t;
typedef snd_pcm_access_t                        mal_snd_pcm_access_t;
typedef snd_pcm_t                               mal_snd_pcm_t;
typedef snd_pcm_hw_params_t                     mal_snd_pcm_hw_params_t;
typedef snd_pcm_sw_params_t                     mal_snd_pcm_sw_params_t;
typedef snd_pcm_format_mask_t                   mal_snd_pcm_format_mask_t;
typedef snd_pcm_info_t                          mal_snd_pcm_info_t;
typedef snd_pcm_channel_area_t                  mal_snd_pcm_channel_area_t;
typedef snd_pcm_chmap_t                         mal_snd_pcm_chmap_t;

// snd_pcm_stream_t
#define MAL_SND_PCM_STREAM_PLAYBACK             SND_PCM_STREAM_PLAYBACK
#define MAL_SND_PCM_STREAM_CAPTURE              SND_PCM_STREAM_CAPTURE

// snd_pcm_format_t
#define MAL_SND_PCM_FORMAT_UNKNOWN              SND_PCM_FORMAT_UNKNOWN
#define MAL_SND_PCM_FORMAT_U8                   SND_PCM_FORMAT_U8
#define MAL_SND_PCM_FORMAT_S16_LE               SND_PCM_FORMAT_S16_LE
#define MAL_SND_PCM_FORMAT_S16_BE               SND_PCM_FORMAT_S16_BE
#define MAL_SND_PCM_FORMAT_S24_LE               SND_PCM_FORMAT_S24_LE
#define MAL_SND_PCM_FORMAT_S24_BE               SND_PCM_FORMAT_S24_BE
#define MAL_SND_PCM_FORMAT_S32_LE               SND_PCM_FORMAT_S32_LE
#define MAL_SND_PCM_FORMAT_S32_BE               SND_PCM_FORMAT_S32_BE
#define MAL_SND_PCM_FORMAT_FLOAT_LE             SND_PCM_FORMAT_FLOAT_LE
#define MAL_SND_PCM_FORMAT_FLOAT_BE             SND_PCM_FORMAT_FLOAT_BE
#define MAL_SND_PCM_FORMAT_FLOAT64_LE           SND_PCM_FORMAT_FLOAT64_LE
#define MAL_SND_PCM_FORMAT_FLOAT64_BE           SND_PCM_FORMAT_FLOAT64_BE
#define MAL_SND_PCM_FORMAT_MU_LAW               SND_PCM_FORMAT_MU_LAW
#define MAL_SND_PCM_FORMAT_A_LAW                SND_PCM_FORMAT_A_LAW
#define MAL_SND_PCM_FORMAT_S24_3LE              SND_PCM_FORMAT_S24_3LE
#define MAL_SND_PCM_FORMAT_S24_3BE              SND_PCM_FORMAT_S24_3BE

// mal_snd_pcm_access_t
#define MAL_SND_PCM_ACCESS_MMAP_INTERLEAVED     SND_PCM_ACCESS_MMAP_INTERLEAVED
#define MAL_SND_PCM_ACCESS_MMAP_NONINTERLEAVED  SND_PCM_ACCESS_MMAP_NONINTERLEAVED
#define MAL_SND_PCM_ACCESS_MMAP_COMPLEX         SND_PCM_ACCESS_MMAP_COMPLEX
#define MAL_SND_PCM_ACCESS_RW_INTERLEAVED       SND_PCM_ACCESS_RW_INTERLEAVED
#define MAL_SND_PCM_ACCESS_RW_NONINTERLEAVED    SND_PCM_ACCESS_RW_NONINTERLEAVED

// Channel positions.
#define MAL_SND_CHMAP_UNKNOWN                   SND_CHMAP_UNKNOWN
#define MAL_SND_CHMAP_NA                        SND_CHMAP_NA
#define MAL_SND_CHMAP_MONO                      SND_CHMAP_MONO
#define MAL_SND_CHMAP_FL                        SND_CHMAP_FL
#define MAL_SND_CHMAP_FR                        SND_CHMAP_FR
#define MAL_SND_CHMAP_RL                        SND_CHMAP_RL
#define MAL_SND_CHMAP_RR                        SND_CHMAP_RR
#define MAL_SND_CHMAP_FC                        SND_CHMAP_FC
#define MAL_SND_CHMAP_LFE                       SND_CHMAP_LFE
#define MAL_SND_CHMAP_SL                        SND_CHMAP_SL
#define MAL_SND_CHMAP_SR                        SND_CHMAP_SR
#define MAL_SND_CHMAP_RC                        SND_CHMAP_RC
#define MAL_SND_CHMAP_FLC                       SND_CHMAP_FLC
#define MAL_SND_CHMAP_FRC                       SND_CHMAP_FRC
#define MAL_SND_CHMAP_RLC                       SND_CHMAP_RLC
#define MAL_SND_CHMAP_RRC                       SND_CHMAP_RRC
#define MAL_SND_CHMAP_FLW                       SND_CHMAP_FLW
#define MAL_SND_CHMAP_FRW                       SND_CHMAP_FRW
#define MAL_SND_CHMAP_FLH                       SND_CHMAP_FLH
#define MAL_SND_CHMAP_FCH                       SND_CHMAP_FCH
#define MAL_SND_CHMAP_FRH                       SND_CHMAP_FRH
#define MAL_SND_CHMAP_TC                        SND_CHMAP_TC
#define MAL_SND_CHMAP_TFL                       SND_CHMAP_TFL
#define MAL_SND_CHMAP_TFR                       SND_CHMAP_TFR
#define MAL_SND_CHMAP_TFC                       SND_CHMAP_TFC
#define MAL_SND_CHMAP_TRL                       SND_CHMAP_TRL
#define MAL_SND_CHMAP_TRR                       SND_CHMAP_TRR
#define MAL_SND_CHMAP_TRC                       SND_CHMAP_TRC
#define MAL_SND_CHMAP_TFLC                      SND_CHMAP_TFLC
#define MAL_SND_CHMAP_TFRC                      SND_CHMAP_TFRC
#define MAL_SND_CHMAP_TSL                       SND_CHMAP_TSL
#define MAL_SND_CHMAP_TSR                       SND_CHMAP_TSR
#define MAL_SND_CHMAP_LLFE                      SND_CHMAP_LLFE
#define MAL_SND_CHMAP_RLFE                      SND_CHMAP_RLFE
#define MAL_SND_CHMAP_BC                        SND_CHMAP_BC
#define MAL_SND_CHMAP_BLC                       SND_CHMAP_BLC
#define MAL_SND_CHMAP_BRC                       SND_CHMAP_BRC

// Open mode flags.
#define MAL_SND_PCM_NO_AUTO_RESAMPLE            SND_PCM_NO_AUTO_RESAMPLE
#define MAL_SND_PCM_NO_AUTO_CHANNELS            SND_PCM_NO_AUTO_CHANNELS
#define MAL_SND_PCM_NO_AUTO_FORMAT              SND_PCM_NO_AUTO_FORMAT
#else
#include <errno.h>  // For EPIPE, etc.
typedef unsigned long                           mal_snd_pcm_uframes_t;
typedef long                                    mal_snd_pcm_sframes_t;
typedef int                                     mal_snd_pcm_stream_t;
typedef int                                     mal_snd_pcm_format_t;
typedef int                                     mal_snd_pcm_access_t;
typedef struct mal_snd_pcm_t                    mal_snd_pcm_t;
typedef struct mal_snd_pcm_hw_params_t          mal_snd_pcm_hw_params_t;
typedef struct mal_snd_pcm_sw_params_t          mal_snd_pcm_sw_params_t;
typedef struct mal_snd_pcm_format_mask_t        mal_snd_pcm_format_mask_t;
typedef struct mal_snd_pcm_info_t               mal_snd_pcm_info_t;
typedef struct
{
    void* addr;
    unsigned int first;
    unsigned int step;
} mal_snd_pcm_channel_area_t;
typedef struct
{
    unsigned int channels;
    unsigned int pos[0];
} mal_snd_pcm_chmap_t;

// snd_pcm_stream_t
#define MAL_SND_PCM_STREAM_PLAYBACK             0
#define MAL_SND_PCM_STREAM_CAPTURE              1

// snd_pcm_format_t
#define MAL_SND_PCM_FORMAT_UNKNOWN              -1
#define MAL_SND_PCM_FORMAT_U8                   1
#define MAL_SND_PCM_FORMAT_S16_LE               2
#define MAL_SND_PCM_FORMAT_S16_BE               3
#define MAL_SND_PCM_FORMAT_S24_LE               6
#define MAL_SND_PCM_FORMAT_S24_BE               7
#define MAL_SND_PCM_FORMAT_S32_LE               10
#define MAL_SND_PCM_FORMAT_S32_BE               11
#define MAL_SND_PCM_FORMAT_FLOAT_LE             14
#define MAL_SND_PCM_FORMAT_FLOAT_BE             15
#define MAL_SND_PCM_FORMAT_FLOAT64_LE           16
#define MAL_SND_PCM_FORMAT_FLOAT64_BE           17
#define MAL_SND_PCM_FORMAT_MU_LAW               20
#define MAL_SND_PCM_FORMAT_A_LAW                21
#define MAL_SND_PCM_FORMAT_S24_3LE              32
#define MAL_SND_PCM_FORMAT_S24_3BE              33

// snd_pcm_access_t
#define MAL_SND_PCM_ACCESS_MMAP_INTERLEAVED     0
#define MAL_SND_PCM_ACCESS_MMAP_NONINTERLEAVED  1
#define MAL_SND_PCM_ACCESS_MMAP_COMPLEX         2
#define MAL_SND_PCM_ACCESS_RW_INTERLEAVED       3
#define MAL_SND_PCM_ACCESS_RW_NONINTERLEAVED    4

// Channel positions.
#define MAL_SND_CHMAP_UNKNOWN                   0
#define MAL_SND_CHMAP_NA                        1
#define MAL_SND_CHMAP_MONO                      2
#define MAL_SND_CHMAP_FL                        3
#define MAL_SND_CHMAP_FR                        4
#define MAL_SND_CHMAP_RL                        5
#define MAL_SND_CHMAP_RR                        6
#define MAL_SND_CHMAP_FC                        7
#define MAL_SND_CHMAP_LFE                       8
#define MAL_SND_CHMAP_SL                        9
#define MAL_SND_CHMAP_SR                        10
#define MAL_SND_CHMAP_RC                        11
#define MAL_SND_CHMAP_FLC                       12
#define MAL_SND_CHMAP_FRC                       13
#define MAL_SND_CHMAP_RLC                       14
#define MAL_SND_CHMAP_RRC                       15
#define MAL_SND_CHMAP_FLW                       16
#define MAL_SND_CHMAP_FRW                       17
#define MAL_SND_CHMAP_FLH                       18
#define MAL_SND_CHMAP_FCH                       19
#define MAL_SND_CHMAP_FRH                       20
#define MAL_SND_CHMAP_TC                        21
#define MAL_SND_CHMAP_TFL                       22
#define MAL_SND_CHMAP_TFR                       23
#define MAL_SND_CHMAP_TFC                       24
#define MAL_SND_CHMAP_TRL                       25
#define MAL_SND_CHMAP_TRR                       26
#define MAL_SND_CHMAP_TRC                       27
#define MAL_SND_CHMAP_TFLC                      28
#define MAL_SND_CHMAP_TFRC                      29
#define MAL_SND_CHMAP_TSL                       30
#define MAL_SND_CHMAP_TSR                       31
#define MAL_SND_CHMAP_LLFE                      32
#define MAL_SND_CHMAP_RLFE                      33
#define MAL_SND_CHMAP_BC                        34
#define MAL_SND_CHMAP_BLC                       35
#define MAL_SND_CHMAP_BRC                       36

// Open mode flags.
#define MAL_SND_PCM_NO_AUTO_RESAMPLE            0x00010000
#define MAL_SND_PCM_NO_AUTO_CHANNELS            0x00020000
#define MAL_SND_PCM_NO_AUTO_FORMAT              0x00040000
#endif

typedef int                   (* mal_snd_pcm_open_proc)                          (mal_snd_pcm_t **pcm, const char *name, mal_snd_pcm_stream_t stream, int mode);
typedef int                   (* mal_snd_pcm_close_proc)                         (mal_snd_pcm_t *pcm);
typedef size_t                (* mal_snd_pcm_hw_params_sizeof_proc)              (void);
typedef int                   (* mal_snd_pcm_hw_params_any_proc)                 (mal_snd_pcm_t *pcm, mal_snd_pcm_hw_params_t *params);
typedef int                   (* mal_snd_pcm_hw_params_set_format_proc)          (mal_snd_pcm_t *pcm, mal_snd_pcm_hw_params_t *params, mal_snd_pcm_format_t val);
typedef int                   (* mal_snd_pcm_hw_params_set_format_first_proc)    (mal_snd_pcm_t *pcm, mal_snd_pcm_hw_params_t *params, mal_snd_pcm_format_t *format);
typedef void                  (* mal_snd_pcm_hw_params_get_format_mask_proc)     (mal_snd_pcm_hw_params_t *params, mal_snd_pcm_format_mask_t *mask);
typedef int                   (* mal_snd_pcm_hw_params_set_channels_near_proc)   (mal_snd_pcm_t *pcm, mal_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                   (* mal_snd_pcm_hw_params_set_rate_resample_proc)   (mal_snd_pcm_t *pcm, mal_snd_pcm_hw_params_t *params, unsigned int val);
typedef int                   (* mal_snd_pcm_hw_params_set_rate_near_proc)       (mal_snd_pcm_t *pcm, mal_snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
typedef int                   (* mal_snd_pcm_hw_params_set_buffer_size_near_proc)(mal_snd_pcm_t *pcm, mal_snd_pcm_hw_params_t *params, mal_snd_pcm_uframes_t *val);
typedef int                   (* mal_snd_pcm_hw_params_set_periods_near_proc)    (mal_snd_pcm_t *pcm, mal_snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
typedef int                   (* mal_snd_pcm_hw_params_set_access_proc)          (mal_snd_pcm_t *pcm, mal_snd_pcm_hw_params_t *params, mal_snd_pcm_access_t _access);
typedef int                   (* mal_snd_pcm_hw_params_get_format_proc)          (const mal_snd_pcm_hw_params_t *params, mal_snd_pcm_format_t *format);
typedef int                   (* mal_snd_pcm_hw_params_get_channels_proc)        (const mal_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                   (* mal_snd_pcm_hw_params_get_channels_min_proc)    (const mal_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                   (* mal_snd_pcm_hw_params_get_channels_max_proc)    (const mal_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                   (* mal_snd_pcm_hw_params_get_rate_proc)            (const mal_snd_pcm_hw_params_t *params, unsigned int *rate, int *dir);
typedef int                   (* mal_snd_pcm_hw_params_get_rate_min_proc)        (const mal_snd_pcm_hw_params_t *params, unsigned int *rate, int *dir);
typedef int                   (* mal_snd_pcm_hw_params_get_rate_max_proc)        (const mal_snd_pcm_hw_params_t *params, unsigned int *rate, int *dir);
typedef int                   (* mal_snd_pcm_hw_params_get_buffer_size_proc)     (const mal_snd_pcm_hw_params_t *params, mal_snd_pcm_uframes_t *val);
typedef int                   (* mal_snd_pcm_hw_params_get_periods_proc)         (const mal_snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
typedef int                   (* mal_snd_pcm_hw_params_get_access_proc)          (const mal_snd_pcm_hw_params_t *params, mal_snd_pcm_access_t *_access);
typedef int                   (* mal_snd_pcm_hw_params_proc)                     (mal_snd_pcm_t *pcm, mal_snd_pcm_hw_params_t *params);
typedef size_t                (* mal_snd_pcm_sw_params_sizeof_proc)              (void);
typedef int                   (* mal_snd_pcm_sw_params_current_proc)             (mal_snd_pcm_t *pcm, mal_snd_pcm_sw_params_t *params);
typedef int                   (* mal_snd_pcm_sw_params_set_avail_min_proc)       (mal_snd_pcm_t *pcm, mal_snd_pcm_sw_params_t *params, mal_snd_pcm_uframes_t val);
typedef int                   (* mal_snd_pcm_sw_params_set_start_threshold_proc) (mal_snd_pcm_t *pcm, mal_snd_pcm_sw_params_t *params, mal_snd_pcm_uframes_t val);
typedef int                   (* mal_snd_pcm_sw_params_proc)                     (mal_snd_pcm_t *pcm, mal_snd_pcm_sw_params_t *params);
typedef size_t                (* mal_snd_pcm_format_mask_sizeof_proc)            (void);
typedef int                   (* mal_snd_pcm_format_mask_test_proc)              (const mal_snd_pcm_format_mask_t *mask, mal_snd_pcm_format_t val);
typedef mal_snd_pcm_chmap_t * (* mal_snd_pcm_get_chmap_proc)                     (mal_snd_pcm_t *pcm);
typedef int                   (* mal_snd_pcm_prepare_proc)                       (mal_snd_pcm_t *pcm);
typedef int                   (* mal_snd_pcm_start_proc)                         (mal_snd_pcm_t *pcm);
typedef int                   (* mal_snd_pcm_drop_proc)                          (mal_snd_pcm_t *pcm);
typedef int                   (* mal_snd_device_name_hint_proc)                  (int card, const char *iface, void ***hints);
typedef char *                (* mal_snd_device_name_get_hint_proc)              (const void *hint, const char *id);
typedef int                   (* mal_snd_card_get_index_proc)                    (const char *name);
typedef int                   (* mal_snd_device_name_free_hint_proc)             (void **hints);
typedef int                   (* mal_snd_pcm_mmap_begin_proc)                    (mal_snd_pcm_t *pcm, const mal_snd_pcm_channel_area_t **areas, mal_snd_pcm_uframes_t *offset, mal_snd_pcm_uframes_t *frames);
typedef mal_snd_pcm_sframes_t (* mal_snd_pcm_mmap_commit_proc)                   (mal_snd_pcm_t *pcm, mal_snd_pcm_uframes_t offset, mal_snd_pcm_uframes_t frames);
typedef int                   (* mal_snd_pcm_recover_proc)                       (mal_snd_pcm_t *pcm, int err, int silent);
typedef mal_snd_pcm_sframes_t (* mal_snd_pcm_readi_proc)                         (mal_snd_pcm_t *pcm, void *buffer, mal_snd_pcm_uframes_t size);
typedef mal_snd_pcm_sframes_t (* mal_snd_pcm_writei_proc)                        (mal_snd_pcm_t *pcm, const void *buffer, mal_snd_pcm_uframes_t size);
typedef mal_snd_pcm_sframes_t (* mal_snd_pcm_avail_proc)                         (mal_snd_pcm_t *pcm);
typedef mal_snd_pcm_sframes_t (* mal_snd_pcm_avail_update_proc)                  (mal_snd_pcm_t *pcm);
typedef int                   (* mal_snd_pcm_wait_proc)                          (mal_snd_pcm_t *pcm, int timeout);
typedef int                   (* mal_snd_pcm_info_proc)                          (mal_snd_pcm_t *pcm, mal_snd_pcm_info_t* info);
typedef size_t                (* mal_snd_pcm_info_sizeof_proc)                   ();
typedef const char*           (* mal_snd_pcm_info_get_name_proc)                 (const mal_snd_pcm_info_t* info);
typedef int                   (* mal_snd_config_update_free_global_proc)         ();

// This array specifies each of the common devices that can be used for both playback and capture.
const char* g_malCommonDeviceNamesALSA[] = {
    "default",
    "null",
    "pulse",
    "jack"
};

// This array allows us to blacklist specific playback devices.
const char* g_malBlacklistedPlaybackDeviceNamesALSA[] = {
    ""
};

// This array allows us to blacklist specific capture devices.
const char* g_malBlacklistedCaptureDeviceNamesALSA[] = {
    ""
};


// This array allows mini_al to control device-specific default buffer sizes. This uses a scaling factor. Order is important. If
// any part of the string is present in the device's name, the associated scale will be used.
static struct
{
    const char* name;
    float scale;
} g_malDefaultBufferSizeScalesALSA[] = {
    {"bcm2835 IEC958/HDMI", 2.0f},
    {"bcm2835 ALSA",        2.0f}
};

float mal_find_default_buffer_size_scale__alsa(const char* deviceName)
{
    if (deviceName == NULL) {
        return 1;
    }

    for (size_t i = 0; i < mal_countof(g_malDefaultBufferSizeScalesALSA); ++i) {
        if (strstr(g_malDefaultBufferSizeScalesALSA[i].name, deviceName) != NULL) {
            return g_malDefaultBufferSizeScalesALSA[i].scale;
        }
    }

    return 1;
}

mal_snd_pcm_format_t mal_convert_mal_format_to_alsa_format(mal_format format)
{
    mal_snd_pcm_format_t ALSAFormats[] = {
        MAL_SND_PCM_FORMAT_UNKNOWN,     // mal_format_unknown
        MAL_SND_PCM_FORMAT_U8,          // mal_format_u8
        MAL_SND_PCM_FORMAT_S16_LE,      // mal_format_s16
        MAL_SND_PCM_FORMAT_S24_3LE,     // mal_format_s24
        MAL_SND_PCM_FORMAT_S32_LE,      // mal_format_s32
        MAL_SND_PCM_FORMAT_FLOAT_LE     // mal_format_f32
    };

    if (mal_is_big_endian()) {
        ALSAFormats[0] = MAL_SND_PCM_FORMAT_UNKNOWN;
        ALSAFormats[1] = MAL_SND_PCM_FORMAT_U8;
        ALSAFormats[2] = MAL_SND_PCM_FORMAT_S16_BE;
        ALSAFormats[3] = MAL_SND_PCM_FORMAT_S24_3BE;
        ALSAFormats[4] = MAL_SND_PCM_FORMAT_S32_BE;
        ALSAFormats[5] = MAL_SND_PCM_FORMAT_FLOAT_BE;
    }


    return ALSAFormats[format];
}

mal_format mal_convert_alsa_format_to_mal_format(mal_snd_pcm_format_t formatALSA)
{
    if (mal_is_little_endian()) {
        switch (formatALSA) {
            case MAL_SND_PCM_FORMAT_S16_LE:   return mal_format_s16;
            case MAL_SND_PCM_FORMAT_S24_3LE:  return mal_format_s24;
            case MAL_SND_PCM_FORMAT_S32_LE:   return mal_format_s32;
            case MAL_SND_PCM_FORMAT_FLOAT_LE: return mal_format_f32;
            default: break;
        }
    } else {
        switch (formatALSA) {
            case MAL_SND_PCM_FORMAT_S16_BE:   return mal_format_s16;
            case MAL_SND_PCM_FORMAT_S24_3BE:  return mal_format_s24;
            case MAL_SND_PCM_FORMAT_S32_BE:   return mal_format_s32;
            case MAL_SND_PCM_FORMAT_FLOAT_BE: return mal_format_f32;
            default: break;
        }
    }

    // Endian agnostic.
    switch (formatALSA) {
        case MAL_SND_PCM_FORMAT_U8: return mal_format_u8;
        default: return mal_format_unknown;
    }
}

mal_channel mal_convert_alsa_channel_position_to_mal_channel(unsigned int alsaChannelPos)
{
    switch (alsaChannelPos)
    {
        case MAL_SND_CHMAP_MONO: return MAL_CHANNEL_MONO;
        case MAL_SND_CHMAP_FL:   return MAL_CHANNEL_FRONT_LEFT;
        case MAL_SND_CHMAP_FR:   return MAL_CHANNEL_FRONT_RIGHT;
        case MAL_SND_CHMAP_RL:   return MAL_CHANNEL_BACK_LEFT;
        case MAL_SND_CHMAP_RR:   return MAL_CHANNEL_BACK_RIGHT;
        case MAL_SND_CHMAP_FC:   return MAL_CHANNEL_FRONT_CENTER;
        case MAL_SND_CHMAP_LFE:  return MAL_CHANNEL_LFE;
        case MAL_SND_CHMAP_SL:   return MAL_CHANNEL_SIDE_LEFT;
        case MAL_SND_CHMAP_SR:   return MAL_CHANNEL_SIDE_RIGHT;
        case MAL_SND_CHMAP_RC:   return MAL_CHANNEL_BACK_CENTER;
        case MAL_SND_CHMAP_FLC:  return MAL_CHANNEL_FRONT_LEFT_CENTER;
        case MAL_SND_CHMAP_FRC:  return MAL_CHANNEL_FRONT_RIGHT_CENTER;
        case MAL_SND_CHMAP_RLC:  return 0;
        case MAL_SND_CHMAP_RRC:  return 0;
        case MAL_SND_CHMAP_FLW:  return 0;
        case MAL_SND_CHMAP_FRW:  return 0;
        case MAL_SND_CHMAP_FLH:  return 0;
        case MAL_SND_CHMAP_FCH:  return 0;
        case MAL_SND_CHMAP_FRH:  return 0;
        case MAL_SND_CHMAP_TC:   return MAL_CHANNEL_TOP_CENTER;
        case MAL_SND_CHMAP_TFL:  return MAL_CHANNEL_TOP_FRONT_LEFT;
        case MAL_SND_CHMAP_TFR:  return MAL_CHANNEL_TOP_FRONT_RIGHT;
        case MAL_SND_CHMAP_TFC:  return MAL_CHANNEL_TOP_FRONT_CENTER;
        case MAL_SND_CHMAP_TRL:  return MAL_CHANNEL_TOP_BACK_LEFT;
        case MAL_SND_CHMAP_TRR:  return MAL_CHANNEL_TOP_BACK_RIGHT;
        case MAL_SND_CHMAP_TRC:  return MAL_CHANNEL_TOP_BACK_CENTER;
        default: break;
    }

    return 0;
}

mal_bool32 mal_is_common_device_name__alsa(const char* name)
{
    for (size_t iName = 0; iName < mal_countof(g_malCommonDeviceNamesALSA); ++iName) {
        if (mal_strcmp(name, g_malCommonDeviceNamesALSA[iName]) == 0) {
            return MAL_TRUE;
        }
    }

    return MAL_FALSE;
}


mal_bool32 mal_is_playback_device_blacklisted__alsa(const char* name)
{
    for (size_t iName = 0; iName < mal_countof(g_malBlacklistedPlaybackDeviceNamesALSA); ++iName) {
        if (mal_strcmp(name, g_malBlacklistedPlaybackDeviceNamesALSA[iName]) == 0) {
            return MAL_TRUE;
        }
    }

    return MAL_FALSE;
}

mal_bool32 mal_is_capture_device_blacklisted__alsa(const char* name)
{
    for (size_t iName = 0; iName < mal_countof(g_malBlacklistedCaptureDeviceNamesALSA); ++iName) {
        if (mal_strcmp(name, g_malBlacklistedCaptureDeviceNamesALSA[iName]) == 0) {
            return MAL_TRUE;
        }
    }

    return MAL_FALSE;
}

mal_bool32 mal_is_device_blacklisted__alsa(mal_device_type deviceType, const char* name)
{
    if (deviceType == mal_device_type_playback) {
        return mal_is_playback_device_blacklisted__alsa(name);
    } else {
        return mal_is_capture_device_blacklisted__alsa(name);
    }
}


const char* mal_find_char(const char* str, char c, int* index)
{
    int i = 0;
    for (;;) {
        if (str[i] == '\0') {
            if (index) *index = -1;
            return NULL;
        }

        if (str[i] == c) {
            if (index) *index = i;
            return str + i;
        }

        i += 1;
    }

    // Should never get here, but treat it as though the character was not found to make me feel
    // better inside.
    if (index) *index = -1;
    return NULL;
}

mal_bool32 mal_is_device_name_in_hw_format__alsa(const char* hwid)
{
    // This function is just checking whether or not hwid is in "hw:%d,%d" format.

    if (hwid == NULL) {
        return MAL_FALSE;
    }

    if (hwid[0] != 'h' || hwid[1] != 'w' || hwid[2] != ':') {
        return MAL_FALSE;
    }

    hwid += 3;

    int commaPos;
    const char* dev = mal_find_char(hwid, ',', &commaPos);
    if (dev == NULL) {
        return MAL_FALSE;
    } else {
        dev += 1;   // Skip past the ",".
    }

    // Check if the part between the ":" and the "," contains only numbers. If not, return false.
    for (int i = 0; i < commaPos; ++i) {
        if (hwid[i] < '0' || hwid[i] > '9') {
            return MAL_FALSE;
        }
    }

    // Check if everything after the "," is numeric. If not, return false.
    int i = 0;
    while (dev[i] != '\0') {
        if (dev[i] < '0' || dev[i] > '9') {
            return MAL_FALSE;
        }
        i += 1;
    }

    return MAL_TRUE;
}

int mal_convert_device_name_to_hw_format__alsa(mal_context* pContext, char* dst, size_t dstSize, const char* src)  // Returns 0 on success, non-0 on error.
{
    // src should look something like this: "hw:CARD=I82801AAICH,DEV=0"

    if (dst == NULL) return -1;
    if (dstSize < 7) return -1;     // Absolute minimum size of the output buffer is 7 bytes.

    *dst = '\0';    // Safety.
    if (src == NULL) return -1;

    // If the input name is already in "hw:%d,%d" format, just return that verbatim.
    if (mal_is_device_name_in_hw_format__alsa(src)) {
        return mal_strcpy_s(dst, dstSize, src);
    }


    int colonPos;
    src = mal_find_char(src, ':', &colonPos);
    if (src == NULL) {
        return -1;  // Couldn't find a colon
    }

    char card[256];

    int commaPos;
    const char* dev = mal_find_char(src, ',', &commaPos);
    if (dev == NULL) {
        dev = "0";
        mal_strncpy_s(card, sizeof(card), src+6, (size_t)-1);   // +6 = ":CARD="
    } else {
        dev = dev + 5;  // +5 = ",DEV="
        mal_strncpy_s(card, sizeof(card), src+6, commaPos-6);   // +6 = ":CARD="
    }

    int cardIndex = ((mal_snd_card_get_index_proc)pContext->alsa.snd_card_get_index)(card);
    if (cardIndex < 0) {
        return -2;  // Failed to retrieve the card index.
    }

    //printf("TESTING: CARD=%s,DEV=%s\n", card, dev);


    // Construction.
    dst[0] = 'h'; dst[1] = 'w'; dst[2] = ':';
    if (mal_itoa_s(cardIndex, dst+3, dstSize-3, 10) != 0) {
        return -3;
    }
    if (mal_strcat_s(dst, dstSize, ",") != 0) {
        return -3;
    }
    if (mal_strcat_s(dst, dstSize, dev) != 0) {
        return -3;
    }

    return 0;
}

mal_bool32 mal_does_id_exist_in_list__alsa(mal_device_id* pUniqueIDs, mal_uint32 count, const char* pHWID)
{
    mal_assert(pHWID != NULL);

    for (mal_uint32 i = 0; i < count; ++i) {
        if (mal_strcmp(pUniqueIDs[i].alsa, pHWID) == 0) {
            return MAL_TRUE;
        }
    }

    return MAL_FALSE;
}


mal_result mal_context_open_pcm__alsa(mal_context* pContext, mal_share_mode shareMode, mal_device_type type, const mal_device_id* pDeviceID, mal_snd_pcm_t** ppPCM)
{
    mal_assert(pContext != NULL);
    mal_assert(ppPCM != NULL);

    *ppPCM = NULL;

    mal_snd_pcm_t* pPCM = NULL;

    

    mal_snd_pcm_stream_t stream = (type == mal_device_type_playback) ? MAL_SND_PCM_STREAM_PLAYBACK : MAL_SND_PCM_STREAM_CAPTURE;
    int openMode = MAL_SND_PCM_NO_AUTO_RESAMPLE | MAL_SND_PCM_NO_AUTO_CHANNELS | MAL_SND_PCM_NO_AUTO_FORMAT;

    if (pDeviceID == NULL) {
        // We're opening the default device. I don't know if trying anything other than "default" is necessary, but it makes
        // me feel better to try as hard as we can get to get _something_ working.
        const char* defaultDeviceNames[] = {
            "default",
            NULL,
            NULL,
            NULL,
            NULL,
            NULL,
            NULL
        };

        if (shareMode == mal_share_mode_exclusive) {
            defaultDeviceNames[1] = "hw";
            defaultDeviceNames[2] = "hw:0";
            defaultDeviceNames[3] = "hw:0,0";
        } else {
            if (type == mal_device_type_playback) {
                defaultDeviceNames[1] = "dmix";
                defaultDeviceNames[2] = "dmix:0";
                defaultDeviceNames[3] = "dmix:0,0";
            } else {
                defaultDeviceNames[1] = "dsnoop";
                defaultDeviceNames[2] = "dsnoop:0";
                defaultDeviceNames[3] = "dsnoop:0,0";
            }
            defaultDeviceNames[4] = "hw";
            defaultDeviceNames[5] = "hw:0";
            defaultDeviceNames[6] = "hw:0,0";
        }

        mal_bool32 isDeviceOpen = MAL_FALSE;
        for (size_t i = 0; i < mal_countof(defaultDeviceNames); ++i) {
            if (defaultDeviceNames[i] != NULL && defaultDeviceNames[i][0] != '\0') {
                if (((mal_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, defaultDeviceNames[i], stream, openMode) == 0) {
                    isDeviceOpen = MAL_TRUE;
                    break;
                }
            }
        }

        if (!isDeviceOpen) {
            return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_open() failed when trying to open an appropriate default device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    } else {
        // We're trying to open a specific device. There's a few things to consider here:
        //
        // mini_al recongnizes a special format of device id that excludes the "hw", "dmix", etc. prefix. It looks like this: ":0,0", ":0,1", etc. When
        // an ID of this format is specified, it indicates to mini_al that it can try different combinations of plugins ("hw", "dmix", etc.) until it
        // finds an appropriate one that works. This comes in very handy when trying to open a device in shared mode ("dmix"), vs exclusive mode ("hw").

        // May end up needing to make small adjustments to the ID, so make a copy.
        mal_device_id deviceID = *pDeviceID;

        mal_bool32 isDeviceOpen = MAL_FALSE;
        if (deviceID.alsa[0] != ':') {
            // The ID is not in ":0,0" format. Use the ID exactly as-is.
            if (((mal_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, deviceID.alsa, stream, openMode) == 0) {
                isDeviceOpen = MAL_TRUE;
            }
        } else {
            // The ID is in ":0,0" format. Try different plugins depending on the shared mode.
            if (deviceID.alsa[1] == '\0') {
                deviceID.alsa[0] = '\0';  // An ID of ":" should be converted to "".
            }

            char hwid[256];
            if (shareMode == mal_share_mode_shared) {
                if (type == mal_device_type_playback) {
                    mal_strcpy_s(hwid, sizeof(hwid), "dmix");
                } else {
                    mal_strcpy_s(hwid, sizeof(hwid), "dsnoop");
                }

                if (mal_strcat_s(hwid, sizeof(hwid), deviceID.alsa) == 0) {
                    if (((mal_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, hwid, stream, openMode) == 0) {
                        isDeviceOpen = MAL_TRUE;
                    }
                }
            }

            // If at this point we still don't have an open device it means we're either preferencing exclusive mode or opening with "dmix"/"dsnoop" failed.
            if (!isDeviceOpen) {
                mal_strcpy_s(hwid, sizeof(hwid), "hw");
                if (mal_strcat_s(hwid, sizeof(hwid), deviceID.alsa) == 0) {
                    if (((mal_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, hwid, stream, openMode) == 0) {
                        isDeviceOpen = MAL_TRUE;
                    }
                }
            }
        }

        if (!isDeviceOpen) {
            return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_open() failed.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    }

    *ppPCM = pPCM;
    return MAL_SUCCESS;
}


mal_bool32 mal_context_is_device_id_equal__alsa(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return mal_strcmp(pID0->alsa, pID1->alsa) == 0;
}

mal_result mal_context_enumerate_devices__alsa(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    mal_bool32 cbResult = MAL_TRUE;

    mal_mutex_lock(&pContext->alsa.internalDeviceEnumLock);

    char** ppDeviceHints;
    if (((mal_snd_device_name_hint_proc)pContext->alsa.snd_device_name_hint)(-1, "pcm", (void***)&ppDeviceHints) < 0) {
        mal_mutex_unlock(&pContext->alsa.internalDeviceEnumLock);
        return MAL_NO_BACKEND;
    }

    mal_device_id* pUniqueIDs = NULL;
    mal_uint32 uniqueIDCount = 0;

    char** ppNextDeviceHint = ppDeviceHints;
    while (*ppNextDeviceHint != NULL) {
        char* NAME = ((mal_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "NAME");
        char* DESC = ((mal_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "DESC");
        char* IOID = ((mal_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "IOID");

        mal_device_type deviceType = mal_device_type_playback;
        if ((IOID == NULL || mal_strcmp(IOID, "Output") == 0)) {
            deviceType = mal_device_type_playback;
        }
        if ((IOID != NULL && mal_strcmp(IOID, "Input" ) == 0)) {
            deviceType = mal_device_type_capture;
        }

        mal_bool32 stopEnumeration = MAL_FALSE;
#if 0
        printf("NAME: %s\n", NAME);
        printf("DESC: %s\n", DESC);
        printf("IOID: %s\n", IOID);

        char hwid2[256];
        mal_convert_device_name_to_hw_format__alsa(pContext, hwid2, sizeof(hwid2), NAME);
        printf("DEVICE ID: %s\n\n", hwid2);
#endif

        char hwid[sizeof(pUniqueIDs->alsa)];
        if (NAME != NULL) {
            if (pContext->config.alsa.useVerboseDeviceEnumeration) {
                // Verbose mode. Use the name exactly as-is.
                mal_strncpy_s(hwid, sizeof(hwid), NAME, (size_t)-1);
            } else {
                // Simplified mode. Use ":%d,%d" format.
                if (mal_convert_device_name_to_hw_format__alsa(pContext, hwid, sizeof(hwid), NAME) == 0) {
                    // At this point, hwid looks like "hw:0,0". In simplified enumeration mode, we actually want to strip off the
                    // plugin name so it looks like ":0,0". The reason for this is that this special format is detected at device
                    // initialization time and is used as an indicator to try and use the most appropriate plugin depending on the
                    // device type and sharing mode.
                    char* dst = hwid;
                    char* src = hwid+2;
                    while ((*dst++ = *src++));
                } else {
                    // Conversion to "hw:%d,%d" failed. Just use the name as-is.
                    mal_strncpy_s(hwid, sizeof(hwid), NAME, (size_t)-1);
                }

                if (mal_does_id_exist_in_list__alsa(pUniqueIDs, uniqueIDCount, hwid)) {
                    goto next_device;   // The device has already been enumerated. Move on to the next one.
                } else {
                    // The device has not yet been enumerated. Make sure it's added to our list so that it's not enumerated again.
                    mal_device_id* pNewUniqueIDs = (mal_device_id*)mal_realloc(pUniqueIDs, sizeof(*pUniqueIDs) * (uniqueIDCount + 1));
                    if (pNewUniqueIDs == NULL) {
                        goto next_device;   // Failed to allocate memory.
                    }

                    pUniqueIDs = pNewUniqueIDs;
                    mal_copy_memory(pUniqueIDs[uniqueIDCount].alsa, hwid, sizeof(hwid));
                    uniqueIDCount += 1;
                }
            }
        } else {
            mal_zero_memory(hwid, sizeof(hwid));
        }

        mal_device_info deviceInfo;
        mal_zero_object(&deviceInfo);
        mal_strncpy_s(deviceInfo.id.alsa, sizeof(deviceInfo.id.alsa), hwid, (size_t)-1);

        // DESC is the friendly name. We treat this slightly differently depending on whether or not we are using verbose
        // device enumeration. In verbose mode we want to take the entire description so that the end-user can distinguish
        // between the subdevices of each card/dev pair. In simplified mode, however, we only want the first part of the
        // description.
        //
        // The value in DESC seems to be split into two lines, with the first line being the name of the device and the
        // second line being a description of the device. I don't like having the description be across two lines because
        // it makes formatting ugly and annoying. I'm therefore deciding to put it all on a single line with the second line
        // being put into parentheses. In simplified mode I'm just stripping the second line entirely.
        if (DESC != NULL) {
            int lfPos;
            const char* line2 = mal_find_char(DESC, '\n', &lfPos);
            if (line2 != NULL) {
                line2 += 1; // Skip past the new-line character.

                if (pContext->config.alsa.useVerboseDeviceEnumeration) {
                    // Verbose mode. Put the second line in brackets.
                    mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), DESC, lfPos);
                    mal_strcat_s (deviceInfo.name, sizeof(deviceInfo.name), " (");
                    mal_strcat_s (deviceInfo.name, sizeof(deviceInfo.name), line2);
                    mal_strcat_s (deviceInfo.name, sizeof(deviceInfo.name), ")");
                } else {
                    // Simplified mode. Strip the second line entirely.
                    mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), DESC, lfPos);
                }
            } else {
                // There's no second line. Just copy the whole description.
                mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), DESC, (size_t)-1);
            }
        }

        if (!mal_is_device_blacklisted__alsa(deviceType, NAME)) {
            cbResult = callback(pContext, deviceType, &deviceInfo, pUserData);
        }

        // Some devices are both playback and capture, but they are only enumerated by ALSA once. We need to fire the callback
        // again for the other device type in this case. We do this for known devices.
        if (cbResult) {
            if (mal_is_common_device_name__alsa(NAME)) {
                if (deviceType == mal_device_type_playback) {
                    if (!mal_is_capture_device_blacklisted__alsa(NAME)) {
                        cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
                    }
                } else {
                    if (!mal_is_playback_device_blacklisted__alsa(NAME)) {
                        cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
                    }
                }
            }
        }

        if (cbResult == MAL_FALSE) {
            stopEnumeration = MAL_TRUE;
        }

    next_device:
        free(NAME);
        free(DESC);
        free(IOID);
        ppNextDeviceHint += 1;

        // We need to stop enumeration if the callback returned false.
        if (stopEnumeration) {
            break;
        }
    }

    mal_free(pUniqueIDs);
    ((mal_snd_device_name_free_hint_proc)pContext->alsa.snd_device_name_free_hint)((void**)ppDeviceHints);

    mal_mutex_unlock(&pContext->alsa.internalDeviceEnumLock);

    return MAL_SUCCESS;
}


typedef struct
{
    mal_device_type deviceType;
    const mal_device_id* pDeviceID;
    mal_share_mode shareMode;
    mal_device_info* pDeviceInfo;
    mal_bool32 foundDevice;
} mal_context_get_device_info_enum_callback_data__alsa;

mal_bool32 mal_context_get_device_info_enum_callback__alsa(mal_context* pContext, mal_device_type deviceType, const mal_device_info* pDeviceInfo, void* pUserData)
{
    mal_context_get_device_info_enum_callback_data__alsa* pData = (mal_context_get_device_info_enum_callback_data__alsa*)pUserData;
    mal_assert(pData != NULL);

    if (pData->pDeviceID == NULL && mal_strcmp(pDeviceInfo->id.alsa, "default") == 0) {
        mal_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pDeviceInfo->name, (size_t)-1);
        pData->foundDevice = MAL_TRUE;
    } else {
        if (pData->deviceType == deviceType && mal_context_is_device_id_equal__alsa(pContext, pData->pDeviceID, &pDeviceInfo->id)) {
            mal_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pDeviceInfo->name, (size_t)-1);
            pData->foundDevice = MAL_TRUE;
        }
    }

    // Keep enumerating until we have found the device.
    return !pData->foundDevice;
}

mal_result mal_context_get_device_info__alsa(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);

    // We just enumerate to find basic information about the device.
    mal_context_get_device_info_enum_callback_data__alsa data;
    data.deviceType = deviceType;
    data.pDeviceID = pDeviceID;
    data.shareMode = shareMode;
    data.pDeviceInfo = pDeviceInfo;
    data.foundDevice = MAL_FALSE;
    mal_result result = mal_context_enumerate_devices__alsa(pContext, mal_context_get_device_info_enum_callback__alsa, &data);
    if (result != MAL_SUCCESS) {
        return result;
    }

    if (!data.foundDevice) {
        return MAL_NO_DEVICE;
    }


    // For detailed info we need to open the device.
    mal_snd_pcm_t* pPCM;
    result = mal_context_open_pcm__alsa(pContext, shareMode, deviceType, pDeviceID, &pPCM);
    if (result != MAL_SUCCESS) {
        return result;
    }

    // We need to initialize a HW parameters object in order to know what formats are supported.
    mal_snd_pcm_hw_params_t* pHWParams = (mal_snd_pcm_hw_params_t*)alloca(((mal_snd_pcm_hw_params_sizeof_proc)pContext->alsa.snd_pcm_hw_params_sizeof)());
    mal_zero_memory(pHWParams, ((mal_snd_pcm_hw_params_sizeof_proc)pContext->alsa.snd_pcm_hw_params_sizeof)());

    if (((mal_snd_pcm_hw_params_any_proc)pContext->alsa.snd_pcm_hw_params_any)(pPCM, pHWParams) < 0) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to initialize hardware parameters. snd_pcm_hw_params_any() failed.", MAL_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
    }

    int sampleRateDir = 0;

    ((mal_snd_pcm_hw_params_get_channels_min_proc)pContext->alsa.snd_pcm_hw_params_get_channels_min)(pHWParams, &pDeviceInfo->minChannels);
    ((mal_snd_pcm_hw_params_get_channels_max_proc)pContext->alsa.snd_pcm_hw_params_get_channels_max)(pHWParams, &pDeviceInfo->maxChannels);
    ((mal_snd_pcm_hw_params_get_rate_min_proc)pContext->alsa.snd_pcm_hw_params_get_rate_min)(pHWParams, &pDeviceInfo->minSampleRate, &sampleRateDir);
    ((mal_snd_pcm_hw_params_get_rate_max_proc)pContext->alsa.snd_pcm_hw_params_get_rate_max)(pHWParams, &pDeviceInfo->maxSampleRate, &sampleRateDir);

    // Formats.
    mal_snd_pcm_format_mask_t* pFormatMask = (mal_snd_pcm_format_mask_t*)alloca(((mal_snd_pcm_format_mask_sizeof_proc)pContext->alsa.snd_pcm_format_mask_sizeof)());
    mal_zero_memory(pFormatMask, ((mal_snd_pcm_format_mask_sizeof_proc)pContext->alsa.snd_pcm_format_mask_sizeof)());
    ((mal_snd_pcm_hw_params_get_format_mask_proc)pContext->alsa.snd_pcm_hw_params_get_format_mask)(pHWParams, pFormatMask);

    pDeviceInfo->formatCount = 0;
    if (((mal_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MAL_SND_PCM_FORMAT_U8)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = mal_format_u8;
    }
    if (((mal_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MAL_SND_PCM_FORMAT_S16_LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = mal_format_s16;
    }
    if (((mal_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MAL_SND_PCM_FORMAT_S24_3LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = mal_format_s24;
    }
    if (((mal_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MAL_SND_PCM_FORMAT_S32_LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = mal_format_s32;
    }
    if (((mal_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MAL_SND_PCM_FORMAT_FLOAT_LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = mal_format_f32;
    }

    ((mal_snd_pcm_close_proc)pContext->alsa.snd_pcm_close)(pPCM);
    return MAL_SUCCESS;
}


// Waits for a number of frames to become available for either capture or playback. The return
// value is the number of frames available.
//
// This will return early if the main loop is broken with mal_device__break_main_loop().
mal_uint32 mal_device__wait_for_frames__alsa(mal_device* pDevice, mal_bool32* pRequiresRestart)
{
    mal_assert(pDevice != NULL);

    if (pRequiresRestart) *pRequiresRestart = MAL_FALSE;

    // I want it so that this function returns the period size in frames. We just wait until that number of frames are available and then return.
    mal_uint32 periodSizeInFrames = pDevice->bufferSizeInFrames / pDevice->periods;
    while (!pDevice->alsa.breakFromMainLoop) {
        mal_snd_pcm_sframes_t framesAvailable = ((mal_snd_pcm_avail_update_proc)pDevice->pContext->alsa.snd_pcm_avail_update)((mal_snd_pcm_t*)pDevice->alsa.pPCM);
        if (framesAvailable < 0) {
            if (framesAvailable == -EPIPE) {
                if (((mal_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((mal_snd_pcm_t*)pDevice->alsa.pPCM, framesAvailable, MAL_TRUE) < 0) {
                    return 0;
                }

                // A device recovery means a restart for mmap mode.
                if (pRequiresRestart) {
                    *pRequiresRestart = MAL_TRUE;
                }

                // Try again, but if it fails this time just return an error.
                framesAvailable = ((mal_snd_pcm_avail_update_proc)pDevice->pContext->alsa.snd_pcm_avail_update)((mal_snd_pcm_t*)pDevice->alsa.pPCM);
                if (framesAvailable < 0) {
                    return 0;
                }
            }
        }

        if (framesAvailable >= periodSizeInFrames) {
            return periodSizeInFrames;
        }

        if (framesAvailable < periodSizeInFrames) {
            // Less than a whole period is available so keep waiting.
            int waitResult = ((mal_snd_pcm_wait_proc)pDevice->pContext->alsa.snd_pcm_wait)((mal_snd_pcm_t*)pDevice->alsa.pPCM, -1);
            if (waitResult < 0) {
                if (waitResult == -EPIPE) {
                    if (((mal_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((mal_snd_pcm_t*)pDevice->alsa.pPCM, waitResult, MAL_TRUE) < 0) {
                        return 0;
                    }

                    // A device recovery means a restart for mmap mode.
                    if (pRequiresRestart) {
                        *pRequiresRestart = MAL_TRUE;
                    }
                }
            }
        }
    }

    // We'll get here if the loop was terminated. Just return whatever's available.
    mal_snd_pcm_sframes_t framesAvailable = ((mal_snd_pcm_avail_update_proc)pDevice->pContext->alsa.snd_pcm_avail_update)((mal_snd_pcm_t*)pDevice->alsa.pPCM);
    if (framesAvailable < 0) {
        return 0;
    }

    return framesAvailable;
}

mal_bool32 mal_device_write__alsa(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);
    if (!mal_device_is_started(pDevice) && mal_device__get_state(pDevice) != MAL_STATE_STARTING) {
        return MAL_FALSE;
    }
    if (pDevice->alsa.breakFromMainLoop) {
        return MAL_FALSE;
    }

    if (pDevice->alsa.isUsingMMap) {
        // mmap.
        mal_bool32 requiresRestart;
        mal_uint32 framesAvailable = mal_device__wait_for_frames__alsa(pDevice, &requiresRestart);
        if (framesAvailable == 0) {
            return MAL_FALSE;
        }

        // Don't bother asking the client for more audio data if we're just stopping the device anyway.
        if (pDevice->alsa.breakFromMainLoop) {
            return MAL_FALSE;
        }

        const mal_snd_pcm_channel_area_t* pAreas;
        mal_snd_pcm_uframes_t mappedOffset;
        mal_snd_pcm_uframes_t mappedFrames = framesAvailable;
        while (framesAvailable > 0) {
            int result = ((mal_snd_pcm_mmap_begin_proc)pDevice->pContext->alsa.snd_pcm_mmap_begin)((mal_snd_pcm_t*)pDevice->alsa.pPCM, &pAreas, &mappedOffset, &mappedFrames);
            if (result < 0) {
                return MAL_FALSE;
            }

            if (mappedFrames > 0) {
                void* pBuffer = (mal_uint8*)pAreas[0].addr + ((pAreas[0].first + (mappedOffset * pAreas[0].step)) / 8);
                mal_device__read_frames_from_client(pDevice, mappedFrames, pBuffer);
            }

            result = ((mal_snd_pcm_mmap_commit_proc)pDevice->pContext->alsa.snd_pcm_mmap_commit)((mal_snd_pcm_t*)pDevice->alsa.pPCM, mappedOffset, mappedFrames);
            if (result < 0 || (mal_snd_pcm_uframes_t)result != mappedFrames) {
                ((mal_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((mal_snd_pcm_t*)pDevice->alsa.pPCM, result, MAL_TRUE);
                return MAL_FALSE;
            }

            if (requiresRestart) {
                if (((mal_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((mal_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
                    return MAL_FALSE;
                }
            }

            if (framesAvailable >= mappedFrames) {
                framesAvailable -= mappedFrames;
            } else {
                framesAvailable = 0;
            }
        }
    } else {
        // readi/writei.
        while (!pDevice->alsa.breakFromMainLoop) {
            mal_uint32 framesAvailable = mal_device__wait_for_frames__alsa(pDevice, NULL);
            if (framesAvailable == 0) {
                continue;
            }

            // Don't bother asking the client for more audio data if we're just stopping the device anyway.
            if (pDevice->alsa.breakFromMainLoop) {
                return MAL_FALSE;
            }

            mal_device__read_frames_from_client(pDevice, framesAvailable, pDevice->alsa.pIntermediaryBuffer);

            mal_snd_pcm_sframes_t framesWritten = ((mal_snd_pcm_writei_proc)pDevice->pContext->alsa.snd_pcm_writei)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
            if (framesWritten < 0) {
                if (framesWritten == -EAGAIN) {
                    continue;   // Just keep trying...
                } else if (framesWritten == -EPIPE) {
                    // Underrun. Just recover and try writing again.
                    if (((mal_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((mal_snd_pcm_t*)pDevice->alsa.pPCM, framesWritten, MAL_TRUE) < 0) {
                        mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to recover device after underrun.", MAL_FAILED_TO_START_BACKEND_DEVICE);
                        return MAL_FALSE;
                    }

                    framesWritten = ((mal_snd_pcm_writei_proc)pDevice->pContext->alsa.snd_pcm_writei)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
                    if (framesWritten < 0) {
                        mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to write data to the internal device.", MAL_FAILED_TO_SEND_DATA_TO_DEVICE);
                        return MAL_FALSE;
                    }

                    break;  // Success.
                } else {
                    mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_writei() failed when writing initial data.", MAL_FAILED_TO_SEND_DATA_TO_DEVICE);
                    return MAL_FALSE;
                }
            } else {
                break;  // Success.
            }
        }
    }

    return MAL_TRUE;
}

mal_bool32 mal_device_read__alsa(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);
    if (!mal_device_is_started(pDevice)) {
        return MAL_FALSE;
    }
    if (pDevice->alsa.breakFromMainLoop) {
        return MAL_FALSE;
    }

    mal_uint32 framesToSend = 0;
    void* pBuffer = NULL;
    if (pDevice->alsa.pIntermediaryBuffer == NULL) {
        // mmap.
        mal_bool32 requiresRestart;
        mal_uint32 framesAvailable = mal_device__wait_for_frames__alsa(pDevice, &requiresRestart);
        if (framesAvailable == 0) {
            return MAL_FALSE;
        }

        const mal_snd_pcm_channel_area_t* pAreas;
        mal_snd_pcm_uframes_t mappedOffset;
        mal_snd_pcm_uframes_t mappedFrames = framesAvailable;
        while (framesAvailable > 0) {
            int result = ((mal_snd_pcm_mmap_begin_proc)pDevice->pContext->alsa.snd_pcm_mmap_begin)((mal_snd_pcm_t*)pDevice->alsa.pPCM, &pAreas, &mappedOffset, &mappedFrames);
            if (result < 0) {
                return MAL_FALSE;
            }

            if (mappedFrames > 0) {
                void* pBuffer = (mal_uint8*)pAreas[0].addr + ((pAreas[0].first + (mappedOffset * pAreas[0].step)) / 8);
                mal_device__send_frames_to_client(pDevice, mappedFrames, pBuffer);
            }

            result = ((mal_snd_pcm_mmap_commit_proc)pDevice->pContext->alsa.snd_pcm_mmap_commit)((mal_snd_pcm_t*)pDevice->alsa.pPCM, mappedOffset, mappedFrames);
            if (result < 0 || (mal_snd_pcm_uframes_t)result != mappedFrames) {
                ((mal_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((mal_snd_pcm_t*)pDevice->alsa.pPCM, result, MAL_TRUE);
                return MAL_FALSE;
            }

            if (requiresRestart) {
                if (((mal_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((mal_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
                    return MAL_FALSE;
                }
            }

            if (framesAvailable >= mappedFrames) {
                framesAvailable -= mappedFrames;
            } else {
                framesAvailable = 0;
            }
        }
    } else {
        // readi/writei.
        mal_snd_pcm_sframes_t framesRead = 0;
        while (!pDevice->alsa.breakFromMainLoop) {
            mal_uint32 framesAvailable = mal_device__wait_for_frames__alsa(pDevice, NULL);
            if (framesAvailable == 0) {
                continue;
            }

            framesRead = ((mal_snd_pcm_readi_proc)pDevice->pContext->alsa.snd_pcm_readi)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
            if (framesRead < 0) {
                if (framesRead == -EAGAIN) {
                    continue;   // Just keep trying...
                } else if (framesRead == -EPIPE) {
                    // Overrun. Just recover and try reading again.
                    if (((mal_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((mal_snd_pcm_t*)pDevice->alsa.pPCM, framesRead, MAL_TRUE) < 0) {
                        mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to recover device after overrun.", MAL_FAILED_TO_START_BACKEND_DEVICE);
                        return MAL_FALSE;
                    }

                    framesRead = ((mal_snd_pcm_readi_proc)pDevice->pContext->alsa.snd_pcm_readi)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
                    if (framesRead < 0) {
                        mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to read data from the internal device.", MAL_FAILED_TO_READ_DATA_FROM_DEVICE);
                        return MAL_FALSE;
                    }

                    break;  // Success.
                } else {
                    return MAL_FALSE;
                }
            } else {
                break;  // Success.
            }
        }

        framesToSend = framesRead;
        pBuffer = pDevice->alsa.pIntermediaryBuffer;
    }

    if (framesToSend > 0) {
        mal_device__send_frames_to_client(pDevice, framesToSend, pBuffer);
    }

    return MAL_TRUE;
}

void mal_device_uninit__alsa(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if ((mal_snd_pcm_t*)pDevice->alsa.pPCM) {
        ((mal_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)((mal_snd_pcm_t*)pDevice->alsa.pPCM);

        if (pDevice->alsa.pIntermediaryBuffer != NULL) {
            mal_free(pDevice->alsa.pIntermediaryBuffer);
        }
    }
}

mal_result mal_device_init__alsa(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pContext;

    mal_assert(pDevice != NULL);
    mal_zero_object(&pDevice->alsa);

    mal_snd_pcm_format_t formatALSA = mal_convert_mal_format_to_alsa_format(pConfig->format);

    mal_result result = mal_context_open_pcm__alsa(pContext, pConfig->shareMode, type, pDeviceID, (mal_snd_pcm_t**)&pDevice->alsa.pPCM);
    if (result != MAL_SUCCESS) {
        return result;
    }

    // We may be scaling the size of the buffer.
    float bufferSizeScaleFactor = 1;

    // If using the default buffer size we may want to apply some device-specific scaling for known devices that have peculiar  latency characteristics (looking at you Raspberry Pi!).
    if (pDevice->usingDefaultBufferSize) {
        mal_snd_pcm_info_t* pInfo = (mal_snd_pcm_info_t*)alloca(((mal_snd_pcm_info_sizeof_proc)pContext->alsa.snd_pcm_info_sizeof)());
        mal_zero_memory(pInfo, ((mal_snd_pcm_info_sizeof_proc)pContext->alsa.snd_pcm_info_sizeof)());

        // We may need to scale the size of the buffer depending on the device.
        if (((mal_snd_pcm_info_proc)pContext->alsa.snd_pcm_info)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pInfo) == 0) {
            const char* deviceName = ((mal_snd_pcm_info_get_name_proc)pContext->alsa.snd_pcm_info_get_name)(pInfo);
            if (deviceName != NULL) {
                if (mal_strcmp(deviceName, "default") == 0) {
                    // It's the default device. We need to use DESC from snd_device_name_hint().
                    char** ppDeviceHints;
                    if (((mal_snd_device_name_hint_proc)pContext->alsa.snd_device_name_hint)(-1, "pcm", (void***)&ppDeviceHints) < 0) {
                        return MAL_NO_BACKEND;
                    }

                    char** ppNextDeviceHint = ppDeviceHints;
                    while (*ppNextDeviceHint != NULL) {
                        char* NAME = ((mal_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "NAME");
                        char* DESC = ((mal_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "DESC");
                        char* IOID = ((mal_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "IOID");

                        mal_bool32 foundDevice = MAL_FALSE;
                        if ((type == mal_device_type_playback && (IOID == NULL || mal_strcmp(IOID, "Output") == 0)) ||
                            (type == mal_device_type_capture  && (IOID != NULL && mal_strcmp(IOID, "Input" ) == 0))) {
                            if (mal_strcmp(NAME, deviceName) == 0) {
                                bufferSizeScaleFactor = mal_find_default_buffer_size_scale__alsa(DESC);
                                foundDevice = MAL_TRUE;
                            }
                        }

                        free(NAME);
                        free(DESC);
                        free(IOID);
                        ppNextDeviceHint += 1;

                        if (foundDevice) {
                            break;
                        }
                    }

                    ((mal_snd_device_name_free_hint_proc)pContext->alsa.snd_device_name_free_hint)((void**)ppDeviceHints);
                } else {
                    bufferSizeScaleFactor = mal_find_default_buffer_size_scale__alsa(deviceName);
                }
            }
        }
    }


    // Hardware parameters.
    mal_snd_pcm_hw_params_t* pHWParams = (mal_snd_pcm_hw_params_t*)alloca(((mal_snd_pcm_hw_params_sizeof_proc)pContext->alsa.snd_pcm_hw_params_sizeof)());
    mal_zero_memory(pHWParams, ((mal_snd_pcm_hw_params_sizeof_proc)pContext->alsa.snd_pcm_hw_params_sizeof)());

    if (((mal_snd_pcm_hw_params_any_proc)pContext->alsa.snd_pcm_hw_params_any)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pHWParams) < 0) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to initialize hardware parameters. snd_pcm_hw_params_any() failed.", MAL_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
    }


    // MMAP Mode
    //
    // Try using interleaved MMAP access. If this fails, fall back to standard readi/writei.
    pDevice->alsa.isUsingMMap = MAL_FALSE;
    if (!pConfig->alsa.noMMap && pDevice->type != mal_device_type_capture) {    // <-- Disabling MMAP mode for capture devices because I apparently do not have a device that supports it which means I can't test it... Contributions welcome.
        if (((mal_snd_pcm_hw_params_set_access_proc)pContext->alsa.snd_pcm_hw_params_set_access)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pHWParams, MAL_SND_PCM_ACCESS_MMAP_INTERLEAVED) == 0) {
            pDevice->alsa.isUsingMMap = MAL_TRUE;
        }
    }

    if (!pDevice->alsa.isUsingMMap) {
        if (((mal_snd_pcm_hw_params_set_access_proc)pContext->alsa.snd_pcm_hw_params_set_access)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pHWParams, MAL_SND_PCM_ACCESS_RW_INTERLEAVED) < 0) {;
            mal_device_uninit__alsa(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to set access mode to neither SND_PCM_ACCESS_MMAP_INTERLEAVED nor SND_PCM_ACCESS_RW_INTERLEAVED. snd_pcm_hw_params_set_access() failed.", MAL_FORMAT_NOT_SUPPORTED);
        }
    }


    // Most important properties first. The documentation for OSS (yes, I know this is ALSA!) recommends format, channels, then sample rate. I can't
    // find any documentation for ALSA specifically, so I'm going to copy the recommendation for OSS.

    // Format.
    // Try getting every supported format.
    mal_snd_pcm_format_mask_t* pFormatMask = (mal_snd_pcm_format_mask_t*)alloca(((mal_snd_pcm_format_mask_sizeof_proc)pContext->alsa.snd_pcm_format_mask_sizeof)());
    mal_zero_memory(pFormatMask, ((mal_snd_pcm_format_mask_sizeof_proc)pContext->alsa.snd_pcm_format_mask_sizeof)());

    ((mal_snd_pcm_hw_params_get_format_mask_proc)pContext->alsa.snd_pcm_hw_params_get_format_mask)(pHWParams, pFormatMask);

    // At this point we should have a list of supported formats, so now we need to find the best one. We first check if the requested format is
    // supported, and if so, use that one. If it's not supported, we just run though a list of formats and try to find the best one.
    if (!((mal_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, formatALSA)) {
        // The requested format is not supported so now try running through the list of formats and return the best one.
        mal_snd_pcm_format_t preferredFormatsALSA[] = {
            MAL_SND_PCM_FORMAT_S16_LE,      // mal_format_s16
            MAL_SND_PCM_FORMAT_FLOAT_LE,    // mal_format_f32
            MAL_SND_PCM_FORMAT_S32_LE,      // mal_format_s32
            MAL_SND_PCM_FORMAT_S24_3LE,     // mal_format_s24
            MAL_SND_PCM_FORMAT_U8           // mal_format_u8
        };

        if (mal_is_big_endian()) {
            preferredFormatsALSA[0] = MAL_SND_PCM_FORMAT_S16_BE;
            preferredFormatsALSA[1] = MAL_SND_PCM_FORMAT_FLOAT_BE;
            preferredFormatsALSA[2] = MAL_SND_PCM_FORMAT_S32_BE;
            preferredFormatsALSA[3] = MAL_SND_PCM_FORMAT_S24_3BE;
            preferredFormatsALSA[4] = MAL_SND_PCM_FORMAT_U8;
        }


        formatALSA = MAL_SND_PCM_FORMAT_UNKNOWN;
        for (size_t i = 0; i < (sizeof(preferredFormatsALSA) / sizeof(preferredFormatsALSA[0])); ++i) {
            if (((mal_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, preferredFormatsALSA[i])) {
                formatALSA = preferredFormatsALSA[i];
                break;
            }
        }

        if (formatALSA == MAL_SND_PCM_FORMAT_UNKNOWN) {
            mal_device_uninit__alsa(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Format not supported. The device does not support any mini_al formats.", MAL_FORMAT_NOT_SUPPORTED);
        }
    }

    if (((mal_snd_pcm_hw_params_set_format_proc)pContext->alsa.snd_pcm_hw_params_set_format)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pHWParams, formatALSA) < 0) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Format not supported. snd_pcm_hw_params_set_format() failed.", MAL_FORMAT_NOT_SUPPORTED);
    }

    pDevice->internalFormat = mal_convert_alsa_format_to_mal_format(formatALSA);
    if (pDevice->internalFormat == mal_format_unknown) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] The chosen format is not supported by mini_al.", MAL_FORMAT_NOT_SUPPORTED);
    }


    // Channels.
    unsigned int channels = pConfig->channels;
    if (((mal_snd_pcm_hw_params_set_channels_near_proc)pContext->alsa.snd_pcm_hw_params_set_channels_near)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pHWParams, &channels) < 0) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to set channel count. snd_pcm_hw_params_set_channels_near() failed.", MAL_FORMAT_NOT_SUPPORTED);
    }
    pDevice->internalChannels = (mal_uint32)channels;


    // Sample Rate. It appears there's either a bug in ALSA, a bug in some drivers, or I'm doing something silly; but having resampling
    // enabled causes problems with some device configurations when used in conjunction with MMAP access mode. To fix this problem we
    // need to disable resampling.
    //
    // To reproduce this problem, open the "plug:dmix" device, and set the sample rate to 44100. Internally, it looks like dmix uses a
    // sample rate of 48000. The hardware parameters will get set correctly with no errors, but it looks like the 44100 -> 48000 resampling
    // doesn't work properly - but only with MMAP access mode. You will notice skipping/crackling in the audio, and it'll run at a slightly
    // faster rate.
    //
    // mini_al has built-in support for sample rate conversion (albeit low quality at the moment), so disabling resampling should be fine
    // for us. The only problem is that it won't be taking advantage of any kind of hardware-accelerated resampling and it won't be very
    // good quality until I get a chance to improve the quality of mini_al's software sample rate conversion.
    //
    // I don't currently know if the dmix plugin is the only one with this error. Indeed, this is the only one I've been able to reproduce
    // this error with. In the future, we may want to restrict the disabling of resampling to only known bad plugins.
    ((mal_snd_pcm_hw_params_set_rate_resample_proc)pContext->alsa.snd_pcm_hw_params_set_rate_resample)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pHWParams, 0);

    unsigned int sampleRate = pConfig->sampleRate;
    if (((mal_snd_pcm_hw_params_set_rate_near_proc)pContext->alsa.snd_pcm_hw_params_set_rate_near)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pHWParams, &sampleRate, 0) < 0) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Sample rate not supported. snd_pcm_hw_params_set_rate_near() failed.", MAL_FORMAT_NOT_SUPPORTED);
    }
    pDevice->internalSampleRate = (mal_uint32)sampleRate;


    // At this point we know the internal sample rate which means we can calculate the buffer size in frames.
    if (pDevice->bufferSizeInFrames == 0) {
        pDevice->bufferSizeInFrames = mal_scale_buffer_size(mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, pDevice->internalSampleRate), bufferSizeScaleFactor);
    }

    // Buffer Size
    mal_snd_pcm_uframes_t actualBufferSize = pDevice->bufferSizeInFrames;
    if (((mal_snd_pcm_hw_params_set_buffer_size_near_proc)pContext->alsa.snd_pcm_hw_params_set_buffer_size_near)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pHWParams, &actualBufferSize) < 0) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to set buffer size for device. snd_pcm_hw_params_set_buffer_size() failed.", MAL_FORMAT_NOT_SUPPORTED);
    }
    pDevice->bufferSizeInFrames = actualBufferSize;

    // Periods.
    mal_uint32 periods = pConfig->periods;
    if (((mal_snd_pcm_hw_params_set_periods_near_proc)pContext->alsa.snd_pcm_hw_params_set_periods_near)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pHWParams, &periods, NULL) < 0) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to set period count. snd_pcm_hw_params_set_periods_near() failed.", MAL_FORMAT_NOT_SUPPORTED);
    }
    pDevice->periods = periods;


    // Apply hardware parameters.
    if (((mal_snd_pcm_hw_params_proc)pContext->alsa.snd_pcm_hw_params)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pHWParams) < 0) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to set hardware parameters. snd_pcm_hw_params() failed.", MAL_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
    }



    // Software parameters.
    mal_snd_pcm_sw_params_t* pSWParams = (mal_snd_pcm_sw_params_t*)alloca(((mal_snd_pcm_sw_params_sizeof_proc)pContext->alsa.snd_pcm_sw_params_sizeof)());
    mal_zero_memory(pSWParams, ((mal_snd_pcm_sw_params_sizeof_proc)pContext->alsa.snd_pcm_sw_params_sizeof)());

    if (((mal_snd_pcm_sw_params_current_proc)pContext->alsa.snd_pcm_sw_params_current)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pSWParams) != 0) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to initialize software parameters. snd_pcm_sw_params_current() failed.", MAL_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
    }

    if (((mal_snd_pcm_sw_params_set_avail_min_proc)pContext->alsa.snd_pcm_sw_params_set_avail_min)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pSWParams, /*(pDevice->sampleRate/1000) * 1*/ mal_prev_power_of_2(pDevice->bufferSizeInFrames/pDevice->periods)) != 0) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_sw_params_set_avail_min() failed.", MAL_FORMAT_NOT_SUPPORTED);
    }

    if (type == mal_device_type_playback && !pDevice->alsa.isUsingMMap) {   // Only playback devices in writei/readi mode need a start threshold.
        if (((mal_snd_pcm_sw_params_set_start_threshold_proc)pContext->alsa.snd_pcm_sw_params_set_start_threshold)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pSWParams, /*(pDevice->sampleRate/1000) * 1*/ pDevice->bufferSizeInFrames/pDevice->periods) != 0) { //mal_prev_power_of_2(pDevice->bufferSizeInFrames/pDevice->periods)
            mal_device_uninit__alsa(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to set start threshold for playback device. snd_pcm_sw_params_set_start_threshold() failed.", MAL_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
        }
    }

    if (((mal_snd_pcm_sw_params_proc)pContext->alsa.snd_pcm_sw_params)((mal_snd_pcm_t*)pDevice->alsa.pPCM, pSWParams) != 0) {
        mal_device_uninit__alsa(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to set software parameters. snd_pcm_sw_params() failed.", MAL_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
    }



    // If we're _not_ using mmap we need to use an intermediary buffer.
    if (!pDevice->alsa.isUsingMMap) {
        pDevice->alsa.pIntermediaryBuffer = mal_malloc(pDevice->bufferSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat));
        if (pDevice->alsa.pIntermediaryBuffer == NULL) {
            mal_device_uninit__alsa(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to allocate memory for intermediary buffer.", MAL_OUT_OF_MEMORY);
        }
    }

    // Grab the internal channel map. For now we're not going to bother trying to change the channel map and
    // instead just do it ourselves.
    mal_snd_pcm_chmap_t* pChmap = ((mal_snd_pcm_get_chmap_proc)pContext->alsa.snd_pcm_get_chmap)((mal_snd_pcm_t*)pDevice->alsa.pPCM);
    if (pChmap != NULL) {
        // There are cases where the returned channel map can have a different channel count than was returned by snd_pcm_hw_params_set_channels_near().
        if (pChmap->channels >= pDevice->internalChannels) {
            // Drop excess channels.
            for (mal_uint32 iChannel = 0; iChannel < pDevice->internalChannels; ++iChannel) {
                pDevice->internalChannelMap[iChannel] = mal_convert_alsa_channel_position_to_mal_channel(pChmap->pos[iChannel]);
            }
        } else {
            // Excess channels use defaults. Do an initial fill with defaults, overwrite the first pChmap->channels, validate to ensure there are no duplicate
            // channels. If validation fails, fall back to defaults.

            // Fill with defaults.
            mal_get_standard_channel_map(mal_standard_channel_map_alsa, pDevice->internalChannels, pDevice->internalChannelMap);

            // Overwrite first pChmap->channels channels.
            for (mal_uint32 iChannel = 0; iChannel < pChmap->channels; ++iChannel) {
                pDevice->internalChannelMap[iChannel] = mal_convert_alsa_channel_position_to_mal_channel(pChmap->pos[iChannel]);
            }

            // Validate.
            mal_bool32 isValid = MAL_TRUE;
            for (mal_uint32 i = 0; i < pDevice->internalChannels && isValid; ++i) {
                for (mal_uint32 j = i+1; j < pDevice->internalChannels; ++j) {
                    if (pDevice->internalChannelMap[i] == pDevice->internalChannelMap[j]) {
                        isValid = MAL_FALSE;
                        break;
                    }
                }
            }

            // If our channel map is invalid, fall back to defaults.
            if (!isValid) {
                mal_get_standard_channel_map(mal_standard_channel_map_alsa, pDevice->internalChannels, pDevice->internalChannelMap);
            }
        }

        free(pChmap);
        pChmap = NULL;
    } else {
        // Could not retrieve the channel map. Fall back to a hard-coded assumption.
        mal_get_standard_channel_map(mal_standard_channel_map_alsa, pDevice->internalChannels, pDevice->internalChannelMap);
    }

    return MAL_SUCCESS;
}


mal_result mal_device__start_backend__alsa(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // Prepare the device first...
    if (((mal_snd_pcm_prepare_proc)pDevice->pContext->alsa.snd_pcm_prepare)((mal_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to prepare device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
    }

    // ... and then grab an initial chunk from the client. After this is done, the device should
    // automatically start playing, since that's how we configured the software parameters.
    if (pDevice->type == mal_device_type_playback) {
        if (!mal_device_write__alsa(pDevice)) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to write initial chunk of data to the playback device.", MAL_FAILED_TO_SEND_DATA_TO_DEVICE);
        }

        // mmap mode requires an explicit start.
        if (pDevice->alsa.isUsingMMap) {
            if (((mal_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((mal_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to start capture device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
            }
        }
    } else {
        if (((mal_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((mal_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[ALSA] Failed to start capture device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__alsa(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    ((mal_snd_pcm_drop_proc)pDevice->pContext->alsa.snd_pcm_drop)((mal_snd_pcm_t*)pDevice->alsa.pPCM);
    return MAL_SUCCESS;
}

mal_result mal_device__break_main_loop__alsa(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // First we tell the main loop that we're breaking...
    pDevice->alsa.breakFromMainLoop = MAL_TRUE;

    // Then we need to force snd_pcm_wait() to return.
    //((mal_snd_pcm_drop_proc)pDevice->pContext->alsa.snd_pcm_drop)((mal_snd_pcm_t*)pDevice->alsa.pPCM);

    return MAL_SUCCESS;
}

mal_result mal_device__main_loop__alsa(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->alsa.breakFromMainLoop = MAL_FALSE;
    if (pDevice->type == mal_device_type_playback) {
        // Playback. Read from client, write to device.
        while (!pDevice->alsa.breakFromMainLoop && mal_device_write__alsa(pDevice)) {
        }
    } else {
        // Capture. Read from device, write to client.
        while (!pDevice->alsa.breakFromMainLoop && mal_device_read__alsa(pDevice)) {
        }
    }

    return MAL_SUCCESS;
}


mal_result mal_context_uninit__alsa(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_alsa);

    // Clean up memory for memory leak checkers.
    ((mal_snd_config_update_free_global_proc)pContext->alsa.snd_config_update_free_global)();

#ifndef MAL_NO_RUNTIME_LINKING
    mal_dlclose(pContext->alsa.asoundSO);
#endif

    mal_mutex_uninit(&pContext->alsa.internalDeviceEnumLock);

    return MAL_SUCCESS;
}

mal_result mal_context_init__alsa(mal_context* pContext)
{
    mal_assert(pContext != NULL);

#ifndef MAL_NO_RUNTIME_LINKING
    pContext->alsa.asoundSO = mal_dlopen("libasound.so");
    if (pContext->alsa.asoundSO == NULL) {
        return MAL_NO_BACKEND;
    }

    pContext->alsa.snd_pcm_open                           = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_open");
    pContext->alsa.snd_pcm_close                          = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_close");
    pContext->alsa.snd_pcm_hw_params_sizeof               = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_sizeof");
    pContext->alsa.snd_pcm_hw_params_any                  = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_any");
    pContext->alsa.snd_pcm_hw_params_set_format           = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_format");
    pContext->alsa.snd_pcm_hw_params_set_format_first     = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_format_first");
    pContext->alsa.snd_pcm_hw_params_get_format_mask      = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_format_mask");
    pContext->alsa.snd_pcm_hw_params_set_channels_near    = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_channels_near");
    pContext->alsa.snd_pcm_hw_params_set_rate_resample    = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_rate_resample");
    pContext->alsa.snd_pcm_hw_params_set_rate_near        = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_rate_near");
    pContext->alsa.snd_pcm_hw_params_set_buffer_size_near = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_buffer_size_near");
    pContext->alsa.snd_pcm_hw_params_set_periods_near     = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_periods_near");
    pContext->alsa.snd_pcm_hw_params_set_access           = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_access");
    pContext->alsa.snd_pcm_hw_params_get_format           = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_format");
    pContext->alsa.snd_pcm_hw_params_get_channels         = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_channels");
    pContext->alsa.snd_pcm_hw_params_get_channels_min     = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_channels_min");
    pContext->alsa.snd_pcm_hw_params_get_channels_max     = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_channels_max");
    pContext->alsa.snd_pcm_hw_params_get_rate             = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_rate");
    pContext->alsa.snd_pcm_hw_params_get_rate_min         = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_rate_min");
    pContext->alsa.snd_pcm_hw_params_get_rate_max         = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_rate_max");
    pContext->alsa.snd_pcm_hw_params_get_buffer_size      = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_buffer_size");
    pContext->alsa.snd_pcm_hw_params_get_periods          = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_periods");
    pContext->alsa.snd_pcm_hw_params_get_access           = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_access");
    pContext->alsa.snd_pcm_hw_params                      = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params");
    pContext->alsa.snd_pcm_sw_params_sizeof               = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params_sizeof");
    pContext->alsa.snd_pcm_sw_params_current              = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params_current");
    pContext->alsa.snd_pcm_sw_params_set_avail_min        = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params_set_avail_min");
    pContext->alsa.snd_pcm_sw_params_set_start_threshold  = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params_set_start_threshold");
    pContext->alsa.snd_pcm_sw_params                      = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params");
    pContext->alsa.snd_pcm_format_mask_sizeof             = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_format_mask_sizeof");
    pContext->alsa.snd_pcm_format_mask_test               = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_format_mask_test");
    pContext->alsa.snd_pcm_get_chmap                      = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_get_chmap");
    pContext->alsa.snd_pcm_prepare                        = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_prepare");
    pContext->alsa.snd_pcm_start                          = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_start");
    pContext->alsa.snd_pcm_drop                           = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_drop");
    pContext->alsa.snd_device_name_hint                   = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_device_name_hint");
    pContext->alsa.snd_device_name_get_hint               = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_device_name_get_hint");
    pContext->alsa.snd_card_get_index                     = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_card_get_index");
    pContext->alsa.snd_device_name_free_hint              = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_device_name_free_hint");
    pContext->alsa.snd_pcm_mmap_begin                     = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_mmap_begin");
    pContext->alsa.snd_pcm_mmap_commit                    = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_mmap_commit");
    pContext->alsa.snd_pcm_recover                        = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_recover");
    pContext->alsa.snd_pcm_readi                          = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_readi");
    pContext->alsa.snd_pcm_writei                         = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_writei");
    pContext->alsa.snd_pcm_avail                          = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_avail");
    pContext->alsa.snd_pcm_avail_update                   = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_avail_update");
    pContext->alsa.snd_pcm_wait                           = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_wait");
    pContext->alsa.snd_pcm_info                           = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_info");
    pContext->alsa.snd_pcm_info_sizeof                    = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_info_sizeof");
    pContext->alsa.snd_pcm_info_get_name                  = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_pcm_info_get_name");
    pContext->alsa.snd_config_update_free_global          = (mal_proc)mal_dlsym(pContext->alsa.asoundSO, "snd_config_update_free_global");
#else
    // The system below is just for type safety.
    mal_snd_pcm_open_proc                           _snd_pcm_open                           = snd_pcm_open;
    mal_snd_pcm_close_proc                          _snd_pcm_close                          = snd_pcm_close;
    mal_snd_pcm_hw_params_sizeof_proc               _snd_pcm_hw_params_sizeof               = snd_pcm_hw_params_sizeof;
    mal_snd_pcm_hw_params_any_proc                  _snd_pcm_hw_params_any                  = snd_pcm_hw_params_any;
    mal_snd_pcm_hw_params_set_format_proc           _snd_pcm_hw_params_set_format           = snd_pcm_hw_params_set_format;
    mal_snd_pcm_hw_params_set_format_first_proc     _snd_pcm_hw_params_set_format_first     = snd_pcm_hw_params_set_format_first;
    mal_snd_pcm_hw_params_get_format_mask_proc      _snd_pcm_hw_params_get_format_mask      = snd_pcm_hw_params_get_format_mask;
    mal_snd_pcm_hw_params_set_channels_near_proc    _snd_pcm_hw_params_set_channels_near    = snd_pcm_hw_params_set_channels_near;
    mal_snd_pcm_hw_params_set_rate_resample_proc    _snd_pcm_hw_params_set_rate_resample    = snd_pcm_hw_params_set_rate_resample;
    mal_snd_pcm_hw_params_set_rate_near_proc        _snd_pcm_hw_params_set_rate_near        = snd_pcm_hw_params_set_rate_near;
    mal_snd_pcm_hw_params_set_buffer_size_near_proc _snd_pcm_hw_params_set_buffer_size_near = snd_pcm_hw_params_set_buffer_size_near;
    mal_snd_pcm_hw_params_set_periods_near_proc     _snd_pcm_hw_params_set_periods_near     = snd_pcm_hw_params_set_periods_near;
    mal_snd_pcm_hw_params_set_access_proc           _snd_pcm_hw_params_set_access           = snd_pcm_hw_params_set_access;
    mal_snd_pcm_hw_params_get_format_proc           _snd_pcm_hw_params_get_format           = snd_pcm_hw_params_get_format;
    mal_snd_pcm_hw_params_get_channels_proc         _snd_pcm_hw_params_get_channels         = snd_pcm_hw_params_get_channels;
    mal_snd_pcm_hw_params_get_channels_min_proc     _snd_pcm_hw_params_get_channels_min     = snd_pcm_hw_params_get_channels_min;
    mal_snd_pcm_hw_params_get_channels_max_proc     _snd_pcm_hw_params_get_channels_max     = snd_pcm_hw_params_get_channels_max;
    mal_snd_pcm_hw_params_get_rate_proc             _snd_pcm_hw_params_get_rate             = snd_pcm_hw_params_get_rate;
    mal_snd_pcm_hw_params_get_rate_min_proc         _snd_pcm_hw_params_get_rate_min         = snd_pcm_hw_params_get_rate_min;
    mal_snd_pcm_hw_params_get_rate_max_proc         _snd_pcm_hw_params_get_rate_max         = snd_pcm_hw_params_get_rate_max;
    mal_snd_pcm_hw_params_get_buffer_size_proc      _snd_pcm_hw_params_get_buffer_size      = snd_pcm_hw_params_get_buffer_size;
    mal_snd_pcm_hw_params_get_periods_proc          _snd_pcm_hw_params_get_periods          = snd_pcm_hw_params_get_periods;
    mal_snd_pcm_hw_params_get_access_proc           _snd_pcm_hw_params_get_access           = snd_pcm_hw_params_get_access;
    mal_snd_pcm_hw_params_proc                      _snd_pcm_hw_params                      = snd_pcm_hw_params;
    mal_snd_pcm_sw_params_sizeof_proc               _snd_pcm_sw_params_sizeof               = snd_pcm_sw_params_sizeof;
    mal_snd_pcm_sw_params_current_proc              _snd_pcm_sw_params_current              = snd_pcm_sw_params_current;
    mal_snd_pcm_sw_params_set_avail_min_proc        _snd_pcm_sw_params_set_avail_min        = snd_pcm_sw_params_set_avail_min;
    mal_snd_pcm_sw_params_set_start_threshold_proc  _snd_pcm_sw_params_set_start_threshold  = snd_pcm_sw_params_set_start_threshold;
    mal_snd_pcm_sw_params_proc                      _snd_pcm_sw_params                      = snd_pcm_sw_params;
    mal_snd_pcm_format_mask_sizeof_proc             _snd_pcm_format_mask_sizeof             = snd_pcm_format_mask_sizeof;
    mal_snd_pcm_format_mask_test_proc               _snd_pcm_format_mask_test               = snd_pcm_format_mask_test;
    mal_snd_pcm_get_chmap_proc                      _snd_pcm_get_chmap                      = snd_pcm_get_chmap;
    mal_snd_pcm_prepare_proc                        _snd_pcm_prepare                        = snd_pcm_prepare;
    mal_snd_pcm_start_proc                          _snd_pcm_start                          = snd_pcm_start;
    mal_snd_pcm_drop_proc                           _snd_pcm_drop                           = snd_pcm_drop;
    mal_snd_device_name_hint_proc                   _snd_device_name_hint                   = snd_device_name_hint;
    mal_snd_device_name_get_hint_proc               _snd_device_name_get_hint               = snd_device_name_get_hint;
    mal_snd_card_get_index_proc                     _snd_card_get_index                     = snd_card_get_index;
    mal_snd_device_name_free_hint_proc              _snd_device_name_free_hint              = snd_device_name_free_hint;
    mal_snd_pcm_mmap_begin_proc                     _snd_pcm_mmap_begin                     = snd_pcm_mmap_begin;
    mal_snd_pcm_mmap_commit_proc                    _snd_pcm_mmap_commit                    = snd_pcm_mmap_commit;
    mal_snd_pcm_recover_proc                        _snd_pcm_recover                        = snd_pcm_recover;
    mal_snd_pcm_readi_proc                          _snd_pcm_readi                          = snd_pcm_readi;
    mal_snd_pcm_writei_proc                         _snd_pcm_writei                         = snd_pcm_writei;
    mal_snd_pcm_avail_proc                          _snd_pcm_avail                          = snd_pcm_avail;
    mal_snd_pcm_avail_update_proc                   _snd_pcm_avail_update                   = snd_pcm_avail_update;
    mal_snd_pcm_wait_proc                           _snd_pcm_wait                           = snd_pcm_wait;
    mal_snd_pcm_info_proc                           _snd_pcm_info                           = snd_pcm_info;
    mal_snd_pcm_info_sizeof_proc                    _snd_pcm_info_sizeof                    = snd_pcm_info_sizeof;
    mal_snd_pcm_info_get_name_proc                  _snd_pcm_info_get_name                  = snd_pcm_info_get_name;
    mal_snd_config_update_free_global_proc          _snd_config_update_free_global          = snd_config_update_free_global;

    pContext->alsa.snd_pcm_open                           = (mal_proc)_snd_pcm_open;
    pContext->alsa.snd_pcm_close                          = (mal_proc)_snd_pcm_close;
    pContext->alsa.snd_pcm_hw_params_sizeof               = (mal_proc)_snd_pcm_hw_params_sizeof;
    pContext->alsa.snd_pcm_hw_params_any                  = (mal_proc)_snd_pcm_hw_params_any;
    pContext->alsa.snd_pcm_hw_params_set_format           = (mal_proc)_snd_pcm_hw_params_set_format;
    pContext->alsa.snd_pcm_hw_params_set_format_first     = (mal_proc)_snd_pcm_hw_params_set_format_first;
    pContext->alsa.snd_pcm_hw_params_get_format_mask      = (mal_proc)_snd_pcm_hw_params_get_format_mask;
    pContext->alsa.snd_pcm_hw_params_set_channels_near    = (mal_proc)_snd_pcm_hw_params_set_channels_near;
    pContext->alsa.snd_pcm_hw_params_set_rate_resample    = (mal_proc)_snd_pcm_hw_params_set_rate_resample;
    pContext->alsa.snd_pcm_hw_params_set_rate_near        = (mal_proc)_snd_pcm_hw_params_set_rate_near;
    pContext->alsa.snd_pcm_hw_params_set_buffer_size_near = (mal_proc)_snd_pcm_hw_params_set_buffer_size_near;
    pContext->alsa.snd_pcm_hw_params_set_periods_near     = (mal_proc)_snd_pcm_hw_params_set_periods_near;
    pContext->alsa.snd_pcm_hw_params_set_access           = (mal_proc)_snd_pcm_hw_params_set_access;
    pContext->alsa.snd_pcm_hw_params_get_format           = (mal_proc)_snd_pcm_hw_params_get_format;
    pContext->alsa.snd_pcm_hw_params_get_channels         = (mal_proc)_snd_pcm_hw_params_get_channels;
    pContext->alsa.snd_pcm_hw_params_get_channels_min     = (mal_proc)_snd_pcm_hw_params_get_channels_min;
    pContext->alsa.snd_pcm_hw_params_get_channels_max     = (mal_proc)_snd_pcm_hw_params_get_channels_max;
    pContext->alsa.snd_pcm_hw_params_get_rate             = (mal_proc)_snd_pcm_hw_params_get_rate;
    pContext->alsa.snd_pcm_hw_params_get_buffer_size      = (mal_proc)_snd_pcm_hw_params_get_buffer_size;
    pContext->alsa.snd_pcm_hw_params_get_periods          = (mal_proc)_snd_pcm_hw_params_get_periods;
    pContext->alsa.snd_pcm_hw_params_get_access           = (mal_proc)_snd_pcm_hw_params_get_access;
    pContext->alsa.snd_pcm_hw_params                      = (mal_proc)_snd_pcm_hw_params;
    pContext->alsa.snd_pcm_sw_params_sizeof               = (mal_proc)_snd_pcm_sw_params_sizeof;
    pContext->alsa.snd_pcm_sw_params_current              = (mal_proc)_snd_pcm_sw_params_current;
    pContext->alsa.snd_pcm_sw_params_set_avail_min        = (mal_proc)_snd_pcm_sw_params_set_avail_min;
    pContext->alsa.snd_pcm_sw_params_set_start_threshold  = (mal_proc)_snd_pcm_sw_params_set_start_threshold;
    pContext->alsa.snd_pcm_sw_params                      = (mal_proc)_snd_pcm_sw_params;
    pContext->alsa.snd_pcm_format_mask_sizeof             = (mal_proc)_snd_pcm_format_mask_sizeof;
    pContext->alsa.snd_pcm_format_mask_test               = (mal_proc)_snd_pcm_format_mask_test;
    pContext->alsa.snd_pcm_get_chmap                      = (mal_proc)_snd_pcm_get_chmap;
    pContext->alsa.snd_pcm_prepare                        = (mal_proc)_snd_pcm_prepare;
    pContext->alsa.snd_pcm_start                          = (mal_proc)_snd_pcm_start;
    pContext->alsa.snd_pcm_drop                           = (mal_proc)_snd_pcm_drop;
    pContext->alsa.snd_device_name_hint                   = (mal_proc)_snd_device_name_hint;
    pContext->alsa.snd_device_name_get_hint               = (mal_proc)_snd_device_name_get_hint;
    pContext->alsa.snd_card_get_index                     = (mal_proc)_snd_card_get_index;
    pContext->alsa.snd_device_name_free_hint              = (mal_proc)_snd_device_name_free_hint;
    pContext->alsa.snd_pcm_mmap_begin                     = (mal_proc)_snd_pcm_mmap_begin;
    pContext->alsa.snd_pcm_mmap_commit                    = (mal_proc)_snd_pcm_mmap_commit;
    pContext->alsa.snd_pcm_recover                        = (mal_proc)_snd_pcm_recover;
    pContext->alsa.snd_pcm_readi                          = (mal_proc)_snd_pcm_readi;
    pContext->alsa.snd_pcm_writei                         = (mal_proc)_snd_pcm_writei;
    pContext->alsa.snd_pcm_avail                          = (mal_proc)_snd_pcm_avail;
    pContext->alsa.snd_pcm_avail_update                   = (mal_proc)_snd_pcm_avail_update;
    pContext->alsa.snd_pcm_wait                           = (mal_proc)_snd_pcm_wait;
    pContext->alsa.snd_pcm_info                           = (mal_proc)_snd_pcm_info;
    pContext->alsa.snd_pcm_info_sizeof                    = (mal_proc)_snd_pcm_info_sizeof;
    pContext->alsa.snd_pcm_info_get_name                  = (mal_proc)_snd_pcm_info_get_name;
    pContext->alsa.snd_config_update_free_global          = (mal_proc)_snd_config_update_free_global;
#endif

    if (mal_mutex_init(pContext, &pContext->alsa.internalDeviceEnumLock) != MAL_SUCCESS) {
        mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[ALSA] WARNING: Failed to initialize mutex for internal device enumeration.", MAL_ERROR);
    }

    pContext->onUninit              = mal_context_uninit__alsa;
    pContext->onDeviceIDEqual       = mal_context_is_device_id_equal__alsa;
    pContext->onEnumDevices         = mal_context_enumerate_devices__alsa;
    pContext->onGetDeviceInfo       = mal_context_get_device_info__alsa;
    pContext->onDeviceInit          = mal_device_init__alsa;
    pContext->onDeviceUninit        = mal_device_uninit__alsa;
    pContext->onDeviceStart         = mal_device__start_backend__alsa;
    pContext->onDeviceStop          = mal_device__stop_backend__alsa;
    pContext->onDeviceBreakMainLoop = mal_device__break_main_loop__alsa;
    pContext->onDeviceMainLoop      = mal_device__main_loop__alsa;

    return MAL_SUCCESS;
}
#endif  // ALSA



///////////////////////////////////////////////////////////////////////////////
//
// PulseAudio Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_PULSEAUDIO

// It is assumed pulseaudio.h is available when compile-time linking is being used. We use this for type safety when using
// compile time linking (we don't have this luxury when using runtime linking without headers).
//
// When using compile time linking, each of our mal_* equivalents should use the sames types as defined by the header. The
// reason for this is that it allow us to take advantage of proper type safety.
#ifdef MAL_NO_RUNTIME_LINKING
#include <pulse/pulseaudio.h>

#define MAL_PA_OK                                       PA_OK
#define MAL_PA_ERR_ACCESS                               PA_ERR_ACCESS
#define MAL_PA_ERR_INVALID                              PA_ERR_INVALID
#define MAL_PA_ERR_NOENTITY                             PA_ERR_NOENTITY

#define MAL_PA_CHANNELS_MAX                             PA_CHANNELS_MAX
#define MAL_PA_RATE_MAX                                 PA_RATE_MAX

typedef pa_context_flags_t mal_pa_context_flags_t;
#define MAL_PA_CONTEXT_NOFLAGS                          PA_CONTEXT_NOFLAGS
#define MAL_PA_CONTEXT_NOAUTOSPAWN                      PA_CONTEXT_NOAUTOSPAWN
#define MAL_PA_CONTEXT_NOFAIL                           PA_CONTEXT_NOFAIL

typedef pa_stream_flags_t mal_pa_stream_flags_t;
#define MAL_PA_STREAM_NOFLAGS                           PA_STREAM_NOFLAGS
#define MAL_PA_STREAM_START_CORKED                      PA_STREAM_START_CORKED
#define MAL_PA_STREAM_INTERPOLATE_TIMING                PA_STREAM_INTERPOLATE_TIMING
#define MAL_PA_STREAM_NOT_MONOTONIC                     PA_STREAM_NOT_MONOTONIC
#define MAL_PA_STREAM_AUTO_TIMING_UPDATE                PA_STREAM_AUTO_TIMING_UPDATE
#define MAL_PA_STREAM_NO_REMAP_CHANNELS                 PA_STREAM_NO_REMAP_CHANNELS
#define MAL_PA_STREAM_NO_REMIX_CHANNELS                 PA_STREAM_NO_REMIX_CHANNELS
#define MAL_PA_STREAM_FIX_FORMAT                        PA_STREAM_FIX_FORMAT
#define MAL_PA_STREAM_FIX_RATE                          PA_STREAM_FIX_RATE
#define MAL_PA_STREAM_FIX_CHANNELS                      PA_STREAM_FIX_CHANNELS
#define MAL_PA_STREAM_DONT_MOVE                         PA_STREAM_DONT_MOVE
#define MAL_PA_STREAM_VARIABLE_RATE                     PA_STREAM_VARIABLE_RATE
#define MAL_PA_STREAM_PEAK_DETECT                       PA_STREAM_PEAK_DETECT
#define MAL_PA_STREAM_START_MUTED                       PA_STREAM_START_MUTED
#define MAL_PA_STREAM_ADJUST_LATENCY                    PA_STREAM_ADJUST_LATENCY
#define MAL_PA_STREAM_EARLY_REQUESTS                    PA_STREAM_EARLY_REQUESTS
#define MAL_PA_STREAM_DONT_INHIBIT_AUTO_SUSPEND         PA_STREAM_DONT_INHIBIT_AUTO_SUSPEND
#define MAL_PA_STREAM_START_UNMUTED                     PA_STREAM_START_UNMUTED
#define MAL_PA_STREAM_FAIL_ON_SUSPEND                   PA_STREAM_FAIL_ON_SUSPEND
#define MAL_PA_STREAM_RELATIVE_VOLUME                   PA_STREAM_RELATIVE_VOLUME
#define MAL_PA_STREAM_PASSTHROUGH                       PA_STREAM_PASSTHROUGH

typedef pa_sink_flags_t mal_pa_sink_flags_t;
#define MAL_PA_SINK_NOFLAGS                             PA_SINK_NOFLAGS
#define MAL_PA_SINK_HW_VOLUME_CTRL                      PA_SINK_HW_VOLUME_CTRL
#define MAL_PA_SINK_LATENCY                             PA_SINK_LATENCY
#define MAL_PA_SINK_HARDWARE                            PA_SINK_HARDWARE
#define MAL_PA_SINK_NETWORK                             PA_SINK_NETWORK
#define MAL_PA_SINK_HW_MUTE_CTRL                        PA_SINK_HW_MUTE_CTRL
#define MAL_PA_SINK_DECIBEL_VOLUME                      PA_SINK_DECIBEL_VOLUME
#define MAL_PA_SINK_FLAT_VOLUME                         PA_SINK_FLAT_VOLUME
#define MAL_PA_SINK_DYNAMIC_LATENCY                     PA_SINK_DYNAMIC_LATENCY
#define MAL_PA_SINK_SET_FORMATS                         PA_SINK_SET_FORMATS

typedef pa_source_flags_t mal_pa_source_flags_t;
#define MAL_PA_SOURCE_NOFLAGS                           PA_SOURCE_NOFLAGS
#define MAL_PA_SOURCE_HW_VOLUME_CTRL                    PA_SOURCE_HW_VOLUME_CTRL
#define MAL_PA_SOURCE_LATENCY                           PA_SOURCE_LATENCY
#define MAL_PA_SOURCE_HARDWARE                          PA_SOURCE_HARDWARE
#define MAL_PA_SOURCE_NETWORK                           PA_SOURCE_NETWORK
#define MAL_PA_SOURCE_HW_MUTE_CTRL                      PA_SOURCE_HW_MUTE_CTRL
#define MAL_PA_SOURCE_DECIBEL_VOLUME                    PA_SOURCE_DECIBEL_VOLUME
#define MAL_PA_SOURCE_DYNAMIC_LATENCY                   PA_SOURCE_DYNAMIC_LATENCY
#define MAL_PA_SOURCE_FLAT_VOLUME                       PA_SOURCE_FLAT_VOLUME

typedef pa_context_state_t mal_pa_context_state_t;
#define MAL_PA_CONTEXT_UNCONNECTED                      PA_CONTEXT_UNCONNECTED
#define MAL_PA_CONTEXT_CONNECTING                       PA_CONTEXT_CONNECTING
#define MAL_PA_CONTEXT_AUTHORIZING                      PA_CONTEXT_AUTHORIZING
#define MAL_PA_CONTEXT_SETTING_NAME                     PA_CONTEXT_SETTING_NAME
#define MAL_PA_CONTEXT_READY                            PA_CONTEXT_READY
#define MAL_PA_CONTEXT_FAILED                           PA_CONTEXT_FAILED
#define MAL_PA_CONTEXT_TERMINATED                       PA_CONTEXT_TERMINATED

typedef pa_stream_state_t mal_pa_stream_state_t;
#define MAL_PA_STREAM_UNCONNECTED                       PA_STREAM_UNCONNECTED
#define MAL_PA_STREAM_CREATING                          PA_STREAM_CREATING
#define MAL_PA_STREAM_READY                             PA_STREAM_READY
#define MAL_PA_STREAM_FAILED                            PA_STREAM_FAILED
#define MAL_PA_STREAM_TERMINATED                        PA_STREAM_TERMINATED

typedef pa_operation_state_t mal_pa_operation_state_t;
#define MAL_PA_OPERATION_RUNNING                        PA_OPERATION_RUNNING
#define MAL_PA_OPERATION_DONE                           PA_OPERATION_DONE
#define MAL_PA_OPERATION_CANCELLED                      PA_OPERATION_CANCELLED

typedef pa_sink_state_t mal_pa_sink_state_t;
#define MAL_PA_SINK_INVALID_STATE                       PA_SINK_INVALID_STATE
#define MAL_PA_SINK_RUNNING                             PA_SINK_RUNNING
#define MAL_PA_SINK_IDLE                                PA_SINK_IDLE
#define MAL_PA_SINK_SUSPENDED                           PA_SINK_SUSPENDED

typedef pa_source_state_t mal_pa_source_state_t;
#define MAL_PA_SOURCE_INVALID_STATE                     PA_SOURCE_INVALID_STATE
#define MAL_PA_SOURCE_RUNNING                           PA_SOURCE_RUNNING
#define MAL_PA_SOURCE_IDLE                              PA_SOURCE_IDLE
#define MAL_PA_SOURCE_SUSPENDED                         PA_SOURCE_SUSPENDED

typedef pa_seek_mode_t mal_pa_seek_mode_t;
#define MAL_PA_SEEK_RELATIVE                            PA_SEEK_RELATIVE
#define MAL_PA_SEEK_ABSOLUTE                            PA_SEEK_ABSOLUTE
#define MAL_PA_SEEK_RELATIVE_ON_READ                    PA_SEEK_RELATIVE_ON_READ
#define MAL_PA_SEEK_RELATIVE_END                        PA_SEEK_RELATIVE_END

typedef pa_channel_position_t mal_pa_channel_position_t;
#define MAL_PA_CHANNEL_POSITION_INVALID                 PA_CHANNEL_POSITION_INVALID
#define MAL_PA_CHANNEL_POSITION_MONO                    PA_CHANNEL_POSITION_MONO
#define MAL_PA_CHANNEL_POSITION_FRONT_LEFT              PA_CHANNEL_POSITION_FRONT_LEFT
#define MAL_PA_CHANNEL_POSITION_FRONT_RIGHT             PA_CHANNEL_POSITION_FRONT_RIGHT
#define MAL_PA_CHANNEL_POSITION_FRONT_CENTER            PA_CHANNEL_POSITION_FRONT_CENTER
#define MAL_PA_CHANNEL_POSITION_REAR_CENTER             PA_CHANNEL_POSITION_REAR_CENTER
#define MAL_PA_CHANNEL_POSITION_REAR_LEFT               PA_CHANNEL_POSITION_REAR_LEFT
#define MAL_PA_CHANNEL_POSITION_REAR_RIGHT              PA_CHANNEL_POSITION_REAR_RIGHT
#define MAL_PA_CHANNEL_POSITION_LFE                     PA_CHANNEL_POSITION_LFE
#define MAL_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER    PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER
#define MAL_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER   PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER
#define MAL_PA_CHANNEL_POSITION_SIDE_LEFT               PA_CHANNEL_POSITION_SIDE_LEFT
#define MAL_PA_CHANNEL_POSITION_SIDE_RIGHT              PA_CHANNEL_POSITION_SIDE_RIGHT
#define MAL_PA_CHANNEL_POSITION_AUX0                    PA_CHANNEL_POSITION_AUX0
#define MAL_PA_CHANNEL_POSITION_AUX1                    PA_CHANNEL_POSITION_AUX1
#define MAL_PA_CHANNEL_POSITION_AUX2                    PA_CHANNEL_POSITION_AUX2
#define MAL_PA_CHANNEL_POSITION_AUX3                    PA_CHANNEL_POSITION_AUX3
#define MAL_PA_CHANNEL_POSITION_AUX4                    PA_CHANNEL_POSITION_AUX4
#define MAL_PA_CHANNEL_POSITION_AUX5                    PA_CHANNEL_POSITION_AUX5
#define MAL_PA_CHANNEL_POSITION_AUX6                    PA_CHANNEL_POSITION_AUX6
#define MAL_PA_CHANNEL_POSITION_AUX7                    PA_CHANNEL_POSITION_AUX7
#define MAL_PA_CHANNEL_POSITION_AUX8                    PA_CHANNEL_POSITION_AUX8
#define MAL_PA_CHANNEL_POSITION_AUX9                    PA_CHANNEL_POSITION_AUX9
#define MAL_PA_CHANNEL_POSITION_AUX10                   PA_CHANNEL_POSITION_AUX10
#define MAL_PA_CHANNEL_POSITION_AUX11                   PA_CHANNEL_POSITION_AUX11
#define MAL_PA_CHANNEL_POSITION_AUX12                   PA_CHANNEL_POSITION_AUX12
#define MAL_PA_CHANNEL_POSITION_AUX13                   PA_CHANNEL_POSITION_AUX13
#define MAL_PA_CHANNEL_POSITION_AUX14                   PA_CHANNEL_POSITION_AUX14
#define MAL_PA_CHANNEL_POSITION_AUX15                   PA_CHANNEL_POSITION_AUX15
#define MAL_PA_CHANNEL_POSITION_AUX16                   PA_CHANNEL_POSITION_AUX16
#define MAL_PA_CHANNEL_POSITION_AUX17                   PA_CHANNEL_POSITION_AUX17
#define MAL_PA_CHANNEL_POSITION_AUX18                   PA_CHANNEL_POSITION_AUX18
#define MAL_PA_CHANNEL_POSITION_AUX19                   PA_CHANNEL_POSITION_AUX19
#define MAL_PA_CHANNEL_POSITION_AUX20                   PA_CHANNEL_POSITION_AUX20
#define MAL_PA_CHANNEL_POSITION_AUX21                   PA_CHANNEL_POSITION_AUX21
#define MAL_PA_CHANNEL_POSITION_AUX22                   PA_CHANNEL_POSITION_AUX22
#define MAL_PA_CHANNEL_POSITION_AUX23                   PA_CHANNEL_POSITION_AUX23
#define MAL_PA_CHANNEL_POSITION_AUX24                   PA_CHANNEL_POSITION_AUX24
#define MAL_PA_CHANNEL_POSITION_AUX25                   PA_CHANNEL_POSITION_AUX25
#define MAL_PA_CHANNEL_POSITION_AUX26                   PA_CHANNEL_POSITION_AUX26
#define MAL_PA_CHANNEL_POSITION_AUX27                   PA_CHANNEL_POSITION_AUX27
#define MAL_PA_CHANNEL_POSITION_AUX28                   PA_CHANNEL_POSITION_AUX28
#define MAL_PA_CHANNEL_POSITION_AUX29                   PA_CHANNEL_POSITION_AUX29
#define MAL_PA_CHANNEL_POSITION_AUX30                   PA_CHANNEL_POSITION_AUX30
#define MAL_PA_CHANNEL_POSITION_AUX31                   PA_CHANNEL_POSITION_AUX31
#define MAL_PA_CHANNEL_POSITION_TOP_CENTER              PA_CHANNEL_POSITION_TOP_CENTER
#define MAL_PA_CHANNEL_POSITION_TOP_FRONT_LEFT          PA_CHANNEL_POSITION_TOP_FRONT_LEFT
#define MAL_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT         PA_CHANNEL_POSITION_TOP_FRONT_RIGHT
#define MAL_PA_CHANNEL_POSITION_TOP_FRONT_CENTER        PA_CHANNEL_POSITION_TOP_FRONT_CENTER
#define MAL_PA_CHANNEL_POSITION_TOP_REAR_LEFT           PA_CHANNEL_POSITION_TOP_REAR_LEFT
#define MAL_PA_CHANNEL_POSITION_TOP_REAR_RIGHT          PA_CHANNEL_POSITION_TOP_REAR_RIGHT
#define MAL_PA_CHANNEL_POSITION_TOP_REAR_CENTER         PA_CHANNEL_POSITION_TOP_REAR_CENTER
#define MAL_PA_CHANNEL_POSITION_LEFT                    PA_CHANNEL_POSITION_LEFT
#define MAL_PA_CHANNEL_POSITION_RIGHT                   PA_CHANNEL_POSITION_RIGHT
#define MAL_PA_CHANNEL_POSITION_CENTER                  PA_CHANNEL_POSITION_CENTER
#define MAL_PA_CHANNEL_POSITION_SUBWOOFER               PA_CHANNEL_POSITION_SUBWOOFER

typedef pa_channel_map_def_t mal_pa_channel_map_def_t;
#define MAL_PA_CHANNEL_MAP_AIFF                         PA_CHANNEL_MAP_AIFF
#define MAL_PA_CHANNEL_MAP_ALSA                         PA_CHANNEL_MAP_ALSA
#define MAL_PA_CHANNEL_MAP_AUX                          PA_CHANNEL_MAP_AUX
#define MAL_PA_CHANNEL_MAP_WAVEEX                       PA_CHANNEL_MAP_WAVEEX
#define MAL_PA_CHANNEL_MAP_OSS                          PA_CHANNEL_MAP_OSS
#define MAL_PA_CHANNEL_MAP_DEFAULT                      PA_CHANNEL_MAP_DEFAULT

typedef pa_sample_format_t mal_pa_sample_format_t;
#define MAL_PA_SAMPLE_INVALID                           PA_SAMPLE_INVALID
#define MAL_PA_SAMPLE_U8                                PA_SAMPLE_U8
#define MAL_PA_SAMPLE_ALAW                              PA_SAMPLE_ALAW
#define MAL_PA_SAMPLE_ULAW                              PA_SAMPLE_ULAW
#define MAL_PA_SAMPLE_S16LE                             PA_SAMPLE_S16LE
#define MAL_PA_SAMPLE_S16BE                             PA_SAMPLE_S16BE
#define MAL_PA_SAMPLE_FLOAT32LE                         PA_SAMPLE_FLOAT32LE
#define MAL_PA_SAMPLE_FLOAT32BE                         PA_SAMPLE_FLOAT32BE
#define MAL_PA_SAMPLE_S32LE                             PA_SAMPLE_S32LE
#define MAL_PA_SAMPLE_S32BE                             PA_SAMPLE_S32BE
#define MAL_PA_SAMPLE_S24LE                             PA_SAMPLE_S24LE
#define MAL_PA_SAMPLE_S24BE                             PA_SAMPLE_S24BE
#define MAL_PA_SAMPLE_S24_32LE                          PA_SAMPLE_S24_32LE
#define MAL_PA_SAMPLE_S24_32BE                          PA_SAMPLE_S24_32BE

typedef pa_mainloop     mal_pa_mainloop;
typedef pa_mainloop_api mal_pa_mainloop_api;
typedef pa_context      mal_pa_context;
typedef pa_operation    mal_pa_operation;
typedef pa_stream       mal_pa_stream;
typedef pa_spawn_api    mal_pa_spawn_api;
typedef pa_buffer_attr  mal_pa_buffer_attr;
typedef pa_channel_map  mal_pa_channel_map;
typedef pa_cvolume      mal_pa_cvolume;
typedef pa_sample_spec  mal_pa_sample_spec;
typedef pa_sink_info    mal_pa_sink_info;
typedef pa_source_info  mal_pa_source_info;

typedef pa_context_notify_cb_t mal_pa_context_notify_cb_t;
typedef pa_sink_info_cb_t mal_pa_sink_info_cb_t;
typedef pa_source_info_cb_t mal_pa_source_info_cb_t;
typedef pa_stream_success_cb_t mal_pa_stream_success_cb_t;
typedef pa_stream_request_cb_t mal_pa_stream_request_cb_t;
typedef pa_free_cb_t mal_pa_free_cb_t;
#else
#define MAL_PA_OK                                       0
#define MAL_PA_ERR_ACCESS                               1
#define MAL_PA_ERR_INVALID                              2
#define MAL_PA_ERR_NOENTITY                             5

#define MAL_PA_CHANNELS_MAX                             32
#define MAL_PA_RATE_MAX                                 384000

typedef int mal_pa_context_flags_t;
#define MAL_PA_CONTEXT_NOFLAGS                          0x00000000
#define MAL_PA_CONTEXT_NOAUTOSPAWN                      0x00000001
#define MAL_PA_CONTEXT_NOFAIL                           0x00000002

typedef int mal_pa_stream_flags_t;
#define MAL_PA_STREAM_NOFLAGS                           0x00000000
#define MAL_PA_STREAM_START_CORKED                      0x00000001
#define MAL_PA_STREAM_INTERPOLATE_TIMING                0x00000002
#define MAL_PA_STREAM_NOT_MONOTONIC                     0x00000004
#define MAL_PA_STREAM_AUTO_TIMING_UPDATE                0x00000008
#define MAL_PA_STREAM_NO_REMAP_CHANNELS                 0x00000010
#define MAL_PA_STREAM_NO_REMIX_CHANNELS                 0x00000020
#define MAL_PA_STREAM_FIX_FORMAT                        0x00000040
#define MAL_PA_STREAM_FIX_RATE                          0x00000080
#define MAL_PA_STREAM_FIX_CHANNELS                      0x00000100
#define MAL_PA_STREAM_DONT_MOVE                         0x00000200
#define MAL_PA_STREAM_VARIABLE_RATE                     0x00000400
#define MAL_PA_STREAM_PEAK_DETECT                       0x00000800
#define MAL_PA_STREAM_START_MUTED                       0x00001000
#define MAL_PA_STREAM_ADJUST_LATENCY                    0x00002000
#define MAL_PA_STREAM_EARLY_REQUESTS                    0x00004000
#define MAL_PA_STREAM_DONT_INHIBIT_AUTO_SUSPEND         0x00008000
#define MAL_PA_STREAM_START_UNMUTED                     0x00010000
#define MAL_PA_STREAM_FAIL_ON_SUSPEND                   0x00020000
#define MAL_PA_STREAM_RELATIVE_VOLUME                   0x00040000
#define MAL_PA_STREAM_PASSTHROUGH                       0x00080000

typedef int mal_pa_sink_flags_t;
#define MAL_PA_SINK_NOFLAGS                             0x00000000
#define MAL_PA_SINK_HW_VOLUME_CTRL                      0x00000001
#define MAL_PA_SINK_LATENCY                             0x00000002
#define MAL_PA_SINK_HARDWARE                            0x00000004
#define MAL_PA_SINK_NETWORK                             0x00000008
#define MAL_PA_SINK_HW_MUTE_CTRL                        0x00000010
#define MAL_PA_SINK_DECIBEL_VOLUME                      0x00000020
#define MAL_PA_SINK_FLAT_VOLUME                         0x00000040
#define MAL_PA_SINK_DYNAMIC_LATENCY                     0x00000080
#define MAL_PA_SINK_SET_FORMATS                         0x00000100

typedef int mal_pa_source_flags_t;
#define MAL_PA_SOURCE_NOFLAGS                           0x00000000
#define MAL_PA_SOURCE_HW_VOLUME_CTRL                    0x00000001
#define MAL_PA_SOURCE_LATENCY                           0x00000002
#define MAL_PA_SOURCE_HARDWARE                          0x00000004
#define MAL_PA_SOURCE_NETWORK                           0x00000008
#define MAL_PA_SOURCE_HW_MUTE_CTRL                      0x00000010
#define MAL_PA_SOURCE_DECIBEL_VOLUME                    0x00000020
#define MAL_PA_SOURCE_DYNAMIC_LATENCY                   0x00000040
#define MAL_PA_SOURCE_FLAT_VOLUME                       0x00000080

typedef int mal_pa_context_state_t;
#define MAL_PA_CONTEXT_UNCONNECTED                      0
#define MAL_PA_CONTEXT_CONNECTING                       1
#define MAL_PA_CONTEXT_AUTHORIZING                      2
#define MAL_PA_CONTEXT_SETTING_NAME                     3
#define MAL_PA_CONTEXT_READY                            4
#define MAL_PA_CONTEXT_FAILED                           5
#define MAL_PA_CONTEXT_TERMINATED                       6

typedef int mal_pa_stream_state_t;
#define MAL_PA_STREAM_UNCONNECTED                       0
#define MAL_PA_STREAM_CREATING                          1
#define MAL_PA_STREAM_READY                             2
#define MAL_PA_STREAM_FAILED                            3
#define MAL_PA_STREAM_TERMINATED                        4

typedef int mal_pa_operation_state_t;
#define MAL_PA_OPERATION_RUNNING                        0
#define MAL_PA_OPERATION_DONE                           1
#define MAL_PA_OPERATION_CANCELLED                      2

typedef int mal_pa_sink_state_t;
#define MAL_PA_SINK_INVALID_STATE                       -1
#define MAL_PA_SINK_RUNNING                             0
#define MAL_PA_SINK_IDLE                                1
#define MAL_PA_SINK_SUSPENDED                           2

typedef int mal_pa_source_state_t;
#define MAL_PA_SOURCE_INVALID_STATE                     -1
#define MAL_PA_SOURCE_RUNNING                           0
#define MAL_PA_SOURCE_IDLE                              1
#define MAL_PA_SOURCE_SUSPENDED                         2

typedef int mal_pa_seek_mode_t;
#define MAL_PA_SEEK_RELATIVE                            0
#define MAL_PA_SEEK_ABSOLUTE                            1
#define MAL_PA_SEEK_RELATIVE_ON_READ                    2
#define MAL_PA_SEEK_RELATIVE_END                        3

typedef int mal_pa_channel_position_t;
#define MAL_PA_CHANNEL_POSITION_INVALID                 -1
#define MAL_PA_CHANNEL_POSITION_MONO                    0
#define MAL_PA_CHANNEL_POSITION_FRONT_LEFT              1
#define MAL_PA_CHANNEL_POSITION_FRONT_RIGHT             2
#define MAL_PA_CHANNEL_POSITION_FRONT_CENTER            3
#define MAL_PA_CHANNEL_POSITION_REAR_CENTER             4
#define MAL_PA_CHANNEL_POSITION_REAR_LEFT               5
#define MAL_PA_CHANNEL_POSITION_REAR_RIGHT              6
#define MAL_PA_CHANNEL_POSITION_LFE                     7
#define MAL_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER    8
#define MAL_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER   9
#define MAL_PA_CHANNEL_POSITION_SIDE_LEFT               10
#define MAL_PA_CHANNEL_POSITION_SIDE_RIGHT              11
#define MAL_PA_CHANNEL_POSITION_AUX0                    12
#define MAL_PA_CHANNEL_POSITION_AUX1                    13
#define MAL_PA_CHANNEL_POSITION_AUX2                    14
#define MAL_PA_CHANNEL_POSITION_AUX3                    15
#define MAL_PA_CHANNEL_POSITION_AUX4                    16
#define MAL_PA_CHANNEL_POSITION_AUX5                    17
#define MAL_PA_CHANNEL_POSITION_AUX6                    18
#define MAL_PA_CHANNEL_POSITION_AUX7                    19
#define MAL_PA_CHANNEL_POSITION_AUX8                    20
#define MAL_PA_CHANNEL_POSITION_AUX9                    21
#define MAL_PA_CHANNEL_POSITION_AUX10                   22
#define MAL_PA_CHANNEL_POSITION_AUX11                   23
#define MAL_PA_CHANNEL_POSITION_AUX12                   24
#define MAL_PA_CHANNEL_POSITION_AUX13                   25
#define MAL_PA_CHANNEL_POSITION_AUX14                   26
#define MAL_PA_CHANNEL_POSITION_AUX15                   27
#define MAL_PA_CHANNEL_POSITION_AUX16                   28
#define MAL_PA_CHANNEL_POSITION_AUX17                   29
#define MAL_PA_CHANNEL_POSITION_AUX18                   30
#define MAL_PA_CHANNEL_POSITION_AUX19                   31
#define MAL_PA_CHANNEL_POSITION_AUX20                   32
#define MAL_PA_CHANNEL_POSITION_AUX21                   33
#define MAL_PA_CHANNEL_POSITION_AUX22                   34
#define MAL_PA_CHANNEL_POSITION_AUX23                   35
#define MAL_PA_CHANNEL_POSITION_AUX24                   36
#define MAL_PA_CHANNEL_POSITION_AUX25                   37
#define MAL_PA_CHANNEL_POSITION_AUX26                   38
#define MAL_PA_CHANNEL_POSITION_AUX27                   39
#define MAL_PA_CHANNEL_POSITION_AUX28                   40
#define MAL_PA_CHANNEL_POSITION_AUX29                   41
#define MAL_PA_CHANNEL_POSITION_AUX30                   42
#define MAL_PA_CHANNEL_POSITION_AUX31                   43
#define MAL_PA_CHANNEL_POSITION_TOP_CENTER              44
#define MAL_PA_CHANNEL_POSITION_TOP_FRONT_LEFT          45
#define MAL_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT         46
#define MAL_PA_CHANNEL_POSITION_TOP_FRONT_CENTER        47
#define MAL_PA_CHANNEL_POSITION_TOP_REAR_LEFT           48
#define MAL_PA_CHANNEL_POSITION_TOP_REAR_RIGHT          49
#define MAL_PA_CHANNEL_POSITION_TOP_REAR_CENTER         50
#define MAL_PA_CHANNEL_POSITION_LEFT                    MAL_PA_CHANNEL_POSITION_FRONT_LEFT
#define MAL_PA_CHANNEL_POSITION_RIGHT                   MAL_PA_CHANNEL_POSITION_FRONT_RIGHT
#define MAL_PA_CHANNEL_POSITION_CENTER                  MAL_PA_CHANNEL_POSITION_FRONT_CENTER
#define MAL_PA_CHANNEL_POSITION_SUBWOOFER               MAL_PA_CHANNEL_POSITION_LFE

typedef int mal_pa_channel_map_def_t;
#define MAL_PA_CHANNEL_MAP_AIFF                         0
#define MAL_PA_CHANNEL_MAP_ALSA                         1
#define MAL_PA_CHANNEL_MAP_AUX                          2
#define MAL_PA_CHANNEL_MAP_WAVEEX                       3
#define MAL_PA_CHANNEL_MAP_OSS                          4
#define MAL_PA_CHANNEL_MAP_DEFAULT                      MAL_PA_CHANNEL_MAP_AIFF

typedef int mal_pa_sample_format_t;
#define MAL_PA_SAMPLE_INVALID                           -1
#define MAL_PA_SAMPLE_U8                                0
#define MAL_PA_SAMPLE_ALAW                              1
#define MAL_PA_SAMPLE_ULAW                              2
#define MAL_PA_SAMPLE_S16LE                             3
#define MAL_PA_SAMPLE_S16BE                             4
#define MAL_PA_SAMPLE_FLOAT32LE                         5
#define MAL_PA_SAMPLE_FLOAT32BE                         6
#define MAL_PA_SAMPLE_S32LE                             7
#define MAL_PA_SAMPLE_S32BE                             8
#define MAL_PA_SAMPLE_S24LE                             9
#define MAL_PA_SAMPLE_S24BE                             10
#define MAL_PA_SAMPLE_S24_32LE                          11
#define MAL_PA_SAMPLE_S24_32BE                          12

typedef struct mal_pa_mainloop     mal_pa_mainloop;
typedef struct mal_pa_mainloop_api mal_pa_mainloop_api;
typedef struct mal_pa_context      mal_pa_context;
typedef struct mal_pa_operation    mal_pa_operation;
typedef struct mal_pa_stream       mal_pa_stream;
typedef struct mal_pa_spawn_api    mal_pa_spawn_api;

typedef struct
{
    mal_uint32 maxlength;
    mal_uint32 tlength;
    mal_uint32 prebuf;
    mal_uint32 minreq;
    mal_uint32 fragsize;
} mal_pa_buffer_attr;

typedef struct
{
    mal_uint8 channels;
    mal_pa_channel_position_t map[MAL_PA_CHANNELS_MAX];
} mal_pa_channel_map;

typedef struct
{
    mal_uint8 channels;
    mal_uint32 values[MAL_PA_CHANNELS_MAX];
} mal_pa_cvolume;

typedef struct
{
    mal_pa_sample_format_t format;
    mal_uint32 rate;
    mal_uint8 channels;
} mal_pa_sample_spec;

typedef struct
{
    const char* name;
    mal_uint32 index;
    const char* description;
    mal_pa_sample_spec sample_spec;
    mal_pa_channel_map channel_map;
    mal_uint32 owner_module;
    mal_pa_cvolume volume;
    int mute;
    mal_uint32 monitor_source;
    const char* monitor_source_name;
    mal_uint64 latency;
    const char* driver;
    mal_pa_sink_flags_t flags;
    void* proplist;
    mal_uint64 configured_latency;
    mal_uint32 base_volume;
    mal_pa_sink_state_t state;
    mal_uint32 n_volume_steps;
    mal_uint32 card;
    mal_uint32 n_ports;
    void** ports;
    void* active_port;
    mal_uint8 n_formats;
    void** formats;
} mal_pa_sink_info;

typedef struct
{
    const char *name;
    mal_uint32 index;
    const char *description;
    mal_pa_sample_spec sample_spec;
    mal_pa_channel_map channel_map;
    mal_uint32 owner_module;
    mal_pa_cvolume volume;
    int mute;
    mal_uint32 monitor_of_sink;
    const char *monitor_of_sink_name;
    mal_uint64 latency;
    const char *driver;
    mal_pa_source_flags_t flags;
    void* proplist;
    mal_uint64 configured_latency;
    mal_uint32 base_volume;
    mal_pa_source_state_t state;
    mal_uint32 n_volume_steps;
    mal_uint32 card;
    mal_uint32 n_ports;
    void** ports;
    void* active_port;
    mal_uint8 n_formats;
    void** formats;
} mal_pa_source_info;

typedef void (* mal_pa_context_notify_cb_t)(mal_pa_context* c, void* userdata);
typedef void (* mal_pa_sink_info_cb_t)     (mal_pa_context* c, const mal_pa_sink_info* i, int eol, void* userdata);
typedef void (* mal_pa_source_info_cb_t)   (mal_pa_context* c, const mal_pa_source_info* i, int eol, void* userdata);
typedef void (* mal_pa_stream_success_cb_t)(mal_pa_stream* s, int success, void* userdata);
typedef void (* mal_pa_stream_request_cb_t)(mal_pa_stream* s, size_t nbytes, void* userdata);
typedef void (* mal_pa_free_cb_t)          (void* p);
#endif


typedef mal_pa_mainloop*          (* mal_pa_mainloop_new_proc)                   ();
typedef void                      (* mal_pa_mainloop_free_proc)                  (mal_pa_mainloop* m);
typedef mal_pa_mainloop_api*      (* mal_pa_mainloop_get_api_proc)               (mal_pa_mainloop* m);
typedef int                       (* mal_pa_mainloop_iterate_proc)               (mal_pa_mainloop* m, int block, int* retval);
typedef void                      (* mal_pa_mainloop_wakeup_proc)                (mal_pa_mainloop* m);
typedef mal_pa_context*           (* mal_pa_context_new_proc)                    (mal_pa_mainloop_api* mainloop, const char* name);
typedef void                      (* mal_pa_context_unref_proc)                  (mal_pa_context* c);
typedef int                       (* mal_pa_context_connect_proc)                (mal_pa_context* c, const char* server, mal_pa_context_flags_t flags, const mal_pa_spawn_api* api);
typedef void                      (* mal_pa_context_disconnect_proc)             (mal_pa_context* c);
typedef void                      (* mal_pa_context_set_state_callback_proc)     (mal_pa_context* c, mal_pa_context_notify_cb_t cb, void* userdata);
typedef mal_pa_context_state_t    (* mal_pa_context_get_state_proc)              (mal_pa_context* c);
typedef mal_pa_operation*         (* mal_pa_context_get_sink_info_list_proc)     (mal_pa_context* c, mal_pa_sink_info_cb_t cb, void* userdata);
typedef mal_pa_operation*         (* mal_pa_context_get_source_info_list_proc)   (mal_pa_context* c, mal_pa_source_info_cb_t cb, void* userdata);
typedef mal_pa_operation*         (* mal_pa_context_get_sink_info_by_name_proc)  (mal_pa_context* c, const char* name, mal_pa_sink_info_cb_t cb, void* userdata);
typedef mal_pa_operation*         (* mal_pa_context_get_source_info_by_name_proc)(mal_pa_context* c, const char* name, mal_pa_source_info_cb_t cb, void* userdata);
typedef void                      (* mal_pa_operation_unref_proc)                (mal_pa_operation* o);
typedef mal_pa_operation_state_t  (* mal_pa_operation_get_state_proc)            (mal_pa_operation* o);
typedef mal_pa_channel_map*       (* mal_pa_channel_map_init_extend_proc)        (mal_pa_channel_map* m, unsigned channels, mal_pa_channel_map_def_t def);
typedef int                       (* mal_pa_channel_map_valid_proc)              (const mal_pa_channel_map* m);
typedef int                       (* mal_pa_channel_map_compatible_proc)         (const mal_pa_channel_map* m, const mal_pa_sample_spec* ss);
typedef mal_pa_stream*            (* mal_pa_stream_new_proc)                     (mal_pa_context* c, const char* name, const mal_pa_sample_spec* ss, const mal_pa_channel_map* map);
typedef void                      (* mal_pa_stream_unref_proc)                   (mal_pa_stream* s);
typedef int                       (* mal_pa_stream_connect_playback_proc)        (mal_pa_stream* s, const char* dev, const mal_pa_buffer_attr* attr, mal_pa_stream_flags_t flags, const mal_pa_cvolume* volume, mal_pa_stream* sync_stream);
typedef int                       (* mal_pa_stream_connect_record_proc)          (mal_pa_stream* s, const char* dev, const mal_pa_buffer_attr* attr, mal_pa_stream_flags_t flags);
typedef int                       (* mal_pa_stream_disconnect_proc)              (mal_pa_stream* s);
typedef mal_pa_stream_state_t     (* mal_pa_stream_get_state_proc)               (mal_pa_stream* s);
typedef const mal_pa_sample_spec* (* mal_pa_stream_get_sample_spec_proc)         (mal_pa_stream* s);
typedef const mal_pa_channel_map* (* mal_pa_stream_get_channel_map_proc)         (mal_pa_stream* s);
typedef const mal_pa_buffer_attr* (* mal_pa_stream_get_buffer_attr_proc)         (mal_pa_stream* s);
typedef const char*               (* mal_pa_stream_get_device_name_proc)         (mal_pa_stream* s);
typedef void                      (* mal_pa_stream_set_write_callback_proc)      (mal_pa_stream* s, mal_pa_stream_request_cb_t cb, void* userdata);
typedef void                      (* mal_pa_stream_set_read_callback_proc)       (mal_pa_stream* s, mal_pa_stream_request_cb_t cb, void* userdata);
typedef mal_pa_operation*         (* mal_pa_stream_flush_proc)                   (mal_pa_stream* s, mal_pa_stream_success_cb_t cb, void* userdata);
typedef mal_pa_operation*         (* mal_pa_stream_drain_proc)                   (mal_pa_stream* s, mal_pa_stream_success_cb_t cb, void* userdata);
typedef mal_pa_operation*         (* mal_pa_stream_cork_proc)                    (mal_pa_stream* s, int b, mal_pa_stream_success_cb_t cb, void* userdata);
typedef mal_pa_operation*         (* mal_pa_stream_trigger_proc)                 (mal_pa_stream* s, mal_pa_stream_success_cb_t cb, void* userdata);
typedef int                       (* mal_pa_stream_begin_write_proc)             (mal_pa_stream* s, void** data, size_t* nbytes);
typedef int                       (* mal_pa_stream_write_proc)                   (mal_pa_stream* s, const void* data, size_t nbytes, mal_pa_free_cb_t free_cb, int64_t offset, mal_pa_seek_mode_t seek);
typedef int                       (* mal_pa_stream_peek_proc)                    (mal_pa_stream* s, const void** data, size_t* nbytes);
typedef int                       (* mal_pa_stream_drop_proc)                    (mal_pa_stream* s);

typedef struct
{
    mal_uint32 count;
    mal_uint32 capacity;
    mal_device_info* pInfo;
} mal_pulse_device_enum_data;

mal_result mal_result_from_pulse(int result)
{
    switch (result) {
        case MAL_PA_OK:           return MAL_SUCCESS;
        case MAL_PA_ERR_ACCESS:   return MAL_ACCESS_DENIED;
        case MAL_PA_ERR_INVALID:  return MAL_INVALID_ARGS;
        case MAL_PA_ERR_NOENTITY: return MAL_NO_DEVICE;
        default:                  return MAL_ERROR;
    }
}

#if 0
mal_pa_sample_format_t mal_format_to_pulse(mal_format format)
{
    if (mal_is_little_endian()) {
        switch (format) {
            case mal_format_s16: return MAL_PA_SAMPLE_S16LE;
            case mal_format_s24: return MAL_PA_SAMPLE_S24LE;
            case mal_format_s32: return MAL_PA_SAMPLE_S32LE;
            case mal_format_f32: return MAL_PA_SAMPLE_FLOAT32LE;
            default: break;
        }
    } else {
        switch (format) {
            case mal_format_s16: return MAL_PA_SAMPLE_S16BE;
            case mal_format_s24: return MAL_PA_SAMPLE_S24BE;
            case mal_format_s32: return MAL_PA_SAMPLE_S32BE;
            case mal_format_f32: return MAL_PA_SAMPLE_FLOAT32BE;
            default: break;
        }
    }

    // Endian agnostic.
    switch (format) {
        case mal_format_u8: return MAL_PA_SAMPLE_U8;
        default: return MAL_PA_SAMPLE_INVALID;
    }
}
#endif

mal_format mal_format_from_pulse(mal_pa_sample_format_t format)
{
    if (mal_is_little_endian()) {
        switch (format) {
            case MAL_PA_SAMPLE_S16LE:     return mal_format_s16;
            case MAL_PA_SAMPLE_S24LE:     return mal_format_s24;
            case MAL_PA_SAMPLE_S32LE:     return mal_format_s32;
            case MAL_PA_SAMPLE_FLOAT32LE: return mal_format_f32;
            default: break;
        }
    } else {
        switch (format) {
            case MAL_PA_SAMPLE_S16BE:     return mal_format_s16;
            case MAL_PA_SAMPLE_S24BE:     return mal_format_s24;
            case MAL_PA_SAMPLE_S32BE:     return mal_format_s32;
            case MAL_PA_SAMPLE_FLOAT32BE: return mal_format_f32;
            default: break;
        }
    }

    // Endian agnostic.
    switch (format) {
        case MAL_PA_SAMPLE_U8: return mal_format_u8;
        default: return mal_format_unknown;
    }
}

mal_channel mal_channel_position_from_pulse(mal_pa_channel_position_t position)
{
    switch (position)
    {
        case MAL_PA_CHANNEL_POSITION_INVALID:               return MAL_CHANNEL_NONE;
        case MAL_PA_CHANNEL_POSITION_MONO:                  return MAL_CHANNEL_MONO;
        case MAL_PA_CHANNEL_POSITION_FRONT_LEFT:            return MAL_CHANNEL_FRONT_LEFT;
        case MAL_PA_CHANNEL_POSITION_FRONT_RIGHT:           return MAL_CHANNEL_FRONT_RIGHT;
        case MAL_PA_CHANNEL_POSITION_FRONT_CENTER:          return MAL_CHANNEL_FRONT_CENTER;
        case MAL_PA_CHANNEL_POSITION_REAR_CENTER:           return MAL_CHANNEL_BACK_CENTER;
        case MAL_PA_CHANNEL_POSITION_REAR_LEFT:             return MAL_CHANNEL_BACK_LEFT;
        case MAL_PA_CHANNEL_POSITION_REAR_RIGHT:            return MAL_CHANNEL_BACK_RIGHT;
        case MAL_PA_CHANNEL_POSITION_LFE:                   return MAL_CHANNEL_LFE;
        case MAL_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER:  return MAL_CHANNEL_FRONT_LEFT_CENTER;
        case MAL_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER: return MAL_CHANNEL_FRONT_RIGHT_CENTER;
        case MAL_PA_CHANNEL_POSITION_SIDE_LEFT:             return MAL_CHANNEL_SIDE_LEFT;
        case MAL_PA_CHANNEL_POSITION_SIDE_RIGHT:            return MAL_CHANNEL_SIDE_RIGHT;
        case MAL_PA_CHANNEL_POSITION_AUX0:                  return MAL_CHANNEL_AUX_0;
        case MAL_PA_CHANNEL_POSITION_AUX1:                  return MAL_CHANNEL_AUX_1;
        case MAL_PA_CHANNEL_POSITION_AUX2:                  return MAL_CHANNEL_AUX_2;
        case MAL_PA_CHANNEL_POSITION_AUX3:                  return MAL_CHANNEL_AUX_3;
        case MAL_PA_CHANNEL_POSITION_AUX4:                  return MAL_CHANNEL_AUX_4;
        case MAL_PA_CHANNEL_POSITION_AUX5:                  return MAL_CHANNEL_AUX_5;
        case MAL_PA_CHANNEL_POSITION_AUX6:                  return MAL_CHANNEL_AUX_6;
        case MAL_PA_CHANNEL_POSITION_AUX7:                  return MAL_CHANNEL_AUX_7;
        case MAL_PA_CHANNEL_POSITION_AUX8:                  return MAL_CHANNEL_AUX_8;
        case MAL_PA_CHANNEL_POSITION_AUX9:                  return MAL_CHANNEL_AUX_9;
        case MAL_PA_CHANNEL_POSITION_AUX10:                 return MAL_CHANNEL_AUX_10;
        case MAL_PA_CHANNEL_POSITION_AUX11:                 return MAL_CHANNEL_AUX_11;
        case MAL_PA_CHANNEL_POSITION_AUX12:                 return MAL_CHANNEL_AUX_12;
        case MAL_PA_CHANNEL_POSITION_AUX13:                 return MAL_CHANNEL_AUX_13;
        case MAL_PA_CHANNEL_POSITION_AUX14:                 return MAL_CHANNEL_AUX_14;
        case MAL_PA_CHANNEL_POSITION_AUX15:                 return MAL_CHANNEL_AUX_15;
        case MAL_PA_CHANNEL_POSITION_AUX16:                 return MAL_CHANNEL_AUX_16;
        case MAL_PA_CHANNEL_POSITION_AUX17:                 return MAL_CHANNEL_AUX_17;
        case MAL_PA_CHANNEL_POSITION_AUX18:                 return MAL_CHANNEL_AUX_18;
        case MAL_PA_CHANNEL_POSITION_AUX19:                 return MAL_CHANNEL_AUX_19;
        case MAL_PA_CHANNEL_POSITION_AUX20:                 return MAL_CHANNEL_AUX_20;
        case MAL_PA_CHANNEL_POSITION_AUX21:                 return MAL_CHANNEL_AUX_21;
        case MAL_PA_CHANNEL_POSITION_AUX22:                 return MAL_CHANNEL_AUX_22;
        case MAL_PA_CHANNEL_POSITION_AUX23:                 return MAL_CHANNEL_AUX_23;
        case MAL_PA_CHANNEL_POSITION_AUX24:                 return MAL_CHANNEL_AUX_24;
        case MAL_PA_CHANNEL_POSITION_AUX25:                 return MAL_CHANNEL_AUX_25;
        case MAL_PA_CHANNEL_POSITION_AUX26:                 return MAL_CHANNEL_AUX_26;
        case MAL_PA_CHANNEL_POSITION_AUX27:                 return MAL_CHANNEL_AUX_27;
        case MAL_PA_CHANNEL_POSITION_AUX28:                 return MAL_CHANNEL_AUX_28;
        case MAL_PA_CHANNEL_POSITION_AUX29:                 return MAL_CHANNEL_AUX_29;
        case MAL_PA_CHANNEL_POSITION_AUX30:                 return MAL_CHANNEL_AUX_30;
        case MAL_PA_CHANNEL_POSITION_AUX31:                 return MAL_CHANNEL_AUX_31;
        case MAL_PA_CHANNEL_POSITION_TOP_CENTER:            return MAL_CHANNEL_TOP_CENTER;
        case MAL_PA_CHANNEL_POSITION_TOP_FRONT_LEFT:        return MAL_CHANNEL_TOP_FRONT_LEFT;
        case MAL_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT:       return MAL_CHANNEL_TOP_FRONT_RIGHT;
        case MAL_PA_CHANNEL_POSITION_TOP_FRONT_CENTER:      return MAL_CHANNEL_TOP_FRONT_CENTER;
        case MAL_PA_CHANNEL_POSITION_TOP_REAR_LEFT:         return MAL_CHANNEL_TOP_BACK_LEFT;
        case MAL_PA_CHANNEL_POSITION_TOP_REAR_RIGHT:        return MAL_CHANNEL_TOP_BACK_RIGHT;
        case MAL_PA_CHANNEL_POSITION_TOP_REAR_CENTER:       return MAL_CHANNEL_TOP_BACK_CENTER;
        default: return MAL_CHANNEL_NONE;
    }
}

#if 0
mal_pa_channel_position_t mal_channel_position_to_pulse(mal_channel position)
{
    switch (position)
    {
        case MAL_CHANNEL_NONE:               return MAL_PA_CHANNEL_POSITION_INVALID;
        case MAL_CHANNEL_FRONT_LEFT:         return MAL_PA_CHANNEL_POSITION_FRONT_LEFT;
        case MAL_CHANNEL_FRONT_RIGHT:        return MAL_PA_CHANNEL_POSITION_FRONT_RIGHT;
        case MAL_CHANNEL_FRONT_CENTER:       return MAL_PA_CHANNEL_POSITION_FRONT_CENTER;
        case MAL_CHANNEL_LFE:                return MAL_PA_CHANNEL_POSITION_LFE;
        case MAL_CHANNEL_BACK_LEFT:          return MAL_PA_CHANNEL_POSITION_REAR_LEFT;
        case MAL_CHANNEL_BACK_RIGHT:         return MAL_PA_CHANNEL_POSITION_REAR_RIGHT;
        case MAL_CHANNEL_FRONT_LEFT_CENTER:  return MAL_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER;
        case MAL_CHANNEL_FRONT_RIGHT_CENTER: return MAL_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER;
        case MAL_CHANNEL_BACK_CENTER:        return MAL_PA_CHANNEL_POSITION_REAR_CENTER;
        case MAL_CHANNEL_SIDE_LEFT:          return MAL_PA_CHANNEL_POSITION_SIDE_LEFT;
        case MAL_CHANNEL_SIDE_RIGHT:         return MAL_PA_CHANNEL_POSITION_SIDE_RIGHT;
        case MAL_CHANNEL_TOP_CENTER:         return MAL_PA_CHANNEL_POSITION_TOP_CENTER;
        case MAL_CHANNEL_TOP_FRONT_LEFT:     return MAL_PA_CHANNEL_POSITION_TOP_FRONT_LEFT;
        case MAL_CHANNEL_TOP_FRONT_CENTER:   return MAL_PA_CHANNEL_POSITION_TOP_FRONT_CENTER;
        case MAL_CHANNEL_TOP_FRONT_RIGHT:    return MAL_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT;
        case MAL_CHANNEL_TOP_BACK_LEFT:      return MAL_PA_CHANNEL_POSITION_TOP_REAR_LEFT;
        case MAL_CHANNEL_TOP_BACK_CENTER:    return MAL_PA_CHANNEL_POSITION_TOP_REAR_CENTER;
        case MAL_CHANNEL_TOP_BACK_RIGHT:     return MAL_PA_CHANNEL_POSITION_TOP_REAR_RIGHT;
        case MAL_CHANNEL_19:                 return MAL_PA_CHANNEL_POSITION_AUX18;
        case MAL_CHANNEL_20:                 return MAL_PA_CHANNEL_POSITION_AUX19;
        case MAL_CHANNEL_21:                 return MAL_PA_CHANNEL_POSITION_AUX20;
        case MAL_CHANNEL_22:                 return MAL_PA_CHANNEL_POSITION_AUX21;
        case MAL_CHANNEL_23:                 return MAL_PA_CHANNEL_POSITION_AUX22;
        case MAL_CHANNEL_24:                 return MAL_PA_CHANNEL_POSITION_AUX23;
        case MAL_CHANNEL_25:                 return MAL_PA_CHANNEL_POSITION_AUX24;
        case MAL_CHANNEL_26:                 return MAL_PA_CHANNEL_POSITION_AUX25;
        case MAL_CHANNEL_27:                 return MAL_PA_CHANNEL_POSITION_AUX26;
        case MAL_CHANNEL_28:                 return MAL_PA_CHANNEL_POSITION_AUX27;
        case MAL_CHANNEL_29:                 return MAL_PA_CHANNEL_POSITION_AUX28;
        case MAL_CHANNEL_30:                 return MAL_PA_CHANNEL_POSITION_AUX29;
        case MAL_CHANNEL_31:                 return MAL_PA_CHANNEL_POSITION_AUX30;
        case MAL_CHANNEL_32:                 return MAL_PA_CHANNEL_POSITION_AUX31;
        default: return (mal_pa_channel_position_t)position;
    }
}
#endif


mal_result mal_wait_for_operation__pulse(mal_context* pContext, mal_pa_mainloop* pMainLoop, mal_pa_operation* pOP)
{
    mal_assert(pContext != NULL);
    mal_assert(pMainLoop != NULL);
    mal_assert(pOP != NULL);

    while (((mal_pa_operation_get_state_proc)pContext->pulse.pa_operation_get_state)(pOP) != MAL_PA_OPERATION_DONE) {
        int error = ((mal_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)(pMainLoop, 1, NULL);
        if (error < 0) {
            return mal_result_from_pulse(error);
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_device__wait_for_operation__pulse(mal_device* pDevice, mal_pa_operation* pOP)
{
    mal_assert(pDevice != NULL);
    mal_assert(pOP != NULL);

    return mal_wait_for_operation__pulse(pDevice->pContext, (mal_pa_mainloop*)pDevice->pulse.pMainLoop, pOP);
}


mal_bool32 mal_context_is_device_id_equal__pulse(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return mal_strcmp(pID0->pulse, pID1->pulse) == 0;
}


typedef struct
{
    mal_context* pContext;
    mal_enum_devices_callback_proc callback;
    void* pUserData;
    mal_bool32 isTerminated;
} mal_context_enumerate_devices_callback_data__pulse;

void mal_context_enumerate_devices_sink_callback__pulse(mal_pa_context* pPulseContext, const mal_pa_sink_info* pSinkInfo, int endOfList, void* pUserData)
{
    mal_context_enumerate_devices_callback_data__pulse* pData = (mal_context_enumerate_devices_callback_data__pulse*)pUserData;
    mal_assert(pData != NULL);

    if (endOfList || pData->isTerminated) {
        return;
    }

    mal_device_info deviceInfo;
    mal_zero_object(&deviceInfo);

    // The name from PulseAudio is the ID for mini_al.
    if (pSinkInfo->name != NULL) {
        mal_strncpy_s(deviceInfo.id.pulse, sizeof(deviceInfo.id.pulse), pSinkInfo->name, (size_t)-1);
    }

    // The description from PulseAudio is the name for mini_al.
    if (pSinkInfo->description != NULL) {
        mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), pSinkInfo->description, (size_t)-1);
    }

    pData->isTerminated = !pData->callback(pData->pContext, mal_device_type_playback, &deviceInfo, pData->pUserData);
}

void mal_context_enumerate_devices_source_callback__pulse(mal_pa_context* pPulseContext, const mal_pa_source_info* pSinkInfo, int endOfList, void* pUserData)
{
    mal_context_enumerate_devices_callback_data__pulse* pData = (mal_context_enumerate_devices_callback_data__pulse*)pUserData;
    mal_assert(pData != NULL);

    if (endOfList || pData->isTerminated) {
        return;
    }

    mal_device_info deviceInfo;
    mal_zero_object(&deviceInfo);

    // The name from PulseAudio is the ID for mini_al.
    if (pSinkInfo->name != NULL) {
        mal_strncpy_s(deviceInfo.id.pulse, sizeof(deviceInfo.id.pulse), pSinkInfo->name, (size_t)-1);
    }

    // The description from PulseAudio is the name for mini_al.
    if (pSinkInfo->description != NULL) {
        mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), pSinkInfo->description, (size_t)-1);
    }

    pData->isTerminated = !pData->callback(pData->pContext, mal_device_type_capture, &deviceInfo, pData->pUserData);
}

mal_result mal_context_enumerate_devices__pulse(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    mal_result result = MAL_SUCCESS;

    mal_context_enumerate_devices_callback_data__pulse callbackData;
    callbackData.pContext = pContext;
    callbackData.callback = callback;
    callbackData.pUserData = pUserData;
    callbackData.isTerminated = MAL_FALSE;

    mal_pa_operation* pOP = NULL;

    mal_pa_mainloop* pMainLoop = ((mal_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
    if (pMainLoop == NULL) {
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    mal_pa_mainloop_api* pAPI = ((mal_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)(pMainLoop);
    if (pAPI == NULL) {
        ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    mal_pa_context* pPulseContext = ((mal_pa_context_new_proc)pContext->pulse.pa_context_new)(pAPI, pContext->config.pulse.pApplicationName);
    if (pPulseContext == NULL) {
        ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    int error = ((mal_pa_context_connect_proc)pContext->pulse.pa_context_connect)(pPulseContext, pContext->config.pulse.pServerName, 0, NULL);
    if (error != MAL_PA_OK) {
        ((mal_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
        ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return mal_result_from_pulse(error);
    }

    while (((mal_pa_context_get_state_proc)pContext->pulse.pa_context_get_state)(pPulseContext) != MAL_PA_CONTEXT_READY) {
        error = ((mal_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)(pMainLoop, 1, NULL);
        if (error < 0) {
            result = mal_result_from_pulse(error);
            goto done;
        }
    }


    // Playback.
    if (!callbackData.isTerminated) {
        pOP = ((mal_pa_context_get_sink_info_list_proc)pContext->pulse.pa_context_get_sink_info_list)(pPulseContext, mal_context_enumerate_devices_sink_callback__pulse, &callbackData);
        if (pOP == NULL) {
            result = MAL_ERROR;
            goto done;
        }

        result = mal_wait_for_operation__pulse(pContext, pMainLoop, pOP);
        ((mal_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        if (result != MAL_SUCCESS) {
            goto done;
        }
    }


    // Capture.
    if (!callbackData.isTerminated) {
        pOP = ((mal_pa_context_get_source_info_list_proc)pContext->pulse.pa_context_get_source_info_list)(pPulseContext, mal_context_enumerate_devices_source_callback__pulse, &callbackData);
        if (pOP == NULL) {
            result = MAL_ERROR;
            goto done;
        }

        result = mal_wait_for_operation__pulse(pContext, pMainLoop, pOP);
        ((mal_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        if (result != MAL_SUCCESS) {
            goto done;
        }
    }

done:
    ((mal_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)(pPulseContext);
    ((mal_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
    ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
    return result;
}


typedef struct
{
    mal_device_info* pDeviceInfo;
    mal_bool32 foundDevice;
} mal_context_get_device_info_callback_data__pulse;

void mal_context_get_device_info_sink_callback__pulse(mal_pa_context* pPulseContext, const mal_pa_sink_info* pInfo, int endOfList, void* pUserData)
{
    if (endOfList > 0) {
        return;
    }

    mal_context_get_device_info_callback_data__pulse* pData = (mal_context_get_device_info_callback_data__pulse*)pUserData;
    mal_assert(pData != NULL);
    pData->foundDevice = MAL_TRUE;

    if (pInfo->name != NULL) {
        mal_strncpy_s(pData->pDeviceInfo->id.pulse, sizeof(pData->pDeviceInfo->id.pulse), pInfo->name, (size_t)-1);
    }

    if (pInfo->description != NULL) {
        mal_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pInfo->description, (size_t)-1);
    }

    pData->pDeviceInfo->minChannels   = pInfo->sample_spec.channels;
    pData->pDeviceInfo->maxChannels   = pInfo->sample_spec.channels;
    pData->pDeviceInfo->minSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->maxSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->formatCount = 1;
    pData->pDeviceInfo->formats[0] = mal_format_from_pulse(pInfo->sample_spec.format);
}

void mal_context_get_device_info_source_callback__pulse(mal_pa_context* pPulseContext, const mal_pa_source_info* pInfo, int endOfList, void* pUserData)
{
    if (endOfList > 0) {
        return;
    }

    mal_context_get_device_info_callback_data__pulse* pData = (mal_context_get_device_info_callback_data__pulse*)pUserData;
    mal_assert(pData != NULL);
    pData->foundDevice = MAL_TRUE;

    if (pInfo->name != NULL) {
        mal_strncpy_s(pData->pDeviceInfo->id.pulse, sizeof(pData->pDeviceInfo->id.pulse), pInfo->name, (size_t)-1);
    }

    if (pInfo->description != NULL) {
        mal_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pInfo->description, (size_t)-1);
    }

    pData->pDeviceInfo->minChannels = pInfo->sample_spec.channels;
    pData->pDeviceInfo->maxChannels = pInfo->sample_spec.channels;
    pData->pDeviceInfo->minSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->maxSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->formatCount = 1;
    pData->pDeviceInfo->formats[0] = mal_format_from_pulse(pInfo->sample_spec.format);
}

mal_result mal_context_get_device_info__pulse(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    (void)shareMode;

    mal_result result = MAL_SUCCESS;

    mal_context_get_device_info_callback_data__pulse callbackData;
    callbackData.pDeviceInfo = pDeviceInfo;
    callbackData.foundDevice = MAL_FALSE;

    mal_pa_operation* pOP = NULL;

    mal_pa_mainloop* pMainLoop = ((mal_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
    if (pMainLoop == NULL) {
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    mal_pa_mainloop_api* pAPI = ((mal_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)(pMainLoop);
    if (pAPI == NULL) {
        ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    mal_pa_context* pPulseContext = ((mal_pa_context_new_proc)pContext->pulse.pa_context_new)(pAPI, pContext->config.pulse.pApplicationName);
    if (pPulseContext == NULL) {
        ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    int error = ((mal_pa_context_connect_proc)pContext->pulse.pa_context_connect)(pPulseContext, pContext->config.pulse.pServerName, 0, NULL);
    if (error != MAL_PA_OK) {
        ((mal_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
        ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return mal_result_from_pulse(error);
    }

    while (((mal_pa_context_get_state_proc)pContext->pulse.pa_context_get_state)(pPulseContext) != MAL_PA_CONTEXT_READY) {
        error = ((mal_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)(pMainLoop, 1, NULL);
        if (error < 0) {
            result = mal_result_from_pulse(error);
            goto done;
        }
    }

    if (deviceType == mal_device_type_playback) {
        pOP = ((mal_pa_context_get_sink_info_by_name_proc)pContext->pulse.pa_context_get_sink_info_by_name)(pPulseContext, pDeviceID->pulse, mal_context_get_device_info_sink_callback__pulse, &callbackData);
    } else {
        pOP = ((mal_pa_context_get_source_info_by_name_proc)pContext->pulse.pa_context_get_source_info_by_name)(pPulseContext, pDeviceID->pulse, mal_context_get_device_info_source_callback__pulse, &callbackData);
    }

    if (pOP != NULL) {
        mal_wait_for_operation__pulse(pContext, pMainLoop, pOP);
        ((mal_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
    } else {
        result = MAL_ERROR;
        goto done;
    }

    if (!callbackData.foundDevice) {
        result = MAL_NO_DEVICE;
        goto done;
    }


done:
    ((mal_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)(pPulseContext);
    ((mal_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
    ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
    return result;
}


void mal_pulse_device_state_callback(mal_pa_context* pPulseContext, void* pUserData)
{
    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    pDevice->pulse.pulseContextState = ((mal_pa_context_get_state_proc)pContext->pulse.pa_context_get_state)(pPulseContext);
}

void mal_pulse_device_write_callback(mal_pa_stream* pStream, size_t sizeInBytes, void* pUserData)
{
    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    size_t bytesRemaining = sizeInBytes;
    while (bytesRemaining > 0) {
        size_t bytesToReadFromClient = bytesRemaining;
        if (bytesToReadFromClient > 0xFFFFFFFF) {
            bytesToReadFromClient = 0xFFFFFFFF;
        }

        void* pBuffer = NULL;
        int error = ((mal_pa_stream_begin_write_proc)pContext->pulse.pa_stream_begin_write)((mal_pa_stream*)pDevice->pulse.pStream, &pBuffer, &bytesToReadFromClient);
        if (error < 0) {
            mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to retrieve write buffer for sending data to the device.", mal_result_from_pulse(error));
            return;
        }

        if (pBuffer != NULL && bytesToReadFromClient > 0) {
            mal_uint32 framesToReadFromClient = (mal_uint32)bytesToReadFromClient / (pDevice->internalChannels*mal_get_bytes_per_sample(pDevice->internalFormat));
            if (framesToReadFromClient > 0) {
                mal_device__read_frames_from_client(pDevice, framesToReadFromClient, pBuffer);

                error = ((mal_pa_stream_write_proc)pContext->pulse.pa_stream_write)((mal_pa_stream*)pDevice->pulse.pStream, pBuffer, bytesToReadFromClient, NULL, 0, MAL_PA_SEEK_RELATIVE);
                if (error < 0) {
                    mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to write data to the PulseAudio stream.", mal_result_from_pulse(error));
                    return;
                }
            }

            bytesRemaining -= bytesToReadFromClient;
        }
    }
}

void mal_pulse_device_read_callback(mal_pa_stream* pStream, size_t sizeInBytes, void* pUserData)
{
    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    size_t bytesRemaining = sizeInBytes;
    while (bytesRemaining > 0) {
        size_t bytesToSendToClient = bytesRemaining;
        if (bytesToSendToClient > 0xFFFFFFFF) {
            bytesToSendToClient = 0xFFFFFFFF;
        }

        const void* pBuffer = NULL;
        int error = ((mal_pa_stream_peek_proc)pContext->pulse.pa_stream_peek)((mal_pa_stream*)pDevice->pulse.pStream, &pBuffer, &sizeInBytes);
        if (error < 0) {
            mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to retrieve read buffer for reading data from the device.", mal_result_from_pulse(error));
            return;
        }

        if (pBuffer != NULL) {
            mal_uint32 framesToSendToClient = (mal_uint32)bytesToSendToClient / (pDevice->internalChannels*mal_get_bytes_per_sample(pDevice->internalFormat));
            if (framesToSendToClient > 0) {
                mal_device__send_frames_to_client(pDevice, framesToSendToClient, pBuffer);
            }
        }

        error = ((mal_pa_stream_drop_proc)pContext->pulse.pa_stream_drop)((mal_pa_stream*)pDevice->pulse.pStream);
        if (error < 0) {
            mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to drop fragment from the PulseAudio stream.", mal_result_from_pulse(error));
        }

        bytesRemaining -= bytesToSendToClient;
    }
}

void mal_device_sink_info_callback(mal_pa_context* pPulseContext, const mal_pa_sink_info* pInfo, int endOfList, void* pUserData)
{
    if (endOfList > 0) {
        return;
    }

    mal_pa_sink_info* pInfoOut = (mal_pa_sink_info*)pUserData;
    mal_assert(pInfoOut != NULL);

    *pInfoOut = *pInfo;
}

void mal_device_source_info_callback(mal_pa_context* pPulseContext, const mal_pa_source_info* pInfo, int endOfList, void* pUserData)
{
    if (endOfList > 0) {
        return;
    }

    mal_pa_source_info* pInfoOut = (mal_pa_source_info*)pUserData;
    mal_assert(pInfoOut != NULL);

    *pInfoOut = *pInfo;
}

void mal_device_sink_name_callback(mal_pa_context* pPulseContext, const mal_pa_sink_info* pInfo, int endOfList, void* pUserData)
{
    if (endOfList > 0) {
        return;
    }

    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    mal_strncpy_s(pDevice->name, sizeof(pDevice->name), pInfo->description, (size_t)-1);
}

void mal_device_source_name_callback(mal_pa_context* pPulseContext, const mal_pa_source_info* pInfo, int endOfList, void* pUserData)
{
    if (endOfList > 0) {
        return;
    }

    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    mal_strncpy_s(pDevice->name, sizeof(pDevice->name), pInfo->description, (size_t)-1);
}

void mal_device_uninit__pulse(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    ((mal_pa_stream_disconnect_proc)pContext->pulse.pa_stream_disconnect)((mal_pa_stream*)pDevice->pulse.pStream);
    ((mal_pa_stream_unref_proc)pContext->pulse.pa_stream_unref)((mal_pa_stream*)pDevice->pulse.pStream);
    ((mal_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)((mal_pa_context*)pDevice->pulse.pPulseContext);
    ((mal_pa_context_unref_proc)pContext->pulse.pa_context_unref)((mal_pa_context*)pDevice->pulse.pPulseContext);
    ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)((mal_pa_mainloop*)pDevice->pulse.pMainLoop);
}

mal_result mal_device_init__pulse(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pContext;

    mal_assert(pDevice != NULL);
    mal_zero_object(&pDevice->pulse);

    mal_result result = MAL_SUCCESS;
    int error = 0;

    const char* dev = NULL;
    if (pDeviceID != NULL) {
        dev = pDeviceID->pulse;
    }

    mal_uint32 bufferSizeInFrames = pConfig->bufferSizeInFrames;

    mal_pa_sink_info sinkInfo;
    mal_pa_source_info sourceInfo;
    mal_pa_operation* pOP = NULL;

    mal_pa_sample_spec ss;
    mal_pa_channel_map cmap;
    mal_pa_buffer_attr attr;

    const mal_pa_sample_spec* pActualSS   = NULL;
    const mal_pa_channel_map* pActualCMap = NULL;
    const mal_pa_buffer_attr* pActualAttr = NULL;



    pDevice->pulse.pMainLoop = ((mal_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
    if (pDevice->pulse.pMainLoop == NULL) {
        result = mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create main loop for device.", MAL_FAILED_TO_INIT_BACKEND);
        goto on_error0;
    }

    pDevice->pulse.pAPI = ((mal_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)((mal_pa_mainloop*)pDevice->pulse.pMainLoop);
    if (pDevice->pulse.pAPI == NULL) {
        result = mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to retrieve PulseAudio main loop.", MAL_FAILED_TO_INIT_BACKEND);
        goto on_error1;
    }

    pDevice->pulse.pPulseContext = ((mal_pa_context_new_proc)pContext->pulse.pa_context_new)((mal_pa_mainloop_api*)pDevice->pulse.pAPI, pContext->config.pulse.pApplicationName);
    if (pDevice->pulse.pPulseContext == NULL) {
        result = mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create PulseAudio context for device.", MAL_FAILED_TO_INIT_BACKEND);
        goto on_error1;
    }

    error = ((mal_pa_context_connect_proc)pContext->pulse.pa_context_connect)((mal_pa_context*)pDevice->pulse.pPulseContext, pContext->config.pulse.pServerName, (pContext->config.pulse.tryAutoSpawn) ? 0 : MAL_PA_CONTEXT_NOAUTOSPAWN, NULL);
    if (error != MAL_PA_OK) {
        result = mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to connect PulseAudio context.", mal_result_from_pulse(error));
        goto on_error2;
    }


    pDevice->pulse.pulseContextState = MAL_PA_CONTEXT_UNCONNECTED;
    ((mal_pa_context_set_state_callback_proc)pContext->pulse.pa_context_set_state_callback)((mal_pa_context*)pDevice->pulse.pPulseContext, mal_pulse_device_state_callback, pDevice);

    // Wait for PulseAudio to get itself ready before returning.
    for (;;) {
        if (pDevice->pulse.pulseContextState == MAL_PA_CONTEXT_READY) {
            break;
        } else {
            error = ((mal_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)((mal_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);    // 1 = block.
            if (error < 0) {
                result = mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] The PulseAudio main loop returned an error while connecting the PulseAudio context.", mal_result_from_pulse(error));
                goto on_error3;
            }
            continue;
        }

        // An error may have occurred.
        if (pDevice->pulse.pulseContextState == MAL_PA_CONTEXT_FAILED || pDevice->pulse.pulseContextState == MAL_PA_CONTEXT_TERMINATED) {
            result = mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] An error occurred while connecting the PulseAudio context.", MAL_ERROR);
            goto on_error3;
        }

        error = ((mal_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)((mal_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
        if (error < 0) {
            result = mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] The PulseAudio main loop returned an error while connecting the PulseAudio context.", mal_result_from_pulse(error));
            goto on_error3;
        }
    }


    if (type == mal_device_type_playback) {
        pOP = ((mal_pa_context_get_sink_info_by_name_proc)pContext->pulse.pa_context_get_sink_info_by_name)((mal_pa_context*)pDevice->pulse.pPulseContext, dev, mal_device_sink_info_callback, &sinkInfo);
    } else {
        pOP = ((mal_pa_context_get_source_info_by_name_proc)pContext->pulse.pa_context_get_source_info_by_name)((mal_pa_context*)pDevice->pulse.pPulseContext, dev, mal_device_source_info_callback, &sourceInfo);
    }

    if (pOP != NULL) {
        mal_device__wait_for_operation__pulse(pDevice, pOP);
        ((mal_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
    }


#if 0
    mal_pa_sample_spec deviceSS;
    mal_pa_channel_map deviceCMap;
    if (type == mal_device_type_playback) {
        deviceSS = sinkInfo.sample_spec;
        deviceCMap = sinkInfo.channel_map;
    } else {
        deviceSS = sourceInfo.sample_spec;
        deviceCMap = sourceInfo.channel_map;
    }

    if (pDevice->usingDefaultFormat) {
        ss.format = deviceSS.format;
    } else {
        ss.format = mal_format_to_pulse(pConfig->format);
    }
    if (ss.format == MAL_PA_SAMPLE_INVALID) {
        ss.format = MAL_PA_SAMPLE_S16LE;
    }

    if (pDevice->usingDefaultChannels) {
        ss.channels = deviceSS.channels;
    } else {
        ss.channels = pConfig->channels;
    }

    if (pDevice->usingDefaultSampleRate) {
        ss.rate = deviceSS.rate;
    } else {
        ss.rate = pConfig->sampleRate;
    }


    if (pDevice->usingDefaultChannelMap) {
        cmap = deviceCMap;
    } else {
        cmap.channels = pConfig->channels;
        for (mal_uint32 iChannel = 0; iChannel < pConfig->channels; ++iChannel) {
            cmap.map[iChannel] = mal_channel_position_to_pulse(pConfig->channelMap[iChannel]);
        }

        if (((mal_pa_channel_map_valid_proc)pContext->pulse.pa_channel_map_valid)(&cmap) == 0 || ((mal_pa_channel_map_compatible_proc)pContext->pulse.pa_channel_map_compatible)(&cmap, &ss) == 0) {
            ((mal_pa_channel_map_init_extend_proc)pContext->pulse.pa_channel_map_init_extend)(&cmap, ss.channels, MAL_PA_CHANNEL_MAP_DEFAULT);     // The channel map is invalid, so just fall back to the default.
        }
    }
#else
    if (type == mal_device_type_playback) {
        ss = sinkInfo.sample_spec;
        cmap = sinkInfo.channel_map;
    } else {
        ss = sourceInfo.sample_spec;
        cmap = sourceInfo.channel_map;
    }
#endif

    // Buffer size.
    bufferSizeInFrames = pDevice->bufferSizeInFrames;
    if (bufferSizeInFrames == 0) {
        bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, ss.rate);

        // PulseAudio seems to need a bit of an bit of size to the buffer to be reliable.
        if (pDevice->usingDefaultBufferSize) {
            float bufferSizeScaleFactor = 1.0f;
            if (type == mal_device_type_capture) {
                bufferSizeScaleFactor = 2.0f;
            }

            bufferSizeInFrames = mal_scale_buffer_size(bufferSizeInFrames, bufferSizeScaleFactor);
        }
    }

    attr.maxlength = bufferSizeInFrames * mal_get_bytes_per_sample(mal_format_from_pulse(ss.format))*ss.channels;
    attr.tlength   = attr.maxlength / pConfig->periods;
    attr.prebuf    = (mal_uint32)-1;
    attr.minreq    = attr.tlength;
    attr.fragsize  = attr.tlength;

    char streamName[256];
    if (pConfig->pulse.pStreamName != NULL) {
        mal_strncpy_s(streamName, sizeof(streamName), pConfig->pulse.pStreamName, (size_t)-1);
    } else {
        static int g_StreamCounter = 0;
        mal_strcpy_s(streamName, sizeof(streamName), "mini_al:");
        mal_itoa_s(g_StreamCounter, streamName + 8, sizeof(streamName)-8, 10);  // 8 = strlen("mini_al:")
        g_StreamCounter += 1;
    }

    pDevice->pulse.pStream = ((mal_pa_stream_new_proc)pContext->pulse.pa_stream_new)((mal_pa_context*)pDevice->pulse.pPulseContext, streamName, &ss, &cmap);
    if (pDevice->pulse.pStream == NULL) {
        result = mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create PulseAudio stream.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        goto on_error3;
    }



    if (type == mal_device_type_playback) {
        error = ((mal_pa_stream_connect_playback_proc)pContext->pulse.pa_stream_connect_playback)((mal_pa_stream*)pDevice->pulse.pStream, dev, &attr, MAL_PA_STREAM_START_CORKED, NULL, NULL);
    } else {
        error = ((mal_pa_stream_connect_record_proc)pContext->pulse.pa_stream_connect_record)((mal_pa_stream*)pDevice->pulse.pStream, dev, &attr, MAL_PA_STREAM_START_CORKED);
    }

    if (error != MAL_PA_OK) {
        result = mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to connect PulseAudio stream.", mal_result_from_pulse(error));
        goto on_error4;
    }

    while (((mal_pa_stream_get_state_proc)pContext->pulse.pa_stream_get_state)((mal_pa_stream*)pDevice->pulse.pStream) != MAL_PA_STREAM_READY) {
        error = ((mal_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)((mal_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
        if (error < 0) {
            result = mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] The PulseAudio main loop returned an error while connecting the PulseAudio stream.", mal_result_from_pulse(error));
            goto on_error5;
        }
    }


    // Internal format.
    pActualSS = ((mal_pa_stream_get_sample_spec_proc)pContext->pulse.pa_stream_get_sample_spec)((mal_pa_stream*)pDevice->pulse.pStream);
    if (pActualSS != NULL) {
        ss = *pActualSS;
    }

    pDevice->internalFormat = mal_format_from_pulse(ss.format);
    pDevice->internalChannels = ss.channels;
    pDevice->internalSampleRate = ss.rate;


    // Internal channel map.
    pActualCMap = ((mal_pa_stream_get_channel_map_proc)pContext->pulse.pa_stream_get_channel_map)((mal_pa_stream*)pDevice->pulse.pStream);
    if (pActualCMap != NULL) {
        cmap = *pActualCMap;
    }

    for (mal_uint32 iChannel = 0; iChannel < pDevice->internalChannels; ++iChannel) {
        pDevice->internalChannelMap[iChannel] = mal_channel_position_from_pulse(cmap.map[iChannel]);
    }


    // Buffer size.
    pActualAttr = ((mal_pa_stream_get_buffer_attr_proc)pContext->pulse.pa_stream_get_buffer_attr)((mal_pa_stream*)pDevice->pulse.pStream);
    if (pActualAttr != NULL) {
        attr = *pActualAttr;
    }

    pDevice->bufferSizeInFrames = attr.maxlength / (mal_get_bytes_per_sample(pDevice->internalFormat)*pDevice->internalChannels);
    pDevice->periods = attr.maxlength / attr.tlength;


    // Grab the name of the device if we can.
    dev = ((mal_pa_stream_get_device_name_proc)pContext->pulse.pa_stream_get_device_name)((mal_pa_stream*)pDevice->pulse.pStream);
    if (dev != NULL) {
        mal_pa_operation* pOP = NULL;
        if (type == mal_device_type_playback) {
            pOP = ((mal_pa_context_get_sink_info_by_name_proc)pContext->pulse.pa_context_get_sink_info_by_name)((mal_pa_context*)pDevice->pulse.pPulseContext, dev, mal_device_sink_name_callback, pDevice);
        } else {
            pOP = ((mal_pa_context_get_source_info_by_name_proc)pContext->pulse.pa_context_get_source_info_by_name)((mal_pa_context*)pDevice->pulse.pPulseContext, dev, mal_device_source_name_callback, pDevice);
        }

        if (pOP != NULL) {
            mal_device__wait_for_operation__pulse(pDevice, pOP);
            ((mal_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        }
    }


    // Set callbacks for reading and writing data to/from the PulseAudio stream.
    if (type == mal_device_type_playback) {
        ((mal_pa_stream_set_write_callback_proc)pContext->pulse.pa_stream_set_write_callback)((mal_pa_stream*)pDevice->pulse.pStream, mal_pulse_device_write_callback, pDevice);
    } else {
        ((mal_pa_stream_set_read_callback_proc)pContext->pulse.pa_stream_set_read_callback)((mal_pa_stream*)pDevice->pulse.pStream, mal_pulse_device_read_callback, pDevice);
    }


    pDevice->pulse.fragmentSizeInBytes = attr.tlength;

    return MAL_SUCCESS;


on_error5: ((mal_pa_stream_disconnect_proc)pContext->pulse.pa_stream_disconnect)((mal_pa_stream*)pDevice->pulse.pStream);
on_error4: ((mal_pa_stream_unref_proc)pContext->pulse.pa_stream_unref)((mal_pa_stream*)pDevice->pulse.pStream);
on_error3: ((mal_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)((mal_pa_context*)pDevice->pulse.pPulseContext);
on_error2: ((mal_pa_context_unref_proc)pContext->pulse.pa_context_unref)((mal_pa_context*)pDevice->pulse.pPulseContext);
on_error1: ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)((mal_pa_mainloop*)pDevice->pulse.pMainLoop);
on_error0:
    return result;
}


void mal_pulse_operation_complete_callback(mal_pa_stream* pStream, int success, void* pUserData)
{
    mal_bool32* pIsSuccessful = (mal_bool32*)pUserData;
    mal_assert(pIsSuccessful != NULL);

    *pIsSuccessful = (mal_bool32)success;
}

mal_result mal_device__cork_stream__pulse(mal_device* pDevice, int cork)
{
    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    mal_bool32 wasSuccessful = MAL_FALSE;
    mal_pa_operation* pOP = ((mal_pa_stream_cork_proc)pContext->pulse.pa_stream_cork)((mal_pa_stream*)pDevice->pulse.pStream, cork, mal_pulse_operation_complete_callback, &wasSuccessful);
    if (pOP == NULL) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to cork PulseAudio stream.", (cork == 0) ? MAL_FAILED_TO_START_BACKEND_DEVICE : MAL_FAILED_TO_STOP_BACKEND_DEVICE);
    }

    mal_result result = mal_device__wait_for_operation__pulse(pDevice, pOP);
    ((mal_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);

    if (result != MAL_SUCCESS) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] An error occurred while waiting for the PulseAudio stream to cork.", result);
    }

    if (!wasSuccessful) {
        if (cork) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to stop PulseAudio stream.", MAL_FAILED_TO_STOP_BACKEND_DEVICE);
        } else {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to start PulseAudio stream.", MAL_FAILED_TO_START_BACKEND_DEVICE);
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_device__start_backend__pulse(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    // For both playback and capture we need to uncork the stream. Afterwards, for playback we need to fill in an initial chunk
    // of data, equal to the trigger length. That should then start actual playback.
    mal_result result = mal_device__cork_stream__pulse(pDevice, 0);
    if (result != MAL_SUCCESS) {
        return result;
    }

    // A playback device is started by simply writing data to it. For capture we do nothing.
    if (pDevice->type == mal_device_type_playback) {
        // Playback.
        mal_pulse_device_write_callback((mal_pa_stream*)pDevice->pulse.pStream, pDevice->pulse.fragmentSizeInBytes, pDevice);

        // Force an immediate start of the device just to be sure.
        mal_pa_operation* pOP = ((mal_pa_stream_trigger_proc)pContext->pulse.pa_stream_trigger)((mal_pa_stream*)pDevice->pulse.pStream, NULL, NULL);
        if (pOP != NULL) {
             mal_device__wait_for_operation__pulse(pDevice, pOP);
            ((mal_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        }
    } else {
        // Capture. Do nothing.
    }

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__pulse(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    mal_result result = mal_device__cork_stream__pulse(pDevice, 1);
    if (result != MAL_SUCCESS) {
        return result;
    }

    // For playback, buffers need to be flushed. For capture they need to be drained.
    mal_bool32 wasSuccessful;
    mal_pa_operation* pOP = NULL;
    if (pDevice->type == mal_device_type_playback) {
        pOP = ((mal_pa_stream_flush_proc)pContext->pulse.pa_stream_flush)((mal_pa_stream*)pDevice->pulse.pStream, mal_pulse_operation_complete_callback, &wasSuccessful);
    } else {
        pOP = ((mal_pa_stream_drain_proc)pContext->pulse.pa_stream_drain)((mal_pa_stream*)pDevice->pulse.pStream, mal_pulse_operation_complete_callback, &wasSuccessful);
    }

    if (pOP == NULL) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to flush buffers after stopping PulseAudio stream.", MAL_ERROR);
    }

    result = mal_device__wait_for_operation__pulse(pDevice, pOP);
    ((mal_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);

    if (result != MAL_SUCCESS) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] An error occurred while waiting for the PulseAudio stream to flush.", result);
    }

    if (!wasSuccessful) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[PulseAudio] Failed to flush buffers after stopping PulseAudio stream.", MAL_ERROR);
    }

    return MAL_SUCCESS;
}

mal_result mal_device__break_main_loop__pulse(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    pDevice->pulse.breakFromMainLoop = MAL_TRUE;
    ((mal_pa_mainloop_wakeup_proc)pContext->pulse.pa_mainloop_wakeup)((mal_pa_mainloop*)pDevice->pulse.pMainLoop);

    return MAL_SUCCESS;
}

mal_result mal_device__main_loop__pulse(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    pDevice->pulse.breakFromMainLoop = MAL_FALSE;
    while (!pDevice->pulse.breakFromMainLoop) {
        // Break from the main loop if the device isn't started anymore. Likely what's happened is the application
        // has requested that the device be stopped.
        if (!mal_device_is_started(pDevice)) {
            break;
        }

        int resultPA = ((mal_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)((mal_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
        if (resultPA < 0) {
            break;  // Some error occurred.
        }
    }

    return MAL_SUCCESS;
}


mal_result mal_context_uninit__pulse(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_pulseaudio);

#ifndef MAL_NO_RUNTIME_LINKING
    mal_dlclose(pContext->pulse.pulseSO);
#endif

    return MAL_SUCCESS;
}

mal_result mal_context_init__pulse(mal_context* pContext)
{
    mal_assert(pContext != NULL);

#ifndef MAL_NO_RUNTIME_LINKING
    // libpulse.so
    const char* libpulseNames[] = {
        "libpulse.so",
        "libpulse.so.0"
    };

    for (size_t i = 0; i < mal_countof(libpulseNames); ++i) {
        pContext->pulse.pulseSO = mal_dlopen(libpulseNames[i]);
        if (pContext->pulse.pulseSO != NULL) {
            break;
        }
    }

    if (pContext->pulse.pulseSO == NULL) {
        return MAL_NO_BACKEND;
    }

    pContext->pulse.pa_mainloop_new                    = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_mainloop_new");
    pContext->pulse.pa_mainloop_free                   = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_mainloop_free");
    pContext->pulse.pa_mainloop_get_api                = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_mainloop_get_api");
    pContext->pulse.pa_mainloop_iterate                = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_mainloop_iterate");
    pContext->pulse.pa_mainloop_wakeup                 = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_mainloop_wakeup");
    pContext->pulse.pa_context_new                     = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_context_new");
    pContext->pulse.pa_context_unref                   = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_context_unref");
    pContext->pulse.pa_context_connect                 = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_context_connect");
    pContext->pulse.pa_context_disconnect              = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_context_disconnect");
    pContext->pulse.pa_context_set_state_callback      = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_context_set_state_callback");
    pContext->pulse.pa_context_get_state               = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_context_get_state");
    pContext->pulse.pa_context_get_sink_info_list      = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_context_get_sink_info_list");
    pContext->pulse.pa_context_get_source_info_list    = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_context_get_source_info_list");
    pContext->pulse.pa_context_get_sink_info_by_name   = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_context_get_sink_info_by_name");
    pContext->pulse.pa_context_get_source_info_by_name = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_context_get_source_info_by_name");
    pContext->pulse.pa_operation_unref                 = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_operation_unref");
    pContext->pulse.pa_operation_get_state             = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_operation_get_state");
    pContext->pulse.pa_channel_map_init_extend         = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_channel_map_init_extend");
    pContext->pulse.pa_channel_map_valid               = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_channel_map_valid");
    pContext->pulse.pa_channel_map_compatible          = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_channel_map_compatible");
    pContext->pulse.pa_stream_new                      = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_new");
    pContext->pulse.pa_stream_unref                    = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_unref");
    pContext->pulse.pa_stream_connect_playback         = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_connect_playback");
    pContext->pulse.pa_stream_connect_record           = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_connect_record");
    pContext->pulse.pa_stream_disconnect               = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_disconnect");
    pContext->pulse.pa_stream_get_state                = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_get_state");
    pContext->pulse.pa_stream_get_sample_spec          = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_get_sample_spec");
    pContext->pulse.pa_stream_get_channel_map          = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_get_channel_map");
    pContext->pulse.pa_stream_get_buffer_attr          = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_get_buffer_attr");
    pContext->pulse.pa_stream_get_device_name          = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_get_device_name");
    pContext->pulse.pa_stream_set_write_callback       = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_set_write_callback");
    pContext->pulse.pa_stream_set_read_callback        = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_set_read_callback");
    pContext->pulse.pa_stream_flush                    = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_flush");
    pContext->pulse.pa_stream_drain                    = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_drain");
    pContext->pulse.pa_stream_cork                     = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_cork");
    pContext->pulse.pa_stream_trigger                  = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_trigger");
    pContext->pulse.pa_stream_begin_write              = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_begin_write");
    pContext->pulse.pa_stream_write                    = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_write");
    pContext->pulse.pa_stream_peek                     = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_peek");
    pContext->pulse.pa_stream_drop                     = (mal_proc)mal_dlsym(pContext->pulse.pulseSO, "pa_stream_drop");
#else
    // This strange assignment system is just for type safety.
    mal_pa_mainloop_new_proc                    _pa_mainloop_new                   = pa_mainloop_new;
    mal_pa_mainloop_free_proc                   _pa_mainloop_free                  = pa_mainloop_free;
    mal_pa_mainloop_get_api_proc                _pa_mainloop_get_api               = pa_mainloop_get_api;
    mal_pa_mainloop_iterate_proc                _pa_mainloop_iterate               = pa_mainloop_iterate;
    mal_pa_mainloop_wakeup_proc                 _pa_mainloop_wakeup                = pa_mainloop_wakeup;
    mal_pa_context_new_proc                     _pa_context_new                    = pa_context_new;
    mal_pa_context_unref_proc                   _pa_context_unref                  = pa_context_unref;
    mal_pa_context_connect_proc                 _pa_context_connect                = pa_context_connect;
    mal_pa_context_disconnect_proc              _pa_context_disconnect             = pa_context_disconnect;
    mal_pa_context_set_state_callback_proc      _pa_context_set_state_callback     = pa_context_set_state_callback;
    mal_pa_context_get_state_proc               _pa_context_get_state              = pa_context_get_state;
    mal_pa_context_get_sink_info_list_proc      _pa_context_get_sink_info_list     = pa_context_get_sink_info_list;
    mal_pa_context_get_source_info_list_proc    _pa_context_get_source_info_list   = pa_context_get_source_info_list;
    mal_pa_context_get_sink_info_by_name_proc   _pa_context_get_sink_info_by_name  = pa_context_get_sink_info_by_name;
    mal_pa_context_get_source_info_by_name_proc _pa_context_get_source_info_by_name= pa_context_get_source_info_by_name;
    mal_pa_operation_unref_proc                 _pa_operation_unref                = pa_operation_unref;
    mal_pa_operation_get_state_proc             _pa_operation_get_state            = pa_operation_get_state;
    mal_pa_channel_map_init_extend_proc         _pa_channel_map_init_extend        = pa_channel_map_init_extend;
    mal_pa_channel_map_valid_proc               _pa_channel_map_valid              = pa_channel_map_valid;
    mal_pa_channel_map_compatible_proc          _pa_channel_map_compatible         = pa_channel_map_compatible;
    mal_pa_stream_new_proc                      _pa_stream_new                     = pa_stream_new;
    mal_pa_stream_unref_proc                    _pa_stream_unref                   = pa_stream_unref;
    mal_pa_stream_connect_playback_proc         _pa_stream_connect_playback        = pa_stream_connect_playback;
    mal_pa_stream_connect_record_proc           _pa_stream_connect_record          = pa_stream_connect_record;
    mal_pa_stream_disconnect_proc               _pa_stream_disconnect              = pa_stream_disconnect;
    mal_pa_stream_get_state_proc                _pa_stream_get_state               = pa_stream_get_state;
    mal_pa_stream_get_sample_spec_proc          _pa_stream_get_sample_spec         = pa_stream_get_sample_spec;
    mal_pa_stream_get_channel_map_proc          _pa_stream_get_channel_map         = pa_stream_get_channel_map;
    mal_pa_stream_get_buffer_attr_proc          _pa_stream_get_buffer_attr         = pa_stream_get_buffer_attr;
    mal_pa_stream_get_device_name_proc          _pa_stream_get_device_name         = pa_stream_get_device_name;
    mal_pa_stream_set_write_callback_proc       _pa_stream_set_write_callback      = pa_stream_set_write_callback;
    mal_pa_stream_set_read_callback_proc        _pa_stream_set_read_callback       = pa_stream_set_read_callback;
    mal_pa_stream_flush_proc                    _pa_stream_flush                   = pa_stream_flush;
    mal_pa_stream_drain_proc                    _pa_stream_drain                   = pa_stream_drain;
    mal_pa_stream_cork_proc                     _pa_stream_cork                    = pa_stream_cork;
    mal_pa_stream_trigger_proc                  _pa_stream_trigger                 = pa_stream_trigger;
    mal_pa_stream_begin_write_proc              _pa_stream_begin_write             = pa_stream_begin_write;
    mal_pa_stream_write_proc                    _pa_stream_write                   = pa_stream_write;
    mal_pa_stream_peek_proc                     _pa_stream_peek                    = pa_stream_peek;
    mal_pa_stream_drop_proc                     _pa_stream_drop                    = pa_stream_drop;

    pContext->pulse.pa_mainloop_new                    = (mal_proc)_pa_mainloop_new;
    pContext->pulse.pa_mainloop_free                   = (mal_proc)_pa_mainloop_free;
    pContext->pulse.pa_mainloop_get_api                = (mal_proc)_pa_mainloop_get_api;
    pContext->pulse.pa_mainloop_iterate                = (mal_proc)_pa_mainloop_iterate;
    pContext->pulse.pa_mainloop_wakeup                 = (mal_proc)_pa_mainloop_wakeup;
    pContext->pulse.pa_context_new                     = (mal_proc)_pa_context_new;
    pContext->pulse.pa_context_unref                   = (mal_proc)_pa_context_unref;
    pContext->pulse.pa_context_connect                 = (mal_proc)_pa_context_connect;
    pContext->pulse.pa_context_disconnect              = (mal_proc)_pa_context_disconnect;
    pContext->pulse.pa_context_set_state_callback      = (mal_proc)_pa_context_set_state_callback;
    pContext->pulse.pa_context_get_state               = (mal_proc)_pa_context_get_state;
    pContext->pulse.pa_context_get_sink_info_list      = (mal_proc)_pa_context_get_sink_info_list;
    pContext->pulse.pa_context_get_source_info_list    = (mal_proc)_pa_context_get_source_info_list;
    pContext->pulse.pa_context_get_sink_info_by_name   = (mal_proc)_pa_context_get_sink_info_by_name;
    pContext->pulse.pa_context_get_source_info_by_name = (mal_proc)_pa_context_get_source_info_by_name;
    pContext->pulse.pa_operation_unref                 = (mal_proc)_pa_operation_unref;
    pContext->pulse.pa_operation_get_state             = (mal_proc)_pa_operation_get_state;
    pContext->pulse.pa_channel_map_init_extend         = (mal_proc)_pa_channel_map_init_extend;
    pContext->pulse.pa_channel_map_valid               = (mal_proc)_pa_channel_map_valid;
    pContext->pulse.pa_channel_map_compatible          = (mal_proc)_pa_channel_map_compatible;
    pContext->pulse.pa_stream_new                      = (mal_proc)_pa_stream_new;
    pContext->pulse.pa_stream_unref                    = (mal_proc)_pa_stream_unref;
    pContext->pulse.pa_stream_connect_playback         = (mal_proc)_pa_stream_connect_playback;
    pContext->pulse.pa_stream_connect_record           = (mal_proc)_pa_stream_connect_record;
    pContext->pulse.pa_stream_disconnect               = (mal_proc)_pa_stream_disconnect;
    pContext->pulse.pa_stream_get_state                = (mal_proc)_pa_stream_get_state;
    pContext->pulse.pa_stream_get_sample_spec          = (mal_proc)_pa_stream_get_sample_spec;
    pContext->pulse.pa_stream_get_channel_map          = (mal_proc)_pa_stream_get_channel_map;
    pContext->pulse.pa_stream_get_buffer_attr          = (mal_proc)_pa_stream_get_buffer_attr;
    pContext->pulse.pa_stream_get_device_name          = (mal_proc)_pa_stream_get_device_name;
    pContext->pulse.pa_stream_set_write_callback       = (mal_proc)_pa_stream_set_write_callback;
    pContext->pulse.pa_stream_set_read_callback        = (mal_proc)_pa_stream_set_read_callback;
    pContext->pulse.pa_stream_flush                    = (mal_proc)_pa_stream_flush;
    pContext->pulse.pa_stream_drain                    = (mal_proc)_pa_stream_drain;
    pContext->pulse.pa_stream_cork                     = (mal_proc)_pa_stream_cork;
    pContext->pulse.pa_stream_trigger                  = (mal_proc)_pa_stream_trigger;
    pContext->pulse.pa_stream_begin_write              = (mal_proc)_pa_stream_begin_write;
    pContext->pulse.pa_stream_write                    = (mal_proc)_pa_stream_write;
    pContext->pulse.pa_stream_peek                     = (mal_proc)_pa_stream_peek;
    pContext->pulse.pa_stream_drop                     = (mal_proc)_pa_stream_drop;
#endif

    pContext->onUninit              = mal_context_uninit__pulse;
    pContext->onDeviceIDEqual       = mal_context_is_device_id_equal__pulse;
    pContext->onEnumDevices         = mal_context_enumerate_devices__pulse;
    pContext->onGetDeviceInfo       = mal_context_get_device_info__pulse;
    pContext->onDeviceInit          = mal_device_init__pulse;
    pContext->onDeviceUninit        = mal_device_uninit__pulse;
    pContext->onDeviceStart         = mal_device__start_backend__pulse;
    pContext->onDeviceStop          = mal_device__stop_backend__pulse;
    pContext->onDeviceBreakMainLoop = mal_device__break_main_loop__pulse;
    pContext->onDeviceMainLoop      = mal_device__main_loop__pulse;

    
    // Although we have found the libpulse library, it doesn't necessarily mean PulseAudio is useable. We need to initialize
    // and connect a dummy PulseAudio context to test PulseAudio's usability.
    mal_pa_mainloop* pMainLoop = ((mal_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
    if (pMainLoop == NULL) {
        return MAL_NO_BACKEND;
    }

    mal_pa_mainloop_api* pAPI = ((mal_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)(pMainLoop);
    if (pAPI == NULL) {
        ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MAL_NO_BACKEND;
    }

    mal_pa_context* pPulseContext = ((mal_pa_context_new_proc)pContext->pulse.pa_context_new)(pAPI, pContext->config.pulse.pApplicationName);
    if (pPulseContext == NULL) {
        ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MAL_NO_BACKEND;
    }

    int error = ((mal_pa_context_connect_proc)pContext->pulse.pa_context_connect)(pPulseContext, pContext->config.pulse.pServerName, 0, NULL);
    if (error != MAL_PA_OK) {
        ((mal_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
        ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MAL_NO_BACKEND;
    }

    ((mal_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)(pPulseContext);
    ((mal_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
    ((mal_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
    return MAL_SUCCESS;
}
#endif


///////////////////////////////////////////////////////////////////////////////
//
// JACK Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_JACK

// It is assumed jack.h is available when compile-time linking is being used.
#ifdef MAL_NO_RUNTIME_LINKING
#include <jack/jack.h>

typedef jack_nframes_t              mal_jack_nframes_t;
typedef jack_options_t              mal_jack_options_t;
typedef jack_status_t               mal_jack_status_t;
typedef jack_client_t               mal_jack_client_t;
typedef jack_port_t                 mal_jack_port_t;
typedef JackProcessCallback         mal_JackProcessCallback;
typedef JackBufferSizeCallback      mal_JackBufferSizeCallback;
typedef JackShutdownCallback        mal_JackShutdownCallback;
#define MAL_JACK_DEFAULT_AUDIO_TYPE JACK_DEFAULT_AUDIO_TYPE
#define mal_JackNoStartServer       JackNoStartServer
#define mal_JackPortIsInput         JackPortIsInput
#define mal_JackPortIsOutput        JackPortIsOutput
#define mal_JackPortIsPhysical      JackPortIsPhysical
#else
typedef mal_uint32               mal_jack_nframes_t;
typedef int                      mal_jack_options_t;
typedef int                      mal_jack_status_t;
typedef struct mal_jack_client_t mal_jack_client_t;
typedef struct mal_jack_port_t   mal_jack_port_t;
typedef int  (* mal_JackProcessCallback)   (mal_jack_nframes_t nframes, void* arg);
typedef int  (* mal_JackBufferSizeCallback)(mal_jack_nframes_t nframes, void* arg);
typedef void (* mal_JackShutdownCallback)  (void* arg);
#define MAL_JACK_DEFAULT_AUDIO_TYPE "32 bit float mono audio"
#define mal_JackNoStartServer       1
#define mal_JackPortIsInput         1
#define mal_JackPortIsOutput        2
#define mal_JackPortIsPhysical      4
#endif

typedef mal_jack_client_t* (* mal_jack_client_open_proc)             (const char* client_name, mal_jack_options_t options, mal_jack_status_t* status, ...);
typedef int                (* mal_jack_client_close_proc)            (mal_jack_client_t* client);
typedef int                (* mal_jack_client_name_size_proc)        ();
typedef int                (* mal_jack_set_process_callback_proc)    (mal_jack_client_t* client, mal_JackProcessCallback process_callback, void* arg);
typedef int                (* mal_jack_set_buffer_size_callback_proc)(mal_jack_client_t* client, mal_JackBufferSizeCallback bufsize_callback, void* arg);
typedef void               (* mal_jack_on_shutdown_proc)             (mal_jack_client_t* client, mal_JackShutdownCallback function, void* arg);
typedef mal_jack_nframes_t (* mal_jack_get_sample_rate_proc)         (mal_jack_client_t* client);
typedef mal_jack_nframes_t (* mal_jack_get_buffer_size_proc)         (mal_jack_client_t* client);
typedef const char**       (* mal_jack_get_ports_proc)               (mal_jack_client_t* client, const char* port_name_pattern, const char* type_name_pattern, unsigned long flags);
typedef int                (* mal_jack_activate_proc)                (mal_jack_client_t* client);
typedef int                (* mal_jack_deactivate_proc)              (mal_jack_client_t* client);
typedef int                (* mal_jack_connect_proc)                 (mal_jack_client_t* client, const char* source_port, const char* destination_port);
typedef mal_jack_port_t*   (* mal_jack_port_register_proc)           (mal_jack_client_t* client, const char* port_name, const char* port_type, unsigned long flags, unsigned long buffer_size);
typedef const char*        (* mal_jack_port_name_proc)               (const mal_jack_port_t* port);
typedef void*              (* mal_jack_port_get_buffer_proc)         (mal_jack_port_t* port, mal_jack_nframes_t nframes);
typedef void               (* mal_jack_free_proc)                    (void* ptr);

mal_result mal_context_open_client__jack(mal_context* pContext, mal_jack_client_t** ppClient)
{
    mal_assert(pContext != NULL);
    mal_assert(ppClient != NULL);

    if (ppClient) {
        *ppClient = NULL;
    }

    size_t maxClientNameSize = ((mal_jack_client_name_size_proc)pContext->jack.jack_client_name_size)(); // Includes null terminator.

    char clientName[256];
    mal_strncpy_s(clientName, mal_min(sizeof(clientName), maxClientNameSize), (pContext->config.jack.pClientName != NULL) ? pContext->config.jack.pClientName : "mini_al", (size_t)-1);

    mal_jack_status_t status;
    mal_jack_client_t* pClient = ((mal_jack_client_open_proc)pContext->jack.jack_client_open)(clientName, (pContext->config.jack.tryStartServer) ? 0 : mal_JackNoStartServer, &status, NULL);
    if (pClient == NULL) {
        return MAL_FAILED_TO_OPEN_BACKEND_DEVICE;
    }

    if (ppClient) {
        *ppClient = pClient;
    }

    return MAL_SUCCESS;
}

mal_bool32 mal_context_is_device_id_equal__jack(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return pID0->jack == pID1->jack;
}

mal_result mal_context_enumerate_devices__jack(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    mal_bool32 cbResult = MAL_TRUE;

    // Playback.
    if (cbResult) {
        mal_device_info deviceInfo;
        mal_zero_object(&deviceInfo);
        mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
    }

    // Capture.
    if (cbResult) {
        mal_device_info deviceInfo;
        mal_zero_object(&deviceInfo);
        mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
    }

    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info__jack(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);

    (void)pContext;
    (void)shareMode;

    if (pDeviceID != NULL && pDeviceID->jack != 0) {
        return MAL_NO_DEVICE;   // Don't know the device.
    }

    // Name / Description
    if (deviceType == mal_device_type_playback) {
        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    } else {
        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    }

    // Jack only supports f32 and has a specific channel count and sample rate.
    pDeviceInfo->formatCount = 1;
    pDeviceInfo->formats[0] = mal_format_f32;

    // The channel count and sample rate can only be determined by opening the device.
    mal_jack_client_t* pClient;
    mal_result result = mal_context_open_client__jack(pContext, &pClient);
    if (result != MAL_SUCCESS) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[JACK] Failed to open client.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    pDeviceInfo->minSampleRate = ((mal_jack_get_sample_rate_proc)pContext->jack.jack_get_sample_rate)((mal_jack_client_t*)pClient);
    pDeviceInfo->maxSampleRate = pDeviceInfo->minSampleRate;

    pDeviceInfo->minChannels = 0;
    pDeviceInfo->maxChannels = 0;

    const char** ppPorts = ((mal_jack_get_ports_proc)pContext->jack.jack_get_ports)((mal_jack_client_t*)pClient, NULL, NULL, mal_JackPortIsPhysical | ((deviceType == mal_device_type_playback) ? mal_JackPortIsInput : mal_JackPortIsOutput));
    if (ppPorts == NULL) {
        ((mal_jack_client_close_proc)pContext->jack.jack_client_close)((mal_jack_client_t*)pClient);
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[JACK] Failed to query physical ports.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    while (ppPorts[pDeviceInfo->minChannels] != NULL) {
        pDeviceInfo->minChannels += 1;
        pDeviceInfo->maxChannels += 1;
    }

    ((mal_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);
    ((mal_jack_client_close_proc)pContext->jack.jack_client_close)((mal_jack_client_t*)pClient);

    return MAL_SUCCESS;
}


void mal_device_uninit__jack(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    if (pDevice->jack.pClient != NULL) {
        ((mal_jack_client_close_proc)pContext->jack.jack_client_close)((mal_jack_client_t*)pDevice->jack.pClient);
    }
}

void mal_device__jack_shutdown_callback(void* pUserData)
{
    // JACK died. Stop the device.
    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    mal_device_stop(pDevice);
}

int mal_device__jack_buffer_size_callback(mal_jack_nframes_t frameCount, void* pUserData)
{
    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    float* pNewBuffer = (float*)mal_realloc(pDevice->jack.pIntermediaryBuffer, frameCount * (pDevice->internalChannels*mal_get_bytes_per_sample(pDevice->internalFormat)));
    if (pNewBuffer == NULL) {
        return MAL_OUT_OF_MEMORY;
    }

    pDevice->jack.pIntermediaryBuffer = pNewBuffer;
    pDevice->bufferSizeInFrames = frameCount * pDevice->periods;

    return 0;
}

int mal_device__jack_process_callback(mal_jack_nframes_t frameCount, void* pUserData)
{
    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    if (pDevice->type == mal_device_type_playback) {
        mal_device__read_frames_from_client(pDevice, frameCount, pDevice->jack.pIntermediaryBuffer);

        // Channels need to be deinterleaved.
        for (mal_uint32 iChannel = 0; iChannel < pDevice->internalChannels; ++iChannel) {
            float* pDst = (float*)((mal_jack_port_get_buffer_proc)pContext->jack.jack_port_get_buffer)((mal_jack_port_t*)pDevice->jack.pPorts[iChannel], frameCount);
            if (pDst != NULL) {
                const float* pSrc = pDevice->jack.pIntermediaryBuffer + iChannel;
                for (mal_jack_nframes_t iFrame = 0; iFrame < frameCount; ++iFrame) {
                    *pDst = *pSrc;

                    pDst += 1;
                    pSrc += pDevice->internalChannels;
                }
            }
        }
    } else {
        // Channels need to be interleaved.
        for (mal_uint32 iChannel = 0; iChannel < pDevice->internalChannels; ++iChannel) {
            const float* pSrc = (const float*)((mal_jack_port_get_buffer_proc)pContext->jack.jack_port_get_buffer)((mal_jack_port_t*)pDevice->jack.pPorts[iChannel], frameCount);
            if (pSrc != NULL) {
                float* pDst = pDevice->jack.pIntermediaryBuffer + iChannel;
                for (mal_jack_nframes_t iFrame = 0; iFrame < frameCount; ++iFrame) {
                    *pDst = *pSrc;

                    pDst += pDevice->internalChannels;
                    pSrc += 1;
                }
            }
        }

        mal_device__send_frames_to_client(pDevice, frameCount, pDevice->jack.pIntermediaryBuffer);
    }

    return 0;
}

mal_result mal_device_init__jack(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    mal_assert(pContext != NULL);
    mal_assert(pConfig != NULL);
    mal_assert(pDevice != NULL);

    (void)pContext;
    (void)pConfig;

    // Only supporting default devices with JACK.
    if (pDeviceID != NULL && pDeviceID->jack != 0) {
        return MAL_NO_DEVICE;
    }


    // Open the client.
    mal_result result = mal_context_open_client__jack(pContext, (mal_jack_client_t**)&pDevice->jack.pClient);
    if (result != MAL_SUCCESS) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[JACK] Failed to open client.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    // Callbacks.
    if (((mal_jack_set_process_callback_proc)pContext->jack.jack_set_process_callback)((mal_jack_client_t*)pDevice->jack.pClient, mal_device__jack_process_callback, pDevice) != 0) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[JACK] Failed to set process callback.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }
    if (((mal_jack_set_buffer_size_callback_proc)pContext->jack.jack_set_buffer_size_callback)((mal_jack_client_t*)pDevice->jack.pClient, mal_device__jack_buffer_size_callback, pDevice) != 0) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[JACK] Failed to set buffer size callback.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    ((mal_jack_on_shutdown_proc)pContext->jack.jack_on_shutdown)((mal_jack_client_t*)pDevice->jack.pClient, mal_device__jack_shutdown_callback, pDevice);


    // The format is always f32.
    pDevice->internalFormat = mal_format_f32;

    // A port is a channel.
    unsigned long serverPortFlags;
    unsigned long clientPortFlags;
    if (type == mal_device_type_playback) {
        serverPortFlags = mal_JackPortIsInput;
        clientPortFlags = mal_JackPortIsOutput;
    } else {
        serverPortFlags = mal_JackPortIsOutput;
        clientPortFlags = mal_JackPortIsInput;
    }

    const char** ppPorts = ((mal_jack_get_ports_proc)pContext->jack.jack_get_ports)((mal_jack_client_t*)pDevice->jack.pClient, NULL, NULL, mal_JackPortIsPhysical | serverPortFlags);
    if (ppPorts == NULL) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[JACK] Failed to query physical ports.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    pDevice->internalChannels = 0;
    while (ppPorts[pDevice->internalChannels] != NULL) {
        char name[64];
        if (type == mal_device_type_playback) {
            mal_strcpy_s(name, sizeof(name), "playback");
            mal_itoa_s((int)pDevice->internalChannels, name+8, sizeof(name)-8, 10); // 8 = length of "playback"
        } else {
            mal_strcpy_s(name, sizeof(name), "capture");
            mal_itoa_s((int)pDevice->internalChannels, name+7, sizeof(name)-7, 10); // 7 = length of "capture"
        }

        pDevice->jack.pPorts[pDevice->internalChannels] = ((mal_jack_port_register_proc)pContext->jack.jack_port_register)((mal_jack_client_t*)pDevice->jack.pClient, name, MAL_JACK_DEFAULT_AUDIO_TYPE, clientPortFlags, 0);
        if (pDevice->jack.pPorts[pDevice->internalChannels] == NULL) {
            ((mal_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);
            mal_device_uninit__jack(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[JACK] Failed to register ports.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        pDevice->internalChannels += 1;
    }

    ((mal_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);
    ppPorts = NULL;

    // We set the sample rate here, but apparently this can change. This is incompatible with mini_al, so changing sample rates will not be supported.
    pDevice->internalSampleRate = ((mal_jack_get_sample_rate_proc)pContext->jack.jack_get_sample_rate)((mal_jack_client_t*)pDevice->jack.pClient);

    // I don't think the channel map can be queried, so just use defaults for now.
    mal_get_standard_channel_map(mal_standard_channel_map_alsa, pDevice->internalChannels, pDevice->internalChannelMap);

    // The buffer size in frames can change.
    pDevice->periods = 2;
    pDevice->bufferSizeInFrames = ((mal_jack_get_buffer_size_proc)pContext->jack.jack_get_buffer_size)((mal_jack_client_t*)pDevice->jack.pClient) * pDevice->periods;

    // Initial allocation for the intermediary buffer.
    pDevice->jack.pIntermediaryBuffer = (float*)mal_malloc((pDevice->bufferSizeInFrames/pDevice->periods)*(pDevice->internalChannels*mal_get_bytes_per_sample(pDevice->internalFormat)));
    if (pDevice->jack.pIntermediaryBuffer == NULL) {
        mal_device_uninit__jack(pDevice);
        return MAL_OUT_OF_MEMORY;
    }

    return MAL_SUCCESS;
}


mal_result mal_device__start_backend__jack(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    int resultJACK = ((mal_jack_activate_proc)pContext->jack.jack_activate)((mal_jack_client_t*)pDevice->jack.pClient);
    if (resultJACK != 0) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[JACK] Failed to activate the JACK client.", MAL_FAILED_TO_START_BACKEND_DEVICE);
    }

    const char** ppServerPorts;
    if (pDevice->type == mal_device_type_playback) {
        ppServerPorts = ((mal_jack_get_ports_proc)pContext->jack.jack_get_ports)((mal_jack_client_t*)pDevice->jack.pClient, NULL, NULL, mal_JackPortIsPhysical | mal_JackPortIsInput);
    } else {
        ppServerPorts = ((mal_jack_get_ports_proc)pContext->jack.jack_get_ports)((mal_jack_client_t*)pDevice->jack.pClient, NULL, NULL, mal_JackPortIsPhysical | mal_JackPortIsOutput);
    }

    if (ppServerPorts == NULL) {
        ((mal_jack_deactivate_proc)pContext->jack.jack_deactivate)((mal_jack_client_t*)pDevice->jack.pClient);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[JACK] Failed to retrieve physical ports.", MAL_ERROR);
    }

    for (size_t i = 0; ppServerPorts[i] != NULL; ++i) {
        const char* pServerPort = ppServerPorts[i];
        mal_assert(pServerPort != NULL);

        const char* pClientPort = ((mal_jack_port_name_proc)pContext->jack.jack_port_name)((mal_jack_port_t*)pDevice->jack.pPorts[i]);
        mal_assert(pClientPort != NULL);

        if (pDevice->type == mal_device_type_playback) {
            resultJACK = ((mal_jack_connect_proc)pContext->jack.jack_connect)((mal_jack_client_t*)pDevice->jack.pClient, pClientPort, pServerPort);
        } else {
            resultJACK = ((mal_jack_connect_proc)pContext->jack.jack_connect)((mal_jack_client_t*)pDevice->jack.pClient, pServerPort, pClientPort);
        }

        if (resultJACK != 0) {
            ((mal_jack_free_proc)pContext->jack.jack_free)((void*)ppServerPorts);
            ((mal_jack_deactivate_proc)pContext->jack.jack_deactivate)((mal_jack_client_t*)pDevice->jack.pClient);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[JACK] Failed to connect ports.", MAL_ERROR);
        }
    }

    ((mal_jack_free_proc)pContext->jack.jack_free)((void*)ppServerPorts);

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__jack(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    mal_context* pContext = pDevice->pContext;
    mal_assert(pContext != NULL);

    if (((mal_jack_deactivate_proc)pContext->jack.jack_deactivate)((mal_jack_client_t*)pDevice->jack.pClient) != 0) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[JACK] An error occurred when deactivating the JACK client.", MAL_ERROR);
    }
    
    mal_device__set_state(pDevice, MAL_STATE_STOPPED);
    mal_stop_proc onStop = pDevice->onStop;
    if (onStop) {
        onStop(pDevice);
    }

    return MAL_SUCCESS;
}


mal_result mal_context_uninit__jack(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_jack);

#ifndef MAL_NO_RUNTIME_LINKING
    mal_dlclose(pContext->jack.jackSO);
#endif

    return MAL_SUCCESS;
}

mal_result mal_context_init__jack(mal_context* pContext)
{
    mal_assert(pContext != NULL);

#ifndef MAL_NO_RUNTIME_LINKING
    // libjack.so
    const char* libjackNames[] = {
#ifdef MAL_WIN32
        "libjack.dll"
#else
        "libjack.so",
        "libjack.so.0"
#endif
    };

    for (size_t i = 0; i < mal_countof(libjackNames); ++i) {
        pContext->jack.jackSO = mal_dlopen(libjackNames[i]);
        if (pContext->jack.jackSO != NULL) {
            break;
        }
    }

    if (pContext->jack.jackSO == NULL) {
        return MAL_NO_BACKEND;
    }

    pContext->jack.jack_client_open              = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_client_open");
    pContext->jack.jack_client_close             = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_client_close");
    pContext->jack.jack_client_name_size         = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_client_name_size");
    pContext->jack.jack_set_process_callback     = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_set_process_callback");
    pContext->jack.jack_set_buffer_size_callback = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_set_buffer_size_callback");
    pContext->jack.jack_on_shutdown              = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_on_shutdown");
    pContext->jack.jack_get_sample_rate          = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_get_sample_rate");
    pContext->jack.jack_get_buffer_size          = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_get_buffer_size");
    pContext->jack.jack_get_ports                = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_get_ports");
    pContext->jack.jack_activate                 = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_activate");
    pContext->jack.jack_deactivate               = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_deactivate");
    pContext->jack.jack_connect                  = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_connect");
    pContext->jack.jack_port_register            = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_port_register");
    pContext->jack.jack_port_name                = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_port_name");
    pContext->jack.jack_port_get_buffer          = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_port_get_buffer");
    pContext->jack.jack_free                     = (mal_proc)mal_dlsym(pContext->jack.jackSO, "jack_free");
#else
    // This strange assignment system is here just to ensure type safety of mini_al's function pointer
    // types. If anything differs slightly the compiler should throw a warning.
    mal_jack_client_open_proc              _jack_client_open              = jack_client_open;
    mal_jack_client_close_proc             _jack_client_close             = jack_client_close;
    mal_jack_client_name_size_proc         _jack_client_name_size         = jack_client_name_size;
    mal_jack_set_process_callback_proc     _jack_set_process_callback     = jack_set_process_callback;
    mal_jack_set_buffer_size_callback_proc _jack_set_buffer_size_callback = jack_set_buffer_size_callback;
    mal_jack_on_shutdown_proc              _jack_on_shutdown              = jack_on_shutdown;
    mal_jack_get_sample_rate_proc          _jack_get_sample_rate          = jack_get_sample_rate;
    mal_jack_get_buffer_size_proc          _jack_get_buffer_size          = jack_get_buffer_size;
    mal_jack_get_ports_proc                _jack_get_ports                = jack_get_ports;
    mal_jack_activate_proc                 _jack_activate                 = jack_activate;
    mal_jack_deactivate_proc               _jack_deactivate               = jack_deactivate;
    mal_jack_connect_proc                  _jack_connect                  = jack_connect;
    mal_jack_port_register_proc            _jack_port_register            = jack_port_register;
    mal_jack_port_name_proc                _jack_port_name                = jack_port_name;
    mal_jack_port_get_buffer_proc          _jack_port_get_buffer          = jack_port_get_buffer;
    mal_jack_free_proc                     _jack_free                     = jack_free;

    pContext->jack.jack_client_open              = (mal_proc)_jack_client_open;
    pContext->jack.jack_client_close             = (mal_proc)_jack_client_close;
    pContext->jack.jack_client_name_size         = (mal_proc)_jack_client_name_size;
    pContext->jack.jack_set_process_callback     = (mal_proc)_jack_set_process_callback;
    pContext->jack.jack_set_buffer_size_callback = (mal_proc)_jack_set_buffer_size_callback;
    pContext->jack.jack_on_shutdown              = (mal_proc)_jack_on_shutdown;
    pContext->jack.jack_get_sample_rate          = (mal_proc)_jack_get_sample_rate;
    pContext->jack.jack_get_buffer_size          = (mal_proc)_jack_get_buffer_size;
    pContext->jack.jack_get_ports                = (mal_proc)_jack_get_ports;
    pContext->jack.jack_activate                 = (mal_proc)_jack_activate;
    pContext->jack.jack_deactivate               = (mal_proc)_jack_deactivate;
    pContext->jack.jack_connect                  = (mal_proc)_jack_connect;
    pContext->jack.jack_port_register            = (mal_proc)_jack_port_register;
    pContext->jack.jack_port_name                = (mal_proc)_jack_port_name;
    pContext->jack.jack_port_get_buffer          = (mal_proc)_jack_port_get_buffer;
    pContext->jack.jack_free                     = (mal_proc)_jack_free;
#endif

    pContext->isBackendAsynchronous = MAL_TRUE;

    pContext->onUninit        = mal_context_uninit__jack;
    pContext->onDeviceIDEqual = mal_context_is_device_id_equal__jack;
    pContext->onEnumDevices   = mal_context_enumerate_devices__jack;
    pContext->onGetDeviceInfo = mal_context_get_device_info__jack;
    pContext->onDeviceInit    = mal_device_init__jack;
    pContext->onDeviceUninit  = mal_device_uninit__jack;
    pContext->onDeviceStart   = mal_device__start_backend__jack;
    pContext->onDeviceStop    = mal_device__stop_backend__jack;


    // Getting here means the JACK library is installed, but it doesn't necessarily mean it's usable. We need to quickly test this by connecting
    // a temporary client.
    mal_jack_client_t* pDummyClient;
    mal_result result = mal_context_open_client__jack(pContext, &pDummyClient);
    if (result != MAL_SUCCESS) {
        return MAL_NO_BACKEND;
    }

    ((mal_jack_client_close_proc)pContext->jack.jack_client_close)((mal_jack_client_t*)pDummyClient);
    return MAL_SUCCESS;
}
#endif  // JACK



///////////////////////////////////////////////////////////////////////////////
//
// Core Audio Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_COREAUDIO
#include <TargetConditionals.h>

#if defined(TARGET_OS_IPHONE) && TARGET_OS_IPHONE == 1
    #define MAL_APPLE_MOBILE
#else
    #define MAL_APPLE_DESKTOP
#endif

#if defined(MAL_APPLE_DESKTOP)
#include <CoreAudio/CoreAudio.h>
#else
#include <AVFoundation/AVFoundation.h>
#endif

#include <AudioToolbox/AudioToolbox.h>

// CoreFoundation
typedef Boolean (* mal_CFStringGetCString_proc)(CFStringRef theString, char* buffer, CFIndex bufferSize, CFStringEncoding encoding);

// CoreAudio
#if defined(MAL_APPLE_DESKTOP)
typedef OSStatus (* mal_AudioObjectGetPropertyData_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32* ioDataSize, void* outData);
typedef OSStatus (* mal_AudioObjectGetPropertyDataSize_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32* outDataSize);
typedef OSStatus (* mal_AudioObjectSetPropertyData_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32 inDataSize, const void* inData);
typedef OSStatus (* mal_AudioObjectAddPropertyListener_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, AudioObjectPropertyListenerProc inListener, void* inClientData);
#endif

// AudioToolbox
typedef AudioComponent (* mal_AudioComponentFindNext_proc)(AudioComponent inComponent, const AudioComponentDescription* inDesc);
typedef OSStatus (* mal_AudioComponentInstanceDispose_proc)(AudioComponentInstance inInstance);
typedef OSStatus (* mal_AudioComponentInstanceNew_proc)(AudioComponent inComponent, AudioComponentInstance* outInstance);
typedef OSStatus (* mal_AudioOutputUnitStart_proc)(AudioUnit inUnit);
typedef OSStatus (* mal_AudioOutputUnitStop_proc)(AudioUnit inUnit);
typedef OSStatus (* mal_AudioUnitAddPropertyListener_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitPropertyListenerProc inProc, void* inProcUserData);
typedef OSStatus (* mal_AudioUnitGetProperty_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitScope inScope, AudioUnitElement inElement, void* outData, UInt32* ioDataSize);
typedef OSStatus (* mal_AudioUnitSetProperty_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitScope inScope, AudioUnitElement inElement, const void* inData, UInt32 inDataSize);
typedef OSStatus (* mal_AudioUnitInitialize_proc)(AudioUnit inUnit);
typedef OSStatus (* mal_AudioUnitRender_proc)(AudioUnit inUnit, AudioUnitRenderActionFlags* ioActionFlags, const AudioTimeStamp* inTimeStamp, UInt32 inOutputBusNumber, UInt32 inNumberFrames, AudioBufferList* ioData);


#define MAL_COREAUDIO_OUTPUT_BUS    0
#define MAL_COREAUDIO_INPUT_BUS     1

mal_result mal_device_reinit_internal__coreaudio(mal_device* pDevice, mal_bool32 disposePreviousAudioUnit);


// Core Audio
//
// So far, Core Audio has been the worst backend to work with due to being both unintuitive and having almost no documentation
// apart from comments in the headers (which admittedly are quite good). For my own purposes, and for anybody out there whose
// needing to figure out how this darn thing works, I'm going to outline a few things here.
//
// Since mini_al is a fairly low-level API, one of the things it needs is control over specific devices, and it needs to be
// able to identify whether or not it can be used as playback and/or capture. The AudioObject API is the only one I've seen
// that supports this level of detail. There was some public domain sample code I stumbled across that used the AudioComponent
// and AudioUnit APIs, but I couldn't see anything that gave low-level control over device selection and capabilities (the
// distinction between playback and capture in particular). Therefore, mini_al is using the AudioObject API.
//
// Most (all?) functions in the AudioObject API take a AudioObjectID as it's input. This is the device identifier. When
// retrieving global information, such as the device list, you use kAudioObjectSystemObject. When retrieving device-specific
// data, you pass in the ID for that device. In order to retrieve device-specific IDs you need to enumerate over each of the
// devices. This is done using the AudioObjectGetPropertyDataSize() and AudioObjectGetPropertyData() APIs which seem to be
// the central APIs for retrieving information about the system and specific devices.
//
// To use the AudioObjectGetPropertyData() API you need to use the notion of a property address. A property address is a
// structure with three variables and is used to identify which property you are getting or setting. The first is the "selector"
// which is basically the specific property that you're wanting to retrieve or set. The second is the "scope", which is
// typically set to kAudioObjectPropertyScopeGlobal, kAudioObjectPropertyScopeInput for input-specific properties and
// kAudioObjectPropertyScopeOutput for output-specific properties. The last is the "element" which is always set to
// kAudioObjectPropertyElementMaster in mini_al's case. I don't know of any cases where this would be set to anything different.
//
// Back to the earlier issue of device retrieval, you first use the AudioObjectGetPropertyDataSize() API to retrieve the size
// of the raw data which is just a list of AudioDeviceID's. You use the kAudioObjectSystemObject AudioObjectID, and a property
// address with the kAudioHardwarePropertyDevices selector and the kAudioObjectPropertyScopeGlobal scope. Once you have the
// size, allocate a block of memory of that size and then call AudioObjectGetPropertyData(). The data is just a list of
// AudioDeviceID's so just do "dataSize/sizeof(AudioDeviceID)" to know the device count.

mal_result mal_result_from_OSStatus(OSStatus status)
{
    switch (status)
    {
        case noErr:                                   return MAL_SUCCESS;
    #if defined(MAL_APPLE_DESKTOP)
        case kAudioHardwareNotRunningError:           return MAL_DEVICE_NOT_STARTED;
        case kAudioHardwareUnspecifiedError:          return MAL_ERROR;
        case kAudioHardwareUnknownPropertyError:      return MAL_INVALID_ARGS;
        case kAudioHardwareBadPropertySizeError:      return MAL_INVALID_OPERATION;
        case kAudioHardwareIllegalOperationError:     return MAL_INVALID_OPERATION;
        case kAudioHardwareBadObjectError:            return MAL_INVALID_ARGS;
        case kAudioHardwareBadDeviceError:            return MAL_INVALID_ARGS;
        case kAudioHardwareBadStreamError:            return MAL_INVALID_ARGS;
        case kAudioHardwareUnsupportedOperationError: return MAL_INVALID_OPERATION;
        case kAudioDeviceUnsupportedFormatError:      return MAL_FORMAT_NOT_SUPPORTED;
        case kAudioDevicePermissionsError:            return MAL_ACCESS_DENIED;
    #endif
        default:                                      return MAL_ERROR;
    }
}

#if 0
mal_channel mal_channel_from_AudioChannelBitmap(AudioChannelBitmap bit)
{
    switch (bit)
    {
        case kAudioChannelBit_Left:                 return MAL_CHANNEL_LEFT;
        case kAudioChannelBit_Right:                return MAL_CHANNEL_RIGHT;
        case kAudioChannelBit_Center:               return MAL_CHANNEL_FRONT_CENTER;
        case kAudioChannelBit_LFEScreen:            return MAL_CHANNEL_LFE;
        case kAudioChannelBit_LeftSurround:         return MAL_CHANNEL_BACK_LEFT;
        case kAudioChannelBit_RightSurround:        return MAL_CHANNEL_BACK_RIGHT;
        case kAudioChannelBit_LeftCenter:           return MAL_CHANNEL_FRONT_LEFT_CENTER;
        case kAudioChannelBit_RightCenter:          return MAL_CHANNEL_FRONT_RIGHT_CENTER;
        case kAudioChannelBit_CenterSurround:       return MAL_CHANNEL_BACK_CENTER;
        case kAudioChannelBit_LeftSurroundDirect:   return MAL_CHANNEL_SIDE_LEFT;
        case kAudioChannelBit_RightSurroundDirect:  return MAL_CHANNEL_SIDE_RIGHT;
        case kAudioChannelBit_TopCenterSurround:    return MAL_CHANNEL_TOP_CENTER;
        case kAudioChannelBit_VerticalHeightLeft:   return MAL_CHANNEL_TOP_FRONT_LEFT;
        case kAudioChannelBit_VerticalHeightCenter: return MAL_CHANNEL_TOP_FRONT_CENTER;
        case kAudioChannelBit_VerticalHeightRight:  return MAL_CHANNEL_TOP_FRONT_RIGHT;
        case kAudioChannelBit_TopBackLeft:          return MAL_CHANNEL_TOP_BACK_LEFT;
        case kAudioChannelBit_TopBackCenter:        return MAL_CHANNEL_TOP_BACK_CENTER;
        case kAudioChannelBit_TopBackRight:         return MAL_CHANNEL_TOP_BACK_RIGHT;
        default:                                    return MAL_CHANNEL_NONE;
    }
}
#endif

mal_channel mal_channel_from_AudioChannelLabel(AudioChannelLabel label)
{
    switch (label)
    {
        case kAudioChannelLabel_Unknown:              return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_Unused:               return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_UseCoordinates:       return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_Left:                 return MAL_CHANNEL_LEFT;
        case kAudioChannelLabel_Right:                return MAL_CHANNEL_RIGHT;
        case kAudioChannelLabel_Center:               return MAL_CHANNEL_FRONT_CENTER;
        case kAudioChannelLabel_LFEScreen:            return MAL_CHANNEL_LFE;
        case kAudioChannelLabel_LeftSurround:         return MAL_CHANNEL_BACK_LEFT;
        case kAudioChannelLabel_RightSurround:        return MAL_CHANNEL_BACK_RIGHT;
        case kAudioChannelLabel_LeftCenter:           return MAL_CHANNEL_FRONT_LEFT_CENTER;
        case kAudioChannelLabel_RightCenter:          return MAL_CHANNEL_FRONT_RIGHT_CENTER;
        case kAudioChannelLabel_CenterSurround:       return MAL_CHANNEL_BACK_CENTER;
        case kAudioChannelLabel_LeftSurroundDirect:   return MAL_CHANNEL_SIDE_LEFT;
        case kAudioChannelLabel_RightSurroundDirect:  return MAL_CHANNEL_SIDE_RIGHT;
        case kAudioChannelLabel_TopCenterSurround:    return MAL_CHANNEL_TOP_CENTER;
        case kAudioChannelLabel_VerticalHeightLeft:   return MAL_CHANNEL_TOP_FRONT_LEFT;
        case kAudioChannelLabel_VerticalHeightCenter: return MAL_CHANNEL_TOP_FRONT_CENTER;
        case kAudioChannelLabel_VerticalHeightRight:  return MAL_CHANNEL_TOP_FRONT_RIGHT;
        case kAudioChannelLabel_TopBackLeft:          return MAL_CHANNEL_TOP_BACK_LEFT;
        case kAudioChannelLabel_TopBackCenter:        return MAL_CHANNEL_TOP_BACK_CENTER;
        case kAudioChannelLabel_TopBackRight:         return MAL_CHANNEL_TOP_BACK_RIGHT;
        case kAudioChannelLabel_RearSurroundLeft:     return MAL_CHANNEL_BACK_LEFT;
        case kAudioChannelLabel_RearSurroundRight:    return MAL_CHANNEL_BACK_RIGHT;
        case kAudioChannelLabel_LeftWide:             return MAL_CHANNEL_SIDE_LEFT;
        case kAudioChannelLabel_RightWide:            return MAL_CHANNEL_SIDE_RIGHT;
        case kAudioChannelLabel_LFE2:                 return MAL_CHANNEL_LFE;
        case kAudioChannelLabel_LeftTotal:            return MAL_CHANNEL_LEFT;
        case kAudioChannelLabel_RightTotal:           return MAL_CHANNEL_RIGHT;
        case kAudioChannelLabel_HearingImpaired:      return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_Narration:            return MAL_CHANNEL_MONO;
        case kAudioChannelLabel_Mono:                 return MAL_CHANNEL_MONO;
        case kAudioChannelLabel_DialogCentricMix:     return MAL_CHANNEL_MONO;
        case kAudioChannelLabel_CenterSurroundDirect: return MAL_CHANNEL_BACK_CENTER;
        case kAudioChannelLabel_Haptic:               return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_W:          return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_X:          return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_Y:          return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_Z:          return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_MS_Mid:               return MAL_CHANNEL_LEFT;
        case kAudioChannelLabel_MS_Side:              return MAL_CHANNEL_RIGHT;
        case kAudioChannelLabel_XY_X:                 return MAL_CHANNEL_LEFT;
        case kAudioChannelLabel_XY_Y:                 return MAL_CHANNEL_RIGHT;
        case kAudioChannelLabel_HeadphonesLeft:       return MAL_CHANNEL_LEFT;
        case kAudioChannelLabel_HeadphonesRight:      return MAL_CHANNEL_RIGHT;
        case kAudioChannelLabel_ClickTrack:           return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_ForeignLanguage:      return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_Discrete:             return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_Discrete_0:           return MAL_CHANNEL_AUX_0;
        case kAudioChannelLabel_Discrete_1:           return MAL_CHANNEL_AUX_1;
        case kAudioChannelLabel_Discrete_2:           return MAL_CHANNEL_AUX_2;
        case kAudioChannelLabel_Discrete_3:           return MAL_CHANNEL_AUX_3;
        case kAudioChannelLabel_Discrete_4:           return MAL_CHANNEL_AUX_4;
        case kAudioChannelLabel_Discrete_5:           return MAL_CHANNEL_AUX_5;
        case kAudioChannelLabel_Discrete_6:           return MAL_CHANNEL_AUX_6;
        case kAudioChannelLabel_Discrete_7:           return MAL_CHANNEL_AUX_7;
        case kAudioChannelLabel_Discrete_8:           return MAL_CHANNEL_AUX_8;
        case kAudioChannelLabel_Discrete_9:           return MAL_CHANNEL_AUX_9;
        case kAudioChannelLabel_Discrete_10:          return MAL_CHANNEL_AUX_10;
        case kAudioChannelLabel_Discrete_11:          return MAL_CHANNEL_AUX_11;
        case kAudioChannelLabel_Discrete_12:          return MAL_CHANNEL_AUX_12;
        case kAudioChannelLabel_Discrete_13:          return MAL_CHANNEL_AUX_13;
        case kAudioChannelLabel_Discrete_14:          return MAL_CHANNEL_AUX_14;
        case kAudioChannelLabel_Discrete_15:          return MAL_CHANNEL_AUX_15;
        case kAudioChannelLabel_Discrete_65535:       return MAL_CHANNEL_NONE;
        
    #if 0   // Introduced in a later version of macOS.
        case kAudioChannelLabel_HOA_ACN:              return MAL_CHANNEL_NONE;
        case kAudioChannelLabel_HOA_ACN_0:            return MAL_CHANNEL_AUX_0;
        case kAudioChannelLabel_HOA_ACN_1:            return MAL_CHANNEL_AUX_1;
        case kAudioChannelLabel_HOA_ACN_2:            return MAL_CHANNEL_AUX_2;
        case kAudioChannelLabel_HOA_ACN_3:            return MAL_CHANNEL_AUX_3;
        case kAudioChannelLabel_HOA_ACN_4:            return MAL_CHANNEL_AUX_4;
        case kAudioChannelLabel_HOA_ACN_5:            return MAL_CHANNEL_AUX_5;
        case kAudioChannelLabel_HOA_ACN_6:            return MAL_CHANNEL_AUX_6;
        case kAudioChannelLabel_HOA_ACN_7:            return MAL_CHANNEL_AUX_7;
        case kAudioChannelLabel_HOA_ACN_8:            return MAL_CHANNEL_AUX_8;
        case kAudioChannelLabel_HOA_ACN_9:            return MAL_CHANNEL_AUX_9;
        case kAudioChannelLabel_HOA_ACN_10:           return MAL_CHANNEL_AUX_10;
        case kAudioChannelLabel_HOA_ACN_11:           return MAL_CHANNEL_AUX_11;
        case kAudioChannelLabel_HOA_ACN_12:           return MAL_CHANNEL_AUX_12;
        case kAudioChannelLabel_HOA_ACN_13:           return MAL_CHANNEL_AUX_13;
        case kAudioChannelLabel_HOA_ACN_14:           return MAL_CHANNEL_AUX_14;
        case kAudioChannelLabel_HOA_ACN_15:           return MAL_CHANNEL_AUX_15;
        case kAudioChannelLabel_HOA_ACN_65024:        return MAL_CHANNEL_NONE;
    #endif
        
        default:                                      return MAL_CHANNEL_NONE;
    }
}

mal_result mal_format_from_AudioStreamBasicDescription(const AudioStreamBasicDescription* pDescription, mal_format* pFormatOut)
{
    mal_assert(pDescription != NULL);
    mal_assert(pFormatOut != NULL);
    
    *pFormatOut = mal_format_unknown;   // Safety.
    
    // There's a few things mini_al doesn't support.
    if (pDescription->mFormatID != kAudioFormatLinearPCM) {
        return MAL_FORMAT_NOT_SUPPORTED;
    }
    
    // We don't support any non-packed formats that are aligned high.
    if ((pDescription->mFormatFlags & kLinearPCMFormatFlagIsAlignedHigh) != 0) {
        return MAL_FORMAT_NOT_SUPPORTED;
    }

    // Only supporting native-endian.
    if ((mal_is_little_endian() && (pDescription->mFormatFlags & kAudioFormatFlagIsBigEndian) != 0) || (mal_is_big_endian() && (pDescription->mFormatFlags & kAudioFormatFlagIsBigEndian) == 0)) {
        return MAL_FORMAT_NOT_SUPPORTED;
    }
    
    // We are not currently supporting non-interleaved formats (this will be added in a future version of mini_al).
    //if ((pDescription->mFormatFlags & kAudioFormatFlagIsNonInterleaved) != 0) {
    //    return MAL_FORMAT_NOT_SUPPORTED;
    //}

    if ((pDescription->mFormatFlags & kLinearPCMFormatFlagIsFloat) != 0) {
        if (pDescription->mBitsPerChannel == 32) {
            *pFormatOut = mal_format_f32;
            return MAL_SUCCESS;
        }
    } else {
        if ((pDescription->mFormatFlags & kLinearPCMFormatFlagIsSignedInteger) != 0) {
            if (pDescription->mBitsPerChannel == 16) {
                *pFormatOut = mal_format_s16;
                return MAL_SUCCESS;
            } else if (pDescription->mBitsPerChannel == 24) {
                if (pDescription->mBytesPerFrame == (pDescription->mBitsPerChannel/8 * pDescription->mChannelsPerFrame)) {
                    *pFormatOut = mal_format_s24;
                    return MAL_SUCCESS;
                } else {
                    if (pDescription->mBytesPerFrame/pDescription->mChannelsPerFrame == sizeof(mal_int32)) {
                        // TODO: Implement mal_format_s24_32.
                        //*pFormatOut = mal_format_s24_32;
                        //return MAL_SUCCESS;
                        return MAL_FORMAT_NOT_SUPPORTED;
                    }
                }
            } else if (pDescription->mBitsPerChannel == 32) {
                *pFormatOut = mal_format_s32;
                return MAL_SUCCESS;
            }
        } else {
            if (pDescription->mBitsPerChannel == 8) {
                *pFormatOut = mal_format_u8;
                return MAL_SUCCESS;
            }
        }
    }
    
    // Getting here means the format is not supported.
    return MAL_FORMAT_NOT_SUPPORTED;
}

#if defined(MAL_APPLE_DESKTOP)
mal_result mal_get_device_object_ids__coreaudio(mal_context* pContext, UInt32* pDeviceCount, AudioObjectID** ppDeviceObjectIDs) // NOTE: Free the returned buffer with mal_free().
{
    mal_assert(pContext != NULL);
    mal_assert(pDeviceCount != NULL);
    mal_assert(ppDeviceObjectIDs != NULL);
    (void)pContext;

    // Safety.
    *pDeviceCount = 0;
    *ppDeviceObjectIDs = NULL;
    
    AudioObjectPropertyAddress propAddressDevices;
    propAddressDevices.mSelector = kAudioHardwarePropertyDevices;
    propAddressDevices.mScope    = kAudioObjectPropertyScopeGlobal;
    propAddressDevices.mElement  = kAudioObjectPropertyElementMaster;

    UInt32 deviceObjectsDataSize;
    OSStatus status = ((mal_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(kAudioObjectSystemObject, &propAddressDevices, 0, NULL, &deviceObjectsDataSize);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    AudioObjectID* pDeviceObjectIDs = (AudioObjectID*)mal_malloc(deviceObjectsDataSize);
    if (pDeviceObjectIDs == NULL) {
        return MAL_OUT_OF_MEMORY;
    }
    
    status = ((mal_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(kAudioObjectSystemObject, &propAddressDevices, 0, NULL, &deviceObjectsDataSize, pDeviceObjectIDs);
    if (status != noErr) {
        mal_free(pDeviceObjectIDs);
        return mal_result_from_OSStatus(status);
    }
    
    *pDeviceCount = deviceObjectsDataSize / sizeof(AudioObjectID);
    *ppDeviceObjectIDs = pDeviceObjectIDs;
    return MAL_SUCCESS;
}

mal_result mal_get_AudioObject_uid_as_CFStringRef(mal_context* pContext, AudioObjectID objectID, CFStringRef* pUID)
{
    mal_assert(pContext != NULL);

    AudioObjectPropertyAddress propAddress;
    propAddress.mSelector = kAudioDevicePropertyDeviceUID;
    propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;

    UInt32 dataSize = sizeof(*pUID);
    OSStatus status = ((mal_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(objectID, &propAddress, 0, NULL, &dataSize, pUID);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    return MAL_SUCCESS;
}

mal_result mal_get_AudioObject_uid(mal_context* pContext, AudioObjectID objectID, size_t bufferSize, char* bufferOut)
{
    mal_assert(pContext != NULL);

    CFStringRef uid;
    mal_result result = mal_get_AudioObject_uid_as_CFStringRef(pContext, objectID, &uid);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    if (!((mal_CFStringGetCString_proc)pContext->coreaudio.CFStringGetCString)(uid, bufferOut, bufferSize, kCFStringEncodingUTF8)) {
        return MAL_ERROR;
    }
    
    return MAL_SUCCESS;
}

mal_result mal_get_AudioObject_name(mal_context* pContext, AudioObjectID objectID, size_t bufferSize, char* bufferOut)
{
    mal_assert(pContext != NULL);

    AudioObjectPropertyAddress propAddress;
    propAddress.mSelector = kAudioDevicePropertyDeviceNameCFString;
    propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;

    CFStringRef deviceName = NULL;
    UInt32 dataSize = sizeof(deviceName);
    OSStatus status = ((mal_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(objectID, &propAddress, 0, NULL, &dataSize, &deviceName);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    if (!((mal_CFStringGetCString_proc)pContext->coreaudio.CFStringGetCString)(deviceName, bufferOut, bufferSize, kCFStringEncodingUTF8)) {
        return MAL_ERROR;
    }
    
    return MAL_SUCCESS;
}

mal_bool32 mal_does_AudioObject_support_scope(mal_context* pContext, AudioObjectID deviceObjectID, AudioObjectPropertyScope scope)
{
    mal_assert(pContext != NULL);

    // To know whether or not a device is an input device we need ot look at the stream configuration. If it has an output channel it's a
    // playback device.
    AudioObjectPropertyAddress propAddress;
    propAddress.mSelector = kAudioDevicePropertyStreamConfiguration;
    propAddress.mScope    = scope;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    UInt32 dataSize;
    OSStatus status = ((mal_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return MAL_FALSE;
    }
    
    AudioBufferList* pBufferList = (AudioBufferList*)mal_malloc(dataSize);
    if (pBufferList == NULL) {
        return MAL_FALSE;   // Out of memory.
    }
    
    status = ((mal_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pBufferList);
    if (status != noErr) {
        mal_free(pBufferList);
        return MAL_FALSE;
    }

    mal_bool32 isSupported = MAL_FALSE;
    if (pBufferList->mNumberBuffers > 0) {
        isSupported = MAL_TRUE;
    }
    
    mal_free(pBufferList);
    return isSupported;
}

mal_bool32 mal_does_AudioObject_support_playback(mal_context* pContext, AudioObjectID deviceObjectID)
{
    return mal_does_AudioObject_support_scope(pContext, deviceObjectID, kAudioObjectPropertyScopeOutput);
}

mal_bool32 mal_does_AudioObject_support_capture(mal_context* pContext, AudioObjectID deviceObjectID)
{
    return mal_does_AudioObject_support_scope(pContext, deviceObjectID, kAudioObjectPropertyScopeInput);
}


mal_result mal_get_AudioObject_stream_descriptions(mal_context* pContext, AudioObjectID deviceObjectID, mal_device_type deviceType, UInt32* pDescriptionCount, AudioStreamRangedDescription** ppDescriptions)    // NOTE: Free the returned pointer with mal_free().
{
    mal_assert(pContext != NULL);
    mal_assert(pDescriptionCount != NULL);
    mal_assert(ppDescriptions != NULL);
    
    // TODO: Experiment with kAudioStreamPropertyAvailablePhysicalFormats instead of (or in addition to) kAudioStreamPropertyAvailableVirtualFormats. My
    //       MacBook Pro uses s24/32 format, however, which mini_al does not currently support.
    AudioObjectPropertyAddress propAddress;
    propAddress.mSelector = kAudioStreamPropertyAvailableVirtualFormats; //kAudioStreamPropertyAvailablePhysicalFormats;
    propAddress.mScope    = (deviceType == mal_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    UInt32 dataSize;
    OSStatus status = ((mal_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    AudioStreamRangedDescription* pDescriptions = (AudioStreamRangedDescription*)mal_malloc(dataSize);
    if (pDescriptions == NULL) {
        return MAL_OUT_OF_MEMORY;
    }
    
    status = ((mal_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pDescriptions);
    if (status != noErr) {
        mal_free(pDescriptions);
        return mal_result_from_OSStatus(status);
    }
    
    *pDescriptionCount = dataSize / sizeof(*pDescriptions);
    *ppDescriptions = pDescriptions;
    return MAL_SUCCESS;
}



mal_result mal_get_AudioObject_channel_layout(mal_context* pContext, AudioObjectID deviceObjectID, mal_device_type deviceType, AudioChannelLayout** ppChannelLayout)   // NOTE: Free the returned pointer with mal_free().
{
    mal_assert(pContext != NULL);
    mal_assert(ppChannelLayout != NULL);
    
    *ppChannelLayout = NULL;    // Safety.
    
    AudioObjectPropertyAddress propAddress;
    propAddress.mSelector = kAudioDevicePropertyPreferredChannelLayout;
    propAddress.mScope    = (deviceType == mal_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    UInt32 dataSize;
    OSStatus status = ((mal_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    AudioChannelLayout* pChannelLayout = (AudioChannelLayout*)mal_malloc(dataSize);
    if (pChannelLayout == NULL) {
        return MAL_OUT_OF_MEMORY;
    }
    
    status = ((mal_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pChannelLayout);
    if (status != noErr) {
        mal_free(pChannelLayout);
        return mal_result_from_OSStatus(status);
    }
    
    *ppChannelLayout = pChannelLayout;
    return MAL_SUCCESS;
}

mal_result mal_get_AudioObject_channel_count(mal_context* pContext, AudioObjectID deviceObjectID, mal_device_type deviceType, mal_uint32* pChannelCount)
{
    mal_assert(pContext != NULL);
    mal_assert(pChannelCount != NULL);
    
    *pChannelCount = 0; // Safety.

    AudioChannelLayout* pChannelLayout;
    mal_result result = mal_get_AudioObject_channel_layout(pContext, deviceObjectID, deviceType, &pChannelLayout);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelDescriptions) {
        *pChannelCount = pChannelLayout->mNumberChannelDescriptions;
    } else if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelBitmap) {
        *pChannelCount = mal_count_set_bits(pChannelLayout->mChannelBitmap);
    } else {
        *pChannelCount = AudioChannelLayoutTag_GetNumberOfChannels(pChannelLayout->mChannelLayoutTag);
    }
    
    mal_free(pChannelLayout);
    return MAL_SUCCESS;
}

mal_result mal_get_channel_map_from_AudioChannelLayout(AudioChannelLayout* pChannelLayout, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    mal_assert(pChannelLayout != NULL);
    
    if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelDescriptions) {
        for (UInt32 iChannel = 0; iChannel < pChannelLayout->mNumberChannelDescriptions; ++iChannel) {
            channelMap[iChannel] = mal_channel_from_AudioChannelLabel(pChannelLayout->mChannelDescriptions[iChannel].mChannelLabel);
        }
    } else
#if 0
    if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelBitmap) {
        // This is the same kind of system that's used by Windows audio APIs.
        UInt32 iChannel = 0;
        AudioChannelBitmap bitmap = pChannelLayout->mChannelBitmap;
        for (UInt32 iBit = 0; iBit < 32; ++iBit) {
            AudioChannelBitmap bit = bitmap & (1 << iBit);
            if (bit != 0) {
                channelMap[iChannel++] = mal_channel_from_AudioChannelBit(bit);
            }
        }
    } else
#endif
    {
        // Need to use the tag to determine the channel map. For now I'm just assuming a default channel map, but later on this should
        // be updated to determine the mapping based on the tag.
        UInt32 channelCount = AudioChannelLayoutTag_GetNumberOfChannels(pChannelLayout->mChannelLayoutTag);
        switch (pChannelLayout->mChannelLayoutTag)
        {
            case kAudioChannelLayoutTag_Mono:
            case kAudioChannelLayoutTag_Stereo:
            case kAudioChannelLayoutTag_StereoHeadphones:
            case kAudioChannelLayoutTag_MatrixStereo:
            case kAudioChannelLayoutTag_MidSide:
            case kAudioChannelLayoutTag_XY:
            case kAudioChannelLayoutTag_Binaural:
            case kAudioChannelLayoutTag_Ambisonic_B_Format:
            {
                mal_get_standard_channel_map(mal_standard_channel_map_default, channelCount, channelMap);
            } break;
            
            case kAudioChannelLayoutTag_Octagonal:
            {
                channelMap[7] = MAL_CHANNEL_SIDE_RIGHT;
                channelMap[6] = MAL_CHANNEL_SIDE_LEFT;
            } // Intentional fallthrough.
            case kAudioChannelLayoutTag_Hexagonal:
            {
                channelMap[5] = MAL_CHANNEL_BACK_CENTER;
            } // Intentional fallthrough.
            case kAudioChannelLayoutTag_Pentagonal:
            {
                channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
            } // Intentional fallghrough.
            case kAudioChannelLayoutTag_Quadraphonic:
            {
                channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
                channelMap[2] = MAL_CHANNEL_BACK_LEFT;
                channelMap[1] = MAL_CHANNEL_RIGHT;
                channelMap[0] = MAL_CHANNEL_LEFT;
            } break;
            
            // TODO: Add support for more tags here.
        
            default:
            {
                mal_get_standard_channel_map(mal_standard_channel_map_default, channelCount, channelMap);
            } break;
        }
    }
    
    return MAL_SUCCESS;
}

mal_result mal_get_AudioObject_channel_map(mal_context* pContext, AudioObjectID deviceObjectID, mal_device_type deviceType, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    mal_assert(pContext != NULL);
    
    AudioChannelLayout* pChannelLayout;
    mal_result result = mal_get_AudioObject_channel_layout(pContext, deviceObjectID, deviceType, &pChannelLayout);
    if (result != MAL_SUCCESS) {
        return result;  // Rather than always failing here, would it be more robust to simply assume a default?
    }
    
    result = mal_get_channel_map_from_AudioChannelLayout(pChannelLayout, channelMap);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    return result;
}

mal_result mal_get_AudioObject_sample_rates(mal_context* pContext, AudioObjectID deviceObjectID, mal_device_type deviceType, UInt32* pSampleRateRangesCount, AudioValueRange** ppSampleRateRanges)   // NOTE: Free the returned pointer with mal_free().
{
    mal_assert(pContext != NULL);
    mal_assert(pSampleRateRangesCount != NULL);
    mal_assert(ppSampleRateRanges != NULL);
  
    // Safety.
    *pSampleRateRangesCount = 0;
    *ppSampleRateRanges = NULL;
    
    AudioObjectPropertyAddress propAddress;
    propAddress.mSelector = kAudioDevicePropertyAvailableNominalSampleRates;
    propAddress.mScope    = (deviceType == mal_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    UInt32 dataSize;
    OSStatus status = ((mal_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    AudioValueRange* pSampleRateRanges = (AudioValueRange*)mal_malloc(dataSize);
    if (pSampleRateRanges == NULL) {
        return MAL_OUT_OF_MEMORY;
    }
    
    status = ((mal_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pSampleRateRanges);
    if (status != noErr) {
        mal_free(pSampleRateRanges);
        return mal_result_from_OSStatus(status);
    }
    
    *pSampleRateRangesCount = dataSize / sizeof(*pSampleRateRanges);
    *ppSampleRateRanges = pSampleRateRanges;
    return MAL_SUCCESS;
}

mal_result mal_get_AudioObject_get_closest_sample_rate(mal_context* pContext, AudioObjectID deviceObjectID, mal_device_type deviceType, mal_uint32 sampleRateIn, mal_uint32* pSampleRateOut)
{
    mal_assert(pContext != NULL);
    mal_assert(pSampleRateOut != NULL);
    
    *pSampleRateOut = 0;    // Safety.
    
    UInt32 sampleRateRangeCount;
    AudioValueRange* pSampleRateRanges;
    mal_result result = mal_get_AudioObject_sample_rates(pContext, deviceObjectID, deviceType, &sampleRateRangeCount, &pSampleRateRanges);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    if (sampleRateRangeCount == 0) {
        mal_free(pSampleRateRanges);
        return MAL_ERROR;   // Should never hit this case should we?
    }
    
    if (sampleRateIn == 0) {
        // Search in order of mini_al's preferred priority.
        for (UInt32 iMALSampleRate = 0; iMALSampleRate < mal_countof(g_malStandardSampleRatePriorities); ++iMALSampleRate) {
            mal_uint32 malSampleRate = g_malStandardSampleRatePriorities[iMALSampleRate];
            for (UInt32 iCASampleRate = 0; iCASampleRate < sampleRateRangeCount; ++iCASampleRate) {
                AudioValueRange caSampleRate = pSampleRateRanges[iCASampleRate];
                if (caSampleRate.mMinimum <= malSampleRate && caSampleRate.mMaximum >= malSampleRate) {
                    *pSampleRateOut = malSampleRate;
                    mal_free(pSampleRateRanges);
                    return MAL_SUCCESS;
                }
            }
        }
        
        // If we get here it means none of mini_al's standard sample rates matched any of the supported sample rates from the device. In this
        // case we just fall back to the first one reported by Core Audio.
        mal_assert(sampleRateRangeCount > 0);
        
        *pSampleRateOut = pSampleRateRanges[0].mMinimum;
        mal_free(pSampleRateRanges);
        return MAL_SUCCESS;
    } else {
        // Find the closest match to this sample rate.
        UInt32 currentAbsoluteDifference = INT32_MAX;
        UInt32 iCurrentClosestRange = (UInt32)-1;
        for (UInt32 iRange = 0; iRange < sampleRateRangeCount; ++iRange) {
            if (pSampleRateRanges[iRange].mMinimum <= sampleRateIn && pSampleRateRanges[iRange].mMaximum >= sampleRateIn) {
                *pSampleRateOut = sampleRateIn;
                mal_free(pSampleRateRanges);
                return MAL_SUCCESS;
            } else {
                UInt32 absoluteDifference;
                if (pSampleRateRanges[iRange].mMinimum > sampleRateIn) {
                    absoluteDifference = pSampleRateRanges[iRange].mMinimum - sampleRateIn;
                } else {
                    absoluteDifference = sampleRateIn - pSampleRateRanges[iRange].mMaximum;
                }
                
                if (currentAbsoluteDifference > absoluteDifference) {
                    currentAbsoluteDifference = absoluteDifference;
                    iCurrentClosestRange = iRange;
                }
            }
        }
        
        mal_assert(iCurrentClosestRange != (UInt32)-1);
        
        *pSampleRateOut = pSampleRateRanges[iCurrentClosestRange].mMinimum;
        mal_free(pSampleRateRanges);
        return MAL_SUCCESS;
    }
    
    // Should never get here, but it would mean we weren't able to find any suitable sample rates.
    //mal_free(pSampleRateRanges);
    //return MAL_ERROR;
}


mal_result mal_get_AudioObject_closest_buffer_size_in_frames(mal_context* pContext, AudioObjectID deviceObjectID, mal_device_type deviceType, mal_uint32 bufferSizeInFramesIn, mal_uint32* pBufferSizeInFramesOut)
{
    mal_assert(pContext != NULL);
    mal_assert(pBufferSizeInFramesOut != NULL);
    
    *pBufferSizeInFramesOut = 0;    // Safety.
    
    AudioObjectPropertyAddress propAddress;
    propAddress.mSelector = kAudioDevicePropertyBufferFrameSizeRange;
    propAddress.mScope    = (deviceType == mal_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;

    AudioValueRange bufferSizeRange;
    UInt32 dataSize = sizeof(bufferSizeRange);
    OSStatus status = ((mal_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, &bufferSizeRange);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    // This is just a clamp.
    if (bufferSizeInFramesIn < bufferSizeRange.mMinimum) {
        *pBufferSizeInFramesOut = (mal_uint32)bufferSizeRange.mMinimum;
    } else if (bufferSizeInFramesIn > bufferSizeRange.mMaximum) {
        *pBufferSizeInFramesOut = (mal_uint32)bufferSizeRange.mMaximum;
    } else {
        *pBufferSizeInFramesOut = bufferSizeInFramesIn;
    }

    return MAL_SUCCESS;
}

mal_result mal_set_AudioObject_buffer_size_in_frames(mal_context* pContext, AudioObjectID deviceObjectID, mal_device_type deviceType, mal_uint32* pBufferSizeInOut)
{
    mal_assert(pContext != NULL);

    mal_uint32 chosenBufferSizeInFrames;
    mal_result result = mal_get_AudioObject_closest_buffer_size_in_frames(pContext, deviceObjectID, deviceType, *pBufferSizeInOut, &chosenBufferSizeInFrames);
    if (result != MAL_SUCCESS) {
        return result;
    }

    // Try setting the size of the buffer... If this fails we just use whatever is currently set.
    AudioObjectPropertyAddress propAddress;
    propAddress.mSelector = kAudioDevicePropertyBufferFrameSize;
    propAddress.mScope    = (deviceType == mal_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    ((mal_AudioObjectSetPropertyData_proc)pContext->coreaudio.AudioObjectSetPropertyData)(deviceObjectID, &propAddress, 0, NULL, sizeof(chosenBufferSizeInFrames), &chosenBufferSizeInFrames);
    
    // Get the actual size of the buffer.
    UInt32 dataSize = sizeof(*pBufferSizeInOut);
    OSStatus status = ((mal_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, &chosenBufferSizeInFrames);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    *pBufferSizeInOut = chosenBufferSizeInFrames;
    return MAL_SUCCESS;
}


mal_result mal_find_AudioObjectID(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, AudioObjectID* pDeviceObjectID)
{
    mal_assert(pContext != NULL);
    mal_assert(pDeviceObjectID != NULL);

    // Safety.
    *pDeviceObjectID = 0;
    
    if (pDeviceID == NULL) {
        // Default device.
        AudioObjectPropertyAddress propAddressDefaultDevice;
        propAddressDefaultDevice.mScope = kAudioObjectPropertyScopeGlobal;
        propAddressDefaultDevice.mElement = kAudioObjectPropertyElementMaster;
        if (type == mal_device_type_playback) {
            propAddressDefaultDevice.mSelector = kAudioHardwarePropertyDefaultOutputDevice;
        } else {
            propAddressDefaultDevice.mSelector = kAudioHardwarePropertyDefaultInputDevice;
        }
        
        UInt32 defaultDeviceObjectIDSize = sizeof(AudioObjectID);
        AudioObjectID defaultDeviceObjectID;
        OSStatus status = ((mal_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(kAudioObjectSystemObject, &propAddressDefaultDevice, 0, NULL, &defaultDeviceObjectIDSize, &defaultDeviceObjectID);
        if (status == noErr) {
            *pDeviceObjectID = defaultDeviceObjectID;
            return MAL_SUCCESS;
        }
    } else {
        // Explicit device.
        UInt32 deviceCount;
        AudioObjectID* pDeviceObjectIDs;
        mal_result result = mal_get_device_object_ids__coreaudio(pContext, &deviceCount, &pDeviceObjectIDs);
        if (result != MAL_SUCCESS) {
            return result;
        }
        
        for (UInt32 iDevice = 0; iDevice < deviceCount; ++iDevice) {
            AudioObjectID deviceObjectID = pDeviceObjectIDs[iDevice];
            
            char uid[256];
            if (mal_get_AudioObject_uid(pContext, deviceObjectID, sizeof(uid), uid) != MAL_SUCCESS) {
                continue;
            }
            
            if (type == mal_device_type_playback) {
                if (mal_does_AudioObject_support_playback(pContext, deviceObjectID)) {
                    if (strcmp(uid, pDeviceID->coreaudio) == 0) {
                        *pDeviceObjectID = deviceObjectID;
                        return MAL_SUCCESS;
                    }
                }
            } else {
                if (mal_does_AudioObject_support_capture(pContext, deviceObjectID)) {
                    if (strcmp(uid, pDeviceID->coreaudio) == 0) {
                        *pDeviceObjectID = deviceObjectID;
                        return MAL_SUCCESS;
                    }
                }
            }
        }
    }
    
    // If we get here it means we couldn't find the device.
    return MAL_NO_DEVICE;
}


mal_result mal_find_best_format__coreaudio(mal_context* pContext, AudioObjectID deviceObjectID, mal_device_type deviceType, mal_format format, mal_uint32 channels, mal_uint32 sampleRate, mal_bool32 usingDefaultFormat, mal_bool32 usingDefaultChannels, mal_bool32 usingDefaultSampleRate, AudioStreamBasicDescription* pFormat)
{
    UInt32 deviceFormatDescriptionCount;
    AudioStreamRangedDescription* pDeviceFormatDescriptions;
    mal_result result = mal_get_AudioObject_stream_descriptions(pContext, deviceObjectID, deviceType, &deviceFormatDescriptionCount, &pDeviceFormatDescriptions);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    mal_uint32 desiredSampleRate = sampleRate;
    if (usingDefaultSampleRate) {
        // When using the device's default sample rate, we get the highest priority standard rate supported by the device. Otherwise
        // we just use the pre-set rate.
        for (mal_uint32 iStandardRate = 0; iStandardRate < mal_countof(g_malStandardSampleRatePriorities); ++iStandardRate) {
            mal_uint32 standardRate = g_malStandardSampleRatePriorities[iStandardRate];
            
            mal_bool32 foundRate = MAL_FALSE;
            for (UInt32 iDeviceRate = 0; iDeviceRate < deviceFormatDescriptionCount; ++iDeviceRate) {
                mal_uint32 deviceRate = (mal_uint32)pDeviceFormatDescriptions[iDeviceRate].mFormat.mSampleRate;
                
                if (deviceRate == standardRate) {
                    desiredSampleRate = standardRate;
                    foundRate = MAL_TRUE;
                    break;
                }
            }
            
            if (foundRate) {
                break;
            }
        }
    }
    
    mal_uint32 desiredChannelCount = channels;
    if (usingDefaultChannels) {
        mal_get_AudioObject_channel_count(pContext, deviceObjectID, deviceType, &desiredChannelCount);    // <-- Not critical if this fails.
    }
    
    mal_format desiredFormat = format;
    if (usingDefaultFormat) {
        desiredFormat = g_malFormatPriorities[0];
    }
    
    // If we get here it means we don't have an exact match to what the client is asking for. We'll need to find the closest one. The next
    // loop will check for formats that have the same sample rate to what we're asking for. If there is, we prefer that one in all cases.
    AudioStreamBasicDescription bestDeviceFormatSoFar;
    mal_zero_object(&bestDeviceFormatSoFar);
    
    mal_bool32 hasSupportedFormat = MAL_FALSE;
    for (UInt32 iFormat = 0; iFormat < deviceFormatDescriptionCount; ++iFormat) {
        mal_format format;
        mal_result formatResult = mal_format_from_AudioStreamBasicDescription(&pDeviceFormatDescriptions[iFormat].mFormat, &format);
        if (formatResult == MAL_SUCCESS && format != mal_format_unknown) {
            hasSupportedFormat = MAL_TRUE;
            bestDeviceFormatSoFar = pDeviceFormatDescriptions[iFormat].mFormat;
            break;
        }
    }
    
    if (!hasSupportedFormat) {
        return MAL_FORMAT_NOT_SUPPORTED;
    }
    
    
    for (UInt32 iFormat = 0; iFormat < deviceFormatDescriptionCount; ++iFormat) {
        AudioStreamBasicDescription thisDeviceFormat = pDeviceFormatDescriptions[iFormat].mFormat;
    
        // If the format is not supported by mini_al we need to skip this one entirely.
        mal_format thisSampleFormat;
        mal_result formatResult = mal_format_from_AudioStreamBasicDescription(&pDeviceFormatDescriptions[iFormat].mFormat, &thisSampleFormat);
        if (formatResult != MAL_SUCCESS || thisSampleFormat == mal_format_unknown) {
            continue;   // The format is not supported by mini_al. Skip.
        }
        
        mal_format bestSampleFormatSoFar;
        mal_format_from_AudioStreamBasicDescription(&bestDeviceFormatSoFar, &bestSampleFormatSoFar);
        
    
        // Getting here means the format is supported by mini_al which makes this format a candidate.
        if (thisDeviceFormat.mSampleRate != desiredSampleRate) {
            // The sample rate does not match, but this format could still be usable, although it's a very low priority. If the best format
            // so far has an equal sample rate we can just ignore this one.
            if (bestDeviceFormatSoFar.mSampleRate == desiredSampleRate) {
                continue;   // The best sample rate so far has the same sample rate as what we requested which means it's still the best so far. Skip this format.
            } else {
                // In this case, neither the best format so far nor this one have the same sample rate. Check the channel count next.
                if (thisDeviceFormat.mChannelsPerFrame != desiredChannelCount) {
                    // This format has a different sample rate _and_ a different channel count.
                    if (bestDeviceFormatSoFar.mChannelsPerFrame == desiredChannelCount) {
                        continue;   // No change to the best format.
                    } else {
                        // Both this format and the best so far have different sample rates and different channel counts. Whichever has the
                        // best format is the new best.
                        if (mal_get_format_priority_index(thisSampleFormat) < mal_get_format_priority_index(bestSampleFormatSoFar)) {
                            bestDeviceFormatSoFar = thisDeviceFormat;
                            continue;
                        } else {
                            continue;   // No change to the best format.
                        }
                    }
                } else {
                    // This format has a different sample rate but the desired channel count.
                    if (bestDeviceFormatSoFar.mChannelsPerFrame == desiredChannelCount) {
                        // Both this format and the best so far have the desired channel count. Whichever has the best format is the new best.
                        if (mal_get_format_priority_index(thisSampleFormat) < mal_get_format_priority_index(bestSampleFormatSoFar)) {
                            bestDeviceFormatSoFar = thisDeviceFormat;
                            continue;
                        } else {
                            continue;   // No change to the best format for now.
                        }
                    } else {
                        // This format has the desired channel count, but the best so far does not. We have a new best.
                        bestDeviceFormatSoFar = thisDeviceFormat;
                        continue;
                    }
                }
            }
        } else {
            // The sample rates match which makes this format a very high priority contender. If the best format so far has a different
            // sample rate it needs to be replaced with this one.
            if (bestDeviceFormatSoFar.mSampleRate != desiredSampleRate) {
                bestDeviceFormatSoFar = thisDeviceFormat;
                continue;
            } else {
                // In this case both this format and the best format so far have the same sample rate. Check the channel count next.
                if (thisDeviceFormat.mChannelsPerFrame == desiredChannelCount) {
                    // In this case this format has the same channel count as what the client is requesting. If the best format so far has
                    // a different count, this one becomes the new best.
                    if (bestDeviceFormatSoFar.mChannelsPerFrame != desiredChannelCount) {
                        bestDeviceFormatSoFar = thisDeviceFormat;
                        continue;
                    } else {
                        // In this case both this format and the best so far have the ideal sample rate and channel count. Check the format.
                        if (thisSampleFormat == desiredFormat) {
                            bestDeviceFormatSoFar = thisDeviceFormat;
                            break;  // Found the exact match.
                        } else {
                            // The formats are different. The new best format is the one with the highest priority format according to mini_al.
                            if (mal_get_format_priority_index(thisSampleFormat) < mal_get_format_priority_index(bestSampleFormatSoFar)) {
                                bestDeviceFormatSoFar = thisDeviceFormat;
                                continue;
                            } else {
                                continue;   // No change to the best format for now.
                            }
                        }
                    }
                } else {
                    // In this case the channel count is different to what the client has requested. If the best so far has the same channel
                    // count as the requested count then it remains the best.
                    if (bestDeviceFormatSoFar.mChannelsPerFrame == desiredChannelCount) {
                        continue;
                    } else {
                        // This is the case where both have the same sample rate (good) but different channel counts. Right now both have about
                        // the same priority, but we need to compare the format now.
                        if (thisSampleFormat == bestSampleFormatSoFar) {
                            if (mal_get_format_priority_index(thisSampleFormat) < mal_get_format_priority_index(bestSampleFormatSoFar)) {
                                bestDeviceFormatSoFar = thisDeviceFormat;
                                continue;
                            } else {
                                continue;   // No change to the best format for now.
                            }
                        }
                    }
                }
            }
        }
    }
    
    *pFormat = bestDeviceFormatSoFar;
    return MAL_SUCCESS;
}
#endif



mal_bool32 mal_context_is_device_id_equal__coreaudio(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return strcmp(pID0->coreaudio, pID1->coreaudio) == 0;
}

mal_result mal_context_enumerate_devices__coreaudio(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);
    
#if defined(MAL_APPLE_DESKTOP)
    UInt32 deviceCount;
    AudioObjectID* pDeviceObjectIDs;
    mal_result result = mal_get_device_object_ids__coreaudio(pContext, &deviceCount, &pDeviceObjectIDs);
    if (result != MAL_SUCCESS) {
        return result;
    }
  
    for (UInt32 iDevice = 0; iDevice < deviceCount; ++iDevice) {
        AudioObjectID deviceObjectID = pDeviceObjectIDs[iDevice];

        mal_device_info info;
        mal_zero_object(&info);
        if (mal_get_AudioObject_uid(pContext, deviceObjectID, sizeof(info.id.coreaudio), info.id.coreaudio) != MAL_SUCCESS) {
            continue;
        }
        if (mal_get_AudioObject_name(pContext, deviceObjectID, sizeof(info.name), info.name) != MAL_SUCCESS) {
            continue;
        }

        if (mal_does_AudioObject_support_playback(pContext, deviceObjectID)) {
            if (!callback(pContext, mal_device_type_playback, &info, pUserData)) {
                break;
            }
        }
        if (mal_does_AudioObject_support_capture(pContext, deviceObjectID)) {
            if (!callback(pContext, mal_device_type_capture, &info, pUserData)) {
                break;
            }
        }
    }
    
    mal_free(pDeviceObjectIDs);
#else
    // Only supporting default devices on non-Desktop platforms.
    mal_device_info info;
    
    mal_zero_object(&info);
    mal_strncpy_s(info.name, sizeof(info.name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    if (!callback(pContext, mal_device_type_playback, &info, pUserData)) {
        return MAL_SUCCESS;
    }
    
    mal_zero_object(&info);
    mal_strncpy_s(info.name, sizeof(info.name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    if (!callback(pContext, mal_device_type_capture, &info, pUserData)) {
        return MAL_SUCCESS;
    }
#endif
    
    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info__coreaudio(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    (void)shareMode;
    (void)pDeviceInfo;
    
#if defined(MAL_APPLE_DESKTOP)
    // Desktop
    // =======
    AudioObjectID deviceObjectID;
    mal_result result = mal_find_AudioObjectID(pContext, deviceType, pDeviceID, &deviceObjectID);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    result = mal_get_AudioObject_uid(pContext, deviceObjectID, sizeof(pDeviceInfo->id.coreaudio), pDeviceInfo->id.coreaudio);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    result = mal_get_AudioObject_name(pContext, deviceObjectID, sizeof(pDeviceInfo->name), pDeviceInfo->name);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    // Formats.
    UInt32 streamDescriptionCount;
    AudioStreamRangedDescription* pStreamDescriptions;
    result = mal_get_AudioObject_stream_descriptions(pContext, deviceObjectID, deviceType, &streamDescriptionCount, &pStreamDescriptions);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    for (UInt32 iStreamDescription = 0; iStreamDescription < streamDescriptionCount; ++iStreamDescription) {
        mal_format format;
        result = mal_format_from_AudioStreamBasicDescription(&pStreamDescriptions[iStreamDescription].mFormat, &format);
        if (result != MAL_SUCCESS) {
            continue;
        }
        
        mal_assert(format != mal_format_unknown);
        
        // Make sure the format isn't already in the output list.
        mal_bool32 exists = MAL_FALSE;
        for (mal_uint32 iOutputFormat = 0; iOutputFormat < pDeviceInfo->formatCount; ++iOutputFormat) {
            if (pDeviceInfo->formats[iOutputFormat] == format) {
                exists = MAL_TRUE;
                break;
            }
        }
        
        if (!exists) {
            pDeviceInfo->formats[pDeviceInfo->formatCount++] = format;
        }
    }
    
    mal_free(pStreamDescriptions);
    
    
    // Channels.
    result = mal_get_AudioObject_channel_count(pContext, deviceObjectID, deviceType, &pDeviceInfo->minChannels);
    if (result != MAL_SUCCESS) {
        return result;
    }
    pDeviceInfo->maxChannels = pDeviceInfo->minChannels;
    
    
    // Sample rates.
    UInt32 sampleRateRangeCount;
    AudioValueRange* pSampleRateRanges;
    result = mal_get_AudioObject_sample_rates(pContext, deviceObjectID, deviceType, &sampleRateRangeCount, &pSampleRateRanges);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    if (sampleRateRangeCount > 0) {
        pDeviceInfo->minSampleRate = UINT32_MAX;
        pDeviceInfo->maxSampleRate = 0;
        for (UInt32 iSampleRate = 0; iSampleRate < sampleRateRangeCount; ++iSampleRate) {
            if (pDeviceInfo->minSampleRate > pSampleRateRanges[iSampleRate].mMinimum) {
                pDeviceInfo->minSampleRate = pSampleRateRanges[iSampleRate].mMinimum;
            }
            if (pDeviceInfo->maxSampleRate < pSampleRateRanges[iSampleRate].mMaximum) {
                pDeviceInfo->maxSampleRate = pSampleRateRanges[iSampleRate].mMaximum;
            }
        }
    }
#else
    // Mobile
    // ======
    if (deviceType == mal_device_type_playback) {
        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    } else {
        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    }
    
    // Retrieving device information is more annoying on mobile than desktop. For simplicity I'm locking this down to whatever format is
    // reported on a temporary I/O unit. The problem, however, is that this doesn't return a value for the sample rate which we need to
    // retrieve from the AVAudioSession shared instance.
    AudioComponentDescription desc;
    desc.componentType = kAudioUnitType_Output;
    desc.componentSubType = kAudioUnitSubType_RemoteIO;
    desc.componentManufacturer = kAudioUnitManufacturer_Apple;
    desc.componentFlags = 0;
    desc.componentFlagsMask = 0;
    
    AudioComponent component = ((mal_AudioComponentFindNext_proc)pContext->coreaudio.AudioComponentFindNext)(NULL, &desc);
    if (component == NULL) {
        return MAL_FAILED_TO_INIT_BACKEND;
    }
    
    AudioUnit audioUnit;
    OSStatus status = ((mal_AudioComponentInstanceNew_proc)pContext->coreaudio.AudioComponentInstanceNew)(component, &audioUnit);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    AudioUnitScope   formatScope   = (deviceType == mal_device_type_playback) ? kAudioUnitScope_Input : kAudioUnitScope_Output;
    AudioUnitElement formatElement = (deviceType == mal_device_type_playback) ? MAL_COREAUDIO_OUTPUT_BUS : MAL_COREAUDIO_INPUT_BUS;
    
    AudioStreamBasicDescription bestFormat;
    UInt32 propSize = sizeof(bestFormat);
    status = ((mal_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, &propSize);
    if (status != noErr) {
        ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(audioUnit);
        return mal_result_from_OSStatus(status);
    }
    
    ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(audioUnit);
    audioUnit = NULL;
    
    
    pDeviceInfo->minChannels = bestFormat.mChannelsPerFrame;
    pDeviceInfo->maxChannels = bestFormat.mChannelsPerFrame;
    
    pDeviceInfo->formatCount = 1;
    mal_result result = mal_format_from_AudioStreamBasicDescription(&bestFormat, &pDeviceInfo->formats[0]);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    // It looks like Apple are wanting to push the whole AVAudioSession thing. Thus, we need to use that to determine device settings. To do
    // this we just get the shared instance and inspect.
    @autoreleasepool {
        AVAudioSession* pAudioSession = [AVAudioSession sharedInstance];
        mal_assert(pAudioSession != NULL);

        pDeviceInfo->minSampleRate = (mal_uint32)pAudioSession.sampleRate;
        pDeviceInfo->maxSampleRate = pDeviceInfo->minSampleRate;
    }
#endif
    
    return MAL_SUCCESS;
}


void mal_device_uninit__coreaudio(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);
    mal_assert(mal_device__get_state(pDevice) == MAL_STATE_UNINITIALIZED);
    
    ((mal_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnit);
    
    if (pDevice->coreaudio.pAudioBufferList) {
        mal_free(pDevice->coreaudio.pAudioBufferList);
    }
}


OSStatus mal_on_output__coreaudio(void* pUserData, AudioUnitRenderActionFlags* pActionFlags, const AudioTimeStamp* pTimeStamp, UInt32 busNumber, UInt32 frameCount, AudioBufferList* pBufferList)
{
    (void)pActionFlags;
    (void)pTimeStamp;
    (void)busNumber;

    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);
    
#if defined(MAL_DEBUG_OUTPUT)
    printf("INFO: Output Callback: busNumber=%d, frameCount=%d, mNumberBuffers=%d\n", busNumber, frameCount, pBufferList->mNumberBuffers);
#endif
    
    // For now we can assume everything is interleaved.
    for (UInt32 iBuffer = 0; iBuffer < pBufferList->mNumberBuffers; ++iBuffer) {
        if (pBufferList->mBuffers[iBuffer].mNumberChannels == pDevice->internalChannels) {
            mal_uint32 frameCountForThisBuffer = pBufferList->mBuffers[iBuffer].mDataByteSize / mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels);
            if (frameCountForThisBuffer > 0) {
                mal_device__read_frames_from_client(pDevice, frameCountForThisBuffer, pBufferList->mBuffers[iBuffer].mData);
            }
            
        #if defined(MAL_DEBUG_OUTPUT)
            printf("  frameCount=%d, mNumberChannels=%d, mDataByteSize=%d\n", frameCount, pBufferList->mBuffers[iBuffer].mNumberChannels, pBufferList->mBuffers[iBuffer].mDataByteSize);
        #endif
        } else {
            // This case is where the number of channels in the output buffer do not match our internal channels. It could mean that it's
            // not interleaved, in which case we can't handle right now since mini_al does not yet support non-interleaved streams. We just
            // output silence here.
            mal_zero_memory(pBufferList->mBuffers[iBuffer].mData, pBufferList->mBuffers[iBuffer].mDataByteSize);

        #if defined(MAL_DEBUG_OUTPUT)
            printf("  WARNING: Outputting silence. frameCount=%d, mNumberChannels=%d, mDataByteSize=%d\n", frameCount, pBufferList->mBuffers[iBuffer].mNumberChannels, pBufferList->mBuffers[iBuffer].mDataByteSize);
        #endif
        }
    }
    
    return noErr;
}

OSStatus mal_on_input__coreaudio(void* pUserData, AudioUnitRenderActionFlags* pActionFlags, const AudioTimeStamp* pTimeStamp, UInt32 busNumber, UInt32 frameCount, AudioBufferList* pUnusedBufferList)
{
    (void)pActionFlags;
    (void)pTimeStamp;
    (void)busNumber;
    (void)frameCount;
    (void)pUnusedBufferList;
    
    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);
    
    AudioBufferList* pRenderedBufferList = (AudioBufferList*)pDevice->coreaudio.pAudioBufferList;
    mal_assert(pRenderedBufferList);
    
#if defined(MAL_DEBUG_OUTPUT)
    printf("INFO: Input Callback: busNumber=%d, frameCount=%d, mNumberBuffers=%d\n", busNumber, frameCount, pRenderedBufferList->mNumberBuffers);
#endif
    
    OSStatus status = ((mal_AudioUnitRender_proc)pDevice->pContext->coreaudio.AudioUnitRender)((AudioUnit)pDevice->coreaudio.audioUnit, pActionFlags, pTimeStamp, busNumber, frameCount, pRenderedBufferList);
    if (status != noErr) {
    #if defined(MAL_DEBUG_OUTPUT)
        printf("  ERROR: AudioUnitRender() failed with %d\n", status);
    #endif
        return status;
    }
    
    // For now we can assume everything is interleaved.
    for (UInt32 iBuffer = 0; iBuffer < pRenderedBufferList->mNumberBuffers; ++iBuffer) {
        if (pRenderedBufferList->mBuffers[iBuffer].mNumberChannels == pDevice->internalChannels) {
            mal_device__send_frames_to_client(pDevice, frameCount, pRenderedBufferList->mBuffers[iBuffer].mData);
        #if defined(MAL_DEBUG_OUTPUT)
            printf("  mDataByteSize=%d\n", pRenderedBufferList->mBuffers[iBuffer].mDataByteSize);
        #endif
        } else {
            // This case is where the number of channels in the output buffer do not match our internal channels. It could mean that it's
            // not interleaved, in which case we can't handle right now since mini_al does not yet support non-interleaved streams.
        }
    }

    return noErr;
}

void on_start_stop__coreaudio(void* pUserData, AudioUnit audioUnit, AudioUnitPropertyID propertyID, AudioUnitScope scope, AudioUnitElement element)
{
    (void)propertyID;
    
    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);
    
    // There's been a report of a deadlock here when triggered by mal_device_uninit(). It looks like
    // AudioUnitGetProprty (called below) and AudioComponentInstanceDispose (called in mal_device_uninit)
    // can try waiting on the same lock. I'm going to try working around this by not calling any Core
    // Audio APIs in the callback when the device has been stopped or initialized.
    if (mal_device__get_state(pDevice) == MAL_STATE_UNINITIALIZED || mal_device__get_state(pDevice) == MAL_STATE_STOPPING) {
        mal_stop_proc onStop = pDevice->onStop;
        if (onStop) {
            onStop(pDevice);
        }
    } else {
        UInt32 isRunning;
        UInt32 isRunningSize = sizeof(isRunning);
        OSStatus status = ((mal_AudioUnitGetProperty_proc)pDevice->pContext->coreaudio.AudioUnitGetProperty)(audioUnit, kAudioOutputUnitProperty_IsRunning, scope, element, &isRunning, &isRunningSize);
        if (status != noErr) {
            return; // Don't really know what to do in this case... just ignore it, I suppose...
        }
        
        if (!isRunning) {
            // The stop event is a bit annoying in Core Audio because it will be called when we automatically switch the default device. Some scenarios to consider:
            //
            // 1) When the device is unplugged, this will be called _before_ the default device change notification.
            // 2) When the device is changed via the default device change notification, this will be called _after_ the switch.
            //
            // For case #1, we just check if there's a new default device available. If so, we just ignore the stop event. For case #2 we check a flag.
            if (pDevice->isDefaultDevice && mal_device__get_state(pDevice) != MAL_STATE_STOPPING && mal_device__get_state(pDevice) != MAL_STATE_STOPPED) {
                // It looks like the device is switching through an external event, such as the user unplugging the device or changing the default device
                // via the operating system's sound settings. If we're re-initializing the device, we just terminate because we want the stopping of the
                // device to be seamless to the client (we don't want them receiving the onStop event and thinking that the device has stopped when it
                // hasn't!).
                if (pDevice->coreaudio.isSwitchingDevice) {
                    return;
                }
                
                // Getting here means the device is not reinitializing which means it may have been unplugged. From what I can see, it looks like Core Audio
                // will try switching to the new default device seamlessly. We need to somehow find a way to determine whether or not Core Audio will most
                // likely be successful in switching to the new device.
                //
                // TODO: Try to predict if Core Audio will switch devices. If not, the onStop callback needs to be posted.
                return;
            }
            
            // Getting here means we need to stop the device.
            mal_stop_proc onStop = pDevice->onStop;
            if (onStop) {
                onStop(pDevice);
            }
        }
    }
}

#if defined(MAL_APPLE_DESKTOP)
OSStatus mal_default_output_device_changed__coreaudio(AudioObjectID objectID, UInt32 addressCount, const AudioObjectPropertyAddress* pAddresses, void* pUserData)
{
    (void)objectID;

    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);
    
    if (pDevice->isDefaultDevice) {
        // Not sure if I really need to check this, but it makes me feel better.
        if (addressCount == 0) {
            return noErr;
        }
    
        if ((pDevice->type == mal_device_type_playback && pAddresses[0].mSelector == kAudioHardwarePropertyDefaultOutputDevice) ||
            (pDevice->type == mal_device_type_capture  && pAddresses[0].mSelector == kAudioHardwarePropertyDefaultInputDevice)) {
#ifdef MAL_DEBUG_OUTPUT
            printf("Device Changed: addressCount=%d, pAddresses[0].mElement=%d\n", addressCount, pAddresses[0].mElement);
#endif
            pDevice->coreaudio.isSwitchingDevice = MAL_TRUE;
            mal_result reinitResult = mal_device_reinit_internal__coreaudio(pDevice, MAL_TRUE);
            pDevice->coreaudio.isSwitchingDevice = MAL_FALSE;
            
            if (reinitResult == MAL_SUCCESS) {
                mal_device__post_init_setup(pDevice);
                
                // Make sure we resume the device if applicable.
                if (mal_device__get_state(pDevice) == MAL_STATE_STARTED) {
                    mal_result startResult = pDevice->pContext->onDeviceStart(pDevice);
                    if (startResult != MAL_SUCCESS) {
                        mal_device__set_state(pDevice, MAL_STATE_STOPPED);
                    }
                }
            }
        }
    }
    
    return noErr;
}
#endif

typedef struct
{
    // Input.
    mal_format formatIn;
    mal_uint32 channelsIn;
    mal_uint32 sampleRateIn;
    mal_channel channelMapIn[MAL_MAX_CHANNELS];
    mal_uint32 bufferSizeInFramesIn;
    mal_uint32 bufferSizeInMillisecondsIn;
    mal_uint32 periodsIn;
    mal_bool32 usingDefaultFormat;
    mal_bool32 usingDefaultChannels;
    mal_bool32 usingDefaultSampleRate;
    mal_bool32 usingDefaultChannelMap;
    mal_share_mode shareMode;

    // Output.
#if defined(MAL_APPLE_DESKTOP)
    AudioObjectID deviceObjectID;
#endif
    AudioComponent component;
    AudioUnit audioUnit;
    AudioBufferList* pAudioBufferList;  // Only used for input devices.
    mal_format formatOut;
    mal_uint32 channelsOut;
    mal_uint32 sampleRateOut;
    mal_channel channelMapOut[MAL_MAX_CHANNELS];
    mal_uint32 bufferSizeInFramesOut;
    mal_uint32 periodsOut;
    mal_bool32 exclusiveMode;
    char deviceName[256];
} mal_device_init_internal_data__coreaudio;

mal_result mal_device_init_internal__coreaudio(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_device_init_internal_data__coreaudio* pData, void* pDevice_DoNotReference)   /* <-- pDevice is typed as void* intentionally so as to avoid accidentally referencing it. */
{
    mal_assert(pContext != NULL);
    mal_assert(deviceType == mal_device_type_playback || deviceType == mal_device_type_capture);
    
#if defined(MAL_APPLE_DESKTOP)
    pData->deviceObjectID = 0;
#endif
    pData->component = NULL;
    pData->audioUnit = NULL;
    pData->pAudioBufferList = NULL;
    
    mal_result result;
    
#if defined(MAL_APPLE_DESKTOP)
    AudioObjectID deviceObjectID;
    result = mal_find_AudioObjectID(pContext, deviceType, pDeviceID, &deviceObjectID);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    pData->deviceObjectID = deviceObjectID;
#endif
    
    // Core audio doesn't really use the notion of a period so we can leave this unmodified, but not too over the top.
    pData->periodsOut = pData->periodsIn;
    if (pData->periodsOut < 1) {
        pData->periodsOut = 1;
    }
    if (pData->periodsOut > 16) {
        pData->periodsOut = 16;
    }
    

    // Audio component.
    AudioComponentDescription desc;
    desc.componentType = kAudioUnitType_Output;
#if defined(MAL_APPLE_DESKTOP)
    desc.componentSubType = kAudioUnitSubType_HALOutput;
#else
    desc.componentSubType = kAudioUnitSubType_RemoteIO;
#endif
    desc.componentManufacturer = kAudioUnitManufacturer_Apple;
    desc.componentFlags = 0;
    desc.componentFlagsMask = 0;
    
    pData->component = ((mal_AudioComponentFindNext_proc)pContext->coreaudio.AudioComponentFindNext)(NULL, &desc);
    if (pData->component == NULL) {
        return MAL_FAILED_TO_INIT_BACKEND;
    }
    
    
    // Audio unit.
    OSStatus status = ((mal_AudioComponentInstanceNew_proc)pContext->coreaudio.AudioComponentInstanceNew)(pData->component, (AudioUnit*)&pData->audioUnit);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    
    // The input/output buses need to be explicitly enabled and disabled. We set the flag based on the output unit first, then we just swap it for input.
    UInt32 enableIOFlag = 1;
    if (deviceType == mal_device_type_capture) {
        enableIOFlag = 0;
    }
    
    status = ((mal_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_EnableIO, kAudioUnitScope_Output, MAL_COREAUDIO_OUTPUT_BUS, &enableIOFlag, sizeof(enableIOFlag));
    if (status != noErr) {
        ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return mal_result_from_OSStatus(status);
    }
    
    enableIOFlag = (enableIOFlag == 0) ? 1 : 0;
    status = ((mal_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_EnableIO, kAudioUnitScope_Input, MAL_COREAUDIO_INPUT_BUS, &enableIOFlag, sizeof(enableIOFlag));
    if (status != noErr) {
        ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return mal_result_from_OSStatus(status);
    }
    
    
    // Set the device to use with this audio unit. This is only used on desktop since we are using defaults on mobile.
#if defined(MAL_APPLE_DESKTOP)
    status = ((mal_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_CurrentDevice, kAudioUnitScope_Global, (deviceType == mal_device_type_playback) ? MAL_COREAUDIO_OUTPUT_BUS : MAL_COREAUDIO_INPUT_BUS, &deviceObjectID, sizeof(AudioDeviceID));
    if (status != noErr) {
        ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return mal_result_from_OSStatus(result);
    }
#endif
    
    // Format. This is the hardest part of initialization because there's a few variables to take into account.
    //   1) The format must be supported by the device.
    //   2) The format must be supported mini_al.
    //   3) There's a priority that mini_al prefers.
    //
    // Ideally we would like to use a format that's as close to the hardware as possible so we can get as close to a passthrough as possible. The
    // most important property is the sample rate. mini_al can do format conversion for any sample rate and channel count, but cannot do the same
    // for the sample data format. If the sample data format is not supported by mini_al it must be ignored completely.
    //
    // On mobile platforms this is a bit different. We just force the use of whatever the audio unit's current format is set to.
    AudioStreamBasicDescription bestFormat;
    {
        AudioUnitScope   formatScope   = (deviceType == mal_device_type_playback) ? kAudioUnitScope_Input : kAudioUnitScope_Output;
        AudioUnitElement formatElement = (deviceType == mal_device_type_playback) ? MAL_COREAUDIO_OUTPUT_BUS : MAL_COREAUDIO_INPUT_BUS;
    
    #if defined(MAL_APPLE_DESKTOP)
        result = mal_find_best_format__coreaudio(pContext, deviceObjectID, deviceType, pData->formatIn, pData->channelsIn, pData->sampleRateIn, pData->usingDefaultFormat, pData->usingDefaultChannels, pData->usingDefaultSampleRate, &bestFormat);
        if (result != MAL_SUCCESS) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return result;
        }
        
        // From what I can see, Apple's documentation implies that we should keep the sample rate consistent.
        AudioStreamBasicDescription origFormat;
        UInt32 origFormatSize = sizeof(origFormat);
        if (deviceType == mal_device_type_playback) {
            status = ((mal_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Output, MAL_COREAUDIO_OUTPUT_BUS, &origFormat, &origFormatSize);
        } else {
            status = ((mal_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Input, MAL_COREAUDIO_INPUT_BUS, &origFormat, &origFormatSize);
        }
        
        if (status != noErr) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return result;
        }
        
        bestFormat.mSampleRate = origFormat.mSampleRate;
        
        status = ((mal_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, sizeof(bestFormat));
        if (status != noErr) {
            // We failed to set the format, so fall back to the current format of the audio unit.
            bestFormat = origFormat;
        }
    #else
        UInt32 propSize = sizeof(bestFormat);
        status = ((mal_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, &propSize);
        if (status != noErr) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return mal_result_from_OSStatus(status);
        }
        
        // Sample rate is a little different here because for some reason kAudioUnitProperty_StreamFormat returns 0... Oh well. We need to instead try
        // setting the sample rate to what the user has requested and then just see the results of it. Need to use some Objective-C here for this since
        // it depends on Apple's AVAudioSession API. To do this we just get the shared AVAudioSession instance and then set it. Note that from what I
        // can tell, it looks like the sample rate is shared between playback and capture for everything.
        @autoreleasepool {
            AVAudioSession* pAudioSession = [AVAudioSession sharedInstance];
            mal_assert(pAudioSession != NULL);
            
            [pAudioSession setPreferredSampleRate:(double)pData->sampleRateIn error:nil];
            bestFormat.mSampleRate = pAudioSession.sampleRate;
        }
        
        status = ((mal_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, sizeof(bestFormat));
        if (status != noErr) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return mal_result_from_OSStatus(status);
        }
    #endif
        
        result = mal_format_from_AudioStreamBasicDescription(&bestFormat, &pData->formatOut);
        if (result != MAL_SUCCESS) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return result;
        }
        
        if (pData->formatOut == mal_format_unknown) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return MAL_FORMAT_NOT_SUPPORTED;
        }
        
        pData->channelsOut = bestFormat.mChannelsPerFrame;
        pData->sampleRateOut = bestFormat.mSampleRate;
    }
    
    
    // Internal channel map.
#if defined(MAL_APPLE_DESKTOP)
    result = mal_get_AudioObject_channel_map(pContext, deviceObjectID, deviceType, pData->channelMapOut);
    if (result != MAL_SUCCESS) {
        return result;
    }
#else
    // TODO: Figure out how to get the channel map using AVAudioSession.
    mal_get_standard_channel_map(mal_standard_channel_map_default, pData->channelsOut, pData->channelMapOut);
#endif
    
    
    // Buffer size. Not allowing this to be configurable on iOS.
    mal_uint32 actualBufferSizeInFrames = pData->bufferSizeInFramesIn;
    
#if defined(MAL_APPLE_DESKTOP)
    if (actualBufferSizeInFrames == 0) {
        actualBufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pData->bufferSizeInMillisecondsIn, pData->sampleRateOut);
    }
    
    actualBufferSizeInFrames = actualBufferSizeInFrames / pData->periodsOut;
    result = mal_set_AudioObject_buffer_size_in_frames(pContext, deviceObjectID, deviceType, &actualBufferSizeInFrames);
    if (result != MAL_SUCCESS) {
        return result;
    }
#else
    actualBufferSizeInFrames = 4096;
#endif

    pData->bufferSizeInFramesOut = actualBufferSizeInFrames * pData->periodsOut;
    
    // During testing I discovered that the buffer size can be too big. You'll get an error like this:
    //
    //   kAudioUnitErr_TooManyFramesToProcess : inFramesToProcess=4096, mMaxFramesPerSlice=512
    //
    // Note how inFramesToProcess is smaller than mMaxFramesPerSlice. To fix, we need to set kAudioUnitProperty_MaximumFramesPerSlice to that
    // of the size of our buffer, or do it the other way around and set our buffer size to the kAudioUnitProperty_MaximumFramesPerSlice.
    {
        /*AudioUnitScope propScope = (deviceType == mal_device_type_playback) ? kAudioUnitScope_Input : kAudioUnitScope_Output;
        AudioUnitElement propBus = (deviceType == mal_device_type_playback) ? MAL_COREAUDIO_OUTPUT_BUS : MAL_COREAUDIO_INPUT_BUS;
    
        status = ((mal_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_MaximumFramesPerSlice, propScope, propBus, &actualBufferSizeInFrames, sizeof(actualBufferSizeInFrames));
        if (status != noErr) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return mal_result_from_OSStatus(status);
        }*/
        
        status = ((mal_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_MaximumFramesPerSlice, kAudioUnitScope_Global, 0, &actualBufferSizeInFrames, sizeof(actualBufferSizeInFrames));
        if (status != noErr) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return mal_result_from_OSStatus(status);
        }
    }
    
    // We need a buffer list if this is an input device. We render into this in the input callback.
    if (deviceType == mal_device_type_capture) {
        mal_bool32 isInterleaved = (bestFormat.mFormatFlags & kAudioFormatFlagIsNonInterleaved) == 0;
        
        size_t allocationSize = sizeof(AudioBufferList) - sizeof(AudioBuffer);  // Subtract sizeof(AudioBuffer) because that part is dynamically sized.
        if (isInterleaved) {
            // Interleaved case. This is the simple case because we just have one buffer.
            allocationSize += sizeof(AudioBuffer) * 1;
            allocationSize += actualBufferSizeInFrames * mal_get_bytes_per_frame(pData->formatOut, pData->channelsOut);
        } else {
            // Non-interleaved case. This is the more complex case because there's more than one buffer.
            allocationSize += sizeof(AudioBuffer) * pData->channelsOut;
            allocationSize += actualBufferSizeInFrames * mal_get_bytes_per_sample(pData->formatOut) * pData->channelsOut;
        }
        
        AudioBufferList* pBufferList = (AudioBufferList*)mal_malloc(allocationSize);
        if (pBufferList == NULL) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return MAL_OUT_OF_MEMORY;
        }
        
        if (isInterleaved) {
            pBufferList->mNumberBuffers = 1;
            pBufferList->mBuffers[0].mNumberChannels = pData->channelsOut;
            pBufferList->mBuffers[0].mDataByteSize = actualBufferSizeInFrames * mal_get_bytes_per_frame(pData->formatOut, pData->channelsOut);
            pBufferList->mBuffers[0].mData = (mal_uint8*)pBufferList + sizeof(AudioBufferList);
        } else {
            pBufferList->mNumberBuffers = pData->channelsOut;
            for (mal_uint32 iBuffer = 0; iBuffer < pBufferList->mNumberBuffers; ++iBuffer) {
                pBufferList->mBuffers[iBuffer].mNumberChannels = 1;
                pBufferList->mBuffers[iBuffer].mDataByteSize = actualBufferSizeInFrames * mal_get_bytes_per_sample(pData->formatOut);
                pBufferList->mBuffers[iBuffer].mData = (mal_uint8*)pBufferList + ((sizeof(AudioBufferList) - sizeof(AudioBuffer)) + (sizeof(AudioBuffer) * pData->channelsOut)) + (actualBufferSizeInFrames * mal_get_bytes_per_sample(pData->formatOut) * iBuffer);
            }
        }
        
        pData->pAudioBufferList = pBufferList;
    }
    
    // Callbacks.
    AURenderCallbackStruct callbackInfo;
    callbackInfo.inputProcRefCon = pDevice_DoNotReference;
    if (deviceType == mal_device_type_playback) {
        callbackInfo.inputProc = mal_on_output__coreaudio;
        status = ((mal_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_SetRenderCallback, kAudioUnitScope_Global, MAL_COREAUDIO_OUTPUT_BUS, &callbackInfo, sizeof(callbackInfo));
        if (status != noErr) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return mal_result_from_OSStatus(status);
        }
    } else {
        callbackInfo.inputProc = mal_on_input__coreaudio;
        status = ((mal_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_SetInputCallback, kAudioUnitScope_Global, MAL_COREAUDIO_INPUT_BUS, &callbackInfo, sizeof(callbackInfo));
        if (status != noErr) {
            ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return mal_result_from_OSStatus(status);
        }
    }
    
    // We need to listen for stop events.
    status = ((mal_AudioUnitAddPropertyListener_proc)pContext->coreaudio.AudioUnitAddPropertyListener)(pData->audioUnit, kAudioOutputUnitProperty_IsRunning, on_start_stop__coreaudio, pDevice_DoNotReference);
    if (status != noErr) {
        ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return mal_result_from_OSStatus(status);
    }
    
    // Initialize the audio unit.
    status = ((mal_AudioUnitInitialize_proc)pContext->coreaudio.AudioUnitInitialize)(pData->audioUnit);
    if (status != noErr) {
        mal_free(pData->pAudioBufferList);
        pData->pAudioBufferList = NULL;
        ((mal_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return mal_result_from_OSStatus(status);
    }
    
    // Grab the name.
#if defined(MAL_APPLE_DESKTOP)
    mal_get_AudioObject_name(pContext, deviceObjectID, sizeof(pData->deviceName), pData->deviceName);
#endif
    
    return result;
}

mal_result mal_device_reinit_internal__coreaudio(mal_device* pDevice, mal_bool32 disposePreviousAudioUnit)
{
    mal_device_init_internal_data__coreaudio data;
    data.formatIn = pDevice->format;
    data.channelsIn = pDevice->channels;
    data.sampleRateIn = pDevice->sampleRate;
    mal_copy_memory(data.channelMapIn, pDevice->channelMap, sizeof(pDevice->channelMap));
    data.bufferSizeInFramesIn = pDevice->bufferSizeInFrames;
    data.bufferSizeInMillisecondsIn = pDevice->bufferSizeInMilliseconds;
    data.periodsIn = pDevice->periods;
    data.usingDefaultFormat = pDevice->usingDefaultFormat;
    data.usingDefaultChannels = pDevice->usingDefaultChannels;
    data.usingDefaultSampleRate = pDevice->usingDefaultSampleRate;
    data.usingDefaultChannelMap = pDevice->usingDefaultChannelMap;
    data.shareMode = pDevice->initConfig.shareMode;

    mal_result result = mal_device_init_internal__coreaudio(pDevice->pContext, pDevice->type, NULL, &data, (void*)pDevice);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    // We have successfully initialized the new objects. We now need to uninitialize the previous objects and re-set them.
    if (disposePreviousAudioUnit) {
        pDevice->pContext->onDeviceStop(pDevice);
        ((mal_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnit);
    }
    if (pDevice->coreaudio.pAudioBufferList) {
        mal_free(pDevice->coreaudio.pAudioBufferList);
    }
    
#if defined(MAL_APPLE_DESKTOP)
    pDevice->coreaudio.deviceObjectID = (mal_uint32)data.deviceObjectID;
#endif
    pDevice->coreaudio.component = (mal_ptr)data.component;
    pDevice->coreaudio.audioUnit = (mal_ptr)data.audioUnit;
    pDevice->coreaudio.pAudioBufferList = (mal_ptr)data.pAudioBufferList;
    
    pDevice->internalFormat = data.formatOut;
    pDevice->internalChannels = data.channelsOut;
    pDevice->internalSampleRate = data.sampleRateOut;
    mal_copy_memory(pDevice->internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
    pDevice->bufferSizeInFrames = data.bufferSizeInFramesOut;
    pDevice->periods = data.periodsOut;
    pDevice->exclusiveMode = MAL_FALSE;
    mal_strcpy_s(pDevice->name, sizeof(pDevice->name), data.deviceName);
    
    return MAL_SUCCESS;
}


mal_result mal_device_init__coreaudio(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pConfig;

    mal_assert(pContext != NULL);
    mal_assert(pConfig != NULL);
    mal_assert(pDevice != NULL);
    mal_assert(deviceType == mal_device_type_playback || deviceType == mal_device_type_capture);
    
    mal_device_init_internal_data__coreaudio data;
    data.formatIn = pDevice->format;
    data.channelsIn = pDevice->channels;
    data.sampleRateIn = pDevice->sampleRate;
    mal_copy_memory(data.channelMapIn, pDevice->channelMap, sizeof(pDevice->channelMap));
    data.bufferSizeInFramesIn = pDevice->bufferSizeInFrames;
    data.bufferSizeInMillisecondsIn = pDevice->bufferSizeInMilliseconds;
    data.periodsIn = pDevice->periods;
    data.usingDefaultFormat = pDevice->usingDefaultFormat;
    data.usingDefaultChannels = pDevice->usingDefaultChannels;
    data.usingDefaultSampleRate = pDevice->usingDefaultSampleRate;
    data.usingDefaultChannelMap = pDevice->usingDefaultChannelMap;
    data.shareMode = pDevice->initConfig.shareMode;

    mal_result result = mal_device_init_internal__coreaudio(pDevice->pContext, pDevice->type, NULL, &data, (void*)pDevice);
    if (result != MAL_SUCCESS) {
        return result;
    }
    
    // We have successfully initialized the new objects. We now need to uninitialize the previous objects and re-set them.
    pDevice->pContext->onDeviceStop(pDevice);
    ((mal_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnit);
    if (pDevice->coreaudio.pAudioBufferList) {
        mal_free(pDevice->coreaudio.pAudioBufferList);
    }
    
#if defined(MAL_APPLE_DESKTOP)
    pDevice->coreaudio.deviceObjectID = (mal_uint32)data.deviceObjectID;
#endif
    pDevice->coreaudio.component = (mal_ptr)data.component;
    pDevice->coreaudio.audioUnit = (mal_ptr)data.audioUnit;
    pDevice->coreaudio.pAudioBufferList = (mal_ptr)data.pAudioBufferList;
    
    pDevice->internalFormat = data.formatOut;
    pDevice->internalChannels = data.channelsOut;
    pDevice->internalSampleRate = data.sampleRateOut;
    mal_copy_memory(pDevice->internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
    pDevice->bufferSizeInFrames = data.bufferSizeInFramesOut;
    pDevice->periods = data.periodsOut;
    pDevice->exclusiveMode = MAL_FALSE;
    mal_strcpy_s(pDevice->name, sizeof(pDevice->name), data.deviceName);
    
#if defined(MAL_APPLE_DESKTOP)
    // If we are using the default device we'll need to listen for changes to the system's default device so we can seemlessly
    // switch the device in the background.
    AudioObjectPropertyAddress propAddress;
    propAddress.mSelector = (deviceType == mal_device_type_playback) ? kAudioHardwarePropertyDefaultOutputDevice : kAudioHardwarePropertyDefaultInputDevice;
    propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    ((mal_AudioObjectAddPropertyListener_proc)pDevice->pContext->coreaudio.AudioObjectAddPropertyListener)(kAudioObjectSystemObject, &propAddress, &mal_default_output_device_changed__coreaudio, pDevice);
#endif

    return MAL_SUCCESS;
}


mal_result mal_device__start_backend__coreaudio(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);
    
    OSStatus status = ((mal_AudioOutputUnitStart_proc)pDevice->pContext->coreaudio.AudioOutputUnitStart)((AudioUnit)pDevice->coreaudio.audioUnit);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__coreaudio(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);
    
    OSStatus status = ((mal_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnit);
    if (status != noErr) {
        return mal_result_from_OSStatus(status);
    }
    
    return MAL_SUCCESS;
}


mal_result mal_context_uninit__coreaudio(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_coreaudio);
    
#if !defined(MAL_NO_RUNTIME_LINKING) && !defined(MAL_APPLE_MOBILE)
    mal_dlclose(pContext->coreaudio.hAudioUnit);
    mal_dlclose(pContext->coreaudio.hCoreAudio);
    mal_dlclose(pContext->coreaudio.hCoreFoundation);
#endif

    (void)pContext;
    return MAL_SUCCESS;
}

mal_result mal_context_init__coreaudio(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    
#if !defined(MAL_NO_RUNTIME_LINKING) && !defined(MAL_APPLE_MOBILE)
    pContext->coreaudio.hCoreFoundation = mal_dlopen("CoreFoundation.framework/CoreFoundation");
    if (pContext->coreaudio.hCoreFoundation == NULL) {
        return MAL_API_NOT_FOUND;
    }
    
    pContext->coreaudio.CFStringGetCString             = mal_dlsym(pContext->coreaudio.hCoreFoundation, "CFStringGetCString");
    
    
    pContext->coreaudio.hCoreAudio = mal_dlopen("CoreAudio.framework/CoreAudio");
    if (pContext->coreaudio.hCoreAudio == NULL) {
        mal_dlclose(pContext->coreaudio.hCoreFoundation);
        return MAL_API_NOT_FOUND;
    }
    
    pContext->coreaudio.AudioObjectGetPropertyData     = mal_dlsym(pContext->coreaudio.hCoreAudio, "AudioObjectGetPropertyData");
    pContext->coreaudio.AudioObjectGetPropertyDataSize = mal_dlsym(pContext->coreaudio.hCoreAudio, "AudioObjectGetPropertyDataSize");
    pContext->coreaudio.AudioObjectSetPropertyData     = mal_dlsym(pContext->coreaudio.hCoreAudio, "AudioObjectSetPropertyData");
    pContext->coreaudio.AudioObjectAddPropertyListener = mal_dlsym(pContext->coreaudio.hCoreAudio, "AudioObjectAddPropertyListener");

    
    // It looks like Apple has moved some APIs from AudioUnit into AudioToolbox on more recent versions of macOS. They are still
    // defined in AudioUnit, but just in case they decide to remove them from there entirely I'm going to implement a fallback.
    // The way it'll work is that it'll first try AudioUnit, and if the required symbols are not present there we'll fall back to
    // AudioToolbox.
    pContext->coreaudio.hAudioUnit = mal_dlopen("AudioUnit.framework/AudioUnit");
    if (pContext->coreaudio.hAudioUnit == NULL) {
        mal_dlclose(pContext->coreaudio.hCoreAudio);
        mal_dlclose(pContext->coreaudio.hCoreFoundation);
        return MAL_API_NOT_FOUND;
    }
    
    if (mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioComponentFindNext") == NULL) {
        // Couldn't find the required symbols in AudioUnit, so fall back to AudioToolbox.
        mal_dlclose(pContext->coreaudio.hAudioUnit);
        pContext->coreaudio.hAudioUnit = mal_dlopen("AudioToolbox.framework/AudioToolbox");
        if (pContext->coreaudio.hAudioUnit == NULL) {
            mal_dlclose(pContext->coreaudio.hCoreAudio);
            mal_dlclose(pContext->coreaudio.hCoreFoundation);
            return MAL_API_NOT_FOUND;
        }
    }
    
    pContext->coreaudio.AudioComponentFindNext         = mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioComponentFindNext");
    pContext->coreaudio.AudioComponentInstanceDispose  = mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioComponentInstanceDispose");
    pContext->coreaudio.AudioComponentInstanceNew      = mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioComponentInstanceNew");
    pContext->coreaudio.AudioOutputUnitStart           = mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioOutputUnitStart");
    pContext->coreaudio.AudioOutputUnitStop            = mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioOutputUnitStop");
    pContext->coreaudio.AudioUnitAddPropertyListener   = mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitAddPropertyListener");
    pContext->coreaudio.AudioUnitGetProperty           = mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitGetProperty");
    pContext->coreaudio.AudioUnitSetProperty           = mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitSetProperty");
    pContext->coreaudio.AudioUnitInitialize            = mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitInitialize");
    pContext->coreaudio.AudioUnitRender                = mal_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitRender");
#else
    pContext->coreaudio.CFStringGetCString             = (mal_proc)CFStringGetCString;
    
    #if defined(MAL_APPLE_DESKTOP)
    pContext->coreaudio.AudioObjectGetPropertyData     = (mal_proc)AudioObjectGetPropertyData;
    pContext->coreaudio.AudioObjectGetPropertyDataSize = (mal_proc)AudioObjectGetPropertyDataSize;
    pContext->coreaudio.AudioObjectSetPropertyData     = (mal_proc)AudioObjectSetPropertyData;
    pContext->coreaudio.AudioObjectAddPropertyListener = (mal_proc)AudioObjectAddPropertyListener;
    #endif
    
    pContext->coreaudio.AudioComponentFindNext         = (mal_proc)AudioComponentFindNext;
    pContext->coreaudio.AudioComponentInstanceDispose  = (mal_proc)AudioComponentInstanceDispose;
    pContext->coreaudio.AudioComponentInstanceNew      = (mal_proc)AudioComponentInstanceNew;
    pContext->coreaudio.AudioOutputUnitStart           = (mal_proc)AudioOutputUnitStart;
    pContext->coreaudio.AudioOutputUnitStop            = (mal_proc)AudioOutputUnitStop;
    pContext->coreaudio.AudioUnitAddPropertyListener   = (mal_proc)AudioUnitAddPropertyListener;
    pContext->coreaudio.AudioUnitGetProperty           = (mal_proc)AudioUnitGetProperty;
    pContext->coreaudio.AudioUnitSetProperty           = (mal_proc)AudioUnitSetProperty;
    pContext->coreaudio.AudioUnitInitialize            = (mal_proc)AudioUnitInitialize;
    pContext->coreaudio.AudioUnitRender                = (mal_proc)AudioUnitRender;
#endif

    pContext->isBackendAsynchronous = MAL_TRUE;
    
    pContext->onUninit        = mal_context_uninit__coreaudio;
    pContext->onDeviceIDEqual = mal_context_is_device_id_equal__coreaudio;
    pContext->onEnumDevices   = mal_context_enumerate_devices__coreaudio;
    pContext->onGetDeviceInfo = mal_context_get_device_info__coreaudio;
    pContext->onDeviceInit    = mal_device_init__coreaudio;
    pContext->onDeviceUninit  = mal_device_uninit__coreaudio;
    pContext->onDeviceStart   = mal_device__start_backend__coreaudio;
    pContext->onDeviceStop    = mal_device__stop_backend__coreaudio;

    return MAL_SUCCESS;
}
#endif  // Core Audio



///////////////////////////////////////////////////////////////////////////////
//
// sndio Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_SNDIO
#include <fcntl.h>
#include <sys/stat.h>

// Only supporting OpenBSD. This did not work very well at all on FreeBSD when I tried it. Not sure if this is due
// to mini_al's implementation or if it's some kind of system configuration issue, but basically the default device
// just doesn't emit any sound, or at times you'll hear tiny pieces. I will consider enabling this when there's
// demand for it or if I can get it tested and debugged more thoroughly.

//#if defined(__NetBSD__) || defined(__OpenBSD__)
//#include <sys/audioio.h>
//#endif
//#if defined(__FreeBSD__) || defined(__DragonFly__)
//#include <sys/soundcard.h>
//#endif

#define MAL_SIO_DEVANY	"default"
#define MAL_SIO_PLAY	1
#define MAL_SIO_REC		2
#define MAL_SIO_NENC	8
#define MAL_SIO_NCHAN	8
#define MAL_SIO_NRATE	16
#define MAL_SIO_NCONF	4

struct mal_sio_hdl; // <-- Opaque

struct mal_sio_par
{
    unsigned int bits;
    unsigned int bps;
    unsigned int sig;
    unsigned int le;
    unsigned int msb;
    unsigned int rchan;
    unsigned int pchan;
    unsigned int rate;
    unsigned int bufsz;
    unsigned int xrun;
    unsigned int round;
    unsigned int appbufsz;
    int __pad[3];
    unsigned int __magic;
};

struct mal_sio_enc
{
    unsigned int bits;
    unsigned int bps;
    unsigned int sig;
    unsigned int le;
    unsigned int msb;
};

struct mal_sio_conf
{
    unsigned int enc;
    unsigned int rchan;
    unsigned int pchan;
    unsigned int rate;
};

struct mal_sio_cap
{
    struct mal_sio_enc enc[MAL_SIO_NENC];
	unsigned int rchan[MAL_SIO_NCHAN];
	unsigned int pchan[MAL_SIO_NCHAN];
	unsigned int rate[MAL_SIO_NRATE];
	int __pad[7];
	unsigned int nconf;
	struct mal_sio_conf confs[MAL_SIO_NCONF];
};

typedef struct mal_sio_hdl* (* mal_sio_open_proc)   (const char*, unsigned int, int);
typedef void                (* mal_sio_close_proc)  (struct mal_sio_hdl*);
typedef int                 (* mal_sio_setpar_proc) (struct mal_sio_hdl*, struct mal_sio_par*);
typedef int                 (* mal_sio_getpar_proc) (struct mal_sio_hdl*, struct mal_sio_par*);
typedef int                 (* mal_sio_getcap_proc) (struct mal_sio_hdl*, struct mal_sio_cap*);
typedef size_t              (* mal_sio_write_proc)  (struct mal_sio_hdl*, const void*, size_t);
typedef size_t              (* mal_sio_read_proc)   (struct mal_sio_hdl*, void*, size_t);
typedef int                 (* mal_sio_start_proc)  (struct mal_sio_hdl*);
typedef int                 (* mal_sio_stop_proc)   (struct mal_sio_hdl*);
typedef int                 (* mal_sio_initpar_proc)(struct mal_sio_par*);

mal_format mal_format_from_sio_enc__sndio(unsigned int bits, unsigned int bps, unsigned int sig, unsigned int le, unsigned int msb)
{
    // We only support native-endian right now.
    if ((mal_is_little_endian() && le == 0) || (mal_is_big_endian() && le == 1)) {
        return mal_format_unknown;
    }
    
    if (bits ==  8 && bps == 1 && sig == 0) {
        return mal_format_u8;
    }
    if (bits == 16 && bps == 2 && sig == 1) {
        return mal_format_s16;
    }
    if (bits == 24 && bps == 3 && sig == 1) {
        return mal_format_s24;
    }
    if (bits == 24 && bps == 4 && sig == 1 && msb == 0) {
        //return mal_format_s24_32;
    }
    if (bits == 32 && bps == 4 && sig == 1) {
        return mal_format_s32;
    }
    
    return mal_format_unknown;
}

mal_format mal_find_best_format_from_sio_cap__sndio(struct mal_sio_cap* caps)
{
    mal_assert(caps != NULL);
    
    mal_format bestFormat = mal_format_unknown;
    for (unsigned int iConfig = 0; iConfig < caps->nconf; iConfig += 1) {
        for (unsigned int iEncoding = 0; iEncoding < MAL_SIO_NENC; iEncoding += 1) {
            if ((caps->confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            unsigned int bits = caps->enc[iEncoding].bits;
            unsigned int bps  = caps->enc[iEncoding].bps;
            unsigned int sig  = caps->enc[iEncoding].sig;
            unsigned int le   = caps->enc[iEncoding].le;
            unsigned int msb  = caps->enc[iEncoding].msb;
            mal_format format = mal_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format == mal_format_unknown) {
                continue;   // Format not supported.
            }
            
            if (bestFormat == mal_format_unknown) {
                bestFormat = format;
            } else {
                if (mal_get_format_priority_index(bestFormat) > mal_get_format_priority_index(format)) {    // <-- Lower = better.
                    bestFormat = format;
                }
            }
        }
    }
    
    return mal_format_unknown;
}

mal_uint32 mal_find_best_channels_from_sio_cap__sndio(struct mal_sio_cap* caps, mal_device_type deviceType, mal_format requiredFormat)
{
    mal_assert(caps != NULL);
    mal_assert(requiredFormat != mal_format_unknown);
    
    // Just pick whatever configuration has the most channels.
    mal_uint32 maxChannels = 0;
    for (unsigned int iConfig = 0; iConfig < caps->nconf; iConfig += 1) {
        // The encoding should be of requiredFormat.
        for (unsigned int iEncoding = 0; iEncoding < MAL_SIO_NENC; iEncoding += 1) {
            if ((caps->confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            unsigned int bits = caps->enc[iEncoding].bits;
            unsigned int bps  = caps->enc[iEncoding].bps;
            unsigned int sig  = caps->enc[iEncoding].sig;
            unsigned int le   = caps->enc[iEncoding].le;
            unsigned int msb  = caps->enc[iEncoding].msb;
            mal_format format = mal_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format != requiredFormat) {
                continue;
            }
            
            // Getting here means the format is supported. Iterate over each channel count and grab the biggest one.
            for (unsigned int iChannel = 0; iChannel < MAL_SIO_NCHAN; iChannel += 1) {
                unsigned int chan = 0;
                if (deviceType == mal_device_type_playback) {
                    chan = caps->confs[iConfig].pchan;
                } else {
                    chan = caps->confs[iConfig].rchan;
                }
            
                if ((chan & (1UL << iChannel)) == 0) {
                    continue;
                }
                
                unsigned int channels;
                if (deviceType == mal_device_type_playback) {
                    channels = caps->pchan[iChannel];
                } else {
                    channels = caps->rchan[iChannel];
                }
                
                if (maxChannels < channels) {
                    maxChannels = channels;
                }
            }
        }
    }
    
    return maxChannels;
}

mal_uint32 mal_find_best_sample_rate_from_sio_cap__sndio(struct mal_sio_cap* caps, mal_device_type deviceType, mal_format requiredFormat, mal_uint32 requiredChannels)
{
    mal_assert(caps != NULL);
    mal_assert(requiredFormat != mal_format_unknown);
    mal_assert(requiredChannels > 0);
    mal_assert(requiredChannels <= MAL_MAX_CHANNELS);
    
    mal_uint32 firstSampleRate = 0; // <-- If the device does not support a standard rate we'll fall back to the first one that's found.
    
    mal_uint32 bestSampleRate = 0;    
    for (unsigned int iConfig = 0; iConfig < caps->nconf; iConfig += 1) {
        // The encoding should be of requiredFormat.
        for (unsigned int iEncoding = 0; iEncoding < MAL_SIO_NENC; iEncoding += 1) {
            if ((caps->confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            unsigned int bits = caps->enc[iEncoding].bits;
            unsigned int bps  = caps->enc[iEncoding].bps;
            unsigned int sig  = caps->enc[iEncoding].sig;
            unsigned int le   = caps->enc[iEncoding].le;
            unsigned int msb  = caps->enc[iEncoding].msb;
            mal_format format = mal_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format != requiredFormat) {
                continue;
            }
            
            // Getting here means the format is supported. Iterate over each channel count and grab the biggest one.
            for (unsigned int iChannel = 0; iChannel < MAL_SIO_NCHAN; iChannel += 1) {
                unsigned int chan = 0;
                if (deviceType == mal_device_type_playback) {
                    chan = caps->confs[iConfig].pchan;
                } else {
                    chan = caps->confs[iConfig].rchan;
                }
            
                if ((chan & (1UL << iChannel)) == 0) {
                    continue;
                }
                
                unsigned int channels;
                if (deviceType == mal_device_type_playback) {
                    channels = caps->pchan[iChannel];
                } else {
                    channels = caps->rchan[iChannel];
                }
                
                if (channels != requiredChannels) {
                    continue;
                }
                
                // Getting here means we have found a compatible encoding/channel pair.
                for (unsigned int iRate = 0; iRate < MAL_SIO_NRATE; iRate += 1) {
                    mal_uint32 rate = (mal_uint32)caps->rate[iRate];
                
                    if (firstSampleRate == 0) {
                        firstSampleRate = rate;
                    }
                    
                    // Disregard this rate if it's not a standard one.
                    mal_uint32 ratePriority = mal_get_standard_sample_rate_priority_index(rate);
                    if (ratePriority == (mal_uint32)-1) {
                        continue;
                    }
                    
                    if (mal_get_standard_sample_rate_priority_index(bestSampleRate) > ratePriority) {   // Lower = better.
                        bestSampleRate = rate;
                    }
                }
            }
        }
    }
    
    // If a standard sample rate was not found just fall back to the first one that was iterated.
    if (bestSampleRate == 0) {
        bestSampleRate = firstSampleRate;
    }
    
    return bestSampleRate;
}


mal_bool32 mal_context_is_device_id_equal__sndio(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return mal_strcmp(pID0->sndio, pID1->sndio) == 0;
}

mal_result mal_context_enumerate_devices__sndio(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);
    
    // sndio doesn't seem to have a good device enumeration API, so I'm therefore only enumerating
    // over default devices for now.
    mal_bool32 isTerminating = MAL_FALSE;
    struct mal_sio_hdl* handle;
    
    // Playback.
    if (!isTerminating) {
        handle = ((mal_sio_open_proc)pContext->sndio.sio_open)(MAL_SIO_DEVANY, MAL_SIO_PLAY, 0);
        if (handle != NULL) {
            // Supports playback.
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            mal_strcpy_s(deviceInfo.id.sndio, sizeof(deviceInfo.id.sndio), MAL_SIO_DEVANY);
            mal_strcpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME);
            
            isTerminating = !callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
            
            ((mal_sio_close_proc)pContext->sndio.sio_close)(handle);
        }
    }
    
    // Capture.
    if (!isTerminating) {
        handle = ((mal_sio_open_proc)pContext->sndio.sio_open)(MAL_SIO_DEVANY, MAL_SIO_REC, 0);
        if (handle != NULL) {
            // Supports capture.
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            mal_strcpy_s(deviceInfo.id.sndio, sizeof(deviceInfo.id.sndio), "default");
            mal_strcpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_CAPTURE_DEVICE_NAME);

            isTerminating = !callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
            
            ((mal_sio_close_proc)pContext->sndio.sio_close)(handle);
        }
    }
    
    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info__sndio(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    (void)shareMode;
    
    // We need to open the device before we can get information about it.
    char devid[256];
    if (pDeviceID == NULL) {
        mal_strcpy_s(devid, sizeof(devid), MAL_SIO_DEVANY);
        mal_strcpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), (deviceType == mal_device_type_playback) ? MAL_DEFAULT_PLAYBACK_DEVICE_NAME : MAL_DEFAULT_CAPTURE_DEVICE_NAME);
    } else {
        mal_strcpy_s(devid, sizeof(devid), pDeviceID->sndio);
        mal_strcpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), devid);
    }
    
    struct mal_sio_hdl* handle = ((mal_sio_open_proc)pContext->sndio.sio_open)(devid, (deviceType == mal_device_type_playback) ? MAL_SIO_PLAY : MAL_SIO_REC, 0);
    if (handle == NULL) {
        return MAL_NO_DEVICE;
    }
    
    struct mal_sio_cap caps;
    if (((mal_sio_getcap_proc)pContext->sndio.sio_getcap)(handle, &caps) == 0) {
        return MAL_ERROR;
    }
    
    for (unsigned int iConfig = 0; iConfig < caps.nconf; iConfig += 1) {
        // The main thing we care about is that the encoding is supported by mini_al. If it is, we want to give
        // preference to some formats over others.
        for (unsigned int iEncoding = 0; iEncoding < MAL_SIO_NENC; iEncoding += 1) {
            if ((caps.confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            unsigned int bits = caps.enc[iEncoding].bits;
            unsigned int bps  = caps.enc[iEncoding].bps;
            unsigned int sig  = caps.enc[iEncoding].sig;
            unsigned int le   = caps.enc[iEncoding].le;
            unsigned int msb  = caps.enc[iEncoding].msb;
            mal_format format = mal_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format == mal_format_unknown) {
                continue;   // Format not supported.
            }
            
            // Add this format if it doesn't already exist.
            mal_bool32 formatExists = MAL_FALSE;
            for (mal_uint32 iExistingFormat = 0; iExistingFormat < pDeviceInfo->formatCount; iExistingFormat += 1) {
                if (pDeviceInfo->formats[iExistingFormat] == format) {
                    formatExists = MAL_TRUE;
                    break;
                }
            }
            
            if (!formatExists) {
                pDeviceInfo->formats[pDeviceInfo->formatCount++] = format;
            }
        }
        
        // Channels.
        for (unsigned int iChannel = 0; iChannel < MAL_SIO_NCHAN; iChannel += 1) {
            unsigned int chan = 0;
            if (deviceType == mal_device_type_playback) {
                chan = caps.confs[iConfig].pchan;
            } else {
                chan = caps.confs[iConfig].rchan;
            }
        
            if ((chan & (1UL << iChannel)) == 0) {
                continue;
            }
            
            unsigned int channels;
            if (deviceType == mal_device_type_playback) {
                channels = caps.pchan[iChannel];
            } else {
                channels = caps.rchan[iChannel];
            }
            
            if (pDeviceInfo->minChannels > channels) {
                pDeviceInfo->minChannels = channels;
            }
            if (pDeviceInfo->maxChannels < channels) {
                pDeviceInfo->maxChannels = channels;
            }
        }
        
        // Sample rates.
        for (unsigned int iRate = 0; iRate < MAL_SIO_NRATE; iRate += 1) {
            if ((caps.confs[iConfig].rate & (1UL << iRate)) != 0) {
                unsigned int rate = caps.rate[iRate];
                if (pDeviceInfo->minSampleRate > rate) {
                    pDeviceInfo->minSampleRate = rate;
                }
                if (pDeviceInfo->maxSampleRate < rate) {
                    pDeviceInfo->maxSampleRate = rate;
                }
            }
        }
    }

    ((mal_sio_close_proc)pContext->sndio.sio_close)(handle);
    return MAL_SUCCESS;
}

void mal_device_uninit__sndio(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    ((mal_sio_close_proc)pDevice->pContext->sndio.sio_close)((struct mal_sio_hdl*)pDevice->sndio.handle);
    mal_free(pDevice->sndio.pIntermediaryBuffer);
}

mal_result mal_device_init__sndio(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pContext;

    mal_assert(pDevice != NULL);
    mal_zero_object(&pDevice->sndio);
    
    const char* deviceName = MAL_SIO_DEVANY;
//#if defined(__FreeBSD__) || defined(__DragonFly__)
//    deviceName = "rsnd/0";
//#else
    if (pDeviceID != NULL) {
        deviceName = pDeviceID->sndio;
    }
    
    pDevice->sndio.handle = (mal_ptr)((mal_sio_open_proc)pContext->sndio.sio_open)(deviceName, (deviceType == mal_device_type_playback) ? MAL_SIO_PLAY : MAL_SIO_REC, 0);
    if (pDevice->sndio.handle == NULL) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[sndio] Failed to open device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }
    
    // We need to retrieve the device caps to determine the most appropriate format to use.
    struct mal_sio_cap caps;
    if (((mal_sio_getcap_proc)pContext->sndio.sio_getcap)((struct mal_sio_hdl*)pDevice->sndio.handle, &caps) == 0) {
        ((mal_sio_close_proc)pContext->sndio.sio_close)((struct mal_sio_hdl*)pDevice->sndio.handle);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[sndio] Failed to retrieve device caps.", MAL_ERROR);
    }
    
    mal_format desiredFormat = pDevice->format;
    if (pDevice->usingDefaultFormat) {
        desiredFormat = mal_find_best_format_from_sio_cap__sndio(&caps);
    }
    
    if (desiredFormat == mal_format_unknown) {
        desiredFormat = pDevice->format;
    }
    

    // Note: sndio reports a huge range of available channels. This is inconvenient for us because there's no real
    // way, as far as I can tell, to get the _actual_ channel count of the device. I'm therefore restricting this
    // to the requested channels, regardless of whether or not the default channel count is requested.
    //
    // For hardware devices, I'm suspecting only a single channel count will be reported and we can safely use the
    // value returned by mal_find_best_channels_from_sio_cap__sndio().
    mal_uint32 desiredChannels = pDevice->channels;
    if (pDevice->usingDefaultChannels) {
        if (strlen(deviceName) > strlen("rsnd/") && strncmp(deviceName, "rsnd/", strlen("rsnd/")) == 0) {
            desiredChannels = mal_find_best_channels_from_sio_cap__sndio(&caps, deviceType, desiredFormat);
        }
    }

    if (desiredChannels == 0) {
        desiredChannels = pDevice->channels;
    }

    
    mal_uint32 desiredSampleRate = pDevice->sampleRate;
    if (pDevice->usingDefaultSampleRate) {
        desiredSampleRate = mal_find_best_sample_rate_from_sio_cap__sndio(&caps, deviceType, desiredFormat, desiredChannels);
    }
    
    if (desiredSampleRate == 0) {
        desiredSampleRate = pDevice->sampleRate;
    }
    
    
    struct mal_sio_par par;
    ((mal_sio_initpar_proc)pDevice->pContext->sndio.sio_initpar)(&par);
    par.msb = 0;
    par.le  = mal_is_little_endian();
    
    switch (desiredFormat) {
        case mal_format_u8:
        {
            par.bits = 8;
            par.bps  = 1;
            par.sig  = 0;
        } break;
        
        case mal_format_s24:
        {
            par.bits = 24;
            par.bps  = 3;
            par.sig  = 1;
        } break;
        
        case mal_format_s32:
        {
            par.bits = 32;
            par.bps  = 4;
            par.sig  = 1;
        } break;
        
        case mal_format_s16:
        case mal_format_f32:
        default:
        {
            par.bits = 16;
            par.bps  = 2;
            par.sig  = 1;
        } break;
    }
    
    if (deviceType == mal_device_type_playback) {
        par.pchan = desiredChannels;
    } else {
        par.rchan = desiredChannels;
    }

    par.rate = desiredSampleRate;
    
    // Try calculating an appropriate default buffer size after we have the sample rate.
    mal_uint32 desiredBufferSizeInFrames = pDevice->bufferSizeInFrames;
    if (desiredBufferSizeInFrames == 0) {
        desiredBufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, par.rate);
    }

    par.round = desiredBufferSizeInFrames / pDevice->periods;
    par.appbufsz = par.round * pDevice->periods;
    
    if (((mal_sio_setpar_proc)pContext->sndio.sio_setpar)((struct mal_sio_hdl*)pDevice->sndio.handle, &par) == 0) {
        ((mal_sio_close_proc)pContext->sndio.sio_close)((struct mal_sio_hdl*)pDevice->sndio.handle);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[sndio] Failed to set buffer size.", MAL_FORMAT_NOT_SUPPORTED);
    }
    if (((mal_sio_getpar_proc)pContext->sndio.sio_getpar)((struct mal_sio_hdl*)pDevice->sndio.handle, &par) == 0) {
        ((mal_sio_close_proc)pContext->sndio.sio_close)((struct mal_sio_hdl*)pDevice->sndio.handle);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[sndio] Failed to retrieve buffer size.", MAL_FORMAT_NOT_SUPPORTED);
    }
    
    pDevice->internalFormat = mal_format_from_sio_enc__sndio(par.bits, par.bps, par.sig, par.le, par.msb);
    
    if (deviceType == mal_device_type_playback) {
        pDevice->internalChannels = par.pchan;
    } else {
        pDevice->internalChannels = par.rchan;
    }
    
    pDevice->internalSampleRate = par.rate;

    pDevice->periods = par.appbufsz / par.round;
    if (pDevice->periods < 2) {
        pDevice->periods = 2;
    }
    pDevice->bufferSizeInFrames = par.round * pDevice->periods;
    pDevice->sndio.fragmentSizeInFrames = par.round;
    
    mal_get_standard_channel_map(mal_standard_channel_map_sndio, pDevice->internalChannels, pDevice->internalChannelMap);
    
    // The device is always shared with sndio.
    pDevice->exclusiveMode = MAL_FALSE;
    
    pDevice->sndio.pIntermediaryBuffer = mal_malloc(pDevice->sndio.fragmentSizeInFrames * mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels));
    if (pDevice->sndio.pIntermediaryBuffer == NULL) {
        ((mal_sio_close_proc)pContext->sndio.sio_close)((struct mal_sio_hdl*)pDevice->sndio.handle);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[sndio] Failed to allocate memory for intermediary buffer.", MAL_OUT_OF_MEMORY);
    }
    
#ifdef MAL_DEBUG_OUTPUT
    printf("DEVICE INFO\n");
    printf("    Format:      %s\n", mal_get_format_name(pDevice->internalFormat));
    printf("    Channels:    %d\n", pDevice->internalChannels);
    printf("    Sample Rate: %d\n", pDevice->internalSampleRate);
    printf("    Buffer Size: %d\n", pDevice->bufferSizeInFrames);
    printf("    Periods:     %d\n", pDevice->periods);
    printf("    appbufsz:    %d\n", par.appbufsz);
    printf("    round:       %d\n", par.round);
#endif
    
    return MAL_SUCCESS;
}

mal_result mal_device__start_backend__sndio(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);
    
    if (((mal_sio_start_proc)pDevice->pContext->sndio.sio_start)((struct mal_sio_hdl*)pDevice->sndio.handle) == 0) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[sndio] Failed to start backend device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
    }

    // The device is started by the next calls to read() and write(). For playback it's simple - just read
    // data from the client, then write it to the device with write() which will in turn start the device.
    // For capture it's a bit less intuitive - we do nothing (it'll be started automatically by the first
    // call to read().
    if (pDevice->type == mal_device_type_playback) {
        // Playback. Need to load the entire buffer, which means we need to write a fragment for each period.
        for (mal_uint32 iPeriod = 0; iPeriod < pDevice->periods; iPeriod += 1) { 
            mal_device__read_frames_from_client(pDevice, pDevice->sndio.fragmentSizeInFrames, pDevice->sndio.pIntermediaryBuffer);

            int bytesWritten = ((mal_sio_write_proc)pDevice->pContext->sndio.sio_write)((struct mal_sio_hdl*)pDevice->sndio.handle, pDevice->sndio.pIntermediaryBuffer, pDevice->sndio.fragmentSizeInFrames * mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels));
            if (bytesWritten == 0) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[sndio] Failed to send initial chunk of data to the device.", MAL_FAILED_TO_SEND_DATA_TO_DEVICE);
            }
        }
    } else {
        // Capture. Do nothing.
    }

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__sndio(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    ((mal_sio_stop_proc)pDevice->pContext->sndio.sio_stop)((struct mal_sio_hdl*)pDevice->sndio.handle);
    return MAL_SUCCESS;
}

mal_result mal_device__break_main_loop__sndio(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->sndio.breakFromMainLoop = MAL_TRUE;
    return MAL_SUCCESS;
}

mal_result mal_device__main_loop__sndio(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->sndio.breakFromMainLoop = MAL_FALSE;
    while (!pDevice->sndio.breakFromMainLoop) {
        // Break from the main loop if the device isn't started anymore. Likely what's happened is the application
        // has requested that the device be stopped.
        if (!mal_device_is_started(pDevice)) {
            break;
        }

        if (pDevice->type == mal_device_type_playback) {
            // Playback.
            mal_device__read_frames_from_client(pDevice, pDevice->sndio.fragmentSizeInFrames, pDevice->sndio.pIntermediaryBuffer);

            int bytesWritten = ((mal_sio_write_proc)pDevice->pContext->sndio.sio_write)((struct mal_sio_hdl*)pDevice->sndio.handle, pDevice->sndio.pIntermediaryBuffer, pDevice->sndio.fragmentSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat));
            if (bytesWritten == 0) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[sndio] Failed to send data from the client to the device.", MAL_FAILED_TO_SEND_DATA_TO_DEVICE);
            }
        } else {
            // Capture.
            int bytesRead = ((mal_sio_read_proc)pDevice->pContext->sndio.sio_read)((struct mal_sio_hdl*)pDevice->sndio.handle, pDevice->sndio.pIntermediaryBuffer, pDevice->sndio.fragmentSizeInFrames * mal_get_bytes_per_sample(pDevice->internalFormat));
            if (bytesRead == 0) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[sndio] Failed to read data from the device to be sent to the client.", MAL_FAILED_TO_READ_DATA_FROM_DEVICE);
            }

            mal_uint32 framesRead = (mal_uint32)bytesRead / pDevice->internalChannels / mal_get_bytes_per_sample(pDevice->internalFormat);
            mal_device__send_frames_to_client(pDevice, framesRead, pDevice->sndio.pIntermediaryBuffer);
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_context_uninit__sndio(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_sndio);

    (void)pContext;
    return MAL_SUCCESS;
}

mal_result mal_context_init__sndio(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    
#ifndef MAL_NO_RUNTIME_LINKING
    // libpulse.so
    const char* libsndioNames[] = {
        "libsndio.so"
    };

    for (size_t i = 0; i < mal_countof(libsndioNames); ++i) {
        pContext->sndio.sndioSO = mal_dlopen(libsndioNames[i]);
        if (pContext->sndio.sndioSO != NULL) {
            break;
        }
    }

    if (pContext->sndio.sndioSO == NULL) {
        return MAL_NO_BACKEND;
    }
    
    pContext->sndio.sio_open    = (mal_proc)mal_dlsym(pContext->sndio.sndioSO, "sio_open");
    pContext->sndio.sio_close   = (mal_proc)mal_dlsym(pContext->sndio.sndioSO, "sio_close");
    pContext->sndio.sio_setpar  = (mal_proc)mal_dlsym(pContext->sndio.sndioSO, "sio_setpar");
    pContext->sndio.sio_getpar  = (mal_proc)mal_dlsym(pContext->sndio.sndioSO, "sio_getpar");
    pContext->sndio.sio_getcap  = (mal_proc)mal_dlsym(pContext->sndio.sndioSO, "sio_getcap");
    pContext->sndio.sio_write   = (mal_proc)mal_dlsym(pContext->sndio.sndioSO, "sio_write");
    pContext->sndio.sio_read    = (mal_proc)mal_dlsym(pContext->sndio.sndioSO, "sio_read");
    pContext->sndio.sio_start   = (mal_proc)mal_dlsym(pContext->sndio.sndioSO, "sio_start");
    pContext->sndio.sio_stop    = (mal_proc)mal_dlsym(pContext->sndio.sndioSO, "sio_stop");
    pContext->sndio.sio_initpar = (mal_proc)mal_dlsym(pContext->sndio.sndioSO, "sio_initpar");
#else
    pContext->sndio.sio_open    = sio_open;
    pContext->sndio.sio_close   = sio_close;
    pContext->sndio.sio_setpar  = sio_setpar;
    pContext->sndio.sio_getpar  = sio_getpar;
    pContext->sndio.sio_getcap  = sio_getcap;
    pContext->sndio.sio_write   = sio_write;
    pContext->sndio.sio_read    = sio_read;
    pContext->sndio.sio_start   = sio_start;
    pContext->sndio.sio_stop    = sio_stop;
    pContext->sndio.sio_initpar = sio_initpar;
#endif

    pContext->onUninit              = mal_context_uninit__sndio;
    pContext->onDeviceIDEqual       = mal_context_is_device_id_equal__sndio;
    pContext->onEnumDevices         = mal_context_enumerate_devices__sndio;
    pContext->onGetDeviceInfo       = mal_context_get_device_info__sndio;
    pContext->onDeviceInit          = mal_device_init__sndio;
    pContext->onDeviceUninit        = mal_device_uninit__sndio;
    pContext->onDeviceStart         = mal_device__start_backend__sndio;
    pContext->onDeviceStop          = mal_device__stop_backend__sndio;
    pContext->onDeviceBreakMainLoop = mal_device__break_main_loop__sndio;
    pContext->onDeviceMainLoop      = mal_device__main_loop__sndio;

    return MAL_SUCCESS;
}
#endif  // sndio



///////////////////////////////////////////////////////////////////////////////
//
// audio(4) Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_AUDIO4
#include <fcntl.h>
#include <poll.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/audioio.h>

#if defined(__OpenBSD__)
    #include <sys/param.h>
    #if defined(OpenBSD) && OpenBSD >= 201709
        #define MAL_AUDIO4_USE_NEW_API
    #endif
#endif

void mal_construct_device_id__audio4(char* id, size_t idSize, const char* base, int deviceIndex)
{
    mal_assert(id != NULL);
    mal_assert(idSize > 0);
    mal_assert(deviceIndex >= 0);
    
    size_t baseLen = strlen(base);
    mal_assert(idSize > baseLen);
    
    mal_strcpy_s(id, idSize, base);
    mal_itoa_s(deviceIndex, id+baseLen, idSize-baseLen, 10);
}

mal_result mal_extract_device_index_from_id__audio4(const char* id, const char* base, int* pIndexOut)
{
    mal_assert(id != NULL);
    mal_assert(base != NULL);
    mal_assert(pIndexOut != NULL);
    
    size_t idLen = strlen(id);
    size_t baseLen = strlen(base);
    if (idLen <= baseLen) {
        return MAL_ERROR;   // Doesn't look like the id starts with the base.
    }
    
    if (strncmp(id, base, baseLen) != 0) {
        return MAL_ERROR;   // ID does not begin with base.
    }
    
    const char* deviceIndexStr = id + baseLen;
    if (deviceIndexStr[0] == '\0') {
        return MAL_ERROR;   // No index specified in the ID.
    }
    
    if (pIndexOut) {
        *pIndexOut = atoi(deviceIndexStr);
    }
    
    return MAL_SUCCESS;
}

mal_bool32 mal_context_is_device_id_equal__audio4(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return mal_strcmp(pID0->audio4, pID1->audio4) == 0;
}

#if !defined(MAL_AUDIO4_USE_NEW_API)
mal_format mal_format_from_encoding__audio4(unsigned int encoding, unsigned int precision)
{
    if (precision == 8 && (encoding == AUDIO_ENCODING_ULINEAR || encoding == AUDIO_ENCODING_ULINEAR || encoding == AUDIO_ENCODING_ULINEAR_LE || encoding == AUDIO_ENCODING_ULINEAR_BE)) {
        return mal_format_u8;
    } else {
        if (mal_is_little_endian() && encoding == AUDIO_ENCODING_SLINEAR_LE) {
            if (precision == 16) {
                return mal_format_s16;
            } else if (precision == 24) {
                return mal_format_s24;
            } else if (precision == 32) {
                return mal_format_s32;
            }
        } else if (mal_is_big_endian() && encoding == AUDIO_ENCODING_SLINEAR_BE) {
            if (precision == 16) {
                return mal_format_s16;
            } else if (precision == 24) {
                return mal_format_s24;
            } else if (precision == 32) {
                return mal_format_s32;
            }
        }
    }

    return mal_format_unknown;  // Encoding not supported.
}

mal_format mal_format_from_prinfo__audio4(struct audio_prinfo* prinfo)
{
    return mal_format_from_encoding__audio4(prinfo->encoding, prinfo->precision);
}
#else
mal_format mal_format_from_swpar__audio4(struct audio_swpar* par)
{
    if (par->bits == 8 && par->bps == 1 && par->sig == 0) {
        return mal_format_u8;
    }
    if (par->bits == 16 && par->bps == 2 && par->sig == 1 && par->le == mal_is_little_endian()) {
        return mal_format_s16;
    }
    if (par->bits == 24 && par->bps == 3 && par->sig == 1 && par->le == mal_is_little_endian()) {
        return mal_format_s24;
    }
    if (par->bits == 32 && par->bps == 4 && par->sig == 1 && par->le == mal_is_little_endian()) {
        return mal_format_f32;
    }

    // Format not supported.
    return mal_format_unknown;
}
#endif

mal_result mal_context_get_device_info_from_fd__audio4(mal_context* pContext, mal_device_type deviceType, int fd, mal_device_info* pInfoOut)
{
    mal_assert(pContext != NULL);
    mal_assert(fd >= 0);
    mal_assert(pInfoOut != NULL);
    
    (void)pContext;
    (void)deviceType;

    audio_device_t fdDevice;
    if (ioctl(fd, AUDIO_GETDEV, &fdDevice) < 0) {
        return MAL_ERROR;   // Failed to retrieve device info.
    }

    // Name.
    mal_strcpy_s(pInfoOut->name, sizeof(pInfoOut->name), fdDevice.name);

#if !defined(MAL_AUDIO4_USE_NEW_API)
    // Supported formats. We get this by looking at the encodings. 
    int counter = 0;
    for (;;) {
        audio_encoding_t encoding;
        mal_zero_object(&encoding);
        encoding.index = counter;
        if (ioctl(fd, AUDIO_GETENC, &encoding) < 0) {
            break;
        }

        mal_format format = mal_format_from_encoding__audio4(encoding.encoding, encoding.precision);
        if (format != mal_format_unknown) {
            pInfoOut->formats[pInfoOut->formatCount++] = format;
        }

        counter += 1;
    }

    audio_info_t fdInfo;
    if (ioctl(fd, AUDIO_GETINFO, &fdInfo) < 0) {
        return MAL_ERROR;
    }

    if (deviceType == mal_device_type_playback) {
        pInfoOut->minChannels = fdInfo.play.channels; 
        pInfoOut->maxChannels = fdInfo.play.channels;
        pInfoOut->minSampleRate = fdInfo.play.sample_rate;
        pInfoOut->maxSampleRate = fdInfo.play.sample_rate;
    } else {
        pInfoOut->minChannels = fdInfo.record.channels;
        pInfoOut->maxChannels = fdInfo.record.channels;
        pInfoOut->minSampleRate = fdInfo.record.sample_rate;
        pInfoOut->maxSampleRate = fdInfo.record.sample_rate;
    }
#else
    struct audio_swpar fdPar;
    if (ioctl(fd, AUDIO_GETPAR, &fdPar) < 0) {
        return MAL_ERROR;
    }
    
    mal_format format = mal_format_from_swpar__audio4(&fdPar);
    if (format == mal_format_unknown) {
        return MAL_FORMAT_NOT_SUPPORTED;
    }
    pInfoOut->formats[pInfoOut->formatCount++] = format;
    
    if (deviceType == mal_device_type_playback) {
        pInfoOut->minChannels = fdPar.pchan;
        pInfoOut->maxChannels = fdPar.pchan;
    } else {
        pInfoOut->minChannels = fdPar.rchan;
        pInfoOut->maxChannels = fdPar.rchan;
    }
    
    pInfoOut->minSampleRate = fdPar.rate;
    pInfoOut->maxSampleRate = fdPar.rate;
#endif
    
    return MAL_SUCCESS;
}

mal_result mal_context_enumerate_devices__audio4(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);
    
    const int maxDevices = 64;
    
    // Every device will be named "/dev/audioN", with a "/dev/audioctlN" equivalent. We use the "/dev/audioctlN"
    // version here since we can open it even when another process has control of the "/dev/audioN" device.
    char devpath[256];
    for (int iDevice = 0; iDevice < maxDevices; ++iDevice) {
        mal_strcpy_s(devpath, sizeof(devpath), "/dev/audioctl");
        mal_itoa_s(iDevice, devpath+strlen(devpath), sizeof(devpath)-strlen(devpath), 10);
    
        struct stat st;
        if (stat(devpath, &st) < 0) {
            break;
        }

        // The device exists, but we need to check if it's usable as playback and/or capture.
        int fd;
        mal_bool32 isTerminating = MAL_FALSE;
        
        // Playback.
        if (!isTerminating) {
            fd = open(devpath, O_RDONLY, 0);
            if (fd >= 0) {
                // Supports playback.
                mal_device_info deviceInfo;
                mal_zero_object(&deviceInfo);
                mal_construct_device_id__audio4(deviceInfo.id.audio4, sizeof(deviceInfo.id.audio4), "/dev/audio", iDevice);
                if (mal_context_get_device_info_from_fd__audio4(pContext, mal_device_type_playback, fd, &deviceInfo) == MAL_SUCCESS) {
                    isTerminating = !callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
                }
                
                close(fd);
            }
        }
        
        // Capture.
        if (!isTerminating) {
            fd = open(devpath, O_WRONLY, 0);
            if (fd >= 0) {
                // Supports capture.
                mal_device_info deviceInfo;
                mal_zero_object(&deviceInfo);
                mal_construct_device_id__audio4(deviceInfo.id.audio4, sizeof(deviceInfo.id.audio4), "/dev/audio", iDevice);
                if (mal_context_get_device_info_from_fd__audio4(pContext, mal_device_type_capture, fd, &deviceInfo) == MAL_SUCCESS) {
                    isTerminating = !callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
                }
                
                close(fd);
            }
        }
        
        if (isTerminating) {
            break;
        }
    }
    
    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info__audio4(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    (void)shareMode;
    
    // We need to open the "/dev/audioctlN" device to get the info. To do this we need to extract the number
    // from the device ID which will be in "/dev/audioN" format.
    int fd = -1;
    int deviceIndex = -1;
    char ctlid[256];
    if (pDeviceID == NULL) {
        // Default device.
        mal_strcpy_s(ctlid, sizeof(ctlid), "/dev/audioctl");
    } else {
        // Specific device. We need to convert from "/dev/audioN" to "/dev/audioctlN".
        mal_result result = mal_extract_device_index_from_id__audio4(pDeviceID->audio4, "/dev/audio", &deviceIndex);
        if (result != MAL_SUCCESS) {
            return result;
        }
        
        mal_construct_device_id__audio4(ctlid, sizeof(ctlid), "/dev/audioctl", deviceIndex);
    }
    
    fd = open(ctlid, (deviceType == mal_device_type_playback) ? O_WRONLY : O_RDONLY, 0);
    if (fd == -1) {
        return MAL_NO_DEVICE;
    }
    
    if (deviceIndex == -1) {
        mal_strcpy_s(pDeviceInfo->id.audio4, sizeof(pDeviceInfo->id.audio4), "/dev/audio");
    } else {
        mal_construct_device_id__audio4(pDeviceInfo->id.audio4, sizeof(pDeviceInfo->id.audio4), "/dev/audio", deviceIndex);
    }
    
    mal_result result = mal_context_get_device_info_from_fd__audio4(pContext, deviceType, fd, pDeviceInfo);
    
    close(fd);
    return result;
}

void mal_device_uninit__audio4(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    close(pDevice->audio4.fd);
    mal_free(pDevice->audio4.pIntermediaryBuffer);
}

mal_result mal_device_init__audio4(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pContext;

    mal_assert(pDevice != NULL);
    mal_zero_object(&pDevice->audio4);
    pDevice->audio4.fd = -1;
    
    // The first thing to do is open the file.
    const char* deviceName = "/dev/audio";
    if (pDeviceID != NULL) {
        deviceName = pDeviceID->audio4;
    }
    
    pDevice->audio4.fd = open(deviceName, ((deviceType == mal_device_type_playback) ? O_WRONLY : O_RDONLY) | O_NONBLOCK, 0);
    if (pDevice->audio4.fd == -1) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to open device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

#if !defined(MAL_AUDIO4_USE_NEW_API)
    audio_info_t fdInfo;
    AUDIO_INITINFO(&fdInfo);

    struct audio_prinfo* prinfo;
    if (deviceType == mal_device_type_playback) {
        prinfo = &fdInfo.play;
        fdInfo.mode = AUMODE_PLAY;
    } else {
        prinfo = &fdInfo.record;
        fdInfo.mode = AUMODE_RECORD;
    }

    // Format. Note that it looks like audio4 does not support floating point formats. In this case
    // we just fall back to s16.
    switch (pDevice->format)
    {
        case mal_format_u8:
        {
            prinfo->encoding = AUDIO_ENCODING_ULINEAR;
            prinfo->precision = 8;
        } break;

        case mal_format_s24:
        {
            prinfo->encoding = (mal_is_little_endian()) ? AUDIO_ENCODING_SLINEAR_LE : AUDIO_ENCODING_SLINEAR_BE;
            prinfo->precision = 24;
        } break;

        case mal_format_s32:
        {
            prinfo->encoding = (mal_is_little_endian()) ? AUDIO_ENCODING_SLINEAR_LE : AUDIO_ENCODING_SLINEAR_BE;
            prinfo->precision = 32;
        } break;

        case mal_format_s16:
        case mal_format_f32:
        default:
        {
            prinfo->encoding = (mal_is_little_endian()) ? AUDIO_ENCODING_SLINEAR_LE : AUDIO_ENCODING_SLINEAR_BE;
            prinfo->precision = 16;
        } break;
    }

    // We always want to the use the devices native channel count and sample rate.
    mal_device_info nativeInfo;
    mal_result result = mal_context_get_device_info(pContext, deviceType, pDeviceID, pConfig->shareMode, &nativeInfo);
    if (result != MAL_SUCCESS) {
        close(pDevice->audio4.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to retrieve device format.", result);
    }

    prinfo->channels = nativeInfo.maxChannels;
    prinfo->sample_rate = nativeInfo.maxSampleRate;
    
    // We need to apply the settings so far so we can get back the actual sample rate which we need for calculating
    // the default buffer size below.
    if (ioctl(pDevice->audio4.fd, AUDIO_SETINFO, &fdInfo) < 0) {
        close(pDevice->audio4.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to set device format. AUDIO_SETINFO failed.", MAL_FORMAT_NOT_SUPPORTED);
    }
    
    if (ioctl(pDevice->audio4.fd, AUDIO_GETINFO, &fdInfo) < 0) {
        close(pDevice->audio4.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] AUDIO_GETINFO failed.", MAL_FORMAT_NOT_SUPPORTED);
    }
    
    pDevice->internalFormat = mal_format_from_prinfo__audio4(prinfo);
    if (pDevice->internalFormat == mal_format_unknown) {
        close(pDevice->audio4.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] The device's internal device format is not supported by mini_al. The device is unusable.", MAL_FORMAT_NOT_SUPPORTED);
    }

    pDevice->internalChannels = prinfo->channels;
    pDevice->internalSampleRate = prinfo->sample_rate;



    // Try calculating an appropriate default buffer size.
    if (pDevice->bufferSizeInFrames == 0) {
        pDevice->bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, pDevice->internalSampleRate);
    }

    // What mini_al calls a fragment, audio4 calls a block.
    mal_uint32 fragmentSizeInBytes = pDevice->bufferSizeInFrames * mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels);
    if (fragmentSizeInBytes < 16) {
        fragmentSizeInBytes = 16;
    }

    
    AUDIO_INITINFO(&fdInfo);
    fdInfo.hiwat = mal_max(pDevice->periods, 5);
    fdInfo.lowat = (unsigned int)(fdInfo.hiwat * 0.75);
    fdInfo.blocksize = fragmentSizeInBytes / fdInfo.hiwat;
    if (ioctl(pDevice->audio4.fd, AUDIO_SETINFO, &fdInfo) < 0) {
        close(pDevice->audio4.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to set internal buffer size. AUDIO_SETINFO failed.", MAL_FORMAT_NOT_SUPPORTED);
    }
    
    pDevice->periods = fdInfo.hiwat;
    pDevice->bufferSizeInFrames = (fdInfo.blocksize * fdInfo.hiwat) / mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels);
    pDevice->audio4.fragmentSizeInFrames = fdInfo.blocksize / mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels);
#else
    // We need to retrieve the format of the device so we can know the channel count and sample rate. Then we
    // can calculate the buffer size.
    struct audio_swpar fdPar;
    if (ioctl(pDevice->audio4.fd, AUDIO_GETPAR, &fdPar) < 0) {
        close(pDevice->audio4.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to retrieve initial device parameters.", MAL_FORMAT_NOT_SUPPORTED);
    }
    
    // Set the initial internal formats so we can do calculations below.
    pDevice->internalFormat = mal_format_from_swpar__audio4(&fdPar);
    if (deviceType == mal_device_type_playback) {
        pDevice->internalChannels = fdPar.pchan;
    } else {
        pDevice->internalChannels = fdPar.rchan;
    }
    pDevice->internalSampleRate = fdPar.rate;
    
    if (pDevice->bufferSizeInFrames == 0) {
        pDevice->bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, pDevice->internalSampleRate);
    }
    
    // What mini_al calls a fragment, audio4 calls a block.
    mal_uint32 bufferSizeInBytes = pDevice->bufferSizeInFrames * mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels);
    if (bufferSizeInBytes < 16) {
        bufferSizeInBytes = 16;
    }
    
    fdPar.nblks = pDevice->periods;
    fdPar.round = bufferSizeInBytes / fdPar.nblks;
    
    if (ioctl(pDevice->audio4.fd, AUDIO_SETPAR, &fdPar) < 0) {
        close(pDevice->audio4.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to set device parameters.", MAL_FORMAT_NOT_SUPPORTED);
    }
    
    if (ioctl(pDevice->audio4.fd, AUDIO_GETPAR, &fdPar) < 0) {
        close(pDevice->audio4.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to retrieve actual device parameters.", MAL_FORMAT_NOT_SUPPORTED);
    }
    
    pDevice->internalFormat = mal_format_from_swpar__audio4(&fdPar);
    if (deviceType == mal_device_type_playback) {
        pDevice->internalChannels = fdPar.pchan;
    } else {
        pDevice->internalChannels = fdPar.rchan;
    }
    pDevice->internalSampleRate = fdPar.rate;
    
    pDevice->periods = fdPar.nblks;
    pDevice->bufferSizeInFrames = (fdPar.nblks * fdPar.round) / mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels);
    pDevice->audio4.fragmentSizeInFrames = fdPar.round / mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels);
#endif


    // For the channel map, I'm not sure how to query the channel map (or if it's even possible). I'm just
    // using the channels defined in FreeBSD's sound(4) man page.
    mal_get_standard_channel_map(mal_standard_channel_map_sound4, pDevice->internalChannels, pDevice->internalChannelMap);


    // The version of the operating system dictates whether or not the device is exclusive or shared. NetBSD
    // introduced in-kernel mixing which means it's shared. All other BSD flavours are exclusive as far as
    // I'm aware.
#if defined(__NetBSD_Version__) && __NetBSD_Version__ >= 800000000
    pDevice->exclusiveMode = MAL_FALSE;
#else
    pDevice->exclusiveMode = MAL_TRUE;
#endif

    
    // When not using MMAP mode we need to use an intermediary buffer to the data transfer between the client
    // and device. Everything is done by the size of a fragment.
    pDevice->audio4.pIntermediaryBuffer = mal_malloc(pDevice->audio4.fragmentSizeInFrames * mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels));
    if (pDevice->audio4.pIntermediaryBuffer == NULL) {
        close(pDevice->audio4.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to allocate memory for intermediary buffer.", MAL_OUT_OF_MEMORY);
    }

    
    return MAL_SUCCESS;
}

mal_result mal_device__start_backend__audio4(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->audio4.fd == -1) {
        return MAL_INVALID_ARGS;
    }

    // The device is started by the next calls to read() and write(). For playback it's simple - just read
    // data from the client, then write it to the device with write() which will in turn start the device.
    // For capture it's a bit less intuitive - we do nothing (it'll be started automatically by the first
    // call to read().
    if (pDevice->type == mal_device_type_playback) {
        // Playback. Need to load the entire buffer, which means we need to write a fragment for each period.
        for (mal_uint32 iPeriod = 0; iPeriod < pDevice->periods; iPeriod += 1) { 
            mal_device__read_frames_from_client(pDevice, pDevice->audio4.fragmentSizeInFrames, pDevice->audio4.pIntermediaryBuffer);

            int bytesWritten = write(pDevice->audio4.fd, pDevice->audio4.pIntermediaryBuffer, pDevice->audio4.fragmentSizeInFrames * mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels));
            if (bytesWritten == -1) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to send initial chunk of data to the device.", MAL_FAILED_TO_SEND_DATA_TO_DEVICE);
            }
        }
    } else {
        // Capture. Do nothing.
    }

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__audio4(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->audio4.fd == -1) {
        return MAL_INVALID_ARGS;
    }

#if !defined(MAL_AUDIO4_USE_NEW_API)
    if (ioctl(pDevice->audio4.fd, AUDIO_FLUSH, 0) < 0) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to stop device. AUDIO_FLUSH failed.", MAL_FAILED_TO_STOP_BACKEND_DEVICE);
    }
#else
    if (ioctl(pDevice->audio4.fd, AUDIO_STOP, 0) < 0) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to stop device. AUDIO_STOP failed.", MAL_FAILED_TO_STOP_BACKEND_DEVICE);
    }
#endif

    return MAL_SUCCESS;
}

mal_result mal_device__break_main_loop__audio4(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->audio4.breakFromMainLoop = MAL_TRUE;
    return MAL_SUCCESS;
}

mal_result mal_device__wait__audio4(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    struct pollfd fds[1];
    fds[0].fd     = pDevice->audio4.fd;
    fds[0].events = (pDevice->type == mal_device_type_playback) ? (POLLOUT | POLLWRBAND) : (POLLIN | POLLPRI);
    int timeout = 2 * 1000;
    int ioresult = poll(fds, mal_countof(fds), timeout);
    if (ioresult < 0) {
    #ifdef MAL_DEBUG_OUTPUT
        printf("poll() failed: timeout=%d, ioresult=%d\n", pDevice->bufferSizeInMilliseconds, ioresult);
    #endif
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to wait for device.", MAL_ERROR);
    }

    // Check for a timeout. This has been annoying in my testing. In my testing, when the device is unplugged it will just
    // hang on the next calls to write(), ioctl(), etc. The only way I have figured out how to handle this is to wait for
    // a timeout from poll(). In the unplugging case poll() will timeout, however there's no indication that the device is
    // unusable - no flags are set, no errors are reported, nothing. To work around this I have decided to outright fail
    // in the event of a timeout.
    if (ioresult == 0) {
        // Check for errors.
        if ((fds[0].revents & (POLLERR | POLLNVAL)) != 0) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to wait for device.", MAL_NO_DEVICE);
        }
        if ((fds[0].revents & (POLLHUP)) != 0) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to wait for device. Disconnected.", MAL_NO_DEVICE);
        }

        // A return value of 0 from poll indicates a timeout.
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Timeout while waiting for device.", MAL_TIMEOUT);
    }

    mal_assert(ioresult > 0);
    return MAL_SUCCESS;
}

mal_result mal_device__main_loop__audio4(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->audio4.breakFromMainLoop = MAL_FALSE;
    while (!pDevice->audio4.breakFromMainLoop) {
        // Break from the main loop if the device isn't started anymore. Likely what's happened is the application
        // has requested that the device be stopped.
        if (!mal_device_is_started(pDevice)) {
            break;
        }

        if (pDevice->type == mal_device_type_playback) {
            // Playback.
            mal_device__read_frames_from_client(pDevice, pDevice->audio4.fragmentSizeInFrames, pDevice->audio4.pIntermediaryBuffer);

            // Wait for data to become available.
            mal_result result = mal_device__wait__audio4(pDevice);
            if (result != MAL_SUCCESS) {
                return result;
            }

            size_t bytesToWrite = pDevice->audio4.fragmentSizeInFrames * mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels);
            while (bytesToWrite > 0) {
                ssize_t bytesWritten = write(pDevice->audio4.fd, pDevice->audio4.pIntermediaryBuffer, bytesToWrite);
                if (bytesWritten < 0) {
                    return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to send data from the client to the device.", MAL_FAILED_TO_SEND_DATA_TO_DEVICE);
                }

                if (bytesWritten == 0) {
                    return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to write any data to the device.", MAL_FAILED_TO_SEND_DATA_TO_DEVICE);
                }

                bytesToWrite -= bytesWritten;
            }
        } else {
            // Capture.
            mal_result result = mal_device__wait__audio4(pDevice);
            if (result != MAL_SUCCESS) {
                return result;
            }

            size_t totalBytesRead = 0;
            size_t bytesToRead = pDevice->audio4.fragmentSizeInFrames * mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels);
            while (bytesToRead > 0) {
                ssize_t bytesRead = read(pDevice->audio4.fd, pDevice->audio4.pIntermediaryBuffer, bytesToRead);
                if (bytesRead < 0) { 
                    if (errno == EAGAIN) {
                        break;
                    }

                    return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to read data from the device to be sent to the client.", MAL_FAILED_TO_READ_DATA_FROM_DEVICE);
                }

                if (bytesRead == 0) {
                    return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[audio4] Failed to read any data from the device.", MAL_FAILED_TO_READ_DATA_FROM_DEVICE);
                }

                bytesToRead -= bytesRead;
                totalBytesRead += bytesRead;
            }

            mal_uint32 framesRead = (mal_uint32)totalBytesRead / mal_get_bytes_per_frame(pDevice->internalFormat, pDevice->internalChannels);
            mal_device__send_frames_to_client(pDevice, framesRead, pDevice->audio4.pIntermediaryBuffer);
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_context_uninit__audio4(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_audio4);

    (void)pContext;
    return MAL_SUCCESS;
}

mal_result mal_context_init__audio4(mal_context* pContext)
{
    mal_assert(pContext != NULL);

    pContext->onUninit              = mal_context_uninit__audio4;
    pContext->onDeviceIDEqual       = mal_context_is_device_id_equal__audio4;
    pContext->onEnumDevices         = mal_context_enumerate_devices__audio4;
    pContext->onGetDeviceInfo       = mal_context_get_device_info__audio4;
    pContext->onDeviceInit          = mal_device_init__audio4;
    pContext->onDeviceUninit        = mal_device_uninit__audio4;
    pContext->onDeviceStart         = mal_device__start_backend__audio4;
    pContext->onDeviceStop          = mal_device__stop_backend__audio4;
    pContext->onDeviceBreakMainLoop = mal_device__break_main_loop__audio4;
    pContext->onDeviceMainLoop      = mal_device__main_loop__audio4;

    return MAL_SUCCESS;
}
#endif  // audio4


///////////////////////////////////////////////////////////////////////////////
//
// OSS Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_OSS
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/soundcard.h>

#ifndef SNDCTL_DSP_HALT
#define SNDCTL_DSP_HALT SNDCTL_DSP_RESET
#endif

int mal_open_temp_device__oss()
{
    // The OSS sample code uses "/dev/mixer" as the device for getting system properties so I'm going to do the same.
    int fd = open("/dev/mixer", O_RDONLY, 0);
    if (fd >= 0) {
        return fd;
    }

    return -1;
}

mal_result mal_context_open_device__oss(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, int* pfd)
{
    mal_assert(pContext != NULL);
    mal_assert(pfd != NULL);
    (void)pContext;

    *pfd = -1;

    char deviceName[64];
    if (pDeviceID != NULL) {
        mal_strncpy_s(deviceName, sizeof(deviceName), pDeviceID->oss, (size_t)-1);
    } else {
        mal_strncpy_s(deviceName, sizeof(deviceName), "/dev/dsp", (size_t)-1);
    }

    *pfd = open(deviceName, (type == mal_device_type_playback) ? O_WRONLY : O_RDONLY, 0);
    if (*pfd == -1) {
        return MAL_FAILED_TO_OPEN_BACKEND_DEVICE;
    }

    return MAL_SUCCESS;
}

mal_bool32 mal_context_is_device_id_equal__oss(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return mal_strcmp(pID0->oss, pID1->oss) == 0;
}

mal_result mal_context_enumerate_devices__oss(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    int fd = mal_open_temp_device__oss();
    if (fd == -1) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to open a temporary device for retrieving system information used for device enumeration.", MAL_NO_BACKEND);
    }

    oss_sysinfo si;
    int result = ioctl(fd, SNDCTL_SYSINFO, &si);
    if (result != -1) {
        for (int iAudioDevice = 0; iAudioDevice < si.numaudios; ++iAudioDevice) {
            oss_audioinfo ai;
            ai.dev = iAudioDevice;
            result = ioctl(fd, SNDCTL_AUDIOINFO, &ai);
            if (result != -1) {
                if (ai.devnode[0] != '\0') {    // <-- Can be blank, according to documentation.
                    mal_device_info deviceInfo;
                    mal_zero_object(&deviceInfo);

                    // ID
                    mal_strncpy_s(deviceInfo.id.oss, sizeof(deviceInfo.id.oss), ai.devnode, (size_t)-1);

                    // The human readable device name should be in the "ai.handle" variable, but it can
                    // sometimes be empty in which case we just fall back to "ai.name" which is less user
                    // friendly, but usually has a value.
                    if (ai.handle[0] != '\0') {
                        mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), ai.handle, (size_t)-1);
                    } else {
                        mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), ai.name, (size_t)-1);
                    }

                    // The device can be both playback and capture.
                    mal_bool32 isTerminating = MAL_FALSE;
                    if (!isTerminating && (ai.caps & PCM_CAP_OUTPUT) != 0) {
                        isTerminating = !callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
                    }
                    if (!isTerminating && (ai.caps & PCM_CAP_INPUT) != 0) {
                        isTerminating = !callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
                    }

                    if (isTerminating) {
                        break;
                    }
                }
            }
        }
    } else {
        close(fd);
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to retrieve system information for device enumeration.", MAL_NO_BACKEND);
    }

    close(fd);
    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info__oss(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    (void)shareMode;

    // Handle the default device a little differently.
    if (pDeviceID == NULL) {
        if (deviceType == mal_device_type_playback) {
            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        } else {
            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        }

        return MAL_SUCCESS;
    }


    // If we get here it means we are _not_ using the default device.
    mal_bool32 foundDevice = MAL_FALSE;

    int fdTemp = mal_open_temp_device__oss();
    if (fdTemp == -1) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to open a temporary device for retrieving system information used for device enumeration.", MAL_NO_BACKEND);
    }

    oss_sysinfo si;
    int result = ioctl(fdTemp, SNDCTL_SYSINFO, &si);
    if (result != -1) {
        for (int iAudioDevice = 0; iAudioDevice < si.numaudios; ++iAudioDevice) {
            oss_audioinfo ai;
            ai.dev = iAudioDevice;
            result = ioctl(fdTemp, SNDCTL_AUDIOINFO, &ai);
            if (result != -1) {
                if (mal_strcmp(ai.devnode, pDeviceID->oss) == 0) {
                    // It has the same name, so now just confirm the type.
                    if ((deviceType == mal_device_type_playback && ((ai.caps & PCM_CAP_OUTPUT) != 0)) ||
                        (deviceType == mal_device_type_capture  && ((ai.caps & PCM_CAP_INPUT)  != 0))) {
                        // ID
                        mal_strncpy_s(pDeviceInfo->id.oss, sizeof(pDeviceInfo->id.oss), ai.devnode, (size_t)-1);

                        // The human readable device name should be in the "ai.handle" variable, but it can
                        // sometimes be empty in which case we just fall back to "ai.name" which is less user
                        // friendly, but usually has a value.
                        if (ai.handle[0] != '\0') {
                            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), ai.handle, (size_t)-1);
                        } else {
                            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), ai.name, (size_t)-1);
                        }

                        pDeviceInfo->minChannels = ai.min_channels;
                        pDeviceInfo->maxChannels = ai.max_channels;
                        pDeviceInfo->minSampleRate = ai.min_rate;
                        pDeviceInfo->maxSampleRate = ai.max_rate;
                        pDeviceInfo->formatCount = 0;

                        unsigned int formatMask;
                        if (deviceType == mal_device_type_playback) {
                            formatMask = ai.oformats;
                        } else {
                            formatMask = ai.iformats;
                        }

                        if ((formatMask & AFMT_U8) != 0) {
                            pDeviceInfo->formats[pDeviceInfo->formatCount++] = mal_format_u8;
                        }
                        if (((formatMask & AFMT_S16_LE) != 0 && mal_is_little_endian()) || (AFMT_S16_BE && mal_is_big_endian())) {
                            pDeviceInfo->formats[pDeviceInfo->formatCount++] = mal_format_s16;
                        }
                        if (((formatMask & AFMT_S32_LE) != 0 && mal_is_little_endian()) || (AFMT_S32_BE && mal_is_big_endian())) {
                            pDeviceInfo->formats[pDeviceInfo->formatCount++] = mal_format_s32;
                        }

                        foundDevice = MAL_TRUE;
                        break;
                    }
                }
            }
        }
    } else {
        close(fdTemp);
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to retrieve system information for device enumeration.", MAL_NO_BACKEND);
    }


    close(fdTemp);

    if (!foundDevice) {
        return MAL_NO_DEVICE;
    }


    return MAL_SUCCESS;
}

void mal_device_uninit__oss(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    close(pDevice->oss.fd);
    mal_free(pDevice->oss.pIntermediaryBuffer);
}

mal_result mal_device_init__oss(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pContext;

    mal_assert(pDevice != NULL);
    mal_zero_object(&pDevice->oss);

    mal_result result = mal_context_open_device__oss(pContext, type, pDeviceID, &pDevice->oss.fd);
    if (result != MAL_SUCCESS) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to open device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    // The OSS documantation is very clear about the order we should be initializing the device's properties:
    //   1) Format
    //   2) Channels
    //   3) Sample rate.

    // Format.
    int ossFormat = AFMT_U8;
    switch (pDevice->format) {
        case mal_format_s16: ossFormat = (mal_is_little_endian()) ? AFMT_S16_LE : AFMT_S16_BE; break;
        case mal_format_s24: ossFormat = (mal_is_little_endian()) ? AFMT_S32_LE : AFMT_S32_BE; break;
        case mal_format_s32: ossFormat = (mal_is_little_endian()) ? AFMT_S32_LE : AFMT_S32_BE; break;
        case mal_format_f32: ossFormat = (mal_is_little_endian()) ? AFMT_S32_LE : AFMT_S32_BE; break;
        case mal_format_u8:
        default: ossFormat = AFMT_U8; break;
    }
    int ossResult = ioctl(pDevice->oss.fd, SNDCTL_DSP_SETFMT, &ossFormat);
    if (ossResult == -1) {
        close(pDevice->oss.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to set format.", MAL_FORMAT_NOT_SUPPORTED);
    }

    if (ossFormat == AFMT_U8) {
        pDevice->internalFormat = mal_format_u8;
    } else {
        if (mal_is_little_endian()) {
            switch (ossFormat) {
                case AFMT_S16_LE: pDevice->internalFormat = mal_format_s16; break;
                case AFMT_S32_LE: pDevice->internalFormat = mal_format_s32; break;
                default: mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] The device's internal format is not supported by mini_al.", MAL_FORMAT_NOT_SUPPORTED);
            }
        } else {
            switch (ossFormat) {
                case AFMT_S16_BE: pDevice->internalFormat = mal_format_s16; break;
                case AFMT_S32_BE: pDevice->internalFormat = mal_format_s32; break;
                default: mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] The device's internal format is not supported by mini_al.", MAL_FORMAT_NOT_SUPPORTED);
            }
        }
    }

    // Channels.
    int ossChannels = (int)pConfig->channels;
    ossResult = ioctl(pDevice->oss.fd, SNDCTL_DSP_CHANNELS, &ossChannels);
    if (ossResult == -1) {
        close(pDevice->oss.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to set channel count.", MAL_FORMAT_NOT_SUPPORTED);
    }

    pDevice->internalChannels = ossChannels;


    // Sample rate.
    int ossSampleRate = (int)pConfig->sampleRate;
    ossResult = ioctl(pDevice->oss.fd, SNDCTL_DSP_SPEED, &ossSampleRate);
    if (ossResult == -1) {
        close(pDevice->oss.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to set sample rate.", MAL_FORMAT_NOT_SUPPORTED);
    }

    pDevice->internalSampleRate = ossSampleRate;


    // Try calculating an appropriate default buffer size.
    if (pDevice->bufferSizeInFrames == 0) {
        pDevice->bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, pDevice->internalSampleRate);
    }

    // The documentation says that the fragment settings should be set as soon as possible, but I'm not sure if
    // it should be done before or after format/channels/rate.
    //
    // OSS wants the fragment size in bytes and a power of 2. When setting, we specify the power, not the actual
    // value.
    mal_uint32 fragmentSizeInBytes = mal_round_to_power_of_2(pDevice->bufferSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat));
    if (fragmentSizeInBytes < 16) {
        fragmentSizeInBytes = 16;
    }

    mal_uint32 ossFragmentSizePower = 4;
    fragmentSizeInBytes >>= 4;
    while (fragmentSizeInBytes >>= 1) {
        ossFragmentSizePower += 1;
    }

    int ossFragment = (int)((pDevice->periods << 16) | ossFragmentSizePower);
    ossResult = ioctl(pDevice->oss.fd, SNDCTL_DSP_SETFRAGMENT, &ossFragment);
    if (ossResult == -1) {
        close(pDevice->oss.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to set fragment size and period count.", MAL_FORMAT_NOT_SUPPORTED);
    }

    int actualFragmentSizeInBytes = 1 << (ossFragment & 0xFFFF);
    pDevice->oss.fragmentSizeInFrames = actualFragmentSizeInBytes / mal_get_bytes_per_sample(pDevice->internalFormat) / pDevice->internalChannels;

    pDevice->periods = (mal_uint32)(ossFragment >> 16);
    pDevice->bufferSizeInFrames = (mal_uint32)(pDevice->oss.fragmentSizeInFrames * pDevice->periods);

    // Set the internal channel map. Not sure if this can be queried. For now just using the channel layouts defined in FreeBSD's sound(4) man page.
    mal_get_standard_channel_map(mal_standard_channel_map_sound4, pDevice->internalChannels, pDevice->internalChannelMap);

    // OSS seems to be shared.
    pDevice->exclusiveMode = MAL_FALSE;

    // When not using MMAP mode, we need to use an intermediary buffer for the client <-> device transfer. We do
    // everything by the size of a fragment.
    pDevice->oss.pIntermediaryBuffer = mal_malloc(actualFragmentSizeInBytes);
    if (pDevice->oss.pIntermediaryBuffer == NULL) {
        close(pDevice->oss.fd);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to allocate memory for intermediary buffer.", MAL_OUT_OF_MEMORY);
    }

    return MAL_SUCCESS;
}


mal_result mal_device__start_backend__oss(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // The device is started by the next calls to read() and write(). For playback it's simple - just read
    // data from the client, then write it to the device with write() which will in turn start the device.
    // For capture it's a bit less intuitive - we do nothing (it'll be started automatically by the first
    // call to read().
    if (pDevice->type == mal_device_type_playback) {
        // Playback.
        mal_device__read_frames_from_client(pDevice, pDevice->oss.fragmentSizeInFrames, pDevice->oss.pIntermediaryBuffer);

        int bytesWritten = write(pDevice->oss.fd, pDevice->oss.pIntermediaryBuffer, pDevice->oss.fragmentSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat));
        if (bytesWritten == -1) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to send initial chunk of data to the device.", MAL_FAILED_TO_SEND_DATA_TO_DEVICE);
        }
    } else {
        // Capture. Do nothing.
    }

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__oss(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // We want to use SNDCTL_DSP_HALT. From the documentation:
    //
    //   In multithreaded applications SNDCTL_DSP_HALT (SNDCTL_DSP_RESET) must only be called by the thread
    //   that actually reads/writes the audio device. It must not be called by some master thread to kill the
    //   audio thread. The audio thread will not stop or get any kind of notification that the device was
    //   stopped by the master thread. The device gets stopped but the next read or write call will silently
    //   restart the device.
    //
    // This is actually safe in our case, because this function is only ever called from within our worker
    // thread anyway. Just keep this in mind, though...

    int result = ioctl(pDevice->oss.fd, SNDCTL_DSP_HALT, 0);
    if (result == -1) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to stop device. SNDCTL_DSP_HALT failed.", MAL_FAILED_TO_STOP_BACKEND_DEVICE);
    }

    return MAL_SUCCESS;
}

mal_result mal_device__break_main_loop__oss(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->oss.breakFromMainLoop = MAL_TRUE;
    return MAL_SUCCESS;
}

mal_result mal_device__main_loop__oss(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->oss.breakFromMainLoop = MAL_FALSE;
    while (!pDevice->oss.breakFromMainLoop) {
        // Break from the main loop if the device isn't started anymore. Likely what's happened is the application
        // has requested that the device be stopped.
        if (!mal_device_is_started(pDevice)) {
            break;
        }

        if (pDevice->type == mal_device_type_playback) {
            // Playback.
            mal_device__read_frames_from_client(pDevice, pDevice->oss.fragmentSizeInFrames, pDevice->oss.pIntermediaryBuffer);

            int bytesWritten = write(pDevice->oss.fd, pDevice->oss.pIntermediaryBuffer, pDevice->oss.fragmentSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat));
            if (bytesWritten < 0) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to send data from the client to the device.", MAL_FAILED_TO_SEND_DATA_TO_DEVICE);
            }
        } else {
            // Capture.
            int bytesRead = read(pDevice->oss.fd, pDevice->oss.pIntermediaryBuffer, pDevice->oss.fragmentSizeInFrames * mal_get_bytes_per_sample(pDevice->internalFormat));
            if (bytesRead < 0) {
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to read data from the device to be sent to the client.", MAL_FAILED_TO_READ_DATA_FROM_DEVICE);
            }

            mal_uint32 framesRead = (mal_uint32)bytesRead / pDevice->internalChannels / mal_get_bytes_per_sample(pDevice->internalFormat);
            mal_device__send_frames_to_client(pDevice, framesRead, pDevice->oss.pIntermediaryBuffer);
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_context_uninit__oss(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_oss);

    (void)pContext;
    return MAL_SUCCESS;
}

mal_result mal_context_init__oss(mal_context* pContext)
{
    mal_assert(pContext != NULL);

    // Try opening a temporary device first so we can get version information. This is closed at the end.
    int fd = mal_open_temp_device__oss();
    if (fd == -1) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to open temporary device for retrieving system properties.", MAL_NO_BACKEND);   // Looks liks OSS isn't installed, or there are no available devices.
    }

    // Grab the OSS version.
    int ossVersion = 0;
    int result = ioctl(fd, OSS_GETVERSION, &ossVersion);
    if (result == -1) {
        close(fd);
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[OSS] Failed to retrieve OSS version.", MAL_NO_BACKEND);
    }

    pContext->oss.versionMajor = ((ossVersion & 0xFF0000) >> 16);
    pContext->oss.versionMinor = ((ossVersion & 0x00FF00) >> 8);

    pContext->onUninit              = mal_context_uninit__oss;
    pContext->onDeviceIDEqual       = mal_context_is_device_id_equal__oss;
    pContext->onEnumDevices         = mal_context_enumerate_devices__oss;
    pContext->onGetDeviceInfo       = mal_context_get_device_info__oss;
    pContext->onDeviceInit          = mal_device_init__oss;
    pContext->onDeviceUninit        = mal_device_uninit__oss;
    pContext->onDeviceStart         = mal_device__start_backend__oss;
    pContext->onDeviceStop          = mal_device__stop_backend__oss;
    pContext->onDeviceBreakMainLoop = mal_device__break_main_loop__oss;
    pContext->onDeviceMainLoop      = mal_device__main_loop__oss;

    close(fd);
    return MAL_SUCCESS;
}
#endif  // OSS


///////////////////////////////////////////////////////////////////////////////
//
// OpenSL|ES Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_OPENSL
#include <SLES/OpenSLES.h>
#ifdef MAL_ANDROID
#include <SLES/OpenSLES_Android.h>
#endif

// OpenSL|ES has one-per-application objects :(
SLObjectItf g_malEngineObjectSL = NULL;
SLEngineItf g_malEngineSL = NULL;
mal_uint32 g_malOpenSLInitCounter = 0;

#define MAL_OPENSL_OBJ(p)         (*((SLObjectItf)(p)))
#define MAL_OPENSL_OUTPUTMIX(p)   (*((SLOutputMixItf)(p)))
#define MAL_OPENSL_PLAY(p)        (*((SLPlayItf)(p)))
#define MAL_OPENSL_RECORD(p)      (*((SLRecordItf)(p)))

#ifdef MAL_ANDROID
#define MAL_OPENSL_BUFFERQUEUE(p) (*((SLAndroidSimpleBufferQueueItf)(p)))
#else
#define MAL_OPENSL_BUFFERQUEUE(p) (*((SLBufferQueueItf)(p)))
#endif

// Converts an individual OpenSL-style channel identifier (SL_SPEAKER_FRONT_LEFT, etc.) to mini_al.
mal_uint8 mal_channel_id_to_mal__opensl(SLuint32 id)
{
    switch (id)
    {
        case SL_SPEAKER_FRONT_LEFT:            return MAL_CHANNEL_FRONT_LEFT;
        case SL_SPEAKER_FRONT_RIGHT:           return MAL_CHANNEL_FRONT_RIGHT;
        case SL_SPEAKER_FRONT_CENTER:          return MAL_CHANNEL_FRONT_CENTER;
        case SL_SPEAKER_LOW_FREQUENCY:         return MAL_CHANNEL_LFE;
        case SL_SPEAKER_BACK_LEFT:             return MAL_CHANNEL_BACK_LEFT;
        case SL_SPEAKER_BACK_RIGHT:            return MAL_CHANNEL_BACK_RIGHT;
        case SL_SPEAKER_FRONT_LEFT_OF_CENTER:  return MAL_CHANNEL_FRONT_LEFT_CENTER;
        case SL_SPEAKER_FRONT_RIGHT_OF_CENTER: return MAL_CHANNEL_FRONT_RIGHT_CENTER;
        case SL_SPEAKER_BACK_CENTER:           return MAL_CHANNEL_BACK_CENTER;
        case SL_SPEAKER_SIDE_LEFT:             return MAL_CHANNEL_SIDE_LEFT;
        case SL_SPEAKER_SIDE_RIGHT:            return MAL_CHANNEL_SIDE_RIGHT;
        case SL_SPEAKER_TOP_CENTER:            return MAL_CHANNEL_TOP_CENTER;
        case SL_SPEAKER_TOP_FRONT_LEFT:        return MAL_CHANNEL_TOP_FRONT_LEFT;
        case SL_SPEAKER_TOP_FRONT_CENTER:      return MAL_CHANNEL_TOP_FRONT_CENTER;
        case SL_SPEAKER_TOP_FRONT_RIGHT:       return MAL_CHANNEL_TOP_FRONT_RIGHT;
        case SL_SPEAKER_TOP_BACK_LEFT:         return MAL_CHANNEL_TOP_BACK_LEFT;
        case SL_SPEAKER_TOP_BACK_CENTER:       return MAL_CHANNEL_TOP_BACK_CENTER;
        case SL_SPEAKER_TOP_BACK_RIGHT:        return MAL_CHANNEL_TOP_BACK_RIGHT;
        default: return 0;
    }
}

// Converts an individual mini_al channel identifier (MAL_CHANNEL_FRONT_LEFT, etc.) to OpenSL-style.
SLuint32 mal_channel_id_to_opensl(mal_uint8 id)
{
    switch (id)
    {
        case MAL_CHANNEL_MONO:               return SL_SPEAKER_FRONT_CENTER;
        case MAL_CHANNEL_FRONT_LEFT:         return SL_SPEAKER_FRONT_LEFT;
        case MAL_CHANNEL_FRONT_RIGHT:        return SL_SPEAKER_FRONT_RIGHT;
        case MAL_CHANNEL_FRONT_CENTER:       return SL_SPEAKER_FRONT_CENTER;
        case MAL_CHANNEL_LFE:                return SL_SPEAKER_LOW_FREQUENCY;
        case MAL_CHANNEL_BACK_LEFT:          return SL_SPEAKER_BACK_LEFT;
        case MAL_CHANNEL_BACK_RIGHT:         return SL_SPEAKER_BACK_RIGHT;
        case MAL_CHANNEL_FRONT_LEFT_CENTER:  return SL_SPEAKER_FRONT_LEFT_OF_CENTER;
        case MAL_CHANNEL_FRONT_RIGHT_CENTER: return SL_SPEAKER_FRONT_RIGHT_OF_CENTER;
        case MAL_CHANNEL_BACK_CENTER:        return SL_SPEAKER_BACK_CENTER;
        case MAL_CHANNEL_SIDE_LEFT:          return SL_SPEAKER_SIDE_LEFT;
        case MAL_CHANNEL_SIDE_RIGHT:         return SL_SPEAKER_SIDE_RIGHT;
        case MAL_CHANNEL_TOP_CENTER:         return SL_SPEAKER_TOP_CENTER;
        case MAL_CHANNEL_TOP_FRONT_LEFT:     return SL_SPEAKER_TOP_FRONT_LEFT;
        case MAL_CHANNEL_TOP_FRONT_CENTER:   return SL_SPEAKER_TOP_FRONT_CENTER;
        case MAL_CHANNEL_TOP_FRONT_RIGHT:    return SL_SPEAKER_TOP_FRONT_RIGHT;
        case MAL_CHANNEL_TOP_BACK_LEFT:      return SL_SPEAKER_TOP_BACK_LEFT;
        case MAL_CHANNEL_TOP_BACK_CENTER:    return SL_SPEAKER_TOP_BACK_CENTER;
        case MAL_CHANNEL_TOP_BACK_RIGHT:     return SL_SPEAKER_TOP_BACK_RIGHT;
        default: return 0;
    }
}

// Converts a channel mapping to an OpenSL-style channel mask.
SLuint32 mal_channel_map_to_channel_mask__opensl(const mal_channel channelMap[MAL_MAX_CHANNELS], mal_uint32 channels)
{
    SLuint32 channelMask = 0;
    for (mal_uint32 iChannel = 0; iChannel < channels; ++iChannel) {
        channelMask |= mal_channel_id_to_opensl(channelMap[iChannel]);
    }

    return channelMask;
}

// Converts an OpenSL-style channel mask to a mini_al channel map.
void mal_channel_mask_to_channel_map__opensl(SLuint32 channelMask, mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    if (channels == 1 && channelMask == 0) {
        channelMap[0] = MAL_CHANNEL_MONO;
    } else if (channels == 2 && channelMask == 0) {
        channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
        channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
    } else {
        if (channels == 1 && (channelMask & SL_SPEAKER_FRONT_CENTER) != 0) {
            channelMap[0] = MAL_CHANNEL_MONO;
        } else {
            // Just iterate over each bit.
            mal_uint32 iChannel = 0;
            for (mal_uint32 iBit = 0; iBit < 32; ++iBit) {
                SLuint32 bitValue = (channelMask & (1UL << iBit));
                if (bitValue != 0) {
                    // The bit is set.
                    channelMap[iChannel] = mal_channel_id_to_mal__opensl(bitValue);
                    iChannel += 1;
                }
            }
        }
    }
}

SLuint32 mal_round_to_standard_sample_rate__opensl(SLuint32 samplesPerSec)
{
    if (samplesPerSec <= SL_SAMPLINGRATE_8) {
        return SL_SAMPLINGRATE_8;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_11_025) {
        return SL_SAMPLINGRATE_11_025;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_12) {
        return SL_SAMPLINGRATE_12;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_16) {
        return SL_SAMPLINGRATE_16;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_22_05) {
        return SL_SAMPLINGRATE_22_05;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_24) {
        return SL_SAMPLINGRATE_24;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_32) {
        return SL_SAMPLINGRATE_32;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_44_1) {
        return SL_SAMPLINGRATE_44_1;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_48) {
        return SL_SAMPLINGRATE_48;
    }

    // Android doesn't support more than 48000.
#ifndef MAL_ANDROID
    if (samplesPerSec <= SL_SAMPLINGRATE_64) {
        return SL_SAMPLINGRATE_64;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_88_2) {
        return SL_SAMPLINGRATE_88_2;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_96) {
        return SL_SAMPLINGRATE_96;
    }
    if (samplesPerSec <= SL_SAMPLINGRATE_192) {
        return SL_SAMPLINGRATE_192;
    }
#endif

    return SL_SAMPLINGRATE_16;
}


mal_bool32 mal_context_is_device_id_equal__opensl(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return pID0->opensl == pID1->opensl;
}

mal_result mal_context_enumerate_devices__opensl(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    // TODO: Test Me.
    //
    // This is currently untested, so for now we are just returning default devices.
#if 0 && !defined(MAL_ANDROID)
    mal_bool32 isTerminated = MAL_FALSE;

    SLuint32 pDeviceIDs[128];
    SLint32 deviceCount = sizeof(pDeviceIDs) / sizeof(pDeviceIDs[0]);

    SLAudioIODeviceCapabilitiesItf deviceCaps;
    SLresult resultSL = (*g_malEngineObjectSL)->GetInterface(g_malEngineObjectSL, SL_IID_AUDIOIODEVICECAPABILITIES, &deviceCaps);
    if (resultSL != SL_RESULT_SUCCESS) {
        // The interface may not be supported so just report a default device.
        goto return_default_device;
    }

    // Playback
    if (!isTerminated) {
        resultSL = (*deviceCaps)->GetAvailableAudioOutputs(deviceCaps, &deviceCount, pDeviceIDs);
        if (resultSL != SL_RESULT_SUCCESS) {
            return MAL_NO_DEVICE;
        }

        for (SLint32 iDevice = 0; iDevice < deviceCount; ++iDevice) {
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            deviceInfo.id.opensl = pDeviceIDs[iDevice];

            SLAudioOutputDescriptor desc;
            resultSL = (*deviceCaps)->QueryAudioOutputCapabilities(deviceCaps, deviceInfo.id.opensl, &desc);
            if (resultSL == SL_RESULT_SUCCESS) {
                mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), (const char*)desc.pDeviceName, (size_t)-1);

                mal_bool32 cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
                if (cbResult == MAL_FALSE) {
                    isTerminated = MAL_TRUE;
                    break;
                }
            }
        }
    }

    // Capture
    if (!isTerminated) {
        resultSL = (*deviceCaps)->GetAvailableAudioInputs(deviceCaps, &deviceCount, pDeviceIDs);
        if (resultSL != SL_RESULT_SUCCESS) {
            return MAL_NO_DEVICE;
        }

        for (SLint32 iDevice = 0; iDevice < deviceCount; ++iDevice) {
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            deviceInfo.id.opensl = pDeviceIDs[iDevice];

            SLAudioInputDescriptor desc;
            resultSL = (*deviceCaps)->QueryAudioInputCapabilities(deviceCaps, deviceInfo.id.opensl, &desc);
            if (resultSL == SL_RESULT_SUCCESS) {
                mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), (const char*)desc.deviceName, (size_t)-1);

                mal_bool32 cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
                if (cbResult == MAL_FALSE) {
                    isTerminated = MAL_TRUE;
                    break;
                }
            }
        }
    }

    return MAL_SUCCESS;
#else
    goto return_default_device;
#endif

return_default_device:;
    mal_bool32 cbResult = MAL_TRUE;

    // Playback.
    if (cbResult) {
        mal_device_info deviceInfo;
        mal_zero_object(&deviceInfo);
        mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
    }

    // Capture.
    if (cbResult) {
        mal_device_info deviceInfo;
        mal_zero_object(&deviceInfo);
        mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
    }

    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info__opensl(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    (void)shareMode;

    // TODO: Test Me.
    //
    // This is currently untested, so for now we are just returning default devices.
#if 0 && !defined(MAL_ANDROID)
    SLAudioIODeviceCapabilitiesItf deviceCaps;
    SLresult resultSL = (*g_malEngineObjectSL)->GetInterface(g_malEngineObjectSL, SL_IID_AUDIOIODEVICECAPABILITIES, &deviceCaps);
    if (resultSL != SL_RESULT_SUCCESS) {
        // The interface may not be supported so just report a default device.
        goto return_default_device;
    }

    if (deviceType == mal_device_type_playback) {
        SLAudioOutputDescriptor desc;
        resultSL = (*deviceCaps)->QueryAudioOutputCapabilities(deviceCaps, pDeviceID->opensl, &desc);
        if (resultSL != SL_RESULT_SUCCESS) {
            return MAL_NO_DEVICE;
        }

        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), (const char*)desc.pDeviceName, (size_t)-1);
    } else {
        SLAudioInputDescriptor desc;
        resultSL = (*deviceCaps)->QueryAudioInputCapabilities(deviceCaps, pDeviceID->opensl, &desc);
        if (resultSL != SL_RESULT_SUCCESS) {
            return MAL_NO_DEVICE;
        }

        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), (const char*)desc.deviceName, (size_t)-1);
    }

    goto return_detailed_info;
#else
    goto return_default_device;
#endif

return_default_device:
    if (pDeviceID != NULL) {
        if ((deviceType == mal_device_type_playback && pDeviceID->opensl != SL_DEFAULTDEVICEID_AUDIOOUTPUT) ||
            (deviceType == mal_device_type_capture  && pDeviceID->opensl != SL_DEFAULTDEVICEID_AUDIOINPUT)) {
            return MAL_NO_DEVICE;   // Don't know the device.
        }
    }

    // Name / Description
    if (deviceType == mal_device_type_playback) {
        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    } else {
        mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    }

    goto return_detailed_info;


return_detailed_info:

    // For now we're just outputting a set of values that are supported by the API but not necessarily supported
    // by the device natively. Later on we should work on this so that it more closely reflects the device's
    // actual native format.
    pDeviceInfo->minChannels = 1;
    pDeviceInfo->maxChannels = 2;
    pDeviceInfo->minSampleRate = 8000;
    pDeviceInfo->maxSampleRate = 48000;
    pDeviceInfo->formatCount = 2;
    pDeviceInfo->formats[0] = mal_format_u8;
    pDeviceInfo->formats[1] = mal_format_s16;
#if defined(MAL_ANDROID) && __ANDROID_API__ >= 21
    pDeviceInfo->formats[pDeviceInfo->formatCount] = mal_format_f32;
    pDeviceInfo->formatCount += 1;
#endif

    return MAL_SUCCESS;
}


#ifdef MAL_ANDROID
//void mal_buffer_queue_callback__opensl_android(SLAndroidSimpleBufferQueueItf pBufferQueue, SLuint32 eventFlags, const void* pBuffer, SLuint32 bufferSize, SLuint32 dataUsed, void* pContext)
void mal_buffer_queue_callback__opensl_android(SLAndroidSimpleBufferQueueItf pBufferQueue, void* pUserData)
{
    (void)pBufferQueue;

    // For now, only supporting Android implementations of OpenSL|ES since that's the only one I've
    // been able to test with and I currently depend on Android-specific extensions (simple buffer
    // queues).
#ifndef MAL_ANDROID
    return MAL_NO_BACKEND;
#endif

    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    // For now, don't do anything unless the buffer was fully processed. From what I can tell, it looks like
    // OpenSL|ES 1.1 improves on buffer queues to the point that we could much more intelligently handle this,
    // but unfortunately it looks like Android is only supporting OpenSL|ES 1.0.1 for now :(
    if (pDevice->state != MAL_STATE_STARTED) {
        return;
    }

    size_t periodSizeInBytes = pDevice->opensl.periodSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat);
    mal_uint8* pBuffer = pDevice->opensl.pBuffer + (pDevice->opensl.currentBufferIndex * periodSizeInBytes);

    if (pDevice->type == mal_device_type_playback) {
        if (pDevice->state != MAL_STATE_STARTED) {
            return;
        }

        mal_device__read_frames_from_client(pDevice, pDevice->opensl.periodSizeInFrames, pBuffer);

        SLresult resultSL = MAL_OPENSL_BUFFERQUEUE(pDevice->opensl.pBufferQueue)->Enqueue((SLAndroidSimpleBufferQueueItf)pDevice->opensl.pBufferQueue, pBuffer, periodSizeInBytes);
        if (resultSL != SL_RESULT_SUCCESS) {
            return;
        }
    } else {
        mal_device__send_frames_to_client(pDevice, pDevice->opensl.periodSizeInFrames, pBuffer);

        SLresult resultSL = MAL_OPENSL_BUFFERQUEUE(pDevice->opensl.pBufferQueue)->Enqueue((SLAndroidSimpleBufferQueueItf)pDevice->opensl.pBufferQueue, pBuffer, periodSizeInBytes);
        if (resultSL != SL_RESULT_SUCCESS) {
            return;
        }
    }

    pDevice->opensl.currentBufferIndex = (pDevice->opensl.currentBufferIndex + 1) % pDevice->periods;
}
#endif

void mal_device_uninit__opensl(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // Uninit device.
    if (pDevice->type == mal_device_type_playback) {
        if (pDevice->opensl.pAudioPlayerObj) MAL_OPENSL_OBJ(pDevice->opensl.pAudioPlayerObj)->Destroy((SLObjectItf)pDevice->opensl.pAudioPlayerObj);
        if (pDevice->opensl.pOutputMixObj) MAL_OPENSL_OBJ(pDevice->opensl.pOutputMixObj)->Destroy((SLObjectItf)pDevice->opensl.pOutputMixObj);
    } else {
        if (pDevice->opensl.pAudioRecorderObj) MAL_OPENSL_OBJ(pDevice->opensl.pAudioRecorderObj)->Destroy((SLObjectItf)pDevice->opensl.pAudioRecorderObj);
    }

    mal_free(pDevice->opensl.pBuffer);
}

mal_result mal_device_init__opensl(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    (void)pContext;

    // For now, only supporting Android implementations of OpenSL|ES since that's the only one I've
    // been able to test with and I currently depend on Android-specific extensions (simple buffer
    // queues).
#ifndef MAL_ANDROID
    return MAL_NO_BACKEND;
#endif

    // Use s32 as the internal format for when floating point is specified.
    if (pConfig->format == mal_format_f32) {
        pDevice->internalFormat = mal_format_s32;
    }

    // Now we can start initializing the device properly.
    mal_assert(pDevice != NULL);
    mal_zero_object(&pDevice->opensl);

    SLDataLocator_AndroidSimpleBufferQueue queue;
    queue.locatorType = SL_DATALOCATOR_ANDROIDSIMPLEBUFFERQUEUE;
    queue.numBuffers = pConfig->periods;

    SLDataFormat_PCM* pFormat = NULL;

#if defined(MAL_ANDROID) && __ANDROID_API__ >= 21
    SLAndroidDataFormat_PCM_EX pcmEx;
    if (pDevice->format == mal_format_f32 /*|| pDevice->format == mal_format_f64*/) {
        pcmEx.formatType = SL_ANDROID_DATAFORMAT_PCM_EX;
        pcmEx.representation = SL_ANDROID_PCM_REPRESENTATION_FLOAT;
    } else {
        pcmEx.formatType = SL_DATAFORMAT_PCM;
    }
    pFormat = (SLDataFormat_PCM*)&pcmEx;
#else
    SLDataFormat_PCM pcm;
    pcm.formatType = SL_DATAFORMAT_PCM;
    pFormat = &pcm;
#endif

    pFormat->numChannels   = pDevice->channels;
    pFormat->samplesPerSec = mal_round_to_standard_sample_rate__opensl(pDevice->sampleRate * 1000);  // In millihertz.
    pFormat->bitsPerSample = mal_get_bytes_per_sample(pDevice->format)*8;
    pFormat->containerSize = pFormat->bitsPerSample;  // Always tightly packed for now.
    pFormat->channelMask   = mal_channel_map_to_channel_mask__opensl(pConfig->channelMap, pFormat->numChannels);
    pFormat->endianness    = (mal_is_little_endian()) ? SL_BYTEORDER_LITTLEENDIAN : SL_BYTEORDER_BIGENDIAN;

    // Android has a few restrictions on the format as documented here: https://developer.android.com/ndk/guides/audio/opensl-for-android.html
    //  - Only mono and stereo is supported.
    //  - Only u8 and s16 formats are supported.
    //  - Maximum sample rate of 48000.
#ifdef MAL_ANDROID
    if (pFormat->numChannels > 2) {
        pFormat->numChannels = 2;
    }
#if __ANDROID_API__ >= 21
    if (pFormat->formatType == SL_ANDROID_DATAFORMAT_PCM_EX) {
        // It's floating point.
        mal_assert(pcmEx.representation == SL_ANDROID_PCM_REPRESENTATION_FLOAT);
        if (pFormat->bitsPerSample > 32) {
            pFormat->bitsPerSample = 32;
        }
    } else {
        if (pFormat->bitsPerSample > 16) {
            pFormat->bitsPerSample = 16;
        }
    }
#else
    if (pFormat->bitsPerSample > 16) {
        pFormat->bitsPerSample = 16;
    }
#endif
    pFormat->containerSize = pFormat->bitsPerSample;  // Always tightly packed for now.

    if (pFormat->samplesPerSec > SL_SAMPLINGRATE_48) {
        pFormat->samplesPerSec = SL_SAMPLINGRATE_48;
    }
#endif

    if (type == mal_device_type_playback) {
        SLresult resultSL = (*g_malEngineSL)->CreateOutputMix(g_malEngineSL, (SLObjectItf*)&pDevice->opensl.pOutputMixObj, 0, NULL, NULL);
        if (resultSL != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to create output mix.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (MAL_OPENSL_OBJ(pDevice->opensl.pOutputMixObj)->Realize((SLObjectItf)pDevice->opensl.pOutputMixObj, SL_BOOLEAN_FALSE)) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to realize output mix object.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (MAL_OPENSL_OBJ(pDevice->opensl.pOutputMixObj)->GetInterface((SLObjectItf)pDevice->opensl.pOutputMixObj, SL_IID_OUTPUTMIX, &pDevice->opensl.pOutputMix) != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to retrieve SL_IID_OUTPUTMIX interface.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        // Set the output device.
        if (pDeviceID != NULL) {
            SLuint32 deviceID_OpenSL = pDeviceID->opensl;
            MAL_OPENSL_OUTPUTMIX(pDevice->opensl.pOutputMix)->ReRoute((SLOutputMixItf)pDevice->opensl.pOutputMix, 1, &deviceID_OpenSL);
        }

        SLDataSource source;
        source.pLocator = &queue;
        source.pFormat = pFormat;

        SLDataLocator_OutputMix outmixLocator;
        outmixLocator.locatorType = SL_DATALOCATOR_OUTPUTMIX;
        outmixLocator.outputMix = (SLObjectItf)pDevice->opensl.pOutputMixObj;

        SLDataSink sink;
        sink.pLocator = &outmixLocator;
        sink.pFormat = NULL;

        const SLInterfaceID itfIDs1[] = {SL_IID_ANDROIDSIMPLEBUFFERQUEUE};
        const SLboolean itfIDsRequired1[] = {SL_BOOLEAN_TRUE};
        resultSL = (*g_malEngineSL)->CreateAudioPlayer(g_malEngineSL, (SLObjectItf*)&pDevice->opensl.pAudioPlayerObj, &source, &sink, 1, itfIDs1, itfIDsRequired1);
        if (resultSL == SL_RESULT_CONTENT_UNSUPPORTED) {
            // Unsupported format. Fall back to something safer and try again. If this fails, just abort.
            pFormat->formatType = SL_DATAFORMAT_PCM;
            pFormat->numChannels = 2;
            pFormat->samplesPerSec = SL_SAMPLINGRATE_16;
            pFormat->bitsPerSample = 16;
            pFormat->containerSize = pFormat->bitsPerSample;  // Always tightly packed for now.
            pFormat->channelMask = SL_SPEAKER_FRONT_LEFT | SL_SPEAKER_FRONT_RIGHT;
            resultSL = (*g_malEngineSL)->CreateAudioPlayer(g_malEngineSL, (SLObjectItf*)&pDevice->opensl.pAudioPlayerObj, &source, &sink, 1, itfIDs1, itfIDsRequired1);
        }

        if (resultSL != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to create audio player.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }


        if (MAL_OPENSL_OBJ(pDevice->opensl.pAudioPlayerObj)->Realize((SLObjectItf)pDevice->opensl.pAudioPlayerObj, SL_BOOLEAN_FALSE) != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to realize audio player.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (MAL_OPENSL_OBJ(pDevice->opensl.pAudioPlayerObj)->GetInterface((SLObjectItf)pDevice->opensl.pAudioPlayerObj, SL_IID_PLAY, &pDevice->opensl.pAudioPlayer) != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to retrieve SL_IID_PLAY interface.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (MAL_OPENSL_OBJ(pDevice->opensl.pAudioPlayerObj)->GetInterface((SLObjectItf)pDevice->opensl.pAudioPlayerObj, SL_IID_ANDROIDSIMPLEBUFFERQUEUE, &pDevice->opensl.pBufferQueue) != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to retrieve SL_IID_ANDROIDSIMPLEBUFFERQUEUE interface.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (MAL_OPENSL_BUFFERQUEUE(pDevice->opensl.pBufferQueue)->RegisterCallback((SLAndroidSimpleBufferQueueItf)pDevice->opensl.pBufferQueue, mal_buffer_queue_callback__opensl_android, pDevice) != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to register buffer queue callback.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    } else {
        SLDataLocator_IODevice locatorDevice;
        locatorDevice.locatorType = SL_DATALOCATOR_IODEVICE;
        locatorDevice.deviceType = SL_IODEVICE_AUDIOINPUT;
        locatorDevice.deviceID = (pDeviceID == NULL) ? SL_DEFAULTDEVICEID_AUDIOINPUT : pDeviceID->opensl;
        locatorDevice.device = NULL;

        SLDataSource source;
        source.pLocator = &locatorDevice;
        source.pFormat = NULL;

        SLDataSink sink;
        sink.pLocator = &queue;
        sink.pFormat = pFormat;

        const SLInterfaceID itfIDs1[] = {SL_IID_ANDROIDSIMPLEBUFFERQUEUE};
        const SLboolean itfIDsRequired1[] = {SL_BOOLEAN_TRUE};
        SLresult resultSL = (*g_malEngineSL)->CreateAudioRecorder(g_malEngineSL, (SLObjectItf*)&pDevice->opensl.pAudioRecorderObj, &source, &sink, 1, itfIDs1, itfIDsRequired1);
        if (resultSL == SL_RESULT_CONTENT_UNSUPPORTED) {
            // Unsupported format. Fall back to something safer and try again. If this fails, just abort.
            pFormat->formatType = SL_DATAFORMAT_PCM;
            pFormat->numChannels = 1;
            pFormat->samplesPerSec = SL_SAMPLINGRATE_16;
            pFormat->bitsPerSample = 16;
            pFormat->containerSize = pFormat->bitsPerSample;  // Always tightly packed for now.
            pFormat->channelMask = SL_SPEAKER_FRONT_LEFT | SL_SPEAKER_FRONT_RIGHT;
            resultSL = (*g_malEngineSL)->CreateAudioRecorder(g_malEngineSL, (SLObjectItf*)&pDevice->opensl.pAudioRecorderObj, &source, &sink, 1, itfIDs1, itfIDsRequired1);
        }

        if (resultSL != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to create audio recorder.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (MAL_OPENSL_OBJ(pDevice->opensl.pAudioRecorderObj)->Realize((SLObjectItf)pDevice->opensl.pAudioRecorderObj, SL_BOOLEAN_FALSE) != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to realize audio recorder.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (MAL_OPENSL_OBJ(pDevice->opensl.pAudioRecorderObj)->GetInterface((SLObjectItf)pDevice->opensl.pAudioRecorderObj, SL_IID_RECORD, &pDevice->opensl.pAudioRecorder) != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to retrieve SL_IID_RECORD interface.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (MAL_OPENSL_OBJ(pDevice->opensl.pAudioRecorderObj)->GetInterface((SLObjectItf)pDevice->opensl.pAudioRecorderObj, SL_IID_ANDROIDSIMPLEBUFFERQUEUE, &pDevice->opensl.pBufferQueue) != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to retrieve SL_IID_ANDROIDSIMPLEBUFFERQUEUE interface.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (MAL_OPENSL_BUFFERQUEUE(pDevice->opensl.pBufferQueue)->RegisterCallback((SLAndroidSimpleBufferQueueItf)pDevice->opensl.pBufferQueue, mal_buffer_queue_callback__opensl_android, pDevice) != SL_RESULT_SUCCESS) {
            mal_device_uninit__opensl(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to register buffer queue callback.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    }


    // The internal format is determined by the pFormat object.
    mal_bool32 isFloatingPoint = MAL_FALSE;
#if defined(MAL_ANDROID) && __ANDROID_API__ >= 21
    if (pFormat->formatType == SL_ANDROID_DATAFORMAT_PCM_EX) {
        mal_assert(pcmEx.representation == SL_ANDROID_PCM_REPRESENTATION_FLOAT);
        isFloatingPoint = MAL_TRUE;
    }
#endif
    if (isFloatingPoint) {
        if (pFormat->bitsPerSample == 32) {
            pDevice->internalFormat = mal_format_f32;
        }
#if 0
        if (pFormat->bitsPerSample == 64) {
            pDevice->internalFormat = mal_format_f64;
        }
#endif
    } else {
        if (pFormat->bitsPerSample == 8) {
            pDevice->internalFormat = mal_format_u8;
        } else if (pFormat->bitsPerSample == 16) {
            pDevice->internalFormat = mal_format_s16;
        } else if (pFormat->bitsPerSample == 24) {
            pDevice->internalFormat = mal_format_s24;
        } else if (pFormat->bitsPerSample == 32) {
            pDevice->internalFormat = mal_format_s32;
        }
    }

    pDevice->internalChannels = pFormat->numChannels;
    pDevice->internalSampleRate = pFormat->samplesPerSec / 1000;
    mal_channel_mask_to_channel_map__opensl(pFormat->channelMask, pDevice->internalChannels, pDevice->internalChannelMap);

    // Try calculating an appropriate default buffer size.
    if (pDevice->bufferSizeInFrames == 0) {
        pDevice->bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, pDevice->internalSampleRate);
    }

    pDevice->opensl.currentBufferIndex = 0;
    pDevice->opensl.periodSizeInFrames = pDevice->bufferSizeInFrames / pConfig->periods;
    pDevice->bufferSizeInFrames = pDevice->opensl.periodSizeInFrames * pConfig->periods;

    size_t bufferSizeInBytes = pDevice->bufferSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat);
    pDevice->opensl.pBuffer = (mal_uint8*)mal_malloc(bufferSizeInBytes);
    if (pDevice->opensl.pBuffer == NULL) {
        mal_device_uninit__opensl(pDevice);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to allocate memory for data buffer.", MAL_OUT_OF_MEMORY);
    }

    mal_zero_memory(pDevice->opensl.pBuffer, bufferSizeInBytes);

    return MAL_SUCCESS;
}

mal_result mal_device__start_backend__opensl(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->type == mal_device_type_playback) {
        SLresult resultSL = MAL_OPENSL_PLAY(pDevice->opensl.pAudioPlayer)->SetPlayState((SLPlayItf)pDevice->opensl.pAudioPlayer, SL_PLAYSTATE_PLAYING);
        if (resultSL != SL_RESULT_SUCCESS) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to start internal playback device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
        }

        // We need to enqueue a buffer for each period.
        mal_device__read_frames_from_client(pDevice, pDevice->bufferSizeInFrames, pDevice->opensl.pBuffer);

        size_t periodSizeInBytes = pDevice->opensl.periodSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat);
        for (mal_uint32 iPeriod = 0; iPeriod < pDevice->periods; ++iPeriod) {
            resultSL = MAL_OPENSL_BUFFERQUEUE(pDevice->opensl.pBufferQueue)->Enqueue((SLAndroidSimpleBufferQueueItf)pDevice->opensl.pBufferQueue, pDevice->opensl.pBuffer + (periodSizeInBytes * iPeriod), periodSizeInBytes);
            if (resultSL != SL_RESULT_SUCCESS) {
                MAL_OPENSL_PLAY(pDevice->opensl.pAudioPlayer)->SetPlayState((SLPlayItf)pDevice->opensl.pAudioPlayer, SL_PLAYSTATE_STOPPED);
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to enqueue buffer for playback device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
            }
        }
    } else {
        SLresult resultSL = MAL_OPENSL_RECORD(pDevice->opensl.pAudioRecorder)->SetRecordState((SLRecordItf)pDevice->opensl.pAudioRecorder, SL_RECORDSTATE_RECORDING);
        if (resultSL != SL_RESULT_SUCCESS) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to start internal capture device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
        }

        size_t periodSizeInBytes = pDevice->opensl.periodSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat);
        for (mal_uint32 iPeriod = 0; iPeriod < pDevice->periods; ++iPeriod) {
            resultSL = MAL_OPENSL_BUFFERQUEUE(pDevice->opensl.pBufferQueue)->Enqueue((SLAndroidSimpleBufferQueueItf)pDevice->opensl.pBufferQueue, pDevice->opensl.pBuffer + (periodSizeInBytes * iPeriod), periodSizeInBytes);
            if (resultSL != SL_RESULT_SUCCESS) {
                MAL_OPENSL_RECORD(pDevice->opensl.pAudioRecorder)->SetRecordState((SLRecordItf)pDevice->opensl.pAudioRecorder, SL_RECORDSTATE_STOPPED);
                return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to enqueue buffer for capture device.", MAL_FAILED_TO_START_BACKEND_DEVICE);
            }
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__opensl(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->type == mal_device_type_playback) {
        SLresult resultSL = MAL_OPENSL_PLAY(pDevice->opensl.pAudioPlayer)->SetPlayState((SLPlayItf)pDevice->opensl.pAudioPlayer, SL_PLAYSTATE_STOPPED);
        if (resultSL != SL_RESULT_SUCCESS) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to stop internal playback device.", MAL_FAILED_TO_STOP_BACKEND_DEVICE);
        }
    } else {
        SLresult resultSL = MAL_OPENSL_RECORD(pDevice->opensl.pAudioRecorder)->SetRecordState((SLRecordItf)pDevice->opensl.pAudioRecorder, SL_RECORDSTATE_STOPPED);
        if (resultSL != SL_RESULT_SUCCESS) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "[OpenSL] Failed to stop internal capture device.", MAL_FAILED_TO_STOP_BACKEND_DEVICE);
        }
    }

    // Make sure any queued buffers are cleared.
    MAL_OPENSL_BUFFERQUEUE(pDevice->opensl.pBufferQueue)->Clear((SLAndroidSimpleBufferQueueItf)pDevice->opensl.pBufferQueue);

    // Make sure the client is aware that the device has stopped. There may be an OpenSL|ES callback for this, but I haven't found it.
    mal_device__set_state(pDevice, MAL_STATE_STOPPED);
    mal_stop_proc onStop = pDevice->onStop;
    if (onStop) {
        onStop(pDevice);
    }

    return MAL_SUCCESS;
}


mal_result mal_context_uninit__opensl(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_opensl);
    (void)pContext;

    // Uninit global data.
    if (g_malOpenSLInitCounter > 0) {
        if (mal_atomic_decrement_32(&g_malOpenSLInitCounter) == 0) {
            (*g_malEngineObjectSL)->Destroy(g_malEngineObjectSL);
        }
    }

    return MAL_SUCCESS;
}

mal_result mal_context_init__opensl(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    (void)pContext;

    // Initialize global data first if applicable.
    if (mal_atomic_increment_32(&g_malOpenSLInitCounter) == 1) {
        SLresult resultSL = slCreateEngine(&g_malEngineObjectSL, 0, NULL, 0, NULL, NULL);
        if (resultSL != SL_RESULT_SUCCESS) {
            mal_atomic_decrement_32(&g_malOpenSLInitCounter);
            return MAL_NO_BACKEND;
        }

        (*g_malEngineObjectSL)->Realize(g_malEngineObjectSL, SL_BOOLEAN_FALSE);

        resultSL = (*g_malEngineObjectSL)->GetInterface(g_malEngineObjectSL, SL_IID_ENGINE, &g_malEngineSL);
        if (resultSL != SL_RESULT_SUCCESS) {
            (*g_malEngineObjectSL)->Destroy(g_malEngineObjectSL);
            mal_atomic_decrement_32(&g_malOpenSLInitCounter);
            return MAL_NO_BACKEND;
        }
    }

    pContext->isBackendAsynchronous = MAL_TRUE;

    pContext->onUninit        = mal_context_uninit__opensl;
    pContext->onDeviceIDEqual = mal_context_is_device_id_equal__opensl;
    pContext->onEnumDevices   = mal_context_enumerate_devices__opensl;
    pContext->onGetDeviceInfo = mal_context_get_device_info__opensl;
    pContext->onDeviceInit    = mal_device_init__opensl;
    pContext->onDeviceUninit  = mal_device_uninit__opensl;
    pContext->onDeviceStart   = mal_device__start_backend__opensl;
    pContext->onDeviceStop    = mal_device__stop_backend__opensl;

    return MAL_SUCCESS;
}
#endif  // OpenSL|ES

///////////////////////////////////////////////////////////////////////////////
//
// OpenAL Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_OPENAL
#ifdef MAL_WIN32
#define MAL_AL_APIENTRY __cdecl
#else
#define MAL_AL_APIENTRY
#endif

#ifdef MAL_NO_RUNTIME_LINKING
    #if defined(MAL_APPLE)
        #include <OpenAL/al.h>
        #include <OpenAL/alc.h>
    #else
        #include <AL/al.h>
        #include <AL/alc.h>
    #endif
#endif

typedef struct mal_ALCdevice_struct  mal_ALCdevice;
typedef struct mal_ALCcontext_struct mal_ALCcontext;
typedef char                         mal_ALCboolean;
typedef char                         mal_ALCchar;
typedef signed char                  mal_ALCbyte;
typedef unsigned char                mal_ALCubyte;
typedef short                        mal_ALCshort;
typedef unsigned short               mal_ALCushort;
typedef int                          mal_ALCint;
typedef unsigned int                 mal_ALCuint;
typedef int                          mal_ALCsizei;
typedef int                          mal_ALCenum;
typedef float                        mal_ALCfloat;
typedef double                       mal_ALCdouble;
typedef void                         mal_ALCvoid;

typedef mal_ALCboolean               mal_ALboolean;
typedef mal_ALCchar                  mal_ALchar;
typedef mal_ALCbyte                  mal_ALbyte;
typedef mal_ALCubyte                 mal_ALubyte;
typedef mal_ALCshort                 mal_ALshort;
typedef mal_ALCushort                mal_ALushort;
typedef mal_ALCint                   mal_ALint;
typedef mal_ALCuint                  mal_ALuint;
typedef mal_ALCsizei                 mal_ALsizei;
typedef mal_ALCenum                  mal_ALenum;
typedef mal_ALCfloat                 mal_ALfloat;
typedef mal_ALCdouble                mal_ALdouble;
typedef mal_ALCvoid                  mal_ALvoid;

#define MAL_ALC_DEVICE_SPECIFIER            0x1005
#define MAL_ALC_CAPTURE_DEVICE_SPECIFIER    0x310
#define MAL_ALC_CAPTURE_SAMPLES             0x312

#define MAL_AL_SOURCE_STATE                 0x1010
#define MAL_AL_INITIAL                      0x1011
#define MAL_AL_PLAYING                      0x1012
#define MAL_AL_PAUSED                       0x1013
#define MAL_AL_STOPPED                      0x1014
#define MAL_AL_BUFFERS_PROCESSED            0x1016

#define MAL_AL_FORMAT_MONO8                 0x1100
#define MAL_AL_FORMAT_MONO16                0x1101
#define MAL_AL_FORMAT_STEREO8               0x1102
#define MAL_AL_FORMAT_STEREO16              0x1103
#define MAL_AL_FORMAT_MONO_FLOAT32          0x10010
#define MAL_AL_FORMAT_STEREO_FLOAT32        0x10011
#define MAL_AL_FORMAT_51CHN16               0x120B
#define MAL_AL_FORMAT_51CHN32               0x120C
#define MAL_AL_FORMAT_51CHN8                0x120A
#define MAL_AL_FORMAT_61CHN16               0x120E
#define MAL_AL_FORMAT_61CHN32               0x120F
#define MAL_AL_FORMAT_61CHN8                0x120D
#define MAL_AL_FORMAT_71CHN16               0x1211
#define MAL_AL_FORMAT_71CHN32               0x1212
#define MAL_AL_FORMAT_71CHN8                0x1210
#define MAL_AL_FORMAT_QUAD16                0x1205
#define MAL_AL_FORMAT_QUAD32                0x1206
#define MAL_AL_FORMAT_QUAD8                 0x1204
#define MAL_AL_FORMAT_REAR16                0x1208
#define MAL_AL_FORMAT_REAR32                0x1209
#define MAL_AL_FORMAT_REAR8                 0x1207

typedef mal_ALCcontext*    (MAL_AL_APIENTRY * MAL_LPALCCREATECONTEXT)      (mal_ALCdevice *device, const mal_ALCint *attrlist);
typedef mal_ALCboolean     (MAL_AL_APIENTRY * MAL_LPALCMAKECONTEXTCURRENT) (mal_ALCcontext *context);
typedef void               (MAL_AL_APIENTRY * MAL_LPALCPROCESSCONTEXT)     (mal_ALCcontext *context);
typedef void               (MAL_AL_APIENTRY * MAL_LPALCSUSPENDCONTEXT)     (mal_ALCcontext *context);
typedef void               (MAL_AL_APIENTRY * MAL_LPALCDESTROYCONTEXT)     (mal_ALCcontext *context);
typedef mal_ALCcontext*    (MAL_AL_APIENTRY * MAL_LPALCGETCURRENTCONTEXT)  (void);
typedef mal_ALCdevice*     (MAL_AL_APIENTRY * MAL_LPALCGETCONTEXTSDEVICE)  (mal_ALCcontext *context);
typedef mal_ALCdevice*     (MAL_AL_APIENTRY * MAL_LPALCOPENDEVICE)         (const mal_ALCchar *devicename);
typedef mal_ALCboolean     (MAL_AL_APIENTRY * MAL_LPALCCLOSEDEVICE)        (mal_ALCdevice *device);
typedef mal_ALCenum        (MAL_AL_APIENTRY * MAL_LPALCGETERROR)           (mal_ALCdevice *device);
typedef mal_ALCboolean     (MAL_AL_APIENTRY * MAL_LPALCISEXTENSIONPRESENT) (mal_ALCdevice *device, const mal_ALCchar *extname);
typedef void*              (MAL_AL_APIENTRY * MAL_LPALCGETPROCADDRESS)     (mal_ALCdevice *device, const mal_ALCchar *funcname);
typedef mal_ALCenum        (MAL_AL_APIENTRY * MAL_LPALCGETENUMVALUE)       (mal_ALCdevice *device, const mal_ALCchar *enumname);
typedef const mal_ALCchar* (MAL_AL_APIENTRY * MAL_LPALCGETSTRING)          (mal_ALCdevice *device, mal_ALCenum param);
typedef void               (MAL_AL_APIENTRY * MAL_LPALCGETINTEGERV)        (mal_ALCdevice *device, mal_ALCenum param, mal_ALCsizei size, mal_ALCint *values);
typedef mal_ALCdevice*     (MAL_AL_APIENTRY * MAL_LPALCCAPTUREOPENDEVICE)  (const mal_ALCchar *devicename, mal_ALCuint frequency, mal_ALCenum format, mal_ALCsizei buffersize);
typedef mal_ALCboolean     (MAL_AL_APIENTRY * MAL_LPALCCAPTURECLOSEDEVICE) (mal_ALCdevice *device);
typedef void               (MAL_AL_APIENTRY * MAL_LPALCCAPTURESTART)       (mal_ALCdevice *device);
typedef void               (MAL_AL_APIENTRY * MAL_LPALCCAPTURESTOP)        (mal_ALCdevice *device);
typedef void               (MAL_AL_APIENTRY * MAL_LPALCCAPTURESAMPLES)     (mal_ALCdevice *device, mal_ALCvoid *buffer, mal_ALCsizei samples);

typedef void               (MAL_AL_APIENTRY * MAL_LPALENABLE)              (mal_ALenum capability);
typedef void               (MAL_AL_APIENTRY * MAL_LPALDISABLE)             (mal_ALenum capability);
typedef mal_ALboolean      (MAL_AL_APIENTRY * MAL_LPALISENABLED)           (mal_ALenum capability);
typedef const mal_ALchar*  (MAL_AL_APIENTRY * MAL_LPALGETSTRING)           (mal_ALenum param);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETBOOLEANV)         (mal_ALenum param, mal_ALboolean *values);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETINTEGERV)         (mal_ALenum param, mal_ALint *values);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETFLOATV)           (mal_ALenum param, mal_ALfloat *values);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETDOUBLEV)          (mal_ALenum param, mal_ALdouble *values);
typedef mal_ALboolean      (MAL_AL_APIENTRY * MAL_LPALGETBOOLEAN)          (mal_ALenum param);
typedef mal_ALint          (MAL_AL_APIENTRY * MAL_LPALGETINTEGER)          (mal_ALenum param);
typedef mal_ALfloat        (MAL_AL_APIENTRY * MAL_LPALGETFLOAT)            (mal_ALenum param);
typedef mal_ALdouble       (MAL_AL_APIENTRY * MAL_LPALGETDOUBLE)           (mal_ALenum param);
typedef mal_ALenum         (MAL_AL_APIENTRY * MAL_LPALGETERROR)            (void);
typedef mal_ALboolean      (MAL_AL_APIENTRY * MAL_LPALISEXTENSIONPRESENT)  (const mal_ALchar *extname);
typedef void*              (MAL_AL_APIENTRY * MAL_LPALGETPROCADDRESS)      (const mal_ALchar *fname);
typedef mal_ALenum         (MAL_AL_APIENTRY * MAL_LPALGETENUMVALUE)        (const mal_ALchar *ename);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGENSOURCES)          (mal_ALsizei n, mal_ALuint *sources);
typedef void               (MAL_AL_APIENTRY * MAL_LPALDELETESOURCES)       (mal_ALsizei n, const mal_ALuint *sources);
typedef mal_ALboolean      (MAL_AL_APIENTRY * MAL_LPALISSOURCE)            (mal_ALuint source);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEF)             (mal_ALuint source, mal_ALenum param, mal_ALfloat value);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCE3F)            (mal_ALuint source, mal_ALenum param, mal_ALfloat value1, mal_ALfloat value2, mal_ALfloat value3);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEFV)            (mal_ALuint source, mal_ALenum param, const mal_ALfloat *values);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEI)             (mal_ALuint source, mal_ALenum param, mal_ALint value);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCE3I)            (mal_ALuint source, mal_ALenum param, mal_ALint value1, mal_ALint value2, mal_ALint value3);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEIV)            (mal_ALuint source, mal_ALenum param, const mal_ALint *values);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETSOURCEF)          (mal_ALuint source, mal_ALenum param, mal_ALfloat *value);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETSOURCE3F)         (mal_ALuint source, mal_ALenum param, mal_ALfloat *value1, mal_ALfloat *value2, mal_ALfloat *value3);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETSOURCEFV)         (mal_ALuint source, mal_ALenum param, mal_ALfloat *values);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETSOURCEI)          (mal_ALuint source, mal_ALenum param, mal_ALint *value);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETSOURCE3I)         (mal_ALuint source, mal_ALenum param, mal_ALint *value1, mal_ALint *value2, mal_ALint *value3);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETSOURCEIV)         (mal_ALuint source, mal_ALenum param, mal_ALint *values);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEPLAYV)         (mal_ALsizei n, const mal_ALuint *sources);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCESTOPV)         (mal_ALsizei n, const mal_ALuint *sources);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEREWINDV)       (mal_ALsizei n, const mal_ALuint *sources);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEPAUSEV)        (mal_ALsizei n, const mal_ALuint *sources);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEPLAY)          (mal_ALuint source);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCESTOP)          (mal_ALuint source);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEREWIND)        (mal_ALuint source);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEPAUSE)         (mal_ALuint source);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEQUEUEBUFFERS)  (mal_ALuint source, mal_ALsizei nb, const mal_ALuint *buffers);
typedef void               (MAL_AL_APIENTRY * MAL_LPALSOURCEUNQUEUEBUFFERS)(mal_ALuint source, mal_ALsizei nb, mal_ALuint *buffers);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGENBUFFERS)          (mal_ALsizei n, mal_ALuint *buffers);
typedef void               (MAL_AL_APIENTRY * MAL_LPALDELETEBUFFERS)       (mal_ALsizei n, const mal_ALuint *buffers);
typedef mal_ALboolean      (MAL_AL_APIENTRY * MAL_LPALISBUFFER)            (mal_ALuint buffer);
typedef void               (MAL_AL_APIENTRY * MAL_LPALBUFFERDATA)          (mal_ALuint buffer, mal_ALenum format, const mal_ALvoid *data, mal_ALsizei size, mal_ALsizei freq);
typedef void               (MAL_AL_APIENTRY * MAL_LPALBUFFERF)             (mal_ALuint buffer, mal_ALenum param, mal_ALfloat value);
typedef void               (MAL_AL_APIENTRY * MAL_LPALBUFFER3F)            (mal_ALuint buffer, mal_ALenum param, mal_ALfloat value1, mal_ALfloat value2, mal_ALfloat value3);
typedef void               (MAL_AL_APIENTRY * MAL_LPALBUFFERFV)            (mal_ALuint buffer, mal_ALenum param, const mal_ALfloat *values);
typedef void               (MAL_AL_APIENTRY * MAL_LPALBUFFERI)             (mal_ALuint buffer, mal_ALenum param, mal_ALint value);
typedef void               (MAL_AL_APIENTRY * MAL_LPALBUFFER3I)            (mal_ALuint buffer, mal_ALenum param, mal_ALint value1, mal_ALint value2, mal_ALint value3);
typedef void               (MAL_AL_APIENTRY * MAL_LPALBUFFERIV)            (mal_ALuint buffer, mal_ALenum param, const mal_ALint *values);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETBUFFERF)          (mal_ALuint buffer, mal_ALenum param, mal_ALfloat *value);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETBUFFER3F)         (mal_ALuint buffer, mal_ALenum param, mal_ALfloat *value1, mal_ALfloat *value2, mal_ALfloat *value3);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETBUFFERFV)         (mal_ALuint buffer, mal_ALenum param, mal_ALfloat *values);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETBUFFERI)          (mal_ALuint buffer, mal_ALenum param, mal_ALint *value);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETBUFFER3I)         (mal_ALuint buffer, mal_ALenum param, mal_ALint *value1, mal_ALint *value2, mal_ALint *value3);
typedef void               (MAL_AL_APIENTRY * MAL_LPALGETBUFFERIV)         (mal_ALuint buffer, mal_ALenum param, mal_ALint *values);

mal_bool32 mal_context_is_device_id_equal__openal(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return mal_strcmp(pID0->openal, pID1->openal) == 0;
}

mal_result mal_context_enumerate_devices__openal(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

    if (pContext->openal.isEnumerationSupported) {
        mal_bool32 isTerminated = MAL_FALSE;

        // Playback
        if (!isTerminated) {
            const mal_ALCchar* pPlaybackDeviceNames = ((MAL_LPALCGETSTRING)pContext->openal.alcGetString)(NULL, MAL_ALC_DEVICE_SPECIFIER);
            if (pPlaybackDeviceNames == NULL) {
                return MAL_NO_DEVICE;
            }

            // Each device is stored in pDeviceNames, separated by a null-terminator. The string itself is double-null-terminated.
            const mal_ALCchar* pNextPlaybackDeviceName = pPlaybackDeviceNames;
            while (pNextPlaybackDeviceName[0] != '\0') {
                mal_device_info deviceInfo;
                mal_zero_object(&deviceInfo);
                mal_strncpy_s(deviceInfo.id.openal, sizeof(deviceInfo.id.openal), (const char*)pNextPlaybackDeviceName, (size_t)-1);
                mal_strncpy_s(deviceInfo.name,      sizeof(deviceInfo.name),      (const char*)pNextPlaybackDeviceName, (size_t)-1);

                mal_bool32 cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
                if (cbResult == MAL_FALSE) {
                    isTerminated = MAL_TRUE;
                    break;
                }

                // Move to the next device name.
                while (*pNextPlaybackDeviceName != '\0') {
                    pNextPlaybackDeviceName += 1;
                }

                // Skip past the null terminator.
                pNextPlaybackDeviceName += 1;
            };
        }

        // Capture
        if (!isTerminated) {
            const mal_ALCchar* pCaptureDeviceNames = ((MAL_LPALCGETSTRING)pContext->openal.alcGetString)(NULL, MAL_ALC_CAPTURE_DEVICE_SPECIFIER);
            if (pCaptureDeviceNames == NULL) {
                return MAL_NO_DEVICE;
            }

            const mal_ALCchar* pNextCaptureDeviceName = pCaptureDeviceNames;
            while (pNextCaptureDeviceName[0] != '\0') {
                mal_device_info deviceInfo;
                mal_zero_object(&deviceInfo);
                mal_strncpy_s(deviceInfo.id.openal, sizeof(deviceInfo.id.openal), (const char*)pNextCaptureDeviceName, (size_t)-1);
                mal_strncpy_s(deviceInfo.name,      sizeof(deviceInfo.name),      (const char*)pNextCaptureDeviceName, (size_t)-1);

                mal_bool32 cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
                if (cbResult == MAL_FALSE) {
                    isTerminated = MAL_TRUE;
                    break;
                }

                // Move to the next device name.
                while (*pNextCaptureDeviceName != '\0') {
                    pNextCaptureDeviceName += 1;
                }

                // Skip past the null terminator.
                pNextCaptureDeviceName += 1;
            };
        }
    } else {
        // Enumeration is not supported. Use default devices.
        mal_bool32 cbResult = MAL_TRUE;

        // Playback.
        if (cbResult) {
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
            cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
        }

        // Capture.
        if (cbResult) {
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
            cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
        }
    }

    return MAL_SUCCESS;
}


typedef struct
{
    mal_device_type deviceType;
    const mal_device_id* pDeviceID;
    mal_share_mode shareMode;
    mal_device_info* pDeviceInfo;
    mal_bool32 foundDevice;
} mal_context_get_device_info_enum_callback_data__openal;

mal_bool32 mal_context_get_device_info_enum_callback__openal(mal_context* pContext, mal_device_type deviceType, const mal_device_info* pDeviceInfo, void* pUserData)
{
    mal_context_get_device_info_enum_callback_data__openal* pData = (mal_context_get_device_info_enum_callback_data__openal*)pUserData;
    mal_assert(pData != NULL);

    if (pData->deviceType == deviceType && mal_context_is_device_id_equal__openal(pContext, pData->pDeviceID, &pDeviceInfo->id)) {
        mal_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pDeviceInfo->name, (size_t)-1);
        pData->foundDevice = MAL_TRUE;
    }

    // Keep enumerating until we have found the device.
    return !pData->foundDevice;
}

mal_result mal_context_get_device_info__openal(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    (void)shareMode;

    // Name / Description
    if (pDeviceID == NULL) {
        if (deviceType == mal_device_type_playback) {
            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        } else {
            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        }

        return MAL_SUCCESS;
    } else {
        mal_context_get_device_info_enum_callback_data__openal data;
        data.deviceType = deviceType;
        data.pDeviceID = pDeviceID;
        data.shareMode = shareMode;
        data.pDeviceInfo = pDeviceInfo;
        data.foundDevice = MAL_FALSE;
        mal_result result = mal_context_enumerate_devices__openal(pContext, mal_context_get_device_info_enum_callback__openal, &data);
        if (result != MAL_SUCCESS) {
            return result;
        }

        if (!data.foundDevice) {
            return MAL_NO_DEVICE;
        }
    }

    // mini_al's OpenAL backend only supports:
    //   - mono and stereo
    //   - u8, s16 and f32
    //   - All standard sample rates
    pDeviceInfo->minChannels = 1;
    pDeviceInfo->maxChannels = 2;
    pDeviceInfo->minSampleRate = MAL_MIN_SAMPLE_RATE;
    pDeviceInfo->maxSampleRate = MAL_MAX_SAMPLE_RATE;
    pDeviceInfo->formatCount = 2;
    pDeviceInfo->formats[0] = mal_format_u8;
    pDeviceInfo->formats[1] = mal_format_s16;
    if (pContext->openal.isFloat32Supported) {
        pDeviceInfo->formats[pDeviceInfo->formatCount] = mal_format_f32;
        pDeviceInfo->formatCount += 1;
    }

    return MAL_SUCCESS;
}


void mal_device_uninit__openal(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // Delete buffers and source first.
    ((MAL_LPALCMAKECONTEXTCURRENT)pDevice->pContext->openal.alcMakeContextCurrent)((mal_ALCcontext*)pDevice->openal.pContextALC);
    if (pDevice->openal.sourceAL != 0) {
        ((MAL_LPALDELETESOURCES)pDevice->pContext->openal.alDeleteSources)(1, (const mal_ALuint*)&pDevice->openal.sourceAL);
    }
    if (pDevice->periods > 0 && pDevice->openal.buffersAL[0] != 0) {
        ((MAL_LPALDELETEBUFFERS)pDevice->pContext->openal.alDeleteBuffers)(pDevice->periods, (const mal_ALuint*)pDevice->openal.buffersAL);
    }


    // Now that resources have been deleted we can destroy the OpenAL context and close the device.
    ((MAL_LPALCMAKECONTEXTCURRENT)pDevice->pContext->openal.alcMakeContextCurrent)(NULL);
    ((MAL_LPALCDESTROYCONTEXT)pDevice->pContext->openal.alcDestroyContext)((mal_ALCcontext*)pDevice->openal.pContextALC);

    if (pDevice->type == mal_device_type_playback) {
        ((MAL_LPALCCLOSEDEVICE)pDevice->pContext->openal.alcCloseDevice)((mal_ALCdevice*)pDevice->openal.pDeviceALC);
    } else {
        ((MAL_LPALCCAPTURECLOSEDEVICE)pDevice->pContext->openal.alcCaptureCloseDevice)((mal_ALCdevice*)pDevice->openal.pDeviceALC);
    }

    mal_free(pDevice->openal.pIntermediaryBuffer);
}

mal_result mal_device_init__openal(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    if (pDevice->periods > MAL_MAX_PERIODS_OPENAL) {
        pDevice->periods = MAL_MAX_PERIODS_OPENAL;
    }

    // Try calculating an appropriate default buffer size.
    if (pDevice->bufferSizeInFrames == 0) {
        pDevice->bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, pDevice->sampleRate);
        if (pDevice->usingDefaultBufferSize) {
            float bufferSizeScaleFactor = 3;
            pDevice->bufferSizeInFrames = mal_scale_buffer_size(pDevice->bufferSizeInFrames, bufferSizeScaleFactor);
        }
    }

    mal_ALCsizei bufferSizeInSamplesAL = pDevice->bufferSizeInFrames;
    mal_ALCuint frequencyAL = pConfig->sampleRate;

    mal_uint32 channelsAL = 0;

    // OpenAL currently only supports only mono and stereo. TODO: Check for the AL_EXT_MCFORMATS extension and use one of those formats for quad, 5.1, etc.
    mal_ALCenum formatAL = 0;
    if (pConfig->channels == 1) {
        // Mono.
        channelsAL = 1;
        if (pConfig->format == mal_format_f32) {
            if (pContext->openal.isFloat32Supported) {
                formatAL = MAL_AL_FORMAT_MONO_FLOAT32;
            } else {
                formatAL = MAL_AL_FORMAT_MONO16;
            }
        } else if (pConfig->format == mal_format_s32) {
            formatAL = MAL_AL_FORMAT_MONO16;
        } else if (pConfig->format == mal_format_s24) {
            formatAL = MAL_AL_FORMAT_MONO16;
        } else if (pConfig->format == mal_format_s16) {
            formatAL = MAL_AL_FORMAT_MONO16;
        } else if (pConfig->format == mal_format_u8) {
            formatAL = MAL_AL_FORMAT_MONO8;
        }
    } else {
        // Stereo.
        channelsAL = 2;
        if (pConfig->format == mal_format_f32) {
            if (pContext->openal.isFloat32Supported) {
                formatAL = MAL_AL_FORMAT_STEREO_FLOAT32;
            } else {
                formatAL = MAL_AL_FORMAT_STEREO16;
            }
        } else if (pConfig->format == mal_format_s32) {
            formatAL = MAL_AL_FORMAT_STEREO16;
        } else if (pConfig->format == mal_format_s24) {
            formatAL = MAL_AL_FORMAT_STEREO16;
        } else if (pConfig->format == mal_format_s16) {
            formatAL = MAL_AL_FORMAT_STEREO16;
        } else if (pConfig->format == mal_format_u8) {
            formatAL = MAL_AL_FORMAT_STEREO8;
        }
    }

    if (formatAL == 0) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[OpenAL] Format not supported.", MAL_FORMAT_NOT_SUPPORTED);
    }

    bufferSizeInSamplesAL *= channelsAL;


    // OpenAL feels a bit unintuitive to me... The global object is a device, and it would appear that each device can have
    // many context's...
    mal_ALCdevice* pDeviceALC = NULL;
    if (type == mal_device_type_playback) {
        pDeviceALC = ((MAL_LPALCOPENDEVICE)pContext->openal.alcOpenDevice)((pDeviceID == NULL) ? NULL : pDeviceID->openal);
    } else {
        pDeviceALC = ((MAL_LPALCCAPTUREOPENDEVICE)pContext->openal.alcCaptureOpenDevice)((pDeviceID == NULL) ? NULL : pDeviceID->openal, frequencyAL, formatAL, bufferSizeInSamplesAL);
    }

    if (pDeviceALC == NULL) {
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[OpenAL] Failed to open device.", MAL_FAILED_TO_INIT_BACKEND);
    }

    // A context is only required for playback.
    mal_ALCcontext* pContextALC = NULL;
    if (pDevice->type == mal_device_type_playback) {
        pContextALC = ((MAL_LPALCCREATECONTEXT)pContext->openal.alcCreateContext)(pDeviceALC, NULL);
        if (pContextALC == NULL) {
            ((MAL_LPALCCLOSEDEVICE)pDevice->pContext->openal.alcCloseDevice)(pDeviceALC);
            return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[OpenAL] Failed to open OpenAL context.", MAL_FAILED_TO_INIT_BACKEND);
        }

        ((MAL_LPALCMAKECONTEXTCURRENT)pDevice->pContext->openal.alcMakeContextCurrent)(pContextALC);

        mal_ALuint sourceAL;
        ((MAL_LPALGENSOURCES)pDevice->pContext->openal.alGenSources)(1, &sourceAL);
        pDevice->openal.sourceAL = sourceAL;

        // We create the buffers, but only fill and queue them when the device is started.
        mal_ALuint buffersAL[MAL_MAX_PERIODS_OPENAL];
        ((MAL_LPALGENBUFFERS)pDevice->pContext->openal.alGenBuffers)(pDevice->periods, buffersAL);
        for (mal_uint32 i = 0; i < pDevice->periods; ++i) {
            pDevice->openal.buffersAL[i] = buffersAL[i];
        }
    }

    pDevice->internalChannels = channelsAL;
    pDevice->internalSampleRate = frequencyAL;

    switch (formatAL)
    {
        case MAL_AL_FORMAT_MONO8:
        case MAL_AL_FORMAT_STEREO8:
        case MAL_AL_FORMAT_REAR8:
        case MAL_AL_FORMAT_QUAD8:
        case MAL_AL_FORMAT_51CHN8:
        case MAL_AL_FORMAT_61CHN8:
        case MAL_AL_FORMAT_71CHN8:
        {
            pDevice->internalFormat = mal_format_u8;
        } break;

        case MAL_AL_FORMAT_MONO16:
        case MAL_AL_FORMAT_STEREO16:
        case MAL_AL_FORMAT_REAR16:
        case MAL_AL_FORMAT_QUAD16:
        case MAL_AL_FORMAT_51CHN16:
        case MAL_AL_FORMAT_61CHN16:
        case MAL_AL_FORMAT_71CHN16:
        {
            pDevice->internalFormat = mal_format_s16;
        } break;

        case MAL_AL_FORMAT_REAR32:
        case MAL_AL_FORMAT_QUAD32:
        case MAL_AL_FORMAT_51CHN32:
        case MAL_AL_FORMAT_61CHN32:
        case MAL_AL_FORMAT_71CHN32:
        {
            pDevice->internalFormat = mal_format_s32;
        } break;

        case MAL_AL_FORMAT_MONO_FLOAT32:
        case MAL_AL_FORMAT_STEREO_FLOAT32:
        {
            pDevice->internalFormat = mal_format_f32;
        } break;
    }

    // From what I can tell, the ordering of channels is fixed for OpenAL.
    switch (formatAL)
    {
        case MAL_AL_FORMAT_MONO8:
        case MAL_AL_FORMAT_MONO16:
        case MAL_AL_FORMAT_MONO_FLOAT32:
        {
            pDevice->internalChannelMap[0] = MAL_CHANNEL_FRONT_CENTER;
        } break;

        case MAL_AL_FORMAT_STEREO8:
        case MAL_AL_FORMAT_STEREO16:
        case MAL_AL_FORMAT_STEREO_FLOAT32:
        {
            pDevice->internalChannelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            pDevice->internalChannelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
        } break;

        case MAL_AL_FORMAT_REAR8:
        case MAL_AL_FORMAT_REAR16:
        case MAL_AL_FORMAT_REAR32:
        {
            pDevice->internalChannelMap[0] = MAL_CHANNEL_BACK_LEFT;
            pDevice->internalChannelMap[1] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case MAL_AL_FORMAT_QUAD8:
        case MAL_AL_FORMAT_QUAD16:
        case MAL_AL_FORMAT_QUAD32:
        {
            pDevice->internalChannelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            pDevice->internalChannelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            pDevice->internalChannelMap[2] = MAL_CHANNEL_BACK_LEFT;
            pDevice->internalChannelMap[3] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case MAL_AL_FORMAT_51CHN8:
        case MAL_AL_FORMAT_51CHN16:
        case MAL_AL_FORMAT_51CHN32:
        {
            pDevice->internalChannelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            pDevice->internalChannelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            pDevice->internalChannelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            pDevice->internalChannelMap[3] = MAL_CHANNEL_LFE;
            pDevice->internalChannelMap[4] = MAL_CHANNEL_BACK_LEFT;
            pDevice->internalChannelMap[5] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case MAL_AL_FORMAT_61CHN8:
        case MAL_AL_FORMAT_61CHN16:
        case MAL_AL_FORMAT_61CHN32:
        {
            pDevice->internalChannelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            pDevice->internalChannelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            pDevice->internalChannelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            pDevice->internalChannelMap[3] = MAL_CHANNEL_LFE;
            pDevice->internalChannelMap[4] = MAL_CHANNEL_BACK_CENTER;
            pDevice->internalChannelMap[5] = MAL_CHANNEL_SIDE_LEFT;
            pDevice->internalChannelMap[6] = MAL_CHANNEL_SIDE_RIGHT;
        } break;

        case MAL_AL_FORMAT_71CHN8:
        case MAL_AL_FORMAT_71CHN16:
        case MAL_AL_FORMAT_71CHN32:
        {
            pDevice->internalChannelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            pDevice->internalChannelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            pDevice->internalChannelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            pDevice->internalChannelMap[3] = MAL_CHANNEL_LFE;
            pDevice->internalChannelMap[4] = MAL_CHANNEL_BACK_LEFT;
            pDevice->internalChannelMap[5] = MAL_CHANNEL_BACK_RIGHT;
            pDevice->internalChannelMap[6] = MAL_CHANNEL_SIDE_LEFT;
            pDevice->internalChannelMap[7] = MAL_CHANNEL_SIDE_RIGHT;
        } break;

        default: break;
    }

    pDevice->openal.pDeviceALC = pDeviceALC;
    pDevice->openal.pContextALC = pContextALC;
    pDevice->openal.formatAL = formatAL;
    pDevice->openal.subBufferSizeInFrames = pDevice->bufferSizeInFrames / pDevice->periods;
    pDevice->openal.pIntermediaryBuffer = (mal_uint8*)mal_malloc(pDevice->openal.subBufferSizeInFrames * channelsAL * mal_get_bytes_per_sample(pDevice->internalFormat));
    if (pDevice->openal.pIntermediaryBuffer == NULL) {
        mal_device_uninit__openal(pDevice);
        return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "[OpenAL] Failed to allocate memory for intermediary buffer.", MAL_OUT_OF_MEMORY);
    }

    return MAL_SUCCESS;
}

mal_result mal_device__start_backend__openal(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->type == mal_device_type_playback) {
        // Playback.
        //
        // When starting playback we want to ensure each buffer is filled and queued before playing the source.
        pDevice->openal.iNextBuffer = 0;

        ((MAL_LPALCMAKECONTEXTCURRENT)pDevice->pContext->openal.alcMakeContextCurrent)((mal_ALCcontext*)pDevice->openal.pContextALC);

        for (mal_uint32 i = 0; i < pDevice->periods; ++i) {
            mal_device__read_frames_from_client(pDevice, pDevice->openal.subBufferSizeInFrames, pDevice->openal.pIntermediaryBuffer);

            mal_ALuint bufferAL = pDevice->openal.buffersAL[i];
            ((MAL_LPALBUFFERDATA)pDevice->pContext->openal.alBufferData)(bufferAL, pDevice->openal.formatAL, pDevice->openal.pIntermediaryBuffer, pDevice->openal.subBufferSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat), pDevice->internalSampleRate);
            ((MAL_LPALSOURCEQUEUEBUFFERS)pDevice->pContext->openal.alSourceQueueBuffers)(pDevice->openal.sourceAL, 1, &bufferAL);
        }

        // Start the source only after filling and queueing each buffer.
        ((MAL_LPALSOURCEPLAY)pDevice->pContext->openal.alSourcePlay)(pDevice->openal.sourceAL);
    } else {
        // Capture.
        ((MAL_LPALCCAPTURESTART)pDevice->pContext->openal.alcCaptureStart)((mal_ALCdevice*)pDevice->openal.pDeviceALC);
    }

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__openal(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->type == mal_device_type_playback) {
        ((MAL_LPALCMAKECONTEXTCURRENT)pDevice->pContext->openal.alcMakeContextCurrent)((mal_ALCcontext*)pDevice->openal.pContextALC);
        ((MAL_LPALSOURCESTOP)pDevice->pContext->openal.alSourceStop)(pDevice->openal.sourceAL);
    } else {
        ((MAL_LPALCCAPTURESTOP)pDevice->pContext->openal.alcCaptureStop)((mal_ALCdevice*)pDevice->openal.pDeviceALC);
    }

    return MAL_SUCCESS;
}

mal_result mal_device__break_main_loop__openal(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->openal.breakFromMainLoop = MAL_TRUE;
    return MAL_SUCCESS;
}

mal_uint32 mal_device__get_available_frames__openal(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    if (pDevice->type == mal_device_type_playback) {
        ((MAL_LPALCMAKECONTEXTCURRENT)pDevice->pContext->openal.alcMakeContextCurrent)((mal_ALCcontext*)pDevice->openal.pContextALC);

        mal_ALint processedBufferCount = 0;
        ((MAL_LPALGETSOURCEI)pDevice->pContext->openal.alGetSourcei)(pDevice->openal.sourceAL, MAL_AL_BUFFERS_PROCESSED, &processedBufferCount);

        return processedBufferCount * pDevice->openal.subBufferSizeInFrames;
    } else {
        mal_ALint samplesAvailable = 0;
        ((MAL_LPALCGETINTEGERV)pDevice->pContext->openal.alcGetIntegerv)((mal_ALCdevice*)pDevice->openal.pDeviceALC, MAL_ALC_CAPTURE_SAMPLES, 1, &samplesAvailable);

        return samplesAvailable / pDevice->channels;
    }
}

mal_uint32 mal_device__wait_for_frames__openal(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    while (!pDevice->openal.breakFromMainLoop) {
        mal_uint32 framesAvailable = mal_device__get_available_frames__openal(pDevice);
        if (framesAvailable > 0) {
            return framesAvailable;
        }

        mal_sleep(1);
    }

    // We'll get here if the loop was terminated. When capturing we want to return whatever is available. For playback we just drop it.
    if (pDevice->type == mal_device_type_playback) {
        return 0;
    } else {
        return mal_device__get_available_frames__openal(pDevice);
    }
}

mal_result mal_device__main_loop__openal(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    pDevice->openal.breakFromMainLoop = MAL_FALSE;
    while (!pDevice->openal.breakFromMainLoop) {
        mal_uint32 framesAvailable = mal_device__wait_for_frames__openal(pDevice);
        if (framesAvailable == 0) {
            continue;
        }

        // If it's a playback device, don't bother grabbing more data if the device is being stopped.
        if (pDevice->openal.breakFromMainLoop && pDevice->type == mal_device_type_playback) {
            return MAL_FALSE;
        }

        if (pDevice->type == mal_device_type_playback) {
            while (framesAvailable > 0) {
                mal_uint32 framesToRead = (framesAvailable > pDevice->openal.subBufferSizeInFrames) ? pDevice->openal.subBufferSizeInFrames : framesAvailable;

                mal_ALuint bufferAL = pDevice->openal.buffersAL[pDevice->openal.iNextBuffer];
                pDevice->openal.iNextBuffer = (pDevice->openal.iNextBuffer + 1) % pDevice->periods;

                mal_device__read_frames_from_client(pDevice, framesToRead, pDevice->openal.pIntermediaryBuffer);

                ((MAL_LPALCMAKECONTEXTCURRENT)pDevice->pContext->openal.alcMakeContextCurrent)((mal_ALCcontext*)pDevice->openal.pContextALC);
                ((MAL_LPALSOURCEUNQUEUEBUFFERS)pDevice->pContext->openal.alSourceUnqueueBuffers)(pDevice->openal.sourceAL, 1, &bufferAL);
                ((MAL_LPALBUFFERDATA)pDevice->pContext->openal.alBufferData)(bufferAL, pDevice->openal.formatAL, pDevice->openal.pIntermediaryBuffer, pDevice->openal.subBufferSizeInFrames * pDevice->internalChannels * mal_get_bytes_per_sample(pDevice->internalFormat), pDevice->internalSampleRate);
                ((MAL_LPALSOURCEQUEUEBUFFERS)pDevice->pContext->openal.alSourceQueueBuffers)(pDevice->openal.sourceAL, 1, &bufferAL);

                framesAvailable -= framesToRead;
            }


            // There's a chance the source has stopped playing due to there not being any buffer's queue. Make sure it's restarted.
            mal_ALenum state;
            ((MAL_LPALGETSOURCEI)pDevice->pContext->openal.alGetSourcei)(pDevice->openal.sourceAL, MAL_AL_SOURCE_STATE, &state);

            if (state != MAL_AL_PLAYING) {
                ((MAL_LPALSOURCEPLAY)pDevice->pContext->openal.alSourcePlay)(pDevice->openal.sourceAL);
            }
        } else {
            while (framesAvailable > 0) {
                mal_uint32 framesToSend = (framesAvailable > pDevice->openal.subBufferSizeInFrames) ? pDevice->openal.subBufferSizeInFrames : framesAvailable;
                ((MAL_LPALCCAPTURESAMPLES)pDevice->pContext->openal.alcCaptureSamples)((mal_ALCdevice*)pDevice->openal.pDeviceALC, pDevice->openal.pIntermediaryBuffer, framesToSend);

                mal_device__send_frames_to_client(pDevice, framesToSend, pDevice->openal.pIntermediaryBuffer);
                framesAvailable -= framesToSend;
            }
        }
    }

    return MAL_SUCCESS;
}


mal_result mal_context_uninit__openal(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_openal);

#ifndef MAL_NO_RUNTIME_LINKING
    mal_dlclose(pContext->openal.hOpenAL);
#endif

    return MAL_SUCCESS;
}

mal_result mal_context_init__openal(mal_context* pContext)
{
    mal_assert(pContext != NULL);

#ifndef MAL_NO_RUNTIME_LINKING
    const char* libNames[] = {
#if defined(MAL_WIN32)
        "OpenAL32.dll",
        "soft_oal.dll"
#endif
#if defined(MAL_UNIX) && !defined(MAL_APPLE)
        "libopenal.so",
        "libopenal.so.1"
#endif
#if defined(MAL_APPLE)
        "OpenAL.framework/OpenAL"
#endif
    };

    for (size_t i = 0; i < mal_countof(libNames); ++i) {
        pContext->openal.hOpenAL = mal_dlopen(libNames[i]);
        if (pContext->openal.hOpenAL != NULL) {
            break;
        }
    }

    if (pContext->openal.hOpenAL == NULL) {
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    pContext->openal.alcCreateContext       = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcCreateContext");
    pContext->openal.alcMakeContextCurrent  = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcMakeContextCurrent");
    pContext->openal.alcProcessContext      = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcProcessContext");
    pContext->openal.alcSuspendContext      = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcSuspendContext");
    pContext->openal.alcDestroyContext      = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcDestroyContext");
    pContext->openal.alcGetCurrentContext   = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcGetCurrentContext");
    pContext->openal.alcGetContextsDevice   = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcGetContextsDevice");
    pContext->openal.alcOpenDevice          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcOpenDevice");
    pContext->openal.alcCloseDevice         = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcCloseDevice");
    pContext->openal.alcGetError            = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcGetError");
    pContext->openal.alcIsExtensionPresent  = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcIsExtensionPresent");
    pContext->openal.alcGetProcAddress      = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcGetProcAddress");
    pContext->openal.alcGetEnumValue        = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcGetEnumValue");
    pContext->openal.alcGetString           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcGetString");
    pContext->openal.alcGetIntegerv         = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcGetIntegerv");
    pContext->openal.alcCaptureOpenDevice   = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcCaptureOpenDevice");
    pContext->openal.alcCaptureCloseDevice  = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcCaptureCloseDevice");
    pContext->openal.alcCaptureStart        = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcCaptureStart");
    pContext->openal.alcCaptureStop         = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcCaptureStop");
    pContext->openal.alcCaptureSamples      = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alcCaptureSamples");

    pContext->openal.alEnable               = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alEnable");
    pContext->openal.alDisable              = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alDisable");
    pContext->openal.alIsEnabled            = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alIsEnabled");
    pContext->openal.alGetString            = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetString");
    pContext->openal.alGetBooleanv          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetBooleanv");
    pContext->openal.alGetIntegerv          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetIntegerv");
    pContext->openal.alGetFloatv            = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetFloatv");
    pContext->openal.alGetDoublev           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetDoublev");
    pContext->openal.alGetBoolean           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetBoolean");
    pContext->openal.alGetInteger           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetInteger");
    pContext->openal.alGetFloat             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetFloat");
    pContext->openal.alGetDouble            = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetDouble");
    pContext->openal.alGetError             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetError");
    pContext->openal.alIsExtensionPresent   = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alIsExtensionPresent");
    pContext->openal.alGetProcAddress       = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetProcAddress");
    pContext->openal.alGetEnumValue         = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetEnumValue");
    pContext->openal.alGenSources           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGenSources");
    pContext->openal.alDeleteSources        = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alDeleteSources");
    pContext->openal.alIsSource             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alIsSource");
    pContext->openal.alSourcef              = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourcef");
    pContext->openal.alSource3f             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSource3f");
    pContext->openal.alSourcefv             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourcefv");
    pContext->openal.alSourcei              = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourcei");
    pContext->openal.alSource3i             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSource3i");
    pContext->openal.alSourceiv             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourceiv");
    pContext->openal.alGetSourcef           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetSourcef");
    pContext->openal.alGetSource3f          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetSource3f");
    pContext->openal.alGetSourcefv          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetSourcefv");
    pContext->openal.alGetSourcei           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetSourcei");
    pContext->openal.alGetSource3i          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetSource3i");
    pContext->openal.alGetSourceiv          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetSourceiv");
    pContext->openal.alSourcePlayv          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourcePlayv");
    pContext->openal.alSourceStopv          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourceStopv");
    pContext->openal.alSourceRewindv        = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourceRewindv");
    pContext->openal.alSourcePausev         = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourcePausev");
    pContext->openal.alSourcePlay           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourcePlay");
    pContext->openal.alSourceStop           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourceStop");
    pContext->openal.alSourceRewind         = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourceRewind");
    pContext->openal.alSourcePause          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourcePause");
    pContext->openal.alSourceQueueBuffers   = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourceQueueBuffers");
    pContext->openal.alSourceUnqueueBuffers = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alSourceUnqueueBuffers");
    pContext->openal.alGenBuffers           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGenBuffers");
    pContext->openal.alDeleteBuffers        = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alDeleteBuffers");
    pContext->openal.alIsBuffer             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alIsBuffer");
    pContext->openal.alBufferData           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alBufferData");
    pContext->openal.alBufferf              = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alBufferf");
    pContext->openal.alBuffer3f             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alBuffer3f");
    pContext->openal.alBufferfv             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alBufferfv");
    pContext->openal.alBufferi              = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alBufferi");
    pContext->openal.alBuffer3i             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alBuffer3i");
    pContext->openal.alBufferiv             = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alBufferiv");
    pContext->openal.alGetBufferf           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetBufferf");
    pContext->openal.alGetBuffer3f          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetBuffer3f");
    pContext->openal.alGetBufferfv          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetBufferfv");
    pContext->openal.alGetBufferi           = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetBufferi");
    pContext->openal.alGetBuffer3i          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetBuffer3i");
    pContext->openal.alGetBufferiv          = (mal_proc)mal_dlsym(pContext->openal.hOpenAL, "alGetBufferiv");
#else
    pContext->openal.alcCreateContext       = (mal_proc)alcCreateContext;
    pContext->openal.alcMakeContextCurrent  = (mal_proc)alcMakeContextCurrent;
    pContext->openal.alcProcessContext      = (mal_proc)alcProcessContext;
    pContext->openal.alcSuspendContext      = (mal_proc)alcSuspendContext;
    pContext->openal.alcDestroyContext      = (mal_proc)alcDestroyContext;
    pContext->openal.alcGetCurrentContext   = (mal_proc)alcGetCurrentContext;
    pContext->openal.alcGetContextsDevice   = (mal_proc)alcGetContextsDevice;
    pContext->openal.alcOpenDevice          = (mal_proc)alcOpenDevice;
    pContext->openal.alcCloseDevice         = (mal_proc)alcCloseDevice;
    pContext->openal.alcGetError            = (mal_proc)alcGetError;
    pContext->openal.alcIsExtensionPresent  = (mal_proc)alcIsExtensionPresent;
    pContext->openal.alcGetProcAddress      = (mal_proc)alcGetProcAddress;
    pContext->openal.alcGetEnumValue        = (mal_proc)alcGetEnumValue;
    pContext->openal.alcGetString           = (mal_proc)alcGetString;
    pContext->openal.alcGetIntegerv         = (mal_proc)alcGetIntegerv;
    pContext->openal.alcCaptureOpenDevice   = (mal_proc)alcCaptureOpenDevice;
    pContext->openal.alcCaptureCloseDevice  = (mal_proc)alcCaptureCloseDevice;
    pContext->openal.alcCaptureStart        = (mal_proc)alcCaptureStart;
    pContext->openal.alcCaptureStop         = (mal_proc)alcCaptureStop;
    pContext->openal.alcCaptureSamples      = (mal_proc)alcCaptureSamples;

    pContext->openal.alEnable               = (mal_proc)alEnable;
    pContext->openal.alDisable              = (mal_proc)alDisable;
    pContext->openal.alIsEnabled            = (mal_proc)alIsEnabled;
    pContext->openal.alGetString            = (mal_proc)alGetString;
    pContext->openal.alGetBooleanv          = (mal_proc)alGetBooleanv;
    pContext->openal.alGetIntegerv          = (mal_proc)alGetIntegerv;
    pContext->openal.alGetFloatv            = (mal_proc)alGetFloatv;
    pContext->openal.alGetDoublev           = (mal_proc)alGetDoublev;
    pContext->openal.alGetBoolean           = (mal_proc)alGetBoolean;
    pContext->openal.alGetInteger           = (mal_proc)alGetInteger;
    pContext->openal.alGetFloat             = (mal_proc)alGetFloat;
    pContext->openal.alGetDouble            = (mal_proc)alGetDouble;
    pContext->openal.alGetError             = (mal_proc)alGetError;
    pContext->openal.alIsExtensionPresent   = (mal_proc)alIsExtensionPresent;
    pContext->openal.alGetProcAddress       = (mal_proc)alGetProcAddress;
    pContext->openal.alGetEnumValue         = (mal_proc)alGetEnumValue;
    pContext->openal.alGenSources           = (mal_proc)alGenSources;
    pContext->openal.alDeleteSources        = (mal_proc)alDeleteSources;
    pContext->openal.alIsSource             = (mal_proc)alIsSource;
    pContext->openal.alSourcef              = (mal_proc)alSourcef;
    pContext->openal.alSource3f             = (mal_proc)alSource3f;
    pContext->openal.alSourcefv             = (mal_proc)alSourcefv;
    pContext->openal.alSourcei              = (mal_proc)alSourcei;
    pContext->openal.alSource3i             = (mal_proc)alSource3i;
    pContext->openal.alSourceiv             = (mal_proc)alSourceiv;
    pContext->openal.alGetSourcef           = (mal_proc)alGetSourcef;
    pContext->openal.alGetSource3f          = (mal_proc)alGetSource3f;
    pContext->openal.alGetSourcefv          = (mal_proc)alGetSourcefv;
    pContext->openal.alGetSourcei           = (mal_proc)alGetSourcei;
    pContext->openal.alGetSource3i          = (mal_proc)alGetSource3i;
    pContext->openal.alGetSourceiv          = (mal_proc)alGetSourceiv;
    pContext->openal.alSourcePlayv          = (mal_proc)alSourcePlayv;
    pContext->openal.alSourceStopv          = (mal_proc)alSourceStopv;
    pContext->openal.alSourceRewindv        = (mal_proc)alSourceRewindv;
    pContext->openal.alSourcePausev         = (mal_proc)alSourcePausev;
    pContext->openal.alSourcePlay           = (mal_proc)alSourcePlay;
    pContext->openal.alSourceStop           = (mal_proc)alSourceStop;
    pContext->openal.alSourceRewind         = (mal_proc)alSourceRewind;
    pContext->openal.alSourcePause          = (mal_proc)alSourcePause;
    pContext->openal.alSourceQueueBuffers   = (mal_proc)alSourceQueueBuffers;
    pContext->openal.alSourceUnqueueBuffers = (mal_proc)alSourceUnqueueBuffers;
    pContext->openal.alGenBuffers           = (mal_proc)alGenBuffers;
    pContext->openal.alDeleteBuffers        = (mal_proc)alDeleteBuffers;
    pContext->openal.alIsBuffer             = (mal_proc)alIsBuffer;
    pContext->openal.alBufferData           = (mal_proc)alBufferData;
    pContext->openal.alBufferf              = (mal_proc)alBufferf;
    pContext->openal.alBuffer3f             = (mal_proc)alBuffer3f;
    pContext->openal.alBufferfv             = (mal_proc)alBufferfv;
    pContext->openal.alBufferi              = (mal_proc)alBufferi;
    pContext->openal.alBuffer3i             = (mal_proc)alBuffer3i;
    pContext->openal.alBufferiv             = (mal_proc)alBufferiv;
    pContext->openal.alGetBufferf           = (mal_proc)alGetBufferf;
    pContext->openal.alGetBuffer3f          = (mal_proc)alGetBuffer3f;
    pContext->openal.alGetBufferfv          = (mal_proc)alGetBufferfv;
    pContext->openal.alGetBufferi           = (mal_proc)alGetBufferi;
    pContext->openal.alGetBuffer3i          = (mal_proc)alGetBuffer3i;
    pContext->openal.alGetBufferiv          = (mal_proc)alGetBufferiv;
#endif

    // We depend on the ALC_ENUMERATION_EXT extension for enumeration. If this is not supported we fall back to default devices.
    pContext->openal.isEnumerationSupported = ((MAL_LPALCISEXTENSIONPRESENT)pContext->openal.alcIsExtensionPresent)(NULL, "ALC_ENUMERATION_EXT");
    pContext->openal.isFloat32Supported     = ((MAL_LPALISEXTENSIONPRESENT)pContext->openal.alIsExtensionPresent)("AL_EXT_float32");
    pContext->openal.isMCFormatsSupported   = ((MAL_LPALISEXTENSIONPRESENT)pContext->openal.alIsExtensionPresent)("AL_EXT_MCFORMATS");

    pContext->onUninit              = mal_context_uninit__openal;
    pContext->onDeviceIDEqual       = mal_context_is_device_id_equal__openal;
    pContext->onEnumDevices         = mal_context_enumerate_devices__openal;
    pContext->onGetDeviceInfo       = mal_context_get_device_info__openal;
    pContext->onDeviceInit          = mal_device_init__openal;
    pContext->onDeviceUninit        = mal_device_uninit__openal;
    pContext->onDeviceStart         = mal_device__start_backend__openal;
    pContext->onDeviceStop          = mal_device__stop_backend__openal;
    pContext->onDeviceBreakMainLoop = mal_device__break_main_loop__openal;
    pContext->onDeviceMainLoop      = mal_device__main_loop__openal;

    return MAL_SUCCESS;
}
#endif  // OpenAL



///////////////////////////////////////////////////////////////////////////////
//
// SDL Backend
//
///////////////////////////////////////////////////////////////////////////////
#ifdef MAL_HAS_SDL

//#define MAL_USE_SDL_1

#define MAL_SDL_INIT_AUDIO                      0x00000010
#define MAL_AUDIO_U8                            0x0008
#define MAL_AUDIO_S16                           0x8010
#define MAL_AUDIO_S32                           0x8020
#define MAL_AUDIO_F32                           0x8120
#define MAL_SDL_AUDIO_ALLOW_FREQUENCY_CHANGE    0x00000001
#define MAL_SDL_AUDIO_ALLOW_FORMAT_CHANGE       0x00000002
#define MAL_SDL_AUDIO_ALLOW_CHANNELS_CHANGE     0x00000004
#define MAL_SDL_AUDIO_ALLOW_ANY_CHANGE          (MAL_SDL_AUDIO_ALLOW_FREQUENCY_CHANGE | MAL_SDL_AUDIO_ALLOW_FORMAT_CHANGE | MAL_SDL_AUDIO_ALLOW_CHANNELS_CHANGE)

// If we are linking at compile time we'll just #include SDL.h. Otherwise we can just redeclare some stuff to avoid the
// need for development packages to be installed.
#ifdef MAL_NO_RUNTIME_LINKING
    #define SDL_MAIN_HANDLED
    #ifdef MAL_EMSCRIPTEN
        #include <SDL/SDL.h>

        // For now just use SDL 1.2 with Emscripten. This avoids the need for "-s USE_SDL=2" at compile time.
        #ifndef MAL_USE_SDL_1
        #define MAL_USE_SDL_1
        #endif
    #else
        #include <SDL2/SDL.h>
    #endif

    typedef SDL_AudioCallback   MAL_SDL_AudioCallback;
    typedef SDL_AudioSpec       MAL_SDL_AudioSpec;
    typedef SDL_AudioFormat     MAL_SDL_AudioFormat;
    typedef SDL_AudioDeviceID   MAL_SDL_AudioDeviceID;
#else
    typedef void (* MAL_SDL_AudioCallback)(void* userdata, mal_uint8* stream, int len);
    typedef mal_uint16 MAL_SDL_AudioFormat;
    typedef mal_uint32 MAL_SDL_AudioDeviceID;

    typedef struct MAL_SDL_AudioSpec
    {
        int freq;
        MAL_SDL_AudioFormat format;
        mal_uint8 channels;
        mal_uint8 silence;
        mal_uint16 samples;
        mal_uint16 padding;
        mal_uint32 size;
        MAL_SDL_AudioCallback callback;
        void* userdata;
    } MAL_SDL_AudioSpec;
#endif

typedef int                   (* MAL_PFN_SDL_InitSubSystem)(mal_uint32 flags);
typedef void                  (* MAL_PFN_SDL_QuitSubSystem)(mal_uint32 flags);
typedef int                   (* MAL_PFN_SDL_GetNumAudioDevices)(int iscapture);
typedef const char*           (* MAL_PFN_SDL_GetAudioDeviceName)(int index, int iscapture);
typedef void                  (* MAL_PFN_SDL_CloseAudio)(void);
typedef void                  (* MAL_PFN_SDL_CloseAudioDevice)(MAL_SDL_AudioDeviceID dev);
typedef int                   (* MAL_PFN_SDL_OpenAudio)(MAL_SDL_AudioSpec* desired, MAL_SDL_AudioSpec* obtained);
typedef MAL_SDL_AudioDeviceID (* MAL_PFN_SDL_OpenAudioDevice)(const char* device, int iscapture, const MAL_SDL_AudioSpec* desired, MAL_SDL_AudioSpec* obtained, int allowed_changes);
typedef void                  (* MAL_PFN_SDL_PauseAudio)(int pause_on);
typedef void                  (* MAL_PFN_SDL_PauseAudioDevice)(MAL_SDL_AudioDeviceID dev, int pause_on);

MAL_SDL_AudioFormat mal_format_to_sdl(mal_format format)
{
    switch (format)
    {
    case mal_format_unknown: return 0;
    case mal_format_u8:      return MAL_AUDIO_U8;
    case mal_format_s16:     return MAL_AUDIO_S16;
    case mal_format_s24:     return MAL_AUDIO_S32;  // Closest match.
    case mal_format_s32:     return MAL_AUDIO_S32;
    default:                 return 0;
    }
}

mal_format mal_format_from_sdl(MAL_SDL_AudioFormat format)
{
    switch (format)
    {
        case MAL_AUDIO_U8:  return mal_format_u8;
        case MAL_AUDIO_S16: return mal_format_s16;
        case MAL_AUDIO_S32: return mal_format_s32;
        case MAL_AUDIO_F32: return mal_format_f32;
        default:            return mal_format_unknown;
    }
}

mal_bool32 mal_context_is_device_id_equal__sdl(mal_context* pContext, const mal_device_id* pID0, const mal_device_id* pID1)
{
    mal_assert(pContext != NULL);
    mal_assert(pID0 != NULL);
    mal_assert(pID1 != NULL);
    (void)pContext;

    return pID0->sdl == pID1->sdl;
}

mal_result mal_context_enumerate_devices__sdl(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    mal_assert(pContext != NULL);
    mal_assert(callback != NULL);

#ifndef MAL_USE_SDL_1
    if (!pContext->sdl.usingSDL1) {
        mal_bool32 isTerminated = MAL_FALSE;

        // Playback
        if (!isTerminated) {
            int deviceCount = ((MAL_PFN_SDL_GetNumAudioDevices)pContext->sdl.SDL_GetNumAudioDevices)(0);
            for (int i = 0; i < deviceCount; ++i) {
                mal_device_info deviceInfo;
                mal_zero_object(&deviceInfo);

                deviceInfo.id.sdl = i;
                mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), ((MAL_PFN_SDL_GetAudioDeviceName)pContext->sdl.SDL_GetAudioDeviceName)(i, 0), (size_t)-1);

                mal_bool32 cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
                if (cbResult == MAL_FALSE) {
                    isTerminated = MAL_TRUE;
                    break;
                }
            }
        }

        // Capture
        if (!isTerminated) {
            int deviceCount = ((MAL_PFN_SDL_GetNumAudioDevices)pContext->sdl.SDL_GetNumAudioDevices)(1);
            for (int i = 0; i < deviceCount; ++i) {
                mal_device_info deviceInfo;
                mal_zero_object(&deviceInfo);

                deviceInfo.id.sdl = i;
                mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), ((MAL_PFN_SDL_GetAudioDeviceName)pContext->sdl.SDL_GetAudioDeviceName)(i, 1), (size_t)-1);

                mal_bool32 cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
                if (cbResult == MAL_FALSE) {
                    isTerminated = MAL_TRUE;
                    break;
                }
            }
        }
    } else
#endif
    {
        // SDL1 only uses default devices, and does not support capture.
        mal_bool32 cbResult = MAL_TRUE;

        // Playback.
        if (cbResult) {
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
            cbResult = callback(pContext, mal_device_type_playback, &deviceInfo, pUserData);
        }

#if 0   // No capture with SDL1.
        // Capture.
        if (cbResult) {
            mal_device_info deviceInfo;
            mal_zero_object(&deviceInfo);
            mal_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
            cbResult = callback(pContext, mal_device_type_capture, &deviceInfo, pUserData);
        }
#endif
    }

    return MAL_SUCCESS;
}

mal_result mal_context_get_device_info__sdl(mal_context* pContext, mal_device_type deviceType, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    mal_assert(pContext != NULL);
    (void)shareMode;

#ifndef MAL_USE_SDL_1
    if (!pContext->sdl.usingSDL1) {
        if (pDeviceID == NULL) {
            if (deviceType == mal_device_type_playback) {
                pDeviceInfo->id.sdl = 0;
                mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
            } else {
                pDeviceInfo->id.sdl = 0;
                mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
            }
        } else {
            pDeviceInfo->id.sdl = pDeviceID->sdl;
            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), ((MAL_PFN_SDL_GetAudioDeviceName)pContext->sdl.SDL_GetAudioDeviceName)(pDeviceID->sdl, (deviceType == mal_device_type_playback) ? 0 : 1), (size_t)-1);
        }
    } else
#endif
    {
        // SDL1 uses default devices.
        if (deviceType == mal_device_type_playback) {
            pDeviceInfo->id.sdl = 0;
            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        } else {
            pDeviceInfo->id.sdl = 0;
            mal_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        }
    }

    // To get an accurate idea on the backend's native format we need to open the device. Not ideal, but it's the only way. An
    // alternative to this is to report all channel counts, sample rates and formats, but that doesn't offer a good representation
    // of the device's _actual_ ideal format.
    //
    // Note: With Emscripten, it looks like non-zero values need to be specified for desiredSpec. Whatever is specified in
    // desiredSpec will be used by SDL since it uses it just does it's own format conversion internally. Therefore, from what
    // I can tell, there's no real way to know the device's actual format which means I'm just going to fall back to the full
    // range of channels and sample rates on Emscripten builds.
#if defined(__EMSCRIPTEN__)
    pDeviceInfo->minChannels = MAL_MIN_CHANNELS;
    pDeviceInfo->maxChannels = MAL_MAX_CHANNELS;
    pDeviceInfo->minSampleRate = MAL_MIN_SAMPLE_RATE;
    pDeviceInfo->maxSampleRate = MAL_MAX_SAMPLE_RATE;
    pDeviceInfo->formatCount = 3;
    pDeviceInfo->formats[0] = mal_format_u8;
    pDeviceInfo->formats[1] = mal_format_s16;
    pDeviceInfo->formats[2] = mal_format_s32;
#else
    MAL_SDL_AudioSpec desiredSpec, obtainedSpec;
    mal_zero_memory(&desiredSpec, sizeof(desiredSpec));

#ifndef MAL_USE_SDL_1
    if (!pContext->sdl.usingSDL1) {
        int isCapture = (deviceType == mal_device_type_playback) ? 0 : 1;

        const char* pDeviceName = NULL;
        if (pDeviceID != NULL) {
            pDeviceName = ((MAL_PFN_SDL_GetAudioDeviceName)pContext->sdl.SDL_GetAudioDeviceName)(pDeviceID->sdl, isCapture);
        }

        MAL_SDL_AudioDeviceID tempDeviceID = ((MAL_PFN_SDL_OpenAudioDevice)pContext->sdl.SDL_OpenAudioDevice)(pDeviceName, isCapture, &desiredSpec, &obtainedSpec, MAL_SDL_AUDIO_ALLOW_ANY_CHANGE);
        if (tempDeviceID == 0) {
            return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "Failed to open SDL device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        ((MAL_PFN_SDL_CloseAudioDevice)pContext->sdl.SDL_CloseAudioDevice)(tempDeviceID);
    } else
#endif
    {
        // SDL1 uses default devices.
        (void)pDeviceID;

        // SDL1 only supports playback as far as I can tell.
        if (deviceType != mal_device_type_playback) {
            return MAL_NO_DEVICE;
        }

        MAL_SDL_AudioDeviceID tempDeviceID = ((MAL_PFN_SDL_OpenAudio)pContext->sdl.SDL_OpenAudio)(&desiredSpec, &obtainedSpec);
        if (tempDeviceID != 0) {
            return mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_ERROR, "Failed to open SDL device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        ((MAL_PFN_SDL_CloseAudio)pContext->sdl.SDL_CloseAudio)();
    }

    pDeviceInfo->minChannels = obtainedSpec.channels;
    pDeviceInfo->maxChannels = obtainedSpec.channels;
    pDeviceInfo->minSampleRate = obtainedSpec.freq;
    pDeviceInfo->maxSampleRate = obtainedSpec.freq;
    pDeviceInfo->formatCount = 1;
    if (obtainedSpec.format == MAL_AUDIO_U8) {
        pDeviceInfo->formats[0] = mal_format_u8;
    } else if (obtainedSpec.format == MAL_AUDIO_S16) {
        pDeviceInfo->formats[0] = mal_format_s16;
    } else if (obtainedSpec.format == MAL_AUDIO_S32) {
        pDeviceInfo->formats[0] = mal_format_s32;
    } else if (obtainedSpec.format == MAL_AUDIO_F32) {
        pDeviceInfo->formats[0] = mal_format_f32;
    } else {
        return MAL_FORMAT_NOT_SUPPORTED;
    }
#endif

    return MAL_SUCCESS;
}


void mal_device_uninit__sdl(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

#ifndef MAL_USE_SDL_1
    if (!pDevice->pContext->sdl.usingSDL1) {
        ((MAL_PFN_SDL_CloseAudioDevice)pDevice->pContext->sdl.SDL_CloseAudioDevice)(pDevice->sdl.deviceID);
    } else
#endif
    {
        ((MAL_PFN_SDL_CloseAudio)pDevice->pContext->sdl.SDL_CloseAudio)();
    }
}


void mal_audio_callback__sdl(void* pUserData, mal_uint8* pBuffer, int bufferSizeInBytes)
{
    mal_device* pDevice = (mal_device*)pUserData;
    mal_assert(pDevice != NULL);

    mal_uint32 bufferSizeInFrames = (mal_uint32)bufferSizeInBytes / mal_get_bytes_per_sample(pDevice->internalFormat) / pDevice->internalChannels;

    if (pDevice->type == mal_device_type_playback) {
        mal_device__read_frames_from_client(pDevice, bufferSizeInFrames, pBuffer);
    } else {
        mal_device__send_frames_to_client(pDevice, bufferSizeInFrames, pBuffer);
    }
}

mal_result mal_device_init__sdl(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, const mal_device_config* pConfig, mal_device* pDevice)
{
    mal_assert(pContext != NULL);
    mal_assert(pConfig != NULL);
    mal_assert(pDevice != NULL);

    (void)pContext;

    if (pDevice->bufferSizeInFrames == 0) {
        pDevice->bufferSizeInFrames = mal_calculate_buffer_size_in_frames_from_milliseconds(pDevice->bufferSizeInMilliseconds, pDevice->sampleRate);
    }

    // SDL wants the buffer size to be a power of 2. The SDL_AudioSpec property for this is only a Uint16, so we need
    // to explicitly clamp this because it will be easy to overflow.
    mal_uint32 bufferSize = pDevice->bufferSizeInFrames;
    if (bufferSize > 32768) {
        bufferSize = 32768;
    } else {
        bufferSize = mal_next_power_of_2(bufferSize);
    }

    mal_assert(bufferSize <= 32768);


    MAL_SDL_AudioSpec desiredSpec, obtainedSpec;
    mal_zero_memory(&desiredSpec, sizeof(desiredSpec));
    desiredSpec.freq     = (int)pConfig->sampleRate;
    desiredSpec.format   = mal_format_to_sdl(pConfig->format);
    desiredSpec.channels = (mal_uint8)pConfig->channels;
    desiredSpec.samples  = (mal_uint16)bufferSize;
    desiredSpec.callback = mal_audio_callback__sdl;
    desiredSpec.userdata = pDevice;

    // Fall back to f32 if we don't have an appropriate mapping between mini_al and SDL.
    if (desiredSpec.format == 0) {
        desiredSpec.format = MAL_AUDIO_F32;
    }

#ifndef MAL_USE_SDL_1
    if (!pDevice->pContext->sdl.usingSDL1) {
        int isCapture = (type == mal_device_type_playback) ? 0 : 1;

        const char* pDeviceName = NULL;
        if (pDeviceID != NULL) {
            pDeviceName = ((MAL_PFN_SDL_GetAudioDeviceName)pDevice->pContext->sdl.SDL_GetAudioDeviceName)(pDeviceID->sdl, isCapture);
        }

        pDevice->sdl.deviceID = ((MAL_PFN_SDL_OpenAudioDevice)pDevice->pContext->sdl.SDL_OpenAudioDevice)(pDeviceName, isCapture, &desiredSpec, &obtainedSpec, MAL_SDL_AUDIO_ALLOW_ANY_CHANGE);
        if (pDevice->sdl.deviceID == 0) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "Failed to open SDL2 device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    } else
#endif
    {
        // SDL1 uses default devices.
        (void)pDeviceID;

        // SDL1 only supports playback as far as I can tell.
        if (type != mal_device_type_playback) {
            return MAL_NO_DEVICE;
        }

        // SDL1 does not support floating point formats.
        if (desiredSpec.format == MAL_AUDIO_F32) {
            desiredSpec.format  = MAL_AUDIO_S16;
        }

        int deviceID = ((MAL_PFN_SDL_OpenAudio)pDevice->pContext->sdl.SDL_OpenAudio)(&desiredSpec, &obtainedSpec);
        if (deviceID < 0) {
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "Failed to open SDL1 device.", MAL_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        pDevice->sdl.deviceID = (mal_uint32)deviceID;
    }

    pDevice->internalFormat     = mal_format_from_sdl(obtainedSpec.format);
    pDevice->internalChannels   = obtainedSpec.channels;
    pDevice->internalSampleRate = (mal_uint32)obtainedSpec.freq;
    mal_get_standard_channel_map(mal_standard_channel_map_default, pDevice->internalChannels, pDevice->internalChannelMap);
    pDevice->bufferSizeInFrames = obtainedSpec.samples;
    pDevice->periods            = 1;    // SDL doesn't seem to tell us what the period count is. Just set this 1.

#ifdef MAL_DEBUG_OUTPUT
    printf("=== SDL CONFIG ===\n");
    printf("REQUESTED -> RECEIVED\n");
    printf("    FORMAT:                 %s -> %s\n", mal_get_format_name(pConfig->format), mal_get_format_name(pDevice->internalFormat));
    printf("    CHANNELS:               %d -> %d\n", desiredSpec.channels, obtainedSpec.channels);
    printf("    SAMPLE RATE:            %d -> %d\n", desiredSpec.freq, obtainedSpec.freq);
    printf("    BUFFER SIZE IN SAMPLES: %d -> %d\n", desiredSpec.samples, obtainedSpec.samples);
#endif

    return MAL_SUCCESS;
}

mal_result mal_device__start_backend__sdl(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

#ifndef MAL_USE_SDL_1
    if (!pDevice->pContext->sdl.usingSDL1) {
        ((MAL_PFN_SDL_PauseAudioDevice)pDevice->pContext->sdl.SDL_PauseAudioDevice)(pDevice->sdl.deviceID, 0);
    } else
#endif
    {
        ((MAL_PFN_SDL_PauseAudio)pDevice->pContext->sdl.SDL_PauseAudio)(0);
    }

    return MAL_SUCCESS;
}

mal_result mal_device__stop_backend__sdl(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

#ifndef MAL_USE_SDL_1
    if (!pDevice->pContext->sdl.usingSDL1) {
        ((MAL_PFN_SDL_PauseAudioDevice)pDevice->pContext->sdl.SDL_PauseAudioDevice)(pDevice->sdl.deviceID, 1);
    } else
#endif
    {
        ((MAL_PFN_SDL_PauseAudio)pDevice->pContext->sdl.SDL_PauseAudio)(1);
    }
    
    mal_device__set_state(pDevice, MAL_STATE_STOPPED);
    mal_stop_proc onStop = pDevice->onStop;
    if (onStop) {
        onStop(pDevice);
    }

    return MAL_SUCCESS;
}


mal_result mal_context_uninit__sdl(mal_context* pContext)
{
    mal_assert(pContext != NULL);
    mal_assert(pContext->backend == mal_backend_sdl);

    ((MAL_PFN_SDL_QuitSubSystem)pContext->sdl.SDL_QuitSubSystem)(MAL_SDL_INIT_AUDIO);
    return MAL_SUCCESS;
}

mal_result mal_context_init__sdl(mal_context* pContext)
{
    mal_assert(pContext != NULL);

#ifndef MAL_NO_RUNTIME_LINKING
    // Run-time linking.
    const char* libNames[] = {
#if defined(MAL_WIN32)
        "SDL2.dll",
        "SDL.dll"
#elif defined(MAL_APPLE)
        "SDL2.framework/SDL2",
        "SDL.framework/SDL"
#else
        "libSDL2-2.0.so.0",
        "libSDL-1.2.so.0"
#endif
    };

    for (size_t i = 0; i < mal_countof(libNames); ++i) {
        pContext->sdl.hSDL = mal_dlopen(libNames[i]);
        if (pContext->sdl.hSDL != NULL) {
            break;
        }
    }

    if (pContext->sdl.hSDL == NULL) {
        return MAL_NO_BACKEND;  // Couldn't find SDL2.dll, etc. Most likely it's not installed.
    }

    pContext->sdl.SDL_InitSubSystem      = mal_dlsym(pContext->sdl.hSDL, "SDL_InitSubSystem");
    pContext->sdl.SDL_QuitSubSystem      = mal_dlsym(pContext->sdl.hSDL, "SDL_QuitSubSystem");
    pContext->sdl.SDL_CloseAudio         = mal_dlsym(pContext->sdl.hSDL, "SDL_CloseAudio");
    pContext->sdl.SDL_OpenAudio          = mal_dlsym(pContext->sdl.hSDL, "SDL_OpenAudio");
    pContext->sdl.SDL_PauseAudio         = mal_dlsym(pContext->sdl.hSDL, "SDL_PauseAudio");
#ifndef MAL_USE_SDL_1
    pContext->sdl.SDL_GetNumAudioDevices = mal_dlsym(pContext->sdl.hSDL, "SDL_GetNumAudioDevices");
    pContext->sdl.SDL_GetAudioDeviceName = mal_dlsym(pContext->sdl.hSDL, "SDL_GetAudioDeviceName");
    pContext->sdl.SDL_CloseAudioDevice   = mal_dlsym(pContext->sdl.hSDL, "SDL_CloseAudioDevice");
    pContext->sdl.SDL_OpenAudioDevice    = mal_dlsym(pContext->sdl.hSDL, "SDL_OpenAudioDevice");
    pContext->sdl.SDL_PauseAudioDevice   = mal_dlsym(pContext->sdl.hSDL, "SDL_PauseAudioDevice");
#endif
#else
    // Compile-time linking.
    pContext->sdl.SDL_InitSubSystem      = (mal_proc)SDL_InitSubSystem;
    pContext->sdl.SDL_QuitSubSystem      = (mal_proc)SDL_QuitSubSystem;
    pContext->sdl.SDL_CloseAudio         = (mal_proc)SDL_CloseAudio;
    pContext->sdl.SDL_OpenAudio          = (mal_proc)SDL_OpenAudio;
    pContext->sdl.SDL_PauseAudio         = (mal_proc)SDL_PauseAudio;
#ifndef MAL_USE_SDL_1
    pContext->sdl.SDL_GetNumAudioDevices = (mal_proc)SDL_GetNumAudioDevices;
    pContext->sdl.SDL_GetAudioDeviceName = (mal_proc)SDL_GetAudioDeviceName;
    pContext->sdl.SDL_CloseAudioDevice   = (mal_proc)SDL_CloseAudioDevice;
    pContext->sdl.SDL_OpenAudioDevice    = (mal_proc)SDL_OpenAudioDevice;
    pContext->sdl.SDL_PauseAudioDevice   = (mal_proc)SDL_PauseAudioDevice;
#endif
#endif

    // We need to determine whether or not we are using SDL2 or SDL1. We can know this by looking at whether or not certain
    // function pointers are NULL.
    if (pContext->sdl.SDL_GetNumAudioDevices == NULL ||
        pContext->sdl.SDL_GetAudioDeviceName == NULL ||
        pContext->sdl.SDL_CloseAudioDevice   == NULL ||
        pContext->sdl.SDL_OpenAudioDevice    == NULL ||
        pContext->sdl.SDL_PauseAudioDevice   == NULL) {
        pContext->sdl.usingSDL1 = MAL_TRUE;
    }

    int resultSDL = ((MAL_PFN_SDL_InitSubSystem)pContext->sdl.SDL_InitSubSystem)(MAL_SDL_INIT_AUDIO);
    if (resultSDL != 0) {
        return MAL_ERROR;
    }

    pContext->isBackendAsynchronous = MAL_TRUE;

    pContext->onUninit        = mal_context_uninit__sdl;
    pContext->onDeviceIDEqual = mal_context_is_device_id_equal__sdl;
    pContext->onEnumDevices   = mal_context_enumerate_devices__sdl;
    pContext->onGetDeviceInfo = mal_context_get_device_info__sdl;
    pContext->onDeviceInit    = mal_device_init__sdl;
    pContext->onDeviceUninit  = mal_device_uninit__sdl;
    pContext->onDeviceStart   = mal_device__start_backend__sdl;
    pContext->onDeviceStop    = mal_device__stop_backend__sdl;

    return MAL_SUCCESS;
}
#endif  // SDL



mal_bool32 mal__is_channel_map_valid(const mal_channel* channelMap, mal_uint32 channels)
{
    // A blank channel map should be allowed, in which case it should use an appropriate default which will depend on context.
    if (channelMap[0] != MAL_CHANNEL_NONE) {
        if (channels == 0) {
            return MAL_FALSE;   // No channels.
        }

        // A channel cannot be present in the channel map more than once.
        for (mal_uint32 iChannel = 0; iChannel < channels; ++iChannel) {
            for (mal_uint32 jChannel = iChannel + 1; jChannel < channels; ++jChannel) {
                if (channelMap[iChannel] == channelMap[jChannel]) {
                    return MAL_FALSE;
                }
            }
        }
    }

    return MAL_TRUE;
}


void mal_device__post_init_setup(mal_device* pDevice)
{
    mal_assert(pDevice != NULL);

    // Make sure the internal channel map was set correctly by the backend. If it's not valid, just fall back to defaults.
    if (!mal_channel_map_valid(pDevice->internalChannels, pDevice->internalChannelMap)) {
        mal_get_standard_channel_map(mal_standard_channel_map_default, pDevice->internalChannels, pDevice->internalChannelMap);
    }


    // If the format/channels/rate is using defaults we need to set these to be the same as the internal config.
    if (pDevice->usingDefaultFormat) {
        pDevice->format = pDevice->internalFormat;
    }
    if (pDevice->usingDefaultChannels) {
        pDevice->channels = pDevice->internalChannels;
    }
    if (pDevice->usingDefaultSampleRate) {
        pDevice->sampleRate = pDevice->internalSampleRate;
    }
    if (pDevice->usingDefaultChannelMap) {
        mal_copy_memory(pDevice->channelMap, pDevice->internalChannelMap, sizeof(pDevice->channelMap));
    }

    // Buffer size. The backend will have set bufferSizeInFrames. We need to calculate bufferSizeInMilliseconds here.
    pDevice->bufferSizeInMilliseconds = pDevice->bufferSizeInFrames / (pDevice->internalSampleRate/1000);


    // We need a DSP object which is where samples are moved through in order to convert them to the
    // format required by the backend.
    mal_dsp_config dspConfig = mal_dsp_config_init_new();
    dspConfig.neverConsumeEndOfInput = MAL_TRUE;
    dspConfig.pUserData = pDevice;
    if (pDevice->type == mal_device_type_playback) {
        dspConfig.formatIn      = pDevice->format;
        dspConfig.channelsIn    = pDevice->channels;
        dspConfig.sampleRateIn  = pDevice->sampleRate;
        mal_copy_memory(dspConfig.channelMapIn, pDevice->channelMap, sizeof(dspConfig.channelMapIn));
        dspConfig.formatOut     = pDevice->internalFormat;
        dspConfig.channelsOut   = pDevice->internalChannels;
        dspConfig.sampleRateOut = pDevice->internalSampleRate;
        mal_copy_memory(dspConfig.channelMapOut, pDevice->internalChannelMap, sizeof(dspConfig.channelMapOut));
        dspConfig.onRead = mal_device__on_read_from_client;
        mal_dsp_init(&dspConfig, &pDevice->dsp);
    } else {
        dspConfig.formatIn      = pDevice->internalFormat;
        dspConfig.channelsIn    = pDevice->internalChannels;
        dspConfig.sampleRateIn  = pDevice->internalSampleRate;
        mal_copy_memory(dspConfig.channelMapIn, pDevice->internalChannelMap, sizeof(dspConfig.channelMapIn));
        dspConfig.formatOut     = pDevice->format;
        dspConfig.channelsOut   = pDevice->channels;
        dspConfig.sampleRateOut = pDevice->sampleRate;
        mal_copy_memory(dspConfig.channelMapOut, pDevice->channelMap, sizeof(dspConfig.channelMapOut));
        dspConfig.onRead = mal_device__on_read_from_device;
        mal_dsp_init(&dspConfig, &pDevice->dsp);
    }
}


mal_thread_result MAL_THREADCALL mal_worker_thread(void* pData)
{
    mal_device* pDevice = (mal_device*)pData;
    mal_assert(pDevice != NULL);

#ifdef MAL_WIN32
    mal_CoInitializeEx(pDevice->pContext, NULL, MAL_COINIT_VALUE);
#endif

    // When the device is being initialized it's initial state is set to MAL_STATE_UNINITIALIZED. Before returning from
    // mal_device_init(), the state needs to be set to something valid. In mini_al the device's default state immediately
    // after initialization is stopped, so therefore we need to mark the device as such. mini_al will wait on the worker
    // thread to signal an event to know when the worker thread is ready for action.
    mal_device__set_state(pDevice, MAL_STATE_STOPPED);
    mal_event_signal(&pDevice->stopEvent);

    for (;;) {
        // We wait on an event to know when something has requested that the device be started and the main loop entered.
        mal_event_wait(&pDevice->wakeupEvent);

        // Default result code.
        pDevice->workResult = MAL_SUCCESS;

        // If the reason for the wake up is that we are terminating, just break from the loop.
        if (mal_device__get_state(pDevice) == MAL_STATE_UNINITIALIZED) {
            break;
        }

        // Getting to this point means the device is wanting to get started. The function that has requested that the device
        // be started will be waiting on an event (pDevice->startEvent) which means we need to make sure we signal the event
        // in both the success and error case. It's important that the state of the device is set _before_ signaling the event.
        mal_assert(mal_device__get_state(pDevice) == MAL_STATE_STARTING);

        pDevice->workResult = pDevice->pContext->onDeviceStart(pDevice);
        if (pDevice->workResult != MAL_SUCCESS) {
            mal_device__set_state(pDevice, MAL_STATE_STOPPED);
            mal_event_signal(&pDevice->startEvent);
            continue;
        }

        // At this point the device should be started.
        mal_device__set_state(pDevice, MAL_STATE_STARTED);
        mal_event_signal(&pDevice->startEvent);


        // Now we just enter the main loop. When the main loop is terminated the device needs to be marked as stopped. This can
        // be broken with mal_device__break_main_loop().
        mal_result mainLoopResult = pDevice->pContext->onDeviceMainLoop(pDevice);
        if (mainLoopResult != MAL_SUCCESS && pDevice->isDefaultDevice && mal_device__get_state(pDevice) == MAL_STATE_STARTED && !pDevice->exclusiveMode) {
            // Something has failed during the main loop. It could be that the device has been lost. If it's the default device,
            // we can try switching over to the new default device by uninitializing and reinitializing.
            mal_result reinitResult = MAL_ERROR;
            if (pDevice->pContext->onDeviceReinit) {
                reinitResult = pDevice->pContext->onDeviceReinit(pDevice);
            } else {
                pDevice->pContext->onDeviceStop(pDevice);
                mal_device__set_state(pDevice, MAL_STATE_STOPPED);

                pDevice->pContext->onDeviceUninit(pDevice);
                mal_device__set_state(pDevice, MAL_STATE_UNINITIALIZED);

                reinitResult = pDevice->pContext->onDeviceInit(pDevice->pContext, pDevice->type, NULL, &pDevice->initConfig, pDevice);
            }

            // Perform the post initialization setup just in case the data conversion pipeline needs to be reinitialized.
            if (reinitResult == MAL_SUCCESS) {
                mal_device__post_init_setup(pDevice);
            }

            // If reinitialization was successful, loop back to the start.
            if (reinitResult == MAL_SUCCESS) {
                mal_device__set_state(pDevice, MAL_STATE_STARTING); // <-- The device is restarting.
                mal_event_signal(&pDevice->wakeupEvent);
                continue;
            }
        }


        // Getting here means we have broken from the main loop which happens the application has requested that device be stopped. Note that this
        // may have actually already happened above if the device was lost and mini_al has attempted to re-initialize the device. In this case we
        // don't want to be doing this a second time.
        if (mal_device__get_state(pDevice) != MAL_STATE_UNINITIALIZED) {
            pDevice->pContext->onDeviceStop(pDevice);
        }

        // After the device has stopped, make sure an event is posted.
        mal_stop_proc onStop = pDevice->onStop;
        if (onStop) {
            onStop(pDevice);
        }

        // A function somewhere is waiting for the device to have stopped for real so we need to signal an event to allow it to continue. Note that
        // it's possible that the device has been uninitialized which means we need to _not_ change the status to stopped. We cannot go from an
        // uninitialized state to stopped state.
        if (mal_device__get_state(pDevice) != MAL_STATE_UNINITIALIZED) {
            mal_device__set_state(pDevice, MAL_STATE_STOPPED);
            mal_event_signal(&pDevice->stopEvent);
        }
    }

    // Make sure we aren't continuously waiting on a stop event.
    mal_event_signal(&pDevice->stopEvent);  // <-- Is this still needed?

#ifdef MAL_WIN32
    mal_CoUninitialize(pDevice->pContext);
#endif

    return (mal_thread_result)0;
}


// Helper for determining whether or not the given device is initialized.
mal_bool32 mal_device__is_initialized(mal_device* pDevice)
{
    if (pDevice == NULL) return MAL_FALSE;
    return mal_device__get_state(pDevice) != MAL_STATE_UNINITIALIZED;
}


#ifdef MAL_WIN32
mal_result mal_context_uninit_backend_apis__win32(mal_context* pContext)
{
    mal_CoUninitialize(pContext);
    mal_dlclose(pContext->win32.hUser32DLL);
    mal_dlclose(pContext->win32.hOle32DLL);
    mal_dlclose(pContext->win32.hAdvapi32DLL);

    return MAL_SUCCESS;
}

mal_result mal_context_init_backend_apis__win32(mal_context* pContext)
{
#ifdef MAL_WIN32_DESKTOP
    // Ole32.dll
    pContext->win32.hOle32DLL = mal_dlopen("ole32.dll");
    if (pContext->win32.hOle32DLL == NULL) {
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    pContext->win32.CoInitializeEx   = (mal_proc)mal_dlsym(pContext->win32.hOle32DLL, "CoInitializeEx");
    pContext->win32.CoUninitialize   = (mal_proc)mal_dlsym(pContext->win32.hOle32DLL, "CoUninitialize");
    pContext->win32.CoCreateInstance = (mal_proc)mal_dlsym(pContext->win32.hOle32DLL, "CoCreateInstance");
    pContext->win32.CoTaskMemFree    = (mal_proc)mal_dlsym(pContext->win32.hOle32DLL, "CoTaskMemFree");
    pContext->win32.PropVariantClear = (mal_proc)mal_dlsym(pContext->win32.hOle32DLL, "PropVariantClear");
    pContext->win32.StringFromGUID2  = (mal_proc)mal_dlsym(pContext->win32.hOle32DLL, "StringFromGUID2");


    // User32.dll
    pContext->win32.hUser32DLL = mal_dlopen("user32.dll");
    if (pContext->win32.hUser32DLL == NULL) {
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    pContext->win32.GetForegroundWindow = (mal_proc)mal_dlsym(pContext->win32.hUser32DLL, "GetForegroundWindow");
    pContext->win32.GetDesktopWindow    = (mal_proc)mal_dlsym(pContext->win32.hUser32DLL, "GetDesktopWindow");


    // Advapi32.dll
    pContext->win32.hAdvapi32DLL = mal_dlopen("advapi32.dll");
    if (pContext->win32.hAdvapi32DLL == NULL) {
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    pContext->win32.RegOpenKeyExA    = (mal_proc)mal_dlsym(pContext->win32.hAdvapi32DLL, "RegOpenKeyExA");
    pContext->win32.RegCloseKey      = (mal_proc)mal_dlsym(pContext->win32.hAdvapi32DLL, "RegCloseKey");
    pContext->win32.RegQueryValueExA = (mal_proc)mal_dlsym(pContext->win32.hAdvapi32DLL, "RegQueryValueExA");
#endif

    mal_CoInitializeEx(pContext, NULL, MAL_COINIT_VALUE);
    return MAL_SUCCESS;
}
#else
mal_result mal_context_uninit_backend_apis__nix(mal_context* pContext)
{
#if defined(MAL_USE_RUNTIME_LINKING_FOR_PTHREAD) && !defined(MAL_NO_RUNTIME_LINKING)
    mal_dlclose(pContext->posix.pthreadSO);
#else
    (void)pContext;
#endif

    return MAL_SUCCESS;
}

mal_result mal_context_init_backend_apis__nix(mal_context* pContext)
{
    // pthread
#if defined(MAL_USE_RUNTIME_LINKING_FOR_PTHREAD) && !defined(MAL_NO_RUNTIME_LINKING)
    const char* libpthreadFileNames[] = {
        "libpthread.so",
        "libpthread.so.0",
        "libpthread.dylib"
    };

    for (size_t i = 0; i < sizeof(libpthreadFileNames) / sizeof(libpthreadFileNames[0]); ++i) {
        pContext->posix.pthreadSO = mal_dlopen(libpthreadFileNames[i]);
        if (pContext->posix.pthreadSO != NULL) {
            break;
        }
    }

    if (pContext->posix.pthreadSO == NULL) {
        return MAL_FAILED_TO_INIT_BACKEND;
    }

    pContext->posix.pthread_create              = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_create");
    pContext->posix.pthread_join                = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_join");
    pContext->posix.pthread_mutex_init          = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_mutex_init");
    pContext->posix.pthread_mutex_destroy       = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_mutex_destroy");
    pContext->posix.pthread_mutex_lock          = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_mutex_lock");
    pContext->posix.pthread_mutex_unlock        = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_mutex_unlock");
    pContext->posix.pthread_cond_init           = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_cond_init");
    pContext->posix.pthread_cond_destroy        = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_cond_destroy");
    pContext->posix.pthread_cond_wait           = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_cond_wait");
    pContext->posix.pthread_cond_signal         = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_cond_signal");
    pContext->posix.pthread_attr_init           = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_attr_init");
    pContext->posix.pthread_attr_destroy        = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_attr_destroy");
    pContext->posix.pthread_attr_setschedpolicy = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_attr_setschedpolicy");
    pContext->posix.pthread_attr_getschedparam  = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_attr_getschedparam");
    pContext->posix.pthread_attr_setschedparam  = (mal_proc)mal_dlsym(pContext->posix.pthreadSO, "pthread_attr_setschedparam");
#else
    pContext->posix.pthread_create              = (mal_proc)pthread_create;
    pContext->posix.pthread_join                = (mal_proc)pthread_join;
    pContext->posix.pthread_mutex_init          = (mal_proc)pthread_mutex_init;
    pContext->posix.pthread_mutex_destroy       = (mal_proc)pthread_mutex_destroy;
    pContext->posix.pthread_mutex_lock          = (mal_proc)pthread_mutex_lock;
    pContext->posix.pthread_mutex_unlock        = (mal_proc)pthread_mutex_unlock;
    pContext->posix.pthread_cond_init           = (mal_proc)pthread_cond_init;
    pContext->posix.pthread_cond_destroy        = (mal_proc)pthread_cond_destroy;
    pContext->posix.pthread_cond_wait           = (mal_proc)pthread_cond_wait;
    pContext->posix.pthread_cond_signal         = (mal_proc)pthread_cond_signal;
    pContext->posix.pthread_attr_init           = (mal_proc)pthread_attr_init;
    pContext->posix.pthread_attr_destroy        = (mal_proc)pthread_attr_destroy;
#if !defined(__EMSCRIPTEN__)
    pContext->posix.pthread_attr_setschedpolicy = (mal_proc)pthread_attr_setschedpolicy;
    pContext->posix.pthread_attr_getschedparam  = (mal_proc)pthread_attr_getschedparam;
    pContext->posix.pthread_attr_setschedparam  = (mal_proc)pthread_attr_setschedparam;
#endif
#endif

    return MAL_SUCCESS;
}
#endif

mal_result mal_context_init_backend_apis(mal_context* pContext)
{
    mal_result result;
#ifdef MAL_WIN32
    result = mal_context_init_backend_apis__win32(pContext);
#else
    result = mal_context_init_backend_apis__nix(pContext);
#endif

    return result;
}

mal_result mal_context_uninit_backend_apis(mal_context* pContext)
{
    mal_result result;
#ifdef MAL_WIN32
    result = mal_context_uninit_backend_apis__win32(pContext);
#else
    result = mal_context_uninit_backend_apis__nix(pContext);
#endif

    return result;
}


mal_bool32 mal_context_is_backend_asynchronous(mal_context* pContext)
{
    return pContext->isBackendAsynchronous;
}

mal_result mal_context_init(const mal_backend backends[], mal_uint32 backendCount, const mal_context_config* pConfig, mal_context* pContext)
{
    if (pContext == NULL) {
        return MAL_INVALID_ARGS;
    }

    mal_zero_object(pContext);

    // Always make sure the config is set first to ensure properties are available as soon as possible.
    if (pConfig != NULL) {
        pContext->config = *pConfig;
    } else {
        pContext->config = mal_context_config_init(NULL);
    }

    // Backend APIs need to be initialized first. This is where external libraries will be loaded and linked.
    mal_result result = mal_context_init_backend_apis(pContext);
    if (result != MAL_SUCCESS) {
        return result;
    }

    mal_backend* pBackendsToIterate = (mal_backend*)backends;
    mal_uint32 backendsToIterateCount = backendCount;
    if (pBackendsToIterate == NULL) {
        pBackendsToIterate = (mal_backend*)g_malDefaultBackends;
        backendsToIterateCount = mal_countof(g_malDefaultBackends);
    }

    mal_assert(pBackendsToIterate != NULL);

    for (mal_uint32 iBackend = 0; iBackend < backendsToIterateCount; ++iBackend) {
        mal_backend backend = pBackendsToIterate[iBackend];

        result = MAL_NO_BACKEND;
        switch (backend) {
        #ifdef MAL_HAS_WASAPI
            case mal_backend_wasapi:
            {
                result = mal_context_init__wasapi(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_DSOUND
            case mal_backend_dsound:
            {
                result = mal_context_init__dsound(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_WINMM
            case mal_backend_winmm:
            {
                result = mal_context_init__winmm(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_ALSA
            case mal_backend_alsa:
            {
                result = mal_context_init__alsa(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_PULSEAUDIO
            case mal_backend_pulseaudio:
            {
                result = mal_context_init__pulse(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_JACK
            case mal_backend_jack:
            {
                result = mal_context_init__jack(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_COREAUDIO
            case mal_backend_coreaudio:
            {
                result = mal_context_init__coreaudio(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_SNDIO
            case mal_backend_sndio:
            {
                result = mal_context_init__sndio(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_AUDIO4
            case mal_backend_audio4:
            {
                result = mal_context_init__audio4(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_OSS
            case mal_backend_oss:
            {
                result = mal_context_init__oss(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_OPENSL
            case mal_backend_opensl:
            {
                result = mal_context_init__opensl(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_OPENAL
            case mal_backend_openal:
            {
                result = mal_context_init__openal(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_SDL
            case mal_backend_sdl:
            {
                result = mal_context_init__sdl(pContext);
            } break;
        #endif
        #ifdef MAL_HAS_NULL
            case mal_backend_null:
            {
                result = mal_context_init__null(pContext);
            } break;
        #endif

            default: break;
        }

        // If this iteration was successful, return.
        if (result == MAL_SUCCESS) {
            result = mal_mutex_init(pContext, &pContext->deviceEnumLock);
            if (result != MAL_SUCCESS) {
                mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_WARNING, "Failed to initialize mutex for device enumeration. mal_context_get_devices() is not thread safe.", MAL_FAILED_TO_CREATE_MUTEX);
            }
            result = mal_mutex_init(pContext, &pContext->deviceInfoLock);
            if (result != MAL_SUCCESS) {
                mal_context_post_error(pContext, NULL, MAL_LOG_LEVEL_WARNING, "Failed to initialize mutex for device info retrieval. mal_context_get_device_info() is not thread safe.", MAL_FAILED_TO_CREATE_MUTEX);
            }

            pContext->backend = backend;
            return result;
        }
    }

    // If we get here it means an error occurred.
    mal_zero_object(pContext);  // Safety.
    return MAL_NO_BACKEND;
}

mal_result mal_context_uninit(mal_context* pContext)
{
    if (pContext == NULL) {
        return MAL_INVALID_ARGS;
    }

    pContext->onUninit(pContext);

    mal_context_uninit_backend_apis(pContext);
    mal_mutex_uninit(&pContext->deviceEnumLock);
    mal_mutex_uninit(&pContext->deviceInfoLock);
    mal_free(pContext->pDeviceInfos);

    return MAL_SUCCESS;
}


mal_result mal_context_enumerate_devices(mal_context* pContext, mal_enum_devices_callback_proc callback, void* pUserData)
{
    if (pContext == NULL || pContext->onEnumDevices == NULL || callback == NULL) {
        return MAL_INVALID_ARGS;
    }

    mal_result result;
    mal_mutex_lock(&pContext->deviceEnumLock);
    {
        result = pContext->onEnumDevices(pContext, callback, pUserData);
    }
    mal_mutex_unlock(&pContext->deviceEnumLock);

    return result;
}


mal_bool32 mal_context_get_devices__enum_callback(mal_context* pContext, mal_device_type type, const mal_device_info* pInfo, void* pUserData)
{
    (void)pUserData;

    // We need to insert the device info into our main internal buffer. Where it goes depends on the device type. If it's a capture device
    // it's just appended to the end. If it's a playback device it's inserted just before the first capture device.

    // First make sure we have room. Since the number of devices we add to the list is usually relatively small I've decided to use a
    // simple fixed size increment for buffer expansion.
    const mal_uint32 bufferExpansionCount = 2;
    const mal_uint32 totalDeviceInfoCount = pContext->playbackDeviceInfoCount + pContext->captureDeviceInfoCount;

    if (pContext->deviceInfoCapacity >= totalDeviceInfoCount) {
        mal_uint32 newCapacity = totalDeviceInfoCount + bufferExpansionCount;
        mal_device_info* pNewInfos = (mal_device_info*)mal_realloc(pContext->pDeviceInfos, sizeof(*pContext->pDeviceInfos)*newCapacity);
        if (pNewInfos == NULL) {
            return MAL_FALSE;   // Out of memory.
        }

        pContext->pDeviceInfos = pNewInfos;
        pContext->deviceInfoCapacity = newCapacity;
    }

    if (type == mal_device_type_playback) {
        // Playback. Insert just before the first capture device.

        // The first thing to do is move all of the capture devices down a slot.
        mal_uint32 iFirstCaptureDevice = pContext->playbackDeviceInfoCount;
        for (size_t iCaptureDevice = totalDeviceInfoCount; iCaptureDevice > iFirstCaptureDevice; --iCaptureDevice) {
            pContext->pDeviceInfos[iCaptureDevice] = pContext->pDeviceInfos[iCaptureDevice-1];
        }

        // Now just insert where the first capture device was before moving it down a slot.
        pContext->pDeviceInfos[iFirstCaptureDevice] = *pInfo;
        pContext->playbackDeviceInfoCount += 1;
    } else {
        // Capture. Insert at the end.
        pContext->pDeviceInfos[totalDeviceInfoCount] = *pInfo;
        pContext->captureDeviceInfoCount += 1;
    }

    return MAL_TRUE;
}

mal_result mal_context_get_devices(mal_context* pContext, mal_device_info** ppPlaybackDeviceInfos, mal_uint32* pPlaybackDeviceCount, mal_device_info** ppCaptureDeviceInfos, mal_uint32* pCaptureDeviceCount)
{
    // Safety.
    if (ppPlaybackDeviceInfos != NULL) *ppPlaybackDeviceInfos = NULL;
    if (pPlaybackDeviceCount  != NULL) *pPlaybackDeviceCount  = 0;
    if (ppCaptureDeviceInfos  != NULL) *ppCaptureDeviceInfos  = NULL;
    if (pCaptureDeviceCount   != NULL) *pCaptureDeviceCount   = 0;

    if (pContext == NULL || pContext->onEnumDevices == NULL) {
        return MAL_INVALID_ARGS;
    }

    // Note that we don't use mal_context_enumerate_devices() here because we want to do locking at a higher level.
    mal_result result;
    mal_mutex_lock(&pContext->deviceEnumLock);
    {
        // Reset everything first.
        pContext->playbackDeviceInfoCount = 0;
        pContext->captureDeviceInfoCount = 0;

        // Now enumerate over available devices.
        result = pContext->onEnumDevices(pContext, mal_context_get_devices__enum_callback, NULL);
        if (result == MAL_SUCCESS) {
            // Playback devices.
            if (ppPlaybackDeviceInfos != NULL) {
                *ppPlaybackDeviceInfos = pContext->pDeviceInfos;
            }
            if (pPlaybackDeviceCount != NULL) {
                *pPlaybackDeviceCount = pContext->playbackDeviceInfoCount;
            }

            // Capture devices.
            if (ppCaptureDeviceInfos != NULL) {
                *ppCaptureDeviceInfos = pContext->pDeviceInfos + pContext->playbackDeviceInfoCount; // Capture devices come after playback devices.
            }
            if (pCaptureDeviceCount != NULL) {
                *pCaptureDeviceCount = pContext->captureDeviceInfoCount;
            }
        }
    }
    mal_mutex_unlock(&pContext->deviceEnumLock);

    return result;
}

mal_result mal_context_get_device_info(mal_context* pContext, mal_device_type type, const mal_device_id* pDeviceID, mal_share_mode shareMode, mal_device_info* pDeviceInfo)
{
    // NOTE: Do not clear pDeviceInfo on entry. The reason is the pDeviceID may actually point to pDeviceInfo->id which will break things.
    if (pContext == NULL || pDeviceInfo == NULL) {
        return MAL_INVALID_ARGS;
    }

    mal_device_info deviceInfo;
    mal_zero_object(&deviceInfo);

    // Help the backend out by copying over the device ID if we have one.
    if (pDeviceID != NULL) {
        mal_copy_memory(&deviceInfo.id, pDeviceID, sizeof(*pDeviceID));
    }

    // The backend may have an optimized device info retrieval function. If so, try that first.
    if (pContext->onGetDeviceInfo != NULL) {
        mal_result result;
        mal_mutex_lock(&pContext->deviceInfoLock);
        {
            result = pContext->onGetDeviceInfo(pContext, type, pDeviceID, shareMode, &deviceInfo);
        }
        mal_mutex_unlock(&pContext->deviceInfoLock);

        // Clamp ranges.
        deviceInfo.minChannels   = mal_max(deviceInfo.minChannels,   MAL_MIN_CHANNELS);
        deviceInfo.maxChannels   = mal_min(deviceInfo.maxChannels,   MAL_MAX_CHANNELS);
        deviceInfo.minSampleRate = mal_max(deviceInfo.minSampleRate, MAL_MIN_SAMPLE_RATE);
        deviceInfo.maxSampleRate = mal_min(deviceInfo.maxSampleRate, MAL_MAX_SAMPLE_RATE);

        *pDeviceInfo = deviceInfo;
        return result;
    }

    // Getting here means onGetDeviceInfo has not been set.
    return MAL_ERROR;
}


mal_result mal_device_init(mal_context* pContext, mal_device_type type, mal_device_id* pDeviceID, const mal_device_config* pConfig, void* pUserData, mal_device* pDevice)
{
    if (pContext == NULL) {
        return mal_device_init_ex(NULL, 0, NULL, type, pDeviceID, pConfig, pUserData, pDevice);
    }


    if (pDevice == NULL) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_init() called with invalid arguments (pDevice == NULL).",  MAL_INVALID_ARGS);
    }

    // The config is allowed to be NULL, in which case we default to mal_device_config_init_default().
    mal_device_config config;
    if (pConfig == NULL) {
        config = mal_device_config_init_default();
    } else {
        config = *pConfig;
    }

    // Basic config validation.
    if (config.channels > MAL_MAX_CHANNELS) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_init() called with an invalid config. Channel count cannot exceed 32.", MAL_INVALID_DEVICE_CONFIG);
    }
    if (!mal__is_channel_map_valid(config.channelMap, config.channels)) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_init() called with invalid config. Channel map is invalid.", MAL_INVALID_DEVICE_CONFIG);
    }


    mal_zero_object(pDevice);
    pDevice->pContext = pContext;
    pDevice->initConfig = config;

    // Set the user data and log callback ASAP to ensure it is available for the entire initialization process.
    pDevice->pUserData = pUserData;
    pDevice->onStop = config.onStopCallback;
    pDevice->onSend = config.onSendCallback;
    pDevice->onRecv = config.onRecvCallback;

    if (((size_t)pDevice % sizeof(pDevice)) != 0) {
        if (pContext->config.onLog) {
            pContext->config.onLog(pContext, pDevice, "WARNING: mal_device_init() called for a device that is not properly aligned. Thread safety is not supported.");
        }
    }

    if (pDeviceID == NULL) {
        pDevice->isDefaultDevice = MAL_TRUE;
    }


    // When passing in 0 for the format/channels/rate/chmap it means the device will be using whatever is chosen by the backend. If everything is set
    // to defaults it means the format conversion pipeline will run on a fast path where data transfer is just passed straight through to the backend.
    if (config.format == mal_format_unknown) {
        config.format = MAL_DEFAULT_FORMAT;
        pDevice->usingDefaultFormat = MAL_TRUE;
    }
    if (config.channels == 0) {
        config.channels = MAL_DEFAULT_CHANNELS;
        pDevice->usingDefaultChannels = MAL_TRUE;
    }
    if (config.sampleRate == 0) {
        config.sampleRate = MAL_DEFAULT_SAMPLE_RATE;
        pDevice->usingDefaultSampleRate = MAL_TRUE;
    }
    if (config.channelMap[0] == MAL_CHANNEL_NONE) {
        pDevice->usingDefaultChannelMap = MAL_TRUE;
    }


    // Default buffer size.
    if (config.bufferSizeInMilliseconds == 0 && config.bufferSizeInFrames == 0) {
        config.bufferSizeInMilliseconds = (config.performanceProfile == mal_performance_profile_low_latency) ? MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY : MAL_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE;
        pDevice->usingDefaultBufferSize = MAL_TRUE;
    }

    // Default periods.
    if (config.periods == 0) {
        config.periods = MAL_DEFAULT_PERIODS;
        pDevice->usingDefaultPeriods = MAL_TRUE;
    }

    pDevice->type = type;
    pDevice->format = config.format;
    pDevice->channels = config.channels;
    pDevice->sampleRate = config.sampleRate;
    mal_copy_memory(pDevice->channelMap, config.channelMap, sizeof(config.channelMap[0]) * config.channels);
    pDevice->bufferSizeInFrames = config.bufferSizeInFrames;
    pDevice->bufferSizeInMilliseconds = config.bufferSizeInMilliseconds;
    pDevice->periods = config.periods;

    // The internal format, channel count and sample rate can be modified by the backend.
    pDevice->internalFormat = pDevice->format;
    pDevice->internalChannels = pDevice->channels;
    pDevice->internalSampleRate = pDevice->sampleRate;
    mal_copy_memory(pDevice->internalChannelMap, pDevice->channelMap, sizeof(pDevice->channelMap));

    if (mal_mutex_init(pContext, &pDevice->lock) != MAL_SUCCESS) {
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "Failed to create mutex.", MAL_FAILED_TO_CREATE_MUTEX);
    }

    // When the device is started, the worker thread is the one that does the actual startup of the backend device. We
    // use a semaphore to wait for the background thread to finish the work. The same applies for stopping the device.
    //
    // Each of these semaphores is released internally by the worker thread when the work is completed. The start
    // semaphore is also used to wake up the worker thread.
    if (mal_event_init(pContext, &pDevice->wakeupEvent) != MAL_SUCCESS) {
        mal_mutex_uninit(&pDevice->lock);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "Failed to create worker thread wakeup event.", MAL_FAILED_TO_CREATE_EVENT);
    }
    if (mal_event_init(pContext, &pDevice->startEvent) != MAL_SUCCESS) {
        mal_event_uninit(&pDevice->wakeupEvent);
        mal_mutex_uninit(&pDevice->lock);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "Failed to create worker thread start event.", MAL_FAILED_TO_CREATE_EVENT);
    }
    if (mal_event_init(pContext, &pDevice->stopEvent) != MAL_SUCCESS) {
        mal_event_uninit(&pDevice->startEvent);
        mal_event_uninit(&pDevice->wakeupEvent);
        mal_mutex_uninit(&pDevice->lock);
        return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "Failed to create worker thread stop event.", MAL_FAILED_TO_CREATE_EVENT);
    }


    mal_result result = pContext->onDeviceInit(pContext, type, pDeviceID, &config, pDevice);
    if (result != MAL_SUCCESS) {
        return MAL_NO_BACKEND;  // The error message will have been posted with mal_post_error() by the source of the error so don't bother calling it here.
    }

    mal_device__post_init_setup(pDevice);


    // If the backend did not fill out a name for the device, try a generic method.
    if (pDevice->name[0] == '\0') {
        if (mal_context__try_get_device_name_by_id(pContext, type, pDeviceID, pDevice->name, sizeof(pDevice->name)) != MAL_SUCCESS) {
            // We failed to get the device name, so fall back to some generic names.
            if (pDeviceID == NULL) {
                if (type == mal_device_type_playback) {
                    mal_strncpy_s(pDevice->name, sizeof(pDevice->name), MAL_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
                } else {
                    mal_strncpy_s(pDevice->name, sizeof(pDevice->name), MAL_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
                }
            } else {
                if (type == mal_device_type_playback) {
                    mal_strncpy_s(pDevice->name, sizeof(pDevice->name), "Playback Device", (size_t)-1);
                } else {
                    mal_strncpy_s(pDevice->name, sizeof(pDevice->name), "Capture Device", (size_t)-1);
                }
            }
        }
    }


    // Some backends don't require the worker thread.
    if (!mal_context_is_backend_asynchronous(pContext)) {
        // The worker thread.
        if (mal_thread_create(pContext, &pDevice->thread, mal_worker_thread, pDevice) != MAL_SUCCESS) {
            mal_device_uninit(pDevice);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "Failed to create worker thread.", MAL_FAILED_TO_CREATE_THREAD);
        }

        // Wait for the worker thread to put the device into it's stopped state for real.
        mal_event_wait(&pDevice->stopEvent);
    } else {
        mal_device__set_state(pDevice, MAL_STATE_STOPPED);
    }


#ifdef MAL_DEBUG_OUTPUT
    printf("[WASAPI] %s (%s)\n", pDevice->name, (pDevice->type == mal_device_type_playback) ? "Playback" : "Capture");
    printf("  Format:      %s -> %s\n", mal_get_format_name(pDevice->format), mal_get_format_name(pDevice->internalFormat));
    printf("  Channels:    %d -> %d\n", pDevice->channels, pDevice->internalChannels);
    printf("  Sample Rate: %d -> %d\n", pDevice->sampleRate, pDevice->internalSampleRate);
#endif


    mal_assert(mal_device__get_state(pDevice) == MAL_STATE_STOPPED);
    return MAL_SUCCESS;
}

mal_result mal_device_init_ex(const mal_backend backends[], mal_uint32 backendCount, const mal_context_config* pContextConfig, mal_device_type type, mal_device_id* pDeviceID, const mal_device_config* pConfig, void* pUserData, mal_device* pDevice)
{
    mal_context* pContext = (mal_context*)mal_malloc(sizeof(*pContext));
    if (pContext == NULL) {
        return MAL_OUT_OF_MEMORY;
    }

    mal_backend* pBackendsToIterate = (mal_backend*)backends;
    mal_uint32 backendsToIterateCount = backendCount;
    if (pBackendsToIterate == NULL) {
        pBackendsToIterate = (mal_backend*)g_malDefaultBackends;
        backendsToIterateCount = mal_countof(g_malDefaultBackends);
    }

    mal_result result = MAL_NO_BACKEND;

    for (mal_uint32 iBackend = 0; iBackend < backendsToIterateCount; ++iBackend) {
        result = mal_context_init(&pBackendsToIterate[iBackend], 1, pContextConfig, pContext);
        if (result == MAL_SUCCESS) {
            result = mal_device_init(pContext, type, pDeviceID, pConfig, pUserData, pDevice);
            if (result == MAL_SUCCESS) {
                break;  // Success.
            } else {
                mal_context_uninit(pContext);   // Failure.
            }
        }
    }

    if (result != MAL_SUCCESS) {
        mal_free(pContext);
        return result;
    }

    pDevice->isOwnerOfContext = MAL_TRUE;
    return result;
}

void mal_device_uninit(mal_device* pDevice)
{
    if (!mal_device__is_initialized(pDevice)) return;

    // Make sure the device is stopped first. The backends will probably handle this naturally,
    // but I like to do it explicitly for my own sanity.
    if (mal_device_is_started(pDevice)) {
        while (mal_device_stop(pDevice) == MAL_DEVICE_BUSY) {
            mal_sleep(1);
        }
    }

    // Putting the device into an uninitialized state will make the worker thread return.
    mal_device__set_state(pDevice, MAL_STATE_UNINITIALIZED);

    // Wake up the worker thread and wait for it to properly terminate.
    if (!mal_context_is_backend_asynchronous(pDevice->pContext)) {
        mal_event_signal(&pDevice->wakeupEvent);
        mal_thread_wait(&pDevice->thread);
    }

    pDevice->pContext->onDeviceUninit(pDevice);

    mal_event_uninit(&pDevice->stopEvent);
    mal_event_uninit(&pDevice->startEvent);
    mal_event_uninit(&pDevice->wakeupEvent);
    mal_mutex_uninit(&pDevice->lock);

    if (pDevice->isOwnerOfContext) {
        mal_context_uninit(pDevice->pContext);
        mal_free(pDevice->pContext);
    }

    mal_zero_object(pDevice);
}

void mal_device_set_recv_callback(mal_device* pDevice, mal_recv_proc proc)
{
    if (pDevice == NULL) return;
    mal_atomic_exchange_ptr(&pDevice->onRecv, proc);
}

void mal_device_set_send_callback(mal_device* pDevice, mal_send_proc proc)
{
    if (pDevice == NULL) return;
    mal_atomic_exchange_ptr(&pDevice->onSend, proc);
}

void mal_device_set_stop_callback(mal_device* pDevice, mal_stop_proc proc)
{
    if (pDevice == NULL) return;
    mal_atomic_exchange_ptr(&pDevice->onStop, proc);
}

mal_result mal_device_start(mal_device* pDevice)
{
    if (pDevice == NULL) return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_start() called with invalid arguments (pDevice == NULL).", MAL_INVALID_ARGS);
    if (mal_device__get_state(pDevice) == MAL_STATE_UNINITIALIZED) return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_start() called for an uninitialized device.", MAL_DEVICE_NOT_INITIALIZED);

    mal_result result = MAL_ERROR;
    mal_mutex_lock(&pDevice->lock);
    {
        // Be a bit more descriptive if the device is already started or is already in the process of starting. This is likely
        // a bug with the application.
        if (mal_device__get_state(pDevice) == MAL_STATE_STARTING) {
            mal_mutex_unlock(&pDevice->lock);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_start() called while another thread is already starting it.", MAL_DEVICE_ALREADY_STARTING);
        }
        if (mal_device__get_state(pDevice) == MAL_STATE_STARTED) {
            mal_mutex_unlock(&pDevice->lock);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_start() called for a device that's already started.", MAL_DEVICE_ALREADY_STARTED);
        }

        // The device needs to be in a stopped state. If it's not, we just let the caller know the device is busy.
        if (mal_device__get_state(pDevice) != MAL_STATE_STOPPED) {
            mal_mutex_unlock(&pDevice->lock);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_start() called while another thread is in the process of stopping it.", MAL_DEVICE_BUSY);
        }

        mal_device__set_state(pDevice, MAL_STATE_STARTING);

        // Asynchronous backends need to be handled differently.
        if (mal_context_is_backend_asynchronous(pDevice->pContext)) {
            result = pDevice->pContext->onDeviceStart(pDevice);
            if (result == MAL_SUCCESS) {
                mal_device__set_state(pDevice, MAL_STATE_STARTED);
            }
        } else {
            // Synchronous backends are started by signaling an event that's being waited on in the worker thread. We first wake up the
            // thread and then wait for the start event.
            mal_event_signal(&pDevice->wakeupEvent);

            // Wait for the worker thread to finish starting the device. Note that the worker thread will be the one
            // who puts the device into the started state. Don't call mal_device__set_state() here.
            mal_event_wait(&pDevice->startEvent);
            result = pDevice->workResult;
        }
    }
    mal_mutex_unlock(&pDevice->lock);

    return result;
}

mal_result mal_device_stop(mal_device* pDevice)
{
    if (pDevice == NULL) return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_stop() called with invalid arguments (pDevice == NULL).", MAL_INVALID_ARGS);
    if (mal_device__get_state(pDevice) == MAL_STATE_UNINITIALIZED) return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_stop() called for an uninitialized device.", MAL_DEVICE_NOT_INITIALIZED);

    mal_result result = MAL_ERROR;
    mal_mutex_lock(&pDevice->lock);
    {
        // Be a bit more descriptive if the device is already stopped or is already in the process of stopping. This is likely
        // a bug with the application.
        if (mal_device__get_state(pDevice) == MAL_STATE_STOPPING) {
            mal_mutex_unlock(&pDevice->lock);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_stop() called while another thread is already stopping it.", MAL_DEVICE_ALREADY_STOPPING);
        }
        if (mal_device__get_state(pDevice) == MAL_STATE_STOPPED) {
            mal_mutex_unlock(&pDevice->lock);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_stop() called for a device that's already stopped.", MAL_DEVICE_ALREADY_STOPPED);
        }

        // The device needs to be in a started state. If it's not, we just let the caller know the device is busy.
        if (mal_device__get_state(pDevice) != MAL_STATE_STARTED) {
            mal_mutex_unlock(&pDevice->lock);
            return mal_post_error(pDevice, MAL_LOG_LEVEL_ERROR, "mal_device_stop() called while another thread is in the process of starting it.", MAL_DEVICE_BUSY);
        }

        mal_device__set_state(pDevice, MAL_STATE_STOPPING);

        // There's no need to wake up the thread like we do when starting.

        // Asynchronous backends need to be handled differently.
        if (mal_context_is_backend_asynchronous(pDevice->pContext)) {
            result = pDevice->pContext->onDeviceStop(pDevice);
        } else {
            // Synchronous backends.

            // When we get here the worker thread is likely in a wait state while waiting for the backend device to deliver or request
            // audio data. We need to force these to return as quickly as possible.
            pDevice->pContext->onDeviceBreakMainLoop(pDevice);

            // We need to wait for the worker thread to become available for work before returning. Note that the worker thread will be
            // the one who puts the device into the stopped state. Don't call mal_device__set_state() here.
            mal_event_wait(&pDevice->stopEvent);
            result = MAL_SUCCESS;
        }
    }
    mal_mutex_unlock(&pDevice->lock);

    return result;
}

mal_bool32 mal_device_is_started(mal_device* pDevice)
{
    if (pDevice == NULL) return MAL_FALSE;
    return mal_device__get_state(pDevice) == MAL_STATE_STARTED;
}

mal_uint32 mal_device_get_buffer_size_in_bytes(mal_device* pDevice)
{
    if (pDevice == NULL) return 0;
    return pDevice->bufferSizeInFrames * pDevice->channels * mal_get_bytes_per_sample(pDevice->format);
}

mal_context_config mal_context_config_init(mal_log_proc onLog)
{
    mal_context_config config;
    mal_zero_object(&config);

    config.onLog = onLog;

    return config;
}


mal_device_config mal_device_config_init_default()
{
    mal_device_config config;
    mal_zero_object(&config);

    return config;
}

mal_device_config mal_device_config_init_default_capture(mal_recv_proc onRecvCallback)
{
    mal_device_config config = mal_device_config_init_default();
    config.onRecvCallback = onRecvCallback;

    return config;
}

mal_device_config mal_device_config_init_default_playback(mal_send_proc onSendCallback)
{
    mal_device_config config = mal_device_config_init_default();
    config.onSendCallback = onSendCallback;

    return config;
}


mal_device_config mal_device_config_init_ex(mal_format format, mal_uint32 channels, mal_uint32 sampleRate, mal_channel channelMap[MAL_MAX_CHANNELS], mal_recv_proc onRecvCallback, mal_send_proc onSendCallback)
{
    mal_device_config config = mal_device_config_init_default();

    config.format = format;
    config.channels = channels;
    config.sampleRate = sampleRate;
    config.onRecvCallback = onRecvCallback;
    config.onSendCallback = onSendCallback;

    if (channels > 0) {
        if (channelMap == NULL) {
            if (channels > 8) {
                mal_zero_memory(config.channelMap, sizeof(mal_channel)*MAL_MAX_CHANNELS);
            } else {
                mal_get_standard_channel_map(mal_standard_channel_map_default, channels, config.channelMap);
            }
        } else {
            mal_copy_memory(config.channelMap, channelMap, sizeof(config.channelMap));
        }
    } else {
        mal_zero_memory(config.channelMap, sizeof(mal_channel)*MAL_MAX_CHANNELS);
    }

    return config;
}
#endif  // MAL_NO_DEVICE_IO


void mal_get_standard_channel_map_microsoft(mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    // Based off the speaker configurations mentioned here: https://docs.microsoft.com/en-us/windows-hardware/drivers/ddi/content/ksmedia/ns-ksmedia-ksaudio_channel_config
    switch (channels)
    {
        case 1:
        {
            channelMap[0] = MAL_CHANNEL_MONO;
        } break;

        case 2:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
        } break;

        case 3: // Not defined, but best guess.
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
        } break;

        case 4:
        {
#ifndef MAL_USE_QUAD_MICROSOFT_CHANNEL_MAP
            // Surround. Using the Surround profile has the advantage of the 3rd channel (MAL_CHANNEL_FRONT_CENTER) mapping nicely
            // with higher channel counts.
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_BACK_CENTER;
#else
            // Quad.
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
#endif
        } break;

        case 5: // Not defined, but best guess.
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_BACK_LEFT;
            channelMap[4] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case 6:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_LFE;
            channelMap[4] = MAL_CHANNEL_SIDE_LEFT;
            channelMap[5] = MAL_CHANNEL_SIDE_RIGHT;
        } break;

        case 7: // Not defined, but best guess.
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_LFE;
            channelMap[4] = MAL_CHANNEL_BACK_CENTER;
            channelMap[5] = MAL_CHANNEL_SIDE_LEFT;
            channelMap[6] = MAL_CHANNEL_SIDE_RIGHT;
        } break;

        case 8:
        default:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_LFE;
            channelMap[4] = MAL_CHANNEL_BACK_LEFT;
            channelMap[5] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[6] = MAL_CHANNEL_SIDE_LEFT;
            channelMap[7] = MAL_CHANNEL_SIDE_RIGHT;
        } break;
    }

    // Remainder.
    if (channels > 8) {
        for (mal_uint32 iChannel = 8; iChannel < MAL_MAX_CHANNELS; ++iChannel) {
            channelMap[iChannel] = (mal_channel)(MAL_CHANNEL_AUX_0 + (iChannel-8));
        }
    }
}

void mal_get_standard_channel_map_alsa(mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    switch (channels)
    {
        case 1:
        {
            channelMap[0] = MAL_CHANNEL_MONO;
        } break;

        case 2:
        {
            channelMap[0] = MAL_CHANNEL_LEFT;
            channelMap[1] = MAL_CHANNEL_RIGHT;
        } break;

        case 3:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
        } break;

        case 4:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case 5:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
        } break;

        case 6:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[5] = MAL_CHANNEL_LFE;
        } break;

        case 7:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[5] = MAL_CHANNEL_LFE;
            channelMap[6] = MAL_CHANNEL_BACK_CENTER;
        } break;

        case 8:
        default:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[5] = MAL_CHANNEL_LFE;
            channelMap[6] = MAL_CHANNEL_SIDE_LEFT;
            channelMap[7] = MAL_CHANNEL_SIDE_RIGHT;
        } break;
    }

    // Remainder.
    if (channels > 8) {
        for (mal_uint32 iChannel = 8; iChannel < MAL_MAX_CHANNELS; ++iChannel) {
            channelMap[iChannel] = (mal_channel)(MAL_CHANNEL_AUX_0 + (iChannel-8));
        }
    }
}

void mal_get_standard_channel_map_rfc3551(mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    switch (channels)
    {
        case 1:
        {
            channelMap[0] = MAL_CHANNEL_MONO;
        } break;

        case 2:
        {
            channelMap[0] = MAL_CHANNEL_LEFT;
            channelMap[1] = MAL_CHANNEL_RIGHT;
        } break;

        case 3:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
        } break;

        case 4:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[2] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[3] = MAL_CHANNEL_BACK_CENTER;
        } break;

        case 5:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_BACK_LEFT;
            channelMap[4] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case 6:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_SIDE_LEFT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[4] = MAL_CHANNEL_SIDE_RIGHT;
            channelMap[5] = MAL_CHANNEL_BACK_CENTER;
        } break;
    }

    // Remainder.
    if (channels > 8) {
        for (mal_uint32 iChannel = 6; iChannel < MAL_MAX_CHANNELS; ++iChannel) {
            channelMap[iChannel] = (mal_channel)(MAL_CHANNEL_AUX_0 + (iChannel-6));
        }
    }
}

void mal_get_standard_channel_map_flac(mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    switch (channels)
    {
        case 1:
        {
            channelMap[0] = MAL_CHANNEL_MONO;
        } break;

        case 2:
        {
            channelMap[0] = MAL_CHANNEL_LEFT;
            channelMap[1] = MAL_CHANNEL_RIGHT;
        } break;

        case 3:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
        } break;

        case 4:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case 5:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_BACK_LEFT;
            channelMap[4] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case 6:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_LFE;
            channelMap[4] = MAL_CHANNEL_BACK_LEFT;
            channelMap[5] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case 7:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_LFE;
            channelMap[4] = MAL_CHANNEL_BACK_CENTER;
            channelMap[5] = MAL_CHANNEL_SIDE_LEFT;
            channelMap[6] = MAL_CHANNEL_SIDE_RIGHT;
        } break;

        case 8:
        default:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[3] = MAL_CHANNEL_LFE;
            channelMap[4] = MAL_CHANNEL_BACK_LEFT;
            channelMap[5] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[6] = MAL_CHANNEL_SIDE_LEFT;
            channelMap[7] = MAL_CHANNEL_SIDE_RIGHT;
        } break;
    }

    // Remainder.
    if (channels > 8) {
        for (mal_uint32 iChannel = 8; iChannel < MAL_MAX_CHANNELS; ++iChannel) {
            channelMap[iChannel] = (mal_channel)(MAL_CHANNEL_AUX_0 + (iChannel-8));
        }
    }
}

void mal_get_standard_channel_map_vorbis(mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    // In Vorbis' type 0 channel mapping, the first two channels are not always the standard left/right - it
    // will have the center speaker where the right usually goes. Why?!
    switch (channels)
    {
        case 1:
        {
            channelMap[0] = MAL_CHANNEL_MONO;
        } break;

        case 2:
        {
            channelMap[0] = MAL_CHANNEL_LEFT;
            channelMap[1] = MAL_CHANNEL_RIGHT;
        } break;

        case 3:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[2] = MAL_CHANNEL_FRONT_RIGHT;
        } break;

        case 4:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case 5:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[2] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[3] = MAL_CHANNEL_BACK_LEFT;
            channelMap[4] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case 6:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[2] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[3] = MAL_CHANNEL_BACK_LEFT;
            channelMap[4] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[5] = MAL_CHANNEL_LFE;
        } break;

        case 7:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[2] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[3] = MAL_CHANNEL_SIDE_LEFT;
            channelMap[4] = MAL_CHANNEL_SIDE_RIGHT;
            channelMap[5] = MAL_CHANNEL_BACK_CENTER;
            channelMap[6] = MAL_CHANNEL_LFE;
        } break;

        case 8:
        default:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[2] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[3] = MAL_CHANNEL_SIDE_LEFT;
            channelMap[4] = MAL_CHANNEL_SIDE_RIGHT;
            channelMap[5] = MAL_CHANNEL_BACK_LEFT;
            channelMap[6] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[7] = MAL_CHANNEL_LFE;
        } break;
    }

    // Remainder.
    if (channels > 8) {
        for (mal_uint32 iChannel = 8; iChannel < MAL_MAX_CHANNELS; ++iChannel) {
            channelMap[iChannel] = (mal_channel)(MAL_CHANNEL_AUX_0 + (iChannel-8));
        }
    }
}

void mal_get_standard_channel_map_sound4(mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    switch (channels)
    {
        case 1:
        {
            channelMap[0] = MAL_CHANNEL_MONO;
        } break;

        case 2:
        {
            channelMap[0] = MAL_CHANNEL_LEFT;
            channelMap[1] = MAL_CHANNEL_RIGHT;
        } break;

        case 3:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_CENTER;
        } break;

        case 4:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case 5:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
        } break;

        case 6:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[5] = MAL_CHANNEL_LFE;
        } break;

        case 7:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[5] = MAL_CHANNEL_BACK_CENTER;
            channelMap[6] = MAL_CHANNEL_LFE;
        } break;

        case 8:
        default:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[5] = MAL_CHANNEL_LFE;
            channelMap[6] = MAL_CHANNEL_SIDE_LEFT;
            channelMap[7] = MAL_CHANNEL_SIDE_RIGHT;
        } break;
    }

    // Remainder.
    if (channels > 8) {
        for (mal_uint32 iChannel = 8; iChannel < MAL_MAX_CHANNELS; ++iChannel) {
            channelMap[iChannel] = (mal_channel)(MAL_CHANNEL_AUX_0 + (iChannel-8));
        }
    }
}

void mal_get_standard_channel_map_sndio(mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    switch (channels)
    {
        case 1:
        {
            channelMap[0] = MAL_CHANNEL_MONO;
        } break;

        case 2:
        {
            channelMap[0] = MAL_CHANNEL_LEFT;
            channelMap[1] = MAL_CHANNEL_RIGHT;
        } break;

        case 3:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_FRONT_CENTER;
        } break;

        case 4:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
        } break;

        case 5:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
        } break;

        case 6:
        default:
        {
            channelMap[0] = MAL_CHANNEL_FRONT_LEFT;
            channelMap[1] = MAL_CHANNEL_FRONT_RIGHT;
            channelMap[2] = MAL_CHANNEL_BACK_LEFT;
            channelMap[3] = MAL_CHANNEL_BACK_RIGHT;
            channelMap[4] = MAL_CHANNEL_FRONT_CENTER;
            channelMap[5] = MAL_CHANNEL_LFE;
        } break;
    }

    // Remainder.
    if (channels > 6) {
        for (mal_uint32 iChannel = 6; iChannel < MAL_MAX_CHANNELS; ++iChannel) {
            channelMap[iChannel] = (mal_channel)(MAL_CHANNEL_AUX_0 + (iChannel-6));
        }
    }
}

void mal_get_standard_channel_map(mal_standard_channel_map standardChannelMap, mal_uint32 channels, mal_channel channelMap[MAL_MAX_CHANNELS])
{
    switch (standardChannelMap)
    {
        case mal_standard_channel_map_alsa:
        {
            mal_get_standard_channel_map_alsa(channels, channelMap);
        } break;

        case mal_standard_channel_map_rfc3551:
        {
            mal_get_standard_channel_map_rfc3551(channels, channelMap);
        } break;

        case mal_standard_channel_map_flac:
        {
            mal_get_standard_channel_map_flac(channels, channelMap);
        } break;

        case mal_standard_channel_map_vorbis:
        {
            mal_get_standard_channel_map_vorbis(channels, channelMap);
        } break;

        case mal_standard_channel_map_sound4:
        {
            mal_get_standard_channel_map_sound4(channels, channelMap);
        } break;
        
        case mal_standard_channel_map_sndio:
        {
            mal_get_standard_channel_map_sndio(channels, channelMap);
        } break;

        case mal_standard_channel_map_microsoft:
        default:
        {
            mal_get_standard_channel_map_microsoft(channels, channelMap);
        } break;
    }
}

void mal_channel_map_copy(mal_channel* pOut, const mal_channel* pIn, mal_uint32 channels)
{
    if (pOut != NULL && pIn != NULL && channels > 0) {
        mal_copy_memory(pOut, pIn, sizeof(*pOut) * channels);
    }
}

mal_bool32 mal_channel_map_valid(mal_uint32 channels, const mal_channel channelMap[MAL_MAX_CHANNELS])
{
    if (channelMap == NULL) {
        return MAL_FALSE;
    }

    // A channel count of 0 is invalid.
    if (channels == 0) {
        return MAL_FALSE;
    }

    // It does not make sense to have a mono channel when there is more than 1 channel.
    if (channels > 1) {
        for (mal_uint32 iChannel = 0; iChannel < channels; ++iChannel) {
            if (channelMap[iChannel] == MAL_CHANNEL_MONO) {
                return MAL_FALSE;
            }
        }
    }

    return MAL_TRUE;
}

mal_bool32 mal_channel_map_equal(mal_uint32 channels, const mal_channel channelMapA[MAL_MAX_CHANNELS], const mal_channel channelMapB[MAL_MAX_CHANNELS])
{
    if (channelMapA == channelMapB) {
        return MAL_FALSE;
    }

    if (channels == 0 || channels > MAL_MAX_CHANNELS) {
        return MAL_FALSE;
    }

    for (mal_uint32 iChannel = 0; iChannel < channels; ++iChannel) {
        if (channelMapA[iChannel] != channelMapB[iChannel]) {
            return MAL_FALSE;
        }
    }

    return MAL_TRUE;
}

mal_bool32 mal_channel_map_blank(mal_uint32 channels, const mal_channel channelMap[MAL_MAX_CHANNELS])
{
    for (mal_uint32 iChannel = 0; iChannel < channels; ++iChannel) {
        if (channelMap[iChannel] != MAL_CHANNEL_NONE) {
            return MAL_FALSE;
        }
    }

    return MAL_TRUE;
}

mal_bool32 mal_channel_map_contains_channel_position(mal_uint32 channels, const mal_channel channelMap[MAL_MAX_CHANNELS], mal_channel channelPosition)
{
    for (mal_uint32 iChannel = 0; iChannel < channels; ++iChannel) {
        if (channelMap[iChannel] == channelPosition) {
            return MAL_TRUE;
        }
    }

    return MAL_FALSE;
}




//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Format Conversion.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//#define MAL_USE_REFERENCE_CONVERSION_APIS   1
//#define MAL_USE_SSE

void mal_copy_memory_64(void* dst, const void* src, mal_uint64 sizeInBytes)
{
#if 0xFFFFFFFFFFFFFFFF <= MAL_SIZE_MAX
    mal_copy_memory(dst, src, (size_t)sizeInBytes);
#else
    while (sizeInBytes > 0) {
        mal_uint64 bytesToCopyNow = sizeInBytes;
        if (bytesToCopyNow > MAL_SIZE_MAX) {
            bytesToCopyNow = MAL_SIZE_MAX;
        }

        mal_copy_memory(dst, src, (size_t)bytesToCopyNow);  // Safe cast to size_t.

        sizeInBytes -= bytesToCopyNow;
        dst = (      void*)((      mal_uint8*)dst + bytesToCopyNow);
        src = (const void*)((const mal_uint8*)src + bytesToCopyNow);
    }
#endif
}

void mal_zero_memory_64(void* dst, mal_uint64 sizeInBytes)
{
#if 0xFFFFFFFFFFFFFFFF <= MAL_SIZE_MAX
    mal_zero_memory(dst, (size_t)sizeInBytes);
#else
    while (sizeInBytes > 0) {
        mal_uint64 bytesToZeroNow = sizeInBytes;
        if (bytesToZeroNow > MAL_SIZE_MAX) {
            bytesToZeroNow = MAL_SIZE_MAX;
        }

        mal_zero_memory(dst, (size_t)bytesToZeroNow);  // Safe cast to size_t.

        sizeInBytes -= bytesToZeroNow;
        dst = (void*)((mal_uint8*)dst + bytesToZeroNow);
    }
#endif
}


// u8
void mal_pcm_u8_to_u8(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    (void)ditherMode;
    mal_copy_memory_64(dst, src, count * sizeof(mal_uint8));
}


void mal_pcm_u8_to_s16__reference(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    (void)ditherMode;

    mal_int16* dst_s16 = (mal_int16*)dst;
    const mal_uint8* src_u8 = (const mal_uint8*)src;

    mal_uint64 i;
    for (i = 0; i < count; i += 1) {
        mal_int16 x = src_u8[i];
        x = x - 128;
        x = x << 8;
        dst_s16[i] = x;
    }
}

void mal_pcm_u8_to_s16__optimized(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s16__reference(dst, src, count, ditherMode);
}

#if defined(MAL_SUPPORT_SSE2)
void mal_pcm_u8_to_s16__sse2(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s16__optimized(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_AVX2)
void mal_pcm_u8_to_s16__avx2(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s16__optimized(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_AVX512)
void mal_pcm_u8_to_s16__avx512(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s16__avx2(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_NEON)
void mal_pcm_u8_to_s16__neon(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s16__optimized(dst, src, count, ditherMode);
}
#endif

void mal_pcm_u8_to_s16(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
#ifdef MAL_USE_REFERENCE_CONVERSION_APIS
    mal_pcm_u8_to_s16__reference(dst, src, count, ditherMode);
#else
    mal_pcm_u8_to_s16__optimized(dst, src, count, ditherMode);
#endif
}


void mal_pcm_u8_to_s24__reference(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    (void)ditherMode;

    mal_uint8* dst_s24 = (mal_uint8*)dst;
    const mal_uint8* src_u8 = (const mal_uint8*)src;

    mal_uint64 i;
    for (i = 0; i < count; i += 1) {
        mal_int16 x = src_u8[i];
        x = x - 128;

        dst_s24[i*3+0] = 0;
        dst_s24[i*3+1] = 0;
        dst_s24[i*3+2] = (mal_uint8)((mal_int8)x);
    }
}

void mal_pcm_u8_to_s24__optimized(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s24__reference(dst, src, count, ditherMode);
}

#if defined(MAL_SUPPORT_SSE2)
void mal_pcm_u8_to_s24__sse2(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s24__optimized(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_AVX2)
void mal_pcm_u8_to_s24__avx2(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s24__optimized(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_AVX512)
void mal_pcm_u8_to_s24__avx512(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s24__avx2(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_NEON)
void mal_pcm_u8_to_s24__neon(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s24__optimized(dst, src, count, ditherMode);
}
#endif

void mal_pcm_u8_to_s24(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
#ifdef MAL_USE_REFERENCE_CONVERSION_APIS
    mal_pcm_u8_to_s24__reference(dst, src, count, ditherMode);
#else
    mal_pcm_u8_to_s24__optimized(dst, src, count, ditherMode);
#endif
}


void mal_pcm_u8_to_s32__reference(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    (void)ditherMode;

    mal_int32* dst_s32 = (mal_int32*)dst;
    const mal_uint8* src_u8 = (const mal_uint8*)src;

    mal_uint64 i;
    for (i = 0; i < count; i += 1) {
        mal_int32 x = src_u8[i];
        x = x - 128;
        x = x << 24;
        dst_s32[i] = x;
    }
}

void mal_pcm_u8_to_s32__optimized(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s32__reference(dst, src, count, ditherMode);
}

#if defined(MAL_SUPPORT_SSE2)
void mal_pcm_u8_to_s32__sse2(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s32__optimized(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_AVX2)
void mal_pcm_u8_to_s32__avx2(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s32__optimized(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_AVX512)
void mal_pcm_u8_to_s32__avx512(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s32__avx2(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_NEON)
void mal_pcm_u8_to_s32__neon(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_s32__optimized(dst, src, count, ditherMode);
}
#endif

void mal_pcm_u8_to_s32(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
#ifdef MAL_USE_REFERENCE_CONVERSION_APIS
    mal_pcm_u8_to_s32__reference(dst, src, count, ditherMode);
#else
    mal_pcm_u8_to_s32__optimized(dst, src, count, ditherMode);
#endif
}


void mal_pcm_u8_to_f32__reference(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    (void)ditherMode;

    float* dst_f32 = (float*)dst;
    const mal_uint8* src_u8 = (const mal_uint8*)src;

    mal_uint64 i;
    for (i = 0; i < count; i += 1) {
        float x = (float)src_u8[i];
        x = x * 0.00784313725490196078f;    // 0..255 to 0..2
        x = x - 1;                          // 0..2 to -1..1

        dst_f32[i] = x;
    }
}

void mal_pcm_u8_to_f32__optimized(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_f32__reference(dst, src, count, ditherMode);
}

#if defined(MAL_SUPPORT_SSE2)
void mal_pcm_u8_to_f32__sse2(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_f32__optimized(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_AVX2)
void mal_pcm_u8_to_f32__avx2(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_f32__optimized(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_AVX512)
void mal_pcm_u8_to_f32__avx512(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_f32__avx2(dst, src, count, ditherMode);
}
#endif
#if defined(MAL_SUPPORT_NEON)
void mal_pcm_u8_to_f32__neon(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
    mal_pcm_u8_to_f32__optimized(dst, src, count, ditherMode);
}
#endif

void mal_pcm_u8_to_f32(void* dst, const void* src, mal_uint64 count, mal_dither_mode ditherMode)
{
#ifdef MAL_USE_REFERENCE_CONVERSION_APIS
    mal_pcm_u8_to_f32__reference(dst, src, count, ditherMode);
