/*
Audio playback and capture library. Choice of public domain or MIT-0. See license statements at the end of this file.
miniaudio - v0.10.21 - 2020-10-30

David Reid - mackron@gmail.com

Website:       https://miniaud.io
Documentation: https://miniaud.io/docs
GitHub:        https://github.com/mackron/miniaudio
*/
#include "miniaudio.h"

#ifndef miniaudio_c
#define miniaudio_c

#include <assert.h>
#include <limits.h> /* For INT_MAX */
#include <math.h>   /* sin(), etc. */

#include <stdarg.h>
#include <stdio.h>
#if !defined(_MSC_VER) && !defined(__DMC__)
    #include <strings.h>    /* For strcasecmp(). */
    #include <wchar.h>      /* For wcslen(), wcsrtombs() */
#endif

#ifdef MA_WIN32
#include <windows.h>
#else
#include <stdlib.h>     /* For malloc(), free(), wcstombs(). */
#include <string.h>     /* For memset() */
#include <sched.h>
#include <sys/time.h>   /* select() (used for ma_sleep()). */
#endif

#include <sys/stat.h>   /* For fstat(), etc. */

#ifdef MA_EMSCRIPTEN
#include <emscripten/emscripten.h>
#endif

#if !defined(MA_64BIT) && !defined(MA_32BIT)
#ifdef _WIN32
#ifdef _WIN64
#define MA_64BIT
#else
#define MA_32BIT
#endif
#endif
#endif

#if !defined(MA_64BIT) && !defined(MA_32BIT)
#ifdef __GNUC__
#ifdef __LP64__
#define MA_64BIT
#else
#define MA_32BIT
#endif
#endif
#endif

#if !defined(MA_64BIT) && !defined(MA_32BIT)
#include <stdint.h>
#if INTPTR_MAX == INT64_MAX
#define MA_64BIT
#else
#define MA_32BIT
#endif
#endif

/* Architecture Detection */
#if defined(__x86_64__) || defined(_M_X64)
#define MA_X64
#elif defined(__i386) || defined(_M_IX86)
#define MA_X86
#elif defined(__arm__) || defined(_M_ARM)
#define MA_ARM
#endif

/* Cannot currently support AVX-512 if AVX is disabled. */
#if !defined(MA_NO_AVX512) && defined(MA_NO_AVX2)
#define MA_NO_AVX512
#endif

/* Intrinsics Support */
#if defined(MA_X64) || defined(MA_X86)
    #if defined(_MSC_VER) && !defined(__clang__)
        /* MSVC. */
        #if _MSC_VER >= 1400 && !defined(MA_NO_SSE2)   /* 2005 */
            #define MA_SUPPORT_SSE2
        #endif
        /*#if _MSC_VER >= 1600 && !defined(MA_NO_AVX)*/    /* 2010 */
        /*    #define MA_SUPPORT_AVX*/
        /*#endif*/
        #if _MSC_VER >= 1700 && !defined(MA_NO_AVX2)   /* 2012 */
            #define MA_SUPPORT_AVX2
        #endif
        #if _MSC_VER >= 1910 && !defined(MA_NO_AVX512) /* 2017 */
            #define MA_SUPPORT_AVX512
        #endif
    #else
        /* Assume GNUC-style. */
        #if defined(__SSE2__) && !defined(MA_NO_SSE2)
            #define MA_SUPPORT_SSE2
        #endif
        /*#if defined(__AVX__) && !defined(MA_NO_AVX)*/
        /*    #define MA_SUPPORT_AVX*/
        /*#endif*/
        #if defined(__AVX2__) && !defined(MA_NO_AVX2)
            #define MA_SUPPORT_AVX2
        #endif
        #if defined(__AVX512F__) && !defined(MA_NO_AVX512)
            #define MA_SUPPORT_AVX512
        #endif
    #endif

    /* If at this point we still haven't determined compiler support for the intrinsics just fall back to __has_include. */
    #if !defined(__GNUC__) && !defined(__clang__) && defined(__has_include)
        #if !defined(MA_SUPPORT_SSE2)   && !defined(MA_NO_SSE2)   && __has_include(<emmintrin.h>)
            #define MA_SUPPORT_SSE2
        #endif
        /*#if !defined(MA_SUPPORT_AVX)    && !defined(MA_NO_AVX)    && __has_include(<immintrin.h>)*/
        /*    #define MA_SUPPORT_AVX*/
        /*#endif*/
        #if !defined(MA_SUPPORT_AVX2)   && !defined(MA_NO_AVX2)   && __has_include(<immintrin.h>)
            #define MA_SUPPORT_AVX2
        #endif
        #if !defined(MA_SUPPORT_AVX512) && !defined(MA_NO_AVX512) && __has_include(<zmmintrin.h>)
            #define MA_SUPPORT_AVX512
        #endif
    #endif

    #if defined(MA_SUPPORT_AVX512)
        #include <immintrin.h>  /* Not a mistake. Intentionally including <immintrin.h> instead of <zmmintrin.h> because otherwise the compiler will complain. */
    #elif defined(MA_SUPPORT_AVX2) || defined(MA_SUPPORT_AVX)
        #include <immintrin.h>
    #elif defined(MA_SUPPORT_SSE2)
        #include <emmintrin.h>
    #endif
#endif

#if defined(MA_ARM)
    #if !defined(MA_NO_NEON) && (defined(__ARM_NEON) || defined(__aarch64__) || defined(_M_ARM64))
        #define MA_SUPPORT_NEON
    #endif

    /* Fall back to looking for the #include file. */
    #if !defined(__GNUC__) && !defined(__clang__) && defined(__has_include)
        #if !defined(MA_SUPPORT_NEON) && !defined(MA_NO_NEON) && __has_include(<arm_neon.h>)
            #define MA_SUPPORT_NEON
        #endif
    #endif

    #if defined(MA_SUPPORT_NEON)
        #include <arm_neon.h>
    #endif
#endif

/* Begin globally disabled warnings. */
#if defined(_MSC_VER)
    #pragma warning(push)
    #pragma warning(disable:4752)   /* found Intel(R) Advanced Vector Extensions; consider using /arch:AVX */
#endif

#if defined(MA_X64) || defined(MA_X86)
    #if defined(_MSC_VER) && !defined(__clang__)
        #if _MSC_VER >= 1400
            #include <intrin.h>
            static MA_INLINE void ma_cpuid(int info[4], int fid)
            {
                __cpuid(info, fid);
            }
        #else
            #define MA_NO_CPUID
        #endif

        #if _MSC_VER >= 1600 && (defined(_MSC_FULL_VER) && _MSC_FULL_VER >= 160040219)
            static MA_INLINE unsigned __int64 ma_xgetbv(int reg)
            {
                return _xgetbv(reg);
            }
        #else
            #define MA_NO_XGETBV
        #endif
    #elif (defined(__GNUC__) || defined(__clang__)) && !defined(MA_ANDROID)
        static MA_INLINE void ma_cpuid(int info[4], int fid)
        {
            /*
            It looks like the -fPIC option uses the ebx register which GCC complains about. We can work around this by just using a different register, the
            specific register of which I'm letting the compiler decide on. The "k" prefix is used to specify a 32-bit register. The {...} syntax is for
            supporting different assembly dialects.
            
            What's basically happening is that we're saving and restoring the ebx register manually.
            */
            #if defined(DRFLAC_X86) && defined(__PIC__)
                __asm__ __volatile__ (
                    "xchg{l} {%%}ebx, %k1;"
                    "cpuid;"
                    "xchg{l} {%%}ebx, %k1;"
                    : "=a"(info[0]), "=&r"(info[1]), "=c"(info[2]), "=d"(info[3]) : "a"(fid), "c"(0)
                );
            #else
                __asm__ __volatile__ (
                    "cpuid" : "=a"(info[0]), "=b"(info[1]), "=c"(info[2]), "=d"(info[3]) : "a"(fid), "c"(0)
                );
            #endif
        }

        static MA_INLINE ma_uint64 ma_xgetbv(int reg)
        {
            unsigned int hi;
            unsigned int lo;

            __asm__ __volatile__ (
                "xgetbv" : "=a"(lo), "=d"(hi) : "c"(reg)
            );

            return ((ma_uint64)hi << 32) | (ma_uint64)lo;
        }
    #else
        #define MA_NO_CPUID
        #define MA_NO_XGETBV
    #endif
#else
    #define MA_NO_CPUID
    #define MA_NO_XGETBV
#endif

static MA_INLINE ma_bool32 ma_has_sse2(void)
{
#if defined(MA_SUPPORT_SSE2)
    #if (defined(MA_X64) || defined(MA_X86)) && !defined(MA_NO_SSE2)
        #if defined(MA_X64)
            return MA_TRUE;    /* 64-bit targets always support SSE2. */
        #elif (defined(_M_IX86_FP) && _M_IX86_FP == 2) || defined(__SSE2__)
            return MA_TRUE;    /* If the compiler is allowed to freely generate SSE2 code we can assume support. */
        #else
            #if defined(MA_NO_CPUID)
                return MA_FALSE;
            #else
                int info[4];
                ma_cpuid(info, 1);
                return (info[3] & (1 << 26)) != 0;
            #endif
        #endif
    #else
        return MA_FALSE;       /* SSE2 is only supported on x86 and x64 architectures. */
    #endif
#else
    return MA_FALSE;           /* No compiler support. */
#endif
}

#if 0
static MA_INLINE ma_bool32 ma_has_avx()
{
#if defined(MA_SUPPORT_AVX)
    #if (defined(MA_X64) || defined(MA_X86)) && !defined(MA_NO_AVX)
        #if defined(_AVX_) || defined(__AVX__)
            return MA_TRUE;    /* If the compiler is allowed to freely generate AVX code we can assume support. */
        #else
            /* AVX requires both CPU and OS support. */
            #if defined(MA_NO_CPUID) || defined(MA_NO_XGETBV)
                return MA_FALSE;
            #else
                int info[4];
                ma_cpuid(info, 1);
                if (((info[2] & (1 << 27)) != 0) && ((info[2] & (1 << 28)) != 0)) {
                    ma_uint64 xrc = ma_xgetbv(0);
                    if ((xrc & 0x06) == 0x06) {
                        return MA_TRUE;
                    } else {
                        return MA_FALSE;
                    }
                } else {
                    return MA_FALSE;
                }
            #endif
        #endif
    #else
        return MA_FALSE;       /* AVX is only supported on x86 and x64 architectures. */
    #endif
#else
    return MA_FALSE;           /* No compiler support. */
#endif
}
#endif

static MA_INLINE ma_bool32 ma_has_avx2(void)
{
#if defined(MA_SUPPORT_AVX2)
    #if (defined(MA_X64) || defined(MA_X86)) && !defined(MA_NO_AVX2)
        #if defined(_AVX2_) || defined(__AVX2__)
            return MA_TRUE;    /* If the compiler is allowed to freely generate AVX2 code we can assume support. */
        #else
            /* AVX2 requires both CPU and OS support. */
            #if defined(MA_NO_CPUID) || defined(MA_NO_XGETBV)
                return MA_FALSE;
            #else
                int info1[4];
                int info7[4];
                ma_cpuid(info1, 1);
                ma_cpuid(info7, 7);
                if (((info1[2] & (1 << 27)) != 0) && ((info7[1] & (1 << 5)) != 0)) {
                    ma_uint64 xrc = ma_xgetbv(0);
                    if ((xrc & 0x06) == 0x06) {
                        return MA_TRUE;
                    } else {
                        return MA_FALSE;
                    }
                } else {
                    return MA_FALSE;
                }
            #endif
        #endif
    #else
        return MA_FALSE;       /* AVX2 is only supported on x86 and x64 architectures. */
    #endif
#else
    return MA_FALSE;           /* No compiler support. */
#endif
}

static MA_INLINE ma_bool32 ma_has_avx512f(void)
{
#if defined(MA_SUPPORT_AVX512)
    #if (defined(MA_X64) || defined(MA_X86)) && !defined(MA_NO_AVX512)
        #if defined(__AVX512F__)
            return MA_TRUE;    /* If the compiler is allowed to freely generate AVX-512F code we can assume support. */
        #else
            /* AVX-512 requires both CPU and OS support. */
            #if defined(MA_NO_CPUID) || defined(MA_NO_XGETBV)
                return MA_FALSE;
            #else
                int info1[4];
                int info7[4];
                ma_cpuid(info1, 1);
                ma_cpuid(info7, 7);
                if (((info1[2] & (1 << 27)) != 0) && ((info7[1] & (1 << 16)) != 0)) {
                    ma_uint64 xrc = ma_xgetbv(0);
                    if ((xrc & 0xE6) == 0xE6) {
                        return MA_TRUE;
                    } else {
                        return MA_FALSE;
                    }
                } else {
                    return MA_FALSE;
                }
            #endif
        #endif
    #else
        return MA_FALSE;       /* AVX-512F is only supported on x86 and x64 architectures. */
    #endif
#else
    return MA_FALSE;           /* No compiler support. */
#endif
}

static MA_INLINE ma_bool32 ma_has_neon(void)
{
#if defined(MA_SUPPORT_NEON)
    #if defined(MA_ARM) && !defined(MA_NO_NEON)
        #if (defined(__ARM_NEON) || defined(__aarch64__) || defined(_M_ARM64))
            return MA_TRUE;    /* If the compiler is allowed to freely generate NEON code we can assume support. */
        #else
            /* TODO: Runtime check. */
            return MA_FALSE;
        #endif
    #else
        return MA_FALSE;       /* NEON is only supported on ARM architectures. */
    #endif
#else
    return MA_FALSE;           /* No compiler support. */
#endif
}

#define MA_SIMD_NONE    0
#define MA_SIMD_SSE2    1
#define MA_SIMD_AVX2    2
#define MA_SIMD_NEON    3

#ifndef MA_PREFERRED_SIMD
    #  if defined(MA_SUPPORT_SSE2) && defined(MA_PREFER_SSE2)
        #define MA_PREFERRED_SIMD MA_SIMD_SSE2
    #elif defined(MA_SUPPORT_AVX2) && defined(MA_PREFER_AVX2)
        #define MA_PREFERRED_SIMD MA_SIMD_AVX2
    #elif defined(MA_SUPPORT_NEON) && defined(MA_PREFER_NEON)
        #define MA_PREFERRED_SIMD MA_SIMD_NEON
    #else
        #define MA_PREFERRED_SIMD MA_SIMD_NONE
    #endif
#endif


#if defined(_MSC_VER) && _MSC_VER >= 1400
    #define MA_HAS_BYTESWAP16_INTRINSIC
    #define MA_HAS_BYTESWAP32_INTRINSIC
    #define MA_HAS_BYTESWAP64_INTRINSIC
#elif defined(__clang__)
    #if defined(__has_builtin)
        #if __has_builtin(__builtin_bswap16)
            #define MA_HAS_BYTESWAP16_INTRINSIC
        #endif
        #if __has_builtin(__builtin_bswap32)
            #define MA_HAS_BYTESWAP32_INTRINSIC
        #endif
        #if __has_builtin(__builtin_bswap64)
            #define MA_HAS_BYTESWAP64_INTRINSIC
        #endif
    #endif
#elif defined(__GNUC__)
    #if ((__GNUC__ > 4) || (__GNUC__ == 4 && __GNUC_MINOR__ >= 3))
        #define MA_HAS_BYTESWAP32_INTRINSIC
        #define MA_HAS_BYTESWAP64_INTRINSIC
    #endif
    #if ((__GNUC__ > 4) || (__GNUC__ == 4 && __GNUC_MINOR__ >= 8))
        #define MA_HAS_BYTESWAP16_INTRINSIC
    #endif
#endif


static MA_INLINE ma_bool32 ma_is_little_endian(void)
{
#if defined(MA_X86) || defined(MA_X64)
    return MA_TRUE;
#else
    int n = 1;
    return (*(char*)&n) == 1;
#endif
}

static MA_INLINE ma_bool32 ma_is_big_endian(void)
{
    return !ma_is_little_endian();
}


static MA_INLINE ma_uint32 ma_swap_endian_uint32(ma_uint32 n)
{
#ifdef MA_HAS_BYTESWAP32_INTRINSIC
    #if defined(_MSC_VER)
        return _byteswap_ulong(n);
    #elif defined(__GNUC__) || defined(__clang__)
        #if defined(MA_ARM) && (defined(__ARM_ARCH) && __ARM_ARCH >= 6) && !defined(MA_64BIT)   /* <-- 64-bit inline assembly has not been tested, so disabling for now. */
            /* Inline assembly optimized implementation for ARM. In my testing, GCC does not generate optimized code with __builtin_bswap32(). */
            ma_uint32 r;
            __asm__ __volatile__ (
            #if defined(MA_64BIT)
                "rev %w[out], %w[in]" : [out]"=r"(r) : [in]"r"(n)   /* <-- This is untested. If someone in the community could test this, that would be appreciated! */
            #else
                "rev %[out], %[in]" : [out]"=r"(r) : [in]"r"(n)
            #endif
            );
            return r;
        #else
            return __builtin_bswap32(n);
        #endif
    #else
        #error "This compiler does not support the byte swap intrinsic."
    #endif
#else
    return ((n & 0xFF000000) >> 24) |
           ((n & 0x00FF0000) >>  8) |
           ((n & 0x0000FF00) <<  8) |
           ((n & 0x000000FF) << 24);
#endif
}


#if !defined(MA_EMSCRIPTEN)
#ifdef MA_WIN32
static void ma_sleep__win32(ma_uint32 milliseconds)
{
    Sleep((DWORD)milliseconds);
}
#endif
#ifdef MA_POSIX
static void ma_sleep__posix(ma_uint32 milliseconds)
{
#ifdef MA_EMSCRIPTEN
    (void)milliseconds;
    MA_ASSERT(MA_FALSE);  /* The Emscripten build should never sleep. */
#else
    #if _POSIX_C_SOURCE >= 199309L
        struct timespec ts;
        ts.tv_sec  = milliseconds / 1000;
        ts.tv_nsec = milliseconds % 1000 * 1000000;
        nanosleep(&ts, NULL);
    #else
        struct timeval tv;
        tv.tv_sec  = milliseconds / 1000;
        tv.tv_usec = milliseconds % 1000 * 1000;
        select(0, NULL, NULL, NULL, &tv);
    #endif
#endif
}
#endif

static void ma_sleep(ma_uint32 milliseconds)
{
#ifdef MA_WIN32
    ma_sleep__win32(milliseconds);
#endif
#ifdef MA_POSIX
    ma_sleep__posix(milliseconds);
#endif
}
#endif

static MA_INLINE void ma_yield()
{
#if defined(__i386) || defined(_M_IX86) || defined(__x86_64__) || defined(_M_X64)
    /* x86/x64 */
    #if defined(_MSC_VER) && !defined(__clang__)
        #if _MSC_VER >= 1400
            _mm_pause();
        #else
            __asm pause;
        #endif
    #else
        __asm__ __volatile__ ("pause");
    #endif
#elif (defined(__arm__) && defined(__ARM_ARCH) && __ARM_ARCH >= 7) || (defined(_M_ARM) && _M_ARM >= 7) || defined(__ARM_ARCH_6K__) || defined(__ARM_ARCH_6T2__)
    /* ARM */
    #if defined(_MSC_VER)
        /* Apparently there is a __yield() intrinsic that's compatible with ARM, but I cannot find documentation for it nor can I find where it's declared. */
        __yield();
    #else
        __asm__ __volatile__ ("yield"); /* ARMv6K/ARMv6T2 and above. */
    #endif
#else
    /* Unknown or unsupported architecture. No-op. */
#endif
}



#ifndef MA_COINIT_VALUE
#define MA_COINIT_VALUE    0   /* 0 = COINIT_MULTITHREADED */
#endif



#ifndef MA_PI
#define MA_PI      3.14159265358979323846264f
#endif
#ifndef MA_PI_D
#define MA_PI_D    3.14159265358979323846264
#endif
#ifndef MA_TAU
#define MA_TAU     6.28318530717958647693f
#endif
#ifndef MA_TAU_D
#define MA_TAU_D   6.28318530717958647693
#endif


/* The default format when ma_format_unknown (0) is requested when initializing a device. */
#ifndef MA_DEFAULT_FORMAT
#define MA_DEFAULT_FORMAT                                   ma_format_f32
#endif

/* The default channel count to use when 0 is used when initializing a device. */
#ifndef MA_DEFAULT_CHANNELS
#define MA_DEFAULT_CHANNELS                                 2
#endif

/* The default sample rate to use when 0 is used when initializing a device. */
#ifndef MA_DEFAULT_SAMPLE_RATE
#define MA_DEFAULT_SAMPLE_RATE                              48000
#endif

/* Default periods when none is specified in ma_device_init(). More periods means more work on the CPU. */
#ifndef MA_DEFAULT_PERIODS
#define MA_DEFAULT_PERIODS                                  3
#endif

/* The default period size in milliseconds for low latency mode. */
#ifndef MA_DEFAULT_PERIOD_SIZE_IN_MILLISECONDS_LOW_LATENCY
#define MA_DEFAULT_PERIOD_SIZE_IN_MILLISECONDS_LOW_LATENCY  10
#endif

/* The default buffer size in milliseconds for conservative mode. */
#ifndef MA_DEFAULT_PERIOD_SIZE_IN_MILLISECONDS_CONSERVATIVE
#define MA_DEFAULT_PERIOD_SIZE_IN_MILLISECONDS_CONSERVATIVE 100
#endif

/* The default LPF filter order for linear resampling. Note that this is clamped to MA_MAX_FILTER_ORDER. */
#ifndef MA_DEFAULT_RESAMPLER_LPF_ORDER
    #if MA_MAX_FILTER_ORDER >= 4
        #define MA_DEFAULT_RESAMPLER_LPF_ORDER  4
    #else
        #define MA_DEFAULT_RESAMPLER_LPF_ORDER  MA_MAX_FILTER_ORDER
    #endif
#endif


#if defined(__GNUC__)
    #pragma GCC diagnostic push
    #pragma GCC diagnostic ignored "-Wunused-variable"
#endif
/* Standard sample rates, in order of priority. */
static ma_uint32 g_maStandardSampleRatePriorities[] = {
    MA_SAMPLE_RATE_48000,  /* Most common */
    MA_SAMPLE_RATE_44100,

    MA_SAMPLE_RATE_32000,  /* Lows */
    MA_SAMPLE_RATE_24000,
    MA_SAMPLE_RATE_22050,

    MA_SAMPLE_RATE_88200,  /* Highs */
    MA_SAMPLE_RATE_96000,
    MA_SAMPLE_RATE_176400,
    MA_SAMPLE_RATE_192000,

    MA_SAMPLE_RATE_16000,  /* Extreme lows */
    MA_SAMPLE_RATE_11025,
    MA_SAMPLE_RATE_8000,

    MA_SAMPLE_RATE_352800, /* Extreme highs */
    MA_SAMPLE_RATE_384000
};

static ma_format g_maFormatPriorities[] = {
    ma_format_s16,         /* Most common */
    ma_format_f32,
    
    /*ma_format_s24_32,*/    /* Clean alignment */
    ma_format_s32,
    
    ma_format_s24,         /* Unclean alignment */
    
    ma_format_u8           /* Low quality */
};
#if defined(__GNUC__)
    #pragma GCC diagnostic pop
#endif


MA_API void ma_version(ma_uint32* pMajor, ma_uint32* pMinor, ma_uint32* pRevision)
{
    if (pMajor) {
        *pMajor = MA_VERSION_MAJOR;
    }

    if (pMinor) {
        *pMinor = MA_VERSION_MINOR;
    }

    if (pRevision) {
        *pRevision = MA_VERSION_REVISION;
    }
}

MA_API const char* ma_version_string(void)
{
    return MA_VERSION_STRING;
}


/******************************************************************************

Standard Library Stuff

******************************************************************************/
#ifndef MA_MALLOC
#ifdef MA_WIN32
#define MA_MALLOC(sz) HeapAlloc(GetProcessHeap(), 0, (sz))
#else
#define MA_MALLOC(sz) malloc((sz))
#endif
#endif

#ifndef MA_REALLOC
#ifdef MA_WIN32
#define MA_REALLOC(p, sz) (((sz) > 0) ? ((p) ? HeapReAlloc(GetProcessHeap(), 0, (p), (sz)) : HeapAlloc(GetProcessHeap(), 0, (sz))) : ((VOID*)(size_t)(HeapFree(GetProcessHeap(), 0, (p)) & 0)))
#else
#define MA_REALLOC(p, sz) realloc((p), (sz))
#endif
#endif

#ifndef MA_FREE
#ifdef MA_WIN32
#define MA_FREE(p) HeapFree(GetProcessHeap(), 0, (p))
#else
#define MA_FREE(p) free((p))
#endif
#endif

#ifndef MA_ZERO_MEMORY
#ifdef MA_WIN32
#define MA_ZERO_MEMORY(p, sz) ZeroMemory((p), (sz))
#else
#define MA_ZERO_MEMORY(p, sz) memset((p), 0, (sz))
#endif
#endif

#ifndef MA_COPY_MEMORY
#ifdef MA_WIN32
#define MA_COPY_MEMORY(dst, src, sz) CopyMemory((dst), (src), (sz))
#else
#define MA_COPY_MEMORY(dst, src, sz) memcpy((dst), (src), (sz))
#endif
#endif

#ifndef MA_ASSERT
#ifdef MA_WIN32
#define MA_ASSERT(condition) assert(condition)
#else
#define MA_ASSERT(condition) assert(condition)
#endif
#endif

#define MA_ZERO_OBJECT(p) MA_ZERO_MEMORY((p), sizeof(*(p)))

#define ma_countof(x)               (sizeof(x) / sizeof(x[0]))
#define ma_max(x, y)                (((x) > (y)) ? (x) : (y))
#define ma_min(x, y)                (((x) < (y)) ? (x) : (y))
#define ma_abs(x)                   (((x) > 0) ? (x) : -(x))
#define ma_clamp(x, lo, hi)         (ma_max(lo, ma_min(x, hi)))
#define ma_offset_ptr(p, offset)    (((ma_uint8*)(p)) + (offset))

#define ma_buffer_frame_capacity(buffer, channels, format) (sizeof(buffer) / ma_get_bytes_per_sample(format) / (channels))

static MA_INLINE double ma_sin(double x)
{
    /* TODO: Implement custom sin(x). */
    return sin(x);
}

static MA_INLINE double ma_exp(double x)
{
    /* TODO: Implement custom exp(x). */
    return exp(x);
}

static MA_INLINE double ma_log(double x)
{
    /* TODO: Implement custom log(x). */
    return log(x);
}

static MA_INLINE double ma_pow(double x, double y)
{
    /* TODO: Implement custom pow(x, y). */
    return pow(x, y);
}

static MA_INLINE double ma_sqrt(double x)
{
    /* TODO: Implement custom sqrt(x). */
    return sqrt(x);
}


static MA_INLINE double ma_cos(double x)
{
    return ma_sin((MA_PI_D*0.5) - x);
}

static MA_INLINE double ma_log10(double x)
{
    return ma_log(x) * 0.43429448190325182765;
}

static MA_INLINE float ma_powf(float x, float y)
{
    return (float)ma_pow((double)x, (double)y);
}

static MA_INLINE float ma_log10f(float x)
{
    return (float)ma_log10((double)x);
}


/*
Return Values:
  0:  Success
  22: EINVAL
  34: ERANGE

Not using symbolic constants for errors because I want to avoid #including errno.h
*/
MA_API int ma_strcpy_s(char* dst, size_t dstSizeInBytes, const char* src)
{
    size_t i;

    if (dst == 0) {
        return 22;
    }
    if (dstSizeInBytes == 0) {
        return 34;
    }
    if (src == 0) {
        dst[0] = '\0';
        return 22;
    }

    for (i = 0; i < dstSizeInBytes && src[i] != '\0'; ++i) {
        dst[i] = src[i];
    }

    if (i < dstSizeInBytes) {
        dst[i] = '\0';
        return 0;
    }

    dst[0] = '\0';
    return 34;
}

MA_API int ma_strncpy_s(char* dst, size_t dstSizeInBytes, const char* src, size_t count)
{
    size_t maxcount;
    size_t i;

    if (dst == 0) {
        return 22;
    }
    if (dstSizeInBytes == 0) {
        return 34;
    }
    if (src == 0) {
        dst[0] = '\0';
        return 22;
    }

    maxcount = count;
    if (count == ((size_t)-1) || count >= dstSizeInBytes) {        /* -1 = _TRUNCATE */
        maxcount = dstSizeInBytes - 1;
    }

    for (i = 0; i < maxcount && src[i] != '\0'; ++i) {
        dst[i] = src[i];
    }

    if (src[i] == '\0' || i == count || count == ((size_t)-1)) {
        dst[i] = '\0';
        return 0;
    }

    dst[0] = '\0';
    return 34;
}

MA_API int ma_strcat_s(char* dst, size_t dstSizeInBytes, const char* src)
{
    char* dstorig;

    if (dst == 0) {
        return 22;
    }
    if (dstSizeInBytes == 0) {
        return 34;
    }
    if (src == 0) {
        dst[0] = '\0';
        return 22;
    }

    dstorig = dst;

    while (dstSizeInBytes > 0 && dst[0] != '\0') {
        dst += 1;
        dstSizeInBytes -= 1;
    }

    if (dstSizeInBytes == 0) {
        return 22;  /* Unterminated. */
    }


    while (dstSizeInBytes > 0 && src[0] != '\0') {
        *dst++ = *src++;
        dstSizeInBytes -= 1;
    }

    if (dstSizeInBytes > 0) {
        dst[0] = '\0';
    } else {
        dstorig[0] = '\0';
        return 34;
    }

    return 0;
}

MA_API int ma_strncat_s(char* dst, size_t dstSizeInBytes, const char* src, size_t count)
{
    char* dstorig;

    if (dst == 0) {
        return 22;
    }
    if (dstSizeInBytes == 0) {
        return 34;
    }
    if (src == 0) {
        return 22;
    }

    dstorig = dst;

    while (dstSizeInBytes > 0 && dst[0] != '\0') {
        dst += 1;
        dstSizeInBytes -= 1;
    }

    if (dstSizeInBytes == 0) {
        return 22;  /* Unterminated. */
    }


    if (count == ((size_t)-1)) {        /* _TRUNCATE */
        count = dstSizeInBytes - 1;
    }

    while (dstSizeInBytes > 0 && src[0] != '\0' && count > 0) {
        *dst++ = *src++;
        dstSizeInBytes -= 1;
        count -= 1;
    }

    if (dstSizeInBytes > 0) {
        dst[0] = '\0';
    } else {
        dstorig[0] = '\0';
        return 34;
    }

    return 0;
}

MA_API int ma_itoa_s(int value, char* dst, size_t dstSizeInBytes, int radix)
{
    int sign;
    unsigned int valueU;
    char* dstEnd;

    if (dst == NULL || dstSizeInBytes == 0) {
        return 22;
    }
    if (radix < 2 || radix > 36) {
        dst[0] = '\0';
        return 22;
    }

    sign = (value < 0 && radix == 10) ? -1 : 1;     /* The negative sign is only used when the base is 10. */

    if (value < 0) {
        valueU = -value;
    } else {
        valueU = value;
    }

    dstEnd = dst;
    do
    {
        int remainder = valueU % radix;
        if (remainder > 9) {
            *dstEnd = (char)((remainder - 10) + 'a');
        } else {
            *dstEnd = (char)(remainder + '0');
        }

        dstEnd += 1;
        dstSizeInBytes -= 1;
        valueU /= radix;
    } while (dstSizeInBytes > 0 && valueU > 0);

    if (dstSizeInBytes == 0) {
        dst[0] = '\0';
        return 22;  /* Ran out of room in the output buffer. */
    }

    if (sign < 0) {
        *dstEnd++ = '-';
        dstSizeInBytes -= 1;
    }

    if (dstSizeInBytes == 0) {
        dst[0] = '\0';
        return 22;  /* Ran out of room in the output buffer. */
    }

    *dstEnd = '\0';


    /* At this point the string will be reversed. */
    dstEnd -= 1;
    while (dst < dstEnd) {
        char temp = *dst;
        *dst = *dstEnd;
        *dstEnd = temp;

        dst += 1;
        dstEnd -= 1;
    }

    return 0;
}

MA_API int ma_strcmp(const char* str1, const char* str2)
{
    if (str1 == str2) return  0;

    /* These checks differ from the standard implementation. It's not important, but I prefer it just for sanity. */
    if (str1 == NULL) return -1;
    if (str2 == NULL) return  1;

    for (;;) {
        if (str1[0] == '\0') {
            break;
        }
        if (str1[0] != str2[0]) {
            break;
        }

        str1 += 1;
        str2 += 1;
    }

    return ((unsigned char*)str1)[0] - ((unsigned char*)str2)[0];
}

MA_API int ma_strappend(char* dst, size_t dstSize, const char* srcA, const char* srcB)
{
    int result;

    result = ma_strncpy_s(dst, dstSize, srcA, (size_t)-1);
    if (result != 0) {
        return result;
    }

    result = ma_strncat_s(dst, dstSize, srcB, (size_t)-1);
    if (result != 0) {
        return result;
    }

    return result;
}

MA_API char* ma_copy_string(const char* src, const ma_allocation_callbacks* pAllocationCallbacks)
{
    size_t sz = strlen(src)+1;
    char* dst = (char*)ma_malloc(sz, pAllocationCallbacks);
    if (dst == NULL) {
        return NULL;
    }

    ma_strcpy_s(dst, sz, src);

    return dst;
}


#include <errno.h>
static ma_result ma_result_from_errno(int e)
{
    switch (e)
    {
        case 0: return MA_SUCCESS;
    #ifdef EPERM
        case EPERM: return MA_INVALID_OPERATION;
    #endif
    #ifdef ENOENT
        case ENOENT: return MA_DOES_NOT_EXIST;
    #endif
    #ifdef ESRCH
        case ESRCH: return MA_DOES_NOT_EXIST;
    #endif
    #ifdef EINTR
        case EINTR: return MA_INTERRUPT;
    #endif
    #ifdef EIO
        case EIO: return MA_IO_ERROR;
    #endif
    #ifdef ENXIO
        case ENXIO: return MA_DOES_NOT_EXIST;
    #endif
    #ifdef E2BIG
        case E2BIG: return MA_INVALID_ARGS;
    #endif
    #ifdef ENOEXEC
        case ENOEXEC: return MA_INVALID_FILE;
    #endif
    #ifdef EBADF
        case EBADF: return MA_INVALID_FILE;
    #endif
    #ifdef ECHILD
        case ECHILD: return MA_ERROR;
    #endif
    #ifdef EAGAIN
        case EAGAIN: return MA_UNAVAILABLE;
    #endif
    #ifdef ENOMEM
        case ENOMEM: return MA_OUT_OF_MEMORY;
    #endif
    #ifdef EACCES
        case EACCES: return MA_ACCESS_DENIED;
    #endif
    #ifdef EFAULT
        case EFAULT: return MA_BAD_ADDRESS;
    #endif
    #ifdef ENOTBLK
        case ENOTBLK: return MA_ERROR;
    #endif
    #ifdef EBUSY
        case EBUSY: return MA_BUSY;
    #endif
    #ifdef EEXIST
        case EEXIST: return MA_ALREADY_EXISTS;
    #endif
    #ifdef EXDEV
        case EXDEV: return MA_ERROR;
    #endif
    #ifdef ENODEV
        case ENODEV: return MA_DOES_NOT_EXIST;
    #endif
    #ifdef ENOTDIR
        case ENOTDIR: return MA_NOT_DIRECTORY;
    #endif
    #ifdef EISDIR
        case EISDIR: return MA_IS_DIRECTORY;
    #endif
    #ifdef EINVAL
        case EINVAL: return MA_INVALID_ARGS;
    #endif
    #ifdef ENFILE
        case ENFILE: return MA_TOO_MANY_OPEN_FILES;
    #endif
    #ifdef EMFILE
        case EMFILE: return MA_TOO_MANY_OPEN_FILES;
    #endif
    #ifdef ENOTTY
        case ENOTTY: return MA_INVALID_OPERATION;
    #endif
    #ifdef ETXTBSY
        case ETXTBSY: return MA_BUSY;
    #endif
    #ifdef EFBIG
        case EFBIG: return MA_TOO_BIG;
    #endif
    #ifdef ENOSPC
        case ENOSPC: return MA_NO_SPACE;
    #endif
    #ifdef ESPIPE
        case ESPIPE: return MA_BAD_SEEK;
    #endif
    #ifdef EROFS
        case EROFS: return MA_ACCESS_DENIED;
    #endif
    #ifdef EMLINK
        case EMLINK: return MA_TOO_MANY_LINKS;
    #endif
    #ifdef EPIPE
        case EPIPE: return MA_BAD_PIPE;
    #endif
    #ifdef EDOM
        case EDOM: return MA_OUT_OF_RANGE;
    #endif
    #ifdef ERANGE
        case ERANGE: return MA_OUT_OF_RANGE;
    #endif
    #ifdef EDEADLK
        case EDEADLK: return MA_DEADLOCK;
    #endif
    #ifdef ENAMETOOLONG
        case ENAMETOOLONG: return MA_PATH_TOO_LONG;
    #endif
    #ifdef ENOLCK
        case ENOLCK: return MA_ERROR;
    #endif
    #ifdef ENOSYS
        case ENOSYS: return MA_NOT_IMPLEMENTED;
    #endif
    #ifdef ENOTEMPTY
        case ENOTEMPTY: return MA_DIRECTORY_NOT_EMPTY;
    #endif
    #ifdef ELOOP
        case ELOOP: return MA_TOO_MANY_LINKS;
    #endif
    #ifdef ENOMSG
        case ENOMSG: return MA_NO_MESSAGE;
    #endif
    #ifdef EIDRM
        case EIDRM: return MA_ERROR;
    #endif
    #ifdef ECHRNG
        case ECHRNG: return MA_ERROR;
    #endif
    #ifdef EL2NSYNC
        case EL2NSYNC: return MA_ERROR;
    #endif
    #ifdef EL3HLT
        case EL3HLT: return MA_ERROR;
    #endif
    #ifdef EL3RST
        case EL3RST: return MA_ERROR;
    #endif
    #ifdef ELNRNG
        case ELNRNG: return MA_OUT_OF_RANGE;
    #endif
    #ifdef EUNATCH
        case EUNATCH: return MA_ERROR;
    #endif
    #ifdef ENOCSI
        case ENOCSI: return MA_ERROR;
    #endif
    #ifdef EL2HLT
        case EL2HLT: return MA_ERROR;
    #endif
    #ifdef EBADE
        case EBADE: return MA_ERROR;
    #endif
    #ifdef EBADR
        case EBADR: return MA_ERROR;
    #endif
    #ifdef EXFULL
        case EXFULL: return MA_ERROR;
    #endif
    #ifdef ENOANO
        case ENOANO: return MA_ERROR;
    #endif
    #ifdef EBADRQC
        case EBADRQC: return MA_ERROR;
    #endif
    #ifdef EBADSLT
        case EBADSLT: return MA_ERROR;
    #endif
    #ifdef EBFONT
        case EBFONT: return MA_INVALID_FILE;
    #endif
    #ifdef ENOSTR
        case ENOSTR: return MA_ERROR;
    #endif
    #ifdef ENODATA
        case ENODATA: return MA_NO_DATA_AVAILABLE;
    #endif
    #ifdef ETIME
        case ETIME: return MA_TIMEOUT;
    #endif
    #ifdef ENOSR
        case ENOSR: return MA_NO_DATA_AVAILABLE;
    #endif
    #ifdef ENONET
        case ENONET: return MA_NO_NETWORK;
    #endif
    #ifdef ENOPKG
        case ENOPKG: return MA_ERROR;
    #endif
    #ifdef EREMOTE
        case EREMOTE: return MA_ERROR;
    #endif
    #ifdef ENOLINK
        case ENOLINK: return MA_ERROR;
    #endif
    #ifdef EADV
        case EADV: return MA_ERROR;
    #endif
    #ifdef ESRMNT
        case ESRMNT: return MA_ERROR;
    #endif
    #ifdef ECOMM
        case ECOMM: return MA_ERROR;
    #endif
    #ifdef EPROTO
        case EPROTO: return MA_ERROR;
    #endif
    #ifdef EMULTIHOP
        case EMULTIHOP: return MA_ERROR;
    #endif
    #ifdef EDOTDOT
        case EDOTDOT: return MA_ERROR;
    #endif
    #ifdef EBADMSG
        case EBADMSG: return MA_BAD_MESSAGE;
    #endif
    #ifdef EOVERFLOW
        case EOVERFLOW: return MA_TOO_BIG;
    #endif
    #ifdef ENOTUNIQ
        case ENOTUNIQ: return MA_NOT_UNIQUE;
    #endif
    #ifdef EBADFD
        case EBADFD: return MA_ERROR;
    #endif
    #ifdef EREMCHG
        case EREMCHG: return MA_ERROR;
    #endif
    #ifdef ELIBACC
        case ELIBACC: return MA_ACCESS_DENIED;
    #endif
    #ifdef ELIBBAD
        case ELIBBAD: return MA_INVALID_FILE;
    #endif
    #ifdef ELIBSCN
        case ELIBSCN: return MA_INVALID_FILE;
    #endif
    #ifdef ELIBMAX
        case ELIBMAX: return MA_ERROR;
    #endif
    #ifdef ELIBEXEC
        case ELIBEXEC: return MA_ERROR;
    #endif
    #ifdef EILSEQ
        case EILSEQ: return MA_INVALID_DATA;
    #endif
    #ifdef ERESTART
        case ERESTART: return MA_ERROR;
    #endif
    #ifdef ESTRPIPE
        case ESTRPIPE: return MA_ERROR;
    #endif
    #ifdef EUSERS
        case EUSERS: return MA_ERROR;
    #endif
    #ifdef ENOTSOCK
        case ENOTSOCK: return MA_NOT_SOCKET;
    #endif
    #ifdef EDESTADDRREQ
        case EDESTADDRREQ: return MA_NO_ADDRESS;
    #endif
    #ifdef EMSGSIZE
        case EMSGSIZE: return MA_TOO_BIG;
    #endif
    #ifdef EPROTOTYPE
        case EPROTOTYPE: return MA_BAD_PROTOCOL;
    #endif
    #ifdef ENOPROTOOPT
        case ENOPROTOOPT: return MA_PROTOCOL_UNAVAILABLE;
    #endif
    #ifdef EPROTONOSUPPORT
        case EPROTONOSUPPORT: return MA_PROTOCOL_NOT_SUPPORTED;
    #endif
    #ifdef ESOCKTNOSUPPORT
        case ESOCKTNOSUPPORT: return MA_SOCKET_NOT_SUPPORTED;
    #endif
    #ifdef EOPNOTSUPP
        case EOPNOTSUPP: return MA_INVALID_OPERATION;
    #endif
    #ifdef EPFNOSUPPORT
        case EPFNOSUPPORT: return MA_PROTOCOL_FAMILY_NOT_SUPPORTED;
    #endif
    #ifdef EAFNOSUPPORT
        case EAFNOSUPPORT: return MA_ADDRESS_FAMILY_NOT_SUPPORTED;
    #endif
    #ifdef EADDRINUSE
        case EADDRINUSE: return MA_ALREADY_IN_USE;
    #endif
    #ifdef EADDRNOTAVAIL
        case EADDRNOTAVAIL: return MA_ERROR;
    #endif
    #ifdef ENETDOWN
        case ENETDOWN: return MA_NO_NETWORK;
    #endif
    #ifdef ENETUNREACH
        case ENETUNREACH: return MA_NO_NETWORK;
    #endif
    #ifdef ENETRESET
        case ENETRESET: return MA_NO_NETWORK;
    #endif
    #ifdef ECONNABORTED
        case ECONNABORTED: return MA_NO_NETWORK;
    #endif
    #ifdef ECONNRESET
        case ECONNRESET: return MA_CONNECTION_RESET;
    #endif
    #ifdef ENOBUFS
        case ENOBUFS: return MA_NO_SPACE;
    #endif
    #ifdef EISCONN
        case EISCONN: return MA_ALREADY_CONNECTED;
    #endif
    #ifdef ENOTCONN
        case ENOTCONN: return MA_NOT_CONNECTED;
    #endif
    #ifdef ESHUTDOWN
        case ESHUTDOWN: return MA_ERROR;
    #endif
    #ifdef ETOOMANYREFS
        case ETOOMANYREFS: return MA_ERROR;
    #endif
    #ifdef ETIMEDOUT
        case ETIMEDOUT: return MA_TIMEOUT;
    #endif
    #ifdef ECONNREFUSED
        case ECONNREFUSED: return MA_CONNECTION_REFUSED;
    #endif
    #ifdef EHOSTDOWN
        case EHOSTDOWN: return MA_NO_HOST;
    #endif
    #ifdef EHOSTUNREACH
        case EHOSTUNREACH: return MA_NO_HOST;
    #endif
    #ifdef EALREADY
        case EALREADY: return MA_IN_PROGRESS;
    #endif
    #ifdef EINPROGRESS
        case EINPROGRESS: return MA_IN_PROGRESS;
    #endif
    #ifdef ESTALE
        case ESTALE: return MA_INVALID_FILE;
    #endif
    #ifdef EUCLEAN
        case EUCLEAN: return MA_ERROR;
    #endif
    #ifdef ENOTNAM
        case ENOTNAM: return MA_ERROR;
    #endif
    #ifdef ENAVAIL
        case ENAVAIL: return MA_ERROR;
    #endif
    #ifdef EISNAM
        case EISNAM: return MA_ERROR;
    #endif
    #ifdef EREMOTEIO
        case EREMOTEIO: return MA_IO_ERROR;
    #endif
    #ifdef EDQUOT
        case EDQUOT: return MA_NO_SPACE;
    #endif
    #ifdef ENOMEDIUM
        case ENOMEDIUM: return MA_DOES_NOT_EXIST;
    #endif
    #ifdef EMEDIUMTYPE
        case EMEDIUMTYPE: return MA_ERROR;
    #endif
    #ifdef ECANCELED
        case ECANCELED: return MA_CANCELLED;
    #endif
    #ifdef ENOKEY
        case ENOKEY: return MA_ERROR;
    #endif
    #ifdef EKEYEXPIRED
        case EKEYEXPIRED: return MA_ERROR;
    #endif
    #ifdef EKEYREVOKED
        case EKEYREVOKED: return MA_ERROR;
    #endif
    #ifdef EKEYREJECTED
        case EKEYREJECTED: return MA_ERROR;
    #endif
    #ifdef EOWNERDEAD
        case EOWNERDEAD: return MA_ERROR;
    #endif
    #ifdef ENOTRECOVERABLE
        case ENOTRECOVERABLE: return MA_ERROR;
    #endif
    #ifdef ERFKILL
        case ERFKILL: return MA_ERROR;
    #endif
    #ifdef EHWPOISON
        case EHWPOISON: return MA_ERROR;
    #endif
        default: return MA_ERROR;
    }
}

MA_API ma_result ma_fopen(FILE** ppFile, const char* pFilePath, const char* pOpenMode)
{
#if _MSC_VER && _MSC_VER >= 1400
    errno_t err;
#endif

    if (ppFile != NULL) {
        *ppFile = NULL;  /* Safety. */
    }

    if (pFilePath == NULL || pOpenMode == NULL || ppFile == NULL) {
        return MA_INVALID_ARGS;
    }

#if _MSC_VER && _MSC_VER >= 1400
    err = fopen_s(ppFile, pFilePath, pOpenMode);
    if (err != 0) {
        return ma_result_from_errno(err);
    }
#else
#if defined(_WIN32) || defined(__APPLE__)
    *ppFile = fopen(pFilePath, pOpenMode);
#else
    #if defined(_FILE_OFFSET_BITS) && _FILE_OFFSET_BITS == 64 && defined(_LARGEFILE64_SOURCE)
        *ppFile = fopen64(pFilePath, pOpenMode);
    #else
        *ppFile = fopen(pFilePath, pOpenMode);
    #endif
#endif
    if (*ppFile == NULL) {
        ma_result result = ma_result_from_errno(errno);
        if (result == MA_SUCCESS) {
            result = MA_ERROR;   /* Just a safety check to make sure we never ever return success when pFile == NULL. */
        }

        return result;
    }
#endif

    return MA_SUCCESS;
}



/*
_wfopen() isn't always available in all compilation environments.

    * Windows only.
    * MSVC seems to support it universally as far back as VC6 from what I can tell (haven't checked further back).
    * MinGW-64 (both 32- and 64-bit) seems to support it.
    * MinGW wraps it in !defined(__STRICT_ANSI__).

This can be reviewed as compatibility issues arise. The preference is to use _wfopen_s() and _wfopen() as opposed to the wcsrtombs()
fallback, so if you notice your compiler not detecting this properly I'm happy to look at adding support.
*/
#if defined(_WIN32)
    #if defined(_MSC_VER) || defined(__MINGW64__) || !defined(__STRICT_ANSI__)
        #define MA_HAS_WFOPEN
    #endif
#endif

MA_API ma_result ma_wfopen(FILE** ppFile, const wchar_t* pFilePath, const wchar_t* pOpenMode, const ma_allocation_callbacks* pAllocationCallbacks)
{
    if (ppFile != NULL) {
        *ppFile = NULL;  /* Safety. */
    }

    if (pFilePath == NULL || pOpenMode == NULL || ppFile == NULL) {
        return MA_INVALID_ARGS;
    }

#if defined(MA_HAS_WFOPEN)
    {
        /* Use _wfopen() on Windows. */
    #if defined(_MSC_VER) && _MSC_VER >= 1400
        errno_t err = _wfopen_s(ppFile, pFilePath, pOpenMode);
        if (err != 0) {
            return ma_result_from_errno(err);
        }
    #else
        *ppFile = _wfopen(pFilePath, pOpenMode);
        if (*ppFile == NULL) {
            return ma_result_from_errno(errno);
        }
    #endif
        (void)pAllocationCallbacks;
    }
#else
    /*
    Use fopen() on anything other than Windows. Requires a conversion. This is annoying because fopen() is locale specific. The only real way I can
    think of to do this is with wcsrtombs(). Note that wcstombs() is apparently not thread-safe because it uses a static global mbstate_t object for
    maintaining state. I've checked this with -std=c89 and it works, but if somebody get's a compiler error I'll look into improving compatibility.
    */
    {
        mbstate_t mbs;
        size_t lenMB;
        const wchar_t* pFilePathTemp = pFilePath;
        char* pFilePathMB = NULL;
        char pOpenModeMB[32] = {0};

        /* Get the length first. */
        MA_ZERO_OBJECT(&mbs);
        lenMB = wcsrtombs(NULL, &pFilePathTemp, 0, &mbs);
        if (lenMB == (size_t)-1) {
            return ma_result_from_errno(errno);
        }

        pFilePathMB = (char*)ma_malloc(lenMB + 1, pAllocationCallbacks);
        if (pFilePathMB == NULL) {
            return MA_OUT_OF_MEMORY;
        }

        pFilePathTemp = pFilePath;
        MA_ZERO_OBJECT(&mbs);
        wcsrtombs(pFilePathMB, &pFilePathTemp, lenMB + 1, &mbs);

        /* The open mode should always consist of ASCII characters so we should be able to do a trivial conversion. */
        {
            size_t i = 0;
            for (;;) {
                if (pOpenMode[i] == 0) {
                    pOpenModeMB[i] = '\0';
                    break;
                }

                pOpenModeMB[i] = (char)pOpenMode[i];
                i += 1;
            }
        }

        *ppFile = fopen(pFilePathMB, pOpenModeMB);

        ma_free(pFilePathMB, pAllocationCallbacks);
    }

    if (*ppFile == NULL) {
        return MA_ERROR;
    }
#endif

    return MA_SUCCESS;
}



static MA_INLINE void ma_copy_memory_64(void* dst, const void* src, ma_uint64 sizeInBytes)
{
#if 0xFFFFFFFFFFFFFFFF <= MA_SIZE_MAX
    MA_COPY_MEMORY(dst, src, (size_t)sizeInBytes);
#else
    while (sizeInBytes > 0) {
        ma_uint64 bytesToCopyNow = sizeInBytes;
        if (bytesToCopyNow > MA_SIZE_MAX) {
            bytesToCopyNow = MA_SIZE_MAX;
        }

        MA_COPY_MEMORY(dst, src, (size_t)bytesToCopyNow);  /* Safe cast to size_t. */

        sizeInBytes -= bytesToCopyNow;
        dst = (      void*)((      ma_uint8*)dst + bytesToCopyNow);
        src = (const void*)((const ma_uint8*)src + bytesToCopyNow);
    }
#endif
}

static MA_INLINE void ma_zero_memory_64(void* dst, ma_uint64 sizeInBytes)
{
#if 0xFFFFFFFFFFFFFFFF <= MA_SIZE_MAX
    MA_ZERO_MEMORY(dst, (size_t)sizeInBytes);
#else
    while (sizeInBytes > 0) {
        ma_uint64 bytesToZeroNow = sizeInBytes;
        if (bytesToZeroNow > MA_SIZE_MAX) {
            bytesToZeroNow = MA_SIZE_MAX;
        }

        MA_ZERO_MEMORY(dst, (size_t)bytesToZeroNow);  /* Safe cast to size_t. */

        sizeInBytes -= bytesToZeroNow;
        dst = (void*)((ma_uint8*)dst + bytesToZeroNow);
    }
#endif
}


/* Thanks to good old Bit Twiddling Hacks for this one: http://graphics.stanford.edu/~seander/bithacks.html#RoundUpPowerOf2 */
static MA_INLINE unsigned int ma_next_power_of_2(unsigned int x)
{
    x--;
    x |= x >> 1;
    x |= x >> 2;
    x |= x >> 4;
    x |= x >> 8;
    x |= x >> 16;
    x++;

    return x;
}

static MA_INLINE unsigned int ma_prev_power_of_2(unsigned int x)
{
    return ma_next_power_of_2(x) >> 1;
}

static MA_INLINE unsigned int ma_round_to_power_of_2(unsigned int x)
{
    unsigned int prev = ma_prev_power_of_2(x);
    unsigned int next = ma_next_power_of_2(x);
    if ((next - x) > (x - prev)) {
        return prev;
    } else {
        return next;
    }
}

static MA_INLINE unsigned int ma_count_set_bits(unsigned int x)
{
    unsigned int count = 0;
    while (x != 0) {
        if (x & 1) {
            count += 1;
        }
        
        x = x >> 1;
    }
    
    return count;
}



/* Clamps an f32 sample to -1..1 */
static MA_INLINE float ma_clip_f32(float x)
{
    if (x < -1) return -1;
    if (x > +1) return +1;
    return x;
}

static MA_INLINE float ma_mix_f32(float x, float y, float a)
{
    return x*(1-a) + y*a;
}
static MA_INLINE float ma_mix_f32_fast(float x, float y, float a)
{
    float r0 = (y - x);
    float r1 = r0*a;
    return x + r1;
    /*return x + (y - x)*a;*/
}

#if defined(MA_SUPPORT_SSE2)
static MA_INLINE __m128 ma_mix_f32_fast__sse2(__m128 x, __m128 y, __m128 a)
{
    return _mm_add_ps(x, _mm_mul_ps(_mm_sub_ps(y, x), a));
}
#endif
#if defined(MA_SUPPORT_AVX2)
static MA_INLINE __m256 ma_mix_f32_fast__avx2(__m256 x, __m256 y, __m256 a)
{
    return _mm256_add_ps(x, _mm256_mul_ps(_mm256_sub_ps(y, x), a));
}
#endif
#if defined(MA_SUPPORT_AVX512)
static MA_INLINE __m512 ma_mix_f32_fast__avx512(__m512 x, __m512 y, __m512 a)
{
    return _mm512_add_ps(x, _mm512_mul_ps(_mm512_sub_ps(y, x), a));
}
#endif
#if defined(MA_SUPPORT_NEON)
static MA_INLINE float32x4_t ma_mix_f32_fast__neon(float32x4_t x, float32x4_t y, float32x4_t a)
{
    return vaddq_f32(x, vmulq_f32(vsubq_f32(y, x), a));
}
#endif


static MA_INLINE double ma_mix_f64(double x, double y, double a)
{
    return x*(1-a) + y*a;
}
static MA_INLINE double ma_mix_f64_fast(double x, double y, double a)
{
    return x + (y - x)*a;
}

static MA_INLINE float ma_scale_to_range_f32(float x, float lo, float hi)
{
    return lo + x*(hi-lo);
}


/*
Greatest common factor using Euclid's algorithm iteratively.
*/
static MA_INLINE ma_uint32 ma_gcf_u32(ma_uint32 a, ma_uint32 b)
{
    for (;;) {
        if (b == 0) {
            break;
        } else {
            ma_uint32 t = a;
            a = b;
            b = t % a;
        }
    }

    return a;
}


/*
Random Number Generation

miniaudio uses the LCG random number generation algorithm. This is good enough for audio.

Note that miniaudio's global LCG implementation uses global state which is _not_ thread-local. When this is called across
multiple threads, results will be unpredictable. However, it won't crash and results will still be random enough for
miniaudio's purposes.
*/
#ifndef MA_DEFAULT_LCG_SEED
#define MA_DEFAULT_LCG_SEED 4321
#endif

#define MA_LCG_M   2147483647
#define MA_LCG_A   48271
#define MA_LCG_C   0

static ma_lcg g_maLCG = {MA_DEFAULT_LCG_SEED}; /* Non-zero initial seed. Use ma_seed() to use an explicit seed. */

static MA_INLINE void ma_lcg_seed(ma_lcg* pLCG, ma_int32 seed)
{
    MA_ASSERT(pLCG != NULL);
    pLCG->state = seed;
}

static MA_INLINE ma_int32 ma_lcg_rand_s32(ma_lcg* pLCG)
{
    pLCG->state = (MA_LCG_A * pLCG->state + MA_LCG_C) % MA_LCG_M;
    return pLCG->state;
}

static MA_INLINE ma_uint32 ma_lcg_rand_u32(ma_lcg* pLCG)
{
    return (ma_uint32)ma_lcg_rand_s32(pLCG);
}

static MA_INLINE ma_int16 ma_lcg_rand_s16(ma_lcg* pLCG)
{
    return (ma_int16)(ma_lcg_rand_s32(pLCG) & 0xFFFF);
}

static MA_INLINE double ma_lcg_rand_f64(ma_lcg* pLCG)
{
    return ma_lcg_rand_s32(pLCG) / (double)0x7FFFFFFF;
}

static MA_INLINE float ma_lcg_rand_f32(ma_lcg* pLCG)
{
    return (float)ma_lcg_rand_f64(pLCG);
}

static MA_INLINE float ma_lcg_rand_range_f32(ma_lcg* pLCG, float lo, float hi)
{
    return ma_scale_to_range_f32(ma_lcg_rand_f32(pLCG), lo, hi);
}

static MA_INLINE ma_int32 ma_lcg_rand_range_s32(ma_lcg* pLCG, ma_int32 lo, ma_int32 hi)
{
    if (lo == hi) {
        return lo;
    }

    return lo + ma_lcg_rand_u32(pLCG) / (0xFFFFFFFF / (hi - lo + 1) + 1);
}



static MA_INLINE void ma_seed(ma_int32 seed)
{
    ma_lcg_seed(&g_maLCG, seed);
}

static MA_INLINE ma_int32 ma_rand_s32(void)
{
    return ma_lcg_rand_s32(&g_maLCG);
}

static MA_INLINE ma_uint32 ma_rand_u32(void)
{
    return ma_lcg_rand_u32(&g_maLCG);
}

static MA_INLINE double ma_rand_f64(void)
{
    return ma_lcg_rand_f64(&g_maLCG);
}

static MA_INLINE float ma_rand_f32(void)
{
    return ma_lcg_rand_f32(&g_maLCG);
}

static MA_INLINE float ma_rand_range_f32(float lo, float hi)
{
    return ma_lcg_rand_range_f32(&g_maLCG, lo, hi);
}

static MA_INLINE ma_int32 ma_rand_range_s32(ma_int32 lo, ma_int32 hi)
{
    return ma_lcg_rand_range_s32(&g_maLCG, lo, hi);
}


static MA_INLINE float ma_dither_f32_rectangle(float ditherMin, float ditherMax)
{
    return ma_rand_range_f32(ditherMin, ditherMax);
}

static MA_INLINE float ma_dither_f32_triangle(float ditherMin, float ditherMax)
{
    float a = ma_rand_range_f32(ditherMin, 0);
    float b = ma_rand_range_f32(0, ditherMax);
    return a + b;
}

static MA_INLINE float ma_dither_f32(ma_dither_mode ditherMode, float ditherMin, float ditherMax)
{
    if (ditherMode == ma_dither_mode_rectangle) {
        return ma_dither_f32_rectangle(ditherMin, ditherMax);
    }
    if (ditherMode == ma_dither_mode_triangle) {
        return ma_dither_f32_triangle(ditherMin, ditherMax);
    }

    return 0;
}

static MA_INLINE ma_int32 ma_dither_s32(ma_dither_mode ditherMode, ma_int32 ditherMin, ma_int32 ditherMax)
{
    if (ditherMode == ma_dither_mode_rectangle) {
        ma_int32 a = ma_rand_range_s32(ditherMin, ditherMax);
        return a;
    }
    if (ditherMode == ma_dither_mode_triangle) {
        ma_int32 a = ma_rand_range_s32(ditherMin, 0);
        ma_int32 b = ma_rand_range_s32(0, ditherMax);
        return a + b;
    }

    return 0;
}


/**************************************************************************************************************************************************************

Atomics

**************************************************************************************************************************************************************/
/* c89atomic.h begin */
#ifndef c89atomic_h
#define c89atomic_h
#if defined(__cplusplus)
extern "C" {
#endif
typedef   signed char           c89atomic_int8;
typedef unsigned char           c89atomic_uint8;
typedef   signed short          c89atomic_int16;
typedef unsigned short          c89atomic_uint16;
typedef   signed int            c89atomic_int32;
typedef unsigned int            c89atomic_uint32;
#if defined(_MSC_VER)
    typedef   signed __int64    c89atomic_int64;
    typedef unsigned __int64    c89atomic_uint64;
#else
    #if defined(__GNUC__)
        #pragma GCC diagnostic push
        #pragma GCC diagnostic ignored "-Wlong-long"
        #if defined(__clang__)
            #pragma GCC diagnostic ignored "-Wc++11-long-long"
        #endif
    #endif
    typedef   signed long long  c89atomic_int64;
    typedef unsigned long long  c89atomic_uint64;
    #if defined(__GNUC__)
        #pragma GCC diagnostic pop
    #endif
#endif
typedef int                     c89atomic_memory_order;
typedef unsigned char           c89atomic_bool;
typedef unsigned char           c89atomic_flag;
#if !defined(C89ATOMIC_64BIT) && !defined(C89ATOMIC_32BIT)
#ifdef _WIN32
#ifdef _WIN64
#define C89ATOMIC_64BIT
#else
#define C89ATOMIC_32BIT
#endif
#endif
#endif
#if !defined(C89ATOMIC_64BIT) && !defined(C89ATOMIC_32BIT)
#ifdef __GNUC__
#ifdef __LP64__
#define C89ATOMIC_64BIT
#else
#define C89ATOMIC_32BIT
#endif
#endif
#endif
#if !defined(C89ATOMIC_64BIT) && !defined(C89ATOMIC_32BIT)
#include <stdint.h>
#if INTPTR_MAX == INT64_MAX
#define C89ATOMIC_64BIT
#else
#define C89ATOMIC_32BIT
#endif
#endif
#if defined(__x86_64__) || defined(_M_X64)
#define C89ATOMIC_X64
#elif defined(__i386) || defined(_M_IX86)
#define C89ATOMIC_X86
#elif defined(__arm__) || defined(_M_ARM)
#define C89ATOMIC_ARM
#endif
#ifdef _MSC_VER
    #define C89ATOMIC_INLINE __forceinline
#elif defined(__GNUC__)
    #if defined(__STRICT_ANSI__)
        #define C89ATOMIC_INLINE __inline__ __attribute__((always_inline))
    #else
        #define C89ATOMIC_INLINE inline __attribute__((always_inline))
    #endif
#else
    #define C89ATOMIC_INLINE
#endif
#if defined(_MSC_VER)
    #define c89atomic_memory_order_relaxed  0
    #define c89atomic_memory_order_consume  1
    #define c89atomic_memory_order_acquire  2
    #define c89atomic_memory_order_release  3
    #define c89atomic_memory_order_acq_rel  4
    #define c89atomic_memory_order_seq_cst  5
    #if _MSC_VER >= 1400
        #include <intrin.h>
        #define c89atomic_exchange_explicit_8( dst, src, order) (c89atomic_uint8 )_InterlockedExchange8 ((volatile char* )dst, (char )src)
        #define c89atomic_exchange_explicit_16(dst, src, order) (c89atomic_uint16)_InterlockedExchange16((volatile short*)dst, (short)src)
        #define c89atomic_exchange_explicit_32(dst, src, order) (c89atomic_uint32)_InterlockedExchange  ((volatile long* )dst, (long )src)
    #if defined(C89ATOMIC_64BIT)
        #define c89atomic_exchange_explicit_64(dst, src, order) (c89atomic_uint64)_InterlockedExchange64((volatile long long*)dst, (long long)src)
    #endif
        #define c89atomic_fetch_add_explicit_8( dst, src, order) (c89atomic_uint8 )_InterlockedExchangeAdd8 ((volatile char* )dst, (char )src)
        #define c89atomic_fetch_add_explicit_16(dst, src, order) (c89atomic_uint16)_InterlockedExchangeAdd16((volatile short*)dst, (short)src)
        #define c89atomic_fetch_add_explicit_32(dst, src, order) (c89atomic_uint32)_InterlockedExchangeAdd  ((volatile long* )dst, (long )src)
    #if defined(C89ATOMIC_64BIT)
        #define c89atomic_fetch_add_explicit_64(dst, src, order) (c89atomic_uint64)_InterlockedExchangeAdd64((volatile long long*)dst, (long long)src)
    #endif
        #define c89atomic_compare_and_swap_8( dst, expected, desired) (c89atomic_uint8 )_InterlockedCompareExchange8 ((volatile char*     )dst, (char     )desired, (char     )expected)
        #define c89atomic_compare_and_swap_16(dst, expected, desired) (c89atomic_uint16)_InterlockedCompareExchange16((volatile short*    )dst, (short    )desired, (short    )expected)
        #define c89atomic_compare_and_swap_32(dst, expected, desired) (c89atomic_uint32)_InterlockedCompareExchange  ((volatile long*     )dst, (long     )desired, (long     )expected)
        #define c89atomic_compare_and_swap_64(dst, expected, desired) (c89atomic_uint64)_InterlockedCompareExchange64((volatile long long*)dst, (long long)desired, (long long)expected)
    #if defined(C89ATOMIC_X64)
        #define c89atomic_thread_fence(order)   __faststorefence()
    #else
        static C89ATOMIC_INLINE void c89atomic_thread_fence(c89atomic_memory_order order)
        {
            volatile c89atomic_uint32 barrier = 0;
            (void)order;
            c89atomic_fetch_add_explicit_32(&barrier, 0, order);
        }
    #endif
    #else
        #if defined(__i386) || defined(_M_IX86)
            static C89ATOMIC_INLINE void __stdcall c89atomic_thread_fence(int order)
            {
                volatile c89atomic_uint32 barrier;
                (void)order;
                __asm {
                    xchg barrier, eax
                }
            }
            static C89ATOMIC_INLINE c89atomic_uint8 __stdcall c89atomic_exchange_explicit_8(volatile c89atomic_uint8* dst, c89atomic_uint8 src, int order)
            {
                (void)order;
                __asm {
                    mov ecx, dst
                    mov al,  src
                    lock xchg [ecx], al
                }
            }
            static C89ATOMIC_INLINE c89atomic_uint16 __stdcall c89atomic_exchange_explicit_16(volatile c89atomic_uint16* dst, c89atomic_uint16 src, int order)
            {
                (void)order;
                __asm {
                    mov ecx, dst
                    mov ax,  src
                    lock xchg [ecx], ax
                }
            }
            static C89ATOMIC_INLINE c89atomic_uint32 __stdcall c89atomic_exchange_explicit_32(volatile c89atomic_uint32* dst, c89atomic_uint32 src, int order)
            {
                (void)order;
                __asm {
                    mov ecx, dst
                    mov eax, src
                    lock xchg [ecx], eax
                }
            }
            static C89ATOMIC_INLINE c89atomic_uint8 __stdcall c89atomic_fetch_add_explicit_8(volatile c89atomic_uint8* dst, c89atomic_uint8 src, int order)
            {
                (void)order;
                __asm {
                    mov ecx, dst
                    mov al,  src
                    lock xadd [ecx], al
                }
            }
            static C89ATOMIC_INLINE c89atomic_uint16 __stdcall c89atomic_fetch_add_explicit_16(volatile c89atomic_uint16* dst, c89atomic_uint16 src, int order)
            {
                (void)order;
                __asm {
                    mov ecx, dst
                    mov ax,  src
                    lock xadd [ecx], ax
                }
            }
            static C89ATOMIC_INLINE c89atomic_uint32 __stdcall c89atomic_fetch_add_explicit_32(volatile c89atomic_uint32* dst, c89atomic_uint32 src, int order)
            {
                (void)order;
                __asm {
                    mov ecx, dst
                    mov eax, src
                    lock xadd [ecx], eax
                }
            }
            static C89ATOMIC_INLINE c89atomic_uint8 __stdcall c89atomic_compare_and_swap_8(volatile c89atomic_uint8* dst, c89atomic_uint8 expected, c89atomic_uint8 desired)
            {
                __asm {
                    mov ecx, dst
                    mov al,  expected
                    mov dl,  desired
                    lock cmpxchg [ecx], dl
                }
            }
            static C89ATOMIC_INLINE c89atomic_uint16 __stdcall c89atomic_compare_and_swap_16(volatile c89atomic_uint16* dst, c89atomic_uint16 expected, c89atomic_uint16 desired)
            {
                __asm {
                    mov ecx, dst
                    mov ax,  expected
                    mov dx,  desired
                    lock cmpxchg [ecx], dx
                }
            }
            static C89ATOMIC_INLINE c89atomic_uint32 __stdcall c89atomic_compare_and_swap_32(volatile c89atomic_uint32* dst, c89atomic_uint32 expected, c89atomic_uint32 desired)
            {
                __asm {
                    mov ecx, dst
                    mov eax, expected
                    mov edx, desired
                    lock cmpxchg [ecx], edx
                }
            }
            static C89ATOMIC_INLINE c89atomic_uint64 __stdcall c89atomic_compare_and_swap_64(volatile c89atomic_uint64* dst, c89atomic_uint64 expected, c89atomic_uint64 desired)
            {
                __asm {
                    mov esi, dst
                    mov eax, dword ptr expected
                    mov edx, dword ptr expected + 4
                    mov ebx, dword ptr desired
                    mov ecx, dword ptr desired + 4
                    lock cmpxchg8b qword ptr [esi]
                }
            }
        #else
            error "Unsupported architecture."
        #endif
    #endif
    #define c89atomic_compiler_fence()      c89atomic_thread_fence(c89atomic_memory_order_seq_cst)
    #define c89atomic_signal_fence(order)   c89atomic_thread_fence(order)
    #define c89atomic_load_explicit_8( ptr, order) c89atomic_compare_and_swap_8 (ptr, 0, 0)
    #define c89atomic_load_explicit_16(ptr, order) c89atomic_compare_and_swap_16(ptr, 0, 0)
    #define c89atomic_load_explicit_32(ptr, order) c89atomic_compare_and_swap_32(ptr, 0, 0)
    #define c89atomic_load_explicit_64(ptr, order) c89atomic_compare_and_swap_64(ptr, 0, 0)
    #define c89atomic_store_explicit_8( dst, src, order) (void)c89atomic_exchange_explicit_8 (dst, src, order)
    #define c89atomic_store_explicit_16(dst, src, order) (void)c89atomic_exchange_explicit_16(dst, src, order)
    #define c89atomic_store_explicit_32(dst, src, order) (void)c89atomic_exchange_explicit_32(dst, src, order)
    #define c89atomic_store_explicit_64(dst, src, order) (void)c89atomic_exchange_explicit_64(dst, src, order)
#if defined(C89ATOMIC_32BIT)
    static C89ATOMIC_INLINE c89atomic_uint64 __stdcall c89atomic_exchange_explicit_64(volatile c89atomic_uint64* dst, c89atomic_uint64 src, int order)
    {
        volatile c89atomic_uint64 oldValue;
        do {
            oldValue = *dst;
        } while (c89atomic_compare_and_swap_64(dst, oldValue, src) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint64 __stdcall c89atomic_fetch_add_explicit_64(volatile c89atomic_uint64* dst, c89atomic_uint64 src, int order)
    {
        volatile c89atomic_uint64 oldValue;
        volatile c89atomic_uint64 newValue;
        do {
            oldValue = *dst;
            newValue = oldValue + src;
        } while (c89atomic_compare_and_swap_64(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
#endif
    static C89ATOMIC_INLINE c89atomic_uint8 __stdcall c89atomic_fetch_sub_explicit_8(volatile c89atomic_uint8* dst, c89atomic_uint8 src, int order)
    {
        volatile c89atomic_uint8 oldValue;
        volatile c89atomic_uint8 newValue;
        do {
            oldValue = *dst;
            newValue = (c89atomic_uint8)(oldValue - src);
        } while (c89atomic_compare_and_swap_8(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint16 __stdcall c89atomic_fetch_sub_explicit_16(volatile c89atomic_uint16* dst, c89atomic_uint16 src, int order)
    {
        volatile c89atomic_uint16 oldValue;
        volatile c89atomic_uint16 newValue;
        do {
            oldValue = *dst;
            newValue = (c89atomic_uint16)(oldValue - src);
        } while (c89atomic_compare_and_swap_16(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint32 __stdcall c89atomic_fetch_sub_explicit_32(volatile c89atomic_uint32* dst, c89atomic_uint32 src, int order)
    {
        volatile c89atomic_uint32 oldValue;
        volatile c89atomic_uint32 newValue;
        do {
            oldValue = *dst;
            newValue = oldValue - src;
        } while (c89atomic_compare_and_swap_32(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint64 __stdcall c89atomic_fetch_sub_explicit_64(volatile c89atomic_uint64* dst, c89atomic_uint64 src, int order)
    {
        volatile c89atomic_uint64 oldValue;
        volatile c89atomic_uint64 newValue;
        do {
            oldValue = *dst;
            newValue = oldValue - src;
        } while (c89atomic_compare_and_swap_64(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint8 __stdcall c89atomic_fetch_and_explicit_8(volatile c89atomic_uint8* dst, c89atomic_uint8 src, int order)
    {
        volatile c89atomic_uint8 oldValue;
        volatile c89atomic_uint8 newValue;
        do {
            oldValue = *dst;
            newValue = (c89atomic_uint8)(oldValue & src);
        } while (c89atomic_compare_and_swap_8(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint16 __stdcall c89atomic_fetch_and_explicit_16(volatile c89atomic_uint16* dst, c89atomic_uint16 src, int order)
    {
        volatile c89atomic_uint16 oldValue;
        volatile c89atomic_uint16 newValue;
        do {
            oldValue = *dst;
            newValue = (c89atomic_uint16)(oldValue & src);
        } while (c89atomic_compare_and_swap_16(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint32 __stdcall c89atomic_fetch_and_explicit_32(volatile c89atomic_uint32* dst, c89atomic_uint32 src, int order)
    {
        volatile c89atomic_uint32 oldValue;
        volatile c89atomic_uint32 newValue;
        do {
            oldValue = *dst;
            newValue = oldValue & src;
        } while (c89atomic_compare_and_swap_32(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint64 __stdcall c89atomic_fetch_and_explicit_64(volatile c89atomic_uint64* dst, c89atomic_uint64 src, int order)
    {
        volatile c89atomic_uint64 oldValue;
        volatile c89atomic_uint64 newValue;
        do {
            oldValue = *dst;
            newValue = oldValue & src;
        } while (c89atomic_compare_and_swap_64(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint8 __stdcall c89atomic_fetch_xor_explicit_8(volatile c89atomic_uint8* dst, c89atomic_uint8 src, int order)
    {
        volatile c89atomic_uint8 oldValue;
        volatile c89atomic_uint8 newValue;
        do {
            oldValue = *dst;
            newValue = (c89atomic_uint8)(oldValue ^ src);
        } while (c89atomic_compare_and_swap_8(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint16 __stdcall c89atomic_fetch_xor_explicit_16(volatile c89atomic_uint16* dst, c89atomic_uint16 src, int order)
    {
        volatile c89atomic_uint16 oldValue;
        volatile c89atomic_uint16 newValue;
        do {
            oldValue = *dst;
            newValue = (c89atomic_uint16)(oldValue ^ src);
        } while (c89atomic_compare_and_swap_16(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint32 __stdcall c89atomic_fetch_xor_explicit_32(volatile c89atomic_uint32* dst, c89atomic_uint32 src, int order)
    {
        volatile c89atomic_uint32 oldValue;
        volatile c89atomic_uint32 newValue;
        do {
            oldValue = *dst;
            newValue = oldValue ^ src;
        } while (c89atomic_compare_and_swap_32(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint64 __stdcall c89atomic_fetch_xor_explicit_64(volatile c89atomic_uint64* dst, c89atomic_uint64 src, int order)
    {
        volatile c89atomic_uint64 oldValue;
        volatile c89atomic_uint64 newValue;
        do {
            oldValue = *dst;
            newValue = oldValue ^ src;
        } while (c89atomic_compare_and_swap_64(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint8 __stdcall c89atomic_fetch_or_explicit_8(volatile c89atomic_uint8* dst, c89atomic_uint8 src, int order)
    {
        volatile c89atomic_uint8 oldValue;
        volatile c89atomic_uint8 newValue;
        do {
            oldValue = *dst;
            newValue = (c89atomic_uint8)(oldValue | src);
        } while (c89atomic_compare_and_swap_8(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint16 __stdcall c89atomic_fetch_or_explicit_16(volatile c89atomic_uint16* dst, c89atomic_uint16 src, int order)
    {
        volatile c89atomic_uint16 oldValue;
        volatile c89atomic_uint16 newValue;
        do {
            oldValue = *dst;
            newValue = (c89atomic_uint16)(oldValue | src);
        } while (c89atomic_compare_and_swap_16(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint32 __stdcall c89atomic_fetch_or_explicit_32(volatile c89atomic_uint32* dst, c89atomic_uint32 src, int order)
    {
        volatile c89atomic_uint32 oldValue;
        volatile c89atomic_uint32 newValue;
        do {
            oldValue = *dst;
            newValue = oldValue | src;
        } while (c89atomic_compare_and_swap_32(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint64 __stdcall c89atomic_fetch_or_explicit_64(volatile c89atomic_uint64* dst, c89atomic_uint64 src, int order)
    {
        volatile c89atomic_uint64 oldValue;
        volatile c89atomic_uint64 newValue;
        do {
            oldValue = *dst;
            newValue = oldValue | src;
        } while (c89atomic_compare_and_swap_64(dst, oldValue, newValue) != oldValue);
        (void)order;
        return oldValue;
    }
    #define c89atomic_test_and_set_explicit_8( dst, order) c89atomic_exchange_explicit_8 (dst, 1, order)
    #define c89atomic_test_and_set_explicit_16(dst, order) c89atomic_exchange_explicit_16(dst, 1, order)
    #define c89atomic_test_and_set_explicit_32(dst, order) c89atomic_exchange_explicit_32(dst, 1, order)
    #define c89atomic_test_and_set_explicit_64(dst, order) c89atomic_exchange_explicit_64(dst, 1, order)
    #define c89atomic_clear_explicit_8( dst, order) c89atomic_store_explicit_8 (dst, 0, order)
    #define c89atomic_clear_explicit_16(dst, order) c89atomic_store_explicit_16(dst, 0, order)
    #define c89atomic_clear_explicit_32(dst, order) c89atomic_store_explicit_32(dst, 0, order)
    #define c89atomic_clear_explicit_64(dst, order) c89atomic_store_explicit_64(dst, 0, order)
    #define c89atomic_flag_test_and_set_explicit(ptr, order)    (c89atomic_flag)c89atomic_test_and_set_explicit_8(ptr, order)
    #define c89atomic_flag_clear_explicit(ptr, order)           c89atomic_clear_explicit_8(ptr, order)
#elif defined(__clang__) || (defined(__GNUC__) && (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC__ >= 7)))
    #define C89ATOMIC_HAS_NATIVE_COMPARE_EXCHANGE
    #define C89ATOMIC_HAS_NATIVE_IS_LOCK_FREE
    #define c89atomic_memory_order_relaxed  __ATOMIC_RELAXED
    #define c89atomic_memory_order_consume  __ATOMIC_CONSUME
    #define c89atomic_memory_order_acquire  __ATOMIC_ACQUIRE
    #define c89atomic_memory_order_release  __ATOMIC_RELEASE
    #define c89atomic_memory_order_acq_rel  __ATOMIC_ACQ_REL
    #define c89atomic_memory_order_seq_cst  __ATOMIC_SEQ_CST
    #define c89atomic_compiler_fence()      __asm__ __volatile__("":::"memory")
    #define c89atomic_thread_fence(order)   __atomic_thread_fence(order)
    #define c89atomic_signal_fence(order)   __atomic_signal_fence(order)
    #define c89atomic_is_lock_free_8(ptr)   __atomic_is_lock_free(1, ptr)
    #define c89atomic_is_lock_free_16(ptr)  __atomic_is_lock_free(2, ptr)
    #define c89atomic_is_lock_free_32(ptr)  __atomic_is_lock_free(4, ptr)
    #define c89atomic_is_lock_free_64(ptr)  __atomic_is_lock_free(8, ptr)
    #define c89atomic_flag_test_and_set_explicit(dst, order)        (c89atomic_flag)__atomic_test_and_set(dst, order)
    #define c89atomic_flag_clear_explicit(dst, order)               __atomic_clear(dst, order)
    #define c89atomic_test_and_set_explicit_8( dst, order)          __atomic_exchange_n(dst, 1, order)
    #define c89atomic_test_and_set_explicit_16(dst, order)          __atomic_exchange_n(dst, 1, order)
    #define c89atomic_test_and_set_explicit_32(dst, order)          __atomic_exchange_n(dst, 1, order)
    #define c89atomic_test_and_set_explicit_64(dst, order)          __atomic_exchange_n(dst, 1, order)
    #define c89atomic_clear_explicit_8( dst, order)                 __atomic_store_n(dst, 0, order)
    #define c89atomic_clear_explicit_16(dst, order)                 __atomic_store_n(dst, 0, order)
    #define c89atomic_clear_explicit_32(dst, order)                 __atomic_store_n(dst, 0, order)
    #define c89atomic_clear_explicit_64(dst, order)                 __atomic_store_n(dst, 0, order)
    #define c89atomic_store_explicit_8( dst, src, order)            __atomic_store_n(dst, src, order)
    #define c89atomic_store_explicit_16(dst, src, order)            __atomic_store_n(dst, src, order)
    #define c89atomic_store_explicit_32(dst, src, order)            __atomic_store_n(dst, src, order)
    #define c89atomic_store_explicit_64(dst, src, order)            __atomic_store_n(dst, src, order)
    #define c89atomic_load_explicit_8( dst, order)                  __atomic_load_n(dst, order)
    #define c89atomic_load_explicit_16(dst, order)                  __atomic_load_n(dst, order)
    #define c89atomic_load_explicit_32(dst, order)                  __atomic_load_n(dst, order)
    #define c89atomic_load_explicit_64(dst, order)                  __atomic_load_n(dst, order)
    #define c89atomic_exchange_explicit_8( dst, src, order)         __atomic_exchange_n(dst, src, order)
    #define c89atomic_exchange_explicit_16(dst, src, order)         __atomic_exchange_n(dst, src, order)
    #define c89atomic_exchange_explicit_32(dst, src, order)         __atomic_exchange_n(dst, src, order)
    #define c89atomic_exchange_explicit_64(dst, src, order)         __atomic_exchange_n(dst, src, order)
    #define c89atomic_compare_exchange_strong_explicit_8( dst, expected, desired, successOrder, failureOrder)   __atomic_compare_exchange_n(dst, expected, desired, 0, successOrder, failureOrder)
    #define c89atomic_compare_exchange_strong_explicit_16(dst, expected, desired, successOrder, failureOrder)   __atomic_compare_exchange_n(dst, expected, desired, 0, successOrder, failureOrder)
    #define c89atomic_compare_exchange_strong_explicit_32(dst, expected, desired, successOrder, failureOrder)   __atomic_compare_exchange_n(dst, expected, desired, 0, successOrder, failureOrder)
    #define c89atomic_compare_exchange_strong_explicit_64(dst, expected, desired, successOrder, failureOrder)   __atomic_compare_exchange_n(dst, expected, desired, 0, successOrder, failureOrder)
    #define c89atomic_compare_exchange_weak_explicit_8( dst, expected, desired, successOrder, failureOrder)     __atomic_compare_exchange_n(dst, expected, desired, 1, successOrder, failureOrder)
    #define c89atomic_compare_exchange_weak_explicit_16(dst, expected, desired, successOrder, failureOrder)     __atomic_compare_exchange_n(dst, expected, desired, 1, successOrder, failureOrder)
    #define c89atomic_compare_exchange_weak_explicit_32(dst, expected, desired, successOrder, failureOrder)     __atomic_compare_exchange_n(dst, expected, desired, 1, successOrder, failureOrder)
    #define c89atomic_compare_exchange_weak_explicit_64(dst, expected, desired, successOrder, failureOrder)     __atomic_compare_exchange_n(dst, expected, desired, 1, successOrder, failureOrder)
    #define c89atomic_fetch_add_explicit_8( dst, src, order)        __atomic_fetch_add(dst, src, order)
    #define c89atomic_fetch_add_explicit_16(dst, src, order)        __atomic_fetch_add(dst, src, order)
    #define c89atomic_fetch_add_explicit_32(dst, src, order)        __atomic_fetch_add(dst, src, order)
    #define c89atomic_fetch_add_explicit_64(dst, src, order)        __atomic_fetch_add(dst, src, order)
    #define c89atomic_fetch_sub_explicit_8( dst, src, order)        __atomic_fetch_sub(dst, src, order)
    #define c89atomic_fetch_sub_explicit_16(dst, src, order)        __atomic_fetch_sub(dst, src, order)
    #define c89atomic_fetch_sub_explicit_32(dst, src, order)        __atomic_fetch_sub(dst, src, order)
    #define c89atomic_fetch_sub_explicit_64(dst, src, order)        __atomic_fetch_sub(dst, src, order)
    #define c89atomic_fetch_or_explicit_8( dst, src, order)         __atomic_fetch_or(dst, src, order)
    #define c89atomic_fetch_or_explicit_16(dst, src, order)         __atomic_fetch_or(dst, src, order)
    #define c89atomic_fetch_or_explicit_32(dst, src, order)         __atomic_fetch_or(dst, src, order)
    #define c89atomic_fetch_or_explicit_64(dst, src, order)         __atomic_fetch_or(dst, src, order)
    #define c89atomic_fetch_xor_explicit_8( dst, src, order)        __atomic_fetch_xor(dst, src, order)
    #define c89atomic_fetch_xor_explicit_16(dst, src, order)        __atomic_fetch_xor(dst, src, order)
    #define c89atomic_fetch_xor_explicit_32(dst, src, order)        __atomic_fetch_xor(dst, src, order)
    #define c89atomic_fetch_xor_explicit_64(dst, src, order)        __atomic_fetch_xor(dst, src, order)
    #define c89atomic_fetch_and_explicit_8( dst, src, order)        __atomic_fetch_and(dst, src, order)
    #define c89atomic_fetch_and_explicit_16(dst, src, order)        __atomic_fetch_and(dst, src, order)
    #define c89atomic_fetch_and_explicit_32(dst, src, order)        __atomic_fetch_and(dst, src, order)
    #define c89atomic_fetch_and_explicit_64(dst, src, order)        __atomic_fetch_and(dst, src, order)
    #define c89atomic_compare_and_swap_8 (dst, expected, desired)   __sync_val_compare_and_swap(dst, expected, desired)
    #define c89atomic_compare_and_swap_16(dst, expected, desired)   __sync_val_compare_and_swap(dst, expected, desired)
    #define c89atomic_compare_and_swap_32(dst, expected, desired)   __sync_val_compare_and_swap(dst, expected, desired)
    #define c89atomic_compare_and_swap_64(dst, expected, desired)   __sync_val_compare_and_swap(dst, expected, desired)
#else
    #define c89atomic_memory_order_relaxed  1
    #define c89atomic_memory_order_consume  2
    #define c89atomic_memory_order_acquire  3
    #define c89atomic_memory_order_release  4
    #define c89atomic_memory_order_acq_rel  5
    #define c89atomic_memory_order_seq_cst  6
    #define c89atomic_compiler_fence()      __asm__ __volatile__("":::"memory")
    #define c89atomic_thread_fence(order)   __sync_synchronize()
    #define c89atomic_signal_fence(order)   c89atomic_thread_fence(order)
    static C89ATOMIC_INLINE c89atomic_uint8 c89atomic_exchange_explicit_8(volatile c89atomic_uint8* dst, c89atomic_uint8 src, c89atomic_memory_order order)
    {
        if (order > c89atomic_memory_order_acquire) {
            __sync_synchronize();
        }
        return __sync_lock_test_and_set(dst, src);
    }
    static C89ATOMIC_INLINE c89atomic_uint16 c89atomic_exchange_explicit_16(volatile c89atomic_uint16* dst, c89atomic_uint16 src, c89atomic_memory_order order)
    {
        volatile c89atomic_uint16 oldValue;
        do {
            oldValue = *dst;
        } while (__sync_val_compare_and_swap(dst, oldValue, src) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint32 c89atomic_exchange_explicit_32(volatile c89atomic_uint32* dst, c89atomic_uint32 src, c89atomic_memory_order order)
    {
        volatile c89atomic_uint32 oldValue;
        do {
            oldValue = *dst;
        } while (__sync_val_compare_and_swap(dst, oldValue, src) != oldValue);
        (void)order;
        return oldValue;
    }
    static C89ATOMIC_INLINE c89atomic_uint64 c89atomic_exchange_explicit_64(volatile c89atomic_uint64* dst, c89atomic_uint64 src, c89atomic_memory_order order)
    {
        volatile c89atomic_uint64 oldValue;
        do {
            oldValue = *dst;
        } while (__sync_val_compare_and_swap(dst, oldValue, src) != oldValue);
        (void)order;
        return oldValue;
    }
    #define c89atomic_fetch_add_explicit_8( dst, src, order)        __sync_fetch_and_add(dst, src)
    #define c89atomic_fetch_add_explicit_16(dst, src, order)        __sync_fetch_and_add(dst, src)
    #define c89atomic_fetch_add_explicit_32(dst, src, order)        __sync_fetch_and_add(dst, src)
    #define c89atomic_fetch_add_explicit_64(dst, src, order)        __sync_fetch_and_add(dst, src)
    #define c89atomic_fetch_sub_explicit_8( dst, src, order)        __sync_fetch_and_sub(dst, src)
    #define c89atomic_fetch_sub_explicit_16(dst, src, order)        __sync_fetch_and_sub(dst, src)
    #define c89atomic_fetch_sub_explicit_32(dst, src, order)        __sync_fetch_and_sub(dst, src)
    #define c89atomic_fetch_sub_explicit_64(dst, src, order)        __sync_fetch_and_sub(dst, src)
    #define c89atomic_fetch_or_explicit_8( dst, src, order)         __sync_fetch_and_or(dst, src)
    #define c89atomic_fetch_or_explicit_16(dst, src, order)         __sync_fetch_and_or(dst, src)
    #define c89atomic_fetch_or_explicit_32(dst, src, order)         __sync_fetch_and_or(dst, src)
    #define c89atomic_fetch_or_explicit_64(dst, src, order)         __sync_fetch_and_or(dst, src)
    #define c89atomic_fetch_xor_explicit_8( dst, src, order)        __sync_fetch_and_xor(dst, src)
    #define c89atomic_fetch_xor_explicit_16(dst, src, order)        __sync_fetch_and_xor(dst, src)
    #define c89atomic_fetch_xor_explicit_32(dst, src, order)        __sync_fetch_and_xor(dst, src)
    #define c89atomic_fetch_xor_explicit_64(dst, src, order)        __sync_fetch_and_xor(dst, src)
    #define c89atomic_fetch_and_explicit_8( dst, src, order)        __sync_fetch_and_and(dst, src)
    #define c89atomic_fetch_and_explicit_16(dst, src, order)        __sync_fetch_and_and(dst, src)
    #define c89atomic_fetch_and_explicit_32(dst, src, order)        __sync_fetch_and_and(dst, src)
    #define c89atomic_fetch_and_explicit_64(dst, src, order)        __sync_fetch_and_and(dst, src)
    #define c89atomic_compare_and_swap_8( dst, expected, desired)   __sync_val_compare_and_swap(dst, expected, desired)
    #define c89atomic_compare_and_swap_16(dst, expected, desired)   __sync_val_compare_and_swap(dst, expected, desired)
    #define c89atomic_compare_and_swap_32(dst, expected, desired)   __sync_val_compare_and_swap(dst, expected, desired)
    #define c89atomic_compare_and_swap_64(dst, expected, desired)   __sync_val_compare_and_swap(dst, expected, desired)
    #define c89atomic_load_explicit_8( ptr, order)                  c89atomic_compare_and_swap_8 (ptr, 0, 0)
    #define c89atomic_load_explicit_16(ptr, order)                  c89atomic_compare_and_swap_16(ptr, 0, 0)
    #define c89atomic_load_explicit_32(ptr, order)                  c89atomic_compare_and_swap_32(ptr, 0, 0)
    #define c89atomic_load_explicit_64(ptr, order)                  c89atomic_compare_and_swap_64(ptr, 0, 0)
    #define c89atomic_store_explicit_8( dst, src, order)            (void)c89atomic_exchange_explicit_8 (dst, src, order)
    #define c89atomic_store_explicit_16(dst, src, order)            (void)c89atomic_exchange_explicit_16(dst, src, order)
    #define c89atomic_store_explicit_32(dst, src, order)            (void)c89atomic_exchange_explicit_32(dst, src, order)
    #define c89atomic_store_explicit_64(dst, src, order)            (void)c89atomic_exchange_explicit_64(dst, src, order)
    #define c89atomic_test_and_set_explicit_8( dst, order)          c89atomic_exchange_explicit_8 (dst, 1, order)
    #define c89atomic_test_and_set_explicit_16(dst, order)          c89atomic_exchange_explicit_16(dst, 1, order)
    #define c89atomic_test_and_set_explicit_32(dst, order)          c89atomic_exchange_explicit_32(dst, 1, order)
    #define c89atomic_test_and_set_explicit_64(dst, order)          c89atomic_exchange_explicit_64(dst, 1, order)
    #define c89atomic_clear_explicit_8( dst, order)                 c89atomic_store_explicit_8 (dst, 0, order)
    #define c89atomic_clear_explicit_16(dst, order)                 c89atomic_store_explicit_16(dst, 0, order)
    #define c89atomic_clear_explicit_32(dst, order)                 c89atomic_store_explicit_32(dst, 0, order)
    #define c89atomic_clear_explicit_64(dst, order)                 c89atomic_store_explicit_64(dst, 0, order)
    #define c89atomic_flag_test_and_set_explicit(ptr, order)        (c89atomic_flag)c89atomic_test_and_set_explicit_8(ptr, order)
    #define c89atomic_flag_clear_explicit(ptr, order)               c89atomic_clear_explicit_8(ptr, order)
#endif
#if !defined(C89ATOMIC_HAS_NATIVE_COMPARE_EXCHANGE)
c89atomic_bool c89atomic_compare_exchange_strong_explicit_8(volatile c89atomic_uint8* dst, volatile c89atomic_uint8* expected, c89atomic_uint8 desired, c89atomic_memory_order successOrder, c89atomic_memory_order failureOrder)
{
    c89atomic_uint8 expectedValue;
    c89atomic_uint8 result;
    (void)successOrder;
    (void)failureOrder;
    expectedValue = c89atomic_load_explicit_8(expected, c89atomic_memory_order_seq_cst);
    result = c89atomic_compare_and_swap_8(dst, expectedValue, desired);
    if (result == expectedValue) {
        return 1;
    } else {
        c89atomic_store_explicit_8(expected, result, failureOrder);
        return 0;
    }
}
c89atomic_bool c89atomic_compare_exchange_strong_explicit_16(volatile c89atomic_uint16* dst, volatile c89atomic_uint16* expected, c89atomic_uint16 desired, c89atomic_memory_order successOrder, c89atomic_memory_order failureOrder)
{
    c89atomic_uint16 expectedValue;
    c89atomic_uint16 result;
    (void)successOrder;
    (void)failureOrder;
    expectedValue = c89atomic_load_explicit_16(expected, c89atomic_memory_order_seq_cst);
    result = c89atomic_compare_and_swap_16(dst, expectedValue, desired);
    if (result == expectedValue) {
        return 1;
    } else {
        c89atomic_store_explicit_16(expected, result, failureOrder);
        return 0;
    }
}
c89atomic_bool c89atomic_compare_exchange_strong_explicit_32(volatile c89atomic_uint32* dst, volatile c89atomic_uint32* expected, c89atomic_uint32 desired, c89atomic_memory_order successOrder, c89atomic_memory_order failureOrder)
{
    c89atomic_uint32 expectedValue;
    c89atomic_uint32 result;
    (void)successOrder;
    (void)failureOrder;
    expectedValue = c89atomic_load_explicit_32(expected, c89atomic_memory_order_seq_cst);
    result = c89atomic_compare_and_swap_32(dst, expectedValue, desired);
    if (result == expectedValue) {
        return 1;
    } else {
        c89atomic_store_explicit_32(expected, result, failureOrder);
        return 0;
    }
}
c89atomic_bool c89atomic_compare_exchange_strong_explicit_64(volatile c89atomic_uint64* dst, volatile c89atomic_uint64* expected, c89atomic_uint64 desired, c89atomic_memory_order successOrder, c89atomic_memory_order failureOrder)
{
    c89atomic_uint64 expectedValue;
    c89atomic_uint64 result;
    (void)successOrder;
    (void)failureOrder;
    expectedValue = c89atomic_load_explicit_64(expected, c89atomic_memory_order_seq_cst);
    result = c89atomic_compare_and_swap_64(dst, expectedValue, desired);
    if (result == expectedValue) {
        return 1;
    } else {
        c89atomic_store_explicit_64(expected, result, failureOrder);
        return 0;
    }
}
#define c89atomic_compare_exchange_weak_explicit_8( dst, expected, desired, successOrder, failureOrder) c89atomic_compare_exchange_strong_explicit_8 (dst, expected, desired, successOrder, failureOrder)
#define c89atomic_compare_exchange_weak_explicit_16(dst, expected, desired, successOrder, failureOrder) c89atomic_compare_exchange_strong_explicit_16(dst, expected, desired, successOrder, failureOrder)
#define c89atomic_compare_exchange_weak_explicit_32(dst, expected, desired, successOrder, failureOrder) c89atomic_compare_exchange_strong_explicit_32(dst, expected, desired, successOrder, failureOrder)
#define c89atomic_compare_exchange_weak_explicit_64(dst, expected, desired, successOrder, failureOrder) c89atomic_compare_exchange_strong_explicit_64(dst, expected, desired, successOrder, failureOrder)
#endif
#if !defined(C89ATOMIC_HAS_NATIVE_IS_LOCK_FREE)
    #define c89atomic_is_lock_free_8( ptr)  1
    #define c89atomic_is_lock_free_16(ptr)  1
    #define c89atomic_is_lock_free_32(ptr)  1
    #if defined(C89ATOMIC_64BIT)
        #define c89atomic_is_lock_free_64(ptr)  1
    #else
        #if defined(C89ATOMIC_X86) || defined(C89ATOMIC_X64)
            #define c89atomic_is_lock_free_64(ptr)  1
        #else
            #define c89atomic_is_lock_free_64(ptr)  0
        #endif
    #endif
#endif
#if defined(C89ATOMIC_64BIT)
    #define c89atomic_is_lock_free_ptr(ptr)                                                                     c89atomic_is_lock_free_64((volatile c89atomic_uint64*)ptr)
    #define c89atomic_load_explicit_ptr(ptr, order)                                                             (void*)c89atomic_load_explicit_64((volatile c89atomic_uint64*)ptr, order)
    #define c89atomic_store_explicit_ptr(dst, src, order)                                                       (void*)c89atomic_store_explicit_64((volatile c89atomic_uint64*)dst, (c89atomic_uint64)src, order)
    #define c89atomic_exchange_explicit_ptr(dst, src, order)                                                    (void*)c89atomic_exchange_explicit_64((volatile c89atomic_uint64*)dst, (c89atomic_uint64)src, order)
    #define c89atomic_compare_exchange_strong_explicit_ptr(dst, expected, desired, successOrder, failureOrder)  c89atomic_compare_exchange_strong_explicit_64((volatile c89atomic_uint64*)dst, (volatile c89atomic_uint64*)expected, (c89atomic_uint64)desired, successOrder, failureOrder)
    #define c89atomic_compare_exchange_weak_explicit_ptr(dst, expected, desired, successOrder, failureOrder)    c89atomic_compare_exchange_weak_explicit_64((volatile c89atomic_uint64*)dst, (volatile c89atomic_uint64*)expected, (c89atomic_uint64)desired, successOrder, failureOrder)
    #define c89atomic_compare_and_swap_ptr(dst, expected, desired)                                              (void*)c89atomic_compare_and_swap_64 ((volatile c89atomic_uint64*)dst, (c89atomic_uint64)expected, (c89atomic_uint64)desired)
#elif defined(C89ATOMIC_32BIT)
    #define c89atomic_is_lock_free_ptr(ptr)                                                                     c89atomic_is_lock_free_32((volatile c89atomic_uint32*)ptr)
    #define c89atomic_load_explicit_ptr(ptr, order)                                                             (void*)c89atomic_load_explicit_32((volatile c89atomic_uint32*)ptr, order)
    #define c89atomic_store_explicit_ptr(dst, src, order)                                                       (void*)c89atomic_store_explicit_32((volatile c89atomic_uint32*)dst, (c89atomic_uint32)src, order)
    #define c89atomic_exchange_explicit_ptr(dst, src, order)                                                    (void*)c89atomic_exchange_explicit_32((volatile c89atomic_uint32*)dst, (c89atomic_uint32)src, order)
    #define c89atomic_compare_exchange_strong_explicit_ptr(dst, expected, desired, successOrder, failureOrder)  c89atomic_compare_exchange_strong_explicit_32((volatile c89atomic_uint32*)dst, (volatile c89atomic_uint32*)expected, (c89atomic_uint32)desired, successOrder, failureOrder)
    #define c89atomic_compare_exchange_weak_explicit_ptr(dst, expected, desired, successOrder, failureOrder)    c89atomic_compare_exchange_weak_explicit_32((volatile c89atomic_uint32*)dst, (volatile c89atomic_uint32*)expected, (c89atomic_uint32)desired, successOrder, failureOrder)
    #define c89atomic_compare_and_swap_ptr(dst, expected, desired)                                              (void*)c89atomic_compare_and_swap_32((volatile c89atomic_uint32*)dst, (c89atomic_uint32)expected, (c89atomic_uint32)desired)
#else
    error "Unsupported architecture."
#endif
#define c89atomic_flag_test_and_set(ptr)                                c89atomic_flag_test_and_set_explicit(ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_flag_clear(ptr)                                       c89atomic_flag_clear_explicit(ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_test_and_set_8( ptr)                                  c89atomic_test_and_set_explicit_8 (ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_test_and_set_16(ptr)                                  c89atomic_test_and_set_explicit_16(ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_test_and_set_32(ptr)                                  c89atomic_test_and_set_explicit_32(ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_test_and_set_64(ptr)                                  c89atomic_test_and_set_explicit_64(ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_clear_8( ptr)                                         c89atomic_clear_explicit_8 (ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_clear_16(ptr)                                         c89atomic_clear_explicit_16(ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_clear_32(ptr)                                         c89atomic_clear_explicit_32(ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_clear_64(ptr)                                         c89atomic_clear_explicit_64(ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_store_8(  dst, src)                                   c89atomic_store_explicit_8 ( dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_store_16( dst, src)                                   c89atomic_store_explicit_16( dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_store_32( dst, src)                                   c89atomic_store_explicit_32( dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_store_64( dst, src)                                   c89atomic_store_explicit_64( dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_store_ptr(dst, src)                                   c89atomic_store_explicit_ptr(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_load_8(  ptr)                                         c89atomic_load_explicit_8 ( ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_load_16( ptr)                                         c89atomic_load_explicit_16( ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_load_32( ptr)                                         c89atomic_load_explicit_32( ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_load_64( ptr)                                         c89atomic_load_explicit_64( ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_load_ptr(ptr)                                         c89atomic_load_explicit_ptr(ptr, c89atomic_memory_order_seq_cst)
#define c89atomic_exchange_8(  dst, src)                                c89atomic_exchange_explicit_8 ( dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_exchange_16( dst, src)                                c89atomic_exchange_explicit_16( dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_exchange_32( dst, src)                                c89atomic_exchange_explicit_32( dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_exchange_64( dst, src)                                c89atomic_exchange_explicit_64( dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_exchange_ptr(dst, src)                                c89atomic_exchange_explicit_ptr(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_compare_exchange_strong_8(  dst, expected, desired)   c89atomic_compare_exchange_strong_explicit_8 ( dst, expected, desired, c89atomic_memory_order_seq_cst, c89atomic_memory_order_seq_cst)
#define c89atomic_compare_exchange_strong_16( dst, expected, desired)   c89atomic_compare_exchange_strong_explicit_16( dst, expected, desired, c89atomic_memory_order_seq_cst, c89atomic_memory_order_seq_cst)
#define c89atomic_compare_exchange_strong_32( dst, expected, desired)   c89atomic_compare_exchange_strong_explicit_32( dst, expected, desired, c89atomic_memory_order_seq_cst, c89atomic_memory_order_seq_cst)
#define c89atomic_compare_exchange_strong_64( dst, expected, desired)   c89atomic_compare_exchange_strong_explicit_64( dst, expected, desired, c89atomic_memory_order_seq_cst, c89atomic_memory_order_seq_cst)
#define c89atomic_compare_exchange_strong_ptr(dst, expected, desired)   c89atomic_compare_exchange_strong_explicit_ptr(dst, expected, desired, c89atomic_memory_order_seq_cst, c89atomic_memory_order_seq_cst)
#define c89atomic_compare_exchange_weak_8(  dst, expected, desired)     c89atomic_compare_exchange_weak_explicit_8 ( dst, expected, desired, c89atomic_memory_order_seq_cst, c89atomic_memory_order_seq_cst)
#define c89atomic_compare_exchange_weak_16( dst, expected, desired)     c89atomic_compare_exchange_weak_explicit_16( dst, expected, desired, c89atomic_memory_order_seq_cst, c89atomic_memory_order_seq_cst)
#define c89atomic_compare_exchange_weak_32( dst, expected, desired)     c89atomic_compare_exchange_weak_explicit_32( dst, expected, desired, c89atomic_memory_order_seq_cst, c89atomic_memory_order_seq_cst)
#define c89atomic_compare_exchange_weak_64( dst, expected, desired)     c89atomic_compare_exchange_weak_explicit_64( dst, expected, desired, c89atomic_memory_order_seq_cst, c89atomic_memory_order_seq_cst)
#define c89atomic_compare_exchange_weak_ptr(dst, expected, desired)     c89atomic_compare_exchange_weak_explicit_ptr(dst, expected, desired, c89atomic_memory_order_seq_cst, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_add_8( dst, src)                                c89atomic_fetch_add_explicit_8 (dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_add_16(dst, src)                                c89atomic_fetch_add_explicit_16(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_add_32(dst, src)                                c89atomic_fetch_add_explicit_32(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_add_64(dst, src)                                c89atomic_fetch_add_explicit_64(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_sub_8( dst, src)                                c89atomic_fetch_sub_explicit_8 (dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_sub_16(dst, src)                                c89atomic_fetch_sub_explicit_16(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_sub_32(dst, src)                                c89atomic_fetch_sub_explicit_32(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_sub_64(dst, src)                                c89atomic_fetch_sub_explicit_64(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_or_8( dst, src)                                 c89atomic_fetch_or_explicit_8 (dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_or_16(dst, src)                                 c89atomic_fetch_or_explicit_16(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_or_32(dst, src)                                 c89atomic_fetch_or_explicit_32(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_or_64(dst, src)                                 c89atomic_fetch_or_explicit_64(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_xor_8( dst, src)                                c89atomic_fetch_xor_explicit_8 (dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_xor_16(dst, src)                                c89atomic_fetch_xor_explicit_16(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_xor_32(dst, src)                                c89atomic_fetch_xor_explicit_32(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_xor_64(dst, src)                                c89atomic_fetch_xor_explicit_64(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_and_8( dst, src)                                c89atomic_fetch_and_explicit_8 (dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_and_16(dst, src)                                c89atomic_fetch_and_explicit_16(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_and_32(dst, src)                                c89atomic_fetch_and_explicit_32(dst, src, c89atomic_memory_order_seq_cst)
#define c89atomic_fetch_and_64(dst, src)                                c89atomic_fetch_and_explicit_64(dst, src, c89atomic_memory_order_seq_cst)
#if defined(__cplusplus)
}
#endif
#endif
/* c89atomic.h end */



static void* ma__malloc_default(size_t sz, void* pUserData)
{
    (void)pUserData;
    return MA_MALLOC(sz);
}

static void* ma__realloc_default(void* p, size_t sz, void* pUserData)
{
    (void)pUserData;
    return MA_REALLOC(p, sz);
}

static void ma__free_default(void* p, void* pUserData)
{
    (void)pUserData;
    MA_FREE(p);
}


static void* ma__malloc_from_callbacks(size_t sz, const ma_allocation_callbacks* pAllocationCallbacks)
{
    if (pAllocationCallbacks == NULL) {
        return NULL;
    }

    if (pAllocationCallbacks->onMalloc != NULL) {
        return pAllocationCallbacks->onMalloc(sz, pAllocationCallbacks->pUserData);
    }

    /* Try using realloc(). */
    if (pAllocationCallbacks->onRealloc != NULL) {
        return pAllocationCallbacks->onRealloc(NULL, sz, pAllocationCallbacks->pUserData);
    }

    return NULL;
}

static void* ma__realloc_from_callbacks(void* p, size_t szNew, size_t szOld, const ma_allocation_callbacks* pAllocationCallbacks)
{
    if (pAllocationCallbacks == NULL) {
        return NULL;
    }

    if (pAllocationCallbacks->onRealloc != NULL) {
        return pAllocationCallbacks->onRealloc(p, szNew, pAllocationCallbacks->pUserData);
    }

    /* Try emulating realloc() in terms of malloc()/free(). */
    if (pAllocationCallbacks->onMalloc != NULL && pAllocationCallbacks->onFree != NULL) {
        void* p2;

        p2 = pAllocationCallbacks->onMalloc(szNew, pAllocationCallbacks->pUserData);
        if (p2 == NULL) {
            return NULL;
        }

        if (p != NULL) {
            MA_COPY_MEMORY(p2, p, szOld);
            pAllocationCallbacks->onFree(p, pAllocationCallbacks->pUserData);
        }

        return p2;
    }

    return NULL;
}

static MA_INLINE void* ma__calloc_from_callbacks(size_t sz, const ma_allocation_callbacks* pAllocationCallbacks)
{
    void* p = ma__malloc_from_callbacks(sz, pAllocationCallbacks);
    if (p != NULL) {
        MA_ZERO_MEMORY(p, sz);
    }

    return p;
}

static void ma__free_from_callbacks(void* p, const ma_allocation_callbacks* pAllocationCallbacks)
{
    if (p == NULL || pAllocationCallbacks == NULL) {
        return;
    }

    if (pAllocationCallbacks->onFree != NULL) {
        pAllocationCallbacks->onFree(p, pAllocationCallbacks->pUserData);
    }
}

static ma_allocation_callbacks ma_allocation_callbacks_init_default(void)
{
    ma_allocation_callbacks callbacks;
    callbacks.pUserData = NULL;
    callbacks.onMalloc  = ma__malloc_default;
    callbacks.onRealloc = ma__realloc_default;
    callbacks.onFree    = ma__free_default;

    return callbacks;
}

static ma_result ma_allocation_callbacks_init_copy(ma_allocation_callbacks* pDst, const ma_allocation_callbacks* pSrc)
{
    if (pDst == NULL) {
        return MA_INVALID_ARGS;
    }

    if (pSrc == NULL) {
        *pDst = ma_allocation_callbacks_init_default();
    } else {
        if (pSrc->pUserData == NULL && pSrc->onFree == NULL && pSrc->onMalloc == NULL && pSrc->onRealloc == NULL) {
            *pDst = ma_allocation_callbacks_init_default();
        } else {
            if (pSrc->onFree == NULL || (pSrc->onMalloc == NULL && pSrc->onRealloc == NULL)) {
                return MA_INVALID_ARGS;    /* Invalid allocation callbacks. */
            } else {
                *pDst = *pSrc;
            }
        }
    }

    return MA_SUCCESS;
}


MA_API ma_uint64 ma_calculate_frame_count_after_resampling(ma_uint32 sampleRateOut, ma_uint32 sampleRateIn, ma_uint64 frameCountIn)
{
    /* For robustness we're going to use a resampler object to calculate this since that already has a way of calculating this. */
    ma_result result;
    ma_uint64 frameCountOut;
    ma_resampler_config config;
    ma_resampler resampler;

    if (sampleRateOut == sampleRateIn) {
        return frameCountIn;
    }

    config = ma_resampler_config_init(ma_format_s16, 1, sampleRateIn, sampleRateOut, ma_resample_algorithm_linear);
    result = ma_resampler_init(&config, &resampler);
    if (result != MA_SUCCESS) {
        return 0;
    }

    frameCountOut = ma_resampler_get_expected_output_frame_count(&resampler, frameCountIn);

    ma_resampler_uninit(&resampler);
    return frameCountOut;
}

#ifndef MA_DATA_CONVERTER_STACK_BUFFER_SIZE
#define MA_DATA_CONVERTER_STACK_BUFFER_SIZE     4096
#endif



#if defined(MA_WIN32)
static ma_result ma_result_from_GetLastError(DWORD error)
{
    switch (error)
    {
        case ERROR_SUCCESS:             return MA_SUCCESS;
        case ERROR_PATH_NOT_FOUND:      return MA_DOES_NOT_EXIST;
        case ERROR_TOO_MANY_OPEN_FILES: return MA_TOO_MANY_OPEN_FILES;
        case ERROR_NOT_ENOUGH_MEMORY:   return MA_OUT_OF_MEMORY;
        case ERROR_DISK_FULL:           return MA_NO_SPACE;
        case ERROR_HANDLE_EOF:          return MA_END_OF_FILE;
        case ERROR_NEGATIVE_SEEK:       return MA_BAD_SEEK;
        case ERROR_INVALID_PARAMETER:   return MA_INVALID_ARGS;
        case ERROR_ACCESS_DENIED:       return MA_ACCESS_DENIED;
        case ERROR_SEM_TIMEOUT:         return MA_TIMEOUT;
        case ERROR_FILE_NOT_FOUND:      return MA_DOES_NOT_EXIST;
        default: break;
    }

    return MA_ERROR;
}
#endif  /* MA_WIN32 */


/*******************************************************************************

Threading

*******************************************************************************/
#ifndef MA_NO_THREADING
#ifdef MA_WIN32
    #define MA_THREADCALL WINAPI
    typedef unsigned long ma_thread_result;
#else
    #define MA_THREADCALL
    typedef void* ma_thread_result;
#endif
typedef ma_thread_result (MA_THREADCALL * ma_thread_entry_proc)(void* pData);

static MA_INLINE ma_result ma_spinlock_lock_ex(ma_spinlock* pSpinlock, ma_bool32 yield)
{
    if (pSpinlock == NULL) {
        return MA_INVALID_ARGS;
    }

    for (;;) {
        if (c89atomic_flag_test_and_set_explicit(pSpinlock, c89atomic_memory_order_acquire) == 0) {
            break;
        }

        while (c89atomic_load_explicit_8(pSpinlock, c89atomic_memory_order_relaxed) == 1) {
            if (yield) {
                ma_yield();
            }
        }
    }

    return MA_SUCCESS;
}

MA_API ma_result ma_spinlock_lock(ma_spinlock* pSpinlock)
{
    return ma_spinlock_lock_ex(pSpinlock, MA_TRUE);
}

MA_API ma_result ma_spinlock_lock_noyield(ma_spinlock* pSpinlock)
{
    return ma_spinlock_lock_ex(pSpinlock, MA_FALSE);
}

MA_API ma_result ma_spinlock_unlock(ma_spinlock* pSpinlock)
{
    if (pSpinlock == NULL) {
        return MA_INVALID_ARGS;
    }

    c89atomic_flag_clear_explicit(pSpinlock, c89atomic_memory_order_release);
    return MA_SUCCESS;
}

#ifdef MA_WIN32
static int ma_thread_priority_to_win32(ma_thread_priority priority)
{
    switch (priority) {
        case ma_thread_priority_idle:     return THREAD_PRIORITY_IDLE;
        case ma_thread_priority_lowest:   return THREAD_PRIORITY_LOWEST;
        case ma_thread_priority_low:      return THREAD_PRIORITY_BELOW_NORMAL;
        case ma_thread_priority_normal:   return THREAD_PRIORITY_NORMAL;
        case ma_thread_priority_high:     return THREAD_PRIORITY_ABOVE_NORMAL;
        case ma_thread_priority_highest:  return THREAD_PRIORITY_HIGHEST;
        case ma_thread_priority_realtime: return THREAD_PRIORITY_TIME_CRITICAL;
        default:                          return THREAD_PRIORITY_NORMAL;
    }
}

static ma_result ma_thread_create__win32(ma_thread* pThread, ma_thread_priority priority, size_t stackSize, ma_thread_entry_proc entryProc, void* pData)
{
    *pThread = CreateThread(NULL, stackSize, entryProc, pData, 0, NULL);
    if (*pThread == NULL) {
        return ma_result_from_GetLastError(GetLastError());
    }

    SetThreadPriority((HANDLE)*pThread, ma_thread_priority_to_win32(priority));

    return MA_SUCCESS;
}

static void ma_thread_wait__win32(ma_thread* pThread)
{
    WaitForSingleObject((HANDLE)*pThread, INFINITE);
}


static ma_result ma_mutex_init__win32(ma_mutex* pMutex)
{
    *pMutex = CreateEventW(NULL, FALSE, TRUE, NULL);
    if (*pMutex == NULL) {
        return ma_result_from_GetLastError(GetLastError());
    }

    return MA_SUCCESS;
}

static void ma_mutex_uninit__win32(ma_mutex* pMutex)
{
    CloseHandle((HANDLE)*pMutex);
}

static void ma_mutex_lock__win32(ma_mutex* pMutex)
{
    WaitForSingleObject((HANDLE)*pMutex, INFINITE);
}

static void ma_mutex_unlock__win32(ma_mutex* pMutex)
{
    SetEvent((HANDLE)*pMutex);
}


static ma_result ma_event_init__win32(ma_event* pEvent)
{
    *pEvent = CreateEventW(NULL, FALSE, FALSE, NULL);
    if (*pEvent == NULL) {
        return ma_result_from_GetLastError(GetLastError());
    }

    return MA_SUCCESS;
}

static void ma_event_uninit__win32(ma_event* pEvent)
{
    CloseHandle((HANDLE)*pEvent);
}

static ma_result ma_event_wait__win32(ma_event* pEvent)
{
    DWORD result = WaitForSingleObject((HANDLE)*pEvent, INFINITE);
    if (result == WAIT_OBJECT_0) {
        return MA_SUCCESS;
    }

    if (result == WAIT_TIMEOUT) {
        return MA_TIMEOUT;
    }

    return ma_result_from_GetLastError(GetLastError());
}

static ma_result ma_event_signal__win32(ma_event* pEvent)
{
    BOOL result = SetEvent((HANDLE)*pEvent);
    if (result == 0) {
        return ma_result_from_GetLastError(GetLastError());
    }

    return MA_SUCCESS;
}


static ma_result ma_semaphore_init__win32(int initialValue, ma_semaphore* pSemaphore)
{
    *pSemaphore = CreateSemaphoreW(NULL, (LONG)initialValue, LONG_MAX, NULL);
    if (*pSemaphore == NULL) {
        return ma_result_from_GetLastError(GetLastError());
    }

    return MA_SUCCESS;
}

static void ma_semaphore_uninit__win32(ma_semaphore* pSemaphore)
{
    CloseHandle((HANDLE)*pSemaphore);
}

static ma_result ma_semaphore_wait__win32(ma_semaphore* pSemaphore)
{
    DWORD result = WaitForSingleObject((HANDLE)*pSemaphore, INFINITE);
    if (result == WAIT_OBJECT_0) {
        return MA_SUCCESS;
    }

    if (result == WAIT_TIMEOUT) {
        return MA_TIMEOUT;
    }

    return ma_result_from_GetLastError(GetLastError());
}

static ma_result ma_semaphore_release__win32(ma_semaphore* pSemaphore)
{
    BOOL result = ReleaseSemaphore((HANDLE)*pSemaphore, 1, NULL);
    if (result == 0) {
        return ma_result_from_GetLastError(GetLastError());
    }

    return MA_SUCCESS;
}
#endif


#ifdef MA_POSIX
static ma_result ma_thread_create__posix(ma_thread* pThread, ma_thread_priority priority, size_t stackSize, ma_thread_entry_proc entryProc, void* pData)
{
    int result;
    pthread_attr_t* pAttr = NULL;

#if !defined(__EMSCRIPTEN__)
    /* Try setting the thread priority. It's not critical if anything fails here. */
    pthread_attr_t attr;
    if (pthread_attr_init(&attr) == 0) {
        int scheduler = -1;
        if (priority == ma_thread_priority_idle) {
#ifdef SCHED_IDLE
            if (pthread_attr_setschedpolicy(&attr, SCHED_IDLE) == 0) {
                scheduler = SCHED_IDLE;
            }
#endif
        } else if (priority == ma_thread_priority_realtime) {
#ifdef SCHED_FIFO
            if (pthread_attr_setschedpolicy(&attr, SCHED_FIFO) == 0) {
                scheduler = SCHED_FIFO;
            }
#endif
#ifdef MA_LINUX
        } else {
            scheduler = sched_getscheduler(0);
#endif
        }

        if (stackSize > 0) {
            pthread_attr_setstacksize(&attr, stackSize);
        }

        if (scheduler != -1) {
            int priorityMin = sched_get_priority_min(scheduler);
            int priorityMax = sched_get_priority_max(scheduler);
            int priorityStep = (priorityMax - priorityMin) / 7;  /* 7 = number of priorities supported by miniaudio. */

            struct sched_param sched;
            if (pthread_attr_getschedparam(&attr, &sched) == 0) {
                if (priority == ma_thread_priority_idle) {
                    sched.sched_priority = priorityMin;
                } else if (priority == ma_thread_priority_realtime) {
                    sched.sched_priority = priorityMax;
                } else {
                    sched.sched_priority += ((int)priority + 5) * priorityStep;  /* +5 because the lowest priority is -5. */
                    if (sched.sched_priority < priorityMin) {
                        sched.sched_priority = priorityMin;
                    }
                    if (sched.sched_priority > priorityMax) {
                        sched.sched_priority = priorityMax;
                    }
                }

                if (pthread_attr_setschedparam(&attr, &sched) == 0) {
                    pAttr = &attr;
                }
            }
        }

        pthread_attr_destroy(&attr);
    }
#else
    /* It's the emscripten build. We'll have a few unused parameters. */
    (void)priority;
    (void)stackSize;
#endif

    result = pthread_create(pThread, pAttr, entryProc, pData);
    if (result != 0) {
        return ma_result_from_errno(result);
    }

    return MA_SUCCESS;
}

static void ma_thread_wait__posix(ma_thread* pThread)
{
    pthread_join(*pThread, NULL);
}


static ma_result ma_mutex_init__posix(ma_mutex* pMutex)
{
    int result = pthread_mutex_init((pthread_mutex_t*)pMutex, NULL);
    if (result != 0) {
        return ma_result_from_errno(result);
    }

    return MA_SUCCESS;
}

static void ma_mutex_uninit__posix(ma_mutex* pMutex)
{
    pthread_mutex_destroy((pthread_mutex_t*)pMutex);
}

static void ma_mutex_lock__posix(ma_mutex* pMutex)
{
    pthread_mutex_lock((pthread_mutex_t*)pMutex);
}

static void ma_mutex_unlock__posix(ma_mutex* pMutex)
{
    pthread_mutex_unlock((pthread_mutex_t*)pMutex);
}


static ma_result ma_event_init__posix(ma_event* pEvent)
{
    int result;

    result = pthread_mutex_init(&pEvent->lock, NULL);
    if (result != 0) {
        return ma_result_from_errno(result);
    }

    result = pthread_cond_init(&pEvent->cond, NULL);
    if (result != 0) {
        pthread_mutex_destroy(&pEvent->lock);
        return ma_result_from_errno(result);
    }

    pEvent->value = 0;
    return MA_SUCCESS;
}

static void ma_event_uninit__posix(ma_event* pEvent)
{
    pthread_cond_destroy(&pEvent->cond);
    pthread_mutex_destroy(&pEvent->lock);
}

static ma_result ma_event_wait__posix(ma_event* pEvent)
{
    pthread_mutex_lock(&pEvent->lock);
    {
        while (pEvent->value == 0) {
            pthread_cond_wait(&pEvent->cond, &pEvent->lock);
        }
        pEvent->value = 0;  /* Auto-reset. */
    }
    pthread_mutex_unlock(&pEvent->lock);

    return MA_SUCCESS;
}

static ma_result ma_event_signal__posix(ma_event* pEvent)
{
    pthread_mutex_lock(&pEvent->lock);
    {
        pEvent->value = 1;
        pthread_cond_signal(&pEvent->cond);
    }
    pthread_mutex_unlock(&pEvent->lock);

    return MA_SUCCESS;
}


static ma_result ma_semaphore_init__posix(int initialValue, ma_semaphore* pSemaphore)
{
    int result;

    if (pSemaphore == NULL) {
        return MA_INVALID_ARGS;
    }

    pSemaphore->value = initialValue;

    result = pthread_mutex_init(&pSemaphore->lock, NULL);
    if (result != 0) {
        return ma_result_from_errno(result);  /* Failed to create mutex. */
    }

    result = pthread_cond_init(&pSemaphore->cond, NULL);
    if (result != 0) {
        pthread_mutex_destroy(&pSemaphore->lock);
        return ma_result_from_errno(result);  /* Failed to create condition variable. */
    }

    return MA_SUCCESS;
}

static void ma_semaphore_uninit__posix(ma_semaphore* pSemaphore)
{
    if (pSemaphore == NULL) {
        return;
    }

    pthread_cond_destroy(&pSemaphore->cond);
    pthread_mutex_destroy(&pSemaphore->lock);
}

static ma_result ma_semaphore_wait__posix(ma_semaphore* pSemaphore)
{
    if (pSemaphore == NULL) {
        return MA_INVALID_ARGS;
    }

    pthread_mutex_lock(&pSemaphore->lock);
    {
        /* We need to wait on a condition variable before escaping. We can't return from this function until the semaphore has been signaled. */
        while (pSemaphore->value == 0) {
            pthread_cond_wait(&pSemaphore->cond, &pSemaphore->lock);
        }

        pSemaphore->value -= 1;
    }
    pthread_mutex_unlock(&pSemaphore->lock);

    return MA_SUCCESS;
}

static ma_result ma_semaphore_release__posix(ma_semaphore* pSemaphore)
{
    if (pSemaphore == NULL) {
        return MA_INVALID_ARGS;
    }

    pthread_mutex_lock(&pSemaphore->lock);
    {
        pSemaphore->value += 1;
        pthread_cond_signal(&pSemaphore->cond);
    }
    pthread_mutex_unlock(&pSemaphore->lock);

    return MA_SUCCESS;
}
#endif

static ma_result ma_thread_create(ma_thread* pThread, ma_thread_priority priority, size_t stackSize, ma_thread_entry_proc entryProc, void* pData)
{
    if (pThread == NULL || entryProc == NULL) {
        return MA_FALSE;
    }

#ifdef MA_WIN32
    return ma_thread_create__win32(pThread, priority, stackSize, entryProc, pData);
#endif
#ifdef MA_POSIX
    return ma_thread_create__posix(pThread, priority, stackSize, entryProc, pData);
#endif
}

static void ma_thread_wait(ma_thread* pThread)
{
    if (pThread == NULL) {
        return;
    }

#ifdef MA_WIN32
    ma_thread_wait__win32(pThread);
#endif
#ifdef MA_POSIX
    ma_thread_wait__posix(pThread);
#endif
}


MA_API ma_result ma_mutex_init(ma_mutex* pMutex)
{
    if (pMutex == NULL) {
        MA_ASSERT(MA_FALSE);    /* Fire an assert to the caller is aware of this bug. */
        return MA_INVALID_ARGS;
    }

#ifdef MA_WIN32
    return ma_mutex_init__win32(pMutex);
#endif
#ifdef MA_POSIX
    return ma_mutex_init__posix(pMutex);
#endif
}

MA_API void ma_mutex_uninit(ma_mutex* pMutex)
{
    if (pMutex == NULL) {
        return;
    }

#ifdef MA_WIN32
    ma_mutex_uninit__win32(pMutex);
#endif
#ifdef MA_POSIX
    ma_mutex_uninit__posix(pMutex);
#endif
}

MA_API void ma_mutex_lock(ma_mutex* pMutex)
{
    if (pMutex == NULL) {
        MA_ASSERT(MA_FALSE);    /* Fire an assert to the caller is aware of this bug. */
        return;
    }

#ifdef MA_WIN32
    ma_mutex_lock__win32(pMutex);
#endif
#ifdef MA_POSIX
    ma_mutex_lock__posix(pMutex);
#endif
}

MA_API void ma_mutex_unlock(ma_mutex* pMutex)
{
    if (pMutex == NULL) {
        MA_ASSERT(MA_FALSE);    /* Fire an assert to the caller is aware of this bug. */
        return;
}

#ifdef MA_WIN32
    ma_mutex_unlock__win32(pMutex);
#endif
#ifdef MA_POSIX
    ma_mutex_unlock__posix(pMutex);
#endif
}


MA_API ma_result ma_event_init(ma_event* pEvent)
{
    if (pEvent == NULL) {
        MA_ASSERT(MA_FALSE);    /* Fire an assert to the caller is aware of this bug. */
        return MA_INVALID_ARGS;
    }

#ifdef MA_WIN32
    return ma_event_init__win32(pEvent);
#endif
#ifdef MA_POSIX
    return ma_event_init__posix(pEvent);
#endif
}

#if 0
static ma_result ma_event_alloc_and_init(ma_event** ppEvent, ma_allocation_callbacks* pAllocationCallbacks)
{
    ma_result result;
    ma_event* pEvent;

    if (ppEvent == NULL) {
        return MA_INVALID_ARGS;
    }

    *ppEvent = NULL;

    pEvent = ma_malloc(sizeof(*pEvent), pAllocationCallbacks/*, MA_ALLOCATION_TYPE_EVENT*/);
    if (pEvent == NULL) {
        return MA_OUT_OF_MEMORY;
    }

    result = ma_event_init(pEvent);
    if (result != MA_SUCCESS) {
        ma_free(pEvent, pAllocationCallbacks/*, MA_ALLOCATION_TYPE_EVENT*/);
        return result;
    }

    *ppEvent = pEvent;
    return result;
}
#endif

MA_API void ma_event_uninit(ma_event* pEvent)
{
    if (pEvent == NULL) {
        return;
    }

#ifdef MA_WIN32
    ma_event_uninit__win32(pEvent);
#endif
#ifdef MA_POSIX
    ma_event_uninit__posix(pEvent);
#endif
}

#if 0
static void ma_event_uninit_and_free(ma_event* pEvent, ma_allocation_callbacks* pAllocationCallbacks)
{
    if (pEvent == NULL) {
        return;
    }

    ma_event_uninit(pEvent);
    ma_free(pEvent, pAllocationCallbacks/*, MA_ALLOCATION_TYPE_EVENT*/);
}
#endif

MA_API ma_result ma_event_wait(ma_event* pEvent)
{
    if (pEvent == NULL) {
        MA_ASSERT(MA_FALSE);    /* Fire an assert to the caller is aware of this bug. */
        return MA_INVALID_ARGS;
    }

#ifdef MA_WIN32
    return ma_event_wait__win32(pEvent);
#endif
#ifdef MA_POSIX
    return ma_event_wait__posix(pEvent);
#endif
}

MA_API ma_result ma_event_signal(ma_event* pEvent)
{
    if (pEvent == NULL) {
        MA_ASSERT(MA_FALSE);    /* Fire an assert to the caller is aware of this bug. */
        return MA_INVALID_ARGS;
    }

#ifdef MA_WIN32
    return ma_event_signal__win32(pEvent);
#endif
#ifdef MA_POSIX
    return ma_event_signal__posix(pEvent);
#endif
}


MA_API ma_result ma_semaphore_init(int initialValue, ma_semaphore* pSemaphore)
{
    if (pSemaphore == NULL) {
        MA_ASSERT(MA_FALSE);    /* Fire an assert so the caller is aware of this bug. */
        return MA_INVALID_ARGS;
    }

#ifdef MA_WIN32
    return ma_semaphore_init__win32(initialValue, pSemaphore);
#endif
#ifdef MA_POSIX
    return ma_semaphore_init__posix(initialValue, pSemaphore);
#endif
}

MA_API void ma_semaphore_uninit(ma_semaphore* pSemaphore)
{
    if (pSemaphore == NULL) {
        MA_ASSERT(MA_FALSE);    /* Fire an assert so the caller is aware of this bug. */
        return;
    }

#ifdef MA_WIN32
    ma_semaphore_uninit__win32(pSemaphore);
#endif
#ifdef MA_POSIX
    ma_semaphore_uninit__posix(pSemaphore);
#endif
}

MA_API ma_result ma_semaphore_wait(ma_semaphore* pSemaphore)
{
    if (pSemaphore == NULL) {
        MA_ASSERT(MA_FALSE);    /* Fire an assert so the caller is aware of this bug. */
        return MA_INVALID_ARGS;
    }

#ifdef MA_WIN32
    return ma_semaphore_wait__win32(pSemaphore);
#endif
#ifdef MA_POSIX
    return ma_semaphore_wait__posix(pSemaphore);
#endif
}

MA_API ma_result ma_semaphore_release(ma_semaphore* pSemaphore)
{
    if (pSemaphore == NULL) {
        MA_ASSERT(MA_FALSE);    /* Fire an assert so the caller is aware of this bug. */
        return MA_INVALID_ARGS;
    }

#ifdef MA_WIN32
    return ma_semaphore_release__win32(pSemaphore);
#endif
#ifdef MA_POSIX
    return ma_semaphore_release__posix(pSemaphore);
#endif
}
#else
/* MA_NO_THREADING is set which means threading is disabled. Threading is required by some API families. If any of these are enabled we need to throw an error. */
#ifndef MA_NO_DEVICE_IO
#error "MA_NO_THREADING cannot be used without MA_NO_DEVICE_IO";
#endif
#endif  /* MA_NO_THREADING */


/************************************************************************************************************************************************************
*************************************************************************************************************************************************************

DEVICE I/O
==========

*************************************************************************************************************************************************************
************************************************************************************************************************************************************/
#ifndef MA_NO_DEVICE_IO
#ifdef MA_WIN32
    #include <objbase.h>
    #include <mmreg.h>
    #include <mmsystem.h>
#endif

#if defined(MA_APPLE) && (__MAC_OS_X_VERSION_MIN_REQUIRED < 101200)
    #include <mach/mach_time.h> /* For mach_absolute_time() */
#endif

#ifdef MA_POSIX
    #include <sys/types.h>
    #include <unistd.h>
    #include <dlfcn.h>
#endif

/*
Unfortunately using runtime linking for pthreads causes problems. This has occurred for me when testing on FreeBSD. When
using runtime linking, deadlocks can occur (for me it happens when loading data from fread()). It turns out that doing
compile-time linking fixes this. I'm not sure why this happens, but the safest way I can think of to fix this is to simply
disable runtime linking by default. To enable runtime linking, #define this before the implementation of this file. I am
not officially supporting this, but I'm leaving it here in case it's useful for somebody, somewhere.
*/
/*#define MA_USE_RUNTIME_LINKING_FOR_PTHREAD*/

/* Disable run-time linking on certain backends. */
#ifndef MA_NO_RUNTIME_LINKING
    #if defined(MA_ANDROID) || defined(MA_EMSCRIPTEN)
        #define MA_NO_RUNTIME_LINKING
    #endif
#endif

/*
Check if we have the necessary development packages for each backend at the top so we can use this to determine whether or not
certain unused functions and variables can be excluded from the build to avoid warnings.
*/
#ifdef MA_ENABLE_WASAPI
    #define MA_HAS_WASAPI      /* Every compiler should support WASAPI */
#endif
#ifdef MA_ENABLE_DSOUND
    #define MA_HAS_DSOUND      /* Every compiler should support DirectSound. */
#endif
#ifdef MA_ENABLE_WINMM
    #define MA_HAS_WINMM       /* Every compiler I'm aware of supports WinMM. */
#endif
#ifdef MA_ENABLE_ALSA
    #define MA_HAS_ALSA
    #ifdef MA_NO_RUNTIME_LINKING
        #ifdef __has_include
            #if !__has_include(<alsa/asoundlib.h>)
                #undef MA_HAS_ALSA
            #endif
        #endif
    #endif
#endif
#ifdef MA_ENABLE_PULSEAUDIO
    #define MA_HAS_PULSEAUDIO
    #ifdef MA_NO_RUNTIME_LINKING
        #ifdef __has_include
            #if !__has_include(<pulse/pulseaudio.h>)
                #undef MA_HAS_PULSEAUDIO
            #endif
        #endif
    #endif
#endif
#ifdef MA_ENABLE_JACK
    #define MA_HAS_JACK
    #ifdef MA_NO_RUNTIME_LINKING
        #ifdef __has_include
            #if !__has_include(<jack/jack.h>)
                #undef MA_HAS_JACK
            #endif
        #endif
    #endif
#endif
#ifdef MA_ENABLE_COREAUDIO
    #define MA_HAS_COREAUDIO
#endif
#ifdef MA_ENABLE_SNDIO
    #define MA_HAS_SNDIO
#endif
#ifdef MA_ENABLE_AUDIO4
    #define MA_HAS_AUDIO4
#endif
#ifdef MA_ENABLE_OSS
    #define MA_HAS_OSS
#endif
#ifdef MA_ENABLE_AAUDIO
    #define MA_HAS_AAUDIO
#endif
#ifdef MA_ENABLE_OPENSL
    #define MA_HAS_OPENSL
#endif
#ifdef MA_ENABLE_WEBAUDIO
    #define MA_HAS_WEBAUDIO
#endif
#ifdef MA_ENABLE_NULL
    #define MA_HAS_NULL    /* Everything supports the null backend. */
#endif

MA_API const char* ma_get_backend_name(ma_backend backend)
{
    switch (backend)
    {
        case ma_backend_wasapi:     return "WASAPI";
        case ma_backend_dsound:     return "DirectSound";
        case ma_backend_winmm:      return "WinMM";
        case ma_backend_coreaudio:  return "Core Audio";
        case ma_backend_sndio:      return "sndio";
        case ma_backend_audio4:     return "audio(4)";
        case ma_backend_oss:        return "OSS";
        case ma_backend_pulseaudio: return "PulseAudio";
        case ma_backend_alsa:       return "ALSA";
        case ma_backend_jack:       return "JACK";
        case ma_backend_aaudio:     return "AAudio";
        case ma_backend_opensl:     return "OpenSL|ES";
        case ma_backend_webaudio:   return "Web Audio";
        case ma_backend_null:       return "Null";
        default:                    return "Unknown";
    }
}

MA_API ma_bool32 ma_is_backend_enabled(ma_backend backend)
{
    /*
    This looks a little bit gross, but we want all backends to be included in the switch to avoid warnings on some compilers
    about some enums not being handled by the switch statement.
    */
    switch (backend)
    {
        case ma_backend_wasapi:
        #if defined(MA_HAS_WASAPI)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_dsound:
        #if defined(MA_HAS_DSOUND)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_winmm:
        #if defined(MA_HAS_WINMM)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_coreaudio:
        #if defined(MA_HAS_COREAUDIO)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_sndio:
        #if defined(MA_HAS_SNDIO)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_audio4:
        #if defined(MA_HAS_AUDIO4)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_oss:
        #if defined(MA_HAS_OSS)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_pulseaudio:
        #if defined(MA_HAS_PULSEAUDIO)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_alsa:
        #if defined(MA_HAS_ALSA)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_jack:
        #if defined(MA_HAS_JACK)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_aaudio:
        #if defined(MA_HAS_AAUDIO)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_opensl:
        #if defined(MA_HAS_OPENSL)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_webaudio:
        #if defined(MA_HAS_WEBAUDIO)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif
        case ma_backend_null:
        #if defined(MA_HAS_NULL)
            return MA_TRUE;
        #else
            return MA_FALSE;
        #endif

        default: return MA_FALSE;
    }
}

MA_API ma_result ma_get_enabled_backends(ma_backend* pBackends, size_t backendCap, size_t* pBackendCount)
{
    size_t backendCount;
    size_t iBackend;
    ma_result result = MA_SUCCESS;

    if (pBackendCount == NULL) {
        return MA_INVALID_ARGS;
    }

    backendCount = 0;

    for (iBackend = 0; iBackend <= ma_backend_null; iBackend += 1) {
        ma_backend backend = (ma_backend)iBackend;

        if (ma_is_backend_enabled(backend)) {
            /* The backend is enabled. Try adding it to the list. If there's no room, MA_NO_SPACE needs to be returned. */
            if (backendCount == backendCap) {
                result = MA_NO_SPACE;
                break;
            } else {
                pBackends[backendCount] = backend;
                backendCount += 1;
            }
        }
    }

    if (pBackendCount != NULL) {
        *pBackendCount = backendCount;
    }

    return result;
}

MA_API ma_bool32 ma_is_loopback_supported(ma_backend backend)
{
    switch (backend)
    {
        case ma_backend_wasapi:     return MA_TRUE;
        case ma_backend_dsound:     return MA_FALSE;
        case ma_backend_winmm:      return MA_FALSE;
        case ma_backend_coreaudio:  return MA_FALSE;
        case ma_backend_sndio:      return MA_FALSE;
        case ma_backend_audio4:     return MA_FALSE;
        case ma_backend_oss:        return MA_FALSE;
        case ma_backend_pulseaudio: return MA_FALSE;
        case ma_backend_alsa:       return MA_FALSE;
        case ma_backend_jack:       return MA_FALSE;
        case ma_backend_aaudio:     return MA_FALSE;
        case ma_backend_opensl:     return MA_FALSE;
        case ma_backend_webaudio:   return MA_FALSE;
        case ma_backend_null:       return MA_FALSE;
        default:                    return MA_FALSE;
    }
}



#ifdef MA_WIN32
/* WASAPI error codes. */
#define MA_AUDCLNT_E_NOT_INITIALIZED              ((HRESULT)0x88890001)
#define MA_AUDCLNT_E_ALREADY_INITIALIZED          ((HRESULT)0x88890002)
#define MA_AUDCLNT_E_WRONG_ENDPOINT_TYPE          ((HRESULT)0x88890003)
#define MA_AUDCLNT_E_DEVICE_INVALIDATED           ((HRESULT)0x88890004)
#define MA_AUDCLNT_E_NOT_STOPPED                  ((HRESULT)0x88890005)
#define MA_AUDCLNT_E_BUFFER_TOO_LARGE             ((HRESULT)0x88890006)
#define MA_AUDCLNT_E_OUT_OF_ORDER                 ((HRESULT)0x88890007)
#define MA_AUDCLNT_E_UNSUPPORTED_FORMAT           ((HRESULT)0x88890008)
#define MA_AUDCLNT_E_INVALID_SIZE                 ((HRESULT)0x88890009)
#define MA_AUDCLNT_E_DEVICE_IN_USE                ((HRESULT)0x8889000A)
#define MA_AUDCLNT_E_BUFFER_OPERATION_PENDING     ((HRESULT)0x8889000B)
#define MA_AUDCLNT_E_THREAD_NOT_REGISTERED        ((HRESULT)0x8889000C)
#define MA_AUDCLNT_E_NO_SINGLE_PROCESS            ((HRESULT)0x8889000D)
#define MA_AUDCLNT_E_EXCLUSIVE_MODE_NOT_ALLOWED   ((HRESULT)0x8889000E)
#define MA_AUDCLNT_E_ENDPOINT_CREATE_FAILED       ((HRESULT)0x8889000F)
#define MA_AUDCLNT_E_SERVICE_NOT_RUNNING          ((HRESULT)0x88890010)
#define MA_AUDCLNT_E_EVENTHANDLE_NOT_EXPECTED     ((HRESULT)0x88890011)
#define MA_AUDCLNT_E_EXCLUSIVE_MODE_ONLY          ((HRESULT)0x88890012)
#define MA_AUDCLNT_E_BUFDURATION_PERIOD_NOT_EQUAL ((HRESULT)0x88890013)
#define MA_AUDCLNT_E_EVENTHANDLE_NOT_SET          ((HRESULT)0x88890014)
#define MA_AUDCLNT_E_INCORRECT_BUFFER_SIZE        ((HRESULT)0x88890015)
#define MA_AUDCLNT_E_BUFFER_SIZE_ERROR            ((HRESULT)0x88890016)
#define MA_AUDCLNT_E_CPUUSAGE_EXCEEDED            ((HRESULT)0x88890017)
#define MA_AUDCLNT_E_BUFFER_ERROR                 ((HRESULT)0x88890018)
#define MA_AUDCLNT_E_BUFFER_SIZE_NOT_ALIGNED      ((HRESULT)0x88890019)
#define MA_AUDCLNT_E_INVALID_DEVICE_PERIOD        ((HRESULT)0x88890020)
#define MA_AUDCLNT_E_INVALID_STREAM_FLAG          ((HRESULT)0x88890021)
#define MA_AUDCLNT_E_ENDPOINT_OFFLOAD_NOT_CAPABLE ((HRESULT)0x88890022)
#define MA_AUDCLNT_E_OUT_OF_OFFLOAD_RESOURCES     ((HRESULT)0x88890023)
#define MA_AUDCLNT_E_OFFLOAD_MODE_ONLY            ((HRESULT)0x88890024)
#define MA_AUDCLNT_E_NONOFFLOAD_MODE_ONLY         ((HRESULT)0x88890025)
#define MA_AUDCLNT_E_RESOURCES_INVALIDATED        ((HRESULT)0x88890026)
#define MA_AUDCLNT_E_RAW_MODE_UNSUPPORTED         ((HRESULT)0x88890027)
#define MA_AUDCLNT_E_ENGINE_PERIODICITY_LOCKED    ((HRESULT)0x88890028)
#define MA_AUDCLNT_E_ENGINE_FORMAT_LOCKED         ((HRESULT)0x88890029)
#define MA_AUDCLNT_E_HEADTRACKING_ENABLED         ((HRESULT)0x88890030)
#define MA_AUDCLNT_E_HEADTRACKING_UNSUPPORTED     ((HRESULT)0x88890040)
#define MA_AUDCLNT_S_BUFFER_EMPTY                 ((HRESULT)0x08890001)
#define MA_AUDCLNT_S_THREAD_ALREADY_REGISTERED    ((HRESULT)0x08890002)
#define MA_AUDCLNT_S_POSITION_STALLED             ((HRESULT)0x08890003)

#define MA_DS_OK                                  ((HRESULT)0)
#define MA_DS_NO_VIRTUALIZATION                   ((HRESULT)0x0878000A)
#define MA_DSERR_ALLOCATED                        ((HRESULT)0x8878000A)
#define MA_DSERR_CONTROLUNAVAIL                   ((HRESULT)0x8878001E)
#define MA_DSERR_INVALIDPARAM                     ((HRESULT)0x80070057) /*E_INVALIDARG*/
#define MA_DSERR_INVALIDCALL                      ((HRESULT)0x88780032)
#define MA_DSERR_GENERIC                          ((HRESULT)0x80004005) /*E_FAIL*/
#define MA_DSERR_PRIOLEVELNEEDED                  ((HRESULT)0x88780046)
#define MA_DSERR_OUTOFMEMORY                      ((HRESULT)0x8007000E) /*E_OUTOFMEMORY*/
#define MA_DSERR_BADFORMAT                        ((HRESULT)0x88780064)
#define MA_DSERR_UNSUPPORTED                      ((HRESULT)0x80004001) /*E_NOTIMPL*/
#define MA_DSERR_NODRIVER                         ((HRESULT)0x88780078)
#define MA_DSERR_ALREADYINITIALIZED               ((HRESULT)0x88780082)
#define MA_DSERR_NOAGGREGATION                    ((HRESULT)0x80040110) /*CLASS_E_NOAGGREGATION*/
#define MA_DSERR_BUFFERLOST                       ((HRESULT)0x88780096)
#define MA_DSERR_OTHERAPPHASPRIO                  ((HRESULT)0x887800A0)
#define MA_DSERR_UNINITIALIZED                    ((HRESULT)0x887800AA)
#define MA_DSERR_NOINTERFACE                      ((HRESULT)0x80004002) /*E_NOINTERFACE*/
#define MA_DSERR_ACCESSDENIED                     ((HRESULT)0x80070005) /*E_ACCESSDENIED*/
#define MA_DSERR_BUFFERTOOSMALL                   ((HRESULT)0x887800B4)
#define MA_DSERR_DS8_REQUIRED                     ((HRESULT)0x887800BE)
#define MA_DSERR_SENDLOOP                         ((HRESULT)0x887800C8)
#define MA_DSERR_BADSENDBUFFERGUID                ((HRESULT)0x887800D2)
#define MA_DSERR_OBJECTNOTFOUND                   ((HRESULT)0x88781161)
#define MA_DSERR_FXUNAVAILABLE                    ((HRESULT)0x887800DC)

static ma_result ma_result_from_HRESULT(HRESULT hr)
{
    switch (hr)
    {
        case NOERROR:                                   return MA_SUCCESS;
        /*case S_OK:                                      return MA_SUCCESS;*/

        case E_POINTER:                                 return MA_INVALID_ARGS;
        case E_UNEXPECTED:                              return MA_ERROR;
        case E_NOTIMPL:                                 return MA_NOT_IMPLEMENTED;
        case E_OUTOFMEMORY:                             return MA_OUT_OF_MEMORY;
        case E_INVALIDARG:                              return MA_INVALID_ARGS;
        case E_NOINTERFACE:                             return MA_API_NOT_FOUND;
        case E_HANDLE:                                  return MA_INVALID_ARGS;
        case E_ABORT:                                   return MA_ERROR;
        case E_FAIL:                                    return MA_ERROR;
        case E_ACCESSDENIED:                            return MA_ACCESS_DENIED;

        /* WASAPI */
        case MA_AUDCLNT_E_NOT_INITIALIZED:              return MA_DEVICE_NOT_INITIALIZED;
        case MA_AUDCLNT_E_ALREADY_INITIALIZED:          return MA_DEVICE_ALREADY_INITIALIZED;
        case MA_AUDCLNT_E_WRONG_ENDPOINT_TYPE:          return MA_INVALID_ARGS;
        case MA_AUDCLNT_E_DEVICE_INVALIDATED:           return MA_UNAVAILABLE;
        case MA_AUDCLNT_E_NOT_STOPPED:                  return MA_DEVICE_NOT_STOPPED;
        case MA_AUDCLNT_E_BUFFER_TOO_LARGE:             return MA_TOO_BIG;
        case MA_AUDCLNT_E_OUT_OF_ORDER:                 return MA_INVALID_OPERATION;
        case MA_AUDCLNT_E_UNSUPPORTED_FORMAT:           return MA_FORMAT_NOT_SUPPORTED;
        case MA_AUDCLNT_E_INVALID_SIZE:                 return MA_INVALID_ARGS;
        case MA_AUDCLNT_E_DEVICE_IN_USE:                return MA_BUSY;
        case MA_AUDCLNT_E_BUFFER_OPERATION_PENDING:     return MA_INVALID_OPERATION;
        case MA_AUDCLNT_E_THREAD_NOT_REGISTERED:        return MA_DOES_NOT_EXIST;
        case MA_AUDCLNT_E_NO_SINGLE_PROCESS:            return MA_INVALID_OPERATION;
        case MA_AUDCLNT_E_EXCLUSIVE_MODE_NOT_ALLOWED:   return MA_SHARE_MODE_NOT_SUPPORTED;
        case MA_AUDCLNT_E_ENDPOINT_CREATE_FAILED:       return MA_FAILED_TO_OPEN_BACKEND_DEVICE;
        case MA_AUDCLNT_E_SERVICE_NOT_RUNNING:          return MA_NOT_CONNECTED;
        case MA_AUDCLNT_E_EVENTHANDLE_NOT_EXPECTED:     return MA_INVALID_ARGS;
        case MA_AUDCLNT_E_EXCLUSIVE_MODE_ONLY:          return MA_SHARE_MODE_NOT_SUPPORTED;
        case MA_AUDCLNT_E_BUFDURATION_PERIOD_NOT_EQUAL: return MA_INVALID_ARGS;
        case MA_AUDCLNT_E_EVENTHANDLE_NOT_SET:          return MA_INVALID_ARGS;
        case MA_AUDCLNT_E_INCORRECT_BUFFER_SIZE:        return MA_INVALID_ARGS;
        case MA_AUDCLNT_E_BUFFER_SIZE_ERROR:            return MA_INVALID_ARGS;
        case MA_AUDCLNT_E_CPUUSAGE_EXCEEDED:            return MA_ERROR;
        case MA_AUDCLNT_E_BUFFER_ERROR:                 return MA_ERROR;
        case MA_AUDCLNT_E_BUFFER_SIZE_NOT_ALIGNED:      return MA_INVALID_ARGS;
        case MA_AUDCLNT_E_INVALID_DEVICE_PERIOD:        return MA_INVALID_ARGS;
        case MA_AUDCLNT_E_INVALID_STREAM_FLAG:          return MA_INVALID_ARGS;
        case MA_AUDCLNT_E_ENDPOINT_OFFLOAD_NOT_CAPABLE: return MA_INVALID_OPERATION;
        case MA_AUDCLNT_E_OUT_OF_OFFLOAD_RESOURCES:     return MA_OUT_OF_MEMORY;
        case MA_AUDCLNT_E_OFFLOAD_MODE_ONLY:            return MA_INVALID_OPERATION;
        case MA_AUDCLNT_E_NONOFFLOAD_MODE_ONLY:         return MA_INVALID_OPERATION;
        case MA_AUDCLNT_E_RESOURCES_INVALIDATED:        return MA_INVALID_DATA;
        case MA_AUDCLNT_E_RAW_MODE_UNSUPPORTED:         return MA_INVALID_OPERATION;
        case MA_AUDCLNT_E_ENGINE_PERIODICITY_LOCKED:    return MA_INVALID_OPERATION;
        case MA_AUDCLNT_E_ENGINE_FORMAT_LOCKED:         return MA_INVALID_OPERATION;
        case MA_AUDCLNT_E_HEADTRACKING_ENABLED:         return MA_INVALID_OPERATION;
        case MA_AUDCLNT_E_HEADTRACKING_UNSUPPORTED:     return MA_INVALID_OPERATION;
        case MA_AUDCLNT_S_BUFFER_EMPTY:                 return MA_NO_SPACE;
        case MA_AUDCLNT_S_THREAD_ALREADY_REGISTERED:    return MA_ALREADY_EXISTS;
        case MA_AUDCLNT_S_POSITION_STALLED:             return MA_ERROR;

        /* DirectSound */
        /*case MA_DS_OK:                                  return MA_SUCCESS;*/          /* S_OK */
        case MA_DS_NO_VIRTUALIZATION:                   return MA_SUCCESS;
        case MA_DSERR_ALLOCATED:                        return MA_ALREADY_IN_USE;
        case MA_DSERR_CONTROLUNAVAIL:                   return MA_INVALID_OPERATION;
        /*case MA_DSERR_INVALIDPARAM:                    return MA_INVALID_ARGS;*/      /* E_INVALIDARG */
        case MA_DSERR_INVALIDCALL:                      return MA_INVALID_OPERATION;
        /*case MA_DSERR_GENERIC:                          return MA_ERROR;*/            /* E_FAIL */
        case MA_DSERR_PRIOLEVELNEEDED:                  return MA_INVALID_OPERATION;
        /*case MA_DSERR_OUTOFMEMORY:                      return MA_OUT_OF_MEMORY;*/    /* E_OUTOFMEMORY */
        case MA_DSERR_BADFORMAT:                        return MA_FORMAT_NOT_SUPPORTED;
        /*case MA_DSERR_UNSUPPORTED:                      return MA_NOT_IMPLEMENTED;*/  /* E_NOTIMPL */
        case MA_DSERR_NODRIVER:                         return MA_FAILED_TO_INIT_BACKEND;
        case MA_DSERR_ALREADYINITIALIZED:               return MA_DEVICE_ALREADY_INITIALIZED;
        case MA_DSERR_NOAGGREGATION:                    return MA_ERROR;
        case MA_DSERR_BUFFERLOST:                       return MA_UNAVAILABLE;
        case MA_DSERR_OTHERAPPHASPRIO:                  return MA_ACCESS_DENIED;
        case MA_DSERR_UNINITIALIZED:                    return MA_DEVICE_NOT_INITIALIZED;
        /*case MA_DSERR_NOINTERFACE:                      return MA_API_NOT_FOUND;*/    /* E_NOINTERFACE */
        /*case MA_DSERR_ACCESSDENIED:                     return MA_ACCESS_DENIED;*/    /* E_ACCESSDENIED */
        case MA_DSERR_BUFFERTOOSMALL:                   return MA_NO_SPACE;
        case MA_DSERR_DS8_REQUIRED:                     return MA_INVALID_OPERATION;
        case MA_DSERR_SENDLOOP:                         return MA_DEADLOCK;
        case MA_DSERR_BADSENDBUFFERGUID:                return MA_INVALID_ARGS;
        case MA_DSERR_OBJECTNOTFOUND:                   return MA_NO_DEVICE;
        case MA_DSERR_FXUNAVAILABLE:                    return MA_UNAVAILABLE;

        default:                                        return MA_ERROR;
    }
}

typedef HRESULT (WINAPI * MA_PFN_CoInitializeEx)(LPVOID pvReserved, DWORD  dwCoInit);
typedef void    (WINAPI * MA_PFN_CoUninitialize)(void);
typedef HRESULT (WINAPI * MA_PFN_CoCreateInstance)(REFCLSID rclsid, LPUNKNOWN pUnkOuter, DWORD dwClsContext, REFIID riid, LPVOID *ppv);
typedef void    (WINAPI * MA_PFN_CoTaskMemFree)(LPVOID pv);
typedef HRESULT (WINAPI * MA_PFN_PropVariantClear)(PROPVARIANT *pvar);
typedef int     (WINAPI * MA_PFN_StringFromGUID2)(const GUID* const rguid, LPOLESTR lpsz, int cchMax);

typedef HWND (WINAPI * MA_PFN_GetForegroundWindow)(void);
typedef HWND (WINAPI * MA_PFN_GetDesktopWindow)(void);

/* Microsoft documents these APIs as returning LSTATUS, but the Win32 API shipping with some compilers do not define it. It's just a LONG. */
typedef LONG (WINAPI * MA_PFN_RegOpenKeyExA)(HKEY hKey, LPCSTR lpSubKey, DWORD ulOptions, REGSAM samDesired, PHKEY phkResult);
typedef LONG (WINAPI * MA_PFN_RegCloseKey)(HKEY hKey);
typedef LONG (WINAPI * MA_PFN_RegQueryValueExA)(HKEY hKey, LPCSTR lpValueName, LPDWORD lpReserved, LPDWORD lpType, LPBYTE lpData, LPDWORD lpcbData);
#endif


#define MA_STATE_UNINITIALIZED     0
#define MA_STATE_STOPPED           1   /* The device's default state after initialization. */
#define MA_STATE_STARTED           2   /* The worker thread is in it's main loop waiting for the driver to request or deliver audio data. */
#define MA_STATE_STARTING          3   /* Transitioning from a stopped state to started. */
#define MA_STATE_STOPPING          4   /* Transitioning from a started state to stopped. */

#define MA_DEFAULT_PLAYBACK_DEVICE_NAME    "Default Playback Device"
#define MA_DEFAULT_CAPTURE_DEVICE_NAME     "Default Capture Device"


MA_API const char* ma_log_level_to_string(ma_uint32 logLevel)
{
    switch (logLevel)
    {
        case MA_LOG_LEVEL_VERBOSE: return "";
        case MA_LOG_LEVEL_INFO:    return "INFO";
        case MA_LOG_LEVEL_WARNING: return "WARNING";
        case MA_LOG_LEVEL_ERROR:   return "ERROR";
        default:                   return "ERROR";
    }
}

/* Posts a log message. */
static void ma_post_log_message(ma_context* pContext, ma_device* pDevice, ma_uint32 logLevel, const char* message)
{
    if (pContext == NULL) {
        if (pDevice != NULL) {
            pContext = pDevice->pContext;
        }
    }

    /* All logs must be output when debug output is enabled. */
#if defined(MA_DEBUG_OUTPUT)
    printf("%s: %s\n", ma_log_level_to_string(logLevel), message);
#endif

    if (pContext == NULL) {
        return;
    }
    
#if defined(MA_LOG_LEVEL)
    if (logLevel <= MA_LOG_LEVEL) {
        ma_log_proc onLog;

        onLog = pContext->logCallback;
        if (onLog) {
            onLog(pContext, pDevice, logLevel, message);
        }
    }
#endif
}

/*
We need to emulate _vscprintf() for the VC6 build. This can be more efficient, but since it's only VC6, and it's just a
logging function, I'm happy to keep this simple. In the VC6 build we can implement this in terms of _vsnprintf().
*/
#if defined(_MSC_VER) && _MSC_VER < 1900
int ma_vscprintf(const char* format, va_list args)
{
#if _MSC_VER > 1200
    return _vscprintf(format, args);
#else
    int result;
    char* pTempBuffer = NULL;
    size_t tempBufferCap = 1024;

    if (format == NULL) {
        errno = EINVAL;
        return -1;
    }

	for (;;) {
        char* pNewTempBuffer = (char*)ma_realloc(pTempBuffer, tempBufferCap, NULL);    /* TODO: Add support for custom memory allocators? */
        if (pNewTempBuffer == NULL) {
            ma_free(pTempBuffer, NULL);
            errno = ENOMEM;
            return -1;  /* Out of memory. */
        }

        pTempBuffer = pNewTempBuffer;

        result = _vsnprintf(pTempBuffer, tempBufferCap, format, args);
        ma_free(pTempBuffer, NULL);
        
        if (result != -1) {
            break;  /* Got it. */
        }

        /* Buffer wasn't big enough. Ideally it'd be nice to use an error code to know the reason for sure, but this is reliable enough. */
        tempBufferCap *= 2;
	}

    return result;
#endif
}
#endif

/* Posts a formatted log message. */
static void ma_post_log_messagev(ma_context* pContext, ma_device* pDevice, ma_uint32 logLevel, const char* pFormat, va_list args)
{
#if (!defined(_MSC_VER) || _MSC_VER >= 1900) && !defined(__STRICT_ANSI__)
    {
        char pFormattedMessage[1024];
        vsnprintf(pFormattedMessage, sizeof(pFormattedMessage), pFormat, args);
        ma_post_log_message(pContext, pDevice, logLevel, pFormattedMessage);
    }
#else
    {
        /*
        Without snprintf() we need to first measure the string and then heap allocate it. I'm only aware of Visual Studio having support for this without snprintf(), so we'll
        need to restrict this branch to Visual Studio. For other compilers we need to just not support formatted logging because I don't want the security risk of overflowing
        a fixed sized stack allocated buffer.
        */
    #if defined(_MSC_VER) && _MSC_VER >= 1200   /* 1200 = VC6 */
        int formattedLen;
        va_list args2;

    #if _MSC_VER >= 1800
        va_copy(args2, args);
    #else
        args2 = args;
    #endif
        formattedLen = ma_vscprintf(pFormat, args2);
        va_end(args2);

        if (formattedLen > 0) {
            char* pFormattedMessage = NULL;
            ma_allocation_callbacks* pAllocationCallbacks = NULL;

            /* Make sure we have a context so we can allocate memory. */
            if (pContext == NULL) {
                if (pDevice != NULL) {
                    pContext = pDevice->pContext;
                }
            }

            if (pContext != NULL) {
                pAllocationCallbacks = &pContext->allocationCallbacks;
            }

            pFormattedMessage = (char*)ma_malloc(formattedLen + 1, pAllocationCallbacks);
            if (pFormattedMessage != NULL) {
                /* We'll get errors on newer versions of Visual Studio if we try to use vsprintf().  */
            #if _MSC_VER >= 1400    /* 1400 = Visual Studio 2005 */
                vsprintf_s(pFormattedMessage, formattedLen + 1, pFormat, args);
            #else
                vsprintf(pFormattedMessage, pFormat, args);
            #endif
                
                ma_post_log_message(pContext, pDevice, logLevel, pFormattedMessage);
                ma_free(pFormattedMessage, pAllocationCallbacks);
            }
        }
    #else
        /* Can't do anything because we don't have a safe way of to emulate vsnprintf() without a manual solution. */
        (void)pContext;
        (void)pDevice;
        (void)logLevel;
        (void)pFormat;
        (void)args;
    #endif
    }
#endif
}

MA_API void ma_post_log_messagef(ma_context* pContext, ma_device* pDevice, ma_uint32 logLevel, const char* pFormat, ...)
{
    va_list args;
    va_start(args, pFormat);
    {
        ma_post_log_messagev(pContext, pDevice, logLevel, pFormat, args);
    }
    va_end(args);
}

/* Posts an log message. Throw a breakpoint in here if you're needing to debug. The return value is always "resultCode". */
static ma_result ma_context_post_error(ma_context* pContext, ma_device* pDevice, ma_uint32 logLevel, const char* message, ma_result resultCode)
{
    ma_post_log_message(pContext, pDevice, logLevel, message);
    return resultCode;
}

static ma_result ma_post_error(ma_device* pDevice, ma_uint32 logLevel, const char* message, ma_result resultCode)
{
    return ma_context_post_error(NULL, pDevice, logLevel, message, resultCode);
}


/*******************************************************************************

Timing

*******************************************************************************/
#ifdef MA_WIN32
    static LARGE_INTEGER g_ma_TimerFrequency = {{0}};
    static void ma_timer_init(ma_timer* pTimer)
    {
        LARGE_INTEGER counter;

        if (g_ma_TimerFrequency.QuadPart == 0) {
            QueryPerformanceFrequency(&g_ma_TimerFrequency);
        }

        QueryPerformanceCounter(&counter);
        pTimer->counter = counter.QuadPart;
    }

    static double ma_timer_get_time_in_seconds(ma_timer* pTimer)
    {
        LARGE_INTEGER counter;
        if (!QueryPerformanceCounter(&counter)) {
            return 0;
        }

        return (double)(counter.QuadPart - pTimer->counter) / g_ma_TimerFrequency.QuadPart;
    }
#elif defined(MA_APPLE) && (__MAC_OS_X_VERSION_MIN_REQUIRED < 101200)
    static ma_uint64 g_ma_TimerFrequency = 0;
    static void ma_timer_init(ma_timer* pTimer)
    {
        mach_timebase_info_data_t baseTime;
        mach_timebase_info(&baseTime);
        g_ma_TimerFrequency = (baseTime.denom * 1e9) / baseTime.numer;

        pTimer->counter = mach_absolute_time();
    }

    static double ma_timer_get_time_in_seconds(ma_timer* pTimer)
    {
        ma_uint64 newTimeCounter = mach_absolute_time();
        ma_uint64 oldTimeCounter = pTimer->counter;

        return (newTimeCounter - oldTimeCounter) / g_ma_TimerFrequency;
    }
#elif defined(MA_EMSCRIPTEN)
    static MA_INLINE void ma_timer_init(ma_timer* pTimer)
    {
        pTimer->counterD = emscripten_get_now();
    }

    static MA_INLINE double ma_timer_get_time_in_seconds(ma_timer* pTimer)
    {
        return (emscripten_get_now() - pTimer->counterD) / 1000;    /* Emscripten is in milliseconds. */
    }
#else
    #if _POSIX_C_SOURCE >= 199309L
        #if defined(CLOCK_MONOTONIC)
            #define MA_CLOCK_ID CLOCK_MONOTONIC
        #else
            #define MA_CLOCK_ID CLOCK_REALTIME
        #endif

        static void ma_timer_init(ma_timer* pTimer)
        {
            struct timespec newTime;
            clock_gettime(MA_CLOCK_ID, &newTime);

            pTimer->counter = (newTime.tv_sec * 1000000000) + newTime.tv_nsec;
        }

        static double ma_timer_get_time_in_seconds(ma_timer* pTimer)
        {
            ma_uint64 newTimeCounter;
            ma_uint64 oldTimeCounter;

            struct timespec newTime;
            clock_gettime(MA_CLOCK_ID, &newTime);

            newTimeCounter = (newTime.tv_sec * 1000000000) + newTime.tv_nsec;
            oldTimeCounter = pTimer->counter;

            return (newTimeCounter - oldTimeCounter) / 1000000000.0;
        }
    #else
        static void ma_timer_init(ma_timer* pTimer)
        {
            struct timeval newTime;
            gettimeofday(&newTime, NULL);

            pTimer->counter = (newTime.tv_sec * 1000000) + newTime.tv_usec;
        }

        static double ma_timer_get_time_in_seconds(ma_timer* pTimer)
        {
            ma_uint64 newTimeCounter;
            ma_uint64 oldTimeCounter;

            struct timeval newTime;
            gettimeofday(&newTime, NULL);

            newTimeCounter = (newTime.tv_sec * 1000000) + newTime.tv_usec;
            oldTimeCounter = pTimer->counter;

            return (newTimeCounter - oldTimeCounter) / 1000000.0;
        }
    #endif
#endif


/*******************************************************************************

Dynamic Linking

*******************************************************************************/
MA_API ma_handle ma_dlopen(ma_context* pContext, const char* filename)
{
    ma_handle handle;

#if MA_LOG_LEVEL >= MA_LOG_LEVEL_VERBOSE
    if (pContext != NULL) {
        char message[256];
        ma_strappend(message, sizeof(message), "Loading library: ", filename);
        ma_post_log_message(pContext, NULL, MA_LOG_LEVEL_VERBOSE, message);
    }
#endif

#ifdef _WIN32
#ifdef MA_WIN32_DESKTOP
    handle = (ma_handle)LoadLibraryA(filename);
#else
    /* *sigh* It appears there is no ANSI version of LoadPackagedLibrary()... */
    WCHAR filenameW[4096];
    if (MultiByteToWideChar(CP_UTF8, 0, filename, -1, filenameW, sizeof(filenameW)) == 0) {
        handle = NULL;
    } else {
        handle = (ma_handle)LoadPackagedLibrary(filenameW, 0);
    }
#endif
#else
    handle = (ma_handle)dlopen(filename, RTLD_NOW);
#endif

    /*
    I'm not considering failure to load a library an error nor a warning because seamlessly falling through to a lower-priority
    backend is a deliberate design choice. Instead I'm logging it as an informational message.
    */
#if MA_LOG_LEVEL >= MA_LOG_LEVEL_INFO
    if (handle == NULL) {
        char message[256];
        ma_strappend(message, sizeof(message), "Failed to load library: ", filename);
        ma_post_log_message(pContext, NULL, MA_LOG_LEVEL_INFO, message);
    }
#endif

    (void)pContext; /* It's possible for pContext to be unused. */
    return handle;
}

MA_API void ma_dlclose(ma_context* pContext, ma_handle handle)
{
#ifdef _WIN32
    FreeLibrary((HMODULE)handle);
#else
    dlclose((void*)handle);
#endif

    (void)pContext;
}

MA_API ma_proc ma_dlsym(ma_context* pContext, ma_handle handle, const char* symbol)
{
    ma_proc proc;

#if MA_LOG_LEVEL >= MA_LOG_LEVEL_VERBOSE
    if (pContext != NULL) {
        char message[256];
        ma_strappend(message, sizeof(message), "Loading symbol: ", symbol);
        ma_post_log_message(pContext, NULL, MA_LOG_LEVEL_VERBOSE, message);
    }
#endif

#ifdef _WIN32
    proc = (ma_proc)GetProcAddress((HMODULE)handle, symbol);
#else
#if defined(__GNUC__) && (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ >= 6))
    #pragma GCC diagnostic push
    #pragma GCC diagnostic ignored "-Wpedantic"
#endif
    proc = (ma_proc)dlsym((void*)handle, symbol);
#if defined(__GNUC__) && (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ >= 6))
    #pragma GCC diagnostic pop
#endif
#endif

#if MA_LOG_LEVEL >= MA_LOG_LEVEL_WARNING
    if (handle == NULL) {
        char message[256];
        ma_strappend(message, sizeof(message), "Failed to load symbol: ", symbol);
        ma_post_log_message(pContext, NULL, MA_LOG_LEVEL_WARNING, message);
    }
#endif

    (void)pContext; /* It's possible for pContext to be unused. */
    return proc;
}


#if 0
static ma_uint32 ma_get_closest_standard_sample_rate(ma_uint32 sampleRateIn)
{
    ma_uint32 closestRate = 0;
    ma_uint32 closestDiff = 0xFFFFFFFF;
    size_t iStandardRate;

    for (iStandardRate = 0; iStandardRate < ma_countof(g_maStandardSampleRatePriorities); ++iStandardRate) {
        ma_uint32 standardRate = g_maStandardSampleRatePriorities[iStandardRate];
        ma_uint32 diff;

        if (sampleRateIn > standardRate) {
            diff = sampleRateIn - standardRate;
        } else {
            diff = standardRate - sampleRateIn;
        }

        if (diff == 0) {
            return standardRate;    /* The input sample rate is a standard rate. */
        }

        if (closestDiff > diff) {
            closestDiff = diff;
            closestRate = standardRate;
        }
    }

    return closestRate;
}
#endif


static void ma_device__on_data(ma_device* pDevice, void* pFramesOut, const void* pFramesIn, ma_uint32 frameCount)
{
    float masterVolumeFactor;
    
    masterVolumeFactor = pDevice->masterVolumeFactor;

    if (pDevice->onData) {
        if (!pDevice->noPreZeroedOutputBuffer && pFramesOut != NULL) {
            ma_silence_pcm_frames(pFramesOut, frameCount, pDevice->playback.format, pDevice->playback.channels);
        }

        /* Volume control of input makes things a bit awkward because the input buffer is read-only. We'll need to use a temp buffer and loop in this case. */
        if (pFramesIn != NULL && masterVolumeFactor < 1) {
            ma_uint8 tempFramesIn[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
            ma_uint32 bpfCapture  = ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
            ma_uint32 bpfPlayback = ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
            ma_uint32 totalFramesProcessed = 0;
            while (totalFramesProcessed < frameCount) {
                ma_uint32 framesToProcessThisIteration = frameCount - totalFramesProcessed;
                if (framesToProcessThisIteration > sizeof(tempFramesIn)/bpfCapture) {
                    framesToProcessThisIteration = sizeof(tempFramesIn)/bpfCapture;
                }

                ma_copy_and_apply_volume_factor_pcm_frames(tempFramesIn, ma_offset_ptr(pFramesIn, totalFramesProcessed*bpfCapture), framesToProcessThisIteration, pDevice->capture.format, pDevice->capture.channels, masterVolumeFactor);

                pDevice->onData(pDevice, ma_offset_ptr(pFramesOut, totalFramesProcessed*bpfPlayback), tempFramesIn, framesToProcessThisIteration);

                totalFramesProcessed += framesToProcessThisIteration;
            }
        } else {
            pDevice->onData(pDevice, pFramesOut, pFramesIn, frameCount);
        }

        /* Volume control and clipping for playback devices. */
        if (pFramesOut != NULL) {
            if (masterVolumeFactor < 1) {
                if (pFramesIn == NULL) {    /* <-- In full-duplex situations, the volume will have been applied to the input samples before the data callback. Applying it again post-callback will incorrectly compound it. */
                    ma_apply_volume_factor_pcm_frames(pFramesOut, frameCount, pDevice->playback.format, pDevice->playback.channels, masterVolumeFactor);
                }
            }

            if (!pDevice->noClip && pDevice->playback.format == ma_format_f32) {
                ma_clip_pcm_frames_f32((float*)pFramesOut, frameCount, pDevice->playback.channels);
            }
        }
    }
}



/* A helper function for reading sample data from the client. */
static void ma_device__read_frames_from_client(ma_device* pDevice, ma_uint32 frameCount, void* pFramesOut)
{
    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(frameCount > 0);
    MA_ASSERT(pFramesOut != NULL);

    if (pDevice->playback.converter.isPassthrough) {
        ma_device__on_data(pDevice, pFramesOut, NULL, frameCount);
    } else {
        ma_result result;
        ma_uint64 totalFramesReadOut;
        ma_uint64 totalFramesReadIn;
        void* pRunningFramesOut;

        totalFramesReadOut = 0;
        totalFramesReadIn  = 0;
        pRunningFramesOut  = pFramesOut;

        while (totalFramesReadOut < frameCount) {
            ma_uint8 pIntermediaryBuffer[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];  /* In client format. */
            ma_uint64 intermediaryBufferCap = sizeof(pIntermediaryBuffer) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
            ma_uint64 framesToReadThisIterationIn;
            ma_uint64 framesReadThisIterationIn;
            ma_uint64 framesToReadThisIterationOut;
            ma_uint64 framesReadThisIterationOut;
            ma_uint64 requiredInputFrameCount;

            framesToReadThisIterationOut = (frameCount - totalFramesReadOut);
            framesToReadThisIterationIn = framesToReadThisIterationOut;
            if (framesToReadThisIterationIn > intermediaryBufferCap) {
                framesToReadThisIterationIn = intermediaryBufferCap;
            }

            requiredInputFrameCount = ma_data_converter_get_required_input_frame_count(&pDevice->playback.converter, framesToReadThisIterationOut);
            if (framesToReadThisIterationIn > requiredInputFrameCount) {
                framesToReadThisIterationIn = requiredInputFrameCount;
            }

            if (framesToReadThisIterationIn > 0) {
                ma_device__on_data(pDevice, pIntermediaryBuffer, NULL, (ma_uint32)framesToReadThisIterationIn);
                totalFramesReadIn += framesToReadThisIterationIn;
            }

            /*
            At this point we have our decoded data in input format and now we need to convert to output format. Note that even if we didn't read any
            input frames, we still want to try processing frames because there may some output frames generated from cached input data.
            */
            framesReadThisIterationIn  = framesToReadThisIterationIn;
            framesReadThisIterationOut = framesToReadThisIterationOut;
            result = ma_data_converter_process_pcm_frames(&pDevice->playback.converter, pIntermediaryBuffer, &framesReadThisIterationIn, pRunningFramesOut, &framesReadThisIterationOut);
            if (result != MA_SUCCESS) {
                break;
            }

            totalFramesReadOut += framesReadThisIterationOut;
            pRunningFramesOut   = ma_offset_ptr(pRunningFramesOut, framesReadThisIterationOut * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));

            if (framesReadThisIterationIn == 0 && framesReadThisIterationOut == 0) {
                break;  /* We're done. */
            }
        }
    }
}

/* A helper for sending sample data to the client. */
static void ma_device__send_frames_to_client(ma_device* pDevice, ma_uint32 frameCountInDeviceFormat, const void* pFramesInDeviceFormat)
{
    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(frameCountInDeviceFormat > 0);
    MA_ASSERT(pFramesInDeviceFormat != NULL);

    if (pDevice->capture.converter.isPassthrough) {
        ma_device__on_data(pDevice, NULL, pFramesInDeviceFormat, frameCountInDeviceFormat);
    } else {
        ma_result result;
        ma_uint8 pFramesInClientFormat[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
        ma_uint64 framesInClientFormatCap = sizeof(pFramesInClientFormat) / ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
        ma_uint64 totalDeviceFramesProcessed = 0;
        ma_uint64 totalClientFramesProcessed = 0;
        const void* pRunningFramesInDeviceFormat = pFramesInDeviceFormat;

        /* We just keep going until we've exhaused all of our input frames and cannot generate any more output frames. */
        for (;;) {
            ma_uint64 deviceFramesProcessedThisIteration;
            ma_uint64 clientFramesProcessedThisIteration;

            deviceFramesProcessedThisIteration = (frameCountInDeviceFormat - totalDeviceFramesProcessed);
            clientFramesProcessedThisIteration = framesInClientFormatCap;

            result = ma_data_converter_process_pcm_frames(&pDevice->capture.converter, pRunningFramesInDeviceFormat, &deviceFramesProcessedThisIteration, pFramesInClientFormat, &clientFramesProcessedThisIteration);
            if (result != MA_SUCCESS) {
                break;
            }

            if (clientFramesProcessedThisIteration > 0) {
                ma_device__on_data(pDevice, NULL, pFramesInClientFormat, (ma_uint32)clientFramesProcessedThisIteration);    /* Safe cast. */
            }

            pRunningFramesInDeviceFormat = ma_offset_ptr(pRunningFramesInDeviceFormat, deviceFramesProcessedThisIteration * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));
            totalDeviceFramesProcessed  += deviceFramesProcessedThisIteration;
            totalClientFramesProcessed  += clientFramesProcessedThisIteration;

            if (deviceFramesProcessedThisIteration == 0 && clientFramesProcessedThisIteration == 0) {
                break;  /* We're done. */
            }
        }
    }
}


/* We only want to expose ma_device__handle_duplex_callback_capture() and ma_device__handle_duplex_callback_playback() if we have an asynchronous backend enabled. */
#if defined(MA_HAS_JACK)      || \
    defined(MA_HAS_COREAUDIO) || \
    defined(MA_HAS_AAUDIO)    || \
    defined(MA_HAS_OPENSL)    || \
    defined(MA_HAS_WEBAUDIO)
static ma_result ma_device__handle_duplex_callback_capture(ma_device* pDevice, ma_uint32 frameCountInDeviceFormat, const void* pFramesInDeviceFormat, ma_pcm_rb* pRB)
{
    ma_result result;
    ma_uint32 totalDeviceFramesProcessed = 0;
    const void* pRunningFramesInDeviceFormat = pFramesInDeviceFormat;

    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(frameCountInDeviceFormat > 0);
    MA_ASSERT(pFramesInDeviceFormat != NULL);
    MA_ASSERT(pRB != NULL);
    
    /* Write to the ring buffer. The ring buffer is in the client format which means we need to convert. */
    for (;;) {
        ma_uint32 framesToProcessInDeviceFormat = (frameCountInDeviceFormat - totalDeviceFramesProcessed);
        ma_uint32 framesToProcessInClientFormat = MA_DATA_CONVERTER_STACK_BUFFER_SIZE / ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
        ma_uint64 framesProcessedInDeviceFormat;
        ma_uint64 framesProcessedInClientFormat;
        void* pFramesInClientFormat;

        result = ma_pcm_rb_acquire_write(pRB, &framesToProcessInClientFormat, &pFramesInClientFormat);
        if (result != MA_SUCCESS) {
            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "Failed to acquire capture PCM frames from ring buffer.", result);
            break;
        }

        if (framesToProcessInClientFormat == 0) {
            if (ma_pcm_rb_pointer_distance(pRB) == (ma_int32)ma_pcm_rb_get_subbuffer_size(pRB)) {
                break;  /* Overrun. Not enough room in the ring buffer for input frame. Excess frames are dropped. */
            }
        }

        /* Convert. */
        framesProcessedInDeviceFormat = framesToProcessInDeviceFormat;
        framesProcessedInClientFormat = framesToProcessInClientFormat;
        result = ma_data_converter_process_pcm_frames(&pDevice->capture.converter, pRunningFramesInDeviceFormat, &framesProcessedInDeviceFormat, pFramesInClientFormat, &framesProcessedInClientFormat);
        if (result != MA_SUCCESS) {
            break;
        }

        result = ma_pcm_rb_commit_write(pRB, (ma_uint32)framesProcessedInClientFormat, pFramesInClientFormat);  /* Safe cast. */
        if (result != MA_SUCCESS) {
            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "Failed to commit capture PCM frames to ring buffer.", result);
            break;
        }

        pRunningFramesInDeviceFormat = ma_offset_ptr(pRunningFramesInDeviceFormat, framesProcessedInDeviceFormat * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));
        totalDeviceFramesProcessed += (ma_uint32)framesProcessedInDeviceFormat; /* Safe cast. */

        /* We're done when we're unable to process any client nor device frames. */
        if (framesProcessedInClientFormat == 0 && framesProcessedInDeviceFormat == 0) {
            break;  /* Done. */
        }
    }

    return MA_SUCCESS;
}

static ma_result ma_device__handle_duplex_callback_playback(ma_device* pDevice, ma_uint32 frameCount, void* pFramesInInternalFormat, ma_pcm_rb* pRB)
{
    ma_result result;
    ma_uint8 playbackFramesInExternalFormat[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
    ma_uint8 silentInputFrames[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
    ma_uint32 totalFramesToReadFromClient;
    ma_uint32 totalFramesReadFromClient;
    ma_uint32 totalFramesReadOut = 0;

    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(frameCount > 0);
    MA_ASSERT(pFramesInInternalFormat != NULL);
    MA_ASSERT(pRB != NULL);
    
    /*
    Sitting in the ring buffer should be captured data from the capture callback in external format. If there's not enough data in there for
    the whole frameCount frames we just use silence instead for the input data.
    */
    MA_ZERO_MEMORY(silentInputFrames, sizeof(silentInputFrames));

    /* We need to calculate how many output frames are required to be read from the client to completely fill frameCount internal frames. */
    totalFramesToReadFromClient = (ma_uint32)ma_data_converter_get_required_input_frame_count(&pDevice->playback.converter, frameCount);
    totalFramesReadFromClient = 0;
    while (totalFramesReadFromClient < totalFramesToReadFromClient && ma_device_is_started(pDevice)) {
        ma_uint32 framesRemainingFromClient;
        ma_uint32 framesToProcessFromClient;
        ma_uint32 inputFrameCount;
        void* pInputFrames;

        framesRemainingFromClient = (totalFramesToReadFromClient - totalFramesReadFromClient);
        framesToProcessFromClient = sizeof(playbackFramesInExternalFormat) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
        if (framesToProcessFromClient > framesRemainingFromClient) {
            framesToProcessFromClient = framesRemainingFromClient;
        }

        /* We need to grab captured samples before firing the callback. If there's not enough input samples we just pass silence. */
        inputFrameCount = framesToProcessFromClient;
        result = ma_pcm_rb_acquire_read(pRB, &inputFrameCount, &pInputFrames);
        if (result == MA_SUCCESS) {
            if (inputFrameCount > 0) {
                /* Use actual input frames. */
                ma_device__on_data(pDevice, playbackFramesInExternalFormat, pInputFrames, inputFrameCount);
            } else {
                if (ma_pcm_rb_pointer_distance(pRB) == 0) {
                    break;  /* Underrun. */
                }
            }
            
            /* We're done with the captured samples. */
            result = ma_pcm_rb_commit_read(pRB, inputFrameCount, pInputFrames);
            if (result != MA_SUCCESS) {
                break; /* Don't know what to do here... Just abandon ship. */
            }
        } else {
            /* Use silent input frames. */
            inputFrameCount = ma_min(
                sizeof(playbackFramesInExternalFormat) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels),
                sizeof(silentInputFrames)              / ma_get_bytes_per_frame(pDevice->capture.format,  pDevice->capture.channels)
            );

            ma_device__on_data(pDevice, playbackFramesInExternalFormat, silentInputFrames, inputFrameCount);
        }

        /* We have samples in external format so now we need to convert to internal format and output to the device. */
        {
            ma_uint64 framesConvertedIn  = inputFrameCount;
            ma_uint64 framesConvertedOut = (frameCount - totalFramesReadOut);
            ma_data_converter_process_pcm_frames(&pDevice->playback.converter, playbackFramesInExternalFormat, &framesConvertedIn, pFramesInInternalFormat, &framesConvertedOut);

            totalFramesReadFromClient += (ma_uint32)framesConvertedIn;  /* Safe cast. */
            totalFramesReadOut        += (ma_uint32)framesConvertedOut; /* Safe cast. */
            pFramesInInternalFormat    = ma_offset_ptr(pFramesInInternalFormat, framesConvertedOut * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));
        }
    }

    return MA_SUCCESS;
}
#endif  /* Asynchronous backends. */

/* A helper for changing the state of the device. */
static MA_INLINE void ma_device__set_state(ma_device* pDevice, ma_uint32 newState)
{
    c89atomic_exchange_32(&pDevice->state, newState);
}

/* A helper for getting the state of the device. */
static MA_INLINE ma_uint32 ma_device__get_state(ma_device* pDevice)
{
    return pDevice->state;
}


#ifdef MA_WIN32
    GUID MA_GUID_KSDATAFORMAT_SUBTYPE_PCM        = {0x00000001, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};
    GUID MA_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT = {0x00000003, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};
    /*GUID MA_GUID_KSDATAFORMAT_SUBTYPE_ALAW       = {0x00000006, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};*/
    /*GUID MA_GUID_KSDATAFORMAT_SUBTYPE_MULAW      = {0x00000007, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};*/
#endif


typedef struct
{
    ma_device_type deviceType;
    const ma_device_id* pDeviceID;
    char* pName;
    size_t nameBufferSize;
    ma_bool32 foundDevice;
} ma_context__try_get_device_name_by_id__enum_callback_data;

static ma_bool32 ma_context__try_get_device_name_by_id__enum_callback(ma_context* pContext, ma_device_type deviceType, const ma_device_info* pDeviceInfo, void* pUserData)
{
    ma_context__try_get_device_name_by_id__enum_callback_data* pData = (ma_context__try_get_device_name_by_id__enum_callback_data*)pUserData;
    MA_ASSERT(pData != NULL);

    if (pData->deviceType == deviceType) {
        if (pContext->onDeviceIDEqual(pContext, pData->pDeviceID, &pDeviceInfo->id)) {
            ma_strncpy_s(pData->pName, pData->nameBufferSize, pDeviceInfo->name, (size_t)-1);
            pData->foundDevice = MA_TRUE;
        }
    }

    return !pData->foundDevice;
}

/*
Generic function for retrieving the name of a device by it's ID.

This function simply enumerates every device and then retrieves the name of the first device that has the same ID.
*/
static ma_result ma_context__try_get_device_name_by_id(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, char* pName, size_t nameBufferSize)
{
    ma_result result;
    ma_context__try_get_device_name_by_id__enum_callback_data data;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pName != NULL);

    if (pDeviceID == NULL) {
        return MA_NO_DEVICE;
    }

    data.deviceType = deviceType;
    data.pDeviceID = pDeviceID;
    data.pName = pName;
    data.nameBufferSize = nameBufferSize;
    data.foundDevice = MA_FALSE;
    result = ma_context_enumerate_devices(pContext, ma_context__try_get_device_name_by_id__enum_callback, &data);
    if (result != MA_SUCCESS) {
        return result;
    }

    if (!data.foundDevice) {
        return MA_NO_DEVICE;
    } else {
        return MA_SUCCESS;
    }
}


MA_API ma_uint32 ma_get_format_priority_index(ma_format format) /* Lower = better. */
{
    ma_uint32 i;
    for (i = 0; i < ma_countof(g_maFormatPriorities); ++i) {
        if (g_maFormatPriorities[i] == format) {
            return i;
        }
    }

    /* Getting here means the format could not be found or is equal to ma_format_unknown. */
    return (ma_uint32)-1;
}

static ma_result ma_device__post_init_setup(ma_device* pDevice, ma_device_type deviceType);


/*******************************************************************************

Null Backend

*******************************************************************************/
#ifdef MA_HAS_NULL

#define MA_DEVICE_OP_NONE__NULL    0
#define MA_DEVICE_OP_START__NULL   1
#define MA_DEVICE_OP_SUSPEND__NULL 2
#define MA_DEVICE_OP_KILL__NULL    3

static ma_thread_result MA_THREADCALL ma_device_thread__null(void* pData)
{
    ma_device* pDevice = (ma_device*)pData;
    MA_ASSERT(pDevice != NULL);

    for (;;) {  /* Keep the thread alive until the device is uninitialized. */
        /* Wait for an operation to be requested. */
        ma_event_wait(&pDevice->null_device.operationEvent);

        /* At this point an event should have been triggered. */

        /* Starting the device needs to put the thread into a loop. */
        if (pDevice->null_device.operation == MA_DEVICE_OP_START__NULL) {
            c89atomic_exchange_32(&pDevice->null_device.operation, MA_DEVICE_OP_NONE__NULL);

            /* Reset the timer just in case. */
            ma_timer_init(&pDevice->null_device.timer);

            /* Keep looping until an operation has been requested. */
            while (pDevice->null_device.operation != MA_DEVICE_OP_NONE__NULL && pDevice->null_device.operation != MA_DEVICE_OP_START__NULL) {
                ma_sleep(10); /* Don't hog the CPU. */
            }

            /* Getting here means a suspend or kill operation has been requested. */
            c89atomic_exchange_32((c89atomic_uint32*)&pDevice->null_device.operationResult, MA_SUCCESS);
            ma_event_signal(&pDevice->null_device.operationCompletionEvent);
            continue;
        }

        /* Suspending the device means we need to stop the timer and just continue the loop. */
        if (pDevice->null_device.operation == MA_DEVICE_OP_SUSPEND__NULL) {
            c89atomic_exchange_32(&pDevice->null_device.operation, MA_DEVICE_OP_NONE__NULL);

            /* We need to add the current run time to the prior run time, then reset the timer. */
            pDevice->null_device.priorRunTime += ma_timer_get_time_in_seconds(&pDevice->null_device.timer);
            ma_timer_init(&pDevice->null_device.timer);

            /* We're done. */
            c89atomic_exchange_32((c89atomic_uint32*)&pDevice->null_device.operationResult, MA_SUCCESS);
            ma_event_signal(&pDevice->null_device.operationCompletionEvent);
            continue;
        }

        /* Killing the device means we need to get out of this loop so that this thread can terminate. */
        if (pDevice->null_device.operation == MA_DEVICE_OP_KILL__NULL) {
            c89atomic_exchange_32(&pDevice->null_device.operation, MA_DEVICE_OP_NONE__NULL);
            c89atomic_exchange_32((c89atomic_uint32*)&pDevice->null_device.operationResult, MA_SUCCESS);
            ma_event_signal(&pDevice->null_device.operationCompletionEvent);
            break;
        }

        /* Getting a signal on a "none" operation probably means an error. Return invalid operation. */
        if (pDevice->null_device.operation == MA_DEVICE_OP_NONE__NULL) {
            MA_ASSERT(MA_FALSE);  /* <-- Trigger this in debug mode to ensure developers are aware they're doing something wrong (or there's a bug in a miniaudio). */
            c89atomic_exchange_32((c89atomic_uint32*)&pDevice->null_device.operationResult, (c89atomic_uint32)MA_INVALID_OPERATION);
            ma_event_signal(&pDevice->null_device.operationCompletionEvent);
            continue;   /* Continue the loop. Don't terminate. */
        }
    }

    return (ma_thread_result)0;
}

static ma_result ma_device_do_operation__null(ma_device* pDevice, ma_uint32 operation)
{
    c89atomic_exchange_32(&pDevice->null_device.operation, operation);
    if (ma_event_signal(&pDevice->null_device.operationEvent) != MA_SUCCESS) {
        return MA_ERROR;
    }

    if (ma_event_wait(&pDevice->null_device.operationCompletionEvent) != MA_SUCCESS) {
        return MA_ERROR;
    }

    return pDevice->null_device.operationResult;
}

static ma_uint64 ma_device_get_total_run_time_in_frames__null(ma_device* pDevice)
{
    ma_uint32 internalSampleRate;
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        internalSampleRate = pDevice->capture.internalSampleRate;
    } else {
        internalSampleRate = pDevice->playback.internalSampleRate;
    }


    return (ma_uint64)((pDevice->null_device.priorRunTime + ma_timer_get_time_in_seconds(&pDevice->null_device.timer)) * internalSampleRate);
}

static ma_bool32 ma_context_is_device_id_equal__null(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pID0 != NULL);
    MA_ASSERT(pID1 != NULL);
    (void)pContext;

    return pID0->nullbackend == pID1->nullbackend;
}

static ma_result ma_context_enumerate_devices__null(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_bool32 cbResult = MA_TRUE;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(callback != NULL);

    /* Playback. */
    if (cbResult) {
        ma_device_info deviceInfo;
        MA_ZERO_OBJECT(&deviceInfo);
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), "NULL Playback Device", (size_t)-1);
        cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
    }

    /* Capture. */
    if (cbResult) {
        ma_device_info deviceInfo;
        MA_ZERO_OBJECT(&deviceInfo);
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), "NULL Capture Device", (size_t)-1);
        cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
    }

    return MA_SUCCESS;
}

static ma_result ma_context_get_device_info__null(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_uint32 iFormat;

    MA_ASSERT(pContext != NULL);

    if (pDeviceID != NULL && pDeviceID->nullbackend != 0) {
        return MA_NO_DEVICE;   /* Don't know the device. */
    }

    /* Name / Description */
    if (deviceType == ma_device_type_playback) {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), "NULL Playback Device", (size_t)-1);
    } else {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), "NULL Capture Device", (size_t)-1);
    }

    /* Support everything on the null backend. */
    pDeviceInfo->formatCount = ma_format_count - 1;    /* Minus one because we don't want to include ma_format_unknown. */
    for (iFormat = 0; iFormat < pDeviceInfo->formatCount; ++iFormat) {
        pDeviceInfo->formats[iFormat] = (ma_format)(iFormat + 1);  /* +1 to skip over ma_format_unknown. */
    }

    pDeviceInfo->minChannels   = 1;
    pDeviceInfo->maxChannels   = MA_MAX_CHANNELS;
    pDeviceInfo->minSampleRate = MA_SAMPLE_RATE_8000;
    pDeviceInfo->maxSampleRate = MA_SAMPLE_RATE_384000;

    (void)pContext;
    (void)shareMode;
    return MA_SUCCESS;
}


static void ma_device_uninit__null(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    /* Keep it clean and wait for the device thread to finish before returning. */
    ma_device_do_operation__null(pDevice, MA_DEVICE_OP_KILL__NULL);

    /* At this point the loop in the device thread is as good as terminated so we can uninitialize our events. */
    ma_event_uninit(&pDevice->null_device.operationCompletionEvent);
    ma_event_uninit(&pDevice->null_device.operationEvent);
}

static ma_result ma_device_init__null(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result;
    ma_uint32 periodSizeInFrames;

    MA_ASSERT(pDevice != NULL);

    MA_ZERO_OBJECT(&pDevice->null_device);

    if (pConfig->deviceType == ma_device_type_loopback) {
        return MA_DEVICE_TYPE_NOT_SUPPORTED;
    }

    periodSizeInFrames = pConfig->periodSizeInFrames;
    if (periodSizeInFrames == 0) {
        periodSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(pConfig->periodSizeInMilliseconds, pConfig->sampleRate);
    }

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_strncpy_s(pDevice->capture.name,  sizeof(pDevice->capture.name),  "NULL Capture Device",  (size_t)-1);
        pDevice->capture.internalFormat             = pConfig->capture.format;
        pDevice->capture.internalChannels           = pConfig->capture.channels;
        ma_channel_map_copy(pDevice->capture.internalChannelMap, pConfig->capture.channelMap, ma_min(pConfig->capture.channels, MA_MAX_CHANNELS));
        pDevice->capture.internalPeriodSizeInFrames = periodSizeInFrames;
        pDevice->capture.internalPeriods            = pConfig->periods;
    }
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_strncpy_s(pDevice->playback.name, sizeof(pDevice->playback.name), "NULL Playback Device", (size_t)-1);
        pDevice->playback.internalFormat             = pConfig->playback.format;
        pDevice->playback.internalChannels           = pConfig->playback.channels;
        ma_channel_map_copy(pDevice->playback.internalChannelMap, pConfig->playback.channelMap, ma_min(pConfig->playback.channels, MA_MAX_CHANNELS));
        pDevice->playback.internalPeriodSizeInFrames = periodSizeInFrames;
        pDevice->playback.internalPeriods            = pConfig->periods;
    }

    /*
    In order to get timing right, we need to create a thread that does nothing but keeps track of the timer. This timer is started when the
    first period is "written" to it, and then stopped in ma_device_stop__null().
    */
    result = ma_event_init(&pDevice->null_device.operationEvent);
    if (result != MA_SUCCESS) {
        return result;
    }

    result = ma_event_init(&pDevice->null_device.operationCompletionEvent);
    if (result != MA_SUCCESS) {
        return result;
    }

    result = ma_thread_create(&pDevice->thread, pContext->threadPriority, 0, ma_device_thread__null, pDevice);
    if (result != MA_SUCCESS) {
        return result;
    }

    return MA_SUCCESS;
}

static ma_result ma_device_start__null(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    ma_device_do_operation__null(pDevice, MA_DEVICE_OP_START__NULL);

    c89atomic_exchange_32(&pDevice->null_device.isStarted, MA_TRUE);
    return MA_SUCCESS;
}

static ma_result ma_device_stop__null(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    ma_device_do_operation__null(pDevice, MA_DEVICE_OP_SUSPEND__NULL);

    c89atomic_exchange_32(&pDevice->null_device.isStarted, MA_FALSE);
    return MA_SUCCESS;
}

static ma_result ma_device_write__null(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount, ma_uint32* pFramesWritten)
{
    ma_result result = MA_SUCCESS;
    ma_uint32 totalPCMFramesProcessed;
    ma_bool32 wasStartedOnEntry;

    if (pFramesWritten != NULL) {
        *pFramesWritten = 0;
    }

    wasStartedOnEntry = pDevice->null_device.isStarted;

    /* Keep going until everything has been read. */
    totalPCMFramesProcessed = 0;
    while (totalPCMFramesProcessed < frameCount) {
        ma_uint64 targetFrame;

        /* If there are any frames remaining in the current period, consume those first. */
        if (pDevice->null_device.currentPeriodFramesRemainingPlayback > 0) {
            ma_uint32 framesRemaining = (frameCount - totalPCMFramesProcessed);
            ma_uint32 framesToProcess = pDevice->null_device.currentPeriodFramesRemainingPlayback;
            if (framesToProcess > framesRemaining) {
                framesToProcess = framesRemaining;
            }

            /* We don't actually do anything with pPCMFrames, so just mark it as unused to prevent a warning. */
            (void)pPCMFrames;

            pDevice->null_device.currentPeriodFramesRemainingPlayback -= framesToProcess;
            totalPCMFramesProcessed += framesToProcess;
        }

        /* If we've consumed the current period we'll need to mark it as such an ensure the device is started if it's not already. */
        if (pDevice->null_device.currentPeriodFramesRemainingPlayback == 0) {
            pDevice->null_device.currentPeriodFramesRemainingPlayback = 0;

            if (!pDevice->null_device.isStarted && !wasStartedOnEntry) {
                result = ma_device_start__null(pDevice);
                if (result != MA_SUCCESS) {
                    break;
                }
            }
        }

        /* If we've consumed the whole buffer we can return now. */
        MA_ASSERT(totalPCMFramesProcessed <= frameCount);
        if (totalPCMFramesProcessed == frameCount) {
            break;
        }

        /* Getting here means we've still got more frames to consume, we but need to wait for it to become available. */
        targetFrame = pDevice->null_device.lastProcessedFramePlayback;
        for (;;) {
            ma_uint64 currentFrame;

            /* Stop waiting if the device has been stopped. */
            if (!pDevice->null_device.isStarted) {
                break;
            }

            currentFrame = ma_device_get_total_run_time_in_frames__null(pDevice);
            if (currentFrame >= targetFrame) {
                break;
            }

            /* Getting here means we haven't yet reached the target sample, so continue waiting. */
            ma_sleep(10);
        }

        pDevice->null_device.lastProcessedFramePlayback          += pDevice->playback.internalPeriodSizeInFrames;
        pDevice->null_device.currentPeriodFramesRemainingPlayback = pDevice->playback.internalPeriodSizeInFrames;
    }

    if (pFramesWritten != NULL) {
        *pFramesWritten = totalPCMFramesProcessed;
    }

    return result;
}

static ma_result ma_device_read__null(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount, ma_uint32* pFramesRead)
{
    ma_result result = MA_SUCCESS;
    ma_uint32 totalPCMFramesProcessed;

    if (pFramesRead != NULL) {
        *pFramesRead = 0;
    }

    /* Keep going until everything has been read. */
    totalPCMFramesProcessed = 0;
    while (totalPCMFramesProcessed < frameCount) {
        ma_uint64 targetFrame;

        /* If there are any frames remaining in the current period, consume those first. */
        if (pDevice->null_device.currentPeriodFramesRemainingCapture > 0) {
            ma_uint32 bpf = ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
            ma_uint32 framesRemaining = (frameCount - totalPCMFramesProcessed);
            ma_uint32 framesToProcess = pDevice->null_device.currentPeriodFramesRemainingCapture;
            if (framesToProcess > framesRemaining) {
                framesToProcess = framesRemaining;
            }

            /* We need to ensure the output buffer is zeroed. */
            MA_ZERO_MEMORY(ma_offset_ptr(pPCMFrames, totalPCMFramesProcessed*bpf), framesToProcess*bpf);

            pDevice->null_device.currentPeriodFramesRemainingCapture -= framesToProcess;
            totalPCMFramesProcessed += framesToProcess;
        }

        /* If we've consumed the current period we'll need to mark it as such an ensure the device is started if it's not already. */
        if (pDevice->null_device.currentPeriodFramesRemainingCapture == 0) {
            pDevice->null_device.currentPeriodFramesRemainingCapture = 0;
        }

        /* If we've consumed the whole buffer we can return now. */
        MA_ASSERT(totalPCMFramesProcessed <= frameCount);
        if (totalPCMFramesProcessed == frameCount) {
            break;
        }

        /* Getting here means we've still got more frames to consume, we but need to wait for it to become available. */
        targetFrame = pDevice->null_device.lastProcessedFrameCapture + pDevice->capture.internalPeriodSizeInFrames;
        for (;;) {
            ma_uint64 currentFrame;

            /* Stop waiting if the device has been stopped. */
            if (!pDevice->null_device.isStarted) {
                break;
            }

            currentFrame = ma_device_get_total_run_time_in_frames__null(pDevice);
            if (currentFrame >= targetFrame) {
                break;
            }

            /* Getting here means we haven't yet reached the target sample, so continue waiting. */
            ma_sleep(10);
        }

        pDevice->null_device.lastProcessedFrameCapture          += pDevice->capture.internalPeriodSizeInFrames;
        pDevice->null_device.currentPeriodFramesRemainingCapture = pDevice->capture.internalPeriodSizeInFrames;
    }

    if (pFramesRead != NULL) {
        *pFramesRead = totalPCMFramesProcessed;
    }

    return result;
}

static ma_result ma_device_main_loop__null(ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;
    ma_bool32 exitLoop = MA_FALSE;
    ma_uint8  capturedDeviceData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
    ma_uint8  playbackDeviceData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
    ma_uint32 capturedDeviceDataCapInFrames = sizeof(capturedDeviceData) / ma_get_bytes_per_frame(pDevice->capture.internalFormat,  pDevice->capture.internalChannels);
    ma_uint32 playbackDeviceDataCapInFrames = sizeof(playbackDeviceData) / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
    
    MA_ASSERT(pDevice != NULL);

    /* The capture device needs to be started immediately. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        result = ma_device_start__null(pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    while (ma_device__get_state(pDevice) == MA_STATE_STARTED && !exitLoop) {
        switch (pDevice->type)
        {
            case ma_device_type_duplex:
            {
                /* The process is: device_read -> convert -> callback -> convert -> device_write */
                ma_uint32 totalCapturedDeviceFramesProcessed = 0;
                ma_uint32 capturedDevicePeriodSizeInFrames = ma_min(pDevice->capture.internalPeriodSizeInFrames, pDevice->playback.internalPeriodSizeInFrames);
                    
                while (totalCapturedDeviceFramesProcessed < capturedDevicePeriodSizeInFrames) {
                    ma_uint32 capturedDeviceFramesRemaining;
                    ma_uint32 capturedDeviceFramesProcessed;
                    ma_uint32 capturedDeviceFramesToProcess;
                    ma_uint32 capturedDeviceFramesToTryProcessing = capturedDevicePeriodSizeInFrames - totalCapturedDeviceFramesProcessed;
                    if (capturedDeviceFramesToTryProcessing > capturedDeviceDataCapInFrames) {
                        capturedDeviceFramesToTryProcessing = capturedDeviceDataCapInFrames;
                    }

                    result = ma_device_read__null(pDevice, capturedDeviceData, capturedDeviceFramesToTryProcessing, &capturedDeviceFramesToProcess);
                    if (result != MA_SUCCESS) {
                        exitLoop = MA_TRUE;
                        break;
                    }

                    capturedDeviceFramesRemaining = capturedDeviceFramesToProcess;
                    capturedDeviceFramesProcessed = 0;

                    /* At this point we have our captured data in device format and we now need to convert it to client format. */
                    for (;;) {
                        ma_uint8  capturedClientData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                        ma_uint8  playbackClientData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                        ma_uint32 capturedClientDataCapInFrames = sizeof(capturedClientData) / ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
                        ma_uint32 playbackClientDataCapInFrames = sizeof(playbackClientData) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
                        ma_uint64 capturedClientFramesToProcessThisIteration = ma_min(capturedClientDataCapInFrames, playbackClientDataCapInFrames);
                        ma_uint64 capturedDeviceFramesToProcessThisIteration = capturedDeviceFramesRemaining;
                        ma_uint8* pRunningCapturedDeviceFrames = ma_offset_ptr(capturedDeviceData, capturedDeviceFramesProcessed * ma_get_bytes_per_frame(pDevice->capture.internalFormat,  pDevice->capture.internalChannels));
                        
                        /* Convert capture data from device format to client format. */
                        result = ma_data_converter_process_pcm_frames(&pDevice->capture.converter, pRunningCapturedDeviceFrames, &capturedDeviceFramesToProcessThisIteration, capturedClientData, &capturedClientFramesToProcessThisIteration);
                        if (result != MA_SUCCESS) {
                            break;
                        }

                        /*
                        If we weren't able to generate any output frames it must mean we've exhaused all of our input. The only time this would not be the case is if capturedClientData was too small
                        which should never be the case when it's of the size MA_DATA_CONVERTER_STACK_BUFFER_SIZE.
                        */
                        if (capturedClientFramesToProcessThisIteration == 0) {
                            break;
                        }

                        ma_device__on_data(pDevice, playbackClientData, capturedClientData, (ma_uint32)capturedClientFramesToProcessThisIteration);    /* Safe cast .*/

                        capturedDeviceFramesProcessed += (ma_uint32)capturedDeviceFramesToProcessThisIteration; /* Safe cast. */
                        capturedDeviceFramesRemaining -= (ma_uint32)capturedDeviceFramesToProcessThisIteration; /* Safe cast. */

                        /* At this point the playbackClientData buffer should be holding data that needs to be written to the device. */
                        for (;;) {
                            ma_uint64 convertedClientFrameCount = capturedClientFramesToProcessThisIteration;
                            ma_uint64 convertedDeviceFrameCount = playbackDeviceDataCapInFrames;
                            result = ma_data_converter_process_pcm_frames(&pDevice->playback.converter, playbackClientData, &convertedClientFrameCount, playbackDeviceData, &convertedDeviceFrameCount);
                            if (result != MA_SUCCESS) {
                                break;
                            }

                            result = ma_device_write__null(pDevice, playbackDeviceData, (ma_uint32)convertedDeviceFrameCount, NULL);   /* Safe cast. */
                            if (result != MA_SUCCESS) {
                                exitLoop = MA_TRUE;
                                break;
                            }

                            capturedClientFramesToProcessThisIteration -= (ma_uint32)convertedClientFrameCount;  /* Safe cast. */
                            if (capturedClientFramesToProcessThisIteration == 0) {
                                break;
                            }
                        }

                        /* In case an error happened from ma_device_write__null()... */
                        if (result != MA_SUCCESS) {
                            exitLoop = MA_TRUE;
                            break;
                        }
                    }

                    totalCapturedDeviceFramesProcessed += capturedDeviceFramesProcessed;
                }
            } break;

            case ma_device_type_capture:
            {
                ma_uint32 periodSizeInFrames = pDevice->capture.internalPeriodSizeInFrames;
                ma_uint32 framesReadThisPeriod = 0;
                while (framesReadThisPeriod < periodSizeInFrames) {
                    ma_uint32 framesRemainingInPeriod = periodSizeInFrames - framesReadThisPeriod;
                    ma_uint32 framesProcessed;
                    ma_uint32 framesToReadThisIteration = framesRemainingInPeriod;
                    if (framesToReadThisIteration > capturedDeviceDataCapInFrames) {
                        framesToReadThisIteration = capturedDeviceDataCapInFrames;
                    }

                    result = ma_device_read__null(pDevice, capturedDeviceData, framesToReadThisIteration, &framesProcessed);
                    if (result != MA_SUCCESS) {
                        exitLoop = MA_TRUE;
                        break;
                    }

                    ma_device__send_frames_to_client(pDevice, framesProcessed, capturedDeviceData);

                    framesReadThisPeriod += framesProcessed;
                }
            } break;

            case ma_device_type_playback:
            {
                /* We write in chunks of the period size, but use a stack allocated buffer for the intermediary. */
                ma_uint32 periodSizeInFrames = pDevice->playback.internalPeriodSizeInFrames;
                ma_uint32 framesWrittenThisPeriod = 0;
                while (framesWrittenThisPeriod < periodSizeInFrames) {
                    ma_uint32 framesRemainingInPeriod = periodSizeInFrames - framesWrittenThisPeriod;
                    ma_uint32 framesProcessed;
                    ma_uint32 framesToWriteThisIteration = framesRemainingInPeriod;
                    if (framesToWriteThisIteration > playbackDeviceDataCapInFrames) {
                        framesToWriteThisIteration = playbackDeviceDataCapInFrames;
                    }

                    ma_device__read_frames_from_client(pDevice, framesToWriteThisIteration, playbackDeviceData);

                    result = ma_device_write__null(pDevice, playbackDeviceData, framesToWriteThisIteration, &framesProcessed);
                    if (result != MA_SUCCESS) {
                        exitLoop = MA_TRUE;
                        break;
                    }

                    framesWrittenThisPeriod += framesProcessed;
                }
            } break;

            /* To silence a warning. Will never hit this. */
            case ma_device_type_loopback:
            default: break;
        }
    }


    /* Here is where the device is started. */
    ma_device_stop__null(pDevice);

    return result;
}

static ma_result ma_context_uninit__null(ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pContext->backend == ma_backend_null);

    (void)pContext;
    return MA_SUCCESS;
}

static ma_result ma_context_init__null(const ma_context_config* pConfig, ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);

    (void)pConfig;

    pContext->onUninit         = ma_context_uninit__null;
    pContext->onDeviceIDEqual  = ma_context_is_device_id_equal__null;
    pContext->onEnumDevices    = ma_context_enumerate_devices__null;
    pContext->onGetDeviceInfo  = ma_context_get_device_info__null;
    pContext->onDeviceInit     = ma_device_init__null;
    pContext->onDeviceUninit   = ma_device_uninit__null;
    pContext->onDeviceStart    = NULL; /* Not required for synchronous backends. */
    pContext->onDeviceStop     = NULL; /* Not required for synchronous backends. */
    pContext->onDeviceMainLoop = ma_device_main_loop__null;

    /* The null backend always works. */
    return MA_SUCCESS;
}
#endif



/*******************************************************************************

WIN32 COMMON

*******************************************************************************/
#if defined(MA_WIN32)
#if defined(MA_WIN32_DESKTOP)
    #define ma_CoInitializeEx(pContext, pvReserved, dwCoInit)                          ((MA_PFN_CoInitializeEx)pContext->win32.CoInitializeEx)(pvReserved, dwCoInit)
    #define ma_CoUninitialize(pContext)                                                ((MA_PFN_CoUninitialize)pContext->win32.CoUninitialize)()
    #define ma_CoCreateInstance(pContext, rclsid, pUnkOuter, dwClsContext, riid, ppv)  ((MA_PFN_CoCreateInstance)pContext->win32.CoCreateInstance)(rclsid, pUnkOuter, dwClsContext, riid, ppv)
    #define ma_CoTaskMemFree(pContext, pv)                                             ((MA_PFN_CoTaskMemFree)pContext->win32.CoTaskMemFree)(pv)
    #define ma_PropVariantClear(pContext, pvar)                                        ((MA_PFN_PropVariantClear)pContext->win32.PropVariantClear)(pvar)
#else
    #define ma_CoInitializeEx(pContext, pvReserved, dwCoInit)                          CoInitializeEx(pvReserved, dwCoInit)
    #define ma_CoUninitialize(pContext)                                                CoUninitialize()
    #define ma_CoCreateInstance(pContext, rclsid, pUnkOuter, dwClsContext, riid, ppv)  CoCreateInstance(rclsid, pUnkOuter, dwClsContext, riid, ppv)
    #define ma_CoTaskMemFree(pContext, pv)                                             CoTaskMemFree(pv)
    #define ma_PropVariantClear(pContext, pvar)                                        PropVariantClear(pvar)
#endif

#if !defined(MAXULONG_PTR)
typedef size_t DWORD_PTR;
#endif

#if !defined(WAVE_FORMAT_44M08)
#define WAVE_FORMAT_44M08 0x00000100
#define WAVE_FORMAT_44S08 0x00000200
#define WAVE_FORMAT_44M16 0x00000400
#define WAVE_FORMAT_44S16 0x00000800
#define WAVE_FORMAT_48M08 0x00001000
#define WAVE_FORMAT_48S08 0x00002000
#define WAVE_FORMAT_48M16 0x00004000
#define WAVE_FORMAT_48S16 0x00008000
#define WAVE_FORMAT_96M08 0x00010000
#define WAVE_FORMAT_96S08 0x00020000
#define WAVE_FORMAT_96M16 0x00040000
#define WAVE_FORMAT_96S16 0x00080000
#endif

#ifndef SPEAKER_FRONT_LEFT
#define SPEAKER_FRONT_LEFT            0x1
#define SPEAKER_FRONT_RIGHT           0x2
#define SPEAKER_FRONT_CENTER          0x4
#define SPEAKER_LOW_FREQUENCY         0x8
#define SPEAKER_BACK_LEFT             0x10
#define SPEAKER_BACK_RIGHT            0x20
#define SPEAKER_FRONT_LEFT_OF_CENTER  0x40
#define SPEAKER_FRONT_RIGHT_OF_CENTER 0x80
#define SPEAKER_BACK_CENTER           0x100
#define SPEAKER_SIDE_LEFT             0x200
#define SPEAKER_SIDE_RIGHT            0x400
#define SPEAKER_TOP_CENTER            0x800
#define SPEAKER_TOP_FRONT_LEFT        0x1000
#define SPEAKER_TOP_FRONT_CENTER      0x2000
#define SPEAKER_TOP_FRONT_RIGHT       0x4000
#define SPEAKER_TOP_BACK_LEFT         0x8000
#define SPEAKER_TOP_BACK_CENTER       0x10000
#define SPEAKER_TOP_BACK_RIGHT        0x20000
#endif

/*
The SDK that comes with old versions of MSVC (VC6, for example) does not appear to define WAVEFORMATEXTENSIBLE. We
define our own implementation in this case.
*/
#if (defined(_MSC_VER) && !defined(_WAVEFORMATEXTENSIBLE_)) || defined(__DMC__)
typedef struct
{
    WAVEFORMATEX Format;
    union
    {
        WORD wValidBitsPerSample;
        WORD wSamplesPerBlock;
        WORD wReserved;
    } Samples;
    DWORD dwChannelMask;
    GUID SubFormat;
} WAVEFORMATEXTENSIBLE;
#endif

#ifndef WAVE_FORMAT_EXTENSIBLE
#define WAVE_FORMAT_EXTENSIBLE  0xFFFE
#endif

#ifndef WAVE_FORMAT_IEEE_FLOAT
#define WAVE_FORMAT_IEEE_FLOAT  0x0003
#endif

/* Converts an individual Win32-style channel identifier (SPEAKER_FRONT_LEFT, etc.) to miniaudio. */
static ma_uint8 ma_channel_id_to_ma__win32(DWORD id)
{
    switch (id)
    {
        case SPEAKER_FRONT_LEFT:            return MA_CHANNEL_FRONT_LEFT;
        case SPEAKER_FRONT_RIGHT:           return MA_CHANNEL_FRONT_RIGHT;
        case SPEAKER_FRONT_CENTER:          return MA_CHANNEL_FRONT_CENTER;
        case SPEAKER_LOW_FREQUENCY:         return MA_CHANNEL_LFE;
        case SPEAKER_BACK_LEFT:             return MA_CHANNEL_BACK_LEFT;
        case SPEAKER_BACK_RIGHT:            return MA_CHANNEL_BACK_RIGHT;
        case SPEAKER_FRONT_LEFT_OF_CENTER:  return MA_CHANNEL_FRONT_LEFT_CENTER;
        case SPEAKER_FRONT_RIGHT_OF_CENTER: return MA_CHANNEL_FRONT_RIGHT_CENTER;
        case SPEAKER_BACK_CENTER:           return MA_CHANNEL_BACK_CENTER;
        case SPEAKER_SIDE_LEFT:             return MA_CHANNEL_SIDE_LEFT;
        case SPEAKER_SIDE_RIGHT:            return MA_CHANNEL_SIDE_RIGHT;
        case SPEAKER_TOP_CENTER:            return MA_CHANNEL_TOP_CENTER;
        case SPEAKER_TOP_FRONT_LEFT:        return MA_CHANNEL_TOP_FRONT_LEFT;
        case SPEAKER_TOP_FRONT_CENTER:      return MA_CHANNEL_TOP_FRONT_CENTER;
        case SPEAKER_TOP_FRONT_RIGHT:       return MA_CHANNEL_TOP_FRONT_RIGHT;
        case SPEAKER_TOP_BACK_LEFT:         return MA_CHANNEL_TOP_BACK_LEFT;
        case SPEAKER_TOP_BACK_CENTER:       return MA_CHANNEL_TOP_BACK_CENTER;
        case SPEAKER_TOP_BACK_RIGHT:        return MA_CHANNEL_TOP_BACK_RIGHT;
        default: return 0;
    }
}

/* Converts an individual miniaudio channel identifier (MA_CHANNEL_FRONT_LEFT, etc.) to Win32-style. */
static DWORD ma_channel_id_to_win32(DWORD id)
{
    switch (id)
    {
        case MA_CHANNEL_MONO:               return SPEAKER_FRONT_CENTER;
        case MA_CHANNEL_FRONT_LEFT:         return SPEAKER_FRONT_LEFT;
        case MA_CHANNEL_FRONT_RIGHT:        return SPEAKER_FRONT_RIGHT;
        case MA_CHANNEL_FRONT_CENTER:       return SPEAKER_FRONT_CENTER;
        case MA_CHANNEL_LFE:                return SPEAKER_LOW_FREQUENCY;
        case MA_CHANNEL_BACK_LEFT:          return SPEAKER_BACK_LEFT;
        case MA_CHANNEL_BACK_RIGHT:         return SPEAKER_BACK_RIGHT;
        case MA_CHANNEL_FRONT_LEFT_CENTER:  return SPEAKER_FRONT_LEFT_OF_CENTER;
        case MA_CHANNEL_FRONT_RIGHT_CENTER: return SPEAKER_FRONT_RIGHT_OF_CENTER;
        case MA_CHANNEL_BACK_CENTER:        return SPEAKER_BACK_CENTER;
        case MA_CHANNEL_SIDE_LEFT:          return SPEAKER_SIDE_LEFT;
        case MA_CHANNEL_SIDE_RIGHT:         return SPEAKER_SIDE_RIGHT;
        case MA_CHANNEL_TOP_CENTER:         return SPEAKER_TOP_CENTER;
        case MA_CHANNEL_TOP_FRONT_LEFT:     return SPEAKER_TOP_FRONT_LEFT;
        case MA_CHANNEL_TOP_FRONT_CENTER:   return SPEAKER_TOP_FRONT_CENTER;
        case MA_CHANNEL_TOP_FRONT_RIGHT:    return SPEAKER_TOP_FRONT_RIGHT;
        case MA_CHANNEL_TOP_BACK_LEFT:      return SPEAKER_TOP_BACK_LEFT;
        case MA_CHANNEL_TOP_BACK_CENTER:    return SPEAKER_TOP_BACK_CENTER;
        case MA_CHANNEL_TOP_BACK_RIGHT:     return SPEAKER_TOP_BACK_RIGHT;
        default: return 0;
    }
}

/* Converts a channel mapping to a Win32-style channel mask. */
static DWORD ma_channel_map_to_channel_mask__win32(const ma_channel* pChannelMap, ma_uint32 channels)
{
    DWORD dwChannelMask = 0;
    ma_uint32 iChannel;

    for (iChannel = 0; iChannel < channels; ++iChannel) {
        dwChannelMask |= ma_channel_id_to_win32(pChannelMap[iChannel]);
    }

    return dwChannelMask;
}

/* Converts a Win32-style channel mask to a miniaudio channel map. */
static void ma_channel_mask_to_channel_map__win32(DWORD dwChannelMask, ma_uint32 channels, ma_channel* pChannelMap)
{
    if (channels == 1 && dwChannelMask == 0) {
        pChannelMap[0] = MA_CHANNEL_MONO;
    } else if (channels == 2 && dwChannelMask == 0) {
        pChannelMap[0] = MA_CHANNEL_FRONT_LEFT;
        pChannelMap[1] = MA_CHANNEL_FRONT_RIGHT;
    } else {
        if (channels == 1 && (dwChannelMask & SPEAKER_FRONT_CENTER) != 0) {
            pChannelMap[0] = MA_CHANNEL_MONO;
        } else {
            /* Just iterate over each bit. */
            ma_uint32 iChannel = 0;
            ma_uint32 iBit;

            for (iBit = 0; iBit < 32 && iChannel < channels; ++iBit) {
                DWORD bitValue = (dwChannelMask & (1UL << iBit));
                if (bitValue != 0) {
                    /* The bit is set. */
                    pChannelMap[iChannel] = ma_channel_id_to_ma__win32(bitValue);
                    iChannel += 1;
                }
            }
        }
    }
}

#ifdef __cplusplus
static ma_bool32 ma_is_guid_equal(const void* a, const void* b)
{
    return IsEqualGUID(*(const GUID*)a, *(const GUID*)b);
}
#else
#define ma_is_guid_equal(a, b) IsEqualGUID((const GUID*)a, (const GUID*)b)
#endif

static MA_INLINE ma_bool32 ma_is_guid_null(const void* guid)
{
    static GUID nullguid = {0x00000000, 0x0000, 0x0000, {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}};
    return ma_is_guid_equal(guid, &nullguid);
}

static ma_format ma_format_from_WAVEFORMATEX(const WAVEFORMATEX* pWF)
{
    MA_ASSERT(pWF != NULL);

    if (pWF->wFormatTag == WAVE_FORMAT_EXTENSIBLE) {
        const WAVEFORMATEXTENSIBLE* pWFEX = (const WAVEFORMATEXTENSIBLE*)pWF;
        if (ma_is_guid_equal(&pWFEX->SubFormat, &MA_GUID_KSDATAFORMAT_SUBTYPE_PCM)) {
            if (pWFEX->Samples.wValidBitsPerSample == 32) {
                return ma_format_s32;
            }
            if (pWFEX->Samples.wValidBitsPerSample == 24) {
                if (pWFEX->Format.wBitsPerSample == 32) {
                    /*return ma_format_s24_32;*/
                }
                if (pWFEX->Format.wBitsPerSample == 24) {
                    return ma_format_s24;
                }
            }
            if (pWFEX->Samples.wValidBitsPerSample == 16) {
                return ma_format_s16;
            }
            if (pWFEX->Samples.wValidBitsPerSample == 8) {
                return ma_format_u8;
            }
        }
        if (ma_is_guid_equal(&pWFEX->SubFormat, &MA_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT)) {
            if (pWFEX->Samples.wValidBitsPerSample == 32) {
                return ma_format_f32;
            }
            /*
            if (pWFEX->Samples.wValidBitsPerSample == 64) {
                return ma_format_f64;
            }
            */
        }
    } else {
        if (pWF->wFormatTag == WAVE_FORMAT_PCM) {
            if (pWF->wBitsPerSample == 32) {
                return ma_format_s32;
            }
            if (pWF->wBitsPerSample == 24) {
                return ma_format_s24;
            }
            if (pWF->wBitsPerSample == 16) {
                return ma_format_s16;
            }
            if (pWF->wBitsPerSample == 8) {
                return ma_format_u8;
            }
        }
        if (pWF->wFormatTag == WAVE_FORMAT_IEEE_FLOAT) {
            if (pWF->wBitsPerSample == 32) {
                return ma_format_f32;
            }
            if (pWF->wBitsPerSample == 64) {
                /*return ma_format_f64;*/
            }
        }
    }

    return ma_format_unknown;
}
#endif


/*******************************************************************************

WASAPI Backend

*******************************************************************************/
#ifdef MA_HAS_WASAPI
#if 0
#if defined(_MSC_VER)
    #pragma warning(push)
    #pragma warning(disable:4091)   /* 'typedef ': ignored on left of '' when no variable is declared */
#endif
#include <audioclient.h>
#include <mmdeviceapi.h>
#if defined(_MSC_VER)
    #pragma warning(pop)
#endif
#endif  /* 0 */

/* Some compilers don't define VerifyVersionInfoW. Need to write this ourselves. */
#define MA_WIN32_WINNT_VISTA    0x0600
#define MA_VER_MINORVERSION     0x01
#define MA_VER_MAJORVERSION     0x02
#define MA_VER_SERVICEPACKMAJOR 0x20
#define MA_VER_GREATER_EQUAL    0x03

typedef struct  {
    DWORD dwOSVersionInfoSize;
    DWORD dwMajorVersion;
    DWORD dwMinorVersion;
    DWORD dwBuildNumber;
    DWORD dwPlatformId;
    WCHAR szCSDVersion[128];
    WORD  wServicePackMajor;
    WORD  wServicePackMinor;
    WORD  wSuiteMask;
    BYTE  wProductType;
    BYTE  wReserved;
} ma_OSVERSIONINFOEXW;

typedef BOOL      (WINAPI * ma_PFNVerifyVersionInfoW) (ma_OSVERSIONINFOEXW* lpVersionInfo, DWORD dwTypeMask, DWORDLONG dwlConditionMask);
typedef ULONGLONG (WINAPI * ma_PFNVerSetConditionMask)(ULONGLONG dwlConditionMask, DWORD dwTypeBitMask, BYTE dwConditionMask);


#ifndef PROPERTYKEY_DEFINED
#define PROPERTYKEY_DEFINED
typedef struct
{
    GUID fmtid;
    DWORD pid;
} PROPERTYKEY;
#endif

/* Some compilers don't define PropVariantInit(). We just do this ourselves since it's just a memset(). */
static MA_INLINE void ma_PropVariantInit(PROPVARIANT* pProp)
{
    MA_ZERO_OBJECT(pProp);
}


static const PROPERTYKEY MA_PKEY_Device_FriendlyName             = {{0xA45C254E, 0xDF1C, 0x4EFD, {0x80, 0x20, 0x67, 0xD1, 0x46, 0xA8, 0x50, 0xE0}}, 14};
static const PROPERTYKEY MA_PKEY_AudioEngine_DeviceFormat        = {{0xF19F064D, 0x82C,  0x4E27, {0xBC, 0x73, 0x68, 0x82, 0xA1, 0xBB, 0x8E, 0x4C}},  0};

static const IID MA_IID_IUnknown                                 = {0x00000000, 0x0000, 0x0000, {0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46}}; /* 00000000-0000-0000-C000-000000000046 */
static const IID MA_IID_IAgileObject                             = {0x94EA2B94, 0xE9CC, 0x49E0, {0xC0, 0xFF, 0xEE, 0x64, 0xCA, 0x8F, 0x5B, 0x90}}; /* 94EA2B94-E9CC-49E0-C0FF-EE64CA8F5B90 */

static const IID MA_IID_IAudioClient                             = {0x1CB9AD4C, 0xDBFA, 0x4C32, {0xB1, 0x78, 0xC2, 0xF5, 0x68, 0xA7, 0x03, 0xB2}}; /* 1CB9AD4C-DBFA-4C32-B178-C2F568A703B2 = __uuidof(IAudioClient) */
static const IID MA_IID_IAudioClient2                            = {0x726778CD, 0xF60A, 0x4EDA, {0x82, 0xDE, 0xE4, 0x76, 0x10, 0xCD, 0x78, 0xAA}}; /* 726778CD-F60A-4EDA-82DE-E47610CD78AA = __uuidof(IAudioClient2) */
static const IID MA_IID_IAudioClient3                            = {0x7ED4EE07, 0x8E67, 0x4CD4, {0x8C, 0x1A, 0x2B, 0x7A, 0x59, 0x87, 0xAD, 0x42}}; /* 7ED4EE07-8E67-4CD4-8C1A-2B7A5987AD42 = __uuidof(IAudioClient3) */
static const IID MA_IID_IAudioRenderClient                       = {0xF294ACFC, 0x3146, 0x4483, {0xA7, 0xBF, 0xAD, 0xDC, 0xA7, 0xC2, 0x60, 0xE2}}; /* F294ACFC-3146-4483-A7BF-ADDCA7C260E2 = __uuidof(IAudioRenderClient) */
static const IID MA_IID_IAudioCaptureClient                      = {0xC8ADBD64, 0xE71E, 0x48A0, {0xA4, 0xDE, 0x18, 0x5C, 0x39, 0x5C, 0xD3, 0x17}}; /* C8ADBD64-E71E-48A0-A4DE-185C395CD317 = __uuidof(IAudioCaptureClient) */
static const IID MA_IID_IMMNotificationClient                    = {0x7991EEC9, 0x7E89, 0x4D85, {0x83, 0x90, 0x6C, 0x70, 0x3C, 0xEC, 0x60, 0xC0}}; /* 7991EEC9-7E89-4D85-8390-6C703CEC60C0 = __uuidof(IMMNotificationClient) */
#ifndef MA_WIN32_DESKTOP
static const IID MA_IID_DEVINTERFACE_AUDIO_RENDER                = {0xE6327CAD, 0xDCEC, 0x4949, {0xAE, 0x8A, 0x99, 0x1E, 0x97, 0x6A, 0x79, 0xD2}}; /* E6327CAD-DCEC-4949-AE8A-991E976A79D2 */
static const IID MA_IID_DEVINTERFACE_AUDIO_CAPTURE               = {0x2EEF81BE, 0x33FA, 0x4800, {0x96, 0x70, 0x1C, 0xD4, 0x74, 0x97, 0x2C, 0x3F}}; /* 2EEF81BE-33FA-4800-9670-1CD474972C3F */
static const IID MA_IID_IActivateAudioInterfaceCompletionHandler = {0x41D949AB, 0x9862, 0x444A, {0x80, 0xF6, 0xC2, 0x61, 0x33, 0x4D, 0xA5, 0xEB}}; /* 41D949AB-9862-444A-80F6-C261334DA5EB */
#endif

static const IID MA_CLSID_MMDeviceEnumerator_Instance            = {0xBCDE0395, 0xE52F, 0x467C, {0x8E, 0x3D, 0xC4, 0x57, 0x92, 0x91, 0x69, 0x2E}}; /* BCDE0395-E52F-467C-8E3D-C4579291692E = __uuidof(MMDeviceEnumerator) */
static const IID MA_IID_IMMDeviceEnumerator_Instance             = {0xA95664D2, 0x9614, 0x4F35, {0xA7, 0x46, 0xDE, 0x8D, 0xB6, 0x36, 0x17, 0xE6}}; /* A95664D2-9614-4F35-A746-DE8DB63617E6 = __uuidof(IMMDeviceEnumerator) */
#ifdef __cplusplus
#define MA_CLSID_MMDeviceEnumerator MA_CLSID_MMDeviceEnumerator_Instance
#define MA_IID_IMMDeviceEnumerator  MA_IID_IMMDeviceEnumerator_Instance
#else
#define MA_CLSID_MMDeviceEnumerator &MA_CLSID_MMDeviceEnumerator_Instance
#define MA_IID_IMMDeviceEnumerator  &MA_IID_IMMDeviceEnumerator_Instance
#endif

typedef struct ma_IUnknown                                 ma_IUnknown;
#ifdef MA_WIN32_DESKTOP
#define MA_MM_DEVICE_STATE_ACTIVE                          1
#define MA_MM_DEVICE_STATE_DISABLED                        2
#define MA_MM_DEVICE_STATE_NOTPRESENT                      4
#define MA_MM_DEVICE_STATE_UNPLUGGED                       8

typedef struct ma_IMMDeviceEnumerator                      ma_IMMDeviceEnumerator;
typedef struct ma_IMMDeviceCollection                      ma_IMMDeviceCollection;
typedef struct ma_IMMDevice                                ma_IMMDevice;
#else
typedef struct ma_IActivateAudioInterfaceCompletionHandler ma_IActivateAudioInterfaceCompletionHandler;
typedef struct ma_IActivateAudioInterfaceAsyncOperation    ma_IActivateAudioInterfaceAsyncOperation;
#endif
typedef struct ma_IPropertyStore                           ma_IPropertyStore;
typedef struct ma_IAudioClient                             ma_IAudioClient;
typedef struct ma_IAudioClient2                            ma_IAudioClient2;
typedef struct ma_IAudioClient3                            ma_IAudioClient3;
typedef struct ma_IAudioRenderClient                       ma_IAudioRenderClient;
typedef struct ma_IAudioCaptureClient                      ma_IAudioCaptureClient;

typedef ma_int64                                           MA_REFERENCE_TIME;

#define MA_AUDCLNT_STREAMFLAGS_CROSSPROCESS                0x00010000
#define MA_AUDCLNT_STREAMFLAGS_LOOPBACK                    0x00020000
#define MA_AUDCLNT_STREAMFLAGS_EVENTCALLBACK               0x00040000
#define MA_AUDCLNT_STREAMFLAGS_NOPERSIST                   0x00080000
#define MA_AUDCLNT_STREAMFLAGS_RATEADJUST                  0x00100000
#define MA_AUDCLNT_STREAMFLAGS_SRC_DEFAULT_QUALITY         0x08000000
#define MA_AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM              0x80000000
#define MA_AUDCLNT_SESSIONFLAGS_EXPIREWHENUNOWNED          0x10000000
#define MA_AUDCLNT_SESSIONFLAGS_DISPLAY_HIDE               0x20000000
#define MA_AUDCLNT_SESSIONFLAGS_DISPLAY_HIDEWHENEXPIRED    0x40000000

/* Buffer flags. */
#define MA_AUDCLNT_BUFFERFLAGS_DATA_DISCONTINUITY          1
#define MA_AUDCLNT_BUFFERFLAGS_SILENT                      2
#define MA_AUDCLNT_BUFFERFLAGS_TIMESTAMP_ERROR             4

typedef enum
{
    ma_eRender  = 0,
    ma_eCapture = 1,
    ma_eAll     = 2
} ma_EDataFlow;

typedef enum
{
    ma_eConsole        = 0,
    ma_eMultimedia     = 1,
    ma_eCommunications = 2
} ma_ERole;

typedef enum
{
    MA_AUDCLNT_SHAREMODE_SHARED,
    MA_AUDCLNT_SHAREMODE_EXCLUSIVE
} MA_AUDCLNT_SHAREMODE;

typedef enum
{
    MA_AudioCategory_Other = 0  /* <-- miniaudio is only caring about Other. */
} MA_AUDIO_STREAM_CATEGORY;

typedef struct
{
    UINT32 cbSize;
    BOOL bIsOffload;
    MA_AUDIO_STREAM_CATEGORY eCategory;
} ma_AudioClientProperties;

/* IUnknown */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IUnknown* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IUnknown* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IUnknown* pThis);
} ma_IUnknownVtbl;
struct ma_IUnknown
{
    ma_IUnknownVtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IUnknown_QueryInterface(ma_IUnknown* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IUnknown_AddRef(ma_IUnknown* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IUnknown_Release(ma_IUnknown* pThis)                                                { return pThis->lpVtbl->Release(pThis); }

#ifdef MA_WIN32_DESKTOP
    /* IMMNotificationClient */
    typedef struct
    {
        /* IUnknown */
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IMMNotificationClient* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IMMNotificationClient* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (ma_IMMNotificationClient* pThis);

        /* IMMNotificationClient */
        HRESULT (STDMETHODCALLTYPE * OnDeviceStateChanged)  (ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID, DWORD dwNewState);
        HRESULT (STDMETHODCALLTYPE * OnDeviceAdded)         (ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID);
        HRESULT (STDMETHODCALLTYPE * OnDeviceRemoved)       (ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID);
        HRESULT (STDMETHODCALLTYPE * OnDefaultDeviceChanged)(ma_IMMNotificationClient* pThis, ma_EDataFlow dataFlow, ma_ERole role, LPCWSTR pDefaultDeviceID);
        HRESULT (STDMETHODCALLTYPE * OnPropertyValueChanged)(ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID, const PROPERTYKEY key);
    } ma_IMMNotificationClientVtbl;

    /* IMMDeviceEnumerator */
    typedef struct
    {
        /* IUnknown */
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IMMDeviceEnumerator* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IMMDeviceEnumerator* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (ma_IMMDeviceEnumerator* pThis);

        /* IMMDeviceEnumerator */
        HRESULT (STDMETHODCALLTYPE * EnumAudioEndpoints)                    (ma_IMMDeviceEnumerator* pThis, ma_EDataFlow dataFlow, DWORD dwStateMask, ma_IMMDeviceCollection** ppDevices);
        HRESULT (STDMETHODCALLTYPE * GetDefaultAudioEndpoint)               (ma_IMMDeviceEnumerator* pThis, ma_EDataFlow dataFlow, ma_ERole role, ma_IMMDevice** ppEndpoint);
        HRESULT (STDMETHODCALLTYPE * GetDevice)                             (ma_IMMDeviceEnumerator* pThis, LPCWSTR pID, ma_IMMDevice** ppDevice);
        HRESULT (STDMETHODCALLTYPE * RegisterEndpointNotificationCallback)  (ma_IMMDeviceEnumerator* pThis, ma_IMMNotificationClient* pClient);
        HRESULT (STDMETHODCALLTYPE * UnregisterEndpointNotificationCallback)(ma_IMMDeviceEnumerator* pThis, ma_IMMNotificationClient* pClient);
    } ma_IMMDeviceEnumeratorVtbl;
    struct ma_IMMDeviceEnumerator
    {
        ma_IMMDeviceEnumeratorVtbl* lpVtbl;
    };
    static MA_INLINE HRESULT ma_IMMDeviceEnumerator_QueryInterface(ma_IMMDeviceEnumerator* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    static MA_INLINE ULONG   ma_IMMDeviceEnumerator_AddRef(ma_IMMDeviceEnumerator* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    static MA_INLINE ULONG   ma_IMMDeviceEnumerator_Release(ma_IMMDeviceEnumerator* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    static MA_INLINE HRESULT ma_IMMDeviceEnumerator_EnumAudioEndpoints(ma_IMMDeviceEnumerator* pThis, ma_EDataFlow dataFlow, DWORD dwStateMask, ma_IMMDeviceCollection** ppDevices) { return pThis->lpVtbl->EnumAudioEndpoints(pThis, dataFlow, dwStateMask, ppDevices); }
    static MA_INLINE HRESULT ma_IMMDeviceEnumerator_GetDefaultAudioEndpoint(ma_IMMDeviceEnumerator* pThis, ma_EDataFlow dataFlow, ma_ERole role, ma_IMMDevice** ppEndpoint) { return pThis->lpVtbl->GetDefaultAudioEndpoint(pThis, dataFlow, role, ppEndpoint); }
    static MA_INLINE HRESULT ma_IMMDeviceEnumerator_GetDevice(ma_IMMDeviceEnumerator* pThis, LPCWSTR pID, ma_IMMDevice** ppDevice)        { return pThis->lpVtbl->GetDevice(pThis, pID, ppDevice); }
    static MA_INLINE HRESULT ma_IMMDeviceEnumerator_RegisterEndpointNotificationCallback(ma_IMMDeviceEnumerator* pThis, ma_IMMNotificationClient* pClient) { return pThis->lpVtbl->RegisterEndpointNotificationCallback(pThis, pClient); }
    static MA_INLINE HRESULT ma_IMMDeviceEnumerator_UnregisterEndpointNotificationCallback(ma_IMMDeviceEnumerator* pThis, ma_IMMNotificationClient* pClient) { return pThis->lpVtbl->UnregisterEndpointNotificationCallback(pThis, pClient); }


    /* IMMDeviceCollection */
    typedef struct
    {
        /* IUnknown */
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IMMDeviceCollection* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IMMDeviceCollection* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (ma_IMMDeviceCollection* pThis);

        /* IMMDeviceCollection */
        HRESULT (STDMETHODCALLTYPE * GetCount)(ma_IMMDeviceCollection* pThis, UINT* pDevices);
        HRESULT (STDMETHODCALLTYPE * Item)    (ma_IMMDeviceCollection* pThis, UINT nDevice, ma_IMMDevice** ppDevice);
    } ma_IMMDeviceCollectionVtbl;
    struct ma_IMMDeviceCollection
    {
        ma_IMMDeviceCollectionVtbl* lpVtbl;
    };
    static MA_INLINE HRESULT ma_IMMDeviceCollection_QueryInterface(ma_IMMDeviceCollection* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    static MA_INLINE ULONG   ma_IMMDeviceCollection_AddRef(ma_IMMDeviceCollection* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    static MA_INLINE ULONG   ma_IMMDeviceCollection_Release(ma_IMMDeviceCollection* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    static MA_INLINE HRESULT ma_IMMDeviceCollection_GetCount(ma_IMMDeviceCollection* pThis, UINT* pDevices)                               { return pThis->lpVtbl->GetCount(pThis, pDevices); }
    static MA_INLINE HRESULT ma_IMMDeviceCollection_Item(ma_IMMDeviceCollection* pThis, UINT nDevice, ma_IMMDevice** ppDevice)            { return pThis->lpVtbl->Item(pThis, nDevice, ppDevice); }


    /* IMMDevice */
    typedef struct
    {
        /* IUnknown */
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IMMDevice* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IMMDevice* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (ma_IMMDevice* pThis);

        /* IMMDevice */
        HRESULT (STDMETHODCALLTYPE * Activate)         (ma_IMMDevice* pThis, const IID* const iid, DWORD dwClsCtx, PROPVARIANT* pActivationParams, void** ppInterface);
        HRESULT (STDMETHODCALLTYPE * OpenPropertyStore)(ma_IMMDevice* pThis, DWORD stgmAccess, ma_IPropertyStore** ppProperties);
        HRESULT (STDMETHODCALLTYPE * GetId)            (ma_IMMDevice* pThis, LPWSTR *pID);
        HRESULT (STDMETHODCALLTYPE * GetState)         (ma_IMMDevice* pThis, DWORD *pState);
    } ma_IMMDeviceVtbl;
    struct ma_IMMDevice
    {
        ma_IMMDeviceVtbl* lpVtbl;
    };
    static MA_INLINE HRESULT ma_IMMDevice_QueryInterface(ma_IMMDevice* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    static MA_INLINE ULONG   ma_IMMDevice_AddRef(ma_IMMDevice* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    static MA_INLINE ULONG   ma_IMMDevice_Release(ma_IMMDevice* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    static MA_INLINE HRESULT ma_IMMDevice_Activate(ma_IMMDevice* pThis, const IID* const iid, DWORD dwClsCtx, PROPVARIANT* pActivationParams, void** ppInterface) { return pThis->lpVtbl->Activate(pThis, iid, dwClsCtx, pActivationParams, ppInterface); }
    static MA_INLINE HRESULT ma_IMMDevice_OpenPropertyStore(ma_IMMDevice* pThis, DWORD stgmAccess, ma_IPropertyStore** ppProperties) { return pThis->lpVtbl->OpenPropertyStore(pThis, stgmAccess, ppProperties); }
    static MA_INLINE HRESULT ma_IMMDevice_GetId(ma_IMMDevice* pThis, LPWSTR *pID)                                     { return pThis->lpVtbl->GetId(pThis, pID); }
    static MA_INLINE HRESULT ma_IMMDevice_GetState(ma_IMMDevice* pThis, DWORD *pState)                                { return pThis->lpVtbl->GetState(pThis, pState); }
#else
    /* IActivateAudioInterfaceAsyncOperation */
    typedef struct
    {
        /* IUnknown */
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IActivateAudioInterfaceAsyncOperation* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IActivateAudioInterfaceAsyncOperation* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (ma_IActivateAudioInterfaceAsyncOperation* pThis);

        /* IActivateAudioInterfaceAsyncOperation */
        HRESULT (STDMETHODCALLTYPE * GetActivateResult)(ma_IActivateAudioInterfaceAsyncOperation* pThis, HRESULT *pActivateResult, ma_IUnknown** ppActivatedInterface);
    } ma_IActivateAudioInterfaceAsyncOperationVtbl;
    struct ma_IActivateAudioInterfaceAsyncOperation
    {
        ma_IActivateAudioInterfaceAsyncOperationVtbl* lpVtbl;
    };
    static MA_INLINE HRESULT ma_IActivateAudioInterfaceAsyncOperation_QueryInterface(ma_IActivateAudioInterfaceAsyncOperation* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    static MA_INLINE ULONG   ma_IActivateAudioInterfaceAsyncOperation_AddRef(ma_IActivateAudioInterfaceAsyncOperation* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    static MA_INLINE ULONG   ma_IActivateAudioInterfaceAsyncOperation_Release(ma_IActivateAudioInterfaceAsyncOperation* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    static MA_INLINE HRESULT ma_IActivateAudioInterfaceAsyncOperation_GetActivateResult(ma_IActivateAudioInterfaceAsyncOperation* pThis, HRESULT *pActivateResult, ma_IUnknown** ppActivatedInterface) { return pThis->lpVtbl->GetActivateResult(pThis, pActivateResult, ppActivatedInterface); }
#endif

/* IPropertyStore */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IPropertyStore* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IPropertyStore* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IPropertyStore* pThis);

    /* IPropertyStore */
    HRESULT (STDMETHODCALLTYPE * GetCount)(ma_IPropertyStore* pThis, DWORD* pPropCount);
    HRESULT (STDMETHODCALLTYPE * GetAt)   (ma_IPropertyStore* pThis, DWORD propIndex, PROPERTYKEY* pPropKey);
    HRESULT (STDMETHODCALLTYPE * GetValue)(ma_IPropertyStore* pThis, const PROPERTYKEY* const pKey, PROPVARIANT* pPropVar);
    HRESULT (STDMETHODCALLTYPE * SetValue)(ma_IPropertyStore* pThis, const PROPERTYKEY* const pKey, const PROPVARIANT* const pPropVar);
    HRESULT (STDMETHODCALLTYPE * Commit)  (ma_IPropertyStore* pThis);
} ma_IPropertyStoreVtbl;
struct ma_IPropertyStore
{
    ma_IPropertyStoreVtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IPropertyStore_QueryInterface(ma_IPropertyStore* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IPropertyStore_AddRef(ma_IPropertyStore* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IPropertyStore_Release(ma_IPropertyStore* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IPropertyStore_GetCount(ma_IPropertyStore* pThis, DWORD* pPropCount)                            { return pThis->lpVtbl->GetCount(pThis, pPropCount); }
static MA_INLINE HRESULT ma_IPropertyStore_GetAt(ma_IPropertyStore* pThis, DWORD propIndex, PROPERTYKEY* pPropKey)          { return pThis->lpVtbl->GetAt(pThis, propIndex, pPropKey); }
static MA_INLINE HRESULT ma_IPropertyStore_GetValue(ma_IPropertyStore* pThis, const PROPERTYKEY* const pKey, PROPVARIANT* pPropVar) { return pThis->lpVtbl->GetValue(pThis, pKey, pPropVar); }
static MA_INLINE HRESULT ma_IPropertyStore_SetValue(ma_IPropertyStore* pThis, const PROPERTYKEY* const pKey, const PROPVARIANT* const pPropVar) { return pThis->lpVtbl->SetValue(pThis, pKey, pPropVar); }
static MA_INLINE HRESULT ma_IPropertyStore_Commit(ma_IPropertyStore* pThis)                                                 { return pThis->lpVtbl->Commit(pThis); }


/* IAudioClient */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IAudioClient* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IAudioClient* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IAudioClient* pThis);

    /* IAudioClient */
    HRESULT (STDMETHODCALLTYPE * Initialize)       (ma_IAudioClient* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
    HRESULT (STDMETHODCALLTYPE * GetBufferSize)    (ma_IAudioClient* pThis, ma_uint32* pNumBufferFrames);
    HRESULT (STDMETHODCALLTYPE * GetStreamLatency) (ma_IAudioClient* pThis, MA_REFERENCE_TIME* pLatency);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPadding)(ma_IAudioClient* pThis, ma_uint32* pNumPaddingFrames);
    HRESULT (STDMETHODCALLTYPE * IsFormatSupported)(ma_IAudioClient* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch);
    HRESULT (STDMETHODCALLTYPE * GetMixFormat)     (ma_IAudioClient* pThis, WAVEFORMATEX** ppDeviceFormat);
    HRESULT (STDMETHODCALLTYPE * GetDevicePeriod)  (ma_IAudioClient* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod);
    HRESULT (STDMETHODCALLTYPE * Start)            (ma_IAudioClient* pThis);
    HRESULT (STDMETHODCALLTYPE * Stop)             (ma_IAudioClient* pThis);
    HRESULT (STDMETHODCALLTYPE * Reset)            (ma_IAudioClient* pThis);
    HRESULT (STDMETHODCALLTYPE * SetEventHandle)   (ma_IAudioClient* pThis, HANDLE eventHandle);
    HRESULT (STDMETHODCALLTYPE * GetService)       (ma_IAudioClient* pThis, const IID* const riid, void** pp);
} ma_IAudioClientVtbl;
struct ma_IAudioClient
{
    ma_IAudioClientVtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IAudioClient_QueryInterface(ma_IAudioClient* pThis, const IID* const riid, void** ppObject)    { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IAudioClient_AddRef(ma_IAudioClient* pThis)                                                    { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IAudioClient_Release(ma_IAudioClient* pThis)                                                   { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IAudioClient_Initialize(ma_IAudioClient* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid) { return pThis->lpVtbl->Initialize(pThis, shareMode, streamFlags, bufferDuration, periodicity, pFormat, pAudioSessionGuid); }
static MA_INLINE HRESULT ma_IAudioClient_GetBufferSize(ma_IAudioClient* pThis, ma_uint32* pNumBufferFrames)                { return pThis->lpVtbl->GetBufferSize(pThis, pNumBufferFrames); }
static MA_INLINE HRESULT ma_IAudioClient_GetStreamLatency(ma_IAudioClient* pThis, MA_REFERENCE_TIME* pLatency)             { return pThis->lpVtbl->GetStreamLatency(pThis, pLatency); }
static MA_INLINE HRESULT ma_IAudioClient_GetCurrentPadding(ma_IAudioClient* pThis, ma_uint32* pNumPaddingFrames)           { return pThis->lpVtbl->GetCurrentPadding(pThis, pNumPaddingFrames); }
static MA_INLINE HRESULT ma_IAudioClient_IsFormatSupported(ma_IAudioClient* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch) { return pThis->lpVtbl->IsFormatSupported(pThis, shareMode, pFormat, ppClosestMatch); }
static MA_INLINE HRESULT ma_IAudioClient_GetMixFormat(ma_IAudioClient* pThis, WAVEFORMATEX** ppDeviceFormat)               { return pThis->lpVtbl->GetMixFormat(pThis, ppDeviceFormat); }
static MA_INLINE HRESULT ma_IAudioClient_GetDevicePeriod(ma_IAudioClient* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod) { return pThis->lpVtbl->GetDevicePeriod(pThis, pDefaultDevicePeriod, pMinimumDevicePeriod); }
static MA_INLINE HRESULT ma_IAudioClient_Start(ma_IAudioClient* pThis)                                                     { return pThis->lpVtbl->Start(pThis); }
static MA_INLINE HRESULT ma_IAudioClient_Stop(ma_IAudioClient* pThis)                                                      { return pThis->lpVtbl->Stop(pThis); }
static MA_INLINE HRESULT ma_IAudioClient_Reset(ma_IAudioClient* pThis)                                                     { return pThis->lpVtbl->Reset(pThis); }
static MA_INLINE HRESULT ma_IAudioClient_SetEventHandle(ma_IAudioClient* pThis, HANDLE eventHandle)                        { return pThis->lpVtbl->SetEventHandle(pThis, eventHandle); }
static MA_INLINE HRESULT ma_IAudioClient_GetService(ma_IAudioClient* pThis, const IID* const riid, void** pp)              { return pThis->lpVtbl->GetService(pThis, riid, pp); }

/* IAudioClient2 */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IAudioClient2* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IAudioClient2* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IAudioClient2* pThis);

    /* IAudioClient */
    HRESULT (STDMETHODCALLTYPE * Initialize)       (ma_IAudioClient2* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
    HRESULT (STDMETHODCALLTYPE * GetBufferSize)    (ma_IAudioClient2* pThis, ma_uint32* pNumBufferFrames);
    HRESULT (STDMETHODCALLTYPE * GetStreamLatency) (ma_IAudioClient2* pThis, MA_REFERENCE_TIME* pLatency);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPadding)(ma_IAudioClient2* pThis, ma_uint32* pNumPaddingFrames);
    HRESULT (STDMETHODCALLTYPE * IsFormatSupported)(ma_IAudioClient2* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch);
    HRESULT (STDMETHODCALLTYPE * GetMixFormat)     (ma_IAudioClient2* pThis, WAVEFORMATEX** ppDeviceFormat);
    HRESULT (STDMETHODCALLTYPE * GetDevicePeriod)  (ma_IAudioClient2* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod);
    HRESULT (STDMETHODCALLTYPE * Start)            (ma_IAudioClient2* pThis);
    HRESULT (STDMETHODCALLTYPE * Stop)             (ma_IAudioClient2* pThis);
    HRESULT (STDMETHODCALLTYPE * Reset)            (ma_IAudioClient2* pThis);
    HRESULT (STDMETHODCALLTYPE * SetEventHandle)   (ma_IAudioClient2* pThis, HANDLE eventHandle);
    HRESULT (STDMETHODCALLTYPE * GetService)       (ma_IAudioClient2* pThis, const IID* const riid, void** pp);

    /* IAudioClient2 */
    HRESULT (STDMETHODCALLTYPE * IsOffloadCapable)   (ma_IAudioClient2* pThis, MA_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable);
    HRESULT (STDMETHODCALLTYPE * SetClientProperties)(ma_IAudioClient2* pThis, const ma_AudioClientProperties* pProperties);
    HRESULT (STDMETHODCALLTYPE * GetBufferSizeLimits)(ma_IAudioClient2* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MA_REFERENCE_TIME* pMinBufferDuration, MA_REFERENCE_TIME* pMaxBufferDuration);
} ma_IAudioClient2Vtbl;
struct ma_IAudioClient2
{
    ma_IAudioClient2Vtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IAudioClient2_QueryInterface(ma_IAudioClient2* pThis, const IID* const riid, void** ppObject)    { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IAudioClient2_AddRef(ma_IAudioClient2* pThis)                                                    { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IAudioClient2_Release(ma_IAudioClient2* pThis)                                                   { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IAudioClient2_Initialize(ma_IAudioClient2* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid) { return pThis->lpVtbl->Initialize(pThis, shareMode, streamFlags, bufferDuration, periodicity, pFormat, pAudioSessionGuid); }
static MA_INLINE HRESULT ma_IAudioClient2_GetBufferSize(ma_IAudioClient2* pThis, ma_uint32* pNumBufferFrames)                { return pThis->lpVtbl->GetBufferSize(pThis, pNumBufferFrames); }
static MA_INLINE HRESULT ma_IAudioClient2_GetStreamLatency(ma_IAudioClient2* pThis, MA_REFERENCE_TIME* pLatency)             { return pThis->lpVtbl->GetStreamLatency(pThis, pLatency); }
static MA_INLINE HRESULT ma_IAudioClient2_GetCurrentPadding(ma_IAudioClient2* pThis, ma_uint32* pNumPaddingFrames)           { return pThis->lpVtbl->GetCurrentPadding(pThis, pNumPaddingFrames); }
static MA_INLINE HRESULT ma_IAudioClient2_IsFormatSupported(ma_IAudioClient2* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch) { return pThis->lpVtbl->IsFormatSupported(pThis, shareMode, pFormat, ppClosestMatch); }
static MA_INLINE HRESULT ma_IAudioClient2_GetMixFormat(ma_IAudioClient2* pThis, WAVEFORMATEX** ppDeviceFormat)               { return pThis->lpVtbl->GetMixFormat(pThis, ppDeviceFormat); }
static MA_INLINE HRESULT ma_IAudioClient2_GetDevicePeriod(ma_IAudioClient2* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod) { return pThis->lpVtbl->GetDevicePeriod(pThis, pDefaultDevicePeriod, pMinimumDevicePeriod); }
static MA_INLINE HRESULT ma_IAudioClient2_Start(ma_IAudioClient2* pThis)                                                     { return pThis->lpVtbl->Start(pThis); }
static MA_INLINE HRESULT ma_IAudioClient2_Stop(ma_IAudioClient2* pThis)                                                      { return pThis->lpVtbl->Stop(pThis); }
static MA_INLINE HRESULT ma_IAudioClient2_Reset(ma_IAudioClient2* pThis)                                                     { return pThis->lpVtbl->Reset(pThis); }
static MA_INLINE HRESULT ma_IAudioClient2_SetEventHandle(ma_IAudioClient2* pThis, HANDLE eventHandle)                        { return pThis->lpVtbl->SetEventHandle(pThis, eventHandle); }
static MA_INLINE HRESULT ma_IAudioClient2_GetService(ma_IAudioClient2* pThis, const IID* const riid, void** pp)              { return pThis->lpVtbl->GetService(pThis, riid, pp); }
static MA_INLINE HRESULT ma_IAudioClient2_IsOffloadCapable(ma_IAudioClient2* pThis, MA_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable) { return pThis->lpVtbl->IsOffloadCapable(pThis, category, pOffloadCapable); }
static MA_INLINE HRESULT ma_IAudioClient2_SetClientProperties(ma_IAudioClient2* pThis, const ma_AudioClientProperties* pProperties)           { return pThis->lpVtbl->SetClientProperties(pThis, pProperties); }
static MA_INLINE HRESULT ma_IAudioClient2_GetBufferSizeLimits(ma_IAudioClient2* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MA_REFERENCE_TIME* pMinBufferDuration, MA_REFERENCE_TIME* pMaxBufferDuration) { return pThis->lpVtbl->GetBufferSizeLimits(pThis, pFormat, eventDriven, pMinBufferDuration, pMaxBufferDuration); }


/* IAudioClient3 */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IAudioClient3* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IAudioClient3* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IAudioClient3* pThis);

    /* IAudioClient */
    HRESULT (STDMETHODCALLTYPE * Initialize)       (ma_IAudioClient3* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
    HRESULT (STDMETHODCALLTYPE * GetBufferSize)    (ma_IAudioClient3* pThis, ma_uint32* pNumBufferFrames);
    HRESULT (STDMETHODCALLTYPE * GetStreamLatency) (ma_IAudioClient3* pThis, MA_REFERENCE_TIME* pLatency);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPadding)(ma_IAudioClient3* pThis, ma_uint32* pNumPaddingFrames);
    HRESULT (STDMETHODCALLTYPE * IsFormatSupported)(ma_IAudioClient3* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch);
    HRESULT (STDMETHODCALLTYPE * GetMixFormat)     (ma_IAudioClient3* pThis, WAVEFORMATEX** ppDeviceFormat);
    HRESULT (STDMETHODCALLTYPE * GetDevicePeriod)  (ma_IAudioClient3* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod);
    HRESULT (STDMETHODCALLTYPE * Start)            (ma_IAudioClient3* pThis);
    HRESULT (STDMETHODCALLTYPE * Stop)             (ma_IAudioClient3* pThis);
    HRESULT (STDMETHODCALLTYPE * Reset)            (ma_IAudioClient3* pThis);
    HRESULT (STDMETHODCALLTYPE * SetEventHandle)   (ma_IAudioClient3* pThis, HANDLE eventHandle);
    HRESULT (STDMETHODCALLTYPE * GetService)       (ma_IAudioClient3* pThis, const IID* const riid, void** pp);

    /* IAudioClient2 */
    HRESULT (STDMETHODCALLTYPE * IsOffloadCapable)   (ma_IAudioClient3* pThis, MA_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable);
    HRESULT (STDMETHODCALLTYPE * SetClientProperties)(ma_IAudioClient3* pThis, const ma_AudioClientProperties* pProperties);
    HRESULT (STDMETHODCALLTYPE * GetBufferSizeLimits)(ma_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MA_REFERENCE_TIME* pMinBufferDuration, MA_REFERENCE_TIME* pMaxBufferDuration);

    /* IAudioClient3 */
    HRESULT (STDMETHODCALLTYPE * GetSharedModeEnginePeriod)       (ma_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, UINT32* pDefaultPeriodInFrames, UINT32* pFundamentalPeriodInFrames, UINT32* pMinPeriodInFrames, UINT32* pMaxPeriodInFrames);
    HRESULT (STDMETHODCALLTYPE * GetCurrentSharedModeEnginePeriod)(ma_IAudioClient3* pThis, WAVEFORMATEX** ppFormat, UINT32* pCurrentPeriodInFrames);
    HRESULT (STDMETHODCALLTYPE * InitializeSharedAudioStream)     (ma_IAudioClient3* pThis, DWORD streamFlags, UINT32 periodInFrames, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
} ma_IAudioClient3Vtbl;
struct ma_IAudioClient3
{
    ma_IAudioClient3Vtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IAudioClient3_QueryInterface(ma_IAudioClient3* pThis, const IID* const riid, void** ppObject)    { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IAudioClient3_AddRef(ma_IAudioClient3* pThis)                                                    { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IAudioClient3_Release(ma_IAudioClient3* pThis)                                                   { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IAudioClient3_Initialize(ma_IAudioClient3* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid) { return pThis->lpVtbl->Initialize(pThis, shareMode, streamFlags, bufferDuration, periodicity, pFormat, pAudioSessionGuid); }
static MA_INLINE HRESULT ma_IAudioClient3_GetBufferSize(ma_IAudioClient3* pThis, ma_uint32* pNumBufferFrames)                { return pThis->lpVtbl->GetBufferSize(pThis, pNumBufferFrames); }
static MA_INLINE HRESULT ma_IAudioClient3_GetStreamLatency(ma_IAudioClient3* pThis, MA_REFERENCE_TIME* pLatency)             { return pThis->lpVtbl->GetStreamLatency(pThis, pLatency); }
static MA_INLINE HRESULT ma_IAudioClient3_GetCurrentPadding(ma_IAudioClient3* pThis, ma_uint32* pNumPaddingFrames)           { return pThis->lpVtbl->GetCurrentPadding(pThis, pNumPaddingFrames); }
static MA_INLINE HRESULT ma_IAudioClient3_IsFormatSupported(ma_IAudioClient3* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch) { return pThis->lpVtbl->IsFormatSupported(pThis, shareMode, pFormat, ppClosestMatch); }
static MA_INLINE HRESULT ma_IAudioClient3_GetMixFormat(ma_IAudioClient3* pThis, WAVEFORMATEX** ppDeviceFormat)               { return pThis->lpVtbl->GetMixFormat(pThis, ppDeviceFormat); }
static MA_INLINE HRESULT ma_IAudioClient3_GetDevicePeriod(ma_IAudioClient3* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod) { return pThis->lpVtbl->GetDevicePeriod(pThis, pDefaultDevicePeriod, pMinimumDevicePeriod); }
static MA_INLINE HRESULT ma_IAudioClient3_Start(ma_IAudioClient3* pThis)                                                     { return pThis->lpVtbl->Start(pThis); }
static MA_INLINE HRESULT ma_IAudioClient3_Stop(ma_IAudioClient3* pThis)                                                      { return pThis->lpVtbl->Stop(pThis); }
static MA_INLINE HRESULT ma_IAudioClient3_Reset(ma_IAudioClient3* pThis)                                                     { return pThis->lpVtbl->Reset(pThis); }
static MA_INLINE HRESULT ma_IAudioClient3_SetEventHandle(ma_IAudioClient3* pThis, HANDLE eventHandle)                        { return pThis->lpVtbl->SetEventHandle(pThis, eventHandle); }
static MA_INLINE HRESULT ma_IAudioClient3_GetService(ma_IAudioClient3* pThis, const IID* const riid, void** pp)              { return pThis->lpVtbl->GetService(pThis, riid, pp); }
static MA_INLINE HRESULT ma_IAudioClient3_IsOffloadCapable(ma_IAudioClient3* pThis, MA_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable) { return pThis->lpVtbl->IsOffloadCapable(pThis, category, pOffloadCapable); }
static MA_INLINE HRESULT ma_IAudioClient3_SetClientProperties(ma_IAudioClient3* pThis, const ma_AudioClientProperties* pProperties)           { return pThis->lpVtbl->SetClientProperties(pThis, pProperties); }
static MA_INLINE HRESULT ma_IAudioClient3_GetBufferSizeLimits(ma_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MA_REFERENCE_TIME* pMinBufferDuration, MA_REFERENCE_TIME* pMaxBufferDuration) { return pThis->lpVtbl->GetBufferSizeLimits(pThis, pFormat, eventDriven, pMinBufferDuration, pMaxBufferDuration); }
static MA_INLINE HRESULT ma_IAudioClient3_GetSharedModeEnginePeriod(ma_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, UINT32* pDefaultPeriodInFrames, UINT32* pFundamentalPeriodInFrames, UINT32* pMinPeriodInFrames, UINT32* pMaxPeriodInFrames) { return pThis->lpVtbl->GetSharedModeEnginePeriod(pThis, pFormat, pDefaultPeriodInFrames, pFundamentalPeriodInFrames, pMinPeriodInFrames, pMaxPeriodInFrames); }
static MA_INLINE HRESULT ma_IAudioClient3_GetCurrentSharedModeEnginePeriod(ma_IAudioClient3* pThis, WAVEFORMATEX** ppFormat, UINT32* pCurrentPeriodInFrames) { return pThis->lpVtbl->GetCurrentSharedModeEnginePeriod(pThis, ppFormat, pCurrentPeriodInFrames); }
static MA_INLINE HRESULT ma_IAudioClient3_InitializeSharedAudioStream(ma_IAudioClient3* pThis, DWORD streamFlags, UINT32 periodInFrames, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGUID) { return pThis->lpVtbl->InitializeSharedAudioStream(pThis, streamFlags, periodInFrames, pFormat, pAudioSessionGUID); }


/* IAudioRenderClient */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IAudioRenderClient* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IAudioRenderClient* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IAudioRenderClient* pThis);

    /* IAudioRenderClient */
    HRESULT (STDMETHODCALLTYPE * GetBuffer)    (ma_IAudioRenderClient* pThis, ma_uint32 numFramesRequested, BYTE** ppData);
    HRESULT (STDMETHODCALLTYPE * ReleaseBuffer)(ma_IAudioRenderClient* pThis, ma_uint32 numFramesWritten, DWORD dwFlags);
} ma_IAudioRenderClientVtbl;
struct ma_IAudioRenderClient
{
    ma_IAudioRenderClientVtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IAudioRenderClient_QueryInterface(ma_IAudioRenderClient* pThis, const IID* const riid, void** ppObject)   { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IAudioRenderClient_AddRef(ma_IAudioRenderClient* pThis)                                                   { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IAudioRenderClient_Release(ma_IAudioRenderClient* pThis)                                                  { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IAudioRenderClient_GetBuffer(ma_IAudioRenderClient* pThis, ma_uint32 numFramesRequested, BYTE** ppData)   { return pThis->lpVtbl->GetBuffer(pThis, numFramesRequested, ppData); }
static MA_INLINE HRESULT ma_IAudioRenderClient_ReleaseBuffer(ma_IAudioRenderClient* pThis, ma_uint32 numFramesWritten, DWORD dwFlags) { return pThis->lpVtbl->ReleaseBuffer(pThis, numFramesWritten, dwFlags); }


/* IAudioCaptureClient */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IAudioCaptureClient* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IAudioCaptureClient* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IAudioCaptureClient* pThis);

    /* IAudioRenderClient */
    HRESULT (STDMETHODCALLTYPE * GetBuffer)        (ma_IAudioCaptureClient* pThis, BYTE** ppData, ma_uint32* pNumFramesToRead, DWORD* pFlags, ma_uint64* pDevicePosition, ma_uint64* pQPCPosition);
    HRESULT (STDMETHODCALLTYPE * ReleaseBuffer)    (ma_IAudioCaptureClient* pThis, ma_uint32 numFramesRead);
    HRESULT (STDMETHODCALLTYPE * GetNextPacketSize)(ma_IAudioCaptureClient* pThis, ma_uint32* pNumFramesInNextPacket);
} ma_IAudioCaptureClientVtbl;
struct ma_IAudioCaptureClient
{
    ma_IAudioCaptureClientVtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IAudioCaptureClient_QueryInterface(ma_IAudioCaptureClient* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IAudioCaptureClient_AddRef(ma_IAudioCaptureClient* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IAudioCaptureClient_Release(ma_IAudioCaptureClient* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IAudioCaptureClient_GetBuffer(ma_IAudioCaptureClient* pThis, BYTE** ppData, ma_uint32* pNumFramesToRead, DWORD* pFlags, ma_uint64* pDevicePosition, ma_uint64* pQPCPosition) { return pThis->lpVtbl->GetBuffer(pThis, ppData, pNumFramesToRead, pFlags, pDevicePosition, pQPCPosition); }
static MA_INLINE HRESULT ma_IAudioCaptureClient_ReleaseBuffer(ma_IAudioCaptureClient* pThis, ma_uint32 numFramesRead)                 { return pThis->lpVtbl->ReleaseBuffer(pThis, numFramesRead); }
static MA_INLINE HRESULT ma_IAudioCaptureClient_GetNextPacketSize(ma_IAudioCaptureClient* pThis, ma_uint32* pNumFramesInNextPacket)   { return pThis->lpVtbl->GetNextPacketSize(pThis, pNumFramesInNextPacket); }

#ifndef MA_WIN32_DESKTOP
#include <mmdeviceapi.h>
typedef struct ma_completion_handler_uwp ma_completion_handler_uwp;

typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_completion_handler_uwp* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_completion_handler_uwp* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_completion_handler_uwp* pThis);

    /* IActivateAudioInterfaceCompletionHandler */
    HRESULT (STDMETHODCALLTYPE * ActivateCompleted)(ma_completion_handler_uwp* pThis, ma_IActivateAudioInterfaceAsyncOperation* pActivateOperation);
} ma_completion_handler_uwp_vtbl;
struct ma_completion_handler_uwp
{
    ma_completion_handler_uwp_vtbl* lpVtbl;
    ma_uint32 counter;
    HANDLE hEvent;
};

static HRESULT STDMETHODCALLTYPE ma_completion_handler_uwp_QueryInterface(ma_completion_handler_uwp* pThis, const IID* const riid, void** ppObject)
{
    /*
    We need to "implement" IAgileObject which is just an indicator that's used internally by WASAPI for some multithreading management. To
    "implement" this, we just make sure we return pThis when the IAgileObject is requested.
    */
    if (!ma_is_guid_equal(riid, &MA_IID_IUnknown) && !ma_is_guid_equal(riid, &MA_IID_IActivateAudioInterfaceCompletionHandler) && !ma_is_guid_equal(riid, &MA_IID_IAgileObject)) {
        *ppObject = NULL;
        return E_NOINTERFACE;
    }

    /* Getting here means the IID is IUnknown or IMMNotificationClient. */
    *ppObject = (void*)pThis;
    ((ma_completion_handler_uwp_vtbl*)pThis->lpVtbl)->AddRef(pThis);
    return S_OK;
}

static ULONG STDMETHODCALLTYPE ma_completion_handler_uwp_AddRef(ma_completion_handler_uwp* pThis)
{
    return (ULONG)c89atomic_fetch_add_32(&pThis->counter, 1) + 1;
}

static ULONG STDMETHODCALLTYPE ma_completion_handler_uwp_Release(ma_completion_handler_uwp* pThis)
{
    ma_uint32 newRefCount = c89atomic_fetch_sub_32(&pThis->counter, 1) - 1;
    if (newRefCount == 0) {
        return 0;   /* We don't free anything here because we never allocate the object on the heap. */
    }

    return (ULONG)newRefCount;
}

static HRESULT STDMETHODCALLTYPE ma_completion_handler_uwp_ActivateCompleted(ma_completion_handler_uwp* pThis, ma_IActivateAudioInterfaceAsyncOperation* pActivateOperation)
{
    (void)pActivateOperation;
    SetEvent(pThis->hEvent);
    return S_OK;
}


static ma_completion_handler_uwp_vtbl g_maCompletionHandlerVtblInstance = {
    ma_completion_handler_uwp_QueryInterface,
    ma_completion_handler_uwp_AddRef,
    ma_completion_handler_uwp_Release,
    ma_completion_handler_uwp_ActivateCompleted
};

static ma_result ma_completion_handler_uwp_init(ma_completion_handler_uwp* pHandler)
{
    MA_ASSERT(pHandler != NULL);
    MA_ZERO_OBJECT(pHandler);

    pHandler->lpVtbl = &g_maCompletionHandlerVtblInstance;
    pHandler->counter = 1;
    pHandler->hEvent = CreateEventW(NULL, FALSE, FALSE, NULL);
    if (pHandler->hEvent == NULL) {
        return ma_result_from_GetLastError(GetLastError());
    }

    return MA_SUCCESS;
}

static void ma_completion_handler_uwp_uninit(ma_completion_handler_uwp* pHandler)
{
    if (pHandler->hEvent != NULL) {
        CloseHandle(pHandler->hEvent);
    }
}

static void ma_completion_handler_uwp_wait(ma_completion_handler_uwp* pHandler)
{
    WaitForSingleObject(pHandler->hEvent, INFINITE);
}
#endif  /* !MA_WIN32_DESKTOP */

/* We need a virtual table for our notification client object that's used for detecting changes to the default device. */
#ifdef MA_WIN32_DESKTOP
static HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_QueryInterface(ma_IMMNotificationClient* pThis, const IID* const riid, void** ppObject)
{
    /*
    We care about two interfaces - IUnknown and IMMNotificationClient. If the requested IID is something else
    we just return E_NOINTERFACE. Otherwise we need to increment the reference counter and return S_OK.
    */
    if (!ma_is_guid_equal(riid, &MA_IID_IUnknown) && !ma_is_guid_equal(riid, &MA_IID_IMMNotificationClient)) {
        *ppObject = NULL;
        return E_NOINTERFACE;
    }

    /* Getting here means the IID is IUnknown or IMMNotificationClient. */
    *ppObject = (void*)pThis;
    ((ma_IMMNotificationClientVtbl*)pThis->lpVtbl)->AddRef(pThis);
    return S_OK;
}

static ULONG STDMETHODCALLTYPE ma_IMMNotificationClient_AddRef(ma_IMMNotificationClient* pThis)
{
    return (ULONG)c89atomic_fetch_add_32(&pThis->counter, 1) + 1;
}

static ULONG STDMETHODCALLTYPE ma_IMMNotificationClient_Release(ma_IMMNotificationClient* pThis)
{
    ma_uint32 newRefCount = c89atomic_fetch_sub_32(&pThis->counter, 1) - 1;
    if (newRefCount == 0) {
        return 0;   /* We don't free anything here because we never allocate the object on the heap. */
    }

    return (ULONG)newRefCount;
}

static HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_OnDeviceStateChanged(ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID, DWORD dwNewState)
{
    ma_bool32 isThisDevice = MA_FALSE;

#ifdef MA_DEBUG_OUTPUT
    /*printf("IMMNotificationClient_OnDeviceStateChanged(pDeviceID=%S, dwNewState=%u)\n", (pDeviceID != NULL) ? pDeviceID : L"(NULL)", (unsigned int)dwNewState);*/
#endif

    if ((dwNewState & MA_MM_DEVICE_STATE_ACTIVE) != 0) {
        return S_OK;
    }

    /*
    There have been reports of a hang when a playback device is disconnected. The idea with this code is to explicitly stop the device if we detect
    that the device is disabled or has been unplugged.
    */
    if (pThis->pDevice->wasapi.allowCaptureAutoStreamRouting && (pThis->pDevice->type == ma_device_type_capture || pThis->pDevice->type == ma_device_type_duplex || pThis->pDevice->type == ma_device_type_loopback)) {
        if (wcscmp(pThis->pDevice->capture.id.wasapi, pDeviceID) == 0) {
            isThisDevice = MA_TRUE;
        }
    }

    if (pThis->pDevice->wasapi.allowPlaybackAutoStreamRouting && (pThis->pDevice->type == ma_device_type_playback || pThis->pDevice->type == ma_device_type_duplex)) {
        if (wcscmp(pThis->pDevice->playback.id.wasapi, pDeviceID) == 0) {
            isThisDevice = MA_TRUE;
        }
    }

    if (isThisDevice) {
        ma_device_stop(pThis->pDevice);
    }
    
    return S_OK;
}

static HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_OnDeviceAdded(ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID)
{
#ifdef MA_DEBUG_OUTPUT
    /*printf("IMMNotificationClient_OnDeviceAdded(pDeviceID=%S)\n", (pDeviceID != NULL) ? pDeviceID : L"(NULL)");*/
#endif

    /* We don't need to worry about this event for our purposes. */
    (void)pThis;
    (void)pDeviceID;
    return S_OK;
}

static HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_OnDeviceRemoved(ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID)
{
#ifdef MA_DEBUG_OUTPUT
    /*printf("IMMNotificationClient_OnDeviceRemoved(pDeviceID=%S)\n", (pDeviceID != NULL) ? pDeviceID : L"(NULL)");*/
#endif

    /* We don't need to worry about this event for our purposes. */
    (void)pThis;
    (void)pDeviceID;
    return S_OK;
}

static HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_OnDefaultDeviceChanged(ma_IMMNotificationClient* pThis, ma_EDataFlow dataFlow, ma_ERole role, LPCWSTR pDefaultDeviceID)
{
#ifdef MA_DEBUG_OUTPUT
    /*printf("IMMNotificationClient_OnDefaultDeviceChanged(dataFlow=%d, role=%d, pDefaultDeviceID=%S)\n", dataFlow, role, (pDefaultDeviceID != NULL) ? pDefaultDeviceID : L"(NULL)");*/
#endif

    /* We only ever use the eConsole role in miniaudio. */
    if (role != ma_eConsole) {
        return S_OK;
    }

    /* We only care about devices with the same data flow and role as the current device. */
    if ((pThis->pDevice->type == ma_device_type_playback && dataFlow != ma_eRender) ||
        (pThis->pDevice->type == ma_device_type_capture  && dataFlow != ma_eCapture)) {
        return S_OK;
    }

    /* Don't do automatic stream routing if we're not allowed. */
    if ((dataFlow == ma_eRender  && pThis->pDevice->wasapi.allowPlaybackAutoStreamRouting == MA_FALSE) ||
        (dataFlow == ma_eCapture && pThis->pDevice->wasapi.allowCaptureAutoStreamRouting  == MA_FALSE)) {
        return S_OK;
    }

    /*
    Not currently supporting automatic stream routing in exclusive mode. This is not working correctly on my machine due to
    AUDCLNT_E_DEVICE_IN_USE errors when reinitializing the device. If this is a bug in miniaudio, we can try re-enabling this once
    it's fixed.
    */
    if ((dataFlow == ma_eRender  && pThis->pDevice->playback.shareMode == ma_share_mode_exclusive) ||
        (dataFlow == ma_eCapture && pThis->pDevice->capture.shareMode  == ma_share_mode_exclusive)) {
        return S_OK;
    }

    /*
    We don't change the device here - we change it in the worker thread to keep synchronization simple. To do this I'm just setting a flag to
    indicate that the default device has changed. Loopback devices are treated as capture devices so we need to do a bit of a dance to handle
    that properly.
    */
    if (dataFlow == ma_eRender  && pThis->pDevice->type != ma_device_type_loopback) {
        c89atomic_exchange_32(&pThis->pDevice->wasapi.hasDefaultPlaybackDeviceChanged, MA_TRUE);
    }
    if (dataFlow == ma_eCapture || pThis->pDevice->type == ma_device_type_loopback) {
        c89atomic_exchange_32(&pThis->pDevice->wasapi.hasDefaultCaptureDeviceChanged,  MA_TRUE);
    }

    (void)pDefaultDeviceID;
    return S_OK;
}

static HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_OnPropertyValueChanged(ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID, const PROPERTYKEY key)
{
#ifdef MA_DEBUG_OUTPUT
    /*printf("IMMNotificationClient_OnPropertyValueChanged(pDeviceID=%S)\n", (pDeviceID != NULL) ? pDeviceID : L"(NULL)");*/
#endif

    (void)pThis;
    (void)pDeviceID;
    (void)key;
    return S_OK;
}

static ma_IMMNotificationClientVtbl g_maNotificationCientVtbl = {
    ma_IMMNotificationClient_QueryInterface,
    ma_IMMNotificationClient_AddRef,
    ma_IMMNotificationClient_Release,
    ma_IMMNotificationClient_OnDeviceStateChanged,
    ma_IMMNotificationClient_OnDeviceAdded,
    ma_IMMNotificationClient_OnDeviceRemoved,
    ma_IMMNotificationClient_OnDefaultDeviceChanged,
    ma_IMMNotificationClient_OnPropertyValueChanged
};
#endif  /* MA_WIN32_DESKTOP */

#ifdef MA_WIN32_DESKTOP
typedef ma_IMMDevice ma_WASAPIDeviceInterface;
#else
typedef ma_IUnknown ma_WASAPIDeviceInterface;
#endif



static ma_bool32 ma_context_is_device_id_equal__wasapi(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pID0 != NULL);
    MA_ASSERT(pID1 != NULL);
    (void)pContext;

    return memcmp(pID0->wasapi, pID1->wasapi, sizeof(pID0->wasapi)) == 0;
}

static void ma_set_device_info_from_WAVEFORMATEX(const WAVEFORMATEX* pWF, ma_device_info* pInfo)
{
    MA_ASSERT(pWF != NULL);
    MA_ASSERT(pInfo != NULL);

    pInfo->formatCount   = 1;
    pInfo->formats[0]    = ma_format_from_WAVEFORMATEX(pWF);
    pInfo->minChannels   = pWF->nChannels;
    pInfo->maxChannels   = pWF->nChannels;
    pInfo->minSampleRate = pWF->nSamplesPerSec;
    pInfo->maxSampleRate = pWF->nSamplesPerSec;
}

static ma_result ma_context_get_device_info_from_IAudioClient__wasapi(ma_context* pContext, /*ma_IMMDevice**/void* pMMDevice, ma_IAudioClient* pAudioClient, ma_share_mode shareMode, ma_device_info* pInfo)
{
    MA_ASSERT(pAudioClient != NULL);
    MA_ASSERT(pInfo != NULL);

    /* We use a different technique to retrieve the device information depending on whether or not we are using shared or exclusive mode. */
    if (shareMode == ma_share_mode_shared) {
        /* Shared Mode. We use GetMixFormat() here. */
        WAVEFORMATEX* pWF = NULL;
        HRESULT hr = ma_IAudioClient_GetMixFormat((ma_IAudioClient*)pAudioClient, (WAVEFORMATEX**)&pWF);
        if (SUCCEEDED(hr)) {
            ma_set_device_info_from_WAVEFORMATEX(pWF, pInfo);
            return MA_SUCCESS;
        } else {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve mix format for device info retrieval.", ma_result_from_HRESULT(hr));
        }
    } else {
        /* Exlcusive Mode. We repeatedly call IsFormatSupported() here. This is not currently support on UWP. */
#ifdef MA_WIN32_DESKTOP
        /*
        The first thing to do is get the format from PKEY_AudioEngine_DeviceFormat. This should give us a channel count we assume is
        correct which will simplify our searching.
        */
        ma_IPropertyStore *pProperties;
        HRESULT hr = ma_IMMDevice_OpenPropertyStore((ma_IMMDevice*)pMMDevice, STGM_READ, &pProperties);
        if (SUCCEEDED(hr)) {
            PROPVARIANT var;
            ma_PropVariantInit(&var);

            hr = ma_IPropertyStore_GetValue(pProperties, &MA_PKEY_AudioEngine_DeviceFormat, &var);
            if (SUCCEEDED(hr)) {
                WAVEFORMATEX* pWF = (WAVEFORMATEX*)var.blob.pBlobData;
                ma_set_device_info_from_WAVEFORMATEX(pWF, pInfo);

                /*
                In my testing, the format returned by PKEY_AudioEngine_DeviceFormat is suitable for exclusive mode so we check this format
                first. If this fails, fall back to a search.
                */
                hr = ma_IAudioClient_IsFormatSupported((ma_IAudioClient*)pAudioClient, MA_AUDCLNT_SHAREMODE_EXCLUSIVE, pWF, NULL);
                ma_PropVariantClear(pContext, &var);

                if (FAILED(hr)) {
                    /*
                    The format returned by PKEY_AudioEngine_DeviceFormat is not supported, so fall back to a search. We assume the channel
                    count returned by MA_PKEY_AudioEngine_DeviceFormat is valid and correct. For simplicity we're only returning one format.
                    */
                    ma_uint32 channels = pInfo->minChannels;
                    ma_format formatsToSearch[] = {
                        ma_format_s16,
                        ma_format_s24,
                        /*ma_format_s24_32,*/
                        ma_format_f32,
                        ma_format_s32,
                        ma_format_u8
                    };
                    ma_channel defaultChannelMap[MA_MAX_CHANNELS];
                    WAVEFORMATEXTENSIBLE wf;
                    ma_bool32 found;
                    ma_uint32 iFormat;

                    /* Make sure we don't overflow the channel map. */
                    if (channels > MA_MAX_CHANNELS) {
                        channels = MA_MAX_CHANNELS;
                    }

                    ma_get_standard_channel_map(ma_standard_channel_map_microsoft, channels, defaultChannelMap);

                    MA_ZERO_OBJECT(&wf);
                    wf.Format.cbSize     = sizeof(wf);
                    wf.Format.wFormatTag = WAVE_FORMAT_EXTENSIBLE;
                    wf.Format.nChannels  = (WORD)channels;
                    wf.dwChannelMask     = ma_channel_map_to_channel_mask__win32(defaultChannelMap, channels);

                    found = MA_FALSE;
                    for (iFormat = 0; iFormat < ma_countof(formatsToSearch); ++iFormat) {
                        ma_format format = formatsToSearch[iFormat];
                        ma_uint32 iSampleRate;

                        wf.Format.wBitsPerSample       = (WORD)(ma_get_bytes_per_sample(format)*8);
                        wf.Format.nBlockAlign          = (WORD)(wf.Format.nChannels * wf.Format.wBitsPerSample / 8);
                        wf.Format.nAvgBytesPerSec      = wf.Format.nBlockAlign * wf.Format.nSamplesPerSec;
                        wf.Samples.wValidBitsPerSample = /*(format == ma_format_s24_32) ? 24 :*/ wf.Format.wBitsPerSample;
                        if (format == ma_format_f32) {
                            wf.SubFormat = MA_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT;
                        } else {
                            wf.SubFormat = MA_GUID_KSDATAFORMAT_SUBTYPE_PCM;
                        }

                        for (iSampleRate = 0; iSampleRate < ma_countof(g_maStandardSampleRatePriorities); ++iSampleRate) {
                            wf.Format.nSamplesPerSec = g_maStandardSampleRatePriorities[iSampleRate];

                            hr = ma_IAudioClient_IsFormatSupported((ma_IAudioClient*)pAudioClient, MA_AUDCLNT_SHAREMODE_EXCLUSIVE, (WAVEFORMATEX*)&wf, NULL);
                            if (SUCCEEDED(hr)) {
                                ma_set_device_info_from_WAVEFORMATEX((WAVEFORMATEX*)&wf, pInfo);
                                found = MA_TRUE;
                                break;
                            }
                        }

                        if (found) {
                            break;
                        }
                    }

                    if (!found) {
                        ma_IPropertyStore_Release(pProperties);
                        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to find suitable device format for device info retrieval.", MA_FORMAT_NOT_SUPPORTED);
                    }
                }
            } else {
                ma_IPropertyStore_Release(pProperties);
                return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve device format for device info retrieval.", ma_result_from_HRESULT(hr));
            }

            ma_IPropertyStore_Release(pProperties);
        } else {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to open property store for device info retrieval.", ma_result_from_HRESULT(hr));
        }

        return MA_SUCCESS;
#else
        /* Exclusive mode not fully supported in UWP right now. */
        return MA_ERROR;
#endif
    }
}

#ifdef MA_WIN32_DESKTOP
static ma_EDataFlow ma_device_type_to_EDataFlow(ma_device_type deviceType)
{
    if (deviceType == ma_device_type_playback) {
        return ma_eRender;
    } else if (deviceType == ma_device_type_capture) {
        return ma_eCapture;
    } else {
        MA_ASSERT(MA_FALSE);
        return ma_eRender; /* Should never hit this. */
    }
}

static ma_result ma_context_create_IMMDeviceEnumerator__wasapi(ma_context* pContext, ma_IMMDeviceEnumerator** ppDeviceEnumerator)
{
    HRESULT hr;
    ma_IMMDeviceEnumerator* pDeviceEnumerator;

    MA_ASSERT(pContext           != NULL);
    MA_ASSERT(ppDeviceEnumerator != NULL);

    *ppDeviceEnumerator = NULL; /* Safety. */

    hr = ma_CoCreateInstance(pContext, MA_CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, MA_IID_IMMDeviceEnumerator, (void**)&pDeviceEnumerator);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create device enumerator.", ma_result_from_HRESULT(hr));
    }

    *ppDeviceEnumerator = pDeviceEnumerator;

    return MA_SUCCESS;
}

static LPWSTR ma_context_get_default_device_id_from_IMMDeviceEnumerator__wasapi(ma_context* pContext, ma_IMMDeviceEnumerator* pDeviceEnumerator, ma_device_type deviceType)
{
    HRESULT hr;
    ma_IMMDevice* pMMDefaultDevice = NULL;
    LPWSTR pDefaultDeviceID = NULL;
    ma_EDataFlow dataFlow;
    ma_ERole role;

    MA_ASSERT(pContext          != NULL);
    MA_ASSERT(pDeviceEnumerator != NULL);

    /* Grab the EDataFlow type from the device type. */
    dataFlow = ma_device_type_to_EDataFlow(deviceType);

    /* The role is always eConsole, but we may make this configurable later. */
    role = ma_eConsole;

    hr = ma_IMMDeviceEnumerator_GetDefaultAudioEndpoint(pDeviceEnumerator, dataFlow, role, &pMMDefaultDevice);
    if (FAILED(hr)) {
        return NULL;
    }

    hr = ma_IMMDevice_GetId(pMMDefaultDevice, &pDefaultDeviceID);

    ma_IMMDevice_Release(pMMDefaultDevice);
    pMMDefaultDevice = NULL;

    if (FAILED(hr)) {
        return NULL;
    }

    return pDefaultDeviceID;
}

static LPWSTR ma_context_get_default_device_id__wasapi(ma_context* pContext, ma_device_type deviceType)    /* Free the returned pointer with ma_CoTaskMemFree() */
{
    ma_result result;
    ma_IMMDeviceEnumerator* pDeviceEnumerator;
    LPWSTR pDefaultDeviceID = NULL;

    MA_ASSERT(pContext != NULL);

    result = ma_context_create_IMMDeviceEnumerator__wasapi(pContext, &pDeviceEnumerator);
    if (result != MA_SUCCESS) {
        return NULL;
    }

    pDefaultDeviceID = ma_context_get_default_device_id_from_IMMDeviceEnumerator__wasapi(pContext, pDeviceEnumerator, deviceType);
    
    ma_IMMDeviceEnumerator_Release(pDeviceEnumerator);
    return pDefaultDeviceID;
}

static ma_result ma_context_get_MMDevice__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_IMMDevice** ppMMDevice)
{
    ma_IMMDeviceEnumerator* pDeviceEnumerator;
    HRESULT hr;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(ppMMDevice != NULL);

    hr = ma_CoCreateInstance(pContext, MA_CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, MA_IID_IMMDeviceEnumerator, (void**)&pDeviceEnumerator);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create IMMDeviceEnumerator.", ma_result_from_HRESULT(hr));
    }

    if (pDeviceID == NULL) {
        hr = ma_IMMDeviceEnumerator_GetDefaultAudioEndpoint(pDeviceEnumerator, (deviceType == ma_device_type_capture) ? ma_eCapture : ma_eRender, ma_eConsole, ppMMDevice);
    } else {
        hr = ma_IMMDeviceEnumerator_GetDevice(pDeviceEnumerator, pDeviceID->wasapi, ppMMDevice);
    }

    ma_IMMDeviceEnumerator_Release(pDeviceEnumerator);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve IMMDevice.", ma_result_from_HRESULT(hr));
    }

    return MA_SUCCESS;
}

static ma_result ma_context_get_device_info_from_MMDevice__wasapi(ma_context* pContext, ma_IMMDevice* pMMDevice, ma_share_mode shareMode, LPWSTR pDefaultDeviceID, ma_bool32 onlySimpleInfo, ma_device_info* pInfo)
{
    LPWSTR pDeviceID;
    HRESULT hr;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pMMDevice != NULL);
    MA_ASSERT(pInfo != NULL);

    /* ID. */
    hr = ma_IMMDevice_GetId(pMMDevice, &pDeviceID);
    if (SUCCEEDED(hr)) {
        size_t idlen = wcslen(pDeviceID);
        if (idlen+1 > ma_countof(pInfo->id.wasapi)) {
            ma_CoTaskMemFree(pContext, pDeviceID);
            MA_ASSERT(MA_FALSE);  /* NOTE: If this is triggered, please report it. It means the format of the ID must haved change and is too long to fit in our fixed sized buffer. */
            return MA_ERROR;
        }

        MA_COPY_MEMORY(pInfo->id.wasapi, pDeviceID, idlen * sizeof(wchar_t));
        pInfo->id.wasapi[idlen] = '\0';

        if (pDefaultDeviceID != NULL) {
            if (wcscmp(pDeviceID, pDefaultDeviceID) == 0) {
                /* It's a default device. */
                pInfo->_private.isDefault = MA_TRUE;
            }
        }

        ma_CoTaskMemFree(pContext, pDeviceID);
    }

    {
        ma_IPropertyStore *pProperties;
        hr = ma_IMMDevice_OpenPropertyStore(pMMDevice, STGM_READ, &pProperties);
        if (SUCCEEDED(hr)) {
            PROPVARIANT var;

            /* Description / Friendly Name */
            ma_PropVariantInit(&var);
            hr = ma_IPropertyStore_GetValue(pProperties, &MA_PKEY_Device_FriendlyName, &var);
            if (SUCCEEDED(hr)) {
                WideCharToMultiByte(CP_UTF8, 0, var.pwszVal, -1, pInfo->name, sizeof(pInfo->name), 0, FALSE);
                ma_PropVariantClear(pContext, &var);
            }

            ma_IPropertyStore_Release(pProperties);
        }
    }

    /* Format */
    if (!onlySimpleInfo) {
        ma_IAudioClient* pAudioClient;
        hr = ma_IMMDevice_Activate(pMMDevice, &MA_IID_IAudioClient, CLSCTX_ALL, NULL, (void**)&pAudioClient);
        if (SUCCEEDED(hr)) {
            ma_result result = ma_context_get_device_info_from_IAudioClient__wasapi(pContext, pMMDevice, pAudioClient, shareMode, pInfo);
            
            ma_IAudioClient_Release(pAudioClient);
            return result;
        } else {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to activate audio client for device info retrieval.", ma_result_from_HRESULT(hr));
        }
    }

    return MA_SUCCESS;
}

static ma_result ma_context_enumerate_devices_by_type__wasapi(ma_context* pContext, ma_IMMDeviceEnumerator* pDeviceEnumerator, ma_device_type deviceType, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_result result = MA_SUCCESS;
    UINT deviceCount;
    HRESULT hr;
    ma_uint32 iDevice;
    LPWSTR pDefaultDeviceID = NULL;
    ma_IMMDeviceCollection* pDeviceCollection = NULL;
    
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(callback != NULL);

    /* Grab the default device. We use this to know whether or not flag the returned device info as being the default. */
    pDefaultDeviceID = ma_context_get_default_device_id_from_IMMDeviceEnumerator__wasapi(pContext, pDeviceEnumerator, deviceType);

    /* We need to enumerate the devices which returns a device collection. */
    hr = ma_IMMDeviceEnumerator_EnumAudioEndpoints(pDeviceEnumerator, ma_device_type_to_EDataFlow(deviceType), MA_MM_DEVICE_STATE_ACTIVE, &pDeviceCollection);
    if (SUCCEEDED(hr)) {
        hr = ma_IMMDeviceCollection_GetCount(pDeviceCollection, &deviceCount);
        if (FAILED(hr)) {
            result = ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to get device count.", ma_result_from_HRESULT(hr));
            goto done;
        }

        for (iDevice = 0; iDevice < deviceCount; ++iDevice) {
            ma_device_info deviceInfo;
            ma_IMMDevice* pMMDevice;
        
            MA_ZERO_OBJECT(&deviceInfo);

            hr = ma_IMMDeviceCollection_Item(pDeviceCollection, iDevice, &pMMDevice);
            if (SUCCEEDED(hr)) {
                result = ma_context_get_device_info_from_MMDevice__wasapi(pContext, pMMDevice, ma_share_mode_shared, pDefaultDeviceID, MA_TRUE, &deviceInfo);   /* MA_TRUE = onlySimpleInfo. */

                ma_IMMDevice_Release(pMMDevice);
                if (result == MA_SUCCESS) {
                    ma_bool32 cbResult = callback(pContext, deviceType, &deviceInfo, pUserData);
                    if (cbResult == MA_FALSE) {
                        break;
                    }
                }
            }
        }
    }

done:
    if (pDefaultDeviceID != NULL) {
        ma_CoTaskMemFree(pContext, pDefaultDeviceID);
        pDefaultDeviceID = NULL;
    }

    if (pDeviceCollection != NULL) {
        ma_IMMDeviceCollection_Release(pDeviceCollection);
        pDeviceCollection = NULL;
    }

    return result;
}

static ma_result ma_context_get_IAudioClient_Desktop__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_IAudioClient** ppAudioClient, ma_IMMDevice** ppMMDevice)
{
    ma_result result;
    HRESULT hr;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(ppAudioClient != NULL);
    MA_ASSERT(ppMMDevice != NULL);

    result = ma_context_get_MMDevice__wasapi(pContext, deviceType, pDeviceID, ppMMDevice);
    if (result != MA_SUCCESS) {
        return result;
    }

    hr = ma_IMMDevice_Activate(*ppMMDevice, &MA_IID_IAudioClient, CLSCTX_ALL, NULL, (void**)ppAudioClient);
    if (FAILED(hr)) {
        return ma_result_from_HRESULT(hr);
    }

    return MA_SUCCESS;
}
#else
static ma_result ma_context_get_IAudioClient_UWP__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_IAudioClient** ppAudioClient, ma_IUnknown** ppActivatedInterface)
{
    ma_IActivateAudioInterfaceAsyncOperation *pAsyncOp = NULL;
    ma_completion_handler_uwp completionHandler;
    IID iid;
    LPOLESTR iidStr;
    HRESULT hr;
    ma_result result;
    HRESULT activateResult;
    ma_IUnknown* pActivatedInterface;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(ppAudioClient != NULL);

    if (pDeviceID != NULL) {
        MA_COPY_MEMORY(&iid, pDeviceID->wasapi, sizeof(iid));
    } else {
        if (deviceType == ma_device_type_playback) {
            iid = MA_IID_DEVINTERFACE_AUDIO_RENDER;
        } else {
            iid = MA_IID_DEVINTERFACE_AUDIO_CAPTURE;
        }
    }

#if defined(__cplusplus)
    hr = StringFromIID(iid, &iidStr);
#else
    hr = StringFromIID(&iid, &iidStr);
#endif
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to convert device IID to string for ActivateAudioInterfaceAsync(). Out of memory.", ma_result_from_HRESULT(hr));
    }

    result = ma_completion_handler_uwp_init(&completionHandler);
    if (result != MA_SUCCESS) {
        ma_CoTaskMemFree(pContext, iidStr);
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create event for waiting for ActivateAudioInterfaceAsync().", result);
    }

#if defined(__cplusplus)
    hr = ActivateAudioInterfaceAsync(iidStr, MA_IID_IAudioClient, NULL, (IActivateAudioInterfaceCompletionHandler*)&completionHandler, (IActivateAudioInterfaceAsyncOperation**)&pAsyncOp);
#else
    hr = ActivateAudioInterfaceAsync(iidStr, &MA_IID_IAudioClient, NULL, (IActivateAudioInterfaceCompletionHandler*)&completionHandler, (IActivateAudioInterfaceAsyncOperation**)&pAsyncOp);
#endif
    if (FAILED(hr)) {
        ma_completion_handler_uwp_uninit(&completionHandler);
        ma_CoTaskMemFree(pContext, iidStr);
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] ActivateAudioInterfaceAsync() failed.", ma_result_from_HRESULT(hr));
    }

    ma_CoTaskMemFree(pContext, iidStr);

    /* Wait for the async operation for finish. */
    ma_completion_handler_uwp_wait(&completionHandler);
    ma_completion_handler_uwp_uninit(&completionHandler);

    hr = ma_IActivateAudioInterfaceAsyncOperation_GetActivateResult(pAsyncOp, &activateResult, &pActivatedInterface);
    ma_IActivateAudioInterfaceAsyncOperation_Release(pAsyncOp);

    if (FAILED(hr) || FAILED(activateResult)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to activate device.", FAILED(hr) ? ma_result_from_HRESULT(hr) : ma_result_from_HRESULT(activateResult));
    }

    /* Here is where we grab the IAudioClient interface. */
    hr = ma_IUnknown_QueryInterface(pActivatedInterface, &MA_IID_IAudioClient, (void**)ppAudioClient);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to query IAudioClient interface.", ma_result_from_HRESULT(hr));
    }

    if (ppActivatedInterface) {
        *ppActivatedInterface = pActivatedInterface;
    } else {
        ma_IUnknown_Release(pActivatedInterface);
    }

    return MA_SUCCESS;
}
#endif

static ma_result ma_context_get_IAudioClient__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_IAudioClient** ppAudioClient, ma_WASAPIDeviceInterface** ppDeviceInterface)
{
#ifdef MA_WIN32_DESKTOP
    return ma_context_get_IAudioClient_Desktop__wasapi(pContext, deviceType, pDeviceID, ppAudioClient, ppDeviceInterface);
#else
    return ma_context_get_IAudioClient_UWP__wasapi(pContext, deviceType, pDeviceID, ppAudioClient, ppDeviceInterface);
#endif
}


static ma_result ma_context_enumerate_devices__wasapi(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    /* Different enumeration for desktop and UWP. */
#ifdef MA_WIN32_DESKTOP
    /* Desktop */
    HRESULT hr;
    ma_IMMDeviceEnumerator* pDeviceEnumerator;

    hr = ma_CoCreateInstance(pContext, MA_CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, MA_IID_IMMDeviceEnumerator, (void**)&pDeviceEnumerator);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create device enumerator.", ma_result_from_HRESULT(hr));
    }

    ma_context_enumerate_devices_by_type__wasapi(pContext, pDeviceEnumerator, ma_device_type_playback, callback, pUserData);
    ma_context_enumerate_devices_by_type__wasapi(pContext, pDeviceEnumerator, ma_device_type_capture,  callback, pUserData);

    ma_IMMDeviceEnumerator_Release(pDeviceEnumerator);
#else
    /*
    UWP
    
    The MMDevice API is only supported on desktop applications. For now, while I'm still figuring out how to properly enumerate
    over devices without using MMDevice, I'm restricting devices to defaults.
    
    Hint: DeviceInformation::FindAllAsync() with DeviceClass.AudioCapture/AudioRender. https://blogs.windows.com/buildingapps/2014/05/15/real-time-audio-in-windows-store-and-windows-phone-apps/
    */
    if (callback) {
        ma_bool32 cbResult = MA_TRUE;

        /* Playback. */
        if (cbResult) {
            ma_device_info deviceInfo;
            MA_ZERO_OBJECT(&deviceInfo);
            ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
            deviceInfo._private.isDefault = MA_TRUE;
            cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
        }

        /* Capture. */
        if (cbResult) {
            ma_device_info deviceInfo;
            MA_ZERO_OBJECT(&deviceInfo);
            ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
            deviceInfo._private.isDefault = MA_TRUE;
            cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
        }
    }
#endif

    return MA_SUCCESS;
}

static ma_result ma_context_get_device_info__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
#ifdef MA_WIN32_DESKTOP
    ma_result result;
    ma_IMMDevice* pMMDevice = NULL;
    LPWSTR pDefaultDeviceID = NULL;
    
    result = ma_context_get_MMDevice__wasapi(pContext, deviceType, pDeviceID, &pMMDevice);
    if (result != MA_SUCCESS) {
        return result;
    }

    /* We need the default device ID so we can set the isDefault flag in the device info. */
    pDefaultDeviceID = ma_context_get_default_device_id__wasapi(pContext, deviceType);

    result = ma_context_get_device_info_from_MMDevice__wasapi(pContext, pMMDevice, shareMode, pDefaultDeviceID, MA_FALSE, pDeviceInfo);   /* MA_FALSE = !onlySimpleInfo. */

    if (pDefaultDeviceID != NULL) {
        ma_CoTaskMemFree(pContext, pDefaultDeviceID);
        pDefaultDeviceID = NULL;
    }

    ma_IMMDevice_Release(pMMDevice);

    return result;
#else
    ma_IAudioClient* pAudioClient;
    ma_result result;

    /* UWP currently only uses default devices. */
    if (deviceType == ma_device_type_playback) {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    } else {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    }

    /* Not currently supporting exclusive mode on UWP. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_ERROR;
    }

    result = ma_context_get_IAudioClient_UWP__wasapi(pContext, deviceType, pDeviceID, &pAudioClient, NULL);
    if (result != MA_SUCCESS) {
        return result;
    }

    result = ma_context_get_device_info_from_IAudioClient__wasapi(pContext, NULL, pAudioClient, shareMode, pDeviceInfo);

    pDeviceInfo->_private.isDefault = MA_TRUE;  /* UWP only supports default devices. */

    ma_IAudioClient_Release(pAudioClient);
    return result;
#endif
}

static void ma_device_uninit__wasapi(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

#ifdef MA_WIN32_DESKTOP
    if (pDevice->wasapi.pDeviceEnumerator) {
        ((ma_IMMDeviceEnumerator*)pDevice->wasapi.pDeviceEnumerator)->lpVtbl->UnregisterEndpointNotificationCallback((ma_IMMDeviceEnumerator*)pDevice->wasapi.pDeviceEnumerator, &pDevice->wasapi.notificationClient);
        ma_IMMDeviceEnumerator_Release((ma_IMMDeviceEnumerator*)pDevice->wasapi.pDeviceEnumerator);
    }
#endif

    if (pDevice->wasapi.pRenderClient) {
        ma_IAudioRenderClient_Release((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient);
    }
    if (pDevice->wasapi.pCaptureClient) {
        ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
    }

    if (pDevice->wasapi.pAudioClientPlayback) {
        ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
    }
    if (pDevice->wasapi.pAudioClientCapture) {
        ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
    }

    if (pDevice->wasapi.hEventPlayback) {
        CloseHandle(pDevice->wasapi.hEventPlayback);
    }
    if (pDevice->wasapi.hEventCapture) {
        CloseHandle(pDevice->wasapi.hEventCapture);
    }
}


typedef struct
{
    /* Input. */
    ma_format formatIn;
    ma_uint32 channelsIn;
    ma_uint32 sampleRateIn;
    ma_channel channelMapIn[MA_MAX_CHANNELS];
    ma_uint32 periodSizeInFramesIn;
    ma_uint32 periodSizeInMillisecondsIn;
    ma_uint32 periodsIn;
    ma_bool32 usingDefaultFormat;
    ma_bool32 usingDefaultChannels;
    ma_bool32 usingDefaultSampleRate;
    ma_bool32 usingDefaultChannelMap;
    ma_share_mode shareMode;
    ma_bool32 noAutoConvertSRC;
    ma_bool32 noDefaultQualitySRC;
    ma_bool32 noHardwareOffloading;

    /* Output. */
    ma_IAudioClient* pAudioClient;
    ma_IAudioRenderClient* pRenderClient;
    ma_IAudioCaptureClient* pCaptureClient;
    ma_format formatOut;
    ma_uint32 channelsOut;
    ma_uint32 sampleRateOut;
    ma_channel channelMapOut[MA_MAX_CHANNELS];
    ma_uint32 periodSizeInFramesOut;
    ma_uint32 periodsOut;
    ma_bool32 usingAudioClient3;
    char deviceName[256];
} ma_device_init_internal_data__wasapi;

static ma_result ma_device_init_internal__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_device_init_internal_data__wasapi* pData)
{
    HRESULT hr;
    ma_result result = MA_SUCCESS;
    const char* errorMsg = "";
    MA_AUDCLNT_SHAREMODE shareMode = MA_AUDCLNT_SHAREMODE_SHARED;
    DWORD streamFlags = 0;
    MA_REFERENCE_TIME periodDurationInMicroseconds;
    ma_bool32 wasInitializedUsingIAudioClient3 = MA_FALSE;
    WAVEFORMATEXTENSIBLE wf;
    ma_WASAPIDeviceInterface* pDeviceInterface = NULL;
    ma_IAudioClient2* pAudioClient2;
    ma_uint32 nativeSampleRate;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pData != NULL);

    /* This function is only used to initialize one device type: either playback, capture or loopback. Never full-duplex. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    pData->pAudioClient = NULL;
    pData->pRenderClient = NULL;
    pData->pCaptureClient = NULL;

    streamFlags = MA_AUDCLNT_STREAMFLAGS_EVENTCALLBACK;
    if (!pData->noAutoConvertSRC && !pData->usingDefaultSampleRate && pData->shareMode != ma_share_mode_exclusive) {    /* <-- Exclusive streams must use the native sample rate. */
        streamFlags |= MA_AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM;
    }
    if (!pData->noDefaultQualitySRC && !pData->usingDefaultSampleRate && (streamFlags & MA_AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM) != 0) {
        streamFlags |= MA_AUDCLNT_STREAMFLAGS_SRC_DEFAULT_QUALITY;
    }
    if (deviceType == ma_device_type_loopback) {
        streamFlags |= MA_AUDCLNT_STREAMFLAGS_LOOPBACK;
    }

    result = ma_context_get_IAudioClient__wasapi(pContext, deviceType, pDeviceID, &pData->pAudioClient, &pDeviceInterface);
    if (result != MA_SUCCESS) {
        goto done;
    }

    MA_ZERO_OBJECT(&wf);

    /* Try enabling hardware offloading. */
    if (!pData->noHardwareOffloading) {
        hr = ma_IAudioClient_QueryInterface(pData->pAudioClient, &MA_IID_IAudioClient2, (void**)&pAudioClient2);
        if (SUCCEEDED(hr)) {
            BOOL isHardwareOffloadingSupported = 0;
            hr = ma_IAudioClient2_IsOffloadCapable(pAudioClient2, MA_AudioCategory_Other, &isHardwareOffloadingSupported);
            if (SUCCEEDED(hr) && isHardwareOffloadingSupported) {
                ma_AudioClientProperties clientProperties;
                MA_ZERO_OBJECT(&clientProperties);
                clientProperties.cbSize = sizeof(clientProperties);
                clientProperties.bIsOffload = 1;
                clientProperties.eCategory = MA_AudioCategory_Other;
                ma_IAudioClient2_SetClientProperties(pAudioClient2, &clientProperties);
            }

            pAudioClient2->lpVtbl->Release(pAudioClient2);
        }
    }

    /* Here is where we try to determine the best format to use with the device. If the client if wanting exclusive mode, first try finding the best format for that. If this fails, fall back to shared mode. */
    result = MA_FORMAT_NOT_SUPPORTED;
    if (pData->shareMode == ma_share_mode_exclusive) {
    #ifdef MA_WIN32_DESKTOP
        /* In exclusive mode on desktop we always use the backend's native format. */
        ma_IPropertyStore* pStore = NULL;
        hr = ma_IMMDevice_OpenPropertyStore(pDeviceInterface, STGM_READ, &pStore);
        if (SUCCEEDED(hr)) {
            PROPVARIANT prop;
            ma_PropVariantInit(&prop);
            hr = ma_IPropertyStore_GetValue(pStore, &MA_PKEY_AudioEngine_DeviceFormat, &prop);
            if (SUCCEEDED(hr)) {
                WAVEFORMATEX* pActualFormat = (WAVEFORMATEX*)prop.blob.pBlobData;
                hr = ma_IAudioClient_IsFormatSupported((ma_IAudioClient*)pData->pAudioClient, MA_AUDCLNT_SHAREMODE_EXCLUSIVE, pActualFormat, NULL);
                if (SUCCEEDED(hr)) {
                    MA_COPY_MEMORY(&wf, pActualFormat, sizeof(WAVEFORMATEXTENSIBLE));
                }

                ma_PropVariantClear(pContext, &prop);
            }

            ma_IPropertyStore_Release(pStore);
        }
    #else
        /*
        I do not know how to query the device's native format on UWP so for now I'm just disabling support for
        exclusive mode. The alternative is to enumerate over different formats and check IsFormatSupported()
        until you find one that works.
        
        TODO: Add support for exclusive mode to UWP.
        */
        hr = S_FALSE;
    #endif

        if (hr == S_OK) {
            shareMode = MA_AUDCLNT_SHAREMODE_EXCLUSIVE;
            result = MA_SUCCESS;
        } else {
            result = MA_SHARE_MODE_NOT_SUPPORTED;
        }
    } else {
        /* In shared mode we are always using the format reported by the operating system. */
        WAVEFORMATEXTENSIBLE* pNativeFormat = NULL;
        hr = ma_IAudioClient_GetMixFormat((ma_IAudioClient*)pData->pAudioClient, (WAVEFORMATEX**)&pNativeFormat);
        if (hr != S_OK) {
            result = MA_FORMAT_NOT_SUPPORTED;
        } else {
            MA_COPY_MEMORY(&wf, pNativeFormat, sizeof(wf));
            result = MA_SUCCESS;
        }

        ma_CoTaskMemFree(pContext, pNativeFormat);

        shareMode = MA_AUDCLNT_SHAREMODE_SHARED;
    }

    /* Return an error if we still haven't found a format. */
    if (result != MA_SUCCESS) {
        errorMsg = "[WASAPI] Failed to find best device mix format.";
        goto done;
    }

    /*
    Override the native sample rate with the one requested by the caller, but only if we're not using the default sample rate. We'll use
    WASAPI to perform the sample rate conversion.
    */
    nativeSampleRate = wf.Format.nSamplesPerSec;
    if (streamFlags & MA_AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM) {
        wf.Format.nSamplesPerSec = pData->sampleRateIn;
        wf.Format.nAvgBytesPerSec = wf.Format.nSamplesPerSec * wf.Format.nBlockAlign;
    }

    pData->formatOut = ma_format_from_WAVEFORMATEX((WAVEFORMATEX*)&wf);
    if (pData->formatOut == ma_format_unknown) {
        /*
        The format isn't supported. This is almost certainly because the exclusive mode format isn't supported by miniaudio. We need to return MA_SHARE_MODE_NOT_SUPPORTED
        in this case so that the caller can detect it and fall back to shared mode if desired. We should never get here if shared mode was requested, but just for
        completeness we'll check for it and return MA_FORMAT_NOT_SUPPORTED.
        */
        if (shareMode == ma_share_mode_exclusive) {
            result = MA_SHARE_MODE_NOT_SUPPORTED;
        } else {
            result = MA_FORMAT_NOT_SUPPORTED;
        }
        
        errorMsg = "[WASAPI] Native format not supported.";
        goto done;
    }

    pData->channelsOut = wf.Format.nChannels;
    pData->sampleRateOut = wf.Format.nSamplesPerSec;

    /* Get the internal channel map based on the channel mask. */
    ma_channel_mask_to_channel_map__win32(wf.dwChannelMask, pData->channelsOut, pData->channelMapOut);

    /* Period size. */
    pData->periodsOut = pData->periodsIn;
    pData->periodSizeInFramesOut = pData->periodSizeInFramesIn;
    if (pData->periodSizeInFramesOut == 0) {
        pData->periodSizeInFramesOut = ma_calculate_buffer_size_in_frames_from_milliseconds(pData->periodSizeInMillisecondsIn, wf.Format.nSamplesPerSec);
    }

    periodDurationInMicroseconds = ((ma_uint64)pData->periodSizeInFramesOut * 1000 * 1000) / wf.Format.nSamplesPerSec;


    /* Slightly different initialization for shared and exclusive modes. We try exclusive mode first, and if it fails, fall back to shared mode. */
    if (shareMode == MA_AUDCLNT_SHAREMODE_EXCLUSIVE) {
        MA_REFERENCE_TIME bufferDuration = periodDurationInMicroseconds * 10;

        /*
        If the periodicy is too small, Initialize() will fail with AUDCLNT_E_INVALID_DEVICE_PERIOD. In this case we should just keep increasing
        it and trying it again.
        */
        hr = E_FAIL;
        for (;;) {
            hr = ma_IAudioClient_Initialize((ma_IAudioClient*)pData->pAudioClient, shareMode, streamFlags, bufferDuration, bufferDuration, (WAVEFORMATEX*)&wf, NULL);
            if (hr == MA_AUDCLNT_E_INVALID_DEVICE_PERIOD) {
                if (bufferDuration > 500*10000) {
                    break;
                } else {
                    if (bufferDuration == 0) {  /* <-- Just a sanity check to prevent an infinit loop. Should never happen, but it makes me feel better. */
                        break;
                    }

                    bufferDuration = bufferDuration * 2;
                    continue;
                }
            } else {
                break;
            }
        }
        
        if (hr == MA_AUDCLNT_E_BUFFER_SIZE_NOT_ALIGNED) {
            ma_uint32 bufferSizeInFrames;
            hr = ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pData->pAudioClient, &bufferSizeInFrames);
            if (SUCCEEDED(hr)) {
                bufferDuration = (MA_REFERENCE_TIME)((10000.0 * 1000 / wf.Format.nSamplesPerSec * bufferSizeInFrames) + 0.5);

                /* Unfortunately we need to release and re-acquire the audio client according to MSDN. Seems silly - why not just call IAudioClient_Initialize() again?! */
                ma_IAudioClient_Release((ma_IAudioClient*)pData->pAudioClient);

            #ifdef MA_WIN32_DESKTOP
                hr = ma_IMMDevice_Activate(pDeviceInterface, &MA_IID_IAudioClient, CLSCTX_ALL, NULL, (void**)&pData->pAudioClient);
            #else
                hr = ma_IUnknown_QueryInterface(pDeviceInterface, &MA_IID_IAudioClient, (void**)&pData->pAudioClient);
            #endif

                if (SUCCEEDED(hr)) {
                    hr = ma_IAudioClient_Initialize((ma_IAudioClient*)pData->pAudioClient, shareMode, streamFlags, bufferDuration, bufferDuration, (WAVEFORMATEX*)&wf, NULL);
                }
            }
        }

        if (FAILED(hr)) {
            /* Failed to initialize in exclusive mode. Don't fall back to shared mode - instead tell the client about it. They can reinitialize in shared mode if they want. */
            if (hr == E_ACCESSDENIED) {
                errorMsg = "[WASAPI] Failed to initialize device in exclusive mode. Access denied.", result = MA_ACCESS_DENIED;
            } else if (hr == MA_AUDCLNT_E_DEVICE_IN_USE) {
                errorMsg = "[WASAPI] Failed to initialize device in exclusive mode. Device in use.", result = MA_BUSY;
            } else {
                errorMsg = "[WASAPI] Failed to initialize device in exclusive mode."; result = ma_result_from_HRESULT(hr);
            }
            goto done;
        }
    }

    if (shareMode == MA_AUDCLNT_SHAREMODE_SHARED) {
        /*
        Low latency shared mode via IAudioClient3.

        NOTE
        ====
        Contrary to the documentation on MSDN (https://docs.microsoft.com/en-us/windows/win32/api/audioclient/nf-audioclient-iaudioclient3-initializesharedaudiostream), the
        use of AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM and AUDCLNT_STREAMFLAGS_SRC_DEFAULT_QUALITY with IAudioClient3_InitializeSharedAudioStream() absolutely does not work. Using
        any of these flags will result in HRESULT code 0x88890021. The other problem is that calling IAudioClient3_GetSharedModeEnginePeriod() with a sample rate different to
        that returned by IAudioClient_GetMixFormat() also results in an error. I'm therefore disabling low-latency shared mode with AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM.
        */
#ifndef MA_WASAPI_NO_LOW_LATENCY_SHARED_MODE
        if ((streamFlags & MA_AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM) == 0 || nativeSampleRate == wf.Format.nSamplesPerSec) {
            ma_IAudioClient3* pAudioClient3 = NULL;
            hr = ma_IAudioClient_QueryInterface(pData->pAudioClient, &MA_IID_IAudioClient3, (void**)&pAudioClient3);
            if (SUCCEEDED(hr)) {
                UINT32 defaultPeriodInFrames;
                UINT32 fundamentalPeriodInFrames;
                UINT32 minPeriodInFrames;
                UINT32 maxPeriodInFrames;
                hr = ma_IAudioClient3_GetSharedModeEnginePeriod(pAudioClient3, (WAVEFORMATEX*)&wf, &defaultPeriodInFrames, &fundamentalPeriodInFrames, &minPeriodInFrames, &maxPeriodInFrames);
                if (SUCCEEDED(hr)) {
                    UINT32 desiredPeriodInFrames = pData->periodSizeInFramesOut;
                    UINT32 actualPeriodInFrames  = desiredPeriodInFrames;

                    /* Make sure the period size is a multiple of fundamentalPeriodInFrames. */
                    actualPeriodInFrames = actualPeriodInFrames / fundamentalPeriodInFrames;
                    actualPeriodInFrames = actualPeriodInFrames * fundamentalPeriodInFrames;

                    /* The period needs to be clamped between minPeriodInFrames and maxPeriodInFrames. */
                    actualPeriodInFrames = ma_clamp(actualPeriodInFrames, minPeriodInFrames, maxPeriodInFrames);

                #if defined(MA_DEBUG_OUTPUT)
                    printf("[WASAPI] Trying IAudioClient3_InitializeSharedAudioStream(actualPeriodInFrames=%d)\n", actualPeriodInFrames);
                    printf("    defaultPeriodInFrames=%d\n", defaultPeriodInFrames);
                    printf("    fundamentalPeriodInFrames=%d\n", fundamentalPeriodInFrames);
                    printf("    minPeriodInFrames=%d\n", minPeriodInFrames);
                    printf("    maxPeriodInFrames=%d\n", maxPeriodInFrames);
                #endif

                    /* If the client requested a largish buffer than we don't actually want to use low latency shared mode because it forces small buffers. */
                    if (actualPeriodInFrames >= desiredPeriodInFrames) {
                        /*
                        MA_AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM | MA_AUDCLNT_STREAMFLAGS_SRC_DEFAULT_QUALITY must not be in the stream flags. If either of these are specified,
                        IAudioClient3_InitializeSharedAudioStream() will fail.
                        */
                        hr = ma_IAudioClient3_InitializeSharedAudioStream(pAudioClient3, streamFlags & ~(MA_AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM | MA_AUDCLNT_STREAMFLAGS_SRC_DEFAULT_QUALITY), actualPeriodInFrames, (WAVEFORMATEX*)&wf, NULL);
                        if (SUCCEEDED(hr)) {
                            wasInitializedUsingIAudioClient3 = MA_TRUE;
                            pData->periodSizeInFramesOut = actualPeriodInFrames;
                        #if defined(MA_DEBUG_OUTPUT)
                            printf("[WASAPI] Using IAudioClient3\n");
                            printf("    periodSizeInFramesOut=%d\n", pData->periodSizeInFramesOut);
                        #endif
                        } else {
                        #if defined(MA_DEBUG_OUTPUT)
                            printf("[WASAPI] IAudioClient3_InitializeSharedAudioStream failed. Falling back to IAudioClient.\n");
                        #endif    
                        }
                    } else {
                    #if defined(MA_DEBUG_OUTPUT)
                        printf("[WASAPI] Not using IAudioClient3 because the desired period size is larger than the maximum supported by IAudioClient3.\n");
                    #endif
                    }
                } else {
                #if defined(MA_DEBUG_OUTPUT)
                    printf("[WASAPI] IAudioClient3_GetSharedModeEnginePeriod failed. Falling back to IAudioClient.\n");
                #endif
                }

                ma_IAudioClient3_Release(pAudioClient3);
                pAudioClient3 = NULL;
            }
        }
#else
    #if defined(MA_DEBUG_OUTPUT)
        printf("[WASAPI] Not using IAudioClient3 because MA_WASAPI_NO_LOW_LATENCY_SHARED_MODE is enabled.\n");
    #endif
#endif

        /* If we don't have an IAudioClient3 then we need to use the normal initialization routine. */
        if (!wasInitializedUsingIAudioClient3) {
            MA_REFERENCE_TIME bufferDuration = periodDurationInMicroseconds * pData->periodsOut * 10;   /* <-- Multiply by 10 for microseconds to 100-nanoseconds. */
            hr = ma_IAudioClient_Initialize((ma_IAudioClient*)pData->pAudioClient, shareMode, streamFlags, bufferDuration, 0, (WAVEFORMATEX*)&wf, NULL);
            if (FAILED(hr)) {
                if (hr == E_ACCESSDENIED) {
                    errorMsg = "[WASAPI] Failed to initialize device. Access denied.", result = MA_ACCESS_DENIED;
                } else if (hr == MA_AUDCLNT_E_DEVICE_IN_USE) {
                    errorMsg = "[WASAPI] Failed to initialize device. Device in use.", result = MA_BUSY;
                } else {
                    errorMsg = "[WASAPI] Failed to initialize device.", result = ma_result_from_HRESULT(hr);
                }

                goto done;
            }
        }
    }

    if (!wasInitializedUsingIAudioClient3) {
        ma_uint32 bufferSizeInFrames;
        hr = ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pData->pAudioClient, &bufferSizeInFrames);
        if (FAILED(hr)) {
            errorMsg = "[WASAPI] Failed to get audio client's actual buffer size.", result = ma_result_from_HRESULT(hr);
            goto done;
        }

        pData->periodSizeInFramesOut = bufferSizeInFrames / pData->periodsOut;
    }

    pData->usingAudioClient3 = wasInitializedUsingIAudioClient3;

    if (deviceType == ma_device_type_playback) {
        hr = ma_IAudioClient_GetService((ma_IAudioClient*)pData->pAudioClient, &MA_IID_IAudioRenderClient, (void**)&pData->pRenderClient);
    } else {
        hr = ma_IAudioClient_GetService((ma_IAudioClient*)pData->pAudioClient, &MA_IID_IAudioCaptureClient, (void**)&pData->pCaptureClient);
    }

    if (FAILED(hr)) {
        errorMsg = "[WASAPI] Failed to get audio client service.", result = ma_result_from_HRESULT(hr);
        goto done;
    }


    /* Grab the name of the device. */
#ifdef MA_WIN32_DESKTOP
    {
        ma_IPropertyStore *pProperties;
        hr = ma_IMMDevice_OpenPropertyStore(pDeviceInterface, STGM_READ, &pProperties);
        if (SUCCEEDED(hr)) {
            PROPVARIANT varName;
            ma_PropVariantInit(&varName);
            hr = ma_IPropertyStore_GetValue(pProperties, &MA_PKEY_Device_FriendlyName, &varName);
            if (SUCCEEDED(hr)) {
                WideCharToMultiByte(CP_UTF8, 0, varName.pwszVal, -1, pData->deviceName, sizeof(pData->deviceName), 0, FALSE);
                ma_PropVariantClear(pContext, &varName);
            }

            ma_IPropertyStore_Release(pProperties);
        }
    }
#endif

done:
    /* Clean up. */
#ifdef MA_WIN32_DESKTOP
    if (pDeviceInterface != NULL) {
        ma_IMMDevice_Release(pDeviceInterface);
    }
#else
    if (pDeviceInterface != NULL) {
        ma_IUnknown_Release(pDeviceInterface);
    }
#endif

    if (result != MA_SUCCESS) {
        if (pData->pRenderClient) {
            ma_IAudioRenderClient_Release((ma_IAudioRenderClient*)pData->pRenderClient);
            pData->pRenderClient = NULL;
        }
        if (pData->pCaptureClient) {
            ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pData->pCaptureClient);
            pData->pCaptureClient = NULL;
        }
        if (pData->pAudioClient) {
            ma_IAudioClient_Release((ma_IAudioClient*)pData->pAudioClient);
            pData->pAudioClient = NULL;
        }

        if (errorMsg != NULL && errorMsg[0] != '\0') {
            ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, errorMsg, result);
        }

        return result;
    } else {
        return MA_SUCCESS;
    }
}

static ma_result ma_device_reinit__wasapi(ma_device* pDevice, ma_device_type deviceType)
{
    ma_device_init_internal_data__wasapi data;
    ma_result result;

    MA_ASSERT(pDevice != NULL);

    /* We only re-initialize the playback or capture device. Never a full-duplex device. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    if (deviceType == ma_device_type_playback) {
        data.formatIn               = pDevice->playback.format;
        data.channelsIn             = pDevice->playback.channels;
        MA_COPY_MEMORY(data.channelMapIn, pDevice->playback.channelMap, sizeof(pDevice->playback.channelMap));
        data.shareMode              = pDevice->playback.shareMode;
        data.usingDefaultFormat     = pDevice->playback.usingDefaultFormat;
        data.usingDefaultChannels   = pDevice->playback.usingDefaultChannels;
        data.usingDefaultChannelMap = pDevice->playback.usingDefaultChannelMap;
    } else {
        data.formatIn               = pDevice->capture.format;
        data.channelsIn             = pDevice->capture.channels;
        MA_COPY_MEMORY(data.channelMapIn, pDevice->capture.channelMap, sizeof(pDevice->capture.channelMap));
        data.shareMode              = pDevice->capture.shareMode;
        data.usingDefaultFormat     = pDevice->capture.usingDefaultFormat;
        data.usingDefaultChannels   = pDevice->capture.usingDefaultChannels;
        data.usingDefaultChannelMap = pDevice->capture.usingDefaultChannelMap;
    }
    
    data.sampleRateIn               = pDevice->sampleRate;
    data.usingDefaultSampleRate     = pDevice->usingDefaultSampleRate;
    data.periodSizeInFramesIn       = pDevice->wasapi.originalPeriodSizeInFrames;
    data.periodSizeInMillisecondsIn = pDevice->wasapi.originalPeriodSizeInMilliseconds;
    data.periodsIn                  = pDevice->wasapi.originalPeriods;
    data.noAutoConvertSRC           = pDevice->wasapi.noAutoConvertSRC;
    data.noDefaultQualitySRC        = pDevice->wasapi.noDefaultQualitySRC;
    data.noHardwareOffloading       = pDevice->wasapi.noHardwareOffloading;
    result = ma_device_init_internal__wasapi(pDevice->pContext, deviceType, NULL, &data);
    if (result != MA_SUCCESS) {
        return result;
    }

    /* At this point we have some new objects ready to go. We need to uninitialize the previous ones and then set the new ones. */
    if (deviceType == ma_device_type_capture || deviceType == ma_device_type_loopback) {
        if (pDevice->wasapi.pCaptureClient) {
            ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
            pDevice->wasapi.pCaptureClient = NULL;
        }

        if (pDevice->wasapi.pAudioClientCapture) {
            ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
            pDevice->wasapi.pAudioClientCapture = NULL;
        }

        pDevice->wasapi.pAudioClientCapture         = data.pAudioClient;
        pDevice->wasapi.pCaptureClient              = data.pCaptureClient;

        pDevice->capture.internalFormat             = data.formatOut;
        pDevice->capture.internalChannels           = data.channelsOut;
        pDevice->capture.internalSampleRate         = data.sampleRateOut;
        MA_COPY_MEMORY(pDevice->capture.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->capture.internalPeriodSizeInFrames = data.periodSizeInFramesOut;
        pDevice->capture.internalPeriods            = data.periodsOut;
        ma_strcpy_s(pDevice->capture.name, sizeof(pDevice->capture.name), data.deviceName);

        ma_IAudioClient_SetEventHandle((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture,  pDevice->wasapi.hEventCapture);

        pDevice->wasapi.periodSizeInFramesCapture = data.periodSizeInFramesOut;
        ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture, &pDevice->wasapi.actualPeriodSizeInFramesCapture);

        /* The device may be in a started state. If so we need to immediately restart it. */
        if (pDevice->wasapi.isStartedCapture) {
            HRESULT hr = ma_IAudioClient_Start((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
            if (FAILED(hr)) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal capture device after reinitialization.", ma_result_from_HRESULT(hr));
            }
        }
    }

    if (deviceType == ma_device_type_playback) {
        if (pDevice->wasapi.pRenderClient) {
            ma_IAudioRenderClient_Release((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient);
            pDevice->wasapi.pRenderClient = NULL;
        }

        if (pDevice->wasapi.pAudioClientPlayback) {
            ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
            pDevice->wasapi.pAudioClientPlayback = NULL;
        }

        pDevice->wasapi.pAudioClientPlayback         = data.pAudioClient;
        pDevice->wasapi.pRenderClient                = data.pRenderClient;

        pDevice->playback.internalFormat             = data.formatOut;
        pDevice->playback.internalChannels           = data.channelsOut;
        pDevice->playback.internalSampleRate         = data.sampleRateOut;
        MA_COPY_MEMORY(pDevice->playback.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->playback.internalPeriodSizeInFrames = data.periodSizeInFramesOut;
        pDevice->playback.internalPeriods            = data.periodsOut;
        ma_strcpy_s(pDevice->playback.name, sizeof(pDevice->playback.name), data.deviceName);

        ma_IAudioClient_SetEventHandle((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, pDevice->wasapi.hEventPlayback);

        pDevice->wasapi.periodSizeInFramesPlayback = data.periodSizeInFramesOut;
        ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, &pDevice->wasapi.actualPeriodSizeInFramesPlayback);

        /* The device may be in a started state. If so we need to immediately restart it. */
        if (pDevice->wasapi.isStartedPlayback) {
            HRESULT hr = ma_IAudioClient_Start((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
            if (FAILED(hr)) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal playback device after reinitialization.", ma_result_from_HRESULT(hr));
            }
        }
    }

    return MA_SUCCESS;
}

static ma_result ma_device_init__wasapi(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;

#ifdef MA_WIN32_DESKTOP
    HRESULT hr;
    ma_IMMDeviceEnumerator* pDeviceEnumerator;
#endif

    (void)pContext;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pDevice != NULL);

    MA_ZERO_OBJECT(&pDevice->wasapi);
    pDevice->wasapi.originalPeriodSizeInFrames       = pConfig->periodSizeInFrames;
    pDevice->wasapi.originalPeriodSizeInMilliseconds = pConfig->periodSizeInMilliseconds;
    pDevice->wasapi.originalPeriods                  = pConfig->periods;
    pDevice->wasapi.noAutoConvertSRC                 = pConfig->wasapi.noAutoConvertSRC;
    pDevice->wasapi.noDefaultQualitySRC              = pConfig->wasapi.noDefaultQualitySRC;
    pDevice->wasapi.noHardwareOffloading             = pConfig->wasapi.noHardwareOffloading;

    /* Exclusive mode is not allowed with loopback. */
    if (pConfig->deviceType == ma_device_type_loopback && pConfig->playback.shareMode == ma_share_mode_exclusive) {
        return MA_INVALID_DEVICE_CONFIG;
    }

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex || pConfig->deviceType == ma_device_type_loopback) {
        ma_device_init_internal_data__wasapi data;
        data.formatIn                   = pConfig->capture.format;
        data.channelsIn                 = pConfig->capture.channels;
        data.sampleRateIn               = pConfig->sampleRate;
        MA_COPY_MEMORY(data.channelMapIn, pConfig->capture.channelMap, sizeof(pConfig->capture.channelMap));
        data.usingDefaultFormat         = pDevice->capture.usingDefaultFormat;
        data.usingDefaultChannels       = pDevice->capture.usingDefaultChannels;
        data.usingDefaultSampleRate     = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap     = pDevice->capture.usingDefaultChannelMap;
        data.shareMode                  = pConfig->capture.shareMode;
        data.periodSizeInFramesIn       = pConfig->periodSizeInFrames;
        data.periodSizeInMillisecondsIn = pConfig->periodSizeInMilliseconds;
        data.periodsIn                  = pConfig->periods;
        data.noAutoConvertSRC           = pConfig->wasapi.noAutoConvertSRC;
        data.noDefaultQualitySRC        = pConfig->wasapi.noDefaultQualitySRC;
        data.noHardwareOffloading       = pConfig->wasapi.noHardwareOffloading;

        result = ma_device_init_internal__wasapi(pDevice->pContext, (pConfig->deviceType == ma_device_type_loopback) ? ma_device_type_loopback : ma_device_type_capture, pConfig->capture.pDeviceID, &data);
        if (result != MA_SUCCESS) {
            return result;
        }

        pDevice->wasapi.pAudioClientCapture         = data.pAudioClient;
        pDevice->wasapi.pCaptureClient              = data.pCaptureClient;

        pDevice->capture.internalFormat             = data.formatOut;
        pDevice->capture.internalChannels           = data.channelsOut;
        pDevice->capture.internalSampleRate         = data.sampleRateOut;
        MA_COPY_MEMORY(pDevice->capture.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->capture.internalPeriodSizeInFrames = data.periodSizeInFramesOut;
        pDevice->capture.internalPeriods            = data.periodsOut;
        ma_strcpy_s(pDevice->capture.name, sizeof(pDevice->capture.name), data.deviceName);

        /*
        The event for capture needs to be manual reset for the same reason as playback. We keep the initial state set to unsignaled,
        however, because we want to block until we actually have something for the first call to ma_device_read().
        */
        pDevice->wasapi.hEventCapture = CreateEventW(NULL, FALSE, FALSE, NULL);  /* Auto reset, unsignaled by default. */
        if (pDevice->wasapi.hEventCapture == NULL) {
            result = ma_result_from_GetLastError(GetLastError());

            if (pDevice->wasapi.pCaptureClient != NULL) {
                ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
                pDevice->wasapi.pCaptureClient = NULL;
            }
            if (pDevice->wasapi.pAudioClientCapture != NULL) {
                ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
                pDevice->wasapi.pAudioClientCapture = NULL;
            }

            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create event for capture.", result);
        }
        ma_IAudioClient_SetEventHandle((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture, pDevice->wasapi.hEventCapture);

        pDevice->wasapi.periodSizeInFramesCapture = data.periodSizeInFramesOut;
        ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture, &pDevice->wasapi.actualPeriodSizeInFramesCapture);
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_device_init_internal_data__wasapi data;
        data.formatIn                   = pConfig->playback.format;
        data.channelsIn                 = pConfig->playback.channels;
        data.sampleRateIn               = pConfig->sampleRate;
        MA_COPY_MEMORY(data.channelMapIn, pConfig->playback.channelMap, sizeof(pConfig->playback.channelMap));
        data.usingDefaultFormat         = pDevice->playback.usingDefaultFormat;
        data.usingDefaultChannels       = pDevice->playback.usingDefaultChannels;
        data.usingDefaultSampleRate     = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap     = pDevice->playback.usingDefaultChannelMap;
        data.shareMode                  = pConfig->playback.shareMode;
        data.periodSizeInFramesIn       = pConfig->periodSizeInFrames;
        data.periodSizeInMillisecondsIn = pConfig->periodSizeInMilliseconds;
        data.periodsIn                  = pConfig->periods;
        data.noAutoConvertSRC           = pConfig->wasapi.noAutoConvertSRC;
        data.noDefaultQualitySRC        = pConfig->wasapi.noDefaultQualitySRC;
        data.noHardwareOffloading       = pConfig->wasapi.noHardwareOffloading;

        result = ma_device_init_internal__wasapi(pDevice->pContext, ma_device_type_playback, pConfig->playback.pDeviceID, &data);
        if (result != MA_SUCCESS) {
            if (pConfig->deviceType == ma_device_type_duplex) {
                if (pDevice->wasapi.pCaptureClient != NULL) {
                    ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
                    pDevice->wasapi.pCaptureClient = NULL;
                }
                if (pDevice->wasapi.pAudioClientCapture != NULL) {
                    ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
                    pDevice->wasapi.pAudioClientCapture = NULL;
                }

                CloseHandle(pDevice->wasapi.hEventCapture);
                pDevice->wasapi.hEventCapture = NULL;
            }
            return result;
        }

        pDevice->wasapi.pAudioClientPlayback         = data.pAudioClient;
        pDevice->wasapi.pRenderClient                = data.pRenderClient;

        pDevice->playback.internalFormat             = data.formatOut;
        pDevice->playback.internalChannels           = data.channelsOut;
        pDevice->playback.internalSampleRate         = data.sampleRateOut;
        MA_COPY_MEMORY(pDevice->playback.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->playback.internalPeriodSizeInFrames = data.periodSizeInFramesOut;
        pDevice->playback.internalPeriods            = data.periodsOut;
        ma_strcpy_s(pDevice->playback.name, sizeof(pDevice->playback.name), data.deviceName);

        /*
        The event for playback is needs to be manual reset because we want to explicitly control the fact that it becomes signalled
        only after the whole available space has been filled, never before.

        The playback event also needs to be initially set to a signaled state so that the first call to ma_device_write() is able
        to get passed WaitForMultipleObjects().
        */
        pDevice->wasapi.hEventPlayback = CreateEventW(NULL, FALSE, TRUE, NULL);  /* Auto reset, signaled by default. */
        if (pDevice->wasapi.hEventPlayback == NULL) {
            result = ma_result_from_GetLastError(GetLastError());

            if (pConfig->deviceType == ma_device_type_duplex) {
                if (pDevice->wasapi.pCaptureClient != NULL) {
                    ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
                    pDevice->wasapi.pCaptureClient = NULL;
                }
                if (pDevice->wasapi.pAudioClientCapture != NULL) {
                    ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
                    pDevice->wasapi.pAudioClientCapture = NULL;
                }

                CloseHandle(pDevice->wasapi.hEventCapture);
                pDevice->wasapi.hEventCapture = NULL;
            }

            if (pDevice->wasapi.pRenderClient != NULL) {
                ma_IAudioRenderClient_Release((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient);
                pDevice->wasapi.pRenderClient = NULL;
            }
            if (pDevice->wasapi.pAudioClientPlayback != NULL) {
                ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
                pDevice->wasapi.pAudioClientPlayback = NULL;
            }

            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create event for playback.", result);
        }
        ma_IAudioClient_SetEventHandle((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, pDevice->wasapi.hEventPlayback);

        pDevice->wasapi.periodSizeInFramesPlayback = data.periodSizeInFramesOut;
        ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, &pDevice->wasapi.actualPeriodSizeInFramesPlayback);
    }

    /*
    We need to register a notification client to detect when the device has been disabled, unplugged or re-routed (when the default device changes). When
    we are connecting to the default device we want to do automatic stream routing when the device is disabled or unplugged. Otherwise we want to just
    stop the device outright and let the application handle it.
    */
#ifdef MA_WIN32_DESKTOP
    if (pConfig->wasapi.noAutoStreamRouting == MA_FALSE) {
        if ((pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.pDeviceID == NULL) {
            pDevice->wasapi.allowCaptureAutoStreamRouting = MA_TRUE;
        }
        if ((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.pDeviceID == NULL) {
            pDevice->wasapi.allowPlaybackAutoStreamRouting = MA_TRUE;
        }
    }

    hr = ma_CoCreateInstance(pContext, MA_CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, MA_IID_IMMDeviceEnumerator, (void**)&pDeviceEnumerator);
    if (FAILED(hr)) {
        ma_device_uninit__wasapi(pDevice);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create device enumerator.", ma_result_from_HRESULT(hr));
    }

    pDevice->wasapi.notificationClient.lpVtbl  = (void*)&g_maNotificationCientVtbl;
    pDevice->wasapi.notificationClient.counter = 1;
    pDevice->wasapi.notificationClient.pDevice = pDevice;

    hr = pDeviceEnumerator->lpVtbl->RegisterEndpointNotificationCallback(pDeviceEnumerator, &pDevice->wasapi.notificationClient);
    if (SUCCEEDED(hr)) {
        pDevice->wasapi.pDeviceEnumerator = (ma_ptr)pDeviceEnumerator;
    } else {
        /* Not the end of the world if we fail to register the notification callback. We just won't support automatic stream routing. */
        ma_IMMDeviceEnumerator_Release(pDeviceEnumerator);
    }
#endif

    c89atomic_exchange_32(&pDevice->wasapi.isStartedCapture,  MA_FALSE);
    c89atomic_exchange_32(&pDevice->wasapi.isStartedPlayback, MA_FALSE);

    return MA_SUCCESS;
}

static ma_result ma_device__get_available_frames__wasapi(ma_device* pDevice, ma_IAudioClient* pAudioClient, ma_uint32* pFrameCount)
{
    ma_uint32 paddingFramesCount;
    HRESULT hr;
    ma_share_mode shareMode;

    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(pFrameCount != NULL);
    
    *pFrameCount = 0;

    if ((ma_ptr)pAudioClient != pDevice->wasapi.pAudioClientPlayback && (ma_ptr)pAudioClient != pDevice->wasapi.pAudioClientCapture) {
        return MA_INVALID_OPERATION;
    }

    hr = ma_IAudioClient_GetCurrentPadding(pAudioClient, &paddingFramesCount);
    if (FAILED(hr)) {
        return ma_result_from_HRESULT(hr);
    }

    /* Slightly different rules for exclusive and shared modes. */
    shareMode = ((ma_ptr)pAudioClient == pDevice->wasapi.pAudioClientPlayback) ? pDevice->playback.shareMode : pDevice->capture.shareMode;
    if (shareMode == ma_share_mode_exclusive) {
        *pFrameCount = paddingFramesCount;
    } else {
        if ((ma_ptr)pAudioClient == pDevice->wasapi.pAudioClientPlayback) {
            *pFrameCount = pDevice->wasapi.actualPeriodSizeInFramesPlayback - paddingFramesCount;
        } else {
            *pFrameCount = paddingFramesCount;
        }
    }

    return MA_SUCCESS;
}

static ma_bool32 ma_device_is_reroute_required__wasapi(ma_device* pDevice, ma_device_type deviceType)
{
    MA_ASSERT(pDevice != NULL);

    if (deviceType == ma_device_type_playback) {
        return pDevice->wasapi.hasDefaultPlaybackDeviceChanged;
    }

    if (deviceType == ma_device_type_capture || deviceType == ma_device_type_loopback) {
        return pDevice->wasapi.hasDefaultCaptureDeviceChanged;
    }
    
    return MA_FALSE;
}

static ma_result ma_device_reroute__wasapi(ma_device* pDevice, ma_device_type deviceType)
{
    ma_result result;

    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    if (deviceType == ma_device_type_playback) {
        c89atomic_exchange_32(&pDevice->wasapi.hasDefaultPlaybackDeviceChanged, MA_FALSE);
    }
    if (deviceType == ma_device_type_capture || deviceType == ma_device_type_loopback) {
        c89atomic_exchange_32(&pDevice->wasapi.hasDefaultCaptureDeviceChanged,  MA_FALSE);
    }
    

    #ifdef MA_DEBUG_OUTPUT
        printf("=== CHANGING DEVICE ===\n");
    #endif

    result = ma_device_reinit__wasapi(pDevice, deviceType);
    if (result != MA_SUCCESS) {
        return result;
    }

    ma_device__post_init_setup(pDevice, deviceType);

    return MA_SUCCESS;
}


static ma_result ma_device_stop__wasapi(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    /*
    It's possible for the main loop to get stuck if the device is disconnected.
    
    In loopback mode it's possible for WaitForSingleObject() to get stuck in a deadlock when nothing is being played. When nothing
    is being played, the event is never signalled internally by WASAPI which means we will deadlock when stopping the device.
    */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex || pDevice->type == ma_device_type_loopback) {
        SetEvent((HANDLE)pDevice->wasapi.hEventCapture);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        SetEvent((HANDLE)pDevice->wasapi.hEventPlayback);
    }

    return MA_SUCCESS;
}


static ma_result ma_device_main_loop__wasapi(ma_device* pDevice)
{
    ma_result result;
    HRESULT hr;
    ma_bool32 exitLoop = MA_FALSE;
    ma_uint32 framesWrittenToPlaybackDevice = 0;
    ma_uint32 mappedDeviceBufferSizeInFramesCapture = 0;
    ma_uint32 mappedDeviceBufferSizeInFramesPlayback = 0;
    ma_uint32 mappedDeviceBufferFramesRemainingCapture = 0;
    ma_uint32 mappedDeviceBufferFramesRemainingPlayback = 0;
    BYTE* pMappedDeviceBufferCapture = NULL;
    BYTE* pMappedDeviceBufferPlayback = NULL;
    ma_uint32 bpfCaptureDevice = ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
    ma_uint32 bpfPlaybackDevice = ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
    ma_uint32 bpfCaptureClient = ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
    ma_uint32 bpfPlaybackClient = ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
    ma_uint8  inputDataInClientFormat[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
    ma_uint32 inputDataInClientFormatCap = sizeof(inputDataInClientFormat) / bpfCaptureClient;
    ma_uint8  outputDataInClientFormat[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
    ma_uint32 outputDataInClientFormatCap = sizeof(outputDataInClientFormat) / bpfPlaybackClient;
    ma_uint32 outputDataInClientFormatCount = 0;
    ma_uint32 outputDataInClientFormatConsumed = 0;
    ma_uint32 periodSizeInFramesCapture = 0;

    MA_ASSERT(pDevice != NULL);

    /* The capture device needs to be started immediately. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex || pDevice->type == ma_device_type_loopback) {
        periodSizeInFramesCapture = pDevice->capture.internalPeriodSizeInFrames;

        hr = ma_IAudioClient_Start((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal capture device.", ma_result_from_HRESULT(hr));
        }
        c89atomic_exchange_32(&pDevice->wasapi.isStartedCapture, MA_TRUE);
    }

    while (ma_device__get_state(pDevice) == MA_STATE_STARTED && !exitLoop) {
        /* We may need to reroute the device. */
        if (ma_device_is_reroute_required__wasapi(pDevice, ma_device_type_playback)) {
            result = ma_device_reroute__wasapi(pDevice, ma_device_type_playback);
            if (result != MA_SUCCESS) {
                exitLoop = MA_TRUE;
                break;
            }
        }
        if (ma_device_is_reroute_required__wasapi(pDevice, ma_device_type_capture)) {
            result = ma_device_reroute__wasapi(pDevice, (pDevice->type == ma_device_type_loopback) ? ma_device_type_loopback : ma_device_type_capture);
            if (result != MA_SUCCESS) {
                exitLoop = MA_TRUE;
                break;
            }
        }

        switch (pDevice->type)
        {
            case ma_device_type_duplex:
            {
                ma_uint32 framesAvailableCapture;
                ma_uint32 framesAvailablePlayback;
                DWORD flagsCapture;    /* Passed to IAudioCaptureClient_GetBuffer(). */

                /* The process is to map the playback buffer and fill it as quickly as possible from input data. */
                if (pMappedDeviceBufferPlayback == NULL) {
                    /* WASAPI is weird with exclusive mode. You need to wait on the event _before_ querying the available frames. */
                    if (pDevice->playback.shareMode == ma_share_mode_exclusive) {
                        if (WaitForSingleObject(pDevice->wasapi.hEventPlayback, INFINITE) != WAIT_OBJECT_0) {
                            return MA_ERROR;   /* Wait failed. */
                        }
                    }

                    result = ma_device__get_available_frames__wasapi(pDevice, (ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, &framesAvailablePlayback);
                    if (result != MA_SUCCESS) {
                        return result;
                    }

                    /*printf("TRACE 1: framesAvailablePlayback=%d\n", framesAvailablePlayback);*/


                    /* In exclusive mode, the frame count needs to exactly match the value returned by GetCurrentPadding(). */
                    if (pDevice->playback.shareMode != ma_share_mode_exclusive) {
                        if (framesAvailablePlayback > pDevice->wasapi.periodSizeInFramesPlayback) {
                            framesAvailablePlayback = pDevice->wasapi.periodSizeInFramesPlayback;
                        }
                    }

                    /* If there's no frames available in the playback device we need to wait for more. */
                    if (framesAvailablePlayback == 0) {
                        /* In exclusive mode we waited at the top. */
                        if (pDevice->playback.shareMode != ma_share_mode_exclusive) {
                            if (WaitForSingleObject(pDevice->wasapi.hEventPlayback, INFINITE) != WAIT_OBJECT_0) {
                                return MA_ERROR;   /* Wait failed. */
                            }
                        }

                        continue;
                    }

                    /* We're ready to map the playback device's buffer. We don't release this until it's been entirely filled. */
                    hr = ma_IAudioRenderClient_GetBuffer((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient, framesAvailablePlayback, &pMappedDeviceBufferPlayback);
                    if (FAILED(hr)) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from playback device in preparation for writing to the device.", ma_result_from_HRESULT(hr));
                        exitLoop = MA_TRUE;
                        break;
                    }

                    mappedDeviceBufferSizeInFramesPlayback    = framesAvailablePlayback;
                    mappedDeviceBufferFramesRemainingPlayback = framesAvailablePlayback;
                }

                /* At this point we should have a buffer available for output. We need to keep writing input samples to it. */
                for (;;) {
                    /* Try grabbing some captured data if we haven't already got a mapped buffer. */
                    if (pMappedDeviceBufferCapture == NULL) {
                        if (pDevice->capture.shareMode == ma_share_mode_shared) {
                            if (WaitForSingleObject(pDevice->wasapi.hEventCapture, INFINITE) != WAIT_OBJECT_0) {
                                return MA_ERROR;   /* Wait failed. */
                            }
                        }

                        result = ma_device__get_available_frames__wasapi(pDevice, (ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture, &framesAvailableCapture);
                        if (result != MA_SUCCESS) {
                            exitLoop = MA_TRUE;
                            break;
                        }

                        /*printf("TRACE 2: framesAvailableCapture=%d\n", framesAvailableCapture);*/

                        /* Wait for more if nothing is available. */
                        if (framesAvailableCapture == 0) {
                            /* In exclusive mode we waited at the top. */
                            if (pDevice->capture.shareMode != ma_share_mode_shared) {
                                if (WaitForSingleObject(pDevice->wasapi.hEventCapture, INFINITE) != WAIT_OBJECT_0) {
                                    return MA_ERROR;   /* Wait failed. */
                                }
                            }

                            continue;
                        }

                        /* Getting here means there's data available for writing to the output device. */
                        mappedDeviceBufferSizeInFramesCapture = ma_min(framesAvailableCapture, periodSizeInFramesCapture);
                        hr = ma_IAudioCaptureClient_GetBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, (BYTE**)&pMappedDeviceBufferCapture, &mappedDeviceBufferSizeInFramesCapture, &flagsCapture, NULL, NULL);
                        if (FAILED(hr)) {
                            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from capture device in preparation for writing to the device.", ma_result_from_HRESULT(hr));
                            exitLoop = MA_TRUE;
                            break;
                        }


                        /* Overrun detection. */
                        if ((flagsCapture & MA_AUDCLNT_BUFFERFLAGS_DATA_DISCONTINUITY) != 0) {
                            /* Glitched. Probably due to an overrun. */
                        #ifdef MA_DEBUG_OUTPUT
                            printf("[WASAPI] Data discontinuity (possible overrun). framesAvailableCapture=%d, mappedBufferSizeInFramesCapture=%d\n", framesAvailableCapture, mappedDeviceBufferSizeInFramesCapture);
                        #endif

                            /*
                            Exeriment: If we get an overrun it probably means we're straddling the end of the buffer. In order to prevent a never-ending sequence of glitches let's experiment
                            by dropping every frame until we're left with only a single period. To do this we just keep retrieving and immediately releasing buffers until we're down to the
                            last period.
                            */
                            if (framesAvailableCapture >= pDevice->wasapi.actualPeriodSizeInFramesCapture) {
                            #ifdef MA_DEBUG_OUTPUT
                                printf("[WASAPI] Synchronizing capture stream. ");
                            #endif
                                do
                                {
                                    hr = ma_IAudioCaptureClient_ReleaseBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, mappedDeviceBufferSizeInFramesCapture);
                                    if (FAILED(hr)) {
                                        break;
                                    }

                                    framesAvailableCapture -= mappedDeviceBufferSizeInFramesCapture;
                                    
                                    if (framesAvailableCapture > 0) {
                                        mappedDeviceBufferSizeInFramesCapture = ma_min(framesAvailableCapture, periodSizeInFramesCapture);
                                        hr = ma_IAudioCaptureClient_GetBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, (BYTE**)&pMappedDeviceBufferCapture, &mappedDeviceBufferSizeInFramesCapture, &flagsCapture, NULL, NULL);
                                        if (FAILED(hr)) {
                                            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from capture device in preparation for writing to the device.", ma_result_from_HRESULT(hr));
                                            exitLoop = MA_TRUE;
                                            break;
                                        }
                                    } else {
                                        pMappedDeviceBufferCapture = NULL;
                                        mappedDeviceBufferSizeInFramesCapture = 0;
                                    }
                                } while (framesAvailableCapture > periodSizeInFramesCapture);
                            #ifdef MA_DEBUG_OUTPUT
                                printf("framesAvailableCapture=%d, mappedBufferSizeInFramesCapture=%d\n", framesAvailableCapture, mappedDeviceBufferSizeInFramesCapture);
                            #endif
                            }
                        } else {
                        #ifdef MA_DEBUG_OUTPUT
                            if (flagsCapture != 0) {
                                printf("[WASAPI] Capture Flags: %ld\n", flagsCapture);
                            }
                        #endif
                        }

                        mappedDeviceBufferFramesRemainingCapture = mappedDeviceBufferSizeInFramesCapture;
                    }


                    /* At this point we should have both input and output data available. We now need to convert the data and post it to the client. */
                    for (;;) {
                        BYTE* pRunningDeviceBufferCapture;
                        BYTE* pRunningDeviceBufferPlayback;
                        ma_uint32 framesToProcess;
                        ma_uint32 framesProcessed;

                        pRunningDeviceBufferCapture  = pMappedDeviceBufferCapture  + ((mappedDeviceBufferSizeInFramesCapture  - mappedDeviceBufferFramesRemainingCapture ) * bpfCaptureDevice);
                        pRunningDeviceBufferPlayback = pMappedDeviceBufferPlayback + ((mappedDeviceBufferSizeInFramesPlayback - mappedDeviceBufferFramesRemainingPlayback) * bpfPlaybackDevice);
                        
                        /* There may be some data sitting in the converter that needs to be processed first. Once this is exhaused, run the data callback again. */
                        if (!pDevice->playback.converter.isPassthrough && outputDataInClientFormatConsumed < outputDataInClientFormatCount) {
                            ma_uint64 convertedFrameCountClient = (outputDataInClientFormatCount - outputDataInClientFormatConsumed);
                            ma_uint64 convertedFrameCountDevice = mappedDeviceBufferFramesRemainingPlayback;
                            void* pConvertedFramesClient = outputDataInClientFormat + (outputDataInClientFormatConsumed * bpfPlaybackClient);
                            void* pConvertedFramesDevice = pRunningDeviceBufferPlayback;
                            result = ma_data_converter_process_pcm_frames(&pDevice->playback.converter, pConvertedFramesClient, &convertedFrameCountClient, pConvertedFramesDevice, &convertedFrameCountDevice);
                            if (result != MA_SUCCESS) {
                                break;
                            }

                            outputDataInClientFormatConsumed          += (ma_uint32)convertedFrameCountClient;  /* Safe cast. */
                            mappedDeviceBufferFramesRemainingPlayback -= (ma_uint32)convertedFrameCountDevice;  /* Safe cast. */

                            if (mappedDeviceBufferFramesRemainingPlayback == 0) {
                                break;
                            }
                        }

                        /*
                        Getting here means we need to fire the callback. If format conversion is unnecessary, we can optimize this by passing the pointers to the internal
                        buffers directly to the callback.
                        */
                        if (pDevice->capture.converter.isPassthrough && pDevice->playback.converter.isPassthrough) {
                            /* Optimal path. We can pass mapped pointers directly to the callback. */
                            framesToProcess = ma_min(mappedDeviceBufferFramesRemainingCapture, mappedDeviceBufferFramesRemainingPlayback);
                            framesProcessed = framesToProcess;

                            ma_device__on_data(pDevice, pRunningDeviceBufferPlayback, pRunningDeviceBufferCapture, framesToProcess);

                            mappedDeviceBufferFramesRemainingCapture  -= framesProcessed;
                            mappedDeviceBufferFramesRemainingPlayback -= framesProcessed;

                            if (mappedDeviceBufferFramesRemainingCapture == 0) {
                                break;  /* Exhausted input data. */
                            }
                            if (mappedDeviceBufferFramesRemainingPlayback == 0) {
                                break;  /* Exhausted output data. */
                            }
                        } else if (pDevice->capture.converter.isPassthrough) {
                            /* The input buffer is a passthrough, but the playback buffer requires a conversion. */
                            framesToProcess = ma_min(mappedDeviceBufferFramesRemainingCapture, outputDataInClientFormatCap);
                            framesProcessed = framesToProcess;

                            ma_device__on_data(pDevice, outputDataInClientFormat, pRunningDeviceBufferCapture, framesToProcess);
                            outputDataInClientFormatCount    = framesProcessed;
                            outputDataInClientFormatConsumed = 0;

                            mappedDeviceBufferFramesRemainingCapture -= framesProcessed;
                            if (mappedDeviceBufferFramesRemainingCapture == 0) {
                                break;  /* Exhausted input data. */
                            }
                        } else if (pDevice->playback.converter.isPassthrough) {
                            /* The input buffer requires conversion, the playback buffer is passthrough. */
                            ma_uint64 capturedDeviceFramesToProcess = mappedDeviceBufferFramesRemainingCapture;
                            ma_uint64 capturedClientFramesToProcess = ma_min(inputDataInClientFormatCap, mappedDeviceBufferFramesRemainingPlayback);

                            result = ma_data_converter_process_pcm_frames(&pDevice->capture.converter, pRunningDeviceBufferCapture, &capturedDeviceFramesToProcess, inputDataInClientFormat, &capturedClientFramesToProcess);
                            if (result != MA_SUCCESS) {
                                break;
                            }

                            if (capturedClientFramesToProcess == 0) {
                                break;
                            }

                            ma_device__on_data(pDevice, pRunningDeviceBufferPlayback, inputDataInClientFormat, (ma_uint32)capturedClientFramesToProcess);   /* Safe cast. */

                            mappedDeviceBufferFramesRemainingCapture  -= (ma_uint32)capturedDeviceFramesToProcess;
                            mappedDeviceBufferFramesRemainingPlayback -= (ma_uint32)capturedClientFramesToProcess;
                        } else {
                            ma_uint64 capturedDeviceFramesToProcess = mappedDeviceBufferFramesRemainingCapture;
                            ma_uint64 capturedClientFramesToProcess = ma_min(inputDataInClientFormatCap, outputDataInClientFormatCap);

                            result = ma_data_converter_process_pcm_frames(&pDevice->capture.converter, pRunningDeviceBufferCapture, &capturedDeviceFramesToProcess, inputDataInClientFormat, &capturedClientFramesToProcess);
                            if (result != MA_SUCCESS) {
                                break;
                            }

                            if (capturedClientFramesToProcess == 0) {
                                break;
                            }

                            ma_device__on_data(pDevice, outputDataInClientFormat, inputDataInClientFormat, (ma_uint32)capturedClientFramesToProcess);
                            
                            mappedDeviceBufferFramesRemainingCapture -= (ma_uint32)capturedDeviceFramesToProcess;
                            outputDataInClientFormatCount             = (ma_uint32)capturedClientFramesToProcess;
                            outputDataInClientFormatConsumed          = 0;
                        }
                    }


                    /* If at this point we've run out of capture data we need to release the buffer. */
                    if (mappedDeviceBufferFramesRemainingCapture == 0 && pMappedDeviceBufferCapture != NULL) {
                        hr = ma_IAudioCaptureClient_ReleaseBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, mappedDeviceBufferSizeInFramesCapture);
                        if (FAILED(hr)) {
                            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to release internal buffer from capture device after reading from the device.", ma_result_from_HRESULT(hr));
                            exitLoop = MA_TRUE;
                            break;
                        }

                        /*printf("TRACE: Released capture buffer\n");*/

                        pMappedDeviceBufferCapture = NULL;
                        mappedDeviceBufferFramesRemainingCapture = 0;
                        mappedDeviceBufferSizeInFramesCapture    = 0;
                    }

                    /* Get out of this loop if we're run out of room in the playback buffer. */
                    if (mappedDeviceBufferFramesRemainingPlayback == 0) {
                        break;
                    }
                }


                /* If at this point we've run out of data we need to release the buffer. */
                if (mappedDeviceBufferFramesRemainingPlayback == 0 && pMappedDeviceBufferPlayback != NULL) {
                    hr = ma_IAudioRenderClient_ReleaseBuffer((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient, mappedDeviceBufferSizeInFramesPlayback, 0);
                    if (FAILED(hr)) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to release internal buffer from playback device after writing to the device.", ma_result_from_HRESULT(hr));
                        exitLoop = MA_TRUE;
                        break;
                    }

                    /*printf("TRACE: Released playback buffer\n");*/
                    framesWrittenToPlaybackDevice += mappedDeviceBufferSizeInFramesPlayback;

                    pMappedDeviceBufferPlayback = NULL;
                    mappedDeviceBufferFramesRemainingPlayback = 0;
                    mappedDeviceBufferSizeInFramesPlayback    = 0;
                }

                if (!pDevice->wasapi.isStartedPlayback) {
                    ma_uint32 startThreshold = pDevice->playback.internalPeriodSizeInFrames * 1;

                    /* Prevent a deadlock. If we don't clamp against the actual buffer size we'll never end up starting the playback device which will result in a deadlock. */
                    if (startThreshold > pDevice->wasapi.actualPeriodSizeInFramesPlayback) {
                        startThreshold = pDevice->wasapi.actualPeriodSizeInFramesPlayback;
                    }

                    if (pDevice->playback.shareMode == ma_share_mode_exclusive || framesWrittenToPlaybackDevice >= startThreshold) {
                        hr = ma_IAudioClient_Start((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
                        if (FAILED(hr)) {
                            ma_IAudioClient_Stop((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
                            ma_IAudioClient_Reset((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
                            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal playback device.", ma_result_from_HRESULT(hr));
                        }
                        c89atomic_exchange_32(&pDevice->wasapi.isStartedPlayback, MA_TRUE);
                    }
                }
            } break;



            case ma_device_type_capture:
            case ma_device_type_loopback:
            {
                ma_uint32 framesAvailableCapture;
                DWORD flagsCapture;    /* Passed to IAudioCaptureClient_GetBuffer(). */

                /* Wait for data to become available first. */
                if (WaitForSingleObject(pDevice->wasapi.hEventCapture, INFINITE) != WAIT_OBJECT_0) {
                    exitLoop = MA_TRUE;
                    break;   /* Wait failed. */
                }

                /* See how many frames are available. Since we waited at the top, I don't think this should ever return 0. I'm checking for this anyway. */
                result = ma_device__get_available_frames__wasapi(pDevice, (ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture, &framesAvailableCapture);
                if (result != MA_SUCCESS) {
                    exitLoop = MA_TRUE;
                    break;
                }

                if (framesAvailableCapture < pDevice->wasapi.periodSizeInFramesCapture) {
                    continue;   /* Nothing available. Keep waiting. */
                }

                /* Map the data buffer in preparation for sending to the client. */
                mappedDeviceBufferSizeInFramesCapture = framesAvailableCapture;
                hr = ma_IAudioCaptureClient_GetBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, (BYTE**)&pMappedDeviceBufferCapture, &mappedDeviceBufferSizeInFramesCapture, &flagsCapture, NULL, NULL);
                if (FAILED(hr)) {
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from capture device in preparation for writing to the device.", ma_result_from_HRESULT(hr));
                    exitLoop = MA_TRUE;
                    break;
                }

                /* Overrun detection. */
                if ((flagsCapture & MA_AUDCLNT_BUFFERFLAGS_DATA_DISCONTINUITY) != 0) {
                    /* Glitched. Probably due to an overrun. */
                #ifdef MA_DEBUG_OUTPUT
                    printf("[WASAPI] Data discontinuity (possible overrun). framesAvailableCapture=%d, mappedBufferSizeInFramesCapture=%d\n", framesAvailableCapture, mappedDeviceBufferSizeInFramesCapture);
                #endif

                    /*
                    Exeriment: If we get an overrun it probably means we're straddling the end of the buffer. In order to prevent a never-ending sequence of glitches let's experiment
                    by dropping every frame until we're left with only a single period. To do this we just keep retrieving and immediately releasing buffers until we're down to the
                    last period.
                    */
                    if (framesAvailableCapture >= pDevice->wasapi.actualPeriodSizeInFramesCapture) {
                    #ifdef MA_DEBUG_OUTPUT
                        printf("[WASAPI] Synchronizing capture stream. ");
                    #endif
                        do
                        {
                            hr = ma_IAudioCaptureClient_ReleaseBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, mappedDeviceBufferSizeInFramesCapture);
                            if (FAILED(hr)) {
                                break;
                            }

                            framesAvailableCapture -= mappedDeviceBufferSizeInFramesCapture;
                                    
                            if (framesAvailableCapture > 0) {
                                mappedDeviceBufferSizeInFramesCapture = ma_min(framesAvailableCapture, periodSizeInFramesCapture);
                                hr = ma_IAudioCaptureClient_GetBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, (BYTE**)&pMappedDeviceBufferCapture, &mappedDeviceBufferSizeInFramesCapture, &flagsCapture, NULL, NULL);
                                if (FAILED(hr)) {
                                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from capture device in preparation for writing to the device.", ma_result_from_HRESULT(hr));
                                    exitLoop = MA_TRUE;
                                    break;
                                }
                            } else {
                                pMappedDeviceBufferCapture = NULL;
                                mappedDeviceBufferSizeInFramesCapture = 0;
                            }
                        } while (framesAvailableCapture > periodSizeInFramesCapture);
                    #ifdef MA_DEBUG_OUTPUT
                        printf("framesAvailableCapture=%d, mappedBufferSizeInFramesCapture=%d\n", framesAvailableCapture, mappedDeviceBufferSizeInFramesCapture);
                    #endif
                    }
                } else {
                #ifdef MA_DEBUG_OUTPUT
                    if (flagsCapture != 0) {
                        printf("[WASAPI] Capture Flags: %ld\n", flagsCapture);
                    }
                #endif
                }

                /* We should have a buffer at this point, but let's just do a sanity check anyway. */
                if (mappedDeviceBufferSizeInFramesCapture > 0 && pMappedDeviceBufferCapture != NULL) {
                    ma_device__send_frames_to_client(pDevice, mappedDeviceBufferSizeInFramesCapture, pMappedDeviceBufferCapture);

                    /* At this point we're done with the buffer. */
                    hr = ma_IAudioCaptureClient_ReleaseBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, mappedDeviceBufferSizeInFramesCapture);
                    pMappedDeviceBufferCapture = NULL;    /* <-- Important. Not doing this can result in an error once we leave this loop because it will use this to know whether or not a final ReleaseBuffer() needs to be called. */
                    mappedDeviceBufferSizeInFramesCapture = 0;
                    if (FAILED(hr)) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to release internal buffer from capture device after reading from the device.", ma_result_from_HRESULT(hr));
                        exitLoop = MA_TRUE;
                        break;
                    }
                }
            } break;



            case ma_device_type_playback:
            {
                ma_uint32 framesAvailablePlayback;

                /* Wait for space to become available first. */
                if (WaitForSingleObject(pDevice->wasapi.hEventPlayback, INFINITE) != WAIT_OBJECT_0) {
                    exitLoop = MA_TRUE;
                    break;   /* Wait failed. */
                }

                /* Check how much space is available. If this returns 0 we just keep waiting. */
                result = ma_device__get_available_frames__wasapi(pDevice, (ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, &framesAvailablePlayback);
                if (result != MA_SUCCESS) {
                    exitLoop = MA_TRUE;
                    break;
                }

                if (framesAvailablePlayback < pDevice->wasapi.periodSizeInFramesPlayback) {
                    continue;   /* No space available. */
                }

                /* Map a the data buffer in preparation for the callback. */
                hr = ma_IAudioRenderClient_GetBuffer((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient, framesAvailablePlayback, &pMappedDeviceBufferPlayback);
                if (FAILED(hr)) {
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from playback device in preparation for writing to the device.", ma_result_from_HRESULT(hr));
                    exitLoop = MA_TRUE;
                    break;
                }

                /* We should have a buffer at this point. */
                ma_device__read_frames_from_client(pDevice, framesAvailablePlayback, pMappedDeviceBufferPlayback);

                /* At this point we're done writing to the device and we just need to release the buffer. */
                hr = ma_IAudioRenderClient_ReleaseBuffer((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient, framesAvailablePlayback, 0);
                pMappedDeviceBufferPlayback = NULL;    /* <-- Important. Not doing this can result in an error once we leave this loop because it will use this to know whether or not a final ReleaseBuffer() needs to be called. */
                mappedDeviceBufferSizeInFramesPlayback = 0;

                if (FAILED(hr)) {
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to release internal buffer from playback device after writing to the device.", ma_result_from_HRESULT(hr));
                    exitLoop = MA_TRUE;
                    break;
                }

                framesWrittenToPlaybackDevice += framesAvailablePlayback;
                if (!pDevice->wasapi.isStartedPlayback) {
                    if (pDevice->playback.shareMode == ma_share_mode_exclusive || framesWrittenToPlaybackDevice >= pDevice->playback.internalPeriodSizeInFrames*1) {
                        hr = ma_IAudioClient_Start((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
                        if (FAILED(hr)) {
                            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal playback device.", ma_result_from_HRESULT(hr));
                            exitLoop = MA_TRUE;
                            break;
                        }
                        c89atomic_exchange_32(&pDevice->wasapi.isStartedPlayback, MA_TRUE);
                    }
                }
            } break;

            default: return MA_INVALID_ARGS;
        }
    }

    /* Here is where the device needs to be stopped. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex || pDevice->type == ma_device_type_loopback) {
        /* Any mapped buffers need to be released. */
        if (pMappedDeviceBufferCapture != NULL) {
            hr = ma_IAudioCaptureClient_ReleaseBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, mappedDeviceBufferSizeInFramesCapture);
        }

        hr = ma_IAudioClient_Stop((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to stop internal capture device.", ma_result_from_HRESULT(hr));
        }

        /* The audio client needs to be reset otherwise restarting will fail. */
        hr = ma_IAudioClient_Reset((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to reset internal capture device.", ma_result_from_HRESULT(hr));
        }

        c89atomic_exchange_32(&pDevice->wasapi.isStartedCapture, MA_FALSE);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        /* Any mapped buffers need to be released. */
        if (pMappedDeviceBufferPlayback != NULL) {
            hr = ma_IAudioRenderClient_ReleaseBuffer((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient, mappedDeviceBufferSizeInFramesPlayback, 0);
        }

        /*
        The buffer needs to be drained before stopping the device. Not doing this will result in the last few frames not getting output to
        the speakers. This is a problem for very short sounds because it'll result in a significant portion of it not getting played.
        */
        if (pDevice->wasapi.isStartedPlayback) {
            /* We need to make sure we put a timeout here or else we'll risk getting stuck in a deadlock in some cases. */
            DWORD waitTime = pDevice->wasapi.actualPeriodSizeInFramesPlayback / pDevice->playback.internalSampleRate;

            if (pDevice->playback.shareMode == ma_share_mode_exclusive) {
                WaitForSingleObject(pDevice->wasapi.hEventPlayback, waitTime);
            } else {
                ma_uint32 prevFramesAvaialablePlayback = (ma_uint32)-1;
                ma_uint32 framesAvailablePlayback;
                for (;;) {
                    result = ma_device__get_available_frames__wasapi(pDevice, (ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, &framesAvailablePlayback);
                    if (result != MA_SUCCESS) {
                        break;
                    }

                    if (framesAvailablePlayback >= pDevice->wasapi.actualPeriodSizeInFramesPlayback) {
                        break;
                    }

                    /*
                    Just a safety check to avoid an infinite loop. If this iteration results in a situation where the number of available frames
                    has not changed, get out of the loop. I don't think this should ever happen, but I think it's nice to have just in case.
                    */
                    if (framesAvailablePlayback == prevFramesAvaialablePlayback) {
                        break;
                    }
                    prevFramesAvaialablePlayback = framesAvailablePlayback;

                    WaitForSingleObject(pDevice->wasapi.hEventPlayback, waitTime);
                    ResetEvent(pDevice->wasapi.hEventPlayback); /* Manual reset. */
                }
            }
        }

        hr = ma_IAudioClient_Stop((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to stop internal playback device.", ma_result_from_HRESULT(hr));
        }

        /* The audio client needs to be reset otherwise restarting will fail. */
        hr = ma_IAudioClient_Reset((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to reset internal playback device.", ma_result_from_HRESULT(hr));
        }

        c89atomic_exchange_32(&pDevice->wasapi.isStartedPlayback, MA_FALSE);
    }

    return MA_SUCCESS;
}

static ma_result ma_context_uninit__wasapi(ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pContext->backend == ma_backend_wasapi);
    (void)pContext;

    return MA_SUCCESS;
}

static ma_result ma_context_init__wasapi(const ma_context_config* pConfig, ma_context* pContext)
{
    ma_result result = MA_SUCCESS;

    MA_ASSERT(pContext != NULL);

    (void)pConfig;

#ifdef MA_WIN32_DESKTOP
    /*
    WASAPI is only supported in Vista SP1 and newer. The reason for SP1 and not the base version of Vista is that event-driven
    exclusive mode does not work until SP1.

    Unfortunately older compilers don't define these functions so we need to dynamically load them in order to avoid a lin error.
    */
    {
        ma_OSVERSIONINFOEXW osvi;
        ma_handle kernel32DLL;
        ma_PFNVerifyVersionInfoW _VerifyVersionInfoW;
        ma_PFNVerSetConditionMask _VerSetConditionMask;

        kernel32DLL = ma_dlopen(pContext, "kernel32.dll");
        if (kernel32DLL == NULL) {
            return MA_NO_BACKEND;
        }

        _VerifyVersionInfoW = (ma_PFNVerifyVersionInfoW)ma_dlsym(pContext, kernel32DLL, "VerifyVersionInfoW");
        _VerSetConditionMask = (ma_PFNVerSetConditionMask)ma_dlsym(pContext, kernel32DLL, "VerSetConditionMask");
        if (_VerifyVersionInfoW == NULL || _VerSetConditionMask == NULL) {
            ma_dlclose(pContext, kernel32DLL);
            return MA_NO_BACKEND;
        }

        MA_ZERO_OBJECT(&osvi);
        osvi.dwOSVersionInfoSize = sizeof(osvi);
        osvi.dwMajorVersion = ((MA_WIN32_WINNT_VISTA >> 8) & 0xFF);
        osvi.dwMinorVersion = ((MA_WIN32_WINNT_VISTA >> 0) & 0xFF);
        osvi.wServicePackMajor = 1;
        if (_VerifyVersionInfoW(&osvi, MA_VER_MAJORVERSION | MA_VER_MINORVERSION | MA_VER_SERVICEPACKMAJOR, _VerSetConditionMask(_VerSetConditionMask(_VerSetConditionMask(0, MA_VER_MAJORVERSION, MA_VER_GREATER_EQUAL), MA_VER_MINORVERSION, MA_VER_GREATER_EQUAL), MA_VER_SERVICEPACKMAJOR, MA_VER_GREATER_EQUAL))) {
            result = MA_SUCCESS;
        } else {
            result = MA_NO_BACKEND;
        }

        ma_dlclose(pContext, kernel32DLL);
    }
#endif

    if (result != MA_SUCCESS) {
        return result;
    }

    pContext->onUninit         = ma_context_uninit__wasapi;
    pContext->onDeviceIDEqual  = ma_context_is_device_id_equal__wasapi;
    pContext->onEnumDevices    = ma_context_enumerate_devices__wasapi;
    pContext->onGetDeviceInfo  = ma_context_get_device_info__wasapi;
    pContext->onDeviceInit     = ma_device_init__wasapi;
    pContext->onDeviceUninit   = ma_device_uninit__wasapi;
    pContext->onDeviceStart    = NULL;                      /* Not used. Started in onDeviceMainLoop. */
    pContext->onDeviceStop     = ma_device_stop__wasapi;    /* Required to ensure the capture event is signalled when stopping a loopback device while nothing is playing. */
    pContext->onDeviceMainLoop = ma_device_main_loop__wasapi;

    return result;
}
#endif

/******************************************************************************

DirectSound Backend

******************************************************************************/
#ifdef MA_HAS_DSOUND
/*#include <dsound.h>*/

static const GUID MA_GUID_IID_DirectSoundNotify = {0xb0210783, 0x89cd, 0x11d0, {0xaf, 0x08, 0x00, 0xa0, 0xc9, 0x25, 0xcd, 0x16}};

/* miniaudio only uses priority or exclusive modes. */
#define MA_DSSCL_NORMAL                 1
#define MA_DSSCL_PRIORITY               2
#define MA_DSSCL_EXCLUSIVE              3
#define MA_DSSCL_WRITEPRIMARY           4

#define MA_DSCAPS_PRIMARYMONO           0x00000001
#define MA_DSCAPS_PRIMARYSTEREO         0x00000002
#define MA_DSCAPS_PRIMARY8BIT           0x00000004
#define MA_DSCAPS_PRIMARY16BIT          0x00000008
#define MA_DSCAPS_CONTINUOUSRATE        0x00000010
#define MA_DSCAPS_EMULDRIVER            0x00000020
#define MA_DSCAPS_CERTIFIED             0x00000040
#define MA_DSCAPS_SECONDARYMONO         0x00000100
#define MA_DSCAPS_SECONDARYSTEREO       0x00000200
#define MA_DSCAPS_SECONDARY8BIT         0x00000400
#define MA_DSCAPS_SECONDARY16BIT        0x00000800

#define MA_DSBCAPS_PRIMARYBUFFER        0x00000001
#define MA_DSBCAPS_STATIC               0x00000002
#define MA_DSBCAPS_LOCHARDWARE          0x00000004
#define MA_DSBCAPS_LOCSOFTWARE          0x00000008
#define MA_DSBCAPS_CTRL3D               0x00000010
#define MA_DSBCAPS_CTRLFREQUENCY        0x00000020
#define MA_DSBCAPS_CTRLPAN              0x00000040
#define MA_DSBCAPS_CTRLVOLUME           0x00000080
#define MA_DSBCAPS_CTRLPOSITIONNOTIFY   0x00000100
#define MA_DSBCAPS_CTRLFX               0x00000200
#define MA_DSBCAPS_STICKYFOCUS          0x00004000
#define MA_DSBCAPS_GLOBALFOCUS          0x00008000
#define MA_DSBCAPS_GETCURRENTPOSITION2  0x00010000
#define MA_DSBCAPS_MUTE3DATMAXDISTANCE  0x00020000
#define MA_DSBCAPS_LOCDEFER             0x00040000
#define MA_DSBCAPS_TRUEPLAYPOSITION     0x00080000

#define MA_DSBPLAY_LOOPING              0x00000001
#define MA_DSBPLAY_LOCHARDWARE          0x00000002
#define MA_DSBPLAY_LOCSOFTWARE          0x00000004
#define MA_DSBPLAY_TERMINATEBY_TIME     0x00000008
#define MA_DSBPLAY_TERMINATEBY_DISTANCE 0x00000010
#define MA_DSBPLAY_TERMINATEBY_PRIORITY 0x00000020

#define MA_DSCBSTART_LOOPING            0x00000001

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwReserved;
    WAVEFORMATEX* lpwfxFormat;
    GUID guid3DAlgorithm;
} MA_DSBUFFERDESC;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwReserved;
    WAVEFORMATEX* lpwfxFormat;
    DWORD dwFXCount;
    void* lpDSCFXDesc;  /* <-- miniaudio doesn't use this, so set to void*. */
} MA_DSCBUFFERDESC;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwMinSecondarySampleRate;
    DWORD dwMaxSecondarySampleRate;
    DWORD dwPrimaryBuffers;
    DWORD dwMaxHwMixingAllBuffers;
    DWORD dwMaxHwMixingStaticBuffers;
    DWORD dwMaxHwMixingStreamingBuffers;
    DWORD dwFreeHwMixingAllBuffers;
    DWORD dwFreeHwMixingStaticBuffers;
    DWORD dwFreeHwMixingStreamingBuffers;
    DWORD dwMaxHw3DAllBuffers;
    DWORD dwMaxHw3DStaticBuffers;
    DWORD dwMaxHw3DStreamingBuffers;
    DWORD dwFreeHw3DAllBuffers;
    DWORD dwFreeHw3DStaticBuffers;
    DWORD dwFreeHw3DStreamingBuffers;
    DWORD dwTotalHwMemBytes;
    DWORD dwFreeHwMemBytes;
    DWORD dwMaxContigFreeHwMemBytes;
    DWORD dwUnlockTransferRateHwBuffers;
    DWORD dwPlayCpuOverheadSwBuffers;
    DWORD dwReserved1;
    DWORD dwReserved2;
} MA_DSCAPS;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwUnlockTransferRate;
    DWORD dwPlayCpuOverhead;
} MA_DSBCAPS;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwFormats;
    DWORD dwChannels;
} MA_DSCCAPS;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwReserved;
} MA_DSCBCAPS;

typedef struct
{
    DWORD  dwOffset;
    HANDLE hEventNotify;
} MA_DSBPOSITIONNOTIFY;

typedef struct ma_IDirectSound              ma_IDirectSound;
typedef struct ma_IDirectSoundBuffer        ma_IDirectSoundBuffer;
typedef struct ma_IDirectSoundCapture       ma_IDirectSoundCapture;
typedef struct ma_IDirectSoundCaptureBuffer ma_IDirectSoundCaptureBuffer;
typedef struct ma_IDirectSoundNotify        ma_IDirectSoundNotify;


/*
COM objects. The way these work is that you have a vtable (a list of function pointers, kind of
like how C++ works internally), and then you have a structure with a single member, which is a
pointer to the vtable. The vtable is where the methods of the object are defined. Methods need
to be in a specific order, and parent classes need to have their methods declared first.
*/

/* IDirectSound */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IDirectSound* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IDirectSound* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IDirectSound* pThis);

    /* IDirectSound */
    HRESULT (STDMETHODCALLTYPE * CreateSoundBuffer)   (ma_IDirectSound* pThis, const MA_DSBUFFERDESC* pDSBufferDesc, ma_IDirectSoundBuffer** ppDSBuffer, void* pUnkOuter);
    HRESULT (STDMETHODCALLTYPE * GetCaps)             (ma_IDirectSound* pThis, MA_DSCAPS* pDSCaps);
    HRESULT (STDMETHODCALLTYPE * DuplicateSoundBuffer)(ma_IDirectSound* pThis, ma_IDirectSoundBuffer* pDSBufferOriginal, ma_IDirectSoundBuffer** ppDSBufferDuplicate);
    HRESULT (STDMETHODCALLTYPE * SetCooperativeLevel) (ma_IDirectSound* pThis, HWND hwnd, DWORD dwLevel);
    HRESULT (STDMETHODCALLTYPE * Compact)             (ma_IDirectSound* pThis);
    HRESULT (STDMETHODCALLTYPE * GetSpeakerConfig)    (ma_IDirectSound* pThis, DWORD* pSpeakerConfig);
    HRESULT (STDMETHODCALLTYPE * SetSpeakerConfig)    (ma_IDirectSound* pThis, DWORD dwSpeakerConfig);
    HRESULT (STDMETHODCALLTYPE * Initialize)          (ma_IDirectSound* pThis, const GUID* pGuidDevice);
} ma_IDirectSoundVtbl;
struct ma_IDirectSound
{
    ma_IDirectSoundVtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IDirectSound_QueryInterface(ma_IDirectSound* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IDirectSound_AddRef(ma_IDirectSound* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IDirectSound_Release(ma_IDirectSound* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IDirectSound_CreateSoundBuffer(ma_IDirectSound* pThis, const MA_DSBUFFERDESC* pDSBufferDesc, ma_IDirectSoundBuffer** ppDSBuffer, void* pUnkOuter) { return pThis->lpVtbl->CreateSoundBuffer(pThis, pDSBufferDesc, ppDSBuffer, pUnkOuter); }
static MA_INLINE HRESULT ma_IDirectSound_GetCaps(ma_IDirectSound* pThis, MA_DSCAPS* pDSCaps)                            { return pThis->lpVtbl->GetCaps(pThis, pDSCaps); }
static MA_INLINE HRESULT ma_IDirectSound_DuplicateSoundBuffer(ma_IDirectSound* pThis, ma_IDirectSoundBuffer* pDSBufferOriginal, ma_IDirectSoundBuffer** ppDSBufferDuplicate) { return pThis->lpVtbl->DuplicateSoundBuffer(pThis, pDSBufferOriginal, ppDSBufferDuplicate); }
static MA_INLINE HRESULT ma_IDirectSound_SetCooperativeLevel(ma_IDirectSound* pThis, HWND hwnd, DWORD dwLevel)          { return pThis->lpVtbl->SetCooperativeLevel(pThis, hwnd, dwLevel); }
static MA_INLINE HRESULT ma_IDirectSound_Compact(ma_IDirectSound* pThis)                                                { return pThis->lpVtbl->Compact(pThis); }
static MA_INLINE HRESULT ma_IDirectSound_GetSpeakerConfig(ma_IDirectSound* pThis, DWORD* pSpeakerConfig)                { return pThis->lpVtbl->GetSpeakerConfig(pThis, pSpeakerConfig); }
static MA_INLINE HRESULT ma_IDirectSound_SetSpeakerConfig(ma_IDirectSound* pThis, DWORD dwSpeakerConfig)                { return pThis->lpVtbl->SetSpeakerConfig(pThis, dwSpeakerConfig); }
static MA_INLINE HRESULT ma_IDirectSound_Initialize(ma_IDirectSound* pThis, const GUID* pGuidDevice)                    { return pThis->lpVtbl->Initialize(pThis, pGuidDevice); }


/* IDirectSoundBuffer */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IDirectSoundBuffer* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IDirectSoundBuffer* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IDirectSoundBuffer* pThis);

    /* IDirectSoundBuffer */
    HRESULT (STDMETHODCALLTYPE * GetCaps)           (ma_IDirectSoundBuffer* pThis, MA_DSBCAPS* pDSBufferCaps);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPosition)(ma_IDirectSoundBuffer* pThis, DWORD* pCurrentPlayCursor, DWORD* pCurrentWriteCursor);
    HRESULT (STDMETHODCALLTYPE * GetFormat)         (ma_IDirectSoundBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten);
    HRESULT (STDMETHODCALLTYPE * GetVolume)         (ma_IDirectSoundBuffer* pThis, LONG* pVolume);
    HRESULT (STDMETHODCALLTYPE * GetPan)            (ma_IDirectSoundBuffer* pThis, LONG* pPan);
    HRESULT (STDMETHODCALLTYPE * GetFrequency)      (ma_IDirectSoundBuffer* pThis, DWORD* pFrequency);
    HRESULT (STDMETHODCALLTYPE * GetStatus)         (ma_IDirectSoundBuffer* pThis, DWORD* pStatus);
    HRESULT (STDMETHODCALLTYPE * Initialize)        (ma_IDirectSoundBuffer* pThis, ma_IDirectSound* pDirectSound, const MA_DSBUFFERDESC* pDSBufferDesc);
    HRESULT (STDMETHODCALLTYPE * Lock)              (ma_IDirectSoundBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * Play)              (ma_IDirectSoundBuffer* pThis, DWORD dwReserved1, DWORD dwPriority, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * SetCurrentPosition)(ma_IDirectSoundBuffer* pThis, DWORD dwNewPosition);
    HRESULT (STDMETHODCALLTYPE * SetFormat)         (ma_IDirectSoundBuffer* pThis, const WAVEFORMATEX* pFormat);
    HRESULT (STDMETHODCALLTYPE * SetVolume)         (ma_IDirectSoundBuffer* pThis, LONG volume);
    HRESULT (STDMETHODCALLTYPE * SetPan)            (ma_IDirectSoundBuffer* pThis, LONG pan);
    HRESULT (STDMETHODCALLTYPE * SetFrequency)      (ma_IDirectSoundBuffer* pThis, DWORD dwFrequency);
    HRESULT (STDMETHODCALLTYPE * Stop)              (ma_IDirectSoundBuffer* pThis);
    HRESULT (STDMETHODCALLTYPE * Unlock)            (ma_IDirectSoundBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2);
    HRESULT (STDMETHODCALLTYPE * Restore)           (ma_IDirectSoundBuffer* pThis);
} ma_IDirectSoundBufferVtbl;
struct ma_IDirectSoundBuffer
{
    ma_IDirectSoundBufferVtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IDirectSoundBuffer_QueryInterface(ma_IDirectSoundBuffer* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IDirectSoundBuffer_AddRef(ma_IDirectSoundBuffer* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IDirectSoundBuffer_Release(ma_IDirectSoundBuffer* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_GetCaps(ma_IDirectSoundBuffer* pThis, MA_DSBCAPS* pDSBufferCaps)                     { return pThis->lpVtbl->GetCaps(pThis, pDSBufferCaps); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_GetCurrentPosition(ma_IDirectSoundBuffer* pThis, DWORD* pCurrentPlayCursor, DWORD* pCurrentWriteCursor) { return pThis->lpVtbl->GetCurrentPosition(pThis, pCurrentPlayCursor, pCurrentWriteCursor); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_GetFormat(ma_IDirectSoundBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten) { return pThis->lpVtbl->GetFormat(pThis, pFormat, dwSizeAllocated, pSizeWritten); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_GetVolume(ma_IDirectSoundBuffer* pThis, LONG* pVolume)                               { return pThis->lpVtbl->GetVolume(pThis, pVolume); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_GetPan(ma_IDirectSoundBuffer* pThis, LONG* pPan)                                     { return pThis->lpVtbl->GetPan(pThis, pPan); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_GetFrequency(ma_IDirectSoundBuffer* pThis, DWORD* pFrequency)                        { return pThis->lpVtbl->GetFrequency(pThis, pFrequency); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_GetStatus(ma_IDirectSoundBuffer* pThis, DWORD* pStatus)                              { return pThis->lpVtbl->GetStatus(pThis, pStatus); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_Initialize(ma_IDirectSoundBuffer* pThis, ma_IDirectSound* pDirectSound, const MA_DSBUFFERDESC* pDSBufferDesc) { return pThis->lpVtbl->Initialize(pThis, pDirectSound, pDSBufferDesc); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_Lock(ma_IDirectSoundBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags) { return pThis->lpVtbl->Lock(pThis, dwOffset, dwBytes, ppAudioPtr1, pAudioBytes1, ppAudioPtr2, pAudioBytes2, dwFlags); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_Play(ma_IDirectSoundBuffer* pThis, DWORD dwReserved1, DWORD dwPriority, DWORD dwFlags) { return pThis->lpVtbl->Play(pThis, dwReserved1, dwPriority, dwFlags); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_SetCurrentPosition(ma_IDirectSoundBuffer* pThis, DWORD dwNewPosition)                { return pThis->lpVtbl->SetCurrentPosition(pThis, dwNewPosition); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_SetFormat(ma_IDirectSoundBuffer* pThis, const WAVEFORMATEX* pFormat)                 { return pThis->lpVtbl->SetFormat(pThis, pFormat); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_SetVolume(ma_IDirectSoundBuffer* pThis, LONG volume)                                 { return pThis->lpVtbl->SetVolume(pThis, volume); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_SetPan(ma_IDirectSoundBuffer* pThis, LONG pan)                                       { return pThis->lpVtbl->SetPan(pThis, pan); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_SetFrequency(ma_IDirectSoundBuffer* pThis, DWORD dwFrequency)                        { return pThis->lpVtbl->SetFrequency(pThis, dwFrequency); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_Stop(ma_IDirectSoundBuffer* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_Unlock(ma_IDirectSoundBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2) { return pThis->lpVtbl->Unlock(pThis, pAudioPtr1, dwAudioBytes1, pAudioPtr2, dwAudioBytes2); }
static MA_INLINE HRESULT ma_IDirectSoundBuffer_Restore(ma_IDirectSoundBuffer* pThis)                                                { return pThis->lpVtbl->Restore(pThis); }


/* IDirectSoundCapture */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IDirectSoundCapture* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IDirectSoundCapture* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IDirectSoundCapture* pThis);

    /* IDirectSoundCapture */
    HRESULT (STDMETHODCALLTYPE * CreateCaptureBuffer)(ma_IDirectSoundCapture* pThis, const MA_DSCBUFFERDESC* pDSCBufferDesc, ma_IDirectSoundCaptureBuffer** ppDSCBuffer, void* pUnkOuter);
    HRESULT (STDMETHODCALLTYPE * GetCaps)            (ma_IDirectSoundCapture* pThis, MA_DSCCAPS* pDSCCaps);
    HRESULT (STDMETHODCALLTYPE * Initialize)         (ma_IDirectSoundCapture* pThis, const GUID* pGuidDevice);
} ma_IDirectSoundCaptureVtbl;
struct ma_IDirectSoundCapture
{
    ma_IDirectSoundCaptureVtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IDirectSoundCapture_QueryInterface(ma_IDirectSoundCapture* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IDirectSoundCapture_AddRef(ma_IDirectSoundCapture* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IDirectSoundCapture_Release(ma_IDirectSoundCapture* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IDirectSoundCapture_CreateCaptureBuffer(ma_IDirectSoundCapture* pThis, const MA_DSCBUFFERDESC* pDSCBufferDesc, ma_IDirectSoundCaptureBuffer** ppDSCBuffer, void* pUnkOuter) { return pThis->lpVtbl->CreateCaptureBuffer(pThis, pDSCBufferDesc, ppDSCBuffer, pUnkOuter); }
static MA_INLINE HRESULT ma_IDirectSoundCapture_GetCaps            (ma_IDirectSoundCapture* pThis, MA_DSCCAPS* pDSCCaps)              { return pThis->lpVtbl->GetCaps(pThis, pDSCCaps); }
static MA_INLINE HRESULT ma_IDirectSoundCapture_Initialize         (ma_IDirectSoundCapture* pThis, const GUID* pGuidDevice)           { return pThis->lpVtbl->Initialize(pThis, pGuidDevice); }


/* IDirectSoundCaptureBuffer */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IDirectSoundCaptureBuffer* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IDirectSoundCaptureBuffer* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IDirectSoundCaptureBuffer* pThis);

    /* IDirectSoundCaptureBuffer */
    HRESULT (STDMETHODCALLTYPE * GetCaps)           (ma_IDirectSoundCaptureBuffer* pThis, MA_DSCBCAPS* pDSCBCaps);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPosition)(ma_IDirectSoundCaptureBuffer* pThis, DWORD* pCapturePosition, DWORD* pReadPosition);
    HRESULT (STDMETHODCALLTYPE * GetFormat)         (ma_IDirectSoundCaptureBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten);
    HRESULT (STDMETHODCALLTYPE * GetStatus)         (ma_IDirectSoundCaptureBuffer* pThis, DWORD* pStatus);
    HRESULT (STDMETHODCALLTYPE * Initialize)        (ma_IDirectSoundCaptureBuffer* pThis, ma_IDirectSoundCapture* pDirectSoundCapture, const MA_DSCBUFFERDESC* pDSCBufferDesc);
    HRESULT (STDMETHODCALLTYPE * Lock)              (ma_IDirectSoundCaptureBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * Start)             (ma_IDirectSoundCaptureBuffer* pThis, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * Stop)              (ma_IDirectSoundCaptureBuffer* pThis);
    HRESULT (STDMETHODCALLTYPE * Unlock)            (ma_IDirectSoundCaptureBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2);
} ma_IDirectSoundCaptureBufferVtbl;
struct ma_IDirectSoundCaptureBuffer
{
    ma_IDirectSoundCaptureBufferVtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IDirectSoundCaptureBuffer_QueryInterface(ma_IDirectSoundCaptureBuffer* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IDirectSoundCaptureBuffer_AddRef(ma_IDirectSoundCaptureBuffer* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IDirectSoundCaptureBuffer_Release(ma_IDirectSoundCaptureBuffer* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IDirectSoundCaptureBuffer_GetCaps(ma_IDirectSoundCaptureBuffer* pThis, MA_DSCBCAPS* pDSCBCaps)                        { return pThis->lpVtbl->GetCaps(pThis, pDSCBCaps); }
static MA_INLINE HRESULT ma_IDirectSoundCaptureBuffer_GetCurrentPosition(ma_IDirectSoundCaptureBuffer* pThis, DWORD* pCapturePosition, DWORD* pReadPosition) { return pThis->lpVtbl->GetCurrentPosition(pThis, pCapturePosition, pReadPosition); }
static MA_INLINE HRESULT ma_IDirectSoundCaptureBuffer_GetFormat(ma_IDirectSoundCaptureBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten) { return pThis->lpVtbl->GetFormat(pThis, pFormat, dwSizeAllocated, pSizeWritten); }
static MA_INLINE HRESULT ma_IDirectSoundCaptureBuffer_GetStatus(ma_IDirectSoundCaptureBuffer* pThis, DWORD* pStatus)                              { return pThis->lpVtbl->GetStatus(pThis, pStatus); }
static MA_INLINE HRESULT ma_IDirectSoundCaptureBuffer_Initialize(ma_IDirectSoundCaptureBuffer* pThis, ma_IDirectSoundCapture* pDirectSoundCapture, const MA_DSCBUFFERDESC* pDSCBufferDesc) { return pThis->lpVtbl->Initialize(pThis, pDirectSoundCapture, pDSCBufferDesc); }
static MA_INLINE HRESULT ma_IDirectSoundCaptureBuffer_Lock(ma_IDirectSoundCaptureBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags) { return pThis->lpVtbl->Lock(pThis, dwOffset, dwBytes, ppAudioPtr1, pAudioBytes1, ppAudioPtr2, pAudioBytes2, dwFlags); }
static MA_INLINE HRESULT ma_IDirectSoundCaptureBuffer_Start(ma_IDirectSoundCaptureBuffer* pThis, DWORD dwFlags)                                   { return pThis->lpVtbl->Start(pThis, dwFlags); }
static MA_INLINE HRESULT ma_IDirectSoundCaptureBuffer_Stop(ma_IDirectSoundCaptureBuffer* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
static MA_INLINE HRESULT ma_IDirectSoundCaptureBuffer_Unlock(ma_IDirectSoundCaptureBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2) { return pThis->lpVtbl->Unlock(pThis, pAudioPtr1, dwAudioBytes1, pAudioPtr2, dwAudioBytes2); }


/* IDirectSoundNotify */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IDirectSoundNotify* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IDirectSoundNotify* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IDirectSoundNotify* pThis);

    /* IDirectSoundNotify */
    HRESULT (STDMETHODCALLTYPE * SetNotificationPositions)(ma_IDirectSoundNotify* pThis, DWORD dwPositionNotifies, const MA_DSBPOSITIONNOTIFY* pPositionNotifies);
} ma_IDirectSoundNotifyVtbl;
struct ma_IDirectSoundNotify
{
    ma_IDirectSoundNotifyVtbl* lpVtbl;
};
static MA_INLINE HRESULT ma_IDirectSoundNotify_QueryInterface(ma_IDirectSoundNotify* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
static MA_INLINE ULONG   ma_IDirectSoundNotify_AddRef(ma_IDirectSoundNotify* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
static MA_INLINE ULONG   ma_IDirectSoundNotify_Release(ma_IDirectSoundNotify* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
static MA_INLINE HRESULT ma_IDirectSoundNotify_SetNotificationPositions(ma_IDirectSoundNotify* pThis, DWORD dwPositionNotifies, const MA_DSBPOSITIONNOTIFY* pPositionNotifies) { return pThis->lpVtbl->SetNotificationPositions(pThis, dwPositionNotifies, pPositionNotifies); }


typedef BOOL    (CALLBACK * ma_DSEnumCallbackAProc)             (LPGUID pDeviceGUID, LPCSTR pDeviceDescription, LPCSTR pModule, LPVOID pContext);
typedef HRESULT (WINAPI   * ma_DirectSoundCreateProc)           (const GUID* pcGuidDevice, ma_IDirectSound** ppDS8, LPUNKNOWN pUnkOuter);
typedef HRESULT (WINAPI   * ma_DirectSoundEnumerateAProc)       (ma_DSEnumCallbackAProc pDSEnumCallback, LPVOID pContext);
typedef HRESULT (WINAPI   * ma_DirectSoundCaptureCreateProc)    (const GUID* pcGuidDevice, ma_IDirectSoundCapture** ppDSC8, LPUNKNOWN pUnkOuter);
typedef HRESULT (WINAPI   * ma_DirectSoundCaptureEnumerateAProc)(ma_DSEnumCallbackAProc pDSEnumCallback, LPVOID pContext);

static ma_uint32 ma_get_best_sample_rate_within_range(ma_uint32 sampleRateMin, ma_uint32 sampleRateMax)
{
    /* Normalize the range in case we were given something stupid. */
    if (sampleRateMin < MA_MIN_SAMPLE_RATE) {
        sampleRateMin = MA_MIN_SAMPLE_RATE;
    }
    if (sampleRateMax > MA_MAX_SAMPLE_RATE) {
        sampleRateMax = MA_MAX_SAMPLE_RATE;
    }
    if (sampleRateMin > sampleRateMax) {
        sampleRateMin = sampleRateMax;
    }

    if (sampleRateMin == sampleRateMax) {
        return sampleRateMax;
    } else {
        size_t iStandardRate;
        for (iStandardRate = 0; iStandardRate < ma_countof(g_maStandardSampleRatePriorities); ++iStandardRate) {
            ma_uint32 standardRate = g_maStandardSampleRatePriorities[iStandardRate];
            if (standardRate >= sampleRateMin && standardRate <= sampleRateMax) {
                return standardRate;
            }
        }
    }

    /* Should never get here. */
    MA_ASSERT(MA_FALSE);
    return 0;
}

/*
Retrieves the channel count and channel map for the given speaker configuration. If the speaker configuration is unknown,
the channel count and channel map will be left unmodified.
*/
static void ma_get_channels_from_speaker_config__dsound(DWORD speakerConfig, WORD* pChannelsOut, DWORD* pChannelMapOut)
{
    WORD  channels;
    DWORD channelMap;

    channels = 0;
    if (pChannelsOut != NULL) {
        channels = *pChannelsOut;
    }

    channelMap = 0;
    if (pChannelMapOut != NULL) {
        channelMap = *pChannelMapOut;
    }

    /*
    The speaker configuration is a combination of speaker config and speaker geometry. The lower 8 bits is what we care about. The upper
    16 bits is for the geometry.
    */
    switch ((BYTE)(speakerConfig)) {
        case 1 /*DSSPEAKER_HEADPHONE*/:                          channels = 2; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT; break;
        case 2 /*DSSPEAKER_MONO*/:                               channels = 1; channelMap = SPEAKER_FRONT_CENTER; break;
        case 3 /*DSSPEAKER_QUAD*/:                               channels = 4; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT; break;
        case 4 /*DSSPEAKER_STEREO*/:                             channels = 2; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT; break;
        case 5 /*DSSPEAKER_SURROUND*/:                           channels = 4; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_BACK_CENTER; break;
        case 6 /*DSSPEAKER_5POINT1_BACK*/ /*DSSPEAKER_5POINT1*/: channels = 6; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT; break;
        case 7 /*DSSPEAKER_7POINT1_WIDE*/ /*DSSPEAKER_7POINT1*/: channels = 8; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT | SPEAKER_FRONT_LEFT_OF_CENTER | SPEAKER_FRONT_RIGHT_OF_CENTER; break;
        case 8 /*DSSPEAKER_7POINT1_SURROUND*/:                   channels = 8; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT | SPEAKER_SIDE_LEFT | SPEAKER_SIDE_RIGHT; break;
        case 9 /*DSSPEAKER_5POINT1_SURROUND*/:                   channels = 6; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_SIDE_LEFT | SPEAKER_SIDE_RIGHT; break;
        default: break;
    }

    if (pChannelsOut != NULL) {
        *pChannelsOut = channels;
    }

    if (pChannelMapOut != NULL) {
        *pChannelMapOut = channelMap;
    }
}


static ma_result ma_context_create_IDirectSound__dsound(ma_context* pContext, ma_share_mode shareMode, const ma_device_id* pDeviceID, ma_IDirectSound** ppDirectSound)
{
    ma_IDirectSound* pDirectSound;
    HWND hWnd;
    HRESULT hr;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(ppDirectSound != NULL);

    *ppDirectSound = NULL;
    pDirectSound = NULL;

    if (FAILED(((ma_DirectSoundCreateProc)pContext->dsound.DirectSoundCreate)((pDeviceID == NULL) ? NULL : (const GUID*)pDeviceID->dsound, &pDirectSound, NULL))) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[DirectSound] DirectSoundCreate() failed for playback device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    /* The cooperative level must be set before doing anything else. */
    hWnd = ((MA_PFN_GetForegroundWindow)pContext->win32.GetForegroundWindow)();
    if (hWnd == NULL) {
        hWnd = ((MA_PFN_GetDesktopWindow)pContext->win32.GetDesktopWindow)();
    }

    hr = ma_IDirectSound_SetCooperativeLevel(pDirectSound, hWnd, (shareMode == ma_share_mode_exclusive) ? MA_DSSCL_EXCLUSIVE : MA_DSSCL_PRIORITY);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_SetCooperateiveLevel() failed for playback device.", ma_result_from_HRESULT(hr));
    }

    *ppDirectSound = pDirectSound;
    return MA_SUCCESS;
}

static ma_result ma_context_create_IDirectSoundCapture__dsound(ma_context* pContext, ma_share_mode shareMode, const ma_device_id* pDeviceID, ma_IDirectSoundCapture** ppDirectSoundCapture)
{
    ma_IDirectSoundCapture* pDirectSoundCapture;
    HRESULT hr;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(ppDirectSoundCapture != NULL);

    /* DirectSound does not support exclusive mode for capture. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    *ppDirectSoundCapture = NULL;
    pDirectSoundCapture = NULL;

    hr = ((ma_DirectSoundCaptureCreateProc)pContext->dsound.DirectSoundCaptureCreate)((pDeviceID == NULL) ? NULL : (const GUID*)pDeviceID->dsound, &pDirectSoundCapture, NULL);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[DirectSound] DirectSoundCaptureCreate() failed for capture device.", ma_result_from_HRESULT(hr));
    }

    *ppDirectSoundCapture = pDirectSoundCapture;
    return MA_SUCCESS;
}

static ma_result ma_context_get_format_info_for_IDirectSoundCapture__dsound(ma_context* pContext, ma_IDirectSoundCapture* pDirectSoundCapture, WORD* pChannels, WORD* pBitsPerSample, DWORD* pSampleRate)
{
    HRESULT hr;
    MA_DSCCAPS caps;
    WORD bitsPerSample;
    DWORD sampleRate;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pDirectSoundCapture != NULL);

    if (pChannels) {
        *pChannels = 0;
    }
    if (pBitsPerSample) {
        *pBitsPerSample = 0;
    }
    if (pSampleRate) {
        *pSampleRate = 0;
    }

    MA_ZERO_OBJECT(&caps);
    caps.dwSize = sizeof(caps);
    hr = ma_IDirectSoundCapture_GetCaps(pDirectSoundCapture, &caps);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCapture_GetCaps() failed for capture device.", ma_result_from_HRESULT(hr));
    }

    if (pChannels) {
        *pChannels = (WORD)caps.dwChannels;
    }

    /* The device can support multiple formats. We just go through the different formats in order of priority and pick the first one. This the same type of system as the WinMM backend. */
    bitsPerSample = 16;
    sampleRate = 48000;

    if (caps.dwChannels == 1) {
        if ((caps.dwFormats & WAVE_FORMAT_48M16) != 0) {
            sampleRate = 48000;
        } else if ((caps.dwFormats & WAVE_FORMAT_44M16) != 0) {
            sampleRate = 44100;
        } else if ((caps.dwFormats & WAVE_FORMAT_2M16) != 0) {
            sampleRate = 22050;
        } else if ((caps.dwFormats & WAVE_FORMAT_1M16) != 0) {
            sampleRate = 11025;
        } else if ((caps.dwFormats & WAVE_FORMAT_96M16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((caps.dwFormats & WAVE_FORMAT_48M08) != 0) {
                sampleRate = 48000;
            } else if ((caps.dwFormats & WAVE_FORMAT_44M08) != 0) {
                sampleRate = 44100;
            } else if ((caps.dwFormats & WAVE_FORMAT_2M08) != 0) {
                sampleRate = 22050;
            } else if ((caps.dwFormats & WAVE_FORMAT_1M08) != 0) {
                sampleRate = 11025;
            } else if ((caps.dwFormats & WAVE_FORMAT_96M08) != 0) {
                sampleRate = 96000;
            } else {
                bitsPerSample = 16;  /* Didn't find it. Just fall back to 16-bit. */
            }
        }
    } else if (caps.dwChannels == 2) {
        if ((caps.dwFormats & WAVE_FORMAT_48S16) != 0) {
            sampleRate = 48000;
        } else if ((caps.dwFormats & WAVE_FORMAT_44S16) != 0) {
            sampleRate = 44100;
        } else if ((caps.dwFormats & WAVE_FORMAT_2S16) != 0) {
            sampleRate = 22050;
        } else if ((caps.dwFormats & WAVE_FORMAT_1S16) != 0) {
            sampleRate = 11025;
        } else if ((caps.dwFormats & WAVE_FORMAT_96S16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((caps.dwFormats & WAVE_FORMAT_48S08) != 0) {
                sampleRate = 48000;
            } else if ((caps.dwFormats & WAVE_FORMAT_44S08) != 0) {
                sampleRate = 44100;
            } else if ((caps.dwFormats & WAVE_FORMAT_2S08) != 0) {
                sampleRate = 22050;
            } else if ((caps.dwFormats & WAVE_FORMAT_1S08) != 0) {
                sampleRate = 11025;
            } else if ((caps.dwFormats & WAVE_FORMAT_96S08) != 0) {
                sampleRate = 96000;
            } else {
                bitsPerSample = 16;  /* Didn't find it. Just fall back to 16-bit. */
            }
        }
    }

    if (pBitsPerSample) {
        *pBitsPerSample = bitsPerSample;
    }
    if (pSampleRate) {
        *pSampleRate = sampleRate;
    }

    return MA_SUCCESS;
}

static ma_bool32 ma_context_is_device_id_equal__dsound(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pID0 != NULL);
    MA_ASSERT(pID1 != NULL);
    (void)pContext;

    return memcmp(pID0->dsound, pID1->dsound, sizeof(pID0->dsound)) == 0;
}


typedef struct
{
    ma_context* pContext;
    ma_device_type deviceType;
    ma_enum_devices_callback_proc callback;
    void* pUserData;
    ma_bool32 terminated;
} ma_context_enumerate_devices_callback_data__dsound;

static BOOL CALLBACK ma_context_enumerate_devices_callback__dsound(LPGUID lpGuid, LPCSTR lpcstrDescription, LPCSTR lpcstrModule, LPVOID lpContext)
{
    ma_context_enumerate_devices_callback_data__dsound* pData = (ma_context_enumerate_devices_callback_data__dsound*)lpContext;
    ma_device_info deviceInfo;    

    MA_ZERO_OBJECT(&deviceInfo);

    /* ID. */
    if (lpGuid != NULL) {
        MA_COPY_MEMORY(deviceInfo.id.dsound, lpGuid, 16);
    } else {
        MA_ZERO_MEMORY(deviceInfo.id.dsound, 16);
    }

    /* Name / Description */
    ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), lpcstrDescription, (size_t)-1);


    /* Call the callback function, but make sure we stop enumerating if the callee requested so. */
    MA_ASSERT(pData != NULL);
    pData->terminated = !pData->callback(pData->pContext, pData->deviceType, &deviceInfo, pData->pUserData);
    if (pData->terminated) {
        return FALSE;   /* Stop enumeration. */
    } else {
        return TRUE;    /* Continue enumeration. */
    }

    (void)lpcstrModule;
}

static ma_result ma_context_enumerate_devices__dsound(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_context_enumerate_devices_callback_data__dsound data;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(callback != NULL);

    data.pContext = pContext;
    data.callback = callback;
    data.pUserData = pUserData;
    data.terminated = MA_FALSE;

    /* Playback. */
    if (!data.terminated) {
        data.deviceType = ma_device_type_playback;
        ((ma_DirectSoundEnumerateAProc)pContext->dsound.DirectSoundEnumerateA)(ma_context_enumerate_devices_callback__dsound, &data);
    }

    /* Capture. */
    if (!data.terminated) {
        data.deviceType = ma_device_type_capture;
        ((ma_DirectSoundCaptureEnumerateAProc)pContext->dsound.DirectSoundCaptureEnumerateA)(ma_context_enumerate_devices_callback__dsound, &data);
    }

    return MA_SUCCESS;
}


typedef struct
{
    const ma_device_id* pDeviceID;
    ma_device_info* pDeviceInfo;
    ma_bool32 found;
} ma_context_get_device_info_callback_data__dsound;

static BOOL CALLBACK ma_context_get_device_info_callback__dsound(LPGUID lpGuid, LPCSTR lpcstrDescription, LPCSTR lpcstrModule, LPVOID lpContext)
{
    ma_context_get_device_info_callback_data__dsound* pData = (ma_context_get_device_info_callback_data__dsound*)lpContext;
    MA_ASSERT(pData != NULL);

    if ((pData->pDeviceID == NULL || ma_is_guid_null(pData->pDeviceID->dsound)) && (lpGuid == NULL || ma_is_guid_null(lpGuid))) {
        /* Default device. */
        ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), lpcstrDescription, (size_t)-1);
        pData->found = MA_TRUE;
        return FALSE;   /* Stop enumeration. */
    } else {
        /* Not the default device. */
        if (lpGuid != NULL && pData->pDeviceID != NULL) {
            if (memcmp(pData->pDeviceID->dsound, lpGuid, sizeof(pData->pDeviceID->dsound)) == 0) {
                ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), lpcstrDescription, (size_t)-1);
                pData->found = MA_TRUE;
                return FALSE;   /* Stop enumeration. */
            }
        }
    }

    (void)lpcstrModule;
    return TRUE;
}

static ma_result ma_context_get_device_info__dsound(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_result result;
    HRESULT hr;

    /* Exclusive mode and capture not supported with DirectSound. */
    if (deviceType == ma_device_type_capture && shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    if (pDeviceID != NULL) {
        ma_context_get_device_info_callback_data__dsound data;

        /* ID. */
        MA_COPY_MEMORY(pDeviceInfo->id.dsound, pDeviceID->dsound, 16);

        /* Name / Description. This is retrieved by enumerating over each device until we find that one that matches the input ID. */
        data.pDeviceID = pDeviceID;
        data.pDeviceInfo = pDeviceInfo;
        data.found = MA_FALSE;
        if (deviceType == ma_device_type_playback) {
            ((ma_DirectSoundEnumerateAProc)pContext->dsound.DirectSoundEnumerateA)(ma_context_get_device_info_callback__dsound, &data);
        } else {
            ((ma_DirectSoundCaptureEnumerateAProc)pContext->dsound.DirectSoundCaptureEnumerateA)(ma_context_get_device_info_callback__dsound, &data);
        }

        if (!data.found) {
            return MA_NO_DEVICE;
        }
    } else {
        /* I don't think there's a way to get the name of the default device with DirectSound. In this case we just need to use defaults. */

        /* ID */
        MA_ZERO_MEMORY(pDeviceInfo->id.dsound, 16);

        /* Name / Description */
        if (deviceType == ma_device_type_playback) {
            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        } else {
            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        }
    }

    /* Retrieving detailed information is slightly different depending on the device type. */
    if (deviceType == ma_device_type_playback) {
        /* Playback. */
        ma_IDirectSound* pDirectSound;
        MA_DSCAPS caps;
        ma_uint32 iFormat;

        result = ma_context_create_IDirectSound__dsound(pContext, shareMode, pDeviceID, &pDirectSound);
        if (result != MA_SUCCESS) {
            return result;
        }

        MA_ZERO_OBJECT(&caps);
        caps.dwSize = sizeof(caps);
        hr = ma_IDirectSound_GetCaps(pDirectSound, &caps);
        if (FAILED(hr)) {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_GetCaps() failed for playback device.", ma_result_from_HRESULT(hr));
        }

        if ((caps.dwFlags & MA_DSCAPS_PRIMARYSTEREO) != 0) {
            /* It supports at least stereo, but could support more. */
            WORD channels = 2;

            /* Look at the speaker configuration to get a better idea on the channel count. */
            DWORD speakerConfig;
            hr = ma_IDirectSound_GetSpeakerConfig(pDirectSound, &speakerConfig);
            if (SUCCEEDED(hr)) {
                ma_get_channels_from_speaker_config__dsound(speakerConfig, &channels, NULL);
            }

            pDeviceInfo->minChannels = channels;
            pDeviceInfo->maxChannels = channels;
        } else {
            /* It does not support stereo, which means we are stuck with mono. */
            pDeviceInfo->minChannels = 1;
            pDeviceInfo->maxChannels = 1;
        }

        /* Sample rate. */
        if ((caps.dwFlags & MA_DSCAPS_CONTINUOUSRATE) != 0) {
            pDeviceInfo->minSampleRate = caps.dwMinSecondarySampleRate;
            pDeviceInfo->maxSampleRate = caps.dwMaxSecondarySampleRate;

            /*
            On my machine the min and max sample rates can return 100 and 200000 respectively. I'd rather these be within
            the range of our standard sample rates so I'm clamping.
            */
            if (caps.dwMinSecondarySampleRate < MA_MIN_SAMPLE_RATE && caps.dwMaxSecondarySampleRate >= MA_MIN_SAMPLE_RATE) {
                pDeviceInfo->minSampleRate = MA_MIN_SAMPLE_RATE;
            }
            if (caps.dwMaxSecondarySampleRate > MA_MAX_SAMPLE_RATE && caps.dwMinSecondarySampleRate <= MA_MAX_SAMPLE_RATE) {
                pDeviceInfo->maxSampleRate = MA_MAX_SAMPLE_RATE;
            }
        } else {
            /* Only supports a single sample rate. Set both min an max to the same thing. Do not clamp within the standard rates. */
            pDeviceInfo->minSampleRate = caps.dwMaxSecondarySampleRate;
            pDeviceInfo->maxSampleRate = caps.dwMaxSecondarySampleRate;
        }

        /* DirectSound can support all formats. */
        pDeviceInfo->formatCount = ma_format_count - 1;    /* Minus one because we don't want to include ma_format_unknown. */
        for (iFormat = 0; iFormat < pDeviceInfo->formatCount; ++iFormat) {
            pDeviceInfo->formats[iFormat] = (ma_format)(iFormat + 1);  /* +1 to skip over ma_format_unknown. */
        }

        ma_IDirectSound_Release(pDirectSound);
    } else {
        /*
        Capture. This is a little different to playback due to the say the supported formats are reported. Technically capture
        devices can support a number of different formats, but for simplicity and consistency with ma_device_init() I'm just
        reporting the best format.
        */
        ma_IDirectSoundCapture* pDirectSoundCapture;
        WORD channels;
        WORD bitsPerSample;
        DWORD sampleRate;

        result = ma_context_create_IDirectSoundCapture__dsound(pContext, shareMode, pDeviceID, &pDirectSoundCapture);
        if (result != MA_SUCCESS) {
            return result;
        }

        result = ma_context_get_format_info_for_IDirectSoundCapture__dsound(pContext, pDirectSoundCapture, &channels, &bitsPerSample, &sampleRate);
        if (result != MA_SUCCESS) {
            ma_IDirectSoundCapture_Release(pDirectSoundCapture);
            return result;
        }

        pDeviceInfo->minChannels = channels;
        pDeviceInfo->maxChannels = channels;
        pDeviceInfo->minSampleRate = sampleRate;
        pDeviceInfo->maxSampleRate = sampleRate;
        pDeviceInfo->formatCount = 1;
        if (bitsPerSample == 8) {
            pDeviceInfo->formats[0] = ma_format_u8;
        } else if (bitsPerSample == 16) {
            pDeviceInfo->formats[0] = ma_format_s16;
        } else if (bitsPerSample == 24) {
            pDeviceInfo->formats[0] = ma_format_s24;
        } else if (bitsPerSample == 32) {
            pDeviceInfo->formats[0] = ma_format_s32;
        } else {
            ma_IDirectSoundCapture_Release(pDirectSoundCapture);
            return MA_FORMAT_NOT_SUPPORTED;
        }

        ma_IDirectSoundCapture_Release(pDirectSoundCapture);
    }

    return MA_SUCCESS;
}



static void ma_device_uninit__dsound(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    if (pDevice->dsound.pCaptureBuffer != NULL) {
        ma_IDirectSoundCaptureBuffer_Release((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer);
    }
    if (pDevice->dsound.pCapture != NULL) {
        ma_IDirectSoundCapture_Release((ma_IDirectSoundCapture*)pDevice->dsound.pCapture);
    }

    if (pDevice->dsound.pPlaybackBuffer != NULL) {
        ma_IDirectSoundBuffer_Release((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer);
    }
    if (pDevice->dsound.pPlaybackPrimaryBuffer != NULL) {
        ma_IDirectSoundBuffer_Release((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackPrimaryBuffer);
    }
    if (pDevice->dsound.pPlayback != NULL) {
        ma_IDirectSound_Release((ma_IDirectSound*)pDevice->dsound.pPlayback);
    }
}

static ma_result ma_config_to_WAVEFORMATEXTENSIBLE(ma_format format, ma_uint32 channels, ma_uint32 sampleRate, const ma_channel* pChannelMap, WAVEFORMATEXTENSIBLE* pWF)
{
    GUID subformat;

    switch (format)
    {
        case ma_format_u8:
        case ma_format_s16:
        case ma_format_s24:
        /*case ma_format_s24_32:*/
        case ma_format_s32:
        {
            subformat = MA_GUID_KSDATAFORMAT_SUBTYPE_PCM;
        } break;

        case ma_format_f32:
        {
            subformat = MA_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT;
        } break;

        default:
        return MA_FORMAT_NOT_SUPPORTED;
    }

    MA_ZERO_OBJECT(pWF);
    pWF->Format.cbSize               = sizeof(*pWF);
    pWF->Format.wFormatTag           = WAVE_FORMAT_EXTENSIBLE;
    pWF->Format.nChannels            = (WORD)channels;
    pWF->Format.nSamplesPerSec       = (DWORD)sampleRate;
    pWF->Format.wBitsPerSample       = (WORD)(ma_get_bytes_per_sample(format)*8);
    pWF->Format.nBlockAlign          = (WORD)(pWF->Format.nChannels * pWF->Format.wBitsPerSample / 8);
    pWF->Format.nAvgBytesPerSec      = pWF->Format.nBlockAlign * pWF->Format.nSamplesPerSec;
    pWF->Samples.wValidBitsPerSample = pWF->Format.wBitsPerSample;
    pWF->dwChannelMask               = ma_channel_map_to_channel_mask__win32(pChannelMap, channels);
    pWF->SubFormat                   = subformat;

    return MA_SUCCESS;
}

static ma_result ma_device_init__dsound(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result;
    HRESULT hr;
    ma_uint32 periodSizeInMilliseconds;

    MA_ASSERT(pDevice != NULL);
    MA_ZERO_OBJECT(&pDevice->dsound);

    if (pConfig->deviceType == ma_device_type_loopback) {
        return MA_DEVICE_TYPE_NOT_SUPPORTED;
    }

    periodSizeInMilliseconds = pConfig->periodSizeInMilliseconds;
    if (periodSizeInMilliseconds == 0) {
        periodSizeInMilliseconds = ma_calculate_buffer_size_in_milliseconds_from_frames(pConfig->periodSizeInFrames, pConfig->sampleRate);
    }
    
    /* DirectSound should use a latency of about 20ms per period for low latency mode. */
    if (pDevice->usingDefaultBufferSize) {
        if (pConfig->performanceProfile == ma_performance_profile_low_latency) {
            periodSizeInMilliseconds =  20;
        } else {
            periodSizeInMilliseconds = 200;
        }
    }

    /* DirectSound breaks down with tiny buffer sizes (bad glitching and silent output). I am therefore restricting the size of the buffer to a minimum of 20 milliseconds. */
    if (periodSizeInMilliseconds < 20) {
        periodSizeInMilliseconds = 20;
    }

    /*
    Unfortunately DirectSound uses different APIs and data structures for playback and catpure devices. We need to initialize
    the capture device first because we'll want to match it's buffer size and period count on the playback side if we're using
    full-duplex mode.
    */
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        WAVEFORMATEXTENSIBLE wf;
        MA_DSCBUFFERDESC descDS;
        ma_uint32 periodSizeInFrames;
        char rawdata[1024]; /* <-- Ugly hack to avoid a malloc() due to a crappy DirectSound API. */
        WAVEFORMATEXTENSIBLE* pActualFormat;

        result = ma_config_to_WAVEFORMATEXTENSIBLE(pConfig->capture.format, pConfig->capture.channels, pConfig->sampleRate, pConfig->capture.channelMap, &wf);
        if (result != MA_SUCCESS) {
            return result;
        }

        result = ma_context_create_IDirectSoundCapture__dsound(pContext, pConfig->capture.shareMode, pConfig->capture.pDeviceID, (ma_IDirectSoundCapture**)&pDevice->dsound.pCapture);
        if (result != MA_SUCCESS) {
            ma_device_uninit__dsound(pDevice);
            return result;
        }

        result = ma_context_get_format_info_for_IDirectSoundCapture__dsound(pContext, (ma_IDirectSoundCapture*)pDevice->dsound.pCapture, &wf.Format.nChannels, &wf.Format.wBitsPerSample, &wf.Format.nSamplesPerSec);
        if (result != MA_SUCCESS) {
            ma_device_uninit__dsound(pDevice);
            return result;
        }

        wf.Format.nBlockAlign          = (WORD)(wf.Format.nChannels * wf.Format.wBitsPerSample / 8);
        wf.Format.nAvgBytesPerSec      = wf.Format.nBlockAlign * wf.Format.nSamplesPerSec;
        wf.Samples.wValidBitsPerSample = wf.Format.wBitsPerSample;
        wf.SubFormat                   = MA_GUID_KSDATAFORMAT_SUBTYPE_PCM;

        /* The size of the buffer must be a clean multiple of the period count. */
        periodSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(periodSizeInMilliseconds, wf.Format.nSamplesPerSec);

        MA_ZERO_OBJECT(&descDS);
        descDS.dwSize = sizeof(descDS);
        descDS.dwFlags = 0;
        descDS.dwBufferBytes = periodSizeInFrames * pConfig->periods * ma_get_bytes_per_frame(pDevice->capture.internalFormat, wf.Format.nChannels);
        descDS.lpwfxFormat = (WAVEFORMATEX*)&wf;
        hr = ma_IDirectSoundCapture_CreateCaptureBuffer((ma_IDirectSoundCapture*)pDevice->dsound.pCapture, &descDS, (ma_IDirectSoundCaptureBuffer**)&pDevice->dsound.pCaptureBuffer, NULL);
        if (FAILED(hr)) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCapture_CreateCaptureBuffer() failed for capture device.", ma_result_from_HRESULT(hr));
        }

        /* Get the _actual_ properties of the buffer. */
        pActualFormat = (WAVEFORMATEXTENSIBLE*)rawdata;
        hr = ma_IDirectSoundCaptureBuffer_GetFormat((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, (WAVEFORMATEX*)pActualFormat, sizeof(rawdata), NULL);
        if (FAILED(hr)) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to retrieve the actual format of the capture device's buffer.", ma_result_from_HRESULT(hr));
        }

        pDevice->capture.internalFormat = ma_format_from_WAVEFORMATEX((WAVEFORMATEX*)pActualFormat);
        pDevice->capture.internalChannels = pActualFormat->Format.nChannels;
        pDevice->capture.internalSampleRate = pActualFormat->Format.nSamplesPerSec;

        /* Get the internal channel map based on the channel mask. */
        if (pActualFormat->Format.wFormatTag == WAVE_FORMAT_EXTENSIBLE) {
            ma_channel_mask_to_channel_map__win32(pActualFormat->dwChannelMask, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);
        } else {
            ma_channel_mask_to_channel_map__win32(wf.dwChannelMask, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);
        }

        /*
        After getting the actual format the size of the buffer in frames may have actually changed. However, we want this to be as close to what the
        user has asked for as possible, so let's go ahead and release the old capture buffer and create a new one in this case.
        */
        if (periodSizeInFrames != (descDS.dwBufferBytes / ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels) / pConfig->periods)) {
            descDS.dwBufferBytes = periodSizeInFrames * ma_get_bytes_per_frame(pDevice->capture.internalFormat, wf.Format.nChannels) * pConfig->periods;
            ma_IDirectSoundCaptureBuffer_Release((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer);

            hr = ma_IDirectSoundCapture_CreateCaptureBuffer((ma_IDirectSoundCapture*)pDevice->dsound.pCapture, &descDS, (ma_IDirectSoundCaptureBuffer**)&pDevice->dsound.pCaptureBuffer, NULL);
            if (FAILED(hr)) {
                ma_device_uninit__dsound(pDevice);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Second attempt at IDirectSoundCapture_CreateCaptureBuffer() failed for capture device.", ma_result_from_HRESULT(hr));
            }
        }

        /* DirectSound should give us a buffer exactly the size we asked for. */
        pDevice->capture.internalPeriodSizeInFrames = periodSizeInFrames;
        pDevice->capture.internalPeriods            = pConfig->periods;
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        WAVEFORMATEXTENSIBLE wf;
        MA_DSBUFFERDESC descDSPrimary;
        MA_DSCAPS caps;
        char rawdata[1024]; /* <-- Ugly hack to avoid a malloc() due to a crappy DirectSound API. */
        WAVEFORMATEXTENSIBLE* pActualFormat;
        ma_uint32 periodSizeInFrames;
        MA_DSBUFFERDESC descDS;

        result = ma_config_to_WAVEFORMATEXTENSIBLE(pConfig->playback.format, pConfig->playback.channels, pConfig->sampleRate, pConfig->playback.channelMap, &wf);
        if (result != MA_SUCCESS) {
            return result;
        }

        result = ma_context_create_IDirectSound__dsound(pContext, pConfig->playback.shareMode, pConfig->playback.pDeviceID, (ma_IDirectSound**)&pDevice->dsound.pPlayback);
        if (result != MA_SUCCESS) {
            ma_device_uninit__dsound(pDevice);
            return result;
        }

        MA_ZERO_OBJECT(&descDSPrimary);
        descDSPrimary.dwSize  = sizeof(MA_DSBUFFERDESC);
        descDSPrimary.dwFlags = MA_DSBCAPS_PRIMARYBUFFER | MA_DSBCAPS_CTRLVOLUME;
        hr = ma_IDirectSound_CreateSoundBuffer((ma_IDirectSound*)pDevice->dsound.pPlayback, &descDSPrimary, (ma_IDirectSoundBuffer**)&pDevice->dsound.pPlaybackPrimaryBuffer, NULL);
        if (FAILED(hr)) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_CreateSoundBuffer() failed for playback device's primary buffer.", ma_result_from_HRESULT(hr));
        }


        /* We may want to make some adjustments to the format if we are using defaults. */
        MA_ZERO_OBJECT(&caps);
        caps.dwSize = sizeof(caps);
        hr = ma_IDirectSound_GetCaps((ma_IDirectSound*)pDevice->dsound.pPlayback, &caps);
        if (FAILED(hr)) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_GetCaps() failed for playback device.", ma_result_from_HRESULT(hr));
        }

        if (pDevice->playback.usingDefaultChannels) {
            if ((caps.dwFlags & MA_DSCAPS_PRIMARYSTEREO) != 0) {
                DWORD speakerConfig;

                /* It supports at least stereo, but could support more. */
                wf.Format.nChannels = 2;

                /* Look at the speaker configuration to get a better idea on the channel count. */
                if (SUCCEEDED(ma_IDirectSound_GetSpeakerConfig((ma_IDirectSound*)pDevice->dsound.pPlayback, &speakerConfig))) {
                    ma_get_channels_from_speaker_config__dsound(speakerConfig, &wf.Format.nChannels, &wf.dwChannelMask);
                }
            } else {
                /* It does not support stereo, which means we are stuck with mono. */
                wf.Format.nChannels = 1;
            }
        }

        if (pDevice->usingDefaultSampleRate) {
            /* We base the sample rate on the values returned by GetCaps(). */
            if ((caps.dwFlags & MA_DSCAPS_CONTINUOUSRATE) != 0) {
                wf.Format.nSamplesPerSec = ma_get_best_sample_rate_within_range(caps.dwMinSecondarySampleRate, caps.dwMaxSecondarySampleRate);
            } else {
                wf.Format.nSamplesPerSec = caps.dwMaxSecondarySampleRate;
            }
        }

        wf.Format.nBlockAlign     = (WORD)(wf.Format.nChannels * wf.Format.wBitsPerSample / 8);
        wf.Format.nAvgBytesPerSec = wf.Format.nBlockAlign * wf.Format.nSamplesPerSec;

        /*
        From MSDN:
        
        The method succeeds even if the hardware does not support the requested format; DirectSound sets the buffer to the closest
        supported format. To determine whether this has happened, an application can call the GetFormat method for the primary buffer
        and compare the result with the format that was requested with the SetFormat method.
        */
        hr = ma_IDirectSoundBuffer_SetFormat((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackPrimaryBuffer, (WAVEFORMATEX*)&wf);
        if (FAILED(hr)) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to set format of playback device's primary buffer.", ma_result_from_HRESULT(hr));
        }

        /* Get the _actual_ properties of the buffer. */
        pActualFormat = (WAVEFORMATEXTENSIBLE*)rawdata;
        hr = ma_IDirectSoundBuffer_GetFormat((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackPrimaryBuffer, (WAVEFORMATEX*)pActualFormat, sizeof(rawdata), NULL);
        if (FAILED(hr)) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to retrieve the actual format of the playback device's primary buffer.", ma_result_from_HRESULT(hr));
        }

        pDevice->playback.internalFormat = ma_format_from_WAVEFORMATEX((WAVEFORMATEX*)pActualFormat);
        pDevice->playback.internalChannels = pActualFormat->Format.nChannels;
        pDevice->playback.internalSampleRate = pActualFormat->Format.nSamplesPerSec;

        /* Get the internal channel map based on the channel mask. */
        if (pActualFormat->Format.wFormatTag == WAVE_FORMAT_EXTENSIBLE) {
            ma_channel_mask_to_channel_map__win32(pActualFormat->dwChannelMask, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);
        } else {
            ma_channel_mask_to_channel_map__win32(wf.dwChannelMask, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);
        }

        /* The size of the buffer must be a clean multiple of the period count. */
        periodSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(periodSizeInMilliseconds, pDevice->playback.internalSampleRate);

        /*
        Meaning of dwFlags (from MSDN):
        
        DSBCAPS_CTRLPOSITIONNOTIFY
          The buffer has position notification capability.
        
        DSBCAPS_GLOBALFOCUS
          With this flag set, an application using DirectSound can continue to play its buffers if the user switches focus to
          another application, even if the new application uses DirectSound.
        
        DSBCAPS_GETCURRENTPOSITION2
          In the first version of DirectSound, the play cursor was significantly ahead of the actual playing sound on emulated
          sound cards; it was directly behind the write cursor. Now, if the DSBCAPS_GETCURRENTPOSITION2 flag is specified, the
          application can get a more accurate play cursor.
        */
        MA_ZERO_OBJECT(&descDS);
        descDS.dwSize = sizeof(descDS);
        descDS.dwFlags = MA_DSBCAPS_CTRLPOSITIONNOTIFY | MA_DSBCAPS_GLOBALFOCUS | MA_DSBCAPS_GETCURRENTPOSITION2;
        descDS.dwBufferBytes = periodSizeInFrames * pConfig->periods * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
        descDS.lpwfxFormat = (WAVEFORMATEX*)&wf;
        hr = ma_IDirectSound_CreateSoundBuffer((ma_IDirectSound*)pDevice->dsound.pPlayback, &descDS, (ma_IDirectSoundBuffer**)&pDevice->dsound.pPlaybackBuffer, NULL);
        if (FAILED(hr)) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_CreateSoundBuffer() failed for playback device's secondary buffer.", ma_result_from_HRESULT(hr));
        }

        /* DirectSound should give us a buffer exactly the size we asked for. */
        pDevice->playback.internalPeriodSizeInFrames = periodSizeInFrames;
        pDevice->playback.internalPeriods            = pConfig->periods;
    }

    (void)pContext;
    return MA_SUCCESS;
}


static ma_result ma_device_main_loop__dsound(ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;
    ma_uint32 bpfDeviceCapture  = ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
    ma_uint32 bpfDevicePlayback = ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
    HRESULT hr;
    DWORD lockOffsetInBytesCapture;
    DWORD lockSizeInBytesCapture;
    DWORD mappedSizeInBytesCapture;
    DWORD mappedDeviceFramesProcessedCapture;
    void* pMappedDeviceBufferCapture;
    DWORD lockOffsetInBytesPlayback;
    DWORD lockSizeInBytesPlayback;
    DWORD mappedSizeInBytesPlayback;
    void* pMappedDeviceBufferPlayback;
    DWORD prevReadCursorInBytesCapture = 0;
    DWORD prevPlayCursorInBytesPlayback = 0;
    ma_bool32 physicalPlayCursorLoopFlagPlayback = 0;
    DWORD virtualWriteCursorInBytesPlayback = 0;
    ma_bool32 virtualWriteCursorLoopFlagPlayback = 0;
    ma_bool32 isPlaybackDeviceStarted = MA_FALSE;
    ma_uint32 framesWrittenToPlaybackDevice = 0;   /* For knowing whether or not the playback device needs to be started. */
    ma_uint32 waitTimeInMilliseconds = 1;

    MA_ASSERT(pDevice != NULL);

    /* The first thing to do is start the capture device. The playback device is only started after the first period is written. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        if (FAILED(ma_IDirectSoundCaptureBuffer_Start((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, MA_DSCBSTART_LOOPING))) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCaptureBuffer_Start() failed.", MA_FAILED_TO_START_BACKEND_DEVICE);
        }
    }
    
    while (ma_device__get_state(pDevice) == MA_STATE_STARTED) {
        switch (pDevice->type)
        {
            case ma_device_type_duplex:
            {
                DWORD physicalCaptureCursorInBytes;
                DWORD physicalReadCursorInBytes;
                hr = ma_IDirectSoundCaptureBuffer_GetCurrentPosition((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, &physicalCaptureCursorInBytes, &physicalReadCursorInBytes);
                if (FAILED(hr)) {
                    return ma_result_from_HRESULT(hr);
                }

                /* If nothing is available we just sleep for a bit and return from this iteration. */
                if (physicalReadCursorInBytes == prevReadCursorInBytesCapture) {
                    ma_sleep(waitTimeInMilliseconds);
                    continue; /* Nothing is available in the capture buffer. */
                }

                /*
                The current position has moved. We need to map all of the captured samples and write them to the playback device, making sure
                we don't return until every frame has been copied over.
                */
                if (prevReadCursorInBytesCapture < physicalReadCursorInBytes) {
                    /* The capture position has not looped. This is the simple case. */
                    lockOffsetInBytesCapture = prevReadCursorInBytesCapture;
                    lockSizeInBytesCapture   = (physicalReadCursorInBytes - prevReadCursorInBytesCapture);
                } else {
                    /*
                    The capture position has looped. This is the more complex case. Map to the end of the buffer. If this does not return anything,
                    do it again from the start.
                    */
                    if (prevReadCursorInBytesCapture < pDevice->capture.internalPeriodSizeInFrames*pDevice->capture.internalPeriods*bpfDeviceCapture) {
                        /* Lock up to the end of the buffer. */
                        lockOffsetInBytesCapture = prevReadCursorInBytesCapture;
                        lockSizeInBytesCapture   = (pDevice->capture.internalPeriodSizeInFrames*pDevice->capture.internalPeriods*bpfDeviceCapture) - prevReadCursorInBytesCapture;
                    } else {
                        /* Lock starting from the start of the buffer. */
                        lockOffsetInBytesCapture = 0;
                        lockSizeInBytesCapture   = physicalReadCursorInBytes;
                    }
                }

                if (lockSizeInBytesCapture == 0) {
                    ma_sleep(waitTimeInMilliseconds);
                    continue; /* Nothing is available in the capture buffer. */
                }

                hr = ma_IDirectSoundCaptureBuffer_Lock((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, lockOffsetInBytesCapture, lockSizeInBytesCapture, &pMappedDeviceBufferCapture, &mappedSizeInBytesCapture, NULL, NULL, 0);
                if (FAILED(hr)) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to map buffer from capture device in preparation for writing to the device.", ma_result_from_HRESULT(hr));
                }


                /* At this point we have some input data that we need to output. We do not return until every mapped frame of the input data is written to the playback device. */
                mappedDeviceFramesProcessedCapture = 0;

                for (;;) {  /* Keep writing to the playback device. */
                    ma_uint8  inputFramesInClientFormat[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                    ma_uint32 inputFramesInClientFormatCap = sizeof(inputFramesInClientFormat) / ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
                    ma_uint8  outputFramesInClientFormat[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                    ma_uint32 outputFramesInClientFormatCap = sizeof(outputFramesInClientFormat) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
                    ma_uint32 outputFramesInClientFormatCount;
                    ma_uint32 outputFramesInClientFormatConsumed = 0;
                    ma_uint64 clientCapturedFramesToProcess = ma_min(inputFramesInClientFormatCap, outputFramesInClientFormatCap);
                    ma_uint64 deviceCapturedFramesToProcess = (mappedSizeInBytesCapture / bpfDeviceCapture) - mappedDeviceFramesProcessedCapture;
                    void* pRunningMappedDeviceBufferCapture = ma_offset_ptr(pMappedDeviceBufferCapture, mappedDeviceFramesProcessedCapture * bpfDeviceCapture);

                    result = ma_data_converter_process_pcm_frames(&pDevice->capture.converter, pRunningMappedDeviceBufferCapture, &deviceCapturedFramesToProcess, inputFramesInClientFormat, &clientCapturedFramesToProcess);
                    if (result != MA_SUCCESS) {
                        break;
                    }

                    outputFramesInClientFormatCount     = (ma_uint32)clientCapturedFramesToProcess;
                    mappedDeviceFramesProcessedCapture += (ma_uint32)deviceCapturedFramesToProcess;

                    ma_device__on_data(pDevice, outputFramesInClientFormat, inputFramesInClientFormat, (ma_uint32)clientCapturedFramesToProcess);

                    /* At this point we have input and output data in client format. All we need to do now is convert it to the output device format. This may take a few passes. */
                    for (;;) {
                        ma_uint32 framesWrittenThisIteration;
                        DWORD physicalPlayCursorInBytes;
                        DWORD physicalWriteCursorInBytes;
                        DWORD availableBytesPlayback;
                        DWORD silentPaddingInBytes = 0; /* <-- Must be initialized to 0. */

                        /* We need the physical play and write cursors. */
                        if (FAILED(ma_IDirectSoundBuffer_GetCurrentPosition((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, &physicalPlayCursorInBytes, &physicalWriteCursorInBytes))) {
                            break;
                        }

                        if (physicalPlayCursorInBytes < prevPlayCursorInBytesPlayback) {
                            physicalPlayCursorLoopFlagPlayback = !physicalPlayCursorLoopFlagPlayback;
                        }
                        prevPlayCursorInBytesPlayback  = physicalPlayCursorInBytes;

                        /* If there's any bytes available for writing we can do that now. The space between the virtual cursor position and play cursor. */
                        if (physicalPlayCursorLoopFlagPlayback == virtualWriteCursorLoopFlagPlayback) {
                            /* Same loop iteration. The available bytes wraps all the way around from the virtual write cursor to the physical play cursor. */
                            if (physicalPlayCursorInBytes <= virtualWriteCursorInBytesPlayback) {
                                availableBytesPlayback  = (pDevice->playback.internalPeriodSizeInFrames*pDevice->playback.internalPeriods*bpfDevicePlayback) - virtualWriteCursorInBytesPlayback;
                                availableBytesPlayback += physicalPlayCursorInBytes;    /* Wrap around. */
                            } else {
                                /* This is an error. */
                            #ifdef MA_DEBUG_OUTPUT
                                printf("[DirectSound] (Duplex/Playback) WARNING: Play cursor has moved in front of the write cursor (same loop iterations). physicalPlayCursorInBytes=%ld, virtualWriteCursorInBytes=%ld.\n", physicalPlayCursorInBytes, virtualWriteCursorInBytesPlayback);
                            #endif
                                availableBytesPlayback = 0;
                            }
                        } else {
                            /* Different loop iterations. The available bytes only goes from the virtual write cursor to the physical play cursor. */
                            if (physicalPlayCursorInBytes >= virtualWriteCursorInBytesPlayback) {
                                availableBytesPlayback = physicalPlayCursorInBytes - virtualWriteCursorInBytesPlayback;
                            } else {
                                /* This is an error. */
                            #ifdef MA_DEBUG_OUTPUT
                                printf("[DirectSound] (Duplex/Playback) WARNING: Write cursor has moved behind the play cursor (different loop iterations). physicalPlayCursorInBytes=%ld, virtualWriteCursorInBytes=%ld.\n", physicalPlayCursorInBytes, virtualWriteCursorInBytesPlayback);
                            #endif
                                availableBytesPlayback = 0;
                            }
                        }

                    #ifdef MA_DEBUG_OUTPUT
                        /*printf("[DirectSound] (Duplex/Playback) physicalPlayCursorInBytes=%d, availableBytesPlayback=%d\n", physicalPlayCursorInBytes, availableBytesPlayback);*/
                    #endif

                        /* If there's no room available for writing we need to wait for more. */
                        if (availableBytesPlayback == 0) {
                            /* If we haven't started the device yet, this will never get beyond 0. In this case we need to get the device started. */
                            if (!isPlaybackDeviceStarted) {
                                hr = ma_IDirectSoundBuffer_Play((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0, 0, MA_DSBPLAY_LOOPING);
                                if (FAILED(hr)) {
                                    ma_IDirectSoundCaptureBuffer_Stop((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer);
                                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Play() failed.", ma_result_from_HRESULT(hr));
                                }
                                isPlaybackDeviceStarted = MA_TRUE;
                            } else {
                                ma_sleep(waitTimeInMilliseconds);
                                continue;
                            }
                        }


                        /* Getting here means there room available somewhere. We limit this to either the end of the buffer or the physical play cursor, whichever is closest. */
                        lockOffsetInBytesPlayback = virtualWriteCursorInBytesPlayback;
                        if (physicalPlayCursorLoopFlagPlayback == virtualWriteCursorLoopFlagPlayback) {
                            /* Same loop iteration. Go up to the end of the buffer. */
                            lockSizeInBytesPlayback = (pDevice->playback.internalPeriodSizeInFrames*pDevice->playback.internalPeriods*bpfDevicePlayback) - virtualWriteCursorInBytesPlayback;
                        } else {
                            /* Different loop iterations. Go up to the physical play cursor. */
                            lockSizeInBytesPlayback = physicalPlayCursorInBytes - virtualWriteCursorInBytesPlayback;
                        }

                        hr = ma_IDirectSoundBuffer_Lock((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, lockOffsetInBytesPlayback, lockSizeInBytesPlayback, &pMappedDeviceBufferPlayback, &mappedSizeInBytesPlayback, NULL, NULL, 0);
                        if (FAILED(hr)) {
                            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to map buffer from playback device in preparation for writing to the device.", ma_result_from_HRESULT(hr));
                            break;
                        }

                        /*
                        Experiment: If the playback buffer is being starved, pad it with some silence to get it back in sync. This will cause a glitch, but it may prevent
                        endless glitching due to it constantly running out of data.
                        */
                        if (isPlaybackDeviceStarted) {
                            DWORD bytesQueuedForPlayback = (pDevice->playback.internalPeriodSizeInFrames*pDevice->playback.internalPeriods*bpfDevicePlayback) - availableBytesPlayback;
                            if (bytesQueuedForPlayback < (pDevice->playback.internalPeriodSizeInFrames*bpfDevicePlayback)) {
                                silentPaddingInBytes   = (pDevice->playback.internalPeriodSizeInFrames*2*bpfDevicePlayback) - bytesQueuedForPlayback;
                                if (silentPaddingInBytes > lockSizeInBytesPlayback) {
                                    silentPaddingInBytes = lockSizeInBytesPlayback;
                                }

                        #ifdef MA_DEBUG_OUTPUT
                                printf("[DirectSound] (Duplex/Playback) Playback buffer starved. availableBytesPlayback=%ld, silentPaddingInBytes=%ld\n", availableBytesPlayback, silentPaddingInBytes);
                        #endif
                            }
                        }

                        /* At this point we have a buffer for output. */
                        if (silentPaddingInBytes > 0) {
                            MA_ZERO_MEMORY(pMappedDeviceBufferPlayback, silentPaddingInBytes);
                            framesWrittenThisIteration = silentPaddingInBytes/bpfDevicePlayback;
                        } else {
                            ma_uint64 convertedFrameCountIn  = (outputFramesInClientFormatCount - outputFramesInClientFormatConsumed);
                            ma_uint64 convertedFrameCountOut = mappedSizeInBytesPlayback/bpfDevicePlayback;
                            void* pConvertedFramesIn  = ma_offset_ptr(outputFramesInClientFormat, outputFramesInClientFormatConsumed * bpfDevicePlayback);
                            void* pConvertedFramesOut = pMappedDeviceBufferPlayback;

                            result = ma_data_converter_process_pcm_frames(&pDevice->playback.converter, pConvertedFramesIn, &convertedFrameCountIn, pConvertedFramesOut, &convertedFrameCountOut);
                            if (result != MA_SUCCESS) {
                                break;
                            }

                            outputFramesInClientFormatConsumed += (ma_uint32)convertedFrameCountOut;
                            framesWrittenThisIteration          = (ma_uint32)convertedFrameCountOut;
                        }
                        

                        hr = ma_IDirectSoundBuffer_Unlock((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, pMappedDeviceBufferPlayback, framesWrittenThisIteration*bpfDevicePlayback, NULL, 0);
                        if (FAILED(hr)) {
                            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to unlock internal buffer from playback device after writing to the device.", ma_result_from_HRESULT(hr));
                            break;
                        }

                        virtualWriteCursorInBytesPlayback += framesWrittenThisIteration*bpfDevicePlayback;
                        if ((virtualWriteCursorInBytesPlayback/bpfDevicePlayback) == pDevice->playback.internalPeriodSizeInFrames*pDevice->playback.internalPeriods) {
                            virtualWriteCursorInBytesPlayback  = 0;
                            virtualWriteCursorLoopFlagPlayback = !virtualWriteCursorLoopFlagPlayback;
                        }
                        
                        /*
                        We may need to start the device. We want two full periods to be written before starting the playback device. Having an extra period adds
                        a bit of a buffer to prevent the playback buffer from getting starved.
                        */
                        framesWrittenToPlaybackDevice += framesWrittenThisIteration;
                        if (!isPlaybackDeviceStarted && framesWrittenToPlaybackDevice >= (pDevice->playback.internalPeriodSizeInFrames*2)) {
                            hr = ma_IDirectSoundBuffer_Play((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0, 0, MA_DSBPLAY_LOOPING);
                            if (FAILED(hr)) {
                                ma_IDirectSoundCaptureBuffer_Stop((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer);
                                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Play() failed.", ma_result_from_HRESULT(hr));
                            }
                            isPlaybackDeviceStarted = MA_TRUE;
                        }

                        if (framesWrittenThisIteration < mappedSizeInBytesPlayback/bpfDevicePlayback) {
                            break;  /* We're finished with the output data.*/
                        }
                    }

                    if (clientCapturedFramesToProcess == 0) {
                        break;  /* We just consumed every input sample. */
                    }
                }


                /* At this point we're done with the mapped portion of the capture buffer. */
                hr = ma_IDirectSoundCaptureBuffer_Unlock((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, pMappedDeviceBufferCapture, mappedSizeInBytesCapture, NULL, 0);
                if (FAILED(hr)) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to unlock internal buffer from capture device after reading from the device.", ma_result_from_HRESULT(hr));
                }
                prevReadCursorInBytesCapture = (lockOffsetInBytesCapture + mappedSizeInBytesCapture);
            } break;



            case ma_device_type_capture:
            {
                DWORD physicalCaptureCursorInBytes;
                DWORD physicalReadCursorInBytes;
                hr = ma_IDirectSoundCaptureBuffer_GetCurrentPosition((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, &physicalCaptureCursorInBytes, &physicalReadCursorInBytes);
                if (FAILED(hr)) {
                    return MA_ERROR;
                }

                /* If the previous capture position is the same as the current position we need to wait a bit longer. */
                if (prevReadCursorInBytesCapture == physicalReadCursorInBytes) {
                    ma_sleep(waitTimeInMilliseconds);
                    continue;
                }

                /* Getting here means we have capture data available. */
                if (prevReadCursorInBytesCapture < physicalReadCursorInBytes) {
                    /* The capture position has not looped. This is the simple case. */
                    lockOffsetInBytesCapture = prevReadCursorInBytesCapture;
                    lockSizeInBytesCapture   = (physicalReadCursorInBytes - prevReadCursorInBytesCapture);
                } else {
                    /*
                    The capture position has looped. This is the more complex case. Map to the end of the buffer. If this does not return anything,
                    do it again from the start.
                    */
                    if (prevReadCursorInBytesCapture < pDevice->capture.internalPeriodSizeInFrames*pDevice->capture.internalPeriods*bpfDeviceCapture) {
                        /* Lock up to the end of the buffer. */
                        lockOffsetInBytesCapture = prevReadCursorInBytesCapture;
                        lockSizeInBytesCapture   = (pDevice->capture.internalPeriodSizeInFrames*pDevice->capture.internalPeriods*bpfDeviceCapture) - prevReadCursorInBytesCapture;
                    } else {
                        /* Lock starting from the start of the buffer. */
                        lockOffsetInBytesCapture = 0;
                        lockSizeInBytesCapture   = physicalReadCursorInBytes;
                    }
                }

            #ifdef MA_DEBUG_OUTPUT
                /*printf("[DirectSound] (Capture) physicalCaptureCursorInBytes=%d, physicalReadCursorInBytes=%d\n", physicalCaptureCursorInBytes, physicalReadCursorInBytes);*/
                /*printf("[DirectSound] (Capture) lockOffsetInBytesCapture=%d, lockSizeInBytesCapture=%d\n", lockOffsetInBytesCapture, lockSizeInBytesCapture);*/
            #endif

                if (lockSizeInBytesCapture < pDevice->capture.internalPeriodSizeInFrames) {
                    ma_sleep(waitTimeInMilliseconds);
                    continue; /* Nothing is available in the capture buffer. */
                }

                hr = ma_IDirectSoundCaptureBuffer_Lock((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, lockOffsetInBytesCapture, lockSizeInBytesCapture, &pMappedDeviceBufferCapture, &mappedSizeInBytesCapture, NULL, NULL, 0);
                if (FAILED(hr)) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to map buffer from capture device in preparation for writing to the device.", ma_result_from_HRESULT(hr));
                }

            #ifdef MA_DEBUG_OUTPUT
                if (lockSizeInBytesCapture != mappedSizeInBytesCapture) {
                    printf("[DirectSound] (Capture) lockSizeInBytesCapture=%ld != mappedSizeInBytesCapture=%ld\n", lockSizeInBytesCapture, mappedSizeInBytesCapture);
                }
            #endif

                ma_device__send_frames_to_client(pDevice, mappedSizeInBytesCapture/bpfDeviceCapture, pMappedDeviceBufferCapture);

                hr = ma_IDirectSoundCaptureBuffer_Unlock((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, pMappedDeviceBufferCapture, mappedSizeInBytesCapture, NULL, 0);
                if (FAILED(hr)) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to unlock internal buffer from capture device after reading from the device.", ma_result_from_HRESULT(hr));
                }
                prevReadCursorInBytesCapture = lockOffsetInBytesCapture + mappedSizeInBytesCapture;

                if (prevReadCursorInBytesCapture == (pDevice->capture.internalPeriodSizeInFrames*pDevice->capture.internalPeriods*bpfDeviceCapture)) {
                    prevReadCursorInBytesCapture = 0;
                }
            } break;



            case ma_device_type_playback:
            {
                DWORD availableBytesPlayback;
                DWORD physicalPlayCursorInBytes;
                DWORD physicalWriteCursorInBytes;
                hr = ma_IDirectSoundBuffer_GetCurrentPosition((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, &physicalPlayCursorInBytes, &physicalWriteCursorInBytes);
                if (FAILED(hr)) {
                    break;
                }

                if (physicalPlayCursorInBytes < prevPlayCursorInBytesPlayback) {
                    physicalPlayCursorLoopFlagPlayback = !physicalPlayCursorLoopFlagPlayback;
                }
                prevPlayCursorInBytesPlayback  = physicalPlayCursorInBytes;

                /* If there's any bytes available for writing we can do that now. The space between the virtual cursor position and play cursor. */
                if (physicalPlayCursorLoopFlagPlayback == virtualWriteCursorLoopFlagPlayback) {
                    /* Same loop iteration. The available bytes wraps all the way around from the virtual write cursor to the physical play cursor. */
                    if (physicalPlayCursorInBytes <= virtualWriteCursorInBytesPlayback) {
                        availableBytesPlayback  = (pDevice->playback.internalPeriodSizeInFrames*pDevice->playback.internalPeriods*bpfDevicePlayback) - virtualWriteCursorInBytesPlayback;
                        availableBytesPlayback += physicalPlayCursorInBytes;    /* Wrap around. */
                    } else {
                        /* This is an error. */
                    #ifdef MA_DEBUG_OUTPUT
                        printf("[DirectSound] (Playback) WARNING: Play cursor has moved in front of the write cursor (same loop iterations). physicalPlayCursorInBytes=%ld, virtualWriteCursorInBytes=%ld.\n", physicalPlayCursorInBytes, virtualWriteCursorInBytesPlayback);
                    #endif
                        availableBytesPlayback = 0;
                    }
                } else {
                    /* Different loop iterations. The available bytes only goes from the virtual write cursor to the physical play cursor. */
                    if (physicalPlayCursorInBytes >= virtualWriteCursorInBytesPlayback) {
                        availableBytesPlayback = physicalPlayCursorInBytes - virtualWriteCursorInBytesPlayback;
                    } else {
                        /* This is an error. */
                    #ifdef MA_DEBUG_OUTPUT
                        printf("[DirectSound] (Playback) WARNING: Write cursor has moved behind the play cursor (different loop iterations). physicalPlayCursorInBytes=%ld, virtualWriteCursorInBytes=%ld.\n", physicalPlayCursorInBytes, virtualWriteCursorInBytesPlayback);
                    #endif
                        availableBytesPlayback = 0;
                    }
                }

            #ifdef MA_DEBUG_OUTPUT
                /*printf("[DirectSound] (Playback) physicalPlayCursorInBytes=%d, availableBytesPlayback=%d\n", physicalPlayCursorInBytes, availableBytesPlayback);*/
            #endif

                /* If there's no room available for writing we need to wait for more. */
                if (availableBytesPlayback < pDevice->playback.internalPeriodSizeInFrames) {
                    /* If we haven't started the device yet, this will never get beyond 0. In this case we need to get the device started. */
                    if (availableBytesPlayback == 0 && !isPlaybackDeviceStarted) {
                        hr = ma_IDirectSoundBuffer_Play((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0, 0, MA_DSBPLAY_LOOPING);
                        if (FAILED(hr)) {
                            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Play() failed.", ma_result_from_HRESULT(hr));
                        }
                        isPlaybackDeviceStarted = MA_TRUE;
                    } else {
                        ma_sleep(waitTimeInMilliseconds);
                        continue;
                    }
                }

                /* Getting here means there room available somewhere. We limit this to either the end of the buffer or the physical play cursor, whichever is closest. */
                lockOffsetInBytesPlayback = virtualWriteCursorInBytesPlayback;
                if (physicalPlayCursorLoopFlagPlayback == virtualWriteCursorLoopFlagPlayback) {
                    /* Same loop iteration. Go up to the end of the buffer. */
                    lockSizeInBytesPlayback = (pDevice->playback.internalPeriodSizeInFrames*pDevice->playback.internalPeriods*bpfDevicePlayback) - virtualWriteCursorInBytesPlayback;
                } else {
                    /* Different loop iterations. Go up to the physical play cursor. */
                    lockSizeInBytesPlayback = physicalPlayCursorInBytes - virtualWriteCursorInBytesPlayback;
                }

                hr = ma_IDirectSoundBuffer_Lock((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, lockOffsetInBytesPlayback, lockSizeInBytesPlayback, &pMappedDeviceBufferPlayback, &mappedSizeInBytesPlayback, NULL, NULL, 0);
                if (FAILED(hr)) {
                    result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to map buffer from playback device in preparation for writing to the device.", ma_result_from_HRESULT(hr));
                    break;
                }

                /* At this point we have a buffer for output. */
                ma_device__read_frames_from_client(pDevice, (mappedSizeInBytesPlayback/bpfDevicePlayback), pMappedDeviceBufferPlayback);

                hr = ma_IDirectSoundBuffer_Unlock((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, pMappedDeviceBufferPlayback, mappedSizeInBytesPlayback, NULL, 0);
                if (FAILED(hr)) {
                    result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to unlock internal buffer from playback device after writing to the device.", ma_result_from_HRESULT(hr));
                    break;
                }

                virtualWriteCursorInBytesPlayback += mappedSizeInBytesPlayback;
                if (virtualWriteCursorInBytesPlayback == pDevice->playback.internalPeriodSizeInFrames*pDevice->playback.internalPeriods*bpfDevicePlayback) {
                    virtualWriteCursorInBytesPlayback  = 0;
                    virtualWriteCursorLoopFlagPlayback = !virtualWriteCursorLoopFlagPlayback;
                }
                        
                /*
                We may need to start the device. We want two full periods to be written before starting the playback device. Having an extra period adds
                a bit of a buffer to prevent the playback buffer from getting starved.
                */
                framesWrittenToPlaybackDevice += mappedSizeInBytesPlayback/bpfDevicePlayback;
                if (!isPlaybackDeviceStarted && framesWrittenToPlaybackDevice >= pDevice->playback.internalPeriodSizeInFrames) {
                    hr = ma_IDirectSoundBuffer_Play((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0, 0, MA_DSBPLAY_LOOPING);
                    if (FAILED(hr)) {
                        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Play() failed.", ma_result_from_HRESULT(hr));
                    }
                    isPlaybackDeviceStarted = MA_TRUE;
                }
            } break;


            default: return MA_INVALID_ARGS;   /* Invalid device type. */
        }

        if (result != MA_SUCCESS) {
            return result;
        }
    }

    /* Getting here means the device is being stopped. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        hr = ma_IDirectSoundCaptureBuffer_Stop((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCaptureBuffer_Stop() failed.", ma_result_from_HRESULT(hr));
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        /* The playback device should be drained before stopping. All we do is wait until the available bytes is equal to the size of the buffer. */
        if (isPlaybackDeviceStarted) {
            for (;;) {
                DWORD availableBytesPlayback = 0;
                DWORD physicalPlayCursorInBytes;
                DWORD physicalWriteCursorInBytes;
                hr = ma_IDirectSoundBuffer_GetCurrentPosition((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, &physicalPlayCursorInBytes, &physicalWriteCursorInBytes);
                if (FAILED(hr)) {
                    break;
                }

                if (physicalPlayCursorInBytes < prevPlayCursorInBytesPlayback) {
                    physicalPlayCursorLoopFlagPlayback = !physicalPlayCursorLoopFlagPlayback;
                }
                prevPlayCursorInBytesPlayback  = physicalPlayCursorInBytes;

                if (physicalPlayCursorLoopFlagPlayback == virtualWriteCursorLoopFlagPlayback) {
                    /* Same loop iteration. The available bytes wraps all the way around from the virtual write cursor to the physical play cursor. */
                    if (physicalPlayCursorInBytes <= virtualWriteCursorInBytesPlayback) {
                        availableBytesPlayback  = (pDevice->playback.internalPeriodSizeInFrames*pDevice->playback.internalPeriods*bpfDevicePlayback) - virtualWriteCursorInBytesPlayback;
                        availableBytesPlayback += physicalPlayCursorInBytes;    /* Wrap around. */
                    } else {
                        break;
                    }
                } else {
                    /* Different loop iterations. The available bytes only goes from the virtual write cursor to the physical play cursor. */
                    if (physicalPlayCursorInBytes >= virtualWriteCursorInBytesPlayback) {
                        availableBytesPlayback = physicalPlayCursorInBytes - virtualWriteCursorInBytesPlayback;
                    } else {
                        break;
                    }
                }

                if (availableBytesPlayback >= (pDevice->playback.internalPeriodSizeInFrames*pDevice->playback.internalPeriods*bpfDevicePlayback)) {
                    break;
                }

                ma_sleep(waitTimeInMilliseconds);
            }
        }

        hr = ma_IDirectSoundBuffer_Stop((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Stop() failed.", ma_result_from_HRESULT(hr));
        }

        ma_IDirectSoundBuffer_SetCurrentPosition((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0);
    }

    return MA_SUCCESS;
}

static ma_result ma_context_uninit__dsound(ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pContext->backend == ma_backend_dsound);

    ma_dlclose(pContext, pContext->dsound.hDSoundDLL);

    return MA_SUCCESS;
}

static ma_result ma_context_init__dsound(const ma_context_config* pConfig, ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);

    (void)pConfig;

    pContext->dsound.hDSoundDLL = ma_dlopen(pContext, "dsound.dll");
    if (pContext->dsound.hDSoundDLL == NULL) {
        return MA_API_NOT_FOUND;
    }

    pContext->dsound.DirectSoundCreate            = ma_dlsym(pContext, pContext->dsound.hDSoundDLL, "DirectSoundCreate");
    pContext->dsound.DirectSoundEnumerateA        = ma_dlsym(pContext, pContext->dsound.hDSoundDLL, "DirectSoundEnumerateA");
    pContext->dsound.DirectSoundCaptureCreate     = ma_dlsym(pContext, pContext->dsound.hDSoundDLL, "DirectSoundCaptureCreate");
    pContext->dsound.DirectSoundCaptureEnumerateA = ma_dlsym(pContext, pContext->dsound.hDSoundDLL, "DirectSoundCaptureEnumerateA");

    pContext->onUninit         = ma_context_uninit__dsound;
    pContext->onDeviceIDEqual  = ma_context_is_device_id_equal__dsound;
    pContext->onEnumDevices    = ma_context_enumerate_devices__dsound;
    pContext->onGetDeviceInfo  = ma_context_get_device_info__dsound;
    pContext->onDeviceInit     = ma_device_init__dsound;
    pContext->onDeviceUninit   = ma_device_uninit__dsound;
    pContext->onDeviceStart    = NULL;  /* Not used. Started in onDeviceMainLoop. */
    pContext->onDeviceStop     = NULL;  /* Not used. Stopped in onDeviceMainLoop. */
    pContext->onDeviceMainLoop = ma_device_main_loop__dsound;

    return MA_SUCCESS;
}
#endif



/******************************************************************************

WinMM Backend

******************************************************************************/
#ifdef MA_HAS_WINMM

/*
Some older compilers don't have WAVEOUTCAPS2A and WAVEINCAPS2A, so we'll need to write this ourselves. These structures
are exactly the same as the older ones but they have a few GUIDs for manufacturer/product/name identification. I'm keeping
the names the same as the Win32 library for consistency, but namespaced to avoid naming conflicts with the Win32 version.
*/
typedef struct
{
    WORD wMid;
    WORD wPid;
    MMVERSION vDriverVersion;
    CHAR szPname[MAXPNAMELEN];
    DWORD dwFormats;
    WORD wChannels;
    WORD wReserved1;
    DWORD dwSupport;
    GUID ManufacturerGuid;
    GUID ProductGuid;
    GUID NameGuid;
} MA_WAVEOUTCAPS2A;
typedef struct
{
    WORD wMid;
    WORD wPid;
    MMVERSION vDriverVersion;
    CHAR szPname[MAXPNAMELEN];
    DWORD dwFormats;
    WORD wChannels;
    WORD wReserved1;
    GUID ManufacturerGuid;
    GUID ProductGuid;
    GUID NameGuid;
} MA_WAVEINCAPS2A;

typedef UINT     (WINAPI * MA_PFN_waveOutGetNumDevs)(void);
typedef MMRESULT (WINAPI * MA_PFN_waveOutGetDevCapsA)(ma_uintptr uDeviceID, LPWAVEOUTCAPSA pwoc, UINT cbwoc);
typedef MMRESULT (WINAPI * MA_PFN_waveOutOpen)(LPHWAVEOUT phwo, UINT uDeviceID, LPCWAVEFORMATEX pwfx, DWORD_PTR dwCallback, DWORD_PTR dwInstance, DWORD fdwOpen);
typedef MMRESULT (WINAPI * MA_PFN_waveOutClose)(HWAVEOUT hwo);
typedef MMRESULT (WINAPI * MA_PFN_waveOutPrepareHeader)(HWAVEOUT hwo, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveOutUnprepareHeader)(HWAVEOUT hwo, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveOutWrite)(HWAVEOUT hwo, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveOutReset)(HWAVEOUT hwo);
typedef UINT     (WINAPI * MA_PFN_waveInGetNumDevs)(void);
typedef MMRESULT (WINAPI * MA_PFN_waveInGetDevCapsA)(ma_uintptr uDeviceID, LPWAVEINCAPSA pwic, UINT cbwic);
typedef MMRESULT (WINAPI * MA_PFN_waveInOpen)(LPHWAVEIN phwi, UINT uDeviceID, LPCWAVEFORMATEX pwfx, DWORD_PTR dwCallback, DWORD_PTR dwInstance, DWORD fdwOpen);
typedef MMRESULT (WINAPI * MA_PFN_waveInClose)(HWAVEIN hwi);
typedef MMRESULT (WINAPI * MA_PFN_waveInPrepareHeader)(HWAVEIN hwi, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveInUnprepareHeader)(HWAVEIN hwi, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveInAddBuffer)(HWAVEIN hwi, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveInStart)(HWAVEIN hwi);
typedef MMRESULT (WINAPI * MA_PFN_waveInReset)(HWAVEIN hwi);

static ma_result ma_result_from_MMRESULT(MMRESULT resultMM)
{
    switch (resultMM) {
        case MMSYSERR_NOERROR:      return MA_SUCCESS;
        case MMSYSERR_BADDEVICEID:  return MA_INVALID_ARGS;
        case MMSYSERR_INVALHANDLE:  return MA_INVALID_ARGS;
        case MMSYSERR_NOMEM:        return MA_OUT_OF_MEMORY;
        case MMSYSERR_INVALFLAG:    return MA_INVALID_ARGS;
        case MMSYSERR_INVALPARAM:   return MA_INVALID_ARGS;
        case MMSYSERR_HANDLEBUSY:   return MA_BUSY;
        case MMSYSERR_ERROR:        return MA_ERROR;
        default:                    return MA_ERROR;
    }
}

static char* ma_find_last_character(char* str, char ch)
{
    char* last;

    if (str == NULL) {
        return NULL;
    }

    last = NULL;
    while (*str != '\0') {
        if (*str == ch) {
            last = str;
        }

        str += 1;
    }

    return last;
}

static ma_uint32 ma_get_period_size_in_bytes(ma_uint32 periodSizeInFrames, ma_format format, ma_uint32 channels)
{
    return periodSizeInFrames * ma_get_bytes_per_frame(format, channels);
}


/*
Our own "WAVECAPS" structure that contains generic information shared between WAVEOUTCAPS2 and WAVEINCAPS2 so
we can do things generically and typesafely. Names are being kept the same for consistency.
*/
typedef struct
{
    CHAR szPname[MAXPNAMELEN];
    DWORD dwFormats;
    WORD wChannels;
    GUID NameGuid;
} MA_WAVECAPSA;

static ma_result ma_get_best_info_from_formats_flags__winmm(DWORD dwFormats, WORD channels, WORD* pBitsPerSample, DWORD* pSampleRate)
{
    WORD bitsPerSample = 0;
    DWORD sampleRate = 0;

    if (pBitsPerSample) {
        *pBitsPerSample = 0;
    }
    if (pSampleRate) {
        *pSampleRate = 0;
    }

    if (channels == 1) {
        bitsPerSample = 16;
        if ((dwFormats & WAVE_FORMAT_48M16) != 0) {
            sampleRate = 48000;
        } else if ((dwFormats & WAVE_FORMAT_44M16) != 0) {
            sampleRate = 44100;
        } else if ((dwFormats & WAVE_FORMAT_2M16) != 0) {
            sampleRate = 22050;
        } else if ((dwFormats & WAVE_FORMAT_1M16) != 0) {
            sampleRate = 11025;
        } else if ((dwFormats & WAVE_FORMAT_96M16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((dwFormats & WAVE_FORMAT_48M08) != 0) {
                sampleRate = 48000;
            } else if ((dwFormats & WAVE_FORMAT_44M08) != 0) {
                sampleRate = 44100;
            } else if ((dwFormats & WAVE_FORMAT_2M08) != 0) {
                sampleRate = 22050;
            } else if ((dwFormats & WAVE_FORMAT_1M08) != 0) {
                sampleRate = 11025;
            } else if ((dwFormats & WAVE_FORMAT_96M08) != 0) {
                sampleRate = 96000;
            } else {
                return MA_FORMAT_NOT_SUPPORTED;
            }
        }
    } else {
        bitsPerSample = 16;
        if ((dwFormats & WAVE_FORMAT_48S16) != 0) {
            sampleRate = 48000;
        } else if ((dwFormats & WAVE_FORMAT_44S16) != 0) {
            sampleRate = 44100;
        } else if ((dwFormats & WAVE_FORMAT_2S16) != 0) {
            sampleRate = 22050;
        } else if ((dwFormats & WAVE_FORMAT_1S16) != 0) {
            sampleRate = 11025;
        } else if ((dwFormats & WAVE_FORMAT_96S16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((dwFormats & WAVE_FORMAT_48S08) != 0) {
                sampleRate = 48000;
            } else if ((dwFormats & WAVE_FORMAT_44S08) != 0) {
                sampleRate = 44100;
            } else if ((dwFormats & WAVE_FORMAT_2S08) != 0) {
                sampleRate = 22050;
            } else if ((dwFormats & WAVE_FORMAT_1S08) != 0) {
                sampleRate = 11025;
            } else if ((dwFormats & WAVE_FORMAT_96S08) != 0) {
                sampleRate = 96000;
            } else {
                return MA_FORMAT_NOT_SUPPORTED;
            }
        }
    }

    if (pBitsPerSample) {
        *pBitsPerSample = bitsPerSample;
    }
    if (pSampleRate) {
        *pSampleRate = sampleRate;
    }

    return MA_SUCCESS;
}

static ma_result ma_formats_flags_to_WAVEFORMATEX__winmm(DWORD dwFormats, WORD channels, WAVEFORMATEX* pWF)
{
    MA_ASSERT(pWF != NULL);

    MA_ZERO_OBJECT(pWF);
    pWF->cbSize     = sizeof(*pWF);
    pWF->wFormatTag = WAVE_FORMAT_PCM;
    pWF->nChannels  = (WORD)channels;
    if (pWF->nChannels > 2) {
        pWF->nChannels = 2;
    }

    if (channels == 1) {
        pWF->wBitsPerSample = 16;
        if ((dwFormats & WAVE_FORMAT_48M16) != 0) {
            pWF->nSamplesPerSec = 48000;
        } else if ((dwFormats & WAVE_FORMAT_44M16) != 0) {
            pWF->nSamplesPerSec = 44100;
        } else if ((dwFormats & WAVE_FORMAT_2M16) != 0) {
            pWF->nSamplesPerSec = 22050;
        } else if ((dwFormats & WAVE_FORMAT_1M16) != 0) {
            pWF->nSamplesPerSec = 11025;
        } else if ((dwFormats & WAVE_FORMAT_96M16) != 0) {
            pWF->nSamplesPerSec = 96000;
        } else {
            pWF->wBitsPerSample = 8;
            if ((dwFormats & WAVE_FORMAT_48M08) != 0) {
                pWF->nSamplesPerSec = 48000;
            } else if ((dwFormats & WAVE_FORMAT_44M08) != 0) {
                pWF->nSamplesPerSec = 44100;
            } else if ((dwFormats & WAVE_FORMAT_2M08) != 0) {
                pWF->nSamplesPerSec = 22050;
            } else if ((dwFormats & WAVE_FORMAT_1M08) != 0) {
                pWF->nSamplesPerSec = 11025;
            } else if ((dwFormats & WAVE_FORMAT_96M08) != 0) {
                pWF->nSamplesPerSec = 96000;
            } else {
                return MA_FORMAT_NOT_SUPPORTED;
            }
        }
    } else {
        pWF->wBitsPerSample = 16;
        if ((dwFormats & WAVE_FORMAT_48S16) != 0) {
            pWF->nSamplesPerSec = 48000;
        } else if ((dwFormats & WAVE_FORMAT_44S16) != 0) {
            pWF->nSamplesPerSec = 44100;
        } else if ((dwFormats & WAVE_FORMAT_2S16) != 0) {
            pWF->nSamplesPerSec = 22050;
        } else if ((dwFormats & WAVE_FORMAT_1S16) != 0) {
            pWF->nSamplesPerSec = 11025;
        } else if ((dwFormats & WAVE_FORMAT_96S16) != 0) {
            pWF->nSamplesPerSec = 96000;
        } else {
            pWF->wBitsPerSample = 8;
            if ((dwFormats & WAVE_FORMAT_48S08) != 0) {
                pWF->nSamplesPerSec = 48000;
            } else if ((dwFormats & WAVE_FORMAT_44S08) != 0) {
                pWF->nSamplesPerSec = 44100;
            } else if ((dwFormats & WAVE_FORMAT_2S08) != 0) {
                pWF->nSamplesPerSec = 22050;
            } else if ((dwFormats & WAVE_FORMAT_1S08) != 0) {
                pWF->nSamplesPerSec = 11025;
            } else if ((dwFormats & WAVE_FORMAT_96S08) != 0) {
                pWF->nSamplesPerSec = 96000;
            } else {
                return MA_FORMAT_NOT_SUPPORTED;
            }
        }
    }

    pWF->nBlockAlign     = (WORD)(pWF->nChannels * pWF->wBitsPerSample / 8);
    pWF->nAvgBytesPerSec = pWF->nBlockAlign * pWF->nSamplesPerSec;

    return MA_SUCCESS;
}

static ma_result ma_context_get_device_info_from_WAVECAPS(ma_context* pContext, MA_WAVECAPSA* pCaps, ma_device_info* pDeviceInfo)
{
    WORD bitsPerSample;
    DWORD sampleRate;
    ma_result result;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pCaps != NULL);
    MA_ASSERT(pDeviceInfo != NULL);

    /*
    Name / Description
    
    Unfortunately the name specified in WAVE(OUT/IN)CAPS2 is limited to 31 characters. This results in an unprofessional looking
    situation where the names of the devices are truncated. To help work around this, we need to look at the name GUID and try
    looking in the registry for the full name. If we can't find it there, we need to just fall back to the default name.
    */

    /* Set the default to begin with. */
    ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), pCaps->szPname, (size_t)-1);

    /*
    Now try the registry. There's a few things to consider here:
    - The name GUID can be null, in which we case we just need to stick to the original 31 characters.
    - If the name GUID is not present in the registry we'll also need to stick to the original 31 characters.
    - I like consistency, so I want the returned device names to be consistent with those returned by WASAPI and DirectSound. The
      problem, however is that WASAPI and DirectSound use "<component> (<name>)" format (such as "Speakers (High Definition Audio)"),
      but WinMM does not specificy the component name. From my admittedly limited testing, I've notice the component name seems to
      usually fit within the 31 characters of the fixed sized buffer, so what I'm going to do is parse that string for the component
      name, and then concatenate the name from the registry.
    */
    if (!ma_is_guid_null(&pCaps->NameGuid)) {
        wchar_t guidStrW[256];
        if (((MA_PFN_StringFromGUID2)pContext->win32.StringFromGUID2)(&pCaps->NameGuid, guidStrW, ma_countof(guidStrW)) > 0) {
            char guidStr[256];
            char keyStr[1024];
            HKEY hKey;

            WideCharToMultiByte(CP_UTF8, 0, guidStrW, -1, guidStr, sizeof(guidStr), 0, FALSE);

            ma_strcpy_s(keyStr, sizeof(keyStr), "SYSTEM\\CurrentControlSet\\Control\\MediaCategories\\");
            ma_strcat_s(keyStr, sizeof(keyStr), guidStr);

            if (((MA_PFN_RegOpenKeyExA)pContext->win32.RegOpenKeyExA)(HKEY_LOCAL_MACHINE, keyStr, 0, KEY_READ, &hKey) == ERROR_SUCCESS) {
                BYTE nameFromReg[512];
                DWORD nameFromRegSize = sizeof(nameFromReg);
                result = ((MA_PFN_RegQueryValueExA)pContext->win32.RegQueryValueExA)(hKey, "Name", 0, NULL, (LPBYTE)nameFromReg, (LPDWORD)&nameFromRegSize);
                ((MA_PFN_RegCloseKey)pContext->win32.RegCloseKey)(hKey);

                if (result == ERROR_SUCCESS) {
                    /* We have the value from the registry, so now we need to construct the name string. */
                    char name[1024];
                    if (ma_strcpy_s(name, sizeof(name), pDeviceInfo->name) == 0) {
                        char* nameBeg = ma_find_last_character(name, '(');
                        if (nameBeg != NULL) {
                            size_t leadingLen = (nameBeg - name);
                            ma_strncpy_s(nameBeg + 1, sizeof(name) - leadingLen, (const char*)nameFromReg, (size_t)-1);

                            /* The closing ")", if it can fit. */
                            if (leadingLen + nameFromRegSize < sizeof(name)-1) {
                                ma_strcat_s(name, sizeof(name), ")");
                            }

                            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), name, (size_t)-1);
                        }
                    }
                }
            }
        }
    }


    result = ma_get_best_info_from_formats_flags__winmm(pCaps->dwFormats, pCaps->wChannels, &bitsPerSample, &sampleRate);
    if (result != MA_SUCCESS) {
        return result;
    }

    pDeviceInfo->minChannels = pCaps->wChannels;
    pDeviceInfo->maxChannels = pCaps->wChannels;
    pDeviceInfo->minSampleRate = sampleRate;
    pDeviceInfo->maxSampleRate = sampleRate;
    pDeviceInfo->formatCount = 1;
    if (bitsPerSample == 8) {
        pDeviceInfo->formats[0] = ma_format_u8;
    } else if (bitsPerSample == 16) {
        pDeviceInfo->formats[0] = ma_format_s16;
    } else if (bitsPerSample == 24) {
        pDeviceInfo->formats[0] = ma_format_s24;
    } else if (bitsPerSample == 32) {
        pDeviceInfo->formats[0] = ma_format_s32;
    } else {
        return MA_FORMAT_NOT_SUPPORTED;
    }

    return MA_SUCCESS;
}

static ma_result ma_context_get_device_info_from_WAVEOUTCAPS2(ma_context* pContext, MA_WAVEOUTCAPS2A* pCaps, ma_device_info* pDeviceInfo)
{
    MA_WAVECAPSA caps;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pCaps != NULL);
    MA_ASSERT(pDeviceInfo != NULL);

    MA_COPY_MEMORY(caps.szPname, pCaps->szPname, sizeof(caps.szPname));
    caps.dwFormats = pCaps->dwFormats;
    caps.wChannels = pCaps->wChannels;
    caps.NameGuid = pCaps->NameGuid;
    return ma_context_get_device_info_from_WAVECAPS(pContext, &caps, pDeviceInfo);
}

static ma_result ma_context_get_device_info_from_WAVEINCAPS2(ma_context* pContext, MA_WAVEINCAPS2A* pCaps, ma_device_info* pDeviceInfo)
{
    MA_WAVECAPSA caps;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pCaps != NULL);
    MA_ASSERT(pDeviceInfo != NULL);

    MA_COPY_MEMORY(caps.szPname, pCaps->szPname, sizeof(caps.szPname));
    caps.dwFormats = pCaps->dwFormats;
    caps.wChannels = pCaps->wChannels;
    caps.NameGuid = pCaps->NameGuid;
    return ma_context_get_device_info_from_WAVECAPS(pContext, &caps, pDeviceInfo);
}


static ma_bool32 ma_context_is_device_id_equal__winmm(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pID0 != NULL);
    MA_ASSERT(pID1 != NULL);
    (void)pContext;

    return pID0->winmm == pID1->winmm;
}

static ma_result ma_context_enumerate_devices__winmm(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    UINT playbackDeviceCount;
    UINT captureDeviceCount;
    UINT iPlaybackDevice;
    UINT iCaptureDevice;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(callback != NULL);

    /* Playback. */
    playbackDeviceCount = ((MA_PFN_waveOutGetNumDevs)pContext->winmm.waveOutGetNumDevs)();
    for (iPlaybackDevice = 0; iPlaybackDevice < playbackDeviceCount; ++iPlaybackDevice) {
        MMRESULT result;
        MA_WAVEOUTCAPS2A caps;

        MA_ZERO_OBJECT(&caps);

        result = ((MA_PFN_waveOutGetDevCapsA)pContext->winmm.waveOutGetDevCapsA)(iPlaybackDevice, (WAVEOUTCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            ma_device_info deviceInfo;

            MA_ZERO_OBJECT(&deviceInfo);
            deviceInfo.id.winmm = iPlaybackDevice;

            if (ma_context_get_device_info_from_WAVEOUTCAPS2(pContext, &caps, &deviceInfo) == MA_SUCCESS) {
                ma_bool32 cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
                if (cbResult == MA_FALSE) {
                    return MA_SUCCESS; /* Enumeration was stopped. */
                }
            }
        }
    }

    /* Capture. */
    captureDeviceCount = ((MA_PFN_waveInGetNumDevs)pContext->winmm.waveInGetNumDevs)();
    for (iCaptureDevice = 0; iCaptureDevice < captureDeviceCount; ++iCaptureDevice) {
        MMRESULT result;
        MA_WAVEINCAPS2A caps;

        MA_ZERO_OBJECT(&caps);

        result = ((MA_PFN_waveInGetDevCapsA)pContext->winmm.waveInGetDevCapsA)(iCaptureDevice, (WAVEINCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            ma_device_info deviceInfo;

            MA_ZERO_OBJECT(&deviceInfo);
            deviceInfo.id.winmm = iCaptureDevice;

            if (ma_context_get_device_info_from_WAVEINCAPS2(pContext, &caps, &deviceInfo) == MA_SUCCESS) {
                ma_bool32 cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
                if (cbResult == MA_FALSE) {
                    return MA_SUCCESS; /* Enumeration was stopped. */
                }
            }
        }
    }

    return MA_SUCCESS;
}

static ma_result ma_context_get_device_info__winmm(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    UINT winMMDeviceID;

    MA_ASSERT(pContext != NULL);

    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    winMMDeviceID = 0;
    if (pDeviceID != NULL) {
        winMMDeviceID = (UINT)pDeviceID->winmm;
    }

    pDeviceInfo->id.winmm = winMMDeviceID;

    if (deviceType == ma_device_type_playback) {
        MMRESULT result;
        MA_WAVEOUTCAPS2A caps;

        MA_ZERO_OBJECT(&caps);
        
        result = ((MA_PFN_waveOutGetDevCapsA)pContext->winmm.waveOutGetDevCapsA)(winMMDeviceID, (WAVEOUTCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            return ma_context_get_device_info_from_WAVEOUTCAPS2(pContext, &caps, pDeviceInfo);
        }
    } else {
        MMRESULT result;
        MA_WAVEINCAPS2A caps;

        MA_ZERO_OBJECT(&caps);
        
        result = ((MA_PFN_waveInGetDevCapsA)pContext->winmm.waveInGetDevCapsA)(winMMDeviceID, (WAVEINCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            return ma_context_get_device_info_from_WAVEINCAPS2(pContext, &caps, pDeviceInfo);
        }
    }

    return MA_NO_DEVICE;
}


static void ma_device_uninit__winmm(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((MA_PFN_waveInClose)pDevice->pContext->winmm.waveInClose)((HWAVEIN)pDevice->winmm.hDeviceCapture);
        CloseHandle((HANDLE)pDevice->winmm.hEventCapture);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ((MA_PFN_waveOutReset)pDevice->pContext->winmm.waveOutReset)((HWAVEOUT)pDevice->winmm.hDevicePlayback);
        ((MA_PFN_waveOutClose)pDevice->pContext->winmm.waveOutClose)((HWAVEOUT)pDevice->winmm.hDevicePlayback);
        CloseHandle((HANDLE)pDevice->winmm.hEventPlayback);
    }

    ma__free_from_callbacks(pDevice->winmm._pHeapData, &pDevice->pContext->allocationCallbacks);

    MA_ZERO_OBJECT(&pDevice->winmm);   /* Safety. */
}

static ma_result ma_device_init__winmm(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    const char* errorMsg = "";
    ma_result errorCode = MA_ERROR;
    ma_result result = MA_SUCCESS;
    ma_uint32 heapSize;
    UINT winMMDeviceIDPlayback = 0;
    UINT winMMDeviceIDCapture  = 0;
    ma_uint32 periodSizeInMilliseconds;

    MA_ASSERT(pDevice != NULL);
    MA_ZERO_OBJECT(&pDevice->winmm);

    if (pConfig->deviceType == ma_device_type_loopback) {
        return MA_DEVICE_TYPE_NOT_SUPPORTED;
    }

    /* No exlusive mode with WinMM. */
    if (((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.shareMode == ma_share_mode_exclusive) ||
        ((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.shareMode  == ma_share_mode_exclusive)) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    periodSizeInMilliseconds = pConfig->periodSizeInMilliseconds;
    if (periodSizeInMilliseconds == 0) {
        periodSizeInMilliseconds = ma_calculate_buffer_size_in_milliseconds_from_frames(pConfig->periodSizeInFrames, pConfig->sampleRate);
    }
    
    /* WinMM has horrible latency. */
    if (pDevice->usingDefaultBufferSize) {
        if (pConfig->performanceProfile == ma_performance_profile_low_latency) {
            periodSizeInMilliseconds =  40;
        } else {
            periodSizeInMilliseconds = 400;
        }
    }


    if (pConfig->playback.pDeviceID != NULL) {
        winMMDeviceIDPlayback = (UINT)pConfig->playback.pDeviceID->winmm;
    }
    if (pConfig->capture.pDeviceID != NULL) {
        winMMDeviceIDCapture = (UINT)pConfig->capture.pDeviceID->winmm;
    }

    /* The capture device needs to be initialized first. */
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        WAVEINCAPSA caps;
        WAVEFORMATEX wf;
        MMRESULT resultMM;

        /* We use an event to know when a new fragment needs to be enqueued. */
        pDevice->winmm.hEventCapture = (ma_handle)CreateEventW(NULL, TRUE, TRUE, NULL);
        if (pDevice->winmm.hEventCapture == NULL) {
            errorMsg = "[WinMM] Failed to create event for fragment enqueing for the capture device.", errorCode = ma_result_from_GetLastError(GetLastError());
            goto on_error;
        }

        /* The format should be based on the device's actual format. */
        if (((MA_PFN_waveInGetDevCapsA)pContext->winmm.waveInGetDevCapsA)(winMMDeviceIDCapture, &caps, sizeof(caps)) != MMSYSERR_NOERROR) {
            errorMsg = "[WinMM] Failed to retrieve internal device caps.", errorCode = MA_FORMAT_NOT_SUPPORTED;
            goto on_error;
        }

        result = ma_formats_flags_to_WAVEFORMATEX__winmm(caps.dwFormats, caps.wChannels, &wf);
        if (result != MA_SUCCESS) {
            errorMsg = "[WinMM] Could not find appropriate format for internal device.", errorCode = result;
            goto on_error;
        }

        resultMM = ((MA_PFN_waveInOpen)pDevice->pContext->winmm.waveInOpen)((LPHWAVEIN)&pDevice->winmm.hDeviceCapture, winMMDeviceIDCapture, &wf, (DWORD_PTR)pDevice->winmm.hEventCapture, (DWORD_PTR)pDevice, CALLBACK_EVENT | WAVE_ALLOWSYNC);
        if (resultMM != MMSYSERR_NOERROR) {
            errorMsg = "[WinMM] Failed to open capture device.", errorCode = MA_FAILED_TO_OPEN_BACKEND_DEVICE;
            goto on_error;
        }

        pDevice->capture.internalFormat             = ma_format_from_WAVEFORMATEX(&wf);
        pDevice->capture.internalChannels           = wf.nChannels;
        pDevice->capture.internalSampleRate         = wf.nSamplesPerSec;
        ma_get_standard_channel_map(ma_standard_channel_map_microsoft, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);
        pDevice->capture.internalPeriods            = pConfig->periods;
        pDevice->capture.internalPeriodSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(periodSizeInMilliseconds, pDevice->capture.internalSampleRate);
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        WAVEOUTCAPSA caps;
        WAVEFORMATEX wf;
        MMRESULT resultMM;

        /* We use an event to know when a new fragment needs to be enqueued. */
        pDevice->winmm.hEventPlayback = (ma_handle)CreateEvent(NULL, TRUE, TRUE, NULL);
        if (pDevice->winmm.hEventPlayback == NULL) {
            errorMsg = "[WinMM] Failed to create event for fragment enqueing for the playback device.", errorCode = ma_result_from_GetLastError(GetLastError());
            goto on_error;
        }

        /* The format should be based on the device's actual format. */
        if (((MA_PFN_waveOutGetDevCapsA)pContext->winmm.waveOutGetDevCapsA)(winMMDeviceIDPlayback, &caps, sizeof(caps)) != MMSYSERR_NOERROR) {
            errorMsg = "[WinMM] Failed to retrieve internal device caps.", errorCode = MA_FORMAT_NOT_SUPPORTED;
            goto on_error;
        }

        result = ma_formats_flags_to_WAVEFORMATEX__winmm(caps.dwFormats, caps.wChannels, &wf);
        if (result != MA_SUCCESS) {
            errorMsg = "[WinMM] Could not find appropriate format for internal device.", errorCode = result;
            goto on_error;
        }

        resultMM = ((MA_PFN_waveOutOpen)pContext->winmm.waveOutOpen)((LPHWAVEOUT)&pDevice->winmm.hDevicePlayback, winMMDeviceIDPlayback, &wf, (DWORD_PTR)pDevice->winmm.hEventPlayback, (DWORD_PTR)pDevice, CALLBACK_EVENT | WAVE_ALLOWSYNC);
        if (resultMM != MMSYSERR_NOERROR) {
            errorMsg = "[WinMM] Failed to open playback device.", errorCode = MA_FAILED_TO_OPEN_BACKEND_DEVICE;
            goto on_error;
        }

        pDevice->playback.internalFormat             = ma_format_from_WAVEFORMATEX(&wf);
        pDevice->playback.internalChannels           = wf.nChannels;
        pDevice->playback.internalSampleRate         = wf.nSamplesPerSec;
        ma_get_standard_channel_map(ma_standard_channel_map_microsoft, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);
        pDevice->playback.internalPeriods            = pConfig->periods;
        pDevice->playback.internalPeriodSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(periodSizeInMilliseconds, pDevice->playback.internalSampleRate);
    }

    /*
    The heap allocated data is allocated like so:
    
    [Capture WAVEHDRs][Playback WAVEHDRs][Capture Intermediary Buffer][Playback Intermediary Buffer]
    */
    heapSize = 0;
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        heapSize += sizeof(WAVEHDR)*pDevice->capture.internalPeriods + (pDevice->capture.internalPeriodSizeInFrames*pDevice->capture.internalPeriods*ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));
    }
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        heapSize += sizeof(WAVEHDR)*pDevice->playback.internalPeriods + (pDevice->playback.internalPeriodSizeInFrames*pDevice->playback.internalPeriods*ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));
    }

    pDevice->winmm._pHeapData = (ma_uint8*)ma__calloc_from_callbacks(heapSize, &pContext->allocationCallbacks);
    if (pDevice->winmm._pHeapData == NULL) {
        errorMsg = "[WinMM] Failed to allocate memory for the intermediary buffer.", errorCode = MA_OUT_OF_MEMORY;
        goto on_error;
    }

    MA_ZERO_MEMORY(pDevice->winmm._pHeapData, heapSize);

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_uint32 iPeriod;

        if (pConfig->deviceType == ma_device_type_capture) {
            pDevice->winmm.pWAVEHDRCapture            = pDevice->winmm._pHeapData;
            pDevice->winmm.pIntermediaryBufferCapture = pDevice->winmm._pHeapData + (sizeof(WAVEHDR)*(pDevice->capture.internalPeriods));
        } else {
            pDevice->winmm.pWAVEHDRCapture            = pDevice->winmm._pHeapData;
            pDevice->winmm.pIntermediaryBufferCapture = pDevice->winmm._pHeapData + (sizeof(WAVEHDR)*(pDevice->capture.internalPeriods + pDevice->playback.internalPeriods));
        }

        /* Prepare headers. */
        for (iPeriod = 0; iPeriod < pDevice->capture.internalPeriods; ++iPeriod) {
            ma_uint32 periodSizeInBytes = ma_get_period_size_in_bytes(pDevice->capture.internalPeriodSizeInFrames, pDevice->capture.internalFormat, pDevice->capture.internalChannels);

            ((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod].lpData         = (LPSTR)(pDevice->winmm.pIntermediaryBufferCapture + (periodSizeInBytes*iPeriod));
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod].dwBufferLength = periodSizeInBytes;
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod].dwFlags        = 0L;
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod].dwLoops        = 0L;
            ((MA_PFN_waveInPrepareHeader)pContext->winmm.waveInPrepareHeader)((HWAVEIN)pDevice->winmm.hDeviceCapture, &((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod], sizeof(WAVEHDR));

            /*
            The user data of the WAVEHDR structure is a single flag the controls whether or not it is ready for writing. Consider it to be named "isLocked". A value of 0 means
            it's unlocked and available for writing. A value of 1 means it's locked.
            */
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod].dwUser = 0;
        }
    }
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_uint32 iPeriod;

        if (pConfig->deviceType == ma_device_type_playback) {
            pDevice->winmm.pWAVEHDRPlayback            = pDevice->winmm._pHeapData;
            pDevice->winmm.pIntermediaryBufferPlayback = pDevice->winmm._pHeapData + (sizeof(WAVEHDR)*pDevice->playback.internalPeriods);
        } else {
            pDevice->winmm.pWAVEHDRPlayback            = pDevice->winmm._pHeapData + (sizeof(WAVEHDR)*(pDevice->capture.internalPeriods));
            pDevice->winmm.pIntermediaryBufferPlayback = pDevice->winmm._pHeapData + (sizeof(WAVEHDR)*(pDevice->capture.internalPeriods + pDevice->playback.internalPeriods)) + (pDevice->capture.internalPeriodSizeInFrames*pDevice->capture.internalPeriods*ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));
        }

        /* Prepare headers. */
        for (iPeriod = 0; iPeriod < pDevice->playback.internalPeriods; ++iPeriod) {
            ma_uint32 periodSizeInBytes = ma_get_period_size_in_bytes(pDevice->playback.internalPeriodSizeInFrames, pDevice->playback.internalFormat, pDevice->playback.internalChannels);

            ((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod].lpData         = (LPSTR)(pDevice->winmm.pIntermediaryBufferPlayback + (periodSizeInBytes*iPeriod));
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod].dwBufferLength = periodSizeInBytes;
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod].dwFlags        = 0L;
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod].dwLoops        = 0L;
            ((MA_PFN_waveOutPrepareHeader)pContext->winmm.waveOutPrepareHeader)((HWAVEOUT)pDevice->winmm.hDevicePlayback, &((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod], sizeof(WAVEHDR));

            /*
            The user data of the WAVEHDR structure is a single flag the controls whether or not it is ready for writing. Consider it to be named "isLocked". A value of 0 means
            it's unlocked and available for writing. A value of 1 means it's locked.
            */
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod].dwUser = 0;
        }
    }

    return MA_SUCCESS;

on_error:
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        if (pDevice->winmm.pWAVEHDRCapture != NULL) {
            ma_uint32 iPeriod;
            for (iPeriod = 0; iPeriod < pDevice->capture.internalPeriods; ++iPeriod) {
                ((MA_PFN_waveInUnprepareHeader)pContext->winmm.waveInUnprepareHeader)((HWAVEIN)pDevice->winmm.hDeviceCapture, &((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod], sizeof(WAVEHDR));
            }
        }

        ((MA_PFN_waveInClose)pContext->winmm.waveInClose)((HWAVEIN)pDevice->winmm.hDeviceCapture);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        if (pDevice->winmm.pWAVEHDRCapture != NULL) {
            ma_uint32 iPeriod;
            for (iPeriod = 0; iPeriod < pDevice->playback.internalPeriods; ++iPeriod) {
                ((MA_PFN_waveOutUnprepareHeader)pContext->winmm.waveOutUnprepareHeader)((HWAVEOUT)pDevice->winmm.hDevicePlayback, &((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod], sizeof(WAVEHDR));
            }
        }

        ((MA_PFN_waveOutClose)pContext->winmm.waveOutClose)((HWAVEOUT)pDevice->winmm.hDevicePlayback);
    }

    ma__free_from_callbacks(pDevice->winmm._pHeapData, &pContext->allocationCallbacks);
    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, errorMsg, errorCode);
}

static ma_result ma_device_stop__winmm(ma_device* pDevice)
{
    MMRESULT resultMM;

    MA_ASSERT(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        if (pDevice->winmm.hDeviceCapture == NULL) {
            return MA_INVALID_ARGS;
        }

        resultMM = ((MA_PFN_waveInReset)pDevice->pContext->winmm.waveInReset)((HWAVEIN)pDevice->winmm.hDeviceCapture);
        if (resultMM != MMSYSERR_NOERROR) {
            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] WARNING: Failed to reset capture device.", ma_result_from_MMRESULT(resultMM));
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ma_uint32 iPeriod;
        WAVEHDR* pWAVEHDR;

        if (pDevice->winmm.hDevicePlayback == NULL) {
            return MA_INVALID_ARGS;
        }

        /* We need to drain the device. To do this we just loop over each header and if it's locked just wait for the event. */
        pWAVEHDR = (WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback;
        for (iPeriod = 0; iPeriod < pDevice->playback.internalPeriods; iPeriod += 1) {
            if (pWAVEHDR[iPeriod].dwUser == 1) { /* 1 = locked. */
                if (WaitForSingleObject((HANDLE)pDevice->winmm.hEventPlayback, INFINITE) != WAIT_OBJECT_0) {
                    break;  /* An error occurred so just abandon ship and stop the device without draining. */
                }

                pWAVEHDR[iPeriod].dwUser = 0;
            }
        }

        resultMM = ((MA_PFN_waveOutReset)pDevice->pContext->winmm.waveOutReset)((HWAVEOUT)pDevice->winmm.hDevicePlayback);
        if (resultMM != MMSYSERR_NOERROR) {
            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] WARNING: Failed to reset playback device.", ma_result_from_MMRESULT(resultMM));
        }
    }

    return MA_SUCCESS;
}

static ma_result ma_device_write__winmm(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount, ma_uint32* pFramesWritten)
{
    ma_result result = MA_SUCCESS;
    MMRESULT resultMM;
    ma_uint32 totalFramesWritten;
    WAVEHDR* pWAVEHDR;

    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(pPCMFrames != NULL);

    if (pFramesWritten != NULL) {
        *pFramesWritten = 0;
    }

    pWAVEHDR = (WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback;

    /* Keep processing as much data as possible. */
    totalFramesWritten = 0;
    while (totalFramesWritten < frameCount) {
        /* If the current header has some space available we need to write part of it. */
        if (pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwUser == 0) { /* 0 = unlocked. */
            /*
            This header has room in it. We copy as much of it as we can. If we end up fully consuming the buffer we need to
            write it out and move on to the next iteration.
            */
            ma_uint32 bpf = ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
            ma_uint32 framesRemainingInHeader = (pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwBufferLength/bpf) - pDevice->winmm.headerFramesConsumedPlayback;

            ma_uint32 framesToCopy = ma_min(framesRemainingInHeader, (frameCount - totalFramesWritten));
            const void* pSrc = ma_offset_ptr(pPCMFrames, totalFramesWritten*bpf);
            void* pDst = ma_offset_ptr(pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].lpData, pDevice->winmm.headerFramesConsumedPlayback*bpf);
            MA_COPY_MEMORY(pDst, pSrc, framesToCopy*bpf);

            pDevice->winmm.headerFramesConsumedPlayback += framesToCopy;
            totalFramesWritten += framesToCopy;

            /* If we've consumed the buffer entirely we need to write it out to the device. */
            if (pDevice->winmm.headerFramesConsumedPlayback == (pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwBufferLength/bpf)) {
                pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwUser = 1;            /* 1 = locked. */
                pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwFlags &= ~WHDR_DONE; /* <-- Need to make sure the WHDR_DONE flag is unset. */

                /* Make sure the event is reset to a non-signaled state to ensure we don't prematurely return from WaitForSingleObject(). */
                ResetEvent((HANDLE)pDevice->winmm.hEventPlayback);

                /* The device will be started here. */
                resultMM = ((MA_PFN_waveOutWrite)pDevice->pContext->winmm.waveOutWrite)((HWAVEOUT)pDevice->winmm.hDevicePlayback, &pWAVEHDR[pDevice->winmm.iNextHeaderPlayback], sizeof(WAVEHDR));
                if (resultMM != MMSYSERR_NOERROR) {
                    result = ma_result_from_MMRESULT(resultMM);
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] waveOutWrite() failed.", result);
                    break;
                }

                /* Make sure we move to the next header. */
                pDevice->winmm.iNextHeaderPlayback = (pDevice->winmm.iNextHeaderPlayback + 1) % pDevice->playback.internalPeriods;
                pDevice->winmm.headerFramesConsumedPlayback = 0;
            }

            /* If at this point we have consumed the entire input buffer we can return. */
            MA_ASSERT(totalFramesWritten <= frameCount);
            if (totalFramesWritten == frameCount) {
                break;
            }

            /* Getting here means there's more to process. */
            continue;
        }

        /* Getting here means there isn't enough room in the buffer and we need to wait for one to become available. */
        if (WaitForSingleObject((HANDLE)pDevice->winmm.hEventPlayback, INFINITE) != WAIT_OBJECT_0) {
            result = MA_ERROR;
            break;
        }

        /* Something happened. If the next buffer has been marked as done we need to reset a bit of state. */
        if ((pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwFlags & WHDR_DONE) != 0) {
            pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwUser = 0;    /* 0 = unlocked (make it available for writing). */
            pDevice->winmm.headerFramesConsumedPlayback = 0;
        }

        /* If the device has been stopped we need to break. */
        if (ma_device__get_state(pDevice) != MA_STATE_STARTED) {
            break;
        }
    }

    if (pFramesWritten != NULL) {
        *pFramesWritten = totalFramesWritten;
    }

    return result;
}

static ma_result ma_device_read__winmm(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount, ma_uint32* pFramesRead)
{
    ma_result result = MA_SUCCESS;
    MMRESULT resultMM;
    ma_uint32 totalFramesRead;
    WAVEHDR* pWAVEHDR;

    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(pPCMFrames != NULL);

    if (pFramesRead != NULL) {
        *pFramesRead = 0;
    }

    pWAVEHDR = (WAVEHDR*)pDevice->winmm.pWAVEHDRCapture;

    /* Keep processing as much data as possible. */
    totalFramesRead = 0;
    while (totalFramesRead < frameCount) {
        /* If the current header has some space available we need to write part of it. */
        if (pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwUser == 0) { /* 0 = unlocked. */
            /* The buffer is available for reading. If we fully consume it we need to add it back to the buffer. */
            ma_uint32 bpf = ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
            ma_uint32 framesRemainingInHeader = (pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwBufferLength/bpf) - pDevice->winmm.headerFramesConsumedCapture;

            ma_uint32 framesToCopy = ma_min(framesRemainingInHeader, (frameCount - totalFramesRead));
            const void* pSrc = ma_offset_ptr(pWAVEHDR[pDevice->winmm.iNextHeaderCapture].lpData, pDevice->winmm.headerFramesConsumedCapture*bpf);
            void* pDst = ma_offset_ptr(pPCMFrames, totalFramesRead*bpf);
            MA_COPY_MEMORY(pDst, pSrc, framesToCopy*bpf);

            pDevice->winmm.headerFramesConsumedCapture += framesToCopy;
            totalFramesRead += framesToCopy;

            /* If we've consumed the buffer entirely we need to add it back to the device. */
            if (pDevice->winmm.headerFramesConsumedCapture == (pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwBufferLength/bpf)) {
                pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwUser = 1;            /* 1 = locked. */
                pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwFlags &= ~WHDR_DONE; /* <-- Need to make sure the WHDR_DONE flag is unset. */

                /* Make sure the event is reset to a non-signaled state to ensure we don't prematurely return from WaitForSingleObject(). */
                ResetEvent((HANDLE)pDevice->winmm.hEventCapture);

                /* The device will be started here. */
                resultMM = ((MA_PFN_waveInAddBuffer)pDevice->pContext->winmm.waveInAddBuffer)((HWAVEIN)pDevice->winmm.hDeviceCapture, &((LPWAVEHDR)pDevice->winmm.pWAVEHDRCapture)[pDevice->winmm.iNextHeaderCapture], sizeof(WAVEHDR));
                if (resultMM != MMSYSERR_NOERROR) {
                    result = ma_result_from_MMRESULT(resultMM);
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] waveInAddBuffer() failed.", result);
                    break;
                }

                /* Make sure we move to the next header. */
                pDevice->winmm.iNextHeaderCapture = (pDevice->winmm.iNextHeaderCapture + 1) % pDevice->capture.internalPeriods;
                pDevice->winmm.headerFramesConsumedCapture = 0;
            }

            /* If at this point we have filled the entire input buffer we can return. */
            MA_ASSERT(totalFramesRead <= frameCount);
            if (totalFramesRead == frameCount) {
                break;
            }

            /* Getting here means there's more to process. */
            continue;
        }

        /* Getting here means there isn't enough any data left to send to the client which means we need to wait for more. */
        if (WaitForSingleObject((HANDLE)pDevice->winmm.hEventCapture, INFINITE) != WAIT_OBJECT_0) {
            result = MA_ERROR;
            break;
        }

        /* Something happened. If the next buffer has been marked as done we need to reset a bit of state. */
        if ((pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwFlags & WHDR_DONE) != 0) {
            pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwUser = 0;    /* 0 = unlocked (make it available for reading). */
            pDevice->winmm.headerFramesConsumedCapture = 0;
        }

        /* If the device has been stopped we need to break. */
        if (ma_device__get_state(pDevice) != MA_STATE_STARTED) {
            break;
        }
    }

    if (pFramesRead != NULL) {
        *pFramesRead = totalFramesRead;
    }

    return result;
}

static ma_result ma_device_main_loop__winmm(ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;
    ma_bool32 exitLoop = MA_FALSE;
    ma_uint8  capturedDeviceData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
    ma_uint8  playbackDeviceData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
    ma_uint32 capturedDeviceDataCapInFrames = sizeof(capturedDeviceData) / ma_get_bytes_per_frame(pDevice->capture.internalFormat,  pDevice->capture.internalChannels);
    ma_uint32 playbackDeviceDataCapInFrames = sizeof(playbackDeviceData) / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
    
    MA_ASSERT(pDevice != NULL);

    /* The capture device needs to be started immediately. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        MMRESULT resultMM;
        WAVEHDR* pWAVEHDR;
        ma_uint32 iPeriod;

        pWAVEHDR = (WAVEHDR*)pDevice->winmm.pWAVEHDRCapture;

        /* Make sure the event is reset to a non-signaled state to ensure we don't prematurely return from WaitForSingleObject(). */
        ResetEvent((HANDLE)pDevice->winmm.hEventCapture);

        /* To start the device we attach all of the buffers and then start it. As the buffers are filled with data we will get notifications. */
        for (iPeriod = 0; iPeriod < pDevice->capture.internalPeriods; ++iPeriod) {
            resultMM = ((MA_PFN_waveInAddBuffer)pDevice->pContext->winmm.waveInAddBuffer)((HWAVEIN)pDevice->winmm.hDeviceCapture, &((LPWAVEHDR)pDevice->winmm.pWAVEHDRCapture)[iPeriod], sizeof(WAVEHDR));
            if (resultMM != MMSYSERR_NOERROR) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] Failed to attach input buffers to capture device in preparation for capture.", ma_result_from_MMRESULT(resultMM));
            }

            /* Make sure all of the buffers start out locked. We don't want to access them until the backend tells us we can. */
            pWAVEHDR[iPeriod].dwUser = 1;   /* 1 = locked. */
        }

        /* Capture devices need to be explicitly started, unlike playback devices. */
        resultMM = ((MA_PFN_waveInStart)pDevice->pContext->winmm.waveInStart)((HWAVEIN)pDevice->winmm.hDeviceCapture);
        if (resultMM != MMSYSERR_NOERROR) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] Failed to start backend device.", ma_result_from_MMRESULT(resultMM));
        }
    }


    while (ma_device__get_state(pDevice) == MA_STATE_STARTED && !exitLoop) {
        switch (pDevice->type)
        {
            case ma_device_type_duplex:
            {
                /* The process is: device_read -> convert -> callback -> convert -> device_write */
                ma_uint32 totalCapturedDeviceFramesProcessed = 0;
                ma_uint32 capturedDevicePeriodSizeInFrames = ma_min(pDevice->capture.internalPeriodSizeInFrames, pDevice->playback.internalPeriodSizeInFrames);
                    
                while (totalCapturedDeviceFramesProcessed < capturedDevicePeriodSizeInFrames) {
                    ma_uint32 capturedDeviceFramesRemaining;
                    ma_uint32 capturedDeviceFramesProcessed;
                    ma_uint32 capturedDeviceFramesToProcess;
                    ma_uint32 capturedDeviceFramesToTryProcessing = capturedDevicePeriodSizeInFrames - totalCapturedDeviceFramesProcessed;
                    if (capturedDeviceFramesToTryProcessing > capturedDeviceDataCapInFrames) {
                        capturedDeviceFramesToTryProcessing = capturedDeviceDataCapInFrames;
                    }

                    result = ma_device_read__winmm(pDevice, capturedDeviceData, capturedDeviceFramesToTryProcessing, &capturedDeviceFramesToProcess);
                    if (result != MA_SUCCESS) {
                        exitLoop = MA_TRUE;
                        break;
                    }

                    capturedDeviceFramesRemaining = capturedDeviceFramesToProcess;
                    capturedDeviceFramesProcessed = 0;

                    for (;;) {
                        ma_uint8  capturedClientData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                        ma_uint8  playbackClientData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                        ma_uint32 capturedClientDataCapInFrames = sizeof(capturedClientData) / ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
                        ma_uint32 playbackClientDataCapInFrames = sizeof(playbackClientData) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
                        ma_uint64 capturedClientFramesToProcessThisIteration = ma_min(capturedClientDataCapInFrames, playbackClientDataCapInFrames);
                        ma_uint64 capturedDeviceFramesToProcessThisIteration = capturedDeviceFramesRemaining;
                        ma_uint8* pRunningCapturedDeviceFrames = ma_offset_ptr(capturedDeviceData, capturedDeviceFramesProcessed * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));

                        /* Convert capture data from device format to client format. */
                        result = ma_data_converter_process_pcm_frames(&pDevice->capture.converter, pRunningCapturedDeviceFrames, &capturedDeviceFramesToProcessThisIteration, capturedClientData, &capturedClientFramesToProcessThisIteration);
                        if (result != MA_SUCCESS) {
                            break;
                        }

                        /*
                        If we weren't able to generate any output frames it must mean we've exhaused all of our input. The only time this would not be the case is if capturedClientData was too small
                        which should never be the case when it's of the size MA_DATA_CONVERTER_STACK_BUFFER_SIZE.
                        */
                        if (capturedClientFramesToProcessThisIteration == 0) {
                            break;
                        }

                        ma_device__on_data(pDevice, playbackClientData, capturedClientData, (ma_uint32)capturedClientFramesToProcessThisIteration);    /* Safe cast .*/

                        capturedDeviceFramesProcessed += (ma_uint32)capturedDeviceFramesToProcessThisIteration; /* Safe cast. */
                        capturedDeviceFramesRemaining -= (ma_uint32)capturedDeviceFramesToProcessThisIteration; /* Safe cast. */

                        /* At this point the playbackClientData buffer should be holding data that needs to be written to the device. */
                        for (;;) {
                            ma_uint64 convertedClientFrameCount = capturedClientFramesToProcessThisIteration;
                            ma_uint64 convertedDeviceFrameCount = playbackDeviceDataCapInFrames;
                            result = ma_data_converter_process_pcm_frames(&pDevice->playback.converter, playbackClientData, &convertedClientFrameCount, playbackDeviceData, &convertedDeviceFrameCount);
                            if (result != MA_SUCCESS) {
                                break;
                            }

                            result = ma_device_write__winmm(pDevice, playbackDeviceData, (ma_uint32)convertedDeviceFrameCount, NULL);  /* Safe cast. */
                            if (result != MA_SUCCESS) {
                                exitLoop = MA_TRUE;
                                break;
                            }

                            capturedClientFramesToProcessThisIteration -= (ma_uint32)convertedClientFrameCount;  /* Safe cast. */
                            if (capturedClientFramesToProcessThisIteration == 0) {
                                break;
                            }
                        }

                        /* In case an error happened from ma_device_write__winmm()... */
                        if (result != MA_SUCCESS) {
                            exitLoop = MA_TRUE;
                            break;
                        }
                    }

                    totalCapturedDeviceFramesProcessed += capturedDeviceFramesProcessed;
                }
            } break;

            case ma_device_type_capture:
            {
                /* We read in chunks of the period size, but use a stack allocated buffer for the intermediary. */
                ma_uint32 periodSizeInFrames = pDevice->capture.internalPeriodSizeInFrames;
                ma_uint32 framesReadThisPeriod = 0;
                while (framesReadThisPeriod < periodSizeInFrames) {
                    ma_uint32 framesRemainingInPeriod = periodSizeInFrames - framesReadThisPeriod;
                    ma_uint32 framesProcessed;
                    ma_uint32 framesToReadThisIteration = framesRemainingInPeriod;
                    if (framesToReadThisIteration > capturedDeviceDataCapInFrames) {
                        framesToReadThisIteration = capturedDeviceDataCapInFrames;
                    }

                    result = ma_device_read__winmm(pDevice, capturedDeviceData, framesToReadThisIteration, &framesProcessed);
                    if (result != MA_SUCCESS) {
                        exitLoop = MA_TRUE;
                        break;
                    }

                    ma_device__send_frames_to_client(pDevice, framesProcessed, capturedDeviceData);

                    framesReadThisPeriod += framesProcessed;
                }
            } break;

            case ma_device_type_playback:
            {
                /* We write in chunks of the period size, but use a stack allocated buffer for the intermediary. */
                ma_uint32 periodSizeInFrames = pDevice->playback.internalPeriodSizeInFrames;
                ma_uint32 framesWrittenThisPeriod = 0;
                while (framesWrittenThisPeriod < periodSizeInFrames) {
                    ma_uint32 framesRemainingInPeriod = periodSizeInFrames - framesWrittenThisPeriod;
                    ma_uint32 framesProcessed;
                    ma_uint32 framesToWriteThisIteration = framesRemainingInPeriod;
                    if (framesToWriteThisIteration > playbackDeviceDataCapInFrames) {
                        framesToWriteThisIteration = playbackDeviceDataCapInFrames;
                    }

                    ma_device__read_frames_from_client(pDevice, framesToWriteThisIteration, playbackDeviceData);

                    result = ma_device_write__winmm(pDevice, playbackDeviceData, framesToWriteThisIteration, &framesProcessed);
                    if (result != MA_SUCCESS) {
                        exitLoop = MA_TRUE;
                        break;
                    }

                    framesWrittenThisPeriod += framesProcessed;
                }
            } break;

            /* To silence a warning. Will never hit this. */
            case ma_device_type_loopback:
            default: break;
        }
    }


    /* Here is where the device is started. */
    ma_device_stop__winmm(pDevice);

    return result;
}

static ma_result ma_context_uninit__winmm(ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pContext->backend == ma_backend_winmm);

    ma_dlclose(pContext, pContext->winmm.hWinMM);
    return MA_SUCCESS;
}

static ma_result ma_context_init__winmm(const ma_context_config* pConfig, ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);

    (void)pConfig;

    pContext->winmm.hWinMM = ma_dlopen(pContext, "winmm.dll");
    if (pContext->winmm.hWinMM == NULL) {
        return MA_NO_BACKEND;
    }

    pContext->winmm.waveOutGetNumDevs      = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveOutGetNumDevs");
    pContext->winmm.waveOutGetDevCapsA     = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveOutGetDevCapsA");
    pContext->winmm.waveOutOpen            = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveOutOpen");
    pContext->winmm.waveOutClose           = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveOutClose");
    pContext->winmm.waveOutPrepareHeader   = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveOutPrepareHeader");
    pContext->winmm.waveOutUnprepareHeader = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveOutUnprepareHeader");
    pContext->winmm.waveOutWrite           = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveOutWrite");
    pContext->winmm.waveOutReset           = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveOutReset");
    pContext->winmm.waveInGetNumDevs       = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveInGetNumDevs");
    pContext->winmm.waveInGetDevCapsA      = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveInGetDevCapsA");
    pContext->winmm.waveInOpen             = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveInOpen");
    pContext->winmm.waveInClose            = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveInClose");
    pContext->winmm.waveInPrepareHeader    = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveInPrepareHeader");
    pContext->winmm.waveInUnprepareHeader  = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveInUnprepareHeader");
    pContext->winmm.waveInAddBuffer        = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveInAddBuffer");
    pContext->winmm.waveInStart            = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveInStart");
    pContext->winmm.waveInReset            = ma_dlsym(pContext, pContext->winmm.hWinMM, "waveInReset");

    pContext->onUninit         = ma_context_uninit__winmm;
    pContext->onDeviceIDEqual  = ma_context_is_device_id_equal__winmm;
    pContext->onEnumDevices    = ma_context_enumerate_devices__winmm;
    pContext->onGetDeviceInfo  = ma_context_get_device_info__winmm;
    pContext->onDeviceInit     = ma_device_init__winmm;
    pContext->onDeviceUninit   = ma_device_uninit__winmm;
    pContext->onDeviceStart    = NULL; /* Not used with synchronous backends. */
    pContext->onDeviceStop     = NULL; /* Not used with synchronous backends. */
    pContext->onDeviceMainLoop = ma_device_main_loop__winmm;

    return MA_SUCCESS;
}
#endif




/******************************************************************************

ALSA Backend

******************************************************************************/
#ifdef MA_HAS_ALSA

#ifdef MA_NO_RUNTIME_LINKING

/* asoundlib.h marks some functions with "inline" which isn't always supported. Need to emulate it. */
#if !defined(__cplusplus)
    #if defined(__STRICT_ANSI__)
        #if !defined(inline)
            #define inline __inline__ __attribute__((always_inline))
            #define MA_INLINE_DEFINED
        #endif
    #endif
#endif
#include <alsa/asoundlib.h>
#if defined(MA_INLINE_DEFINED)
    #undef inline
    #undef MA_INLINE_DEFINED
#endif

typedef snd_pcm_uframes_t                       ma_snd_pcm_uframes_t;
typedef snd_pcm_sframes_t                       ma_snd_pcm_sframes_t;
typedef snd_pcm_stream_t                        ma_snd_pcm_stream_t;
typedef snd_pcm_format_t                        ma_snd_pcm_format_t;
typedef snd_pcm_access_t                        ma_snd_pcm_access_t;
typedef snd_pcm_t                               ma_snd_pcm_t;
typedef snd_pcm_hw_params_t                     ma_snd_pcm_hw_params_t;
typedef snd_pcm_sw_params_t                     ma_snd_pcm_sw_params_t;
typedef snd_pcm_format_mask_t                   ma_snd_pcm_format_mask_t;
typedef snd_pcm_info_t                          ma_snd_pcm_info_t;
typedef snd_pcm_channel_area_t                  ma_snd_pcm_channel_area_t;
typedef snd_pcm_chmap_t                         ma_snd_pcm_chmap_t;
typedef snd_pcm_state_t                         ma_snd_pcm_state_t;

/* snd_pcm_stream_t */
#define MA_SND_PCM_STREAM_PLAYBACK              SND_PCM_STREAM_PLAYBACK
#define MA_SND_PCM_STREAM_CAPTURE               SND_PCM_STREAM_CAPTURE

/* snd_pcm_format_t */
#define MA_SND_PCM_FORMAT_UNKNOWN               SND_PCM_FORMAT_UNKNOWN
#define MA_SND_PCM_FORMAT_U8                    SND_PCM_FORMAT_U8
#define MA_SND_PCM_FORMAT_S16_LE                SND_PCM_FORMAT_S16_LE
#define MA_SND_PCM_FORMAT_S16_BE                SND_PCM_FORMAT_S16_BE
#define MA_SND_PCM_FORMAT_S24_LE                SND_PCM_FORMAT_S24_LE
#define MA_SND_PCM_FORMAT_S24_BE                SND_PCM_FORMAT_S24_BE
#define MA_SND_PCM_FORMAT_S32_LE                SND_PCM_FORMAT_S32_LE
#define MA_SND_PCM_FORMAT_S32_BE                SND_PCM_FORMAT_S32_BE
#define MA_SND_PCM_FORMAT_FLOAT_LE              SND_PCM_FORMAT_FLOAT_LE
#define MA_SND_PCM_FORMAT_FLOAT_BE              SND_PCM_FORMAT_FLOAT_BE
#define MA_SND_PCM_FORMAT_FLOAT64_LE            SND_PCM_FORMAT_FLOAT64_LE
#define MA_SND_PCM_FORMAT_FLOAT64_BE            SND_PCM_FORMAT_FLOAT64_BE
#define MA_SND_PCM_FORMAT_MU_LAW                SND_PCM_FORMAT_MU_LAW
#define MA_SND_PCM_FORMAT_A_LAW                 SND_PCM_FORMAT_A_LAW
#define MA_SND_PCM_FORMAT_S24_3LE               SND_PCM_FORMAT_S24_3LE
#define MA_SND_PCM_FORMAT_S24_3BE               SND_PCM_FORMAT_S24_3BE

/* ma_snd_pcm_access_t */
#define MA_SND_PCM_ACCESS_MMAP_INTERLEAVED      SND_PCM_ACCESS_MMAP_INTERLEAVED
#define MA_SND_PCM_ACCESS_MMAP_NONINTERLEAVED   SND_PCM_ACCESS_MMAP_NONINTERLEAVED
#define MA_SND_PCM_ACCESS_MMAP_COMPLEX          SND_PCM_ACCESS_MMAP_COMPLEX
#define MA_SND_PCM_ACCESS_RW_INTERLEAVED        SND_PCM_ACCESS_RW_INTERLEAVED
#define MA_SND_PCM_ACCESS_RW_NONINTERLEAVED     SND_PCM_ACCESS_RW_NONINTERLEAVED

/* Channel positions. */
#define MA_SND_CHMAP_UNKNOWN                    SND_CHMAP_UNKNOWN
#define MA_SND_CHMAP_NA                         SND_CHMAP_NA
#define MA_SND_CHMAP_MONO                       SND_CHMAP_MONO
#define MA_SND_CHMAP_FL                         SND_CHMAP_FL
#define MA_SND_CHMAP_FR                         SND_CHMAP_FR
#define MA_SND_CHMAP_RL                         SND_CHMAP_RL
#define MA_SND_CHMAP_RR                         SND_CHMAP_RR
#define MA_SND_CHMAP_FC                         SND_CHMAP_FC
#define MA_SND_CHMAP_LFE                        SND_CHMAP_LFE
#define MA_SND_CHMAP_SL                         SND_CHMAP_SL
#define MA_SND_CHMAP_SR                         SND_CHMAP_SR
#define MA_SND_CHMAP_RC                         SND_CHMAP_RC
#define MA_SND_CHMAP_FLC                        SND_CHMAP_FLC
#define MA_SND_CHMAP_FRC                        SND_CHMAP_FRC
#define MA_SND_CHMAP_RLC                        SND_CHMAP_RLC
#define MA_SND_CHMAP_RRC                        SND_CHMAP_RRC
#define MA_SND_CHMAP_FLW                        SND_CHMAP_FLW
#define MA_SND_CHMAP_FRW                        SND_CHMAP_FRW
#define MA_SND_CHMAP_FLH                        SND_CHMAP_FLH
#define MA_SND_CHMAP_FCH                        SND_CHMAP_FCH
#define MA_SND_CHMAP_FRH                        SND_CHMAP_FRH
#define MA_SND_CHMAP_TC                         SND_CHMAP_TC
#define MA_SND_CHMAP_TFL                        SND_CHMAP_TFL
#define MA_SND_CHMAP_TFR                        SND_CHMAP_TFR
#define MA_SND_CHMAP_TFC                        SND_CHMAP_TFC
#define MA_SND_CHMAP_TRL                        SND_CHMAP_TRL
#define MA_SND_CHMAP_TRR                        SND_CHMAP_TRR
#define MA_SND_CHMAP_TRC                        SND_CHMAP_TRC
#define MA_SND_CHMAP_TFLC                       SND_CHMAP_TFLC
#define MA_SND_CHMAP_TFRC                       SND_CHMAP_TFRC
#define MA_SND_CHMAP_TSL                        SND_CHMAP_TSL
#define MA_SND_CHMAP_TSR                        SND_CHMAP_TSR
#define MA_SND_CHMAP_LLFE                       SND_CHMAP_LLFE
#define MA_SND_CHMAP_RLFE                       SND_CHMAP_RLFE
#define MA_SND_CHMAP_BC                         SND_CHMAP_BC
#define MA_SND_CHMAP_BLC                        SND_CHMAP_BLC
#define MA_SND_CHMAP_BRC                        SND_CHMAP_BRC

/* Open mode flags. */
#define MA_SND_PCM_NO_AUTO_RESAMPLE             SND_PCM_NO_AUTO_RESAMPLE
#define MA_SND_PCM_NO_AUTO_CHANNELS             SND_PCM_NO_AUTO_CHANNELS
#define MA_SND_PCM_NO_AUTO_FORMAT               SND_PCM_NO_AUTO_FORMAT
#else
#include <errno.h>  /* For EPIPE, etc. */
typedef unsigned long                           ma_snd_pcm_uframes_t;
typedef long                                    ma_snd_pcm_sframes_t;
typedef int                                     ma_snd_pcm_stream_t;
typedef int                                     ma_snd_pcm_format_t;
typedef int                                     ma_snd_pcm_access_t;
typedef int                                     ma_snd_pcm_state_t;
typedef struct ma_snd_pcm_t                     ma_snd_pcm_t;
typedef struct ma_snd_pcm_hw_params_t           ma_snd_pcm_hw_params_t;
typedef struct ma_snd_pcm_sw_params_t           ma_snd_pcm_sw_params_t;
typedef struct ma_snd_pcm_format_mask_t         ma_snd_pcm_format_mask_t;
typedef struct ma_snd_pcm_info_t                ma_snd_pcm_info_t;
typedef struct
{
    void* addr;
    unsigned int first;
    unsigned int step;
} ma_snd_pcm_channel_area_t;
typedef struct
{
    unsigned int channels;
    unsigned int pos[1];
} ma_snd_pcm_chmap_t;

/* snd_pcm_state_t */
#define MA_SND_PCM_STATE_OPEN                  0
#define MA_SND_PCM_STATE_SETUP                 1
#define MA_SND_PCM_STATE_PREPARED              2
#define MA_SND_PCM_STATE_RUNNING               3
#define MA_SND_PCM_STATE_XRUN                  4
#define MA_SND_PCM_STATE_DRAINING              5
#define MA_SND_PCM_STATE_PAUSED                6
#define MA_SND_PCM_STATE_SUSPENDED             7
#define MA_SND_PCM_STATE_DISCONNECTED          8

/* snd_pcm_stream_t */
#define MA_SND_PCM_STREAM_PLAYBACK             0
#define MA_SND_PCM_STREAM_CAPTURE              1

/* snd_pcm_format_t */
#define MA_SND_PCM_FORMAT_UNKNOWN              -1
#define MA_SND_PCM_FORMAT_U8                   1
#define MA_SND_PCM_FORMAT_S16_LE               2
#define MA_SND_PCM_FORMAT_S16_BE               3
#define MA_SND_PCM_FORMAT_S24_LE               6
#define MA_SND_PCM_FORMAT_S24_BE               7
#define MA_SND_PCM_FORMAT_S32_LE               10
#define MA_SND_PCM_FORMAT_S32_BE               11
#define MA_SND_PCM_FORMAT_FLOAT_LE             14
#define MA_SND_PCM_FORMAT_FLOAT_BE             15
#define MA_SND_PCM_FORMAT_FLOAT64_LE           16
#define MA_SND_PCM_FORMAT_FLOAT64_BE           17
#define MA_SND_PCM_FORMAT_MU_LAW               20
#define MA_SND_PCM_FORMAT_A_LAW                21
#define MA_SND_PCM_FORMAT_S24_3LE              32
#define MA_SND_PCM_FORMAT_S24_3BE              33

/* snd_pcm_access_t */
#define MA_SND_PCM_ACCESS_MMAP_INTERLEAVED     0
#define MA_SND_PCM_ACCESS_MMAP_NONINTERLEAVED  1
#define MA_SND_PCM_ACCESS_MMAP_COMPLEX         2
#define MA_SND_PCM_ACCESS_RW_INTERLEAVED       3
#define MA_SND_PCM_ACCESS_RW_NONINTERLEAVED    4

/* Channel positions. */
#define MA_SND_CHMAP_UNKNOWN                   0
#define MA_SND_CHMAP_NA                        1
#define MA_SND_CHMAP_MONO                      2
#define MA_SND_CHMAP_FL                        3
#define MA_SND_CHMAP_FR                        4
#define MA_SND_CHMAP_RL                        5
#define MA_SND_CHMAP_RR                        6
#define MA_SND_CHMAP_FC                        7
#define MA_SND_CHMAP_LFE                       8
#define MA_SND_CHMAP_SL                        9
#define MA_SND_CHMAP_SR                        10
#define MA_SND_CHMAP_RC                        11
#define MA_SND_CHMAP_FLC                       12
#define MA_SND_CHMAP_FRC                       13
#define MA_SND_CHMAP_RLC                       14
#define MA_SND_CHMAP_RRC                       15
#define MA_SND_CHMAP_FLW                       16
#define MA_SND_CHMAP_FRW                       17
#define MA_SND_CHMAP_FLH                       18
#define MA_SND_CHMAP_FCH                       19
#define MA_SND_CHMAP_FRH                       20
#define MA_SND_CHMAP_TC                        21
#define MA_SND_CHMAP_TFL                       22
#define MA_SND_CHMAP_TFR                       23
#define MA_SND_CHMAP_TFC                       24
#define MA_SND_CHMAP_TRL                       25
#define MA_SND_CHMAP_TRR                       26
#define MA_SND_CHMAP_TRC                       27
#define MA_SND_CHMAP_TFLC                      28
#define MA_SND_CHMAP_TFRC                      29
#define MA_SND_CHMAP_TSL                       30
#define MA_SND_CHMAP_TSR                       31
#define MA_SND_CHMAP_LLFE                      32
#define MA_SND_CHMAP_RLFE                      33
#define MA_SND_CHMAP_BC                        34
#define MA_SND_CHMAP_BLC                       35
#define MA_SND_CHMAP_BRC                       36

/* Open mode flags. */
#define MA_SND_PCM_NO_AUTO_RESAMPLE            0x00010000
#define MA_SND_PCM_NO_AUTO_CHANNELS            0x00020000
#define MA_SND_PCM_NO_AUTO_FORMAT              0x00040000
#endif

typedef int                  (* ma_snd_pcm_open_proc)                          (ma_snd_pcm_t **pcm, const char *name, ma_snd_pcm_stream_t stream, int mode);
typedef int                  (* ma_snd_pcm_close_proc)                         (ma_snd_pcm_t *pcm);
typedef size_t               (* ma_snd_pcm_hw_params_sizeof_proc)              (void);
typedef int                  (* ma_snd_pcm_hw_params_any_proc)                 (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params);
typedef int                  (* ma_snd_pcm_hw_params_set_format_proc)          (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, ma_snd_pcm_format_t val);
typedef int                  (* ma_snd_pcm_hw_params_set_format_first_proc)    (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, ma_snd_pcm_format_t *format);
typedef void                 (* ma_snd_pcm_hw_params_get_format_mask_proc)     (ma_snd_pcm_hw_params_t *params, ma_snd_pcm_format_mask_t *mask);
typedef int                  (* ma_snd_pcm_hw_params_set_channels_near_proc)   (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                  (* ma_snd_pcm_hw_params_set_rate_resample_proc)   (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, unsigned int val);
typedef int                  (* ma_snd_pcm_hw_params_set_rate_near_proc)       (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_set_buffer_size_near_proc)(ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, ma_snd_pcm_uframes_t *val);
typedef int                  (* ma_snd_pcm_hw_params_set_periods_near_proc)    (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_set_access_proc)          (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, ma_snd_pcm_access_t _access);
typedef int                  (* ma_snd_pcm_hw_params_get_format_proc)          (const ma_snd_pcm_hw_params_t *params, ma_snd_pcm_format_t *format);
typedef int                  (* ma_snd_pcm_hw_params_get_channels_proc)        (const ma_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                  (* ma_snd_pcm_hw_params_get_channels_min_proc)    (const ma_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                  (* ma_snd_pcm_hw_params_get_channels_max_proc)    (const ma_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                  (* ma_snd_pcm_hw_params_get_rate_proc)            (const ma_snd_pcm_hw_params_t *params, unsigned int *rate, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_get_rate_min_proc)        (const ma_snd_pcm_hw_params_t *params, unsigned int *rate, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_get_rate_max_proc)        (const ma_snd_pcm_hw_params_t *params, unsigned int *rate, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_get_buffer_size_proc)     (const ma_snd_pcm_hw_params_t *params, ma_snd_pcm_uframes_t *val);
typedef int                  (* ma_snd_pcm_hw_params_get_periods_proc)         (const ma_snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_get_access_proc)          (const ma_snd_pcm_hw_params_t *params, ma_snd_pcm_access_t *_access);
typedef int                  (* ma_snd_pcm_hw_params_proc)                     (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params);
typedef size_t               (* ma_snd_pcm_sw_params_sizeof_proc)              (void);
typedef int                  (* ma_snd_pcm_sw_params_current_proc)             (ma_snd_pcm_t *pcm, ma_snd_pcm_sw_params_t *params);
typedef int                  (* ma_snd_pcm_sw_params_get_boundary_proc)        (const ma_snd_pcm_sw_params_t *params, ma_snd_pcm_uframes_t* val);
typedef int                  (* ma_snd_pcm_sw_params_set_avail_min_proc)       (ma_snd_pcm_t *pcm, ma_snd_pcm_sw_params_t *params, ma_snd_pcm_uframes_t val);
typedef int                  (* ma_snd_pcm_sw_params_set_start_threshold_proc) (ma_snd_pcm_t *pcm, ma_snd_pcm_sw_params_t *params, ma_snd_pcm_uframes_t val);
typedef int                  (* ma_snd_pcm_sw_params_set_stop_threshold_proc)  (ma_snd_pcm_t *pcm, ma_snd_pcm_sw_params_t *params, ma_snd_pcm_uframes_t val);
typedef int                  (* ma_snd_pcm_sw_params_proc)                     (ma_snd_pcm_t *pcm, ma_snd_pcm_sw_params_t *params);
typedef size_t               (* ma_snd_pcm_format_mask_sizeof_proc)            (void);
typedef int                  (* ma_snd_pcm_format_mask_test_proc)              (const ma_snd_pcm_format_mask_t *mask, ma_snd_pcm_format_t val);
typedef ma_snd_pcm_chmap_t * (* ma_snd_pcm_get_chmap_proc)                     (ma_snd_pcm_t *pcm);
typedef ma_snd_pcm_state_t   (* ma_snd_pcm_state_proc)                         (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_prepare_proc)                       (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_start_proc)                         (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_drop_proc)                          (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_drain_proc)                         (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_device_name_hint_proc)                  (int card, const char *iface, void ***hints);
typedef char *               (* ma_snd_device_name_get_hint_proc)              (const void *hint, const char *id);
typedef int                  (* ma_snd_card_get_index_proc)                    (const char *name);
typedef int                  (* ma_snd_device_name_free_hint_proc)             (void **hints);
typedef int                  (* ma_snd_pcm_mmap_begin_proc)                    (ma_snd_pcm_t *pcm, const ma_snd_pcm_channel_area_t **areas, ma_snd_pcm_uframes_t *offset, ma_snd_pcm_uframes_t *frames);
typedef ma_snd_pcm_sframes_t (* ma_snd_pcm_mmap_commit_proc)                   (ma_snd_pcm_t *pcm, ma_snd_pcm_uframes_t offset, ma_snd_pcm_uframes_t frames);
typedef int                  (* ma_snd_pcm_recover_proc)                       (ma_snd_pcm_t *pcm, int err, int silent);
typedef ma_snd_pcm_sframes_t (* ma_snd_pcm_readi_proc)                         (ma_snd_pcm_t *pcm, void *buffer, ma_snd_pcm_uframes_t size);
typedef ma_snd_pcm_sframes_t (* ma_snd_pcm_writei_proc)                        (ma_snd_pcm_t *pcm, const void *buffer, ma_snd_pcm_uframes_t size);
typedef ma_snd_pcm_sframes_t (* ma_snd_pcm_avail_proc)                         (ma_snd_pcm_t *pcm);
typedef ma_snd_pcm_sframes_t (* ma_snd_pcm_avail_update_proc)                  (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_wait_proc)                          (ma_snd_pcm_t *pcm, int timeout);
typedef int                  (* ma_snd_pcm_info_proc)                          (ma_snd_pcm_t *pcm, ma_snd_pcm_info_t* info);
typedef size_t               (* ma_snd_pcm_info_sizeof_proc)                   (void);
typedef const char*          (* ma_snd_pcm_info_get_name_proc)                 (const ma_snd_pcm_info_t* info);
typedef int                  (* ma_snd_config_update_free_global_proc)         (void);

/* This array specifies each of the common devices that can be used for both playback and capture. */
static const char* g_maCommonDeviceNamesALSA[] = {
    "default",
    "null",
    "pulse",
    "jack"
};

/* This array allows us to blacklist specific playback devices. */
static const char* g_maBlacklistedPlaybackDeviceNamesALSA[] = {
    ""
};

/* This array allows us to blacklist specific capture devices. */
static const char* g_maBlacklistedCaptureDeviceNamesALSA[] = {
    ""
};


/*
This array allows miniaudio to control device-specific default buffer sizes. This uses a scaling factor. Order is important. If
any part of the string is present in the device's name, the associated scale will be used.
*/
static struct
{
    const char* name;
    float scale;
} g_maDefaultBufferSizeScalesALSA[] = {
    {"bcm2835 IEC958/HDMI", 2.0f},
    {"bcm2835 ALSA",        2.0f}
};

static float ma_find_default_buffer_size_scale__alsa(const char* deviceName)
{
    size_t i;

    if (deviceName == NULL) {
        return 1;
    }

    for (i = 0; i < ma_countof(g_maDefaultBufferSizeScalesALSA); ++i) {
        if (strstr(g_maDefaultBufferSizeScalesALSA[i].name, deviceName) != NULL) {
            return g_maDefaultBufferSizeScalesALSA[i].scale;
        }
    }

    return 1;
}

static ma_snd_pcm_format_t ma_convert_ma_format_to_alsa_format(ma_format format)
{
    ma_snd_pcm_format_t ALSAFormats[] = {
        MA_SND_PCM_FORMAT_UNKNOWN,     /* ma_format_unknown */
        MA_SND_PCM_FORMAT_U8,          /* ma_format_u8 */
        MA_SND_PCM_FORMAT_S16_LE,      /* ma_format_s16 */
        MA_SND_PCM_FORMAT_S24_3LE,     /* ma_format_s24 */
        MA_SND_PCM_FORMAT_S32_LE,      /* ma_format_s32 */
        MA_SND_PCM_FORMAT_FLOAT_LE     /* ma_format_f32 */
    };

    if (ma_is_big_endian()) {
        ALSAFormats[0] = MA_SND_PCM_FORMAT_UNKNOWN;
        ALSAFormats[1] = MA_SND_PCM_FORMAT_U8;
        ALSAFormats[2] = MA_SND_PCM_FORMAT_S16_BE;
        ALSAFormats[3] = MA_SND_PCM_FORMAT_S24_3BE;
        ALSAFormats[4] = MA_SND_PCM_FORMAT_S32_BE;
        ALSAFormats[5] = MA_SND_PCM_FORMAT_FLOAT_BE;
    }

    return ALSAFormats[format];
}

static ma_format ma_format_from_alsa(ma_snd_pcm_format_t formatALSA)
{
    if (ma_is_little_endian()) {
        switch (formatALSA) {
            case MA_SND_PCM_FORMAT_S16_LE:   return ma_format_s16;
            case MA_SND_PCM_FORMAT_S24_3LE:  return ma_format_s24;
            case MA_SND_PCM_FORMAT_S32_LE:   return ma_format_s32;
            case MA_SND_PCM_FORMAT_FLOAT_LE: return ma_format_f32;
            default: break;
        }
    } else {
        switch (formatALSA) {
            case MA_SND_PCM_FORMAT_S16_BE:   return ma_format_s16;
            case MA_SND_PCM_FORMAT_S24_3BE:  return ma_format_s24;
            case MA_SND_PCM_FORMAT_S32_BE:   return ma_format_s32;
            case MA_SND_PCM_FORMAT_FLOAT_BE: return ma_format_f32;
            default: break;
        }
    }

    /* Endian agnostic. */
    switch (formatALSA) {
        case MA_SND_PCM_FORMAT_U8: return ma_format_u8;
        default: return ma_format_unknown;
    }
}

static ma_channel ma_convert_alsa_channel_position_to_ma_channel(unsigned int alsaChannelPos)
{
    switch (alsaChannelPos)
    {
        case MA_SND_CHMAP_MONO: return MA_CHANNEL_MONO;
        case MA_SND_CHMAP_FL:   return MA_CHANNEL_FRONT_LEFT;
        case MA_SND_CHMAP_FR:   return MA_CHANNEL_FRONT_RIGHT;
        case MA_SND_CHMAP_RL:   return MA_CHANNEL_BACK_LEFT;
        case MA_SND_CHMAP_RR:   return MA_CHANNEL_BACK_RIGHT;
        case MA_SND_CHMAP_FC:   return MA_CHANNEL_FRONT_CENTER;
        case MA_SND_CHMAP_LFE:  return MA_CHANNEL_LFE;
        case MA_SND_CHMAP_SL:   return MA_CHANNEL_SIDE_LEFT;
        case MA_SND_CHMAP_SR:   return MA_CHANNEL_SIDE_RIGHT;
        case MA_SND_CHMAP_RC:   return MA_CHANNEL_BACK_CENTER;
        case MA_SND_CHMAP_FLC:  return MA_CHANNEL_FRONT_LEFT_CENTER;
        case MA_SND_CHMAP_FRC:  return MA_CHANNEL_FRONT_RIGHT_CENTER;
        case MA_SND_CHMAP_RLC:  return 0;
        case MA_SND_CHMAP_RRC:  return 0;
        case MA_SND_CHMAP_FLW:  return 0;
        case MA_SND_CHMAP_FRW:  return 0;
        case MA_SND_CHMAP_FLH:  return 0;
        case MA_SND_CHMAP_FCH:  return 0;
        case MA_SND_CHMAP_FRH:  return 0;
        case MA_SND_CHMAP_TC:   return MA_CHANNEL_TOP_CENTER;
        case MA_SND_CHMAP_TFL:  return MA_CHANNEL_TOP_FRONT_LEFT;
        case MA_SND_CHMAP_TFR:  return MA_CHANNEL_TOP_FRONT_RIGHT;
        case MA_SND_CHMAP_TFC:  return MA_CHANNEL_TOP_FRONT_CENTER;
        case MA_SND_CHMAP_TRL:  return MA_CHANNEL_TOP_BACK_LEFT;
        case MA_SND_CHMAP_TRR:  return MA_CHANNEL_TOP_BACK_RIGHT;
        case MA_SND_CHMAP_TRC:  return MA_CHANNEL_TOP_BACK_CENTER;
        default: break;
    }

    return 0;
}

static ma_bool32 ma_is_common_device_name__alsa(const char* name)
{
    size_t iName;
    for (iName = 0; iName < ma_countof(g_maCommonDeviceNamesALSA); ++iName) {
        if (ma_strcmp(name, g_maCommonDeviceNamesALSA[iName]) == 0) {
            return MA_TRUE;
        }
    }

    return MA_FALSE;
}


static ma_bool32 ma_is_playback_device_blacklisted__alsa(const char* name)
{
    size_t iName;
    for (iName = 0; iName < ma_countof(g_maBlacklistedPlaybackDeviceNamesALSA); ++iName) {
        if (ma_strcmp(name, g_maBlacklistedPlaybackDeviceNamesALSA[iName]) == 0) {
            return MA_TRUE;
        }
    }

    return MA_FALSE;
}

static ma_bool32 ma_is_capture_device_blacklisted__alsa(const char* name)
{
    size_t iName;
    for (iName = 0; iName < ma_countof(g_maBlacklistedCaptureDeviceNamesALSA); ++iName) {
        if (ma_strcmp(name, g_maBlacklistedCaptureDeviceNamesALSA[iName]) == 0) {
            return MA_TRUE;
        }
    }

    return MA_FALSE;
}

static ma_bool32 ma_is_device_blacklisted__alsa(ma_device_type deviceType, const char* name)
{
    if (deviceType == ma_device_type_playback) {
        return ma_is_playback_device_blacklisted__alsa(name);
    } else {
        return ma_is_capture_device_blacklisted__alsa(name);
    }
}


static const char* ma_find_char(const char* str, char c, int* index)
{
    int i = 0;
    for (;;) {
        if (str[i] == '\0') {
            if (index) *index = -1;
            return NULL;
        }

        if (str[i] == c) {
            if (index) *index = i;
            return str + i;
        }

        i += 1;
    }

    /* Should never get here, but treat it as though the character was not found to make me feel better inside. */
    if (index) *index = -1;
    return NULL;
}

static ma_bool32 ma_is_device_name_in_hw_format__alsa(const char* hwid)
{
    /* This function is just checking whether or not hwid is in "hw:%d,%d" format. */

    int commaPos;
    const char* dev;
    int i;

    if (hwid == NULL) {
        return MA_FALSE;
    }

    if (hwid[0] != 'h' || hwid[1] != 'w' || hwid[2] != ':') {
        return MA_FALSE;
    }

    hwid += 3;

    dev = ma_find_char(hwid, ',', &commaPos);
    if (dev == NULL) {
        return MA_FALSE;
    } else {
        dev += 1;   /* Skip past the ",". */
    }

    /* Check if the part between the ":" and the "," contains only numbers. If not, return false. */
    for (i = 0; i < commaPos; ++i) {
        if (hwid[i] < '0' || hwid[i] > '9') {
            return MA_FALSE;
        }
    }

    /* Check if everything after the "," is numeric. If not, return false. */
    i = 0;
    while (dev[i] != '\0') {
        if (dev[i] < '0' || dev[i] > '9') {
            return MA_FALSE;
        }
        i += 1;
    }

    return MA_TRUE;
}

static int ma_convert_device_name_to_hw_format__alsa(ma_context* pContext, char* dst, size_t dstSize, const char* src)  /* Returns 0 on success, non-0 on error. */
{
    /* src should look something like this: "hw:CARD=I82801AAICH,DEV=0" */

    int colonPos;
    int commaPos;
    char card[256];
    const char* dev;
    int cardIndex;

    if (dst == NULL) {
        return -1;
    }
    if (dstSize < 7) {
        return -1;     /* Absolute minimum size of the output buffer is 7 bytes. */
    }

    *dst = '\0';    /* Safety. */
    if (src == NULL) {
        return -1;
    }

    /* If the input name is already in "hw:%d,%d" format, just return that verbatim. */
    if (ma_is_device_name_in_hw_format__alsa(src)) {
        return ma_strcpy_s(dst, dstSize, src);
    }

    src = ma_find_char(src, ':', &colonPos);
    if (src == NULL) {
        return -1;  /* Couldn't find a colon */
    }

    dev = ma_find_char(src, ',', &commaPos);
    if (dev == NULL) {
        dev = "0";
        ma_strncpy_s(card, sizeof(card), src+6, (size_t)-1);   /* +6 = ":CARD=" */
    } else {
        dev = dev + 5;  /* +5 = ",DEV=" */
        ma_strncpy_s(card, sizeof(card), src+6, commaPos-6);   /* +6 = ":CARD=" */
    }

    cardIndex = ((ma_snd_card_get_index_proc)pContext->alsa.snd_card_get_index)(card);
    if (cardIndex < 0) {
        return -2;  /* Failed to retrieve the card index. */
    }

    /*printf("TESTING: CARD=%s,DEV=%s\n", card, dev); */


    /* Construction. */
    dst[0] = 'h'; dst[1] = 'w'; dst[2] = ':';
    if (ma_itoa_s(cardIndex, dst+3, dstSize-3, 10) != 0) {
        return -3;
    }
    if (ma_strcat_s(dst, dstSize, ",") != 0) {
        return -3;
    }
    if (ma_strcat_s(dst, dstSize, dev) != 0) {
        return -3;
    }

    return 0;
}

static ma_bool32 ma_does_id_exist_in_list__alsa(ma_device_id* pUniqueIDs, ma_uint32 count, const char* pHWID)
{
    ma_uint32 i;

    MA_ASSERT(pHWID != NULL);

    for (i = 0; i < count; ++i) {
        if (ma_strcmp(pUniqueIDs[i].alsa, pHWID) == 0) {
            return MA_TRUE;
        }
    }

    return MA_FALSE;
}


static ma_result ma_context_open_pcm__alsa(ma_context* pContext, ma_share_mode shareMode, ma_device_type deviceType, const ma_device_id* pDeviceID, int openMode, ma_snd_pcm_t** ppPCM)
{
    ma_snd_pcm_t* pPCM;
    ma_snd_pcm_stream_t stream;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(ppPCM != NULL);

    *ppPCM = NULL;
    pPCM = NULL;

    stream = (deviceType == ma_device_type_playback) ? MA_SND_PCM_STREAM_PLAYBACK : MA_SND_PCM_STREAM_CAPTURE;

    if (pDeviceID == NULL) {
        ma_bool32 isDeviceOpen;
        size_t i;

        /*
        We're opening the default device. I don't know if trying anything other than "default" is necessary, but it makes
        me feel better to try as hard as we can get to get _something_ working.
        */
        const char* defaultDeviceNames[] = {
            "default",
            NULL,
            NULL,
            NULL,
            NULL,
            NULL,
            NULL
        };

        if (shareMode == ma_share_mode_exclusive) {
            defaultDeviceNames[1] = "hw";
            defaultDeviceNames[2] = "hw:0";
            defaultDeviceNames[3] = "hw:0,0";
        } else {
            if (deviceType == ma_device_type_playback) {
                defaultDeviceNames[1] = "dmix";
                defaultDeviceNames[2] = "dmix:0";
                defaultDeviceNames[3] = "dmix:0,0";
            } else {
                defaultDeviceNames[1] = "dsnoop";
                defaultDeviceNames[2] = "dsnoop:0";
                defaultDeviceNames[3] = "dsnoop:0,0";
            }
            defaultDeviceNames[4] = "hw";
            defaultDeviceNames[5] = "hw:0";
            defaultDeviceNames[6] = "hw:0,0";
        }

        isDeviceOpen = MA_FALSE;
        for (i = 0; i < ma_countof(defaultDeviceNames); ++i) {
            if (defaultDeviceNames[i] != NULL && defaultDeviceNames[i][0] != '\0') {
                if (((ma_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, defaultDeviceNames[i], stream, openMode) == 0) {
                    isDeviceOpen = MA_TRUE;
                    break;
                }
            }
        }

        if (!isDeviceOpen) {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_open() failed when trying to open an appropriate default device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    } else {
        /*
        We're trying to open a specific device. There's a few things to consider here:
        
        miniaudio recongnizes a special format of device id that excludes the "hw", "dmix", etc. prefix. It looks like this: ":0,0", ":0,1", etc. When
        an ID of this format is specified, it indicates to miniaudio that it can try different combinations of plugins ("hw", "dmix", etc.) until it
        finds an appropriate one that works. This comes in very handy when trying to open a device in shared mode ("dmix"), vs exclusive mode ("hw").
        */

        /* May end up needing to make small adjustments to the ID, so make a copy. */
        ma_device_id deviceID = *pDeviceID;
        int resultALSA = -ENODEV;

        if (deviceID.alsa[0] != ':') {
            /* The ID is not in ":0,0" format. Use the ID exactly as-is. */
            resultALSA = ((ma_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, deviceID.alsa, stream, openMode);
        } else {
            char hwid[256];

            /* The ID is in ":0,0" format. Try different plugins depending on the shared mode. */
            if (deviceID.alsa[1] == '\0') {
                deviceID.alsa[0] = '\0';  /* An ID of ":" should be converted to "". */
            }

            if (shareMode == ma_share_mode_shared) {
                if (deviceType == ma_device_type_playback) {
                    ma_strcpy_s(hwid, sizeof(hwid), "dmix");
                } else {
                    ma_strcpy_s(hwid, sizeof(hwid), "dsnoop");
                }

                if (ma_strcat_s(hwid, sizeof(hwid), deviceID.alsa) == 0) {
                    resultALSA = ((ma_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, hwid, stream, openMode);
                }
            }

            /* If at this point we still don't have an open device it means we're either preferencing exclusive mode or opening with "dmix"/"dsnoop" failed. */
            if (resultALSA != 0) {
                ma_strcpy_s(hwid, sizeof(hwid), "hw");
                if (ma_strcat_s(hwid, sizeof(hwid), deviceID.alsa) == 0) {
                    resultALSA = ((ma_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, hwid, stream, openMode);
                }
            }
        }

        if (resultALSA < 0) {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_open() failed.", ma_result_from_errno(-resultALSA));
        }
    }

    *ppPCM = pPCM;
    return MA_SUCCESS;
}


static ma_bool32 ma_context_is_device_id_equal__alsa(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pID0 != NULL);
    MA_ASSERT(pID1 != NULL);
    (void)pContext;

    return ma_strcmp(pID0->alsa, pID1->alsa) == 0;
}

static ma_result ma_context_enumerate_devices__alsa(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    int resultALSA;
    ma_bool32 cbResult = MA_TRUE;
    char** ppDeviceHints;
    ma_device_id* pUniqueIDs = NULL;
    ma_uint32 uniqueIDCount = 0;
    char** ppNextDeviceHint;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(callback != NULL);

    ma_mutex_lock(&pContext->alsa.internalDeviceEnumLock);

    resultALSA = ((ma_snd_device_name_hint_proc)pContext->alsa.snd_device_name_hint)(-1, "pcm", (void***)&ppDeviceHints);
    if (resultALSA < 0) {
        ma_mutex_unlock(&pContext->alsa.internalDeviceEnumLock);
        return ma_result_from_errno(-resultALSA);
    }

    ppNextDeviceHint = ppDeviceHints;
    while (*ppNextDeviceHint != NULL) {
        char* NAME = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "NAME");
        char* DESC = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "DESC");
        char* IOID = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "IOID");
        ma_device_type deviceType = ma_device_type_playback;
        ma_bool32 stopEnumeration = MA_FALSE;
        char hwid[sizeof(pUniqueIDs->alsa)];
        ma_device_info deviceInfo;

        if ((IOID == NULL || ma_strcmp(IOID, "Output") == 0)) {
            deviceType = ma_device_type_playback;
        }
        if ((IOID != NULL && ma_strcmp(IOID, "Input" ) == 0)) {
            deviceType = ma_device_type_capture;
        }

        if (NAME != NULL) {
            if (pContext->alsa.useVerboseDeviceEnumeration) {
                /* Verbose mode. Use the name exactly as-is. */
                ma_strncpy_s(hwid, sizeof(hwid), NAME, (size_t)-1);
            } else {
                /* Simplified mode. Use ":%d,%d" format. */
                if (ma_convert_device_name_to_hw_format__alsa(pContext, hwid, sizeof(hwid), NAME) == 0) {
                    /*
                    At this point, hwid looks like "hw:0,0". In simplified enumeration mode, we actually want to strip off the
                    plugin name so it looks like ":0,0". The reason for this is that this special format is detected at device
                    initialization time and is used as an indicator to try and use the most appropriate plugin depending on the
                    device type and sharing mode.
                    */
                    char* dst = hwid;
                    char* src = hwid+2;
                    while ((*dst++ = *src++));
                } else {
                    /* Conversion to "hw:%d,%d" failed. Just use the name as-is. */
                    ma_strncpy_s(hwid, sizeof(hwid), NAME, (size_t)-1);
                }

                if (ma_does_id_exist_in_list__alsa(pUniqueIDs, uniqueIDCount, hwid)) {
                    goto next_device;   /* The device has already been enumerated. Move on to the next one. */
                } else {
                    /* The device has not yet been enumerated. Make sure it's added to our list so that it's not enumerated again. */
                    size_t oldCapacity = sizeof(*pUniqueIDs) *  uniqueIDCount;
                    size_t newCapacity = sizeof(*pUniqueIDs) * (uniqueIDCount + 1);
                    ma_device_id* pNewUniqueIDs = (ma_device_id*)ma__realloc_from_callbacks(pUniqueIDs, newCapacity, oldCapacity, &pContext->allocationCallbacks);
                    if (pNewUniqueIDs == NULL) {
                        goto next_device;   /* Failed to allocate memory. */
                    }

                    pUniqueIDs = pNewUniqueIDs;
                    MA_COPY_MEMORY(pUniqueIDs[uniqueIDCount].alsa, hwid, sizeof(hwid));
                    uniqueIDCount += 1;
                }
            }
        } else {
            MA_ZERO_MEMORY(hwid, sizeof(hwid));
        }

        MA_ZERO_OBJECT(&deviceInfo);
        ma_strncpy_s(deviceInfo.id.alsa, sizeof(deviceInfo.id.alsa), hwid, (size_t)-1);

        /*
        DESC is the friendly name. We treat this slightly differently depending on whether or not we are using verbose
        device enumeration. In verbose mode we want to take the entire description so that the end-user can distinguish
        between the subdevices of each card/dev pair. In simplified mode, however, we only want the first part of the
        description.
        
        The value in DESC seems to be split into two lines, with the first line being the name of the device and the
        second line being a description of the device. I don't like having the description be across two lines because
        it makes formatting ugly and annoying. I'm therefore deciding to put it all on a single line with the second line
        being put into parentheses. In simplified mode I'm just stripping the second line entirely.
        */
        if (DESC != NULL) {
            int lfPos;
            const char* line2 = ma_find_char(DESC, '\n', &lfPos);
            if (line2 != NULL) {
                line2 += 1; /* Skip past the new-line character. */

                if (pContext->alsa.useVerboseDeviceEnumeration) {
                    /* Verbose mode. Put the second line in brackets. */
                    ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), DESC, lfPos);
                    ma_strcat_s (deviceInfo.name, sizeof(deviceInfo.name), " (");
                    ma_strcat_s (deviceInfo.name, sizeof(deviceInfo.name), line2);
                    ma_strcat_s (deviceInfo.name, sizeof(deviceInfo.name), ")");
                } else {
                    /* Simplified mode. Strip the second line entirely. */
                    ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), DESC, lfPos);
                }
            } else {
                /* There's no second line. Just copy the whole description. */
                ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), DESC, (size_t)-1);
            }
        }

        if (!ma_is_device_blacklisted__alsa(deviceType, NAME)) {
            cbResult = callback(pContext, deviceType, &deviceInfo, pUserData);
        }

        /*
        Some devices are both playback and capture, but they are only enumerated by ALSA once. We need to fire the callback
        again for the other device type in this case. We do this for known devices.
        */
        if (cbResult) {
            if (ma_is_common_device_name__alsa(NAME)) {
                if (deviceType == ma_device_type_playback) {
                    if (!ma_is_capture_device_blacklisted__alsa(NAME)) {
                        cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
                    }
                } else {
                    if (!ma_is_playback_device_blacklisted__alsa(NAME)) {
                        cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
                    }
                }
            }
        }

        if (cbResult == MA_FALSE) {
            stopEnumeration = MA_TRUE;
        }

    next_device:
        free(NAME);
        free(DESC);
        free(IOID);
        ppNextDeviceHint += 1;

        /* We need to stop enumeration if the callback returned false. */
        if (stopEnumeration) {
            break;
        }
    }

    ma__free_from_callbacks(pUniqueIDs, &pContext->allocationCallbacks);
    ((ma_snd_device_name_free_hint_proc)pContext->alsa.snd_device_name_free_hint)((void**)ppDeviceHints);

    ma_mutex_unlock(&pContext->alsa.internalDeviceEnumLock);

    return MA_SUCCESS;
}


typedef struct
{
    ma_device_type deviceType;
    const ma_device_id* pDeviceID;
    ma_share_mode shareMode;
    ma_device_info* pDeviceInfo;
    ma_bool32 foundDevice;
} ma_context_get_device_info_enum_callback_data__alsa;

static ma_bool32 ma_context_get_device_info_enum_callback__alsa(ma_context* pContext, ma_device_type deviceType, const ma_device_info* pDeviceInfo, void* pUserData)
{
    ma_context_get_device_info_enum_callback_data__alsa* pData = (ma_context_get_device_info_enum_callback_data__alsa*)pUserData;
    MA_ASSERT(pData != NULL);

    if (pData->pDeviceID == NULL && ma_strcmp(pDeviceInfo->id.alsa, "default") == 0) {
        ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pDeviceInfo->name, (size_t)-1);
        pData->foundDevice = MA_TRUE;
    } else {
        if (pData->deviceType == deviceType && ma_context_is_device_id_equal__alsa(pContext, pData->pDeviceID, &pDeviceInfo->id)) {
            ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pDeviceInfo->name, (size_t)-1);
            pData->foundDevice = MA_TRUE;
        }
    }

    /* Keep enumerating until we have found the device. */
    return !pData->foundDevice;
}

static ma_result ma_context_get_device_info__alsa(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_context_get_device_info_enum_callback_data__alsa data;
    ma_result result;
    int resultALSA;
    ma_snd_pcm_t* pPCM;
    ma_snd_pcm_hw_params_t* pHWParams;
    ma_snd_pcm_format_mask_t* pFormatMask;
    int sampleRateDir = 0;

    MA_ASSERT(pContext != NULL);

    /* We just enumerate to find basic information about the device. */
    data.deviceType = deviceType;
    data.pDeviceID = pDeviceID;
    data.shareMode = shareMode;
    data.pDeviceInfo = pDeviceInfo;
    data.foundDevice = MA_FALSE;
    result = ma_context_enumerate_devices__alsa(pContext, ma_context_get_device_info_enum_callback__alsa, &data);
    if (result != MA_SUCCESS) {
        return result;
    }

    if (!data.foundDevice) {
        return MA_NO_DEVICE;
    }

    /* For detailed info we need to open the device. */
    result = ma_context_open_pcm__alsa(pContext, shareMode, deviceType, pDeviceID, 0, &pPCM);
    if (result != MA_SUCCESS) {
        return result;
    }

    /* We need to initialize a HW parameters object in order to know what formats are supported. */
    pHWParams = (ma_snd_pcm_hw_params_t*)ma__calloc_from_callbacks(((ma_snd_pcm_hw_params_sizeof_proc)pContext->alsa.snd_pcm_hw_params_sizeof)(), &pContext->allocationCallbacks);
    if (pHWParams == NULL) {
        return MA_OUT_OF_MEMORY;
    }

    resultALSA = ((ma_snd_pcm_hw_params_any_proc)pContext->alsa.snd_pcm_hw_params_any)(pPCM, pHWParams);
    if (resultALSA < 0) {
        ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to initialize hardware parameters. snd_pcm_hw_params_any() failed.", ma_result_from_errno(-resultALSA));
    }

    ((ma_snd_pcm_hw_params_get_channels_min_proc)pContext->alsa.snd_pcm_hw_params_get_channels_min)(pHWParams, &pDeviceInfo->minChannels);
    ((ma_snd_pcm_hw_params_get_channels_max_proc)pContext->alsa.snd_pcm_hw_params_get_channels_max)(pHWParams, &pDeviceInfo->maxChannels);
    ((ma_snd_pcm_hw_params_get_rate_min_proc)pContext->alsa.snd_pcm_hw_params_get_rate_min)(pHWParams, &pDeviceInfo->minSampleRate, &sampleRateDir);
    ((ma_snd_pcm_hw_params_get_rate_max_proc)pContext->alsa.snd_pcm_hw_params_get_rate_max)(pHWParams, &pDeviceInfo->maxSampleRate, &sampleRateDir);

    /* Formats. */
    pFormatMask = (ma_snd_pcm_format_mask_t*)ma__calloc_from_callbacks(((ma_snd_pcm_format_mask_sizeof_proc)pContext->alsa.snd_pcm_format_mask_sizeof)(), &pContext->allocationCallbacks);
    if (pFormatMask == NULL) {
        ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
        return MA_OUT_OF_MEMORY;
    }

    ((ma_snd_pcm_hw_params_get_format_mask_proc)pContext->alsa.snd_pcm_hw_params_get_format_mask)(pHWParams, pFormatMask);

    pDeviceInfo->formatCount = 0;
    if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MA_SND_PCM_FORMAT_U8)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_u8;
    }
    if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MA_SND_PCM_FORMAT_S16_LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_s16;
    }
    if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MA_SND_PCM_FORMAT_S24_3LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_s24;
    }
    if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MA_SND_PCM_FORMAT_S32_LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_s32;
    }
    if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MA_SND_PCM_FORMAT_FLOAT_LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_f32;
    }

    ma__free_from_callbacks(pFormatMask, &pContext->allocationCallbacks);
    ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);

    ((ma_snd_pcm_close_proc)pContext->alsa.snd_pcm_close)(pPCM);
    return MA_SUCCESS;
}


#if 0
/*
Waits for a number of frames to become available for either capture or playback. The return
value is the number of frames available.

This will return early if the main loop is broken with ma_device__break_main_loop().
*/
static ma_uint32 ma_device__wait_for_frames__alsa(ma_device* pDevice, ma_bool32* pRequiresRestart)
{
    MA_ASSERT(pDevice != NULL);

    if (pRequiresRestart) *pRequiresRestart = MA_FALSE;

    /* I want it so that this function returns the period size in frames. We just wait until that number of frames are available and then return. */
    ma_uint32 periodSizeInFrames = pDevice->bufferSizeInFrames / pDevice->periods;
    while (!pDevice->alsa.breakFromMainLoop) {
        ma_snd_pcm_sframes_t framesAvailable = ((ma_snd_pcm_avail_update_proc)pDevice->pContext->alsa.snd_pcm_avail_update)((ma_snd_pcm_t*)pDevice->alsa.pPCM);
        if (framesAvailable < 0) {
            if (framesAvailable == -EPIPE) {
                if (((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, framesAvailable, MA_TRUE) < 0) {
                    return 0;
                }

                /* A device recovery means a restart for mmap mode. */
                if (pRequiresRestart) {
                    *pRequiresRestart = MA_TRUE;
                }

                /* Try again, but if it fails this time just return an error. */
                framesAvailable = ((ma_snd_pcm_avail_update_proc)pDevice->pContext->alsa.snd_pcm_avail_update)((ma_snd_pcm_t*)pDevice->alsa.pPCM);
                if (framesAvailable < 0) {
                    return 0;
                }
            }
        }

        if (framesAvailable >= periodSizeInFrames) {
            return periodSizeInFrames;
        }

        if (framesAvailable < periodSizeInFrames) {
            /* Less than a whole period is available so keep waiting. */
            int waitResult = ((ma_snd_pcm_wait_proc)pDevice->pContext->alsa.snd_pcm_wait)((ma_snd_pcm_t*)pDevice->alsa.pPCM, -1);
            if (waitResult < 0) {
                if (waitResult == -EPIPE) {
                    if (((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, waitResult, MA_TRUE) < 0) {
                        return 0;
                    }

                    /* A device recovery means a restart for mmap mode. */
                    if (pRequiresRestart) {
                        *pRequiresRestart = MA_TRUE;
                    }
                }
            }
        }
    }

    /* We'll get here if the loop was terminated. Just return whatever's available. */
    ma_snd_pcm_sframes_t framesAvailable = ((ma_snd_pcm_avail_update_proc)pDevice->pContext->alsa.snd_pcm_avail_update)((ma_snd_pcm_t*)pDevice->alsa.pPCM);
    if (framesAvailable < 0) {
        return 0;
    }

    return framesAvailable;
}

static ma_bool32 ma_device_read_from_client_and_write__alsa(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);
    if (!ma_device_is_started(pDevice) && ma_device__get_state(pDevice) != MA_STATE_STARTING) {
        return MA_FALSE;
    }
    if (pDevice->alsa.breakFromMainLoop) {
        return MA_FALSE;
    }

    if (pDevice->alsa.isUsingMMap) {
        /* mmap. */
        ma_bool32 requiresRestart;
        ma_uint32 framesAvailable = ma_device__wait_for_frames__alsa(pDevice, &requiresRestart);
        if (framesAvailable == 0) {
            return MA_FALSE;
        }

        /* Don't bother asking the client for more audio data if we're just stopping the device anyway. */
        if (pDevice->alsa.breakFromMainLoop) {
            return MA_FALSE;
        }

        const ma_snd_pcm_channel_area_t* pAreas;
        ma_snd_pcm_uframes_t mappedOffset;
        ma_snd_pcm_uframes_t mappedFrames = framesAvailable;
        while (framesAvailable > 0) {
            int result = ((ma_snd_pcm_mmap_begin_proc)pDevice->pContext->alsa.snd_pcm_mmap_begin)((ma_snd_pcm_t*)pDevice->alsa.pPCM, &pAreas, &mappedOffset, &mappedFrames);
            if (result < 0) {
                return MA_FALSE;
            }

            if (mappedFrames > 0) {
                void* pBuffer = (ma_uint8*)pAreas[0].addr + ((pAreas[0].first + (mappedOffset * pAreas[0].step)) / 8);
                ma_device__read_frames_from_client(pDevice, mappedFrames, pBuffer);
            }

            result = ((ma_snd_pcm_mmap_commit_proc)pDevice->pContext->alsa.snd_pcm_mmap_commit)((ma_snd_pcm_t*)pDevice->alsa.pPCM, mappedOffset, mappedFrames);
            if (result < 0 || (ma_snd_pcm_uframes_t)result != mappedFrames) {
                ((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, result, MA_TRUE);
                return MA_FALSE;
            }

            if (requiresRestart) {
                if (((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
                    return MA_FALSE;
                }
            }

            if (framesAvailable >= mappedFrames) {
                framesAvailable -= mappedFrames;
            } else {
                framesAvailable = 0;
            }
        }
    } else {
        /* readi/writei. */
        while (!pDevice->alsa.breakFromMainLoop) {
            ma_uint32 framesAvailable = ma_device__wait_for_frames__alsa(pDevice, NULL);
            if (framesAvailable == 0) {
                continue;
            }

            /* Don't bother asking the client for more audio data if we're just stopping the device anyway. */
            if (pDevice->alsa.breakFromMainLoop) {
                return MA_FALSE;
            }

            ma_device__read_frames_from_client(pDevice, framesAvailable, pDevice->alsa.pIntermediaryBuffer);

            ma_snd_pcm_sframes_t framesWritten = ((ma_snd_pcm_writei_proc)pDevice->pContext->alsa.snd_pcm_writei)((ma_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
            if (framesWritten < 0) {
                if (framesWritten == -EAGAIN) {
                    continue;   /* Just keep trying... */
                } else if (framesWritten == -EPIPE) {
                    /* Underrun. Just recover and try writing again. */
                    if (((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, framesWritten, MA_TRUE) < 0) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to recover device after underrun.", MA_FAILED_TO_START_BACKEND_DEVICE);
                        return MA_FALSE;
                    }

                    framesWritten = ((ma_snd_pcm_writei_proc)pDevice->pContext->alsa.snd_pcm_writei)((ma_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
                    if (framesWritten < 0) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to write data to the internal device.", ma_result_from_errno((int)-framesWritten));
                        return MA_FALSE;
                    }

                    break;  /* Success. */
                } else {
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_writei() failed when writing initial data.", ma_result_from_errno((int)-framesWritten));
                    return MA_FALSE;
                }
            } else {
                break;  /* Success. */
            }
        }
    }

    return MA_TRUE;
}

static ma_bool32 ma_device_read_and_send_to_client__alsa(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);
    if (!ma_device_is_started(pDevice)) {
        return MA_FALSE;
    }
    if (pDevice->alsa.breakFromMainLoop) {
        return MA_FALSE;
    }

    ma_uint32 framesToSend = 0;
    void* pBuffer = NULL;
    if (pDevice->alsa.pIntermediaryBuffer == NULL) {
        /* mmap. */
        ma_bool32 requiresRestart;
        ma_uint32 framesAvailable = ma_device__wait_for_frames__alsa(pDevice, &requiresRestart);
        if (framesAvailable == 0) {
            return MA_FALSE;
        }

        const ma_snd_pcm_channel_area_t* pAreas;
        ma_snd_pcm_uframes_t mappedOffset;
        ma_snd_pcm_uframes_t mappedFrames = framesAvailable;
        while (framesAvailable > 0) {
            int result = ((ma_snd_pcm_mmap_begin_proc)pDevice->pContext->alsa.snd_pcm_mmap_begin)((ma_snd_pcm_t*)pDevice->alsa.pPCM, &pAreas, &mappedOffset, &mappedFrames);
            if (result < 0) {
                return MA_FALSE;
            }

            if (mappedFrames > 0) {
                void* pBuffer = (ma_uint8*)pAreas[0].addr + ((pAreas[0].first + (mappedOffset * pAreas[0].step)) / 8);
                ma_device__send_frames_to_client(pDevice, mappedFrames, pBuffer);
            }

            result = ((ma_snd_pcm_mmap_commit_proc)pDevice->pContext->alsa.snd_pcm_mmap_commit)((ma_snd_pcm_t*)pDevice->alsa.pPCM, mappedOffset, mappedFrames);
            if (result < 0 || (ma_snd_pcm_uframes_t)result != mappedFrames) {
                ((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, result, MA_TRUE);
                return MA_FALSE;
            }

            if (requiresRestart) {
                if (((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
                    return MA_FALSE;
                }
            }

            if (framesAvailable >= mappedFrames) {
                framesAvailable -= mappedFrames;
            } else {
                framesAvailable = 0;
            }
        }
    } else {
        /* readi/writei. */
        ma_snd_pcm_sframes_t framesRead = 0;
        while (!pDevice->alsa.breakFromMainLoop) {
            ma_uint32 framesAvailable = ma_device__wait_for_frames__alsa(pDevice, NULL);
            if (framesAvailable == 0) {
                continue;
            }

            framesRead = ((ma_snd_pcm_readi_proc)pDevice->pContext->alsa.snd_pcm_readi)((ma_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
            if (framesRead < 0) {
                if (framesRead == -EAGAIN) {
                    continue;   /* Just keep trying... */
                } else if (framesRead == -EPIPE) {
                    /* Overrun. Just recover and try reading again. */
                    if (((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, framesRead, MA_TRUE) < 0) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to recover device after overrun.", MA_FAILED_TO_START_BACKEND_DEVICE);
                        return MA_FALSE;
                    }

                    framesRead = ((ma_snd_pcm_readi_proc)pDevice->pContext->alsa.snd_pcm_readi)((ma_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
                    if (framesRead < 0) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to read data from the internal device.", ma_result_from_errno((int)-framesRead));
                        return MA_FALSE;
                    }

                    break;  /* Success. */
                } else {
                    return MA_FALSE;
                }
            } else {
                break;  /* Success. */
            }
        }

        framesToSend = framesRead;
        pBuffer = pDevice->alsa.pIntermediaryBuffer;
    }

    if (framesToSend > 0) {
        ma_device__send_frames_to_client(pDevice, framesToSend, pBuffer);
    }

    return MA_TRUE;
}
#endif /* 0 */

static void ma_device_uninit__alsa(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    if ((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture) {
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture);
    }

    if ((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback) {
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback);
    }
}

static ma_result ma_device_init_by_type__alsa(ma_context* pContext, const ma_device_config* pConfig, ma_device_type deviceType, ma_device* pDevice)
{
    ma_result result;
    int resultALSA;
    ma_snd_pcm_t* pPCM;
    ma_bool32 isUsingMMap;
    ma_snd_pcm_format_t formatALSA;
    ma_share_mode shareMode;
    const ma_device_id* pDeviceID;
    ma_format internalFormat;
    ma_uint32 internalChannels;
    ma_uint32 internalSampleRate;
    ma_channel internalChannelMap[MA_MAX_CHANNELS];
    ma_uint32 internalPeriodSizeInFrames;
    ma_uint32 internalPeriods;
    int openMode;
    ma_snd_pcm_hw_params_t* pHWParams;
    ma_snd_pcm_sw_params_t* pSWParams;
    ma_snd_pcm_uframes_t bufferBoundary;
    float bufferSizeScaleFactor;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pConfig != NULL);
    MA_ASSERT(deviceType != ma_device_type_duplex); /* This function should only be called for playback _or_ capture, never duplex. */
    MA_ASSERT(pDevice != NULL);

    formatALSA = ma_convert_ma_format_to_alsa_format((deviceType == ma_device_type_capture) ? pConfig->capture.format : pConfig->playback.format);
    shareMode  = (deviceType == ma_device_type_capture) ? pConfig->capture.shareMode : pConfig->playback.shareMode;
    pDeviceID  = (deviceType == ma_device_type_capture) ? pConfig->capture.pDeviceID : pConfig->playback.pDeviceID;

    openMode = 0;
    if (pConfig->alsa.noAutoResample) {
        openMode |= MA_SND_PCM_NO_AUTO_RESAMPLE;
    }
    if (pConfig->alsa.noAutoChannels) {
        openMode |= MA_SND_PCM_NO_AUTO_CHANNELS;
    }
    if (pConfig->alsa.noAutoFormat) {
        openMode |= MA_SND_PCM_NO_AUTO_FORMAT;
    }

    result = ma_context_open_pcm__alsa(pContext, shareMode, deviceType, pDeviceID, openMode, &pPCM);
    if (result != MA_SUCCESS) {
        return result;
    }

    /* If using the default buffer size we may want to apply some device-specific scaling for known devices that have peculiar latency characteristics */
    bufferSizeScaleFactor = 1;
    if (pDevice->usingDefaultBufferSize) {
        ma_snd_pcm_info_t* pInfo = (ma_snd_pcm_info_t*)ma__calloc_from_callbacks(((ma_snd_pcm_info_sizeof_proc)pContext->alsa.snd_pcm_info_sizeof)(), &pContext->allocationCallbacks);
        if (pInfo == NULL) {
            return MA_OUT_OF_MEMORY;
        }

        /* We may need to scale the size of the buffer depending on the device. */
        if (((ma_snd_pcm_info_proc)pContext->alsa.snd_pcm_info)(pPCM, pInfo) == 0) {
            const char* deviceName = ((ma_snd_pcm_info_get_name_proc)pContext->alsa.snd_pcm_info_get_name)(pInfo);
            if (deviceName != NULL) {
                if (ma_strcmp(deviceName, "default") == 0) {
                    char** ppDeviceHints;
                    char** ppNextDeviceHint;

                    /* It's the default device. We need to use DESC from snd_device_name_hint(). */
                    if (((ma_snd_device_name_hint_proc)pContext->alsa.snd_device_name_hint)(-1, "pcm", (void***)&ppDeviceHints) < 0) {
                        ma__free_from_callbacks(pInfo, &pContext->allocationCallbacks);
                        return MA_NO_BACKEND;
                    }

                    ppNextDeviceHint = ppDeviceHints;
                    while (*ppNextDeviceHint != NULL) {
                        char* NAME = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "NAME");
                        char* DESC = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "DESC");
                        char* IOID = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "IOID");

                        ma_bool32 foundDevice = MA_FALSE;
                        if ((deviceType == ma_device_type_playback && (IOID == NULL || ma_strcmp(IOID, "Output") == 0)) ||
                            (deviceType == ma_device_type_capture  && (IOID != NULL && ma_strcmp(IOID, "Input" ) == 0))) {
                            if (ma_strcmp(NAME, deviceName) == 0) {
                                bufferSizeScaleFactor = ma_find_default_buffer_size_scale__alsa(DESC);
                                foundDevice = MA_TRUE;
                            }
                        }

                        free(NAME);
                        free(DESC);
                        free(IOID);
                        ppNextDeviceHint += 1;

                        if (foundDevice) {
                            break;
                        }
                    }

                    ((ma_snd_device_name_free_hint_proc)pContext->alsa.snd_device_name_free_hint)((void**)ppDeviceHints);
                } else {
                    bufferSizeScaleFactor = ma_find_default_buffer_size_scale__alsa(deviceName);
                }
            }
        }

        ma__free_from_callbacks(pInfo, &pContext->allocationCallbacks);
    }


    /* Hardware parameters. */
    pHWParams = (ma_snd_pcm_hw_params_t*)ma__calloc_from_callbacks(((ma_snd_pcm_hw_params_sizeof_proc)pContext->alsa.snd_pcm_hw_params_sizeof)(), &pContext->allocationCallbacks);
    if (pHWParams == NULL) {
        return MA_OUT_OF_MEMORY;
    }

    resultALSA = ((ma_snd_pcm_hw_params_any_proc)pContext->alsa.snd_pcm_hw_params_any)(pPCM, pHWParams);
    if (resultALSA < 0) {
        ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to initialize hardware parameters. snd_pcm_hw_params_any() failed.", ma_result_from_errno(-resultALSA));
    }

    /* MMAP Mode. Try using interleaved MMAP access. If this fails, fall back to standard readi/writei. */
    isUsingMMap = MA_FALSE;
#if 0   /* NOTE: MMAP mode temporarily disabled. */
    if (deviceType != ma_device_type_capture) {    /* <-- Disabling MMAP mode for capture devices because I apparently do not have a device that supports it which means I can't test it... Contributions welcome. */
        if (!pConfig->alsa.noMMap && ma_device__is_async(pDevice)) {
            if (((ma_snd_pcm_hw_params_set_access_proc)pContext->alsa.snd_pcm_hw_params_set_access)(pPCM, pHWParams, MA_SND_PCM_ACCESS_MMAP_INTERLEAVED) == 0) {
                pDevice->alsa.isUsingMMap = MA_TRUE;
            }
        }
    }
#endif

    if (!isUsingMMap) {
        resultALSA = ((ma_snd_pcm_hw_params_set_access_proc)pContext->alsa.snd_pcm_hw_params_set_access)(pPCM, pHWParams, MA_SND_PCM_ACCESS_RW_INTERLEAVED);
        if (resultALSA < 0) {
            ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set access mode to neither SND_PCM_ACCESS_MMAP_INTERLEAVED nor SND_PCM_ACCESS_RW_INTERLEAVED. snd_pcm_hw_params_set_access() failed.", ma_result_from_errno(-resultALSA));
        }
    }

    /*
    Most important properties first. The documentation for OSS (yes, I know this is ALSA!) recommends format, channels, then sample rate. I can't
    find any documentation for ALSA specifically, so I'm going to copy the recommendation for OSS.
    */

    /* Format. */
    {
        ma_snd_pcm_format_mask_t* pFormatMask;

        /* Try getting every supported format first. */
        pFormatMask = (ma_snd_pcm_format_mask_t*)ma__calloc_from_callbacks(((ma_snd_pcm_format_mask_sizeof_proc)pContext->alsa.snd_pcm_format_mask_sizeof)(), &pContext->allocationCallbacks);
        if (pFormatMask == NULL) {
            ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return MA_OUT_OF_MEMORY;
        }

        ((ma_snd_pcm_hw_params_get_format_mask_proc)pContext->alsa.snd_pcm_hw_params_get_format_mask)(pHWParams, pFormatMask);

        /*
        At this point we should have a list of supported formats, so now we need to find the best one. We first check if the requested format is
        supported, and if so, use that one. If it's not supported, we just run though a list of formats and try to find the best one.
        */
        if (!((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, formatALSA)) {
            size_t i;

            /* The requested format is not supported so now try running through the list of formats and return the best one. */
            ma_snd_pcm_format_t preferredFormatsALSA[] = {
                MA_SND_PCM_FORMAT_S16_LE,      /* ma_format_s16 */
                MA_SND_PCM_FORMAT_FLOAT_LE,    /* ma_format_f32 */
                MA_SND_PCM_FORMAT_S32_LE,      /* ma_format_s32 */
                MA_SND_PCM_FORMAT_S24_3LE,     /* ma_format_s24 */
                MA_SND_PCM_FORMAT_U8           /* ma_format_u8 */
            };

            if (ma_is_big_endian()) {
                preferredFormatsALSA[0] = MA_SND_PCM_FORMAT_S16_BE;
                preferredFormatsALSA[1] = MA_SND_PCM_FORMAT_FLOAT_BE;
                preferredFormatsALSA[2] = MA_SND_PCM_FORMAT_S32_BE;
                preferredFormatsALSA[3] = MA_SND_PCM_FORMAT_S24_3BE;
                preferredFormatsALSA[4] = MA_SND_PCM_FORMAT_U8;
            }

            formatALSA = MA_SND_PCM_FORMAT_UNKNOWN;
            for (i = 0; i < (sizeof(preferredFormatsALSA) / sizeof(preferredFormatsALSA[0])); ++i) {
                if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, preferredFormatsALSA[i])) {
                    formatALSA = preferredFormatsALSA[i];
                    break;
                }
            }

            if (formatALSA == MA_SND_PCM_FORMAT_UNKNOWN) {
                ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
                ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Format not supported. The device does not support any miniaudio formats.", MA_FORMAT_NOT_SUPPORTED);
            }
        }

        ma__free_from_callbacks(pFormatMask, &pContext->allocationCallbacks);
        pFormatMask = NULL;

        resultALSA = ((ma_snd_pcm_hw_params_set_format_proc)pContext->alsa.snd_pcm_hw_params_set_format)(pPCM, pHWParams, formatALSA);
        if (resultALSA < 0) {
            ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Format not supported. snd_pcm_hw_params_set_format() failed.", ma_result_from_errno(-resultALSA));
        }
        
        internalFormat = ma_format_from_alsa(formatALSA);
        if (internalFormat == ma_format_unknown) {
            ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] The chosen format is not supported by miniaudio.", MA_FORMAT_NOT_SUPPORTED);
        }
    }

    /* Channels. */
    {
        unsigned int channels = (deviceType == ma_device_type_capture) ? pConfig->capture.channels : pConfig->playback.channels;
        resultALSA = ((ma_snd_pcm_hw_params_set_channels_near_proc)pContext->alsa.snd_pcm_hw_params_set_channels_near)(pPCM, pHWParams, &channels);
        if (resultALSA < 0) {
            ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set channel count. snd_pcm_hw_params_set_channels_near() failed.", ma_result_from_errno(-resultALSA));
        }
        internalChannels = (ma_uint32)channels;
    }

    /* Sample Rate */
    {
        unsigned int sampleRate;

        /*
        It appears there's either a bug in ALSA, a bug in some drivers, or I'm doing something silly; but having resampling enabled causes
        problems with some device configurations when used in conjunction with MMAP access mode. To fix this problem we need to disable
        resampling.
        
        To reproduce this problem, open the "plug:dmix" device, and set the sample rate to 44100. Internally, it looks like dmix uses a
        sample rate of 48000. The hardware parameters will get set correctly with no errors, but it looks like the 44100 -> 48000 resampling
        doesn't work properly - but only with MMAP access mode. You will notice skipping/crackling in the audio, and it'll run at a slightly
        faster rate.
        
        miniaudio has built-in support for sample rate conversion (albeit low quality at the moment), so disabling resampling should be fine
        for us. The only problem is that it won't be taking advantage of any kind of hardware-accelerated resampling and it won't be very
        good quality until I get a chance to improve the quality of miniaudio's software sample rate conversion.
        
        I don't currently know if the dmix plugin is the only one with this error. Indeed, this is the only one I've been able to reproduce
        this error with. In the future, we may want to restrict the disabling of resampling to only known bad plugins.
        */
        ((ma_snd_pcm_hw_params_set_rate_resample_proc)pContext->alsa.snd_pcm_hw_params_set_rate_resample)(pPCM, pHWParams, 0);

        sampleRate = pConfig->sampleRate;
        resultALSA = ((ma_snd_pcm_hw_params_set_rate_near_proc)pContext->alsa.snd_pcm_hw_params_set_rate_near)(pPCM, pHWParams, &sampleRate, 0);
        if (resultALSA < 0) {
            ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Sample rate not supported. snd_pcm_hw_params_set_rate_near() failed.", ma_result_from_errno(-resultALSA));
        }
        internalSampleRate = (ma_uint32)sampleRate;
    }

    /* Periods. */
    {
        ma_uint32 periods = pConfig->periods;
        resultALSA = ((ma_snd_pcm_hw_params_set_periods_near_proc)pContext->alsa.snd_pcm_hw_params_set_periods_near)(pPCM, pHWParams, &periods, NULL);
        if (resultALSA < 0) {
            ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set period count. snd_pcm_hw_params_set_periods_near() failed.", ma_result_from_errno(-resultALSA));
        }
        internalPeriods = periods;
    }

    /* Buffer Size */
    {
        ma_snd_pcm_uframes_t actualBufferSizeInFrames = pConfig->periodSizeInFrames * internalPeriods;
        if (actualBufferSizeInFrames == 0) {
            actualBufferSizeInFrames = ma_scale_buffer_size(ma_calculate_buffer_size_in_frames_from_milliseconds(pConfig->periodSizeInMilliseconds, internalSampleRate), bufferSizeScaleFactor) * internalPeriods;
        }

        resultALSA = ((ma_snd_pcm_hw_params_set_buffer_size_near_proc)pContext->alsa.snd_pcm_hw_params_set_buffer_size_near)(pPCM, pHWParams, &actualBufferSizeInFrames);
        if (resultALSA < 0) {
            ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set buffer size for device. snd_pcm_hw_params_set_buffer_size() failed.", ma_result_from_errno(-resultALSA));
        }
        internalPeriodSizeInFrames = actualBufferSizeInFrames / internalPeriods;
    }

    /* Apply hardware parameters. */
    resultALSA = ((ma_snd_pcm_hw_params_proc)pContext->alsa.snd_pcm_hw_params)(pPCM, pHWParams);
    if (resultALSA < 0) {
        ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set hardware parameters. snd_pcm_hw_params() failed.", ma_result_from_errno(-resultALSA));
    }

    ma__free_from_callbacks(pHWParams, &pContext->allocationCallbacks);
    pHWParams = NULL;


    /* Software parameters. */
    pSWParams = (ma_snd_pcm_sw_params_t*)ma__calloc_from_callbacks(((ma_snd_pcm_sw_params_sizeof_proc)pContext->alsa.snd_pcm_sw_params_sizeof)(), &pContext->allocationCallbacks);
    if (pSWParams == NULL) {
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return MA_OUT_OF_MEMORY;
    }

    resultALSA = ((ma_snd_pcm_sw_params_current_proc)pContext->alsa.snd_pcm_sw_params_current)(pPCM, pSWParams);
    if (resultALSA < 0) {
        ma__free_from_callbacks(pSWParams, &pContext->allocationCallbacks);
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to initialize software parameters. snd_pcm_sw_params_current() failed.", ma_result_from_errno(-resultALSA));
    }

    resultALSA = ((ma_snd_pcm_sw_params_set_avail_min_proc)pContext->alsa.snd_pcm_sw_params_set_avail_min)(pPCM, pSWParams, ma_prev_power_of_2(internalPeriodSizeInFrames));
    if (resultALSA < 0) {
        ma__free_from_callbacks(pSWParams, &pContext->allocationCallbacks);
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_sw_params_set_avail_min() failed.", ma_result_from_errno(-resultALSA));
    }

    resultALSA = ((ma_snd_pcm_sw_params_get_boundary_proc)pContext->alsa.snd_pcm_sw_params_get_boundary)(pSWParams, &bufferBoundary);
    if (resultALSA < 0) {
        bufferBoundary = internalPeriodSizeInFrames * internalPeriods;
    }

    /*printf("TRACE: bufferBoundary=%ld\n", bufferBoundary);*/

    if (deviceType == ma_device_type_playback && !isUsingMMap) {   /* Only playback devices in writei/readi mode need a start threshold. */
        /*
        Subtle detail here with the start threshold. When in playback-only mode (no full-duplex) we can set the start threshold to
        the size of a period. But for full-duplex we need to set it such that it is at least two periods.
        */
        resultALSA = ((ma_snd_pcm_sw_params_set_start_threshold_proc)pContext->alsa.snd_pcm_sw_params_set_start_threshold)(pPCM, pSWParams, internalPeriodSizeInFrames*2);
        if (resultALSA < 0) {
            ma__free_from_callbacks(pSWParams, &pContext->allocationCallbacks);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set start threshold for playback device. snd_pcm_sw_params_set_start_threshold() failed.", ma_result_from_errno(-resultALSA));
        }

        resultALSA = ((ma_snd_pcm_sw_params_set_stop_threshold_proc)pContext->alsa.snd_pcm_sw_params_set_stop_threshold)(pPCM, pSWParams, bufferBoundary);
        if (resultALSA < 0) { /* Set to boundary to loop instead of stop in the event of an xrun. */
            ma__free_from_callbacks(pSWParams, &pContext->allocationCallbacks);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set stop threshold for playback device. snd_pcm_sw_params_set_stop_threshold() failed.", ma_result_from_errno(-resultALSA));
        }
    }

    resultALSA = ((ma_snd_pcm_sw_params_proc)pContext->alsa.snd_pcm_sw_params)(pPCM, pSWParams);
    if (resultALSA < 0) {
        ma__free_from_callbacks(pSWParams, &pContext->allocationCallbacks);
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set software parameters. snd_pcm_sw_params() failed.", ma_result_from_errno(-resultALSA));
    }

    ma__free_from_callbacks(pSWParams, &pContext->allocationCallbacks);
    pSWParams = NULL;


    /* Grab the internal channel map. For now we're not going to bother trying to change the channel map and instead just do it ourselves. */
    {
        ma_snd_pcm_chmap_t* pChmap = ((ma_snd_pcm_get_chmap_proc)pContext->alsa.snd_pcm_get_chmap)(pPCM);
        if (pChmap != NULL) {
            ma_uint32 iChannel;

            /* There are cases where the returned channel map can have a different channel count than was returned by snd_pcm_hw_params_set_channels_near(). */
            if (pChmap->channels >= internalChannels) {
                /* Drop excess channels. */
                for (iChannel = 0; iChannel < internalChannels; ++iChannel) {
                    internalChannelMap[iChannel] = ma_convert_alsa_channel_position_to_ma_channel(pChmap->pos[iChannel]);
                }
            } else {
                ma_uint32 i;

                /*
                Excess channels use defaults. Do an initial fill with defaults, overwrite the first pChmap->channels, validate to ensure there are no duplicate
                channels. If validation fails, fall back to defaults.
                */
                ma_bool32 isValid = MA_TRUE;

                /* Fill with defaults. */
                ma_get_standard_channel_map(ma_standard_channel_map_alsa, internalChannels, internalChannelMap);

                /* Overwrite first pChmap->channels channels. */
                for (iChannel = 0; iChannel < pChmap->channels; ++iChannel) {
                    internalChannelMap[iChannel] = ma_convert_alsa_channel_position_to_ma_channel(pChmap->pos[iChannel]);
                }

                /* Validate. */
                for (i = 0; i < internalChannels && isValid; ++i) {
                    ma_uint32 j;
                    for (j = i+1; j < internalChannels; ++j) {
                        if (internalChannelMap[i] == internalChannelMap[j]) {
                            isValid = MA_FALSE;
                            break;
                        }
                    }
                }

                /* If our channel map is invalid, fall back to defaults. */
                if (!isValid) {
                    ma_get_standard_channel_map(ma_standard_channel_map_alsa, internalChannels, internalChannelMap);
                }
            }

            free(pChmap);
            pChmap = NULL;
        } else {
            /* Could not retrieve the channel map. Fall back to a hard-coded assumption. */
            ma_get_standard_channel_map(ma_standard_channel_map_alsa, internalChannels, internalChannelMap);
        }
    }


    /* We're done. Prepare the device. */
    resultALSA = ((ma_snd_pcm_prepare_proc)pDevice->pContext->alsa.snd_pcm_prepare)(pPCM);
    if (resultALSA < 0) {
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to prepare device.", ma_result_from_errno(-resultALSA));
    }


    if (deviceType == ma_device_type_capture) {
        pDevice->alsa.pPCMCapture                    = (ma_ptr)pPCM;
        pDevice->alsa.isUsingMMapCapture             = isUsingMMap;
        pDevice->capture.internalFormat              = internalFormat;
        pDevice->capture.internalChannels            = internalChannels;
        pDevice->capture.internalSampleRate          = internalSampleRate;
        ma_channel_map_copy(pDevice->capture.internalChannelMap, internalChannelMap, ma_min(internalChannels, MA_MAX_CHANNELS));
        pDevice->capture.internalPeriodSizeInFrames  = internalPeriodSizeInFrames;
        pDevice->capture.internalPeriods             = internalPeriods;
    } else {
        pDevice->alsa.pPCMPlayback                   = (ma_ptr)pPCM;
        pDevice->alsa.isUsingMMapPlayback            = isUsingMMap;
        pDevice->playback.internalFormat             = internalFormat;
        pDevice->playback.internalChannels           = internalChannels;
        pDevice->playback.internalSampleRate         = internalSampleRate;
        ma_channel_map_copy(pDevice->playback.internalChannelMap, internalChannelMap, ma_min(internalChannels, MA_MAX_CHANNELS));
        pDevice->playback.internalPeriodSizeInFrames = internalPeriodSizeInFrames;
        pDevice->playback.internalPeriods            = internalPeriods;
    }

    return MA_SUCCESS;
}

static ma_result ma_device_init__alsa(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    MA_ZERO_OBJECT(&pDevice->alsa);

    if (pConfig->deviceType == ma_device_type_loopback) {
        return MA_DEVICE_TYPE_NOT_SUPPORTED;
    }

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_by_type__alsa(pContext, pConfig, ma_device_type_capture, pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_by_type__alsa(pContext, pConfig, ma_device_type_playback, pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    return MA_SUCCESS;
}

static ma_result ma_device_read__alsa(ma_device* pDevice, void* pFramesOut, ma_uint32 frameCount, ma_uint32* pFramesRead)
{
    ma_snd_pcm_sframes_t resultALSA;

    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(pFramesOut != NULL);

    if (pFramesRead != NULL) {
        *pFramesRead = 0;
    }

    for (;;) {
        resultALSA = ((ma_snd_pcm_readi_proc)pDevice->pContext->alsa.snd_pcm_readi)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture, pFramesOut, frameCount);
        if (resultALSA >= 0) {
            break;  /* Success. */
        } else {
            if (resultALSA == -EAGAIN) {
                /*printf("TRACE: EGAIN (read)\n");*/
                continue;   /* Try again. */
            } else if (resultALSA == -EPIPE) {
            #if defined(MA_DEBUG_OUTPUT)
                printf("TRACE: EPIPE (read)\n");
            #endif

                /* Overrun. Recover and try again. If this fails we need to return an error. */
                resultALSA = ((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture, resultALSA, MA_TRUE);
                if (resultALSA < 0) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to recover device after overrun.", ma_result_from_errno((int)-resultALSA));
                }

                resultALSA = ((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture);
                if (resultALSA < 0) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to start device after underrun.", ma_result_from_errno((int)-resultALSA));
                }

                resultALSA = ((ma_snd_pcm_readi_proc)pDevice->pContext->alsa.snd_pcm_readi)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture, pFramesOut, frameCount);
                if (resultALSA < 0) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to read data from the internal device.", ma_result_from_errno((int)-resultALSA));
                }
            }
        }
    }

    if (pFramesRead != NULL) {
        *pFramesRead = resultALSA;
    }

    return MA_SUCCESS;
}

static ma_result ma_device_write__alsa(ma_device* pDevice, const void* pFrames, ma_uint32 frameCount, ma_uint32* pFramesWritten)
{
    ma_snd_pcm_sframes_t resultALSA;

    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(pFrames != NULL);

    if (pFramesWritten != NULL) {
        *pFramesWritten = 0;
    }

    for (;;) {
        resultALSA = ((ma_snd_pcm_writei_proc)pDevice->pContext->alsa.snd_pcm_writei)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback, pFrames, frameCount);
        if (resultALSA >= 0) {
            break;  /* Success. */
        } else {
            if (resultALSA == -EAGAIN) {
                /*printf("TRACE: EGAIN (write)\n");*/
                continue;   /* Try again. */
            } else if (resultALSA == -EPIPE) {
            #if defined(MA_DEBUG_OUTPUT)
                printf("TRACE: EPIPE (write)\n");
            #endif

                /* Underrun. Recover and try again. If this fails we need to return an error. */
                resultALSA = ((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback, resultALSA, MA_TRUE);
                if (resultALSA < 0) { /* MA_TRUE=silent (don't print anything on error). */
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to recover device after underrun.", ma_result_from_errno((int)-resultALSA));
                }

                /*
                In my testing I have had a situation where writei() does not automatically restart the device even though I've set it
                up as such in the software parameters. What will happen is writei() will block indefinitely even though the number of
                frames is well beyond the auto-start threshold. To work around this I've needed to add an explicit start here. Not sure
                if this is me just being stupid and not recovering the device properly, but this definitely feels like something isn't
                quite right here.
                */
                resultALSA = ((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback);
                if (resultALSA < 0) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to start device after underrun.", ma_result_from_errno((int)-resultALSA));
                }

                resultALSA = ((ma_snd_pcm_writei_proc)pDevice->pContext->alsa.snd_pcm_writei)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback, pFrames, frameCount);
                if (resultALSA < 0) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to write data to device after underrun.", ma_result_from_errno((int)-resultALSA));
                }
            }
        }
    }

    if (pFramesWritten != NULL) {
        *pFramesWritten = resultALSA;
    }

    return MA_SUCCESS;
}

static ma_result ma_device_main_loop__alsa(ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;
    int resultALSA;
    ma_bool32 exitLoop = MA_FALSE;

    MA_ASSERT(pDevice != NULL);

    /* Capture devices need to be started immediately. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        resultALSA = ((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture);
        if (resultALSA < 0) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to start device in preparation for reading.", ma_result_from_errno(-resultALSA));
        }
    }

    while (ma_device__get_state(pDevice) == MA_STATE_STARTED && !exitLoop) {
        switch (pDevice->type)
        {
            case ma_device_type_duplex:
            {
                if (pDevice->alsa.isUsingMMapCapture || pDevice->alsa.isUsingMMapPlayback) {
                    /* MMAP */
                    return MA_INVALID_OPERATION;    /* Not yet implemented. */
                } else {
                    /* readi() and writei() */

                    /* The process is: device_read -> convert -> callback -> convert -> device_write */
                    ma_uint32 totalCapturedDeviceFramesProcessed = 0;
                    ma_uint32 capturedDevicePeriodSizeInFrames = ma_min(pDevice->capture.internalPeriodSizeInFrames, pDevice->playback.internalPeriodSizeInFrames);
                    
                    while (totalCapturedDeviceFramesProcessed < capturedDevicePeriodSizeInFrames) {
                        ma_uint8  capturedDeviceData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                        ma_uint8  playbackDeviceData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                        ma_uint32 capturedDeviceDataCapInFrames = sizeof(capturedDeviceData) / ma_get_bytes_per_frame(pDevice->capture.internalFormat,  pDevice->capture.internalChannels);
                        ma_uint32 playbackDeviceDataCapInFrames = sizeof(playbackDeviceData) / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                        ma_uint32 capturedDeviceFramesRemaining;
                        ma_uint32 capturedDeviceFramesProcessed;
                        ma_uint32 capturedDeviceFramesToProcess;
                        ma_uint32 capturedDeviceFramesToTryProcessing = capturedDevicePeriodSizeInFrames - totalCapturedDeviceFramesProcessed;
                        if (capturedDeviceFramesToTryProcessing > capturedDeviceDataCapInFrames) {
                            capturedDeviceFramesToTryProcessing = capturedDeviceDataCapInFrames;
                        }

                        result = ma_device_read__alsa(pDevice, capturedDeviceData, capturedDeviceFramesToTryProcessing, &capturedDeviceFramesToProcess);
                        if (result != MA_SUCCESS) {
                            exitLoop = MA_TRUE;
                            break;
                        }

                        capturedDeviceFramesRemaining = capturedDeviceFramesToProcess;
                        capturedDeviceFramesProcessed = 0;

                        for (;;) {
                            ma_uint8  capturedClientData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                            ma_uint8  playbackClientData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                            ma_uint32 capturedClientDataCapInFrames = sizeof(capturedClientData) / ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
                            ma_uint32 playbackClientDataCapInFrames = sizeof(playbackClientData) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
                            ma_uint64 capturedClientFramesToProcessThisIteration = ma_min(capturedClientDataCapInFrames, playbackClientDataCapInFrames);
                            ma_uint64 capturedDeviceFramesToProcessThisIteration = capturedDeviceFramesRemaining;
                            ma_uint8* pRunningCapturedDeviceFrames = ma_offset_ptr(capturedDeviceData, capturedDeviceFramesProcessed * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));

                            /* Convert capture data from device format to client format. */
                            result = ma_data_converter_process_pcm_frames(&pDevice->capture.converter, pRunningCapturedDeviceFrames, &capturedDeviceFramesToProcessThisIteration, capturedClientData, &capturedClientFramesToProcessThisIteration);
                            if (result != MA_SUCCESS) {
                                break;
                            }

                            /*
                            If we weren't able to generate any output frames it must mean we've exhaused all of our input. The only time this would not be the case is if capturedClientData was too small
                            which should never be the case when it's of the size MA_DATA_CONVERTER_STACK_BUFFER_SIZE.
                            */
                            if (capturedClientFramesToProcessThisIteration == 0) {
                                break;
                            }

                            ma_device__on_data(pDevice, playbackClientData, capturedClientData, (ma_uint32)capturedClientFramesToProcessThisIteration);    /* Safe cast .*/

                            capturedDeviceFramesProcessed += (ma_uint32)capturedDeviceFramesToProcessThisIteration; /* Safe cast. */
                            capturedDeviceFramesRemaining -= (ma_uint32)capturedDeviceFramesToProcessThisIteration; /* Safe cast. */

                            /* At this point the playbackClientData buffer should be holding data that needs to be written to the device. */
                            for (;;) {
                                ma_uint64 convertedClientFrameCount = capturedClientFramesToProcessThisIteration;
                                ma_uint64 convertedDeviceFrameCount = playbackDeviceDataCapInFrames;
                                result = ma_data_converter_process_pcm_frames(&pDevice->playback.converter, playbackClientData, &convertedClientFrameCount, playbackDeviceData, &convertedDeviceFrameCount);
                                if (result != MA_SUCCESS) {
                                    break;
                                }

                                result = ma_device_write__alsa(pDevice, playbackDeviceData, (ma_uint32)convertedDeviceFrameCount, NULL);    /* Safe cast. */
                                if (result != MA_SUCCESS) {
                                    exitLoop = MA_TRUE;
                                    break;
                                }

                                capturedClientFramesToProcessThisIteration -= (ma_uint32)convertedClientFrameCount;  /* Safe cast. */
                                if (capturedClientFramesToProcessThisIteration == 0) {
                                    break;
                                }
                            }

                            /* In case an error happened from ma_device_write__alsa()... */
                            if (result != MA_SUCCESS) {
                                exitLoop = MA_TRUE;
                                break;
                            }
                        }

                        totalCapturedDeviceFramesProcessed += capturedDeviceFramesProcessed;
                    }
                }
            } break;

            case ma_device_type_capture:
            {
                if (pDevice->alsa.isUsingMMapCapture) {
                    /* MMAP */
                    return MA_INVALID_OPERATION;    /* Not yet implemented. */
                } else {
                    /* readi() */

                    /* We read in chunks of the period size, but use a stack allocated buffer for the intermediary. */
                    ma_uint8 intermediaryBuffer[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                    ma_uint32 intermediaryBufferSizeInFrames = sizeof(intermediaryBuffer) / ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
                    ma_uint32 periodSizeInFrames = pDevice->capture.internalPeriodSizeInFrames;
                    ma_uint32 framesReadThisPeriod = 0;
                    while (framesReadThisPeriod < periodSizeInFrames) {
                        ma_uint32 framesRemainingInPeriod = periodSizeInFrames - framesReadThisPeriod;
                        ma_uint32 framesProcessed;
                        ma_uint32 framesToReadThisIteration = framesRemainingInPeriod;
                        if (framesToReadThisIteration > intermediaryBufferSizeInFrames) {
                            framesToReadThisIteration = intermediaryBufferSizeInFrames;
                        }

                        result = ma_device_read__alsa(pDevice, intermediaryBuffer, framesToReadThisIteration, &framesProcessed);
                        if (result != MA_SUCCESS) {
                            exitLoop = MA_TRUE;
                            break;
                        }

                        ma_device__send_frames_to_client(pDevice, framesProcessed, intermediaryBuffer);

                        framesReadThisPeriod += framesProcessed;
                    }
                }
            } break;

            case ma_device_type_playback:
            {
                if (pDevice->alsa.isUsingMMapPlayback) {
                    /* MMAP */
                    return MA_INVALID_OPERATION;    /* Not yet implemented. */
                } else {
                    /* writei() */

                    /* We write in chunks of the period size, but use a stack allocated buffer for the intermediary. */
                    ma_uint8 intermediaryBuffer[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                    ma_uint32 intermediaryBufferSizeInFrames = sizeof(intermediaryBuffer) / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                    ma_uint32 periodSizeInFrames = pDevice->playback.internalPeriodSizeInFrames;
                    ma_uint32 framesWrittenThisPeriod = 0;
                    while (framesWrittenThisPeriod < periodSizeInFrames) {
                        ma_uint32 framesRemainingInPeriod = periodSizeInFrames - framesWrittenThisPeriod;
                        ma_uint32 framesProcessed;
                        ma_uint32 framesToWriteThisIteration = framesRemainingInPeriod;
                        if (framesToWriteThisIteration > intermediaryBufferSizeInFrames) {
                            framesToWriteThisIteration = intermediaryBufferSizeInFrames;
                        }

                        ma_device__read_frames_from_client(pDevice, framesToWriteThisIteration, intermediaryBuffer);

                        result = ma_device_write__alsa(pDevice, intermediaryBuffer, framesToWriteThisIteration, &framesProcessed);
                        if (result != MA_SUCCESS) {
                            exitLoop = MA_TRUE;
                            break;
                        }

                        framesWrittenThisPeriod += framesProcessed;
                    }
                }
            } break;

            /* To silence a warning. Will never hit this. */
            case ma_device_type_loopback:
            default: break;
        } 
    }

    /* Here is where the device needs to be stopped. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_snd_pcm_drain_proc)pDevice->pContext->alsa.snd_pcm_drain)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture);

        /* We need to prepare the device again, otherwise we won't be able to restart the device. */
        if (((ma_snd_pcm_prepare_proc)pDevice->pContext->alsa.snd_pcm_prepare)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture) < 0) {
    #ifdef MA_DEBUG_OUTPUT
            printf("[ALSA] Failed to prepare capture device after stopping.\n");
    #endif
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ((ma_snd_pcm_drain_proc)pDevice->pContext->alsa.snd_pcm_drain)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback);

        /* We need to prepare the device again, otherwise we won't be able to restart the device. */
        if (((ma_snd_pcm_prepare_proc)pDevice->pContext->alsa.snd_pcm_prepare)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback) < 0) {
    #ifdef MA_DEBUG_OUTPUT
            printf("[ALSA] Failed to prepare playback device after stopping.\n");
    #endif
        }
    }

    return result;
}

static ma_result ma_context_uninit__alsa(ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pContext->backend == ma_backend_alsa);

    /* Clean up memory for memory leak checkers. */
    ((ma_snd_config_update_free_global_proc)pContext->alsa.snd_config_update_free_global)();

#ifndef MA_NO_RUNTIME_LINKING
    ma_dlclose(pContext, pContext->alsa.asoundSO);
#endif

    ma_mutex_uninit(&pContext->alsa.internalDeviceEnumLock);

    return MA_SUCCESS;
}

static ma_result ma_context_init__alsa(const ma_context_config* pConfig, ma_context* pContext)
{
#ifndef MA_NO_RUNTIME_LINKING
    const char* libasoundNames[] = {
        "libasound.so.2",
        "libasound.so"
    };
    size_t i;

    for (i = 0; i < ma_countof(libasoundNames); ++i) {
        pContext->alsa.asoundSO = ma_dlopen(pContext, libasoundNames[i]);
        if (pContext->alsa.asoundSO != NULL) {
            break;
        }
    }

    if (pContext->alsa.asoundSO == NULL) {
#ifdef MA_DEBUG_OUTPUT
        printf("[ALSA] Failed to open shared object.\n");
#endif
        return MA_NO_BACKEND;
    }

    pContext->alsa.snd_pcm_open                           = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_open");
    pContext->alsa.snd_pcm_close                          = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_close");
    pContext->alsa.snd_pcm_hw_params_sizeof               = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_sizeof");
    pContext->alsa.snd_pcm_hw_params_any                  = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_any");
    pContext->alsa.snd_pcm_hw_params_set_format           = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_set_format");
    pContext->alsa.snd_pcm_hw_params_set_format_first     = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_set_format_first");
    pContext->alsa.snd_pcm_hw_params_get_format_mask      = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_format_mask");
    pContext->alsa.snd_pcm_hw_params_set_channels_near    = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_set_channels_near");
    pContext->alsa.snd_pcm_hw_params_set_rate_resample    = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_set_rate_resample");
    pContext->alsa.snd_pcm_hw_params_set_rate_near        = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_set_rate_near");
    pContext->alsa.snd_pcm_hw_params_set_buffer_size_near = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_set_buffer_size_near");
    pContext->alsa.snd_pcm_hw_params_set_periods_near     = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_set_periods_near");
    pContext->alsa.snd_pcm_hw_params_set_access           = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_set_access");
    pContext->alsa.snd_pcm_hw_params_get_format           = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_format");
    pContext->alsa.snd_pcm_hw_params_get_channels         = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_channels");
    pContext->alsa.snd_pcm_hw_params_get_channels_min     = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_channels_min");
    pContext->alsa.snd_pcm_hw_params_get_channels_max     = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_channels_max");
    pContext->alsa.snd_pcm_hw_params_get_rate             = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_rate");
    pContext->alsa.snd_pcm_hw_params_get_rate_min         = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_rate_min");
    pContext->alsa.snd_pcm_hw_params_get_rate_max         = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_rate_max");
    pContext->alsa.snd_pcm_hw_params_get_buffer_size      = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_buffer_size");
    pContext->alsa.snd_pcm_hw_params_get_periods          = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_periods");
    pContext->alsa.snd_pcm_hw_params_get_access           = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params_get_access");
    pContext->alsa.snd_pcm_hw_params                      = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_hw_params");
    pContext->alsa.snd_pcm_sw_params_sizeof               = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_sw_params_sizeof");
    pContext->alsa.snd_pcm_sw_params_current              = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_sw_params_current");
    pContext->alsa.snd_pcm_sw_params_get_boundary         = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_sw_params_get_boundary");
    pContext->alsa.snd_pcm_sw_params_set_avail_min        = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_sw_params_set_avail_min");
    pContext->alsa.snd_pcm_sw_params_set_start_threshold  = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_sw_params_set_start_threshold");
    pContext->alsa.snd_pcm_sw_params_set_stop_threshold   = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_sw_params_set_stop_threshold");
    pContext->alsa.snd_pcm_sw_params                      = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_sw_params");
    pContext->alsa.snd_pcm_format_mask_sizeof             = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_format_mask_sizeof");
    pContext->alsa.snd_pcm_format_mask_test               = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_format_mask_test");
    pContext->alsa.snd_pcm_get_chmap                      = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_get_chmap");
    pContext->alsa.snd_pcm_state                          = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_state");
    pContext->alsa.snd_pcm_prepare                        = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_prepare");
    pContext->alsa.snd_pcm_start                          = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_start");
    pContext->alsa.snd_pcm_drop                           = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_drop");
    pContext->alsa.snd_pcm_drain                          = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_drain");
    pContext->alsa.snd_device_name_hint                   = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_device_name_hint");
    pContext->alsa.snd_device_name_get_hint               = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_device_name_get_hint");
    pContext->alsa.snd_card_get_index                     = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_card_get_index");
    pContext->alsa.snd_device_name_free_hint              = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_device_name_free_hint");
    pContext->alsa.snd_pcm_mmap_begin                     = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_mmap_begin");
    pContext->alsa.snd_pcm_mmap_commit                    = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_mmap_commit");
    pContext->alsa.snd_pcm_recover                        = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_recover");
    pContext->alsa.snd_pcm_readi                          = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_readi");
    pContext->alsa.snd_pcm_writei                         = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_writei");
    pContext->alsa.snd_pcm_avail                          = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_avail");
    pContext->alsa.snd_pcm_avail_update                   = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_avail_update");
    pContext->alsa.snd_pcm_wait                           = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_wait");
    pContext->alsa.snd_pcm_info                           = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_info");
    pContext->alsa.snd_pcm_info_sizeof                    = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_info_sizeof");
    pContext->alsa.snd_pcm_info_get_name                  = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_pcm_info_get_name");
    pContext->alsa.snd_config_update_free_global          = (ma_proc)ma_dlsym(pContext, pContext->alsa.asoundSO, "snd_config_update_free_global");
#else
    /* The system below is just for type safety. */
    ma_snd_pcm_open_proc                           _snd_pcm_open                           = snd_pcm_open;
    ma_snd_pcm_close_proc                          _snd_pcm_close                          = snd_pcm_close;
    ma_snd_pcm_hw_params_sizeof_proc               _snd_pcm_hw_params_sizeof               = snd_pcm_hw_params_sizeof;
    ma_snd_pcm_hw_params_any_proc                  _snd_pcm_hw_params_any                  = snd_pcm_hw_params_any;
    ma_snd_pcm_hw_params_set_format_proc           _snd_pcm_hw_params_set_format           = snd_pcm_hw_params_set_format;
    ma_snd_pcm_hw_params_set_format_first_proc     _snd_pcm_hw_params_set_format_first     = snd_pcm_hw_params_set_format_first;
    ma_snd_pcm_hw_params_get_format_mask_proc      _snd_pcm_hw_params_get_format_mask      = snd_pcm_hw_params_get_format_mask;
    ma_snd_pcm_hw_params_set_channels_near_proc    _snd_pcm_hw_params_set_channels_near    = snd_pcm_hw_params_set_channels_near;
    ma_snd_pcm_hw_params_set_rate_resample_proc    _snd_pcm_hw_params_set_rate_resample    = snd_pcm_hw_params_set_rate_resample;
    ma_snd_pcm_hw_params_set_rate_near_proc        _snd_pcm_hw_params_set_rate_near        = snd_pcm_hw_params_set_rate_near;
    ma_snd_pcm_hw_params_set_buffer_size_near_proc _snd_pcm_hw_params_set_buffer_size_near = snd_pcm_hw_params_set_buffer_size_near;
    ma_snd_pcm_hw_params_set_periods_near_proc     _snd_pcm_hw_params_set_periods_near     = snd_pcm_hw_params_set_periods_near;
    ma_snd_pcm_hw_params_set_access_proc           _snd_pcm_hw_params_set_access           = snd_pcm_hw_params_set_access;
    ma_snd_pcm_hw_params_get_format_proc           _snd_pcm_hw_params_get_format           = snd_pcm_hw_params_get_format;
    ma_snd_pcm_hw_params_get_channels_proc         _snd_pcm_hw_params_get_channels         = snd_pcm_hw_params_get_channels;
    ma_snd_pcm_hw_params_get_channels_min_proc     _snd_pcm_hw_params_get_channels_min     = snd_pcm_hw_params_get_channels_min;
    ma_snd_pcm_hw_params_get_channels_max_proc     _snd_pcm_hw_params_get_channels_max     = snd_pcm_hw_params_get_channels_max;
    ma_snd_pcm_hw_params_get_rate_proc             _snd_pcm_hw_params_get_rate             = snd_pcm_hw_params_get_rate;
    ma_snd_pcm_hw_params_get_rate_min_proc         _snd_pcm_hw_params_get_rate_min         = snd_pcm_hw_params_get_rate_min;
    ma_snd_pcm_hw_params_get_rate_max_proc         _snd_pcm_hw_params_get_rate_max         = snd_pcm_hw_params_get_rate_max;
    ma_snd_pcm_hw_params_get_buffer_size_proc      _snd_pcm_hw_params_get_buffer_size      = snd_pcm_hw_params_get_buffer_size;
    ma_snd_pcm_hw_params_get_periods_proc          _snd_pcm_hw_params_get_periods          = snd_pcm_hw_params_get_periods;
    ma_snd_pcm_hw_params_get_access_proc           _snd_pcm_hw_params_get_access           = snd_pcm_hw_params_get_access;
    ma_snd_pcm_hw_params_proc                      _snd_pcm_hw_params                      = snd_pcm_hw_params;
    ma_snd_pcm_sw_params_sizeof_proc               _snd_pcm_sw_params_sizeof               = snd_pcm_sw_params_sizeof;
    ma_snd_pcm_sw_params_current_proc              _snd_pcm_sw_params_current              = snd_pcm_sw_params_current;
    ma_snd_pcm_sw_params_get_boundary_proc         _snd_pcm_sw_params_get_boundary         = snd_pcm_sw_params_get_boundary;
    ma_snd_pcm_sw_params_set_avail_min_proc        _snd_pcm_sw_params_set_avail_min        = snd_pcm_sw_params_set_avail_min;
    ma_snd_pcm_sw_params_set_start_threshold_proc  _snd_pcm_sw_params_set_start_threshold  = snd_pcm_sw_params_set_start_threshold;
    ma_snd_pcm_sw_params_set_stop_threshold_proc   _snd_pcm_sw_params_set_stop_threshold   = snd_pcm_sw_params_set_stop_threshold;
    ma_snd_pcm_sw_params_proc                      _snd_pcm_sw_params                      = snd_pcm_sw_params;
    ma_snd_pcm_format_mask_sizeof_proc             _snd_pcm_format_mask_sizeof             = snd_pcm_format_mask_sizeof;
    ma_snd_pcm_format_mask_test_proc               _snd_pcm_format_mask_test               = snd_pcm_format_mask_test;
    ma_snd_pcm_get_chmap_proc                      _snd_pcm_get_chmap                      = snd_pcm_get_chmap;
    ma_snd_pcm_state_proc                          _snd_pcm_state                          = snd_pcm_state;
    ma_snd_pcm_prepare_proc                        _snd_pcm_prepare                        = snd_pcm_prepare;
    ma_snd_pcm_start_proc                          _snd_pcm_start                          = snd_pcm_start;
    ma_snd_pcm_drop_proc                           _snd_pcm_drop                           = snd_pcm_drop;
    ma_snd_pcm_drain_proc                          _snd_pcm_drain                          = snd_pcm_drain;
    ma_snd_device_name_hint_proc                   _snd_device_name_hint                   = snd_device_name_hint;
    ma_snd_device_name_get_hint_proc               _snd_device_name_get_hint               = snd_device_name_get_hint;
    ma_snd_card_get_index_proc                     _snd_card_get_index                     = snd_card_get_index;
    ma_snd_device_name_free_hint_proc              _snd_device_name_free_hint              = snd_device_name_free_hint;
    ma_snd_pcm_mmap_begin_proc                     _snd_pcm_mmap_begin                     = snd_pcm_mmap_begin;
    ma_snd_pcm_mmap_commit_proc                    _snd_pcm_mmap_commit                    = snd_pcm_mmap_commit;
    ma_snd_pcm_recover_proc                        _snd_pcm_recover                        = snd_pcm_recover;
    ma_snd_pcm_readi_proc                          _snd_pcm_readi                          = snd_pcm_readi;
    ma_snd_pcm_writei_proc                         _snd_pcm_writei                         = snd_pcm_writei;
    ma_snd_pcm_avail_proc                          _snd_pcm_avail                          = snd_pcm_avail;
    ma_snd_pcm_avail_update_proc                   _snd_pcm_avail_update                   = snd_pcm_avail_update;
    ma_snd_pcm_wait_proc                           _snd_pcm_wait                           = snd_pcm_wait;
    ma_snd_pcm_info_proc                           _snd_pcm_info                           = snd_pcm_info;
    ma_snd_pcm_info_sizeof_proc                    _snd_pcm_info_sizeof                    = snd_pcm_info_sizeof;
    ma_snd_pcm_info_get_name_proc                  _snd_pcm_info_get_name                  = snd_pcm_info_get_name;
    ma_snd_config_update_free_global_proc          _snd_config_update_free_global          = snd_config_update_free_global;

    pContext->alsa.snd_pcm_open                           = (ma_proc)_snd_pcm_open;
    pContext->alsa.snd_pcm_close                          = (ma_proc)_snd_pcm_close;
    pContext->alsa.snd_pcm_hw_params_sizeof               = (ma_proc)_snd_pcm_hw_params_sizeof;
    pContext->alsa.snd_pcm_hw_params_any                  = (ma_proc)_snd_pcm_hw_params_any;
    pContext->alsa.snd_pcm_hw_params_set_format           = (ma_proc)_snd_pcm_hw_params_set_format;
    pContext->alsa.snd_pcm_hw_params_set_format_first     = (ma_proc)_snd_pcm_hw_params_set_format_first;
    pContext->alsa.snd_pcm_hw_params_get_format_mask      = (ma_proc)_snd_pcm_hw_params_get_format_mask;
    pContext->alsa.snd_pcm_hw_params_set_channels_near    = (ma_proc)_snd_pcm_hw_params_set_channels_near;
    pContext->alsa.snd_pcm_hw_params_set_rate_resample    = (ma_proc)_snd_pcm_hw_params_set_rate_resample;
    pContext->alsa.snd_pcm_hw_params_set_rate_near        = (ma_proc)_snd_pcm_hw_params_set_rate_near;
    pContext->alsa.snd_pcm_hw_params_set_buffer_size_near = (ma_proc)_snd_pcm_hw_params_set_buffer_size_near;
    pContext->alsa.snd_pcm_hw_params_set_periods_near     = (ma_proc)_snd_pcm_hw_params_set_periods_near;
    pContext->alsa.snd_pcm_hw_params_set_access           = (ma_proc)_snd_pcm_hw_params_set_access;
    pContext->alsa.snd_pcm_hw_params_get_format           = (ma_proc)_snd_pcm_hw_params_get_format;
    pContext->alsa.snd_pcm_hw_params_get_channels         = (ma_proc)_snd_pcm_hw_params_get_channels;
    pContext->alsa.snd_pcm_hw_params_get_channels_min     = (ma_proc)_snd_pcm_hw_params_get_channels_min;
    pContext->alsa.snd_pcm_hw_params_get_channels_max     = (ma_proc)_snd_pcm_hw_params_get_channels_max;
    pContext->alsa.snd_pcm_hw_params_get_rate             = (ma_proc)_snd_pcm_hw_params_get_rate;
    pContext->alsa.snd_pcm_hw_params_get_rate_min         = (ma_proc)_snd_pcm_hw_params_get_rate_min;
    pContext->alsa.snd_pcm_hw_params_get_rate_max         = (ma_proc)_snd_pcm_hw_params_get_rate_max;
    pContext->alsa.snd_pcm_hw_params_get_buffer_size      = (ma_proc)_snd_pcm_hw_params_get_buffer_size;
    pContext->alsa.snd_pcm_hw_params_get_periods          = (ma_proc)_snd_pcm_hw_params_get_periods;
    pContext->alsa.snd_pcm_hw_params_get_access           = (ma_proc)_snd_pcm_hw_params_get_access;
    pContext->alsa.snd_pcm_hw_params                      = (ma_proc)_snd_pcm_hw_params;
    pContext->alsa.snd_pcm_sw_params_sizeof               = (ma_proc)_snd_pcm_sw_params_sizeof;
    pContext->alsa.snd_pcm_sw_params_current              = (ma_proc)_snd_pcm_sw_params_current;
    pContext->alsa.snd_pcm_sw_params_get_boundary         = (ma_proc)_snd_pcm_sw_params_get_boundary;
    pContext->alsa.snd_pcm_sw_params_set_avail_min        = (ma_proc)_snd_pcm_sw_params_set_avail_min;
    pContext->alsa.snd_pcm_sw_params_set_start_threshold  = (ma_proc)_snd_pcm_sw_params_set_start_threshold;
    pContext->alsa.snd_pcm_sw_params_set_stop_threshold   = (ma_proc)_snd_pcm_sw_params_set_stop_threshold;
    pContext->alsa.snd_pcm_sw_params                      = (ma_proc)_snd_pcm_sw_params;
    pContext->alsa.snd_pcm_format_mask_sizeof             = (ma_proc)_snd_pcm_format_mask_sizeof;
    pContext->alsa.snd_pcm_format_mask_test               = (ma_proc)_snd_pcm_format_mask_test;
    pContext->alsa.snd_pcm_get_chmap                      = (ma_proc)_snd_pcm_get_chmap;
    pContext->alsa.snd_pcm_state                          = (ma_proc)_snd_pcm_state;
    pContext->alsa.snd_pcm_prepare                        = (ma_proc)_snd_pcm_prepare;
    pContext->alsa.snd_pcm_start                          = (ma_proc)_snd_pcm_start;
    pContext->alsa.snd_pcm_drop                           = (ma_proc)_snd_pcm_drop;
    pContext->alsa.snd_pcm_drain                          = (ma_proc)_snd_pcm_drain;
    pContext->alsa.snd_device_name_hint                   = (ma_proc)_snd_device_name_hint;
    pContext->alsa.snd_device_name_get_hint               = (ma_proc)_snd_device_name_get_hint;
    pContext->alsa.snd_card_get_index                     = (ma_proc)_snd_card_get_index;
    pContext->alsa.snd_device_name_free_hint              = (ma_proc)_snd_device_name_free_hint;
    pContext->alsa.snd_pcm_mmap_begin                     = (ma_proc)_snd_pcm_mmap_begin;
    pContext->alsa.snd_pcm_mmap_commit                    = (ma_proc)_snd_pcm_mmap_commit;
    pContext->alsa.snd_pcm_recover                        = (ma_proc)_snd_pcm_recover;
    pContext->alsa.snd_pcm_readi                          = (ma_proc)_snd_pcm_readi;
    pContext->alsa.snd_pcm_writei                         = (ma_proc)_snd_pcm_writei;
    pContext->alsa.snd_pcm_avail                          = (ma_proc)_snd_pcm_avail;
    pContext->alsa.snd_pcm_avail_update                   = (ma_proc)_snd_pcm_avail_update;
    pContext->alsa.snd_pcm_wait                           = (ma_proc)_snd_pcm_wait;
    pContext->alsa.snd_pcm_info                           = (ma_proc)_snd_pcm_info;
    pContext->alsa.snd_pcm_info_sizeof                    = (ma_proc)_snd_pcm_info_sizeof;
    pContext->alsa.snd_pcm_info_get_name                  = (ma_proc)_snd_pcm_info_get_name;
    pContext->alsa.snd_config_update_free_global          = (ma_proc)_snd_config_update_free_global;
#endif

    pContext->alsa.useVerboseDeviceEnumeration = pConfig->alsa.useVerboseDeviceEnumeration;

    if (ma_mutex_init(&pContext->alsa.internalDeviceEnumLock) != MA_SUCCESS) {
        ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[ALSA] WARNING: Failed to initialize mutex for internal device enumeration.", MA_ERROR);
    }

    pContext->onUninit         = ma_context_uninit__alsa;
    pContext->onDeviceIDEqual  = ma_context_is_device_id_equal__alsa;
    pContext->onEnumDevices    = ma_context_enumerate_devices__alsa;
    pContext->onGetDeviceInfo  = ma_context_get_device_info__alsa;
    pContext->onDeviceInit     = ma_device_init__alsa;
    pContext->onDeviceUninit   = ma_device_uninit__alsa;
    pContext->onDeviceStart    = NULL; /* Not used. Started in the main loop. */
    pContext->onDeviceStop     = NULL; /* Not used. Started in the main loop. */
    pContext->onDeviceMainLoop = ma_device_main_loop__alsa;

    return MA_SUCCESS;
}
#endif  /* ALSA */



/******************************************************************************

PulseAudio Backend

******************************************************************************/
#ifdef MA_HAS_PULSEAUDIO
/*
It is assumed pulseaudio.h is available when compile-time linking is being used. We use this for type safety when using
compile time linking (we don't have this luxury when using runtime linking without headers).

When using compile time linking, each of our ma_* equivalents should use the sames types as defined by the header. The
reason for this is that it allow us to take advantage of proper type safety.
*/
#ifdef MA_NO_RUNTIME_LINKING

/* pulseaudio.h marks some functions with "inline" which isn't always supported. Need to emulate it. */
#if !defined(__cplusplus)
    #if defined(__STRICT_ANSI__)
        #if !defined(inline)
            #define inline __inline__ __attribute__((always_inline))
            #define MA_INLINE_DEFINED
        #endif
    #endif
#endif
#include <pulse/pulseaudio.h>
#if defined(MA_INLINE_DEFINED)
    #undef inline
    #undef MA_INLINE_DEFINED
#endif

#define MA_PA_OK                                       PA_OK
#define MA_PA_ERR_ACCESS                               PA_ERR_ACCESS
#define MA_PA_ERR_INVALID                              PA_ERR_INVALID
#define MA_PA_ERR_NOENTITY                             PA_ERR_NOENTITY

#define MA_PA_CHANNELS_MAX                             PA_CHANNELS_MAX
#define MA_PA_RATE_MAX                                 PA_RATE_MAX

typedef pa_context_flags_t ma_pa_context_flags_t;
#define MA_PA_CONTEXT_NOFLAGS                          PA_CONTEXT_NOFLAGS
#define MA_PA_CONTEXT_NOAUTOSPAWN                      PA_CONTEXT_NOAUTOSPAWN
#define MA_PA_CONTEXT_NOFAIL                           PA_CONTEXT_NOFAIL

typedef pa_stream_flags_t ma_pa_stream_flags_t;
#define MA_PA_STREAM_NOFLAGS                           PA_STREAM_NOFLAGS
#define MA_PA_STREAM_START_CORKED                      PA_STREAM_START_CORKED
#define MA_PA_STREAM_INTERPOLATE_TIMING                PA_STREAM_INTERPOLATE_TIMING
#define MA_PA_STREAM_NOT_MONOTONIC                     PA_STREAM_NOT_MONOTONIC
#define MA_PA_STREAM_AUTO_TIMING_UPDATE                PA_STREAM_AUTO_TIMING_UPDATE
#define MA_PA_STREAM_NO_REMAP_CHANNELS                 PA_STREAM_NO_REMAP_CHANNELS
#define MA_PA_STREAM_NO_REMIX_CHANNELS                 PA_STREAM_NO_REMIX_CHANNELS
#define MA_PA_STREAM_FIX_FORMAT                        PA_STREAM_FIX_FORMAT
#define MA_PA_STREAM_FIX_RATE                          PA_STREAM_FIX_RATE
#define MA_PA_STREAM_FIX_CHANNELS                      PA_STREAM_FIX_CHANNELS
#define MA_PA_STREAM_DONT_MOVE                         PA_STREAM_DONT_MOVE
#define MA_PA_STREAM_VARIABLE_RATE                     PA_STREAM_VARIABLE_RATE
#define MA_PA_STREAM_PEAK_DETECT                       PA_STREAM_PEAK_DETECT
#define MA_PA_STREAM_START_MUTED                       PA_STREAM_START_MUTED
#define MA_PA_STREAM_ADJUST_LATENCY                    PA_STREAM_ADJUST_LATENCY
#define MA_PA_STREAM_EARLY_REQUESTS                    PA_STREAM_EARLY_REQUESTS
#define MA_PA_STREAM_DONT_INHIBIT_AUTO_SUSPEND         PA_STREAM_DONT_INHIBIT_AUTO_SUSPEND
#define MA_PA_STREAM_START_UNMUTED                     PA_STREAM_START_UNMUTED
#define MA_PA_STREAM_FAIL_ON_SUSPEND                   PA_STREAM_FAIL_ON_SUSPEND
#define MA_PA_STREAM_RELATIVE_VOLUME                   PA_STREAM_RELATIVE_VOLUME
#define MA_PA_STREAM_PASSTHROUGH                       PA_STREAM_PASSTHROUGH

typedef pa_sink_flags_t ma_pa_sink_flags_t;
#define MA_PA_SINK_NOFLAGS                             PA_SINK_NOFLAGS
#define MA_PA_SINK_HW_VOLUME_CTRL                      PA_SINK_HW_VOLUME_CTRL
#define MA_PA_SINK_LATENCY                             PA_SINK_LATENCY
#define MA_PA_SINK_HARDWARE                            PA_SINK_HARDWARE
#define MA_PA_SINK_NETWORK                             PA_SINK_NETWORK
#define MA_PA_SINK_HW_MUTE_CTRL                        PA_SINK_HW_MUTE_CTRL
#define MA_PA_SINK_DECIBEL_VOLUME                      PA_SINK_DECIBEL_VOLUME
#define MA_PA_SINK_FLAT_VOLUME                         PA_SINK_FLAT_VOLUME
#define MA_PA_SINK_DYNAMIC_LATENCY                     PA_SINK_DYNAMIC_LATENCY
#define MA_PA_SINK_SET_FORMATS                         PA_SINK_SET_FORMATS

typedef pa_source_flags_t ma_pa_source_flags_t;
#define MA_PA_SOURCE_NOFLAGS                           PA_SOURCE_NOFLAGS
#define MA_PA_SOURCE_HW_VOLUME_CTRL                    PA_SOURCE_HW_VOLUME_CTRL
#define MA_PA_SOURCE_LATENCY                           PA_SOURCE_LATENCY
#define MA_PA_SOURCE_HARDWARE                          PA_SOURCE_HARDWARE
#define MA_PA_SOURCE_NETWORK                           PA_SOURCE_NETWORK
#define MA_PA_SOURCE_HW_MUTE_CTRL                      PA_SOURCE_HW_MUTE_CTRL
#define MA_PA_SOURCE_DECIBEL_VOLUME                    PA_SOURCE_DECIBEL_VOLUME
#define MA_PA_SOURCE_DYNAMIC_LATENCY                   PA_SOURCE_DYNAMIC_LATENCY
#define MA_PA_SOURCE_FLAT_VOLUME                       PA_SOURCE_FLAT_VOLUME

typedef pa_context_state_t ma_pa_context_state_t;
#define MA_PA_CONTEXT_UNCONNECTED                      PA_CONTEXT_UNCONNECTED
#define MA_PA_CONTEXT_CONNECTING                       PA_CONTEXT_CONNECTING
#define MA_PA_CONTEXT_AUTHORIZING                      PA_CONTEXT_AUTHORIZING
#define MA_PA_CONTEXT_SETTING_NAME                     PA_CONTEXT_SETTING_NAME
#define MA_PA_CONTEXT_READY                            PA_CONTEXT_READY
#define MA_PA_CONTEXT_FAILED                           PA_CONTEXT_FAILED
#define MA_PA_CONTEXT_TERMINATED                       PA_CONTEXT_TERMINATED

typedef pa_stream_state_t ma_pa_stream_state_t;
#define MA_PA_STREAM_UNCONNECTED                       PA_STREAM_UNCONNECTED
#define MA_PA_STREAM_CREATING                          PA_STREAM_CREATING
#define MA_PA_STREAM_READY                             PA_STREAM_READY
#define MA_PA_STREAM_FAILED                            PA_STREAM_FAILED
#define MA_PA_STREAM_TERMINATED                        PA_STREAM_TERMINATED

typedef pa_operation_state_t ma_pa_operation_state_t;
#define MA_PA_OPERATION_RUNNING                        PA_OPERATION_RUNNING
#define MA_PA_OPERATION_DONE                           PA_OPERATION_DONE
#define MA_PA_OPERATION_CANCELLED                      PA_OPERATION_CANCELLED

typedef pa_sink_state_t ma_pa_sink_state_t;
#define MA_PA_SINK_INVALID_STATE                       PA_SINK_INVALID_STATE
#define MA_PA_SINK_RUNNING                             PA_SINK_RUNNING
#define MA_PA_SINK_IDLE                                PA_SINK_IDLE
#define MA_PA_SINK_SUSPENDED                           PA_SINK_SUSPENDED

typedef pa_source_state_t ma_pa_source_state_t;
#define MA_PA_SOURCE_INVALID_STATE                     PA_SOURCE_INVALID_STATE
#define MA_PA_SOURCE_RUNNING                           PA_SOURCE_RUNNING
#define MA_PA_SOURCE_IDLE                              PA_SOURCE_IDLE
#define MA_PA_SOURCE_SUSPENDED                         PA_SOURCE_SUSPENDED

typedef pa_seek_mode_t ma_pa_seek_mode_t;
#define MA_PA_SEEK_RELATIVE                            PA_SEEK_RELATIVE
#define MA_PA_SEEK_ABSOLUTE                            PA_SEEK_ABSOLUTE
#define MA_PA_SEEK_RELATIVE_ON_READ                    PA_SEEK_RELATIVE_ON_READ
#define MA_PA_SEEK_RELATIVE_END                        PA_SEEK_RELATIVE_END

typedef pa_channel_position_t ma_pa_channel_position_t;
#define MA_PA_CHANNEL_POSITION_INVALID                 PA_CHANNEL_POSITION_INVALID
#define MA_PA_CHANNEL_POSITION_MONO                    PA_CHANNEL_POSITION_MONO
#define MA_PA_CHANNEL_POSITION_FRONT_LEFT              PA_CHANNEL_POSITION_FRONT_LEFT
#define MA_PA_CHANNEL_POSITION_FRONT_RIGHT             PA_CHANNEL_POSITION_FRONT_RIGHT
#define MA_PA_CHANNEL_POSITION_FRONT_CENTER            PA_CHANNEL_POSITION_FRONT_CENTER
#define MA_PA_CHANNEL_POSITION_REAR_CENTER             PA_CHANNEL_POSITION_REAR_CENTER
#define MA_PA_CHANNEL_POSITION_REAR_LEFT               PA_CHANNEL_POSITION_REAR_LEFT
#define MA_PA_CHANNEL_POSITION_REAR_RIGHT              PA_CHANNEL_POSITION_REAR_RIGHT
#define MA_PA_CHANNEL_POSITION_LFE                     PA_CHANNEL_POSITION_LFE
#define MA_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER    PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER
#define MA_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER   PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER
#define MA_PA_CHANNEL_POSITION_SIDE_LEFT               PA_CHANNEL_POSITION_SIDE_LEFT
#define MA_PA_CHANNEL_POSITION_SIDE_RIGHT              PA_CHANNEL_POSITION_SIDE_RIGHT
#define MA_PA_CHANNEL_POSITION_AUX0                    PA_CHANNEL_POSITION_AUX0
#define MA_PA_CHANNEL_POSITION_AUX1                    PA_CHANNEL_POSITION_AUX1
#define MA_PA_CHANNEL_POSITION_AUX2                    PA_CHANNEL_POSITION_AUX2
#define MA_PA_CHANNEL_POSITION_AUX3                    PA_CHANNEL_POSITION_AUX3
#define MA_PA_CHANNEL_POSITION_AUX4                    PA_CHANNEL_POSITION_AUX4
#define MA_PA_CHANNEL_POSITION_AUX5                    PA_CHANNEL_POSITION_AUX5
#define MA_PA_CHANNEL_POSITION_AUX6                    PA_CHANNEL_POSITION_AUX6
#define MA_PA_CHANNEL_POSITION_AUX7                    PA_CHANNEL_POSITION_AUX7
#define MA_PA_CHANNEL_POSITION_AUX8                    PA_CHANNEL_POSITION_AUX8
#define MA_PA_CHANNEL_POSITION_AUX9                    PA_CHANNEL_POSITION_AUX9
#define MA_PA_CHANNEL_POSITION_AUX10                   PA_CHANNEL_POSITION_AUX10
#define MA_PA_CHANNEL_POSITION_AUX11                   PA_CHANNEL_POSITION_AUX11
#define MA_PA_CHANNEL_POSITION_AUX12                   PA_CHANNEL_POSITION_AUX12
#define MA_PA_CHANNEL_POSITION_AUX13                   PA_CHANNEL_POSITION_AUX13
#define MA_PA_CHANNEL_POSITION_AUX14                   PA_CHANNEL_POSITION_AUX14
#define MA_PA_CHANNEL_POSITION_AUX15                   PA_CHANNEL_POSITION_AUX15
#define MA_PA_CHANNEL_POSITION_AUX16                   PA_CHANNEL_POSITION_AUX16
#define MA_PA_CHANNEL_POSITION_AUX17                   PA_CHANNEL_POSITION_AUX17
#define MA_PA_CHANNEL_POSITION_AUX18                   PA_CHANNEL_POSITION_AUX18
#define MA_PA_CHANNEL_POSITION_AUX19                   PA_CHANNEL_POSITION_AUX19
#define MA_PA_CHANNEL_POSITION_AUX20                   PA_CHANNEL_POSITION_AUX20
#define MA_PA_CHANNEL_POSITION_AUX21                   PA_CHANNEL_POSITION_AUX21
#define MA_PA_CHANNEL_POSITION_AUX22                   PA_CHANNEL_POSITION_AUX22
#define MA_PA_CHANNEL_POSITION_AUX23                   PA_CHANNEL_POSITION_AUX23
#define MA_PA_CHANNEL_POSITION_AUX24                   PA_CHANNEL_POSITION_AUX24
#define MA_PA_CHANNEL_POSITION_AUX25                   PA_CHANNEL_POSITION_AUX25
#define MA_PA_CHANNEL_POSITION_AUX26                   PA_CHANNEL_POSITION_AUX26
#define MA_PA_CHANNEL_POSITION_AUX27                   PA_CHANNEL_POSITION_AUX27
#define MA_PA_CHANNEL_POSITION_AUX28                   PA_CHANNEL_POSITION_AUX28
#define MA_PA_CHANNEL_POSITION_AUX29                   PA_CHANNEL_POSITION_AUX29
#define MA_PA_CHANNEL_POSITION_AUX30                   PA_CHANNEL_POSITION_AUX30
#define MA_PA_CHANNEL_POSITION_AUX31                   PA_CHANNEL_POSITION_AUX31
#define MA_PA_CHANNEL_POSITION_TOP_CENTER              PA_CHANNEL_POSITION_TOP_CENTER
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_LEFT          PA_CHANNEL_POSITION_TOP_FRONT_LEFT
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT         PA_CHANNEL_POSITION_TOP_FRONT_RIGHT
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_CENTER        PA_CHANNEL_POSITION_TOP_FRONT_CENTER
#define MA_PA_CHANNEL_POSITION_TOP_REAR_LEFT           PA_CHANNEL_POSITION_TOP_REAR_LEFT
#define MA_PA_CHANNEL_POSITION_TOP_REAR_RIGHT          PA_CHANNEL_POSITION_TOP_REAR_RIGHT
#define MA_PA_CHANNEL_POSITION_TOP_REAR_CENTER         PA_CHANNEL_POSITION_TOP_REAR_CENTER
#define MA_PA_CHANNEL_POSITION_LEFT                    PA_CHANNEL_POSITION_LEFT
#define MA_PA_CHANNEL_POSITION_RIGHT                   PA_CHANNEL_POSITION_RIGHT
#define MA_PA_CHANNEL_POSITION_CENTER                  PA_CHANNEL_POSITION_CENTER
#define MA_PA_CHANNEL_POSITION_SUBWOOFER               PA_CHANNEL_POSITION_SUBWOOFER

typedef pa_channel_map_def_t ma_pa_channel_map_def_t;
#define MA_PA_CHANNEL_MAP_AIFF                         PA_CHANNEL_MAP_AIFF
#define MA_PA_CHANNEL_MAP_ALSA                         PA_CHANNEL_MAP_ALSA
#define MA_PA_CHANNEL_MAP_AUX                          PA_CHANNEL_MAP_AUX
#define MA_PA_CHANNEL_MAP_WAVEEX                       PA_CHANNEL_MAP_WAVEEX
#define MA_PA_CHANNEL_MAP_OSS                          PA_CHANNEL_MAP_OSS
#define MA_PA_CHANNEL_MAP_DEFAULT                      PA_CHANNEL_MAP_DEFAULT

typedef pa_sample_format_t ma_pa_sample_format_t;
#define MA_PA_SAMPLE_INVALID                           PA_SAMPLE_INVALID
#define MA_PA_SAMPLE_U8                                PA_SAMPLE_U8
#define MA_PA_SAMPLE_ALAW                              PA_SAMPLE_ALAW
#define MA_PA_SAMPLE_ULAW                              PA_SAMPLE_ULAW
#define MA_PA_SAMPLE_S16LE                             PA_SAMPLE_S16LE
#define MA_PA_SAMPLE_S16BE                             PA_SAMPLE_S16BE
#define MA_PA_SAMPLE_FLOAT32LE                         PA_SAMPLE_FLOAT32LE
#define MA_PA_SAMPLE_FLOAT32BE                         PA_SAMPLE_FLOAT32BE
#define MA_PA_SAMPLE_S32LE                             PA_SAMPLE_S32LE
#define MA_PA_SAMPLE_S32BE                             PA_SAMPLE_S32BE
#define MA_PA_SAMPLE_S24LE                             PA_SAMPLE_S24LE
#define MA_PA_SAMPLE_S24BE                             PA_SAMPLE_S24BE
#define MA_PA_SAMPLE_S24_32LE                          PA_SAMPLE_S24_32LE
#define MA_PA_SAMPLE_S24_32BE                          PA_SAMPLE_S24_32BE

typedef pa_mainloop     ma_pa_mainloop;
typedef pa_mainloop_api ma_pa_mainloop_api;
typedef pa_context      ma_pa_context;
typedef pa_operation    ma_pa_operation;
typedef pa_stream       ma_pa_stream;
typedef pa_spawn_api    ma_pa_spawn_api;
typedef pa_buffer_attr  ma_pa_buffer_attr;
typedef pa_channel_map  ma_pa_channel_map;
typedef pa_cvolume      ma_pa_cvolume;
typedef pa_sample_spec  ma_pa_sample_spec;
typedef pa_sink_info    ma_pa_sink_info;
typedef pa_source_info  ma_pa_source_info;

typedef pa_context_notify_cb_t ma_pa_context_notify_cb_t;
typedef pa_sink_info_cb_t ma_pa_sink_info_cb_t;
typedef pa_source_info_cb_t ma_pa_source_info_cb_t;
typedef pa_stream_success_cb_t ma_pa_stream_success_cb_t;
typedef pa_stream_request_cb_t ma_pa_stream_request_cb_t;
typedef pa_free_cb_t ma_pa_free_cb_t;
#else
#define MA_PA_OK                                       0
#define MA_PA_ERR_ACCESS                               1
#define MA_PA_ERR_INVALID                              2
#define MA_PA_ERR_NOENTITY                             5

#define MA_PA_CHANNELS_MAX                             32
#define MA_PA_RATE_MAX                                 384000

typedef int ma_pa_context_flags_t;
#define MA_PA_CONTEXT_NOFLAGS                          0x00000000
#define MA_PA_CONTEXT_NOAUTOSPAWN                      0x00000001
#define MA_PA_CONTEXT_NOFAIL                           0x00000002

typedef int ma_pa_stream_flags_t;
#define MA_PA_STREAM_NOFLAGS                           0x00000000
#define MA_PA_STREAM_START_CORKED                      0x00000001
#define MA_PA_STREAM_INTERPOLATE_TIMING                0x00000002
#define MA_PA_STREAM_NOT_MONOTONIC                     0x00000004
#define MA_PA_STREAM_AUTO_TIMING_UPDATE                0x00000008
#define MA_PA_STREAM_NO_REMAP_CHANNELS                 0x00000010
#define MA_PA_STREAM_NO_REMIX_CHANNELS                 0x00000020
#define MA_PA_STREAM_FIX_FORMAT                        0x00000040
#define MA_PA_STREAM_FIX_RATE                          0x00000080
#define MA_PA_STREAM_FIX_CHANNELS                      0x00000100
#define MA_PA_STREAM_DONT_MOVE                         0x00000200
#define MA_PA_STREAM_VARIABLE_RATE                     0x00000400
#define MA_PA_STREAM_PEAK_DETECT                       0x00000800
#define MA_PA_STREAM_START_MUTED                       0x00001000
#define MA_PA_STREAM_ADJUST_LATENCY                    0x00002000
#define MA_PA_STREAM_EARLY_REQUESTS                    0x00004000
#define MA_PA_STREAM_DONT_INHIBIT_AUTO_SUSPEND         0x00008000
#define MA_PA_STREAM_START_UNMUTED                     0x00010000
#define MA_PA_STREAM_FAIL_ON_SUSPEND                   0x00020000
#define MA_PA_STREAM_RELATIVE_VOLUME                   0x00040000
#define MA_PA_STREAM_PASSTHROUGH                       0x00080000

typedef int ma_pa_sink_flags_t;
#define MA_PA_SINK_NOFLAGS                             0x00000000
#define MA_PA_SINK_HW_VOLUME_CTRL                      0x00000001
#define MA_PA_SINK_LATENCY                             0x00000002
#define MA_PA_SINK_HARDWARE                            0x00000004
#define MA_PA_SINK_NETWORK                             0x00000008
#define MA_PA_SINK_HW_MUTE_CTRL                        0x00000010
#define MA_PA_SINK_DECIBEL_VOLUME                      0x00000020
#define MA_PA_SINK_FLAT_VOLUME                         0x00000040
#define MA_PA_SINK_DYNAMIC_LATENCY                     0x00000080
#define MA_PA_SINK_SET_FORMATS                         0x00000100

typedef int ma_pa_source_flags_t;
#define MA_PA_SOURCE_NOFLAGS                           0x00000000
#define MA_PA_SOURCE_HW_VOLUME_CTRL                    0x00000001
#define MA_PA_SOURCE_LATENCY                           0x00000002
#define MA_PA_SOURCE_HARDWARE                          0x00000004
#define MA_PA_SOURCE_NETWORK                           0x00000008
#define MA_PA_SOURCE_HW_MUTE_CTRL                      0x00000010
#define MA_PA_SOURCE_DECIBEL_VOLUME                    0x00000020
#define MA_PA_SOURCE_DYNAMIC_LATENCY                   0x00000040
#define MA_PA_SOURCE_FLAT_VOLUME                       0x00000080

typedef int ma_pa_context_state_t;
#define MA_PA_CONTEXT_UNCONNECTED                      0
#define MA_PA_CONTEXT_CONNECTING                       1
#define MA_PA_CONTEXT_AUTHORIZING                      2
#define MA_PA_CONTEXT_SETTING_NAME                     3
#define MA_PA_CONTEXT_READY                            4
#define MA_PA_CONTEXT_FAILED                           5
#define MA_PA_CONTEXT_TERMINATED                       6

typedef int ma_pa_stream_state_t;
#define MA_PA_STREAM_UNCONNECTED                       0
#define MA_PA_STREAM_CREATING                          1
#define MA_PA_STREAM_READY                             2
#define MA_PA_STREAM_FAILED                            3
#define MA_PA_STREAM_TERMINATED                        4

typedef int ma_pa_operation_state_t;
#define MA_PA_OPERATION_RUNNING                        0
#define MA_PA_OPERATION_DONE                           1
#define MA_PA_OPERATION_CANCELLED                      2

typedef int ma_pa_sink_state_t;
#define MA_PA_SINK_INVALID_STATE                       -1
#define MA_PA_SINK_RUNNING                             0
#define MA_PA_SINK_IDLE                                1
#define MA_PA_SINK_SUSPENDED                           2

typedef int ma_pa_source_state_t;
#define MA_PA_SOURCE_INVALID_STATE                     -1
#define MA_PA_SOURCE_RUNNING                           0
#define MA_PA_SOURCE_IDLE                              1
#define MA_PA_SOURCE_SUSPENDED                         2

typedef int ma_pa_seek_mode_t;
#define MA_PA_SEEK_RELATIVE                            0
#define MA_PA_SEEK_ABSOLUTE                            1
#define MA_PA_SEEK_RELATIVE_ON_READ                    2
#define MA_PA_SEEK_RELATIVE_END                        3

typedef int ma_pa_channel_position_t;
#define MA_PA_CHANNEL_POSITION_INVALID                 -1
#define MA_PA_CHANNEL_POSITION_MONO                    0
#define MA_PA_CHANNEL_POSITION_FRONT_LEFT              1
#define MA_PA_CHANNEL_POSITION_FRONT_RIGHT             2
#define MA_PA_CHANNEL_POSITION_FRONT_CENTER            3
#define MA_PA_CHANNEL_POSITION_REAR_CENTER             4
#define MA_PA_CHANNEL_POSITION_REAR_LEFT               5
#define MA_PA_CHANNEL_POSITION_REAR_RIGHT              6
#define MA_PA_CHANNEL_POSITION_LFE                     7
#define MA_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER    8
#define MA_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER   9
#define MA_PA_CHANNEL_POSITION_SIDE_LEFT               10
#define MA_PA_CHANNEL_POSITION_SIDE_RIGHT              11
#define MA_PA_CHANNEL_POSITION_AUX0                    12
#define MA_PA_CHANNEL_POSITION_AUX1                    13
#define MA_PA_CHANNEL_POSITION_AUX2                    14
#define MA_PA_CHANNEL_POSITION_AUX3                    15
#define MA_PA_CHANNEL_POSITION_AUX4                    16
#define MA_PA_CHANNEL_POSITION_AUX5                    17
#define MA_PA_CHANNEL_POSITION_AUX6                    18
#define MA_PA_CHANNEL_POSITION_AUX7                    19
#define MA_PA_CHANNEL_POSITION_AUX8                    20
#define MA_PA_CHANNEL_POSITION_AUX9                    21
#define MA_PA_CHANNEL_POSITION_AUX10                   22
#define MA_PA_CHANNEL_POSITION_AUX11                   23
#define MA_PA_CHANNEL_POSITION_AUX12                   24
#define MA_PA_CHANNEL_POSITION_AUX13                   25
#define MA_PA_CHANNEL_POSITION_AUX14                   26
#define MA_PA_CHANNEL_POSITION_AUX15                   27
#define MA_PA_CHANNEL_POSITION_AUX16                   28
#define MA_PA_CHANNEL_POSITION_AUX17                   29
#define MA_PA_CHANNEL_POSITION_AUX18                   30
#define MA_PA_CHANNEL_POSITION_AUX19                   31
#define MA_PA_CHANNEL_POSITION_AUX20                   32
#define MA_PA_CHANNEL_POSITION_AUX21                   33
#define MA_PA_CHANNEL_POSITION_AUX22                   34
#define MA_PA_CHANNEL_POSITION_AUX23                   35
#define MA_PA_CHANNEL_POSITION_AUX24                   36
#define MA_PA_CHANNEL_POSITION_AUX25                   37
#define MA_PA_CHANNEL_POSITION_AUX26                   38
#define MA_PA_CHANNEL_POSITION_AUX27                   39
#define MA_PA_CHANNEL_POSITION_AUX28                   40
#define MA_PA_CHANNEL_POSITION_AUX29                   41
#define MA_PA_CHANNEL_POSITION_AUX30                   42
#define MA_PA_CHANNEL_POSITION_AUX31                   43
#define MA_PA_CHANNEL_POSITION_TOP_CENTER              44
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_LEFT          45
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT         46
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_CENTER        47
#define MA_PA_CHANNEL_POSITION_TOP_REAR_LEFT           48
#define MA_PA_CHANNEL_POSITION_TOP_REAR_RIGHT          49
#define MA_PA_CHANNEL_POSITION_TOP_REAR_CENTER         50
#define MA_PA_CHANNEL_POSITION_LEFT                    MA_PA_CHANNEL_POSITION_FRONT_LEFT
#define MA_PA_CHANNEL_POSITION_RIGHT                   MA_PA_CHANNEL_POSITION_FRONT_RIGHT
#define MA_PA_CHANNEL_POSITION_CENTER                  MA_PA_CHANNEL_POSITION_FRONT_CENTER
#define MA_PA_CHANNEL_POSITION_SUBWOOFER               MA_PA_CHANNEL_POSITION_LFE

typedef int ma_pa_channel_map_def_t;
#define MA_PA_CHANNEL_MAP_AIFF                         0
#define MA_PA_CHANNEL_MAP_ALSA                         1
#define MA_PA_CHANNEL_MAP_AUX                          2
#define MA_PA_CHANNEL_MAP_WAVEEX                       3
#define MA_PA_CHANNEL_MAP_OSS                          4
#define MA_PA_CHANNEL_MAP_DEFAULT                      MA_PA_CHANNEL_MAP_AIFF

typedef int ma_pa_sample_format_t;
#define MA_PA_SAMPLE_INVALID                           -1
#define MA_PA_SAMPLE_U8                                0
#define MA_PA_SAMPLE_ALAW                              1
#define MA_PA_SAMPLE_ULAW                              2
#define MA_PA_SAMPLE_S16LE                             3
#define MA_PA_SAMPLE_S16BE                             4
#define MA_PA_SAMPLE_FLOAT32LE                         5
#define MA_PA_SAMPLE_FLOAT32BE                         6
#define MA_PA_SAMPLE_S32LE                             7
#define MA_PA_SAMPLE_S32BE                             8
#define MA_PA_SAMPLE_S24LE                             9
#define MA_PA_SAMPLE_S24BE                             10
#define MA_PA_SAMPLE_S24_32LE                          11
#define MA_PA_SAMPLE_S24_32BE                          12

typedef struct ma_pa_mainloop     ma_pa_mainloop;
typedef struct ma_pa_mainloop_api ma_pa_mainloop_api;
typedef struct ma_pa_context      ma_pa_context;
typedef struct ma_pa_operation    ma_pa_operation;
typedef struct ma_pa_stream       ma_pa_stream;
typedef struct ma_pa_spawn_api    ma_pa_spawn_api;

typedef struct
{
    ma_uint32 maxlength;
    ma_uint32 tlength;
    ma_uint32 prebuf;
    ma_uint32 minreq;
    ma_uint32 fragsize;
} ma_pa_buffer_attr;

typedef struct
{
    ma_uint8 channels;
    ma_pa_channel_position_t map[MA_PA_CHANNELS_MAX];
} ma_pa_channel_map;

typedef struct
{
    ma_uint8 channels;
    ma_uint32 values[MA_PA_CHANNELS_MAX];
} ma_pa_cvolume;

typedef struct
{
    ma_pa_sample_format_t format;
    ma_uint32 rate;
    ma_uint8 channels;
} ma_pa_sample_spec;

typedef struct
{
    const char* name;
    ma_uint32 index;
    const char* description;
    ma_pa_sample_spec sample_spec;
    ma_pa_channel_map channel_map;
    ma_uint32 owner_module;
    ma_pa_cvolume volume;
    int mute;
    ma_uint32 monitor_source;
    const char* monitor_source_name;
    ma_uint64 latency;
    const char* driver;
    ma_pa_sink_flags_t flags;
    void* proplist;
    ma_uint64 configured_latency;
    ma_uint32 base_volume;
    ma_pa_sink_state_t state;
    ma_uint32 n_volume_steps;
    ma_uint32 card;
    ma_uint32 n_ports;
    void** ports;
    void* active_port;
    ma_uint8 n_formats;
    void** formats;
} ma_pa_sink_info;

typedef struct
{
    const char *name;
    ma_uint32 index;
    const char *description;
    ma_pa_sample_spec sample_spec;
    ma_pa_channel_map channel_map;
    ma_uint32 owner_module;
    ma_pa_cvolume volume;
    int mute;
    ma_uint32 monitor_of_sink;
    const char *monitor_of_sink_name;
    ma_uint64 latency;
    const char *driver;
    ma_pa_source_flags_t flags;
    void* proplist;
    ma_uint64 configured_latency;
    ma_uint32 base_volume;
    ma_pa_source_state_t state;
    ma_uint32 n_volume_steps;
    ma_uint32 card;
    ma_uint32 n_ports;
    void** ports;
    void* active_port;
    ma_uint8 n_formats;
    void** formats;
} ma_pa_source_info;

typedef void (* ma_pa_context_notify_cb_t)(ma_pa_context* c, void* userdata);
typedef void (* ma_pa_sink_info_cb_t)     (ma_pa_context* c, const ma_pa_sink_info* i, int eol, void* userdata);
typedef void (* ma_pa_source_info_cb_t)   (ma_pa_context* c, const ma_pa_source_info* i, int eol, void* userdata);
typedef void (* ma_pa_stream_success_cb_t)(ma_pa_stream* s, int success, void* userdata);
typedef void (* ma_pa_stream_request_cb_t)(ma_pa_stream* s, size_t nbytes, void* userdata);
typedef void (* ma_pa_free_cb_t)          (void* p);
#endif


typedef ma_pa_mainloop*          (* ma_pa_mainloop_new_proc)                   (void);
typedef void                     (* ma_pa_mainloop_free_proc)                  (ma_pa_mainloop* m);
typedef ma_pa_mainloop_api*      (* ma_pa_mainloop_get_api_proc)               (ma_pa_mainloop* m);
typedef int                      (* ma_pa_mainloop_iterate_proc)               (ma_pa_mainloop* m, int block, int* retval);
typedef void                     (* ma_pa_mainloop_wakeup_proc)                (ma_pa_mainloop* m);
typedef ma_pa_context*           (* ma_pa_context_new_proc)                    (ma_pa_mainloop_api* mainloop, const char* name);
typedef void                     (* ma_pa_context_unref_proc)                  (ma_pa_context* c);
typedef int                      (* ma_pa_context_connect_proc)                (ma_pa_context* c, const char* server, ma_pa_context_flags_t flags, const ma_pa_spawn_api* api);
typedef void                     (* ma_pa_context_disconnect_proc)             (ma_pa_context* c);
typedef void                     (* ma_pa_context_set_state_callback_proc)     (ma_pa_context* c, ma_pa_context_notify_cb_t cb, void* userdata);
typedef ma_pa_context_state_t    (* ma_pa_context_get_state_proc)              (ma_pa_context* c);
typedef ma_pa_operation*         (* ma_pa_context_get_sink_info_list_proc)     (ma_pa_context* c, ma_pa_sink_info_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_context_get_source_info_list_proc)   (ma_pa_context* c, ma_pa_source_info_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_context_get_sink_info_by_name_proc)  (ma_pa_context* c, const char* name, ma_pa_sink_info_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_context_get_source_info_by_name_proc)(ma_pa_context* c, const char* name, ma_pa_source_info_cb_t cb, void* userdata);
typedef void                     (* ma_pa_operation_unref_proc)                (ma_pa_operation* o);
typedef ma_pa_operation_state_t  (* ma_pa_operation_get_state_proc)            (ma_pa_operation* o);
typedef ma_pa_channel_map*       (* ma_pa_channel_map_init_extend_proc)        (ma_pa_channel_map* m, unsigned channels, ma_pa_channel_map_def_t def);
typedef int                      (* ma_pa_channel_map_valid_proc)              (const ma_pa_channel_map* m);
typedef int                      (* ma_pa_channel_map_compatible_proc)         (const ma_pa_channel_map* m, const ma_pa_sample_spec* ss);
typedef ma_pa_stream*            (* ma_pa_stream_new_proc)                     (ma_pa_context* c, const char* name, const ma_pa_sample_spec* ss, const ma_pa_channel_map* map);
typedef void                     (* ma_pa_stream_unref_proc)                   (ma_pa_stream* s);
typedef int                      (* ma_pa_stream_connect_playback_proc)        (ma_pa_stream* s, const char* dev, const ma_pa_buffer_attr* attr, ma_pa_stream_flags_t flags, const ma_pa_cvolume* volume, ma_pa_stream* sync_stream);
typedef int                      (* ma_pa_stream_connect_record_proc)          (ma_pa_stream* s, const char* dev, const ma_pa_buffer_attr* attr, ma_pa_stream_flags_t flags);
typedef int                      (* ma_pa_stream_disconnect_proc)              (ma_pa_stream* s);
typedef ma_pa_stream_state_t     (* ma_pa_stream_get_state_proc)               (ma_pa_stream* s);
typedef const ma_pa_sample_spec* (* ma_pa_stream_get_sample_spec_proc)         (ma_pa_stream* s);
typedef const ma_pa_channel_map* (* ma_pa_stream_get_channel_map_proc)         (ma_pa_stream* s);
typedef const ma_pa_buffer_attr* (* ma_pa_stream_get_buffer_attr_proc)         (ma_pa_stream* s);
typedef ma_pa_operation*         (* ma_pa_stream_set_buffer_attr_proc)         (ma_pa_stream* s, const ma_pa_buffer_attr* attr, ma_pa_stream_success_cb_t cb, void* userdata);
typedef const char*              (* ma_pa_stream_get_device_name_proc)         (ma_pa_stream* s);
typedef void                     (* ma_pa_stream_set_write_callback_proc)      (ma_pa_stream* s, ma_pa_stream_request_cb_t cb, void* userdata);
typedef void                     (* ma_pa_stream_set_read_callback_proc)       (ma_pa_stream* s, ma_pa_stream_request_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_stream_flush_proc)                   (ma_pa_stream* s, ma_pa_stream_success_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_stream_drain_proc)                   (ma_pa_stream* s, ma_pa_stream_success_cb_t cb, void* userdata);
typedef int                      (* ma_pa_stream_is_corked_proc)               (ma_pa_stream* s);
typedef ma_pa_operation*         (* ma_pa_stream_cork_proc)                    (ma_pa_stream* s, int b, ma_pa_stream_success_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_stream_trigger_proc)                 (ma_pa_stream* s, ma_pa_stream_success_cb_t cb, void* userdata);
typedef int                      (* ma_pa_stream_begin_write_proc)             (ma_pa_stream* s, void** data, size_t* nbytes);
typedef int                      (* ma_pa_stream_write_proc)                   (ma_pa_stream* s, const void* data, size_t nbytes, ma_pa_free_cb_t free_cb, int64_t offset, ma_pa_seek_mode_t seek);
typedef int                      (* ma_pa_stream_peek_proc)                    (ma_pa_stream* s, const void** data, size_t* nbytes);
typedef int                      (* ma_pa_stream_drop_proc)                    (ma_pa_stream* s);
typedef size_t                   (* ma_pa_stream_writable_size_proc)           (ma_pa_stream* s);
typedef size_t                   (* ma_pa_stream_readable_size_proc)           (ma_pa_stream* s);

typedef struct
{
    ma_uint32 count;
    ma_uint32 capacity;
    ma_device_info* pInfo;
} ma_pulse_device_enum_data;

static ma_result ma_result_from_pulse(int result)
{
    switch (result) {
        case MA_PA_OK:           return MA_SUCCESS;
        case MA_PA_ERR_ACCESS:   return MA_ACCESS_DENIED;
        case MA_PA_ERR_INVALID:  return MA_INVALID_ARGS;
        case MA_PA_ERR_NOENTITY: return MA_NO_DEVICE;
        default:                 return MA_ERROR;
    }
}

#if 0
static ma_pa_sample_format_t ma_format_to_pulse(ma_format format)
{
    if (ma_is_little_endian()) {
        switch (format) {
            case ma_format_s16: return MA_PA_SAMPLE_S16LE;
            case ma_format_s24: return MA_PA_SAMPLE_S24LE;
            case ma_format_s32: return MA_PA_SAMPLE_S32LE;
            case ma_format_f32: return MA_PA_SAMPLE_FLOAT32LE;
            default: break;
        }
    } else {
        switch (format) {
            case ma_format_s16: return MA_PA_SAMPLE_S16BE;
            case ma_format_s24: return MA_PA_SAMPLE_S24BE;
            case ma_format_s32: return MA_PA_SAMPLE_S32BE;
            case ma_format_f32: return MA_PA_SAMPLE_FLOAT32BE;
            default: break;
        }
    }

    /* Endian agnostic. */
    switch (format) {
        case ma_format_u8: return MA_PA_SAMPLE_U8;
        default: return MA_PA_SAMPLE_INVALID;
    }
}
#endif

static ma_format ma_format_from_pulse(ma_pa_sample_format_t format)
{
    if (ma_is_little_endian()) {
        switch (format) {
            case MA_PA_SAMPLE_S16LE:     return ma_format_s16;
            case MA_PA_SAMPLE_S24LE:     return ma_format_s24;
            case MA_PA_SAMPLE_S32LE:     return ma_format_s32;
            case MA_PA_SAMPLE_FLOAT32LE: return ma_format_f32;
            default: break;
        }
    } else {
        switch (format) {
            case MA_PA_SAMPLE_S16BE:     return ma_format_s16;
            case MA_PA_SAMPLE_S24BE:     return ma_format_s24;
            case MA_PA_SAMPLE_S32BE:     return ma_format_s32;
            case MA_PA_SAMPLE_FLOAT32BE: return ma_format_f32;
            default: break;
        }
    }

    /* Endian agnostic. */
    switch (format) {
        case MA_PA_SAMPLE_U8: return ma_format_u8;
        default: return ma_format_unknown;
    }
}

static ma_channel ma_channel_position_from_pulse(ma_pa_channel_position_t position)
{
    switch (position)
    {
        case MA_PA_CHANNEL_POSITION_INVALID:               return MA_CHANNEL_NONE;
        case MA_PA_CHANNEL_POSITION_MONO:                  return MA_CHANNEL_MONO;
        case MA_PA_CHANNEL_POSITION_FRONT_LEFT:            return MA_CHANNEL_FRONT_LEFT;
        case MA_PA_CHANNEL_POSITION_FRONT_RIGHT:           return MA_CHANNEL_FRONT_RIGHT;
        case MA_PA_CHANNEL_POSITION_FRONT_CENTER:          return MA_CHANNEL_FRONT_CENTER;
        case MA_PA_CHANNEL_POSITION_REAR_CENTER:           return MA_CHANNEL_BACK_CENTER;
        case MA_PA_CHANNEL_POSITION_REAR_LEFT:             return MA_CHANNEL_BACK_LEFT;
        case MA_PA_CHANNEL_POSITION_REAR_RIGHT:            return MA_CHANNEL_BACK_RIGHT;
        case MA_PA_CHANNEL_POSITION_LFE:                   return MA_CHANNEL_LFE;
        case MA_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER:  return MA_CHANNEL_FRONT_LEFT_CENTER;
        case MA_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER: return MA_CHANNEL_FRONT_RIGHT_CENTER;
        case MA_PA_CHANNEL_POSITION_SIDE_LEFT:             return MA_CHANNEL_SIDE_LEFT;
        case MA_PA_CHANNEL_POSITION_SIDE_RIGHT:            return MA_CHANNEL_SIDE_RIGHT;
        case MA_PA_CHANNEL_POSITION_AUX0:                  return MA_CHANNEL_AUX_0;
        case MA_PA_CHANNEL_POSITION_AUX1:                  return MA_CHANNEL_AUX_1;
        case MA_PA_CHANNEL_POSITION_AUX2:                  return MA_CHANNEL_AUX_2;
        case MA_PA_CHANNEL_POSITION_AUX3:                  return MA_CHANNEL_AUX_3;
        case MA_PA_CHANNEL_POSITION_AUX4:                  return MA_CHANNEL_AUX_4;
        case MA_PA_CHANNEL_POSITION_AUX5:                  return MA_CHANNEL_AUX_5;
        case MA_PA_CHANNEL_POSITION_AUX6:                  return MA_CHANNEL_AUX_6;
        case MA_PA_CHANNEL_POSITION_AUX7:                  return MA_CHANNEL_AUX_7;
        case MA_PA_CHANNEL_POSITION_AUX8:                  return MA_CHANNEL_AUX_8;
        case MA_PA_CHANNEL_POSITION_AUX9:                  return MA_CHANNEL_AUX_9;
        case MA_PA_CHANNEL_POSITION_AUX10:                 return MA_CHANNEL_AUX_10;
        case MA_PA_CHANNEL_POSITION_AUX11:                 return MA_CHANNEL_AUX_11;
        case MA_PA_CHANNEL_POSITION_AUX12:                 return MA_CHANNEL_AUX_12;
        case MA_PA_CHANNEL_POSITION_AUX13:                 return MA_CHANNEL_AUX_13;
        case MA_PA_CHANNEL_POSITION_AUX14:                 return MA_CHANNEL_AUX_14;
        case MA_PA_CHANNEL_POSITION_AUX15:                 return MA_CHANNEL_AUX_15;
        case MA_PA_CHANNEL_POSITION_AUX16:                 return MA_CHANNEL_AUX_16;
        case MA_PA_CHANNEL_POSITION_AUX17:                 return MA_CHANNEL_AUX_17;
        case MA_PA_CHANNEL_POSITION_AUX18:                 return MA_CHANNEL_AUX_18;
        case MA_PA_CHANNEL_POSITION_AUX19:                 return MA_CHANNEL_AUX_19;
        case MA_PA_CHANNEL_POSITION_AUX20:                 return MA_CHANNEL_AUX_20;
        case MA_PA_CHANNEL_POSITION_AUX21:                 return MA_CHANNEL_AUX_21;
        case MA_PA_CHANNEL_POSITION_AUX22:                 return MA_CHANNEL_AUX_22;
        case MA_PA_CHANNEL_POSITION_AUX23:                 return MA_CHANNEL_AUX_23;
        case MA_PA_CHANNEL_POSITION_AUX24:                 return MA_CHANNEL_AUX_24;
        case MA_PA_CHANNEL_POSITION_AUX25:                 return MA_CHANNEL_AUX_25;
        case MA_PA_CHANNEL_POSITION_AUX26:                 return MA_CHANNEL_AUX_26;
        case MA_PA_CHANNEL_POSITION_AUX27:                 return MA_CHANNEL_AUX_27;
        case MA_PA_CHANNEL_POSITION_AUX28:                 return MA_CHANNEL_AUX_28;
        case MA_PA_CHANNEL_POSITION_AUX29:                 return MA_CHANNEL_AUX_29;
        case MA_PA_CHANNEL_POSITION_AUX30:                 return MA_CHANNEL_AUX_30;
        case MA_PA_CHANNEL_POSITION_AUX31:                 return MA_CHANNEL_AUX_31;
        case MA_PA_CHANNEL_POSITION_TOP_CENTER:            return MA_CHANNEL_TOP_CENTER;
        case MA_PA_CHANNEL_POSITION_TOP_FRONT_LEFT:        return MA_CHANNEL_TOP_FRONT_LEFT;
        case MA_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT:       return MA_CHANNEL_TOP_FRONT_RIGHT;
        case MA_PA_CHANNEL_POSITION_TOP_FRONT_CENTER:      return MA_CHANNEL_TOP_FRONT_CENTER;
        case MA_PA_CHANNEL_POSITION_TOP_REAR_LEFT:         return MA_CHANNEL_TOP_BACK_LEFT;
        case MA_PA_CHANNEL_POSITION_TOP_REAR_RIGHT:        return MA_CHANNEL_TOP_BACK_RIGHT;
        case MA_PA_CHANNEL_POSITION_TOP_REAR_CENTER:       return MA_CHANNEL_TOP_BACK_CENTER;
        default: return MA_CHANNEL_NONE;
    }
}

#if 0
static ma_pa_channel_position_t ma_channel_position_to_pulse(ma_channel position)
{
    switch (position)
    {
        case MA_CHANNEL_NONE:               return MA_PA_CHANNEL_POSITION_INVALID;
        case MA_CHANNEL_FRONT_LEFT:         return MA_PA_CHANNEL_POSITION_FRONT_LEFT;
        case MA_CHANNEL_FRONT_RIGHT:        return MA_PA_CHANNEL_POSITION_FRONT_RIGHT;
        case MA_CHANNEL_FRONT_CENTER:       return MA_PA_CHANNEL_POSITION_FRONT_CENTER;
        case MA_CHANNEL_LFE:                return MA_PA_CHANNEL_POSITION_LFE;
        case MA_CHANNEL_BACK_LEFT:          return MA_PA_CHANNEL_POSITION_REAR_LEFT;
        case MA_CHANNEL_BACK_RIGHT:         return MA_PA_CHANNEL_POSITION_REAR_RIGHT;
        case MA_CHANNEL_FRONT_LEFT_CENTER:  return MA_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER;
        case MA_CHANNEL_FRONT_RIGHT_CENTER: return MA_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER;
        case MA_CHANNEL_BACK_CENTER:        return MA_PA_CHANNEL_POSITION_REAR_CENTER;
        case MA_CHANNEL_SIDE_LEFT:          return MA_PA_CHANNEL_POSITION_SIDE_LEFT;
        case MA_CHANNEL_SIDE_RIGHT:         return MA_PA_CHANNEL_POSITION_SIDE_RIGHT;
        case MA_CHANNEL_TOP_CENTER:         return MA_PA_CHANNEL_POSITION_TOP_CENTER;
        case MA_CHANNEL_TOP_FRONT_LEFT:     return MA_PA_CHANNEL_POSITION_TOP_FRONT_LEFT;
        case MA_CHANNEL_TOP_FRONT_CENTER:   return MA_PA_CHANNEL_POSITION_TOP_FRONT_CENTER;
        case MA_CHANNEL_TOP_FRONT_RIGHT:    return MA_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT;
        case MA_CHANNEL_TOP_BACK_LEFT:      return MA_PA_CHANNEL_POSITION_TOP_REAR_LEFT;
        case MA_CHANNEL_TOP_BACK_CENTER:    return MA_PA_CHANNEL_POSITION_TOP_REAR_CENTER;
        case MA_CHANNEL_TOP_BACK_RIGHT:     return MA_PA_CHANNEL_POSITION_TOP_REAR_RIGHT;
        case MA_CHANNEL_19:                 return MA_PA_CHANNEL_POSITION_AUX18;
        case MA_CHANNEL_20:                 return MA_PA_CHANNEL_POSITION_AUX19;
        case MA_CHANNEL_21:                 return MA_PA_CHANNEL_POSITION_AUX20;
        case MA_CHANNEL_22:                 return MA_PA_CHANNEL_POSITION_AUX21;
        case MA_CHANNEL_23:                 return MA_PA_CHANNEL_POSITION_AUX22;
        case MA_CHANNEL_24:                 return MA_PA_CHANNEL_POSITION_AUX23;
        case MA_CHANNEL_25:                 return MA_PA_CHANNEL_POSITION_AUX24;
        case MA_CHANNEL_26:                 return MA_PA_CHANNEL_POSITION_AUX25;
        case MA_CHANNEL_27:                 return MA_PA_CHANNEL_POSITION_AUX26;
        case MA_CHANNEL_28:                 return MA_PA_CHANNEL_POSITION_AUX27;
        case MA_CHANNEL_29:                 return MA_PA_CHANNEL_POSITION_AUX28;
        case MA_CHANNEL_30:                 return MA_PA_CHANNEL_POSITION_AUX29;
        case MA_CHANNEL_31:                 return MA_PA_CHANNEL_POSITION_AUX30;
        case MA_CHANNEL_32:                 return MA_PA_CHANNEL_POSITION_AUX31;
        default: return (ma_pa_channel_position_t)position;
    }
}
#endif

static ma_result ma_wait_for_operation__pulse(ma_context* pContext, ma_pa_mainloop* pMainLoop, ma_pa_operation* pOP)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pMainLoop != NULL);
    MA_ASSERT(pOP != NULL);

    while (((ma_pa_operation_get_state_proc)pContext->pulse.pa_operation_get_state)(pOP) == MA_PA_OPERATION_RUNNING) {
        int error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)(pMainLoop, 1, NULL);
        if (error < 0) {
            return ma_result_from_pulse(error);
        }
    }

    return MA_SUCCESS;
}

static ma_result ma_device__wait_for_operation__pulse(ma_device* pDevice, ma_pa_operation* pOP)
{
    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(pOP != NULL);

    return ma_wait_for_operation__pulse(pDevice->pContext, (ma_pa_mainloop*)pDevice->pulse.pMainLoop, pOP);
}


static ma_bool32 ma_context_is_device_id_equal__pulse(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pID0 != NULL);
    MA_ASSERT(pID1 != NULL);
    (void)pContext;

    return ma_strcmp(pID0->pulse, pID1->pulse) == 0;
}


typedef struct
{
    ma_context* pContext;
    ma_enum_devices_callback_proc callback;
    void* pUserData;
    ma_bool32 isTerminated;
} ma_context_enumerate_devices_callback_data__pulse;

static void ma_context_enumerate_devices_sink_callback__pulse(ma_pa_context* pPulseContext, const ma_pa_sink_info* pSinkInfo, int endOfList, void* pUserData)
{
    ma_context_enumerate_devices_callback_data__pulse* pData = (ma_context_enumerate_devices_callback_data__pulse*)pUserData;
    ma_device_info deviceInfo;

    MA_ASSERT(pData != NULL);

    if (endOfList || pData->isTerminated) {
        return;
    }

    MA_ZERO_OBJECT(&deviceInfo);

    /* The name from PulseAudio is the ID for miniaudio. */
    if (pSinkInfo->name != NULL) {
        ma_strncpy_s(deviceInfo.id.pulse, sizeof(deviceInfo.id.pulse), pSinkInfo->name, (size_t)-1);
    }

    /* The description from PulseAudio is the name for miniaudio. */
    if (pSinkInfo->description != NULL) {
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), pSinkInfo->description, (size_t)-1);
    }

    pData->isTerminated = !pData->callback(pData->pContext, ma_device_type_playback, &deviceInfo, pData->pUserData);

    (void)pPulseContext; /* Unused. */
}

static void ma_context_enumerate_devices_source_callback__pulse(ma_pa_context* pPulseContext, const ma_pa_source_info* pSinkInfo, int endOfList, void* pUserData)
{
    ma_context_enumerate_devices_callback_data__pulse* pData = (ma_context_enumerate_devices_callback_data__pulse*)pUserData;
    ma_device_info deviceInfo;

    MA_ASSERT(pData != NULL);

    if (endOfList || pData->isTerminated) {
        return;
    }

    MA_ZERO_OBJECT(&deviceInfo);

    /* The name from PulseAudio is the ID for miniaudio. */
    if (pSinkInfo->name != NULL) {
        ma_strncpy_s(deviceInfo.id.pulse, sizeof(deviceInfo.id.pulse), pSinkInfo->name, (size_t)-1);
    }

    /* The description from PulseAudio is the name for miniaudio. */
    if (pSinkInfo->description != NULL) {
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), pSinkInfo->description, (size_t)-1);
    }

    pData->isTerminated = !pData->callback(pData->pContext, ma_device_type_capture, &deviceInfo, pData->pUserData);

    (void)pPulseContext; /* Unused. */
}

static ma_result ma_context_enumerate_devices__pulse(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_result result = MA_SUCCESS;
    ma_context_enumerate_devices_callback_data__pulse callbackData;
    ma_pa_operation* pOP = NULL;
    ma_pa_mainloop* pMainLoop;
    ma_pa_mainloop_api* pAPI;
    ma_pa_context* pPulseContext;
    int error;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(callback != NULL);

    callbackData.pContext = pContext;
    callbackData.callback = callback;
    callbackData.pUserData = pUserData;
    callbackData.isTerminated = MA_FALSE;

    pMainLoop = ((ma_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
    if (pMainLoop == NULL) {
        return MA_FAILED_TO_INIT_BACKEND;
    }

    pAPI = ((ma_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)(pMainLoop);
    if (pAPI == NULL) {
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MA_FAILED_TO_INIT_BACKEND;
    }

    pPulseContext = ((ma_pa_context_new_proc)pContext->pulse.pa_context_new)(pAPI, pContext->pulse.pApplicationName);
    if (pPulseContext == NULL) {
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MA_FAILED_TO_INIT_BACKEND;
    }

    error = ((ma_pa_context_connect_proc)pContext->pulse.pa_context_connect)(pPulseContext, pContext->pulse.pServerName, (pContext->pulse.tryAutoSpawn) ? 0 : MA_PA_CONTEXT_NOAUTOSPAWN, NULL);
    if (error != MA_PA_OK) {
        ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return ma_result_from_pulse(error);
    }

    for (;;) {
        ma_pa_context_state_t state = ((ma_pa_context_get_state_proc)pContext->pulse.pa_context_get_state)(pPulseContext);
        if (state == MA_PA_CONTEXT_READY) {
            break;  /* Success. */
        }
        if (state == MA_PA_CONTEXT_CONNECTING || state == MA_PA_CONTEXT_AUTHORIZING || state == MA_PA_CONTEXT_SETTING_NAME) {
            error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)(pMainLoop, 1, NULL);
            if (error < 0) {
                result = ma_result_from_pulse(error);
                goto done;
            }

#ifdef MA_DEBUG_OUTPUT
            printf("[PulseAudio] pa_context_get_state() returned %d. Waiting.\n", state);
#endif
            continue;   /* Keep trying. */
        }
        if (state == MA_PA_CONTEXT_UNCONNECTED || state == MA_PA_CONTEXT_FAILED || state == MA_PA_CONTEXT_TERMINATED) {
#ifdef MA_DEBUG_OUTPUT
            printf("[PulseAudio] pa_context_get_state() returned %d. Failed.\n", state);
#endif
            goto done;  /* Failed. */
        }
    }


    /* Playback. */
    if (!callbackData.isTerminated) {
        pOP = ((ma_pa_context_get_sink_info_list_proc)pContext->pulse.pa_context_get_sink_info_list)(pPulseContext, ma_context_enumerate_devices_sink_callback__pulse, &callbackData);
        if (pOP == NULL) {
            result = MA_ERROR;
            goto done;
        }

        result = ma_wait_for_operation__pulse(pContext, pMainLoop, pOP);
        ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        if (result != MA_SUCCESS) {
            goto done;
        }
    }


    /* Capture. */
    if (!callbackData.isTerminated) {
        pOP = ((ma_pa_context_get_source_info_list_proc)pContext->pulse.pa_context_get_source_info_list)(pPulseContext, ma_context_enumerate_devices_source_callback__pulse, &callbackData);
        if (pOP == NULL) {
            result = MA_ERROR;
            goto done;
        }

        result = ma_wait_for_operation__pulse(pContext, pMainLoop, pOP);
        ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        if (result != MA_SUCCESS) {
            goto done;
        }
    }

done:
    ((ma_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)(pPulseContext);
    ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
    ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
    return result;
}


typedef struct
{
    ma_device_info* pDeviceInfo;
    ma_bool32 foundDevice;
} ma_context_get_device_info_callback_data__pulse;

static void ma_context_get_device_info_sink_callback__pulse(ma_pa_context* pPulseContext, const ma_pa_sink_info* pInfo, int endOfList, void* pUserData)
{
    ma_context_get_device_info_callback_data__pulse* pData = (ma_context_get_device_info_callback_data__pulse*)pUserData;

    if (endOfList > 0) {
        return;
    }

    MA_ASSERT(pData != NULL);
    pData->foundDevice = MA_TRUE;

    if (pInfo->name != NULL) {
        ma_strncpy_s(pData->pDeviceInfo->id.pulse, sizeof(pData->pDeviceInfo->id.pulse), pInfo->name, (size_t)-1);
    }

    if (pInfo->description != NULL) {
        ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pInfo->description, (size_t)-1);
    }

    pData->pDeviceInfo->minChannels   = pInfo->sample_spec.channels;
    pData->pDeviceInfo->maxChannels   = pInfo->sample_spec.channels;
    pData->pDeviceInfo->minSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->maxSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->formatCount = 1;
    pData->pDeviceInfo->formats[0] = ma_format_from_pulse(pInfo->sample_spec.format);

    (void)pPulseContext; /* Unused. */
}

static void ma_context_get_device_info_source_callback__pulse(ma_pa_context* pPulseContext, const ma_pa_source_info* pInfo, int endOfList, void* pUserData)
{
    ma_context_get_device_info_callback_data__pulse* pData = (ma_context_get_device_info_callback_data__pulse*)pUserData;

    if (endOfList > 0) {
        return;
    }

    MA_ASSERT(pData != NULL);
    pData->foundDevice = MA_TRUE;

    if (pInfo->name != NULL) {
        ma_strncpy_s(pData->pDeviceInfo->id.pulse, sizeof(pData->pDeviceInfo->id.pulse), pInfo->name, (size_t)-1);
    }

    if (pInfo->description != NULL) {
        ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pInfo->description, (size_t)-1);
    }

    pData->pDeviceInfo->minChannels = pInfo->sample_spec.channels;
    pData->pDeviceInfo->maxChannels = pInfo->sample_spec.channels;
    pData->pDeviceInfo->minSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->maxSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->formatCount = 1;
    pData->pDeviceInfo->formats[0] = ma_format_from_pulse(pInfo->sample_spec.format);

    (void)pPulseContext; /* Unused. */
}

static ma_result ma_context_get_device_info__pulse(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_result result = MA_SUCCESS;
    ma_context_get_device_info_callback_data__pulse callbackData;
    ma_pa_operation* pOP = NULL;
    ma_pa_mainloop* pMainLoop;
    ma_pa_mainloop_api* pAPI;
    ma_pa_context* pPulseContext;
    int error;

    MA_ASSERT(pContext != NULL);

    /* No exclusive mode with the PulseAudio backend. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    callbackData.pDeviceInfo = pDeviceInfo;
    callbackData.foundDevice = MA_FALSE;

    pMainLoop = ((ma_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
    if (pMainLoop == NULL) {
        return MA_FAILED_TO_INIT_BACKEND;
    }

    pAPI = ((ma_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)(pMainLoop);
    if (pAPI == NULL) {
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MA_FAILED_TO_INIT_BACKEND;
    }

    pPulseContext = ((ma_pa_context_new_proc)pContext->pulse.pa_context_new)(pAPI, pContext->pulse.pApplicationName);
    if (pPulseContext == NULL) {
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MA_FAILED_TO_INIT_BACKEND;
    }

    error = ((ma_pa_context_connect_proc)pContext->pulse.pa_context_connect)(pPulseContext, pContext->pulse.pServerName, 0, NULL);
    if (error != MA_PA_OK) {
        ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return ma_result_from_pulse(error);
    }

    for (;;) {
        ma_pa_context_state_t state = ((ma_pa_context_get_state_proc)pContext->pulse.pa_context_get_state)(pPulseContext);
        if (state == MA_PA_CONTEXT_READY) {
            break;  /* Success. */
        }
        if (state == MA_PA_CONTEXT_CONNECTING || state == MA_PA_CONTEXT_AUTHORIZING || state == MA_PA_CONTEXT_SETTING_NAME) {
            error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)(pMainLoop, 1, NULL);
            if (error < 0) {
                result = ma_result_from_pulse(error);
                goto done;
            }

#ifdef MA_DEBUG_OUTPUT
            printf("[PulseAudio] pa_context_get_state() returned %d. Waiting.\n", state);
#endif
            continue;   /* Keep trying. */
        }
        if (state == MA_PA_CONTEXT_UNCONNECTED || state == MA_PA_CONTEXT_FAILED || state == MA_PA_CONTEXT_TERMINATED) {
#ifdef MA_DEBUG_OUTPUT
            printf("[PulseAudio] pa_context_get_state() returned %d. Failed.\n", state);
#endif
            goto done;  /* Failed. */
        }
    }

    if (deviceType == ma_device_type_playback) {
        pOP = ((ma_pa_context_get_sink_info_by_name_proc)pContext->pulse.pa_context_get_sink_info_by_name)(pPulseContext, pDeviceID->pulse, ma_context_get_device_info_sink_callback__pulse, &callbackData);
    } else {
        pOP = ((ma_pa_context_get_source_info_by_name_proc)pContext->pulse.pa_context_get_source_info_by_name)(pPulseContext, pDeviceID->pulse, ma_context_get_device_info_source_callback__pulse, &callbackData);
    }

    if (pOP != NULL) {
        ma_wait_for_operation__pulse(pContext, pMainLoop, pOP);
        ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
    } else {
        result = MA_ERROR;
        goto done;
    }

    if (!callbackData.foundDevice) {
        result = MA_NO_DEVICE;
        goto done;
    }


done:
    ((ma_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)(pPulseContext);
    ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
    ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
    return result;
}


static void ma_pulse_device_state_callback(ma_pa_context* pPulseContext, void* pUserData)
{
    ma_device* pDevice;
    ma_context* pContext;

    pDevice = (ma_device*)pUserData;
    MA_ASSERT(pDevice != NULL);

    pContext = pDevice->pContext;
    MA_ASSERT(pContext != NULL);

    pDevice->pulse.pulseContextState = ((ma_pa_context_get_state_proc)pContext->pulse.pa_context_get_state)(pPulseContext);
}

static void ma_device_sink_info_callback(ma_pa_context* pPulseContext, const ma_pa_sink_info* pInfo, int endOfList, void* pUserData)
{
    ma_pa_sink_info* pInfoOut;

    if (endOfList > 0) {
        return;
    }

    pInfoOut = (ma_pa_sink_info*)pUserData;
    MA_ASSERT(pInfoOut != NULL);

    *pInfoOut = *pInfo;

    (void)pPulseContext; /* Unused. */
}

static void ma_device_source_info_callback(ma_pa_context* pPulseContext, const ma_pa_source_info* pInfo, int endOfList, void* pUserData)
{
    ma_pa_source_info* pInfoOut;

    if (endOfList > 0) {
        return;
    }

    pInfoOut = (ma_pa_source_info*)pUserData;
    MA_ASSERT(pInfoOut != NULL);

    *pInfoOut = *pInfo;

    (void)pPulseContext; /* Unused. */
}

static void ma_device_sink_name_callback(ma_pa_context* pPulseContext, const ma_pa_sink_info* pInfo, int endOfList, void* pUserData)
{
    ma_device* pDevice;

    if (endOfList > 0) {
        return;
    }

    pDevice = (ma_device*)pUserData;
    MA_ASSERT(pDevice != NULL);

    ma_strncpy_s(pDevice->playback.name, sizeof(pDevice->playback.name), pInfo->description, (size_t)-1);

    (void)pPulseContext; /* Unused. */
}

static void ma_device_source_name_callback(ma_pa_context* pPulseContext, const ma_pa_source_info* pInfo, int endOfList, void* pUserData)
{
    ma_device* pDevice;

    if (endOfList > 0) {
        return;
    }

    pDevice = (ma_device*)pUserData;
    MA_ASSERT(pDevice != NULL);

    ma_strncpy_s(pDevice->capture.name, sizeof(pDevice->capture.name), pInfo->description, (size_t)-1);

    (void)pPulseContext; /* Unused. */
}

static void ma_device_uninit__pulse(ma_device* pDevice)
{
    ma_context* pContext;

    MA_ASSERT(pDevice != NULL);

    pContext = pDevice->pContext;
    MA_ASSERT(pContext != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_pa_stream_disconnect_proc)pContext->pulse.pa_stream_disconnect)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
        ((ma_pa_stream_unref_proc)pContext->pulse.pa_stream_unref)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
    }
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ((ma_pa_stream_disconnect_proc)pContext->pulse.pa_stream_disconnect)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
        ((ma_pa_stream_unref_proc)pContext->pulse.pa_stream_unref)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
    }

    ((ma_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)((ma_pa_context*)pDevice->pulse.pPulseContext);
    ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)((ma_pa_context*)pDevice->pulse.pPulseContext);
    ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)((ma_pa_mainloop*)pDevice->pulse.pMainLoop);
}

static ma_pa_buffer_attr ma_device__pa_buffer_attr_new(ma_uint32 periodSizeInFrames, ma_uint32 periods, const ma_pa_sample_spec* ss)
{
    ma_pa_buffer_attr attr;
    attr.maxlength = periodSizeInFrames * periods * ma_get_bytes_per_frame(ma_format_from_pulse(ss->format), ss->channels);
    attr.tlength   = attr.maxlength / periods;
    attr.prebuf    = (ma_uint32)-1;
    attr.minreq    = (ma_uint32)-1;
    attr.fragsize  = attr.maxlength / periods;

    return attr;
}

static ma_pa_stream* ma_device__pa_stream_new__pulse(ma_device* pDevice, const char* pStreamName, const ma_pa_sample_spec* ss, const ma_pa_channel_map* cmap)
{
    static int g_StreamCounter = 0;
    char actualStreamName[256];

    if (pStreamName != NULL) {
        ma_strncpy_s(actualStreamName, sizeof(actualStreamName), pStreamName, (size_t)-1);
    } else {
        ma_strcpy_s(actualStreamName, sizeof(actualStreamName), "miniaudio:");
        ma_itoa_s(g_StreamCounter, actualStreamName + 8, sizeof(actualStreamName)-8, 10);  /* 8 = strlen("miniaudio:") */
    }
    g_StreamCounter += 1;

    return ((ma_pa_stream_new_proc)pDevice->pContext->pulse.pa_stream_new)((ma_pa_context*)pDevice->pulse.pPulseContext, actualStreamName, ss, cmap);
}

static ma_result ma_device_init__pulse(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;
    int error = 0;
    const char* devPlayback = NULL;
    const char* devCapture  = NULL;
    ma_uint32 periodSizeInMilliseconds;
    ma_pa_sink_info sinkInfo;
    ma_pa_source_info sourceInfo;
    ma_pa_operation* pOP = NULL;
    ma_pa_sample_spec ss;
    ma_pa_channel_map cmap;
    ma_pa_buffer_attr attr;
    const ma_pa_sample_spec* pActualSS   = NULL;
    const ma_pa_channel_map* pActualCMap = NULL;
    const ma_pa_buffer_attr* pActualAttr = NULL;
    ma_uint32 iChannel;
    ma_pa_stream_flags_t streamFlags;

    MA_ASSERT(pDevice != NULL);
    MA_ZERO_OBJECT(&pDevice->pulse);

    if (pConfig->deviceType == ma_device_type_loopback) {
        return MA_DEVICE_TYPE_NOT_SUPPORTED;
    }

    /* No exclusive mode with the PulseAudio backend. */
    if (((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.shareMode == ma_share_mode_exclusive) ||
        ((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.shareMode  == ma_share_mode_exclusive)) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    if ((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.pDeviceID != NULL) {
        devPlayback = pConfig->playback.pDeviceID->pulse;
    }
    if ((pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.pDeviceID != NULL) {
        devCapture = pConfig->capture.pDeviceID->pulse;
    }

    periodSizeInMilliseconds = pConfig->periodSizeInMilliseconds;
    if (periodSizeInMilliseconds == 0) {
        periodSizeInMilliseconds = ma_calculate_buffer_size_in_milliseconds_from_frames(pConfig->periodSizeInFrames, pConfig->sampleRate);
    }

    pDevice->pulse.pMainLoop = ((ma_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
    if (pDevice->pulse.pMainLoop == NULL) {
        result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create main loop for device.", MA_FAILED_TO_INIT_BACKEND);
        goto on_error0;
    }

    pDevice->pulse.pAPI = ((ma_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)((ma_pa_mainloop*)pDevice->pulse.pMainLoop);
    if (pDevice->pulse.pAPI == NULL) {
        result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to retrieve PulseAudio main loop.", MA_FAILED_TO_INIT_BACKEND);
        goto on_error1;
    }

    pDevice->pulse.pPulseContext = ((ma_pa_context_new_proc)pContext->pulse.pa_context_new)((ma_pa_mainloop_api*)pDevice->pulse.pAPI, pContext->pulse.pApplicationName);
    if (pDevice->pulse.pPulseContext == NULL) {
        result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create PulseAudio context for device.", MA_FAILED_TO_INIT_BACKEND);
        goto on_error1;
    }

    error = ((ma_pa_context_connect_proc)pContext->pulse.pa_context_connect)((ma_pa_context*)pDevice->pulse.pPulseContext, pContext->pulse.pServerName, (pContext->pulse.tryAutoSpawn) ? 0 : MA_PA_CONTEXT_NOAUTOSPAWN, NULL);
    if (error != MA_PA_OK) {
        result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to connect PulseAudio context.", ma_result_from_pulse(error));
        goto on_error2;
    }


    pDevice->pulse.pulseContextState = MA_PA_CONTEXT_UNCONNECTED;
    ((ma_pa_context_set_state_callback_proc)pContext->pulse.pa_context_set_state_callback)((ma_pa_context*)pDevice->pulse.pPulseContext, ma_pulse_device_state_callback, pDevice);

    /* Wait for PulseAudio to get itself ready before returning. */
    for (;;) {
        if (pDevice->pulse.pulseContextState == MA_PA_CONTEXT_READY) {
            break;
        }

        /* An error may have occurred. */
        if (pDevice->pulse.pulseContextState == MA_PA_CONTEXT_FAILED || pDevice->pulse.pulseContextState == MA_PA_CONTEXT_TERMINATED) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] An error occurred while connecting the PulseAudio context.", MA_ERROR);
            goto on_error3;
        }

        error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)((ma_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
        if (error < 0) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] The PulseAudio main loop returned an error while connecting the PulseAudio context.", ma_result_from_pulse(error));
            goto on_error3;
        }
    }

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        pOP = ((ma_pa_context_get_source_info_by_name_proc)pContext->pulse.pa_context_get_source_info_by_name)((ma_pa_context*)pDevice->pulse.pPulseContext, devCapture, ma_device_source_info_callback, &sourceInfo);
        if (pOP != NULL) {
            ma_device__wait_for_operation__pulse(pDevice, pOP);
            ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        } else {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to retrieve source info for capture device.", ma_result_from_pulse(error));
            goto on_error3;
        }

        ss = sourceInfo.sample_spec;
        cmap = sourceInfo.channel_map;

        pDevice->capture.internalPeriodSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(periodSizeInMilliseconds, ss.rate);
        pDevice->capture.internalPeriods            = pConfig->periods;

        attr = ma_device__pa_buffer_attr_new(pDevice->capture.internalPeriodSizeInFrames, pConfig->periods, &ss);
    #ifdef MA_DEBUG_OUTPUT
        printf("[PulseAudio] Capture attr: maxlength=%d, tlength=%d, prebuf=%d, minreq=%d, fragsize=%d; internalPeriodSizeInFrames=%d\n", attr.maxlength, attr.tlength, attr.prebuf, attr.minreq, attr.fragsize, pDevice->capture.internalPeriodSizeInFrames);
    #endif

        pDevice->pulse.pStreamCapture = ma_device__pa_stream_new__pulse(pDevice, pConfig->pulse.pStreamNameCapture, &ss, &cmap);
        if (pDevice->pulse.pStreamCapture == NULL) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create PulseAudio capture stream.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
            goto on_error3;
        }

        streamFlags = MA_PA_STREAM_START_CORKED | MA_PA_STREAM_FIX_FORMAT | MA_PA_STREAM_FIX_RATE | MA_PA_STREAM_FIX_CHANNELS;
        if (devCapture != NULL) {
            streamFlags |= MA_PA_STREAM_DONT_MOVE;
        }

        error = ((ma_pa_stream_connect_record_proc)pContext->pulse.pa_stream_connect_record)((ma_pa_stream*)pDevice->pulse.pStreamCapture, devCapture, &attr, streamFlags);
        if (error != MA_PA_OK) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to connect PulseAudio capture stream.", ma_result_from_pulse(error));
            goto on_error4;
        }

        while (((ma_pa_stream_get_state_proc)pContext->pulse.pa_stream_get_state)((ma_pa_stream*)pDevice->pulse.pStreamCapture) != MA_PA_STREAM_READY) {
            error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)((ma_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
            if (error < 0) {
                result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] The PulseAudio main loop returned an error while connecting the PulseAudio capture stream.", ma_result_from_pulse(error));
                goto on_error5;
            }
        }

        /* Internal format. */
        pActualSS = ((ma_pa_stream_get_sample_spec_proc)pContext->pulse.pa_stream_get_sample_spec)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
        if (pActualSS != NULL) {
            /* If anything has changed between the requested and the actual sample spec, we need to update the buffer. */
            if (ss.format != pActualSS->format || ss.channels != pActualSS->channels || ss.rate != pActualSS->rate) {
                attr = ma_device__pa_buffer_attr_new(pDevice->capture.internalPeriodSizeInFrames, pConfig->periods, pActualSS);

                pOP = ((ma_pa_stream_set_buffer_attr_proc)pContext->pulse.pa_stream_set_buffer_attr)((ma_pa_stream*)pDevice->pulse.pStreamCapture, &attr, NULL, NULL);
                if (pOP != NULL) {
                    ma_device__wait_for_operation__pulse(pDevice, pOP);
                    ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
                }
            }

            ss = *pActualSS;
        }

        pDevice->capture.internalFormat     = ma_format_from_pulse(ss.format);
        pDevice->capture.internalChannels   = ss.channels;
        pDevice->capture.internalSampleRate = ss.rate;

        /* Internal channel map. */
        pActualCMap = ((ma_pa_stream_get_channel_map_proc)pContext->pulse.pa_stream_get_channel_map)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
        if (pActualCMap != NULL) {
            cmap = *pActualCMap;
        }
        for (iChannel = 0; iChannel < pDevice->capture.internalChannels; ++iChannel) {
            pDevice->capture.internalChannelMap[iChannel] = ma_channel_position_from_pulse(cmap.map[iChannel]);
        }

        /* Buffer. */
        pActualAttr = ((ma_pa_stream_get_buffer_attr_proc)pContext->pulse.pa_stream_get_buffer_attr)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
        if (pActualAttr != NULL) {
            attr = *pActualAttr;
        }
        pDevice->capture.internalPeriods            = attr.maxlength / attr.fragsize;
        pDevice->capture.internalPeriodSizeInFrames = attr.maxlength / ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels) / pDevice->capture.internalPeriods;
    #ifdef MA_DEBUG_OUTPUT
        printf("[PulseAudio] Capture actual attr: maxlength=%d, tlength=%d, prebuf=%d, minreq=%d, fragsize=%d; internalPeriodSizeInFrames=%d\n", attr.maxlength, attr.tlength, attr.prebuf, attr.minreq, attr.fragsize, pDevice->capture.internalPeriodSizeInFrames);
    #endif

        /* Name. */
        devCapture = ((ma_pa_stream_get_device_name_proc)pContext->pulse.pa_stream_get_device_name)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
        if (devCapture != NULL) {
            ma_pa_operation* pOP = ((ma_pa_context_get_source_info_by_name_proc)pContext->pulse.pa_context_get_source_info_by_name)((ma_pa_context*)pDevice->pulse.pPulseContext, devCapture, ma_device_source_name_callback, pDevice);
            if (pOP != NULL) {
                ma_device__wait_for_operation__pulse(pDevice, pOP);
                ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
            }
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        pOP = ((ma_pa_context_get_sink_info_by_name_proc)pContext->pulse.pa_context_get_sink_info_by_name)((ma_pa_context*)pDevice->pulse.pPulseContext, devPlayback, ma_device_sink_info_callback, &sinkInfo);
        if (pOP != NULL) {
            ma_device__wait_for_operation__pulse(pDevice, pOP);
            ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        } else {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to retrieve sink info for playback device.", ma_result_from_pulse(error));
            goto on_error3;
        }

        ss = sinkInfo.sample_spec;
        cmap = sinkInfo.channel_map;

        pDevice->playback.internalPeriodSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(periodSizeInMilliseconds, ss.rate);
        pDevice->playback.internalPeriods            = pConfig->periods;

        attr = ma_device__pa_buffer_attr_new(pDevice->playback.internalPeriodSizeInFrames, pConfig->periods, &ss);
    #ifdef MA_DEBUG_OUTPUT
        printf("[PulseAudio] Playback attr: maxlength=%d, tlength=%d, prebuf=%d, minreq=%d, fragsize=%d; internalPeriodSizeInFrames=%d\n", attr.maxlength, attr.tlength, attr.prebuf, attr.minreq, attr.fragsize, pDevice->playback.internalPeriodSizeInFrames);
    #endif

        pDevice->pulse.pStreamPlayback = ma_device__pa_stream_new__pulse(pDevice, pConfig->pulse.pStreamNamePlayback, &ss, &cmap);
        if (pDevice->pulse.pStreamPlayback == NULL) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create PulseAudio playback stream.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
            goto on_error3;
        }

        streamFlags = MA_PA_STREAM_START_CORKED | MA_PA_STREAM_FIX_FORMAT | MA_PA_STREAM_FIX_RATE | MA_PA_STREAM_FIX_CHANNELS;
        if (devPlayback != NULL) {
            streamFlags |= MA_PA_STREAM_DONT_MOVE;
        }

        error = ((ma_pa_stream_connect_playback_proc)pContext->pulse.pa_stream_connect_playback)((ma_pa_stream*)pDevice->pulse.pStreamPlayback, devPlayback, &attr, streamFlags, NULL, NULL);
        if (error != MA_PA_OK) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to connect PulseAudio playback stream.", ma_result_from_pulse(error));
            goto on_error6;
        }

        while (((ma_pa_stream_get_state_proc)pContext->pulse.pa_stream_get_state)((ma_pa_stream*)pDevice->pulse.pStreamPlayback) != MA_PA_STREAM_READY) {
            error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)((ma_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
            if (error < 0) {
                result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] The PulseAudio main loop returned an error while connecting the PulseAudio playback stream.", ma_result_from_pulse(error));
                goto on_error7;
            }
        }

        /* Internal format. */
        pActualSS = ((ma_pa_stream_get_sample_spec_proc)pContext->pulse.pa_stream_get_sample_spec)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
        if (pActualSS != NULL) {
            /* If anything has changed between the requested and the actual sample spec, we need to update the buffer. */
            if (ss.format != pActualSS->format || ss.channels != pActualSS->channels || ss.rate != pActualSS->rate) {
                attr = ma_device__pa_buffer_attr_new(pDevice->playback.internalPeriodSizeInFrames, pConfig->periods, pActualSS);

                pOP = ((ma_pa_stream_set_buffer_attr_proc)pContext->pulse.pa_stream_set_buffer_attr)((ma_pa_stream*)pDevice->pulse.pStreamPlayback, &attr, NULL, NULL);
                if (pOP != NULL) {
                    ma_device__wait_for_operation__pulse(pDevice, pOP);
                    ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
                }
            }

            ss = *pActualSS;
        }

        pDevice->playback.internalFormat     = ma_format_from_pulse(ss.format);
        pDevice->playback.internalChannels   = ss.channels;
        pDevice->playback.internalSampleRate = ss.rate;

        /* Internal channel map. */
        pActualCMap = ((ma_pa_stream_get_channel_map_proc)pContext->pulse.pa_stream_get_channel_map)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
        if (pActualCMap != NULL) {
            cmap = *pActualCMap;
        }
        for (iChannel = 0; iChannel < pDevice->playback.internalChannels; ++iChannel) {
            pDevice->playback.internalChannelMap[iChannel] = ma_channel_position_from_pulse(cmap.map[iChannel]);
        }

        /* Buffer. */
        pActualAttr = ((ma_pa_stream_get_buffer_attr_proc)pContext->pulse.pa_stream_get_buffer_attr)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
        if (pActualAttr != NULL) {
            attr = *pActualAttr;
        }
        pDevice->playback.internalPeriods            = attr.maxlength / attr.tlength;
        pDevice->playback.internalPeriodSizeInFrames = attr.maxlength / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels) / pDevice->playback.internalPeriods;
    #ifdef MA_DEBUG_OUTPUT
        printf("[PulseAudio] Playback actual attr: maxlength=%d, tlength=%d, prebuf=%d, minreq=%d, fragsize=%d; internalPeriodSizeInFrames=%d\n", attr.maxlength, attr.tlength, attr.prebuf, attr.minreq, attr.fragsize, pDevice->playback.internalPeriodSizeInFrames);
    #endif

        /* Name. */
        devPlayback = ((ma_pa_stream_get_device_name_proc)pContext->pulse.pa_stream_get_device_name)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
        if (devPlayback != NULL) {
            ma_pa_operation* pOP = ((ma_pa_context_get_sink_info_by_name_proc)pContext->pulse.pa_context_get_sink_info_by_name)((ma_pa_context*)pDevice->pulse.pPulseContext, devPlayback, ma_device_sink_name_callback, pDevice);
            if (pOP != NULL) {
                ma_device__wait_for_operation__pulse(pDevice, pOP);
                ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
            }
        }
    }

    return MA_SUCCESS;


on_error7:
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ((ma_pa_stream_disconnect_proc)pContext->pulse.pa_stream_disconnect)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
    }
on_error6:
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ((ma_pa_stream_unref_proc)pContext->pulse.pa_stream_unref)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
    }
on_error5:
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ((ma_pa_stream_disconnect_proc)pContext->pulse.pa_stream_disconnect)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
    }
on_error4:
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ((ma_pa_stream_unref_proc)pContext->pulse.pa_stream_unref)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
    }
on_error3: ((ma_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)((ma_pa_context*)pDevice->pulse.pPulseContext);
on_error2: ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)((ma_pa_context*)pDevice->pulse.pPulseContext);
on_error1: ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)((ma_pa_mainloop*)pDevice->pulse.pMainLoop);
on_error0:
    return result;
}


static void ma_pulse_operation_complete_callback(ma_pa_stream* pStream, int success, void* pUserData)
{
    ma_bool32* pIsSuccessful = (ma_bool32*)pUserData;
    MA_ASSERT(pIsSuccessful != NULL);

    *pIsSuccessful = (ma_bool32)success;

    (void)pStream; /* Unused. */
}

static ma_result ma_device__cork_stream__pulse(ma_device* pDevice, ma_device_type deviceType, int cork)
{
    ma_context* pContext = pDevice->pContext;
    ma_bool32 wasSuccessful;
    ma_pa_stream* pStream;
    ma_pa_operation* pOP;
    ma_result result;

    /* This should not be called with a duplex device type. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    wasSuccessful = MA_FALSE;

    pStream = (ma_pa_stream*)((deviceType == ma_device_type_capture) ? pDevice->pulse.pStreamCapture : pDevice->pulse.pStreamPlayback);
    MA_ASSERT(pStream != NULL);

    pOP = ((ma_pa_stream_cork_proc)pContext->pulse.pa_stream_cork)(pStream, cork, ma_pulse_operation_complete_callback, &wasSuccessful);
    if (pOP == NULL) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to cork PulseAudio stream.", (cork == 0) ? MA_FAILED_TO_START_BACKEND_DEVICE : MA_FAILED_TO_STOP_BACKEND_DEVICE);
    }

    result = ma_device__wait_for_operation__pulse(pDevice, pOP);
    ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);

    if (result != MA_SUCCESS) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] An error occurred while waiting for the PulseAudio stream to cork.", result);
    }

    if (!wasSuccessful) {
        if (cork) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to stop PulseAudio stream.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
        } else {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to start PulseAudio stream.", MA_FAILED_TO_START_BACKEND_DEVICE);
        }
    }

    return MA_SUCCESS;
}

static ma_result ma_device_stop__pulse(ma_device* pDevice)
{
    ma_result result;
    ma_bool32 wasSuccessful;
    ma_pa_operation* pOP;

    MA_ASSERT(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        result = ma_device__cork_stream__pulse(pDevice, ma_device_type_capture, 1);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        /* The stream needs to be drained if it's a playback device. */
        pOP = ((ma_pa_stream_drain_proc)pDevice->pContext->pulse.pa_stream_drain)((ma_pa_stream*)pDevice->pulse.pStreamPlayback, ma_pulse_operation_complete_callback, &wasSuccessful);
        if (pOP != NULL) {
            ma_device__wait_for_operation__pulse(pDevice, pOP);
            ((ma_pa_operation_unref_proc)pDevice->pContext->pulse.pa_operation_unref)(pOP);
        }

        result = ma_device__cork_stream__pulse(pDevice, ma_device_type_playback, 1);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    return MA_SUCCESS;
}

static ma_result ma_device_write__pulse(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount, ma_uint32* pFramesWritten)
{
    ma_uint32 totalFramesWritten;

    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(pPCMFrames != NULL);
    MA_ASSERT(frameCount > 0);

    if (pFramesWritten != NULL) {
        *pFramesWritten = 0;
    }

    totalFramesWritten = 0;
    while (totalFramesWritten < frameCount) {
        if (ma_device__get_state(pDevice) != MA_STATE_STARTED) {
            return MA_DEVICE_NOT_STARTED;
        }

        /* Place the data into the mapped buffer if we have one. */
        if (pDevice->pulse.pMappedBufferPlayback != NULL && pDevice->pulse.mappedBufferFramesRemainingPlayback > 0) {
            ma_uint32 bpf = ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
            ma_uint32 mappedBufferFramesConsumed = pDevice->pulse.mappedBufferFramesCapacityPlayback - pDevice->pulse.mappedBufferFramesRemainingPlayback;

            void* pDst = (ma_uint8*)pDevice->pulse.pMappedBufferPlayback + (mappedBufferFramesConsumed * bpf);
            const void* pSrc = (const ma_uint8*)pPCMFrames + (totalFramesWritten * bpf);
            ma_uint32  framesToCopy = ma_min(pDevice->pulse.mappedBufferFramesRemainingPlayback, (frameCount - totalFramesWritten));
            MA_COPY_MEMORY(pDst, pSrc, framesToCopy * bpf);

            pDevice->pulse.mappedBufferFramesRemainingPlayback -= framesToCopy;
            totalFramesWritten += framesToCopy;
        }

        /*
        Getting here means we've run out of data in the currently mapped chunk. We need to write this to the device and then try
        mapping another chunk. If this fails we need to wait for space to become available.
        */
        if (pDevice->pulse.mappedBufferFramesCapacityPlayback > 0 && pDevice->pulse.mappedBufferFramesRemainingPlayback == 0) {
            size_t nbytes = pDevice->pulse.mappedBufferFramesCapacityPlayback * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);

            int error = ((ma_pa_stream_write_proc)pDevice->pContext->pulse.pa_stream_write)((ma_pa_stream*)pDevice->pulse.pStreamPlayback, pDevice->pulse.pMappedBufferPlayback, nbytes, NULL, 0, MA_PA_SEEK_RELATIVE);
            if (error < 0) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to write data to the PulseAudio stream.", ma_result_from_pulse(error));
            }

            pDevice->pulse.pMappedBufferPlayback = NULL;
            pDevice->pulse.mappedBufferFramesRemainingPlayback = 0;
            pDevice->pulse.mappedBufferFramesCapacityPlayback = 0;
        }

        MA_ASSERT(totalFramesWritten <= frameCount);
        if (totalFramesWritten == frameCount) {
            break;
        }

        /* Getting here means we need to map a new buffer. If we don't have enough space we need to wait for more. */
        for (;;) {
            size_t writableSizeInBytes;

            /* If the device has been corked, don't try to continue. */
            if (((ma_pa_stream_is_corked_proc)pDevice->pContext->pulse.pa_stream_is_corked)((ma_pa_stream*)pDevice->pulse.pStreamPlayback)) {
                break;
            }

            writableSizeInBytes = ((ma_pa_stream_writable_size_proc)pDevice->pContext->pulse.pa_stream_writable_size)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
            if (writableSizeInBytes != (size_t)-1) {
                if (writableSizeInBytes > 0) {
                    /* Data is avaialable. */
                    size_t bytesToMap = writableSizeInBytes;
                    int error = ((ma_pa_stream_begin_write_proc)pDevice->pContext->pulse.pa_stream_begin_write)((ma_pa_stream*)pDevice->pulse.pStreamPlayback, &pDevice->pulse.pMappedBufferPlayback, &bytesToMap);
                    if (error < 0) {
                        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to map write buffer.", ma_result_from_pulse(error));
                    }

                    pDevice->pulse.mappedBufferFramesCapacityPlayback  = bytesToMap / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                    pDevice->pulse.mappedBufferFramesRemainingPlayback = pDevice->pulse.mappedBufferFramesCapacityPlayback;

                    break;
                } else {
                    /* No data available. Need to wait for more. */
                    int error = ((ma_pa_mainloop_iterate_proc)pDevice->pContext->pulse.pa_mainloop_iterate)((ma_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
                    if (error < 0) {
                        return ma_result_from_pulse(error);
                    }

                    continue;
                }
            } else {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to query the stream's writable size.", MA_ERROR);
            }
        }
    }

    if (pFramesWritten != NULL) {
        *pFramesWritten = totalFramesWritten;
    }

    return MA_SUCCESS;
}

static ma_result ma_device_read__pulse(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount, ma_uint32* pFramesRead)
{
    ma_uint32 totalFramesRead;

    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(pPCMFrames != NULL);
    MA_ASSERT(frameCount > 0);

    if (pFramesRead != NULL) {
        *pFramesRead = 0;
    }

    totalFramesRead = 0;
    while (totalFramesRead < frameCount) {
        if (ma_device__get_state(pDevice) != MA_STATE_STARTED) {
            return MA_DEVICE_NOT_STARTED;
        }

        /*
        If a buffer is mapped we need to read from that first. Once it's consumed we need to drop it. Note that pDevice->pulse.pMappedBufferCapture can be null in which
        case it could be a hole. In this case we just write zeros into the output buffer.
        */
        if (pDevice->pulse.mappedBufferFramesRemainingCapture > 0) {
            ma_uint32 bpf = ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
            ma_uint32 mappedBufferFramesConsumed = pDevice->pulse.mappedBufferFramesCapacityCapture - pDevice->pulse.mappedBufferFramesRemainingCapture;

            ma_uint32  framesToCopy = ma_min(pDevice->pulse.mappedBufferFramesRemainingCapture, (frameCount - totalFramesRead));
            void* pDst = (ma_uint8*)pPCMFrames + (totalFramesRead * bpf);

            /*
            This little bit of logic here is specifically for PulseAudio and it's hole management. The buffer pointer will be set to NULL
            when the current fragment is a hole. For a hole we just output silence.
            */
            if (pDevice->pulse.pMappedBufferCapture != NULL) {
                const void* pSrc = (const ma_uint8*)pDevice->pulse.pMappedBufferCapture + (mappedBufferFramesConsumed * bpf);
                MA_COPY_MEMORY(pDst, pSrc, framesToCopy * bpf);
            } else {
                MA_ZERO_MEMORY(pDst, framesToCopy * bpf);
            #if defined(MA_DEBUG_OUTPUT)
                printf("[PulseAudio] ma_device_read__pulse: Filling hole with silence.\n");
            #endif
            }

            pDevice->pulse.mappedBufferFramesRemainingCapture -= framesToCopy;
            totalFramesRead += framesToCopy;
        }

        /*
        Getting here means we've run out of data in the currently mapped chunk. We need to drop this from the device and then try
        mapping another chunk. If this fails we need to wait for data to become available.
        */
        if (pDevice->pulse.mappedBufferFramesCapacityCapture > 0 && pDevice->pulse.mappedBufferFramesRemainingCapture == 0) {
            int error;

        #if defined(MA_DEBUG_OUTPUT)
            printf("[PulseAudio] ma_device_read__pulse: Call pa_stream_drop()\n");
        #endif

            error = ((ma_pa_stream_drop_proc)pDevice->pContext->pulse.pa_stream_drop)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
            if (error != 0) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to drop fragment.", ma_result_from_pulse(error));
            }

            pDevice->pulse.pMappedBufferCapture = NULL;
            pDevice->pulse.mappedBufferFramesRemainingCapture = 0;
            pDevice->pulse.mappedBufferFramesCapacityCapture = 0;
        }

        MA_ASSERT(totalFramesRead <= frameCount);
        if (totalFramesRead == frameCount) {
            break;
        }

        /* Getting here means we need to map a new buffer. If we don't have enough data we wait for more. */
        for (;;) {
            int error;
            size_t bytesMapped;

            if (ma_device__get_state(pDevice) != MA_STATE_STARTED) {
                break;
            }

            /* If the device has been corked, don't try to continue. */
            if (((ma_pa_stream_is_corked_proc)pDevice->pContext->pulse.pa_stream_is_corked)((ma_pa_stream*)pDevice->pulse.pStreamCapture)) {
            #if defined(MA_DEBUG_OUTPUT)
                printf("[PulseAudio] ma_device_read__pulse: Corked.\n");
            #endif
                break;
            }

            MA_ASSERT(pDevice->pulse.pMappedBufferCapture == NULL); /* <-- We're about to map a buffer which means we shouldn't have an existing mapping. */

            error = ((ma_pa_stream_peek_proc)pDevice->pContext->pulse.pa_stream_peek)((ma_pa_stream*)pDevice->pulse.pStreamCapture, &pDevice->pulse.pMappedBufferCapture, &bytesMapped);
            if (error < 0) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to peek capture buffer.", ma_result_from_pulse(error));
            }

            if (bytesMapped > 0) {
                pDevice->pulse.mappedBufferFramesCapacityCapture  = bytesMapped / ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
                pDevice->pulse.mappedBufferFramesRemainingCapture = pDevice->pulse.mappedBufferFramesCapacityCapture;

                #if defined(MA_DEBUG_OUTPUT)
                    printf("[PulseAudio] ma_device_read__pulse: Mapped. mappedBufferFramesCapacityCapture=%d, mappedBufferFramesRemainingCapture=%d\n", pDevice->pulse.mappedBufferFramesCapacityCapture, pDevice->pulse.mappedBufferFramesRemainingCapture);
                #endif

                if (pDevice->pulse.pMappedBufferCapture == NULL) {
                    /* It's a hole. */
                    #if defined(MA_DEBUG_OUTPUT)
                        printf("[PulseAudio] ma_device_read__pulse: Call pa_stream_peek(). Hole.\n");
                    #endif
                }

                break;
            } else {
                if (pDevice->pulse.pMappedBufferCapture == NULL) {
                    /* Nothing available yet. Need to wait for more. */

                    /*
                    I have had reports of a deadlock in this part of the code. I have reproduced this when using the "Built-in Audio Analogue Stereo" device without
                    an actual microphone connected. I'm experimenting here by not blocking in pa_mainloop_iterate() and instead sleep for a bit when there are no
                    dispatches.
                    */
                    error = ((ma_pa_mainloop_iterate_proc)pDevice->pContext->pulse.pa_mainloop_iterate)((ma_pa_mainloop*)pDevice->pulse.pMainLoop, 0, NULL);
                    if (error < 0) {
                        return ma_result_from_pulse(error);
                    }

                    /* Sleep for a bit if nothing was dispatched. */
                    if (error == 0) {
                        ma_sleep(1);
                    }

                #if defined(MA_DEBUG_OUTPUT)
                    printf("[PulseAudio] ma_device_read__pulse: No data available. Waiting. mappedBufferFramesCapacityCapture=%d, mappedBufferFramesRemainingCapture=%d\n", pDevice->pulse.mappedBufferFramesCapacityCapture, pDevice->pulse.mappedBufferFramesRemainingCapture);
                #endif
                } else {
                    /* Getting here means we mapped 0 bytes, but have a non-NULL buffer. I don't think this should ever happen. */
                    MA_ASSERT(MA_FALSE);
                }
            }
        }
    }

    if (pFramesRead != NULL) {
        *pFramesRead = totalFramesRead;
    }

    return MA_SUCCESS;
}

static ma_result ma_device_main_loop__pulse(ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;
    ma_bool32 exitLoop = MA_FALSE;

    MA_ASSERT(pDevice != NULL);

    /* The stream needs to be uncorked first. We do this at the top for both capture and playback for PulseAudio. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        result = ma_device__cork_stream__pulse(pDevice, ma_device_type_capture, 0);
        if (result != MA_SUCCESS) {
            return result;
        }
    }
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        result = ma_device__cork_stream__pulse(pDevice, ma_device_type_playback, 0);
        if (result != MA_SUCCESS) {
            return result;
        }
    }


    while (ma_device__get_state(pDevice) == MA_STATE_STARTED && !exitLoop) {
        switch (pDevice->type)
        {
            case ma_device_type_duplex:
            {
                /* The process is: device_read -> convert -> callback -> convert -> device_write */
                ma_uint32 totalCapturedDeviceFramesProcessed = 0;
                ma_uint32 capturedDevicePeriodSizeInFrames = ma_min(pDevice->capture.internalPeriodSizeInFrames, pDevice->playback.internalPeriodSizeInFrames);
                    
                while (totalCapturedDeviceFramesProcessed < capturedDevicePeriodSizeInFrames) {
                    ma_uint8  capturedDeviceData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                    ma_uint8  playbackDeviceData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                    ma_uint32 capturedDeviceDataCapInFrames = sizeof(capturedDeviceData) / ma_get_bytes_per_frame(pDevice->capture.internalFormat,  pDevice->capture.internalChannels);
                    ma_uint32 playbackDeviceDataCapInFrames = sizeof(playbackDeviceData) / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                    ma_uint32 capturedDeviceFramesRemaining;
                    ma_uint32 capturedDeviceFramesProcessed;
                    ma_uint32 capturedDeviceFramesToProcess;
                    ma_uint32 capturedDeviceFramesToTryProcessing = capturedDevicePeriodSizeInFrames - totalCapturedDeviceFramesProcessed;
                    if (capturedDeviceFramesToTryProcessing > capturedDeviceDataCapInFrames) {
                        capturedDeviceFramesToTryProcessing = capturedDeviceDataCapInFrames;
                    }

                    result = ma_device_read__pulse(pDevice, capturedDeviceData, capturedDeviceFramesToTryProcessing, &capturedDeviceFramesToProcess);
                    if (result != MA_SUCCESS) {
                        exitLoop = MA_TRUE;
                        break;
                    }

                    capturedDeviceFramesRemaining = capturedDeviceFramesToProcess;
                    capturedDeviceFramesProcessed = 0;

                    for (;;) {
                        ma_uint8  capturedClientData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                        ma_uint8  playbackClientData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                        ma_uint32 capturedClientDataCapInFrames = sizeof(capturedClientData) / ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
                        ma_uint32 playbackClientDataCapInFrames = sizeof(playbackClientData) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
                        ma_uint64 capturedClientFramesToProcessThisIteration = ma_min(capturedClientDataCapInFrames, playbackClientDataCapInFrames);
                        ma_uint64 capturedDeviceFramesToProcessThisIteration = capturedDeviceFramesRemaining;
                        ma_uint8* pRunningCapturedDeviceFrames = ma_offset_ptr(capturedDeviceData, capturedDeviceFramesProcessed * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));

                        /* Convert capture data from device format to client format. */
                        result = ma_data_converter_process_pcm_frames(&pDevice->capture.converter, pRunningCapturedDeviceFrames, &capturedDeviceFramesToProcessThisIteration, capturedClientData, &capturedClientFramesToProcessThisIteration);
                        if (result != MA_SUCCESS) {
                            break;
                        }

                        /*
                        If we weren't able to generate any output frames it must mean we've exhaused all of our input. The only time this would not be the case is if capturedClientData was too small
                        which should never be the case when it's of the size MA_DATA_CONVERTER_STACK_BUFFER_SIZE.
                        */
                        if (capturedClientFramesToProcessThisIteration == 0) {
                            break;
                        }

                        ma_device__on_data(pDevice, playbackClientData, capturedClientData, (ma_uint32)capturedClientFramesToProcessThisIteration);    /* Safe cast .*/

                        capturedDeviceFramesProcessed += (ma_uint32)capturedDeviceFramesToProcessThisIteration; /* Safe cast. */
                        capturedDeviceFramesRemaining -= (ma_uint32)capturedDeviceFramesToProcessThisIteration; /* Safe cast. */

                        /* At this point the playbackClientData buffer should be holding data that needs to be written to the device. */
                        for (;;) {
                            ma_uint64 convertedClientFrameCount = capturedClientFramesToProcessThisIteration;
                            ma_uint64 convertedDeviceFrameCount = playbackDeviceDataCapInFrames;
                            result = ma_data_converter_process_pcm_frames(&pDevice->playback.converter, playbackClientData, &convertedClientFrameCount, playbackDeviceData, &convertedDeviceFrameCount);
                            if (result != MA_SUCCESS) {
                                break;
                            }

                            result = ma_device_write__pulse(pDevice, playbackDeviceData, (ma_uint32)convertedDeviceFrameCount, NULL);   /* Safe cast. */
                            if (result != MA_SUCCESS) {
                                exitLoop = MA_TRUE;
                                break;
                            }

                            capturedClientFramesToProcessThisIteration -= (ma_uint32)convertedClientFrameCount;  /* Safe cast. */
                            if (capturedClientFramesToProcessThisIteration == 0) {
                                break;
                            }
                        }

                        /* In case an error happened from ma_device_write__pulse()... */
                        if (result != MA_SUCCESS) {
                            exitLoop = MA_TRUE;
                            break;
                        }
                    }

                    totalCapturedDeviceFramesProcessed += capturedDeviceFramesProcessed;
                }
            } break;

            case ma_device_type_capture:
            {
                ma_uint8 intermediaryBuffer[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                ma_uint32 intermediaryBufferSizeInFrames = sizeof(intermediaryBuffer) / ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
                ma_uint32 periodSizeInFrames = pDevice->capture.internalPeriodSizeInFrames;
                ma_uint32 framesReadThisPeriod = 0;
                while (framesReadThisPeriod < periodSizeInFrames) {
                    ma_uint32 framesRemainingInPeriod = periodSizeInFrames - framesReadThisPeriod;
                    ma_uint32 framesProcessed;
                    ma_uint32 framesToReadThisIteration = framesRemainingInPeriod;
                    if (framesToReadThisIteration > intermediaryBufferSizeInFrames) {
                        framesToReadThisIteration = intermediaryBufferSizeInFrames;
                    }

                    result = ma_device_read__pulse(pDevice, intermediaryBuffer, framesToReadThisIteration, &framesProcessed);
                    if (result != MA_SUCCESS) {
                        exitLoop = MA_TRUE;
                        break;
                    }

                    ma_device__send_frames_to_client(pDevice, framesProcessed, intermediaryBuffer);

                    framesReadThisPeriod += framesProcessed;
                }
            } break;

            case ma_device_type_playback:
            {
                ma_uint8 intermediaryBuffer[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                ma_uint32 intermediaryBufferSizeInFrames = sizeof(intermediaryBuffer) / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                ma_uint32 periodSizeInFrames = pDevice->playback.internalPeriodSizeInFrames;
                ma_uint32 framesWrittenThisPeriod = 0;
                while (framesWrittenThisPeriod < periodSizeInFrames) {
                    ma_uint32 framesRemainingInPeriod = periodSizeInFrames - framesWrittenThisPeriod;
                    ma_uint32 framesProcessed;
                    ma_uint32 framesToWriteThisIteration = framesRemainingInPeriod;
                    if (framesToWriteThisIteration > intermediaryBufferSizeInFrames) {
                        framesToWriteThisIteration = intermediaryBufferSizeInFrames;
                    }

                    ma_device__read_frames_from_client(pDevice, framesToWriteThisIteration, intermediaryBuffer);

                    result = ma_device_write__pulse(pDevice, intermediaryBuffer, framesToWriteThisIteration, &framesProcessed);
                    if (result != MA_SUCCESS) {
                        exitLoop = MA_TRUE;
                        break;
                    }

                    framesWrittenThisPeriod += framesProcessed;
                }
            } break;

            /* To silence a warning. Will never hit this. */
            case ma_device_type_loopback:
            default: break;
        }
    }

    /* Here is where the device needs to be stopped. */
    ma_device_stop__pulse(pDevice);

    return result;
}


static ma_result ma_context_uninit__pulse(ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pContext->backend == ma_backend_pulseaudio);

    ma_free(pContext->pulse.pServerName, &pContext->allocationCallbacks);
    pContext->pulse.pServerName = NULL;

    ma_free(pContext->pulse.pApplicationName, &pContext->allocationCallbacks);
    pContext->pulse.pApplicationName = NULL;

#ifndef MA_NO_RUNTIME_LINKING
    ma_dlclose(pContext, pContext->pulse.pulseSO);
#endif

    return MA_SUCCESS;
}

static ma_result ma_context_init__pulse(const ma_context_config* pConfig, ma_context* pContext)
{
#ifndef MA_NO_RUNTIME_LINKING
    const char* libpulseNames[] = {
        "libpulse.so",
        "libpulse.so.0"
    };
    size_t i;

    for (i = 0; i < ma_countof(libpulseNames); ++i) {
        pContext->pulse.pulseSO = ma_dlopen(pContext, libpulseNames[i]);
        if (pContext->pulse.pulseSO != NULL) {
            break;
        }
    }

    if (pContext->pulse.pulseSO == NULL) {
        return MA_NO_BACKEND;
    }

    pContext->pulse.pa_mainloop_new                    = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_mainloop_new");
    pContext->pulse.pa_mainloop_free                   = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_mainloop_free");
    pContext->pulse.pa_mainloop_get_api                = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_mainloop_get_api");
    pContext->pulse.pa_mainloop_iterate                = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_mainloop_iterate");
    pContext->pulse.pa_mainloop_wakeup                 = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_mainloop_wakeup");
    pContext->pulse.pa_context_new                     = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_context_new");
    pContext->pulse.pa_context_unref                   = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_context_unref");
    pContext->pulse.pa_context_connect                 = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_context_connect");
    pContext->pulse.pa_context_disconnect              = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_context_disconnect");
    pContext->pulse.pa_context_set_state_callback      = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_context_set_state_callback");
    pContext->pulse.pa_context_get_state               = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_context_get_state");
    pContext->pulse.pa_context_get_sink_info_list      = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_context_get_sink_info_list");
    pContext->pulse.pa_context_get_source_info_list    = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_context_get_source_info_list");
    pContext->pulse.pa_context_get_sink_info_by_name   = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_context_get_sink_info_by_name");
    pContext->pulse.pa_context_get_source_info_by_name = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_context_get_source_info_by_name");
    pContext->pulse.pa_operation_unref                 = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_operation_unref");
    pContext->pulse.pa_operation_get_state             = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_operation_get_state");
    pContext->pulse.pa_channel_map_init_extend         = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_channel_map_init_extend");
    pContext->pulse.pa_channel_map_valid               = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_channel_map_valid");
    pContext->pulse.pa_channel_map_compatible          = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_channel_map_compatible");
    pContext->pulse.pa_stream_new                      = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_new");
    pContext->pulse.pa_stream_unref                    = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_unref");
    pContext->pulse.pa_stream_connect_playback         = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_connect_playback");
    pContext->pulse.pa_stream_connect_record           = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_connect_record");
    pContext->pulse.pa_stream_disconnect               = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_disconnect");
    pContext->pulse.pa_stream_get_state                = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_get_state");
    pContext->pulse.pa_stream_get_sample_spec          = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_get_sample_spec");
    pContext->pulse.pa_stream_get_channel_map          = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_get_channel_map");
    pContext->pulse.pa_stream_get_buffer_attr          = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_get_buffer_attr");
    pContext->pulse.pa_stream_set_buffer_attr          = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_set_buffer_attr");
    pContext->pulse.pa_stream_get_device_name          = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_get_device_name");
    pContext->pulse.pa_stream_set_write_callback       = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_set_write_callback");
    pContext->pulse.pa_stream_set_read_callback        = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_set_read_callback");
    pContext->pulse.pa_stream_flush                    = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_flush");
    pContext->pulse.pa_stream_drain                    = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_drain");
    pContext->pulse.pa_stream_is_corked                = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_is_corked");
    pContext->pulse.pa_stream_cork                     = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_cork");
    pContext->pulse.pa_stream_trigger                  = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_trigger");
    pContext->pulse.pa_stream_begin_write              = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_begin_write");
    pContext->pulse.pa_stream_write                    = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_write");
    pContext->pulse.pa_stream_peek                     = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_peek");
    pContext->pulse.pa_stream_drop                     = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_drop");
    pContext->pulse.pa_stream_writable_size            = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_writable_size");
    pContext->pulse.pa_stream_readable_size            = (ma_proc)ma_dlsym(pContext, pContext->pulse.pulseSO, "pa_stream_readable_size");
#else
    /* This strange assignment system is just for type safety. */
    ma_pa_mainloop_new_proc                    _pa_mainloop_new                   = pa_mainloop_new;
    ma_pa_mainloop_free_proc                   _pa_mainloop_free                  = pa_mainloop_free;
    ma_pa_mainloop_get_api_proc                _pa_mainloop_get_api               = pa_mainloop_get_api;
    ma_pa_mainloop_iterate_proc                _pa_mainloop_iterate               = pa_mainloop_iterate;
    ma_pa_mainloop_wakeup_proc                 _pa_mainloop_wakeup                = pa_mainloop_wakeup;
    ma_pa_context_new_proc                     _pa_context_new                    = pa_context_new;
    ma_pa_context_unref_proc                   _pa_context_unref                  = pa_context_unref;
    ma_pa_context_connect_proc                 _pa_context_connect                = pa_context_connect;
    ma_pa_context_disconnect_proc              _pa_context_disconnect             = pa_context_disconnect;
    ma_pa_context_set_state_callback_proc      _pa_context_set_state_callback     = pa_context_set_state_callback;
    ma_pa_context_get_state_proc               _pa_context_get_state              = pa_context_get_state;
    ma_pa_context_get_sink_info_list_proc      _pa_context_get_sink_info_list     = pa_context_get_sink_info_list;
    ma_pa_context_get_source_info_list_proc    _pa_context_get_source_info_list   = pa_context_get_source_info_list;
    ma_pa_context_get_sink_info_by_name_proc   _pa_context_get_sink_info_by_name  = pa_context_get_sink_info_by_name;
    ma_pa_context_get_source_info_by_name_proc _pa_context_get_source_info_by_name= pa_context_get_source_info_by_name;
    ma_pa_operation_unref_proc                 _pa_operation_unref                = pa_operation_unref;
    ma_pa_operation_get_state_proc             _pa_operation_get_state            = pa_operation_get_state;
    ma_pa_channel_map_init_extend_proc         _pa_channel_map_init_extend        = pa_channel_map_init_extend;
    ma_pa_channel_map_valid_proc               _pa_channel_map_valid              = pa_channel_map_valid;
    ma_pa_channel_map_compatible_proc          _pa_channel_map_compatible         = pa_channel_map_compatible;
    ma_pa_stream_new_proc                      _pa_stream_new                     = pa_stream_new;
    ma_pa_stream_unref_proc                    _pa_stream_unref                   = pa_stream_unref;
    ma_pa_stream_connect_playback_proc         _pa_stream_connect_playback        = pa_stream_connect_playback;
    ma_pa_stream_connect_record_proc           _pa_stream_connect_record          = pa_stream_connect_record;
    ma_pa_stream_disconnect_proc               _pa_stream_disconnect              = pa_stream_disconnect;
    ma_pa_stream_get_state_proc                _pa_stream_get_state               = pa_stream_get_state;
    ma_pa_stream_get_sample_spec_proc          _pa_stream_get_sample_spec         = pa_stream_get_sample_spec;
    ma_pa_stream_get_channel_map_proc          _pa_stream_get_channel_map         = pa_stream_get_channel_map;
    ma_pa_stream_get_buffer_attr_proc          _pa_stream_get_buffer_attr         = pa_stream_get_buffer_attr;
    ma_pa_stream_set_buffer_attr_proc          _pa_stream_set_buffer_attr         = pa_stream_set_buffer_attr;
    ma_pa_stream_get_device_name_proc          _pa_stream_get_device_name         = pa_stream_get_device_name;
    ma_pa_stream_set_write_callback_proc       _pa_stream_set_write_callback      = pa_stream_set_write_callback;
    ma_pa_stream_set_read_callback_proc        _pa_stream_set_read_callback       = pa_stream_set_read_callback;
    ma_pa_stream_flush_proc                    _pa_stream_flush                   = pa_stream_flush;
    ma_pa_stream_drain_proc                    _pa_stream_drain                   = pa_stream_drain;
    ma_pa_stream_is_corked_proc                _pa_stream_is_corked               = pa_stream_is_corked;
    ma_pa_stream_cork_proc                     _pa_stream_cork                    = pa_stream_cork;
    ma_pa_stream_trigger_proc                  _pa_stream_trigger                 = pa_stream_trigger;
    ma_pa_stream_begin_write_proc              _pa_stream_begin_write             = pa_stream_begin_write;
    ma_pa_stream_write_proc                    _pa_stream_write                   = pa_stream_write;
    ma_pa_stream_peek_proc                     _pa_stream_peek                    = pa_stream_peek;
    ma_pa_stream_drop_proc                     _pa_stream_drop                    = pa_stream_drop;
    ma_pa_stream_writable_size_proc            _pa_stream_writable_size           = pa_stream_writable_size;
    ma_pa_stream_readable_size_proc            _pa_stream_readable_size           = pa_stream_readable_size;

    pContext->pulse.pa_mainloop_new                    = (ma_proc)_pa_mainloop_new;
    pContext->pulse.pa_mainloop_free                   = (ma_proc)_pa_mainloop_free;
    pContext->pulse.pa_mainloop_get_api                = (ma_proc)_pa_mainloop_get_api;
    pContext->pulse.pa_mainloop_iterate                = (ma_proc)_pa_mainloop_iterate;
    pContext->pulse.pa_mainloop_wakeup                 = (ma_proc)_pa_mainloop_wakeup;
    pContext->pulse.pa_context_new                     = (ma_proc)_pa_context_new;
    pContext->pulse.pa_context_unref                   = (ma_proc)_pa_context_unref;
    pContext->pulse.pa_context_connect                 = (ma_proc)_pa_context_connect;
    pContext->pulse.pa_context_disconnect              = (ma_proc)_pa_context_disconnect;
    pContext->pulse.pa_context_set_state_callback      = (ma_proc)_pa_context_set_state_callback;
    pContext->pulse.pa_context_get_state               = (ma_proc)_pa_context_get_state;
    pContext->pulse.pa_context_get_sink_info_list      = (ma_proc)_pa_context_get_sink_info_list;
    pContext->pulse.pa_context_get_source_info_list    = (ma_proc)_pa_context_get_source_info_list;
    pContext->pulse.pa_context_get_sink_info_by_name   = (ma_proc)_pa_context_get_sink_info_by_name;
    pContext->pulse.pa_context_get_source_info_by_name = (ma_proc)_pa_context_get_source_info_by_name;
    pContext->pulse.pa_operation_unref                 = (ma_proc)_pa_operation_unref;
    pContext->pulse.pa_operation_get_state             = (ma_proc)_pa_operation_get_state;
    pContext->pulse.pa_channel_map_init_extend         = (ma_proc)_pa_channel_map_init_extend;
    pContext->pulse.pa_channel_map_valid               = (ma_proc)_pa_channel_map_valid;
    pContext->pulse.pa_channel_map_compatible          = (ma_proc)_pa_channel_map_compatible;
    pContext->pulse.pa_stream_new                      = (ma_proc)_pa_stream_new;
    pContext->pulse.pa_stream_unref                    = (ma_proc)_pa_stream_unref;
    pContext->pulse.pa_stream_connect_playback         = (ma_proc)_pa_stream_connect_playback;
    pContext->pulse.pa_stream_connect_record           = (ma_proc)_pa_stream_connect_record;
    pContext->pulse.pa_stream_disconnect               = (ma_proc)_pa_stream_disconnect;
    pContext->pulse.pa_stream_get_state                = (ma_proc)_pa_stream_get_state;
    pContext->pulse.pa_stream_get_sample_spec          = (ma_proc)_pa_stream_get_sample_spec;
    pContext->pulse.pa_stream_get_channel_map          = (ma_proc)_pa_stream_get_channel_map;
    pContext->pulse.pa_stream_get_buffer_attr          = (ma_proc)_pa_stream_get_buffer_attr;
    pContext->pulse.pa_stream_set_buffer_attr          = (ma_proc)_pa_stream_set_buffer_attr;
    pContext->pulse.pa_stream_get_device_name          = (ma_proc)_pa_stream_get_device_name;
    pContext->pulse.pa_stream_set_write_callback       = (ma_proc)_pa_stream_set_write_callback;
    pContext->pulse.pa_stream_set_read_callback        = (ma_proc)_pa_stream_set_read_callback;
    pContext->pulse.pa_stream_flush                    = (ma_proc)_pa_stream_flush;
    pContext->pulse.pa_stream_drain                    = (ma_proc)_pa_stream_drain;
    pContext->pulse.pa_stream_is_corked                = (ma_proc)_pa_stream_is_corked;
    pContext->pulse.pa_stream_cork                     = (ma_proc)_pa_stream_cork;
    pContext->pulse.pa_stream_trigger                  = (ma_proc)_pa_stream_trigger;
    pContext->pulse.pa_stream_begin_write              = (ma_proc)_pa_stream_begin_write;
    pContext->pulse.pa_stream_write                    = (ma_proc)_pa_stream_write;
    pContext->pulse.pa_stream_peek                     = (ma_proc)_pa_stream_peek;
    pContext->pulse.pa_stream_drop                     = (ma_proc)_pa_stream_drop;
    pContext->pulse.pa_stream_writable_size            = (ma_proc)_pa_stream_writable_size;
    pContext->pulse.pa_stream_readable_size            = (ma_proc)_pa_stream_readable_size;
#endif

    pContext->onUninit         = ma_context_uninit__pulse;
    pContext->onDeviceIDEqual  = ma_context_is_device_id_equal__pulse;
    pContext->onEnumDevices    = ma_context_enumerate_devices__pulse;
    pContext->onGetDeviceInfo  = ma_context_get_device_info__pulse;
    pContext->onDeviceInit     = ma_device_init__pulse;
    pContext->onDeviceUninit   = ma_device_uninit__pulse;
    pContext->onDeviceStart    = NULL;
    pContext->onDeviceStop     = NULL;
    pContext->onDeviceMainLoop = ma_device_main_loop__pulse;

    if (pConfig->pulse.pApplicationName) {
        pContext->pulse.pApplicationName = ma_copy_string(pConfig->pulse.pApplicationName, &pContext->allocationCallbacks);
    }
    if (pConfig->pulse.pServerName) {
        pContext->pulse.pServerName = ma_copy_string(pConfig->pulse.pServerName, &pContext->allocationCallbacks);
    }
    pContext->pulse.tryAutoSpawn = pConfig->pulse.tryAutoSpawn;
    
    /*
    Although we have found the libpulse library, it doesn't necessarily mean PulseAudio is useable. We need to initialize
    and connect a dummy PulseAudio context to test PulseAudio's usability.
    */
    {
        ma_pa_mainloop* pMainLoop;
        ma_pa_mainloop_api* pAPI;
        ma_pa_context* pPulseContext;
        int error;

        pMainLoop = ((ma_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
        if (pMainLoop == NULL) {
            ma_free(pContext->pulse.pServerName, &pContext->allocationCallbacks);
            ma_free(pContext->pulse.pApplicationName, &pContext->allocationCallbacks);
        #ifndef MA_NO_RUNTIME_LINKING
            ma_dlclose(pContext, pContext->pulse.pulseSO);
        #endif
            return MA_NO_BACKEND;
        }

        pAPI = ((ma_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)(pMainLoop);
        if (pAPI == NULL) {
            ma_free(pContext->pulse.pServerName, &pContext->allocationCallbacks);
            ma_free(pContext->pulse.pApplicationName, &pContext->allocationCallbacks);
            ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        #ifndef MA_NO_RUNTIME_LINKING
            ma_dlclose(pContext, pContext->pulse.pulseSO);
        #endif
            return MA_NO_BACKEND;
        }

        pPulseContext = ((ma_pa_context_new_proc)pContext->pulse.pa_context_new)(pAPI, pContext->pulse.pApplicationName);
        if (pPulseContext == NULL) {
            ma_free(pContext->pulse.pServerName, &pContext->allocationCallbacks);
            ma_free(pContext->pulse.pApplicationName, &pContext->allocationCallbacks);
            ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        #ifndef MA_NO_RUNTIME_LINKING
            ma_dlclose(pContext, pContext->pulse.pulseSO);
        #endif
            return MA_NO_BACKEND;
        }

        error = ((ma_pa_context_connect_proc)pContext->pulse.pa_context_connect)(pPulseContext, pContext->pulse.pServerName, 0, NULL);
        if (error != MA_PA_OK) {
            ma_free(pContext->pulse.pServerName, &pContext->allocationCallbacks);
            ma_free(pContext->pulse.pApplicationName, &pContext->allocationCallbacks);
            ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
            ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        #ifndef MA_NO_RUNTIME_LINKING
            ma_dlclose(pContext, pContext->pulse.pulseSO);
        #endif
            return MA_NO_BACKEND;
        }

        ((ma_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)(pPulseContext);
        ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
    }

    return MA_SUCCESS;
}
#endif


/******************************************************************************

JACK Backend

******************************************************************************/
#ifdef MA_HAS_JACK

/* It is assumed jack.h is available when compile-time linking is being used. */
#ifdef MA_NO_RUNTIME_LINKING
#include <jack/jack.h>

typedef jack_nframes_t              ma_jack_nframes_t;
typedef jack_options_t              ma_jack_options_t;
typedef jack_status_t               ma_jack_status_t;
typedef jack_client_t               ma_jack_client_t;
typedef jack_port_t                 ma_jack_port_t;
typedef JackProcessCallback         ma_JackProcessCallback;
typedef JackBufferSizeCallback      ma_JackBufferSizeCallback;
typedef JackShutdownCallback        ma_JackShutdownCallback;
#define MA_JACK_DEFAULT_AUDIO_TYPE  JACK_DEFAULT_AUDIO_TYPE
#define ma_JackNoStartServer        JackNoStartServer
#define ma_JackPortIsInput          JackPortIsInput
#define ma_JackPortIsOutput         JackPortIsOutput
#define ma_JackPortIsPhysical       JackPortIsPhysical
#else
typedef ma_uint32               ma_jack_nframes_t;
typedef int                     ma_jack_options_t;
typedef int                     ma_jack_status_t;
typedef struct ma_jack_client_t ma_jack_client_t;
typedef struct ma_jack_port_t   ma_jack_port_t;
typedef int  (* ma_JackProcessCallback)   (ma_jack_nframes_t nframes, void* arg);
typedef int  (* ma_JackBufferSizeCallback)(ma_jack_nframes_t nframes, void* arg);
typedef void (* ma_JackShutdownCallback)  (void* arg);
#define MA_JACK_DEFAULT_AUDIO_TYPE "32 bit float mono audio"
#define ma_JackNoStartServer       1
#define ma_JackPortIsInput         1
#define ma_JackPortIsOutput        2
#define ma_JackPortIsPhysical      4
#endif

typedef ma_jack_client_t* (* ma_jack_client_open_proc)             (const char* client_name, ma_jack_options_t options, ma_jack_status_t* status, ...);
typedef int               (* ma_jack_client_close_proc)            (ma_jack_client_t* client);
typedef int               (* ma_jack_client_name_size_proc)        (void);
typedef int               (* ma_jack_set_process_callback_proc)    (ma_jack_client_t* client, ma_JackProcessCallback process_callback, void* arg);
typedef int               (* ma_jack_set_buffer_size_callback_proc)(ma_jack_client_t* client, ma_JackBufferSizeCallback bufsize_callback, void* arg);
typedef void              (* ma_jack_on_shutdown_proc)             (ma_jack_client_t* client, ma_JackShutdownCallback function, void* arg);
typedef ma_jack_nframes_t (* ma_jack_get_sample_rate_proc)         (ma_jack_client_t* client);
typedef ma_jack_nframes_t (* ma_jack_get_buffer_size_proc)         (ma_jack_client_t* client);
typedef const char**      (* ma_jack_get_ports_proc)               (ma_jack_client_t* client, const char* port_name_pattern, const char* type_name_pattern, unsigned long flags);
typedef int               (* ma_jack_activate_proc)                (ma_jack_client_t* client);
typedef int               (* ma_jack_deactivate_proc)              (ma_jack_client_t* client);
typedef int               (* ma_jack_connect_proc)                 (ma_jack_client_t* client, const char* source_port, const char* destination_port);
typedef ma_jack_port_t*   (* ma_jack_port_register_proc)           (ma_jack_client_t* client, const char* port_name, const char* port_type, unsigned long flags, unsigned long buffer_size);
typedef const char*       (* ma_jack_port_name_proc)               (const ma_jack_port_t* port);
typedef void*             (* ma_jack_port_get_buffer_proc)         (ma_jack_port_t* port, ma_jack_nframes_t nframes);
typedef void              (* ma_jack_free_proc)                    (void* ptr);

static ma_result ma_context_open_client__jack(ma_context* pContext, ma_jack_client_t** ppClient)
{
    size_t maxClientNameSize;
    char clientName[256];
    ma_jack_status_t status;
    ma_jack_client_t* pClient;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(ppClient != NULL);

    if (ppClient) {
        *ppClient = NULL;
    }

    maxClientNameSize = ((ma_jack_client_name_size_proc)pContext->jack.jack_client_name_size)(); /* Includes null terminator. */
    ma_strncpy_s(clientName, ma_min(sizeof(clientName), maxClientNameSize), (pContext->jack.pClientName != NULL) ? pContext->jack.pClientName : "miniaudio", (size_t)-1);

    pClient = ((ma_jack_client_open_proc)pContext->jack.jack_client_open)(clientName, (pContext->jack.tryStartServer) ? 0 : ma_JackNoStartServer, &status, NULL);
    if (pClient == NULL) {
        return MA_FAILED_TO_OPEN_BACKEND_DEVICE;
    }

    if (ppClient) {
        *ppClient = pClient;
    }

    return MA_SUCCESS;
}

static ma_bool32 ma_context_is_device_id_equal__jack(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pID0 != NULL);
    MA_ASSERT(pID1 != NULL);
    (void)pContext;

    return pID0->jack == pID1->jack;
}

static ma_result ma_context_enumerate_devices__jack(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_bool32 cbResult = MA_TRUE;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(callback != NULL);

    /* Playback. */
    if (cbResult) {
        ma_device_info deviceInfo;
        MA_ZERO_OBJECT(&deviceInfo);
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
    }

    /* Capture. */
    if (cbResult) {
        ma_device_info deviceInfo;
        MA_ZERO_OBJECT(&deviceInfo);
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
    }

    return MA_SUCCESS;
}

static ma_result ma_context_get_device_info__jack(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_jack_client_t* pClient;
    ma_result result;
    const char** ppPorts;

    MA_ASSERT(pContext != NULL);

    /* No exclusive mode with the JACK backend. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    if (pDeviceID != NULL && pDeviceID->jack != 0) {
        return MA_NO_DEVICE;   /* Don't know the device. */
    }

    /* Name / Description */
    if (deviceType == ma_device_type_playback) {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    } else {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    }

    /* Jack only supports f32 and has a specific channel count and sample rate. */
    pDeviceInfo->formatCount = 1;
    pDeviceInfo->formats[0] = ma_format_f32;

    /* The channel count and sample rate can only be determined by opening the device. */
    result = ma_context_open_client__jack(pContext, &pClient);
    if (result != MA_SUCCESS) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[JACK] Failed to open client.", result);
    }

    pDeviceInfo->minSampleRate = ((ma_jack_get_sample_rate_proc)pContext->jack.jack_get_sample_rate)((ma_jack_client_t*)pClient);
    pDeviceInfo->maxSampleRate = pDeviceInfo->minSampleRate;

    pDeviceInfo->minChannels = 0;
    pDeviceInfo->maxChannels = 0;

    ppPorts = ((ma_jack_get_ports_proc)pContext->jack.jack_get_ports)((ma_jack_client_t*)pClient, NULL, MA_JACK_DEFAULT_AUDIO_TYPE, ma_JackPortIsPhysical | ((deviceType == ma_device_type_playback) ? ma_JackPortIsInput : ma_JackPortIsOutput));
    if (ppPorts == NULL) {
        ((ma_jack_client_close_proc)pContext->jack.jack_client_close)((ma_jack_client_t*)pClient);
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[JACK] Failed to query physical ports.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    while (ppPorts[pDeviceInfo->minChannels] != NULL) {
        pDeviceInfo->minChannels += 1;
        pDeviceInfo->maxChannels += 1;
    }

    ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);
    ((ma_jack_client_close_proc)pContext->jack.jack_client_close)((ma_jack_client_t*)pClient);

    (void)pContext;
    return MA_SUCCESS;
}


static void ma_device_uninit__jack(ma_device* pDevice)
{
    ma_context* pContext;

    MA_ASSERT(pDevice != NULL);

    pContext = pDevice->pContext;
    MA_ASSERT(pContext != NULL);

    if (pDevice->jack.pClient != NULL) {
        ((ma_jack_client_close_proc)pContext->jack.jack_client_close)((ma_jack_client_t*)pDevice->jack.pClient);
    }

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ma__free_from_callbacks(pDevice->jack.pIntermediaryBufferCapture, &pDevice->pContext->allocationCallbacks);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ma__free_from_callbacks(pDevice->jack.pIntermediaryBufferPlayback, &pDevice->pContext->allocationCallbacks);
    }

    if (pDevice->type == ma_device_type_duplex) {
        ma_pcm_rb_uninit(&pDevice->jack.duplexRB);
    }
}

static void ma_device__jack_shutdown_callback(void* pUserData)
{
    /* JACK died. Stop the device. */
    ma_device* pDevice = (ma_device*)pUserData;
    MA_ASSERT(pDevice != NULL);

    ma_device_stop(pDevice);
}

static int ma_device__jack_buffer_size_callback(ma_jack_nframes_t frameCount, void* pUserData)
{
    ma_device* pDevice = (ma_device*)pUserData;
    MA_ASSERT(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        size_t newBufferSize = frameCount * (pDevice->capture.internalChannels * ma_get_bytes_per_sample(pDevice->capture.internalFormat));
        float* pNewBuffer = (float*)ma__calloc_from_callbacks(newBufferSize, &pDevice->pContext->allocationCallbacks);
        if (pNewBuffer == NULL) {
            return MA_OUT_OF_MEMORY;
        }

        ma__free_from_callbacks(pDevice->jack.pIntermediaryBufferCapture, &pDevice->pContext->allocationCallbacks);

        pDevice->jack.pIntermediaryBufferCapture = pNewBuffer;
        pDevice->playback.internalPeriodSizeInFrames = frameCount;
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        size_t newBufferSize = frameCount * (pDevice->playback.internalChannels * ma_get_bytes_per_sample(pDevice->playback.internalFormat));
        float* pNewBuffer = (float*)ma__calloc_from_callbacks(newBufferSize, &pDevice->pContext->allocationCallbacks);
        if (pNewBuffer == NULL) {
            return MA_OUT_OF_MEMORY;
        }

        ma__free_from_callbacks(pDevice->jack.pIntermediaryBufferPlayback, &pDevice->pContext->allocationCallbacks);

        pDevice->jack.pIntermediaryBufferPlayback = pNewBuffer;
        pDevice->playback.internalPeriodSizeInFrames = frameCount;
    }

    return 0;
}

static int ma_device__jack_process_callback(ma_jack_nframes_t frameCount, void* pUserData)
{
    ma_device* pDevice;
    ma_context* pContext;
    ma_uint32 iChannel;

    pDevice = (ma_device*)pUserData;
    MA_ASSERT(pDevice != NULL);

    pContext = pDevice->pContext;
    MA_ASSERT(pContext != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        /* Channels need to be interleaved. */
        for (iChannel = 0; iChannel < pDevice->capture.internalChannels; ++iChannel) {
            const float* pSrc = (const float*)((ma_jack_port_get_buffer_proc)pContext->jack.jack_port_get_buffer)((ma_jack_port_t*)pDevice->jack.pPortsCapture[iChannel], frameCount);
            if (pSrc != NULL) {
                float* pDst = pDevice->jack.pIntermediaryBufferCapture + iChannel;
                ma_jack_nframes_t iFrame;
                for (iFrame = 0; iFrame < frameCount; ++iFrame) {
                    *pDst = *pSrc;

                    pDst += pDevice->capture.internalChannels;
                    pSrc += 1;
                }
            }
        }

        if (pDevice->type == ma_device_type_duplex) {
            ma_device__handle_duplex_callback_capture(pDevice, frameCount, pDevice->jack.pIntermediaryBufferCapture, &pDevice->jack.duplexRB);
        } else {
            ma_device__send_frames_to_client(pDevice, frameCount, pDevice->jack.pIntermediaryBufferCapture);
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        if (pDevice->type == ma_device_type_duplex) {
            ma_device__handle_duplex_callback_playback(pDevice, frameCount, pDevice->jack.pIntermediaryBufferPlayback, &pDevice->jack.duplexRB);
        } else {
            ma_device__read_frames_from_client(pDevice, frameCount, pDevice->jack.pIntermediaryBufferPlayback);
        }

        /* Channels need to be deinterleaved. */
        for (iChannel = 0; iChannel < pDevice->playback.internalChannels; ++iChannel) {
            float* pDst = (float*)((ma_jack_port_get_buffer_proc)pContext->jack.jack_port_get_buffer)((ma_jack_port_t*)pDevice->jack.pPortsPlayback[iChannel], frameCount);
            if (pDst != NULL) {
                const float* pSrc = pDevice->jack.pIntermediaryBufferPlayback + iChannel;
                ma_jack_nframes_t iFrame;
                for (iFrame = 0; iFrame < frameCount; ++iFrame) {
                    *pDst = *pSrc;

                    pDst += 1;
                    pSrc += pDevice->playback.internalChannels;
                }
            }
        }
    }

    return 0;
}

static ma_result ma_device_init__jack(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result;
    ma_uint32 periods;
    ma_uint32 periodSizeInFrames;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pConfig != NULL);
    MA_ASSERT(pDevice != NULL);

    if (pConfig->deviceType == ma_device_type_loopback) {
        return MA_DEVICE_TYPE_NOT_SUPPORTED;
    }

    /* Only supporting default devices with JACK. */
    if (((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.pDeviceID != NULL && pConfig->playback.pDeviceID->jack != 0) ||
        ((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.pDeviceID  != NULL && pConfig->capture.pDeviceID->jack  != 0)) {
        return MA_NO_DEVICE;
    }

    /* No exclusive mode with the JACK backend. */
    if (((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.shareMode == ma_share_mode_exclusive) ||
        ((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.shareMode  == ma_share_mode_exclusive)) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    /* Open the client. */
    result = ma_context_open_client__jack(pContext, (ma_jack_client_t**)&pDevice->jack.pClient);
    if (result != MA_SUCCESS) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to open client.", result);
    }

    /* Callbacks. */
    if (((ma_jack_set_process_callback_proc)pContext->jack.jack_set_process_callback)((ma_jack_client_t*)pDevice->jack.pClient, ma_device__jack_process_callback, pDevice) != 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to set process callback.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }
    if (((ma_jack_set_buffer_size_callback_proc)pContext->jack.jack_set_buffer_size_callback)((ma_jack_client_t*)pDevice->jack.pClient, ma_device__jack_buffer_size_callback, pDevice) != 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to set buffer size callback.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    ((ma_jack_on_shutdown_proc)pContext->jack.jack_on_shutdown)((ma_jack_client_t*)pDevice->jack.pClient, ma_device__jack_shutdown_callback, pDevice);


    /* The buffer size in frames can change. */
    periods            = pConfig->periods;
    periodSizeInFrames = ((ma_jack_get_buffer_size_proc)pContext->jack.jack_get_buffer_size)((ma_jack_client_t*)pDevice->jack.pClient);
    
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        const char** ppPorts;

        pDevice->capture.internalFormat = ma_format_f32;
        pDevice->capture.internalChannels = 0;
        pDevice->capture.internalSampleRate = ((ma_jack_get_sample_rate_proc)pContext->jack.jack_get_sample_rate)((ma_jack_client_t*)pDevice->jack.pClient);
        ma_get_standard_channel_map(ma_standard_channel_map_alsa, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);

        ppPorts = ((ma_jack_get_ports_proc)pContext->jack.jack_get_ports)((ma_jack_client_t*)pDevice->jack.pClient, NULL, MA_JACK_DEFAULT_AUDIO_TYPE, ma_JackPortIsPhysical | ma_JackPortIsOutput);
        if (ppPorts == NULL) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to query physical ports.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        while (ppPorts[pDevice->capture.internalChannels] != NULL) {
            char name[64];
            ma_strcpy_s(name, sizeof(name), "capture");
            ma_itoa_s((int)pDevice->capture.internalChannels, name+7, sizeof(name)-7, 10); /* 7 = length of "capture" */

            pDevice->jack.pPortsCapture[pDevice->capture.internalChannels] = ((ma_jack_port_register_proc)pContext->jack.jack_port_register)((ma_jack_client_t*)pDevice->jack.pClient, name, MA_JACK_DEFAULT_AUDIO_TYPE, ma_JackPortIsInput, 0);
            if (pDevice->jack.pPortsCapture[pDevice->capture.internalChannels] == NULL) {
                ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);
                ma_device_uninit__jack(pDevice);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to register ports.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
            }

            pDevice->capture.internalChannels += 1;
        }

        ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);

        pDevice->capture.internalPeriodSizeInFrames = periodSizeInFrames;
        pDevice->capture.internalPeriods            = periods;

        pDevice->jack.pIntermediaryBufferCapture = (float*)ma__calloc_from_callbacks(pDevice->capture.internalPeriodSizeInFrames * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels), &pContext->allocationCallbacks);
        if (pDevice->jack.pIntermediaryBufferCapture == NULL) {
            ma_device_uninit__jack(pDevice);
            return MA_OUT_OF_MEMORY;
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        const char** ppPorts;

        pDevice->playback.internalFormat = ma_format_f32;
        pDevice->playback.internalChannels = 0;
        pDevice->playback.internalSampleRate = ((ma_jack_get_sample_rate_proc)pContext->jack.jack_get_sample_rate)((ma_jack_client_t*)pDevice->jack.pClient);
        ma_get_standard_channel_map(ma_standard_channel_map_alsa, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);

        ppPorts = ((ma_jack_get_ports_proc)pContext->jack.jack_get_ports)((ma_jack_client_t*)pDevice->jack.pClient, NULL, MA_JACK_DEFAULT_AUDIO_TYPE, ma_JackPortIsPhysical | ma_JackPortIsInput);
        if (ppPorts == NULL) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to query physical ports.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        while (ppPorts[pDevice->playback.internalChannels] != NULL) {
            char name[64];
            ma_strcpy_s(name, sizeof(name), "playback");
            ma_itoa_s((int)pDevice->playback.internalChannels, name+8, sizeof(name)-8, 10); /* 8 = length of "playback" */

            pDevice->jack.pPortsPlayback[pDevice->playback.internalChannels] = ((ma_jack_port_register_proc)pContext->jack.jack_port_register)((ma_jack_client_t*)pDevice->jack.pClient, name, MA_JACK_DEFAULT_AUDIO_TYPE, ma_JackPortIsOutput, 0);
            if (pDevice->jack.pPortsPlayback[pDevice->playback.internalChannels] == NULL) {
                ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);
                ma_device_uninit__jack(pDevice);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to register ports.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
            }

            pDevice->playback.internalChannels += 1;
        }

        ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);

        pDevice->playback.internalPeriodSizeInFrames = periodSizeInFrames;
        pDevice->playback.internalPeriods            = periods;

        pDevice->jack.pIntermediaryBufferPlayback = (float*)ma__calloc_from_callbacks(pDevice->playback.internalPeriodSizeInFrames * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels), &pContext->allocationCallbacks);
        if (pDevice->jack.pIntermediaryBufferPlayback == NULL) {
            ma_device_uninit__jack(pDevice);
            return MA_OUT_OF_MEMORY;
        }
    }

    if (pDevice->type == ma_device_type_duplex) {
        ma_uint32 rbSizeInFrames = (ma_uint32)ma_calculate_frame_count_after_resampling(pDevice->sampleRate, pDevice->capture.internalSampleRate, pDevice->capture.internalPeriodSizeInFrames * pDevice->capture.internalPeriods);
        result = ma_pcm_rb_init(pDevice->capture.format, pDevice->capture.channels, rbSizeInFrames, NULL, &pDevice->pContext->allocationCallbacks, &pDevice->jack.duplexRB);
        if (result != MA_SUCCESS) {
            ma_device_uninit__jack(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to initialize ring buffer.", result);
        }

        /* We need a period to act as a buffer for cases where the playback and capture device's end up desyncing. */
        {
            ma_uint32 marginSizeInFrames = rbSizeInFrames / pDevice->capture.internalPeriods;
            void* pMarginData;
            ma_pcm_rb_acquire_write(&pDevice->jack.duplexRB, &marginSizeInFrames, &pMarginData);
            {
                MA_ZERO_MEMORY(pMarginData, marginSizeInFrames * ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels));
            }
            ma_pcm_rb_commit_write(&pDevice->jack.duplexRB, marginSizeInFrames, pMarginData);
        }
    }

    return MA_SUCCESS;
}


static ma_result ma_device_start__jack(ma_device* pDevice)
{
    ma_context* pContext = pDevice->pContext;
    int resultJACK;
    size_t i;

    resultJACK = ((ma_jack_activate_proc)pContext->jack.jack_activate)((ma_jack_client_t*)pDevice->jack.pClient);
    if (resultJACK != 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to activate the JACK client.", MA_FAILED_TO_START_BACKEND_DEVICE);
    }

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        const char** ppServerPorts = ((ma_jack_get_ports_proc)pContext->jack.jack_get_ports)((ma_jack_client_t*)pDevice->jack.pClient, NULL, MA_JACK_DEFAULT_AUDIO_TYPE, ma_JackPortIsPhysical | ma_JackPortIsOutput);
        if (ppServerPorts == NULL) {
            ((ma_jack_deactivate_proc)pContext->jack.jack_deactivate)((ma_jack_client_t*)pDevice->jack.pClient);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to retrieve physical ports.", MA_ERROR);
        }

        for (i = 0; ppServerPorts[i] != NULL; ++i) {
            const char* pServerPort = ppServerPorts[i];
            const char* pClientPort = ((ma_jack_port_name_proc)pContext->jack.jack_port_name)((ma_jack_port_t*)pDevice->jack.pPortsCapture[i]);

            resultJACK = ((ma_jack_connect_proc)pContext->jack.jack_connect)((ma_jack_client_t*)pDevice->jack.pClient, pServerPort, pClientPort);
            if (resultJACK != 0) {
                ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppServerPorts);
                ((ma_jack_deactivate_proc)pContext->jack.jack_deactivate)((ma_jack_client_t*)pDevice->jack.pClient);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to connect ports.", MA_ERROR);
            }
        }

        ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppServerPorts);
    }
    
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        const char** ppServerPorts = ((ma_jack_get_ports_proc)pContext->jack.jack_get_ports)((ma_jack_client_t*)pDevice->jack.pClient, NULL, MA_JACK_DEFAULT_AUDIO_TYPE, ma_JackPortIsPhysical | ma_JackPortIsInput);
        if (ppServerPorts == NULL) {
            ((ma_jack_deactivate_proc)pContext->jack.jack_deactivate)((ma_jack_client_t*)pDevice->jack.pClient);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to retrieve physical ports.", MA_ERROR);
        }

        for (i = 0; ppServerPorts[i] != NULL; ++i) {
            const char* pServerPort = ppServerPorts[i];
            const char* pClientPort = ((ma_jack_port_name_proc)pContext->jack.jack_port_name)((ma_jack_port_t*)pDevice->jack.pPortsPlayback[i]);

            resultJACK = ((ma_jack_connect_proc)pContext->jack.jack_connect)((ma_jack_client_t*)pDevice->jack.pClient, pClientPort, pServerPort);
            if (resultJACK != 0) {
                ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppServerPorts);
                ((ma_jack_deactivate_proc)pContext->jack.jack_deactivate)((ma_jack_client_t*)pDevice->jack.pClient);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to connect ports.", MA_ERROR);
            }
        }

        ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppServerPorts);
    }

    return MA_SUCCESS;
}

static ma_result ma_device_stop__jack(ma_device* pDevice)
{
    ma_context* pContext = pDevice->pContext;
    ma_stop_proc onStop;

    if (((ma_jack_deactivate_proc)pContext->jack.jack_deactivate)((ma_jack_client_t*)pDevice->jack.pClient) != 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] An error occurred when deactivating the JACK client.", MA_ERROR);
    }
    
    onStop = pDevice->onStop;
    if (onStop) {
        onStop(pDevice);
    }

    return MA_SUCCESS;
}


static ma_result ma_context_uninit__jack(ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pContext->backend == ma_backend_jack);

    ma_free(pContext->jack.pClientName, &pContext->allocationCallbacks);
    pContext->jack.pClientName = NULL;

#ifndef MA_NO_RUNTIME_LINKING
    ma_dlclose(pContext, pContext->jack.jackSO);
#endif

    return MA_SUCCESS;
}

static ma_result ma_context_init__jack(const ma_context_config* pConfig, ma_context* pContext)
{
#ifndef MA_NO_RUNTIME_LINKING
    const char* libjackNames[] = {
#ifdef MA_WIN32
        "libjack.dll"
#else
        "libjack.so",
        "libjack.so.0"
#endif
    };
    size_t i;

    for (i = 0; i < ma_countof(libjackNames); ++i) {
        pContext->jack.jackSO = ma_dlopen(pContext, libjackNames[i]);
        if (pContext->jack.jackSO != NULL) {
            break;
        }
    }

    if (pContext->jack.jackSO == NULL) {
        return MA_NO_BACKEND;
    }

    pContext->jack.jack_client_open              = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_client_open");
    pContext->jack.jack_client_close             = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_client_close");
    pContext->jack.jack_client_name_size         = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_client_name_size");
    pContext->jack.jack_set_process_callback     = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_set_process_callback");
    pContext->jack.jack_set_buffer_size_callback = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_set_buffer_size_callback");
    pContext->jack.jack_on_shutdown              = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_on_shutdown");
    pContext->jack.jack_get_sample_rate          = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_get_sample_rate");
    pContext->jack.jack_get_buffer_size          = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_get_buffer_size");
    pContext->jack.jack_get_ports                = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_get_ports");
    pContext->jack.jack_activate                 = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_activate");
    pContext->jack.jack_deactivate               = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_deactivate");
    pContext->jack.jack_connect                  = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_connect");
    pContext->jack.jack_port_register            = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_port_register");
    pContext->jack.jack_port_name                = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_port_name");
    pContext->jack.jack_port_get_buffer          = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_port_get_buffer");
    pContext->jack.jack_free                     = (ma_proc)ma_dlsym(pContext, pContext->jack.jackSO, "jack_free");
#else
    /*
    This strange assignment system is here just to ensure type safety of miniaudio's function pointer
    types. If anything differs slightly the compiler should throw a warning.
    */
    ma_jack_client_open_proc              _jack_client_open              = jack_client_open;
    ma_jack_client_close_proc             _jack_client_close             = jack_client_close;
    ma_jack_client_name_size_proc         _jack_client_name_size         = jack_client_name_size;
    ma_jack_set_process_callback_proc     _jack_set_process_callback     = jack_set_process_callback;
    ma_jack_set_buffer_size_callback_proc _jack_set_buffer_size_callback = jack_set_buffer_size_callback;
    ma_jack_on_shutdown_proc              _jack_on_shutdown              = jack_on_shutdown;
    ma_jack_get_sample_rate_proc          _jack_get_sample_rate          = jack_get_sample_rate;
    ma_jack_get_buffer_size_proc          _jack_get_buffer_size          = jack_get_buffer_size;
    ma_jack_get_ports_proc                _jack_get_ports                = jack_get_ports;
    ma_jack_activate_proc                 _jack_activate                 = jack_activate;
    ma_jack_deactivate_proc               _jack_deactivate               = jack_deactivate;
    ma_jack_connect_proc                  _jack_connect                  = jack_connect;
    ma_jack_port_register_proc            _jack_port_register            = jack_port_register;
    ma_jack_port_name_proc                _jack_port_name                = jack_port_name;
    ma_jack_port_get_buffer_proc          _jack_port_get_buffer          = jack_port_get_buffer;
    ma_jack_free_proc                     _jack_free                     = jack_free;

    pContext->jack.jack_client_open              = (ma_proc)_jack_client_open;
    pContext->jack.jack_client_close             = (ma_proc)_jack_client_close;
    pContext->jack.jack_client_name_size         = (ma_proc)_jack_client_name_size;
    pContext->jack.jack_set_process_callback     = (ma_proc)_jack_set_process_callback;
    pContext->jack.jack_set_buffer_size_callback = (ma_proc)_jack_set_buffer_size_callback;
    pContext->jack.jack_on_shutdown              = (ma_proc)_jack_on_shutdown;
    pContext->jack.jack_get_sample_rate          = (ma_proc)_jack_get_sample_rate;
    pContext->jack.jack_get_buffer_size          = (ma_proc)_jack_get_buffer_size;
    pContext->jack.jack_get_ports                = (ma_proc)_jack_get_ports;
    pContext->jack.jack_activate                 = (ma_proc)_jack_activate;
    pContext->jack.jack_deactivate               = (ma_proc)_jack_deactivate;
    pContext->jack.jack_connect                  = (ma_proc)_jack_connect;
    pContext->jack.jack_port_register            = (ma_proc)_jack_port_register;
    pContext->jack.jack_port_name                = (ma_proc)_jack_port_name;
    pContext->jack.jack_port_get_buffer          = (ma_proc)_jack_port_get_buffer;
    pContext->jack.jack_free                     = (ma_proc)_jack_free;
#endif

    pContext->isBackendAsynchronous = MA_TRUE;

    pContext->onUninit        = ma_context_uninit__jack;
    pContext->onDeviceIDEqual = ma_context_is_device_id_equal__jack;
    pContext->onEnumDevices   = ma_context_enumerate_devices__jack;
    pContext->onGetDeviceInfo = ma_context_get_device_info__jack;
    pContext->onDeviceInit    = ma_device_init__jack;
    pContext->onDeviceUninit  = ma_device_uninit__jack;
    pContext->onDeviceStart   = ma_device_start__jack;
    pContext->onDeviceStop    = ma_device_stop__jack;

    if (pConfig->jack.pClientName != NULL) {
        pContext->jack.pClientName = ma_copy_string(pConfig->jack.pClientName, &pContext->allocationCallbacks);
    }
    pContext->jack.tryStartServer = pConfig->jack.tryStartServer;

    /*
    Getting here means the JACK library is installed, but it doesn't necessarily mean it's usable. We need to quickly test this by connecting
    a temporary client.
    */
    {
        ma_jack_client_t* pDummyClient;
        ma_result result = ma_context_open_client__jack(pContext, &pDummyClient);
        if (result != MA_SUCCESS) {
            ma_free(pContext->jack.pClientName, &pContext->allocationCallbacks);
        #ifndef MA_NO_RUNTIME_LINKING
            ma_dlclose(pContext, pContext->jack.jackSO);
        #endif
            return MA_NO_BACKEND;
        }

        ((ma_jack_client_close_proc)pContext->jack.jack_client_close)((ma_jack_client_t*)pDummyClient);
    }

    return MA_SUCCESS;
}
#endif  /* JACK */



/******************************************************************************

Core Audio Backend

******************************************************************************/
#ifdef MA_HAS_COREAUDIO
#include <TargetConditionals.h>

#if defined(TARGET_OS_IPHONE) && TARGET_OS_IPHONE == 1
    #define MA_APPLE_MOBILE
    #if defined(TARGET_OS_TV) && TARGET_OS_TV == 1
        #define MA_APPLE_TV
    #endif
    #if defined(TARGET_OS_WATCH) && TARGET_OS_WATCH == 1
        #define MA_APPLE_WATCH
    #endif
#else
    #define MA_APPLE_DESKTOP
#endif

#if defined(MA_APPLE_DESKTOP)
#include <CoreAudio/CoreAudio.h>
#else
#include <AVFoundation/AVFoundation.h>
#endif

#include <AudioToolbox/AudioToolbox.h>

/* CoreFoundation */
typedef Boolean (* ma_CFStringGetCString_proc)(CFStringRef theString, char* buffer, CFIndex bufferSize, CFStringEncoding encoding);
typedef void (* ma_CFRelease_proc)(CFTypeRef cf);

/* CoreAudio */
#if defined(MA_APPLE_DESKTOP)
typedef OSStatus (* ma_AudioObjectGetPropertyData_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32* ioDataSize, void* outData);
typedef OSStatus (* ma_AudioObjectGetPropertyDataSize_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32* outDataSize);
typedef OSStatus (* ma_AudioObjectSetPropertyData_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32 inDataSize, const void* inData);
typedef OSStatus (* ma_AudioObjectAddPropertyListener_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, AudioObjectPropertyListenerProc inListener, void* inClientData);
typedef OSStatus (* ma_AudioObjectRemovePropertyListener_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, AudioObjectPropertyListenerProc inListener, void* inClientData);
#endif

/* AudioToolbox */
typedef AudioComponent (* ma_AudioComponentFindNext_proc)(AudioComponent inComponent, const AudioComponentDescription* inDesc);
typedef OSStatus (* ma_AudioComponentInstanceDispose_proc)(AudioComponentInstance inInstance);
typedef OSStatus (* ma_AudioComponentInstanceNew_proc)(AudioComponent inComponent, AudioComponentInstance* outInstance);
typedef OSStatus (* ma_AudioOutputUnitStart_proc)(AudioUnit inUnit);
typedef OSStatus (* ma_AudioOutputUnitStop_proc)(AudioUnit inUnit);
typedef OSStatus (* ma_AudioUnitAddPropertyListener_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitPropertyListenerProc inProc, void* inProcUserData);
typedef OSStatus (* ma_AudioUnitGetPropertyInfo_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitScope inScope, AudioUnitElement inElement, UInt32* outDataSize, Boolean* outWriteable);
typedef OSStatus (* ma_AudioUnitGetProperty_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitScope inScope, AudioUnitElement inElement, void* outData, UInt32* ioDataSize);
typedef OSStatus (* ma_AudioUnitSetProperty_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitScope inScope, AudioUnitElement inElement, const void* inData, UInt32 inDataSize);
typedef OSStatus (* ma_AudioUnitInitialize_proc)(AudioUnit inUnit);
typedef OSStatus (* ma_AudioUnitRender_proc)(AudioUnit inUnit, AudioUnitRenderActionFlags* ioActionFlags, const AudioTimeStamp* inTimeStamp, UInt32 inOutputBusNumber, UInt32 inNumberFrames, AudioBufferList* ioData);


#define MA_COREAUDIO_OUTPUT_BUS    0
#define MA_COREAUDIO_INPUT_BUS     1

#if defined(MA_APPLE_DESKTOP)
static ma_result ma_device_reinit_internal__coreaudio(ma_device* pDevice, ma_device_type deviceType, ma_bool32 disposePreviousAudioUnit);
#endif

/*
Core Audio

So far, Core Audio has been the worst backend to work with due to being both unintuitive and having almost no documentation
apart from comments in the headers (which admittedly are quite good). For my own purposes, and for anybody out there whose
needing to figure out how this darn thing works, I'm going to outline a few things here.

Since miniaudio is a fairly low-level API, one of the things it needs is control over specific devices, and it needs to be
able to identify whether or not it can be used as playback and/or capture. The AudioObject API is the only one I've seen
that supports this level of detail. There was some public domain sample code I stumbled across that used the AudioComponent
and AudioUnit APIs, but I couldn't see anything that gave low-level control over device selection and capabilities (the
distinction between playback and capture in particular). Therefore, miniaudio is using the AudioObject API.

Most (all?) functions in the AudioObject API take a AudioObjectID as it's input. This is the device identifier. When
retrieving global information, such as the device list, you use kAudioObjectSystemObject. When retrieving device-specific
data, you pass in the ID for that device. In order to retrieve device-specific IDs you need to enumerate over each of the
devices. This is done using the AudioObjectGetPropertyDataSize() and AudioObjectGetPropertyData() APIs which seem to be
the central APIs for retrieving information about the system and specific devices.

To use the AudioObjectGetPropertyData() API you need to use the notion of a property address. A property address is a
structure with three variables and is used to identify which property you are getting or setting. The first is the "selector"
which is basically the specific property that you're wanting to retrieve or set. The second is the "scope", which is
typically set to kAudioObjectPropertyScopeGlobal, kAudioObjectPropertyScopeInput for input-specific properties and
kAudioObjectPropertyScopeOutput for output-specific properties. The last is the "element" which is always set to
kAudioObjectPropertyElementMaster in miniaudio's case. I don't know of any cases where this would be set to anything different.

Back to the earlier issue of device retrieval, you first use the AudioObjectGetPropertyDataSize() API to retrieve the size
of the raw data which is just a list of AudioDeviceID's. You use the kAudioObjectSystemObject AudioObjectID, and a property
address with the kAudioHardwarePropertyDevices selector and the kAudioObjectPropertyScopeGlobal scope. Once you have the
size, allocate a block of memory of that size and then call AudioObjectGetPropertyData(). The data is just a list of
AudioDeviceID's so just do "dataSize/sizeof(AudioDeviceID)" to know the device count.
*/

static ma_result ma_result_from_OSStatus(OSStatus status)
{
    switch (status)
    {
        case noErr:                                   return MA_SUCCESS;
    #if defined(MA_APPLE_DESKTOP)
        case kAudioHardwareNotRunningError:           return MA_DEVICE_NOT_STARTED;
        case kAudioHardwareUnspecifiedError:          return MA_ERROR;
        case kAudioHardwareUnknownPropertyError:      return MA_INVALID_ARGS;
        case kAudioHardwareBadPropertySizeError:      return MA_INVALID_OPERATION;
        case kAudioHardwareIllegalOperationError:     return MA_INVALID_OPERATION;
        case kAudioHardwareBadObjectError:            return MA_INVALID_ARGS;
        case kAudioHardwareBadDeviceError:            return MA_INVALID_ARGS;
        case kAudioHardwareBadStreamError:            return MA_INVALID_ARGS;
        case kAudioHardwareUnsupportedOperationError: return MA_INVALID_OPERATION;
        case kAudioDeviceUnsupportedFormatError:      return MA_FORMAT_NOT_SUPPORTED;
        case kAudioDevicePermissionsError:            return MA_ACCESS_DENIED;
    #endif
        default:                                      return MA_ERROR;
    }
}

#if 0
static ma_channel ma_channel_from_AudioChannelBitmap(AudioChannelBitmap bit)
{
    switch (bit)
    {
        case kAudioChannelBit_Left:                 return MA_CHANNEL_LEFT;
        case kAudioChannelBit_Right:                return MA_CHANNEL_RIGHT;
        case kAudioChannelBit_Center:               return MA_CHANNEL_FRONT_CENTER;
        case kAudioChannelBit_LFEScreen:            return MA_CHANNEL_LFE;
        case kAudioChannelBit_LeftSurround:         return MA_CHANNEL_BACK_LEFT;
        case kAudioChannelBit_RightSurround:        return MA_CHANNEL_BACK_RIGHT;
        case kAudioChannelBit_LeftCenter:           return MA_CHANNEL_FRONT_LEFT_CENTER;
        case kAudioChannelBit_RightCenter:          return MA_CHANNEL_FRONT_RIGHT_CENTER;
        case kAudioChannelBit_CenterSurround:       return MA_CHANNEL_BACK_CENTER;
        case kAudioChannelBit_LeftSurroundDirect:   return MA_CHANNEL_SIDE_LEFT;
        case kAudioChannelBit_RightSurroundDirect:  return MA_CHANNEL_SIDE_RIGHT;
        case kAudioChannelBit_TopCenterSurround:    return MA_CHANNEL_TOP_CENTER;
        case kAudioChannelBit_VerticalHeightLeft:   return MA_CHANNEL_TOP_FRONT_LEFT;
        case kAudioChannelBit_VerticalHeightCenter: return MA_CHANNEL_TOP_FRONT_CENTER;
        case kAudioChannelBit_VerticalHeightRight:  return MA_CHANNEL_TOP_FRONT_RIGHT;
        case kAudioChannelBit_TopBackLeft:          return MA_CHANNEL_TOP_BACK_LEFT;
        case kAudioChannelBit_TopBackCenter:        return MA_CHANNEL_TOP_BACK_CENTER;
        case kAudioChannelBit_TopBackRight:         return MA_CHANNEL_TOP_BACK_RIGHT;
        default:                                    return MA_CHANNEL_NONE;
    }
}
#endif

static ma_result ma_format_from_AudioStreamBasicDescription(const AudioStreamBasicDescription* pDescription, ma_format* pFormatOut)
{
    MA_ASSERT(pDescription != NULL);
    MA_ASSERT(pFormatOut != NULL);
    
    *pFormatOut = ma_format_unknown;   /* Safety. */
    
    /* There's a few things miniaudio doesn't support. */
    if (pDescription->mFormatID != kAudioFormatLinearPCM) {
        return MA_FORMAT_NOT_SUPPORTED;
    }
    
    /* We don't support any non-packed formats that are aligned high. */
    if ((pDescription->mFormatFlags & kLinearPCMFormatFlagIsAlignedHigh) != 0) {
        return MA_FORMAT_NOT_SUPPORTED;
    }

    /* Only supporting native-endian. */
    if ((ma_is_little_endian() && (pDescription->mFormatFlags & kAudioFormatFlagIsBigEndian) != 0) || (ma_is_big_endian() && (pDescription->mFormatFlags & kAudioFormatFlagIsBigEndian) == 0)) {
        return MA_FORMAT_NOT_SUPPORTED;
    }
    
    /* We are not currently supporting non-interleaved formats (this will be added in a future version of miniaudio). */
    /*if ((pDescription->mFormatFlags & kAudioFormatFlagIsNonInterleaved) != 0) {
        return MA_FORMAT_NOT_SUPPORTED;
    }*/

    if ((pDescription->mFormatFlags & kLinearPCMFormatFlagIsFloat) != 0) {
        if (pDescription->mBitsPerChannel == 32) {
            *pFormatOut = ma_format_f32;
            return MA_SUCCESS;
        }
    } else {
        if ((pDescription->mFormatFlags & kLinearPCMFormatFlagIsSignedInteger) != 0) {
            if (pDescription->mBitsPerChannel == 16) {
                *pFormatOut = ma_format_s16;
                return MA_SUCCESS;
            } else if (pDescription->mBitsPerChannel == 24) {
                if (pDescription->mBytesPerFrame == (pDescription->mBitsPerChannel/8 * pDescription->mChannelsPerFrame)) {
                    *pFormatOut = ma_format_s24;
                    return MA_SUCCESS;
                } else {
                    if (pDescription->mBytesPerFrame/pDescription->mChannelsPerFrame == sizeof(ma_int32)) {
                        /* TODO: Implement ma_format_s24_32. */
                        /**pFormatOut = ma_format_s24_32;*/
                        /*return MA_SUCCESS;*/
                        return MA_FORMAT_NOT_SUPPORTED;
                    }
                }
            } else if (pDescription->mBitsPerChannel == 32) {
                *pFormatOut = ma_format_s32;
                return MA_SUCCESS;
            }
        } else {
            if (pDescription->mBitsPerChannel == 8) {
                *pFormatOut = ma_format_u8;
                return MA_SUCCESS;
            }
        }
    }
    
    /* Getting here means the format is not supported. */
    return MA_FORMAT_NOT_SUPPORTED;
}

#if defined(MA_APPLE_DESKTOP)
static ma_channel ma_channel_from_AudioChannelLabel(AudioChannelLabel label)
{
    switch (label)
    {
        case kAudioChannelLabel_Unknown:              return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Unused:               return MA_CHANNEL_NONE;
        case kAudioChannelLabel_UseCoordinates:       return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Left:                 return MA_CHANNEL_LEFT;
        case kAudioChannelLabel_Right:                return MA_CHANNEL_RIGHT;
        case kAudioChannelLabel_Center:               return MA_CHANNEL_FRONT_CENTER;
        case kAudioChannelLabel_LFEScreen:            return MA_CHANNEL_LFE;
        case kAudioChannelLabel_LeftSurround:         return MA_CHANNEL_BACK_LEFT;
        case kAudioChannelLabel_RightSurround:        return MA_CHANNEL_BACK_RIGHT;
        case kAudioChannelLabel_LeftCenter:           return MA_CHANNEL_FRONT_LEFT_CENTER;
        case kAudioChannelLabel_RightCenter:          return MA_CHANNEL_FRONT_RIGHT_CENTER;
        case kAudioChannelLabel_CenterSurround:       return MA_CHANNEL_BACK_CENTER;
        case kAudioChannelLabel_LeftSurroundDirect:   return MA_CHANNEL_SIDE_LEFT;
        case kAudioChannelLabel_RightSurroundDirect:  return MA_CHANNEL_SIDE_RIGHT;
        case kAudioChannelLabel_TopCenterSurround:    return MA_CHANNEL_TOP_CENTER;
        case kAudioChannelLabel_VerticalHeightLeft:   return MA_CHANNEL_TOP_FRONT_LEFT;
        case kAudioChannelLabel_VerticalHeightCenter: return MA_CHANNEL_TOP_FRONT_CENTER;
        case kAudioChannelLabel_VerticalHeightRight:  return MA_CHANNEL_TOP_FRONT_RIGHT;
        case kAudioChannelLabel_TopBackLeft:          return MA_CHANNEL_TOP_BACK_LEFT;
        case kAudioChannelLabel_TopBackCenter:        return MA_CHANNEL_TOP_BACK_CENTER;
        case kAudioChannelLabel_TopBackRight:         return MA_CHANNEL_TOP_BACK_RIGHT;
        case kAudioChannelLabel_RearSurroundLeft:     return MA_CHANNEL_BACK_LEFT;
        case kAudioChannelLabel_RearSurroundRight:    return MA_CHANNEL_BACK_RIGHT;
        case kAudioChannelLabel_LeftWide:             return MA_CHANNEL_SIDE_LEFT;
        case kAudioChannelLabel_RightWide:            return MA_CHANNEL_SIDE_RIGHT;
        case kAudioChannelLabel_LFE2:                 return MA_CHANNEL_LFE;
        case kAudioChannelLabel_LeftTotal:            return MA_CHANNEL_LEFT;
        case kAudioChannelLabel_RightTotal:           return MA_CHANNEL_RIGHT;
        case kAudioChannelLabel_HearingImpaired:      return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Narration:            return MA_CHANNEL_MONO;
        case kAudioChannelLabel_Mono:                 return MA_CHANNEL_MONO;
        case kAudioChannelLabel_DialogCentricMix:     return MA_CHANNEL_MONO;
        case kAudioChannelLabel_CenterSurroundDirect: return MA_CHANNEL_BACK_CENTER;
        case kAudioChannelLabel_Haptic:               return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_W:          return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_X:          return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_Y:          return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_Z:          return MA_CHANNEL_NONE;
        case kAudioChannelLabel_MS_Mid:               return MA_CHANNEL_LEFT;
        case kAudioChannelLabel_MS_Side:              return MA_CHANNEL_RIGHT;
        case kAudioChannelLabel_XY_X:                 return MA_CHANNEL_LEFT;
        case kAudioChannelLabel_XY_Y:                 return MA_CHANNEL_RIGHT;
        case kAudioChannelLabel_HeadphonesLeft:       return MA_CHANNEL_LEFT;
        case kAudioChannelLabel_HeadphonesRight:      return MA_CHANNEL_RIGHT;
        case kAudioChannelLabel_ClickTrack:           return MA_CHANNEL_NONE;
        case kAudioChannelLabel_ForeignLanguage:      return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Discrete:             return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Discrete_0:           return MA_CHANNEL_AUX_0;
        case kAudioChannelLabel_Discrete_1:           return MA_CHANNEL_AUX_1;
        case kAudioChannelLabel_Discrete_2:           return MA_CHANNEL_AUX_2;
        case kAudioChannelLabel_Discrete_3:           return MA_CHANNEL_AUX_3;
        case kAudioChannelLabel_Discrete_4:           return MA_CHANNEL_AUX_4;
        case kAudioChannelLabel_Discrete_5:           return MA_CHANNEL_AUX_5;
        case kAudioChannelLabel_Discrete_6:           return MA_CHANNEL_AUX_6;
        case kAudioChannelLabel_Discrete_7:           return MA_CHANNEL_AUX_7;
        case kAudioChannelLabel_Discrete_8:           return MA_CHANNEL_AUX_8;
        case kAudioChannelLabel_Discrete_9:           return MA_CHANNEL_AUX_9;
        case kAudioChannelLabel_Discrete_10:          return MA_CHANNEL_AUX_10;
        case kAudioChannelLabel_Discrete_11:          return MA_CHANNEL_AUX_11;
        case kAudioChannelLabel_Discrete_12:          return MA_CHANNEL_AUX_12;
        case kAudioChannelLabel_Discrete_13:          return MA_CHANNEL_AUX_13;
        case kAudioChannelLabel_Discrete_14:          return MA_CHANNEL_AUX_14;
        case kAudioChannelLabel_Discrete_15:          return MA_CHANNEL_AUX_15;
        case kAudioChannelLabel_Discrete_65535:       return MA_CHANNEL_NONE;
        
    #if 0   /* Introduced in a later version of macOS. */
        case kAudioChannelLabel_HOA_ACN:              return MA_CHANNEL_NONE;
        case kAudioChannelLabel_HOA_ACN_0:            return MA_CHANNEL_AUX_0;
        case kAudioChannelLabel_HOA_ACN_1:            return MA_CHANNEL_AUX_1;
        case kAudioChannelLabel_HOA_ACN_2:            return MA_CHANNEL_AUX_2;
        case kAudioChannelLabel_HOA_ACN_3:            return MA_CHANNEL_AUX_3;
        case kAudioChannelLabel_HOA_ACN_4:            return MA_CHANNEL_AUX_4;
        case kAudioChannelLabel_HOA_ACN_5:            return MA_CHANNEL_AUX_5;
        case kAudioChannelLabel_HOA_ACN_6:            return MA_CHANNEL_AUX_6;
        case kAudioChannelLabel_HOA_ACN_7:            return MA_CHANNEL_AUX_7;
        case kAudioChannelLabel_HOA_ACN_8:            return MA_CHANNEL_AUX_8;
        case kAudioChannelLabel_HOA_ACN_9:            return MA_CHANNEL_AUX_9;
        case kAudioChannelLabel_HOA_ACN_10:           return MA_CHANNEL_AUX_10;
        case kAudioChannelLabel_HOA_ACN_11:           return MA_CHANNEL_AUX_11;
        case kAudioChannelLabel_HOA_ACN_12:           return MA_CHANNEL_AUX_12;
        case kAudioChannelLabel_HOA_ACN_13:           return MA_CHANNEL_AUX_13;
        case kAudioChannelLabel_HOA_ACN_14:           return MA_CHANNEL_AUX_14;
        case kAudioChannelLabel_HOA_ACN_15:           return MA_CHANNEL_AUX_15;
        case kAudioChannelLabel_HOA_ACN_65024:        return MA_CHANNEL_NONE;
    #endif
        
        default:                                      return MA_CHANNEL_NONE;
    }
}

static ma_result ma_get_channel_map_from_AudioChannelLayout(AudioChannelLayout* pChannelLayout, ma_channel* pChannelMap, size_t channelMapCap)
{
    MA_ASSERT(pChannelLayout != NULL);
    
    if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelDescriptions) {
        UInt32 iChannel;
        for (iChannel = 0; iChannel < pChannelLayout->mNumberChannelDescriptions && iChannel < channelMapCap; ++iChannel) {
            pChannelMap[iChannel] = ma_channel_from_AudioChannelLabel(pChannelLayout->mChannelDescriptions[iChannel].mChannelLabel);
        }
    } else
#if 0
    if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelBitmap) {
        /* This is the same kind of system that's used by Windows audio APIs. */
        UInt32 iChannel = 0;
        UInt32 iBit;
        AudioChannelBitmap bitmap = pChannelLayout->mChannelBitmap;
        for (iBit = 0; iBit < 32 && iChannel < channelMapCap; ++iBit) {
            AudioChannelBitmap bit = bitmap & (1 << iBit);
            if (bit != 0) {
                pChannelMap[iChannel++] = ma_channel_from_AudioChannelBit(bit);
            }
        }
    } else
#endif
    {
        /*
        Need to use the tag to determine the channel map. For now I'm just assuming a default channel map, but later on this should
        be updated to determine the mapping based on the tag.
        */
        UInt32 channelCount;

        /* Our channel map retrieval APIs below take 32-bit integers, so we'll want to clamp the channel map capacity. */
        if (channelMapCap > 0xFFFFFFFF) {
            channelMapCap = 0xFFFFFFFF;
        }

        channelCount = ma_min(AudioChannelLayoutTag_GetNumberOfChannels(pChannelLayout->mChannelLayoutTag), (UInt32)channelMapCap);

        switch (pChannelLayout->mChannelLayoutTag)
        {
            case kAudioChannelLayoutTag_Mono:
            case kAudioChannelLayoutTag_Stereo:
            case kAudioChannelLayoutTag_StereoHeadphones:
            case kAudioChannelLayoutTag_MatrixStereo:
            case kAudioChannelLayoutTag_MidSide:
            case kAudioChannelLayoutTag_XY:
            case kAudioChannelLayoutTag_Binaural:
            case kAudioChannelLayoutTag_Ambisonic_B_Format:
            {
                ma_get_standard_channel_map(ma_standard_channel_map_default, channelCount, pChannelMap);
            } break;
            
            case kAudioChannelLayoutTag_Octagonal:
            {
                pChannelMap[7] = MA_CHANNEL_SIDE_RIGHT;
                pChannelMap[6] = MA_CHANNEL_SIDE_LEFT;
            } /* Intentional fallthrough. */
            case kAudioChannelLayoutTag_Hexagonal:
            {
                pChannelMap[5] = MA_CHANNEL_BACK_CENTER;
            } /* Intentional fallthrough. */
            case kAudioChannelLayoutTag_Pentagonal:
            {
                pChannelMap[4] = MA_CHANNEL_FRONT_CENTER;
            } /* Intentional fallghrough. */
            case kAudioChannelLayoutTag_Quadraphonic:
            {
                pChannelMap[3] = MA_CHANNEL_BACK_RIGHT;
                pChannelMap[2] = MA_CHANNEL_BACK_LEFT;
                pChannelMap[1] = MA_CHANNEL_RIGHT;
                pChannelMap[0] = MA_CHANNEL_LEFT;
            } break;
            
            /* TODO: Add support for more tags here. */
        
            default:
            {
                ma_get_standard_channel_map(ma_standard_channel_map_default, channelCount, pChannelMap);
            } break;
        }
    }
    
    return MA_SUCCESS;
}

static ma_result ma_get_device_object_ids__coreaudio(ma_context* pContext, UInt32* pDeviceCount, AudioObjectID** ppDeviceObjectIDs) /* NOTE: Free the returned buffer with ma_free(). */
{
    AudioObjectPropertyAddress propAddressDevices;
    UInt32 deviceObjectsDataSize;
    OSStatus status;
    AudioObjectID* pDeviceObjectIDs;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pDeviceCount != NULL);
    MA_ASSERT(ppDeviceObjectIDs != NULL);

    /* Safety. */
    *pDeviceCount = 0;
    *ppDeviceObjectIDs = NULL;
    
    propAddressDevices.mSelector = kAudioHardwarePropertyDevices;
    propAddressDevices.mScope    = kAudioObjectPropertyScopeGlobal;
    propAddressDevices.mElement  = kAudioObjectPropertyElementMaster;

    status = ((ma_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(kAudioObjectSystemObject, &propAddressDevices, 0, NULL, &deviceObjectsDataSize);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    pDeviceObjectIDs = (AudioObjectID*)ma_malloc(deviceObjectsDataSize, &pContext->allocationCallbacks);
    if (pDeviceObjectIDs == NULL) {
        return MA_OUT_OF_MEMORY;
    }
    
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(kAudioObjectSystemObject, &propAddressDevices, 0, NULL, &deviceObjectsDataSize, pDeviceObjectIDs);
    if (status != noErr) {
        ma_free(pDeviceObjectIDs, &pContext->allocationCallbacks);
        return ma_result_from_OSStatus(status);
    }
    
    *pDeviceCount = deviceObjectsDataSize / sizeof(AudioObjectID);
    *ppDeviceObjectIDs = pDeviceObjectIDs;

    return MA_SUCCESS;
}

static ma_result ma_get_AudioObject_uid_as_CFStringRef(ma_context* pContext, AudioObjectID objectID, CFStringRef* pUID)
{
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;

    MA_ASSERT(pContext != NULL);

    propAddress.mSelector = kAudioDevicePropertyDeviceUID;
    propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;

    dataSize = sizeof(*pUID);
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(objectID, &propAddress, 0, NULL, &dataSize, pUID);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    return MA_SUCCESS;
}

static ma_result ma_get_AudioObject_uid(ma_context* pContext, AudioObjectID objectID, size_t bufferSize, char* bufferOut)
{
    CFStringRef uid;
    ma_result result;

    MA_ASSERT(pContext != NULL);

    result = ma_get_AudioObject_uid_as_CFStringRef(pContext, objectID, &uid);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    if (!((ma_CFStringGetCString_proc)pContext->coreaudio.CFStringGetCString)(uid, bufferOut, bufferSize, kCFStringEncodingUTF8)) {
        return MA_ERROR;
    }
    
    ((ma_CFRelease_proc)pContext->coreaudio.CFRelease)(uid);
    return MA_SUCCESS;
}

static ma_result ma_get_AudioObject_name(ma_context* pContext, AudioObjectID objectID, size_t bufferSize, char* bufferOut)
{
    AudioObjectPropertyAddress propAddress;
    CFStringRef deviceName = NULL;
    UInt32 dataSize;
    OSStatus status;

    MA_ASSERT(pContext != NULL);

    propAddress.mSelector = kAudioDevicePropertyDeviceNameCFString;
    propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;

    dataSize = sizeof(deviceName);
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(objectID, &propAddress, 0, NULL, &dataSize, &deviceName);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    if (!((ma_CFStringGetCString_proc)pContext->coreaudio.CFStringGetCString)(deviceName, bufferOut, bufferSize, kCFStringEncodingUTF8)) {
        return MA_ERROR;
    }
    
    ((ma_CFRelease_proc)pContext->coreaudio.CFRelease)(deviceName);
    return MA_SUCCESS;
}

static ma_bool32 ma_does_AudioObject_support_scope(ma_context* pContext, AudioObjectID deviceObjectID, AudioObjectPropertyScope scope)
{
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;
    AudioBufferList* pBufferList;
    ma_bool32 isSupported;

    MA_ASSERT(pContext != NULL);

    /* To know whether or not a device is an input device we need ot look at the stream configuration. If it has an output channel it's a playback device. */
    propAddress.mSelector = kAudioDevicePropertyStreamConfiguration;
    propAddress.mScope    = scope;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    status = ((ma_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return MA_FALSE;
    }
    
    pBufferList = (AudioBufferList*)ma__malloc_from_callbacks(dataSize, &pContext->allocationCallbacks);
    if (pBufferList == NULL) {
        return MA_FALSE;   /* Out of memory. */
    }
    
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pBufferList);
    if (status != noErr) {
        ma__free_from_callbacks(pBufferList, &pContext->allocationCallbacks);
        return MA_FALSE;
    }

    isSupported = MA_FALSE;
    if (pBufferList->mNumberBuffers > 0) {
        isSupported = MA_TRUE;
    }
    
    ma__free_from_callbacks(pBufferList, &pContext->allocationCallbacks);
    return isSupported;
}

static ma_bool32 ma_does_AudioObject_support_playback(ma_context* pContext, AudioObjectID deviceObjectID)
{
    return ma_does_AudioObject_support_scope(pContext, deviceObjectID, kAudioObjectPropertyScopeOutput);
}

static ma_bool32 ma_does_AudioObject_support_capture(ma_context* pContext, AudioObjectID deviceObjectID)
{
    return ma_does_AudioObject_support_scope(pContext, deviceObjectID, kAudioObjectPropertyScopeInput);
}


static ma_result ma_get_AudioObject_stream_descriptions(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, UInt32* pDescriptionCount, AudioStreamRangedDescription** ppDescriptions) /* NOTE: Free the returned pointer with ma_free(). */
{
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;
    AudioStreamRangedDescription* pDescriptions;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pDescriptionCount != NULL);
    MA_ASSERT(ppDescriptions != NULL);
    
    /*
    TODO: Experiment with kAudioStreamPropertyAvailablePhysicalFormats instead of (or in addition to) kAudioStreamPropertyAvailableVirtualFormats. My
          MacBook Pro uses s24/32 format, however, which miniaudio does not currently support.
    */
    propAddress.mSelector = kAudioStreamPropertyAvailableVirtualFormats; /*kAudioStreamPropertyAvailablePhysicalFormats;*/
    propAddress.mScope    = (deviceType == ma_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    status = ((ma_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    pDescriptions = (AudioStreamRangedDescription*)ma_malloc(dataSize, &pContext->allocationCallbacks);
    if (pDescriptions == NULL) {
        return MA_OUT_OF_MEMORY;
    }
    
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pDescriptions);
    if (status != noErr) {
        ma_free(pDescriptions, &pContext->allocationCallbacks);
        return ma_result_from_OSStatus(status);
    }
    
    *pDescriptionCount = dataSize / sizeof(*pDescriptions);
    *ppDescriptions = pDescriptions;
    return MA_SUCCESS;
}


static ma_result ma_get_AudioObject_channel_layout(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, AudioChannelLayout** ppChannelLayout)   /* NOTE: Free the returned pointer with ma_free(). */
{
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;
    AudioChannelLayout* pChannelLayout;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(ppChannelLayout != NULL);
    
    *ppChannelLayout = NULL;    /* Safety. */
    
    propAddress.mSelector = kAudioDevicePropertyPreferredChannelLayout;
    propAddress.mScope    = (deviceType == ma_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    status = ((ma_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    pChannelLayout = (AudioChannelLayout*)ma_malloc(dataSize, &pContext->allocationCallbacks);
    if (pChannelLayout == NULL) {
        return MA_OUT_OF_MEMORY;
    }
    
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pChannelLayout);
    if (status != noErr) {
        ma_free(pChannelLayout, &pContext->allocationCallbacks);
        return ma_result_from_OSStatus(status);
    }
    
    *ppChannelLayout = pChannelLayout;
    return MA_SUCCESS;
}

static ma_result ma_get_AudioObject_channel_count(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_uint32* pChannelCount)
{
    AudioChannelLayout* pChannelLayout;
    ma_result result;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pChannelCount != NULL);
    
    *pChannelCount = 0; /* Safety. */

    result = ma_get_AudioObject_channel_layout(pContext, deviceObjectID, deviceType, &pChannelLayout);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelDescriptions) {
        *pChannelCount = pChannelLayout->mNumberChannelDescriptions;
    } else if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelBitmap) {
        *pChannelCount = ma_count_set_bits(pChannelLayout->mChannelBitmap);
    } else {
        *pChannelCount = AudioChannelLayoutTag_GetNumberOfChannels(pChannelLayout->mChannelLayoutTag);
    }
    
    ma_free(pChannelLayout, &pContext->allocationCallbacks);
    return MA_SUCCESS;
}

#if 0
static ma_result ma_get_AudioObject_channel_map(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_channel* pChannelMap, size_t channelMapCap)
{
    AudioChannelLayout* pChannelLayout;
    ma_result result;

    MA_ASSERT(pContext != NULL);
    
    result = ma_get_AudioObject_channel_layout(pContext, deviceObjectID, deviceType, &pChannelLayout);
    if (result != MA_SUCCESS) {
        return result;  /* Rather than always failing here, would it be more robust to simply assume a default? */
    }
    
    result = ma_get_channel_map_from_AudioChannelLayout(pChannelLayout, pChannelMap, channelMapCap);
    if (result != MA_SUCCESS) {
        ma_free(pChannelLayout, &pContext->allocationCallbacks);
        return result;
    }
    
    ma_free(pChannelLayout, &pContext->allocationCallbacks);
    return result;
}
#endif

static ma_result ma_get_AudioObject_sample_rates(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, UInt32* pSampleRateRangesCount, AudioValueRange** ppSampleRateRanges)   /* NOTE: Free the returned pointer with ma_free(). */
{
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;
    AudioValueRange* pSampleRateRanges;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pSampleRateRangesCount != NULL);
    MA_ASSERT(ppSampleRateRanges != NULL);
  
    /* Safety. */
    *pSampleRateRangesCount = 0;
    *ppSampleRateRanges = NULL;
    
    propAddress.mSelector = kAudioDevicePropertyAvailableNominalSampleRates;
    propAddress.mScope    = (deviceType == ma_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    status = ((ma_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    pSampleRateRanges = (AudioValueRange*)ma_malloc(dataSize, &pContext->allocationCallbacks);
    if (pSampleRateRanges == NULL) {
        return MA_OUT_OF_MEMORY;
    }
    
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pSampleRateRanges);
    if (status != noErr) {
        ma_free(pSampleRateRanges, &pContext->allocationCallbacks);
        return ma_result_from_OSStatus(status);
    }
    
    *pSampleRateRangesCount = dataSize / sizeof(*pSampleRateRanges);
    *ppSampleRateRanges = pSampleRateRanges;
    return MA_SUCCESS;
}

#if 0
static ma_result ma_get_AudioObject_get_closest_sample_rate(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_uint32 sampleRateIn, ma_uint32* pSampleRateOut)
{
    UInt32 sampleRateRangeCount;
    AudioValueRange* pSampleRateRanges;
    ma_result result;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pSampleRateOut != NULL);
    
    *pSampleRateOut = 0;    /* Safety. */
    
    result = ma_get_AudioObject_sample_rates(pContext, deviceObjectID, deviceType, &sampleRateRangeCount, &pSampleRateRanges);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    if (sampleRateRangeCount == 0) {
        ma_free(pSampleRateRanges, &pContext->allocationCallbacks);
        return MA_ERROR;   /* Should never hit this case should we? */
    }
    
    if (sampleRateIn == 0) {
        /* Search in order of miniaudio's preferred priority. */
        UInt32 iMALSampleRate;
        for (iMALSampleRate = 0; iMALSampleRate < ma_countof(g_maStandardSampleRatePriorities); ++iMALSampleRate) {
            ma_uint32 malSampleRate = g_maStandardSampleRatePriorities[iMALSampleRate];
            UInt32 iCASampleRate;
            for (iCASampleRate = 0; iCASampleRate < sampleRateRangeCount; ++iCASampleRate) {
                AudioValueRange caSampleRate = pSampleRateRanges[iCASampleRate];
                if (caSampleRate.mMinimum <= malSampleRate && caSampleRate.mMaximum >= malSampleRate) {
                    *pSampleRateOut = malSampleRate;
                    ma_free(pSampleRateRanges, &pContext->allocationCallbacks);
                    return MA_SUCCESS;
                }
            }
        }
        
        /*
        If we get here it means none of miniaudio's standard sample rates matched any of the supported sample rates from the device. In this
        case we just fall back to the first one reported by Core Audio.
        */
        MA_ASSERT(sampleRateRangeCount > 0);
        
        *pSampleRateOut = pSampleRateRanges[0].mMinimum;
        ma_free(pSampleRateRanges, &pContext->allocationCallbacks);
        return MA_SUCCESS;
    } else {
        /* Find the closest match to this sample rate. */
        UInt32 currentAbsoluteDifference = INT32_MAX;
        UInt32 iCurrentClosestRange = (UInt32)-1;
        UInt32 iRange;
        for (iRange = 0; iRange < sampleRateRangeCount; ++iRange) {
            if (pSampleRateRanges[iRange].mMinimum <= sampleRateIn && pSampleRateRanges[iRange].mMaximum >= sampleRateIn) {
                *pSampleRateOut = sampleRateIn;
                ma_free(pSampleRateRanges, &pContext->allocationCallbacks);
                return MA_SUCCESS;
            } else {
                UInt32 absoluteDifference;
                if (pSampleRateRanges[iRange].mMinimum > sampleRateIn) {
                    absoluteDifference = pSampleRateRanges[iRange].mMinimum - sampleRateIn;
                } else {
                    absoluteDifference = sampleRateIn - pSampleRateRanges[iRange].mMaximum;
                }
                
                if (currentAbsoluteDifference > absoluteDifference) {
                    currentAbsoluteDifference = absoluteDifference;
                    iCurrentClosestRange = iRange;
                }
            }
        }
        
        MA_ASSERT(iCurrentClosestRange != (UInt32)-1);
        
        *pSampleRateOut = pSampleRateRanges[iCurrentClosestRange].mMinimum;
        ma_free(pSampleRateRanges, &pContext->allocationCallbacks);
        return MA_SUCCESS;
    }
    
    /* Should never get here, but it would mean we weren't able to find any suitable sample rates. */
    /*ma_free(pSampleRateRanges, &pContext->allocationCallbacks);*/
    /*return MA_ERROR;*/
}
#endif

static ma_result ma_get_AudioObject_closest_buffer_size_in_frames(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_uint32 bufferSizeInFramesIn, ma_uint32* pBufferSizeInFramesOut)
{
    AudioObjectPropertyAddress propAddress;
    AudioValueRange bufferSizeRange;
    UInt32 dataSize;
    OSStatus status;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pBufferSizeInFramesOut != NULL);
    
    *pBufferSizeInFramesOut = 0;    /* Safety. */
    
    propAddress.mSelector = kAudioDevicePropertyBufferFrameSizeRange;
    propAddress.mScope    = (deviceType == ma_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;

    dataSize = sizeof(bufferSizeRange);
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, &bufferSizeRange);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    /* This is just a clamp. */
    if (bufferSizeInFramesIn < bufferSizeRange.mMinimum) {
        *pBufferSizeInFramesOut = (ma_uint32)bufferSizeRange.mMinimum;
    } else if (bufferSizeInFramesIn > bufferSizeRange.mMaximum) {
        *pBufferSizeInFramesOut = (ma_uint32)bufferSizeRange.mMaximum;
    } else {
        *pBufferSizeInFramesOut = bufferSizeInFramesIn;
    }

    return MA_SUCCESS;
}

static ma_result ma_set_AudioObject_buffer_size_in_frames(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_uint32* pPeriodSizeInOut)
{
    ma_result result;
    ma_uint32 chosenBufferSizeInFrames;
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;

    MA_ASSERT(pContext != NULL);

    result = ma_get_AudioObject_closest_buffer_size_in_frames(pContext, deviceObjectID, deviceType, *pPeriodSizeInOut, &chosenBufferSizeInFrames);
    if (result != MA_SUCCESS) {
        return result;
    }

    /* Try setting the size of the buffer... If this fails we just use whatever is currently set. */
    propAddress.mSelector = kAudioDevicePropertyBufferFrameSize;
    propAddress.mScope    = (deviceType == ma_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    ((ma_AudioObjectSetPropertyData_proc)pContext->coreaudio.AudioObjectSetPropertyData)(deviceObjectID, &propAddress, 0, NULL, sizeof(chosenBufferSizeInFrames), &chosenBufferSizeInFrames);
    
    /* Get the actual size of the buffer. */
    dataSize = sizeof(*pPeriodSizeInOut);
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, &chosenBufferSizeInFrames);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    *pPeriodSizeInOut = chosenBufferSizeInFrames;
    return MA_SUCCESS;
}


static ma_result ma_find_AudioObjectID(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, AudioObjectID* pDeviceObjectID)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pDeviceObjectID != NULL);

    /* Safety. */
    *pDeviceObjectID = 0;
    
    if (pDeviceID == NULL) {
        /* Default device. */
        AudioObjectPropertyAddress propAddressDefaultDevice;
        UInt32 defaultDeviceObjectIDSize = sizeof(AudioObjectID);
        AudioObjectID defaultDeviceObjectID;
        OSStatus status;

        propAddressDefaultDevice.mScope = kAudioObjectPropertyScopeGlobal;
        propAddressDefaultDevice.mElement = kAudioObjectPropertyElementMaster;
        if (deviceType == ma_device_type_playback) {
            propAddressDefaultDevice.mSelector = kAudioHardwarePropertyDefaultOutputDevice;
        } else {
            propAddressDefaultDevice.mSelector = kAudioHardwarePropertyDefaultInputDevice;
        }
        
        defaultDeviceObjectIDSize = sizeof(AudioObjectID);
        status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(kAudioObjectSystemObject, &propAddressDefaultDevice, 0, NULL, &defaultDeviceObjectIDSize, &defaultDeviceObjectID);
        if (status == noErr) {
            *pDeviceObjectID = defaultDeviceObjectID;
            return MA_SUCCESS;
        }
    } else {
        /* Explicit device. */
        UInt32 deviceCount;
        AudioObjectID* pDeviceObjectIDs;
        ma_result result;
        UInt32 iDevice;

        result = ma_get_device_object_ids__coreaudio(pContext, &deviceCount, &pDeviceObjectIDs);
        if (result != MA_SUCCESS) {
            return result;
        }
        
        for (iDevice = 0; iDevice < deviceCount; ++iDevice) {
            AudioObjectID deviceObjectID = pDeviceObjectIDs[iDevice];
            
            char uid[256];
            if (ma_get_AudioObject_uid(pContext, deviceObjectID, sizeof(uid), uid) != MA_SUCCESS) {
                continue;
            }
            
            if (deviceType == ma_device_type_playback) {
                if (ma_does_AudioObject_support_playback(pContext, deviceObjectID)) {
                    if (strcmp(uid, pDeviceID->coreaudio) == 0) {
                        *pDeviceObjectID = deviceObjectID;
                        ma_free(pDeviceObjectIDs, &pContext->allocationCallbacks);
                        return MA_SUCCESS;
                    }
                }
            } else {
                if (ma_does_AudioObject_support_capture(pContext, deviceObjectID)) {
                    if (strcmp(uid, pDeviceID->coreaudio) == 0) {
                        *pDeviceObjectID = deviceObjectID;
                        ma_free(pDeviceObjectIDs, &pContext->allocationCallbacks);
                        return MA_SUCCESS;
                    }
                }
            }
        }

        ma_free(pDeviceObjectIDs, &pContext->allocationCallbacks);
    }
    
    /* If we get here it means we couldn't find the device. */
    return MA_NO_DEVICE;
}


static ma_result ma_find_best_format__coreaudio(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_format format, ma_uint32 channels, ma_uint32 sampleRate, ma_bool32 usingDefaultFormat, ma_bool32 usingDefaultChannels, ma_bool32 usingDefaultSampleRate, const AudioStreamBasicDescription* pOrigFormat, AudioStreamBasicDescription* pFormat)
{
    UInt32 deviceFormatDescriptionCount;
    AudioStreamRangedDescription* pDeviceFormatDescriptions;
    ma_result result;
    ma_uint32 desiredSampleRate;
    ma_uint32 desiredChannelCount;
    ma_format desiredFormat;
    AudioStreamBasicDescription bestDeviceFormatSoFar;
    ma_bool32 hasSupportedFormat;
    UInt32 iFormat;

    result = ma_get_AudioObject_stream_descriptions(pContext, deviceObjectID, deviceType, &deviceFormatDescriptionCount, &pDeviceFormatDescriptions);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    desiredSampleRate = sampleRate;
    if (usingDefaultSampleRate) {
        desiredSampleRate = pOrigFormat->mSampleRate;
    }
    
    desiredChannelCount = channels;
    if (usingDefaultChannels) {
        desiredChannelCount = pOrigFormat->mChannelsPerFrame;
    }
    
    desiredFormat = format;
    if (usingDefaultFormat) {
        result = ma_format_from_AudioStreamBasicDescription(pOrigFormat, &desiredFormat);
        if (result != MA_SUCCESS || desiredFormat == ma_format_unknown) {
            desiredFormat = g_maFormatPriorities[0];
        }
    }
    
    /*
    If we get here it means we don't have an exact match to what the client is asking for. We'll need to find the closest one. The next
    loop will check for formats that have the same sample rate to what we're asking for. If there is, we prefer that one in all cases.
    */
    MA_ZERO_OBJECT(&bestDeviceFormatSoFar);
    
    hasSupportedFormat = MA_FALSE;
    for (iFormat = 0; iFormat < deviceFormatDescriptionCount; ++iFormat) {
        ma_format format;
        ma_result formatResult = ma_format_from_AudioStreamBasicDescription(&pDeviceFormatDescriptions[iFormat].mFormat, &format);
        if (formatResult == MA_SUCCESS && format != ma_format_unknown) {
            hasSupportedFormat = MA_TRUE;
            bestDeviceFormatSoFar = pDeviceFormatDescriptions[iFormat].mFormat;
            break;
        }
    }
    
    if (!hasSupportedFormat) {
        ma_free(pDeviceFormatDescriptions, &pContext->allocationCallbacks);
        return MA_FORMAT_NOT_SUPPORTED;
    }
    
    
    for (iFormat = 0; iFormat < deviceFormatDescriptionCount; ++iFormat) {
        AudioStreamBasicDescription thisDeviceFormat = pDeviceFormatDescriptions[iFormat].mFormat;
        ma_format thisSampleFormat;
        ma_result formatResult;
        ma_format bestSampleFormatSoFar;

        /* If the format is not supported by miniaudio we need to skip this one entirely. */
        formatResult = ma_format_from_AudioStreamBasicDescription(&pDeviceFormatDescriptions[iFormat].mFormat, &thisSampleFormat);
        if (formatResult != MA_SUCCESS || thisSampleFormat == ma_format_unknown) {
            continue;   /* The format is not supported by miniaudio. Skip. */
        }
        
        ma_format_from_AudioStreamBasicDescription(&bestDeviceFormatSoFar, &bestSampleFormatSoFar);
        
        /* Getting here means the format is supported by miniaudio which makes this format a candidate. */
        if (thisDeviceFormat.mSampleRate != desiredSampleRate) {
            /*
            The sample rate does not match, but this format could still be usable, although it's a very low priority. If the best format
            so far has an equal sample rate we can just ignore this one.
            */
            if (bestDeviceFormatSoFar.mSampleRate == desiredSampleRate) {
                continue;   /* The best sample rate so far has the same sample rate as what we requested which means it's still the best so far. Skip this format. */
            } else {
                /* In this case, neither the best format so far nor this one have the same sample rate. Check the channel count next. */
                if (thisDeviceFormat.mChannelsPerFrame != desiredChannelCount) {
                    /* This format has a different sample rate _and_ a different channel count. */
                    if (bestDeviceFormatSoFar.mChannelsPerFrame == desiredChannelCount) {
                        continue;   /* No change to the best format. */
                    } else {
                        /*
                        Both this format and the best so far have different sample rates and different channel counts. Whichever has the
                        best format is the new best.
                        */
                        if (ma_get_format_priority_index(thisSampleFormat) < ma_get_format_priority_index(bestSampleFormatSoFar)) {
                            bestDeviceFormatSoFar = thisDeviceFormat;
                            continue;
                        } else {
                            continue;   /* No change to the best format. */
                        }
                    }
                } else {
                    /* This format has a different sample rate but the desired channel count. */
                    if (bestDeviceFormatSoFar.mChannelsPerFrame == desiredChannelCount) {
                        /* Both this format and the best so far have the desired channel count. Whichever has the best format is the new best. */
                        if (ma_get_format_priority_index(thisSampleFormat) < ma_get_format_priority_index(bestSampleFormatSoFar)) {
                            bestDeviceFormatSoFar = thisDeviceFormat;
                            continue;
                        } else {
                            continue;   /* No change to the best format for now. */
                        }
                    } else {
                        /* This format has the desired channel count, but the best so far does not. We have a new best. */
                        bestDeviceFormatSoFar = thisDeviceFormat;
                        continue;
                    }
                }
            }
        } else {
            /*
            The sample rates match which makes this format a very high priority contender. If the best format so far has a different
            sample rate it needs to be replaced with this one.
            */
            if (bestDeviceFormatSoFar.mSampleRate != desiredSampleRate) {
                bestDeviceFormatSoFar = thisDeviceFormat;
                continue;
            } else {
                /* In this case both this format and the best format so far have the same sample rate. Check the channel count next. */
                if (thisDeviceFormat.mChannelsPerFrame == desiredChannelCount) {
                    /*
                    In this case this format has the same channel count as what the client is requesting. If the best format so far has
                    a different count, this one becomes the new best.
                    */
                    if (bestDeviceFormatSoFar.mChannelsPerFrame != desiredChannelCount) {
                        bestDeviceFormatSoFar = thisDeviceFormat;
                        continue;
                    } else {
                        /* In this case both this format and the best so far have the ideal sample rate and channel count. Check the format. */
                        if (thisSampleFormat == desiredFormat) {
                            bestDeviceFormatSoFar = thisDeviceFormat;
                            break;  /* Found the exact match. */
                        } else {
                            /* The formats are different. The new best format is the one with the highest priority format according to miniaudio. */
                            if (ma_get_format_priority_index(thisSampleFormat) < ma_get_format_priority_index(bestSampleFormatSoFar)) {
                                bestDeviceFormatSoFar = thisDeviceFormat;
                                continue;
                            } else {
                                continue;   /* No change to the best format for now. */
                            }
                        }
                    }
                } else {
                    /*
                    In this case the channel count is different to what the client has requested. If the best so far has the same channel
                    count as the requested count then it remains the best.
                    */
                    if (bestDeviceFormatSoFar.mChannelsPerFrame == desiredChannelCount) {
                        continue;
                    } else {
                        /*
                        This is the case where both have the same sample rate (good) but different channel counts. Right now both have about
                        the same priority, but we need to compare the format now.
                        */
                        if (thisSampleFormat == bestSampleFormatSoFar) {
                            if (ma_get_format_priority_index(thisSampleFormat) < ma_get_format_priority_index(bestSampleFormatSoFar)) {
                                bestDeviceFormatSoFar = thisDeviceFormat;
                                continue;
                            } else {
                                continue;   /* No change to the best format for now. */
                            }
                        }
                    }
                }
            }
        }
    }
    
    *pFormat = bestDeviceFormatSoFar;

    ma_free(pDeviceFormatDescriptions, &pContext->allocationCallbacks);
    return MA_SUCCESS;
}

static ma_result ma_get_AudioUnit_channel_map(ma_context* pContext, AudioUnit audioUnit, ma_device_type deviceType, ma_channel* pChannelMap, size_t channelMapCap)
{
    AudioUnitScope deviceScope;
    AudioUnitElement deviceBus;
    UInt32 channelLayoutSize;
    OSStatus status;
    AudioChannelLayout* pChannelLayout;
    ma_result result;

    MA_ASSERT(pContext != NULL);
    
    if (deviceType == ma_device_type_playback) {
        deviceScope = kAudioUnitScope_Output;
        deviceBus = MA_COREAUDIO_OUTPUT_BUS;
    } else {
        deviceScope = kAudioUnitScope_Input;
        deviceBus = MA_COREAUDIO_INPUT_BUS;
    }
    
    status = ((ma_AudioUnitGetPropertyInfo_proc)pContext->coreaudio.AudioUnitGetPropertyInfo)(audioUnit, kAudioUnitProperty_AudioChannelLayout, deviceScope, deviceBus, &channelLayoutSize, NULL);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    pChannelLayout = (AudioChannelLayout*)ma__malloc_from_callbacks(channelLayoutSize, &pContext->allocationCallbacks);
    if (pChannelLayout == NULL) {
        return MA_OUT_OF_MEMORY;
    }
    
    status = ((ma_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(audioUnit, kAudioUnitProperty_AudioChannelLayout, deviceScope, deviceBus, pChannelLayout, &channelLayoutSize);
    if (status != noErr) {
        ma__free_from_callbacks(pChannelLayout, &pContext->allocationCallbacks);
        return ma_result_from_OSStatus(status);
    }
    
    result = ma_get_channel_map_from_AudioChannelLayout(pChannelLayout, pChannelMap, channelMapCap);
    if (result != MA_SUCCESS) {
        ma__free_from_callbacks(pChannelLayout, &pContext->allocationCallbacks);
        return result;
    }

    ma__free_from_callbacks(pChannelLayout, &pContext->allocationCallbacks);
    return MA_SUCCESS;
}
#endif /* MA_APPLE_DESKTOP */

static ma_bool32 ma_context_is_device_id_equal__coreaudio(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pID0 != NULL);
    MA_ASSERT(pID1 != NULL);
    (void)pContext;

    return strcmp(pID0->coreaudio, pID1->coreaudio) == 0;
}

#if !defined(MA_APPLE_DESKTOP)
static void ma_AVAudioSessionPortDescription_to_device_info(AVAudioSessionPortDescription* pPortDesc, ma_device_info* pInfo)
{
    MA_ZERO_OBJECT(pInfo);
    ma_strncpy_s(pInfo->name,         sizeof(pInfo->name),         [pPortDesc.portName UTF8String], (size_t)-1);
    ma_strncpy_s(pInfo->id.coreaudio, sizeof(pInfo->id.coreaudio), [pPortDesc.UID      UTF8String], (size_t)-1);
}
#endif

static ma_result ma_context_enumerate_devices__coreaudio(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
#if defined(MA_APPLE_DESKTOP)
    UInt32 deviceCount;
    AudioObjectID* pDeviceObjectIDs;
    ma_result result;
    UInt32 iDevice;

    result = ma_get_device_object_ids__coreaudio(pContext, &deviceCount, &pDeviceObjectIDs);
    if (result != MA_SUCCESS) {
        return result;
    }
  
    for (iDevice = 0; iDevice < deviceCount; ++iDevice) {
        AudioObjectID deviceObjectID = pDeviceObjectIDs[iDevice];
        ma_device_info info;

        MA_ZERO_OBJECT(&info);
        if (ma_get_AudioObject_uid(pContext, deviceObjectID, sizeof(info.id.coreaudio), info.id.coreaudio) != MA_SUCCESS) {
            continue;
        }
        if (ma_get_AudioObject_name(pContext, deviceObjectID, sizeof(info.name), info.name) != MA_SUCCESS) {
            continue;
        }

        if (ma_does_AudioObject_support_playback(pContext, deviceObjectID)) {
            if (!callback(pContext, ma_device_type_playback, &info, pUserData)) {
                break;
            }
        }
        if (ma_does_AudioObject_support_capture(pContext, deviceObjectID)) {
            if (!callback(pContext, ma_device_type_capture, &info, pUserData)) {
                break;
            }
        }
    }
    
    ma_free(pDeviceObjectIDs, &pContext->allocationCallbacks);
#else
    ma_device_info info;
    NSArray *pInputs  = [[[AVAudioSession sharedInstance] currentRoute] inputs];
    NSArray *pOutputs = [[[AVAudioSession sharedInstance] currentRoute] outputs];
    
    for (AVAudioSessionPortDescription* pPortDesc in pOutputs) {
        ma_AVAudioSessionPortDescription_to_device_info(pPortDesc, &info);
        if (!callback(pContext, ma_device_type_playback, &info, pUserData)) {
            return MA_SUCCESS;
        }
    }
    
    for (AVAudioSessionPortDescription* pPortDesc in pInputs) {
        ma_AVAudioSessionPortDescription_to_device_info(pPortDesc, &info);
        if (!callback(pContext, ma_device_type_capture, &info, pUserData)) {
            return MA_SUCCESS;
        }
    }
#endif
    
    return MA_SUCCESS;
}

static ma_result ma_context_get_device_info__coreaudio(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_result result;

    MA_ASSERT(pContext != NULL);

    /* No exclusive mode with the Core Audio backend for now. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }
    
#if defined(MA_APPLE_DESKTOP)
    /* Desktop */
    {
        AudioObjectID deviceObjectID;
        UInt32 streamDescriptionCount;
        AudioStreamRangedDescription* pStreamDescriptions;
        UInt32 iStreamDescription;
        UInt32 sampleRateRangeCount;
        AudioValueRange* pSampleRateRanges;

        result = ma_find_AudioObjectID(pContext, deviceType, pDeviceID, &deviceObjectID);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        result = ma_get_AudioObject_uid(pContext, deviceObjectID, sizeof(pDeviceInfo->id.coreaudio), pDeviceInfo->id.coreaudio);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        result = ma_get_AudioObject_name(pContext, deviceObjectID, sizeof(pDeviceInfo->name), pDeviceInfo->name);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        /* Formats. */
        result = ma_get_AudioObject_stream_descriptions(pContext, deviceObjectID, deviceType, &streamDescriptionCount, &pStreamDescriptions);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        for (iStreamDescription = 0; iStreamDescription < streamDescriptionCount; ++iStreamDescription) {
            ma_format format;
            ma_bool32 formatExists = MA_FALSE;
            ma_uint32 iOutputFormat;

            result = ma_format_from_AudioStreamBasicDescription(&pStreamDescriptions[iStreamDescription].mFormat, &format);
            if (result != MA_SUCCESS) {
                continue;
            }
        
            MA_ASSERT(format != ma_format_unknown);
        
            /* Make sure the format isn't already in the output list. */
            for (iOutputFormat = 0; iOutputFormat < pDeviceInfo->formatCount; ++iOutputFormat) {
                if (pDeviceInfo->formats[iOutputFormat] == format) {
                    formatExists = MA_TRUE;
                    break;
                }
            }
        
            if (!formatExists) {
                pDeviceInfo->formats[pDeviceInfo->formatCount++] = format;
            }
        }
    
        ma_free(pStreamDescriptions, &pContext->allocationCallbacks);
    
    
        /* Channels. */
        result = ma_get_AudioObject_channel_count(pContext, deviceObjectID, deviceType, &pDeviceInfo->minChannels);
        if (result != MA_SUCCESS) {
            return result;
        }
        pDeviceInfo->maxChannels = pDeviceInfo->minChannels;
    
    
        /* Sample rates. */
        result = ma_get_AudioObject_sample_rates(pContext, deviceObjectID, deviceType, &sampleRateRangeCount, &pSampleRateRanges);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        if (sampleRateRangeCount > 0) {
            UInt32 iSampleRate;
            pDeviceInfo->minSampleRate = UINT32_MAX;
            pDeviceInfo->maxSampleRate = 0;
            for (iSampleRate = 0; iSampleRate < sampleRateRangeCount; ++iSampleRate) {
                if (pDeviceInfo->minSampleRate > pSampleRateRanges[iSampleRate].mMinimum) {
                    pDeviceInfo->minSampleRate = pSampleRateRanges[iSampleRate].mMinimum;
                }
                if (pDeviceInfo->maxSampleRate < pSampleRateRanges[iSampleRate].mMaximum) {
                    pDeviceInfo->maxSampleRate = pSampleRateRanges[iSampleRate].mMaximum;
                }
            }
        }
    }
#else
    /* Mobile */
    {
        AudioComponentDescription desc;
        AudioComponent component;
        AudioUnit audioUnit;
        OSStatus status;
        AudioUnitScope formatScope;
        AudioUnitElement formatElement;
        AudioStreamBasicDescription bestFormat;
        UInt32 propSize;
        
        /* We want to ensure we use a consistent device name to device enumeration. */
        if (pDeviceID != NULL) {
            ma_bool32 found = MA_FALSE;
            if (deviceType == ma_device_type_playback) {
                NSArray *pOutputs = [[[AVAudioSession sharedInstance] currentRoute] outputs];
                for (AVAudioSessionPortDescription* pPortDesc in pOutputs) {
                    if (strcmp(pDeviceID->coreaudio, [pPortDesc.UID UTF8String]) == 0) {
                        ma_AVAudioSessionPortDescription_to_device_info(pPortDesc, pDeviceInfo);
                        found = MA_TRUE;
                        break;
                    }
                }
            } else {
                NSArray *pInputs = [[[AVAudioSession sharedInstance] currentRoute] inputs];
                for (AVAudioSessionPortDescription* pPortDesc in pInputs) {
                    if (strcmp(pDeviceID->coreaudio, [pPortDesc.UID UTF8String]) == 0) {
                        ma_AVAudioSessionPortDescription_to_device_info(pPortDesc, pDeviceInfo);
                        found = MA_TRUE;
                        break;
                    }
                }
            }
            
            if (!found) {
                return MA_DOES_NOT_EXIST;
            }
        } else {
            if (deviceType == ma_device_type_playback) {
                ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
            } else {
                ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
            }
        }
        
    
        /*
        Retrieving device information is more annoying on mobile than desktop. For simplicity I'm locking this down to whatever format is
        reported on a temporary I/O unit. The problem, however, is that this doesn't return a value for the sample rate which we need to
        retrieve from the AVAudioSession shared instance.
        */
        desc.componentType = kAudioUnitType_Output;
        desc.componentSubType = kAudioUnitSubType_RemoteIO;
        desc.componentManufacturer = kAudioUnitManufacturer_Apple;
        desc.componentFlags = 0;
        desc.componentFlagsMask = 0;
    
        component = ((ma_AudioComponentFindNext_proc)pContext->coreaudio.AudioComponentFindNext)(NULL, &desc);
        if (component == NULL) {
            return MA_FAILED_TO_INIT_BACKEND;
        }
    
        status = ((ma_AudioComponentInstanceNew_proc)pContext->coreaudio.AudioComponentInstanceNew)(component, &audioUnit);
        if (status != noErr) {
            return ma_result_from_OSStatus(status);
        }
    
        formatScope   = (deviceType == ma_device_type_playback) ? kAudioUnitScope_Input : kAudioUnitScope_Output;
        formatElement = (deviceType == ma_device_type_playback) ? MA_COREAUDIO_OUTPUT_BUS : MA_COREAUDIO_INPUT_BUS;
    
        propSize = sizeof(bestFormat);
        status = ((ma_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, &propSize);
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(audioUnit);
            return ma_result_from_OSStatus(status);
        }
    
        ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(audioUnit);
        audioUnit = NULL;
    
    
        pDeviceInfo->minChannels = bestFormat.mChannelsPerFrame;
        pDeviceInfo->maxChannels = bestFormat.mChannelsPerFrame;
    
        pDeviceInfo->formatCount = 1;
        result = ma_format_from_AudioStreamBasicDescription(&bestFormat, &pDeviceInfo->formats[0]);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        /*
        It looks like Apple are wanting to push the whole AVAudioSession thing. Thus, we need to use that to determine device settings. To do
        this we just get the shared instance and inspect.
        */
        @autoreleasepool {
            AVAudioSession* pAudioSession = [AVAudioSession sharedInstance];
            MA_ASSERT(pAudioSession != NULL);

            pDeviceInfo->minSampleRate = (ma_uint32)pAudioSession.sampleRate;
            pDeviceInfo->maxSampleRate = pDeviceInfo->minSampleRate;
        }
    }
#endif
    
    (void)pDeviceInfo; /* Unused. */
    return MA_SUCCESS;
}


static OSStatus ma_on_output__coreaudio(void* pUserData, AudioUnitRenderActionFlags* pActionFlags, const AudioTimeStamp* pTimeStamp, UInt32 busNumber, UInt32 frameCount, AudioBufferList* pBufferList)
{
    ma_device* pDevice = (ma_device*)pUserData;
    ma_stream_layout layout;

    MA_ASSERT(pDevice != NULL);

#if defined(MA_DEBUG_OUTPUT)
    printf("INFO: Output Callback: busNumber=%d, frameCount=%d, mNumberBuffers=%d\n", busNumber, frameCount, pBufferList->mNumberBuffers);
#endif

    /* We need to check whether or not we are outputting interleaved or non-interleaved samples. The way we do this is slightly different for each type. */
    layout = ma_stream_layout_interleaved;
    if (pBufferList->mBuffers[0].mNumberChannels != pDevice->playback.internalChannels) {
        layout = ma_stream_layout_deinterleaved;
    }
    
    if (layout == ma_stream_layout_interleaved) {
        /* For now we can assume everything is interleaved. */
        UInt32 iBuffer;
        for (iBuffer = 0; iBuffer < pBufferList->mNumberBuffers; ++iBuffer) {
            if (pBufferList->mBuffers[iBuffer].mNumberChannels == pDevice->playback.internalChannels) {
                ma_uint32 frameCountForThisBuffer = pBufferList->mBuffers[iBuffer].mDataByteSize / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                if (frameCountForThisBuffer > 0) {
                    if (pDevice->type == ma_device_type_duplex) {
                        ma_device__handle_duplex_callback_playback(pDevice, frameCountForThisBuffer, pBufferList->mBuffers[iBuffer].mData, &pDevice->coreaudio.duplexRB);
                    } else {
                        ma_device__read_frames_from_client(pDevice, frameCountForThisBuffer, pBufferList->mBuffers[iBuffer].mData);
                    }
                }
                
            #if defined(MA_DEBUG_OUTPUT)
                printf("  frameCount=%d, mNumberChannels=%d, mDataByteSize=%d\n", frameCount, pBufferList->mBuffers[iBuffer].mNumberChannels, pBufferList->mBuffers[iBuffer].mDataByteSize);
            #endif
            } else {
                /*
                This case is where the number of channels in the output buffer do not match our internal channels. It could mean that it's
                not interleaved, in which case we can't handle right now since miniaudio does not yet support non-interleaved streams. We just
                output silence here.
                */
                MA_ZERO_MEMORY(pBufferList->mBuffers[iBuffer].mData, pBufferList->mBuffers[iBuffer].mDataByteSize);

            #if defined(MA_DEBUG_OUTPUT)
                printf("  WARNING: Outputting silence. frameCount=%d, mNumberChannels=%d, mDataByteSize=%d\n", frameCount, pBufferList->mBuffers[iBuffer].mNumberChannels, pBufferList->mBuffers[iBuffer].mDataByteSize);
            #endif
            }
        }
    } else {
        /* This is the deinterleaved case. We need to update each buffer in groups of internalChannels. This assumes each buffer is the same size. */
        MA_ASSERT(pDevice->playback.internalChannels <= MA_MAX_CHANNELS);   /* This should heve been validated at initialization time. */
        
        /*
        For safety we'll check that the internal channels is a multiple of the buffer count. If it's not it means something
        very strange has happened and we're not going to support it.
        */
        if ((pBufferList->mNumberBuffers % pDevice->playback.internalChannels) == 0) {
            ma_uint8 tempBuffer[4096];
            UInt32 iBuffer;
        
            for (iBuffer = 0; iBuffer < pBufferList->mNumberBuffers; iBuffer += pDevice->playback.internalChannels) {
                ma_uint32 frameCountPerBuffer = pBufferList->mBuffers[iBuffer].mDataByteSize / ma_get_bytes_per_sample(pDevice->playback.internalFormat);
                ma_uint32 framesRemaining = frameCountPerBuffer;

                while (framesRemaining > 0) {
                    void* ppDeinterleavedBuffers[MA_MAX_CHANNELS];
                    ma_uint32 iChannel;
                    ma_uint32 framesToRead = sizeof(tempBuffer) / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                    if (framesToRead > framesRemaining) {
                        framesToRead = framesRemaining;
                    }
                    
                    if (pDevice->type == ma_device_type_duplex) {
                        ma_device__handle_duplex_callback_playback(pDevice, framesToRead, tempBuffer, &pDevice->coreaudio.duplexRB);
                    } else {
                        ma_device__read_frames_from_client(pDevice, framesToRead, tempBuffer);
                    }
                    
                    for (iChannel = 0; iChannel < pDevice->playback.internalChannels; ++iChannel) {
                        ppDeinterleavedBuffers[iChannel] = (void*)ma_offset_ptr(pBufferList->mBuffers[iBuffer+iChannel].mData, (frameCountPerBuffer - framesRemaining) * ma_get_bytes_per_sample(pDevice->playback.internalFormat));
                    }
                    
                    ma_deinterleave_pcm_frames(pDevice->playback.internalFormat, pDevice->playback.internalChannels, framesToRead, tempBuffer, ppDeinterleavedBuffers);
                    
                    framesRemaining -= framesToRead;
                }
            }
        }
    }
    
    (void)pActionFlags;
    (void)pTimeStamp;
    (void)busNumber;
    (void)frameCount;

    return noErr;
}

static OSStatus ma_on_input__coreaudio(void* pUserData, AudioUnitRenderActionFlags* pActionFlags, const AudioTimeStamp* pTimeStamp, UInt32 busNumber, UInt32 frameCount, AudioBufferList* pUnusedBufferList)
{
    ma_device* pDevice = (ma_device*)pUserData;
    AudioBufferList* pRenderedBufferList;
    ma_stream_layout layout;
    OSStatus status;

    MA_ASSERT(pDevice != NULL);
    
    pRenderedBufferList = (AudioBufferList*)pDevice->coreaudio.pAudioBufferList;
    MA_ASSERT(pRenderedBufferList);
    
    /* We need to check whether or not we are outputting interleaved or non-interleaved samples. The way we do this is slightly different for each type. */
    layout = ma_stream_layout_interleaved;
    if (pRenderedBufferList->mBuffers[0].mNumberChannels != pDevice->capture.internalChannels) {
        layout = ma_stream_layout_deinterleaved;
    }
    
#if defined(MA_DEBUG_OUTPUT)
    printf("INFO: Input Callback: busNumber=%d, frameCount=%d, mNumberBuffers=%d\n", busNumber, frameCount, pRenderedBufferList->mNumberBuffers);
#endif
    
    status = ((ma_AudioUnitRender_proc)pDevice->pContext->coreaudio.AudioUnitRender)((AudioUnit)pDevice->coreaudio.audioUnitCapture, pActionFlags, pTimeStamp, busNumber, frameCount, pRenderedBufferList);
    if (status != noErr) {
    #if defined(MA_DEBUG_OUTPUT)
        printf("  ERROR: AudioUnitRender() failed with %d\n", status);
    #endif
        return status;
    }
    
    if (layout == ma_stream_layout_interleaved) {
        UInt32 iBuffer;
        for (iBuffer = 0; iBuffer < pRenderedBufferList->mNumberBuffers; ++iBuffer) {
            if (pRenderedBufferList->mBuffers[iBuffer].mNumberChannels == pDevice->capture.internalChannels) {
                if (pDevice->type == ma_device_type_duplex) {
                    ma_device__handle_duplex_callback_capture(pDevice, frameCount, pRenderedBufferList->mBuffers[iBuffer].mData, &pDevice->coreaudio.duplexRB);
                } else {
                    ma_device__send_frames_to_client(pDevice, frameCount, pRenderedBufferList->mBuffers[iBuffer].mData);
                }
            #if defined(MA_DEBUG_OUTPUT)
                printf("  mDataByteSize=%d\n", pRenderedBufferList->mBuffers[iBuffer].mDataByteSize);
            #endif
            } else {
                /*
                This case is where the number of channels in the output buffer do not match our internal channels. It could mean that it's
                not interleaved, in which case we can't handle right now since miniaudio does not yet support non-interleaved streams.
                */
                ma_uint8 silentBuffer[4096];
                ma_uint32 framesRemaining;
                
                MA_ZERO_MEMORY(silentBuffer, sizeof(silentBuffer));
                
                framesRemaining = frameCount;
                while (framesRemaining > 0) {
                    ma_uint32 framesToSend = sizeof(silentBuffer) / ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
                    if (framesToSend > framesRemaining) {
                        framesToSend = framesRemaining;
                    }
                    
                    if (pDevice->type == ma_device_type_duplex) {
                        ma_device__handle_duplex_callback_capture(pDevice, framesToSend, silentBuffer, &pDevice->coreaudio.duplexRB);
                    } else {
                        ma_device__send_frames_to_client(pDevice, framesToSend, silentBuffer);
                    }
                    
                    framesRemaining -= framesToSend;
                }
                
            #if defined(MA_DEBUG_OUTPUT)
                printf("  WARNING: Outputting silence. frameCount=%d, mNumberChannels=%d, mDataByteSize=%d\n", frameCount, pRenderedBufferList->mBuffers[iBuffer].mNumberChannels, pRenderedBufferList->mBuffers[iBuffer].mDataByteSize);
            #endif
            }
        }
    } else {
        /* This is the deinterleaved case. We need to interleave the audio data before sending it to the client. This assumes each buffer is the same size. */
        MA_ASSERT(pDevice->capture.internalChannels <= MA_MAX_CHANNELS);    /* This should have been validated at initialization time. */
        
        /*
        For safety we'll check that the internal channels is a multiple of the buffer count. If it's not it means something
        very strange has happened and we're not going to support it.
        */
        if ((pRenderedBufferList->mNumberBuffers % pDevice->capture.internalChannels) == 0) {
            ma_uint8 tempBuffer[4096];
            UInt32 iBuffer;
            for (iBuffer = 0; iBuffer < pRenderedBufferList->mNumberBuffers; iBuffer += pDevice->capture.internalChannels) {
                ma_uint32 framesRemaining = frameCount;
                while (framesRemaining > 0) {
                    void* ppDeinterleavedBuffers[MA_MAX_CHANNELS];
                    ma_uint32 iChannel;
                    ma_uint32 framesToSend = sizeof(tempBuffer) / ma_get_bytes_per_sample(pDevice->capture.internalFormat);
                    if (framesToSend > framesRemaining) {
                        framesToSend = framesRemaining;
                    }
                    
                    for (iChannel = 0; iChannel < pDevice->capture.internalChannels; ++iChannel) {
                        ppDeinterleavedBuffers[iChannel] = (void*)ma_offset_ptr(pRenderedBufferList->mBuffers[iBuffer+iChannel].mData, (frameCount - framesRemaining) * ma_get_bytes_per_sample(pDevice->capture.internalFormat));
                    }
                    
                    ma_interleave_pcm_frames(pDevice->capture.internalFormat, pDevice->capture.internalChannels, framesToSend, (const void**)ppDeinterleavedBuffers, tempBuffer);

                    if (pDevice->type == ma_device_type_duplex) {
                        ma_device__handle_duplex_callback_capture(pDevice, framesToSend, tempBuffer, &pDevice->coreaudio.duplexRB);
                    } else {
                        ma_device__send_frames_to_client(pDevice, framesToSend, tempBuffer);
                    }

                    framesRemaining -= framesToSend;
                }
            }
        }
    }

    (void)pActionFlags;
    (void)pTimeStamp;
    (void)busNumber;
    (void)frameCount;
    (void)pUnusedBufferList;

    return noErr;
}

static void on_start_stop__coreaudio(void* pUserData, AudioUnit audioUnit, AudioUnitPropertyID propertyID, AudioUnitScope scope, AudioUnitElement element)
{
    ma_device* pDevice = (ma_device*)pUserData;
    MA_ASSERT(pDevice != NULL);
    
    /*
    There's been a report of a deadlock here when triggered by ma_device_uninit(). It looks like
    AudioUnitGetProprty (called below) and AudioComponentInstanceDispose (called in ma_device_uninit)
    can try waiting on the same lock. I'm going to try working around this by not calling any Core
    Audio APIs in the callback when the device has been stopped or uninitialized.
    */
    if (ma_device__get_state(pDevice) == MA_STATE_UNINITIALIZED || ma_device__get_state(pDevice) == MA_STATE_STOPPING || ma_device__get_state(pDevice) == MA_STATE_STOPPED) {
        ma_stop_proc onStop = pDevice->onStop;
        if (onStop) {
            onStop(pDevice);
        }
        
        ma_event_signal(&pDevice->coreaudio.stopEvent);
    } else {
        UInt32 isRunning;
        UInt32 isRunningSize = sizeof(isRunning);
        OSStatus status = ((ma_AudioUnitGetProperty_proc)pDevice->pContext->coreaudio.AudioUnitGetProperty)(audioUnit, kAudioOutputUnitProperty_IsRunning, scope, element, &isRunning, &isRunningSize);
        if (status != noErr) {
            return; /* Don't really know what to do in this case... just ignore it, I suppose... */
        }
        
        if (!isRunning) {
            ma_stop_proc onStop;

            /*
            The stop event is a bit annoying in Core Audio because it will be called when we automatically switch the default device. Some scenarios to consider:
            
            1) When the device is unplugged, this will be called _before_ the default device change notification.
            2) When the device is changed via the default device change notification, this will be called _after_ the switch.
            
            For case #1, we just check if there's a new default device available. If so, we just ignore the stop event. For case #2 we check a flag.
            */
            if (((audioUnit == pDevice->coreaudio.audioUnitPlayback) && pDevice->coreaudio.isDefaultPlaybackDevice) ||
                ((audioUnit == pDevice->coreaudio.audioUnitCapture)  && pDevice->coreaudio.isDefaultCaptureDevice)) {
                /*
                It looks like the device is switching through an external event, such as the user unplugging the device or changing the default device
                via the operating system's sound settings. If we're re-initializing the device, we just terminate because we want the stopping of the
                device to be seamless to the client (we don't want them receiving the onStop event and thinking that the device has stopped when it
                hasn't!).
                */
                if (((audioUnit == pDevice->coreaudio.audioUnitPlayback) && pDevice->coreaudio.isSwitchingPlaybackDevice) ||
                    ((audioUnit == pDevice->coreaudio.audioUnitCapture)  && pDevice->coreaudio.isSwitchingCaptureDevice)) {
                    return;
                }
                
                /*
                Getting here means the device is not reinitializing which means it may have been unplugged. From what I can see, it looks like Core Audio
                will try switching to the new default device seamlessly. We need to somehow find a way to determine whether or not Core Audio will most
                likely be successful in switching to the new device.
                
                TODO: Try to predict if Core Audio will switch devices. If not, the onStop callback needs to be posted.
                */
                return;
            }
            
            /* Getting here means we need to stop the device. */
            onStop = pDevice->onStop;
            if (onStop) {
                onStop(pDevice);
            }
        }
    }

    (void)propertyID; /* Unused. */
}

#if defined(MA_APPLE_DESKTOP)
static ma_spinlock g_DeviceTrackingInitLock_CoreAudio = 0;  /* A spinlock for mutal exclusion of the init/uninit of the global tracking data. Initialization to 0 is what we need. */
static ma_uint32   g_DeviceTrackingInitCounter_CoreAudio = 0;
static ma_mutex    g_DeviceTrackingMutex_CoreAudio;
static ma_device** g_ppTrackedDevices_CoreAudio = NULL;
static ma_uint32   g_TrackedDeviceCap_CoreAudio = 0;
static ma_uint32   g_TrackedDeviceCount_CoreAudio = 0;

static OSStatus ma_default_device_changed__coreaudio(AudioObjectID objectID, UInt32 addressCount, const AudioObjectPropertyAddress* pAddresses, void* pUserData)
{
    ma_device_type deviceType;
    
    /* Not sure if I really need to check this, but it makes me feel better. */
    if (addressCount == 0) {
        return noErr;
    }
    
    if (pAddresses[0].mSelector == kAudioHardwarePropertyDefaultOutputDevice) {
        deviceType = ma_device_type_playback;
    } else if (pAddresses[0].mSelector == kAudioHardwarePropertyDefaultInputDevice) {
        deviceType = ma_device_type_capture;
    } else {
        return noErr;   /* Should never hit this. */
    }
    
    ma_mutex_lock(&g_DeviceTrackingMutex_CoreAudio);
    {
        ma_uint32 iDevice;
        for (iDevice = 0; iDevice < g_TrackedDeviceCount_CoreAudio; iDevice += 1) {
            ma_result reinitResult;
            ma_device* pDevice;
            
            pDevice = g_ppTrackedDevices_CoreAudio[iDevice];
            if (pDevice->type == deviceType || pDevice->type == ma_device_type_duplex) {
                if (deviceType == ma_device_type_playback) {
                    pDevice->coreaudio.isSwitchingPlaybackDevice = MA_TRUE;
                    reinitResult = ma_device_reinit_internal__coreaudio(pDevice, deviceType, MA_TRUE);
                    pDevice->coreaudio.isSwitchingPlaybackDevice = MA_FALSE;
                } else {
                    pDevice->coreaudio.isSwitchingCaptureDevice = MA_TRUE;
                    reinitResult = ma_device_reinit_internal__coreaudio(pDevice, deviceType, MA_TRUE);
                    pDevice->coreaudio.isSwitchingCaptureDevice = MA_FALSE;
                }
                
                if (reinitResult == MA_SUCCESS) {
                    ma_device__post_init_setup(pDevice, deviceType);
            
                    /* Restart the device if required. If this fails we need to stop the device entirely. */
                    if (ma_device__get_state(pDevice) == MA_STATE_STARTED) {
                        OSStatus status;
                        if (deviceType == ma_device_type_playback) {
                            status = ((ma_AudioOutputUnitStart_proc)pDevice->pContext->coreaudio.AudioOutputUnitStart)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
                            if (status != noErr) {
                                if (pDevice->type == ma_device_type_duplex) {
                                    ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
                                }
                                ma_device__set_state(pDevice, MA_STATE_STOPPED);
                            }
                        } else if (deviceType == ma_device_type_capture) {
                            status = ((ma_AudioOutputUnitStart_proc)pDevice->pContext->coreaudio.AudioOutputUnitStart)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
                            if (status != noErr) {
                                if (pDevice->type == ma_device_type_duplex) {
                                    ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
                                }
                                ma_device__set_state(pDevice, MA_STATE_STOPPED);
                            }
                        }
                    }
                }
            }
        }
    }
    ma_mutex_unlock(&g_DeviceTrackingMutex_CoreAudio);
    
    /* Unused parameters. */
    (void)objectID;
    (void)pUserData;

    return noErr;
}

static ma_result ma_context__init_device_tracking__coreaudio(ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);
    
    ma_spinlock_lock(&g_DeviceTrackingInitLock_CoreAudio);
    {
        /* Don't do anything if we've already initializd device tracking. */
        if (g_DeviceTrackingInitCounter_CoreAudio == 0) {
            AudioObjectPropertyAddress propAddress;
            propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
            propAddress.mElement  = kAudioObjectPropertyElementMaster;
            
            ma_mutex_init(&g_DeviceTrackingMutex_CoreAudio);
            
            propAddress.mSelector = kAudioHardwarePropertyDefaultInputDevice;
            ((ma_AudioObjectAddPropertyListener_proc)pContext->coreaudio.AudioObjectAddPropertyListener)(kAudioObjectSystemObject, &propAddress, &ma_default_device_changed__coreaudio, NULL);
            
            propAddress.mSelector = kAudioHardwarePropertyDefaultOutputDevice;
            ((ma_AudioObjectAddPropertyListener_proc)pContext->coreaudio.AudioObjectAddPropertyListener)(kAudioObjectSystemObject, &propAddress, &ma_default_device_changed__coreaudio, NULL);
            
            g_DeviceTrackingInitCounter_CoreAudio += 1;
        }
    }
    ma_spinlock_unlock(&g_DeviceTrackingInitLock_CoreAudio);

    return MA_SUCCESS;
}

static ma_result ma_context__uninit_device_tracking__coreaudio(ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);
    
    ma_spinlock_lock(&g_DeviceTrackingInitLock_CoreAudio);
    {
        g_DeviceTrackingInitCounter_CoreAudio -= 1;
        
        if (g_DeviceTrackingInitCounter_CoreAudio == 0) {
            AudioObjectPropertyAddress propAddress;
            propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
            propAddress.mElement  = kAudioObjectPropertyElementMaster;
            
            propAddress.mSelector = kAudioHardwarePropertyDefaultInputDevice;
            ((ma_AudioObjectRemovePropertyListener_proc)pContext->coreaudio.AudioObjectRemovePropertyListener)(kAudioObjectSystemObject, &propAddress, &ma_default_device_changed__coreaudio, NULL);
            
            propAddress.mSelector = kAudioHardwarePropertyDefaultOutputDevice;
            ((ma_AudioObjectRemovePropertyListener_proc)pContext->coreaudio.AudioObjectRemovePropertyListener)(kAudioObjectSystemObject, &propAddress, &ma_default_device_changed__coreaudio, NULL);

            /* At this point there should be no tracked devices. If not there's an error somewhere. */
            if (g_ppTrackedDevices_CoreAudio != NULL) {
                ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_WARNING, "You have uninitialized all contexts while an associated device is still active.", MA_INVALID_OPERATION);
            }
            
            ma_mutex_uninit(&g_DeviceTrackingMutex_CoreAudio);
        }
    }
    ma_spinlock_unlock(&g_DeviceTrackingInitLock_CoreAudio);
    
    return MA_SUCCESS;
}

static ma_result ma_device__track__coreaudio(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);
    
    ma_mutex_lock(&g_DeviceTrackingMutex_CoreAudio);
    {
        /* Allocate memory if required. */
        if (g_TrackedDeviceCap_CoreAudio <= g_TrackedDeviceCount_CoreAudio) {
            ma_uint32 oldCap;
            ma_uint32 newCap;
            ma_device** ppNewDevices;
            
            oldCap = g_TrackedDeviceCap_CoreAudio;
            newCap = g_TrackedDeviceCap_CoreAudio * 2;
            if (newCap == 0) {
                newCap = 1;
            }
            
            ppNewDevices = (ma_device**)ma__realloc_from_callbacks(g_ppTrackedDevices_CoreAudio, sizeof(*g_ppTrackedDevices_CoreAudio)*newCap, sizeof(*g_ppTrackedDevices_CoreAudio)*oldCap, &pDevice->pContext->allocationCallbacks);
            if (ppNewDevices == NULL) {
                ma_mutex_unlock(&g_DeviceTrackingMutex_CoreAudio);
                return MA_OUT_OF_MEMORY;
            }
            
            g_ppTrackedDevices_CoreAudio = ppNewDevices;
            g_TrackedDeviceCap_CoreAudio = newCap;
        }
        
        g_ppTrackedDevices_CoreAudio[g_TrackedDeviceCount_CoreAudio] = pDevice;
        g_TrackedDeviceCount_CoreAudio += 1;
    }
    ma_mutex_unlock(&g_DeviceTrackingMutex_CoreAudio);
    
    return MA_SUCCESS;
}

static ma_result ma_device__untrack__coreaudio(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);
    
    ma_mutex_lock(&g_DeviceTrackingMutex_CoreAudio);
    {
        ma_uint32 iDevice;
        for (iDevice = 0; iDevice < g_TrackedDeviceCount_CoreAudio; iDevice += 1) {
            if (g_ppTrackedDevices_CoreAudio[iDevice] == pDevice) {
                /* We've found the device. We now need to remove it from the list. */
                ma_uint32 jDevice;
                for (jDevice = iDevice; jDevice < g_TrackedDeviceCount_CoreAudio-1; jDevice += 1) {
                    g_ppTrackedDevices_CoreAudio[jDevice] = g_ppTrackedDevices_CoreAudio[jDevice+1];
                }
                
                g_TrackedDeviceCount_CoreAudio -= 1;
                
                /* If there's nothing else in the list we need to free memory. */
                if (g_TrackedDeviceCount_CoreAudio == 0) {
                    ma__free_from_callbacks(g_ppTrackedDevices_CoreAudio, &pDevice->pContext->allocationCallbacks);
                    g_ppTrackedDevices_CoreAudio = NULL;
                    g_TrackedDeviceCap_CoreAudio = 0;
                }
            
                break;
            }
        }
    }
    ma_mutex_unlock(&g_DeviceTrackingMutex_CoreAudio);

    return MA_SUCCESS;
}
#endif

#if defined(MA_APPLE_MOBILE)
@interface ma_router_change_handler:NSObject {
    ma_device* m_pDevice;
}
@end

@implementation ma_router_change_handler
-(id)init:(ma_device*)pDevice
{
    self = [super init];
    m_pDevice = pDevice;

    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(handle_route_change:) name:AVAudioSessionRouteChangeNotification object:[AVAudioSession sharedInstance]];

    return self;
}

-(void)dealloc
{
    [self remove_handler];
}

-(void)remove_handler
{
    [[NSNotificationCenter defaultCenter] removeObserver:self name:@"AVAudioSessionRouteChangeNotification" object:nil];
}

-(void)handle_route_change:(NSNotification*)pNotification
{
    AVAudioSession* pSession = [AVAudioSession sharedInstance];

    NSInteger reason = [[[pNotification userInfo] objectForKey:AVAudioSessionRouteChangeReasonKey] integerValue];
    switch (reason)
    {
        case AVAudioSessionRouteChangeReasonOldDeviceUnavailable:
        {
        #if defined(MA_DEBUG_OUTPUT)
            printf("[Core Audio] Route Changed: AVAudioSessionRouteChangeReasonOldDeviceUnavailable\n");
        #endif
        } break;

        case AVAudioSessionRouteChangeReasonNewDeviceAvailable:
        {
        #if defined(MA_DEBUG_OUTPUT)
            printf("[Core Audio] Route Changed: AVAudioSessionRouteChangeReasonNewDeviceAvailable\n");
        #endif
        } break;

        case AVAudioSessionRouteChangeReasonNoSuitableRouteForCategory:
        {
        #if defined(MA_DEBUG_OUTPUT)
            printf("[Core Audio] Route Changed: AVAudioSessionRouteChangeReasonNoSuitableRouteForCategory\n");
        #endif
        } break;

        case AVAudioSessionRouteChangeReasonWakeFromSleep:
        {
        #if defined(MA_DEBUG_OUTPUT)
            printf("[Core Audio] Route Changed: AVAudioSessionRouteChangeReasonWakeFromSleep\n");
        #endif
        } break;

        case AVAudioSessionRouteChangeReasonOverride:
        {
        #if defined(MA_DEBUG_OUTPUT)
            printf("[Core Audio] Route Changed: AVAudioSessionRouteChangeReasonOverride\n");
        #endif
        } break;

        case AVAudioSessionRouteChangeReasonCategoryChange:
        {
        #if defined(MA_DEBUG_OUTPUT)
            printf("[Core Audio] Route Changed: AVAudioSessionRouteChangeReasonCategoryChange\n");
        #endif
        } break;

        case AVAudioSessionRouteChangeReasonUnknown:
        default:
        {
        #if defined(MA_DEBUG_OUTPUT)
            printf("[Core Audio] Route Changed: AVAudioSessionRouteChangeReasonUnknown\n");
        #endif
        } break;
    }

    m_pDevice->sampleRate = (ma_uint32)pSession.sampleRate;

    if (m_pDevice->type == ma_device_type_capture || m_pDevice->type == ma_device_type_duplex) {
        m_pDevice->capture.channels = (ma_uint32)pSession.inputNumberOfChannels;
        ma_device__post_init_setup(m_pDevice, ma_device_type_capture);
    }
    if (m_pDevice->type == ma_device_type_playback || m_pDevice->type == ma_device_type_duplex) {
        m_pDevice->playback.channels = (ma_uint32)pSession.outputNumberOfChannels;
        ma_device__post_init_setup(m_pDevice, ma_device_type_playback);
    }
}
@end
#endif

static void ma_device_uninit__coreaudio(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);
    MA_ASSERT(ma_device__get_state(pDevice) == MA_STATE_UNINITIALIZED);
    
#if defined(MA_APPLE_DESKTOP)
    /*
    Make sure we're no longer tracking the device. It doesn't matter if we call this for a non-default device because it'll
    just gracefully ignore it.
    */
    ma_device__untrack__coreaudio(pDevice);
#endif
#if defined(MA_APPLE_MOBILE)
    if (pDevice->coreaudio.pRouteChangeHandler != NULL) {
        ma_router_change_handler* pRouteChangeHandler = (__bridge_transfer ma_router_change_handler*)pDevice->coreaudio.pRouteChangeHandler;
        [pRouteChangeHandler remove_handler];
    }
#endif
    
    if (pDevice->coreaudio.audioUnitCapture != NULL) {
        ((ma_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
    }
    if (pDevice->coreaudio.audioUnitPlayback != NULL) {
        ((ma_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
    }
    
    if (pDevice->coreaudio.pAudioBufferList) {
        ma__free_from_callbacks(pDevice->coreaudio.pAudioBufferList, &pDevice->pContext->allocationCallbacks);
    }

    if (pDevice->type == ma_device_type_duplex) {
        ma_pcm_rb_uninit(&pDevice->coreaudio.duplexRB);
    }
}

typedef struct
{
    /* Input. */
    ma_format formatIn;
    ma_uint32 channelsIn;
    ma_uint32 sampleRateIn;
    ma_channel channelMapIn[MA_MAX_CHANNELS];
    ma_uint32 periodSizeInFramesIn;
    ma_uint32 periodSizeInMillisecondsIn;
    ma_uint32 periodsIn;
    ma_bool32 usingDefaultFormat;
    ma_bool32 usingDefaultChannels;
    ma_bool32 usingDefaultSampleRate;
    ma_bool32 usingDefaultChannelMap;
    ma_share_mode shareMode;
    ma_bool32 registerStopEvent;

    /* Output. */
#if defined(MA_APPLE_DESKTOP)
    AudioObjectID deviceObjectID;
#endif
    AudioComponent component;
    AudioUnit audioUnit;
    AudioBufferList* pAudioBufferList;  /* Only used for input devices. */
    ma_format formatOut;
    ma_uint32 channelsOut;
    ma_uint32 sampleRateOut;
    ma_channel channelMapOut[MA_MAX_CHANNELS];
    ma_uint32 periodSizeInFramesOut;
    ma_uint32 periodsOut;
    char deviceName[256];
} ma_device_init_internal_data__coreaudio;

static ma_result ma_device_init_internal__coreaudio(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_device_init_internal_data__coreaudio* pData, void* pDevice_DoNotReference)   /* <-- pDevice is typed as void* intentionally so as to avoid accidentally referencing it. */
{
    ma_result result;
    OSStatus status;
    UInt32 enableIOFlag;
    AudioStreamBasicDescription bestFormat;
    ma_uint32 actualPeriodSizeInFrames;
    AURenderCallbackStruct callbackInfo;
#if defined(MA_APPLE_DESKTOP)
    AudioObjectID deviceObjectID;
#endif

    /* This API should only be used for a single device type: playback or capture. No full-duplex mode. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(deviceType == ma_device_type_playback || deviceType == ma_device_type_capture);

#if defined(MA_APPLE_DESKTOP)
    pData->deviceObjectID = 0;
#endif
    pData->component = NULL;
    pData->audioUnit = NULL;
    pData->pAudioBufferList = NULL;
    
#if defined(MA_APPLE_DESKTOP)
    result = ma_find_AudioObjectID(pContext, deviceType, pDeviceID, &deviceObjectID);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    pData->deviceObjectID = deviceObjectID;
#endif
    
    /* Core audio doesn't really use the notion of a period so we can leave this unmodified, but not too over the top. */
    pData->periodsOut = pData->periodsIn;
    if (pData->periodsOut == 0) {
        pData->periodsOut = MA_DEFAULT_PERIODS;
    }
    if (pData->periodsOut > 16) {
        pData->periodsOut = 16;
    }
    
    
    /* Audio unit. */
    status = ((ma_AudioComponentInstanceNew_proc)pContext->coreaudio.AudioComponentInstanceNew)((AudioComponent)pContext->coreaudio.component, (AudioUnit*)&pData->audioUnit);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    
    /* The input/output buses need to be explicitly enabled and disabled. We set the flag based on the output unit first, then we just swap it for input. */
    enableIOFlag = 1;
    if (deviceType == ma_device_type_capture) {
        enableIOFlag = 0;
    }
    
    status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_EnableIO, kAudioUnitScope_Output, MA_COREAUDIO_OUTPUT_BUS, &enableIOFlag, sizeof(enableIOFlag));
    if (status != noErr) {
        ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return ma_result_from_OSStatus(status);
    }
    
    enableIOFlag = (enableIOFlag == 0) ? 1 : 0;
    status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_EnableIO, kAudioUnitScope_Input, MA_COREAUDIO_INPUT_BUS, &enableIOFlag, sizeof(enableIOFlag));
    if (status != noErr) {
        ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return ma_result_from_OSStatus(status);
    }
    
    
    /* Set the device to use with this audio unit. This is only used on desktop since we are using defaults on mobile. */
#if defined(MA_APPLE_DESKTOP)
    status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_CurrentDevice, kAudioUnitScope_Global, (deviceType == ma_device_type_playback) ? MA_COREAUDIO_OUTPUT_BUS : MA_COREAUDIO_INPUT_BUS, &deviceObjectID, sizeof(AudioDeviceID));
    if (status != noErr) {
        ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return ma_result_from_OSStatus(result);
    }
#else
    /*
    For some reason it looks like Apple is only allowing selection of the input device. There does not appear to be any way to change
    the default output route. I have no idea why this is like this, but for now we'll only be able to configure capture devices.
    */
    if (pDeviceID != NULL) {
        if (deviceType == ma_device_type_capture) {
            ma_bool32 found = MA_FALSE;
            NSArray *pInputs = [[[AVAudioSession sharedInstance] currentRoute] inputs];
            for (AVAudioSessionPortDescription* pPortDesc in pInputs) {
                if (strcmp(pDeviceID->coreaudio, [pPortDesc.UID UTF8String]) == 0) {
                    [[AVAudioSession sharedInstance] setPreferredInput:pPortDesc error:nil];
                    found = MA_TRUE;
                    break;
                }
            }
            
            if (found == MA_FALSE) {
                return MA_DOES_NOT_EXIST;
            }
        }
    }
#endif
    
    /*
    Format. This is the hardest part of initialization because there's a few variables to take into account.
      1) The format must be supported by the device.
      2) The format must be supported miniaudio.
      3) There's a priority that miniaudio prefers.
    
    Ideally we would like to use a format that's as close to the hardware as possible so we can get as close to a passthrough as possible. The
    most important property is the sample rate. miniaudio can do format conversion for any sample rate and channel count, but cannot do the same
    for the sample data format. If the sample data format is not supported by miniaudio it must be ignored completely.
    
    On mobile platforms this is a bit different. We just force the use of whatever the audio unit's current format is set to.
    */
    {
        AudioUnitScope   formatScope   = (deviceType == ma_device_type_playback) ? kAudioUnitScope_Input : kAudioUnitScope_Output;
        AudioUnitElement formatElement = (deviceType == ma_device_type_playback) ? MA_COREAUDIO_OUTPUT_BUS : MA_COREAUDIO_INPUT_BUS;

    #if defined(MA_APPLE_DESKTOP)
        AudioStreamBasicDescription origFormat;
        UInt32 origFormatSize;
        
        origFormatSize = sizeof(origFormat);
        status = ((ma_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &origFormat, &origFormatSize);
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return result;
        }

        result = ma_find_best_format__coreaudio(pContext, deviceObjectID, deviceType, pData->formatIn, pData->channelsIn, pData->sampleRateIn, pData->usingDefaultFormat, pData->usingDefaultChannels, pData->usingDefaultSampleRate, &origFormat, &bestFormat);
        if (result != MA_SUCCESS) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return result;
        }
        
        /*
        Update 2020-10-10:
        
        I cannot remember where I read this in the documentation and I cannot find it again. For now I'm going to remove this
        and see what the feedback from the community is like. If this results in issues we can add it back in again. The idea
        is that the closest sample rate natively supported by the backend to the requested sample rate should be used if possible.
        */
    #if 0
        /* From what I can see, Apple's documentation implies that we should keep the sample rate consistent. */
        bestFormat.mSampleRate = origFormat.mSampleRate;
    #endif
        
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, sizeof(bestFormat));
        if (status != noErr) {
            /* We failed to set the format, so fall back to the current format of the audio unit. */
            bestFormat = origFormat;
        }
    #else
        UInt32 propSize = sizeof(bestFormat);
        status = ((ma_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, &propSize);
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
        
        /*
        Sample rate is a little different here because for some reason kAudioUnitProperty_StreamFormat returns 0... Oh well. We need to instead try
        setting the sample rate to what the user has requested and then just see the results of it. Need to use some Objective-C here for this since
        it depends on Apple's AVAudioSession API. To do this we just get the shared AVAudioSession instance and then set it. Note that from what I
        can tell, it looks like the sample rate is shared between playback and capture for everything.
        */
        @autoreleasepool {
            AVAudioSession* pAudioSession = [AVAudioSession sharedInstance];
            MA_ASSERT(pAudioSession != NULL);
            
            [pAudioSession setPreferredSampleRate:(double)pData->sampleRateIn error:nil];
            bestFormat.mSampleRate = pAudioSession.sampleRate;

            /*
            I've had a report that the channel count returned by AudioUnitGetProperty above is inconsistent with
            AVAudioSession outputNumberOfChannels. I'm going to try using the AVAudioSession values instead.
            */
            if (deviceType == ma_device_type_playback) {
                bestFormat.mChannelsPerFrame = (UInt32)pAudioSession.outputNumberOfChannels;
            }
            if (deviceType == ma_device_type_capture) {
                bestFormat.mChannelsPerFrame = (UInt32)pAudioSession.inputNumberOfChannels;
            }
        }
        
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, sizeof(bestFormat));
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
    #endif
        
        result = ma_format_from_AudioStreamBasicDescription(&bestFormat, &pData->formatOut);
        if (result != MA_SUCCESS) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return result;
        }
        
        if (pData->formatOut == ma_format_unknown) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return MA_FORMAT_NOT_SUPPORTED;
        }
        
        pData->channelsOut = bestFormat.mChannelsPerFrame;
        pData->sampleRateOut = bestFormat.mSampleRate;
    }

    /* Clamp the channel count for safety. */
    if (pData->channelsOut > MA_MAX_CHANNELS) {
        pData->channelsOut = MA_MAX_CHANNELS;
    }
    
    /*
    Internal channel map. This is weird in my testing. If I use the AudioObject to get the
    channel map, the channel descriptions are set to "Unknown" for some reason. To work around
    this it looks like retrieving it from the AudioUnit will work. However, and this is where
    it gets weird, it doesn't seem to work with capture devices, nor at all on iOS... Therefore
    I'm going to fall back to a default assumption in these cases.
    */
#if defined(MA_APPLE_DESKTOP)
    result = ma_get_AudioUnit_channel_map(pContext, pData->audioUnit, deviceType, pData->channelMapOut, pData->channelsOut);
    if (result != MA_SUCCESS) {
    #if 0
        /* Try falling back to the channel map from the AudioObject. */
        result = ma_get_AudioObject_channel_map(pContext, deviceObjectID, deviceType, pData->channelMapOut, pData->channelsOut);
        if (result != MA_SUCCESS) {
            return result;
        }
    #else
        /* Fall back to default assumptions. */
        ma_get_standard_channel_map(ma_standard_channel_map_default, pData->channelsOut, pData->channelMapOut);
    #endif
    }
#else
    /* TODO: Figure out how to get the channel map using AVAudioSession. */
    ma_get_standard_channel_map(ma_standard_channel_map_default, pData->channelsOut, pData->channelMapOut);
#endif
    

    /* Buffer size. Not allowing this to be configurable on iOS. */
    actualPeriodSizeInFrames = pData->periodSizeInFramesIn;
    
#if defined(MA_APPLE_DESKTOP)
    if (actualPeriodSizeInFrames == 0) {
        actualPeriodSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(pData->periodSizeInMillisecondsIn, pData->sampleRateOut);
    }
    
    result = ma_set_AudioObject_buffer_size_in_frames(pContext, deviceObjectID, deviceType, &actualPeriodSizeInFrames);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    pData->periodSizeInFramesOut = actualPeriodSizeInFrames;
#else
    actualPeriodSizeInFrames = 2048;
    pData->periodSizeInFramesOut = actualPeriodSizeInFrames;
#endif


    /*
    During testing I discovered that the buffer size can be too big. You'll get an error like this:
    
      kAudioUnitErr_TooManyFramesToProcess : inFramesToProcess=4096, mMaxFramesPerSlice=512
    
    Note how inFramesToProcess is smaller than mMaxFramesPerSlice. To fix, we need to set kAudioUnitProperty_MaximumFramesPerSlice to that
    of the size of our buffer, or do it the other way around and set our buffer size to the kAudioUnitProperty_MaximumFramesPerSlice.
    */
    {
        /*AudioUnitScope propScope = (deviceType == ma_device_type_playback) ? kAudioUnitScope_Input : kAudioUnitScope_Output;
        AudioUnitElement propBus = (deviceType == ma_device_type_playback) ? MA_COREAUDIO_OUTPUT_BUS : MA_COREAUDIO_INPUT_BUS;
    
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_MaximumFramesPerSlice, propScope, propBus, &actualBufferSizeInFrames, sizeof(actualBufferSizeInFrames));
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }*/
        
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_MaximumFramesPerSlice, kAudioUnitScope_Global, 0, &actualPeriodSizeInFrames, sizeof(actualPeriodSizeInFrames));
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
    }
    
    /* We need a buffer list if this is an input device. We render into this in the input callback. */
    if (deviceType == ma_device_type_capture) {
        ma_bool32 isInterleaved = (bestFormat.mFormatFlags & kAudioFormatFlagIsNonInterleaved) == 0;
        size_t allocationSize;
        AudioBufferList* pBufferList;

        allocationSize = sizeof(AudioBufferList) - sizeof(AudioBuffer);  /* Subtract sizeof(AudioBuffer) because that part is dynamically sized. */
        if (isInterleaved) {
            /* Interleaved case. This is the simple case because we just have one buffer. */
            allocationSize += sizeof(AudioBuffer) * 1;
            allocationSize += actualPeriodSizeInFrames * ma_get_bytes_per_frame(pData->formatOut, pData->channelsOut);
        } else {
            /* Non-interleaved case. This is the more complex case because there's more than one buffer. */
            allocationSize += sizeof(AudioBuffer) * pData->channelsOut;
            allocationSize += actualPeriodSizeInFrames * ma_get_bytes_per_sample(pData->formatOut) * pData->channelsOut;
        }
        
        pBufferList = (AudioBufferList*)ma__malloc_from_callbacks(allocationSize, &pContext->allocationCallbacks);
        if (pBufferList == NULL) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return MA_OUT_OF_MEMORY;
        }
        
        if (isInterleaved) {
            pBufferList->mNumberBuffers = 1;
            pBufferList->mBuffers[0].mNumberChannels = pData->channelsOut;
            pBufferList->mBuffers[0].mDataByteSize   = actualPeriodSizeInFrames * ma_get_bytes_per_frame(pData->formatOut, pData->channelsOut);
            pBufferList->mBuffers[0].mData           = (ma_uint8*)pBufferList + sizeof(AudioBufferList);
        } else {
            ma_uint32 iBuffer;
            pBufferList->mNumberBuffers = pData->channelsOut;
            for (iBuffer = 0; iBuffer < pBufferList->mNumberBuffers; ++iBuffer) {
                pBufferList->mBuffers[iBuffer].mNumberChannels = 1;
                pBufferList->mBuffers[iBuffer].mDataByteSize   = actualPeriodSizeInFrames * ma_get_bytes_per_sample(pData->formatOut);
                pBufferList->mBuffers[iBuffer].mData           = (ma_uint8*)pBufferList + ((sizeof(AudioBufferList) - sizeof(AudioBuffer)) + (sizeof(AudioBuffer) * pData->channelsOut)) + (actualPeriodSizeInFrames * ma_get_bytes_per_sample(pData->formatOut) * iBuffer);
            }
        }
        
        pData->pAudioBufferList = pBufferList;
    }
    
    /* Callbacks. */
    callbackInfo.inputProcRefCon = pDevice_DoNotReference;
    if (deviceType == ma_device_type_playback) {
        callbackInfo.inputProc = ma_on_output__coreaudio;
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_SetRenderCallback, kAudioUnitScope_Global, MA_COREAUDIO_OUTPUT_BUS, &callbackInfo, sizeof(callbackInfo));
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
    } else {
        callbackInfo.inputProc = ma_on_input__coreaudio;
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_SetInputCallback, kAudioUnitScope_Global, MA_COREAUDIO_INPUT_BUS, &callbackInfo, sizeof(callbackInfo));
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
    }
    
    /* We need to listen for stop events. */
    if (pData->registerStopEvent) {
        status = ((ma_AudioUnitAddPropertyListener_proc)pContext->coreaudio.AudioUnitAddPropertyListener)(pData->audioUnit, kAudioOutputUnitProperty_IsRunning, on_start_stop__coreaudio, pDevice_DoNotReference);
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
    }
    
    /* Initialize the audio unit. */
    status = ((ma_AudioUnitInitialize_proc)pContext->coreaudio.AudioUnitInitialize)(pData->audioUnit);
    if (status != noErr) {
        ma__free_from_callbacks(pData->pAudioBufferList, &pContext->allocationCallbacks);
        pData->pAudioBufferList = NULL;
        ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return ma_result_from_OSStatus(status);
    }
    
    /* Grab the name. */
#if defined(MA_APPLE_DESKTOP)
    ma_get_AudioObject_name(pContext, deviceObjectID, sizeof(pData->deviceName), pData->deviceName);
#else
    if (deviceType == ma_device_type_playback) {
        ma_strcpy_s(pData->deviceName, sizeof(pData->deviceName), MA_DEFAULT_PLAYBACK_DEVICE_NAME);
    } else {
        ma_strcpy_s(pData->deviceName, sizeof(pData->deviceName), MA_DEFAULT_CAPTURE_DEVICE_NAME);
    }
#endif
    
    return result;
}

#if defined(MA_APPLE_DESKTOP)
static ma_result ma_device_reinit_internal__coreaudio(ma_device* pDevice, ma_device_type deviceType, ma_bool32 disposePreviousAudioUnit)
{
    ma_device_init_internal_data__coreaudio data;
    ma_result result;

    /* This should only be called for playback or capture, not duplex. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    if (deviceType == ma_device_type_capture) {
        data.formatIn               = pDevice->capture.format;
        data.channelsIn             = pDevice->capture.channels;
        data.sampleRateIn           = pDevice->sampleRate;
        MA_COPY_MEMORY(data.channelMapIn, pDevice->capture.channelMap, sizeof(pDevice->capture.channelMap));
        data.usingDefaultFormat     = pDevice->capture.usingDefaultFormat;
        data.usingDefaultChannels   = pDevice->capture.usingDefaultChannels;
        data.usingDefaultSampleRate = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap = pDevice->capture.usingDefaultChannelMap;
        data.shareMode              = pDevice->capture.shareMode;
        data.registerStopEvent      = MA_TRUE;
        
        if (disposePreviousAudioUnit) {
            ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
            ((ma_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
        }
        if (pDevice->coreaudio.pAudioBufferList) {
            ma__free_from_callbacks(pDevice->coreaudio.pAudioBufferList, &pDevice->pContext->allocationCallbacks);
        }
    } else if (deviceType == ma_device_type_playback) {
        data.formatIn               = pDevice->playback.format;
        data.channelsIn             = pDevice->playback.channels;
        data.sampleRateIn           = pDevice->sampleRate;
        MA_COPY_MEMORY(data.channelMapIn, pDevice->playback.channelMap, sizeof(pDevice->playback.channelMap));
        data.usingDefaultFormat     = pDevice->playback.usingDefaultFormat;
        data.usingDefaultChannels   = pDevice->playback.usingDefaultChannels;
        data.usingDefaultSampleRate = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap = pDevice->playback.usingDefaultChannelMap;
        data.shareMode              = pDevice->playback.shareMode;
        data.registerStopEvent      = (pDevice->type != ma_device_type_duplex);
        
        if (disposePreviousAudioUnit) {
            ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
            ((ma_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
        }
    }
    data.periodSizeInFramesIn       = pDevice->coreaudio.originalPeriodSizeInFrames;
    data.periodSizeInMillisecondsIn = pDevice->coreaudio.originalPeriodSizeInMilliseconds;
    data.periodsIn                  = pDevice->coreaudio.originalPeriods;

    /* Need at least 3 periods for duplex. */
    if (data.periodsIn < 3 && pDevice->type == ma_device_type_duplex) {
        data.periodsIn = 3;
    }

    result = ma_device_init_internal__coreaudio(pDevice->pContext, deviceType, NULL, &data, (void*)pDevice);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    if (deviceType == ma_device_type_capture) {
    #if defined(MA_APPLE_DESKTOP)
        pDevice->coreaudio.deviceObjectIDCapture     = (ma_uint32)data.deviceObjectID;
    #endif
        pDevice->coreaudio.audioUnitCapture          = (ma_ptr)data.audioUnit;
        pDevice->coreaudio.pAudioBufferList          = (ma_ptr)data.pAudioBufferList;
        
        pDevice->capture.internalFormat              = data.formatOut;
        pDevice->capture.internalChannels            = data.channelsOut;
        pDevice->capture.internalSampleRate          = data.sampleRateOut;
        MA_COPY_MEMORY(pDevice->capture.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->capture.internalPeriodSizeInFrames  = data.periodSizeInFramesOut;
        pDevice->capture.internalPeriods             = data.periodsOut;
    } else if (deviceType == ma_device_type_playback) {
    #if defined(MA_APPLE_DESKTOP)
        pDevice->coreaudio.deviceObjectIDPlayback    = (ma_uint32)data.deviceObjectID;
    #endif
        pDevice->coreaudio.audioUnitPlayback         = (ma_ptr)data.audioUnit;
        
        pDevice->playback.internalFormat             = data.formatOut;
        pDevice->playback.internalChannels           = data.channelsOut;
        pDevice->playback.internalSampleRate         = data.sampleRateOut;
        MA_COPY_MEMORY(pDevice->playback.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->playback.internalPeriodSizeInFrames = data.periodSizeInFramesOut;
        pDevice->playback.internalPeriods            = data.periodsOut;
    }
    
    return MA_SUCCESS;
}
#endif /* MA_APPLE_DESKTOP */

static ma_result ma_device_init__coreaudio(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pConfig != NULL);
    MA_ASSERT(pDevice != NULL);

    if (pConfig->deviceType == ma_device_type_loopback) {
        return MA_DEVICE_TYPE_NOT_SUPPORTED;
    }

    /* No exclusive mode with the Core Audio backend for now. */
    if (((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.shareMode  == ma_share_mode_exclusive) ||
        ((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.shareMode == ma_share_mode_exclusive)) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }
    
    /* Capture needs to be initialized first. */
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_device_init_internal_data__coreaudio data;
        data.formatIn                   = pConfig->capture.format;
        data.channelsIn                 = pConfig->capture.channels;
        data.sampleRateIn               = pConfig->sampleRate;
        MA_COPY_MEMORY(data.channelMapIn, pConfig->capture.channelMap, sizeof(pConfig->capture.channelMap));
        data.usingDefaultFormat         = pDevice->capture.usingDefaultFormat;
        data.usingDefaultChannels       = pDevice->capture.usingDefaultChannels;
        data.usingDefaultSampleRate     = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap     = pDevice->capture.usingDefaultChannelMap;
        data.shareMode                  = pConfig->capture.shareMode;
        data.periodSizeInFramesIn       = pConfig->periodSizeInFrames;
        data.periodSizeInMillisecondsIn = pConfig->periodSizeInMilliseconds;
        data.periodsIn                  = pConfig->periods;
        data.registerStopEvent          = MA_TRUE;

        /* Need at least 3 periods for duplex. */
        if (data.periodsIn < 3 && pConfig->deviceType == ma_device_type_duplex) {
            data.periodsIn = 3;
        }
        
        result = ma_device_init_internal__coreaudio(pDevice->pContext, ma_device_type_capture, pConfig->capture.pDeviceID, &data, (void*)pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
        
        pDevice->coreaudio.isDefaultCaptureDevice   = (pConfig->capture.pDeviceID == NULL);
    #if defined(MA_APPLE_DESKTOP)
        pDevice->coreaudio.deviceObjectIDCapture    = (ma_uint32)data.deviceObjectID;
    #endif
        pDevice->coreaudio.audioUnitCapture         = (ma_ptr)data.audioUnit;
        pDevice->coreaudio.pAudioBufferList         = (ma_ptr)data.pAudioBufferList;
        
        pDevice->capture.internalFormat             = data.formatOut;
        pDevice->capture.internalChannels           = data.channelsOut;
        pDevice->capture.internalSampleRate         = data.sampleRateOut;
        MA_COPY_MEMORY(pDevice->capture.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->capture.internalPeriodSizeInFrames = data.periodSizeInFramesOut;
        pDevice->capture.internalPeriods            = data.periodsOut;
        
    #if defined(MA_APPLE_DESKTOP)
        /*
        If we are using the default device we'll need to listen for changes to the system's default device so we can seemlessly
        switch the device in the background.
        */
        if (pConfig->capture.pDeviceID == NULL) {
            ma_device__track__coreaudio(pDevice);
        }
    #endif
    }
    
    /* Playback. */
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_device_init_internal_data__coreaudio data;
        data.formatIn                   = pConfig->playback.format;
        data.channelsIn                 = pConfig->playback.channels;
        data.sampleRateIn               = pConfig->sampleRate;
        MA_COPY_MEMORY(data.channelMapIn, pConfig->playback.channelMap, sizeof(pConfig->playback.channelMap));
        data.usingDefaultFormat         = pDevice->playback.usingDefaultFormat;
        data.usingDefaultChannels       = pDevice->playback.usingDefaultChannels;
        data.usingDefaultSampleRate     = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap     = pDevice->playback.usingDefaultChannelMap;
        data.shareMode                  = pConfig->playback.shareMode;
        
        /* In full-duplex mode we want the playback buffer to be the same size as the capture buffer. */
        if (pConfig->deviceType == ma_device_type_duplex) {
            data.periodSizeInFramesIn       = pDevice->capture.internalPeriodSizeInFrames;
            data.periodsIn                  = pDevice->capture.internalPeriods;
            data.registerStopEvent          = MA_FALSE;
        } else {
            data.periodSizeInFramesIn       = pConfig->periodSizeInFrames;
            data.periodSizeInMillisecondsIn = pConfig->periodSizeInMilliseconds;
            data.periodsIn                  = pConfig->periods;
            data.registerStopEvent          = MA_TRUE;
        }
        
        result = ma_device_init_internal__coreaudio(pDevice->pContext, ma_device_type_playback, pConfig->playback.pDeviceID, &data, (void*)pDevice);
        if (result != MA_SUCCESS) {
            if (pConfig->deviceType == ma_device_type_duplex) {
                ((ma_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
                if (pDevice->coreaudio.pAudioBufferList) {
                    ma__free_from_callbacks(pDevice->coreaudio.pAudioBufferList, &pDevice->pContext->allocationCallbacks);
                }
            }
            return result;
        }
        
        pDevice->coreaudio.isDefaultPlaybackDevice   = (pConfig->playback.pDeviceID == NULL);
    #if defined(MA_APPLE_DESKTOP)
        pDevice->coreaudio.deviceObjectIDPlayback    = (ma_uint32)data.deviceObjectID;
    #endif
        pDevice->coreaudio.audioUnitPlayback         = (ma_ptr)data.audioUnit;
        
        pDevice->playback.internalFormat             = data.formatOut;
        pDevice->playback.internalChannels           = data.channelsOut;
        pDevice->playback.internalSampleRate         = data.sampleRateOut;
        MA_COPY_MEMORY(pDevice->playback.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->playback.internalPeriodSizeInFrames = data.periodSizeInFramesOut;
        pDevice->playback.internalPeriods            = data.periodsOut;
        
    #if defined(MA_APPLE_DESKTOP)
        /*
        If we are using the default device we'll need to listen for changes to the system's default device so we can seemlessly
        switch the device in the background.
        */
        if (pConfig->playback.pDeviceID == NULL && (pConfig->deviceType != ma_device_type_duplex || pConfig->capture.pDeviceID != NULL)) {
            ma_device__track__coreaudio(pDevice);
        }
    #endif
    }
    
    pDevice->coreaudio.originalPeriodSizeInFrames       = pConfig->periodSizeInFrames;
    pDevice->coreaudio.originalPeriodSizeInMilliseconds = pConfig->periodSizeInMilliseconds;
    pDevice->coreaudio.originalPeriods                  = pConfig->periods;
    
    /*
    When stopping the device, a callback is called on another thread. We need to wait for this callback
    before returning from ma_device_stop(). This event is used for this.
    */
    ma_event_init(&pDevice->coreaudio.stopEvent);

    /* Need a ring buffer for duplex mode. */
    if (pConfig->deviceType == ma_device_type_duplex) {
        ma_uint32 rbSizeInFrames = (ma_uint32)ma_calculate_frame_count_after_resampling(pDevice->sampleRate, pDevice->capture.internalSampleRate, pDevice->capture.internalPeriodSizeInFrames * pDevice->capture.internalPeriods);
        ma_result result = ma_pcm_rb_init(pDevice->capture.format, pDevice->capture.channels, rbSizeInFrames, NULL, &pDevice->pContext->allocationCallbacks, &pDevice->coreaudio.duplexRB);
        if (result != MA_SUCCESS) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[Core Audio] Failed to initialize ring buffer.", result);
        }

        /* We need a period to act as a buffer for cases where the playback and capture device's end up desyncing. */
        {
            ma_uint32 bufferSizeInFrames = rbSizeInFrames / pDevice->capture.internalPeriods;
            void* pBufferData;
            ma_pcm_rb_acquire_write(&pDevice->coreaudio.duplexRB, &bufferSizeInFrames, &pBufferData);
            {
                MA_ZERO_MEMORY(pBufferData, bufferSizeInFrames * ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels));
            }
            ma_pcm_rb_commit_write(&pDevice->coreaudio.duplexRB, bufferSizeInFrames, pBufferData);
        }
    }

    /*
    We need to detect when a route has changed so we can update the data conversion pipeline accordingly. This is done
    differently on non-Desktop Apple platforms.
    */
#if defined(MA_APPLE_MOBILE)
    pDevice->coreaudio.pRouteChangeHandler = (__bridge_retained void*)[[ma_router_change_handler alloc] init:pDevice];
#endif

    return MA_SUCCESS;
}


static ma_result ma_device_start__coreaudio(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);
    
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        OSStatus status = ((ma_AudioOutputUnitStart_proc)pDevice->pContext->coreaudio.AudioOutputUnitStart)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
        if (status != noErr) {
            return ma_result_from_OSStatus(status);
        }
    }
    
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        OSStatus status = ((ma_AudioOutputUnitStart_proc)pDevice->pContext->coreaudio.AudioOutputUnitStart)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
        if (status != noErr) {
            if (pDevice->type == ma_device_type_duplex) {
                ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
            }
            return ma_result_from_OSStatus(status);
        }
    }
    
    return MA_SUCCESS;
}

static ma_result ma_device_stop__coreaudio(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    /* It's not clear from the documentation whether or not AudioOutputUnitStop() actually drains the device or not. */

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        OSStatus status = ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
        if (status != noErr) {
            return ma_result_from_OSStatus(status);
        }
    }
    
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        OSStatus status = ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
        if (status != noErr) {
            return ma_result_from_OSStatus(status);
        }
    }
    
    /* We need to wait for the callback to finish before returning. */
    ma_event_wait(&pDevice->coreaudio.stopEvent);
    return MA_SUCCESS;
}


static ma_result ma_context_uninit__coreaudio(ma_context* pContext)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pContext->backend == ma_backend_coreaudio);
    
#if defined(MA_APPLE_MOBILE)
    if (!pContext->coreaudio.noAudioSessionDeactivate) {
        if (![[AVAudioSession sharedInstance] setActive:false error:nil]) {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "Failed to deactivate audio session.", MA_FAILED_TO_INIT_BACKEND);
        }
    }
#endif
    
#if !defined(MA_NO_RUNTIME_LINKING) && !defined(MA_APPLE_MOBILE)
    ma_dlclose(pContext, pContext->coreaudio.hAudioUnit);
    ma_dlclose(pContext, pContext->coreaudio.hCoreAudio);
    ma_dlclose(pContext, pContext->coreaudio.hCoreFoundation);
#endif

#if !defined(MA_APPLE_MOBILE)
    ma_context__uninit_device_tracking__coreaudio(pContext);
#endif

    (void)pContext;
    return MA_SUCCESS;
}

#if defined(MA_APPLE_MOBILE)
static AVAudioSessionCategory ma_to_AVAudioSessionCategory(ma_ios_session_category category)
{
    /* The "default" and "none" categories are treated different and should not be used as an input into this function. */
    MA_ASSERT(category != ma_ios_session_category_default);
    MA_ASSERT(category != ma_ios_session_category_none);

    switch (category) {
        case ma_ios_session_category_ambient:         return AVAudioSessionCategoryAmbient;
        case ma_ios_session_category_solo_ambient:    return AVAudioSessionCategorySoloAmbient;
        case ma_ios_session_category_playback:        return AVAudioSessionCategoryPlayback;
        case ma_ios_session_category_record:          return AVAudioSessionCategoryRecord;
        case ma_ios_session_category_play_and_record: return AVAudioSessionCategoryPlayAndRecord;
        case ma_ios_session_category_multi_route:     return AVAudioSessionCategoryMultiRoute;
        case ma_ios_session_category_none:            return AVAudioSessionCategoryAmbient;
        case ma_ios_session_category_default:         return AVAudioSessionCategoryAmbient;
        default:                                      return AVAudioSessionCategoryAmbient;
    }
}
#endif

static ma_result ma_context_init__coreaudio(const ma_context_config* pConfig, ma_context* pContext)
{
#if !defined(MA_APPLE_MOBILE)
    ma_result result;
#endif

    MA_ASSERT(pConfig != NULL);
    MA_ASSERT(pContext != NULL);

#if defined(MA_APPLE_MOBILE)
    @autoreleasepool {
        AVAudioSession* pAudioSession = [AVAudioSession sharedInstance];
        AVAudioSessionCategoryOptions options = pConfig->coreaudio.sessionCategoryOptions;

        MA_ASSERT(pAudioSession != NULL);

        if (pConfig->coreaudio.sessionCategory == ma_ios_session_category_default) {
            /*
            I'm going to use trial and error to determine our default session category. First we'll try PlayAndRecord. If that fails
            we'll try Playback and if that fails we'll try record. If all of these fail we'll just not set the category.
            */
        #if !defined(MA_APPLE_TV) && !defined(MA_APPLE_WATCH)
            options |= AVAudioSessionCategoryOptionDefaultToSpeaker;
        #endif

            if ([pAudioSession setCategory: AVAudioSessionCategoryPlayAndRecord withOptions:options error:nil]) {
                /* Using PlayAndRecord */
            } else if ([pAudioSession setCategory: AVAudioSessionCategoryPlayback withOptions:options error:nil]) {
                /* Using Playback */
            } else if ([pAudioSession setCategory: AVAudioSessionCategoryRecord withOptions:options error:nil]) {
                /* Using Record */
            } else {
                /* Leave as default? */
            }
        } else {
            if (pConfig->coreaudio.sessionCategory != ma_ios_session_category_none) {
                if (![pAudioSession setCategory: ma_to_AVAudioSessionCategory(pConfig->coreaudio.sessionCategory) withOptions:options error:nil]) {
                    return MA_INVALID_OPERATION;    /* Failed to set session category. */
                }
            }
        }
        
        if (!pConfig->coreaudio.noAudioSessionActivate) {
            if (![pAudioSession setActive:true error:nil]) {
                return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "Failed to activate audio session.", MA_FAILED_TO_INIT_BACKEND);
            }
        }
    }
#endif
    
#if !defined(MA_NO_RUNTIME_LINKING) && !defined(MA_APPLE_MOBILE)
    pContext->coreaudio.hCoreFoundation = ma_dlopen(pContext, "CoreFoundation.framework/CoreFoundation");
    if (pContext->coreaudio.hCoreFoundation == NULL) {
        return MA_API_NOT_FOUND;
    }
    
    pContext->coreaudio.CFStringGetCString = ma_dlsym(pContext, pContext->coreaudio.hCoreFoundation, "CFStringGetCString");
    pContext->coreaudio.CFRelease          = ma_dlsym(pContext, pContext->coreaudio.hCoreFoundation, "CFRelease");
    
    
    pContext->coreaudio.hCoreAudio = ma_dlopen(pContext, "CoreAudio.framework/CoreAudio");
    if (pContext->coreaudio.hCoreAudio == NULL) {
        ma_dlclose(pContext, pContext->coreaudio.hCoreFoundation);
        return MA_API_NOT_FOUND;
    }
    
    pContext->coreaudio.AudioObjectGetPropertyData        = ma_dlsym(pContext, pContext->coreaudio.hCoreAudio, "AudioObjectGetPropertyData");
    pContext->coreaudio.AudioObjectGetPropertyDataSize    = ma_dlsym(pContext, pContext->coreaudio.hCoreAudio, "AudioObjectGetPropertyDataSize");
    pContext->coreaudio.AudioObjectSetPropertyData        = ma_dlsym(pContext, pContext->coreaudio.hCoreAudio, "AudioObjectSetPropertyData");
    pContext->coreaudio.AudioObjectAddPropertyListener    = ma_dlsym(pContext, pContext->coreaudio.hCoreAudio, "AudioObjectAddPropertyListener");
    pContext->coreaudio.AudioObjectRemovePropertyListener = ma_dlsym(pContext, pContext->coreaudio.hCoreAudio, "AudioObjectRemovePropertyListener");

    /*
    It looks like Apple has moved some APIs from AudioUnit into AudioToolbox on more recent versions of macOS. They are still
    defined in AudioUnit, but just in case they decide to remove them from there entirely I'm going to implement a fallback.
    The way it'll work is that it'll first try AudioUnit, and if the required symbols are not present there we'll fall back to
    AudioToolbox.
    */
    pContext->coreaudio.hAudioUnit = ma_dlopen(pContext, "AudioUnit.framework/AudioUnit");
    if (pContext->coreaudio.hAudioUnit == NULL) {
        ma_dlclose(pContext, pContext->coreaudio.hCoreAudio);
        ma_dlclose(pContext, pContext->coreaudio.hCoreFoundation);
        return MA_API_NOT_FOUND;
    }
    
    if (ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioComponentFindNext") == NULL) {
        /* Couldn't find the required symbols in AudioUnit, so fall back to AudioToolbox. */
        ma_dlclose(pContext, pContext->coreaudio.hAudioUnit);
        pContext->coreaudio.hAudioUnit = ma_dlopen(pContext, "AudioToolbox.framework/AudioToolbox");
        if (pContext->coreaudio.hAudioUnit == NULL) {
            ma_dlclose(pContext, pContext->coreaudio.hCoreAudio);
            ma_dlclose(pContext, pContext->coreaudio.hCoreFoundation);
            return MA_API_NOT_FOUND;
        }
    }
    
    pContext->coreaudio.AudioComponentFindNext            = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioComponentFindNext");
    pContext->coreaudio.AudioComponentInstanceDispose     = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioComponentInstanceDispose");
    pContext->coreaudio.AudioComponentInstanceNew         = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioComponentInstanceNew");
    pContext->coreaudio.AudioOutputUnitStart              = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioOutputUnitStart");
    pContext->coreaudio.AudioOutputUnitStop               = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioOutputUnitStop");
    pContext->coreaudio.AudioUnitAddPropertyListener      = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioUnitAddPropertyListener");
    pContext->coreaudio.AudioUnitGetPropertyInfo          = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioUnitGetPropertyInfo");
    pContext->coreaudio.AudioUnitGetProperty              = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioUnitGetProperty");
    pContext->coreaudio.AudioUnitSetProperty              = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioUnitSetProperty");
    pContext->coreaudio.AudioUnitInitialize               = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioUnitInitialize");
    pContext->coreaudio.AudioUnitRender                   = ma_dlsym(pContext, pContext->coreaudio.hAudioUnit, "AudioUnitRender");
#else
    pContext->coreaudio.CFStringGetCString                = (ma_proc)CFStringGetCString;
    pContext->coreaudio.CFRelease                         = (ma_proc)CFRelease;
    
    #if defined(MA_APPLE_DESKTOP)
    pContext->coreaudio.AudioObjectGetPropertyData        = (ma_proc)AudioObjectGetPropertyData;
    pContext->coreaudio.AudioObjectGetPropertyDataSize    = (ma_proc)AudioObjectGetPropertyDataSize;
    pContext->coreaudio.AudioObjectSetPropertyData        = (ma_proc)AudioObjectSetPropertyData;
    pContext->coreaudio.AudioObjectAddPropertyListener    = (ma_proc)AudioObjectAddPropertyListener;
    pContext->coreaudio.AudioObjectRemovePropertyListener = (ma_proc)AudioObjectRemovePropertyListener;
    #endif
    
    pContext->coreaudio.AudioComponentFindNext            = (ma_proc)AudioComponentFindNext;
    pContext->coreaudio.AudioComponentInstanceDispose     = (ma_proc)AudioComponentInstanceDispose;
    pContext->coreaudio.AudioComponentInstanceNew         = (ma_proc)AudioComponentInstanceNew;
    pContext->coreaudio.AudioOutputUnitStart              = (ma_proc)AudioOutputUnitStart;
    pContext->coreaudio.AudioOutputUnitStop               = (ma_proc)AudioOutputUnitStop;
    pContext->coreaudio.AudioUnitAddPropertyListener      = (ma_proc)AudioUnitAddPropertyListener;
    pContext->coreaudio.AudioUnitGetPropertyInfo          = (ma_proc)AudioUnitGetPropertyInfo;
    pContext->coreaudio.AudioUnitGetProperty              = (ma_proc)AudioUnitGetProperty;
    pContext->coreaudio.AudioUnitSetProperty              = (ma_proc)AudioUnitSetProperty;
    pContext->coreaudio.AudioUnitInitialize               = (ma_proc)AudioUnitInitialize;
    pContext->coreaudio.AudioUnitRender                   = (ma_proc)AudioUnitRender;
#endif

    pContext->isBackendAsynchronous = MA_TRUE;
    
    pContext->onUninit        = ma_context_uninit__coreaudio;
    pContext->onDeviceIDEqual = ma_context_is_device_id_equal__coreaudio;
    pContext->onEnumDevices   = ma_context_enumerate_devices__coreaudio;
    pContext->onGetDeviceInfo = ma_context_get_device_info__coreaudio;
    pContext->onDeviceInit    = ma_device_init__coreaudio;
    pContext->onDeviceUninit  = ma_device_uninit__coreaudio;
    pContext->onDeviceStart   = ma_device_start__coreaudio;
    pContext->onDeviceStop    = ma_device_stop__coreaudio;
    
    /* Audio component. */
    {
        AudioComponentDescription desc;
        desc.componentType         = kAudioUnitType_Output;
    #if defined(MA_APPLE_DESKTOP)
        desc.componentSubType      = kAudioUnitSubType_HALOutput;
    #else
        desc.componentSubType      = kAudioUnitSubType_RemoteIO;
    #endif
        desc.componentManufacturer = kAudioUnitManufacturer_Apple;
        desc.componentFlags        = 0;
        desc.componentFlagsMask    = 0;
    
        pContext->coreaudio.component = ((ma_AudioComponentFindNext_proc)pContext->coreaudio.AudioComponentFindNext)(NULL, &desc);
        if (pContext->coreaudio.component == NULL) {
        #if !defined(MA_NO_RUNTIME_LINKING) && !defined(MA_APPLE_MOBILE)
            ma_dlclose(pContext, pContext->coreaudio.hAudioUnit);
            ma_dlclose(pContext, pContext->coreaudio.hCoreAudio);
            ma_dlclose(pContext, pContext->coreaudio.hCoreFoundation);
        #endif
            return MA_FAILED_TO_INIT_BACKEND;
        }
    }
    
#if !defined(MA_APPLE_MOBILE)
    result = ma_context__init_device_tracking__coreaudio(pContext);
    if (result != MA_SUCCESS) {
    #if !defined(MA_NO_RUNTIME_LINKING) && !defined(MA_APPLE_MOBILE)
        ma_dlclose(pContext, pContext->coreaudio.hAudioUnit);
        ma_dlclose(pContext, pContext->coreaudio.hCoreAudio);
        ma_dlclose(pContext, pContext->coreaudio.hCoreFoundation);
    #endif
        return result;
    }
#endif

    pContext->coreaudio.noAudioSessionDeactivate = pConfig->coreaudio.noAudioSessionDeactivate;

    return MA_SUCCESS;
}
#endif  /* Core Audio */



/******************************************************************************

sndio Backend

******************************************************************************/
#ifdef MA_HAS_SNDIO
#include <fcntl.h>

/*
Only supporting OpenBSD. This did not work very well at all on FreeBSD when I tried it. Not sure if this is due
to miniaudio's implementation or if it's some kind of system configuration issue, but basically the default device
just doesn't emit any sound, or at times you'll hear tiny pieces. I will consider enabling this when there's
demand for it or if I can get it tested and debugged more thoroughly.
*/
#if 0
#if defined(__NetBSD__) || defined(__OpenBSD__)
#include <sys/audioio.h>
#endif
#if defined(__FreeBSD__) || defined(__DragonFly__)
#include <sys/soundcard.h>
#endif
#endif

#define MA_SIO_DEVANY   "default"
#define MA_SIO_PLAY     1
#define MA_SIO_REC      2
#define MA_SIO_NENC     8
#define MA_SIO_NCHAN    8
#define MA_SIO_NRATE    16
#define MA_SIO_NCONF    4

struct ma_sio_hdl; /* <-- Opaque */

struct ma_sio_par
{
    unsigned int bits;
    unsigned int bps;
    unsigned int sig;
    unsigned int le;
    unsigned int msb;
    unsigned int rchan;
    unsigned int pchan;
    unsigned int rate;
    unsigned int bufsz;
    unsigned int xrun;
    unsigned int round;
    unsigned int appbufsz;
    int __pad[3];
    unsigned int __magic;
};

struct ma_sio_enc
{
    unsigned int bits;
    unsigned int bps;
    unsigned int sig;
    unsigned int le;
    unsigned int msb;
};

struct ma_sio_conf
{
    unsigned int enc;
    unsigned int rchan;
    unsigned int pchan;
    unsigned int rate;
};

struct ma_sio_cap
{
    struct ma_sio_enc enc[MA_SIO_NENC];
    unsigned int rchan[MA_SIO_NCHAN];
    unsigned int pchan[MA_SIO_NCHAN];
    unsigned int rate[MA_SIO_NRATE];
    int __pad[7];
    unsigned int nconf;
    struct ma_sio_conf confs[MA_SIO_NCONF];
};

typedef struct ma_sio_hdl* (* ma_sio_open_proc)   (const char*, unsigned int, int);
typedef void               (* ma_sio_close_proc)  (struct ma_sio_hdl*);
typedef int                (* ma_sio_setpar_proc) (struct ma_sio_hdl*, struct ma_sio_par*);
typedef int                (* ma_sio_getpar_proc) (struct ma_sio_hdl*, struct ma_sio_par*);
typedef int                (* ma_sio_getcap_proc) (struct ma_sio_hdl*, struct ma_sio_cap*);
typedef size_t             (* ma_sio_write_proc)  (struct ma_sio_hdl*, const void*, size_t);
typedef size_t             (* ma_sio_read_proc)   (struct ma_sio_hdl*, void*, size_t);
typedef int                (* ma_sio_start_proc)  (struct ma_sio_hdl*);
typedef int                (* ma_sio_stop_proc)   (struct ma_sio_hdl*);
typedef int                (* ma_sio_initpar_proc)(struct ma_sio_par*);

static ma_uint32 ma_get_standard_sample_rate_priority_index__sndio(ma_uint32 sampleRate)   /* Lower = higher priority */
{
    ma_uint32 i;
    for (i = 0; i < ma_countof(g_maStandardSampleRatePriorities); ++i) {
        if (g_maStandardSampleRatePriorities[i] == sampleRate) {
            return i;
        }
    }

    return (ma_uint32)-1;
}

static ma_format ma_format_from_sio_enc__sndio(unsigned int bits, unsigned int bps, unsigned int sig, unsigned int le, unsigned int msb)
{
    /* We only support native-endian right now. */
    if ((ma_is_little_endian() && le == 0) || (ma_is_big_endian() && le == 1)) {
        return ma_format_unknown;
    }
    
    if (bits ==  8 && bps == 1 && sig == 0) {
        return ma_format_u8;
    }
    if (bits == 16 && bps == 2 && sig == 1) {
        return ma_format_s16;
    }
    if (bits == 24 && bps == 3 && sig == 1) {
        return ma_format_s24;
    }
    if (bits == 24 && bps == 4 && sig == 1 && msb == 0) {
        /*return ma_format_s24_32;*/
    }
    if (bits == 32 && bps == 4 && sig == 1) {
        return ma_format_s32;
    }
    
    return ma_format_unknown;
}

static ma_format ma_find_best_format_from_sio_cap__sndio(struct ma_sio_cap* caps)
{
    ma_format bestFormat;
    unsigned int iConfig;

    MA_ASSERT(caps != NULL);
    
    bestFormat = ma_format_unknown;
    for (iConfig = 0; iConfig < caps->nconf; iConfig += 1) {
        unsigned int iEncoding;
        for (iEncoding = 0; iEncoding < MA_SIO_NENC; iEncoding += 1) {
            unsigned int bits;
            unsigned int bps;
            unsigned int sig;
            unsigned int le;
            unsigned int msb;
            ma_format format;

            if ((caps->confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            bits = caps->enc[iEncoding].bits;
            bps  = caps->enc[iEncoding].bps;
            sig  = caps->enc[iEncoding].sig;
            le   = caps->enc[iEncoding].le;
            msb  = caps->enc[iEncoding].msb;
            format = ma_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format == ma_format_unknown) {
                continue;   /* Format not supported. */
            }
            
            if (bestFormat == ma_format_unknown) {
                bestFormat = format;
            } else {
                if (ma_get_format_priority_index(bestFormat) > ma_get_format_priority_index(format)) {    /* <-- Lower = better. */
                    bestFormat = format;
                }
            }
        }
    }
    
    return bestFormat;
}

static ma_uint32 ma_find_best_channels_from_sio_cap__sndio(struct ma_sio_cap* caps, ma_device_type deviceType, ma_format requiredFormat)
{
    ma_uint32 maxChannels;
    unsigned int iConfig;

    MA_ASSERT(caps != NULL);
    MA_ASSERT(requiredFormat != ma_format_unknown);
    
    /* Just pick whatever configuration has the most channels. */
    maxChannels = 0;
    for (iConfig = 0; iConfig < caps->nconf; iConfig += 1) {
        /* The encoding should be of requiredFormat. */
        unsigned int iEncoding;
        for (iEncoding = 0; iEncoding < MA_SIO_NENC; iEncoding += 1) {
            unsigned int iChannel;
            unsigned int bits;
            unsigned int bps;
            unsigned int sig;
            unsigned int le;
            unsigned int msb;
            ma_format format;

            if ((caps->confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            bits = caps->enc[iEncoding].bits;
            bps  = caps->enc[iEncoding].bps;
            sig  = caps->enc[iEncoding].sig;
            le   = caps->enc[iEncoding].le;
            msb  = caps->enc[iEncoding].msb;
            format = ma_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format != requiredFormat) {
                continue;
            }
            
            /* Getting here means the format is supported. Iterate over each channel count and grab the biggest one. */
            for (iChannel = 0; iChannel < MA_SIO_NCHAN; iChannel += 1) {
                unsigned int chan = 0;
                unsigned int channels;

                if (deviceType == ma_device_type_playback) {
                    chan = caps->confs[iConfig].pchan;
                } else {
                    chan = caps->confs[iConfig].rchan;
                }
            
                if ((chan & (1UL << iChannel)) == 0) {
                    continue;
                }
                
                if (deviceType == ma_device_type_playback) {
                    channels = caps->pchan[iChannel];
                } else {
                    channels = caps->rchan[iChannel];
                }
                
                if (maxChannels < channels) {
                    maxChannels = channels;
                }
            }
        }
    }
    
    return maxChannels;
}

static ma_uint32 ma_find_best_sample_rate_from_sio_cap__sndio(struct ma_sio_cap* caps, ma_device_type deviceType, ma_format requiredFormat, ma_uint32 requiredChannels)
{
    ma_uint32 firstSampleRate;
    ma_uint32 bestSampleRate;
    unsigned int iConfig;

    MA_ASSERT(caps != NULL);
    MA_ASSERT(requiredFormat != ma_format_unknown);
    MA_ASSERT(requiredChannels > 0);
    MA_ASSERT(requiredChannels <= MA_MAX_CHANNELS);
    
    firstSampleRate = 0; /* <-- If the device does not support a standard rate we'll fall back to the first one that's found. */
    bestSampleRate  = 0;

    for (iConfig = 0; iConfig < caps->nconf; iConfig += 1) {
        /* The encoding should be of requiredFormat. */
        unsigned int iEncoding;
        for (iEncoding = 0; iEncoding < MA_SIO_NENC; iEncoding += 1) {
            unsigned int iChannel;
            unsigned int bits;
            unsigned int bps;
            unsigned int sig;
            unsigned int le;
            unsigned int msb;
            ma_format format;

            if ((caps->confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            bits = caps->enc[iEncoding].bits;
            bps  = caps->enc[iEncoding].bps;
            sig  = caps->enc[iEncoding].sig;
            le   = caps->enc[iEncoding].le;
            msb  = caps->enc[iEncoding].msb;
            format = ma_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format != requiredFormat) {
                continue;
            }
            
            /* Getting here means the format is supported. Iterate over each channel count and grab the biggest one. */
            for (iChannel = 0; iChannel < MA_SIO_NCHAN; iChannel += 1) {
                unsigned int chan = 0;
                unsigned int channels;
                unsigned int iRate;

                if (deviceType == ma_device_type_playback) {
                    chan = caps->confs[iConfig].pchan;
                } else {
                    chan = caps->confs[iConfig].rchan;
                }
            
                if ((chan & (1UL << iChannel)) == 0) {
                    continue;
                }
                
                if (deviceType == ma_device_type_playback) {
                    channels = caps->pchan[iChannel];
                } else {
                    channels = caps->rchan[iChannel];
                }
                
                if (channels != requiredChannels) {
                    continue;
                }
                
                /* Getting here means we have found a compatible encoding/channel pair. */
                for (iRate = 0; iRate < MA_SIO_NRATE; iRate += 1) {
                    ma_uint32 rate = (ma_uint32)caps->rate[iRate];
                    ma_uint32 ratePriority;
                
                    if (firstSampleRate == 0) {
                        firstSampleRate = rate;
                    }
                    
                    /* Disregard this rate if it's not a standard one. */
                    ratePriority = ma_get_standard_sample_rate_priority_index__sndio(rate);
                    if (ratePriority == (ma_uint32)-1) {
                        continue;
                    }
                    
                    if (ma_get_standard_sample_rate_priority_index__sndio(bestSampleRate) > ratePriority) {   /* Lower = better. */
                        bestSampleRate = rate;
                    }
                }
            }
        }
    }
    
    /* If a standard sample rate was not found just fall back to the first one that was iterated. */
    if (bestSampleRate == 0) {
        bestSampleRate = firstSampleRate;
    }
    
    return bestSampleRate;
}


static ma_bool32 ma_context_is_device_id_equal__sndio(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    MA_ASSERT(pContext != NULL);
    MA_ASSERT(pID0 != NULL);
    MA_ASSERT(pID1 != NULL);
    (void)pContext;

    return ma_strcmp(pID0->sndio, pID1->sndio) == 0;
}

static ma_result ma_context_enumerate_devices__sndio(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_bool32 isTerminating = MA_FALSE;
    struct ma_sio_hdl* handle;

    MA_ASSERT(pContext != NULL);
    MA_ASSERT(callback != NULL);
    
    /* sndio doesn't seem to have a good device enumeration API, so I'm therefore only enumerating over default devices for now. */
    
    /* Playback. */
    if (!isTerminating) {
        handle = ((ma_sio_open_proc)pContext->sndio.sio_open)(MA_SIO_DEVANY, MA_SIO_PLAY, 0);
        if (handle != NULL) {
            /* Supports playback. */
            ma_device_info deviceInfo;
            MA_ZERO_OBJECT(&deviceInfo);
            ma_strcpy_s(deviceInfo.id.sndio, sizeof(deviceInfo.id.sndio), MA_SIO_DEVANY);
            ma_strcpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_PLAYBACK_DEVICE_NAME);
            
            isTerminating = !callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
            
            ((ma_sio_close_proc)pContext->sndio.sio_close)(handle);
        }
    }
    
    /* Capture. */
    if (!isTerminating) {
        handle = ((ma_sio_open_proc)pContext->sndio.sio_open)(MA_SIO_DEVANY, MA_SIO_REC, 0);
        if (handle != NULL) {
            /* Supports capture. */
            ma_device_info deviceInfo;
            MA_ZERO_OBJECT(&deviceInfo);
            ma_strcpy_s(deviceInfo.id.sndio, sizeof(deviceInfo.id.sndio), "default");
            ma_strcpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_CAPTURE_DEVICE_NAME);

            isTerminating = !callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
            
            ((ma_sio_close_proc)pContext->sndio.sio_close)(handle);
        }
    }
    
    return MA_SUCCESS;
}

static ma_result ma_context_get_device_info__sndio(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    char devid[256];
    struct ma_sio_hdl* handle;
    struct ma_sio_cap caps;
    unsigned int iConfig;

    MA_ASSERT(pContext != NULL);
    (void)shareMode;
    
    /* We need to open the device before we can get information about it. */
    if (pDeviceID == NULL) {
        ma_strcpy_s(devid, sizeof(devid), MA_SIO_DEVANY);
        ma_strcpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), (deviceType == ma_device_type_playback) ? MA_DEFAULT_PLAYBACK_DEVICE_NAME : MA_DEFAULT_CAPTURE_DEVICE_NAME);
    } else {
        ma_strcpy_s(devid, sizeof(devid), pDeviceID->sndio);
        ma_strcpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), devid);
    }
    
    handle = ((ma_sio_open_proc)pContext->sndio.sio_open)(devid, (deviceType == ma_device_type_playback) ? MA_SIO_PLAY : MA_SIO_REC, 0);
    if (handle == NULL) {
        return MA_NO_DEVICE;
    }
    
    if (((ma_sio_getcap_proc)pContext->sndio.sio_getcap)(handle, &caps) == 0) {
        return MA_ERROR;
    }
    
    for (iConfig = 0; iConfig < caps.nconf; iConfig += 1) {
        /*
        The main thing we care about is that the encoding is supported by miniaudio. If it is, we want to give
        preference to some formats over others.
        */
        unsigned int iEncoding;
        unsigned int iChannel;
        unsigned int iRate;

        for (iEncoding = 0; iEncoding < MA_SIO_NENC; iEncoding += 1) {
            unsigned int bits;
            unsigned int bps;
            unsigned int sig;
            unsigned int le;
            unsigned int msb;
            ma_format format;
            ma_bool32 formatExists = MA_FALSE;
            ma_uint32 iExistingFormat;

            if ((caps.confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            bits = caps.enc[iEncoding].bits;
            bps  = caps.enc[iEncoding].bps;
            sig  = caps.enc[iEncoding].sig;
            le   = caps.enc[iEncoding].le;
            msb  = caps.enc[iEncoding].msb;
            format = ma_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format == ma_format_unknown) {
                continue;   /* Format not supported. */
            }
            
            /* Add this format if it doesn't already exist. */
            for (iExistingFormat = 0; iExistingFormat < pDeviceInfo->formatCount; iExistingFormat += 1) {
                if (pDeviceInfo->formats[iExistingFormat] == format) {
                    formatExists = MA_TRUE;
                    break;
                }
            }
            
            if (!formatExists) {
                pDeviceInfo->formats[pDeviceInfo->formatCount++] = format;
            }
        }
        
        /* Channels. */
        for (iChannel = 0; iChannel < MA_SIO_NCHAN; iChannel += 1) {
            unsigned int chan = 0;
            unsigned int channels;

            if (deviceType == ma_device_type_playback) {
                chan = caps.confs[iConfig].pchan;
            } else {
                chan = caps.confs[iConfig].rchan;
            }
        
            if ((chan & (1UL << iChannel)) == 0) {
                continue;
            }
            
            if (deviceType == ma_device_type_playback) {
                channels = caps.pchan[iChannel];
            } else {
                channels = caps.rchan[iChannel];
            }
            
            if (pDeviceInfo->minChannels > channels) {
                pDeviceInfo->minChannels = channels;
            }
            if (pDeviceInfo->maxChannels < channels) {
                pDeviceInfo->maxChannels = channels;
            }
        }
        
        /* Sample rates. */
        for (iRate = 0; iRate < MA_SIO_NRATE; iRate += 1) {
            if ((caps.confs[iConfig].rate & (1UL << iRate)) != 0) {
                unsigned int rate = caps.rate[iRate];
                if (pDeviceInfo->minSampleRate > rate) {
                    pDeviceInfo->minSampleRate = rate;
                }
                if (pDeviceInfo->maxSampleRate < rate) {
                    pDeviceInfo->maxSampleRate = rate;
                }
            }
        }
    }

    ((ma_sio_close_proc)pContext->sndio.sio_close)(handle);
    return MA_SUCCESS;
}

static void ma_device_uninit__sndio(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_sio_close_proc)pDevice->pContext->sndio.sio_close)((struct ma_sio_hdl*)pDevice->sndio.handleCapture);
    }

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_sio_close_proc)pDevice->pContext->sndio.sio_close)((struct ma_sio_hdl*)pDevice->sndio.handlePlayback);
    }
}

static ma_result ma_device_init_handle__sndio(ma_context* pContext, const ma_device_config* pConfig, ma_device_type deviceType, ma_device* pDevice)
{
    const char* pDeviceName;
    ma_ptr handle;
    int openFlags = 0;
    struct ma_sio_cap caps;
    struct ma_sio_par par;
    ma_device_id* pDeviceID;
    ma_format format;
    ma_uint32 channels;
    ma_uint32 sampleRate;
    ma_format internalFormat;
    ma_uint32 internalChannels;
    ma_uint32 internalSampleRate;
    ma_uint32 internalPeriodSizeInFrames;
    ma_uint32 internalPeriods;

    MA_ASSERT(pContext   != NULL);
    MA_ASSERT(pConfig    != NULL);
    MA_ASSERT(deviceType != ma_device_type_duplex);
    MA_ASSERT(pDevice    != NULL);

    if (deviceType == ma_device_type_capture) {
        openFlags  = MA_SIO_REC;
        pDeviceID  = pConfig->capture.pDeviceID;
        format     = pConfig->capture.format;
        channels   = pConfig->capture.channels;
        sampleRate = pConfig->sampleRate;
    } else {
        openFlags = MA_SIO_PLAY;
        pDeviceID  = pConfig->playback.pDeviceID;
        format     = pConfig->playback.format;
        channels   = pConfig->playback.channels;
        sampleRate = pConfig->sampleRate;
    }

    pDeviceName = MA_SIO_DEVANY;
    if (pDeviceID != NULL) {
        pDeviceName = pDeviceID->sndio;
    }

    handle = (ma_ptr)((ma_sio_open_proc)pContext->sndio.sio_open)(pDeviceName, openFlags, 0);
    if (handle == NULL) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to open device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    /* We need to retrieve the device caps to determine the most appropriate format to use. */
    if (((ma_sio_getcap_proc)pContext->sndio.sio_getcap)((struct ma_sio_hdl*)handle, &caps) == 0) {
        ((ma_sio_close_proc)pContext->sndio.sio_close)((struct ma_sio_hdl*)handle);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to retrieve device caps.", MA_ERROR);
    }

    /*
    Note: sndio reports a huge range of available channels. This is inconvenient for us because there's no real
    way, as far as I can tell, to get the _actual_ channel count of the device. I'm therefore restricting this
    to the requested channels, regardless of whether or not the default channel count is requested.
    
    For hardware devices, I'm suspecting only a single channel count will be reported and we can safely use the
    value returned by ma_find_best_channels_from_sio_cap__sndio().
    */
    if (deviceType == ma_device_type_capture) {
        if (pDevice->capture.usingDefaultFormat) {
            format = ma_find_best_format_from_sio_cap__sndio(&caps);
        }
        if (pDevice->capture.usingDefaultChannels) {
            if (strlen(pDeviceName) > strlen("rsnd/") && strncmp(pDeviceName, "rsnd/", strlen("rsnd/")) == 0) {
                channels = ma_find_best_channels_from_sio_cap__sndio(&caps, deviceType, format);
            }
        }
    } else {
        if (pDevice->playback.usingDefaultFormat) {
            format = ma_find_best_format_from_sio_cap__sndio(&caps);
        }
        if (pDevice->playback.usingDefaultChannels) {
            if (strlen(pDeviceName) > strlen("rsnd/") && strncmp(pDeviceName, "rsnd/", strlen("rsnd/")) == 0) {
                channels = ma_find_best_channels_from_sio_cap__sndio(&caps, deviceType, format);
            }
        }
    }
    
    if (pDevice->usingDefaultSampleRate) {
        sampleRate = ma_find_best_sample_rate_from_sio_cap__sndio(&caps, pConfig->deviceType, format, channels);
    }


    ((ma_sio_initpar_proc)pDevice->pContext->sndio.sio_initpar)(&par);
    par.msb = 0;
    par.le  = ma_is_little_endian();
    
    switch (format) {
        case ma_format_u8:
        {
            par.bits = 8;
            par.bps  = 1;
            par.sig  = 0;
        } break;
        
        case ma_format_s24:
        {
            par.bits = 24;
            par.bps  = 3;
            par.sig  = 1;
        } break;
        
        case ma_format_s32:
        {
            par.bits = 32;
            par.bps  = 4;
            par.sig  = 1;
        } break;
        
        case ma_format_s16:
        case ma_format_f32:
        default:
        {
            par.bits = 16;
            par.bps  = 2;
            par.sig  = 1;
        } break;
    }
    
    if (deviceType == ma_device_type_capture) {
        par.rchan = channels;
    } else {
        par.pchan = channels;
    }

    par.rate = sampleRate;

    internalPeriodSizeInFrames = pConfig->periodSizeInFrames;
    if (internalPeriodSizeInFrames == 0) {
        internalPeriodSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(pConfig->periodSizeInMilliseconds, par.rate);
    }

    par.round    = internalPeriodSizeInFrames;
    par.appbufsz = par.round * pConfig->periods;
    
    if (((ma_sio_setpar_proc)pContext->sndio.sio_setpar)((struct ma_sio_hdl*)handle, &par) == 0) {
        ((ma_sio_close_proc)pContext->sndio.sio_close)((struct ma_sio_hdl*)handle);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to set buffer size.", MA_FORMAT_NOT_SUPPORTED);
    }
    if (((ma_sio_getpar_proc)pContext->sndio.sio_getpar)((struct ma_sio_hdl*)handle, &par) == 0) {
        ((ma_sio_close_proc)pContext->sndio.sio_close)((struct ma_sio_hdl*)handle);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to retrieve buffer size.", MA_FORMAT_NOT_SUPPORTED);
    }

    internalFormat             = ma_format_from_sio_enc__sndio(par.bits, par.bps, par.sig, par.le, par.msb);
    internalChannels           = (deviceType == ma_device_type_capture) ? par.rchan : par.pchan;
    internalSampleRate         = par.rate;
    internalPeriods            = par.appbufsz / par.round;
    internalPeriodSizeInFrames = par.round;

    if (deviceType == ma_device_type_capture) {
        pDevice->sndio.handleCapture                 = handle;
        pDevice->capture.internalFormat              = internalFormat;
        pDevice->capture.internalChannels            = internalChannels;
        pDevice->capture.internalSampleRate          = internalSampleRate;
        ma_get_standard_channel_map(ma_standard_channel_map_sndio, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);
        pDevice->capture.internalPeriodSizeInFrames  = internalPeriodSizeInFrames;
        pDevice->capture.internalPeriods             = internalPeriods;
    } else {
        pDevice->sndio.handlePlayback                = handle;
        pDevice->playback.internalFormat             = internalFormat;
        pDevice->playback.internalChannels           = internalChannels;
        pDevice->playback.internalSampleRate         = internalSampleRate;
        ma_get_standard_channel_map(ma_standard_channel_map_sndio, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);
        pDevice->playback.internalPeriodSizeInFrames = internalPeriodSizeInFrames;
        pDevice->playback.internalPeriods            = internalPeriods;
    }

#ifdef MA_DEBUG_OUTPUT
    printf("DEVICE INFO\n");
    printf("    Format:      %s\n", ma_get_format_name(internalFormat));
    printf("    Channels:    %d\n", internalChannels);
    printf("    Sample Rate: %d\n", internalSampleRate);
    printf("    Period Size: %d\n", internalPeriodSizeInFrames);
    printf("    Periods:     %d\n", internalPeriods);
    printf("    appbufsz:    %d\n", par.appbufsz);
    printf("    round:       %d\n", par.round);
#endif

    return MA_SUCCESS;
}

static ma_result ma_device_init__sndio(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    MA_ZERO_OBJECT(&pDevice->sndio);

    if (pConfig->deviceType == ma_device_type_loopback) {
        return MA_DEVICE_TYPE_NOT_SUPPORTED;
    }

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_handle__sndio(pContext, pConfig, ma_device_type_capture, pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_handle__sndio(pContext, pConfig, ma_device_type_playback, pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    return MA_SUCCESS;
}

static ma_result ma_device_stop__sndio(ma_device* pDevice)
{
    MA_ASSERT(pDevice != NULL);

    /*
    From the documentation:

        The sio_stop() function puts the audio subsystem in the same state as before sio_start() is called. It stops recording, drains the play buffer and then
        stops playback. If samples to play are queued but playback hasn't started yet then playback is forced immediately; playback will actually stop once the
        buffer is drained. In no case are samples in the play buffer discarded.

    Therefore, sio_stop() performs all of the necessary draining for us.
    */

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_sio_stop_proc)pDevice->pContext->sndio.sio_stop)((struct ma_sio_hdl*)pDevice->sndio.handleCapture);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ((ma_sio_stop_proc)pDevice->pContext->sndio.sio_stop)((struct ma_sio_hdl*)pDevice->sndio.handlePlayback);
    }

    return MA_SUCCESS;
}

static ma_result ma_device_write__sndio(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount, ma_uint32* pFramesWritten)
{
    int result;

    if (pFramesWritten != NULL) {
        *pFramesWritten = 0;
    }

    result = ((ma_sio_write_proc)pDevice->pContext->sndio.sio_write)((struct ma_sio_hdl*)pDevice->sndio.handlePlayback, pPCMFrames, frameCount * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));
    if (result == 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to send data from the client to the device.", MA_IO_ERROR);
    }

    if (pFramesWritten != NULL) {
        *pFramesWritten = frameCount;
    }
    
    return MA_SUCCESS;
}

static ma_result ma_device_read__sndio(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount, ma_uint32* pFramesRead)
{
    int result;

    if (pFramesRead != NULL) {
        *pFramesRead = 0;
    }

    result = ((ma_sio_read_proc)pDevice->pContext->sndio.sio_read)((struct ma_sio_hdl*)pDevice->sndio.handleCapture, pPCMFrames, frameCount * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));
    if (result == 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to read data from the device to be sent to the device.", MA_IO_ERROR);
    }

    if (pFramesRead != NULL) {
        *pFramesRead = frameCount;
    }
    
    return MA_SUCCESS;
}

static ma_result ma_device_main_loop__sndio(ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;
    ma_bool32 exitLoop = MA_FALSE;

    /* Devices need to be started here. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_sio_start_proc)pDevice->pContext->sndio.sio_start)((struct ma_sio_hdl*)pDevice->sndio.handleCapture);
    }
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ((ma_sio_start_proc)pDevice->pContext->sndio.sio_start)((struct ma_sio_hdl*)pDevice->sndio.handlePlayback);   /* <-- Doesn't actually playback until data is written. */
    }

    while (ma_device__get_state(pDevice) == MA_STATE_STARTED && !exitLoop) {
        switch (pDevice->type)
        {
            case ma_device_type_duplex:
            {
                /* The process is: device_read -> convert -> callback -> convert -> device_write */
                ma_uint32 totalCapturedDeviceFramesProcessed = 0;
                ma_uint32 capturedDevicePeriodSizeInFrames = ma_min(pDevice->capture.internalPeriodSizeInFrames, pDevice->playback.internalPeriodSizeInFrames);
                    
                while (totalCapturedDeviceFramesProcessed < capturedDevicePeriodSizeInFrames) {
                    ma_uint8  capturedDeviceData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                    ma_uint8  playbackDeviceData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                    ma_uint32 capturedDeviceDataCapInFrames = sizeof(capturedDeviceData) / ma_get_bytes_per_frame(pDevice->capture.internalFormat,  pDevice->capture.internalChannels);
                    ma_uint32 playbackDeviceDataCapInFrames = sizeof(playbackDeviceData) / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                    ma_uint32 capturedDeviceFramesRemaining;
                    ma_uint32 capturedDeviceFramesProcessed;
                    ma_uint32 capturedDeviceFramesToProcess;
                    ma_uint32 capturedDeviceFramesToTryProcessing = capturedDevicePeriodSizeInFrames - totalCapturedDeviceFramesProcessed;
                    if (capturedDeviceFramesToTryProcessing > capturedDeviceDataCapInFrames) {
                        capturedDeviceFramesToTryProcessing = capturedDeviceDataCapInFrames;
                    }

                    result = ma_device_read__sndio(pDevice, capturedDeviceData, capturedDeviceFramesToTryProcessing, &capturedDeviceFramesToProcess);
                    if (result != MA_SUCCESS) {
                        exitLoop = MA_TRUE;
                        break;
                    }

                    capturedDeviceFramesRemaining = capturedDeviceFramesToProcess;
                    capturedDeviceFramesProcessed = 0;

                    for (;;) {
                        ma_uint8  capturedClientData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                        ma_uint8  playbackClientData[MA_DATA_CONVERTER_STACK_BUFFER_SIZE];
                        ma_uint32 capturedClientDataCapInFrames = sizeof(capturedClientData) / ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
                        ma_uint32 playbackClientDataCapInFrames = sizeof(playbackClientData) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
                        ma_uint64 capturedClientFramesToProcessThisIteration = ma_min(capturedClientDataCapInFrames, playbackClientDataCapInFrames);
                        ma_uint64 capturedDeviceFramesToProcessThisIteration = capturedDeviceFramesRemaining;
                        ma_uint8* pRunningCapturedDeviceFrames = ma_offset_ptr(capturedDeviceData, capturedDeviceFramesProcessed * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));

                        /* Convert capture data from device format to client format. */
                        result = ma_data_converter_process_pcm_frames(&pDevice->capture.converter, pRunningCapturedDeviceFrames, &capturedDeviceFramesToProcessThisIteration, capturedClientData, &capturedClientFramesToProcessThisIteration);
                        if (result != MA_SUCCESS) {
                            break;
                        }

                        /*
                        If we weren't able to generate any output frames it must mean we've exhaused all of our input. The only time this would not be the case is if capturedClientData was too small
                        which should never be the case when it's of the size MA_DATA_CONVERTER_STACK_BUFFER_SIZE.
                        */
                        if (capturedClientFramesToProcessThisIteration == 0) {
                            break;
                        }

                        ma_device__on_data(pDevice, playbackClientData, capturedClientData, (ma_uint32)capturedClientFramesToProcessThisIteration);    /* Safe cast .*/

                        capturedDeviceFramesProcessed += (ma_uint32)capturedDeviceFramesToProcessThisIteration; /* Safe cast. */
                        capturedDeviceFramesRemaining -= (ma_uint32)capturedDeviceFramesToProcessThisIteration; /* Safe cast. */

                        /* At this point the playbackClientData buffer should be holding data that needs to be written to the device. */
                        for (;;) {
                            ma_uint64 convertedClientFrameCount = capturedClientFramesToProcessThisIteration;
                            ma_uint64 convertedDeviceFrameCount = playbackDeviceDataCapInFrames;
                            result = ma_data_converter_process_pcm_frames(&pDevice->playback.converter, playbackClientData, &convertedClientFrameCount, playbackDeviceData, &convertedDeviceFrameCount);
                            if (result != MA_SUCCESS) {
                                break;
                            }

                            result = ma_device_write__sndio(pDevice, playbackDeviceData, (ma_uint32)convertedDeviceFrameCount, NULL);   /* Safe cast. */
                            if (result != MA_SUCCESS) {
                                exitLoop = MA_TRUE;
                                break