/*
Audio playback and capture library. Choice of public domain or MIT-0. See license statements at the end of this file.
miniaudio (formerly mini_al) - v0.9.4 - 2019-05-xx

David Reid - davidreidsoftware@gmail.com
*/

/*
MAJOR CHANGES IN VERSION 0.9
============================
Version 0.9 includes major API changes, centered mostly around full-duplex and the rebrand to "miniaudio". Before I go into
detail about the major changes I would like to apologize. I know it's annoying dealing with breaking API changes, but I think
it's best to get these changes out of the way now while the library is still relatively young and unknown.

There's been a lot of refactoring with this release so there's a good chance a few bugs have been introduced. I apologize in
advance for this. You may want to hold off on upgrading for the short term if you're worried. If mini_al v0.8.14 works for
you, and you don't need full-duplex support, you can avoid upgrading (though you won't be getting future bug fixes).


Rebranding to "miniaudio"
-------------------------
The decision was made to rename mini_al to miniaudio. Don't worry, it's the same project. The reason for this is simple:

1) Having the word "audio" in the title makes it immediately clear that the library is related to audio; and
2) I don't like the look of the underscore.

This rebrand has necessitated a change in namespace from "mal" to "ma". I know this is annoying, and I apologize, but it's
better to get this out of the road now rather than later. Also, since there are necessary API changes for full-duplex support
I think it's better to just get the namespace change over and done with at the same time as the full-duplex changes. I'm hoping
this will be the last of the major API changes. Fingers crossed!

The implementation define is now "#define MINIAUDIO_IMPLEMENTATION". You can also use "#define MA_IMPLEMENTATION" if that's
your preference.


Full-Duplex Support
-------------------
The major feature added to version 0.9 is full-duplex. This has necessitated a few API changes.

1) The data callback has now changed. Previously there was one type of callback for playback and another for capture. I wanted
   to avoid a third callback just for full-duplex so the decision was made to break this API and unify the callbacks. Now,
   there is just one callback which is the same for all three modes (playback, capture, duplex). The new callback looks like
   the following:

       void data_callback(ma_device* pDevice, void* pOutput, const void* pInput, ma_uint32 frameCount);

   This callback allows you to move data straight out of the input buffer and into the output buffer in full-duplex mode. In
   playback-only mode, pInput will be null. Likewise, pOutput will be null in capture-only mode. The sample count is no longer
   returned from the callback since it's not necessary for miniaudio anymore.

2) The device config needed to change in order to support full-duplex. Full-duplex requires the ability to allow the client
   to choose a different PCM format for the playback and capture sides. The old ma_device_config object simply did not allow
   this and needed to change. With these changes you now specify the device ID, format, channels, channel map and share mode
   on a per-playback and per-capture basis (see example below). The sample rate must be the same for playback and capture.

   Since the device config API has changed I have also decided to take the opportunity to simplify device initialization. Now,
   the device ID, device type and callback user data are set in the config. ma_device_init() is now simplified down to taking
   just the context, device config and a pointer to the device object being initialized. The rationale for this change is that
   it just makes more sense to me that these are set as part of the config like everything else.

   Example device initialization:

       ma_device_config config = ma_device_config_init(ma_device_type_duplex);   // Or ma_device_type_playback or ma_device_type_capture.
       config.playback.pDeviceID = &myPlaybackDeviceID; // Or NULL for the default playback device.
       config.playback.format    = ma_format_f32;
       config.playback.channels  = 2;
       config.capture.pDeviceID  = &myCaptureDeviceID;  // Or NULL for the default capture device.
       config.capture.format     = ma_format_s16;
       config.capture.channels   = 1;
       config.sampleRate         = 44100;
       config.dataCallback       = data_callback;
       config.pUserData          = &myUserData;

       result = ma_device_init(&myContext, &config, &device);
       if (result != MA_SUCCESS) {
           ... handle error ...
       }

   Note that the "onDataCallback" member of ma_device_config has been renamed to "dataCallback". Also, "onStopCallback" has
   been renamed to "stopCallback".

This is the first pass for full-duplex and there is a known bug. You will hear crackling on the following backends when sample
rate conversion is required for the playback device:
  - Core Audio
  - JACK
  - AAudio
  - OpenSL
  - WebAudio

In addition to the above, not all platforms have been absolutely thoroughly tested simply because I lack the hardware for such
thorough testing. If you experience a bug, an issue report on GitHub or an email would be greatly appreciated (and a sample
program that reproduces the issue if possible).


Other API Changes
-----------------
In addition to the above, the following API changes have been made:

- The log callback is no longer passed to ma_context_config_init(). Instead you need to set it manually after initialization.
- The onLogCallback member of ma_context_config has been renamed to "logCallback".
- The log callback now takes a logLevel parameter. The new callback looks like: void log_callback(ma_context* pContext, ma_device* pDevice, ma_uint32 logLevel, const char* message)
  - You can use ma_log_level_to_string() to convert the logLevel to human readable text if you want to log it.
- Some APIs have been renamed:
  - mal_decoder_read()          -> ma_decoder_read_pcm_frames()
  - mal_decoder_seek_to_frame() -> ma_decoder_seek_to_pcm_frame()
  - mal_sine_wave_read()        -> ma_sine_wave_read_f32()
  - mal_sine_wave_read_ex()     -> ma_sine_wave_read_f32_ex()
- Some APIs have been removed:
  - mal_device_get_buffer_size_in_bytes()
  - mal_device_set_recv_callback()
  - mal_device_set_send_callback()
  - mal_src_set_input_sample_rate()
  - mal_src_set_output_sample_rate()
- Error codes have been rearranged. If you're a binding maintainer you will need to update.
- The ma_backend enums have been rearranged to priority order. The rationale for this is to simplify automatic backend selection
  and to make it easier to see the priority. If you're a binding maintainer you will need to update.
- ma_dsp has been renamed to ma_pcm_converter. The rationale for this change is that I'm expecting "ma_dsp" to conflict with
  some future planned high-level APIs.
- For functions that take a pointer/count combo, such as ma_decoder_read_pcm_frames(), the parameter order has changed so that
  the pointer comes before the count. The rationale for this is to keep it consistent with things like memcpy().


Miscellaneous Changes
---------------------
The following miscellaneous changes have also been made.

- The AAudio backend has been added for Android 8 and above. This is Android's new "High-Performance Audio" API. (For the
  record, this is one of the nicest audio APIs out there, just behind the BSD audio APIs).
- The WebAudio backend has been added. This is based on ScriptProcessorNode. This removes the need for SDL.
- The SDL and OpenAL backends have been removed. These were originally implemented to add support for platforms for which miniaudio
  was not explicitly supported. These are no longer needed and have therefore been removed.
- Device initialization now fails if the requested share mode is not supported. If you ask for exclusive mode, you either get an
  exclusive mode device, or an error. The rationale for this change is to give the client more control over how to handle cases
  when the desired shared mode is unavailable.
- A lock-free ring buffer API has been added. There are two varients of this. "ma_rb" operates on bytes, whereas "ma_pcm_rb"
  operates on PCM frames.
- The library is now licensed as a choice of Public Domain (Unlicense) _or_ MIT-0 (No Attribution) which is the same as MIT, but
  removes the attribution requirement. The rationale for this is to support countries that don't recognize public domain.
*/

/*
ABOUT
=====
miniaudio is a single file library for audio playback and capture. It's written in C (compilable as
C++) and released into the public domain.

Supported Backends:
  - WASAPI
  - DirectSound
  - WinMM
  - Core Audio (Apple)
  - ALSA
  - PulseAudio
  - JACK
  - sndio (OpenBSD)
  - audio(4) (NetBSD and OpenBSD)
  - OSS (FreeBSD)
  - AAudio (Android 8.0+)
  - OpenSL|ES (Android only)
  - Web Audio (Emscripten)
  - Null (Silence)

Supported Formats:
  - Unsigned 8-bit PCM
  - Signed 16-bit PCM
  - Signed 24-bit PCM (tightly packed)
  - Signed 32-bit PCM
  - IEEE 32-bit floating point PCM


USAGE
=====
miniaudio is a single-file library. To use it, do something like the following in one .c file.
  #define MINIAUDIO_IMPLEMENTATION
  #include "miniaudio.h"

You can then #include this file in other parts of the program as you would with any other header file.

miniaudio uses an asynchronous, callback based API. You initialize a device with a configuration (sample rate,
channel count, etc.) which includes the callback you want to use to handle data transmission to/from the
device. In the callback you either read from a data pointer in the case of playback or write to it in the case
of capture.

Playback Example
----------------
  void data_callback(ma_device* pDevice, void* pOutput, const void* pInput, ma_uint32 frameCount)
  {
      ma_decoder* pDecoder = (ma_decoder*)pDevice->pUserData;
      if (pDecoder == NULL) {
          return;
      }
  
      ma_decoder_read_pcm_frames(pDecoder, frameCount, pOutput);
  }

  ...

  ma_device_config config = ma_device_config_init(ma_device_type_playback);
  config.playback.format   = decoder.outputFormat;
  config.playback.channels = decoder.outputChannels;
  config.sampleRate        = decoder.outputSampleRate;
  config.dataCallback      = data_callback;
  config.pUserData         = &decoder;

  ma_device device;
  if (ma_device_init(NULL, &config, &device) != MA_SUCCESS) {
      ... An error occurred ...
  }

  ma_device_start(&device);     // The device is sleeping by default so you'll need to start it manually.

  ...

  ma_device_uninit(&device);    // This will stop the device so no need to do that manually.


BUILDING
========
miniaudio should Just Work by adding it to your project's source tree. You do not need to download or install
any dependencies. See below for platform-specific details.

If you want to disable a specific backend, #define the appropriate MA_NO_* option before the implementation.

Note that GCC and Clang requires "-msse2", "-mavx2", etc. for SIMD optimizations.


Building for Windows
--------------------
The Windows build should compile clean on all popular compilers without the need to configure any include paths
nor link to any libraries.

Building for macOS and iOS
--------------------------
The macOS build should compile clean without the need to download any dependencies or link to any libraries or
frameworks. The iOS build needs to be compiled as Objective-C (sorry) and will need to link the relevant frameworks
but should Just Work with Xcode.

Building for Linux
------------------
The Linux build only requires linking to -ldl, -lpthread and -lm. You do not need any development packages.

Building for BSD
----------------
The BSD build only requires linking to -ldl, -lpthread and -lm. NetBSD uses audio(4), OpenBSD uses sndio and
FreeBSD uses OSS.

Building for Android
--------------------
AAudio is the highest priority backend on Android. This should work out out of the box without needing any kind of
compiler configuration. Support for AAudio starts with Android 8 which means older versions will fall back to
OpenSL|ES which requires API level 16+.

Building for Emscripten
-----------------------
The Emscripten build emits Web Audio JavaScript directly and should Just Work without any configuration.


NOTES
=====
- This library uses an asynchronous API for delivering and requesting audio data. Each device will have
  it's own worker thread which is managed by the library.
- If ma_device_init() is called with a device that's not aligned to the platform's natural alignment
  boundary (4 bytes on 32-bit, 8 bytes on 64-bit), it will _not_ be thread-safe. The reason for this
  is that it depends on members of ma_device being correctly aligned for atomic assignments.
- Sample data is always native-endian and interleaved. For example, ma_format_s16 means signed 16-bit
  integer samples, interleaved. Let me know if you need non-interleaved and I'll look into it.
- The sndio backend is currently only enabled on OpenBSD builds.
- The audio(4) backend is supported on OpenBSD, but you may need to disable sndiod before you can use it.
- Automatic stream routing is enabled on a per-backend basis. Support is explicitly enabled for WASAPI
  and Core Audio, however other backends such as PulseAudio may naturally support it, though not all have
  been tested.


BACKEND NUANCES
===============

PulseAudio
----------
- If you experience bad glitching/noise on Arch Linux, consider this fix from the Arch wiki:
    https://wiki.archlinux.org/index.php/PulseAudio/Troubleshooting#Glitches,_skips_or_crackling
  Alternatively, consider using a different backend such as ALSA.

Android
-------
- To capture audio on Android, remember to add the RECORD_AUDIO permission to your manifest:
    <uses-permission android:name="android.permission.RECORD_AUDIO" />
- With OpenSL|ES, only a single ma_context can be active at any given time. This is due to a limitation with OpenSL|ES.
- With AAudio, only default devices are enumerated. This is due to AAudio not having an enumeration API (devices are
  enumerated through Java). You can however perform your own device enumeration through Java and then set the ID in the
  ma_device_id structure (ma_device_id.aaudio) and pass it to ma_device_init().
- The backend API will perform resampling where possible. The reason for this as opposed to using miniaudio's built-in
  resampler is to take advantage of any potential device-specific optimizations the driver may implement.

UWP
---
- UWP only supports default playback and capture devices.
- UWP requires the Microphone capability to be enabled in the application's manifest (Package.appxmanifest):
      <Package ...>
          ...
          <Capabilities>
              <DeviceCapability Name="microphone" />
          </Capabilities>
      </Package>

Web Audio / Emscripten
----------------------
- The first time a context is initialized it will create a global object called "mal" whose primary purpose is to act
  as a factory for device objects.
- Currently the Web Audio backend uses ScriptProcessorNode's, but this may need to change later as they've been deprecated.
- Google is implementing a policy in their browsers that prevent automatic media output without first receiving some kind
  of user input. See here for details: https://developers.google.com/web/updates/2017/09/autoplay-policy-changes. Starting
  the device may fail if you try to start playback without first handling some kind of user input.


OPTIONS
=======
#define these options before including this file.

#define MA_NO_WASAPI
  Disables the WASAPI backend.

#define MA_NO_DSOUND
  Disables the DirectSound backend.

#define MA_NO_WINMM
  Disables the WinMM backend.

#define MA_NO_ALSA
  Disables the ALSA backend.

#define MA_NO_PULSEAUDIO
  Disables the PulseAudio backend.

#define MA_NO_JACK
  Disables the JACK backend.

#define MA_NO_COREAUDIO
  Disables the Core Audio backend.

#define MA_NO_SNDIO
  Disables the sndio backend.

#define MA_NO_AUDIO4
  Disables the audio(4) backend.

#define MA_NO_OSS
  Disables the OSS backend.

#define MA_NO_AAUDIO
  Disables the AAudio backend.

#define MA_NO_OPENSL
  Disables the OpenSL|ES backend.

#define MA_NO_WEBAUDIO
  Disables the Web Audio backend.

#define MA_NO_NULL
  Disables the null backend.

#define MA_DEFAULT_PERIODS
  When a period count of 0 is specified when a device is initialized, it will default to this.

#define MA_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY
#define MA_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE
  When a buffer size of 0 is specified when a device is initialized it will default to a buffer of this size, depending
  on the chosen performance profile. These can be increased or decreased depending on your specific requirements.

#define MA_NO_DECODING
  Disables the decoding APIs.

#define MA_NO_DEVICE_IO
  Disables playback and recording. This will disable ma_context and ma_device APIs. This is useful if you only want to
  use miniaudio's data conversion and/or decoding APIs. 

#define MA_NO_STDIO
  Disables file IO APIs.

#define MA_NO_SSE2
  Disables SSE2 optimizations.

#define MA_NO_AVX2
  Disables AVX2 optimizations.

#define MA_NO_AVX512
  Disables AVX-512 optimizations.

#define MA_NO_NEON
  Disables NEON optimizations.

#define MA_LOG_LEVEL <Level>
  Sets the logging level. Set level to one of the following:
    MA_LOG_LEVEL_VERBOSE
    MA_LOG_LEVEL_INFO
    MA_LOG_LEVEL_WARNING
    MA_LOG_LEVEL_ERROR

#define MA_DEBUT_OUTPUT
  Enable printf() debug output.

#define MA_COINIT_VALUE
  Windows only. The value to pass to internal calls to CoInitializeEx(). Defaults to COINIT_MULTITHREADED.


DEFINITIONS
===========
This section defines common terms used throughout miniaudio. Unfortunately there is often ambiguity in the use of terms
throughout the audio space, so this section is intended to clarify how miniaudio uses each term.

Sample
------
A sample is a single unit of audio data. If the sample format is f32, then one sample is one 32-bit floating point number.

Frame / PCM Frame
-----------------
A frame is a groups of samples equal to the number of channels. For a stereo stream a frame is 2 samples, a mono frame
is 1 sample, a 5.1 surround sound frame is 6 samples, etc. The terms "frame" and "PCM frame" are the same thing in
miniaudio. Note that this is different to a compressed frame. If ever miniaudio needs to refer to a compressed frame, such
as a FLAC frame, it will always clarify what it's referring to with something like "FLAC frame" or whatnot.

Channel
-------
A stream of monaural audio that is emitted from an individual speaker in a speaker system, or received from an individual
microphone in a microphone system. A stereo stream has two channels (a left channel, and a right channel), a 5.1 surround
sound system has 6 channels, etc. Some audio systems refer to a channel as a complex audio stream that's mixed with other
channels to produce the final mix - this is completely different to miniaudio's use of the term "channel" and should not be
confused.

Sample Rate
-----------
The sample rate in miniaudio is always expressed in Hz, such as 44100, 48000, etc. It's the number of PCM frames that are
processed per second.

Formats
-------
Throughout miniaudio you will see references to different sample formats:
    u8  - Unsigned 8-bit integer
    s16 - Signed 16-bit integer
    s24 - Signed 24-bit integer (tightly packed).
    s32 - Signed 32-bit integer
    f32 - 32-bit floating point
*/

#ifndef miniaudio_h
#define miniaudio_h

#ifdef __cplusplus
extern "C" {
#endif

#if defined(_MSC_VER)
    #pragma warning(push)
    #pragma warning(disable:4201)   /* nonstandard extension used: nameless struct/union */
    #pragma warning(disable:4324)   /* structure was padded due to alignment specifier */
#else
    #pragma GCC diagnostic push
    #pragma GCC diagnostic ignored "-Wpedantic" /* For ISO C99 doesn't support unnamed structs/unions [-Wpedantic] */
#endif

/* Platform/backend detection. */
#ifdef _WIN32
    #define MA_WIN32
    #if defined(WINAPI_FAMILY) && (WINAPI_FAMILY == WINAPI_FAMILY_PC_APP || WINAPI_FAMILY == WINAPI_FAMILY_PHONE_APP)
        #define MA_WIN32_UWP
    #else
        #define MA_WIN32_DESKTOP
    #endif
#else
    #define MA_POSIX
    #include <pthread.h>    /* Unfortunate #include, but needed for pthread_t, pthread_mutex_t and pthread_cond_t types. */

    #ifdef __unix__
        #define MA_UNIX
        #if defined(__DragonFly__) || defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
            #define MA_BSD
        #endif
    #endif
    #ifdef __linux__
        #define MA_LINUX
    #endif
    #ifdef __APPLE__
        #define MA_APPLE
    #endif
    #ifdef __ANDROID__
        #define MA_ANDROID
    #endif
    #ifdef __EMSCRIPTEN__
        #define MA_EMSCRIPTEN
    #endif
#endif

#include <stddef.h> /* For size_t. */

/* Sized types. Prefer built-in types. Fall back to stdint. */
#ifdef _MSC_VER
    #if defined(__clang__)
        #pragma GCC diagnostic push
        #pragma GCC diagnostic ignored "-Wlanguage-extension-token"
        #pragma GCC diagnostic ignored "-Wc++11-long-long"
    #endif
    typedef   signed __int8  ma_int8;
    typedef unsigned __int8  ma_uint8;
    typedef   signed __int16 ma_int16;
    typedef unsigned __int16 ma_uint16;
    typedef   signed __int32 ma_int32;
    typedef unsigned __int32 ma_uint32;
    typedef   signed __int64 ma_int64;
    typedef unsigned __int64 ma_uint64;
    #if defined(__clang__)
        #pragma GCC diagnostic pop
    #endif
#else
    #define MA_HAS_STDINT
    #include <stdint.h>
    typedef int8_t   ma_int8;
    typedef uint8_t  ma_uint8;
    typedef int16_t  ma_int16;
    typedef uint16_t ma_uint16;
    typedef int32_t  ma_int32;
    typedef uint32_t ma_uint32;
    typedef int64_t  ma_int64;
    typedef uint64_t ma_uint64;
#endif

#ifdef MA_HAS_STDINT
    typedef uintptr_t ma_uintptr;
#else
    #if defined(_WIN32)
        #if defined(_WIN64)
            typedef ma_uint64 ma_uintptr;
        #else
            typedef ma_uint32 ma_uintptr;
        #endif
    #elif defined(__GNUC__)
        #if defined(__LP64__)
            typedef ma_uint64 ma_uintptr;
        #else
            typedef ma_uint32 ma_uintptr;
        #endif
    #else
        typedef ma_uint64 ma_uintptr;   /* Fallback. */
    #endif
#endif

typedef ma_uint8                   ma_bool8;
typedef ma_uint32                  ma_bool32;
#define MA_TRUE                    1
#define MA_FALSE                   0

typedef void* ma_handle;
typedef void* ma_ptr;
typedef void (* ma_proc)(void);

#if defined(_MSC_VER) && !defined(_WCHAR_T_DEFINED)
typedef ma_uint16 wchar_t;
#endif

/* Define NULL for some compilers. */
#ifndef NULL
#define NULL 0
#endif

#if defined(SIZE_MAX)
    #define MA_SIZE_MAX    SIZE_MAX
#else
    #define MA_SIZE_MAX    0xFFFFFFFF  /* When SIZE_MAX is not defined by the standard library just default to the maximum 32-bit unsigned integer. */
#endif


#ifdef _MSC_VER
#define MA_INLINE __forceinline
#else
#ifdef __GNUC__
#define MA_INLINE __inline__ __attribute__((always_inline))
#else
#define MA_INLINE
#endif
#endif

#ifdef _MSC_VER
#define MA_ALIGN(alignment) __declspec(align(alignment))
#elif !defined(__DMC__)
#define MA_ALIGN(alignment) __attribute__((aligned(alignment)))
#else
#define MA_ALIGN(alignment)
#endif

#ifdef _MSC_VER
#define MA_ALIGNED_STRUCT(alignment) MA_ALIGN(alignment) struct
#else
#define MA_ALIGNED_STRUCT(alignment) struct MA_ALIGN(alignment)
#endif

/* SIMD alignment in bytes. Currently set to 64 bytes in preparation for future AVX-512 optimizations. */
#define MA_SIMD_ALIGNMENT  64


/* Logging levels */
#define MA_LOG_LEVEL_VERBOSE   4
#define MA_LOG_LEVEL_INFO      3
#define MA_LOG_LEVEL_WARNING   2
#define MA_LOG_LEVEL_ERROR     1

#ifndef MA_LOG_LEVEL
#define MA_LOG_LEVEL           MA_LOG_LEVEL_ERROR
#endif

typedef struct ma_context ma_context;
typedef struct ma_device ma_device;

typedef ma_uint8 ma_channel;
#define MA_CHANNEL_NONE                                0
#define MA_CHANNEL_MONO                                1
#define MA_CHANNEL_FRONT_LEFT                          2
#define MA_CHANNEL_FRONT_RIGHT                         3
#define MA_CHANNEL_FRONT_CENTER                        4
#define MA_CHANNEL_LFE                                 5
#define MA_CHANNEL_BACK_LEFT                           6
#define MA_CHANNEL_BACK_RIGHT                          7
#define MA_CHANNEL_FRONT_LEFT_CENTER                   8
#define MA_CHANNEL_FRONT_RIGHT_CENTER                  9
#define MA_CHANNEL_BACK_CENTER                         10
#define MA_CHANNEL_SIDE_LEFT                           11
#define MA_CHANNEL_SIDE_RIGHT                          12
#define MA_CHANNEL_TOP_CENTER                          13
#define MA_CHANNEL_TOP_FRONT_LEFT                      14
#define MA_CHANNEL_TOP_FRONT_CENTER                    15
#define MA_CHANNEL_TOP_FRONT_RIGHT                     16
#define MA_CHANNEL_TOP_BACK_LEFT                       17
#define MA_CHANNEL_TOP_BACK_CENTER                     18
#define MA_CHANNEL_TOP_BACK_RIGHT                      19
#define MA_CHANNEL_AUX_0                               20
#define MA_CHANNEL_AUX_1                               21
#define MA_CHANNEL_AUX_2                               22
#define MA_CHANNEL_AUX_3                               23
#define MA_CHANNEL_AUX_4                               24
#define MA_CHANNEL_AUX_5                               25
#define MA_CHANNEL_AUX_6                               26
#define MA_CHANNEL_AUX_7                               27
#define MA_CHANNEL_AUX_8                               28
#define MA_CHANNEL_AUX_9                               29
#define MA_CHANNEL_AUX_10                              30
#define MA_CHANNEL_AUX_11                              31
#define MA_CHANNEL_AUX_12                              32
#define MA_CHANNEL_AUX_13                              33
#define MA_CHANNEL_AUX_14                              34
#define MA_CHANNEL_AUX_15                              35
#define MA_CHANNEL_AUX_16                              36
#define MA_CHANNEL_AUX_17                              37
#define MA_CHANNEL_AUX_18                              38
#define MA_CHANNEL_AUX_19                              39
#define MA_CHANNEL_AUX_20                              40
#define MA_CHANNEL_AUX_21                              41
#define MA_CHANNEL_AUX_22                              42
#define MA_CHANNEL_AUX_23                              43
#define MA_CHANNEL_AUX_24                              44
#define MA_CHANNEL_AUX_25                              45
#define MA_CHANNEL_AUX_26                              46
#define MA_CHANNEL_AUX_27                              47
#define MA_CHANNEL_AUX_28                              48
#define MA_CHANNEL_AUX_29                              49
#define MA_CHANNEL_AUX_30                              50
#define MA_CHANNEL_AUX_31                              51
#define MA_CHANNEL_LEFT                                MA_CHANNEL_FRONT_LEFT
#define MA_CHANNEL_RIGHT                               MA_CHANNEL_FRONT_RIGHT
#define MA_CHANNEL_POSITION_COUNT                      MA_CHANNEL_AUX_31 + 1


typedef int ma_result;
#define MA_SUCCESS                                      0

/* General errors. */
#define MA_ERROR                                       -1      /* A generic error. */
#define MA_INVALID_ARGS                                -2
#define MA_INVALID_OPERATION                           -3
#define MA_OUT_OF_MEMORY                               -4
#define MA_ACCESS_DENIED                               -5
#define MA_TOO_LARGE                                   -6
#define MA_TIMEOUT                                     -7

/* General miniaudio-specific errors. */
#define MA_FORMAT_NOT_SUPPORTED                        -100
#define MA_DEVICE_TYPE_NOT_SUPPORTED                   -101
#define MA_SHARE_MODE_NOT_SUPPORTED                    -102
#define MA_NO_BACKEND                                  -103
#define MA_NO_DEVICE                                   -104
#define MA_API_NOT_FOUND                               -105
#define MA_INVALID_DEVICE_CONFIG                       -106

/* State errors. */
#define MA_DEVICE_BUSY                                 -200
#define MA_DEVICE_NOT_INITIALIZED                      -201
#define MA_DEVICE_NOT_STARTED                          -202
#define MA_DEVICE_UNAVAILABLE                          -203

/* Operation errors. */
#define MA_FAILED_TO_MAP_DEVICE_BUFFER                 -300
#define MA_FAILED_TO_UNMAP_DEVICE_BUFFER               -301
#define MA_FAILED_TO_INIT_BACKEND                      -302
#define MA_FAILED_TO_READ_DATA_FROM_CLIENT             -303
#define MA_FAILED_TO_READ_DATA_FROM_DEVICE             -304
#define MA_FAILED_TO_SEND_DATA_TO_CLIENT               -305
#define MA_FAILED_TO_SEND_DATA_TO_DEVICE               -306
#define MA_FAILED_TO_OPEN_BACKEND_DEVICE               -307
#define MA_FAILED_TO_START_BACKEND_DEVICE              -308
#define MA_FAILED_TO_STOP_BACKEND_DEVICE               -309
#define MA_FAILED_TO_CONFIGURE_BACKEND_DEVICE          -310
#define MA_FAILED_TO_CREATE_MUTEX                      -311
#define MA_FAILED_TO_CREATE_EVENT                      -312
#define MA_FAILED_TO_CREATE_THREAD                     -313


/* Standard sample rates. */
#define MA_SAMPLE_RATE_8000                            8000
#define MA_SAMPLE_RATE_11025                           11025
#define MA_SAMPLE_RATE_16000                           16000
#define MA_SAMPLE_RATE_22050                           22050
#define MA_SAMPLE_RATE_24000                           24000
#define MA_SAMPLE_RATE_32000                           32000
#define MA_SAMPLE_RATE_44100                           44100
#define MA_SAMPLE_RATE_48000                           48000
#define MA_SAMPLE_RATE_88200                           88200
#define MA_SAMPLE_RATE_96000                           96000
#define MA_SAMPLE_RATE_176400                          176400
#define MA_SAMPLE_RATE_192000                          192000
#define MA_SAMPLE_RATE_352800                          352800
#define MA_SAMPLE_RATE_384000                          384000

#define MA_MIN_PCM_SAMPLE_SIZE_IN_BYTES                1   /* For simplicity, miniaudio does not support PCM samples that are not byte aligned. */
#define MA_MAX_PCM_SAMPLE_SIZE_IN_BYTES                8
#define MA_MIN_CHANNELS                                1
#define MA_MAX_CHANNELS                                32
#define MA_MIN_SAMPLE_RATE                             MA_SAMPLE_RATE_8000
#define MA_MAX_SAMPLE_RATE                             MA_SAMPLE_RATE_384000
#define MA_SRC_SINC_MIN_WINDOW_WIDTH                   2
#define MA_SRC_SINC_MAX_WINDOW_WIDTH                   32
#define MA_SRC_SINC_DEFAULT_WINDOW_WIDTH               32
#define MA_SRC_SINC_LOOKUP_TABLE_RESOLUTION            8
#define MA_SRC_INPUT_BUFFER_SIZE_IN_SAMPLES            256

typedef enum
{
    ma_stream_format_pcm = 0,
} ma_stream_format;

typedef enum
{
    ma_stream_layout_interleaved = 0,
    ma_stream_layout_deinterleaved
} ma_stream_layout;

typedef enum
{
    ma_dither_mode_none = 0,
    ma_dither_mode_rectangle,
    ma_dither_mode_triangle
} ma_dither_mode;

typedef enum
{
    /*
    I like to keep these explicitly defined because they're used as a key into a lookup table. When items are
    added to this, make sure there are no gaps and that they're added to the lookup table in ma_get_bytes_per_sample().
    */
    ma_format_unknown = 0,     /* Mainly used for indicating an error, but also used as the default for the output format for decoders. */
    ma_format_u8      = 1,
    ma_format_s16     = 2,     /* Seems to be the most widely supported format. */
    ma_format_s24     = 3,     /* Tightly packed. 3 bytes per sample. */
    ma_format_s32     = 4,
    ma_format_f32     = 5,
    ma_format_count
} ma_format;

typedef enum
{
    ma_channel_mix_mode_rectangular = 0,   /* Simple averaging based on the plane(s) the channel is sitting on. */
    ma_channel_mix_mode_simple,            /* Drop excess channels; zeroed out extra channels. */
    ma_channel_mix_mode_custom_weights,    /* Use custom weights specified in ma_channel_router_config. */
    ma_channel_mix_mode_planar_blend = ma_channel_mix_mode_rectangular,
    ma_channel_mix_mode_default = ma_channel_mix_mode_planar_blend
} ma_channel_mix_mode;

typedef enum
{
    ma_standard_channel_map_microsoft,
    ma_standard_channel_map_alsa,
    ma_standard_channel_map_rfc3551,   /* Based off AIFF. */
    ma_standard_channel_map_flac,
    ma_standard_channel_map_vorbis,
    ma_standard_channel_map_sound4,    /* FreeBSD's sound(4). */
    ma_standard_channel_map_sndio,     /* www.sndio.org/tips.html */
    ma_standard_channel_map_webaudio = ma_standard_channel_map_flac, /* https://webaudio.github.io/web-audio-api/#ChannelOrdering. Only 1, 2, 4 and 6 channels are defined, but can fill in the gaps with logical assumptions. */
    ma_standard_channel_map_default = ma_standard_channel_map_microsoft
} ma_standard_channel_map;

typedef enum
{
    ma_performance_profile_low_latency = 0,
    ma_performance_profile_conservative
} ma_performance_profile;


typedef struct ma_format_converter ma_format_converter;
typedef ma_uint32 (* ma_format_converter_read_proc)              (ma_format_converter* pConverter, ma_uint32 frameCount, void* pFramesOut, void* pUserData);
typedef ma_uint32 (* ma_format_converter_read_deinterleaved_proc)(ma_format_converter* pConverter, ma_uint32 frameCount, void** ppSamplesOut, void* pUserData);

typedef struct
{
    ma_format formatIn;
    ma_format formatOut;
    ma_uint32 channels;
    ma_stream_format streamFormatIn;
    ma_stream_format streamFormatOut;
    ma_dither_mode ditherMode;
    ma_bool32 noSSE2   : 1;
    ma_bool32 noAVX2   : 1;
    ma_bool32 noAVX512 : 1;
    ma_bool32 noNEON   : 1;
    ma_format_converter_read_proc onRead;
    ma_format_converter_read_deinterleaved_proc onReadDeinterleaved;
    void* pUserData;
} ma_format_converter_config;

struct ma_format_converter
{
    ma_format_converter_config config;
    ma_bool32 useSSE2   : 1;
    ma_bool32 useAVX2   : 1;
    ma_bool32 useAVX512 : 1;
    ma_bool32 useNEON   : 1;
    void (* onConvertPCM)(void* dst, const void* src, ma_uint64 count, ma_dither_mode ditherMode);
    void (* onInterleavePCM)(void* dst, const void** src, ma_uint64 frameCount, ma_uint32 channels);
    void (* onDeinterleavePCM)(void** dst, const void* src, ma_uint64 frameCount, ma_uint32 channels);
};



typedef struct ma_channel_router ma_channel_router;
typedef ma_uint32 (* ma_channel_router_read_deinterleaved_proc)(ma_channel_router* pRouter, ma_uint32 frameCount, void** ppSamplesOut, void* pUserData);

typedef struct
{
    ma_uint32 channelsIn;
    ma_uint32 channelsOut;
    ma_channel channelMapIn[MA_MAX_CHANNELS];
    ma_channel channelMapOut[MA_MAX_CHANNELS];
    ma_channel_mix_mode mixingMode;
    float weights[MA_MAX_CHANNELS][MA_MAX_CHANNELS];  /* [in][out]. Only used when mixingMode is set to ma_channel_mix_mode_custom_weights. */
    ma_bool32 noSSE2   : 1;
    ma_bool32 noAVX2   : 1;
    ma_bool32 noAVX512 : 1;
    ma_bool32 noNEON   : 1;
    ma_channel_router_read_deinterleaved_proc onReadDeinterleaved;
    void* pUserData;
} ma_channel_router_config;

struct ma_channel_router
{
    ma_channel_router_config config;
    ma_bool32 isPassthrough   : 1;
    ma_bool32 isSimpleShuffle : 1;
    ma_bool32 useSSE2         : 1;
    ma_bool32 useAVX2         : 1;
    ma_bool32 useAVX512       : 1;
    ma_bool32 useNEON         : 1;
    ma_uint8 shuffleTable[MA_MAX_CHANNELS];
};



typedef struct ma_src ma_src;
typedef ma_uint32 (* ma_src_read_deinterleaved_proc)(ma_src* pSRC, ma_uint32 frameCount, void** ppSamplesOut, void* pUserData); /* Returns the number of frames that were read. */

typedef enum
{
    ma_src_algorithm_linear = 0,
    ma_src_algorithm_sinc,
    ma_src_algorithm_none,
    ma_src_algorithm_default = ma_src_algorithm_linear
} ma_src_algorithm;

typedef enum
{
    ma_src_sinc_window_function_hann = 0,
    ma_src_sinc_window_function_rectangular,
    ma_src_sinc_window_function_default = ma_src_sinc_window_function_hann
} ma_src_sinc_window_function;

typedef struct
{
    ma_src_sinc_window_function windowFunction;
    ma_uint32 windowWidth;
} ma_src_config_sinc;

typedef struct
{
    ma_uint32 sampleRateIn;
    ma_uint32 sampleRateOut;
    ma_uint32 channels;
    ma_src_algorithm algorithm;
    ma_bool32 neverConsumeEndOfInput : 1;
    ma_bool32 noSSE2   : 1;
    ma_bool32 noAVX2   : 1;
    ma_bool32 noAVX512 : 1;
    ma_bool32 noNEON   : 1;
    ma_src_read_deinterleaved_proc onReadDeinterleaved;
    void* pUserData;
    union
    {
        ma_src_config_sinc sinc;
    };
} ma_src_config;

MA_ALIGNED_STRUCT(MA_SIMD_ALIGNMENT) ma_src
{
    union
    {
        struct
        {
            MA_ALIGN(MA_SIMD_ALIGNMENT) float input[MA_MAX_CHANNELS][MA_SRC_INPUT_BUFFER_SIZE_IN_SAMPLES];
            float timeIn;
            ma_uint32 leftoverFrames;
        } linear;

        struct
        {
            MA_ALIGN(MA_SIMD_ALIGNMENT) float input[MA_MAX_CHANNELS][MA_SRC_SINC_MAX_WINDOW_WIDTH*2 + MA_SRC_INPUT_BUFFER_SIZE_IN_SAMPLES];
            float timeIn;
            ma_uint32 inputFrameCount;     /* The number of frames sitting in the input buffer, not including the first half of the window. */
            ma_uint32 windowPosInSamples;  /* An offset of <input>. */
            float table[MA_SRC_SINC_MAX_WINDOW_WIDTH*1 * MA_SRC_SINC_LOOKUP_TABLE_RESOLUTION]; /* Precomputed lookup table. The +1 is used to avoid the need for an overflow check. */
        } sinc;
    };

    ma_src_config config;
    ma_bool32 isEndOfInputLoaded : 1;
    ma_bool32 useSSE2   : 1;
    ma_bool32 useAVX2   : 1;
    ma_bool32 useAVX512 : 1;
    ma_bool32 useNEON   : 1;
};

typedef struct ma_pcm_converter ma_pcm_converter;
typedef ma_uint32 (* ma_pcm_converter_read_proc)(ma_pcm_converter* pDSP, void* pSamplesOut, ma_uint32 frameCount, void* pUserData);

typedef struct
{
    ma_format formatIn;
    ma_uint32 channelsIn;
    ma_uint32 sampleRateIn;
    ma_channel channelMapIn[MA_MAX_CHANNELS];
    ma_format formatOut;
    ma_uint32 channelsOut;
    ma_uint32 sampleRateOut;
    ma_channel channelMapOut[MA_MAX_CHANNELS];
    ma_channel_mix_mode channelMixMode;
    ma_dither_mode ditherMode;
    ma_src_algorithm srcAlgorithm;
    ma_bool32 allowDynamicSampleRate;
    ma_bool32 neverConsumeEndOfInput : 1;  /* <-- For SRC. */
    ma_bool32 noSSE2   : 1;
    ma_bool32 noAVX2   : 1;
    ma_bool32 noAVX512 : 1;
    ma_bool32 noNEON   : 1;
    ma_pcm_converter_read_proc onRead;
    void* pUserData;
    union
    {
        ma_src_config_sinc sinc;
    };
} ma_pcm_converter_config;

MA_ALIGNED_STRUCT(MA_SIMD_ALIGNMENT) ma_pcm_converter
{
    ma_pcm_converter_read_proc onRead;
    void* pUserData;
    ma_format_converter formatConverterIn;             /* For converting data to f32 in preparation for further processing. */
    ma_format_converter formatConverterOut;            /* For converting data to the requested output format. Used as the final step in the processing pipeline. */
    ma_channel_router channelRouter;                   /* For channel conversion. */
    ma_src src;                                        /* For sample rate conversion. */
    ma_bool32 isDynamicSampleRateAllowed     : 1;      /* ma_pcm_converter_set_input_sample_rate() and ma_pcm_converter_set_output_sample_rate() will fail if this is set to false. */
    ma_bool32 isPreFormatConversionRequired  : 1;
    ma_bool32 isPostFormatConversionRequired : 1;
    ma_bool32 isChannelRoutingRequired       : 1;
    ma_bool32 isSRCRequired                  : 1;
    ma_bool32 isChannelRoutingAtStart        : 1;
    ma_bool32 isPassthrough                  : 1;      /* <-- Will be set to true when the DSP pipeline is an optimized passthrough. */
};


/************************************************************************************************************************************************************
*************************************************************************************************************************************************************

DATA CONVERSION
===============

This section contains the APIs for data conversion. You will find everything here for channel mapping, sample format conversion, resampling, etc.

*************************************************************************************************************************************************************
************************************************************************************************************************************************************/

/************************************************************************************************************************************************************

Channel Maps
============

Below is the channel map used by ma_standard_channel_map_default:

|---------------|------------------------------|
| Channel Count | Mapping                      |
|---------------|------------------------------|
| 1 (Mono)      | 0: MA_CHANNEL_MONO           |
|---------------|------------------------------|
| 2 (Stereo)    | 0: MA_CHANNEL_FRONT_LEFT     |
|               | 1: MA_CHANNEL_FRONT_RIGHT    |
|---------------|------------------------------|
| 3             | 0: MA_CHANNEL_FRONT_LEFT     |
|               | 1: MA_CHANNEL_FRONT_RIGHT    |
|               | 2: MA_CHANNEL_FRONT_CENTER   |
|---------------|------------------------------|
| 4 (Surround)  | 0: MA_CHANNEL_FRONT_LEFT     |
|               | 1: MA_CHANNEL_FRONT_RIGHT    |
|               | 2: MA_CHANNEL_FRONT_CENTER   |
|               | 3: MA_CHANNEL_BACK_CENTER    |
|---------------|------------------------------|
| 5             | 0: MA_CHANNEL_FRONT_LEFT     |
|               | 1: MA_CHANNEL_FRONT_RIGHT    |
|               | 2: MA_CHANNEL_FRONT_CENTER   |
|               | 3: MA_CHANNEL_BACK_LEFT      |
|               | 4: MA_CHANNEL_BACK_RIGHT     |
|---------------|------------------------------|
| 6 (5.1)       | 0: MA_CHANNEL_FRONT_LEFT     |
|               | 1: MA_CHANNEL_FRONT_RIGHT    |
|               | 2: MA_CHANNEL_FRONT_CENTER   |
|               | 3: MA_CHANNEL_LFE            |
|               | 4: MA_CHANNEL_SIDE_LEFT      |
|               | 5: MA_CHANNEL_SIDE_RIGHT     |
|---------------|------------------------------|
| 7             | 0: MA_CHANNEL_FRONT_LEFT     |
|               | 1: MA_CHANNEL_FRONT_RIGHT    |
|               | 2: MA_CHANNEL_FRONT_CENTER   |
|               | 3: MA_CHANNEL_LFE            |
|               | 4: MA_CHANNEL_BACK_CENTER    |
|               | 4: MA_CHANNEL_SIDE_LEFT      |
|               | 5: MA_CHANNEL_SIDE_RIGHT     |
|---------------|------------------------------|
| 8 (7.1)       | 0: MA_CHANNEL_FRONT_LEFT     |
|               | 1: MA_CHANNEL_FRONT_RIGHT    |
|               | 2: MA_CHANNEL_FRONT_CENTER   |
|               | 3: MA_CHANNEL_LFE            |
|               | 4: MA_CHANNEL_BACK_LEFT      |
|               | 5: MA_CHANNEL_BACK_RIGHT     |
|               | 6: MA_CHANNEL_SIDE_LEFT      |
|               | 7: MA_CHANNEL_SIDE_RIGHT     |
|---------------|------------------------------|
| Other         | All channels set to 0. This  |
|               | is equivalent to the same    |
|               | mapping as the device.       |
|---------------|------------------------------|

************************************************************************************************************************************************************/

/*
Helper for retrieving a standard channel map.
*/
void ma_get_standard_channel_map(ma_standard_channel_map standardChannelMap, ma_uint32 channels, ma_channel channelMap[MA_MAX_CHANNELS]);

/*
Copies a channel map.
*/
void ma_channel_map_copy(ma_channel* pOut, const ma_channel* pIn, ma_uint32 channels);


/*
Determines whether or not a channel map is valid.

A blank channel map is valid (all channels set to MA_CHANNEL_NONE). The way a blank channel map is handled is context specific, but
is usually treated as a passthrough.

Invalid channel maps:
  - A channel map with no channels
  - A channel map with more than one channel and a mono channel
*/
ma_bool32 ma_channel_map_valid(ma_uint32 channels, const ma_channel channelMap[MA_MAX_CHANNELS]);

/*
Helper for comparing two channel maps for equality.

This assumes the channel count is the same between the two.
*/
ma_bool32 ma_channel_map_equal(ma_uint32 channels, const ma_channel channelMapA[MA_MAX_CHANNELS], const ma_channel channelMapB[MA_MAX_CHANNELS]);

/*
Helper for determining if a channel map is blank (all channels set to MA_CHANNEL_NONE).
*/
ma_bool32 ma_channel_map_blank(ma_uint32 channels, const ma_channel channelMap[MA_MAX_CHANNELS]);

/*
Helper for determining whether or not a channel is present in the given channel map.
*/
ma_bool32 ma_channel_map_contains_channel_position(ma_uint32 channels, const ma_channel channelMap[MA_MAX_CHANNELS], ma_channel channelPosition);


/************************************************************************************************************************************************************

Format Conversion
=================
The format converter serves two purposes:
  1) Conversion between data formats (u8 to f32, etc.)
  2) Interleaving and deinterleaving

When initializing a converter, you specify the input and output formats (u8, s16, etc.) and read callbacks. There are two read callbacks - one for
interleaved input data (onRead) and another for deinterleaved input data (onReadDeinterleaved). You implement whichever is most convenient for you. You
can implement both, but it's not recommended as it just introduces unnecessary complexity.

To read data as interleaved samples, use ma_format_converter_read(). Otherwise use ma_format_converter_read_deinterleaved().

Dithering
---------
The format converter also supports dithering. Dithering can be set using ditherMode variable in the config, like so.

  pConfig->ditherMode = ma_dither_mode_rectangle;

The different dithering modes include the following, in order of efficiency:
  - None:      ma_dither_mode_none
  - Rectangle: ma_dither_mode_rectangle
  - Triangle:  ma_dither_mode_triangle

Note that even if the dither mode is set to something other than ma_dither_mode_none, it will be ignored for conversions where dithering is not needed.
Dithering is available for the following conversions:
  - s16 -> u8
  - s24 -> u8
  - s32 -> u8
  - f32 -> u8
  - s24 -> s16
  - s32 -> s16
  - f32 -> s16

Note that it is not an error to pass something other than ma_dither_mode_none for conversions where dither is not used. It will just be ignored.

************************************************************************************************************************************************************/

/*
Initializes a format converter.
*/
ma_result ma_format_converter_init(const ma_format_converter_config* pConfig, ma_format_converter* pConverter);

/*
Reads data from the format converter as interleaved channels.
*/
ma_uint64 ma_format_converter_read(ma_format_converter* pConverter, ma_uint64 frameCount, void* pFramesOut, void* pUserData);

/*
Reads data from the format converter as deinterleaved channels.
*/
ma_uint64 ma_format_converter_read_deinterleaved(ma_format_converter* pConverter, ma_uint64 frameCount, void** ppSamplesOut, void* pUserData);

/*
Helper for initializing a format converter config.
*/
ma_format_converter_config ma_format_converter_config_init_new(void);
ma_format_converter_config ma_format_converter_config_init(ma_format formatIn, ma_format formatOut, ma_uint32 channels, ma_format_converter_read_proc onRead, void* pUserData);
ma_format_converter_config ma_format_converter_config_init_deinterleaved(ma_format formatIn, ma_format formatOut, ma_uint32 channels, ma_format_converter_read_deinterleaved_proc onReadDeinterleaved, void* pUserData);



/************************************************************************************************************************************************************

Channel Routing
===============
There are two main things you can do with the channel router:
  1) Rearrange channels
  2) Convert from one channel count to another

Channel Rearrangement
---------------------
A simple example of channel rearrangement may be swapping the left and right channels in a stereo stream. To do this you just pass in the same channel
count for both the input and output with channel maps that contain the same channels (in a different order).

Channel Conversion
------------------
The channel router can also convert from one channel count to another, such as converting a 5.1 stream to stero. When changing the channel count, the
router will first perform a 1:1 mapping of channel positions that are present in both the input and output channel maps. The second thing it will do
is distribute the input mono channel (if any) across all output channels, excluding any None and LFE channels. If there is an output mono channel, all
input channels will be averaged, excluding any None and LFE channels.

The last case to consider is when a channel position in the input channel map is not present in the output channel map, and vice versa. In this case the
channel router will perform a blend of other related channels to produce an audible channel. There are several blending modes.
  1) Simple
     Unmatched channels are silenced.
  2) Planar Blending
     Channels are blended based on a set of planes that each speaker emits audio from.

Rectangular / Planar Blending
-----------------------------
In this mode, channel positions are associated with a set of planes where the channel conceptually emits audio from. An example is the front/left speaker.
This speaker is positioned to the front of the listener, so you can think of it as emitting audio from the front plane. It is also positioned to the left
of the listener so you can think of it as also emitting audio from the left plane. Now consider the (unrealistic) situation where the input channel map
contains only the front/left channel position, but the output channel map contains both the front/left and front/center channel. When deciding on the audio
data to send to the front/center speaker (which has no 1:1 mapping with an input channel) we need to use some logic based on our available input channel
positions.

As mentioned earlier, our front/left speaker is, conceptually speaking, emitting audio from the front _and_ the left planes. Similarly, the front/center
speaker is emitting audio from _only_ the front plane. What these two channels have in common is that they are both emitting audio from the front plane.
Thus, it makes sense that the front/center speaker should receive some contribution from the front/left channel. How much contribution depends on their
planar relationship (thus the name of this blending technique).

Because the front/left channel is emitting audio from two planes (front and left), you can think of it as though it's willing to dedicate 50% of it's total
volume to each of it's planes (a channel position emitting from 1 plane would be willing to given 100% of it's total volume to that plane, and a channel
position emitting from 3 planes would be willing to given 33% of it's total volume to each plane). Similarly, the front/center speaker is emitting audio
from only one plane so you can think of it as though it's willing to _take_ 100% of it's volume from front plane emissions. Now, since the front/left
channel is willing to _give_ 50% of it's total volume to the front plane, and the front/center speaker is willing to _take_ 100% of it's total volume
from the front, you can imagine that 50% of the front/left speaker will be given to the front/center speaker.

Usage
-----
To use the channel router you need to specify three things:
  1) The input channel count and channel map
  2) The output channel count and channel map
  3) The mixing mode to use in the case where a 1:1 mapping is unavailable

Note that input and output data is always deinterleaved 32-bit floating point.

Initialize the channel router with ma_channel_router_init(). You will need to pass in a config object which specifies the input and output configuration,
mixing mode and a callback for sending data to the router. This callback will be called when input data needs to be sent to the router for processing. Note
that the mixing mode is only used when a 1:1 mapping is unavailable. This includes the custom weights mode.

Read data from the channel router with ma_channel_router_read_deinterleaved(). Output data is always 32-bit floating point.

************************************************************************************************************************************************************/

/*
Initializes a channel router where it is assumed that the input data is non-interleaved.
*/
ma_result ma_channel_router_init(const ma_channel_router_config* pConfig, ma_channel_router* pRouter);

/*
Reads data from the channel router as deinterleaved channels.
*/
ma_uint64 ma_channel_router_read_deinterleaved(ma_channel_router* pRouter, ma_uint64 frameCount, void** ppSamplesOut, void* pUserData);

/*
Helper for initializing a channel router config.
*/
ma_channel_router_config ma_channel_router_config_init(ma_uint32 channelsIn, const ma_channel channelMapIn[MA_MAX_CHANNELS], ma_uint32 channelsOut, const ma_channel channelMapOut[MA_MAX_CHANNELS], ma_channel_mix_mode mixingMode, ma_channel_router_read_deinterleaved_proc onRead, void* pUserData);


/************************************************************************************************************************************************************

Sample Rate Conversion
======================

************************************************************************************************************************************************************/

/*
Initializes a sample rate conversion object.
*/
ma_result ma_src_init(const ma_src_config* pConfig, ma_src* pSRC);

/*
Dynamically adjusts the sample rate.

This is useful for dynamically adjust pitch. Keep in mind, however, that this will speed up or slow down the sound. If this
is not acceptable you will need to use your own algorithm.
*/
ma_result ma_src_set_sample_rate(ma_src* pSRC, ma_uint32 sampleRateIn, ma_uint32 sampleRateOut);

/*
Reads a number of frames.

Returns the number of frames actually read.
*/
ma_uint64 ma_src_read_deinterleaved(ma_src* pSRC, ma_uint64 frameCount, void** ppSamplesOut, void* pUserData);

/*
Helper for creating a sample rate conversion config.
*/
ma_src_config ma_src_config_init_new(void);
ma_src_config ma_src_config_init(ma_uint32 sampleRateIn, ma_uint32 sampleRateOut, ma_uint32 channels, ma_src_read_deinterleaved_proc onReadDeinterleaved, void* pUserData);


/************************************************************************************************************************************************************

Conversion

************************************************************************************************************************************************************/

/*
Initializes a DSP object.
*/
ma_result ma_pcm_converter_init(const ma_pcm_converter_config* pConfig, ma_pcm_converter* pDSP);

/*
Dynamically adjusts the input sample rate.

This will fail is the DSP was not initialized with allowDynamicSampleRate.

DEPRECATED. Use ma_pcm_converter_set_sample_rate() instead.
*/
ma_result ma_pcm_converter_set_input_sample_rate(ma_pcm_converter* pDSP, ma_uint32 sampleRateOut);

/*
Dynamically adjusts the output sample rate.

This is useful for dynamically adjust pitch. Keep in mind, however, that this will speed up or slow down the sound. If this
is not acceptable you will need to use your own algorithm.

This will fail is the DSP was not initialized with allowDynamicSampleRate.

DEPRECATED. Use ma_pcm_converter_set_sample_rate() instead.
*/
ma_result ma_pcm_converter_set_output_sample_rate(ma_pcm_converter* pDSP, ma_uint32 sampleRateOut);

/*
Dynamically adjusts the output sample rate.

This is useful for dynamically adjust pitch. Keep in mind, however, that this will speed up or slow down the sound. If this
is not acceptable you will need to use your own algorithm.

This will fail if the DSP was not initialized with allowDynamicSampleRate.
*/
ma_result ma_pcm_converter_set_sample_rate(ma_pcm_converter* pDSP, ma_uint32 sampleRateIn, ma_uint32 sampleRateOut);

/*
Reads a number of frames and runs them through the DSP processor.
*/
ma_uint64 ma_pcm_converter_read(ma_pcm_converter* pDSP, void* pFramesOut, ma_uint64 frameCount);

/*
Helper for initializing a ma_pcm_converter_config object.
*/
ma_pcm_converter_config ma_pcm_converter_config_init_new(void);
ma_pcm_converter_config ma_pcm_converter_config_init(ma_format formatIn, ma_uint32 channelsIn, ma_uint32 sampleRateIn, ma_format formatOut, ma_uint32 channelsOut, ma_uint32 sampleRateOut, ma_pcm_converter_read_proc onRead, void* pUserData);
ma_pcm_converter_config ma_pcm_converter_config_init_ex(ma_format formatIn, ma_uint32 channelsIn, ma_uint32 sampleRateIn, ma_channel channelMapIn[MA_MAX_CHANNELS], ma_format formatOut, ma_uint32 channelsOut, ma_uint32 sampleRateOut,  ma_channel channelMapOut[MA_MAX_CHANNELS], ma_pcm_converter_read_proc onRead, void* pUserData);

/*
High-level helper for doing a full format conversion in one go. Returns the number of output frames. Call this with pOut set to NULL to
determine the required size of the output buffer.

A return value of 0 indicates an error.

This function is useful for one-off bulk conversions, but if you're streaming data you should use the DSP APIs instead.
*/
ma_uint64 ma_convert_frames(void* pOut, ma_format formatOut, ma_uint32 channelsOut, ma_uint32 sampleRateOut, const void* pIn, ma_format formatIn, ma_uint32 channelsIn, ma_uint32 sampleRateIn, ma_uint64 frameCount);
ma_uint64 ma_convert_frames_ex(void* pOut, ma_format formatOut, ma_uint32 channelsOut, ma_uint32 sampleRateOut, ma_channel channelMapOut[MA_MAX_CHANNELS], const void* pIn, ma_format formatIn, ma_uint32 channelsIn, ma_uint32 sampleRateIn, ma_channel channelMapIn[MA_MAX_CHANNELS], ma_uint64 frameCount);


/************************************************************************************************************************************************************

Ring Buffer
===========

Features
--------
- Lock free (assuming single producer, single consumer)
- Support for interleaved and deinterleaved streams
- Allows the caller to allocate their own block of memory

Usage
-----
- Call ma_rb_init() to initialize a simple buffer, with an optional pre-allocated buffer. If you pass in NULL
  for the pre-allocated buffer, it will be allocated for you and free()'d in ma_rb_uninit(). If you pass in
  your own pre-allocated buffer, free()-ing is left to you.

- Call ma_rb_init_ex() if you need a deinterleaved buffer. The data for each sub-buffer is offset from each
  other based on the stride. Use ma_rb_get_subbuffer_stride(), ma_rb_get_subbuffer_offset() and
  ma_rb_get_subbuffer_ptr() to manage your sub-buffers.

- Use ma_rb_acquire_read() and ma_rb_acquire_write() to retrieve a pointer to a section of the ring buffer.
  You specify the number of bytes you need, and on output it will set to what was actually acquired. If the
  read or write pointer is positioned such that the number of bytes requested will require a loop, it will be
  clamped to the end of the buffer. Therefore, the number of bytes you're given may be less than the number
  you requested.

- After calling ma_rb_acquire_read/write(), you do your work on the buffer and then "commit" it with
  ma_rb_commit_read/write(). This is where the read/write pointers are updated. When you commit you need to
  pass in the buffer that was returned by the earlier call to ma_rb_acquire_read/write() and is only used
  for validation. The number of bytes passed to ma_rb_commit_read/write() is what's used to increment the
  pointers.

- If you want to correct for drift between the write pointer and the read pointer you can use a combination
  of ma_rb_pointer_distance(), ma_rb_seek_read() and ma_rb_seek_write(). Note that you can only move the
  pointers forward, and you should only move the read pointer forward via the consumer thread, and the write
  pointer forward by the producer thread. If there is too much space between the pointers, move the read
  pointer forward. If there is too little space between the pointers, move the write pointer forward.


Notes
-----
- Thread safety depends on a single producer, single consumer model. Only one thread is allowed to write, and
  only one thread is allowed to read. The producer is the only one allowed to move the write pointer, and the
  consumer is the only one allowed to move the read pointer.
- Operates on bytes. Use ma_pcm_rb to operate in terms of PCM frames.
- Maximum buffer size in bytes is 0x7FFFFFFF-(MA_SIMD_ALIGNMENT-1) because of reasons.


PCM Ring Buffer
===============
This is the same as the regular ring buffer, except that it works on PCM frames instead of bytes.

************************************************************************************************************************************************************/
typedef struct
{
    void* pBuffer;
    ma_uint32 subbufferSizeInBytes;
    ma_uint32 subbufferCount;
    ma_uint32 subbufferStrideInBytes;
    volatile ma_uint32 encodedReadOffset;  /* Most significant bit is the loop flag. Lower 31 bits contains the actual offset in bytes. */
    volatile ma_uint32 encodedWriteOffset; /* Most significant bit is the loop flag. Lower 31 bits contains the actual offset in bytes. */
    ma_bool32 ownsBuffer          : 1;     /* Used to know whether or not miniaudio is responsible for free()-ing the buffer. */
    ma_bool32 clearOnWriteAcquire : 1;     /* When set, clears the acquired write buffer before returning from ma_rb_acquire_write(). */
} ma_rb;

ma_result ma_rb_init_ex(size_t subbufferSizeInBytes, size_t subbufferCount, size_t subbufferStrideInBytes, void* pOptionalPreallocatedBuffer, ma_rb* pRB);
ma_result ma_rb_init(size_t bufferSizeInBytes, void* pOptionalPreallocatedBuffer, ma_rb* pRB);
void ma_rb_uninit(ma_rb* pRB);
ma_result ma_rb_acquire_read(ma_rb* pRB, size_t* pSizeInBytes, void** ppBufferOut);
ma_result ma_rb_commit_read(ma_rb* pRB, size_t sizeInBytes, void* pBufferOut);
ma_result ma_rb_acquire_write(ma_rb* pRB, size_t* pSizeInBytes, void** ppBufferOut);
ma_result ma_rb_commit_write(ma_rb* pRB, size_t sizeInBytes, void* pBufferOut);
ma_result ma_rb_seek_read(ma_rb* pRB, size_t offsetInBytes);
ma_result ma_rb_seek_write(ma_rb* pRB, size_t offsetInBytes);
ma_int32 ma_rb_pointer_distance(ma_rb* pRB);    /* Returns the distance between the write pointer and the read pointer. Should never be negative for a correct program. */
size_t ma_rb_get_subbuffer_size(ma_rb* pRB);
size_t ma_rb_get_subbuffer_stride(ma_rb* pRB);
size_t ma_rb_get_subbuffer_offset(ma_rb* pRB, size_t subbufferIndex);
void* ma_rb_get_subbuffer_ptr(ma_rb* pRB, size_t subbufferIndex, void* pBuffer);


typedef struct
{
    ma_rb rb;
    ma_format format;
    ma_uint32 channels;
} ma_pcm_rb;

ma_result ma_pcm_rb_init_ex(ma_format format, ma_uint32 channels, ma_uint32 subbufferSizeInFrames, ma_uint32 subbufferCount, ma_uint32 subbufferStrideInFrames, void* pOptionalPreallocatedBuffer, ma_pcm_rb* pRB);
ma_result ma_pcm_rb_init(ma_format format, ma_uint32 channels, ma_uint32 bufferSizeInFrames, void* pOptionalPreallocatedBuffer, ma_pcm_rb* pRB);
void ma_pcm_rb_uninit(ma_pcm_rb* pRB);
ma_result ma_pcm_rb_acquire_read(ma_pcm_rb* pRB, ma_uint32* pSizeInFrames, void** ppBufferOut);
ma_result ma_pcm_rb_commit_read(ma_pcm_rb* pRB, ma_uint32 sizeInFrames, void* pBufferOut);
ma_result ma_pcm_rb_acquire_write(ma_pcm_rb* pRB, ma_uint32* pSizeInFrames, void** ppBufferOut);
ma_result ma_pcm_rb_commit_write(ma_pcm_rb* pRB, ma_uint32 sizeInFrames, void* pBufferOut);
ma_result ma_pcm_rb_seek_read(ma_pcm_rb* pRB, ma_uint32 offsetInFrames);
ma_result ma_pcm_rb_seek_write(ma_pcm_rb* pRB, ma_uint32 offsetInFrames);
ma_int32 ma_pcm_rb_pointer_disance(ma_pcm_rb* pRB); /* Return value is in frames. */
ma_uint32 ma_pcm_rb_get_subbuffer_size(ma_pcm_rb* pRB);
ma_uint32 ma_pcm_rb_get_subbuffer_stride(ma_pcm_rb* pRB);
ma_uint32 ma_pcm_rb_get_subbuffer_offset(ma_pcm_rb* pRB, ma_uint32 subbufferIndex);
void* ma_pcm_rb_get_subbuffer_ptr(ma_pcm_rb* pRB, ma_uint32 subbufferIndex, void* pBuffer);


/************************************************************************************************************************************************************

Miscellaneous Helpers

************************************************************************************************************************************************************/

/*
malloc(). Calls MA_MALLOC().
*/
void* ma_malloc(size_t sz);

/*
realloc(). Calls MA_REALLOC().
*/
void* ma_realloc(void* p, size_t sz);

/*
free(). Calls MA_FREE().
*/
void ma_free(void* p);

/*
Performs an aligned malloc, with the assumption that the alignment is a power of 2.
*/
void* ma_aligned_malloc(size_t sz, size_t alignment);

/*
Free's an aligned malloc'd buffer.
*/
void ma_aligned_free(void* p);

/*
Retrieves a friendly name for a format.
*/
const char* ma_get_format_name(ma_format format);

/*
Blends two frames in floating point format.
*/
void ma_blend_f32(float* pOut, float* pInA, float* pInB, float factor, ma_uint32 channels);

/*
Retrieves the size of a sample in bytes for the given format.

This API is efficient and is implemented using a lookup table.

Thread Safety: SAFE
  This API is pure.
*/
ma_uint32 ma_get_bytes_per_sample(ma_format format);
static MA_INLINE ma_uint32 ma_get_bytes_per_frame(ma_format format, ma_uint32 channels) { return ma_get_bytes_per_sample(format) * channels; }

/*
Converts a log level to a string.
*/
const char* ma_log_level_to_string(ma_uint32 logLevel);


/************************************************************************************************************************************************************

Format Conversion

************************************************************************************************************************************************************/
void ma_pcm_u8_to_s16(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_u8_to_s24(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_u8_to_s32(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_u8_to_f32(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s16_to_u8(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s16_to_s24(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s16_to_s32(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s16_to_f32(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s24_to_u8(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s24_to_s16(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s24_to_s32(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s24_to_f32(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s32_to_u8(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s32_to_s16(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s32_to_s24(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_s32_to_f32(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_f32_to_u8(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_f32_to_s16(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_f32_to_s24(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_f32_to_s32(void* pOut, const void* pIn, ma_uint64 count, ma_dither_mode ditherMode);
void ma_pcm_convert(void* pOut, ma_format formatOut, const void* pIn, ma_format formatIn, ma_uint64 sampleCount, ma_dither_mode ditherMode);

/*
Deinterleaves an interleaved buffer.
*/
void ma_deinterleave_pcm_frames(ma_format format, ma_uint32 channels, ma_uint64 frameCount, const void* pInterleavedPCMFrames, void** ppDeinterleavedPCMFrames);

/*
Interleaves a group of deinterleaved buffers.
*/
void ma_interleave_pcm_frames(ma_format format, ma_uint32 channels, ma_uint64 frameCount, const void** ppDeinterleavedPCMFrames, void* pInterleavedPCMFrames);


/************************************************************************************************************************************************************
*************************************************************************************************************************************************************

DEVICE I/O
==========

This section contains the APIs for device playback and capture. Here is where you'll find ma_device_init(), etc.

*************************************************************************************************************************************************************
************************************************************************************************************************************************************/
#ifndef MA_NO_DEVICE_IO
/* Some backends are only supported on certain platforms. */
#if defined(MA_WIN32)
    #define MA_SUPPORT_WASAPI
    #if defined(MA_WIN32_DESKTOP)  /* DirectSound and WinMM backends are only supported on desktops. */
        #define MA_SUPPORT_DSOUND
        #define MA_SUPPORT_WINMM
        #define MA_SUPPORT_JACK    /* JACK is technically supported on Windows, but I don't know how many people use it in practice... */
    #endif
#endif
#if defined(MA_UNIX)
    #if defined(MA_LINUX)
        #if !defined(MA_ANDROID)   /* ALSA is not supported on Android. */
            #define MA_SUPPORT_ALSA
        #endif
    #endif
    #if !defined(MA_BSD) && !defined(MA_ANDROID) && !defined(MA_EMSCRIPTEN)
        #define MA_SUPPORT_PULSEAUDIO
        #define MA_SUPPORT_JACK
    #endif
    #if defined(MA_ANDROID)
        #define MA_SUPPORT_AAUDIO
        #define MA_SUPPORT_OPENSL
    #endif
    #if defined(__OpenBSD__)        /* <-- Change this to "#if defined(MA_BSD)" to enable sndio on all BSD flavors. */
        #define MA_SUPPORT_SNDIO    /* sndio is only supported on OpenBSD for now. May be expanded later if there's demand. */
    #endif
    #if defined(__NetBSD__) || defined(__OpenBSD__)
        #define MA_SUPPORT_AUDIO4   /* Only support audio(4) on platforms with known support. */
    #endif
    #if defined(__FreeBSD__) || defined(__DragonFly__)
        #define MA_SUPPORT_OSS      /* Only support OSS on specific platforms with known support. */
    #endif
#endif
#if defined(MA_APPLE)
    #define MA_SUPPORT_COREAUDIO
#endif
#if defined(MA_EMSCRIPTEN)
    #define MA_SUPPORT_WEBAUDIO
#endif

/* Explicitly disable the Null backend for Emscripten because it uses a background thread which is not properly supported right now. */
#if !defined(MA_EMSCRIPTEN)
#define MA_SUPPORT_NULL
#endif


#if !defined(MA_NO_WASAPI) && defined(MA_SUPPORT_WASAPI)
    #define MA_ENABLE_WASAPI
#endif
#if !defined(MA_NO_DSOUND) && defined(MA_SUPPORT_DSOUND)
    #define MA_ENABLE_DSOUND
#endif
#if !defined(MA_NO_WINMM) && defined(MA_SUPPORT_WINMM)
    #define MA_ENABLE_WINMM
#endif
#if !defined(MA_NO_ALSA) && defined(MA_SUPPORT_ALSA)
    #define MA_ENABLE_ALSA
#endif
#if !defined(MA_NO_PULSEAUDIO) && defined(MA_SUPPORT_PULSEAUDIO)
    #define MA_ENABLE_PULSEAUDIO
#endif
#if !defined(MA_NO_JACK) && defined(MA_SUPPORT_JACK)
    #define MA_ENABLE_JACK
#endif
#if !defined(MA_NO_COREAUDIO) && defined(MA_SUPPORT_COREAUDIO)
    #define MA_ENABLE_COREAUDIO
#endif
#if !defined(MA_NO_SNDIO) && defined(MA_SUPPORT_SNDIO)
    #define MA_ENABLE_SNDIO
#endif
#if !defined(MA_NO_AUDIO4) && defined(MA_SUPPORT_AUDIO4)
    #define MA_ENABLE_AUDIO4
#endif
#if !defined(MA_NO_OSS) && defined(MA_SUPPORT_OSS)
    #define MA_ENABLE_OSS
#endif
#if !defined(MA_NO_AAUDIO) && defined(MA_SUPPORT_AAUDIO)
    #define MA_ENABLE_AAUDIO
#endif
#if !defined(MA_NO_OPENSL) && defined(MA_SUPPORT_OPENSL)
    #define MA_ENABLE_OPENSL
#endif
#if !defined(MA_NO_WEBAUDIO) && defined(MA_SUPPORT_WEBAUDIO)
    #define MA_ENABLE_WEBAUDIO
#endif
#if !defined(MA_NO_NULL) && defined(MA_SUPPORT_NULL)
    #define MA_ENABLE_NULL
#endif

#ifdef MA_SUPPORT_WASAPI
/* We need a IMMNotificationClient object for WASAPI. */
typedef struct
{
    void* lpVtbl;
    ma_uint32 counter;
    ma_device* pDevice;
} ma_IMMNotificationClient;
#endif

/* Backend enums must be in priority order. */
typedef enum
{
    ma_backend_wasapi,
    ma_backend_dsound,
    ma_backend_winmm,
    ma_backend_coreaudio,
    ma_backend_sndio,
    ma_backend_audio4,
    ma_backend_oss,
    ma_backend_pulseaudio,
    ma_backend_alsa,
    ma_backend_jack,
    ma_backend_aaudio,
    ma_backend_opensl,
    ma_backend_webaudio,
    ma_backend_null    /* <-- Must always be the last item. Lowest priority, and used as the terminator for backend enumeration. */
} ma_backend;

/* Thread priorties should be ordered such that the default priority of the worker thread is 0. */
typedef enum
{
    ma_thread_priority_idle     = -5,
    ma_thread_priority_lowest   = -4,
    ma_thread_priority_low      = -3,
    ma_thread_priority_normal   = -2,
    ma_thread_priority_high     = -1,
    ma_thread_priority_highest  =  0,
    ma_thread_priority_realtime =  1,
    ma_thread_priority_default  =  0
} ma_thread_priority;

typedef struct
{
    ma_context* pContext;

    union
    {
#ifdef MA_WIN32
        struct
        {
            /*HANDLE*/ ma_handle hThread;
        } win32;
#endif
#ifdef MA_POSIX
        struct
        {
            pthread_t thread;
        } posix;
#endif
        int _unused;
    };
} ma_thread;

typedef struct
{
    ma_context* pContext;

    union
    {
#ifdef MA_WIN32
        struct
        {
            /*HANDLE*/ ma_handle hMutex;
        } win32;
#endif
#ifdef MA_POSIX
        struct
        {
            pthread_mutex_t mutex;
        } posix;
#endif
        int _unused;
    };
} ma_mutex;

typedef struct
{
    ma_context* pContext;

    union
    {
#ifdef MA_WIN32
        struct
        {
            /*HANDLE*/ ma_handle hEvent;
        } win32;
#endif
#ifdef MA_POSIX
        struct
        {
            pthread_mutex_t mutex;
            pthread_cond_t condition;
            ma_uint32 value;
        } posix;
#endif
        int _unused;
    };
} ma_event;


/*
The callback for processing audio data from the device.

pOutput is a pointer to a buffer that will receive audio data that will later be played back through the speakers. This will be non-null
for a playback or full-duplex device and null for a capture device.

pInput is a pointer to a buffer containing input data from the device. This will be non-null for a capture or full-duplex device, and
null for a playback device.

frameCount is the number of PCM frames to process. If an output buffer is provided (pOutput is not null), applications should write out
to the entire output buffer.

Do _not_ call any miniaudio APIs from the callback. Attempting the stop the device can result in a deadlock. The proper way to stop the
device is to call ma_device_stop() from a different thread, normally the main application thread.
*/
typedef void (* ma_device_callback_proc)(ma_device* pDevice, void* pOutput, const void* pInput, ma_uint32 frameCount);

/*
The callback for when the device has been stopped.

This will be called when the device is stopped explicitly with ma_device_stop() and also called implicitly when the device is stopped
through external forces such as being unplugged or an internal error occuring.

Do not restart the device from the callback.
*/
typedef void (* ma_stop_proc)(ma_device* pDevice);

/*
The callback for handling log messages.

It is possible for pDevice to be null in which case the log originated from the context. If it is non-null you can assume the message
came from the device.

logLevel is one of the following:
    MA_LOG_LEVEL_VERBOSE
    MA_LOG_LEVEL_INFO
    MA_LOG_LEVEL_WARNING
    MA_LOG_LEVEL_ERROR
*/
typedef void (* ma_log_proc)(ma_context* pContext, ma_device* pDevice, ma_uint32 logLevel, const char* message);

typedef enum
{
    ma_device_type_playback = 1,
    ma_device_type_capture  = 2,
    ma_device_type_duplex   = ma_device_type_playback | ma_device_type_capture,
} ma_device_type;

typedef enum
{
    ma_share_mode_shared = 0,
    ma_share_mode_exclusive,
} ma_share_mode;

typedef union
{
#ifdef MA_SUPPORT_WASAPI
    wchar_t wasapi[64];             /* WASAPI uses a wchar_t string for identification. */
#endif
#ifdef MA_SUPPORT_DSOUND
    ma_uint8 dsound[16];            /* DirectSound uses a GUID for identification. */
#endif
#ifdef MA_SUPPORT_WINMM
    /*UINT_PTR*/ ma_uint32 winmm;   /* When creating a device, WinMM expects a Win32 UINT_PTR for device identification. In practice it's actually just a UINT. */
#endif
#ifdef MA_SUPPORT_ALSA
    char alsa[256];                 /* ALSA uses a name string for identification. */
#endif
#ifdef MA_SUPPORT_PULSEAUDIO
    char pulse[256];                /* PulseAudio uses a name string for identification. */
#endif
#ifdef MA_SUPPORT_JACK
    int jack;                       /* JACK always uses default devices. */
#endif
#ifdef MA_SUPPORT_COREAUDIO
    char coreaudio[256];            /* Core Audio uses a string for identification. */
#endif
#ifdef MA_SUPPORT_SNDIO
    char sndio[256];                /* "snd/0", etc. */
#endif
#ifdef MA_SUPPORT_AUDIO4
    char audio4[256];               /* "/dev/audio", etc. */
#endif
#ifdef MA_SUPPORT_OSS
    char oss[64];                   /* "dev/dsp0", etc. "dev/dsp" for the default device. */
#endif
#ifdef MA_SUPPORT_AAUDIO
    ma_int32 aaudio;                /* AAudio uses a 32-bit integer for identification. */
#endif
#ifdef MA_SUPPORT_OPENSL
    ma_uint32 opensl;               /* OpenSL|ES uses a 32-bit unsigned integer for identification. */
#endif
#ifdef MA_SUPPORT_WEBAUDIO
    char webaudio[32];              /* Web Audio always uses default devices for now, but if this changes it'll be a GUID. */
#endif
#ifdef MA_SUPPORT_NULL
    int nullbackend;                /* The null backend uses an integer for device IDs. */
#endif
} ma_device_id;

typedef struct
{
    /* Basic info. This is the only information guaranteed to be filled in during device enumeration. */
    ma_device_id id;
    char name[256];

    /*
    Detailed info. As much of this is filled as possible with ma_context_get_device_info(). Note that you are allowed to initialize
    a device with settings outside of this range, but it just means the data will be converted using miniaudio's data conversion
    pipeline before sending the data to/from the device. Most programs will need to not worry about these values, but it's provided
    here mainly for informational purposes or in the rare case that someone might find it useful.
    
    These will be set to 0 when returned by ma_context_enumerate_devices() or ma_context_get_devices().
    */
    ma_uint32 formatCount;
    ma_format formats[ma_format_count];
    ma_uint32 minChannels;
    ma_uint32 maxChannels;
    ma_uint32 minSampleRate;
    ma_uint32 maxSampleRate;
} ma_device_info;

typedef union
{
    ma_int64 counter;
    double counterD;
} ma_timer;

typedef struct
{
    ma_device_type deviceType;
    ma_uint32 sampleRate;
    ma_uint32 bufferSizeInFrames;
    ma_uint32 bufferSizeInMilliseconds;
    ma_uint32 periods;
    ma_performance_profile performanceProfile;
    ma_device_callback_proc dataCallback;
    ma_stop_proc stopCallback;
    void* pUserData;
    struct
    {
        ma_device_id* pDeviceID;
        ma_format format;
        ma_uint32 channels;
        ma_channel channelMap[MA_MAX_CHANNELS];
        ma_share_mode shareMode;
    } playback;
    struct
    {
        ma_device_id* pDeviceID;
        ma_format format;
        ma_uint32 channels;
        ma_channel channelMap[MA_MAX_CHANNELS];
        ma_share_mode shareMode;
    } capture;

    struct
    {
        ma_bool32 noMMap;  /* Disables MMap mode. */
    } alsa;
    struct
    {
        const char* pStreamNamePlayback;
        const char* pStreamNameCapture;
    } pulse;
} ma_device_config;

typedef struct
{
    ma_log_proc logCallback;
    ma_thread_priority threadPriority;
    void* pUserData;

    struct
    {
        ma_bool32 useVerboseDeviceEnumeration;
    } alsa;
    struct
    {
        const char* pApplicationName;
        const char* pServerName;
        ma_bool32 tryAutoSpawn; /* Enables autospawning of the PulseAudio daemon if necessary. */
    } pulse;
    struct
    {
        const char* pClientName;
        ma_bool32 tryStartServer;
    } jack;
} ma_context_config;

typedef ma_bool32 (* ma_enum_devices_callback_proc)(ma_context* pContext, ma_device_type deviceType, const ma_device_info* pInfo, void* pUserData);

struct ma_context
{
    ma_backend backend;                    /* DirectSound, ALSA, etc. */
    ma_log_proc logCallback;
    ma_thread_priority threadPriority;
    void* pUserData;
    ma_mutex deviceEnumLock;               /* Used to make ma_context_get_devices() thread safe. */
    ma_mutex deviceInfoLock;               /* Used to make ma_context_get_device_info() thread safe. */
    ma_uint32 deviceInfoCapacity;          /* Total capacity of pDeviceInfos. */
    ma_uint32 playbackDeviceInfoCount;
    ma_uint32 captureDeviceInfoCount;
    ma_device_info* pDeviceInfos;          /* Playback devices first, then capture. */
    ma_bool32 isBackendAsynchronous : 1;   /* Set when the context is initialized. Set to 1 for asynchronous backends such as Core Audio and JACK. Do not modify. */

    ma_result (* onUninit        )(ma_context* pContext);
    ma_bool32 (* onDeviceIDEqual )(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1);
    ma_result (* onEnumDevices   )(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData);    /* Return false from the callback to stop enumeration. */
    ma_result (* onGetDeviceInfo )(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo);
    ma_result (* onDeviceInit    )(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice);
    void      (* onDeviceUninit  )(ma_device* pDevice);
    ma_result (* onDeviceStart   )(ma_device* pDevice);
    ma_result (* onDeviceStop    )(ma_device* pDevice);
    ma_result (* onDeviceWrite   )(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount);    /* Data is in internal device format. */
    ma_result (* onDeviceRead    )(ma_device* pDevice,       void* pPCMFrames, ma_uint32 frameCount);    /* Data is in internal device format. */
    ma_result (* onDeviceMainLoop)(ma_device* pDevice);

    union
    {
#ifdef MA_SUPPORT_WASAPI
        struct
        {
            int _unused;
        } wasapi;
#endif
#ifdef MA_SUPPORT_DSOUND
        struct
        {
            ma_handle hDSoundDLL;
            ma_proc DirectSoundCreate;
            ma_proc DirectSoundEnumerateA;
            ma_proc DirectSoundCaptureCreate;
            ma_proc DirectSoundCaptureEnumerateA;
        } dsound;
#endif
#ifdef MA_SUPPORT_WINMM
        struct
        {
            ma_handle hWinMM;
            ma_proc waveOutGetNumDevs;
            ma_proc waveOutGetDevCapsA;
            ma_proc waveOutOpen;
            ma_proc waveOutClose;
            ma_proc waveOutPrepareHeader;
            ma_proc waveOutUnprepareHeader;
            ma_proc waveOutWrite;
            ma_proc waveOutReset;
            ma_proc waveInGetNumDevs;
            ma_proc waveInGetDevCapsA;
            ma_proc waveInOpen;
            ma_proc waveInClose;
            ma_proc waveInPrepareHeader;
            ma_proc waveInUnprepareHeader;
            ma_proc waveInAddBuffer;
            ma_proc waveInStart;
            ma_proc waveInReset;
        } winmm;
#endif
#ifdef MA_SUPPORT_ALSA
        struct
        {
            ma_handle asoundSO;
            ma_proc snd_pcm_open;
            ma_proc snd_pcm_close;
            ma_proc snd_pcm_hw_params_sizeof;
            ma_proc snd_pcm_hw_params_any;
            ma_proc snd_pcm_hw_params_set_format;
            ma_proc snd_pcm_hw_params_set_format_first;
            ma_proc snd_pcm_hw_params_get_format_mask;
            ma_proc snd_pcm_hw_params_set_channels_near;
            ma_proc snd_pcm_hw_params_set_rate_resample;
            ma_proc snd_pcm_hw_params_set_rate_near;
            ma_proc snd_pcm_hw_params_set_buffer_size_near;
            ma_proc snd_pcm_hw_params_set_periods_near;
            ma_proc snd_pcm_hw_params_set_access;
            ma_proc snd_pcm_hw_params_get_format;
            ma_proc snd_pcm_hw_params_get_channels;
            ma_proc snd_pcm_hw_params_get_channels_min;
            ma_proc snd_pcm_hw_params_get_channels_max;
            ma_proc snd_pcm_hw_params_get_rate;
            ma_proc snd_pcm_hw_params_get_rate_min;
            ma_proc snd_pcm_hw_params_get_rate_max;
            ma_proc snd_pcm_hw_params_get_buffer_size;
            ma_proc snd_pcm_hw_params_get_periods;
            ma_proc snd_pcm_hw_params_get_access;
            ma_proc snd_pcm_hw_params;
            ma_proc snd_pcm_sw_params_sizeof;
            ma_proc snd_pcm_sw_params_current;
            ma_proc snd_pcm_sw_params_get_boundary;
            ma_proc snd_pcm_sw_params_set_avail_min;
            ma_proc snd_pcm_sw_params_set_start_threshold;
            ma_proc snd_pcm_sw_params_set_stop_threshold;
            ma_proc snd_pcm_sw_params;
            ma_proc snd_pcm_format_mask_sizeof;
            ma_proc snd_pcm_format_mask_test;
            ma_proc snd_pcm_get_chmap;
            ma_proc snd_pcm_state;
            ma_proc snd_pcm_prepare;
            ma_proc snd_pcm_start;
            ma_proc snd_pcm_drop;
            ma_proc snd_pcm_drain;
            ma_proc snd_device_name_hint;
            ma_proc snd_device_name_get_hint;
            ma_proc snd_card_get_index;
            ma_proc snd_device_name_free_hint;
            ma_proc snd_pcm_mmap_begin;
            ma_proc snd_pcm_mmap_commit;
            ma_proc snd_pcm_recover;
            ma_proc snd_pcm_readi;
            ma_proc snd_pcm_writei;
            ma_proc snd_pcm_avail;
            ma_proc snd_pcm_avail_update;
            ma_proc snd_pcm_wait;
            ma_proc snd_pcm_info;
            ma_proc snd_pcm_info_sizeof;
            ma_proc snd_pcm_info_get_name;
            ma_proc snd_config_update_free_global;

            ma_mutex internalDeviceEnumLock;
            ma_bool32 useVerboseDeviceEnumeration;
        } alsa;
#endif
#ifdef MA_SUPPORT_PULSEAUDIO
        struct
        {
            ma_handle pulseSO;
            ma_proc pa_mainloop_new;
            ma_proc pa_mainloop_free;
            ma_proc pa_mainloop_get_api;
            ma_proc pa_mainloop_iterate;
            ma_proc pa_mainloop_wakeup;
            ma_proc pa_context_new;
            ma_proc pa_context_unref;
            ma_proc pa_context_connect;
            ma_proc pa_context_disconnect;
            ma_proc pa_context_set_state_callback;
            ma_proc pa_context_get_state;
            ma_proc pa_context_get_sink_info_list;
            ma_proc pa_context_get_source_info_list;
            ma_proc pa_context_get_sink_info_by_name;
            ma_proc pa_context_get_source_info_by_name;
            ma_proc pa_operation_unref;
            ma_proc pa_operation_get_state;
            ma_proc pa_channel_map_init_extend;
            ma_proc pa_channel_map_valid;
            ma_proc pa_channel_map_compatible;
            ma_proc pa_stream_new;
            ma_proc pa_stream_unref;
            ma_proc pa_stream_connect_playback;
            ma_proc pa_stream_connect_record;
            ma_proc pa_stream_disconnect;
            ma_proc pa_stream_get_state;
            ma_proc pa_stream_get_sample_spec;
            ma_proc pa_stream_get_channel_map;
            ma_proc pa_stream_get_buffer_attr;
            ma_proc pa_stream_set_buffer_attr;
            ma_proc pa_stream_get_device_name;
            ma_proc pa_stream_set_write_callback;
            ma_proc pa_stream_set_read_callback;
            ma_proc pa_stream_flush;
            ma_proc pa_stream_drain;
            ma_proc pa_stream_is_corked;
            ma_proc pa_stream_cork;
            ma_proc pa_stream_trigger;
            ma_proc pa_stream_begin_write;
            ma_proc pa_stream_write;
            ma_proc pa_stream_peek;
            ma_proc pa_stream_drop;
            ma_proc pa_stream_writable_size;
            ma_proc pa_stream_readable_size;

            char* pApplicationName;
            char* pServerName;
            ma_bool32 tryAutoSpawn;
        } pulse;
#endif
#ifdef MA_SUPPORT_JACK
        struct
        {
            ma_handle jackSO;
            ma_proc jack_client_open;
            ma_proc jack_client_close;
            ma_proc jack_client_name_size;
            ma_proc jack_set_process_callback;
            ma_proc jack_set_buffer_size_callback;
            ma_proc jack_on_shutdown;
            ma_proc jack_get_sample_rate;
            ma_proc jack_get_buffer_size;
            ma_proc jack_get_ports;
            ma_proc jack_activate;
            ma_proc jack_deactivate;
            ma_proc jack_connect;
            ma_proc jack_port_register;
            ma_proc jack_port_name;
            ma_proc jack_port_get_buffer;
            ma_proc jack_free;

            char* pClientName;
            ma_bool32 tryStartServer;
        } jack;
#endif
#ifdef MA_SUPPORT_COREAUDIO
        struct
        {
            ma_handle hCoreFoundation;
            ma_proc CFStringGetCString;
            
            ma_handle hCoreAudio;
            ma_proc AudioObjectGetPropertyData;
            ma_proc AudioObjectGetPropertyDataSize;
            ma_proc AudioObjectSetPropertyData;
            ma_proc AudioObjectAddPropertyListener;
            
            ma_handle hAudioUnit;  /* Could possibly be set to AudioToolbox on later versions of macOS. */
            ma_proc AudioComponentFindNext;
            ma_proc AudioComponentInstanceDispose;
            ma_proc AudioComponentInstanceNew;
            ma_proc AudioOutputUnitStart;
            ma_proc AudioOutputUnitStop;
            ma_proc AudioUnitAddPropertyListener;
            ma_proc AudioUnitGetPropertyInfo;
            ma_proc AudioUnitGetProperty;
            ma_proc AudioUnitSetProperty;
            ma_proc AudioUnitInitialize;
            ma_proc AudioUnitRender;
            
            /*AudioComponent*/ ma_ptr component;
        } coreaudio;
#endif
#ifdef MA_SUPPORT_SNDIO
        struct
        {
            ma_handle sndioSO;
            ma_proc sio_open;
            ma_proc sio_close;
            ma_proc sio_setpar;
            ma_proc sio_getpar;
            ma_proc sio_getcap;
            ma_proc sio_start;
            ma_proc sio_stop;
            ma_proc sio_read;
            ma_proc sio_write;
            ma_proc sio_onmove;
            ma_proc sio_nfds;
            ma_proc sio_pollfd;
            ma_proc sio_revents;
            ma_proc sio_eof;
            ma_proc sio_setvol;
            ma_proc sio_onvol;
            ma_proc sio_initpar;
        } sndio;
#endif
#ifdef MA_SUPPORT_AUDIO4
        struct
        {
            int _unused;
        } audio4;
#endif
#ifdef MA_SUPPORT_OSS
        struct
        {
            int versionMajor;
            int versionMinor;
        } oss;
#endif
#ifdef MA_SUPPORT_AAUDIO
        struct
        {
            ma_handle hAAudio; /* libaaudio.so */
            ma_proc AAudio_createStreamBuilder;
            ma_proc AAudioStreamBuilder_delete;
            ma_proc AAudioStreamBuilder_setDeviceId;
            ma_proc AAudioStreamBuilder_setDirection;
            ma_proc AAudioStreamBuilder_setSharingMode;
            ma_proc AAudioStreamBuilder_setFormat;
            ma_proc AAudioStreamBuilder_setChannelCount;
            ma_proc AAudioStreamBuilder_setSampleRate;
            ma_proc AAudioStreamBuilder_setBufferCapacityInFrames;
            ma_proc AAudioStreamBuilder_setFramesPerDataCallback;
            ma_proc AAudioStreamBuilder_setDataCallback;
            ma_proc AAudioStreamBuilder_setPerformanceMode;
            ma_proc AAudioStreamBuilder_openStream;
            ma_proc AAudioStream_close;
            ma_proc AAudioStream_getState;
            ma_proc AAudioStream_waitForStateChange;
            ma_proc AAudioStream_getFormat;
            ma_proc AAudioStream_getChannelCount;
            ma_proc AAudioStream_getSampleRate;
            ma_proc AAudioStream_getBufferCapacityInFrames;
            ma_proc AAudioStream_getFramesPerDataCallback;
            ma_proc AAudioStream_getFramesPerBurst;
            ma_proc AAudioStream_requestStart;
            ma_proc AAudioStream_requestStop;
        } aaudio;
#endif
#ifdef MA_SUPPORT_OPENSL
        struct
        {
            int _unused;
        } opensl;
#endif
#ifdef MA_SUPPORT_WEBAUDIO
        struct
        {
            int _unused;
        } webaudio;
#endif
#ifdef MA_SUPPORT_NULL
        struct
        {
            int _unused;
        } null_backend;
#endif
    };

    union
    {
#ifdef MA_WIN32
        struct
        {
            /*HMODULE*/ ma_handle hOle32DLL;
            ma_proc CoInitializeEx;
            ma_proc CoUninitialize;
            ma_proc CoCreateInstance;
            ma_proc CoTaskMemFree;
            ma_proc PropVariantClear;
            ma_proc StringFromGUID2;

            /*HMODULE*/ ma_handle hUser32DLL;
            ma_proc GetForegroundWindow;
            ma_proc GetDesktopWindow;

            /*HMODULE*/ ma_handle hAdvapi32DLL;
            ma_proc RegOpenKeyExA;
            ma_proc RegCloseKey;
            ma_proc RegQueryValueExA;
        } win32;
#endif
#ifdef MA_POSIX
        struct
        {
            ma_handle pthreadSO;
            ma_proc pthread_create;
            ma_proc pthread_join;
            ma_proc pthread_mutex_init;
            ma_proc pthread_mutex_destroy;
            ma_proc pthread_mutex_lock;
            ma_proc pthread_mutex_unlock;
            ma_proc pthread_cond_init;
            ma_proc pthread_cond_destroy;
            ma_proc pthread_cond_wait;
            ma_proc pthread_cond_signal;
            ma_proc pthread_attr_init;
            ma_proc pthread_attr_destroy;
            ma_proc pthread_attr_setschedpolicy;
            ma_proc pthread_attr_getschedparam;
            ma_proc pthread_attr_setschedparam;
        } posix;
#endif
        int _unused;
    };
};

MA_ALIGNED_STRUCT(MA_SIMD_ALIGNMENT) ma_device
{
    ma_context* pContext;
    ma_device_type type;
    ma_uint32 sampleRate;
    ma_uint32 state;
    ma_device_callback_proc onData;
    ma_stop_proc onStop;
    void* pUserData;                        /* Application defined data. */
    ma_mutex lock;
    ma_event wakeupEvent;
    ma_event startEvent;
    ma_event stopEvent;
    ma_thread thread;
    ma_result workResult;                   /* This is set by the worker thread after it's finished doing a job. */
    ma_bool32 usingDefaultSampleRate : 1;
    ma_bool32 usingDefaultBufferSize : 1;
    ma_bool32 usingDefaultPeriods    : 1;
    ma_bool32 isOwnerOfContext       : 1;   /* When set to true, uninitializing the device will also uninitialize the context. Set to true when NULL is passed into ma_device_init(). */
    struct
    {
        char name[256];                     /* Maybe temporary. Likely to be replaced with a query API. */
        ma_share_mode shareMode;            /* Set to whatever was passed in when the device was initialized. */
        ma_bool32 usingDefaultFormat     : 1;
        ma_bool32 usingDefaultChannels   : 1;
        ma_bool32 usingDefaultChannelMap : 1;
        ma_format format;
        ma_uint32 channels;
        ma_channel channelMap[MA_MAX_CHANNELS];
        ma_format internalFormat;
        ma_uint32 internalChannels;
        ma_uint32 internalSampleRate;
        ma_channel internalChannelMap[MA_MAX_CHANNELS];
        ma_uint32 internalBufferSizeInFrames;
        ma_uint32 internalPeriods;
        ma_pcm_converter converter;
        ma_uint32 _dspFrameCount;           /* Internal use only. Used as the data source when reading from the device. */
        const ma_uint8* _dspFrames;         /* ^^^ AS ABOVE ^^^ */
    } playback;
    struct
    {
        char name[256];                     /* Maybe temporary. Likely to be replaced with a query API. */
        ma_share_mode shareMode;            /* Set to whatever was passed in when the device was initialized. */
        ma_bool32 usingDefaultFormat     : 1;
        ma_bool32 usingDefaultChannels   : 1;
        ma_bool32 usingDefaultChannelMap : 1;
        ma_format format;
        ma_uint32 channels;
        ma_channel channelMap[MA_MAX_CHANNELS];
        ma_format internalFormat;
        ma_uint32 internalChannels;
        ma_uint32 internalSampleRate;
        ma_channel internalChannelMap[MA_MAX_CHANNELS];
        ma_uint32 internalBufferSizeInFrames;
        ma_uint32 internalPeriods;
        ma_pcm_converter converter;
        ma_uint32 _dspFrameCount;           /* Internal use only. Used as the data source when reading from the device. */
        const ma_uint8* _dspFrames;         /* ^^^ AS ABOVE ^^^ */
    } capture;

    union
    {
#ifdef MA_SUPPORT_WASAPI
        struct
        {
            /*IAudioClient**/ ma_ptr pAudioClientPlayback;
            /*IAudioClient**/ ma_ptr pAudioClientCapture;
            /*IAudioRenderClient**/ ma_ptr pRenderClient;
            /*IAudioCaptureClient**/ ma_ptr pCaptureClient;
            /*IMMDeviceEnumerator**/ ma_ptr pDeviceEnumerator; /* Used for IMMNotificationClient notifications. Required for detecting default device changes. */
            ma_IMMNotificationClient notificationClient;
            /*HANDLE*/ ma_handle hEventPlayback;               /* Auto reset. Initialized to signaled. */
            /*HANDLE*/ ma_handle hEventCapture;                /* Auto reset. Initialized to unsignaled. */
            ma_uint32 actualBufferSizeInFramesPlayback;        /* Value from GetBufferSize(). internalBufferSizeInFrames is not set to the _actual_ buffer size when low-latency shared mode is being used due to the way the IAudioClient3 API works. */
            ma_uint32 actualBufferSizeInFramesCapture;
            ma_uint32 originalBufferSizeInFrames;
            ma_uint32 originalBufferSizeInMilliseconds;
            ma_uint32 originalPeriods;
            ma_bool32 hasDefaultPlaybackDeviceChanged;         /* <-- Make sure this is always a whole 32-bits because we use atomic assignments. */
            ma_bool32 hasDefaultCaptureDeviceChanged;          /* <-- Make sure this is always a whole 32-bits because we use atomic assignments. */
            ma_uint32 periodSizeInFramesPlayback;
            ma_uint32 periodSizeInFramesCapture;
            ma_bool32 isStartedCapture;
            ma_bool32 isStartedPlayback;
        } wasapi;
#endif
#ifdef MA_SUPPORT_DSOUND
        struct
        {
            /*LPDIRECTSOUND*/ ma_ptr pPlayback;
            /*LPDIRECTSOUNDBUFFER*/ ma_ptr pPlaybackPrimaryBuffer;
            /*LPDIRECTSOUNDBUFFER*/ ma_ptr pPlaybackBuffer;
            /*LPDIRECTSOUNDCAPTURE*/ ma_ptr pCapture;
            /*LPDIRECTSOUNDCAPTUREBUFFER*/ ma_ptr pCaptureBuffer;
        } dsound;
#endif
#ifdef MA_SUPPORT_WINMM
        struct
        {
            /*HWAVEOUT*/ ma_handle hDevicePlayback;
            /*HWAVEIN*/ ma_handle hDeviceCapture;
            /*HANDLE*/ ma_handle hEventPlayback;
            /*HANDLE*/ ma_handle hEventCapture;
            ma_uint32 fragmentSizeInFrames;
            ma_uint32 fragmentSizeInBytes;
            ma_uint32 iNextHeaderPlayback;             /* [0,periods). Used as an index into pWAVEHDRPlayback. */
            ma_uint32 iNextHeaderCapture;              /* [0,periods). Used as an index into pWAVEHDRCapture. */
            ma_uint32 headerFramesConsumedPlayback;    /* The number of PCM frames consumed in the buffer in pWAVEHEADER[iNextHeader]. */
            ma_uint32 headerFramesConsumedCapture;     /* ^^^ */
            /*WAVEHDR**/ ma_uint8* pWAVEHDRPlayback;   /* One instantiation for each period. */
            /*WAVEHDR**/ ma_uint8* pWAVEHDRCapture;    /* One instantiation for each period. */
            ma_uint8* pIntermediaryBufferPlayback;
            ma_uint8* pIntermediaryBufferCapture;
            ma_uint8* _pHeapData;                      /* Used internally and is used for the heap allocated data for the intermediary buffer and the WAVEHDR structures. */
            ma_bool32 isStarted;
        } winmm;
#endif
#ifdef MA_SUPPORT_ALSA
        struct
        {
            /*snd_pcm_t**/ ma_ptr pPCMPlayback;
            /*snd_pcm_t**/ ma_ptr pPCMCapture;
            ma_bool32 isUsingMMapPlayback : 1;
            ma_bool32 isUsingMMapCapture  : 1;
        } alsa;
#endif
#ifdef MA_SUPPORT_PULSEAUDIO
        struct
        {
            /*pa_mainloop**/ ma_ptr pMainLoop;
            /*pa_mainloop_api**/ ma_ptr pAPI;
            /*pa_context**/ ma_ptr pPulseContext;
            /*pa_stream**/ ma_ptr pStreamPlayback;
            /*pa_stream**/ ma_ptr pStreamCapture;
            /*pa_context_state*/ ma_uint32 pulseContextState;
            void* pMappedBufferPlayback;
            const void* pMappedBufferCapture;
            ma_uint32 mappedBufferFramesRemainingPlayback;
            ma_uint32 mappedBufferFramesRemainingCapture;
            ma_uint32 mappedBufferFramesCapacityPlayback;
            ma_uint32 mappedBufferFramesCapacityCapture;
            ma_bool32 breakFromMainLoop : 1;
        } pulse;
#endif
#ifdef MA_SUPPORT_JACK
        struct
        {
            /*jack_client_t**/ ma_ptr pClient;
            /*jack_port_t**/ ma_ptr pPortsPlayback[MA_MAX_CHANNELS];
            /*jack_port_t**/ ma_ptr pPortsCapture[MA_MAX_CHANNELS];
            float* pIntermediaryBufferPlayback; /* Typed as a float because JACK is always floating point. */
            float* pIntermediaryBufferCapture;
            ma_pcm_rb duplexRB;
        } jack;
#endif
#ifdef MA_SUPPORT_COREAUDIO
        struct
        {
            ma_uint32 deviceObjectIDPlayback;
            ma_uint32 deviceObjectIDCapture;
            /*AudioUnit*/ ma_ptr audioUnitPlayback;
            /*AudioUnit*/ ma_ptr audioUnitCapture;
            /*AudioBufferList**/ ma_ptr pAudioBufferList;  /* Only used for input devices. */
            ma_event stopEvent;
            ma_uint32 originalBufferSizeInFrames;
            ma_uint32 originalBufferSizeInMilliseconds;
            ma_uint32 originalPeriods;
            ma_bool32 isDefaultPlaybackDevice;
            ma_bool32 isDefaultCaptureDevice;
            ma_bool32 isSwitchingPlaybackDevice;   /* <-- Set to true when the default device has changed and miniaudio is in the process of switching. */
            ma_bool32 isSwitchingCaptureDevice;    /* <-- Set to true when the default device has changed and miniaudio is in the process of switching. */
            ma_pcm_rb duplexRB;
        } coreaudio;
#endif
#ifdef MA_SUPPORT_SNDIO
        struct
        {
            ma_ptr handlePlayback;
            ma_ptr handleCapture;
            ma_bool32 isStartedPlayback;
            ma_bool32 isStartedCapture;
        } sndio;
#endif
#ifdef MA_SUPPORT_AUDIO4
        struct
        {
            int fdPlayback;
            int fdCapture;
        } audio4;
#endif
#ifdef MA_SUPPORT_OSS
        struct
        {
            int fdPlayback;
            int fdCapture;
        } oss;
#endif
#ifdef MA_SUPPORT_AAUDIO
        struct
        {
            /*AAudioStream**/ ma_ptr pStreamPlayback;
            /*AAudioStream**/ ma_ptr pStreamCapture;
            ma_pcm_rb duplexRB;
        } aaudio;
#endif
#ifdef MA_SUPPORT_OPENSL
        struct
        {
            /*SLObjectItf*/ ma_ptr pOutputMixObj;
            /*SLOutputMixItf*/ ma_ptr pOutputMix;
            /*SLObjectItf*/ ma_ptr pAudioPlayerObj;
            /*SLPlayItf*/ ma_ptr pAudioPlayer;
            /*SLObjectItf*/ ma_ptr pAudioRecorderObj;
            /*SLRecordItf*/ ma_ptr pAudioRecorder;
            /*SLAndroidSimpleBufferQueueItf*/ ma_ptr pBufferQueuePlayback;
            /*SLAndroidSimpleBufferQueueItf*/ ma_ptr pBufferQueueCapture;
            ma_uint32 currentBufferIndexPlayback;
            ma_uint32 currentBufferIndexCapture;
            ma_uint8* pBufferPlayback;      /* This is malloc()'d and is used for storing audio data. Typed as ma_uint8 for easy offsetting. */
            ma_uint8* pBufferCapture;
            ma_pcm_rb duplexRB;
        } opensl;
#endif
#ifdef MA_SUPPORT_WEBAUDIO
        struct
        {
            int indexPlayback;              /* We use a factory on the JavaScript side to manage devices and use an index for JS/C interop. */
            int indexCapture;
            ma_pcm_rb duplexRB;             /* In external capture format. */
        } webaudio;
#endif
#ifdef MA_SUPPORT_NULL
        struct
        {
            ma_thread deviceThread;
            ma_event operationEvent;
            ma_event operationCompletionEvent;
            ma_uint32 operation;
            ma_result operationResult;
            ma_timer timer;
            double priorRunTime;
            ma_uint32 currentPeriodFramesRemainingPlayback;
            ma_uint32 currentPeriodFramesRemainingCapture;
            ma_uint64 lastProcessedFramePlayback;
            ma_uint32 lastProcessedFrameCapture;
            ma_bool32 isStarted;
        } null_device;
#endif
    };
};
#if defined(_MSC_VER)
    #pragma warning(pop)
#else
    #pragma GCC diagnostic pop  /* For ISO C99 doesn't support unnamed structs/unions [-Wpedantic] */
#endif

/*
Initializes a context.

The context is used for selecting and initializing the relevant backends.

Note that the location of the context cannot change throughout it's lifetime. Consider allocating
the ma_context object with malloc() if this is an issue. The reason for this is that a pointer
to the context is stored in the ma_device structure.

<backends> is used to allow the application to prioritize backends depending on it's specific
requirements. This can be null in which case it uses the default priority, which is as follows:
  - WASAPI
  - DirectSound
  - WinMM
  - Core Audio (Apple)
  - sndio
  - audio(4)
  - OSS
  - PulseAudio
  - ALSA
  - JACK
  - AAudio
  - OpenSL|ES
  - Web Audio / Emscripten
  - Null

<pConfig> is used to configure the context. Use the logCallback config to set a callback for whenever a
log message is posted. The priority of the worker thread can be set with the threadPriority config.

It is recommended that only a single context is active at any given time because it's a bulky data
structure which performs run-time linking for the relevant backends every time it's initialized.

Return Value:
  MA_SUCCESS if successful; any other error code otherwise.

Thread Safety: UNSAFE
*/
ma_result ma_context_init(const ma_backend backends[], ma_uint32 backendCount, const ma_context_config* pConfig, ma_context* pContext);

/*
Uninitializes a context.

Results are undefined if you call this while any device created by this context is still active.

Return Value:
  MA_SUCCESS if successful; any other error code otherwise.

Thread Safety: UNSAFE
*/
ma_result ma_context_uninit(ma_context* pContext);

/*
Enumerates over every device (both playback and capture).

This is a lower-level enumeration function to the easier to use ma_context_get_devices(). Use
ma_context_enumerate_devices() if you would rather not incur an internal heap allocation, or
it simply suits your code better.

Do _not_ assume the first enumerated device of a given type is the default device.

Some backends and platforms may only support default playback and capture devices.

Note that this only retrieves the ID and name/description of the device. The reason for only
retrieving basic information is that it would otherwise require opening the backend device in
order to probe it for more detailed information which can be inefficient. Consider using
ma_context_get_device_info() for this, but don't call it from within the enumeration callback.

In general, you should not do anything complicated from within the callback. In particular, do
not try initializing a device from within the callback.

Consider using ma_context_get_devices() for a simpler and safer API, albeit at the expense of
an internal heap allocation.

Returning false from the callback will stop enumeration. Returning true will continue enumeration.

Return Value:
  MA_SUCCESS if successful; any other error code otherwise.

Thread Safety: SAFE
  This is guarded using a simple mutex lock.
*/
ma_result ma_context_enumerate_devices(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData);

/*
Retrieves basic information about every active playback and/or capture device.

You can pass in NULL for the playback or capture lists in which case they'll be ignored.

It is _not_ safe to assume the first device in the list is the default device.

The returned pointers will become invalid upon the next call this this function, or when the
context is uninitialized. Do not free the returned pointers.

This function follows the same enumeration rules as ma_context_enumerate_devices(). See
documentation for ma_context_enumerate_devices() for more information.

Return Value:
  MA_SUCCESS if successful; any other error code otherwise.

Thread Safety: SAFE
  Since each call to this function invalidates the pointers from the previous call, you
  should not be calling this simultaneously across multiple threads. Instead, you need to
  make a copy of the returned data with your own higher level synchronization.
*/
ma_result ma_context_get_devices(ma_context* pContext, ma_device_info** ppPlaybackDeviceInfos, ma_uint32* pPlaybackDeviceCount, ma_device_info** ppCaptureDeviceInfos, ma_uint32* pCaptureDeviceCount);

/*
Retrieves information about a device with the given ID.

Do _not_ call this from within the ma_context_enumerate_devices() callback.

It's possible for a device to have different information and capabilities depending on wether or
not it's opened in shared or exclusive mode. For example, in shared mode, WASAPI always uses
floating point samples for mixing, but in exclusive mode it can be anything. Therefore, this
function allows you to specify which share mode you want information for. Note that not all
backends and devices support shared or exclusive mode, in which case this function will fail
if the requested share mode is unsupported.

This leaves pDeviceInfo unmodified in the result of an error.

Return Value:
  MA_SUCCESS if successful; any other error code otherwise.

Thread Safety: SAFE
  This is guarded using a simple mutex lock.
*/
ma_result ma_context_get_device_info(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo);

/*
Initializes a device.

The context can be null in which case it uses the default. This is equivalent to passing in a
context that was initialized like so:

    ma_context_init(NULL, 0, NULL, &context);

Do not pass in null for the context if you are needing to open multiple devices. You can,
however, use null when initializing the first device, and then use device.pContext for the
initialization of other devices.

The device's configuration is controlled with pConfig. This allows you to configure the sample
format, channel count, sample rate, etc. Before calling ma_device_init(), you will need to
initialize a ma_device_config object using ma_device_config_init(). You must set the callback in
the device config.

Passing in 0 to any property in pConfig will force the use of a default value. In the case of
sample format, channel count, sample rate and channel map it will default to the values used by
the backend's internal device. For the size of the buffer you can set bufferSizeInFrames or
bufferSizeInMilliseconds (if both are set it will prioritize bufferSizeInFrames). If both are
set to zero, it will default to MA_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY or
MA_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE, depending on whether or not performanceProfile
is set to ma_performance_profile_low_latency or ma_performance_profile_conservative.

If you request exclusive mode and the backend does not support it an error will be returned. For
robustness, you may want to first try initializing the device in exclusive mode, and then fall back
to shared mode if required. Alternatively you can just request shared mode (the default if you
leave it unset in the config) which is the most reliable option. Some backends do not have a
practical way of choosing whether or not the device should be exclusive or not (ALSA, for example)
in which case it just acts as a hint. Unless you have special requirements you should try avoiding
exclusive mode as it's intrusive to the user. Starting with Windows 10, miniaudio will use low-latency
shared mode where possible which may make exclusive mode unnecessary.

When sending or receiving data to/from a device, miniaudio will internally perform a format
conversion to convert between the format specified by pConfig and the format used internally by
the backend. If you pass in NULL for pConfig or 0 for the sample format, channel count,
sample rate _and_ channel map, data transmission will run on an optimized pass-through fast path.

The buffer size should be treated as a hint. miniaudio will try it's best to use exactly what you
ask for, but it may differ. You should not assume the number of frames specified in each call to
the data callback is exactly what you originally specified.

The <periods> property controls how frequently the background thread is woken to check for more
data. It's tied to the buffer size, so as an example, if your buffer size is equivalent to 10
milliseconds and you have 2 periods, the CPU will wake up approximately every 5 milliseconds.

When compiling for UWP you must ensure you call this function on the main UI thread because the
operating system may need to present the user with a message asking for permissions. Please refer
to the official documentation for ActivateAudioInterfaceAsync() for more information.

ALSA Specific: When initializing the default device, requesting shared mode will try using the
"dmix" device for playback and the "dsnoop" device for capture. If these fail it will try falling
back to the "hw" device.

Return Value:
  MA_SUCCESS if successful; any other error code otherwise.

Thread Safety: UNSAFE
  It is not safe to call this function simultaneously for different devices because some backends
  depend on and mutate global state (such as OpenSL|ES). The same applies to calling this at the
  same time as ma_device_uninit().
*/
ma_result ma_device_init(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice);

/*
Initializes a device without a context, with extra parameters for controlling the configuration
of the internal self-managed context.

See ma_device_init() and ma_context_init().
*/
ma_result ma_device_init_ex(const ma_backend backends[], ma_uint32 backendCount, const ma_context_config* pContextConfig, const ma_device_config* pConfig, ma_device* pDevice);

/*
Uninitializes a device.

This will explicitly stop the device. You do not need to call ma_device_stop() beforehand, but it's
harmless if you do.

Return Value:
  MA_SUCCESS if successful; any other error code otherwise.

Thread Safety: UNSAFE
  As soon as this API is called the device should be considered undefined. All bets are off if you
  try using the device at the same time as uninitializing it.
*/
void ma_device_uninit(ma_device* pDevice);

/*
Sets the callback to use when the device has stopped, either explicitly or as a result of an error.

Thread Safety: SAFE
  This API is implemented as a simple atomic assignment.
*/
void ma_device_set_stop_callback(ma_device* pDevice, ma_stop_proc proc);

/*
Activates the device. For playback devices this begins playback. For capture devices it begins
recording.

For a playback device, this will retrieve an initial chunk of audio data from the client before
returning. The reason for this is to ensure there is valid audio data in the buffer, which needs
to be done _before_ the device begins playback.

This API waits until the backend device has been started for real by the worker thread. It also
waits on a mutex for thread-safety.

Return Value:
  MA_SUCCESS if successful; any other error code otherwise.

Thread Safety: SAFE
*/
ma_result ma_device_start(ma_device* pDevice);

/*
Puts the device to sleep, but does not uninitialize it. Use ma_device_start() to start it up again.

This API needs to wait on the worker thread to stop the backend device properly before returning. It
also waits on a mutex for thread-safety. In addition, some backends need to wait for the device to
finish playback/recording of the current fragment which can take some time (usually proportionate to
the buffer size that was specified at initialization time).

This should not drop unprocessed samples. Backends are required to either pause the stream in-place
or drain the buffer if pausing is not possible. The reason for this is that stopping the device and
the resuming it with ma_device_start() (which you might do when your program loses focus) may result
in a situation where those samples are never output to the speakers or received from the microphone
which can in turn result in de-syncs.

Return Value:
  MA_SUCCESS if successful; any other error code otherwise.

Thread Safety: SAFE
*/
ma_result ma_device_stop(ma_device* pDevice);

/*
Determines whether or not the device is started.

This is implemented as a simple accessor.

Return Value:
  True if the device is started, false otherwise.

Thread Safety: SAFE
  If another thread calls ma_device_start() or ma_device_stop() at this same time as this function
  is called, there's a very small chance the return value will be out of sync.
*/
ma_bool32 ma_device_is_started(ma_device* pDevice);


/*
Helper function for initializing a ma_context_config object.
*/
ma_context_config ma_context_config_init(void);

/*
Initializes a device config.

By default, the device config will use native device settings (format, channels, sample rate, etc.). Using native
settings means you will get an optimized pass-through data transmission pipeline to and from the device, but you will
need to do all format conversions manually. Normally you would want to use a known format that your program can handle
natively, which you can do by specifying it after this function returns, like so:

    ma_device_config config = ma_device_config_init(ma_device_type_playback);
    config.callback = my_data_callback;
    config.pUserData = pMyUserData;
    config.format = ma_format_f32;
    config.channels = 2;
    config.sampleRate = 44100;

In this case miniaudio will perform all of the necessary data conversion for you behind the scenes.

Currently miniaudio only supports asynchronous, callback based data delivery which means you must specify callback. A
pointer to user data can also be specified which is set in the pUserData member of the ma_device object.

To specify a channel map you can use ma_get_standard_channel_map():

    ma_get_standard_channel_map(ma_standard_channel_map_default, config.channels, config.channelMap);

Alternatively you can set the channel map manually if you need something specific or something that isn't one of miniaudio's
stock channel maps.

By default the system's default device will be used. Set the pDeviceID member to a pointer to a ma_device_id object to 
use a specific device. You can enumerate over the devices with ma_context_enumerate_devices() or ma_context_get_devices()
which will give you access to the device ID. Set pDeviceID to NULL to use the default device.

The device type can be one of the ma_device_type's:
  ma_device_type_playback
  ma_device_type_capture
  ma_device_type_duplex

Thread Safety: SAFE
*/
ma_device_config ma_device_config_init(ma_device_type deviceType);


/************************************************************************************************************************************************************

Utiltities

************************************************************************************************************************************************************/

/*
Creates a mutex.

A mutex must be created from a valid context. A mutex is initially unlocked.
*/
ma_result ma_mutex_init(ma_context* pContext, ma_mutex* pMutex);

/*
Deletes a mutex.
*/
void ma_mutex_uninit(ma_mutex* pMutex);

/*
Locks a mutex with an infinite timeout.
*/
void ma_mutex_lock(ma_mutex* pMutex);

/*
Unlocks a mutex.
*/
void ma_mutex_unlock(ma_mutex* pMutex);


/*
Retrieves a friendly name for a backend.
*/
const char* ma_get_backend_name(ma_backend backend);

/*
Adjust buffer size based on a scaling factor.

This just multiplies the base size by the scaling factor, making sure it's a size of at least 1.
*/
ma_uint32 ma_scale_buffer_size(ma_uint32 baseBufferSize, float scale);

/*
Calculates a buffer size in milliseconds from the specified number of frames and sample rate.
*/
ma_uint32 ma_calculate_buffer_size_in_milliseconds_from_frames(ma_uint32 bufferSizeInFrames, ma_uint32 sampleRate);

/*
Calculates a buffer size in frames from the specified number of milliseconds and sample rate.
*/
ma_uint32 ma_calculate_buffer_size_in_frames_from_milliseconds(ma_uint32 bufferSizeInMilliseconds, ma_uint32 sampleRate);

/*
Retrieves the default buffer size in milliseconds based on the specified performance profile.
*/
ma_uint32 ma_get_default_buffer_size_in_milliseconds(ma_performance_profile performanceProfile);

/*
Calculates a buffer size in frames for the specified performance profile and scale factor.
*/
ma_uint32 ma_get_default_buffer_size_in_frames(ma_performance_profile performanceProfile, ma_uint32 sampleRate);

/*
Copies silent frames into the given buffer.
*/
void ma_zero_pcm_frames(void* p, ma_uint32 frameCount, ma_format format, ma_uint32 channels);

#endif  /* MA_NO_DEVICE_IO */




/************************************************************************************************************************************************************

Decoding

************************************************************************************************************************************************************/
#ifndef MA_NO_DECODING

typedef struct ma_decoder ma_decoder;

typedef enum
{
    ma_seek_origin_start,
    ma_seek_origin_current
} ma_seek_origin;

typedef size_t    (* ma_decoder_read_proc)             (ma_decoder* pDecoder, void* pBufferOut, size_t bytesToRead); /* Returns the number of bytes read. */
typedef ma_bool32 (* ma_decoder_seek_proc)             (ma_decoder* pDecoder, int byteOffset, ma_seek_origin origin);
typedef ma_result (* ma_decoder_seek_to_pcm_frame_proc)(ma_decoder* pDecoder, ma_uint64 frameIndex);
typedef ma_result (* ma_decoder_uninit_proc)           (ma_decoder* pDecoder);

typedef struct
{
    ma_format format;      /* Set to 0 or ma_format_unknown to use the stream's internal format. */
    ma_uint32 channels;    /* Set to 0 to use the stream's internal channels. */
    ma_uint32 sampleRate;  /* Set to 0 to use the stream's internal sample rate. */
    ma_channel channelMap[MA_MAX_CHANNELS];
    ma_channel_mix_mode channelMixMode;
    ma_dither_mode ditherMode;
    ma_src_algorithm srcAlgorithm;
    union
    {
        ma_src_config_sinc sinc;
    } src;
} ma_decoder_config;

struct ma_decoder
{
    ma_decoder_read_proc onRead;
    ma_decoder_seek_proc onSeek;
    void* pUserData;
    ma_uint64  readPointer; /* Used for returning back to a previous position after analysing the stream or whatnot. */
    ma_format  internalFormat;
    ma_uint32  internalChannels;
    ma_uint32  internalSampleRate;
    ma_channel internalChannelMap[MA_MAX_CHANNELS];
    ma_format  outputFormat;
    ma_uint32  outputChannels;
    ma_uint32  outputSampleRate;
    ma_channel outputChannelMap[MA_MAX_CHANNELS];
    ma_pcm_converter dsp;   /* <-- Format conversion is achieved by running frames through this. */
    ma_decoder_seek_to_pcm_frame_proc onSeekToPCMFrame;
    ma_decoder_uninit_proc onUninit;
    void* pInternalDecoder; /* <-- The drwav/drflac/stb_vorbis/etc. objects. */
    struct
    {
        const ma_uint8* pData;
        size_t dataSize;
        size_t currentReadPos;
    } memory;               /* Only used for decoders that were opened against a block of memory. */
};

ma_decoder_config ma_decoder_config_init(ma_format outputFormat, ma_uint32 outputChannels, ma_uint32 outputSampleRate);

ma_result ma_decoder_init(ma_decoder_read_proc onRead, ma_decoder_seek_proc onSeek, void* pUserData, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_wav(ma_decoder_read_proc onRead, ma_decoder_seek_proc onSeek, void* pUserData, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_flac(ma_decoder_read_proc onRead, ma_decoder_seek_proc onSeek, void* pUserData, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_vorbis(ma_decoder_read_proc onRead, ma_decoder_seek_proc onSeek, void* pUserData, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_mp3(ma_decoder_read_proc onRead, ma_decoder_seek_proc onSeek, void* pUserData, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_raw(ma_decoder_read_proc onRead, ma_decoder_seek_proc onSeek, void* pUserData, const ma_decoder_config* pConfigIn, const ma_decoder_config* pConfigOut, ma_decoder* pDecoder);

ma_result ma_decoder_init_memory(const void* pData, size_t dataSize, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_memory_wav(const void* pData, size_t dataSize, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_memory_flac(const void* pData, size_t dataSize, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_memory_vorbis(const void* pData, size_t dataSize, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_memory_mp3(const void* pData, size_t dataSize, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_memory_raw(const void* pData, size_t dataSize, const ma_decoder_config* pConfigIn, const ma_decoder_config* pConfigOut, ma_decoder* pDecoder);

#ifndef MA_NO_STDIO
ma_result ma_decoder_init_file(const char* pFilePath, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
ma_result ma_decoder_init_file_wav(const char* pFilePath, const ma_decoder_config* pConfig, ma_decoder* pDecoder);
#endif

ma_result ma_decoder_uninit(ma_decoder* pDecoder);

ma_uint64 ma_decoder_read_pcm_frames(ma_decoder* pDecoder, void* pFramesOut, ma_uint64 frameCount);
ma_result ma_decoder_seek_to_pcm_frame(ma_decoder* pDecoder, ma_uint64 frameIndex);

/*
Helper for opening and decoding a file into a heap allocated block of memory. Free the returned pointer with ma_free(). On input,
pConfig should be set to what you want. On output it will be set to what you got.
*/
#ifndef MA_NO_STDIO
ma_result ma_decode_file(const char* pFilePath, ma_decoder_config* pConfig, ma_uint64* pFrameCountOut, void** ppDataOut);
#endif
ma_result ma_decode_memory(const void* pData, size_t dataSize, ma_decoder_config* pConfig, ma_uint64* pFrameCountOut, void** ppDataOut);

#endif  /* MA_NO_DECODING */


/************************************************************************************************************************************************************

Generation

************************************************************************************************************************************************************/
typedef struct
{
    double amplitude;
    double periodsPerSecond;
    double delta;
    double time;
} ma_sine_wave;

ma_result ma_sine_wave_init(double amplitude, double period, ma_uint32 sampleRate, ma_sine_wave* pSineWave);
ma_uint64 ma_sine_wave_read_f32(ma_sine_wave* pSineWave, ma_uint64 count, float* pSamples);
ma_uint64 ma_sine_wave_read_f32_ex(ma_sine_wave* pSineWave, ma_uint64 frameCount, ma_uint32 channels, ma_stream_layout layout, float** ppFrames);

#ifdef __cplusplus
}
#endif
#endif  /* miniaudio_h */



/************************************************************************************************************************************************************
*************************************************************************************************************************************************************

IMPLEMENTATION

*************************************************************************************************************************************************************
************************************************************************************************************************************************************/
#if defined(MINIAUDIO_IMPLEMENTATION) || defined(MA_IMPLEMENTATION)
#include <assert.h>
#include <limits.h> /* For INT_MAX */
#include <math.h>   /* sin(), etc. */

#if defined(MA_DEBUG_OUTPUT)
#include <stdio.h>  /* for printf() for debug output */
#endif

#ifdef MA_WIN32
#include <windows.h>
#include <objbase.h>
#include <mmreg.h>
#include <mmsystem.h>
#else
#include <stdlib.h> /* For malloc()/free() */
#include <string.h> /* For memset() */
#endif

#if defined(MA_APPLE) && (__MAC_OS_X_VERSION_MIN_REQUIRED < 101200)
#include <mach/mach_time.h> /* For mach_absolute_time() */
#endif

#ifdef MA_POSIX
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <dlfcn.h>
#endif

#ifdef MA_EMSCRIPTEN
#include <emscripten/emscripten.h>
#endif

#if !defined(MA_64BIT) && !defined(MA_32BIT)
#ifdef _WIN32
#ifdef _WIN64
#define MA_64BIT
#else
#define MA_32BIT
#endif
#endif
#endif

#if !defined(MA_64BIT) && !defined(MA_32BIT)
#ifdef __GNUC__
#ifdef __LP64__
#define MA_64BIT
#else
#define MA_32BIT
#endif
#endif
#endif

#if !defined(MA_64BIT) && !defined(MA_32BIT)
#include <stdint.h>
#if INTPTR_MAX == INT64_MAX
#define MA_64BIT
#else
#define MA_32BIT
#endif
#endif

/* Architecture Detection */
#if defined(__x86_64__) || defined(_M_X64)
#define MA_X64
#elif defined(__i386) || defined(_M_IX86)
#define MA_X86
#elif defined(__arm__) || defined(_M_ARM)
#define MA_ARM
#endif

/* Cannot currently support AVX-512 if AVX is disabled. */
#if !defined(MA_NO_AVX512) && defined(MA_NO_AVX2)
#define MA_NO_AVX512
#endif

/* Intrinsics Support */
#if defined(MA_X64) || defined(MA_X86)
    #if defined(_MSC_VER) && !defined(__clang__)
        /* MSVC. */
        #if !defined(MA_NO_SSE2)   /* Assume all MSVC compilers support SSE2 intrinsics. */
            #define MA_SUPPORT_SSE2
        #endif
        /*#if _MSC_VER >= 1600 && !defined(MA_NO_AVX)*/    /* 2010 */
        /*    #define MA_SUPPORT_AVX*/
        /*#endif*/
        #if _MSC_VER >= 1700 && !defined(MA_NO_AVX2)   /* 2012 */
            #define MA_SUPPORT_AVX2
        #endif
        #if _MSC_VER >= 1910 && !defined(MA_NO_AVX512) /* 2017 */
            #define MA_SUPPORT_AVX512
        #endif
    #else
        /* Assume GNUC-style. */
        #if defined(__SSE2__) && !defined(MA_NO_SSE2)
            #define MA_SUPPORT_SSE2
        #endif
        /*#if defined(__AVX__) && !defined(MA_NO_AVX)*/
        /*    #define MA_SUPPORT_AVX*/
        /*#endif*/
        #if defined(__AVX2__) && !defined(MA_NO_AVX2)
            #define MA_SUPPORT_AVX2
        #endif
        #if defined(__AVX512F__) && !defined(MA_NO_AVX512)
            #define MA_SUPPORT_AVX512
        #endif
    #endif

    /* If at this point we still haven't determined compiler support for the intrinsics just fall back to __has_include. */
    #if !defined(__GNUC__) && !defined(__clang__) && defined(__has_include)
        #if !defined(MA_SUPPORT_SSE2)   && !defined(MA_NO_SSE2)   && __has_include(<emmintrin.h>)
            #define MA_SUPPORT_SSE2
        #endif
        /*#if !defined(MA_SUPPORT_AVX)    && !defined(MA_NO_AVX)    && __has_include(<immintrin.h>)*/
        /*    #define MA_SUPPORT_AVX*/
        /*#endif*/
        #if !defined(MA_SUPPORT_AVX2)   && !defined(MA_NO_AVX2)   && __has_include(<immintrin.h>)
            #define MA_SUPPORT_AVX2
        #endif
        #if !defined(MA_SUPPORT_AVX512) && !defined(MA_NO_AVX512) && __has_include(<zmmintrin.h>)
            #define MA_SUPPORT_AVX512
        #endif
    #endif

    #if defined(MA_SUPPORT_AVX512)
        #include <immintrin.h>  /* Not a mistake. Intentionally including <immintrin.h> instead of <zmmintrin.h> because otherwise the compiler will complain. */
    #elif defined(MA_SUPPORT_AVX2) || defined(MA_SUPPORT_AVX)
        #include <immintrin.h>
    #elif defined(MA_SUPPORT_SSE2)
        #include <emmintrin.h>
    #endif
#endif

#if defined(MA_ARM)
    #if !defined(MA_NO_NEON) && (defined(__ARM_NEON) || defined(__aarch64__) || defined(_M_ARM64))
        #define MA_SUPPORT_NEON
    #endif

    /* Fall back to looking for the #include file. */
    #if !defined(__GNUC__) && !defined(__clang__) && defined(__has_include)
        #if !defined(MA_SUPPORT_NEON) && !defined(MA_NO_NEON) && __has_include(<arm_neon.h>)
            #define MA_SUPPORT_NEON
        #endif
    #endif

    #if defined(MA_SUPPORT_NEON)
        #include <arm_neon.h>
    #endif
#endif

#if defined(_MSC_VER)
    #pragma warning(push)
    #pragma warning(disable:4752)   /* found Intel(R) Advanced Vector Extensions; consider using /arch:AVX */
#endif

#if defined(MA_X64) || defined(MA_X86)
    #if defined(_MSC_VER) && !defined(__clang__)
        #if _MSC_VER >= 1400
            #include <intrin.h>
            static MA_INLINE void ma_cpuid(int info[4], int fid)
            {
                __cpuid(info, fid);
            }
        #else
            #define MA_NO_CPUID
        #endif

        #if _MSC_VER >= 1600
            static MA_INLINE unsigned __int64 ma_xgetbv(int reg)
            {
                return _xgetbv(reg);
            }
        #else
            #define MA_NO_XGETBV
        #endif
    #elif (defined(__GNUC__) || defined(__clang__)) && !defined(MA_ANDROID)
        static MA_INLINE void ma_cpuid(int info[4], int fid)
        {
            /*
            It looks like the -fPIC option uses the ebx register which GCC complains about. We can work around this by just using a different register, the
            specific register of which I'm letting the compiler decide on. The "k" prefix is used to specify a 32-bit register. The {...} syntax is for
            supporting different assembly dialects.
            
            What's basically happening is that we're saving and restoring the ebx register manually.
            */
            #if defined(DRFLAC_X86) && defined(__PIC__)
                __asm__ __volatile__ (
                    "xchg{l} {%%}ebx, %k1;"
                    "cpuid;"
                    "xchg{l} {%%}ebx, %k1;"
                    : "=a"(info[0]), "=&r"(info[1]), "=c"(info[2]), "=d"(info[3]) : "a"(fid), "c"(0)
                );
            #else
                __asm__ __volatile__ (
                    "cpuid" : "=a"(info[0]), "=b"(info[1]), "=c"(info[2]), "=d"(info[3]) : "a"(fid), "c"(0)
                );
            #endif
        }

        static MA_INLINE ma_uint64 ma_xgetbv(int reg)
        {
            unsigned int hi;
            unsigned int lo;

            __asm__ __volatile__ (
                "xgetbv" : "=a"(lo), "=d"(hi) : "c"(reg)
            );

            return ((ma_uint64)hi << 32) | (ma_uint64)lo;
        }
    #else
        #define MA_NO_CPUID
        #define MA_NO_XGETBV
    #endif
#else
    #define MA_NO_CPUID
    #define MA_NO_XGETBV
#endif

static MA_INLINE ma_bool32 ma_has_sse2()
{
#if defined(MA_SUPPORT_SSE2)
    #if (defined(MA_X64) || defined(MA_X86)) && !defined(MA_NO_SSE2)
        #if defined(MA_X64)
            return MA_TRUE;    /* 64-bit targets always support SSE2. */
        #elif (defined(_M_IX86_FP) && _M_IX86_FP == 2) || defined(__SSE2__)
            return MA_TRUE;    /* If the compiler is allowed to freely generate SSE2 code we can assume support. */
        #else
            #if defined(MA_NO_CPUID)
                return MA_FALSE;
            #else
                int info[4];
                ma_cpuid(info, 1);
                return (info[3] & (1 << 26)) != 0;
            #endif
        #endif
    #else
        return MA_FALSE;       /* SSE2 is only supported on x86 and x64 architectures. */
    #endif
#else
    return MA_FALSE;           /* No compiler support. */
#endif
}

#if 0
static MA_INLINE ma_bool32 ma_has_avx()
{
#if defined(MA_SUPPORT_AVX)
    #if (defined(MA_X64) || defined(MA_X86)) && !defined(MA_NO_AVX)
        #if defined(_AVX_) || defined(__AVX__)
            return MA_TRUE;    /* If the compiler is allowed to freely generate AVX code we can assume support. */
        #else
            /* AVX requires both CPU and OS support. */
            #if defined(MA_NO_CPUID) || defined(MA_NO_XGETBV)
                return MA_FALSE;
            #else
                int info[4];
                ma_cpuid(info, 1);
                if (((info[2] & (1 << 27)) != 0) && ((info[2] & (1 << 28)) != 0)) {
                    ma_uint64 xrc = ma_xgetbv(0);
                    if ((xrc & 0x06) == 0x06) {
                        return MA_TRUE;
                    } else {
                        return MA_FALSE;
                    }
                } else {
                    return MA_FALSE;
                }
            #endif
        #endif
    #else
        return MA_FALSE;       /* AVX is only supported on x86 and x64 architectures. */
    #endif
#else
    return MA_FALSE;           /* No compiler support. */
#endif
}
#endif

static MA_INLINE ma_bool32 ma_has_avx2()
{
#if defined(MA_SUPPORT_AVX2)
    #if (defined(MA_X64) || defined(MA_X86)) && !defined(MA_NO_AVX2)
        #if defined(_AVX2_) || defined(__AVX2__)
            return MA_TRUE;    /* If the compiler is allowed to freely generate AVX2 code we can assume support. */
        #else
            /* AVX2 requires both CPU and OS support. */
            #if defined(MA_NO_CPUID) || defined(MA_NO_XGETBV)
                return MA_FALSE;
            #else
                int info1[4];
                int info7[4];
                ma_cpuid(info1, 1);
                ma_cpuid(info7, 7);
                if (((info1[2] & (1 << 27)) != 0) && ((info7[1] & (1 << 5)) != 0)) {
                    ma_uint64 xrc = ma_xgetbv(0);
                    if ((xrc & 0x06) == 0x06) {
                        return MA_TRUE;
                    } else {
                        return MA_FALSE;
                    }
                } else {
                    return MA_FALSE;
                }
            #endif
        #endif
    #else
        return MA_FALSE;       /* AVX2 is only supported on x86 and x64 architectures. */
    #endif
#else
    return MA_FALSE;           /* No compiler support. */
#endif
}

static MA_INLINE ma_bool32 ma_has_avx512f()
{
#if defined(MA_SUPPORT_AVX512)
    #if (defined(MA_X64) || defined(MA_X86)) && !defined(MA_NO_AVX512)
        #if defined(__AVX512F__)
            return MA_TRUE;    /* If the compiler is allowed to freely generate AVX-512F code we can assume support. */
        #else
            /* AVX-512 requires both CPU and OS support. */
            #if defined(MA_NO_CPUID) || defined(MA_NO_XGETBV)
                return MA_FALSE;
            #else
                int info1[4];
                int info7[4];
                ma_cpuid(info1, 1);
                ma_cpuid(info7, 7);
                if (((info1[2] & (1 << 27)) != 0) && ((info7[1] & (1 << 16)) != 0)) {
                    ma_uint64 xrc = ma_xgetbv(0);
                    if ((xrc & 0xE6) == 0xE6) {
                        return MA_TRUE;
                    } else {
                        return MA_FALSE;
                    }
                } else {
                    return MA_FALSE;
                }
            #endif
        #endif
    #else
        return MA_FALSE;       /* AVX-512F is only supported on x86 and x64 architectures. */
    #endif
#else
    return MA_FALSE;           /* No compiler support. */
#endif
}

static MA_INLINE ma_bool32 ma_has_neon()
{
#if defined(MA_SUPPORT_NEON)
    #if defined(MA_ARM) && !defined(MA_NO_NEON)
        #if (defined(__ARM_NEON) || defined(__aarch64__) || defined(_M_ARM64))
            return MA_TRUE;    /* If the compiler is allowed to freely generate NEON code we can assume support. */
        #else
            /* TODO: Runtime check. */
            return MA_FALSE;
        #endif
    #else
        return MA_FALSE;       /* NEON is only supported on ARM architectures. */
    #endif
#else
    return MA_FALSE;           /* No compiler support. */
#endif
}


static MA_INLINE ma_bool32 ma_is_little_endian()
{
#if defined(MA_X86) || defined(MA_X64)
    return MA_TRUE;
#else
    int n = 1;
    return (*(char*)&n) == 1;
#endif
}

static MA_INLINE ma_bool32 ma_is_big_endian()
{
    return !ma_is_little_endian();
}


#ifndef MA_COINIT_VALUE
#define MA_COINIT_VALUE    0   /* 0 = COINIT_MULTITHREADED*/
#endif



#ifndef MA_PI
#define MA_PI      3.14159265358979323846264f
#endif
#ifndef MA_PI_D
#define MA_PI_D    3.14159265358979323846264
#endif
#ifndef MA_TAU
#define MA_TAU     6.28318530717958647693f
#endif
#ifndef MA_TAU_D
#define MA_TAU_D   6.28318530717958647693
#endif


/* The default format when ma_format_unknown (0) is requested when initializing a device. */
#ifndef MA_DEFAULT_FORMAT
#define MA_DEFAULT_FORMAT                                  ma_format_f32
#endif

/* The default channel count to use when 0 is used when initializing a device. */
#ifndef MA_DEFAULT_CHANNELS
#define MA_DEFAULT_CHANNELS                                2
#endif

/* The default sample rate to use when 0 is used when initializing a device. */
#ifndef MA_DEFAULT_SAMPLE_RATE
#define MA_DEFAULT_SAMPLE_RATE                             48000
#endif

/* Default periods when none is specified in ma_device_init(). More periods means more work on the CPU. */
#ifndef MA_DEFAULT_PERIODS
#define MA_DEFAULT_PERIODS                                 3
#endif

/* The base buffer size in milliseconds for low latency mode. */
#ifndef MA_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY
#define MA_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY    (10*MA_DEFAULT_PERIODS)
#endif

/* The base buffer size in milliseconds for conservative mode. */
#ifndef MA_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE
#define MA_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE   (100*MA_DEFAULT_PERIODS)
#endif


/* Standard sample rates, in order of priority. */
ma_uint32 g_maStandardSampleRatePriorities[] = {
    MA_SAMPLE_RATE_48000,  /* Most common */
    MA_SAMPLE_RATE_44100,

    MA_SAMPLE_RATE_32000,  /* Lows */
    MA_SAMPLE_RATE_24000,
    MA_SAMPLE_RATE_22050,

    MA_SAMPLE_RATE_88200,  /* Highs */
    MA_SAMPLE_RATE_96000,
    MA_SAMPLE_RATE_176400,
    MA_SAMPLE_RATE_192000,

    MA_SAMPLE_RATE_16000,  /* Extreme lows */
    MA_SAMPLE_RATE_11025,
    MA_SAMPLE_RATE_8000,

    MA_SAMPLE_RATE_352800, /* Extreme highs */
    MA_SAMPLE_RATE_384000
};

ma_format g_maFormatPriorities[] = {
    ma_format_s16,         /* Most common */
    ma_format_f32,
    
    /*ma_format_s24_32,*/    /* Clean alignment */
    ma_format_s32,
    
    ma_format_s24,         /* Unclean alignment */
    
    ma_format_u8           /* Low quality */
};



/******************************************************************************

Standard Library Stuff

******************************************************************************/
#ifndef MA_MALLOC
#ifdef MA_WIN32
#define MA_MALLOC(sz) HeapAlloc(GetProcessHeap(), 0, (sz))
#else
#define MA_MALLOC(sz) malloc((sz))
#endif
#endif

#ifndef MA_REALLOC
#ifdef MA_WIN32
#define MA_REALLOC(p, sz) (((sz) > 0) ? ((p) ? HeapReAlloc(GetProcessHeap(), 0, (p), (sz)) : HeapAlloc(GetProcessHeap(), 0, (sz))) : ((VOID*)(size_t)(HeapFree(GetProcessHeap(), 0, (p)) & 0)))
#else
#define MA_REALLOC(p, sz) realloc((p), (sz))
#endif
#endif

#ifndef MA_FREE
#ifdef MA_WIN32
#define MA_FREE(p) HeapFree(GetProcessHeap(), 0, (p))
#else
#define MA_FREE(p) free((p))
#endif
#endif

#ifndef MA_ZERO_MEMORY
#ifdef MA_WIN32
#define MA_ZERO_MEMORY(p, sz) ZeroMemory((p), (sz))
#else
#define MA_ZERO_MEMORY(p, sz) memset((p), 0, (sz))
#endif
#endif

#ifndef MA_COPY_MEMORY
#ifdef MA_WIN32
#define MA_COPY_MEMORY(dst, src, sz) CopyMemory((dst), (src), (sz))
#else
#define MA_COPY_MEMORY(dst, src, sz) memcpy((dst), (src), (sz))
#endif
#endif

#ifndef MA_ASSERT
#ifdef MA_WIN32
#define MA_ASSERT(condition) assert(condition)
#else
#define MA_ASSERT(condition) assert(condition)
#endif
#endif

#define ma_zero_memory MA_ZERO_MEMORY
#define ma_copy_memory MA_COPY_MEMORY
#define ma_assert      MA_ASSERT

#define ma_zero_object(p)          ma_zero_memory((p), sizeof(*(p)))
#define ma_countof(x)              (sizeof(x) / sizeof(x[0]))
#define ma_max(x, y)               (((x) > (y)) ? (x) : (y))
#define ma_min(x, y)               (((x) < (y)) ? (x) : (y))
#define ma_clamp(x, lo, hi)        (ma_max(lo, ma_min(x, hi)))
#define ma_offset_ptr(p, offset)   (((ma_uint8*)(p)) + (offset))

#define ma_buffer_frame_capacity(buffer, channels, format) (sizeof(buffer) / ma_get_bytes_per_sample(format) / (channels))

/*
Return Values:
  0:  Success
  22: EINVAL
  34: ERANGE

Not using symbolic constants for errors because I want to avoid #including errno.h
*/
int ma_strcpy_s(char* dst, size_t dstSizeInBytes, const char* src)
{
    size_t i;

    if (dst == 0) {
        return 22;
    }
    if (dstSizeInBytes == 0) {
        return 34;
    }
    if (src == 0) {
        dst[0] = '\0';
        return 22;
    }

    for (i = 0; i < dstSizeInBytes && src[i] != '\0'; ++i) {
        dst[i] = src[i];
    }

    if (i < dstSizeInBytes) {
        dst[i] = '\0';
        return 0;
    }

    dst[0] = '\0';
    return 34;
}

int ma_strncpy_s(char* dst, size_t dstSizeInBytes, const char* src, size_t count)
{
    size_t maxcount;
    size_t i;

    if (dst == 0) {
        return 22;
    }
    if (dstSizeInBytes == 0) {
        return 34;
    }
    if (src == 0) {
        dst[0] = '\0';
        return 22;
    }

    maxcount = count;
    if (count == ((size_t)-1) || count >= dstSizeInBytes) {        /* -1 = _TRUNCATE */
        maxcount = dstSizeInBytes - 1;
    }

    for (i = 0; i < maxcount && src[i] != '\0'; ++i) {
        dst[i] = src[i];
    }

    if (src[i] == '\0' || i == count || count == ((size_t)-1)) {
        dst[i] = '\0';
        return 0;
    }

    dst[0] = '\0';
    return 34;
}

int ma_strcat_s(char* dst, size_t dstSizeInBytes, const char* src)
{
    char* dstorig;

    if (dst == 0) {
        return 22;
    }
    if (dstSizeInBytes == 0) {
        return 34;
    }
    if (src == 0) {
        dst[0] = '\0';
        return 22;
    }

    dstorig = dst;

    while (dstSizeInBytes > 0 && dst[0] != '\0') {
        dst += 1;
        dstSizeInBytes -= 1;
    }

    if (dstSizeInBytes == 0) {
        return 22;  /* Unterminated. */
    }


    while (dstSizeInBytes > 0 && src[0] != '\0') {
        *dst++ = *src++;
        dstSizeInBytes -= 1;
    }

    if (dstSizeInBytes > 0) {
        dst[0] = '\0';
    } else {
        dstorig[0] = '\0';
        return 34;
    }

    return 0;
}

int ma_itoa_s(int value, char* dst, size_t dstSizeInBytes, int radix)
{
    int sign;
    unsigned int valueU;
    char* dstEnd;

    if (dst == NULL || dstSizeInBytes == 0) {
        return 22;
    }
    if (radix < 2 || radix > 36) {
        dst[0] = '\0';
        return 22;
    }

    sign = (value < 0 && radix == 10) ? -1 : 1;     /* The negative sign is only used when the base is 10. */

    if (value < 0) {
        valueU = -value;
    } else {
        valueU = value;
    }

    dstEnd = dst;
    do
    {
        int remainder = valueU % radix;
        if (remainder > 9) {
            *dstEnd = (char)((remainder - 10) + 'a');
        } else {
            *dstEnd = (char)(remainder + '0');
        }

        dstEnd += 1;
        dstSizeInBytes -= 1;
        valueU /= radix;
    } while (dstSizeInBytes > 0 && valueU > 0);

    if (dstSizeInBytes == 0) {
        dst[0] = '\0';
        return 22;  /* Ran out of room in the output buffer. */
    }

    if (sign < 0) {
        *dstEnd++ = '-';
        dstSizeInBytes -= 1;
    }

    if (dstSizeInBytes == 0) {
        dst[0] = '\0';
        return 22;  /* Ran out of room in the output buffer. */
    }

    *dstEnd = '\0';


    /* At this point the string will be reversed. */
    dstEnd -= 1;
    while (dst < dstEnd) {
        char temp = *dst;
        *dst = *dstEnd;
        *dstEnd = temp;

        dst += 1;
        dstEnd -= 1;
    }

    return 0;
}

int ma_strcmp(const char* str1, const char* str2)
{
    if (str1 == str2) return  0;

    /* These checks differ from the standard implementation. It's not important, but I prefer it just for sanity. */
    if (str1 == NULL) return -1;
    if (str2 == NULL) return  1;

    for (;;) {
        if (str1[0] == '\0') {
            break;
        }
        if (str1[0] != str2[0]) {
            break;
        }

        str1 += 1;
        str2 += 1;
    }

    return ((unsigned char*)str1)[0] - ((unsigned char*)str2)[0];
}

char* ma_copy_string(const char* src)
{
    size_t sz = strlen(src)+1;
    char* dst = (char*)ma_malloc(sz);
    if (dst == NULL) {
        return NULL;
    }

    ma_strcpy_s(dst, sz, src);

    return dst;
}


/* Thanks to good old Bit Twiddling Hacks for this one: http://graphics.stanford.edu/~seander/bithacks.html#RoundUpPowerOf2 */
static MA_INLINE unsigned int ma_next_power_of_2(unsigned int x)
{
    x--;
    x |= x >> 1;
    x |= x >> 2;
    x |= x >> 4;
    x |= x >> 8;
    x |= x >> 16;
    x++;

    return x;
}

static MA_INLINE unsigned int ma_prev_power_of_2(unsigned int x)
{
    return ma_next_power_of_2(x) >> 1;
}

static MA_INLINE unsigned int ma_round_to_power_of_2(unsigned int x)
{
    unsigned int prev = ma_prev_power_of_2(x);
    unsigned int next = ma_next_power_of_2(x);
    if ((next - x) > (x - prev)) {
        return prev;
    } else {
        return next;
    }
}

static MA_INLINE unsigned int ma_count_set_bits(unsigned int x)
{
    unsigned int count = 0;
    while (x != 0) {
        if (x & 1) {
            count += 1;
        }
        
        x = x >> 1;
    }
    
    return count;
}



/* Clamps an f32 sample to -1..1 */
static MA_INLINE float ma_clip_f32(float x)
{
    if (x < -1) return -1;
    if (x > +1) return +1;
    return x;
}

static MA_INLINE float ma_mix_f32(float x, float y, float a)
{
    return x*(1-a) + y*a;
}
static MA_INLINE float ma_mix_f32_fast(float x, float y, float a)
{
    float r0 = (y - x);
    float r1 = r0*a;
    return x + r1;
    /*return x + (y - x)*a;*/
}

#if defined(MA_SUPPORT_SSE2)
static MA_INLINE __m128 ma_mix_f32_fast__sse2(__m128 x, __m128 y, __m128 a)
{
    return _mm_add_ps(x, _mm_mul_ps(_mm_sub_ps(y, x), a));
}
#endif
#if defined(MA_SUPPORT_AVX2)
static MA_INLINE __m256 ma_mix_f32_fast__avx2(__m256 x, __m256 y, __m256 a)
{
    return _mm256_add_ps(x, _mm256_mul_ps(_mm256_sub_ps(y, x), a));
}
#endif
#if defined(MA_SUPPORT_AVX512)
static MA_INLINE __m512 ma_mix_f32_fast__avx512(__m512 x, __m512 y, __m512 a)
{
    return _mm512_add_ps(x, _mm512_mul_ps(_mm512_sub_ps(y, x), a));
}
#endif
#if defined(MA_SUPPORT_NEON)
static MA_INLINE float32x4_t ma_mix_f32_fast__neon(float32x4_t x, float32x4_t y, float32x4_t a)
{
    return vaddq_f32(x, vmulq_f32(vsubq_f32(y, x), a));
}
#endif


static MA_INLINE double ma_mix_f64(double x, double y, double a)
{
    return x*(1-a) + y*a;
}
static MA_INLINE double ma_mix_f64_fast(double x, double y, double a)
{
    return x + (y - x)*a;
}

static MA_INLINE float ma_scale_to_range_f32(float x, float lo, float hi)
{
    return lo + x*(hi-lo);
}


/*
Random Number Generation

miniaudio uses the LCG random number generation algorithm. This is good enough for audio.

Note that miniaudio's LCG implementation uses global state which is _not_ thread-local. When this is called across
multiple threads, results will be unpredictable. However, it won't crash and results will still be random enough
for miniaudio's purposes.
*/
#define MA_LCG_M   2147483647
#define MA_LCG_A   48271
#define MA_LCG_C   0
static ma_int32 g_maLCG;

void ma_seed(ma_int32 seed)
{
    g_maLCG = seed;
}

ma_int32 ma_rand_s32()
{
    ma_int32 lcg = g_maLCG;
    ma_int32 r = (MA_LCG_A * lcg + MA_LCG_C) % MA_LCG_M;
    g_maLCG = r;
    return r;
}

double ma_rand_f64()
{
    return (ma_rand_s32() + 0x80000000) / (double)0x7FFFFFFF;
}

float ma_rand_f32()
{
    return (float)ma_rand_f64();
}

static MA_INLINE float ma_rand_range_f32(float lo, float hi)
{
    return ma_scale_to_range_f32(ma_rand_f32(), lo, hi);
}

static MA_INLINE ma_int32 ma_rand_range_s32(ma_int32 lo, ma_int32 hi)
{
    double x = ma_rand_f64();
    return lo + (ma_int32)(x*(hi-lo));
}


static MA_INLINE float ma_dither_f32_rectangle(float ditherMin, float ditherMax)
{
    return ma_rand_range_f32(ditherMin, ditherMax);
}

static MA_INLINE float ma_dither_f32_triangle(float ditherMin, float ditherMax)
{
    float a = ma_rand_range_f32(ditherMin, 0);
    float b = ma_rand_range_f32(0, ditherMax);
    return a + b;
}

static MA_INLINE float ma_dither_f32(ma_dither_mode ditherMode, float ditherMin, float ditherMax)
{
    if (ditherMode == ma_dither_mode_rectangle) {
        return ma_dither_f32_rectangle(ditherMin, ditherMax);
    }
    if (ditherMode == ma_dither_mode_triangle) {
        return ma_dither_f32_triangle(ditherMin, ditherMax);
    }

    return 0;
}

static MA_INLINE ma_int32 ma_dither_s32(ma_dither_mode ditherMode, ma_int32 ditherMin, ma_int32 ditherMax)
{
    if (ditherMode == ma_dither_mode_rectangle) {
        ma_int32 a = ma_rand_range_s32(ditherMin, ditherMax);
        return a;
    }
    if (ditherMode == ma_dither_mode_triangle) {
        ma_int32 a = ma_rand_range_s32(ditherMin, 0);
        ma_int32 b = ma_rand_range_s32(0, ditherMax);
        return a + b;
    }

    return 0;
}


/*
Splits a buffer into parts of equal length and of the given alignment. The returned size of the split buffers will be a
multiple of the alignment. The alignment must be a power of 2.
*/
void ma_split_buffer(void* pBuffer, size_t bufferSize, size_t splitCount, size_t alignment, void** ppBuffersOut, size_t* pSplitSizeOut)
{
    ma_uintptr pBufferUnaligned;
    ma_uintptr pBufferAligned;
    size_t unalignedBytes;
    size_t splitSize;

    if (pSplitSizeOut) {
        *pSplitSizeOut = 0;
    }

    if (pBuffer == NULL || bufferSize == 0 || splitCount == 0) {
        return;
    }

    if (alignment == 0) {
        alignment = 1;
    }

    pBufferUnaligned = (ma_uintptr)pBuffer;
    pBufferAligned   = (pBufferUnaligned + (alignment-1)) & ~(alignment-1);
    unalignedBytes   = (size_t)(pBufferAligned - pBufferUnaligned);

    splitSize = 0;
    if (bufferSize >= unalignedBytes) {
        splitSize = (bufferSize - unalignedBytes) / splitCount;
        splitSize = splitSize & ~(alignment-1);
    }

    if (ppBuffersOut != NULL) {
        size_t i;
        for (i = 0; i < splitCount; ++i) {
            ppBuffersOut[i] = (ma_uint8*)(pBufferAligned + (splitSize*i));
        }
    }

    if (pSplitSizeOut) {
        *pSplitSizeOut = splitSize;
    }
}


/******************************************************************************

Atomics

******************************************************************************/
#if defined(_WIN32) && !defined(__GNUC__)
#define ma_memory_barrier()            MemoryBarrier()
#define ma_atomic_exchange_32(a, b)    InterlockedExchange((LONG*)a, (LONG)b)
#define ma_atomic_exchange_64(a, b)    InterlockedExchange64((LONGLONG*)a, (LONGLONG)b)
#define ma_atomic_increment_32(a)      InterlockedIncrement((LONG*)a)
#define ma_atomic_decrement_32(a)      InterlockedDecrement((LONG*)a)
#else
#define ma_memory_barrier()            __sync_synchronize()
#define ma_atomic_exchange_32(a, b)    (void)__sync_lock_test_and_set(a, b); __sync_synchronize()
#define ma_atomic_exchange_64(a, b)    (void)__sync_lock_test_and_set(a, b); __sync_synchronize()
#define ma_atomic_increment_32(a)      __sync_add_and_fetch(a, 1)
#define ma_atomic_decrement_32(a)      __sync_sub_and_fetch(a, 1)
#endif

#ifdef MA_64BIT
#define ma_atomic_exchange_ptr ma_atomic_exchange_64
#endif
#ifdef MA_32BIT
#define ma_atomic_exchange_ptr ma_atomic_exchange_32
#endif


ma_uint32 ma_get_standard_sample_rate_priority_index(ma_uint32 sampleRate)   /* Lower = higher priority */
{
    ma_uint32 i;
    for (i = 0; i < ma_countof(g_maStandardSampleRatePriorities); ++i) {
        if (g_maStandardSampleRatePriorities[i] == sampleRate) {
            return i;
        }
    }

    return (ma_uint32)-1;
}

ma_uint64 ma_calculate_frame_count_after_src(ma_uint32 sampleRateOut, ma_uint32 sampleRateIn, ma_uint64 frameCountIn)
{
    double    srcRatio       = (double)sampleRateOut / sampleRateIn;
    double    frameCountOutF = frameCountIn * srcRatio;
    ma_uint64 frameCountOut  = (ma_uint64)frameCountOutF;

    /* If the output frame count is fractional, make sure we add an extra frame to ensure there's enough room for that last sample. */
    if ((frameCountOutF - frameCountOut) > 0.0) {
        frameCountOut += 1;
    }

    return frameCountOut;
}


/************************************************************************************************************************************************************
*************************************************************************************************************************************************************

DEVICE I/O
==========

*************************************************************************************************************************************************************
************************************************************************************************************************************************************/
#ifndef MA_NO_DEVICE_IO
/*
Unfortunately using runtime linking for pthreads causes problems. This has occurred for me when testing on FreeBSD. When
using runtime linking, deadlocks can occur (for me it happens when loading data from fread()). It turns out that doing
compile-time linking fixes this. I'm not sure why this happens, but the safest way I can think of to fix this is to simply
disable runtime linking by default. To enable runtime linking, #define this before the implementation of this file. I am
not officially supporting this, but I'm leaving it here in case it's useful for somebody, somewhere.
*/
/*#define MA_USE_RUNTIME_LINKING_FOR_PTHREAD*/

/* Disable run-time linking on certain backends. */
#ifndef MA_NO_RUNTIME_LINKING
    #if defined(MA_ANDROID) || defined(MA_EMSCRIPTEN)
        #define MA_NO_RUNTIME_LINKING
    #endif
#endif

/*
Check if we have the necessary development packages for each backend at the top so we can use this to determine whether or not
certain unused functions and variables can be excluded from the build to avoid warnings.
*/
#ifdef MA_ENABLE_WASAPI
    #define MA_HAS_WASAPI      /* Every compiler should support WASAPI */
#endif
#ifdef MA_ENABLE_DSOUND
    #define MA_HAS_DSOUND      /* Every compiler should support DirectSound. */
#endif
#ifdef MA_ENABLE_WINMM
    #define MA_HAS_WINMM       /* Every compiler I'm aware of supports WinMM. */
#endif
#ifdef MA_ENABLE_ALSA
    #define MA_HAS_ALSA
    #ifdef MA_NO_RUNTIME_LINKING
        #ifdef __has_include
            #if !__has_include(<alsa/asoundlib.h>)
                #undef MA_HAS_ALSA
            #endif
        #endif
    #endif
#endif
#ifdef MA_ENABLE_PULSEAUDIO
    #define MA_HAS_PULSEAUDIO
    #ifdef MA_NO_RUNTIME_LINKING
        #ifdef __has_include
            #if !__has_include(<pulse/pulseaudio.h>)
                #undef MA_HAS_PULSEAUDIO
            #endif
        #endif
    #endif
#endif
#ifdef MA_ENABLE_JACK
    #define MA_HAS_JACK
    #ifdef MA_NO_RUNTIME_LINKING
        #ifdef __has_include
            #if !__has_include(<jack/jack.h>)
                #undef MA_HAS_JACK
            #endif
        #endif
    #endif
#endif
#ifdef MA_ENABLE_COREAUDIO
    #define MA_HAS_COREAUDIO
#endif
#ifdef MA_ENABLE_SNDIO
    #define MA_HAS_SNDIO
#endif
#ifdef MA_ENABLE_AUDIO4
    #define MA_HAS_AUDIO4
#endif
#ifdef MA_ENABLE_OSS
    #define MA_HAS_OSS
#endif
#ifdef MA_ENABLE_AAUDIO
    #define MA_HAS_AAUDIO
#endif
#ifdef MA_ENABLE_OPENSL
    #define MA_HAS_OPENSL
#endif
#ifdef MA_ENABLE_WEBAUDIO
    #define MA_HAS_WEBAUDIO
#endif
#ifdef MA_ENABLE_NULL
    #define MA_HAS_NULL    /* Everything supports the null backend. */
#endif

const char* ma_get_backend_name(ma_backend backend)
{
    switch (backend)
    {
        case ma_backend_wasapi:     return "WASAPI";
        case ma_backend_dsound:     return "DirectSound";
        case ma_backend_winmm:      return "WinMM";
        case ma_backend_coreaudio:  return "Core Audio";
        case ma_backend_sndio:      return "sndio";
        case ma_backend_audio4:     return "audio(4)";
        case ma_backend_oss:        return "OSS";
        case ma_backend_pulseaudio: return "PulseAudio";
        case ma_backend_alsa:       return "ALSA";
        case ma_backend_jack:       return "JACK";
        case ma_backend_aaudio:     return "AAudio";
        case ma_backend_opensl:     return "OpenSL|ES";
        case ma_backend_webaudio:   return "Web Audio";
        case ma_backend_null:       return "Null";
        default:                     return "Unknown";
    }
}



#ifdef MA_WIN32
    #define MA_THREADCALL WINAPI
    typedef unsigned long ma_thread_result;
#else
    #define MA_THREADCALL
    typedef void* ma_thread_result;
#endif
typedef ma_thread_result (MA_THREADCALL * ma_thread_entry_proc)(void* pData);

#ifdef MA_WIN32
typedef HRESULT (WINAPI * MA_PFN_CoInitializeEx)(LPVOID pvReserved, DWORD  dwCoInit);
typedef void    (WINAPI * MA_PFN_CoUninitialize)();
typedef HRESULT (WINAPI * MA_PFN_CoCreateInstance)(REFCLSID rclsid, LPUNKNOWN pUnkOuter, DWORD dwClsContext, REFIID riid, LPVOID *ppv);
typedef void    (WINAPI * MA_PFN_CoTaskMemFree)(LPVOID pv);
typedef HRESULT (WINAPI * MA_PFN_PropVariantClear)(PROPVARIANT *pvar);
typedef int     (WINAPI * MA_PFN_StringFromGUID2)(const GUID* const rguid, LPOLESTR lpsz, int cchMax);

typedef HWND (WINAPI * MA_PFN_GetForegroundWindow)();
typedef HWND (WINAPI * MA_PFN_GetDesktopWindow)();

/* Microsoft documents these APIs as returning LSTATUS, but the Win32 API shipping with some compilers do not define it. It's just a LONG. */
typedef LONG (WINAPI * MA_PFN_RegOpenKeyExA)(HKEY hKey, LPCSTR lpSubKey, DWORD ulOptions, REGSAM samDesired, PHKEY phkResult);
typedef LONG (WINAPI * MA_PFN_RegCloseKey)(HKEY hKey);
typedef LONG (WINAPI * MA_PFN_RegQueryValueExA)(HKEY hKey, LPCSTR lpValueName, LPDWORD lpReserved, LPDWORD lpType, LPBYTE lpData, LPDWORD lpcbData);
#endif


#define MA_STATE_UNINITIALIZED     0
#define MA_STATE_STOPPED           1   /* The device's default state after initialization. */
#define MA_STATE_STARTED           2   /* The worker thread is in it's main loop waiting for the driver to request or deliver audio data. */
#define MA_STATE_STARTING          3   /* Transitioning from a stopped state to started. */
#define MA_STATE_STOPPING          4   /* Transitioning from a started state to stopped. */

#define MA_DEFAULT_PLAYBACK_DEVICE_NAME    "Default Playback Device"
#define MA_DEFAULT_CAPTURE_DEVICE_NAME     "Default Capture Device"


/*******************************************************************************

Timing

*******************************************************************************/
#ifdef MA_WIN32
LARGE_INTEGER g_ma_TimerFrequency = {{0}};
void ma_timer_init(ma_timer* pTimer)
{
    LARGE_INTEGER counter;

    if (g_ma_TimerFrequency.QuadPart == 0) {
        QueryPerformanceFrequency(&g_ma_TimerFrequency);
    }

    QueryPerformanceCounter(&counter);
    pTimer->counter = counter.QuadPart;
}

double ma_timer_get_time_in_seconds(ma_timer* pTimer)
{
    LARGE_INTEGER counter;
    if (!QueryPerformanceCounter(&counter)) {
        return 0;
    }

    return (double)(counter.QuadPart - pTimer->counter) / g_ma_TimerFrequency.QuadPart;
}
#elif defined(MA_APPLE) && (__MAC_OS_X_VERSION_MIN_REQUIRED < 101200)
ma_uint64 g_ma_TimerFrequency = 0;
void ma_timer_init(ma_timer* pTimer)
{
    mach_timebase_info_data_t baseTime;
    mach_timebase_info(&baseTime);
    g_ma_TimerFrequency = (baseTime.denom * 1e9) / baseTime.numer;

    pTimer->counter = mach_absolute_time();
}

double ma_timer_get_time_in_seconds(ma_timer* pTimer)
{
    ma_uint64 newTimeCounter = mach_absolute_time();
    ma_uint64 oldTimeCounter = pTimer->counter;

    return (newTimeCounter - oldTimeCounter) / g_ma_TimerFrequency;
}
#elif defined(MA_EMSCRIPTEN)
void ma_timer_init(ma_timer* pTimer)
{
    pTimer->counterD = emscripten_get_now();
}

double ma_timer_get_time_in_seconds(ma_timer* pTimer)
{
    return (emscripten_get_now() - pTimer->counterD) / 1000;    /* Emscripten is in milliseconds. */
}
#else
#if _POSIX_C_SOURCE >= 199309L
#if defined(CLOCK_MONOTONIC)
    #define MA_CLOCK_ID CLOCK_MONOTONIC
#else
    #define MA_CLOCK_ID CLOCK_REALTIME
#endif

void ma_timer_init(ma_timer* pTimer)
{
    struct timespec newTime;
    clock_gettime(MA_CLOCK_ID, &newTime);

    pTimer->counter = (newTime.tv_sec * 1000000000) + newTime.tv_nsec;
}

double ma_timer_get_time_in_seconds(ma_timer* pTimer)
{
    ma_uint64 newTimeCounter;
    ma_uint64 oldTimeCounter;

    struct timespec newTime;
    clock_gettime(MA_CLOCK_ID, &newTime);

    newTimeCounter = (newTime.tv_sec * 1000000000) + newTime.tv_nsec;
    oldTimeCounter = pTimer->counter;

    return (newTimeCounter - oldTimeCounter) / 1000000000.0;
}
#else
void ma_timer_init(ma_timer* pTimer)
{
    struct timeval newTime;
    gettimeofday(&newTime, NULL);

    pTimer->counter = (newTime.tv_sec * 1000000) + newTime.tv_usec;
}

double ma_timer_get_time_in_seconds(ma_timer* pTimer)
{
    ma_uint64 newTimeCounter;
    ma_uint64 oldTimeCounter;

    struct timeval newTime;
    gettimeofday(&newTime, NULL);

    newTimeCounter = (newTime.tv_sec * 1000000) + newTime.tv_usec;
    oldTimeCounter = pTimer->counter;

    return (newTimeCounter - oldTimeCounter) / 1000000.0;
}
#endif
#endif


/*******************************************************************************

Dynamic Linking

*******************************************************************************/
ma_handle ma_dlopen(const char* filename)
{
#ifdef _WIN32
#ifdef MA_WIN32_DESKTOP
    return (ma_handle)LoadLibraryA(filename);
#else
    /* *sigh* It appears there is no ANSI version of LoadPackagedLibrary()... */
    WCHAR filenameW[4096];
    if (MultiByteToWideChar(CP_UTF8, 0, filename, -1, filenameW, sizeof(filenameW)) == 0) {
        return NULL;
    }

    return (ma_handle)LoadPackagedLibrary(filenameW, 0);
#endif
#else
    return (ma_handle)dlopen(filename, RTLD_NOW);
#endif
}

void ma_dlclose(ma_handle handle)
{
#ifdef _WIN32
    FreeLibrary((HMODULE)handle);
#else
    dlclose((void*)handle);
#endif
}

ma_proc ma_dlsym(ma_handle handle, const char* symbol)
{
#ifdef _WIN32
    return (ma_proc)GetProcAddress((HMODULE)handle, symbol);
#else
#if defined(__GNUC__) && (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ >= 6))
    #pragma GCC diagnostic push
    #pragma GCC diagnostic ignored "-Wpedantic"
#endif
    return (ma_proc)dlsym((void*)handle, symbol);
#if defined(__GNUC__) && (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ >= 6))
    #pragma GCC diagnostic pop
#endif
#endif
}


/*******************************************************************************

Threading

*******************************************************************************/
#ifdef MA_WIN32
int ma_thread_priority_to_win32(ma_thread_priority priority)
{
    switch (priority) {
        case ma_thread_priority_idle:     return THREAD_PRIORITY_IDLE;
        case ma_thread_priority_lowest:   return THREAD_PRIORITY_LOWEST;
        case ma_thread_priority_low:      return THREAD_PRIORITY_BELOW_NORMAL;
        case ma_thread_priority_normal:   return THREAD_PRIORITY_NORMAL;
        case ma_thread_priority_high:     return THREAD_PRIORITY_ABOVE_NORMAL;
        case ma_thread_priority_highest:  return THREAD_PRIORITY_HIGHEST;
        case ma_thread_priority_realtime: return THREAD_PRIORITY_TIME_CRITICAL;
        default: return ma_thread_priority_normal;
    }
}

ma_result ma_thread_create__win32(ma_context* pContext, ma_thread* pThread, ma_thread_entry_proc entryProc, void* pData)
{
    pThread->win32.hThread = CreateThread(NULL, 0, entryProc, pData, 0, NULL);
    if (pThread->win32.hThread == NULL) {
        return MA_FAILED_TO_CREATE_THREAD;
    }

    SetThreadPriority((HANDLE)pThread->win32.hThread, ma_thread_priority_to_win32(pContext->threadPriority));

    return MA_SUCCESS;
}

void ma_thread_wait__win32(ma_thread* pThread)
{
    WaitForSingleObject(pThread->win32.hThread, INFINITE);
}

void ma_sleep__win32(ma_uint32 milliseconds)
{
    Sleep((DWORD)milliseconds);
}


ma_result ma_mutex_init__win32(ma_context* pContext, ma_mutex* pMutex)
{
    (void)pContext;

    pMutex->win32.hMutex = CreateEventA(NULL, FALSE, TRUE, NULL);
    if (pMutex->win32.hMutex == NULL) {
        return MA_FAILED_TO_CREATE_MUTEX;
    }

    return MA_SUCCESS;
}

void ma_mutex_uninit__win32(ma_mutex* pMutex)
{
    CloseHandle(pMutex->win32.hMutex);
}

void ma_mutex_lock__win32(ma_mutex* pMutex)
{
    WaitForSingleObject(pMutex->win32.hMutex, INFINITE);
}

void ma_mutex_unlock__win32(ma_mutex* pMutex)
{
    SetEvent(pMutex->win32.hMutex);
}


ma_result ma_event_init__win32(ma_context* pContext, ma_event* pEvent)
{
    (void)pContext;

    pEvent->win32.hEvent = CreateEventW(NULL, FALSE, FALSE, NULL);
    if (pEvent->win32.hEvent == NULL) {
        return MA_FAILED_TO_CREATE_EVENT;
    }

    return MA_SUCCESS;
}

void ma_event_uninit__win32(ma_event* pEvent)
{
    CloseHandle(pEvent->win32.hEvent);
}

ma_bool32 ma_event_wait__win32(ma_event* pEvent)
{
    return WaitForSingleObject(pEvent->win32.hEvent, INFINITE) == WAIT_OBJECT_0;
}

ma_bool32 ma_event_signal__win32(ma_event* pEvent)
{
    return SetEvent(pEvent->win32.hEvent);
}
#endif


#ifdef MA_POSIX
#include <sched.h>

typedef int (* ma_pthread_create_proc)(pthread_t *thread, const pthread_attr_t *attr, void *(*start_routine) (void *), void *arg);
typedef int (* ma_pthread_join_proc)(pthread_t thread, void **retval);
typedef int (* ma_pthread_mutex_init_proc)(pthread_mutex_t *__mutex, const pthread_mutexattr_t *__mutexattr);
typedef int (* ma_pthread_mutex_destroy_proc)(pthread_mutex_t *__mutex);
typedef int (* ma_pthread_mutex_lock_proc)(pthread_mutex_t *__mutex);
typedef int (* ma_pthread_mutex_unlock_proc)(pthread_mutex_t *__mutex);
typedef int (* ma_pthread_cond_init_proc)(pthread_cond_t *__restrict __cond, const pthread_condattr_t *__restrict __cond_attr);
typedef int (* ma_pthread_cond_destroy_proc)(pthread_cond_t *__cond);
typedef int (* ma_pthread_cond_signal_proc)(pthread_cond_t *__cond);
typedef int (* ma_pthread_cond_wait_proc)(pthread_cond_t *__restrict __cond, pthread_mutex_t *__restrict __mutex);
typedef int (* ma_pthread_attr_init_proc)(pthread_attr_t *attr);
typedef int (* ma_pthread_attr_destroy_proc)(pthread_attr_t *attr);
typedef int (* ma_pthread_attr_setschedpolicy_proc)(pthread_attr_t *attr, int policy);
typedef int (* ma_pthread_attr_getschedparam_proc)(const pthread_attr_t *attr, struct sched_param *param);
typedef int (* ma_pthread_attr_setschedparam_proc)(pthread_attr_t *attr, const struct sched_param *param);

ma_result ma_thread_create__posix(ma_context* pContext, ma_thread* pThread, ma_thread_entry_proc entryProc, void* pData)
{
    int result;
    pthread_attr_t* pAttr = NULL;

#if !defined(__EMSCRIPTEN__)
    /* Try setting the thread priority. It's not critical if anything fails here. */
    pthread_attr_t attr;
    if (((ma_pthread_attr_init_proc)pContext->posix.pthread_attr_init)(&attr) == 0) {
        int scheduler = -1;
        if (pContext->threadPriority == ma_thread_priority_idle) {
#ifdef SCHED_IDLE
            if (((ma_pthread_attr_setschedpolicy_proc)pContext->posix.pthread_attr_setschedpolicy)(&attr, SCHED_IDLE) == 0) {
                scheduler = SCHED_IDLE;
            }
#endif
        } else if (pContext->threadPriority == ma_thread_priority_realtime) {
#ifdef SCHED_FIFO
            if (((ma_pthread_attr_setschedpolicy_proc)pContext->posix.pthread_attr_setschedpolicy)(&attr, SCHED_FIFO) == 0) {
                scheduler = SCHED_FIFO;
            }
#endif
#ifdef MA_LINUX
        } else {
            scheduler = sched_getscheduler(0);
#endif
        }

        if (scheduler != -1) {
            int priorityMin = sched_get_priority_min(scheduler);
            int priorityMax = sched_get_priority_max(scheduler);
            int priorityStep = (priorityMax - priorityMin) / 7;  /* 7 = number of priorities supported by miniaudio. */

            struct sched_param sched;
            if (((ma_pthread_attr_getschedparam_proc)pContext->posix.pthread_attr_getschedparam)(&attr, &sched) == 0) {
                if (pContext->threadPriority == ma_thread_priority_idle) {
                    sched.sched_priority = priorityMin;
                } else if (pContext->threadPriority == ma_thread_priority_realtime) {
                    sched.sched_priority = priorityMax;
                } else {
                    sched.sched_priority += ((int)pContext->threadPriority + 5) * priorityStep;  /* +5 because the lowest priority is -5. */
                    if (sched.sched_priority < priorityMin) {
                        sched.sched_priority = priorityMin;
                    }
                    if (sched.sched_priority > priorityMax) {
                        sched.sched_priority = priorityMax;
                    }
                }

                if (((ma_pthread_attr_setschedparam_proc)pContext->posix.pthread_attr_setschedparam)(&attr, &sched) == 0) {
                    pAttr = &attr;
                }
            }
        }

        ((ma_pthread_attr_destroy_proc)pContext->posix.pthread_attr_destroy)(&attr);
    }
#endif

    result = ((ma_pthread_create_proc)pContext->posix.pthread_create)(&pThread->posix.thread, pAttr, entryProc, pData);
    if (result != 0) {
        return MA_FAILED_TO_CREATE_THREAD;
    }

    return MA_SUCCESS;
}

void ma_thread_wait__posix(ma_thread* pThread)
{
    ((ma_pthread_join_proc)pThread->pContext->posix.pthread_join)(pThread->posix.thread, NULL);
}

void ma_sleep__posix(ma_uint32 milliseconds)
{
#ifdef MA_EMSCRIPTEN
    (void)milliseconds;
    ma_assert(MA_FALSE);  /* The Emscripten build should never sleep. */
#else
    #if _POSIX_C_SOURCE >= 199309L
        struct timespec ts;
        ts.tv_sec  = milliseconds / 1000000;
        ts.tv_nsec = milliseconds % 1000000 * 1000000;
        nanosleep(&ts, NULL);
    #else
        struct timeval tv;
        tv.tv_sec  = milliseconds / 1000;
        tv.tv_usec = milliseconds % 1000 * 1000;
        select(0, NULL, NULL, NULL, &tv);
    #endif
#endif
}


ma_result ma_mutex_init__posix(ma_context* pContext, ma_mutex* pMutex)
{
    int result = ((ma_pthread_mutex_init_proc)pContext->posix.pthread_mutex_init)(&pMutex->posix.mutex, NULL);
    if (result != 0) {
        return MA_FAILED_TO_CREATE_MUTEX;
    }

    return MA_SUCCESS;
}

void ma_mutex_uninit__posix(ma_mutex* pMutex)
{
    ((ma_pthread_mutex_destroy_proc)pMutex->pContext->posix.pthread_mutex_destroy)(&pMutex->posix.mutex);
}

void ma_mutex_lock__posix(ma_mutex* pMutex)
{
    ((ma_pthread_mutex_lock_proc)pMutex->pContext->posix.pthread_mutex_lock)(&pMutex->posix.mutex);
}

void ma_mutex_unlock__posix(ma_mutex* pMutex)
{
    ((ma_pthread_mutex_unlock_proc)pMutex->pContext->posix.pthread_mutex_unlock)(&pMutex->posix.mutex);
}


ma_result ma_event_init__posix(ma_context* pContext, ma_event* pEvent)
{
    if (((ma_pthread_mutex_init_proc)pContext->posix.pthread_mutex_init)(&pEvent->posix.mutex, NULL) != 0) {
        return MA_FAILED_TO_CREATE_MUTEX;
    }

    if (((ma_pthread_cond_init_proc)pContext->posix.pthread_cond_init)(&pEvent->posix.condition, NULL) != 0) {
        return MA_FAILED_TO_CREATE_EVENT;
    }

    pEvent->posix.value = 0;
    return MA_SUCCESS;
}

void ma_event_uninit__posix(ma_event* pEvent)
{
    ((ma_pthread_cond_destroy_proc)pEvent->pContext->posix.pthread_cond_destroy)(&pEvent->posix.condition);
    ((ma_pthread_mutex_destroy_proc)pEvent->pContext->posix.pthread_mutex_destroy)(&pEvent->posix.mutex);
}

ma_bool32 ma_event_wait__posix(ma_event* pEvent)
{
    ((ma_pthread_mutex_lock_proc)pEvent->pContext->posix.pthread_mutex_lock)(&pEvent->posix.mutex);
    {
        while (pEvent->posix.value == 0) {
            ((ma_pthread_cond_wait_proc)pEvent->pContext->posix.pthread_cond_wait)(&pEvent->posix.condition, &pEvent->posix.mutex);
        }
        pEvent->posix.value = 0;  /* Auto-reset. */
    }
    ((ma_pthread_mutex_unlock_proc)pEvent->pContext->posix.pthread_mutex_unlock)(&pEvent->posix.mutex);

    return MA_TRUE;
}

ma_bool32 ma_event_signal__posix(ma_event* pEvent)
{
    ((ma_pthread_mutex_lock_proc)pEvent->pContext->posix.pthread_mutex_lock)(&pEvent->posix.mutex);
    {
        pEvent->posix.value = 1;
        ((ma_pthread_cond_signal_proc)pEvent->pContext->posix.pthread_cond_signal)(&pEvent->posix.condition);
    }
    ((ma_pthread_mutex_unlock_proc)pEvent->pContext->posix.pthread_mutex_unlock)(&pEvent->posix.mutex);

    return MA_TRUE;
}
#endif

ma_result ma_thread_create(ma_context* pContext, ma_thread* pThread, ma_thread_entry_proc entryProc, void* pData)
{
    if (pContext == NULL || pThread == NULL || entryProc == NULL) {
        return MA_FALSE;
    }

    pThread->pContext = pContext;

#ifdef MA_WIN32
    return ma_thread_create__win32(pContext, pThread, entryProc, pData);
#endif
#ifdef MA_POSIX
    return ma_thread_create__posix(pContext, pThread, entryProc, pData);
#endif
}

void ma_thread_wait(ma_thread* pThread)
{
    if (pThread == NULL) {
        return;
    }

#ifdef MA_WIN32
    ma_thread_wait__win32(pThread);
#endif
#ifdef MA_POSIX
    ma_thread_wait__posix(pThread);
#endif
}

void ma_sleep(ma_uint32 milliseconds)
{
#ifdef MA_WIN32
    ma_sleep__win32(milliseconds);
#endif
#ifdef MA_POSIX
    ma_sleep__posix(milliseconds);
#endif
}


ma_result ma_mutex_init(ma_context* pContext, ma_mutex* pMutex)
{
    if (pContext == NULL || pMutex == NULL) {
        return MA_INVALID_ARGS;
    }

    pMutex->pContext = pContext;

#ifdef MA_WIN32
    return ma_mutex_init__win32(pContext, pMutex);
#endif
#ifdef MA_POSIX
    return ma_mutex_init__posix(pContext, pMutex);
#endif
}

void ma_mutex_uninit(ma_mutex* pMutex)
{
    if (pMutex == NULL || pMutex->pContext == NULL) {
        return;
    }

#ifdef MA_WIN32
    ma_mutex_uninit__win32(pMutex);
#endif
#ifdef MA_POSIX
    ma_mutex_uninit__posix(pMutex);
#endif
}

void ma_mutex_lock(ma_mutex* pMutex)
{
    if (pMutex == NULL || pMutex->pContext == NULL) {
        return;
    }

#ifdef MA_WIN32
    ma_mutex_lock__win32(pMutex);
#endif
#ifdef MA_POSIX
    ma_mutex_lock__posix(pMutex);
#endif
}

void ma_mutex_unlock(ma_mutex* pMutex)
{
    if (pMutex == NULL || pMutex->pContext == NULL) {
        return;
}

#ifdef MA_WIN32
    ma_mutex_unlock__win32(pMutex);
#endif
#ifdef MA_POSIX
    ma_mutex_unlock__posix(pMutex);
#endif
}


ma_result ma_event_init(ma_context* pContext, ma_event* pEvent)
{
    if (pContext == NULL || pEvent == NULL) {
        return MA_FALSE;
    }

    pEvent->pContext = pContext;

#ifdef MA_WIN32
    return ma_event_init__win32(pContext, pEvent);
#endif
#ifdef MA_POSIX
    return ma_event_init__posix(pContext, pEvent);
#endif
}

void ma_event_uninit(ma_event* pEvent)
{
    if (pEvent == NULL || pEvent->pContext == NULL) {
        return;
    }

#ifdef MA_WIN32
    ma_event_uninit__win32(pEvent);
#endif
#ifdef MA_POSIX
    ma_event_uninit__posix(pEvent);
#endif
}

ma_bool32 ma_event_wait(ma_event* pEvent)
{
    if (pEvent == NULL || pEvent->pContext == NULL) {
        return MA_FALSE;
    }

#ifdef MA_WIN32
    return ma_event_wait__win32(pEvent);
#endif
#ifdef MA_POSIX
    return ma_event_wait__posix(pEvent);
#endif
}

ma_bool32 ma_event_signal(ma_event* pEvent)
{
    if (pEvent == NULL || pEvent->pContext == NULL) {
        return MA_FALSE;
    }

#ifdef MA_WIN32
    return ma_event_signal__win32(pEvent);
#endif
#ifdef MA_POSIX
    return ma_event_signal__posix(pEvent);
#endif
}


ma_uint32 ma_get_best_sample_rate_within_range(ma_uint32 sampleRateMin, ma_uint32 sampleRateMax)
{
    /* Normalize the range in case we were given something stupid. */
    if (sampleRateMin < MA_MIN_SAMPLE_RATE) {
        sampleRateMin = MA_MIN_SAMPLE_RATE;
    }
    if (sampleRateMax > MA_MAX_SAMPLE_RATE) {
        sampleRateMax = MA_MAX_SAMPLE_RATE;
    }
    if (sampleRateMin > sampleRateMax) {
        sampleRateMin = sampleRateMax;
    }

    if (sampleRateMin == sampleRateMax) {
        return sampleRateMax;
    } else {
        size_t iStandardRate;
        for (iStandardRate = 0; iStandardRate < ma_countof(g_maStandardSampleRatePriorities); ++iStandardRate) {
            ma_uint32 standardRate = g_maStandardSampleRatePriorities[iStandardRate];
            if (standardRate >= sampleRateMin && standardRate <= sampleRateMax) {
                return standardRate;
            }
        }
    }

    /* Should never get here. */
    ma_assert(MA_FALSE);
    return 0;
}

ma_uint32 ma_get_closest_standard_sample_rate(ma_uint32 sampleRateIn)
{
    ma_uint32 closestRate = 0;
    ma_uint32 closestDiff = 0xFFFFFFFF;
    size_t iStandardRate;

    for (iStandardRate = 0; iStandardRate < ma_countof(g_maStandardSampleRatePriorities); ++iStandardRate) {
        ma_uint32 standardRate = g_maStandardSampleRatePriorities[iStandardRate];
        ma_uint32 diff;

        if (sampleRateIn > standardRate) {
            diff = sampleRateIn - standardRate;
        } else {
            diff = standardRate - sampleRateIn;
        }

        if (diff == 0) {
            return standardRate;    /* The input sample rate is a standard rate. */
        }

        if (closestDiff > diff) {
            closestDiff = diff;
            closestRate = standardRate;
        }
    }

    return closestRate;
}


ma_uint32 ma_scale_buffer_size(ma_uint32 baseBufferSize, float scale)
{
    return ma_max(1, (ma_uint32)(baseBufferSize*scale));
}

ma_uint32 ma_calculate_buffer_size_in_milliseconds_from_frames(ma_uint32 bufferSizeInFrames, ma_uint32 sampleRate)
{
    return bufferSizeInFrames / (sampleRate/1000);
}

ma_uint32 ma_calculate_buffer_size_in_frames_from_milliseconds(ma_uint32 bufferSizeInMilliseconds, ma_uint32 sampleRate)
{
    return bufferSizeInMilliseconds * (sampleRate/1000); 
}

ma_uint32 ma_get_default_buffer_size_in_milliseconds(ma_performance_profile performanceProfile)
{
    if (performanceProfile == ma_performance_profile_low_latency) {
        return MA_BASE_BUFFER_SIZE_IN_MILLISECONDS_LOW_LATENCY;
    } else {
        return MA_BASE_BUFFER_SIZE_IN_MILLISECONDS_CONSERVATIVE;
    }
}

ma_uint32 ma_get_default_buffer_size_in_frames(ma_performance_profile performanceProfile, ma_uint32 sampleRate)
{
    ma_uint32 bufferSizeInMilliseconds;
    ma_uint32 sampleRateMS;

    bufferSizeInMilliseconds = ma_get_default_buffer_size_in_milliseconds(performanceProfile);
    if (bufferSizeInMilliseconds == 0) {
        bufferSizeInMilliseconds = 1;
    }

    sampleRateMS = (sampleRate/1000);
    if (sampleRateMS == 0) {
        sampleRateMS = 1;
    }

    return bufferSizeInMilliseconds * sampleRateMS;
}

ma_uint32 ma_get_fragment_size_in_bytes(ma_uint32 bufferSizeInFrames, ma_uint32 periods, ma_format format, ma_uint32 channels)
{
    ma_uint32 fragmentSizeInFrames = bufferSizeInFrames / periods;
    return fragmentSizeInFrames * ma_get_bytes_per_frame(format, channels);
}

void ma_zero_pcm_frames(void* p, ma_uint32 frameCount, ma_format format, ma_uint32 channels)
{
    ma_zero_memory(p, frameCount * ma_get_bytes_per_frame(format, channels));
}


const char* ma_log_level_to_string(ma_uint32 logLevel)
{
    switch (logLevel)
    {
        case MA_LOG_LEVEL_VERBOSE: return "";
        case MA_LOG_LEVEL_INFO:    return "INFO";
        case MA_LOG_LEVEL_WARNING: return "WARNING";
        case MA_LOG_LEVEL_ERROR:   return "ERROR";
        default:                   return "ERROR";
    }
}

/* Posts a log message. */
void ma_log(ma_context* pContext, ma_device* pDevice, ma_uint32 logLevel, const char* message)
{
    if (pContext == NULL) {
        return;
    }
    
#if defined(MA_LOG_LEVEL)
    if (logLevel <= MA_LOG_LEVEL) {
        ma_log_proc onLog;

    #if defined(MA_DEBUG_OUTPUT)
        if (logLevel <= MA_LOG_LEVEL) {
            printf("%s: %s\n", ma_log_level_to_string(logLevel), message);
        }
    #endif
    
        onLog = pContext->logCallback;
        if (onLog) {
            onLog(pContext, pDevice, logLevel, message);
        }
    }
#endif
}

/* Posts an error. Throw a breakpoint in here if you're needing to debug. The return value is always "resultCode". */
ma_result ma_context_post_error(ma_context* pContext, ma_device* pDevice, ma_uint32 logLevel, const char* message, ma_result resultCode)
{
    /* Derive the context from the device if necessary. */
    if (pContext == NULL) {
        if (pDevice != NULL) {
            pContext = pDevice->pContext;
        }
    }

    ma_log(pContext, pDevice, logLevel, message);
    return resultCode;
}

ma_result ma_post_error(ma_device* pDevice, ma_uint32 logLevel, const char* message, ma_result resultCode)
{
    return ma_context_post_error(NULL, pDevice, logLevel, message, resultCode);
}



/* The callback for reading from the client -> DSP -> device. */
ma_uint32 ma_device__on_read_from_client(ma_pcm_converter* pDSP, void* pFramesOut, ma_uint32 frameCount, void* pUserData)
{
    ma_device* pDevice = (ma_device*)pUserData;
    ma_device_callback_proc onData;

    ma_assert(pDevice != NULL);

    ma_zero_pcm_frames(pFramesOut, frameCount, pDevice->playback.format, pDevice->playback.channels);

    onData = pDevice->onData;
    if (onData) {
        onData(pDevice, pFramesOut, NULL, frameCount);
        return frameCount;
    }

    (void)pDSP;
    return 0;
}

/* The PCM converter callback for reading from a buffer. */
ma_uint32 ma_device__pcm_converter__on_read_from_buffer_capture(ma_pcm_converter* pConverter, void* pFramesOut, ma_uint32 frameCount, void* pUserData)
{
    ma_device* pDevice = (ma_device*)pUserData;
    ma_uint32 framesToRead;
    ma_uint32 bytesToRead;

    ma_assert(pDevice != NULL);

    if (pDevice->capture._dspFrameCount == 0) {
        return 0;   /* Nothing left. */
    }

    framesToRead = frameCount;
    if (framesToRead > pDevice->capture._dspFrameCount) {
        framesToRead = pDevice->capture._dspFrameCount;
    }

    bytesToRead = framesToRead * ma_get_bytes_per_frame(pConverter->formatConverterIn.config.formatIn, pConverter->channelRouter.config.channelsIn);
    ma_copy_memory(pFramesOut, pDevice->capture._dspFrames, bytesToRead);
    pDevice->capture._dspFrameCount -= framesToRead;
    pDevice->capture._dspFrames     += bytesToRead;

    return framesToRead;
}

ma_uint32 ma_device__pcm_converter__on_read_from_buffer_playback(ma_pcm_converter* pConverter, void* pFramesOut, ma_uint32 frameCount, void* pUserData)
{
    ma_device* pDevice = (ma_device*)pUserData;
    ma_uint32 framesToRead;
    ma_uint32 bytesToRead;

    ma_assert(pDevice != NULL);

    if (pDevice->playback._dspFrameCount == 0) {
        return 0;   /* Nothing left. */
    }

    framesToRead = frameCount;
    if (framesToRead > pDevice->playback._dspFrameCount) {
        framesToRead = pDevice->playback._dspFrameCount;
    }

    bytesToRead = framesToRead * ma_get_bytes_per_frame(pConverter->formatConverterIn.config.formatIn, pConverter->channelRouter.config.channelsIn);
    ma_copy_memory(pFramesOut, pDevice->playback._dspFrames, bytesToRead);
    pDevice->playback._dspFrameCount -= framesToRead;
    pDevice->playback._dspFrames     += bytesToRead;

    return framesToRead;
}



/* A helper function for reading sample data from the client. */
static MA_INLINE void ma_device__read_frames_from_client(ma_device* pDevice, ma_uint32 frameCount, void* pSamples)
{
    ma_device_callback_proc onData;

    ma_assert(pDevice != NULL);
    ma_assert(frameCount > 0);
    ma_assert(pSamples != NULL);

    onData = pDevice->onData;
    if (onData) {
        if (pDevice->playback.converter.isPassthrough) {
            ma_zero_pcm_frames(pSamples, frameCount, pDevice->playback.format, pDevice->playback.channels);
            onData(pDevice, pSamples, NULL, frameCount);
        } else {
            ma_pcm_converter_read(&pDevice->playback.converter, pSamples, frameCount);
        }
    }
}

/* A helper for sending sample data to the client. */
static MA_INLINE void ma_device__send_frames_to_client(ma_device* pDevice, ma_uint32 frameCount, const void* pSamples)
{
    ma_device_callback_proc onData;

    ma_assert(pDevice != NULL);
    ma_assert(frameCount > 0);
    ma_assert(pSamples != NULL);

    onData = pDevice->onData;
    if (onData) {
        if (pDevice->capture.converter.isPassthrough) {
            onData(pDevice, NULL, pSamples, frameCount);
        } else {
            ma_uint8 chunkBuffer[4096];
            ma_uint32 chunkFrameCount;

            pDevice->capture._dspFrameCount = frameCount;
            pDevice->capture._dspFrames = (const ma_uint8*)pSamples;

            chunkFrameCount = sizeof(chunkBuffer) / ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);

            for (;;) {
                ma_uint32 framesJustRead = (ma_uint32)ma_pcm_converter_read(&pDevice->capture.converter, chunkBuffer, chunkFrameCount);
                if (framesJustRead == 0) {
                    break;
                }

                onData(pDevice, NULL, chunkBuffer, framesJustRead);

                if (framesJustRead < chunkFrameCount) {
                    break;
                }
            }
        }
    }
}

static MA_INLINE ma_result ma_device__handle_duplex_callback_capture(ma_device* pDevice, ma_uint32 frameCount, const void* pFramesInInternalFormat, ma_pcm_rb* pRB)
{
    ma_result result;

    ma_assert(pDevice != NULL);
    ma_assert(frameCount > 0);
    ma_assert(pFramesInInternalFormat != NULL);
    ma_assert(pRB != NULL);
    
    pDevice->capture._dspFrameCount = (ma_uint32)frameCount;
    pDevice->capture._dspFrames     = (const ma_uint8*)pFramesInInternalFormat;

    /* Write to the ring buffer. The ring buffer is in the external format. */
    for (;;) {
        ma_uint32 framesProcessed;
        ma_uint32 framesToProcess = 256;
        void* pFramesInExternalFormat;

        result = ma_pcm_rb_acquire_write(pRB, &framesToProcess, &pFramesInExternalFormat);
        if (result != MA_SUCCESS) {
            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "Failed to acquire capture PCM frames from ring buffer.", result);
            break;
        }

        if (framesToProcess == 0) {
            if (ma_pcm_rb_pointer_disance(pRB) == (ma_int32)ma_pcm_rb_get_subbuffer_size(pRB)) {
                break;  /* Overrun. Not enough room in the ring buffer for input frame. Excess frames are dropped. */
            }
        }

        /* Convert. */
        framesProcessed = (ma_uint32)ma_pcm_converter_read(&pDevice->capture.converter, pFramesInExternalFormat, framesToProcess);

        result = ma_pcm_rb_commit_write(pRB, framesProcessed, pFramesInExternalFormat);
        if (result != MA_SUCCESS) {
            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "Failed to commit capture PCM frames to ring buffer.", result);
            break;
        }

        if (framesProcessed < framesToProcess) {
            break;  /* Done. */
        }
    }

    return MA_SUCCESS;
}

static MA_INLINE ma_result ma_device__handle_duplex_callback_playback(ma_device* pDevice, ma_uint32 frameCount, void* pFramesInInternalFormat, ma_pcm_rb* pRB)
{
    ma_result result;
    ma_uint8 playbackFramesInExternalFormat[4096];
    ma_uint8 silentInputFrames[4096];
    ma_uint32 totalFramesToReadFromClient;
    ma_uint32 totalFramesReadFromClient;

    ma_assert(pDevice != NULL);
    ma_assert(frameCount > 0);
    ma_assert(pFramesInInternalFormat != NULL);
    ma_assert(pRB != NULL);
    
    /*
    Sitting in the ring buffer should be captured data from the capture callback in external format. If there's not enough data in there for
    the whole frameCount frames we just use silence instead for the input data.
    */
    ma_zero_memory(silentInputFrames, sizeof(silentInputFrames));

    /* We need to calculate how many output frames are required to be read from the client to completely fill frameCount internal frames. */
    totalFramesToReadFromClient = (ma_uint32)ma_calculate_frame_count_after_src(pDevice->sampleRate, pDevice->playback.internalSampleRate, frameCount); /* ma_pcm_converter_get_required_input_frame_count(&pDevice->playback.converter, (ma_uint32)frameCount); */
    totalFramesReadFromClient = 0;
    while (totalFramesReadFromClient < totalFramesToReadFromClient && ma_device_is_started(pDevice)) {
        ma_uint32 framesRemainingFromClient;
        ma_uint32 framesToProcessFromClient;
        ma_uint32 inputFrameCount;
        void* pInputFrames;

        framesRemainingFromClient = (totalFramesToReadFromClient - totalFramesReadFromClient);
        framesToProcessFromClient = sizeof(playbackFramesInExternalFormat) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);
        if (framesToProcessFromClient > framesRemainingFromClient) {
            framesToProcessFromClient = framesRemainingFromClient;
        }

        /* We need to grab captured samples before firing the callback. If there's not enough input samples we just pass silence. */
        inputFrameCount = framesToProcessFromClient;
        result = ma_pcm_rb_acquire_read(pRB, &inputFrameCount, &pInputFrames);
        if (result == MA_SUCCESS) {
            if (inputFrameCount > 0) {
                /* Use actual input frames. */
                pDevice->onData(pDevice, playbackFramesInExternalFormat, pInputFrames, inputFrameCount);
            } else {
                if (ma_pcm_rb_pointer_disance(pRB) == 0) {
                    break;  /* Underrun. */
                }
            }
            
            /* We're done with the captured samples. */
            result = ma_pcm_rb_commit_read(pRB, inputFrameCount, pInputFrames);
            if (result != MA_SUCCESS) {
                break; /* Don't know what to do here... Just abandon ship. */
            }
        } else {
            /* Use silent input frames. */
            inputFrameCount = ma_min(
                sizeof(playbackFramesInExternalFormat) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels),
                sizeof(silentInputFrames)              / ma_get_bytes_per_frame(pDevice->capture.format,  pDevice->capture.channels)
            );

            pDevice->onData(pDevice, playbackFramesInExternalFormat, silentInputFrames, inputFrameCount);
        }

        /* We have samples in external format so now we need to convert to internal format and output to the device. */
        pDevice->playback._dspFrameCount = inputFrameCount;
        pDevice->playback._dspFrames     = (const ma_uint8*)playbackFramesInExternalFormat;
        ma_pcm_converter_read(&pDevice->playback.converter, pFramesInInternalFormat, inputFrameCount);

        totalFramesReadFromClient += inputFrameCount;
        pFramesInInternalFormat = ma_offset_ptr(pFramesInInternalFormat, inputFrameCount * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));
    }

    return MA_SUCCESS;
}

/* A helper for changing the state of the device. */
static MA_INLINE void ma_device__set_state(ma_device* pDevice, ma_uint32 newState)
{
    ma_atomic_exchange_32(&pDevice->state, newState);
}

/* A helper for getting the state of the device. */
static MA_INLINE ma_uint32 ma_device__get_state(ma_device* pDevice)
{
    return pDevice->state;
}

/* A helper for determining whether or not the device is running in async mode. */
static MA_INLINE ma_bool32 ma_device__is_async(ma_device* pDevice)
{
    return pDevice->onData != NULL;
}


#ifdef MA_WIN32
    GUID MA_GUID_KSDATAFORMAT_SUBTYPE_PCM        = {0x00000001, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};
    GUID MA_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT = {0x00000003, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};
    /*GUID MA_GUID_KSDATAFORMAT_SUBTYPE_ALAW       = {0x00000006, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};*/
    /*GUID MA_GUID_KSDATAFORMAT_SUBTYPE_MULAW      = {0x00000007, 0x0000, 0x0010, {0x80, 0x00, 0x00, 0xaa, 0x00, 0x38, 0x9b, 0x71}};*/
#endif


ma_bool32 ma_context__device_id_equal(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);

    if (pID0 == pID1) return MA_TRUE;

    if ((pID0 == NULL && pID1 != NULL) ||
        (pID0 != NULL && pID1 == NULL)) {
        return MA_FALSE;
    }

    if (pContext->onDeviceIDEqual) {
        return pContext->onDeviceIDEqual(pContext, pID0, pID1);
    }

    return MA_FALSE;
}


typedef struct
{
    ma_device_type deviceType;
    const ma_device_id* pDeviceID;
    char* pName;
    size_t nameBufferSize;
    ma_bool32 foundDevice;
} ma_context__try_get_device_name_by_id__enum_callback_data;

ma_bool32 ma_context__try_get_device_name_by_id__enum_callback(ma_context* pContext, ma_device_type deviceType, const ma_device_info* pDeviceInfo, void* pUserData)
{
    ma_context__try_get_device_name_by_id__enum_callback_data* pData = (ma_context__try_get_device_name_by_id__enum_callback_data*)pUserData;
    ma_assert(pData != NULL);

    if (pData->deviceType == deviceType) {
        if (pContext->onDeviceIDEqual(pContext, pData->pDeviceID, &pDeviceInfo->id)) {
            ma_strncpy_s(pData->pName, pData->nameBufferSize, pDeviceInfo->name, (size_t)-1);
            pData->foundDevice = MA_TRUE;
        }
    }

    return !pData->foundDevice;
}

/*
Generic function for retrieving the name of a device by it's ID.

This function simply enumerates every device and then retrieves the name of the first device that has the same ID.
*/
ma_result ma_context__try_get_device_name_by_id(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, char* pName, size_t nameBufferSize)
{
    ma_result result;
    ma_context__try_get_device_name_by_id__enum_callback_data data;

    ma_assert(pContext != NULL);
    ma_assert(pName != NULL);

    if (pDeviceID == NULL) {
        return MA_NO_DEVICE;
    }

    data.deviceType = deviceType;
    data.pDeviceID = pDeviceID;
    data.pName = pName;
    data.nameBufferSize = nameBufferSize;
    data.foundDevice = MA_FALSE;
    result = ma_context_enumerate_devices(pContext, ma_context__try_get_device_name_by_id__enum_callback, &data);
    if (result != MA_SUCCESS) {
        return result;
    }

    if (!data.foundDevice) {
        return MA_NO_DEVICE;
    } else {
        return MA_SUCCESS;
    }
}


ma_uint32 ma_get_format_priority_index(ma_format format) /* Lower = better. */
{
    ma_uint32 i;
    for (i = 0; i < ma_countof(g_maFormatPriorities); ++i) {
        if (g_maFormatPriorities[i] == format) {
            return i;
        }
    }

    /* Getting here means the format could not be found or is equal to ma_format_unknown. */
    return (ma_uint32)-1;
}

void ma_device__post_init_setup(ma_device* pDevice, ma_device_type deviceType);


/*******************************************************************************

Null Backend

*******************************************************************************/
#ifdef MA_HAS_NULL

#define MA_DEVICE_OP_NONE__NULL    0
#define MA_DEVICE_OP_START__NULL   1
#define MA_DEVICE_OP_SUSPEND__NULL 2
#define MA_DEVICE_OP_KILL__NULL    3

ma_thread_result MA_THREADCALL ma_device_thread__null(void* pData)
{
    ma_device* pDevice = (ma_device*)pData;
    ma_assert(pDevice != NULL);

    for (;;) {  /* Keep the thread alive until the device is uninitialized. */
        /* Wait for an operation to be requested. */
        ma_event_wait(&pDevice->null_device.operationEvent);

        /* At this point an event should have been triggered. */

        /* Starting the device needs to put the thread into a loop. */
        if (pDevice->null_device.operation == MA_DEVICE_OP_START__NULL) {
            ma_atomic_exchange_32(&pDevice->null_device.operation, MA_DEVICE_OP_NONE__NULL);

            /* Reset the timer just in case. */
            ma_timer_init(&pDevice->null_device.timer);

            /* Keep looping until an operation has been requested. */
            while (pDevice->null_device.operation != MA_DEVICE_OP_NONE__NULL && pDevice->null_device.operation != MA_DEVICE_OP_START__NULL) {
                ma_sleep(10); /* Don't hog the CPU. */
            }

            /* Getting here means a suspend or kill operation has been requested. */
            ma_atomic_exchange_32(&pDevice->null_device.operationResult, MA_SUCCESS);
            ma_event_signal(&pDevice->null_device.operationCompletionEvent);
            continue;
        }

        /* Suspending the device means we need to stop the timer and just continue the loop. */
        if (pDevice->null_device.operation == MA_DEVICE_OP_SUSPEND__NULL) {
            ma_atomic_exchange_32(&pDevice->null_device.operation, MA_DEVICE_OP_NONE__NULL);

            /* We need to add the current run time to the prior run time, then reset the timer. */
            pDevice->null_device.priorRunTime += ma_timer_get_time_in_seconds(&pDevice->null_device.timer);
            ma_timer_init(&pDevice->null_device.timer);

            /* We're done. */
            ma_atomic_exchange_32(&pDevice->null_device.operationResult, MA_SUCCESS);
            ma_event_signal(&pDevice->null_device.operationCompletionEvent);
            continue;
        }

        /* Killing the device means we need to get out of this loop so that this thread can terminate. */
        if (pDevice->null_device.operation == MA_DEVICE_OP_KILL__NULL) {
            ma_atomic_exchange_32(&pDevice->null_device.operation, MA_DEVICE_OP_NONE__NULL);
            ma_atomic_exchange_32(&pDevice->null_device.operationResult, MA_SUCCESS);
            ma_event_signal(&pDevice->null_device.operationCompletionEvent);
            break;
        }

        /* Getting a signal on a "none" operation probably means an error. Return invalid operation. */
        if (pDevice->null_device.operation == MA_DEVICE_OP_NONE__NULL) {
            ma_assert(MA_FALSE);  /* <-- Trigger this in debug mode to ensure developers are aware they're doing something wrong (or there's a bug in a miniaudio). */
            ma_atomic_exchange_32(&pDevice->null_device.operationResult, MA_INVALID_OPERATION);
            ma_event_signal(&pDevice->null_device.operationCompletionEvent);
            continue;   /* Continue the loop. Don't terminate. */
        }
    }

    return (ma_thread_result)0;
}

ma_result ma_device_do_operation__null(ma_device* pDevice, ma_uint32 operation)
{
    ma_atomic_exchange_32(&pDevice->null_device.operation, operation);
    if (!ma_event_signal(&pDevice->null_device.operationEvent)) {
        return MA_ERROR;
    }

    if (!ma_event_wait(&pDevice->null_device.operationCompletionEvent)) {
        return MA_ERROR;
    }

    return pDevice->null_device.operationResult;
}

ma_uint64 ma_device_get_total_run_time_in_frames__null(ma_device* pDevice)
{
    ma_uint32 internalSampleRate;
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        internalSampleRate = pDevice->capture.internalSampleRate;
    } else {
        internalSampleRate = pDevice->playback.internalSampleRate;
    }


    return (ma_uint64)((pDevice->null_device.priorRunTime + ma_timer_get_time_in_seconds(&pDevice->null_device.timer)) * internalSampleRate);
}

ma_bool32 ma_context_is_device_id_equal__null(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return pID0->nullbackend == pID1->nullbackend;
}

ma_result ma_context_enumerate_devices__null(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_bool32 cbResult = MA_TRUE;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);

    /* Playback. */
    if (cbResult) {
        ma_device_info deviceInfo;
        ma_zero_object(&deviceInfo);
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), "NULL Playback Device", (size_t)-1);
        cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
    }

    /* Capture. */
    if (cbResult) {
        ma_device_info deviceInfo;
        ma_zero_object(&deviceInfo);
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), "NULL Capture Device", (size_t)-1);
        cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
    }

    return MA_SUCCESS;
}

ma_result ma_context_get_device_info__null(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_uint32 iFormat;

    ma_assert(pContext != NULL);

    if (pDeviceID != NULL && pDeviceID->nullbackend != 0) {
        return MA_NO_DEVICE;   /* Don't know the device. */
    }

    /* Name / Description */
    if (deviceType == ma_device_type_playback) {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), "NULL Playback Device", (size_t)-1);
    } else {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), "NULL Capture Device", (size_t)-1);
    }

    /* Support everything on the null backend. */
    pDeviceInfo->formatCount = ma_format_count - 1;    /* Minus one because we don't want to include ma_format_unknown. */
    for (iFormat = 0; iFormat < pDeviceInfo->formatCount; ++iFormat) {
        pDeviceInfo->formats[iFormat] = (ma_format)(iFormat + 1);  /* +1 to skip over ma_format_unknown. */
    }

    pDeviceInfo->minChannels   = 1;
    pDeviceInfo->maxChannels   = MA_MAX_CHANNELS;
    pDeviceInfo->minSampleRate = MA_SAMPLE_RATE_8000;
    pDeviceInfo->maxSampleRate = MA_SAMPLE_RATE_384000;

    (void)pContext;
    (void)shareMode;
    return MA_SUCCESS;
}


void ma_device_uninit__null(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    /* Keep it clean and wait for the device thread to finish before returning. */
    ma_device_do_operation__null(pDevice, MA_DEVICE_OP_KILL__NULL);

    /* At this point the loop in the device thread is as good as terminated so we can uninitialize our events. */
    ma_event_uninit(&pDevice->null_device.operationCompletionEvent);
    ma_event_uninit(&pDevice->null_device.operationEvent);
}

ma_result ma_device_init__null(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result;
    ma_uint32 bufferSizeInFrames;

    ma_assert(pDevice != NULL);

    ma_zero_object(&pDevice->null_device);

    bufferSizeInFrames = pConfig->bufferSizeInFrames;
    if (bufferSizeInFrames == 0) {
        bufferSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(pConfig->bufferSizeInMilliseconds, pConfig->sampleRate);
    }

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_strncpy_s(pDevice->capture.name,  sizeof(pDevice->capture.name),  "NULL Capture Device",  (size_t)-1);
        pDevice->capture.internalFormat             = pConfig->capture.format;
        pDevice->capture.internalChannels           = pConfig->capture.channels;
        ma_channel_map_copy(pDevice->capture.internalChannelMap, pConfig->capture.channelMap, pConfig->capture.channels);
        pDevice->capture.internalBufferSizeInFrames = bufferSizeInFrames;
        pDevice->capture.internalPeriods            = pConfig->periods;
    }
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_strncpy_s(pDevice->playback.name, sizeof(pDevice->playback.name), "NULL Playback Device", (size_t)-1);
        pDevice->playback.internalFormat             = pConfig->playback.format;
        pDevice->playback.internalChannels           = pConfig->playback.channels;
        ma_channel_map_copy(pDevice->playback.internalChannelMap, pConfig->playback.channelMap, pConfig->playback.channels);
        pDevice->playback.internalBufferSizeInFrames = bufferSizeInFrames;
        pDevice->playback.internalPeriods            = pConfig->periods;
    }

    /*
    In order to get timing right, we need to create a thread that does nothing but keeps track of the timer. This timer is started when the
    first period is "written" to it, and then stopped in ma_device_stop__null().
    */
    result = ma_event_init(pContext, &pDevice->null_device.operationEvent);
    if (result != MA_SUCCESS) {
        return result;
    }

    result = ma_event_init(pContext, &pDevice->null_device.operationCompletionEvent);
    if (result != MA_SUCCESS) {
        return result;
    }

    result = ma_thread_create(pContext, &pDevice->thread, ma_device_thread__null, pDevice);
    if (result != MA_SUCCESS) {
        return result;
    }

    return MA_SUCCESS;
}

ma_result ma_device_start__null(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    ma_device_do_operation__null(pDevice, MA_DEVICE_OP_START__NULL);

    ma_atomic_exchange_32(&pDevice->null_device.isStarted, MA_TRUE);
    return MA_SUCCESS;
}

ma_result ma_device_stop__null(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    ma_device_do_operation__null(pDevice, MA_DEVICE_OP_SUSPEND__NULL);

    ma_atomic_exchange_32(&pDevice->null_device.isStarted, MA_FALSE);
    return MA_SUCCESS;
}

ma_result ma_device_write__null(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount)
{
    ma_result result = MA_SUCCESS;
    ma_uint32 totalPCMFramesProcessed;
    ma_bool32 wasStartedOnEntry;

    wasStartedOnEntry = pDevice->null_device.isStarted;

    /* Keep going until everything has been read. */
    totalPCMFramesProcessed = 0;
    while (totalPCMFramesProcessed < frameCount) {
        ma_uint64 targetFrame;

        /* If there are any frames remaining in the current period, consume those first. */
        if (pDevice->null_device.currentPeriodFramesRemainingPlayback > 0) {
            ma_uint32 framesRemaining = (frameCount - totalPCMFramesProcessed);
            ma_uint32 framesToProcess = pDevice->null_device.currentPeriodFramesRemainingPlayback;
            if (framesToProcess > framesRemaining) {
                framesToProcess = framesRemaining;
            }

            /* We don't actually do anything with pPCMFrames, so just mark it as unused to prevent a warning. */
            (void)pPCMFrames;

            pDevice->null_device.currentPeriodFramesRemainingPlayback -= framesToProcess;
            totalPCMFramesProcessed += framesToProcess;
        }

        /* If we've consumed the current period we'll need to mark it as such an ensure the device is started if it's not already. */
        if (pDevice->null_device.currentPeriodFramesRemainingPlayback == 0) {
            pDevice->null_device.currentPeriodFramesRemainingPlayback = 0;

            if (!pDevice->null_device.isStarted && !wasStartedOnEntry) {
                result = ma_device_start__null(pDevice);
                if (result != MA_SUCCESS) {
                    break;
                }
            }
        }

        /* If we've consumed the whole buffer we can return now. */
        ma_assert(totalPCMFramesProcessed <= frameCount);
        if (totalPCMFramesProcessed == frameCount) {
            break;
        }

        /* Getting here means we've still got more frames to consume, we but need to wait for it to become available. */
        targetFrame = pDevice->null_device.lastProcessedFramePlayback;
        for (;;) {
            ma_uint64 currentFrame;

            /* Stop waiting if the device has been stopped. */
            if (!pDevice->null_device.isStarted) {
                break;
            }

            currentFrame = ma_device_get_total_run_time_in_frames__null(pDevice);
            if (currentFrame >= targetFrame) {
                break;
            }

            /* Getting here means we haven't yet reached the target sample, so continue waiting. */
            ma_sleep(10);
        }

        pDevice->null_device.lastProcessedFramePlayback          += pDevice->playback.internalBufferSizeInFrames / pDevice->playback.internalPeriods;
        pDevice->null_device.currentPeriodFramesRemainingPlayback = pDevice->playback.internalBufferSizeInFrames / pDevice->playback.internalPeriods;
    }

    return result;
}

ma_result ma_device_read__null(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount)
{
    ma_result result = MA_SUCCESS;
    ma_uint32 totalPCMFramesProcessed;

    /* The device needs to be started immediately. */
    if (!pDevice->null_device.isStarted) {
        result = ma_device_start__null(pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    /* Keep going until everything has been read. */
    totalPCMFramesProcessed = 0;
    while (totalPCMFramesProcessed < frameCount) {
        ma_uint64 targetFrame;

        /* If there are any frames remaining in the current period, consume those first. */
        if (pDevice->null_device.currentPeriodFramesRemainingCapture > 0) {
            ma_uint32 bpf = ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
            ma_uint32 framesRemaining = (frameCount - totalPCMFramesProcessed);
            ma_uint32 framesToProcess = pDevice->null_device.currentPeriodFramesRemainingCapture;
            if (framesToProcess > framesRemaining) {
                framesToProcess = framesRemaining;
            }

            /* We need to ensured the output buffer is zeroed. */
            ma_zero_memory(ma_offset_ptr(pPCMFrames, totalPCMFramesProcessed*bpf), framesToProcess*bpf);

            pDevice->null_device.currentPeriodFramesRemainingCapture -= framesToProcess;
            totalPCMFramesProcessed += framesToProcess;
        }

        /* If we've consumed the current period we'll need to mark it as such an ensure the device is started if it's not already. */
        if (pDevice->null_device.currentPeriodFramesRemainingCapture == 0) {
            pDevice->null_device.currentPeriodFramesRemainingCapture = 0;
        }

        /* If we've consumed the whole buffer we can return now. */
        ma_assert(totalPCMFramesProcessed <= frameCount);
        if (totalPCMFramesProcessed == frameCount) {
            break;
        }

        /* Getting here means we've still got more frames to consume, we but need to wait for it to become available. */
        targetFrame = pDevice->null_device.lastProcessedFrameCapture + (pDevice->capture.internalBufferSizeInFrames / pDevice->capture.internalPeriods);
        for (;;) {
            ma_uint64 currentFrame;

            /* Stop waiting if the device has been stopped. */
            if (!pDevice->null_device.isStarted) {
                break;
            }

            currentFrame = ma_device_get_total_run_time_in_frames__null(pDevice);
            if (currentFrame >= targetFrame) {
                break;
            }

            /* Getting here means we haven't yet reached the target sample, so continue waiting. */
            ma_sleep(10);
        }

        pDevice->null_device.lastProcessedFrameCapture          += pDevice->capture.internalBufferSizeInFrames / pDevice->capture.internalPeriods;
        pDevice->null_device.currentPeriodFramesRemainingCapture = pDevice->capture.internalBufferSizeInFrames / pDevice->capture.internalPeriods;
    }

    return result;
}

ma_result ma_context_uninit__null(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_null);

    (void)pContext;
    return MA_SUCCESS;
}

ma_result ma_context_init__null(const ma_context_config* pConfig, ma_context* pContext)
{
    ma_assert(pContext != NULL);

    (void)pConfig;

    pContext->onUninit              = ma_context_uninit__null;
    pContext->onDeviceIDEqual       = ma_context_is_device_id_equal__null;
    pContext->onEnumDevices         = ma_context_enumerate_devices__null;
    pContext->onGetDeviceInfo       = ma_context_get_device_info__null;
    pContext->onDeviceInit          = ma_device_init__null;
    pContext->onDeviceUninit        = ma_device_uninit__null;
    pContext->onDeviceStart         = ma_device_start__null;
    pContext->onDeviceStop          = ma_device_stop__null;
    pContext->onDeviceWrite         = ma_device_write__null;
    pContext->onDeviceRead          = ma_device_read__null;

    /* The null backend always works. */
    return MA_SUCCESS;
}
#endif


/*******************************************************************************

WIN32 COMMON

*******************************************************************************/
#if defined(MA_WIN32)
#if defined(MA_WIN32_DESKTOP)
    #define ma_CoInitializeEx(pContext, pvReserved, dwCoInit)                          ((MA_PFN_CoInitializeEx)pContext->win32.CoInitializeEx)(pvReserved, dwCoInit)
    #define ma_CoUninitialize(pContext)                                                ((MA_PFN_CoUninitialize)pContext->win32.CoUninitialize)()
    #define ma_CoCreateInstance(pContext, rclsid, pUnkOuter, dwClsContext, riid, ppv)  ((MA_PFN_CoCreateInstance)pContext->win32.CoCreateInstance)(rclsid, pUnkOuter, dwClsContext, riid, ppv)
    #define ma_CoTaskMemFree(pContext, pv)                                             ((MA_PFN_CoTaskMemFree)pContext->win32.CoTaskMemFree)(pv)
    #define ma_PropVariantClear(pContext, pvar)                                        ((MA_PFN_PropVariantClear)pContext->win32.PropVariantClear)(pvar)
#else
    #define ma_CoInitializeEx(pContext, pvReserved, dwCoInit)                          CoInitializeEx(pvReserved, dwCoInit)
    #define ma_CoUninitialize(pContext)                                                CoUninitialize()
    #define ma_CoCreateInstance(pContext, rclsid, pUnkOuter, dwClsContext, riid, ppv)  CoCreateInstance(rclsid, pUnkOuter, dwClsContext, riid, ppv)
    #define ma_CoTaskMemFree(pContext, pv)                                             CoTaskMemFree(pv)
    #define ma_PropVariantClear(pContext, pvar)                                        PropVariantClear(pvar)
#endif

#if !defined(MAXULONG_PTR)
typedef size_t DWORD_PTR;
#endif

#if !defined(WAVE_FORMAT_44M08)
#define WAVE_FORMAT_44M08 0x00000100
#define WAVE_FORMAT_44S08 0x00000200
#define WAVE_FORMAT_44M16 0x00000400
#define WAVE_FORMAT_44S16 0x00000800
#define WAVE_FORMAT_48M08 0x00001000
#define WAVE_FORMAT_48S08 0x00002000
#define WAVE_FORMAT_48M16 0x00004000
#define WAVE_FORMAT_48S16 0x00008000
#define WAVE_FORMAT_96M08 0x00010000
#define WAVE_FORMAT_96S08 0x00020000
#define WAVE_FORMAT_96M16 0x00040000
#define WAVE_FORMAT_96S16 0x00080000
#endif

#ifndef SPEAKER_FRONT_LEFT
#define SPEAKER_FRONT_LEFT            0x1
#define SPEAKER_FRONT_RIGHT           0x2
#define SPEAKER_FRONT_CENTER          0x4
#define SPEAKER_LOW_FREQUENCY         0x8
#define SPEAKER_BACK_LEFT             0x10
#define SPEAKER_BACK_RIGHT            0x20
#define SPEAKER_FRONT_LEFT_OF_CENTER  0x40
#define SPEAKER_FRONT_RIGHT_OF_CENTER 0x80
#define SPEAKER_BACK_CENTER           0x100
#define SPEAKER_SIDE_LEFT             0x200
#define SPEAKER_SIDE_RIGHT            0x400
#define SPEAKER_TOP_CENTER            0x800
#define SPEAKER_TOP_FRONT_LEFT        0x1000
#define SPEAKER_TOP_FRONT_CENTER      0x2000
#define SPEAKER_TOP_FRONT_RIGHT       0x4000
#define SPEAKER_TOP_BACK_LEFT         0x8000
#define SPEAKER_TOP_BACK_CENTER       0x10000
#define SPEAKER_TOP_BACK_RIGHT        0x20000
#endif

/*
The SDK that comes with old versions of MSVC (VC6, for example) does not appear to define WAVEFORMATEXTENSIBLE. We
define our own implementation in this case.
*/
#if (defined(_MSC_VER) && !defined(_WAVEFORMATEXTENSIBLE_)) || defined(__DMC__)
typedef struct
{
    WAVEFORMATEX Format;
    union
    {
        WORD wValidBitsPerSample;
        WORD wSamplesPerBlock;
        WORD wReserved;
    } Samples;
    DWORD dwChannelMask;
    GUID SubFormat;
} WAVEFORMATEXTENSIBLE;
#endif

#ifndef WAVE_FORMAT_EXTENSIBLE
#define WAVE_FORMAT_EXTENSIBLE  0xFFFE
#endif

#ifndef WAVE_FORMAT_IEEE_FLOAT
#define WAVE_FORMAT_IEEE_FLOAT  0x0003
#endif

GUID MA_GUID_NULL = {0x00000000, 0x0000, 0x0000, {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}};

/* Converts an individual Win32-style channel identifier (SPEAKER_FRONT_LEFT, etc.) to miniaudio. */
ma_uint8 ma_channel_id_to_ma__win32(DWORD id)
{
    switch (id)
    {
        case SPEAKER_FRONT_LEFT:            return MA_CHANNEL_FRONT_LEFT;
        case SPEAKER_FRONT_RIGHT:           return MA_CHANNEL_FRONT_RIGHT;
        case SPEAKER_FRONT_CENTER:          return MA_CHANNEL_FRONT_CENTER;
        case SPEAKER_LOW_FREQUENCY:         return MA_CHANNEL_LFE;
        case SPEAKER_BACK_LEFT:             return MA_CHANNEL_BACK_LEFT;
        case SPEAKER_BACK_RIGHT:            return MA_CHANNEL_BACK_RIGHT;
        case SPEAKER_FRONT_LEFT_OF_CENTER:  return MA_CHANNEL_FRONT_LEFT_CENTER;
        case SPEAKER_FRONT_RIGHT_OF_CENTER: return MA_CHANNEL_FRONT_RIGHT_CENTER;
        case SPEAKER_BACK_CENTER:           return MA_CHANNEL_BACK_CENTER;
        case SPEAKER_SIDE_LEFT:             return MA_CHANNEL_SIDE_LEFT;
        case SPEAKER_SIDE_RIGHT:            return MA_CHANNEL_SIDE_RIGHT;
        case SPEAKER_TOP_CENTER:            return MA_CHANNEL_TOP_CENTER;
        case SPEAKER_TOP_FRONT_LEFT:        return MA_CHANNEL_TOP_FRONT_LEFT;
        case SPEAKER_TOP_FRONT_CENTER:      return MA_CHANNEL_TOP_FRONT_CENTER;
        case SPEAKER_TOP_FRONT_RIGHT:       return MA_CHANNEL_TOP_FRONT_RIGHT;
        case SPEAKER_TOP_BACK_LEFT:         return MA_CHANNEL_TOP_BACK_LEFT;
        case SPEAKER_TOP_BACK_CENTER:       return MA_CHANNEL_TOP_BACK_CENTER;
        case SPEAKER_TOP_BACK_RIGHT:        return MA_CHANNEL_TOP_BACK_RIGHT;
        default: return 0;
    }
}

/* Converts an individual miniaudio channel identifier (MA_CHANNEL_FRONT_LEFT, etc.) to Win32-style. */
DWORD ma_channel_id_to_win32(DWORD id)
{
    switch (id)
    {
        case MA_CHANNEL_MONO:               return SPEAKER_FRONT_CENTER;
        case MA_CHANNEL_FRONT_LEFT:         return SPEAKER_FRONT_LEFT;
        case MA_CHANNEL_FRONT_RIGHT:        return SPEAKER_FRONT_RIGHT;
        case MA_CHANNEL_FRONT_CENTER:       return SPEAKER_FRONT_CENTER;
        case MA_CHANNEL_LFE:                return SPEAKER_LOW_FREQUENCY;
        case MA_CHANNEL_BACK_LEFT:          return SPEAKER_BACK_LEFT;
        case MA_CHANNEL_BACK_RIGHT:         return SPEAKER_BACK_RIGHT;
        case MA_CHANNEL_FRONT_LEFT_CENTER:  return SPEAKER_FRONT_LEFT_OF_CENTER;
        case MA_CHANNEL_FRONT_RIGHT_CENTER: return SPEAKER_FRONT_RIGHT_OF_CENTER;
        case MA_CHANNEL_BACK_CENTER:        return SPEAKER_BACK_CENTER;
        case MA_CHANNEL_SIDE_LEFT:          return SPEAKER_SIDE_LEFT;
        case MA_CHANNEL_SIDE_RIGHT:         return SPEAKER_SIDE_RIGHT;
        case MA_CHANNEL_TOP_CENTER:         return SPEAKER_TOP_CENTER;
        case MA_CHANNEL_TOP_FRONT_LEFT:     return SPEAKER_TOP_FRONT_LEFT;
        case MA_CHANNEL_TOP_FRONT_CENTER:   return SPEAKER_TOP_FRONT_CENTER;
        case MA_CHANNEL_TOP_FRONT_RIGHT:    return SPEAKER_TOP_FRONT_RIGHT;
        case MA_CHANNEL_TOP_BACK_LEFT:      return SPEAKER_TOP_BACK_LEFT;
        case MA_CHANNEL_TOP_BACK_CENTER:    return SPEAKER_TOP_BACK_CENTER;
        case MA_CHANNEL_TOP_BACK_RIGHT:     return SPEAKER_TOP_BACK_RIGHT;
        default: return 0;
    }
}

/* Converts a channel mapping to a Win32-style channel mask. */
DWORD ma_channel_map_to_channel_mask__win32(const ma_channel channelMap[MA_MAX_CHANNELS], ma_uint32 channels)
{
    DWORD dwChannelMask = 0;
    ma_uint32 iChannel;

    for (iChannel = 0; iChannel < channels; ++iChannel) {
        dwChannelMask |= ma_channel_id_to_win32(channelMap[iChannel]);
    }

    return dwChannelMask;
}

/* Converts a Win32-style channel mask to a miniaudio channel map. */
void ma_channel_mask_to_channel_map__win32(DWORD dwChannelMask, ma_uint32 channels, ma_channel channelMap[MA_MAX_CHANNELS])
{
    if (channels == 1 && dwChannelMask == 0) {
        channelMap[0] = MA_CHANNEL_MONO;
    } else if (channels == 2 && dwChannelMask == 0) {
        channelMap[0] = MA_CHANNEL_FRONT_LEFT;
        channelMap[1] = MA_CHANNEL_FRONT_RIGHT;
    } else {
        if (channels == 1 && (dwChannelMask & SPEAKER_FRONT_CENTER) != 0) {
            channelMap[0] = MA_CHANNEL_MONO;
        } else {
            /* Just iterate over each bit. */
            ma_uint32 iChannel = 0;
            ma_uint32 iBit;

            for (iBit = 0; iBit < 32; ++iBit) {
                DWORD bitValue = (dwChannelMask & (1UL << iBit));
                if (bitValue != 0) {
                    /* The bit is set. */
                    channelMap[iChannel] = ma_channel_id_to_ma__win32(bitValue);
                    iChannel += 1;
                }
            }
        }
    }
}

#ifdef __cplusplus
ma_bool32 ma_is_guid_equal(const void* a, const void* b)
{
    return IsEqualGUID(*(const GUID*)a, *(const GUID*)b);
}
#else
#define ma_is_guid_equal(a, b) IsEqualGUID((const GUID*)a, (const GUID*)b)
#endif

ma_format ma_format_from_WAVEFORMATEX(const WAVEFORMATEX* pWF)
{
    ma_assert(pWF != NULL);

    if (pWF->wFormatTag == WAVE_FORMAT_EXTENSIBLE) {
        const WAVEFORMATEXTENSIBLE* pWFEX = (const WAVEFORMATEXTENSIBLE*)pWF;
        if (ma_is_guid_equal(&pWFEX->SubFormat, &MA_GUID_KSDATAFORMAT_SUBTYPE_PCM)) {
            if (pWFEX->Samples.wValidBitsPerSample == 32) {
                return ma_format_s32;
            }
            if (pWFEX->Samples.wValidBitsPerSample == 24) {
                if (pWFEX->Format.wBitsPerSample == 32) {
                    /*return ma_format_s24_32;*/
                }
                if (pWFEX->Format.wBitsPerSample == 24) {
                    return ma_format_s24;
                }
            }
            if (pWFEX->Samples.wValidBitsPerSample == 16) {
                return ma_format_s16;
            }
            if (pWFEX->Samples.wValidBitsPerSample == 8) {
                return ma_format_u8;
            }
        }
        if (ma_is_guid_equal(&pWFEX->SubFormat, &MA_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT)) {
            if (pWFEX->Samples.wValidBitsPerSample == 32) {
                return ma_format_f32;
            }
            /*
            if (pWFEX->Samples.wValidBitsPerSample == 64) {
                return ma_format_f64;
            }
            */
        }
    } else {
        if (pWF->wFormatTag == WAVE_FORMAT_PCM) {
            if (pWF->wBitsPerSample == 32) {
                return ma_format_s32;
            }
            if (pWF->wBitsPerSample == 24) {
                return ma_format_s24;
            }
            if (pWF->wBitsPerSample == 16) {
                return ma_format_s16;
            }
            if (pWF->wBitsPerSample == 8) {
                return ma_format_u8;
            }
        }
        if (pWF->wFormatTag == WAVE_FORMAT_IEEE_FLOAT) {
            if (pWF->wBitsPerSample == 32) {
                return ma_format_f32;
            }
            if (pWF->wBitsPerSample == 64) {
                /*return ma_format_f64;*/
            }
        }
    }

    return ma_format_unknown;
}
#endif


/*******************************************************************************

WASAPI Backend

*******************************************************************************/
#ifdef MA_HAS_WASAPI
#if 0
#if defined(_MSC_VER)
    #pragma warning(push)
    #pragma warning(disable:4091)   /* 'typedef ': ignored on left of '' when no variable is declared */
#endif
#include <audioclient.h>
#include <mmdeviceapi.h>
#if defined(_MSC_VER)
    #pragma warning(pop)
#endif
#endif  /* 0 */


/* Some compilers don't define VerifyVersionInfoW. Need to write this ourselves. */
#if defined(__DMC__)
#define _WIN32_WINNT_VISTA      0x0600
#define VER_MINORVERSION        0x01
#define VER_MAJORVERSION        0x02
#define VER_SERVICEPACKMAJOR    0x20
#define VER_GREATER_EQUAL       0x03

typedef struct  {
    DWORD dwOSVersionInfoSize;
    DWORD dwMajorVersion;
    DWORD dwMinorVersion;
    DWORD dwBuildNumber;
    DWORD dwPlatformId;
    WCHAR szCSDVersion[128];
    WORD  wServicePackMajor;
    WORD  wServicePackMinor;
    WORD  wSuiteMask;
    BYTE  wProductType;
    BYTE  wReserved;
} ma_OSVERSIONINFOEXW;

BOOL WINAPI VerifyVersionInfoW(ma_OSVERSIONINFOEXW* lpVersionInfo, DWORD dwTypeMask, DWORDLONG dwlConditionMask);
ULONGLONG WINAPI VerSetConditionMask(ULONGLONG dwlConditionMask, DWORD dwTypeBitMask, BYTE dwConditionMask);
#else
typedef OSVERSIONINFOEXW ma_OSVERSIONINFOEXW;
#endif


#ifndef PROPERTYKEY_DEFINED
#define PROPERTYKEY_DEFINED
typedef struct
{
    GUID fmtid;
    DWORD pid;
} PROPERTYKEY;
#endif

/* Some compilers don't define PropVariantInit(). We just do this ourselves since it's just a memset(). */
static MA_INLINE void ma_PropVariantInit(PROPVARIANT* pProp)
{
    ma_zero_object(pProp);
}


const PROPERTYKEY MA_PKEY_Device_FriendlyName             = {{0xA45C254E, 0xDF1C, 0x4EFD, {0x80, 0x20, 0x67, 0xD1, 0x46, 0xA8, 0x50, 0xE0}}, 14};
const PROPERTYKEY MA_PKEY_AudioEngine_DeviceFormat        = {{0xF19F064D, 0x82C,  0x4E27, {0xBC, 0x73, 0x68, 0x82, 0xA1, 0xBB, 0x8E, 0x4C}},  0};

const IID MA_IID_IUnknown                                 = {0x00000000, 0x0000, 0x0000, {0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46}}; /* 00000000-0000-0000-C000-000000000046 */
const IID MA_IID_IAgileObject                             = {0x94EA2B94, 0xE9CC, 0x49E0, {0xC0, 0xFF, 0xEE, 0x64, 0xCA, 0x8F, 0x5B, 0x90}}; /* 94EA2B94-E9CC-49E0-C0FF-EE64CA8F5B90 */

const IID MA_IID_IAudioClient                             = {0x1CB9AD4C, 0xDBFA, 0x4C32, {0xB1, 0x78, 0xC2, 0xF5, 0x68, 0xA7, 0x03, 0xB2}}; /* 1CB9AD4C-DBFA-4C32-B178-C2F568A703B2 = __uuidof(IAudioClient) */
const IID MA_IID_IAudioClient2                            = {0x726778CD, 0xF60A, 0x4EDA, {0x82, 0xDE, 0xE4, 0x76, 0x10, 0xCD, 0x78, 0xAA}}; /* 726778CD-F60A-4EDA-82DE-E47610CD78AA = __uuidof(IAudioClient2) */
const IID MA_IID_IAudioClient3                            = {0x7ED4EE07, 0x8E67, 0x4CD4, {0x8C, 0x1A, 0x2B, 0x7A, 0x59, 0x87, 0xAD, 0x42}}; /* 7ED4EE07-8E67-4CD4-8C1A-2B7A5987AD42 = __uuidof(IAudioClient3) */
const IID MA_IID_IAudioRenderClient                       = {0xF294ACFC, 0x3146, 0x4483, {0xA7, 0xBF, 0xAD, 0xDC, 0xA7, 0xC2, 0x60, 0xE2}}; /* F294ACFC-3146-4483-A7BF-ADDCA7C260E2 = __uuidof(IAudioRenderClient) */
const IID MA_IID_IAudioCaptureClient                      = {0xC8ADBD64, 0xE71E, 0x48A0, {0xA4, 0xDE, 0x18, 0x5C, 0x39, 0x5C, 0xD3, 0x17}}; /* C8ADBD64-E71E-48A0-A4DE-185C395CD317 = __uuidof(IAudioCaptureClient) */
const IID MA_IID_IMMNotificationClient                    = {0x7991EEC9, 0x7E89, 0x4D85, {0x83, 0x90, 0x6C, 0x70, 0x3C, 0xEC, 0x60, 0xC0}}; /* 7991EEC9-7E89-4D85-8390-6C703CEC60C0 = __uuidof(IMMNotificationClient) */
#ifndef MA_WIN32_DESKTOP
const IID MA_IID_DEVINTERFACE_AUDIO_RENDER                = {0xE6327CAD, 0xDCEC, 0x4949, {0xAE, 0x8A, 0x99, 0x1E, 0x97, 0x6A, 0x79, 0xD2}}; /* E6327CAD-DCEC-4949-AE8A-991E976A79D2 */
const IID MA_IID_DEVINTERFACE_AUDIO_CAPTURE               = {0x2EEF81BE, 0x33FA, 0x4800, {0x96, 0x70, 0x1C, 0xD4, 0x74, 0x97, 0x2C, 0x3F}}; /* 2EEF81BE-33FA-4800-9670-1CD474972C3F */
const IID MA_IID_IActivateAudioInterfaceCompletionHandler = {0x41D949AB, 0x9862, 0x444A, {0x80, 0xF6, 0xC2, 0x61, 0x33, 0x4D, 0xA5, 0xEB}}; /* 41D949AB-9862-444A-80F6-C261334DA5EB */
#endif

const IID MA_CLSID_MMDeviceEnumerator_Instance            = {0xBCDE0395, 0xE52F, 0x467C, {0x8E, 0x3D, 0xC4, 0x57, 0x92, 0x91, 0x69, 0x2E}}; /* BCDE0395-E52F-467C-8E3D-C4579291692E = __uuidof(MMDeviceEnumerator) */
const IID MA_IID_IMMDeviceEnumerator_Instance             = {0xA95664D2, 0x9614, 0x4F35, {0xA7, 0x46, 0xDE, 0x8D, 0xB6, 0x36, 0x17, 0xE6}}; /* A95664D2-9614-4F35-A746-DE8DB63617E6 = __uuidof(IMMDeviceEnumerator) */
#ifdef __cplusplus
#define MA_CLSID_MMDeviceEnumerator MA_CLSID_MMDeviceEnumerator_Instance
#define MA_IID_IMMDeviceEnumerator  MA_IID_IMMDeviceEnumerator_Instance
#else
#define MA_CLSID_MMDeviceEnumerator &MA_CLSID_MMDeviceEnumerator_Instance
#define MA_IID_IMMDeviceEnumerator  &MA_IID_IMMDeviceEnumerator_Instance
#endif

typedef struct ma_IUnknown                                 ma_IUnknown;
#ifdef MA_WIN32_DESKTOP
#define MA_MM_DEVICE_STATE_ACTIVE                          1
#define MA_MM_DEVICE_STATE_DISABLED                        2
#define MA_MM_DEVICE_STATE_NOTPRESENT                      4
#define MA_MM_DEVICE_STATE_UNPLUGGED                       8

typedef struct ma_IMMDeviceEnumerator                      ma_IMMDeviceEnumerator;
typedef struct ma_IMMDeviceCollection                      ma_IMMDeviceCollection;
typedef struct ma_IMMDevice                                ma_IMMDevice;
#else
typedef struct ma_IActivateAudioInterfaceCompletionHandler ma_IActivateAudioInterfaceCompletionHandler;
typedef struct ma_IActivateAudioInterfaceAsyncOperation    ma_IActivateAudioInterfaceAsyncOperation;
#endif
typedef struct ma_IPropertyStore                           ma_IPropertyStore;
typedef struct ma_IAudioClient                             ma_IAudioClient;
typedef struct ma_IAudioClient2                            ma_IAudioClient2;
typedef struct ma_IAudioClient3                            ma_IAudioClient3;
typedef struct ma_IAudioRenderClient                       ma_IAudioRenderClient;
typedef struct ma_IAudioCaptureClient                      ma_IAudioCaptureClient;

typedef ma_int64                                           MA_REFERENCE_TIME;

#define MA_AUDCLNT_STREAMFLAGS_CROSSPROCESS                0x00010000
#define MA_AUDCLNT_STREAMFLAGS_LOOPBACK                    0x00020000
#define MA_AUDCLNT_STREAMFLAGS_EVENTCALLBACK               0x00040000
#define MA_AUDCLNT_STREAMFLAGS_NOPERSIST                   0x00080000
#define MA_AUDCLNT_STREAMFLAGS_RATEADJUST                  0x00100000
#define MA_AUDCLNT_STREAMFLAGS_SRC_DEFAULT_QUALITY         0x08000000
#define MA_AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM              0x80000000
#define MA_AUDCLNT_SESSIONFLAGS_EXPIREWHENUNOWNED          0x10000000
#define MA_AUDCLNT_SESSIONFLAGS_DISPLAY_HIDE               0x20000000
#define MA_AUDCLNT_SESSIONFLAGS_DISPLAY_HIDEWHENEXPIRED    0x40000000

/* We only care about a few error codes. */
#define MA_AUDCLNT_E_INVALID_DEVICE_PERIOD                 (-2004287456)
#define MA_AUDCLNT_E_BUFFER_SIZE_NOT_ALIGNED               (-2004287463)
#define MA_AUDCLNT_S_BUFFER_EMPTY                          (143196161)
#define MA_AUDCLNT_E_DEVICE_IN_USE                         (-2004287478)

typedef enum
{
    ma_eRender  = 0,
    ma_eCapture = 1,
    ma_eAll     = 2
} ma_EDataFlow;

typedef enum
{
    ma_eConsole        = 0,
    ma_eMultimedia     = 1,
    ma_eCommunications = 2
} ma_ERole;

typedef enum
{
    MA_AUDCLNT_SHAREMODE_SHARED,
    MA_AUDCLNT_SHAREMODE_EXCLUSIVE
} MA_AUDCLNT_SHAREMODE;

typedef enum
{
    MA_AudioCategory_Other = 0  /* <-- miniaudio is only caring about Other. */
} MA_AUDIO_STREAM_CATEGORY;

typedef struct
{
    UINT32 cbSize;
    BOOL bIsOffload;
    MA_AUDIO_STREAM_CATEGORY eCategory;
} ma_AudioClientProperties;

/* IUnknown */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IUnknown* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IUnknown* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IUnknown* pThis);
} ma_IUnknownVtbl;
struct ma_IUnknown
{
    ma_IUnknownVtbl* lpVtbl;
};
HRESULT ma_IUnknown_QueryInterface(ma_IUnknown* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IUnknown_AddRef(ma_IUnknown* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IUnknown_Release(ma_IUnknown* pThis)                                                { return pThis->lpVtbl->Release(pThis); }

#ifdef MA_WIN32_DESKTOP
    /* IMMNotificationClient */
    typedef struct
    {
        /* IUnknown */
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IMMNotificationClient* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IMMNotificationClient* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (ma_IMMNotificationClient* pThis);

        /* IMMNotificationClient */
        HRESULT (STDMETHODCALLTYPE * OnDeviceStateChanged)  (ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID, DWORD dwNewState);
        HRESULT (STDMETHODCALLTYPE * OnDeviceAdded)         (ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID);
        HRESULT (STDMETHODCALLTYPE * OnDeviceRemoved)       (ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID);
        HRESULT (STDMETHODCALLTYPE * OnDefaultDeviceChanged)(ma_IMMNotificationClient* pThis, ma_EDataFlow dataFlow, ma_ERole role, LPCWSTR pDefaultDeviceID);
        HRESULT (STDMETHODCALLTYPE * OnPropertyValueChanged)(ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID, const PROPERTYKEY key);
    } ma_IMMNotificationClientVtbl;

    /* IMMDeviceEnumerator */
    typedef struct
    {
        /* IUnknown */
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IMMDeviceEnumerator* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IMMDeviceEnumerator* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (ma_IMMDeviceEnumerator* pThis);

        /* IMMDeviceEnumerator */
        HRESULT (STDMETHODCALLTYPE * EnumAudioEndpoints)                    (ma_IMMDeviceEnumerator* pThis, ma_EDataFlow dataFlow, DWORD dwStateMask, ma_IMMDeviceCollection** ppDevices);
        HRESULT (STDMETHODCALLTYPE * GetDefaultAudioEndpoint)               (ma_IMMDeviceEnumerator* pThis, ma_EDataFlow dataFlow, ma_ERole role, ma_IMMDevice** ppEndpoint);
        HRESULT (STDMETHODCALLTYPE * GetDevice)                             (ma_IMMDeviceEnumerator* pThis, LPCWSTR pID, ma_IMMDevice** ppDevice);
        HRESULT (STDMETHODCALLTYPE * RegisterEndpointNotificationCallback)  (ma_IMMDeviceEnumerator* pThis, ma_IMMNotificationClient* pClient);
        HRESULT (STDMETHODCALLTYPE * UnregisterEndpointNotificationCallback)(ma_IMMDeviceEnumerator* pThis, ma_IMMNotificationClient* pClient);
    } ma_IMMDeviceEnumeratorVtbl;
    struct ma_IMMDeviceEnumerator
    {
        ma_IMMDeviceEnumeratorVtbl* lpVtbl;
    };
    HRESULT ma_IMMDeviceEnumerator_QueryInterface(ma_IMMDeviceEnumerator* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    ULONG   ma_IMMDeviceEnumerator_AddRef(ma_IMMDeviceEnumerator* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    ULONG   ma_IMMDeviceEnumerator_Release(ma_IMMDeviceEnumerator* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    HRESULT ma_IMMDeviceEnumerator_EnumAudioEndpoints(ma_IMMDeviceEnumerator* pThis, ma_EDataFlow dataFlow, DWORD dwStateMask, ma_IMMDeviceCollection** ppDevices) { return pThis->lpVtbl->EnumAudioEndpoints(pThis, dataFlow, dwStateMask, ppDevices); }
    HRESULT ma_IMMDeviceEnumerator_GetDefaultAudioEndpoint(ma_IMMDeviceEnumerator* pThis, ma_EDataFlow dataFlow, ma_ERole role, ma_IMMDevice** ppEndpoint) { return pThis->lpVtbl->GetDefaultAudioEndpoint(pThis, dataFlow, role, ppEndpoint); }
    HRESULT ma_IMMDeviceEnumerator_GetDevice(ma_IMMDeviceEnumerator* pThis, LPCWSTR pID, ma_IMMDevice** ppDevice)       { return pThis->lpVtbl->GetDevice(pThis, pID, ppDevice); }
    HRESULT ma_IMMDeviceEnumerator_RegisterEndpointNotificationCallback(ma_IMMDeviceEnumerator* pThis, ma_IMMNotificationClient* pClient) { return pThis->lpVtbl->RegisterEndpointNotificationCallback(pThis, pClient); }
    HRESULT ma_IMMDeviceEnumerator_UnregisterEndpointNotificationCallback(ma_IMMDeviceEnumerator* pThis, ma_IMMNotificationClient* pClient) { return pThis->lpVtbl->UnregisterEndpointNotificationCallback(pThis, pClient); }


    /* IMMDeviceCollection */
    typedef struct
    {
        /* IUnknown */
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IMMDeviceCollection* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IMMDeviceCollection* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (ma_IMMDeviceCollection* pThis);

        /* IMMDeviceCollection */
        HRESULT (STDMETHODCALLTYPE * GetCount)(ma_IMMDeviceCollection* pThis, UINT* pDevices);
        HRESULT (STDMETHODCALLTYPE * Item)    (ma_IMMDeviceCollection* pThis, UINT nDevice, ma_IMMDevice** ppDevice);
    } ma_IMMDeviceCollectionVtbl;
    struct ma_IMMDeviceCollection
    {
        ma_IMMDeviceCollectionVtbl* lpVtbl;
    };
    HRESULT ma_IMMDeviceCollection_QueryInterface(ma_IMMDeviceCollection* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    ULONG   ma_IMMDeviceCollection_AddRef(ma_IMMDeviceCollection* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    ULONG   ma_IMMDeviceCollection_Release(ma_IMMDeviceCollection* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    HRESULT ma_IMMDeviceCollection_GetCount(ma_IMMDeviceCollection* pThis, UINT* pDevices)                               { return pThis->lpVtbl->GetCount(pThis, pDevices); }
    HRESULT ma_IMMDeviceCollection_Item(ma_IMMDeviceCollection* pThis, UINT nDevice, ma_IMMDevice** ppDevice)           { return pThis->lpVtbl->Item(pThis, nDevice, ppDevice); }


    /* IMMDevice */
    typedef struct
    {
        /* IUnknown */
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IMMDevice* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IMMDevice* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (ma_IMMDevice* pThis);

        /* IMMDevice */
        HRESULT (STDMETHODCALLTYPE * Activate)         (ma_IMMDevice* pThis, const IID* const iid, DWORD dwClsCtx, PROPVARIANT* pActivationParams, void** ppInterface);
        HRESULT (STDMETHODCALLTYPE * OpenPropertyStore)(ma_IMMDevice* pThis, DWORD stgmAccess, ma_IPropertyStore** ppProperties);
        HRESULT (STDMETHODCALLTYPE * GetId)            (ma_IMMDevice* pThis, LPWSTR *pID);
        HRESULT (STDMETHODCALLTYPE * GetState)         (ma_IMMDevice* pThis, DWORD *pState);
    } ma_IMMDeviceVtbl;
    struct ma_IMMDevice
    {
        ma_IMMDeviceVtbl* lpVtbl;
    };
    HRESULT ma_IMMDevice_QueryInterface(ma_IMMDevice* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    ULONG   ma_IMMDevice_AddRef(ma_IMMDevice* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    ULONG   ma_IMMDevice_Release(ma_IMMDevice* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    HRESULT ma_IMMDevice_Activate(ma_IMMDevice* pThis, const IID* const iid, DWORD dwClsCtx, PROPVARIANT* pActivationParams, void** ppInterface) { return pThis->lpVtbl->Activate(pThis, iid, dwClsCtx, pActivationParams, ppInterface); }
    HRESULT ma_IMMDevice_OpenPropertyStore(ma_IMMDevice* pThis, DWORD stgmAccess, ma_IPropertyStore** ppProperties) { return pThis->lpVtbl->OpenPropertyStore(pThis, stgmAccess, ppProperties); }
    HRESULT ma_IMMDevice_GetId(ma_IMMDevice* pThis, LPWSTR *pID)                                     { return pThis->lpVtbl->GetId(pThis, pID); }
    HRESULT ma_IMMDevice_GetState(ma_IMMDevice* pThis, DWORD *pState)                                { return pThis->lpVtbl->GetState(pThis, pState); }
#else
    /* IActivateAudioInterfaceAsyncOperation */
    typedef struct
    {
        /* IUnknown */
        HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IActivateAudioInterfaceAsyncOperation* pThis, const IID* const riid, void** ppObject);
        ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IActivateAudioInterfaceAsyncOperation* pThis);
        ULONG   (STDMETHODCALLTYPE * Release)       (ma_IActivateAudioInterfaceAsyncOperation* pThis);

        /* IActivateAudioInterfaceAsyncOperation */
        HRESULT (STDMETHODCALLTYPE * GetActivateResult)(ma_IActivateAudioInterfaceAsyncOperation* pThis, HRESULT *pActivateResult, ma_IUnknown** ppActivatedInterface);
    } ma_IActivateAudioInterfaceAsyncOperationVtbl;
    struct ma_IActivateAudioInterfaceAsyncOperation
    {
        ma_IActivateAudioInterfaceAsyncOperationVtbl* lpVtbl;
    };
    HRESULT ma_IActivateAudioInterfaceAsyncOperation_QueryInterface(ma_IActivateAudioInterfaceAsyncOperation* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
    ULONG   ma_IActivateAudioInterfaceAsyncOperation_AddRef(ma_IActivateAudioInterfaceAsyncOperation* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
    ULONG   ma_IActivateAudioInterfaceAsyncOperation_Release(ma_IActivateAudioInterfaceAsyncOperation* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
    HRESULT ma_IActivateAudioInterfaceAsyncOperation_GetActivateResult(ma_IActivateAudioInterfaceAsyncOperation* pThis, HRESULT *pActivateResult, ma_IUnknown** ppActivatedInterface) { return pThis->lpVtbl->GetActivateResult(pThis, pActivateResult, ppActivatedInterface); }
#endif

/* IPropertyStore */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IPropertyStore* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IPropertyStore* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IPropertyStore* pThis);

    /* IPropertyStore */
    HRESULT (STDMETHODCALLTYPE * GetCount)(ma_IPropertyStore* pThis, DWORD* pPropCount);
    HRESULT (STDMETHODCALLTYPE * GetAt)   (ma_IPropertyStore* pThis, DWORD propIndex, PROPERTYKEY* pPropKey);
    HRESULT (STDMETHODCALLTYPE * GetValue)(ma_IPropertyStore* pThis, const PROPERTYKEY* const pKey, PROPVARIANT* pPropVar);
    HRESULT (STDMETHODCALLTYPE * SetValue)(ma_IPropertyStore* pThis, const PROPERTYKEY* const pKey, const PROPVARIANT* const pPropVar);
    HRESULT (STDMETHODCALLTYPE * Commit)  (ma_IPropertyStore* pThis);
} ma_IPropertyStoreVtbl;
struct ma_IPropertyStore
{
    ma_IPropertyStoreVtbl* lpVtbl;
};
HRESULT ma_IPropertyStore_QueryInterface(ma_IPropertyStore* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IPropertyStore_AddRef(ma_IPropertyStore* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IPropertyStore_Release(ma_IPropertyStore* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IPropertyStore_GetCount(ma_IPropertyStore* pThis, DWORD* pPropCount)                            { return pThis->lpVtbl->GetCount(pThis, pPropCount); }
HRESULT ma_IPropertyStore_GetAt(ma_IPropertyStore* pThis, DWORD propIndex, PROPERTYKEY* pPropKey)          { return pThis->lpVtbl->GetAt(pThis, propIndex, pPropKey); }
HRESULT ma_IPropertyStore_GetValue(ma_IPropertyStore* pThis, const PROPERTYKEY* const pKey, PROPVARIANT* pPropVar) { return pThis->lpVtbl->GetValue(pThis, pKey, pPropVar); }
HRESULT ma_IPropertyStore_SetValue(ma_IPropertyStore* pThis, const PROPERTYKEY* const pKey, const PROPVARIANT* const pPropVar) { return pThis->lpVtbl->SetValue(pThis, pKey, pPropVar); }
HRESULT ma_IPropertyStore_Commit(ma_IPropertyStore* pThis)                                                 { return pThis->lpVtbl->Commit(pThis); }


/* IAudioClient */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IAudioClient* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IAudioClient* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IAudioClient* pThis);

    /* IAudioClient */
    HRESULT (STDMETHODCALLTYPE * Initialize)       (ma_IAudioClient* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
    HRESULT (STDMETHODCALLTYPE * GetBufferSize)    (ma_IAudioClient* pThis, ma_uint32* pNumBufferFrames);
    HRESULT (STDMETHODCALLTYPE * GetStreamLatency) (ma_IAudioClient* pThis, MA_REFERENCE_TIME* pLatency);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPadding)(ma_IAudioClient* pThis, ma_uint32* pNumPaddingFrames);
    HRESULT (STDMETHODCALLTYPE * IsFormatSupported)(ma_IAudioClient* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch);
    HRESULT (STDMETHODCALLTYPE * GetMixFormat)     (ma_IAudioClient* pThis, WAVEFORMATEX** ppDeviceFormat);
    HRESULT (STDMETHODCALLTYPE * GetDevicePeriod)  (ma_IAudioClient* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod);
    HRESULT (STDMETHODCALLTYPE * Start)            (ma_IAudioClient* pThis);
    HRESULT (STDMETHODCALLTYPE * Stop)             (ma_IAudioClient* pThis);
    HRESULT (STDMETHODCALLTYPE * Reset)            (ma_IAudioClient* pThis);
    HRESULT (STDMETHODCALLTYPE * SetEventHandle)   (ma_IAudioClient* pThis, HANDLE eventHandle);
    HRESULT (STDMETHODCALLTYPE * GetService)       (ma_IAudioClient* pThis, const IID* const riid, void** pp);
} ma_IAudioClientVtbl;
struct ma_IAudioClient
{
    ma_IAudioClientVtbl* lpVtbl;
};
HRESULT ma_IAudioClient_QueryInterface(ma_IAudioClient* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IAudioClient_AddRef(ma_IAudioClient* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IAudioClient_Release(ma_IAudioClient* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IAudioClient_Initialize(ma_IAudioClient* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid) { return pThis->lpVtbl->Initialize(pThis, shareMode, streamFlags, bufferDuration, periodicity, pFormat, pAudioSessionGuid); }
HRESULT ma_IAudioClient_GetBufferSize(ma_IAudioClient* pThis, ma_uint32* pNumBufferFrames)                { return pThis->lpVtbl->GetBufferSize(pThis, pNumBufferFrames); }
HRESULT ma_IAudioClient_GetStreamLatency(ma_IAudioClient* pThis, MA_REFERENCE_TIME* pLatency)             { return pThis->lpVtbl->GetStreamLatency(pThis, pLatency); }
HRESULT ma_IAudioClient_GetCurrentPadding(ma_IAudioClient* pThis, ma_uint32* pNumPaddingFrames)           { return pThis->lpVtbl->GetCurrentPadding(pThis, pNumPaddingFrames); }
HRESULT ma_IAudioClient_IsFormatSupported(ma_IAudioClient* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch) { return pThis->lpVtbl->IsFormatSupported(pThis, shareMode, pFormat, ppClosestMatch); }
HRESULT ma_IAudioClient_GetMixFormat(ma_IAudioClient* pThis, WAVEFORMATEX** ppDeviceFormat)            { return pThis->lpVtbl->GetMixFormat(pThis, ppDeviceFormat); }
HRESULT ma_IAudioClient_GetDevicePeriod(ma_IAudioClient* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod) { return pThis->lpVtbl->GetDevicePeriod(pThis, pDefaultDevicePeriod, pMinimumDevicePeriod); }
HRESULT ma_IAudioClient_Start(ma_IAudioClient* pThis)                                                  { return pThis->lpVtbl->Start(pThis); }
HRESULT ma_IAudioClient_Stop(ma_IAudioClient* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
HRESULT ma_IAudioClient_Reset(ma_IAudioClient* pThis)                                                  { return pThis->lpVtbl->Reset(pThis); }
HRESULT ma_IAudioClient_SetEventHandle(ma_IAudioClient* pThis, HANDLE eventHandle)                     { return pThis->lpVtbl->SetEventHandle(pThis, eventHandle); }
HRESULT ma_IAudioClient_GetService(ma_IAudioClient* pThis, const IID* const riid, void** pp)           { return pThis->lpVtbl->GetService(pThis, riid, pp); }

/* IAudioClient2 */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IAudioClient2* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IAudioClient2* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IAudioClient2* pThis);

    /* IAudioClient */
    HRESULT (STDMETHODCALLTYPE * Initialize)       (ma_IAudioClient2* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
    HRESULT (STDMETHODCALLTYPE * GetBufferSize)    (ma_IAudioClient2* pThis, ma_uint32* pNumBufferFrames);
    HRESULT (STDMETHODCALLTYPE * GetStreamLatency) (ma_IAudioClient2* pThis, MA_REFERENCE_TIME* pLatency);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPadding)(ma_IAudioClient2* pThis, ma_uint32* pNumPaddingFrames);
    HRESULT (STDMETHODCALLTYPE * IsFormatSupported)(ma_IAudioClient2* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch);
    HRESULT (STDMETHODCALLTYPE * GetMixFormat)     (ma_IAudioClient2* pThis, WAVEFORMATEX** ppDeviceFormat);
    HRESULT (STDMETHODCALLTYPE * GetDevicePeriod)  (ma_IAudioClient2* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod);
    HRESULT (STDMETHODCALLTYPE * Start)            (ma_IAudioClient2* pThis);
    HRESULT (STDMETHODCALLTYPE * Stop)             (ma_IAudioClient2* pThis);
    HRESULT (STDMETHODCALLTYPE * Reset)            (ma_IAudioClient2* pThis);
    HRESULT (STDMETHODCALLTYPE * SetEventHandle)   (ma_IAudioClient2* pThis, HANDLE eventHandle);
    HRESULT (STDMETHODCALLTYPE * GetService)       (ma_IAudioClient2* pThis, const IID* const riid, void** pp);

    /* IAudioClient2 */
    HRESULT (STDMETHODCALLTYPE * IsOffloadCapable)   (ma_IAudioClient2* pThis, MA_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable);
    HRESULT (STDMETHODCALLTYPE * SetClientProperties)(ma_IAudioClient2* pThis, const ma_AudioClientProperties* pProperties);
    HRESULT (STDMETHODCALLTYPE * GetBufferSizeLimits)(ma_IAudioClient2* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MA_REFERENCE_TIME* pMinBufferDuration, MA_REFERENCE_TIME* pMaxBufferDuration);
} ma_IAudioClient2Vtbl;
struct ma_IAudioClient2
{
    ma_IAudioClient2Vtbl* lpVtbl;
};
HRESULT ma_IAudioClient2_QueryInterface(ma_IAudioClient2* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IAudioClient2_AddRef(ma_IAudioClient2* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IAudioClient2_Release(ma_IAudioClient2* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IAudioClient2_Initialize(ma_IAudioClient2* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid) { return pThis->lpVtbl->Initialize(pThis, shareMode, streamFlags, bufferDuration, periodicity, pFormat, pAudioSessionGuid); }
HRESULT ma_IAudioClient2_GetBufferSize(ma_IAudioClient2* pThis, ma_uint32* pNumBufferFrames)                { return pThis->lpVtbl->GetBufferSize(pThis, pNumBufferFrames); }
HRESULT ma_IAudioClient2_GetStreamLatency(ma_IAudioClient2* pThis, MA_REFERENCE_TIME* pLatency)             { return pThis->lpVtbl->GetStreamLatency(pThis, pLatency); }
HRESULT ma_IAudioClient2_GetCurrentPadding(ma_IAudioClient2* pThis, ma_uint32* pNumPaddingFrames)           { return pThis->lpVtbl->GetCurrentPadding(pThis, pNumPaddingFrames); }
HRESULT ma_IAudioClient2_IsFormatSupported(ma_IAudioClient2* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch) { return pThis->lpVtbl->IsFormatSupported(pThis, shareMode, pFormat, ppClosestMatch); }
HRESULT ma_IAudioClient2_GetMixFormat(ma_IAudioClient2* pThis, WAVEFORMATEX** ppDeviceFormat)            { return pThis->lpVtbl->GetMixFormat(pThis, ppDeviceFormat); }
HRESULT ma_IAudioClient2_GetDevicePeriod(ma_IAudioClient2* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod) { return pThis->lpVtbl->GetDevicePeriod(pThis, pDefaultDevicePeriod, pMinimumDevicePeriod); }
HRESULT ma_IAudioClient2_Start(ma_IAudioClient2* pThis)                                                  { return pThis->lpVtbl->Start(pThis); }
HRESULT ma_IAudioClient2_Stop(ma_IAudioClient2* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
HRESULT ma_IAudioClient2_Reset(ma_IAudioClient2* pThis)                                                  { return pThis->lpVtbl->Reset(pThis); }
HRESULT ma_IAudioClient2_SetEventHandle(ma_IAudioClient2* pThis, HANDLE eventHandle)                     { return pThis->lpVtbl->SetEventHandle(pThis, eventHandle); }
HRESULT ma_IAudioClient2_GetService(ma_IAudioClient2* pThis, const IID* const riid, void** pp)           { return pThis->lpVtbl->GetService(pThis, riid, pp); }
HRESULT ma_IAudioClient2_IsOffloadCapable(ma_IAudioClient2* pThis, MA_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable) { return pThis->lpVtbl->IsOffloadCapable(pThis, category, pOffloadCapable); }
HRESULT ma_IAudioClient2_SetClientProperties(ma_IAudioClient2* pThis, const ma_AudioClientProperties* pProperties)           { return pThis->lpVtbl->SetClientProperties(pThis, pProperties); }
HRESULT ma_IAudioClient2_GetBufferSizeLimits(ma_IAudioClient2* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MA_REFERENCE_TIME* pMinBufferDuration, MA_REFERENCE_TIME* pMaxBufferDuration) { return pThis->lpVtbl->GetBufferSizeLimits(pThis, pFormat, eventDriven, pMinBufferDuration, pMaxBufferDuration); }


/* IAudioClient3 */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IAudioClient3* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IAudioClient3* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IAudioClient3* pThis);

    /* IAudioClient */
    HRESULT (STDMETHODCALLTYPE * Initialize)       (ma_IAudioClient3* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
    HRESULT (STDMETHODCALLTYPE * GetBufferSize)    (ma_IAudioClient3* pThis, ma_uint32* pNumBufferFrames);
    HRESULT (STDMETHODCALLTYPE * GetStreamLatency) (ma_IAudioClient3* pThis, MA_REFERENCE_TIME* pLatency);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPadding)(ma_IAudioClient3* pThis, ma_uint32* pNumPaddingFrames);
    HRESULT (STDMETHODCALLTYPE * IsFormatSupported)(ma_IAudioClient3* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch);
    HRESULT (STDMETHODCALLTYPE * GetMixFormat)     (ma_IAudioClient3* pThis, WAVEFORMATEX** ppDeviceFormat);
    HRESULT (STDMETHODCALLTYPE * GetDevicePeriod)  (ma_IAudioClient3* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod);
    HRESULT (STDMETHODCALLTYPE * Start)            (ma_IAudioClient3* pThis);
    HRESULT (STDMETHODCALLTYPE * Stop)             (ma_IAudioClient3* pThis);
    HRESULT (STDMETHODCALLTYPE * Reset)            (ma_IAudioClient3* pThis);
    HRESULT (STDMETHODCALLTYPE * SetEventHandle)   (ma_IAudioClient3* pThis, HANDLE eventHandle);
    HRESULT (STDMETHODCALLTYPE * GetService)       (ma_IAudioClient3* pThis, const IID* const riid, void** pp);

    /* IAudioClient2 */
    HRESULT (STDMETHODCALLTYPE * IsOffloadCapable)   (ma_IAudioClient3* pThis, MA_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable);
    HRESULT (STDMETHODCALLTYPE * SetClientProperties)(ma_IAudioClient3* pThis, const ma_AudioClientProperties* pProperties);
    HRESULT (STDMETHODCALLTYPE * GetBufferSizeLimits)(ma_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MA_REFERENCE_TIME* pMinBufferDuration, MA_REFERENCE_TIME* pMaxBufferDuration);

    /* IAudioClient3 */
    HRESULT (STDMETHODCALLTYPE * GetSharedModeEnginePeriod)       (ma_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, UINT32* pDefaultPeriodInFrames, UINT32* pFundamentalPeriodInFrames, UINT32* pMinPeriodInFrames, UINT32* pMaxPeriodInFrames);
    HRESULT (STDMETHODCALLTYPE * GetCurrentSharedModeEnginePeriod)(ma_IAudioClient3* pThis, WAVEFORMATEX** ppFormat, UINT32* pCurrentPeriodInFrames);
    HRESULT (STDMETHODCALLTYPE * InitializeSharedAudioStream)     (ma_IAudioClient3* pThis, DWORD streamFlags, UINT32 periodInFrames, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid);
} ma_IAudioClient3Vtbl;
struct ma_IAudioClient3
{
    ma_IAudioClient3Vtbl* lpVtbl;
};
HRESULT ma_IAudioClient3_QueryInterface(ma_IAudioClient3* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IAudioClient3_AddRef(ma_IAudioClient3* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IAudioClient3_Release(ma_IAudioClient3* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IAudioClient3_Initialize(ma_IAudioClient3* pThis, MA_AUDCLNT_SHAREMODE shareMode, DWORD streamFlags, MA_REFERENCE_TIME bufferDuration, MA_REFERENCE_TIME periodicity, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGuid) { return pThis->lpVtbl->Initialize(pThis, shareMode, streamFlags, bufferDuration, periodicity, pFormat, pAudioSessionGuid); }
HRESULT ma_IAudioClient3_GetBufferSize(ma_IAudioClient3* pThis, ma_uint32* pNumBufferFrames)                { return pThis->lpVtbl->GetBufferSize(pThis, pNumBufferFrames); }
HRESULT ma_IAudioClient3_GetStreamLatency(ma_IAudioClient3* pThis, MA_REFERENCE_TIME* pLatency)             { return pThis->lpVtbl->GetStreamLatency(pThis, pLatency); }
HRESULT ma_IAudioClient3_GetCurrentPadding(ma_IAudioClient3* pThis, ma_uint32* pNumPaddingFrames)           { return pThis->lpVtbl->GetCurrentPadding(pThis, pNumPaddingFrames); }
HRESULT ma_IAudioClient3_IsFormatSupported(ma_IAudioClient3* pThis, MA_AUDCLNT_SHAREMODE shareMode, const WAVEFORMATEX* pFormat, WAVEFORMATEX** ppClosestMatch) { return pThis->lpVtbl->IsFormatSupported(pThis, shareMode, pFormat, ppClosestMatch); }
HRESULT ma_IAudioClient3_GetMixFormat(ma_IAudioClient3* pThis, WAVEFORMATEX** ppDeviceFormat)            { return pThis->lpVtbl->GetMixFormat(pThis, ppDeviceFormat); }
HRESULT ma_IAudioClient3_GetDevicePeriod(ma_IAudioClient3* pThis, MA_REFERENCE_TIME* pDefaultDevicePeriod, MA_REFERENCE_TIME* pMinimumDevicePeriod) { return pThis->lpVtbl->GetDevicePeriod(pThis, pDefaultDevicePeriod, pMinimumDevicePeriod); }
HRESULT ma_IAudioClient3_Start(ma_IAudioClient3* pThis)                                                  { return pThis->lpVtbl->Start(pThis); }
HRESULT ma_IAudioClient3_Stop(ma_IAudioClient3* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
HRESULT ma_IAudioClient3_Reset(ma_IAudioClient3* pThis)                                                  { return pThis->lpVtbl->Reset(pThis); }
HRESULT ma_IAudioClient3_SetEventHandle(ma_IAudioClient3* pThis, HANDLE eventHandle)                     { return pThis->lpVtbl->SetEventHandle(pThis, eventHandle); }
HRESULT ma_IAudioClient3_GetService(ma_IAudioClient3* pThis, const IID* const riid, void** pp)           { return pThis->lpVtbl->GetService(pThis, riid, pp); }
HRESULT ma_IAudioClient3_IsOffloadCapable(ma_IAudioClient3* pThis, MA_AUDIO_STREAM_CATEGORY category, BOOL* pOffloadCapable) { return pThis->lpVtbl->IsOffloadCapable(pThis, category, pOffloadCapable); }
HRESULT ma_IAudioClient3_SetClientProperties(ma_IAudioClient3* pThis, const ma_AudioClientProperties* pProperties)           { return pThis->lpVtbl->SetClientProperties(pThis, pProperties); }
HRESULT ma_IAudioClient3_GetBufferSizeLimits(ma_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, BOOL eventDriven, MA_REFERENCE_TIME* pMinBufferDuration, MA_REFERENCE_TIME* pMaxBufferDuration) { return pThis->lpVtbl->GetBufferSizeLimits(pThis, pFormat, eventDriven, pMinBufferDuration, pMaxBufferDuration); }
HRESULT ma_IAudioClient3_GetSharedModeEnginePeriod(ma_IAudioClient3* pThis, const WAVEFORMATEX* pFormat, UINT32* pDefaultPeriodInFrames, UINT32* pFundamentalPeriodInFrames, UINT32* pMinPeriodInFrames, UINT32* pMaxPeriodInFrames) { return pThis->lpVtbl->GetSharedModeEnginePeriod(pThis, pFormat, pDefaultPeriodInFrames, pFundamentalPeriodInFrames, pMinPeriodInFrames, pMaxPeriodInFrames); }
HRESULT ma_IAudioClient3_GetCurrentSharedModeEnginePeriod(ma_IAudioClient3* pThis, WAVEFORMATEX** ppFormat, UINT32* pCurrentPeriodInFrames) { return pThis->lpVtbl->GetCurrentSharedModeEnginePeriod(pThis, ppFormat, pCurrentPeriodInFrames); }
HRESULT ma_IAudioClient3_InitializeSharedAudioStream(ma_IAudioClient3* pThis, DWORD streamFlags, UINT32 periodInFrames, const WAVEFORMATEX* pFormat, const GUID* pAudioSessionGUID) { return pThis->lpVtbl->InitializeSharedAudioStream(pThis, streamFlags, periodInFrames, pFormat, pAudioSessionGUID); }


/* IAudioRenderClient */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IAudioRenderClient* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IAudioRenderClient* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IAudioRenderClient* pThis);

    /* IAudioRenderClient */
    HRESULT (STDMETHODCALLTYPE * GetBuffer)    (ma_IAudioRenderClient* pThis, ma_uint32 numFramesRequested, BYTE** ppData);
    HRESULT (STDMETHODCALLTYPE * ReleaseBuffer)(ma_IAudioRenderClient* pThis, ma_uint32 numFramesWritten, DWORD dwFlags);
} ma_IAudioRenderClientVtbl;
struct ma_IAudioRenderClient
{
    ma_IAudioRenderClientVtbl* lpVtbl;
};
HRESULT ma_IAudioRenderClient_QueryInterface(ma_IAudioRenderClient* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IAudioRenderClient_AddRef(ma_IAudioRenderClient* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IAudioRenderClient_Release(ma_IAudioRenderClient* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IAudioRenderClient_GetBuffer(ma_IAudioRenderClient* pThis, ma_uint32 numFramesRequested, BYTE** ppData)    { return pThis->lpVtbl->GetBuffer(pThis, numFramesRequested, ppData); }
HRESULT ma_IAudioRenderClient_ReleaseBuffer(ma_IAudioRenderClient* pThis, ma_uint32 numFramesWritten, DWORD dwFlags)  { return pThis->lpVtbl->ReleaseBuffer(pThis, numFramesWritten, dwFlags); }


/* IAudioCaptureClient */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IAudioCaptureClient* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IAudioCaptureClient* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IAudioCaptureClient* pThis);

    /* IAudioRenderClient */
    HRESULT (STDMETHODCALLTYPE * GetBuffer)        (ma_IAudioCaptureClient* pThis, BYTE** ppData, ma_uint32* pNumFramesToRead, DWORD* pFlags, ma_uint64* pDevicePosition, ma_uint64* pQPCPosition);
    HRESULT (STDMETHODCALLTYPE * ReleaseBuffer)    (ma_IAudioCaptureClient* pThis, ma_uint32 numFramesRead);
    HRESULT (STDMETHODCALLTYPE * GetNextPacketSize)(ma_IAudioCaptureClient* pThis, ma_uint32* pNumFramesInNextPacket);
} ma_IAudioCaptureClientVtbl;
struct ma_IAudioCaptureClient
{
    ma_IAudioCaptureClientVtbl* lpVtbl;
};
HRESULT ma_IAudioCaptureClient_QueryInterface(ma_IAudioCaptureClient* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IAudioCaptureClient_AddRef(ma_IAudioCaptureClient* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IAudioCaptureClient_Release(ma_IAudioCaptureClient* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IAudioCaptureClient_GetBuffer(ma_IAudioCaptureClient* pThis, BYTE** ppData, ma_uint32* pNumFramesToRead, DWORD* pFlags, ma_uint64* pDevicePosition, ma_uint64* pQPCPosition) { return pThis->lpVtbl->GetBuffer(pThis, ppData, pNumFramesToRead, pFlags, pDevicePosition, pQPCPosition); }
HRESULT ma_IAudioCaptureClient_ReleaseBuffer(ma_IAudioCaptureClient* pThis, ma_uint32 numFramesRead)                    { return pThis->lpVtbl->ReleaseBuffer(pThis, numFramesRead); }
HRESULT ma_IAudioCaptureClient_GetNextPacketSize(ma_IAudioCaptureClient* pThis, ma_uint32* pNumFramesInNextPacket)      { return pThis->lpVtbl->GetNextPacketSize(pThis, pNumFramesInNextPacket); }

#ifndef MA_WIN32_DESKTOP
#include <mmdeviceapi.h>
typedef struct ma_completion_handler_uwp ma_completion_handler_uwp;

typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_completion_handler_uwp* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_completion_handler_uwp* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_completion_handler_uwp* pThis);

    /* IActivateAudioInterfaceCompletionHandler */
    HRESULT (STDMETHODCALLTYPE * ActivateCompleted)(ma_completion_handler_uwp* pThis, ma_IActivateAudioInterfaceAsyncOperation* pActivateOperation);
} ma_completion_handler_uwp_vtbl;
struct ma_completion_handler_uwp
{
    ma_completion_handler_uwp_vtbl* lpVtbl;
    ma_uint32 counter;
    HANDLE hEvent;
};

HRESULT STDMETHODCALLTYPE ma_completion_handler_uwp_QueryInterface(ma_completion_handler_uwp* pThis, const IID* const riid, void** ppObject)
{
    /*
    We need to "implement" IAgileObject which is just an indicator that's used internally by WASAPI for some multithreading management. To
    "implement" this, we just make sure we return pThis when the IAgileObject is requested.
    */
    if (!ma_is_guid_equal(riid, &MA_IID_IUnknown) && !ma_is_guid_equal(riid, &MA_IID_IActivateAudioInterfaceCompletionHandler) && !ma_is_guid_equal(riid, &MA_IID_IAgileObject)) {
        *ppObject = NULL;
        return E_NOINTERFACE;
    }

    /* Getting here means the IID is IUnknown or IMMNotificationClient. */
    *ppObject = (void*)pThis;
    ((ma_completion_handler_uwp_vtbl*)pThis->lpVtbl)->AddRef(pThis);
    return S_OK;
}

ULONG STDMETHODCALLTYPE ma_completion_handler_uwp_AddRef(ma_completion_handler_uwp* pThis)
{
    return (ULONG)ma_atomic_increment_32(&pThis->counter);
}

ULONG STDMETHODCALLTYPE ma_completion_handler_uwp_Release(ma_completion_handler_uwp* pThis)
{
    ma_uint32 newRefCount = ma_atomic_decrement_32(&pThis->counter);
    if (newRefCount == 0) {
        return 0;   /* We don't free anything here because we never allocate the object on the heap. */
    }

    return (ULONG)newRefCount;
}

HRESULT STDMETHODCALLTYPE ma_completion_handler_uwp_ActivateCompleted(ma_completion_handler_uwp* pThis, ma_IActivateAudioInterfaceAsyncOperation* pActivateOperation)
{
    (void)pActivateOperation;
    SetEvent(pThis->hEvent);
    return S_OK;
}


static ma_completion_handler_uwp_vtbl g_maCompletionHandlerVtblInstance = {
    ma_completion_handler_uwp_QueryInterface,
    ma_completion_handler_uwp_AddRef,
    ma_completion_handler_uwp_Release,
    ma_completion_handler_uwp_ActivateCompleted
};

ma_result ma_completion_handler_uwp_init(ma_completion_handler_uwp* pHandler)
{
    ma_assert(pHandler != NULL);
    ma_zero_object(pHandler);

    pHandler->lpVtbl = &g_maCompletionHandlerVtblInstance;
    pHandler->counter = 1;
    pHandler->hEvent = CreateEventA(NULL, FALSE, FALSE, NULL);
    if (pHandler->hEvent == NULL) {
        return MA_ERROR;
    }

    return MA_SUCCESS;
}

void ma_completion_handler_uwp_uninit(ma_completion_handler_uwp* pHandler)
{
    if (pHandler->hEvent != NULL) {
        CloseHandle(pHandler->hEvent);
    }
}

void ma_completion_handler_uwp_wait(ma_completion_handler_uwp* pHandler)
{
    WaitForSingleObject(pHandler->hEvent, INFINITE);
}
#endif  /* !MA_WIN32_DESKTOP */

/* We need a virtual table for our notification client object that's used for detecting changes to the default device. */
#ifdef MA_WIN32_DESKTOP
HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_QueryInterface(ma_IMMNotificationClient* pThis, const IID* const riid, void** ppObject)
{
    /*
    We care about two interfaces - IUnknown and IMMNotificationClient. If the requested IID is something else
    we just return E_NOINTERFACE. Otherwise we need to increment the reference counter and return S_OK.
    */
    if (!ma_is_guid_equal(riid, &MA_IID_IUnknown) && !ma_is_guid_equal(riid, &MA_IID_IMMNotificationClient)) {
        *ppObject = NULL;
        return E_NOINTERFACE;
    }

    /* Getting here means the IID is IUnknown or IMMNotificationClient. */
    *ppObject = (void*)pThis;
    ((ma_IMMNotificationClientVtbl*)pThis->lpVtbl)->AddRef(pThis);
    return S_OK;
}

ULONG STDMETHODCALLTYPE ma_IMMNotificationClient_AddRef(ma_IMMNotificationClient* pThis)
{
    return (ULONG)ma_atomic_increment_32(&pThis->counter);
}

ULONG STDMETHODCALLTYPE ma_IMMNotificationClient_Release(ma_IMMNotificationClient* pThis)
{
    ma_uint32 newRefCount = ma_atomic_decrement_32(&pThis->counter);
    if (newRefCount == 0) {
        return 0;   /* We don't free anything here because we never allocate the object on the heap. */
    }

    return (ULONG)newRefCount;
}


HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_OnDeviceStateChanged(ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID, DWORD dwNewState)
{
#ifdef MA_DEBUG_OUTPUT
    printf("IMMNotificationClient_OnDeviceStateChanged(pDeviceID=%S, dwNewState=%u)\n", (pDeviceID != NULL) ? pDeviceID : L"(NULL)", (unsigned int)dwNewState);
#endif

    (void)pThis;
    (void)pDeviceID;
    (void)dwNewState;
    return S_OK;
}

HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_OnDeviceAdded(ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID)
{
#ifdef MA_DEBUG_OUTPUT
    printf("IMMNotificationClient_OnDeviceAdded(pDeviceID=%S)\n", (pDeviceID != NULL) ? pDeviceID : L"(NULL)");
#endif

    /* We don't need to worry about this event for our purposes. */
    (void)pThis;
    (void)pDeviceID;
    return S_OK;
}

HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_OnDeviceRemoved(ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID)
{
#ifdef MA_DEBUG_OUTPUT
    printf("IMMNotificationClient_OnDeviceRemoved(pDeviceID=%S)\n", (pDeviceID != NULL) ? pDeviceID : L"(NULL)");
#endif

    /* We don't need to worry about this event for our purposes. */
    (void)pThis;
    (void)pDeviceID;
    return S_OK;
}

HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_OnDefaultDeviceChanged(ma_IMMNotificationClient* pThis, ma_EDataFlow dataFlow, ma_ERole role, LPCWSTR pDefaultDeviceID)
{
#ifdef MA_DEBUG_OUTPUT
    printf("IMMNotificationClient_OnDefaultDeviceChanged(dataFlow=%d, role=%d, pDefaultDeviceID=%S)\n", dataFlow, role, (pDefaultDeviceID != NULL) ? pDefaultDeviceID : L"(NULL)");
#endif

    /* We only ever use the eConsole role in miniaudio. */
    if (role != ma_eConsole) {
        return S_OK;
    }

    /* We only care about devices with the same data flow and role as the current device. */
    if ((pThis->pDevice->type == ma_device_type_playback && dataFlow != ma_eRender) ||
        (pThis->pDevice->type == ma_device_type_capture  && dataFlow != ma_eCapture)) {
        return S_OK;
    }

    /*
    Not currently supporting automatic stream routing in exclusive mode. This is not working correctly on my machine due to
    AUDCLNT_E_DEVICE_IN_USE errors when reinitializing the device. If this is a bug in miniaudio, we can try re-enabling this once
    it's fixed.
    */
    if ((dataFlow == ma_eRender  && pThis->pDevice->playback.shareMode == ma_share_mode_exclusive) ||
        (dataFlow == ma_eCapture && pThis->pDevice->capture.shareMode  == ma_share_mode_exclusive)) {
        return S_OK;
    }

    /*
    We don't change the device here - we change it in the worker thread to keep synchronization simple. To do this I'm just setting a flag to
    indicate that the default device has changed.
    */
    if (dataFlow == ma_eRender) {
        ma_atomic_exchange_32(&pThis->pDevice->wasapi.hasDefaultPlaybackDeviceChanged, MA_TRUE);
    }
    if (dataFlow == ma_eCapture) {
        ma_atomic_exchange_32(&pThis->pDevice->wasapi.hasDefaultCaptureDeviceChanged,  MA_TRUE);
    }

    (void)pDefaultDeviceID;
    return S_OK;
}

HRESULT STDMETHODCALLTYPE ma_IMMNotificationClient_OnPropertyValueChanged(ma_IMMNotificationClient* pThis, LPCWSTR pDeviceID, const PROPERTYKEY key)
{
#ifdef MA_DEBUG_OUTPUT
    printf("IMMNotificationClient_OnPropertyValueChanged(pDeviceID=%S)\n", (pDeviceID != NULL) ? pDeviceID : L"(NULL)");
#endif

    (void)pThis;
    (void)pDeviceID;
    (void)key;
    return S_OK;
}

static ma_IMMNotificationClientVtbl g_maNotificationCientVtbl = {
    ma_IMMNotificationClient_QueryInterface,
    ma_IMMNotificationClient_AddRef,
    ma_IMMNotificationClient_Release,
    ma_IMMNotificationClient_OnDeviceStateChanged,
    ma_IMMNotificationClient_OnDeviceAdded,
    ma_IMMNotificationClient_OnDeviceRemoved,
    ma_IMMNotificationClient_OnDefaultDeviceChanged,
    ma_IMMNotificationClient_OnPropertyValueChanged
};
#endif  /* MA_WIN32_DESKTOP */

#ifdef MA_WIN32_DESKTOP
typedef ma_IMMDevice ma_WASAPIDeviceInterface;
#else
typedef ma_IUnknown ma_WASAPIDeviceInterface;
#endif



ma_bool32 ma_context_is_device_id_equal__wasapi(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return memcmp(pID0->wasapi, pID1->wasapi, sizeof(pID0->wasapi)) == 0;
}

void ma_set_device_info_from_WAVEFORMATEX(const WAVEFORMATEX* pWF, ma_device_info* pInfo)
{
    ma_assert(pWF != NULL);
    ma_assert(pInfo != NULL);

    pInfo->formatCount   = 1;
    pInfo->formats[0]    = ma_format_from_WAVEFORMATEX(pWF);
    pInfo->minChannels   = pWF->nChannels;
    pInfo->maxChannels   = pWF->nChannels;
    pInfo->minSampleRate = pWF->nSamplesPerSec;
    pInfo->maxSampleRate = pWF->nSamplesPerSec;
}

ma_result ma_context_get_device_info_from_IAudioClient__wasapi(ma_context* pContext, /*ma_IMMDevice**/void* pMMDevice, ma_IAudioClient* pAudioClient, ma_share_mode shareMode, ma_device_info* pInfo)
{
    ma_assert(pAudioClient != NULL);
    ma_assert(pInfo != NULL);

    /* We use a different technique to retrieve the device information depending on whether or not we are using shared or exclusive mode. */
    if (shareMode == ma_share_mode_shared) {
        /* Shared Mode. We use GetMixFormat() here. */
        WAVEFORMATEX* pWF = NULL;
        HRESULT hr = ma_IAudioClient_GetMixFormat((ma_IAudioClient*)pAudioClient, (WAVEFORMATEX**)&pWF);
        if (SUCCEEDED(hr)) {
            ma_set_device_info_from_WAVEFORMATEX(pWF, pInfo);
            return MA_SUCCESS;
        } else {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve mix format for device info retrieval.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    } else {
        /* Exlcusive Mode. We repeatedly call IsFormatSupported() here. This is not currently support on UWP. */
#ifdef MA_WIN32_DESKTOP
        /*
        The first thing to do is get the format from PKEY_AudioEngine_DeviceFormat. This should give us a channel count we assume is
        correct which will simplify our searching.
        */
        ma_IPropertyStore *pProperties;
        HRESULT hr = ma_IMMDevice_OpenPropertyStore((ma_IMMDevice*)pMMDevice, STGM_READ, &pProperties);
        if (SUCCEEDED(hr)) {
            PROPVARIANT var;
            ma_PropVariantInit(&var);

            hr = ma_IPropertyStore_GetValue(pProperties, &MA_PKEY_AudioEngine_DeviceFormat, &var);
            if (SUCCEEDED(hr)) {
                WAVEFORMATEX* pWF = (WAVEFORMATEX*)var.blob.pBlobData;
                ma_set_device_info_from_WAVEFORMATEX(pWF, pInfo);

                /*
                In my testing, the format returned by PKEY_AudioEngine_DeviceFormat is suitable for exclusive mode so we check this format
                first. If this fails, fall back to a search.
                */
                hr = ma_IAudioClient_IsFormatSupported((ma_IAudioClient*)pAudioClient, MA_AUDCLNT_SHAREMODE_EXCLUSIVE, pWF, NULL);
                ma_PropVariantClear(pContext, &var);

                if (FAILED(hr)) {
                    /*
                    The format returned by PKEY_AudioEngine_DeviceFormat is not supported, so fall back to a search. We assume the channel
                    count returned by MA_PKEY_AudioEngine_DeviceFormat is valid and correct. For simplicity we're only returning one format.
                    */
                    ma_uint32 channels = pInfo->minChannels;
                    ma_format formatsToSearch[] = {
                        ma_format_s16,
                        ma_format_s24,
                        /*ma_format_s24_32,*/
                        ma_format_f32,
                        ma_format_s32,
                        ma_format_u8
                    };
                    ma_channel defaultChannelMap[MA_MAX_CHANNELS];
                    WAVEFORMATEXTENSIBLE wf;
                    ma_bool32 found;
                    ma_uint32 iFormat;

                    ma_get_standard_channel_map(ma_standard_channel_map_microsoft, channels, defaultChannelMap);

                    ma_zero_object(&wf);
                    wf.Format.cbSize     = sizeof(wf);
                    wf.Format.wFormatTag = WAVE_FORMAT_EXTENSIBLE;
                    wf.Format.nChannels  = (WORD)channels;
                    wf.dwChannelMask     = ma_channel_map_to_channel_mask__win32(defaultChannelMap, channels);

                    found = MA_FALSE;
                    for (iFormat = 0; iFormat < ma_countof(formatsToSearch); ++iFormat) {
                        ma_format format = formatsToSearch[iFormat];
                        ma_uint32 iSampleRate;

                        wf.Format.wBitsPerSample       = (WORD)ma_get_bytes_per_sample(format)*8;
                        wf.Format.nBlockAlign          = (wf.Format.nChannels * wf.Format.wBitsPerSample) / 8;
                        wf.Format.nAvgBytesPerSec      = wf.Format.nBlockAlign * wf.Format.nSamplesPerSec;
                        wf.Samples.wValidBitsPerSample = /*(format == ma_format_s24_32) ? 24 :*/ wf.Format.wBitsPerSample;
                        if (format == ma_format_f32) {
                            wf.SubFormat = MA_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT;
                        } else {
                            wf.SubFormat = MA_GUID_KSDATAFORMAT_SUBTYPE_PCM;
                        }

                        for (iSampleRate = 0; iSampleRate < ma_countof(g_maStandardSampleRatePriorities); ++iSampleRate) {
                            wf.Format.nSamplesPerSec = g_maStandardSampleRatePriorities[iSampleRate];

                            hr = ma_IAudioClient_IsFormatSupported((ma_IAudioClient*)pAudioClient, MA_AUDCLNT_SHAREMODE_EXCLUSIVE, (WAVEFORMATEX*)&wf, NULL);
                            if (SUCCEEDED(hr)) {
                                ma_set_device_info_from_WAVEFORMATEX((WAVEFORMATEX*)&wf, pInfo);
                                found = MA_TRUE;
                                break;
                            }
                        }

                        if (found) {
                            break;
                        }
                    }

                    if (!found) {
                        ma_IPropertyStore_Release(pProperties);
                        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to find suitable device format for device info retrieval.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
                    }
                }
            } else {
                ma_IPropertyStore_Release(pProperties);
                return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve device format for device info retrieval.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
            }
        } else {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to open property store for device info retrieval.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        return MA_SUCCESS;
#else
        /* Exclusive mode not fully supported in UWP right now. */
        return MA_ERROR;
#endif
    }
}

#ifdef MA_WIN32_DESKTOP
ma_result ma_context_get_MMDevice__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_IMMDevice** ppMMDevice)
{
    ma_IMMDeviceEnumerator* pDeviceEnumerator;
    HRESULT hr;

    ma_assert(pContext != NULL);
    ma_assert(ppMMDevice != NULL);

    hr = ma_CoCreateInstance(pContext, MA_CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, MA_IID_IMMDeviceEnumerator, (void**)&pDeviceEnumerator);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create IMMDeviceEnumerator.", MA_FAILED_TO_INIT_BACKEND);
    }

    if (pDeviceID == NULL) {
        hr = ma_IMMDeviceEnumerator_GetDefaultAudioEndpoint(pDeviceEnumerator, (deviceType == ma_device_type_playback) ? ma_eRender : ma_eCapture, ma_eConsole, ppMMDevice);
    } else {
        hr = ma_IMMDeviceEnumerator_GetDevice(pDeviceEnumerator, pDeviceID->wasapi, ppMMDevice);
    }

    ma_IMMDeviceEnumerator_Release(pDeviceEnumerator);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve IMMDevice.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    return MA_SUCCESS;
}

ma_result ma_context_get_device_info_from_MMDevice__wasapi(ma_context* pContext, ma_IMMDevice* pMMDevice, ma_share_mode shareMode, ma_bool32 onlySimpleInfo, ma_device_info* pInfo)
{
    LPWSTR id;
    HRESULT hr;

    ma_assert(pContext != NULL);
    ma_assert(pMMDevice != NULL);
    ma_assert(pInfo != NULL);

    /* ID. */
    hr = ma_IMMDevice_GetId(pMMDevice, &id);
    if (SUCCEEDED(hr)) {
        size_t idlen = wcslen(id);
        if (idlen+1 > ma_countof(pInfo->id.wasapi)) {
            ma_CoTaskMemFree(pContext, id);
            ma_assert(MA_FALSE);  /* NOTE: If this is triggered, please report it. It means the format of the ID must haved change and is too long to fit in our fixed sized buffer. */
            return MA_ERROR;
        }

        ma_copy_memory(pInfo->id.wasapi, id, idlen * sizeof(wchar_t));
        pInfo->id.wasapi[idlen] = '\0';

        ma_CoTaskMemFree(pContext, id);
    }

    {
        ma_IPropertyStore *pProperties;
        hr = ma_IMMDevice_OpenPropertyStore(pMMDevice, STGM_READ, &pProperties);
        if (SUCCEEDED(hr)) {
            PROPVARIANT var;

            /* Description / Friendly Name */
            ma_PropVariantInit(&var);
            hr = ma_IPropertyStore_GetValue(pProperties, &MA_PKEY_Device_FriendlyName, &var);
            if (SUCCEEDED(hr)) {
                WideCharToMultiByte(CP_UTF8, 0, var.pwszVal, -1, pInfo->name, sizeof(pInfo->name), 0, FALSE);
                ma_PropVariantClear(pContext, &var);
            }

            ma_IPropertyStore_Release(pProperties);
        }
    }

    /* Format */
    if (!onlySimpleInfo) {
        ma_IAudioClient* pAudioClient;
        hr = ma_IMMDevice_Activate(pMMDevice, &MA_IID_IAudioClient, CLSCTX_ALL, NULL, (void**)&pAudioClient);
        if (SUCCEEDED(hr)) {
            ma_result result = ma_context_get_device_info_from_IAudioClient__wasapi(pContext, pMMDevice, pAudioClient, shareMode, pInfo);
            
            ma_IAudioClient_Release(pAudioClient);
            return result;
        } else {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to activate audio client for device info retrieval.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    }

    return MA_SUCCESS;
}

ma_result ma_context_enumerate_device_collection__wasapi(ma_context* pContext, ma_IMMDeviceCollection* pDeviceCollection, ma_device_type deviceType, ma_enum_devices_callback_proc callback, void* pUserData)
{
    UINT deviceCount;
    HRESULT hr;
    ma_uint32 iDevice;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);

    hr = ma_IMMDeviceCollection_GetCount(pDeviceCollection, &deviceCount);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to get playback device count.", MA_NO_DEVICE);
    }

    for (iDevice = 0; iDevice < deviceCount; ++iDevice) {
        ma_device_info deviceInfo;
        ma_IMMDevice* pMMDevice;
        
        ma_zero_object(&deviceInfo);

        hr = ma_IMMDeviceCollection_Item(pDeviceCollection, iDevice, &pMMDevice);
        if (SUCCEEDED(hr)) {
            ma_result result = ma_context_get_device_info_from_MMDevice__wasapi(pContext, pMMDevice, ma_share_mode_shared, MA_TRUE, &deviceInfo);   /* MA_TRUE = onlySimpleInfo. */

            ma_IMMDevice_Release(pMMDevice);
            if (result == MA_SUCCESS) {
                ma_bool32 cbResult = callback(pContext, deviceType, &deviceInfo, pUserData);
                if (cbResult == MA_FALSE) {
                    break;
                }
            }
        }
    }

    return MA_SUCCESS;
}
#endif

#ifdef MA_WIN32_DESKTOP
ma_result ma_context_get_IAudioClient_Desktop__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_IAudioClient** ppAudioClient, ma_IMMDevice** ppMMDevice)
{
    ma_result result;
    HRESULT hr;

    ma_assert(pContext != NULL);
    ma_assert(ppAudioClient != NULL);
    ma_assert(ppMMDevice != NULL);

    result = ma_context_get_MMDevice__wasapi(pContext, deviceType, pDeviceID, ppMMDevice);
    if (result != MA_SUCCESS) {
        return result;
    }

    hr = ma_IMMDevice_Activate(*ppMMDevice, &MA_IID_IAudioClient, CLSCTX_ALL, NULL, (void**)ppAudioClient);
    if (FAILED(hr)) {
        return MA_FAILED_TO_OPEN_BACKEND_DEVICE;
    }

    return MA_SUCCESS;
}
#else
ma_result ma_context_get_IAudioClient_UWP__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_IAudioClient** ppAudioClient, ma_IUnknown** ppActivatedInterface)
{
    ma_IActivateAudioInterfaceAsyncOperation *pAsyncOp = NULL;
    ma_completion_handler_uwp completionHandler;
    IID iid;
    LPOLESTR iidStr;
    HRESULT hr;
    ma_result result;
    HRESULT activateResult;
    ma_IUnknown* pActivatedInterface;

    ma_assert(pContext != NULL);
    ma_assert(ppAudioClient != NULL);

    if (pDeviceID != NULL) {
        ma_copy_memory(&iid, pDeviceID->wasapi, sizeof(iid));
    } else {
        if (deviceType == ma_device_type_playback) {
            iid = MA_IID_DEVINTERFACE_AUDIO_RENDER;
        } else {
            iid = MA_IID_DEVINTERFACE_AUDIO_CAPTURE;
        }
    }

#if defined(__cplusplus)
    hr = StringFromIID(iid, &iidStr);
#else
    hr = StringFromIID(&iid, &iidStr);
#endif
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to convert device IID to string for ActivateAudioInterfaceAsync(). Out of memory.", MA_OUT_OF_MEMORY);
    }

    result = ma_completion_handler_uwp_init(&completionHandler);
    if (result != MA_SUCCESS) {
        ma_CoTaskMemFree(pContext, iidStr);
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create event for waiting for ActivateAudioInterfaceAsync().", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

#if defined(__cplusplus)
    hr = ActivateAudioInterfaceAsync(iidStr, MA_IID_IAudioClient, NULL, (IActivateAudioInterfaceCompletionHandler*)&completionHandler, (IActivateAudioInterfaceAsyncOperation**)&pAsyncOp);
#else
    hr = ActivateAudioInterfaceAsync(iidStr, &MA_IID_IAudioClient, NULL, (IActivateAudioInterfaceCompletionHandler*)&completionHandler, (IActivateAudioInterfaceAsyncOperation**)&pAsyncOp);
#endif
    if (FAILED(hr)) {
        ma_completion_handler_uwp_uninit(&completionHandler);
        ma_CoTaskMemFree(pContext, iidStr);
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] ActivateAudioInterfaceAsync() failed.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    ma_CoTaskMemFree(pContext, iidStr);

    /* Wait for the async operation for finish. */
    ma_completion_handler_uwp_wait(&completionHandler);
    ma_completion_handler_uwp_uninit(&completionHandler);

    hr = ma_IActivateAudioInterfaceAsyncOperation_GetActivateResult(pAsyncOp, &activateResult, &pActivatedInterface);
    ma_IActivateAudioInterfaceAsyncOperation_Release(pAsyncOp);

    if (FAILED(hr) || FAILED(activateResult)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to activate device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    /* Here is where we grab the IAudioClient interface. */
    hr = ma_IUnknown_QueryInterface(pActivatedInterface, &MA_IID_IAudioClient, (void**)ppAudioClient);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to query IAudioClient interface.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    if (ppActivatedInterface) {
        *ppActivatedInterface = pActivatedInterface;
    } else {
        ma_IUnknown_Release(pActivatedInterface);
    }

    return MA_SUCCESS;
}
#endif

ma_result ma_context_get_IAudioClient__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_IAudioClient** ppAudioClient, ma_WASAPIDeviceInterface** ppDeviceInterface)
{
#ifdef MA_WIN32_DESKTOP
    return ma_context_get_IAudioClient_Desktop__wasapi(pContext, deviceType, pDeviceID, ppAudioClient, ppDeviceInterface);
#else
    return ma_context_get_IAudioClient_UWP__wasapi(pContext, deviceType, pDeviceID, ppAudioClient, ppDeviceInterface);
#endif
}


ma_result ma_context_enumerate_devices__wasapi(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    /* Different enumeration for desktop and UWP. */
#ifdef MA_WIN32_DESKTOP
    /* Desktop */
    HRESULT hr;
    ma_IMMDeviceEnumerator* pDeviceEnumerator;
    ma_IMMDeviceCollection* pDeviceCollection;

    hr = ma_CoCreateInstance(pContext, MA_CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, MA_IID_IMMDeviceEnumerator, (void**)&pDeviceEnumerator);
    if (FAILED(hr)) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create device enumerator.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    /* Playback. */
    hr = ma_IMMDeviceEnumerator_EnumAudioEndpoints(pDeviceEnumerator, ma_eRender, MA_MM_DEVICE_STATE_ACTIVE, &pDeviceCollection);
    if (SUCCEEDED(hr)) {
        ma_context_enumerate_device_collection__wasapi(pContext, pDeviceCollection, ma_device_type_playback, callback, pUserData);
        ma_IMMDeviceCollection_Release(pDeviceCollection);
    }

    /* Capture. */
    hr = ma_IMMDeviceEnumerator_EnumAudioEndpoints(pDeviceEnumerator, ma_eCapture, MA_MM_DEVICE_STATE_ACTIVE, &pDeviceCollection);
    if (SUCCEEDED(hr)) {
        ma_context_enumerate_device_collection__wasapi(pContext, pDeviceCollection, ma_device_type_capture, callback, pUserData);
        ma_IMMDeviceCollection_Release(pDeviceCollection);
    }

    ma_IMMDeviceEnumerator_Release(pDeviceEnumerator);
#else
    /*
    UWP
    
    The MMDevice API is only supported on desktop applications. For now, while I'm still figuring out how to properly enumerate
    over devices without using MMDevice, I'm restricting devices to defaults.
    
    Hint: DeviceInformation::FindAllAsync() with DeviceClass.AudioCapture/AudioRender. https://blogs.windows.com/buildingapps/2014/05/15/real-time-audio-in-windows-store-and-windows-phone-apps/
    */
    if (callback) {
        ma_bool32 cbResult = MA_TRUE;

        /* Playback. */
        if (cbResult) {
            ma_device_info deviceInfo;
            ma_zero_object(&deviceInfo);
            ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
            cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
        }

        /* Capture. */
        if (cbResult) {
            ma_device_info deviceInfo;
            ma_zero_object(&deviceInfo);
            ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
            cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
        }
    }
#endif

    return MA_SUCCESS;
}

ma_result ma_context_get_device_info__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
#ifdef MA_WIN32_DESKTOP
    ma_IMMDevice* pMMDevice = NULL;
    ma_result result;
    
    result = ma_context_get_MMDevice__wasapi(pContext, deviceType, pDeviceID, &pMMDevice);
    if (result != MA_SUCCESS) {
        return result;
    }

    result = ma_context_get_device_info_from_MMDevice__wasapi(pContext, pMMDevice, shareMode, MA_FALSE, pDeviceInfo);   /* MA_FALSE = !onlySimpleInfo. */

    ma_IMMDevice_Release(pMMDevice);
    return result;
#else
    ma_IAudioClient* pAudioClient;
    ma_result result;

    /* UWP currently only uses default devices. */
    if (deviceType == ma_device_type_playback) {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    } else {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    }

    /* Not currently supporting exclusive mode on UWP. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_ERROR;
    }

    result = ma_context_get_IAudioClient_UWP__wasapi(pContext, deviceType, pDeviceID, &pAudioClient, NULL);
    if (result != MA_SUCCESS) {
        return result;
    }

    result = ma_context_get_device_info_from_IAudioClient__wasapi(pContext, NULL, pAudioClient, shareMode, pDeviceInfo);

    ma_IAudioClient_Release(pAudioClient);
    return result;
#endif
}

void ma_device_uninit__wasapi(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

#ifdef MA_WIN32_DESKTOP
    if (pDevice->wasapi.pDeviceEnumerator) {
        ((ma_IMMDeviceEnumerator*)pDevice->wasapi.pDeviceEnumerator)->lpVtbl->UnregisterEndpointNotificationCallback((ma_IMMDeviceEnumerator*)pDevice->wasapi.pDeviceEnumerator, &pDevice->wasapi.notificationClient);
        ma_IMMDeviceEnumerator_Release((ma_IMMDeviceEnumerator*)pDevice->wasapi.pDeviceEnumerator);
    }
#endif

    if (pDevice->wasapi.pRenderClient) {
        ma_IAudioRenderClient_Release((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient);
    }
    if (pDevice->wasapi.pCaptureClient) {
        ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
    }

    if (pDevice->wasapi.pAudioClientPlayback) {
        ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
    }
    if (pDevice->wasapi.pAudioClientCapture) {
        ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
    }

    if (pDevice->wasapi.hEventPlayback) {
        CloseHandle(pDevice->wasapi.hEventPlayback);
    }
    if (pDevice->wasapi.hEventCapture) {
        CloseHandle(pDevice->wasapi.hEventCapture);
    }
}


typedef struct
{
    /* Input. */
    ma_format formatIn;
    ma_uint32 channelsIn;
    ma_uint32 sampleRateIn;
    ma_channel channelMapIn[MA_MAX_CHANNELS];
    ma_uint32 bufferSizeInFramesIn;
    ma_uint32 bufferSizeInMillisecondsIn;
    ma_uint32 periodsIn;
    ma_bool32 usingDefaultFormat;
    ma_bool32 usingDefaultChannels;
    ma_bool32 usingDefaultSampleRate;
    ma_bool32 usingDefaultChannelMap;
    ma_share_mode shareMode;

    /* Output. */
    ma_IAudioClient* pAudioClient;
    ma_IAudioRenderClient* pRenderClient;
    ma_IAudioCaptureClient* pCaptureClient;
    ma_format formatOut;
    ma_uint32 channelsOut;
    ma_uint32 sampleRateOut;
    ma_channel channelMapOut[MA_MAX_CHANNELS];
    ma_uint32 bufferSizeInFramesOut;
    ma_uint32 periodSizeInFramesOut;
    ma_uint32 periodsOut;
    char deviceName[256];
} ma_device_init_internal_data__wasapi;

ma_result ma_device_init_internal__wasapi(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_device_init_internal_data__wasapi* pData)
{
    HRESULT hr;
    ma_result result = MA_SUCCESS;
    const char* errorMsg = "";
    MA_AUDCLNT_SHAREMODE shareMode = MA_AUDCLNT_SHAREMODE_SHARED;
    MA_REFERENCE_TIME bufferDurationInMicroseconds;
    ma_bool32 wasInitializedUsingIAudioClient3 = MA_FALSE;
    WAVEFORMATEXTENSIBLE wf;
    ma_WASAPIDeviceInterface* pDeviceInterface = NULL;
    ma_IAudioClient2* pAudioClient2;

    ma_assert(pContext != NULL);
    ma_assert(pData != NULL);

    /* This function is only used to initialize one device type: either playback or capture. Never full-duplex. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    pData->pAudioClient = NULL;
    pData->pRenderClient = NULL;
    pData->pCaptureClient = NULL;

    result = ma_context_get_IAudioClient__wasapi(pContext, deviceType, pDeviceID, &pData->pAudioClient, &pDeviceInterface);
    if (result != MA_SUCCESS) {
        goto done;
    }


    /* Try enabling hardware offloading. */
    hr = ma_IAudioClient_QueryInterface(pData->pAudioClient, &MA_IID_IAudioClient2, (void**)&pAudioClient2);
    if (SUCCEEDED(hr)) {
        BOOL isHardwareOffloadingSupported = 0;
        hr = ma_IAudioClient2_IsOffloadCapable(pAudioClient2, MA_AudioCategory_Other, &isHardwareOffloadingSupported);
        if (SUCCEEDED(hr) && isHardwareOffloadingSupported) {
            ma_AudioClientProperties clientProperties;
            ma_zero_object(&clientProperties);
            clientProperties.cbSize = sizeof(clientProperties);
            clientProperties.bIsOffload = 1;
            clientProperties.eCategory = MA_AudioCategory_Other;
            ma_IAudioClient2_SetClientProperties(pAudioClient2, &clientProperties);
        }
    }


    /* Here is where we try to determine the best format to use with the device. If the client if wanting exclusive mode, first try finding the best format for that. If this fails, fall back to shared mode. */
    result = MA_FORMAT_NOT_SUPPORTED;
    if (pData->shareMode == ma_share_mode_exclusive) {
    #ifdef MA_WIN32_DESKTOP
        /* In exclusive mode on desktop we always use the backend's native format. */
        ma_IPropertyStore* pStore = NULL;
        hr = ma_IMMDevice_OpenPropertyStore(pDeviceInterface, STGM_READ, &pStore);
        if (SUCCEEDED(hr)) {
            PROPVARIANT prop;
            ma_PropVariantInit(&prop);
            hr = ma_IPropertyStore_GetValue(pStore, &MA_PKEY_AudioEngine_DeviceFormat, &prop);
            if (SUCCEEDED(hr)) {
                WAVEFORMATEX* pActualFormat = (WAVEFORMATEX*)prop.blob.pBlobData;
                hr = ma_IAudioClient_IsFormatSupported((ma_IAudioClient*)pData->pAudioClient, MA_AUDCLNT_SHAREMODE_EXCLUSIVE, pActualFormat, NULL);
                if (SUCCEEDED(hr)) {
                    ma_copy_memory(&wf, pActualFormat, sizeof(WAVEFORMATEXTENSIBLE));
                }

                ma_PropVariantClear(pContext, &prop);
            }

            ma_IPropertyStore_Release(pStore);
        }
    #else
        /*
        I do not know how to query the device's native format on UWP so for now I'm just disabling support for
        exclusive mode. The alternative is to enumerate over different formats and check IsFormatSupported()
        until you find one that works.
        
        TODO: Add support for exclusive mode to UWP.
        */
        hr = S_FALSE;
    #endif

        if (hr == S_OK) {
            shareMode = MA_AUDCLNT_SHAREMODE_EXCLUSIVE;
            result = MA_SUCCESS;
        } else {
            result = MA_SHARE_MODE_NOT_SUPPORTED;
        }
    } else {
        /* In shared mode we are always using the format reported by the operating system. */
        WAVEFORMATEXTENSIBLE* pNativeFormat = NULL;
        hr = ma_IAudioClient_GetMixFormat((ma_IAudioClient*)pData->pAudioClient, (WAVEFORMATEX**)&pNativeFormat);
        if (hr != S_OK) {
            result = MA_FORMAT_NOT_SUPPORTED;
        } else {
            ma_copy_memory(&wf, pNativeFormat, sizeof(wf));
            result = MA_SUCCESS;
        }

        ma_CoTaskMemFree(pContext, pNativeFormat);

        shareMode = MA_AUDCLNT_SHAREMODE_SHARED;
    }

    /* Return an error if we still haven't found a format. */
    if (result != MA_SUCCESS) {
        errorMsg = "[WASAPI] Failed to find best device mix format.";
        goto done;
    }

    pData->formatOut = ma_format_from_WAVEFORMATEX((WAVEFORMATEX*)&wf);
    pData->channelsOut = wf.Format.nChannels;
    pData->sampleRateOut = wf.Format.nSamplesPerSec;

    /* Get the internal channel map based on the channel mask. */
    ma_channel_mask_to_channel_map__win32(wf.dwChannelMask, pData->channelsOut, pData->channelMapOut);

    /* If we're using a default buffer size we need to calculate it based on the efficiency of the system. */
    pData->periodsOut = pData->periodsIn;
    pData->bufferSizeInFramesOut = pData->bufferSizeInFramesIn;
    if (pData->bufferSizeInFramesOut == 0) {
        pData->bufferSizeInFramesOut = ma_calculate_buffer_size_in_frames_from_milliseconds(pData->bufferSizeInMillisecondsIn, pData->sampleRateOut);
    }

    bufferDurationInMicroseconds = ((ma_uint64)pData->bufferSizeInFramesOut * 1000 * 1000) / pData->sampleRateOut;


    /* Slightly different initialization for shared and exclusive modes. We try exclusive mode first, and if it fails, fall back to shared mode. */
    if (shareMode == MA_AUDCLNT_SHAREMODE_EXCLUSIVE) {
        MA_REFERENCE_TIME bufferDuration = (bufferDurationInMicroseconds / pData->periodsOut) * 10;

        /*
        If the periodicy is too small, Initialize() will fail with AUDCLNT_E_INVALID_DEVICE_PERIOD. In this case we should just keep increasing
        it and trying it again.
        */
        hr = E_FAIL;
        for (;;) {
            hr = ma_IAudioClient_Initialize((ma_IAudioClient*)pData->pAudioClient, shareMode, MA_AUDCLNT_STREAMFLAGS_EVENTCALLBACK, bufferDuration, bufferDuration, (WAVEFORMATEX*)&wf, NULL);
            if (hr == MA_AUDCLNT_E_INVALID_DEVICE_PERIOD) {
                if (bufferDuration > 500*10000) {
                    break;
                } else {
                    if (bufferDuration == 0) {  /* <-- Just a sanity check to prevent an infinit loop. Should never happen, but it makes me feel better. */
                        break;
                    }

                    bufferDuration = bufferDuration * 2;
                    continue;
                }
            } else {
                break;
            }
        }
        
        if (hr == MA_AUDCLNT_E_BUFFER_SIZE_NOT_ALIGNED) {
            UINT bufferSizeInFrames;
            hr = ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pData->pAudioClient, &bufferSizeInFrames);
            if (SUCCEEDED(hr)) {
                bufferDuration = (MA_REFERENCE_TIME)((10000.0 * 1000 / wf.Format.nSamplesPerSec * bufferSizeInFrames) + 0.5);

                /* Unfortunately we need to release and re-acquire the audio client according to MSDN. Seems silly - why not just call IAudioClient_Initialize() again?! */
                ma_IAudioClient_Release((ma_IAudioClient*)pData->pAudioClient);

            #ifdef MA_WIN32_DESKTOP
                hr = ma_IMMDevice_Activate(pDeviceInterface, &MA_IID_IAudioClient, CLSCTX_ALL, NULL, (void**)&pData->pAudioClient);
            #else
                hr = ma_IUnknown_QueryInterface(pDeviceInterface, &MA_IID_IAudioClient, (void**)&pData->pAudioClient);
            #endif

                if (SUCCEEDED(hr)) {
                    hr = ma_IAudioClient_Initialize((ma_IAudioClient*)pData->pAudioClient, shareMode, MA_AUDCLNT_STREAMFLAGS_EVENTCALLBACK, bufferDuration, bufferDuration, (WAVEFORMATEX*)&wf, NULL);
                }
            }
        }

        if (FAILED(hr)) {
            /* Failed to initialize in exclusive mode. Don't fall back to shared mode - instead tell the client about it. They can reinitialize in shared mode if they want. */
            if (hr == E_ACCESSDENIED) {
                errorMsg = "[WASAPI] Failed to initialize device in exclusive mode. Access denied.", result = MA_ACCESS_DENIED;
            } else if (hr == MA_AUDCLNT_E_DEVICE_IN_USE) {
                errorMsg = "[WASAPI] Failed to initialize device in exclusive mode. Device in use.", result = MA_DEVICE_BUSY;
            } else {
                errorMsg = "[WASAPI] Failed to initialize device in exclusive mode."; result = MA_SHARE_MODE_NOT_SUPPORTED;
            }
            goto done;
        }
    }

    if (shareMode == MA_AUDCLNT_SHAREMODE_SHARED) {
        /* Low latency shared mode via IAudioClient3. */
#ifndef MA_WASAPI_NO_LOW_LATENCY_SHARED_MODE
        ma_IAudioClient3* pAudioClient3 = NULL;
        hr = ma_IAudioClient_QueryInterface(pData->pAudioClient, &MA_IID_IAudioClient3, (void**)&pAudioClient3);
        if (SUCCEEDED(hr)) {
            UINT32 defaultPeriodInFrames;
            UINT32 fundamentalPeriodInFrames;
            UINT32 minPeriodInFrames;
            UINT32 maxPeriodInFrames;
            hr = ma_IAudioClient3_GetSharedModeEnginePeriod(pAudioClient3, (WAVEFORMATEX*)&wf, &defaultPeriodInFrames, &fundamentalPeriodInFrames, &minPeriodInFrames, &maxPeriodInFrames);
            if (SUCCEEDED(hr)) {
                UINT32 desiredPeriodInFrames = pData->bufferSizeInFramesOut / pData->periodsOut;
                UINT32 actualPeriodInFrames  = desiredPeriodInFrames;

                /* Make sure the period size is a multiple of fundamentalPeriodInFrames. */
                actualPeriodInFrames = actualPeriodInFrames / fundamentalPeriodInFrames;
                actualPeriodInFrames = actualPeriodInFrames * fundamentalPeriodInFrames;

                /* The period needs to be clamped between minPeriodInFrames and maxPeriodInFrames. */
                actualPeriodInFrames = ma_clamp(actualPeriodInFrames, minPeriodInFrames, maxPeriodInFrames);

                /* If the client requested a largish buffer than we don't actually want to use low latency shared mode because it forces small buffers. */
                if (actualPeriodInFrames >= desiredPeriodInFrames) {
                    hr = ma_IAudioClient3_InitializeSharedAudioStream(pAudioClient3, MA_AUDCLNT_STREAMFLAGS_EVENTCALLBACK, actualPeriodInFrames, (WAVEFORMATEX*)&wf, NULL);
                    if (SUCCEEDED(hr)) {
                        wasInitializedUsingIAudioClient3 = MA_TRUE;
                        pData->periodSizeInFramesOut = actualPeriodInFrames;
                        pData->bufferSizeInFramesOut = actualPeriodInFrames * pData->periodsOut;
                    }
                }
            }

            ma_IAudioClient3_Release(pAudioClient3);
            pAudioClient3 = NULL;
        }
#endif

        /* If we don't have an IAudioClient3 then we need to use the normal initialization routine. */
        if (!wasInitializedUsingIAudioClient3) {
            MA_REFERENCE_TIME bufferDuration = bufferDurationInMicroseconds*10;
            hr = ma_IAudioClient_Initialize((ma_IAudioClient*)pData->pAudioClient, shareMode, MA_AUDCLNT_STREAMFLAGS_EVENTCALLBACK, bufferDuration, 0, (WAVEFORMATEX*)&wf, NULL);
            if (FAILED(hr)) {
                if (hr == E_ACCESSDENIED) {
                    errorMsg = "[WASAPI] Failed to initialize device. Access denied.", result = MA_ACCESS_DENIED;
                } else if (hr == MA_AUDCLNT_E_DEVICE_IN_USE) {
                    errorMsg = "[WASAPI] Failed to initialize device. Device in use.", result = MA_DEVICE_BUSY;
                } else {
                    errorMsg = "[WASAPI] Failed to initialize device.", result = MA_FAILED_TO_OPEN_BACKEND_DEVICE;
                }

                goto done;
            }
        }
    }

    if (!wasInitializedUsingIAudioClient3) {
        hr = ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pData->pAudioClient, &pData->bufferSizeInFramesOut);
        if (FAILED(hr)) {
            errorMsg = "[WASAPI] Failed to get audio client's actual buffer size.", result = MA_FAILED_TO_OPEN_BACKEND_DEVICE;
            goto done;
        }

        pData->periodSizeInFramesOut = pData->bufferSizeInFramesOut / pData->periodsOut;
    }

    if (deviceType == ma_device_type_playback) {
        hr = ma_IAudioClient_GetService((ma_IAudioClient*)pData->pAudioClient, &MA_IID_IAudioRenderClient, (void**)&pData->pRenderClient);
    } else {
        hr = ma_IAudioClient_GetService((ma_IAudioClient*)pData->pAudioClient, &MA_IID_IAudioCaptureClient, (void**)&pData->pCaptureClient);
    }

    if (FAILED(hr)) {
        errorMsg = "[WASAPI] Failed to get audio client service.", result = MA_API_NOT_FOUND;
        goto done;
    }


    /* Grab the name of the device. */
#ifdef MA_WIN32_DESKTOP
    {
        ma_IPropertyStore *pProperties;
        hr = ma_IMMDevice_OpenPropertyStore(pDeviceInterface, STGM_READ, &pProperties);
        if (SUCCEEDED(hr)) {
            PROPVARIANT varName;
            ma_PropVariantInit(&varName);
            hr = ma_IPropertyStore_GetValue(pProperties, &MA_PKEY_Device_FriendlyName, &varName);
            if (SUCCEEDED(hr)) {
                WideCharToMultiByte(CP_UTF8, 0, varName.pwszVal, -1, pData->deviceName, sizeof(pData->deviceName), 0, FALSE);
                ma_PropVariantClear(pContext, &varName);
            }

            ma_IPropertyStore_Release(pProperties);
        }
    }
#endif

done:
    /* Clean up. */
#ifdef MA_WIN32_DESKTOP
    if (pDeviceInterface != NULL) {
        ma_IMMDevice_Release(pDeviceInterface);
    }
#else
    if (pDeviceInterface != NULL) {
        ma_IUnknown_Release(pDeviceInterface);
    }
#endif

    if (result != MA_SUCCESS) {
        if (pData->pRenderClient) {
            ma_IAudioRenderClient_Release((ma_IAudioRenderClient*)pData->pRenderClient);
            pData->pRenderClient = NULL;
        }
        if (pData->pCaptureClient) {
            ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pData->pCaptureClient);
            pData->pCaptureClient = NULL;
        }
        if (pData->pAudioClient) {
            ma_IAudioClient_Release((ma_IAudioClient*)pData->pAudioClient);
            pData->pAudioClient = NULL;
        }

        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, errorMsg, result);
    } else {
        return MA_SUCCESS;
    }
}

ma_result ma_device_reinit__wasapi(ma_device* pDevice, ma_device_type deviceType)
{
    ma_device_init_internal_data__wasapi data;
    ma_result result;

    ma_assert(pDevice != NULL);

    /* We only re-initialize the playback or capture device. Never a full-duplex device. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    if (deviceType == ma_device_type_capture) {
        data.formatIn               = pDevice->capture.format;
        data.channelsIn             = pDevice->capture.channels;
        ma_copy_memory(data.channelMapIn, pDevice->capture.channelMap, sizeof(pDevice->capture.channelMap));
        data.shareMode              = pDevice->capture.shareMode;
        data.usingDefaultFormat     = pDevice->capture.usingDefaultFormat;
        data.usingDefaultChannels   = pDevice->capture.usingDefaultChannels;
        data.usingDefaultChannelMap = pDevice->capture.usingDefaultChannelMap;
    } else {
        data.formatIn               = pDevice->playback.format;
        data.channelsIn             = pDevice->playback.channels;
        ma_copy_memory(data.channelMapIn, pDevice->playback.channelMap, sizeof(pDevice->playback.channelMap));
        data.shareMode              = pDevice->playback.shareMode;
        data.usingDefaultFormat     = pDevice->playback.usingDefaultFormat;
        data.usingDefaultChannels   = pDevice->playback.usingDefaultChannels;
        data.usingDefaultChannelMap = pDevice->playback.usingDefaultChannelMap;
    }
    
    data.sampleRateIn               = pDevice->sampleRate;
    data.usingDefaultSampleRate     = pDevice->usingDefaultSampleRate;
    data.bufferSizeInFramesIn       = pDevice->wasapi.originalBufferSizeInFrames;
    data.bufferSizeInMillisecondsIn = pDevice->wasapi.originalBufferSizeInMilliseconds;
    data.periodsIn                  = pDevice->wasapi.originalPeriods;
    result = ma_device_init_internal__wasapi(pDevice->pContext, deviceType, NULL, &data);
    if (result != MA_SUCCESS) {
        return result;
    }

    /* At this point we have some new objects ready to go. We need to uninitialize the previous ones and then set the new ones. */
    if (deviceType == ma_device_type_capture) {
        if (pDevice->wasapi.pCaptureClient) {
            ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
            pDevice->wasapi.pCaptureClient = NULL;
        }

        if (pDevice->wasapi.pAudioClientCapture) {
            ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
            pDevice->wasapi.pAudioClientCapture = NULL;
        }

        pDevice->wasapi.pAudioClientCapture         = data.pAudioClient;
        pDevice->wasapi.pCaptureClient              = data.pCaptureClient;

        pDevice->capture.internalFormat             = data.formatOut;
        pDevice->capture.internalChannels           = data.channelsOut;
        pDevice->capture.internalSampleRate         = data.sampleRateOut;
        ma_copy_memory(pDevice->capture.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->capture.internalBufferSizeInFrames = data.bufferSizeInFramesOut;
        pDevice->capture.internalPeriods            = data.periodsOut;
        ma_strcpy_s(pDevice->capture.name, sizeof(pDevice->capture.name), data.deviceName);

        ma_IAudioClient_SetEventHandle((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture,  pDevice->wasapi.hEventCapture);

        pDevice->wasapi.periodSizeInFramesCapture = data.periodSizeInFramesOut;
        ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture, &pDevice->wasapi.actualBufferSizeInFramesCapture);

        /* The device may be in a started state. If so we need to immediately restart it. */
        if (pDevice->wasapi.isStartedCapture) {
            HRESULT hr = ma_IAudioClient_Start((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
            if (FAILED(hr)) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal capture device after reinitialization.", MA_FAILED_TO_START_BACKEND_DEVICE);
            }
        }
    }

    if (deviceType == ma_device_type_playback) {
        if (pDevice->wasapi.pRenderClient) {
            ma_IAudioRenderClient_Release((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient);
            pDevice->wasapi.pRenderClient = NULL;
        }

        if (pDevice->wasapi.pAudioClientPlayback) {
            ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
            pDevice->wasapi.pAudioClientPlayback = NULL;
        }

        pDevice->wasapi.pAudioClientPlayback         = data.pAudioClient;
        pDevice->wasapi.pRenderClient                = data.pRenderClient;

        pDevice->playback.internalFormat             = data.formatOut;
        pDevice->playback.internalChannels           = data.channelsOut;
        pDevice->playback.internalSampleRate         = data.sampleRateOut;
        ma_copy_memory(pDevice->playback.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->playback.internalBufferSizeInFrames = data.bufferSizeInFramesOut;
        pDevice->playback.internalPeriods            = data.periodsOut;
        ma_strcpy_s(pDevice->playback.name, sizeof(pDevice->playback.name), data.deviceName);

        ma_IAudioClient_SetEventHandle((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, pDevice->wasapi.hEventPlayback);

        pDevice->wasapi.periodSizeInFramesPlayback = data.periodSizeInFramesOut;
        ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, &pDevice->wasapi.actualBufferSizeInFramesPlayback);

        /* The device may be in a started state. If so we need to immediately restart it. */
        if (pDevice->wasapi.isStartedPlayback) {
            HRESULT hr = ma_IAudioClient_Start((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
            if (FAILED(hr)) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal playback device after reinitialization.", MA_FAILED_TO_START_BACKEND_DEVICE);
            }
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_init__wasapi(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;

    (void)pContext;

    ma_assert(pContext != NULL);
    ma_assert(pDevice != NULL);

    ma_zero_object(&pDevice->wasapi);
    pDevice->wasapi.originalBufferSizeInFrames       = pConfig->bufferSizeInFrames;
    pDevice->wasapi.originalBufferSizeInMilliseconds = pConfig->bufferSizeInMilliseconds;
    pDevice->wasapi.originalPeriods                  = pConfig->periods;

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_device_init_internal_data__wasapi data;
        data.formatIn                   = pConfig->capture.format;
        data.channelsIn                 = pConfig->capture.channels;
        data.sampleRateIn               = pConfig->sampleRate;
        ma_copy_memory(data.channelMapIn, pConfig->capture.channelMap, sizeof(pConfig->capture.channelMap));
        data.usingDefaultFormat         = pDevice->capture.usingDefaultFormat;
        data.usingDefaultChannels       = pDevice->capture.usingDefaultChannels;
        data.usingDefaultSampleRate     = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap     = pDevice->capture.usingDefaultChannelMap;
        data.shareMode                  = pConfig->capture.shareMode;
        data.bufferSizeInFramesIn       = pConfig->bufferSizeInFrames;
        data.bufferSizeInMillisecondsIn = pConfig->bufferSizeInMilliseconds;
        data.periodsIn                  = pConfig->periods;

        result = ma_device_init_internal__wasapi(pDevice->pContext, ma_device_type_capture, pConfig->capture.pDeviceID, &data);
        if (result != MA_SUCCESS) {
            return result;
        }

        pDevice->wasapi.pAudioClientCapture         = data.pAudioClient;
        pDevice->wasapi.pCaptureClient              = data.pCaptureClient;

        pDevice->capture.internalFormat             = data.formatOut;
        pDevice->capture.internalChannels           = data.channelsOut;
        pDevice->capture.internalSampleRate         = data.sampleRateOut;
        ma_copy_memory(pDevice->capture.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->capture.internalBufferSizeInFrames = data.bufferSizeInFramesOut;
        pDevice->capture.internalPeriods            = data.periodsOut;
        ma_strcpy_s(pDevice->capture.name, sizeof(pDevice->capture.name), data.deviceName);

        /*
        The event for capture needs to be manual reset for the same reason as playback. We keep the initial state set to unsignaled,
        however, because we want to block until we actually have something for the first call to ma_device_read().
        */
        pDevice->wasapi.hEventCapture = CreateEventA(NULL, FALSE, FALSE, NULL);  /* Auto reset, unsignaled by default. */
        if (pDevice->wasapi.hEventCapture == NULL) {
            if (pDevice->wasapi.pCaptureClient != NULL) {
                ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
                pDevice->wasapi.pCaptureClient = NULL;
            }
            if (pDevice->wasapi.pAudioClientCapture != NULL) {
                ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
                pDevice->wasapi.pAudioClientCapture = NULL;
            }

            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create event for capture.", MA_FAILED_TO_CREATE_EVENT);
        }
        ma_IAudioClient_SetEventHandle((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture, pDevice->wasapi.hEventCapture);

        pDevice->wasapi.periodSizeInFramesCapture = data.periodSizeInFramesOut;
        ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture, &pDevice->wasapi.actualBufferSizeInFramesCapture);
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_device_init_internal_data__wasapi data;
        data.formatIn                   = pConfig->playback.format;
        data.channelsIn                 = pConfig->playback.channels;
        data.sampleRateIn               = pConfig->sampleRate;
        ma_copy_memory(data.channelMapIn, pConfig->playback.channelMap, sizeof(pConfig->playback.channelMap));
        data.usingDefaultFormat         = pDevice->playback.usingDefaultFormat;
        data.usingDefaultChannels       = pDevice->playback.usingDefaultChannels;
        data.usingDefaultSampleRate     = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap     = pDevice->playback.usingDefaultChannelMap;
        data.shareMode                  = pConfig->playback.shareMode;
        data.bufferSizeInFramesIn       = pConfig->bufferSizeInFrames;
        data.bufferSizeInMillisecondsIn = pConfig->bufferSizeInMilliseconds;
        data.periodsIn                  = pConfig->periods;

        result = ma_device_init_internal__wasapi(pDevice->pContext, ma_device_type_playback, pConfig->playback.pDeviceID, &data);
        if (result != MA_SUCCESS) {
            if (pConfig->deviceType == ma_device_type_duplex) {
                if (pDevice->wasapi.pCaptureClient != NULL) {
                    ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
                    pDevice->wasapi.pCaptureClient = NULL;
                }
                if (pDevice->wasapi.pAudioClientCapture != NULL) {
                    ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
                    pDevice->wasapi.pAudioClientCapture = NULL;
                }

                CloseHandle(pDevice->wasapi.hEventCapture);
                pDevice->wasapi.hEventCapture = NULL;
            }
            return result;
        }

        pDevice->wasapi.pAudioClientPlayback         = data.pAudioClient;
        pDevice->wasapi.pRenderClient                = data.pRenderClient;

        pDevice->playback.internalFormat             = data.formatOut;
        pDevice->playback.internalChannels           = data.channelsOut;
        pDevice->playback.internalSampleRate         = data.sampleRateOut;
        ma_copy_memory(pDevice->playback.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->playback.internalBufferSizeInFrames = data.bufferSizeInFramesOut;
        pDevice->playback.internalPeriods            = data.periodsOut;
        ma_strcpy_s(pDevice->playback.name, sizeof(pDevice->playback.name), data.deviceName);

        /*
        The event for playback is needs to be manual reset because we want to explicitly control the fact that it becomes signalled
        only after the whole available space has been filled, never before.

        The playback event also needs to be initially set to a signaled state so that the first call to ma_device_write() is able
        to get passed WaitForMultipleObjects().
        */
        pDevice->wasapi.hEventPlayback = CreateEventA(NULL, FALSE, TRUE, NULL);  /* Auto reset, signaled by default. */
        if (pDevice->wasapi.hEventPlayback == NULL) {
            if (pConfig->deviceType == ma_device_type_duplex) {
                if (pDevice->wasapi.pCaptureClient != NULL) {
                    ma_IAudioCaptureClient_Release((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient);
                    pDevice->wasapi.pCaptureClient = NULL;
                }
                if (pDevice->wasapi.pAudioClientCapture != NULL) {
                    ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
                    pDevice->wasapi.pAudioClientCapture = NULL;
                }

                CloseHandle(pDevice->wasapi.hEventCapture);
                pDevice->wasapi.hEventCapture = NULL;
            }

            if (pDevice->wasapi.pRenderClient != NULL) {
                ma_IAudioRenderClient_Release((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient);
                pDevice->wasapi.pRenderClient = NULL;
            }
            if (pDevice->wasapi.pAudioClientPlayback != NULL) {
                ma_IAudioClient_Release((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
                pDevice->wasapi.pAudioClientPlayback = NULL;
            }

            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create event for playback.", MA_FAILED_TO_CREATE_EVENT);
        }
        ma_IAudioClient_SetEventHandle((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, pDevice->wasapi.hEventPlayback);

        pDevice->wasapi.periodSizeInFramesPlayback = data.periodSizeInFramesOut;
        ma_IAudioClient_GetBufferSize((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, &pDevice->wasapi.actualBufferSizeInFramesPlayback);
    }

    /*
    We need to get notifications of when the default device changes. We do this through a device enumerator by
    registering a IMMNotificationClient with it. We only care about this if it's the default device.
    */
#ifdef MA_WIN32_DESKTOP
    {
        ma_IMMDeviceEnumerator* pDeviceEnumerator;
        HRESULT hr = ma_CoCreateInstance(pContext, MA_CLSID_MMDeviceEnumerator, NULL, CLSCTX_ALL, MA_IID_IMMDeviceEnumerator, (void**)&pDeviceEnumerator);
        if (FAILED(hr)) {
            ma_device_uninit__wasapi(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to create device enumerator.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        pDevice->wasapi.notificationClient.lpVtbl  = (void*)&g_maNotificationCientVtbl;
        pDevice->wasapi.notificationClient.counter = 1;
        pDevice->wasapi.notificationClient.pDevice = pDevice;

        hr = pDeviceEnumerator->lpVtbl->RegisterEndpointNotificationCallback(pDeviceEnumerator, &pDevice->wasapi.notificationClient);
        if (SUCCEEDED(hr)) {
            pDevice->wasapi.pDeviceEnumerator = (ma_ptr)pDeviceEnumerator;
        } else {
            /* Not the end of the world if we fail to register the notification callback. We just won't support automatic stream routing. */
            ma_IMMDeviceEnumerator_Release(pDeviceEnumerator);
        }
    }
#endif

    ma_atomic_exchange_32(&pDevice->wasapi.isStartedCapture,  MA_FALSE);
    ma_atomic_exchange_32(&pDevice->wasapi.isStartedPlayback, MA_FALSE);

    return MA_SUCCESS;
}

ma_result ma_device__get_available_frames__wasapi(ma_device* pDevice, ma_IAudioClient* pAudioClient, ma_uint32* pFrameCount)
{
    ma_uint32 paddingFramesCount;
    HRESULT hr;
    ma_share_mode shareMode;

    ma_assert(pDevice != NULL);
    ma_assert(pFrameCount != NULL);
    
    *pFrameCount = 0;

    if ((ma_ptr)pAudioClient != pDevice->wasapi.pAudioClientPlayback && (ma_ptr)pAudioClient != pDevice->wasapi.pAudioClientCapture) {
        return MA_INVALID_OPERATION;
    }

    hr = ma_IAudioClient_GetCurrentPadding(pAudioClient, &paddingFramesCount);
    if (FAILED(hr)) {
        return MA_DEVICE_UNAVAILABLE;
    }

    /* Slightly different rules for exclusive and shared modes. */
    shareMode = ((ma_ptr)pAudioClient == pDevice->wasapi.pAudioClientPlayback) ? pDevice->playback.shareMode : pDevice->capture.shareMode;
    if (shareMode == ma_share_mode_exclusive) {
        *pFrameCount = paddingFramesCount;
    } else {
        if ((ma_ptr)pAudioClient == pDevice->wasapi.pAudioClientPlayback) {
            *pFrameCount = pDevice->wasapi.actualBufferSizeInFramesPlayback - paddingFramesCount;
        } else {
            *pFrameCount = paddingFramesCount;
        }
    }

    return MA_SUCCESS;
}

ma_bool32 ma_device_is_reroute_required__wasapi(ma_device* pDevice, ma_device_type deviceType)
{
    ma_assert(pDevice != NULL);

    if (deviceType == ma_device_type_playback) {
        return pDevice->wasapi.hasDefaultPlaybackDeviceChanged;
    }

    if (deviceType == ma_device_type_capture) {
        return pDevice->wasapi.hasDefaultCaptureDeviceChanged;
    }
    
    return MA_FALSE;
}

ma_result ma_device_reroute__wasapi(ma_device* pDevice, ma_device_type deviceType)
{
    ma_result result;

    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    if (deviceType == ma_device_type_playback) {
        ma_atomic_exchange_32(&pDevice->wasapi.hasDefaultPlaybackDeviceChanged, MA_FALSE);
    }
    if (deviceType == ma_device_type_capture) {
        ma_atomic_exchange_32(&pDevice->wasapi.hasDefaultCaptureDeviceChanged,  MA_FALSE);
    }
    

    #ifdef MA_DEBUG_OUTPUT
        printf("=== CHANGING DEVICE ===\n");
    #endif

    result = ma_device_reinit__wasapi(pDevice, deviceType);
    if (result != MA_SUCCESS) {
        return result;
    }

    ma_device__post_init_setup(pDevice, deviceType);

    return MA_SUCCESS;
}


ma_result ma_device_main_loop__wasapi(ma_device* pDevice)
{
    ma_result result;
    HRESULT hr;
    ma_bool32 exitLoop = MA_FALSE;
    ma_uint32 framesWrittenToPlaybackDevice = 0;
    ma_uint32 mappedBufferSizeInFramesCapture = 0;
    ma_uint32 mappedBufferSizeInFramesPlayback = 0;
    ma_uint32 mappedBufferFramesRemainingCapture = 0;
    ma_uint32 mappedBufferFramesRemainingPlayback = 0;
    BYTE* pMappedBufferCapture = NULL;
    BYTE* pMappedBufferPlayback = NULL;
    ma_uint32 bpfCapture = ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
    ma_uint32 bpfPlayback = ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
    ma_uint8  inputDataInExternalFormat[4096];
    ma_uint32 inputDataInExternalFormatCap = sizeof(inputDataInExternalFormat) / bpfCapture;
    ma_uint8  outputDataInExternalFormat[4096];
    ma_uint32 outputDataInExternalFormatCap = sizeof(outputDataInExternalFormat) / bpfPlayback;

    ma_assert(pDevice != NULL);

    /* The playback device needs to be started immediately. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        hr = ma_IAudioClient_Start((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal capture device.", MA_FAILED_TO_START_BACKEND_DEVICE);
        }
        ma_atomic_exchange_32(&pDevice->wasapi.isStartedCapture, MA_TRUE);
    }

    while (ma_device__get_state(pDevice) == MA_STATE_STARTED && !exitLoop) {
        /* We may need to reroute the device. */
        if (ma_device_is_reroute_required__wasapi(pDevice, ma_device_type_playback)) {
            result = ma_device_reroute__wasapi(pDevice, ma_device_type_playback);
            if (result != MA_SUCCESS) {
                exitLoop = MA_TRUE;
                break;
            }
        }
        if (ma_device_is_reroute_required__wasapi(pDevice, ma_device_type_capture)) {
            result = ma_device_reroute__wasapi(pDevice, ma_device_type_capture);
            if (result != MA_SUCCESS) {
                exitLoop = MA_TRUE;
                break;
            }
        }

        switch (pDevice->type)
        {
            case ma_device_type_duplex:
            {
                ma_uint32 framesAvailableCapture;
                ma_uint32 framesAvailablePlayback;
                DWORD flagsCapture;    /* Passed to IAudioCaptureClient_GetBuffer(). */

                /* The process is to map the playback buffer and fill it as quickly as possible from input data. */
                if (pMappedBufferPlayback == NULL) {
                    /* WASAPI is weird with exclusive mode. You need to wait on the event _before_ querying the available frames. */
                    if (pDevice->playback.shareMode == ma_share_mode_exclusive) {
                        if (WaitForSingleObject(pDevice->wasapi.hEventPlayback, INFINITE) == WAIT_FAILED) {
                            return MA_ERROR;   /* Wait failed. */
                        }
                    }

                    result = ma_device__get_available_frames__wasapi(pDevice, (ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, &framesAvailablePlayback);
                    if (result != MA_SUCCESS) {
                        return result;
                    }

                    /*printf("TRACE 1: framesAvailablePlayback=%d\n", framesAvailablePlayback);*/


                    /* In exclusive mode, the frame count needs to exactly match the value returned by GetCurrentPadding(). */
                    if (pDevice->playback.shareMode != ma_share_mode_exclusive) {
                        if (framesAvailablePlayback >= pDevice->wasapi.periodSizeInFramesPlayback) {
                            framesAvailablePlayback  = pDevice->wasapi.periodSizeInFramesPlayback;
                        }
                    }

                    /* If there's no frames available in the playback device we need to wait for more. */
                    if (framesAvailablePlayback == 0) {
                        /* In exclusive mode we waited at the top. */
                        if (pDevice->playback.shareMode != ma_share_mode_exclusive) {
                            if (WaitForSingleObject(pDevice->wasapi.hEventPlayback, INFINITE) == WAIT_FAILED) {
                                return MA_ERROR;   /* Wait failed. */
                            }
                        }

                        continue;
                    }

                    /* We're ready to map the playback device's buffer. We don't release this until it's been entirely filled. */
                    hr = ma_IAudioRenderClient_GetBuffer((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient, framesAvailablePlayback, &pMappedBufferPlayback);
                    if (FAILED(hr)) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from playback device in preparation for writing to the device.", MA_FAILED_TO_MAP_DEVICE_BUFFER);
                        exitLoop = MA_TRUE;
                        break;
                    }

                    mappedBufferSizeInFramesPlayback = framesAvailablePlayback;
                    mappedBufferFramesRemainingPlayback = framesAvailablePlayback;
                }

                /* At this point we should have a buffer available for output. We need to keep writing input samples to it. */
                for (;;) {
                    /* Try grabbing some captured data if we haven't already got a mapped buffer. */
                    if (pMappedBufferCapture == NULL) {
                        if (pDevice->capture.shareMode == ma_share_mode_shared) {
                            if (WaitForSingleObject(pDevice->wasapi.hEventCapture, INFINITE) == WAIT_FAILED) {
                                return MA_ERROR;   /* Wait failed. */
                            }
                        }

                        result = ma_device__get_available_frames__wasapi(pDevice, (ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture, &framesAvailableCapture);
                        if (result != MA_SUCCESS) {
                            exitLoop = MA_TRUE;
                            break;
                        }

                        /*printf("TRACE 2: framesAvailableCapture=%d\n", framesAvailableCapture);*/

                        /* Wait for more if nothing is available. */
                        if (framesAvailableCapture == 0) {
                            /* In exclusive mode we waited at the top. */
                            if (pDevice->capture.shareMode != ma_share_mode_shared) {
                                if (WaitForSingleObject(pDevice->wasapi.hEventCapture, INFINITE) == WAIT_FAILED) {
                                    return MA_ERROR;   /* Wait failed. */
                                }
                            }

                            continue;
                        }

                        /* Getting here means there's data available for writing to the output device. */
                        mappedBufferSizeInFramesCapture = framesAvailableCapture;
                        hr = ma_IAudioCaptureClient_GetBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, (BYTE**)&pMappedBufferCapture, &mappedBufferSizeInFramesCapture, &flagsCapture, NULL, NULL);
                        if (FAILED(hr)) {
                            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from capture device in preparation for writing to the device.", MA_FAILED_TO_MAP_DEVICE_BUFFER);
                            exitLoop = MA_TRUE;
                            break;
                        }

                        /* TODO: How do we handle the capture flags returned by GetBuffer()? In particular, AUDCLNT_BUFFERFLAGS_SILENT (1). */
                    #ifdef MA_DEBUG_OUTPUT
                        if (flagsCapture != 0) {
                            printf("[WASAPI] Capture Flags: %d\n", flagsCapture);
                        }
                    #endif

                        mappedBufferFramesRemainingCapture = mappedBufferSizeInFramesCapture;

                        pDevice->capture._dspFrameCount = mappedBufferSizeInFramesCapture;
                        pDevice->capture._dspFrames     = (const ma_uint8*)pMappedBufferCapture;
                    }


                    /* At this point we should have both input and output data available. We now need to post it to the convert the data and post it to the client. */
                    for (;;) {
                        BYTE* pRunningBufferCapture;
                        BYTE* pRunningBufferPlayback;
                        ma_uint32 framesToProcess;
                        ma_uint32 framesProcessed;

                        pRunningBufferCapture  = pMappedBufferCapture  + ((mappedBufferSizeInFramesCapture  - mappedBufferFramesRemainingCapture ) * bpfPlayback);
                        pRunningBufferPlayback = pMappedBufferPlayback + ((mappedBufferSizeInFramesPlayback - mappedBufferFramesRemainingPlayback) * bpfPlayback);
                        
                        /* There may be some data sitting in the converter that needs to be processed first. Once this is exhaused, run the data callback again. */
                        if (!pDevice->playback.converter.isPassthrough) {
                            framesProcessed = (ma_uint32)ma_pcm_converter_read(&pDevice->playback.converter, pRunningBufferPlayback, mappedBufferFramesRemainingPlayback);
                            if (framesProcessed > 0) {
                                mappedBufferFramesRemainingPlayback -= framesProcessed;
                                if (mappedBufferFramesRemainingPlayback == 0) {
                                    break;
                                }
                            }
                        }

                        /*
                        Getting here means we need to fire the callback. If format conversion is unnecessary, we can optimize this by passing the pointers to the internal
                        buffers directly to the callback.
                        */
                        if (pDevice->capture.converter.isPassthrough && pDevice->playback.converter.isPassthrough) {
                            /* Optimal path. We can pass mapped pointers directly to the callback. */
                            framesToProcess = ma_min(mappedBufferFramesRemainingCapture, mappedBufferFramesRemainingPlayback);
                            framesProcessed = framesToProcess;

                            pDevice->onData(pDevice, pRunningBufferPlayback, pRunningBufferCapture, framesToProcess);

                            mappedBufferFramesRemainingCapture  -= framesProcessed;
                            mappedBufferFramesRemainingPlayback -= framesProcessed;

                            if (mappedBufferFramesRemainingCapture == 0) {
                                break;  /* Exhausted input data. */
                            }
                            if (mappedBufferFramesRemainingPlayback == 0) {
                                break;  /* Exhausted output data. */
                            }
                        } else if (pDevice->capture.converter.isPassthrough) {
                            /* The input buffer is a passthrough, but the playback buffer requires a conversion. */
                            framesToProcess = ma_min(mappedBufferFramesRemainingCapture, outputDataInExternalFormatCap);
                            framesProcessed = framesToProcess;

                            pDevice->onData(pDevice, outputDataInExternalFormat, pRunningBufferCapture, framesToProcess);
                            mappedBufferFramesRemainingCapture -= framesProcessed;

                            pDevice->playback._dspFrameCount = framesProcessed;
                            pDevice->playback._dspFrames     = (const ma_uint8*)outputDataInExternalFormat;

                            if (mappedBufferFramesRemainingCapture == 0) {
                                break;  /* Exhausted input data. */
                            }
                        } else if (pDevice->playback.converter.isPassthrough) {
                            /* The input buffer requires conversion, the playback buffer is passthrough. */
                            framesToProcess = ma_min(inputDataInExternalFormatCap, mappedBufferFramesRemainingPlayback);
                            framesProcessed = (ma_uint32)ma_pcm_converter_read(&pDevice->capture.converter, inputDataInExternalFormat, framesToProcess);
                            if (framesProcessed == 0) {
                                /* Getting here means we've run out of input data. */
                                mappedBufferFramesRemainingCapture = 0;
                                break;
                            }

                            pDevice->onData(pDevice, pRunningBufferPlayback, inputDataInExternalFormat, framesProcessed);
                            mappedBufferFramesRemainingPlayback -= framesProcessed;

                            if (framesProcessed < framesToProcess) {
                                mappedBufferFramesRemainingCapture = 0;
                                break;  /* Exhausted input data. */
                            }

                            if (mappedBufferFramesRemainingPlayback == 0) {
                                break;  /* Exhausted output data. */
                            }
                        } else {
                            framesToProcess = ma_min(inputDataInExternalFormatCap, outputDataInExternalFormatCap);
                            framesProcessed = (ma_uint32)ma_pcm_converter_read(&pDevice->capture.converter, inputDataInExternalFormat, framesToProcess);
                            if (framesProcessed == 0) {
                                /* Getting here means we've run out of input data. */
                                mappedBufferFramesRemainingCapture = 0;
                                break;
                            }

                            pDevice->onData(pDevice, outputDataInExternalFormat, inputDataInExternalFormat, framesProcessed);

                            pDevice->playback._dspFrameCount = framesProcessed;
                            pDevice->playback._dspFrames     = (const ma_uint8*)outputDataInExternalFormat;

                            if (framesProcessed < framesToProcess) {
                                /* Getting here means we've run out of input data. */
                                mappedBufferFramesRemainingCapture = 0;
                                break;
                            }
                        }
                    }


                    /* If at this point we've run out of capture data we need to release the buffer. */
                    if (mappedBufferFramesRemainingCapture == 0 && pMappedBufferCapture != NULL) {
                        hr = ma_IAudioCaptureClient_ReleaseBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, mappedBufferSizeInFramesCapture);
                        if (FAILED(hr)) {
                            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to release internal buffer from capture device after reading from the device.", MA_FAILED_TO_UNMAP_DEVICE_BUFFER);
                            exitLoop = MA_TRUE;
                            break;
                        }

                        /*printf("TRACE: Released capture buffer\n");*/

                        pMappedBufferCapture = NULL;
                        mappedBufferFramesRemainingCapture = 0;
                        mappedBufferSizeInFramesCapture = 0;
                    }

                    /* Get out of this loop if we're run out of room in the playback buffer. */
                    if (mappedBufferFramesRemainingPlayback == 0) {
                        break;
                    }
                }


                /* If at this point we've run out of data we need to release the buffer. */
                if (mappedBufferFramesRemainingPlayback == 0 && pMappedBufferPlayback != NULL) {
                    hr = ma_IAudioRenderClient_ReleaseBuffer((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient, mappedBufferSizeInFramesPlayback, 0);
                    if (FAILED(hr)) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to release internal buffer from playback device after writing to the device.", MA_FAILED_TO_UNMAP_DEVICE_BUFFER);
                        exitLoop = MA_TRUE;
                        break;
                    }

                    /*printf("TRACE: Released playback buffer\n");*/
                    framesWrittenToPlaybackDevice += mappedBufferSizeInFramesPlayback;

                    pMappedBufferPlayback = NULL;
                    mappedBufferFramesRemainingPlayback = 0;
                    mappedBufferSizeInFramesPlayback = 0;
                }

                if (!pDevice->wasapi.isStartedPlayback) {
                    if (pDevice->playback.shareMode == ma_share_mode_exclusive || framesWrittenToPlaybackDevice >= (pDevice->playback.internalBufferSizeInFrames/pDevice->playback.internalPeriods)*2) {
                        hr = ma_IAudioClient_Start((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
                        if (FAILED(hr)) {
                            ma_IAudioClient_Stop((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
                            ma_IAudioClient_Reset((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
                            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal playback device.", MA_FAILED_TO_START_BACKEND_DEVICE);
                        }
                        ma_atomic_exchange_32(&pDevice->wasapi.isStartedPlayback, MA_TRUE);
                    }
                }
            } break;



            case ma_device_type_capture:
            {
                ma_uint32 framesAvailableCapture;
                DWORD flagsCapture;    /* Passed to IAudioCaptureClient_GetBuffer(). */

                /* Wait for data to become available first. */
                if (WaitForSingleObject(pDevice->wasapi.hEventCapture, INFINITE) == WAIT_FAILED) {
                    exitLoop = MA_TRUE;
                    break;   /* Wait failed. */
                }

                /* See how many frames are available. Since we waited at the top, I don't think this should ever return 0. I'm checking for this anyway. */
                result = ma_device__get_available_frames__wasapi(pDevice, (ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture, &framesAvailableCapture);
                if (result != MA_SUCCESS) {
                    exitLoop = MA_TRUE;
                    break;
                }

                if (framesAvailableCapture < pDevice->wasapi.periodSizeInFramesCapture) {
                    continue;   /* Nothing available. Keep waiting. */
                }

                /* Map a the data buffer in preparation for sending to the client. */
                mappedBufferSizeInFramesCapture = framesAvailableCapture;
                hr = ma_IAudioCaptureClient_GetBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, (BYTE**)&pMappedBufferCapture, &mappedBufferSizeInFramesCapture, &flagsCapture, NULL, NULL);
                if (FAILED(hr)) {
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from capture device in preparation for writing to the device.", MA_FAILED_TO_MAP_DEVICE_BUFFER);
                    exitLoop = MA_TRUE;
                    break;
                }

                /* We should have a buffer at this point. */
                ma_device__send_frames_to_client(pDevice, mappedBufferSizeInFramesCapture, pMappedBufferCapture);

                /* At this point we're done with the buffer. */
                hr = ma_IAudioCaptureClient_ReleaseBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, mappedBufferSizeInFramesCapture);
                pMappedBufferCapture = NULL;    /* <-- Important. Not doing this can result in an error once we leave this loop because it will use this to know whether or not a final ReleaseBuffer() needs to be called. */
                mappedBufferSizeInFramesCapture = 0;
                if (FAILED(hr)) {
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to release internal buffer from capture device after reading from the device.", MA_FAILED_TO_UNMAP_DEVICE_BUFFER);
                    exitLoop = MA_TRUE;
                    break;
                }
            } break;



            case ma_device_type_playback:
            {
                ma_uint32 framesAvailablePlayback;

                /* Wait for space to become available first. */
                if (WaitForSingleObject(pDevice->wasapi.hEventPlayback, INFINITE) == WAIT_FAILED) {
                    exitLoop = MA_TRUE;
                    break;   /* Wait failed. */
                }

                /* Check how much space is available. If this returns 0 we just keep waiting. */
                result = ma_device__get_available_frames__wasapi(pDevice, (ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, &framesAvailablePlayback);
                if (result != MA_SUCCESS) {
                    exitLoop = MA_TRUE;
                    break;
                }

                if (framesAvailablePlayback < pDevice->wasapi.periodSizeInFramesPlayback) {
                    continue;   /* No space available. */
                }

                /* Map a the data buffer in preparation for the callback. */
                hr = ma_IAudioRenderClient_GetBuffer((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient, framesAvailablePlayback, &pMappedBufferPlayback);
                if (FAILED(hr)) {
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to retrieve internal buffer from playback device in preparation for writing to the device.", MA_FAILED_TO_MAP_DEVICE_BUFFER);
                    exitLoop = MA_TRUE;
                    break;
                }

                /* We should have a buffer at this point. */
                ma_device__read_frames_from_client(pDevice, framesAvailablePlayback, pMappedBufferPlayback);

                /* At this point we're done writing to the device and we just need to release the buffer. */
                hr = ma_IAudioRenderClient_ReleaseBuffer((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient, framesAvailablePlayback, 0);
                pMappedBufferPlayback = NULL;    /* <-- Important. Not doing this can result in an error once we leave this loop because it will use this to know whether or not a final ReleaseBuffer() needs to be called. */
                mappedBufferSizeInFramesPlayback = 0;

                if (FAILED(hr)) {
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to release internal buffer from playback device after writing to the device.", MA_FAILED_TO_UNMAP_DEVICE_BUFFER);
                    exitLoop = MA_TRUE;
                    break;
                }

                framesWrittenToPlaybackDevice += framesAvailablePlayback;
                if (!pDevice->wasapi.isStartedPlayback) {
                    if (pDevice->playback.shareMode == ma_share_mode_exclusive || framesWrittenToPlaybackDevice >= (pDevice->playback.internalBufferSizeInFrames/pDevice->playback.internalPeriods)*1) {
                        hr = ma_IAudioClient_Start((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
                        if (FAILED(hr)) {
                            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to start internal playback device.", MA_FAILED_TO_START_BACKEND_DEVICE);
                            exitLoop = MA_TRUE;
                            break;
                        }
                        ma_atomic_exchange_32(&pDevice->wasapi.isStartedPlayback, MA_TRUE);
                    }
                }
            } break;

            default: return MA_INVALID_ARGS;
        }
    }

    /* Here is where the device needs to be stopped. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        /* Any mapped buffers need to be released. */
        if (pMappedBufferCapture != NULL) {
            hr = ma_IAudioCaptureClient_ReleaseBuffer((ma_IAudioCaptureClient*)pDevice->wasapi.pCaptureClient, mappedBufferSizeInFramesCapture);
        }

        hr = ma_IAudioClient_Stop((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to stop internal capture device.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
        }

        /* The audio client needs to be reset otherwise restarting will fail. */
        hr = ma_IAudioClient_Reset((ma_IAudioClient*)pDevice->wasapi.pAudioClientCapture);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to reset internal capture device.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
        }

        ma_atomic_exchange_32(&pDevice->wasapi.isStartedCapture, MA_FALSE);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        /* Any mapped buffers need to be released. */
        if (pMappedBufferPlayback != NULL) {
            hr = ma_IAudioRenderClient_ReleaseBuffer((ma_IAudioRenderClient*)pDevice->wasapi.pRenderClient, mappedBufferSizeInFramesPlayback, 0);
        }

        /*
        The buffer needs to be drained before stopping the device. Not doing this will result in the last few frames not getting output to
        the speakers. This is a problem for very short sounds because it'll result in a significant potion of it not getting played.
        */
        if (pDevice->wasapi.isStartedPlayback) {
            if (pDevice->playback.shareMode == ma_share_mode_exclusive) {
                WaitForSingleObject(pDevice->wasapi.hEventPlayback, INFINITE);
            } else {
                ma_uint32 prevFramesAvaialablePlayback = (size_t)-1;
                ma_uint32 framesAvailablePlayback;
                for (;;) {
                    result = ma_device__get_available_frames__wasapi(pDevice, (ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback, &framesAvailablePlayback);
                    if (result != MA_SUCCESS) {
                        break;
                    }

                    if (framesAvailablePlayback >= pDevice->wasapi.actualBufferSizeInFramesPlayback) {
                        break;
                    }

                    /*
                    Just a safety check to avoid an infinite loop. If this iteration results in a situation where the number of available frames
                    has not changed, get out of the loop. I don't think this should ever happen, but I think it's nice to have just in case.
                    */
                    if (framesAvailablePlayback == prevFramesAvaialablePlayback) {
                        break;
                    }
                    prevFramesAvaialablePlayback = framesAvailablePlayback;

                    WaitForSingleObject(pDevice->wasapi.hEventPlayback, INFINITE);
                    ResetEvent(pDevice->wasapi.hEventPlayback); /* Manual reset. */
                }
            }
        }

        hr = ma_IAudioClient_Stop((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to stop internal playback device.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
        }

        /* The audio client needs to be reset otherwise restarting will fail. */
        hr = ma_IAudioClient_Reset((ma_IAudioClient*)pDevice->wasapi.pAudioClientPlayback);
        if (FAILED(hr)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WASAPI] Failed to reset internal playback device.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
        }

        ma_atomic_exchange_32(&pDevice->wasapi.isStartedPlayback, MA_FALSE);
    }

    return MA_SUCCESS;
}

ma_result ma_context_uninit__wasapi(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_wasapi);
    (void)pContext;

    return MA_SUCCESS;
}

ma_result ma_context_init__wasapi(const ma_context_config* pConfig, ma_context* pContext)
{
    ma_result result = MA_SUCCESS;

    ma_assert(pContext != NULL);

    (void)pContext;
    (void)pConfig;

#ifdef MA_WIN32_DESKTOP
    /*
    WASAPI is only supported in Vista SP1 and newer. The reason for SP1 and not the base version of Vista is that event-driven
    exclusive mode does not work until SP1.
    */
    {
        ma_OSVERSIONINFOEXW osvi;
        ma_zero_object(&osvi);
        osvi.dwOSVersionInfoSize = sizeof(osvi);
        osvi.dwMajorVersion = HIBYTE(_WIN32_WINNT_VISTA);
        osvi.dwMinorVersion = LOBYTE(_WIN32_WINNT_VISTA);
        osvi.wServicePackMajor = 1;
        if (VerifyVersionInfoW(&osvi, VER_MAJORVERSION | VER_MINORVERSION | VER_SERVICEPACKMAJOR, VerSetConditionMask(VerSetConditionMask(VerSetConditionMask(0, VER_MAJORVERSION, VER_GREATER_EQUAL), VER_MINORVERSION, VER_GREATER_EQUAL), VER_SERVICEPACKMAJOR, VER_GREATER_EQUAL))) {
            result = MA_SUCCESS;
        } else {
            result = MA_NO_BACKEND;
        }
    }
#endif

    if (result != MA_SUCCESS) {
        return result;
    }

    pContext->onUninit         = ma_context_uninit__wasapi;
    pContext->onDeviceIDEqual  = ma_context_is_device_id_equal__wasapi;
    pContext->onEnumDevices    = ma_context_enumerate_devices__wasapi;
    pContext->onGetDeviceInfo  = ma_context_get_device_info__wasapi;
    pContext->onDeviceInit     = ma_device_init__wasapi;
    pContext->onDeviceUninit   = ma_device_uninit__wasapi;
    pContext->onDeviceStart    = NULL; /* Not used. Started in onDeviceMainLoop. */
    pContext->onDeviceStop     = NULL; /* Not used. Stopped in onDeviceMainLoop. */
    pContext->onDeviceWrite    = NULL;
    pContext->onDeviceRead     = NULL;
    pContext->onDeviceMainLoop = ma_device_main_loop__wasapi;

    return result;
}
#endif

/******************************************************************************

DirectSound Backend

******************************************************************************/
#ifdef MA_HAS_DSOUND
/*#include <dsound.h>*/

GUID MA_GUID_IID_DirectSoundNotify = {0xb0210783, 0x89cd, 0x11d0, {0xaf, 0x08, 0x00, 0xa0, 0xc9, 0x25, 0xcd, 0x16}};

/* miniaudio only uses priority or exclusive modes. */
#define MA_DSSCL_NORMAL                 1
#define MA_DSSCL_PRIORITY               2
#define MA_DSSCL_EXCLUSIVE              3
#define MA_DSSCL_WRITEPRIMARY           4

#define MA_DSCAPS_PRIMARYMONO           0x00000001
#define MA_DSCAPS_PRIMARYSTEREO         0x00000002
#define MA_DSCAPS_PRIMARY8BIT           0x00000004
#define MA_DSCAPS_PRIMARY16BIT          0x00000008
#define MA_DSCAPS_CONTINUOUSRATE        0x00000010
#define MA_DSCAPS_EMULDRIVER            0x00000020
#define MA_DSCAPS_CERTIFIED             0x00000040
#define MA_DSCAPS_SECONDARYMONO         0x00000100
#define MA_DSCAPS_SECONDARYSTEREO       0x00000200
#define MA_DSCAPS_SECONDARY8BIT         0x00000400
#define MA_DSCAPS_SECONDARY16BIT        0x00000800

#define MA_DSBCAPS_PRIMARYBUFFER        0x00000001
#define MA_DSBCAPS_STATIC               0x00000002
#define MA_DSBCAPS_LOCHARDWARE          0x00000004
#define MA_DSBCAPS_LOCSOFTWARE          0x00000008
#define MA_DSBCAPS_CTRL3D               0x00000010
#define MA_DSBCAPS_CTRLFREQUENCY        0x00000020
#define MA_DSBCAPS_CTRLPAN              0x00000040
#define MA_DSBCAPS_CTRLVOLUME           0x00000080
#define MA_DSBCAPS_CTRLPOSITIONNOTIFY   0x00000100
#define MA_DSBCAPS_CTRLFX               0x00000200
#define MA_DSBCAPS_STICKYFOCUS          0x00004000
#define MA_DSBCAPS_GLOBALFOCUS          0x00008000
#define MA_DSBCAPS_GETCURRENTPOSITION2  0x00010000
#define MA_DSBCAPS_MUTE3DATMAXDISTANCE  0x00020000
#define MA_DSBCAPS_LOCDEFER             0x00040000
#define MA_DSBCAPS_TRUEPLAYPOSITION     0x00080000

#define MA_DSBPLAY_LOOPING              0x00000001
#define MA_DSBPLAY_LOCHARDWARE          0x00000002
#define MA_DSBPLAY_LOCSOFTWARE          0x00000004
#define MA_DSBPLAY_TERMINATEBY_TIME     0x00000008
#define MA_DSBPLAY_TERMINATEBY_DISTANCE 0x00000010
#define MA_DSBPLAY_TERMINATEBY_PRIORITY 0x00000020

#define MA_DSCBSTART_LOOPING            0x00000001

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwReserved;
    WAVEFORMATEX* lpwfxFormat;
    GUID guid3DAlgorithm;
} MA_DSBUFFERDESC;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwReserved;
    WAVEFORMATEX* lpwfxFormat;
    DWORD dwFXCount;
    void* lpDSCFXDesc;  /* <-- miniaudio doesn't use this, so set to void*. */
} MA_DSCBUFFERDESC;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwMinSecondarySampleRate;
    DWORD dwMaxSecondarySampleRate;
    DWORD dwPrimaryBuffers;
    DWORD dwMaxHwMixingAllBuffers;
    DWORD dwMaxHwMixingStaticBuffers;
    DWORD dwMaxHwMixingStreamingBuffers;
    DWORD dwFreeHwMixingAllBuffers;
    DWORD dwFreeHwMixingStaticBuffers;
    DWORD dwFreeHwMixingStreamingBuffers;
    DWORD dwMaxHw3DAllBuffers;
    DWORD dwMaxHw3DStaticBuffers;
    DWORD dwMaxHw3DStreamingBuffers;
    DWORD dwFreeHw3DAllBuffers;
    DWORD dwFreeHw3DStaticBuffers;
    DWORD dwFreeHw3DStreamingBuffers;
    DWORD dwTotalHwMemBytes;
    DWORD dwFreeHwMemBytes;
    DWORD dwMaxContigFreeHwMemBytes;
    DWORD dwUnlockTransferRateHwBuffers;
    DWORD dwPlayCpuOverheadSwBuffers;
    DWORD dwReserved1;
    DWORD dwReserved2;
} MA_DSCAPS;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwUnlockTransferRate;
    DWORD dwPlayCpuOverhead;
} MA_DSBCAPS;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwFormats;
    DWORD dwChannels;
} MA_DSCCAPS;

typedef struct
{
    DWORD dwSize;
    DWORD dwFlags;
    DWORD dwBufferBytes;
    DWORD dwReserved;
} MA_DSCBCAPS;

typedef struct
{
    DWORD  dwOffset;
    HANDLE hEventNotify;
} MA_DSBPOSITIONNOTIFY;

typedef struct ma_IDirectSound              ma_IDirectSound;
typedef struct ma_IDirectSoundBuffer        ma_IDirectSoundBuffer;
typedef struct ma_IDirectSoundCapture       ma_IDirectSoundCapture;
typedef struct ma_IDirectSoundCaptureBuffer ma_IDirectSoundCaptureBuffer;
typedef struct ma_IDirectSoundNotify        ma_IDirectSoundNotify;


/*
COM objects. The way these work is that you have a vtable (a list of function pointers, kind of
like how C++ works internally), and then you have a structure with a single member, which is a
pointer to the vtable. The vtable is where the methods of the object are defined. Methods need
to be in a specific order, and parent classes need to have their methods declared first.
*/

/* IDirectSound */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IDirectSound* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IDirectSound* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IDirectSound* pThis);

    /* IDirectSound */
    HRESULT (STDMETHODCALLTYPE * CreateSoundBuffer)   (ma_IDirectSound* pThis, const MA_DSBUFFERDESC* pDSBufferDesc, ma_IDirectSoundBuffer** ppDSBuffer, void* pUnkOuter);
    HRESULT (STDMETHODCALLTYPE * GetCaps)             (ma_IDirectSound* pThis, MA_DSCAPS* pDSCaps);
    HRESULT (STDMETHODCALLTYPE * DuplicateSoundBuffer)(ma_IDirectSound* pThis, ma_IDirectSoundBuffer* pDSBufferOriginal, ma_IDirectSoundBuffer** ppDSBufferDuplicate);
    HRESULT (STDMETHODCALLTYPE * SetCooperativeLevel) (ma_IDirectSound* pThis, HWND hwnd, DWORD dwLevel);
    HRESULT (STDMETHODCALLTYPE * Compact)             (ma_IDirectSound* pThis);
    HRESULT (STDMETHODCALLTYPE * GetSpeakerConfig)    (ma_IDirectSound* pThis, DWORD* pSpeakerConfig);
    HRESULT (STDMETHODCALLTYPE * SetSpeakerConfig)    (ma_IDirectSound* pThis, DWORD dwSpeakerConfig);
    HRESULT (STDMETHODCALLTYPE * Initialize)          (ma_IDirectSound* pThis, const GUID* pGuidDevice);
} ma_IDirectSoundVtbl;
struct ma_IDirectSound
{
    ma_IDirectSoundVtbl* lpVtbl;
};
HRESULT ma_IDirectSound_QueryInterface(ma_IDirectSound* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IDirectSound_AddRef(ma_IDirectSound* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IDirectSound_Release(ma_IDirectSound* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IDirectSound_CreateSoundBuffer(ma_IDirectSound* pThis, const MA_DSBUFFERDESC* pDSBufferDesc, ma_IDirectSoundBuffer** ppDSBuffer, void* pUnkOuter) { return pThis->lpVtbl->CreateSoundBuffer(pThis, pDSBufferDesc, ppDSBuffer, pUnkOuter); }
HRESULT ma_IDirectSound_GetCaps(ma_IDirectSound* pThis, MA_DSCAPS* pDSCaps)                           { return pThis->lpVtbl->GetCaps(pThis, pDSCaps); }
HRESULT ma_IDirectSound_DuplicateSoundBuffer(ma_IDirectSound* pThis, ma_IDirectSoundBuffer* pDSBufferOriginal, ma_IDirectSoundBuffer** ppDSBufferDuplicate) { return pThis->lpVtbl->DuplicateSoundBuffer(pThis, pDSBufferOriginal, ppDSBufferDuplicate); }
HRESULT ma_IDirectSound_SetCooperativeLevel(ma_IDirectSound* pThis, HWND hwnd, DWORD dwLevel)          { return pThis->lpVtbl->SetCooperativeLevel(pThis, hwnd, dwLevel); }
HRESULT ma_IDirectSound_Compact(ma_IDirectSound* pThis)                                                { return pThis->lpVtbl->Compact(pThis); }
HRESULT ma_IDirectSound_GetSpeakerConfig(ma_IDirectSound* pThis, DWORD* pSpeakerConfig)                { return pThis->lpVtbl->GetSpeakerConfig(pThis, pSpeakerConfig); }
HRESULT ma_IDirectSound_SetSpeakerConfig(ma_IDirectSound* pThis, DWORD dwSpeakerConfig)                { return pThis->lpVtbl->SetSpeakerConfig(pThis, dwSpeakerConfig); }
HRESULT ma_IDirectSound_Initialize(ma_IDirectSound* pThis, const GUID* pGuidDevice)                    { return pThis->lpVtbl->Initialize(pThis, pGuidDevice); }


/* IDirectSoundBuffer */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IDirectSoundBuffer* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IDirectSoundBuffer* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IDirectSoundBuffer* pThis);

    /* IDirectSoundBuffer */
    HRESULT (STDMETHODCALLTYPE * GetCaps)           (ma_IDirectSoundBuffer* pThis, MA_DSBCAPS* pDSBufferCaps);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPosition)(ma_IDirectSoundBuffer* pThis, DWORD* pCurrentPlayCursor, DWORD* pCurrentWriteCursor);
    HRESULT (STDMETHODCALLTYPE * GetFormat)         (ma_IDirectSoundBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten);
    HRESULT (STDMETHODCALLTYPE * GetVolume)         (ma_IDirectSoundBuffer* pThis, LONG* pVolume);
    HRESULT (STDMETHODCALLTYPE * GetPan)            (ma_IDirectSoundBuffer* pThis, LONG* pPan);
    HRESULT (STDMETHODCALLTYPE * GetFrequency)      (ma_IDirectSoundBuffer* pThis, DWORD* pFrequency);
    HRESULT (STDMETHODCALLTYPE * GetStatus)         (ma_IDirectSoundBuffer* pThis, DWORD* pStatus);
    HRESULT (STDMETHODCALLTYPE * Initialize)        (ma_IDirectSoundBuffer* pThis, ma_IDirectSound* pDirectSound, const MA_DSBUFFERDESC* pDSBufferDesc);
    HRESULT (STDMETHODCALLTYPE * Lock)              (ma_IDirectSoundBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * Play)              (ma_IDirectSoundBuffer* pThis, DWORD dwReserved1, DWORD dwPriority, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * SetCurrentPosition)(ma_IDirectSoundBuffer* pThis, DWORD dwNewPosition);
    HRESULT (STDMETHODCALLTYPE * SetFormat)         (ma_IDirectSoundBuffer* pThis, const WAVEFORMATEX* pFormat);
    HRESULT (STDMETHODCALLTYPE * SetVolume)         (ma_IDirectSoundBuffer* pThis, LONG volume);
    HRESULT (STDMETHODCALLTYPE * SetPan)            (ma_IDirectSoundBuffer* pThis, LONG pan);
    HRESULT (STDMETHODCALLTYPE * SetFrequency)      (ma_IDirectSoundBuffer* pThis, DWORD dwFrequency);
    HRESULT (STDMETHODCALLTYPE * Stop)              (ma_IDirectSoundBuffer* pThis);
    HRESULT (STDMETHODCALLTYPE * Unlock)            (ma_IDirectSoundBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2);
    HRESULT (STDMETHODCALLTYPE * Restore)           (ma_IDirectSoundBuffer* pThis);
} ma_IDirectSoundBufferVtbl;
struct ma_IDirectSoundBuffer
{
    ma_IDirectSoundBufferVtbl* lpVtbl;
};
HRESULT ma_IDirectSoundBuffer_QueryInterface(ma_IDirectSoundBuffer* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IDirectSoundBuffer_AddRef(ma_IDirectSoundBuffer* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IDirectSoundBuffer_Release(ma_IDirectSoundBuffer* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IDirectSoundBuffer_GetCaps(ma_IDirectSoundBuffer* pThis, MA_DSBCAPS* pDSBufferCaps)                    { return pThis->lpVtbl->GetCaps(pThis, pDSBufferCaps); }
HRESULT ma_IDirectSoundBuffer_GetCurrentPosition(ma_IDirectSoundBuffer* pThis, DWORD* pCurrentPlayCursor, DWORD* pCurrentWriteCursor) { return pThis->lpVtbl->GetCurrentPosition(pThis, pCurrentPlayCursor, pCurrentWriteCursor); }
HRESULT ma_IDirectSoundBuffer_GetFormat(ma_IDirectSoundBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten) { return pThis->lpVtbl->GetFormat(pThis, pFormat, dwSizeAllocated, pSizeWritten); }
HRESULT ma_IDirectSoundBuffer_GetVolume(ma_IDirectSoundBuffer* pThis, LONG* pVolume)                               { return pThis->lpVtbl->GetVolume(pThis, pVolume); }
HRESULT ma_IDirectSoundBuffer_GetPan(ma_IDirectSoundBuffer* pThis, LONG* pPan)                                     { return pThis->lpVtbl->GetPan(pThis, pPan); }
HRESULT ma_IDirectSoundBuffer_GetFrequency(ma_IDirectSoundBuffer* pThis, DWORD* pFrequency)                        { return pThis->lpVtbl->GetFrequency(pThis, pFrequency); }
HRESULT ma_IDirectSoundBuffer_GetStatus(ma_IDirectSoundBuffer* pThis, DWORD* pStatus)                              { return pThis->lpVtbl->GetStatus(pThis, pStatus); }
HRESULT ma_IDirectSoundBuffer_Initialize(ma_IDirectSoundBuffer* pThis, ma_IDirectSound* pDirectSound, const MA_DSBUFFERDESC* pDSBufferDesc) { return pThis->lpVtbl->Initialize(pThis, pDirectSound, pDSBufferDesc); }
HRESULT ma_IDirectSoundBuffer_Lock(ma_IDirectSoundBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags) { return pThis->lpVtbl->Lock(pThis, dwOffset, dwBytes, ppAudioPtr1, pAudioBytes1, ppAudioPtr2, pAudioBytes2, dwFlags); }
HRESULT ma_IDirectSoundBuffer_Play(ma_IDirectSoundBuffer* pThis, DWORD dwReserved1, DWORD dwPriority, DWORD dwFlags) { return pThis->lpVtbl->Play(pThis, dwReserved1, dwPriority, dwFlags); }
HRESULT ma_IDirectSoundBuffer_SetCurrentPosition(ma_IDirectSoundBuffer* pThis, DWORD dwNewPosition)                { return pThis->lpVtbl->SetCurrentPosition(pThis, dwNewPosition); }
HRESULT ma_IDirectSoundBuffer_SetFormat(ma_IDirectSoundBuffer* pThis, const WAVEFORMATEX* pFormat)                 { return pThis->lpVtbl->SetFormat(pThis, pFormat); }
HRESULT ma_IDirectSoundBuffer_SetVolume(ma_IDirectSoundBuffer* pThis, LONG volume)                                 { return pThis->lpVtbl->SetVolume(pThis, volume); }
HRESULT ma_IDirectSoundBuffer_SetPan(ma_IDirectSoundBuffer* pThis, LONG pan)                                       { return pThis->lpVtbl->SetPan(pThis, pan); }
HRESULT ma_IDirectSoundBuffer_SetFrequency(ma_IDirectSoundBuffer* pThis, DWORD dwFrequency)                        { return pThis->lpVtbl->SetFrequency(pThis, dwFrequency); }
HRESULT ma_IDirectSoundBuffer_Stop(ma_IDirectSoundBuffer* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
HRESULT ma_IDirectSoundBuffer_Unlock(ma_IDirectSoundBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2) { return pThis->lpVtbl->Unlock(pThis, pAudioPtr1, dwAudioBytes1, pAudioPtr2, dwAudioBytes2); }
HRESULT ma_IDirectSoundBuffer_Restore(ma_IDirectSoundBuffer* pThis)                                                { return pThis->lpVtbl->Restore(pThis); }


/* IDirectSoundCapture */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IDirectSoundCapture* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IDirectSoundCapture* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IDirectSoundCapture* pThis);

    /* IDirectSoundCapture */
    HRESULT (STDMETHODCALLTYPE * CreateCaptureBuffer)(ma_IDirectSoundCapture* pThis, const MA_DSCBUFFERDESC* pDSCBufferDesc, ma_IDirectSoundCaptureBuffer** ppDSCBuffer, void* pUnkOuter);
    HRESULT (STDMETHODCALLTYPE * GetCaps)            (ma_IDirectSoundCapture* pThis, MA_DSCCAPS* pDSCCaps);
    HRESULT (STDMETHODCALLTYPE * Initialize)         (ma_IDirectSoundCapture* pThis, const GUID* pGuidDevice);
} ma_IDirectSoundCaptureVtbl;
struct ma_IDirectSoundCapture
{
    ma_IDirectSoundCaptureVtbl* lpVtbl;
};
HRESULT ma_IDirectSoundCapture_QueryInterface(ma_IDirectSoundCapture* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IDirectSoundCapture_AddRef(ma_IDirectSoundCapture* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IDirectSoundCapture_Release(ma_IDirectSoundCapture* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IDirectSoundCapture_CreateCaptureBuffer(ma_IDirectSoundCapture* pThis, const MA_DSCBUFFERDESC* pDSCBufferDesc, ma_IDirectSoundCaptureBuffer** ppDSCBuffer, void* pUnkOuter) { return pThis->lpVtbl->CreateCaptureBuffer(pThis, pDSCBufferDesc, ppDSCBuffer, pUnkOuter); }
HRESULT ma_IDirectSoundCapture_GetCaps            (ma_IDirectSoundCapture* pThis, MA_DSCCAPS* pDSCCaps)             { return pThis->lpVtbl->GetCaps(pThis, pDSCCaps); }
HRESULT ma_IDirectSoundCapture_Initialize         (ma_IDirectSoundCapture* pThis, const GUID* pGuidDevice)           { return pThis->lpVtbl->Initialize(pThis, pGuidDevice); }


/* IDirectSoundCaptureBuffer */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IDirectSoundCaptureBuffer* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IDirectSoundCaptureBuffer* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IDirectSoundCaptureBuffer* pThis);

    /* IDirectSoundCaptureBuffer */
    HRESULT (STDMETHODCALLTYPE * GetCaps)           (ma_IDirectSoundCaptureBuffer* pThis, MA_DSCBCAPS* pDSCBCaps);
    HRESULT (STDMETHODCALLTYPE * GetCurrentPosition)(ma_IDirectSoundCaptureBuffer* pThis, DWORD* pCapturePosition, DWORD* pReadPosition);
    HRESULT (STDMETHODCALLTYPE * GetFormat)         (ma_IDirectSoundCaptureBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten);
    HRESULT (STDMETHODCALLTYPE * GetStatus)         (ma_IDirectSoundCaptureBuffer* pThis, DWORD* pStatus);
    HRESULT (STDMETHODCALLTYPE * Initialize)        (ma_IDirectSoundCaptureBuffer* pThis, ma_IDirectSoundCapture* pDirectSoundCapture, const MA_DSCBUFFERDESC* pDSCBufferDesc);
    HRESULT (STDMETHODCALLTYPE * Lock)              (ma_IDirectSoundCaptureBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * Start)             (ma_IDirectSoundCaptureBuffer* pThis, DWORD dwFlags);
    HRESULT (STDMETHODCALLTYPE * Stop)              (ma_IDirectSoundCaptureBuffer* pThis);
    HRESULT (STDMETHODCALLTYPE * Unlock)            (ma_IDirectSoundCaptureBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2);
} ma_IDirectSoundCaptureBufferVtbl;
struct ma_IDirectSoundCaptureBuffer
{
    ma_IDirectSoundCaptureBufferVtbl* lpVtbl;
};
HRESULT ma_IDirectSoundCaptureBuffer_QueryInterface(ma_IDirectSoundCaptureBuffer* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IDirectSoundCaptureBuffer_AddRef(ma_IDirectSoundCaptureBuffer* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IDirectSoundCaptureBuffer_Release(ma_IDirectSoundCaptureBuffer* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IDirectSoundCaptureBuffer_GetCaps(ma_IDirectSoundCaptureBuffer* pThis, MA_DSCBCAPS* pDSCBCaps)                       { return pThis->lpVtbl->GetCaps(pThis, pDSCBCaps); }
HRESULT ma_IDirectSoundCaptureBuffer_GetCurrentPosition(ma_IDirectSoundCaptureBuffer* pThis, DWORD* pCapturePosition, DWORD* pReadPosition) { return pThis->lpVtbl->GetCurrentPosition(pThis, pCapturePosition, pReadPosition); }
HRESULT ma_IDirectSoundCaptureBuffer_GetFormat(ma_IDirectSoundCaptureBuffer* pThis, WAVEFORMATEX* pFormat, DWORD dwSizeAllocated, DWORD* pSizeWritten) { return pThis->lpVtbl->GetFormat(pThis, pFormat, dwSizeAllocated, pSizeWritten); }
HRESULT ma_IDirectSoundCaptureBuffer_GetStatus(ma_IDirectSoundCaptureBuffer* pThis, DWORD* pStatus)                              { return pThis->lpVtbl->GetStatus(pThis, pStatus); }
HRESULT ma_IDirectSoundCaptureBuffer_Initialize(ma_IDirectSoundCaptureBuffer* pThis, ma_IDirectSoundCapture* pDirectSoundCapture, const MA_DSCBUFFERDESC* pDSCBufferDesc) { return pThis->lpVtbl->Initialize(pThis, pDirectSoundCapture, pDSCBufferDesc); }
HRESULT ma_IDirectSoundCaptureBuffer_Lock(ma_IDirectSoundCaptureBuffer* pThis, DWORD dwOffset, DWORD dwBytes, void** ppAudioPtr1, DWORD* pAudioBytes1, void** ppAudioPtr2, DWORD* pAudioBytes2, DWORD dwFlags) { return pThis->lpVtbl->Lock(pThis, dwOffset, dwBytes, ppAudioPtr1, pAudioBytes1, ppAudioPtr2, pAudioBytes2, dwFlags); }
HRESULT ma_IDirectSoundCaptureBuffer_Start(ma_IDirectSoundCaptureBuffer* pThis, DWORD dwFlags)                                   { return pThis->lpVtbl->Start(pThis, dwFlags); }
HRESULT ma_IDirectSoundCaptureBuffer_Stop(ma_IDirectSoundCaptureBuffer* pThis)                                                   { return pThis->lpVtbl->Stop(pThis); }
HRESULT ma_IDirectSoundCaptureBuffer_Unlock(ma_IDirectSoundCaptureBuffer* pThis, void* pAudioPtr1, DWORD dwAudioBytes1, void* pAudioPtr2, DWORD dwAudioBytes2) { return pThis->lpVtbl->Unlock(pThis, pAudioPtr1, dwAudioBytes1, pAudioPtr2, dwAudioBytes2); }


/* IDirectSoundNotify */
typedef struct
{
    /* IUnknown */
    HRESULT (STDMETHODCALLTYPE * QueryInterface)(ma_IDirectSoundNotify* pThis, const IID* const riid, void** ppObject);
    ULONG   (STDMETHODCALLTYPE * AddRef)        (ma_IDirectSoundNotify* pThis);
    ULONG   (STDMETHODCALLTYPE * Release)       (ma_IDirectSoundNotify* pThis);

    /* IDirectSoundNotify */
    HRESULT (STDMETHODCALLTYPE * SetNotificationPositions)(ma_IDirectSoundNotify* pThis, DWORD dwPositionNotifies, const MA_DSBPOSITIONNOTIFY* pPositionNotifies);
} ma_IDirectSoundNotifyVtbl;
struct ma_IDirectSoundNotify
{
    ma_IDirectSoundNotifyVtbl* lpVtbl;
};
HRESULT ma_IDirectSoundNotify_QueryInterface(ma_IDirectSoundNotify* pThis, const IID* const riid, void** ppObject) { return pThis->lpVtbl->QueryInterface(pThis, riid, ppObject); }
ULONG   ma_IDirectSoundNotify_AddRef(ma_IDirectSoundNotify* pThis)                                                 { return pThis->lpVtbl->AddRef(pThis); }
ULONG   ma_IDirectSoundNotify_Release(ma_IDirectSoundNotify* pThis)                                                { return pThis->lpVtbl->Release(pThis); }
HRESULT ma_IDirectSoundNotify_SetNotificationPositions(ma_IDirectSoundNotify* pThis, DWORD dwPositionNotifies, const MA_DSBPOSITIONNOTIFY* pPositionNotifies) { return pThis->lpVtbl->SetNotificationPositions(pThis, dwPositionNotifies, pPositionNotifies); }


typedef BOOL    (CALLBACK * ma_DSEnumCallbackAProc)             (LPGUID pDeviceGUID, LPCSTR pDeviceDescription, LPCSTR pModule, LPVOID pContext);
typedef HRESULT (WINAPI   * ma_DirectSoundCreateProc)           (const GUID* pcGuidDevice, ma_IDirectSound** ppDS8, LPUNKNOWN pUnkOuter);
typedef HRESULT (WINAPI   * ma_DirectSoundEnumerateAProc)       (ma_DSEnumCallbackAProc pDSEnumCallback, LPVOID pContext);
typedef HRESULT (WINAPI   * ma_DirectSoundCaptureCreateProc)    (const GUID* pcGuidDevice, ma_IDirectSoundCapture** ppDSC8, LPUNKNOWN pUnkOuter);
typedef HRESULT (WINAPI   * ma_DirectSoundCaptureEnumerateAProc)(ma_DSEnumCallbackAProc pDSEnumCallback, LPVOID pContext);


/*
Retrieves the channel count and channel map for the given speaker configuration. If the speaker configuration is unknown,
the channel count and channel map will be left unmodified.
*/
void ma_get_channels_from_speaker_config__dsound(DWORD speakerConfig, WORD* pChannelsOut, DWORD* pChannelMapOut)
{
    WORD  channels;
    DWORD channelMap;

    channels = 0;
    if (pChannelsOut != NULL) {
        channels = *pChannelsOut;
    }

    channelMap = 0;
    if (pChannelMapOut != NULL) {
        channelMap = *pChannelMapOut;
    }

    /*
    The speaker configuration is a combination of speaker config and speaker geometry. The lower 8 bits is what we care about. The upper
    16 bits is for the geometry.
    */
    switch ((BYTE)(speakerConfig)) {
        case 1 /*DSSPEAKER_HEADPHONE*/:                          channels = 2; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT; break;
        case 2 /*DSSPEAKER_MONO*/:                               channels = 1; channelMap = SPEAKER_FRONT_CENTER; break;
        case 3 /*DSSPEAKER_QUAD*/:                               channels = 4; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT; break;
        case 4 /*DSSPEAKER_STEREO*/:                             channels = 2; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT; break;
        case 5 /*DSSPEAKER_SURROUND*/:                           channels = 4; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_BACK_CENTER; break;
        case 6 /*DSSPEAKER_5POINT1_BACK*/ /*DSSPEAKER_5POINT1*/: channels = 6; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT; break;
        case 7 /*DSSPEAKER_7POINT1_WIDE*/ /*DSSPEAKER_7POINT1*/: channels = 8; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT | SPEAKER_FRONT_LEFT_OF_CENTER | SPEAKER_FRONT_RIGHT_OF_CENTER; break;
        case 8 /*DSSPEAKER_7POINT1_SURROUND*/:                   channels = 8; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_BACK_LEFT | SPEAKER_BACK_RIGHT | SPEAKER_SIDE_LEFT | SPEAKER_SIDE_RIGHT; break;
        case 9 /*DSSPEAKER_5POINT1_SURROUND*/:                   channels = 6; channelMap = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT | SPEAKER_FRONT_CENTER | SPEAKER_LOW_FREQUENCY | SPEAKER_SIDE_LEFT | SPEAKER_SIDE_RIGHT; break;
        default: break;
    }

    if (pChannelsOut != NULL) {
        *pChannelsOut = channels;
    }

    if (pChannelMapOut != NULL) {
        *pChannelMapOut = channelMap;
    }
}


ma_result ma_context_create_IDirectSound__dsound(ma_context* pContext, ma_share_mode shareMode, const ma_device_id* pDeviceID, ma_IDirectSound** ppDirectSound)
{
    ma_IDirectSound* pDirectSound;
    HWND hWnd;

    ma_assert(pContext != NULL);
    ma_assert(ppDirectSound != NULL);

    *ppDirectSound = NULL;
    pDirectSound = NULL;

    if (FAILED(((ma_DirectSoundCreateProc)pContext->dsound.DirectSoundCreate)((pDeviceID == NULL) ? NULL : (const GUID*)pDeviceID->dsound, &pDirectSound, NULL))) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[DirectSound] DirectSoundCreate() failed for playback device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    /* The cooperative level must be set before doing anything else. */
    hWnd = ((MA_PFN_GetForegroundWindow)pContext->win32.GetForegroundWindow)();
    if (hWnd == NULL) {
        hWnd = ((MA_PFN_GetDesktopWindow)pContext->win32.GetDesktopWindow)();
    }
    if (FAILED(ma_IDirectSound_SetCooperativeLevel(pDirectSound, hWnd, (shareMode == ma_share_mode_exclusive) ? MA_DSSCL_EXCLUSIVE : MA_DSSCL_PRIORITY))) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_SetCooperateiveLevel() failed for playback device.", MA_SHARE_MODE_NOT_SUPPORTED);
    }

    *ppDirectSound = pDirectSound;
    return MA_SUCCESS;
}

ma_result ma_context_create_IDirectSoundCapture__dsound(ma_context* pContext, ma_share_mode shareMode, const ma_device_id* pDeviceID, ma_IDirectSoundCapture** ppDirectSoundCapture)
{
    ma_IDirectSoundCapture* pDirectSoundCapture;

    ma_assert(pContext != NULL);
    ma_assert(ppDirectSoundCapture != NULL);

    /* DirectSound does not support exclusive mode for capture. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    *ppDirectSoundCapture = NULL;
    pDirectSoundCapture = NULL;

    if (FAILED(((ma_DirectSoundCaptureCreateProc)pContext->dsound.DirectSoundCaptureCreate)((pDeviceID == NULL) ? NULL : (const GUID*)pDeviceID->dsound, &pDirectSoundCapture, NULL))) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[DirectSound] DirectSoundCaptureCreate() failed for capture device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    *ppDirectSoundCapture = pDirectSoundCapture;
    return MA_SUCCESS;
}

ma_result ma_context_get_format_info_for_IDirectSoundCapture__dsound(ma_context* pContext, ma_IDirectSoundCapture* pDirectSoundCapture, WORD* pChannels, WORD* pBitsPerSample, DWORD* pSampleRate)
{
    MA_DSCCAPS caps;
    WORD bitsPerSample;
    DWORD sampleRate;

    ma_assert(pContext != NULL);
    ma_assert(pDirectSoundCapture != NULL);

    if (pChannels) {
        *pChannels = 0;
    }
    if (pBitsPerSample) {
        *pBitsPerSample = 0;
    }
    if (pSampleRate) {
        *pSampleRate = 0;
    }

    ma_zero_object(&caps);
    caps.dwSize = sizeof(caps);
    if (FAILED(ma_IDirectSoundCapture_GetCaps(pDirectSoundCapture, &caps))) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCapture_GetCaps() failed for capture device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    if (pChannels) {
        *pChannels = (WORD)caps.dwChannels;
    }

    /* The device can support multiple formats. We just go through the different formats in order of priority and pick the first one. This the same type of system as the WinMM backend. */
    bitsPerSample = 16;
    sampleRate = 48000;

    if (caps.dwChannels == 1) {
        if ((caps.dwFormats & WAVE_FORMAT_48M16) != 0) {
            sampleRate = 48000;
        } else if ((caps.dwFormats & WAVE_FORMAT_44M16) != 0) {
            sampleRate = 44100;
        } else if ((caps.dwFormats & WAVE_FORMAT_2M16) != 0) {
            sampleRate = 22050;
        } else if ((caps.dwFormats & WAVE_FORMAT_1M16) != 0) {
            sampleRate = 11025;
        } else if ((caps.dwFormats & WAVE_FORMAT_96M16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((caps.dwFormats & WAVE_FORMAT_48M08) != 0) {
                sampleRate = 48000;
            } else if ((caps.dwFormats & WAVE_FORMAT_44M08) != 0) {
                sampleRate = 44100;
            } else if ((caps.dwFormats & WAVE_FORMAT_2M08) != 0) {
                sampleRate = 22050;
            } else if ((caps.dwFormats & WAVE_FORMAT_1M08) != 0) {
                sampleRate = 11025;
            } else if ((caps.dwFormats & WAVE_FORMAT_96M08) != 0) {
                sampleRate = 96000;
            } else {
                bitsPerSample = 16;  /* Didn't find it. Just fall back to 16-bit. */
            }
        }
    } else if (caps.dwChannels == 2) {
        if ((caps.dwFormats & WAVE_FORMAT_48S16) != 0) {
            sampleRate = 48000;
        } else if ((caps.dwFormats & WAVE_FORMAT_44S16) != 0) {
            sampleRate = 44100;
        } else if ((caps.dwFormats & WAVE_FORMAT_2S16) != 0) {
            sampleRate = 22050;
        } else if ((caps.dwFormats & WAVE_FORMAT_1S16) != 0) {
            sampleRate = 11025;
        } else if ((caps.dwFormats & WAVE_FORMAT_96S16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((caps.dwFormats & WAVE_FORMAT_48S08) != 0) {
                sampleRate = 48000;
            } else if ((caps.dwFormats & WAVE_FORMAT_44S08) != 0) {
                sampleRate = 44100;
            } else if ((caps.dwFormats & WAVE_FORMAT_2S08) != 0) {
                sampleRate = 22050;
            } else if ((caps.dwFormats & WAVE_FORMAT_1S08) != 0) {
                sampleRate = 11025;
            } else if ((caps.dwFormats & WAVE_FORMAT_96S08) != 0) {
                sampleRate = 96000;
            } else {
                bitsPerSample = 16;  /* Didn't find it. Just fall back to 16-bit. */
            }
        }
    }

    if (pBitsPerSample) {
        *pBitsPerSample = bitsPerSample;
    }
    if (pSampleRate) {
        *pSampleRate = sampleRate;
    }

    return MA_SUCCESS;
}

ma_bool32 ma_context_is_device_id_equal__dsound(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return memcmp(pID0->dsound, pID1->dsound, sizeof(pID0->dsound)) == 0;
}


typedef struct
{
    ma_context* pContext;
    ma_device_type deviceType;
    ma_enum_devices_callback_proc callback;
    void* pUserData;
    ma_bool32 terminated;
} ma_context_enumerate_devices_callback_data__dsound;

BOOL CALLBACK ma_context_enumerate_devices_callback__dsound(LPGUID lpGuid, LPCSTR lpcstrDescription, LPCSTR lpcstrModule, LPVOID lpContext)
{
    ma_context_enumerate_devices_callback_data__dsound* pData = (ma_context_enumerate_devices_callback_data__dsound*)lpContext;
    ma_device_info deviceInfo;    

    ma_zero_object(&deviceInfo);

    /* ID. */
    if (lpGuid != NULL) {
        ma_copy_memory(deviceInfo.id.dsound, lpGuid, 16);
    } else {
        ma_zero_memory(deviceInfo.id.dsound, 16);
    }

    /* Name / Description */
    ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), lpcstrDescription, (size_t)-1);


    /* Call the callback function, but make sure we stop enumerating if the callee requested so. */
    ma_assert(pData != NULL);
    pData->terminated = !pData->callback(pData->pContext, pData->deviceType, &deviceInfo, pData->pUserData);
    if (pData->terminated) {
        return FALSE;   /* Stop enumeration. */
    } else {
        return TRUE;    /* Continue enumeration. */
    }

    (void)lpcstrModule;
}

ma_result ma_context_enumerate_devices__dsound(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_context_enumerate_devices_callback_data__dsound data;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);

    data.pContext = pContext;
    data.callback = callback;
    data.pUserData = pUserData;
    data.terminated = MA_FALSE;

    /* Playback. */
    if (!data.terminated) {
        data.deviceType = ma_device_type_playback;
        ((ma_DirectSoundEnumerateAProc)pContext->dsound.DirectSoundEnumerateA)(ma_context_enumerate_devices_callback__dsound, &data);
    }

    /* Capture. */
    if (!data.terminated) {
        data.deviceType = ma_device_type_capture;
        ((ma_DirectSoundCaptureEnumerateAProc)pContext->dsound.DirectSoundCaptureEnumerateA)(ma_context_enumerate_devices_callback__dsound, &data);
    }

    return MA_SUCCESS;
}


typedef struct
{
    const ma_device_id* pDeviceID;
    ma_device_info* pDeviceInfo;
    ma_bool32 found;
} ma_context_get_device_info_callback_data__dsound;

BOOL CALLBACK ma_context_get_device_info_callback__dsound(LPGUID lpGuid, LPCSTR lpcstrDescription, LPCSTR lpcstrModule, LPVOID lpContext)
{
    ma_context_get_device_info_callback_data__dsound* pData = (ma_context_get_device_info_callback_data__dsound*)lpContext;
    ma_assert(pData != NULL);

    if ((pData->pDeviceID == NULL || ma_is_guid_equal(pData->pDeviceID->dsound, &MA_GUID_NULL)) && (lpGuid == NULL || ma_is_guid_equal(lpGuid, &MA_GUID_NULL))) {
        /* Default device. */
        ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), lpcstrDescription, (size_t)-1);
        pData->found = MA_TRUE;
        return FALSE;   /* Stop enumeration. */
    } else {
        /* Not the default device. */
        if (lpGuid != NULL) {
            if (memcmp(pData->pDeviceID->dsound, lpGuid, sizeof(pData->pDeviceID->dsound)) == 0) {
                ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), lpcstrDescription, (size_t)-1);
                pData->found = MA_TRUE;
                return FALSE;   /* Stop enumeration. */
            }
        }
    }

    (void)lpcstrModule;
    return TRUE;
}

ma_result ma_context_get_device_info__dsound(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    /* Exclusive mode and capture not supported with DirectSound. */
    if (deviceType == ma_device_type_capture && shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    if (pDeviceID != NULL) {
        ma_context_get_device_info_callback_data__dsound data;

        /* ID. */
        ma_copy_memory(pDeviceInfo->id.dsound, pDeviceID->dsound, 16);

        /* Name / Description. This is retrieved by enumerating over each device until we find that one that matches the input ID. */
        data.pDeviceID = pDeviceID;
        data.pDeviceInfo = pDeviceInfo;
        data.found = MA_FALSE;
        if (deviceType == ma_device_type_playback) {
            ((ma_DirectSoundEnumerateAProc)pContext->dsound.DirectSoundEnumerateA)(ma_context_get_device_info_callback__dsound, &data);
        } else {
            ((ma_DirectSoundCaptureEnumerateAProc)pContext->dsound.DirectSoundCaptureEnumerateA)(ma_context_get_device_info_callback__dsound, &data);
        }

        if (!data.found) {
            return MA_NO_DEVICE;
        }
    } else {
        /* I don't think there's a way to get the name of the default device with DirectSound. In this case we just need to use defaults. */

        /* ID */
        ma_zero_memory(pDeviceInfo->id.dsound, 16);

        /* Name / Description */
        if (deviceType == ma_device_type_playback) {
            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        } else {
            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        }
    }

    /* Retrieving detailed information is slightly different depending on the device type. */
    if (deviceType == ma_device_type_playback) {
        /* Playback. */
        ma_IDirectSound* pDirectSound;
        ma_result result;
        MA_DSCAPS caps;
        ma_uint32 iFormat;

        result = ma_context_create_IDirectSound__dsound(pContext, shareMode, pDeviceID, &pDirectSound);
        if (result != MA_SUCCESS) {
            return result;
        }

        ma_zero_object(&caps);
        caps.dwSize = sizeof(caps);
        if (FAILED(ma_IDirectSound_GetCaps(pDirectSound, &caps))) {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_GetCaps() failed for playback device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if ((caps.dwFlags & MA_DSCAPS_PRIMARYSTEREO) != 0) {
            /* It supports at least stereo, but could support more. */
            WORD channels = 2;

            /* Look at the speaker configuration to get a better idea on the channel count. */
            DWORD speakerConfig;
            if (SUCCEEDED(ma_IDirectSound_GetSpeakerConfig(pDirectSound, &speakerConfig))) {
                ma_get_channels_from_speaker_config__dsound(speakerConfig, &channels, NULL);
            }

            pDeviceInfo->minChannels = channels;
            pDeviceInfo->maxChannels = channels;
        } else {
            /* It does not support stereo, which means we are stuck with mono. */
            pDeviceInfo->minChannels = 1;
            pDeviceInfo->maxChannels = 1;
        }

        /* Sample rate. */
        if ((caps.dwFlags & MA_DSCAPS_CONTINUOUSRATE) != 0) {
            pDeviceInfo->minSampleRate = caps.dwMinSecondarySampleRate;
            pDeviceInfo->maxSampleRate = caps.dwMaxSecondarySampleRate;

            /*
            On my machine the min and max sample rates can return 100 and 200000 respectively. I'd rather these be within
            the range of our standard sample rates so I'm clamping.
            */
            if (caps.dwMinSecondarySampleRate < MA_MIN_SAMPLE_RATE && caps.dwMaxSecondarySampleRate >= MA_MIN_SAMPLE_RATE) {
                pDeviceInfo->minSampleRate = MA_MIN_SAMPLE_RATE;
            }
            if (caps.dwMaxSecondarySampleRate > MA_MAX_SAMPLE_RATE && caps.dwMinSecondarySampleRate <= MA_MAX_SAMPLE_RATE) {
                pDeviceInfo->maxSampleRate = MA_MAX_SAMPLE_RATE;
            }
        } else {
            /* Only supports a single sample rate. Set both min an max to the same thing. Do not clamp within the standard rates. */
            pDeviceInfo->minSampleRate = caps.dwMaxSecondarySampleRate;
            pDeviceInfo->maxSampleRate = caps.dwMaxSecondarySampleRate;
        }

        /* DirectSound can support all formats. */
        pDeviceInfo->formatCount = ma_format_count - 1;    /* Minus one because we don't want to include ma_format_unknown. */
        for (iFormat = 0; iFormat < pDeviceInfo->formatCount; ++iFormat) {
            pDeviceInfo->formats[iFormat] = (ma_format)(iFormat + 1);  /* +1 to skip over ma_format_unknown. */
        }

        ma_IDirectSound_Release(pDirectSound);
    } else {
        /*
        Capture. This is a little different to playback due to the say the supported formats are reported. Technically capture
        devices can support a number of different formats, but for simplicity and consistency with ma_device_init() I'm just
        reporting the best format.
        */
        ma_IDirectSoundCapture* pDirectSoundCapture;
        ma_result result;
        WORD channels;
        WORD bitsPerSample;
        DWORD sampleRate;

        result = ma_context_create_IDirectSoundCapture__dsound(pContext, shareMode, pDeviceID, &pDirectSoundCapture);
        if (result != MA_SUCCESS) {
            return result;
        }

        result = ma_context_get_format_info_for_IDirectSoundCapture__dsound(pContext, pDirectSoundCapture, &channels, &bitsPerSample, &sampleRate);
        if (result != MA_SUCCESS) {
            ma_IDirectSoundCapture_Release(pDirectSoundCapture);
            return result;
        }

        pDeviceInfo->minChannels = channels;
        pDeviceInfo->maxChannels = channels;
        pDeviceInfo->minSampleRate = sampleRate;
        pDeviceInfo->maxSampleRate = sampleRate;
        pDeviceInfo->formatCount = 1;
        if (bitsPerSample == 8) {
            pDeviceInfo->formats[0] = ma_format_u8;
        } else if (bitsPerSample == 16) {
            pDeviceInfo->formats[0] = ma_format_s16;
        } else if (bitsPerSample == 24) {
            pDeviceInfo->formats[0] = ma_format_s24;
        } else if (bitsPerSample == 32) {
            pDeviceInfo->formats[0] = ma_format_s32;
        } else {
            ma_IDirectSoundCapture_Release(pDirectSoundCapture);
            return MA_FORMAT_NOT_SUPPORTED;
        }

        ma_IDirectSoundCapture_Release(pDirectSoundCapture);
    }

    return MA_SUCCESS;
}


typedef struct
{
    ma_uint32 deviceCount;
    ma_uint32 infoCount;
    ma_device_info* pInfo;
} ma_device_enum_data__dsound;

BOOL CALLBACK ma_enum_devices_callback__dsound(LPGUID lpGuid, LPCSTR lpcstrDescription, LPCSTR lpcstrModule, LPVOID lpContext)
{
    ma_device_enum_data__dsound* pData = (ma_device_enum_data__dsound*)lpContext;
    ma_assert(pData != NULL);

    if (pData->pInfo != NULL) {
        if (pData->infoCount > 0) {
            ma_zero_object(pData->pInfo);
            ma_strncpy_s(pData->pInfo->name, sizeof(pData->pInfo->name), lpcstrDescription, (size_t)-1);

            if (lpGuid != NULL) {
                ma_copy_memory(pData->pInfo->id.dsound, lpGuid, 16);
            } else {
                ma_zero_memory(pData->pInfo->id.dsound, 16);
            }

            pData->pInfo += 1;
            pData->infoCount -= 1;
            pData->deviceCount += 1;
        }
    } else {
        pData->deviceCount += 1;
    }

    (void)lpcstrModule;
    return TRUE;
}

void ma_device_uninit__dsound(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->dsound.pCaptureBuffer != NULL) {
        ma_IDirectSoundCaptureBuffer_Release((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer);
    }
    if (pDevice->dsound.pCapture != NULL) {
        ma_IDirectSoundCapture_Release((ma_IDirectSoundCapture*)pDevice->dsound.pCapture);
    }

    if (pDevice->dsound.pPlaybackBuffer != NULL) {
        ma_IDirectSoundBuffer_Release((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer);
    }
    if (pDevice->dsound.pPlaybackPrimaryBuffer != NULL) {
        ma_IDirectSoundBuffer_Release((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackPrimaryBuffer);
    }
    if (pDevice->dsound.pPlayback != NULL) {
        ma_IDirectSound_Release((ma_IDirectSound*)pDevice->dsound.pPlayback);
    }
}

ma_result ma_config_to_WAVEFORMATEXTENSIBLE(ma_format format, ma_uint32 channels, ma_uint32 sampleRate, const ma_channel* pChannelMap, WAVEFORMATEXTENSIBLE* pWF)
{
    GUID subformat;

    switch (format)
    {
        case ma_format_u8:
        case ma_format_s16:
        case ma_format_s24:
        /*case ma_format_s24_32:*/
        case ma_format_s32:
        {
            subformat = MA_GUID_KSDATAFORMAT_SUBTYPE_PCM;
        } break;

        case ma_format_f32:
        {
            subformat = MA_GUID_KSDATAFORMAT_SUBTYPE_IEEE_FLOAT;
        } break;

        default:
        return MA_FORMAT_NOT_SUPPORTED;
    }

    ma_zero_object(pWF);
    pWF->Format.cbSize               = sizeof(*pWF);
    pWF->Format.wFormatTag           = WAVE_FORMAT_EXTENSIBLE;
    pWF->Format.nChannels            = (WORD)channels;
    pWF->Format.nSamplesPerSec       = (DWORD)sampleRate;
    pWF->Format.wBitsPerSample       = (WORD)ma_get_bytes_per_sample(format)*8;
    pWF->Format.nBlockAlign          = (pWF->Format.nChannels * pWF->Format.wBitsPerSample) / 8;
    pWF->Format.nAvgBytesPerSec      = pWF->Format.nBlockAlign * pWF->Format.nSamplesPerSec;
    pWF->Samples.wValidBitsPerSample = pWF->Format.wBitsPerSample;
    pWF->dwChannelMask               = ma_channel_map_to_channel_mask__win32(pChannelMap, channels);
    pWF->SubFormat                   = subformat;

    return MA_SUCCESS;
}

ma_result ma_device_init__dsound(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result;
    ma_uint32 bufferSizeInMilliseconds;

    ma_assert(pDevice != NULL);
    ma_zero_object(&pDevice->dsound);

    bufferSizeInMilliseconds = pConfig->bufferSizeInMilliseconds;
    if (bufferSizeInMilliseconds == 0) {
        bufferSizeInMilliseconds = ma_calculate_buffer_size_in_milliseconds_from_frames(pConfig->bufferSizeInFrames, pConfig->sampleRate);
    }
    
    /* DirectSound should use a latency of about 20ms per period for low latency mode. */
    if (pDevice->usingDefaultBufferSize) {
        if (pConfig->performanceProfile == ma_performance_profile_low_latency) {
            bufferSizeInMilliseconds =  20 * pConfig->periods;
        } else {
            bufferSizeInMilliseconds = 200 * pConfig->periods;
        }
    }

    /* DirectSound breaks down with tiny buffer sizes (bad glitching and silent output). I am therefore restricting the size of the buffer to a minimum of 20 milliseconds. */
    if ((bufferSizeInMilliseconds/pConfig->periods) < 20) {
        bufferSizeInMilliseconds = pConfig->periods * 20;
    }

    /*
    Unfortunately DirectSound uses different APIs and data structures for playback and catpure devices. We need to initialize
    the capture device first because we'll want to match it's buffer size and period count on the playback side if we're using
    full-duplex mode.
    */
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        WAVEFORMATEXTENSIBLE wf;
        MA_DSCBUFFERDESC descDS;
        ma_uint32 bufferSizeInFrames;
        char rawdata[1024]; /* <-- Ugly hack to avoid a malloc() due to a crappy DirectSound API. */
        WAVEFORMATEXTENSIBLE* pActualFormat;

        result = ma_config_to_WAVEFORMATEXTENSIBLE(pConfig->capture.format, pConfig->capture.channels, pConfig->sampleRate, pConfig->capture.channelMap, &wf);
        if (result != MA_SUCCESS) {
            return result;
        }

        result = ma_context_create_IDirectSoundCapture__dsound(pContext, pConfig->capture.shareMode, pConfig->capture.pDeviceID, (ma_IDirectSoundCapture**)&pDevice->dsound.pCapture);
        if (result != MA_SUCCESS) {
            ma_device_uninit__dsound(pDevice);
            return result;
        }

        result = ma_context_get_format_info_for_IDirectSoundCapture__dsound(pContext, (ma_IDirectSoundCapture*)pDevice->dsound.pCapture, &wf.Format.nChannels, &wf.Format.wBitsPerSample, &wf.Format.nSamplesPerSec);
        if (result != MA_SUCCESS) {
            ma_device_uninit__dsound(pDevice);
            return result;
        }

        wf.Format.nBlockAlign          = (wf.Format.nChannels * wf.Format.wBitsPerSample) / 8;
        wf.Format.nAvgBytesPerSec      = wf.Format.nBlockAlign * wf.Format.nSamplesPerSec;
        wf.Samples.wValidBitsPerSample = wf.Format.wBitsPerSample;
        wf.SubFormat                   = MA_GUID_KSDATAFORMAT_SUBTYPE_PCM;

        /* The size of the buffer must be a clean multiple of the period count. */
        bufferSizeInFrames = (ma_calculate_buffer_size_in_frames_from_milliseconds(bufferSizeInMilliseconds, wf.Format.nSamplesPerSec) / pConfig->periods) * pConfig->periods;

        ma_zero_object(&descDS);
        descDS.dwSize = sizeof(descDS);
        descDS.dwFlags = 0;
        descDS.dwBufferBytes = bufferSizeInFrames * ma_get_bytes_per_frame(pDevice->capture.internalFormat, wf.Format.nChannels);
        descDS.lpwfxFormat = (WAVEFORMATEX*)&wf;
        if (FAILED(ma_IDirectSoundCapture_CreateCaptureBuffer((ma_IDirectSoundCapture*)pDevice->dsound.pCapture, &descDS, (ma_IDirectSoundCaptureBuffer**)&pDevice->dsound.pCaptureBuffer, NULL))) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCapture_CreateCaptureBuffer() failed for capture device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        /* Get the _actual_ properties of the buffer. */
        pActualFormat = (WAVEFORMATEXTENSIBLE*)rawdata;
        if (FAILED(ma_IDirectSoundCaptureBuffer_GetFormat((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, (WAVEFORMATEX*)pActualFormat, sizeof(rawdata), NULL))) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to retrieve the actual format of the capture device's buffer.", MA_FORMAT_NOT_SUPPORTED);
        }

        pDevice->capture.internalFormat = ma_format_from_WAVEFORMATEX((WAVEFORMATEX*)pActualFormat);
        pDevice->capture.internalChannels = pActualFormat->Format.nChannels;
        pDevice->capture.internalSampleRate = pActualFormat->Format.nSamplesPerSec;

        /* Get the internal channel map based on the channel mask. */
        if (pActualFormat->Format.wFormatTag == WAVE_FORMAT_EXTENSIBLE) {
            ma_channel_mask_to_channel_map__win32(pActualFormat->dwChannelMask, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);
        } else {
            ma_channel_mask_to_channel_map__win32(wf.dwChannelMask, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);
        }

        /*
        After getting the actual format the size of the buffer in frames may have actually changed. However, we want this to be as close to what the
        user has asked for as possible, so let's go ahead and release the old capture buffer and create a new one in this case.
        */
        if (bufferSizeInFrames != (descDS.dwBufferBytes / ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels))) {
            descDS.dwBufferBytes = bufferSizeInFrames * ma_get_bytes_per_frame(pDevice->capture.internalFormat, wf.Format.nChannels);
            ma_IDirectSoundCaptureBuffer_Release((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer);

            if (FAILED(ma_IDirectSoundCapture_CreateCaptureBuffer((ma_IDirectSoundCapture*)pDevice->dsound.pCapture, &descDS, (ma_IDirectSoundCaptureBuffer**)&pDevice->dsound.pCaptureBuffer, NULL))) {
                ma_device_uninit__dsound(pDevice);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Second attempt at IDirectSoundCapture_CreateCaptureBuffer() failed for capture device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
            }
        }

        /* DirectSound should give us a buffer exactly the size we asked for. */
        pDevice->capture.internalBufferSizeInFrames = bufferSizeInFrames;
        pDevice->capture.internalPeriods            = pConfig->periods;
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        WAVEFORMATEXTENSIBLE wf;
        MA_DSBUFFERDESC descDSPrimary;
        MA_DSCAPS caps;
        char rawdata[1024]; /* <-- Ugly hack to avoid a malloc() due to a crappy DirectSound API. */
        WAVEFORMATEXTENSIBLE* pActualFormat;
        ma_uint32 bufferSizeInFrames;
        MA_DSBUFFERDESC descDS;

        result = ma_config_to_WAVEFORMATEXTENSIBLE(pConfig->playback.format, pConfig->playback.channels, pConfig->sampleRate, pConfig->playback.channelMap, &wf);
        if (result != MA_SUCCESS) {
            return result;
        }

        result = ma_context_create_IDirectSound__dsound(pContext, pConfig->playback.shareMode, pConfig->playback.pDeviceID, (ma_IDirectSound**)&pDevice->dsound.pPlayback);
        if (result != MA_SUCCESS) {
            ma_device_uninit__dsound(pDevice);
            return result;
        }

        ma_zero_object(&descDSPrimary);
        descDSPrimary.dwSize  = sizeof(MA_DSBUFFERDESC);
        descDSPrimary.dwFlags = MA_DSBCAPS_PRIMARYBUFFER | MA_DSBCAPS_CTRLVOLUME;
        if (FAILED(ma_IDirectSound_CreateSoundBuffer((ma_IDirectSound*)pDevice->dsound.pPlayback, &descDSPrimary, (ma_IDirectSoundBuffer**)&pDevice->dsound.pPlaybackPrimaryBuffer, NULL))) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_CreateSoundBuffer() failed for playback device's primary buffer.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }


        /* We may want to make some adjustments to the format if we are using defaults. */
        ma_zero_object(&caps);
        caps.dwSize = sizeof(caps);
        if (FAILED(ma_IDirectSound_GetCaps((ma_IDirectSound*)pDevice->dsound.pPlayback, &caps))) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_GetCaps() failed for playback device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        if (pDevice->playback.usingDefaultChannels) {
            if ((caps.dwFlags & MA_DSCAPS_PRIMARYSTEREO) != 0) {
                DWORD speakerConfig;

                /* It supports at least stereo, but could support more. */
                wf.Format.nChannels = 2;

                /* Look at the speaker configuration to get a better idea on the channel count. */
                if (SUCCEEDED(ma_IDirectSound_GetSpeakerConfig((ma_IDirectSound*)pDevice->dsound.pPlayback, &speakerConfig))) {
                    ma_get_channels_from_speaker_config__dsound(speakerConfig, &wf.Format.nChannels, &wf.dwChannelMask);
                }
            } else {
                /* It does not support stereo, which means we are stuck with mono. */
                wf.Format.nChannels = 1;
            }
        }

        if (pDevice->usingDefaultSampleRate) {
            /* We base the sample rate on the values returned by GetCaps(). */
            if ((caps.dwFlags & MA_DSCAPS_CONTINUOUSRATE) != 0) {
                wf.Format.nSamplesPerSec = ma_get_best_sample_rate_within_range(caps.dwMinSecondarySampleRate, caps.dwMaxSecondarySampleRate);
            } else {
                wf.Format.nSamplesPerSec = caps.dwMaxSecondarySampleRate;
            }
        }

        wf.Format.nBlockAlign     = (wf.Format.nChannels * wf.Format.wBitsPerSample) / 8;
        wf.Format.nAvgBytesPerSec = wf.Format.nBlockAlign * wf.Format.nSamplesPerSec;

        /*
        From MSDN:
        
        The method succeeds even if the hardware does not support the requested format; DirectSound sets the buffer to the closest
        supported format. To determine whether this has happened, an application can call the GetFormat method for the primary buffer
        and compare the result with the format that was requested with the SetFormat method.
        */
        if (FAILED(ma_IDirectSoundBuffer_SetFormat((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackPrimaryBuffer, (WAVEFORMATEX*)&wf))) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to set format of playback device's primary buffer.", MA_FORMAT_NOT_SUPPORTED);
        }

        /* Get the _actual_ properties of the buffer. */
        pActualFormat = (WAVEFORMATEXTENSIBLE*)rawdata;
        if (FAILED(ma_IDirectSoundBuffer_GetFormat((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackPrimaryBuffer, (WAVEFORMATEX*)pActualFormat, sizeof(rawdata), NULL))) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to retrieve the actual format of the playback device's primary buffer.", MA_FORMAT_NOT_SUPPORTED);
        }

        pDevice->playback.internalFormat = ma_format_from_WAVEFORMATEX((WAVEFORMATEX*)pActualFormat);
        pDevice->playback.internalChannels = pActualFormat->Format.nChannels;
        pDevice->playback.internalSampleRate = pActualFormat->Format.nSamplesPerSec;

        /* Get the internal channel map based on the channel mask. */
        if (pActualFormat->Format.wFormatTag == WAVE_FORMAT_EXTENSIBLE) {
            ma_channel_mask_to_channel_map__win32(pActualFormat->dwChannelMask, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);
        } else {
            ma_channel_mask_to_channel_map__win32(wf.dwChannelMask, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);
        }

        /* The size of the buffer must be a clean multiple of the period count. */
        bufferSizeInFrames = (ma_calculate_buffer_size_in_frames_from_milliseconds(bufferSizeInMilliseconds, pDevice->playback.internalSampleRate) / pConfig->periods) * pConfig->periods;

        /*
        Meaning of dwFlags (from MSDN):
        
        DSBCAPS_CTRLPOSITIONNOTIFY
          The buffer has position notification capability.
        
        DSBCAPS_GLOBALFOCUS
          With this flag set, an application using DirectSound can continue to play its buffers if the user switches focus to
          another application, even if the new application uses DirectSound.
        
        DSBCAPS_GETCURRENTPOSITION2
          In the first version of DirectSound, the play cursor was significantly ahead of the actual playing sound on emulated
          sound cards; it was directly behind the write cursor. Now, if the DSBCAPS_GETCURRENTPOSITION2 flag is specified, the
          application can get a more accurate play cursor.
        */
        ma_zero_object(&descDS);
        descDS.dwSize = sizeof(descDS);
        descDS.dwFlags = MA_DSBCAPS_CTRLPOSITIONNOTIFY | MA_DSBCAPS_GLOBALFOCUS | MA_DSBCAPS_GETCURRENTPOSITION2;
        descDS.dwBufferBytes = bufferSizeInFrames * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
        descDS.lpwfxFormat = (WAVEFORMATEX*)&wf;
        if (FAILED(ma_IDirectSound_CreateSoundBuffer((ma_IDirectSound*)pDevice->dsound.pPlayback, &descDS, (ma_IDirectSoundBuffer**)&pDevice->dsound.pPlaybackBuffer, NULL))) {
            ma_device_uninit__dsound(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSound_CreateSoundBuffer() failed for playback device's secondary buffer.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        /* DirectSound should give us a buffer exactly the size we asked for. */
        pDevice->playback.internalBufferSizeInFrames = bufferSizeInFrames;
        pDevice->playback.internalPeriods            = pConfig->periods;
    }

    (void)pContext;
    return MA_SUCCESS;
}


ma_result ma_device_main_loop__dsound(ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;
    ma_uint32 bpfCapture  = ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
    ma_uint32 bpfPlayback = ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
    HRESULT hr;
    DWORD lockOffsetInBytesCapture;
    DWORD lockSizeInBytesCapture;
    DWORD mappedSizeInBytesCapture;
    void* pMappedBufferCapture;
    DWORD lockOffsetInBytesPlayback;
    DWORD lockSizeInBytesPlayback;
    DWORD mappedSizeInBytesPlayback;
    void* pMappedBufferPlayback;
    DWORD prevReadCursorInBytesCapture = 0;
    DWORD prevPlayCursorInBytesPlayback = 0;
    ma_bool32 physicalPlayCursorLoopFlagPlayback = 0;
    DWORD virtualWriteCursorInBytesPlayback = 0;
    ma_bool32 virtualWriteCursorLoopFlagPlayback = 0;
    ma_bool32 isPlaybackDeviceStarted = MA_FALSE;
    ma_uint32 framesWrittenToPlaybackDevice = 0;   /* For knowing whether or not the playback device needs to be started. */
    ma_uint32 waitTimeInMilliseconds = 1;

    ma_assert(pDevice != NULL);

    /* The first thing to do is start the capture device. The playback device is only started after the first period is written. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        if (FAILED(ma_IDirectSoundCaptureBuffer_Start((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, MA_DSCBSTART_LOOPING))) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCaptureBuffer_Start() failed.", MA_FAILED_TO_START_BACKEND_DEVICE);
        }
    }
    
    while (ma_device__get_state(pDevice) == MA_STATE_STARTED) {
        switch (pDevice->type)
        {
            case ma_device_type_duplex:
            {
                DWORD physicalCaptureCursorInBytes;
                DWORD physicalReadCursorInBytes;
                if (FAILED(ma_IDirectSoundCaptureBuffer_GetCurrentPosition((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, &physicalCaptureCursorInBytes, &physicalReadCursorInBytes))) {
                    return MA_ERROR;
                }

                /* If nothing is available we just sleep for a bit and return from this iteration. */
                if (physicalReadCursorInBytes == prevReadCursorInBytesCapture) {
                    ma_sleep(waitTimeInMilliseconds);
                    continue; /* Nothing is available in the capture buffer. */
                }

                /*
                The current position has moved. We need to map all of the captured samples and write them to the playback device, making sure
                we don't return until every frame has been copied over.
                */
                if (prevReadCursorInBytesCapture < physicalReadCursorInBytes) {
                    /* The capture position has not looped. This is the simple case. */
                    lockOffsetInBytesCapture = prevReadCursorInBytesCapture;
                    lockSizeInBytesCapture   = (physicalReadCursorInBytes - prevReadCursorInBytesCapture);
                } else {
                    /*
                    The capture position has looped. This is the more complex case. Map to the end of the buffer. If this does not return anything,
                    do it again from the start.
                    */
                    if (prevReadCursorInBytesCapture < pDevice->capture.internalBufferSizeInFrames*bpfCapture) {
                        /* Lock up to the end of the buffer. */
                        lockOffsetInBytesCapture = prevReadCursorInBytesCapture;
                        lockSizeInBytesCapture   = (pDevice->capture.internalBufferSizeInFrames*bpfCapture) - prevReadCursorInBytesCapture;
                    } else {
                        /* Lock starting from the start of the buffer. */
                        lockOffsetInBytesCapture = 0;
                        lockSizeInBytesCapture   = physicalReadCursorInBytes;
                    }
                }

                if (lockSizeInBytesCapture == 0) {
                    ma_sleep(waitTimeInMilliseconds);
                    continue; /* Nothing is available in the capture buffer. */
                }

                hr = ma_IDirectSoundCaptureBuffer_Lock((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, lockOffsetInBytesCapture, lockSizeInBytesCapture, &pMappedBufferCapture, &mappedSizeInBytesCapture, NULL, NULL, 0);
                if (FAILED(hr)) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to map buffer from capture device in preparation for writing to the device.", MA_FAILED_TO_MAP_DEVICE_BUFFER);
                }


                /* At this point we have some input data that we need to output. We do not return until every mapped frame of the input data is written to the playback device. */
                pDevice->capture._dspFrameCount = mappedSizeInBytesCapture / bpfCapture;
                pDevice->capture._dspFrames     = (const ma_uint8*)pMappedBufferCapture;
                for (;;) {  /* Keep writing to the playback device. */
                    ma_uint8  inputFramesInExternalFormat[4096];
                    ma_uint32 inputFramesInExternalFormatCap = sizeof(inputFramesInExternalFormat) / ma_get_bytes_per_frame(pDevice->capture.format, pDevice->capture.channels);
                    ma_uint32 inputFramesInExternalFormatCount;
                    ma_uint8  outputFramesInExternalFormat[4096];
                    ma_uint32 outputFramesInExternalFormatCap = sizeof(outputFramesInExternalFormat) / ma_get_bytes_per_frame(pDevice->playback.format, pDevice->playback.channels);

                    inputFramesInExternalFormatCount = (ma_uint32)ma_pcm_converter_read(&pDevice->capture.converter, inputFramesInExternalFormat, ma_min(inputFramesInExternalFormatCap, outputFramesInExternalFormatCap));
                    if (inputFramesInExternalFormatCount == 0) {
                        break;  /* No more input data. */
                    }

                    pDevice->onData(pDevice, outputFramesInExternalFormat, inputFramesInExternalFormat, inputFramesInExternalFormatCount);

                    /* At this point we have input and output data in external format. All we need to do now is convert it to the output format. This may take a few passes. */
                    pDevice->playback._dspFrameCount = inputFramesInExternalFormatCount;
                    pDevice->playback._dspFrames     = (const ma_uint8*)outputFramesInExternalFormat;
                    for (;;) {
                        ma_uint32 framesWrittenThisIteration;
                        DWORD physicalPlayCursorInBytes;
                        DWORD physicalWriteCursorInBytes;
                        DWORD availableBytesPlayback;
                        DWORD silentPaddingInBytes = 0; /* <-- Must be initialized to 0. */

                        /* We need the physical play and write cursors. */
                        if (FAILED(ma_IDirectSoundBuffer_GetCurrentPosition((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, &physicalPlayCursorInBytes, &physicalWriteCursorInBytes))) {
                            break;
                        }

                        if (physicalPlayCursorInBytes < prevPlayCursorInBytesPlayback) {
                            physicalPlayCursorLoopFlagPlayback = !physicalPlayCursorLoopFlagPlayback;
                        }
                        prevPlayCursorInBytesPlayback  = physicalPlayCursorInBytes;

                        /* If there's any bytes available for writing we can do that now. The space between the virtual cursor position and play cursor. */
                        if (physicalPlayCursorLoopFlagPlayback == virtualWriteCursorLoopFlagPlayback) {
                            /* Same loop iteration. The available bytes wraps all the way around from the virtual write cursor to the physical play cursor. */
                            if (physicalPlayCursorInBytes <= virtualWriteCursorInBytesPlayback) {
                                availableBytesPlayback  = (pDevice->playback.internalBufferSizeInFrames*bpfPlayback) - virtualWriteCursorInBytesPlayback;
                                availableBytesPlayback += physicalPlayCursorInBytes;    /* Wrap around. */
                            } else {
                                /* This is an error. */
                            #ifdef MA_DEBUG_OUTPUT
                                printf("[DirectSound] (Duplex/Playback) WARNING: Play cursor has moved in front of the write cursor (same loop iterations). physicalPlayCursorInBytes=%d, virtualWriteCursorInBytes=%d.\n", physicalPlayCursorInBytes, virtualWriteCursorInBytesPlayback);
                            #endif
                                availableBytesPlayback = 0;
                            }
                        } else {
                            /* Different loop iterations. The available bytes only goes from the virtual write cursor to the physical play cursor. */
                            if (physicalPlayCursorInBytes >= virtualWriteCursorInBytesPlayback) {
                                availableBytesPlayback = physicalPlayCursorInBytes - virtualWriteCursorInBytesPlayback;
                            } else {
                                /* This is an error. */
                            #ifdef MA_DEBUG_OUTPUT
                                printf("[DirectSound] (Duplex/Playback) WARNING: Write cursor has moved behind the play cursor (different loop iterations). physicalPlayCursorInBytes=%d, virtualWriteCursorInBytes=%d.\n", physicalPlayCursorInBytes, virtualWriteCursorInBytesPlayback);
                            #endif
                                availableBytesPlayback = 0;
                            }
                        }

                    #ifdef MA_DEBUG_OUTPUT
                        /*printf("[DirectSound] (Duplex/Playback) physicalPlayCursorInBytes=%d, availableBytesPlayback=%d\n", physicalPlayCursorInBytes, availableBytesPlayback);*/
                    #endif

                        /* If there's no room available for writing we need to wait for more. */
                        if (availableBytesPlayback == 0) {
                            /* If we haven't started the device yet, this will never get beyond 0. In this case we need to get the device started. */
                            if (!isPlaybackDeviceStarted) {
                                if (FAILED(ma_IDirectSoundBuffer_Play((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0, 0, MA_DSBPLAY_LOOPING))) {
                                    ma_IDirectSoundCaptureBuffer_Stop((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer);
                                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Play() failed.", MA_FAILED_TO_START_BACKEND_DEVICE);
                                }
                                isPlaybackDeviceStarted = MA_TRUE;
                            } else {
                                ma_sleep(waitTimeInMilliseconds);
                                continue;
                            }
                        }


                        /* Getting here means there room available somewhere. We limit this to either the end of the buffer or the physical play cursor, whichever is closest. */
                        lockOffsetInBytesPlayback = virtualWriteCursorInBytesPlayback;
                        if (physicalPlayCursorLoopFlagPlayback == virtualWriteCursorLoopFlagPlayback) {
                            /* Same loop iteration. Go up to the end of the buffer. */
                            lockSizeInBytesPlayback = (pDevice->playback.internalBufferSizeInFrames*bpfPlayback) - virtualWriteCursorInBytesPlayback;
                        } else {
                            /* Different loop iterations. Go up to the physical play cursor. */
                            lockSizeInBytesPlayback = physicalPlayCursorInBytes - virtualWriteCursorInBytesPlayback;
                        }

                        hr = ma_IDirectSoundBuffer_Lock((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, lockOffsetInBytesPlayback, lockSizeInBytesPlayback, &pMappedBufferPlayback, &mappedSizeInBytesPlayback, NULL, NULL, 0);
                        if (FAILED(hr)) {
                            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to map buffer from playback device in preparation for writing to the device.", MA_FAILED_TO_MAP_DEVICE_BUFFER);
                            break;
                        }

                        /*
                        Experiment: If the playback buffer is being starved, pad it with some silence to get it back in sync. This will cause a glitch, but it may prevent
                        endless glitching due to it constantly running out of data.
                        */
                        if (isPlaybackDeviceStarted) {
                            DWORD bytesQueuedForPlayback = (pDevice->playback.internalBufferSizeInFrames*bpfPlayback) - availableBytesPlayback;
                            if (bytesQueuedForPlayback < ((pDevice->playback.internalBufferSizeInFrames/pDevice->playback.internalPeriods)*bpfPlayback)) {
                                silentPaddingInBytes = ((pDevice->playback.internalBufferSizeInFrames/pDevice->playback.internalPeriods)*2*bpfPlayback) - bytesQueuedForPlayback;
                                if (silentPaddingInBytes > lockSizeInBytesPlayback) {
                                    silentPaddingInBytes = lockSizeInBytesPlayback;
                                }

                        #ifdef MA_DEBUG_OUTPUT
                                printf("[DirectSound] (Duplex/Playback) Playback buffer starved. availableBytesPlayback=%d, silentPaddingInBytes=%d\n", availableBytesPlayback, silentPaddingInBytes);
                        #endif
                            }
                        }

                        /* At this point we have a buffer for output. */
                        if (silentPaddingInBytes > 0) {
                            ma_zero_memory(pMappedBufferPlayback, silentPaddingInBytes);
                            framesWrittenThisIteration = silentPaddingInBytes/bpfPlayback;
                        } else {
                            framesWrittenThisIteration = (ma_uint32)ma_pcm_converter_read(&pDevice->playback.converter, pMappedBufferPlayback, mappedSizeInBytesPlayback/bpfPlayback);
                        }
                        

                        hr = ma_IDirectSoundBuffer_Unlock((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, pMappedBufferPlayback, framesWrittenThisIteration*bpfPlayback, NULL, 0);
                        if (FAILED(hr)) {
                            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to unlock internal buffer from playback device after writing to the device.", MA_FAILED_TO_UNMAP_DEVICE_BUFFER);
                            break;
                        }

                        virtualWriteCursorInBytesPlayback += framesWrittenThisIteration*bpfPlayback;
                        if ((virtualWriteCursorInBytesPlayback/bpfPlayback) == pDevice->playback.internalBufferSizeInFrames) {
                            virtualWriteCursorInBytesPlayback  = 0;
                            virtualWriteCursorLoopFlagPlayback = !virtualWriteCursorLoopFlagPlayback;
                        }
                        
                        /*
                        We may need to start the device. We want two full periods to be written before starting the playback device. Having an extra period adds
                        a bit of a buffer to prevent the playback buffer from getting starved.
                        */
                        framesWrittenToPlaybackDevice += framesWrittenThisIteration;
                        if (!isPlaybackDeviceStarted && framesWrittenToPlaybackDevice >= ((pDevice->playback.internalBufferSizeInFrames/pDevice->playback.internalPeriods)*2)) {
                            if (FAILED(ma_IDirectSoundBuffer_Play((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0, 0, MA_DSBPLAY_LOOPING))) {
                                ma_IDirectSoundCaptureBuffer_Stop((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer);
                                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Play() failed.", MA_FAILED_TO_START_BACKEND_DEVICE);
                            }
                            isPlaybackDeviceStarted = MA_TRUE;
                        }

                        if (framesWrittenThisIteration < mappedSizeInBytesPlayback/bpfPlayback) {
                            break;  /* We're finished with the output data.*/
                        }
                    }

                    if (inputFramesInExternalFormatCount < inputFramesInExternalFormatCap) {
                        break;  /* We just consumed every input sample. */
                    }
                }


                /* At this point we're done with the mapped portion of the capture buffer. */
                hr = ma_IDirectSoundCaptureBuffer_Unlock((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, pMappedBufferCapture, mappedSizeInBytesCapture, NULL, 0);
                if (FAILED(hr)) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to unlock internal buffer from capture device after reading from the device.", MA_FAILED_TO_UNMAP_DEVICE_BUFFER);
                }
                prevReadCursorInBytesCapture = (lockOffsetInBytesCapture + mappedSizeInBytesCapture);
            } break;



            case ma_device_type_capture:
            {
                DWORD physicalCaptureCursorInBytes;
                DWORD physicalReadCursorInBytes;
                if (FAILED(ma_IDirectSoundCaptureBuffer_GetCurrentPosition((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, &physicalCaptureCursorInBytes, &physicalReadCursorInBytes))) {
                    return MA_ERROR;
                }

                /* If the previous capture position is the same as the current position we need to wait a bit longer. */
                if (prevReadCursorInBytesCapture == physicalReadCursorInBytes) {
                    ma_sleep(waitTimeInMilliseconds);
                    continue;
                }

                /* Getting here means we have capture data available. */
                if (prevReadCursorInBytesCapture < physicalReadCursorInBytes) {
                    /* The capture position has not looped. This is the simple case. */
                    lockOffsetInBytesCapture = prevReadCursorInBytesCapture;
                    lockSizeInBytesCapture   = (physicalReadCursorInBytes - prevReadCursorInBytesCapture);
                } else {
                    /*
                    The capture position has looped. This is the more complex case. Map to the end of the buffer. If this does not return anything,
                    do it again from the start.
                    */
                    if (prevReadCursorInBytesCapture < pDevice->capture.internalBufferSizeInFrames*bpfCapture) {
                        /* Lock up to the end of the buffer. */
                        lockOffsetInBytesCapture = prevReadCursorInBytesCapture;
                        lockSizeInBytesCapture   = (pDevice->capture.internalBufferSizeInFrames*bpfCapture) - prevReadCursorInBytesCapture;
                    } else {
                        /* Lock starting from the start of the buffer. */
                        lockOffsetInBytesCapture = 0;
                        lockSizeInBytesCapture   = physicalReadCursorInBytes;
                    }
                }

            #ifdef MA_DEBUG_OUTPUT
                /*printf("[DirectSound] (Capture) physicalCaptureCursorInBytes=%d, physicalReadCursorInBytes=%d\n", physicalCaptureCursorInBytes, physicalReadCursorInBytes);*/
                /*printf("[DirectSound] (Capture) lockOffsetInBytesCapture=%d, lockSizeInBytesCapture=%d\n", lockOffsetInBytesCapture, lockSizeInBytesCapture);*/
            #endif

                if (lockSizeInBytesCapture < (pDevice->capture.internalBufferSizeInFrames/pDevice->capture.internalPeriods)) {
                    ma_sleep(waitTimeInMilliseconds);
                    continue; /* Nothing is available in the capture buffer. */
                }

                hr = ma_IDirectSoundCaptureBuffer_Lock((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, lockOffsetInBytesCapture, lockSizeInBytesCapture, &pMappedBufferCapture, &mappedSizeInBytesCapture, NULL, NULL, 0);
                if (FAILED(hr)) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to map buffer from capture device in preparation for writing to the device.", MA_FAILED_TO_MAP_DEVICE_BUFFER);
                }

            #ifdef MA_DEBUG_OUTPUT
                if (lockSizeInBytesCapture != mappedSizeInBytesCapture) {
                    printf("[DirectSound] (Capture) lockSizeInBytesCapture=%d != mappedSizeInBytesCapture=%d\n", lockSizeInBytesCapture, mappedSizeInBytesCapture);
                }
            #endif

                ma_device__send_frames_to_client(pDevice, mappedSizeInBytesCapture/bpfCapture, pMappedBufferCapture);

                hr = ma_IDirectSoundCaptureBuffer_Unlock((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer, pMappedBufferCapture, mappedSizeInBytesCapture, NULL, 0);
                if (FAILED(hr)) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to unlock internal buffer from capture device after reading from the device.", MA_FAILED_TO_UNMAP_DEVICE_BUFFER);
                }
                prevReadCursorInBytesCapture = lockOffsetInBytesCapture + mappedSizeInBytesCapture;

                if (prevReadCursorInBytesCapture == (pDevice->capture.internalBufferSizeInFrames*bpfCapture)) {
                    prevReadCursorInBytesCapture = 0;
                }
            } break;



            case ma_device_type_playback:
            {
                DWORD availableBytesPlayback;
                DWORD physicalPlayCursorInBytes;
                DWORD physicalWriteCursorInBytes;
                if (FAILED(ma_IDirectSoundBuffer_GetCurrentPosition((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, &physicalPlayCursorInBytes, &physicalWriteCursorInBytes))) {
                    break;
                }

                if (physicalPlayCursorInBytes < prevPlayCursorInBytesPlayback) {
                    physicalPlayCursorLoopFlagPlayback = !physicalPlayCursorLoopFlagPlayback;
                }
                prevPlayCursorInBytesPlayback  = physicalPlayCursorInBytes;

                /* If there's any bytes available for writing we can do that now. The space between the virtual cursor position and play cursor. */
                if (physicalPlayCursorLoopFlagPlayback == virtualWriteCursorLoopFlagPlayback) {
                    /* Same loop iteration. The available bytes wraps all the way around from the virtual write cursor to the physical play cursor. */
                    if (physicalPlayCursorInBytes <= virtualWriteCursorInBytesPlayback) {
                        availableBytesPlayback  = (pDevice->playback.internalBufferSizeInFrames*bpfPlayback) - virtualWriteCursorInBytesPlayback;
                        availableBytesPlayback += physicalPlayCursorInBytes;    /* Wrap around. */
                    } else {
                        /* This is an error. */
                    #ifdef MA_DEBUG_OUTPUT
                        printf("[DirectSound] (Playback) WARNING: Play cursor has moved in front of the write cursor (same loop iterations). physicalPlayCursorInBytes=%d, virtualWriteCursorInBytes=%d.\n", physicalPlayCursorInBytes, virtualWriteCursorInBytesPlayback);
                    #endif
                        availableBytesPlayback = 0;
                    }
                } else {
                    /* Different loop iterations. The available bytes only goes from the virtual write cursor to the physical play cursor. */
                    if (physicalPlayCursorInBytes >= virtualWriteCursorInBytesPlayback) {
                        availableBytesPlayback = physicalPlayCursorInBytes - virtualWriteCursorInBytesPlayback;
                    } else {
                        /* This is an error. */
                    #ifdef MA_DEBUG_OUTPUT
                        printf("[DirectSound] (Playback) WARNING: Write cursor has moved behind the play cursor (different loop iterations). physicalPlayCursorInBytes=%d, virtualWriteCursorInBytes=%d.\n", physicalPlayCursorInBytes, virtualWriteCursorInBytesPlayback);
                    #endif
                        availableBytesPlayback = 0;
                    }
                }

            #ifdef MA_DEBUG_OUTPUT
                /*printf("[DirectSound] (Playback) physicalPlayCursorInBytes=%d, availableBytesPlayback=%d\n", physicalPlayCursorInBytes, availableBytesPlayback);*/
            #endif

                /* If there's no room available for writing we need to wait for more. */
                if (availableBytesPlayback < (pDevice->playback.internalBufferSizeInFrames/pDevice->playback.internalPeriods)) {
                    /* If we haven't started the device yet, this will never get beyond 0. In this case we need to get the device started. */
                    if (availableBytesPlayback == 0 && !isPlaybackDeviceStarted) {
                        if (FAILED(ma_IDirectSoundBuffer_Play((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0, 0, MA_DSBPLAY_LOOPING))) {
                            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Play() failed.", MA_FAILED_TO_START_BACKEND_DEVICE);
                        }
                        isPlaybackDeviceStarted = MA_TRUE;
                    } else {
                        ma_sleep(waitTimeInMilliseconds);
                        continue;
                    }
                }

                /* Getting here means there room available somewhere. We limit this to either the end of the buffer or the physical play cursor, whichever is closest. */
                lockOffsetInBytesPlayback = virtualWriteCursorInBytesPlayback;
                if (physicalPlayCursorLoopFlagPlayback == virtualWriteCursorLoopFlagPlayback) {
                    /* Same loop iteration. Go up to the end of the buffer. */
                    lockSizeInBytesPlayback = (pDevice->playback.internalBufferSizeInFrames*bpfPlayback) - virtualWriteCursorInBytesPlayback;
                } else {
                    /* Different loop iterations. Go up to the physical play cursor. */
                    lockSizeInBytesPlayback = physicalPlayCursorInBytes - virtualWriteCursorInBytesPlayback;
                }

                hr = ma_IDirectSoundBuffer_Lock((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, lockOffsetInBytesPlayback, lockSizeInBytesPlayback, &pMappedBufferPlayback, &mappedSizeInBytesPlayback, NULL, NULL, 0);
                if (FAILED(hr)) {
                    result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to map buffer from playback device in preparation for writing to the device.", MA_FAILED_TO_MAP_DEVICE_BUFFER);
                    break;
                }

                /* At this point we have a buffer for output. */
                ma_device__read_frames_from_client(pDevice, (mappedSizeInBytesPlayback/bpfPlayback), pMappedBufferPlayback);

                hr = ma_IDirectSoundBuffer_Unlock((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, pMappedBufferPlayback, mappedSizeInBytesPlayback, NULL, 0);
                if (FAILED(hr)) {
                    result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] Failed to unlock internal buffer from playback device after writing to the device.", MA_FAILED_TO_UNMAP_DEVICE_BUFFER);
                    break;
                }

                virtualWriteCursorInBytesPlayback += mappedSizeInBytesPlayback;
                if (virtualWriteCursorInBytesPlayback == pDevice->playback.internalBufferSizeInFrames*bpfPlayback) {
                    virtualWriteCursorInBytesPlayback  = 0;
                    virtualWriteCursorLoopFlagPlayback = !virtualWriteCursorLoopFlagPlayback;
                }
                        
                /*
                We may need to start the device. We want two full periods to be written before starting the playback device. Having an extra period adds
                a bit of a buffer to prevent the playback buffer from getting starved.
                */
                framesWrittenToPlaybackDevice += mappedSizeInBytesPlayback/bpfPlayback;
                if (!isPlaybackDeviceStarted && framesWrittenToPlaybackDevice >= (pDevice->playback.internalBufferSizeInFrames/pDevice->playback.internalPeriods)) {
                    if (FAILED(ma_IDirectSoundBuffer_Play((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0, 0, MA_DSBPLAY_LOOPING))) {
                        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Play() failed.", MA_FAILED_TO_START_BACKEND_DEVICE);
                    }
                    isPlaybackDeviceStarted = MA_TRUE;
                }
            } break;


            default: return MA_INVALID_ARGS;   /* Invalid device type. */
        }

        if (result != MA_SUCCESS) {
            return result;
        }
    }

    /* Getting here means the device is being stopped. */
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        if (FAILED(ma_IDirectSoundCaptureBuffer_Stop((ma_IDirectSoundCaptureBuffer*)pDevice->dsound.pCaptureBuffer))) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundCaptureBuffer_Stop() failed.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        /* The playback device should be drained before stopping. All we do is wait until the available bytes is equal to the size of the buffer. */
        if (isPlaybackDeviceStarted) {
            for (;;) {
                DWORD availableBytesPlayback = 0;
                DWORD physicalPlayCursorInBytes;
                DWORD physicalWriteCursorInBytes;
                if (FAILED(ma_IDirectSoundBuffer_GetCurrentPosition((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, &physicalPlayCursorInBytes, &physicalWriteCursorInBytes))) {
                    break;
                }

                if (physicalPlayCursorInBytes < prevPlayCursorInBytesPlayback) {
                    physicalPlayCursorLoopFlagPlayback = !physicalPlayCursorLoopFlagPlayback;
                }
                prevPlayCursorInBytesPlayback  = physicalPlayCursorInBytes;

                if (physicalPlayCursorLoopFlagPlayback == virtualWriteCursorLoopFlagPlayback) {
                    /* Same loop iteration. The available bytes wraps all the way around from the virtual write cursor to the physical play cursor. */
                    if (physicalPlayCursorInBytes <= virtualWriteCursorInBytesPlayback) {
                        availableBytesPlayback  = (pDevice->playback.internalBufferSizeInFrames*bpfPlayback) - virtualWriteCursorInBytesPlayback;
                        availableBytesPlayback += physicalPlayCursorInBytes;    /* Wrap around. */
                    } else {
                        break;
                    }
                } else {
                    /* Different loop iterations. The available bytes only goes from the virtual write cursor to the physical play cursor. */
                    if (physicalPlayCursorInBytes >= virtualWriteCursorInBytesPlayback) {
                        availableBytesPlayback = physicalPlayCursorInBytes - virtualWriteCursorInBytesPlayback;
                    } else {
                        break;
                    }
                }

                if (availableBytesPlayback >= (pDevice->playback.internalBufferSizeInFrames*bpfPlayback)) {
                    break;
                }

                ma_sleep(waitTimeInMilliseconds);
            }
        }

        if (FAILED(ma_IDirectSoundBuffer_Stop((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer))) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[DirectSound] IDirectSoundBuffer_Stop() failed.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
        }

        ma_IDirectSoundBuffer_SetCurrentPosition((ma_IDirectSoundBuffer*)pDevice->dsound.pPlaybackBuffer, 0);
    }

    return MA_SUCCESS;
}

ma_result ma_context_uninit__dsound(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_dsound);

    ma_dlclose(pContext->dsound.hDSoundDLL);

    return MA_SUCCESS;
}

ma_result ma_context_init__dsound(const ma_context_config* pConfig, ma_context* pContext)
{
    ma_assert(pContext != NULL);

    (void)pConfig;

    pContext->dsound.hDSoundDLL = ma_dlopen("dsound.dll");
    if (pContext->dsound.hDSoundDLL == NULL) {
        return MA_API_NOT_FOUND;
    }

    pContext->dsound.DirectSoundCreate            = ma_dlsym(pContext->dsound.hDSoundDLL, "DirectSoundCreate");
    pContext->dsound.DirectSoundEnumerateA        = ma_dlsym(pContext->dsound.hDSoundDLL, "DirectSoundEnumerateA");
    pContext->dsound.DirectSoundCaptureCreate     = ma_dlsym(pContext->dsound.hDSoundDLL, "DirectSoundCaptureCreate");
    pContext->dsound.DirectSoundCaptureEnumerateA = ma_dlsym(pContext->dsound.hDSoundDLL, "DirectSoundCaptureEnumerateA");

    pContext->onUninit         = ma_context_uninit__dsound;
    pContext->onDeviceIDEqual  = ma_context_is_device_id_equal__dsound;
    pContext->onEnumDevices    = ma_context_enumerate_devices__dsound;
    pContext->onGetDeviceInfo  = ma_context_get_device_info__dsound;
    pContext->onDeviceInit     = ma_device_init__dsound;
    pContext->onDeviceUninit   = ma_device_uninit__dsound;
    pContext->onDeviceStart    = NULL;  /* Not used. Started in onDeviceMainLoop. */
    pContext->onDeviceStop     = NULL;  /* Not used. Stopped in onDeviceMainLoop. */
    pContext->onDeviceWrite    = NULL;
    pContext->onDeviceRead     = NULL;
    pContext->onDeviceMainLoop = ma_device_main_loop__dsound;

    return MA_SUCCESS;
}
#endif



/******************************************************************************

WinMM Backend

******************************************************************************/
#ifdef MA_HAS_WINMM

/*
Some older compilers don't have WAVEOUTCAPS2A and WAVEINCAPS2A, so we'll need to write this ourselves. These structures
are exactly the same as the older ones but they have a few GUIDs for manufacturer/product/name identification. I'm keeping
the names the same as the Win32 library for consistency, but namespaced to avoid naming conflicts with the Win32 version.
*/
typedef struct
{
    WORD wMid;
    WORD wPid;
    MMVERSION vDriverVersion;
    CHAR szPname[MAXPNAMELEN];
    DWORD dwFormats;
    WORD wChannels;
    WORD wReserved1;
    DWORD dwSupport;
    GUID ManufacturerGuid;
    GUID ProductGuid;
    GUID NameGuid;
} MA_WAVEOUTCAPS2A;
typedef struct
{
    WORD wMid;
    WORD wPid;
    MMVERSION vDriverVersion;
    CHAR szPname[MAXPNAMELEN];
    DWORD dwFormats;
    WORD wChannels;
    WORD wReserved1;
    GUID ManufacturerGuid;
    GUID ProductGuid;
    GUID NameGuid;
} MA_WAVEINCAPS2A;

typedef UINT     (WINAPI * MA_PFN_waveOutGetNumDevs)(void);
typedef MMRESULT (WINAPI * MA_PFN_waveOutGetDevCapsA)(ma_uintptr uDeviceID, LPWAVEOUTCAPSA pwoc, UINT cbwoc);
typedef MMRESULT (WINAPI * MA_PFN_waveOutOpen)(LPHWAVEOUT phwo, UINT uDeviceID, LPCWAVEFORMATEX pwfx, DWORD_PTR dwCallback, DWORD_PTR dwInstance, DWORD fdwOpen);
typedef MMRESULT (WINAPI * MA_PFN_waveOutClose)(HWAVEOUT hwo);
typedef MMRESULT (WINAPI * MA_PFN_waveOutPrepareHeader)(HWAVEOUT hwo, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveOutUnprepareHeader)(HWAVEOUT hwo, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveOutWrite)(HWAVEOUT hwo, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveOutReset)(HWAVEOUT hwo);
typedef UINT     (WINAPI * MA_PFN_waveInGetNumDevs)(void);
typedef MMRESULT (WINAPI * MA_PFN_waveInGetDevCapsA)(ma_uintptr uDeviceID, LPWAVEINCAPSA pwic, UINT cbwic);
typedef MMRESULT (WINAPI * MA_PFN_waveInOpen)(LPHWAVEIN phwi, UINT uDeviceID, LPCWAVEFORMATEX pwfx, DWORD_PTR dwCallback, DWORD_PTR dwInstance, DWORD fdwOpen);
typedef MMRESULT (WINAPI * MA_PFN_waveInClose)(HWAVEIN hwi);
typedef MMRESULT (WINAPI * MA_PFN_waveInPrepareHeader)(HWAVEIN hwi, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveInUnprepareHeader)(HWAVEIN hwi, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveInAddBuffer)(HWAVEIN hwi, LPWAVEHDR pwh, UINT cbwh);
typedef MMRESULT (WINAPI * MA_PFN_waveInStart)(HWAVEIN hwi);
typedef MMRESULT (WINAPI * MA_PFN_waveInReset)(HWAVEIN hwi);

ma_result ma_result_from_MMRESULT(MMRESULT resultMM)
{
    switch (resultMM) {
        case MMSYSERR_NOERROR:      return MA_SUCCESS;
        case MMSYSERR_BADDEVICEID:  return MA_INVALID_ARGS;
        case MMSYSERR_INVALHANDLE:  return MA_INVALID_ARGS;
        case MMSYSERR_NOMEM:        return MA_OUT_OF_MEMORY;
        case MMSYSERR_INVALFLAG:    return MA_INVALID_ARGS;
        case MMSYSERR_INVALPARAM:   return MA_INVALID_ARGS;
        case MMSYSERR_HANDLEBUSY:   return MA_DEVICE_BUSY;
        case MMSYSERR_ERROR:        return MA_ERROR;
        default:                    return MA_ERROR;
    }
}

char* ma_find_last_character(char* str, char ch)
{
    char* last;

    if (str == NULL) {
        return NULL;
    }

    last = NULL;
    while (*str != '\0') {
        if (*str == ch) {
            last = str;
        }

        str += 1;
    }

    return last;
}


/*
Our own "WAVECAPS" structure that contains generic information shared between WAVEOUTCAPS2 and WAVEINCAPS2 so
we can do things generically and typesafely. Names are being kept the same for consistency.
*/
typedef struct
{
    CHAR szPname[MAXPNAMELEN];
    DWORD dwFormats;
    WORD wChannels;
    GUID NameGuid;
} MA_WAVECAPSA;

ma_result ma_get_best_info_from_formats_flags__winmm(DWORD dwFormats, WORD channels, WORD* pBitsPerSample, DWORD* pSampleRate)
{
    WORD bitsPerSample = 0;
    DWORD sampleRate = 0;

    if (pBitsPerSample) {
        *pBitsPerSample = 0;
    }
    if (pSampleRate) {
        *pSampleRate = 0;
    }

    if (channels == 1) {
        bitsPerSample = 16;
        if ((dwFormats & WAVE_FORMAT_48M16) != 0) {
            sampleRate = 48000;
        } else if ((dwFormats & WAVE_FORMAT_44M16) != 0) {
            sampleRate = 44100;
        } else if ((dwFormats & WAVE_FORMAT_2M16) != 0) {
            sampleRate = 22050;
        } else if ((dwFormats & WAVE_FORMAT_1M16) != 0) {
            sampleRate = 11025;
        } else if ((dwFormats & WAVE_FORMAT_96M16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((dwFormats & WAVE_FORMAT_48M08) != 0) {
                sampleRate = 48000;
            } else if ((dwFormats & WAVE_FORMAT_44M08) != 0) {
                sampleRate = 44100;
            } else if ((dwFormats & WAVE_FORMAT_2M08) != 0) {
                sampleRate = 22050;
            } else if ((dwFormats & WAVE_FORMAT_1M08) != 0) {
                sampleRate = 11025;
            } else if ((dwFormats & WAVE_FORMAT_96M08) != 0) {
                sampleRate = 96000;
            } else {
                return MA_FORMAT_NOT_SUPPORTED;
            }
        }
    } else {
        bitsPerSample = 16;
        if ((dwFormats & WAVE_FORMAT_48S16) != 0) {
            sampleRate = 48000;
        } else if ((dwFormats & WAVE_FORMAT_44S16) != 0) {
            sampleRate = 44100;
        } else if ((dwFormats & WAVE_FORMAT_2S16) != 0) {
            sampleRate = 22050;
        } else if ((dwFormats & WAVE_FORMAT_1S16) != 0) {
            sampleRate = 11025;
        } else if ((dwFormats & WAVE_FORMAT_96S16) != 0) {
            sampleRate = 96000;
        } else {
            bitsPerSample = 8;
            if ((dwFormats & WAVE_FORMAT_48S08) != 0) {
                sampleRate = 48000;
            } else if ((dwFormats & WAVE_FORMAT_44S08) != 0) {
                sampleRate = 44100;
            } else if ((dwFormats & WAVE_FORMAT_2S08) != 0) {
                sampleRate = 22050;
            } else if ((dwFormats & WAVE_FORMAT_1S08) != 0) {
                sampleRate = 11025;
            } else if ((dwFormats & WAVE_FORMAT_96S08) != 0) {
                sampleRate = 96000;
            } else {
                return MA_FORMAT_NOT_SUPPORTED;
            }
        }
    }

    if (pBitsPerSample) {
        *pBitsPerSample = bitsPerSample;
    }
    if (pSampleRate) {
        *pSampleRate = sampleRate;
    }

    return MA_SUCCESS;
}

ma_result ma_formats_flags_to_WAVEFORMATEX__winmm(DWORD dwFormats, WORD channels, WAVEFORMATEX* pWF)
{
    ma_assert(pWF != NULL);

    ma_zero_object(pWF);
    pWF->cbSize     = sizeof(*pWF);
    pWF->wFormatTag = WAVE_FORMAT_PCM;
    pWF->nChannels  = (WORD)channels;
    if (pWF->nChannels > 2) {
        pWF->nChannels = 2;
    }

    if (channels == 1) {
        pWF->wBitsPerSample = 16;
        if ((dwFormats & WAVE_FORMAT_48M16) != 0) {
            pWF->nSamplesPerSec = 48000;
        } else if ((dwFormats & WAVE_FORMAT_44M16) != 0) {
            pWF->nSamplesPerSec = 44100;
        } else if ((dwFormats & WAVE_FORMAT_2M16) != 0) {
            pWF->nSamplesPerSec = 22050;
        } else if ((dwFormats & WAVE_FORMAT_1M16) != 0) {
            pWF->nSamplesPerSec = 11025;
        } else if ((dwFormats & WAVE_FORMAT_96M16) != 0) {
            pWF->nSamplesPerSec = 96000;
        } else {
            pWF->wBitsPerSample = 8;
            if ((dwFormats & WAVE_FORMAT_48M08) != 0) {
                pWF->nSamplesPerSec = 48000;
            } else if ((dwFormats & WAVE_FORMAT_44M08) != 0) {
                pWF->nSamplesPerSec = 44100;
            } else if ((dwFormats & WAVE_FORMAT_2M08) != 0) {
                pWF->nSamplesPerSec = 22050;
            } else if ((dwFormats & WAVE_FORMAT_1M08) != 0) {
                pWF->nSamplesPerSec = 11025;
            } else if ((dwFormats & WAVE_FORMAT_96M08) != 0) {
                pWF->nSamplesPerSec = 96000;
            } else {
                return MA_FORMAT_NOT_SUPPORTED;
            }
        }
    } else {
        pWF->wBitsPerSample = 16;
        if ((dwFormats & WAVE_FORMAT_48S16) != 0) {
            pWF->nSamplesPerSec = 48000;
        } else if ((dwFormats & WAVE_FORMAT_44S16) != 0) {
            pWF->nSamplesPerSec = 44100;
        } else if ((dwFormats & WAVE_FORMAT_2S16) != 0) {
            pWF->nSamplesPerSec = 22050;
        } else if ((dwFormats & WAVE_FORMAT_1S16) != 0) {
            pWF->nSamplesPerSec = 11025;
        } else if ((dwFormats & WAVE_FORMAT_96S16) != 0) {
            pWF->nSamplesPerSec = 96000;
        } else {
            pWF->wBitsPerSample = 8;
            if ((dwFormats & WAVE_FORMAT_48S08) != 0) {
                pWF->nSamplesPerSec = 48000;
            } else if ((dwFormats & WAVE_FORMAT_44S08) != 0) {
                pWF->nSamplesPerSec = 44100;
            } else if ((dwFormats & WAVE_FORMAT_2S08) != 0) {
                pWF->nSamplesPerSec = 22050;
            } else if ((dwFormats & WAVE_FORMAT_1S08) != 0) {
                pWF->nSamplesPerSec = 11025;
            } else if ((dwFormats & WAVE_FORMAT_96S08) != 0) {
                pWF->nSamplesPerSec = 96000;
            } else {
                return MA_FORMAT_NOT_SUPPORTED;
            }
        }
    }

    pWF->nBlockAlign     = (pWF->nChannels * pWF->wBitsPerSample) / 8;
    pWF->nAvgBytesPerSec = pWF->nBlockAlign * pWF->nSamplesPerSec;

    return MA_SUCCESS;
}

ma_result ma_context_get_device_info_from_WAVECAPS(ma_context* pContext, MA_WAVECAPSA* pCaps, ma_device_info* pDeviceInfo)
{
    WORD bitsPerSample;
    DWORD sampleRate;
    ma_result result;

    ma_assert(pContext != NULL);
    ma_assert(pCaps != NULL);
    ma_assert(pDeviceInfo != NULL);

    /*
    Name / Description
    
    Unfortunately the name specified in WAVE(OUT/IN)CAPS2 is limited to 31 characters. This results in an unprofessional looking
    situation where the names of the devices are truncated. To help work around this, we need to look at the name GUID and try
    looking in the registry for the full name. If we can't find it there, we need to just fall back to the default name.
    */

    /* Set the default to begin with. */
    ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), pCaps->szPname, (size_t)-1);

    /*
    Now try the registry. There's a few things to consider here:
    - The name GUID can be null, in which we case we just need to stick to the original 31 characters.
    - If the name GUID is not present in the registry we'll also need to stick to the original 31 characters.
    - I like consistency, so I want the returned device names to be consistent with those returned by WASAPI and DirectSound. The
      problem, however is that WASAPI and DirectSound use "<component> (<name>)" format (such as "Speakers (High Definition Audio)"),
      but WinMM does not specificy the component name. From my admittedly limited testing, I've notice the component name seems to
      usually fit within the 31 characters of the fixed sized buffer, so what I'm going to do is parse that string for the component
      name, and then concatenate the name from the registry.
    */
    if (!ma_is_guid_equal(&pCaps->NameGuid, &MA_GUID_NULL)) {
        wchar_t guidStrW[256];
        if (((MA_PFN_StringFromGUID2)pContext->win32.StringFromGUID2)(&pCaps->NameGuid, guidStrW, ma_countof(guidStrW)) > 0) {
            char guidStr[256];
            char keyStr[1024];
            HKEY hKey;

            WideCharToMultiByte(CP_UTF8, 0, guidStrW, -1, guidStr, sizeof(guidStr), 0, FALSE);

            ma_strcpy_s(keyStr, sizeof(keyStr), "SYSTEM\\CurrentControlSet\\Control\\MediaCategories\\");
            ma_strcat_s(keyStr, sizeof(keyStr), guidStr);

            if (((MA_PFN_RegOpenKeyExA)pContext->win32.RegOpenKeyExA)(HKEY_LOCAL_MACHINE, keyStr, 0, KEY_READ, &hKey) == ERROR_SUCCESS) {
                BYTE nameFromReg[512];
                DWORD nameFromRegSize = sizeof(nameFromReg);
                result = ((MA_PFN_RegQueryValueExA)pContext->win32.RegQueryValueExA)(hKey, "Name", 0, NULL, (LPBYTE)nameFromReg, (LPDWORD)&nameFromRegSize);
                ((MA_PFN_RegCloseKey)pContext->win32.RegCloseKey)(hKey);

                if (result == ERROR_SUCCESS) {
                    /* We have the value from the registry, so now we need to construct the name string. */
                    char name[1024];
                    if (ma_strcpy_s(name, sizeof(name), pDeviceInfo->name) == 0) {
                        char* nameBeg = ma_find_last_character(name, '(');
                        if (nameBeg != NULL) {
                            size_t leadingLen = (nameBeg - name);
                            ma_strncpy_s(nameBeg + 1, sizeof(name) - leadingLen, (const char*)nameFromReg, (size_t)-1);

                            /* The closing ")", if it can fit. */
                            if (leadingLen + nameFromRegSize < sizeof(name)-1) {
                                ma_strcat_s(name, sizeof(name), ")");
                            }

                            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), name, (size_t)-1);
                        }
                    }
                }
            }
        }
    }


    result = ma_get_best_info_from_formats_flags__winmm(pCaps->dwFormats, pCaps->wChannels, &bitsPerSample, &sampleRate);
    if (result != MA_SUCCESS) {
        return result;
    }

    pDeviceInfo->minChannels = pCaps->wChannels;
    pDeviceInfo->maxChannels = pCaps->wChannels;
    pDeviceInfo->minSampleRate = sampleRate;
    pDeviceInfo->maxSampleRate = sampleRate;
    pDeviceInfo->formatCount = 1;
    if (bitsPerSample == 8) {
        pDeviceInfo->formats[0] = ma_format_u8;
    } else if (bitsPerSample == 16) {
        pDeviceInfo->formats[0] = ma_format_s16;
    } else if (bitsPerSample == 24) {
        pDeviceInfo->formats[0] = ma_format_s24;
    } else if (bitsPerSample == 32) {
        pDeviceInfo->formats[0] = ma_format_s32;
    } else {
        return MA_FORMAT_NOT_SUPPORTED;
    }

    return MA_SUCCESS;
}

ma_result ma_context_get_device_info_from_WAVEOUTCAPS2(ma_context* pContext, MA_WAVEOUTCAPS2A* pCaps, ma_device_info* pDeviceInfo)
{
    MA_WAVECAPSA caps;

    ma_assert(pContext != NULL);
    ma_assert(pCaps != NULL);
    ma_assert(pDeviceInfo != NULL);

    ma_copy_memory(caps.szPname, pCaps->szPname, sizeof(caps.szPname));
    caps.dwFormats = pCaps->dwFormats;
    caps.wChannels = pCaps->wChannels;
    caps.NameGuid = pCaps->NameGuid;
    return ma_context_get_device_info_from_WAVECAPS(pContext, &caps, pDeviceInfo);
}

ma_result ma_context_get_device_info_from_WAVEINCAPS2(ma_context* pContext, MA_WAVEINCAPS2A* pCaps, ma_device_info* pDeviceInfo)
{
    MA_WAVECAPSA caps;

    ma_assert(pContext != NULL);
    ma_assert(pCaps != NULL);
    ma_assert(pDeviceInfo != NULL);

    ma_copy_memory(caps.szPname, pCaps->szPname, sizeof(caps.szPname));
    caps.dwFormats = pCaps->dwFormats;
    caps.wChannels = pCaps->wChannels;
    caps.NameGuid = pCaps->NameGuid;
    return ma_context_get_device_info_from_WAVECAPS(pContext, &caps, pDeviceInfo);
}


ma_bool32 ma_context_is_device_id_equal__winmm(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return pID0->winmm == pID1->winmm;
}

ma_result ma_context_enumerate_devices__winmm(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    UINT playbackDeviceCount;
    UINT captureDeviceCount;
    UINT iPlaybackDevice;
    UINT iCaptureDevice;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);

    /* Playback. */
    playbackDeviceCount = ((MA_PFN_waveOutGetNumDevs)pContext->winmm.waveOutGetNumDevs)();
    for (iPlaybackDevice = 0; iPlaybackDevice < playbackDeviceCount; ++iPlaybackDevice) {
        MMRESULT result;
        MA_WAVEOUTCAPS2A caps;

        ma_zero_object(&caps);

        result = ((MA_PFN_waveOutGetDevCapsA)pContext->winmm.waveOutGetDevCapsA)(iPlaybackDevice, (WAVEOUTCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            ma_device_info deviceInfo;

            ma_zero_object(&deviceInfo);
            deviceInfo.id.winmm = iPlaybackDevice;

            if (ma_context_get_device_info_from_WAVEOUTCAPS2(pContext, &caps, &deviceInfo) == MA_SUCCESS) {
                ma_bool32 cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
                if (cbResult == MA_FALSE) {
                    return MA_SUCCESS; /* Enumeration was stopped. */
                }
            }
        }
    }

    /* Capture. */
    captureDeviceCount = ((MA_PFN_waveInGetNumDevs)pContext->winmm.waveInGetNumDevs)();
    for (iCaptureDevice = 0; iCaptureDevice < captureDeviceCount; ++iCaptureDevice) {
        MMRESULT result;
        MA_WAVEINCAPS2A caps;

        ma_zero_object(&caps);

        result = ((MA_PFN_waveInGetDevCapsA)pContext->winmm.waveInGetDevCapsA)(iCaptureDevice, (WAVEINCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            ma_device_info deviceInfo;

            ma_zero_object(&deviceInfo);
            deviceInfo.id.winmm = iCaptureDevice;

            if (ma_context_get_device_info_from_WAVEINCAPS2(pContext, &caps, &deviceInfo) == MA_SUCCESS) {
                ma_bool32 cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
                if (cbResult == MA_FALSE) {
                    return MA_SUCCESS; /* Enumeration was stopped. */
                }
            }
        }
    }

    return MA_SUCCESS;
}

ma_result ma_context_get_device_info__winmm(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    UINT winMMDeviceID;

    ma_assert(pContext != NULL);

    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    winMMDeviceID = 0;
    if (pDeviceID != NULL) {
        winMMDeviceID = (UINT)pDeviceID->winmm;
    }

    pDeviceInfo->id.winmm = winMMDeviceID;

    if (deviceType == ma_device_type_playback) {
        MMRESULT result;
        MA_WAVEOUTCAPS2A caps;

        ma_zero_object(&caps);
        
        result = ((MA_PFN_waveOutGetDevCapsA)pContext->winmm.waveOutGetDevCapsA)(winMMDeviceID, (WAVEOUTCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            return ma_context_get_device_info_from_WAVEOUTCAPS2(pContext, &caps, pDeviceInfo);
        }
    } else {
        MMRESULT result;
        MA_WAVEINCAPS2A caps;

        ma_zero_object(&caps);
        
        result = ((MA_PFN_waveInGetDevCapsA)pContext->winmm.waveInGetDevCapsA)(winMMDeviceID, (WAVEINCAPSA*)&caps, sizeof(caps));
        if (result == MMSYSERR_NOERROR) {
            return ma_context_get_device_info_from_WAVEINCAPS2(pContext, &caps, pDeviceInfo);
        }
    }

    return MA_NO_DEVICE;
}


void ma_device_uninit__winmm(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((MA_PFN_waveInClose)pDevice->pContext->winmm.waveInClose)((HWAVEIN)pDevice->winmm.hDeviceCapture);
        CloseHandle((HANDLE)pDevice->winmm.hEventCapture);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ((MA_PFN_waveOutReset)pDevice->pContext->winmm.waveOutReset)((HWAVEOUT)pDevice->winmm.hDevicePlayback);
        ((MA_PFN_waveOutClose)pDevice->pContext->winmm.waveOutClose)((HWAVEOUT)pDevice->winmm.hDevicePlayback);
        CloseHandle((HANDLE)pDevice->winmm.hEventPlayback);
    }

    ma_free(pDevice->winmm._pHeapData);

    ma_zero_object(&pDevice->winmm);   /* Safety. */
}

ma_result ma_device_init__winmm(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    const char* errorMsg = "";
    ma_result errorCode = MA_ERROR;
    ma_result result = MA_SUCCESS;
    ma_uint32 heapSize;
    UINT winMMDeviceIDPlayback = 0;
    UINT winMMDeviceIDCapture  = 0;
    ma_uint32 bufferSizeInMilliseconds;

    ma_assert(pDevice != NULL);
    ma_zero_object(&pDevice->winmm);

    /* No exlusive mode with WinMM. */
    if (((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.shareMode == ma_share_mode_exclusive) ||
        ((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.shareMode  == ma_share_mode_exclusive)) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    bufferSizeInMilliseconds = pConfig->bufferSizeInMilliseconds;
    if (bufferSizeInMilliseconds == 0) {
        bufferSizeInMilliseconds = ma_calculate_buffer_size_in_milliseconds_from_frames(pConfig->bufferSizeInFrames, pConfig->sampleRate);
    }
    
    /* WinMM has horrible latency. */
    if (pDevice->usingDefaultBufferSize) {
        if (pConfig->performanceProfile == ma_performance_profile_low_latency) {
            bufferSizeInMilliseconds =  40 * pConfig->periods;
        } else {
            bufferSizeInMilliseconds = 400 * pConfig->periods;
        }
    }


    if (pConfig->playback.pDeviceID != NULL) {
        winMMDeviceIDPlayback = (UINT)pConfig->playback.pDeviceID->winmm;
    }
    if (pConfig->capture.pDeviceID != NULL) {
        winMMDeviceIDCapture = (UINT)pConfig->capture.pDeviceID->winmm;
    }

    /* The capture device needs to be initialized first. */
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        WAVEINCAPSA caps;
        WAVEFORMATEX wf;
        MMRESULT resultMM;

        /* We use an event to know when a new fragment needs to be enqueued. */
        pDevice->winmm.hEventCapture = (ma_handle)CreateEvent(NULL, TRUE, TRUE, NULL);
        if (pDevice->winmm.hEventCapture == NULL) {
            errorMsg = "[WinMM] Failed to create event for fragment enqueing for the capture device.", errorCode = MA_FAILED_TO_CREATE_EVENT;
            goto on_error;
        }

        /* The format should be based on the device's actual format. */
        if (((MA_PFN_waveInGetDevCapsA)pContext->winmm.waveInGetDevCapsA)(winMMDeviceIDCapture, &caps, sizeof(caps)) != MMSYSERR_NOERROR) {
            errorMsg = "[WinMM] Failed to retrieve internal device caps.", errorCode = MA_FORMAT_NOT_SUPPORTED;
            goto on_error;
        }

        result = ma_formats_flags_to_WAVEFORMATEX__winmm(caps.dwFormats, caps.wChannels, &wf);
        if (result != MA_SUCCESS) {
            errorMsg = "[WinMM] Could not find appropriate format for internal device.", errorCode = result;
            goto on_error;
        }

        resultMM = ((MA_PFN_waveInOpen)pDevice->pContext->winmm.waveInOpen)((LPHWAVEIN)&pDevice->winmm.hDeviceCapture, winMMDeviceIDCapture, &wf, (DWORD_PTR)pDevice->winmm.hEventCapture, (DWORD_PTR)pDevice, CALLBACK_EVENT | WAVE_ALLOWSYNC);
        if (resultMM != MMSYSERR_NOERROR) {
            errorMsg = "[WinMM] Failed to open capture device.", errorCode = MA_FAILED_TO_OPEN_BACKEND_DEVICE;
            goto on_error;
        }

        pDevice->capture.internalFormat     = ma_format_from_WAVEFORMATEX(&wf);
        pDevice->capture.internalChannels   = wf.nChannels;
        pDevice->capture.internalSampleRate = wf.nSamplesPerSec;
        ma_get_standard_channel_map(ma_standard_channel_map_microsoft, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);
        pDevice->capture.internalPeriods = pConfig->periods;
        pDevice->capture.internalBufferSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(bufferSizeInMilliseconds, pDevice->capture.internalSampleRate);
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        WAVEOUTCAPSA caps;
        WAVEFORMATEX wf;
        MMRESULT resultMM;

        /* We use an event to know when a new fragment needs to be enqueued. */
        pDevice->winmm.hEventPlayback = (ma_handle)CreateEvent(NULL, TRUE, TRUE, NULL);
        if (pDevice->winmm.hEventPlayback == NULL) {
            errorMsg = "[WinMM] Failed to create event for fragment enqueing for the playback device.", errorCode = MA_FAILED_TO_CREATE_EVENT;
            goto on_error;
        }

        /* The format should be based on the device's actual format. */
        if (((MA_PFN_waveOutGetDevCapsA)pContext->winmm.waveOutGetDevCapsA)(winMMDeviceIDPlayback, &caps, sizeof(caps)) != MMSYSERR_NOERROR) {
            errorMsg = "[WinMM] Failed to retrieve internal device caps.", errorCode = MA_FORMAT_NOT_SUPPORTED;
            goto on_error;
        }

        result = ma_formats_flags_to_WAVEFORMATEX__winmm(caps.dwFormats, caps.wChannels, &wf);
        if (result != MA_SUCCESS) {
            errorMsg = "[WinMM] Could not find appropriate format for internal device.", errorCode = result;
            goto on_error;
        }

        resultMM = ((MA_PFN_waveOutOpen)pContext->winmm.waveOutOpen)((LPHWAVEOUT)&pDevice->winmm.hDevicePlayback, winMMDeviceIDPlayback, &wf, (DWORD_PTR)pDevice->winmm.hEventPlayback, (DWORD_PTR)pDevice, CALLBACK_EVENT | WAVE_ALLOWSYNC);
        if (resultMM != MMSYSERR_NOERROR) {
            errorMsg = "[WinMM] Failed to open playback device.", errorCode = MA_FAILED_TO_OPEN_BACKEND_DEVICE;
            goto on_error;
        }

        pDevice->playback.internalFormat     = ma_format_from_WAVEFORMATEX(&wf);
        pDevice->playback.internalChannels   = wf.nChannels;
        pDevice->playback.internalSampleRate = wf.nSamplesPerSec;
        ma_get_standard_channel_map(ma_standard_channel_map_microsoft, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);
        pDevice->playback.internalPeriods = pConfig->periods;
        pDevice->playback.internalBufferSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(bufferSizeInMilliseconds, pDevice->playback.internalSampleRate);
    }

    /*
    The heap allocated data is allocated like so:
    
    [Capture WAVEHDRs][Playback WAVEHDRs][Capture Intermediary Buffer][Playback Intermediary Buffer]
    */
    heapSize = 0;
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        heapSize += sizeof(WAVEHDR)*pDevice->capture.internalPeriods + (pDevice->capture.internalBufferSizeInFrames*ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));
    }
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        heapSize += sizeof(WAVEHDR)*pDevice->playback.internalPeriods + (pDevice->playback.internalBufferSizeInFrames*ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));
    }

    pDevice->winmm._pHeapData = (ma_uint8*)ma_malloc(heapSize);
    if (pDevice->winmm._pHeapData == NULL) {
        errorMsg = "[WinMM] Failed to allocate memory for the intermediary buffer.", errorCode = MA_OUT_OF_MEMORY;
        goto on_error;
    }

    ma_zero_memory(pDevice->winmm._pHeapData, heapSize);

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_uint32 iPeriod;

        if (pConfig->deviceType == ma_device_type_capture) {
            pDevice->winmm.pWAVEHDRCapture            = pDevice->winmm._pHeapData;
            pDevice->winmm.pIntermediaryBufferCapture = pDevice->winmm._pHeapData + (sizeof(WAVEHDR)*(pDevice->capture.internalPeriods));
        } else {
            pDevice->winmm.pWAVEHDRCapture            = pDevice->winmm._pHeapData;
            pDevice->winmm.pIntermediaryBufferCapture = pDevice->winmm._pHeapData + (sizeof(WAVEHDR)*(pDevice->capture.internalPeriods + pDevice->playback.internalPeriods));
        }

        /* Prepare headers. */
        for (iPeriod = 0; iPeriod < pDevice->capture.internalPeriods; ++iPeriod) {
            ma_uint32 fragmentSizeInBytes = ma_get_fragment_size_in_bytes(pDevice->capture.internalBufferSizeInFrames, pDevice->capture.internalPeriods, pDevice->capture.internalFormat, pDevice->capture.internalChannels);

            ((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod].lpData         = (LPSTR)(pDevice->winmm.pIntermediaryBufferCapture + (fragmentSizeInBytes*iPeriod));
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod].dwBufferLength = fragmentSizeInBytes;
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod].dwFlags        = 0L;
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod].dwLoops        = 0L;
            ((MA_PFN_waveInPrepareHeader)pContext->winmm.waveInPrepareHeader)((HWAVEIN)pDevice->winmm.hDeviceCapture, &((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod], sizeof(WAVEHDR));

            /*
            The user data of the WAVEHDR structure is a single flag the controls whether or not it is ready for writing. Consider it to be named "isLocked". A value of 0 means
            it's unlocked and available for writing. A value of 1 means it's locked.
            */
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod].dwUser = 0;
        }
    }
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_uint32 iPeriod;

        if (pConfig->deviceType == ma_device_type_playback) {
            pDevice->winmm.pWAVEHDRPlayback            = pDevice->winmm._pHeapData;
            pDevice->winmm.pIntermediaryBufferPlayback = pDevice->winmm._pHeapData + (sizeof(WAVEHDR)*pDevice->playback.internalPeriods);
        } else {
            pDevice->winmm.pWAVEHDRPlayback            = pDevice->winmm._pHeapData + (sizeof(WAVEHDR)*(pDevice->capture.internalPeriods));
            pDevice->winmm.pIntermediaryBufferPlayback = pDevice->winmm._pHeapData + (sizeof(WAVEHDR)*(pDevice->capture.internalPeriods + pDevice->playback.internalPeriods)) + (pDevice->playback.internalBufferSizeInFrames*ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));
        }

        /* Prepare headers. */
        for (iPeriod = 0; iPeriod < pDevice->playback.internalPeriods; ++iPeriod) {
            ma_uint32 fragmentSizeInBytes = ma_get_fragment_size_in_bytes(pDevice->playback.internalBufferSizeInFrames, pDevice->playback.internalPeriods, pDevice->playback.internalFormat, pDevice->playback.internalChannels);

            ((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod].lpData         = (LPSTR)(pDevice->winmm.pIntermediaryBufferPlayback + (fragmentSizeInBytes*iPeriod));
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod].dwBufferLength = fragmentSizeInBytes;
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod].dwFlags        = 0L;
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod].dwLoops        = 0L;
            ((MA_PFN_waveOutPrepareHeader)pContext->winmm.waveOutPrepareHeader)((HWAVEOUT)pDevice->winmm.hDevicePlayback, &((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod], sizeof(WAVEHDR));

            /*
            The user data of the WAVEHDR structure is a single flag the controls whether or not it is ready for writing. Consider it to be named "isLocked". A value of 0 means
            it's unlocked and available for writing. A value of 1 means it's locked.
            */
            ((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod].dwUser = 0;
        }
    }

    return MA_SUCCESS;

on_error:
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        if (pDevice->winmm.pWAVEHDRCapture != NULL) {
            ma_uint32 iPeriod;
            for (iPeriod = 0; iPeriod < pDevice->capture.internalPeriods; ++iPeriod) {
                ((MA_PFN_waveInUnprepareHeader)pContext->winmm.waveInUnprepareHeader)((HWAVEIN)pDevice->winmm.hDeviceCapture, &((WAVEHDR*)pDevice->winmm.pWAVEHDRCapture)[iPeriod], sizeof(WAVEHDR));
            }
        }

        ((MA_PFN_waveInClose)pContext->winmm.waveInClose)((HWAVEIN)pDevice->winmm.hDeviceCapture);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        if (pDevice->winmm.pWAVEHDRCapture != NULL) {
            ma_uint32 iPeriod;
            for (iPeriod = 0; iPeriod < pDevice->playback.internalPeriods; ++iPeriod) {
                ((MA_PFN_waveOutUnprepareHeader)pContext->winmm.waveOutUnprepareHeader)((HWAVEOUT)pDevice->winmm.hDevicePlayback, &((WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback)[iPeriod], sizeof(WAVEHDR));
            }
        }

        ((MA_PFN_waveOutClose)pContext->winmm.waveOutClose)((HWAVEOUT)pDevice->winmm.hDevicePlayback);
    }

    ma_free(pDevice->winmm._pHeapData);
    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, errorMsg, errorCode);
}

ma_result ma_device_stop__winmm(ma_device* pDevice)
{
    MMRESULT resultMM;

    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        if (pDevice->winmm.hDeviceCapture == NULL) {
            return MA_INVALID_ARGS;
        }

        resultMM = ((MA_PFN_waveInReset)pDevice->pContext->winmm.waveInReset)((HWAVEIN)pDevice->winmm.hDeviceCapture);
        if (resultMM != MMSYSERR_NOERROR) {
            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] WARNING: Failed to reset capture device.", ma_result_from_MMRESULT(resultMM));
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        if (pDevice->winmm.hDevicePlayback == NULL) {
            return MA_INVALID_ARGS;
        }

        resultMM = ((MA_PFN_waveOutReset)pDevice->pContext->winmm.waveOutReset)((HWAVEOUT)pDevice->winmm.hDevicePlayback);
        if (resultMM != MMSYSERR_NOERROR) {
            ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] WARNING: Failed to reset playback device.", ma_result_from_MMRESULT(resultMM));
        }
    }

    ma_atomic_exchange_32(&pDevice->winmm.isStarted, MA_FALSE);
    return MA_SUCCESS;
}

ma_result ma_device_write__winmm(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount)
{
    ma_result result = MA_SUCCESS;
    MMRESULT resultMM;
    ma_uint32 totalFramesWritten;
    WAVEHDR* pWAVEHDR;

    ma_assert(pDevice != NULL);
    ma_assert(pPCMFrames != NULL);

    pWAVEHDR = (WAVEHDR*)pDevice->winmm.pWAVEHDRPlayback;

    /* Keep processing as much data as possible. */
    totalFramesWritten = 0;
    while (totalFramesWritten < frameCount) {
        /* If the current header has some space available we need to write part of it. */
        if (pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwUser == 0) { /* 0 = unlocked. */
            /*
            This header has room in it. We copy as much of it as we can. If we end up fully consuming the buffer we need to
            write it out and move on to the next iteration.
            */
            ma_uint32 bpf = ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
            ma_uint32 framesRemainingInHeader = (pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwBufferLength/bpf) - pDevice->winmm.headerFramesConsumedPlayback;

            ma_uint32 framesToCopy = ma_min(framesRemainingInHeader, (frameCount - totalFramesWritten));
            const void* pSrc = ma_offset_ptr(pPCMFrames, totalFramesWritten*bpf);
            void* pDst = ma_offset_ptr(pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].lpData, pDevice->winmm.headerFramesConsumedPlayback*bpf);
            ma_copy_memory(pDst, pSrc, framesToCopy*bpf);

            pDevice->winmm.headerFramesConsumedPlayback += framesToCopy;
            totalFramesWritten += framesToCopy;

            /* If we've consumed the buffer entirely we need to write it out to the device. */
            if (pDevice->winmm.headerFramesConsumedPlayback == (pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwBufferLength/bpf)) {
                pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwUser = 1;            /* 1 = locked. */
                pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwFlags &= ~WHDR_DONE; /* <-- Need to make sure the WHDR_DONE flag is unset. */

                /* Make sure the event is reset to a non-signaled state to ensure we don't prematurely return from WaitForSingleObject(). */
                ResetEvent((HANDLE)pDevice->winmm.hEventPlayback);

                /* The device will be started here. */
                resultMM = ((MA_PFN_waveOutWrite)pDevice->pContext->winmm.waveOutWrite)((HWAVEOUT)pDevice->winmm.hDevicePlayback, &pWAVEHDR[pDevice->winmm.iNextHeaderPlayback], sizeof(WAVEHDR));
                if (resultMM != MMSYSERR_NOERROR) {
                    result = ma_result_from_MMRESULT(resultMM);
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] waveOutWrite() failed.", result);
                    break;
                }
                ma_atomic_exchange_32(&pDevice->winmm.isStarted, MA_TRUE);

                /* Make sure we move to the next header. */
                pDevice->winmm.iNextHeaderPlayback = (pDevice->winmm.iNextHeaderPlayback + 1) % pDevice->playback.internalPeriods;
                pDevice->winmm.headerFramesConsumedPlayback = 0;
            }

            /* If at this point we have consumed the entire input buffer we can return. */
            ma_assert(totalFramesWritten <= frameCount);
            if (totalFramesWritten == frameCount) {
                break;
            }

            /* Getting here means there's more to process. */
            continue;
        }

        /* Getting here means there isn't enough room in the buffer and we need to wait for one to become available. */
        if (WaitForSingleObject((HANDLE)pDevice->winmm.hEventPlayback, INFINITE) != WAIT_OBJECT_0) {
            result = MA_ERROR;
            break;
        }

        /* Something happened. If the next buffer has been marked as done we need to reset a bit of state. */
        if ((pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwFlags & WHDR_DONE) != 0) {
            pWAVEHDR[pDevice->winmm.iNextHeaderPlayback].dwUser = 0;    /* 0 = unlocked (make it available for writing). */
            pDevice->winmm.headerFramesConsumedPlayback = 0;
        }

        /* If the device has been stopped we need to break. */
        if (!pDevice->winmm.isStarted) {
            break;
        }
    }

    return result;
}

ma_result ma_device_read__winmm(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount)
{
    ma_result result = MA_SUCCESS;
    MMRESULT resultMM;
    ma_uint32 totalFramesRead;
    WAVEHDR* pWAVEHDR;

    ma_assert(pDevice != NULL);
    ma_assert(pPCMFrames != NULL);

    pWAVEHDR = (WAVEHDR*)pDevice->winmm.pWAVEHDRCapture;

    /* We want to start the device immediately. */
    if (!pDevice->winmm.isStarted) {
        ma_uint32 iPeriod;

        /* Make sure the event is reset to a non-signaled state to ensure we don't prematurely return from WaitForSingleObject(). */
        ResetEvent((HANDLE)pDevice->winmm.hEventCapture);

        /* To start the device we attach all of the buffers and then start it. As the buffers are filled with data we will get notifications. */
        for (iPeriod = 0; iPeriod < pDevice->capture.internalPeriods; ++iPeriod) {
            resultMM = ((MA_PFN_waveInAddBuffer)pDevice->pContext->winmm.waveInAddBuffer)((HWAVEIN)pDevice->winmm.hDeviceCapture, &((LPWAVEHDR)pDevice->winmm.pWAVEHDRCapture)[iPeriod], sizeof(WAVEHDR));
            if (resultMM != MMSYSERR_NOERROR) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] Failed to attach input buffers to capture device in preparation for capture.", ma_result_from_MMRESULT(resultMM));
            }

            /* Make sure all of the buffers start out locked. We don't want to access them until the backend tells us we can. */
            pWAVEHDR[iPeriod].dwUser = 1;   /* 1 = locked. */
        }

        /* Capture devices need to be explicitly started, unlike playback devices. */
        resultMM = ((MA_PFN_waveInStart)pDevice->pContext->winmm.waveInStart)((HWAVEIN)pDevice->winmm.hDeviceCapture);
        if (resultMM != MMSYSERR_NOERROR) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] Failed to start backend device.", ma_result_from_MMRESULT(resultMM));
        }

        ma_atomic_exchange_32(&pDevice->winmm.isStarted, MA_TRUE);
    }

    /* Keep processing as much data as possible. */
    totalFramesRead = 0;
    while (totalFramesRead < frameCount) {
        /* If the current header has some space available we need to write part of it. */
        if (pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwUser == 0) { /* 0 = unlocked. */
            /* The buffer is available for reading. If we fully consume it we need to add it back to the buffer. */
            ma_uint32 bpf = ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
            ma_uint32 framesRemainingInHeader = (pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwBufferLength/bpf) - pDevice->winmm.headerFramesConsumedCapture;

            ma_uint32 framesToCopy = ma_min(framesRemainingInHeader, (frameCount - totalFramesRead));
            const void* pSrc = ma_offset_ptr(pWAVEHDR[pDevice->winmm.iNextHeaderCapture].lpData, pDevice->winmm.headerFramesConsumedCapture*bpf);
            void* pDst = ma_offset_ptr(pPCMFrames, totalFramesRead*bpf);
            ma_copy_memory(pDst, pSrc, framesToCopy*bpf);

            pDevice->winmm.headerFramesConsumedCapture += framesToCopy;
            totalFramesRead += framesToCopy;

            /* If we've consumed the buffer entirely we need to add it back to the device. */
            if (pDevice->winmm.headerFramesConsumedCapture == (pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwBufferLength/bpf)) {
                pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwUser = 1;            /* 1 = locked. */
                pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwFlags &= ~WHDR_DONE; /* <-- Need to make sure the WHDR_DONE flag is unset. */

                /* Make sure the event is reset to a non-signaled state to ensure we don't prematurely return from WaitForSingleObject(). */
                ResetEvent((HANDLE)pDevice->winmm.hEventCapture);

                /* The device will be started here. */
                resultMM = ((MA_PFN_waveInAddBuffer)pDevice->pContext->winmm.waveInAddBuffer)((HWAVEIN)pDevice->winmm.hDeviceCapture, &((LPWAVEHDR)pDevice->winmm.pWAVEHDRCapture)[pDevice->winmm.iNextHeaderCapture], sizeof(WAVEHDR));
                if (resultMM != MMSYSERR_NOERROR) {
                    result = ma_result_from_MMRESULT(resultMM);
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[WinMM] waveInAddBuffer() failed.", result);
                    break;
                }

                /* Make sure we move to the next header. */
                pDevice->winmm.iNextHeaderCapture = (pDevice->winmm.iNextHeaderCapture + 1) % pDevice->capture.internalPeriods;
                pDevice->winmm.headerFramesConsumedCapture = 0;
            }

            /* If at this point we have filled the entire input buffer we can return. */
            ma_assert(totalFramesRead <= frameCount);
            if (totalFramesRead == frameCount) {
                break;
            }

            /* Getting here means there's more to process. */
            continue;
        }

        /* Getting here means there isn't enough any data left to send to the client which means we need to wait for more. */
        if (WaitForSingleObject((HANDLE)pDevice->winmm.hEventCapture, INFINITE) != WAIT_OBJECT_0) {
            result = MA_ERROR;
            break;
        }

        /* Something happened. If the next buffer has been marked as done we need to reset a bit of state. */
        if ((pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwFlags & WHDR_DONE) != 0) {
            pWAVEHDR[pDevice->winmm.iNextHeaderCapture].dwUser = 0;    /* 0 = unlocked (make it available for reading). */
            pDevice->winmm.headerFramesConsumedCapture = 0;
        }

        /* If the device has been stopped we need to break. */
        if (!pDevice->winmm.isStarted) {
            break;
        }
    }

    return result;
}

ma_result ma_context_uninit__winmm(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_winmm);

    ma_dlclose(pContext->winmm.hWinMM);
    return MA_SUCCESS;
}

ma_result ma_context_init__winmm(const ma_context_config* pConfig, ma_context* pContext)
{
    ma_assert(pContext != NULL);

    (void)pConfig;

    pContext->winmm.hWinMM = ma_dlopen("winmm.dll");
    if (pContext->winmm.hWinMM == NULL) {
        return MA_NO_BACKEND;
    }

    pContext->winmm.waveOutGetNumDevs      = ma_dlsym(pContext->winmm.hWinMM, "waveOutGetNumDevs");
    pContext->winmm.waveOutGetDevCapsA     = ma_dlsym(pContext->winmm.hWinMM, "waveOutGetDevCapsA");
    pContext->winmm.waveOutOpen            = ma_dlsym(pContext->winmm.hWinMM, "waveOutOpen");
    pContext->winmm.waveOutClose           = ma_dlsym(pContext->winmm.hWinMM, "waveOutClose");
    pContext->winmm.waveOutPrepareHeader   = ma_dlsym(pContext->winmm.hWinMM, "waveOutPrepareHeader");
    pContext->winmm.waveOutUnprepareHeader = ma_dlsym(pContext->winmm.hWinMM, "waveOutUnprepareHeader");
    pContext->winmm.waveOutWrite           = ma_dlsym(pContext->winmm.hWinMM, "waveOutWrite");
    pContext->winmm.waveOutReset           = ma_dlsym(pContext->winmm.hWinMM, "waveOutReset");
    pContext->winmm.waveInGetNumDevs       = ma_dlsym(pContext->winmm.hWinMM, "waveInGetNumDevs");
    pContext->winmm.waveInGetDevCapsA      = ma_dlsym(pContext->winmm.hWinMM, "waveInGetDevCapsA");
    pContext->winmm.waveInOpen             = ma_dlsym(pContext->winmm.hWinMM, "waveInOpen");
    pContext->winmm.waveInClose            = ma_dlsym(pContext->winmm.hWinMM, "waveInClose");
    pContext->winmm.waveInPrepareHeader    = ma_dlsym(pContext->winmm.hWinMM, "waveInPrepareHeader");
    pContext->winmm.waveInUnprepareHeader  = ma_dlsym(pContext->winmm.hWinMM, "waveInUnprepareHeader");
    pContext->winmm.waveInAddBuffer        = ma_dlsym(pContext->winmm.hWinMM, "waveInAddBuffer");
    pContext->winmm.waveInStart            = ma_dlsym(pContext->winmm.hWinMM, "waveInStart");
    pContext->winmm.waveInReset            = ma_dlsym(pContext->winmm.hWinMM, "waveInReset");

    pContext->onUninit        = ma_context_uninit__winmm;
    pContext->onDeviceIDEqual = ma_context_is_device_id_equal__winmm;
    pContext->onEnumDevices   = ma_context_enumerate_devices__winmm;
    pContext->onGetDeviceInfo = ma_context_get_device_info__winmm;
    pContext->onDeviceInit    = ma_device_init__winmm;
    pContext->onDeviceUninit  = ma_device_uninit__winmm;
    pContext->onDeviceStart   = NULL; /* Not used. Started in onDeviceWrite/onDeviceRead. */
    pContext->onDeviceStop    = ma_device_stop__winmm;
    pContext->onDeviceWrite   = ma_device_write__winmm;
    pContext->onDeviceRead    = ma_device_read__winmm;

    return MA_SUCCESS;
}
#endif




/******************************************************************************

ALSA Backend

******************************************************************************/
#ifdef MA_HAS_ALSA

#ifdef MA_NO_RUNTIME_LINKING
#include <alsa/asoundlib.h>
typedef snd_pcm_uframes_t                       ma_snd_pcm_uframes_t;
typedef snd_pcm_sframes_t                       ma_snd_pcm_sframes_t;
typedef snd_pcm_stream_t                        ma_snd_pcm_stream_t;
typedef snd_pcm_format_t                        ma_snd_pcm_format_t;
typedef snd_pcm_access_t                        ma_snd_pcm_access_t;
typedef snd_pcm_t                               ma_snd_pcm_t;
typedef snd_pcm_hw_params_t                     ma_snd_pcm_hw_params_t;
typedef snd_pcm_sw_params_t                     ma_snd_pcm_sw_params_t;
typedef snd_pcm_format_mask_t                   ma_snd_pcm_format_mask_t;
typedef snd_pcm_info_t                          ma_snd_pcm_info_t;
typedef snd_pcm_channel_area_t                  ma_snd_pcm_channel_area_t;
typedef snd_pcm_chmap_t                         ma_snd_pcm_chmap_t;

/* snd_pcm_stream_t */
#define MA_SND_PCM_STREAM_PLAYBACK              SND_PCM_STREAM_PLAYBACK
#define MA_SND_PCM_STREAM_CAPTURE               SND_PCM_STREAM_CAPTURE

/* snd_pcm_format_t */
#define MA_SND_PCM_FORMAT_UNKNOWN               SND_PCM_FORMAT_UNKNOWN
#define MA_SND_PCM_FORMAT_U8                    SND_PCM_FORMAT_U8
#define MA_SND_PCM_FORMAT_S16_LE                SND_PCM_FORMAT_S16_LE
#define MA_SND_PCM_FORMAT_S16_BE                SND_PCM_FORMAT_S16_BE
#define MA_SND_PCM_FORMAT_S24_LE                SND_PCM_FORMAT_S24_LE
#define MA_SND_PCM_FORMAT_S24_BE                SND_PCM_FORMAT_S24_BE
#define MA_SND_PCM_FORMAT_S32_LE                SND_PCM_FORMAT_S32_LE
#define MA_SND_PCM_FORMAT_S32_BE                SND_PCM_FORMAT_S32_BE
#define MA_SND_PCM_FORMAT_FLOAT_LE              SND_PCM_FORMAT_FLOAT_LE
#define MA_SND_PCM_FORMAT_FLOAT_BE              SND_PCM_FORMAT_FLOAT_BE
#define MA_SND_PCM_FORMAT_FLOAT64_LE            SND_PCM_FORMAT_FLOAT64_LE
#define MA_SND_PCM_FORMAT_FLOAT64_BE            SND_PCM_FORMAT_FLOAT64_BE
#define MA_SND_PCM_FORMAT_MU_LAW                SND_PCM_FORMAT_MU_LAW
#define MA_SND_PCM_FORMAT_A_LAW                 SND_PCM_FORMAT_A_LAW
#define MA_SND_PCM_FORMAT_S24_3LE               SND_PCM_FORMAT_S24_3LE
#define MA_SND_PCM_FORMAT_S24_3BE               SND_PCM_FORMAT_S24_3BE

/* ma_snd_pcm_access_t */
#define MA_SND_PCM_ACCESS_MMAP_INTERLEAVED      SND_PCM_ACCESS_MMAP_INTERLEAVED
#define MA_SND_PCM_ACCESS_MMAP_NONINTERLEAVED   SND_PCM_ACCESS_MMAP_NONINTERLEAVED
#define MA_SND_PCM_ACCESS_MMAP_COMPLEX          SND_PCM_ACCESS_MMAP_COMPLEX
#define MA_SND_PCM_ACCESS_RW_INTERLEAVED        SND_PCM_ACCESS_RW_INTERLEAVED
#define MA_SND_PCM_ACCESS_RW_NONINTERLEAVED     SND_PCM_ACCESS_RW_NONINTERLEAVED

/* Channel positions. */
#define MA_SND_CHMAP_UNKNOWN                    SND_CHMAP_UNKNOWN
#define MA_SND_CHMAP_NA                         SND_CHMAP_NA
#define MA_SND_CHMAP_MONO                       SND_CHMAP_MONO
#define MA_SND_CHMAP_FL                         SND_CHMAP_FL
#define MA_SND_CHMAP_FR                         SND_CHMAP_FR
#define MA_SND_CHMAP_RL                         SND_CHMAP_RL
#define MA_SND_CHMAP_RR                         SND_CHMAP_RR
#define MA_SND_CHMAP_FC                         SND_CHMAP_FC
#define MA_SND_CHMAP_LFE                        SND_CHMAP_LFE
#define MA_SND_CHMAP_SL                         SND_CHMAP_SL
#define MA_SND_CHMAP_SR                         SND_CHMAP_SR
#define MA_SND_CHMAP_RC                         SND_CHMAP_RC
#define MA_SND_CHMAP_FLC                        SND_CHMAP_FLC
#define MA_SND_CHMAP_FRC                        SND_CHMAP_FRC
#define MA_SND_CHMAP_RLC                        SND_CHMAP_RLC
#define MA_SND_CHMAP_RRC                        SND_CHMAP_RRC
#define MA_SND_CHMAP_FLW                        SND_CHMAP_FLW
#define MA_SND_CHMAP_FRW                        SND_CHMAP_FRW
#define MA_SND_CHMAP_FLH                        SND_CHMAP_FLH
#define MA_SND_CHMAP_FCH                        SND_CHMAP_FCH
#define MA_SND_CHMAP_FRH                        SND_CHMAP_FRH
#define MA_SND_CHMAP_TC                         SND_CHMAP_TC
#define MA_SND_CHMAP_TFL                        SND_CHMAP_TFL
#define MA_SND_CHMAP_TFR                        SND_CHMAP_TFR
#define MA_SND_CHMAP_TFC                        SND_CHMAP_TFC
#define MA_SND_CHMAP_TRL                        SND_CHMAP_TRL
#define MA_SND_CHMAP_TRR                        SND_CHMAP_TRR
#define MA_SND_CHMAP_TRC                        SND_CHMAP_TRC
#define MA_SND_CHMAP_TFLC                       SND_CHMAP_TFLC
#define MA_SND_CHMAP_TFRC                       SND_CHMAP_TFRC
#define MA_SND_CHMAP_TSL                        SND_CHMAP_TSL
#define MA_SND_CHMAP_TSR                        SND_CHMAP_TSR
#define MA_SND_CHMAP_LLFE                       SND_CHMAP_LLFE
#define MA_SND_CHMAP_RLFE                       SND_CHMAP_RLFE
#define MA_SND_CHMAP_BC                         SND_CHMAP_BC
#define MA_SND_CHMAP_BLC                        SND_CHMAP_BLC
#define MA_SND_CHMAP_BRC                        SND_CHMAP_BRC

/* Open mode flags. */
#define MA_SND_PCM_NO_AUTO_RESAMPLE             SND_PCM_NO_AUTO_RESAMPLE
#define MA_SND_PCM_NO_AUTO_CHANNELS             SND_PCM_NO_AUTO_CHANNELS
#define MA_SND_PCM_NO_AUTO_FORMAT               SND_PCM_NO_AUTO_FORMAT
#else
#include <errno.h>  /* For EPIPE, etc. */
typedef unsigned long                           ma_snd_pcm_uframes_t;
typedef long                                    ma_snd_pcm_sframes_t;
typedef int                                     ma_snd_pcm_stream_t;
typedef int                                     ma_snd_pcm_format_t;
typedef int                                     ma_snd_pcm_access_t;
typedef struct ma_snd_pcm_t                     ma_snd_pcm_t;
typedef struct ma_snd_pcm_hw_params_t           ma_snd_pcm_hw_params_t;
typedef struct ma_snd_pcm_sw_params_t           ma_snd_pcm_sw_params_t;
typedef struct ma_snd_pcm_format_mask_t         ma_snd_pcm_format_mask_t;
typedef struct ma_snd_pcm_info_t                ma_snd_pcm_info_t;
typedef struct
{
    void* addr;
    unsigned int first;
    unsigned int step;
} ma_snd_pcm_channel_area_t;
typedef struct
{
    unsigned int channels;
    unsigned int pos[1];
} ma_snd_pcm_chmap_t;

/* snd_pcm_state_t */
#define MA_SND_PCM_STATE_OPEN                  0
#define MA_SND_PCM_STATE_SETUP                 1
#define MA_SND_PCM_STATE_PREPARED              2
#define MA_SND_PCM_STATE_RUNNING               3
#define MA_SND_PCM_STATE_XRUN                  4
#define MA_SND_PCM_STATE_DRAINING              5
#define MA_SND_PCM_STATE_PAUSED                6
#define MA_SND_PCM_STATE_SUSPENDED             7
#define MA_SND_PCM_STATE_DISCONNECTED          8

/* snd_pcm_stream_t */
#define MA_SND_PCM_STREAM_PLAYBACK             0
#define MA_SND_PCM_STREAM_CAPTURE              1

/* snd_pcm_format_t */
#define MA_SND_PCM_FORMAT_UNKNOWN              -1
#define MA_SND_PCM_FORMAT_U8                   1
#define MA_SND_PCM_FORMAT_S16_LE               2
#define MA_SND_PCM_FORMAT_S16_BE               3
#define MA_SND_PCM_FORMAT_S24_LE               6
#define MA_SND_PCM_FORMAT_S24_BE               7
#define MA_SND_PCM_FORMAT_S32_LE               10
#define MA_SND_PCM_FORMAT_S32_BE               11
#define MA_SND_PCM_FORMAT_FLOAT_LE             14
#define MA_SND_PCM_FORMAT_FLOAT_BE             15
#define MA_SND_PCM_FORMAT_FLOAT64_LE           16
#define MA_SND_PCM_FORMAT_FLOAT64_BE           17
#define MA_SND_PCM_FORMAT_MU_LAW               20
#define MA_SND_PCM_FORMAT_A_LAW                21
#define MA_SND_PCM_FORMAT_S24_3LE              32
#define MA_SND_PCM_FORMAT_S24_3BE              33

/* snd_pcm_access_t */
#define MA_SND_PCM_ACCESS_MMAP_INTERLEAVED     0
#define MA_SND_PCM_ACCESS_MMAP_NONINTERLEAVED  1
#define MA_SND_PCM_ACCESS_MMAP_COMPLEX         2
#define MA_SND_PCM_ACCESS_RW_INTERLEAVED       3
#define MA_SND_PCM_ACCESS_RW_NONINTERLEAVED    4

/* Channel positions. */
#define MA_SND_CHMAP_UNKNOWN                   0
#define MA_SND_CHMAP_NA                        1
#define MA_SND_CHMAP_MONO                      2
#define MA_SND_CHMAP_FL                        3
#define MA_SND_CHMAP_FR                        4
#define MA_SND_CHMAP_RL                        5
#define MA_SND_CHMAP_RR                        6
#define MA_SND_CHMAP_FC                        7
#define MA_SND_CHMAP_LFE                       8
#define MA_SND_CHMAP_SL                        9
#define MA_SND_CHMAP_SR                        10
#define MA_SND_CHMAP_RC                        11
#define MA_SND_CHMAP_FLC                       12
#define MA_SND_CHMAP_FRC                       13
#define MA_SND_CHMAP_RLC                       14
#define MA_SND_CHMAP_RRC                       15
#define MA_SND_CHMAP_FLW                       16
#define MA_SND_CHMAP_FRW                       17
#define MA_SND_CHMAP_FLH                       18
#define MA_SND_CHMAP_FCH                       19
#define MA_SND_CHMAP_FRH                       20
#define MA_SND_CHMAP_TC                        21
#define MA_SND_CHMAP_TFL                       22
#define MA_SND_CHMAP_TFR                       23
#define MA_SND_CHMAP_TFC                       24
#define MA_SND_CHMAP_TRL                       25
#define MA_SND_CHMAP_TRR                       26
#define MA_SND_CHMAP_TRC                       27
#define MA_SND_CHMAP_TFLC                      28
#define MA_SND_CHMAP_TFRC                      29
#define MA_SND_CHMAP_TSL                       30
#define MA_SND_CHMAP_TSR                       31
#define MA_SND_CHMAP_LLFE                      32
#define MA_SND_CHMAP_RLFE                      33
#define MA_SND_CHMAP_BC                        34
#define MA_SND_CHMAP_BLC                       35
#define MA_SND_CHMAP_BRC                       36

/* Open mode flags. */
#define MA_SND_PCM_NO_AUTO_RESAMPLE            0x00010000
#define MA_SND_PCM_NO_AUTO_CHANNELS            0x00020000
#define MA_SND_PCM_NO_AUTO_FORMAT              0x00040000
#endif

typedef int                  (* ma_snd_pcm_open_proc)                          (ma_snd_pcm_t **pcm, const char *name, ma_snd_pcm_stream_t stream, int mode);
typedef int                  (* ma_snd_pcm_close_proc)                         (ma_snd_pcm_t *pcm);
typedef size_t               (* ma_snd_pcm_hw_params_sizeof_proc)              (void);
typedef int                  (* ma_snd_pcm_hw_params_any_proc)                 (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params);
typedef int                  (* ma_snd_pcm_hw_params_set_format_proc)          (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, ma_snd_pcm_format_t val);
typedef int                  (* ma_snd_pcm_hw_params_set_format_first_proc)    (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, ma_snd_pcm_format_t *format);
typedef void                 (* ma_snd_pcm_hw_params_get_format_mask_proc)     (ma_snd_pcm_hw_params_t *params, ma_snd_pcm_format_mask_t *mask);
typedef int                  (* ma_snd_pcm_hw_params_set_channels_near_proc)   (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                  (* ma_snd_pcm_hw_params_set_rate_resample_proc)   (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, unsigned int val);
typedef int                  (* ma_snd_pcm_hw_params_set_rate_near_proc)       (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_set_buffer_size_near_proc)(ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, ma_snd_pcm_uframes_t *val);
typedef int                  (* ma_snd_pcm_hw_params_set_periods_near_proc)    (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_set_access_proc)          (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params, ma_snd_pcm_access_t _access);
typedef int                  (* ma_snd_pcm_hw_params_get_format_proc)          (const ma_snd_pcm_hw_params_t *params, ma_snd_pcm_format_t *format);
typedef int                  (* ma_snd_pcm_hw_params_get_channels_proc)        (const ma_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                  (* ma_snd_pcm_hw_params_get_channels_min_proc)    (const ma_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                  (* ma_snd_pcm_hw_params_get_channels_max_proc)    (const ma_snd_pcm_hw_params_t *params, unsigned int *val);
typedef int                  (* ma_snd_pcm_hw_params_get_rate_proc)            (const ma_snd_pcm_hw_params_t *params, unsigned int *rate, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_get_rate_min_proc)        (const ma_snd_pcm_hw_params_t *params, unsigned int *rate, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_get_rate_max_proc)        (const ma_snd_pcm_hw_params_t *params, unsigned int *rate, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_get_buffer_size_proc)     (const ma_snd_pcm_hw_params_t *params, ma_snd_pcm_uframes_t *val);
typedef int                  (* ma_snd_pcm_hw_params_get_periods_proc)         (const ma_snd_pcm_hw_params_t *params, unsigned int *val, int *dir);
typedef int                  (* ma_snd_pcm_hw_params_get_access_proc)          (const ma_snd_pcm_hw_params_t *params, ma_snd_pcm_access_t *_access);
typedef int                  (* ma_snd_pcm_hw_params_proc)                     (ma_snd_pcm_t *pcm, ma_snd_pcm_hw_params_t *params);
typedef size_t               (* ma_snd_pcm_sw_params_sizeof_proc)              (void);
typedef int                  (* ma_snd_pcm_sw_params_current_proc)             (ma_snd_pcm_t *pcm, ma_snd_pcm_sw_params_t *params);
typedef int                  (* ma_snd_pcm_sw_params_get_boundary_proc)        (ma_snd_pcm_sw_params_t *params, ma_snd_pcm_uframes_t* val);
typedef int                  (* ma_snd_pcm_sw_params_set_avail_min_proc)       (ma_snd_pcm_t *pcm, ma_snd_pcm_sw_params_t *params, ma_snd_pcm_uframes_t val);
typedef int                  (* ma_snd_pcm_sw_params_set_start_threshold_proc) (ma_snd_pcm_t *pcm, ma_snd_pcm_sw_params_t *params, ma_snd_pcm_uframes_t val);
typedef int                  (* ma_snd_pcm_sw_params_set_stop_threshold_proc)  (ma_snd_pcm_t *pcm, ma_snd_pcm_sw_params_t *params, ma_snd_pcm_uframes_t val);
typedef int                  (* ma_snd_pcm_sw_params_proc)                     (ma_snd_pcm_t *pcm, ma_snd_pcm_sw_params_t *params);
typedef size_t               (* ma_snd_pcm_format_mask_sizeof_proc)            (void);
typedef int                  (* ma_snd_pcm_format_mask_test_proc)              (const ma_snd_pcm_format_mask_t *mask, ma_snd_pcm_format_t val);
typedef ma_snd_pcm_chmap_t * (* ma_snd_pcm_get_chmap_proc)                     (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_state_proc)                         (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_prepare_proc)                       (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_start_proc)                         (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_drop_proc)                          (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_drain_proc)                         (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_device_name_hint_proc)                  (int card, const char *iface, void ***hints);
typedef char *               (* ma_snd_device_name_get_hint_proc)              (const void *hint, const char *id);
typedef int                  (* ma_snd_card_get_index_proc)                    (const char *name);
typedef int                  (* ma_snd_device_name_free_hint_proc)             (void **hints);
typedef int                  (* ma_snd_pcm_mmap_begin_proc)                    (ma_snd_pcm_t *pcm, const ma_snd_pcm_channel_area_t **areas, ma_snd_pcm_uframes_t *offset, ma_snd_pcm_uframes_t *frames);
typedef ma_snd_pcm_sframes_t (* ma_snd_pcm_mmap_commit_proc)                   (ma_snd_pcm_t *pcm, ma_snd_pcm_uframes_t offset, ma_snd_pcm_uframes_t frames);
typedef int                  (* ma_snd_pcm_recover_proc)                       (ma_snd_pcm_t *pcm, int err, int silent);
typedef ma_snd_pcm_sframes_t (* ma_snd_pcm_readi_proc)                         (ma_snd_pcm_t *pcm, void *buffer, ma_snd_pcm_uframes_t size);
typedef ma_snd_pcm_sframes_t (* ma_snd_pcm_writei_proc)                        (ma_snd_pcm_t *pcm, const void *buffer, ma_snd_pcm_uframes_t size);
typedef ma_snd_pcm_sframes_t (* ma_snd_pcm_avail_proc)                         (ma_snd_pcm_t *pcm);
typedef ma_snd_pcm_sframes_t (* ma_snd_pcm_avail_update_proc)                  (ma_snd_pcm_t *pcm);
typedef int                  (* ma_snd_pcm_wait_proc)                          (ma_snd_pcm_t *pcm, int timeout);
typedef int                  (* ma_snd_pcm_info_proc)                          (ma_snd_pcm_t *pcm, ma_snd_pcm_info_t* info);
typedef size_t               (* ma_snd_pcm_info_sizeof_proc)                   ();
typedef const char*          (* ma_snd_pcm_info_get_name_proc)                 (const ma_snd_pcm_info_t* info);
typedef int                  (* ma_snd_config_update_free_global_proc)         ();

/* This array specifies each of the common devices that can be used for both playback and capture. */
const char* g_maCommonDeviceNamesALSA[] = {
    "default",
    "null",
    "pulse",
    "jack"
};

/* This array allows us to blacklist specific playback devices. */
const char* g_maBlacklistedPlaybackDeviceNamesALSA[] = {
    ""
};

/* This array allows us to blacklist specific capture devices. */
const char* g_maBlacklistedCaptureDeviceNamesALSA[] = {
    ""
};


/*
This array allows miniaudio to control device-specific default buffer sizes. This uses a scaling factor. Order is important. If
any part of the string is present in the device's name, the associated scale will be used.
*/
static struct
{
    const char* name;
    float scale;
} g_maDefaultBufferSizeScalesALSA[] = {
    {"bcm2835 IEC958/HDMI", 2.0f},
    {"bcm2835 ALSA",        2.0f}
};

float ma_find_default_buffer_size_scale__alsa(const char* deviceName)
{
    size_t i;

    if (deviceName == NULL) {
        return 1;
    }

    for (i = 0; i < ma_countof(g_maDefaultBufferSizeScalesALSA); ++i) {
        if (strstr(g_maDefaultBufferSizeScalesALSA[i].name, deviceName) != NULL) {
            return g_maDefaultBufferSizeScalesALSA[i].scale;
        }
    }

    return 1;
}

ma_snd_pcm_format_t ma_convert_ma_format_to_alsa_format(ma_format format)
{
    ma_snd_pcm_format_t ALSAFormats[] = {
        MA_SND_PCM_FORMAT_UNKNOWN,     /* ma_format_unknown */
        MA_SND_PCM_FORMAT_U8,          /* ma_format_u8 */
        MA_SND_PCM_FORMAT_S16_LE,      /* ma_format_s16 */
        MA_SND_PCM_FORMAT_S24_3LE,     /* ma_format_s24 */
        MA_SND_PCM_FORMAT_S32_LE,      /* ma_format_s32 */
        MA_SND_PCM_FORMAT_FLOAT_LE     /* ma_format_f32 */
    };

    if (ma_is_big_endian()) {
        ALSAFormats[0] = MA_SND_PCM_FORMAT_UNKNOWN;
        ALSAFormats[1] = MA_SND_PCM_FORMAT_U8;
        ALSAFormats[2] = MA_SND_PCM_FORMAT_S16_BE;
        ALSAFormats[3] = MA_SND_PCM_FORMAT_S24_3BE;
        ALSAFormats[4] = MA_SND_PCM_FORMAT_S32_BE;
        ALSAFormats[5] = MA_SND_PCM_FORMAT_FLOAT_BE;
    }

    return ALSAFormats[format];
}

ma_format ma_format_from_alsa(ma_snd_pcm_format_t formatALSA)
{
    if (ma_is_little_endian()) {
        switch (formatALSA) {
            case MA_SND_PCM_FORMAT_S16_LE:   return ma_format_s16;
            case MA_SND_PCM_FORMAT_S24_3LE:  return ma_format_s24;
            case MA_SND_PCM_FORMAT_S32_LE:   return ma_format_s32;
            case MA_SND_PCM_FORMAT_FLOAT_LE: return ma_format_f32;
            default: break;
        }
    } else {
        switch (formatALSA) {
            case MA_SND_PCM_FORMAT_S16_BE:   return ma_format_s16;
            case MA_SND_PCM_FORMAT_S24_3BE:  return ma_format_s24;
            case MA_SND_PCM_FORMAT_S32_BE:   return ma_format_s32;
            case MA_SND_PCM_FORMAT_FLOAT_BE: return ma_format_f32;
            default: break;
        }
    }

    /* Endian agnostic. */
    switch (formatALSA) {
        case MA_SND_PCM_FORMAT_U8: return ma_format_u8;
        default: return ma_format_unknown;
    }
}

ma_channel ma_convert_alsa_channel_position_to_ma_channel(unsigned int alsaChannelPos)
{
    switch (alsaChannelPos)
    {
        case MA_SND_CHMAP_MONO: return MA_CHANNEL_MONO;
        case MA_SND_CHMAP_FL:   return MA_CHANNEL_FRONT_LEFT;
        case MA_SND_CHMAP_FR:   return MA_CHANNEL_FRONT_RIGHT;
        case MA_SND_CHMAP_RL:   return MA_CHANNEL_BACK_LEFT;
        case MA_SND_CHMAP_RR:   return MA_CHANNEL_BACK_RIGHT;
        case MA_SND_CHMAP_FC:   return MA_CHANNEL_FRONT_CENTER;
        case MA_SND_CHMAP_LFE:  return MA_CHANNEL_LFE;
        case MA_SND_CHMAP_SL:   return MA_CHANNEL_SIDE_LEFT;
        case MA_SND_CHMAP_SR:   return MA_CHANNEL_SIDE_RIGHT;
        case MA_SND_CHMAP_RC:   return MA_CHANNEL_BACK_CENTER;
        case MA_SND_CHMAP_FLC:  return MA_CHANNEL_FRONT_LEFT_CENTER;
        case MA_SND_CHMAP_FRC:  return MA_CHANNEL_FRONT_RIGHT_CENTER;
        case MA_SND_CHMAP_RLC:  return 0;
        case MA_SND_CHMAP_RRC:  return 0;
        case MA_SND_CHMAP_FLW:  return 0;
        case MA_SND_CHMAP_FRW:  return 0;
        case MA_SND_CHMAP_FLH:  return 0;
        case MA_SND_CHMAP_FCH:  return 0;
        case MA_SND_CHMAP_FRH:  return 0;
        case MA_SND_CHMAP_TC:   return MA_CHANNEL_TOP_CENTER;
        case MA_SND_CHMAP_TFL:  return MA_CHANNEL_TOP_FRONT_LEFT;
        case MA_SND_CHMAP_TFR:  return MA_CHANNEL_TOP_FRONT_RIGHT;
        case MA_SND_CHMAP_TFC:  return MA_CHANNEL_TOP_FRONT_CENTER;
        case MA_SND_CHMAP_TRL:  return MA_CHANNEL_TOP_BACK_LEFT;
        case MA_SND_CHMAP_TRR:  return MA_CHANNEL_TOP_BACK_RIGHT;
        case MA_SND_CHMAP_TRC:  return MA_CHANNEL_TOP_BACK_CENTER;
        default: break;
    }

    return 0;
}

ma_bool32 ma_is_common_device_name__alsa(const char* name)
{
    size_t iName;
    for (iName = 0; iName < ma_countof(g_maCommonDeviceNamesALSA); ++iName) {
        if (ma_strcmp(name, g_maCommonDeviceNamesALSA[iName]) == 0) {
            return MA_TRUE;
        }
    }

    return MA_FALSE;
}


ma_bool32 ma_is_playback_device_blacklisted__alsa(const char* name)
{
    size_t iName;
    for (iName = 0; iName < ma_countof(g_maBlacklistedPlaybackDeviceNamesALSA); ++iName) {
        if (ma_strcmp(name, g_maBlacklistedPlaybackDeviceNamesALSA[iName]) == 0) {
            return MA_TRUE;
        }
    }

    return MA_FALSE;
}

ma_bool32 ma_is_capture_device_blacklisted__alsa(const char* name)
{
    size_t iName;
    for (iName = 0; iName < ma_countof(g_maBlacklistedCaptureDeviceNamesALSA); ++iName) {
        if (ma_strcmp(name, g_maBlacklistedCaptureDeviceNamesALSA[iName]) == 0) {
            return MA_TRUE;
        }
    }

    return MA_FALSE;
}

ma_bool32 ma_is_device_blacklisted__alsa(ma_device_type deviceType, const char* name)
{
    if (deviceType == ma_device_type_playback) {
        return ma_is_playback_device_blacklisted__alsa(name);
    } else {
        return ma_is_capture_device_blacklisted__alsa(name);
    }
}


const char* ma_find_char(const char* str, char c, int* index)
{
    int i = 0;
    for (;;) {
        if (str[i] == '\0') {
            if (index) *index = -1;
            return NULL;
        }

        if (str[i] == c) {
            if (index) *index = i;
            return str + i;
        }

        i += 1;
    }

    /* Should never get here, but treat it as though the character was not found to make me feel better inside. */
    if (index) *index = -1;
    return NULL;
}

ma_bool32 ma_is_device_name_in_hw_format__alsa(const char* hwid)
{
    /* This function is just checking whether or not hwid is in "hw:%d,%d" format. */

    int commaPos;
    const char* dev;
    int i;

    if (hwid == NULL) {
        return MA_FALSE;
    }

    if (hwid[0] != 'h' || hwid[1] != 'w' || hwid[2] != ':') {
        return MA_FALSE;
    }

    hwid += 3;

    dev = ma_find_char(hwid, ',', &commaPos);
    if (dev == NULL) {
        return MA_FALSE;
    } else {
        dev += 1;   /* Skip past the ",". */
    }

    /* Check if the part between the ":" and the "," contains only numbers. If not, return false. */
    for (i = 0; i < commaPos; ++i) {
        if (hwid[i] < '0' || hwid[i] > '9') {
            return MA_FALSE;
        }
    }

    /* Check if everything after the "," is numeric. If not, return false. */
    i = 0;
    while (dev[i] != '\0') {
        if (dev[i] < '0' || dev[i] > '9') {
            return MA_FALSE;
        }
        i += 1;
    }

    return MA_TRUE;
}

int ma_convert_device_name_to_hw_format__alsa(ma_context* pContext, char* dst, size_t dstSize, const char* src)  /* Returns 0 on success, non-0 on error. */
{
    /* src should look something like this: "hw:CARD=I82801AAICH,DEV=0" */

    int colonPos;
    int commaPos;
    char card[256];
    const char* dev;
    int cardIndex;

    if (dst == NULL) {
        return -1;
    }
    if (dstSize < 7) {
        return -1;     /* Absolute minimum size of the output buffer is 7 bytes. */
    }

    *dst = '\0';    /* Safety. */
    if (src == NULL) {
        return -1;
    }

    /* If the input name is already in "hw:%d,%d" format, just return that verbatim. */
    if (ma_is_device_name_in_hw_format__alsa(src)) {
        return ma_strcpy_s(dst, dstSize, src);
    }

    src = ma_find_char(src, ':', &colonPos);
    if (src == NULL) {
        return -1;  /* Couldn't find a colon */
    }

    dev = ma_find_char(src, ',', &commaPos);
    if (dev == NULL) {
        dev = "0";
        ma_strncpy_s(card, sizeof(card), src+6, (size_t)-1);   /* +6 = ":CARD=" */
    } else {
        dev = dev + 5;  /* +5 = ",DEV=" */
        ma_strncpy_s(card, sizeof(card), src+6, commaPos-6);   /* +6 = ":CARD=" */
    }

    cardIndex = ((ma_snd_card_get_index_proc)pContext->alsa.snd_card_get_index)(card);
    if (cardIndex < 0) {
        return -2;  /* Failed to retrieve the card index. */
    }

    /*printf("TESTING: CARD=%s,DEV=%s\n", card, dev); */


    /* Construction. */
    dst[0] = 'h'; dst[1] = 'w'; dst[2] = ':';
    if (ma_itoa_s(cardIndex, dst+3, dstSize-3, 10) != 0) {
        return -3;
    }
    if (ma_strcat_s(dst, dstSize, ",") != 0) {
        return -3;
    }
    if (ma_strcat_s(dst, dstSize, dev) != 0) {
        return -3;
    }

    return 0;
}

ma_bool32 ma_does_id_exist_in_list__alsa(ma_device_id* pUniqueIDs, ma_uint32 count, const char* pHWID)
{
    ma_uint32 i;

    ma_assert(pHWID != NULL);

    for (i = 0; i < count; ++i) {
        if (ma_strcmp(pUniqueIDs[i].alsa, pHWID) == 0) {
            return MA_TRUE;
        }
    }

    return MA_FALSE;
}


ma_result ma_context_open_pcm__alsa(ma_context* pContext, ma_share_mode shareMode, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_snd_pcm_t** ppPCM)
{
    ma_snd_pcm_t* pPCM;
    ma_snd_pcm_stream_t stream;
    int openMode;

    ma_assert(pContext != NULL);
    ma_assert(ppPCM != NULL);

    *ppPCM = NULL;
    pPCM = NULL;

    stream   = (deviceType == ma_device_type_playback) ? MA_SND_PCM_STREAM_PLAYBACK : MA_SND_PCM_STREAM_CAPTURE;
    openMode = MA_SND_PCM_NO_AUTO_RESAMPLE | MA_SND_PCM_NO_AUTO_CHANNELS | MA_SND_PCM_NO_AUTO_FORMAT;

    if (pDeviceID == NULL) {
        ma_bool32 isDeviceOpen;
        size_t i;

        /*
        We're opening the default device. I don't know if trying anything other than "default" is necessary, but it makes
        me feel better to try as hard as we can get to get _something_ working.
        */
        const char* defaultDeviceNames[] = {
            "default",
            NULL,
            NULL,
            NULL,
            NULL,
            NULL,
            NULL
        };

        if (shareMode == ma_share_mode_exclusive) {
            defaultDeviceNames[1] = "hw";
            defaultDeviceNames[2] = "hw:0";
            defaultDeviceNames[3] = "hw:0,0";
        } else {
            if (deviceType == ma_device_type_playback) {
                defaultDeviceNames[1] = "dmix";
                defaultDeviceNames[2] = "dmix:0";
                defaultDeviceNames[3] = "dmix:0,0";
            } else {
                defaultDeviceNames[1] = "dsnoop";
                defaultDeviceNames[2] = "dsnoop:0";
                defaultDeviceNames[3] = "dsnoop:0,0";
            }
            defaultDeviceNames[4] = "hw";
            defaultDeviceNames[5] = "hw:0";
            defaultDeviceNames[6] = "hw:0,0";
        }

        isDeviceOpen = MA_FALSE;
        for (i = 0; i < ma_countof(defaultDeviceNames); ++i) {
            if (defaultDeviceNames[i] != NULL && defaultDeviceNames[i][0] != '\0') {
                if (((ma_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, defaultDeviceNames[i], stream, openMode) == 0) {
                    isDeviceOpen = MA_TRUE;
                    break;
                }
            }
        }

        if (!isDeviceOpen) {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_open() failed when trying to open an appropriate default device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    } else {
        /*
        We're trying to open a specific device. There's a few things to consider here:
        
        miniaudio recongnizes a special format of device id that excludes the "hw", "dmix", etc. prefix. It looks like this: ":0,0", ":0,1", etc. When
        an ID of this format is specified, it indicates to miniaudio that it can try different combinations of plugins ("hw", "dmix", etc.) until it
        finds an appropriate one that works. This comes in very handy when trying to open a device in shared mode ("dmix"), vs exclusive mode ("hw").
        */

        /* May end up needing to make small adjustments to the ID, so make a copy. */
        ma_device_id deviceID = *pDeviceID;
        ma_bool32 isDeviceOpen = MA_FALSE;

        if (deviceID.alsa[0] != ':') {
            /* The ID is not in ":0,0" format. Use the ID exactly as-is. */
            if (((ma_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, deviceID.alsa, stream, openMode) == 0) {
                isDeviceOpen = MA_TRUE;
            }
        } else {
            char hwid[256];

            /* The ID is in ":0,0" format. Try different plugins depending on the shared mode. */
            if (deviceID.alsa[1] == '\0') {
                deviceID.alsa[0] = '\0';  /* An ID of ":" should be converted to "". */
            }

            if (shareMode == ma_share_mode_shared) {
                if (deviceType == ma_device_type_playback) {
                    ma_strcpy_s(hwid, sizeof(hwid), "dmix");
                } else {
                    ma_strcpy_s(hwid, sizeof(hwid), "dsnoop");
                }

                if (ma_strcat_s(hwid, sizeof(hwid), deviceID.alsa) == 0) {
                    if (((ma_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, hwid, stream, openMode) == 0) {
                        isDeviceOpen = MA_TRUE;
                    }
                }
            }

            /* If at this point we still don't have an open device it means we're either preferencing exclusive mode or opening with "dmix"/"dsnoop" failed. */
            if (!isDeviceOpen) {
                ma_strcpy_s(hwid, sizeof(hwid), "hw");
                if (ma_strcat_s(hwid, sizeof(hwid), deviceID.alsa) == 0) {
                    if (((ma_snd_pcm_open_proc)pContext->alsa.snd_pcm_open)(&pPCM, hwid, stream, openMode) == 0) {
                        isDeviceOpen = MA_TRUE;
                    }
                }
            }
        }

        if (!isDeviceOpen) {
            return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_open() failed.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    }

    *ppPCM = pPCM;
    return MA_SUCCESS;
}


ma_bool32 ma_context_is_device_id_equal__alsa(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return ma_strcmp(pID0->alsa, pID1->alsa) == 0;
}

ma_result ma_context_enumerate_devices__alsa(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_bool32 cbResult = MA_TRUE;
    char** ppDeviceHints;
    ma_device_id* pUniqueIDs = NULL;
    ma_uint32 uniqueIDCount = 0;
    char** ppNextDeviceHint;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);

    ma_mutex_lock(&pContext->alsa.internalDeviceEnumLock);

    if (((ma_snd_device_name_hint_proc)pContext->alsa.snd_device_name_hint)(-1, "pcm", (void***)&ppDeviceHints) < 0) {
        ma_mutex_unlock(&pContext->alsa.internalDeviceEnumLock);
        return MA_NO_BACKEND;
    }

    ppNextDeviceHint = ppDeviceHints;
    while (*ppNextDeviceHint != NULL) {
        char* NAME = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "NAME");
        char* DESC = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "DESC");
        char* IOID = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "IOID");
        ma_device_type deviceType = ma_device_type_playback;
        ma_bool32 stopEnumeration = MA_FALSE;
        char hwid[sizeof(pUniqueIDs->alsa)];
        ma_device_info deviceInfo;

        if ((IOID == NULL || ma_strcmp(IOID, "Output") == 0)) {
            deviceType = ma_device_type_playback;
        }
        if ((IOID != NULL && ma_strcmp(IOID, "Input" ) == 0)) {
            deviceType = ma_device_type_capture;
        }

        if (NAME != NULL) {
            if (pContext->alsa.useVerboseDeviceEnumeration) {
                /* Verbose mode. Use the name exactly as-is. */
                ma_strncpy_s(hwid, sizeof(hwid), NAME, (size_t)-1);
            } else {
                /* Simplified mode. Use ":%d,%d" format. */
                if (ma_convert_device_name_to_hw_format__alsa(pContext, hwid, sizeof(hwid), NAME) == 0) {
                    /*
                    At this point, hwid looks like "hw:0,0". In simplified enumeration mode, we actually want to strip off the
                    plugin name so it looks like ":0,0". The reason for this is that this special format is detected at device
                    initialization time and is used as an indicator to try and use the most appropriate plugin depending on the
                    device type and sharing mode.
                    */
                    char* dst = hwid;
                    char* src = hwid+2;
                    while ((*dst++ = *src++));
                } else {
                    /* Conversion to "hw:%d,%d" failed. Just use the name as-is. */
                    ma_strncpy_s(hwid, sizeof(hwid), NAME, (size_t)-1);
                }

                if (ma_does_id_exist_in_list__alsa(pUniqueIDs, uniqueIDCount, hwid)) {
                    goto next_device;   /* The device has already been enumerated. Move on to the next one. */
                } else {
                    /* The device has not yet been enumerated. Make sure it's added to our list so that it's not enumerated again. */
                    ma_device_id* pNewUniqueIDs = (ma_device_id*)ma_realloc(pUniqueIDs, sizeof(*pUniqueIDs) * (uniqueIDCount + 1));
                    if (pNewUniqueIDs == NULL) {
                        goto next_device;   /* Failed to allocate memory. */
                    }

                    pUniqueIDs = pNewUniqueIDs;
                    ma_copy_memory(pUniqueIDs[uniqueIDCount].alsa, hwid, sizeof(hwid));
                    uniqueIDCount += 1;
                }
            }
        } else {
            ma_zero_memory(hwid, sizeof(hwid));
        }

        ma_zero_object(&deviceInfo);
        ma_strncpy_s(deviceInfo.id.alsa, sizeof(deviceInfo.id.alsa), hwid, (size_t)-1);

        /*
        DESC is the friendly name. We treat this slightly differently depending on whether or not we are using verbose
        device enumeration. In verbose mode we want to take the entire description so that the end-user can distinguish
        between the subdevices of each card/dev pair. In simplified mode, however, we only want the first part of the
        description.
        
        The value in DESC seems to be split into two lines, with the first line being the name of the device and the
        second line being a description of the device. I don't like having the description be across two lines because
        it makes formatting ugly and annoying. I'm therefore deciding to put it all on a single line with the second line
        being put into parentheses. In simplified mode I'm just stripping the second line entirely.
        */
        if (DESC != NULL) {
            int lfPos;
            const char* line2 = ma_find_char(DESC, '\n', &lfPos);
            if (line2 != NULL) {
                line2 += 1; /* Skip past the new-line character. */

                if (pContext->alsa.useVerboseDeviceEnumeration) {
                    /* Verbose mode. Put the second line in brackets. */
                    ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), DESC, lfPos);
                    ma_strcat_s (deviceInfo.name, sizeof(deviceInfo.name), " (");
                    ma_strcat_s (deviceInfo.name, sizeof(deviceInfo.name), line2);
                    ma_strcat_s (deviceInfo.name, sizeof(deviceInfo.name), ")");
                } else {
                    /* Simplified mode. Strip the second line entirely. */
                    ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), DESC, lfPos);
                }
            } else {
                /* There's no second line. Just copy the whole description. */
                ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), DESC, (size_t)-1);
            }
        }

        if (!ma_is_device_blacklisted__alsa(deviceType, NAME)) {
            cbResult = callback(pContext, deviceType, &deviceInfo, pUserData);
        }

        /*
        Some devices are both playback and capture, but they are only enumerated by ALSA once. We need to fire the callback
        again for the other device type in this case. We do this for known devices.
        */
        if (cbResult) {
            if (ma_is_common_device_name__alsa(NAME)) {
                if (deviceType == ma_device_type_playback) {
                    if (!ma_is_capture_device_blacklisted__alsa(NAME)) {
                        cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
                    }
                } else {
                    if (!ma_is_playback_device_blacklisted__alsa(NAME)) {
                        cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
                    }
                }
            }
        }

        if (cbResult == MA_FALSE) {
            stopEnumeration = MA_TRUE;
        }

    next_device:
        free(NAME);
        free(DESC);
        free(IOID);
        ppNextDeviceHint += 1;

        /* We need to stop enumeration if the callback returned false. */
        if (stopEnumeration) {
            break;
        }
    }

    ma_free(pUniqueIDs);
    ((ma_snd_device_name_free_hint_proc)pContext->alsa.snd_device_name_free_hint)((void**)ppDeviceHints);

    ma_mutex_unlock(&pContext->alsa.internalDeviceEnumLock);

    return MA_SUCCESS;
}


typedef struct
{
    ma_device_type deviceType;
    const ma_device_id* pDeviceID;
    ma_share_mode shareMode;
    ma_device_info* pDeviceInfo;
    ma_bool32 foundDevice;
} ma_context_get_device_info_enum_callback_data__alsa;

ma_bool32 ma_context_get_device_info_enum_callback__alsa(ma_context* pContext, ma_device_type deviceType, const ma_device_info* pDeviceInfo, void* pUserData)
{
    ma_context_get_device_info_enum_callback_data__alsa* pData = (ma_context_get_device_info_enum_callback_data__alsa*)pUserData;
    ma_assert(pData != NULL);

    if (pData->pDeviceID == NULL && ma_strcmp(pDeviceInfo->id.alsa, "default") == 0) {
        ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pDeviceInfo->name, (size_t)-1);
        pData->foundDevice = MA_TRUE;
    } else {
        if (pData->deviceType == deviceType && ma_context_is_device_id_equal__alsa(pContext, pData->pDeviceID, &pDeviceInfo->id)) {
            ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pDeviceInfo->name, (size_t)-1);
            pData->foundDevice = MA_TRUE;
        }
    }

    /* Keep enumerating until we have found the device. */
    return !pData->foundDevice;
}

ma_result ma_context_get_device_info__alsa(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_context_get_device_info_enum_callback_data__alsa data;
    ma_result result;
    ma_snd_pcm_t* pPCM;
    ma_snd_pcm_hw_params_t* pHWParams;
    ma_snd_pcm_format_mask_t* pFormatMask;
    int sampleRateDir = 0;

    ma_assert(pContext != NULL);

    /* We just enumerate to find basic information about the device. */
    data.deviceType = deviceType;
    data.pDeviceID = pDeviceID;
    data.shareMode = shareMode;
    data.pDeviceInfo = pDeviceInfo;
    data.foundDevice = MA_FALSE;
    result = ma_context_enumerate_devices__alsa(pContext, ma_context_get_device_info_enum_callback__alsa, &data);
    if (result != MA_SUCCESS) {
        return result;
    }

    if (!data.foundDevice) {
        return MA_NO_DEVICE;
    }

    /* For detailed info we need to open the device. */
    result = ma_context_open_pcm__alsa(pContext, shareMode, deviceType, pDeviceID, &pPCM);
    if (result != MA_SUCCESS) {
        return result;
    }

    /* We need to initialize a HW parameters object in order to know what formats are supported. */
    pHWParams = (ma_snd_pcm_hw_params_t*)calloc(1, ((ma_snd_pcm_hw_params_sizeof_proc)pContext->alsa.snd_pcm_hw_params_sizeof)());
    if (pHWParams == NULL) {
        return MA_OUT_OF_MEMORY;
    }

    if (((ma_snd_pcm_hw_params_any_proc)pContext->alsa.snd_pcm_hw_params_any)(pPCM, pHWParams) < 0) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to initialize hardware parameters. snd_pcm_hw_params_any() failed.", MA_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
    }

    ((ma_snd_pcm_hw_params_get_channels_min_proc)pContext->alsa.snd_pcm_hw_params_get_channels_min)(pHWParams, &pDeviceInfo->minChannels);
    ((ma_snd_pcm_hw_params_get_channels_max_proc)pContext->alsa.snd_pcm_hw_params_get_channels_max)(pHWParams, &pDeviceInfo->maxChannels);
    ((ma_snd_pcm_hw_params_get_rate_min_proc)pContext->alsa.snd_pcm_hw_params_get_rate_min)(pHWParams, &pDeviceInfo->minSampleRate, &sampleRateDir);
    ((ma_snd_pcm_hw_params_get_rate_max_proc)pContext->alsa.snd_pcm_hw_params_get_rate_max)(pHWParams, &pDeviceInfo->maxSampleRate, &sampleRateDir);

    /* Formats. */
    pFormatMask = (ma_snd_pcm_format_mask_t*)calloc(1, ((ma_snd_pcm_format_mask_sizeof_proc)pContext->alsa.snd_pcm_format_mask_sizeof)());
    if (pFormatMask == NULL) {
        return MA_OUT_OF_MEMORY;
    }

    ((ma_snd_pcm_hw_params_get_format_mask_proc)pContext->alsa.snd_pcm_hw_params_get_format_mask)(pHWParams, pFormatMask);

    pDeviceInfo->formatCount = 0;
    if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MA_SND_PCM_FORMAT_U8)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_u8;
    }
    if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MA_SND_PCM_FORMAT_S16_LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_s16;
    }
    if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MA_SND_PCM_FORMAT_S24_3LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_s24;
    }
    if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MA_SND_PCM_FORMAT_S32_LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_s32;
    }
    if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, MA_SND_PCM_FORMAT_FLOAT_LE)) {
        pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_f32;
    }

    ma_free(pFormatMask);
    ma_free(pHWParams);

    ((ma_snd_pcm_close_proc)pContext->alsa.snd_pcm_close)(pPCM);
    return MA_SUCCESS;
}


#if 0
/*
Waits for a number of frames to become available for either capture or playback. The return
value is the number of frames available.

This will return early if the main loop is broken with ma_device__break_main_loop().
*/
ma_uint32 ma_device__wait_for_frames__alsa(ma_device* pDevice, ma_bool32* pRequiresRestart)
{
    ma_assert(pDevice != NULL);

    if (pRequiresRestart) *pRequiresRestart = MA_FALSE;

    /* I want it so that this function returns the period size in frames. We just wait until that number of frames are available and then return. */
    ma_uint32 periodSizeInFrames = pDevice->bufferSizeInFrames / pDevice->periods;
    while (!pDevice->alsa.breakFromMainLoop) {
        ma_snd_pcm_sframes_t framesAvailable = ((ma_snd_pcm_avail_update_proc)pDevice->pContext->alsa.snd_pcm_avail_update)((ma_snd_pcm_t*)pDevice->alsa.pPCM);
        if (framesAvailable < 0) {
            if (framesAvailable == -EPIPE) {
                if (((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, framesAvailable, MA_TRUE) < 0) {
                    return 0;
                }

                /* A device recovery means a restart for mmap mode. */
                if (pRequiresRestart) {
                    *pRequiresRestart = MA_TRUE;
                }

                /* Try again, but if it fails this time just return an error. */
                framesAvailable = ((ma_snd_pcm_avail_update_proc)pDevice->pContext->alsa.snd_pcm_avail_update)((ma_snd_pcm_t*)pDevice->alsa.pPCM);
                if (framesAvailable < 0) {
                    return 0;
                }
            }
        }

        if (framesAvailable >= periodSizeInFrames) {
            return periodSizeInFrames;
        }

        if (framesAvailable < periodSizeInFrames) {
            /* Less than a whole period is available so keep waiting. */
            int waitResult = ((ma_snd_pcm_wait_proc)pDevice->pContext->alsa.snd_pcm_wait)((ma_snd_pcm_t*)pDevice->alsa.pPCM, -1);
            if (waitResult < 0) {
                if (waitResult == -EPIPE) {
                    if (((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, waitResult, MA_TRUE) < 0) {
                        return 0;
                    }

                    /* A device recovery means a restart for mmap mode. */
                    if (pRequiresRestart) {
                        *pRequiresRestart = MA_TRUE;
                    }
                }
            }
        }
    }

    /* We'll get here if the loop was terminated. Just return whatever's available. */
    ma_snd_pcm_sframes_t framesAvailable = ((ma_snd_pcm_avail_update_proc)pDevice->pContext->alsa.snd_pcm_avail_update)((ma_snd_pcm_t*)pDevice->alsa.pPCM);
    if (framesAvailable < 0) {
        return 0;
    }

    return framesAvailable;
}

ma_bool32 ma_device_read_from_client_and_write__alsa(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);
    if (!ma_device_is_started(pDevice) && ma_device__get_state(pDevice) != MA_STATE_STARTING) {
        return MA_FALSE;
    }
    if (pDevice->alsa.breakFromMainLoop) {
        return MA_FALSE;
    }

    if (pDevice->alsa.isUsingMMap) {
        /* mmap. */
        ma_bool32 requiresRestart;
        ma_uint32 framesAvailable = ma_device__wait_for_frames__alsa(pDevice, &requiresRestart);
        if (framesAvailable == 0) {
            return MA_FALSE;
        }

        /* Don't bother asking the client for more audio data if we're just stopping the device anyway. */
        if (pDevice->alsa.breakFromMainLoop) {
            return MA_FALSE;
        }

        const ma_snd_pcm_channel_area_t* pAreas;
        ma_snd_pcm_uframes_t mappedOffset;
        ma_snd_pcm_uframes_t mappedFrames = framesAvailable;
        while (framesAvailable > 0) {
            int result = ((ma_snd_pcm_mmap_begin_proc)pDevice->pContext->alsa.snd_pcm_mmap_begin)((ma_snd_pcm_t*)pDevice->alsa.pPCM, &pAreas, &mappedOffset, &mappedFrames);
            if (result < 0) {
                return MA_FALSE;
            }

            if (mappedFrames > 0) {
                void* pBuffer = (ma_uint8*)pAreas[0].addr + ((pAreas[0].first + (mappedOffset * pAreas[0].step)) / 8);
                ma_device__read_frames_from_client(pDevice, mappedFrames, pBuffer);
            }

            result = ((ma_snd_pcm_mmap_commit_proc)pDevice->pContext->alsa.snd_pcm_mmap_commit)((ma_snd_pcm_t*)pDevice->alsa.pPCM, mappedOffset, mappedFrames);
            if (result < 0 || (ma_snd_pcm_uframes_t)result != mappedFrames) {
                ((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, result, MA_TRUE);
                return MA_FALSE;
            }

            if (requiresRestart) {
                if (((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
                    return MA_FALSE;
                }
            }

            if (framesAvailable >= mappedFrames) {
                framesAvailable -= mappedFrames;
            } else {
                framesAvailable = 0;
            }
        }
    } else {
        /* readi/writei. */
        while (!pDevice->alsa.breakFromMainLoop) {
            ma_uint32 framesAvailable = ma_device__wait_for_frames__alsa(pDevice, NULL);
            if (framesAvailable == 0) {
                continue;
            }

            /* Don't bother asking the client for more audio data if we're just stopping the device anyway. */
            if (pDevice->alsa.breakFromMainLoop) {
                return MA_FALSE;
            }

            ma_device__read_frames_from_client(pDevice, framesAvailable, pDevice->alsa.pIntermediaryBuffer);

            ma_snd_pcm_sframes_t framesWritten = ((ma_snd_pcm_writei_proc)pDevice->pContext->alsa.snd_pcm_writei)((ma_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
            if (framesWritten < 0) {
                if (framesWritten == -EAGAIN) {
                    continue;   /* Just keep trying... */
                } else if (framesWritten == -EPIPE) {
                    /* Underrun. Just recover and try writing again. */
                    if (((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, framesWritten, MA_TRUE) < 0) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to recover device after underrun.", MA_FAILED_TO_START_BACKEND_DEVICE);
                        return MA_FALSE;
                    }

                    framesWritten = ((ma_snd_pcm_writei_proc)pDevice->pContext->alsa.snd_pcm_writei)((ma_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
                    if (framesWritten < 0) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to write data to the internal device.", MA_FAILED_TO_SEND_DATA_TO_DEVICE);
                        return MA_FALSE;
                    }

                    break;  /* Success. */
                } else {
                    ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_writei() failed when writing initial data.", MA_FAILED_TO_SEND_DATA_TO_DEVICE);
                    return MA_FALSE;
                }
            } else {
                break;  /* Success. */
            }
        }
    }

    return MA_TRUE;
}

ma_bool32 ma_device_read_and_send_to_client__alsa(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);
    if (!ma_device_is_started(pDevice)) {
        return MA_FALSE;
    }
    if (pDevice->alsa.breakFromMainLoop) {
        return MA_FALSE;
    }

    ma_uint32 framesToSend = 0;
    void* pBuffer = NULL;
    if (pDevice->alsa.pIntermediaryBuffer == NULL) {
        /* mmap. */
        ma_bool32 requiresRestart;
        ma_uint32 framesAvailable = ma_device__wait_for_frames__alsa(pDevice, &requiresRestart);
        if (framesAvailable == 0) {
            return MA_FALSE;
        }

        const ma_snd_pcm_channel_area_t* pAreas;
        ma_snd_pcm_uframes_t mappedOffset;
        ma_snd_pcm_uframes_t mappedFrames = framesAvailable;
        while (framesAvailable > 0) {
            int result = ((ma_snd_pcm_mmap_begin_proc)pDevice->pContext->alsa.snd_pcm_mmap_begin)((ma_snd_pcm_t*)pDevice->alsa.pPCM, &pAreas, &mappedOffset, &mappedFrames);
            if (result < 0) {
                return MA_FALSE;
            }

            if (mappedFrames > 0) {
                void* pBuffer = (ma_uint8*)pAreas[0].addr + ((pAreas[0].first + (mappedOffset * pAreas[0].step)) / 8);
                ma_device__send_frames_to_client(pDevice, mappedFrames, pBuffer);
            }

            result = ((ma_snd_pcm_mmap_commit_proc)pDevice->pContext->alsa.snd_pcm_mmap_commit)((ma_snd_pcm_t*)pDevice->alsa.pPCM, mappedOffset, mappedFrames);
            if (result < 0 || (ma_snd_pcm_uframes_t)result != mappedFrames) {
                ((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, result, MA_TRUE);
                return MA_FALSE;
            }

            if (requiresRestart) {
                if (((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
                    return MA_FALSE;
                }
            }

            if (framesAvailable >= mappedFrames) {
                framesAvailable -= mappedFrames;
            } else {
                framesAvailable = 0;
            }
        }
    } else {
        /* readi/writei. */
        ma_snd_pcm_sframes_t framesRead = 0;
        while (!pDevice->alsa.breakFromMainLoop) {
            ma_uint32 framesAvailable = ma_device__wait_for_frames__alsa(pDevice, NULL);
            if (framesAvailable == 0) {
                continue;
            }

            framesRead = ((ma_snd_pcm_readi_proc)pDevice->pContext->alsa.snd_pcm_readi)((ma_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
            if (framesRead < 0) {
                if (framesRead == -EAGAIN) {
                    continue;   /* Just keep trying... */
                } else if (framesRead == -EPIPE) {
                    /* Overrun. Just recover and try reading again. */
                    if (((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCM, framesRead, MA_TRUE) < 0) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to recover device after overrun.", MA_FAILED_TO_START_BACKEND_DEVICE);
                        return MA_FALSE;
                    }

                    framesRead = ((ma_snd_pcm_readi_proc)pDevice->pContext->alsa.snd_pcm_readi)((ma_snd_pcm_t*)pDevice->alsa.pPCM, pDevice->alsa.pIntermediaryBuffer, framesAvailable);
                    if (framesRead < 0) {
                        ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to read data from the internal device.", MA_FAILED_TO_READ_DATA_FROM_DEVICE);
                        return MA_FALSE;
                    }

                    break;  /* Success. */
                } else {
                    return MA_FALSE;
                }
            } else {
                break;  /* Success. */
            }
        }

        framesToSend = framesRead;
        pBuffer = pDevice->alsa.pIntermediaryBuffer;
    }

    if (framesToSend > 0) {
        ma_device__send_frames_to_client(pDevice, framesToSend, pBuffer);
    }

    return MA_TRUE;
}
#endif /* 0 */

void ma_device_uninit__alsa(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if ((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture) {
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture);
    }

    if ((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback) {
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback);
    }
}

ma_result ma_device_init_by_type__alsa(ma_context* pContext, const ma_device_config* pConfig, ma_device_type deviceType, ma_device* pDevice)
{
    ma_result result;
    ma_snd_pcm_t* pPCM;
    ma_bool32 isUsingMMap;
    ma_snd_pcm_format_t formatALSA;
    ma_share_mode shareMode;
    ma_device_id* pDeviceID;
    ma_format internalFormat;
    ma_uint32 internalChannels;
    ma_uint32 internalSampleRate;
    ma_channel internalChannelMap[MA_MAX_CHANNELS];
    ma_uint32 internalBufferSizeInFrames;
    ma_uint32 internalPeriods;
    ma_snd_pcm_hw_params_t* pHWParams;
    ma_snd_pcm_sw_params_t* pSWParams;
    ma_snd_pcm_uframes_t bufferBoundary;
    float bufferSizeScaleFactor;

    ma_assert(pContext != NULL);
    ma_assert(pConfig != NULL);
    ma_assert(deviceType != ma_device_type_duplex); /* This function should only be called for playback _or_ capture, never duplex. */
    ma_assert(pDevice != NULL);

    formatALSA = ma_convert_ma_format_to_alsa_format((deviceType == ma_device_type_capture) ? pConfig->capture.format : pConfig->playback.format);
    shareMode  = (deviceType == ma_device_type_capture) ? pConfig->capture.shareMode : pConfig->playback.shareMode;
    pDeviceID  = (deviceType == ma_device_type_capture) ? pConfig->capture.pDeviceID : pConfig->playback.pDeviceID;

    result = ma_context_open_pcm__alsa(pContext, shareMode, deviceType, pDeviceID, &pPCM);
    if (result != MA_SUCCESS) {
        return result;
    }

    /* If using the default buffer size we may want to apply some device-specific scaling for known devices that have peculiar latency characteristics */
    bufferSizeScaleFactor = 1;
    if (pDevice->usingDefaultBufferSize) {
        ma_snd_pcm_info_t* pInfo = (ma_snd_pcm_info_t*)calloc(1, ((ma_snd_pcm_info_sizeof_proc)pContext->alsa.snd_pcm_info_sizeof)());
        if (pInfo == NULL) {
            return MA_OUT_OF_MEMORY;
        }

        /* We may need to scale the size of the buffer depending on the device. */
        if (((ma_snd_pcm_info_proc)pContext->alsa.snd_pcm_info)(pPCM, pInfo) == 0) {
            const char* deviceName = ((ma_snd_pcm_info_get_name_proc)pContext->alsa.snd_pcm_info_get_name)(pInfo);
            if (deviceName != NULL) {
                if (ma_strcmp(deviceName, "default") == 0) {
                    char** ppDeviceHints;
                    char** ppNextDeviceHint;

                    /* It's the default device. We need to use DESC from snd_device_name_hint(). */
                    if (((ma_snd_device_name_hint_proc)pContext->alsa.snd_device_name_hint)(-1, "pcm", (void***)&ppDeviceHints) < 0) {
                        ma_free(pInfo);
                        return MA_NO_BACKEND;
                    }

                    ppNextDeviceHint = ppDeviceHints;
                    while (*ppNextDeviceHint != NULL) {
                        char* NAME = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "NAME");
                        char* DESC = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "DESC");
                        char* IOID = ((ma_snd_device_name_get_hint_proc)pContext->alsa.snd_device_name_get_hint)(*ppNextDeviceHint, "IOID");

                        ma_bool32 foundDevice = MA_FALSE;
                        if ((deviceType == ma_device_type_playback && (IOID == NULL || ma_strcmp(IOID, "Output") == 0)) ||
                            (deviceType == ma_device_type_capture  && (IOID != NULL && ma_strcmp(IOID, "Input" ) == 0))) {
                            if (ma_strcmp(NAME, deviceName) == 0) {
                                bufferSizeScaleFactor = ma_find_default_buffer_size_scale__alsa(DESC);
                                foundDevice = MA_TRUE;
                            }
                        }

                        free(NAME);
                        free(DESC);
                        free(IOID);
                        ppNextDeviceHint += 1;

                        if (foundDevice) {
                            break;
                        }
                    }

                    ((ma_snd_device_name_free_hint_proc)pContext->alsa.snd_device_name_free_hint)((void**)ppDeviceHints);
                } else {
                    bufferSizeScaleFactor = ma_find_default_buffer_size_scale__alsa(deviceName);
                }
            }
        }

        ma_free(pInfo);
    }


    /* Hardware parameters. */
    pHWParams = (ma_snd_pcm_hw_params_t*)calloc(1, ((ma_snd_pcm_hw_params_sizeof_proc)pContext->alsa.snd_pcm_hw_params_sizeof)());
    if (pHWParams == NULL) {
        return MA_OUT_OF_MEMORY;
    }

    if (((ma_snd_pcm_hw_params_any_proc)pContext->alsa.snd_pcm_hw_params_any)(pPCM, pHWParams) < 0) {
        ma_free(pHWParams);
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to initialize hardware parameters. snd_pcm_hw_params_any() failed.", MA_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
    }

    /* MMAP Mode. Try using interleaved MMAP access. If this fails, fall back to standard readi/writei. */
    isUsingMMap = MA_FALSE;
#if 0   /* NOTE: MMAP mode temporarily disabled. */
    if (deviceType != ma_device_type_capture) {    /* <-- Disabling MMAP mode for capture devices because I apparently do not have a device that supports it which means I can't test it... Contributions welcome. */
        if (!pConfig->alsa.noMMap && ma_device__is_async(pDevice)) {
            if (((ma_snd_pcm_hw_params_set_access_proc)pContext->alsa.snd_pcm_hw_params_set_access)(pPCM, pHWParams, MA_SND_PCM_ACCESS_MMAP_INTERLEAVED) == 0) {
                pDevice->alsa.isUsingMMap = MA_TRUE;
            }
        }
    }
#endif

    if (!isUsingMMap) {
        if (((ma_snd_pcm_hw_params_set_access_proc)pContext->alsa.snd_pcm_hw_params_set_access)(pPCM, pHWParams, MA_SND_PCM_ACCESS_RW_INTERLEAVED) < 0) {
            ma_free(pHWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set access mode to neither SND_PCM_ACCESS_MMAP_INTERLEAVED nor SND_PCM_ACCESS_RW_INTERLEAVED. snd_pcm_hw_params_set_access() failed.", MA_FORMAT_NOT_SUPPORTED);
        }
    }

    /*
    Most important properties first. The documentation for OSS (yes, I know this is ALSA!) recommends format, channels, then sample rate. I can't
    find any documentation for ALSA specifically, so I'm going to copy the recommendation for OSS.
    */

    /* Format. */
    {
        ma_snd_pcm_format_mask_t* pFormatMask;

        /* Try getting every supported format first. */
        pFormatMask = (ma_snd_pcm_format_mask_t*)calloc(1, ((ma_snd_pcm_format_mask_sizeof_proc)pContext->alsa.snd_pcm_format_mask_sizeof)());
        if (pFormatMask == NULL) {
            ma_free(pHWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return MA_OUT_OF_MEMORY;
        }

        ((ma_snd_pcm_hw_params_get_format_mask_proc)pContext->alsa.snd_pcm_hw_params_get_format_mask)(pHWParams, pFormatMask);

        /*
        At this point we should have a list of supported formats, so now we need to find the best one. We first check if the requested format is
        supported, and if so, use that one. If it's not supported, we just run though a list of formats and try to find the best one.
        */
        if (!((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, formatALSA)) {
            size_t i;

            /* The requested format is not supported so now try running through the list of formats and return the best one. */
            ma_snd_pcm_format_t preferredFormatsALSA[] = {
                MA_SND_PCM_FORMAT_S16_LE,      /* ma_format_s16 */
                MA_SND_PCM_FORMAT_FLOAT_LE,    /* ma_format_f32 */
                MA_SND_PCM_FORMAT_S32_LE,      /* ma_format_s32 */
                MA_SND_PCM_FORMAT_S24_3LE,     /* ma_format_s24 */
                MA_SND_PCM_FORMAT_U8           /* ma_format_u8 */
            };

            if (ma_is_big_endian()) {
                preferredFormatsALSA[0] = MA_SND_PCM_FORMAT_S16_BE;
                preferredFormatsALSA[1] = MA_SND_PCM_FORMAT_FLOAT_BE;
                preferredFormatsALSA[2] = MA_SND_PCM_FORMAT_S32_BE;
                preferredFormatsALSA[3] = MA_SND_PCM_FORMAT_S24_3BE;
                preferredFormatsALSA[4] = MA_SND_PCM_FORMAT_U8;
            }

            formatALSA = MA_SND_PCM_FORMAT_UNKNOWN;
            for (i = 0; i < (sizeof(preferredFormatsALSA) / sizeof(preferredFormatsALSA[0])); ++i) {
                if (((ma_snd_pcm_format_mask_test_proc)pContext->alsa.snd_pcm_format_mask_test)(pFormatMask, preferredFormatsALSA[i])) {
                    formatALSA = preferredFormatsALSA[i];
                    break;
                }
            }

            if (formatALSA == MA_SND_PCM_FORMAT_UNKNOWN) {
                ma_free(pHWParams);
                ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Format not supported. The device does not support any miniaudio formats.", MA_FORMAT_NOT_SUPPORTED);
            }
        }

        ma_free(pFormatMask);
        pFormatMask = NULL;

        if (((ma_snd_pcm_hw_params_set_format_proc)pContext->alsa.snd_pcm_hw_params_set_format)(pPCM, pHWParams, formatALSA) < 0) {
            ma_free(pHWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Format not supported. snd_pcm_hw_params_set_format() failed.", MA_FORMAT_NOT_SUPPORTED);
        }
        
        internalFormat = ma_format_from_alsa(formatALSA);
        if (internalFormat == ma_format_unknown) {
            ma_free(pHWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] The chosen format is not supported by miniaudio.", MA_FORMAT_NOT_SUPPORTED);
        }
    }

    /* Channels. */
    {
        unsigned int channels = (deviceType == ma_device_type_capture) ? pConfig->capture.channels : pConfig->playback.channels;
        if (((ma_snd_pcm_hw_params_set_channels_near_proc)pContext->alsa.snd_pcm_hw_params_set_channels_near)(pPCM, pHWParams, &channels) < 0) {
            ma_free(pHWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set channel count. snd_pcm_hw_params_set_channels_near() failed.", MA_FORMAT_NOT_SUPPORTED);
        }
        internalChannels = (ma_uint32)channels;
    }

    /* Sample Rate */
    {
        unsigned int sampleRate;

        /*
        It appears there's either a bug in ALSA, a bug in some drivers, or I'm doing something silly; but having resampling enabled causes
        problems with some device configurations when used in conjunction with MMAP access mode. To fix this problem we need to disable
        resampling.
        
        To reproduce this problem, open the "plug:dmix" device, and set the sample rate to 44100. Internally, it looks like dmix uses a
        sample rate of 48000. The hardware parameters will get set correctly with no errors, but it looks like the 44100 -> 48000 resampling
        doesn't work properly - but only with MMAP access mode. You will notice skipping/crackling in the audio, and it'll run at a slightly
        faster rate.
        
        miniaudio has built-in support for sample rate conversion (albeit low quality at the moment), so disabling resampling should be fine
        for us. The only problem is that it won't be taking advantage of any kind of hardware-accelerated resampling and it won't be very
        good quality until I get a chance to improve the quality of miniaudio's software sample rate conversion.
        
        I don't currently know if the dmix plugin is the only one with this error. Indeed, this is the only one I've been able to reproduce
        this error with. In the future, we may want to restrict the disabling of resampling to only known bad plugins.
        */
        ((ma_snd_pcm_hw_params_set_rate_resample_proc)pContext->alsa.snd_pcm_hw_params_set_rate_resample)(pPCM, pHWParams, 0);

        sampleRate = pConfig->sampleRate;
        if (((ma_snd_pcm_hw_params_set_rate_near_proc)pContext->alsa.snd_pcm_hw_params_set_rate_near)(pPCM, pHWParams, &sampleRate, 0) < 0) {
            ma_free(pHWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Sample rate not supported. snd_pcm_hw_params_set_rate_near() failed.", MA_FORMAT_NOT_SUPPORTED);
        }
        internalSampleRate = (ma_uint32)sampleRate;
    }

    /* Buffer Size */
    {
        ma_snd_pcm_uframes_t actualBufferSizeInFrames = pConfig->bufferSizeInFrames;
        if (actualBufferSizeInFrames == 0) {
            actualBufferSizeInFrames = ma_scale_buffer_size(ma_calculate_buffer_size_in_frames_from_milliseconds(pConfig->bufferSizeInMilliseconds, internalSampleRate), bufferSizeScaleFactor);
        }

        if (((ma_snd_pcm_hw_params_set_buffer_size_near_proc)pContext->alsa.snd_pcm_hw_params_set_buffer_size_near)(pPCM, pHWParams, &actualBufferSizeInFrames) < 0) {
            ma_free(pHWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set buffer size for device. snd_pcm_hw_params_set_buffer_size() failed.", MA_FORMAT_NOT_SUPPORTED);
        }
        internalBufferSizeInFrames = actualBufferSizeInFrames;
    }

    /* Periods. */
    {
        ma_uint32 periods = pConfig->periods;
        if (((ma_snd_pcm_hw_params_set_periods_near_proc)pContext->alsa.snd_pcm_hw_params_set_periods_near)(pPCM, pHWParams, &periods, NULL) < 0) {
            ma_free(pHWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set period count. snd_pcm_hw_params_set_periods_near() failed.", MA_FORMAT_NOT_SUPPORTED);
        }
        internalPeriods = periods;
    }

    /* Apply hardware parameters. */
    if (((ma_snd_pcm_hw_params_proc)pContext->alsa.snd_pcm_hw_params)(pPCM, pHWParams) < 0) {
        ma_free(pHWParams);
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set hardware parameters. snd_pcm_hw_params() failed.", MA_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
    }

    ma_free(pHWParams);
    pHWParams = NULL;


    /* Software parameters. */
    pSWParams = (ma_snd_pcm_sw_params_t*)calloc(1, ((ma_snd_pcm_sw_params_sizeof_proc)pContext->alsa.snd_pcm_sw_params_sizeof)());
    if (pSWParams == NULL) {
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return MA_OUT_OF_MEMORY;
    }

    if (((ma_snd_pcm_sw_params_current_proc)pContext->alsa.snd_pcm_sw_params_current)(pPCM, pSWParams) != 0) {
        ma_free(pSWParams);
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to initialize software parameters. snd_pcm_sw_params_current() failed.", MA_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
    }

    if (deviceType == ma_device_type_capture) {
        if (((ma_snd_pcm_sw_params_set_avail_min_proc)pContext->alsa.snd_pcm_sw_params_set_avail_min)(pPCM, pSWParams, 1) != 0) {
            ma_free(pSWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_sw_params_set_avail_min() failed.", MA_FORMAT_NOT_SUPPORTED);
        }
    } else {
        if (((ma_snd_pcm_sw_params_set_avail_min_proc)pContext->alsa.snd_pcm_sw_params_set_avail_min)(pPCM, pSWParams, 1) != 0) {
            ma_free(pSWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] snd_pcm_sw_params_set_avail_min() failed.", MA_FORMAT_NOT_SUPPORTED);
        }
    }
    

    if (((ma_snd_pcm_sw_params_get_boundary_proc)pContext->alsa.snd_pcm_sw_params_get_boundary)(pSWParams, &bufferBoundary) < 0) {
        bufferBoundary = internalBufferSizeInFrames;
    }

    /*printf("TRACE: bufferBoundary=%ld\n", bufferBoundary);*/

    if (deviceType == ma_device_type_playback && !isUsingMMap) {   /* Only playback devices in writei/readi mode need a start threshold. */
        /*
        Subtle detail here with the start threshold. When in playback-only mode (no full-duplex) we can set the start threshold to
        the size of a period. But for full-duplex we need to set it such that it is at least two periods.
        */
        if (((ma_snd_pcm_sw_params_set_start_threshold_proc)pContext->alsa.snd_pcm_sw_params_set_start_threshold)(pPCM, pSWParams, internalBufferSizeInFrames) != 0) {
            ma_free(pSWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set start threshold for playback device. snd_pcm_sw_params_set_start_threshold() failed.", MA_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
        }
        if (((ma_snd_pcm_sw_params_set_stop_threshold_proc)pContext->alsa.snd_pcm_sw_params_set_stop_threshold)(pPCM, pSWParams, bufferBoundary) != 0) { /* Set to boundary to loop instead of stop in the event of an xrun. */
            ma_free(pSWParams);
            ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set stop threshold for playback device. snd_pcm_sw_params_set_stop_threshold() failed.", MA_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
        }
    }

    if (((ma_snd_pcm_sw_params_proc)pContext->alsa.snd_pcm_sw_params)(pPCM, pSWParams) != 0) {
        ma_free(pSWParams);
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to set software parameters. snd_pcm_sw_params() failed.", MA_FAILED_TO_CONFIGURE_BACKEND_DEVICE);
    }

    ma_free(pSWParams);
    pSWParams = NULL;


    /* Grab the internal channel map. For now we're not going to bother trying to change the channel map and instead just do it ourselves. */
    {
        ma_snd_pcm_chmap_t* pChmap = ((ma_snd_pcm_get_chmap_proc)pContext->alsa.snd_pcm_get_chmap)(pPCM);
        if (pChmap != NULL) {
            ma_uint32 iChannel;

            /* There are cases where the returned channel map can have a different channel count than was returned by snd_pcm_hw_params_set_channels_near(). */
            if (pChmap->channels >= internalChannels) {
                /* Drop excess channels. */
                for (iChannel = 0; iChannel < internalChannels; ++iChannel) {
                    internalChannelMap[iChannel] = ma_convert_alsa_channel_position_to_ma_channel(pChmap->pos[iChannel]);
                }
            } else {
                ma_uint32 i;

                /*
                Excess channels use defaults. Do an initial fill with defaults, overwrite the first pChmap->channels, validate to ensure there are no duplicate
                channels. If validation fails, fall back to defaults.
                */
                ma_bool32 isValid = MA_TRUE;

                /* Fill with defaults. */
                ma_get_standard_channel_map(ma_standard_channel_map_alsa, internalChannels, internalChannelMap);

                /* Overwrite first pChmap->channels channels. */
                for (iChannel = 0; iChannel < pChmap->channels; ++iChannel) {
                    internalChannelMap[iChannel] = ma_convert_alsa_channel_position_to_ma_channel(pChmap->pos[iChannel]);
                }

                /* Validate. */
                for (i = 0; i < internalChannels && isValid; ++i) {
                    ma_uint32 j;
                    for (j = i+1; j < internalChannels; ++j) {
                        if (internalChannelMap[i] == internalChannelMap[j]) {
                            isValid = MA_FALSE;
                            break;
                        }
                    }
                }

                /* If our channel map is invalid, fall back to defaults. */
                if (!isValid) {
                    ma_get_standard_channel_map(ma_standard_channel_map_alsa, internalChannels, internalChannelMap);
                }
            }

            free(pChmap);
            pChmap = NULL;
        } else {
            /* Could not retrieve the channel map. Fall back to a hard-coded assumption. */
            ma_get_standard_channel_map(ma_standard_channel_map_alsa, internalChannels, internalChannelMap);
        }
    }


    /* We're done. Prepare the device. */
    if (((ma_snd_pcm_prepare_proc)pDevice->pContext->alsa.snd_pcm_prepare)(pPCM) < 0) {
        ((ma_snd_pcm_close_proc)pDevice->pContext->alsa.snd_pcm_close)(pPCM);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to prepare device.", MA_FAILED_TO_START_BACKEND_DEVICE);
    }


    if (deviceType == ma_device_type_capture) {
        pDevice->alsa.pPCMCapture                    = (ma_ptr)pPCM;
        pDevice->alsa.isUsingMMapCapture             = isUsingMMap;
        pDevice->capture.internalFormat              = internalFormat;
        pDevice->capture.internalChannels            = internalChannels;
        pDevice->capture.internalSampleRate          = internalSampleRate;
        ma_channel_map_copy(pDevice->capture.internalChannelMap, internalChannelMap, internalChannels);
        pDevice->capture.internalBufferSizeInFrames  = internalBufferSizeInFrames;
        pDevice->capture.internalPeriods             = internalPeriods;
    } else {
        pDevice->alsa.pPCMPlayback                   = (ma_ptr)pPCM;
        pDevice->alsa.isUsingMMapPlayback            = isUsingMMap;
        pDevice->playback.internalFormat             = internalFormat;
        pDevice->playback.internalChannels           = internalChannels;
        pDevice->playback.internalSampleRate         = internalSampleRate;
        ma_channel_map_copy(pDevice->playback.internalChannelMap, internalChannelMap, internalChannels);
        pDevice->playback.internalBufferSizeInFrames = internalBufferSizeInFrames;
        pDevice->playback.internalPeriods            = internalPeriods;
    }

    return MA_SUCCESS;
}

ma_result ma_device_init__alsa(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    ma_zero_object(&pDevice->alsa);

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_by_type__alsa(pContext, pConfig, ma_device_type_capture, pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_by_type__alsa(pContext, pConfig, ma_device_type_playback, pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    return MA_SUCCESS;
}

#if 0
ma_result ma_device_start__alsa(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    /* Prepare the device first... */
    if (((ma_snd_pcm_prepare_proc)pDevice->pContext->alsa.snd_pcm_prepare)((ma_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to prepare device.", MA_FAILED_TO_START_BACKEND_DEVICE);
    }

    /*
    ... and then grab an initial chunk from the client. After this is done, the device should
    automatically start playing, since that's how we configured the software parameters.
    */
    if (pDevice->type == ma_device_type_playback) {
        if (!ma_device_read_from_client_and_write__alsa(pDevice)) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to write initial chunk of data to the playback device.", MA_FAILED_TO_SEND_DATA_TO_DEVICE);
        }

        /* mmap mode requires an explicit start. */
        if (pDevice->alsa.isUsingMMap) {
            if (((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to start capture device.", MA_FAILED_TO_START_BACKEND_DEVICE);
            }
        }
    } else {
        if (((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCM) < 0) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to start capture device.", MA_FAILED_TO_START_BACKEND_DEVICE);
        }
    }

    return MA_SUCCESS;
}
#endif /* 0 */

ma_result ma_device_stop__alsa(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_snd_pcm_drain_proc)pDevice->pContext->alsa.snd_pcm_drain)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        /* Using drain instead of drop because ma_device_stop() is defined such that pending frames are processed before returning. */
        ((ma_snd_pcm_drain_proc)pDevice->pContext->alsa.snd_pcm_drain)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback);
    }

    
    return MA_SUCCESS;
}

ma_result ma_device_write__alsa(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount)
{
    ma_snd_pcm_sframes_t resultALSA;
    ma_uint32 totalPCMFramesProcessed;

    ma_assert(pDevice != NULL);
    ma_assert(pPCMFrames != NULL);

    /*printf("TRACE: Enter write()\n");*/

    totalPCMFramesProcessed = 0;
    while (totalPCMFramesProcessed < frameCount) {
        const void* pSrc = ma_offset_ptr(pPCMFrames, totalPCMFramesProcessed * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));
        ma_uint32 framesRemaining = (frameCount - totalPCMFramesProcessed);

        /*printf("TRACE: Writing %d frames (frameCount=%d)\n", framesRemaining, frameCount);*/

        resultALSA = ((ma_snd_pcm_writei_proc)pDevice->pContext->alsa.snd_pcm_writei)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback, pSrc, framesRemaining);
        if (resultALSA < 0) {
            if (resultALSA == -EAGAIN) {
                /*printf("TRACE: EGAIN (write)\n");*/
                continue;   /* Try again. */
            } else if (resultALSA == -EPIPE) {
                /*printf("TRACE: EPIPE (write)\n");*/

                /* Underrun. Recover and try again. If this fails we need to return an error. */
                if (((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback, resultALSA, MA_TRUE) < 0) { /* MA_TRUE=silent (don't print anything on error). */
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to recover device after underrun.", MA_FAILED_TO_START_BACKEND_DEVICE);
                }

                /*
                In my testing I have had a situation where writei() does not automatically restart the device even though I've set it
                up as such in the software parameters. What will happen is writei() will block indefinitely even though the number of
                frames is well beyond the auto-start threshold. To work around this I've needed to add an explicit start here. Not sure
                if this is me just being stupid and not recovering the device properly, but this definitely feels like something isn't
                quite right here.
                */
                if (((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback) < 0) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to start device after underrun.", MA_FAILED_TO_START_BACKEND_DEVICE);
                }

                resultALSA = ((ma_snd_pcm_writei_proc)pDevice->pContext->alsa.snd_pcm_writei)((ma_snd_pcm_t*)pDevice->alsa.pPCMPlayback, pSrc, framesRemaining);
                if (resultALSA < 0) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to write data to device after underrun.", MA_FAILED_TO_START_BACKEND_DEVICE);
                }
            }
        }

        totalPCMFramesProcessed += resultALSA;
    }

    return MA_SUCCESS;
}

ma_result ma_device_read__alsa(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount)
{
    ma_snd_pcm_sframes_t resultALSA;
    ma_uint32 totalPCMFramesProcessed;

    ma_assert(pDevice != NULL);
    ma_assert(pPCMFrames != NULL);

    /* We need to explicitly start the device if it isn't already. */
    if (((ma_snd_pcm_state_proc)pDevice->pContext->alsa.snd_pcm_state)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture) != MA_SND_PCM_STATE_RUNNING) {
        if (((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture) < 0) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to start device in preparation for reading.", MA_FAILED_TO_START_BACKEND_DEVICE);
        }
    }

    totalPCMFramesProcessed = 0;
    while (totalPCMFramesProcessed < frameCount) {
        void* pDst = ma_offset_ptr(pPCMFrames, totalPCMFramesProcessed * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));
        ma_uint32 framesRemaining = (frameCount - totalPCMFramesProcessed);

        /*printf("TRACE: snd_pcm_readi(framesRemaining=%d)\n", framesRemaining);*/

        resultALSA = ((ma_snd_pcm_readi_proc)pDevice->pContext->alsa.snd_pcm_readi)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture, pDst, framesRemaining);
        if (resultALSA < 0) {
            if (resultALSA == -EAGAIN) {
                /*printf("TRACE: EGAIN (read)\n");*/
                continue;
            } else if (resultALSA == -EPIPE) {
                /*printf("TRACE: EPIPE (read)\n");*/

                /* Overrun. Recover and try again. If this fails we need to return an error. */
                if (((ma_snd_pcm_recover_proc)pDevice->pContext->alsa.snd_pcm_recover)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture, resultALSA, MA_TRUE) < 0) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to recover device after overrun.", MA_FAILED_TO_START_BACKEND_DEVICE);
                }

                if (((ma_snd_pcm_start_proc)pDevice->pContext->alsa.snd_pcm_start)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture) < 0) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to start device after underrun.", MA_FAILED_TO_START_BACKEND_DEVICE);
                }

                resultALSA = ((ma_snd_pcm_readi_proc)pDevice->pContext->alsa.snd_pcm_readi)((ma_snd_pcm_t*)pDevice->alsa.pPCMCapture, pDst, framesRemaining);
                if (resultALSA < 0) {
                    return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[ALSA] Failed to read data from the internal device.", MA_FAILED_TO_READ_DATA_FROM_DEVICE);
                }
            }
        }

        totalPCMFramesProcessed += resultALSA;
    }

    return MA_SUCCESS;
}

#if 0
ma_result ma_device_break_main_loop__alsa(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    pDevice->alsa.breakFromMainLoop = MA_TRUE;
    return MA_SUCCESS;
}

ma_result ma_device_main_loop__alsa(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    pDevice->alsa.breakFromMainLoop = MA_FALSE;
    if (pDevice->type == ma_device_type_playback) {
        /* Playback. Read from client, write to device. */
        while (!pDevice->alsa.breakFromMainLoop && ma_device_read_from_client_and_write__alsa(pDevice)) {
        }
    } else {
        /* Capture. Read from device, write to client. */
        while (!pDevice->alsa.breakFromMainLoop && ma_device_read_and_send_to_client__alsa(pDevice)) {
        }
    }

    return MA_SUCCESS;
}
#endif /* 0 */

ma_result ma_context_uninit__alsa(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_alsa);

    /* Clean up memory for memory leak checkers. */
    ((ma_snd_config_update_free_global_proc)pContext->alsa.snd_config_update_free_global)();

#ifndef MA_NO_RUNTIME_LINKING
    ma_dlclose(pContext->alsa.asoundSO);
#endif

    ma_mutex_uninit(&pContext->alsa.internalDeviceEnumLock);

    return MA_SUCCESS;
}

ma_result ma_context_init__alsa(const ma_context_config* pConfig, ma_context* pContext)
{
#ifndef MA_NO_RUNTIME_LINKING
    const char* libasoundNames[] = {
        "libasound.so.2",
        "libasound.so"
    };
    size_t i;

    for (i = 0; i < ma_countof(libasoundNames); ++i) {
        pContext->alsa.asoundSO = ma_dlopen(libasoundNames[i]);
        if (pContext->alsa.asoundSO != NULL) {
            break;
        }
    }

    if (pContext->alsa.asoundSO == NULL) {
#ifdef MA_DEBUG_OUTPUT
        printf("[ALSA] Failed to open shared object.\n");
#endif
        return MA_NO_BACKEND;
    }

    pContext->alsa.snd_pcm_open                           = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_open");
    pContext->alsa.snd_pcm_close                          = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_close");
    pContext->alsa.snd_pcm_hw_params_sizeof               = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_sizeof");
    pContext->alsa.snd_pcm_hw_params_any                  = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_any");
    pContext->alsa.snd_pcm_hw_params_set_format           = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_format");
    pContext->alsa.snd_pcm_hw_params_set_format_first     = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_format_first");
    pContext->alsa.snd_pcm_hw_params_get_format_mask      = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_format_mask");
    pContext->alsa.snd_pcm_hw_params_set_channels_near    = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_channels_near");
    pContext->alsa.snd_pcm_hw_params_set_rate_resample    = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_rate_resample");
    pContext->alsa.snd_pcm_hw_params_set_rate_near        = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_rate_near");
    pContext->alsa.snd_pcm_hw_params_set_buffer_size_near = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_buffer_size_near");
    pContext->alsa.snd_pcm_hw_params_set_periods_near     = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_periods_near");
    pContext->alsa.snd_pcm_hw_params_set_access           = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_set_access");
    pContext->alsa.snd_pcm_hw_params_get_format           = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_format");
    pContext->alsa.snd_pcm_hw_params_get_channels         = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_channels");
    pContext->alsa.snd_pcm_hw_params_get_channels_min     = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_channels_min");
    pContext->alsa.snd_pcm_hw_params_get_channels_max     = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_channels_max");
    pContext->alsa.snd_pcm_hw_params_get_rate             = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_rate");
    pContext->alsa.snd_pcm_hw_params_get_rate_min         = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_rate_min");
    pContext->alsa.snd_pcm_hw_params_get_rate_max         = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_rate_max");
    pContext->alsa.snd_pcm_hw_params_get_buffer_size      = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_buffer_size");
    pContext->alsa.snd_pcm_hw_params_get_periods          = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_periods");
    pContext->alsa.snd_pcm_hw_params_get_access           = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params_get_access");
    pContext->alsa.snd_pcm_hw_params                      = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_hw_params");
    pContext->alsa.snd_pcm_sw_params_sizeof               = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params_sizeof");
    pContext->alsa.snd_pcm_sw_params_current              = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params_current");
    pContext->alsa.snd_pcm_sw_params_get_boundary         = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params_get_boundary");
    pContext->alsa.snd_pcm_sw_params_set_avail_min        = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params_set_avail_min");
    pContext->alsa.snd_pcm_sw_params_set_start_threshold  = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params_set_start_threshold");
    pContext->alsa.snd_pcm_sw_params_set_stop_threshold   = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params_set_stop_threshold");
    pContext->alsa.snd_pcm_sw_params                      = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_sw_params");
    pContext->alsa.snd_pcm_format_mask_sizeof             = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_format_mask_sizeof");
    pContext->alsa.snd_pcm_format_mask_test               = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_format_mask_test");
    pContext->alsa.snd_pcm_get_chmap                      = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_get_chmap");
    pContext->alsa.snd_pcm_state                          = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_state");
    pContext->alsa.snd_pcm_prepare                        = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_prepare");
    pContext->alsa.snd_pcm_start                          = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_start");
    pContext->alsa.snd_pcm_drop                           = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_drop");
    pContext->alsa.snd_pcm_drain                          = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_drain");
    pContext->alsa.snd_device_name_hint                   = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_device_name_hint");
    pContext->alsa.snd_device_name_get_hint               = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_device_name_get_hint");
    pContext->alsa.snd_card_get_index                     = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_card_get_index");
    pContext->alsa.snd_device_name_free_hint              = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_device_name_free_hint");
    pContext->alsa.snd_pcm_mmap_begin                     = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_mmap_begin");
    pContext->alsa.snd_pcm_mmap_commit                    = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_mmap_commit");
    pContext->alsa.snd_pcm_recover                        = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_recover");
    pContext->alsa.snd_pcm_readi                          = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_readi");
    pContext->alsa.snd_pcm_writei                         = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_writei");
    pContext->alsa.snd_pcm_avail                          = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_avail");
    pContext->alsa.snd_pcm_avail_update                   = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_avail_update");
    pContext->alsa.snd_pcm_wait                           = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_wait");
    pContext->alsa.snd_pcm_info                           = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_info");
    pContext->alsa.snd_pcm_info_sizeof                    = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_info_sizeof");
    pContext->alsa.snd_pcm_info_get_name                  = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_pcm_info_get_name");
    pContext->alsa.snd_config_update_free_global          = (ma_proc)ma_dlsym(pContext->alsa.asoundSO, "snd_config_update_free_global");
#else
    /* The system below is just for type safety. */
    ma_snd_pcm_open_proc                           _snd_pcm_open                           = snd_pcm_open;
    ma_snd_pcm_close_proc                          _snd_pcm_close                          = snd_pcm_close;
    ma_snd_pcm_hw_params_sizeof_proc               _snd_pcm_hw_params_sizeof               = snd_pcm_hw_params_sizeof;
    ma_snd_pcm_hw_params_any_proc                  _snd_pcm_hw_params_any                  = snd_pcm_hw_params_any;
    ma_snd_pcm_hw_params_set_format_proc           _snd_pcm_hw_params_set_format           = snd_pcm_hw_params_set_format;
    ma_snd_pcm_hw_params_set_format_first_proc     _snd_pcm_hw_params_set_format_first     = snd_pcm_hw_params_set_format_first;
    ma_snd_pcm_hw_params_get_format_mask_proc      _snd_pcm_hw_params_get_format_mask      = snd_pcm_hw_params_get_format_mask;
    ma_snd_pcm_hw_params_set_channels_near_proc    _snd_pcm_hw_params_set_channels_near    = snd_pcm_hw_params_set_channels_near;
    ma_snd_pcm_hw_params_set_rate_resample_proc    _snd_pcm_hw_params_set_rate_resample    = snd_pcm_hw_params_set_rate_resample;
    ma_snd_pcm_hw_params_set_rate_near_proc        _snd_pcm_hw_params_set_rate_near        = snd_pcm_hw_params_set_rate_near;
    ma_snd_pcm_hw_params_set_buffer_size_near_proc _snd_pcm_hw_params_set_buffer_size_near = snd_pcm_hw_params_set_buffer_size_near;
    ma_snd_pcm_hw_params_set_periods_near_proc     _snd_pcm_hw_params_set_periods_near     = snd_pcm_hw_params_set_periods_near;
    ma_snd_pcm_hw_params_set_access_proc           _snd_pcm_hw_params_set_access           = snd_pcm_hw_params_set_access;
    ma_snd_pcm_hw_params_get_format_proc           _snd_pcm_hw_params_get_format           = snd_pcm_hw_params_get_format;
    ma_snd_pcm_hw_params_get_channels_proc         _snd_pcm_hw_params_get_channels         = snd_pcm_hw_params_get_channels;
    ma_snd_pcm_hw_params_get_channels_min_proc     _snd_pcm_hw_params_get_channels_min     = snd_pcm_hw_params_get_channels_min;
    ma_snd_pcm_hw_params_get_channels_max_proc     _snd_pcm_hw_params_get_channels_max     = snd_pcm_hw_params_get_channels_max;
    ma_snd_pcm_hw_params_get_rate_proc             _snd_pcm_hw_params_get_rate             = snd_pcm_hw_params_get_rate;
    ma_snd_pcm_hw_params_get_rate_min_proc         _snd_pcm_hw_params_get_rate_min         = snd_pcm_hw_params_get_rate_min;
    ma_snd_pcm_hw_params_get_rate_max_proc         _snd_pcm_hw_params_get_rate_max         = snd_pcm_hw_params_get_rate_max;
    ma_snd_pcm_hw_params_get_buffer_size_proc      _snd_pcm_hw_params_get_buffer_size      = snd_pcm_hw_params_get_buffer_size;
    ma_snd_pcm_hw_params_get_periods_proc          _snd_pcm_hw_params_get_periods          = snd_pcm_hw_params_get_periods;
    ma_snd_pcm_hw_params_get_access_proc           _snd_pcm_hw_params_get_access           = snd_pcm_hw_params_get_access;
    ma_snd_pcm_hw_params_proc                      _snd_pcm_hw_params                      = snd_pcm_hw_params;
    ma_snd_pcm_sw_params_sizeof_proc               _snd_pcm_sw_params_sizeof               = snd_pcm_sw_params_sizeof;
    ma_snd_pcm_sw_params_current_proc              _snd_pcm_sw_params_current              = snd_pcm_sw_params_current;
    ma_snd_pcm_sw_params_get_boundary_proc         _snd_pcm_sw_params_get_boundary         = snd_pcm_sw_params_get_boundary;
    ma_snd_pcm_sw_params_set_avail_min_proc        _snd_pcm_sw_params_set_avail_min        = snd_pcm_sw_params_set_avail_min;
    ma_snd_pcm_sw_params_set_start_threshold_proc  _snd_pcm_sw_params_set_start_threshold  = snd_pcm_sw_params_set_start_threshold;
    ma_snd_pcm_sw_params_set_stop_threshold_proc   _snd_pcm_sw_params_set_stop_threshold   = snd_pcm_sw_params_set_stop_threshold;
    ma_snd_pcm_sw_params_proc                      _snd_pcm_sw_params                      = snd_pcm_sw_params;
    ma_snd_pcm_format_mask_sizeof_proc             _snd_pcm_format_mask_sizeof             = snd_pcm_format_mask_sizeof;
    ma_snd_pcm_format_mask_test_proc               _snd_pcm_format_mask_test               = snd_pcm_format_mask_test;
    ma_snd_pcm_get_chmap_proc                      _snd_pcm_get_chmap                      = snd_pcm_get_chmap;
    ma_snd_pcm_state_proc                          _snd_pcm_state                          = snd_pcm_state;
    ma_snd_pcm_prepare_proc                        _snd_pcm_prepare                        = snd_pcm_prepare;
    ma_snd_pcm_start_proc                          _snd_pcm_start                          = snd_pcm_start;
    ma_snd_pcm_drop_proc                           _snd_pcm_drop                           = snd_pcm_drop;
    ma_snd_pcm_drain_proc                          _snd_pcm_drain                          = snd_pcm_drain;
    ma_snd_device_name_hint_proc                   _snd_device_name_hint                   = snd_device_name_hint;
    ma_snd_device_name_get_hint_proc               _snd_device_name_get_hint               = snd_device_name_get_hint;
    ma_snd_card_get_index_proc                     _snd_card_get_index                     = snd_card_get_index;
    ma_snd_device_name_free_hint_proc              _snd_device_name_free_hint              = snd_device_name_free_hint;
    ma_snd_pcm_mmap_begin_proc                     _snd_pcm_mmap_begin                     = snd_pcm_mmap_begin;
    ma_snd_pcm_mmap_commit_proc                    _snd_pcm_mmap_commit                    = snd_pcm_mmap_commit;
    ma_snd_pcm_recover_proc                        _snd_pcm_recover                        = snd_pcm_recover;
    ma_snd_pcm_readi_proc                          _snd_pcm_readi                          = snd_pcm_readi;
    ma_snd_pcm_writei_proc                         _snd_pcm_writei                         = snd_pcm_writei;
    ma_snd_pcm_avail_proc                          _snd_pcm_avail                          = snd_pcm_avail;
    ma_snd_pcm_avail_update_proc                   _snd_pcm_avail_update                   = snd_pcm_avail_update;
    ma_snd_pcm_wait_proc                           _snd_pcm_wait                           = snd_pcm_wait;
    ma_snd_pcm_info_proc                           _snd_pcm_info                           = snd_pcm_info;
    ma_snd_pcm_info_sizeof_proc                    _snd_pcm_info_sizeof                    = snd_pcm_info_sizeof;
    ma_snd_pcm_info_get_name_proc                  _snd_pcm_info_get_name                  = snd_pcm_info_get_name;
    ma_snd_config_update_free_global_proc          _snd_config_update_free_global          = snd_config_update_free_global;

    pContext->alsa.snd_pcm_open                           = (ma_proc)_snd_pcm_open;
    pContext->alsa.snd_pcm_close                          = (ma_proc)_snd_pcm_close;
    pContext->alsa.snd_pcm_hw_params_sizeof               = (ma_proc)_snd_pcm_hw_params_sizeof;
    pContext->alsa.snd_pcm_hw_params_any                  = (ma_proc)_snd_pcm_hw_params_any;
    pContext->alsa.snd_pcm_hw_params_set_format           = (ma_proc)_snd_pcm_hw_params_set_format;
    pContext->alsa.snd_pcm_hw_params_set_format_first     = (ma_proc)_snd_pcm_hw_params_set_format_first;
    pContext->alsa.snd_pcm_hw_params_get_format_mask      = (ma_proc)_snd_pcm_hw_params_get_format_mask;
    pContext->alsa.snd_pcm_hw_params_set_channels_near    = (ma_proc)_snd_pcm_hw_params_set_channels_near;
    pContext->alsa.snd_pcm_hw_params_set_rate_resample    = (ma_proc)_snd_pcm_hw_params_set_rate_resample;
    pContext->alsa.snd_pcm_hw_params_set_rate_near        = (ma_proc)_snd_pcm_hw_params_set_rate_near;
    pContext->alsa.snd_pcm_hw_params_set_buffer_size_near = (ma_proc)_snd_pcm_hw_params_set_buffer_size_near;
    pContext->alsa.snd_pcm_hw_params_set_periods_near     = (ma_proc)_snd_pcm_hw_params_set_periods_near;
    pContext->alsa.snd_pcm_hw_params_set_access           = (ma_proc)_snd_pcm_hw_params_set_access;
    pContext->alsa.snd_pcm_hw_params_get_format           = (ma_proc)_snd_pcm_hw_params_get_format;
    pContext->alsa.snd_pcm_hw_params_get_channels         = (ma_proc)_snd_pcm_hw_params_get_channels;
    pContext->alsa.snd_pcm_hw_params_get_channels_min     = (ma_proc)_snd_pcm_hw_params_get_channels_min;
    pContext->alsa.snd_pcm_hw_params_get_channels_max     = (ma_proc)_snd_pcm_hw_params_get_channels_max;
    pContext->alsa.snd_pcm_hw_params_get_rate             = (ma_proc)_snd_pcm_hw_params_get_rate;
    pContext->alsa.snd_pcm_hw_params_get_buffer_size      = (ma_proc)_snd_pcm_hw_params_get_buffer_size;
    pContext->alsa.snd_pcm_hw_params_get_periods          = (ma_proc)_snd_pcm_hw_params_get_periods;
    pContext->alsa.snd_pcm_hw_params_get_access           = (ma_proc)_snd_pcm_hw_params_get_access;
    pContext->alsa.snd_pcm_hw_params                      = (ma_proc)_snd_pcm_hw_params;
    pContext->alsa.snd_pcm_sw_params_sizeof               = (ma_proc)_snd_pcm_sw_params_sizeof;
    pContext->alsa.snd_pcm_sw_params_current              = (ma_proc)_snd_pcm_sw_params_current;
    pContext->alsa.snd_pcm_sw_params_get_boundary         = (ma_proc)_snd_pcm_sw_params_get_boundary;
    pContext->alsa.snd_pcm_sw_params_set_avail_min        = (ma_proc)_snd_pcm_sw_params_set_avail_min;
    pContext->alsa.snd_pcm_sw_params_set_start_threshold  = (ma_proc)_snd_pcm_sw_params_set_start_threshold;
    pContext->alsa.snd_pcm_sw_params_set_stop_threshold   = (ma_proc)_snd_pcm_sw_params_set_stop_threshold;
    pContext->alsa.snd_pcm_sw_params                      = (ma_proc)_snd_pcm_sw_params;
    pContext->alsa.snd_pcm_format_mask_sizeof             = (ma_proc)_snd_pcm_format_mask_sizeof;
    pContext->alsa.snd_pcm_format_mask_test               = (ma_proc)_snd_pcm_format_mask_test;
    pContext->alsa.snd_pcm_get_chmap                      = (ma_proc)_snd_pcm_get_chmap;
    pContext->alsa.snd_pcm_state                          = (ma_proc)_snd_pcm_state;
    pContext->alsa.snd_pcm_prepare                        = (ma_proc)_snd_pcm_prepare;
    pContext->alsa.snd_pcm_start                          = (ma_proc)_snd_pcm_start;
    pContext->alsa.snd_pcm_drop                           = (ma_proc)_snd_pcm_drop;
    pContext->alsa.snd_pcm_drain                          = (ma_proc)_snd_pcm_drain;
    pContext->alsa.snd_device_name_hint                   = (ma_proc)_snd_device_name_hint;
    pContext->alsa.snd_device_name_get_hint               = (ma_proc)_snd_device_name_get_hint;
    pContext->alsa.snd_card_get_index                     = (ma_proc)_snd_card_get_index;
    pContext->alsa.snd_device_name_free_hint              = (ma_proc)_snd_device_name_free_hint;
    pContext->alsa.snd_pcm_mmap_begin                     = (ma_proc)_snd_pcm_mmap_begin;
    pContext->alsa.snd_pcm_mmap_commit                    = (ma_proc)_snd_pcm_mmap_commit;
    pContext->alsa.snd_pcm_recover                        = (ma_proc)_snd_pcm_recover;
    pContext->alsa.snd_pcm_readi                          = (ma_proc)_snd_pcm_readi;
    pContext->alsa.snd_pcm_writei                         = (ma_proc)_snd_pcm_writei;
    pContext->alsa.snd_pcm_avail                          = (ma_proc)_snd_pcm_avail;
    pContext->alsa.snd_pcm_avail_update                   = (ma_proc)_snd_pcm_avail_update;
    pContext->alsa.snd_pcm_wait                           = (ma_proc)_snd_pcm_wait;
    pContext->alsa.snd_pcm_info                           = (ma_proc)_snd_pcm_info;
    pContext->alsa.snd_pcm_info_sizeof                    = (ma_proc)_snd_pcm_info_sizeof;
    pContext->alsa.snd_pcm_info_get_name                  = (ma_proc)_snd_pcm_info_get_name;
    pContext->alsa.snd_config_update_free_global          = (ma_proc)_snd_config_update_free_global;
#endif

    pContext->alsa.useVerboseDeviceEnumeration = pConfig->alsa.useVerboseDeviceEnumeration;

    if (ma_mutex_init(pContext, &pContext->alsa.internalDeviceEnumLock) != MA_SUCCESS) {
        ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[ALSA] WARNING: Failed to initialize mutex for internal device enumeration.", MA_ERROR);
    }

    pContext->onUninit              = ma_context_uninit__alsa;
    pContext->onDeviceIDEqual       = ma_context_is_device_id_equal__alsa;
    pContext->onEnumDevices         = ma_context_enumerate_devices__alsa;
    pContext->onGetDeviceInfo       = ma_context_get_device_info__alsa;
    pContext->onDeviceInit          = ma_device_init__alsa;
    pContext->onDeviceUninit        = ma_device_uninit__alsa;
    pContext->onDeviceStart         = NULL; /*ma_device_start__alsa;*/
    pContext->onDeviceStop          = ma_device_stop__alsa;
    pContext->onDeviceWrite         = ma_device_write__alsa;
    pContext->onDeviceRead          = ma_device_read__alsa;

    return MA_SUCCESS;
}
#endif  /* ALSA */



/******************************************************************************

PulseAudio Backend

******************************************************************************/
#ifdef MA_HAS_PULSEAUDIO
/*
It is assumed pulseaudio.h is available when compile-time linking is being used. We use this for type safety when using
compile time linking (we don't have this luxury when using runtime linking without headers).

When using compile time linking, each of our ma_* equivalents should use the sames types as defined by the header. The
reason for this is that it allow us to take advantage of proper type safety.
*/
#ifdef MA_NO_RUNTIME_LINKING
#include <pulse/pulseaudio.h>

#define MA_PA_OK                                       PA_OK
#define MA_PA_ERR_ACCESS                               PA_ERR_ACCESS
#define MA_PA_ERR_INVALID                              PA_ERR_INVALID
#define MA_PA_ERR_NOENTITY                             PA_ERR_NOENTITY

#define MA_PA_CHANNELS_MAX                             PA_CHANNELS_MAX
#define MA_PA_RATE_MAX                                 PA_RATE_MAX

typedef pa_context_flags_t ma_pa_context_flags_t;
#define MA_PA_CONTEXT_NOFLAGS                          PA_CONTEXT_NOFLAGS
#define MA_PA_CONTEXT_NOAUTOSPAWN                      PA_CONTEXT_NOAUTOSPAWN
#define MA_PA_CONTEXT_NOFAIL                           PA_CONTEXT_NOFAIL

typedef pa_stream_flags_t ma_pa_stream_flags_t;
#define MA_PA_STREAM_NOFLAGS                           PA_STREAM_NOFLAGS
#define MA_PA_STREAM_START_CORKED                      PA_STREAM_START_CORKED
#define MA_PA_STREAM_INTERPOLATE_TIMING                PA_STREAM_INTERPOLATE_TIMING
#define MA_PA_STREAM_NOT_MONOTONIC                     PA_STREAM_NOT_MONOTONIC
#define MA_PA_STREAM_AUTO_TIMING_UPDATE                PA_STREAM_AUTO_TIMING_UPDATE
#define MA_PA_STREAM_NO_REMAP_CHANNELS                 PA_STREAM_NO_REMAP_CHANNELS
#define MA_PA_STREAM_NO_REMIX_CHANNELS                 PA_STREAM_NO_REMIX_CHANNELS
#define MA_PA_STREAM_FIX_FORMAT                        PA_STREAM_FIX_FORMAT
#define MA_PA_STREAM_FIX_RATE                          PA_STREAM_FIX_RATE
#define MA_PA_STREAM_FIX_CHANNELS                      PA_STREAM_FIX_CHANNELS
#define MA_PA_STREAM_DONT_MOVE                         PA_STREAM_DONT_MOVE
#define MA_PA_STREAM_VARIABLE_RATE                     PA_STREAM_VARIABLE_RATE
#define MA_PA_STREAM_PEAK_DETECT                       PA_STREAM_PEAK_DETECT
#define MA_PA_STREAM_START_MUTED                       PA_STREAM_START_MUTED
#define MA_PA_STREAM_ADJUST_LATENCY                    PA_STREAM_ADJUST_LATENCY
#define MA_PA_STREAM_EARLY_REQUESTS                    PA_STREAM_EARLY_REQUESTS
#define MA_PA_STREAM_DONT_INHIBIT_AUTO_SUSPEND         PA_STREAM_DONT_INHIBIT_AUTO_SUSPEND
#define MA_PA_STREAM_START_UNMUTED                     PA_STREAM_START_UNMUTED
#define MA_PA_STREAM_FAIL_ON_SUSPEND                   PA_STREAM_FAIL_ON_SUSPEND
#define MA_PA_STREAM_RELATIVE_VOLUME                   PA_STREAM_RELATIVE_VOLUME
#define MA_PA_STREAM_PASSTHROUGH                       PA_STREAM_PASSTHROUGH

typedef pa_sink_flags_t ma_pa_sink_flags_t;
#define MA_PA_SINK_NOFLAGS                             PA_SINK_NOFLAGS
#define MA_PA_SINK_HW_VOLUME_CTRL                      PA_SINK_HW_VOLUME_CTRL
#define MA_PA_SINK_LATENCY                             PA_SINK_LATENCY
#define MA_PA_SINK_HARDWARE                            PA_SINK_HARDWARE
#define MA_PA_SINK_NETWORK                             PA_SINK_NETWORK
#define MA_PA_SINK_HW_MUTE_CTRL                        PA_SINK_HW_MUTE_CTRL
#define MA_PA_SINK_DECIBEL_VOLUME                      PA_SINK_DECIBEL_VOLUME
#define MA_PA_SINK_FLAT_VOLUME                         PA_SINK_FLAT_VOLUME
#define MA_PA_SINK_DYNAMIC_LATENCY                     PA_SINK_DYNAMIC_LATENCY
#define MA_PA_SINK_SET_FORMATS                         PA_SINK_SET_FORMATS

typedef pa_source_flags_t ma_pa_source_flags_t;
#define MA_PA_SOURCE_NOFLAGS                           PA_SOURCE_NOFLAGS
#define MA_PA_SOURCE_HW_VOLUME_CTRL                    PA_SOURCE_HW_VOLUME_CTRL
#define MA_PA_SOURCE_LATENCY                           PA_SOURCE_LATENCY
#define MA_PA_SOURCE_HARDWARE                          PA_SOURCE_HARDWARE
#define MA_PA_SOURCE_NETWORK                           PA_SOURCE_NETWORK
#define MA_PA_SOURCE_HW_MUTE_CTRL                      PA_SOURCE_HW_MUTE_CTRL
#define MA_PA_SOURCE_DECIBEL_VOLUME                    PA_SOURCE_DECIBEL_VOLUME
#define MA_PA_SOURCE_DYNAMIC_LATENCY                   PA_SOURCE_DYNAMIC_LATENCY
#define MA_PA_SOURCE_FLAT_VOLUME                       PA_SOURCE_FLAT_VOLUME

typedef pa_context_state_t ma_pa_context_state_t;
#define MA_PA_CONTEXT_UNCONNECTED                      PA_CONTEXT_UNCONNECTED
#define MA_PA_CONTEXT_CONNECTING                       PA_CONTEXT_CONNECTING
#define MA_PA_CONTEXT_AUTHORIZING                      PA_CONTEXT_AUTHORIZING
#define MA_PA_CONTEXT_SETTING_NAME                     PA_CONTEXT_SETTING_NAME
#define MA_PA_CONTEXT_READY                            PA_CONTEXT_READY
#define MA_PA_CONTEXT_FAILED                           PA_CONTEXT_FAILED
#define MA_PA_CONTEXT_TERMINATED                       PA_CONTEXT_TERMINATED

typedef pa_stream_state_t ma_pa_stream_state_t;
#define MA_PA_STREAM_UNCONNECTED                       PA_STREAM_UNCONNECTED
#define MA_PA_STREAM_CREATING                          PA_STREAM_CREATING
#define MA_PA_STREAM_READY                             PA_STREAM_READY
#define MA_PA_STREAM_FAILED                            PA_STREAM_FAILED
#define MA_PA_STREAM_TERMINATED                        PA_STREAM_TERMINATED

typedef pa_operation_state_t ma_pa_operation_state_t;
#define MA_PA_OPERATION_RUNNING                        PA_OPERATION_RUNNING
#define MA_PA_OPERATION_DONE                           PA_OPERATION_DONE
#define MA_PA_OPERATION_CANCELLED                      PA_OPERATION_CANCELLED

typedef pa_sink_state_t ma_pa_sink_state_t;
#define MA_PA_SINK_INVALID_STATE                       PA_SINK_INVALID_STATE
#define MA_PA_SINK_RUNNING                             PA_SINK_RUNNING
#define MA_PA_SINK_IDLE                                PA_SINK_IDLE
#define MA_PA_SINK_SUSPENDED                           PA_SINK_SUSPENDED

typedef pa_source_state_t ma_pa_source_state_t;
#define MA_PA_SOURCE_INVALID_STATE                     PA_SOURCE_INVALID_STATE
#define MA_PA_SOURCE_RUNNING                           PA_SOURCE_RUNNING
#define MA_PA_SOURCE_IDLE                              PA_SOURCE_IDLE
#define MA_PA_SOURCE_SUSPENDED                         PA_SOURCE_SUSPENDED

typedef pa_seek_mode_t ma_pa_seek_mode_t;
#define MA_PA_SEEK_RELATIVE                            PA_SEEK_RELATIVE
#define MA_PA_SEEK_ABSOLUTE                            PA_SEEK_ABSOLUTE
#define MA_PA_SEEK_RELATIVE_ON_READ                    PA_SEEK_RELATIVE_ON_READ
#define MA_PA_SEEK_RELATIVE_END                        PA_SEEK_RELATIVE_END

typedef pa_channel_position_t ma_pa_channel_position_t;
#define MA_PA_CHANNEL_POSITION_INVALID                 PA_CHANNEL_POSITION_INVALID
#define MA_PA_CHANNEL_POSITION_MONO                    PA_CHANNEL_POSITION_MONO
#define MA_PA_CHANNEL_POSITION_FRONT_LEFT              PA_CHANNEL_POSITION_FRONT_LEFT
#define MA_PA_CHANNEL_POSITION_FRONT_RIGHT             PA_CHANNEL_POSITION_FRONT_RIGHT
#define MA_PA_CHANNEL_POSITION_FRONT_CENTER            PA_CHANNEL_POSITION_FRONT_CENTER
#define MA_PA_CHANNEL_POSITION_REAR_CENTER             PA_CHANNEL_POSITION_REAR_CENTER
#define MA_PA_CHANNEL_POSITION_REAR_LEFT               PA_CHANNEL_POSITION_REAR_LEFT
#define MA_PA_CHANNEL_POSITION_REAR_RIGHT              PA_CHANNEL_POSITION_REAR_RIGHT
#define MA_PA_CHANNEL_POSITION_LFE                     PA_CHANNEL_POSITION_LFE
#define MA_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER    PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER
#define MA_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER   PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER
#define MA_PA_CHANNEL_POSITION_SIDE_LEFT               PA_CHANNEL_POSITION_SIDE_LEFT
#define MA_PA_CHANNEL_POSITION_SIDE_RIGHT              PA_CHANNEL_POSITION_SIDE_RIGHT
#define MA_PA_CHANNEL_POSITION_AUX0                    PA_CHANNEL_POSITION_AUX0
#define MA_PA_CHANNEL_POSITION_AUX1                    PA_CHANNEL_POSITION_AUX1
#define MA_PA_CHANNEL_POSITION_AUX2                    PA_CHANNEL_POSITION_AUX2
#define MA_PA_CHANNEL_POSITION_AUX3                    PA_CHANNEL_POSITION_AUX3
#define MA_PA_CHANNEL_POSITION_AUX4                    PA_CHANNEL_POSITION_AUX4
#define MA_PA_CHANNEL_POSITION_AUX5                    PA_CHANNEL_POSITION_AUX5
#define MA_PA_CHANNEL_POSITION_AUX6                    PA_CHANNEL_POSITION_AUX6
#define MA_PA_CHANNEL_POSITION_AUX7                    PA_CHANNEL_POSITION_AUX7
#define MA_PA_CHANNEL_POSITION_AUX8                    PA_CHANNEL_POSITION_AUX8
#define MA_PA_CHANNEL_POSITION_AUX9                    PA_CHANNEL_POSITION_AUX9
#define MA_PA_CHANNEL_POSITION_AUX10                   PA_CHANNEL_POSITION_AUX10
#define MA_PA_CHANNEL_POSITION_AUX11                   PA_CHANNEL_POSITION_AUX11
#define MA_PA_CHANNEL_POSITION_AUX12                   PA_CHANNEL_POSITION_AUX12
#define MA_PA_CHANNEL_POSITION_AUX13                   PA_CHANNEL_POSITION_AUX13
#define MA_PA_CHANNEL_POSITION_AUX14                   PA_CHANNEL_POSITION_AUX14
#define MA_PA_CHANNEL_POSITION_AUX15                   PA_CHANNEL_POSITION_AUX15
#define MA_PA_CHANNEL_POSITION_AUX16                   PA_CHANNEL_POSITION_AUX16
#define MA_PA_CHANNEL_POSITION_AUX17                   PA_CHANNEL_POSITION_AUX17
#define MA_PA_CHANNEL_POSITION_AUX18                   PA_CHANNEL_POSITION_AUX18
#define MA_PA_CHANNEL_POSITION_AUX19                   PA_CHANNEL_POSITION_AUX19
#define MA_PA_CHANNEL_POSITION_AUX20                   PA_CHANNEL_POSITION_AUX20
#define MA_PA_CHANNEL_POSITION_AUX21                   PA_CHANNEL_POSITION_AUX21
#define MA_PA_CHANNEL_POSITION_AUX22                   PA_CHANNEL_POSITION_AUX22
#define MA_PA_CHANNEL_POSITION_AUX23                   PA_CHANNEL_POSITION_AUX23
#define MA_PA_CHANNEL_POSITION_AUX24                   PA_CHANNEL_POSITION_AUX24
#define MA_PA_CHANNEL_POSITION_AUX25                   PA_CHANNEL_POSITION_AUX25
#define MA_PA_CHANNEL_POSITION_AUX26                   PA_CHANNEL_POSITION_AUX26
#define MA_PA_CHANNEL_POSITION_AUX27                   PA_CHANNEL_POSITION_AUX27
#define MA_PA_CHANNEL_POSITION_AUX28                   PA_CHANNEL_POSITION_AUX28
#define MA_PA_CHANNEL_POSITION_AUX29                   PA_CHANNEL_POSITION_AUX29
#define MA_PA_CHANNEL_POSITION_AUX30                   PA_CHANNEL_POSITION_AUX30
#define MA_PA_CHANNEL_POSITION_AUX31                   PA_CHANNEL_POSITION_AUX31
#define MA_PA_CHANNEL_POSITION_TOP_CENTER              PA_CHANNEL_POSITION_TOP_CENTER
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_LEFT          PA_CHANNEL_POSITION_TOP_FRONT_LEFT
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT         PA_CHANNEL_POSITION_TOP_FRONT_RIGHT
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_CENTER        PA_CHANNEL_POSITION_TOP_FRONT_CENTER
#define MA_PA_CHANNEL_POSITION_TOP_REAR_LEFT           PA_CHANNEL_POSITION_TOP_REAR_LEFT
#define MA_PA_CHANNEL_POSITION_TOP_REAR_RIGHT          PA_CHANNEL_POSITION_TOP_REAR_RIGHT
#define MA_PA_CHANNEL_POSITION_TOP_REAR_CENTER         PA_CHANNEL_POSITION_TOP_REAR_CENTER
#define MA_PA_CHANNEL_POSITION_LEFT                    PA_CHANNEL_POSITION_LEFT
#define MA_PA_CHANNEL_POSITION_RIGHT                   PA_CHANNEL_POSITION_RIGHT
#define MA_PA_CHANNEL_POSITION_CENTER                  PA_CHANNEL_POSITION_CENTER
#define MA_PA_CHANNEL_POSITION_SUBWOOFER               PA_CHANNEL_POSITION_SUBWOOFER

typedef pa_channel_map_def_t ma_pa_channel_map_def_t;
#define MA_PA_CHANNEL_MAP_AIFF                         PA_CHANNEL_MAP_AIFF
#define MA_PA_CHANNEL_MAP_ALSA                         PA_CHANNEL_MAP_ALSA
#define MA_PA_CHANNEL_MAP_AUX                          PA_CHANNEL_MAP_AUX
#define MA_PA_CHANNEL_MAP_WAVEEX                       PA_CHANNEL_MAP_WAVEEX
#define MA_PA_CHANNEL_MAP_OSS                          PA_CHANNEL_MAP_OSS
#define MA_PA_CHANNEL_MAP_DEFAULT                      PA_CHANNEL_MAP_DEFAULT

typedef pa_sample_format_t ma_pa_sample_format_t;
#define MA_PA_SAMPLE_INVALID                           PA_SAMPLE_INVALID
#define MA_PA_SAMPLE_U8                                PA_SAMPLE_U8
#define MA_PA_SAMPLE_ALAW                              PA_SAMPLE_ALAW
#define MA_PA_SAMPLE_ULAW                              PA_SAMPLE_ULAW
#define MA_PA_SAMPLE_S16LE                             PA_SAMPLE_S16LE
#define MA_PA_SAMPLE_S16BE                             PA_SAMPLE_S16BE
#define MA_PA_SAMPLE_FLOAT32LE                         PA_SAMPLE_FLOAT32LE
#define MA_PA_SAMPLE_FLOAT32BE                         PA_SAMPLE_FLOAT32BE
#define MA_PA_SAMPLE_S32LE                             PA_SAMPLE_S32LE
#define MA_PA_SAMPLE_S32BE                             PA_SAMPLE_S32BE
#define MA_PA_SAMPLE_S24LE                             PA_SAMPLE_S24LE
#define MA_PA_SAMPLE_S24BE                             PA_SAMPLE_S24BE
#define MA_PA_SAMPLE_S24_32LE                          PA_SAMPLE_S24_32LE
#define MA_PA_SAMPLE_S24_32BE                          PA_SAMPLE_S24_32BE

typedef pa_mainloop     ma_pa_mainloop;
typedef pa_mainloop_api ma_pa_mainloop_api;
typedef pa_context      ma_pa_context;
typedef pa_operation    ma_pa_operation;
typedef pa_stream       ma_pa_stream;
typedef pa_spawn_api    ma_pa_spawn_api;
typedef pa_buffer_attr  ma_pa_buffer_attr;
typedef pa_channel_map  ma_pa_channel_map;
typedef pa_cvolume      ma_pa_cvolume;
typedef pa_sample_spec  ma_pa_sample_spec;
typedef pa_sink_info    ma_pa_sink_info;
typedef pa_source_info  ma_pa_source_info;

typedef pa_context_notify_cb_t ma_pa_context_notify_cb_t;
typedef pa_sink_info_cb_t ma_pa_sink_info_cb_t;
typedef pa_source_info_cb_t ma_pa_source_info_cb_t;
typedef pa_stream_success_cb_t ma_pa_stream_success_cb_t;
typedef pa_stream_request_cb_t ma_pa_stream_request_cb_t;
typedef pa_free_cb_t ma_pa_free_cb_t;
#else
#define MA_PA_OK                                       0
#define MA_PA_ERR_ACCESS                               1
#define MA_PA_ERR_INVALID                              2
#define MA_PA_ERR_NOENTITY                             5

#define MA_PA_CHANNELS_MAX                             32
#define MA_PA_RATE_MAX                                 384000

typedef int ma_pa_context_flags_t;
#define MA_PA_CONTEXT_NOFLAGS                          0x00000000
#define MA_PA_CONTEXT_NOAUTOSPAWN                      0x00000001
#define MA_PA_CONTEXT_NOFAIL                           0x00000002

typedef int ma_pa_stream_flags_t;
#define MA_PA_STREAM_NOFLAGS                           0x00000000
#define MA_PA_STREAM_START_CORKED                      0x00000001
#define MA_PA_STREAM_INTERPOLATE_TIMING                0x00000002
#define MA_PA_STREAM_NOT_MONOTONIC                     0x00000004
#define MA_PA_STREAM_AUTO_TIMING_UPDATE                0x00000008
#define MA_PA_STREAM_NO_REMAP_CHANNELS                 0x00000010
#define MA_PA_STREAM_NO_REMIX_CHANNELS                 0x00000020
#define MA_PA_STREAM_FIX_FORMAT                        0x00000040
#define MA_PA_STREAM_FIX_RATE                          0x00000080
#define MA_PA_STREAM_FIX_CHANNELS                      0x00000100
#define MA_PA_STREAM_DONT_MOVE                         0x00000200
#define MA_PA_STREAM_VARIABLE_RATE                     0x00000400
#define MA_PA_STREAM_PEAK_DETECT                       0x00000800
#define MA_PA_STREAM_START_MUTED                       0x00001000
#define MA_PA_STREAM_ADJUST_LATENCY                    0x00002000
#define MA_PA_STREAM_EARLY_REQUESTS                    0x00004000
#define MA_PA_STREAM_DONT_INHIBIT_AUTO_SUSPEND         0x00008000
#define MA_PA_STREAM_START_UNMUTED                     0x00010000
#define MA_PA_STREAM_FAIL_ON_SUSPEND                   0x00020000
#define MA_PA_STREAM_RELATIVE_VOLUME                   0x00040000
#define MA_PA_STREAM_PASSTHROUGH                       0x00080000

typedef int ma_pa_sink_flags_t;
#define MA_PA_SINK_NOFLAGS                             0x00000000
#define MA_PA_SINK_HW_VOLUME_CTRL                      0x00000001
#define MA_PA_SINK_LATENCY                             0x00000002
#define MA_PA_SINK_HARDWARE                            0x00000004
#define MA_PA_SINK_NETWORK                             0x00000008
#define MA_PA_SINK_HW_MUTE_CTRL                        0x00000010
#define MA_PA_SINK_DECIBEL_VOLUME                      0x00000020
#define MA_PA_SINK_FLAT_VOLUME                         0x00000040
#define MA_PA_SINK_DYNAMIC_LATENCY                     0x00000080
#define MA_PA_SINK_SET_FORMATS                         0x00000100

typedef int ma_pa_source_flags_t;
#define MA_PA_SOURCE_NOFLAGS                           0x00000000
#define MA_PA_SOURCE_HW_VOLUME_CTRL                    0x00000001
#define MA_PA_SOURCE_LATENCY                           0x00000002
#define MA_PA_SOURCE_HARDWARE                          0x00000004
#define MA_PA_SOURCE_NETWORK                           0x00000008
#define MA_PA_SOURCE_HW_MUTE_CTRL                      0x00000010
#define MA_PA_SOURCE_DECIBEL_VOLUME                    0x00000020
#define MA_PA_SOURCE_DYNAMIC_LATENCY                   0x00000040
#define MA_PA_SOURCE_FLAT_VOLUME                       0x00000080

typedef int ma_pa_context_state_t;
#define MA_PA_CONTEXT_UNCONNECTED                      0
#define MA_PA_CONTEXT_CONNECTING                       1
#define MA_PA_CONTEXT_AUTHORIZING                      2
#define MA_PA_CONTEXT_SETTING_NAME                     3
#define MA_PA_CONTEXT_READY                            4
#define MA_PA_CONTEXT_FAILED                           5
#define MA_PA_CONTEXT_TERMINATED                       6

typedef int ma_pa_stream_state_t;
#define MA_PA_STREAM_UNCONNECTED                       0
#define MA_PA_STREAM_CREATING                          1
#define MA_PA_STREAM_READY                             2
#define MA_PA_STREAM_FAILED                            3
#define MA_PA_STREAM_TERMINATED                        4

typedef int ma_pa_operation_state_t;
#define MA_PA_OPERATION_RUNNING                        0
#define MA_PA_OPERATION_DONE                           1
#define MA_PA_OPERATION_CANCELLED                      2

typedef int ma_pa_sink_state_t;
#define MA_PA_SINK_INVALID_STATE                       -1
#define MA_PA_SINK_RUNNING                             0
#define MA_PA_SINK_IDLE                                1
#define MA_PA_SINK_SUSPENDED                           2

typedef int ma_pa_source_state_t;
#define MA_PA_SOURCE_INVALID_STATE                     -1
#define MA_PA_SOURCE_RUNNING                           0
#define MA_PA_SOURCE_IDLE                              1
#define MA_PA_SOURCE_SUSPENDED                         2

typedef int ma_pa_seek_mode_t;
#define MA_PA_SEEK_RELATIVE                            0
#define MA_PA_SEEK_ABSOLUTE                            1
#define MA_PA_SEEK_RELATIVE_ON_READ                    2
#define MA_PA_SEEK_RELATIVE_END                        3

typedef int ma_pa_channel_position_t;
#define MA_PA_CHANNEL_POSITION_INVALID                 -1
#define MA_PA_CHANNEL_POSITION_MONO                    0
#define MA_PA_CHANNEL_POSITION_FRONT_LEFT              1
#define MA_PA_CHANNEL_POSITION_FRONT_RIGHT             2
#define MA_PA_CHANNEL_POSITION_FRONT_CENTER            3
#define MA_PA_CHANNEL_POSITION_REAR_CENTER             4
#define MA_PA_CHANNEL_POSITION_REAR_LEFT               5
#define MA_PA_CHANNEL_POSITION_REAR_RIGHT              6
#define MA_PA_CHANNEL_POSITION_LFE                     7
#define MA_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER    8
#define MA_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER   9
#define MA_PA_CHANNEL_POSITION_SIDE_LEFT               10
#define MA_PA_CHANNEL_POSITION_SIDE_RIGHT              11
#define MA_PA_CHANNEL_POSITION_AUX0                    12
#define MA_PA_CHANNEL_POSITION_AUX1                    13
#define MA_PA_CHANNEL_POSITION_AUX2                    14
#define MA_PA_CHANNEL_POSITION_AUX3                    15
#define MA_PA_CHANNEL_POSITION_AUX4                    16
#define MA_PA_CHANNEL_POSITION_AUX5                    17
#define MA_PA_CHANNEL_POSITION_AUX6                    18
#define MA_PA_CHANNEL_POSITION_AUX7                    19
#define MA_PA_CHANNEL_POSITION_AUX8                    20
#define MA_PA_CHANNEL_POSITION_AUX9                    21
#define MA_PA_CHANNEL_POSITION_AUX10                   22
#define MA_PA_CHANNEL_POSITION_AUX11                   23
#define MA_PA_CHANNEL_POSITION_AUX12                   24
#define MA_PA_CHANNEL_POSITION_AUX13                   25
#define MA_PA_CHANNEL_POSITION_AUX14                   26
#define MA_PA_CHANNEL_POSITION_AUX15                   27
#define MA_PA_CHANNEL_POSITION_AUX16                   28
#define MA_PA_CHANNEL_POSITION_AUX17                   29
#define MA_PA_CHANNEL_POSITION_AUX18                   30
#define MA_PA_CHANNEL_POSITION_AUX19                   31
#define MA_PA_CHANNEL_POSITION_AUX20                   32
#define MA_PA_CHANNEL_POSITION_AUX21                   33
#define MA_PA_CHANNEL_POSITION_AUX22                   34
#define MA_PA_CHANNEL_POSITION_AUX23                   35
#define MA_PA_CHANNEL_POSITION_AUX24                   36
#define MA_PA_CHANNEL_POSITION_AUX25                   37
#define MA_PA_CHANNEL_POSITION_AUX26                   38
#define MA_PA_CHANNEL_POSITION_AUX27                   39
#define MA_PA_CHANNEL_POSITION_AUX28                   40
#define MA_PA_CHANNEL_POSITION_AUX29                   41
#define MA_PA_CHANNEL_POSITION_AUX30                   42
#define MA_PA_CHANNEL_POSITION_AUX31                   43
#define MA_PA_CHANNEL_POSITION_TOP_CENTER              44
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_LEFT          45
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT         46
#define MA_PA_CHANNEL_POSITION_TOP_FRONT_CENTER        47
#define MA_PA_CHANNEL_POSITION_TOP_REAR_LEFT           48
#define MA_PA_CHANNEL_POSITION_TOP_REAR_RIGHT          49
#define MA_PA_CHANNEL_POSITION_TOP_REAR_CENTER         50
#define MA_PA_CHANNEL_POSITION_LEFT                    MA_PA_CHANNEL_POSITION_FRONT_LEFT
#define MA_PA_CHANNEL_POSITION_RIGHT                   MA_PA_CHANNEL_POSITION_FRONT_RIGHT
#define MA_PA_CHANNEL_POSITION_CENTER                  MA_PA_CHANNEL_POSITION_FRONT_CENTER
#define MA_PA_CHANNEL_POSITION_SUBWOOFER               MA_PA_CHANNEL_POSITION_LFE

typedef int ma_pa_channel_map_def_t;
#define MA_PA_CHANNEL_MAP_AIFF                         0
#define MA_PA_CHANNEL_MAP_ALSA                         1
#define MA_PA_CHANNEL_MAP_AUX                          2
#define MA_PA_CHANNEL_MAP_WAVEEX                       3
#define MA_PA_CHANNEL_MAP_OSS                          4
#define MA_PA_CHANNEL_MAP_DEFAULT                      MA_PA_CHANNEL_MAP_AIFF

typedef int ma_pa_sample_format_t;
#define MA_PA_SAMPLE_INVALID                           -1
#define MA_PA_SAMPLE_U8                                0
#define MA_PA_SAMPLE_ALAW                              1
#define MA_PA_SAMPLE_ULAW                              2
#define MA_PA_SAMPLE_S16LE                             3
#define MA_PA_SAMPLE_S16BE                             4
#define MA_PA_SAMPLE_FLOAT32LE                         5
#define MA_PA_SAMPLE_FLOAT32BE                         6
#define MA_PA_SAMPLE_S32LE                             7
#define MA_PA_SAMPLE_S32BE                             8
#define MA_PA_SAMPLE_S24LE                             9
#define MA_PA_SAMPLE_S24BE                             10
#define MA_PA_SAMPLE_S24_32LE                          11
#define MA_PA_SAMPLE_S24_32BE                          12

typedef struct ma_pa_mainloop     ma_pa_mainloop;
typedef struct ma_pa_mainloop_api ma_pa_mainloop_api;
typedef struct ma_pa_context      ma_pa_context;
typedef struct ma_pa_operation    ma_pa_operation;
typedef struct ma_pa_stream       ma_pa_stream;
typedef struct ma_pa_spawn_api    ma_pa_spawn_api;

typedef struct
{
    ma_uint32 maxlength;
    ma_uint32 tlength;
    ma_uint32 prebuf;
    ma_uint32 minreq;
    ma_uint32 fragsize;
} ma_pa_buffer_attr;

typedef struct
{
    ma_uint8 channels;
    ma_pa_channel_position_t map[MA_PA_CHANNELS_MAX];
} ma_pa_channel_map;

typedef struct
{
    ma_uint8 channels;
    ma_uint32 values[MA_PA_CHANNELS_MAX];
} ma_pa_cvolume;

typedef struct
{
    ma_pa_sample_format_t format;
    ma_uint32 rate;
    ma_uint8 channels;
} ma_pa_sample_spec;

typedef struct
{
    const char* name;
    ma_uint32 index;
    const char* description;
    ma_pa_sample_spec sample_spec;
    ma_pa_channel_map channel_map;
    ma_uint32 owner_module;
    ma_pa_cvolume volume;
    int mute;
    ma_uint32 monitor_source;
    const char* monitor_source_name;
    ma_uint64 latency;
    const char* driver;
    ma_pa_sink_flags_t flags;
    void* proplist;
    ma_uint64 configured_latency;
    ma_uint32 base_volume;
    ma_pa_sink_state_t state;
    ma_uint32 n_volume_steps;
    ma_uint32 card;
    ma_uint32 n_ports;
    void** ports;
    void* active_port;
    ma_uint8 n_formats;
    void** formats;
} ma_pa_sink_info;

typedef struct
{
    const char *name;
    ma_uint32 index;
    const char *description;
    ma_pa_sample_spec sample_spec;
    ma_pa_channel_map channel_map;
    ma_uint32 owner_module;
    ma_pa_cvolume volume;
    int mute;
    ma_uint32 monitor_of_sink;
    const char *monitor_of_sink_name;
    ma_uint64 latency;
    const char *driver;
    ma_pa_source_flags_t flags;
    void* proplist;
    ma_uint64 configured_latency;
    ma_uint32 base_volume;
    ma_pa_source_state_t state;
    ma_uint32 n_volume_steps;
    ma_uint32 card;
    ma_uint32 n_ports;
    void** ports;
    void* active_port;
    ma_uint8 n_formats;
    void** formats;
} ma_pa_source_info;

typedef void (* ma_pa_context_notify_cb_t)(ma_pa_context* c, void* userdata);
typedef void (* ma_pa_sink_info_cb_t)     (ma_pa_context* c, const ma_pa_sink_info* i, int eol, void* userdata);
typedef void (* ma_pa_source_info_cb_t)   (ma_pa_context* c, const ma_pa_source_info* i, int eol, void* userdata);
typedef void (* ma_pa_stream_success_cb_t)(ma_pa_stream* s, int success, void* userdata);
typedef void (* ma_pa_stream_request_cb_t)(ma_pa_stream* s, size_t nbytes, void* userdata);
typedef void (* ma_pa_free_cb_t)          (void* p);
#endif


typedef ma_pa_mainloop*          (* ma_pa_mainloop_new_proc)                   ();
typedef void                     (* ma_pa_mainloop_free_proc)                  (ma_pa_mainloop* m);
typedef ma_pa_mainloop_api*      (* ma_pa_mainloop_get_api_proc)               (ma_pa_mainloop* m);
typedef int                      (* ma_pa_mainloop_iterate_proc)               (ma_pa_mainloop* m, int block, int* retval);
typedef void                     (* ma_pa_mainloop_wakeup_proc)                (ma_pa_mainloop* m);
typedef ma_pa_context*           (* ma_pa_context_new_proc)                    (ma_pa_mainloop_api* mainloop, const char* name);
typedef void                     (* ma_pa_context_unref_proc)                  (ma_pa_context* c);
typedef int                      (* ma_pa_context_connect_proc)                (ma_pa_context* c, const char* server, ma_pa_context_flags_t flags, const ma_pa_spawn_api* api);
typedef void                     (* ma_pa_context_disconnect_proc)             (ma_pa_context* c);
typedef void                     (* ma_pa_context_set_state_callback_proc)     (ma_pa_context* c, ma_pa_context_notify_cb_t cb, void* userdata);
typedef ma_pa_context_state_t    (* ma_pa_context_get_state_proc)              (ma_pa_context* c);
typedef ma_pa_operation*         (* ma_pa_context_get_sink_info_list_proc)     (ma_pa_context* c, ma_pa_sink_info_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_context_get_source_info_list_proc)   (ma_pa_context* c, ma_pa_source_info_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_context_get_sink_info_by_name_proc)  (ma_pa_context* c, const char* name, ma_pa_sink_info_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_context_get_source_info_by_name_proc)(ma_pa_context* c, const char* name, ma_pa_source_info_cb_t cb, void* userdata);
typedef void                     (* ma_pa_operation_unref_proc)                (ma_pa_operation* o);
typedef ma_pa_operation_state_t  (* ma_pa_operation_get_state_proc)            (ma_pa_operation* o);
typedef ma_pa_channel_map*       (* ma_pa_channel_map_init_extend_proc)        (ma_pa_channel_map* m, unsigned channels, ma_pa_channel_map_def_t def);
typedef int                      (* ma_pa_channel_map_valid_proc)              (const ma_pa_channel_map* m);
typedef int                      (* ma_pa_channel_map_compatible_proc)         (const ma_pa_channel_map* m, const ma_pa_sample_spec* ss);
typedef ma_pa_stream*            (* ma_pa_stream_new_proc)                     (ma_pa_context* c, const char* name, const ma_pa_sample_spec* ss, const ma_pa_channel_map* map);
typedef void                     (* ma_pa_stream_unref_proc)                   (ma_pa_stream* s);
typedef int                      (* ma_pa_stream_connect_playback_proc)        (ma_pa_stream* s, const char* dev, const ma_pa_buffer_attr* attr, ma_pa_stream_flags_t flags, const ma_pa_cvolume* volume, ma_pa_stream* sync_stream);
typedef int                      (* ma_pa_stream_connect_record_proc)          (ma_pa_stream* s, const char* dev, const ma_pa_buffer_attr* attr, ma_pa_stream_flags_t flags);
typedef int                      (* ma_pa_stream_disconnect_proc)              (ma_pa_stream* s);
typedef ma_pa_stream_state_t     (* ma_pa_stream_get_state_proc)               (ma_pa_stream* s);
typedef const ma_pa_sample_spec* (* ma_pa_stream_get_sample_spec_proc)         (ma_pa_stream* s);
typedef const ma_pa_channel_map* (* ma_pa_stream_get_channel_map_proc)         (ma_pa_stream* s);
typedef const ma_pa_buffer_attr* (* ma_pa_stream_get_buffer_attr_proc)         (ma_pa_stream* s);
typedef ma_pa_operation*         (* ma_pa_stream_set_buffer_attr_proc)         (ma_pa_stream* s, const ma_pa_buffer_attr* attr, ma_pa_stream_success_cb_t cb, void* userdata);
typedef const char*              (* ma_pa_stream_get_device_name_proc)         (ma_pa_stream* s);
typedef void                     (* ma_pa_stream_set_write_callback_proc)      (ma_pa_stream* s, ma_pa_stream_request_cb_t cb, void* userdata);
typedef void                     (* ma_pa_stream_set_read_callback_proc)       (ma_pa_stream* s, ma_pa_stream_request_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_stream_flush_proc)                   (ma_pa_stream* s, ma_pa_stream_success_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_stream_drain_proc)                   (ma_pa_stream* s, ma_pa_stream_success_cb_t cb, void* userdata);
typedef int                      (* ma_pa_stream_is_corked_proc)               (ma_pa_stream* s);
typedef ma_pa_operation*         (* ma_pa_stream_cork_proc)                    (ma_pa_stream* s, int b, ma_pa_stream_success_cb_t cb, void* userdata);
typedef ma_pa_operation*         (* ma_pa_stream_trigger_proc)                 (ma_pa_stream* s, ma_pa_stream_success_cb_t cb, void* userdata);
typedef int                      (* ma_pa_stream_begin_write_proc)             (ma_pa_stream* s, void** data, size_t* nbytes);
typedef int                      (* ma_pa_stream_write_proc)                   (ma_pa_stream* s, const void* data, size_t nbytes, ma_pa_free_cb_t free_cb, int64_t offset, ma_pa_seek_mode_t seek);
typedef int                      (* ma_pa_stream_peek_proc)                    (ma_pa_stream* s, const void** data, size_t* nbytes);
typedef int                      (* ma_pa_stream_drop_proc)                    (ma_pa_stream* s);
typedef size_t                   (* ma_pa_stream_writable_size_proc)           (ma_pa_stream* s);
typedef size_t                   (* ma_pa_stream_readable_size_proc)           (ma_pa_stream* s);

typedef struct
{
    ma_uint32 count;
    ma_uint32 capacity;
    ma_device_info* pInfo;
} ma_pulse_device_enum_data;

ma_result ma_result_from_pulse(int result)
{
    switch (result) {
        case MA_PA_OK:           return MA_SUCCESS;
        case MA_PA_ERR_ACCESS:   return MA_ACCESS_DENIED;
        case MA_PA_ERR_INVALID:  return MA_INVALID_ARGS;
        case MA_PA_ERR_NOENTITY: return MA_NO_DEVICE;
        default:                  return MA_ERROR;
    }
}

#if 0
ma_pa_sample_format_t ma_format_to_pulse(ma_format format)
{
    if (ma_is_little_endian()) {
        switch (format) {
            case ma_format_s16: return MA_PA_SAMPLE_S16LE;
            case ma_format_s24: return MA_PA_SAMPLE_S24LE;
            case ma_format_s32: return MA_PA_SAMPLE_S32LE;
            case ma_format_f32: return MA_PA_SAMPLE_FLOAT32LE;
            default: break;
        }
    } else {
        switch (format) {
            case ma_format_s16: return MA_PA_SAMPLE_S16BE;
            case ma_format_s24: return MA_PA_SAMPLE_S24BE;
            case ma_format_s32: return MA_PA_SAMPLE_S32BE;
            case ma_format_f32: return MA_PA_SAMPLE_FLOAT32BE;
            default: break;
        }
    }

    /* Endian agnostic. */
    switch (format) {
        case ma_format_u8: return MA_PA_SAMPLE_U8;
        default: return MA_PA_SAMPLE_INVALID;
    }
}
#endif

ma_format ma_format_from_pulse(ma_pa_sample_format_t format)
{
    if (ma_is_little_endian()) {
        switch (format) {
            case MA_PA_SAMPLE_S16LE:     return ma_format_s16;
            case MA_PA_SAMPLE_S24LE:     return ma_format_s24;
            case MA_PA_SAMPLE_S32LE:     return ma_format_s32;
            case MA_PA_SAMPLE_FLOAT32LE: return ma_format_f32;
            default: break;
        }
    } else {
        switch (format) {
            case MA_PA_SAMPLE_S16BE:     return ma_format_s16;
            case MA_PA_SAMPLE_S24BE:     return ma_format_s24;
            case MA_PA_SAMPLE_S32BE:     return ma_format_s32;
            case MA_PA_SAMPLE_FLOAT32BE: return ma_format_f32;
            default: break;
        }
    }

    /* Endian agnostic. */
    switch (format) {
        case MA_PA_SAMPLE_U8: return ma_format_u8;
        default: return ma_format_unknown;
    }
}

ma_channel ma_channel_position_from_pulse(ma_pa_channel_position_t position)
{
    switch (position)
    {
        case MA_PA_CHANNEL_POSITION_INVALID:               return MA_CHANNEL_NONE;
        case MA_PA_CHANNEL_POSITION_MONO:                  return MA_CHANNEL_MONO;
        case MA_PA_CHANNEL_POSITION_FRONT_LEFT:            return MA_CHANNEL_FRONT_LEFT;
        case MA_PA_CHANNEL_POSITION_FRONT_RIGHT:           return MA_CHANNEL_FRONT_RIGHT;
        case MA_PA_CHANNEL_POSITION_FRONT_CENTER:          return MA_CHANNEL_FRONT_CENTER;
        case MA_PA_CHANNEL_POSITION_REAR_CENTER:           return MA_CHANNEL_BACK_CENTER;
        case MA_PA_CHANNEL_POSITION_REAR_LEFT:             return MA_CHANNEL_BACK_LEFT;
        case MA_PA_CHANNEL_POSITION_REAR_RIGHT:            return MA_CHANNEL_BACK_RIGHT;
        case MA_PA_CHANNEL_POSITION_LFE:                   return MA_CHANNEL_LFE;
        case MA_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER:  return MA_CHANNEL_FRONT_LEFT_CENTER;
        case MA_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER: return MA_CHANNEL_FRONT_RIGHT_CENTER;
        case MA_PA_CHANNEL_POSITION_SIDE_LEFT:             return MA_CHANNEL_SIDE_LEFT;
        case MA_PA_CHANNEL_POSITION_SIDE_RIGHT:            return MA_CHANNEL_SIDE_RIGHT;
        case MA_PA_CHANNEL_POSITION_AUX0:                  return MA_CHANNEL_AUX_0;
        case MA_PA_CHANNEL_POSITION_AUX1:                  return MA_CHANNEL_AUX_1;
        case MA_PA_CHANNEL_POSITION_AUX2:                  return MA_CHANNEL_AUX_2;
        case MA_PA_CHANNEL_POSITION_AUX3:                  return MA_CHANNEL_AUX_3;
        case MA_PA_CHANNEL_POSITION_AUX4:                  return MA_CHANNEL_AUX_4;
        case MA_PA_CHANNEL_POSITION_AUX5:                  return MA_CHANNEL_AUX_5;
        case MA_PA_CHANNEL_POSITION_AUX6:                  return MA_CHANNEL_AUX_6;
        case MA_PA_CHANNEL_POSITION_AUX7:                  return MA_CHANNEL_AUX_7;
        case MA_PA_CHANNEL_POSITION_AUX8:                  return MA_CHANNEL_AUX_8;
        case MA_PA_CHANNEL_POSITION_AUX9:                  return MA_CHANNEL_AUX_9;
        case MA_PA_CHANNEL_POSITION_AUX10:                 return MA_CHANNEL_AUX_10;
        case MA_PA_CHANNEL_POSITION_AUX11:                 return MA_CHANNEL_AUX_11;
        case MA_PA_CHANNEL_POSITION_AUX12:                 return MA_CHANNEL_AUX_12;
        case MA_PA_CHANNEL_POSITION_AUX13:                 return MA_CHANNEL_AUX_13;
        case MA_PA_CHANNEL_POSITION_AUX14:                 return MA_CHANNEL_AUX_14;
        case MA_PA_CHANNEL_POSITION_AUX15:                 return MA_CHANNEL_AUX_15;
        case MA_PA_CHANNEL_POSITION_AUX16:                 return MA_CHANNEL_AUX_16;
        case MA_PA_CHANNEL_POSITION_AUX17:                 return MA_CHANNEL_AUX_17;
        case MA_PA_CHANNEL_POSITION_AUX18:                 return MA_CHANNEL_AUX_18;
        case MA_PA_CHANNEL_POSITION_AUX19:                 return MA_CHANNEL_AUX_19;
        case MA_PA_CHANNEL_POSITION_AUX20:                 return MA_CHANNEL_AUX_20;
        case MA_PA_CHANNEL_POSITION_AUX21:                 return MA_CHANNEL_AUX_21;
        case MA_PA_CHANNEL_POSITION_AUX22:                 return MA_CHANNEL_AUX_22;
        case MA_PA_CHANNEL_POSITION_AUX23:                 return MA_CHANNEL_AUX_23;
        case MA_PA_CHANNEL_POSITION_AUX24:                 return MA_CHANNEL_AUX_24;
        case MA_PA_CHANNEL_POSITION_AUX25:                 return MA_CHANNEL_AUX_25;
        case MA_PA_CHANNEL_POSITION_AUX26:                 return MA_CHANNEL_AUX_26;
        case MA_PA_CHANNEL_POSITION_AUX27:                 return MA_CHANNEL_AUX_27;
        case MA_PA_CHANNEL_POSITION_AUX28:                 return MA_CHANNEL_AUX_28;
        case MA_PA_CHANNEL_POSITION_AUX29:                 return MA_CHANNEL_AUX_29;
        case MA_PA_CHANNEL_POSITION_AUX30:                 return MA_CHANNEL_AUX_30;
        case MA_PA_CHANNEL_POSITION_AUX31:                 return MA_CHANNEL_AUX_31;
        case MA_PA_CHANNEL_POSITION_TOP_CENTER:            return MA_CHANNEL_TOP_CENTER;
        case MA_PA_CHANNEL_POSITION_TOP_FRONT_LEFT:        return MA_CHANNEL_TOP_FRONT_LEFT;
        case MA_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT:       return MA_CHANNEL_TOP_FRONT_RIGHT;
        case MA_PA_CHANNEL_POSITION_TOP_FRONT_CENTER:      return MA_CHANNEL_TOP_FRONT_CENTER;
        case MA_PA_CHANNEL_POSITION_TOP_REAR_LEFT:         return MA_CHANNEL_TOP_BACK_LEFT;
        case MA_PA_CHANNEL_POSITION_TOP_REAR_RIGHT:        return MA_CHANNEL_TOP_BACK_RIGHT;
        case MA_PA_CHANNEL_POSITION_TOP_REAR_CENTER:       return MA_CHANNEL_TOP_BACK_CENTER;
        default: return MA_CHANNEL_NONE;
    }
}

#if 0
ma_pa_channel_position_t ma_channel_position_to_pulse(ma_channel position)
{
    switch (position)
    {
        case MA_CHANNEL_NONE:               return MA_PA_CHANNEL_POSITION_INVALID;
        case MA_CHANNEL_FRONT_LEFT:         return MA_PA_CHANNEL_POSITION_FRONT_LEFT;
        case MA_CHANNEL_FRONT_RIGHT:        return MA_PA_CHANNEL_POSITION_FRONT_RIGHT;
        case MA_CHANNEL_FRONT_CENTER:       return MA_PA_CHANNEL_POSITION_FRONT_CENTER;
        case MA_CHANNEL_LFE:                return MA_PA_CHANNEL_POSITION_LFE;
        case MA_CHANNEL_BACK_LEFT:          return MA_PA_CHANNEL_POSITION_REAR_LEFT;
        case MA_CHANNEL_BACK_RIGHT:         return MA_PA_CHANNEL_POSITION_REAR_RIGHT;
        case MA_CHANNEL_FRONT_LEFT_CENTER:  return MA_PA_CHANNEL_POSITION_FRONT_LEFT_OF_CENTER;
        case MA_CHANNEL_FRONT_RIGHT_CENTER: return MA_PA_CHANNEL_POSITION_FRONT_RIGHT_OF_CENTER;
        case MA_CHANNEL_BACK_CENTER:        return MA_PA_CHANNEL_POSITION_REAR_CENTER;
        case MA_CHANNEL_SIDE_LEFT:          return MA_PA_CHANNEL_POSITION_SIDE_LEFT;
        case MA_CHANNEL_SIDE_RIGHT:         return MA_PA_CHANNEL_POSITION_SIDE_RIGHT;
        case MA_CHANNEL_TOP_CENTER:         return MA_PA_CHANNEL_POSITION_TOP_CENTER;
        case MA_CHANNEL_TOP_FRONT_LEFT:     return MA_PA_CHANNEL_POSITION_TOP_FRONT_LEFT;
        case MA_CHANNEL_TOP_FRONT_CENTER:   return MA_PA_CHANNEL_POSITION_TOP_FRONT_CENTER;
        case MA_CHANNEL_TOP_FRONT_RIGHT:    return MA_PA_CHANNEL_POSITION_TOP_FRONT_RIGHT;
        case MA_CHANNEL_TOP_BACK_LEFT:      return MA_PA_CHANNEL_POSITION_TOP_REAR_LEFT;
        case MA_CHANNEL_TOP_BACK_CENTER:    return MA_PA_CHANNEL_POSITION_TOP_REAR_CENTER;
        case MA_CHANNEL_TOP_BACK_RIGHT:     return MA_PA_CHANNEL_POSITION_TOP_REAR_RIGHT;
        case MA_CHANNEL_19:                 return MA_PA_CHANNEL_POSITION_AUX18;
        case MA_CHANNEL_20:                 return MA_PA_CHANNEL_POSITION_AUX19;
        case MA_CHANNEL_21:                 return MA_PA_CHANNEL_POSITION_AUX20;
        case MA_CHANNEL_22:                 return MA_PA_CHANNEL_POSITION_AUX21;
        case MA_CHANNEL_23:                 return MA_PA_CHANNEL_POSITION_AUX22;
        case MA_CHANNEL_24:                 return MA_PA_CHANNEL_POSITION_AUX23;
        case MA_CHANNEL_25:                 return MA_PA_CHANNEL_POSITION_AUX24;
        case MA_CHANNEL_26:                 return MA_PA_CHANNEL_POSITION_AUX25;
        case MA_CHANNEL_27:                 return MA_PA_CHANNEL_POSITION_AUX26;
        case MA_CHANNEL_28:                 return MA_PA_CHANNEL_POSITION_AUX27;
        case MA_CHANNEL_29:                 return MA_PA_CHANNEL_POSITION_AUX28;
        case MA_CHANNEL_30:                 return MA_PA_CHANNEL_POSITION_AUX29;
        case MA_CHANNEL_31:                 return MA_PA_CHANNEL_POSITION_AUX30;
        case MA_CHANNEL_32:                 return MA_PA_CHANNEL_POSITION_AUX31;
        default: return (ma_pa_channel_position_t)position;
    }
}
#endif

ma_result ma_wait_for_operation__pulse(ma_context* pContext, ma_pa_mainloop* pMainLoop, ma_pa_operation* pOP)
{
    ma_assert(pContext != NULL);
    ma_assert(pMainLoop != NULL);
    ma_assert(pOP != NULL);

    while (((ma_pa_operation_get_state_proc)pContext->pulse.pa_operation_get_state)(pOP) == MA_PA_OPERATION_RUNNING) {
        int error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)(pMainLoop, 1, NULL);
        if (error < 0) {
            return ma_result_from_pulse(error);
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device__wait_for_operation__pulse(ma_device* pDevice, ma_pa_operation* pOP)
{
    ma_assert(pDevice != NULL);
    ma_assert(pOP != NULL);

    return ma_wait_for_operation__pulse(pDevice->pContext, (ma_pa_mainloop*)pDevice->pulse.pMainLoop, pOP);
}


ma_bool32 ma_context_is_device_id_equal__pulse(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return ma_strcmp(pID0->pulse, pID1->pulse) == 0;
}


typedef struct
{
    ma_context* pContext;
    ma_enum_devices_callback_proc callback;
    void* pUserData;
    ma_bool32 isTerminated;
} ma_context_enumerate_devices_callback_data__pulse;

void ma_context_enumerate_devices_sink_callback__pulse(ma_pa_context* pPulseContext, const ma_pa_sink_info* pSinkInfo, int endOfList, void* pUserData)
{
    ma_context_enumerate_devices_callback_data__pulse* pData = (ma_context_enumerate_devices_callback_data__pulse*)pUserData;
    ma_device_info deviceInfo;

    ma_assert(pData != NULL);

    if (endOfList || pData->isTerminated) {
        return;
    }

    ma_zero_object(&deviceInfo);

    /* The name from PulseAudio is the ID for miniaudio. */
    if (pSinkInfo->name != NULL) {
        ma_strncpy_s(deviceInfo.id.pulse, sizeof(deviceInfo.id.pulse), pSinkInfo->name, (size_t)-1);
    }

    /* The description from PulseAudio is the name for miniaudio. */
    if (pSinkInfo->description != NULL) {
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), pSinkInfo->description, (size_t)-1);
    }

    pData->isTerminated = !pData->callback(pData->pContext, ma_device_type_playback, &deviceInfo, pData->pUserData);

    (void)pPulseContext; /* Unused. */
}

void ma_context_enumerate_devices_source_callback__pulse(ma_pa_context* pPulseContext, const ma_pa_source_info* pSinkInfo, int endOfList, void* pUserData)
{
    ma_context_enumerate_devices_callback_data__pulse* pData = (ma_context_enumerate_devices_callback_data__pulse*)pUserData;
    ma_device_info deviceInfo;

    ma_assert(pData != NULL);

    if (endOfList || pData->isTerminated) {
        return;
    }

    ma_zero_object(&deviceInfo);

    /* The name from PulseAudio is the ID for miniaudio. */
    if (pSinkInfo->name != NULL) {
        ma_strncpy_s(deviceInfo.id.pulse, sizeof(deviceInfo.id.pulse), pSinkInfo->name, (size_t)-1);
    }

    /* The description from PulseAudio is the name for miniaudio. */
    if (pSinkInfo->description != NULL) {
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), pSinkInfo->description, (size_t)-1);
    }

    pData->isTerminated = !pData->callback(pData->pContext, ma_device_type_capture, &deviceInfo, pData->pUserData);

    (void)pPulseContext; /* Unused. */
}

ma_result ma_context_enumerate_devices__pulse(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_result result = MA_SUCCESS;
    ma_context_enumerate_devices_callback_data__pulse callbackData;
    ma_pa_operation* pOP = NULL;
    ma_pa_mainloop* pMainLoop;
    ma_pa_mainloop_api* pAPI;
    ma_pa_context* pPulseContext;
    int error;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);

    callbackData.pContext = pContext;
    callbackData.callback = callback;
    callbackData.pUserData = pUserData;
    callbackData.isTerminated = MA_FALSE;

    pMainLoop = ((ma_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
    if (pMainLoop == NULL) {
        return MA_FAILED_TO_INIT_BACKEND;
    }

    pAPI = ((ma_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)(pMainLoop);
    if (pAPI == NULL) {
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MA_FAILED_TO_INIT_BACKEND;
    }

    pPulseContext = ((ma_pa_context_new_proc)pContext->pulse.pa_context_new)(pAPI, pContext->pulse.pApplicationName);
    if (pPulseContext == NULL) {
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MA_FAILED_TO_INIT_BACKEND;
    }

    error = ((ma_pa_context_connect_proc)pContext->pulse.pa_context_connect)(pPulseContext, pContext->pulse.pServerName, (pContext->pulse.tryAutoSpawn) ? 0 : MA_PA_CONTEXT_NOAUTOSPAWN, NULL);
    if (error != MA_PA_OK) {
        ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return ma_result_from_pulse(error);
    }

    for (;;) {
        ma_pa_context_state_t state = ((ma_pa_context_get_state_proc)pContext->pulse.pa_context_get_state)(pPulseContext);
        if (state == MA_PA_CONTEXT_READY) {
            break;  /* Success. */
        }
        if (state == MA_PA_CONTEXT_CONNECTING || state == MA_PA_CONTEXT_AUTHORIZING || state == MA_PA_CONTEXT_SETTING_NAME) {
            error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)(pMainLoop, 1, NULL);
            if (error < 0) {
                result = ma_result_from_pulse(error);
                goto done;
            }

#ifdef MA_DEBUG_OUTPUT
            printf("[PulseAudio] pa_context_get_state() returned %d. Waiting.\n", state);
#endif
            continue;   /* Keep trying. */
        }
        if (state == MA_PA_CONTEXT_UNCONNECTED || state == MA_PA_CONTEXT_FAILED || state == MA_PA_CONTEXT_TERMINATED) {
#ifdef MA_DEBUG_OUTPUT
            printf("[PulseAudio] pa_context_get_state() returned %d. Failed.\n", state);
#endif
            goto done;  /* Failed. */
        }
    }


    /* Playback. */
    if (!callbackData.isTerminated) {
        pOP = ((ma_pa_context_get_sink_info_list_proc)pContext->pulse.pa_context_get_sink_info_list)(pPulseContext, ma_context_enumerate_devices_sink_callback__pulse, &callbackData);
        if (pOP == NULL) {
            result = MA_ERROR;
            goto done;
        }

        result = ma_wait_for_operation__pulse(pContext, pMainLoop, pOP);
        ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        if (result != MA_SUCCESS) {
            goto done;
        }
    }


    /* Capture. */
    if (!callbackData.isTerminated) {
        pOP = ((ma_pa_context_get_source_info_list_proc)pContext->pulse.pa_context_get_source_info_list)(pPulseContext, ma_context_enumerate_devices_source_callback__pulse, &callbackData);
        if (pOP == NULL) {
            result = MA_ERROR;
            goto done;
        }

        result = ma_wait_for_operation__pulse(pContext, pMainLoop, pOP);
        ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        if (result != MA_SUCCESS) {
            goto done;
        }
    }

done:
    ((ma_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)(pPulseContext);
    ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
    ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
    return result;
}


typedef struct
{
    ma_device_info* pDeviceInfo;
    ma_bool32 foundDevice;
} ma_context_get_device_info_callback_data__pulse;

void ma_context_get_device_info_sink_callback__pulse(ma_pa_context* pPulseContext, const ma_pa_sink_info* pInfo, int endOfList, void* pUserData)
{
    ma_context_get_device_info_callback_data__pulse* pData = (ma_context_get_device_info_callback_data__pulse*)pUserData;

    if (endOfList > 0) {
        return;
    }

    ma_assert(pData != NULL);
    pData->foundDevice = MA_TRUE;

    if (pInfo->name != NULL) {
        ma_strncpy_s(pData->pDeviceInfo->id.pulse, sizeof(pData->pDeviceInfo->id.pulse), pInfo->name, (size_t)-1);
    }

    if (pInfo->description != NULL) {
        ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pInfo->description, (size_t)-1);
    }

    pData->pDeviceInfo->minChannels   = pInfo->sample_spec.channels;
    pData->pDeviceInfo->maxChannels   = pInfo->sample_spec.channels;
    pData->pDeviceInfo->minSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->maxSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->formatCount = 1;
    pData->pDeviceInfo->formats[0] = ma_format_from_pulse(pInfo->sample_spec.format);

    (void)pPulseContext; /* Unused. */
}

void ma_context_get_device_info_source_callback__pulse(ma_pa_context* pPulseContext, const ma_pa_source_info* pInfo, int endOfList, void* pUserData)
{
    ma_context_get_device_info_callback_data__pulse* pData = (ma_context_get_device_info_callback_data__pulse*)pUserData;

    if (endOfList > 0) {
        return;
    }

    ma_assert(pData != NULL);
    pData->foundDevice = MA_TRUE;

    if (pInfo->name != NULL) {
        ma_strncpy_s(pData->pDeviceInfo->id.pulse, sizeof(pData->pDeviceInfo->id.pulse), pInfo->name, (size_t)-1);
    }

    if (pInfo->description != NULL) {
        ma_strncpy_s(pData->pDeviceInfo->name, sizeof(pData->pDeviceInfo->name), pInfo->description, (size_t)-1);
    }

    pData->pDeviceInfo->minChannels = pInfo->sample_spec.channels;
    pData->pDeviceInfo->maxChannels = pInfo->sample_spec.channels;
    pData->pDeviceInfo->minSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->maxSampleRate = pInfo->sample_spec.rate;
    pData->pDeviceInfo->formatCount = 1;
    pData->pDeviceInfo->formats[0] = ma_format_from_pulse(pInfo->sample_spec.format);

    (void)pPulseContext; /* Unused. */
}

ma_result ma_context_get_device_info__pulse(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_result result = MA_SUCCESS;
    ma_context_get_device_info_callback_data__pulse callbackData;
    ma_pa_operation* pOP = NULL;
    ma_pa_mainloop* pMainLoop;
    ma_pa_mainloop_api* pAPI;
    ma_pa_context* pPulseContext;
    int error;

    ma_assert(pContext != NULL);

    /* No exclusive mode with the PulseAudio backend. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    callbackData.pDeviceInfo = pDeviceInfo;
    callbackData.foundDevice = MA_FALSE;

    pMainLoop = ((ma_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
    if (pMainLoop == NULL) {
        return MA_FAILED_TO_INIT_BACKEND;
    }

    pAPI = ((ma_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)(pMainLoop);
    if (pAPI == NULL) {
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MA_FAILED_TO_INIT_BACKEND;
    }

    pPulseContext = ((ma_pa_context_new_proc)pContext->pulse.pa_context_new)(pAPI, pContext->pulse.pApplicationName);
    if (pPulseContext == NULL) {
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return MA_FAILED_TO_INIT_BACKEND;
    }

    error = ((ma_pa_context_connect_proc)pContext->pulse.pa_context_connect)(pPulseContext, pContext->pulse.pServerName, 0, NULL);
    if (error != MA_PA_OK) {
        ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        return ma_result_from_pulse(error);
    }

    for (;;) {
        ma_pa_context_state_t state = ((ma_pa_context_get_state_proc)pContext->pulse.pa_context_get_state)(pPulseContext);
        if (state == MA_PA_CONTEXT_READY) {
            break;  /* Success. */
        }
        if (state == MA_PA_CONTEXT_CONNECTING || state == MA_PA_CONTEXT_AUTHORIZING || state == MA_PA_CONTEXT_SETTING_NAME) {
            error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)(pMainLoop, 1, NULL);
            if (error < 0) {
                result = ma_result_from_pulse(error);
                goto done;
            }

#ifdef MA_DEBUG_OUTPUT
            printf("[PulseAudio] pa_context_get_state() returned %d. Waiting.\n", state);
#endif
            continue;   /* Keep trying. */
        }
        if (state == MA_PA_CONTEXT_UNCONNECTED || state == MA_PA_CONTEXT_FAILED || state == MA_PA_CONTEXT_TERMINATED) {
#ifdef MA_DEBUG_OUTPUT
            printf("[PulseAudio] pa_context_get_state() returned %d. Failed.\n", state);
#endif
            goto done;  /* Failed. */
        }
    }

    if (deviceType == ma_device_type_playback) {
        pOP = ((ma_pa_context_get_sink_info_by_name_proc)pContext->pulse.pa_context_get_sink_info_by_name)(pPulseContext, pDeviceID->pulse, ma_context_get_device_info_sink_callback__pulse, &callbackData);
    } else {
        pOP = ((ma_pa_context_get_source_info_by_name_proc)pContext->pulse.pa_context_get_source_info_by_name)(pPulseContext, pDeviceID->pulse, ma_context_get_device_info_source_callback__pulse, &callbackData);
    }

    if (pOP != NULL) {
        ma_wait_for_operation__pulse(pContext, pMainLoop, pOP);
        ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
    } else {
        result = MA_ERROR;
        goto done;
    }

    if (!callbackData.foundDevice) {
        result = MA_NO_DEVICE;
        goto done;
    }


done:
    ((ma_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)(pPulseContext);
    ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
    ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
    return result;
}


void ma_pulse_device_state_callback(ma_pa_context* pPulseContext, void* pUserData)
{
    ma_device* pDevice;
    ma_context* pContext;

    pDevice = (ma_device*)pUserData;
    ma_assert(pDevice != NULL);

    pContext = pDevice->pContext;
    ma_assert(pContext != NULL);

    pDevice->pulse.pulseContextState = ((ma_pa_context_get_state_proc)pContext->pulse.pa_context_get_state)(pPulseContext);
}

void ma_device_sink_info_callback(ma_pa_context* pPulseContext, const ma_pa_sink_info* pInfo, int endOfList, void* pUserData)
{
    ma_pa_sink_info* pInfoOut;

    if (endOfList > 0) {
        return;
    }

    pInfoOut = (ma_pa_sink_info*)pUserData;
    ma_assert(pInfoOut != NULL);

    *pInfoOut = *pInfo;

    (void)pPulseContext; /* Unused. */
}

void ma_device_source_info_callback(ma_pa_context* pPulseContext, const ma_pa_source_info* pInfo, int endOfList, void* pUserData)
{
    ma_pa_source_info* pInfoOut;

    if (endOfList > 0) {
        return;
    }

    pInfoOut = (ma_pa_source_info*)pUserData;
    ma_assert(pInfoOut != NULL);

    *pInfoOut = *pInfo;

    (void)pPulseContext; /* Unused. */
}

void ma_device_sink_name_callback(ma_pa_context* pPulseContext, const ma_pa_sink_info* pInfo, int endOfList, void* pUserData)
{
    ma_device* pDevice;

    if (endOfList > 0) {
        return;
    }

    pDevice = (ma_device*)pUserData;
    ma_assert(pDevice != NULL);

    ma_strncpy_s(pDevice->playback.name, sizeof(pDevice->playback.name), pInfo->description, (size_t)-1);

    (void)pPulseContext; /* Unused. */
}

void ma_device_source_name_callback(ma_pa_context* pPulseContext, const ma_pa_source_info* pInfo, int endOfList, void* pUserData)
{
    ma_device* pDevice;

    if (endOfList > 0) {
        return;
    }

    pDevice = (ma_device*)pUserData;
    ma_assert(pDevice != NULL);

    ma_strncpy_s(pDevice->capture.name, sizeof(pDevice->capture.name), pInfo->description, (size_t)-1);

    (void)pPulseContext; /* Unused. */
}

void ma_device_uninit__pulse(ma_device* pDevice)
{
    ma_context* pContext;

    ma_assert(pDevice != NULL);

    pContext = pDevice->pContext;
    ma_assert(pContext != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_pa_stream_disconnect_proc)pContext->pulse.pa_stream_disconnect)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
        ((ma_pa_stream_unref_proc)pContext->pulse.pa_stream_unref)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
    }
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ((ma_pa_stream_disconnect_proc)pContext->pulse.pa_stream_disconnect)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
        ((ma_pa_stream_unref_proc)pContext->pulse.pa_stream_unref)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
    }

    ((ma_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)((ma_pa_context*)pDevice->pulse.pPulseContext);
    ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)((ma_pa_context*)pDevice->pulse.pPulseContext);
    ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)((ma_pa_mainloop*)pDevice->pulse.pMainLoop);
}

ma_pa_buffer_attr ma_device__pa_buffer_attr_new(ma_uint32 bufferSizeInFrames, ma_uint32 periods, const ma_pa_sample_spec* ss)
{
    ma_pa_buffer_attr attr;
    attr.maxlength = bufferSizeInFrames * ma_get_bytes_per_sample(ma_format_from_pulse(ss->format)) * ss->channels;
    attr.tlength   = attr.maxlength / periods;
    attr.prebuf    = (ma_uint32)-1;
    attr.minreq    = attr.maxlength / periods;
    attr.fragsize  = attr.maxlength / periods;

    return attr;
}

ma_pa_stream* ma_device__pa_stream_new__pulse(ma_device* pDevice, const char* pStreamName, const ma_pa_sample_spec* ss, const ma_pa_channel_map* cmap)
{
    static int g_StreamCounter = 0;
    char actualStreamName[256];

    if (pStreamName != NULL) {
        ma_strncpy_s(actualStreamName, sizeof(actualStreamName), pStreamName, (size_t)-1);
    } else {
        ma_strcpy_s(actualStreamName, sizeof(actualStreamName), "miniaudio:");
        ma_itoa_s(g_StreamCounter, actualStreamName + 8, sizeof(actualStreamName)-8, 10);  /* 8 = strlen("miniaudio:") */
    }
    g_StreamCounter += 1;

    return ((ma_pa_stream_new_proc)pDevice->pContext->pulse.pa_stream_new)((ma_pa_context*)pDevice->pulse.pPulseContext, actualStreamName, ss, cmap);
}

ma_result ma_device_init__pulse(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result = MA_SUCCESS;
    int error = 0;
    const char* devPlayback = NULL;
    const char* devCapture  = NULL;
    ma_uint32 bufferSizeInMilliseconds;
    ma_pa_sink_info sinkInfo;
    ma_pa_source_info sourceInfo;
    ma_pa_operation* pOP = NULL;
    ma_pa_sample_spec ss;
    ma_pa_channel_map cmap;
    ma_pa_buffer_attr attr;
    const ma_pa_sample_spec* pActualSS   = NULL;
    const ma_pa_channel_map* pActualCMap = NULL;
    const ma_pa_buffer_attr* pActualAttr = NULL;
    ma_uint32 iChannel;
    ma_pa_stream_flags_t streamFlags;

    ma_assert(pDevice != NULL);
    ma_zero_object(&pDevice->pulse);

    /* No exclusive mode with the PulseAudio backend. */
    if (((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.shareMode == ma_share_mode_exclusive) ||
        ((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.shareMode  == ma_share_mode_exclusive)) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    if ((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.pDeviceID != NULL) {
        devPlayback = pConfig->playback.pDeviceID->pulse;
    }
    if ((pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.pDeviceID != NULL) {
        devCapture = pConfig->capture.pDeviceID->pulse;
    }

    bufferSizeInMilliseconds = pConfig->bufferSizeInMilliseconds;
    if (bufferSizeInMilliseconds == 0) {
        bufferSizeInMilliseconds = ma_calculate_buffer_size_in_milliseconds_from_frames(pConfig->bufferSizeInFrames, pConfig->sampleRate);
    }

    pDevice->pulse.pMainLoop = ((ma_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
    if (pDevice->pulse.pMainLoop == NULL) {
        result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create main loop for device.", MA_FAILED_TO_INIT_BACKEND);
        goto on_error0;
    }

    pDevice->pulse.pAPI = ((ma_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)((ma_pa_mainloop*)pDevice->pulse.pMainLoop);
    if (pDevice->pulse.pAPI == NULL) {
        result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to retrieve PulseAudio main loop.", MA_FAILED_TO_INIT_BACKEND);
        goto on_error1;
    }

    pDevice->pulse.pPulseContext = ((ma_pa_context_new_proc)pContext->pulse.pa_context_new)((ma_pa_mainloop_api*)pDevice->pulse.pAPI, pContext->pulse.pApplicationName);
    if (pDevice->pulse.pPulseContext == NULL) {
        result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create PulseAudio context for device.", MA_FAILED_TO_INIT_BACKEND);
        goto on_error1;
    }

    error = ((ma_pa_context_connect_proc)pContext->pulse.pa_context_connect)((ma_pa_context*)pDevice->pulse.pPulseContext, pContext->pulse.pServerName, (pContext->pulse.tryAutoSpawn) ? 0 : MA_PA_CONTEXT_NOAUTOSPAWN, NULL);
    if (error != MA_PA_OK) {
        result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to connect PulseAudio context.", ma_result_from_pulse(error));
        goto on_error2;
    }


    pDevice->pulse.pulseContextState = MA_PA_CONTEXT_UNCONNECTED;
    ((ma_pa_context_set_state_callback_proc)pContext->pulse.pa_context_set_state_callback)((ma_pa_context*)pDevice->pulse.pPulseContext, ma_pulse_device_state_callback, pDevice);

    /* Wait for PulseAudio to get itself ready before returning. */
    for (;;) {
        if (pDevice->pulse.pulseContextState == MA_PA_CONTEXT_READY) {
            break;
        }

        /* An error may have occurred. */
        if (pDevice->pulse.pulseContextState == MA_PA_CONTEXT_FAILED || pDevice->pulse.pulseContextState == MA_PA_CONTEXT_TERMINATED) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] An error occurred while connecting the PulseAudio context.", MA_ERROR);
            goto on_error3;
        }

        error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)((ma_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
        if (error < 0) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] The PulseAudio main loop returned an error while connecting the PulseAudio context.", ma_result_from_pulse(error));
            goto on_error3;
        }
    }

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        pOP = ((ma_pa_context_get_source_info_by_name_proc)pContext->pulse.pa_context_get_source_info_by_name)((ma_pa_context*)pDevice->pulse.pPulseContext, devCapture, ma_device_source_info_callback, &sourceInfo);
        if (pOP != NULL) {
            ma_device__wait_for_operation__pulse(pDevice, pOP);
            ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        } else {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to retrieve source info for capture device.", ma_result_from_pulse(error));
            goto on_error3;
        }

        ss = sourceInfo.sample_spec;
        cmap = sourceInfo.channel_map;

        pDevice->capture.internalBufferSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(bufferSizeInMilliseconds, ss.rate);
        pDevice->capture.internalPeriods            = pConfig->periods;

        attr = ma_device__pa_buffer_attr_new(pDevice->capture.internalBufferSizeInFrames, pConfig->periods, &ss);
    #ifdef MA_DEBUG_OUTPUT
        printf("[PulseAudio] Capture attr: maxlength=%d, tlength=%d, prebuf=%d, minreq=%d, fragsize=%d; internalBufferSizeInFrames=%d\n", attr.maxlength, attr.tlength, attr.prebuf, attr.minreq, attr.fragsize, pDevice->capture.internalBufferSizeInFrames);
    #endif

        pDevice->pulse.pStreamCapture = ma_device__pa_stream_new__pulse(pDevice, pConfig->pulse.pStreamNameCapture, &ss, &cmap);
        if (pDevice->pulse.pStreamCapture == NULL) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create PulseAudio capture stream.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
            goto on_error3;
        }

        streamFlags = MA_PA_STREAM_START_CORKED | MA_PA_STREAM_FIX_FORMAT | MA_PA_STREAM_FIX_RATE | MA_PA_STREAM_FIX_CHANNELS;
        if (devCapture != NULL) {
            streamFlags |= MA_PA_STREAM_DONT_MOVE;
        }

        error = ((ma_pa_stream_connect_record_proc)pContext->pulse.pa_stream_connect_record)((ma_pa_stream*)pDevice->pulse.pStreamCapture, devCapture, &attr, streamFlags);
        if (error != MA_PA_OK) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to connect PulseAudio capture stream.", ma_result_from_pulse(error));
            goto on_error4;
        }

        while (((ma_pa_stream_get_state_proc)pContext->pulse.pa_stream_get_state)((ma_pa_stream*)pDevice->pulse.pStreamCapture) != MA_PA_STREAM_READY) {
            error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)((ma_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
            if (error < 0) {
                result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] The PulseAudio main loop returned an error while connecting the PulseAudio capture stream.", ma_result_from_pulse(error));
                goto on_error5;
            }
        }

        /* Internal format. */
        pActualSS = ((ma_pa_stream_get_sample_spec_proc)pContext->pulse.pa_stream_get_sample_spec)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
        if (pActualSS != NULL) {
            /* If anything has changed between the requested and the actual sample spec, we need to update the buffer. */
            if (ss.format != pActualSS->format || ss.channels != pActualSS->channels || ss.rate != pActualSS->rate) {
                attr = ma_device__pa_buffer_attr_new(pDevice->capture.internalBufferSizeInFrames, pConfig->periods, pActualSS);

                pOP = ((ma_pa_stream_set_buffer_attr_proc)pContext->pulse.pa_stream_set_buffer_attr)((ma_pa_stream*)pDevice->pulse.pStreamCapture, &attr, NULL, NULL);
                if (pOP != NULL) {
                    ma_device__wait_for_operation__pulse(pDevice, pOP);
                    ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
                }
            }

            ss = *pActualSS;
        }

        pDevice->capture.internalFormat     = ma_format_from_pulse(ss.format);
        pDevice->capture.internalChannels   = ss.channels;
        pDevice->capture.internalSampleRate = ss.rate;

        /* Internal channel map. */
        pActualCMap = ((ma_pa_stream_get_channel_map_proc)pContext->pulse.pa_stream_get_channel_map)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
        if (pActualCMap != NULL) {
            cmap = *pActualCMap;
        }
        for (iChannel = 0; iChannel < pDevice->capture.internalChannels; ++iChannel) {
            pDevice->capture.internalChannelMap[iChannel] = ma_channel_position_from_pulse(cmap.map[iChannel]);
        }

        /* Buffer. */
        pActualAttr = ((ma_pa_stream_get_buffer_attr_proc)pContext->pulse.pa_stream_get_buffer_attr)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
        if (pActualAttr != NULL) {
            attr = *pActualAttr;
        }
        pDevice->capture.internalBufferSizeInFrames = attr.maxlength / (ma_get_bytes_per_sample(pDevice->capture.internalFormat) * pDevice->capture.internalChannels);
        pDevice->capture.internalPeriods            = attr.maxlength / attr.fragsize;
    #ifdef MA_DEBUG_OUTPUT
        printf("[PulseAudio] Capture actual attr: maxlength=%d, tlength=%d, prebuf=%d, minreq=%d, fragsize=%d; internalBufferSizeInFrames=%d\n", attr.maxlength, attr.tlength, attr.prebuf, attr.minreq, attr.fragsize, pDevice->capture.internalBufferSizeInFrames);
    #endif

        /* Name. */
        devCapture = ((ma_pa_stream_get_device_name_proc)pContext->pulse.pa_stream_get_device_name)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
        if (devCapture != NULL) {
            ma_pa_operation* pOP = ((ma_pa_context_get_source_info_by_name_proc)pContext->pulse.pa_context_get_source_info_by_name)((ma_pa_context*)pDevice->pulse.pPulseContext, devCapture, ma_device_source_name_callback, pDevice);
            if (pOP != NULL) {
                ma_device__wait_for_operation__pulse(pDevice, pOP);
                ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
            }
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        pOP = ((ma_pa_context_get_sink_info_by_name_proc)pContext->pulse.pa_context_get_sink_info_by_name)((ma_pa_context*)pDevice->pulse.pPulseContext, devPlayback, ma_device_sink_info_callback, &sinkInfo);
        if (pOP != NULL) {
            ma_device__wait_for_operation__pulse(pDevice, pOP);
            ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
        } else {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to retrieve sink info for playback device.", ma_result_from_pulse(error));
            goto on_error3;
        }

        ss = sinkInfo.sample_spec;
        cmap = sinkInfo.channel_map;

        pDevice->playback.internalBufferSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(bufferSizeInMilliseconds, ss.rate);
        pDevice->playback.internalPeriods            = pConfig->periods;

        attr = ma_device__pa_buffer_attr_new(pDevice->playback.internalBufferSizeInFrames, pConfig->periods, &ss);
    #ifdef MA_DEBUG_OUTPUT
        printf("[PulseAudio] Playback attr: maxlength=%d, tlength=%d, prebuf=%d, minreq=%d, fragsize=%d; internalBufferSizeInFrames=%d\n", attr.maxlength, attr.tlength, attr.prebuf, attr.minreq, attr.fragsize, pDevice->playback.internalBufferSizeInFrames);
    #endif

        pDevice->pulse.pStreamPlayback = ma_device__pa_stream_new__pulse(pDevice, pConfig->pulse.pStreamNamePlayback, &ss, &cmap);
        if (pDevice->pulse.pStreamPlayback == NULL) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to create PulseAudio playback stream.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
            goto on_error3;
        }

        streamFlags = MA_PA_STREAM_START_CORKED | MA_PA_STREAM_FIX_FORMAT | MA_PA_STREAM_FIX_RATE | MA_PA_STREAM_FIX_CHANNELS;
        if (devPlayback != NULL) {
            streamFlags |= MA_PA_STREAM_DONT_MOVE;
        }

        error = ((ma_pa_stream_connect_playback_proc)pContext->pulse.pa_stream_connect_playback)((ma_pa_stream*)pDevice->pulse.pStreamPlayback, devPlayback, &attr, streamFlags, NULL, NULL);
        if (error != MA_PA_OK) {
            result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to connect PulseAudio playback stream.", ma_result_from_pulse(error));
            goto on_error6;
        }

        while (((ma_pa_stream_get_state_proc)pContext->pulse.pa_stream_get_state)((ma_pa_stream*)pDevice->pulse.pStreamPlayback) != MA_PA_STREAM_READY) {
            error = ((ma_pa_mainloop_iterate_proc)pContext->pulse.pa_mainloop_iterate)((ma_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
            if (error < 0) {
                result = ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] The PulseAudio main loop returned an error while connecting the PulseAudio playback stream.", ma_result_from_pulse(error));
                goto on_error7;
            }
        }

        /* Internal format. */
        pActualSS = ((ma_pa_stream_get_sample_spec_proc)pContext->pulse.pa_stream_get_sample_spec)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
        if (pActualSS != NULL) {
            /* If anything has changed between the requested and the actual sample spec, we need to update the buffer. */
            if (ss.format != pActualSS->format || ss.channels != pActualSS->channels || ss.rate != pActualSS->rate) {
                attr = ma_device__pa_buffer_attr_new(pDevice->playback.internalBufferSizeInFrames, pConfig->periods, pActualSS);

                pOP = ((ma_pa_stream_set_buffer_attr_proc)pContext->pulse.pa_stream_set_buffer_attr)((ma_pa_stream*)pDevice->pulse.pStreamPlayback, &attr, NULL, NULL);
                if (pOP != NULL) {
                    ma_device__wait_for_operation__pulse(pDevice, pOP);
                    ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
                }
            }

            ss = *pActualSS;
        }

        pDevice->playback.internalFormat     = ma_format_from_pulse(ss.format);
        pDevice->playback.internalChannels   = ss.channels;
        pDevice->playback.internalSampleRate = ss.rate;

        /* Internal channel map. */
        pActualCMap = ((ma_pa_stream_get_channel_map_proc)pContext->pulse.pa_stream_get_channel_map)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
        if (pActualCMap != NULL) {
            cmap = *pActualCMap;
        }
        for (iChannel = 0; iChannel < pDevice->playback.internalChannels; ++iChannel) {
            pDevice->playback.internalChannelMap[iChannel] = ma_channel_position_from_pulse(cmap.map[iChannel]);
        }

        /* Buffer. */
        pActualAttr = ((ma_pa_stream_get_buffer_attr_proc)pContext->pulse.pa_stream_get_buffer_attr)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
        if (pActualAttr != NULL) {
            attr = *pActualAttr;
        }
        pDevice->playback.internalBufferSizeInFrames = attr.maxlength / (ma_get_bytes_per_sample(pDevice->playback.internalFormat) * pDevice->playback.internalChannels);
        pDevice->playback.internalPeriods            = /*pConfig->periods;*/attr.maxlength / attr.tlength;
    #ifdef MA_DEBUG_OUTPUT
        printf("[PulseAudio] Playback actual attr: maxlength=%d, tlength=%d, prebuf=%d, minreq=%d, fragsize=%d; internalBufferSizeInFrames=%d\n", attr.maxlength, attr.tlength, attr.prebuf, attr.minreq, attr.fragsize, pDevice->playback.internalBufferSizeInFrames);
    #endif

        /* Name. */
        devPlayback = ((ma_pa_stream_get_device_name_proc)pContext->pulse.pa_stream_get_device_name)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
        if (devPlayback != NULL) {
            ma_pa_operation* pOP = ((ma_pa_context_get_sink_info_by_name_proc)pContext->pulse.pa_context_get_sink_info_by_name)((ma_pa_context*)pDevice->pulse.pPulseContext, devPlayback, ma_device_sink_name_callback, pDevice);
            if (pOP != NULL) {
                ma_device__wait_for_operation__pulse(pDevice, pOP);
                ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);
            }
        }
    }

    return MA_SUCCESS;


on_error7:
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ((ma_pa_stream_disconnect_proc)pContext->pulse.pa_stream_disconnect)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
    }
on_error6:
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ((ma_pa_stream_unref_proc)pContext->pulse.pa_stream_unref)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
    }
on_error5:
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ((ma_pa_stream_disconnect_proc)pContext->pulse.pa_stream_disconnect)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
    }
on_error4:
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ((ma_pa_stream_unref_proc)pContext->pulse.pa_stream_unref)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
    }
on_error3: ((ma_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)((ma_pa_context*)pDevice->pulse.pPulseContext);
on_error2: ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)((ma_pa_context*)pDevice->pulse.pPulseContext);
on_error1: ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)((ma_pa_mainloop*)pDevice->pulse.pMainLoop);
on_error0:
    return result;
}


void ma_pulse_operation_complete_callback(ma_pa_stream* pStream, int success, void* pUserData)
{
    ma_bool32* pIsSuccessful = (ma_bool32*)pUserData;
    ma_assert(pIsSuccessful != NULL);

    *pIsSuccessful = (ma_bool32)success;

    (void)pStream; /* Unused. */
}

ma_result ma_device__cork_stream__pulse(ma_device* pDevice, ma_device_type deviceType, int cork)
{
    ma_context* pContext = pDevice->pContext;
    ma_bool32 wasSuccessful;
    ma_pa_stream* pStream;
    ma_pa_operation* pOP;
    ma_result result;

    /* This should not be called with a duplex device type. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    wasSuccessful = MA_FALSE;

    pStream = (ma_pa_stream*)((deviceType == ma_device_type_capture) ? pDevice->pulse.pStreamCapture : pDevice->pulse.pStreamPlayback);
    ma_assert(pStream != NULL);

    pOP = ((ma_pa_stream_cork_proc)pContext->pulse.pa_stream_cork)(pStream, cork, ma_pulse_operation_complete_callback, &wasSuccessful);
    if (pOP == NULL) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to cork PulseAudio stream.", (cork == 0) ? MA_FAILED_TO_START_BACKEND_DEVICE : MA_FAILED_TO_STOP_BACKEND_DEVICE);
    }

    result = ma_device__wait_for_operation__pulse(pDevice, pOP);
    ((ma_pa_operation_unref_proc)pContext->pulse.pa_operation_unref)(pOP);

    if (result != MA_SUCCESS) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] An error occurred while waiting for the PulseAudio stream to cork.", result);
    }

    if (!wasSuccessful) {
        if (cork) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to stop PulseAudio stream.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
        } else {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to start PulseAudio stream.", MA_FAILED_TO_START_BACKEND_DEVICE);
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_stop__pulse(ma_device* pDevice)
{
    ma_result result;
    ma_bool32 wasSuccessful;
    ma_pa_operation* pOP;

    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        result = ma_device__cork_stream__pulse(pDevice, ma_device_type_capture, 1);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        /* The stream needs to be drained if it's a playback device. */
        pOP = ((ma_pa_stream_drain_proc)pDevice->pContext->pulse.pa_stream_drain)((ma_pa_stream*)pDevice->pulse.pStreamPlayback, ma_pulse_operation_complete_callback, &wasSuccessful);
        if (pOP != NULL) {
            ma_device__wait_for_operation__pulse(pDevice, pOP);
            ((ma_pa_operation_unref_proc)pDevice->pContext->pulse.pa_operation_unref)(pOP);
        }

        result = ma_device__cork_stream__pulse(pDevice, ma_device_type_playback, 1);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_write__pulse(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount)
{
    ma_uint32 totalFramesWritten;

    ma_assert(pDevice != NULL);
    ma_assert(pPCMFrames != NULL);
    ma_assert(frameCount > 0);

    /* The stream needs to be uncorked first. */
    if (((ma_pa_stream_is_corked_proc)pDevice->pContext->pulse.pa_stream_is_corked)((ma_pa_stream*)pDevice->pulse.pStreamPlayback)) {
        ma_result result = ma_device__cork_stream__pulse(pDevice, ma_device_type_playback, 0);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    totalFramesWritten = 0;
    while (totalFramesWritten < frameCount) {
        /* Place the data into the mapped buffer if we have one. */
        if (pDevice->pulse.pMappedBufferPlayback != NULL && pDevice->pulse.mappedBufferFramesRemainingPlayback > 0) {
            ma_uint32 bpf = ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
            ma_uint32 mappedBufferFramesConsumed = pDevice->pulse.mappedBufferFramesCapacityPlayback - pDevice->pulse.mappedBufferFramesRemainingPlayback;

            void* pDst = (ma_uint8*)pDevice->pulse.pMappedBufferPlayback + (mappedBufferFramesConsumed * bpf);
            const void* pSrc = (const ma_uint8*)pPCMFrames + (totalFramesWritten * bpf);
            ma_uint32  framesToCopy = ma_min(pDevice->pulse.mappedBufferFramesRemainingPlayback, (frameCount - totalFramesWritten));
            ma_copy_memory(pDst, pSrc, framesToCopy * bpf);

            pDevice->pulse.mappedBufferFramesRemainingPlayback -= framesToCopy;
            totalFramesWritten += framesToCopy;
        }

        /*
        Getting here means we've run out of data in the currently mapped chunk. We need to write this to the device and then try
        mapping another chunk. If this fails we need to wait for space to become available.
        */
        if (pDevice->pulse.mappedBufferFramesCapacityPlayback > 0 && pDevice->pulse.mappedBufferFramesRemainingPlayback == 0) {
            size_t nbytes = pDevice->pulse.mappedBufferFramesCapacityPlayback * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);

            int error = ((ma_pa_stream_write_proc)pDevice->pContext->pulse.pa_stream_write)((ma_pa_stream*)pDevice->pulse.pStreamPlayback, pDevice->pulse.pMappedBufferPlayback, nbytes, NULL, 0, MA_PA_SEEK_RELATIVE);
            if (error < 0) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to write data to the PulseAudio stream.", ma_result_from_pulse(error));
            }

            pDevice->pulse.pMappedBufferPlayback = NULL;
            pDevice->pulse.mappedBufferFramesRemainingPlayback = 0;
            pDevice->pulse.mappedBufferFramesCapacityPlayback = 0;
        }

        ma_assert(totalFramesWritten <= frameCount);
        if (totalFramesWritten == frameCount) {
            break;
        }

        /* Getting here means we need to map a new buffer. If we don't have enough space we need to wait for more. */
        for (;;) {
            size_t writableSizeInBytes;

            /* If the device has been corked, don't try to continue. */
            if (((ma_pa_stream_is_corked_proc)pDevice->pContext->pulse.pa_stream_is_corked)((ma_pa_stream*)pDevice->pulse.pStreamPlayback)) {
                break;
            }

            writableSizeInBytes = ((ma_pa_stream_writable_size_proc)pDevice->pContext->pulse.pa_stream_writable_size)((ma_pa_stream*)pDevice->pulse.pStreamPlayback);
            if (writableSizeInBytes != (size_t)-1) {
                /*size_t periodSizeInBytes = (pDevice->playback.internalBufferSizeInFrames / pDevice->playback.internalPeriods) * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);*/
                if (writableSizeInBytes > 0) {
                    /* Data is avaialable. */
                    size_t bytesToMap = writableSizeInBytes;
                    int error = ((ma_pa_stream_begin_write_proc)pDevice->pContext->pulse.pa_stream_begin_write)((ma_pa_stream*)pDevice->pulse.pStreamPlayback, &pDevice->pulse.pMappedBufferPlayback, &bytesToMap);
                    if (error < 0) {
                        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to map write buffer.", ma_result_from_pulse(error));
                    }

                    pDevice->pulse.mappedBufferFramesCapacityPlayback  = bytesToMap / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                    pDevice->pulse.mappedBufferFramesRemainingPlayback = pDevice->pulse.mappedBufferFramesCapacityPlayback;

                    break;
                } else {
                    /* No data available. Need to wait for more. */
                    int error = ((ma_pa_mainloop_iterate_proc)pDevice->pContext->pulse.pa_mainloop_iterate)((ma_pa_mainloop*)pDevice->pulse.pMainLoop, 1, NULL);
                    if (error < 0) {
                        return ma_result_from_pulse(error);
                    }

                    continue;
                }
            } else {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to query the stream's writable size.", MA_ERROR);
            }
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_read__pulse(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount)
{
    ma_uint32 totalFramesRead;

    ma_assert(pDevice != NULL);
    ma_assert(pPCMFrames != NULL);
    ma_assert(frameCount > 0);

    /* The stream needs to be uncorked first. */
    if (((ma_pa_stream_is_corked_proc)pDevice->pContext->pulse.pa_stream_is_corked)((ma_pa_stream*)pDevice->pulse.pStreamCapture)) {
        ma_result result = ma_device__cork_stream__pulse(pDevice, ma_device_type_capture, 0);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    totalFramesRead = 0;
    while (totalFramesRead < frameCount) {
        if (ma_device__get_state(pDevice) != MA_STATE_STARTED) {
            break;
        }

        /* If a buffer is mapped we need to write to that first. Once it's consumed we reset the event and unmap it. */
        if (pDevice->pulse.pMappedBufferCapture != NULL && pDevice->pulse.mappedBufferFramesRemainingCapture > 0) {
            ma_uint32 bpf = ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
            ma_uint32 mappedBufferFramesConsumed = pDevice->pulse.mappedBufferFramesCapacityCapture - pDevice->pulse.mappedBufferFramesRemainingCapture;

            ma_uint32  framesToCopy = ma_min(pDevice->pulse.mappedBufferFramesRemainingCapture, (frameCount - totalFramesRead));
            void* pDst = (ma_uint8*)pPCMFrames + (totalFramesRead * bpf);

            /*
            This little bit of logic here is specifically for PulseAudio and it's hole management. The buffer pointer will be set to NULL
            when the current fragment is a hole. For a hole we just output silence.
            */
            if (pDevice->pulse.pMappedBufferCapture != NULL) {
                const void* pSrc = (const ma_uint8*)pDevice->pulse.pMappedBufferCapture + (mappedBufferFramesConsumed * bpf);
                ma_copy_memory(pDst, pSrc, framesToCopy * bpf);
            } else {
                ma_zero_memory(pDst, framesToCopy * bpf);
            }

            pDevice->pulse.mappedBufferFramesRemainingCapture -= framesToCopy;
            totalFramesRead += framesToCopy;
        }

        /*
        Getting here means we've run out of data in the currently mapped chunk. We need to drop this from the device and then try
        mapping another chunk. If this fails we need to wait for data to become available.
        */
        if (pDevice->pulse.mappedBufferFramesCapacityCapture > 0 && pDevice->pulse.mappedBufferFramesRemainingCapture == 0) {
            int error = ((ma_pa_stream_drop_proc)pDevice->pContext->pulse.pa_stream_drop)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
            if (error != 0) {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to drop fragment.", ma_result_from_pulse(error));
            }

            pDevice->pulse.pMappedBufferCapture = NULL;
            pDevice->pulse.mappedBufferFramesRemainingCapture = 0;
            pDevice->pulse.mappedBufferFramesCapacityCapture = 0;
        }

        ma_assert(totalFramesRead <= frameCount);
        if (totalFramesRead == frameCount) {
            break;
        }

        /* Getting here means we need to map a new buffer. If we don't have enough data we wait for more. */
        for (;;) {
            size_t readableSizeInBytes;

            if (ma_device__get_state(pDevice) != MA_STATE_STARTED) {
                break;
            }

            /* If the device has been corked, don't try to continue. */
            if (((ma_pa_stream_is_corked_proc)pDevice->pContext->pulse.pa_stream_is_corked)((ma_pa_stream*)pDevice->pulse.pStreamCapture)) {
                break;
            }

            readableSizeInBytes = ((ma_pa_stream_readable_size_proc)pDevice->pContext->pulse.pa_stream_readable_size)((ma_pa_stream*)pDevice->pulse.pStreamCapture);
            if (readableSizeInBytes != (size_t)-1) {
                /*size_t periodSizeInBytes = (pDevice->capture.internalBufferSizeInFrames / pDevice->capture.internalPeriods) * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);*/
                if (readableSizeInBytes > 0) {
                    /* Data is avaialable. */
                    size_t bytesMapped = (size_t)-1;
                    int error = ((ma_pa_stream_peek_proc)pDevice->pContext->pulse.pa_stream_peek)((ma_pa_stream*)pDevice->pulse.pStreamCapture, &pDevice->pulse.pMappedBufferCapture, &bytesMapped);
                    if (error < 0) {
                        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to peek capture buffer.", ma_result_from_pulse(error));
                    }

                    if (pDevice->pulse.pMappedBufferCapture == NULL && bytesMapped == 0) {
                        /* Nothing available. This shouldn't happen because we checked earlier with pa_stream_readable_size(). I'm going to throw an error in this case. */
                        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Nothing available after peeking capture buffer.", MA_ERROR);
                    }

                    pDevice->pulse.mappedBufferFramesCapacityCapture  = bytesMapped / ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
                    pDevice->pulse.mappedBufferFramesRemainingCapture = pDevice->pulse.mappedBufferFramesCapacityCapture;

                    break;
                } else {
                    /* No data available. Need to wait for more. */

                    /*
                    I have had reports of a deadlock in this part of the code. I have reproduced this when using the "Built-in Audio Analogue Stereo" device without
                    an actual microphone connected. I'm experimenting here by not blocking in pa_mainloop_iterate() and instead sleep for a bit when there are no
                    dispatches.
                    */
                    int error = ((ma_pa_mainloop_iterate_proc)pDevice->pContext->pulse.pa_mainloop_iterate)((ma_pa_mainloop*)pDevice->pulse.pMainLoop, 0, NULL);
                    if (error < 0) {
                        return ma_result_from_pulse(error);
                    }

                    /* Sleep for a bit if nothing was dispatched. */
                    if (error == 0) {
                        ma_sleep(1);
                    }

                    continue;
                }
            } else {
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[PulseAudio] Failed to query the stream's readable size.", MA_ERROR);
            }
        }
    }

    return MA_SUCCESS;
}


ma_result ma_context_uninit__pulse(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_pulseaudio);

    ma_free(pContext->pulse.pServerName);
    pContext->pulse.pServerName = NULL;

    ma_free(pContext->pulse.pApplicationName);
    pContext->pulse.pApplicationName = NULL;

#ifndef MA_NO_RUNTIME_LINKING
    ma_dlclose(pContext->pulse.pulseSO);
#endif

    return MA_SUCCESS;
}

ma_result ma_context_init__pulse(const ma_context_config* pConfig, ma_context* pContext)
{
#ifndef MA_NO_RUNTIME_LINKING
    const char* libpulseNames[] = {
        "libpulse.so",
        "libpulse.so.0"
    };
    size_t i;

    for (i = 0; i < ma_countof(libpulseNames); ++i) {
        pContext->pulse.pulseSO = ma_dlopen(libpulseNames[i]);
        if (pContext->pulse.pulseSO != NULL) {
            break;
        }
    }

    if (pContext->pulse.pulseSO == NULL) {
        return MA_NO_BACKEND;
    }

    pContext->pulse.pa_mainloop_new                    = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_mainloop_new");
    pContext->pulse.pa_mainloop_free                   = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_mainloop_free");
    pContext->pulse.pa_mainloop_get_api                = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_mainloop_get_api");
    pContext->pulse.pa_mainloop_iterate                = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_mainloop_iterate");
    pContext->pulse.pa_mainloop_wakeup                 = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_mainloop_wakeup");
    pContext->pulse.pa_context_new                     = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_context_new");
    pContext->pulse.pa_context_unref                   = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_context_unref");
    pContext->pulse.pa_context_connect                 = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_context_connect");
    pContext->pulse.pa_context_disconnect              = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_context_disconnect");
    pContext->pulse.pa_context_set_state_callback      = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_context_set_state_callback");
    pContext->pulse.pa_context_get_state               = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_context_get_state");
    pContext->pulse.pa_context_get_sink_info_list      = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_context_get_sink_info_list");
    pContext->pulse.pa_context_get_source_info_list    = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_context_get_source_info_list");
    pContext->pulse.pa_context_get_sink_info_by_name   = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_context_get_sink_info_by_name");
    pContext->pulse.pa_context_get_source_info_by_name = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_context_get_source_info_by_name");
    pContext->pulse.pa_operation_unref                 = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_operation_unref");
    pContext->pulse.pa_operation_get_state             = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_operation_get_state");
    pContext->pulse.pa_channel_map_init_extend         = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_channel_map_init_extend");
    pContext->pulse.pa_channel_map_valid               = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_channel_map_valid");
    pContext->pulse.pa_channel_map_compatible          = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_channel_map_compatible");
    pContext->pulse.pa_stream_new                      = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_new");
    pContext->pulse.pa_stream_unref                    = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_unref");
    pContext->pulse.pa_stream_connect_playback         = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_connect_playback");
    pContext->pulse.pa_stream_connect_record           = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_connect_record");
    pContext->pulse.pa_stream_disconnect               = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_disconnect");
    pContext->pulse.pa_stream_get_state                = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_get_state");
    pContext->pulse.pa_stream_get_sample_spec          = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_get_sample_spec");
    pContext->pulse.pa_stream_get_channel_map          = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_get_channel_map");
    pContext->pulse.pa_stream_get_buffer_attr          = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_get_buffer_attr");
    pContext->pulse.pa_stream_set_buffer_attr          = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_set_buffer_attr");
    pContext->pulse.pa_stream_get_device_name          = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_get_device_name");
    pContext->pulse.pa_stream_set_write_callback       = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_set_write_callback");
    pContext->pulse.pa_stream_set_read_callback        = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_set_read_callback");
    pContext->pulse.pa_stream_flush                    = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_flush");
    pContext->pulse.pa_stream_drain                    = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_drain");
    pContext->pulse.pa_stream_is_corked                = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_is_corked");
    pContext->pulse.pa_stream_cork                     = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_cork");
    pContext->pulse.pa_stream_trigger                  = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_trigger");
    pContext->pulse.pa_stream_begin_write              = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_begin_write");
    pContext->pulse.pa_stream_write                    = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_write");
    pContext->pulse.pa_stream_peek                     = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_peek");
    pContext->pulse.pa_stream_drop                     = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_drop");
    pContext->pulse.pa_stream_writable_size            = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_writable_size");
    pContext->pulse.pa_stream_readable_size            = (ma_proc)ma_dlsym(pContext->pulse.pulseSO, "pa_stream_readable_size");
#else
    /* This strange assignment system is just for type safety. */
    ma_pa_mainloop_new_proc                    _pa_mainloop_new                   = pa_mainloop_new;
    ma_pa_mainloop_free_proc                   _pa_mainloop_free                  = pa_mainloop_free;
    ma_pa_mainloop_get_api_proc                _pa_mainloop_get_api               = pa_mainloop_get_api;
    ma_pa_mainloop_iterate_proc                _pa_mainloop_iterate               = pa_mainloop_iterate;
    ma_pa_mainloop_wakeup_proc                 _pa_mainloop_wakeup                = pa_mainloop_wakeup;
    ma_pa_context_new_proc                     _pa_context_new                    = pa_context_new;
    ma_pa_context_unref_proc                   _pa_context_unref                  = pa_context_unref;
    ma_pa_context_connect_proc                 _pa_context_connect                = pa_context_connect;
    ma_pa_context_disconnect_proc              _pa_context_disconnect             = pa_context_disconnect;
    ma_pa_context_set_state_callback_proc      _pa_context_set_state_callback     = pa_context_set_state_callback;
    ma_pa_context_get_state_proc               _pa_context_get_state              = pa_context_get_state;
    ma_pa_context_get_sink_info_list_proc      _pa_context_get_sink_info_list     = pa_context_get_sink_info_list;
    ma_pa_context_get_source_info_list_proc    _pa_context_get_source_info_list   = pa_context_get_source_info_list;
    ma_pa_context_get_sink_info_by_name_proc   _pa_context_get_sink_info_by_name  = pa_context_get_sink_info_by_name;
    ma_pa_context_get_source_info_by_name_proc _pa_context_get_source_info_by_name= pa_context_get_source_info_by_name;
    ma_pa_operation_unref_proc                 _pa_operation_unref                = pa_operation_unref;
    ma_pa_operation_get_state_proc             _pa_operation_get_state            = pa_operation_get_state;
    ma_pa_channel_map_init_extend_proc         _pa_channel_map_init_extend        = pa_channel_map_init_extend;
    ma_pa_channel_map_valid_proc               _pa_channel_map_valid              = pa_channel_map_valid;
    ma_pa_channel_map_compatible_proc          _pa_channel_map_compatible         = pa_channel_map_compatible;
    ma_pa_stream_new_proc                      _pa_stream_new                     = pa_stream_new;
    ma_pa_stream_unref_proc                    _pa_stream_unref                   = pa_stream_unref;
    ma_pa_stream_connect_playback_proc         _pa_stream_connect_playback        = pa_stream_connect_playback;
    ma_pa_stream_connect_record_proc           _pa_stream_connect_record          = pa_stream_connect_record;
    ma_pa_stream_disconnect_proc               _pa_stream_disconnect              = pa_stream_disconnect;
    ma_pa_stream_get_state_proc                _pa_stream_get_state               = pa_stream_get_state;
    ma_pa_stream_get_sample_spec_proc          _pa_stream_get_sample_spec         = pa_stream_get_sample_spec;
    ma_pa_stream_get_channel_map_proc          _pa_stream_get_channel_map         = pa_stream_get_channel_map;
    ma_pa_stream_get_buffer_attr_proc          _pa_stream_get_buffer_attr         = pa_stream_get_buffer_attr;
    ma_pa_stream_set_buffer_attr_proc          _pa_stream_set_buffer_attr         = pa_stream_set_buffer_attr;
    ma_pa_stream_get_device_name_proc          _pa_stream_get_device_name         = pa_stream_get_device_name;
    ma_pa_stream_set_write_callback_proc       _pa_stream_set_write_callback      = pa_stream_set_write_callback;
    ma_pa_stream_set_read_callback_proc        _pa_stream_set_read_callback       = pa_stream_set_read_callback;
    ma_pa_stream_flush_proc                    _pa_stream_flush                   = pa_stream_flush;
    ma_pa_stream_drain_proc                    _pa_stream_drain                   = pa_stream_drain;
    ma_pa_stream_is_corked_proc                _pa_stream_is_corked               = pa_stream_is_corked;
    ma_pa_stream_cork_proc                     _pa_stream_cork                    = pa_stream_cork;
    ma_pa_stream_trigger_proc                  _pa_stream_trigger                 = pa_stream_trigger;
    ma_pa_stream_begin_write_proc              _pa_stream_begin_write             = pa_stream_begin_write;
    ma_pa_stream_write_proc                    _pa_stream_write                   = pa_stream_write;
    ma_pa_stream_peek_proc                     _pa_stream_peek                    = pa_stream_peek;
    ma_pa_stream_drop_proc                     _pa_stream_drop                    = pa_stream_drop;
    ma_pa_stream_writable_size_proc            _pa_stream_writable_size           = pa_stream_writable_size;
    ma_pa_stream_readable_size_proc            _pa_stream_readable_size           = pa_stream_readable_size;

    pContext->pulse.pa_mainloop_new                    = (ma_proc)_pa_mainloop_new;
    pContext->pulse.pa_mainloop_free                   = (ma_proc)_pa_mainloop_free;
    pContext->pulse.pa_mainloop_get_api                = (ma_proc)_pa_mainloop_get_api;
    pContext->pulse.pa_mainloop_iterate                = (ma_proc)_pa_mainloop_iterate;
    pContext->pulse.pa_mainloop_wakeup                 = (ma_proc)_pa_mainloop_wakeup;
    pContext->pulse.pa_context_new                     = (ma_proc)_pa_context_new;
    pContext->pulse.pa_context_unref                   = (ma_proc)_pa_context_unref;
    pContext->pulse.pa_context_connect                 = (ma_proc)_pa_context_connect;
    pContext->pulse.pa_context_disconnect              = (ma_proc)_pa_context_disconnect;
    pContext->pulse.pa_context_set_state_callback      = (ma_proc)_pa_context_set_state_callback;
    pContext->pulse.pa_context_get_state               = (ma_proc)_pa_context_get_state;
    pContext->pulse.pa_context_get_sink_info_list      = (ma_proc)_pa_context_get_sink_info_list;
    pContext->pulse.pa_context_get_source_info_list    = (ma_proc)_pa_context_get_source_info_list;
    pContext->pulse.pa_context_get_sink_info_by_name   = (ma_proc)_pa_context_get_sink_info_by_name;
    pContext->pulse.pa_context_get_source_info_by_name = (ma_proc)_pa_context_get_source_info_by_name;
    pContext->pulse.pa_operation_unref                 = (ma_proc)_pa_operation_unref;
    pContext->pulse.pa_operation_get_state             = (ma_proc)_pa_operation_get_state;
    pContext->pulse.pa_channel_map_init_extend         = (ma_proc)_pa_channel_map_init_extend;
    pContext->pulse.pa_channel_map_valid               = (ma_proc)_pa_channel_map_valid;
    pContext->pulse.pa_channel_map_compatible          = (ma_proc)_pa_channel_map_compatible;
    pContext->pulse.pa_stream_new                      = (ma_proc)_pa_stream_new;
    pContext->pulse.pa_stream_unref                    = (ma_proc)_pa_stream_unref;
    pContext->pulse.pa_stream_connect_playback         = (ma_proc)_pa_stream_connect_playback;
    pContext->pulse.pa_stream_connect_record           = (ma_proc)_pa_stream_connect_record;
    pContext->pulse.pa_stream_disconnect               = (ma_proc)_pa_stream_disconnect;
    pContext->pulse.pa_stream_get_state                = (ma_proc)_pa_stream_get_state;
    pContext->pulse.pa_stream_get_sample_spec          = (ma_proc)_pa_stream_get_sample_spec;
    pContext->pulse.pa_stream_get_channel_map          = (ma_proc)_pa_stream_get_channel_map;
    pContext->pulse.pa_stream_get_buffer_attr          = (ma_proc)_pa_stream_get_buffer_attr;
    pContext->pulse.pa_stream_set_buffer_attr          = (ma_proc)_pa_stream_set_buffer_attr;
    pContext->pulse.pa_stream_get_device_name          = (ma_proc)_pa_stream_get_device_name;
    pContext->pulse.pa_stream_set_write_callback       = (ma_proc)_pa_stream_set_write_callback;
    pContext->pulse.pa_stream_set_read_callback        = (ma_proc)_pa_stream_set_read_callback;
    pContext->pulse.pa_stream_flush                    = (ma_proc)_pa_stream_flush;
    pContext->pulse.pa_stream_drain                    = (ma_proc)_pa_stream_drain;
    pContext->pulse.pa_stream_is_corked                = (ma_proc)_pa_stream_is_corked;
    pContext->pulse.pa_stream_cork                     = (ma_proc)_pa_stream_cork;
    pContext->pulse.pa_stream_trigger                  = (ma_proc)_pa_stream_trigger;
    pContext->pulse.pa_stream_begin_write              = (ma_proc)_pa_stream_begin_write;
    pContext->pulse.pa_stream_write                    = (ma_proc)_pa_stream_write;
    pContext->pulse.pa_stream_peek                     = (ma_proc)_pa_stream_peek;
    pContext->pulse.pa_stream_drop                     = (ma_proc)_pa_stream_drop;
    pContext->pulse.pa_stream_writable_size            = (ma_proc)_pa_stream_writable_size;
    pContext->pulse.pa_stream_readable_size            = (ma_proc)_pa_stream_readable_size;
#endif

    pContext->onUninit         = ma_context_uninit__pulse;
    pContext->onDeviceIDEqual  = ma_context_is_device_id_equal__pulse;
    pContext->onEnumDevices    = ma_context_enumerate_devices__pulse;
    pContext->onGetDeviceInfo  = ma_context_get_device_info__pulse;
    pContext->onDeviceInit     = ma_device_init__pulse;
    pContext->onDeviceUninit   = ma_device_uninit__pulse;
    pContext->onDeviceStart    = NULL;
    pContext->onDeviceStop     = ma_device_stop__pulse;
    pContext->onDeviceWrite    = ma_device_write__pulse;
    pContext->onDeviceRead     = ma_device_read__pulse;

    if (pConfig->pulse.pApplicationName) {
        pContext->pulse.pApplicationName = ma_copy_string(pConfig->pulse.pApplicationName);
    }
    if (pConfig->pulse.pServerName) {
        pContext->pulse.pServerName = ma_copy_string(pConfig->pulse.pServerName);
    }
    pContext->pulse.tryAutoSpawn = pConfig->pulse.tryAutoSpawn;
    
    /*
    Although we have found the libpulse library, it doesn't necessarily mean PulseAudio is useable. We need to initialize
    and connect a dummy PulseAudio context to test PulseAudio's usability.
    */
    {
        ma_pa_mainloop* pMainLoop;
        ma_pa_mainloop_api* pAPI;
        ma_pa_context* pPulseContext;
        int error;

        pMainLoop = ((ma_pa_mainloop_new_proc)pContext->pulse.pa_mainloop_new)();
        if (pMainLoop == NULL) {
            ma_free(pContext->pulse.pServerName);
            ma_free(pContext->pulse.pApplicationName);
        #ifndef MA_NO_RUNTIME_LINKING
            ma_dlclose(pContext->pulse.pulseSO);
        #endif
            return MA_NO_BACKEND;
        }

        pAPI = ((ma_pa_mainloop_get_api_proc)pContext->pulse.pa_mainloop_get_api)(pMainLoop);
        if (pAPI == NULL) {
            ma_free(pContext->pulse.pServerName);
            ma_free(pContext->pulse.pApplicationName);
            ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        #ifndef MA_NO_RUNTIME_LINKING
            ma_dlclose(pContext->pulse.pulseSO);
        #endif
            return MA_NO_BACKEND;
        }

        pPulseContext = ((ma_pa_context_new_proc)pContext->pulse.pa_context_new)(pAPI, pContext->pulse.pApplicationName);
        if (pPulseContext == NULL) {
            ma_free(pContext->pulse.pServerName);
            ma_free(pContext->pulse.pApplicationName);
            ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        #ifndef MA_NO_RUNTIME_LINKING
            ma_dlclose(pContext->pulse.pulseSO);
        #endif
            return MA_NO_BACKEND;
        }

        error = ((ma_pa_context_connect_proc)pContext->pulse.pa_context_connect)(pPulseContext, pContext->pulse.pServerName, 0, NULL);
        if (error != MA_PA_OK) {
            ma_free(pContext->pulse.pServerName);
            ma_free(pContext->pulse.pApplicationName);
            ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
            ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
        #ifndef MA_NO_RUNTIME_LINKING
            ma_dlclose(pContext->pulse.pulseSO);
        #endif
            return MA_NO_BACKEND;
        }

        ((ma_pa_context_disconnect_proc)pContext->pulse.pa_context_disconnect)(pPulseContext);
        ((ma_pa_context_unref_proc)pContext->pulse.pa_context_unref)(pPulseContext);
        ((ma_pa_mainloop_free_proc)pContext->pulse.pa_mainloop_free)(pMainLoop);
    }

    return MA_SUCCESS;
}
#endif


/******************************************************************************

JACK Backend

******************************************************************************/
#ifdef MA_HAS_JACK

/* It is assumed jack.h is available when compile-time linking is being used. */
#ifdef MA_NO_RUNTIME_LINKING
#include <jack/jack.h>

typedef jack_nframes_t              ma_jack_nframes_t;
typedef jack_options_t              ma_jack_options_t;
typedef jack_status_t               ma_jack_status_t;
typedef jack_client_t               ma_jack_client_t;
typedef jack_port_t                 ma_jack_port_t;
typedef JackProcessCallback         ma_JackProcessCallback;
typedef JackBufferSizeCallback      ma_JackBufferSizeCallback;
typedef JackShutdownCallback        ma_JackShutdownCallback;
#define MA_JACK_DEFAULT_AUDIO_TYPE  JACK_DEFAULT_AUDIO_TYPE
#define ma_JackNoStartServer        JackNoStartServer
#define ma_JackPortIsInput          JackPortIsInput
#define ma_JackPortIsOutput         JackPortIsOutput
#define ma_JackPortIsPhysical       JackPortIsPhysical
#else
typedef ma_uint32               ma_jack_nframes_t;
typedef int                     ma_jack_options_t;
typedef int                     ma_jack_status_t;
typedef struct ma_jack_client_t ma_jack_client_t;
typedef struct ma_jack_port_t   ma_jack_port_t;
typedef int  (* ma_JackProcessCallback)   (ma_jack_nframes_t nframes, void* arg);
typedef int  (* ma_JackBufferSizeCallback)(ma_jack_nframes_t nframes, void* arg);
typedef void (* ma_JackShutdownCallback)  (void* arg);
#define MA_JACK_DEFAULT_AUDIO_TYPE "32 bit float mono audio"
#define ma_JackNoStartServer       1
#define ma_JackPortIsInput         1
#define ma_JackPortIsOutput        2
#define ma_JackPortIsPhysical      4
#endif

typedef ma_jack_client_t* (* ma_jack_client_open_proc)             (const char* client_name, ma_jack_options_t options, ma_jack_status_t* status, ...);
typedef int               (* ma_jack_client_close_proc)            (ma_jack_client_t* client);
typedef int               (* ma_jack_client_name_size_proc)        ();
typedef int               (* ma_jack_set_process_callback_proc)    (ma_jack_client_t* client, ma_JackProcessCallback process_callback, void* arg);
typedef int               (* ma_jack_set_buffer_size_callback_proc)(ma_jack_client_t* client, ma_JackBufferSizeCallback bufsize_callback, void* arg);
typedef void              (* ma_jack_on_shutdown_proc)             (ma_jack_client_t* client, ma_JackShutdownCallback function, void* arg);
typedef ma_jack_nframes_t (* ma_jack_get_sample_rate_proc)         (ma_jack_client_t* client);
typedef ma_jack_nframes_t (* ma_jack_get_buffer_size_proc)         (ma_jack_client_t* client);
typedef const char**      (* ma_jack_get_ports_proc)               (ma_jack_client_t* client, const char* port_name_pattern, const char* type_name_pattern, unsigned long flags);
typedef int               (* ma_jack_activate_proc)                (ma_jack_client_t* client);
typedef int               (* ma_jack_deactivate_proc)              (ma_jack_client_t* client);
typedef int               (* ma_jack_connect_proc)                 (ma_jack_client_t* client, const char* source_port, const char* destination_port);
typedef ma_jack_port_t*   (* ma_jack_port_register_proc)           (ma_jack_client_t* client, const char* port_name, const char* port_type, unsigned long flags, unsigned long buffer_size);
typedef const char*       (* ma_jack_port_name_proc)               (const ma_jack_port_t* port);
typedef void*             (* ma_jack_port_get_buffer_proc)         (ma_jack_port_t* port, ma_jack_nframes_t nframes);
typedef void              (* ma_jack_free_proc)                    (void* ptr);

ma_result ma_context_open_client__jack(ma_context* pContext, ma_jack_client_t** ppClient)
{
    size_t maxClientNameSize;
    char clientName[256];
    ma_jack_status_t status;
    ma_jack_client_t* pClient;

    ma_assert(pContext != NULL);
    ma_assert(ppClient != NULL);

    if (ppClient) {
        *ppClient = NULL;
    }

    maxClientNameSize = ((ma_jack_client_name_size_proc)pContext->jack.jack_client_name_size)(); /* Includes null terminator. */
    ma_strncpy_s(clientName, ma_min(sizeof(clientName), maxClientNameSize), (pContext->jack.pClientName != NULL) ? pContext->jack.pClientName : "miniaudio", (size_t)-1);

    pClient = ((ma_jack_client_open_proc)pContext->jack.jack_client_open)(clientName, (pContext->jack.tryStartServer) ? 0 : ma_JackNoStartServer, &status, NULL);
    if (pClient == NULL) {
        return MA_FAILED_TO_OPEN_BACKEND_DEVICE;
    }

    if (ppClient) {
        *ppClient = pClient;
    }

    return MA_SUCCESS;
}

ma_bool32 ma_context_is_device_id_equal__jack(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return pID0->jack == pID1->jack;
}

ma_result ma_context_enumerate_devices__jack(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_bool32 cbResult = MA_TRUE;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);

    /* Playback. */
    if (cbResult) {
        ma_device_info deviceInfo;
        ma_zero_object(&deviceInfo);
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
    }

    /* Capture. */
    if (cbResult) {
        ma_device_info deviceInfo;
        ma_zero_object(&deviceInfo);
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
    }

    return MA_SUCCESS;
}

ma_result ma_context_get_device_info__jack(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_jack_client_t* pClient;
    ma_result result;
    const char** ppPorts;

    ma_assert(pContext != NULL);

    /* No exclusive mode with the JACK backend. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    if (pDeviceID != NULL && pDeviceID->jack != 0) {
        return MA_NO_DEVICE;   /* Don't know the device. */
    }

    /* Name / Description */
    if (deviceType == ma_device_type_playback) {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    } else {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    }

    /* Jack only supports f32 and has a specific channel count and sample rate. */
    pDeviceInfo->formatCount = 1;
    pDeviceInfo->formats[0] = ma_format_f32;

    /* The channel count and sample rate can only be determined by opening the device. */
    result = ma_context_open_client__jack(pContext, &pClient);
    if (result != MA_SUCCESS) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[JACK] Failed to open client.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    pDeviceInfo->minSampleRate = ((ma_jack_get_sample_rate_proc)pContext->jack.jack_get_sample_rate)((ma_jack_client_t*)pClient);
    pDeviceInfo->maxSampleRate = pDeviceInfo->minSampleRate;

    pDeviceInfo->minChannels = 0;
    pDeviceInfo->maxChannels = 0;

    ppPorts = ((ma_jack_get_ports_proc)pContext->jack.jack_get_ports)((ma_jack_client_t*)pClient, NULL, NULL, ma_JackPortIsPhysical | ((deviceType == ma_device_type_playback) ? ma_JackPortIsInput : ma_JackPortIsOutput));
    if (ppPorts == NULL) {
        ((ma_jack_client_close_proc)pContext->jack.jack_client_close)((ma_jack_client_t*)pClient);
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[JACK] Failed to query physical ports.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    while (ppPorts[pDeviceInfo->minChannels] != NULL) {
        pDeviceInfo->minChannels += 1;
        pDeviceInfo->maxChannels += 1;
    }

    ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);
    ((ma_jack_client_close_proc)pContext->jack.jack_client_close)((ma_jack_client_t*)pClient);

    (void)pContext;
    return MA_SUCCESS;
}


void ma_device_uninit__jack(ma_device* pDevice)
{
    ma_context* pContext;

    ma_assert(pDevice != NULL);

    pContext = pDevice->pContext;
    ma_assert(pContext != NULL);

    if (pDevice->jack.pClient != NULL) {
        ((ma_jack_client_close_proc)pContext->jack.jack_client_close)((ma_jack_client_t*)pDevice->jack.pClient);
    }

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ma_free(pDevice->jack.pIntermediaryBufferCapture);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ma_free(pDevice->jack.pIntermediaryBufferPlayback);
    }

    if (pDevice->type == ma_device_type_duplex) {
        ma_pcm_rb_uninit(&pDevice->jack.duplexRB);
    }
}

void ma_device__jack_shutdown_callback(void* pUserData)
{
    /* JACK died. Stop the device. */
    ma_device* pDevice = (ma_device*)pUserData;
    ma_assert(pDevice != NULL);

    ma_device_stop(pDevice);
}

int ma_device__jack_buffer_size_callback(ma_jack_nframes_t frameCount, void* pUserData)
{
    ma_device* pDevice = (ma_device*)pUserData;
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        float* pNewBuffer = (float*)ma_realloc(pDevice->jack.pIntermediaryBufferCapture, frameCount * (pDevice->capture.internalChannels * ma_get_bytes_per_sample(pDevice->capture.internalFormat)));
        if (pNewBuffer == NULL) {
            return MA_OUT_OF_MEMORY;
        }

        pDevice->jack.pIntermediaryBufferCapture = pNewBuffer;
        pDevice->playback.internalBufferSizeInFrames = frameCount * pDevice->capture.internalPeriods;
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        float* pNewBuffer = (float*)ma_realloc(pDevice->jack.pIntermediaryBufferPlayback, frameCount * (pDevice->playback.internalChannels * ma_get_bytes_per_sample(pDevice->playback.internalFormat)));
        if (pNewBuffer == NULL) {
            return MA_OUT_OF_MEMORY;
        }

        pDevice->jack.pIntermediaryBufferPlayback = pNewBuffer;
        pDevice->playback.internalBufferSizeInFrames = frameCount * pDevice->playback.internalPeriods;
    }

    return 0;
}

int ma_device__jack_process_callback(ma_jack_nframes_t frameCount, void* pUserData)
{
    ma_device* pDevice;
    ma_context* pContext;
    ma_uint32 iChannel;

    pDevice = (ma_device*)pUserData;
    ma_assert(pDevice != NULL);

    pContext = pDevice->pContext;
    ma_assert(pContext != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        /* Channels need to be interleaved. */
        for (iChannel = 0; iChannel < pDevice->capture.internalChannels; ++iChannel) {
            const float* pSrc = (const float*)((ma_jack_port_get_buffer_proc)pContext->jack.jack_port_get_buffer)((ma_jack_port_t*)pDevice->jack.pPortsCapture[iChannel], frameCount);
            if (pSrc != NULL) {
                float* pDst = pDevice->jack.pIntermediaryBufferCapture + iChannel;
                ma_jack_nframes_t iFrame;
                for (iFrame = 0; iFrame < frameCount; ++iFrame) {
                    *pDst = *pSrc;

                    pDst += pDevice->capture.internalChannels;
                    pSrc += 1;
                }
            }
        }

        if (pDevice->type == ma_device_type_duplex) {
            ma_device__handle_duplex_callback_capture(pDevice, frameCount, pDevice->jack.pIntermediaryBufferCapture, &pDevice->jack.duplexRB);
        } else {
            ma_device__send_frames_to_client(pDevice, frameCount, pDevice->jack.pIntermediaryBufferCapture);
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        if (pDevice->type == ma_device_type_duplex) {
            ma_device__handle_duplex_callback_playback(pDevice, frameCount, pDevice->jack.pIntermediaryBufferPlayback, &pDevice->jack.duplexRB);
        } else {
            ma_device__read_frames_from_client(pDevice, frameCount, pDevice->jack.pIntermediaryBufferPlayback);
        }

        /* Channels need to be deinterleaved. */
        for (iChannel = 0; iChannel < pDevice->playback.internalChannels; ++iChannel) {
            float* pDst = (float*)((ma_jack_port_get_buffer_proc)pContext->jack.jack_port_get_buffer)((ma_jack_port_t*)pDevice->jack.pPortsPlayback[iChannel], frameCount);
            if (pDst != NULL) {
                const float* pSrc = pDevice->jack.pIntermediaryBufferPlayback + iChannel;
                ma_jack_nframes_t iFrame;
                for (iFrame = 0; iFrame < frameCount; ++iFrame) {
                    *pDst = *pSrc;

                    pDst += 1;
                    pSrc += pDevice->playback.internalChannels;
                }
            }
        }
    }

    return 0;
}

ma_result ma_device_init__jack(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result;
    ma_uint32 periods;
    ma_uint32 bufferSizeInFrames;

    ma_assert(pContext != NULL);
    ma_assert(pConfig != NULL);
    ma_assert(pDevice != NULL);

    /* Only supporting default devices with JACK. */
    if (((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.pDeviceID != NULL && pConfig->playback.pDeviceID->jack != 0) ||
        ((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.pDeviceID  != NULL && pConfig->capture.pDeviceID->jack  != 0)) {
        return MA_NO_DEVICE;
    }

    /* No exclusive mode with the JACK backend. */
    if (((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.shareMode == ma_share_mode_exclusive) ||
        ((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.shareMode  == ma_share_mode_exclusive)) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    /* Open the client. */
    result = ma_context_open_client__jack(pContext, (ma_jack_client_t**)&pDevice->jack.pClient);
    if (result != MA_SUCCESS) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to open client.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    /* Callbacks. */
    if (((ma_jack_set_process_callback_proc)pContext->jack.jack_set_process_callback)((ma_jack_client_t*)pDevice->jack.pClient, ma_device__jack_process_callback, pDevice) != 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to set process callback.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }
    if (((ma_jack_set_buffer_size_callback_proc)pContext->jack.jack_set_buffer_size_callback)((ma_jack_client_t*)pDevice->jack.pClient, ma_device__jack_buffer_size_callback, pDevice) != 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to set buffer size callback.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    ((ma_jack_on_shutdown_proc)pContext->jack.jack_on_shutdown)((ma_jack_client_t*)pDevice->jack.pClient, ma_device__jack_shutdown_callback, pDevice);


    /* The buffer size in frames can change. */
    periods            = 2;
    bufferSizeInFrames = ((ma_jack_get_buffer_size_proc)pContext->jack.jack_get_buffer_size)((ma_jack_client_t*)pDevice->jack.pClient) * periods;
    
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        const char** ppPorts;

        pDevice->capture.internalFormat = ma_format_f32;
        pDevice->capture.internalChannels = 0;
        pDevice->capture.internalSampleRate = ((ma_jack_get_sample_rate_proc)pContext->jack.jack_get_sample_rate)((ma_jack_client_t*)pDevice->jack.pClient);
        ma_get_standard_channel_map(ma_standard_channel_map_alsa, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);

        ppPorts = ((ma_jack_get_ports_proc)pContext->jack.jack_get_ports)((ma_jack_client_t*)pDevice->jack.pClient, NULL, NULL, ma_JackPortIsPhysical | ma_JackPortIsOutput);
        if (ppPorts == NULL) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to query physical ports.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        while (ppPorts[pDevice->capture.internalChannels] != NULL) {
            char name[64];
            ma_strcpy_s(name, sizeof(name), "capture");
            ma_itoa_s((int)pDevice->capture.internalChannels, name+7, sizeof(name)-7, 10); /* 7 = length of "capture" */

            pDevice->jack.pPortsCapture[pDevice->capture.internalChannels] = ((ma_jack_port_register_proc)pContext->jack.jack_port_register)((ma_jack_client_t*)pDevice->jack.pClient, name, MA_JACK_DEFAULT_AUDIO_TYPE, ma_JackPortIsInput, 0);
            if (pDevice->jack.pPortsCapture[pDevice->capture.internalChannels] == NULL) {
                ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);
                ma_device_uninit__jack(pDevice);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to register ports.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
            }

            pDevice->capture.internalChannels += 1;
        }

        ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);

        pDevice->capture.internalBufferSizeInFrames = bufferSizeInFrames;
        pDevice->capture.internalPeriods            = periods;

        pDevice->jack.pIntermediaryBufferCapture = (float*)ma_malloc((pDevice->capture.internalBufferSizeInFrames/pDevice->capture.internalPeriods) * (pDevice->capture.internalChannels * ma_get_bytes_per_sample(pDevice->capture.internalFormat)));
        if (pDevice->jack.pIntermediaryBufferCapture == NULL) {
            ma_device_uninit__jack(pDevice);
            return MA_OUT_OF_MEMORY;
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        const char** ppPorts;

        pDevice->playback.internalFormat = ma_format_f32;
        pDevice->playback.internalChannels = 0;
        pDevice->playback.internalSampleRate = ((ma_jack_get_sample_rate_proc)pContext->jack.jack_get_sample_rate)((ma_jack_client_t*)pDevice->jack.pClient);
        ma_get_standard_channel_map(ma_standard_channel_map_alsa, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);

        ppPorts = ((ma_jack_get_ports_proc)pContext->jack.jack_get_ports)((ma_jack_client_t*)pDevice->jack.pClient, NULL, NULL, ma_JackPortIsPhysical | ma_JackPortIsInput);
        if (ppPorts == NULL) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to query physical ports.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }

        while (ppPorts[pDevice->playback.internalChannels] != NULL) {
            char name[64];
            ma_strcpy_s(name, sizeof(name), "playback");
            ma_itoa_s((int)pDevice->playback.internalChannels, name+8, sizeof(name)-8, 10); /* 8 = length of "playback" */

            pDevice->jack.pPortsPlayback[pDevice->playback.internalChannels] = ((ma_jack_port_register_proc)pContext->jack.jack_port_register)((ma_jack_client_t*)pDevice->jack.pClient, name, MA_JACK_DEFAULT_AUDIO_TYPE, ma_JackPortIsOutput, 0);
            if (pDevice->jack.pPortsPlayback[pDevice->playback.internalChannels] == NULL) {
                ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);
                ma_device_uninit__jack(pDevice);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to register ports.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
            }

            pDevice->playback.internalChannels += 1;
        }

        ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppPorts);

        pDevice->playback.internalBufferSizeInFrames = bufferSizeInFrames;
        pDevice->playback.internalPeriods            = periods;

        pDevice->jack.pIntermediaryBufferPlayback = (float*)ma_malloc((pDevice->playback.internalBufferSizeInFrames/pDevice->playback.internalPeriods) * (pDevice->playback.internalChannels * ma_get_bytes_per_sample(pDevice->playback.internalFormat)));
        if (pDevice->jack.pIntermediaryBufferPlayback == NULL) {
            ma_device_uninit__jack(pDevice);
            return MA_OUT_OF_MEMORY;
        }
    }

    if (pDevice->type == ma_device_type_duplex) {
        ma_uint32 rbSizeInFrames = (ma_uint32)ma_calculate_frame_count_after_src(pDevice->sampleRate, pDevice->capture.internalSampleRate, pDevice->capture.internalBufferSizeInFrames);
        result = ma_pcm_rb_init(pDevice->capture.format, pDevice->capture.channels, rbSizeInFrames, NULL, &pDevice->jack.duplexRB);
        if (result != MA_SUCCESS) {
            ma_device_uninit__jack(pDevice);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to initialize ring buffer.", result);
        }
    }

    return MA_SUCCESS;
}


ma_result ma_device_start__jack(ma_device* pDevice)
{
    ma_context* pContext = pDevice->pContext;
    int resultJACK;
    size_t i;

    resultJACK = ((ma_jack_activate_proc)pContext->jack.jack_activate)((ma_jack_client_t*)pDevice->jack.pClient);
    if (resultJACK != 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to activate the JACK client.", MA_FAILED_TO_START_BACKEND_DEVICE);
    }

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        const char** ppServerPorts = ((ma_jack_get_ports_proc)pContext->jack.jack_get_ports)((ma_jack_client_t*)pDevice->jack.pClient, NULL, NULL, ma_JackPortIsPhysical | ma_JackPortIsOutput);
        if (ppServerPorts == NULL) {
            ((ma_jack_deactivate_proc)pContext->jack.jack_deactivate)((ma_jack_client_t*)pDevice->jack.pClient);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to retrieve physical ports.", MA_ERROR);
        }

        for (i = 0; ppServerPorts[i] != NULL; ++i) {
            const char* pServerPort = ppServerPorts[i];
            const char* pClientPort = ((ma_jack_port_name_proc)pContext->jack.jack_port_name)((ma_jack_port_t*)pDevice->jack.pPortsCapture[i]);

            resultJACK = ((ma_jack_connect_proc)pContext->jack.jack_connect)((ma_jack_client_t*)pDevice->jack.pClient, pServerPort, pClientPort);
            if (resultJACK != 0) {
                ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppServerPorts);
                ((ma_jack_deactivate_proc)pContext->jack.jack_deactivate)((ma_jack_client_t*)pDevice->jack.pClient);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to connect ports.", MA_ERROR);
            }
        }

        ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppServerPorts);
    }
    
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        const char** ppServerPorts = ((ma_jack_get_ports_proc)pContext->jack.jack_get_ports)((ma_jack_client_t*)pDevice->jack.pClient, NULL, NULL, ma_JackPortIsPhysical | ma_JackPortIsInput);
        if (ppServerPorts == NULL) {
            ((ma_jack_deactivate_proc)pContext->jack.jack_deactivate)((ma_jack_client_t*)pDevice->jack.pClient);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to retrieve physical ports.", MA_ERROR);
        }

        for (i = 0; ppServerPorts[i] != NULL; ++i) {
            const char* pServerPort = ppServerPorts[i];
            const char* pClientPort = ((ma_jack_port_name_proc)pContext->jack.jack_port_name)((ma_jack_port_t*)pDevice->jack.pPortsPlayback[i]);

            resultJACK = ((ma_jack_connect_proc)pContext->jack.jack_connect)((ma_jack_client_t*)pDevice->jack.pClient, pClientPort, pServerPort);
            if (resultJACK != 0) {
                ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppServerPorts);
                ((ma_jack_deactivate_proc)pContext->jack.jack_deactivate)((ma_jack_client_t*)pDevice->jack.pClient);
                return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] Failed to connect ports.", MA_ERROR);
            }
        }

        ((ma_jack_free_proc)pContext->jack.jack_free)((void*)ppServerPorts);
    }

    return MA_SUCCESS;
}

ma_result ma_device_stop__jack(ma_device* pDevice)
{
    ma_context* pContext = pDevice->pContext;
    ma_stop_proc onStop;

    if (((ma_jack_deactivate_proc)pContext->jack.jack_deactivate)((ma_jack_client_t*)pDevice->jack.pClient) != 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[JACK] An error occurred when deactivating the JACK client.", MA_ERROR);
    }
    
    onStop = pDevice->onStop;
    if (onStop) {
        onStop(pDevice);
    }

    return MA_SUCCESS;
}


ma_result ma_context_uninit__jack(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_jack);

    ma_free(pContext->jack.pClientName);
    pContext->jack.pClientName = NULL;

#ifndef MA_NO_RUNTIME_LINKING
    ma_dlclose(pContext->jack.jackSO);
#endif

    return MA_SUCCESS;
}

ma_result ma_context_init__jack(const ma_context_config* pConfig, ma_context* pContext)
{
#ifndef MA_NO_RUNTIME_LINKING
    const char* libjackNames[] = {
#ifdef MA_WIN32
        "libjack.dll"
#else
        "libjack.so",
        "libjack.so.0"
#endif
    };
    size_t i;

    for (i = 0; i < ma_countof(libjackNames); ++i) {
        pContext->jack.jackSO = ma_dlopen(libjackNames[i]);
        if (pContext->jack.jackSO != NULL) {
            break;
        }
    }

    if (pContext->jack.jackSO == NULL) {
        return MA_NO_BACKEND;
    }

    pContext->jack.jack_client_open              = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_client_open");
    pContext->jack.jack_client_close             = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_client_close");
    pContext->jack.jack_client_name_size         = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_client_name_size");
    pContext->jack.jack_set_process_callback     = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_set_process_callback");
    pContext->jack.jack_set_buffer_size_callback = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_set_buffer_size_callback");
    pContext->jack.jack_on_shutdown              = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_on_shutdown");
    pContext->jack.jack_get_sample_rate          = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_get_sample_rate");
    pContext->jack.jack_get_buffer_size          = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_get_buffer_size");
    pContext->jack.jack_get_ports                = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_get_ports");
    pContext->jack.jack_activate                 = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_activate");
    pContext->jack.jack_deactivate               = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_deactivate");
    pContext->jack.jack_connect                  = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_connect");
    pContext->jack.jack_port_register            = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_port_register");
    pContext->jack.jack_port_name                = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_port_name");
    pContext->jack.jack_port_get_buffer          = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_port_get_buffer");
    pContext->jack.jack_free                     = (ma_proc)ma_dlsym(pContext->jack.jackSO, "jack_free");
#else
    /*
    This strange assignment system is here just to ensure type safety of miniaudio's function pointer
    types. If anything differs slightly the compiler should throw a warning.
    */
    ma_jack_client_open_proc              _jack_client_open              = jack_client_open;
    ma_jack_client_close_proc             _jack_client_close             = jack_client_close;
    ma_jack_client_name_size_proc         _jack_client_name_size         = jack_client_name_size;
    ma_jack_set_process_callback_proc     _jack_set_process_callback     = jack_set_process_callback;
    ma_jack_set_buffer_size_callback_proc _jack_set_buffer_size_callback = jack_set_buffer_size_callback;
    ma_jack_on_shutdown_proc              _jack_on_shutdown              = jack_on_shutdown;
    ma_jack_get_sample_rate_proc          _jack_get_sample_rate          = jack_get_sample_rate;
    ma_jack_get_buffer_size_proc          _jack_get_buffer_size          = jack_get_buffer_size;
    ma_jack_get_ports_proc                _jack_get_ports                = jack_get_ports;
    ma_jack_activate_proc                 _jack_activate                 = jack_activate;
    ma_jack_deactivate_proc               _jack_deactivate               = jack_deactivate;
    ma_jack_connect_proc                  _jack_connect                  = jack_connect;
    ma_jack_port_register_proc            _jack_port_register            = jack_port_register;
    ma_jack_port_name_proc                _jack_port_name                = jack_port_name;
    ma_jack_port_get_buffer_proc          _jack_port_get_buffer          = jack_port_get_buffer;
    ma_jack_free_proc                     _jack_free                     = jack_free;

    pContext->jack.jack_client_open              = (ma_proc)_jack_client_open;
    pContext->jack.jack_client_close             = (ma_proc)_jack_client_close;
    pContext->jack.jack_client_name_size         = (ma_proc)_jack_client_name_size;
    pContext->jack.jack_set_process_callback     = (ma_proc)_jack_set_process_callback;
    pContext->jack.jack_set_buffer_size_callback = (ma_proc)_jack_set_buffer_size_callback;
    pContext->jack.jack_on_shutdown              = (ma_proc)_jack_on_shutdown;
    pContext->jack.jack_get_sample_rate          = (ma_proc)_jack_get_sample_rate;
    pContext->jack.jack_get_buffer_size          = (ma_proc)_jack_get_buffer_size;
    pContext->jack.jack_get_ports                = (ma_proc)_jack_get_ports;
    pContext->jack.jack_activate                 = (ma_proc)_jack_activate;
    pContext->jack.jack_deactivate               = (ma_proc)_jack_deactivate;
    pContext->jack.jack_connect                  = (ma_proc)_jack_connect;
    pContext->jack.jack_port_register            = (ma_proc)_jack_port_register;
    pContext->jack.jack_port_name                = (ma_proc)_jack_port_name;
    pContext->jack.jack_port_get_buffer          = (ma_proc)_jack_port_get_buffer;
    pContext->jack.jack_free                     = (ma_proc)_jack_free;
#endif

    pContext->isBackendAsynchronous = MA_TRUE;

    pContext->onUninit        = ma_context_uninit__jack;
    pContext->onDeviceIDEqual = ma_context_is_device_id_equal__jack;
    pContext->onEnumDevices   = ma_context_enumerate_devices__jack;
    pContext->onGetDeviceInfo = ma_context_get_device_info__jack;
    pContext->onDeviceInit    = ma_device_init__jack;
    pContext->onDeviceUninit  = ma_device_uninit__jack;
    pContext->onDeviceStart   = ma_device_start__jack;
    pContext->onDeviceStop    = ma_device_stop__jack;

    if (pConfig->jack.pClientName != NULL) {
        pContext->jack.pClientName = ma_copy_string(pConfig->jack.pClientName);
    }
    pContext->jack.tryStartServer = pConfig->jack.tryStartServer;

    /*
    Getting here means the JACK library is installed, but it doesn't necessarily mean it's usable. We need to quickly test this by connecting
    a temporary client.
    */
    {
        ma_jack_client_t* pDummyClient;
        ma_result result = ma_context_open_client__jack(pContext, &pDummyClient);
        if (result != MA_SUCCESS) {
            ma_free(pContext->jack.pClientName);
        #ifndef MA_NO_RUNTIME_LINKING
            ma_dlclose(pContext->jack.jackSO);
        #endif
            return MA_NO_BACKEND;
        }

        ((ma_jack_client_close_proc)pContext->jack.jack_client_close)((ma_jack_client_t*)pDummyClient);
    }

    return MA_SUCCESS;
}
#endif  /* JACK */



/******************************************************************************

Core Audio Backend

******************************************************************************/
#ifdef MA_HAS_COREAUDIO
#include <TargetConditionals.h>

#if defined(TARGET_OS_IPHONE) && TARGET_OS_IPHONE == 1
    #define MA_APPLE_MOBILE
#else
    #define MA_APPLE_DESKTOP
#endif

#if defined(MA_APPLE_DESKTOP)
#include <CoreAudio/CoreAudio.h>
#else
#include <AVFoundation/AVFoundation.h>
#endif

#include <AudioToolbox/AudioToolbox.h>

/* CoreFoundation */
typedef Boolean (* ma_CFStringGetCString_proc)(CFStringRef theString, char* buffer, CFIndex bufferSize, CFStringEncoding encoding);

/* CoreAudio */
#if defined(MA_APPLE_DESKTOP)
typedef OSStatus (* ma_AudioObjectGetPropertyData_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32* ioDataSize, void* outData);
typedef OSStatus (* ma_AudioObjectGetPropertyDataSize_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32* outDataSize);
typedef OSStatus (* ma_AudioObjectSetPropertyData_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, UInt32 inQualifierDataSize, const void* inQualifierData, UInt32 inDataSize, const void* inData);
typedef OSStatus (* ma_AudioObjectAddPropertyListener_proc)(AudioObjectID inObjectID, const AudioObjectPropertyAddress* inAddress, AudioObjectPropertyListenerProc inListener, void* inClientData);
#endif

/* AudioToolbox */
typedef AudioComponent (* ma_AudioComponentFindNext_proc)(AudioComponent inComponent, const AudioComponentDescription* inDesc);
typedef OSStatus (* ma_AudioComponentInstanceDispose_proc)(AudioComponentInstance inInstance);
typedef OSStatus (* ma_AudioComponentInstanceNew_proc)(AudioComponent inComponent, AudioComponentInstance* outInstance);
typedef OSStatus (* ma_AudioOutputUnitStart_proc)(AudioUnit inUnit);
typedef OSStatus (* ma_AudioOutputUnitStop_proc)(AudioUnit inUnit);
typedef OSStatus (* ma_AudioUnitAddPropertyListener_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitPropertyListenerProc inProc, void* inProcUserData);
typedef OSStatus (* ma_AudioUnitGetPropertyInfo_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitScope inScope, AudioUnitElement inElement, UInt32* outDataSize, Boolean* outWriteable);
typedef OSStatus (* ma_AudioUnitGetProperty_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitScope inScope, AudioUnitElement inElement, void* outData, UInt32* ioDataSize);
typedef OSStatus (* ma_AudioUnitSetProperty_proc)(AudioUnit inUnit, AudioUnitPropertyID inID, AudioUnitScope inScope, AudioUnitElement inElement, const void* inData, UInt32 inDataSize);
typedef OSStatus (* ma_AudioUnitInitialize_proc)(AudioUnit inUnit);
typedef OSStatus (* ma_AudioUnitRender_proc)(AudioUnit inUnit, AudioUnitRenderActionFlags* ioActionFlags, const AudioTimeStamp* inTimeStamp, UInt32 inOutputBusNumber, UInt32 inNumberFrames, AudioBufferList* ioData);


#define MA_COREAUDIO_OUTPUT_BUS    0
#define MA_COREAUDIO_INPUT_BUS     1

ma_result ma_device_reinit_internal__coreaudio(ma_device* pDevice, ma_device_type deviceType, ma_bool32 disposePreviousAudioUnit);

/*
Core Audio

So far, Core Audio has been the worst backend to work with due to being both unintuitive and having almost no documentation
apart from comments in the headers (which admittedly are quite good). For my own purposes, and for anybody out there whose
needing to figure out how this darn thing works, I'm going to outline a few things here.

Since miniaudio is a fairly low-level API, one of the things it needs is control over specific devices, and it needs to be
able to identify whether or not it can be used as playback and/or capture. The AudioObject API is the only one I've seen
that supports this level of detail. There was some public domain sample code I stumbled across that used the AudioComponent
and AudioUnit APIs, but I couldn't see anything that gave low-level control over device selection and capabilities (the
distinction between playback and capture in particular). Therefore, miniaudio is using the AudioObject API.

Most (all?) functions in the AudioObject API take a AudioObjectID as it's input. This is the device identifier. When
retrieving global information, such as the device list, you use kAudioObjectSystemObject. When retrieving device-specific
data, you pass in the ID for that device. In order to retrieve device-specific IDs you need to enumerate over each of the
devices. This is done using the AudioObjectGetPropertyDataSize() and AudioObjectGetPropertyData() APIs which seem to be
the central APIs for retrieving information about the system and specific devices.

To use the AudioObjectGetPropertyData() API you need to use the notion of a property address. A property address is a
structure with three variables and is used to identify which property you are getting or setting. The first is the "selector"
which is basically the specific property that you're wanting to retrieve or set. The second is the "scope", which is
typically set to kAudioObjectPropertyScopeGlobal, kAudioObjectPropertyScopeInput for input-specific properties and
kAudioObjectPropertyScopeOutput for output-specific properties. The last is the "element" which is always set to
kAudioObjectPropertyElementMaster in miniaudio's case. I don't know of any cases where this would be set to anything different.

Back to the earlier issue of device retrieval, you first use the AudioObjectGetPropertyDataSize() API to retrieve the size
of the raw data which is just a list of AudioDeviceID's. You use the kAudioObjectSystemObject AudioObjectID, and a property
address with the kAudioHardwarePropertyDevices selector and the kAudioObjectPropertyScopeGlobal scope. Once you have the
size, allocate a block of memory of that size and then call AudioObjectGetPropertyData(). The data is just a list of
AudioDeviceID's so just do "dataSize/sizeof(AudioDeviceID)" to know the device count.
*/

ma_result ma_result_from_OSStatus(OSStatus status)
{
    switch (status)
    {
        case noErr:                                   return MA_SUCCESS;
    #if defined(MA_APPLE_DESKTOP)
        case kAudioHardwareNotRunningError:           return MA_DEVICE_NOT_STARTED;
        case kAudioHardwareUnspecifiedError:          return MA_ERROR;
        case kAudioHardwareUnknownPropertyError:      return MA_INVALID_ARGS;
        case kAudioHardwareBadPropertySizeError:      return MA_INVALID_OPERATION;
        case kAudioHardwareIllegalOperationError:     return MA_INVALID_OPERATION;
        case kAudioHardwareBadObjectError:            return MA_INVALID_ARGS;
        case kAudioHardwareBadDeviceError:            return MA_INVALID_ARGS;
        case kAudioHardwareBadStreamError:            return MA_INVALID_ARGS;
        case kAudioHardwareUnsupportedOperationError: return MA_INVALID_OPERATION;
        case kAudioDeviceUnsupportedFormatError:      return MA_FORMAT_NOT_SUPPORTED;
        case kAudioDevicePermissionsError:            return MA_ACCESS_DENIED;
    #endif
        default:                                      return MA_ERROR;
    }
}

#if 0
ma_channel ma_channel_from_AudioChannelBitmap(AudioChannelBitmap bit)
{
    switch (bit)
    {
        case kAudioChannelBit_Left:                 return MA_CHANNEL_LEFT;
        case kAudioChannelBit_Right:                return MA_CHANNEL_RIGHT;
        case kAudioChannelBit_Center:               return MA_CHANNEL_FRONT_CENTER;
        case kAudioChannelBit_LFEScreen:            return MA_CHANNEL_LFE;
        case kAudioChannelBit_LeftSurround:         return MA_CHANNEL_BACK_LEFT;
        case kAudioChannelBit_RightSurround:        return MA_CHANNEL_BACK_RIGHT;
        case kAudioChannelBit_LeftCenter:           return MA_CHANNEL_FRONT_LEFT_CENTER;
        case kAudioChannelBit_RightCenter:          return MA_CHANNEL_FRONT_RIGHT_CENTER;
        case kAudioChannelBit_CenterSurround:       return MA_CHANNEL_BACK_CENTER;
        case kAudioChannelBit_LeftSurroundDirect:   return MA_CHANNEL_SIDE_LEFT;
        case kAudioChannelBit_RightSurroundDirect:  return MA_CHANNEL_SIDE_RIGHT;
        case kAudioChannelBit_TopCenterSurround:    return MA_CHANNEL_TOP_CENTER;
        case kAudioChannelBit_VerticalHeightLeft:   return MA_CHANNEL_TOP_FRONT_LEFT;
        case kAudioChannelBit_VerticalHeightCenter: return MA_CHANNEL_TOP_FRONT_CENTER;
        case kAudioChannelBit_VerticalHeightRight:  return MA_CHANNEL_TOP_FRONT_RIGHT;
        case kAudioChannelBit_TopBackLeft:          return MA_CHANNEL_TOP_BACK_LEFT;
        case kAudioChannelBit_TopBackCenter:        return MA_CHANNEL_TOP_BACK_CENTER;
        case kAudioChannelBit_TopBackRight:         return MA_CHANNEL_TOP_BACK_RIGHT;
        default:                                    return MA_CHANNEL_NONE;
    }
}
#endif

ma_channel ma_channel_from_AudioChannelLabel(AudioChannelLabel label)
{
    switch (label)
    {
        case kAudioChannelLabel_Unknown:              return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Unused:               return MA_CHANNEL_NONE;
        case kAudioChannelLabel_UseCoordinates:       return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Left:                 return MA_CHANNEL_LEFT;
        case kAudioChannelLabel_Right:                return MA_CHANNEL_RIGHT;
        case kAudioChannelLabel_Center:               return MA_CHANNEL_FRONT_CENTER;
        case kAudioChannelLabel_LFEScreen:            return MA_CHANNEL_LFE;
        case kAudioChannelLabel_LeftSurround:         return MA_CHANNEL_BACK_LEFT;
        case kAudioChannelLabel_RightSurround:        return MA_CHANNEL_BACK_RIGHT;
        case kAudioChannelLabel_LeftCenter:           return MA_CHANNEL_FRONT_LEFT_CENTER;
        case kAudioChannelLabel_RightCenter:          return MA_CHANNEL_FRONT_RIGHT_CENTER;
        case kAudioChannelLabel_CenterSurround:       return MA_CHANNEL_BACK_CENTER;
        case kAudioChannelLabel_LeftSurroundDirect:   return MA_CHANNEL_SIDE_LEFT;
        case kAudioChannelLabel_RightSurroundDirect:  return MA_CHANNEL_SIDE_RIGHT;
        case kAudioChannelLabel_TopCenterSurround:    return MA_CHANNEL_TOP_CENTER;
        case kAudioChannelLabel_VerticalHeightLeft:   return MA_CHANNEL_TOP_FRONT_LEFT;
        case kAudioChannelLabel_VerticalHeightCenter: return MA_CHANNEL_TOP_FRONT_CENTER;
        case kAudioChannelLabel_VerticalHeightRight:  return MA_CHANNEL_TOP_FRONT_RIGHT;
        case kAudioChannelLabel_TopBackLeft:          return MA_CHANNEL_TOP_BACK_LEFT;
        case kAudioChannelLabel_TopBackCenter:        return MA_CHANNEL_TOP_BACK_CENTER;
        case kAudioChannelLabel_TopBackRight:         return MA_CHANNEL_TOP_BACK_RIGHT;
        case kAudioChannelLabel_RearSurroundLeft:     return MA_CHANNEL_BACK_LEFT;
        case kAudioChannelLabel_RearSurroundRight:    return MA_CHANNEL_BACK_RIGHT;
        case kAudioChannelLabel_LeftWide:             return MA_CHANNEL_SIDE_LEFT;
        case kAudioChannelLabel_RightWide:            return MA_CHANNEL_SIDE_RIGHT;
        case kAudioChannelLabel_LFE2:                 return MA_CHANNEL_LFE;
        case kAudioChannelLabel_LeftTotal:            return MA_CHANNEL_LEFT;
        case kAudioChannelLabel_RightTotal:           return MA_CHANNEL_RIGHT;
        case kAudioChannelLabel_HearingImpaired:      return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Narration:            return MA_CHANNEL_MONO;
        case kAudioChannelLabel_Mono:                 return MA_CHANNEL_MONO;
        case kAudioChannelLabel_DialogCentricMix:     return MA_CHANNEL_MONO;
        case kAudioChannelLabel_CenterSurroundDirect: return MA_CHANNEL_BACK_CENTER;
        case kAudioChannelLabel_Haptic:               return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_W:          return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_X:          return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_Y:          return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Ambisonic_Z:          return MA_CHANNEL_NONE;
        case kAudioChannelLabel_MS_Mid:               return MA_CHANNEL_LEFT;
        case kAudioChannelLabel_MS_Side:              return MA_CHANNEL_RIGHT;
        case kAudioChannelLabel_XY_X:                 return MA_CHANNEL_LEFT;
        case kAudioChannelLabel_XY_Y:                 return MA_CHANNEL_RIGHT;
        case kAudioChannelLabel_HeadphonesLeft:       return MA_CHANNEL_LEFT;
        case kAudioChannelLabel_HeadphonesRight:      return MA_CHANNEL_RIGHT;
        case kAudioChannelLabel_ClickTrack:           return MA_CHANNEL_NONE;
        case kAudioChannelLabel_ForeignLanguage:      return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Discrete:             return MA_CHANNEL_NONE;
        case kAudioChannelLabel_Discrete_0:           return MA_CHANNEL_AUX_0;
        case kAudioChannelLabel_Discrete_1:           return MA_CHANNEL_AUX_1;
        case kAudioChannelLabel_Discrete_2:           return MA_CHANNEL_AUX_2;
        case kAudioChannelLabel_Discrete_3:           return MA_CHANNEL_AUX_3;
        case kAudioChannelLabel_Discrete_4:           return MA_CHANNEL_AUX_4;
        case kAudioChannelLabel_Discrete_5:           return MA_CHANNEL_AUX_5;
        case kAudioChannelLabel_Discrete_6:           return MA_CHANNEL_AUX_6;
        case kAudioChannelLabel_Discrete_7:           return MA_CHANNEL_AUX_7;
        case kAudioChannelLabel_Discrete_8:           return MA_CHANNEL_AUX_8;
        case kAudioChannelLabel_Discrete_9:           return MA_CHANNEL_AUX_9;
        case kAudioChannelLabel_Discrete_10:          return MA_CHANNEL_AUX_10;
        case kAudioChannelLabel_Discrete_11:          return MA_CHANNEL_AUX_11;
        case kAudioChannelLabel_Discrete_12:          return MA_CHANNEL_AUX_12;
        case kAudioChannelLabel_Discrete_13:          return MA_CHANNEL_AUX_13;
        case kAudioChannelLabel_Discrete_14:          return MA_CHANNEL_AUX_14;
        case kAudioChannelLabel_Discrete_15:          return MA_CHANNEL_AUX_15;
        case kAudioChannelLabel_Discrete_65535:       return MA_CHANNEL_NONE;
        
    #if 0   /* Introduced in a later version of macOS. */
        case kAudioChannelLabel_HOA_ACN:              return MA_CHANNEL_NONE;
        case kAudioChannelLabel_HOA_ACN_0:            return MA_CHANNEL_AUX_0;
        case kAudioChannelLabel_HOA_ACN_1:            return MA_CHANNEL_AUX_1;
        case kAudioChannelLabel_HOA_ACN_2:            return MA_CHANNEL_AUX_2;
        case kAudioChannelLabel_HOA_ACN_3:            return MA_CHANNEL_AUX_3;
        case kAudioChannelLabel_HOA_ACN_4:            return MA_CHANNEL_AUX_4;
        case kAudioChannelLabel_HOA_ACN_5:            return MA_CHANNEL_AUX_5;
        case kAudioChannelLabel_HOA_ACN_6:            return MA_CHANNEL_AUX_6;
        case kAudioChannelLabel_HOA_ACN_7:            return MA_CHANNEL_AUX_7;
        case kAudioChannelLabel_HOA_ACN_8:            return MA_CHANNEL_AUX_8;
        case kAudioChannelLabel_HOA_ACN_9:            return MA_CHANNEL_AUX_9;
        case kAudioChannelLabel_HOA_ACN_10:           return MA_CHANNEL_AUX_10;
        case kAudioChannelLabel_HOA_ACN_11:           return MA_CHANNEL_AUX_11;
        case kAudioChannelLabel_HOA_ACN_12:           return MA_CHANNEL_AUX_12;
        case kAudioChannelLabel_HOA_ACN_13:           return MA_CHANNEL_AUX_13;
        case kAudioChannelLabel_HOA_ACN_14:           return MA_CHANNEL_AUX_14;
        case kAudioChannelLabel_HOA_ACN_15:           return MA_CHANNEL_AUX_15;
        case kAudioChannelLabel_HOA_ACN_65024:        return MA_CHANNEL_NONE;
    #endif
        
        default:                                      return MA_CHANNEL_NONE;
    }
}

ma_result ma_format_from_AudioStreamBasicDescription(const AudioStreamBasicDescription* pDescription, ma_format* pFormatOut)
{
    ma_assert(pDescription != NULL);
    ma_assert(pFormatOut != NULL);
    
    *pFormatOut = ma_format_unknown;   /* Safety. */
    
    /* There's a few things miniaudio doesn't support. */
    if (pDescription->mFormatID != kAudioFormatLinearPCM) {
        return MA_FORMAT_NOT_SUPPORTED;
    }
    
    /* We don't support any non-packed formats that are aligned high. */
    if ((pDescription->mFormatFlags & kLinearPCMFormatFlagIsAlignedHigh) != 0) {
        return MA_FORMAT_NOT_SUPPORTED;
    }

    /* Only supporting native-endian. */
    if ((ma_is_little_endian() && (pDescription->mFormatFlags & kAudioFormatFlagIsBigEndian) != 0) || (ma_is_big_endian() && (pDescription->mFormatFlags & kAudioFormatFlagIsBigEndian) == 0)) {
        return MA_FORMAT_NOT_SUPPORTED;
    }
    
    /* We are not currently supporting non-interleaved formats (this will be added in a future version of miniaudio). */
    /*if ((pDescription->mFormatFlags & kAudioFormatFlagIsNonInterleaved) != 0) {
        return MA_FORMAT_NOT_SUPPORTED;
    }*/

    if ((pDescription->mFormatFlags & kLinearPCMFormatFlagIsFloat) != 0) {
        if (pDescription->mBitsPerChannel == 32) {
            *pFormatOut = ma_format_f32;
            return MA_SUCCESS;
        }
    } else {
        if ((pDescription->mFormatFlags & kLinearPCMFormatFlagIsSignedInteger) != 0) {
            if (pDescription->mBitsPerChannel == 16) {
                *pFormatOut = ma_format_s16;
                return MA_SUCCESS;
            } else if (pDescription->mBitsPerChannel == 24) {
                if (pDescription->mBytesPerFrame == (pDescription->mBitsPerChannel/8 * pDescription->mChannelsPerFrame)) {
                    *pFormatOut = ma_format_s24;
                    return MA_SUCCESS;
                } else {
                    if (pDescription->mBytesPerFrame/pDescription->mChannelsPerFrame == sizeof(ma_int32)) {
                        /* TODO: Implement ma_format_s24_32. */
                        /**pFormatOut = ma_format_s24_32;*/
                        /*return MA_SUCCESS;*/
                        return MA_FORMAT_NOT_SUPPORTED;
                    }
                }
            } else if (pDescription->mBitsPerChannel == 32) {
                *pFormatOut = ma_format_s32;
                return MA_SUCCESS;
            }
        } else {
            if (pDescription->mBitsPerChannel == 8) {
                *pFormatOut = ma_format_u8;
                return MA_SUCCESS;
            }
        }
    }
    
    /* Getting here means the format is not supported. */
    return MA_FORMAT_NOT_SUPPORTED;
}

ma_result ma_get_channel_map_from_AudioChannelLayout(AudioChannelLayout* pChannelLayout, ma_channel channelMap[MA_MAX_CHANNELS])
{
    ma_assert(pChannelLayout != NULL);
    
    if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelDescriptions) {
        UInt32 iChannel;
        for (iChannel = 0; iChannel < pChannelLayout->mNumberChannelDescriptions; ++iChannel) {
            channelMap[iChannel] = ma_channel_from_AudioChannelLabel(pChannelLayout->mChannelDescriptions[iChannel].mChannelLabel);
        }
    } else
#if 0
    if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelBitmap) {
        /* This is the same kind of system that's used by Windows audio APIs. */
        UInt32 iChannel = 0;
        UInt32 iBit;
        AudioChannelBitmap bitmap = pChannelLayout->mChannelBitmap;
        for (iBit = 0; iBit < 32; ++iBit) {
            AudioChannelBitmap bit = bitmap & (1 << iBit);
            if (bit != 0) {
                channelMap[iChannel++] = ma_channel_from_AudioChannelBit(bit);
            }
        }
    } else
#endif
    {
        /*
        Need to use the tag to determine the channel map. For now I'm just assuming a default channel map, but later on this should
        be updated to determine the mapping based on the tag.
        */
        UInt32 channelCount = AudioChannelLayoutTag_GetNumberOfChannels(pChannelLayout->mChannelLayoutTag);
        switch (pChannelLayout->mChannelLayoutTag)
        {
            case kAudioChannelLayoutTag_Mono:
            case kAudioChannelLayoutTag_Stereo:
            case kAudioChannelLayoutTag_StereoHeadphones:
            case kAudioChannelLayoutTag_MatrixStereo:
            case kAudioChannelLayoutTag_MidSide:
            case kAudioChannelLayoutTag_XY:
            case kAudioChannelLayoutTag_Binaural:
            case kAudioChannelLayoutTag_Ambisonic_B_Format:
            {
                ma_get_standard_channel_map(ma_standard_channel_map_default, channelCount, channelMap);
            } break;
            
            case kAudioChannelLayoutTag_Octagonal:
            {
                channelMap[7] = MA_CHANNEL_SIDE_RIGHT;
                channelMap[6] = MA_CHANNEL_SIDE_LEFT;
            } /* Intentional fallthrough. */
            case kAudioChannelLayoutTag_Hexagonal:
            {
                channelMap[5] = MA_CHANNEL_BACK_CENTER;
            } /* Intentional fallthrough. */
            case kAudioChannelLayoutTag_Pentagonal:
            {
                channelMap[4] = MA_CHANNEL_FRONT_CENTER;
            } /* Intentional fallghrough. */
            case kAudioChannelLayoutTag_Quadraphonic:
            {
                channelMap[3] = MA_CHANNEL_BACK_RIGHT;
                channelMap[2] = MA_CHANNEL_BACK_LEFT;
                channelMap[1] = MA_CHANNEL_RIGHT;
                channelMap[0] = MA_CHANNEL_LEFT;
            } break;
            
            /* TODO: Add support for more tags here. */
        
            default:
            {
                ma_get_standard_channel_map(ma_standard_channel_map_default, channelCount, channelMap);
            } break;
        }
    }
    
    return MA_SUCCESS;
}


#if defined(MA_APPLE_DESKTOP)
ma_result ma_get_device_object_ids__coreaudio(ma_context* pContext, UInt32* pDeviceCount, AudioObjectID** ppDeviceObjectIDs) /* NOTE: Free the returned buffer with ma_free(). */
{
    AudioObjectPropertyAddress propAddressDevices;
    UInt32 deviceObjectsDataSize;
    OSStatus status;
    AudioObjectID* pDeviceObjectIDs;

    ma_assert(pContext != NULL);
    ma_assert(pDeviceCount != NULL);
    ma_assert(ppDeviceObjectIDs != NULL);

    /* Safety. */
    *pDeviceCount = 0;
    *ppDeviceObjectIDs = NULL;
    
    propAddressDevices.mSelector = kAudioHardwarePropertyDevices;
    propAddressDevices.mScope    = kAudioObjectPropertyScopeGlobal;
    propAddressDevices.mElement  = kAudioObjectPropertyElementMaster;

    status = ((ma_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(kAudioObjectSystemObject, &propAddressDevices, 0, NULL, &deviceObjectsDataSize);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    pDeviceObjectIDs = (AudioObjectID*)ma_malloc(deviceObjectsDataSize);
    if (pDeviceObjectIDs == NULL) {
        return MA_OUT_OF_MEMORY;
    }
    
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(kAudioObjectSystemObject, &propAddressDevices, 0, NULL, &deviceObjectsDataSize, pDeviceObjectIDs);
    if (status != noErr) {
        ma_free(pDeviceObjectIDs);
        return ma_result_from_OSStatus(status);
    }
    
    *pDeviceCount = deviceObjectsDataSize / sizeof(AudioObjectID);
    *ppDeviceObjectIDs = pDeviceObjectIDs;

    (void)pContext; /* Unused. */
    return MA_SUCCESS;
}

ma_result ma_get_AudioObject_uid_as_CFStringRef(ma_context* pContext, AudioObjectID objectID, CFStringRef* pUID)
{
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;

    ma_assert(pContext != NULL);

    propAddress.mSelector = kAudioDevicePropertyDeviceUID;
    propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;

    dataSize = sizeof(*pUID);
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(objectID, &propAddress, 0, NULL, &dataSize, pUID);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    return MA_SUCCESS;
}

ma_result ma_get_AudioObject_uid(ma_context* pContext, AudioObjectID objectID, size_t bufferSize, char* bufferOut)
{
    CFStringRef uid;
    ma_result result;

    ma_assert(pContext != NULL);

    result = ma_get_AudioObject_uid_as_CFStringRef(pContext, objectID, &uid);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    if (!((ma_CFStringGetCString_proc)pContext->coreaudio.CFStringGetCString)(uid, bufferOut, bufferSize, kCFStringEncodingUTF8)) {
        return MA_ERROR;
    }
    
    return MA_SUCCESS;
}

ma_result ma_get_AudioObject_name(ma_context* pContext, AudioObjectID objectID, size_t bufferSize, char* bufferOut)
{
    AudioObjectPropertyAddress propAddress;
    CFStringRef deviceName = NULL;
    UInt32 dataSize;
    OSStatus status;

    ma_assert(pContext != NULL);

    propAddress.mSelector = kAudioDevicePropertyDeviceNameCFString;
    propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;

    dataSize = sizeof(deviceName);
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(objectID, &propAddress, 0, NULL, &dataSize, &deviceName);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    if (!((ma_CFStringGetCString_proc)pContext->coreaudio.CFStringGetCString)(deviceName, bufferOut, bufferSize, kCFStringEncodingUTF8)) {
        return MA_ERROR;
    }
    
    return MA_SUCCESS;
}

ma_bool32 ma_does_AudioObject_support_scope(ma_context* pContext, AudioObjectID deviceObjectID, AudioObjectPropertyScope scope)
{
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;
    AudioBufferList* pBufferList;
    ma_bool32 isSupported;

    ma_assert(pContext != NULL);

    /* To know whether or not a device is an input device we need ot look at the stream configuration. If it has an output channel it's a playback device. */
    propAddress.mSelector = kAudioDevicePropertyStreamConfiguration;
    propAddress.mScope    = scope;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    status = ((ma_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return MA_FALSE;
    }
    
    pBufferList = (AudioBufferList*)ma_malloc(dataSize);
    if (pBufferList == NULL) {
        return MA_FALSE;   /* Out of memory. */
    }
    
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pBufferList);
    if (status != noErr) {
        ma_free(pBufferList);
        return MA_FALSE;
    }

    isSupported = MA_FALSE;
    if (pBufferList->mNumberBuffers > 0) {
        isSupported = MA_TRUE;
    }
    
    ma_free(pBufferList);
    return isSupported;
}

ma_bool32 ma_does_AudioObject_support_playback(ma_context* pContext, AudioObjectID deviceObjectID)
{
    return ma_does_AudioObject_support_scope(pContext, deviceObjectID, kAudioObjectPropertyScopeOutput);
}

ma_bool32 ma_does_AudioObject_support_capture(ma_context* pContext, AudioObjectID deviceObjectID)
{
    return ma_does_AudioObject_support_scope(pContext, deviceObjectID, kAudioObjectPropertyScopeInput);
}


ma_result ma_get_AudioObject_stream_descriptions(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, UInt32* pDescriptionCount, AudioStreamRangedDescription** ppDescriptions) /* NOTE: Free the returned pointer with ma_free(). */
{
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;
    AudioStreamRangedDescription* pDescriptions;

    ma_assert(pContext != NULL);
    ma_assert(pDescriptionCount != NULL);
    ma_assert(ppDescriptions != NULL);
    
    /*
    TODO: Experiment with kAudioStreamPropertyAvailablePhysicalFormats instead of (or in addition to) kAudioStreamPropertyAvailableVirtualFormats. My
          MacBook Pro uses s24/32 format, however, which miniaudio does not currently support.
    */
    propAddress.mSelector = kAudioStreamPropertyAvailableVirtualFormats; /*kAudioStreamPropertyAvailablePhysicalFormats;*/
    propAddress.mScope    = (deviceType == ma_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    status = ((ma_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    pDescriptions = (AudioStreamRangedDescription*)ma_malloc(dataSize);
    if (pDescriptions == NULL) {
        return MA_OUT_OF_MEMORY;
    }
    
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pDescriptions);
    if (status != noErr) {
        ma_free(pDescriptions);
        return ma_result_from_OSStatus(status);
    }
    
    *pDescriptionCount = dataSize / sizeof(*pDescriptions);
    *ppDescriptions = pDescriptions;
    return MA_SUCCESS;
}


ma_result ma_get_AudioObject_channel_layout(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, AudioChannelLayout** ppChannelLayout)   /* NOTE: Free the returned pointer with ma_free(). */
{
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;
    AudioChannelLayout* pChannelLayout;

    ma_assert(pContext != NULL);
    ma_assert(ppChannelLayout != NULL);
    
    *ppChannelLayout = NULL;    /* Safety. */
    
    propAddress.mSelector = kAudioDevicePropertyPreferredChannelLayout;
    propAddress.mScope    = (deviceType == ma_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    status = ((ma_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    pChannelLayout = (AudioChannelLayout*)ma_malloc(dataSize);
    if (pChannelLayout == NULL) {
        return MA_OUT_OF_MEMORY;
    }
    
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pChannelLayout);
    if (status != noErr) {
        ma_free(pChannelLayout);
        return ma_result_from_OSStatus(status);
    }
    
    *ppChannelLayout = pChannelLayout;
    return MA_SUCCESS;
}

ma_result ma_get_AudioObject_channel_count(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_uint32* pChannelCount)
{
    AudioChannelLayout* pChannelLayout;
    ma_result result;

    ma_assert(pContext != NULL);
    ma_assert(pChannelCount != NULL);
    
    *pChannelCount = 0; /* Safety. */

    result = ma_get_AudioObject_channel_layout(pContext, deviceObjectID, deviceType, &pChannelLayout);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelDescriptions) {
        *pChannelCount = pChannelLayout->mNumberChannelDescriptions;
    } else if (pChannelLayout->mChannelLayoutTag == kAudioChannelLayoutTag_UseChannelBitmap) {
        *pChannelCount = ma_count_set_bits(pChannelLayout->mChannelBitmap);
    } else {
        *pChannelCount = AudioChannelLayoutTag_GetNumberOfChannels(pChannelLayout->mChannelLayoutTag);
    }
    
    ma_free(pChannelLayout);
    return MA_SUCCESS;
}

ma_result ma_get_AudioObject_channel_map(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_channel channelMap[MA_MAX_CHANNELS])
{
    AudioChannelLayout* pChannelLayout;
    ma_result result;

    ma_assert(pContext != NULL);
    
    result = ma_get_AudioObject_channel_layout(pContext, deviceObjectID, deviceType, &pChannelLayout);
    if (result != MA_SUCCESS) {
        return result;  /* Rather than always failing here, would it be more robust to simply assume a default? */
    }
    
    result = ma_get_channel_map_from_AudioChannelLayout(pChannelLayout, channelMap);
    if (result != MA_SUCCESS) {
        ma_free(pChannelLayout);
        return result;
    }
    
    ma_free(pChannelLayout);
    return result;
}

ma_result ma_get_AudioObject_sample_rates(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, UInt32* pSampleRateRangesCount, AudioValueRange** ppSampleRateRanges)   /* NOTE: Free the returned pointer with ma_free(). */
{
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;
    AudioValueRange* pSampleRateRanges;

    ma_assert(pContext != NULL);
    ma_assert(pSampleRateRangesCount != NULL);
    ma_assert(ppSampleRateRanges != NULL);
  
    /* Safety. */
    *pSampleRateRangesCount = 0;
    *ppSampleRateRanges = NULL;
    
    propAddress.mSelector = kAudioDevicePropertyAvailableNominalSampleRates;
    propAddress.mScope    = (deviceType == ma_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    status = ((ma_AudioObjectGetPropertyDataSize_proc)pContext->coreaudio.AudioObjectGetPropertyDataSize)(deviceObjectID, &propAddress, 0, NULL, &dataSize);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    pSampleRateRanges = (AudioValueRange*)ma_malloc(dataSize);
    if (pSampleRateRanges == NULL) {
        return MA_OUT_OF_MEMORY;
    }
    
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, pSampleRateRanges);
    if (status != noErr) {
        ma_free(pSampleRateRanges);
        return ma_result_from_OSStatus(status);
    }
    
    *pSampleRateRangesCount = dataSize / sizeof(*pSampleRateRanges);
    *ppSampleRateRanges = pSampleRateRanges;
    return MA_SUCCESS;
}

ma_result ma_get_AudioObject_get_closest_sample_rate(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_uint32 sampleRateIn, ma_uint32* pSampleRateOut)
{
    UInt32 sampleRateRangeCount;
    AudioValueRange* pSampleRateRanges;
    ma_result result;

    ma_assert(pContext != NULL);
    ma_assert(pSampleRateOut != NULL);
    
    *pSampleRateOut = 0;    /* Safety. */
    
    result = ma_get_AudioObject_sample_rates(pContext, deviceObjectID, deviceType, &sampleRateRangeCount, &pSampleRateRanges);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    if (sampleRateRangeCount == 0) {
        ma_free(pSampleRateRanges);
        return MA_ERROR;   /* Should never hit this case should we? */
    }
    
    if (sampleRateIn == 0) {
        /* Search in order of miniaudio's preferred priority. */
        UInt32 iMALSampleRate;
        for (iMALSampleRate = 0; iMALSampleRate < ma_countof(g_maStandardSampleRatePriorities); ++iMALSampleRate) {
            ma_uint32 malSampleRate = g_maStandardSampleRatePriorities[iMALSampleRate];
            UInt32 iCASampleRate;
            for (iCASampleRate = 0; iCASampleRate < sampleRateRangeCount; ++iCASampleRate) {
                AudioValueRange caSampleRate = pSampleRateRanges[iCASampleRate];
                if (caSampleRate.mMinimum <= malSampleRate && caSampleRate.mMaximum >= malSampleRate) {
                    *pSampleRateOut = malSampleRate;
                    ma_free(pSampleRateRanges);
                    return MA_SUCCESS;
                }
            }
        }
        
        /*
        If we get here it means none of miniaudio's standard sample rates matched any of the supported sample rates from the device. In this
        case we just fall back to the first one reported by Core Audio.
        */
        ma_assert(sampleRateRangeCount > 0);
        
        *pSampleRateOut = pSampleRateRanges[0].mMinimum;
        ma_free(pSampleRateRanges);
        return MA_SUCCESS;
    } else {
        /* Find the closest match to this sample rate. */
        UInt32 currentAbsoluteDifference = INT32_MAX;
        UInt32 iCurrentClosestRange = (UInt32)-1;
        UInt32 iRange;
        for (iRange = 0; iRange < sampleRateRangeCount; ++iRange) {
            if (pSampleRateRanges[iRange].mMinimum <= sampleRateIn && pSampleRateRanges[iRange].mMaximum >= sampleRateIn) {
                *pSampleRateOut = sampleRateIn;
                ma_free(pSampleRateRanges);
                return MA_SUCCESS;
            } else {
                UInt32 absoluteDifference;
                if (pSampleRateRanges[iRange].mMinimum > sampleRateIn) {
                    absoluteDifference = pSampleRateRanges[iRange].mMinimum - sampleRateIn;
                } else {
                    absoluteDifference = sampleRateIn - pSampleRateRanges[iRange].mMaximum;
                }
                
                if (currentAbsoluteDifference > absoluteDifference) {
                    currentAbsoluteDifference = absoluteDifference;
                    iCurrentClosestRange = iRange;
                }
            }
        }
        
        ma_assert(iCurrentClosestRange != (UInt32)-1);
        
        *pSampleRateOut = pSampleRateRanges[iCurrentClosestRange].mMinimum;
        ma_free(pSampleRateRanges);
        return MA_SUCCESS;
    }
    
    /* Should never get here, but it would mean we weren't able to find any suitable sample rates. */
    /*ma_free(pSampleRateRanges);*/
    /*return MA_ERROR;*/
}


ma_result ma_get_AudioObject_closest_buffer_size_in_frames(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_uint32 bufferSizeInFramesIn, ma_uint32* pBufferSizeInFramesOut)
{
    AudioObjectPropertyAddress propAddress;
    AudioValueRange bufferSizeRange;
    UInt32 dataSize;
    OSStatus status;

    ma_assert(pContext != NULL);
    ma_assert(pBufferSizeInFramesOut != NULL);
    
    *pBufferSizeInFramesOut = 0;    /* Safety. */
    
    propAddress.mSelector = kAudioDevicePropertyBufferFrameSizeRange;
    propAddress.mScope    = (deviceType == ma_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;

    dataSize = sizeof(bufferSizeRange);
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, &bufferSizeRange);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    /* This is just a clamp. */
    if (bufferSizeInFramesIn < bufferSizeRange.mMinimum) {
        *pBufferSizeInFramesOut = (ma_uint32)bufferSizeRange.mMinimum;
    } else if (bufferSizeInFramesIn > bufferSizeRange.mMaximum) {
        *pBufferSizeInFramesOut = (ma_uint32)bufferSizeRange.mMaximum;
    } else {
        *pBufferSizeInFramesOut = bufferSizeInFramesIn;
    }

    return MA_SUCCESS;
}

ma_result ma_set_AudioObject_buffer_size_in_frames(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_uint32* pBufferSizeInOut)
{
    ma_uint32 chosenBufferSizeInFrames;
    AudioObjectPropertyAddress propAddress;
    UInt32 dataSize;
    OSStatus status;

    ma_assert(pContext != NULL);

    ma_result result = ma_get_AudioObject_closest_buffer_size_in_frames(pContext, deviceObjectID, deviceType, *pBufferSizeInOut, &chosenBufferSizeInFrames);
    if (result != MA_SUCCESS) {
        return result;
    }

    /* Try setting the size of the buffer... If this fails we just use whatever is currently set. */
    propAddress.mSelector = kAudioDevicePropertyBufferFrameSize;
    propAddress.mScope    = (deviceType == ma_device_type_playback) ? kAudioObjectPropertyScopeOutput : kAudioObjectPropertyScopeInput;
    propAddress.mElement  = kAudioObjectPropertyElementMaster;
    
    ((ma_AudioObjectSetPropertyData_proc)pContext->coreaudio.AudioObjectSetPropertyData)(deviceObjectID, &propAddress, 0, NULL, sizeof(chosenBufferSizeInFrames), &chosenBufferSizeInFrames);
    
    /* Get the actual size of the buffer. */
    dataSize = sizeof(*pBufferSizeInOut);
    status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(deviceObjectID, &propAddress, 0, NULL, &dataSize, &chosenBufferSizeInFrames);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    *pBufferSizeInOut = chosenBufferSizeInFrames;
    return MA_SUCCESS;
}


ma_result ma_find_AudioObjectID(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, AudioObjectID* pDeviceObjectID)
{
    ma_assert(pContext != NULL);
    ma_assert(pDeviceObjectID != NULL);

    /* Safety. */
    *pDeviceObjectID = 0;
    
    if (pDeviceID == NULL) {
        /* Default device. */
        AudioObjectPropertyAddress propAddressDefaultDevice;
        UInt32 defaultDeviceObjectIDSize = sizeof(AudioObjectID);
        AudioObjectID defaultDeviceObjectID;
        OSStatus status;

        propAddressDefaultDevice.mScope = kAudioObjectPropertyScopeGlobal;
        propAddressDefaultDevice.mElement = kAudioObjectPropertyElementMaster;
        if (deviceType == ma_device_type_playback) {
            propAddressDefaultDevice.mSelector = kAudioHardwarePropertyDefaultOutputDevice;
        } else {
            propAddressDefaultDevice.mSelector = kAudioHardwarePropertyDefaultInputDevice;
        }
        
        defaultDeviceObjectIDSize = sizeof(AudioObjectID);
        status = ((ma_AudioObjectGetPropertyData_proc)pContext->coreaudio.AudioObjectGetPropertyData)(kAudioObjectSystemObject, &propAddressDefaultDevice, 0, NULL, &defaultDeviceObjectIDSize, &defaultDeviceObjectID);
        if (status == noErr) {
            *pDeviceObjectID = defaultDeviceObjectID;
            return MA_SUCCESS;
        }
    } else {
        /* Explicit device. */
        UInt32 deviceCount;
        AudioObjectID* pDeviceObjectIDs;
        ma_result result;
        UInt32 iDevice;

        result = ma_get_device_object_ids__coreaudio(pContext, &deviceCount, &pDeviceObjectIDs);
        if (result != MA_SUCCESS) {
            return result;
        }
        
        for (iDevice = 0; iDevice < deviceCount; ++iDevice) {
            AudioObjectID deviceObjectID = pDeviceObjectIDs[iDevice];
            
            char uid[256];
            if (ma_get_AudioObject_uid(pContext, deviceObjectID, sizeof(uid), uid) != MA_SUCCESS) {
                continue;
            }
            
            if (deviceType == ma_device_type_playback) {
                if (ma_does_AudioObject_support_playback(pContext, deviceObjectID)) {
                    if (strcmp(uid, pDeviceID->coreaudio) == 0) {
                        *pDeviceObjectID = deviceObjectID;
                        return MA_SUCCESS;
                    }
                }
            } else {
                if (ma_does_AudioObject_support_capture(pContext, deviceObjectID)) {
                    if (strcmp(uid, pDeviceID->coreaudio) == 0) {
                        *pDeviceObjectID = deviceObjectID;
                        return MA_SUCCESS;
                    }
                }
            }
        }
    }
    
    /* If we get here it means we couldn't find the device. */
    return MA_NO_DEVICE;
}


ma_result ma_find_best_format__coreaudio(ma_context* pContext, AudioObjectID deviceObjectID, ma_device_type deviceType, ma_format format, ma_uint32 channels, ma_uint32 sampleRate, ma_bool32 usingDefaultFormat, ma_bool32 usingDefaultChannels, ma_bool32 usingDefaultSampleRate, AudioStreamBasicDescription* pFormat)
{
    UInt32 deviceFormatDescriptionCount;
    AudioStreamRangedDescription* pDeviceFormatDescriptions;
    ma_result result;
    ma_uint32 desiredSampleRate;
    ma_uint32 desiredChannelCount;
    ma_format desiredFormat;
    AudioStreamBasicDescription bestDeviceFormatSoFar;
    ma_bool32 hasSupportedFormat;
    UInt32 iFormat;

    result = ma_get_AudioObject_stream_descriptions(pContext, deviceObjectID, deviceType, &deviceFormatDescriptionCount, &pDeviceFormatDescriptions);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    desiredSampleRate = sampleRate;
    if (usingDefaultSampleRate) {
        /*
        When using the device's default sample rate, we get the highest priority standard rate supported by the device. Otherwise
        we just use the pre-set rate.
        */
        ma_uint32 iStandardRate;
        for (iStandardRate = 0; iStandardRate < ma_countof(g_maStandardSampleRatePriorities); ++iStandardRate) {
            ma_uint32 standardRate = g_maStandardSampleRatePriorities[iStandardRate];
            ma_bool32 foundRate = MA_FALSE;
            UInt32 iDeviceRate;

            for (iDeviceRate = 0; iDeviceRate < deviceFormatDescriptionCount; ++iDeviceRate) {
                ma_uint32 deviceRate = (ma_uint32)pDeviceFormatDescriptions[iDeviceRate].mFormat.mSampleRate;
                
                if (deviceRate == standardRate) {
                    desiredSampleRate = standardRate;
                    foundRate = MA_TRUE;
                    break;
                }
            }
            
            if (foundRate) {
                break;
            }
        }
    }
    
    desiredChannelCount = channels;
    if (usingDefaultChannels) {
        ma_get_AudioObject_channel_count(pContext, deviceObjectID, deviceType, &desiredChannelCount);    /* <-- Not critical if this fails. */
    }
    
    desiredFormat = format;
    if (usingDefaultFormat) {
        desiredFormat = g_maFormatPriorities[0];
    }
    
    /*
    If we get here it means we don't have an exact match to what the client is asking for. We'll need to find the closest one. The next
    loop will check for formats that have the same sample rate to what we're asking for. If there is, we prefer that one in all cases.
    */
    ma_zero_object(&bestDeviceFormatSoFar);
    
    hasSupportedFormat = MA_FALSE;
    for (iFormat = 0; iFormat < deviceFormatDescriptionCount; ++iFormat) {
        ma_format format;
        ma_result formatResult = ma_format_from_AudioStreamBasicDescription(&pDeviceFormatDescriptions[iFormat].mFormat, &format);
        if (formatResult == MA_SUCCESS && format != ma_format_unknown) {
            hasSupportedFormat = MA_TRUE;
            bestDeviceFormatSoFar = pDeviceFormatDescriptions[iFormat].mFormat;
            break;
        }
    }
    
    if (!hasSupportedFormat) {
        return MA_FORMAT_NOT_SUPPORTED;
    }
    
    
    for (iFormat = 0; iFormat < deviceFormatDescriptionCount; ++iFormat) {
        AudioStreamBasicDescription thisDeviceFormat = pDeviceFormatDescriptions[iFormat].mFormat;
        ma_format thisSampleFormat;
        ma_result formatResult;
        ma_format bestSampleFormatSoFar;

        /* If the format is not supported by miniaudio we need to skip this one entirely. */
        formatResult = ma_format_from_AudioStreamBasicDescription(&pDeviceFormatDescriptions[iFormat].mFormat, &thisSampleFormat);
        if (formatResult != MA_SUCCESS || thisSampleFormat == ma_format_unknown) {
            continue;   /* The format is not supported by miniaudio. Skip. */
        }
        
        ma_format_from_AudioStreamBasicDescription(&bestDeviceFormatSoFar, &bestSampleFormatSoFar);
        
        /* Getting here means the format is supported by miniaudio which makes this format a candidate. */
        if (thisDeviceFormat.mSampleRate != desiredSampleRate) {
            /*
            The sample rate does not match, but this format could still be usable, although it's a very low priority. If the best format
            so far has an equal sample rate we can just ignore this one.
            */
            if (bestDeviceFormatSoFar.mSampleRate == desiredSampleRate) {
                continue;   /* The best sample rate so far has the same sample rate as what we requested which means it's still the best so far. Skip this format. */
            } else {
                /* In this case, neither the best format so far nor this one have the same sample rate. Check the channel count next. */
                if (thisDeviceFormat.mChannelsPerFrame != desiredChannelCount) {
                    /* This format has a different sample rate _and_ a different channel count. */
                    if (bestDeviceFormatSoFar.mChannelsPerFrame == desiredChannelCount) {
                        continue;   /* No change to the best format. */
                    } else {
                        /*
                        Both this format and the best so far have different sample rates and different channel counts. Whichever has the
                        best format is the new best.
                        */
                        if (ma_get_format_priority_index(thisSampleFormat) < ma_get_format_priority_index(bestSampleFormatSoFar)) {
                            bestDeviceFormatSoFar = thisDeviceFormat;
                            continue;
                        } else {
                            continue;   /* No change to the best format. */
                        }
                    }
                } else {
                    /* This format has a different sample rate but the desired channel count. */
                    if (bestDeviceFormatSoFar.mChannelsPerFrame == desiredChannelCount) {
                        /* Both this format and the best so far have the desired channel count. Whichever has the best format is the new best. */
                        if (ma_get_format_priority_index(thisSampleFormat) < ma_get_format_priority_index(bestSampleFormatSoFar)) {
                            bestDeviceFormatSoFar = thisDeviceFormat;
                            continue;
                        } else {
                            continue;   /* No change to the best format for now. */
                        }
                    } else {
                        /* This format has the desired channel count, but the best so far does not. We have a new best. */
                        bestDeviceFormatSoFar = thisDeviceFormat;
                        continue;
                    }
                }
            }
        } else {
            /*
            The sample rates match which makes this format a very high priority contender. If the best format so far has a different
            sample rate it needs to be replaced with this one.
            */
            if (bestDeviceFormatSoFar.mSampleRate != desiredSampleRate) {
                bestDeviceFormatSoFar = thisDeviceFormat;
                continue;
            } else {
                /* In this case both this format and the best format so far have the same sample rate. Check the channel count next. */
                if (thisDeviceFormat.mChannelsPerFrame == desiredChannelCount) {
                    /*
                    In this case this format has the same channel count as what the client is requesting. If the best format so far has
                    a different count, this one becomes the new best.
                    */
                    if (bestDeviceFormatSoFar.mChannelsPerFrame != desiredChannelCount) {
                        bestDeviceFormatSoFar = thisDeviceFormat;
                        continue;
                    } else {
                        /* In this case both this format and the best so far have the ideal sample rate and channel count. Check the format. */
                        if (thisSampleFormat == desiredFormat) {
                            bestDeviceFormatSoFar = thisDeviceFormat;
                            break;  /* Found the exact match. */
                        } else {
                            /* The formats are different. The new best format is the one with the highest priority format according to miniaudio. */
                            if (ma_get_format_priority_index(thisSampleFormat) < ma_get_format_priority_index(bestSampleFormatSoFar)) {
                                bestDeviceFormatSoFar = thisDeviceFormat;
                                continue;
                            } else {
                                continue;   /* No change to the best format for now. */
                            }
                        }
                    }
                } else {
                    /*
                    In this case the channel count is different to what the client has requested. If the best so far has the same channel
                    count as the requested count then it remains the best.
                    */
                    if (bestDeviceFormatSoFar.mChannelsPerFrame == desiredChannelCount) {
                        continue;
                    } else {
                        /*
                        This is the case where both have the same sample rate (good) but different channel counts. Right now both have about
                        the same priority, but we need to compare the format now.
                        */
                        if (thisSampleFormat == bestSampleFormatSoFar) {
                            if (ma_get_format_priority_index(thisSampleFormat) < ma_get_format_priority_index(bestSampleFormatSoFar)) {
                                bestDeviceFormatSoFar = thisDeviceFormat;
                                continue;
                            } else {
                                continue;   /* No change to the best format for now. */
                            }
                        }
                    }
                }
            }
        }
    }
    
    *pFormat = bestDeviceFormatSoFar;
    return MA_SUCCESS;
}
#endif

ma_result ma_get_AudioUnit_channel_map(ma_context* pContext, AudioUnit audioUnit, ma_device_type deviceType, ma_channel channelMap[MA_MAX_CHANNELS])
{
    AudioUnitScope deviceScope;
    AudioUnitElement deviceBus;
    UInt32 channelLayoutSize;
    OSStatus status;
    AudioChannelLayout* pChannelLayout;
    ma_result result;

    ma_assert(pContext != NULL);
    
    if (deviceType == ma_device_type_playback) {
        deviceScope = kAudioUnitScope_Output;
        deviceBus = MA_COREAUDIO_OUTPUT_BUS;
    } else {
        deviceScope = kAudioUnitScope_Input;
        deviceBus = MA_COREAUDIO_INPUT_BUS;
    }
    
    status = ((ma_AudioUnitGetPropertyInfo_proc)pContext->coreaudio.AudioUnitGetPropertyInfo)(audioUnit, kAudioUnitProperty_AudioChannelLayout, deviceScope, deviceBus, &channelLayoutSize, NULL);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    pChannelLayout = (AudioChannelLayout*)ma_malloc(channelLayoutSize);
    if (pChannelLayout == NULL) {
        return MA_OUT_OF_MEMORY;
    }
    
    status = ((ma_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(audioUnit, kAudioUnitProperty_AudioChannelLayout, deviceScope, deviceBus, pChannelLayout, &channelLayoutSize);
    if (status != noErr) {
        ma_free(pChannelLayout);
        return ma_result_from_OSStatus(status);
    }
    
    result = ma_get_channel_map_from_AudioChannelLayout(pChannelLayout, channelMap);
    if (result != MA_SUCCESS) {
        ma_free(pChannelLayout);
        return result;
    }

    ma_free(pChannelLayout);
    return MA_SUCCESS;
}

ma_bool32 ma_context_is_device_id_equal__coreaudio(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return strcmp(pID0->coreaudio, pID1->coreaudio) == 0;
}

ma_result ma_context_enumerate_devices__coreaudio(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
#if defined(MA_APPLE_DESKTOP)
    UInt32 deviceCount;
    AudioObjectID* pDeviceObjectIDs;
    ma_result result;
    UInt32 iDevice;

    result = ma_get_device_object_ids__coreaudio(pContext, &deviceCount, &pDeviceObjectIDs);
    if (result != MA_SUCCESS) {
        return result;
    }
  
    for (iDevice = 0; iDevice < deviceCount; ++iDevice) {
        AudioObjectID deviceObjectID = pDeviceObjectIDs[iDevice];
        ma_device_info info;

        ma_zero_object(&info);
        if (ma_get_AudioObject_uid(pContext, deviceObjectID, sizeof(info.id.coreaudio), info.id.coreaudio) != MA_SUCCESS) {
            continue;
        }
        if (ma_get_AudioObject_name(pContext, deviceObjectID, sizeof(info.name), info.name) != MA_SUCCESS) {
            continue;
        }

        if (ma_does_AudioObject_support_playback(pContext, deviceObjectID)) {
            if (!callback(pContext, ma_device_type_playback, &info, pUserData)) {
                break;
            }
        }
        if (ma_does_AudioObject_support_capture(pContext, deviceObjectID)) {
            if (!callback(pContext, ma_device_type_capture, &info, pUserData)) {
                break;
            }
        }
    }
    
    ma_free(pDeviceObjectIDs);
#else
    /* Only supporting default devices on non-Desktop platforms. */
    ma_device_info info;
    
    ma_zero_object(&info);
    ma_strncpy_s(info.name, sizeof(info.name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    if (!callback(pContext, ma_device_type_playback, &info, pUserData)) {
        return MA_SUCCESS;
    }
    
    ma_zero_object(&info);
    ma_strncpy_s(info.name, sizeof(info.name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    if (!callback(pContext, ma_device_type_capture, &info, pUserData)) {
        return MA_SUCCESS;
    }
#endif
    
    return MA_SUCCESS;
}

ma_result ma_context_get_device_info__coreaudio(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_result result;

    ma_assert(pContext != NULL);

    /* No exclusive mode with the Core Audio backend for now. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }
    
#if defined(MA_APPLE_DESKTOP)
    /* Desktop */
    {
        AudioObjectID deviceObjectID;
        UInt32 streamDescriptionCount;
        AudioStreamRangedDescription* pStreamDescriptions;
        UInt32 iStreamDescription;
        UInt32 sampleRateRangeCount;
        AudioValueRange* pSampleRateRanges;

        result = ma_find_AudioObjectID(pContext, deviceType, pDeviceID, &deviceObjectID);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        result = ma_get_AudioObject_uid(pContext, deviceObjectID, sizeof(pDeviceInfo->id.coreaudio), pDeviceInfo->id.coreaudio);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        result = ma_get_AudioObject_name(pContext, deviceObjectID, sizeof(pDeviceInfo->name), pDeviceInfo->name);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        /* Formats. */
        result = ma_get_AudioObject_stream_descriptions(pContext, deviceObjectID, deviceType, &streamDescriptionCount, &pStreamDescriptions);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        for (iStreamDescription = 0; iStreamDescription < streamDescriptionCount; ++iStreamDescription) {
            ma_format format;
            ma_bool32 formatExists = MA_FALSE;

            result = ma_format_from_AudioStreamBasicDescription(&pStreamDescriptions[iStreamDescription].mFormat, &format);
            if (result != MA_SUCCESS) {
                continue;
            }
        
            ma_assert(format != ma_format_unknown);
        
            /* Make sure the format isn't already in the output list. */
            for (ma_uint32 iOutputFormat = 0; iOutputFormat < pDeviceInfo->formatCount; ++iOutputFormat) {
                if (pDeviceInfo->formats[iOutputFormat] == format) {
                    formatExists = MA_TRUE;
                    break;
                }
            }
        
            if (!formatExists) {
                pDeviceInfo->formats[pDeviceInfo->formatCount++] = format;
            }
        }
    
        ma_free(pStreamDescriptions);
    
    
        /* Channels. */
        result = ma_get_AudioObject_channel_count(pContext, deviceObjectID, deviceType, &pDeviceInfo->minChannels);
        if (result != MA_SUCCESS) {
            return result;
        }
        pDeviceInfo->maxChannels = pDeviceInfo->minChannels;
    
    
        /* Sample rates. */
        result = ma_get_AudioObject_sample_rates(pContext, deviceObjectID, deviceType, &sampleRateRangeCount, &pSampleRateRanges);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        if (sampleRateRangeCount > 0) {
            pDeviceInfo->minSampleRate = UINT32_MAX;
            pDeviceInfo->maxSampleRate = 0;
            for (UInt32 iSampleRate = 0; iSampleRate < sampleRateRangeCount; ++iSampleRate) {
                if (pDeviceInfo->minSampleRate > pSampleRateRanges[iSampleRate].mMinimum) {
                    pDeviceInfo->minSampleRate = pSampleRateRanges[iSampleRate].mMinimum;
                }
                if (pDeviceInfo->maxSampleRate < pSampleRateRanges[iSampleRate].mMaximum) {
                    pDeviceInfo->maxSampleRate = pSampleRateRanges[iSampleRate].mMaximum;
                }
            }
        }
    }
#else
    /* Mobile */
    {
        AudioComponentDescription desc;
        AudioComponent component;
        AudioUnit audioUnit;
        OSStatus status;
        AudioUnitScope formatScope;
        AudioUnitElement formatElement;
        AudioStreamBasicDescription bestFormat;
        UInt32 propSize;

        if (deviceType == ma_device_type_playback) {
            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        } else {
            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        }
    
        /*
        Retrieving device information is more annoying on mobile than desktop. For simplicity I'm locking this down to whatever format is
        reported on a temporary I/O unit. The problem, however, is that this doesn't return a value for the sample rate which we need to
        retrieve from the AVAudioSession shared instance.
        */
        desc.componentType = kAudioUnitType_Output;
        desc.componentSubType = kAudioUnitSubType_RemoteIO;
        desc.componentManufacturer = kAudioUnitManufacturer_Apple;
        desc.componentFlags = 0;
        desc.componentFlagsMask = 0;
    
        component = ((ma_AudioComponentFindNext_proc)pContext->coreaudio.AudioComponentFindNext)(NULL, &desc);
        if (component == NULL) {
            return MA_FAILED_TO_INIT_BACKEND;
        }
    
        status = ((ma_AudioComponentInstanceNew_proc)pContext->coreaudio.AudioComponentInstanceNew)(component, &audioUnit);
        if (status != noErr) {
            return ma_result_from_OSStatus(status);
        }
    
        formatScope   = (deviceType == ma_device_type_playback) ? kAudioUnitScope_Input : kAudioUnitScope_Output;
        formatElement = (deviceType == ma_device_type_playback) ? MA_COREAUDIO_OUTPUT_BUS : MA_COREAUDIO_INPUT_BUS;
    
        propSize = sizeof(bestFormat);
        status = ((ma_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, &propSize);
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(audioUnit);
            return ma_result_from_OSStatus(status);
        }
    
        ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(audioUnit);
        audioUnit = NULL;
    
    
        pDeviceInfo->minChannels = bestFormat.mChannelsPerFrame;
        pDeviceInfo->maxChannels = bestFormat.mChannelsPerFrame;
    
        pDeviceInfo->formatCount = 1;
        result = ma_format_from_AudioStreamBasicDescription(&bestFormat, &pDeviceInfo->formats[0]);
        if (result != MA_SUCCESS) {
            return result;
        }
    
        /*
        It looks like Apple are wanting to push the whole AVAudioSession thing. Thus, we need to use that to determine device settings. To do
        this we just get the shared instance and inspect.
        */
        @autoreleasepool {
            AVAudioSession* pAudioSession = [AVAudioSession sharedInstance];
            ma_assert(pAudioSession != NULL);

            pDeviceInfo->minSampleRate = (ma_uint32)pAudioSession.sampleRate;
            pDeviceInfo->maxSampleRate = pDeviceInfo->minSampleRate;
        }
    }
#endif
    
    (void)pDeviceInfo; /* Unused. */
    return MA_SUCCESS;
}


void ma_device_uninit__coreaudio(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);
    ma_assert(ma_device__get_state(pDevice) == MA_STATE_UNINITIALIZED);
    
    if (pDevice->coreaudio.audioUnitCapture != NULL) {
        ((ma_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
    }
    if (pDevice->coreaudio.audioUnitPlayback != NULL) {
        ((ma_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
    }
    
    if (pDevice->coreaudio.pAudioBufferList) {
        ma_free(pDevice->coreaudio.pAudioBufferList);
    }

    if (pDevice->type == ma_device_type_duplex) {
        ma_pcm_rb_uninit(&pDevice->coreaudio.duplexRB);
    }
}


OSStatus ma_on_output__coreaudio(void* pUserData, AudioUnitRenderActionFlags* pActionFlags, const AudioTimeStamp* pTimeStamp, UInt32 busNumber, UInt32 frameCount, AudioBufferList* pBufferList)
{
    ma_device* pDevice = (ma_device*)pUserData;
    ma_stream_layout layout;

    ma_assert(pDevice != NULL);

#if defined(MA_DEBUG_OUTPUT)
    printf("INFO: Output Callback: busNumber=%d, frameCount=%d, mNumberBuffers=%d\n", busNumber, frameCount, pBufferList->mNumberBuffers);
#endif

    /* We need to check whether or not we are outputting interleaved or non-interleaved samples. The way we do this is slightly different for each type. */
    layout = ma_stream_layout_interleaved;
    if (pBufferList->mBuffers[0].mNumberChannels != pDevice->playback.internalChannels) {
        layout = ma_stream_layout_deinterleaved;
    }
    
    if (layout == ma_stream_layout_interleaved) {
        /* For now we can assume everything is interleaved. */
        UInt32 iBuffer;
        for (iBuffer = 0; iBuffer < pBufferList->mNumberBuffers; ++iBuffer) {
            if (pBufferList->mBuffers[iBuffer].mNumberChannels == pDevice->playback.internalChannels) {
                ma_uint32 frameCountForThisBuffer = pBufferList->mBuffers[iBuffer].mDataByteSize / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                if (frameCountForThisBuffer > 0) {
                    if (pDevice->type == ma_device_type_duplex) {
                        ma_device__handle_duplex_callback_playback(pDevice, frameCountForThisBuffer, pBufferList->mBuffers[iBuffer].mData, &pDevice->coreaudio.duplexRB);
                    } else {
                        ma_device__read_frames_from_client(pDevice, frameCountForThisBuffer, pBufferList->mBuffers[iBuffer].mData);
                    }
                }
                
            #if defined(MA_DEBUG_OUTPUT)
                printf("  frameCount=%d, mNumberChannels=%d, mDataByteSize=%d\n", frameCount, pBufferList->mBuffers[iBuffer].mNumberChannels, pBufferList->mBuffers[iBuffer].mDataByteSize);
            #endif
            } else {
                /*
                This case is where the number of channels in the output buffer do not match our internal channels. It could mean that it's
                not interleaved, in which case we can't handle right now since miniaudio does not yet support non-interleaved streams. We just
                output silence here.
                */
                ma_zero_memory(pBufferList->mBuffers[iBuffer].mData, pBufferList->mBuffers[iBuffer].mDataByteSize);

            #if defined(MA_DEBUG_OUTPUT)
                printf("  WARNING: Outputting silence. frameCount=%d, mNumberChannels=%d, mDataByteSize=%d\n", frameCount, pBufferList->mBuffers[iBuffer].mNumberChannels, pBufferList->mBuffers[iBuffer].mDataByteSize);
            #endif
            }
        }
    } else {
        /* This is the deinterleaved case. We need to update each buffer in groups of internalChannels. This assumes each buffer is the same size. */
        ma_uint8 tempBuffer[4096];
        UInt32 iBuffer;
        for (iBuffer = 0; iBuffer < pBufferList->mNumberBuffers; iBuffer += pDevice->playback.internalChannels) {
            ma_uint32 frameCountPerBuffer = pBufferList->mBuffers[iBuffer].mDataByteSize / ma_get_bytes_per_sample(pDevice->playback.internalFormat);
            ma_uint32 framesRemaining = frameCountPerBuffer;

            while (framesRemaining > 0) {
                void* ppDeinterleavedBuffers[MA_MAX_CHANNELS];
                ma_uint32 framesToRead = sizeof(tempBuffer) / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);
                if (framesToRead > framesRemaining) {
                    framesToRead = framesRemaining;
                }
                
                if (pDevice->type == ma_device_type_duplex) {
                    ma_device__handle_duplex_callback_playback(pDevice, framesToRead, tempBuffer, &pDevice->coreaudio.duplexRB);
                } else {
                    ma_device__read_frames_from_client(pDevice, framesToRead, tempBuffer);
                }
                
                for (ma_uint32 iChannel = 0; iChannel < pDevice->playback.internalChannels; ++iChannel) {
                    ppDeinterleavedBuffers[iChannel] = (void*)ma_offset_ptr(pBufferList->mBuffers[iBuffer].mData, (frameCountPerBuffer - framesRemaining) * ma_get_bytes_per_sample(pDevice->playback.internalFormat));
                }
                
                ma_deinterleave_pcm_frames(pDevice->playback.internalFormat, pDevice->playback.internalChannels, framesToRead, tempBuffer, ppDeinterleavedBuffers);
                
                framesRemaining -= framesToRead;
            }
        }
    }
    
    (void)pActionFlags;
    (void)pTimeStamp;
    (void)busNumber;

    return noErr;
}

OSStatus ma_on_input__coreaudio(void* pUserData, AudioUnitRenderActionFlags* pActionFlags, const AudioTimeStamp* pTimeStamp, UInt32 busNumber, UInt32 frameCount, AudioBufferList* pUnusedBufferList)
{
    ma_device* pDevice = (ma_device*)pUserData;
    AudioBufferList* pRenderedBufferList;
    ma_stream_layout layout;
    OSStatus status;

    ma_assert(pDevice != NULL);
    
    pRenderedBufferList = (AudioBufferList*)pDevice->coreaudio.pAudioBufferList;
    ma_assert(pRenderedBufferList);
    
    /* We need to check whether or not we are outputting interleaved or non-interleaved samples. The way we do this is slightly different for each type. */
    layout = ma_stream_layout_interleaved;
    if (pRenderedBufferList->mBuffers[0].mNumberChannels != pDevice->capture.internalChannels) {
        layout = ma_stream_layout_deinterleaved;
    }
    
#if defined(MA_DEBUG_OUTPUT)
    printf("INFO: Input Callback: busNumber=%d, frameCount=%d, mNumberBuffers=%d\n", busNumber, frameCount, pRenderedBufferList->mNumberBuffers);
#endif
    
    status = ((ma_AudioUnitRender_proc)pDevice->pContext->coreaudio.AudioUnitRender)((AudioUnit)pDevice->coreaudio.audioUnitCapture, pActionFlags, pTimeStamp, busNumber, frameCount, pRenderedBufferList);
    if (status != noErr) {
    #if defined(MA_DEBUG_OUTPUT)
        printf("  ERROR: AudioUnitRender() failed with %d\n", status);
    #endif
        return status;
    }
    
    if (layout == ma_stream_layout_interleaved) {
        UInt32 iBuffer;
        for (iBuffer = 0; iBuffer < pRenderedBufferList->mNumberBuffers; ++iBuffer) {
            if (pRenderedBufferList->mBuffers[iBuffer].mNumberChannels == pDevice->capture.internalChannels) {
                if (pDevice->type == ma_device_type_duplex) {
                    ma_device__handle_duplex_callback_capture(pDevice, frameCount, pRenderedBufferList->mBuffers[iBuffer].mData, &pDevice->coreaudio.duplexRB);
                } else {
                    ma_device__send_frames_to_client(pDevice, frameCount, pRenderedBufferList->mBuffers[iBuffer].mData);
                }
            #if defined(MA_DEBUG_OUTPUT)
                printf("  mDataByteSize=%d\n", pRenderedBufferList->mBuffers[iBuffer].mDataByteSize);
            #endif
            } else {
                /*
                This case is where the number of channels in the output buffer do not match our internal channels. It could mean that it's
                not interleaved, in which case we can't handle right now since miniaudio does not yet support non-interleaved streams.
                */
                ma_uint8 silentBuffer[4096];
                ma_uint32 framesRemaining;
                
                ma_zero_memory(silentBuffer, sizeof(silentBuffer));
                
                framesRemaining = frameCount;
                while (framesRemaining > 0) {
                    ma_uint32 framesToSend = sizeof(silentBuffer) / ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);
                    if (framesToSend > framesRemaining) {
                        framesToSend = framesRemaining;
                    }
                    
                    if (pDevice->type == ma_device_type_duplex) {
                        ma_device__handle_duplex_callback_capture(pDevice, framesToSend, silentBuffer, &pDevice->coreaudio.duplexRB);
                    } else {
                        ma_device__send_frames_to_client(pDevice, framesToSend, silentBuffer);
                    }
                    
                    framesRemaining -= framesToSend;
                }
                
            #if defined(MA_DEBUG_OUTPUT)
                printf("  WARNING: Outputting silence. frameCount=%d, mNumberChannels=%d, mDataByteSize=%d\n", frameCount, pRenderedBufferList->mBuffers[iBuffer].mNumberChannels, pRenderedBufferList->mBuffers[iBuffer].mDataByteSize);
            #endif
            }
        }
    } else {
        /* This is the deinterleaved case. We need to interleave the audio data before sending it to the client. This assumes each buffer is the same size. */
        ma_uint8 tempBuffer[4096];
        UInt32 iBuffer;
        for (iBuffer = 0; iBuffer < pRenderedBufferList->mNumberBuffers; iBuffer += pDevice->capture.internalChannels) {
            ma_uint32 framesRemaining = frameCount;
            while (framesRemaining > 0) {
                void* ppDeinterleavedBuffers[MA_MAX_CHANNELS];
                ma_uint32 framesToSend = sizeof(tempBuffer) / ma_get_bytes_per_sample(pDevice->capture.internalFormat);
                if (framesToSend > framesRemaining) {
                    framesToSend = framesRemaining;
                }
                
                for (ma_uint32 iChannel = 0; iChannel < pDevice->capture.internalChannels; ++iChannel) {
                    ppDeinterleavedBuffers[iChannel] = (void*)ma_offset_ptr(pRenderedBufferList->mBuffers[iBuffer].mData, (frameCount - framesRemaining) * ma_get_bytes_per_sample(pDevice->capture.internalFormat));
                }
                
                ma_interleave_pcm_frames(pDevice->capture.internalFormat, pDevice->capture.internalChannels, framesToSend, (const void**)ppDeinterleavedBuffers, tempBuffer);

                if (pDevice->type == ma_device_type_duplex) {
                    ma_device__handle_duplex_callback_capture(pDevice, framesToSend, tempBuffer, &pDevice->coreaudio.duplexRB);
                } else {
                    ma_device__send_frames_to_client(pDevice, framesToSend, tempBuffer);
                }

                framesRemaining -= framesToSend;
            }
        }
    }

    (void)pActionFlags;
    (void)pTimeStamp;
    (void)busNumber;
    (void)frameCount;
    (void)pUnusedBufferList;

    return noErr;
}

void on_start_stop__coreaudio(void* pUserData, AudioUnit audioUnit, AudioUnitPropertyID propertyID, AudioUnitScope scope, AudioUnitElement element)
{
    ma_device* pDevice = (ma_device*)pUserData;
    ma_assert(pDevice != NULL);
    
    /*
    There's been a report of a deadlock here when triggered by ma_device_uninit(). It looks like
    AudioUnitGetProprty (called below) and AudioComponentInstanceDispose (called in ma_device_uninit)
    can try waiting on the same lock. I'm going to try working around this by not calling any Core
    Audio APIs in the callback when the device has been stopped or uninitialized.
    */
    if (ma_device__get_state(pDevice) == MA_STATE_UNINITIALIZED || ma_device__get_state(pDevice) == MA_STATE_STOPPING || ma_device__get_state(pDevice) == MA_STATE_STOPPED) {
        ma_stop_proc onStop = pDevice->onStop;
        if (onStop) {
            onStop(pDevice);
        }
        
        ma_event_signal(&pDevice->coreaudio.stopEvent);
    } else {
        UInt32 isRunning;
        UInt32 isRunningSize = sizeof(isRunning);
        OSStatus status = ((ma_AudioUnitGetProperty_proc)pDevice->pContext->coreaudio.AudioUnitGetProperty)(audioUnit, kAudioOutputUnitProperty_IsRunning, scope, element, &isRunning, &isRunningSize);
        if (status != noErr) {
            return; /* Don't really know what to do in this case... just ignore it, I suppose... */
        }
        
        if (!isRunning) {
            ma_stop_proc onStop;

            /*
            The stop event is a bit annoying in Core Audio because it will be called when we automatically switch the default device. Some scenarios to consider:
            
            1) When the device is unplugged, this will be called _before_ the default device change notification.
            2) When the device is changed via the default device change notification, this will be called _after_ the switch.
            
            For case #1, we just check if there's a new default device available. If so, we just ignore the stop event. For case #2 we check a flag.
            */
            if (((audioUnit == pDevice->coreaudio.audioUnitPlayback) && pDevice->coreaudio.isDefaultPlaybackDevice) ||
                ((audioUnit == pDevice->coreaudio.audioUnitCapture)  && pDevice->coreaudio.isDefaultCaptureDevice)) {
                /*
                It looks like the device is switching through an external event, such as the user unplugging the device or changing the default device
                via the operating system's sound settings. If we're re-initializing the device, we just terminate because we want the stopping of the
                device to be seamless to the client (we don't want them receiving the onStop event and thinking that the device has stopped when it
                hasn't!).
                */
                if (((audioUnit == pDevice->coreaudio.audioUnitPlayback) && pDevice->coreaudio.isSwitchingPlaybackDevice) ||
                    ((audioUnit == pDevice->coreaudio.audioUnitCapture)  && pDevice->coreaudio.isSwitchingCaptureDevice)) {
                    return;
                }
                
                /*
                Getting here means the device is not reinitializing which means it may have been unplugged. From what I can see, it looks like Core Audio
                will try switching to the new default device seamlessly. We need to somehow find a way to determine whether or not Core Audio will most
                likely be successful in switching to the new device.
                
                TODO: Try to predict if Core Audio will switch devices. If not, the onStop callback needs to be posted.
                */
                return;
            }
            
            /* Getting here means we need to stop the device. */
            onStop = pDevice->onStop;
            if (onStop) {
                onStop(pDevice);
            }
        }
    }

    (void)propertyID; /* Unused. */
}

#if defined(MA_APPLE_DESKTOP)
OSStatus ma_default_device_changed__coreaudio(AudioObjectID objectID, UInt32 addressCount, const AudioObjectPropertyAddress* pAddresses, void* pUserData)
{
    ma_device* pDevice = (ma_device*)pUserData;
    ma_assert(pDevice != NULL);
    
    /* Not sure if I really need to check this, but it makes me feel better. */
    if (addressCount == 0) {
        return noErr;
    }
    
    if (pAddresses[0].mSelector == kAudioHardwarePropertyDefaultOutputDevice) {
        ma_result reinitResult;

        pDevice->coreaudio.isSwitchingPlaybackDevice = MA_TRUE;
        reinitResult = ma_device_reinit_internal__coreaudio(pDevice, ma_device_type_playback, MA_TRUE);
        pDevice->coreaudio.isSwitchingPlaybackDevice = MA_FALSE;
        
        if (reinitResult == MA_SUCCESS) {
            ma_device__post_init_setup(pDevice, ma_device_type_playback);
            
            /* Restart the device if required. If this fails we need to stop the device entirely. */
            if (ma_device__get_state(pDevice) == MA_STATE_STARTED) {
                OSStatus status = ((ma_AudioOutputUnitStart_proc)pDevice->pContext->coreaudio.AudioOutputUnitStart)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
                if (status != noErr) {
                    if (pDevice->type == ma_device_type_duplex) {
                        ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
                    }
                    ma_device__set_state(pDevice, MA_STATE_STOPPED);
                }
            }
        }
    }
    
    if (pAddresses[0].mSelector == kAudioHardwarePropertyDefaultInputDevice) {
        ma_result reinitResult;

        pDevice->coreaudio.isSwitchingPlaybackDevice = MA_TRUE;
        reinitResult = ma_device_reinit_internal__coreaudio(pDevice, ma_device_type_capture, MA_TRUE);
        pDevice->coreaudio.isSwitchingPlaybackDevice = MA_FALSE;
        
        if (reinitResult == MA_SUCCESS) {
            ma_device__post_init_setup(pDevice, ma_device_type_capture);
            
            /* Restart the device if required. If this fails we need to stop the device entirely. */
            if (ma_device__get_state(pDevice) == MA_STATE_STARTED) {
                OSStatus status = ((ma_AudioOutputUnitStart_proc)pDevice->pContext->coreaudio.AudioOutputUnitStart)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
                if (status != noErr) {
                    if (pDevice->type == ma_device_type_duplex) {
                        ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
                    }
                    ma_device__set_state(pDevice, MA_STATE_STOPPED);
                }
            }
        }
    }
    
    (void)objectID; /* Unused. */
    return noErr;
}
#endif

typedef struct
{
    /* Input. */
    ma_format formatIn;
    ma_uint32 channelsIn;
    ma_uint32 sampleRateIn;
    ma_channel channelMapIn[MA_MAX_CHANNELS];
    ma_uint32 bufferSizeInFramesIn;
    ma_uint32 bufferSizeInMillisecondsIn;
    ma_uint32 periodsIn;
    ma_bool32 usingDefaultFormat;
    ma_bool32 usingDefaultChannels;
    ma_bool32 usingDefaultSampleRate;
    ma_bool32 usingDefaultChannelMap;
    ma_share_mode shareMode;
    ma_bool32 registerStopEvent;

    /* Output. */
#if defined(MA_APPLE_DESKTOP)
    AudioObjectID deviceObjectID;
#endif
    AudioComponent component;
    AudioUnit audioUnit;
    AudioBufferList* pAudioBufferList;  /* Only used for input devices. */
    ma_format formatOut;
    ma_uint32 channelsOut;
    ma_uint32 sampleRateOut;
    ma_channel channelMapOut[MA_MAX_CHANNELS];
    ma_uint32 bufferSizeInFramesOut;
    ma_uint32 periodsOut;
    char deviceName[256];
} ma_device_init_internal_data__coreaudio;

ma_result ma_device_init_internal__coreaudio(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_device_init_internal_data__coreaudio* pData, void* pDevice_DoNotReference)   /* <-- pDevice is typed as void* intentionally so as to avoid accidentally referencing it. */
{
    ma_result result;
    OSStatus status;
    UInt32 enableIOFlag;
    AudioStreamBasicDescription bestFormat;
    ma_uint32 actualBufferSizeInFrames;
    AURenderCallbackStruct callbackInfo;
#if defined(MA_APPLE_DESKTOP)
    AudioObjectID deviceObjectID;
#endif

    /* This API should only be used for a single device type: playback or capture. No full-duplex mode. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    ma_assert(pContext != NULL);
    ma_assert(deviceType == ma_device_type_playback || deviceType == ma_device_type_capture);

#if defined(MA_APPLE_DESKTOP)
    pData->deviceObjectID = 0;
#endif
    pData->component = NULL;
    pData->audioUnit = NULL;
    pData->pAudioBufferList = NULL;
    
    ma_result result;
    
#if defined(MA_APPLE_DESKTOP)
    result = ma_find_AudioObjectID(pContext, deviceType, pDeviceID, &deviceObjectID);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    pData->deviceObjectID = deviceObjectID;
#endif
    
    /* Core audio doesn't really use the notion of a period so we can leave this unmodified, but not too over the top. */
    pData->periodsOut = pData->periodsIn;
    if (pData->periodsOut < 1) {
        pData->periodsOut = 1;
    }
    if (pData->periodsOut > 16) {
        pData->periodsOut = 16;
    }
    
    
    /* Audio unit. */
    status = ((ma_AudioComponentInstanceNew_proc)pContext->coreaudio.AudioComponentInstanceNew)((AudioComponent)pContext->coreaudio.component, (AudioUnit*)&pData->audioUnit);
    if (status != noErr) {
        return ma_result_from_OSStatus(status);
    }
    
    
    /* The input/output buses need to be explicitly enabled and disabled. We set the flag based on the output unit first, then we just swap it for input. */
    enableIOFlag = 1;
    if (deviceType == ma_device_type_capture) {
        enableIOFlag = 0;
    }
    
    status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_EnableIO, kAudioUnitScope_Output, MA_COREAUDIO_OUTPUT_BUS, &enableIOFlag, sizeof(enableIOFlag));
    if (status != noErr) {
        ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return ma_result_from_OSStatus(status);
    }
    
    enableIOFlag = (enableIOFlag == 0) ? 1 : 0;
    status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_EnableIO, kAudioUnitScope_Input, MA_COREAUDIO_INPUT_BUS, &enableIOFlag, sizeof(enableIOFlag));
    if (status != noErr) {
        ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return ma_result_from_OSStatus(status);
    }
    
    
    /* Set the device to use with this audio unit. This is only used on desktop since we are using defaults on mobile. */
#if defined(MA_APPLE_DESKTOP)
    status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_CurrentDevice, kAudioUnitScope_Global, (deviceType == ma_device_type_playback) ? MA_COREAUDIO_OUTPUT_BUS : MA_COREAUDIO_INPUT_BUS, &deviceObjectID, sizeof(AudioDeviceID));
    if (status != noErr) {
        ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return ma_result_from_OSStatus(result);
    }
#endif
    
    /*
    Format. This is the hardest part of initialization because there's a few variables to take into account.
      1) The format must be supported by the device.
      2) The format must be supported miniaudio.
      3) There's a priority that miniaudio prefers.
    
    Ideally we would like to use a format that's as close to the hardware as possible so we can get as close to a passthrough as possible. The
    most important property is the sample rate. miniaudio can do format conversion for any sample rate and channel count, but cannot do the same
    for the sample data format. If the sample data format is not supported by miniaudio it must be ignored completely.
    
    On mobile platforms this is a bit different. We just force the use of whatever the audio unit's current format is set to.
    */
    {
        AudioUnitScope   formatScope   = (deviceType == ma_device_type_playback) ? kAudioUnitScope_Input : kAudioUnitScope_Output;
        AudioUnitElement formatElement = (deviceType == ma_device_type_playback) ? MA_COREAUDIO_OUTPUT_BUS : MA_COREAUDIO_INPUT_BUS;

    #if defined(MA_APPLE_DESKTOP)
        AudioStreamBasicDescription origFormat;
        UInt32 origFormatSize;

        result = ma_find_best_format__coreaudio(pContext, deviceObjectID, deviceType, pData->formatIn, pData->channelsIn, pData->sampleRateIn, pData->usingDefaultFormat, pData->usingDefaultChannels, pData->usingDefaultSampleRate, &bestFormat);
        if (result != MA_SUCCESS) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return result;
        }
        
        /* From what I can see, Apple's documentation implies that we should keep the sample rate consistent. */
        origFormatSize = sizeof(origFormat);
        if (deviceType == ma_device_type_playback) {
            status = ((ma_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Output, MA_COREAUDIO_OUTPUT_BUS, &origFormat, &origFormatSize);
        } else {
            status = ((ma_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Input, MA_COREAUDIO_INPUT_BUS, &origFormat, &origFormatSize);
        }
        
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return result;
        }
        
        bestFormat.mSampleRate = origFormat.mSampleRate;
        
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, sizeof(bestFormat));
        if (status != noErr) {
            /* We failed to set the format, so fall back to the current format of the audio unit. */
            bestFormat = origFormat;
        }
    #else
        UInt32 propSize = sizeof(bestFormat);
        status = ((ma_AudioUnitGetProperty_proc)pContext->coreaudio.AudioUnitGetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, &propSize);
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
        
        /*
        Sample rate is a little different here because for some reason kAudioUnitProperty_StreamFormat returns 0... Oh well. We need to instead try
        setting the sample rate to what the user has requested and then just see the results of it. Need to use some Objective-C here for this since
        it depends on Apple's AVAudioSession API. To do this we just get the shared AVAudioSession instance and then set it. Note that from what I
        can tell, it looks like the sample rate is shared between playback and capture for everything.
        */
        @autoreleasepool {
            AVAudioSession* pAudioSession = [AVAudioSession sharedInstance];
            ma_assert(pAudioSession != NULL);
            
            [pAudioSession setPreferredSampleRate:(double)pData->sampleRateIn error:nil];
            bestFormat.mSampleRate = pAudioSession.sampleRate;
        }
        
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_StreamFormat, formatScope, formatElement, &bestFormat, sizeof(bestFormat));
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
    #endif
        
        result = ma_format_from_AudioStreamBasicDescription(&bestFormat, &pData->formatOut);
        if (result != MA_SUCCESS) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return result;
        }
        
        if (pData->formatOut == ma_format_unknown) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return MA_FORMAT_NOT_SUPPORTED;
        }
        
        pData->channelsOut = bestFormat.mChannelsPerFrame;
        pData->sampleRateOut = bestFormat.mSampleRate;
    }
    
    /*
    Internal channel map. This is weird in my testing. If I use the AudioObject to get the
    channel map, the channel descriptions are set to "Unknown" for some reason. To work around
    this it looks like retrieving it from the AudioUnit will work. However, and this is where
    it gets weird, it doesn't seem to work with capture devices, nor at all on iOS... Therefore
    I'm going to fall back to a default assumption in these cases.
    */
#if defined(MA_APPLE_DESKTOP)
    result = ma_get_AudioUnit_channel_map(pContext, pData->audioUnit, deviceType, pData->channelMapOut);
    if (result != MA_SUCCESS) {
    #if 0
        /* Try falling back to the channel map from the AudioObject. */
        result = ma_get_AudioObject_channel_map(pContext, deviceObjectID, deviceType, pData->channelMapOut);
        if (result != MA_SUCCESS) {
            return result;
        }
    #else
        /* Fall back to default assumptions. */
        ma_get_standard_channel_map(ma_standard_channel_map_default, pData->channelsOut, pData->channelMapOut);
    #endif
    }
#else
    /* TODO: Figure out how to get the channel map using AVAudioSession. */
    ma_get_standard_channel_map(ma_standard_channel_map_default, pData->channelsOut, pData->channelMapOut);
#endif
    

    /* Buffer size. Not allowing this to be configurable on iOS. */
    actualBufferSizeInFrames = pData->bufferSizeInFramesIn;
    
#if defined(MA_APPLE_DESKTOP)
    if (actualBufferSizeInFrames == 0) {
        actualBufferSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(pData->bufferSizeInMillisecondsIn, pData->sampleRateOut);
    }
    
    actualBufferSizeInFrames = actualBufferSizeInFrames / pData->periodsOut;
    result = ma_set_AudioObject_buffer_size_in_frames(pContext, deviceObjectID, deviceType, &actualBufferSizeInFrames);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    pData->bufferSizeInFramesOut = actualBufferSizeInFrames * pData->periodsOut;
#else
    actualBufferSizeInFrames = 4096;
    pData->bufferSizeInFramesOut = actualBufferSizeInFrames;
#endif


    /*
    During testing I discovered that the buffer size can be too big. You'll get an error like this:
    
      kAudioUnitErr_TooManyFramesToProcess : inFramesToProcess=4096, mMaxFramesPerSlice=512
    
    Note how inFramesToProcess is smaller than mMaxFramesPerSlice. To fix, we need to set kAudioUnitProperty_MaximumFramesPerSlice to that
    of the size of our buffer, or do it the other way around and set our buffer size to the kAudioUnitProperty_MaximumFramesPerSlice.
    */
    {
        /*AudioUnitScope propScope = (deviceType == ma_device_type_playback) ? kAudioUnitScope_Input : kAudioUnitScope_Output;
        AudioUnitElement propBus = (deviceType == ma_device_type_playback) ? MA_COREAUDIO_OUTPUT_BUS : MA_COREAUDIO_INPUT_BUS;
    
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_MaximumFramesPerSlice, propScope, propBus, &actualBufferSizeInFrames, sizeof(actualBufferSizeInFrames));
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }*/
        
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_MaximumFramesPerSlice, kAudioUnitScope_Global, 0, &actualBufferSizeInFrames, sizeof(actualBufferSizeInFrames));
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
    }
    
    /* We need a buffer list if this is an input device. We render into this in the input callback. */
    if (deviceType == ma_device_type_capture) {
        ma_bool32 isInterleaved = (bestFormat.mFormatFlags & kAudioFormatFlagIsNonInterleaved) == 0;
        size_t allocationSize;
        AudioBufferList* pBufferList;

        allocationSize = sizeof(AudioBufferList) - sizeof(AudioBuffer);  /* Subtract sizeof(AudioBuffer) because that part is dynamically sized. */
        if (isInterleaved) {
            /* Interleaved case. This is the simple case because we just have one buffer. */
            allocationSize += sizeof(AudioBuffer) * 1;
            allocationSize += actualBufferSizeInFrames * ma_get_bytes_per_frame(pData->formatOut, pData->channelsOut);
        } else {
            /* Non-interleaved case. This is the more complex case because there's more than one buffer. */
            allocationSize += sizeof(AudioBuffer) * pData->channelsOut;
            allocationSize += actualBufferSizeInFrames * ma_get_bytes_per_sample(pData->formatOut) * pData->channelsOut;
        }
        
        pBufferList = (AudioBufferList*)ma_malloc(allocationSize);
        if (pBufferList == NULL) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return MA_OUT_OF_MEMORY;
        }
        
        if (isInterleaved) {
            pBufferList->mNumberBuffers = 1;
            pBufferList->mBuffers[0].mNumberChannels = pData->channelsOut;
            pBufferList->mBuffers[0].mDataByteSize = actualBufferSizeInFrames * ma_get_bytes_per_frame(pData->formatOut, pData->channelsOut);
            pBufferList->mBuffers[0].mData = (ma_uint8*)pBufferList + sizeof(AudioBufferList);
        } else {
            pBufferList->mNumberBuffers = pData->channelsOut;
            for (ma_uint32 iBuffer = 0; iBuffer < pBufferList->mNumberBuffers; ++iBuffer) {
                pBufferList->mBuffers[iBuffer].mNumberChannels = 1;
                pBufferList->mBuffers[iBuffer].mDataByteSize = actualBufferSizeInFrames * ma_get_bytes_per_sample(pData->formatOut);
                pBufferList->mBuffers[iBuffer].mData = (ma_uint8*)pBufferList + ((sizeof(AudioBufferList) - sizeof(AudioBuffer)) + (sizeof(AudioBuffer) * pData->channelsOut)) + (actualBufferSizeInFrames * ma_get_bytes_per_sample(pData->formatOut) * iBuffer);
            }
        }
        
        pData->pAudioBufferList = pBufferList;
    }
    
    /* Callbacks. */
    callbackInfo.inputProcRefCon = pDevice_DoNotReference;
    if (deviceType == ma_device_type_playback) {
        callbackInfo.inputProc = ma_on_output__coreaudio;
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioUnitProperty_SetRenderCallback, kAudioUnitScope_Global, MA_COREAUDIO_OUTPUT_BUS, &callbackInfo, sizeof(callbackInfo));
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
    } else {
        callbackInfo.inputProc = ma_on_input__coreaudio;
        status = ((ma_AudioUnitSetProperty_proc)pContext->coreaudio.AudioUnitSetProperty)(pData->audioUnit, kAudioOutputUnitProperty_SetInputCallback, kAudioUnitScope_Global, MA_COREAUDIO_INPUT_BUS, &callbackInfo, sizeof(callbackInfo));
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
    }
    
    /* We need to listen for stop events. */
    if (pData->registerStopEvent) {
        status = ((ma_AudioUnitAddPropertyListener_proc)pContext->coreaudio.AudioUnitAddPropertyListener)(pData->audioUnit, kAudioOutputUnitProperty_IsRunning, on_start_stop__coreaudio, pDevice_DoNotReference);
        if (status != noErr) {
            ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
            return ma_result_from_OSStatus(status);
        }
    }
    
    /* Initialize the audio unit. */
    status = ((ma_AudioUnitInitialize_proc)pContext->coreaudio.AudioUnitInitialize)(pData->audioUnit);
    if (status != noErr) {
        ma_free(pData->pAudioBufferList);
        pData->pAudioBufferList = NULL;
        ((ma_AudioComponentInstanceDispose_proc)pContext->coreaudio.AudioComponentInstanceDispose)(pData->audioUnit);
        return ma_result_from_OSStatus(status);
    }
    
    /* Grab the name. */
#if defined(MA_APPLE_DESKTOP)
    ma_get_AudioObject_name(pContext, deviceObjectID, sizeof(pData->deviceName), pData->deviceName);
#else
    if (deviceType == ma_device_type_playback) {
        ma_strcpy_s(pData->deviceName, sizeof(pData->deviceName), MA_DEFAULT_PLAYBACK_DEVICE_NAME);
    } else {
        ma_strcpy_s(pData->deviceName, sizeof(pData->deviceName), MA_DEFAULT_CAPTURE_DEVICE_NAME);
    }
#endif
    
    return result;
}

ma_result ma_device_reinit_internal__coreaudio(ma_device* pDevice, ma_device_type deviceType, ma_bool32 disposePreviousAudioUnit)
{
    ma_device_init_internal_data__coreaudio data;
    ma_result result;

    /* This should only be called for playback or capture, not duplex. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    if (deviceType == ma_device_type_capture) {
        data.formatIn               = pDevice->capture.format;
        data.channelsIn             = pDevice->capture.channels;
        data.sampleRateIn           = pDevice->sampleRate;
        ma_copy_memory(data.channelMapIn, pDevice->capture.channelMap, sizeof(pDevice->capture.channelMap));
        data.usingDefaultFormat     = pDevice->capture.usingDefaultFormat;
        data.usingDefaultChannels   = pDevice->capture.usingDefaultChannels;
        data.usingDefaultSampleRate = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap = pDevice->capture.usingDefaultChannelMap;
        data.shareMode              = pDevice->capture.shareMode;
        data.registerStopEvent      = MA_TRUE;
        
        if (disposePreviousAudioUnit) {
            ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
            ((ma_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
        }
        if (pDevice->coreaudio.pAudioBufferList) {
            ma_free(pDevice->coreaudio.pAudioBufferList);
        }
        
    #if defined(MA_APPLE_DESKTOP)
        pDevice->coreaudio.deviceObjectIDCapture = (ma_uint32)data.deviceObjectID;
    #endif
        pDevice->coreaudio.audioUnitCapture = (ma_ptr)data.audioUnit;
        pDevice->coreaudio.pAudioBufferList = (ma_ptr)data.pAudioBufferList;
    }
    if (deviceType == ma_device_type_playback) {
        data.formatIn               = pDevice->playback.format;
        data.channelsIn             = pDevice->playback.channels;
        data.sampleRateIn           = pDevice->sampleRate;
        ma_copy_memory(data.channelMapIn, pDevice->playback.channelMap, sizeof(pDevice->playback.channelMap));
        data.usingDefaultFormat     = pDevice->playback.usingDefaultFormat;
        data.usingDefaultChannels   = pDevice->playback.usingDefaultChannels;
        data.usingDefaultSampleRate = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap = pDevice->playback.usingDefaultChannelMap;
        data.shareMode              = pDevice->playback.shareMode;
        data.registerStopEvent      = (pDevice->type != ma_device_type_duplex);
        
        if (disposePreviousAudioUnit) {
            ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
            ((ma_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
        }
        
    #if defined(MA_APPLE_DESKTOP)
        pDevice->coreaudio.deviceObjectIDPlayback = (ma_uint32)data.deviceObjectID;
    #endif
        pDevice->coreaudio.audioUnitPlayback = (ma_ptr)data.audioUnit;
    }
    data.bufferSizeInFramesIn       = pDevice->coreaudio.originalBufferSizeInFrames;
    data.bufferSizeInMillisecondsIn = pDevice->coreaudio.originalBufferSizeInMilliseconds;
    data.periodsIn                  = pDevice->coreaudio.originalPeriods;

    result = ma_device_init_internal__coreaudio(pDevice->pContext, deviceType, NULL, &data, (void*)pDevice);
    if (result != MA_SUCCESS) {
        return result;
    }
    
    return MA_SUCCESS;
}


ma_result ma_device_init__coreaudio(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result;

    ma_assert(pContext != NULL);
    ma_assert(pConfig != NULL);
    ma_assert(pDevice != NULL);

    /* No exclusive mode with the Core Audio backend for now. */
    if (((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.shareMode  == ma_share_mode_exclusive) ||
        ((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.shareMode == ma_share_mode_exclusive)) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }
    
    /* Capture needs to be initialized first. */
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_device_init_internal_data__coreaudio data;
        data.formatIn                   = pConfig->capture.format;
        data.channelsIn                 = pConfig->capture.channels;
        data.sampleRateIn               = pConfig->sampleRate;
        ma_copy_memory(data.channelMapIn, pConfig->capture.channelMap, sizeof(pConfig->capture.channelMap));
        data.usingDefaultFormat         = pDevice->capture.usingDefaultFormat;
        data.usingDefaultChannels       = pDevice->capture.usingDefaultChannels;
        data.usingDefaultSampleRate     = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap     = pDevice->capture.usingDefaultChannelMap;
        data.shareMode                  = pConfig->capture.shareMode;
        data.bufferSizeInFramesIn       = pConfig->bufferSizeInFrames;
        data.bufferSizeInMillisecondsIn = pConfig->bufferSizeInMilliseconds;
        data.registerStopEvent          = MA_TRUE;
        
        result = ma_device_init_internal__coreaudio(pDevice->pContext, ma_device_type_capture, pConfig->capture.pDeviceID, &data, (void*)pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
        
        pDevice->coreaudio.isDefaultCaptureDevice   = (pConfig->capture.pDeviceID == NULL);
    #if defined(MA_APPLE_DESKTOP)
        pDevice->coreaudio.deviceObjectIDCapture    = (ma_uint32)data.deviceObjectID;
    #endif
        pDevice->coreaudio.audioUnitCapture         = (ma_ptr)data.audioUnit;
        pDevice->coreaudio.pAudioBufferList         = (ma_ptr)data.pAudioBufferList;
        
        pDevice->capture.internalFormat             = data.formatOut;
        pDevice->capture.internalChannels           = data.channelsOut;
        pDevice->capture.internalSampleRate         = data.sampleRateOut;
        ma_copy_memory(pDevice->capture.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->capture.internalBufferSizeInFrames = data.bufferSizeInFramesOut;
        pDevice->capture.internalPeriods            = data.periodsOut;
        
        /* TODO: This needs to be made global. */
    #if defined(MA_APPLE_DESKTOP)
        /*
        If we are using the default device we'll need to listen for changes to the system's default device so we can seemlessly
        switch the device in the background.
        */
        if (pConfig->capture.pDeviceID == NULL) {
            AudioObjectPropertyAddress propAddress;
            propAddress.mSelector = kAudioHardwarePropertyDefaultInputDevice;
            propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
            propAddress.mElement  = kAudioObjectPropertyElementMaster;
            ((ma_AudioObjectAddPropertyListener_proc)pDevice->pContext->coreaudio.AudioObjectAddPropertyListener)(kAudioObjectSystemObject, &propAddress, &ma_default_device_changed__coreaudio, pDevice);
        }
    #endif
    }
    
    /* Playback. */
    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_device_init_internal_data__coreaudio data;
        data.formatIn                   = pConfig->playback.format;
        data.channelsIn                 = pConfig->playback.channels;
        data.sampleRateIn               = pConfig->sampleRate;
        ma_copy_memory(data.channelMapIn, pConfig->playback.channelMap, sizeof(pConfig->playback.channelMap));
        data.usingDefaultFormat         = pDevice->playback.usingDefaultFormat;
        data.usingDefaultChannels       = pDevice->playback.usingDefaultChannels;
        data.usingDefaultSampleRate     = pDevice->usingDefaultSampleRate;
        data.usingDefaultChannelMap     = pDevice->playback.usingDefaultChannelMap;
        data.shareMode                  = pConfig->playback.shareMode;
        
        /* In full-duplex mode we want the playback buffer to be the same size as the capture buffer. */
        if (pConfig->deviceType == ma_device_type_duplex) {
            data.bufferSizeInFramesIn       = pDevice->capture.internalBufferSizeInFrames;
            data.periodsIn                  = pDevice->capture.internalPeriods;
            data.registerStopEvent          = MA_FALSE;
        } else {
            data.bufferSizeInFramesIn       = pConfig->bufferSizeInFrames;
            data.bufferSizeInMillisecondsIn = pConfig->bufferSizeInMilliseconds;
            data.periodsIn                  = pConfig->periods;
            data.registerStopEvent          = MA_TRUE;
        }
        
        result = ma_device_init_internal__coreaudio(pDevice->pContext, ma_device_type_playback, pConfig->playback.pDeviceID, &data, (void*)pDevice);
        if (result != MA_SUCCESS) {
            if (pConfig->deviceType == ma_device_type_duplex) {
                ((ma_AudioComponentInstanceDispose_proc)pDevice->pContext->coreaudio.AudioComponentInstanceDispose)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
                if (pDevice->coreaudio.pAudioBufferList) {
                    ma_free(pDevice->coreaudio.pAudioBufferList);
                }
            }
            return result;
        }
        
        pDevice->coreaudio.isDefaultPlaybackDevice   = (pConfig->playback.pDeviceID == NULL);
    #if defined(MA_APPLE_DESKTOP)
        pDevice->coreaudio.deviceObjectIDPlayback    = (ma_uint32)data.deviceObjectID;
    #endif
        pDevice->coreaudio.audioUnitPlayback         = (ma_ptr)data.audioUnit;
        
        pDevice->playback.internalFormat             = data.formatOut;
        pDevice->playback.internalChannels           = data.channelsOut;
        pDevice->playback.internalSampleRate         = data.sampleRateOut;
        ma_copy_memory(pDevice->playback.internalChannelMap, data.channelMapOut, sizeof(data.channelMapOut));
        pDevice->playback.internalBufferSizeInFrames = data.bufferSizeInFramesOut;
        pDevice->playback.internalPeriods            = data.periodsOut;
        
        /* TODO: This needs to be made global. */
    #if defined(MA_APPLE_DESKTOP)
        /*
        If we are using the default device we'll need to listen for changes to the system's default device so we can seemlessly
        switch the device in the background.
        */
        if (pConfig->playback.pDeviceID == NULL) {
            AudioObjectPropertyAddress propAddress;
            propAddress.mSelector = kAudioHardwarePropertyDefaultOutputDevice;
            propAddress.mScope    = kAudioObjectPropertyScopeGlobal;
            propAddress.mElement  = kAudioObjectPropertyElementMaster;
            ((ma_AudioObjectAddPropertyListener_proc)pDevice->pContext->coreaudio.AudioObjectAddPropertyListener)(kAudioObjectSystemObject, &propAddress, &ma_default_device_changed__coreaudio, pDevice);
        }
    #endif
    }
    
    pDevice->coreaudio.originalBufferSizeInFrames       = pConfig->bufferSizeInFrames;
    pDevice->coreaudio.originalBufferSizeInMilliseconds = pConfig->bufferSizeInMilliseconds;
    pDevice->coreaudio.originalPeriods                  = pConfig->periods;
    
    /*
    When stopping the device, a callback is called on another thread. We need to wait for this callback
    before returning from ma_device_stop(). This event is used for this.
    */
    ma_event_init(pContext, &pDevice->coreaudio.stopEvent);

    /* Need a ring buffer for duplex mode. */
    if (pConfig->deviceType == ma_device_type_duplex) {
        ma_uint32 rbSizeInFrames = (ma_uint32)ma_calculate_frame_count_after_src(pDevice->sampleRate, pDevice->capture.internalSampleRate, pDevice->capture.internalBufferSizeInFrames);
        ma_result result = ma_pcm_rb_init(pDevice->capture.format, pDevice->capture.channels, rbSizeInFrames, NULL, &pDevice->coreaudio.duplexRB);
        if (result != MA_SUCCESS) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[Core Audio] Failed to initialize ring buffer.", result);
        }
    }

    return MA_SUCCESS;
}


ma_result ma_device_start__coreaudio(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);
    
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        OSStatus status = ((ma_AudioOutputUnitStart_proc)pDevice->pContext->coreaudio.AudioOutputUnitStart)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
        if (status != noErr) {
            return ma_result_from_OSStatus(status);
        }
    }
    
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        OSStatus status = ((ma_AudioOutputUnitStart_proc)pDevice->pContext->coreaudio.AudioOutputUnitStart)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
        if (status != noErr) {
            if (pDevice->type == ma_device_type_duplex) {
                ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
            }
            return ma_result_from_OSStatus(status);
        }
    }
    
    return MA_SUCCESS;
}

ma_result ma_device_stop__coreaudio(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);
    
    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        OSStatus status = ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitCapture);
        if (status != noErr) {
            return ma_result_from_OSStatus(status);
        }
    }
    
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        OSStatus status = ((ma_AudioOutputUnitStop_proc)pDevice->pContext->coreaudio.AudioOutputUnitStop)((AudioUnit)pDevice->coreaudio.audioUnitPlayback);
        if (status != noErr) {
            return ma_result_from_OSStatus(status);
        }
    }
    
    /* We need to wait for the callback to finish before returning. */
    ma_event_wait(&pDevice->coreaudio.stopEvent);
    return MA_SUCCESS;
}


ma_result ma_context_uninit__coreaudio(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_coreaudio);
    
#if !defined(MA_NO_RUNTIME_LINKING) && !defined(MA_APPLE_MOBILE)
    ma_dlclose(pContext->coreaudio.hAudioUnit);
    ma_dlclose(pContext->coreaudio.hCoreAudio);
    ma_dlclose(pContext->coreaudio.hCoreFoundation);
#endif

    (void)pContext;
    return MA_SUCCESS;
}

ma_result ma_context_init__coreaudio(const ma_context_config* pConfig, ma_context* pContext)
{
    ma_assert(pContext != NULL);

    (void)pConfig;

#if defined(MA_APPLE_MOBILE)
    @autoreleasepool {
        AVAudioSession* pAudioSession = [AVAudioSession sharedInstance];
        ma_assert(pAudioSession != NULL);

        [pAudioSession setCategory: AVAudioSessionCategoryPlayAndRecord error:nil];
        
        /* By default we want miniaudio to use the speakers instead of the receiver. In the future this may be customizable. */
        ma_bool32 useSpeakers = MA_TRUE;
        if (useSpeakers) {
            [pAudioSession overrideOutputAudioPort:AVAudioSessionPortOverrideSpeaker error:nil];
        }
    }
#endif
    
#if !defined(MA_NO_RUNTIME_LINKING) && !defined(MA_APPLE_MOBILE)
    pContext->coreaudio.hCoreFoundation = ma_dlopen("CoreFoundation.framework/CoreFoundation");
    if (pContext->coreaudio.hCoreFoundation == NULL) {
        return MA_API_NOT_FOUND;
    }
    
    pContext->coreaudio.CFStringGetCString             = ma_dlsym(pContext->coreaudio.hCoreFoundation, "CFStringGetCString");
    
    
    pContext->coreaudio.hCoreAudio = ma_dlopen("CoreAudio.framework/CoreAudio");
    if (pContext->coreaudio.hCoreAudio == NULL) {
        ma_dlclose(pContext->coreaudio.hCoreFoundation);
        return MA_API_NOT_FOUND;
    }
    
    pContext->coreaudio.AudioObjectGetPropertyData     = ma_dlsym(pContext->coreaudio.hCoreAudio, "AudioObjectGetPropertyData");
    pContext->coreaudio.AudioObjectGetPropertyDataSize = ma_dlsym(pContext->coreaudio.hCoreAudio, "AudioObjectGetPropertyDataSize");
    pContext->coreaudio.AudioObjectSetPropertyData     = ma_dlsym(pContext->coreaudio.hCoreAudio, "AudioObjectSetPropertyData");
    pContext->coreaudio.AudioObjectAddPropertyListener = ma_dlsym(pContext->coreaudio.hCoreAudio, "AudioObjectAddPropertyListener");

    /*
    It looks like Apple has moved some APIs from AudioUnit into AudioToolbox on more recent versions of macOS. They are still
    defined in AudioUnit, but just in case they decide to remove them from there entirely I'm going to implement a fallback.
    The way it'll work is that it'll first try AudioUnit, and if the required symbols are not present there we'll fall back to
    AudioToolbox.
    */
    pContext->coreaudio.hAudioUnit = ma_dlopen("AudioUnit.framework/AudioUnit");
    if (pContext->coreaudio.hAudioUnit == NULL) {
        ma_dlclose(pContext->coreaudio.hCoreAudio);
        ma_dlclose(pContext->coreaudio.hCoreFoundation);
        return MA_API_NOT_FOUND;
    }
    
    if (ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioComponentFindNext") == NULL) {
        /* Couldn't find the required symbols in AudioUnit, so fall back to AudioToolbox. */
        ma_dlclose(pContext->coreaudio.hAudioUnit);
        pContext->coreaudio.hAudioUnit = ma_dlopen("AudioToolbox.framework/AudioToolbox");
        if (pContext->coreaudio.hAudioUnit == NULL) {
            ma_dlclose(pContext->coreaudio.hCoreAudio);
            ma_dlclose(pContext->coreaudio.hCoreFoundation);
            return MA_API_NOT_FOUND;
        }
    }
    
    pContext->coreaudio.AudioComponentFindNext         = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioComponentFindNext");
    pContext->coreaudio.AudioComponentInstanceDispose  = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioComponentInstanceDispose");
    pContext->coreaudio.AudioComponentInstanceNew      = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioComponentInstanceNew");
    pContext->coreaudio.AudioOutputUnitStart           = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioOutputUnitStart");
    pContext->coreaudio.AudioOutputUnitStop            = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioOutputUnitStop");
    pContext->coreaudio.AudioUnitAddPropertyListener   = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitAddPropertyListener");
    pContext->coreaudio.AudioUnitGetPropertyInfo       = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitGetPropertyInfo");
    pContext->coreaudio.AudioUnitGetProperty           = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitGetProperty");
    pContext->coreaudio.AudioUnitSetProperty           = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitSetProperty");
    pContext->coreaudio.AudioUnitInitialize            = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitInitialize");
    pContext->coreaudio.AudioUnitRender                = ma_dlsym(pContext->coreaudio.hAudioUnit, "AudioUnitRender");
#else
    pContext->coreaudio.CFStringGetCString             = (ma_proc)CFStringGetCString;
    
    #if defined(MA_APPLE_DESKTOP)
    pContext->coreaudio.AudioObjectGetPropertyData     = (ma_proc)AudioObjectGetPropertyData;
    pContext->coreaudio.AudioObjectGetPropertyDataSize = (ma_proc)AudioObjectGetPropertyDataSize;
    pContext->coreaudio.AudioObjectSetPropertyData     = (ma_proc)AudioObjectSetPropertyData;
    pContext->coreaudio.AudioObjectAddPropertyListener = (ma_proc)AudioObjectAddPropertyListener;
    #endif
    
    pContext->coreaudio.AudioComponentFindNext         = (ma_proc)AudioComponentFindNext;
    pContext->coreaudio.AudioComponentInstanceDispose  = (ma_proc)AudioComponentInstanceDispose;
    pContext->coreaudio.AudioComponentInstanceNew      = (ma_proc)AudioComponentInstanceNew;
    pContext->coreaudio.AudioOutputUnitStart           = (ma_proc)AudioOutputUnitStart;
    pContext->coreaudio.AudioOutputUnitStop            = (ma_proc)AudioOutputUnitStop;
    pContext->coreaudio.AudioUnitAddPropertyListener   = (ma_proc)AudioUnitAddPropertyListener;
    pContext->coreaudio.AudioUnitGetPropertyInfo       = (ma_proc)AudioUnitGetPropertyInfo;
    pContext->coreaudio.AudioUnitGetProperty           = (ma_proc)AudioUnitGetProperty;
    pContext->coreaudio.AudioUnitSetProperty           = (ma_proc)AudioUnitSetProperty;
    pContext->coreaudio.AudioUnitInitialize            = (ma_proc)AudioUnitInitialize;
    pContext->coreaudio.AudioUnitRender                = (ma_proc)AudioUnitRender;
#endif

    pContext->isBackendAsynchronous = MA_TRUE;
    
    pContext->onUninit        = ma_context_uninit__coreaudio;
    pContext->onDeviceIDEqual = ma_context_is_device_id_equal__coreaudio;
    pContext->onEnumDevices   = ma_context_enumerate_devices__coreaudio;
    pContext->onGetDeviceInfo = ma_context_get_device_info__coreaudio;
    pContext->onDeviceInit    = ma_device_init__coreaudio;
    pContext->onDeviceUninit  = ma_device_uninit__coreaudio;
    pContext->onDeviceStart   = ma_device_start__coreaudio;
    pContext->onDeviceStop    = ma_device_stop__coreaudio;
    
    /* Audio component. */
    {
        AudioComponentDescription desc;
        desc.componentType         = kAudioUnitType_Output;
    #if defined(MA_APPLE_DESKTOP)
        desc.componentSubType      = kAudioUnitSubType_HALOutput;
    #else
        desc.componentSubType      = kAudioUnitSubType_RemoteIO;
    #endif
        desc.componentManufacturer = kAudioUnitManufacturer_Apple;
        desc.componentFlags        = 0;
        desc.componentFlagsMask    = 0;
    
        pContext->coreaudio.component = ((ma_AudioComponentFindNext_proc)pContext->coreaudio.AudioComponentFindNext)(NULL, &desc);
        if (pContext->coreaudio.component == NULL) {
    #if !defined(MA_NO_RUNTIME_LINKING) && !defined(MA_APPLE_MOBILE)
            ma_dlclose(pContext->coreaudio.hAudioUnit);
            ma_dlclose(pContext->coreaudio.hCoreAudio);
            ma_dlclose(pContext->coreaudio.hCoreFoundation);
    #endif
            return MA_FAILED_TO_INIT_BACKEND;
        }
    }

    return MA_SUCCESS;
}
#endif  /* Core Audio */



/******************************************************************************

sndio Backend

******************************************************************************/
#ifdef MA_HAS_SNDIO
#include <fcntl.h>
#include <sys/stat.h>

/*
Only supporting OpenBSD. This did not work very well at all on FreeBSD when I tried it. Not sure if this is due
to miniaudio's implementation or if it's some kind of system configuration issue, but basically the default device
just doesn't emit any sound, or at times you'll hear tiny pieces. I will consider enabling this when there's
demand for it or if I can get it tested and debugged more thoroughly.
*/
#if 0
#if defined(__NetBSD__) || defined(__OpenBSD__)
#include <sys/audioio.h>
#endif
#if defined(__FreeBSD__) || defined(__DragonFly__)
#include <sys/soundcard.h>
#endif
#endif

#define MA_SIO_DEVANY   "default"
#define MA_SIO_PLAY     1
#define MA_SIO_REC      2
#define MA_SIO_NENC     8
#define MA_SIO_NCHAN    8
#define MA_SIO_NRATE    16
#define MA_SIO_NCONF    4

struct ma_sio_hdl; /* <-- Opaque */

struct ma_sio_par
{
    unsigned int bits;
    unsigned int bps;
    unsigned int sig;
    unsigned int le;
    unsigned int msb;
    unsigned int rchan;
    unsigned int pchan;
    unsigned int rate;
    unsigned int bufsz;
    unsigned int xrun;
    unsigned int round;
    unsigned int appbufsz;
    int __pad[3];
    unsigned int __magic;
};

struct ma_sio_enc
{
    unsigned int bits;
    unsigned int bps;
    unsigned int sig;
    unsigned int le;
    unsigned int msb;
};

struct ma_sio_conf
{
    unsigned int enc;
    unsigned int rchan;
    unsigned int pchan;
    unsigned int rate;
};

struct ma_sio_cap
{
    struct ma_sio_enc enc[MA_SIO_NENC];
    unsigned int rchan[MA_SIO_NCHAN];
    unsigned int pchan[MA_SIO_NCHAN];
    unsigned int rate[MA_SIO_NRATE];
    int __pad[7];
    unsigned int nconf;
    struct ma_sio_conf confs[MA_SIO_NCONF];
};

typedef struct ma_sio_hdl* (* ma_sio_open_proc)   (const char*, unsigned int, int);
typedef void               (* ma_sio_close_proc)  (struct ma_sio_hdl*);
typedef int                (* ma_sio_setpar_proc) (struct ma_sio_hdl*, struct ma_sio_par*);
typedef int                (* ma_sio_getpar_proc) (struct ma_sio_hdl*, struct ma_sio_par*);
typedef int                (* ma_sio_getcap_proc) (struct ma_sio_hdl*, struct ma_sio_cap*);
typedef size_t             (* ma_sio_write_proc)  (struct ma_sio_hdl*, const void*, size_t);
typedef size_t             (* ma_sio_read_proc)   (struct ma_sio_hdl*, void*, size_t);
typedef int                (* ma_sio_start_proc)  (struct ma_sio_hdl*);
typedef int                (* ma_sio_stop_proc)   (struct ma_sio_hdl*);
typedef int                (* ma_sio_initpar_proc)(struct ma_sio_par*);

ma_format ma_format_from_sio_enc__sndio(unsigned int bits, unsigned int bps, unsigned int sig, unsigned int le, unsigned int msb)
{
    /* We only support native-endian right now. */
    if ((ma_is_little_endian() && le == 0) || (ma_is_big_endian() && le == 1)) {
        return ma_format_unknown;
    }
    
    if (bits ==  8 && bps == 1 && sig == 0) {
        return ma_format_u8;
    }
    if (bits == 16 && bps == 2 && sig == 1) {
        return ma_format_s16;
    }
    if (bits == 24 && bps == 3 && sig == 1) {
        return ma_format_s24;
    }
    if (bits == 24 && bps == 4 && sig == 1 && msb == 0) {
        /*return ma_format_s24_32;*/
    }
    if (bits == 32 && bps == 4 && sig == 1) {
        return ma_format_s32;
    }
    
    return ma_format_unknown;
}

ma_format ma_find_best_format_from_sio_cap__sndio(struct ma_sio_cap* caps)
{
    ma_format bestFormat;
    unsigned int iConfig;

    ma_assert(caps != NULL);
    
    bestFormat = ma_format_unknown;
    for (iConfig = 0; iConfig < caps->nconf; iConfig += 1) {
        unsigned int iEncoding;
        for (iEncoding = 0; iEncoding < MA_SIO_NENC; iEncoding += 1) {
            unsigned int bits;
            unsigned int bps;
            unsigned int sig;
            unsigned int le;
            unsigned int msb;
            ma_format format;

            if ((caps->confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            bits = caps->enc[iEncoding].bits;
            bps  = caps->enc[iEncoding].bps;
            sig  = caps->enc[iEncoding].sig;
            le   = caps->enc[iEncoding].le;
            msb  = caps->enc[iEncoding].msb;
            format = ma_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format == ma_format_unknown) {
                continue;   /* Format not supported. */
            }
            
            if (bestFormat == ma_format_unknown) {
                bestFormat = format;
            } else {
                if (ma_get_format_priority_index(bestFormat) > ma_get_format_priority_index(format)) {    /* <-- Lower = better. */
                    bestFormat = format;
                }
            }
        }
    }
    
    return ma_format_unknown;
}

ma_uint32 ma_find_best_channels_from_sio_cap__sndio(struct ma_sio_cap* caps, ma_device_type deviceType, ma_format requiredFormat)
{
    ma_uint32 maxChannels;
    unsigned int iConfig;

    ma_assert(caps != NULL);
    ma_assert(requiredFormat != ma_format_unknown);
    
    /* Just pick whatever configuration has the most channels. */
    ma_uint32 maxChannels = 0;
    for (iConfig = 0; iConfig < caps->nconf; iConfig += 1) {
        /* The encoding should be of requiredFormat. */
        unsigned int iEncoding;
        for (unsigned int iEncoding = 0; iEncoding < MA_SIO_NENC; iEncoding += 1) {
            unsigned int iChannel;
            unsigned int bits;
            unsigned int bps;
            unsigned int sig;
            unsigned int le;
            unsigned int msb;
            ma_format format;

            if ((caps->confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            bits = caps->enc[iEncoding].bits;
            bps  = caps->enc[iEncoding].bps;
            sig  = caps->enc[iEncoding].sig;
            le   = caps->enc[iEncoding].le;
            msb  = caps->enc[iEncoding].msb;
            format = ma_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format != requiredFormat) {
                continue;
            }
            
            /* Getting here means the format is supported. Iterate over each channel count and grab the biggest one. */
            for (iChannel = 0; iChannel < MA_SIO_NCHAN; iChannel += 1) {
                unsigned int chan = 0;
                unsigned int channels;

                if (deviceType == ma_device_type_playback) {
                    chan = caps->confs[iConfig].pchan;
                } else {
                    chan = caps->confs[iConfig].rchan;
                }
            
                if ((chan & (1UL << iChannel)) == 0) {
                    continue;
                }
                
                if (deviceType == ma_device_type_playback) {
                    channels = caps->pchan[iChannel];
                } else {
                    channels = caps->rchan[iChannel];
                }
                
                if (maxChannels < channels) {
                    maxChannels = channels;
                }
            }
        }
    }
    
    return maxChannels;
}

ma_uint32 ma_find_best_sample_rate_from_sio_cap__sndio(struct ma_sio_cap* caps, ma_device_type deviceType, ma_format requiredFormat, ma_uint32 requiredChannels)
{
    ma_uint32 firstSampleRate;
    ma_uint32 bestSampleRate;
    unsigned int iConfig;

    ma_assert(caps != NULL);
    ma_assert(requiredFormat != ma_format_unknown);
    ma_assert(requiredChannels > 0);
    ma_assert(requiredChannels <= MA_MAX_CHANNELS);
    
    firstSampleRate = 0; /* <-- If the device does not support a standard rate we'll fall back to the first one that's found. */
    bestSampleRate  = 0;

    for (iConfig = 0; iConfig < caps->nconf; iConfig += 1) {
        /* The encoding should be of requiredFormat. */
        unsigned int iEncoding;
        for (iEncoding = 0; iEncoding < MA_SIO_NENC; iEncoding += 1) {
            unsigned int iChannel;
            unsigned int bits;
            unsigned int bps;
            unsigned int sig;
            unsigned int le;
            unsigned int msb;
            ma_format format;

            if ((caps->confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            bits = caps->enc[iEncoding].bits;
            bps  = caps->enc[iEncoding].bps;
            sig  = caps->enc[iEncoding].sig;
            le   = caps->enc[iEncoding].le;
            msb  = caps->enc[iEncoding].msb;
            format = ma_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format != requiredFormat) {
                continue;
            }
            
            /* Getting here means the format is supported. Iterate over each channel count and grab the biggest one. */
            for (iChannel = 0; iChannel < MA_SIO_NCHAN; iChannel += 1) {
                unsigned int chan = 0;
                unsigned int channels;
                unsigned int iRate;

                if (deviceType == ma_device_type_playback) {
                    chan = caps->confs[iConfig].pchan;
                } else {
                    chan = caps->confs[iConfig].rchan;
                }
            
                if ((chan & (1UL << iChannel)) == 0) {
                    continue;
                }
                
                if (deviceType == ma_device_type_playback) {
                    channels = caps->pchan[iChannel];
                } else {
                    channels = caps->rchan[iChannel];
                }
                
                if (channels != requiredChannels) {
                    continue;
                }
                
                /* Getting here means we have found a compatible encoding/channel pair. */
                for (iRate = 0; iRate < MA_SIO_NRATE; iRate += 1) {
                    ma_uint32 rate = (ma_uint32)caps->rate[iRate];
                    ma_uint32 ratePriority;
                
                    if (firstSampleRate == 0) {
                        firstSampleRate = rate;
                    }
                    
                    /* Disregard this rate if it's not a standard one. */
                    ratePriority = ma_get_standard_sample_rate_priority_index(rate);
                    if (ratePriority == (ma_uint32)-1) {
                        continue;
                    }
                    
                    if (ma_get_standard_sample_rate_priority_index(bestSampleRate) > ratePriority) {   /* Lower = better. */
                        bestSampleRate = rate;
                    }
                }
            }
        }
    }
    
    /* If a standard sample rate was not found just fall back to the first one that was iterated. */
    if (bestSampleRate == 0) {
        bestSampleRate = firstSampleRate;
    }
    
    return bestSampleRate;
}


ma_bool32 ma_context_is_device_id_equal__sndio(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return ma_strcmp(pID0->sndio, pID1->sndio) == 0;
}

ma_result ma_context_enumerate_devices__sndio(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_bool32 isTerminating = MA_FALSE;
    struct ma_sio_hdl* handle;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);
    
    /* sndio doesn't seem to have a good device enumeration API, so I'm therefore only enumerating over default devices for now. */
    
    /* Playback. */
    if (!isTerminating) {
        handle = ((ma_sio_open_proc)pContext->sndio.sio_open)(MA_SIO_DEVANY, MA_SIO_PLAY, 0);
        if (handle != NULL) {
            /* Supports playback. */
            ma_device_info deviceInfo;
            ma_zero_object(&deviceInfo);
            ma_strcpy_s(deviceInfo.id.sndio, sizeof(deviceInfo.id.sndio), MA_SIO_DEVANY);
            ma_strcpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_PLAYBACK_DEVICE_NAME);
            
            isTerminating = !callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
            
            ((ma_sio_close_proc)pContext->sndio.sio_close)(handle);
        }
    }
    
    /* Capture. */
    if (!isTerminating) {
        handle = ((ma_sio_open_proc)pContext->sndio.sio_open)(MA_SIO_DEVANY, MA_SIO_REC, 0);
        if (handle != NULL) {
            /* Supports capture. */
            ma_device_info deviceInfo;
            ma_zero_object(&deviceInfo);
            ma_strcpy_s(deviceInfo.id.sndio, sizeof(deviceInfo.id.sndio), "default");
            ma_strcpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_CAPTURE_DEVICE_NAME);

            isTerminating = !callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
            
            ((ma_sio_close_proc)pContext->sndio.sio_close)(handle);
        }
    }
    
    return MA_SUCCESS;
}

ma_result ma_context_get_device_info__sndio(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    char devid[256];
    struct ma_sio_hdl* handle;
    struct ma_sio_cap caps;
    unsigned int iConfig;

    ma_assert(pContext != NULL);
    (void)shareMode;
    
    /* We need to open the device before we can get information about it. */
    if (pDeviceID == NULL) {
        ma_strcpy_s(devid, sizeof(devid), MA_SIO_DEVANY);
        ma_strcpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), (deviceType == ma_device_type_playback) ? MA_DEFAULT_PLAYBACK_DEVICE_NAME : MA_DEFAULT_CAPTURE_DEVICE_NAME);
    } else {
        ma_strcpy_s(devid, sizeof(devid), pDeviceID->sndio);
        ma_strcpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), devid);
    }
    
    handle = ((ma_sio_open_proc)pContext->sndio.sio_open)(devid, (deviceType == ma_device_type_playback) ? MA_SIO_PLAY : MA_SIO_REC, 0);
    if (handle == NULL) {
        return MA_NO_DEVICE;
    }
    
    if (((ma_sio_getcap_proc)pContext->sndio.sio_getcap)(handle, &caps) == 0) {
        return MA_ERROR;
    }
    
    for (iConfig = 0; iConfig < caps.nconf; iConfig += 1) {
        /*
        The main thing we care about is that the encoding is supported by miniaudio. If it is, we want to give
        preference to some formats over others.
        */
        unsigned int iEncoding;
        unsigned int iChannel;
        unsigned int iRate;

        for (iEncoding = 0; iEncoding < MA_SIO_NENC; iEncoding += 1) {
            unsigned int bits;
            unsigned int bps;
            unsigned int sig;
            unsigned int le;
            unsigned int msb;
            ma_format format;
            ma_bool32 formatExists = MA_FALSE;
            ma_uint32 iExistingFormat;

            if ((caps.confs[iConfig].enc & (1UL << iEncoding)) == 0) {
                continue;
            }
            
            bits = caps.enc[iEncoding].bits;
            bps  = caps.enc[iEncoding].bps;
            sig  = caps.enc[iEncoding].sig;
            le   = caps.enc[iEncoding].le;
            msb  = caps.enc[iEncoding].msb;
            format = ma_format_from_sio_enc__sndio(bits, bps, sig, le, msb);
            if (format == ma_format_unknown) {
                continue;   /* Format not supported. */
            }
            
            /* Add this format if it doesn't already exist. */
            for (iExistingFormat = 0; iExistingFormat < pDeviceInfo->formatCount; iExistingFormat += 1) {
                if (pDeviceInfo->formats[iExistingFormat] == format) {
                    formatExists = MA_TRUE;
                    break;
                }
            }
            
            if (!formatExists) {
                pDeviceInfo->formats[pDeviceInfo->formatCount++] = format;
            }
        }
        
        /* Channels. */
        for (unsigned int iChannel = 0; iChannel < MA_SIO_NCHAN; iChannel += 1) {
            unsigned int chan = 0;
            unsigned int channels;

            if (deviceType == ma_device_type_playback) {
                chan = caps.confs[iConfig].pchan;
            } else {
                chan = caps.confs[iConfig].rchan;
            }
        
            if ((chan & (1UL << iChannel)) == 0) {
                continue;
            }
            
            if (deviceType == ma_device_type_playback) {
                channels = caps.pchan[iChannel];
            } else {
                channels = caps.rchan[iChannel];
            }
            
            if (pDeviceInfo->minChannels > channels) {
                pDeviceInfo->minChannels = channels;
            }
            if (pDeviceInfo->maxChannels < channels) {
                pDeviceInfo->maxChannels = channels;
            }
        }
        
        /* Sample rates. */
        for (iRate = 0; iRate < MA_SIO_NRATE; iRate += 1) {
            if ((caps.confs[iConfig].rate & (1UL << iRate)) != 0) {
                unsigned int rate = caps.rate[iRate];
                if (pDeviceInfo->minSampleRate > rate) {
                    pDeviceInfo->minSampleRate = rate;
                }
                if (pDeviceInfo->maxSampleRate < rate) {
                    pDeviceInfo->maxSampleRate = rate;
                }
            }
        }
    }

    ((ma_sio_close_proc)pContext->sndio.sio_close)(handle);
    return MA_SUCCESS;
}

void ma_device_uninit__sndio(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_sio_close_proc)pDevice->pContext->sndio.sio_close)((struct ma_sio_hdl*)pDevice->sndio.handleCapture);
    }

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_sio_close_proc)pDevice->pContext->sndio.sio_close)((struct ma_sio_hdl*)pDevice->sndio.handlePlayback);
    }
}

ma_result ma_device_init_handle__sndio(ma_context* pContext, const ma_device_config* pConfig, ma_device_type deviceType, ma_device* pDevice)
{
    const char* pDeviceName;
    ma_ptr handle;
    int openFlags = 0;
    struct ma_sio_cap caps;
    struct ma_sio_par par;
    ma_device_id* pDeviceID;
    ma_format format;
    ma_uint32 channels;
    ma_uint32 sampleRate;
    ma_format internalFormat;
    ma_uint32 internalChannels;
    ma_uint32 internalSampleRate;
    ma_uint32 internalBufferSizeInFrames;
    ma_uint32 internalPeriods;

    ma_assert(pContext   != NULL);
    ma_assert(pConfig    != NULL);
    ma_assert(deviceType != ma_device_type_duplex);
    ma_assert(pDevice    != NULL);

    if (deviceType == ma_device_type_capture) {
        openFlags  = MA_SIO_REC;
        pDeviceID  = pConfig->capture.pDeviceID;
        format     = pConfig->capture.format;
        channels   = pConfig->capture.channels;
        sampleRate = pConfig->sampleRate;
    } else {
        openFlags = MA_SIO_PLAY;
        pDeviceID  = pConfig->playback.pDeviceID;
        format     = pConfig->playback.format;
        channels   = pConfig->playback.channels;
        sampleRate = pConfig->sampleRate;
    }

    pDeviceName = MA_SIO_DEVANY;
    if (pDeviceID != NULL) {
        pDeviceName = pDeviceID->sndio;
    }

    handle = (ma_ptr)((ma_sio_open_proc)pContext->sndio.sio_open)(pDeviceName, openFlags, 0);
    if (handle == NULL) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to open device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    /* We need to retrieve the device caps to determine the most appropriate format to use. */
    if (((ma_sio_getcap_proc)pContext->sndio.sio_getcap)((struct ma_sio_hdl*)handle, &caps) == 0) {
        ((ma_sio_close_proc)pContext->sndio.sio_close)((struct ma_sio_hdl*)handle);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to retrieve device caps.", MA_ERROR);
    }

    /*
    Note: sndio reports a huge range of available channels. This is inconvenient for us because there's no real
    way, as far as I can tell, to get the _actual_ channel count of the device. I'm therefore restricting this
    to the requested channels, regardless of whether or not the default channel count is requested.
    
    For hardware devices, I'm suspecting only a single channel count will be reported and we can safely use the
    value returned by ma_find_best_channels_from_sio_cap__sndio().
    */
    if (deviceType == ma_device_type_capture) {
        if (pDevice->capture.usingDefaultFormat) {
            format = ma_find_best_format_from_sio_cap__sndio(&caps);
        }
        if (pDevice->capture.usingDefaultChannels) {
            if (strlen(pDeviceName) > strlen("rsnd/") && strncmp(pDeviceName, "rsnd/", strlen("rsnd/")) == 0) {
                channels = ma_find_best_channels_from_sio_cap__sndio(&caps, deviceType, format);
            }
        }
    } else {
        if (pDevice->playback.usingDefaultFormat) {
            format = ma_find_best_format_from_sio_cap__sndio(&caps);
        }
        if (pDevice->playback.usingDefaultChannels) {
            if (strlen(pDeviceName) > strlen("rsnd/") && strncmp(pDeviceName, "rsnd/", strlen("rsnd/")) == 0) {
                channels = ma_find_best_channels_from_sio_cap__sndio(&caps, deviceType, format);
            }
        }
    }
    
    if (pDevice->usingDefaultSampleRate) {
        sampleRate = ma_find_best_sample_rate_from_sio_cap__sndio(&caps, pConfig->deviceType, format, channels);
    }


    ((ma_sio_initpar_proc)pDevice->pContext->sndio.sio_initpar)(&par);
    par.msb = 0;
    par.le  = ma_is_little_endian();
    
    switch (format) {
        case ma_format_u8:
        {
            par.bits = 8;
            par.bps  = 1;
            par.sig  = 0;
        } break;
        
        case ma_format_s24:
        {
            par.bits = 24;
            par.bps  = 3;
            par.sig  = 1;
        } break;
        
        case ma_format_s32:
        {
            par.bits = 32;
            par.bps  = 4;
            par.sig  = 1;
        } break;
        
        case ma_format_s16:
        case ma_format_f32:
        default:
        {
            par.bits = 16;
            par.bps  = 2;
            par.sig  = 1;
        } break;
    }
    
    if (deviceType == ma_device_type_capture) {
        par.rchan = channels;
    } else {
        par.pchan = channels;
    }

    par.rate = sampleRate;

    internalBufferSizeInFrames = pConfig->bufferSizeInFrames;
    if (internalBufferSizeInFrames == 0) {
        internalBufferSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(pConfig->bufferSizeInMilliseconds, par.rate);
    }

    par.round    = internalBufferSizeInFrames / pConfig->periods;
    par.appbufsz = par.round * pConfig->periods;
    
    if (((ma_sio_setpar_proc)pContext->sndio.sio_setpar)((struct ma_sio_hdl*)handle, &par) == 0) {
        ((ma_sio_close_proc)pContext->sndio.sio_close)((struct ma_sio_hdl*)handle);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to set buffer size.", MA_FORMAT_NOT_SUPPORTED);
    }
    if (((ma_sio_getpar_proc)pContext->sndio.sio_getpar)((struct ma_sio_hdl*)handle, &par) == 0) {
        ((ma_sio_close_proc)pContext->sndio.sio_close)((struct ma_sio_hdl*)handle);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to retrieve buffer size.", MA_FORMAT_NOT_SUPPORTED);
    }

    internalFormat             = ma_format_from_sio_enc__sndio(par.bits, par.bps, par.sig, par.le, par.msb);
    internalChannels           = (deviceType == ma_device_type_capture) ? par.rchan : par.pchan;
    internalSampleRate         = par.rate;
    internalPeriods            = par.appbufsz / par.round;
    internalBufferSizeInFrames = par.appbufsz;

    if (deviceType == ma_device_type_capture) {
        pDevice->sndio.handleCapture                 = handle;
        pDevice->capture.internalFormat              = internalFormat;
        pDevice->capture.internalChannels            = internalChannels;
        pDevice->capture.internalSampleRate          = internalSampleRate;
        ma_get_standard_channel_map(ma_standard_channel_map_sndio, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);
        pDevice->capture.internalBufferSizeInFrames  = internalBufferSizeInFrames;
        pDevice->capture.internalPeriods             = internalPeriods;
    } else {
        pDevice->sndio.handlePlayback                = handle;
        pDevice->playback.internalFormat             = internalFormat;
        pDevice->playback.internalChannels           = internalChannels;
        pDevice->playback.internalSampleRate         = internalSampleRate;
        ma_get_standard_channel_map(ma_standard_channel_map_sndio, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);
        pDevice->playback.internalBufferSizeInFrames = internalBufferSizeInFrames;
        pDevice->playback.internalPeriods            = internalPeriods;
    }

#ifdef MA_DEBUG_OUTPUT
    printf("DEVICE INFO\n");
    printf("    Format:      %s\n", ma_get_format_name(internalFormat));
    printf("    Channels:    %d\n", internalChannels);
    printf("    Sample Rate: %d\n", internalSampleRate);
    printf("    Buffer Size: %d\n", internalBufferSizeInFrames);
    printf("    Periods:     %d\n", internalPeriods);
    printf("    appbufsz:    %d\n", par.appbufsz);
    printf("    round:       %d\n", par.round);
#endif

    return MA_SUCCESS;
}

ma_result ma_device_init__sndio(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    ma_zero_object(&pDevice->sndio);

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_handle__sndio(pContext, pConfig, ma_device_type_capture, pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_handle__sndio(pContext, pConfig, ma_device_type_playback, pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_stop__sndio(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ((ma_sio_stop_proc)pDevice->pContext->sndio.sio_stop)((struct ma_sio_hdl*)pDevice->sndio.handleCapture);
        ma_atomic_exchange_32(&pDevice->sndio.isStartedCapture, MA_FALSE);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ((ma_sio_stop_proc)pDevice->pContext->sndio.sio_stop)((struct ma_sio_hdl*)pDevice->sndio.handlePlayback);
        ma_atomic_exchange_32(&pDevice->sndio.isStartedPlayback, MA_FALSE);
    }

    return MA_SUCCESS;
}

ma_result ma_device_write__sndio(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount)
{
    int result;

    if (!pDevice->sndio.isStartedPlayback) {
        ((ma_sio_start_proc)pDevice->pContext->sndio.sio_start)((struct ma_sio_hdl*)pDevice->sndio.handlePlayback);   /* <-- Doesn't actually playback until data is written. */
        ma_atomic_exchange_32(&pDevice->sndio.isStartedPlayback, MA_TRUE);
    }

    result = ((ma_sio_write_proc)pDevice->pContext->sndio.sio_write)((struct ma_sio_hdl*)pDevice->sndio.handlePlayback, pPCMFrames, frameCount * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));
    if (result == 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to send data from the client to the device.", MA_FAILED_TO_SEND_DATA_TO_DEVICE);
    }
    
    return MA_SUCCESS;
}

ma_result ma_device_read__sndio(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount)
{
    int result;

    if (!pDevice->sndio.isStartedCapture) {
        ((ma_sio_start_proc)pDevice->pContext->sndio.sio_start)((struct ma_sio_hdl*)pDevice->sndio.handleCapture);   /* <-- Doesn't actually playback until data is written. */
        ma_atomic_exchange_32(&pDevice->sndio.isStartedCapture, MA_TRUE);
    }
    
    result = ((ma_sio_read_proc)pDevice->pContext->sndio.sio_read)((struct ma_sio_hdl*)pDevice->sndio.handleCapture, pPCMFrames, frameCount * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));
    if (result == 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[sndio] Failed to read data from the device to be sent to the device.", MA_FAILED_TO_SEND_DATA_TO_DEVICE);
    }
    
    return MA_SUCCESS;
}

ma_result ma_context_uninit__sndio(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_sndio);

    (void)pContext;
    return MA_SUCCESS;
}

ma_result ma_context_init__sndio(const ma_context_config* pConfig, ma_context* pContext)
{
#ifndef MA_NO_RUNTIME_LINKING
    const char* libsndioNames[] = {
        "libsndio.so"
    };
    size_t i;

    for (i = 0; i < ma_countof(libsndioNames); ++i) {
        pContext->sndio.sndioSO = ma_dlopen(libsndioNames[i]);
        if (pContext->sndio.sndioSO != NULL) {
            break;
        }
    }

    if (pContext->sndio.sndioSO == NULL) {
        return MA_NO_BACKEND;
    }
    
    pContext->sndio.sio_open    = (ma_proc)ma_dlsym(pContext->sndio.sndioSO, "sio_open");
    pContext->sndio.sio_close   = (ma_proc)ma_dlsym(pContext->sndio.sndioSO, "sio_close");
    pContext->sndio.sio_setpar  = (ma_proc)ma_dlsym(pContext->sndio.sndioSO, "sio_setpar");
    pContext->sndio.sio_getpar  = (ma_proc)ma_dlsym(pContext->sndio.sndioSO, "sio_getpar");
    pContext->sndio.sio_getcap  = (ma_proc)ma_dlsym(pContext->sndio.sndioSO, "sio_getcap");
    pContext->sndio.sio_write   = (ma_proc)ma_dlsym(pContext->sndio.sndioSO, "sio_write");
    pContext->sndio.sio_read    = (ma_proc)ma_dlsym(pContext->sndio.sndioSO, "sio_read");
    pContext->sndio.sio_start   = (ma_proc)ma_dlsym(pContext->sndio.sndioSO, "sio_start");
    pContext->sndio.sio_stop    = (ma_proc)ma_dlsym(pContext->sndio.sndioSO, "sio_stop");
    pContext->sndio.sio_initpar = (ma_proc)ma_dlsym(pContext->sndio.sndioSO, "sio_initpar");
#else
    pContext->sndio.sio_open    = sio_open;
    pContext->sndio.sio_close   = sio_close;
    pContext->sndio.sio_setpar  = sio_setpar;
    pContext->sndio.sio_getpar  = sio_getpar;
    pContext->sndio.sio_getcap  = sio_getcap;
    pContext->sndio.sio_write   = sio_write;
    pContext->sndio.sio_read    = sio_read;
    pContext->sndio.sio_start   = sio_start;
    pContext->sndio.sio_stop    = sio_stop;
    pContext->sndio.sio_initpar = sio_initpar;
#endif

    pContext->onUninit              = ma_context_uninit__sndio;
    pContext->onDeviceIDEqual       = ma_context_is_device_id_equal__sndio;
    pContext->onEnumDevices         = ma_context_enumerate_devices__sndio;
    pContext->onGetDeviceInfo       = ma_context_get_device_info__sndio;
    pContext->onDeviceInit          = ma_device_init__sndio;
    pContext->onDeviceUninit        = ma_device_uninit__sndio;
    pContext->onDeviceStart         = NULL; /* Not required for synchronous backends. */
    pContext->onDeviceStop          = ma_device_stop__sndio;
    pContext->onDeviceWrite         = ma_device_write__sndio;
    pContext->onDeviceRead          = ma_device_read__sndio;

    (void)pConfig;
    return MA_SUCCESS;
}
#endif  /* sndio */



/******************************************************************************

audio(4) Backend

******************************************************************************/
#ifdef MA_HAS_AUDIO4
#include <fcntl.h>
#include <poll.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/audioio.h>

#if defined(__OpenBSD__)
    #include <sys/param.h>
    #if defined(OpenBSD) && OpenBSD >= 201709
        #define MA_AUDIO4_USE_NEW_API
    #endif
#endif

void ma_construct_device_id__audio4(char* id, size_t idSize, const char* base, int deviceIndex)
{
    size_t baseLen;

    ma_assert(id != NULL);
    ma_assert(idSize > 0);
    ma_assert(deviceIndex >= 0);
    
    baseLen = strlen(base);
    ma_assert(idSize > baseLen);
    
    ma_strcpy_s(id, idSize, base);
    ma_itoa_s(deviceIndex, id+baseLen, idSize-baseLen, 10);
}

ma_result ma_extract_device_index_from_id__audio4(const char* id, const char* base, int* pIndexOut)
{
    size_t idLen;
    size_t baseLen;
    const char* deviceIndexStr;

    ma_assert(id != NULL);
    ma_assert(base != NULL);
    ma_assert(pIndexOut != NULL);
    
    idLen = strlen(id);
    baseLen = strlen(base);
    if (idLen <= baseLen) {
        return MA_ERROR;   /* Doesn't look like the id starts with the base. */
    }
    
    if (strncmp(id, base, baseLen) != 0) {
        return MA_ERROR;   /* ID does not begin with base. */
    }
    
    deviceIndexStr = id + baseLen;
    if (deviceIndexStr[0] == '\0') {
        return MA_ERROR;   /* No index specified in the ID. */
    }
    
    if (pIndexOut) {
        *pIndexOut = atoi(deviceIndexStr);
    }
    
    return MA_SUCCESS;
}

ma_bool32 ma_context_is_device_id_equal__audio4(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return ma_strcmp(pID0->audio4, pID1->audio4) == 0;
}

#if !defined(MA_AUDIO4_USE_NEW_API)    /* Old API */
ma_format ma_format_from_encoding__audio4(unsigned int encoding, unsigned int precision)
{
    if (precision == 8 && (encoding == AUDIO_ENCODING_ULINEAR || encoding == AUDIO_ENCODING_ULINEAR || encoding == AUDIO_ENCODING_ULINEAR_LE || encoding == AUDIO_ENCODING_ULINEAR_BE)) {
        return ma_format_u8;
    } else {
        if (ma_is_little_endian() && encoding == AUDIO_ENCODING_SLINEAR_LE) {
            if (precision == 16) {
                return ma_format_s16;
            } else if (precision == 24) {
                return ma_format_s24;
            } else if (precision == 32) {
                return ma_format_s32;
            }
        } else if (ma_is_big_endian() && encoding == AUDIO_ENCODING_SLINEAR_BE) {
            if (precision == 16) {
                return ma_format_s16;
            } else if (precision == 24) {
                return ma_format_s24;
            } else if (precision == 32) {
                return ma_format_s32;
            }
        }
    }

    return ma_format_unknown;  /* Encoding not supported. */
}

void ma_encoding_from_format__audio4(ma_format format, unsigned int* pEncoding, unsigned int* pPrecision)
{
    ma_assert(format     != ma_format_unknown);
    ma_assert(pEncoding  != NULL);
    ma_assert(pPrecision != NULL);

    switch (format)
    {
        case ma_format_u8:
        {
            *pEncoding = AUDIO_ENCODING_ULINEAR;
            *pPrecision = 8;
        } break;

        case ma_format_s24:
        {
            *pEncoding = (ma_is_little_endian()) ? AUDIO_ENCODING_SLINEAR_LE : AUDIO_ENCODING_SLINEAR_BE;
            *pPrecision = 24;
        } break;

        case ma_format_s32:
        {
            *pEncoding = (ma_is_little_endian()) ? AUDIO_ENCODING_SLINEAR_LE : AUDIO_ENCODING_SLINEAR_BE;
            *pPrecision = 32;
        } break;

        case ma_format_s16:
        case ma_format_f32:
        default:
        {
            *pEncoding = (ma_is_little_endian()) ? AUDIO_ENCODING_SLINEAR_LE : AUDIO_ENCODING_SLINEAR_BE;
            *pPrecision = 16;
        } break;
    }
}

ma_format ma_format_from_prinfo__audio4(struct audio_prinfo* prinfo)
{
    return ma_format_from_encoding__audio4(prinfo->encoding, prinfo->precision);
}
#else
ma_format ma_format_from_swpar__audio4(struct audio_swpar* par)
{
    if (par->bits == 8 && par->bps == 1 && par->sig == 0) {
        return ma_format_u8;
    }
    if (par->bits == 16 && par->bps == 2 && par->sig == 1 && par->le == ma_is_little_endian()) {
        return ma_format_s16;
    }
    if (par->bits == 24 && par->bps == 3 && par->sig == 1 && par->le == ma_is_little_endian()) {
        return ma_format_s24;
    }
    if (par->bits == 32 && par->bps == 4 && par->sig == 1 && par->le == ma_is_little_endian()) {
        return ma_format_f32;
    }

    /* Format not supported. */
    return ma_format_unknown;
}
#endif

ma_result ma_context_get_device_info_from_fd__audio4(ma_context* pContext, ma_device_type deviceType, int fd, ma_device_info* pInfoOut)
{
    audio_device_t fdDevice;
#if !defined(MA_AUDIO4_USE_NEW_API)
    int counter = 0;
    audio_info_t fdInfo;
#else
    struct audio_swpar fdPar;
    ma_format format;
#endif

    ma_assert(pContext != NULL);
    ma_assert(fd >= 0);
    ma_assert(pInfoOut != NULL);
    
    (void)pContext;
    (void)deviceType;

    if (ioctl(fd, AUDIO_GETDEV, &fdDevice) < 0) {
        return MA_ERROR;   /* Failed to retrieve device info. */
    }

    /* Name. */
    ma_strcpy_s(pInfoOut->name, sizeof(pInfoOut->name), fdDevice.name);

#if !defined(MA_AUDIO4_USE_NEW_API)
    /* Supported formats. We get this by looking at the encodings. */
    for (;;) {
        audio_encoding_t encoding;
        ma_format format;

        ma_zero_object(&encoding);
        encoding.index = counter;
        if (ioctl(fd, AUDIO_GETENC, &encoding) < 0) {
            break;
        }

        format = ma_format_from_encoding__audio4(encoding.encoding, encoding.precision);
        if (format != ma_format_unknown) {
            pInfoOut->formats[pInfoOut->formatCount++] = format;
        }

        counter += 1;
    }

    if (ioctl(fd, AUDIO_GETINFO, &fdInfo) < 0) {
        return MA_ERROR;
    }

    if (deviceType == ma_device_type_playback) {
        pInfoOut->minChannels = fdInfo.play.channels; 
        pInfoOut->maxChannels = fdInfo.play.channels;
        pInfoOut->minSampleRate = fdInfo.play.sample_rate;
        pInfoOut->maxSampleRate = fdInfo.play.sample_rate;
    } else {
        pInfoOut->minChannels = fdInfo.record.channels;
        pInfoOut->maxChannels = fdInfo.record.channels;
        pInfoOut->minSampleRate = fdInfo.record.sample_rate;
        pInfoOut->maxSampleRate = fdInfo.record.sample_rate;
    }
#else
    if (ioctl(fd, AUDIO_GETPAR, &fdPar) < 0) {
        return MA_ERROR;
    }
    
    format = ma_format_from_swpar__audio4(&fdPar);
    if (format == ma_format_unknown) {
        return MA_FORMAT_NOT_SUPPORTED;
    }
    pInfoOut->formats[pInfoOut->formatCount++] = format;
    
    if (deviceType == ma_device_type_playback) {
        pInfoOut->minChannels = fdPar.pchan;
        pInfoOut->maxChannels = fdPar.pchan;
    } else {
        pInfoOut->minChannels = fdPar.rchan;
        pInfoOut->maxChannels = fdPar.rchan;
    }
    
    pInfoOut->minSampleRate = fdPar.rate;
    pInfoOut->maxSampleRate = fdPar.rate;
#endif
    
    return MA_SUCCESS;
}

ma_result ma_context_enumerate_devices__audio4(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    const int maxDevices = 64;
    char devpath[256];
    int iDevice;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);
    
    /*
    Every device will be named "/dev/audioN", with a "/dev/audioctlN" equivalent. We use the "/dev/audioctlN"
    version here since we can open it even when another process has control of the "/dev/audioN" device.
    */
    for (iDevice = 0; iDevice < maxDevices; ++iDevice) {
        struct stat st;
        int fd;
        ma_bool32 isTerminating = MA_FALSE;

        ma_strcpy_s(devpath, sizeof(devpath), "/dev/audioctl");
        ma_itoa_s(iDevice, devpath+strlen(devpath), sizeof(devpath)-strlen(devpath), 10);
    
        if (stat(devpath, &st) < 0) {
            break;
        }

        /* The device exists, but we need to check if it's usable as playback and/or capture. */
        
        /* Playback. */
        if (!isTerminating) {
            fd = open(devpath, O_RDONLY, 0);
            if (fd >= 0) {
                /* Supports playback. */
                ma_device_info deviceInfo;
                ma_zero_object(&deviceInfo);
                ma_construct_device_id__audio4(deviceInfo.id.audio4, sizeof(deviceInfo.id.audio4), "/dev/audio", iDevice);
                if (ma_context_get_device_info_from_fd__audio4(pContext, ma_device_type_playback, fd, &deviceInfo) == MA_SUCCESS) {
                    isTerminating = !callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
                }
                
                close(fd);
            }
        }
        
        /* Capture. */
        if (!isTerminating) {
            fd = open(devpath, O_WRONLY, 0);
            if (fd >= 0) {
                /* Supports capture. */
                ma_device_info deviceInfo;
                ma_zero_object(&deviceInfo);
                ma_construct_device_id__audio4(deviceInfo.id.audio4, sizeof(deviceInfo.id.audio4), "/dev/audio", iDevice);
                if (ma_context_get_device_info_from_fd__audio4(pContext, ma_device_type_capture, fd, &deviceInfo) == MA_SUCCESS) {
                    isTerminating = !callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
                }
                
                close(fd);
            }
        }
        
        if (isTerminating) {
            break;
        }
    }
    
    return MA_SUCCESS;
}

ma_result ma_context_get_device_info__audio4(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    int fd = -1;
    int deviceIndex = -1;
    char ctlid[256];
    ma_result result;

    ma_assert(pContext != NULL);
    (void)shareMode;
    
    /*
    We need to open the "/dev/audioctlN" device to get the info. To do this we need to extract the number
    from the device ID which will be in "/dev/audioN" format.
    */
    if (pDeviceID == NULL) {
        /* Default device. */
        ma_strcpy_s(ctlid, sizeof(ctlid), "/dev/audioctl");
    } else {
        /* Specific device. We need to convert from "/dev/audioN" to "/dev/audioctlN". */
        result = ma_extract_device_index_from_id__audio4(pDeviceID->audio4, "/dev/audio", &deviceIndex);
        if (result != MA_SUCCESS) {
            return result;
        }
        
        ma_construct_device_id__audio4(ctlid, sizeof(ctlid), "/dev/audioctl", deviceIndex);
    }
    
    fd = open(ctlid, (deviceType == ma_device_type_playback) ? O_WRONLY : O_RDONLY, 0);
    if (fd == -1) {
        return MA_NO_DEVICE;
    }
    
    if (deviceIndex == -1) {
        ma_strcpy_s(pDeviceInfo->id.audio4, sizeof(pDeviceInfo->id.audio4), "/dev/audio");
    } else {
        ma_construct_device_id__audio4(pDeviceInfo->id.audio4, sizeof(pDeviceInfo->id.audio4), "/dev/audio", deviceIndex);
    }
    
    result = ma_context_get_device_info_from_fd__audio4(pContext, deviceType, fd, pDeviceInfo);
    
    close(fd);
    return result;
}

void ma_device_uninit__audio4(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        close(pDevice->audio4.fdCapture);
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        close(pDevice->audio4.fdPlayback);
    }
}

ma_result ma_device_init_fd__audio4(ma_context* pContext, const ma_device_config* pConfig, ma_device_type deviceType, ma_device* pDevice)
{
    const char* pDefaultDeviceNames[] = {
        "/dev/audio",
        "/dev/audio0"
    };
    int fd;
    int fdFlags = 0;
#if !defined(MA_AUDIO4_USE_NEW_API)    /* Old API */
    audio_info_t fdInfo;
#else
    struct audio_swpar fdPar;
#endif
    ma_format internalFormat;
    ma_uint32 internalChannels;
    ma_uint32 internalSampleRate;
    ma_uint32 internalBufferSizeInFrames;
    ma_uint32 internalPeriods;

    ma_assert(pContext   != NULL);
    ma_assert(pConfig    != NULL);
    ma_assert(deviceType != ma_device_type_duplex);
    ma_assert(pDevice    != NULL);

    (void)pContext;

    /* The first thing to do is open the file. */
    if (deviceType == ma_device_type_capture) {
        fdFlags = O_RDONLY;
    } else {
        fdFlags = O_WRONLY;
    }
    fdFlags |= O_NONBLOCK;

    if ((deviceType == ma_device_type_capture && pConfig->capture.pDeviceID == NULL) || (deviceType == ma_device_type_playback && pConfig->playback.pDeviceID == NULL)) {
        /* Default device. */
        for (size_t iDevice = 0; iDevice < ma_countof(pDefaultDeviceNames); ++iDevice) {
            fd = open(pDefaultDeviceNames[iDevice], fdFlags, 0);
            if (fd != -1) {
                break;
            }
        }
    } else {
        /* Specific device. */
        fd = open((deviceType == ma_device_type_capture) ? pConfig->capture.pDeviceID->audio4 : pConfig->playback.pDeviceID->audio4, fdFlags, 0);
    }

    if (fd == -1) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] Failed to open device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

#if !defined(MA_AUDIO4_USE_NEW_API)    /* Old API */
    AUDIO_INITINFO(&fdInfo);

    /* We get the driver to do as much of the data conversion as possible. */
    if (deviceType == ma_device_type_capture) {
        fdInfo.mode = AUMODE_RECORD;
        ma_encoding_from_format__audio4(pConfig->capture.format, &fdInfo.record.encoding, &fdInfo.record.precision);
        fdInfo.record.channels    = pConfig->capture.channels;
        fdInfo.record.sample_rate = pConfig->sampleRate;
    } else {
        fdInfo.mode = AUMODE_PLAY;
        ma_encoding_from_format__audio4(pConfig->playback.format, &fdInfo.play.encoding, &fdInfo.play.precision);
        fdInfo.play.channels    = pConfig->playback.channels;
        fdInfo.play.sample_rate = pConfig->sampleRate;
    }

    if (ioctl(fd, AUDIO_SETINFO, &fdInfo) < 0) {
        close(fd);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] Failed to set device format. AUDIO_SETINFO failed.", MA_FORMAT_NOT_SUPPORTED);
    }
    
    if (ioctl(fd, AUDIO_GETINFO, &fdInfo) < 0) {
        close(fd);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] AUDIO_GETINFO failed.", MA_FORMAT_NOT_SUPPORTED);
    }

    if (deviceType == ma_device_type_capture) {
        internalFormat     = ma_format_from_prinfo__audio4(&fdInfo.record);
        internalChannels   = fdInfo.record.channels;
        internalSampleRate = fdInfo.record.sample_rate;
    } else {
        internalFormat     = ma_format_from_prinfo__audio4(&fdInfo.play);
        internalChannels   = fdInfo.play.channels;
        internalSampleRate = fdInfo.play.sample_rate;
    }

    if (internalFormat == ma_format_unknown) {
        close(fd);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] The device's internal device format is not supported by miniaudio. The device is unusable.", MA_FORMAT_NOT_SUPPORTED);
    }

    /* Buffer. */
    {
        ma_uint32 internalBufferSizeInBytes;

        internalBufferSizeInFrames = pConfig->bufferSizeInFrames;
        if (internalBufferSizeInFrames == 0) {
            internalBufferSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(pConfig->bufferSizeInMilliseconds, internalSampleRate);
        }

        internalBufferSizeInBytes = internalBufferSizeInFrames * ma_get_bytes_per_frame(internalFormat, internalChannels);
        if (internalBufferSizeInBytes < 16) {
            internalBufferSizeInBytes = 16;
        }

        internalPeriods = pConfig->periods;
        if (internalPeriods < 2) {
            internalPeriods = 2;
        }

        /* What miniaudio calls a fragment, audio4 calls a block. */
        AUDIO_INITINFO(&fdInfo);
        fdInfo.hiwat     = internalPeriods;
        fdInfo.lowat     = internalPeriods-1;
        fdInfo.blocksize = internalBufferSizeInBytes / internalPeriods;
        if (ioctl(fd, AUDIO_SETINFO, &fdInfo) < 0) {
            close(fd);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] Failed to set internal buffer size. AUDIO_SETINFO failed.", MA_FORMAT_NOT_SUPPORTED);
        }

        internalPeriods            = fdInfo.hiwat;
        internalBufferSizeInFrames = (fdInfo.blocksize * fdInfo.hiwat) / ma_get_bytes_per_frame(internalFormat, internalChannels);
    }
#else
    /* We need to retrieve the format of the device so we can know the channel count and sample rate. Then we can calculate the buffer size. */
    if (ioctl(fd, AUDIO_GETPAR, &fdPar) < 0) {
        close(fd);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] Failed to retrieve initial device parameters.", MA_FORMAT_NOT_SUPPORTED);
    }

    internalFormat     = ma_format_from_swpar__audio4(&fdPar);
    internalChannels   = (deviceType == ma_device_type_capture) ? fdPar.rchan : fdPar.pchan;
    internalSampleRate = fdPar.rate;

    if (internalFormat == ma_format_unknown) {
        close(fd);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] The device's internal device format is not supported by miniaudio. The device is unusable.", MA_FORMAT_NOT_SUPPORTED);
    }

    /* Buffer. */
    {
        ma_uint32 internalBufferSizeInBytes;

        internalBufferSizeInFrames = pConfig->bufferSizeInFrames;
        if (internalBufferSizeInFrames == 0) {
            internalBufferSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(pConfig->bufferSizeInMilliseconds, internalSampleRate);
        }

        /* What miniaudio calls a fragment, audio4 calls a block. */
        internalBufferSizeInBytes = internalBufferSizeInFrames * ma_get_bytes_per_frame(internalFormat, internalChannels);
        if (internalBufferSizeInBytes < 16) {
            internalBufferSizeInBytes = 16;
        }
    
        fdPar.nblks = pConfig->periods;
        fdPar.round = internalBufferSizeInBytes / fdPar.nblks;
    
        if (ioctl(fd, AUDIO_SETPAR, &fdPar) < 0) {
            close(fd);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] Failed to set device parameters.", MA_FORMAT_NOT_SUPPORTED);
        }

        if (ioctl(fd, AUDIO_GETPAR, &fdPar) < 0) {
            close(fd);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] Failed to retrieve actual device parameters.", MA_FORMAT_NOT_SUPPORTED);
        }
    }

    internalFormat             = ma_format_from_swpar__audio4(&fdPar);
    internalChannels           = (deviceType == ma_device_type_capture) ? fdPar.rchan : fdPar.pchan;
    internalSampleRate         = fdPar.rate;
    internalPeriods            = fdPar.nblks;
    internalBufferSizeInFrames = (fdPar.nblks * fdPar.round) / ma_get_bytes_per_frame(internalFormat, internalChannels);
#endif

    if (internalFormat == ma_format_unknown) {
        close(fd);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] The device's internal device format is not supported by miniaudio. The device is unusable.", MA_FORMAT_NOT_SUPPORTED);
    }

    if (deviceType == ma_device_type_capture) {
        pDevice->audio4.fdCapture                    = fd;
        pDevice->capture.internalFormat              = internalFormat;
        pDevice->capture.internalChannels            = internalChannels;
        pDevice->capture.internalSampleRate          = internalSampleRate;
        ma_get_standard_channel_map(ma_standard_channel_map_sound4, internalChannels, pDevice->capture.internalChannelMap);
        pDevice->capture.internalBufferSizeInFrames  = internalBufferSizeInFrames;
        pDevice->capture.internalPeriods             = internalPeriods;
    } else {
        pDevice->audio4.fdPlayback                   = fd;
        pDevice->playback.internalFormat             = internalFormat;
        pDevice->playback.internalChannels           = internalChannels;
        pDevice->playback.internalSampleRate         = internalSampleRate;
        ma_get_standard_channel_map(ma_standard_channel_map_sound4, internalChannels, pDevice->playback.internalChannelMap);
        pDevice->playback.internalBufferSizeInFrames = internalBufferSizeInFrames;
        pDevice->playback.internalPeriods            = internalPeriods;
    }

    return MA_SUCCESS;
}

ma_result ma_device_init__audio4(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    ma_zero_object(&pDevice->audio4);
    
    pDevice->audio4.fdCapture  = -1;
    pDevice->audio4.fdPlayback = -1;

    /*
    The version of the operating system dictates whether or not the device is exclusive or shared. NetBSD
    introduced in-kernel mixing which means it's shared. All other BSD flavours are exclusive as far as
    I'm aware.
    */
#if defined(__NetBSD_Version__) && __NetBSD_Version__ >= 800000000
    /* NetBSD 8.0+ */
    if (((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.shareMode == ma_share_mode_exclusive) ||
        ((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.shareMode  == ma_share_mode_exclusive)) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }
#else
    /* All other flavors. */
#endif

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_fd__audio4(pContext, pConfig, ma_device_type_capture, pDevice);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_fd__audio4(pContext, pConfig, ma_device_type_playback, pDevice);
        if (result != MA_SUCCESS) {
            if (pConfig->deviceType == ma_device_type_duplex) {
                close(pDevice->audio4.fdCapture);
            }
            return result;
        }
    }

    return MA_SUCCESS;
}

#if 0
ma_result ma_device_start__audio4(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        if (pDevice->audio4.fdCapture == -1) {
            return MA_INVALID_ARGS;
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        if (pDevice->audio4.fdPlayback == -1) {
            return MA_INVALID_ARGS;
        }
    }

    return MA_SUCCESS;
}
#endif

ma_result ma_device_stop_fd__audio4(ma_device* pDevice, int fd)
{
    if (fd == -1) {
        return MA_INVALID_ARGS;
    }

#if !defined(MA_AUDIO4_USE_NEW_API)
    if (ioctl(fd, AUDIO_FLUSH, 0) < 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] Failed to stop device. AUDIO_FLUSH failed.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
    }
#else
    if (ioctl(fd, AUDIO_STOP, 0) < 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] Failed to stop device. AUDIO_STOP failed.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
    }
#endif

    return MA_SUCCESS;
}

ma_result ma_device_stop__audio4(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ma_result result = ma_device_stop_fd__audio4(pDevice, pDevice->audio4.fdCapture);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ma_result result = ma_device_stop_fd__audio4(pDevice, pDevice->audio4.fdPlayback);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_write__audio4(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount)
{
    int result = write(pDevice->audio4.fdPlayback, pPCMFrames, frameCount * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));
    if (result < 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] Failed to write data to the device.", MA_FAILED_TO_SEND_DATA_TO_DEVICE);
    }

    return MA_SUCCESS;
}

ma_result ma_device_read__audio4(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount)
{
    int result = read(pDevice->audio4.fdCapture, pPCMFrames, frameCount * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));
    if (result < 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[audio4] Failed to read data from the device.", MA_FAILED_TO_READ_DATA_FROM_DEVICE);
    }

    return MA_SUCCESS;
}

ma_result ma_context_uninit__audio4(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_audio4);

    (void)pContext;
    return MA_SUCCESS;
}

ma_result ma_context_init__audio4(const ma_context_config* pConfig, ma_context* pContext)
{
    ma_assert(pContext != NULL);

    (void)pConfig;

    pContext->onUninit              = ma_context_uninit__audio4;
    pContext->onDeviceIDEqual       = ma_context_is_device_id_equal__audio4;
    pContext->onEnumDevices         = ma_context_enumerate_devices__audio4;
    pContext->onGetDeviceInfo       = ma_context_get_device_info__audio4;
    pContext->onDeviceInit          = ma_device_init__audio4;
    pContext->onDeviceUninit        = ma_device_uninit__audio4;
    pContext->onDeviceStart         = NULL;
    pContext->onDeviceStop          = ma_device_stop__audio4;
    pContext->onDeviceWrite         = ma_device_write__audio4;
    pContext->onDeviceRead          = ma_device_read__audio4;

    return MA_SUCCESS;
}
#endif  /* audio4 */


/******************************************************************************

OSS Backend

******************************************************************************/
#ifdef MA_HAS_OSS
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/soundcard.h>

#ifndef SNDCTL_DSP_HALT
#define SNDCTL_DSP_HALT SNDCTL_DSP_RESET
#endif

int ma_open_temp_device__oss()
{
    /* The OSS sample code uses "/dev/mixer" as the device for getting system properties so I'm going to do the same. */
    int fd = open("/dev/mixer", O_RDONLY, 0);
    if (fd >= 0) {
        return fd;
    }

    return -1;
}

ma_result ma_context_open_device__oss(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, int* pfd)
{
    const char* deviceName;
    int flags;

    ma_assert(pContext != NULL);
    ma_assert(pfd != NULL);
    (void)pContext;

    *pfd = -1;

    /* This function should only be called for playback or capture, not duplex. */
    if (deviceType == ma_device_type_duplex) {
        return MA_INVALID_ARGS;
    }

    deviceName = "/dev/dsp";
    if (pDeviceID != NULL) {
        deviceName = pDeviceID->oss;
    }

    flags = (deviceType == ma_device_type_playback) ? O_WRONLY : O_RDONLY;
    if (shareMode == ma_share_mode_exclusive) {
        flags |= O_EXCL;
    }

    *pfd = open(deviceName, flags, 0);
    if (*pfd == -1) {
        return MA_FAILED_TO_OPEN_BACKEND_DEVICE;
    }

    return MA_SUCCESS;
}

ma_bool32 ma_context_is_device_id_equal__oss(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return ma_strcmp(pID0->oss, pID1->oss) == 0;
}

ma_result ma_context_enumerate_devices__oss(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    int fd;
    oss_sysinfo si;
    int result;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);

    fd = ma_open_temp_device__oss();
    if (fd == -1) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[OSS] Failed to open a temporary device for retrieving system information used for device enumeration.", MA_NO_BACKEND);
    }

    result = ioctl(fd, SNDCTL_SYSINFO, &si);
    if (result != -1) {
        int iAudioDevice;
        for (iAudioDevice = 0; iAudioDevice < si.numaudios; ++iAudioDevice) {
            oss_audioinfo ai;
            ai.dev = iAudioDevice;
            result = ioctl(fd, SNDCTL_AUDIOINFO, &ai);
            if (result != -1) {
                if (ai.devnode[0] != '\0') {    /* <-- Can be blank, according to documentation. */
                    ma_device_info deviceInfo;
                    ma_bool32 isTerminating = MA_FALSE;

                    ma_zero_object(&deviceInfo);

                    /* ID */
                    ma_strncpy_s(deviceInfo.id.oss, sizeof(deviceInfo.id.oss), ai.devnode, (size_t)-1);

                    /*
                    The human readable device name should be in the "ai.handle" variable, but it can
                    sometimes be empty in which case we just fall back to "ai.name" which is less user
                    friendly, but usually has a value.
                    */
                    if (ai.handle[0] != '\0') {
                        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), ai.handle, (size_t)-1);
                    } else {
                        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), ai.name, (size_t)-1);
                    }

                    /* The device can be both playback and capture. */
                    if (!isTerminating && (ai.caps & PCM_CAP_OUTPUT) != 0) {
                        isTerminating = !callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
                    }
                    if (!isTerminating && (ai.caps & PCM_CAP_INPUT) != 0) {
                        isTerminating = !callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
                    }

                    if (isTerminating) {
                        break;
                    }
                }
            }
        }
    } else {
        close(fd);
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[OSS] Failed to retrieve system information for device enumeration.", MA_NO_BACKEND);
    }

    close(fd);
    return MA_SUCCESS;
}

ma_result ma_context_get_device_info__oss(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_bool32 foundDevice;
    int fdTemp;
    oss_sysinfo si;
    int result;

    ma_assert(pContext != NULL);
    (void)shareMode;

    /* Handle the default device a little differently. */
    if (pDeviceID == NULL) {
        if (deviceType == ma_device_type_playback) {
            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
        } else {
            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
        }

        return MA_SUCCESS;
    }


    /* If we get here it means we are _not_ using the default device. */
    foundDevice = MA_FALSE;

    fdTemp = ma_open_temp_device__oss();
    if (fdTemp == -1) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[OSS] Failed to open a temporary device for retrieving system information used for device enumeration.", MA_NO_BACKEND);
    }

    result = ioctl(fdTemp, SNDCTL_SYSINFO, &si);
    if (result != -1) {
        int iAudioDevice;
        for (iAudioDevice = 0; iAudioDevice < si.numaudios; ++iAudioDevice) {
            oss_audioinfo ai;
            ai.dev = iAudioDevice;
            result = ioctl(fdTemp, SNDCTL_AUDIOINFO, &ai);
            if (result != -1) {
                if (ma_strcmp(ai.devnode, pDeviceID->oss) == 0) {
                    /* It has the same name, so now just confirm the type. */
                    if ((deviceType == ma_device_type_playback && ((ai.caps & PCM_CAP_OUTPUT) != 0)) ||
                        (deviceType == ma_device_type_capture  && ((ai.caps & PCM_CAP_INPUT)  != 0))) {
                        unsigned int formatMask;

                        /* ID */
                        ma_strncpy_s(pDeviceInfo->id.oss, sizeof(pDeviceInfo->id.oss), ai.devnode, (size_t)-1);

                        /*
                        The human readable device name should be in the "ai.handle" variable, but it can
                        sometimes be empty in which case we just fall back to "ai.name" which is less user
                        friendly, but usually has a value.
                        */
                        if (ai.handle[0] != '\0') {
                            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), ai.handle, (size_t)-1);
                        } else {
                            ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), ai.name, (size_t)-1);
                        }

                        pDeviceInfo->minChannels = ai.min_channels;
                        pDeviceInfo->maxChannels = ai.max_channels;
                        pDeviceInfo->minSampleRate = ai.min_rate;
                        pDeviceInfo->maxSampleRate = ai.max_rate;
                        pDeviceInfo->formatCount = 0;

                        if (deviceType == ma_device_type_playback) {
                            formatMask = ai.oformats;
                        } else {
                            formatMask = ai.iformats;
                        }

                        if ((formatMask & AFMT_U8) != 0) {
                            pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_u8;
                        }
                        if (((formatMask & AFMT_S16_LE) != 0 && ma_is_little_endian()) || (AFMT_S16_BE && ma_is_big_endian())) {
                            pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_s16;
                        }
                        if (((formatMask & AFMT_S32_LE) != 0 && ma_is_little_endian()) || (AFMT_S32_BE && ma_is_big_endian())) {
                            pDeviceInfo->formats[pDeviceInfo->formatCount++] = ma_format_s32;
                        }

                        foundDevice = MA_TRUE;
                        break;
                    }
                }
            }
        }
    } else {
        close(fdTemp);
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[OSS] Failed to retrieve system information for device enumeration.", MA_NO_BACKEND);
    }


    close(fdTemp);

    if (!foundDevice) {
        return MA_NO_DEVICE;
    }

    return MA_SUCCESS;
}

void ma_device_uninit__oss(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        close(pDevice->oss.fdCapture);
    }
    
    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        close(pDevice->oss.fdPlayback);
    }
}

int ma_format_to_oss(ma_format format)
{
    int ossFormat = AFMT_U8;
    switch (format) {
        case ma_format_s16: ossFormat = (ma_is_little_endian()) ? AFMT_S16_LE : AFMT_S16_BE; break;
        case ma_format_s24: ossFormat = (ma_is_little_endian()) ? AFMT_S32_LE : AFMT_S32_BE; break;
        case ma_format_s32: ossFormat = (ma_is_little_endian()) ? AFMT_S32_LE : AFMT_S32_BE; break;
        case ma_format_f32: ossFormat = (ma_is_little_endian()) ? AFMT_S16_LE : AFMT_S16_BE; break;
        case ma_format_u8:
        default: ossFormat = AFMT_U8; break;
    }

    return ossFormat;
}

ma_format ma_format_from_oss(int ossFormat)
{
    if (ossFormat == AFMT_U8) {
        return ma_format_u8;
    } else {
        if (ma_is_little_endian()) {
            switch (ossFormat) {
                case AFMT_S16_LE: return ma_format_s16;
                case AFMT_S32_LE: return ma_format_s32;
                default: return ma_format_unknown;
            }
        } else {
            switch (ossFormat) {
                case AFMT_S16_BE: return ma_format_s16;
                case AFMT_S32_BE: return ma_format_s32;
                default: return ma_format_unknown;
            }
        }
    }

    return ma_format_unknown;
}

ma_result ma_device_init_fd__oss(ma_context* pContext, const ma_device_config* pConfig, ma_device_type deviceType, ma_device* pDevice)
{
    ma_result result;
    int ossResult;
    int fd;
    const ma_device_id* pDeviceID = NULL;
    ma_share_mode shareMode;
    int ossFormat;
    int ossChannels;
    int ossSampleRate;
    int ossFragment;

    ma_assert(pContext != NULL);
    ma_assert(pConfig != NULL);
    ma_assert(deviceType != ma_device_type_duplex);
    ma_assert(pDevice != NULL);

    (void)pContext;

    if (deviceType == ma_device_type_capture) {
        pDeviceID     = pConfig->capture.pDeviceID;
        shareMode     = pConfig->capture.shareMode;
        ossFormat     = ma_format_to_oss(pConfig->capture.format);
        ossChannels   = (int)pConfig->capture.channels;
        ossSampleRate = (int)pConfig->sampleRate;
    } else {
        pDeviceID     = pConfig->playback.pDeviceID;
        shareMode     = pConfig->playback.shareMode;
        ossFormat     = ma_format_to_oss(pConfig->playback.format);
        ossChannels   = (int)pConfig->playback.channels;
        ossSampleRate = (int)pConfig->sampleRate;
    }

    result = ma_context_open_device__oss(pContext, deviceType, pDeviceID, shareMode, &fd);
    if (result != MA_SUCCESS) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to open device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
    }

    /*
    The OSS documantation is very clear about the order we should be initializing the device's properties:
      1) Format
      2) Channels
      3) Sample rate.
    */

    /* Format. */
    ossResult = ioctl(fd, SNDCTL_DSP_SETFMT, &ossFormat);
    if (ossResult == -1) {
        close(fd);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to set format.", MA_FORMAT_NOT_SUPPORTED);
    }

    /* Channels. */
    ossResult = ioctl(fd, SNDCTL_DSP_CHANNELS, &ossChannels);
    if (ossResult == -1) {
        close(fd);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to set channel count.", MA_FORMAT_NOT_SUPPORTED);
    }

    /* Sample Rate. */
    ossResult = ioctl(fd, SNDCTL_DSP_SPEED, &ossSampleRate);
    if (ossResult == -1) {
        close(fd);
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to set sample rate.", MA_FORMAT_NOT_SUPPORTED);
    }

    /*
    Buffer.

    The documentation says that the fragment settings should be set as soon as possible, but I'm not sure if
    it should be done before or after format/channels/rate.
    
    OSS wants the fragment size in bytes and a power of 2. When setting, we specify the power, not the actual
    value.
    */
    {
        ma_uint32 fragmentSizeInBytes;
        ma_uint32 bufferSizeInFrames;
        ma_uint32 ossFragmentSizePower;
        
        bufferSizeInFrames = pConfig->bufferSizeInFrames;
        if (bufferSizeInFrames == 0) {
            bufferSizeInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(pConfig->bufferSizeInMilliseconds, (ma_uint32)ossSampleRate);
        }

        fragmentSizeInBytes = ma_round_to_power_of_2((bufferSizeInFrames / pConfig->periods) * ma_get_bytes_per_frame(ma_format_from_oss(ossFormat), ossChannels));
        if (fragmentSizeInBytes < 16) {
            fragmentSizeInBytes = 16;
        }

        ossFragmentSizePower = 4;
        fragmentSizeInBytes >>= 4;
        while (fragmentSizeInBytes >>= 1) {
            ossFragmentSizePower += 1;
        }

        ossFragment = (int)((pConfig->periods << 16) | ossFragmentSizePower);
        ossResult = ioctl(fd, SNDCTL_DSP_SETFRAGMENT, &ossFragment);
        if (ossResult == -1) {
            close(fd);
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to set fragment size and period count.", MA_FORMAT_NOT_SUPPORTED);
        }
    }

    /* Internal settings. */
    if (deviceType == ma_device_type_capture) {
        pDevice->oss.fdCapture                       = fd;
        pDevice->capture.internalFormat              = ma_format_from_oss(ossFormat);
        pDevice->capture.internalChannels            = ossChannels;
        pDevice->capture.internalSampleRate          = ossSampleRate;
        ma_get_standard_channel_map(ma_standard_channel_map_sound4, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap);
        pDevice->capture.internalPeriods             = (ma_uint32)(ossFragment >> 16);
        pDevice->capture.internalBufferSizeInFrames  = (((ma_uint32)(1 << (ossFragment & 0xFFFF))) * pDevice->capture.internalPeriods) / ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels);

        if (pDevice->capture.internalFormat == ma_format_unknown) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] The device's internal format is not supported by miniaudio.", MA_FORMAT_NOT_SUPPORTED);
        }
    } else {
        pDevice->oss.fdPlayback                      = fd;
        pDevice->playback.internalFormat             = ma_format_from_oss(ossFormat);
        pDevice->playback.internalChannels           = ossChannels;
        pDevice->playback.internalSampleRate         = ossSampleRate;
        ma_get_standard_channel_map(ma_standard_channel_map_sound4, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap);
        pDevice->playback.internalPeriods            = (ma_uint32)(ossFragment >> 16);
        pDevice->playback.internalBufferSizeInFrames = (((ma_uint32)(1 << (ossFragment & 0xFFFF))) * pDevice->playback.internalPeriods) / ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels);

        if (pDevice->playback.internalFormat == ma_format_unknown) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] The device's internal format is not supported by miniaudio.", MA_FORMAT_NOT_SUPPORTED);
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_init__oss(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_assert(pContext != NULL);
    ma_assert(pConfig  != NULL);
    ma_assert(pDevice  != NULL);

    ma_zero_object(&pDevice->oss);

    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_fd__oss(pContext, pConfig, ma_device_type_capture, pDevice);
        if (result != MA_SUCCESS) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to open device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        ma_result result = ma_device_init_fd__oss(pContext, pConfig, ma_device_type_playback, pDevice);
        if (result != MA_SUCCESS) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to open device.", MA_FAILED_TO_OPEN_BACKEND_DEVICE);
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_stop__oss(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    /*
    We want to use SNDCTL_DSP_HALT. From the documentation:
    
      In multithreaded applications SNDCTL_DSP_HALT (SNDCTL_DSP_RESET) must only be called by the thread
      that actually reads/writes the audio device. It must not be called by some master thread to kill the
      audio thread. The audio thread will not stop or get any kind of notification that the device was
      stopped by the master thread. The device gets stopped but the next read or write call will silently
      restart the device.
    
    This is actually safe in our case, because this function is only ever called from within our worker
    thread anyway. Just keep this in mind, though...
    */

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        int result = ioctl(pDevice->oss.fdCapture, SNDCTL_DSP_HALT, 0);
        if (result == -1) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to stop device. SNDCTL_DSP_HALT failed.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        int result = ioctl(pDevice->oss.fdPlayback, SNDCTL_DSP_HALT, 0);
        if (result == -1) {
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to stop device. SNDCTL_DSP_HALT failed.", MA_FAILED_TO_STOP_BACKEND_DEVICE);
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_write__oss(ma_device* pDevice, const void* pPCMFrames, ma_uint32 frameCount)
{
    int resultOSS = write(pDevice->oss.fdPlayback, pPCMFrames, frameCount * ma_get_bytes_per_frame(pDevice->playback.internalFormat, pDevice->playback.internalChannels));
    if (resultOSS < 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to send data from the client to the device.", MA_FAILED_TO_SEND_DATA_TO_DEVICE);
    }
    
    return MA_SUCCESS;
}

ma_result ma_device_read__oss(ma_device* pDevice, void* pPCMFrames, ma_uint32 frameCount)
{
    int resultOSS = read(pDevice->oss.fdCapture, pPCMFrames, frameCount * ma_get_bytes_per_frame(pDevice->capture.internalFormat, pDevice->capture.internalChannels));
    if (resultOSS < 0) {
        return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[OSS] Failed to read data from the device to be sent to the client.", MA_FAILED_TO_READ_DATA_FROM_DEVICE);
    }
    
    return MA_SUCCESS;
}

ma_result ma_context_uninit__oss(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_oss);

    (void)pContext;
    return MA_SUCCESS;
}

ma_result ma_context_init__oss(const ma_context_config* pConfig, ma_context* pContext)
{
    int fd;
    int ossVersion;
    int result;

    ma_assert(pContext != NULL);

    (void)pConfig;

    /* Try opening a temporary device first so we can get version information. This is closed at the end. */
    fd = ma_open_temp_device__oss();
    if (fd == -1) {
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[OSS] Failed to open temporary device for retrieving system properties.", MA_NO_BACKEND);   /* Looks liks OSS isn't installed, or there are no available devices. */
    }

    /* Grab the OSS version. */
    ossVersion = 0;
    result = ioctl(fd, OSS_GETVERSION, &ossVersion);
    if (result == -1) {
        close(fd);
        return ma_context_post_error(pContext, NULL, MA_LOG_LEVEL_ERROR, "[OSS] Failed to retrieve OSS version.", MA_NO_BACKEND);
    }

    pContext->oss.versionMajor = ((ossVersion & 0xFF0000) >> 16);
    pContext->oss.versionMinor = ((ossVersion & 0x00FF00) >> 8);

    pContext->onUninit              = ma_context_uninit__oss;
    pContext->onDeviceIDEqual       = ma_context_is_device_id_equal__oss;
    pContext->onEnumDevices         = ma_context_enumerate_devices__oss;
    pContext->onGetDeviceInfo       = ma_context_get_device_info__oss;
    pContext->onDeviceInit          = ma_device_init__oss;
    pContext->onDeviceUninit        = ma_device_uninit__oss;
    pContext->onDeviceStart         = NULL; /* Not required for synchronous backends. */
    pContext->onDeviceStop          = ma_device_stop__oss;
    pContext->onDeviceWrite         = ma_device_write__oss;
    pContext->onDeviceRead          = ma_device_read__oss;

    close(fd);
    return MA_SUCCESS;
}
#endif  /* OSS */


/******************************************************************************

AAudio Backend

******************************************************************************/
#ifdef MA_HAS_AAUDIO
/*#include <AAudio/AAudio.h>*/

#define MA_AAUDIO_UNSPECIFIED 0

typedef int32_t ma_aaudio_result_t;
typedef int32_t ma_aaudio_direction_t;
typedef int32_t ma_aaudio_sharing_mode_t;
typedef int32_t ma_aaudio_format_t;
typedef int32_t ma_aaudio_stream_state_t;
typedef int32_t ma_aaudio_performance_mode_t;
typedef int32_t ma_aaudio_data_callback_result_t;

/* Result codes. miniaudio only cares about the success code. */
#define MA_AAUDIO_OK                               0

/* Directions. */
#define MA_AAUDIO_DIRECTION_OUTPUT                 0
#define MA_AAUDIO_DIRECTION_INPUT                  1

/* Sharing modes. */
#define MA_AAUDIO_SHARING_MODE_EXCLUSIVE           0
#define MA_AAUDIO_SHARING_MODE_SHARED              1

/* Formats. */
#define MA_AAUDIO_FORMAT_PCM_I16                   1
#define MA_AAUDIO_FORMAT_PCM_FLOAT                 2

/* Stream states. */
#define MA_AAUDIO_STREAM_STATE_UNINITIALIZED       0
#define MA_AAUDIO_STREAM_STATE_UNKNOWN             1
#define MA_AAUDIO_STREAM_STATE_OPEN                2
#define MA_AAUDIO_STREAM_STATE_STARTING            3
#define MA_AAUDIO_STREAM_STATE_STARTED             4
#define MA_AAUDIO_STREAM_STATE_PAUSING             5
#define MA_AAUDIO_STREAM_STATE_PAUSED              6
#define MA_AAUDIO_STREAM_STATE_FLUSHING            7
#define MA_AAUDIO_STREAM_STATE_FLUSHED             8
#define MA_AAUDIO_STREAM_STATE_STOPPING            9
#define MA_AAUDIO_STREAM_STATE_STOPPED             10
#define MA_AAUDIO_STREAM_STATE_CLOSING             11
#define MA_AAUDIO_STREAM_STATE_CLOSED              12
#define MA_AAUDIO_STREAM_STATE_DISCONNECTED        13

/* Performance modes. */
#define MA_AAUDIO_PERFORMANCE_MODE_NONE            10
#define MA_AAUDIO_PERFORMANCE_MODE_POWER_SAVING    11
#define MA_AAUDIO_PERFORMANCE_MODE_LOW_LATENCY     12

/* Callback results. */
#define MA_AAUDIO_CALLBACK_RESULT_CONTINUE         0
#define MA_AAUDIO_CALLBACK_RESULT_STOP             1

/* Objects. */
typedef struct ma_AAudioStreamBuilder_t* ma_AAudioStreamBuilder;
typedef struct ma_AAudioStream_t*        ma_AAudioStream;

typedef ma_aaudio_data_callback_result_t (*ma_AAudioStream_dataCallback)(ma_AAudioStream* pStream, void* pUserData, void* pAudioData, int32_t numFrames);

typedef ma_aaudio_result_t       (* MA_PFN_AAudio_createStreamBuilder)                   (ma_AAudioStreamBuilder** ppBuilder);
typedef ma_aaudio_result_t       (* MA_PFN_AAudioStreamBuilder_delete)                   (ma_AAudioStreamBuilder* pBuilder);
typedef void                     (* MA_PFN_AAudioStreamBuilder_setDeviceId)              (ma_AAudioStreamBuilder* pBuilder, int32_t deviceId);
typedef void                     (* MA_PFN_AAudioStreamBuilder_setDirection)             (ma_AAudioStreamBuilder* pBuilder, ma_aaudio_direction_t direction);
typedef void                     (* MA_PFN_AAudioStreamBuilder_setSharingMode)           (ma_AAudioStreamBuilder* pBuilder, ma_aaudio_sharing_mode_t sharingMode);
typedef void                     (* MA_PFN_AAudioStreamBuilder_setFormat)                (ma_AAudioStreamBuilder* pBuilder, ma_aaudio_format_t format);
typedef void                     (* MA_PFN_AAudioStreamBuilder_setChannelCount)          (ma_AAudioStreamBuilder* pBuilder, int32_t channelCount);
typedef void                     (* MA_PFN_AAudioStreamBuilder_setSampleRate)            (ma_AAudioStreamBuilder* pBuilder, int32_t sampleRate);
typedef void                     (* MA_PFN_AAudioStreamBuilder_setBufferCapacityInFrames)(ma_AAudioStreamBuilder* pBuilder, int32_t numFrames);
typedef void                     (* MA_PFN_AAudioStreamBuilder_setFramesPerDataCallback) (ma_AAudioStreamBuilder* pBuilder, int32_t numFrames);
typedef void                     (* MA_PFN_AAudioStreamBuilder_setDataCallback)          (ma_AAudioStreamBuilder* pBuilder, ma_AAudioStream_dataCallback callback, void* pUserData);
typedef void                     (* MA_PFN_AAudioStreamBuilder_setPerformanceMode)       (ma_AAudioStreamBuilder* pBuilder, ma_aaudio_performance_mode_t mode);
typedef ma_aaudio_result_t       (* MA_PFN_AAudioStreamBuilder_openStream)               (ma_AAudioStreamBuilder* pBuilder, ma_AAudioStream** ppStream);
typedef ma_aaudio_result_t       (* MA_PFN_AAudioStream_close)                           (ma_AAudioStream* pStream);
typedef ma_aaudio_stream_state_t (* MA_PFN_AAudioStream_getState)                        (ma_AAudioStream* pStream);
typedef ma_aaudio_result_t       (* MA_PFN_AAudioStream_waitForStateChange)              (ma_AAudioStream* pStream, ma_aaudio_stream_state_t inputState, ma_aaudio_stream_state_t* pNextState, int64_t timeoutInNanoseconds);
typedef ma_aaudio_format_t       (* MA_PFN_AAudioStream_getFormat)                       (ma_AAudioStream* pStream);
typedef int32_t                  (* MA_PFN_AAudioStream_getChannelCount)                 (ma_AAudioStream* pStream);
typedef int32_t                  (* MA_PFN_AAudioStream_getSampleRate)                   (ma_AAudioStream* pStream);
typedef int32_t                  (* MA_PFN_AAudioStream_getBufferCapacityInFrames)       (ma_AAudioStream* pStream);
typedef int32_t                  (* MA_PFN_AAudioStream_getFramesPerDataCallback)        (ma_AAudioStream* pStream);
typedef int32_t                  (* MA_PFN_AAudioStream_getFramesPerBurst)               (ma_AAudioStream* pStream);
typedef ma_aaudio_result_t       (* MA_PFN_AAudioStream_requestStart)                    (ma_AAudioStream* pStream);
typedef ma_aaudio_result_t       (* MA_PFN_AAudioStream_requestStop)                     (ma_AAudioStream* pStream);

ma_result ma_result_from_aaudio(ma_aaudio_result_t resultAA)
{
    switch (resultAA)
    {
        case MA_AAUDIO_OK: return MA_SUCCESS;
        default: break;
    }

    return MA_ERROR;
}

ma_aaudio_data_callback_result_t ma_stream_data_callback_capture__aaudio(ma_AAudioStream* pStream, void* pUserData, void* pAudioData, int32_t frameCount)
{
    ma_device* pDevice = (ma_device*)pUserData;
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_duplex) {
        ma_device__handle_duplex_callback_capture(pDevice, frameCount, pAudioData, &pDevice->aaudio.duplexRB);
    } else {
        ma_device__send_frames_to_client(pDevice, frameCount, pAudioData);     /* Send directly to the client. */
    }

    (void)pStream;
    return MA_AAUDIO_CALLBACK_RESULT_CONTINUE;
}

ma_aaudio_data_callback_result_t ma_stream_data_callback_playback__aaudio(ma_AAudioStream* pStream, void* pUserData, void* pAudioData, int32_t frameCount)
{
    ma_device* pDevice = (ma_device*)pUserData;
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_duplex) {
        ma_device__handle_duplex_callback_playback(pDevice, frameCount, pAudioData, &pDevice->aaudio.duplexRB);
    } else {
        ma_device__read_frames_from_client(pDevice, frameCount, pAudioData);   /* Read directly from the client. */
    }

    (void)pStream;
    return MA_AAUDIO_CALLBACK_RESULT_CONTINUE;
}

ma_result ma_open_stream__aaudio(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, const ma_device_config* pConfig, const ma_device* pDevice, ma_AAudioStream** ppStream)
{
    ma_AAudioStreamBuilder* pBuilder;
    ma_aaudio_result_t resultAA;

    ma_assert(deviceType != ma_device_type_duplex);   /* This function should not be called for a full-duplex device type. */

    *ppStream = NULL;

    resultAA = ((MA_PFN_AAudio_createStreamBuilder)pContext->aaudio.AAudio_createStreamBuilder)(&pBuilder);
    if (resultAA != MA_AAUDIO_OK) {
        return ma_result_from_aaudio(resultAA);
    }

    if (pDeviceID != NULL) {
        ((MA_PFN_AAudioStreamBuilder_setDeviceId)pContext->aaudio.AAudioStreamBuilder_setDeviceId)(pBuilder, pDeviceID->aaudio);
    }

    ((MA_PFN_AAudioStreamBuilder_setDirection)pContext->aaudio.AAudioStreamBuilder_setDirection)(pBuilder, (deviceType == ma_device_type_playback) ? MA_AAUDIO_DIRECTION_OUTPUT : MA_AAUDIO_DIRECTION_INPUT);
    ((MA_PFN_AAudioStreamBuilder_setSharingMode)pContext->aaudio.AAudioStreamBuilder_setSharingMode)(pBuilder, (shareMode == ma_share_mode_shared) ? MA_AAUDIO_SHARING_MODE_SHARED : MA_AAUDIO_SHARING_MODE_EXCLUSIVE);

    if (pConfig != NULL) {
        ma_uint32 bufferCapacityInFrames;

        if (pDevice == NULL || !pDevice->usingDefaultSampleRate) {
            ((MA_PFN_AAudioStreamBuilder_setSampleRate)pContext->aaudio.AAudioStreamBuilder_setSampleRate)(pBuilder, pConfig->sampleRate);
        }

        if (deviceType == ma_device_type_capture) {
            if (pDevice == NULL || !pDevice->capture.usingDefaultChannels) {
                ((MA_PFN_AAudioStreamBuilder_setChannelCount)pContext->aaudio.AAudioStreamBuilder_setChannelCount)(pBuilder, pConfig->capture.channels);
            }
            if (pDevice == NULL || !pDevice->capture.usingDefaultFormat) {
                ((MA_PFN_AAudioStreamBuilder_setFormat)pContext->aaudio.AAudioStreamBuilder_setFormat)(pBuilder, (pConfig->capture.format == ma_format_s16) ? MA_AAUDIO_FORMAT_PCM_I16 : MA_AAUDIO_FORMAT_PCM_FLOAT);
            }
        } else {
            if (pDevice == NULL || !pDevice->playback.usingDefaultChannels) {
                ((MA_PFN_AAudioStreamBuilder_setChannelCount)pContext->aaudio.AAudioStreamBuilder_setChannelCount)(pBuilder, pConfig->playback.channels);
            }
            if (pDevice == NULL || !pDevice->playback.usingDefaultFormat) {
                ((MA_PFN_AAudioStreamBuilder_setFormat)pContext->aaudio.AAudioStreamBuilder_setFormat)(pBuilder, (pConfig->playback.format == ma_format_s16) ? MA_AAUDIO_FORMAT_PCM_I16 : MA_AAUDIO_FORMAT_PCM_FLOAT);
            }
        }

        bufferCapacityInFrames = pConfig->bufferSizeInFrames;
        if (bufferCapacityInFrames == 0) {
            bufferCapacityInFrames = ma_calculate_buffer_size_in_frames_from_milliseconds(pConfig->bufferSizeInMilliseconds, pConfig->sampleRate);
        }
        ((MA_PFN_AAudioStreamBuilder_setBufferCapacityInFrames)pContext->aaudio.AAudioStreamBuilder_setBufferCapacityInFrames)(pBuilder, bufferCapacityInFrames);

        ((MA_PFN_AAudioStreamBuilder_setFramesPerDataCallback)pContext->aaudio.AAudioStreamBuilder_setFramesPerDataCallback)(pBuilder, bufferCapacityInFrames / pConfig->periods);

        if (deviceType == ma_device_type_capture) {
            ((MA_PFN_AAudioStreamBuilder_setDataCallback)pContext->aaudio.AAudioStreamBuilder_setDataCallback)(pBuilder, ma_stream_data_callback_capture__aaudio, (void*)pDevice);
        } else {
            ((MA_PFN_AAudioStreamBuilder_setDataCallback)pContext->aaudio.AAudioStreamBuilder_setDataCallback)(pBuilder, ma_stream_data_callback_playback__aaudio, (void*)pDevice);
        }

        /* Not sure how this affects things, but since there's a mapping between miniaudio's performance profiles and AAudio's performance modes, let go ahead and set it. */
        ((MA_PFN_AAudioStreamBuilder_setPerformanceMode)pContext->aaudio.AAudioStreamBuilder_setPerformanceMode)(pBuilder, (pConfig->performanceProfile == ma_performance_profile_low_latency) ? MA_AAUDIO_PERFORMANCE_MODE_LOW_LATENCY : MA_AAUDIO_PERFORMANCE_MODE_NONE);
    }

    resultAA = ((MA_PFN_AAudioStreamBuilder_openStream)pContext->aaudio.AAudioStreamBuilder_openStream)(pBuilder, ppStream);
    if (resultAA != MA_AAUDIO_OK) {
        *ppStream = NULL;
        ((MA_PFN_AAudioStreamBuilder_delete)pContext->aaudio.AAudioStreamBuilder_delete)(pBuilder);
        return ma_result_from_aaudio(resultAA);
    }

    ((MA_PFN_AAudioStreamBuilder_delete)pContext->aaudio.AAudioStreamBuilder_delete)(pBuilder);
    return MA_SUCCESS;
}

ma_result ma_close_stream__aaudio(ma_context* pContext, ma_AAudioStream* pStream)
{
    return ma_result_from_aaudio(((MA_PFN_AAudioStream_close)pContext->aaudio.AAudioStream_close)(pStream));
}

ma_bool32 ma_has_default_device__aaudio(ma_context* pContext, ma_device_type deviceType)
{
    /* The only way to know this is to try creating a stream. */
    ma_AAudioStream* pStream;
    ma_result result = ma_open_stream__aaudio(pContext, deviceType, NULL, ma_share_mode_shared, NULL, NULL, &pStream);
    if (result != MA_SUCCESS) {
        return MA_FALSE;
    }

    ma_close_stream__aaudio(pContext, pStream);
    return MA_TRUE;
}

ma_result ma_wait_for_simple_state_transition__aaudio(ma_context* pContext, ma_AAudioStream* pStream, ma_aaudio_stream_state_t oldState, ma_aaudio_stream_state_t newState)
{
    ma_aaudio_stream_state_t actualNewState;
    ma_aaudio_result_t resultAA = ((MA_PFN_AAudioStream_waitForStateChange)pContext->aaudio.AAudioStream_waitForStateChange)(pStream, oldState, &actualNewState, 5000000000); /* 5 second timeout. */
    if (resultAA != MA_AAUDIO_OK) {
        return ma_result_from_aaudio(resultAA);
    }

    if (newState != actualNewState) {
        return MA_ERROR;   /* Failed to transition into the expected state. */
    }

    return MA_SUCCESS;
}


ma_bool32 ma_context_is_device_id_equal__aaudio(ma_context* pContext, const ma_device_id* pID0, const ma_device_id* pID1)
{
    ma_assert(pContext != NULL);
    ma_assert(pID0 != NULL);
    ma_assert(pID1 != NULL);
    (void)pContext;

    return pID0->aaudio == pID1->aaudio;
}

ma_result ma_context_enumerate_devices__aaudio(ma_context* pContext, ma_enum_devices_callback_proc callback, void* pUserData)
{
    ma_bool32 cbResult = MA_TRUE;

    ma_assert(pContext != NULL);
    ma_assert(callback != NULL);

    /* Unfortunately AAudio does not have an enumeration API. Therefore I'm only going to report default devices, but only if it can instantiate a stream. */

    /* Playback. */
    if (cbResult) {
        ma_device_info deviceInfo;
        ma_zero_object(&deviceInfo);
        deviceInfo.id.aaudio = MA_AAUDIO_UNSPECIFIED;
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);

        if (ma_has_default_device__aaudio(pContext, ma_device_type_playback)) {
            cbResult = callback(pContext, ma_device_type_playback, &deviceInfo, pUserData);
        }
    }

    /* Capture. */
    if (cbResult) {
        ma_device_info deviceInfo;
        ma_zero_object(&deviceInfo);
        deviceInfo.id.aaudio = MA_AAUDIO_UNSPECIFIED;
        ma_strncpy_s(deviceInfo.name, sizeof(deviceInfo.name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);

        if (ma_has_default_device__aaudio(pContext, ma_device_type_capture)) {
            cbResult = callback(pContext, ma_device_type_capture, &deviceInfo, pUserData);
        }
    }

    return MA_SUCCESS;
}

ma_result ma_context_get_device_info__aaudio(ma_context* pContext, ma_device_type deviceType, const ma_device_id* pDeviceID, ma_share_mode shareMode, ma_device_info* pDeviceInfo)
{
    ma_AAudioStream* pStream;
    ma_result result;

    ma_assert(pContext != NULL);

    /* No exclusive mode with AAudio. */
    if (shareMode == ma_share_mode_exclusive) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    /* ID */
    if (pDeviceID != NULL) {
        pDeviceInfo->id.aaudio = pDeviceID->aaudio;
    } else {
        pDeviceInfo->id.aaudio = MA_AAUDIO_UNSPECIFIED;
    }
    
    /* Name */
    if (deviceType == ma_device_type_playback) {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_PLAYBACK_DEVICE_NAME, (size_t)-1);
    } else {
        ma_strncpy_s(pDeviceInfo->name, sizeof(pDeviceInfo->name), MA_DEFAULT_CAPTURE_DEVICE_NAME, (size_t)-1);
    }


    /* We'll need to open the device to get accurate sample rate and channel count information. */
    result = ma_open_stream__aaudio(pContext, deviceType, pDeviceID, shareMode, NULL, NULL, &pStream);
    if (result != MA_SUCCESS) {
        return result;
    }

    pDeviceInfo->minChannels   = ((MA_PFN_AAudioStream_getChannelCount)pContext->aaudio.AAudioStream_getChannelCount)(pStream);
    pDeviceInfo->maxChannels   = pDeviceInfo->minChannels;
    pDeviceInfo->minSampleRate = ((MA_PFN_AAudioStream_getSampleRate)pContext->aaudio.AAudioStream_getSampleRate)(pStream);
    pDeviceInfo->maxSampleRate = pDeviceInfo->minSampleRate;

    ma_close_stream__aaudio(pContext, pStream);
    pStream = NULL;


    /* AAudio supports s16 and f32. */
    pDeviceInfo->formatCount = 2;
    pDeviceInfo->formats[0] = ma_format_s16;
    pDeviceInfo->formats[1] = ma_format_f32;

    return MA_SUCCESS;
}


void ma_device_uninit__aaudio(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ma_close_stream__aaudio(pDevice->pContext, (ma_AAudioStream*)pDevice->aaudio.pStreamCapture);
        pDevice->aaudio.pStreamCapture = NULL;
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ma_close_stream__aaudio(pDevice->pContext, (ma_AAudioStream*)pDevice->aaudio.pStreamPlayback);
        pDevice->aaudio.pStreamPlayback = NULL;
    }

    if (pDevice->type == ma_device_type_duplex) {
        ma_pcm_rb_uninit(&pDevice->aaudio.duplexRB);
    }
}

ma_result ma_device_init__aaudio(ma_context* pContext, const ma_device_config* pConfig, ma_device* pDevice)
{
    ma_result result;

    ma_assert(pDevice != NULL);

    /* No exclusive mode with AAudio. */
    if (((pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) && pConfig->playback.shareMode == ma_share_mode_exclusive) ||
        ((pConfig->deviceType == ma_device_type_capture  || pConfig->deviceType == ma_device_type_duplex) && pConfig->capture.shareMode  == ma_share_mode_exclusive)) {
        return MA_SHARE_MODE_NOT_SUPPORTED;
    }

    /* We first need to try opening the stream. */
    if (pConfig->deviceType == ma_device_type_capture || pConfig->deviceType == ma_device_type_duplex) {
        int32_t framesPerPeriod;

        result = ma_open_stream__aaudio(pContext, ma_device_type_capture, pConfig->capture.pDeviceID, pConfig->capture.shareMode, pConfig, pDevice, (ma_AAudioStream**)&pDevice->aaudio.pStreamCapture);
        if (result != MA_SUCCESS) {
            return result;  /* Failed to open the AAudio stream. */
        }

        pDevice->capture.internalFormat     = (((MA_PFN_AAudioStream_getFormat)pContext->aaudio.AAudioStream_getFormat)((ma_AAudioStream*)pDevice->aaudio.pStreamCapture) == MA_AAUDIO_FORMAT_PCM_I16) ? ma_format_s16 : ma_format_f32;
        pDevice->capture.internalChannels   = ((MA_PFN_AAudioStream_getChannelCount)pContext->aaudio.AAudioStream_getChannelCount)((ma_AAudioStream*)pDevice->aaudio.pStreamCapture);
        pDevice->capture.internalSampleRate = ((MA_PFN_AAudioStream_getSampleRate)pContext->aaudio.AAudioStream_getSampleRate)((ma_AAudioStream*)pDevice->aaudio.pStreamCapture);
        ma_get_standard_channel_map(ma_standard_channel_map_default, pDevice->capture.internalChannels, pDevice->capture.internalChannelMap); /* <-- Cannot find info on channel order, so assuming a default. */
        pDevice->capture.internalBufferSizeInFrames = ((MA_PFN_AAudioStream_getBufferCapacityInFrames)pContext->aaudio.AAudioStream_getBufferCapacityInFrames)((ma_AAudioStream*)pDevice->aaudio.pStreamCapture);

        /*
        TODO: When synchronous reading and writing is supported, use AAudioStream_getFramesPerBurst() instead of AAudioStream_getFramesPerDataCallback(). Keep
        using AAudioStream_getFramesPerDataCallback() for asynchronous mode, though.
        */
        framesPerPeriod = ((MA_PFN_AAudioStream_getFramesPerDataCallback)pContext->aaudio.AAudioStream_getFramesPerDataCallback)((ma_AAudioStream*)pDevice->aaudio.pStreamCapture);
        if (framesPerPeriod > 0) {
            pDevice->capture.internalPeriods = 1;
        } else {
            pDevice->capture.internalPeriods = pDevice->capture.internalBufferSizeInFrames / framesPerPeriod;
        }
    }

    if (pConfig->deviceType == ma_device_type_playback || pConfig->deviceType == ma_device_type_duplex) {
        int32_t framesPerPeriod;

        result = ma_open_stream__aaudio(pContext, ma_device_type_playback, pConfig->playback.pDeviceID, pConfig->playback.shareMode, pConfig, pDevice, (ma_AAudioStream**)&pDevice->aaudio.pStreamPlayback);
        if (result != MA_SUCCESS) {
            return result;  /* Failed to open the AAudio stream. */
        }

        pDevice->playback.internalFormat     = (((MA_PFN_AAudioStream_getFormat)pContext->aaudio.AAudioStream_getFormat)((ma_AAudioStream*)pDevice->aaudio.pStreamPlayback) == MA_AAUDIO_FORMAT_PCM_I16) ? ma_format_s16 : ma_format_f32;
        pDevice->playback.internalChannels   = ((MA_PFN_AAudioStream_getChannelCount)pContext->aaudio.AAudioStream_getChannelCount)((ma_AAudioStream*)pDevice->aaudio.pStreamPlayback);
        pDevice->playback.internalSampleRate = ((MA_PFN_AAudioStream_getSampleRate)pContext->aaudio.AAudioStream_getSampleRate)((ma_AAudioStream*)pDevice->aaudio.pStreamPlayback);
        ma_get_standard_channel_map(ma_standard_channel_map_default, pDevice->playback.internalChannels, pDevice->playback.internalChannelMap); /* <-- Cannot find info on channel order, so assuming a default. */
        pDevice->playback.internalBufferSizeInFrames = ((MA_PFN_AAudioStream_getBufferCapacityInFrames)pContext->aaudio.AAudioStream_getBufferCapacityInFrames)((ma_AAudioStream*)pDevice->aaudio.pStreamPlayback);

        framesPerPeriod = ((MA_PFN_AAudioStream_getFramesPerDataCallback)pContext->aaudio.AAudioStream_getFramesPerDataCallback)((ma_AAudioStream*)pDevice->aaudio.pStreamPlayback);
        if (framesPerPeriod > 0) {
            pDevice->playback.internalPeriods = 1;
        } else {
            pDevice->playback.internalPeriods = pDevice->playback.internalBufferSizeInFrames / framesPerPeriod;
        }
    }

    if (pConfig->deviceType == ma_device_type_duplex) {
        ma_uint32 rbSizeInFrames = (ma_uint32)ma_calculate_frame_count_after_src(pDevice->sampleRate, pDevice->capture.internalSampleRate, pDevice->capture.internalBufferSizeInFrames);
        ma_result result = ma_pcm_rb_init(pDevice->capture.format, pDevice->capture.channels, rbSizeInFrames, NULL, &pDevice->aaudio.duplexRB);
        if (result != MA_SUCCESS) {
            if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
                ma_close_stream__aaudio(pDevice->pContext, (ma_AAudioStream*)pDevice->aaudio.pStreamCapture);
            }
            if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
                ma_close_stream__aaudio(pDevice->pContext, (ma_AAudioStream*)pDevice->aaudio.pStreamPlayback);
            }
            return ma_post_error(pDevice, MA_LOG_LEVEL_ERROR, "[AAudio] Failed to initialize ring buffer.", result);
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_start_stream__aaudio(ma_device* pDevice, ma_AAudioStream* pStream)
{
    ma_aaudio_result_t resultAA;
    ma_aaudio_stream_state_t currentState;

    ma_assert(pDevice != NULL);

    resultAA = ((MA_PFN_AAudioStream_requestStart)pDevice->pContext->aaudio.AAudioStream_requestStart)(pStream);
    if (resultAA != MA_AAUDIO_OK) {
        return ma_result_from_aaudio(resultAA);
    }

    /* Do we actually need to wait for the device to transition into it's started state? */

    /* The device should be in either a starting or started state. If it's not set to started we need to wait for it to transition. It should go from starting to started. */
    currentState = ((MA_PFN_AAudioStream_getState)pDevice->pContext->aaudio.AAudioStream_getState)(pStream);
    if (currentState != MA_AAUDIO_STREAM_STATE_STARTED) {
        ma_result result;

        if (currentState != MA_AAUDIO_STREAM_STATE_STARTING) {
            return MA_ERROR;   /* Expecting the stream to be a starting or started state. */
        }

        result = ma_wait_for_simple_state_transition__aaudio(pDevice->pContext, pStream, currentState, MA_AAUDIO_STREAM_STATE_STARTED);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_stop_stream__aaudio(ma_device* pDevice, ma_AAudioStream* pStream)
{
    ma_aaudio_result_t resultAA;
    ma_aaudio_stream_state_t currentState;

    ma_assert(pDevice != NULL);

    resultAA = ((MA_PFN_AAudioStream_requestStop)pDevice->pContext->aaudio.AAudioStream_requestStop)(pStream);
    if (resultAA != MA_AAUDIO_OK) {
        return ma_result_from_aaudio(resultAA);
    }

    /* The device should be in either a stopping or stopped state. If it's not set to started we need to wait for it to transition. It should go from stopping to stopped. */
    currentState = ((MA_PFN_AAudioStream_getState)pDevice->pContext->aaudio.AAudioStream_getState)(pStream);
    if (currentState != MA_AAUDIO_STREAM_STATE_STOPPED) {
        ma_result result;

        if (currentState != MA_AAUDIO_STREAM_STATE_STOPPING) {
            return MA_ERROR;   /* Expecting the stream to be a stopping or stopped state. */
        }

        result = ma_wait_for_simple_state_transition__aaudio(pDevice->pContext, pStream, currentState, MA_AAUDIO_STREAM_STATE_STOPPED);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_start__aaudio(ma_device* pDevice)
{
    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ma_result result = ma_device_start_stream__aaudio(pDevice, (ma_AAudioStream*)pDevice->aaudio.pStreamCapture);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ma_result result = ma_device_start_stream__aaudio(pDevice, (ma_AAudioStream*)pDevice->aaudio.pStreamPlayback);
        if (result != MA_SUCCESS) {
            if (pDevice->type == ma_device_type_duplex) {
                ma_device_stop_stream__aaudio(pDevice, (ma_AAudioStream*)pDevice->aaudio.pStreamCapture);
            }
            return result;
        }
    }

    return MA_SUCCESS;
}

ma_result ma_device_stop__aaudio(ma_device* pDevice)
{
    ma_stop_proc onStop;

    ma_assert(pDevice != NULL);

    if (pDevice->type == ma_device_type_capture || pDevice->type == ma_device_type_duplex) {
        ma_result result = ma_device_stop_stream__aaudio(pDevice, (ma_AAudioStream*)pDevice->aaudio.pStreamCapture);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    if (pDevice->type == ma_device_type_playback || pDevice->type == ma_device_type_duplex) {
        ma_result result = ma_device_stop_stream__aaudio(pDevice, (ma_AAudioStream*)pDevice->aaudio.pStreamPlayback);
        if (result != MA_SUCCESS) {
            return result;
        }
    }

    onStop = pDevice->onStop;
    if (onStop) {
        onStop(pDevice);
    }

    return MA_SUCCESS;
}


ma_result ma_context_uninit__aaudio(ma_context* pContext)
{
    ma_assert(pContext != NULL);
    ma_assert(pContext->backend == ma_backend_aaudio);
    
    ma_dlclose(pContext->aaudio.hAAudio);
    pContext->aaudio.hAAudio = NULL;

    return MA_SUCCESS;
}

ma_result ma_context_init__aaudio(const ma_context_config* pConfig, ma_context* pContext)
{
    const char* libNames[] = {
        "libaaudio.so"
    };
    size_t i;

    for (i = 0; i < ma_countof(libNames); ++i) {
        pContext->aaudio.hAAudio = ma_dlopen(libNames[i]);
        if (pContext->aaudio.hAAudio != NULL) {
            break;
        }
    }

    if (pContext->aaudio.hAAudio == NULL) {
        return MA_FAILED_TO_INIT_BACKEND;
    }

    pContext->aaudio.AAudio_createStreamBuilder                    = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudio_createStreamBuilder");
    pContext->aaudio.AAudioStreamBuilder_delete                    = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_delete");
    pContext->aaudio.AAudioStreamBuilder_setDeviceId               = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_setDeviceId");
    pContext->aaudio.AAudioStreamBuilder_setDirection              = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_setDirection");
    pContext->aaudio.AAudioStreamBuilder_setSharingMode            = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_setSharingMode");
    pContext->aaudio.AAudioStreamBuilder_setFormat                 = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_setFormat");
    pContext->aaudio.AAudioStreamBuilder_setChannelCount           = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_setChannelCount");
    pContext->aaudio.AAudioStreamBuilder_setSampleRate             = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_setSampleRate");
    pContext->aaudio.AAudioStreamBuilder_setBufferCapacityInFrames = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_setBufferCapacityInFrames");
    pContext->aaudio.AAudioStreamBuilder_setFramesPerDataCallback  = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_setFramesPerDataCallback");
    pContext->aaudio.AAudioStreamBuilder_setDataCallback           = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_setDataCallback");
    pContext->aaudio.AAudioStreamBuilder_setPerformanceMode        = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_setPerformanceMode");
    pContext->aaudio.AAudioStreamBuilder_openStream                = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStreamBuilder_openStream");
    pContext->aaudio.AAudioStream_close                            = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_close");
    pContext->aaudio.AAudioStream_getState                         = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_getState");
    pContext->aaudio.AAudioStream_waitForStateChange               = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_waitForStateChange");
    pContext->aaudio.AAudioStream_getFormat                        = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_getFormat");
    pContext->aaudio.AAudioStream_getChannelCount                  = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_getChannelCount");
    pContext->aaudio.AAudioStream_getSampleRate                    = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_getSampleRate");
    pContext->aaudio.AAudioStream_getBufferCapacityInFrames        = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_getBufferCapacityInFrames");
    pContext->aaudio.AAudioStream_getFramesPerDataCallback         = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_getFramesPerDataCallback");
    pContext->aaudio.AAudioStream_getFramesPerBurst                = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_getFramesPerBurst");
    pContext->aaudio.AAudioStream_requestStart                     = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_requestStart");
    pContext->aaudio.AAudioStream_requestStop                      = (ma_proc)ma_dlsym(pContext->aaudio.hAAudio, "AAudioStream_requestStop");

    pContext->isBackendAsynchronous = MA_TRUE;

    pContext->onUninit        = ma_context_uninit__aaudio;
    pContext->onDeviceIDEqual = ma_context_is_device_id_equal__aaudio;
    pContext->onEnumDevices   = ma_context_enumerate_devices__aaudio;
    pContext->onGetDeviceInfo = ma_context_get_device_info__aaudio;
    pContext->onDeviceInit    = ma_device_init__aaudio;
    pContext->onDeviceUninit  = ma_device_uninit__aaudio;
    pContext->onDeviceStart   = ma_device_start__aaudio;
    pContext->onDeviceStop    = ma_device_stop__aaudio;

    (void)pConfig;
    return MA_SUCCESS;
}
#endif  /* AAudio */


/****************************************************************************