import { ConsoleLogger, Injectable, OnModuleInit } from '@nestjs/common';
import fs from 'fs';
import path from 'path';
import Mustache from 'mustache';

@Injectable()
export class TemplateService extends ConsoleLogger implements OnModuleInit {
  private templates = new Map<string, string>();
  constructor() {
    super('TemplateService');
  }

  async onModuleInit() {
    const fileList = await fs.promises.readdir('views');
    for (const file of fileList) {
      if (!file.endsWith('.mustache')) {
        continue;
      }
      const template = file.slice(0, file.length - 9);
      this.templates.set(
        template,
        await fs.promises.readFile(path.join('views', file), 'utf-8'),
      );
      this.log(`Loaded template ${template}`);
    }
  }

  render(template: string, view: any = {}, htmlSafe = false) {
    const text = this.templates.get(template);
    if (!text) {
      this.warn(`Template ${template} not found.`);
      return template;
    }
    let result: string;
    if (htmlSafe) {
      result = Mustache.render(text, view);
    } else {
      result = Mustache.render(text, view, undefined, { escape: (v) => v });
    }
    return result.trim();
  }

  getTemplateList() {
    return [...this.templates.keys()];
  }
}
