import { Injectable } from '@nestjs/common';
import { CrudService } from 'nicot';
import { User } from './entities/user.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { PutUserId, UseCommand } from 'koishi-nestjs';
import { Aragami, CacheKey, CacheTTL } from 'aragami';
import { InjectAragami } from 'nestjs-aragami';
import cryptoRandomString from 'crypto-random-string';
import { MycardAuthService, MycardUser } from 'nestjs-mycard';
import { ConfigService } from '@nestjs/config';
import { BindQueryDto } from './dto/bind-query.dto';
import { TemplateService } from '../template/template.service';

@CacheTTL(1800 * 1000)
class BindSession {
  @CacheKey()
  sessionId: string;
  userId: string;

  static create(userId: string) {
    const s = new BindSession();
    s.sessionId = cryptoRandomString({ length: 64, type: 'url-safe' });
    s.userId = userId;
    return s;
  }
}

function loginUrl(callbackUrl: string) {
  let params = new URLSearchParams();
  params.set('return_sso_url', callbackUrl);
  const payload = Buffer.from(params.toString()).toString('base64');
  const url = new URL('https://accounts.moecube.com/signin');
  params = url['searchParams'];
  params.set('sso', payload);
  return url.toString();
}

@Injectable()
export class MycardAccountService extends CrudService(User) {
  private rootUrl = this.config.get<string>('ROOT_URL');
  constructor(
    @InjectRepository(User) repo,
    @InjectAragami() private aragami: Aragami,
    private mycard: MycardAuthService,
    private config: ConfigService,
    private template: TemplateService,
  ) {
    super(repo);
  }

  async findOrCreateUser(id: string) {
    let user = await this.repo.findOne({ where: { id } });
    if (user) {
      return user;
    }
    user = new User();
    user.id = id;
    return this.repo.save(user);
  }

  @UseCommand('unbind', '解绑 MyCard 帐号')
  async unbind(@PutUserId() id: string) {
    const user = await this.findOrCreateUser(id);
    if (!user.mycardId) {
      return '您还未绑定 MyCard 帐号。';
    }
    await this.repo.update({ id }, { mycardId: null });
    return '解绑成功。';
  }

  @UseCommand('bind', '绑定 MyCard 帐号')
  async bind(@PutUserId() id: string, needHint = false) {
    const session = BindSession.create(id);
    await this.aragami.set(session);
    const url = loginUrl(
      `${this.rootUrl}/api/mycard-account/bind?session=${session.sessionId}`,
    );
    return this.template.render('bind-account', { url, needHint });
  }

  async bindCallback(
    dto: BindQueryDto,
  ): Promise<{ error?: string; user?: MycardUser }> {
    const session = await this.aragami.get(BindSession, dto.session);
    if (!session) {
      return { error: '会话不存在。' };
    }
    try {
      const mycardUser = await this.mycard.getUserFromToken(dto.decodeToken());
      await this.findOrCreateUser(session.userId);
      await this.repo.update(
        { id: session.userId },
        { mycardId: mycardUser.id },
      );
      return { user: mycardUser };
    } catch (e) {
      return { error: e.message };
    } finally {
      await this.aragami.del(BindSession, dto.session);
    }
  }
}
