import {
  BadRequestException,
  Injectable,
  InternalServerErrorException,
} from '@nestjs/common';
import { CrudService } from 'nicot';
import { Feedback } from './entities/feedback.entity';
import { InjectRepository } from '@nestjs/typeorm';
import { Not, Repository } from 'typeorm';
import { ConfigService } from '@nestjs/config';
import { Random, Session } from 'koishi';
import { HttpService } from '@nestjs/axios';
import { lastValueFrom } from 'rxjs';
import { PutSession, UseCommand } from 'koishi-nestjs';
import { BotService } from '../bot/bot.service';
import { TemplateService } from '../template/template.service';

@Injectable()
export class FeedbackService extends CrudService(Feedback) {
  private notifyUrl = this.config.get<string>('FEEDBACK_NOTIFY_URL');
  private notifyToken = this.config.get<string>('FEEDBACK_NOTIFY_TOKEN');

  constructor(
    @InjectRepository(Feedback) repo: Repository<Feedback>,
    private config: ConfigService,
    private http: HttpService,
    private botService: BotService,
    private template: TemplateService,
  ) {
    super(repo);
  }

  async notifyFeedback(feedback: Feedback) {
    if (!this.notifyUrl) {
      return;
    }
    try {
      await lastValueFrom(
        this.http.post(
          this.notifyUrl,
          { content: feedback.toNotifyText() },
          {
            headers: {
              Authorization: `Bearer ${this.notifyToken}`,
            },
          },
        ),
      );
    } catch (e) {
      this.log.error(`Failed to notify feedback: ${e.message}`);
    }
  }

  async createFeedback(session: Session, payload: Partial<Feedback>) {
    if (!payload.content) {
      throw new BadRequestException('');
    }
    if (payload.content.length > 10000) {
      throw new BadRequestException('反馈内容过长。');
    }
    const feedback = new Feedback().fromSession(session);
    Object.assign(feedback, payload);
    this.notifyFeedback(feedback).then();
    try {
      await this.repo.save(feedback);
      return await this.repo.count({
        where: { category: payload.category === '周边' ? '周边' : Not('周边') },
      });
    } catch (e) {
      const id = Random.id(10);
      this.log.error(`Failed to save feedback id: ${e.message}`);
      throw new InternalServerErrorException(
        '反馈遇到问题，请提供下列编号与管理员取得联系: ' + id,
      );
    }
  }

  @UseCommand('poll', '调研')
  private async pollCommand() {
    return this.template.render('poll-not-available');
  }

  @UseCommand('feedback', '发送反馈。')
  private async feedbackCommand(@PutSession() session: Session) {
    await session.send(this.template.render('feedback-menu'));
    let choice = 0;
    while (true) {
      const input = await session.prompt();
      if (!input) {
        return;
      }
      if (this.botService.isBuiltinCommand(input)) {
        return session.execute(input);
      }
      choice = parseInt(input.trim());
      if (choice >= 1 && choice <= 4) {
        break;
      }
      await session.send('请选择一个正确的选项（数字1~4）');
    }
    if (choice == 1) {
      return this.template.render('feedback-bug');
    }
    const category = ['游戏功能/改善', '游戏体验问题', '其他'][choice - 2];
    await session.send(this.template.render('feedback-prompt'));
    const content = await session.prompt();
    if (this.botService.isBuiltinCommand(content)) {
      return session.execute(content);
    }
    const count = await this.createFeedback(session, { category, content });
    return this.template.render('feedback-success', { count });
  }

  @UseCommand('feedback.shop', '发送周边反馈。')
  private async shopFeedbackCommand(@PutSession() session: Session) {
    await session.send(this.template.render('feedback-shop'));
    const content = await session.prompt();
    if (this.botService.isBuiltinCommand(content)) {
      return session.execute(content);
    }
    const count = await this.createFeedback(session, {
      category: '周边',
      content,
    });
    return this.template.render('feedback-shop-success', { count });
  }
}
