import { ConsoleLogger, Injectable } from '@nestjs/common';
import {
  InjectContext,
  PluginDef,
  UseCommand,
  UsePlugin,
  WireContextService,
} from 'koishi-nestjs';
import { ConfigService } from '@nestjs/config';
import path from 'path';
import fs from 'fs';
import simpleGit, { ResetMode, SimpleGit } from 'simple-git';
import YGOCardPlugin from 'koishi-plugin-ygocard';
import { Cron } from '@nestjs/schedule';
import { Context } from 'koishi';

async function exists(path: string) {
  try {
    const res = await fs.promises.stat(path);
    return res.isDirectory();
  } catch (e) {
    return false;
  }
}

@Injectable()
export class CdbLoaderService extends ConsoleLogger {
  private repoUrl = this.config.get<string>('CDB_REPO');
  private repoPath = path.join(process.cwd(), 'ygopro-database');
  private repoBranch = this.config.get<string>('CDB_BRANCH');
  private git: SimpleGit;

  constructor(
    private config: ConfigService,
    @InjectContext() private ctx: Context,
  ) {
    super('CdbLoaderService');
  }

  async initRepo() {
    if (!(await exists(path.join(this.repoPath, '.git')))) {
      this.log(`Cloning repo ${this.repoUrl} to ${this.repoPath}`);
      await simpleGit().clone(this.repoUrl, this.repoPath, [
        '--branch',
        this.repoBranch,
      ]);
      this.log('Repo cloned');
    }
    this.git = simpleGit(this.repoPath);
  }

  @UsePlugin()
  async load() {
    await this.initRepo();
    return PluginDef(YGOCardPlugin, {
      databasePaths: [path.join(this.repoPath, 'locales', 'zh-CN')],
      matchCount: 20,
      lang: 'cn',
      usePuppeteer: true,
    });
  }

  @WireContextService()
  private ygocard: YGOCardPlugin;

  @Cron('0 0 3 * * *')
  async updateRepo() {
    this.log('Updating repo');
    await this.git.fetch(['origin', this.repoBranch]);
    await this.git.reset(ResetMode.HARD, ['origin/' + this.repoBranch]);
    if (this.ygocard) {
      await this.ygocard.reloadDatabase();
    }
    this.log('Repo updated');
  }

  /*@UseCommand('reload-cdb')
  async reloadRepo() {
    await this.updateRepo();
    return 'success';
  }*/
}
