import { Module } from '@nestjs/common';
import { ConfigModule, ConfigService } from '@nestjs/config';
import { loadConfig } from './utility/config';
import { TypeOrmModule } from '@nestjs/typeorm';
import { KoishiModule, PluginDef } from 'koishi-nestjs';
import { ReplyService } from './reply/reply.service';
import { CommandCountService } from './command-count/command-count.service';
import WechatBot from 'koishi-plugin-adapter-wechat-official';
import * as HelpPlugin from '@koishijs/plugin-help';
import { CommandCount } from './command-count/entities/command-count.entity';
import { CdbLoaderService } from './cdb-loader/cdb-loader.service';
import { ScheduleModule } from '@nestjs/schedule';
import { FeedbackService } from './feedback/feedback.service';
import PuppeteerPlugin from 'koishi-plugin-puppeteer';
import { HttpModule } from '@nestjs/axios';
import { Feedback } from './feedback/entities/feedback.entity';
import { OnSubscribeService } from './on-subscribe/on-subscribe.service';
import { BotService } from './bot/bot.service';
import { MycardAuthModule } from 'nestjs-mycard';
import { MycardAccountService } from './mycard-account/mycard-account.service';
import { AragamiModule } from 'nestjs-aragami';
import { User } from './mycard-account/entities/user.entity';
import { MycardAccountController } from './mycard-account/mycard-account.controller';
import { TemplateService } from './template/template.service';
import { ArenaScoreService } from './arena-score/arena-score.service';

@Module({
  imports: [
    ConfigModule.forRoot({
      load: [loadConfig],
      isGlobal: true,
      ignoreEnvVars: true,
      ignoreEnvFile: true,
    }),
    TypeOrmModule.forRootAsync({
      inject: [ConfigService],
      useFactory: async (config: ConfigService) => ({
        type: 'postgres',
        entities: [],
        autoLoadEntities: true,
        synchronize: !config.get('DB_NO_INIT'),
        host: config.get('DB_HOST'),
        port: parseInt(config.get('DB_PORT')) || 5432,
        username: config.get('DB_USER'),
        password: config.get('DB_PASS'),
        database: config.get('DB_NAME'),
        supportBigNumbers: true,
        bigNumberStrings: false,
      }),
    }),
    TypeOrmModule.forFeature([CommandCount, Feedback, User]),
    ScheduleModule.forRoot(),
    HttpModule,
    MycardAuthModule.register(),
    AragamiModule.registerAsync({
      inject: [ConfigService],
      useFactory: async (config: ConfigService) => ({
        redis: { uri: config.get('REDIS_URL') },
      }),
    }),
    KoishiModule.registerAsync({
      inject: [ConfigService],
      useWs: true,
      useFactory: async (config: ConfigService) => ({
        usePlugins: [
          PluginDef(HelpPlugin),
          PluginDef(PuppeteerPlugin),
          PluginDef(WechatBot, {
            appId: config.get('WX_ID'),
            appSecret: config.get('WX_SECRET'),
            token: config.get('WX_TOKEN'),
            encodingAESKey: config.get('WX_AESKEY'),
            path: '/wechat',
            menus: [
              {
                type: 'parent',
                name: '常用链接',
                children: [
                  { type: 'view', name: '玩家社区', url: 'https://ygobbs2.com' },
                  {
                    type: 'view',
                    name: '决斗数据库',
                    url: 'https://mycard.world/ygopro/arena',
                  },
                  {
                    type: 'click',
                    name: '游戏下载',
                    command: 'reply.download',
                  },
                  {
                    type: 'view',
                    name: '用户中心',
                    url: 'https://accounts.moecube.com/',
                  },
                  {
                    type: 'view',
                    name: 'MC周边店',
                    url: 'https://shop387046095.taobao.com/',
                  },
                ],
              },
              {
                type: 'parent',
                name: '用户服务',
                children: [
                  {
                    type: 'click',
                    name: 'MyCard 战绩',
                    command: 'arena-score',
                  },
                  {
                    type: 'click',
                    name: '服务器列表',
                    command: 'reply.servers',
                  },
                  {
                    type: 'click',
                    name: '如何更新',
                    command: 'reply.update',
                  },
                  {
                    type: 'click',
                    name: '常见疑难',
                    command: 'reply.faq',
                  },
                  {
                    type: 'click',
                    name: '娘化补丁',
                    command: 'reply.novelai',
                  },
                ],
              },
              {
                type: 'parent',
                name: '游戏功能',
                children: [
                  {
                    type: 'click',
                    name: '随机抽卡',
                    command: 'card --random',
                  },
                  {
                    type: 'click',
                    name: '卡查',
                    command: 'card',
                  },
                  {
                    type: 'click',
                    name: '有奖调研',
                    command: 'poll',
                  },
                  {
                    type: 'click',
                    name: '意见反馈',
                    command: 'feedback',
                  },
                  {
                    type: 'click',
                    name: '周边反馈',
                    command: 'feedback.shop',
                  },
                ],
              },
            ],
          }),
        ],
      }),
    }),
  ],
  providers: [
    ReplyService,
    CommandCountService,
    CdbLoaderService,
    FeedbackService,
    OnSubscribeService,
    BotService,
    MycardAccountService,
    TemplateService,
    ArenaScoreService,
  ],
  controllers: [MycardAccountController],
})
export class AppModule {}
