import { createProxyServer } from '@mutagen-d/node-proxy-server';
import { Server, createConnection } from 'net';
import { hashAddress } from './src/hash-address';
import { createLogger } from 'bunyan';
import { Socket } from 'net';
import Address from 'ipaddr.js';
import dns from 'dns';

declare module 'net' {
  interface Server {
    on(
      event: 'proxy-auth',
      listener: (
        socket: Socket,
        username: string,
        password: string,
        callback: (success: boolean) => void,
      ) => void,
    ): this;
  }
}

const targetIpSym = Symbol('targetIp');
const log = createLogger({ name: 'switcher' });

const server: Server = createProxyServer({
  auth: true,
  createProxyConnection: async (info) => {
    const src: Address.IPv4 | Address.IPv6 = info.socket[targetIpSym];
    log.info(
      `Connecting to ${info.dstHost}:${info.dstPort} with ${src?.toString()}`,
    );
    let socket: Socket;
    if (src) {
      try {
        const lookup = await dns.promises.lookup(
          info.dstHost,
          src.kind() === 'ipv4' ? 4 : 6,
        );
        socket = createConnection({
          host: lookup.address,
          port: info.dstPort,
          localAddress: src.toString(),
        });
      } catch (e) {
        log.warn(
          `Remote ${
            info.dstHost
          } does not support ${src.kind()}, fallback to direct connect.`,
        );
        socket = createConnection({
          host: info.dstHost,
          port: info.dstPort,
        });
      }
    } else {
      socket = createConnection({
        host: info.dstHost,
        port: info.dstPort,
      });
    }
    return new Promise((resolve, reject) => {
      socket.on('connect', () => resolve(socket));
      socket.on('error', (error) => reject(error));
    });
  },
});

server.on('proxy-auth', (socket, username, password, callback) => {
  if (!username && !password) {
    callback(true);
    return;
  }
  const token = `${username}:${password}`;
  const address = hashAddress(process.env.CIDR, token);
  socket[targetIpSym] = address;
  log.info(`Mapped ${token} to ${address.toString()}`);
  callback(true);
});

const port = parseInt(process.env.PORT || '8080');
server.listen(port, process.env.HOST || '0.0.0.0', () =>
  log.info(`Listening on port ${port}`),
);
