"==================================================
" File:         fencview.vim
" Brief:        View a file in different encodings
" Authors:      Ming Bai <mbbill AT gmail DOT com>,
"               Wu Yongwei <wuyongwei AT gmail DOT com>
" Last Change:  2012-02-04 19:18:37
" Version:      4.8
" Licence:      LGPL
"
"
" Usage:
"               Commands:
"                :FencAutoDetect
"                    Auto detect the file encoding.
"                    Built-in detected encodings:
"                     Unicode:
"                        UTF-8
"                     Chinese Simplified:
"                        CP936 (GBK, EUC-CN)
"                     Chinese Traditional:
"                        CP950 (Big5)
"                        EUC-TW
"                     Japanese:
"                        CP932 (SJIS)
"                        EUC-JP
"                     Korean:
"                        CP949 (EUC-KR)
"                :FencView
"                    Open the encoding list window,
"                    <up> and <down> to select an encoding,
"                    enter to reload the file
"               Menu:
"                 Select a file encoding from the
"                 Tools->Encoding menu just like what you do in
"                 firefox/IE.
"               Options:
"                 Set these value in vimrc to config this
"                 plugin. Example:
"                 let g:fencview_autodetect = 1
"
"                 "$FENCVIEW_TELLENC"
"                   Now FencView allows the use of an external
"                   program to determine the file encoding. It
"                   may give you better performance, or better
"                   accuracy, in some cases. Set the environment
"                   variable FENCVIEW_TELLENC (or the vimrc
"                   variable $FENCVIEW_TELLENC) to use this
"                   feature. It is currently tested against
"                   tellenc.
"                   (default: 'tellenc')
"                   It means if tellenc is executable, it will
"                   be used by default. If you still want to use
"                   the detection mechanism of this script, set
"                   the value to "fencview".
"
"                 "g:fencview_autodetect"
"                   Auto detect file encoding when you open a
"                   file.
"                   (default: 1)
"
"                 "g:fencview_auto_patterns"
"                   Set this variable in your vimrc to decide
"                   the pattern of file names to enable
"                   autodetection.
"                   (default: '*.txt,*.htm{l\=}')
"
"                 "g:fencview_checklines"
"                   It checks first and last several lines of
"                   current file, so don't set the value too
"                   large. Set to 0 if you want to check every
"                   line." (default: 10)
"
" Tip:          1 "+iconv" feature is needed. If you are
"                 using Microsoft Windows, make sure you
"                 have iconv.dll in $PATH.
"                 (http://mbbill.googlepages.com/iconv.dll)
"                 Use command ":echo has('iconv')" to check it.
"               2 Make sure there is no modeline at the end of
"                 current file.
"               3 set encoding to utf-8 for better performance.
"               4 No effect to Vim encrypted files.
"
"
"
" Thanks:       jasonal
"
"
"==================================================
" Vim version 7.x is needed. Thanks to Ingo Karkat
if v:version < 700
     finish
endif

" Change cpoptions to make sure line continuation works
let s:cpo_save=&cpo
set cpo&vim

fun! s:escape(name)
  " shellescape() was added by patch 7.0.111
  if exists("*shellescape")
    return shellescape(a:name)
  endif
  return "'" . a:name . "'"
endfun

" variable definition{{{1
if !exists('g:fencview_autodetect')
    let g:fencview_autodetect = 1
endif
if !exists('g:fencview_auto_patterns')
    let g:fencview_auto_patterns='*.txt,*.htm{l\=}'
endif
if !exists('g:fencview_checklines')
    let g:fencview_checklines = 10
endif
if !exists('g:fencview_html_filetypes')
    let g:fencview_html_filetypes='html'
endif
if $FENCVIEW_TELLENC == ''
    let $FENCVIEW_TELLENC ='tellenc'
endif


let s:FencWinName="FencView_8795684"
let s:Fenc8bit=[
            \"latin1    8-bit.characters (ISO 8859-1)",
            \"koi8-r    Russian",
            \"koi8-u    Ukrainian",
            \"macroman  MacRoman (Macintosh encoding)",
            \"cp437     similar to iso-8859-1",
            \"cp737     similar to iso-8859-7",
            \"cp775     Baltic",
            \"cp850     similar to iso-8859-4",
            \"cp852     similar to iso-8859-1",
            \"cp855     similar to iso-8859-2",
            \"cp857     similar to iso-8859-5",
            \"cp860     similar to iso-8859-9",
            \"cp861     similar to iso-8859-1",
            \"cp862     similar to iso-8859-1",
            \"cp863     similar to iso-8859-8",
            \"cp865     similar to iso-8859-1",
            \"cp866     similar to iso-8859-5",
            \"cp869     similar to iso-8859-7",
            \"cp874     Thai",
            \"cp1250    Czech, Polish, etc",
            \"cp1251    Cyrillic",
            \"cp1253    Greek",
            \"cp1254    Turkish",
            \"cp1255    Hebrew",
            \"cp1256    Arabic",
            \"cp1257    Baltic",
            \"cp1258    Vietnamese"]
let s:Fenc16bit=[
            \"gb18030   Simplified Chinese",
            \"cp936     Simplified Chinese (Windows only)",
            \"euc-cn    Simplified Chinese (Unix only)",
            \"cp950     Traditional Chinese (on Unix alias for big5)",
            \"big5      Traditional Chinese (on Windows alias for cp950)",
            \"euc-tw    Traditional Chinese (Unix only)",
            \"cp932     Japanese (Windows only)",
            \"euc-jp    Japanese (Unix only)",
            \"sjis      Japanese (Unix only)",
            \"cp949     Korean (Unix and Windows)",
            \"euc-kr    Korean (Unix only)"]
let s:FencUnicode=[
            \"utf-8     32 bit UTF-8 encoded Unicode (ISO/IEC 10646-1)",
            \"ucs-2     16 bit UCS-2 encoded Unicode (ISO/IEC 10646-1)",
            \"ucs-2le   like ucs-2, little endian",
            \"utf-16    ucs-2 extended with double-words for more characters",
            \"utf-16le  like utf-16, little endian",
            \"ucs-4     32 bit UCS-4 encoded Unicode (ISO/IEC 10646-1)",
            \"ucs-4le   like ucs-4, little endian"]
let g:FencCustom=
            \"Examples:\n".
            \"------------------\n".
            \"iso-8859-n    ISO_8859 variant (n=2 to 15)\n".
            \"8bit-{name}   any 8-bit encoding (Vim specific name)\n".
            \"cp{number}    MS-Windows: any installed single-byte codepage\n".
            \"cp{number}    MS-Windows: any installed double-byte codepage\n".
            \"2byte-{name}  Unix: any double-byte encoding (Vim specific name)"

" cp936TopChars"{{{1
let s:cp936TopChars=[
\0xb5.0xc4,0xd2.0xbb,0xb9.0xfa,0xd4.0xda,0xc8.0xcb,0xc1.0xcb,0xd3.0xd0,0xd6.0xd0,
\0xca.0xc7,0xc4.0xea,0xba.0xcd,0xb4.0xf3,0xd2.0xb5,0xb2.0xbb,0xce.0xaa,0xb7.0xa2,
\0xbb.0xe1,0xb9.0xa4,0xbe.0xad,0xc9.0xcf,0xb5.0xd8,0xca.0xd0,0xd2.0xaa,0xb8.0xf6,
\0xb2.0xfa,0xd5.0xe2,0xb3.0xf6,0xd0.0xd0,0xd7.0xf7,0xc9.0xfa,0xbc.0xd2,0xd2.0xd4,
\0xb3.0xc9,0xb5.0xbd,0xc8.0xd5,0xc3.0xf1,0xc0.0xb4,0xce.0xd2,0xb2.0xbf,0xb6.0xd4,
\0xbd.0xf8,0xb6.0xe0,0xc8.0xab,0xbd.0xa8,0xcb.0xfb,0xb9.0xab,0xbf.0xaa,0xc3.0xc7,
\0xb3.0xa1,0xd5.0xb9,0xca.0xb1,0xc0.0xed,0xd0.0xc2,0xb7.0xbd,0xd6.0xf7,0xc6.0xf3,
\0xd7.0xca,0xca.0xb5,0xd1.0xa7,0xb1.0xa8,0xd6.0xc6,0xd5.0xfe,0xbc.0xc3,0xd3.0xc3,
\0xcd.0xac,0xd3.0xda,0xb7.0xa8,0xb8.0xdf,0xb3.0xa4,0xcf.0xd6,0xb1.0xbe,0xd4.0xc2,
\0xb6.0xa8,0xbb.0xaf,0xbc.0xd3,0xb6.0xaf,0xba.0xcf,0xc6.0xb7,0xd6.0xd8,0xb9.0xd8,
\0xbb.0xfa,0xb7.0xd6,0xc1.0xa6,0xd7.0xd4,0xcd.0xe2,0xd5.0xdf,0xc7.0xf8,0xc4.0xdc,
\0xc9.0xe8,0xba.0xf3,0xbe.0xcd,0xb5.0xc8,0xcc.0xe5,0xcf.0xc2,0xcd.0xf2,0xd4.0xaa,
\0xc9.0xe7,0xb9.0xfd,0xc7.0xb0,0xc3.0xe6,0xc5.0xa9,0xd2.0xb2,0xb5.0xc3,0xd3.0xeb,
\0xcb.0xb5,0xd6.0xae,0xd4.0xb1,0xb6.0xf8,0xce.0xf1,0xc0.0xfb,0xb5.0xe7,0xce.0xc4,
\0xca.0xc2,0xbf.0xc9,0xd6.0xd6,0xd7.0xdc,0xb8.0xc4,0xc8.0xfd,0xb8.0xf7,0xba.0xc3,
\0xbd.0xf0,0xb5.0xda,0xcb.0xbe,0xc6.0xe4,0xb4.0xd3,0xc6.0xbd,0xb4.0xfa,0xb5.0xb1,
\0xcc.0xec,0xcb.0xae,0xca.0xa1,0xcc.0xe1,0xc9.0xcc,0xca.0xae,0xb9.0xdc,0xc4.0xda,
\0xd0.0xa1,0xbc.0xbc,0xce.0xbb,0xc4.0xbf,0xc6.0xf0,0xba.0xa3,0xcb.0xf9,0xc1.0xa2,
\0xd2.0xd1,0xcd.0xa8,0xc8.0xeb,0xc1.0xbf,0xd7.0xd3,0xce.0xca,0xb6.0xc8,0xb1.0xb1,
\0xb1.0xa3,0xd0.0xc4,0xbb.0xb9,0xbf.0xc6,0xce.0xaf,0xb6.0xbc,0xca.0xf5,0xca.0xb9,
\0xc3.0xf7,0xd7.0xc5,0xb4.0xce,0xbd.0xab,0xd4.0xf6,0xbb.0xf9,0xc3.0xfb,0xcf.0xf2,
\0xc3.0xc5,0xd3.0xa6,0xc0.0xef,0xc3.0xc0,0xd3.0xc9,0xb9.0xe6,0xbd.0xf1,0xcc.0xe2,
\0xbc.0xc7,0xb5.0xe3,0xbc.0xc6,0xc8.0xa5,0xc7.0xbf,0xc1.0xbd,0xd0.0xa9,0xb1.0xed,
\0xcf.0xb5,0xb0.0xec,0xbd.0xcc,0xd5.0xfd,0xcc.0xf5,0xd7.0xee,0xb4.0xef,0xcc.0xd8,
\0xb8.0xef,0xca.0xd5,0xb6.0xfe,0xc6.0xda,0xb2.0xa2,0xb3.0xcc,0xb3.0xa7,0xc8.0xe7,
\0xb5.0xc0,0xbc.0xca,0xbc.0xb0,0xce.0xf7,0xbf.0xda,0xbe.0xa9,0xbb.0xaa,0xc8.0xce,
\0xb5.0xf7,0xd0.0xd4,0xb5.0xbc,0xd7.0xe9,0xb6.0xab,0xc2.0xb7,0xbb.0xee,0xb9.0xe3,
\0xd2.0xe2,0xb1.0xc8,0xcd.0xb6,0xbe.0xf6,0xbd.0xbb,0xcd.0xb3,0xb5.0xb3,0xc4.0xcf,
\0xb0.0xb2,0xb4.0xcb,0xc1.0xec,0xbd.0xe1,0xd3.0xaa,0xcf.0xee,0xc7.0xe9,0xbd.0xe2,
\0xd2.0xe9,0xd2.0xe5,0xc9.0xbd,0xcf.0xc8,0xb3.0xb5,0xc8.0xbb,0xbc.0xdb,0xb7.0xc5,
\0xca.0xc0,0xbc.0xe4,0xd2.0xf2,0xb9.0xb2,0xd4.0xba,0xb2.0xbd,0xce.0xef,0xbd.0xe7,
\0xbc.0xaf,0xb0.0xd1,0xb3.0xd6,0xce.0xde,0xb5.0xab,0xb3.0xc7,0xcf.0xe0,0xca.0xe9,
\0xb4.0xe5,0xc7.0xf3,0xd6.0xce,0xc8.0xa1,0xd4.0xad,0xb4.0xa6,0xb8.0xae,0xd1.0xd0,
\0xd6.0xca,0xd0.0xc5,0xcb.0xc4,0xd4.0xcb,0xcf.0xd8,0xbe.0xfc,0xbc.0xfe,0xd3.0xfd,
\0xbe.0xd6,0xb8.0xc9,0xb6.0xd3,0xcd.0xc5,0xd3.0xd6,0xd4.0xec,0xd0.0xce,0xbc.0xb6,
\0xb1.0xea,0xc1.0xaa,0xd7.0xa8,0xc9.0xd9,0xb7.0xd1,0xd0.0xa7,0xbe.0xdd,0xca.0xd6,
\0xca.0xa9,0xc8.0xa8,0xbd.0xad,0xbd.0xfc,0xc9.0xee,0xb8.0xfc,0xc8.0xcf,0xb9.0xfb,
\0xb8.0xf1,0xbc.0xb8,0xbf.0xb4,0xc3.0xbb,0xd6.0xb0,0xb7.0xfe,0xcc.0xa8,0xca.0xbd,
\0xd2.0xe6,0xcf.0xeb,0xca.0xfd,0xb5.0xa5,0xd1.0xf9,0xd6.0xbb,0xb1.0xbb,0xd2.0xda,
\0xc0.0xcf,0xca.0xdc,0xd3.0xc5,0xb3.0xa3,0xcf.0xfa,0xd6.0xbe,0xd5.0xbd,0xc1.0xf7,
\0xba.0xdc,0xbd.0xd3,0xcf.0xe7,0xcd.0xb7,0xb8.0xf8,0xd6.0xc1,0xc4.0xd1,0xb9.0xdb,
\0xd6.0xb8,0xb4.0xb4,0xd6.0xa4,0xd6.0xaf,0xc2.0xdb,0xb1.0xf0,0xce.0xe5,0xd0.0xad,
\0xb1.0xe4,0xb7.0xe7,0xc5.0xfa,0xbc.0xfb,0xbe.0xbf,0xd6.0xa7,0xc4.0xc7,0xb2.0xe9,
\0xd5.0xc5,0xbe.0xab,0xc3.0xbf,0xc1.0xd6,0xd7.0xaa,0xbb.0xae,0xd7.0xbc,0xd7.0xf6,
\0xd0.0xe8,0xb4.0xab,0xd5.0xf9,0xcb.0xb0,0xb9.0xb9,0xbe.0xdf,0xb0.0xd9,0xbb.0xf2,
\0xb2.0xc5,0xbb.0xfd,0xca.0xc6,0xbe.0xd9,0xb1.0xd8,0xd0.0xcd,0xd2.0xd7,0xca.0xd3,
\0xbf.0xec,0xc0.0xee,0xb2.0xce,0xbb.0xd8,0xd2.0xfd,0xd5.0xf2,0xca.0xd7,0xcd.0xc6,
\0xcb.0xbc,0xcd.0xea,0xcf.0xfb,0xd6.0xb5,0xb8.0xc3,0xd7.0xdf,0xd7.0xb0,0xd6.0xda,
\0xd4.0xf0,0xb1.0xb8,0xd6.0xdd,0xb9.0xa9,0xb0.0xfc,0xb8.0xb1,0xbc.0xab,0xd5.0xfb,
\0xc8.0xb7,0xd6.0xaa,0xc3.0xb3,0xbc.0xba,0xbb.0xb7,0xbb.0xb0,0xb7.0xb4,0xc9.0xed,
\0xd1.0xa1,0xd1.0xc7,0xc3.0xb4,0xb4.0xf8,0xb2.0xc9,0xcd.0xf5,0xb2.0xdf,0xd5.0xe6,
\0xc5.0xae,0xcc.0xb8,0xd1.0xcf,0xcb.0xb9,0xbf.0xf6,0xc9.0xab,0xb4.0xf2,0xb5.0xc2,
\0xb8.0xe6,0xbd.0xf6,0xcb.0xfc,0xc6.0xf8,0xc1.0xcf,0xc9.0xf1,0xc2.0xca,0xca.0xb6,
\0xc0.0xcd,0xbe.0xb3,0xd4.0xb4,0xc7.0xe0,0xbb.0xa4,0xc1.0xd0,0xd0.0xcb,0xd0.0xed,
\0xbb.0xa7,0xc2.0xed,0xb8.0xdb,0xd4.0xf2,0xbd.0xda,0xbf.0xee,0xc0.0xad,0xd6.0xb1,
\0xb0.0xb8,0xb9.0xc9,0xb9.0xe2,0xbd.0xcf,0xba.0xd3,0xbb.0xa8,0xb8.0xf9,0xb2.0xbc,
\0xcf.0xdf,0xcd.0xc1,0xbf.0xcb,0xd4.0xd9,0xc8.0xba,0xd2.0xbd,0xc7.0xe5,0xcb.0xd9,
\0xc2.0xc9,0xcb.0xfd,0xd7.0xe5,0xc0.0xfa,0xb7.0xc7,0xb8.0xd0,0xd5.0xbc,0xd0.0xf8,
\0xca.0xa6,0xba.0xce,0xd3.0xb0,0xb9.0xa6,0xb8.0xba,0xd1.0xe9,0xcd.0xfb,0xb2.0xc6,
\0xc0.0xe0,0xbb.0xf5,0xd4.0xbc,0xd2.0xd5,0xca.0xdb,0xc1.0xac,0xbc.0xcd,0xb0.0xb4,
\0xd1.0xb6,0xca.0xb7,0xca.0xbe,0xcf.0xf3,0xd1.0xf8,0xbb.0xf1,0xca.0xaf,0xca.0xb3,
\0xd7.0xa5,0xb8.0xbb,0xc4.0xa3,0xca.0xbc,0xd7.0xa1,0xc8.0xfc,0xbf.0xcd,0xd4.0xbd,
\0xce.0xc5,0xd1.0xeb,0xcf.0xaf,0xbc.0xe1]

" cp950TopChars"{{{1
let s:cp950TopChars=[
\0xaa.0xba,0xa4.0x40,0xb0.0xea,0xa6.0x62,0xa4.0x48,0xa4.0x46,0xa6.0xb3,0xa4.0xa4,
\0xac.0x4f,0xa6.0x7e,0xa9.0x4d,0xa4.0x6a,0xb7.0x7e,0xa4.0xa3,0xac.0xb0,0xb5.0x6f,
\0xb7.0x7c,0xa4.0x75,0xb8.0x67,0xa4.0x57,0xa6.0x61,0xa5.0xab,0xad.0x6e,0xad.0xd3,
\0xb2.0xa3,0xb3.0x6f,0xa5.0x58,0xa6.0xe6,0xa7.0x40,0xa5.0xcd,0xae.0x61,0xa5.0x48,
\0xa6.0xa8,0xa8.0xec,0xa4.0xe9,0xa5.0xc1,0xa8.0xd3,0xa7.0xda,0xb3.0xa1,0xb9.0xef,
\0xb6.0x69,0xa6.0x68,0xa5.0xfe,0xab.0xd8,0xa5.0x4c,0xa4.0xbd,0xb6.0x7d,0xad.0xcc,
\0xb3.0xf5,0xae.0x69,0xae.0xc9,0xb2.0x7a,0xb7.0x73,0xa4.0xe8,0xa5.0x44,0xa5.0xf8,
\0xb8.0xea,0xb9.0xea,0xbe.0xc7,0xb3.0xf8,0xa8.0xee,0xac.0x46,0xc0.0xd9,0xa5.0xce,
\0xa6.0x50,0xa4.0x5f,0xaa.0x6b,0xb0.0xaa,0xaa.0xf8,0xb2.0x7b,0xa5.0xbb,0xa4.0xeb,
\0xa9.0x77,0xa4.0xc6,0xa5.0x5b,0xb0.0xca,0xa6.0x58,0xab.0x7e,0xad.0xab,0xc3.0xf6,
\0xbe.0xf7,0xa4.0xc0,0xa4.0x4f,0xa6.0xdb,0xa5.0x7e,0xaa.0xcc,0xb0.0xcf,0xaf.0xe0,
\0xb3.0x5d,0xab.0xe1,0xb4.0x4e,0xb5.0xa5,0xc5.0xe9,0xa4.0x55,0xb8.0x55,0xa4.0xb8,
\0xaa.0xc0,0xb9.0x4c,0xab.0x65,0xad.0xb1,0xb9.0x41,0xa4.0x5d,0xb1.0x6f,0xbb.0x50,
\0xbb.0xa1,0xa4.0xa7,0xad.0xfb,0xa6.0xd3,0xb0.0xc8,0xa7.0x51,0xb9.0x71,0xa4.0xe5,
\0xa8.0xc6,0xa5.0x69,0xba.0xd8,0xc1.0x60,0xa7.0xef,0xa4.0x54,0xa6.0x55,0xa6.0x6e,
\0xaa.0xf7,0xb2.0xc4,0xa5.0x71,0xa8.0xe4,0xb1.0x71,0xa5.0xad,0xa5.0x4e,0xb7.0xed,
\0xa4.0xd1,0xa4.0xf4,0xac.0xd9,0xb4.0xa3,0xb0.0xd3,0xa4.0x51,0xba.0xde,0xa4.0xba,
\0xa4.0x70,0xa7.0xde,0xa6.0xec,0xa5.0xd8,0xb0.0x5f,0xae.0xfc,0xa9.0xd2,0xa5.0xdf,
\0xa4.0x77,0xb3.0x71,0xa4.0x4a,0xb6.0x71,0xa4.0x6c,0xb0.0xdd,0xab.0xd7,0xa5.0x5f,
\0xab.0x4f,0xa4.0xdf,0xc1.0xd9,0xac.0xec,0xa9.0x65,0xb3.0xa3,0xb3.0x4e,0xa8.0xcf,
\0xa9.0xfa,0xb5.0xdb,0xa6.0xb8,0xb1.0x4e,0xbc.0x57,0xb0.0xf2,0xa6.0x57,0xa6.0x56,
\0xaa.0xf9,0xc0.0xb3,0xb8.0xcc,0xac.0xfc,0xa5.0xd1,0xb3.0x57,0xa4.0xb5,0xc3.0x44,
\0xb0.0x4f,0xc2.0x49,0xad.0x70,0xa5.0x68,0xb1.0x6a,0xa8.0xe2,0xa8.0xc7,0xaa.0xed,
\0xa8.0x74,0xbf.0xec,0xb1.0xd0,0xa5.0xbf,0xb1.0xf8,0xb3.0xcc,0xb9.0x46,0xaf.0x53,
\0xad.0xb2,0xa6.0xac,0xa4.0x47,0xb4.0xc1,0xa8.0xc3,0xb5.0x7b,0xbc.0x74,0xa6.0x70,
\0xb9.0x44,0xbb.0xda,0xa4.0xce,0xa6.0xe8,0xa4.0x66,0xa8.0xca,0xb5.0xd8,0xa5.0xf4,
\0xbd.0xd5,0xa9.0xca,0xbe.0xc9,0xb2.0xd5,0xaa.0x46,0xb8.0xf4,0xac.0xa1,0xbc.0x73,
\0xb7.0x4e,0xa4.0xf1,0xa7.0xeb,0xa8.0x4d,0xa5.0xe6,0xb2.0xce,0xc4.0xd2,0xab.0x6e,
\0xa6.0x77,0xa6.0xb9,0xbb.0xe2,0xb5.0xb2,0xc0.0xe7,0xb6.0xb5,0xb1.0xa1,0xb8.0xd1,
\0xc4.0xb3,0xb8.0x71,0xa4.0x73,0xa5.0xfd,0xa8.0xae,0xb5.0x4d,0xbb.0xf9,0xa9.0xf1,
\0xa5.0x40,0xb6.0xa1,0xa6.0x5d,0xa6.0x40,0xb0.0x7c,0xa8.0x42,0xaa.0xab,0xac.0xc9,
\0xb6.0xb0,0xa7.0xe2,0xab.0xf9,0xb5.0x4c,0xa6.0xfd,0xab.0xb0,0xac.0xdb,0xae.0xd1,
\0xa7.0xf8,0xa8.0x44,0xaa.0x76,0xa8.0xfa,0xad.0xec,0xb3.0x42,0xa9.0xb2,0xac.0xe3,
\0xbd.0xe8,0xab.0x48,0xa5.0x7c,0xb9.0x42,0xbf.0xa4,0xad.0x78,0xa5.0xf3,0xa8.0x7c,
\0xa7.0xbd,0xb7.0x46,0xb6.0xa4,0xb9.0xce,0xa4.0x53,0xb3.0x79,0xa7.0xce,0xaf.0xc5,
\0xbc.0xd0,0xc1.0x70,0xb1.0x4d,0xa4.0xd6,0xb6.0x4f,0xae.0xc4,0xbe.0xda,0xa4.0xe2,
\0xac.0x49,0xc5.0x76,0xa6.0xbf,0xaa.0xf1,0xb2.0x60,0xa7.0xf3,0xbb.0x7b,0xaa.0x47,
\0xae.0xe6,0xb4.0x58,0xac.0xdd,0xa8.0x53,0xc2.0xbe,0xaa.0x41,0xa5.0x78,0xa6.0xa1,
\0xaf.0x71,0xb7.0x51,0xbc.0xc6,0xb3.0xe6,0xbc.0xcb,0xa5.0x75,0xb3.0x51,0xbb.0xf5,
\0xa6.0xd1,0xa8.0xfc,0xc0.0x75,0xb1.0x60,0xbe.0x50,0xa7.0xd3,0xbe.0xd4,0xac.0x79,
\0xab.0xdc,0xb1.0xb5,0xb6.0x6d,0xc0.0x59,0xb5.0xb9,0xa6.0xdc,0xc3.0xf8,0xc6.0x5b,
\0xab.0xfc,0xb3.0xd0,0xb5.0xfd,0xc2.0xb4,0xbd.0xd7,0xa7.0x4f,0xa4.0xad,0xa8.0xf3,
\0xc5.0xdc,0xad.0xb7,0xa7.0xe5,0xa8.0xa3,0xa8.0x73,0xa4.0xe4,0xa8.0xba,0xac.0x64,
\0xb1.0x69,0xba.0xeb,0xa8.0x43,0xaa.0x4c,0xc2.0xe0,0xb9.0xba,0xb7.0xc7,0xb0.0xb5,
\0xbb.0xdd,0xb6.0xc7,0xaa.0xa7,0xb5.0x7c,0xba.0x63,0xa8.0xe3,0xa6.0xca,0xa9.0xce,
\0xa4.0x7e,0xbf.0x6e,0xb6.0xd5,0xc1.0x7c,0xa5.0xb2,0xab.0xac,0xa9.0xf6,0xb5.0xf8,
\0xa7.0xd6,0xa7.0xf5,0xb0.0xd1,0xa6.0x5e,0xa4.0xde,0xc2.0xed,0xad.0xba,0xb1.0xc0,
\0xab.0xe4,0xa7.0xb9,0xae.0xf8,0xad.0xc8,0xb8.0xd3,0xa8.0xab,0xb8.0xcb,0xb2.0xb3,
\0xb3.0x64,0xb3.0xc6,0xa6.0x7b,0xa8.0xd1,0xa5.0x5d,0xb0.0xc6,0xb7.0xa5,0xbe.0xe3,
\0xbd.0x54,0xaa.0xbe,0xb6.0x54,0xa4.0x76,0xc0.0xf4,0xb8.0xdc,0xa4.0xcf,0xa8.0xad,
\0xbf.0xef,0xa8.0xc8,0xbb.0xf2,0xb1.0x61,0xb1.0xc4,0xa4.0xfd,0xb5.0xa6,0xaf.0x75,
\0xa4.0x6b,0xbd.0xcd,0xc4.0x59,0xb4.0xb5,0xaa.0x70,0xa6.0xe2,0xa5.0xb4,0xbc.0x77,
\0xa7.0x69,0xb6.0xc8,0xa5.0xa6,0xae.0xf0,0xae.0xc6,0xaf.0xab,0xb2.0x76,0xc3.0xd1,
\0xb3.0xd2,0xb9.0xd2,0xb7.0xbd,0xab.0x43,0xc5.0x40,0xa6.0x43,0xbf.0xb3,0xb3.0x5c,
\0xa4.0xe1,0xb0.0xa8,0xb4.0xe4,0xab.0x68,0xb8.0x60,0xb4.0xda,0xa9.0xd4,0xaa.0xbd,
\0xae.0xd7,0xaa.0xd1,0xa5.0xfa,0xb8.0xfb,0xaa.0x65,0xaa.0xe1,0xae.0xda,0xa5.0xac,
\0xbd.0x75,0xa4.0x67,0xa7.0x4a,0xa6.0x41,0xb8.0x73,0xc2.0xe5,0xb2.0x4d,0xb3.0x74,
\0xab.0xdf,0xa6.0x6f,0xb1.0xda,0xbe.0xfa,0xab.0x44,0xb7.0x50,0xa6.0xfb,0xc4.0xf2,
\0xae.0x76,0xa6.0xf3,0xbc.0x76,0xa5.0x5c,0xad.0x74,0xc5.0xe7,0xb1.0xe6,0xb0.0x5d,
\0xc3.0xfe,0xb3.0x66,0xac.0xf9,0xc3.0xc0,0xb0.0xe2,0xb3.0x73,0xac.0xf6,0xab.0xf6,
\0xb0.0x54,0xa5.0x76,0xa5.0xdc,0xb6.0x48,0xbe.0x69,0xc0.0xf2,0xa5.0xdb,0xad.0xb9,
\0xa7.0xec,0xb4.0x49,0xbc.0xd2,0xa9.0x6c,0xa6.0xed,0xc1.0xc9,0xab.0xc8,0xb6.0x56,
\0xbb.0x44,0xa5.0xa1,0xae.0x75,0xb0.0xed]

" euc_twTopChars"{{{1
let s:euc_twTopChars=[
\0xce.0xfb,0xc4.0xa1,0xd9.0xcf,0xc7.0xe3,0xc4.0xa9,0xc4.0xa7,0xc8.0xb4,0xc4.0xe3,
\0xd1.0xd2,0xc8.0xa1,0xcc.0xcf,0xc4.0xcb,0xe4.0xc6,0xc4.0xe2,0xd2.0xb3,0xe0.0xf5,
\0xe4.0xc4,0xc4.0xd6,0xe5.0xee,0xc4.0xb8,0xc7.0xe2,0xc6.0xcb,0xd3.0xd1,0xd4.0xb6,
\0xdc.0xa8,0xdd.0xd5,0xc5.0xf8,0xc8.0xe7,0xc9.0xa2,0xc6.0xed,0xd5.0xa5,0xc5.0xe8,
\0xc8.0xa9,0xcc.0xaf,0xc5.0xca,0xc6.0xe1,0xcb.0xf4,0xca.0xbc,0xdd.0xe5,0xe8.0xd7,
\0xe2.0xd0,0xc7.0xe9,0xc7.0xc0,0xd0.0xfa,0xc5.0xec,0xc4.0xfc,0xe2.0xe4,0xd4.0xaf,
\0xde.0xdb,0xd5.0xad,0xd5.0xeb,0xdc.0xa1,0xe4.0xbb,0xc5.0xc9,0xc5.0xe4,0xc7.0xba,
\0xe6.0xf1,0xe8.0xd2,0xf0.0xd0,0xde.0xde,0xcc.0xb1,0xd1.0xc9,0xf4.0xa4,0xc6.0xee,
\0xc7.0xd1,0xc4.0xc0,0xce.0xce,0xd8.0xed,0xcf.0xdb,0xdc.0xa2,0xc6.0xdb,0xc5.0xcc,
\0xcc.0xf9,0xc5.0xa7,0xc5.0xfb,0xd9.0xaf,0xc7.0xd9,0xd0.0xc2,0xd3.0xec,0xf9.0xc3,
\0xf1.0xa2,0xc5.0xa1,0xc4.0xb0,0xc8.0xdc,0xc6.0xc0,0xcf.0xaf,0xd9.0xb4,0xd7.0xe4,
\0xdd.0xc3,0xd1.0xa5,0xde.0xf3,0xe1.0xab,0xfc.0xd5,0xc4.0xb6,0xe5.0xdc,0xc4.0xf7,
\0xcf.0xa3,0xe7.0xb4,0xd0.0xa9,0xd3.0xf2,0xe7.0xa9,0xc4.0xbe,0xda.0xb5,0xea.0xd8,
\0xeb.0xa9,0xc4.0xe6,0xd4.0xde,0xc8.0xd4,0xd9.0xad,0xc9.0xb3,0xe7.0xd9,0xc5.0xc6,
\0xcb.0xe7,0xc6.0xab,0xea.0xa1,0xf4.0xea,0xca.0xd1,0xc4.0xb5,0xc7.0xd6,0xc7.0xef,
\0xcf.0xda,0xdc.0xc9,0xc6.0xb3,0xcc.0xa7,0xda.0xb7,0xc6.0xcd,0xc5.0xee,0xe5.0xb5,
\0xc5.0xb2,0xc5.0xd5,0xd2.0xdc,0xdf.0xc8,0xd9.0xb8,0xc4.0xb2,0xea.0xa7,0xc4.0xf9,
\0xc4.0xd1,0xca.0xc0,0xc8.0xed,0xc6.0xf8,0xd8.0xc4,0xd6.0xc0,0xcd.0xd4,0xc7.0xa1,
\0xc4.0xd8,0xdd.0xd7,0xc4.0xab,0xe2.0xd8,0xc4.0xcd,0xd9.0xc2,0xd0.0xf9,0xc6.0xa1,
\0xcf.0xf1,0xc5.0xc0,0xf5.0xe4,0xd2.0xef,0xcc.0xe7,0xdd.0xe7,0xdd.0xb4,0xcb.0xf0,
\0xcd.0xfc,0xe1.0xe1,0xc8.0xb9,0xd9.0xf2,0xec.0xc1,0xd9.0xd7,0xc7.0xd8,0xc7.0xd7,
\0xcf.0xdc,0xf3.0xdc,0xe6.0xd3,0xd3.0xa1,0xc6.0xf1,0xdd.0xbd,0xc4.0xf4,0xf7.0xee,
\0xd8.0xb4,0xf6.0xb5,0xd3.0xd3,0xc6.0xaa,0xda.0xb0,0xcc.0xa5,0xcb.0xe8,0xcf.0xd0,
\0xcb.0xb7,0xf2.0xd6,0xda.0xf4,0xc6.0xdf,0xdb.0xbe,0xde.0xb2,0xe7.0xae,0xd6.0xd6,
\0xd3.0xf3,0xc8.0xad,0xc4.0xa8,0xdf.0xe6,0xcb.0xe4,0xe1.0xa3,0xec.0xde,0xc7.0xf1,
\0xe7.0xac,0xeb.0xe3,0xc5.0xaf,0xc8.0xe9,0xc4.0xc7,0xcb.0xeb,0xe1.0xde,0xc7.0xb6,
\0xee.0xfe,0xcd.0xcc,0xf0.0xd2,0xdc.0xda,0xce.0xa9,0xe6.0xfb,0xd2.0xa4,0xec.0xdd,
\0xe3.0xf4,0xc5.0xd2,0xca.0xcd,0xca.0xee,0xc7.0xa8,0xdc.0xd3,0xfa.0xdd,0xd0.0xb2,
\0xc7.0xf8,0xc8.0xba,0xeb.0xeb,0xe1.0xb8,0xf4.0xb2,0xe2.0xfa,0xda.0xc5,0xe6.0xd8,
\0xfa.0xbe,0xe5.0xf8,0xc4.0xd4,0xc7.0xbf,0xcb.0xcf,0xe0.0xd3,0xec.0xa4,0xcd.0xf3,
\0xc5.0xe0,0xe2.0xe6,0xc7.0xde,0xc7.0xc1,0xd8.0xe1,0xca.0xe3,0xce.0xec,0xd2.0xcc,
\0xe2.0xf5,0xca.0xc4,0xd1.0xbd,0xe0.0xd2,0xc8.0xfe,0xd0.0xd2,0xd2.0xde,0xd5.0xf3,
\0xca.0xda,0xca.0xe5,0xce.0xd9,0xcc.0xbd,0xd4.0xcf,0xdd.0xa8,0xcd.0xb4,0xd2.0xe6,
\0xef.0xb3,0xcf.0xea,0xc6.0xbe,0xe7.0xaa,0xf1.0xec,0xd3.0xdb,0xc7.0xb5,0xcb.0xbf,
\0xc9.0xfd,0xe3.0xec,0xe2.0xe9,0xe8.0xb6,0xc4.0xb4,0xdd.0xdf,0xca.0xb0,0xd7.0xc8,
\0xed.0xba,0xf4.0xfa,0xd9.0xf1,0xc5.0xb7,0xe2.0xb6,0xd5.0xe6,0xf0.0xe3,0xc5.0xc3,
\0xd1.0xcc,0xfb.0xe2,0xc8.0xc0,0xcf.0xd4,0xdb.0xe5,0xca.0xd5,0xeb.0xa5,0xce.0xaa,
\0xd6.0xaa,0xde.0xfd,0xd2.0xe0,0xca.0xf4,0xf7.0xaa,0xce.0xa4,0xc6.0xba,0xc8.0xa2,
\0xd6.0xf4,0xe3.0xf7,0xed.0xb0,0xde.0xcc,0xed.0xb5,0xc6.0xb7,0xdd.0xb7,0xeb.0xfe,
\0xc8.0xd2,0xcc.0xbf,0xf3.0xc0,0xda.0xa6,0xef.0xda,0xca.0xb5,0xf0.0xdd,0xd1.0xfc,
\0xd0.0xfe,0xda.0xd9,0xe2.0xd4,0xf3.0xa4,0xe1.0xbf,0xc8.0xdd,0xf9.0xc5,0xfd.0xa8,
\0xd1.0xc0,0xde.0xb6,0xe2.0xa5,0xf6.0xfe,0xef.0xa2,0xc9.0xb1,0xc4.0xec,0xcc.0xb6,
\0xfc.0xc8,0xd3.0xf8,0xca.0xc7,0xcb.0xc4,0xcb.0xb6,0xc5.0xc5,0xcb.0xdb,0xd1.0xe7,
\0xda.0xaf,0xea.0xb4,0xca.0xe4,0xce.0xaf,0xf7.0xcb,0xe8.0xa2,0xe4.0xed,0xd8.0xf8,
\0xeb.0xe6,0xe3.0xae,0xce.0xe8,0xe1.0xa4,0xe9.0xac,0xcc.0xa6,0xc8.0xcb,0xcd.0xd0,
\0xc4.0xdf,0xf1.0xd8,0xe3.0xbc,0xf5.0xa8,0xc6.0xd2,0xd0.0xce,0xcd.0xf8,0xe1.0xfe,
\0xca.0xb8,0xca.0xd7,0xd9.0xb6,0xc7.0xdf,0xc5.0xbf,0xf7.0xd8,0xd3.0xfb,0xda.0xe4,
\0xd1.0xa8,0xc9.0xf9,0xd6.0xbc,0xd4.0xab,0xe6.0xda,0xcb.0xcc,0xe6.0xd2,0xdc.0xb8,
\0xdd.0xca,0xde.0xac,0xc7.0xfc,0xcb.0xf2,0xc5.0xfd,0xd9.0xab,0xe4.0xcb,0xf0.0xec,
\0xed.0xfd,0xcf.0xa1,0xe2.0xbb,0xc4.0xd7,0xf4.0xbf,0xe6.0xe3,0xc5.0xb0,0xcb.0xce,
\0xf2.0xd9,0xcb.0xe9,0xeb.0xfb,0xda.0xa7,0xda.0xe8,0xc5.0xde,0xe1.0xac,0xd6.0xf8,
\0xc4.0xcc,0xee.0xf6,0xf9.0xe4,0xdf.0xda,0xce.0xd3,0xc8.0xe3,0xc6.0xd4,0xec.0xe1,
\0xc9.0xcb,0xe3.0xaf,0xc6.0xc6,0xd6.0xb4,0xd5.0xe8,0xd7.0xae,0xdb.0xfb,0xf8.0xfc,
\0xde.0xb8,0xe8.0xba,0xe4.0xe3,0xcf.0xe5,0xfb.0xac,0xc7.0xc4,0xf1.0xfb,0xdd.0xc2,
\0xc5.0xc2,0xd8.0xeb,0xe0.0xab,0xd0.0xac,0xe5.0xe7,0xe0.0xa1,0xcd.0xd6,0xce.0xfe,
\0xd5.0xf9,0xcf.0xb4,0xc7.0xbc,0xe7.0xa4,0xce.0xc8,0xcf.0xc4,0xd5.0xfc,0xc6.0xcc,
\0xee.0xc0,0xc4.0xc8,0xc9.0xac,0xc7.0xc2,0xe5.0xfa,0xf7.0xd0,0xdb.0xd2,0xdd.0xda,
\0xd1.0xa3,0xc7.0xf0,0xda.0xfe,0xf1.0xa5,0xcf.0xe6,0xe3.0xf6,0xc8.0xfc,0xfa.0xfd,
\0xd5.0xba,0xc8.0xf4,0xec.0xe0,0xc5.0xfc,0xd3.0xd7,0xfc.0xd3,0xdb.0xac,0xd8.0xc2,
\0xf9.0xcb,0xdd.0xcc,0xd2.0xfc,0xf8.0xeb,0xd9.0xc7,0xdd.0xd9,0xd2.0xf9,0xd1.0xba,
\0xd8.0xb9,0xc6.0xb8,0xc6.0xfc,0xe2.0xaf,0xef.0xf2,0xf4.0xbd,0xc6.0xfb,0xd3.0xfa,
\0xca.0xce,0xde.0xee,0xed.0xbc,0xcc.0xee,0xc8.0xee,0xf5.0xd4,0xd0.0xea,0xe2.0xbd,
\0xea.0xcc,0xc6.0xc1,0xd5.0xb9,0xd9.0xd2]

" cp932TopChars"{{{1
let s:cp932TopChars=[
\0x82.0xb5,0x82.0xf0,0x82.0xcc,0x82.0xb7,0x82.0xdc,0x82.0xc9,0x82.0xe9,0x81.0x5b,
\0x82.0xc6,0x82.0xcd,0x83.0x93,0x83.0x8b,0x83.0x43,0x83.0x76,0x82.0xc5,0x82.0xa4,
\0x82.0xc4,0x83.0x5e,0x83.0x58,0x82.0xa2,0x8d.0x73,0x82.0xe5,0x82.0xb1,0x82.0xbd,
\0x82.0xea,0x83.0x68,0x82.0xaa,0x93.0xae,0x83.0x62,0x83.0x4a,0x82.0xc8,0x83.0x8c,
\0x83.0x7d,0x83.0x5c,0x82.0xe8,0x88.0xda,0x82.0xb3,0x83.0x52,0x82.0xe7,0x82.0xa9,
\0x83.0x67,0x83.0x74,0x95.0xb6,0x89.0xba,0x83.0x65,0x83.0x40,0x83.0x4c,0x82.0xe0,
\0x89.0x9f,0x83.0x82,0x82.0xab,0x92.0x75,0x8e.0x9a,0x8f.0x9c,0x8d.0xed,0x82.0xe6,
\0x82.0xc1,0x82.0xad,0x82.0xc2,0x8c.0xea,0x93.0xfc,0x82.0xdd,0x83.0x56,0x82.0xbb,
\0x8e.0xa6,0x83.0x49,0x88.0xc8,0x82.0xaf,0x82.0xa6,0x92.0x50,0x8e.0x67,0x8d.0x58,
\0x95.0xcf,0x83.0x87,0x82.0xd6,0x8d.0xf5,0x8e.0x9f,0x8c.0x9f,0x8d.0xc5,0x91.0x53,
\0x91.0x7d,0x88.0xca,0x82.0xa0,0x82.0xdf,0x8a.0xd4,0x8a.0xb7,0x96.0xbc,0x88.0xe1,
\0x8f.0x49,0x95.0xfb,0x8f.0x89,0x83.0x8a,0x83.0x79,0x8c.0xe3,0x95.0xd4,0x8c.0x4a,
\0x89.0xc1,0x91.0x4f,0x83.0x45,0x83.0x42,0x82.0xce,0x97.0x70,0x8e.0xc0,0x82.0xed,
\0x97.0x76,0x90.0xb3,0x90.0x94,0x88.0xea,0x83.0x4e,0x83.0x41,0x82.0xf1,0x82.0xbe,
\0x95.0x94,0x92.0xc7,0x97.0x6c,0x96.0x96,0x91.0xb6,0x8e.0xe6,0x83.0x66,0x8d.0x9e,
\0x8f.0xc1,0x97.0xcd,0x83.0x85,0x83.0x60,0x82.0xb9,0x95.0x5c,0x8a.0x6d,0x92.0xe8,
\0x83.0x77,0x83.0x6d,0x93.0xaa,0x94.0x46,0x96.0xda,0x93.0x78,0x91.0xe5,0x8d.0xec,
\0x8f.0xe3,0x82.0xe2,0x82.0xb6,0x93.0xc7,0x8a.0xae,0x8e.0x6e,0x91.0xbd,0x8f.0xea,
\0x8b.0xe5,0x90.0xe6,0x95.0xdb,0x91.0xcc,0x97.0xb9,0x8a.0x4a,0x8b.0x4e,0x8c.0xa9,
\0x96.0xf1,0x91.0xce,0x8d.0x87,0x92.0x6c,0x83.0x57,0x91.0x49,0x90.0x69,0x90.0xdd,
\0x8f.0x91,0x8e.0x9e,0x8f.0xac,0x8c.0xfc,0x93.0xaf,0x89.0xbd,0x82.0xc7,0x96.0x40,
\0x96.0xdf,0x89.0xf1,0x8f.0x6f,0x83.0x8d,0x82.0xb8,0x82.0xa8,0x8f.0x57,0x8d.0xdb,
\0x8e.0x8e,0x95.0xe2,0x95.0xd2,0x94.0xd4,0x8c.0xbb,0x96.0x7b,0x91.0xf0,0x94.0xf6,
\0x95.0xaa,0x8f.0x43,0x97.0xe1,0x83.0x89,0x96.0xca,0x97.0x97,0x8a.0x6f,0x91.0xb1,
\0x89.0xe6,0x8a.0xf9,0x8f.0x8a,0x8b.0xad,0x8d.0xc4,0x83.0x73,0x83.0x47,0x83.0x46,
\0x93.0xc1,0x96.0xb3,0x90.0xac,0x95.0x4b,0x8a.0x77,0x8a.0x4f,0x8d.0xdd,0x95.0x74,
\0x91.0xbc,0x83.0x6f,0x82.0xde,0x82.0x9f,0x92.0xb2,0x90.0xe0,0x94.0x5c,0x8f.0x4b,
\0x94.0xcd,0x92.0x5b,0x92.0xbc,0x8f.0xf3,0x94.0xc5,0x97.0x88,0x96.0xbe,0x8a.0x87,
\0x91.0xd4,0x8c.0xca,0x8a.0xdc,0x8d.0x86,0x95.0xca,0x93.0xe0,0x90.0x6c,0x92.0x86,
\0x97.0x5e,0x82.0xe4,0x82.0xcb,0x82.0xbf,0x8b.0x74,0x93.0x5c,0x89.0xf0,0x95.0xa1,
\0x8e.0xa9,0x8b.0xf3,0x92.0x6d,0x94.0x92,0x97.0x9d,0x88.0xd7,0x92.0x8d,0x8e.0x63,
\0x8b.0xc6,0x91.0x80,0x8e.0x77,0x94.0xb2,0x8a.0xb5,0x88.0xd3,0x8f.0xee,0x89.0x9e,
\0x92.0xa5,0x8c.0x60,0x8e.0xae,0x8a.0xf4,0x95.0xf1,0x88.0xcd,0x89.0x45,0x89.0xc2,
\0x8b.0xe6,0x95.0x73,0x83.0x80,0x83.0x72,0x83.0x69,0x82.0xeb,0x82.0xd9,0x82.0xd7,
\0x82.0xd1,0x82.0xc3,0x97.0xde,0x91.0xe8,0x90.0xc2,0x8d.0x7e,0x95.0xc2,0x98.0x41,
\0x92.0xca,0x91.0xab,0x90.0xd4,0x91.0x45,0x97.0xfb,0x91.0x67,0x94.0x5b,0x8a.0xc8,
\0x91.0xe6,0x8f.0xcd,0x97.0xa7,0x8e.0xd0,0x94.0x6a,0x96.0xee,0x88.0xd9,0x8f.0xc6,
\0x8c.0x88,0x8b.0x43,0x8a.0xfc,0x8a.0x69,0x90.0x84,0x8e.0x9d,0x96.0x59,0x90.0x53,
\0x93.0xbe,0x96.0xf0,0x88.0xf8,0x8d.0xb6,0x8d.0x44,0x8e.0x51,0x88.0xf3,0x8c.0xf8,
\0x8c.0xb3,0x93.0xad,0x90.0x4d,0x83.0x83,0x83.0x7e,0x83.0x75,0x83.0x5b,0x83.0x59,
\0x83.0x55,0x83.0x4f,0x82.0xd4,0x82.0xbc,0x82.0xb0,0x82.0xac,0x97.0xed,0x95.0x70,
\0x8f.0x87,0x8d.0x80,0x94.0xf1,0x8b.0xf7,0x8c.0xaf,0x92.0xb7,0x94.0x7a,0x98.0x59,
\0x93.0x4b,0x92.0x42,0x91.0xac,0x93.0x72,0x8f.0x71,0x8d.0xda,0x8c.0x79,0x8e.0xd4,
\0x90.0x67,0x93.0xa5,0x8b.0x4d,0x92.0x4e,0x8c.0xeb,0x8f.0xda,0x96.0xf3,0x8c.0x76,
\0x8c.0xbe,0x90.0x65,0x8e.0x8b,0x90.0xbb,0x97.0x8e,0x90.0x46,0x97.0xc7,0x92.0x76,
\0x8e.0xd2,0xe3.0x59,0x8d.0xd7,0x8e.0x86,0x8f.0x83,0x93.0x9c,0x89.0xd3,0x93.0x99,
\0x95.0x84,0x92.0xf6,0x8e.0x84,0x8d.0xbb,0x92.0x5a,0x8a.0xc4,0x93.0x49,0x94.0xad,
\0x97.0xaa,0x8a.0xc3,0x93.0x5f,0x91.0xd7,0x8b.0x81,0x92.0x69,0x8b.0x40,0x8d.0x5c,
\0x8a.0x54,0x8a.0x79,0x8f.0xbc,0x91.0xba,0x8a.0xfa,0x93.0xfa,0x8b.0xb3,0x8a.0xf6,
\0x8c.0x66,0x8b.0x93,0x91.0xc5,0x8e.0xe8,0x8a.0xb4,0x8c.0x62,0x8e.0x76,0x94.0x4f,
\0x89.0xf5,0x8d.0x4f,0x8d.0x4c,0x8f.0xed,0x89.0xaa,0x97.0x65,0x8f.0xa7,0x91.0xbe,
\0x8a.0xee,0x90.0x7d,0x92.0x51,0x96.0xe2,0x8c.0xc4,0x96.0xa1,0x8d.0x90,0x8c.0xc3,
\0x8b.0x79,0x8f.0x5c,0x95.0xd7,0x8d.0x8f,0x97.0x98,0x94.0xbb,0x96.0x60,0x8b.0xa4,
\0x8f.0x5b,0x94.0xf5,0x95.0xd6,0x88.0xcb,0x95.0xb9,0x8e.0x97,0x93.0x60,0x91.0xe3,
\0x8e.0x64,0x8d.0xa1,0x97.0xbc,0x96.0x9c,0x83.0x86,0x83.0x84,0x83.0x71,0x83.0x70,
\0x83.0x6a,0x83.0x54,0x82.0xda,0x82.0xb2,0x81.0x58]

" euc_jpTopChars"{{{1
let s:euc_jpTopChars=[
\0xa4.0xb7,0xa4.0xf2,0xa4.0xce,0xa4.0xb9,0xa4.0xde,0xa4.0xcb,0xa4.0xeb,0xa1.0xbc,
\0xa4.0xc8,0xa4.0xcf,0xa5.0xf3,0xa5.0xeb,0xa5.0xa4,0xa5.0xd7,0xa4.0xc7,0xa4.0xa6,
\0xa4.0xc6,0xa5.0xbf,0xa5.0xb9,0xa4.0xa4,0xb9.0xd4,0xa4.0xe7,0xa4.0xb3,0xa4.0xbf,
\0xa4.0xec,0xa5.0xc9,0xa4.0xac,0xc6.0xb0,0xa5.0xc3,0xa5.0xab,0xa4.0xca,0xa5.0xec,
\0xa5.0xde,0xa5.0xbd,0xa4.0xea,0xb0.0xdc,0xa4.0xb5,0xa5.0xb3,0xa4.0xe9,0xa4.0xab,
\0xa5.0xc8,0xa5.0xd5,0xca.0xb8,0xb2.0xbc,0xa5.0xc6,0xa5.0xa1,0xa5.0xad,0xa4.0xe2,
\0xb2.0xa1,0xa5.0xe2,0xa4.0xad,0xc3.0xd6,0xbb.0xfa,0xbd.0xfc,0xba.0xef,0xa4.0xe8,
\0xa4.0xc3,0xa4.0xaf,0xa4.0xc4,0xb8.0xec,0xc6.0xfe,0xa4.0xdf,0xa5.0xb7,0xa4.0xbd,
\0xbc.0xa8,0xa5.0xaa,0xb0.0xca,0xa4.0xb1,0xa4.0xa8,0xc3.0xb1,0xbb.0xc8,0xb9.0xb9,
\0xca.0xd1,0xa5.0xe7,0xa4.0xd8,0xba.0xf7,0xbc.0xa1,0xb8.0xa1,0xba.0xc7,0xc1.0xb4,
\0xc1.0xde,0xb0.0xcc,0xa4.0xa2,0xa4.0xe1,0xb4.0xd6,0xb4.0xb9,0xcc.0xbe,0xb0.0xe3,
\0xbd.0xaa,0xca.0xfd,0xbd.0xe9,0xa5.0xea,0xa5.0xda,0xb8.0xe5,0xca.0xd6,0xb7.0xab,
\0xb2.0xc3,0xc1.0xb0,0xa5.0xa6,0xa5.0xa3,0xa4.0xd0,0xcd.0xd1,0xbc.0xc2,0xa4.0xef,
\0xcd.0xd7,0xc0.0xb5,0xbf.0xf4,0xb0.0xec,0xa5.0xaf,0xa5.0xa2,0xa4.0xf3,0xa4.0xc0,
\0xc9.0xf4,0xc4.0xc9,0xcd.0xcd,0xcb.0xf6,0xc2.0xb8,0xbc.0xe8,0xa5.0xc7,0xb9.0xfe,
\0xbe.0xc3,0xce.0xcf,0xa5.0xe5,0xa5.0xc1,0xa4.0xbb,0xc9.0xbd,0xb3.0xce,0xc4.0xea,
\0xa5.0xd8,0xa5.0xce,0xc6.0xac,0xc7.0xa7,0xcc.0xdc,0xc5.0xd9,0xc2.0xe7,0xba.0xee,
\0xbe.0xe5,0xa4.0xe4,0xa4.0xb8,0xc6.0xc9,0xb4.0xb0,0xbb.0xcf,0xc2.0xbf,0xbe.0xec,
\0xb6.0xe7,0xc0.0xe8,0xca.0xdd,0xc2.0xce,0xce.0xbb,0xb3.0xab,0xb5.0xaf,0xb8.0xab,
\0xcc.0xf3,0xc2.0xd0,0xb9.0xe7,0xc3.0xcd,0xa5.0xb8,0xc1.0xaa,0xbf.0xca,0xc0.0xdf,
\0xbd.0xf1,0xbb.0xfe,0xbe.0xae,0xb8.0xfe,0xc6.0xb1,0xb2.0xbf,0xa4.0xc9,0xcb.0xa1,
\0xcc.0xe1,0xb2.0xf3,0xbd.0xd0,0xa5.0xed,0xa4.0xba,0xa4.0xaa,0xbd.0xb8,0xba.0xdd,
\0xbb.0xee,0xca.0xe4,0xca.0xd4,0xc8.0xd6,0xb8.0xbd,0xcb.0xdc,0xc2.0xf2,0xc8.0xf8,
\0xca.0xac,0xbd.0xa4,0xce.0xe3,0xa5.0xe9,0xcc.0xcc,0xcd.0xf7,0xb3.0xd0,0xc2.0xb3,
\0xb2.0xe8,0xb4.0xfb,0xbd.0xea,0xb6.0xaf,0xba.0xc6,0xa5.0xd4,0xa5.0xa8,0xa5.0xa7,
\0xc6.0xc3,0xcc.0xb5,0xc0.0xae,0xc9.0xac,0xb3.0xd8,0xb3.0xb0,0xba.0xdf,0xc9.0xd5,
\0xc2.0xbe,0xa5.0xd0,0xa4.0xe0,0xa4.0xa1,0xc4.0xb4,0xc0.0xe2,0xc7.0xbd,0xbd.0xac,
\0xc8.0xcf,0xc3.0xbc,0xc4.0xbe,0xbe.0xf5,0xc8.0xc7,0xcd.0xe8,0xcc.0xc0,0xb3.0xe7,
\0xc2.0xd6,0xb8.0xcc,0xb4.0xde,0xb9.0xe6,0xca.0xcc,0xc6.0xe2,0xbf.0xcd,0xc3.0xe6,
\0xcd.0xbf,0xa4.0xe6,0xa4.0xcd,0xa4.0xc1,0xb5.0xd5,0xc5.0xbd,0xb2.0xf2,0xca.0xa3,
\0xbc.0xab,0xb6.0xf5,0xc3.0xce,0xc7.0xf2,0xcd.0xfd,0xb0.0xd9,0xc3.0xed,0xbb.0xc4,
\0xb6.0xc8,0xc1.0xe0,0xbb.0xd8,0xc8.0xb4,0xb4.0xb7,0xb0.0xd5,0xbe.0xf0,0xb1.0xfe,
\0xc4.0xa7,0xb7.0xc1,0xbc.0xb0,0xb4.0xf6,0xca.0xf3,0xb0.0xcf,0xb1.0xa6,0xb2.0xc4,
\0xb6.0xe8,0xc9.0xd4,0xa5.0xe0,0xa5.0xd3,0xa5.0xca,0xa4.0xed,0xa4.0xdb,0xa4.0xd9,
\0xa4.0xd3,0xa4.0xc5,0xce.0xe0,0xc2.0xea,0xc0.0xc4,0xb9.0xdf,0xca.0xc4,0xcf.0xa2,
\0xc4.0xcc,0xc2.0xad,0xc0.0xd6,0xc1.0xa6,0xce.0xfd,0xc1.0xc8,0xc7.0xbc,0xb4.0xca,
\0xc2.0xe8,0xbe.0xcf,0xce.0xa9,0xbc.0xd2,0xc7.0xcb,0xcc.0xf0,0xb0.0xdb,0xbe.0xc8,
\0xb7.0xe8,0xb5.0xa4,0xb4.0xfe,0xb3.0xca,0xbf.0xe4,0xbb.0xfd,0xcb.0xba,0xbf.0xb4,
\0xc6.0xc0,0xcc.0xf2,0xb0.0xfa,0xba.0xb8,0xb9.0xa5,0xbb.0xb2,0xb0.0xf5,0xb8.0xfa,
\0xb8.0xb5,0xc6.0xaf,0xbf.0xae,0xa5.0xe3,0xa5.0xdf,0xa5.0xd6,0xa5.0xbc,0xa5.0xba,
\0xa5.0xb6,0xa5.0xb0,0xa4.0xd6,0xa4.0xbe,0xa4.0xb2,0xa4.0xae,0xce.0xef,0xc9.0xd1,
\0xbd.0xe7,0xb9.0xe0,0xc8.0xf3,0xb6.0xf9,0xb8.0xb1,0xc4.0xb9,0xc7.0xdb,0xcf.0xba,
\0xc5.0xac,0xc3.0xa3,0xc2.0xae,0xc5.0xd3,0xbd.0xd2,0xba.0xdc,0xb7.0xda,0xbc.0xd6,
\0xbf.0xc8,0xc6.0xa7,0xb5.0xae,0xc3.0xaf,0xb8.0xed,0xbe.0xdc,0xcc.0xf5,0xb7.0xd7,
\0xb8.0xc0,0xbf.0xc6,0xbb.0xeb,0xc0.0xbd,0xcd.0xee,0xbf.0xa7,0xce.0xc9,0xc3.0xd7,
\0xbc.0xd4,0xe5.0xba,0xba.0xd9,0xbb.0xe6,0xbd.0xe3,0xc5.0xfc,0xb2.0xd5,0xc5.0xf9,
\0xc9.0xe4,0xc4.0xf8,0xbb.0xe4,0xba.0xbd,0xc3.0xbb,0xb4.0xc6,0xc5.0xaa,0xc8.0xaf,
\0xce.0xac,0xb4.0xc5,0xc5.0xc0,0xc2.0xd9,0xb5.0xe1,0xc3.0xca,0xb5.0xa1,0xb9.0xbd,
\0xb3.0xb5,0xb3.0xda,0xbe.0xbe,0xc2.0xbc,0xb4.0xfc,0xc6.0xfc,0xb6.0xb5,0xb4.0xf8,
\0xb7.0xc7,0xb5.0xf3,0xc2.0xc7,0xbc.0xea,0xb4.0xb6,0xb7.0xc3,0xbb.0xd7,0xc7.0xb0,
\0xb2.0xf7,0xb9.0xb0,0xb9.0xad,0xbe.0xef,0xb2.0xac,0xcd.0xc6,0xbe.0xa9,0xc2.0xc0,
\0xb4.0xf0,0xbf.0xde,0xc3.0xb2,0xcc.0xe4,0xb8.0xc6,0xcc.0xa3,0xb9.0xf0,0xb8.0xc5,
\0xb5.0xda,0xbd.0xbd,0xca.0xd9,0xb9.0xef,0xcd.0xf8,0xc8.0xbd,0xcb.0xc1,0xb6.0xa6,
\0xbd.0xbc,0xc8.0xf7,0xca.0xd8,0xb0.0xcd,0xca.0xbb,0xbb.0xf7,0xc5.0xc1,0xc2.0xe5,
\0xbb.0xc5,0xba.0xa3,0xce.0xbe,0xcb.0xfc,0xa5.0xe6,0xa5.0xe4,0xa5.0xd2,0xa5.0xd1,
\0xa5.0xcb,0xa5.0xb5,0xa4.0xdc,0xa4.0xb4,0xa1.0xb9]

" cp949TopChars(euc-kr){{{1
let s:cp949TopChars=[
\0xb0.0xa1,0xb0.0xa3,0xb0.0xa9,0xb0.0xaa,0xb0.0xac,0xb0.0xad,0xb0.0xb0,0xb0.0xb3,
\0xb0.0xc5,0xb0.0xc9,0xb0.0xcb,0xb0.0xcd,0xb0.0xd4,0xb0.0xdc,0xb0.0xdf,0xb0.0xe5,
\0xb0.0xe6,0xb0.0xe8,0xb0.0xed,0xb0.0xf7,0xb0.0xf8,0xb0.0xfa,0xb0.0xfc,0xb0.0xfd,
\0xb1.0xb3,0xb1.0xb8,0xb1.0xb9,0xb1.0xd7,0xb1.0xdb,0xb1.0xe2,0xb1.0xe4,0xb1.0xe6,
\0xb1.0xe9,0xb1.0xeb,0xb1.0xee,0xb2.0xd9,0xb2.0xdb,0xb2.0xdc,0xb2.0xdf,0xb2.0xee,
\0xb2.0xef,0xb3.0xa1,0xb3.0xa2,0xb3.0xa6,0xb3.0xaa,0xb3.0xb2,0xb3.0xb5,0xb3.0xbb,
\0xb3.0xc0,0xb3.0xd0,0xb3.0xd1,0xb3.0xd6,0xb3.0xd7,0xb3.0xe4,0xb3.0xf5,0xb4.0xa9,
\0xb4.0xad,0xb4.0xba,0xb4.0xc0,0xb4.0xc2,0xb4.0xc9,0xb4.0xcf,0xb4.0xd9,0xb4.0xdc,
\0xb4.0xdd,0xb4.0xde,0xb4.0xe7,0xb4.0xeb,0xb4.0xf5,0xb4.0xf8,0xb5.0xa5,0xb5.0xb5,
\0xb5.0xb9,0xb5.0xbf,0xb5.0xc7,0xb5.0xc8,0xb5.0xc9,0xb5.0xcb,0xb5.0xce,0xb5.0xd1,
\0xb5.0xda,0xb5.0xe5,0xb5.0xe6,0xb5.0xe7,0xb5.0xe9,0xb5.0xec,0xb5.0xee,0xb5.0xf0,
\0xb5.0xfb,0xb6.0xa7,0xb6.0xb2,0xb6.0xbc,0xb6.0xc7,0xb6.0xc8,0xb6.0xe6,0xb6.0xf3,
\0xb6.0xf4,0xb6.0xf7,0xb6.0xf8,0xb7.0xa1,0xb7.0xa5,0xb7.0xa7,0xb7.0xab,0xb7.0xaf,
\0xb7.0xb1,0xb7.0xb3,0xb7.0xb6,0xb7.0xb8,0xb7.0xba,0xb7.0xc1,0xb7.0xc2,0xb7.0xc9,
\0xb7.0xce,0xb7.0xcf,0xb7.0xd2,0xb7.0xe1,0xb7.0xe7,0xb7.0xf9,0xb8.0xa3,0xb8.0xa5,
\0xb8.0xa6,0xb8.0xa7,0xb8.0xa8,0xb8.0xae,0xb8.0xb0,0xb8.0xb1,0xb8.0xb2,0xb8.0xb3,
\0xb8.0xb6,0xb8.0xb7,0xb8.0xb8,0xb8.0xb9,0xb8.0xbb,0xb8.0xc2,0xb8.0xc5,0xb8.0xd3,
\0xb8.0xd5,0xb8.0xe7,0xb8.0xe9,0xb8.0xed,0xb8.0xee,0xb8.0xf0,0xb8.0xf1,0xb8.0xf8,
\0xb9.0xab,0xb9.0xae,0xb9.0xcc,0xb9.0xd9,0xb9.0xdd,0xb9.0xde,0xb9.0xdf,0xb9.0xe6,
\0xb9.0xe8,0xb9.0xe9,0xb9.0xf6,0xb9.0xf8,0xb9.0xfd,0xba.0xae,0xba.0xaf,0xba.0xb0,
\0xba.0xb8,0xba.0xb9,0xba.0xbb,0xba.0xbc,0xba.0xbe,0xba.0xc1,0xba.0xce,0xba.0xd0,
\0xba.0xd2,0xba.0xd9,0xba.0xf1,0xba.0xf6,0xba.0xfc,0xbb.0xd3,0xbb.0xe7,0xbb.0xe8,
\0xbb.0xec,0xbb.0xef,0xbb.0xf0,0xbb.0xf3,0xbb.0xf5,0xbb.0xf6,0xbb.0xfd,0xbc.0xad,
\0xbc.0xb1,0xbc.0xb3,0xbc.0xb8,0xbc.0xba,0xbc.0xbc,0xbc.0xbd,0xbc.0xc7,0xbc.0xd2,
\0xbc.0xd3,0xbc.0xf6,0xbc.0xf7,0xbc.0xf8,0xbd.0xa9,0xbd.0xb1,0xbd.0xba,0xbd.0xc0,
\0xbd.0xc3,0xbd.0xc4,0xbd.0xc5,0xbd.0xc7,0xbd.0xc9,0xbd.0xca,0xbd.0xcd,0xbd.0xe1,
\0xbd.0xe8,0xbe.0xb2,0xbe.0xb5,0xbe.0xb9,0xbe.0xc0,0xbe.0xc6,0xbe.0xc8,0xbe.0xca,
\0xbe.0xcb,0xbe.0xd6,0xbe.0xdf,0xbe.0xe0,0xbe.0xee,0xbe.0xef,0xbe.0xf0,0xbe.0xf2,
\0xbe.0xf3,0xbe.0xf8,0xbe.0xfa,0xbf.0xa1,0xbf.0xa9,0xbf.0xaa,0xbf.0xac,0xbf.0xad,
\0xbf.0xb4,0xbf.0xb5,0xbf.0xb9,0xbf.0xc0,0xbf.0xc2,0xbf.0xc3,0xbf.0xc5,0xbf.0xc9,
\0xbf.0xcd,0xbf.0xcf,0xbf.0xdc,0xbf.0xde,0xbf.0xe4,0xbf.0xeb,0xbf.0xec,0xbf.0xee,
\0xbf.0xef,0xbf.0xf2,0xbf.0xf3,0xbf.0xf6,0xbf.0xf8,0xc0.0xa7,0xc0.0xad,0xc0.0xaf,
\0xc0.0xb8,0xc0.0xba,0xc0.0xbb,0xc0.0xbd,0xc0.0xc0,0xc0.0xc7,0xc0.0xcc,0xc0.0xcd,
\0xc0.0xce,0xc0.0xcf,0xc0.0xd0,0xc0.0xd4,0xc0.0xd6,0xc0.0xd8,0xc0.0xda,0xc0.0xdb,
\0xc0.0xdf,0xc0.0xe2,0xc0.0xe5,0xc0.0xe6,0xc0.0xe7,0xc0.0xfa,0xc0.0xfb,0xc0.0xfc,
\0xc1.0xa1,0xc1.0xa2,0xc1.0xa4,0xc1.0xa6,0xc1.0xae,0xc1.0xb3,0xc1.0xb6,0xc1.0xb8,
\0xc1.0xbb,0xc1.0xbe,0xc1.0xc1,0xc1.0xd6,0xc1.0xd8,0xc1.0xd9,0xc1.0xdf,0xc1.0xf6,
\0xc1.0xf7,0xc1.0xf8,0xc1.0xfa,0xc1.0xfd,0xc2.0xa6,0xc2.0xb0,0xc2.0xca,0xc2.0xf7,
\0xc2.0xf8,0xc2.0xf9,0xc2.0xfc,0xc3.0xa2,0xc3.0xa3,0xc3.0xa4,0xc3.0xa5,0xc3.0xb3,
\0xc3.0xb5,0xc3.0xb9,0xc3.0xbc,0xc3.0xc4,0xc3.0xc6,0xc3.0xca,0xc3.0xdf,0xc3.0xe2,
\0xc3.0xe6,0xc3.0xeb,0xc4.0xa1,0xc4.0xa3,0xc4.0xa8,0xc4.0xbf,0xc4.0xd1,0xc5.0xa9,
\0xc5.0xad,0xc5.0xb0,0xc5.0xb8,0xc5.0xc2,0xc5.0xc3,0xc5.0xcd,0xc5.0xd8,0xc5.0xdb,
\0xc5.0xe4,0xc5.0xeb,0xc6.0xae,0xc6.0xaf,0xc6.0xc4,0xc6.0xc7,0xc6.0xdb,0xc6.0xe4,
\0xc6.0xec,0xc6.0xed,0xc6.0xf7,0xc6.0xf8,0xc7.0xa5,0xc7.0xc1,0xc7.0xc7,0xc7.0xca,
\0xc7.0xcf,0xc7.0xd1,0xc7.0xd2,0xc7.0xd4,0xc7.0xd5,0xc7.0xd8,0xc7.0xdf,0xc7.0xe0,
\0xc7.0xe2,0xc7.0xe8,0xc7.0xf4,0xc7.0xf6,0xc7.0xfc,0xc8.0xa3,0xc8.0xad,0xc8.0xae,
\0xc8.0xaf,0xc8.0xbd,0xc8.0xc4,0xc8.0xce,0xc8.0xf7,0xc8.0xf9]


function! s:NormalizeEncodingName(enc) "{{{1
    if a:enc=='gbk'
        return 'cp936'
    elseif has('win32') || has('win32unix') || has('win64') || has('iconv')
        if a:enc=='gb2312'
            return 'cp936'
        elseif a:enc=='big5'
            return 'cp950'
        endif
    else " Unix and w/o iconv
        if a:enc=='gb2312'
            return 'euc-cn'
        endif
    endif
    return a:enc
endfunction


function! s:ConvertHtmlEncoding(enc) "{{{1
    if a:enc=~?'^gb2312'
        return 'cp936'          " GB2312 imprecisely means CP936 in HTML
    elseif a:enc==? 'iso-8859-1'
        return 'latin1'         " The canonical encoding name in Vim
    elseif a:enc==? 'utf8'
        return 'utf-8'          " Other encoding aliases should follow here
    else
        return s:NormalizeEncodingName(tolower(a:enc))
    endif
endfunction


function! s:CheckModelineFileEncoding() "{{{1
    if exists("b:fencview_modeline_checked")
        return
    endif
    let b:fencview_modeline_checked=1
    if &modified && &fileencoding!=''
        if s:disable_autodetection<2
            let Syn=&syntax
            exec 'edit! ++enc='.&fileencoding
            if Syn!=''
                let &syntax=Syn
            endif
        elseif exists('s:fencview_manual_enc')
            let &fileencoding=s:fencview_manual_enc
            set nomodified
        endif
    endif
endfunction


function! s:DetectHtmlEncoding() " {{{1
    normal m`
    normal gg
    if search('\c<meta http-equiv=\("\?\)Content-Type\1 content="text/html; charset=[-A-Za-z0-9_]\+">')!=0
        let charset=matchstr(getline('.'), 'text/html; charset=\zs[-A-Za-z0-9_]\+', col('.') - 1)
        let charset=s:ConvertHtmlEncoding(charset)
        normal ``
        if &fileencodings==''
          let auto_encodings=','.&encoding.','
        else
          let auto_encodings=','.&fileencodings.','
        endif
        if charset!=?&fileencoding &&
           \(auto_encodings=~','.&fileencoding.',' || &fileencoding=='')
            try
                let s:disable_autodetection=1
                let Syn=&syntax
                silent! exec 'edit ++enc='.charset
                if Syn!=''
                    let &syntax=Syn
                endif
            finally
                let s:disable_autodetection=0
            endtry
        endif
        return 1
    else
        return 0
    endif
endfunction


function! s:EditManualEncoding(enc, ...) "{{{1
    if a:0>1
        echohl Error | echomsg 'Only one file name should be supplied' | echohl None
        return
    endif
    if a:0==1
        let filename=' '.a:1
    else
        let filename=''
    endif
    try
        let s:disable_autodetection=2
        let s:fencview_manual_enc=a:enc
        let Syn=&syntax
        exec 'edit ++enc='.a:enc.filename
        if Syn!=''
            let &syntax=Syn
        endif
    finally
        let s:disable_autodetection=0
        unlet s:fencview_manual_enc
    endtry
endfunction


function! s:EditAutoEncoding(...) "{{{1
    if s:disable_autodetection || &buftype=='help'
        return
    endif
    if bufname(winnr())==s:FencWinName
        return
    endif
    if  &modified
        echohl Error | echomsg "File is modified!" | echohl None
        return
    endif
    if !has('iconv')
        echohl Error | echomsg "\"+iconv\" feature not found, see Tip #1 in fencview.vim" | echohl None
        return
    endif
    if a:0>1
        echohl Error | echomsg 'Only one file name should be supplied' | echohl None
        return
    endif
    if a:0==1
        let filename=iconv(a:1, &encoding, g:legacy_encoding)
        let filename_e=' '.a:1
    else
        let filename=iconv(expand('%:p'), &encoding, g:legacy_encoding)
        let filename_e=''
    endif
    if a:0==1
        try
            let s:disable_autodetection=1
            exec 'e'.filename_e
        finally
            let s:disable_autodetection=0
        endtry
    endif
    if ','.g:fencview_html_filetypes.',' =~ ','.&filetype.','
        if s:DetectHtmlEncoding()
            return
        endif
    endif
    if ($FENCVIEW_TELLENC == "fencview") || !executable($FENCVIEW_TELLENC)
        call s:FencDetectFileEncoding()
        return
    endif
    try
        if has('gui_running')
            if exists('$VIM_SYSTEM_HIDECONSOLE')
                let vim_system_hideconsole_bak=$VIM_SYSTEM_HIDECONSOLE
            else
                let vim_system_hideconsole_bak=0
            endif
            let $VIM_SYSTEM_HIDECONSOLE=1
        endif
        let result=system($FENCVIEW_TELLENC . ' ' . s:escape(filename))
    finally
        if has('gui_running')
            let $VIM_SYSTEM_HIDECONSOLE=vim_system_hideconsole_bak
        endif
    endtry
    let result=substitute(result, '\n$', '', '')
    if v:shell_error!=0
        echohl Error|echomsg iconv(result, g:legacy_encoding, &encoding)|echohl None
        return
    endif
    let result=s:NormalizeEncodingName(result)
    if result!=&fileencoding && !(result=='ascii' && &fileencoding=='utf-8')
        if result == 'binary'
            echomsg 'Binary file'
            sleep 1
        elseif result == 'unknown'
            echomsg 'Unknown encoding'
            sleep 1
        else
            try
                let s:disable_autodetection=1
                let Syn=&syntax
                exec 'edit ++enc='.result.filename_e
                if Syn!=''
                    let &syntax=Syn
                endif
            finally
                let s:disable_autodetection=0
            endtry
        endif
    endif
endfunction


function! s:ToggleFencView() "{{{1
    let FencWinNr=bufwinnr(s:FencWinName)
    if FencWinNr!=-1
        exec FencWinNr." wincmd w"
        exec "wincmd c"
        return
    endif
    let _tmpfenc=&fenc
    if  &modified
        echohl Error | echomsg "File is modified!" | echohl None
        return
    endif
    let splitLocation="belowright "
    let splitMode="vertical "
    let splitSize=34
    let cmd=splitLocation.splitMode.splitSize.' new '.s:FencWinName
    silent! execute cmd
    setlocal winfixwidth
    setlocal noswapfile
    setlocal buftype=nowrite
    setlocal bufhidden=delete
    setlocal nowrap
    setlocal foldcolumn=0
    setlocal nobuflisted
    setlocal nospell
    setlocal nonumber
    setlocal cursorline
    call append(0,s:Fenc8bit)
    call append(0,s:Fenc16bit)
    call append(0,s:FencUnicode)
    normal Gddgg
    setlocal readonly
    setlocal nomodifiable
    syn match Type "^.\{-}\s"
    syn match Comment "\s.*$"
    if _tmpfenc!=""
        let s=search(_tmpfenc)
        if s!=0
            let _line=getline(line("."))
            let _fenc=substitute(_line,'\s.*$','',"g")
            syn clear Search
            exec "syn match Search \""._line."\""
        endif
    else
        normal gg
    endif
    nnoremap <buffer> <CR> :call <SID>FencSelect()<CR>
    nnoremap <buffer> <2-leftmouse> :call <SID>FencSelect()<CR>
endfunction


function! s:FencSelect() "{{{1
    let _line=getline(line("."))
    let _fenc=substitute(_line,'\s.*$','',"g")
    if _fenc==''
        return
    endif
    syn clear Search
    exec "syn match Search \""._line."\""
    let MainWinNr=winnr("#")
    if MainWinNr==0
        echohl Error | echomsg "Main window not found!" | echohl None
        return
    endif
    exec MainWinNr." wincmd w"
    if  &modified
        echohl Error | echomsg "File is modified!" | echohl None
        return
    endif
    try
        let s:disable_autodetection=2
        exec "edit ++enc="._fenc
    finally
        let s:disable_autodetection=0
    endtry
    let FencWinNr=bufwinnr(s:FencWinName)
    if FencWinNr==-1
        echohl Error | echomsg "Encoding list window not found!" | echohl None
        return
    endif
    exec FencWinNr." wincmd w"
endfunction


function! s:FencCreateMenu() "{{{1
    au! FencView BufEnter
    for i in s:Fenc8bit
        let fencla=substitute(i,'\s.*$','','g')
        let fenname=fencla.'<tab>('.substitute(i,'^.\{-}\s\+','','g').')'
        let fenname=substitute(fenname,' ','\\ ','g')
        let fenname=substitute(fenname,'\.','_','g')
        exec 'menu &Tools.Encodin&g.&8bit\ encodings.'.fenname.' :call FencMenuSel("'.fencla.'")<CR>'
    endfor
    for i in s:Fenc16bit
        let fencla=substitute(i,'\s.*$','','g')
        let fenname=fencla.'<tab>('.substitute(i,'^.\{-}\s\+','','g').')'
        let fenname=substitute(fenname,' ','\\ ','g')
        let fenname=substitute(fenname,'\.','_','g')
        exec 'menu &Tools.Encodin&g.&16bit\ encodings.'.fenname.' :call FencMenuSel("'.fencla.'")<CR>'
    endfor
    for i in s:FencUnicode
        let fencla=substitute(i,'\s.*$','','g')
        let fenname=fencla.'<tab>('.substitute(i,'^.\{-}\s\+','','g').')'
        let fenname=substitute(fenname,' ','\\ ','g')
        let fenname=substitute(fenname,'\.','_','g')
        exec 'menu &Tools.Encodin&g.&Unicode.'.fenname.' :call FencMenuSel("'.fencla.'")<CR>'
    endfor
    menu &Tools.Encodin&g.-sep-                  :
    menu &Tools.Encodin&g.&Auto\ Detect           :FencAutoDetect<CR>
    menu &Tools.Encodin&g.&Toggle\ Encoding\ list :FencView<CR>
    menu &Tools.Encodin&g.&Input\.\.\.            :call FencMenuSel(inputdialog(g:FencCustom))<CR>
endfunction


function! FencMenuSel(fen_name) "{{{1
    if  &modified
        echohl Error | echomsg "File is modified!" | echohl None
        return
    endif
    if a:fen_name==''
        return
    endif
    let WinNr=winnr()
    if bufname(winnr())==s:FencWinName
        return
    endif
    try
        let s:disable_autodetection=2
        exec "edit ++enc=".a:fen_name
    finally
        let s:disable_autodetection=0
    endtry

endfunction


function! s:FencProgressBar(percentage, string) "{{{1
"-----------------------------------------
"   a:percentage -- percent
"   a:string     -- leading description string (empty acceptable)
    let cmdheight=&cmdheight
    if cmdheight < 2
        let &cmdheight=2
    endif
    echo a:string." ".a:percentage."% (ctrl+c to stop)"
    let &cmdheight=cmdheight
    return
endfunction


function! s:FencProbeCp936(c) "{{{1
    if s:cp936_bchar<=0x80
        if a:c<0x80
            return
        else
            let s:cp936_bchar=a:c
            return
        endif
    else
        if a:c<0x40
            let s:cp936_error+=1
            let s:cp936_bchar=0x0
            return
        else
            let wc=s:cp936_bchar.a:c
            let s:cp936_bchar=0x0
            if index(s:cp936TopChars,wc)!=-1
                let s:cp936_count+=1
            endif
        endif
    endif
endfunction


function! s:FencProbeCp950(c) "{{{1
    if s:cp950_bchar<=0x80
        if a:c<0x80
            return
        else
            let s:cp950_bchar=a:c
            return
        endif
    else
        if a:c<0x40
            let s:cp950_error+=1
            let s:cp950_bchar=0x0
            return
        else
            let wc=s:cp950_bchar.a:c
            let s:cp950_bchar=0x0
            if index(s:cp950TopChars,wc)!=-1
                let s:cp950_count+=1
            endif
        endif
    endif
endfunction

function! s:FencProbeEuc_tw(c) "{{{1
    if s:euc_tw_bchar<=0x80
        if a:c<0x80
            return
        else
            let s:euc_tw_bchar=a:c
            return
        endif
    else
        if a:c<0x40
            let s:euc_tw_error+=1
            let s:euc_tw_bchar=0x0
            return
        else
            let wc=s:euc_tw_bchar.a:c
            let s:euc_tw_bchar=0x0
            if index(s:euc_twTopChars,wc)!=-1
                let s:euc_tw_count+=1
            endif
        endif
    endif
endfunction

function! s:FencProbeCp932(c) "{{{1
    if s:cp932_bchar<=0x80
        if a:c<0x80
            return
        else
            let s:cp932_bchar=a:c
            return
        endif
    else
        if a:c<0x40
            let s:cp932_error+=1
            let s:cp932_bchar=0x0
            return
        else
            let wc=s:cp932_bchar.a:c
            let s:cp932_bchar=0x0
            if index(s:cp932TopChars,wc)!=-1
                let s:cp932_count+=1
            endif
        endif
    endif
endfunction

function! s:FencProbeEuc_jp(c) "{{{1
    if s:euc_jp_bchar<=0x80
        if a:c<0x80
            return
        else
            let s:euc_jp_bchar=a:c
            return
        endif
    else
        if a:c<0x40
            let s:euc_jp_error+=1
            let s:euc_jp_bchar=0x0
            return
        else
            let wc=s:euc_jp_bchar.a:c
            let s:euc_jp_bchar=0x0
            if index(s:euc_jpTopChars,wc)!=-1
                let s:euc_jp_count+=1
            endif
        endif
    endif
endfunction

function! s:FencProbeCp949(c) "{{{1
    if s:cp949_bchar<=0x80
        if a:c<0x80
            return
        else
            let s:cp949_bchar=a:c
            return
        endif
    else
        if a:c<0x40
            let s:cp949_error+=1
            let s:cp949_bchar=0x0
            return
        else
            let wc=s:cp949_bchar.a:c
            let s:cp949_bchar=0x0
            if index(s:cp949TopChars,wc)!=-1
                let s:cp949_count+=1
            endif
        endif
    endif
endfunction

function! s:FencProbeUTF8(c) "{{{1
"still not full support here
"U-00000000 - U-0000007F:  0xxxxxxx
"U-00000080 - U-000007FF:  110xxxxx 10xxxxxx
"U-00000800 - U-0000FFFF:  1110xxxx 10xxxxxx 10xxxxxx
"U-00010000 - U-001FFFFF:  11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
"U-00200000 - U-03FFFFFF:  111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
"U-04000000 - U-7FFFFFFF:  1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx

    "if error occurs, we do nothing more but return.
    if s:UTF8_error > 0
        return
    endif

    if s:UTF8_state=="start"
        if a:c<=0x7f "still start state
            return
        elseif (a:c>=0xc2) && (a:c<=0xdf)
            let s:UTF8_state="wait"
            let s:UTF8_waitNr=1
            return
        elseif (a:c>=0xe0) && (a:c<=0xef)
            let s:UTF8_state="wait"
            let s:UTF8_waitNr=2
            return
        elseif (a:c>=0xf0) && (a:c<=0xf7)
            let s:UTF8_state="wait"
            let s:UTF8_waitNr=3
            return
        elseif (a:c>=0xf8) && (a:c<=0xfb)
            let s:UTF8_state="wait"
            let s:UTF8_waitNr=4
            return
        elseif (a:c>=0xf0) && (a:c<=0xfd)
            let s:UTF8_state="wait"
            let s:UTF8_waitNr=5
            return
        else
            let s:UTF8_error=1
            return
        endif
    else    "s:UTF8_state=="wait"
        if (a:c>=0x80) && (a:c<=0xbf)
            let s:UTF8_waitNr-=1
            if s:UTF8_waitNr==0
                let s:UTF8_state="start"
            endif
            return
        else
            let s:UTF8_error=1
            let s:UTF8_waitNr=0
            let s:UTF8_state="start"
            return
        endif
    endif
endfunction

"eol ??
function! s:FencHandleData() "{{{1
    let lnr=0
    let filename=bufname('%')
    if filename=='' || filename==s:FencWinName
        return
    endif
    let A_fbody=readfile(filename,'b')
    if A_fbody==[]
        return
    endif
    "check first and last several lines
    if len(A_fbody)<(g:fencview_checklines*2) || g:fencview_checklines==0
        let fbody=A_fbody
    else
        let fbody=A_fbody[:(g:fencview_checklines-1)]+A_fbody[(0-g:fencview_checklines):]
    endif
    " check BOM
    if s:FencProbeBOM(fbody[0])==1
        return
    endif
    " check header
    let bodylen=len(fbody)
    let _firstline=fbody[0]
    if _firstline[:7]=="VimCrypt"
        let s:FencRes="VimCrypt"
        return
    endif
    for line in fbody
        let lnr+=1
        call s:FencProgressBar(100*lnr/bodylen,' Processing... ',)
        let ci=0
        let ch="\x01"
        while ch!=''
            let ch=line[ci]
            let c=char2nr(ch)
            "=============
            call s:FencProbeUTF8(c)
            call s:FencProbeCp936(c)
            call s:FencProbeCp950(c)
            call s:FencProbeEuc_tw(c)
            call s:FencProbeCp932(c)
            call s:FencProbeEuc_jp(c)
            call s:FencProbeCp949(c)
            if s:FencRes!=''
                return
            endif
            "=============
            let ci=ci+1
        endwhile
    endfor
endfunction

function! s:FencProbeBOM(Firstline) "{{{1
" Vim can probe the file encoding by BOM correctly.
" This function is used to prevent probing other
" encodings by mistake.
    let ah1=a:Firstline[0]
    let ah2=a:Firstline[1]
    let ah3=a:Firstline[2]
    let ah4=a:Firstline[3]
    let a1=char2nr(ah1)
    let a2=char2nr(ah2)
    let a3=char2nr(ah3)
    let a4=char2nr(ah4)
    let s:FencRes="BOM"
    if a1.a2.a3==0xef.0xbb.0xbf "utf-8
        return 1
    elseif a1.a2==0xfe.0xff "utf-16 (ucs-2)
        return 1
    elseif a1.a2==0xff.0xfe "utf-16le (ucs-2le)
        return 1
    elseif a1.a2.a3.a4==0x00.0x00.0xfe.0xff "utf-32 (ucs-4)
        return 1
    elseif a1.a2.a3.a4==0xff.0xfe.0x00.0x00 "utf-32le (ucs-4le)
        return 1
    endif
    let s:FencRes=""
endfunction


function! s:FencInitVar() "{{{1
    "let s:small_regin=15
    "let s:big_regin=30
    let s:FencResCount=0
    let s:FencRes=''
    " cp936
    let s:cp936_bchar=''
    let s:cp936_count=0
    let s:cp936_error=0
    " cp950
    let s:cp950_bchar=''
    let s:cp950_count=0
    let s:cp950_error=0
    " euc_tw
    let s:euc_tw_bchar=''
    let s:euc_tw_count=0
    let s:euc_tw_error=0
    " cp932
    let s:cp932_bchar=''
    let s:cp932_count=0
    let s:cp932_error=0
    " euc_jp
    let s:euc_jp_bchar=''
    let s:euc_jp_count=0
    let s:euc_jp_error=0
    " cp949
    let s:cp949_bchar=''
    let s:cp949_count=0
    let s:cp949_error=0
    " UTF-8
    let s:UTF8_state="start"
    let s:UTF8_waitNr=0
    let s:UTF8_error=0
endfunction


function! s:FencDetectFileEncoding() "{{{1
    if bufname(winnr())==s:FencWinName
        return
    endif
    if  &modified
        echohl Error | echomsg "File is modified!" | echohl None
        return
    endif
    call s:FencInitVar()

    call s:FencHandleData()

    " VimCrypt
    if s:FencRes=="VimCrypt"
        echohl Error | echomsg "This is Vim encrypted file, decrypt it first!" | echohl None
        return
    endif

    " BOM
    if s:FencRes=="BOM"
        let tmp_fenc=&fencs
        set fencs=ucs-bom,utf-8,utf-16,latin1
        exec "e"
        exec "set fencs=".tmp_fenc
        return
    endif

    if s:UTF8_error == 0
        " even if the file is strict utf-8 format, it is still
        " possible to be another encoding.
        let s:FencRes = "utf-8"
        if s:cp936_error == 0 && s:FencResCount < s:cp936_count
            let s:FencResCount = s:cp936_count
            let s:FencRes = "cp936"
        endif
        if s:cp950_error == 0 && s:FencResCount < s:cp950_count
            let s:FencResCount=s:cp950_count
            let s:FencRes = "cp950"
        endif
        if s:euc_tw_error == 0 && s:FencResCount < s:euc_tw_count
            let s:FencResCount=s:euc_tw_count
            let s:FencRes = "euc-tw"
        endif
        if s:cp932_error == 0 && s:FencResCount < s:cp932_count
            let s:FencResCount=s:cp932_count
            let s:FencRes = "cp932"
        endif
        if s:euc_jp_error == 0 && s:FencResCount < s:euc_jp_count
            let s:FencResCount=s:euc_jp_count
            let s:FencRes = "euc-jp"
        endif
        if s:cp949_error == 0 && s:FencResCount < s:cp949_count
            let s:FencResCount=s:cp949_count
            let s:FencRes = "cp949"
        endif
    else
        if s:FencResCount<(s:cp936_count-s:cp936_error)
            let s:FencResCount=s:cp936_count
            let s:FencRes="cp936"
        endif
        if s:FencResCount<(s:cp950_count-s:cp950_error)
            let s:FencResCount=s:cp950_count
            let s:FencRes="cp950"
        endif
        if s:FencResCount<(s:euc_tw_count-s:euc_tw_error)
            let s:FencResCount=s:euc_tw_count
            let s:FencRes="euc-tw"
        endif
        if s:FencResCount<(s:cp932_count-s:cp932_error)
            let s:FencResCount=s:cp932_count
            let s:FencRes="cp932"
        endif
        if s:FencResCount<(s:euc_jp_count-s:euc_jp_error)
            let s:FencResCount=s:euc_jp_count
            let s:FencRes="euc-jp"
        endif
        if s:FencResCount<(s:cp949_count-s:cp949_error)
            let s:FencResCount=s:cp949_count
            let s:FencRes="cp949"
        endif

        if s:FencResCount == 0
            let s:FencRes="ascii"
        endif
    endif

    let Syn=&syntax
    try
        let s:disable_autodetection=2
        if s:FencRes=='ascii'
            exec "edit"
        else
            exec "edit ++enc=".s:FencRes
        endif
    finally
        let s:disable_autodetection=0
    endtry
    if Syn!=''
        let &syntax=Syn
    endif
endfunction


" initialization{{{1
if !exists('g:legacy_encoding')
    if &encoding!~?'^utf' && &encoding!~?'^ucs'
        let g:legacy_encoding=&encoding
    elseif &fileencodings=~?'^ucs-bom,utf-8,[^,]\+' &&
                \!(has('unix') && !has('win32unix'))
        let g:legacy_encoding=matchstr(&fileencodings, '^ucs-bom,utf-8,\zs[^,]\+')
    endif
    if !exists('g:legacy_encoding') || g:legacy_encoding=='default'
        let g:legacy_encoding=''
    endif
endif

let s:disable_autodetection=0

augroup FencView
    au!
augroup END

if has('gui_running')
    augroup FencView
        au BufEnter *               call s:FencCreateMenu()
    augroup END
endif

command! -nargs=0                FencView       call s:ToggleFencView()
command! -nargs=* -complete=file FencAutoDetect call
                               \ s:EditAutoEncoding(<f-args>)
command! -nargs=+ -complete=file FencManualEncoding call
                               \ s:EditManualEncoding(<f-args>)

if g:fencview_autodetect
    exec 'au BufRead ' . g:fencview_auto_patterns .
                \' call s:EditAutoEncoding()'
    exec 'au BufWinEnter ' . g:fencview_auto_patterns .
                \' call s:CheckModelineFileEncoding()'
endif

" Restore cpoptions
let &cpo=s:cpo_save

" vim: set et fdm=marker sts=4 sw=4 tw=64:
