import { CacheKey, CachePrefix, CacheTTL } from 'aragami';
import {
  IsInt,
  IsNotEmpty,
  IsPositive,
  IsString,
  Max,
  MaxLength,
  Min,
} from 'class-validator';
import { ApiProperty } from '@nestjs/swagger';
import { MycardUser } from 'nestjs-mycard';
import * as net from 'net';

export class GameServerInput {
  @IsString()
  @ApiProperty({ description: 'The name of the game server' })
  @IsNotEmpty()
  @MaxLength(32)
  name: string;
  @IsString()
  @ApiProperty({ description: 'The description of the game server' })
  @MaxLength(256)
  description: string;
  @IsString()
  @MaxLength(100)
  @ApiProperty({ description: 'The address of the game server' })
  host: string;
  @IsPositive()
  @IsInt()
  @Min(1)
  @Max(65535)
  @ApiProperty({ description: 'The port of the game server' })
  port: number;

  async testConnection() {
    return new Promise<string>((resolve) => {
      const socket = new net.Socket();

      socket.connect(this.port, this.host);
      socket.setTimeout(5000);

      socket.on('connect', () => {
        socket.destroy();
        resolve(undefined);
      });

      socket.on('error', (error) => {
        socket.destroy();
        resolve(error.message);
      });

      socket.on('timeout', () => {
        socket.destroy();
        resolve('Connection timed out.');
      });
    });
  }
}

@CacheTTL(600000)
@CachePrefix('server')
export class GameServer extends GameServerInput {
  @ApiProperty({ description: 'The ID of the game.' })
  ns: string;
  @ApiProperty({ description: 'The username of the game server owner.' })
  ownerUsername: string;
  @ApiProperty({ description: 'The name of the game server owner.' })
  owner: string;

  fromInput(input: GameServerInput) {
    this.name = input.name;
    this.description = input.description;
    this.host = input.host;
    this.port = input.port;
    return this;
  }

  fromOwner(user: MycardUser) {
    this.ownerUsername = user.username;
    this.owner = user.name;
    return this;
  }

  withNs(ns: string) {
    this.ns = ns;
    return this;
  }

  @CacheKey()
  cacheKey() {
    return `${this.ns}:${this.ownerUsername}:${this.name}`;
  }
}
