import {
  Body,
  Controller,
  Delete,
  Get,
  Param,
  Put,
  Query,
  ValidationPipe,
} from '@nestjs/common';
import { AppService } from './app.service';
import {
  ApiBody,
  ApiCreatedResponse,
  ApiHeader,
  ApiNoContentResponse,
  ApiOkResponse,
  ApiOperation,
  ApiParam,
  ApiQuery,
} from '@nestjs/swagger';
import { GameServer, GameServerInput } from './dto/GameServer.dto';
import {
  BlankReturnMessageDto,
  ReturnMessageDto,
} from './dto/ReturnMessage.dto';
import { MycardUser, PutMycardUser } from 'nestjs-mycard';

const ServerReturnMessageDto = ReturnMessageDto(GameServer);

@Controller('api/server-list/:ns')
export class AppController {
  constructor(private readonly appService: AppService) {}

  @Get()
  @ApiOperation({ summary: 'Get server list' })
  @ApiParam({ name: 'ns', description: 'Namespace' })
  @ApiOkResponse({ type: [GameServer] })
  serverList(@Param('ns') ns: string) {
    return this.appService.serverList(ns);
  }

  @Put()
  @ApiOperation({ summary: 'Register server. Must refresh once every minute.' })
  @ApiParam({ name: 'ns', description: 'Namespace' })
  @ApiHeader({
    name: 'Authorization',
    description: 'Bearer token from MyCard account.',
  })
  @ApiBody({ type: GameServerInput })
  @ApiCreatedResponse({ type: ServerReturnMessageDto })
  async registerServer(
    @Param('ns') ns: string,
    @PutMycardUser() user: MycardUser,
    @Body(
      new ValidationPipe({
        transform: true,
        transformOptions: { enableImplicitConversion: true },
      }),
    )
    input: GameServerInput,
  ) {
    const server = await this.appService.registerServer(ns, user, input);
    return new ServerReturnMessageDto(201, 'success', server);
  }

  @Delete()
  @ApiOperation({ summary: 'Delete server.' })
  @ApiParam({ name: 'ns', description: 'Namespace' })
  @ApiHeader({
    name: 'Authorization',
    description: 'Bearer token from MyCard account.',
  })
  @ApiQuery({ name: 'name', description: 'Server name' })
  @ApiNoContentResponse({ type: BlankReturnMessageDto })
  async deleteServer(
    @Param('ns') ns: string,
    @PutMycardUser() user: MycardUser,
    @Query('name') name: string,
  ) {
    await this.appService.deleteServer(ns, user, name);
    return new BlankReturnMessageDto(204, 'success');
  }
}
