import { Injectable, ConsoleLogger } from '@nestjs/common';
import { Aragami } from 'aragami';
import { InjectAragami } from 'nestjs-aragami';
import { GameServer, GameServerInput } from './dto/GameServer.dto';
import { MycardUser } from 'nestjs-mycard';
import { ConfigService } from '@nestjs/config';
import { BlankReturnMessageDto } from './dto/ReturnMessage.dto';

@Injectable()
export class AppService extends ConsoleLogger {
  private maxServerCount = parseInt(this.config.get('MAX_SERVER_COUNT'));
  constructor(
    @InjectAragami() private readonly aragami: Aragami,
    private config: ConfigService,
  ) {
    super('app');
  }

  async serverList(ns: string) {
    return this.aragami.values(GameServer, `${ns}:`);
  }

  async registerServer(ns: string, user: MycardUser, input: GameServerInput) {
    const server = new GameServer().withNs(ns).fromInput(input).fromOwner(user);
    if (!user.admin) {
      const existingServers = await this.aragami.keys(
        GameServer,
        `${ns}:${user.username}:`,
      );
      if (existingServers.length > this.maxServerCount) {
        throw new BlankReturnMessageDto(
          403,
          'You have too many servers.',
        ).toException();
      } else if (existingServers.length === this.maxServerCount) {
        const existingServer = await this.aragami.get(
          GameServer,
          server.cacheKey(),
        );
        if (!existingServer) {
          throw new BlankReturnMessageDto(
            403,
            'You have too many servers.',
          ).toException();
        }
      }
    }
    const errorReason = await server.testConnection();
    if (errorReason) {
      throw new BlankReturnMessageDto(
        404,
        `Server broken: ${errorReason}`,
      ).toException();
    }
    await this.aragami.set(server);
    return server;
  }

  async deleteServer(ns: string, user: MycardUser, name: string) {
    const server = await this.aragami.get(
      GameServer,
      `${ns}:${user.username}:${name}`,
    );
    if (!server) {
      throw new BlankReturnMessageDto(404, 'Server not found.').toException();
    }
    await this.aragami.del(server);
  }
}
