import os
from fastapi import FastAPI, HTTPException
from fastapi.responses import RedirectResponse
import httpx
from starlette.middleware.cors import CORSMiddleware


app = FastAPI()
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

@app.get("/{route}")
async def redirect_route(route: str):
    env_key = f"ROUTE_{route}"
    target_url = os.getenv(env_key)

    if not target_url:
        raise HTTPException(status_code=404, detail=f"Route {env_key} not found.")

    try:
        async with httpx.AsyncClient() as client:
            response = await client.get(target_url)
            response.raise_for_status()
            redirect_link = response.text.strip()
    except Exception as e:
        raise HTTPException(status_code=501, detail=f"Failed to retrieve redirect link: {str(e)}")

    return RedirectResponse(url=redirect_link, status_code=302)


if __name__ == "__main__":
    import uvicorn

    uvicorn.run(app, proxy_headers=True, host="0.0.0.0", port=3000)

